"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertToNumber = convertToNumber;
exports.roundToFloat16Bits = roundToFloat16Bits;
var _primordials = require("./primordials.cjs");
const mantissaBitLength = 10;
const mantissaMask = 0x3ff;
const exponentMax = 31;
const exponentBias = 15;
function roundToEven(num) {
  const truncated = (0, _primordials.MathTrunc)(num);
  const delta = (0, _primordials.MathAbs)(num - truncated);
  if (delta > 0.5 || delta === 0.5 && truncated % 2 !== 0) {
    return truncated + (0, _primordials.MathSign)(num);
  }
  return truncated;
}
function roundToFloat16Bits(num) {
  const absolute = (0, _primordials.MathAbs)(num);
  const s = num < 0 || (0, _primordials.ObjectIs)(num, -0) ? 1 : 0;
  let m, e;
  if (!(0, _primordials.NumberIsFinite)(absolute)) {
    m = (0, _primordials.NumberIsNaN)(absolute) ? 0x200 : 0;
    e = exponentMax;
  } else {
    let rawE = (0, _primordials.MathFloor)((0, _primordials.MathLog2)(absolute));
    let c = (0, _primordials.MathPow)(2, -rawE);
    if (absolute * c < 1) {
      --rawE;
      c *= 2;
    } else if (absolute * c >= 2) {
      ++rawE;
      c /= 2;
    }
    if (rawE + exponentBias >= exponentMax) {
      m = 0;
      e = exponentMax;
    } else if (rawE + exponentBias >= 1) {
      m = roundToEven((absolute * c - 1) * 0x400);
      e = rawE + exponentBias;
    } else {
      m = roundToEven(absolute * 0x1000000);
      e = 0;
    }
  }
  return s << 15 | e << mantissaBitLength | m;
}
const buffer = new _primordials.NativeArrayBuffer(4);
const floatView = new _primordials.NativeFloat32Array(buffer);
const uint32View = new _primordials.NativeUint32Array(buffer);
const mantissaTable = new _primordials.NativeUint32Array(2048);
for (let i = 1; i < 1024; ++i) {
  let m = i << 13;
  let e = 0;
  while ((m & 0x00800000) === 0) {
    m <<= 1;
    e -= 0x00800000;
  }
  m &= ~0x00800000;
  e += 0x38800000;
  mantissaTable[i] = m | e;
}
for (let i = 1024; i < 2048; ++i) {
  mantissaTable[i] = 0x38000000 + (i - 1024 << 13);
}
const exponentTable = new _primordials.NativeUint32Array(64);
for (let i = 1; i < 31; ++i) {
  exponentTable[i] = i << 23;
}
exponentTable[31] = 0x47800000;
exponentTable[32] = 0x80000000;
for (let i = 33; i < 63; ++i) {
  exponentTable[i] = 0x80000000 + (i - 32 << 23);
}
exponentTable[63] = 0xc7800000;
const offsetTable = new _primordials.NativeUint16Array(64);
for (let i = 1; i < 64; ++i) {
  if (i !== 32) {
    offsetTable[i] = 1024;
  }
}
function convertToNumber(float16bits) {
  const i = float16bits >> mantissaBitLength;
  uint32View[0] = mantissaTable[offsetTable[i] + (float16bits & mantissaMask)] + exponentTable[i];
  return floatView[0];
}