'use strict';

var pgp = require('../lib');

var dateSample = new Date();

// common error messages;
var errors = {
    rawNull: function () {
        return "Values null/undefined cannot be used as raw text.";
    }
};

var dummy = function () {
};

var userObj = {
    name: "John O'Connor",
    dob: new Date(1980, 5, 15),
    active: true
};

describe("Method as.bool", function () {

    it("must correctly convert any boolean-like value", function () {
        expect(pgp.as.bool()).toBe("null");
        expect(pgp.as.bool(null)).toBe("null");
        expect(pgp.as.bool(0)).toBe("false");
        expect(pgp.as.bool(false)).toBe("false");
        expect(pgp.as.bool(1)).toBe("true");
        expect(pgp.as.bool(true)).toBe("true");
        expect(pgp.as.bool(10)).toBe("true");
        expect(pgp.as.bool(-10)).toBe("true");
        expect(pgp.as.bool([])).toBe("true");
        expect(pgp.as.bool({})).toBe("true");
        expect(pgp.as.bool("false")).toBe("true");
    });

    it("must correctly resolve functions", function () {
        expect(pgp.as.bool(dummy)).toBe("null");
        expect(pgp.as.bool(function () {
            return null;
        })).toBe("null");
        expect(pgp.as.bool(function () {
            return true;
        })).toBe("true");
        expect(pgp.as.bool(function () {
            return false;
        })).toBe("false");
    });

});

describe("Method as.number", function () {

    it("must correctly convert any number", function () {
        expect(pgp.as.number()).toBe("null");
        expect(pgp.as.number(null)).toBe("null");
        expect(pgp.as.number(0)).toBe("0");
        expect(pgp.as.number(1)).toBe("1");
        expect(pgp.as.number(1234567890)).toBe("1234567890");
        expect(pgp.as.number(-123.456)).toBe("-123.456");
        expect(pgp.as.number(NaN)).toBe("'NaN'");
        expect(pgp.as.number(1 / 0)).toBe("'+Infinity'");
        expect(pgp.as.number(-1 / 0)).toBe("'-Infinity'");
    });

    it("must correctly resolve functions", function () {
        expect(pgp.as.number(dummy)).toBe("null");
        expect(pgp.as.number(function () {
            return null;
        })).toBe("null");
        expect(pgp.as.number(function () {
            return 123;
        })).toBe("123");
        expect(pgp.as.number(function () {
            return 0;
        })).toBe("0");
        expect(pgp.as.number(function () {
            return -1 / 0;
        })).toBe("'-Infinity'");

        // deep-call test:
        expect(pgp.as.number(function () {
            return function () {
                return function () {
                    return 123;
                }
            };
        })).toBe("123");

    });

    it("must correctly reject invalid values", function () {

        var err = " is not a number.";
        expect(function () {
            pgp.as.number('');
        }).toThrow(new Error("''" + err));

        expect(function () {
            pgp.as.number([1, 2]);
        }).toThrow(new Error("'1,2'" + err));

    });

});

describe("Method as.text", function () {
    it("must correctly convert any text", function () {

        expect(pgp.as.text()).toBe("null");
        expect(pgp.as.text(null)).toBe("null");

        expect(pgp.as.text("")).toBe("''");
        expect(pgp.as.text("", true)).toBe(""); // raw-text test;

        expect(pgp.as.text("some text")).toBe("'some text'");
        expect(pgp.as.text("some text", true)).toBe("some text"); // raw-text test;

        expect(pgp.as.text("'starts with quote")).toBe("'''starts with quote'");
        expect(pgp.as.text("'starts with quote", true)).toBe("'starts with quote"); // raw-text test;

        expect(pgp.as.text("ends with quote'")).toBe("'ends with quote'''");
        expect(pgp.as.text("ends with quote'", true)).toBe("ends with quote'"); // raw-text test;

        expect(pgp.as.text("has '' two quotes")).toBe("'has '''' two quotes'");
        expect(pgp.as.text("has '' two quotes", true)).toBe("has '' two quotes"); // raw-text test;

        expect(pgp.as.text("'")).toBe("''''");
        expect(pgp.as.text("'", true)).toBe("'"); // raw-text test;

        expect(pgp.as.text("''")).toBe("''''''");
        expect(pgp.as.text("''", true)).toBe("''"); // raw-text test;

        expect(pgp.as.text(-123.456)).toBe("'-123.456'");
        expect(pgp.as.text(true)).toBe("'true'");
        expect(pgp.as.text(false)).toBe("'false'");
        expect(pgp.as.text(dateSample)).toBe("'" + dateSample.toString() + "'");

        expect(pgp.as.text([])).toBe("''");
        expect(pgp.as.text([], true)).toBe(""); // raw-text test;

        expect(pgp.as.text([1, "hello"])).toBe("'1,hello'"); // converts string as is;
        expect(pgp.as.text([1, "hello"], true)).toBe("1,hello"); // converts string as is;

        expect(pgp.as.text({})).toBe("'[object Object]'");
    });

    it("must correctly resolve functions", function () {

        expect(pgp.as.text(dummy)).toBe("null");

        expect(pgp.as.text(function () {
            return null;
        })).toBe("null");

        expect(pgp.as.text(function () {
            return 'hello';
        })).toBe("'hello'");

    });

    it("must correctly respond to invalid raw-text requests", function () {
        expect(function () {
            pgp.as.text(undefined, true);
        }).toThrow(new Error(errors.rawNull()));

        expect(function () {
            pgp.as.text(null, true);
        }).toThrow(new Error(errors.rawNull()));

    });

});

describe("Method as.date", function () {
    it("must correctly convert any date", function () {
        expect(pgp.as.date()).toBe("null");
        expect(pgp.as.date(null)).toBe("null");
        expect(pgp.as.date(dateSample)).toBe("'" + dateSample.toUTCString() + "'");
        expect(pgp.as.date(dateSample, true)).toBe(dateSample.toUTCString());
    });

    it("must correctly resolve functions", function () {
        expect(pgp.as.date(dummy)).toBe("null");
        expect(pgp.as.date(function () {
            return null;
        })).toBe("null");
        expect(pgp.as.date(function () {
            return dateSample;
        }, true)).toBe(dateSample.toUTCString());
    });

    it("must correctly reject invalid requests", function () {

        expect(function () {
            pgp.as.date(undefined, true);
        }).toThrow(new Error(errors.rawNull()));

        expect(function () {
            pgp.as.date(null, true);
        }).toThrow(new Error(errors.rawNull()));

        expect(function () {
            pgp.as.date("");
        }).toThrow(new Error("'' is not a Date object."));

        expect(function () {
            pgp.as.date("bla-bla");
        }).toThrow(new Error("'bla-bla' is not a Date object."));

        expect(function () {
            pgp.as.date(123);
        }).toThrow(new Error("'123' is not a Date object."));

        expect(function () {
            pgp.as.date([]);
        }).toThrow(new Error("'' is not a Date object."));

        expect(function () {
            pgp.as.date({});
        }).toThrow(new Error("'[object Object]' is not a Date object."));

    });

});

describe("Method as.csv", function () {

    it("must correctly convert any parameters into CSV", function () {

        ////////////////////////////////
        // positive tests;

        expect(pgp.as.csv()).toBe(""); // test undefined;
        expect(pgp.as.csv([])).toBe(""); // test empty array;
        expect(pgp.as.csv([[]])).toBe("array[]"); // test empty array;
        expect(pgp.as.csv(null)).toBe("null"); // test null;
        expect(pgp.as.csv([null])).toBe("null"); // test null in array;
        expect(pgp.as.csv([undefined])).toBe("null"); // test undefined in array;
        expect(pgp.as.csv([null, undefined])).toBe("null,null"); // test combination of null + undefined in array;

        expect(pgp.as.csv(0)).toBe("0"); // test zero;
        expect(pgp.as.csv([0])).toBe("0"); // test zero in array;
        expect(pgp.as.csv(-123.456)).toBe("-123.456"); // test a float;
        expect(pgp.as.csv([-123.456])).toBe("-123.456"); // test a float in array;

        expect(pgp.as.csv(true)).toBe("true"); // test boolean True;
        expect(pgp.as.csv([true])).toBe("true"); // test boolean True in array;

        expect(pgp.as.csv(false)).toBe("false"); // test boolean False;
        expect(pgp.as.csv([false])).toBe("false"); // test boolean False in array;

        expect(pgp.as.csv("")).toBe("''"); // empty text;
        expect(pgp.as.csv([""])).toBe("''"); // empty text in array;
        expect(pgp.as.csv("simple text")).toBe("'simple text'"); // simple text;
        expect(pgp.as.csv("don't break")).toBe("'don''t break'"); // text with one single-quote symbol;
        expect(pgp.as.csv("test ''")).toBe("'test '''''"); // text with two single-quote symbols;

        expect(pgp.as.csv(dateSample)).toBe("'" + dateSample.toUTCString() + "'"); // test date;
        expect(pgp.as.csv([dateSample])).toBe("'" + dateSample.toUTCString() + "'"); // test date in array;

        expect(pgp.as.csv([userObj])).toBe(pgp.as.text(JSON.stringify(userObj)));

        // test a combination of all possible types;
        expect(pgp.as.csv([12.34, true, "don't break", null, undefined, userObj, dateSample, [1, 2]]))
            .toBe("12.34,true,'don''t break',null,null," + pgp.as.text(JSON.stringify(userObj)) + ",'" + dateSample.toUTCString() + "',array[1,2]");

        // test array-type as a parameter;
        expect(pgp.as.csv([1, [2, 3], 4])).toBe("1,array[2,3],4");
        expect(pgp.as.csv([1, [['two'], ['three']], 4])).toBe("1,array[['two'],['three']],4");

    });

    it("must correctly resolve functions", function () {

        expect(pgp.as.csv(dummy)).toBe("");

        expect(pgp.as.csv(function () {
            return null;
        })).toBe("null");

        expect(pgp.as.csv(function () {
            return 'one';
        })).toBe("'one'");

        expect(pgp.as.csv(function () {
            return ['one', 'two', [1, 2, 3]];
        })).toBe("'one','two',array[1,2,3]");
    });

});

describe("Method as.json", function () {

    it("must correctly convert any object into JSON", function () {
        expect(pgp.as.json()).toBe("null");
        expect(pgp.as.json(null)).toBe("null");
        expect(pgp.as.json({})).toBe("'" + JSON.stringify({}) + "'");
        expect(pgp.as.json(userObj)).toBe(pgp.as.text(JSON.stringify(userObj)));
    });

    it("must correctly resolve functions", function () {

        expect(pgp.as.json(dummy)).toBe("null");
        expect(pgp.as.json(function () {
            return null;
        })).toBe("null");

        expect(pgp.as.json(function () {
            return userObj;
        })).toBe(pgp.as.text(JSON.stringify(userObj)));
    });

    it("must correctly reject invalid requests", function () {
        expect(function () {
            pgp.as.json(null, true);
        }).toThrow(new Error(errors.rawNull()));
        expect(function () {
            pgp.as.json(undefined, true);
        }).toThrow(new Error(errors.rawNull()));
    });
});

describe("Method as.array", function () {

    it("must correctly convert an empty array or value", function () {
        expect(pgp.as.array()).toBe('null');
        expect(pgp.as.array(null)).toBe('null');
        expect(pgp.as.array([])).toBe("array[]");
    });

    it("must correctly convert multi-dimension arrays", function () {
        expect(pgp.as.array([[1, 2], ['three', 'four', [5, 'six', true]]]))
            .toBe("array[[1,2],['three','four',[5,'six',true]]]");
    });

    // 20-dimension test;
    it("must correctly convert arrays of any depth", function () {
        expect(pgp.as.array([[[[[[[[[[[[[[[[[[[[20]]]]]]]]]]]]]]]]]]]]))
            .toBe("array[[[[[[[[[[[[[[[[[[[[20]]]]]]]]]]]]]]]]]]]]");
    });

    it("must correctly resolve functions", function () {

        expect(pgp.as.array(dummy)).toBe("null");

        expect(pgp.as.array(function () {
            return null;
        })).toBe("null");

        expect(pgp.as.array(function () {
            return [1, 2, 3];
        })).toBe("array[1,2,3]");

    });

    it("must correctly reject invalid requests", function () {

        var err = " is not an Array object.";
        expect(function () {
            pgp.as.array(123);
        }).toThrow(new Error("'123'" + err));

        expect(function () {
            pgp.as.array('');
        }).toThrow(new Error("''" + err));

    });

});

describe("Method as.dummy", function () {

    it("must correctly convert any function return", function () {
        expect(pgp.as.func()).toBe('null');
        expect(pgp.as.func(null)).toBe('null');

        expect(pgp.as.func(function () {
            return 1;
        })).toBe("1");

        expect(pgp.as.func(function () {
            return [1, 2, 3];
        })).toBe("array[1,2,3]");

        expect(pgp.as.func(function () {
        })).toBe("null");

        expect(pgp.as.func(function () {
            return null;
        })).toBe("null");

        expect(pgp.as.func(function () {
            return dummy()
            {
                return dummy()
                {
                }
            }
        })).toBe("null");

        expect(pgp.as.func(function () {
            return function () {
                return function () {
                    return true;
                }
            }
        })).toBe("true");

        expect(pgp.as.format("$1,$1^", function () {
            return function () {
                return 'one';
            };
        })).toBe("'one',one");

        // testing function-object context;
        expect(pgp.as.format("${summ}", {
            val1: 1,
            val2: 2,
            summ: function () {
                return this.val1 + this.val2; // `this` must work here;
            }
        })).toBe("3");

        // the same object context must be
        // passed into every sub-function;
        expect(pgp.as.func(function () {
            return function () {
                return function () {
                    return this.test;
                }
            }
        }, false, {
            test: "Hello!"
        })).toBe("'Hello!'");

        /////////////////////////////
        // negative tests;

        expect(function () {
            pgp.as.func(1);
        }).toThrow(new Error("'1' is not a function."));

        expect(function () {
            pgp.as.func(undefined, true);
        }).toThrow(new Error("Values null/undefined cannot be used as raw text."));

        expect(function () {
            pgp.as.func(null, true);
        }).toThrow(new Error("Values null/undefined cannot be used as raw text."));

        expect(function () {
            pgp.as.func(function () {
                throw "internal error";
            });
        }).toThrow("internal error");

        expect(function () {
            pgp.as.func(dummy, false, '');
        }).toThrow(new Error("'' is not an object."));

        expect(function () {
            pgp.as.func(dummy, false, 0);
        }).toThrow(new Error("'0' is not an object."));

    });
});

describe("Method as.format", function () {

    it("must return a correctly formatted string", function () {

        expect(pgp.as.format("", [])).toBe("");

        expect(pgp.as.format("$1", [])).toBe("$1");
        expect(pgp.as.format("$1^", [])).toBe("$1^");

        expect(pgp.as.format("$1")).toBe("$1");
        expect(pgp.as.format("$1", null)).toBe("null");

        expect(pgp.as.format("$1", [undefined])).toBe("null");

        expect(pgp.as.format("$1", "one")).toBe("'one'");
        expect(pgp.as.format("$1^", "one")).toBe("one");

        expect(pgp.as.format("$1", ["one"])).toBe("'one'");
        expect(pgp.as.format("$1^", ["one"])).toBe("one");

        expect(pgp.as.format("$1, $1", "one")).toBe("'one', 'one'");

        expect(pgp.as.format("$1$1", "one")).toBe("'one''one'");
        expect(pgp.as.format("$1^$1^", "one")).toBe("oneone");

        expect(pgp.as.format("$1", [userObj])).toBe(pgp.as.text(JSON.stringify(userObj)));
        expect(pgp.as.format("$1^", [userObj])).toBe(JSON.stringify(userObj));

        expect(pgp.as.format("$1, $2, $3, $4", [true, -12.34, "text", dateSample])).toBe("true, -12.34, 'text', '" + dateSample.toUTCString() + "'");

        expect(pgp.as.format("$1 $1, $2 $2, $1", [1, "two"])).toBe("1 1, 'two' 'two', 1"); // test for repeated variables;

        expect(pgp.as.format("Test: $1", ["don't break quotes!"])).toBe("Test: 'don''t break quotes!'");

        // testing with lots of variables;
        var source = "", dest = "", params = [];
        for (var i = 1; i <= 1000; i++) {
            source += '$' + i;
            dest += i;
            params.push(i);
        }
        expect(pgp.as.format(source, params)).toBe(dest);

        // testing various cases with many variables:
        // - variables next to each other;
        // - variables not defined;
        // - variables are repeated;
        // - long variable names present;
        expect(pgp.as.format("$1$2,$3,$4,$5,$6,$7,$8,$9,$10$11,$12,$13,$14,$15,$1,$3", [1, 2, 'C', 'DDD', 'E', 'F', 'G', 'H', 'I', 88, 99, 'LLL']))
            .toBe("12,'C','DDD','E','F','G','H','I',8899,'LLL',$13,$14,$15,1,'C'");

        // test that variable names are not confused for longer ones;
        expect(pgp.as.format("$11, $1, $111, $1", 123)).toBe("$11, 123, $111, 123");

        // test that variable names are not confused for longer ones,
        // even when they are right next to each other;
        expect(pgp.as.format("$11$1$111$1", 123)).toBe("$11123$111123");

        expect(pgp.as.format("$1, $2", [
            'one', [2, 3]
        ])).toBe("'one', array[2,3]");

        // check that gaps are handled correctly;
        expect(pgp.as.format("$2, $4, $6", [1, 2, 3, 4, 5])).toBe("2, 4, $6");

        ///////////////////////////////////////////////////
        // test that inserting strings with variable names
        // in them doesn't effect formatting;

        // a) for regular variables;
        expect(pgp.as.format("$1, $2, $3^, $4", ["$2", "$3", "$2,$3"])).toBe("'$2', '$3', $2,$3, $4");

        // b) for the Named Parameters;
        expect(pgp.as.format("${one}, ${two}, ${three^}, $four", {
                one: "${two}",
                two: "${three}",
                three: "${two},${three}"
            }
        )).toBe("'${two}', '${three}', ${two},${three}, $four");

    });

    it("must correctly inject raw-text variables", function () {

        expect(pgp.as.format("${name},${name^},${name},${name^}", {
            name: 'me'
        })).toBe("'me',me,'me',me");

        expect(pgp.as.format("$1,$1^,$1,$1^", 'hello')).toBe("'hello',hello,'hello',hello");

        expect(pgp.as.format("$1,$2,$1^,$2^", ['one', 'two']))
            .toBe("'one','two',one,two");

        expect(pgp.as.format("$1^ $2^ $1", ["Don't break", 'this']))
            .toBe("Don't break this 'Don''t break'");

        expect(pgp.as.format("$1^,$1", dateSample))
            .toBe(dateSample.toUTCString() + ",'" + dateSample.toUTCString() + "'");

    });

    it("must correctly reject invalid requests", function () {

        var errEmptyString = "Parameter 'query' must be a text string.";

        expect(function () {
            pgp.as.format();
        }).toThrow(new Error(errEmptyString));

        expect(function () {
            pgp.as.format(null);
        }).toThrow(new Error(errEmptyString));

        expect(function () {
            pgp.as.format(null, [1, 2, 3]);
        }).toThrow(new Error(errEmptyString));

        expect(function () {
            pgp.as.format(123);
        }).toThrow(new Error(errEmptyString));

        expect(function () {
            pgp.as.format(function () {
                return '';
            }, dummy)
        }).toThrow(new Error(errEmptyString));

        expect(function () {
            pgp.as.format("$1^", null);
        }).toThrow(new Error(errors.rawNull()));

        expect(function () {
            pgp.as.format("$1^", [null]);
        }).toThrow(new Error(errors.rawNull()));

        expect(function () {
            pgp.as.format("$1^", [undefined]);
        }).toThrow(new Error(errors.rawNull()));

    });

});

describe("Named Parameters", function () {

    it("must recognize all supported symbols", function () {
        expect(pgp.as.format("${one},$(two),$[three],$<four>,$/five/", {
            one: 1,
            two: 2,
            three: 3,
            four: 4,
            five: 5
        })).toBe("1,2,3,4,5");
    });

    it("must ignore mixed open-close symbols", function () {
        var openers = '{([</', closers = '})]>/';
        for (var i = 0; i < openers.length; i++) {
            for (var k = 0; k < closers.length; k++) {
                var txt = '$' + openers[i] + 'value' + closers[k];
                var s = pgp.as.format(txt, {
                    value: 'hello'
                });
                if (i === k) {
                    expect(s).toBe("'hello'");
                } else {
                    expect(s).toBe(txt);
                }
            }
        }

    });

    it("must ignore internal spaces", function () {
        expect(pgp.as.format("${  one  },$(  two  ),$[  three  ],$<  four  >,$/  five  /", {
            one: 1,
            two: 2,
            three: 3,
            four: 4,
            five: 5
        })).toBe("1,2,3,4,5");
    });

    it("must support short property names", function () {
        expect(pgp.as.format("${$}$(_)$[a]$< _$>$/$$ /", {
            $: 1,
            _: 2,
            a: 3,
            _$: 4,
            $$: 5
        })).toBe("12345");
    });

    it("must support digit-only property names", function () {
        expect(pgp.as.format("${1}, $(2), $[5^]", {
            1: 'one',
            2: 'two',
            5: 'five'
        })).toBe("'one', 'two', five");
    });

    it("must recognize case difference", function () {
        expect(pgp.as.format("${value},$(Value),$[VALUE],$<valuE>,$/vaLue/", {
            value: 1,
            Value: 2,
            VALUE: 3,
            valuE: 4,
            vaLue: 5
        })).toBe("1,2,3,4,5");

        // Negative;
        expect(function () {
            pgp.as.format("$/propName/$(PropName)", {
                propName: undefined
            });
        }).toThrow(new Error("Property 'PropName' doesn't exist."));

    });

    it("must ignore invalid-formatted variables", function () {
        expect(pgp.as.format("$()", {})).toBe("$()");
        expect(pgp.as.format("$((test))", {})).toBe("$((test))");
        expect(pgp.as.format("${{test}}", {})).toBe("${{test}}");
        expect(pgp.as.format("$({test})", {})).toBe("$({test})");
        expect(pgp.as.format("$(^test)", {})).toBe("$(^test)");
        expect(pgp.as.format("${^test}", {})).toBe("${^test}");
        expect(pgp.as.format("$(test^^)", {})).toBe("$(test^^)");
        expect(pgp.as.format("${test^^}", {})).toBe("${test^^}");
    });

    it("must convert all types correctly", function () {
        expect(pgp.as.format("${one},$(two),$[three],$<four>,$/five/", {
            one: undefined,
            two: true,
            three: -123.45,
            four: dateSample,
            five: function () {
                return "text";
            }
        })).toBe("null,true,-123.45,'" + dateSample.toUTCString() + "','text'");
    });

    it("must treat null and undefined values equally", function () {
        // Both null and undefined properties are formatted as null;
        expect(pgp.as.format("${empty1}, $(empty2)", {
            empty1: null,
            empty2: undefined
        })).toBe("null, null");
    });

    it("must throw error when property doesn't exist", function () {
        expect(function () {
            pgp.as.format("${prop1},${prop2}", {
                prop1: 'hello'
            });
        }).toThrow(new Error("Property 'prop2' doesn't exist."));
    });

    describe("'this' formatting", function () {

        it("must recognize 'this'", function () {
            var obj = {
                val1: 123,
                val2: 'hello'
            };
            expect(pgp.as.format("${this}", obj)).toEqual("'" + JSON.stringify(obj) + "'");
        });

        it("must recognize 'this^'", function () {
            var obj = {
                val1: 123,
                val2: 'hello'
            };
            expect(pgp.as.format("${this^}", obj)).toEqual(JSON.stringify(obj));
        });

        it("must ignore 'this' when property exists", function () {
            var obj = {
                this: 'self',
                val1: 123,
                val2: 'hello'
            };
            expect(pgp.as.format("${this^}", obj)).toBe('self');
        });

    });

});

describe("Custom Format", function () {
    function MyType(v) {
        this.value = v;
        this._rawDBType = true;
        this.formatDBType = function () {
            return this.value.toFixed(4);
        }
    }

    var test = new MyType(12.3);
    describe("as array value", function () {
        it("must covert correctly", function () {
            expect(pgp.as.format("$1", [test])).toBe("12.3000");
        });
    });
    describe("as one value", function () {
        it("must covert correctly", function () {
            expect(pgp.as.format("$1", test)).toBe("12.3000");
        });
    });
    describe("for Date override", function () {
        beforeEach(function () {
            Date.prototype.formatDBType = function () {
                function subLevel() {
                    return this.getFullYear();
                }

                return subLevel;
            }
        });
        var today = new Date();
        it("must covert correctly", function () {
            expect(pgp.as.format("$1", today)).toBe(today.getFullYear().toString());
        });
        afterEach(function () {
            delete Date.prototype.formatDBType;
        });
    });
    describe("for Array override", function () {
        beforeEach(function () {
            Array.prototype.formatDBType = function () {
                return new MyType(88); // testing recursive conversion;
            }
        });
        it("must covert correctly", function () {
            expect(pgp.as.format("$1^", [1, 2, 3])).toBe("88.0000");
        });
        afterEach(function () {
            delete Array.prototype.formatDBType;
        });
    });

    describe("with custom object - formatter", function () {
        var values = {
            test: 123
        };

        function CustomFormatter() {
            this.formatDBType = function () {
                return values;
            }
        }

        it("must redirect to named formatting", function () {
            expect(pgp.as.format("${test}", new CustomFormatter)).toBe("123");
        });
    });

});

