/// <reference path="pixi.comments.d.ts" />
/// <reference path="p2.d.ts" />

// Type definitions for Phaser CE
// Project: https://github.com/photonstorm/phaser-ce

declare module "phaser-ce" {
    export = Phaser;
}

declare class Phaser {

    static VERSION: string;
    static DEV_VERSION: string;
    static GAMES: Phaser.Game[];

    static AUTO: number;
    static CANVAS: number;
    static WEBGL: number;
    static HEADLESS: number;
    static WEBGL_MULTI: number;

    static BITMAPDATA: number;
    static BITMAPTEXT: number;
    static BUTTON: number;
    static CANVAS_FILTER: number;
    static CIRCLE: number;
    static ELLIPSE: number;
    static EMITTER: number;
    static GRAPHICS: number;
    static GROUP: number;
    static IMAGE: number;
    static LINE: number;
    static MATRIX: number;
    static POINT: number;
    static POINTER: number;
    static POLYGON: number;
    static RECTANGLE: number;
    static ROUNDEDRECTANGLE: number;
    static RENDERTEXTURE: number;
    static RETROFONT: number;
    static SPRITE: number;
    static SPRITEBATCH: number;
    static TEXT: number;
    static TILEMAP: number;
    static TILEMAPLAYER: number;
    static TILESPRITE: number;
    static WEBGL_FILTER: number;
    static ROPE: number;
    static CREATURE: number;
    static VIDEO: number;

    static NONE: number;
    static LEFT: number;
    static RIGHT: number;
    static UP: number;
    static DOWN: number;

    static HORIZONTAL: number;
    static VERTICAL: number;
    static LANDSCAPE: number;
    static PORTRAIT: number;

    static ANGLE_UP: number;
    static ANGLE_DOWN: number;
    static ANGLE_LEFT: number;
    static ANGLE_RIGHT: number;
    static ANGLE_NORTH_EAST: number;
    static ANGLE_NORTH_WEST: number;
    static ANGLE_SOUTH_EAST: number;
    static ANGLE_SOUTH_WEST: number;

    static TOP_LEFT: number;
    static TOP_CENTER: number;
    static TOP_RIGHT: number;

    static LEFT_TOP: number;
    static LEFT_CENTER: number;
    static LEFT_BOTTOM: number;

    static CENTER: number;

    static RIGHT_TOP: number;
    static RIGHT_CENTER: number;
    static RIGHT_BOTTOM: number;

    static BOTTOM_LEFT: number;
    static BOTTOM_CENTER: number;
    static BOTTOM_RIGHT: number;

}

declare module Phaser {


    /**
    * An Animation instance contains a single animation and the controls to play it.
    * 
    * It is created by the AnimationManager, consists of Animation.Frame objects and belongs to a single Game Object such as a Sprite.
    */
    class Animation {


        /**
        * An Animation instance contains a single animation and the controls to play it.
        * 
        * It is created by the AnimationManager, consists of Animation.Frame objects and belongs to a single Game Object such as a Sprite.
        * 
        * @param game A reference to the currently running game.
        * @param parent A reference to the owner of this Animation.
        * @param name The unique name for this animation, used in playback commands.
        * @param frameData The FrameData object that contains all frames used by this Animation.
        * @param frames An array of numbers or strings indicating which frames to play in which order.
        * @param frameRate The speed at which the animation should play. The speed is given in frames per second. - Default: 60
        * @param loop Whether or not the animation is looped or just plays once.
        */
        constructor(game: Phaser.Game, parent: Phaser.Sprite, name: string, frameData: Phaser.FrameData, frames: number[] | string[], frameRate?: number, loop?: boolean);


        /**
        * The currently displayed frame of the Animation.
        */
        currentFrame: Phaser.Frame;

        /**
        * The delay in ms between each frame of the Animation, based on the given frameRate.
        */
        delay: number;

        /**
        * Gets or sets if this animation will dispatch the onUpdate events upon changing frame.
        */
        enableUpdate: boolean;

        /**
        * Gets or sets the current frame index and updates the Texture Cache for display.
        */
        frame: number;

        /**
        * The total number of frames in the currently loaded FrameData, or -1 if no FrameData is loaded.
        */
        frameTotal: number;

        /**
        * A reference to the currently running Game.
        */
        game: Phaser.Game;

        /**
        * The finished state of the Animation. Set to true once playback completes, false during playback.
        */
        isFinished: boolean;

        /**
        * The paused state of the Animation.
        */
        isPaused: boolean;

        /**
        * The playing state of the Animation. Set to false once playback completes, true during playback.
        */
        isPlaying: boolean;

        /**
        * Should the parent of this Animation be killed when the animation completes?
        */
        killOnComplete: boolean;

        /**
        * The loop state of the Animation.
        */
        loop: boolean;

        /**
        * The number of times the animation has looped since it was last started.
        */
        loopCount: number;

        /**
        * The user defined name given to this Animation.
        */
        name: string;

        /**
        * This event is dispatched when this Animation completes playback. If the animation is set to loop this is never fired, listen for onLoop instead.
        */
        onComplete: Phaser.Signal;

        /**
        * This event is dispatched when this Animation loops.
        */
        onLoop: Phaser.Signal;

        /**
        * This event is dispatched when this Animation starts playback.
        */
        onStart: Phaser.Signal;

        /**
        * This event is dispatched when the Animation changes frame.
        * By default this event is disabled due to its intensive nature. Enable it with: `Animation.enableUpdate = true`.
        * Note that the event is only dispatched with the current frame. In a low-FPS environment Animations
        * will automatically frame-skip to try and claw back time, so do not base your code on expecting to
        * receive a perfectly sequential set of frames from this event.
        */
        onUpdate: Phaser.Signal;

        /**
        * Gets and sets the paused state of this Animation.
        */
        paused: boolean;

        /**
        * Gets and sets the isReversed state of this Animation.
        */
        reversed: boolean;

        /**
        * Gets or sets the current speed of the animation in frames per second. Changing this in a playing animation will take effect from the next frame. Value must be greater than 0.
        */
        speed: number;


        /**
        * Called internally when the animation finishes playback.
        * Sets the isPlaying and isFinished states and dispatches the onAnimationComplete event if it exists on the parent and local onComplete event.
        */
        complete(): void;

        /**
        * Cleans up this animation ready for deletion. Nulls all values and references.
        */
        destroy(): void;

        /**
        * Really handy function for when you are creating arrays of animation data but it's using frame names and not numbers.
        * For example imagine you've got 30 frames named: 'explosion_0001-large' to 'explosion_0030-large'
        * You could use this function to generate those by doing: Phaser.Animation.generateFrameNames('explosion_', 1, 30, '-large', 4);
        * 
        * @param prefix The start of the filename. If the filename was 'explosion_0001-large' the prefix would be 'explosion_'.
        * @param start The number to start sequentially counting from. If your frames are named 'explosion_0001' to 'explosion_0034' the start is 1.
        * @param stop The number to count to. If your frames are named 'explosion_0001' to 'explosion_0034' the stop value is 34.
        * @param suffix The end of the filename. If the filename was 'explosion_0001-large' the prefix would be '-large'. - Default: ''
        * @param zeroPad The number of zeros to pad the min and max values with. If your frames are named 'explosion_0001' to 'explosion_0034' then the zeroPad is 4.
        * @return An array of framenames.
        */
        static generateFrameNames(prefix: string, start: number, stop: number, suffix?: string, zeroPad?: number): string[];

        /**
        * Advances by the given number of frames in the Animation, taking the loop value into consideration.
        * 
        * @param quantity The number of frames to advance. - Default: 1
        */
        next(quantity?: number): void;

        /**
        * Called when the Game enters a paused state.
        */
        onPause(): void;

        /**
        * Called when the Game resumes from a paused state.
        */
        onResume(): void;

        /**
        * Plays this animation.
        * 
        * If you need to jump to a specific frame of this animation, then call `play` and immediately after it,
        * set the frame you require (i.e. `animation.play(); animation.frame = 4`).
        * 
        * @param frameRate The framerate to play the animation at. The speed is given in frames per second. If not provided the previously set frameRate of the Animation is used.
        * @param loop Should the animation be looped after playback. If not provided the previously set loop value of the Animation is used.
        * @param killOnComplete If set to true when the animation completes (only happens if loop=false) the parent Sprite will be killed.
        * @return - A reference to this Animation instance.
        */
        play(frameRate?: number, loop?: boolean, killOnComplete?: boolean): Phaser.Animation;

        /**
        * Moves backwards the given number of frames in the Animation, taking the loop value into consideration.
        * 
        * @param quantity The number of frames to move back. - Default: 1
        */
        previous(quantity?: number): void;

        /**
        * Sets this animation back to the first frame and restarts the animation.
        */
        restart(): void;

        /**
        * Reverses the animation direction.
        * @return The animation instance.
        */
        reverse(): Animation;

        /**
        * Reverses the animation direction for the current/next animation only
        * Once the onComplete event is called this method will be called again and revert
        * the reversed state.
        * @return The animation instance.
        */
        reverseOnce(): Animation;

        /**
        * Sets this animations playback to a given frame with the given ID.
        * 
        * @param frameId The identifier of the frame to set. Can be the name of the frame, the sprite index of the frame, or the animation-local frame index.
        * @param useLocalFrameIndex If you provide a number for frameId, should it use the numeric indexes of the frameData, or the 0-indexed frame index local to the animation.
        */
        setFrame(frameId?: string | number, useLocalFrameIndex?: boolean): void;

        /**
        * Stops playback of this animation and set it to a finished state. If a resetFrame is provided it will stop playback and set frame to the first in the animation.
        * If `dispatchComplete` is true it will dispatch the complete events, otherwise they'll be ignored.
        * 
        * @param resetFrame If true after the animation stops the currentFrame value will be set to the first frame in this animation.
        * @param dispatchComplete Dispatch the Animation.onComplete and parent.onAnimationComplete events?
        */
        stop(resetFrame?: boolean, dispatchComplete?: boolean): void;

        /**
        * Updates this animation. Called automatically by the AnimationManager.
        */
        update(): boolean;

        /**
        * Changes the currentFrame per the _frameIndex, updates the display state,
        * and triggers the update signal.
        * 
        * Returns true if the current frame update was 'successful', false otherwise.
        * 
        * @param signalUpdate If true the `Animation.onUpdate` signal will be dispatched.
        * @param fromPlay Was this call made from the playing of a new animation?
        * @return True if the current frame was updated, otherwise false.
        */
        updateCurrentFrame(signalUpdate: boolean, fromPlay?: boolean): boolean;

        /**
        * Changes the FrameData object this Animation is using.
        * 
        * @param frameData The FrameData object that contains all frames used by this Animation.
        */
        updateFrameData(frameData: FrameData): void;

    }


    /**
    * The Animation Manager is used to add, play and update Phaser Animations.
    * Any Game Object such as Phaser.Sprite that supports animation contains a single AnimationManager instance.
    */
    class AnimationManager {


        /**
        * The Animation Manager is used to add, play and update Phaser Animations.
        * Any Game Object such as Phaser.Sprite that supports animation contains a single AnimationManager instance.
        * 
        * @param sprite A reference to the Game Object that owns this AnimationManager.
        */
        constructor(sprite: Phaser.Sprite);


        /**
        * The currently displayed animation, if any.
        */
        currentAnim: Phaser.Animation;

        /**
        * The currently displayed Frame of animation, if any.
        * This property is only set once an Animation starts playing. Until that point it remains set as `null`.
        */
        currentFrame: Phaser.Frame;

        /**
        * Gets or sets the current frame index and updates the Texture Cache for display.
        */
        frame: number;

        /**
        * The current animations FrameData.
        */
        frameData: Phaser.FrameData;

        /**
        * Gets or sets the current frame name and updates the Texture Cache for display.
        */
        frameName: string;

        /**
        * The total number of frames in the currently loaded FrameData, or -1 if no FrameData is loaded.
        */
        frameTotal: number;

        /**
        * A reference to the currently running Game.
        */
        game: Phaser.Game;

        /**
        * Set to true once animation data has been loaded.
        */
        isLoaded: boolean;

        /**
        * Gets the current animation name, if set.
        */
        name: string;

        /**
        * Gets and sets the paused state of the current animation.
        */
        paused: boolean;

        /**
        * A reference to the parent Sprite that owns this AnimationManager.
        */
        sprite: Phaser.Sprite;

        /**
        * Update the animation data only while the the sprite is {@link Phaser.Sprite#visible}. Set to `false` to continue updating while the sprite is invisible.
        * Default: true
        */
        updateIfVisible: boolean;


        /**
        * Adds a new animation under the given key. Optionally set the frames, frame rate and loop.
        * Animations added in this way are played back with the play function.
        * 
        * @param name The unique (within this Sprite) name for the animation, i.e. "run", "fire", "walk".
        * @param frames An array of numbers/strings that correspond to the frames to add to this animation and in which order. e.g. [1, 2, 3] or ['run0', 'run1', run2]). If null then all frames will be used.
        * @param frameRate The speed at which the animation should play. The speed is given in frames per second. - Default: 60
        * @param loop Whether or not the animation is looped or just plays once.
        * @param useNumericIndex Are the given frames using numeric indexes (default) or strings? - Default: true
        * @return The Animation object that was created.
        */
        add(name: string, frames?: number[] | string[], frameRate?: number, loop?: boolean, useNumericIndex?: boolean): Phaser.Animation;

        /**
        * Loads FrameData into the internal temporary vars and resets the frame index to zero.
        * This is called automatically when a new Sprite is created.
        * 
        * @param frameData The FrameData set to load.
        * @param frame The frame to default to.
        * @return Returns `true` if the frame data was loaded successfully, otherwise `false`
        */
        copyFrameData(frameData: Phaser.FrameData, frame: string | number): boolean;

        /**
        * Destroys all references this AnimationManager contains.
        * Iterates through the list of animations stored in this manager and calls destroy on each of them.
        */
        destroy(): void;

        /**
        * Returns an animation that was previously added by name.
        * 
        * @param name The name of the animation to be returned, e.g. "fire".
        * @return The Animation instance, if found, otherwise null.
        */
        getAnimation(name: string): Phaser.Animation;

        /**
        * Advances by the given number of frames in the current animation, taking the loop value into consideration.
        * 
        * @param quantity The number of frames to advance. - Default: 1
        */
        next(quantity?: number): void;

        /**
        * Play an animation based on the given key. The animation should previously have been added via `animations.add`
        * 
        * If the requested animation is already playing this request will be ignored.
        * If you need to reset an already running animation do so directly on the Animation object itself.
        * 
        * If you need to jump to a specific frame of this animation, then call `play` and immediately after it,
        * set the frame you require (i.e. `animation.play(); animation.frame = 4`).
        * 
        * @param name The name of the animation to be played, e.g. "fire", "walk", "jump".
        * @param frameRate The framerate to play the animation at. The speed is given in frames per second. If not provided the previously set frameRate of the Animation is used.
        * @param loop Should the animation be looped after playback. If not provided the previously set loop value of the Animation is used.
        * @param killOnComplete If set to true when the animation completes (only happens if loop=false) the parent Sprite will be killed.
        * @return A reference to playing Animation instance.
        */
        play(name: string, frameRate?: number, loop?: boolean, killOnComplete?: boolean): Phaser.Animation;

        /**
        * Moves backwards the given number of frames in the current animation, taking the loop value into consideration.
        * 
        * @param quantity The number of frames to move back. - Default: 1
        */
        previous(quantity?: number): void;

        /**
        * Stop playback of an animation. If a name is given that specific animation is stopped, otherwise the current animation is stopped.
        * The currentAnim property of the AnimationManager is automatically set to the animation given.
        * 
        * @param name The name of the animation to be stopped, e.g. "fire". If none is given the currently running animation is stopped.
        * @param resetFrame When the animation is stopped should the currentFrame be set to the first frame of the animation (true) or paused on the last frame displayed (false)
        */
        stop(name?: string, resetFrame?: boolean): void;

        /**
        * The main update function is called by the Sprites update loop. It's responsible for updating animation frames and firing related events.
        * @return True if a new animation frame has been set, otherwise false.
        */
        update(): boolean;

        /**
        * Check whether the frames in the given array are valid and exist.
        * 
        * @param frames An array of frames to be validated.
        * @param useNumericIndex Validate the frames based on their numeric index (true) or string index (false) - Default: true
        * @return True if all given Frames are valid, otherwise false.
        */
        validateFrames(frames: Phaser.Frame[], useNumericIndex?: boolean): boolean;

    }


    /**
    * Responsible for parsing sprite sheet and JSON data into the internal FrameData format that Phaser uses for animations.
    */
    class AnimationParser {


        /**
        * Parse the JSON data and extract the animation frame data from it.
        * 
        * @param game A reference to the currently running game.
        * @param json The JSON data from the Texture Atlas. Must be in Array format.
        * @return A FrameData object containing the parsed frames.
        */
        static JSONData(game: Phaser.Game, json: any): Phaser.FrameData;

        /**
        * Parse the JSON data and extract the animation frame data from it.
        * 
        * @param game A reference to the currently running game.
        * @param json The JSON data from the Texture Atlas. Must be in JSON Hash format.
        * @return A FrameData object containing the parsed frames.
        */
        static JSONDataHash(game: Phaser.Game, json: any): Phaser.FrameData;

        /**
        * Parse the JSON data and extract the animation frame data from it.
        * 
        * @param game A reference to the currently running game.
        * @param json The JSON data from the Texture Atlas. Must be in Pyxel JSON format.
        * @return A FrameData object containing the parsed frames.
        */
        static JSONDataPyxel(game: Phaser.Game, json: any): Phaser.FrameData;

        /**
        * Parse a Sprite Sheet and extract the animation frame data from it.
        * 
        * @param game A reference to the currently running game.
        * @param key The Game.Cache asset key of the Sprite Sheet image or an actual HTML Image element.
        * @param frameWidth The fixed width of each frame of the animation.
        * @param frameHeight The fixed height of each frame of the animation.
        * @param frameMax The total number of animation frames to extract from the Sprite Sheet. The default value of -1 means "extract all frames". - Default: -1
        * @param margin If the frames have been drawn with a margin, specify the amount here.
        * @param spacing If the frames have been drawn with spacing between them, specify the amount here.
        * @param skipFrames Skip a number of frames. Useful when there are multiple sprite sheets in one image.
        * @return A FrameData object containing the parsed frames.
        */
        static spriteSheet(game: Phaser.Game, key: string, frameWidth: number, frameHeight: number, frameMax?: number, margin?: number, spacing?: number, skipFrames?: number): Phaser.FrameData;

        /**
        * Parse the XML data and extract the animation frame data from it.
        * 
        * @param game A reference to the currently running game.
        * @param xml The XML data from the Texture Atlas. Must be in Starling XML format.
        * @return A FrameData object containing the parsed frames.
        */
        static XMLData(game: Phaser.Game, xml: any): Phaser.FrameData;

    }


    /**
    * Audio Sprites are a combination of audio files and a JSON configuration.
    * The JSON follows the format of that created by https://github.com/tonistiigi/audiosprite
    */
    class AudioSprite {


        /**
        * Audio Sprites are a combination of audio files and a JSON configuration.
        * The JSON follows the format of that created by https://github.com/tonistiigi/audiosprite
        * 
        * @param game Reference to the current game instance.
        * @param key Asset key for the sound.
        */
        constructor(game: Phaser.Game, key: string);


        /**
        * A reference to the currently running Game.
        */
        game: Phaser.Game;

        /**
        * Asset key for the Audio Sprite.
        */
        key: string;

        /**
        * JSON audio atlas object.
        */
        config: any;

        /**
        * If a sound is set to auto play, this holds the marker key of it.
        */
        autoplayKey: string;

        /**
        * Is a sound set to autoplay or not?
        */
        autoplay: boolean;

        /**
        * An object containing the Phaser.Sound objects for the Audio Sprite.
        */
        sounds: any;


        /**
        * Get a sound with the given name.
        * 
        * @param marker The name of sound to get.
        * @return The sound instance.
        */
        get(marker: string): Phaser.Sound;

        /**
        * Play a sound with the given name.
        * 
        * @param marker The name of sound to play
        * @param volume Volume of the sound you want to play. If none is given it will use the volume given to the Sound when it was created (which defaults to 1 if none was specified). - Default: 1
        * @return This sound instance.
        */
        play(marker: string, volume?: number): Phaser.Sound;

        /**
        * Stop a sound with the given name.
        * 
        * @param marker The name of sound to stop. If none is given it will stop all sounds in the audio sprite. - Default: ''
        */
        stop(marker: string): Phaser.Sound;

    }


    /**
    * ArraySet is a Set data structure (items must be unique within the set) that also maintains order.
    * This allows specific items to be easily added or removed from the Set.
    * 
    * Item equality (and uniqueness) is determined by the behavior of `Array.indexOf`.
    * 
    * This used primarily by the Input subsystem.
    */
    class ArraySet {


        /**
        * ArraySet is a Set data structure (items must be unique within the set) that also maintains order.
        * This allows specific items to be easily added or removed from the Set.
        * 
        * Item equality (and uniqueness) is determined by the behavior of `Array.indexOf`.
        * 
        * This used primarily by the Input subsystem.
        * 
        * @param list The backing array: if specified the items in the list _must_ be unique, per `Array.indexOf`, and the ownership of the array _should_ be relinquished to the ArraySet. - Default: (new array)
        */
        constructor(list: any[]);


        /**
        * Current cursor position as established by `first` and `next`.
        */
        position: number;

        /**
        * The backing array.
        */
        list: any[];

        /**
        * Number of items in the ArraySet. Same as `list.length`.
        */
        total: number;

        /**
        * Returns the first item and resets the cursor to the start.
        */
        first: any;

        /**
        * Returns the the next item (based on the cursor) and advances the cursor.
        */
        next: any;


        /**
        * Adds a new element to the end of the list.
        * If the item already exists in the list it is not moved.
        * 
        * @param item The element to add to this list.
        * @return The item that was added.
        */
        add(item: any): any;

        /**
        * Gets an item from the set based on the property strictly equaling the value given.
        * Returns null if not found.
        * 
        * @param property The property to check against the value.
        * @param value The value to check if the property strictly equals.
        * @return The item that was found, or null if nothing matched.
        */
        getByKey(property: string, value: any): any;

        /**
        * Gets the index of the item in the list, or -1 if it isn't in the list.
        * 
        * @param item The element to get the list index for.
        * @return The index of the item or -1 if not found.
        */
        getIndex(item: any): number;

        /**
        * Checks for the item within this list.
        * 
        * @param item The element to get the list index for.
        * @return True if the item is found in the list, otherwise false.
        */
        exists(item: any): boolean;

        /**
        * Removes all the items.
        */
        reset(): void;

        /**
        * Removes the given element from this list if it exists.
        * 
        * @param item The item to be removed from the list.
        * @return item - The item that was removed.
        */
        remove(item: any): any;

        /**
        * Removes every member from this ArraySet and optionally destroys it.
        * 
        * @param destroy Call `destroy` on each member as it's removed from this set.
        */
        removeAll(destoy?: boolean): void;

        /**
        * Sets the property `key` to the given value on all members of this list.
        * 
        * @param key The property of the item to set.
        * @param value The value to set the property to.
        */
        setAll(key: any, value: any): void;

        /**
        * Calls a function on all members of this list, using the member as the context for the callback.
        * 
        * If the `key` property is present it must be a function.
        * The function is invoked using the item as the context.
        * 
        * @param key The name of the property with the function to call.
        * @param args Additional parameters that will be passed to the callback.
        */
        callAll(key: string, ...parameter: any[]): void;

    }


    /**
    * Utility functions for dealing with Arrays.
    */
    class ArrayUtils {


        /**
        * Fetch a random entry from the given array.
        * 
        * Will return null if there are no array items that fall within the specified range
        * or if there is no item for the randomly chosen index.
        * 
        * @param objects An array of objects.
        * @param startIndex Optional offset off the front of the array. Default value is 0, or the beginning of the array.
        * @param length Optional restriction on the number of values you want to randomly select from.
        * @return The random object that was selected.
        */
        static getRandomItem<T>(objects: T[], startIndex?: number, length?: number): T;

        /**
        * Removes a random object from the given array and returns it.
        * 
        * Will return null if there are no array items that fall within the specified range
        * or if there is no item for the randomly chosen index.
        * 
        * @param objects An array of objects.
        * @param startIndex Optional offset off the front of the array. Default value is 0, or the beginning of the array.
        * @param length Optional restriction on the number of values you want to randomly select from.
        * @return The random object that was removed.
        */
        static removeRandomItem<T>(objects: T[], startIndex?: number, length?: number): T;

        /**
        * A standard Fisher-Yates Array shuffle implementation which modifies the array in place.
        * 
        * @param array The array to shuffle.
        * @return The original array, now shuffled.
        */
        static shuffle<T>(array: T[]): T[];

        /**
        * Transposes the elements of the given matrix (array of arrays).
        * 
        * @param array The matrix to transpose.
        * @return A new transposed matrix
        */
        static transposeMatrix<T>(array: T[]): T;

        /**
        * Rotates the given matrix (array of arrays).
        * 
        * Based on the routine from {@link http://jsfiddle.net/MrPolywhirl/NH42z/}.
        * 
        * @param matrix The array to rotate; this matrix _may_ be altered.
        * @param direction The amount to rotate: the rotation in degrees (90, -90, 270, -270, 180) or a string command ('rotateLeft', 'rotateRight' or 'rotate180').
        * @return The rotated matrix. The source matrix should be discarded for the returned matrix.
        */
        static rotateMatrix(matrix: any, direction: number | string): any;

        /**
        * Snaps a value to the nearest value in an array.
        * The result will always be in the range `[first_value, last_value]`.
        * 
        * @param value The search value
        * @param arr The input array which _must_ be sorted.
        * @return The nearest value found.
        */
        static findClosest(value: number, arr: number[]): number;

        /**
        * Moves the element from the start of the array to the end, shifting all items in the process.
        * The "rotation" happens to the left.
        * 
        * Before: `[ A, B, C, D, E, F ]`
        * After: `[ B, C, D, E, F, A ]`
        * 
        * See also Phaser.ArrayUtils.rotateRight
        * 
        * @param array The array to rotate. The array is modified.
        * @return The rotated value.
        */
        static rotate(array: any[]): any;

        /**
        * Moves the element from the start of the array to the end, shifting all items in the process.
        * The "rotation" happens to the left.
        * 
        * Before: `[ A, B, C, D, E, F ]`
        * After: `[ B, C, D, E, F, A ]`
        * 
        * See also Phaser.ArrayUtils.rotateRight
        * 
        * @param array The array to rotate. The array is modified.
        * @return The rotated value.
        */
        static rotateLeft(array: any[]): any;

        /**
        * Moves the element from the end of the array to the start, shifting all items in the process.
        * The "rotation" happens to the right.
        * 
        * Before: `[ A, B, C, D, E, F ]`
        * After: `[ F, A, B, C, D, E ]`
        * 
        * See also Phaser.ArrayUtils.rotateLeft.
        * 
        * @param array The array to rotate. The array is modified.
        * @return The shifted value.
        */
        static rotateRight(array: any[]): any;

        /**
        * Create an array representing the inclusive range of numbers (usually integers) in `[start, end]`.
        * This is equivalent to `numberArrayStep(start, end, 1)`.
        * 
        * @param start The minimum value the array starts with.
        * @param end The maximum value the array contains.
        * @return The array of number values.
        */
        static numberArray(start: number, end: number): number[];

        /**
        * Create an array of numbers (positive and/or negative) progressing from `start`
        * up to but not including `end` by advancing by `step`.
        * 
        * If `start` is less than `end` a zero-length range is created unless a negative `step` is specified.
        * 
        * Certain values for `start` and `end` (eg. NaN/undefined/null) are currently coerced to 0;
        * for forward compatibility make sure to pass in actual numbers.
        * 
        * @param start The start of the range.
        * @param end The end of the range.
        * @param step The value to increment or decrement by. - Default: 1
        * @return Returns the new array of numbers.
        */
        static numberArrayStep(start: number, end?: number, step?: number): number[];

    }

    interface BitmapFont {

        base: PIXI.BaseTexture;
        data: HTMLImageElement;
        font: Phaser.BMFont;
        url: string;

    }

    interface BMFont {

        chars: Phaser.BMFontChar[];
        font: string;
        lineHeight: number;
        size: number;

    }

    interface BMFontChar {

        x: number;
        y: number;
        width: number;
        height: number;
        xOffset: number;
        yOffset: number;
        xAdvance: number;
        kerning: number[];
        texture: PIXI.BaseTexture;

    }


    /**
    * A BitmapData object contains a Canvas element to which you can draw anything you like via normal Canvas context operations.
    * A single BitmapData can be used as the texture for one or many Images / Sprites.
    * So if you need to dynamically create a Sprite texture then they are a good choice.
    * 
    * Important note: Every BitmapData creates its own Canvas element. Because BitmapData's are now Game Objects themselves, and don't
    * live on the display list, they are NOT automatically cleared when you change State. Therefore you _must_ call BitmapData.destroy
    * in your State's shutdown method if you wish to free-up the resources the BitmapData used, it will not happen for you.
    */
    class BitmapData {


        /**
        * A BitmapData object contains a Canvas element to which you can draw anything you like via normal Canvas context operations.
        * A single BitmapData can be used as the texture for one or many Images / Sprites.
        * So if you need to dynamically create a Sprite texture then they are a good choice.
        * 
        * Important note: Every BitmapData creates its own Canvas element. Because BitmapData's are now Game Objects themselves, and don't
        * live on the display list, they are NOT automatically cleared when you change State. Therefore you _must_ call BitmapData.destroy
        * in your State's shutdown method if you wish to free-up the resources the BitmapData used, it will not happen for you.
        * 
        * @param game A reference to the currently running game.
        * @param key Internal Phaser reference key for the BitmapData.
        * @param width The width of the BitmapData in pixels. If undefined or zero it's set to a default value. - Default: 256
        * @param height The height of the BitmapData in pixels. If undefined or zero it's set to a default value. - Default: 256
        * @param skipPool When this BitmapData generates its internal canvas to use for rendering, it will get the canvas from the CanvasPool if false, or create its own if true.
        */
        constructor(game: Phaser.Game, key: string, width?: number, height?: number, skipPool?: boolean);


        /**
        * The PIXI.BaseTexture.
        */
        baseTexture: PIXI.BaseTexture;
        buffer: ArrayBuffer;

        /**
        * The canvas to which this BitmapData draws.
        */
        canvas: HTMLCanvasElement;

        /**
        * The 2d context of the canvas.
        */
        context: CanvasRenderingContext2D;

        /**
        * A reference to BitmapData.context.
        */
        ctx: CanvasRenderingContext2D;

        /**
        * A Uint8ClampedArray view into BitmapData.buffer.
        * Note that this is unavailable in some browsers (such as Epic Browser due to its security restrictions)
        */
        data: Uint8Array;

        /**
        * If dirty this BitmapData will be re-rendered.
        */
        dirty: boolean;

        /**
        * If disableTextureUpload is true this BitmapData will never send its image data to the GPU when its dirty flag is true.
        */
        disableTextureUpload: boolean;

        /**
        * A reference to the currently running game.
        */
        game: Phaser.Game;

        /**
        * The height of the BitmapData in pixels.
        */
        height: number;

        /**
        * The context image data.
        * Please note that a call to BitmapData.draw() or BitmapData.copy() does not update immediately this property for performance reason. Use BitmapData.update() to do so.
        * This property is updated automatically after the first game loop, according to the dirty flag property.
        */
        imageData: ImageData;

        /**
        * The key of the BitmapData in the Cache, if stored there.
        */
        key: string;
        op: string;

        /**
        * An Uint32Array view into BitmapData.buffer.
        */
        pixels: Uint32Array;
        smoothed: boolean;

        /**
        * The context property needed for smoothing this Canvas.
        */
        smoothProperty: string;

        /**
        * The PIXI.Texture.
        */
        texture: PIXI.Texture;

        /**
        * The Frame this BitmapData uses for rendering.
        */
        textureFrame: Phaser.Frame;

        /**
        * The const type of this object.
        */
        type: number;

        /**
        * The width of the BitmapData in pixels.
        */
        width: number;


        /**
        * Gets a JavaScript object that has 6 properties set that are used by BitmapData in a transform.
        * 
        * @param translateX The x translate value.
        * @param translateY The y translate value.
        * @param scaleX The scale x value.
        * @param scaleY The scale y value.
        * @param skewX The skew x value.
        * @param skewY The skew y value.
        * @return A JavaScript object containing all of the properties BitmapData needs for transforms.
        */
        static getTransform(translateX: number, translateY: number, scaleX: number, scaleY: number, skewX: number, skewY: number): any;


        /**
        * Updates the given objects so that they use this BitmapData as their texture.
        * This will replace any texture they will currently have set.
        * 
        * @param object Either a single Sprite/Image or an Array of Sprites/Images.
        * @return This BitmapData object for method chaining.
        */
        add(object: any): Phaser.BitmapData;

        /**
        * Creates a new Phaser.Image object, assigns this BitmapData to be its texture, adds it to the world then returns it.
        * 
        * @param x The x coordinate to place the Image at.
        * @param y The y coordinate to place the Image at.
        * @param anchorX Set the x anchor point of the Image. A value between 0 and 1, where 0 is the top-left and 1 is bottom-right.
        * @param anchorY Set the y anchor point of the Image. A value between 0 and 1, where 0 is the top-left and 1 is bottom-right.
        * @param scaleX The horizontal scale factor of the Image. A value of 1 means no scaling. 2 would be twice the size, and so on. - Default: 1
        * @param scaleY The vertical scale factor of the Image. A value of 1 means no scaling. 2 would be twice the size, and so on. - Default: 1
        * @return The newly added Image object.
        */
        addToWorld(x?: number, y?: number, anchorX?: number, anchorY?: number, scaleX?: number, scaleY?: number): Phaser.Image;

        /**
        * Draws the image onto this BitmapData using an image as an alpha mask.
        * 
        * @param source The source to copy from. If you give a string it will try and find the Image in the Game.Cache first. This is quite expensive so try to provide the image itself.
        * @param mask The object to be used as the mask. If you give a string it will try and find the Image in the Game.Cache first. This is quite expensive so try to provide the image itself. If you don't provide a mask it will use this BitmapData as the mask.
        * @param sourceRect A Rectangle where x/y define the coordinates to draw the Source image to and width/height define the size.
        * @param maskRect A Rectangle where x/y define the coordinates to draw the Mask image to and width/height define the size.
        * @return This BitmapData object for method chaining.
        */
        alphaMask(source: any, mask?: any, sourceRect?: Phaser.Rectangle, maskRect?: Phaser.Rectangle): Phaser.BitmapData;

        /**
        * Sets the blend mode to 'lighter'
        * @return This BitmapData object for method chaining.
        */
        blendAdd(): Phaser.BitmapData;

        /**
        * Sets the blend mode to 'color'
        * @return This BitmapData object for method chaining.
        */
        blendColor(): Phaser.BitmapData;

        /**
        * Sets the blend mode to 'color-burn'
        * @return This BitmapData object for method chaining.
        */
        blendColorBurn(): Phaser.BitmapData;

        /**
        * Sets the blend mode to 'color-dodge'
        * @return This BitmapData object for method chaining.
        */
        blendColorDodge(): Phaser.BitmapData;

        /**
        * Sets the blend mode to 'darken'
        * @return This BitmapData object for method chaining.
        */
        blendDarken(): Phaser.BitmapData;

        /**
        * Sets the blend mode to 'destination-atop'
        * @return This BitmapData object for method chaining.
        */
        blendDestinationAtop(): Phaser.BitmapData;

        /**
        * Sets the blend mode to 'destination-in'
        * @return This BitmapData object for method chaining.
        */
        blendDestinationIn(): Phaser.BitmapData;

        /**
        * Sets the blend mode to 'destination-out'
        * @return This BitmapData object for method chaining.
        */
        blendDestinationOut(): Phaser.BitmapData;

        /**
        * Sets the blend mode to 'destination-over'
        * @return This BitmapData object for method chaining.
        */
        blendDestinationOver(): Phaser.BitmapData;

        /**
        * Sets the blend mode to 'difference'
        * @return This BitmapData object for method chaining.
        */
        blendDifference(): Phaser.BitmapData;

        /**
        * Sets the blend mode to 'exclusion'
        * @return This BitmapData object for method chaining.
        */
        blendExclusion(): Phaser.BitmapData;

        /**
        * Sets the blend mode to 'hard-light'
        * @return This BitmapData object for method chaining.
        */
        blendHardLight(): Phaser.BitmapData;

        /**
        * Sets the blend mode to 'hue'
        * @return This BitmapData object for method chaining.
        */
        blendHue(): Phaser.BitmapData;

        /**
        * Sets the blend mode to 'lighten'
        * @return This BitmapData object for method chaining.
        */
        blendLighten(): Phaser.BitmapData;

        /**
        * Sets the blend mode to 'luminosity'
        * @return This BitmapData object for method chaining.
        */
        blendLuminosity(): Phaser.BitmapData;

        /**
        * Sets the blend mode to 'multiply'
        * @return This BitmapData object for method chaining.
        */
        blendMultiply(): Phaser.BitmapData;

        /**
        * Sets the blend mode to 'overlay'
        * @return This BitmapData object for method chaining.
        */
        blendOverlay(): Phaser.BitmapData;

        /**
        * Resets the blend mode (effectively sets it to 'source-over')
        * @return This BitmapData object for method chaining.
        */
        blendReset(): Phaser.BitmapData;

        /**
        * Sets the blend mode to 'saturation'
        * @return This BitmapData object for method chaining.
        */
        blendSaturation(): Phaser.BitmapData;

        /**
        * Sets the blend mode to 'screen'
        * @return This BitmapData object for method chaining.
        */
        blendScreen(): Phaser.BitmapData;

        /**
        * Sets the blend mode to 'soft-light'
        * @return This BitmapData object for method chaining.
        */
        blendSoftLight(): Phaser.BitmapData;

        /**
        * Sets the blend mode to 'source-atop'
        * @return This BitmapData object for method chaining.
        */
        blendSourceAtop(): Phaser.BitmapData;

        /**
        * Sets the blend mode to 'source-in'
        * @return This BitmapData object for method chaining.
        */
        blendSourceIn(): Phaser.BitmapData;

        /**
        * Sets the blend mode to 'source-out'
        * @return This BitmapData object for method chaining.
        */
        blendSourceOut(): Phaser.BitmapData;

        /**
        * Sets the blend mode to 'source-over'
        * @return This BitmapData object for method chaining.
        */
        blendSourceOver(): Phaser.BitmapData;

        /**
        * Sets the blend mode to 'xor'
        * @return This BitmapData object for method chaining.
        */
        blendXor(): Phaser.BitmapData;

        /**
        * Draws a filled Circle to the BitmapData at the given x, y coordinates and radius in size.
        * 
        * @param x The x coordinate to draw the Circle at. This is the center of the circle.
        * @param y The y coordinate to draw the Circle at. This is the center of the circle.
        * @param radius The radius of the Circle in pixels. The radius is half the diameter.
        * @param fillStyle If set the context fillStyle will be set to this value before the circle is drawn.
        * @return This BitmapData object for method chaining.
        */
        circle(x: number, y: number, radius: number, fillStyle?: string): Phaser.BitmapData;

        /**
        * Clears the BitmapData context using a clearRect.
        * 
        * You can optionally define the area to clear.
        * If the arguments are left empty it will clear the entire canvas.
        * 
        * You may need to call BitmapData.update after this in order to clear out the pixel data,
        * but Phaser will not do this automatically for you.
        * 
        * @param x The x coordinate of the top-left of the area to clear.
        * @param y The y coordinate of the top-left of the area to clear.
        * @param width The width of the area to clear. If undefined it will use BitmapData.width.
        * @param height The height of the area to clear. If undefined it will use BitmapData.height.
        * @return This BitmapData object for method chaining.
        */
        clear(x?: number, y?: number, width?: number, height?: number): Phaser.BitmapData;

        /**
        * Clears the BitmapData context using a clearRect.
        */
        cls(): Phaser.BitmapData;

        /**
        * Copies a rectangular area from the source object to this BitmapData. If you give `null` as the source it will copy from itself.
        * 
        * You can optionally resize, translate, rotate, scale, alpha or blend as it's drawn.
        * 
        * All rotation, scaling and drawing takes place around the regions center point by default, but can be changed with the anchor parameters.
        * 
        * Note that the source image can also be this BitmapData, which can create some interesting effects.
        * 
        * This method has a lot of parameters for maximum control.
        * You can use the more friendly methods like `copyRect` and `draw` to avoid having to remember them all.
        * 
        * You may prefer to use `copyTransform` if you're simply trying to draw a Sprite to this BitmapData,
        * and don't wish to translate, scale or rotate it from its original values.
        * 
        * @param source The source to copy from. If you give a string it will try and find the Image in the Game.Cache first. This is quite expensive so try to provide the image itself.
        * @param x The x coordinate representing the top-left of the region to copy from the source image.
        * @param y The y coordinate representing the top-left of the region to copy from the source image.
        * @param width The width of the region to copy from the source image. If not specified it will use the full source image width.
        * @param height The height of the region to copy from the source image. If not specified it will use the full source image height.
        * @param tx The x coordinate to translate to before drawing. If not specified it will default to the `x` parameter. If `null` and `source` is a Display Object, it will default to `source.x`.
        * @param ty The y coordinate to translate to before drawing. If not specified it will default to the `y` parameter. If `null` and `source` is a Display Object, it will default to `source.y`.
        * @param newWidth The new width of the block being copied. If not specified it will default to the `width` parameter.
        * @param newHeight The new height of the block being copied. If not specified it will default to the `height` parameter.
        * @param rotate The angle in radians to rotate the block to before drawing. Rotation takes place around the center by default, but can be changed with the `anchor` parameters.
        * @param anchorX The anchor point around which the block is rotated and scaled. A value between 0 and 1, where 0 is the top-left and 1 is bottom-right.
        * @param anchorY The anchor point around which the block is rotated and scaled. A value between 0 and 1, where 0 is the top-left and 1 is bottom-right.
        * @param scaleX The horizontal scale factor of the block. A value of 1 means no scaling. 2 would be twice the size, and so on. - Default: 1
        * @param scaleY The vertical scale factor of the block. A value of 1 means no scaling. 2 would be twice the size, and so on. - Default: 1
        * @param alpha The alpha that will be set on the context before drawing. A value between 0 (fully transparent) and 1, opaque. - Default: 1
        * @param blendMode The composite blend mode that will be used when drawing. The default is no blend mode at all. This is a Canvas globalCompositeOperation value such as 'lighter' or 'xor'.
        * @param roundPx Should the x and y values be rounded to integers before drawing? This prevents anti-aliasing in some instances.
        * @return This BitmapData object for method chaining.
        */
        copy(source?: any, x?: number, y?: number, width?: number, height?: number, tx?: number, ty?: number, newWidth?: number, newHeight?: number, rotate?: number, anchorX?: number, anchorY?: number, scaleX?: number, scaleY?: number, alpha?: number, blendMode?: string, roundPx?: boolean): Phaser.BitmapData;
        copyPixels(source: any, area: Phaser.Rectangle, x: number, y: number, alpha?: number): void;

        /**
        * Copies the area defined by the Rectangle parameter from the source image to this BitmapData at the given location.
        * 
        * @param source The Image to copy from. If you give a string it will try and find the Image in the Game.Cache.
        * @param area The Rectangle region to copy from the source image.
        * @param x The destination x coordinate to copy the image to.
        * @param y The destination y coordinate to copy the image to.
        * @param alpha The alpha that will be set on the context before drawing. A value between 0 (fully transparent) and 1, opaque. - Default: 1
        * @param blendMode The composite blend mode that will be used when drawing. The default is no blend mode at all. This is a Canvas globalCompositeOperation value such as 'lighter' or 'xor'.
        * @param roundPx Should the x and y values be rounded to integers before drawing? This prevents anti-aliasing in some instances.
        * @return This BitmapData object for method chaining.
        */
        copyRect(source: any, area: Phaser.Rectangle, x?: number, y?: number, alpha?: number, blendMode?: string, roundPx?: boolean): Phaser.BitmapData;

        /**
        * Draws the given `source` Game Object to this BitmapData, using its `worldTransform` property to set the
        * position, scale and rotation of where it is drawn. This function is used internally by `drawGroup`.
        * It takes the objects tint and scale mode into consideration before drawing.
        * 
        * You can optionally specify Blend Mode and Round Pixels arguments.
        * 
        * @param source The Game Object to draw.
        * @param blendMode The composite blend mode that will be used when drawing. The default is no blend mode at all. This is a Canvas globalCompositeOperation value such as 'lighter' or 'xor'.
        * @param roundPx Should the x and y values be rounded to integers before drawing? This prevents anti-aliasing in some instances.
        * @return This BitmapData object for method chaining.
        */
        copyTransform(source: any, blendMode?: string, roundPx?: boolean): Phaser.BitmapData;

        /**
        * Destroys this BitmapData and puts the canvas it was using back into the canvas pool for re-use.
        */
        destroy(): void;

        /**
        * Draws the given Phaser.Sprite, Phaser.Image or Phaser.Text to this BitmapData at the coordinates specified.
        * You can use the optional width and height values to 'stretch' the sprite as it is drawn. This uses drawImage stretching, not scaling.
        * 
        * The children will be drawn at their `x` and `y` world space coordinates. If this is outside the bounds of the BitmapData they won't be visible.
        * When drawing it will take into account the rotation, scale, scaleMode, alpha and tint values.
        * 
        * Note: You should ensure that at least 1 full update has taken place before calling this,
        * otherwise the objects are likely to render incorrectly, if at all.
        * You can trigger an update yourself by calling `stage.updateTransform()` before calling `draw`.
        * 
        * @param source The Sprite, Image or Text object to draw onto this BitmapData.
        * @param x The x coordinate to translate to before drawing. If not specified it will default to `source.x`.
        * @param y The y coordinate to translate to before drawing. If not specified it will default to `source.y`.
        * @param width The new width of the Sprite being copied. If not specified it will default to `source.width`.
        * @param height The new height of the Sprite being copied. If not specified it will default to `source.height`.
        * @param blendMode The composite blend mode that will be used when drawing. The default is no blend mode at all. This is a Canvas globalCompositeOperation value such as 'lighter' or 'xor'.
        * @param roundPx Should the x and y values be rounded to integers before drawing? This prevents anti-aliasing in some instances.
        * @return This BitmapData object for method chaining.
        */
        draw(source: any, x?: number, y?: number, width?: number, height?: number, blendMode?: string, roundPx?: boolean): Phaser.BitmapData;

        /**
        * Draws the Game Object or Group to this BitmapData and then recursively iterates through all of its children.
        * 
        * If a child has an `exists` property then it (and its children) will be only be drawn if exists is `true`.
        * 
        * The children will be drawn at their `x` and `y` world space coordinates. If this is outside the bounds of the BitmapData
        * they won't be drawn. Depending on your requirements you may need to resize the BitmapData in advance to match the
        * bounds of the top-level Game Object.
        * 
        * When drawing it will take into account the child's world rotation, scale and alpha values.
        * 
        * It's perfectly valid to pass in `game.world` as the parent object, and it will iterate through the entire display list.
        * 
        * Note: If you are trying to grab your entire game at the start of a State then you should ensure that at least 1 full update
        * has taken place before doing so, otherwise all of the objects will render with incorrect positions and scales. You can
        * trigger an update yourself by calling `stage.updateTransform()` before calling `drawFull`.
        * 
        * @param parent The Game Object to draw onto this BitmapData and then recursively draw all of its children.
        * @param blendMode The composite blend mode that will be used when drawing. The default is no blend mode at all. This is a Canvas globalCompositeOperation value such as 'lighter' or 'xor'.
        * @param roundPx Should the x and y values be rounded to integers before drawing? This prevents anti-aliasing in some instances.
        * @return This BitmapData object for method chaining.
        */
        drawFull(parent: any, blendMode?: string, roundPx?: boolean): Phaser.BitmapData;

        /**
        * Draws the immediate children of a Phaser.Group to this BitmapData.
        * 
        * It's perfectly valid to pass in `game.world` as the Group, and it will iterate through the entire display list.
        * 
        * Children are drawn _only_ if they have their `exists` property set to `true`, and have image, or RenderTexture, based Textures.
        * 
        * The children will be drawn at their `x` and `y` world space coordinates. If this is outside the bounds of the BitmapData they won't be visible.
        * When drawing it will take into account the rotation, scale, scaleMode, alpha and tint values.
        * 
        * Note: You should ensure that at least 1 full update has taken place before calling this,
        * otherwise the objects are likely to render incorrectly, if at all.
        * You can trigger an update yourself by calling `stage.updateTransform()` before calling `drawGroup`.
        * 
        * @param group The Group to draw onto this BitmapData. Can also be Phaser.World.
        * @param blendMode The composite blend mode that will be used when drawing. The default is no blend mode at all. This is a Canvas globalCompositeOperation value such as 'lighter' or 'xor'.
        * @param roundPx Should the x and y values be rounded to integers before drawing? This prevents anti-aliasing in some instances.
        * @return This BitmapData object for method chaining.
        */
        drawGroup(group: Phaser.Group, blendMode?: string, roundPx?: boolean): Phaser.BitmapData;

        /**
        * Scans this BitmapData for all pixels matching the given r,g,b values and then draws them into the given destination BitmapData.
        * The original BitmapData remains unchanged.
        * The destination BitmapData must be large enough to receive all of the pixels that are scanned unless the 'resize' parameter is true.
        * Although the destination BitmapData is returned from this method, it's actually modified directly in place, meaning this call is perfectly valid:
        * `picture.extract(mask, r, g, b)`
        * You can specify optional r2, g2, b2 color values. If given the pixel written to the destination bitmap will be of the r2, g2, b2 color.
        * If not given it will be written as the same color it was extracted. You can provide one or more alternative colors, allowing you to tint
        * the color during extraction.
        * 
        * @param destination The BitmapData that the extracted pixels will be drawn to.
        * @param r The red color component, in the range 0 - 255.
        * @param g The green color component, in the range 0 - 255.
        * @param b The blue color component, in the range 0 - 255.
        * @param a The alpha color component, in the range 0 - 255 that the new pixel will be drawn at. - Default: 255
        * @param resize Should the destination BitmapData be resized to match this one before the pixels are copied?
        * @param r2 An alternative red color component to be written to the destination, in the range 0 - 255.
        * @param g2 An alternative green color component to be written to the destination, in the range 0 - 255.
        * @param b2 An alternative blue color component to be written to the destination, in the range 0 - 255.
        * @return The BitmapData that the extract pixels were drawn on.
        */
        extract(destination: Phaser.BitmapData, r: number, g: number, b: number, a?: number, resize?: boolean, r2?: number, g2?: number, b2?: number): Phaser.BitmapData;

        /**
        * Fills the BitmapData with the given color.
        * 
        * @param r The red color value, between 0 and 0xFF (255).
        * @param g The green color value, between 0 and 0xFF (255).
        * @param b The blue color value, between 0 and 0xFF (255).
        * @param a The alpha color value, between 0 and 1. - Default: 1
        * @return This BitmapData object for method chaining.
        */
        fill(r: number, g: number, b: number, a?: number): Phaser.BitmapData;

        /**
        * Creates a new Image element by converting this BitmapDatas canvas into a dataURL.
        * 
        * The image is then stored in the image Cache using the key given.
        * 
        * Finally a PIXI.Texture is created based on the image and returned.
        * 
        * You can apply the texture to a sprite or any other supporting object by using either the
        * key or the texture. First call generateTexture:
        * 
        * `var texture = bitmapdata.generateTexture('ball');`
        * 
        * Then you can either apply the texture to a sprite:
        * 
        * `game.add.sprite(0, 0, texture);`
        * 
        * or by using the string based key:
        * 
        * `game.add.sprite(0, 0, 'ball');`
        * 
        * Most browsers now load the image data asynchronously, so you should use a callback:
        * 
        * ```
        * bitmapdata.generateTexture('ball', function (texture) {
        *     game.add.sprite(0, 0, texture);
        *     // or
        *     game.add.sprite(0, 0, 'ball');
        * });
        * ```
        * 
        * If this BitmapData is available during preload, you can use {@link Phaser.Loader#imageFromBitmapData} instead.
        * 
        * @param key The key which will be used to store the image in the Cache.
        * @param callback A function to execute once the texture is generated. It will be passed the newly generated texture.
        * @param callbackContext The context in which to invoke the callback.
        * @return The newly generated texture, or `null` if a callback was passed and the texture isn't available yet.
        */
        generateTexture(key: string, callback?: (texture: PIXI.Texture) => void, callbackContext?: any): PIXI.Texture;

        /**
        * Scans the BitmapData and calculates the bounds. This is a rectangle that defines the extent of all non-transparent pixels.
        * The rectangle returned will extend from the top-left of the image to the bottom-right, excluding transparent pixels.
        * 
        * @param rect If provided this Rectangle object will be populated with the bounds, otherwise a new object will be created.
        * @return A Rectangle whose dimensions encompass the full extent of non-transparent pixels in this BitmapData.
        */
        getBounds(rect?: Phaser.Rectangle): Phaser.Rectangle;

        /**
        * Scans the BitmapData, pixel by pixel, until it encounters a pixel that isn't transparent (i.e. has an alpha value > 0).
        * It then stops scanning and returns an object containing the color of the pixel in r, g and b properties and the location in the x and y properties.
        * 
        * The direction parameter controls from which direction it should start the scan:
        * 
        * 0 = top to bottom
        * 1 = bottom to top
        * 2 = left to right
        * 3 = right to left
        * 
        * @param direction The direction in which to scan for the first pixel. 0 = top to bottom, 1 = bottom to top, 2 = left to right and 3 = right to left.
        * @return Returns an object containing the color of the pixel in the `r`, `g` and `b` properties and the location in the `x` and `y` properties.
        */
        getFirstPixel(direction: number): { r: number; g: number; b: number; x: number; y: number; };

        /**
        * Get the color of a specific pixel in the context into a color object.
        * If you have drawn anything to the BitmapData since it was created you must call BitmapData.update to refresh the array buffer,
        * otherwise this may return out of date color values, or worse - throw a run-time error as it tries to access an array element that doesn't exist.
        * 
        * @param x The x coordinate of the pixel to be set. Must lay within the dimensions of this BitmapData and be an integer, not a float.
        * @param y The y coordinate of the pixel to be set. Must lay within the dimensions of this BitmapData and be an integer, not a float.
        * @param out An object into which 4 properties will be created: r, g, b and a. If not provided a new object will be created.
        * @return An object with the red, green, blue and alpha values set in the r, g, b and a properties.
        */
        getPixel(x: number, y: number, out?: any): any;

        /**
        * Get the color of a specific pixel including its alpha value as a color object containing r,g,b,a and rgba properties.
        * If you have drawn anything to the BitmapData since it was created you must call BitmapData.update to refresh the array buffer,
        * otherwise this may return out of date color values, or worse - throw a run-time error as it tries to access an array element that doesn't exist.
        * 
        * @param x The x coordinate of the pixel to be set. Must lay within the dimensions of this BitmapData and be an integer, not a float.
        * @param y The y coordinate of the pixel to be set. Must lay within the dimensions of this BitmapData and be an integer, not a float.
        * @param out An object into which 3 properties will be created: r, g and b. If not provided a new object will be created.
        * @param hsl Also convert the rgb values into hsl?
        * @param hsv Also convert the rgb values into hsv?
        * @return An object with the red, green and blue values set in the r, g and b properties.
        */
        getPixelRGB(x: number, y: number, out?: any, hsl?: boolean, hsv?: boolean): any;

        /**
        * Get the color of a specific pixel including its alpha value.
        * If you have drawn anything to the BitmapData since it was created you must call BitmapData.update to refresh the array buffer,
        * otherwise this may return out of date color values, or worse - throw a run-time error as it tries to access an array element that doesn't exist.
        * Note that on little-endian systems the format is 0xAABBGGRR and on big-endian the format is 0xRRGGBBAA.
        * 
        * @param x The x coordinate of the pixel to be set. Must lay within the dimensions of this BitmapData and be an integer, not a float.
        * @param y The y coordinate of the pixel to be set. Must lay within the dimensions of this BitmapData and be an integer, not a float.
        * @return A native color value integer (format: 0xAARRGGBB)
        */
        getPixel32(x: number, y: number): number;

        /**
        * Gets all the pixels from the region specified by the given Rectangle object.
        * 
        * @param rect The Rectangle region to get.
        * @return Returns a ImageData object containing a Uint8ClampedArray data property.
        */
        getPixels(rect: Phaser.Rectangle): ImageData;

        /**
        * Gets a JavaScript object that has 6 properties set that are used by BitmapData in a transform.
        * 
        * @param translateX The x translate value.
        * @param translateY The y translate value.
        * @param scaleX The scale x value.
        * @param scaleY The scale y value.
        * @param skewX The skew x value.
        * @param skewY The skew y value.
        * @return A JavaScript object containing all of the properties BitmapData needs for transforms.
        */
        getTransform(translateX: number, translateY: number, scaleX: number, scaleY: number, skewX: number, skewY: number): any;

        /**
        * Draws a line between the coordinates given in the color and thickness specified.
        * 
        * @param x1 The x coordinate to start the line from.
        * @param y1 The y coordinate to start the line from.
        * @param x2 The x coordinate to draw the line to.
        * @param y2 The y coordinate to draw the line to.
        * @param color The stroke color that the line will be drawn in. - Default: '#fff'
        * @param width The line thickness. - Default: 1
        * @return This BitmapData object for method chaining.
        */
        line(x1: number, y1: number, x2: number, y2: number, color?: string, width?: number): Phaser.BitmapData;

        /**
        * Takes the given Game Object, resizes this BitmapData to match it and then draws it into this BitmapDatas canvas, ready for further processing.
        * The source game object is not modified by this operation.
        * If the source object uses a texture as part of a Texture Atlas or Sprite Sheet, only the current frame will be used for sizing.
        * If a string is given it will assume it's a cache key and look in Phaser.Cache for an image key matching the string.
        * 
        * @param source The object that will be used to populate this BitmapData. If you give a string it will try and find the Image in the Game.Cache first.
        * @return This BitmapData object for method chaining.
        */
        load(source: any): Phaser.BitmapData;

        /**
        * Shifts the contents of this BitmapData by the distances given.
        * 
        * The image will wrap-around the edges on all sides if the wrap argument is true (the default).
        * 
        * @param x The amount of pixels to horizontally shift the canvas by. Use a negative value to shift to the left, positive to the right.
        * @param y The amount of pixels to vertically shift the canvas by. Use a negative value to shift up, positive to shift down.
        * @param wrap Wrap the content of the BitmapData. - Default: true
        * @return This BitmapData object for method chaining.
        */
        move(x: number, y: number, wrap?: boolean): Phaser.BitmapData;

        /**
        * Shifts the contents of this BitmapData horizontally.
        * 
        * The image will wrap-around the sides if the wrap argument is true (the default).
        * 
        * @param distance The amount of pixels to horizontally shift the canvas by. Use a negative value to shift to the left, positive to the right.
        * @param wrap Wrap the content of the BitmapData. - Default: true
        * @return This BitmapData object for method chaining.
        */
        moveH(distance: number, wrap?: boolean): Phaser.BitmapData;

        /**
        * Shifts the contents of this BitmapData vertically.
        * 
        * The image will wrap-around the sides if the wrap argument is true (the default).
        * 
        * @param distance The amount of pixels to vertically shift the canvas by. Use a negative value to shift up, positive to shift down.
        * @param wrap Wrap the content of the BitmapData. - Default: true
        * @return This BitmapData object for method chaining.
        */
        moveV(distance: number, wrap?: boolean): Phaser.BitmapData;

        /**
        * Scans through the area specified in this BitmapData and sends the color for every pixel to the given callback along with its x and y coordinates.
        * Whatever value the callback returns is set as the new color for that pixel, unless it returns the same color, in which case it's skipped.
        * Note that the format of the color received will be different depending on if the system is big or little endian.
        * It is expected that your callback will deal with endianess. If you'd rather Phaser did it then use processPixelRGB instead.
        * The callback will also be sent the pixels x and y coordinates respectively.
        * 
        * @param callback The callback that will be sent each pixel color to be processed.
        * @param callbackContext The context under which the callback will be called.
        * @param x The x coordinate of the top-left of the region to process from.
        * @param y The y coordinate of the top-left of the region to process from.
        * @param width The width of the region to process.
        * @param height The height of the region to process.
        * @return This BitmapData object for method chaining.
        */
        processPixel(callback: (color: number, x: number, y: number) => void, callbackContext: any, x?: number, y?: Number, width?: number, height?: number): Phaser.BitmapData;

        /**
        * Scans through the area specified in this BitmapData and sends a color object for every pixel to the given callback.
        * The callback will be sent a color object with 6 properties: `{ r: number, g: number, b: number, a: number, color: number, rgba: string }`.
        * Where r, g, b and a are integers between 0 and 255 representing the color component values for red, green, blue and alpha.
        * The `color` property is an Int32 of the full color. Note the endianess of this will change per system.
        * The `rgba` property is a CSS style rgba() string which can be used with context.fillStyle calls, among others.
        * The callback will also be sent the pixels x and y coordinates respectively.
        * The callback must return either `false`, in which case no change will be made to the pixel, or a new color object.
        * If a new color object is returned the pixel will be set to the r, g, b and a color values given within it.
        * 
        * @param callback The callback that will be sent each pixel color object to be processed.
        * @param callbackContext The context under which the callback will be called.
        * @param x The x coordinate of the top-left of the region to process from.
        * @param y The y coordinate of the top-left of the region to process from.
        * @param width The width of the region to process.
        * @param height The height of the region to process.
        * @return This BitmapData object for method chaining.
        */
        processPixelRGB(callback: (color: ColorComponents, x: number, y: number) => void, callbackContext: any, x?: number, y?: Number, width?: number, height?: number): Phaser.BitmapData;

        /**
        * Draws a filled Rectangle to the BitmapData at the given x, y coordinates and width / height in size.
        * 
        * @param x The x coordinate of the top-left of the Rectangle.
        * @param y The y coordinate of the top-left of the Rectangle.
        * @param width The width of the Rectangle.
        * @param height The height of the Rectangle.
        * @param fillStyle If set the context fillStyle will be set to this value before the rect is drawn.
        * @return This BitmapData object for method chaining.
        */
        rect(x: number, y: number, width: number, height: number, fillStyle?: string): Phaser.BitmapData;

        /**
        * If the game is running in WebGL this will push the texture up to the GPU if it's dirty.
        * This is called automatically if the BitmapData is being used by a Sprite, otherwise you need to remember to call it in your render function.
        * If you wish to suppress this functionality set BitmapData.disableTextureUpload to `true`.
        * @return This BitmapData object for method chaining.
        */
        render(): Phaser.BitmapData;

        /**
        * Replaces all pixels matching one color with another. The color values are given as two sets of RGBA values.
        * An optional region parameter controls if the replacement happens in just a specific area of the BitmapData or the entire thing.
        * 
        * @param r1 The red color value to be replaced. Between 0 and 255.
        * @param g1 The green color value to be replaced. Between 0 and 255.
        * @param b1 The blue color value to be replaced. Between 0 and 255.
        * @param a1 The alpha color value to be replaced. Between 0 and 255.
        * @param r2 The red color value that is the replacement color. Between 0 and 255.
        * @param g2 The green color value that is the replacement color. Between 0 and 255.
        * @param b2 The blue color value that is the replacement color. Between 0 and 255.
        * @param a2 The alpha color value that is the replacement color. Between 0 and 255.
        * @param region The area to perform the search over. If not given it will replace over the whole BitmapData.
        * @return This BitmapData object for method chaining.
        */
        replaceRGB(r1: number, g1: number, b1: number, a1: number, r2: number, g2: number, b2: number, a2: number, region?: Phaser.Rectangle): Phaser.BitmapData;

        /**
        * Resizes the BitmapData. This changes the size of the underlying canvas and refreshes the buffer.
        * 
        * @param width The new width of the BitmapData.
        * @param height The new height of the BitmapData.
        * @return This BitmapData object for method chaining.
        */
        resize(width: number, height: number): Phaser.BitmapData;
        resizeFrame(parent: any, width: number, height: number): void;

        /**
        * Sets the hue, saturation and lightness values on every pixel in the given region, or the whole BitmapData if no region was specified.
        * 
        * @param h The hue, in the range 0 - 1.
        * @param s The saturation, in the range 0 - 1.
        * @param l The lightness, in the range 0 - 1.
        * @param region The area to perform the operation on. If not given it will run over the whole BitmapData.
        * @return This BitmapData object for method chaining.
        */
        setHSL(h?: number, s?: number, l?: number, region?: Phaser.Rectangle): Phaser.BitmapData;

        /**
        * Sets the color of the given pixel to the specified red, green and blue values.
        * 
        * @param x The x coordinate of the pixel to be set. Must lay within the dimensions of this BitmapData and be an integer, not a float.
        * @param y The y coordinate of the pixel to be set. Must lay within the dimensions of this BitmapData and be an integer, not a float.
        * @param red The red color value, between 0 and 0xFF (255).
        * @param green The green color value, between 0 and 0xFF (255).
        * @param blue The blue color value, between 0 and 0xFF (255).
        * @param immediate If `true` the context.putImageData will be called and the dirty flag set. - Default: true
        * @return This BitmapData object for method chaining.
        */
        setPixel(x: number, y: number, red: number, green: number, blue: number, immediate?: boolean): Phaser.BitmapData;

        /**
        * Sets the color of the given pixel to the specified red, green, blue and alpha values.
        * 
        * @param x The x coordinate of the pixel to be set. Must lay within the dimensions of this BitmapData and be an integer, not a float.
        * @param y The y coordinate of the pixel to be set. Must lay within the dimensions of this BitmapData and be an integer, not a float.
        * @param red The red color value, between 0 and 0xFF (255).
        * @param green The green color value, between 0 and 0xFF (255).
        * @param blue The blue color value, between 0 and 0xFF (255).
        * @param alpha The alpha color value, between 0 and 0xFF (255).
        * @param immediate If `true` the context.putImageData will be called and the dirty flag set. - Default: true
        * @return This BitmapData object for method chaining.
        */
        setPixel32(x: number, y: number, red: number, green: number, blue: number, alpha: number, immediate?: boolean): Phaser.BitmapData;

        /**
        * Sets the shadow properties of this BitmapDatas context which will affect all draw operations made to it.
        * You can cancel an existing shadow by calling this method and passing no parameters.
        * Note: At the time of writing (October 2014) Chrome still doesn't support shadowBlur used with drawImage.
        * 
        * @param color The color of the shadow, given in a CSS format, i.e. `#000000` or `rgba(0,0,0,1)`. If `null` or `undefined` the shadow will be reset.
        * @param blur The amount the shadow will be blurred by. Low values = a crisp shadow, high values = a softer shadow. - Default: 5
        * @param x The horizontal offset of the shadow in pixels. - Default: 10
        * @param y The vertical offset of the shadow in pixels. - Default: 10
        * @return This BitmapData object for method chaining.
        */
        shadow(color: string, blur?: number, x?: number, y?: number): Phaser.BitmapData;

        /**
        * Shifts any or all of the hue, saturation and lightness values on every pixel in the given region, or the whole BitmapData if no region was specified.
        * Shifting will add the given value onto the current h, s and l values, not replace them.
        * The hue is wrapped to keep it within the range 0 to 1. Saturation and lightness are clamped to not exceed 1.
        * 
        * @param h The amount to shift the hue by.
        * @param s The amount to shift the saturation by.
        * @param l The amount to shift the lightness by.
        * @param region The area to perform the operation on. If not given it will run over the whole BitmapData.
        * @return This BitmapData object for method chaining.
        */
        shiftHSL(h?: number, s?: number, l?: number, region?: Phaser.Rectangle): Phaser.BitmapData;

        /**
        * Draws text to the BitmapData in the given font and color.
        * The default font is 14px Courier, so useful for quickly drawing debug text.
        * If you need to do a lot of font work to this BitmapData we'd recommend implementing your own text draw method.
        * 
        * @param text The text to write to the BitmapData.
        * @param x The x coordinate of the top-left of the text string.
        * @param y The y coordinate of the top-left of the text string.
        * @param font The font. This is passed directly to Context.font, so anything that can support, this can. - Default: '14px Courier'
        * @param color The color the text will be drawn in. - Default: 'rgb(255,255,255)'
        * @param shadow Draw a single pixel black shadow below the text (offset by text.x/y + 1) - Default: true
        * @return This BitmapData object for method chaining.
        */
        text(text: string, x?: number, y?: number, font?: string, color?: string, shadow?: boolean): Phaser.BitmapData;

        /**
        * Takes the given Line object and image and renders it to this BitmapData as a repeating texture line.
        * 
        * @param line A Phaser.Line object that will be used to plot the start and end of the line.
        * @param image The key of an image in the Phaser.Cache to use as the texture for this line, or an actual Image.
        * @param repeat The pattern repeat mode to use when drawing the line. Either `repeat`, `repeat-x` or `no-repeat`. - Default: 'repeat-x'
        * @return This BitmapData object for method chaining.
        */
        textureLine(line: Phaser.Line, key: string, repeat?: string): Phaser.BitmapData;

        /**
        * This re-creates the BitmapData.imageData from the current context.
        * It then re-builds the ArrayBuffer, the data Uint8ClampedArray reference and the pixels Int32Array.
        * If not given the dimensions defaults to the full size of the context.
        * 
        * Warning: This is a very expensive operation, so use it sparingly.
        * 
        * @param x The x coordinate of the top-left of the image data area to grab from.
        * @param y The y coordinate of the top-left of the image data area to grab from.
        * @param width The width of the image data area. - Default: 1
        * @param height The height of the image data area. - Default: 1
        * @return This BitmapData object for method chaining.
        */
        update(x?: number, y?: number, width?: number, height?: number): Phaser.BitmapData;

        /**
        * Updates a portion of the BitmapData from a source Bitmap.
        * This optimization is important if calling update() on a large Bitmap is causing performance issues.
        * Make sure you use getPixel32() instead of getPixel().
        * This does not work with floating point numbers for x and y.
        * 
        * @param source The BitmapData you wish to copy.
        * @param x The x coordinate of the top-left of the area to copy.
        * @param y The y coordinate of the top-left of the area to copy.
        * @return This BitmapData object for method chaining.
        */
        copyBitmapData(source: Phaser.BitmapData, x: number, y: number): Phaser.BitmapData;

    }


    /**
    * BitmapText objects work by taking a texture file and an XML or JSON file that describes the font structure.
    * It then generates a new Sprite object for each letter of the text, proportionally spaced out and aligned to
    * match the font structure.
    * 
    * BitmapText objects are less flexible than Text objects, in that they have less features such as shadows, fills and the ability
    * to use Web Fonts, however you trade this flexibility for rendering speed. You can also create visually compelling BitmapTexts by
    * processing the font texture in an image editor, applying fills and any other effects required.
    * 
    * To create multi-line text insert \r, \n or \r\n escape codes into the text string.
    * 
    * If you are having performance issues due to the volume of sprites being rendered, and do not require the text to be constantly
    * updating, you can use BitmapText.generateTexture to create a static texture from this BitmapText.
    * 
    * To create a BitmapText data files you can use:
    * 
    * BMFont (Windows, free): http://www.angelcode.com/products/bmfont/
    * Glyph Designer (OS X, commercial): http://www.71squared.com/en/glyphdesigner
    * Littera (Web-based, free): http://kvazars.com/littera/
    * 
    * For most use cases it is recommended to use XML. If you wish to use JSON, the formatting should be equal to the result of
    * converting a valid XML file through the popular X2JS library. An online tool for conversion can be found here: http://codebeautify.org/xmltojson
    * 
    * If you were using an older version of Phaser (< 2.4) and using the DOMish parser hack, please remove this. It isn't required any longer.
    */
    class BitmapText extends PIXI.DisplayObjectContainer {


        /**
        * BitmapText objects work by taking a texture file and an XML or JSON file that describes the font structure.
        * It then generates a new Sprite object for each letter of the text, proportionally spaced out and aligned to
        * match the font structure.
        * 
        * BitmapText objects are less flexible than Text objects, in that they have less features such as shadows, fills and the ability
        * to use Web Fonts, however you trade this flexibility for rendering speed. You can also create visually compelling BitmapTexts by
        * processing the font texture in an image editor, applying fills and any other effects required.
        * 
        * To create multi-line text insert \r, \n or \r\n escape codes into the text string.
        * 
        * If you are having performance issues due to the volume of sprites being rendered, and do not require the text to be constantly
        * updating, you can use BitmapText.generateTexture to create a static texture from this BitmapText.
        * 
        * To create a BitmapText data files you can use:
        * 
        * BMFont (Windows, free): http://www.angelcode.com/products/bmfont/
        * Glyph Designer (OS X, commercial): http://www.71squared.com/en/glyphdesigner
        * Littera (Web-based, free): http://kvazars.com/littera/
        * 
        * For most use cases it is recommended to use XML. If you wish to use JSON, the formatting should be equal to the result of
        * converting a valid XML file through the popular X2JS library. An online tool for conversion can be found here: http://codebeautify.org/xmltojson
        * 
        * If you were using an older version of Phaser (< 2.4) and using the DOMish parser hack, please remove this. It isn't required any longer.
        * 
        * @param game A reference to the currently running game.
        * @param x X coordinate to display the BitmapText object at.
        * @param y Y coordinate to display the BitmapText object at.
        * @param font The key of the BitmapText as stored in Phaser.Cache.
        * @param text The text that will be rendered. This can also be set later via BitmapText.text. - Default: ''
        * @param size The size the font will be rendered at in pixels. - Default: 32
        * @param align The alignment of multi-line text. Has no effect if there is only one line of text. - Default: 'left'
        */
        constructor(game: Phaser.Game, x: number, y: number, font: string, text?: string, size?: number, align?: string);


        /**
        * Alignment for multi-line text ('left', 'center' or 'right'), does not affect single lines of text.
        */
        align: string;

        /**
        * A useful flag to control if the Game Object is alive or dead.
        * 
        * This is set automatically by the Health components `damage` method should the object run out of health.
        * Or you can toggle it via your game code.
        * 
        * This property is mostly just provided to be used by your game - it doesn't effect rendering or logic updates.
        * However you can use `Group.getFirstAlive` in conjunction with this property for fast object pooling and recycling.
        * Default: true
        */
        alive: boolean;

        /**
        * The anchor value of this BitmapText.
        */
        anchor: Phaser.Point;

        /**
        * If the Game Object is enabled for animation (such as a Phaser.Sprite) this is a reference to its AnimationManager instance.
        * Through it you can create, play, pause and stop animations.
        */
        animations: Phaser.AnimationManager;

        /**
        * The angle property is the rotation of the Game Object in *degrees* from its original orientation.
        * 
        * Values from 0 to 180 represent clockwise rotation; values from 0 to -180 represent counterclockwise rotation.
        * 
        * Values outside this range are added to or subtracted from 360 to obtain a value within the range.
        * For example, the statement player.angle = 450 is the same as player.angle = 90.
        * 
        * If you wish to work in radians instead of degrees you can use the property `rotation` instead.
        * Working in radians is slightly faster as it doesn't have to perform any calculations.
        */
        angle: number;

        /**
        * A Game Object with `autoCull` set to true will check its bounds against the World Camera every frame.
        * If it is not intersecting the Camera bounds at any point then it has its `renderable` property set to `false`.
        * This keeps the Game Object alive and still processing updates, but forces it to skip the render step entirely.
        * 
        * This is a relatively expensive operation, especially if enabled on hundreds of Game Objects. So enable it only if you know it's required,
        * or you have tested performance and find it acceptable.
        */
        autoCull: boolean;

        /**
        * `body` is the Game Objects physics body. Once a Game Object is enabled for physics you access all associated
        * properties and methods via it.
        * 
        * By default Game Objects won't add themselves to any physics system and their `body` property will be `null`.
        * 
        * To enable this Game Object for physics you need to call `game.physics.enable(object, system)` where `object` is this object
        * and `system` is the Physics system you are using. If none is given it defaults to `Phaser.Physics.Arcade`.
        * 
        * You can alternatively call `game.physics.arcade.enable(object)`, or add this Game Object to a physics enabled Group.
        * 
        * Important: Enabling a Game Object for P2 or Ninja physics will automatically set its `anchor` property to 0.5,
        * so the physics body is centered on the Game Object.
        * 
        * If you need a different result then adjust or re-create the Body shape offsets manually or reset the anchor after enabling physics.
        */
        body: Phaser.Physics.Arcade.Body | Phaser.Physics.P2.Body | Phaser.Physics.Ninja.Body | any;

        /**
        * The sum of the y and height properties.
        * This is the same as `y + height - offsetY`.
        */
        bottom: number;

        /**
        * The x/y coordinate offset applied to the top-left of the camera that this Game Object will be drawn at if `fixedToCamera` is true.
        * 
        * The values are relative to the top-left of the camera view and in addition to any parent of the Game Object on the display list.
        */
        cameraOffset: Phaser.Point;

        /**
        * If this is set to `true` the Game Object checks if it is within the World bounds each frame.
        * 
        * When it is no longer intersecting the world bounds it dispatches the `onOutOfBounds` event.
        * 
        * If it was *previously* out of bounds but is now intersecting the world bounds again it dispatches the `onEnterBounds` event.
        * 
        * It also optionally kills the Game Object if `outOfBoundsKill` is `true`.
        * 
        * When `checkWorldBounds` is enabled it forces the Game Object to calculate its full bounds every frame.
        * 
        * This is a relatively expensive operation, especially if enabled on hundreds of Game Objects. So enable it only if you know it's required,
        * or you have tested performance and find it acceptable.
        */
        checkWorldBounds: boolean;

        /**
        * An empty Object that belongs to this Game Object.
        * This value isn't ever used internally by Phaser, but may be used by your own code, or
        * by Phaser Plugins, to store data that needs to be associated with the Game Object,
        * without polluting the Game Object directly.
        * Default: {}
        */
        data: any;

        /**
        * As a Game Object runs through its destroy method this flag is set to true,
        * and can be checked in any sub-systems or plugins it is being destroyed from.
        */
        destroyPhase: boolean;

        /**
        * A debug flag designed for use with `Game.enableStep`.
        */
        debug: boolean;

        /**
        * The dirty state of this object.
        */
        dirty: boolean;

        /**
        * All Phaser Game Objects have an Events class which contains all of the events that are dispatched when certain things happen to this
        * Game Object, or any of its components.
        */
        events: Phaser.Events;

        /**
        * Controls if this Game Object is processed by the core game loop.
        * If this Game Object has a physics body it also controls if its physics body is updated or not.
        * When `exists` is set to `false` it will remove its physics body from the physics world if it has one.
        * It also toggles the `visible` property to false as well.
        * 
        * Setting `exists` to true will add its physics body back in to the physics world, if it has one.
        * It will also set the `visible` property to `true`.
        */
        exists: boolean;

        /**
        * A Game Object that is "fixed" to the camera is rendered at a given x/y offsets from the top left of the camera. The offsets
        * are stored in the `cameraOffset` property, which is initialized with the current object coordinates.
        * 
        * The values are adjusted at the rendering stage, overriding the Game Objects actual world position.
        * 
        * The end result is that the Game Object will appear to be 'fixed' to the camera, regardless of where in the game world
        * the camera is viewing. This is useful if for example this Game Object is a UI item that you wish to be visible at all times
        * regardless where in the world the camera is.
        * 
        * Note that the `cameraOffset` values are in addition to any parent of this Game Object on the display list.
        * 
        * Be careful not to set `fixedToCamera` on Game Objects which are in Groups that already have `fixedToCamera` enabled on them.
        */
        fixedToCamera: boolean;

        /**
        * The font the text will be rendered in, i.e. 'Arial'. Must be loaded in the browser before use.
        */
        font: string;

        /**
        * The size of the font in pixels.
        */
        fontSize: number;

        /**
        * A Game Object is considered `fresh` if it has just been created or reset and is yet to receive a renderer transform update.
        * This property is mostly used internally by the physics systems, but is exposed for the use of plugins.
        */
        fresh: boolean;

        /**
        * A reference to the currently running Game.
        */
        game: Phaser.Game;

        /**
        * The Input Handler for this Game Object.
        * 
        * By default it is disabled. If you wish this Game Object to process input events you should enable it with: `inputEnabled = true`.
        * 
        * After you have done this, this property will be a reference to the Phaser InputHandler.
        */
        input: Phaser.InputHandler;

        /**
        * By default a Game Object won't process any input events. By setting `inputEnabled` to true a Phaser.InputHandler is created
        * for this Game Object and it will then start to process click / touch events and more.
        * 
        * You can then access the Input Handler via `this.input`.
        * 
        * Note that Input related events are dispatched from `this.events`, i.e.: `events.onInputDown`.
        * 
        * If you set this property to false it will stop the Input Handler from processing any more input events.
        * 
        * If you want to _temporarily_ disable input for a Game Object, then it's better to set
        * `input.enabled = false`, as it won't reset any of the Input Handlers internal properties.
        * You can then toggle this back on as needed.
        */
        inputEnabled: boolean;

        /**
        * Checks if the Game Objects bounds intersect with the Game Camera bounds.
        * Returns `true` if they do, otherwise `false` if fully outside of the Cameras bounds.
        */
        inCamera: boolean;

        /**
        * Checks if the Game Objects bounds are within, or intersect at any point with the Game World bounds.
        */
        inWorld: boolean;

        /**
        * The key of the image or texture used by this Game Object during rendering.
        * If it is a string it's the string used to retrieve the texture from the Phaser Image Cache.
        * It can also be an instance of a RenderTexture, BitmapData, Video or PIXI.Texture.
        * If a Game Object is created without a key it is automatically assigned the key `__default` which is a 32x32 transparent PNG stored within the Cache.
        * If a Game Object is given a key which doesn't exist in the Image Cache it is re-assigned the key `__missing` which is a 32x32 PNG of a green box with a line through it.
        */
        key: string | Phaser.RenderTexture | Phaser.BitmapData | Phaser.Video | PIXI.Texture;

        /**
        * The left coordinate of the Game Object.
        * This is the same as `x - offsetX`.
        */
        left: number;

        /**
        * A user defined name given to this Game Object.
        * This value isn't ever used internally by Phaser, it is meant as a game level property.
        */
        name: string;

        /**
        * The components this Game Object has installed.
        */
        components: any;

        /**
        * The lifespan allows you to give a Game Object a lifespan in milliseconds.
        * 
        * Once the Game Object is 'born' you can set this to a positive value.
        * 
        * It is automatically decremented by the millisecond equivalent of `game.time.physicsElapsed` each frame.
        * When it reaches zero it will call the `kill` method.
        * 
        * Very handy for particles, bullets, collectibles, or any other short-lived entity.
        */
        lifespan: number;

        /**
        * The maximum display width of this BitmapText in pixels.
        * 
        * If BitmapText.text is longer than maxWidth then the lines will be automatically wrapped
        * based on the last whitespace character found in the line.
        * 
        * If no whitespace was found then no wrapping will take place and consequently the maxWidth value will not be honored.
        * 
        * Disable maxWidth by setting the value to 0. The maximum width of this BitmapText in pixels.
        */
        maxWidth: number;

        /**
        * The amount the Game Object is visually offset from its x coordinate.
        * This is the same as `width * anchor.x`.
        * It will only be > 0 if anchor.x is not equal to zero.
        */
        offsetX: number;

        /**
        * The amount the Game Object is visually offset from its y coordinate.
        * This is the same as `height * anchor.y`.
        * It will only be > 0 if anchor.y is not equal to zero.
        */
        offsetY: number;

        /**
        * If this and the `checkWorldBounds` property are both set to `true` then the `kill` method is called as soon as `inWorld` returns false.
        */
        outOfBoundsKill: boolean;

        /**
        * A Game Object is that is pendingDestroy is flagged to have its destroy method called on the next logic update.
        * You can set it directly to allow you to flag an object to be destroyed on its next update.
        * 
        * This is extremely useful if you wish to destroy an object from within one of its own callbacks
        * such as with Buttons or other Input events.
        */
        pendingDestroy: boolean;

        /**
        * The const physics body type of this object.
        */
        physicsType: number;

        /**
        * The position the Game Object was located in the previous frame.
        */
        previousPosition: Phaser.Point;

        /**
        * The rotation the Game Object was in set to in the previous frame. Value is in radians.
        */
        previousRotation: number;

        /**
        * The coordinates, in pixels, of this DisplayObject, relative to its parent container.
        * 
        * The value of this property does not reflect any positioning happening further up the display list.
        * To obtain that value please see the `worldPosition` property.
        */
        position: Phaser.Point;

        /**
        * The render order ID is used internally by the renderer and Input Manager and should not be modified.
        * This property is mostly used internally by the renderers, but is exposed for the use of plugins.
        */
        renderOrderID: number;

        /**
        * The right coordinate of the Game Object.
        * This is the same as `x + width - offsetX`.
        */
        right: number;

        /**
        * The text to be displayed by this BitmapText object.
        */
        text: string;

        /**
        * Enable or disable texture smoothing for this BitmapText.
        * 
        * The smoothing is applied to the BaseTexture of this font, which all letters of the text reference.
        * 
        * Smoothing is enabled by default.
        */
        smoothed: boolean;

        /**
        * The width in pixels of the overall text area, taking into consideration multi-line text.
        */
        textWidth: number;

        /**
        * The height in pixels of the overall text area, taking into consideration multi-line text.
        */
        textHeight: number;

        /**
        * The tint applied to the BitmapText. This is a hex value. Set to white to disable (0xFFFFFF)
        */
        tint: number;

        /**
        * The y coordinate of the Game Object.
        * This is the same as `y - offsetY`.
        */
        top: number;

        /**
        * The const type of this object.
        */
        type: number;

        /**
        * The world coordinates of this Game Object in pixels.
        * Depending on where in the display list this Game Object is placed this value can differ from `position`,
        * which contains the x/y coordinates relative to the Game Objects parent.
        */
        world: Phaser.Point;

        /**
        * The horizontal position of the DisplayObject, in pixels, relative to its parent.
        * If you need the world position of the DisplayObject, use `DisplayObject.worldPosition` instead.
        */
        x: number;

        /**
        * The vertical position of the DisplayObject, in pixels, relative to its parent.
        * If you need the world position of the DisplayObject, use `DisplayObject.worldPosition` instead.
        */
        y: number;

        /**
        * The z depth of this Game Object within its parent Group.
        * No two objects in a Group can have the same z value.
        * This value is adjusted automatically whenever the Group hierarchy changes.
        * If you wish to re-order the layering of a Game Object then see methods like Group.moveUp or Group.bringToTop.
        */
        z: number;


        /**
        * Aligns this Game Object within another Game Object, or Rectangle, known as the
        * 'container', to one of 9 possible positions.
        * 
        * The container must be a Game Object, or Phaser.Rectangle object. This can include properties
        * such as `World.bounds` or `Camera.view`, for aligning Game Objects within the world
        * and camera bounds. Or it can include other Sprites, Images, Text objects, BitmapText,
        * TileSprites or Buttons.
        * 
        * Please note that aligning a Sprite to another Game Object does **not** make it a child of
        * the container. It simply modifies its position coordinates so it aligns with it.
        * 
        * The position constants you can use are:
        * 
        * `Phaser.TOP_LEFT`, `Phaser.TOP_CENTER`, `Phaser.TOP_RIGHT`, `Phaser.LEFT_CENTER`,
        * `Phaser.CENTER`, `Phaser.RIGHT_CENTER`, `Phaser.BOTTOM_LEFT`,
        * `Phaser.BOTTOM_CENTER` and `Phaser.BOTTOM_RIGHT`.
        * 
        * The Game Objects are placed in such a way that their _bounds_ align with the
        * container, taking into consideration rotation, scale and the anchor property.
        * This allows you to neatly align Game Objects, irrespective of their position value.
        * 
        * The optional `offsetX` and `offsetY` arguments allow you to apply extra spacing to the final
        * aligned position of the Game Object. For example:
        * 
        * `sprite.alignIn(background, Phaser.BOTTOM_RIGHT, -20, -20)`
        * 
        * Would align the `sprite` to the bottom-right, but moved 20 pixels in from the corner.
        * Think of the offsets as applying an adjustment to the containers bounds before the alignment takes place.
        * So providing a negative offset will 'shrink' the container bounds by that amount, and providing a positive
        * one expands it.
        * 
        * @param container The Game Object or Rectangle with which to align this Game Object to. Can also include properties such as `World.bounds` or `Camera.view`.
        * @param position The position constant. One of `Phaser.TOP_LEFT` (default), `Phaser.TOP_CENTER`, `Phaser.TOP_RIGHT`, `Phaser.LEFT_CENTER`, `Phaser.CENTER`, `Phaser.RIGHT_CENTER`, `Phaser.BOTTOM_LEFT`, `Phaser.BOTTOM_CENTER` or `Phaser.BOTTOM_RIGHT`.
        * @param offsetX A horizontal adjustment of the Containers bounds, applied to the aligned position of the Game Object. Use a negative value to shrink the bounds, positive to increase it.
        * @param offsetY A vertical adjustment of the Containers bounds, applied to the aligned position of the Game Object. Use a negative value to shrink the bounds, positive to increase it.
        * @return This Game Object.
        */
        alignIn(container: Phaser.Rectangle | Phaser.Sprite | Phaser.Image | Phaser.Text | Phaser.BitmapText | Phaser.Button | Phaser.Graphics | Phaser.TileSprite, position?: number, offsetX?: number, offsetY?: number): any;

        /**
        * Aligns this Game Object to the side of another Game Object, or Rectangle, known as the
        * 'parent', in one of 11 possible positions.
        * 
        * The parent must be a Game Object, or Phaser.Rectangle object. This can include properties
        * such as `World.bounds` or `Camera.view`, for aligning Game Objects within the world
        * and camera bounds. Or it can include other Sprites, Images, Text objects, BitmapText,
        * TileSprites or Buttons.
        * 
        * Please note that aligning a Sprite to another Game Object does **not** make it a child of
        * the parent. It simply modifies its position coordinates so it aligns with it.
        * 
        * The position constants you can use are:
        * 
        * `Phaser.TOP_LEFT` (default), `Phaser.TOP_CENTER`, `Phaser.TOP_RIGHT`, `Phaser.LEFT_TOP`,
        * `Phaser.LEFT_CENTER`, `Phaser.LEFT_BOTTOM`, `Phaser.RIGHT_TOP`, `Phaser.RIGHT_CENTER`,
        * `Phaser.RIGHT_BOTTOM`, `Phaser.BOTTOM_LEFT`, `Phaser.BOTTOM_CENTER`
        * and `Phaser.BOTTOM_RIGHT`.
        * 
        * The Game Objects are placed in such a way that their _bounds_ align with the
        * parent, taking into consideration rotation, scale and the anchor property.
        * This allows you to neatly align Game Objects, irrespective of their position value.
        * 
        * The optional `offsetX` and `offsetY` arguments allow you to apply extra spacing to the final
        * aligned position of the Game Object. For example:
        * 
        * `sprite.alignTo(background, Phaser.BOTTOM_RIGHT, -20, -20)`
        * 
        * Would align the `sprite` to the bottom-right, but moved 20 pixels in from the corner.
        * Think of the offsets as applying an adjustment to the parents bounds before the alignment takes place.
        * So providing a negative offset will 'shrink' the parent bounds by that amount, and providing a positive
        * one expands it.
        * 
        * @param parent The Game Object or Rectangle with which to align this Game Object to. Can also include properties such as `World.bounds` or `Camera.view`.
        * @param position The position constant. One of `Phaser.TOP_LEFT`, `Phaser.TOP_CENTER`, `Phaser.TOP_RIGHT`, `Phaser.LEFT_TOP`, `Phaser.LEFT_CENTER`, `Phaser.LEFT_BOTTOM`, `Phaser.RIGHT_TOP`, `Phaser.RIGHT_CENTER`, `Phaser.RIGHT_BOTTOM`, `Phaser.BOTTOM_LEFT`, `Phaser.BOTTOM_CENTER` or `Phaser.BOTTOM_RIGHT`.
        * @param offsetX A horizontal adjustment of the Containers bounds, applied to the aligned position of the Game Object. Use a negative value to shrink the bounds, positive to increase it.
        * @param offsetY A vertical adjustment of the Containers bounds, applied to the aligned position of the Game Object. Use a negative value to shrink the bounds, positive to increase it.
        * @return This Game Object.
        */
        alignTo(container: Phaser.Rectangle | Phaser.Sprite | Phaser.Image | Phaser.Text | Phaser.BitmapText | Phaser.Button | Phaser.Graphics | Phaser.TileSprite, position?: number, offsetX?: number, offsetY?: number): any;

        /**
        * Destroys the Game Object. This removes it from its parent group, destroys the input, event and animation handlers if present
        * and nulls its reference to `game`, freeing it up for garbage collection.
        * 
        * If this Game Object has the Events component it will also dispatch the `onDestroy` event.
        * 
        * You can optionally also destroy the BaseTexture this Game Object is using. Be careful if you've
        * more than one Game Object sharing the same BaseTexture.
        * 
        * @param destroyChildren Should every child of this object have its destroy method called as well? - Default: true
        * @param destroyTexture Destroy the BaseTexture this Game Object is using? Note that if another Game Object is sharing the same BaseTexture it will invalidate it.
        */
        destroy(destroyChildren?: boolean): void;

        /**
        * Kills a Game Object. A killed Game Object has its `alive`, `exists` and `visible` properties all set to false.
        * 
        * It will dispatch the `onKilled` event. You can listen to `events.onKilled` for the signal.
        * 
        * Note that killing a Game Object is a way for you to quickly recycle it in an object pool,
        * it doesn't destroy the object or free it up from memory.
        * 
        * If you don't need this Game Object any more you should call `destroy` instead.
        * @return This instance.
        */
        kill(): void;

        /**
        * Automatically called by World.preUpdate.
        */
        postUpdate(): void;

        /**
        * Automatically called by World.preUpdate.
        * @return True if the BitmapText was rendered, otherwise false.
        */
        preUpdate(): void;

        /**
        * If a BitmapText changes from having a large number of characters to having very few characters it will cause lots of
        * Sprites to be retained in the BitmapText._glyphs array. Although they are not attached to the display list they
        * still take up memory while sat in the glyphs pool waiting to be re-used in the future.
        * 
        * If you know that the BitmapText will not grow any larger then you can purge out the excess glyphs from the pool
        * by calling this method.
        * 
        * Calling this doesn't prevent you from increasing the length of the text again in the future.
        * @return The amount of glyphs removed from the pool.
        */
        purgeGlyphs(): number;

        /**
        * Resets the Game Object.
        * 
        * This moves the Game Object to the given x/y world coordinates and sets `fresh`, `exists`,
        * `visible` and `renderable` to true.
        * 
        * If this Game Object has the LifeSpan component it will also set `alive` to true and `health` to the given value.
        * 
        * If this Game Object has a Physics Body it will reset the Body.
        * 
        * @param x The x coordinate (in world space) to position the Game Object at.
        * @param y The y coordinate (in world space) to position the Game Object at.
        * @param health The health to give the Game Object if it has the Health component. - Default: 1
        * @return This instance.
        */
        reset(x: number, y: number, health?: number): Phaser.BitmapText;

        /**
        * Brings a 'dead' Game Object back to life, optionally resetting its health value in the process.
        * 
        * A resurrected Game Object has its `alive`, `exists` and `visible` properties all set to true.
        * 
        * It will dispatch the `onRevived` event. Listen to `events.onRevived` for the signal.
        * 
        * @param health The health to give the Game Object. Only set if the GameObject has the Health component. - Default: 100
        * @return This instance.
        */
        revive(health?: number): Phaser.BitmapText;

        /**
        * Given the input text this will scan the characters until either a newline is encountered,
        * or the line exceeds maxWidth, taking into account kerning, character widths and scaling.
        * 
        * @param data A reference to the font object in the Phaser.Cache.
        * @param scale The scale of the font in relation to the texture.
        * @param text The text to parse.
        * @return An object containing the parsed characters, total pixel width and x offsets.
        */

                                                            /**
                                                            * The width of the displayObjectContainer, setting this will actually modify the scale to achieve the value set
                                                            */

                                                                           /**
                                                                           * The text to be displayed by this BitmapText object.
                                                                           */
        scanLine(data: any, scale: number, text: string): { width: number; text: string; end: boolean; chars: string[] };

        /**
        * The text to be displayed by this BitmapText object.
        * 
        * It's faster to use `BitmapText.text = string`, but this is kept for backwards compatibility.
        * 
        * @param text The text to be displayed by this BitmapText object.
        */
        setText(text: string): void;

        /**
        * Override this method in your own custom objects to handle any update requirements.
        * It is called immediately after `preUpdate` and before `postUpdate`.
        * Remember if this Game Object has any children you should call update on those too.
        */
        update(): void;

        /**
        * Renders text and updates it when needed.
        */
        updateText(): void;

        /**
        * Updates the transform of this object.
        */
        updateTransform(): void;

    }

    class Bullet extends Phaser.Sprite {

        constructor(game: Phaser.Game, x: number, y: number, key?: any, frame?: any);

        kill(): Phaser.Bullet;
        update(): void;

    }


    /**
    * Create a new `Button` object. A Button is a special type of Sprite that is set-up to handle Pointer events automatically.
    * 
    * The four states a Button responds to are:
    * 
    * * 'Over' - when the Pointer moves over the Button. This is also commonly known as 'hover'.
    * * 'Out' - when the Pointer that was previously over the Button moves out of it.
    * * 'Down' - when the Pointer is pressed down on the Button. I.e. touched on a touch enabled device or clicked with the mouse.
    * * 'Up' - when the Pointer that was pressed down on the Button is released again.
    * 
    * A different texture/frame and activation sound can be specified for any of the states.
    * 
    * Frames can be specified as either an integer (the frame ID) or a string (the frame name); the same values that can be used with a Sprite constructor.
    */
    class Button extends Phaser.Image {


        /**
        * Create a new `Button` object. A Button is a special type of Sprite that is set-up to handle Pointer events automatically.
        * 
        * The four states a Button responds to are:
        * 
        * * 'Over' - when the Pointer moves over the Button. This is also commonly known as 'hover'.
        * * 'Out' - when the Pointer that was previously over the Button moves out of it.
        * * 'Down' - when the Pointer is pressed down on the Button. I.e. touched on a touch enabled device or clicked with the mouse.
        * * 'Up' - when the Pointer that was pressed down on the Button is released again.
        * 
        * A different texture/frame and activation sound can be specified for any of the states.
        * 
        * Frames can be specified as either an integer (the frame ID) or a string (the frame name); the same values that can be used with a Sprite constructor.
        * 
        * @param game Current game instance.
        * @param x X position of the Button.
        * @param y Y position of the Button.
        * @param key The image key (in the Game.Cache) to use as the texture for this Button.
        * @param callback The function to call when this Button is pressed.
        * @param callbackContext The context in which the callback will be called (usually 'this').
        * @param overFrame The frame / frameName when the button is in the Over state.
        * @param outFrame The frame / frameName when the button is in the Out state.
        * @param downFrame The frame / frameName when the button is in the Down state.
        * @param upFrame The frame / frameName when the button is in the Up state.
        */
        constructor(game: Phaser.Game, x?: number, y?: number, key?: string, callback?: Function, callbackContext?: any, overFrame?: string | number, outFrame?: string | number, downFrame?: string | number, upFrame?: string | number);


        /**
        * When the Button is touched / clicked and then released you can force it to enter a state of "out" instead of "up".
        * 
        * This can also accept a {@link Phaser.PointerModer pointer mode bitmask} for more refined control.
        */
        forceOut: boolean;

        /**
        * When true the the texture frame will not be automatically switched on up/down/over/out events.
        */
        freezeFrames: boolean;

        /**
        * The Sound to be played when this Buttons Down state is activated.
        */
        onDownSound: Phaser.Sound | Phaser.AudioSprite;

        /**
        * The Sound Marker used in conjunction with the onDownSound.
        */
        onDownSoundMarker: string;

        /**
        * The Signal (or event) dispatched when this Button is in an Down state.
        */
        onInputDown: Phaser.Signal;

        /**
        * The Signal (or event) dispatched when this Button is in an Out state.
        */
        onInputOut: Phaser.Signal;

        /**
        * The Signal (or event) dispatched when this Button is in an Over state.
        */
        onInputOver: Phaser.Signal;

        /**
        * The Signal (or event) dispatched when this Button is in an Up state.
        */
        onInputUp: Phaser.Signal;

        /**
        * The Sound to be played when this Buttons Out state is activated.
        */
        onOutSound: Phaser.Sound | Phaser.AudioSprite;

        /**
        * The Sound Marker used in conjunction with the onOutSound.
        */
        onOutSoundMarker: string;

        /**
        * The Sound to be played when this Buttons Over state is activated.
        */
        onOverSound: Phaser.Sound | Phaser.AudioSprite;

        /**
        * The Sound Marker used in conjunction with the onOverSound.
        */
        onOverSoundMarker: string;

        /**
        * If true then onOver events (such as onOverSound) will only be triggered if the Pointer object causing them was the Mouse Pointer.
        * The frame will still be changed as applicable.
        * Default: true
        */
        onOverMouseOnly: boolean;

        /**
        * The Sound to be played when this Buttons Up state is activated.
        */
        onUpSound: Phaser.Sound | Phaser.AudioSprite;
        onUpSoundMaker: string;

        /**
        * The const physics body type of this object.
        */
        physicsType: number;

        /**
        * The Phaser Object Type.
        */
        type: number;


        /**
        * Clears all of the frames set on this Button.
        */
        clearFrames(): void;

        /**
        * The Sound to be played when a Pointer presses down on this Button.
        * 
        * @param sound The Sound that will be played.
        * @param marker A Sound Marker that will be used in the playback.
        */
        setDownSound(sound: Phaser.Sound | Phaser.AudioSprite, marker?: string): void;

        /**
        * Used to manually set the frames that will be used for the different states of the Button.
        * 
        * Frames can be specified as either an integer (the frame ID) or a string (the frame name); these are the same values that can be used with a Sprite constructor.
        * 
        * @param overFrame The frame / frameName when the button is in the Over state.
        * @param outFrame The frame / frameName when the button is in the Out state.
        * @param downFrame The frame / frameName when the button is in the Down state.
        * @param upFrame The frame / frameName when the button is in the Up state.
        */
        setFrames(overFrame?: string | number, outFrame?: string | number, downFrame?: string | number, upFrame?: string | number): void;

        /**
        * Internal function that handles input events.
        * 
        * @param sprite The Button that the event occurred on.
        * @param pointer The Pointer that activated the Button.
        */
        onInputOverHandler(sprite: Phaser.Button, pointer: Phaser.Pointer): void;

        /**
        * Internal function that handles input events.
        * 
        * @param sprite The Button that the event occurred on.
        * @param pointer The Pointer that activated the Button.
        */
        onInputOutHandler(sprite: Phaser.Button, pointer: Phaser.Pointer): void;

        /**
        * Internal function that handles input events.
        * 
        * @param sprite The Button that the event occurred on.
        * @param pointer The Pointer that activated the Button.
        */
        onInputDownHandler(sprite: Phaser.Button, pointer: Phaser.Pointer): void;

        /**
        * Internal function that handles input events.
        * 
        * @param sprite The Button that the event occurred on.
        * @param pointer The Pointer that activated the Button.
        */
        onInputUpHandler(sprite: Phaser.Button, pointer: Phaser.Pointer, isOver: boolean): void;

        /**
        * Called when this Button is removed from the World.
        */
        removedFromWorld(): void;

        /**
        * The Sound to be played when a Pointer moves out of this Button.
        * 
        * @param sound The Sound that will be played.
        * @param marker A Sound Marker that will be used in the playback.
        */
        setOutSound(sound: Phaser.Sound | Phaser.AudioSprite, marker?: string): void;

        /**
        * The Sound to be played when a Pointer moves over this Button.
        * 
        * @param sound The Sound that will be played.
        * @param marker A Sound Marker that will be used in the playback.
        */
        setOverSound(sound: Phaser.Sound | Phaser.AudioSprite, marker?: string): void;

        /**
        * Sets the sounds to be played whenever this Button is interacted with. Sounds can be either full Sound objects, or markers pointing to a section of a Sound object.
        * The most common forms of sounds are 'hover' effects and 'click' effects, which is why the order of the parameters is overSound then downSound.
        * 
        * Call this function with no parameters to reset all sounds on this Button.
        * 
        * @param overSound Over Button Sound.
        * @param overMarker Over Button Sound Marker.
        * @param downSound Down Button Sound.
        * @param downMarker Down Button Sound Marker.
        * @param outSound Out Button Sound.
        * @param outMarker Out Button Sound Marker.
        * @param upSound Up Button Sound.
        * @param upMarker Up Button Sound Marker.
        */
        setSounds(overSound?: Phaser.Sound | Phaser.AudioSprite, overMarker?: string, downSound?: Phaser.Sound | Phaser.AudioSprite, downMarker?: string, outSound?: Phaser.Sound | Phaser.AudioSprite, outMarker?: string, upSound?: Phaser.Sound | Phaser.AudioSprite, upMarker?: string): void;
        setState(newState: number): void;

        /**
        * The Sound to be played when a Pointer has pressed down and is released from this Button.
        * 
        * @param sound The Sound that will be played.
        * @param marker A Sound Marker that will be used in the playback.
        */
        setUpSound(sound: Phaser.Sound | Phaser.AudioSprite, marker?: string): void;

    }


    /**
    * Enumeration categorizing operational modes of pointers.
    * 
    * PointerType values represent valid bitmasks.
    * For example, a value representing both Mouse and Touch devices
    * can be expressed as `PointerMode.CURSOR | PointerMode.CONTACT`.
    * 
    * Values may be added for future mode categorizations.
    */
    class PointerMode {


        /**
        * A 'CURSOR' is a pointer with a *passive cursor* such as a mouse, touchpad, watcom stylus, or even TV-control arrow-pad.
        * 
        * It has the property that a cursor is passively moved without activating the input.
        * This currently corresponds with {@link Phaser.Pointer#isMouse} property.
        */
        static CURSOR: number;

        /**
        * A 'CONTACT' pointer has an *active cursor* that only tracks movement when actived; notably this is a touch-style input.
        */
        static CONTACT: number;

    }


    /**
    * Phaser has one single cache in which it stores all assets.
    * 
    * The cache is split up into sections, such as images, sounds, video, json, etc. All assets are stored using
    * a unique string-based key as their identifier. Assets stored in different areas of the cache can have the
    * same key, for example 'playerWalking' could be used as the key for both a sprite sheet and an audio file,
    * because they are unique data types.
    * 
    * The cache is automatically populated by the Phaser.Loader. When you use the loader to pull in external assets
    * such as images they are automatically placed into their respective cache. Most common Game Objects, such as
    * Sprites and Videos automatically query the cache to extract the assets they need on instantiation.
    * 
    * You can access the cache from within a State via `this.cache`. From here you can call any public method it has,
    * including adding new entries to it, deleting them or querying them.
    * 
    * Understand that almost without exception when you get an item from the cache it will return a reference to the
    * item stored in the cache, not a copy of it. Therefore if you retrieve an item and then modify it, the original
    * object in the cache will also be updated, even if you don't put it back into the cache again.
    * 
    * By default when you change State the cache is _not_ cleared, although there is an option to clear it should
    * your game require it. In a typical game set-up the cache is populated once after the main game has loaded and
    * then used as an asset store.
    */
    class Cache {


        /**
        * Phaser has one single cache in which it stores all assets.
        * 
        * The cache is split up into sections, such as images, sounds, video, json, etc. All assets are stored using
        * a unique string-based key as their identifier. Assets stored in different areas of the cache can have the
        * same key, for example 'playerWalking' could be used as the key for both a sprite sheet and an audio file,
        * because they are unique data types.
        * 
        * The cache is automatically populated by the Phaser.Loader. When you use the loader to pull in external assets
        * such as images they are automatically placed into their respective cache. Most common Game Objects, such as
        * Sprites and Videos automatically query the cache to extract the assets they need on instantiation.
        * 
        * You can access the cache from within a State via `this.cache`. From here you can call any public method it has,
        * including adding new entries to it, deleting them or querying them.
        * 
        * Understand that almost without exception when you get an item from the cache it will return a reference to the
        * item stored in the cache, not a copy of it. Therefore if you retrieve an item and then modify it, the original
        * object in the cache will also be updated, even if you don't put it back into the cache again.
        * 
        * By default when you change State the cache is _not_ cleared, although there is an option to clear it should
        * your game require it. In a typical game set-up the cache is populated once after the main game has loaded and
        * then used as an asset store.
        * 
        * @param game A reference to the currently running game.
        */
        constructor(game: Phaser.Game);

        static BINARY: number;
        static BITMAPDATA: number;
        static BITMAPFONT: number;
        static CANVAS: number;
        static IMAGE: number;
        static JSON: number;
        static PHYSICS: number;
        static RENDER_TEXTURE: number;
        static SHADER: number;
        static SOUND: number;
        static SPRITE_SHEET: number;
        static TEXT: number;
        static TEXTURE: number;
        static TEXTURE_ATLAS: number;
        static TILEMAP: number;
        static XML: number;
        static VIDEO: number;


        /**
        * The default image used for a texture when no other is specified.
        */
        static DEFAULT: PIXI.Texture;

        /**
        * The default image used for a texture when the source image is missing.
        */
        static MISSING: PIXI.Texture;


        /**
        * Automatically resolve resource URLs to absolute paths for use with the Cache.getURL method.
        */
        autoResolveURL: boolean;

        /**
        * Local reference to game.
        */
        game: Phaser.Game;

        /**
        * This event is dispatched when the sound system is unlocked via a touch event on cellular devices.
        */
        onSoundUnlock: Phaser.Signal;


        /**
        * Add a binary object in to the cache.
        * 
        * @param key The key that this asset will be stored in the cache under. This should be unique within this cache.
        * @param binaryData The binary object to be added to the cache.
        */
        addBinary(key: string, binaryData: any): void;

        /**
        * Add a BitmapData object to the cache.
        * 
        * @param key The key that this asset will be stored in the cache under. This should be unique within this cache.
        * @param bitmapData The BitmapData object to be addded to the cache.
        * @param frameData Optional FrameData set associated with the given BitmapData. If not specified (or `undefined`) a new FrameData object is created containing the Bitmap's Frame. If `null` is supplied then no FrameData will be created. - Default: (auto create)
        * @return The BitmapData object to be addded to the cache.
        */
        addBitmapData(key: string, bitmapData: Phaser.BitmapData, frameData?: Phaser.FrameData): Phaser.BitmapData;

        /**
        * Add a new Bitmap Font to the Cache.
        * 
        * @param key The key that this asset will be stored in the cache under. This should be unique within this cache.
        * @param url The URL the asset was loaded from. If the asset was not loaded externally set to `null`.
        * @param data Extra font data.
        * @param atlasData The Bitmap Font data.
        * @param atlasType The format of the Bitmap Font data file: `json` or `xml`. - Default: 'xml'
        * @param xSpacing If you'd like to add additional horizontal spacing between the characters then set the pixel value here.
        * @param ySpacing If you'd like to add additional vertical spacing between the lines then set the pixel value here.
        */
        addBitmapFont(key: string, url: string, data: any, atlasData: any, atlasType: string, xSpacing?: number, ySpacing?: number): void;

        /**
        * Add a new Bitmap Font to the Cache, where the font texture is part of a Texture Atlas.
        * 
        * The atlas must already exist in the cache, and be available based on the given `atlasKey`.
        * 
        * The `atlasFrame` specifies the name of the frame within the atlas that the Bitmap Font is
        * stored in.
        * 
        * The `dataKey` is the key of the XML or JSON Bitmap Font Data, which must already be in
        * the Cache.
        * 
        * @param key The key that this asset will be stored in the cache under. This should be unique within this cache.
        * @param atlasKey The key of the Texture Atlas in the Cache.
        * @param atlasFrame The frame of the Texture Atlas that the Bitmap Font is in.
        * @param dataKey The key of the Bitmap Font data in the Cache
        * @param dataType The format of the Bitmap Font data: `json` or `xml`. - Default: 'xml'
        * @param xSpacing If you'd like to add additional horizontal spacing between the characters then set the pixel value here.
        * @param ySpacing If you'd like to add additional vertical spacing between the lines then set the pixel value here.
        */
        addBitmapFontFromAtlas(key: string, atlasKey: string, atlasFrame: string, dataKey: string, dataType?: string, xSpacing?: number, ySpacing?: number): void;

        /**
        * Add a new canvas object in to the cache.
        * 
        * @param key The key that this asset will be stored in the cache under. This should be unique within this cache.
        * @param canvas The Canvas DOM element.
        * @param context The context of the canvas element. If not specified it will default go `getContext('2d')`.
        */
        addCanvas(key: string, canvas: HTMLCanvasElement, context?: CanvasRenderingContext2D): void;

        /**
        * Adds a default image to be used in special cases such as WebGL Filters.
        * It uses the special reserved key of `__default`.
        * This method is called automatically when the Cache is created.
        * This image is skipped when `Cache.destroy` is called due to its internal requirements.
        */
        addDefaultImage(): void;

        /**
        * Adds an Image file into the Cache. The file must have already been loaded, typically via Phaser.Loader, but can also have been loaded into the DOM.
        * If an image already exists in the cache with the same key then it is removed and destroyed, and the new image inserted in its place.
        * 
        * If the image has not yet been fetched (successfully or not), a `console.warn` message will be displayed.
        * 
        * @param key The key that this asset will be stored in the cache under. This should be unique within this cache.
        * @param url The URL the asset was loaded from. If the asset was not loaded externally set to `null`.
        * @param data Extra image data.
        * @return The full image object that was added to the cache.
        */
        addImage(key: string, url: string, data: any): HTMLImageElement;

        /**
        * Add a new json object into the cache.
        * 
        * @param key The key that this asset will be stored in the cache under. This should be unique within this cache.
        * @param url The URL the asset was loaded from. If the asset was not loaded externally set to `null`.
        * @param data Extra json data.
        */
        addJSON(key: string, urL: string, data: any): void;

        /**
        * Adds an image to be used when a key is wrong / missing.
        * It uses the special reserved key of `__missing`.
        * This method is called automatically when the Cache is created.
        * This image is skipped when `Cache.destroy` is called due to its internal requirements.
        */
        addMissingImage(): void;

        /**
        * Add a new physics data object to the Cache.
        * 
        * @param key The key that this asset will be stored in the cache under. This should be unique within this cache.
        * @param url The URL the asset was loaded from. If the asset was not loaded externally set to `null`.
        * @param JSONData The physics data object (a JSON file).
        * @param format The format of the physics data.
        */
        addPhysicsData(key: string, url: string, JSONData: any, format: number): void;

        /**
        * Add a new Phaser.RenderTexture in to the cache.
        * 
        * @param key The key that this asset will be stored in the cache under. This should be unique within this cache.
        * @param texture The texture to use as the base of the RenderTexture.
        */
        addRenderTexture(key: string, texture: RenderTexture): void;

        /**
        * Adds a Fragment Shader in to the Cache. The file must have already been loaded, typically via Phaser.Loader.
        * 
        * @param key The key that this asset will be stored in the cache under. This should be unique within this cache.
        * @param url The URL the asset was loaded from. If the asset was not loaded externally set to `null`.
        * @param data Extra shader data.
        */
        addShader(key: string, url: string, data: any): void;

        /**
        * Adds a Sound file into the Cache. The file must have already been loaded, typically via Phaser.Loader.
        * 
        * @param key The key that this asset will be stored in the cache under. This should be unique within this cache.
        * @param url The URL the asset was loaded from. If the asset was not loaded externally set to `null`.
        * @param data Extra sound data.
        * @param webAudio True if the file is using web audio.
        * @param audioTag True if the file is using legacy HTML audio.
        */
        addSound(key: string, url: string, data: any, webAudio: boolean, audioTag: boolean): void;

        /**
        * Add a new sprite sheet in to the cache.
        * 
        * @param key The key that this asset will be stored in the cache under. This should be unique within this cache.
        * @param url The URL the asset was loaded from. If the asset was not loaded externally set to `null`.
        * @param data Extra sprite sheet data.
        * @param frameWidth Width of the sprite sheet.
        * @param frameHeight Height of the sprite sheet.
        * @param frameMax How many frames stored in the sprite sheet. If -1 then it divides the whole sheet evenly. - Default: -1
        * @param margin If the frames have been drawn with a margin, specify the amount here.
        * @param spacing If the frames have been drawn with spacing between them, specify the amount here.
        * @param skipFrames Skip a number of frames. Useful when there are multiple sprite sheets in one image.
        */
        addSpriteSheet(key: string, url: string, data: any, frameWidth: number, frameHeight: number, frameMax?: number, margin?: number, spacing?: number, skipFrames?: number): void;

        /**
        * Add a new text data.
        * 
        * @param key The key that this asset will be stored in the cache under. This should be unique within this cache.
        * @param url The URL the asset was loaded from. If the asset was not loaded externally set to `null`.
        * @param data Extra text data.
        */
        addText(key: string, url: string, data: any): void;

        /**
        * Add a new texture atlas to the Cache.
        * 
        * @param key The key that this asset will be stored in the cache under. This should be unique within this cache.
        * @param url The URL the asset was loaded from. If the asset was not loaded externally set to `null`.
        * @param data Extra texture atlas data.
        * @param atlasData Texture atlas frames data.
        * @param format The format of the texture atlas.
        */
        addTextureAtlas(key: string, url: string, data: any, atlasData: any, format: number): void;

        /**
        * Add a new tilemap to the Cache.
        * 
        * @param key The key that this asset will be stored in the cache under. This should be unique within this cache.
        * @param url The URL the asset was loaded from. If the asset was not loaded externally set to `null`.
        * @param mapData The tilemap data object (either a CSV or JSON file).
        * @param format The format of the tilemap data.
        */
        addTilemap(key: string, url: string, mapData: any, format: number): void;

        /**
        * Adds a Video file into the Cache. The file must have already been loaded, typically via Phaser.Loader.
        * 
        * @param key The key that this asset will be stored in the cache under. This should be unique within this cache.
        * @param url The URL the asset was loaded from. If the asset was not loaded externally set to `null`.
        * @param data Extra video data.
        * @param isBlob True if the file was preloaded via xhr and the data parameter is a Blob. false if a Video tag was created instead.
        */
        addVideo(key: string, url: string, data: any, isBlob?: boolean): void;

        /**
        * Add a new xml object into the cache.
        * 
        * @param key The key that this asset will be stored in the cache under. This should be unique within this cache.
        * @param url The URL the asset was loaded from. If the asset was not loaded externally set to `null`.
        * @param data Extra text data.
        */
        addXML(key: string, url: string, data: any): void;

        /**
        * Checks if the given key exists in the Binary Cache.
        * 
        * @param key The key of the asset within the cache.
        * @return True if the key exists in the cache, otherwise false.
        */
        checkBinaryKey(key: string): boolean;

        /**
        * Checks if the given key exists in the BitmapData Cache.
        * 
        * @param key The key of the asset within the cache.
        * @return True if the key exists in the cache, otherwise false.
        */
        checkBitmapDataKey(key: string): boolean;

        /**
        * Checks if the given key exists in the BitmapFont Cache.
        * 
        * @param key The key of the asset within the cache.
        * @return True if the key exists in the cache, otherwise false.
        */
        checkBitmapFontKey(key: string): boolean;

        /**
        * Checks if the given key exists in the Canvas Cache.
        * 
        * @param key The key of the asset within the cache.
        * @return True if the key exists in the cache, otherwise false.
        */
        checkCanvasKey(key: string): boolean;

        /**
        * Checks if the given key exists in the Image Cache. Note that this also includes Texture Atlases, Sprite Sheets and Retro Fonts.
        * 
        * @param key The key of the asset within the cache.
        * @return True if the key exists in the cache, otherwise false.
        */
        checkImageKey(key: string): boolean;

        /**
        * Checks if the given key exists in the JSON Cache.
        * 
        * @param key The key of the asset within the cache.
        * @return True if the key exists in the cache, otherwise false.
        */
        checkJSONKey(key: string): boolean;

        /**
        * Checks if a key for the given cache object type exists.
        * 
        * @param cache The cache to search. One of the Cache consts such as `Phaser.Cache.IMAGE` or `Phaser.Cache.SOUND`.
        * @param key The key of the asset within the cache.
        * @return True if the key exists, otherwise false.
        */
        checkKey(cache: number, key: string): boolean;

        /**
        * Checks if the given key exists in the Physics Cache.
        * 
        * @param key The key of the asset within the cache.
        * @return True if the key exists in the cache, otherwise false.
        */
        checkPhysicsKey(key: string): boolean;

        /**
        * Checks if the given key exists in the Render Texture Cache.
        * 
        * @param key The key of the asset within the cache.
        * @return True if the key exists in the cache, otherwise false.
        */
        checkRenderTextureKey(key: string): boolean;

        /**
        * Checks if the given key exists in the Fragment Shader Cache.
        * 
        * @param key The key of the asset within the cache.
        * @return True if the key exists in the cache, otherwise false.
        */
        checkShaderKey(key: string): boolean;

        /**
        * Checks if the given key exists in the Sound Cache.
        * 
        * @param key The key of the asset within the cache.
        * @return True if the key exists in the cache, otherwise false.
        */
        checkSoundKey(key: string): boolean;

        /**
        * Checks if the given key exists in the Text Cache.
        * 
        * @param key The key of the asset within the cache.
        * @return True if the key exists in the cache, otherwise false.
        */
        checkTextKey(key: string): boolean;

        /**
        * Checks if the given key exists in the Texture Cache.
        * 
        * @param key The key of the asset within the cache.
        * @return True if the key exists in the cache, otherwise false.
        */
        checkTextureKey(key: string): boolean;

        /**
        * Checks if the given key exists in the Tilemap Cache.
        * 
        * @param key The key of the asset within the cache.
        * @return True if the key exists in the cache, otherwise false.
        */
        checkTilemapKey(key: string): boolean;

        /**
        * Checks if the given URL has been loaded into the Cache.
        * This method will only work if Cache.autoResolveURL was set to `true` before any preloading took place.
        * The method will make a DOM src call to the URL given, so please be aware of this for certain file types, such as Sound files on Firefox
        * which may cause double-load instances.
        * 
        * @param url The url to check for in the cache.
        * @return True if the url exists, otherwise false.
        */
        checkURL(url: string): any;
        checkUrl(url: string): any;

        /**
        * Checks if the given key exists in the XML Cache.
        * 
        * @param key The key of the asset within the cache.
        * @return True if the key exists in the cache, otherwise false.
        */
        checkXMLKey(key: string): boolean;

        /**
        * Checks if the given key exists in the Video Cache.
        * 
        * @param key The key of the asset within the cache.
        * @return True if the key exists in the cache, otherwise false.
        */
        checkVideoKey(key: string): boolean;

        /**
        * Empties out all of the GL Textures from Images stored in the cache.
        * This is called automatically when the WebGL context is lost and then restored.
        */
        clearGLTextures(): void;

        /**
        * Add a new decoded sound.
        * 
        * @param key The key of the asset within the cache.
        * @param data Extra sound data.
        */
        decodedSound(key: string, data: any): void;

        /**
        * Clears the cache. Removes every local cache object reference.
        * If an object in the cache has a `destroy` method it will be called;
        * otherwise, `destroy` will be called on any of the object's `base`, `data`,
        * `frameData`, or `texture` properties.
        */
        destroy(): void;

        /**
        * Gets a PIXI.BaseTexture by key from the given Cache.
        * 
        * @param key Asset key of the image for which you want the BaseTexture for.
        * @param cache The cache to search for the item in. - Default: Phaser.Cache.IMAGE
        * @return The BaseTexture object.
        */
        getBaseTexture(key: string, cache?: number): PIXI.BaseTexture;

        /**
        * Gets a binary object from the cache.
        * 
        * The object is looked-up based on the key given.
        * 
        * Note: If the object cannot be found a `console.warn` message is displayed.
        * 
        * @param key The key of the asset to retrieve from the cache.
        * @return The binary data object.
        */
        getBinary(key: string): any;

        /**
        * Gets a BitmapData object from the cache.
        * 
        * The object is looked-up based on the key given.
        * 
        * Note: If the object cannot be found a `console.warn` message is displayed.
        * 
        * @param key The key of the asset to retrieve from the cache.
        * @return The requested BitmapData object if found, or null if not.
        */
        getBitmapData(key: string): Phaser.BitmapData;

        /**
        * Gets a Bitmap Font object from the cache.
        * 
        * The object is looked-up based on the key given.
        * 
        * Note: If the object cannot be found a `console.warn` message is displayed.
        * 
        * @param key The key of the asset to retrieve from the cache.
        * @return The requested BitmapFont object if found, or null if not.
        */
        getBitmapFont(key: string): Phaser.BitmapFont;

        /**
        * Gets a Canvas object from the cache.
        * 
        * The object is looked-up based on the key given.
        * 
        * Note: If the object cannot be found a `console.warn` message is displayed.
        * 
        * @param key The key of the asset to retrieve from the cache.
        * @return The canvas object or `null` if no item could be found matching the given key.
        */
        getCanvas(key: string): HTMLCanvasElement;

        /**
        * Get a single frame by key. You'd only do this to get the default Frame created for a non-atlas/spritesheet image.
        * 
        * @param key Asset key of the frame data to retrieve from the Cache.
        * @param cache The cache to search for the item in. - Default: Phaser.Cache.IMAGE
        * @return The frame data.
        */
        getFrame(key: string, cache?: number): Phaser.Frame;

        /**
        * Get a single frame out of a frameData set by key.
        * 
        * @param key Asset key of the frame data to retrieve from the Cache.
        * @param index The index of the frame you want to get.
        * @param cache The cache to search. One of the Cache consts such as `Phaser.Cache.IMAGE` or `Phaser.Cache.SOUND`. - Default: Phaser.Cache.IMAGE
        * @return The frame object.
        */
        getFrameByIndex(key: string, index: number, cache?: number): Phaser.Frame;

        /**
        * Get a single frame out of a frameData set by key.
        * 
        * @param key Asset key of the frame data to retrieve from the Cache.
        * @param name The name of the frame you want to get.
        * @param cache The cache to search. One of the Cache consts such as `Phaser.Cache.IMAGE` or `Phaser.Cache.SOUND`. - Default: Phaser.Cache.IMAGE
        * @return The frame object.
        */
        getFrameByName(key: string, name: string, cache?: number): Phaser.Frame;

        /**
        * Get the total number of frames contained in the FrameData object specified by the given key.
        * 
        * @param key Asset key of the FrameData you want.
        * @param cache The cache to search for the item in. - Default: Phaser.Cache.IMAGE
        * @return Then number of frames. 0 if the image is not found.
        */
        getFrameCount(key: string, cache?: number): number;

        /**
        * Gets a Phaser.FrameData object from the Image Cache.
        * 
        * The object is looked-up based on the key given.
        * 
        * Note: If the object cannot be found a `console.warn` message is displayed.
        * 
        * @param key Asset key of the frame data to retrieve from the Cache.
        * @param cache The cache to search for the item in. - Default: Phaser.Cache.IMAGE
        * @return The frame data.
        */
        getFrameData(key: string, cache?: number): Phaser.FrameData;

        /**
        * Gets a Image object from the cache. This returns a DOM Image object, not a Phaser.Image object.
        * 
        * The object is looked-up based on the key given.
        * 
        * Note: If the object cannot be found a `console.warn` message is displayed.
        * 
        * Only the Image cache is searched, which covers images loaded via Loader.image, Sprite Sheets and Texture Atlases.
        * 
        * If you need the image used by a bitmap font or similar then please use those respective 'get' methods.
        * 
        * @param key The key of the asset to retrieve from the cache. If not given or null it will return a default image. If given but not found in the cache it will throw a warning and return the missing image.
        * @param full If true the full image object will be returned, if false just the HTML Image object is returned.
        * @return The Image object if found in the Cache, otherwise `null`. If `full` was true then a JavaScript object is returned.
        */
        getImage(key: string, full?: boolean): HTMLImageElement;

        /**
        * Get an item from a cache based on the given key and property.
        * 
        * This method is mostly used internally by other Cache methods such as `getImage` but is exposed
        * publicly for your own use as well.
        * 
        * @param key The key of the asset within the cache.
        * @param cache The cache to search. One of the Cache consts such as `Phaser.Cache.IMAGE` or `Phaser.Cache.SOUND`.
        * @param method The string name of the method calling getItem. Can be empty, in which case no console warning is output.
        * @param property If you require a specific property from the cache item, specify it here.
        * @return The cached item if found, otherwise `null`. If the key is invalid and `method` is set then a console.warn is output.
        */
        getItem(key: string, cache: number, method?: string, property?: string): any;

        /**
        * Gets a JSON object from the cache.
        * 
        * The object is looked-up based on the key given.
        * 
        * Note: If the object cannot be found a `console.warn` message is displayed.
        * 
        * You can either return the object by reference (the default), or return a clone
        * of it by setting the `clone` argument to `true`.
        * 
        * @param key The key of the asset to retrieve from the cache.
        * @param clone Return a clone of the original object (true) or a reference to it? (false)
        * @return The JSON object, or an Array if the key points to an Array property. If the property wasn't found, it returns null.
        */
        getJSON(key: string, clone?: boolean): any;

        /**
        * Gets all keys used in the requested Cache.
        * 
        * @param cache The Cache you wish to get the keys from. Can be any of the Cache consts such as `Phaser.Cache.IMAGE`, `Phaser.Cache.SOUND` etc. - Default: Phaser.Cache.IMAGE
        * @return The array of keys in the requested cache.
        */
        getKeys(cache: number): string[];

        /**
        * Gets a Physics Data object from the cache.
        * 
        * The object is looked-up based on the key given.
        * 
        * Note: If the object cannot be found a `console.warn` message is displayed.
        * 
        * You can get either the entire data set, a single object or a single fixture of an object from it.
        * 
        * @param key The key of the asset to retrieve from the cache.
        * @param object If specified it will return just the physics object that is part of the given key, if null it will return them all.
        * @param fixtureKey Fixture key of fixture inside an object. This key can be set per fixture with the Phaser Exporter.
        * @return The requested physics object data if found.
        */
        getPhysicsData(key: string, object?: string, fixtureKey?: string): any[];

        /**
        * Gets a RenderTexture object from the cache.
        * 
        * The object is looked-up based on the key given.
        * 
        * Note: If the object cannot be found a `console.warn` message is displayed.
        * 
        * @param key The key of the asset to retrieve from the cache.
        * @return The object with Phaser.RenderTexture and Phaser.Frame.
        */
        getRenderTexture(key: string): Phaser.CachedRenderTexture;

        /**
        * Gets a fragment shader object from the cache.
        * 
        * The object is looked-up based on the key given.
        * 
        * Note: If the object cannot be found a `console.warn` message is displayed.
        * 
        * @param key The key of the asset to retrieve from the cache.
        * @return The shader object.
        */
        getShader(key: string): string;

        /**
        * Gets a Phaser.Sound object from the cache.
        * 
        * The object is looked-up based on the key given.
        * 
        * Note: If the object cannot be found a `console.warn` message is displayed.
        * 
        * @param key The key of the asset to retrieve from the cache.
        * @return The sound object.
        */
        getSound(key: string): Phaser.Sound;

        /**
        * Gets a raw Sound data object from the cache.
        * 
        * The object is looked-up based on the key given.
        * 
        * Note: If the object cannot be found a `console.warn` message is displayed.
        * 
        * @param key The key of the asset to retrieve from the cache.
        * @return The sound data.
        */
        getSoundData(key: string): any;
        getSpriteSheetKey(key: string): boolean;

        /**
        * Gets a Text object from the cache.
        * 
        * The object is looked-up based on the key given.
        * 
        * Note: If the object cannot be found a `console.warn` message is displayed.
        * 
        * @param key The key of the asset to retrieve from the cache.
        * @return The text data.
        */
        getText(key: string): string;
        getTextKeys(): string[];
        getTexture(key: string): Phaser.RenderTexture;
        getTextureAtlasKey(key: string): boolean;

        /**
        * Get a single texture frame by key.
        * 
        * You'd only do this to get the default Frame created for a non-atlas / spritesheet image.
        * 
        * @param key The key of the asset to retrieve from the cache.
        * @return The frame data.
        */
        getTextureFrame(key: string): Phaser.Frame;
        getTilemap(key: string): any;

        /**
        * Gets a raw Tilemap data object from the cache. This will be in either CSV or JSON format.
        * 
        * The object is looked-up based on the key given.
        * 
        * Note: If the object cannot be found a `console.warn` message is displayed.
        * 
        * @param key The key of the asset to retrieve from the cache.
        * @return The raw tilemap data in CSV or JSON format.
        */
        getTilemapData(key: string): any;

        /**
        * Get a cached object by the URL.
        * This only returns a value if you set Cache.autoResolveURL to `true` *before* starting the preload of any assets.
        * Be aware that every call to this function makes a DOM src query, so use carefully and double-check for implications in your target browsers/devices.
        * 
        * @param url The url for the object loaded to get from the cache.
        * @return The cached object.
        */
        getURL(url: string): any;

        /**
        * Gets an XML object from the cache.
        * 
        * The object is looked-up based on the key given.
        * 
        * Note: If the object cannot be found a `console.warn` message is displayed.
        * 
        * @param key The key of the asset to retrieve from the cache.
        * @return The XML object.
        */
        getXML(key: string): any;

        /**
        * Gets a Phaser.Video object from the cache.
        * 
        * The object is looked-up based on the key given.
        * 
        * Note: If the object cannot be found a `console.warn` message is displayed.
        * 
        * @param key The key of the asset to retrieve from the cache.
        * @return The video object.
        */
        getVideo(key: string): Phaser.Video;

        /**
        * Check if the FrameData for the given key exists in the Image Cache.
        * 
        * @param key Asset key of the frame data to retrieve from the Cache.
        * @param cache The cache to search for the item in. - Default: Phaser.Cache.IMAGE
        * @return True if the given key has frameData in the cache, otherwise false.
        */
        hasFrameData(key: string, cache?: number): boolean;

        /**
        * Check if the given sound has finished decoding.
        * 
        * @param key The key of the asset within the cache.
        * @return The decoded state of the Sound object.
        */
        isSoundDecoded(key: string): boolean;

        /**
        * Check if the given sound is ready for playback.
        * A sound is considered ready when it has finished decoding and the device is no longer touch locked.
        * 
        * @param key The key of the asset within the cache.
        * @return True if the sound is decoded and the device is not touch locked.
        */
        isSoundReady(key: string): boolean;
        isSpriteSheet(key: string): boolean;

        /**
        * Reload a Sound file from the server.
        * 
        * @param key The key of the asset within the cache.
        */
        reloadSound(key: string): void;

        /**
        * Fires the onSoundUnlock event when the sound has completed reloading.
        * 
        * @param key The key of the asset within the cache.
        */
        reloadSoundComplete(key: string): void;

        /**
        * Removes a binary file from the cache.
        * 
        * Note that this only removes it from the Phaser.Cache. If you still have references to the data elsewhere
        * then it will persist in memory.
        * 
        * @param key Key of the asset you want to remove.
        */
        removeBinary(key: string): void;

        /**
        * Removes a bitmap data from the cache.
        * 
        * Note that this only removes it from the Phaser.Cache. If you still have references to the data elsewhere
        * then it will persist in memory.
        * 
        * @param key Key of the asset you want to remove.
        */
        removeBitmapData(key: string): void;

        /**
        * Removes a bitmap font from the cache.
        * 
        * Note that this only removes it from the Phaser.Cache. If you still have references to the data elsewhere
        * then it will persist in memory.
        * 
        * @param key Key of the asset you want to remove.
        */
        removeBitmapFont(key: string): void;

        /**
        * Removes a canvas from the cache.
        * 
        * Note that this only removes it from the Phaser.Cache. If you still have references to the data elsewhere
        * then it will persist in memory.
        * 
        * @param key Key of the asset you want to remove.
        */
        removeCanvas(key: string): void;

        /**
        * Removes an image from the cache.
        * 
        * You can optionally elect to destroy it as well. This calls BaseTexture.destroy on it.
        * 
        * Note that this only removes it from the Phaser Cache. If you still have references to the data elsewhere
        * then it will persist in memory.
        * 
        * @param key Key of the asset you want to remove.
        * @param destroyBaseTexture Should the BaseTexture behind this image also be destroyed? - Default: true
        */
        removeImage(key: string, removeFromPixi?: boolean): void;

        /**
        * Removes a json object from the cache.
        * 
        * Note that this only removes it from the Phaser.Cache. If you still have references to the data elsewhere
        * then it will persist in memory.
        * 
        * @param key Key of the asset you want to remove.
        */
        removeJSON(key: string): void;

        /**
        * Removes a physics data file from the cache.
        * 
        * Note that this only removes it from the Phaser.Cache. If you still have references to the data elsewhere
        * then it will persist in memory.
        * 
        * @param key Key of the asset you want to remove.
        */
        removePhysics(key: string): void;

        /**
        * Removes a Render Texture from the cache.
        * 
        * Note that this only removes it from the Phaser.Cache. If you still have references to the data elsewhere
        * then it will persist in memory.
        * 
        * @param key Key of the asset you want to remove.
        */
        removeRenderTexture(key: string): void;

        /**
        * Removes a shader from the cache.
        * 
        * Note that this only removes it from the Phaser.Cache. If you still have references to the data elsewhere
        * then it will persist in memory.
        * 
        * @param key Key of the asset you want to remove.
        */
        removeShader(key: string): void;

        /**
        * Removes a sound from the cache.
        * 
        * If any `Phaser.Sound` objects use the audio file in the cache that you remove with this method, they will
        * _automatically_ destroy themselves. If you wish to have full control over when Sounds are destroyed then
        * you must finish your house-keeping and destroy them all yourself first, before calling this method.
        * 
        * Note that this only removes it from the Phaser.Cache. If you still have references to the data elsewhere
        * then it will persist in memory.
        * 
        * @param key Key of the asset you want to remove.
        */
        removeSound(key: string): void;

        /**
        * Removes a Sprite Sheet from the cache.
        * 
        * Note that this only removes it from the Phaser.Cache. If you still have references to the data elsewhere
        * then it will persist in memory.
        * 
        * @param key Key of the asset you want to remove.
        */
        removeSpriteSheet(key: string): void;

        /**
        * Removes a text file from the cache.
        * 
        * Note that this only removes it from the Phaser.Cache. If you still have references to the data elsewhere
        * then it will persist in memory.
        * 
        * @param key Key of the asset you want to remove.
        */
        removeText(key: string): void;

        /**
        * Removes a Texture Atlas from the cache.
        * 
        * Note that this only removes it from the Phaser.Cache. If you still have references to the data elsewhere
        * then it will persist in memory.
        * 
        * @param key Key of the asset you want to remove.
        */
        removeTextureAtlas(key: string): void;

        /**
        * Removes a tilemap from the cache.
        * 
        * Note that this only removes it from the Phaser.Cache. If you still have references to the data elsewhere
        * then it will persist in memory.
        * 
        * @param key Key of the asset you want to remove.
        */
        removeTilemap(key: string): void;

        /**
        * Removes a xml object from the cache.
        * 
        * Note that this only removes it from the Phaser.Cache. If you still have references to the data elsewhere
        * then it will persist in memory.
        * 
        * @param key Key of the asset you want to remove.
        */
        removeXML(key: string): void;

        /**
        * Removes a video from the cache.
        * 
        * Note that this only removes it from the Phaser.Cache. If you still have references to the data elsewhere
        * then it will persist in memory.
        * 
        * @param key Key of the asset you want to remove.
        */
        removeVideo(key: string): void;

        /**
        * Replaces a set of frameData with a new Phaser.FrameData object.
        * 
        * @param key The unique key by which you will reference this object.
        * @param frameData The new FrameData.
        * @param cache The cache to search. One of the Cache consts such as `Phaser.Cache.IMAGE` or `Phaser.Cache.SOUND`. - Default: Phaser.Cache.IMAGE
        */
        updateFrameData(key: string, frameData: any, cache?: number): void;

        /**
        * Updates the sound object in the cache.
        * 
        * @param key The key of the asset within the cache.
        */
        updateSound(key: string, property: string, value: Phaser.Sound): void;

    }

    interface CachedRenderTexture {

        frame: Phaser.Frame;
        texture: Phaser.RenderTexture;

    }


    /**
    * A Camera is your view into the game world. It has a position and size and renders only those objects within its field of view.
    * The game automatically creates a single Stage sized camera on boot. Move the camera around the world with Phaser.Camera.x/y
    */
    class Camera {


        /**
        * A Camera is your view into the game world. It has a position and size and renders only those objects within its field of view.
        * The game automatically creates a single Stage sized camera on boot. Move the camera around the world with Phaser.Camera.x/y
        * 
        * @param game Game reference to the currently running game.
        * @param id Not being used at the moment, will be when Phaser supports multiple camera
        * @param x Position of the camera on the X axis
        * @param y Position of the camera on the Y axis
        * @param width The width of the view rectangle
        * @param height The height of the view rectangle
        */
        constructor(game: Phaser.Game, id: number, x: number, y: number, width: number, height: number);


        /**
        * A follow style that uses no deadzone.
        */
        static FOLLOW_LOCKON: number;

        /**
        * A follow style that uses a tall, narrow deadzone (0.33 x 0.125) with a center slightly above the view center.
        */
        static FOLLOW_PLATFORMER: number;

        /**
        * A follow style that uses a square deadzone (0.25 of the larger view edge).
        */
        static FOLLOW_TOPDOWN: number;

        /**
        * A follow style that uses a small square deadzone (0.125 of the larger view edge).
        */
        static FOLLOW_TOPDOWN_TIGHT: number;
        static SHAKE_BOTH: number;
        static SHAKE_HORIZONTAL: number;
        static SHAKE_VERTICAL: number;
        static ENABLE_FX: number;


        /**
        * Whether this camera is flush with the World Bounds or not.
        */

                   /**
                   * The Cameras x coordinate. This value is automatically clamped if it falls outside of the World bounds. Gets or sets the cameras x position.
                   */

                               /**
                               * The Cameras y coordinate. This value is automatically clamped if it falls outside of the World bounds. Gets or sets the cameras y position.
                               */
        atLimit: { x: boolean; y: boolean; };

        /**
        * The Camera is bound to this Rectangle and cannot move outside of it. By default it is enabled and set to the size of the World.
        * The Rectangle can be located anywhere in the world and updated as often as you like. If you don't wish the Camera to be bound
        * at all then set this to null. The values can be anything and are in World coordinates, with 0,0 being the top-left of the world. The Rectangle in which the Camera is bounded. Set to null to allow for movement anywhere.
        */
        bounds: Phaser.Rectangle;

        /**
        * Moving inside this Rectangle will not cause the camera to move.
        */
        deadzone: Phaser.Rectangle;

        /**
        * The display object to which all game objects are added. Set by World.boot.
        */
        displayObject: PIXI.DisplayObject;

        /**
        * Reserved for future multiple camera set-ups.
        */
        id: number;

        /**
        * The Graphics object used to handle camera fx such as fade and flash.
        */
        fx: Phaser.Graphics;

        /**
        * A reference to the currently running Game.
        */
        game: Phaser.Game;

        /**
        * The Cameras height. By default this is the same as the Game size and should not be adjusted for now. Gets or sets the cameras height.
        */
        height: number;

        /**
        * The linear interpolation value to use when following a target.
        * The default values of 1 means the camera will instantly snap to the target coordinates.
        * A lower value, such as 0.1 means the camera will more slowly track the target, giving
        * a smooth transition. You can set the horizontal and vertical values independently, and also
        * adjust this value in real-time during your game.
        */
        lerp: Phaser.Point;

        /**
        * The Cameras position. This value is automatically clamped if it falls outside of the World bounds. Gets or sets the cameras xy position using Phaser.Point object.
        */
        position: Phaser.Point;

        /**
        * If a Camera has roundPx set to `true` it will call `view.floor` as part of its update loop, keeping its boundary to integer values. Set this to `false` to disable this from happening.
        * Default: true
        */
        roundPx: boolean;

        /**
        * The scale of the display object to which all game objects are added. Set by World.boot.
        */
        scale: Phaser.Point;

        /**
        * The Cameras shake intensity. Gets or sets the cameras shake intensity.
        */
        shakeIntensity: number;

        /**
        * This signal is dispatched when the camera fade effect completes.
        * When the fade effect completes you will be left with the screen black (or whatever
        * color you faded to). In order to reset this call `Camera.resetFX`. This is called
        * automatically when you change State.
        */
        onFadeComplete: Phaser.Signal;

        /**
        * This signal is dispatched when the camera flash effect completes.
        */
        onFlashComplete: Phaser.Signal;

        /**
        * This signal is dispatched when the camera shake effect completes.
        */
        onShakeComplete: Phaser.Signal;

        /**
        * If the camera is tracking a Sprite, this is a reference to it, otherwise null.
        */
        target: Phaser.Sprite;

        /**
        * The total number of Sprites with `autoCull` set to `true` that are visible by this Camera.
        */
        totalInView: number;

        /**
        * Camera view.
        * The view into the world we wish to render (by default the game dimensions).
        * The x/y values are in world coordinates, not screen coordinates, the width/height is how many pixels to render.
        * Sprites outside of this view are not rendered if Sprite.autoCull is set to `true`. Otherwise they are always rendered.
        */
        view: Phaser.Rectangle;

        /**
        * Whether this camera is visible or not.
        * Default: true
        */
        visible: boolean;

        /**
        * The Cameras width. By default this is the same as the Game size and should not be adjusted for now. Gets or sets the cameras width.
        */
        width: number;

        /**
        * A reference to the game world.
        */
        world: Phaser.World;

        /**
        * The Cameras x coordinate. This value is automatically clamped if it falls outside of the World bounds. Gets or sets the cameras x position.
        */
        x: number;

        /**
        * The Cameras y coordinate. This value is automatically clamped if it falls outside of the World bounds. Gets or sets the cameras y position.
        */
        y: number;


        /**
        * Method called to ensure the camera doesn't venture outside of the game world.
        * Called automatically by Camera.update.
        */
        checkBounds(): void;

        /**
        * This creates a camera fade effect. It works by filling the game with the
        * color specified, over the duration given, ending with a solid fill.
        * 
        * You can use this for things such as transitioning to a new scene.
        * 
        * The game will be left 'filled' at the end of this effect, likely obscuring
        * everything. In order to reset it you can call `Camera.resetFX` and it will clear the
        * fade. Or you can call `Camera.flash` with the same color as the fade, and it will
        * reverse the process, bringing the game back into view again.
        * 
        * When the effect ends the signal Camera.onFadeComplete is dispatched.
        * 
        * @param color The color the game will fade to. I.e. 0x000000 for black, 0xff0000 for red, etc. - Default: 0x000000
        * @param duration The duration of the fade in milliseconds. - Default: 500
        * @param force If a camera flash or fade effect is already running and force is true it will replace the previous effect, resetting the duration.
        * @param alpha The alpha value of the color applied to the fade effect. - Default: 1
        * @return True if the effect was started, otherwise false.
        */
        fade(color?: number, duration?: number, force?: boolean, alpha?: number): boolean;

        /**
        * This creates a camera flash effect. It works by filling the game with the solid fill
        * color specified, and then fading it away to alpha 0 over the duration given.
        * 
        * You can use this for things such as hit feedback effects.
        * 
        * When the effect ends the signal Camera.onFlashComplete is dispatched.
        * 
        * @param color The color of the flash effect. I.e. 0xffffff for white, 0xff0000 for red, etc. - Default: 0xffffff
        * @param duration The duration of the flash effect in milliseconds. - Default: 500
        * @param force If a camera flash or fade effect is already running and force is true it will replace the previous effect, resetting the duration.
        * @param alpha The alpha value of the color applied to the flash effect. - Default: 1
        * @return True if the effect was started, otherwise false.
        */
        flash(color?: number, duration?: number, force?: boolean, alpha?: number): boolean;

        /**
        * Move the camera focus on a display object instantly.
        * 
        * @param displayObject The display object to focus the camera on. Must have visible x/y properties.
        */
        focusOn(displayObject: PIXI.DisplayObject): void;

        /**
        * Move the camera focus on a location instantly.
        * 
        * @param x X position.
        * @param y Y position.
        */
        focusOnXY(x: number, y: number): void;

        /**
        * Tell the camera which sprite to follow.
        * 
        * You can set the follow type and a linear interpolation value.
        * Use low lerp values (such as 0.1) to automatically smooth the camera motion.
        * 
        * If you find you're getting a slight "jitter" effect when following a Sprite it's probably to do with sub-pixel rendering of the Sprite position.
        * This can be disabled by setting `game.renderer.renderSession.roundPixels = true` to force full pixel rendering.
        * 
        * @param target The object you want the camera to track. Set to null to not follow anything.
        * @param style Leverage one of the existing {@link deadzone} presets. If you use a custom deadzone, ignore this parameter and manually specify the deadzone after calling follow().
        * @param lerpX A value between 0 and 1. This value specifies the amount of linear interpolation to use when horizontally tracking the target. The closer the value to 1, the faster the camera will track. - Default: 1
        * @param lerpY A value between 0 and 1. This value specifies the amount of linear interpolation to use when vertically tracking the target. The closer the value to 1, the faster the camera will track. - Default: 1
        */
        follow(target: Phaser.Sprite, style?: number, lerpX?: number, lerpY?: number): void;

        /**
        * Resets the camera back to 0,0 and un-follows any object it may have been tracking.
        * Also immediately resets any camera effects that may have been running such as
        * shake, flash or fade.
        */
        reset(): void;

        /**
        * Resets any active FX, such as a fade or flash and immediately clears it.
        * Useful to calling after a fade in order to remove the fade from the Stage.
        */
        resetFX(): void;

        /**
        * Update the Camera bounds to match the game world.
        */
        setBoundsToWorld(): void;

        /**
        * A helper function to set both the X and Y properties of the camera at once
        * without having to use game.camera.x and game.camera.y.
        * 
        * @param x X position.
        * @param y Y position.
        */
        setPosition(x: number, y: number): void;

        /**
        * Sets the size of the view rectangle given the width and height in parameters.
        * 
        * @param width The desired width.
        * @param height The desired height.
        */
        setSize(width: number, height: number): void;

        /**
        * This creates a camera shake effect. It works by applying a random amount of additional
        * spacing on the x and y axis each frame. You can control the intensity and duration
        * of the effect, and if it should effect both axis or just one.
        * 
        * When the shake effect ends the signal Camera.onShakeComplete is dispatched.
        * 
        * @param intensity The intensity of the camera shake. Given as a percentage of the camera size representing the maximum distance that the camera can move while shaking. - Default: 0.05
        * @param duration The duration of the shake effect in milliseconds. - Default: 500
        * @param force If a camera shake effect is already running and force is true it will replace the previous effect, resetting the duration. - Default: true
        * @param direction The directions in which the camera can shake. Either Phaser.Camera.SHAKE_BOTH, Phaser.Camera.SHAKE_HORIZONTAL or Phaser.Camera.SHAKE_VERTICAL. - Default: Phaser.Camera.SHAKE_BOTH
        * @param shakeBounds Is the effect allowed to shake the camera beyond its bounds (if set?). - Default: true
        * @return True if the shake effect was started, otherwise false.
        */
        shake(intensity?: number, duration?: number, force?: boolean, direction?: number, shakeBounds?: boolean): boolean;

        /**
        * Sets the Camera follow target to null, stopping it from following an object if it's doing so.
        */
        unfollow(): void;

        /**
        * The camera update loop. This is called automatically by the core game loop.
        */
        update(): void;

    }


    /**
    * The Canvas class handles everything related to creating the `canvas` DOM tag that Phaser will use,
    * including styles, offset and aspect ratio.
    */
    class Canvas {


        /**
        * Adds the given canvas element to the DOM. The canvas will be added as a child of the given parent.
        * If no parent is given it will be added as a child of the document.body.
        * 
        * @param canvas The canvas to be added to the DOM.
        * @param parent The DOM element to add the canvas to.
        * @param overflowHidden If set to true it will add the overflow='hidden' style to the parent DOM element. - Default: true
        * @return Returns the source canvas.
        */
        static addToDOM(canvas: HTMLCanvasElement, parent: HTMLElement, overflowHidden?: boolean): HTMLCanvasElement;

        /**
        * Creates a `canvas` DOM element. The element is not automatically added to the document.
        * 
        * @param parent The object that will own the canvas that is created.
        * @param width The width of the canvas element. - Default: 256
        * @param height The height of the canvas element.. - Default: 256
        * @param id If specified, and not the empty string, this will be set as the ID of the canvas element. Otherwise no ID will be set. - Default: (none)
        * @param skipPool If `true` the canvas will not be placed in the CanvasPool global.
        * @return The newly created canvas element.
        */
        static create(parent: HTMLDivElement, width?: number, height?: number, id?: string, skipPool?: boolean): HTMLCanvasElement;

        /**
        * Returns `true` if the given context has image smoothing enabled, otherwise returns `false`.
        * 
        * @param context The context to check for smoothing on.
        * @return True if the given context has image smoothing enabled, otherwise false.
        */
        static getSmoothingEnabled(context: CanvasRenderingContext2D): boolean;

        /**
        * Gets the Smoothing Enabled vendor prefix being used on the given context, or null if not set.
        * 
        * @param context The context to enable or disable the image smoothing on.
        * @return Returns the smoothingEnabled vendor prefix, or null if not set on the context.
        */
        static getSmoothingPrefix(context: CanvasRenderingContext2D): string;

        /**
        * Removes the given canvas element from the DOM.
        * 
        * @param canvas The canvas to be removed from the DOM.
        */
        static removeFromDOM(canvas: HTMLCanvasElement): void;

        /**
        * Sets the background color behind the canvas. This changes the canvas style property.
        * 
        * @param canvas The canvas to set the background color on.
        * @param color The color to set. Can be in the format 'rgb(r,g,b)', or '#RRGGBB' or any valid CSS color. - Default: 'rgb(0,0,0)'
        * @return Returns the source canvas.
        */
        static setBackgroundColor(canvas: HTMLCanvasElement, color: string): HTMLCanvasElement;

        /**
        * Sets the CSS image-rendering property on the given canvas to be 'bicubic' (aka 'auto').
        * Note that if this doesn't given the desired result then see the CanvasUtils.setSmoothingEnabled method.
        * 
        * @param canvas The canvas to set image-rendering bicubic on.
        * @return Returns the source canvas.
        */
        static setImageRenderingBicubic(canvas: HTMLCanvasElement): HTMLCanvasElement;

        /**
        * Sets the CSS image-rendering property on the given canvas to be 'crisp' (aka 'optimize contrast' on webkit).
        * Note that if this doesn't given the desired result then see the setSmoothingEnabled.
        * 
        * @param canvas The canvas to set image-rendering crisp on.
        * @return Returns the source canvas.
        */
        static setImageRenderingCrisp(canvas: HTMLCanvasElement): HTMLCanvasElement;

        /**
        * Sets the Image Smoothing property on the given context. Set to false to disable image smoothing.
        * By default browsers have image smoothing enabled, which isn't always what you visually want, especially
        * when using pixel art in a game. Note that this sets the property on the context itself, so that any image
        * drawn to the context will be affected. This sets the property across all current browsers but support is
        * patchy on earlier browsers, especially on mobile.
        * 
        * @param context The context to enable or disable the image smoothing on.
        * @param value If set to true it will enable image smoothing, false will disable it.
        * @return Returns the source context.
        */
        static setSmoothingEnabled(context: CanvasRenderingContext2D, value: boolean): CanvasRenderingContext2D;

        /**
        * Sets the touch-action property on the canvas style. Can be used to disable default browser touch actions.
        * 
        * @param canvas The canvas to set the touch action on.
        * @param value The touch action to set. Defaults to 'none'.
        * @return The source canvas.
        */
        static setTouchAction(canvas: HTMLCanvasElement, value: string): HTMLCanvasElement;

        /**
        * Sets the transform of the given canvas to the matrix values provided.
        * 
        * @param context The context to set the transform on.
        * @param translateX The value to translate horizontally by.
        * @param translateY The value to translate vertically by.
        * @param scaleX The value to scale horizontally by.
        * @param scaleY The value to scale vertically by.
        * @param skewX The value to skew horizontaly by.
        * @param skewY The value to skew vertically by.
        * @return Returns the source context.
        */
        static setTransform(context: CanvasRenderingContext2D, translateX: number, translateY: number, scaleX: number, scaleY: number, skewX: number, skewY: number): CanvasRenderingContext2D;

        /**
        * Sets the user-select property on the canvas style. Can be used to disable default browser selection actions.
        * 
        * @param canvas The canvas to set the touch action on.
        * @param value The touch action to set. Defaults to 'none'.
        * @return The source canvas.
        */
        static setUserSelect(canvas: HTMLCanvasElement, value?: string): HTMLCanvasElement;

    }


    /**
    * Creates a new Circle object with the center coordinate specified by the x and y parameters and the diameter specified by the diameter parameter.
    * If you call this function without parameters, a circle with x, y, diameter and radius properties set to 0 is created.
    */
    class Circle {


        /**
        * Creates a new Circle object with the center coordinate specified by the x and y parameters and the diameter specified by the diameter parameter.
        * If you call this function without parameters, a circle with x, y, diameter and radius properties set to 0 is created.
        * 
        * @param x The x coordinate of the center of the circle.
        * @param y The y coordinate of the center of the circle.
        * @param diameter The diameter of the circle.
        */
        constructor(x?: number, y?: number, diameter?: number);


        /**
        * The area of this Circle.
        */
        area: number;

        /**
        * The sum of the y and radius properties. Changing the bottom property of a Circle object has no effect on the x and y properties, but does change the diameter. Gets or sets the bottom of the circle.
        */
        bottom: number;

        /**
        * The largest distance between any two points on the circle. The same as the radius * 2. Gets or sets the diameter of the circle.
        */
        diameter: number;

        /**
        * Determines whether or not this Circle object is empty. Will return a value of true if the Circle objects diameter is less than or equal to 0; otherwise false.
        * If set to true it will reset all of the Circle objects properties to 0. A Circle object is empty if its diameter is less than or equal to 0. Gets or sets the empty state of the circle.
        */
        empty: boolean;

        /**
        * The x coordinate of the leftmost point of the circle. Changing the left property of a Circle object has no effect on the x and y properties. However it does affect the diameter, whereas changing the x value does not affect the diameter property.
        */
        left: number;

        /**
        * The length of a line extending from the center of the circle to any point on the circle itself. The same as half the diameter. Gets or sets the radius of the circle.
        */
        radius: number;

        /**
        * The x coordinate of the rightmost point of the circle. Changing the right property of a Circle object has no effect on the x and y properties. However it does affect the diameter, whereas changing the x value does not affect the diameter property. Gets or sets the value of the rightmost point of the circle.
        */
        right: number;

        /**
        * The sum of the y minus the radius property. Changing the top property of a Circle object has no effect on the x and y properties, but does change the diameter. Gets or sets the top of the circle.
        */
        top: number;

        /**
        * The x coordinate of the center of the circle.
        */
        x: number;

        /**
        * The y coordinate of the center of the circle.
        */
        y: number;


        /**
        * Returns a Point object containing the coordinates of a point on the circumference of the Circle based on the given angle.
        * 
        * @param angle The angle in radians (unless asDegrees is true) to return the point from.
        * @param asDegrees Is the given angle in radians (false) or degrees (true)?
        * @param out An optional Point object to put the result in to. If none specified a new Point object will be created.
        * @return The Point object holding the result.
        */
        static circumferencePoint(a: Phaser.Circle, angle: number, asDegrees: boolean, out?: Phaser.Point): Phaser.Point;

        /**
        * Return true if the given x/y coordinates are within this Circle object.
        * 
        * @param x The X value of the coordinate to test.
        * @param y The Y value of the coordinate to test.
        * @return True if the coordinates are within this circle, otherwise false.
        */
        static contains(a: Phaser.Circle, x: number, y: number): boolean;

        /**
        * Determines whether the two Circle objects match. This method compares the x, y and diameter properties.
        * 
        * @param a The first Circle object.
        * @param b The second Circle object.
        * @return A value of true if the object has exactly the same values for the x, y and diameter properties as this Circle object; otherwise false.
        */
        static equals(a: Phaser.Circle, b: Phaser.Circle): boolean;

        /**
        * Determines whether the two Circle objects intersect.
        * This method checks the radius distances between the two Circle objects to see if they intersect.
        * 
        * @param a The first Circle object.
        * @param b The second Circle object.
        * @return A value of true if the specified object intersects with this Circle object; otherwise false.
        */
        static intersects(a: Phaser.Circle, b: Phaser.Circle): boolean;

        /**
        * Checks if the given Circle and Rectangle objects intersect.
        * 
        * @param c The Circle object to test.
        * @param r The Rectangle object to test.
        * @return True if the two objects intersect, otherwise false.
        */
        static intersectsRectangle(c: Phaser.Circle, r: Phaser.Rectangle): boolean;


        /**
        * The circumference of the circle.
        * @return The circumference of the circle.
        */
        circumference(): number;

        /**
        * Returns a Point object containing the coordinates of a point on the circumference of the Circle based on the given angle.
        * 
        * @param angle The angle in radians (unless asDegrees is true) to return the point from.
        * @param asDegrees Is the given angle in radians (false) or degrees (true)?
        * @param out An optional Point object to put the result in to. If none specified a new Point object will be created.
        * @return The Point object holding the result.
        */
        circumferencePoint(angle: number, asDegrees?: boolean, out?: Phaser.Point): Phaser.Point;

        /**
        * Returns a new Circle object with the same values for the x, y, width, and height properties as this Circle object.
        * 
        * @param output Optional Circle object. If given the values will be set into the object, otherwise a brand new Circle object will be created and returned.
        * @return The cloned Circle object.
        */
        clone(output: Phaser.Circle): Phaser.Circle;

        /**
        * Return true if the given x/y coordinates are within this Circle object.
        * 
        * @param x The X value of the coordinate to test.
        * @param y The Y value of the coordinate to test.
        * @return True if the coordinates are within this circle, otherwise false.
        */
        contains(x: number, y: number): boolean;

        /**
        * Copies the x, y and diameter properties from any given object to this Circle.
        * 
        * @param source The object to copy from.
        * @return This Circle object.
        */
        copyFrom(source: any): Circle;

        /**
        * Copies the x, y and diameter properties from this Circle to any given object.
        * 
        * @param dest The object to copy to.
        * @return This dest object.
        */
        copyTo(dest: any): any;

        /**
        * Returns the distance from the center of the Circle object to the given object
        * (can be Circle, Point or anything with x/y properties)
        * 
        * @param dest The target object. Must have visible x and y properties that represent the center of the object.
        * @param round Round the distance to the nearest integer.
        * @return The distance between this Point object and the destination Point object.
        */
        distance(dest: any, round?: boolean): number;

        /**
        * Returns the framing rectangle of the circle as a Phaser.Rectangle object.
        * @return The bounds of the Circle.
        */
        getBounds(): Phaser.Rectangle;

        /**
        * Adjusts the location of the Circle object, as determined by its center coordinate, by the specified amounts.
        * 
        * @param dx Moves the x value of the Circle object by this amount.
        * @param dy Moves the y value of the Circle object by this amount.
        * @return This Circle object.
        */
        offset(dx: number, dy: number): Phaser.Circle;

        /**
        * Adjusts the location of the Circle object using a Point object as a parameter. This method is similar to the Circle.offset() method, except that it takes a Point object as a parameter.
        * 
        * @param point A Point object to use to offset this Circle object (or any valid object with exposed x and y properties).
        * @return This Circle object.
        */
        offsetPoint(point: Phaser.Point): Phaser.Circle;

        /**
        * Returns a uniformly distributed random point from anywhere within this Circle.
        * 
        * @param out A Phaser.Point, or any object with public x/y properties, that the values will be set in.
        *            If no object is provided a new Phaser.Point object will be created. In high performance areas avoid this by re-using an existing object.
        * @return An object containing the random point in its `x` and `y` properties.
        */
        random(out?: Phaser.Point): Phaser.Point;
        scale(x: number, y?: number): Phaser.Rectangle;

        /**
        * Sets the members of Circle to the specified values.
        * 
        * @param x The x coordinate of the center of the circle.
        * @param y The y coordinate of the center of the circle.
        * @param diameter The diameter of the circle.
        * @return This circle object.
        */
        setTo(x: number, y: number, diameter: number): Circle;

        /**
        * Returns a string representation of this object.
        * @return a string representation of the instance.
        */
        toString(): string;

    }


    /**
    * The Phaser.Color class is a set of static methods that assist in color manipulation and conversion.
    */
    class Color {


        /**
        * Return a string containing a hex representation of the given color component.
        * 
        * @param color The color channel to get the hex value for, must be a value between 0 and 255.
        * @return A string of length 2 characters, i.e. 255 = ff, 100 = 64.
        */
        static componentToHex(color: number): string;

        /**
        * A utility function to create a lightweight 'color' object with the default components.
        * Any components that are not specified will default to zero.
        * 
        * This is useful when you want to use a shared color object for the getPixel and getPixelAt methods.
        * 
        * @param r The red color component, in the range 0 - 255.
        * @param g The green color component, in the range 0 - 255.
        * @param b The blue color component, in the range 0 - 255.
        * @param a The alpha color component, in the range 0 - 1. - Default: 1
        * @param h The hue, in the range 0 - 1.
        * @param s The saturation, in the range 0 - 1.
        * @param l The lightness, in the range 0 - 1.
        * @param v The value, in the range 0 - 1.
        * @return The resulting object with r, g, b, a properties and h, s, l and v.
        */
        static createColor(r?: number, g?: number, b?: number, a?: number, h?: number, s?: number, l?: number, v?: number): ColorComponents;

        /**
        * A utility to convert an integer in 0xRRGGBBAA format to a color object.
        * This does not rely on endianness.
        * 
        * @param rgba An RGBA hex
        * @param out The object to use, optional.
        * @return A color object.
        */
        static fromRGBA(rgba: number, out?: ColorComponents): ColorComponents;

        /**
        * Given a native color value (in the format 0xAARRGGBB) this will return the Alpha component, as a value between 0 and 255.
        * 
        * @param color In the format 0xAARRGGBB.
        * @return The Alpha component of the color, will be between 0 and 1 (0 being no Alpha (opaque), 1 full Alpha (transparent)).
        */
        static getAlpha(color: number): number;

        /**
        * Given a native color value (in the format 0xAARRGGBB) this will return the Alpha component as a value between 0 and 1.
        * 
        * @param color In the format 0xAARRGGBB.
        * @return The Alpha component of the color, will be between 0 and 1 (0 being no Alpha (opaque), 1 full Alpha (transparent)).
        */
        static getAlphaFloat(color: number): number;

        /**
        * Given a native color value (in the format 0xAARRGGBB) this will return the Blue component, as a value between 0 and 255.
        * 
        * @param color In the format 0xAARRGGBB.
        * @return The Blue component of the color, will be between 0 and 255 (0 being no color, 255 full Blue).
        */
        static getBlue(color: number): number;

        /**
        * Given 3 color values this will return an integer representation of it.
        * 
        * @param r The red color component, in the range 0 - 255.
        * @param g The green color component, in the range 0 - 255.
        * @param b The blue color component, in the range 0 - 255.
        * @return A native color value integer (format: 0xRRGGBB).
        */
        static getColor(red: number, green: number, blue: number): number;

        /**
        * Given an alpha and 3 color values this will return an integer representation of it.
        * 
        * @param a The alpha color component, in the range 0 - 255.
        * @param r The red color component, in the range 0 - 255.
        * @param g The green color component, in the range 0 - 255.
        * @param b The blue color component, in the range 0 - 255.
        * @return A native color value integer (format: 0xAARRGGBB).
        */
        static getColor32(alpha: number, red: number, green: number, blue: number): number;

        /**
        * Given a native color value (in the format 0xAARRGGBB) this will return the Green component, as a value between 0 and 255.
        * 
        * @param color In the format 0xAARRGGBB.
        * @return The Green component of the color, will be between 0 and 255 (0 being no color, 255 full Green).
        */
        static getGreen(color: number): number;

        /**
        * Returns a random color value between black and white
        * Set the min value to start each channel from the given offset.
        * Set the max value to restrict the maximum color used per channel.
        * 
        * @param min The lowest value to use for the color.
        * @param max The highest value to use for the color. - Default: 255
        * @param alpha The alpha value of the returning color (default 255 = fully opaque). - Default: 255
        * @return 32-bit color value with alpha.
        */
        static getRandomColor(min?: number, max?: number, alpha?: number): number;

        /**
        * Given a native color value (in the format 0xAARRGGBB) this will return the Red component, as a value between 0 and 255.
        * 
        * @param color In the format 0xAARRGGBB.
        * @return The Red component of the color, will be between 0 and 255 (0 being no color, 255 full Red).
        */
        static getRed(color: number): number;

        /**
        * Return the component parts of a color as an Object with the properties alpha, red, green, blue.
        * 
        * Alpha will only be set if it exist in the given color (0xAARRGGBB)
        * 
        * @param color Color in RGB (0xRRGGBB) or ARGB format (0xAARRGGBB).
        * @return An Object with properties: alpha, red, green, blue (also r, g, b and a). Alpha will only be present if a color value > 16777215 was given.
        */
        static getRGB(color: number): RGBColor;

        /**
        * Returns a CSS friendly string value from the given color.
        * 
        * @param color Color in RGB (0xRRGGBB), ARGB format (0xAARRGGBB) or an Object with r, g, b, a properties.
        * @return A string in the format: 'rgba(r,g,b,a)'
        */
        static getWebRGB(color: number | RGBColor): string;

        /**
        * Converts a hex color value to an [R, G, B] array.
        * 
        * @param color The color to convert to an RGB array. In the format 0xRRGGBB.
        * @return An array with element 0 containing the Red value, 1 containing Green, and 2 containing Blue.
        */
        static hexToRGBArray(color: number): number[];

        /**
        * Converts a hex string into an integer color value.
        * 
        * @param hex The hex string to convert. Can be in the short-hand format `#03f` or `#0033ff`.
        * @return The rgb color value in the format 0xAARRGGBB.
        */
        static hexToRGB(h: string): number;

        /**
        * Converts a hex string into a Phaser Color object.
        * 
        * The hex string can supplied as `'#0033ff'` or the short-hand format of `'#03f'`; it can begin with an optional "#" or "0x", or be unprefixed.
        * 
        * An alpha channel is _not_ supported.
        * 
        * @param hex The color string in a hex format.
        * @param out An object into which 3 properties will be created or set: r, g and b. If not provided a new object will be created.
        * @return An object with the red, green and blue values set in the r, g and b properties.
        */
        static hexToColor(hex: string, out?: ColorComponents): ColorComponents;

        /**
        * Converts an HSL (hue, saturation and lightness) color value to RGB.
        * Conversion forumla from http://en.wikipedia.org/wiki/HSL_color_space.
        * Assumes HSL values are contained in the set [0, 1] and returns r, g and b values in the set [0, 255].
        * Based on code by Michael Jackson (https://github.com/mjijackson)
        * 
        * @param h The hue, in the range 0 - 1.
        * @param s The saturation, in the range 0 - 1.
        * @param l The lightness, in the range 0 - 1.
        * @param out An object into which 3 properties will be created: r, g and b. If not provided a new object will be created.
        * @return An object with the red, green and blue values set in the r, g and b properties.
        */
        static HSLtoRGB(h: number, s: number, l: number, out?: ColorComponents): ColorComponents;

        /**
        * Get HSL color wheel values in an array which will be 360 elements in size.
        * 
        * @param s The saturation, in the range 0 - 1. - Default: 0.5
        * @param l The lightness, in the range 0 - 1. - Default: 0.5
        * @return An array containing 360 elements corresponding to the HSL color wheel.
        */
        static HSLColorWheel(s?: number, l?: number): ColorComponents[];

        /**
        * Converts an HSV (hue, saturation and value) color value to RGB.
        * Conversion forumla from http://en.wikipedia.org/wiki/HSL_color_space.
        * Assumes HSV values are contained in the set [0, 1] and returns r, g and b values in the set [0, 255].
        * Based on code by Michael Jackson (https://github.com/mjijackson)
        * 
        * @param h The hue, in the range 0 - 1.
        * @param s The saturation, in the range 0 - 1.
        * @param v The value, in the range 0 - 1.
        * @param out An object into which 3 properties will be created: r, g and b. If not provided a new object will be created.
        * @return An object with the red, green and blue values set in the r, g and b properties.
        */
        static HSVtoRGB(h: number, s: number, v: number, out?: ColorComponents): ColorComponents;

        /**
        * Get HSV color wheel values in an array which will be 360 elements in size.
        * 
        * @param s The saturation, in the range 0 - 1. - Default: 1
        * @param v The value, in the range 0 - 1. - Default: 1
        * @return An array containing 360 elements corresponding to the HSV color wheel.
        */
        static HSVColorWheel(s?: number, v?: number): ColorComponents[];

        /**
        * Converts a hue to an RGB color.
        * Based on code by Michael Jackson (https://github.com/mjijackson)
        * 
        * @param p
        * @param q
        * @param t
        * @return The color component value.
        */
        static hueToColor(p: number, q: number, t: number): number;

        /**
        * Interpolates the two given colours based on the supplied step and currentStep properties.
        * 
        * @param color1 The first color value.
        * @param color2 The second color value.
        * @param steps The number of steps to run the interpolation over.
        * @param currentStep The currentStep value. If the interpolation will take 100 steps, a currentStep value of 50 would be half-way between the two.
        * @param alpha The alpha of the returned color.
        * @return The interpolated color value.
        */
        static interpolateColor(color1: number, color2: number, steps: number, currentStep: number, alpha?: number): number;

        /**
        * Interpolates the two given colours based on the supplied step and currentStep properties.
        * 
        * @param color The first color value.
        * @param r The red color value, between 0 and 0xFF (255).
        * @param g The green color value, between 0 and 0xFF (255).
        * @param b The blue color value, between 0 and 0xFF (255).
        * @param steps The number of steps to run the interpolation over.
        * @param currentStep The currentStep value. If the interpolation will take 100 steps, a currentStep value of 50 would be half-way between the two.
        * @return The interpolated color value.
        */
        static interpolateColorWithRGB(color: number, r: number, g: number, b: number, steps: number, currentStep: number): number;

        /**
        * Interpolates the two given colours based on the supplied step and currentStep properties.
        * 
        * @param r1 The red color value, between 0 and 0xFF (255).
        * @param g1 The green color value, between 0 and 0xFF (255).
        * @param b1 The blue color value, between 0 and 0xFF (255).
        * @param r2 The red color value, between 0 and 0xFF (255).
        * @param g2 The green color value, between 0 and 0xFF (255).
        * @param b2 The blue color value, between 0 and 0xFF (255).
        * @param steps The number of steps to run the interpolation over.
        * @param currentStep The currentStep value. If the interpolation will take 100 steps, a currentStep value of 50 would be half-way between the two.
        * @return The interpolated color value.
        */
        static interpolateRGB(r1: number, g1: number, b1: number, r2: number, g2: number, b2: number, steps: number, currentStep: number): number;

        /**
        * Packs the r, g, b, a components into a single integer, for use with Int32Array.
        * If device is little endian then ABGR order is used. Otherwise RGBA order is used.
        * 
        * @param r The red color component, in the range 0 - 255.
        * @param g The green color component, in the range 0 - 255.
        * @param b The blue color component, in the range 0 - 255.
        * @param a The alpha color component, in the range 0 - 255.
        * @return The packed color as uint32
        */
        static packPixel(r: number, g: number, b: number, a: number): number;

        /**
        * Converts an RGB color array, in the format: [R, G, B], to a hex color value.
        * 
        * @param rgb An array with element 0 containing the Red value, 1 containing Green, and 2 containing Blue.
        * @return The color value, in the format 0xRRGGBB.
        */
        static RGBArrayToHex(rgb: number[]): number;

        /**
        * Converts an RGB color value to HSL (hue, saturation and lightness).
        * Conversion forumla from http://en.wikipedia.org/wiki/HSL_color_space.
        * Assumes RGB values are contained in the set [0, 255] and returns h, s and l in the set [0, 1].
        * Based on code by Michael Jackson (https://github.com/mjijackson)
        * 
        * @param r The red color component, in the range 0 - 255.
        * @param g The green color component, in the range 0 - 255.
        * @param b The blue color component, in the range 0 - 255.
        * @param out An object into which 3 properties will be created, h, s and l. If not provided a new object will be created.
        * @return An object with the hue, saturation and lightness values set in the h, s and l properties.
        */
        static RGBtoHSL(r: number, g: number, b: number, out?: ColorComponents): ColorComponents;

        /**
        * Converts an RGB color value to HSV (hue, saturation and value).
        * Conversion forumla from http://en.wikipedia.org/wiki/HSL_color_space.
        * Assumes RGB values are contained in the set [0, 255] and returns h, s and v in the set [0, 1].
        * Based on code by Michael Jackson (https://github.com/mjijackson)
        * 
        * @param r The red color component, in the range 0 - 255.
        * @param g The green color component, in the range 0 - 255.
        * @param b The blue color component, in the range 0 - 255.
        * @param out An object into which 3 properties will be created, h, s and v. If not provided a new object will be created.
        * @return An object with the hue, saturation and value set in the h, s and v properties.
        */
        static RGBtoHSV(r: number, g: number, b: number, out?: ColorComponents): ColorComponents;

        /**
        * Converts the given color values into a string.
        * If prefix was '#' it will be in the format `#RRGGBB` otherwise `0xAARRGGBB`.
        * 
        * @param r The red color component, in the range 0 - 255.
        * @param g The green color component, in the range 0 - 255.
        * @param b The blue color component, in the range 0 - 255.
        * @param a The alpha color component, in the range 0 - 255. - Default: 255
        * @param prefix The prefix used in the return string. If '#' it will return `#RRGGBB`, else `0xAARRGGBB`. - Default: '#'
        * @return A string containing the color values. If prefix was '#' it will be in the format `#RRGGBB` otherwise `0xAARRGGBB`.
        */
        static RGBtoString(r: number, g: number, b: number, a?: number, prefix?: string): string;

        /**
        * A utility to convert RGBA components to a 32 bit integer in RRGGBBAA format.
        * 
        * @param r The red color component, in the range 0 - 255.
        * @param g The green color component, in the range 0 - 255.
        * @param b The blue color component, in the range 0 - 255.
        * @param a The alpha color component, in the range 0 - 255.
        * @return A RGBA-packed 32 bit integer
        */
        static toRGBA(r: number, g: number, b: number, a: number): number;

        /**
        * Converts RGBA components to a 32 bit integer in AABBGGRR format.
        * 
        * @param r The red color component, in the range 0 - 255.
        * @param g The green color component, in the range 0 - 255.
        * @param b The blue color component, in the range 0 - 255.
        * @param a The alpha color component, in the range 0 - 255.
        * @return A RGBA-packed 32 bit integer
        */
        static toABGR(r: number, g: number, b: number, a: number): number;

        /**
        * Unpacks the r, g, b, a components into the specified color object, or a new
        * object, for use with Int32Array. If little endian, then ABGR order is used when
        * unpacking, otherwise, RGBA order is used. The resulting color object has the
        * `r, g, b, a` properties which are unrelated to endianness.
        * 
        * Note that the integer is assumed to be packed in the correct endianness. On little-endian
        * the format is 0xAABBGGRR and on big-endian the format is 0xRRGGBBAA. If you want a
        * endian-independent method, use fromRGBA(rgba) and toRGBA(r, g, b, a).
        * 
        * @param rgba The integer, packed in endian order by packPixel.
        * @param out An object into which 3 properties will be created: r, g and b. If not provided a new object will be created.
        * @param hsl Also convert the rgb values into hsl?
        * @param hsv Also convert the rgb values into hsv?
        * @return An object with the red, green and blue values set in the r, g and b properties.
        */
        static unpackPixel(rgba: number, out?: ColorComponents, hsl?: boolean, hsv?: boolean): ColorComponents;

        /**
        * Takes a color object and updates the rgba, color and color32 properties.
        * 
        * @param out The color object to update.
        * @return A native color value integer (format: 0xAARRGGBB).
        */
        static updateColor(out: ColorComponents): ColorComponents;

        /**
        * Converts a value - a "hex" string, a "CSS 'web' string", or a number - into red, green, blue, and alpha components.
        * 
        * The value can be a string (see `hexToColor` and `webToColor` for the supported formats) or a packed integer (see `getRGB`).
        * 
        * An alpha channel is _not_ supported when specifying a hex string.
        * 
        * @param value The color expressed as a recognized string format or a packed integer.
        * @param out The object to use for the output. If not provided a new object will be created.
        * @return The (`out`) object with the red, green, blue, and alpha values set as the r/g/b/a properties.
        */
        static valueToColor(value: string, out?: ColorComponents): ColorComponents;

        /**
        * Converts a CSS 'web' string into a Phaser Color object.
        * 
        * The web string can be in the format `'rgb(r,g,b)'` or `'rgba(r,g,b,a)'` where r/g/b are in the range [0..255] and a is in the range [0..1].
        * 
        * @param web The color string in CSS 'web' format.
        * @param out An object into which 4 properties will be created: r, g, b and a. If not provided a new object will be created.
        * @return An object with the red, green, blue and alpha values set in the r, g, b and a properties.
        */
        static webToColor(web: string, out?: ColorComponents): ColorComponents;

        /**
        * Blends the source color, ignoring the backdrop.
        * 
        * @param a The source color to blend, in the range 1 to 255.
        * @param b The backdrop color to blend, in the range 1 to 255.
        * @return The blended color value, in the range 1 to 255.
        */
        static blendNormal(a: number): number;

        /**
        * Selects the lighter of the backdrop and source colors.
        * 
        * @param a The source color to blend, in the range 1 to 255.
        * @param b The backdrop color to blend, in the range 1 to 255.
        * @return The blended color value, in the range 1 to 255.
        */
        static blendLighten(a: number, b: number): number;

        /**
        * Selects the darker of the backdrop and source colors.
        * 
        * @param a The source color to blend, in the range 1 to 255.
        * @param b The backdrop color to blend, in the range 1 to 255.
        * @return The blended color value, in the range 1 to 255.
        */
        static blendDarken(a: number, b: number): number;

        /**
        * Multiplies the backdrop and source color values.
        * The result color is always at least as dark as either of the two constituent
        * colors. Multiplying any color with black produces black;
        * multiplying with white leaves the original color unchanged.
        * 
        * @param a The source color to blend, in the range 1 to 255.
        * @param b The backdrop color to blend, in the range 1 to 255.
        * @return The blended color value, in the range 1 to 255.
        */
        static blendMultiply(a: number, b: number): number;

        /**
        * Takes the average of the source and backdrop colors.
        * 
        * @param a The source color to blend, in the range 1 to 255.
        * @param b The backdrop color to blend, in the range 1 to 255.
        * @return The blended color value, in the range 1 to 255.
        */
        static blendAverage(a: number, b: number): number;

        /**
        * Adds the source and backdrop colors together and returns the value, up to a maximum of 255.
        * 
        * @param a The source color to blend, in the range 1 to 255.
        * @param b The backdrop color to blend, in the range 1 to 255.
        * @return The blended color value, in the range 1 to 255.
        */
        static blendAdd(a: number, b: number): number;

        /**
        * Combines the source and backdrop colors and returns their value minus 255.
        * 
        * @param a The source color to blend, in the range 1 to 255.
        * @param b The backdrop color to blend, in the range 1 to 255.
        * @return The blended color value, in the range 1 to 255.
        */
        static blendSubtract(a: number, b: number): number;

        /**
        * Subtracts the darker of the two constituent colors from the lighter.
        * 
        * Painting with white inverts the backdrop color; painting with black produces no change.
        * 
        * @param a The source color to blend, in the range 1 to 255.
        * @param b The backdrop color to blend, in the range 1 to 255.
        * @return The blended color value, in the range 1 to 255.
        */
        static blendDifference(a: number, b: number): number;

        /**
        * Negation blend mode.
        * 
        * @param a The source color to blend, in the range 1 to 255.
        * @param b The backdrop color to blend, in the range 1 to 255.
        * @return The blended color value, in the range 1 to 255.
        */
        static blendNegation(a: number, b: number): number;

        /**
        * Multiplies the complements of the backdrop and source color values, then complements the result.
        * The result color is always at least as light as either of the two constituent colors.
        * Screening any color with white produces white; screening with black leaves the original color unchanged.
        * 
        * @param a The source color to blend, in the range 1 to 255.
        * @param b The backdrop color to blend, in the range 1 to 255.
        * @return The blended color value, in the range 1 to 255.
        */
        static blendScreen(a: number, b: number): number;

        /**
        * Produces an effect similar to that of the Difference mode, but lower in contrast.
        * Painting with white inverts the backdrop color; painting with black produces no change.
        * 
        * @param a The source color to blend, in the range 1 to 255.
        * @param b The backdrop color to blend, in the range 1 to 255.
        * @return The blended color value, in the range 1 to 255.
        */
        static blendExclusion(a: number, b: number): number;

        /**
        * Multiplies or screens the colors, depending on the backdrop color.
        * Source colors overlay the backdrop while preserving its highlights and shadows.
        * The backdrop color is not replaced, but is mixed with the source color to reflect the lightness or darkness of the backdrop.
        * 
        * @param a The source color to blend, in the range 1 to 255.
        * @param b The backdrop color to blend, in the range 1 to 255.
        * @return The blended color value, in the range 1 to 255.
        */
        static blendOverlay(a: number, b: number): number;

        /**
        * Darkens or lightens the colors, depending on the source color value.
        * 
        * If the source color is lighter than 0.5, the backdrop is lightened, as if it were dodged;
        * this is useful for adding highlights to a scene.
        * 
        * If the source color is darker than 0.5, the backdrop is darkened, as if it were burned in.
        * The degree of lightening or darkening is proportional to the difference between the source color and 0.5;
        * if it is equal to 0.5, the backdrop is unchanged.
        * 
        * Painting with pure black or white produces a distinctly darker or lighter area, but does not result in pure black or white.
        * The effect is similar to shining a diffused spotlight on the backdrop.
        * 
        * @param a The source color to blend, in the range 1 to 255.
        * @param b The backdrop color to blend, in the range 1 to 255.
        * @return The blended color value, in the range 1 to 255.
        */
        static blendSoftLight(a: number, b: number): number;

        /**
        * Multiplies or screens the colors, depending on the source color value.
        * 
        * If the source color is lighter than 0.5, the backdrop is lightened, as if it were screened;
        * this is useful for adding highlights to a scene.
        * 
        * If the source color is darker than 0.5, the backdrop is darkened, as if it were multiplied;
        * this is useful for adding shadows to a scene.
        * 
        * The degree of lightening or darkening is proportional to the difference between the source color and 0.5;
        * if it is equal to 0.5, the backdrop is unchanged.
        * 
        * Painting with pure black or white produces pure black or white. The effect is similar to shining a harsh spotlight on the backdrop.
        * 
        * @param a The source color to blend, in the range 1 to 255.
        * @param b The backdrop color to blend, in the range 1 to 255.
        * @return The blended color value, in the range 1 to 255.
        */
        static blendHardLight(a: number, b: number): number;

        /**
        * Brightens the backdrop color to reflect the source color.
        * Painting with black produces no change.
        * 
        * @param a The source color to blend, in the range 1 to 255.
        * @param b The backdrop color to blend, in the range 1 to 255.
        * @return The blended color value, in the range 1 to 255.
        */
        static blendColorDodge(a: number, b: number): number;

        /**
        * Darkens the backdrop color to reflect the source color.
        * Painting with white produces no change.
        * 
        * @param a The source color to blend, in the range 1 to 255.
        * @param b The backdrop color to blend, in the range 1 to 255.
        * @return The blended color value, in the range 1 to 255.
        */
        static blendColorBurn(a: number, b: number): number;

        /**
        * An alias for blendAdd, it simply sums the values of the two colors.
        * 
        * @param a The source color to blend, in the range 1 to 255.
        * @param b The backdrop color to blend, in the range 1 to 255.
        * @return The blended color value, in the range 1 to 255.
        */
        static blendLinearDodge(a: number, b: number): number;

        /**
        * An alias for blendSubtract, it simply sums the values of the two colors and subtracts 255.
        * 
        * @param a The source color to blend, in the range 1 to 255.
        * @param b The backdrop color to blend, in the range 1 to 255.
        * @return The blended color value, in the range 1 to 255.
        */
        static blendLinearBurn(a: number, b: number): number;

        /**
        * This blend mode combines Linear Dodge and Linear Burn (rescaled so that neutral colors become middle gray).
        * Dodge applies to values of top layer lighter than middle gray, and burn to darker values.
        * The calculation simplifies to the sum of bottom layer and twice the top layer, subtract 128. The contrast decreases.
        * 
        * @param a The source color to blend, in the range 1 to 255.
        * @param b The backdrop color to blend, in the range 1 to 255.
        * @return The blended color value, in the range 1 to 255.
        */
        static blendLinearLight(a: number, b: number): number;

        /**
        * This blend mode combines Color Dodge and Color Burn (rescaled so that neutral colors become middle gray).
        * Dodge applies when values in the top layer are lighter than middle gray, and burn to darker values.
        * The middle gray is the neutral color. When color is lighter than this, this effectively moves the white point of the bottom
        * layer down by twice the difference; when it is darker, the black point is moved up by twice the difference. The perceived contrast increases.
        * 
        * @param a The source color to blend, in the range 1 to 255.
        * @param b The backdrop color to blend, in the range 1 to 255.
        * @return The blended color value, in the range 1 to 255.
        */
        static blendVividLight(a: number, b: number): number;

        /**
        * If the backdrop color (light source) is lighter than 50%, the blendDarken mode is used, and colors lighter than the backdrop color do not change.
        * If the backdrop color is darker than 50% gray, colors lighter than the blend color are replaced, and colors darker than the blend color do not change.
        * 
        * @param a The source color to blend, in the range 1 to 255.
        * @param b The backdrop color to blend, in the range 1 to 255.
        * @return The blended color value, in the range 1 to 255.
        */
        static blendPinLight(a: number, b: number): number;

        /**
        * Runs blendVividLight on the source and backdrop colors.
        * If the resulting color is 128 or more, it receives a value of 255; if less than 128, a value of 0.
        * Therefore, all blended pixels have red, green, and blue channel values of either 0 or 255.
        * This changes all pixels to primary additive colors (red, green, or blue), white, or black.
        * 
        * @param a The source color to blend, in the range 1 to 255.
        * @param b The backdrop color to blend, in the range 1 to 255.
        * @return The blended color value, in the range 1 to 255.
        */
        static blendHardMix(a: number, b: number): number;

        /**
        * Reflect blend mode. This mode is useful when adding shining objects or light zones to images.
        * 
        * @param a The source color to blend, in the range 1 to 255.
        * @param b The backdrop color to blend, in the range 1 to 255.
        * @return The blended color value, in the range 1 to 255.
        */
        static blendReflect(a: number, b: number): number;

        /**
        * Glow blend mode. This mode is a variation of reflect mode with the source and backdrop colors swapped.
        * 
        * @param a The source color to blend, in the range 1 to 255.
        * @param b The backdrop color to blend, in the range 1 to 255.
        * @return The blended color value, in the range 1 to 255.
        */
        static blendGlow(a: number, b: number): number;

        /**
        * Phoenix blend mode. This subtracts the lighter color from the darker color, and adds 255, giving a bright result.
        * 
        * @param a The source color to blend, in the range 1 to 255.
        * @param b The backdrop color to blend, in the range 1 to 255.
        * @return The blended color value, in the range 1 to 255.
        */
        static blendPhoenix(a: number, b: number): number;

    }

    interface RGBColor {
        r: number;
        g: number;
        b: number;
        a: number;
    }
    interface ColorComponents extends RGBColor {
        h: number;
        s: number;
        v: number;
        l: number;
        color: number;
        color32: number;
        rgba: string;
    }


    /**
    * The Phaser.Create class is a collection of smaller helper methods that allow you to generate game content
    * quickly and easily, without the need for any external files. You can create textures for sprites and in
    * coming releases we'll add dynamic sound effect generation support as well (like sfxr).
    * 
    * Access this via `Game.create` (`this.game.create` from within a State object).
    */
    class Create {


        /**
        * The Phaser.Create class is a collection of smaller helper methods that allow you to generate game content
        * quickly and easily, without the need for any external files. You can create textures for sprites and in
        * coming releases we'll add dynamic sound effect generation support as well (like sfxr).
        * 
        * Access this via `Game.create` (`this.game.create` from within a State object).
        * 
        * @param game Game reference to the currently running game.
        */
        constructor(game: Phaser.Game);


        /**
        * A 16 color palette by [Arne](http://androidarts.com/palette/16pal.htm)
        */
        static PALETTE_ARNE: number;

        /**
        * A 16 color JMP inspired palette.
        */
        static PALETTE_JMP: number;

        /**
        * A 16 color CGA inspired palette.
        */
        static PALETTE_CGA: number;

        /**
        * A 16 color C64 inspired palette.
        */
        static PALETTE_C64: number;

        /**
        * A 16 color palette inspired by Japanese computers like the MSX.
        */
        static PALETTE_JAPANESE_MACHINE: number;


        /**
        * The internal BitmapData Create uses to generate textures from.
        */
        bmd: Phaser.BitmapData;

        /**
        * The canvas the BitmapData uses.
        */
        canvas: HTMLCanvasElement;

        /**
        * The 2d context of the canvas.
        */
        ctx: CanvasRenderingContext2D;

        /**
        * A reference to the currently running Game.
        */
        game: Phaser.Game;

        /**
        * A range of 16 color palettes for use with sprite generation.
        */
        palettes: any;


        /**
        * Copies the contents of {@link bmd Create's canvas} to the given BitmapData object, or a new BitmapData object.
        * 
        * @param dest The BitmapData receiving the copied image.
        * @param x The x coordinate to translate to before drawing.
        * @param y The y coordinate to translate to before drawing.
        * @param width The new width of the Sprite being copied.
        * @param height The new height of the Sprite being copied.
        * @param blendMode The composite blend mode that will be used when drawing. The default is no blend mode at all. This is a Canvas globalCompositeOperation value such as 'lighter' or 'xor'.
        * @param roundPx Should the x and y values be rounded to integers before drawing? This prevents anti-aliasing in some instances.
        * @return - The `dest` argument (if passed), or a new BitmapData object
        */
        copy(dest?: Phaser.BitmapData, x?: number, y?: number, width?: number, height?: number, blendMode?: string, roundPx?: boolean): Phaser.BitmapData;

        /**
        * Creates a grid texture based on the given dimensions.
        * 
        * @param key The key used to store this texture in the Phaser Cache.
        * @param width The width of the grid in pixels.
        * @param height The height of the grid in pixels.
        * @param cellWidth The width of the grid cells in pixels.
        * @param cellHeight The height of the grid cells in pixels.
        * @param color The color to draw the grid lines in. Should be a Canvas supported color string like `#ff5500` or `rgba(200,50,3,0.5)`.
        * @param generateTexture When false, a new BitmapData object is returned instead. - Default: true
        * @param callback A function to execute once the texture is generated. It will be passed the newly generated texture.
        * @param callbackContext The context in which to invoke the callback.
        * @return The newly generated texture, or a new BitmapData object if `generateTexture` is false, or `null` if a callback was passed and the texture isn't available yet.
        */
        grid(key: string, width: number, height: number, cellWidth: number, cellHeight: number, color: string, generateTexture?: boolean, callback?: Function, callbackContext?: any): PIXI.Texture;

        /**
        * Generates a new PIXI.Texture from the given data, which can be applied to a Sprite.
        * 
        * This allows you to create game graphics quickly and easily, with no external files but that use actual proper images
        * rather than Phaser.Graphics objects, which are expensive to render and limited in scope.
        * 
        * Each element of the array is a string holding the pixel color values, as mapped to one of the Phaser.Create PALETTE consts.
        * 
        * For example:
        * 
        * `var data = [
        *   ' 333 ',
        *   ' 777 ',
        *   'E333E',
        *   ' 333 ',
        *   ' 3 3 '
        * ];`
        * 
        * `game.create.texture('bob', data);`
        * 
        * The above will create a new texture called `bob`, which will look like a little man wearing a hat. You can then use it
        * for sprites the same way you use any other texture: `game.add.sprite(0, 0, 'bob');`
        * 
        * @param key The key used to store this texture in the Phaser Cache.
        * @param data An array of pixel data.
        * @param pixelWidth The width of each pixel. - Default: 8
        * @param pixelHeight The height of each pixel. - Default: 8
        * @param palette The palette to use when rendering the texture. One of the Phaser.Create.PALETTE consts.
        * @param generateTexture When false, a new BitmapData object is returned instead. - Default: true
        * @param callback A function to execute once the texture is generated. It will be passed the newly generated texture.
        * @param callbackContext The context in which to invoke the callback.
        * @return The newly generated texture, or a new BitmapData object if `generateTexture` is false, or `null` if a callback was passed and the texture isn't available yet.
        */
        texture(key: string, data: any, pixelWidth?: number, pixelHeight?: number, palette?: number, generateTexture?: boolean, callback?: Function, callbackContext?: any): PIXI.Texture;

    }

    interface CursorKeys {

        up: Phaser.Key;
        down: Phaser.Key;
        left: Phaser.Key;
        right: Phaser.Key;

    }


    /**
    * Detects device support capabilities and is responsible for device initialization - see {@link Phaser.Device.whenReady whenReady}.
    * 
    * This class represents a singleton object that can be accessed directly as `game.device`
    * (or, as a fallback, `Phaser.Device` when a game instance is not available) without the need to instantiate it.
    * 
    * Unless otherwise noted the device capabilities are only guaranteed after initialization. Initialization
    * occurs automatically and is guaranteed complete before {@link Phaser.Game} begins its "boot" phase.
    * Feature detection can be modified in the {@link Phaser.Device.onInitialized onInitialized} signal.
    * 
    * When checking features using the exposed properties only the *truth-iness* of the value should be relied upon
    * unless the documentation states otherwise: properties may return `false`, `''`, `null`, or even `undefined`
    * when indicating the lack of a feature.
    * 
    * Uses elements from System.js by MrDoob and Modernizr
    */
    class Device {


        /**
        * Same value as `littleEndian`.
        */
        static LITTLE_ENDIAN: boolean;

        /**
        * This signal is dispatched after device initialization occurs but before any of the ready
        * callbacks (see {@link Phaser.Device.whenReady whenReady}) have been invoked.
        * 
        * Local "patching" for a particular device can/should be done in this event.
        * 
        * _Note_: This signal is removed after the device has been readied; if a handler has not been
        * added _before_ `new Phaser.Game(..)` it is probably too late.
        */
        static onInitialized: Phaser.Signal;

        static checkFullScreenSupport(): void;

        /**
        * Check whether the host environment can play audio.
        * 
        * @param type One of 'mp3, 'ogg', 'm4a', 'wav', 'webm' or 'opus'.
        * @return True if the given file type is supported by the browser, otherwise false.
        */
        static canPlayAudio(type: string): boolean;

        /**
        * Check whether the host environment can play video files.
        * 
        * @param type One of 'mp4, 'ogg', 'webm' or 'mpeg'.
        * @return True if the given file type is supported by the browser, otherwise false.
        */
        static canPlayVideo(type: string): boolean;

        /**
        * Check whether the console is open.
        * Note that this only works in Firefox with Firebug and earlier versions of Chrome.
        * It used to work in Chrome, but then they removed the ability: {@link http://src.chromium.org/viewvc/blink?view=revision&revision=151136}
        */
        static isConsoleOpen(): boolean;

        /**
        * Detect if the host is a an Android Stock browser.
        * This is available before the device "ready" event.
        * 
        * Authors might want to scale down on effects and switch to the CANVAS rendering method on those devices.
        */
        static isAndroidStockBrowser(): string;
        static whenReady: (callback: Function, context?: any) => void;


        /**
        * Is running on android?
        */
        android: boolean;

        /**
        * Set to true if running in Arora.
        */
        arora: boolean;

        /**
        * Are Audio tags available?
        */
        audioData: boolean;
        cancelFullScreen: string;

        /**
        * If the browser isn't capable of handling tinting with alpha this will be false.
        */
        canHandleAlpha: boolean;

        /**
        * Whether or not the {@link http://caniuse.com/#feat=canvas-blending Canvas Blend Modes} are supported, consequently the ability to tint using the multiply method.
        * 
        * Expect `false` in Internet Explorer <= 11.
        */
        canUseMultiply: boolean;

        /**
        * Is canvas available?
        */
        canvas: boolean;

        /**
        * Set to true if running in Chrome.
        */
        chrome: boolean;

        /**
        * Is running on chromeOS?
        */
        chromeOS: boolean;

        /**
        * If running in Chrome this will contain the major version number.
        */
        chromeVersion: number;

        /**
        * Is the game running under CocoonJS?
        */
        cocoonJS: boolean;

        /**
        * Is this game running with CocoonJS.App?
        */
        cocoonJSApp: boolean;

        /**
        * Is the game running under Apache Cordova?
        */
        cordova: boolean;

        /**
        * Is the game running under the Intel Crosswalk XDK?
        */
        crosswalk: boolean;

        /**
        * Is css3D available?
        */
        css3D: boolean;

        /**
        * Is running on a desktop?
        */
        desktop: boolean;

        /**
        * The time the device became ready.
        */
        deviceReadyAt: number;

        /**
        * Is the game running under GitHub Electron?
        */
        electron: boolean;

        /**
        * Is the game running under Ejecta?
        */
        ejecta: boolean;

        /**
        * Set to true if running in Epiphany.
        */
        epiphany: boolean;

        /**
        * Is file available?
        */
        file: boolean;

        /**
        * Is fileSystem available?
        */
        fileSystem: boolean;

        /**
        * Set to true if running in Firefox.
        */
        firefox: boolean;

        /**
        * If running in Firefox this will contain the major version number.
        */
        firefoxVersion: number;
        fullScreen: boolean;
        fullScreenKeyboard: boolean;

        /**
        * Does the device support the getUserMedia API?
        * Default: true
        */
        getUserMedia: boolean;
        game: Phaser.Game;

        /**
        * Can this device play h264 mp4 video files?
        */
        h264Video: boolean;

        /**
        * Can this device play hls video files?
        */
        hlsVideo: boolean;

        /**
        * Set to true if running in Internet Explorer.
        */
        ie: boolean;

        /**
        * If running in Internet Explorer this will contain the major version number. Beyond IE10 you should use Device.trident and Device.tridentVersion.
        */
        ieVersion: number;

        /**
        * Is running on iOS?
        */
        iOS: boolean;

        /**
        * If running in iOS this will contain the major version number.
        */
        iOSVersion: number;

        /**
        * The time as which initialization has completed.
        */
        initialized: boolean;

        /**
        * Is running on iPad?
        */
        iPad: boolean;

        /**
        * Is running on iPhone?
        */
        iPhone: boolean;

        /**
        * Is running on iPhone4?
        */
        iPhone4: boolean;
        kindle: boolean;

        /**
        * Is running on linux?
        */
        linux: boolean;

        /**
        * Is the device big or little endian? (only detected if the browser supports TypedArrays)
        */
        littleEndian: boolean;

        /**
        * Is localStorage available?
        */
        localStorage: boolean;

        /**
        * Can this device play m4a files? True if this device can play m4a files.
        */
        m4a: boolean;

        /**
        * Is running on macOS?
        */
        macOS: boolean;

        /**
        * Set to true if running in Midori.
        */
        midori: boolean;

        /**
        * Set to true if running in Mobile Safari.
        */
        mobileSafari: boolean;

        /**
        * Can this device play mp3 files?
        */
        mp3: boolean;

        /**
        * Can this device play h264 mp4 video files?
        */
        mp4Video: boolean;

        /**
        * Is mspointer available?
        */
        mspointer: boolean;

        /**
        * Is the game running under Node.js?
        */
        node: boolean;

        /**
        * Is the game running under Node-Webkit?
        */
        nodeWebkit: boolean;

        /**
        * Can this device play ogg files?
        */
        ogg: boolean;

        /**
        * Can this device play ogg video files?
        */
        oggVideo: number;

        /**
        * Set to true if running in Opera.
        */
        opera: boolean;

        /**
        * Can this device play opus files?
        */
        opus: boolean;

        /**
        * PixelRatio of the host device?
        */
        pixelRatio: number;

        /**
        * Is Pointer Lock available?
        */
        pointerLock: boolean;

        /**
        * Is the browser running in strict mode (false) or quirks mode? (true)
        */
        quirksMode: boolean;
        requestFullScreen: string;

        /**
        * Set to true if running in Safari.
        */
        safari: boolean;

        /**
        * Set to true if running in the Silk browser (as used on the Amazon Kindle)
        */
        silk: boolean;

        /**
        * Does the device context support 32bit pixel manipulation using array buffer views?
        */
        support32bit: boolean;

        /**
        * Is touch available?
        */
        touch: boolean;

        /**
        * Set to true if running a Trident version of Internet Explorer (IE11+)
        */
        trident: boolean;

        /**
        * If running in Internet Explorer 11 this will contain the major version number. See {@link http://msdn.microsoft.com/en-us/library/ie/ms537503(v=vs.85).aspx}
        */
        tridentVersion: number;

        /**
        * Does the browser support TypedArrays?
        */
        typedArray: boolean;

        /**
        * Does the device support the Vibration API?
        */
        vibration: boolean;
        vita: boolean;

        /**
        * Can this device play wav files?
        */
        wav: boolean;

        /**
        * Set to true if running as a WebApp, i.e. within a WebView
        */
        webApp: boolean;

        /**
        * Is the WebAudio API available?
        */
        webAudio: boolean;

        /**
        * Is webGL available?
        */
        webGL: boolean;

        /**
        * Can this device play webm files?
        */
        webm: boolean;

        /**
        * Can this device play webm video files?
        */
        webmVideo: boolean;

        /**
        * Is running on windows?
        */
        windows: boolean;

        /**
        * Is running on a Windows Phone?
        */
        windowsPhone: boolean;

        /**
        * The newest type of Wheel/Scroll event supported: 'wheel', 'mousewheel', 'DOMMouseScroll'
        */
        wheelEvent: string;

        /**
        * Is worker available?
        */
        worker: boolean;
        wp9Video: boolean;

    }


    /**
    * DeviceButtons belong to both `Phaser.Pointer` and `Phaser.SinglePad` (Gamepad) instances.
    * 
    * For Pointers they represent the various buttons that can exist on mice and pens, such as the left button, right button,
    * middle button and advanced buttons like back and forward.
    * 
    * Access them via `Pointer.leftbutton`, `Pointer.rightButton` and so on.
    * 
    * On Gamepads they represent all buttons on the pad: from shoulder buttons to action buttons.
    * 
    * At the time of writing this there are device limitations you should be aware of:
    * 
    * - On Windows, if you install a mouse driver, and its utility software allows you to customize button actions
    *   (e.g., IntelliPoint and SetPoint), the middle (wheel) button, the 4th button, and the 5th button might not be set,
    *   even when they are pressed.
    * - On Linux (GTK), the 4th button and the 5th button are not supported.
    * - On Mac OS X 10.5 there is no platform API for implementing any advanced buttons.
    */
    class DeviceButton {


        /**
        * DeviceButtons belong to both `Phaser.Pointer` and `Phaser.SinglePad` (Gamepad) instances.
        * 
        * For Pointers they represent the various buttons that can exist on mice and pens, such as the left button, right button,
        * middle button and advanced buttons like back and forward.
        * 
        * Access them via `Pointer.leftbutton`, `Pointer.rightButton` and so on.
        * 
        * On Gamepads they represent all buttons on the pad: from shoulder buttons to action buttons.
        * 
        * At the time of writing this there are device limitations you should be aware of:
        * 
        * - On Windows, if you install a mouse driver, and its utility software allows you to customize button actions
        *   (e.g., IntelliPoint and SetPoint), the middle (wheel) button, the 4th button, and the 5th button might not be set,
        *   even when they are pressed.
        * - On Linux (GTK), the 4th button and the 5th button are not supported.
        * - On Mac OS X 10.5 there is no platform API for implementing any advanced buttons.
        * 
        * @param parent A reference to the parent of this button. Either a Pointer or a Gamepad.
        * @param buttonCode The button code this DeviceButton is responsible for.
        */
        constructor(parent: Phaser.Pointer | Phaser.SinglePad, butonCode: number);


        /**
        * The buttoncode of this button if a Gamepad, or the DOM button event value if a Pointer.
        */
        buttonCode: number;

        /**
        * A reference to the currently running game.
        */
        game: Phaser.Game;

        /**
        * The "down" state of the button.
        */
        isDown: boolean;

        /**
        * The "up" state of the button.
        * Default: true
        */
        isUp: boolean;

        /**
        * This Signal is dispatched every time this DeviceButton is pressed down.
        * It is only dispatched once (until the button is released again).
        * When dispatched it sends 2 arguments: A reference to this DeviceButton and the value of the button.
        */
        onDown: Phaser.Signal;

        /**
        * Gamepad only.
        * This Signal is dispatched every time this DeviceButton changes floating value (between, but not exactly, 0 and 1).
        * When dispatched it sends 2 arguments: A reference to this DeviceButton and the value of the button.
        */
        onFloat: Phaser.Signal;

        /**
        * This Signal is dispatched every time this DeviceButton is released from a down state.
        * It is only dispatched once (until the button is pressed again).
        * When dispatched it sends 2 arguments: A reference to this DeviceButton and the value of the button.
        */
        onUp: Phaser.Signal;
        pad: Phaser.Gamepad;

        /**
        * Gamepad only.
        * If a button is held down this holds down the number of times the button has 'repeated'.
        */
        repeats: number;

        /**
        * The timestamp when the button was last pressed down.
        */
        timeDown: number;

        /**
        * The timestamp when the button was last released.
        */
        timeUp: number;

        /**
        * Button value. Mainly useful for checking analog buttons (like shoulder triggers) on Gamepads.
        */
        value: number;


        /**
        * Destroys this DeviceButton, this disposes of the onDown, onUp and onFloat signals
        * and clears the parent and game references.
        */
        destroy(): void;

        /**
        * Returns the "just pressed" state of this button.
        * Just pressed is considered true if the button was pressed down within the duration given (default 250ms).
        * 
        * @param duration The duration in ms below which the button is considered as being just pressed. - Default: 250
        * @return True if the button is just pressed otherwise false.
        */
        justPressed(duration?: number): boolean;

        /**
        * Returns the "just released" state of this button.
        * Just released is considered as being true if the button was released within the duration given (default 250ms).
        * 
        * @param duration The duration in ms below which the button is considered as being just released. - Default: 250
        * @return True if the button is just released otherwise false.
        */
        justReleased(duration?: number): boolean;
        processButtonDown(value: number): void;
        processButtonFloat(value: number): void;
        processButtonUp(value: number): void;

        /**
        * Resets this DeviceButton, changing it to an isUp state and resetting the duration and repeats counters.
        */
        reset(): void;

    }

    module Easing {

        var Default: Function;
        var Power0: Function;
        var Power1: Function;
        var power2: Function;
        var power3: Function;
        var power4: Function;


        /**
        * Back easing.
        */
        class Back {

            /**
            * Back ease-in.
            * 
            * @param k The value to be tweened.
            * @return The tweened value.
            */
            static In(k: number): number;

            /**
            * Back ease-out.
            * 
            * @param k The value to be tweened.
            * @return The tweened value.
            */
            static Out(k: number): number;

            /**
            * Back ease-in/out.
            * 
            * @param k The value to be tweened.
            * @return The tweened value.
            */
            static InOut(k: number): number;
        }


        /**
        * Bounce easing.
        */
        class Bounce {

            /**
            * Bounce ease-in.
            * 
            * @param k The value to be tweened.
            * @return The tweened value.
            */
            static In(k: number): number;

            /**
            * Bounce ease-out.
            * 
            * @param k The value to be tweened.
            * @return The tweened value.
            */
            static Out(k: number): number;

            /**
            * Bounce ease-in/out.
            * 
            * @param k The value to be tweened.
            * @return The tweened value.
            */
            static InOut(k: number): number;
        }


        /**
        * Circular easing.
        */
        class Circular {

            /**
            * Circular ease-in.
            * 
            * @param k The value to be tweened.
            * @return The tweened value.
            */
            static In(k: number): number;

            /**
            * Circular ease-out.
            * 
            * @param k The value to be tweened.
            * @return The tweened value.
            */
            static Out(k: number): number;

            /**
            * Circular ease-in/out.
            * 
            * @param k The value to be tweened.
            * @return The tweened value.
            */
            static InOut(k: number): number;
        }


        /**
        * Cubic easing.
        */
        class Cubic {

            /**
            * Cubic ease-in.
            * 
            * @param k The value to be tweened.
            * @return The tweened value.
            */
            static In(k: number): number;

            /**
            * Cubic ease-out.
            * 
            * @param k The value to be tweened.
            * @return The tweened value.
            */
            static Out(k: number): number;

            /**
            * Cubic ease-in/out.
            * 
            * @param k The value to be tweened.
            * @return The tweened value.
            */
            static InOut(k: number): number;
        }


        /**
        * Elastic easing.
        */
        class Elastic {

            /**
            * Elastic ease-in.
            * 
            * @param k The value to be tweened.
            * @return The tweened value.
            */
            static In(k: number): number;

            /**
            * Elastic ease-out.
            * 
            * @param k The value to be tweened.
            * @return The tweened value.
            */
            static Out(k: number): number;

            /**
            * Elastic ease-in/out.
            * 
            * @param k The value to be tweened.
            * @return The tweened value.
            */
            static InOut(k: number): number;
        }


        /**
        * Exponential easing.
        */
        class Exponential {

            /**
            * Exponential ease-in.
            * 
            * @param k The value to be tweened.
            * @return The tweened value.
            */
            static In(k: number): number;

            /**
            * Exponential ease-out.
            * 
            * @param k The value to be tweened.
            * @return The tweened value.
            */
            static Out(k: number): number;

            /**
            * Exponential ease-in/out.
            * 
            * @param k The value to be tweened.
            * @return The tweened value.
            */
            static InOut(k: number): number;
        }


        /**
        * Linear easing.
        */
        class Linear {

            /**
            * Linear Easing (no variation).
            * 
            * @param k The value to be tweened.
            * @return k.
            */
            static None(k: number): number;
        }


        /**
        * Quadratic easing.
        */
        class Quadratic {

            /**
            * Ease-in.
            * 
            * @param k The value to be tweened.
            * @return k^2.
            */
            static In(k: number): number;

            /**
            * Ease-out.
            * 
            * @param k The value to be tweened.
            * @return k* (2-k).
            */
            static Out(k: number): number;

            /**
            * Ease-in/out.
            * 
            * @param k The value to be tweened.
            * @return The tweened value.
            */
            static InOut(k: number): number;
        }


        /**
        * Quartic easing.
        */
        class Quartic {

            /**
            * Quartic ease-in.
            * 
            * @param k The value to be tweened.
            * @return The tweened value.
            */
            static In(k: number): number;

            /**
            * Quartic ease-out.
            * 
            * @param k The value to be tweened.
            * @return The tweened value.
            */
            static Out(k: number): number;

            /**
            * Quartic ease-in/out.
            * 
            * @param k The value to be tweened.
            * @return The tweened value.
            */
            static InOut(k: number): number;
        }


        /**
        * Quintic easing.
        */
        class Quintic {

            /**
            * Quintic ease-in.
            * 
            * @param k The value to be tweened.
            * @return The tweened value.
            */
            static In(k: number): number;

            /**
            * Quintic ease-out.
            * 
            * @param k The value to be tweened.
            * @return The tweened value.
            */
            static Out(k: number): number;

            /**
            * Quintic ease-in/out.
            * 
            * @param k The value to be tweened.
            * @return The tweened value.
            */
            static InOut(k: number): number;
        }


        /**
        * Sinusoidal easing.
        */
        class Sinusoidal {

            /**
            * Sinusoidal ease-in.
            * 
            * @param k The value to be tweened.
            * @return The tweened value.
            */
            static In(k: number): number;

            /**
            * Sinusoidal ease-out.
            * 
            * @param k The value to be tweened.
            * @return The tweened value.
            */
            static Out(k: number): number;

            /**
            * Sinusoidal ease-in/out.
            * 
            * @param k The value to be tweened.
            * @return The tweened value.
            */
            static InOut(k: number): number;
        }

    }


    /**
    * Creates a Ellipse object. A curve on a plane surrounding two focal points.
    */
    class Ellipse {


        /**
        * Creates a Ellipse object. A curve on a plane surrounding two focal points.
        * 
        * @param x The X coordinate of the upper-left corner of the framing rectangle of this ellipse.
        * @param y The Y coordinate of the upper-left corner of the framing rectangle of this ellipse.
        * @param width The overall width of this ellipse.
        * @param height The overall height of this ellipse.
        */
        constructor(x?: number, y?: number, width?: number, height?: number);


        /**
        * The sum of the y and height properties. Changing the bottom property of an Ellipse doesn't adjust the y property, but does change the height. Gets or sets the bottom of the ellipse.
        */
        bottom: number;

        /**
        * Determines whether or not this Ellipse object is empty. Will return a value of true if the Ellipse objects dimensions are less than or equal to 0; otherwise false.
        * If set to true it will reset all of the Ellipse objects properties to 0. An Ellipse object is empty if its width or height is less than or equal to 0. Gets or sets the empty state of the ellipse.
        */
        empty: boolean;

        /**
        * The overall height of this ellipse.
        */
        height: number;

        /**
        * The left coordinate of the Ellipse. The same as the X coordinate.
        */
        left: number;

        /**
        * The x coordinate of the rightmost point of the Ellipse. Changing the right property of an Ellipse object has no effect on the x property, but does adjust the width. Gets or sets the value of the rightmost point of the ellipse.
        */
        right: number;

        /**
        * The top of the Ellipse. The same as its y property. Gets or sets the top of the ellipse.
        */
        top: number;

        /**
        * The const type of this object.
        */
        type: number;

        /**
        * The overall width of this ellipse.
        */
        width: number;

        /**
        * The X coordinate of the upper-left corner of the framing rectangle of this ellipse.
        */
        x: number;

        /**
        * The Y coordinate of the upper-left corner of the framing rectangle of this ellipse.
        */
        y: number;

        static constains(a: Phaser.Ellipse, x: number, y: number): boolean;


        /**
        * Returns a new Ellipse object with the same values for the x, y, width, and height properties as this Ellipse object.
        * 
        * @param output Optional Ellipse object. If given the values will be set into the object, otherwise a brand new Ellipse object will be created and returned.
        * @return The cloned Ellipse object.
        */
        clone(output: Phaser.Ellipse): Phaser.Ellipse;

        /**
        * Return true if the given x/y coordinates are within this Ellipse object.
        * 
        * @param x The X value of the coordinate to test.
        * @param y The Y value of the coordinate to test.
        * @return True if the coordinates are within this ellipse, otherwise false.
        */
        contains(x: number, y: number): boolean;

        /**
        * Copies the x, y, width and height properties from any given object to this Ellipse.
        * 
        * @param source The object to copy from.
        * @return This Ellipse object.
        */
        copyFrom(source: any): Phaser.Ellipse;

        /**
        * Copies the x, y, width and height properties from this Ellipse to any given object.
        * 
        * @param dest The object to copy to.
        * @return This dest object.
        */
        copyTo(dest: any): any;

        /**
        * Returns the framing rectangle of the ellipse as a Phaser.Rectangle object.
        * @return The bounds of the Ellipse.
        */
        getBounds(): Phaser.Rectangle;

        /**
        * Returns a uniformly distributed random point from anywhere within this Ellipse.
        * 
        * @param out A Phaser.Point, or any object with public x/y properties, that the values will be set in.
        *            If no object is provided a new Phaser.Point object will be created. In high performance areas avoid this by re-using an existing object.
        * @return An object containing the random point in its `x` and `y` properties.
        */
        random(out?: Phaser.Point): Phaser.Point;

        /**
        * Sets the members of the Ellipse to the specified values.
        * 
        * @param x The X coordinate of the upper-left corner of the framing rectangle of this ellipse.
        * @param y The Y coordinate of the upper-left corner of the framing rectangle of this ellipse.
        * @param width The overall width of this ellipse.
        * @param height The overall height of this ellipse.
        * @return This Ellipse object.
        */
        setTo(x: number, y: number, width: number, height: number): Phaser.Ellipse;

        /**
        * Returns a string representation of this object.
        * @return A string representation of the instance.
        */
        toString(): string;

    }


    /**
    * The Events component is a collection of events fired by the parent Game Object.
    * 
    * Phaser uses what are known as 'Signals' for all event handling. All of the events in
    * this class are signals you can subscribe to, much in the same way you'd "listen" for
    * an event.
    * 
    * For example to tell when a Sprite has been added to a new group, you can bind a function
    * to the `onAddedToGroup` signal:
    * 
    * `sprite.events.onAddedToGroup.add(yourFunction, this);`
    * 
    * Where `yourFunction` is the function you want called when this event occurs.
    * 
    * For more details about how signals work please see the Phaser.Signal class.
    * 
    * The Input-related events will only be dispatched if the Sprite has had `inputEnabled` set to `true`
    * and the Animation-related events only apply to game objects with animations like {@link Phaser.Sprite}.
    */
    class Events {


        /**
        * The Events component is a collection of events fired by the parent Game Object.
        * 
        * Phaser uses what are known as 'Signals' for all event handling. All of the events in
        * this class are signals you can subscribe to, much in the same way you'd "listen" for
        * an event.
        * 
        * For example to tell when a Sprite has been added to a new group, you can bind a function
        * to the `onAddedToGroup` signal:
        * 
        * `sprite.events.onAddedToGroup.add(yourFunction, this);`
        * 
        * Where `yourFunction` is the function you want called when this event occurs.
        * 
        * For more details about how signals work please see the Phaser.Signal class.
        * 
        * The Input-related events will only be dispatched if the Sprite has had `inputEnabled` set to `true`
        * and the Animation-related events only apply to game objects with animations like {@link Phaser.Sprite}.
        * 
        * @param sprite A reference to the game object / Sprite that owns this Events object.
        */
        constructor(sprite: Phaser.Sprite);


        /**
        * The Sprite that owns these events.
        */
        parent: Phaser.Sprite;

        /**
        * This signal is dispatched when this Game Object is added to a new Group.
        * It is sent two arguments:
        * {any} The Game Object that was added to the Group.
        * {Phaser.Group} The Group it was added to.
        */
        onAddedToGroup: Phaser.Signal;

        /**
        * This signal is dispatched when the Game Object is removed from a Group.
        * It is sent two arguments:
        * {any} The Game Object that was removed from the Group.
        * {Phaser.Group} The Group it was removed from.
        */
        onRemovedFromGroup: Phaser.Signal;

        /**
        * This Signal is never used internally by Phaser and is now deprecated.
        */
        onRemovedFromWorld: Phaser.Signal;

        /**
        * This signal is dispatched when the Game Object is killed.
        * This happens when `Sprite.kill()` is called.
        * Please understand the difference between `kill` and `destroy` by looking at their respective methods.
        * It is sent one argument:
        * {any} The Game Object that was killed.
        */
        onKilled: Phaser.Signal;

        /**
        * This signal is dispatched when the Game Object is revived from a previously killed state.
        * This happens when `Sprite.revive()` is called.
        * It is sent one argument:
        * {any} The Game Object that was revived.
        */
        onRevived: Phaser.Signal;

        /**
        * This signal is dispatched when the Game Object leaves the Phaser.World bounds.
        * This signal is only if `Sprite.checkWorldBounds` is set to `true`.
        * It is sent one argument:
        * {any} The Game Object that left the World bounds.
        */
        onOutOfBounds: Phaser.Signal;

        /**
        * This signal is dispatched when the Game Object returns within the Phaser.World bounds, having previously been outside of them.
        * This signal is only if `Sprite.checkWorldBounds` is set to `true`.
        * It is sent one argument:
        * {any} The Game Object that entered the World bounds.
        */
        onEnterBounds: Phaser.Signal;

        /**
        * This signal is dispatched if the Game Object has `inputEnabled` set to `true`,
        * and receives an over event from a Phaser.Pointer.
        * It is sent two arguments:
        * {any} The Game Object that received the event.
        * {Phaser.Pointer} The Phaser.Pointer object that caused the event.
        */
        onInputOver: Phaser.Signal;

        /**
        * This signal is dispatched if the Game Object has `inputEnabled` set to `true`,
        * and receives an out event from a Phaser.Pointer, which was previously over it.
        * It is sent two arguments:
        * {any} The Game Object that received the event.
        * {Phaser.Pointer} The Phaser.Pointer object that caused the event.
        */
        onInputOut: Phaser.Signal;

        /**
        * This signal is dispatched if the Game Object has `inputEnabled` set to `true`,
        * and receives a down event from a Phaser.Pointer. This effectively means the Pointer has been
        * pressed down (but not yet released) on the Game Object.
        * It is sent two arguments:
        * {any} The Game Object that received the event.
        * {Phaser.Pointer} The Phaser.Pointer object that caused the event.
        */
        onInputDown: Phaser.Signal;

        /**
        * This signal is dispatched if the Game Object has `inputEnabled` set to `true`,
        * and receives an up event from a Phaser.Pointer. This effectively means the Pointer had been
        * pressed down, and was then released on the Game Object.
        * It is sent three arguments:
        * {any} The Game Object that received the event.
        * {Phaser.Pointer} The Phaser.Pointer object that caused the event.
        * {boolean} isOver - Is the Pointer still over the Game Object?
        */
        onInputUp: Phaser.Signal;

        /**
        * This signal is dispatched when the Game Object is destroyed.
        * This happens when `Sprite.destroy()` is called, or `Group.destroy()` with `destroyChildren` set to true.
        * It is sent one argument:
        * {any} The Game Object that was destroyed.
        */
        onDestroy: Phaser.Signal;

        /**
        * This signal is dispatched if the Game Object has been `inputEnabled` and `enableDrag` has been set.
        * It is sent when a Phaser.Pointer starts to drag the Game Object, taking into consideration the various
        * drag limitations that may be set.
        * It is sent four arguments:
        * {any} The Game Object that received the event.
        * {Phaser.Pointer} The Phaser.Pointer object that caused the event.
        * {number} The x coordinate that the drag started from.
        * {number} The y coordinate that the drag started from.
        */
        onDragStart: Phaser.Signal;

        /**
        * This signal is dispatched if the Game Object has been `inputEnabled` and `enableDrag` has been set.
        * It is sent when a Phaser.Pointer stops dragging the Game Object.
        * It is sent two arguments:
        * {any} The Game Object that received the event.
        * {Phaser.Pointer} The Phaser.Pointer object that caused the event.
        */
        onDragStop: Phaser.Signal;

        /**
        * This signal is dispatched if the Game Object has been `inputEnabled` and `enableDrag` has been set.
        * It is sent when a Phaser.Pointer is actively dragging the Game Object.
        * Be warned: This is a high volume Signal. Be careful what you bind to it.
        * It is sent six arguments:
        * {any} The Game Object that received the event.
        * {Phaser.Pointer} The Phaser.Pointer object that caused the event.
        * {number} The new x coordinate of the Game Object.
        * {number} The new y coordinate of the Game Object.
        * {Phaser.Point} A Point object that contains the point the Game Object was snapped to, if `snapOnDrag` has been enabled.
        * {boolean} The `fromStart` boolean, indicates if this is the first update immediately after the drag has started.
        */
        onDragUpdate: Phaser.Signal;

        /**
        * This signal is dispatched if the Game Object has the AnimationManager component,
        * and an Animation has been played.
        * You can also listen to `Animation.onStart` rather than via the Game Objects events.
        * It is sent two arguments:
        * {any} The Game Object that received the event.
        * {Phaser.Animation} The Phaser.Animation that was started.
        */
        onAnimationStart: Phaser.Signal;

        /**
        * This signal is dispatched if the Game Object has the AnimationManager component,
        * and an Animation has been stopped (via `animation.stop()` and the `dispatchComplete` argument has been set.
        * You can also listen to `Animation.onComplete` rather than via the Game Objects events.
        * It is sent two arguments:
        * {any} The Game Object that received the event.
        * {Phaser.Animation} The Phaser.Animation that was stopped.
        */
        onAnimationComplete: Phaser.Signal;

        /**
        * This signal is dispatched if the Game Object has the AnimationManager component,
        * and an Animation has looped playback.
        * You can also listen to `Animation.onLoop` rather than via the Game Objects events.
        * It is sent two arguments:
        * {any} The Game Object that received the event.
        * {Phaser.Animation} The Phaser.Animation that looped.
        */
        onAnimationLoop: Phaser.Signal;


        /**
        * Removes all events.
        */
        destroy(): void;

    }


    /**
    * This is a base Filter class to use for any Phaser filter development.
    * If you want to make a custom filter, this should be your base class.
    * 
    * The default uniforms, types and values for all Filters are:
    * 
    * ```
    * resolution: { type: '2f', value: { x: 256, y: 256 }}
    * time: { type: '1f', value: 0 }
    * mouse: { type: '2f', value: { x: 0.0, y: 0.0 } }
    * date: { type: '4fv', value: [ d.getFullYear(),  d.getMonth(),  d.getDate(), d.getHours() *60 * 60 + d.getMinutes() * 60 + d.getSeconds() ] }
    * sampleRate: { type: '1f', value: 44100.0 }
    * iChannel0: { type: 'sampler2D', value: null, textureData: { repeat: true } }
    * iChannel1: { type: 'sampler2D', value: null, textureData: { repeat: true } }
    * iChannel2: { type: 'sampler2D', value: null, textureData: { repeat: true } }
    * iChannel3: { type: 'sampler2D', value: null, textureData: { repeat: true } }
    * ```
    * 
    * The vast majority of filters (including all of those that ship with Phaser) use fragment shaders, and
    * therefore only work in WebGL and are not supported by Canvas at all.
    */
    class Filter extends PIXI.AbstractFilter {


        /**
        * This is a base Filter class to use for any Phaser filter development.
        * If you want to make a custom filter, this should be your base class.
        * 
        * The default uniforms, types and values for all Filters are:
        * 
        * ```
        * resolution: { type: '2f', value: { x: 256, y: 256 }}
        * time: { type: '1f', value: 0 }
        * mouse: { type: '2f', value: { x: 0.0, y: 0.0 } }
        * date: { type: '4fv', value: [ d.getFullYear(),  d.getMonth(),  d.getDate(), d.getHours() *60 * 60 + d.getMinutes() * 60 + d.getSeconds() ] }
        * sampleRate: { type: '1f', value: 44100.0 }
        * iChannel0: { type: 'sampler2D', value: null, textureData: { repeat: true } }
        * iChannel1: { type: 'sampler2D', value: null, textureData: { repeat: true } }
        * iChannel2: { type: 'sampler2D', value: null, textureData: { repeat: true } }
        * iChannel3: { type: 'sampler2D', value: null, textureData: { repeat: true } }
        * ```
        * 
        * The vast majority of filters (including all of those that ship with Phaser) use fragment shaders, and
        * therefore only work in WebGL and are not supported by Canvas at all.
        * 
        * @param game A reference to the currently running game.
        * @param uniforms Uniform mappings object. The uniforms are added on the default uniforms, or replace them if the keys are the same.
        * @param fragmentSrc The fragment shader code. Either an array, one element per line of code, or a string.
        */
        constructor(game: Phaser.Game, uniforms: any, fragmentSrc: string | string[]);


        /**
        * Internal PIXI var.
        * Default: true
        */
        dirty: boolean;

        /**
        * A reference to the currently running game.
        */
        game: Phaser.Game;

        /**
        * The height (resolution uniform)
        */
        height: number;

        /**
        * The fragment shader code.
        */
        fragmentSrc: string | string[];

        /**
        * Internal PIXI var.
        */
        padding: number;

        /**
        * The previous position of the pointer (we don't update the uniform if the same)
        */
        prevPoint: Phaser.Point;

        /**
        * The const type of this object, either Phaser.WEBGL_FILTER or Phaser.CANVAS_FILTER.
        */
        type: number;

        /**
        * Default uniform mappings. Compatible with ShaderToy and GLSLSandbox.
        */
        uniforms: any;

        /**
        * The width (resolution uniform)
        */
        width: number;


        /**
        * Creates a new Phaser.Image object using a blank texture and assigns
        * this Filter to it. The image is then added to the world.
        * 
        * If you don't provide width and height values then Filter.width and Filter.height are used.
        * 
        * If you do provide width and height values then this filter will be resized to match those
        * values.
        * 
        * @param x The x coordinate to place the Image at.
        * @param y The y coordinate to place the Image at.
        * @param width The width of the Image. If not specified (or null) it will use Filter.width. If specified Filter.width will be set to this value.
        * @param height The height of the Image. If not specified (or null) it will use Filter.height. If specified Filter.height will be set to this value.
        * @param anchorX Set the x anchor point of the Image. A value between 0 and 1, where 0 is the top-left and 1 is bottom-right.
        * @param anchorY Set the y anchor point of the Image. A value between 0 and 1, where 0 is the top-left and 1 is bottom-right.
        * @return The newly added Image object.
        */
        addToWorld(x?: number, y?: number, width?: number, height?: number, anchorX?: number, anchorY?: number): Phaser.Image;
        apply(frameBuffer: WebGLFramebuffer): void;

        /**
        * Clear down this Filter and null out references to game.
        */
        destroy(): void;

        /**
        * This should be over-ridden. Will receive a variable number of arguments.
        */
        init(...args: any[]): void;

        /**
        * Set the resolution uniforms on the filter.
        * 
        * @param width The width of the display.
        * @param height The height of the display.
        */
        setResolution(width: number, height: number): void;

        /**
        * Syncs the uniforms between the class object and the shaders.
        */
        syncUniforms(): void;

        /**
        * Updates the filter.
        * 
        * @param pointer A Pointer object to use for the filter. The coordinates are mapped to the mouse uniform.
        */
        update(pointer?: Phaser.Pointer): void;

    }

    module Filter {

        class BinarySerpents extends Phaser.Filter {

            constructor(game: Phaser.Game, width: number, height: number, march?: number, maxDistance?: number);

            fog: number;

        }

        class BlurX extends Phaser.Filter {

            blur: number;

        }

        class BlurY extends Phaser.Filter {

            blur: number;

        }

        class CausticLight extends Phaser.Filter {

            constructor(game: Phaser.Game, width: number, height: number, divisor?: number);

            init(width: number, height: number, divisor?: number): void;

        }

        class CheckerWave extends Phaser.Filter {

            constructor(game: Phaser.Game, width: number, height: number);

            alpha: number;
            cameraX: number;
            cameraY: number;
            cameraZ: number;

            init(width: number, height: number): void;
            setColor1(red: number, green: number, blue: number): void;
            setColor2(red: number, green: number, blue: number): void;

        }

        class ColorBars extends Phaser.Filter {

            constructor(game: Phaser.Game, width: number, height: number);

            alpha: number;

            init(width: number, height: number): void;

        }

        class Fire extends Phaser.Filter {

            constructor(width: number, height: number, alpha?: number, shift?: number);

            alpha: number;
            shift: number;
            speed: number;

            init(width: number, height: number, alpha?: number, shift?: number): void;

        }

        class Gray extends Phaser.Filter {

            gray: number;

        }

        class HueRotate extends Phaser.Filter {

            constructor(game: Phaser.Game, width: number, height: number, texture: any);

            alpha: number;

            init(width: number, height: number, texture: any): void;

        }

        class LazerBeam extends Phaser.Filter {

            init(width: number, height: number, divisor?: number): void;

        }

        class LightBeam extends Phaser.Filter {

            constructor(game: Phaser.Game, width: number, height: number);

            alpha: number;
            blue: number;
            green: number;
            red: number;
            thickness: number;
            speed: number;

            init(width: number, height: number): void;

        }

        class Marble extends Phaser.Filter {

            constructor(game: Phaser.Game, width: number, height: number, speed?: number, intensity?: number);

            alpha: number;
            intensity: number;
            speed: number;

            init(width: number, height: number, speed?: number, intensity?: number): void;

        }

        class Pixelate extends Phaser.Filter {

            size: number;
            sizeX: number;
            sizeY: number;

        }

        class Plasma extends Phaser.Filter {

            constructor(game: Phaser.Game, width: number, height: number, alpha?: number, size?: number);

            alpha: number;
            blueShift: number;
            greenShift: number;
            redShift: number;
            size: number;

            init(width: number, height: number, alpha?: number, size?: number): void;

        }

        class SampleFilter extends Phaser.Filter {

            constructor(game: Phaser.Game, width: number, height: number, divisor?: number);

            init(width: number, height: number, divisor?: number): void;

        }

        class Tunnel extends Phaser.Filter {

            constructor(game: Phaser.Game, width: number, height: number, texture: any);

            alpha: number;
            origin: number;

            init(width: number, height: number, texture: any): void;

        }
    }


    /**
    * WARNING: This is an EXPERIMENTAL class. The API will change significantly in the coming versions and is incomplete.
    * Please try to avoid using in production games with a long time to build.
    * This is also why the documentation is incomplete.
    * 
    * FlexGrid is a a responsive grid manager that works in conjunction with the ScaleManager RESIZE scaling mode and FlexLayers
    * to provide for game object positioning in a responsive manner.
    */
    class FlexGrid {


        /**
        * WARNING: This is an EXPERIMENTAL class. The API will change significantly in the coming versions and is incomplete.
        * Please try to avoid using in production games with a long time to build.
        * This is also why the documentation is incomplete.
        * 
        * FlexGrid is a a responsive grid manager that works in conjunction with the ScaleManager RESIZE scaling mode and FlexLayers
        * to provide for game object positioning in a responsive manner.
        * 
        * @param manager The ScaleManager.
        * @param width The width of the game.
        * @param height The height of the game.
        */
        constructor(manager: Phaser.ScaleManager, width: number, height: number);


        /**
        * A reference to the currently running Game.
        */
        game: Phaser.Game;

        /**
        * A reference to the ScaleManager.
        */
        manager: Phaser.ScaleManager;
        width: number;
        height: number;
        boundsCustom: Phaser.Rectangle;
        boundsFluid: Phaser.Rectangle;
        boundsFull: Phaser.Rectangle;
        boundsNone: Phaser.Rectangle;
        customWidth: number;
        customHeight: number;
        customOffsetX: number;
        customOffsetY: number;

        /**
        * -
        */
        positionCustom: Phaser.Point;
        positionFluid: Phaser.Point;
        positionFull: Phaser.Point;
        positionNone: Phaser.Point;

        /**
        * The scale factor based on the game dimensions vs. the scaled dimensions.
        */
        scaleCustom: Phaser.Point;
        scaleFluid: Phaser.Point;
        scaleFluidInversed: Phaser.Point;
        scaleFull: Phaser.Point;
        scaleNone: Phaser.Point;
        ratioH: number;
        ratioV: number;
        multiplier: number;


        /**
        * A custom layer is centered on the game and maintains its aspect ratio as it scales up and down.
        * 
        * @param width Width of this layer in pixels.
        * @param height Height of this layer in pixels.
        * @param children An array of children that are used to populate the FlexLayer.
        * @return The Layer object.
        */
        createCustomLayer(width: number, height: number, children?: PIXI.DisplayObject[], addToWorld?: boolean): Phaser.FlexLayer;

        /**
        * A fluid layer is centered on the game and maintains its aspect ratio as it scales up and down.
        * 
        * @param children An array of children that are used to populate the FlexLayer.
        * @return The Layer object.
        */
        createFluidLayer(children: PIXI.DisplayObject[]): Phaser.FlexLayer;

        /**
        * A full layer is placed at 0,0 and extends to the full size of the game. Children are scaled according to the fluid ratios.
        * 
        * @param children An array of children that are used to populate the FlexLayer.
        * @return The Layer object.
        */
        createFullLayer(children: PIXI.DisplayObject[]): Phaser.FlexLayer;

        /**
        * A fixed layer is centered on the game and is the size of the required dimensions and is never scaled.
        * 
        * @param children An array of children that are used to populate the FlexLayer.
        * @return The Layer object.
        */
        createFixedLayer(children: PIXI.DisplayObject[]): Phaser.FlexLayer;

        /**
        * Call in the render function to output the bounds rects.
        */
        debug(): void;

        /**
        * Fits a sprites width to the bounds.
        * 
        * @param sprite The Sprite to fit.
        */
        fitSprite(sprite: Phaser.Sprite): void;

        /**
        * Called when the game container changes dimensions.
        * 
        * @param width The new width of the game container.
        * @param height The new height of the game container.
        */
        onResize(width: number, height: number): void;

        /**
        * Updates all internal vars such as the bounds and scale values.
        */
        refresh(): void;

        /**
        * Resets the layer children references
        */
        reset(): void;

        /**
        * Sets the core game size. This resets the w/h parameters and bounds.
        * 
        * @param width The new dimensions.
        * @param height The new dimensions.
        */
        setSize(width: number, height: number): void;

    }


    /**
    * WARNING: This is an EXPERIMENTAL class. The API will change significantly in the coming versions and is incomplete.
    * Please try to avoid using in production games with a long time to build.
    * This is also why the documentation is incomplete.
    * 
    * A responsive grid layer.
    */
    class FlexLayer extends Phaser.Group {


        /**
        * WARNING: This is an EXPERIMENTAL class. The API will change significantly in the coming versions and is incomplete.
        * Please try to avoid using in production games with a long time to build.
        * This is also why the documentation is incomplete.
        * 
        * A responsive grid layer.
        * 
        * @param manager The FlexGrid that owns this FlexLayer.
        * @param position A reference to the Point object used for positioning.
        * @param bounds A reference to the Rectangle used for the layer bounds.
        * @param scale A reference to the Point object used for layer scaling.
        */
        constructor(manager: Phaser.ScaleManager, position: Phaser.Point, bounds: Phaser.Rectangle, scale: Phaser.Point);


        /**
        * A reference to the FlexGrid that owns this layer.
        */
        grid: Phaser.FlexGrid;

        /**
        * A reference to the ScaleManager.
        */
        manager: Phaser.ScaleManager;

        bottomLeft: Phaser.Point;
        bottomMiddle: Phaser.Point;
        bottomRight: Phaser.Point;
        bounds: Phaser.Rectangle;

        /**
        * Should the FlexLayer remain through a State swap?
        */
        persist: boolean;
        position: Phaser.Point;
        scale: Phaser.Point;
        topLeft: Phaser.Point;
        topMiddle: Phaser.Point;
        topRight: Phaser.Point;


        /**
        * Debug.
        */
        debug(): void;

        /**
        * Resize.
        */
        resize(): void;

    }


    /**
    * A Frame is a single frame of an animation and is part of a FrameData collection.
    */
    class Frame {


        /**
        * A Frame is a single frame of an animation and is part of a FrameData collection.
        * 
        * @param index The index of this Frame within the FrameData set it is being added to.
        * @param x X position of the frame within the texture image.
        * @param y Y position of the frame within the texture image.
        * @param width Width of the frame within the texture image.
        * @param height Height of the frame within the texture image.
        * @param name The name of the frame. In Texture Atlas data this is usually set to the filename.
        */
        constructor(index: number, x: number, y: number, width: number, height: number, name: string);


        /**
        * The bottom of the frame (y + height).
        */
        bottom: number;

        /**
        * Center X position within the image to cut from.
        */
        centerX: number;

        /**
        * Center Y position within the image to cut from.
        */
        centerY: number;

        /**
        * The distance from the top left to the bottom-right of this Frame.
        */
        distance: number;

        /**
        * Height of the frame.
        */
        height: number;

        /**
        * The index of this Frame within the FrameData set it is being added to.
        */
        index: number;

        /**
        * Useful for Texture Atlas files (is set to the filename value).
        */
        name: string;

        /**
        * The right of the Frame (x + width).
        */
        right: number;

        /**
        * Is the frame rotated in the source texture?
        */
        rotated: boolean;

        /**
        * Height of the original sprite before it was trimmed.
        */
        sourceSizeH: number;

        /**
        * Width of the original sprite before it was trimmed.
        */
        sourceSizeW: number;

        /**
        * Height of the trimmed sprite.
        */
        spriteSourceSizeH: number;

        /**
        * Width of the trimmed sprite.
        */
        spriteSourceSizeW: number;

        /**
        * X position of the trimmed sprite inside original sprite.
        */
        spriteSourceSizeX: number;

        /**
        * Y position of the trimmed sprite inside original sprite.
        */
        spriteSourceSizeY: number;

        /**
        * Was it trimmed when packed?
        */
        trimmed: boolean;
        uuid: string;

        /**
        * Width of the frame.
        */
        width: number;

        /**
        * X position within the image to cut from.
        */
        x: number;

        /**
        * Y position within the image to cut from.
        */
        y: number;


        /**
        * Clones this Frame into a new Phaser.Frame object and returns it.
        * Note that all properties are cloned, including the name and index.
        * @return An exact copy of this Frame object.
        */
        clone(): Phaser.Frame;

        /**
        * Returns a Rectangle set to the dimensions of this Frame.
        * 
        * @param out A rectangle to copy the frame dimensions to.
        * @return A rectangle.
        */
        getRect(out?: Phaser.Rectangle): Phaser.Rectangle;

        /**
        * If the frame was trimmed when added to the Texture Atlas this records the trim and source data.
        * 
        * @param trimmed If this frame was trimmed or not.
        * @param actualWidth The width of the frame before being trimmed.
        * @param actualHeight The height of the frame before being trimmed.
        * @param destX The destination X position of the trimmed frame for display.
        * @param destY The destination Y position of the trimmed frame for display.
        * @param destWidth The destination width of the trimmed frame for display.
        * @param destHeight The destination height of the trimmed frame for display.
        */
        setTrim(trimmed: boolean, actualWidth: number, actualHeight: number, destX: number, destY: number, destWidth: number, destHeight: number): void;

        /**
        * Adjusts of all the Frame properties based on the given width and height values.
        * 
        * @param width The new width of the Frame.
        * @param height The new height of the Frame.
        */
        resize(width: number, height: number): void;

    }


    /**
    * FrameData is a container for Frame objects, which are the internal representation of animation data in Phaser.
    */
    class FrameData {


        /**
        * The total number of frames in this FrameData set.
        */
        total: number;


        /**
        * Adds a new Frame to this FrameData collection. Typically called by the Animation.Parser and not directly.
        * 
        * @param frame The frame to add to this FrameData set.
        * @return The frame that was just added.
        */
        addFrame(frame: Frame): Phaser.Frame;

        /**
        * Check if there is a Frame with the given name.
        * 
        * @param name The name of the frame you want to check.
        * @return True if the frame is found, otherwise false.
        */
        checkFrameName(name: string): boolean;

        /**
        * Makes a copy of this FrameData including copies (not references) to all of the Frames it contains.
        * @return A clone of this object, including clones of the Frame objects it contains.
        */
        clone(): Phaser.FrameData;

        /**
        * Get a Frame by its numerical index.
        * 
        * @param index The index of the frame you want to get.
        * @return The frame, if found, or undefined.
        */
        getFrame(index: number): Phaser.Frame;

        /**
        * Get a Frame by its frame name.
        * 
        * @param name The name of the frame you want to get.
        * @return The frame, if found, or null.
        */
        getFrameByName(name: string): Phaser.Frame;

        /**
        * Returns all of the Frame indexes in this FrameData set.
        * The frames indexes are returned in the output array, or if none is provided in a new Array object.
        * 
        * @param frames An Array containing the indexes of the frames to retrieve. If undefined or the array is empty then all frames in the FrameData are returned.
        * @param useNumericIndex Are the given frames using numeric indexes (default) or strings? (false) - Default: true
        * @param output If given the results will be appended to the end of this array otherwise a new array will be created.
        * @return An array of all Frame indexes matching the given names or IDs.
        */
        getFrameIndexes(frames?: number[], useNumericIndex?: boolean, output?: number[]): number[];

        /**
        * Returns a range of frames based on the given start and end frame indexes and returns them in an Array.
        * 
        * @param start The starting frame index.
        * @param end The ending frame index.
        * @param output If given the results will be appended to the end of this array otherwise a new array will be created.
        * @return An array of Frames between the start and end index values, or an empty array if none were found.
        */
        getFrameRange(start: number, end: number, output: Phaser.Frame[]): Phaser.Frame[];

        /**
        * Returns all of the Frames in this FrameData set where the frame index is found in the input array.
        * The frames are returned in the output array, or if none is provided in a new Array object.
        * 
        * @param frames An Array containing the indexes of the frames to retrieve. If the array is empty or undefined then all frames in the FrameData are returned.
        * @param useNumericIndex Are the given frames using numeric indexes (default) or strings? (false) - Default: true
        * @param output If given the results will be appended to the end of this array otherwise a new array will be created.
        * @return An array of all Frames in this FrameData set matching the given names or IDs.
        */
        getFrames(frames?: number[], useNumericIndex?: boolean, output?: Phaser.Frame[]): Phaser.Frame[];

    }

    interface IGameConfig {

        enableDebug?: boolean;
        width?: number;
        height?: number;
        renderer?: number;
        parent?: any;
        transparent?: boolean;
        antialias?: boolean;
        resolution?: number;
        preserveDrawingBuffer?: boolean;
        physicsConfig?: any;
        seed?: string;
        state?: Phaser.State;
        forceSetTimeOut?: boolean;
        multiTexture?: boolean;
        scaleMode?: number;
        
    }


    /**
    * Instead of specifying arguments you can also pass a single object instead:
    * 
    * ```
    * var config = {
    *     width: 800,
    *     height: 600,
    *     renderer: Phaser.AUTO,
    *     antialias: true,
    *     multiTexture: true,
    *     state: {
    *         preload: preload,
    *         create: create,
    *         update: update
    *     }
    * }
    * 
    * var game = new Phaser.Game(config);
    * ```
    */
    class Game {


        /**
        * Instead of specifying arguments you can also pass a single object instead:
        * 
        * ```
        * var config = {
        *     width: 800,
        *     height: 600,
        *     renderer: Phaser.AUTO,
        *     antialias: true,
        *     multiTexture: true,
        *     state: {
        *         preload: preload,
        *         create: create,
        *         update: update
        *     }
        * }
        * 
        * var game = new Phaser.Game(config);
        * ```
        * 
        * @param config A single configuration object
        * @param config.antialias  - Default: true
        * @param config.height  - Default: 600
        * @param config.enableDebug Enable {@link Phaser.Utils.Debug}. You can gain a little performance by disabling this in production. - Default: true
        * @param config.fullScreenScaleMode The scaling method used by the ScaleManager when in fullscreen.
        * @param config.fullScreenTarget The DOM element on which the Fullscreen API enter request will be invoked.
        * @param config.multiTexture Enable support for multiple bound Textures in WebGL. Same as `renderer: Phaser.WEBGL_MULTI`.
        * @param config.parent  - Default: ''
        * @param config.physicsConfig
        * @param config.preserveDrawingBuffer Whether or not the contents of the stencil buffer is retained after rendering.
        * @param config.renderer  - Default: Phaser.AUTO
        * @param config.resolution The resolution of your game, as a ratio of canvas pixels to game pixels. - Default: 1
        * @param config.scaleMode The scaling method used by the ScaleManager when not in fullscreen.
        * @param config.seed Seed for {@link Phaser.RandomDataGenerator}.
        * @param config.state
        * @param config.transparent
        * @param config.width  - Default: 800
        */
        constructor(width?: number | string, height?: number | string, renderer?: number, parent?: any, state?: any, transparent?: boolean, antialias?: boolean, physicsConfig?: any);

        /**
        * Instead of specifying arguments you can also pass a single object instead:
        * 
        * ```
        * var config = {
        *     width: 800,
        *     height: 600,
        *     renderer: Phaser.AUTO,
        *     antialias: true,
        *     multiTexture: true,
        *     state: {
        *         preload: preload,
        *         create: create,
        *         update: update
        *     }
        * }
        * 
        * var game = new Phaser.Game(config);
        * ```
        * 
        * @param config A single configuration object
        * @param config.antialias  - Default: true
        * @param config.height  - Default: 600
        * @param config.enableDebug Enable {@link Phaser.Utils.Debug}. You can gain a little performance by disabling this in production. - Default: true
        * @param config.fullScreenScaleMode The scaling method used by the ScaleManager when in fullscreen.
        * @param config.fullScreenTarget The DOM element on which the Fullscreen API enter request will be invoked.
        * @param config.multiTexture Enable support for multiple bound Textures in WebGL. Same as `renderer: Phaser.WEBGL_MULTI`.
        * @param config.parent  - Default: ''
        * @param config.physicsConfig
        * @param config.preserveDrawingBuffer Whether or not the contents of the stencil buffer is retained after rendering.
        * @param config.renderer  - Default: Phaser.AUTO
        * @param config.resolution The resolution of your game, as a ratio of canvas pixels to game pixels. - Default: 1
        * @param config.scaleMode The scaling method used by the ScaleManager when not in fullscreen.
        * @param config.seed Seed for {@link Phaser.RandomDataGenerator}.
        * @param config.state
        * @param config.transparent
        * @param config.width  - Default: 800
        */
        constructor(config: IGameConfig);


        /**
        * Reference to the Phaser.GameObjectFactory.
        */
        add: Phaser.GameObjectFactory;

        /**
        * Anti-alias graphics (as set when the Game is created). By default scaled and rotated images are smoothed in Canvas and WebGL; set `antialias` to false to disable this globally. After the game boots, use `game.stage.smoothed` instead.
        * Default: true
        */
        antialias: boolean;

        /**
        * Reference to the assets cache.
        */
        cache: Phaser.Cache;

        /**
        * A handy reference to world.camera.
        */
        camera: Phaser.Camera;

        /**
        * A handy reference to renderer.view, the canvas that the game is being rendered in to.
        */
        canvas: HTMLCanvasElement;

        /**
        * Clear the Canvas each frame before rendering the display list.
        * You can set this to `false` to gain some performance if your game always contains a background that completely fills the display.
        * Default: true
        */
        clearBeforeRender: boolean;

        /**
        * The Phaser.Game configuration object.
        */
        config: IGameConfig;

        /**
        * A handy reference to renderer.context (only set for CANVAS games, not WebGL)
        */
        context: CanvasRenderingContext2D;
        count: number;

        /**
        * The Asset Generator.
        */
        create: Phaser.Create;

        /**
        * A set of useful debug utilities.
        */
        debug: Phaser.Utils.Debug;

        /**
        * Contains device information and capabilities.
        */
        device: Phaser.Device;

        /**
        * Should the game loop force a logic update, regardless of the delta timer? You can toggle it on the fly.
        */
        forceSingleUpdate: boolean;

        /**
        * If the game is struggling to maintain the desired FPS, this signal will be dispatched.
        * The desired/chosen FPS should probably be closer to the {@link Phaser.Time#suggestedFps} value.
        */
        fpsProblemNotifier: Phaser.Signal;

        /**
        * The current Game Height in pixels.
        * 
        * _Do not modify this property directly:_ use {@link Phaser.ScaleManager#setGameSize} - eg. `game.scale.setGameSize(width, height)` - instead.
        * Default: 600
        */
        height: number;

        /**
        * Phaser Game ID
        */
        id: number;

        /**
        * Reference to the input manager
        */
        input: Phaser.Input;

        /**
        * Whether the game engine is booted, aka available.
        */
        isBooted: boolean;

        /**
        * Is game running or paused?
        */
        isRunning: boolean;

        /**
        * Reference to the assets loader.
        */
        load: Phaser.Loader;

        /**
        * If `false` Phaser will automatically render the display list every update. If `true` the render loop will be skipped.
        * You can toggle this value at run-time to gain exact control over when Phaser renders. This can be useful in certain types of game or application.
        * Please note that if you don't render the display list then none of the game object transforms will be updated, so use this value carefully.
        */
        lockRender: boolean;

        /**
        * Reference to the GameObject Creator.
        */
        make: Phaser.GameObjectCreator;

        /**
        * Reference to the math helper.
        */
        math: Phaser.Math;

        /**
        * Reference to the network class.
        */
        net: Phaser.Net;

        /**
        * This event is fired when the game no longer has focus (typically on page hide).
        */
        onBlur: Phaser.Signal;

        /**
        * This event is fired when the game has focus (typically on page show).
        */
        onFocus: Phaser.Signal;

        /**
        * This event is fired when the game pauses.
        */
        onPause: Phaser.Signal;

        /**
        * This event is fired when the game resumes from a paused state.
        */
        onResume: Phaser.Signal;

        /**
        * The Game's DOM parent (or name thereof), if any, as set when the game was created. The actual parent can be found in `game.canvas.parentNode`. Setting this has no effect after {@link Phaser.ScaleManager} is booted.
        */
        parent: HTMLElement;

        /**
        * The Particle Manager.
        */
        particles: Phaser.Particles;

        /**
        * The paused state of the Game. A paused game doesn't update any of its subsystems.
        * When a game is paused the onPause event is dispatched. When it is resumed the onResume event is dispatched. Gets and sets the paused state of the Game.
        */
        paused: boolean;

        /**
        * An internal property used by enableStep, but also useful to query from your own game objects.
        */
        pendingStep: boolean;

        /**
        * Reference to the physics manager.
        */
        physics: Phaser.Physics;

        /**
        * The Phaser.Physics.World configuration object.
        */
        physicsConfig: any;

        /**
        * Reference to the plugin manager.
        */
        plugins: PluginManager;

        /**
        * The value of the preserveDrawingBuffer flag affects whether or not the contents of the stencil buffer is retained after rendering.
        */
        preserveDrawingBuffer: Boolean;

        /**
        * Automatically handles the core game loop via requestAnimationFrame or setTimeout
        */
        raf: Phaser.RequestAnimationFrame;

        /**
        * The Pixi Renderer.
        */
        renderer: PIXI.CanvasRenderer | PIXI.WebGLRenderer;

        /**
        * The Renderer this game will use. Either Phaser.AUTO, Phaser.CANVAS, Phaser.WEBGL, Phaser.WEBGL_MULTI or Phaser.HEADLESS. After the game boots, renderType reflects the renderer in use: AUTO changes to CANVAS or WEBGL and WEBGL_MULTI changes to WEBGL. HEADLESS skips `render` hooks but not `preRender` or `postRender`; set {@link lockRender} to skip those as well.
        */
        renderType: number;

        /**
        * The resolution of your game, as a ratio of canvas pixels to game pixels. This value is read only, but can be changed at start time it via a game configuration object.
        * Default: 1
        */
        resolution: number;

        /**
        * Instance of repeatable random data generator helper.
        */
        rnd: Phaser.RandomDataGenerator;

        /**
        * The game scale manager.
        */
        scale: Phaser.ScaleManager;
        scratch: Phaser.BitmapData;

        /**
        * Reference to the sound manager.
        */
        sound: Phaser.SoundManager;

        /**
        * Reference to the stage.
        */
        stage: Phaser.Stage;

        /**
        * The StateManager.
        */
        state: Phaser.StateManager;

        /**
        * When stepping is enabled this contains the current step cycle.
        */
        stepCount: number;

        /**
        * Enable core loop stepping with Game.enableStep().
        */
        stepping: boolean;

        /**
        * Reference to the core game clock.
        */
        time: Phaser.Time;

        /**
        * Use a transparent canvas background or not.
        */
        transparent: boolean;

        /**
        * Reference to the tween manager.
        */
        tweens: Phaser.TweenManager;

        /**
        * The ID of the current/last logic update applied this render frame, starting from 0.
        * The first update is `currentUpdateID === 0` and the last update is `currentUpdateID === updatesThisFrame.`
        */
        currentUpdateID: number;

        /**
        * Number of logic updates expected to occur this render frame; will be 1 unless there are catch-ups required (and allowed).
        */
        updatesThisFrame: number;

        /**
        * The current Game Width in pixels.
        * 
        * _Do not modify this property directly:_ use {@link Phaser.ScaleManager#setGameSize} - eg. `game.scale.setGameSize(width, height)` - instead.
        * Default: 800
        */
        width: number;

        /**
        * Reference to the world.
        */
        world: Phaser.World;


        /**
        * Initialize engine sub modules and start the game.
        */
        boot(): void;

        /**
        * Nukes the entire game from orbit.
        * 
        * Calls destroy on Game.state, Game.sound, Game.scale, Game.stage, Game.input, Game.physics and Game.plugins.
        * 
        * Then sets all of those local handlers to null, destroys the renderer, removes the canvas from the DOM
        * and resets the PIXI default renderer.
        */
        destroy(): void;

        /**
        * Disables core game loop stepping.
        */
        disableStep(): void;

        /**
        * Enable core game loop stepping. When enabled you must call game.step() directly (perhaps via a DOM button?)
        * Calling step will advance the game loop by one frame. This is extremely useful for hard to track down errors!
        */
        enableStep(): void;

        /**
        * Called by the Stage visibility handler.
        * 
        * @param event The DOM event that caused the game to pause, if any.
        */
        focusGain(event: any): void;

        /**
        * Called by the Stage visibility handler.
        * 
        * @param event The DOM event that caused the game to pause, if any.
        */
        focusLoss(event: any): void;

        /**
        * Called by the Stage visibility handler.
        * 
        * @param event The DOM event that caused the game to pause, if any.
        */
        gamePaused(event: any): void;

        /**
        * Called by the Stage visibility handler.
        * 
        * @param event The DOM event that caused the game to pause, if any.
        */
        gameResumed(event: any): void;

        /**
        * Parses a Game configuration object.
        */
        parseConfig(config: any): void;
        removeFromDOM(canvas: HTMLCanvasElement): void;

        /**
        * Checks if the device is capable of using the requested renderer and sets it up or an alternative if not.
        */
        setUpRenderer(): void;

        /**
        * Displays a Phaser version debug header in the console.
        */
        showDebugHeader(): void;

        /**
        * When stepping is enabled you must call this function directly (perhaps via a DOM button?) to advance the game loop by one frame.
        * This is extremely useful to hard to track down errors! Use the internal stepCount property to monitor progress.
        */
        step(): void;

        /**
        * The core game loop.
        * 
        * @param time The current time as provided by RequestAnimationFrame.
        */
        update(time: number): void;

        /**
        * Updates all logic subsystems in Phaser. Called automatically by Game.update.
        * 
        * @param timeStep The current timeStep value as determined by Game.update.
        */
        updateLogic(timeStep: number): void;

        /**
        * Runs the Render cycle.
        * It starts by calling State.preRender. In here you can do any last minute adjustments of display objects as required.
        * It then calls the renderer, which renders the entire display list, starting from the Stage object and working down.
        * It then calls plugin.render on any loaded plugins, in the order in which they were enabled.
        * After this State.render is called. Any rendering that happens here will take place on-top of the display list.
        * Finally plugin.postRender is called on any loaded plugins, in the order in which they were enabled.
        * This method is called automatically by Game.update, you don't need to call it directly.
        * Should you wish to have fine-grained control over when Phaser renders then use the `Game.lockRender` boolean.
        * Phaser will only render when this boolean is `false`.
        * 
        * @param elapsedTime The time elapsed since the last update.
        */
        updateRender(timeStep: number): void;

    }


    /**
    * The GameObjectCreator is a quick way to create common game objects _without_ adding them to the game world.
    * The object creator can be accessed with {@linkcode Phaser.Game#make `game.make`}.
    */
    class GameObjectCreator {


        /**
        * The GameObjectCreator is a quick way to create common game objects _without_ adding them to the game world.
        * The object creator can be accessed with {@linkcode Phaser.Game#make `game.make`}.
        * 
        * @param game A reference to the currently running game.
        */
        constructor(game: Phaser.Game);


        /**
        * A reference to the currently running Game.
        */
        game: Phaser.Game;

        /**
        * A reference to the game world.
        */
        world: Phaser.World;


        /**
        * Creates a new Sound object.
        * 
        * @param key The Game.cache key of the sound that this object will use.
        * @param volume The volume at which the sound will be played. - Default: 1
        * @param loop Whether or not the sound will loop.
        * @param connect Controls if the created Sound object will connect to the master gainNode of the SoundManager when running under WebAudio. - Default: true
        * @return The newly created text object.
        */
        audio(key: string, volume?: number, loop?: boolean, connect?: boolean): Phaser.Sound;

        /**
        * Creates a new AudioSprite object.
        * 
        * @param key The Game.cache key of the sound that this object will use.
        * @return The newly created AudioSprite object.
        */
        audioSprite(key: string): Phaser.AudioSprite;

        /**
        * Create a BitmpaData object.
        * 
        * A BitmapData object can be manipulated and drawn to like a traditional Canvas object and used to texture Sprites.
        * 
        * @param width The width of the BitmapData in pixels. - Default: 256
        * @param height The height of the BitmapData in pixels. - Default: 256
        * @param key Asset key for the BitmapData when stored in the Cache (see addToCache parameter). - Default: ''
        * @param addToCache Should this BitmapData be added to the Game.Cache? If so you can retrieve it with Cache.getBitmapData(key)
        * @return The newly created BitmapData object.
        */
        bitmapData(width?: number, height?: number, key?: string, addToCache?: boolean): Phaser.BitmapData;

        /**
        * Create a new BitmapText object.
        * 
        * BitmapText objects work by taking a texture file and an XML file that describes the font structure.
        * It then generates a new Sprite object for each letter of the text, proportionally spaced out and aligned to
        * match the font structure.
        * 
        * BitmapText objects are less flexible than Text objects, in that they have less features such as shadows, fills and the ability
        * to use Web Fonts. However you trade this flexibility for pure rendering speed. You can also create visually compelling BitmapTexts by
        * processing the font texture in an image editor first, applying fills and any other effects required.
        * 
        * To create multi-line text insert \r, \n or \r\n escape codes into the text string.
        * 
        * To create a BitmapText data files you can use:
        * 
        * BMFont (Windows, free): http://www.angelcode.com/products/bmfont/
        * Glyph Designer (OS X, commercial): http://www.71squared.com/en/glyphdesigner
        * Littera (Web-based, free): http://kvazars.com/littera/
        * 
        * @param x X coordinate to display the BitmapText object at.
        * @param y Y coordinate to display the BitmapText object at.
        * @param font The key of the BitmapText as stored in Phaser.Cache.
        * @param text The text that will be rendered. This can also be set later via BitmapText.text. - Default: ''
        * @param size The size the font will be rendered at in pixels. - Default: 32
        * @param align The alignment of multi-line text. Has no effect if there is only one line of text. - Default: 'left'
        * @return The newly created bitmapText object.
        */
        bitmapText(x: number, y: number, font: string, text?: string, size?: number, align?: string): Phaser.BitmapText;

        /**
        * Creates a new Button object.
        * 
        * @param x X position of the new button object.
        * @param y Y position of the new button object.
        * @param key The image key as defined in the Game.Cache to use as the texture for this button.
        * @param callback The function to call when this button is pressed
        * @param callbackContext The context in which the callback will be called (usually 'this')
        * @param overFrame This is the frame or frameName that will be set when this button is in an over state. Give either a number to use a frame ID or a string for a frame name.
        * @param outFrame This is the frame or frameName that will be set when this button is in an out state. Give either a number to use a frame ID or a string for a frame name.
        * @param downFrame This is the frame or frameName that will be set when this button is in a down state. Give either a number to use a frame ID or a string for a frame name.
        * @param upFrame This is the frame or frameName that will be set when this button is in an up state. Give either a number to use a frame ID or a string for a frame name.
        * @return The newly created button object.
        */
        button(x?: number, y?: number, key?: string, callback?: Function, callbackContext?: any, overFrame?: any, outFrame?: any, downFrame?: any, upFrame?: any): Phaser.Button;

        /**
        * Creat a new Emitter.
        * 
        * An Emitter is a lightweight particle emitter. It can be used for one-time explosions or for
        * continuous effects like rain and fire. All it really does is launch Particle objects out
        * at set intervals, and fixes their positions and velocities accorindgly.
        * 
        * @param x The x coordinate within the Emitter that the particles are emitted from.
        * @param y The y coordinate within the Emitter that the particles are emitted from.
        * @param maxParticles The total number of particles in this emitter. - Default: 50
        * @return The newly created emitter object.
        */
        emitter(x?: number, y?: number, maxParticles?: number): Phaser.Particles.Arcade.Emitter;

        /**
        * A WebGL shader/filter that can be applied to Sprites.
        * 
        * @param filter The name of the filter you wish to create, for example HueRotate or SineWave.
        * @param undefined Whatever parameters are needed to be passed to the filter init function.
        * @return The newly created Phaser.Filter object.
        */
        filter(filter: any, ...args: any[]): Phaser.Filter;

        /**
        * Creates a new Graphics object.
        * 
        * @param x X position of the new graphics object.
        * @param y Y position of the new graphics object.
        * @return The newly created graphics object.
        */
        graphics(x?: number, y?: number): Phaser.Graphics;

        /**
        * A Group is a container for display objects that allows for fast pooling, recycling and collision checks.
        * 
        * @param parent The parent Group or DisplayObjectContainer that will hold this group, if any.
        * @param name A name for this Group. Not used internally but useful for debugging. - Default: 'group'
        * @param addToStage If set to true this Group will be added directly to the Game.Stage instead of Game.World.
        * @param enableBody If true all Sprites created with `Group.create` or `Group.createMulitple` will have a physics body created on them. Change the body type with physicsBodyType.
        * @param physicsBodyType If enableBody is true this is the type of physics body that is created on new Sprites. Phaser.Physics.ARCADE, Phaser.Physics.P2, Phaser.Physics.NINJA, etc.
        * @return The newly created Group.
        */
        group(parent?: any, name?: string, addToStage?: boolean, enableBody?: boolean, physicsBodyType?: number): Phaser.Group;

        /**
        * Create a new Image object.
        * 
        * An Image is a light-weight object you can use to display anything that doesn't need physics or animation.
        * It can still rotate, scale, crop and receive input events. This makes it perfect for logos, backgrounds, simple buttons and other non-Sprite graphics.
        * 
        * @param x X position of the image.
        * @param y Y position of the image.
        * @param key This is the image or texture used by the Sprite during rendering. It can be a string which is a reference to the Cache entry, or an instance of a RenderTexture or PIXI.Texture.
        * @param frame If the sprite uses an image from a texture atlas or sprite sheet you can pass the frame here. Either a number for a frame ID or a string for a frame name.
        * @return the newly created sprite object.
        */
        image(x: number, y: number, key?: any, frame?: any): Phaser.Image;

        /**
        * A dynamic initially blank canvas to which images can be drawn.
        * 
        * @param width the width of the RenderTexture. - Default: 100
        * @param height the height of the RenderTexture. - Default: 100
        * @param key Asset key for the RenderTexture when stored in the Cache (see addToCache parameter). - Default: ''
        * @param addToCache Should this RenderTexture be added to the Game.Cache? If so you can retrieve it with Cache.getTexture(key)
        * @return The newly created RenderTexture object.
        */
        renderTexture(width?: number, height?: number, key?: any, addToCache?: boolean): Phaser.RenderTexture;

        /**
        * Create a new RetroFont object.
        * 
        * A RetroFont can be used as a texture for an Image or Sprite and optionally add it to the Cache.
        * A RetroFont uses a bitmap which contains fixed with characters for the font set. You use character spacing to define the set.
        * If you need variable width character support then use a BitmapText object instead. The main difference between a RetroFont and a BitmapText
        * is that a RetroFont creates a single texture that you can apply to a game object, where-as a BitmapText creates one Sprite object per letter of text.
        * The texture can be asssigned or one or multiple images/sprites, but note that the text the RetroFont uses will be shared across them all,
        * i.e. if you need each Image to have different text in it, then you need to create multiple RetroFont objects.
        * 
        * @param font The key of the image in the Game.Cache that the RetroFont will use.
        * @param characterWidth The width of each character in the font set.
        * @param characterHeight The height of each character in the font set.
        * @param chars The characters used in the font set, in display order. You can use the TEXT_SET consts for common font set arrangements.
        * @param charsPerRow The number of characters per row in the font set.
        * @param xSpacing If the characters in the font set have horizontal spacing between them set the required amount here.
        * @param ySpacing If the characters in the font set have vertical spacing between them set the required amount here.
        * @param xOffset If the font set doesn't start at the top left of the given image, specify the X coordinate offset here.
        * @param yOffset If the font set doesn't start at the top left of the given image, specify the Y coordinate offset here.
        * @return The newly created RetroFont texture which can be applied to an Image or Sprite.
        */
        retroFont(font: string, characterWidth: number, characterHeight: number, chars: string, charsPerRow: number, xSpacing?: number, ySpacing?: number, xOffset?: number, yOffset?: number): Phaser.RetroFont;

        /**
        * Creates a new Rope object.
        * 
        * @param x The x coordinate (in world space) to position the Rope at.
        * @param y The y coordinate (in world space) to position the Rope at.
        * @param width The width of the Rope.
        * @param height The height of the Rope.
        * @param key This is the image or texture used by the TileSprite during rendering. It can be a string which is a reference to the Cache entry, or an instance of a RenderTexture or PIXI.Texture.
        * @param frame If this Rope is using part of a sprite sheet or texture atlas you can specify the exact frame to use by giving a string or numeric index.
        * @return The newly created rope object.
        */
        rope(x: number, y: number, key: any, frame?: any, points?: Phaser.Point[]): Phaser.Rope;

        /**
        * Creates a new Sound object.
        * 
        * @param key The Game.cache key of the sound that this object will use.
        * @param volume The volume at which the sound will be played. - Default: 1
        * @param loop Whether or not the sound will loop.
        * @param connect Controls if the created Sound object will connect to the master gainNode of the SoundManager when running under WebAudio. - Default: true
        * @return The newly created text object.
        */
        sound(key: string, volume?: number, loop?: boolean, connect?: boolean): Phaser.Sound;

        /**
        * Create a new Sprite with specific position and sprite sheet key.
        * 
        * @param x X position of the new sprite.
        * @param y Y position of the new sprite.
        * @param key This is the image or texture used by the Sprite during rendering. It can be a string which is a reference to the Cache entry, or an instance of a RenderTexture or PIXI.Texture.
        * @param frame If the sprite uses an image from a texture atlas or sprite sheet you can pass the frame here. Either a number for a frame ID or a string for a frame name.
        * @return the newly created sprite object.
        */
        sprite(x: number, y: number, key?: any, frame?: any): Phaser.Sprite;

        /**
        * Create a new SpriteBatch.
        * 
        * @param parent The parent Group or DisplayObjectContainer that will hold this group, if any.
        * @param name A name for this Group. Not used internally but useful for debugging. - Default: 'group'
        * @param addToStage If set to true this Group will be added directly to the Game.Stage instead of Game.World.
        * @return The newly created group.
        */
        spriteBatch(parent: any, name?: String, addToStage?: boolean): Phaser.SpriteBatch;

        /**
        * Creates a new Text object.
        * 
        * @param x X position of the new text object.
        * @param y Y position of the new text object.
        * @param text The actual text that will be written.
        * @param style The style object containing style attributes like font, font size , etc.
        * @return The newly created text object.
        */
        text(x: number, y: number, text?: string, style?: PhaserTextStyle): Phaser.Text;

        /**
        * Creates a new Phaser.Tilemap object.
        * 
        * The map can either be populated with data from a Tiled JSON file or from a CSV file.
        * To do this pass the Cache key as the first parameter. When using Tiled data you need only provide the key.
        * When using CSV data you must provide the key and the tileWidth and tileHeight parameters.
        * If creating a blank tilemap to be populated later, you can either specify no parameters at all and then use `Tilemap.create` or pass the map and tile dimensions here.
        * Note that all Tilemaps use a base tile size to calculate dimensions from, but that a TilemapLayer may have its own unique tile size that overrides it.
        * 
        * @param key The key of the tilemap data as stored in the Cache. If you're creating a blank map either leave this parameter out or pass `null`.
        * @param tileWidth The pixel width of a single map tile. If using CSV data you must specify this. Not required if using Tiled map data. - Default: 32
        * @param tileHeight The pixel height of a single map tile. If using CSV data you must specify this. Not required if using Tiled map data. - Default: 32
        * @param width The width of the map in tiles. If this map is created from Tiled or CSV data you don't need to specify this. - Default: 10
        * @param height The height of the map in tiles. If this map is created from Tiled or CSV data you don't need to specify this. - Default: 10
        */
        tilemap(key: string, tileWidth?: number, tileHeight?: number, width?: number, height?: number): Phaser.Tilemap;

        /**
        * Creates a new TileSprite object.
        * 
        * @param x The x coordinate (in world space) to position the TileSprite at.
        * @param y The y coordinate (in world space) to position the TileSprite at.
        * @param width The width of the TileSprite.
        * @param height The height of the TileSprite.
        * @param key This is the image or texture used by the TileSprite during rendering. It can be a string which is a reference to the Phaser Image Cache entry, or an instance of a PIXI.Texture or BitmapData.
        * @param frame If this TileSprite is using part of a sprite sheet or texture atlas you can specify the exact frame to use by giving a string or numeric index.
        * @return The newly created tileSprite object.
        */
        tileSprite(x: number, y: number, width: number, height: number, key: any, frame: any): Phaser.TileSprite;

        /**
        * Create a tween object for a specific object.
        * 
        * The object can be any JavaScript object or Phaser object such as Sprite.
        * 
        * @param obj Object the tween will be run on.
        * @return The Tween object.
        */
        tween(obj: any): Phaser.Tween;

    }


    /**
    * The GameObjectFactory is a quick way to create many common game objects
    * using {@linkcode Phaser.Game#add `game.add`}.
    * 
    * Created objects are _automatically added_ to the appropriate Manager, World, or manually specified parent Group.
    */
    class GameObjectFactory {


        /**
        * The GameObjectFactory is a quick way to create many common game objects
        * using {@linkcode Phaser.Game#add `game.add`}.
        * 
        * Created objects are _automatically added_ to the appropriate Manager, World, or manually specified parent Group.
        * 
        * @param game A reference to the currently running game.
        */
        constructor(game: Phaser.Game);


        /**
        * A reference to the currently running Game.
        */
        game: Phaser.Game;

        /**
        * A reference to the game world.
        */
        world: Phaser.World;


        /**
        * Creates a new Sound object.
        * 
        * @param key The Game.cache key of the sound that this object will use.
        * @param volume The volume at which the sound will be played. - Default: 1
        * @param loop Whether or not the sound will loop.
        * @param connect Controls if the created Sound object will connect to the master gainNode of the SoundManager when running under WebAudio. - Default: true
        * @return The newly created sound object.
        */
        audio(key: string, volume?: number, loop?: boolean, connect?: boolean): Phaser.Sound;

        /**
        * Creates a new AudioSprite object.
        * 
        * @param key The Game.cache key of the sound that this object will use.
        * @return The newly created AudioSprite object.
        */
        audioSprite(key: string): Phaser.AudioSprite;

        /**
        * Create a BitmapData object.
        * 
        * A BitmapData object can be manipulated and drawn to like a traditional Canvas object and used to texture Sprites.
        * 
        * @param width The width of the BitmapData in pixels. - Default: 256
        * @param height The height of the BitmapData in pixels. - Default: 256
        * @param key Asset key for the BitmapData when stored in the Cache (see addToCache parameter). - Default: ''
        * @param addToCache Should this BitmapData be added to the Game.Cache? If so you can retrieve it with Cache.getBitmapData(key)
        * @return The newly created BitmapData object.
        */
        bitmapData(width?: number, height?: number, key?: string, addToCache?: boolean): Phaser.BitmapData;

        /**
        * Create a new BitmapText object.
        * 
        * BitmapText objects work by taking a texture file and an XML file that describes the font structure.
        * It then generates a new Sprite object for each letter of the text, proportionally spaced out and aligned to
        * match the font structure.
        * 
        * BitmapText objects are less flexible than Text objects, in that they have less features such as shadows, fills and the ability
        * to use Web Fonts. However you trade this flexibility for pure rendering speed. You can also create visually compelling BitmapTexts by
        * processing the font texture in an image editor first, applying fills and any other effects required.
        * 
        * To create multi-line text insert \r, \n or \r\n escape codes into the text string.
        * 
        * To create a BitmapText data files you can use:
        * 
        * BMFont (Windows, free): http://www.angelcode.com/products/bmfont/
        * Glyph Designer (OS X, commercial): http://www.71squared.com/en/glyphdesigner
        * Littera (Web-based, free): http://kvazars.com/littera/
        * 
        * @param x X coordinate to display the BitmapText object at.
        * @param y Y coordinate to display the BitmapText object at.
        * @param font The key of the BitmapText as stored in Phaser.Cache.
        * @param text The text that will be rendered. This can also be set later via BitmapText.text. - Default: ''
        * @param size The size the font will be rendered at in pixels. - Default: 32
        * @param group Optional Group to add the object to. If not specified it will be added to the World group.
        * @return The newly created bitmapText object.
        */
        bitmapText(x: number, y: number, font: string, text?: string, size?: number, group?: Phaser.Group): Phaser.BitmapText;

        /**
        * Creates a new Button object.
        * 
        * @param x The x coordinate of the Button. The coordinate is relative to any parent container this button may be in.
        * @param y The y coordinate of the Button. The coordinate is relative to any parent container this button may be in.
        * @param key The image key as defined in the Game.Cache to use as the texture for this button.
        * @param callback The function to call when this button is pressed
        * @param callbackContext The context in which the callback will be called (usually 'this')
        * @param overFrame This is the frame or frameName that will be set when this button is in an over state. Give either a number to use a frame ID or a string for a frame name.
        * @param outFrame This is the frame or frameName that will be set when this button is in an out state. Give either a number to use a frame ID or a string for a frame name.
        * @param downFrame This is the frame or frameName that will be set when this button is in a down state. Give either a number to use a frame ID or a string for a frame name.
        * @param upFrame This is the frame or frameName that will be set when this button is in an up state. Give either a number to use a frame ID or a string for a frame name.
        * @param group Optional Group to add the object to. If not specified it will be added to the World group.
        * @return The newly created Button object.
        */
        button(x?: number, y?: number, key?: string, callback?: Function, callbackContext?: any, overFrame?: any, outFrame?: any, downFrame?: any, upFrame?: any, group?: Phaser.Group): Phaser.Button;

        /**
        * Create a new Emitter.
        * 
        * A particle emitter can be used for one-time explosions or for
        * continuous effects like rain and fire. All it really does is launch Particle objects out
        * at set intervals, and fixes their positions and velocities accordingly.
        * 
        * @param x The x coordinate within the Emitter that the particles are emitted from.
        * @param y The y coordinate within the Emitter that the particles are emitted from.
        * @param maxParticles The total number of particles in this emitter. - Default: 50
        * @return The newly created emitter object.
        */
        emitter(x?: number, y?: number, maxParticles?: number): Phaser.Particles.Arcade.Emitter;

        /**
        * Adds an existing display object to the game world.
        * 
        * @param object An instance of Phaser.Sprite, Phaser.Button or any other display object.
        * @return The child that was added to the World.
        */
        existing(object: any): any;

        /**
        * A WebGL shader/filter that can be applied to Sprites.
        * 
        * @param filter The name of the filter you wish to create, for example HueRotate or SineWave.
        * @param undefined Whatever parameters are needed to be passed to the filter init function.
        * @return The newly created Phaser.Filter object.
        */
        filter(filter: string, ...args: any[]): Phaser.Filter;

        /**
        * Creates a new Graphics object.
        * 
        * @param x The x coordinate of the Graphic. The coordinate is relative to any parent container this object may be in.
        * @param y The y coordinate of the Graphic. The coordinate is relative to any parent container this object may be in.
        * @param group Optional Group to add the object to. If not specified it will be added to the World group.
        * @return The newly created graphics object.
        */
        graphics(x?: number, y?: number, group?: Phaser.Group): Phaser.Graphics;

        /**
        * A Group is a container for display objects that allows for fast pooling, recycling and collision checks.
        * 
        * @param parent The parent Group or DisplayObjectContainer that will hold this group, if any. If set to null the Group won't be added to the display list. If undefined it will be added to World by default.
        * @param name A name for this Group. Not used internally but useful for debugging. - Default: 'group'
        * @param addToStage If set to true this Group will be added directly to the Game.Stage instead of Game.World.
        * @param enableBody If true all Sprites created with `Group.create` or `Group.createMulitple` will have a physics body created on them. Change the body type with physicsBodyType.
        * @param physicsBodyType If enableBody is true this is the type of physics body that is created on new Sprites. Phaser.Physics.ARCADE, Phaser.Physics.P2, Phaser.Physics.NINJA, etc.
        * @return The newly created Group.
        */
        group(parent?: any, name?: string, addToStage?: boolean, enableBody?: boolean, physicsBodyType?: number): Phaser.Group;

        /**
        * Create a new `Image` object.
        * 
        * An Image is a light-weight object you can use to display anything that doesn't need physics or animation.
        * 
        * It can still rotate, scale, crop and receive input events.
        * This makes it perfect for logos, backgrounds, simple buttons and other non-Sprite graphics.
        * 
        * @param x The x coordinate of the Image. The coordinate is relative to any parent container this Image may be in.
        * @param y The y coordinate of the Image. The coordinate is relative to any parent container this Image may be in.
        * @param key The image used as a texture by this display object during rendering. If a string Phaser will get for an entry in the Image Cache. Or it can be an instance of a RenderTexture, BitmapData, Video or PIXI.Texture.
        * @param frame If a Texture Atlas or Sprite Sheet is used this allows you to specify the frame to be used. Use either an integer for a Frame ID or a string for a frame name.
        * @param group Optional Group to add the object to. If not specified it will be added to the World group.
        * @return The newly created Image object.
        */
        image(x?: number, y?: number, key?: any, frame?: any, group?: Phaser.Group): Phaser.Image;

        /**
        * A Group is a container for display objects that allows for fast pooling, recycling and collision checks.
        * 
        * A Physics Group is the same as an ordinary Group except that is has enableBody turned on by default, so any Sprites it creates
        * are automatically given a physics body.
        * 
        * @param physicsBodyType If enableBody is true this is the type of physics body that is created on new Sprites. Phaser.Physics.ARCADE, Phaser.Physics.P2JS, Phaser.Physics.NINJA, etc. - Default: Phaser.Physics.ARCADE
        * @param parent The parent Group or DisplayObjectContainer that will hold this group, if any. If set to null the Group won't be added to the display list. If undefined it will be added to World by default.
        * @param name A name for this Group. Not used internally but useful for debugging. - Default: 'group'
        * @param addToStage If set to true this Group will be added directly to the Game.Stage instead of Game.World.
        * @return The newly created Group.
        */
        physicsGroup(physicsBodyType?: number, parent?: any, name?: string, addToStage?: boolean): Phaser.Group;

        /**
        * Add a new Plugin into the PluginManager.
        * 
        * The Plugin must have 2 properties: `game` and `parent`. Plugin.game is set to the game reference the PluginManager uses, and parent is set to the PluginManager.
        * 
        * @param plugin The Plugin to add into the PluginManager. This can be a function or an existing object.
        * @param args Additional parameters that will be passed to the Plugin.init method.
        * @return The Plugin that was added to the manager.
        */
        plugin(plugin: Phaser.Plugin, ...parameter: any[]): Phaser.Plugin;

        /**
        * A dynamic initially blank canvas to which images can be drawn.
        * 
        * @param width the width of the RenderTexture. - Default: 100
        * @param height the height of the RenderTexture. - Default: 100
        * @param key Asset key for the RenderTexture when stored in the Cache (see addToCache parameter). - Default: ''
        * @param addToCache Should this RenderTexture be added to the Game.Cache? If so you can retrieve it with Cache.getTexture(key)
        * @return The newly created RenderTexture object.
        */
        renderTexture(width?: number, height?: number, key?: string, addToCache?: boolean): Phaser.RenderTexture;

        /**
        * Create a new RetroFont object.
        * 
        * A RetroFont can be used as a texture for an Image or Sprite and optionally add it to the Cache.
        * A RetroFont uses a bitmap which contains fixed with characters for the font set. You use character spacing to define the set.
        * If you need variable width character support then use a BitmapText object instead. The main difference between a RetroFont and a BitmapText
        * is that a RetroFont creates a single texture that you can apply to a game object, where-as a BitmapText creates one Sprite object per letter of text.
        * The texture can be asssigned or one or multiple images/sprites, but note that the text the RetroFont uses will be shared across them all,
        * i.e. if you need each Image to have different text in it, then you need to create multiple RetroFont objects.
        * 
        * @param font The key of the image in the Game.Cache that the RetroFont will use.
        * @param characterWidth The width of each character in the font set.
        * @param characterHeight The height of each character in the font set.
        * @param chars The characters used in the font set, in display order. You can use the TEXT_SET consts for common font set arrangements.
        * @param charsPerRow The number of characters per row in the font set.
        * @param xSpacing If the characters in the font set have horizontal spacing between them set the required amount here.
        * @param ySpacing If the characters in the font set have vertical spacing between them set the required amount here.
        * @param xOffset If the font set doesn't start at the top left of the given image, specify the X coordinate offset here.
        * @param yOffset If the font set doesn't start at the top left of the given image, specify the Y coordinate offset here.
        * @return The newly created RetroFont texture which can be applied to an Image or Sprite.
        */
        retroFont(font: string, characterWidth: number, characterHeight: number, chars: string, charsPerRow: number, xSpacing?: number, ySpacing?: number, xOffset?: number, yOffset?: number): Phaser.RetroFont;

        /**
        * Creates a new Rope object.
        * 
        * Example usage: https://github.com/codevinsky/phaser-rope-demo/blob/master/dist/demo.js
        * 
        * @param x The x coordinate of the Rope. The coordinate is relative to any parent container this rope may be in.
        * @param y The y coordinate of the Rope. The coordinate is relative to any parent container this rope may be in.
        * @param key The image used as a texture by this display object during rendering. If a string Phaser will get for an entry in the Image Cache. Or it can be an instance of a RenderTexture, BitmapData, Video or PIXI.Texture.
        * @param frame If a Texture Atlas or Sprite Sheet is used this allows you to specify the frame to be used. Use either an integer for a Frame ID or a string for a frame name.
        * @param points An array of {Phaser.Point}.
        * @param group Optional Group to add the object to. If not specified it will be added to the World group.
        * @return The newly created Rope object.
        */
        rope(x?: number, y?: number, key?: any, frame?: any, points?: Phaser.Point[]): Phaser.Rope;

        /**
        * Creates a new Sound object.
        * 
        * @param key The Game.cache key of the sound that this object will use.
        * @param volume The volume at which the sound will be played. - Default: 1
        * @param loop Whether or not the sound will loop.
        * @param connect Controls if the created Sound object will connect to the master gainNode of the SoundManager when running under WebAudio. - Default: true
        * @return The newly created sound object.
        */
        sound(key: string, volume?: number, loop?: boolean, connect?: boolean): Phaser.Sound;

        /**
        * Create a new Sprite with specific position and sprite sheet key.
        * 
        * At its most basic a Sprite consists of a set of coordinates and a texture that is used when rendered.
        * They also contain additional properties allowing for physics motion (via Sprite.body), input handling (via Sprite.input),
        * events (via Sprite.events), animation (via Sprite.animations), camera culling and more. Please see the Examples for use cases.
        * 
        * @param x The x coordinate of the sprite. The coordinate is relative to any parent container this sprite may be in.
        * @param y The y coordinate of the sprite. The coordinate is relative to any parent container this sprite may be in.
        * @param key The image used as a texture by this display object during rendering. If a string Phaser will get for an entry in the Image Cache. Or it can be an instance of a RenderTexture, BitmapData, Video or PIXI.Texture.
        * @param frame If a Texture Atlas or Sprite Sheet is used this allows you to specify the frame to be used. Use either an integer for a Frame ID or a string for a frame name.
        * @param group Optional Group to add the object to. If not specified it will be added to the World group.
        * @return The newly created Sprite object.
        */
        sprite(x?: number, y?: number, key?: any, frame?: any, group?: Phaser.Group): Phaser.Sprite;

        /**
        * A SpriteBatch is a really fast version of a Phaser Group built solely for speed.
        * Use when you need a lot of sprites or particles all sharing the same texture.
        * The speed gains are specifically for WebGL. In Canvas mode you won't see any real difference.
        * 
        * @param parent The parent Group that will hold this Sprite Batch. Set to `undefined` or `null` to add directly to game.world.
        * @param name A name for this Sprite Batch. Not used internally but useful for debugging. - Default: 'group'
        * @param addToStage If set to true this Sprite Batch will be added directly to the Game.Stage instead of the parent.
        * @return The newly created Sprite Batch.
        */
        spriteBatch(parent: any, name?: string, addToStage?: boolean): Phaser.Group;

        /**
        * Creates a new Text object.
        * 
        * @param x The x coordinate of the Text. The coordinate is relative to any parent container this text may be in.
        * @param y The y coordinate of the Text. The coordinate is relative to any parent container this text may be in.
        * @param text The text string that will be displayed. - Default: ''
        * @param style The style object containing style attributes like font, font size , etc.
        * @param group Optional Group to add the object to. If not specified it will be added to the World group.
        * @return The newly created text object.
        */
        text(x?: number, y?: number, text?: string, style?: PhaserTextStyle, group?: Phaser.Group): Phaser.Text;

        /**
        * Creates a new Phaser.Tilemap object.
        * 
        * The map can either be populated with data from a Tiled JSON file or from a CSV file.
        * To do this pass the Cache key as the first parameter. When using Tiled data you need only provide the key.
        * When using CSV data you must provide the key and the tileWidth and tileHeight parameters.
        * If creating a blank tilemap to be populated later, you can either specify no parameters at all and then use `Tilemap.create` or pass the map and tile dimensions here.
        * Note that all Tilemaps use a base tile size to calculate dimensions from, but that a TilemapLayer may have its own unique tile size that overrides it.
        * 
        * @param key The key of the tilemap data as stored in the Cache. If you're creating a blank map either leave this parameter out or pass `null`.
        * @param tileWidth The pixel width of a single map tile. If using CSV data you must specify this. Not required if using Tiled map data. - Default: 32
        * @param tileHeight The pixel height of a single map tile. If using CSV data you must specify this. Not required if using Tiled map data. - Default: 32
        * @param width The width of the map in tiles. If this map is created from Tiled or CSV data you don't need to specify this. - Default: 10
        * @param height The height of the map in tiles. If this map is created from Tiled or CSV data you don't need to specify this. - Default: 10
        * @return The newly created tilemap object.
        */
        tilemap(key?: string, tileWidth?: number, tileHeight?: number, width?: number, height?: number): Phaser.Tilemap;

        /**
        * Creates a new TileSprite object.
        * 
        * @param x The x coordinate of the TileSprite. The coordinate is relative to any parent container this TileSprite may be in.
        * @param y The y coordinate of the TileSprite. The coordinate is relative to any parent container this TileSprite may be in.
        * @param width The width of the TileSprite.
        * @param height The height of the TileSprite.
        * @param key This is the image or texture used by the TileSprite during rendering. It can be a string which is a reference to the Phaser Image Cache entry, or an instance of a PIXI.Texture or BitmapData.
        * @param frame If a Texture Atlas or Sprite Sheet is used this allows you to specify the frame to be used. Use either an integer for a Frame ID or a string for a frame name.
        * @param group Optional Group to add the object to. If not specified it will be added to the World group.
        * @return The newly created TileSprite object.
        */
        tileSprite(x: number, y: number, width: number, height: number, key?: any, frame?: any, group?: Phaser.Group): Phaser.TileSprite;

        /**
        * Create a tween on a specific object.
        * 
        * The object can be any JavaScript object or Phaser object such as Sprite.
        * 
        * @param object Object the tween will be run on.
        * @return The newly created Phaser.Tween object.
        */
        tween(obj: any): Phaser.Tween;

        /**
        * Weapons provide the ability to easily create a bullet pool and manager.
        * 
        * Weapons fire Phaser.Bullet objects, which are essentially Sprites with a few extra properties.
        * The Bullets are enabled for Arcade Physics. They do not currently work with P2 Physics.
        * 
        * The Bullets are created inside of `Weapon.bullets`, which is a Phaser.Group instance. Anything you
        * can usually do with a Group, such as move it around the display list, iterate it, etc can be done
        * to the bullets Group too.
        * 
        * Bullets can have textures and even animations. You can control the speed at which they are fired,
        * the firing rate, the firing angle, and even set things like gravity for them.
        * 
        * @param quantity The quantity of bullets to seed the Weapon with. If -1 it will set the pool to automatically expand. - Default: 1
        * @param key The image used as a texture by the bullets during rendering. If a string Phaser will get for an entry in the Image Cache. Or it can be an instance of a RenderTexture, BitmapData, Video or PIXI.Texture.
        * @param frame If a Texture Atlas or Sprite Sheet is used this allows you to specify the frame to be used by the bullets. Use either an integer for a Frame ID or a string for a frame name.
        * @param group Optional Group to add the Weapon to. If not specified it will be added to the World group.
        * @param bulletClass The Class of the bullets that are launched by this Weapon. See {@link Phaser.Weapon#bulletClass}
        * @return A Weapon instance.
        */
        weapon(quantity?: number, key?: any, frame?: any, group?: Phaser.Group, bulletClass?: Phaser.Bullet): Phaser.Weapon;

        /**
        * Create a Video object.
        * 
        * This will return a Phaser.Video object which you can pass to a Sprite to be used as a texture.
        * 
        * @param key The key of the video file in the Phaser.Cache that this Video object will play. Set to `null` or leave undefined if you wish to use a webcam as the source. See `startMediaStream` to start webcam capture.
        * @param url If the video hasn't been loaded then you can provide a full URL to the file here (make sure to set key to null)
        * @return The newly created Video object.
        */
        video(key?: string, url?: string): Phaser.Video;

    }


    /**
    * The Gamepad class handles gamepad input and dispatches gamepad events.
    * 
    * Remember to call `gamepad.start()`.
    * 
    * HTML5 GAMEPAD API SUPPORT IS AT AN EXPERIMENTAL STAGE!
    * At moment of writing this (end of 2013) only Chrome supports parts of it out of the box. Firefox supports it
    * via prefs flags (about:config, search gamepad). The browsers map the same controllers differently.
    * This class has constants for Windows 7 Chrome mapping of XBOX 360 controller.
    */
    class Gamepad {


        /**
        * The Gamepad class handles gamepad input and dispatches gamepad events.
        * 
        * Remember to call `gamepad.start()`.
        * 
        * HTML5 GAMEPAD API SUPPORT IS AT AN EXPERIMENTAL STAGE!
        * At moment of writing this (end of 2013) only Chrome supports parts of it out of the box. Firefox supports it
        * via prefs flags (about:config, search gamepad). The browsers map the same controllers differently.
        * This class has constants for Windows 7 Chrome mapping of XBOX 360 controller.
        * 
        * @param game A reference to the currently running game.
        */
        constructor(game: Phaser.Game);

        static BUTTON_0: number;
        static BUTTON_1: number;
        static BUTTON_2: number;
        static BUTTON_3: number;
        static BUTTON_4: number;
        static BUTTON_5: number;
        static BUTTON_6: number;
        static BUTTON_7: number;
        static BUTTON_8: number;
        static BUTTON_9: number;
        static BUTTON_10: number;
        static BUTTON_11: number;
        static BUTTON_12: number;
        static BUTTON_13: number;
        static BUTTON_14: number;
        static BUTTON_15: number;

        static AXIS_0: number;
        static AXIS_1: number;
        static AXIS_2: number;
        static AXIS_3: number;
        static AXIS_4: number;
        static AXIS_5: number;
        static AXIS_6: number;
        static AXIS_7: number;
        static AXIS_8: number;
        static AXIS_9: number;

        static XBOX360_A: number;
        static XBOX360_B: number;
        static XBOX360_X: number;
        static XBOX360_Y: number;
        static XBOX360_LEFT_BUMPER: number;
        static XBOX360_RIGHT_BUMPER: number;
        static XBOX360_LEFT_TRIGGER: number;
        static XBOX360_RIGHT_TRIGGER: number;
        static XBOX360_BACK: number;
        static XBOX360_START: number;
        static XBOX360_STICK_LEFT_BUTTON: number;
        static XBOX360_STICK_RIGHT_BUTTON: number;
        static XBOX360_DPAD_LEFT: number;
        static XBOX360_DPAD_RIGHT: number;
        static XBOX360_DPAD_UP: number;
        static XBOX360_DPAD_DOWN: number;
        static XBOX360_STICK_LEFT_X: number;
        static XBOX360_STICK_LEFT_Y: number;
        static XBOX360_STICK_RIGHT_X: number;
        static XBOX360_STICK_RIGHT_Y: number;

        static PS3XC_X: number;
        static PS3XC_CIRCLE: number;
        static PS3XC_SQUARE: number;
        static PS3XC_TRIANGLE: number;
        static PS3XC_L1: number;
        static PS3XC_R1: number;
        static PS3XC_L2: number;
        static PS3XC_R2: number;
        static PS3XC_SELECT: number;
        static PS3XC_START: number;
        static PS3XC_STICK_LEFT_BUTTON: number;
        static PS3XC_STICK_RIGHT_BUTTON: number;
        static PS3XC_DPAD_UP: number;
        static PS3XC_DPAD_DOWN: number;
        static PS3XC_DPAD_LEFT: number;
        static PS3XC_DPAD_RIGHT: number;
        static PS3XC_STICK_LEFT_X: number;
        static PS3XC_STICK_LEFT_Y: number;
        static PS3XC_STICK_RIGHT_X: number;
        static PS3XC_STICK_RIGHT_Y: number;


        /**
        * If the gamepad input is active or not - if not active it should not be updated from Input.js
        */
        active: boolean;

        /**
        * The context under which the callbacks are run.
        */
        callbackContext: any;

        /**
        * Gamepad input will only be processed if enabled.
        * Default: true
        */
        enabled: boolean;

        /**
        * Local reference to game.
        */
        game: Phaser.Game;
        onAxisCallBack: Function;

        /**
        * This callback is invoked every time any gamepad is connected
        */
        onConnectCallback: Function;

        /**
        * This callback is invoked every time any gamepad is disconnected
        */
        onDisconnectCallback: Function;

        /**
        * This callback is invoked every time any gamepad button is pressed down.
        */
        onDownCallback: Function;

        /**
        * This callback is invoked every time any gamepad button is changed to a value where value > 0 and value < 1.
        */
        onFloatCallback: Function;

        /**
        * This callback is invoked every time any gamepad button is released.
        */
        onUpCallback: Function;

        /**
        * Gamepad #1
        */
        pad1: Phaser.SinglePad;

        /**
        * Gamepad #2
        */
        pad2: Phaser.SinglePad;

        /**
        * Gamepad #3
        */
        pad3: Phaser.SinglePad;

        /**
        * Gamepad #4
        */
        pad4: Phaser.SinglePad;

        /**
        * How many live gamepads are currently connected.
        */
        padsConnected: number;

        /**
        * Whether or not gamepads are supported in current browser.
        */
        supported: boolean;


        /**
        * Add callbacks to the main Gamepad handler to handle connect/disconnect/button down/button up/axis change/float value buttons.
        * 
        * @param context The context under which the callbacks are run.
        * @param callbacks Object that takes six different callback methods:
        *                  onConnectCallback, onDisconnectCallback, onDownCallback, onUpCallback, onAxisCallback, onFloatCallback
        */
        addCallbacks(context: any, callbacks: any): void;

        /**
        * Returns true if the button is currently pressed down, on ANY gamepad.
        * 
        * @param buttonCode The buttonCode of the button to check for.
        * @return True if a button is currently down.
        */
        isDown(buttonCode: number): boolean;

        /**
        * Returns the "just pressed" state of a button from ANY gamepad connected. Just pressed is considered true if the button was pressed down within the duration given (default 250ms).
        * 
        * @param buttonCode The buttonCode of the button to check for.
        * @param duration The duration below which the button is considered as being just pressed. - Default: 250
        * @return True if the button is just pressed otherwise false.
        */
        justPressed(buttonCode: number, duration?: number): boolean;
        justReleased(buttonCode: number, duration?: number): boolean;

        /**
        * Reset all buttons/axes of all gamepads
        */
        reset(): void;

        /**
        * Sets the deadZone variable for all four gamepads
        */
        setDeadZones(value: any): void;

        /**
        * Starts the Gamepad event handling.
        * This MUST be called manually before Phaser will start polling the Gamepad API.
        */
        start(): void;

        /**
        * Stops the Gamepad event handling.
        */
        stop(): void;

        /**
        * Main gamepad update loop. Should not be called manually.
        */
        update(): void;

    }


    /**
    * A Graphics object is a way to draw primitives to your game. Primitives include forms of geometry, such as Rectangles,
    * Circles and Polygons. They also include lines, arcs and curves. When you initially create a Graphics object it will
    * be empty. To 'draw' to it you first specify a lineStyle or fillStyle (or both), and then draw a shape. For example:
    * 
    * ```
    * graphics.beginFill(0xff0000);
    * graphics.drawCircle(50, 50, 100);
    * graphics.endFill();
    * ```
    * 
    * This will draw a circle shape to the Graphics object, with a diameter of 100, located at x: 50, y: 50.
    * 
    * When a Graphics object is rendered it will render differently based on if the game is running under Canvas or
    * WebGL. Under Canvas it will use the HTML Canvas context drawing operations to draw the path. Under WebGL the
    * graphics data is decomposed into polygons. Both of these are expensive processes, especially with complex shapes.
    * 
    * If your Graphics object doesn't change much (or at all) once you've drawn your shape to it, then you will help
    * performance by calling `Graphics.generateTexture`. This will 'bake' the Graphics object into a Texture, and return it.
    * You can then use this Texture for Sprites or other display objects. If your Graphics object updates frequently then
    * you should avoid doing this, as it will constantly generate new textures, which will consume memory.
    * 
    * As you can tell, Graphics objects are a bit of a trade-off. While they are extremely useful, you need to be careful
    * in their complexity and quantity of them in your game.
    */
    class Graphics extends PIXI.DisplayObjectContainer {


        /**
        * A Graphics object is a way to draw primitives to your game. Primitives include forms of geometry, such as Rectangles,
        * Circles and Polygons. They also include lines, arcs and curves. When you initially create a Graphics object it will
        * be empty. To 'draw' to it you first specify a lineStyle or fillStyle (or both), and then draw a shape. For example:
        * 
        * ```
        * graphics.beginFill(0xff0000);
        * graphics.drawCircle(50, 50, 100);
        * graphics.endFill();
        * ```
        * 
        * This will draw a circle shape to the Graphics object, with a diameter of 100, located at x: 50, y: 50.
        * 
        * When a Graphics object is rendered it will render differently based on if the game is running under Canvas or
        * WebGL. Under Canvas it will use the HTML Canvas context drawing operations to draw the path. Under WebGL the
        * graphics data is decomposed into polygons. Both of these are expensive processes, especially with complex shapes.
        * 
        * If your Graphics object doesn't change much (or at all) once you've drawn your shape to it, then you will help
        * performance by calling `Graphics.generateTexture`. This will 'bake' the Graphics object into a Texture, and return it.
        * You can then use this Texture for Sprites or other display objects. If your Graphics object updates frequently then
        * you should avoid doing this, as it will constantly generate new textures, which will consume memory.
        * 
        * As you can tell, Graphics objects are a bit of a trade-off. While they are extremely useful, you need to be careful
        * in their complexity and quantity of them in your game.
        * 
        * @param game Current game instance.
        * @param x X position of the new graphics object.
        * @param y Y position of the new graphics object.
        */
        constructor(game: Phaser.Game, x?: number, y?: number);


        /**
        * A useful flag to control if the Game Object is alive or dead.
        * 
        * This is set automatically by the Health components `damage` method should the object run out of health.
        * Or you can toggle it via your game code.
        * 
        * This property is mostly just provided to be used by your game - it doesn't effect rendering or logic updates.
        * However you can use `Group.getFirstAlive` in conjunction with this property for fast object pooling and recycling.
        * Default: true
        */
        alive: boolean;

        /**
        * The angle property is the rotation of the Game Object in *degrees* from its original orientation.
        * 
        * Values from 0 to 180 represent clockwise rotation; values from 0 to -180 represent counterclockwise rotation.
        * 
        * Values outside this range are added to or subtracted from 360 to obtain a value within the range.
        * For example, the statement player.angle = 450 is the same as player.angle = 90.
        * 
        * If you wish to work in radians instead of degrees you can use the property `rotation` instead.
        * Working in radians is slightly faster as it doesn't have to perform any calculations.
        */
        angle: number;

        /**
        * If the Game Object is enabled for animation (such as a Phaser.Sprite) this is a reference to its AnimationManager instance.
        * Through it you can create, play, pause and stop animations.
        */
        animations: Phaser.AnimationManager;

        /**
        * A Game Object with `autoCull` set to true will check its bounds against the World Camera every frame.
        * If it is not intersecting the Camera bounds at any point then it has its `renderable` property set to `false`.
        * This keeps the Game Object alive and still processing updates, but forces it to skip the render step entirely.
        * 
        * This is a relatively expensive operation, especially if enabled on hundreds of Game Objects. So enable it only if you know it's required,
        * or you have tested performance and find it acceptable.
        */
        autoCull: boolean;

        /**
        * The blend mode to be applied to the graphic shape. Apply a value of PIXI.blendModes.NORMAL to reset the blend mode.
        * Default: PIXI.blendModes.NORMAL;
        */
        blendMode: number;

        /**
        * `body` is the Game Objects physics body. Once a Game Object is enabled for physics you access all associated
        * properties and methods via it.
        * 
        * By default Game Objects won't add themselves to any physics system and their `body` property will be `null`.
        * 
        * To enable this Game Object for physics you need to call `game.physics.enable(object, system)` where `object` is this object
        * and `system` is the Physics system you are using. If none is given it defaults to `Phaser.Physics.Arcade`.
        * 
        * You can alternatively call `game.physics.arcade.enable(object)`, or add this Game Object to a physics enabled Group.
        * 
        * Important: Enabling a Game Object for P2 or Ninja physics will automatically set its `anchor` property to 0.5,
        * so the physics body is centered on the Game Object.
        * 
        * If you need a different result then adjust or re-create the Body shape offsets manually or reset the anchor after enabling physics.
        */
        body: Phaser.Physics.Arcade.Body | Phaser.Physics.P2.Body | Phaser.Physics.Ninja.Body | any;

        /**
        * The sum of the y and height properties.
        * This is the same as `y + height - offsetY`.
        */
        bottom: number;

        /**
        * The bounds' padding used for bounds calculation.
        */
        boundsPadding: number;

        /**
        * The x/y coordinate offset applied to the top-left of the camera that this Game Object will be drawn at if `fixedToCamera` is true.
        * 
        * The values are relative to the top-left of the camera view and in addition to any parent of the Game Object on the display list.
        */
        cameraOffset: Phaser.Point;

        /**
        * The center x coordinate of the Game Object.
        * This is the same as `(x - offsetX) + (width / 2)`.
        */
        centerX: number;

        /**
        * The center y coordinate of the Game Object.
        * This is the same as `(y - offsetY) + (height / 2)`.
        */
        centerY: number;

        /**
        * If this is set to `true` the Game Object checks if it is within the World bounds each frame.
        * 
        * When it is no longer intersecting the world bounds it dispatches the `onOutOfBounds` event.
        * 
        * If it was *previously* out of bounds but is now intersecting the world bounds again it dispatches the `onEnterBounds` event.
        * 
        * It also optionally kills the Game Object if `outOfBoundsKill` is `true`.
        * 
        * When `checkWorldBounds` is enabled it forces the Game Object to calculate its full bounds every frame.
        * 
        * This is a relatively expensive operation, especially if enabled on hundreds of Game Objects. So enable it only if you know it's required,
        * or you have tested performance and find it acceptable.
        */
        checkWorldBounds: boolean;

        /**
        * The components this Game Object has installed.
        */
        components: any;

        /**
        * An empty Object that belongs to this Game Object.
        * This value isn't ever used internally by Phaser, but may be used by your own code, or
        * by Phaser Plugins, to store data that needs to be associated with the Game Object,
        * without polluting the Game Object directly.
        * Default: {}
        */
        data: any;

        /**
        * A debug flag designed for use with `Game.enableStep`.
        */
        debug: boolean;

        /**
        * As a Game Object runs through its destroy method this flag is set to true,
        * and can be checked in any sub-systems or plugins it is being destroyed from.
        */
        destroyPhase: boolean;

        /**
        * All Phaser Game Objects have an Events class which contains all of the events that are dispatched when certain things happen to this
        * Game Object, or any of its components.
        */
        events: Phaser.Events;

        /**
        * Controls if this Game Object is processed by the core game loop.
        * If this Game Object has a physics body it also controls if its physics body is updated or not.
        * When `exists` is set to `false` it will remove its physics body from the physics world if it has one.
        * It also toggles the `visible` property to false as well.
        * 
        * Setting `exists` to true will add its physics body back in to the physics world, if it has one.
        * It will also set the `visible` property to `true`.
        */
        exists: boolean;

        /**
        * The alpha value used when filling the Graphics object.
        */
        fillAlpha: number;

        /**
        * A Game Object that is "fixed" to the camera is rendered at a given x/y offsets from the top left of the camera. The offsets
        * are stored in the `cameraOffset` property, which is initialized with the current object coordinates.
        * 
        * The values are adjusted at the rendering stage, overriding the Game Objects actual world position.
        * 
        * The end result is that the Game Object will appear to be 'fixed' to the camera, regardless of where in the game world
        * the camera is viewing. This is useful if for example this Game Object is a UI item that you wish to be visible at all times
        * regardless where in the world the camera is.
        * 
        * Note that the `cameraOffset` values are in addition to any parent of this Game Object on the display list.
        * 
        * Be careful not to set `fixedToCamera` on Game Objects which are in Groups that already have `fixedToCamera` enabled on them.
        */
        fixedToCamera: boolean;

        /**
        * A Game Object is considered `fresh` if it has just been created or reset and is yet to receive a renderer transform update.
        * This property is mostly used internally by the physics systems, but is exposed for the use of plugins.
        */
        fresh: boolean;

        /**
        * A reference to the currently running Game.
        */
        game: Phaser.Game;

        /**
        * The height of the displayObjectContainer, setting this will actually modify the scale to achieve the value set
        */
        height: number;

        /**
        * Checks if the Game Objects bounds intersect with the Game Camera bounds.
        * Returns `true` if they do, otherwise `false` if fully outside of the Cameras bounds.
        */
        inCamera: boolean;

        /**
        * Checks if the Game Objects bounds are within, or intersect at any point with the Game World bounds.
        */
        inWorld: boolean;

        /**
        * The Input Handler for this Game Object.
        * 
        * By default it is disabled. If you wish this Game Object to process input events you should enable it with: `inputEnabled = true`.
        * 
        * After you have done this, this property will be a reference to the Phaser InputHandler.
        */
        input: Phaser.InputHandler;

        /**
        * By default a Game Object won't process any input events. By setting `inputEnabled` to true a Phaser.InputHandler is created
        * for this Game Object and it will then start to process click / touch events and more.
        * 
        * You can then access the Input Handler via `this.input`.
        * 
        * Note that Input related events are dispatched from `this.events`, i.e.: `events.onInputDown`.
        * 
        * If you set this property to false it will stop the Input Handler from processing any more input events.
        * 
        * If you want to _temporarily_ disable input for a Game Object, then it's better to set
        * `input.enabled = false`, as it won't reset any of the Input Handlers internal properties.
        * You can then toggle this back on as needed.
        */
        inputEnabled: boolean;

        /**
        * Whether this shape is being used as a mask.
        */
        isMask: boolean;

        /**
        * The key of the image or texture used by this Game Object during rendering.
        * If it is a string it's the string used to retrieve the texture from the Phaser Image Cache.
        * It can also be an instance of a RenderTexture, BitmapData, Video or PIXI.Texture.
        * If a Game Object is created without a key it is automatically assigned the key `__default` which is a 32x32 transparent PNG stored within the Cache.
        * If a Game Object is given a key which doesn't exist in the Image Cache it is re-assigned the key `__missing` which is a 32x32 PNG of a green box with a line through it.
        */
        key: string | Phaser.RenderTexture | Phaser.BitmapData | Phaser.Video | PIXI.Texture;

        /**
        * The left coordinate of the Game Object.
        * This is the same as `x - offsetX`.
        */
        left: number;

        /**
        * The lifespan allows you to give a Game Object a lifespan in milliseconds.
        * 
        * Once the Game Object is 'born' you can set this to a positive value.
        * 
        * It is automatically decremented by the millisecond equivalent of `game.time.physicsElapsed` each frame.
        * When it reaches zero it will call the `kill` method.
        * 
        * Very handy for particles, bullets, collectibles, or any other short-lived entity.
        */
        lifespan: number;

        /**
        * The color of any lines drawn.
        * Default: 0
        */
        lineColor: number;

        /**
        * The width (thickness) of any lines drawn.
        */
        lineWidth: number;

        /**
        * A user defined name given to this Game Object.
        * This value isn't ever used internally by Phaser, it is meant as a game level property.
        */
        name: string;

        /**
        * The amount the Game Object is visually offset from its x coordinate.
        * This is the same as `width * anchor.x`.
        * It will only be > 0 if anchor.x is not equal to zero.
        */
        offsetX: number;

        /**
        * The amount the Game Object is visually offset from its y coordinate.
        * This is the same as `height * anchor.y`.
        * It will only be > 0 if anchor.y is not equal to zero.
        */
        offsetY: number;

        /**
        * If this and the `checkWorldBounds` property are both set to `true` then the `kill` method is called as soon as `inWorld` returns false.
        */
        outOfBoundsKill: boolean;

        /**
        * A Game Object is that is pendingDestroy is flagged to have its destroy method called on the next logic update.
        * You can set it directly to allow you to flag an object to be destroyed on its next update.
        * 
        * This is extremely useful if you wish to destroy an object from within one of its own callbacks
        * such as with Buttons or other Input events.
        */
        pendingDestroy: boolean;

        /**
        * The const physics body type of this object.
        */
        physicsType: number;

        /**
        * The coordinates, in pixels, of this DisplayObject, relative to its parent container.
        * 
        * The value of this property does not reflect any positioning happening further up the display list.
        * To obtain that value please see the `worldPosition` property.
        */
        position: Phaser.Point;

        /**
        * The position the Game Object was located in the previous frame.
        */
        previousPosition: Phaser.Point;

        /**
        * The rotation the Game Object was in set to in the previous frame. Value is in radians.
        */
        previousRotation: number;

        /**
        * The render order ID is used internally by the renderer and Input Manager and should not be modified.
        * This property is mostly used internally by the renderers, but is exposed for the use of plugins.
        */
        renderOrderID: number;

        /**
        * The right coordinate of the Game Object.
        * This is the same as `x + width - offsetX`.
        */
        right: number;

        /**
        * The tint applied to the graphic shape. This is a hex value. Apply a value of 0xFFFFFF to reset the tint.
        * Default: 0xFFFFFF
        */
        tint: number;

        /**
        * The y coordinate of the Game Object.
        * This is the same as `y - offsetY`.
        */
        top: number;

        /**
        * The const type of this object.
        */
        type: number;

        /**
        * The width of the displayObjectContainer, setting this will actually modify the scale to achieve the value set
        */
        width: number;

        /**
        * The world coordinates of this Game Object in pixels.
        * Depending on where in the display list this Game Object is placed this value can differ from `position`,
        * which contains the x/y coordinates relative to the Game Objects parent.
        */
        world: Phaser.Point;

        /**
        * The multiplied alpha value of this DisplayObject. A value of 1 is fully opaque. A value of 0 is transparent.
        * This value is the calculated total, based on the alpha values of all parents of this DisplayObjects
        * in the display list.
        * 
        * To obtain, and set, the local alpha value, see the `alpha` property.
        * 
        * Note: This property is only updated at the end of the `updateTransform` call, once per render. Until
        * that happens this property will contain values based on the previous frame. Be mindful of this if
        * accessing this property outside of the normal game flow, i.e. from an asynchronous event callback.
        */
        worldAlpha: number;

        /**
        * The z depth of this Game Object within its parent Group.
        * No two objects in a Group can have the same z value.
        * This value is adjusted automatically whenever the Group hierarchy changes.
        * If you wish to re-order the layering of a Game Object then see methods like Group.moveUp or Group.bringToTop.
        */
        z: number;


        /**
        * Aligns this Game Object within another Game Object, or Rectangle, known as the
        * 'container', to one of 9 possible positions.
        * 
        * The container must be a Game Object, or Phaser.Rectangle object. This can include properties
        * such as `World.bounds` or `Camera.view`, for aligning Game Objects within the world
        * and camera bounds. Or it can include other Sprites, Images, Text objects, BitmapText,
        * TileSprites or Buttons.
        * 
        * Please note that aligning a Sprite to another Game Object does **not** make it a child of
        * the container. It simply modifies its position coordinates so it aligns with it.
        * 
        * The position constants you can use are:
        * 
        * `Phaser.TOP_LEFT`, `Phaser.TOP_CENTER`, `Phaser.TOP_RIGHT`, `Phaser.LEFT_CENTER`,
        * `Phaser.CENTER`, `Phaser.RIGHT_CENTER`, `Phaser.BOTTOM_LEFT`,
        * `Phaser.BOTTOM_CENTER` and `Phaser.BOTTOM_RIGHT`.
        * 
        * The Game Objects are placed in such a way that their _bounds_ align with the
        * container, taking into consideration rotation, scale and the anchor property.
        * This allows you to neatly align Game Objects, irrespective of their position value.
        * 
        * The optional `offsetX` and `offsetY` arguments allow you to apply extra spacing to the final
        * aligned position of the Game Object. For example:
        * 
        * `sprite.alignIn(background, Phaser.BOTTOM_RIGHT, -20, -20)`
        * 
        * Would align the `sprite` to the bottom-right, but moved 20 pixels in from the corner.
        * Think of the offsets as applying an adjustment to the containers bounds before the alignment takes place.
        * So providing a negative offset will 'shrink' the container bounds by that amount, and providing a positive
        * one expands it.
        * 
        * @param container The Game Object or Rectangle with which to align this Game Object to. Can also include properties such as `World.bounds` or `Camera.view`.
        * @param position The position constant. One of `Phaser.TOP_LEFT` (default), `Phaser.TOP_CENTER`, `Phaser.TOP_RIGHT`, `Phaser.LEFT_CENTER`, `Phaser.CENTER`, `Phaser.RIGHT_CENTER`, `Phaser.BOTTOM_LEFT`, `Phaser.BOTTOM_CENTER` or `Phaser.BOTTOM_RIGHT`.
        * @param offsetX A horizontal adjustment of the Containers bounds, applied to the aligned position of the Game Object. Use a negative value to shrink the bounds, positive to increase it.
        * @param offsetY A vertical adjustment of the Containers bounds, applied to the aligned position of the Game Object. Use a negative value to shrink the bounds, positive to increase it.
        * @return This Game Object.
        */
        alignIn(container: Phaser.Rectangle | Phaser.Sprite | Phaser.Image | Phaser.Text | Phaser.BitmapText | Phaser.Button | Phaser.Graphics | Phaser.TileSprite, position?: number, offsetX?: number, offsetY?: number): any;

        /**
        * Aligns this Game Object to the side of another Game Object, or Rectangle, known as the
        * 'parent', in one of 11 possible positions.
        * 
        * The parent must be a Game Object, or Phaser.Rectangle object. This can include properties
        * such as `World.bounds` or `Camera.view`, for aligning Game Objects within the world
        * and camera bounds. Or it can include other Sprites, Images, Text objects, BitmapText,
        * TileSprites or Buttons.
        * 
        * Please note that aligning a Sprite to another Game Object does **not** make it a child of
        * the parent. It simply modifies its position coordinates so it aligns with it.
        * 
        * The position constants you can use are:
        * 
        * `Phaser.TOP_LEFT` (default), `Phaser.TOP_CENTER`, `Phaser.TOP_RIGHT`, `Phaser.LEFT_TOP`,
        * `Phaser.LEFT_CENTER`, `Phaser.LEFT_BOTTOM`, `Phaser.RIGHT_TOP`, `Phaser.RIGHT_CENTER`,
        * `Phaser.RIGHT_BOTTOM`, `Phaser.BOTTOM_LEFT`, `Phaser.BOTTOM_CENTER`
        * and `Phaser.BOTTOM_RIGHT`.
        * 
        * The Game Objects are placed in such a way that their _bounds_ align with the
        * parent, taking into consideration rotation, scale and the anchor property.
        * This allows you to neatly align Game Objects, irrespective of their position value.
        * 
        * The optional `offsetX` and `offsetY` arguments allow you to apply extra spacing to the final
        * aligned position of the Game Object. For example:
        * 
        * `sprite.alignTo(background, Phaser.BOTTOM_RIGHT, -20, -20)`
        * 
        * Would align the `sprite` to the bottom-right, but moved 20 pixels in from the corner.
        * Think of the offsets as applying an adjustment to the parents bounds before the alignment takes place.
        * So providing a negative offset will 'shrink' the parent bounds by that amount, and providing a positive
        * one expands it.
        * 
        * @param parent The Game Object or Rectangle with which to align this Game Object to. Can also include properties such as `World.bounds` or `Camera.view`.
        * @param position The position constant. One of `Phaser.TOP_LEFT`, `Phaser.TOP_CENTER`, `Phaser.TOP_RIGHT`, `Phaser.LEFT_TOP`, `Phaser.LEFT_CENTER`, `Phaser.LEFT_BOTTOM`, `Phaser.RIGHT_TOP`, `Phaser.RIGHT_CENTER`, `Phaser.RIGHT_BOTTOM`, `Phaser.BOTTOM_LEFT`, `Phaser.BOTTOM_CENTER` or `Phaser.BOTTOM_RIGHT`.
        * @param offsetX A horizontal adjustment of the Containers bounds, applied to the aligned position of the Game Object. Use a negative value to shrink the bounds, positive to increase it.
        * @param offsetY A vertical adjustment of the Containers bounds, applied to the aligned position of the Game Object. Use a negative value to shrink the bounds, positive to increase it.
        * @return This Game Object.
        */
        alignTo(container: Phaser.Rectangle | Phaser.Sprite | Phaser.Image | Phaser.Text | Phaser.BitmapText | Phaser.Button | Phaser.Graphics | Phaser.TileSprite, position?: number, offsetX?: number, offsetY?: number): any;

        /**
        * The arc method creates an arc/curve (used to create circles, or parts of circles).
        * 
        * @param cx The x-coordinate of the center of the circle
        * @param cy The y-coordinate of the center of the circle
        * @param radius The radius of the circle
        * @param startAngle The starting angle, in radians (0 is at the 3 o'clock position of the arc's circle)
        * @param endAngle The ending angle, in radians
        * @param anticlockwise Optional. Specifies whether the drawing should be counterclockwise or clockwise. False is default, and indicates clockwise, while true indicates counter-clockwise.
        * @param segments Optional. The number of segments to use when calculating the arc. The default is 40. If you need more fidelity use a higher number.
        */
        arc(cx: number, cy: number, radius: number, startAngle: number, endAngle: number, anticlockwise: boolean): Phaser.Graphics;

        /**
        * The arcTo() method creates an arc/curve between two tangents on the canvas.
        * 
        * "borrowed" from https://code.google.com/p/fxcanvas/ - thanks google!
        * 
        * @param x1 The x-coordinate of the beginning of the arc
        * @param y1 The y-coordinate of the beginning of the arc
        * @param x2 The x-coordinate of the end of the arc
        * @param y2 The y-coordinate of the end of the arc
        * @param radius The radius of the arc
        */
        arcTo(x1: number, y1: number, x2: number, y2: number, radius: number): Phaser.Graphics;

        /**
        * Specifies a simple one-color fill that subsequent calls to other Graphics methods
        * (such as lineTo() or drawCircle()) use when drawing.
        * 
        * @param color the color of the fill
        * @param alpha the alpha of the fill
        */
        beginFill(color?: number, alpha?: number): Phaser.Graphics;

        /**
        * Calculate the points for a bezier curve and then draws it.
        * 
        * @param cpX Control point x
        * @param cpY Control point y
        * @param cpX2 Second Control point x
        * @param cpY2 Second Control point y
        * @param toX Destination point x
        * @param toY Destination point y
        */
        bezierCurveTo(cpX: number, cpY: number, cpX2: number, cpY2: number, toX: number, toY: number): Phaser.Graphics;

        /**
        * Clears the graphics that were drawn to this Graphics object, and resets fill and line style settings.
        */
        clear(): Phaser.Graphics;

        /**
        * Destroy this Graphics instance.
        * 
        * @param destroyChildren Should every child of this object have its destroy method called? - Default: true
        */
        destroy(destroyChildren?: boolean): void;

        /**
        * Destroys a previous cached sprite.
        */
        destroyCachedSprite(): void;

        /**
        * Draws a circle.
        * 
        * @param x The X coordinate of the center of the circle
        * @param y The Y coordinate of the center of the circle
        * @param diameter The diameter of the circle
        */
        drawCircle(x: number, y: number, diameter: number): Phaser.Graphics;

        /**
        * Draws an ellipse.
        * 
        * @param x The X coordinate of the center of the ellipse
        * @param y The Y coordinate of the center of the ellipse
        * @param width The half width of the ellipse
        * @param height The half height of the ellipse
        */
        drawEllipse(x: number, y: number, width: number, height: number): Phaser.Graphics;

        /**
        * Draws a polygon using the given path.
        * 
        * @param path The path data used to construct the polygon. Can either be an array of points or a Phaser.Polygon object.
        */
        drawPolygon(...path: any[]): Phaser.Graphics;

        /**
        * 
        * 
        * @param x The X coord of the top-left of the rectangle
        * @param y The Y coord of the top-left of the rectangle
        * @param width The width of the rectangle
        * @param height The height of the rectangle
        */
        drawRect(x: number, y: number, width: number, height: number): Phaser.Graphics;

        /**
        * 
        * 
        * @param x The X coord of the top-left of the rectangle
        * @param y The Y coord of the top-left of the rectangle
        * @param width The width of the rectangle
        * @param height The height of the rectangle
        * @param radius Radius of the rectangle corners. In WebGL this must be a value between 0 and 9.
        */
        drawRoundedRect(x: number, y: number, width: number, height: number, radius: number): Phaser.Graphics;

        /**
        * Draws the given shape to this Graphics object. Can be any of Circle, Rectangle, Ellipse, Line or Polygon.
        * 
        * @param shape The Shape object to draw.
        * @return The generated GraphicsData object.
        */
        drawShape(shape: Circle): Phaser.GraphicsData;

        /**
        * Draws the given shape to this Graphics object. Can be any of Circle, Rectangle, Ellipse, Line or Polygon.
        * 
        * @param shape The Shape object to draw.
        * @return The generated GraphicsData object.
        */
        drawShape(shape: Ellipse): Phaser.GraphicsData;

        /**
        * Draws the given shape to this Graphics object. Can be any of Circle, Rectangle, Ellipse, Line or Polygon.
        * 
        * @param shape The Shape object to draw.
        * @return The generated GraphicsData object.
        */
        drawShape(shape: Polygon): Phaser.GraphicsData;

        /**
        * Draws the given shape to this Graphics object. Can be any of Circle, Rectangle, Ellipse, Line or Polygon.
        * 
        * @param shape The Shape object to draw.
        * @return The generated GraphicsData object.
        */
        drawShape(shape: Rectangle): Phaser.GraphicsData;

        /**
        * Draws a single {Phaser.Polygon} triangle from a {Phaser.Point} array
        * 
        * @param points An array of Phaser.Points that make up the three vertices of this triangle
        * @param cull Should we check if the triangle is back-facing
        */
        drawTriangle(points: Phaser.Point[], cull?: boolean): void;

        /**
        * Draws {Phaser.Polygon} triangles
        * 
        * @param vertices An array of Phaser.Points or numbers that make up the vertices of the triangles
        * @param {indices An array of numbers that describe what order to draw the vertices in - Default: null}
        * @param cull Should we check if the triangle is back-facing
        */
        drawTriangles(vertices: Phaser.Point[] | number[], indices?: number[], cull?: boolean): void;

        /**
        * Applies a fill to the lines and shapes that were added since the last call to the beginFill() method.
        */
        endFill(): Phaser.Graphics;

        /**
        * Useful function that returns a texture of the graphics object that can then be used to create sprites
        * This can be quite useful if your geometry is complicated and needs to be reused multiple times.
        * 
        * @param resolution The resolution of the texture being generated - Default: 1
        * @param scaleMode Should be one of the PIXI.scaleMode consts
        * @param padding Add optional extra padding to the generated texture (default 0)
        * @return a texture of the graphics object
        */
        generateTexture(resolution?: number, scaleMode?: number, padding?: number): Phaser.RenderTexture;

        /**
        * Kills a Game Object. A killed Game Object has its `alive`, `exists` and `visible` properties all set to false.
        * 
        * It will dispatch the `onKilled` event. You can listen to `events.onKilled` for the signal.
        * 
        * Note that killing a Game Object is a way for you to quickly recycle it in an object pool,
        * it doesn't destroy the object or free it up from memory.
        * 
        * If you don't need this Game Object any more you should call `destroy` instead.
        * @return This instance.
        */
        kill(): Phaser.Graphics;

        /**
        * Specifies the line style used for subsequent calls to Graphics methods such as the lineTo() method or the drawCircle() method.
        * 
        * @param lineWidth width of the line to draw, will update the objects stored style
        * @param color color of the line to draw, will update the objects stored style
        * @param alpha alpha of the line to draw, will update the objects stored style
        */
        lineStyle(lineWidth?: number, color?: number, alpha?: number): Phaser.Graphics;

        /**
        * Draws a line using the current line style from the current drawing position to (x, y);
        * The current drawing position is then set to (x, y).
        * 
        * @param x the X coordinate to draw to
        * @param y the Y coordinate to draw to
        */
        lineTo(x: number, y: number): Phaser.Graphics;

        /**
        * Moves the current drawing position to x, y.
        * 
        * @param x the X coordinate to move to
        * @param y the Y coordinate to move to
        */
        moveTo(x: number, y: number): Phaser.Graphics;

        /**
        * Automatically called by World
        */
        postUpdate(): void;

        /**
        * Automatically called by World.preUpdate.
        */
        preUpdate(): void;

        /**
        * Calculate the points for a quadratic bezier curve and then draws it.
        * Based on: https://stackoverflow.com/questions/785097/how-do-i-implement-a-bezier-curve-in-c
        * 
        * @param cpX Control point x
        * @param cpY Control point y
        * @param toX Destination point x
        * @param toY Destination point y
        */
        quadraticCurveTo(cpX: number, cpY: number, toX: number, toY: number): Phaser.Graphics;

        /**
        * Resets the Game Object.
        * 
        * This moves the Game Object to the given x/y world coordinates and sets `fresh`, `exists`,
        * `visible` and `renderable` to true.
        * 
        * If this Game Object has the LifeSpan component it will also set `alive` to true and `health` to the given value.
        * 
        * If this Game Object has a Physics Body it will reset the Body.
        * 
        * @param x The x coordinate (in world space) to position the Game Object at.
        * @param y The y coordinate (in world space) to position the Game Object at.
        * @param health The health to give the Game Object if it has the Health component. - Default: 1
        * @return This instance.
        */
        reset(x: number, y: number, health?: number): Phaser.Graphics;

        /**
        * Brings a 'dead' Game Object back to life, optionally resetting its health value in the process.
        * 
        * A resurrected Game Object has its `alive`, `exists` and `visible` properties all set to true.
        * 
        * It will dispatch the `onRevived` event. Listen to `events.onRevived` for the signal.
        * 
        * @param health The health to give the Game Object. Only set if the GameObject has the Health component. - Default: 100
        * @return This instance.
        */
        revive(health?: number): Phaser.Graphics;

        /**
        * Override this method in your own custom objects to handle any update requirements.
        * It is called immediately after `preUpdate` and before `postUpdate`.
        * Remember if this Game Object has any children you should call update on those too.
        */
        update(): void;

    }

    class GraphicsData {

        constructor(lineWidth?: number, lineColor?: number, lineAlpha?: number, fillColor?: number, fillAlpha?: number, fill?: boolean, shape?: any);

        lineWidth: number;
        lineColor: number;
        lineAlpha: number;
        fillColor: number;
        fillAlpha: number;
        fill: boolean;
        shape: any;
        type: number;

    }


    /**
    * A Group is a container for {@link DisplayObject display objects} including {@link Phaser.Sprite Sprites} and {@link Phaser.Image Images}.
    * 
    * Groups form the logical tree structure of the display/scene graph where local transformations are applied to children.
    * For instance, all children are also moved/rotated/scaled when the group is moved/rotated/scaled.
    * 
    * In addition, Groups provides support for fast pooling and object recycling.
    * 
    * Groups are also display objects and can be nested as children within other Groups.
    */
    class Group extends PIXI.DisplayObjectContainer {


        /**
        * A Group is a container for {@link DisplayObject display objects} including {@link Phaser.Sprite Sprites} and {@link Phaser.Image Images}.
        * 
        * Groups form the logical tree structure of the display/scene graph where local transformations are applied to children.
        * For instance, all children are also moved/rotated/scaled when the group is moved/rotated/scaled.
        * 
        * In addition, Groups provides support for fast pooling and object recycling.
        * 
        * Groups are also display objects and can be nested as children within other Groups.
        * 
        * @param game A reference to the currently running game.
        * @param parent The parent Group (or other {@link DisplayObject}) that this group will be added to.
        *               If undefined/unspecified the Group will be added to the {@link Phaser.Game#world Game World}; if null the Group will not be added to any parent. - Default: (game world)
        * @param name A name for this group. Not used internally but useful for debugging. - Default: 'group'
        * @param addToStage If true this group will be added directly to the Game.Stage instead of Game.World.
        * @param enableBody If true all Sprites created with {@link #create} or {@link #createMulitple} will have a physics body created on them. Change the body type with {@link #physicsBodyType}.
        * @param physicsBodyType The physics body type to use when physics bodies are automatically added. See {@link #physicsBodyType} for values.
        */
        constructor(game: Phaser.Game, parent?: PIXI.DisplayObjectContainer, name?: string, addToStage?: boolean, enableBody?: boolean, physicsBodyType?: number);


        /**
        * A returnType value, as specified in {@link Phaser.Group#iterate iterate} eg.
        */
        static RETURN_CHILD: number;

        /**
        * A returnType value, as specified in {@link Phaser.Group#iterate iterate} eg.
        */
        static RETURN_NONE: number;

        /**
        * A returnType value, as specified in {@link Phaser.Group#iterate iterate} eg.
        */
        static RETURN_TOTAL: number;

        /**
        * A returnType value, as specified in {@link Phaser.Group#iterate iterate} eg.
        */
        static RETURN_ALL: number;

        /**
        * A sort ordering value, as specified in {@link Phaser.Group#sort sort} eg.
        */
        static SORT_ASCENDING: number;

        /**
        * A sort ordering value, as specified in {@link Phaser.Group#sort sort} eg.
        */
        static SORT_DESCENDING: number;

        alpha: number;

        /**
        * The angle of rotation of the group container, in degrees.
        * 
        * This adjusts the group itself by modifying its local rotation transform.
        * 
        * This has no impact on the rotation/angle properties of the children, but it will update their worldTransform
        * and on-screen orientation and position.
        */
        angle: number;

        /**
        * The alive property is useful for Groups that are children of other Groups and need to be included/excluded in checks like forEachAlive.
        * Default: true
        */
        alive: boolean;

        /**
        * The bottom coordinate of this Group.
        * 
        * It is derived by calling `getBounds`, calculating the Groups dimensions based on its
        * visible children.
        */
        bottom: number;

        /**
        * If this object is {@link Phaser.Group#fixedToCamera fixedToCamera} then this stores the x/y position offset relative to the top-left of the camera view.
        * If the parent of this Group is also `fixedToCamera` then the offset here is in addition to that and should typically be disabled.
        */
        cameraOffset: Phaser.Point;

        /**
        * The center x coordinate of this Group.
        * 
        * It is derived by calling `getBounds`, calculating the Groups dimensions based on its
        * visible children.
        */
        centerX: number;

        /**
        * The center y coordinate of this Group.
        * 
        * It is derived by calling `getBounds`, calculating the Groups dimensions based on its
        * visible children.
        */
        centerY: number;

        /**
        * The type of objects that will be created when using {@link Phaser.Group#create create} or {@link Phaser.Group#createMultiple createMultiple}.
        * 
        * It should extend either Sprite or Image and accept the same constructor arguments: `(game, x, y, key, frame)`.
        * Default: {@link Phaser.Sprite}
        */
        classType: any;

        /**
        * The current display object that the group cursor is pointing to, if any. (Can be set manually.)
        * 
        * The cursor is a way to iterate through the children in a Group using {@link Phaser.Group#next next} and {@link Phaser.Group#previous previous}.
        */
        cursor: any;

        /**
        * The current index of the Group cursor. Advance it with Group.next.
        */
        cursorIndex: number;

        /**
        * If true all Sprites created by, or added to this group, will have a physics body enabled on them.
        * 
        * If there are children already in the Group at the time you set this property, they are not changed.
        * 
        * The default body type is controlled with {@link Phaser.Group#physicsBodyType physicsBodyType}.
        */
        enableBody: boolean;

        /**
        * If true when a physics body is created (via {@link Phaser.Group#enableBody enableBody}) it will create a physics debug object as well.
        * 
        * This only works for P2 bodies.
        */
        enableBodyDebug: boolean;

        /**
        * If exists is false the group will be excluded from collision checks and filters such as {@link Phaser.Group#forEachExists forEachExists}. The group will not call `preUpdate` and `postUpdate` on its children and the children will not receive physics updates or camera/world boundary checks. The group will still be {@link Phaser.Group#visible visible} and will still call `update` on its children.
        * Default: true
        */
        exists: boolean;

        /**
        * A Group that is fixed to the camera uses its x/y coordinates as offsets from the top left of the camera. These are stored in Group.cameraOffset.
        * 
        * Note that the cameraOffset values are in addition to any parent in the display list.
        * So if this Group was in a Group that has x: 200, then this will be added to the cameraOffset.x
        */
        fixedToCamera: boolean;

        /**
        * A reference to the currently running Game.
        */
        game: Phaser.Game;

        /**
        * The hash array is an array belonging to this Group into which you can add any of its children via Group.addToHash and Group.removeFromHash.
        * 
        * Only children of this Group can be added to and removed from the hash.
        * 
        * This hash is used automatically by Phaser Arcade Physics in order to perform non z-index based destructive sorting.
        * However if you don't use Arcade Physics, or this isn't a physics enabled Group, then you can use the hash to perform your own
        * sorting and filtering of Group children without touching their z-index (and therefore display draw order)
        */
        hash: PIXI.DisplayObject[];

        /**
        * A group with `ignoreDestroy` set to `true` ignores all calls to its `destroy` method.
        */
        ignoreDestroy: boolean;

        /**
        * A Group with `inputEnableChildren` set to `true` will automatically call `inputEnabled = true`
        * on any children _added_ to, or _created by_, this Group.
        * 
        * If there are children already in the Group at the time you set this property, they are not changed.
        */
        inputEnableChildren: boolean;

        /**
        * The left coordinate of this Group.
        * 
        * It is derived by calling `getBounds`, calculating the Groups dimensions based on its
        * visible children.
        */
        left: number;

        /**
        * Total number of children in this group, regardless of exists/alive status.
        */
        length: number;

        /**
        * A name for this group. Not used internally but useful for debugging.
        */
        name: string;

        /**
        * This Signal is dispatched whenever a child of this Group emits an onInputDown signal as a result
        * of having been interacted with by a Pointer. You can bind functions to this Signal instead of to
        * every child Sprite.
        * 
        * This Signal is sent 2 arguments: A reference to the Sprite that triggered the signal, and
        * a reference to the Pointer that caused it.
        */
        onChildInputDown: Phaser.Signal;

        /**
        * This Signal is dispatched whenever a child of this Group emits an onInputUp signal as a result
        * of having been interacted with by a Pointer. You can bind functions to this Signal instead of to
        * every child Sprite.
        * 
        * This Signal is sent 3 arguments: A reference to the Sprite that triggered the signal,
        * a reference to the Pointer that caused it, and a boolean value `isOver` that tells you if the Pointer
        * is still over the Sprite or not.
        */
        onChildInputUp: Phaser.Signal;

        /**
        * This Signal is dispatched whenever a child of this Group emits an onInputOver signal as a result
        * of having been interacted with by a Pointer. You can bind functions to this Signal instead of to
        * every child Sprite.
        * 
        * This Signal is sent 2 arguments: A reference to the Sprite that triggered the signal, and
        * a reference to the Pointer that caused it.
        */
        onChildInputOver: Phaser.Signal;

        /**
        * This Signal is dispatched whenever a child of this Group emits an onInputOut signal as a result
        * of having been interacted with by a Pointer. You can bind functions to this Signal instead of to
        * every child Sprite.
        * 
        * This Signal is sent 2 arguments: A reference to the Sprite that triggered the signal, and
        * a reference to the Pointer that caused it.
        */
        onChildInputOut: Phaser.Signal;

        /**
        * This signal is dispatched when the group is destroyed.
        */
        onDestroy: Phaser.Signal;

        /**
        * A Group is that has `pendingDestroy` set to `true` is flagged to have its destroy method
        * called on the next logic update.
        * You can set it directly to flag the Group to be destroyed on its next update.
        * 
        * This is extremely useful if you wish to destroy a Group from within one of its own callbacks
        * or a callback of one of its children.
        */
        pendingDestroy: boolean;

        /**
        * If {@link Phaser.Group#enableBody enableBody} is true this is the type of physics body that is created on new Sprites.
        * 
        * The valid values are {@link Phaser.Physics.ARCADE}, {@link Phaser.Physics.P2JS}, {@link Phaser.Physics.NINJA}, etc.
        */
        physicsBodyType: number;

        /**
        * The const physics body type of this object.
        */
        physicsType: number;

        /**
        * If this Group contains Arcade Physics Sprites you can set a custom sort direction via this property.
        * 
        * It should be set to one of the Phaser.Physics.Arcade sort direction constants:
        * 
        * Phaser.Physics.Arcade.SORT_NONE
        * Phaser.Physics.Arcade.LEFT_RIGHT
        * Phaser.Physics.Arcade.RIGHT_LEFT
        * Phaser.Physics.Arcade.TOP_BOTTOM
        * Phaser.Physics.Arcade.BOTTOM_TOP
        * 
        * If set to `null` the Group will use whatever Phaser.Physics.Arcade.sortDirection is set to. This is the default behavior.
        */
        physicsSortDirection: number;

        /**
        * The coordinates, in pixels, of this DisplayObject, relative to its parent container.
        * 
        * The value of this property does not reflect any positioning happening further up the display list.
        * To obtain that value please see the `worldPosition` property.
        */
        position: Phaser.Point;

        /**
        * The right coordinate of this Group.
        * 
        * It is derived by calling `getBounds`, calculating the Groups dimensions based on its
        * visible children.
        */
        right: number;
        rotation: number;

        /**
        * The scale of this DisplayObject. A scale of 1:1 represents the DisplayObject
        * at its default size. A value of 0.5 would scale this DisplayObject by half, and so on.
        * 
        * The value of this property does not reflect any scaling happening further up the display list.
        * To obtain that value please see the `worldScale` property.
        */
        scale: Phaser.Point;

        /**
        * The top coordinate of this Group.
        * 
        * It is derived by calling `getBounds`, calculating the Groups dimensions based on its
        * visible children.
        */
        top: number;

        /**
        * Total number of existing children in the group.
        */
        total: number;

        /**
        * Internal Phaser Type value.
        */
        type: number;

        /**
        * Skip children with `exists = false` in {@link Phaser.Group#update update}.
        */
        updateOnlyExistingChildren: boolean;
        visible: boolean;

        /**
        * The z-depth value of this object within its parent container/Group - the World is a Group as well.
        * This value must be unique for each child in a Group.
        */
        z: number;


        /**
        * Adds an existing object as the top child in this group.
        * 
        * The child is automatically added to the top of the group, and is displayed above every previous child.
        * 
        * Or if the _optional_ index is specified, the child is added at the location specified by the index value,
        * this allows you to control child ordering.
        * 
        * If the child was already in this Group, it is simply returned, and nothing else happens to it.
        * 
        * If `Group.enableBody` is set, then a physics body will be created on the object, so long as one does not already exist.
        * 
        * If `Group.inputEnableChildren` is set, then an Input Handler will be created on the object, so long as one does not already exist.
        * 
        * Use {@link Phaser.Group#addAt addAt} to control where a child is added. Use {@link Phaser.Group#create create} to create and add a new child.
        * 
        * @param child The display object to add as a child.
        * @param silent If true the child will not dispatch the `onAddedToGroup` event.
        * @param index The index within the group to insert the child to. Where 0 is the bottom of the Group.
        * @return The child that was added to the group.
        */
        add(child: any, silent?: boolean, index?: number): any;

        /**
        * Adds the amount to the given property on all children in this group.
        * 
        * `Group.addAll('x', 10)` will add 10 to the child.x value for each child.
        * 
        * @param property The property to increment, for example 'body.velocity.x' or 'angle'.
        * @param amount The amount to increment the property by. If child.x = 10 then addAll('x', 40) would make child.x = 50.
        * @param checkAlive If true the property will only be changed if the child is alive.
        * @param checkVisible If true the property will only be changed if the child is visible.
        */
        addAll(property: string, amount: number, checkAlive?: boolean, checkVisible?: boolean): void;

        /**
        * Adds an existing object to this group.
        * 
        * The child is added to the group at the location specified by the index value, this allows you to control child ordering.
        * 
        * If `Group.enableBody` is set, then a physics body will be created on the object, so long as one does not already exist.
        * 
        * If `Group.inputEnableChildren` is set, then an Input Handler will be created on the object, so long as one does not already exist.
        * 
        * @param child The display object to add as a child.
        * @param index The index within the group to insert the child to.
        * @param silent If true the child will not dispatch the `onAddedToGroup` event.
        * @return The child that was added to the group.
        */
        addAt(child: any, index: number, silent?: boolean): any;

        /**
        * Adds an array of existing Display Objects to this Group.
        * 
        * The Display Objects are automatically added to the top of this Group, and will render on-top of everything already in this Group.
        * 
        * As well as an array you can also pass another Group as the first argument. In this case all of the children from that
        * Group will be removed from it and added into this Group.
        * 
        * If `Group.enableBody` is set, then a physics body will be created on the objects, so long as one does not already exist.
        * 
        * If `Group.inputEnableChildren` is set, then an Input Handler will be created on the objects, so long as one does not already exist.
        * 
        * @param children An array of display objects or a Phaser.Group. If a Group is given then *all* children will be moved from it.
        * @param silent If true the children will not dispatch the `onAddedToGroup` event.
        * @return The array of children or Group of children that were added to this Group.
        */
        addMultiple(children: any[], silent?: boolean): any[];

        /**
        * Adds a child of this Group into the hash array.
        * This call will return false if the child is not a child of this Group, or is already in the hash.
        * 
        * @param child The display object to add to this Groups hash. Must be a member of this Group already and not present in the hash.
        * @return True if the child was successfully added to the hash, otherwise false.
        */
        addToHash(child: PIXI.DisplayObject): boolean;

        /**
        * This method iterates through all children in the Group (regardless if they are visible or exist)
        * and then changes their position so they are arranged in a Grid formation. Children must have
        * the `alignTo` method in order to be positioned by this call. All default Phaser Game Objects have
        * this.
        * 
        * The grid dimensions are determined by the first four arguments. The `width` and `height` arguments
        * relate to the width and height of the grid respectively.
        * 
        * For example if the Group had 100 children in it:
        * 
        * `Group.align(10, 10, 32, 32)`
        * 
        * This will align all of the children into a grid formation of 10x10, using 32 pixels per
        * grid cell. If you want a wider grid, you could do:
        * 
        * `Group.align(25, 4, 32, 32)`
        * 
        * This will align the children into a grid of 25x4, again using 32 pixels per grid cell.
        * 
        * You can choose to set _either_ the `width` or `height` value to -1. Doing so tells the method
        * to keep on aligning children until there are no children left. For example if this Group had
        * 48 children in it, the following:
        * 
        * `Group.align(-1, 8, 32, 32)`
        * 
        * ... will align the children so that there are 8 children vertically (the second argument),
        * and each row will contain 6 sprites, except the last one, which will contain 5 (totaling 48)
        * 
        * You can also do:
        * 
        * `Group.align(10, -1, 32, 32)`
        * 
        * In this case it will create a grid 10 wide, and as tall as it needs to be in order to fit
        * all of the children in.
        * 
        * The `position` property allows you to control where in each grid cell the child is positioned.
        * This is a constant and can be one of `Phaser.TOP_LEFT` (default), `Phaser.TOP_CENTER`,
        * `Phaser.TOP_RIGHT`, `Phaser.LEFT_CENTER`, `Phaser.CENTER`, `Phaser.RIGHT_CENTER`,
        * `Phaser.BOTTOM_LEFT`, `Phaser.BOTTOM_CENTER` or `Phaser.BOTTOM_RIGHT`.
        * 
        * The final argument; `offset` lets you start the alignment from a specific child index.
        * 
        * @param width The width of the grid in items (not pixels). Set to -1 for a dynamic width. If -1 then you must set an explicit height value.
        * @param height The height of the grid in items (not pixels). Set to -1 for a dynamic height. If -1 then you must set an explicit width value.
        * @param cellWidth The width of each grid cell, in pixels.
        * @param cellHeight The height of each grid cell, in pixels.
        * @param position The position constant. One of `Phaser.TOP_LEFT` (default), `Phaser.TOP_CENTER`, `Phaser.TOP_RIGHT`, `Phaser.LEFT_CENTER`, `Phaser.CENTER`, `Phaser.RIGHT_CENTER`, `Phaser.BOTTOM_LEFT`, `Phaser.BOTTOM_CENTER` or `Phaser.BOTTOM_RIGHT`.
        * @param offset Optional index to start the alignment from. Defaults to zero, the first child in the Group, but can be set to any valid child index value.
        * @return True if the Group children were aligned, otherwise false.
        */
        align(width: number, height: number, cellWidth: number, cellHeight: number, position?: number, offset?: number): boolean;
        alignIn(container: Phaser.Rectangle | Phaser.Sprite | Phaser.Image | Phaser.Text | Phaser.BitmapText | Phaser.Button | Phaser.Graphics | Phaser.TileSprite, position?: number, offsetX?: number, offsetY?: number): Phaser.Group;
        alignTo(container: Phaser.Rectangle | Phaser.Sprite | Phaser.Image | Phaser.Text | Phaser.BitmapText | Phaser.Button | Phaser.Graphics | Phaser.TileSprite, position?: number, offsetX?: number, offsetY?: number): Phaser.Group;

        /**
        * Brings the given child to the top of this group so it renders above all other children.
        * 
        * @param child The child to bring to the top of this group.
        * @return The child that was moved.
        */
        bringToTop(child: any): any;

        /**
        * Calls a function, specified by name, on all on children.
        * 
        * The function is called for all children regardless if they are dead or alive (see callAllExists for different options).
        * After the method parameter and context you can add as many extra parameters as you like, which will all be passed to the child.
        * 
        * @param method Name of the function on the child to call. Deep property lookup is supported.
        * @param context A string containing the context under which the method will be executed. Set to null to default to the child.
        * @param args Additional parameters that will be passed to the method.
        */
        callAll(method: string, context: any, ...parameters: any[]): void;

        /**
        * Calls a function, specified by name, on all children in the group who exist (or do not exist).
        * 
        * After the existsValue parameter you can add as many parameters as you like, which will all be passed to the child callback.
        * 
        * @param callback Name of the function on the children to call.
        * @param existsValue Only children with exists=existsValue will be called.
        * @param parameter Additional parameters that will be passed to the callback.
        */
        callAllExists(callback: string, existsValue: boolean, ...parameters: any[]): void;

        /**
        * Returns a reference to a function that exists on a child of the group based on the given callback array.
        * 
        * @param child The object to inspect.
        * @param callback The array of function names.
        * @param length The size of the array (pre-calculated in callAll).
        */
        callbackFromArray(child: any, callback: Function, length: number): void;

        /**
        * Test that the same property across all children of this group is equal to the given value.
        * 
        * This call doesn't descend down children, so if you have a Group inside of this group, the property will be checked on the group but not its children.
        * 
        * @param key The property, as a string, to be checked. For example: 'body.velocity.x'
        * @param value The value that will be checked.
        * @param checkAlive If set then only children with alive=true will be checked. This includes any Groups that are children.
        * @param checkVisible If set then only children with visible=true will be checked. This includes any Groups that are children.
        * @param force Also return false if the property is missing or undefined (regardless of the `value` argument).
        * @return - True if all eligible children have the given property value (but see `force`); otherwise false.
        */
        checkAll(key: string, value: any, checkAlive?: boolean, checkVisible?: boolean, force?: boolean): boolean;

        /**
        * Test that at least one child of this group has the given property value.
        * 
        * This call doesn't descend down children, so if you have a Group inside of this group, the property will be checked on the group but not its children.
        * 
        * @param key The property, as a string, to be checked. For example: 'body.velocity.x'
        * @param value The value that will be checked.
        * @param checkAlive If set then only children with alive=true will be checked. This includes any Groups that are children.
        * @param checkVisible If set then only children with visible=true will be checked. This includes any Groups that are children.
        * @return - True if at least one eligible child has the given property value; otherwise false.
        */
        checkAny(key: string, value: any, checkAlive?: boolean, checkVisible?: boolean): boolean;

        /**
        * Checks a property for the given value on the child.
        * 
        * @param child The child to check the property value on.
        * @param key The property, as a string, to be checked. For example: 'body.velocity.x'
        * @param value The value that will be checked.
        * @param force Also return false if the property is missing or undefined (regardless of the `value` argument).
        * @return True if `child` is a child of this Group and the property was equal to value, false if not.
        */
        checkProperty(child: any, key: string, value: any, force?: boolean): boolean;

        /**
        * Get the number of dead children in this group.
        * @return The number of children flagged as dead.
        */
        countDead(): number;

        /**
        * Get the number of living children in this group.
        * @return The number of children flagged as alive.
        */
        countLiving(): number;

        /**
        * Creates a new Phaser.Sprite object and adds it to the top of this group.
        * 
        * Use {@link Phaser.Group#classType classType} to change the type of object created.
        * 
        * The child is automatically added to the top of the group, and is displayed above every previous child.
        * 
        * Or if the _optional_ index is specified, the child is added at the location specified by the index value,
        * this allows you to control child ordering.
        * 
        * If `Group.enableBody` is set, then a physics body will be created on the object, so long as one does not already exist.
        * 
        * If `Group.inputEnableChildren` is set, then an Input Handler will be created on the object, so long as one does not already exist.
        * 
        * @param x The x coordinate to display the newly created Sprite at. The value is in relation to the group.x point.
        * @param y The y coordinate to display the newly created Sprite at. The value is in relation to the group.y point.
        * @param key This is the image or texture used by the Sprite during rendering. It can be a string which is a reference to the Cache Image entry, or an instance of a RenderTexture, BitmapData, Video or PIXI.Texture.
        * @param frame If this Sprite is using part of a sprite sheet or texture atlas you can specify the exact frame to use by giving a string or numeric index.
        * @param exists The default exists state of the Sprite. - Default: true
        * @param index The index within the group to insert the child to. Where 0 is the bottom of the Group.
        * @return The child that was created: will be a {@link Phaser.Sprite} unless {@link #classType} has been changed.
        */
        create(x: number, y: number, key?: string | Phaser.RenderTexture | Phaser.BitmapData | Phaser.Video | PIXI.Texture, frame?: string | number, exists?: boolean, index?: number): any;

        /**
        * Creates multiple Phaser.Sprite objects and adds them to the top of this Group.
        * 
        * This method is useful if you need to quickly generate a pool of sprites, such as bullets.
        * 
        * Use {@link Phaser.Group#classType classType} to change the type of object created.
        * 
        * You can provide an array as the `key` and / or `frame` arguments. When you do this
        * it will create `quantity` Sprites for every key (and frame) in the arrays.
        * 
        * For example:
        * 
        * `createMultiple(25, ['ball', 'carrot'])`
        * 
        * In the above code there are 2 keys (ball and carrot) which means that 50 sprites will be
        * created in total, 25 of each. You can also have the `frame` as an array:
        * 
        * `createMultiple(5, 'bricks', [0, 1, 2, 3])`
        * 
        * In the above there is one key (bricks), which is a sprite sheet. The frames array tells
        * this method to use frames 0, 1, 2 and 3. So in total it will create 20 sprites, because
        * the quantity was set to 5, so that is 5 brick sprites of frame 0, 5 brick sprites with
        * frame 1, and so on.
        * 
        * If you set both the key and frame arguments to be arrays then understand it will create
        * a total quantity of sprites equal to the size of both arrays times each other. I.e.:
        * 
        * `createMultiple(20, ['diamonds', 'balls'], [0, 1, 2])`
        * 
        * The above will create 20 'diamonds' of frame 0, 20 with frame 1 and 20 with frame 2.
        * It will then create 20 'balls' of frame 0, 20 with frame 1 and 20 with frame 2.
        * In total it will have created 120 sprites.
        * 
        * By default the Sprites will have their `exists` property set to `false`, and they will be
        * positioned at 0x0, relative to the `Group.x / y` values.
        * 
        * If `Group.enableBody` is set, then a physics body will be created on the objects, so long as one does not already exist.
        * 
        * If `Group.inputEnableChildren` is set, then an Input Handler will be created on the objects, so long as one does not already exist.
        * 
        * @param quantity The number of Sprites to create.
        * @param key The Cache key of the image that the Sprites will use. Or an Array of keys. See the description for details on how the quantity applies when arrays are used.
        * @param frame If the Sprite image contains multiple frames you can specify which one to use here. Or an Array of frames. See the description for details on how the quantity applies when arrays are used.
        * @param exists The default exists state of the Sprite.
        * @return An array containing all of the Sprites that were created.
        */
        createMultiple(quantity: number, key: string | string[], frame?: any | any[], exists?: boolean): any[];

        /**
        * Sort the children in the group according to custom sort function.
        * 
        * The `sortHandler` is provided the two parameters: the two children involved in the comparison (a and b).
        * It should return -1 if `a > b`, 1 if `a < b` or 0 if `a === b`.
        * 
        * @param sortHandler The custom sort function.
        * @param context The context in which the sortHandler is called.
        */
        customSort(sortHandler: Function, context?: any): void;

        /**
        * Destroys this group.
        * 
        * Removes all children, then removes this group from its parent and nulls references.
        * 
        * @param destroyChildren If true `destroy` will be invoked on each removed child. - Default: true
        * @param soft A 'soft destroy' (set to true) doesn't remove this group from its parent or null the game reference. Set to false and it does.
        */
        destroy(destroyChildren?: boolean, soft?: boolean): void;

        /**
        * Divides the given property by the amount on all children in this group.
        * 
        * `Group.divideAll('x', 2)` will half the child.x value for each child.
        * 
        * @param property The property to divide, for example 'body.velocity.x' or 'angle'.
        * @param amount The amount to divide the property by. If child.x = 100 then divideAll('x', 2) would make child.x = 50.
        * @param checkAlive If true the property will only be changed if the child is alive.
        * @param checkVisible If true the property will only be changed if the child is visible.
        */
        divideAll(property: string, amount: number, checkAlive?: boolean, checkVisible?: boolean): void;

        /**
        * Call a function on each child in this group.
        * 
        * Additional arguments for the callback can be specified after the `checkExists` parameter. For example,
        * 
        *     Group.forEach(awardBonusGold, this, true, 100, 500)
        * 
        * would invoke `awardBonusGold` function with the parameters `(child, 100, 500)`.
        * 
        * Note: This check will skip any children which are Groups themselves.
        * 
        * @param callback The function that will be called for each applicable child. The child will be passed as the first argument.
        * @param callbackContext The context in which the function should be called (usually 'this').
        * @param checkExists If set only children matching for which `exists` is true will be passed to the callback, otherwise all children will be passed.
        * @param args Additional arguments to pass to the callback function, after the child item. - Default: (none)
        */
        forEach(callback: Function, callbackContext: any, checkExists?: boolean, ...args: any[]): void;

        /**
        * Call a function on each alive child in this group.
        * 
        * See {@link Phaser.Group#forEach forEach} for details.
        * 
        * @param callback The function that will be called for each applicable child. The child will be passed as the first argument.
        * @param callbackContext The context in which the function should be called (usually 'this').
        * @param args Additional arguments to pass to the callback function, after the child item. - Default: (none)
        */
        forEachAlive(callback: Function, callbackContext?: any, ...args: any[]): void;

        /**
        * Call a function on each dead child in this group.
        * 
        * See {@link Phaser.Group#forEach forEach} for details.
        * 
        * @param callback The function that will be called for each applicable child. The child will be passed as the first argument.
        * @param callbackContext The context in which the function should be called (usually 'this').
        * @param args Additional arguments to pass to the callback function, after the child item. - Default: (none)
        */
        forEachDead(callback: Function, callbackContext?: any, ...args: any[]): void;

        /**
        * Call a function on each existing child in this group.
        * 
        * See {@link Phaser.Group#forEach forEach} for details.
        * 
        * @param callback The function that will be called for each applicable child. The child will be passed as the first argument.
        * @param callbackContext The context in which the function should be called (usually 'this').
        * @param args Additional arguments to pass to the callback function, after the child item. - Default: (none)
        */
        forEachExists(callback: Function, callbackContext?: any): void;

        /**
        * Find children matching a certain predicate.
        * 
        * For example:
        * 
        *     var healthyList = Group.filter(function(child, index, children) {
        *         return child.health > 10 ? true : false;
        *     }, true);
        *     healthyList.callAll('attack');
        * 
        * Note: Currently this will skip any children which are Groups themselves.
        * 
        * @param predicate The function that each child will be evaluated against. Each child of the group will be passed to it as its first parameter, the index as the second, and the entire child array as the third
        * @param checkExists If true, only existing can be selected; otherwise all children can be selected and will be passed to the predicate.
        * @return Returns an array list containing all the children that the predicate returned true for
        */
        filter(predicate: Function, checkExists?: boolean): ArraySet;

        /**
        * Returns all children in this Group.
        * 
        * You can optionally specify a matching criteria using the `property` and `value` arguments.
        * 
        * For example: `getAll('exists', true)` would return only children that have an `exists` property equal to `true`.
        * 
        * Optionally you can specify a start and end index. For example if this Group had 100 children,
        * and you set `startIndex` to 0 and `endIndex` to 50, it would return the first 50 children in the Group.
        * If `property` and `value` are also specified, only children within the given index range are searched.
        * 
        * @param property An optional property to test against the value argument.
        * @param value If property is set then Child.property must strictly equal this value to be included in the results.
        * @param startIndex The first child index to start the search from.
        * @param endIndex The last child index to search up until.
        * @return - An array containing all, some, or none of the Children of this Group.
        */
        getAll(property?: string, value?: any, startIndex?: number, endIndex?: number): any[];

        /**
        * Returns the child found at the given index within this group.
        * 
        * @param index The index to return the child from.
        * @return The child that was found at the given index, or -1 for an invalid index.
        */
        getAt(index: number): PIXI.DisplayObject | number;

        /**
        * Returns the child at the bottom of this group.
        * 
        * The bottom child the child being displayed (rendered) below every other child.
        * @return The child at the bottom of the Group.
        */
        getBottom(): any;

        /**
        * Searches the Group for the first instance of a child with the `name`
        * property matching the given argument. Should more than one child have
        * the same name only the first instance is returned.
        * 
        * @param name The name to search for.
        * @return The first child with a matching name, or null if none were found.
        */
        getByName(name: string): any;

        /**
        * Get the closest child to given Object, with optional callback to filter children.
        * 
        * This can be a Sprite, Group, Image or any object with public x and y properties.
        * 
        * 'close' is determined by the distance from the objects `x` and `y` properties compared to the childs `x` and `y` properties.
        * 
        * You can use the optional `callback` argument to apply your own filter to the distance checks.
        * If the child is closer then the previous child, it will be sent to `callback` as the first argument,
        * with the distance as the second. The callback should return `true` if it passes your
        * filtering criteria, otherwise it should return `false`.
        * 
        * @param object The object used to determine the distance. This can be a Sprite, Group, Image or any object with public x and y properties.
        * @param callback The function that each child will be evaluated against. Each child of the group will be passed to it as its first parameter, with the distance as the second. It should return `true` if the child passes the matching criteria.
        * @param callbackContext The context in which the function should be called (usually 'this').
        * @return The child closest to given object, or `null` if no child was found.
        */
        getClosestTo(object: any, callback?: Function, callbackContext?: any): any;

        /**
        * Get the first child that is alive (`child.alive === true`).
        * 
        * This is handy for choosing a squad leader, etc.
        * 
        * You can use the optional argument `createIfNull` to create a new Game Object if no alive ones were found in this Group.
        * 
        * It works by calling `Group.create` passing it the parameters given to this method, and returning the new child.
        * 
        * If a child *was* found , `createIfNull` is `false` and you provided the additional arguments then the child
        * will be reset and/or have a new texture loaded on it. This is handled by `Group.resetChild`.
        * 
        * @param createIfNull If `true` and no alive children are found a new one is created.
        * @param x The x coordinate to reset the child to. The value is in relation to the group.x point.
        * @param y The y coordinate to reset the child to. The value is in relation to the group.y point.
        * @param key This is the image or texture used by the Sprite during rendering. It can be a string which is a reference to the Cache Image entry, or an instance of a RenderTexture, BitmapData, Video or PIXI.Texture.
        * @param frame If this Sprite is using part of a sprite sheet or texture atlas you can specify the exact frame to use by giving a string or numeric index.
        * @return The alive dead child, or `null` if none found and `createIfNull` was false.
        */
        getFirstAlive(createIfNull?: boolean, x?: number, y?: number, key?: string | Phaser.RenderTexture | Phaser.BitmapData | Phaser.Video | PIXI.Texture, frame?: string | number): any;

        /**
        * Get the first child that is dead (`child.alive === false`).
        * 
        * This is handy for checking if everything has been wiped out and adding to the pool as needed.
        * 
        * You can use the optional argument `createIfNull` to create a new Game Object if no dead ones were found in this Group.
        * 
        * It works by calling `Group.create` passing it the parameters given to this method, and returning the new child.
        * 
        * If a child *was* found , `createIfNull` is `false` and you provided the additional arguments then the child
        * will be reset and/or have a new texture loaded on it. This is handled by `Group.resetChild`.
        * 
        * @param createIfNull If `true` and no dead children are found a new one is created.
        * @param x The x coordinate to reset the child to. The value is in relation to the group.x point.
        * @param y The y coordinate to reset the child to. The value is in relation to the group.y point.
        * @param key This is the image or texture used by the Sprite during rendering. It can be a string which is a reference to the Cache Image entry, or an instance of a RenderTexture, BitmapData, Video or PIXI.Texture.
        * @param frame If this Sprite is using part of a sprite sheet or texture atlas you can specify the exact frame to use by giving a string or numeric index.
        * @return The first dead child, or `null` if none found and `createIfNull` was false.
        */
        getFirstDead(createIfNull?: boolean, x?: number, y?: number, key?: string | Phaser.RenderTexture | Phaser.BitmapData | Phaser.Video | PIXI.Texture, frame?: string | number): any;

        /**
        * Get the first display object that exists, or doesn't exist.
        * 
        * You can use the optional argument `createIfNull` to create a new Game Object if none matching your exists argument were found in this Group.
        * 
        * It works by calling `Group.create` passing it the parameters given to this method, and returning the new child.
        * 
        * If a child *was* found , `createIfNull` is `false` and you provided the additional arguments then the child
        * will be reset and/or have a new texture loaded on it. This is handled by `Group.resetChild`.
        * 
        * @param exists If true, find the first existing child; otherwise find the first non-existing child. - Default: true
        * @param createIfNull If `true` and no alive children are found a new one is created.
        * @param x The x coordinate to reset the child to. The value is in relation to the group.x point.
        * @param y The y coordinate to reset the child to. The value is in relation to the group.y point.
        * @param key This is the image or texture used by the Sprite during rendering. It can be a string which is a reference to the Cache Image entry, or an instance of a RenderTexture, BitmapData, Video or PIXI.Texture.
        * @param frame If this Sprite is using part of a sprite sheet or texture atlas you can specify the exact frame to use by giving a string or numeric index.
        * @return The first child, or `null` if none found and `createIfNull` was false.
        */
        getFirstExists(exists: boolean, createIfNull?: boolean, x?: number, y?: number, key?: string | Phaser.RenderTexture | Phaser.BitmapData | Phaser.Video | PIXI.Texture, frame?: string | number): any;

        /**
        * Get the child furthest away from the given Object, with optional callback to filter children.
        * 
        * This can be a Sprite, Group, Image or any object with public x and y properties.
        * 
        * 'furthest away' is determined by the distance from the objects `x` and `y` properties compared to the childs `x` and `y` properties.
        * 
        * You can use the optional `callback` argument to apply your own filter to the distance checks.
        * If the child is closer then the previous child, it will be sent to `callback` as the first argument,
        * with the distance as the second. The callback should return `true` if it passes your
        * filtering criteria, otherwise it should return `false`.
        * 
        * @param object The object used to determine the distance. This can be a Sprite, Group, Image or any object with public x and y properties.
        * @param callback The function that each child will be evaluated against. Each child of the group will be passed to it as its first parameter, with the distance as the second. It should return `true` if the child passes the matching criteria.
        * @param callbackContext The context in which the function should be called (usually 'this').
        * @return The child furthest from the given object, or `null` if no child was found.
        */
        getFurthestFrom(object: any, callback?: Function, callbackContext?: any): any;

        /**
        * Get the index position of the given child in this group, which should match the child's `z` property.
        * 
        * @param child The child to get the index for.
        * @return The index of the child or -1 if it's not a member of this group.
        */
        getIndex(child: any): number;

        /**
        * Returns a random child from the group.
        * 
        * @param startIndex Offset from the front of the group (lowest child).
        * @param length Restriction on the number of values you want to randomly select from. - Default: (to top)
        * @return A random child of this Group.
        */
        getRandom(startIndex?: number, length?: number): any;

        /**
        * Return the child at the top of this group.
        * 
        * The top child is the child displayed (rendered) above every other child.
        * @return The child at the top of the Group.
        */
        getTop(): any;

        /**
        * Checks if the child has the given property.
        * 
        * Will scan up to 4 levels deep only.
        * 
        * @param child The child to check for the existence of the property on.
        * @param key An array of strings that make up the property.
        * @return True if the child has the property, otherwise false.
        */
        hasProperty(child: any, key: string[]): boolean;

        /**
        * Iterates over the children of the group performing one of several actions for matched children.
        * 
        * A child is considered a match when it has a property, named `key`, whose value is equal to `value`
        * according to a strict equality comparison.
        * 
        * The result depends on the `returnType`:
        * 
        * - {@link Phaser.Group.RETURN_TOTAL RETURN_TOTAL}:
        *     The callback, if any, is applied to all matching children. The number of matched children is returned.
        * - {@link Phaser.Group.RETURN_NONE RETURN_NONE}:
        *     The callback, if any, is applied to all matching children. No value is returned.
        * - {@link Phaser.Group.RETURN_CHILD RETURN_CHILD}:
        *     The callback, if any, is applied to the *first* matching child and the *first* matched child is returned.
        *     If there is no matching child then null is returned.
        * 
        * If `args` is specified it must be an array. The matched child will be assigned to the first
        * element and the entire array will be applied to the callback function.
        * 
        * @param key The child property to check, i.e. 'exists', 'alive', 'health'
        * @param value A child matches if `child[key] === value` is true.
        * @param returnType How to iterate the children and what to return.
        * @param callback Optional function that will be called on each matching child. The matched child is supplied as the first argument.
        * @param callbackContext The context in which the function should be called (usually 'this').
        * @param args The arguments supplied to to the callback; the first array index (argument) will be replaced with the matched child. - Default: (none)
        * @return Returns either an integer (for RETURN_TOTAL), the first matched child (for RETURN_CHILD), or null.
        */
        iterate(key: string, value: any, returnType: number, callback?: Function, callbackContext?: any, ...args: any[]): any;

        /**
        * Kills all children having exists=true.
        */
        killAll(): void;

        /**
        * Moves all children from this Group to the Group given.
        * 
        * @param group The new Group to which the children will be moved to.
        * @param silent If true the children will not dispatch the `onAddedToGroup` event for the new Group.
        * @return The Group to which all the children were moved.
        */
        moveAll(group: Phaser.Group, silent?: boolean): Phaser.Group;

        /**
        * Moves the given child down one place in this group unless it's already at the bottom.
        * 
        * @param child The child to move down in the group.
        * @return The child that was moved.
        */
        moveDown(child: any): any;

        /**
        * Moves the given child up one place in this group unless it's already at the top.
        * 
        * @param child The child to move up in the group.
        * @return The child that was moved.
        */
        moveUp(child: any): any;

        /**
        * Multiplies the given property by the amount on all children in this group.
        * 
        * `Group.multiplyAll('x', 2)` will x2 the child.x value for each child.
        * 
        * @param property The property to multiply, for example 'body.velocity.x' or 'angle'.
        * @param amount The amount to multiply the property by. If child.x = 10 then multiplyAll('x', 2) would make child.x = 20.
        * @param checkAlive If true the property will only be changed if the child is alive.
        * @param checkVisible If true the property will only be changed if the child is visible.
        */
        multiplyAll(property: string, amount: number, checkAlive: boolean, checkVisible: boolean): void;

        /**
        * Advances the group cursor to the next (higher) object in the group.
        * 
        * If the cursor is at the end of the group (top child) it is moved the start of the group (bottom child).
        * @return The child the cursor now points to.
        */
        next(): any;

        /**
        * The core postUpdate - as called by World.
        */
        postUpdate(): void;

        /**
        * The core preUpdate - as called by World.
        */
        preUpdate(): void;

        /**
        * Moves the group cursor to the previous (lower) child in the group.
        * 
        * If the cursor is at the start of the group (bottom child) it is moved to the end (top child).
        * @return The child the cursor now points to.
        */
        previous(): void;

        /**
        * Removes the given child from this group.
        * 
        * This will dispatch an `onRemovedFromGroup` event from the child (if it has one), and optionally destroy the child.
        * 
        * If the group cursor was referring to the removed child it is updated to refer to the next child.
        * 
        * @param child The child to remove.
        * @param destroy If true `destroy` will be invoked on the removed child.
        * @param silent If true the the child will not dispatch the `onRemovedFromGroup` event.
        * @return true if the child was removed from this group, otherwise false.
        */
        remove(child: any, destroy?: boolean, silent?: boolean): boolean;

        /**
        * Removes all children from this Group, but does not remove the group from its parent.
        * 
        * The children can be optionally destroyed as they are removed.
        * 
        * You can also optionally also destroy the BaseTexture the Child is using. Be careful if you've
        * more than one Game Object sharing the same BaseTexture.
        * 
        * @param destroy If true `destroy` will be invoked on each removed child.
        * @param silent If true the children will not dispatch their `onRemovedFromGroup` events.
        * @param destroyTexture If true, and if the `destroy` argument is also true, the BaseTexture belonging to the Child is also destroyed. Note that if another Game Object is sharing the same BaseTexture it will invalidate it.
        */
        removeAll(destroy?: boolean, silent?: boolean, destroyTexture?: boolean): void;

        /**
        * Removes all children from this group whose index falls beteen the given startIndex and endIndex values.
        * 
        * @param startIndex The index to start removing children from.
        * @param endIndex The index to stop removing children at. Must be higher than startIndex. If undefined this method will remove all children between startIndex and the end of the group.
        * @param destroy If true `destroy` will be invoked on each removed child.
        * @param silent If true the children will not dispatch their `onRemovedFromGroup` events.
        */
        removeBetween(startIndex: number, endIndex?: number, destroy?: boolean, silent?: boolean): void;

        /**
        * Removes a child of this Group from the hash array.
        * This call will return false if the child is not in the hash.
        * 
        * @param child The display object to remove from this Groups hash. Must be a member of this Group and in the hash.
        * @return True if the child was successfully removed from the hash, otherwise false.
        */
        removeFromHash(child: PIXI.DisplayObject): boolean;

        /**
        * Replaces a child of this Group with the given newChild. The newChild cannot be a member of this Group.
        * 
        * If `Group.enableBody` is set, then a physics body will be created on the object, so long as one does not already exist.
        * 
        * If `Group.inputEnableChildren` is set, then an Input Handler will be created on the object, so long as one does not already exist.
        * 
        * @param oldChild The child in this group that will be replaced.
        * @param newChild The child to be inserted into this group.
        * @return Returns the oldChild that was replaced within this group.
        */
        replace(oldChild: any, newChild: any): any;

        /**
        * Calls {@link Phaser.Group#resetChild resetChild} on each child (or each existing child).
        * 
        * @param x The x coordinate to reset each child to. The value is in relation to the group.x point.
        * @param y The y coordinate to reset each child to. The value is in relation to the group.y point.
        * @param key The image or texture used by the Sprite during rendering.
        * @param frame The frame of a sprite sheet or texture atlas.
        * @param checkExists Reset only existing children.
        */
        resetAll(x?: number, y?: number, key?: string | Phaser.RenderTexture | Phaser.BitmapData | Phaser.Video | PIXI.Texture, frame?: string | number, checkExists?: boolean): void;

        /**
        * Takes a child and if the `x` and `y` arguments are given it calls `child.reset(x, y)` on it.
        * 
        * If the `key` and optionally the `frame` arguments are given, it calls `child.loadTexture(key, frame)` on it.
        * 
        * The two operations are separate. For example if you just wish to load a new texture then pass `null` as the x and y values.
        * 
        * @param child The child to reset and/or load the texture on.
        * @param x The x coordinate to reset the child to. The value is in relation to the group.x point.
        * @param y The y coordinate to reset the child to. The value is in relation to the group.y point.
        * @param key This is the image or texture used by the Sprite during rendering. It can be a string which is a reference to the Cache Image entry, or an instance of a RenderTexture, BitmapData, Video or PIXI.Texture.
        * @param frame If this Sprite is using part of a sprite sheet or texture atlas you can specify the exact frame to use by giving a string or numeric index.
        * @return The child that was reset: usually a {@link Phaser.Sprite}.
        */
        resetChild(child: any, x?: number, y?: number, key?: string | Phaser.RenderTexture | Phaser.BitmapData | Phaser.Video | PIXI.Texture, frame?: string | number): any;

        /**
        * Sets the group cursor to the first child in the group.
        * 
        * If the optional index parameter is given it sets the cursor to the object at that index instead.
        * 
        * @param index Set the cursor to point to a specific index.
        * @return The child the cursor now points to.
        */
        resetCursor(index?: number): any;

        /**
        * Reverses all children in this group.
        * 
        * This operation applies only to immediate children and does not propagate to subgroups.
        */
        reverse(): void;

        /**
        * Revives all children having exists=false.
        */
        reviveAll(): void;

        /**
        * Places each child at a random position within the given Rectangle (or the {@link Phaser.World#bounds World bounds}).
        * 
        * @param rect A Rectangle. If omitted {@link Phaser.World#bounds} is used. - Default: this.game.world.bounds
        * @param checkExists Place only children with exists=true.
        */
        scatter(rect?: Phaser.Rectangle, checkExists?: boolean): void;

        /**
        * Sends the given child to the bottom of this group so it renders below all other children.
        * 
        * @param child The child to send to the bottom of this group.
        * @return The child that was moved.
        */
        sendToBack(child: any): any;

        /**
        * Quickly set a property on a single child of this group to a new value.
        * 
        * The operation parameter controls how the new value is assigned to the property, from simple replacement to addition and multiplication.
        * 
        * @param child The child to set the property on.
        * @param key The property, as a string, to be set. For example: 'body.velocity.x'
        * @param value The value that will be set.
        * @param checkAlive If set then the child will only be updated if alive=true.
        * @param checkVisible If set then the child will only be updated if visible=true.
        * @param operation Controls how the value is assigned. A value of 0 replaces the value with the new one. A value of 1 adds it, 2 subtracts it, 3 multiplies it and 4 divides it.
        * @param force If `force` is true then the property will be set on the child regardless if it already exists or not. If false and the property doesn't exist, nothing will be set.
        * @return True if the property was set, false if not.
        */
        set(child: any, key: string[], value: any, operation?: number, force?: boolean): boolean;

        /**
        * Quickly set the same property across all children of this group to a new value.
        * 
        * This call doesn't descend down children, so if you have a Group inside of this group, the property will be set on the group but not its children.
        * If you need that ability please see `Group.setAllChildren`.
        * 
        * The operation parameter controls how the new value is assigned to the property, from simple replacement to addition and multiplication.
        * 
        * @param key The property, as a string, to be set. For example: 'body.velocity.x'
        * @param value The value that will be set.
        * @param checkAlive If set then only children with alive=true will be updated. This includes any Groups that are children.
        * @param checkVisible If set then only children with visible=true will be updated. This includes any Groups that are children.
        * @param operation Controls how the value is assigned. A value of 0 replaces the value with the new one. A value of 1 adds it, 2 subtracts it, 3 multiplies it and 4 divides it.
        * @param force If `force` is true then the property will be set on the child regardless if it already exists or not. If false and the property doesn't exist, nothing will be set.
        */
        setAll(key: string, value: any, checkAlive?: boolean, checkVisible?: boolean, operation?: number, force?: boolean): void;

        /**
        * Quickly set the same property across all children of this group, and any child Groups, to a new value.
        * 
        * If this group contains other Groups then the same property is set across their children as well, iterating down until it reaches the bottom.
        * Unlike with `setAll` the property is NOT set on child Groups itself.
        * 
        * The operation parameter controls how the new value is assigned to the property, from simple replacement to addition and multiplication.
        * 
        * @param key The property, as a string, to be set. For example: 'body.velocity.x'
        * @param value The value that will be set.
        * @param checkAlive If set then only children with alive=true will be updated. This includes any Groups that are children.
        * @param checkVisible If set then only children with visible=true will be updated. This includes any Groups that are children.
        * @param operation Controls how the value is assigned. A value of 0 replaces the value with the new one. A value of 1 adds it, 2 subtracts it, 3 multiplies it and 4 divides it.
        * @param force If `force` is true then the property will be set on the child regardless if it already exists or not. If false and the property doesn't exist, nothing will be set.
        */
        setAllChildren(key: string, value: any, checkAlive?: boolean, checkVisible?: boolean, operation?: number, force?: boolean): void;

        /**
        * Sets a property to the given value on the child. The operation parameter controls how the value is set.
        * 
        * The operations are:
        * - 0: set the existing value to the given value; if force is `true` a new property will be created if needed
        * - 1: will add the given value to the value already present.
        * - 2: will subtract the given value from the value already present.
        * - 3: will multiply the value already present by the given value.
        * - 4: will divide the value already present by the given value.
        * 
        * @param child The child to set the property value on.
        * @param key An array of strings that make up the property that will be set.
        * @param value The value that will be set.
        * @param operation Controls how the value is assigned. A value of 0 replaces the value with the new one. A value of 1 adds it, 2 subtracts it, 3 multiplies it and 4 divides it.
        * @param force If `force` is true then the property will be set on the child regardless if it already exists or not. If false and the property doesn't exist, nothing will be set.
        * @return True if the property was set, false if not.
        */
        setProperty(child: any, key: string[], value: any, operation?: number, force?: boolean): boolean;

        /**
        * Orders this Group's children randomly.
        * 
        * This can be more efficient than calling {@link Phaser.Group#getRandom getRandom} repeatedly.
        */
        shuffle(): void;

        /**
        * Sort the children in the group according to a particular key and ordering.
        * 
        * Call this function to sort the group according to a particular key value and order.
        * 
        * For example to depth sort Sprites for Zelda-style game you might call `group.sort('y', Phaser.Group.SORT_ASCENDING)` at the bottom of your `State.update()`.
        * 
        * Internally this uses a standard JavaScript Array sort, so everything that applies there also applies here, including
        * alphabetical sorting, mixing strings and numbers, and Unicode sorting. See MDN for more details.
        * 
        * @param key The name of the property to sort on. Defaults to the objects z-depth value. - Default: 'z'
        * @param order Order ascending ({@link Phaser.Group.SORT_ASCENDING SORT_ASCENDING}) or descending ({@link Phaser.Group.SORT_DESCENDING SORT_DESCENDING}). - Default: Phaser.Group.SORT_ASCENDING
        */
        sort(key?: string, order?: number): void;

        /**
        * Subtracts the amount from the given property on all children in this group.
        * 
        * `Group.subAll('x', 10)` will minus 10 from the child.x value for each child.
        * 
        * @param property The property to decrement, for example 'body.velocity.x' or 'angle'.
        * @param amount The amount to subtract from the property. If child.x = 50 then subAll('x', 40) would make child.x = 10.
        * @param checkAlive If true the property will only be changed if the child is alive.
        * @param checkVisible If true the property will only be changed if the child is visible.
        */
        subAll(property: string, amount: number, checkAlive: boolean, checkVisible: boolean): void;

        /**
        * Swaps the position of two children in this group.
        * 
        * Both children must be in this group, a child cannot be swapped with itself, and unparented children cannot be swapped.
        * 
        * @param child1 The first child to swap.
        * @param child2 The second child to swap.
        */
        swap(child1: any, child2: any): boolean;

        /**
        * The core update - as called by World.
        * 
        * Children with `exists = false` are updated unless {@link Phaser.Group#updateOnlyExistingChildren updateOnlyExistingChildren} is true.
        */
        update(): void;

        /**
        * Internal method that re-applies all of the children's Z values.
        * 
        * This must be called whenever children ordering is altered so that their `z` indices are correctly updated.
        */
        updateZ(): void;

        /**
        * Positions the child found at the given index within this group to the given x and y coordinates.
        * 
        * @param index The index of the child in the group to set the position of.
        * @param x The new x position of the child.
        * @param y The new y position of the child.
        */
        xy(index: number, x: number, y: number): void;

    }


    /**
    * An Image is a light-weight object you can use to display anything that doesn't need health, physics, or complex position monitoring.
    * 
    * It can still rotate, scale, crop and receive input events. This makes it perfect for logos, backgrounds, simple buttons and other non-Sprite graphics.
    */
    class Image extends PIXI.Sprite {


        /**
        * An Image is a light-weight object you can use to display anything that doesn't need health, physics, or complex position monitoring.
        * 
        * It can still rotate, scale, crop and receive input events. This makes it perfect for logos, backgrounds, simple buttons and other non-Sprite graphics.
        * 
        * @param game A reference to the currently running game.
        * @param x The x coordinate of the Image. The coordinate is relative to any parent container this Image may be in.
        * @param y The y coordinate of the Image. The coordinate is relative to any parent container this Image may be in.
        * @param key The texture used by the Image during rendering. It can be a string which is a reference to the Cache entry, or an instance of a RenderTexture, BitmapData or PIXI.Texture. If this argument is omitted, the image will receive {@link Phaser.Cache.DEFAULT the default texture} (as if you had passed '__default'), but its `key` will remain empty.
        * @param frame If this Image is using part of a sprite sheet or texture atlas you can specify the exact frame to use by giving a string or numeric index.
        */
        constructor(game: Phaser.Game, x: number, y: number, key: string | Phaser.RenderTexture | Phaser.BitmapData | PIXI.Texture, frame?: string | number);


        /**
        * A useful flag to control if the Game Object is alive or dead.
        * 
        * This is set automatically by the Health components `damage` method should the object run out of health.
        * Or you can toggle it via your game code.
        * 
        * This property is mostly just provided to be used by your game - it doesn't effect rendering or logic updates.
        * However you can use `Group.getFirstAlive` in conjunction with this property for fast object pooling and recycling.
        * Default: true
        */
        alive: boolean;

        /**
        * The angle property is the rotation of the Game Object in *degrees* from its original orientation.
        * 
        * Values from 0 to 180 represent clockwise rotation; values from 0 to -180 represent counterclockwise rotation.
        * 
        * Values outside this range are added to or subtracted from 360 to obtain a value within the range.
        * For example, the statement player.angle = 450 is the same as player.angle = 90.
        * 
        * If you wish to work in radians instead of degrees you can use the property `rotation` instead.
        * Working in radians is slightly faster as it doesn't have to perform any calculations.
        */
        angle: number;

        /**
        * The anchor sets the origin point of the texture.
        * The default (0, 0) is the top left.
        * (0.5, 0.5) is the center.
        * (1, 1) is the bottom right.
        * 
        * You can modify the default values in PIXI.Sprite.defaultAnchor.
        */
        anchor: Phaser.Point;

        /**
        * If the Game Object is enabled for animation (such as a Phaser.Sprite) this is a reference to its AnimationManager instance.
        * Through it you can create, play, pause and stop animations.
        */
        animations: Phaser.AnimationManager;

        /**
        * A Game Object with `autoCull` set to true will check its bounds against the World Camera every frame.
        * If it is not intersecting the Camera bounds at any point then it has its `renderable` property set to `false`.
        * This keeps the Game Object alive and still processing updates, but forces it to skip the render step entirely.
        * 
        * This is a relatively expensive operation, especially if enabled on hundreds of Game Objects. So enable it only if you know it's required,
        * or you have tested performance and find it acceptable.
        */
        autoCull: boolean;

        /**
        * The sum of the y and height properties.
        * This is the same as `y + height - offsetY`.
        */
        bottom: number;

        /**
        * The x/y coordinate offset applied to the top-left of the camera that this Game Object will be drawn at if `fixedToCamera` is true.
        * 
        * The values are relative to the top-left of the camera view and in addition to any parent of the Game Object on the display list.
        */
        cameraOffset: Phaser.Point;

        /**
        * The center x coordinate of the Game Object.
        * This is the same as `(x - offsetX) + (width / 2)`.
        */
        centerX: number;

        /**
        * The center y coordinate of the Game Object.
        * This is the same as `(y - offsetY) + (height / 2)`.
        */
        centerY: number;

        /**
        * The components this Game Object has installed.
        */
        components: any;

        /**
        * The Rectangle used to crop the texture this Game Object uses.
        * Set this property via `crop`.
        * If you modify this property directly you must call `updateCrop` in order to have the change take effect.
        */
        cropRect: Phaser.Rectangle;

        /**
        * Does this texture require a custom render call? (as set by BitmapData, Video, etc)
        */
        customRender: boolean;

        /**
        * An empty Object that belongs to this Game Object.
        * This value isn't ever used internally by Phaser, but may be used by your own code, or
        * by Phaser Plugins, to store data that needs to be associated with the Game Object,
        * without polluting the Game Object directly.
        * Default: {}
        */
        data: any;

        /**
        * A debug flag designed for use with `Game.enableStep`.
        */
        debug: boolean;
        deltaX: number;
        deltaY: number;
        deltaZ: number;

        /**
        * As a Game Object runs through its destroy method this flag is set to true,
        * and can be checked in any sub-systems or plugins it is being destroyed from.
        */
        destroyPhase: boolean;

        /**
        * All Phaser Game Objects have an Events class which contains all of the events that are dispatched when certain things happen to this
        * Game Object, or any of its components.
        */
        events: Phaser.Events;

        /**
        * Controls if this Sprite is processed by the core Phaser game loops and Group loops (except {@link Phaser.Group#update}).
        * Default: true
        */
        exists: boolean;

        /**
        * A Game Object that is "fixed" to the camera is rendered at a given x/y offsets from the top left of the camera. The offsets
        * are stored in the `cameraOffset` property, which is initialized with the current object coordinates.
        * 
        * The values are adjusted at the rendering stage, overriding the Game Objects actual world position.
        * 
        * The end result is that the Game Object will appear to be 'fixed' to the camera, regardless of where in the game world
        * the camera is viewing. This is useful if for example this Game Object is a UI item that you wish to be visible at all times
        * regardless where in the world the camera is.
        * 
        * Note that the `cameraOffset` values are in addition to any parent of this Game Object on the display list.
        * 
        * Be careful not to set `fixedToCamera` on Game Objects which are in Groups that already have `fixedToCamera` enabled on them.
        */
        fixedToCamera: boolean;

        /**
        * Gets or sets the current frame index of the texture being used to render this Game Object.
        * 
        * To change the frame set `frame` to the index of the new frame in the sprite sheet you wish this Game Object to use,
        * for example: `player.frame = 4`.
        * 
        * If the frame index given doesn't exist it will revert to the first frame found in the texture.
        * 
        * If you are using a texture atlas then you should use the `frameName` property instead.
        * 
        * If you wish to fully replace the texture being used see `loadTexture`.
        */
        frame: string | number;

        /**
        * Gets or sets the current frame name of the texture being used to render this Game Object.
        * 
        * To change the frame set `frameName` to the name of the new frame in the texture atlas you wish this Game Object to use,
        * for example: `player.frameName = "idle"`.
        * 
        * If the frame name given doesn't exist it will revert to the first frame found in the texture and throw a console warning.
        * 
        * If you are using a sprite sheet then you should use the `frame` property instead.
        * 
        * If you wish to fully replace the texture being used see `loadTexture`.
        */
        frameName: string;

        /**
        * A Game Object is considered `fresh` if it has just been created or reset and is yet to receive a renderer transform update.
        * This property is mostly used internally by the physics systems, but is exposed for the use of plugins.
        */
        fresh: boolean;

        /**
        * A reference to the currently running Game.
        */
        game: Phaser.Game;

        /**
        * Checks if the Game Objects bounds intersect with the Game Camera bounds.
        * Returns `true` if they do, otherwise `false` if fully outside of the Cameras bounds.
        */
        inCamera: boolean;

        /**
        * The Input Handler for this Game Object.
        * 
        * By default it is disabled. If you wish this Game Object to process input events you should enable it with: `inputEnabled = true`.
        * 
        * After you have done this, this property will be a reference to the Phaser InputHandler.
        */
        input: Phaser.InputHandler;

        /**
        * By default a Game Object won't process any input events. By setting `inputEnabled` to true a Phaser.InputHandler is created
        * for this Game Object and it will then start to process click / touch events and more.
        * 
        * You can then access the Input Handler via `this.input`.
        * 
        * Note that Input related events are dispatched from `this.events`, i.e.: `events.onInputDown`.
        * 
        * If you set this property to false it will stop the Input Handler from processing any more input events.
        * 
        * If you want to _temporarily_ disable input for a Game Object, then it's better to set
        * `input.enabled = false`, as it won't reset any of the Input Handlers internal properties.
        * You can then toggle this back on as needed.
        */
        inputEnabled: boolean;
        inWorld: boolean;

        /**
        * The key of the image or texture used by this Game Object during rendering.
        * If it is a string it's the string used to retrieve the texture from the Phaser Image Cache.
        * It can also be an instance of a RenderTexture, BitmapData, Video or PIXI.Texture.
        * If a Game Object is created without a key it is automatically assigned the key `__default` which is a 32x32 transparent PNG stored within the Cache.
        * If a Game Object is given a key which doesn't exist in the Image Cache it is re-assigned the key `__missing` which is a 32x32 PNG of a green box with a line through it.
        */
        key: string | Phaser.RenderTexture | Phaser.BitmapData | Phaser.Video | PIXI.Texture;

        /**
        * The lifespan allows you to give a Game Object a lifespan in milliseconds.
        * 
        * Once the Game Object is 'born' you can set this to a positive value.
        * 
        * It is automatically decremented by the millisecond equivalent of `game.time.physicsElapsed` each frame.
        * When it reaches zero it will call the `kill` method.
        * 
        * Very handy for particles, bullets, collectibles, or any other short-lived entity.
        */
        lifespan: number;

        /**
        * The left coordinate of the Game Object.
        * This is the same as `x - offsetX`.
        */
        left: number;

        /**
        * A user defined name given to this Game Object.
        * This value isn't ever used internally by Phaser, it is meant as a game level property.
        */
        name: string;

        /**
        * The amount the Game Object is visually offset from its x coordinate.
        * This is the same as `width * anchor.x`.
        * It will only be > 0 if anchor.x is not equal to zero.
        */
        offsetX: number;

        /**
        * The amount the Game Object is visually offset from its y coordinate.
        * This is the same as `height * anchor.y`.
        * It will only be > 0 if anchor.y is not equal to zero.
        */
        offsetY: number;

        /**
        * A Game Object is that is pendingDestroy is flagged to have its destroy method called on the next logic update.
        * You can set it directly to allow you to flag an object to be destroyed on its next update.
        * 
        * This is extremely useful if you wish to destroy an object from within one of its own callbacks
        * such as with Buttons or other Input events.
        */
        pendingDestroy: boolean;

        /**
        * The coordinates, in pixels, of this DisplayObject, relative to its parent container.
        * 
        * The value of this property does not reflect any positioning happening further up the display list.
        * To obtain that value please see the `worldPosition` property.
        */
        position: Phaser.Point;

        /**
        * The position the Game Object was located in the previous frame.
        */
        previousPosition: Phaser.Point;

        /**
        * The rotation the Game Object was in set to in the previous frame. Value is in radians.
        */
        previousRotation: number;

        /**
        * The render order ID is used internally by the renderer and Input Manager and should not be modified.
        * This property is mostly used internally by the renderers, but is exposed for the use of plugins.
        */
        renderOrderID: number;

        /**
        * The right coordinate of the Game Object.
        * This is the same as `x + width - offsetX`.
        */
        right: number;

        /**
        * The scale of this DisplayObject. A scale of 1:1 represents the DisplayObject
        * at its default size. A value of 0.5 would scale this DisplayObject by half, and so on.
        * 
        * The value of this property does not reflect any scaling happening further up the display list.
        * To obtain that value please see the `worldScale` property.
        */
        scale: Phaser.Point;

        /**
        * Enable or disable texture smoothing for this Game Object.
        * 
        * It only takes effect if the Game Object is using an image based texture.
        * 
        * Smoothing is enabled by default.
        */
        smoothed: boolean;

        /**
        * The y coordinate of the Game Object.
        * This is the same as `y - offsetY`.
        */
        top: number;

        /**
        * The const type of this object.
        */
        type: number;

        /**
        * The world coordinates of this Game Object in pixels.
        * Depending on where in the display list this Game Object is placed this value can differ from `position`,
        * which contains the x/y coordinates relative to the Game Objects parent.
        */
        world: Phaser.Point;

        /**
        * The z depth of this Game Object within its parent Group.
        * No two objects in a Group can have the same z value.
        * This value is adjusted automatically whenever the Group hierarchy changes.
        * If you wish to re-order the layering of a Game Object then see methods like Group.moveUp or Group.bringToTop.
        */
        z: number;


        /**
        * Aligns this Game Object within another Game Object, or Rectangle, known as the
        * 'container', to one of 9 possible positions.
        * 
        * The container must be a Game Object, or Phaser.Rectangle object. This can include properties
        * such as `World.bounds` or `Camera.view`, for aligning Game Objects within the world
        * and camera bounds. Or it can include other Sprites, Images, Text objects, BitmapText,
        * TileSprites or Buttons.
        * 
        * Please note that aligning a Sprite to another Game Object does **not** make it a child of
        * the container. It simply modifies its position coordinates so it aligns with it.
        * 
        * The position constants you can use are:
        * 
        * `Phaser.TOP_LEFT`, `Phaser.TOP_CENTER`, `Phaser.TOP_RIGHT`, `Phaser.LEFT_CENTER`,
        * `Phaser.CENTER`, `Phaser.RIGHT_CENTER`, `Phaser.BOTTOM_LEFT`,
        * `Phaser.BOTTOM_CENTER` and `Phaser.BOTTOM_RIGHT`.
        * 
        * The Game Objects are placed in such a way that their _bounds_ align with the
        * container, taking into consideration rotation, scale and the anchor property.
        * This allows you to neatly align Game Objects, irrespective of their position value.
        * 
        * The optional `offsetX` and `offsetY` arguments allow you to apply extra spacing to the final
        * aligned position of the Game Object. For example:
        * 
        * `sprite.alignIn(background, Phaser.BOTTOM_RIGHT, -20, -20)`
        * 
        * Would align the `sprite` to the bottom-right, but moved 20 pixels in from the corner.
        * Think of the offsets as applying an adjustment to the containers bounds before the alignment takes place.
        * So providing a negative offset will 'shrink' the container bounds by that amount, and providing a positive
        * one expands it.
        * 
        * @param container The Game Object or Rectangle with which to align this Game Object to. Can also include properties such as `World.bounds` or `Camera.view`.
        * @param position The position constant. One of `Phaser.TOP_LEFT` (default), `Phaser.TOP_CENTER`, `Phaser.TOP_RIGHT`, `Phaser.LEFT_CENTER`, `Phaser.CENTER`, `Phaser.RIGHT_CENTER`, `Phaser.BOTTOM_LEFT`, `Phaser.BOTTOM_CENTER` or `Phaser.BOTTOM_RIGHT`.
        * @param offsetX A horizontal adjustment of the Containers bounds, applied to the aligned position of the Game Object. Use a negative value to shrink the bounds, positive to increase it.
        * @param offsetY A vertical adjustment of the Containers bounds, applied to the aligned position of the Game Object. Use a negative value to shrink the bounds, positive to increase it.
        * @return This Game Object.
        */
        alignIn(container: Phaser.Rectangle | Phaser.Sprite | Phaser.Image | Phaser.Text | Phaser.BitmapText | Phaser.Button | Phaser.Graphics | Phaser.TileSprite, position?: number, offsetX?: number, offsetY?: number): any;

        /**
        * Aligns this Game Object to the side of another Game Object, or Rectangle, known as the
        * 'parent', in one of 11 possible positions.
        * 
        * The parent must be a Game Object, or Phaser.Rectangle object. This can include properties
        * such as `World.bounds` or `Camera.view`, for aligning Game Objects within the world
        * and camera bounds. Or it can include other Sprites, Images, Text objects, BitmapText,
        * TileSprites or Buttons.
        * 
        * Please note that aligning a Sprite to another Game Object does **not** make it a child of
        * the parent. It simply modifies its position coordinates so it aligns with it.
        * 
        * The position constants you can use are:
        * 
        * `Phaser.TOP_LEFT` (default), `Phaser.TOP_CENTER`, `Phaser.TOP_RIGHT`, `Phaser.LEFT_TOP`,
        * `Phaser.LEFT_CENTER`, `Phaser.LEFT_BOTTOM`, `Phaser.RIGHT_TOP`, `Phaser.RIGHT_CENTER`,
        * `Phaser.RIGHT_BOTTOM`, `Phaser.BOTTOM_LEFT`, `Phaser.BOTTOM_CENTER`
        * and `Phaser.BOTTOM_RIGHT`.
        * 
        * The Game Objects are placed in such a way that their _bounds_ align with the
        * parent, taking into consideration rotation, scale and the anchor property.
        * This allows you to neatly align Game Objects, irrespective of their position value.
        * 
        * The optional `offsetX` and `offsetY` arguments allow you to apply extra spacing to the final
        * aligned position of the Game Object. For example:
        * 
        * `sprite.alignTo(background, Phaser.BOTTOM_RIGHT, -20, -20)`
        * 
        * Would align the `sprite` to the bottom-right, but moved 20 pixels in from the corner.
        * Think of the offsets as applying an adjustment to the parents bounds before the alignment takes place.
        * So providing a negative offset will 'shrink' the parent bounds by that amount, and providing a positive
        * one expands it.
        * 
        * @param parent The Game Object or Rectangle with which to align this Game Object to. Can also include properties such as `World.bounds` or `Camera.view`.
        * @param position The position constant. One of `Phaser.TOP_LEFT`, `Phaser.TOP_CENTER`, `Phaser.TOP_RIGHT`, `Phaser.LEFT_TOP`, `Phaser.LEFT_CENTER`, `Phaser.LEFT_BOTTOM`, `Phaser.RIGHT_TOP`, `Phaser.RIGHT_CENTER`, `Phaser.RIGHT_BOTTOM`, `Phaser.BOTTOM_LEFT`, `Phaser.BOTTOM_CENTER` or `Phaser.BOTTOM_RIGHT`.
        * @param offsetX A horizontal adjustment of the Containers bounds, applied to the aligned position of the Game Object. Use a negative value to shrink the bounds, positive to increase it.
        * @param offsetY A vertical adjustment of the Containers bounds, applied to the aligned position of the Game Object. Use a negative value to shrink the bounds, positive to increase it.
        * @return This Game Object.
        */
        alignTo(container: Phaser.Rectangle | Phaser.Sprite | Phaser.Image | Phaser.Text | Phaser.BitmapText | Phaser.Button | Phaser.Graphics | Phaser.TileSprite, position?: number, offsetX?: number, offsetY?: number): any;

        /**
        * Brings this Game Object to the top of its parents display list.
        * Visually this means it will render over the top of any old child in the same Group.
        * 
        * If this Game Object hasn't been added to a custom Group then this method will bring it to the top of the Game World,
        * because the World is the root Group from which all Game Objects descend.
        * @return This instance.
        */
        bringToTop(): Phaser.Image;

        /**
        * Crop allows you to crop the texture being used to display this Game Object.
        * Setting a crop rectangle modifies the core texture frame. The Game Object width and height properties will be adjusted accordingly.
        * 
        * Cropping takes place from the top-left and can be modified in real-time either by providing an updated rectangle object to this method,
        * or by modifying `cropRect` property directly and then calling `updateCrop`.
        * 
        * The rectangle object given to this method can be either a `Phaser.Rectangle` or any other object
        * so long as it has public `x`, `y`, `width`, `height`, `right` and `bottom` properties.
        * 
        * A reference to the rectangle is stored in `cropRect` unless the `copy` parameter is `true`,
        * in which case the values are duplicated to a local object.
        * 
        * @param rect The Rectangle used during cropping. Pass null or no parameters to clear a previously set crop rectangle.
        * @param copy If false `cropRect` will be stored as a reference to the given rect. If true it will copy the rect values into a local Phaser Rectangle object stored in cropRect.
        */
        crop(rect: Phaser.Rectangle, copy?: boolean): void;

        /**
        * Destroys the Game Object. This removes it from its parent group, destroys the input, event and animation handlers if present
        * and nulls its reference to `game`, freeing it up for garbage collection.
        * 
        * If this Game Object has the Events component it will also dispatch the `onDestroy` event.
        * 
        * You can optionally also destroy the BaseTexture this Game Object is using. Be careful if you've
        * more than one Game Object sharing the same BaseTexture.
        * 
        * @param destroyChildren Should every child of this object have its destroy method called as well? - Default: true
        * @param destroyTexture Destroy the BaseTexture this Game Object is using? Note that if another Game Object is sharing the same BaseTexture it will invalidate it.
        */
        destroy(destroyChildren?: boolean): void;

        /**
        * Kills a Game Object. A killed Game Object has its `alive`, `exists` and `visible` properties all set to false.
        * 
        * It will dispatch the `onKilled` event. You can listen to `events.onKilled` for the signal.
        * 
        * Note that killing a Game Object is a way for you to quickly recycle it in an object pool,
        * it doesn't destroy the object or free it up from memory.
        * 
        * If you don't need this Game Object any more you should call `destroy` instead.
        * @return This instance.
        */
        kill(): Phaser.Image;

        /**
        * Changes the base texture the Game Object is using. The old texture is removed and the new one is referenced or fetched from the Cache.
        * 
        * If your Game Object is using a frame from a texture atlas and you just wish to change to another frame, then see the `frame` or `frameName` properties instead.
        * 
        * You should only use `loadTexture` if you want to replace the base texture entirely.
        * 
        * Calling this method causes a WebGL texture update, so use sparingly or in low-intensity portions of your game, or if you know the new texture is already on the GPU.
        * 
        * You can use the new const `Phaser.PENDING_ATLAS` as the texture key for any sprite.
        * Doing this then sets the key to be the `frame` argument (the frame is set to zero).
        * 
        * This allows you to create sprites using `load.image` during development, and then change them
        * to use a Texture Atlas later in development by simply searching your code for 'PENDING_ATLAS'
        * and swapping it to be the key of the atlas data.
        * 
        * Note: You cannot use a RenderTexture as a texture for a TileSprite.
        * 
        * @param key This is the image or texture used by the Sprite during rendering. It can be a string which is a reference to the Cache Image entry, or an instance of a RenderTexture, BitmapData, Video or PIXI.Texture.
        * @param frame If this Sprite is using part of a sprite sheet or texture atlas you can specify the exact frame to use by giving a string or numeric index.
        * @param stopAnimation If an animation is already playing on this Sprite you can choose to stop it or let it carry on playing. - Default: true
        */
        loadTexture(key: string | Phaser.RenderTexture | Phaser.BitmapData | Phaser.Video | PIXI.Texture, frame?: string | number, stopAnimation?: boolean): void;

        /**
        * Resizes the Frame dimensions that the Game Object uses for rendering.
        * 
        * You shouldn't normally need to ever call this, but in the case of special texture types such as Video or BitmapData
        * it can be useful to adjust the dimensions directly in this way.
        * 
        * @param parent The parent texture object that caused the resize, i.e. a Phaser.Video object.
        * @param width The new width of the texture.
        * @param height The new height of the texture.
        */
        resizeFrame(parent: any, width: number, height: number): void;

        /**
        * Moves this Game Object down one place in its parents display list.
        * This call has no effect if the Game Object is already at the bottom of the display list.
        * 
        * If this Game Object hasn't been added to a custom Group then this method will move it one object down within the Game World,
        * because the World is the root Group from which all Game Objects descend.
        * @return This instance.
        */
        moveDown(): Phaser.Image;

        /**
        * Moves this Game Object up one place in its parents display list.
        * This call has no effect if the Game Object is already at the top of the display list.
        * 
        * If this Game Object hasn't been added to a custom Group then this method will move it one object up within the Game World,
        * because the World is the root Group from which all Game Objects descend.
        * @return This instance.
        */
        moveUp(): Phaser.Image;

        /**
        * Checks to see if the bounds of this Game Object overlaps with the bounds of the given Display Object,
        * which can be a Sprite, Image, TileSprite or anything that extends those such as Button or provides a `getBounds` method and result.
        * 
        * This check ignores the `hitArea` property if set and runs a `getBounds` comparison on both objects to determine the result.
        * 
        * Therefore it's relatively expensive to use in large quantities, i.e. with lots of Sprites at a high frequency.
        * It should be fine for low-volume testing where physics isn't required.
        * 
        * @param displayObject The display object to check against.
        * @return True if the bounds of this Game Object intersects at any point with the bounds of the given display object.
        */
        overlap(displayObject: Phaser.Sprite | Phaser.Image | Phaser.TileSprite | Phaser.Button | PIXI.DisplayObject): boolean;

        /**
        * Plays an Animation.
        * 
        * The animation should have previously been created via `animations.add`.
        * 
        * If the animation is already playing calling this again won't do anything.
        * If you need to reset an already running animation do so directly on the Animation object itself or via `AnimationManager.stop`.
        * 
        * @param name The name of the animation to be played, e.g. "fire", "walk", "jump". Must have been previously created via 'AnimationManager.add'.
        * @param frameRate The framerate to play the animation at. The speed is given in frames per second. If not provided the previously set frameRate of the Animation is used.
        * @param loop Should the animation be looped after playback. If not provided the previously set loop value of the Animation is used.
        * @param killOnComplete If set to true when the animation completes (only happens if loop=false) the parent Sprite will be killed.
        * @return A reference to playing Animation.
        */
        play(name: string, frameRate?: number, loop?: boolean, killOnComplete?: boolean): Phaser.Animation;

        /**
        * Internal method called by the World postUpdate cycle.
        */
        postUpdate(): void;

        /**
        * Automatically called by World.preUpdate.
        */
        preUpdate(): void;

        /**
        * Resets the Game Object.
        * 
        * This moves the Game Object to the given x/y world coordinates and sets `fresh`, `exists`,
        * `visible` and `renderable` to true.
        * 
        * If this Game Object has the LifeSpan component it will also set `alive` to true and `health` to the given value.
        * 
        * If this Game Object has a Physics Body it will reset the Body.
        * 
        * @param x The x coordinate (in world space) to position the Game Object at.
        * @param y The y coordinate (in world space) to position the Game Object at.
        * @param health The health to give the Game Object if it has the Health component. - Default: 1
        * @return This instance.
        */
        reset(x: number, y: number, health?: number): Phaser.Image;

        /**
        * Resets the texture frame dimensions that the Game Object uses for rendering.
        */
        resetFrame(): void;

        /**
        * Brings a 'dead' Game Object back to life, optionally resetting its health value in the process.
        * 
        * A resurrected Game Object has its `alive`, `exists` and `visible` properties all set to true.
        * 
        * It will dispatch the `onRevived` event. Listen to `events.onRevived` for the signal.
        * 
        * @param health The health to give the Game Object. Only set if the GameObject has the Health component. - Default: 100
        * @return This instance.
        */
        revive(health?: number): Phaser.Image;

        /**
        * Sends this Game Object to the bottom of its parents display list.
        * Visually this means it will render below all other children in the same Group.
        * 
        * If this Game Object hasn't been added to a custom Group then this method will send it to the bottom of the Game World,
        * because the World is the root Group from which all Game Objects descend.
        * @return This instance.
        */
        sendToBack(): Phaser.Image;

        /**
        * Sets the texture frame the Game Object uses for rendering.
        * 
        * This is primarily an internal method used by `loadTexture`, but is exposed for the use of plugins and custom classes.
        * 
        * @param frame The Frame to be used by the texture.
        */
        setFrame(frame: Phaser.Frame): void;

        /**
        * Override this method in your own custom objects to handle any update requirements.
        * It is called immediately after `preUpdate` and before `postUpdate`.
        * Remember if this Game Object has any children you should call update on those too.
        */
        update(): void;

        /**
        * If you have set a crop rectangle on this Game Object via `crop` and since modified the `cropRect` property,
        * or the rectangle it references, then you need to update the crop frame by calling this method.
        */
        updateCrop(): void;

    }


    /**
    * An Image Collection is a special tileset containing mulitple images, with no slicing into each image.
    * 
    * Image Collections are normally created automatically when Tiled data is loaded.
    */
    class ImageCollection {


        /**
        * An Image Collection is a special tileset containing mulitple images, with no slicing into each image.
        * 
        * Image Collections are normally created automatically when Tiled data is loaded.
        * 
        * @param name The name of the image collection in the map data.
        * @param firstgid The first image index this image collection contains.
        * @param width Width of widest image (in pixels). - Default: 32
        * @param height Height of tallest image (in pixels). - Default: 32
        * @param margin The margin around all images in the collection (in pixels).
        * @param spacing The spacing between each image in the collection (in pixels).
        * @param properties Custom Image Collection properties. - Default: {}
        */
        constructor(name: string, firstgid: number, width?: number, height?: number, margin?: number, spacing?: number, properties?: any);


        /**
        * The name of the Image Collection.
        */
        name: string;

        /**
        * The Tiled firstgid value.
        * This is the starting index of the first image index this Image Collection contains.
        */
        firstgid: number;

        /**
        * The width of the widest image (in pixels).
        */
        imageWidth: number;

        /**
        * The height of the tallest image (in pixels).
        */
        imageHeight: number;

        /**
        * The margin around the images in the collection (in pixels).
        * Use `setSpacing` to change.
        */
        imageMargin: number;

        /**
        * The spacing between each image in the collection (in pixels).
        * Use `setSpacing` to change.
        */
        imageSpacing: number;

        /**
        * Image Collection-specific properties that are typically defined in the Tiled editor.
        */
        properties: any;

        /**
        * The cached images that are a part of this collection.
        */
        images: any[];

        /**
        * The total number of images in the image collection.
        */
        total: number;


        /**
        * Add an image to this Image Collection.
        * 
        * @param gid The gid of the image in the Image Collection.
        * @param image The the key of the image in the Image Collection and in the cache.
        */
        addImage(gid: number, image: string): void;

        /**
        * Returns true if and only if this image collection contains the given image index.
        * 
        * @param imageIndex The image index to search for.
        * @return True if this Image Collection contains the given index.
        */
        containsImageIndex(imageIndex: number): boolean;

    }


    /**
    * Phaser.Input is the Input Manager for all types of Input across Phaser, including mouse, keyboard, touch and MSPointer.
    * The Input manager is updated automatically by the core game loop.
    */
    class Input {


        /**
        * Phaser.Input is the Input Manager for all types of Input across Phaser, including mouse, keyboard, touch and MSPointer.
        * The Input manager is updated automatically by the core game loop.
        * 
        * @param game Current game instance.
        */
        constructor(game: Phaser.Game);


        /**
        * The maximum number of pointers that can be added. This excludes the mouse pointer.
        */
        static MAX_POINTERS: number;
        static MOUSE_OVERRIDES_TOUCH: number;
        static MOUSE_TOUCH_COMBINE: number;
        static TOUCH_OVERRIDES_MOUSE: number;


        /**
        * The most recently active Pointer object.
        * 
        * When you've limited max pointers to 1 this will accurately be either the first finger touched or mouse.
        */
        activePointer: Phaser.Pointer;

        /**
        * A Circle object centered on the x/y screen coordinates of the Input.
        * Default size of 44px (Apples recommended "finger tip" size) but can be changed to anything.
        */
        circle: Phaser.Circle;

        /**
        * When enabled, input (eg. Keyboard, Mouse, Touch) will be processed - as long as the individual sources are enabled themselves.
        * 
        * When not enabled, _all_ input sources are ignored. To disable just one type of input; for example, the Mouse, use `input.mouse.enabled = false`.
        * Default: true
        */
        enabled: boolean;

        /**
        * The number of milliseconds between taps of the same Pointer for it to be considered a double tap / click.
        * Default: 300
        */
        doubleTapRate: number;

        /**
        * A reference to the currently running game.
        */
        game: Phaser.Game;

        /**
        * The Gamepad Input manager.
        */
        gamepad: Phaser.Gamepad;

        /**
        * The canvas to which single pixels are drawn in order to perform pixel-perfect hit detection.
        */
        hitCanvas: HTMLCanvasElement;

        /**
        * The context of the pixel perfect hit canvas.
        */
        hitContext: CanvasRenderingContext2D;

        /**
        * The number of milliseconds that the Pointer has to be pressed down for it to fire a onHold event.
        * Default: 2000
        */
        holdRate: number;

        /**
        * A list of interactive objects. The InputHandler components add and remove themselves from this list.
        */
        interactiveItems: Phaser.ArraySet;

        /**
        * The number of milliseconds below which the Pointer is considered justPressed.
        * Default: 200
        */
        justPressedRate: number;

        /**
        * The number of milliseconds below which the Pointer is considered justReleased .
        * Default: 200
        */
        justReleasedRate: number;

        /**
        * The Keyboard Input manager.
        */
        keyboard: Phaser.Keyboard;

        /**
        * The maximum number of Pointers allowed to be active at any one time. A value of -1 is only limited by the total number of pointers. For lots of games it's useful to set this to 1.
        * Default: -1 (Limited by total pointers.)
        */
        maxPointers: number;

        /**
        * You can tell all Pointers to ignore any Game Object with a `priorityID` lower than this value.
        * This is useful when stacking UI layers. Set to zero to disable.
        */
        minPriorityID: number;

        /**
        * The Mouse Input manager.
        * 
        * You should not usually access this manager directly, but instead use Input.mousePointer or Input.activePointer
        * which normalizes all the input values for you, regardless of browser.
        */
        mouse: Phaser.Mouse;

        /**
        * The mouse has its own unique Phaser.Pointer object which you can use if making a desktop specific game.
        */
        mousePointer: Phaser.Pointer;

        /**
        * An array of callbacks that will be fired every time the activePointer receives a move event from the DOM.
        * To add a callback to this array please use `Input.addMoveCallback`.
        */
        moveCallbacks: (pointer: Phaser.Pointer, x: number, y: number) => void[];

        /**
        * The MSPointer Input manager.
        * 
        * You should not usually access this manager directly, but instead use Input.activePointer
        * which normalizes all the input values for you, regardless of browser.
        */
        mspointer: Phaser.MSPointer;

        /**
        * Controls the expected behavior when using a mouse and touch together on a multi-input device.
        */
        multiInputOverride: number;

        /**
        * A Signal that is dispatched each time a pointer is pressed down.
        */
        onDown: Phaser.Signal;

        /**
        * A Signal that is dispatched each time a pointer is held down.
        */
        onHold: Phaser.Signal;

        /**
        * A Signal that is dispatched each time a pointer is tapped.
        */
        onTap: Phaser.Signal;

        /**
        * A Signal that is dispatched each time a pointer is released.
        */
        onUp: Phaser.Signal;

        /**
        * A Pointer object.
        */
        pointer1: Phaser.Pointer;

        /**
        * A Pointer object.
        */
        pointer2: Phaser.Pointer;

        /**
        * A Pointer object.
        */
        pointer3: Phaser.Pointer;

        /**
        * A Pointer object.
        */
        pointer4: Phaser.Pointer;

        /**
        * A Pointer object.
        */
        pointer5: Phaser.Pointer;

        /**
        * A Pointer object.
        */
        pointer6: Phaser.Pointer;

        /**
        * A Pointer object.
        */
        pointer7: Phaser.Pointer;

        /**
        * A Pointer object.
        */
        pointer8: Phaser.Pointer;

        /**
        * A Pointer object.
        */
        pointer9: Phaser.Pointer;

        /**
        * A Pointer object.
        */
        pointer10: Phaser.Pointer;

        /**
        * True if the Input is currently poll rate locked.
        */
        pollLocked: boolean;

        /**
        * How often should the input pointers be checked for updates? A value of 0 means every single frame (60fps); a value of 1 means every other frame (30fps) and so on.
        */
        pollRate: number;

        /**
        * A point object representing the current position of the Pointer.
        */
        position: Phaser.Point;
        pointer: Phaser.Pointer[];

        /**
        * The total number of entries that can be recorded into the Pointer objects tracking history.
        * If the Pointer is tracking one event every 100ms; then a trackLimit of 100 would store the last 10 seconds worth of history.
        * Default: 100
        */
        recordLimit: number;

        /**
        * Sets if the Pointer objects should record a history of x/y coordinates they have passed through.
        * The history is cleared each time the Pointer is pressed down.
        * The history is updated at the rate specified in Input.pollRate
        */
        recordPointerHistory: boolean;

        /**
        * The rate in milliseconds at which the Pointer objects should update their tracking history.
        * Default: 100
        */
        recordRate: number;

        /**
        * If the Input Manager has been reset locked then all calls made to InputManager.reset,
        * such as from a State change, are ignored.
        */
        resetLocked: boolean;

        /**
        * The scale by which all input coordinates are multiplied; calculated by the ScaleManager. In an un-scaled game the values will be x = 1 and y = 1.
        */
        scale: Phaser.Point;

        /**
        * A point object representing the speed of the Pointer. Only really useful in single Pointer games; otherwise see the Pointer objects directly.
        */
        speed: Phaser.Point;

        /**
        * The number of milliseconds that the Pointer has to be pressed down and then released to be considered a tap or click.
        * Default: 200
        */
        tapRate: number;

        /**
        * The total number of active Pointers, not counting the mouse pointer.
        */
        totalActivePointers: number;

        /**
        * The total number of inactive Pointers.
        */
        totalInactivePointers: number;

        /**
        * The Touch Input manager.
        * 
        * You should not usually access this manager directly, but instead use Input.activePointer
        * which normalizes all the input values for you, regardless of browser.
        */
        touch: Phaser.Touch;

        /**
        * The world X coordinate of the most recently active pointer.
        */
        worldX: number;

        /**
        * The world Y coordinate of the most recently active pointer.
        */
        worldY: number;

        /**
        * The X coordinate of the most recently active pointer.
        * This value takes game scaling into account automatically. See Pointer.screenX/clientX for source values.
        */
        x: number;

        /**
        * The Y coordinate of the most recently active pointer.
        * This value takes game scaling into account automatically. See Pointer.screenY/clientY for source values.
        */
        y: number;


        /**
        * Add a new Pointer object to the Input Manager.
        * By default Input creates 3 pointer objects: `mousePointer` (not include in part of general pointer pool), `pointer1` and `pointer2`.
        * This method adds an additional pointer, up to a maximum of Phaser.Input.MAX_POINTERS (default of 10).
        * @return The new Pointer object that was created; null if a new pointer could not be added.
        */
        addPointer(): Phaser.Pointer;

        /**
        * Adds a callback that is fired every time the activePointer receives a DOM move event such as a mousemove or touchmove.
        * 
        * The callback will be sent 4 parameters:
        * 
        * A reference to the Phaser.Pointer object that moved,
        * The x position of the pointer,
        * The y position,
        * A boolean indicating if the movement was the result of a 'click' event (such as a mouse click or touch down).
        * 
        * It will be called every time the activePointer moves, which in a multi-touch game can be a lot of times, so this is best
        * to only use if you've limited input to a single pointer (i.e. mouse or touch).
        * 
        * The callback is added to the Phaser.Input.moveCallbacks array and should be removed with Phaser.Input.deleteMoveCallback.
        * 
        * @param callback The callback that will be called each time the activePointer receives a DOM move event.
        * @param context The context in which the callback will be called.
        */
        addMoveCallback(callback: Function, context: any): number;

        /**
        * Starts the Input Manager running.
        */
        boot(): void;
        countActivePointers(limit?: number): number;

        /**
        * Removes the callback from the Phaser.Input.moveCallbacks array.
        * 
        * @param callback The callback to be removed.
        * @param context The context in which the callback exists.
        */
        deleteMoveCallback(callback: Function, context?: any): void;

        /**
        * Stops all of the Input Managers from running.
        */
        destroy(): void;

        /**
        * This will return the local coordinates of the specified displayObject based on the given Pointer.
        * 
        * @param displayObject The DisplayObject to get the local coordinates for.
        * @param pointer The Pointer to use in the check against the displayObject.
        * @return A point containing the coordinates of the Pointer position relative to the DisplayObject.
        */
        getLocalPosition(displayObject: any, pointer: Phaser.Pointer): Phaser.Point;

        /**
        * Get the first Pointer with the given active state.
        * 
        * @param isActive The state the Pointer should be in - active or inactive?
        * @return A Pointer object or null if no Pointer object matches the requested state.
        */
        getPointer(isActive?: boolean): Phaser.Pointer;

        /**
        * Get the Pointer object whos `pointerId` property matches the given value.
        * 
        * The pointerId property is not set until the Pointer has been used at least once, as its populated by the DOM event.
        * Also it can change every time you press the pointer down if the browser recycles it.
        * 
        * @param pointerId The `pointerId` (not 'id') value to search for.
        * @return A Pointer object or null if no Pointer object matches the requested identifier.
        */
        getPointerFromId(pointerID: number): Phaser.Pointer;

        /**
        * Get the Pointer object whos `identifier` property matches the given identifier value.
        * 
        * The identifier property is not set until the Pointer has been used at least once, as its populated by the DOM event.
        * Also it can change every time you press the pointer down, and is not fixed once set.
        * Note: Not all browsers set the identifier property and it's not part of the W3C spec, so you may need getPointerFromId instead.
        * 
        * @param identifier The Pointer.identifier value to search for.
        * @return A Pointer object or null if no Pointer object matches the requested identifier.
        */
        getPointerFromIdentifier(identifier: number): Phaser.Pointer;

        /**
        * Tests if the pointer hits the given object.
        * 
        * @param displayObject The displayObject to test for a hit.
        * @param pointer The pointer to use for the test.
        * @param localPoint The local translated point.
        */
        hitTest(displayObject: PIXI.DisplayObject, pointer: Phaser.Pointer, localPoint: Phaser.Point): void;

        /**
        * Reset all of the Pointers and Input states.
        * 
        * The optional `hard` parameter will reset any events or callbacks that may be bound.
        * Input.reset is called automatically during a State change or if a game loses focus / visibility.
        * To control control the reset manually set {@link Phaser.InputManager.resetLocked} to `true`.
        * 
        * @param hard A soft reset won't reset any events or callbacks that are bound. A hard reset will.
        */
        reset(hard?: boolean): void;

        /**
        * Resets the speed and old position properties.
        * 
        * @param x Sets the oldPosition.x value.
        * @param y Sets the oldPosition.y value.
        */
        resetSpeed(x: number, y: number): void;

        /**
        * Adds a callback that is fired every time `Pointer.processInteractiveObjects` is called.
        * The purpose of `processInteractiveObjects` is to work out which Game Object the Pointer is going to
        * interact with. It works by polling all of the valid game objects, and then slowly discounting those
        * that don't meet the criteria (i.e. they aren't under the Pointer, are disabled, invisible, etc).
        * 
        * Eventually a short-list of 'candidates' is created. These are all of the Game Objects which are valid
        * for input and overlap with the Pointer. If you need fine-grained control over which of the items is
        * selected then you can use this callback to do so.
        * 
        * The callback will be sent 3 parameters:
        * 
        * 1) A reference to the Phaser.Pointer object that is processing the Items.
        * 2) An array containing all potential interactive candidates. This is an array of `InputHandler` objects, not Sprites.
        * 3) The current 'favorite' candidate, based on its priorityID and position in the display list.
        * 
        * Your callback MUST return one of the candidates sent to it.
        * 
        * @param callback The callback that will be called each time `Pointer.processInteractiveObjects` is called. Set to `null` to disable.
        * @param context The context in which the callback will be called.
        */
        setInteractiveCandidateHandler(callback: Function, context?: any): void;

        /**
        * Find the first free Pointer object and start it, passing in the event data.
        * This is called automatically by Phaser.Touch and Phaser.MSPointer.
        * 
        * @param event The event data from the Touch event.
        * @return The Pointer object that was started or null if no Pointer object is available.
        */
        startPointer(event: any): Phaser.Pointer;

        /**
        * Stops the matching Pointer object, passing in the event data.
        * 
        * @param event The event data from the Touch event.
        * @return The Pointer object that was stopped or null if no Pointer object is available.
        */
        stopPointer(event: any): Phaser.Pointer;

        /**
        * Updates the Input Manager. Called by the core Game loop.
        */
        update(): void;

        /**
        * Updates the matching Pointer object, passing in the event data.
        * This is called automatically and should not normally need to be invoked.
        * 
        * @param event The event data from the Touch event.
        * @return The Pointer object that was updated; null if no pointer was updated.
        */
        updatePointer(event: any): Phaser.Pointer;

    }


    /**
    * The Input Handler is bound to a specific Sprite and is responsible for managing all Input events on that Sprite.
    */
    class InputHandler {


        /**
        * The Input Handler is bound to a specific Sprite and is responsible for managing all Input events on that Sprite.
        * 
        * @param sprite The Sprite object to which this Input Handler belongs.
        */
        constructor(sprite: Phaser.Sprite);


        /**
        * Controls if the Sprite is allowed to be dragged horizontally.
        * Default: true
        */
        allowHorizontalDrag: boolean;

        /**
        * Controls if the Sprite is allowed to be dragged vertically.
        * Default: true
        */
        allowVerticalDrag: boolean;

        /**
        * A region of the game world within which the sprite is restricted during drag.
        */
        boundsRect: Phaser.Rectangle;

        /**
        * A Sprite the bounds of which this sprite is restricted during drag.
        */
        boundsSprite: Phaser.Sprite;

        /**
        * If true when this Sprite is clicked or dragged it will automatically be bought to the top of the Group it is within.
        */
        bringToTop: boolean;

        /**
        * A Point object containing the coordinates of the Pointer when it was first pressed down onto this Sprite.
        */
        downPoint: Phaser.Point;

        /**
        * The distance, in pixels, the pointer has to move while being held down, before the Sprite thinks it is being dragged.
        */
        dragDistanceThreshold: number;

        /**
        * The offset from the Sprites position that dragging takes place from.
        */
        dragOffset: Phaser.Point;

        /**
        * Is the Sprite dragged from its center, or the point at which the Pointer was pressed down upon it?
        */
        dragFromCenter: boolean;

        /**
        * Is this sprite allowed to be dragged by the mouse? true = yes, false = no
        */
        draggable: boolean;

        /**
        * The Point from which the most recent drag started from. Useful if you need to return an object to its starting position.
        */
        dragStartPoint: Phaser.Point;

        /**
        * If enabled, when the Sprite stops being dragged, it will only dispatch the `onDragStop` event, and not the `onInputUp` event. If set to `false` it will dispatch both events.
        */
        dragStopBlocksInputUp: boolean;

        /**
        * The amount of time, in ms, the pointer has to be held down over the Sprite before it thinks it is being dragged.
        */
        dragTimeThreshold: number;

        /**
        * If enabled the Input Handler will process input requests and monitor pointer activity.
        */
        enabled: boolean;

        /**
        * A reference to the currently running game.
        */
        game: Phaser.Game;

        /**
        * Warning: EXPERIMENTAL
        * 
        * @param x
        */
        globalToLocalX(x: number): number;

        /**
        * Warning: EXPERIMENTAL
        * 
        * @param y
        */
        globalToLocalY(y: number): number;

        /**
        * true if the Sprite is being currently dragged.
        */
        isDragged: boolean;

        /**
        * The alpha tolerance threshold. If the alpha value of the pixel matches or is above this value, it's considered a hit.
        * Default: 255
        */
        pixelPerfectAlpha: number;

        /**
        * Set to true to use pixel perfect hit detection when checking if the pointer is over this Sprite when it's clicked or touched.
        * The x/y coordinates of the pointer are tested against the image in combination with the InputHandler.pixelPerfectAlpha value.
        * This feature only works for display objects with image based textures such as Sprites. It won't work on BitmapText or Rope.
        * Warning: This is expensive so only enable if you really need it. Use a pixel perfect check when testing for clicks or touches on the Sprite.
        */
        pixelPerfectClick: boolean;

        /**
        * Set to true to use pixel perfect hit detection when checking if the pointer is over this Sprite.
        * The x/y coordinates of the pointer are tested against the image in combination with the InputHandler.pixelPerfectAlpha value.
        * This feature only works for display objects with image based textures such as Sprites. It won't work on BitmapText or Rope.
        * Warning: This is expensive, especially on mobile (where it's not even needed!) so only enable if required. Also see the less-expensive InputHandler.pixelPerfectClick. Use a pixel perfect check when testing for pointer over.
        */
        pixelPerfectOver: boolean;

        /**
        * The priorityID is used to determine which game objects should get priority when input events occur. For example if you have
        * several Sprites that overlap, by default the one at the top of the display list is given priority for input events. You can
        * stop this from happening by controlling the priorityID value. The higher the value, the more important they are considered to the Input events.
        */
        priorityID: number;

        /**
        * EXPERIMENTAL: Please do not use this property unless you know what it does. Likely to change in the future.
        */
        scaleLayer: boolean;

        /**
        * A Point object that contains by how far the Sprite snap is offset.
        */
        snapOffset: Phaser.Point;

        /**
        * This defines the top-left X coordinate of the snap grid.
        */
        snapOffsetX: number;

        /**
        * This defines the top-left Y coordinate of the snap grid..
        */
        snapOffsetY: number;

        /**
        * When the Sprite is dragged this controls if the center of the Sprite will snap to the pointer on drag or not.
        */
        snapOnDrag: boolean;

        /**
        * When the Sprite is dragged this controls if the Sprite will be snapped on release.
        */
        snapOnRelease: boolean;

        /**
        * If the sprite is set to snap while dragging this holds the point of the most recent 'snap' event.
        */
        snapPoint: Phaser.Point;

        /**
        * When a Sprite has snapping enabled this holds the width of the snap grid.
        */
        snapX: number;

        /**
        * When a Sprite has snapping enabled this holds the height of the snap grid.
        */
        snapY: number;

        /**
        * The Sprite object to which this Input Handler belongs.
        */
        sprite: Phaser.Sprite;

        /**
        * On a desktop browser you can set the 'hand' cursor to appear when moving over the Sprite.
        */
        useHandCursor: boolean;


        /**
        * Bounds Rect check for the sprite drag
        */
        checkBoundsRect(): void;

        /**
        * Parent Sprite Bounds check for the sprite drag.
        */
        checkBoundsSprite(): void;

        /**
        * Runs a pixel perfect check against the given x/y coordinates of the Sprite this InputHandler is bound to.
        * It compares the alpha value of the pixel and if >= InputHandler.pixelPerfectAlpha it returns true.
        * 
        * @param x The x coordinate to check.
        * @param y The y coordinate to check.
        * @param pointer The pointer to get the x/y coordinate from if not passed as the first two parameters.
        * @return true if there is the alpha of the pixel is >= InputHandler.pixelPerfectAlpha
        */
        checkPixel(x: number, y: number, pointer?: Phaser.Pointer): boolean;

        /**
        * Checks if the given pointer is both down and over the Sprite this InputHandler belongs to.
        * Use the `fastTest` flag is to quickly check just the bounding hit area even if `InputHandler.pixelPerfectOver` is `true`.
        * 
        * @param pointer
        * @param fastTest Force a simple hit area check even if `pixelPerfectOver` is true for this object?
        * @return True if the pointer is down, otherwise false.
        */
        checkPointerDown(pointer: Phaser.Pointer, fastTest?: boolean): boolean;

        /**
        * Checks if the given pointer is over the Sprite this InputHandler belongs to.
        * Use the `fastTest` flag is to quickly check just the bounding hit area even if `InputHandler.pixelPerfectOver` is `true`.
        * 
        * @param pointer
        * @param fastTest Force a simple hit area check even if `pixelPerfectOver` is true for this object?
        */
        checkPointerOver(pointer: Phaser.Pointer, fastTest?: boolean): boolean;

        /**
        * Clean up memory.
        */
        destroy(): void;

        /**
        * Stops this sprite from being able to be dragged.
        * If it is currently the target of an active drag it will be stopped immediately; also disables any set callbacks.
        */
        disableDrag(): void;

        /**
        * Stops the sprite from snapping to a grid during drag or release.
        */
        disableSnap(): void;

        /**
        * If the pointer is currently over this Sprite this returns how long it has been there for in milliseconds.
        * 
        * @param pointerId
        * @return The number of milliseconds the pointer has been pressed down on the Sprite, or -1 if not over.
        */
        downDuration(pointerId?: number): number;

        /**
        * Allow this Sprite to be dragged by any valid pointer.
        * 
        * When the drag begins the Sprite.events.onDragStart event will be dispatched.
        * 
        * When the drag completes by way of the user letting go of the pointer that was dragging the sprite, the Sprite.events.onDragStop event is dispatched.
        * 
        * You can control the thresholds over when a drag starts via the properties:
        * 
        * `Pointer.dragDistanceThreshold` the distance, in pixels, that the pointer has to move
        * before the drag will start.
        * 
        * `Pointer.dragTimeThreshold` the time, in ms, that the pointer must be held down on
        * the Sprite before the drag will start.
        * 
        * You can set either (or both) of these properties after enabling a Sprite for drag.
        * 
        * For the duration of the drag the Sprite.events.onDragUpdate event is dispatched. This event is only dispatched when the pointer actually
        * changes position and moves. The event sends 5 parameters: `sprite`, `pointer`, `dragX`, `dragY` and `snapPoint`.
        * 
        * @param lockCenter If false the Sprite will drag from where you click it minus the dragOffset. If true it will center itself to the tip of the mouse pointer.
        * @param bringToTop If true the Sprite will be bought to the top of the rendering list in its current Group.
        * @param pixelPerfect If true it will use a pixel perfect test to see if you clicked the Sprite. False uses the bounding box.
        * @param alphaThreshold If using pixel perfect collision this specifies the alpha level from 0 to 255 above which a collision is processed. - Default: 255
        * @param boundsRect If you want to restrict the drag of this sprite to a specific Rectangle, pass the Phaser.Rectangle here, otherwise it's free to drag anywhere.
        * @param boundsSprite If you want to restrict the drag of this sprite to within the bounding box of another sprite, pass it here.
        */
        enableDrag(lockCenter?: boolean, bringToTop?: boolean, pixelPerfect?: boolean, alphaThreshold?: number, boundsRect?: Phaser.Rectangle, boundsSprite?: Phaser.Sprite): void;

        /**
        * Make this Sprite snap to the given grid either during drag or when it's released.
        * For example 16x16 as the snapX and snapY would make the sprite snap to every 16 pixels.
        * 
        * @param snapX The width of the grid cell to snap to.
        * @param snapY The height of the grid cell to snap to.
        * @param onDrag If true the sprite will snap to the grid while being dragged. - Default: true
        * @param onRelease If true the sprite will snap to the grid when released.
        * @param snapOffsetX Used to offset the top-left starting point of the snap grid.
        * @param snapOffsetY Used to offset the top-left starting point of the snap grid.
        */
        enableSnap(snapX: number, snapY: number, onDrag?: boolean, onRelease?: boolean, snapOffsetX?: number, snapOffsetY?: number): void;

        /**
        * Is this object using pixel perfect checking?
        * @return True if the this InputHandler has either `pixelPerfectClick` or `pixelPerfectOver` set to `true`.
        */
        isPixelPerfect(): boolean;

        /**
        * Returns true if the pointer has left the Sprite within the specified delay time (defaults to 500ms, half a second)
        * 
        * @param pointerId
        * @param delay The time below which the pointer is considered as just out.
        */
        justOut(pointerId?: number, delay?: number): boolean;

        /**
        * Returns true if the pointer has entered the Sprite within the specified delay time (defaults to 500ms, half a second)
        * 
        * @param pointerId
        * @param delay The time below which the pointer is considered as just over.
        */
        justOver(pointerId?: number, delay?: number): boolean;

        /**
        * Returns true if the pointer has touched or clicked on the Sprite within the specified delay time (defaults to 500ms, half a second)
        * 
        * @param pointerId
        * @param delay The time below which the pointer is considered as just over.
        */
        justPressed(pointerId?: number, delay?: number): boolean;

        /**
        * Returns true if the pointer was touching this Sprite, but has been released within the specified delay time (defaults to 500ms, half a second)
        * 
        * @param pointerId
        * @param delay The time below which the pointer is considered as just out.
        */
        justReleased(pointerId?: number, delay?: number): boolean;

        /**
        * If the pointer is currently over this Sprite this returns how long it has been there for in milliseconds.
        * 
        * @param pointerId
        * @return The number of milliseconds the pointer has been over the Sprite, or -1 if not over.
        */
        overDuration(pointerId?: number): number;

        /**
        * If the Pointer is down this returns true.
        * This *only* checks if the Pointer is down, not if it's down over any specific Sprite.
        * 
        * @param pointerId
        * @return - True if the given pointer is down, otherwise false.
        */
        pointerDown(pointerId?: number): boolean;

        /**
        * Is this sprite being dragged by the mouse or not?
        * 
        * @param pointerId
        * @return True if the pointer is dragging an object, otherwise false.
        */
        pointerDragged(pointerId?: number): boolean;

        /**
        * Is the Pointer outside of this Sprite?
        * 
        * @param pointerId The ID number of a Pointer to check. If you don't provide a number it will check all Pointers. - Default: (check all)
        * @return True if the given pointer (if a index was given, or any pointer if not) is out of this object.
        */
        pointerOut(pointerId?: number): boolean;

        /**
        * Is the Pointer over this Sprite?
        * 
        * @param pointerId The ID number of a Pointer to check. If you don't provide a number it will check all Pointers. - Default: (check all)
        * @return - True if the given pointer (if a index was given, or any pointer if not) is over this object.
        */
        pointerOver(pointerId?: number): boolean;

        /**
        * A timestamp representing when the Pointer first touched the touchscreen.
        * 
        * @param pointerId  - Default: (check all)
        */
        pointerTimeDown(pointerId?: number): number;

        /**
        * A timestamp representing when the Pointer left the touchscreen.
        * 
        * @param pointerId
        */
        pointerTimeOut(pointerId?: number): number;

        /**
        * A timestamp representing when the Pointer first touched the touchscreen.
        * 
        * @param pointerId
        */
        pointerTimeOver(pointerId?: number): number;

        /**
        * A timestamp representing when the Pointer left the touchscreen.
        * 
        * @param pointerId
        */
        pointerTimeUp(pointerId?: number): number;

        /**
        * If the Pointer is up this returns true.
        * This *only* checks if the Pointer is up, not if it's up over any specific Sprite.
        * 
        * @param pointerId
        * @return - True if the given pointer is up, otherwise false.
        */
        pointerUp(pointerId?: number): boolean;

        /**
        * The x coordinate of the Input pointer, relative to the top-left of the parent Sprite.
        * This value is only set when the pointer is over this Sprite.
        * 
        * @param pointerId
        * @return The x coordinate of the Input pointer.
        */
        pointerX(pointerId?: number): number;

        /**
        * The y coordinate of the Input pointer, relative to the top-left of the parent Sprite
        * This value is only set when the pointer is over this Sprite.
        * 
        * @param pointerId
        * @return The y coordinate of the Input pointer.
        */
        pointerY(pointerId?: number): number;

        /**
        * Resets the Input Handler and disables it.
        */
        reset(): void;

        /**
        * Restricts this sprite to drag movement only on the given axis. Note: If both are set to false the sprite will never move!
        * 
        * @param allowHorizontal To enable the sprite to be dragged horizontally set to true, otherwise false. - Default: true
        * @param allowVertical To enable the sprite to be dragged vertically set to true, otherwise false. - Default: true
        */
        setDragLock(allowHorizontal?: boolean, allowVertical?: boolean): void;

        /**
        * Starts the Input Handler running. This is called automatically when you enable input on a Sprite, or can be called directly if you need to set a specific priority.
        * 
        * @param priority Higher priority sprites take click priority over low-priority sprites when they are stacked on-top of each other.
        * @param useHandCursor If true the Sprite will show the hand cursor on mouse-over (doesn't apply to mobile browsers)
        * @return The Sprite object to which the Input Handler is bound.
        */
        start(priority: number, useHandCursor: boolean): Phaser.Sprite;

        /**
        * Called by Pointer when drag starts on this Sprite. Should not usually be called directly.
        * 
        * @param pointer
        */
        startDrag(pointer: Phaser.Pointer): void;

        /**
        * Stops the Input Handler from running.
        */
        stop(): void;

        /**
        * Called by Pointer when drag is stopped on this Sprite. Should not usually be called directly.
        * 
        * @param pointer
        */
        stopDrag(pointer: Phaser.Pointer): void;

        /**
        * Internal Update method. This is called automatically and handles the Pointer
        * and drag update loops.
        * 
        * @param pointer
        * @return True if the pointer is still active, otherwise false.
        */
        update(pointer: Phaser.Pointer): void;

        /**
        * Called as a Pointer actively drags this Game Object.
        * 
        * @param pointer The Pointer causing the drag update.
        * @param fromStart True if this is the first update, immediately after the drag has started.
        */
        updateDrag(pointer: Phaser.Pointer): boolean;

        /**
        * Checks if the object this InputHandler is bound to is valid for consideration in the Pointer move event.
        * This is called by Phaser.Pointer and shouldn't typically be called directly.
        * 
        * @param highestID The highest ID currently processed by the Pointer.
        * @param highestRenderID The highest Render Order ID currently processed by the Pointer.
        * @param includePixelPerfect If this object has `pixelPerfectClick` or `pixelPerfectOver` set should it be considered as valid? - Default: true
        * @return True if the object this InputHandler is bound to should be considered as valid for input detection.
        */
        validForInput(highestID: number, highestRenderID: number, includePixelPerfect?: boolean): boolean;

    }


    /**
    * If you need more fine-grained control over the handling of specific keys you can create and use Phaser.Key objects.
    */
    class Key {


        /**
        * If you need more fine-grained control over the handling of specific keys you can create and use Phaser.Key objects.
        * 
        * @param game Current game instance.
        * @param keycode The key code this Key is responsible for. See {@link Phaser.KeyCode}.
        */
        constructor(game: Phaser.Game, keycode: number);


        /**
        * The down state of the ALT key, if pressed at the same time as this key.
        */
        altKey: boolean;

        /**
        * The down state of the CTRL key, if pressed at the same time as this key.
        */
        ctrlKey: boolean;

        /**
        * If the key is down this value holds the duration of that key press and is constantly updated.
        * If the key is up it holds the duration of the previous down session. The number of milliseconds this key has been held down for.
        */
        duration: number;

        /**
        * An enabled key processes its update and dispatches events.
        * A key can be disabled momentarily at runtime instead of deleting it.
        * Default: true
        */
        enabled: boolean;

        /**
        * Stores the most recent DOM event.
        */
        event: any;

        /**
        * A reference to the currently running game.
        */
        game: Phaser.Game;

        /**
        * The "down" state of the key. This will remain `true` for as long as the keyboard thinks this key is held down.
        */
        isDown: boolean;

        /**
        * The "up" state of the key. This will remain `true` for as long as the keyboard thinks this key is up.
        * Default: true
        */
        isUp: boolean;

        /**
        * True if the key has just been pressed (NOTE: requires to be reset, see justDown getter)
        */
        _justDown: boolean;

        /**
        * The justDown value allows you to test if this Key has just been pressed down or not.
        * When you check this value it will return `true` if the Key is down, otherwise `false`.
        * You can only call justDown once per key press. It will only return `true` once, until the Key is released and pressed down again.
        * This allows you to use it in situations where you want to check if this key is down without using a Signal, such as in a core game loop.
        * Default: false
        */
        justDown: boolean;

        /**
        * True if the key has just been pressed (NOTE: requires to be reset, see justDown getter)
        */
        _justUp: boolean;

        /**
        * The justUp value allows you to test if this Key has just been released or not.
        * When you check this value it will return `true` if the Key is up, otherwise `false`.
        * You can only call justUp once per key release. It will only return `true` once, until the Key is pressed down and released again.
        * This allows you to use it in situations where you want to check if this key is up without using a Signal, such as in a core game loop.
        * Default: false
        */
        justUp: boolean;

        /**
        * The keycode of this key.
        */
        keyCode: number;

        /**
        * This Signal is dispatched every time this Key is pressed down. It is only dispatched once (until the key is released again).
        */
        onDown: Phaser.Signal;

        /**
        * A callback that is called while this Key is held down. Warning: Depending on refresh rate that could be 60+ times per second.
        */
        onHoldCallback: Function;

        /**
        * The context under which the onHoldCallback will be called.
        */
        onHoldContext: any;

        /**
        * This Signal is dispatched every time this Key is released. It is only dispatched once (until the key is pressed and released again).
        */
        onUp: Phaser.Signal;

        /**
        * If a key is held down this holds down the number of times the key has 'repeated'.
        */
        repeats: number;

        /**
        * The down state of the SHIFT key, if pressed at the same time as this key.
        */
        shiftKey: boolean;

        /**
        * The timestamp when the key was last pressed down. This is based on Game.time.now.
        */
        timeDown: number;

        /**
        * The timestamp when the key was last released. This is based on Game.time.now.
        */
        timeUp: number;


        /**
        * Returns `true` if the Key was pressed down within the `duration` value given, or `false` if it either isn't down,
        * or was pressed down longer ago than then given duration.
        * 
        * @param duration The duration within which the key is considered as being just pressed. Given in ms. - Default: 50
        * @return True if the key was pressed down within the given duration.
        */
        downDuration(duration?: number): boolean;

        /**
        * Called automatically by Phaser.Keyboard.
        * 
        * @param event The DOM event that triggered this.
        */
        processKeyDown(event: KeyboardEvent): void;

        /**
        * Called automatically by Phaser.Keyboard.
        * 
        * @param event The DOM event that triggered this.
        */
        processKeyUp(event: KeyboardEvent): void;

        /**
        * Resets the state of this Key.
        * 
        * This sets isDown to false, isUp to true, resets the time to be the current time, and _enables_ the key.
        * In addition, if it is a "hard reset", it clears clears any callbacks associated with the onDown and onUp events and removes the onHoldCallback.
        * 
        * @param hard A soft reset won't reset any events or callbacks; a hard reset will. - Default: true
        */
        reset(hard?: boolean): void;

        /**
        * Called automatically by Phaser.Keyboard.
        */
        update(): void;

        /**
        * Returns `true` if the Key was pressed down within the `duration` value given, or `false` if it either isn't down,
        * or was pressed down longer ago than then given duration.
        * 
        * @param duration The duration within which the key is considered as being just released. Given in ms. - Default: 50
        * @return True if the key was released within the given duration.
        */
        upDuration(duration?: number): boolean;

    }


    /**
    * The Keyboard class monitors keyboard input and dispatches keyboard events.
    * 
    * _Note_: many keyboards are unable to process certain combinations of keys due to hardware limitations known as ghosting.
    * See http://www.html5gamedevs.com/topic/4876-impossible-to-use-more-than-2-keyboard-input-buttons-at-the-same-time/ for more details.
    * 
    * Also please be aware that certain browser extensions can disable or override Phaser keyboard handling.
    * For example the Chrome extension vimium is known to disable Phaser from using the D key. And there are others.
    * So please check your extensions before opening Phaser issues.
    */
    class Keyboard {


        /**
        * The Keyboard class monitors keyboard input and dispatches keyboard events.
        * 
        * _Note_: many keyboards are unable to process certain combinations of keys due to hardware limitations known as ghosting.
        * See http://www.html5gamedevs.com/topic/4876-impossible-to-use-more-than-2-keyboard-input-buttons-at-the-same-time/ for more details.
        * 
        * Also please be aware that certain browser extensions can disable or override Phaser keyboard handling.
        * For example the Chrome extension vimium is known to disable Phaser from using the D key. And there are others.
        * So please check your extensions before opening Phaser issues.
        * 
        * @param game A reference to the currently running game.
        */
        constructor(game: Phaser.Game);

        static A: number;
        static B: number;
        static C: number;
        static D: number;
        static E: number;
        static F: number;
        static G: number;
        static H: number;
        static I: number;
        static J: number;
        static K: number;
        static L: number;
        static M: number;
        static N: number;
        static O: number;
        static P: number;
        static Q: number;
        static R: number;
        static S: number;
        static T: number;
        static U: number;
        static V: number;
        static W: number;
        static X: number;
        static Y: number;
        static Z: number;
        static ZERO: number;
        static ONE: number;
        static TWO: number;
        static THREE: number;
        static FOUR: number;
        static FIVE: number;
        static SIX: number;
        static SEVEN: number;
        static EIGHT: number;
        static NINE: number;
        static NUMPAD_0: number;
        static NUMPAD_1: number;
        static NUMPAD_2: number;
        static NUMPAD_3: number;
        static NUMPAD_4: number;
        static NUMPAD_5: number;
        static NUMPAD_6: number;
        static NUMPAD_7: number;
        static NUMPAD_8: number;
        static NUMPAD_9: number;
        static NUMPAD_MULTIPLY: number;
        static NUMPAD_ADD: number;
        static NUMPAD_ENTER: number;
        static NUMPAD_SUBTRACT: number;
        static NUMPAD_DECIMAL: number;
        static NUMPAD_DIVIDE: number;
        static F1: number;
        static F2: number;
        static F3: number;
        static F4: number;
        static F5: number;
        static F6: number;
        static F7: number;
        static F8: number;
        static F9: number;
        static F10: number;
        static F11: number;
        static F12: number;
        static F13: number;
        static F14: number;
        static F15: number;
        static COLON: number;
        static EQUALS: number;
        static COMMA: number;
        static UNDERSCORE: number;
        static PERIOD: number;
        static QUESTION_MARK: number;
        static TILDE: number;
        static OPEN_BRACKET: number;
        static BACKWARD_SLASH: number;
        static CLOSED_BRACKET: number;
        static QUOTES: number;
        static BACKSPACE: number;
        static TAB: number;
        static CLEAR: number;
        static ENTER: number;
        static SHIFT: number;
        static CONTROL: number;
        static ALT: number;
        static CAPS_LOCK: number;
        static ESC: number;
        static SPACEBAR: number;
        static PAGE_UP: number;
        static PAGE_DOWN: number;
        static END: number;
        static HOME: number;
        static LEFT: number;
        static UP: number;
        static RIGHT: number;
        static DOWN: number;
        static INSERT: number;
        static DELETE: number;
        static HELP: number;
        static NUM_LOCK: number;
        static PLUS: number;
        static MINUS: number;


        /**
        * The context under which the callbacks are run.
        */
        callbackContext: any;

        /**
        * Keyboard input will only be processed if enabled.
        * Default: true
        */
        enabled: boolean;

        /**
        * The most recent DOM event from keydown or keyup. This is updated every time a new key is pressed or released.
        */
        event: any;

        /**
        * Local reference to game.
        */
        game: Phaser.Game;

        /**
        * Returns the string value of the most recently pressed key.
        */
        lastChar: string;

        /**
        * Returns the most recently pressed Key. This is a Phaser.Key object and it changes every time a key is pressed.
        */
        lastKey: Phaser.Key;

        /**
        * This callback is invoked every time a key is pressed down, including key repeats when a key is held down.
        */
        onDownCallback: Function;

        /**
        * This callback is invoked every time a DOM onkeypress event is raised, which is only for printable keys.
        */
        onPressCallback: Function;

        /**
        * This callback is invoked every time a key is released.
        */
        onUpCallback: Function;

        /**
        * The most recent DOM event from keypress.
        */
        pressEvent: any;


        /**
        * Add callbacks to the Keyboard handler so that each time a key is pressed down or released the callbacks are activated.
        * 
        * @param context The context under which the callbacks are run.
        * @param onDown This callback is invoked every time a key is pressed down.
        * @param onUp This callback is invoked every time a key is released.
        * @param onPress This callback is invoked every time the onkeypress event is raised.
        */
        addCallbacks(context: any, onDown?: Function, onUp?: Function, onPress?: Function): void;

        /**
        * If you need more fine-grained control over a Key you can create a new Phaser.Key object via this method.
        * The Key object can then be polled, have events attached to it, etc.
        * 
        * @param keycode The {@link Phaser.KeyCode keycode} of the key.
        * @return The Key object which you can store locally and reference directly.
        */
        addKey(keycode: number): Phaser.Key;

        /**
        * A practical way to create an object containing user selected hotkeys.
        * 
        * For example,
        * 
        *     addKeys( { 'up': Phaser.KeyCode.W, 'down': Phaser.KeyCode.S, 'left': Phaser.KeyCode.A, 'right': Phaser.KeyCode.D } );
        * 
        * would return an object containing properties (`up`, `down`, `left` and `right`) referring to {@link Phaser.Key} object.
        * 
        * @param keys A key mapping object, i.e. `{ 'up': Phaser.KeyCode.W, 'down': Phaser.KeyCode.S }` or `{ 'up': 52, 'down': 53 }`.
        * @return An object containing the properties mapped to {@link Phaser.Key} values.
        */
        addKeys(keys: any): any;

        /**
        * By default when a key is pressed Phaser will not stop the event from propagating up to the browser.
        * There are some keys this can be annoying for, like the arrow keys or space bar, which make the browser window scroll.
        * 
        * The `addKeyCapture` method enables consuming keyboard event for specific keys so it doesn't bubble up to the the browser
        * and cause the default browser behavior.
        * 
        * Pass in either a single keycode or an array/hash of keycodes.
        * 
        * @param keycode Either a single {@link Phaser.KeyCode keycode} or an array/hash of keycodes such as `[65, 67, 68]`.
        */
        addKeyCapture(keycode: any): void;

        /**
        * Creates and returns an object containing 4 hotkeys for Up, Down, Left and Right.
        * @return An object containing properties: `up`, `down`, `left` and `right` of {@link Phaser.Key} objects.
        */
        createCursorKeys(): Phaser.CursorKeys;

        /**
        * Clear all set key captures.
        */
        clearCaptures(): void;

        /**
        * Stops the Keyboard event listeners from running (keydown and keyup). They are removed from the window.
        * Also clears all key captures and currently created Key objects.
        */
        destroy(): void;

        /**
        * Returns `true` if the Key was pressed down within the `duration` value given, or `false` if it either isn't down,
        * or was pressed down longer ago than then given duration.
        * 
        * @param keycode The {@link Phaser.KeyCode keycode} of the key to check: i.e. Phaser.KeyCode.UP or Phaser.KeyCode.SPACEBAR.
        * @param duration The duration within which the key is considered as being just pressed. Given in ms. - Default: 50
        * @return True if the key was pressed down within the given duration, false if not or null if the Key wasn't found.
        */
        downDuration(keycode: number, duration?: number): boolean;

        /**
        * Returns true of the key is currently pressed down. Note that it can only detect key presses on the web browser.
        * 
        * @param keycode The {@link Phaser.KeyCode keycode} of the key to check: i.e. Phaser.KeyCode.UP or Phaser.KeyCode.SPACEBAR.
        * @return True if the key is currently down, false if not or null if the Key wasn't found.
        */
        isDown(keycode: number): boolean;

        /**
        * Process the keydown event.
        * 
        * @param event
        */
        processKeyDown(event: KeyboardEvent): void;

        /**
        * Process the keypress event.
        * 
        * @param event
        */
        processKeyPress(event: KeyboardEvent): void;

        /**
        * Process the keyup event.
        * 
        * @param event
        */
        processKeyUp(event: KeyboardEvent): void;

        /**
        * Removes a Key object from the Keyboard manager.
        * 
        * @param keycode The {@link Phaser.KeyCode keycode} of the key to remove.
        */
        removeKey(keycode: number): void;

        /**
        * Removes an existing key capture.
        * 
        * @param keycode The {@link Phaser.KeyCode keycode} to remove capturing of.
        */
        removeKeyCapture(keycode: number): void;

        /**
        * Resets all Keys.
        * 
        * @param hard A soft reset won't reset any events or callbacks that are bound to the Keys. A hard reset will. - Default: true
        */
        reset(hard?: boolean): void;

        /**
        * Starts the Keyboard event listeners running (keydown, keyup and keypress). They are attached to the window.
        * This is called automatically by Phaser.Input and should not normally be invoked directly.
        */
        start(): void;

        /**
        * Stops the Keyboard event listeners from running (keydown, keyup and keypress). They are removed from the window.
        */
        stop(): void;

        /**
        * Updates all currently defined keys.
        */
        update(): void;

        /**
        * Returns `true` if the Key was pressed down within the `duration` value given, or `false` if it either isn't down,
        * or was pressed down longer ago than then given duration.
        * 
        * @param keycode The keycode of the key to check, i.e. Phaser.KeyCode.UP or Phaser.KeyCode.SPACEBAR.
        * @param duration The duration within which the key is considered as being just released. Given in ms. - Default: 50
        * @return True if the key was released within the given duration, false if not or null if the Key wasn't found.
        */
        upDuration(keycode: number, duration?: number): boolean;

    }


    /**
    * A key code represents a physical key on a keyboard.
    * 
    * The KeyCode class contains commonly supported keyboard key codes which can be used
    * as keycode`-parameters in several {@link Phaser.Keyboard} and {@link Phaser.Key} methods.
    * 
    * _Note_: These values should only be used indirectly, eg. as `Phaser.KeyCode.KEY`.
    * Future versions may replace the actual values, such that they remain compatible with `keycode`-parameters.
    * The current implementation maps to the {@link https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/keyCode KeyboardEvent.keyCode} property.
    * 
    * _Note_: Use `Phaser.KeyCode.KEY` instead of `Phaser.Keyboard.KEY` to refer to a key code;
    * the latter approach is supported for compatibility.
    */
    class KeyCode {

        static A: number;
        static B: number;
        static C: number;
        static D: number;
        static E: number;
        static F: number;
        static G: number;
        static H: number;
        static I: number;
        static J: number;
        static K: number;
        static L: number;
        static M: number;
        static N: number;
        static O: number;
        static P: number;
        static Q: number;
        static R: number;
        static S: number;
        static T: number;
        static U: number;
        static V: number;
        static W: number;
        static X: number;
        static Y: number;
        static Z: number;
        static ZERO: number;
        static ONE: number;
        static TWO: number;
        static THREE: number;
        static FOUR: number;
        static FIVE: number;
        static SIX: number;
        static SEVEN: number;
        static EIGHT: number;
        static NINE: number;
        static NUMPAD_0: number;
        static NUMPAD_1: number;
        static NUMPAD_2: number;
        static NUMPAD_3: number;
        static NUMPAD_4: number;
        static NUMPAD_5: number;
        static NUMPAD_6: number;
        static NUMPAD_7: number;
        static NUMPAD_8: number;
        static NUMPAD_9: number;
        static NUMPAD_MULTIPLY: number;
        static NUMPAD_ADD: number;
        static NUMPAD_ENTER: number;
        static NUMPAD_SUBTRACT: number;
        static NUMPAD_DECIMAL: number;
        static NUMPAD_DIVIDE: number;
        static F1: number;
        static F2: number;
        static F3: number;
        static F4: number;
        static F5: number;
        static F6: number;
        static F7: number;
        static F8: number;
        static F9: number;
        static F10: number;
        static F11: number;
        static F12: number;
        static F13: number;
        static F14: number;
        static F15: number;
        static COLON: number;
        static EQUALS: number;
        static COMMA: number;
        static UNDERSCORE: number;
        static PERIOD: number;
        static QUESTION_MARK: number;
        static TILDE: number;
        static OPEN_BRACKET: number;
        static BACKWARD_SLASH: number;
        static CLOSED_BRACKET: number;
        static QUOTES: number;
        static BACKSPACE: number;
        static TAB: number;
        static CLEAR: number;
        static ENTER: number;
        static SHIFT: number;
        static CONTROL: number;
        static ALT: number;
        static CAPS_LOCK: number;
        static ESC: number;
        static SPACEBAR: number;
        static PAGE_UP: number;
        static PAGE_DOWN: number;
        static END: number;
        static HOME: number;
        static LEFT: number;
        static UP: number;
        static RIGHT: number;
        static DOWN: number;
        static INSERT: number;
        static DELETE: number;
        static HELP: number;
        static NUM_LOCK: number;
        static PLUS: number;
        static MINUS: number;

    }


    /**
    * Creates a new Line object with a start and an end point.
    */
    class Line {


        /**
        * Creates a new Line object with a start and an end point.
        * 
        * @param x1 The x coordinate of the start of the line.
        * @param y1 The y coordinate of the start of the line.
        * @param x2 The x coordinate of the end of the line.
        * @param y2 The y coordinate of the end of the line.
        */
        constructor(x1?: number, y1?: number, x2?: number, y2?: number);


        /**
        * Gets the angle of the line in radians.
        */
        angle: number;

        /**
        * The end point of the line.
        */
        end: Phaser.Point;

        /**
        * Gets the height of this bounds of this line.
        */
        height: number;

        /**
        * Gets the left-most point of this line.
        */
        left: number;

        /**
        * Gets the length of the line segment.
        */
        length: number;

        /**
        * Gets the angle in radians of the normal of this line (line.angle - 90 degrees.)
        */
        normalAngle: number;

        /**
        * Gets the x component of the left-hand normal of this line.
        */
        normalX: number;

        /**
        * Gets the y component of the left-hand normal of this line.
        */
        normalY: number;

        /**
        * Gets the perpendicular slope of the line (x/y).
        */
        perpSlope: number;

        /**
        * Gets the right-most point of this line.
        */
        right: number;

        /**
        * Gets the slope of the line (y/x).
        */
        slope: number;

        /**
        * The start point of the line.
        */
        start: Phaser.Point;

        /**
        * Gets the top-most point of this line.
        */
        top: number;

        /**
        * The const type of this object.
        */
        type: number;

        /**
        * Gets the width of this bounds of this line.
        */
        width: number;

        /**
        * Gets the x coordinate of the top left of the bounds around this line.
        */
        x: number;

        /**
        * Gets the y coordinate of the top left of the bounds around this line.
        */
        y: number;


        /**
        * Checks for intersection between two lines as defined by the given start and end points.
        * If asSegment is true it will check for line segment intersection. If asSegment is false it will check for line intersection.
        * Returns the intersection segment of AB and EF as a Point, or null if there is no intersection.
        * Adapted from code by Keith Hair
        * 
        * @param a The start of the first Line to be checked.
        * @param b The end of the first line to be checked.
        * @param e The start of the second Line to be checked.
        * @param f The end of the second line to be checked.
        * @param asSegment If true it will check for segment intersection, otherwise full line intersection. - Default: true
        * @param result A Point object to store the result in, if not given a new one will be created.
        * @return The intersection segment of the two lines as a Point, or null if there is no intersection.
        */
        static intersectsPoints(a: Phaser.Point, b: Phaser.Point, e: Phaser.Point, f: Phaser.Point, asSegment?: boolean, result?: Phaser.Point): Phaser.Point;

        /**
        * Checks for intersection between this line and another Line.
        * If asSegment is true it will check for segment intersection. If asSegment is false it will check for line intersection.
        * Returns the intersection segment of AB and EF as a Point, or null if there is no intersection.
        * 
        * @param line The line to check against this one.
        * @param asSegment If true it will check for segment intersection, otherwise full line intersection. - Default: true
        * @param result A Point object to store the result in, if not given a new one will be created.
        * @return The intersection segment of the two lines as a Point, or null if there is no intersection.
        */
        static intersects(a: Phaser.Line, b: Phaser.Line, asSegment?: boolean, result?: Phaser.Point): Phaser.Point;

        /**
        * Checks for intersection between the Line and a Rectangle shape, or a rectangle-like
        * object, with public `x`, `y`, `right` and `bottom` properties, such as a Sprite or Body.
        * 
        * An intersection is considered valid if:
        * 
        * The line starts within or ends within the rectangle; or
        * The line segment intersects one of the 4 rectangle edges; and
        * The line has a non-zero length; and
        * The rectangle is not empty.
        * 
        * For the purposes of this function rectangles are considered 'solid'.
        * 
        * @param line The line to check for intersection with.
        * @param rect The rectangle, or rectangle-like object, to check for intersection with.
        * @return True if the line intersects with the rectangle edges, or starts or ends within the rectangle.
        */
        static intersectsRectangle(line: Phaser.Line, rect: Phaser.Rectangle): boolean;

        /**
        * Returns the reflected angle between two lines.
        * This is the outgoing angle based on the angle of this line and the normalAngle of the given line.
        * 
        * @param line The line to reflect off this line.
        * @return The reflected angle in radians.
        */
        static reflect(a: Phaser.Line, b: Phaser.Line): number;


        /**
        * Centers this Line on the given coordinates.
        * 
        * The line is centered by positioning the start and end points so that the lines midpoint matches
        * the coordinates given.
        * 
        * @param x The x position to center the line on.
        * @param y The y position to center the line on.
        * @return This line object
        */
        centerOn(x: number, y: number): Phaser.Line;

        /**
        * Returns a new Line object with the same values for the start and end properties as this Line object.
        * 
        * @param output Optional Line object. If given the values will be set into the object, otherwise a brand new Line object will be created and returned.
        * @return The cloned Line object.
        */
        clone(output: Phaser.Line): Phaser.Line;

        /**
        * Using Bresenham's line algorithm this will return an array of all coordinates on this line.
        * The start and end points are rounded before this runs as the algorithm works on integers.
        * 
        * @param stepRate How many steps will we return? 1 = every coordinate on the line, 2 = every other coordinate, etc. - Default: 1
        * @param results The array to store the results in. If not provided a new one will be generated.
        * @return An array of coordinates.
        */
        coordinatesOnLine(stepRate: number, results: any[]): any[];

        /**
        * Sets this line to start at the given `x` and `y` coordinates and for the segment to extend at `angle` for the given `length`.
        * 
        * @param x The x coordinate of the start of the line.
        * @param y The y coordinate of the start of the line.
        * @param angle The angle of the line in radians.
        * @param length The length of the line in pixels.
        * @return This line object
        */
        fromAngle(x: number, y: number, angle: number, length: number): Phaser.Line;

        /**
        * Sets the line to match the x/y coordinates of the two given sprites.
        * Can optionally be calculated from their center coordinates.
        * 
        * @param startSprite The coordinates of this Sprite will be set to the Line.start point.
        * @param endSprite The coordinates of this Sprite will be set to the Line.start point.
        * @param useCenter If true it will use startSprite.centerX, if false startSprite.x.
        * @return This line object
        */
        fromSprite(startSprite: Phaser.Sprite, endSprite: Phaser.Sprite, useCenter?: boolean): Phaser.Line;

        /**
        * Checks for intersection between this line and another Line.
        * If asSegment is true it will check for segment intersection. If asSegment is false it will check for line intersection.
        * Returns the intersection segment of AB and EF as a Point, or null if there is no intersection.
        * 
        * @param line The line to check against this one.
        * @param asSegment If true it will check for segment intersection, otherwise full line intersection. - Default: true
        * @param result A Point object to store the result in, if not given a new one will be created.
        * @return The intersection segment of the two lines as a Point, or null if there is no intersection.
        */
        intersects(line: Phaser.Line, asSegment?: boolean, result?: Phaser.Point): Phaser.Point;

        /**
        * Returns a Point object where the x and y values correspond to the center (or midpoint) of the Line segment.
        * 
        * @param out A Phaser.Point object into which the result will be populated. If not given a new Point object is created.
        * @return A Phaser.Point object with the x and y values set to the center of the line segment.
        */
        midPoint(out?: Phaser.Point): Phaser.Point;

        /**
        * Tests if the given coordinates fall on this line. See pointOnSegment to test against just the line segment.
        * 
        * @param x The line to check against this one.
        * @param y The line to check against this one.
        * @return True if the point is on the line, false if not.
        */
        pointOnLine(x: number, y: number): boolean;

        /**
        * Tests if the given coordinates fall on this line and within the segment. See pointOnLine to test against just the line.
        * 
        * @param x The line to check against this one.
        * @param y The line to check against this one.
        * @return True if the point is on the line and segment, false if not.
        */
        pointOnSegment(x: number, y: number): boolean;

        /**
        * Picks a random point from anywhere on the Line segment and returns it.
        * 
        * @param out A Phaser.Point, or any object with public x/y properties, that the values will be set in.
        *            If no object is provided a new Phaser.Point object will be created. In high performance areas avoid this by re-using an object.
        * @return An object containing the random point in its `x` and `y` properties.
        */
        random(out?: Phaser.Point): Phaser.Point;

        /**
        * Returns the reflected angle between two lines.
        * This is the outgoing angle based on the angle of this line and the normalAngle of the given line.
        * 
        * @param line The line to reflect off this line.
        * @return The reflected angle in radians.
        */
        reflect(line: Phaser.Line): number;

        /**
        * Rotates the line by the amount specified in `angle`.
        * 
        * Rotation takes place from the center of the line.
        * If you wish to rotate around a different point see Line.rotateAround.
        * 
        * If you wish to rotate the ends of the Line then see Line.start.rotate or Line.end.rotate.
        * 
        * @param angle The angle in radians (unless asDegrees is true) to rotate the line by.
        * @param asDegrees Is the given angle in radians (false) or degrees (true)?
        * @return This line object
        */
        rotate(angle: number, asDegrees?: boolean): Phaser.Line;

        /**
        * Rotates the line by the amount specified in `angle`.
        * 
        * Rotation takes place around the coordinates given.
        * 
        * @param x The x coordinate to offset the rotation from.
        * @param y The y coordinate to offset the rotation from.
        * @param angle The angle in radians (unless asDegrees is true) to rotate the line by.
        * @param asDegrees Is the given angle in radians (false) or degrees (true)?
        * @return This line object
        */
        rotateAround(x: number, y: number, angle: number, asDegrees?: boolean): Phaser.Line;

        /**
        * Sets the components of the Line to the specified values.
        * 
        * @param x1 The x coordinate of the start of the line.
        * @param y1 The y coordinate of the start of the line.
        * @param x2 The x coordinate of the end of the line.
        * @param y2 The y coordinate of the end of the line.
        * @return This line object
        */
        setTo(x1?: number, y1?: number, x2?: number, y2?: number): Phaser.Line;

    }


    /**
    * A basic Linked List data structure.
    * 
    * This implementation _modifies_ the `prev` and `next` properties of each item added:
    * - The `prev` and `next` properties must be writable and should not be used for any other purpose.
    * - Items _cannot_ be added to multiple LinkedLists at the same time.
    * - Only objects can be added.
    */
    class LinkedList {


        /**
        * First element in the list.
        */
        first: any;

        /**
        * Last element in the list.
        */
        last: any;

        /**
        * Next element in the list.
        */
        next: any;

        /**
        * Previous element in the list.
        */
        prev: any;

        /**
        * Number of elements in the list.
        */
        total: number;


        /**
        * Adds a new element to this linked list.
        * 
        * @param item The element to add to this list. Can be a Phaser.Sprite or any other object you need to quickly iterate through.
        * @return The item that was added.
        */
        add(item: any): any;

        /**
        * Calls a function on all members of this list, using the member as the context for the callback.
        * The function must exist on the member.
        * 
        * @param callback The function to call.
        */
        callAll(callback: Function): void;

        /**
        * Removes the given element from this linked list if it exists.
        * 
        * @param item The item to be removed from the list.
        */
        remove(item: any): void;

        /**
        * Resets the first, last, next and previous node pointers in this list.
        */
        reset(): void;

    }


    /**
    * The Loader handles loading all external content such as Images, Sounds, Texture Atlases and data files.
    * 
    * The loader uses a combination of tag loading (eg. Image elements) and XHR and provides progress and completion callbacks.
    * 
    * Parallel loading (see {@link Phaser.Loader#enableParallel enableParallel}) is supported and enabled by default.
    * Load-before behavior of parallel resources is controlled by synchronization points as discussed in {@link Phaser.Loader#withSyncPoint withSyncPoint}.
    * 
    * Texture Atlases can be created with tools such as [Texture Packer](https://www.codeandweb.com/texturepacker/phaser) and
    * [Shoebox](http://renderhjs.net/shoebox/)
    */
    class Loader {


        /**
        * The Loader handles loading all external content such as Images, Sounds, Texture Atlases and data files.
        * 
        * The loader uses a combination of tag loading (eg. Image elements) and XHR and provides progress and completion callbacks.
        * 
        * Parallel loading (see {@link Phaser.Loader#enableParallel enableParallel}) is supported and enabled by default.
        * Load-before behavior of parallel resources is controlled by synchronization points as discussed in {@link Phaser.Loader#withSyncPoint withSyncPoint}.
        * 
        * Texture Atlases can be created with tools such as [Texture Packer](https://www.codeandweb.com/texturepacker/phaser) and
        * [Shoebox](http://renderhjs.net/shoebox/)
        * 
        * @param game A reference to the currently running game.
        */
        constructor(game: Phaser.Game);

        static PHYSICS_LIME_CORONA_JSON: number;
        static PHYSICS_PHASER_JSON: number;
        static TEXTURE_ATLAS_JSON_ARRAY: number;
        static TEXTURE_ATLAS_JSON_HASH: number;
        static TEXTURE_ATLAS_XML_STARLING: number;
        static TEXTURE_ATLAS_JSON_PYXEL: number;


        /**
        * If you want to append a URL before the path of any asset you can set this here.
        * Useful if allowing the asset base url to be configured outside of the game code.
        * The string _must_ end with a "/".
        */
        baseURL: string;

        /**
        * Local reference to the Phaser.Cache.
        */
        cache: Phaser.Cache;

        /**
        * The crossOrigin value applied to loaded images. Very often this needs to be set to 'anonymous'.
        */
        crossOrigin: boolean | string;

        /**
        * If true (the default) then parallel downloading will be enabled.
        * 
        * To disable all parallel downloads this must be set to false prior to any resource being loaded.
        */
        enableParallel: boolean;

        /**
        * Local reference to game.
        */
        game: Phaser.Game;

        /**
        * True if all assets in the queue have finished loading.
        */
        hasLoaded: boolean;

        /**
        * Used to map the application mime-types to to the Accept header in XHR requests.
        * If you don't require these mappings, or they cause problems on your server, then
        * remove them from the headers object and the XHR request will not try to use them.
        * 
        * This object can also be used to set the `X-Requested-With` header to
        * `XMLHttpRequest` (or any other value you need). To enable this do:
        * 
        * `this.load.headers.requestedWith = 'XMLHttpRequest'`
        * 
        * before adding anything to the Loader. The XHR loader will then call:
        * 
        * `setRequestHeader('X-Requested-With', this.headers['requestedWith'])`
        * Default: {"undefined":"application/xml"}
        */
        headers: any;

        /**
        * True if the Loader is in the process of loading the queue.
        */
        isLoading: boolean;

        /**
        * The number of concurrent / parallel resources to try and fetch at once.
        * 
        * Many current browsers limit 6 requests per domain; this is slightly conservative.
        * 
        * This should generally be left at the default, but can be set to a higher limit for specific use-cases. Just be careful when setting large values as different browsers could behave differently.
        */
        maxParallelDownloads: number;

        /**
        * This event is dispatched immediately before a file starts loading.
        * It's possible the file may fail (eg. download error, invalid format) after this event is sent.
        * 
        * Params: `(progress, file key, file url)`
        */
        onFileStart: Phaser.Signal;

        /**
        * This event is dispatched when a file has either loaded or failed to load.
        * 
        * Any function bound to this will receive the following parameters:
        * 
        * progress, file key, success?, total loaded files, total files
        * 
        * Where progress is a number between 1 and 100 (inclusive) representing the percentage of the load.
        */
        onFileComplete: Phaser.Signal;

        /**
        * This event is dispatched when a file (or pack) errors as a result of the load request.
        * 
        * For files it will be triggered before `onFileComplete`. For packs it will be triggered before `onPackComplete`.
        * 
        * Params: `(file key, file)`
        */
        onFileError: Phaser.Signal;

        /**
        * This event is dispatched when the final file in the load queue has either loaded or failed.
        */
        onLoadComplete: Phaser.Signal;

        /**
        * This event is dispatched when the loading process starts: before the first file has been requested,
        * but after all the initial packs have been loaded.
        */
        onLoadStart: Phaser.Signal;

        /**
        * This event is dispatched when an asset pack has either loaded or failed to load.
        * 
        * This is called when the asset pack manifest file has loaded and successfully added its contents to the loader queue.
        * 
        * Params: `(pack key, success?, total packs loaded, total packs)`
        */
        onPackComplete: Phaser.Signal;

        /**
        * The value of `path`, if set, is placed before any _relative_ file path given. For example:
        * 
        * `load.path = "images/sprites/";
        * load.image("ball", "ball.png");
        * load.image("tree", "level1/oaktree.png");
        * load.image("boom", "http://server.com/explode.png");`
        * 
        * Would load the `ball` file from `images/sprites/ball.png` and the tree from
        * `images/sprites/level1/oaktree.png` but the file `boom` would load from the URL
        * given as it's an absolute URL.
        * 
        * Please note that the path is added before the filename but *after* the baseURL (if set.)
        * 
        * The string _must_ end with a "/".
        */
        path: string;

        /**
        * You can optionally link a progress sprite with {@link Phaser.Loader#setPreloadSprite setPreloadSprite}.
        * 
        * This property is an object containing: sprite, rect, direction, width and height
        */
        preloadSprite: any;

        /**
        * The rounded load progress percentage value (from 0 to 100). See {@link Phaser.Loader#progressFloat}.
        */
        progress: number;

        /**
        * The non-rounded load progress value (from 0.0 to 100.0).
        * 
        * A general indicator of the progress.
        * It is possible for the progress to decrease, after `onLoadStart`, if more files are dynamically added.
        */
        progressFloat: number;

        /**
        * If true all calls to Loader.reset will be ignored. Useful if you need to create a load queue before swapping to a preloader state.
        */
        resetLocked: boolean;

        /**
        * If true and if the browser supports XDomainRequest, it will be used in preference for XHR.
        * 
        * This is only relevant for IE 9 and should _only_ be enabled for IE 9 clients when required by the server/CDN.
        */
        useXDomainRequest: boolean;


        /**
        * Informs the loader that the given file resource has been fetched and processed;
        * or such a request has failed.
        * 
        * @param file
        * @param error The error message, if any. No message implies no error. - Default: ''
        */
        asyncComplete(file: any, errorMessage?: string): void;

        /**
        * Add a synchronization point to a specific file/asset in the load queue.
        * 
        * This has no effect on already loaded assets.
        * 
        * @param type The type of resource to turn into a sync point (image, audio, xml, etc).
        * @param key Key of the file you want to turn into a sync point.
        * @return This Loader instance.
        */
        addSyncPoint(type: string, key: string): Phaser.Loader;

        /**
        * Internal function that adds a new entry to the file list. Do not call directly.
        * 
        * @param type The type of resource to add to the list (image, audio, xml, etc).
        * @param key The unique Cache ID key of this resource.
        * @param url The URL the asset will be loaded from.
        * @param properties Any additional properties needed to load the file. These are added directly to the added file object and overwrite any defaults. - Default: (none)
        * @param overwrite If true then this will overwrite a file asset of the same type/key. Otherwise it will only add a new asset. If overwrite is true, and the asset is already being loaded (or has been loaded), then it is appended instead.
        * @param extension If no URL is given the Loader will sometimes auto-generate the URL based on the key, using this as the extension.
        * @return This instance of the Phaser Loader.
        */
        addToFileList(type: string, key: string, url?: string, properties?: any, overwrite?: boolean, extension?: string): Phaser.Loader;

        /**
        * Adds a Texture Atlas file to the current load queue.
        * 
        * To create the Texture Atlas you can use tools such as:
        * 
        * [Texture Packer](https://www.codeandweb.com/texturepacker/phaser)
        * [Shoebox](http://renderhjs.net/shoebox/)
        * 
        * If using Texture Packer we recommend you enable "Trim sprite names".
        * If your atlas software has an option to "rotate" the resulting frames, you must disable it.
        * 
        * You can choose to either load the data externally, by providing a URL to a json file.
        * Or you can pass in a JSON object or String via the `atlasData` parameter.
        * If you pass a String the data is automatically run through `JSON.parse` and then immediately added to the Phaser.Cache.
        * 
        * If URLs are provided the files are **not** loaded immediately after calling this method, but are added to the load queue.
        * 
        * The key must be a unique String. It is used to add the file to the Phaser.Cache upon successful load.
        * 
        * Retrieve the file via `Cache.getImage(key)`. JSON files are automatically parsed upon load.
        * If you need to control when the JSON is parsed then use `Loader.text` instead and parse the JSON file as needed.
        * 
        * The URLs can be relative or absolute. If the URL is relative the `Loader.baseURL` and `Loader.path` values will be prepended to it.
        * 
        * If the textureURL isn't specified then the Loader will take the key and create a filename from that.
        * For example if the key is "player" and textureURL is null then the Loader will set the URL to be "player.png".
        * The same is true for the atlasURL. If atlasURL isn't specified and no atlasData has been provided then the Loader will
        * set the atlasURL to be the key. For example if the key is "player" the atlasURL will be set to "player.json".
        * 
        * If you do not desire this action then provide URLs and / or a data object.
        * 
        * @param key Unique asset key of the texture atlas file.
        * @param textureURL URL of the texture atlas image file. If undefined or `null` the url will be set to `<key>.png`, i.e. if `key` was "alien" then the URL will be "alien.png".
        * @param atlasURL URL of the texture atlas data file. If undefined or `null` and no atlasData is given, the url will be set to `<key>.json`, i.e. if `key` was "alien" then the URL will be "alien.json".
        * @param atlasData A JSON or XML data object. You don't need this if the data is being loaded from a URL.
        * @param format The format of the data. Can be Phaser.Loader.TEXTURE_ATLAS_JSON_ARRAY (the default), Phaser.Loader.TEXTURE_ATLAS_JSON_HASH or Phaser.Loader.TEXTURE_ATLAS_XML_STARLING.
        * @return This Loader instance.
        */
        atlas(key: string, textureURL?: string, atlasURL?: string, atlasData?: any, format?: number): Phaser.Loader;

        /**
        * Adds a Texture Atlas file to the current load queue.
        * 
        * Unlike `Loader.atlasJSONHash` this call expects the atlas data to be in a JSON Array format.
        * 
        * To create the Texture Atlas you can use tools such as:
        * 
        * [Texture Packer](https://www.codeandweb.com/texturepacker/phaser)
        * [Shoebox](http://renderhjs.net/shoebox/)
        * 
        * If using Texture Packer we recommend you enable "Trim sprite names".
        * If your atlas software has an option to "rotate" the resulting frames, you must disable it.
        * 
        * You can choose to either load the data externally, by providing a URL to a json file.
        * Or you can pass in a JSON object or String via the `atlasData` parameter.
        * If you pass a String the data is automatically run through `JSON.parse` and then immediately added to the Phaser.Cache.
        * 
        * If URLs are provided the files are **not** loaded immediately after calling this method, but are added to the load queue.
        * 
        * The key must be a unique String. It is used to add the file to the Phaser.Cache upon successful load.
        * 
        * Retrieve the file via `Cache.getImage(key)`. JSON files are automatically parsed upon load.
        * If you need to control when the JSON is parsed then use `Loader.text` instead and parse the JSON file as needed.
        * 
        * The URLs can be relative or absolute. If the URL is relative the `Loader.baseURL` and `Loader.path` values will be prepended to it.
        * 
        * If the textureURL isn't specified then the Loader will take the key and create a filename from that.
        * For example if the key is "player" and textureURL is null then the Loader will set the URL to be "player.png".
        * The same is true for the atlasURL. If atlasURL isn't specified and no atlasData has been provided then the Loader will
        * set the atlasURL to be the key. For example if the key is "player" the atlasURL will be set to "player.json".
        * 
        * If you do not desire this action then provide URLs and / or a data object.
        * 
        * @param key Unique asset key of the texture atlas file.
        * @param textureURL URL of the texture atlas image file. If undefined or `null` the url will be set to `<key>.png`, i.e. if `key` was "alien" then the URL will be "alien.png".
        * @param atlasURL URL of the texture atlas data file. If undefined or `null` and no atlasData is given, the url will be set to `<key>.json`, i.e. if `key` was "alien" then the URL will be "alien.json".
        * @param atlasData A JSON data object. You don't need this if the data is being loaded from a URL.
        * @return This Loader instance.
        */
        atlasJSONArray(key: string, textureURL?: string, atlasURL?: string, atlasData?: any): Phaser.Loader;

        /**
        * Adds a Texture Atlas file to the current load queue.
        * 
        * Unlike `Loader.atlas` this call expects the atlas data to be in a JSON Hash format.
        * 
        * To create the Texture Atlas you can use tools such as:
        * 
        * [Texture Packer](https://www.codeandweb.com/texturepacker/phaser)
        * [Shoebox](http://renderhjs.net/shoebox/)
        * 
        * If using Texture Packer we recommend you enable "Trim sprite names".
        * If your atlas software has an option to "rotate" the resulting frames, you must disable it.
        * 
        * You can choose to either load the data externally, by providing a URL to a json file.
        * Or you can pass in a JSON object or String via the `atlasData` parameter.
        * If you pass a String the data is automatically run through `JSON.parse` and then immediately added to the Phaser.Cache.
        * 
        * If URLs are provided the files are **not** loaded immediately after calling this method, but are added to the load queue.
        * 
        * The key must be a unique String. It is used to add the file to the Phaser.Cache upon successful load.
        * 
        * Retrieve the file via `Cache.getImage(key)`. JSON files are automatically parsed upon load.
        * If you need to control when the JSON is parsed then use `Loader.text` instead and parse the JSON file as needed.
        * 
        * The URLs can be relative or absolute. If the URL is relative the `Loader.baseURL` and `Loader.path` values will be prepended to it.
        * 
        * If the textureURL isn't specified then the Loader will take the key and create a filename from that.
        * For example if the key is "player" and textureURL is null then the Loader will set the URL to be "player.png".
        * The same is true for the atlasURL. If atlasURL isn't specified and no atlasData has been provided then the Loader will
        * set the atlasURL to be the key. For example if the key is "player" the atlasURL will be set to "player.json".
        * 
        * If you do not desire this action then provide URLs and / or a data object.
        * 
        * @param key Unique asset key of the texture atlas file.
        * @param textureURL URL of the texture atlas image file. If undefined or `null` the url will be set to `<key>.png`, i.e. if `key` was "alien" then the URL will be "alien.png".
        * @param atlasURL URL of the texture atlas data file. If undefined or `null` and no atlasData is given, the url will be set to `<key>.json`, i.e. if `key` was "alien" then the URL will be "alien.json".
        * @param atlasData A JSON data object. You don't need this if the data is being loaded from a URL.
        * @return This Loader instance.
        */
        atlasJSONHash(key: string, textureURL?: string, atlasURL?: string, atlasData?: any): Phaser.Loader;

        /**
        * Adds a Texture Atlas file to the current load queue.
        * 
        * This call expects the atlas data to be in the Starling XML data format.
        * 
        * To create the Texture Atlas you can use tools such as:
        * 
        * [Texture Packer](https://www.codeandweb.com/texturepacker/phaser)
        * [Shoebox](http://renderhjs.net/shoebox/)
        * 
        * If using Texture Packer we recommend you enable "Trim sprite names".
        * If your atlas software has an option to "rotate" the resulting frames, you must disable it.
        * 
        * You can choose to either load the data externally, by providing a URL to an xml file.
        * Or you can pass in an XML object or String via the `atlasData` parameter.
        * If you pass a String the data is automatically run through `Loader.parseXML` and then immediately added to the Phaser.Cache.
        * 
        * If URLs are provided the files are **not** loaded immediately after calling this method, but are added to the load queue.
        * 
        * The key must be a unique String. It is used to add the file to the Phaser.Cache upon successful load.
        * 
        * Retrieve the file via `Cache.getImage(key)`. XML files are automatically parsed upon load.
        * If you need to control when the XML is parsed then use `Loader.text` instead and parse the XML file as needed.
        * 
        * The URLs can be relative or absolute. If the URL is relative the `Loader.baseURL` and `Loader.path` values will be prepended to it.
        * 
        * If the textureURL isn't specified then the Loader will take the key and create a filename from that.
        * For example if the key is "player" and textureURL is null then the Loader will set the URL to be "player.png".
        * The same is true for the atlasURL. If atlasURL isn't specified and no atlasData has been provided then the Loader will
        * set the atlasURL to be the key. For example if the key is "player" the atlasURL will be set to "player.xml".
        * 
        * If you do not desire this action then provide URLs and / or a data object.
        * 
        * @param key Unique asset key of the texture atlas file.
        * @param textureURL URL of the texture atlas image file. If undefined or `null` the url will be set to `<key>.png`, i.e. if `key` was "alien" then the URL will be "alien.png".
        * @param atlasURL URL of the texture atlas data file. If undefined or `null` and no atlasData is given, the url will be set to `<key>.json`, i.e. if `key` was "alien" then the URL will be "alien.xml".
        * @param atlasData An XML data object. You don't need this if the data is being loaded from a URL.
        * @return This Loader instance.
        */
        atlasXML(key: string, textureURL?: string, atlasURL?: string, atlasData?: any): Phaser.Loader;

        /**
        * Adds an audio file to the current load queue.
        * 
        * The file is **not** loaded immediately after calling this method. The file is added to the queue ready to be loaded when the loader starts.
        * 
        * The key must be a unique String. It is used to add the file to the Phaser.Cache upon successful load.
        * 
        * Retrieve the file via `Cache.getSound(key)`.
        * 
        * The URL can be relative or absolute. If the URL is relative the `Loader.baseURL` and `Loader.path` values will be prepended to it.
        * 
        * Mobile warning: There are some mobile devices (certain iPad 2 and iPad Mini revisions) that cannot play 48000 Hz audio.
        * When they try to play the audio becomes extremely distorted and buzzes, eventually crashing the sound system.
        * The solution is to use a lower encoding rate such as 44100 Hz.
        * 
        * @param key Unique asset key of the audio file.
        * @param urls Either a single string or an array of URIs or pairs of `{uri: .., type: ..}`.
        *             If an array is specified then the first URI (or URI + mime pair) that is device-compatible will be selected.
        *             For example: `"jump.mp3"`, `['jump.mp3', 'jump.ogg', 'jump.m4a']`, or `[{uri: "data:<opus_resource>", type: 'opus'}, 'fallback.mp3']`.
        *             BLOB and DATA URIs can be used but only support automatic detection when used in the pair form; otherwise the format must be manually checked before adding the resource.
        * @param autoDecode When using Web Audio the audio files can either be decoded at load time or run-time.
        *                   Audio files can't be played until they are decoded and, if specified, this enables immediate decoding. Decoding is a non-blocking async process, however it consumes huge amounts of CPU time on mobiles especially. - Default: true
        * @return This Loader instance.
        */
        audio(key: string, urls: string | string[] | any, autoDecode?: boolean): Phaser.Loader;

        /**
        * A legacy alias for Loader.audioSprite. Please see that method for documentation.
        * 
        * @param key Unique asset key of the audio file.
        * @param urls An array containing the URLs of the audio files, i.e.: [ 'audiosprite.mp3', 'audiosprite.ogg', 'audiosprite.m4a' ] or a single string containing just one URL.
        * @param jsonURL The URL of the audiosprite configuration JSON object. If you wish to pass the data directly set this parameter to null.
        * @param jsonData A JSON object or string containing the audiosprite configuration data. This is ignored if jsonURL is not null.
        * @param autoDecode When using Web Audio the audio files can either be decoded at load time or run-time.
        *                   Audio files can't be played until they are decoded and, if specified, this enables immediate decoding. Decoding is a non-blocking async process, however it consumes huge amounts of CPU time on mobiles especially. - Default: true
        * @return This Loader instance.
        */
        audiosprite(key: string, urls: string[], jsonURL?: string, jsonData?: string | any, autoDecode?: boolean): Phaser.Loader;

        /**
        * Adds a binary file to the current load queue.
        * 
        * The file is **not** loaded immediately after calling this method. The file is added to the queue ready to be loaded when the loader starts.
        * 
        * The key must be a unique String. It is used to add the file to the Phaser.Cache upon successful load.
        * 
        * Retrieve the file via `Cache.getBinary(key)`.
        * 
        * The URL can be relative or absolute. If the URL is relative the `Loader.baseURL` and `Loader.path` values will be prepended to it.
        * 
        * If the URL isn't specified the Loader will take the key and create a filename from that. For example if the key is "alien"
        * and no URL is given then the Loader will set the URL to be "alien.bin". It will always add `.bin` as the extension.
        * If you do not desire this action then provide a URL.
        * 
        * It will be loaded via xhr with a responseType of "arraybuffer". You can specify an optional callback to process the file after load.
        * When the callback is called it will be passed 2 parameters: the key of the file and the file data.
        * 
        * WARNING: If a callback is specified the data will be set to whatever it returns. Always return the data object, even if you didn't modify it.
        * 
        * @param key Unique asset key of the binary file.
        * @param url URL of the binary file. If undefined or `null` the url will be set to `<key>.bin`, i.e. if `key` was "alien" then the URL will be "alien.bin".
        * @param callback Optional callback that will be passed the file after loading, so you can perform additional processing on it. - Default: (none)
        * @param callbackContext The context under which the callback will be applied. If not specified it will use the callback itself as the context.
        * @return This Loader instance.
        */
        binary(key: string, url?: string, callback?: Function, callbackContext?: any): Phaser.Loader;

        /**
        * Adds Bitmap Font files to the current load queue.
        * 
        * To create the Bitmap Font files you can use:
        * 
        * BMFont (Windows, free): http://www.angelcode.com/products/bmfont/
        * Glyph Designer (OS X, commercial): http://www.71squared.com/en/glyphdesigner
        * Littera (Web-based, free): http://kvazars.com/littera/
        * 
        * You can choose to either load the data externally, by providing a URL to an xml file.
        * Or you can pass in an XML object or String via the `xmlData` parameter.
        * If you pass a String the data is automatically run through `Loader.parseXML` and then immediately added to the Phaser.Cache.
        * 
        * If URLs are provided the files are **not** loaded immediately after calling this method, but are added to the load queue.
        * 
        * The key must be a unique String. It is used to add the file to the Phaser.Cache upon successful load.
        * 
        * Retrieve the file via `Cache.getBitmapFont(key)`. XML files are automatically parsed upon load.
        * If you need to control when the XML is parsed then use `Loader.text` instead and parse the XML file as needed.
        * 
        * The URLs can be relative or absolute. If the URL is relative the `Loader.baseURL` and `Loader.path` values will be prepended to it.
        * 
        * If the textureURL isn't specified then the Loader will take the key and create a filename from that.
        * For example if the key is "megaFont" and textureURL is null then the Loader will set the URL to be "megaFont.png".
        * The same is true for the atlasURL. If atlasURL isn't specified and no atlasData has been provided then the Loader will
        * set the atlasURL to be the key. For example if the key is "megaFont" the atlasURL will be set to "megaFont.xml".
        * 
        * If you do not desire this action then provide URLs and / or a data object.
        * 
        * @param key Unique asset key of the bitmap font.
        * @param textureURL URL of the Bitmap Font texture file. If undefined or `null` the url will be set to `<key>.png`, i.e. if `key` was "megaFont" then the URL will be "megaFont.png".
        * @param atlasURL URL of the Bitmap Font atlas file (xml/json). If undefined or `null` AND `atlasData` is null, the url will be set to `<key>.xml`, i.e. if `key` was "megaFont" then the URL will be "megaFont.xml".
        * @param atlasData An optional Bitmap Font atlas in string form (stringified xml/json).
        * @param xSpacing If you'd like to add additional horizontal spacing between the characters then set the pixel value here.
        * @param ySpacing If you'd like to add additional vertical spacing between the lines then set the pixel value here.
        * @return This Loader instance.
        */
        bitmapFont(key: string, textureURL?: string, atlasURL?: string, atlasData?: any, xSpacing?: number, ySpacing?: number): Phaser.Loader;

        /**
        * Check whether a file/asset with a specific key is queued to be loaded.
        * 
        * To access a loaded asset use Phaser.Cache, eg. {@link Phaser.Cache#checkImageKey}
        * 
        * @param type The type asset you want to check.
        * @param key Key of the asset you want to check.
        * @return Return true if exists, otherwise return false.
        */
        checkKeyExists(type: string, key: string): boolean;

        /**
        * Successfully loaded a CSV file - only used for certain types.
        * 
        * @param file File associated with this request
        * @param xhr
        */
        csvLoadComplete(file: any, xhr: XMLHttpRequest): void;

        /**
        * Called when a file has been downloaded and needs to be processed further.
        * 
        * @param file File loaded
        * @param xhr XHR request, unspecified if loaded via other means (eg. tags)
        */
        fileComplete(file: any, xhr: XMLHttpRequest): void;

        /**
        * Error occurred when loading a file.
        * 
        * @param file
        * @param xhr XHR request, unspecified if loaded via other means (eg. tags)
        * @param reason
        */
        fileError(file: any, xhr: XMLHttpRequest, reason: string): void;

        /**
        * The loading is all finished.
        * 
        * @param abnormal True if the loading finished abnormally. - Default: true
        */
        finishedLoading(abnormal?: boolean): void;

        /**
        * Find a file/asset with a specific key.
        * 
        * Only assets in the download file queue will be found.
        * 
        * @param type The type asset you want to check.
        * @param key Key of the asset you want to check.
        * @return Returns an object if found that has 2 properties: `index` and `file`; otherwise a non-true value is returned.
        *         The index may change and should only be used immediately following this call.
        */
        getAsset(type: string, key: string): any;

        /**
        * Get the queue-index of the file/asset with a specific key.
        * 
        * Only assets in the download file queue will be found.
        * 
        * @param type The type asset you want to check.
        * @param key Key of the asset you want to check.
        * @return The index of this key in the filelist, or -1 if not found.
        *         The index may change and should only be used immediately following this call
        */
        getAssetIndex(type: string, key: string): number;

        /**
        * Give a bunch of URLs, return the first URL that has an extension this device thinks it can play.
        * 
        * It is assumed that the device can play "blob:" or "data:" URIs - There is no mime-type checking on data URIs.
        * 
        * @param urls See {@link #audio} for format.
        * @return The URL to try and fetch; or null.
        */
        getAudioURL(urls: any[]): void;

        /**
        * Adds an Image to the current load queue.
        * 
        * The file is **not** loaded immediately after calling this method. The file is added to the queue ready to be loaded when the loader starts.
        * 
        * Phaser can load all common image types: png, jpg, gif and any other format the browser can natively handle.
        * 
        * The key must be a unique String. It is used to add the file to the Phaser.Cache upon successful load.
        * 
        * Retrieve the image via `Cache.getImage(key)`
        * 
        * The URL can be relative or absolute. If the URL is relative the `Loader.baseURL` and `Loader.path` values will be prepended to it.
        * 
        * If the URL isn't specified the Loader will take the key and create a filename from that. For example if the key is "alien"
        * and no URL is given then the Loader will set the URL to be "alien.png". It will always add `.png` as the extension.
        * If you do not desire this action then provide a URL.
        * 
        * This method also supports passing in a texture object as the `url` argument. This allows you to load
        * compressed textures into Phaser. You can also use `Loader.texture` to do this.
        * 
        * Compressed Textures are a WebGL only feature, and require 3rd party tools to create.
        * Available tools include Texture Packer, PVRTexTool, DirectX Texture Tool and Mali Texture Compression Tool.
        * 
        * Supported texture compression formats are: PVRTC, S3TC and ETC1.
        * Supported file formats are: PVR, DDS, KTX and PKM.
        * 
        * The formats that support all 3 compression algorithms are PVR and KTX.
        * PKM only supports ETC1, and DDS only S3TC for now.
        * 
        * The texture path object looks like this:
        * 
        * ```
        * load.image('factory', {
        *     etc1: 'assets/factory_etc1.pkm',
        *     s3tc: 'assets/factory_dxt1.pvr',
        *     pvrtc: 'assets/factory_pvrtc.pvr',
        *     truecolor: 'assets/factory.png'
        * });
        * ```
        * 
        * The `truecolor` property points to a standard PNG file, that will be used if none of the
        * compressed formats are supported by the browser / GPU.
        * 
        * @param key Unique asset key of this image file.
        * @param url URL of an image file. If undefined or `null` the url will be set to `<key>.png`, i.e. if `key` was "alien" then the URL will be "alien.png". Can also be a texture data object.
        * @param overwrite If an unloaded file with a matching key already exists in the queue, this entry will overwrite it.
        * @return This Loader instance.
        */
        image(key: string, url?: string | any, overwrite?: boolean): Phaser.Loader;

        /**
        * Generate an image from a BitmapData object and add it to the current load queue.
        * 
        * @param key Unique asset key for the generated image.
        * @param bitmapData
        * @param overwrite If an unloaded file with a matching key already exists in the queue, this entry will overwrite it.
        * @return This Loader instance.
        */
        imageFromBitmapData(key: string, bitmapData: Phaser.BitmapData, overwrite?: boolean): Phaser.Loader;

        /**
        * Adds an array of images to the current load queue.
        * 
        * It works by passing each element of the array to the Loader.image method.
        * 
        * The files are **not** loaded immediately after calling this method. The files are added to the queue ready to be loaded when the loader starts.
        * 
        * Phaser can load all common image types: png, jpg, gif and any other format the browser can natively handle.
        * 
        * The keys must be unique Strings. They are used to add the files to the Phaser.Cache upon successful load.
        * 
        * Retrieve the images via `Cache.getImage(key)`
        * 
        * The URL can be relative or absolute. If the URL is relative the `Loader.baseURL` and `Loader.path` values will be prepended to it.
        * 
        * If the URL isn't specified the Loader will take the key and create a filename from that. For example if the key is "alien"
        * and no URL is given then the Loader will set the URL to be "alien.png". It will always add `.png` as the extension.
        * If you do not desire this action then provide a URL.
        * 
        * @param keys An array of unique asset keys of the image files.
        * @param urls Optional array of URLs. If undefined or `null` the url will be set to `<key>.png`, i.e. if `key` was "alien" then the URL will be "alien.png". If provided the URLs array length must match the keys array length.
        * @return This Loader instance.
        */
        images(keys: string[], urls?: string[]): Phaser.Loader;

        /**
        * Adds a JSON file to the current load queue.
        * 
        * The file is **not** loaded immediately after calling this method. The file is added to the queue ready to be loaded when the loader starts.
        * 
        * The key must be a unique String. It is used to add the file to the Phaser.Cache upon successful load.
        * 
        * Retrieve the file via `Cache.getJSON(key)`. JSON files are automatically parsed upon load.
        * If you need to control when the JSON is parsed then use `Loader.text` instead and parse the text file as needed.
        * 
        * The URL can be relative or absolute. If the URL is relative the `Loader.baseURL` and `Loader.path` values will be prepended to it.
        * 
        * If the URL isn't specified the Loader will take the key and create a filename from that. For example if the key is "alien"
        * and no URL is given then the Loader will set the URL to be "alien.json". It will always add `.json` as the extension.
        * If you do not desire this action then provide a URL.
        * 
        * @param key Unique asset key of the json file.
        * @param url URL of the JSON file. If undefined or `null` the url will be set to `<key>.json`, i.e. if `key` was "alien" then the URL will be "alien.json".
        * @param overwrite If an unloaded file with a matching key already exists in the queue, this entry will overwrite it.
        * @return This Loader instance.
        */
        json(key: string, url?: string, overwrite?: boolean): Phaser.Loader;

        /**
        * Successfully loaded a JSON file - only used for certain types.
        * 
        * @param file File associated with this request
        * @param xhr
        */
        jsonLoadComplete(file: any, xhr: XMLHttpRequest): void;

        /**
        * Continue async loading through an Audio tag.
        */
        loadAudioTag(file: any): void;

        /**
        * Start fetching a resource.
        * 
        * All code paths, async or otherwise, from this function must return to `asyncComplete`.
        * 
        * @param file
        */
        loadFile(file: any): void;

        /**
        * Continue async loading through an Image tag.
        */
        loadImageTag(file: any): void;

        /**
        * Add a JSON resource pack ('packfile') to the Loader.
        * 
        * A packfile is a JSON file that contains a list of assets to the be loaded.
        * Please see the example 'loader/asset pack' in the Phaser Examples repository.
        * 
        * Packs are always put before the first non-pack file that is not loaded / loading.
        * 
        * This means that all packs added before any loading has started are added to the front
        * of the file queue, in the order added.
        * 
        * The key must be a unique String. It is used to add the file to the Phaser.Cache upon successful load.
        * 
        * The URL of the packfile can be relative or absolute. If the URL is relative the `Loader.baseURL` and `Loader.path` values will be prepended to it.
        * 
        * @param key Unique asset key of this resource pack.
        * @param url URL of the Asset Pack JSON file. If you wish to pass a json object instead set this to null and pass the object as the data parameter.
        * @param data The Asset Pack JSON data. Use this to pass in a json data object rather than loading it from a URL. TODO
        * @param callbackContext Some Loader operations, like Binary and Script require a context for their callbacks. Pass the context here. - Default: (loader)
        * @return This Loader instance.
        */
        pack(key: string, url?: string, data?: any, callbackContext?: any): Phaser.Loader;

        /**
        * Parses string data as XML.
        * 
        * @param data The XML text to parse
        * @return Returns the xml document, or null if such could not parsed to a valid document.
        */
        parseXml(data: string): XMLDocument;

        /**
        * Adds a physics data file to the current load queue.
        * 
        * The data must be in `Lime + Corona` JSON format. [Physics Editor](https://www.codeandweb.com) by code'n'web exports in this format natively.
        * 
        * You can choose to either load the data externally, by providing a URL to a json file.
        * Or you can pass in a JSON object or String via the `data` parameter.
        * If you pass a String the data is automatically run through `JSON.parse` and then immediately added to the Phaser.Cache.
        * 
        * If a URL is provided the file is **not** loaded immediately after calling this method, but is added to the load queue.
        * 
        * The key must be a unique String. It is used to add the file to the Phaser.Cache upon successful load.
        * 
        * Retrieve the file via `Cache.getJSON(key)`. JSON files are automatically parsed upon load.
        * If you need to control when the JSON is parsed then use `Loader.text` instead and parse the text file as needed.
        * 
        * The URL can be relative or absolute. If the URL is relative the `Loader.baseURL` and `Loader.path` values will be prepended to it.
        * 
        * If the URL isn't specified and no data is given then the Loader will take the key and create a filename from that.
        * For example if the key is "alien" and no URL or data is given then the Loader will set the URL to be "alien.json".
        * It will always use `.json` as the extension.
        * 
        * If you do not desire this action then provide a URL or data object.
        * 
        * @param key Unique asset key of the physics json data.
        * @param url URL of the physics data file. If undefined or `null` and no data is given the url will be set to `<key>.json`, i.e. if `key` was "alien" then the URL will be "alien.json".
        * @param data An optional JSON data object. If given then the url is ignored and this JSON object is used for physics data instead.
        * @param format The format of the physics data. - Default: Phaser.Physics.LIME_CORONA_JSON
        * @return This Loader instance.
        */
        physics(key: string, url?: string, data?: any, format?: string): Phaser.Loader;

        /**
        * Process the next item(s) in the file/asset queue.
        * 
        * Process the queue and start loading enough items to fill up the inflight queue.
        * 
        * If a sync-file is encountered then subsequent asset processing is delayed until it completes.
        * The exception to this rule is that packfiles can be downloaded (but not processed) even if
        * there appear other sync files (ie. packs) - this enables multiple packfiles to be fetched in parallel.
        * such as during the start phaser.
        */
        processLoadQueue(): void;

        /**
        * Process pack data. This will usually modify the file list.
        * 
        * @param pack
        */
        processPack(pack: any): void;

        /**
        * Remove all file loading requests - this is _insufficient_ to stop current loading. Use `reset` instead.
        */
        removeAll(): void;

        /**
        * Remove a file/asset from the loading queue.
        * 
        * A file that is loaded or has started loading cannot be removed.
        * 
        * @param type The type of resource to add to the list (image, audio, xml, etc).
        * @param key Key of the file you want to remove.
        */
        removeFile(type: string, key: string): void;

        /**
        * Internal function that replaces an existing entry in the file list with a new one. Do not call directly.
        * 
        * @param type The type of resource to add to the list (image, audio, xml, etc).
        * @param key The unique Cache ID key of this resource.
        * @param url The URL the asset will be loaded from.
        * @param properties Any additional properties needed to load the file.
        */
        replaceInFileList(type: string, key: string, url: string, properties: any): void;

        /**
        * Reset the loader and clear any queued assets. If `Loader.resetLocked` is true this operation will abort.
        * 
        * This will abort any loading and clear any queued assets.
        * 
        * Optionally you can clear any associated events.
        * 
        * @param hard If true then the preload sprite and other artifacts may also be cleared.
        * @param clearEvents If true then the all Loader signals will have removeAll called on them.
        */
        reset(hard?: boolean, clearEvents?: boolean): void;

        /**
        * Called automatically by ScaleManager when the game resizes in RESIZE scalemode.
        * 
        * This can be used to adjust the preloading sprite size, eg.
        */
        resize(): void;

        /**
        * Adds a JavaScript file to the current load queue.
        * 
        * The file is **not** loaded immediately after calling this method. The file is added to the queue ready to be loaded when the loader starts.
        * 
        * The key must be a unique String.
        * 
        * The URL can be relative or absolute. If the URL is relative the `Loader.baseURL` and `Loader.path` values will be prepended to it.
        * 
        * If the URL isn't specified the Loader will take the key and create a filename from that. For example if the key is "alien"
        * and no URL is given then the Loader will set the URL to be "alien.js". It will always add `.js` as the extension.
        * If you do not desire this action then provide a URL.
        * 
        * Upon successful load the JavaScript is automatically turned into a script tag and executed, so be careful what you load!
        * 
        * A callback, which will be invoked as the script tag has been created, can also be specified.
        * The callback must return relevant `data`.
        * 
        * @param key Unique asset key of the script file.
        * @param url URL of the JavaScript file. If undefined or `null` the url will be set to `<key>.js`, i.e. if `key` was "alien" then the URL will be "alien.js".
        * @param callback Optional callback that will be called after the script tag has loaded, so you can perform additional processing. - Default: (none)
        * @param callbackContext The context under which the callback will be applied. If not specified it will use the Phaser Loader as the context. - Default: (loader)
        * @return This Loader instance.
        */
        script(key: string, url?: String, callback?: Function, callbackContext?: any): Phaser.Loader;

        /**
        * Adds a fragment shader file to the current load queue.
        * 
        * The file is **not** loaded immediately after calling this method. The file is added to the queue ready to be loaded when the loader starts.
        * 
        * The key must be a unique String. It is used to add the file to the Phaser.Cache upon successful load.
        * 
        * Retrieve the file via `Cache.getShader(key)`.
        * 
        * The URL can be relative or absolute. If the URL is relative the `Loader.baseURL` and `Loader.path` values will be prepended to it.
        * 
        * If the URL isn't specified the Loader will take the key and create a filename from that. For example if the key is "blur"
        * and no URL is given then the Loader will set the URL to be "blur.frag". It will always add `.frag` as the extension.
        * If you do not desire this action then provide a URL.
        * 
        * @param key Unique asset key of the fragment file.
        * @param url URL of the fragment file. If undefined or `null` the url will be set to `<key>.frag`, i.e. if `key` was "blur" then the URL will be "blur.frag".
        * @param overwrite If an unloaded file with a matching key already exists in the queue, this entry will overwrite it.
        * @return This Loader instance.
        */
        shader(key: string, url?: String, overwrite?: boolean): Phaser.Loader;

        /**
        * Set a Sprite to be a "preload" sprite by passing it to this method.
        * 
        * A "preload" sprite will have its width or height crop adjusted based on the percentage of the loader in real-time.
        * This allows you to easily make loading bars for games.
        * 
        * The sprite will automatically be made visible when calling this.
        * 
        * @param sprite The sprite or image that will be cropped during the load.
        * @param direction A value of zero means the sprite will be cropped horizontally, a value of 1 means its will be cropped vertically.
        */
        setPreloadSprite(sprite: Phaser.Sprite | Phaser.Image, direction?: number): void;

        /**
        * Adds a Sprite Sheet to the current load queue.
        * 
        * The file is **not** loaded immediately after calling this method. The file is added to the queue ready to be loaded when the loader starts.
        * 
        * To clarify the terminology that Phaser uses: A Sprite Sheet is an image containing frames, usually of an animation, that are all equal
        * dimensions and often in sequence. For example if the frame size is 32x32 then every frame in the sprite sheet will be that size.
        * Sometimes (outside of Phaser) the term "sprite sheet" is used to refer to a texture atlas.
        * A Texture Atlas works by packing together images as best it can, using whatever frame sizes it likes, often with cropping and trimming
        * the frames in the process. Software such as Texture Packer, Flash CC or Shoebox all generate texture atlases, not sprite sheets.
        * If you've got an atlas then use `Loader.atlas` instead.
        * 
        * The key must be a unique String. It is used to add the image to the Phaser.Cache upon successful load.
        * 
        * Retrieve the file via `Cache.getImage(key)`. Sprite sheets, being image based, live in the same Cache as all other Images.
        * 
        * The URL can be relative or absolute. If the URL is relative the `Loader.baseURL` and `Loader.path` values will be prepended to it.
        * 
        * If the URL isn't specified the Loader will take the key and create a filename from that. For example if the key is "alien"
        * and no URL is given then the Loader will set the URL to be "alien.png". It will always add `.png` as the extension.
        * If you do not desire this action then provide a URL.
        * 
        * @param key Unique asset key of the sheet file.
        * @param url URL of the sprite sheet file. If undefined or `null` the url will be set to `<key>.png`, i.e. if `key` was "alien" then the URL will be "alien.png".
        * @param frameWidth Width in pixels of a single frame in the sprite sheet.
        * @param frameHeight Height in pixels of a single frame in the sprite sheet.
        * @param frameMax How many frames in this sprite sheet. If not specified it will divide the whole image into frames. - Default: -1
        * @param margin If the frames have been drawn with a margin, specify the amount here.
        * @param spacing If the frames have been drawn with spacing between them, specify the amount here.
        * @param skipFrames Skip a number of frames. Useful when there are multiple sprite sheets in one image.
        * @return This Loader instance.
        */
        spritesheet(key: string, url: string, frameWidth: number, frameHeight: number, frameMax?: number, margin?: number, spacing?: number, skipFrames?: number): Phaser.Loader;

        /**
        * Start loading the assets. Normally you don't need to call this yourself as the StateManager will do so.
        */
        start(): void;

        /**
        * Adds a Text file to the current load queue.
        * 
        * The file is **not** loaded immediately after calling this method. The file is added to the queue ready to be loaded when the loader starts.
        * 
        * The key must be a unique String. It is used to add the file to the Phaser.Cache upon successful load.
        * 
        * Retrieve the file via `Cache.getText(key)`
        * 
        * The URL can be relative or absolute. If the URL is relative the `Loader.baseURL` and `Loader.path` values will be prepended to it.
        * 
        * If the URL isn't specified the Loader will take the key and create a filename from that. For example if the key is "alien"
        * and no URL is given then the Loader will set the URL to be "alien.txt". It will always add `.txt` as the extension.
        * If you do not desire this action then provide a URL.
        * 
        * @param key Unique asset key of the text file.
        * @param url URL of the text file. If undefined or `null` the url will be set to `<key>.txt`, i.e. if `key` was "alien" then the URL will be "alien.txt".
        * @param overwrite If an unloaded file with a matching key already exists in the queue, this entry will overwrite it.
        * @return This Loader instance.
        */
        text(key: string, url?: string, overwrite?: boolean): Phaser.Loader;

        /**
        * Adds a Compressed Texture Image to the current load queue.
        * 
        * Compressed Textures are a WebGL only feature, and require 3rd party tools to create.
        * Available tools include Texture Packer, PVRTexTool, DirectX Texture Tool and Mali Texture Compression Tool.
        * 
        * Supported texture compression formats are: PVRTC, S3TC and ETC1.
        * Supported file formats are: PVR, DDS, KTX and PKM.
        * 
        * The formats that support all 3 compression algorithms are PVR and KTX.
        * PKM only supports ETC1, and DDS only S3TC for now.
        * 
        * The texture path object looks like this:
        * 
        * ```
        * load.texture('factory', {
        *     etc1: 'assets/factory_etc1.pkm',
        *     s3tc: 'assets/factory_dxt1.pvr',
        *     pvrtc: 'assets/factory_pvrtc.pvr',
        *     truecolor: 'assets/factory.png'
        * });
        * ```
        * 
        * The `truecolor` property points to a standard PNG file, that will be used if none of the
        * compressed formats are supported by the browser / GPU.
        * 
        * The file is **not** loaded immediately after calling this method. The file is added to the queue ready to be loaded when the loader starts.
        * 
        * The key must be a unique String. It is used to add the file to the Phaser.Cache upon successful load.
        * 
        * Retrieve the image via `Cache.getImage(key)`
        * 
        * The URL can be relative or absolute. If the URL is relative the `Loader.baseURL` and `Loader.path` values will be prepended to it.
        * 
        * If the URL isn't specified the Loader will take the key and create a filename from that. For example if the key is "alien"
        * and no URL is given then the Loader will set the URL to be "alien.pvr". It will always add `.pvr` as the extension.
        * If you do not desire this action then provide a URL.
        * 
        * @param key Unique asset key of this image file.
        * @param object The texture path data object.
        * @param overwrite If an unloaded file with a matching key already exists in the queue, this entry will overwrite it.
        * @return This Loader instance.
        */
        texture(key: string, object: any, overwrite?: boolean): Phaser.Loader;

        /**
        * Adds a Tile Map data file to the current load queue.
        * 
        * Phaser can load data in two different formats: CSV and Tiled JSON.
        * 
        * Tiled is a free software package, specifically for creating tilemaps, and is available from http://www.mapeditor.org
        * 
        * You can choose to either load the data externally, by providing a URL to a json file.
        * Or you can pass in a JSON object or String via the `data` parameter.
        * If you pass a String the data is automatically run through `JSON.parse` and then immediately added to the Phaser.Cache.
        * 
        * If a URL is provided the file is **not** loaded immediately after calling this method, but is added to the load queue.
        * 
        * The key must be a unique String. It is used to add the file to the Phaser.Cache upon successful load.
        * 
        * Retrieve the file via `Cache.getTilemapData(key)`. JSON files are automatically parsed upon load.
        * If you need to control when the JSON is parsed then use `Loader.text` instead and parse the text file as needed.
        * 
        * The URL can be relative or absolute. If the URL is relative the `Loader.baseURL` and `Loader.path` values will be prepended to it.
        * 
        * If the URL isn't specified and no data is given then the Loader will take the key and create a filename from that.
        * For example if the key is "level1" and no URL or data is given then the Loader will set the URL to be "level1.json".
        * If you set the format to be Tilemap.CSV it will set the URL to be "level1.csv" instead.
        * 
        * If you do not desire this action then provide a URL or data object.
        * 
        * @param key Unique asset key of the tilemap data.
        * @param url URL of the tile map file. If undefined or `null` and no data is given the url will be set to `<key>.json`, i.e. if `key` was "level1" then the URL will be "level1.json".
        * @param data An optional JSON data object. If given then the url is ignored and this JSON object is used for map data instead.
        * @param format The format of the map data. Either Phaser.Tilemap.CSV or Phaser.Tilemap.TILED_JSON. - Default: Phaser.Tilemap.CSV
        * @return This Loader instance.
        */
        tilemap(key: string, url?: string, data?: any, format?: number): Phaser.Loader;

        /**
        * Returns the number of files that have already been loaded, even if they errored.
        * @return The number of files that have already been loaded (even if they errored)
        */
        totalLoadedFiles(): number;

        /**
        * Returns the number of asset packs that have already been loaded, even if they errored.
        * @return The number of asset packs that have already been loaded (even if they errored)
        */
        totalLoadedPacks(): number;

        /**
        * Returns the number of files still waiting to be processed in the load queue. This value decreases as each file in the queue is loaded.
        * @return The number of files that still remain in the load queue.
        */
        totalQueuedFiles(): number;

        /**
        * Returns the number of asset packs still waiting to be processed in the load queue. This value decreases as each pack in the queue is loaded.
        * @return The number of asset packs that still remain in the load queue.
        */
        totalQueuedPacks(): number;

        /**
        * Transforms the asset URL.
        * 
        * The default implementation prepends the baseURL if the url doesn't begin with http or //
        * 
        * @param url The url to transform.
        * @param file The file object being transformed.
        * @return The transformed url. In rare cases where the url isn't specified it will return false instead.
        */
        transformUrl(url: string, file?: any): string;
        updateProgress(): void;

        /**
        * Adds a video file to the current load queue.
        * 
        * The file is **not** loaded immediately after calling this method. The file is added to the queue ready to be loaded when the loader starts.
        * 
        * The key must be a unique String. It is used to add the file to the Phaser.Cache upon successful load.
        * 
        * Retrieve the file via `Cache.getVideo(key)`.
        * 
        * The URL can be relative or absolute. If the URL is relative the `Loader.baseURL` and `Loader.path` values will be prepended to it.
        * 
        * You don't need to preload a video in order to play it in your game. See `Video.createVideoFromURL` for details.
        * 
        * @param key Unique asset key of the video file.
        * @param urls Either a single string or an array of URIs or pairs of `{uri: .., type: ..}`.
        *             If an array is specified then the first URI (or URI + mime pair) that is device-compatible will be selected.
        *             For example: `"boom.mp4"`, `['boom.mp4', 'boom.ogg', 'boom.webm']`, or `[{uri: "data:<opus_resource>", type: 'opus'}, 'fallback.mp4']`.
        *             BLOB and DATA URIs can be used but only support automatic detection when used in the pair form; otherwise the format must be manually checked before adding the resource.
        * @param loadEvent This sets the Video source event to listen for before the load is considered complete.
        *                  'canplaythrough' implies the video has downloaded enough, and bandwidth is high enough that it can be played to completion.
        *                  'canplay' implies the video has downloaded enough to start playing, but not necessarily to finish.
        *                  'loadeddata' just makes sure that the video meta data and first frame have downloaded. Phaser uses this value automatically if the
        *                  browser is detected as being Firefox and no `loadEvent` is given, otherwise it defaults to `canplaythrough`. - Default: 'canplaythrough'
        * @param asBlob Video files can either be loaded via the creation of a video element which has its src property set.
        *               Or they can be loaded via xhr, stored as binary data in memory and then converted to a Blob. This isn't supported in IE9 or Android 2.
        *               If you need to have the same video playing at different times across multiple Sprites then you need to load it as a Blob.
        * @return This Loader instance.
        */
        video(key: string, urls: string | string[] | any, loadEvent?: string, asBlob?: boolean): Phaser.Loader;

        /**
        * Add a synchronization point to the assets / files added within the supplied callback.
        * 
        * A synchronization point denotes that an asset _must_ be completely loaded before
        * subsequent assets can be loaded. An asset marked as a sync-point does not need to wait
        * for previous assets to load (unless they are sync-points). Resources, such as packs, may still
        * be downloaded around sync-points, as long as they do not finalize loading.
        * 
        * @param callback The callback is invoked and is supplied with a single argument: the loader.
        * @param callbackContext Context for the callback. - Default: (loader)
        * @return This Loader instance.
        */
        withSyncPoint(callback: Function, callbackContext?: any): Phaser.Loader;

        /**
        * Adds an XML file to the current load queue.
        * 
        * The file is **not** loaded immediately after calling this method. The file is added to the queue ready to be loaded when the loader starts.
        * 
        * The key must be a unique String. It is used to add the file to the Phaser.Cache upon successful load.
        * 
        * Retrieve the file via `Cache.getXML(key)`.
        * 
        * The URL can be relative or absolute. If the URL is relative the `Loader.baseURL` and `Loader.path` values will be prepended to it.
        * 
        * If the URL isn't specified the Loader will take the key and create a filename from that. For example if the key is "alien"
        * and no URL is given then the Loader will set the URL to be "alien.xml". It will always add `.xml` as the extension.
        * If you do not desire this action then provide a URL.
        * 
        * @param key Unique asset key of the xml file.
        * @param url URL of the XML file. If undefined or `null` the url will be set to `<key>.xml`, i.e. if `key` was "alien" then the URL will be "alien.xml".
        * @param overwrite If an unloaded file with a matching key already exists in the queue, this entry will overwrite it.
        * @return This Loader instance.
        */
        xml(key: string, url?: string, overwrite?: boolean): Phaser.Loader;

        /**
        * Starts the xhr loader.
        * 
        * This is designed specifically to use with asset file processing.
        * 
        * @param file The file/pack to load.
        * @param url The URL of the file.
        * @param type The xhr responseType.
        * @param onload The function to call on success. Invoked in `this` context and supplied with `(file, xhr)` arguments.
        * @param onerror The function to call on error. Invoked in `this` context and supplied with `(file, xhr)` arguments. - Default: fileError
        */
        xhrLoad(file: any, url: string, type: string, onload: Function, onerror?: Function): void;
        xhrLoadWithXDR(file: any, url: string, type: string, onload: Function, onerror?: Function): void;

        /**
        * Successfully loaded an XML file - only used for certain types.
        * 
        * @param file File associated with this request
        * @param xhr
        */
        xmlLoadComplete(file: any, xhr: XMLHttpRequest): void;

    }


    /**
    * Phaser.LoaderParser parses data objects from Phaser.Loader that need more preparation before they can be inserted into the Cache.
    */
    class LoaderParser {


        /**
        * Alias for xmlBitmapFont, for backwards compatibility.
        * 
        * @param xml XML data you want to parse.
        * @param baseTexture The BaseTexture this font uses.
        * @param xSpacing Additional horizontal spacing between the characters.
        * @param ySpacing Additional vertical spacing between the characters.
        * @param frame Optional Frame, if this font is embedded in a texture atlas.
        * @param resolution Optional game resolution to apply to the kerning data. - Default: 1
        * @return The parsed Bitmap Font data.
        */
        static bitmapFont(xml: any, baseTexture: PIXI.BaseTexture, xSpacing?: number, ySpacing?: number, frame?: Phaser.Frame, resolution?: number): any;

        /**
        * Parse a Bitmap Font from an XML file.
        * 
        * @param xml XML data you want to parse.
        * @param baseTexture The BaseTexture this font uses.
        * @param xSpacing Additional horizontal spacing between the characters.
        * @param ySpacing Additional vertical spacing between the characters.
        * @param frame Optional Frame, if this font is embedded in a texture atlas.
        * @param resolution Optional game resolution to apply to the kerning data. - Default: 1
        * @return The parsed Bitmap Font data.
        */
        static xmlBitmapFont(xml: any, baseTexture: PIXI.BaseTexture, xSpacing?: number, ySpacing?: number, frame?: Phaser.Frame, resolution?: number): any;

        /**
        * Parse a Bitmap Font from a JSON file.
        * 
        * @param json JSON data you want to parse.
        * @param baseTexture The BaseTexture this font uses.
        * @param xSpacing Additional horizontal spacing between the characters.
        * @param ySpacing Additional vertical spacing between the characters.
        * @param frame Optional Frame, if this font is embedded in a texture atlas.
        * @param resolution Optional game resolution to apply to the kerning data. - Default: 1
        * @return The parsed Bitmap Font data.
        */
        static jsonBitmapFont(json: any, baseTexture: PIXI.BaseTexture, xSpacing?: number, ySpacing?: number, frame?: Phaser.Frame, resolution?: number): any;

    }


    /**
    * The Matrix is a 3x3 matrix mostly used for display transforms within the renderer.
    * 
    * It is represented like so:
    * 
    * | a | b | tx |
    * | c | d | ty |
    * | 0 | 0 | 1 |
    */
    class Matrix {


        /**
        * 
        * Default: 1
        */
        a: number;

        /**
        * 
        * Default: 0
        */
        b: number;

        /**
        * 
        * Default: 0
        */
        c: number;

        /**
        * 
        * Default: 1
        */
        d: number;

        /**
        * 
        * Default: 0
        */
        tx: number;

        /**
        * 
        * Default: 0
        */
        ty: number;

        /**
        * The const type of this object.
        */
        type: number;


        /**
        * The Matrix is a 3x3 matrix mostly used for display transforms within the renderer.
        * 
        * It is represented like so:
        * 
        * | a | b | tx |
        * | c | d | ty |
        * | 0 | 0 | 1 |
        * 
        * @param a Horizontal scaling - Default: 1
        * @param b Horizontal skewing
        * @param c Vertical skewing
        * @param d Vertical scaling - Default: 1
        * @param tx Horizontal translation
        * @param ty Vertical translation
        */
        constructor(a?: number, b?: number, c?: number, d?: number, tx?: number, ty?: number);


        /**
        * Get a new position with the current transformation applied.
        * 
        * Can be used to go from a childs coordinate space to the world coordinate space (e.g. rendering)
        * 
        * @param pos The origin Point.
        * @param newPos The point that the new position is assigned to. This can be same as input point.
        * @return The new point, transformed through this matrix.
        */
        apply(pos: Phaser.Point, newPos?: Phaser.Point): Phaser.Point;

        /**
        * Get a new position with the inverse of the current transformation applied.
        * 
        * Can be used to go from the world coordinate space to a childs coordinate space. (e.g. input)
        * 
        * @param pos The origin Point.
        * @param newPos The point that the new position is assigned to. This can be same as input point.
        * @return The new point, inverse transformed through this matrix.
        */
        applyInverse(pos: Phaser.Point, newPos?: Phaser.Point): Phaser.Point;

        /**
        * Creates a new Matrix object based on the values of this Matrix.
        * If you provide the output parameter the values of this Matrix will be copied over to it.
        * If the output parameter is blank a new Matrix object will be created.
        * 
        * @param output If provided the values of this Matrix will be copied to it, otherwise a new Matrix object is created.
        * @return A clone of this Matrix.
        */
        clone(output?: Phaser.Matrix): Phaser.Matrix;

        /**
        * Copies the properties from the given Matrix into this Matrix.
        * 
        * @param matrix The Matrix to copy from.
        * @return This Matrix object.
        */
        copyFrom(matrix: Phaser.Matrix): Phaser.Matrix;

        /**
        * Copies the properties from this Matrix to the given Matrix.
        * 
        * @param matrix The Matrix to copy from.
        * @return The destination Matrix object.
        */
        copyTo(matrix: Phaser.Matrix): Phaser.Matrix;

        /**
        * Sets the values of this Matrix to the values in the given array.
        * 
        * The Array elements should be set as follows:
        * 
        * a = array[0]
        * b = array[1]
        * c = array[3]
        * d = array[4]
        * tx = array[2]
        * ty = array[5]
        * 
        * @param array The array to copy from.
        * @return This Matrix object.
        */
        fromArray(array: number[]): Phaser.Matrix;

        /**
        * Sets the values of this Matrix to the given values.
        * 
        * @param a Horizontal scaling
        * @param b Horizontal skewing
        * @param c Vertical skewing
        * @param d Vertical scaling
        * @param tx Horizontal translation
        * @param ty Vertical translation
        * @return This Matrix object.
        */
        setTo(a: number, b: number, c: number, d: number, tx: number, ty: number): Phaser.Matrix;

        /**
        * Creates a Float32 Array with values populated from this Matrix object.
        * 
        * @param transpose Whether the values in the array are transposed or not.
        * @param array If provided the values will be set into this array, otherwise a new Float32Array is created.
        * @return The newly created array which contains the matrix.
        */
        toArray(transpose?: boolean, array?: number[]): number[];

        /**
        * Translates the matrix on the x and y.
        * This is the same as Matrix.tx += x.
        * 
        * @param x The x value to translate on.
        * @param y The y value to translate on.
        * @return This Matrix object.
        */
        translate(x: number, y: number): Phaser.Matrix;

        /**
        * Applies a scale transformation to this matrix.
        * 
        * @param x The amount to scale horizontally.
        * @param y The amount to scale vertically.
        * @return This Matrix object.
        */
        scale(x: number, y: number): Phaser.Matrix;

        /**
        * Applies a rotation transformation to this matrix.
        * 
        * @param angle The angle to rotate by, given in radians.
        * @return This Matrix object.
        */
        rotate(angle: number): Phaser.Matrix;

        /**
        * Appends the given Matrix to this Matrix.
        * 
        * @param matrix The matrix to append to this one.
        * @return This Matrix object.
        */
        append(matrix: Phaser.Matrix): Phaser.Matrix;

        /**
        * Resets this Matrix to an identity (default) matrix.
        * @return This Matrix object.
        */
        identity(): Phaser.Matrix;

    }


    /**
    * A collection of useful mathematical functions.
    * 
    * These are normally accessed through `game.math`.
    */
    class Math {


        /**
        * Find the angle of a segment from (x1, y1) -> (x2, y2).
        * 
        * @param x1 The x coordinate of the first value.
        * @param y1 The y coordinate of the first value.
        * @param x2 The x coordinate of the second value.
        * @param y2 The y coordinate of the second value.
        * @return The angle, in radians.
        */
        static angleBetween(x1: number, y1: number, x2: number, y2: number): number;

        /**
        * Find the angle of a segment from (point1.x, point1.y) -> (point2.x, point2.y).
        * 
        * @param point1 The first point.
        * @param point2 The second point.
        * @return The angle between the two points, in radians.
        */
        static angleBetweenPoints(point1: Phaser.Point, point2: Phaser.Point): number;

        /**
        * Find the angle of a segment from (x1, y1) -> (x2, y2).
        * 
        * The difference between this method and Math.angleBetween is that this assumes the y coordinate travels
        * down the screen.
        * 
        * @param x1 The x coordinate of the first value.
        * @param y1 The y coordinate of the first value.
        * @param x2 The x coordinate of the second value.
        * @param y2 The y coordinate of the second value.
        * @return The angle, in radians.
        */
        static angleBetweenY(x1: number, y1: number, x2: number, y2: number): number;

        /**
        * Find the angle of a segment from (point1.x, point1.y) -> (point2.x, point2.y).
        * 
        * @param point1
        * @param point2
        * @return The angle, in radians.
        */
        static angleBetweenPointsY(point1: Phaser.Point, point2: Phaser.Point): number;

        /**
        * Averages all values passed to the function and returns the result.
        * @return The average of all given values.
        */
        static average(...numbers: number[]): number;

        /**
        * 
        * 
        * @param n
        * @param i
        */
        static bernstein(n: number, i: number): number;

        /**
        * Returns a random float in the range `[min, max)`. If these parameters are not in order than they will be put in order.
        * Default is 0 for `min` and 1 for `max`.
        * 
        * @param min The minimum value. Must be a Number.
        * @param max The maximum value. Must be a Number.
        * @return A floating point number between min (inclusive) and max (exclusive).
        */
        static random(min: number, max: number): number;

        /**
        * Returns a random integer in the range `[min, max]`. If these parameters are not in order than they will be put in order.
        * Default is 0 for `min` and 1 for `max`.
        * 
        * @param min The minimum value. Must be a Number.
        * @param max The maximum value. Must be a Number.
        * @return An integer between min (inclusive) and max (inclusive).
        */
        static between(min: number, max: number): number;

        /**
        * A Bezier Interpolation Method, mostly used by Phaser.Tween.
        * 
        * @param v The input array of values to interpolate between.
        * @param k The percentage of interpolation, between 0 and 1.
        * @return The interpolated value
        */
        static bezierInterpolation(v: number[], k: number): number;

        /**
        * Calculates a catmum rom value.
        * 
        * @param p0
        * @param p1
        * @param p2
        * @param p3
        * @param t
        */
        static catmullRom(p0: number, p1: number, p2: number, p3: number, t: number): number;

        /**
        * A Catmull Rom Interpolation Method, mostly used by Phaser.Tween.
        * 
        * @param v The input array of values to interpolate between.
        * @param k The percentage of interpolation, between 0 and 1.
        * @return The interpolated value
        */
        static catmullRomInterpolation(v: number[], k: number): number;

        /**
        * Ceils to some place comparative to a `base`, default is 10 for decimal place.
        * The `place` is represented by the power applied to `base` to get that place.
        * 
        * @param value The value to round.
        * @param place The place to round to.
        * @param base The base to round in. Default is 10 for decimal. - Default: 10
        * @return The rounded value.
        */
        static ceilTo(value: number, place?: number, base?: number): number;

        /**
        * Force a value within the boundaries by clamping it to the range `min`, `max`.
        * 
        * @param v The value to be clamped.
        * @param min The minimum bounds.
        * @param max The maximum bounds.
        * @return The clamped value.
        */
        static clamp(x: number, a: number, b: number): number;

        /**
        * Clamp `x` to the range `[a, Infinity)`.
        * Roughly the same as `Math.max(x, a)`, except for NaN handling.
        * 
        * @param x
        * @param a
        */
        static clampBottom(x: number, a: number): number;

        /**
        * Convert degrees to radians.
        * 
        * @param degrees Angle in degrees.
        * @return Angle in radians.
        */
        static degToRad(degrees: number): number;

        /**
        * The absolute difference between two values.
        * 
        * @param a The first value to check.
        * @param b The second value to check.
        * @return The absolute difference between the two values.
        */
        static difference(a: number, b: number): number;

        /**
        * Returns the euclidian distance between the two given set of coordinates.
        * 
        * @param x1
        * @param y1
        * @param x2
        * @param y2
        * @return The distance between the two sets of coordinates.
        */
        static distance(x1: number, y1: number, x2: number, y2: number): number;

        /**
        * Returns the euclidean distance squared between the two given set of
        * coordinates (cuts out a square root operation before returning).
        * 
        * @param x1
        * @param y1
        * @param x2
        * @param y2
        * @return The distance squared between the two sets of coordinates.
        */
        static distanceSq(x1: number, y1: number, x2: number, y2: number): number;

        /**
        * Returns the distance between the two given set of coordinates at the power given.
        * 
        * @param x1
        * @param y1
        * @param x2
        * @param y2
        * @param pow  - Default: 2
        * @return The distance between the two sets of coordinates.
        */
        static distancePow(xy: number, y1: number, x2: number, y2: number, pow?: number): number;

        /**
        * 
        * 
        * @param value the number you want to evaluate
        */
        static factorial(value: number): number;

        /**
        * Floors to some place comparative to a `base`, default is 10 for decimal place.
        * The `place` is represented by the power applied to `base` to get that place.
        * 
        * @param value The value to round.
        * @param place The place to round to.
        * @param base The base to round in. Default is 10 for decimal. - Default: 10
        * @return The rounded value.
        */
        static floorTo(value: number, place: number, base: number): number;

        /**
        * Applies a fuzzy ceil to the given value.
        * 
        * @param val The value to ceil.
        * @param epsilon The epsilon (a small value used in the calculation) - Default: 0.0001
        * @return ceiling(val-epsilon)
        */
        static fuzzyCeil(val: number, epsilon?: number): boolean;

        /**
        * Two number are fuzzyEqual if their difference is less than epsilon.
        * 
        * @param a The first number to compare.
        * @param b The second number to compare.
        * @param epsilon The epsilon (a small value used in the calculation) - Default: 0.0001
        * @return True if |a-b|<epsilon
        */
        static fuzzyEqual(a: number, b: number, epsilon?: number): boolean;

        /**
        * `a` is fuzzyLessThan `b` if it is less than b + epsilon.
        * 
        * @param a The first number to compare.
        * @param b The second number to compare.
        * @param epsilon The epsilon (a small value used in the calculation) - Default: 0.0001
        * @return True if a<b+epsilon
        */
        static fuzzyLessThan(a: Number, b: number, epsilon?: number): boolean;

        /**
        * Applies a fuzzy floor to the given value.
        * 
        * @param val The value to floor.
        * @param epsilon The epsilon (a small value used in the calculation) - Default: 0.0001
        * @return floor(val+epsilon)
        */
        static fuzzyFloor(val: number, epsilon?: number): boolean;

        /**
        * `a` is fuzzyGreaterThan `b` if it is more than b - epsilon.
        * 
        * @param a The first number to compare.
        * @param b The second number to compare.
        * @param epsilon The epsilon (a small value used in the calculation) - Default: 0.0001
        * @return True if a>b+epsilon
        */
        static fuzzyGreaterThan(a: number, b: number, epsilon?: number): boolean;

        /**
        * `a` is fuzzyLessThan `b` if it is less than b + epsilon.
        * 
        * @param a The first number to compare.
        * @param b The second number to compare.
        * @param epsilon The epsilon (a small value used in the calculation) - Default: 0.0001
        * @return True if a<b+epsilon
        */
        static fuzzyLessThan(a: number, b: number, epsilon?: number): boolean;

        /**
        * Gets the shortest angle between `angle1` and `angle2`.
        * Both angles must be in the range -180 to 180, which is the same clamped
        * range that `sprite.angle` uses, so you can pass in two sprite angles to
        * this method, and get the shortest angle back between the two of them.
        * 
        * The angle returned will be in the same range. If the returned angle is
        * greater than 0 then it's a counter-clockwise rotation, if < 0 then it's
        * a clockwise rotation.
        * 
        * @param angle1 The first angle. In the range -180 to 180.
        * @param angle2 The second angle. In the range -180 to 180.
        * @return The shortest angle, in degrees. If greater than zero it's a counter-clockwise rotation.
        */
        static getShortestAngle(angle1: number, angle2: number): number;

        /**
        * Given a number, this function returns the closest number that is a power of two.
        * This function is from the Starling Framework.
        * 
        * @param value The value to get the closest power of two from.
        * @return The closest number that is a power of two.
        */
        static getNextPowerOfTwo(value: number): number;

        /**
        * Half PI.
        * Default: ~1.570
        */
        static HALF_PI: number;

        /**
        * Returns true if the number given is even.
        * 
        * @param n The number to check.
        * @return True if the given number is even. False if the given number is odd.
        */
        static isEven(n: number): boolean;

        /**
        * Returns true if the number given is odd.
        * 
        * @param n The number to check.
        * @return True if the given number is odd. False if the given number is even.
        */
        static isOdd(n: number): boolean;

        /**
        * Checks if the given dimensions make a power of two texture.
        * 
        * @param width The width to check.
        * @param height The height to check.
        * @return True if the width and height are a power of two.
        */
        static isPowerOfTwo(width: number, height: number): boolean;

        /**
        * Calculates a linear (interpolation) value over t.
        * 
        * @param p0
        * @param p1
        * @param t A value between 0 and 1.
        */
        static linear(p0: number, p1: number, t: number): number;

        /**
        * A Linear Interpolation Method, mostly used by Phaser.Tween.
        * 
        * @param v The input array of values to interpolate between.
        * @param k The percentage of interpolation, between 0 and 1.
        * @return The interpolated value
        */
        static linearInterpolation(v: number[], k: number): number;

        /**
        * Linear mapping from range <a1, a2> to range <b1, b2>
        * 
        * @param x The value to map
        * @param a1 First endpoint of the range <a1, a2>
        * @param a2 Final endpoint of the range <a1, a2>
        * @param b1 First endpoint of the range <b1, b2>
        * @param b2 Final endpoint of the range  <b1, b2>
        */
        static mapLinear(x: number, a1: number, a2: number, b1: number, b2: number): number;

        /**
        * Variation of Math.max that can be passed either an array of numbers or the numbers as parameters.
        * 
        * Prefer the standard `Math.max` function when appropriate.
        * @return The largest value from those given.
        */
        static max(...numbers: number[]): number;

        /**
        * Adds the given amount to the value, but never lets the value go over the specified maximum.
        * 
        * @param value The value to add the amount to.
        * @param amount The amount to add to the value.
        * @param max The maximum the value is allowed to be.
        * @return The new value.
        */
        static maxAdd(value: number, amount: number, max: number): number;

        /**
        * Variation of Math.max that can be passed a property and either an array of objects or the objects as parameters.
        * It will find the largest matching property value from the given objects.
        * @return The largest value from those given.
        */
        static maxProperty(...numbers: number[]): number;

        /**
        * Variation of Math.min that can be passed either an array of numbers or the numbers as parameters.
        * 
        * Prefer the standard `Math.min` function when appropriate.
        * @return The lowest value from those given.
        */
        static min(...numbers: number[]): number;

        /**
        * Variation of Math.min that can be passed a property and either an array of objects or the objects as parameters.
        * It will find the lowest matching property value from the given objects.
        * @return The lowest value from those given.
        */
        static minProperty(...numbers: number[]): number;

        /**
        * Subtracts the given amount from the value, but never lets the value go below the specified minimum.
        * 
        * @param value The base value.
        * @param amount The amount to subtract from the base value.
        * @param min The minimum the value is allowed to be.
        * @return The new value.
        */
        static minSub(value: number, amount: number, min: number): number;

        /**
        * Normalizes an angle to the [0,2pi) range.
        * 
        * @param angleRad The angle to normalize, in radians.
        * @return The angle, fit within the [0,2pi] range, in radians.
        */
        static normalizeAngle(angle: number, radians?: boolean): number;

        /**
        * Work out what percentage value `a` is of value `b` using the given base.
        * 
        * @param a The value to work out the percentage for.
        * @param b The value you wish to get the percentage of.
        * @param base The base value.
        * @return The percentage a is of b, between 0 and 1.
        */
        static percent(a: number, b: number, base?: number): number;
        static p2px(v: number): number;

        /**
        * Twice PI.
        * Default: ~6.283
        */
        static PI2: number;

        /**
        * Convert radians to degrees.
        * 
        * @param radians Angle in radians.
        * @return Angle in degrees
        */
        static radToDeg(radians: number): number;

        /**
        * Reverses an angle.
        * 
        * @param angleRad The angle to reverse, in radians.
        * @return The reverse angle, in radians.
        */
        static reverseAngle(angleRed: number): number;

        /**
        * Rotates currentAngle towards targetAngle, taking the shortest rotation distance.
        * The lerp argument is the amount to rotate by in this call.
        * 
        * @param currentAngle The current angle, in radians.
        * @param targetAngle The target angle to rotate to, in radians.
        * @param lerp The lerp value to add to the current angle. - Default: 0.05
        * @return The adjusted angle.
        */
        static rotateToAngle(currentAngle: number, targetAngle: number, lerp?: number): number;

        /**
        * Round to the next whole number _away_ from zero.
        * 
        * @param value Any number.
        * @return The rounded value of that number.
        */
        static roundAwayFromZero(value: number): number;

        /**
        * Round to some place comparative to a `base`, default is 10 for decimal place.
        * The `place` is represented by the power applied to `base` to get that place.
        * 
        *     e.g. 2000/7 ~= 285.714285714285714285714 ~= (bin)100011101.1011011011011011
        * 
        *     roundTo(2000/7,3) === 0
        *     roundTo(2000/7,2) == 300
        *     roundTo(2000/7,1) == 290
        *     roundTo(2000/7,0) == 286
        *     roundTo(2000/7,-1) == 285.7
        *     roundTo(2000/7,-2) == 285.71
        *     roundTo(2000/7,-3) == 285.714
        *     roundTo(2000/7,-4) == 285.7143
        *     roundTo(2000/7,-5) == 285.71429
        * 
        *     roundTo(2000/7,3,2)  == 288       -- 100100000
        *     roundTo(2000/7,2,2)  == 284       -- 100011100
        *     roundTo(2000/7,1,2)  == 286       -- 100011110
        *     roundTo(2000/7,0,2)  == 286       -- 100011110
        *     roundTo(2000/7,-1,2) == 285.5     -- 100011101.1
        *     roundTo(2000/7,-2,2) == 285.75    -- 100011101.11
        *     roundTo(2000/7,-3,2) == 285.75    -- 100011101.11
        *     roundTo(2000/7,-4,2) == 285.6875  -- 100011101.1011
        *     roundTo(2000/7,-5,2) == 285.71875 -- 100011101.10111
        * 
        * Note what occurs when we round to the 3rd space (8ths place), 100100000, this is to be assumed
        * because we are rounding 100011.1011011011011011 which rounds up.
        * 
        * @param value The value to round.
        * @param place The place to round to.
        * @param base The base to round in. Default is 10 for decimal. - Default: 10
        * @return The rounded value.
        */
        static roundTo(value: number, place?: number, base?: number): number;

        /**
        * 
        * 
        * @param n
        * @return n mod 1
        */
        static shear(n: number): number;

        /**
        * A value representing the sign of the value: -1 for negative, +1 for positive, 0 if value is 0.
        * 
        * This works differently from `Math.sign` for values of NaN and -0, etc.
        * 
        * @param x
        * @return An integer in {-1, 0, 1}
        */
        static sign(x: number): number;

        /**
        * Generate a sine and cosine table simultaneously and extremely quickly.
        * The parameters allow you to specify the length, amplitude and frequency of the wave.
        * This generator is fast enough to be used in real-time.
        * Code based on research by Franky of scene.at
        * 
        * @param length The length of the wave
        * @param sinAmplitude The amplitude to apply to the sine table (default 1.0) if you need values between say -+ 125 then give 125 as the value
        * @param cosAmplitude The amplitude to apply to the cosine table (default 1.0) if you need values between say -+ 125 then give 125 as the value
        * @param frequency The frequency of the sine and cosine table data
        * @return Returns the table data.
        */
        static sinCosGenerator(length: number, sinAmplitude?: number, cosAmplitude?: number, frequency?: number): { sin: number[]; cos: number[]; };

        /**
        * Returns the length of the hypotenuse connecting two segments of given lengths.
        * 
        * @param a
        * @param b
        * @return The length of the hypotenuse connecting the given lengths.
        */
        static hypot(a: number, b: number): number;

        /**
        * Smootherstep function as detailed at http://en.wikipedia.org/wiki/Smoothstep
        * 
        * @param x The input value.
        * @param min The left edge. Should be smaller than the right edge.
        * @param max The right edge.
        * @return A value between 0 and 1.
        */
        static smootherstep(x: number, min: number, max: number): number;

        /**
        * Smoothstep function as detailed at http://en.wikipedia.org/wiki/Smoothstep
        * 
        * @param x The input value.
        * @param min The left edge. Should be smaller than the right edge.
        * @param max The right edge.
        * @return A value between 0 and 1.
        */
        static smoothstep(x: number, min: number, max: number): number;

        /**
        * Snap a value to nearest grid slice, using rounding.
        * 
        * Example: if you have an interval gap of 5 and a position of 12... you will snap to 10 whereas 14 will snap to 15.
        * 
        * @param input The value to snap.
        * @param gap The interval gap of the grid.
        * @param start Optional starting offset for gap.
        * @return The snapped value.
        */
        static snapTo(input: number, gap: number, start?: number): number;

        /**
        * Snap a value to nearest grid slice, using ceil.
        * 
        * Example: if you have an interval gap of 5 and a position of 12... you will snap to 15.
        * As will 14 will snap to 15... but 16 will snap to 20.
        * 
        * @param input The value to snap.
        * @param gap The interval gap of the grid.
        * @param start Optional starting offset for gap.
        * @return The snapped value.
        */
        static snapToCeil(input: number, gap: number, start?: number): number;

        /**
        * Snap a value to nearest grid slice, using floor.
        * 
        * Example: if you have an interval gap of 5 and a position of 12... you will snap to 10.
        * As will 14 snap to 10... but 16 will snap to 15.
        * 
        * @param input The value to snap.
        * @param gap The interval gap of the grid.
        * @param start Optional starting offset for gap.
        * @return The snapped value.
        */
        static snapToFloor(input: number, gap: number, start?: number): number;

        /**
        * Checks if two values are within the given tolerance of each other.
        * 
        * @param a The first number to check
        * @param b The second number to check
        * @param tolerance The tolerance. Anything equal to or less than this is considered within the range.
        * @return True if a is <= tolerance of b.
        */
        static within(a: number, b: number, tolerance: number): boolean;

        /**
        * Ensures that the value always stays between min and max, by wrapping the value around.
        * 
        * If `max` is not larger than `min` the result is 0.
        * 
        * @param value The value to wrap.
        * @param min The minimum the value is allowed to be.
        * @param max The maximum the value is allowed to be, should be larger than `min`.
        * @return The wrapped value.
        */
        static wrap(value: number, min: number, max: number): number;

        /**
        * Keeps an angle value between -180 and +180; or -PI and PI if radians.
        * 
        * @param angle The angle value to wrap
        * @param radians Set to `true` if the angle is given in radians, otherwise degrees is expected.
        * @return The new angle value; will be the same as the input angle if it was within bounds.
        */
        static wrapAngle(angle: number, radians?: boolean): number;

        /**
        * Adds value to amount and ensures that the result always stays between 0 and max, by wrapping the value around.
        * 
        * Values _must_ be positive integers, and are passed through Math.abs. See {@link Phaser.Math#wrap} for an alternative.
        * 
        * @param value The value to add the amount to.
        * @param amount The amount to add to the value.
        * @param max The maximum the value is allowed to be.
        * @return The wrapped value.
        */
        static wrapValue(value: number, amount: number, max: number): number;

    }

    interface WheelEventProxy {

        bindEvent(event: any): WheelEventProxy;

        type: string;
        deltaMode: number;
        deltaX: number;
        deltaY: number;
        deltaZ: number;

    }


    /**
    * The Mouse class is responsible for handling all aspects of mouse interaction with the browser.
    * 
    * It captures and processes mouse events that happen on the game canvas object.
    * It also adds a single `mouseup` listener to `window` which is used to capture the mouse being released
    * when not over the game.
    * 
    * You should not normally access this class directly, but instead use a Phaser.Pointer object
    * which normalises all game input for you, including accurate button handling.
    */
    class Mouse {


        /**
        * The Mouse class is responsible for handling all aspects of mouse interaction with the browser.
        * 
        * It captures and processes mouse events that happen on the game canvas object.
        * It also adds a single `mouseup` listener to `window` which is used to capture the mouse being released
        * when not over the game.
        * 
        * You should not normally access this class directly, but instead use a Phaser.Pointer object
        * which normalises all game input for you, including accurate button handling.
        * 
        * @param game A reference to the currently running game.
        */
        constructor(game: Phaser.Game);

        static NO_BUTTON: number;
        static LEFT_BUTTON: number;
        static MIDDLE_BUTTON: number;
        static RIGHT_BUTTON: number;
        static BACK_BUTTON: number;
        static FORWARD_BUTTON: number;
        static WHEEL_DOWN: number;
        static WHEEL_UP: number;


        /**
        * This property was removed in Phaser 2.4 and should no longer be used.
        * Instead please see the Pointer button properties such as `Pointer.leftButton`, `Pointer.rightButton` and so on.
        * Or Pointer.button holds the DOM event button value if you require that.
        */
        button: number;

        /**
        * The context under which callbacks are called.
        */
        callbackContext: any;

        /**
        * If true the DOM mouse events will have event.preventDefault applied to them, if false they will propagate fully.
        */
        capture: boolean;

        /**
        * Mouse input will only be processed if enabled.
        * Default: true
        */
        enabled: boolean;

        /**
        * The browser mouse DOM event. Will be null if no mouse event has ever been received.
        * Access this property only inside a Mouse event handler and do not keep references to it.
        */
        event: MouseEvent;

        /**
        * A reference to the currently running game.
        */
        game: Phaser.Game;

        /**
        * A reference to the Phaser Input Manager.
        */
        input: Phaser.Input;

        /**
        * If the mouse has been Pointer Locked successfully this will be set to true.
        */
        locked: boolean;

        /**
        * A callback that can be fired when the mouse is pressed down.
        */
        mouseDownCallback: (event: MouseEvent) => void;

        /**
        * A callback that can be fired when the mouse is no longer over the game canvas.
        */
        mouseOutCallback: (event: MouseEvent) => void;

        /**
        * A callback that can be fired when the mouse enters the game canvas (usually after a mouseout).
        */
        mouseOverCallback: (event: MouseEvent) => void;

        /**
        * A callback that can be fired when the mouse is released from a pressed down state.
        */
        mouseUpCallback: (event: MouseEvent) => void;

        /**
        * A callback that can be fired when the mousewheel is used.
        */
        mouseWheelCallback: (event: MouseEvent) => void;

        /**
        * Internal event handler reference.
        */
        _onMouseDown: (event: MouseEvent) => void;

        /**
        * Internal event handler reference.
        */
        _onMouseMove: (event: MouseEvent) => void;

        /**
        * Internal event handler reference.
        */
        _onMouseUp: (event: MouseEvent) => void;

        /**
        * Internal event handler reference.
        */
        _onMouseOut: (event: MouseEvent) => void;

        /**
        * Internal event handler reference.
        */
        _onMouseOver: (event: MouseEvent) => void;

        /**
        * Internal event handler reference.
        */
        _onMouseWheel: (event: MouseEvent) => void;

        /**
        * Wheel proxy event object, if required. Shared for all wheel events for this mouse.
        */
        _wheelEvent: WheelEventProxy;

        /**
        * This event is dispatched when the browser enters or leaves pointer lock state.
        */
        pointerLock: Phaser.Signal;

        /**
        * If true Pointer.stop will be called if the mouse leaves the game canvas.
        */
        stopOnGameOut: boolean;

        /**
        * The direction of the _last_ mousewheel usage 1 for up -1 for down.
        */
        wheelDelta: number;


        /**
        * The internal method that handles the mouse down event from the browser.
        * 
        * @param event The native event from the browser. This gets stored in Mouse.event.
        */
        onMouseDown(event: MouseEvent): void;

        /**
        * The internal method that handles the mouse move event from the browser.
        * 
        * @param event The native event from the browser. This gets stored in Mouse.event.
        */
        onMouseMove(event: MouseEvent): void;

        /**
        * The internal method that handles the mouse out event from the browser.
        * 
        * @param event The native event from the browser. This gets stored in Mouse.event.
        */
        onMouseOut(event: MouseEvent): void;

        /**
        * The internal method that handles the mouse over event from the browser.
        * 
        * @param event The native event from the browser. This gets stored in Mouse.event.
        */
        onMouseOver(event: MouseEvent): void;

        /**
        * The internal method that handles the mouse up event from the browser.
        * 
        * @param event The native event from the browser. This gets stored in Mouse.event.
        */
        onMouseUp(event: MouseEvent): void;

        /**
        * The internal method that handles the mouse up event from the window.
        * 
        * @param event The native event from the browser. This gets stored in Mouse.event.
        */
        onMouseUpGlobal(event: MouseEvent): void;

        /**
        * The internal method that handles the mouse wheel event from the browser.
        * 
        * @param event The native event from the browser.
        */
        onMouseWheel(event: MouseEvent): void;

        /**
        * Internal pointerLockChange handler.
        * 
        * @param event The native event from the browser. This gets stored in Mouse.event.
        */
        pointerLockChange(event: MouseEvent): void;

        /**
        * Internal release pointer lock handler.
        */
        releasePointerLock(): void;

        /**
        * If the browser supports it you can request that the pointer be locked to the browser window.
        * This is classically known as 'FPS controls', where the pointer can't leave the browser until the user presses an exit key.
        * If the browser successfully enters a locked state the event Phaser.Mouse.pointerLock will be dispatched and the first parameter will be 'true'.
        */
        requestPointerLock(): void;

        /**
        * Starts the event listeners running.
        */
        start(): void;

        /**
        * Stop the event listeners.
        */
        stop(): void;

    }


    /**
    * The MSPointer class handles Microsoft touch interactions with the game and the resulting Pointer objects.
    * 
    * It will work only in Internet Explorer 10+ and Windows Store or Windows Phone 8 apps using JavaScript.
    * http://msdn.microsoft.com/en-us/library/ie/hh673557(v=vs.85).aspx
    * 
    * You should not normally access this class directly, but instead use a Phaser.Pointer object which
    * normalises all game input for you including accurate button handling.
    * 
    * Please note that at the current time of writing Phaser does not yet support chorded button interactions:
    * http://www.w3.org/TR/pointerevents/#chorded-button-interactions
    */
    class MSPointer {


        /**
        * The MSPointer class handles Microsoft touch interactions with the game and the resulting Pointer objects.
        * 
        * It will work only in Internet Explorer 10+ and Windows Store or Windows Phone 8 apps using JavaScript.
        * http://msdn.microsoft.com/en-us/library/ie/hh673557(v=vs.85).aspx
        * 
        * You should not normally access this class directly, but instead use a Phaser.Pointer object which
        * normalises all game input for you including accurate button handling.
        * 
        * Please note that at the current time of writing Phaser does not yet support chorded button interactions:
        * http://www.w3.org/TR/pointerevents/#chorded-button-interactions
        * 
        * @param game A reference to the currently running game.
        */
        constructor(game: Phaser.Game);


        /**
        * This property was removed in Phaser 2.4 and should no longer be used.
        * Instead please see the Pointer button properties such as `Pointer.leftButton`, `Pointer.rightButton` and so on.
        * Or Pointer.button holds the DOM event button value if you require that.
        */
        button: number;

        /**
        * If true the Pointer events will have event.preventDefault applied to them, if false they will propagate fully.
        */
        capture: boolean;

        /**
        * The context under which callbacks are called (defaults to game).
        */
        callbackContext: any;

        /**
        * The browser MSPointer DOM event. Will be null if no event has ever been received.
        * Access this property only inside a Pointer event handler and do not keep references to it.
        */
        event: MSPointerEvent;

        /**
        * A reference to the currently running game.
        */
        game: Phaser.Game;

        /**
        * A reference to the Phaser Input Manager.
        */
        input: Phaser.Input;

        onPointerDown: (event: MSPointerEvent) => void;
        onPointerMove: (event: MSPointerEvent) => void;
        onPointerUp: (event: MSPointerEvent) => void;
        mouseDownCallback: (event: MSPointerEvent) => void;
        mouseMoveCallback: (event: MSPointerEvent) => void;
        mouseUpCallback: (event: MSPointerEvent) => void;

        /**
        * A callback that can be fired on a MSPointerDown event.
        */
        pointerDownCallback: (event: MSPointerEvent) => void;

        /**
        * A callback that can be fired on a MSPointerMove event.
        */
        pointerMoveCallback: (event: MSPointerEvent) => void;

        /**
        * A callback that can be fired on a MSPointerUp event.
        */
        pointerUpCallback: (event: MSPointerEvent) => void;


        /**
        * Starts the event listeners running.
        */
        start(): void;

        /**
        * Stop the event listeners.
        */
        stop(): void;

    }


    /**
    * Phaser.Net handles browser URL related tasks such as checking host names, domain names and query string manipulation.
    */
    class Net {


        /**
        * Phaser.Net handles browser URL related tasks such as checking host names, domain names and query string manipulation.
        * 
        * @param game A reference to the currently running game.
        */
        constructor(game: Phaser.Game);

        game: Phaser.Game;


        /**
        * Compares the given domain name against the hostname of the browser containing the game.
        * If the domain name is found it returns true.
        * You can specify a part of a domain, for example 'google' would match 'google.com', 'google.co.uk', etc.
        * Do not include 'http://' at the start.
        * 
        * @param domain
        * @return true if the given domain fragment can be found in the window.location.hostname
        */
        checkDomainName(domain: string): boolean;

        /**
        * Takes a Uniform Resource Identifier (URI) component (previously created by encodeURIComponent or by a similar routine) and
        * decodes it, replacing \ with spaces in the return. Used internally by the Net classes.
        * 
        * @param value The URI component to be decoded.
        * @return The decoded value.
        */
        decodeURI(value: string): string;

        /**
        * Returns the hostname given by the browser.
        */
        getHostName(): string;

        /**
        * Returns the Query String as an object.
        * If you specify a parameter it will return just the value of that parameter, should it exist.
        * 
        * @param parameter If specified this will return just the value for that key. - Default: ''
        * @return An object containing the key value pairs found in the query string or just the value if a parameter was given.
        */
        getQueryString(parameter?: string): string;

        /**
        * Updates a value on the Query String and returns it in full.
        * If the value doesn't already exist it is set.
        * If the value exists it is replaced with the new value given. If you don't provide a new value it is removed from the query string.
        * Optionally you can redirect to the new url, or just return it as a string.
        * 
        * @param key The querystring key to update.
        * @param value The new value to be set. If it already exists it will be replaced.
        * @param redirect If true the browser will issue a redirect to the url with the new querystring.
        * @param url The URL to modify. If none is given it uses window.location.href.
        * @return If redirect is false then the modified url and query string is returned.
        */
        updateQueryString(key: string, value: any, redirect?: boolean, url?: string): string;

    }


    /**
    * Create a new `Particle` object. Particles are extended Sprites that are emitted by a particle emitter such as Phaser.Particles.Arcade.Emitter.
    */
    class Particle extends Phaser.Sprite {


        /**
        * Create a new `Particle` object. Particles are extended Sprites that are emitted by a particle emitter such as Phaser.Particles.Arcade.Emitter.
        * 
        * @param game A reference to the currently running game.
        * @param x The x coordinate (in world space) to position the Particle at.
        * @param y The y coordinate (in world space) to position the Particle at.
        * @param key This is the image or texture used by the Particle during rendering. It can be a string which is a reference to the Cache entry, or an instance of a RenderTexture or PIXI.Texture.
        * @param frame If this Particle is using part of a sprite sheet or texture atlas you can specify the exact frame to use by giving a string or numeric index.
        */
        constructor(game: Phaser.Game, x: number, y: number, key?: any, frame?: any);


        /**
        * A Game Object is considered `fresh` if it has just been created or reset and is yet to receive a renderer transform update.
        * This property is mostly used internally by the physics systems, but is exposed for the use of plugins.
        */
        fresh: boolean;


        /**
        * Called by the Emitter when this particle is emitted. Left empty for you to over-ride as required.
        */
        onEmit(): void;

        /**
        * Resets the Particle. This places the Particle at the given x/y world coordinates and then
        * sets alive, exists, visible and renderable all to true. Also resets the outOfBounds state and health values.
        * If the Particle has a physics body that too is reset.
        * 
        * @param x The x coordinate (in world space) to position the Particle at.
        * @param y The y coordinate (in world space) to position the Particle at.
        * @param health The health to give the Particle. - Default: 1
        * @return This instance.
        */
        reset(x: number, y: number, health?: number): Phaser.Particle;

        /**
        * Called by the Emitter if autoAlpha has been enabled. Passes over the alpha ease data and resets the alpha counter.
        */
        setAlphaData(data: any[]): void;

        /**
        * Called by the Emitter if autoScale has been enabled. Passes over the scale ease data and resets the scale counter.
        */
        setScaleData(data: any[]): void;

        /**
        * Updates the Particle scale or alpha if autoScale and autoAlpha are set.
        */
        update(): void;

    }


    /**
    * Phaser.Particles is the Particle Manager for the game. It is called during the game update loop and in turn updates any Emitters attached to it.
    */
    class Particles {


        /**
        * Phaser.Particles is the Particle Manager for the game. It is called during the game update loop and in turn updates any Emitters attached to it.
        * 
        * @param game A reference to the currently running game.
        */
        constructor(game: Phaser.Game);


        /**
        * Internal emitters store.
        */
        emitters: any;

        /**
        * A reference to the currently running Game.
        */
        game: Phaser.Game;

        /**
        * -
        */
        ID: number;


        /**
        * Adds a new Particle Emitter to the Particle Manager.
        * 
        * @param emitter The emitter to be added to the particle manager.
        * @return The emitter that was added.
        */
        add(emitter: Phaser.Particles.Arcade.Emitter): Phaser.Particles.Arcade.Emitter;

        /**
        * Removes an existing Particle Emitter from the Particle Manager.
        * 
        * @param emitter The emitter to remove.
        */
        remove(emitter: Phaser.Particles.Arcade.Emitter): void;

        /**
        * Updates all Emitters who have their exists value set to true.
        * 
        * Phaser no longer uses this method; Emitters receive updates via {@link Phaser.Stage#update} instead.
        */
        update(): void;

    }

    module Particles {

        module Arcade {


            /**
            * Emitter is a lightweight particle emitter that uses Arcade Physics.
            * It can be used for one-time explosions or for continuous effects like rain and fire.
            * All it really does is launch Particle objects out at set intervals, and fixes their positions and velocities accordingly.
            */
            class Emitter extends Phaser.Group {


                /**
                * Emitter is a lightweight particle emitter that uses Arcade Physics.
                * It can be used for one-time explosions or for continuous effects like rain and fire.
                * All it really does is launch Particle objects out at set intervals, and fixes their positions and velocities accordingly.
                * 
                * @param game Current game instance.
                * @param x The x coordinate within the Emitter that the particles are emitted from.
                * @param y The y coordinate within the Emitter that the particles are emitted from.
                * @param maxParticles The total number of particles in this emitter. - Default: 50
                */
                constructor(game: Phaser.Game, x?: number, y?: number, maxParticles?: number);


                /**
                * An array of the calculated alpha easing data applied to particles with alphaRates > 0.
                */
                alphaData: any[];

                /**
                * When a new Particle is emitted this controls if it will automatically change alpha. Use Emitter.setAlpha to configure.
                */
                autoAlpha: boolean;

                /**
                * When a new Particle is emitted this controls if it will automatically scale in size. Use Emitter.setScale to configure.
                */
                autoScale: boolean;

                /**
                * The angle of rotation of the group container, in degrees.
                * 
                * This adjusts the group itself by modifying its local rotation transform.
                * 
                * This has no impact on the rotation/angle properties of the children, but it will update their worldTransform
                * and on-screen orientation and position.
                */
                angle: number;

                /**
                * The angular drag component of particles launched from the emitter if they are rotating.
                */
                angularDrag: number;

                /**
                * Gets the bottom position of the Emitter.
                */
                bottom: number;

                /**
                * How much each particle should bounce on each axis. 1 = full bounce, 0 = no bounce.
                */
                bounce: Phaser.Point;
                count: {emitted: number; failed: number; totalEmitted: number; totalFailed: number};

                /**
                * The point the particles are emitted from.
                * Emitter.x and Emitter.y control the containers location, which updates all current particles
                * Emitter.emitX and Emitter.emitY control the emission location relative to the x/y position.
                */
                emitX: number;

                /**
                * The point the particles are emitted from.
                * Emitter.x and Emitter.y control the containers location, which updates all current particles
                * Emitter.emitX and Emitter.emitY control the emission location relative to the x/y position.
                */
                emitY: number;

                /**
                * If exists is false the group will be excluded from collision checks and filters such as {@link Phaser.Group#forEachExists forEachExists}. The group will not call `preUpdate` and `postUpdate` on its children and the children will not receive physics updates or camera/world boundary checks. The group will still be {@link Phaser.Group#visible visible} and will still call `update` on its children.
                * Default: true
                */
                exists: boolean;

                /**
                * How often a particle is emitted in ms (if emitter is started with Explode === false).
                * Default: 100
                */
                frequency: number;

                /**
                * Sets the `body.gravity` of each particle sprite to this on launch.
                */
                gravity: Phaser.Point;
                group: Phaser.Group;

                /**
                * Gets or sets the height of the Emitter. This is the region in which a particle can be emitted.
                */
                height: number;

                /**
                * Gets the left position of the Emitter.
                */
                left: number;

                /**
                * How long each particle lives once it is emitted in ms. Default is 2 seconds. Set lifespan to 'zero' for particles to live forever.
                * Default: 2000
                */
                lifespan: number;

                /**
                * The number of particles released during one particle's {@link Phaser.Particles.Arcade.Emitter#lifespan lifespan}, after calling {@link Phaser.Particles.Arcade.Emitter#flow flow}.
                */
                lifespanOutput: number;

                /**
                * The maximum angle of initial particle velocities, in degrees. When set to a non-null value (with {@link Phaser.Particles.Arcade.Emitter#minAngle minAngle}), {@link Phaser.Particles.Arcade.Emitter#minSpeed minSpeed} and {@link Phaser.Particles.Arcade.Emitter#maxSpeed maxSpeed} are used and {@link Phaser.Particles.Arcade.Emitter#minParticleSpeed minParticleSpeed} and {@link Phaser.Particles.Arcade.Emitter#maxParticleSpeed maxParticleSpeed} are ignored.
                */
                maxAngle: number;

                /**
                * The total number of particles in this emitter.
                */
                maxParticles: number;

                /**
                * The maximum possible scale of a particle. This is applied to the X and Y axis. If you need to control each axis see maxParticleScaleX.
                * Default: 1
                */
                maxParticleScale: number;

                /**
                * The maximum possible velocity of a particle.
                */
                maxParticleSpeed: Phaser.Point;

                /**
                * The maximum possible angular velocity of a particle.
                * Default: 360
                */
                maxRotation: number;

                /**
                * The maximum initial speed of particles released within {@link Phaser.Particles.Arcade.Emitter#minAngle minAngle} and {@link Phaser.Particles.Arcade.Emitter#maxAngle maxAngle}.
                * Default: 100
                */
                maxSpeed: number;

                /**
                * The minimum angle of initial particle velocities, in degrees. When set to a non-null value (with {@link Phaser.Particles.Arcade.Emitter#maxAngle maxAngle}), {@link Phaser.Particles.Arcade.Emitter#minSpeed minSpeed} and {@link Phaser.Particles.Arcade.Emitter#maxSpeed maxSpeed} are used and {@link Phaser.Particles.Arcade.Emitter#minParticleSpeed minParticleSpeed} and {@link Phaser.Particles.Arcade.Emitter#maxParticleSpeed maxParticleSpeed} are ignored.
                */
                minAngle: number;

                /**
                * The minimum possible scale of a particle. This is applied to the X and Y axis. If you need to control each axis see minParticleScaleX.
                * Default: 1
                */
                minParticleScale: number;

                /**
                * The minimum possible velocity of a particle.
                */
                minParticleSpeed: Phaser.Point;

                /**
                * The minimum possible angular velocity of a particle.
                */
                minRotation: number;

                /**
                * The minimum initial speed of particles released within {@link Phaser.Particles.Arcade.Emitter#minAngle minAngle} and {@link Phaser.Particles.Arcade.Emitter#maxAngle maxAngle}.
                */
                minSpeed: number;

                /**
                * A handy string name for this emitter. Can be set to anything.
                */
                name: string;

                /**
                * Determines whether the emitter is currently emitting particles. It is totally safe to directly toggle this.
                */
                on: boolean;

                /**
                * The number of particles released per second, after calling {@link Phaser.Particles.Arcade.Emitter#flow flow}.
                */
                output: number;

                /**
                * When a particle is created its anchor will be set to match this Point object (defaults to x/y: 0.5 to aid in rotation)
                */
                particleAnchor: Phaser.Point;

                /**
                * If this is `true` then when the Particle is emitted it will be bought to the top of the Emitters display list.
                */
                particleBringToTop: boolean;

                /**
                * If this is `true` then when the Particle is emitted it will be sent to the back of the Emitters display list.
                */
                particleSendToBack: boolean;

                /**
                * For emitting your own particle class types. They must extend Phaser.Particle.
                */
                particleClass: any;

                /**
                * The X and Y drag component of particles launched from the emitter.
                */
                particleDrag: Phaser.Point;

                /**
                * The const physics body type of this object.
                */
                physicsType: number;

                /**
                * The coordinates, in pixels, of this DisplayObject, relative to its parent container.
                * 
                * The value of this property does not reflect any positioning happening further up the display list.
                * To obtain that value please see the `worldPosition` property.
                */
                position: Phaser.Point;

                /**
                * The expected number of unreleased particles after a flow interval of {@link Phaser.Particles.Arcade.Emitter#lifespan lifespan}, after calling {@link Phaser.Particles.Arcade.Emitter#flow flow}.
                */
                remainder: number;

                /**
                * Gets the right position of the Emitter.
                */
                right: number;

                /**
                * An array of the calculated scale easing data applied to particles with scaleRates > 0.
                */
                scaleData: any[];

                /**
                * Gets the top position of the Emitter.
                */
                top: number;

                /**
                * Internal Phaser Type value.
                */
                type: number;

                /**
                * Gets or sets the width of the Emitter. This is the region in which a particle can be emitted.
                */
                width: number;

                /**
                * Gets or sets the x position of the Emitter.
                */
                x: number;

                /**
                * Gets or sets the y position of the Emitter.
                */
                y: number;


                /**
                * Change the emitters center to match the center of any object with a `center` property, such as a Sprite.
                * If the object doesn't have a center property it will be set to object.x + object.width / 2
                * 
                * @param object The object that you wish to match the center with.
                * @return This Emitter instance.
                */
                at(object: any): Phaser.Particles.Arcade.Emitter;

                /**
                * This function is used internally to emit the next particle in the queue.
                * 
                * However it can also be called externally to emit a particle.
                * 
                * When called externally you can use the arguments to override any defaults the Emitter has set.
                * 
                * @param x The x coordinate to emit the particle from. If `null` or `undefined` it will use `Emitter.emitX` or if the Emitter has a width > 1 a random value between `Emitter.left` and `Emitter.right`.
                * @param y The y coordinate to emit the particle from. If `null` or `undefined` it will use `Emitter.emitY` or if the Emitter has a height > 1 a random value between `Emitter.top` and `Emitter.bottom`.
                * @param key This is the image or texture used by the Particle during rendering. It can be a string which is a reference to the Cache Image entry, or an instance of a RenderTexture, BitmapData, Video or PIXI.Texture.
                * @param frame If this Particle is using part of a sprite sheet or texture atlas you can specify the exact frame to use by giving a string or numeric index.
                * @return True if a particle was emitted, otherwise false.
                */
                emitParticle(x?: number, y?: number, key?: string | Phaser.RenderTexture | Phaser.BitmapData | Phaser.Video | PIXI.Texture, frame?: string | number): boolean;

                /**
                * Call this function to emit the given quantity of particles at all once (an explosion)
                * 
                * @param lifespan How long each particle lives once emitted in ms. 0 = forever.
                * @param quantity How many particles to launch. - Default: this.maxParticles
                * @return This Emitter instance.
                */
                explode(lifespan?: number, quantity?: number): Phaser.Particles.Arcade.Emitter;

                /**
                * Call this function to start emitting a flow of particles at the given frequency.
                * It will carry on going until the total given is reached.
                * Each time the flow is run the quantity number of particles will be emitted together.
                * If you set the total to be 20 and quantity to be 5 then flow will emit 4 times in total (4 x 5 = 20 total)
                * If you set the total to be -1 then no quantity cap is used and it will keep emitting.
                * 
                * @param lifespan How long each particle lives once emitted in ms. 0 = forever.
                * @param frequency Frequency is how often to emit the particles, given in ms. - Default: 250
                * @param quantity How many particles to launch each time the frequency is met. Can never be > Emitter.maxParticles. - Default: 1
                * @param total How many particles to launch in total. If -1 it will carry on indefinitely. - Default: -1
                * @param immediate Should the flow start immediately (true) or wait until the first frequency event? (false) - Default: true
                * @return This Emitter instance.
                */
                flow(lifespan?: number, frequency?: number, quantity?: number, total?: number, immediate?: boolean): Phaser.Particles.Arcade.Emitter;

                /**
                * Call this function to turn off all the particles and the emitter.
                * @return This Emitter instance.
                */
                kill(): Phaser.Particles.Arcade.Emitter;

                /**
                * This function generates a new set of particles for use by this emitter.
                * The particles are stored internally waiting to be emitted via Emitter.start.
                * 
                * @param keys A string or an array of strings that the particle sprites will use as their texture. If an array one is picked at random.
                * @param frames A frame number, or array of frames that the sprite will use. If an array one is picked at random.
                * @param quantity The number of particles to generate. If not given it will use the value of Emitter.maxParticles. If the value is greater than Emitter.maxParticles it will use Emitter.maxParticles as the quantity.
                * @param collide If you want the particles to be able to collide with other Arcade Physics bodies then set this to true.
                * @param collideWorldBounds A particle can be set to collide against the World bounds automatically and rebound back into the World if this is set to true. Otherwise it will leave the World.
                * @param particleArguments Custom arguments to pass to your particle class
                * @return This Emitter instance.
                */
                makeParticles(keys: any, frames?: any, quantity?: number, collide?: boolean, collideWorldBounds?: boolean, particleArguments?: any): Phaser.Particles.Arcade.Emitter;
                reset(x: number, y: number, health?: number): Phaser.Particles;

                /**
                * A more compact way of setting the alpha constraints of the particles.
                * The rate parameter, if set to a value above zero, lets you set the speed at which the Particle change in alpha from min to max.
                * If rate is zero, which is the default, the particle won't change alpha - instead it will pick a random alpha between min and max on emit.
                * 
                * @param min The minimum value for this range. - Default: 1
                * @param max The maximum value for this range. - Default: 1
                * @param rate The rate (in ms) at which the particles will change in alpha from min to max, or set to zero to pick a random alpha between the two.
                * @param ease If you've set a rate > 0 this is the easing formula applied between the min and max values. - Default: Phaser.Easing.Linear.None
                * @param yoyo If you've set a rate > 0 you can set if the ease will yoyo or not (i.e. ease back to its original values)
                * @return This Emitter instance.
                */
                setAlpha(min?: number, max?: number, rate?: number, ease?: (k: number) => number, yoyo?: boolean): Phaser.Particles.Arcade.Emitter;

                /**
                * Sets a radial pattern for emitting particles.
                * 
                * This is a shorthand for setting {@link Phaser.Particles.Arcade.Emitter#minAngle minAngle}, {@link Phaser.Particles.Arcade.Emitter#maxAngle maxAngle}, {@link Phaser.Particles.Arcade.Emitter#minSpeed minSpeed}, and {@link Phaser.Particles.Arcade.Emitter#maxSpeed maxSpeed}.
                * 
                * To remove the radial pattern, use `setAngle(null, null)`.
                * 
                * @param minAngle The minimum angle of initial particle velocities, in degrees.
                * @param maxAngle The maximum angle of initial particle velocities, in degrees.
                * @param minSpeed The minimum initial particle speed.
                * @param maxSpeed The maximum initial particle speed.
                * @return This Emitter instance.
                */
                setAngle(minAngle: number, maxAngle: number, minSpeed?: number, maxSpeed?: number): Phaser.Particles.Arcade.Emitter;

                /**
                * A more compact way of setting the angular velocity constraints of the particles.
                * 
                * @param min The minimum value for this range.
                * @param max The maximum value for this range.
                * @return This Emitter instance.
                */
                setRotation(min?: number, max?: number): Phaser.Particles.Arcade.Emitter;

                /**
                * A more compact way of setting the scale constraints of the particles.
                * The rate parameter, if set to a value above zero, lets you set the speed and ease which the Particle uses to change in scale from min to max across both axis.
                * If rate is zero, which is the default, the particle won't change scale during update, instead it will pick a random scale between min and max on emit.
                * 
                * @param minX The minimum value of Particle.scale.x. - Default: 1
                * @param maxX The maximum value of Particle.scale.x. - Default: 1
                * @param minY The minimum value of Particle.scale.y. - Default: 1
                * @param maxY The maximum value of Particle.scale.y. - Default: 1
                * @param rate The rate (in ms) at which the particles will change in scale from min to max, or set to zero to pick a random size between the two.
                * @param ease If you've set a rate > 0 this is the easing formula applied between the min and max values. - Default: Phaser.Easing.Linear.None
                * @param yoyo If you've set a rate > 0 you can set if the ease will yoyo or not (i.e. ease back to its original values)
                * @return This Emitter instance.
                */
                setScale(minX?: number, maxX?: number, minY?: number, maxY?: number, rate?: number, ease?: (k: number) => number, yoyo?: boolean): Phaser.Particles.Arcade.Emitter;

                /**
                * A more compact way of setting the width and height of the emitter.
                * 
                * @param width The desired width of the emitter (particles are spawned randomly within these dimensions).
                * @param height The desired height of the emitter.
                * @return This Emitter instance.
                */
                setSize(width: number, height: number): Phaser.Particles.Arcade.Emitter;

                /**
                * A more compact way of setting the X velocity range of the emitter.
                * 
                * @param min The minimum value for this range.
                * @param max The maximum value for this range.
                * @return This Emitter instance.
                */
                setXSpeed(min: number, max: number): Phaser.Particles.Arcade.Emitter;

                /**
                * A more compact way of setting the Y velocity range of the emitter.
                * 
                * @param min The minimum value for this range.
                * @param max The maximum value for this range.
                * @return This Emitter instance.
                */
                setYSpeed(min: number, max: number): Phaser.Particles.Arcade.Emitter;

                /**
                * Start emitting particles.
                * 
                * {@link Phaser.Particles.Arcade.Emitter#explode explode} and {@link Phaser.Particles.Arcade.Emitter#flow flow} are simpler methods.
                * 
                * There are two patterns, based on the `explode` argument:
                * 
                * - `start(true, lifespan=0, null, quantity)`
                * - `start(false, lifespan=0, frequency=250, quantity=0)`
                * 
                * When `explode` is true or `forceQuantity` is true, `start` emits `quantity` particles immediately. You should pass a nonzero `quantity`.
                * 
                * When `explode` is false and `forceQuantity` is false, `start` emits 1 particle every interval of `frequency` ms. If `quantity` is not zero, the emitter turns itself off after `quantity` particles have been released. If `quantity` is zero, the emitter keeps emitting particles as long as they are available. To emit more than 1 particle per flow interval, use {@link Phaser.Particles.Arcade.Emitter#flow flow} instead.
                * 
                * `forceQuantity` seems equivalent to `explode` and can probably be avoided.
                * 
                * @param explode Whether the particles should all burst out at once (true) or at the frequency given (false). - Default: true
                * @param lifespan How long each particle lives once emitted in ms. 0 = forever.
                * @param frequency Frequency is how often to emit 1 particle when `explode` is false. Value given in ms. Ignored if `explode` is set to true. - Default: 250
                * @param quantity How many particles to launch in total (not larger than {@link Phaser.Particles.Arcade.Emitter#maxParticles maxParticles}).
                * @param forceQuantity Equivalent to `explodes`.
                * @return This Emitter instance.
                */
                start(explode?: boolean, lifespan?: number, frequency?: number, quantity?: number, forceQuantity?: boolean): Phaser.Particles.Arcade.Emitter;

                /**
                * Called automatically by the game loop, decides when to launch particles and when to "die".
                */
                update(): void;

                /**
                * Handy for bringing game objects "back to life". Just sets alive and exists back to true.
                * @return This Emitter instance.
                */
                revive(): Phaser.Particles.Arcade.Emitter;

            }
        }
    }


    /**
    * The Physics Manager is responsible for looking after all of the running physics systems.
    * Phaser supports 4 physics systems: Arcade Physics, P2, Ninja Physics and Box2D via a commercial plugin.
    * 
    * Game Objects (such as Sprites) can only belong to 1 physics system, but you can have multiple systems active in a single game.
    * 
    * For example you could have P2 managing a polygon-built terrain landscape that an vehicle drives over, while it could be firing bullets that use the
    * faster (due to being much simpler) Arcade Physics system.
    */
    class Physics {


        /**
        * The Physics Manager is responsible for looking after all of the running physics systems.
        * Phaser supports 4 physics systems: Arcade Physics, P2, Ninja Physics and Box2D via a commercial plugin.
        * 
        * Game Objects (such as Sprites) can only belong to 1 physics system, but you can have multiple systems active in a single game.
        * 
        * For example you could have P2 managing a polygon-built terrain landscape that an vehicle drives over, while it could be firing bullets that use the
        * faster (due to being much simpler) Arcade Physics system.
        * 
        * @param game A reference to the currently running game.
        * @param physicsConfig A physics configuration object to pass to the Physics world on creation.
        */
        constructor(game: Phaser.Game, config?: any);

        static ARCADE: number;
        static P2JS: number;
        static NINJA: number;
        static BOX2D: number;
        static CHIPMUNK: number;
        static MATTERJS: number;


        /**
        * The Arcade Physics system.
        */
        arcade: Phaser.Physics.Arcade;

        /**
        * The physics configuration object as passed to the game on creation.
        */
        config: any;

        /**
        * Local reference to game.
        */
        game: Phaser.Game;

        /**
        * The N+ Ninja Physics system.
        */
        ninja: Phaser.Physics.Ninja;

        /**
        * The P2.JS Physics system.
        */
        p2: Phaser.Physics.P2;

        /**
        * The Box2D Physics system.
        */
        box2d: any;


        /**
        * Clears down all active physics systems. This doesn't destroy them, it just clears them of objects and is called when the State changes.
        */
        clear(): void;

        /**
        * Destroys all active physics systems. Usually only called on a Game Shutdown, not on a State swap.
        */
        destroy(): void;

        /**
        * This will create a default physics body on the given game object or array of objects.
        * A game object can only have 1 physics body active at any one time, and it can't be changed until the object is destroyed.
        * It can be for any of the physics systems that have been started:
        * 
        * Phaser.Physics.Arcade - A light weight AABB based collision system with basic separation.
        * Phaser.Physics.P2JS - A full-body advanced physics system supporting multiple object shapes, polygon loading, contact materials, springs and constraints.
        * Phaser.Physics.NINJA - A port of Metanet Softwares N+ physics system. Advanced AABB and Circle vs. Tile collision.
        * Phaser.Physics.BOX2D - A port of https://code.google.com/p/box2d-html5
        * Phaser.Physics.MATTER - A full-body and light-weight advanced physics system (still in development)
        * Phaser.Physics.CHIPMUNK is still in development.
        * 
        * If you require more control over what type of body is created, for example to create a Ninja Physics Circle instead of the default AABB, then see the
        * individual physics systems `enable` methods instead of using this generic one.
        * 
        * @param object The game object to create the physics body on. Can also be an array of objects, a body will be created on every object in the array.
        * @param system The physics system that will be used to create the body. Defaults to Arcade Physics. - Default: Phaser.Physics.ARCADE
        * @param debug Enable the debug drawing for this body. Defaults to false.
        */
        enable(object: any, system?: number, debug?: boolean): void;

        /**
        * Parses the Physics Configuration object passed to the Game constructor and starts any physics systems specified within.
        */
        parseConfig(): void;

        /**
        * preUpdate checks.
        */
        preUpdate(): void;

        /**
        * Resets the active physics system. Called automatically on a Phaser.State swap.
        */
        reset(): void;

        /**
        * Updates the physics bounds to match the world dimensions.
        */
        setBoundsToWorld(): void;

        /**
        * This will create an instance of the requested physics simulation.
        * Phaser.Physics.Arcade is running by default, but all others need activating directly.
        * 
        * You can start the following physics systems:
        * 
        * Phaser.Physics.P2JS - A full-body advanced physics system by Stefan Hedman.
        * Phaser.Physics.NINJA - A port of Metanet Softwares N+ physics system.
        * Phaser.Physics.BOX2D - A commercial Phaser Plugin (see http://phaser.io)
        * 
        * Both Ninja Physics and Box2D require their respective plugins to be loaded before you can start them.
        * They are not bundled into the core Phaser library.
        * 
        * If the physics world has already been created (i.e. in another state in your game) then
        * calling startSystem will reset the physics world, not re-create it. If you need to start them again from their constructors
        * then set Phaser.Physics.p2 (or whichever system you want to recreate) to `null` before calling `startSystem`.
        * 
        * @param system The physics system to start: Phaser.Physics.ARCADE, Phaser.Physics.P2JS, Phaser.Physics.NINJA or Phaser.Physics.BOX2D.
        */
        startSystem(system: number): void;

        /**
        * Updates all running physics systems.
        */
        update(): void;

    }


    /**
    * A Video object that takes a previously loaded Video from the Phaser Cache and handles playback of it.
    * 
    * Alternatively it takes a getUserMedia feed from an active webcam and streams the contents of that to
    * the Video instead (see `startMediaStream` method)
    * 
    * The video can then be applied to a Sprite as a texture. If multiple Sprites share the same Video texture and playback
    * changes (i.e. you pause the video, or seek to a new time) then this change will be seen across all Sprites simultaneously.
    * 
    * Due to a bug in IE11 you cannot play a video texture to a Sprite in WebGL. For IE11 force Canvas mode.
    * 
    * If you need each Sprite to be able to play a video fully independently then you will need one Video object per Sprite.
    * Please understand the obvious performance implications of doing this, and the memory required to hold videos in RAM.
    * 
    * On some mobile browsers such as iOS Safari, you cannot play a video until the user has explicitly touched the screen.
    * This works in the same way as audio unlocking. Phaser will handle the touch unlocking for you, however unlike with audio
    * it's worth noting that every single Video needs to be touch unlocked, not just the first one. You can use the `changeSource`
    * method to try and work around this limitation, but see the method help for details.
    * 
    * Small screen devices, especially iPod and iPhone will launch the video in its own native video player,
    * outside of the Safari browser. There is no way to avoid this, it's a device imposed limitation.
    * 
    * Note: On iOS if you need to detect when the user presses the 'Done' button (before the video ends)
    * then you need to add your own event listener
    */
    export class Video {


        /**
        * A reference to the currently running game.
        */
        game: Phaser.Game;

        /**
        * The key of the Video in the Cache, if stored there. Will be `null` if this Video is using the webcam instead.
        * Default: null
        */
        key: string;

        /**
        * The HTML Video Element that is added to the document.
        */
        video: HTMLVideoElement;
        baseTexture: PIXI.BaseTexture;

        /**
        * The PIXI.Texture.
        */
        texture: PIXI.Texture;

        /**
        * The Frame this video uses for rendering.
        */
        textureFrame: Phaser.Frame;

        /**
        * The const type of this object.
        */
        type: number;

        /**
        * If true this video will never send its image data to the GPU when its dirty flag is true. This only applies in WebGL.
        */
        disableTextureUpload: boolean;
        dirty: boolean;


        /**
        * The current time of the video in seconds. If set the video will attempt to seek to that point in time.
        */
        currentTime: number;

        /**
        * The duration of the video in seconds.
        */
        duration: number;

        /**
        * The progress of this video. This is a value between 0 and 1, where 0 is the start and 1 is the end of the video.
        */
        progress: number;

        /**
        * Gets or sets the muted state of the Video.
        */
        mute: boolean;

        /**
        * Gets or sets the paused state of the Video.
        * If the video is still touch locked (such as on iOS devices) this call has no effect.
        */
        paused: boolean;

        /**
        * Gets or sets the volume of the Video, a value between 0 and 1. The value given is clamped to the range 0 to 1.
        */
        volume: boolean;

        /**
        * Gets or sets the playback rate of the Video. This is the speed at which the video is playing.
        */
        playbackRate: boolean;

        /**
        * True if the video is currently playing (and not paused or ended), otherwise false.
        */
        playing: boolean;

        /**
        * Gets or sets if the Video is set to loop.
        * Please note that at present some browsers (i.e. Chrome) do not support *seamless* video looping.
        * If the video isn't yet set this will always return false.
        */
        loop: boolean;

        /**
        * The width of the video in pixels.
        */
        width: number;

        /**
        * The height of the video in pixels.
        */
        height: number;

        /**
        * The Video Stream data. Only set if this Video is streaming from the webcam via `startMediaStream`.
        */
        videoStream: any;

        /**
        * Is there a streaming video source? I.e. from a webcam.
        */
        isStreaming: boolean;

        /**
        * A snapshot grabbed from the video. This is initially black. Populate it by calling Video.grab().
        * When called the BitmapData is updated with a grab taken from the current video playing or active video stream.
        * If Phaser has been compiled without BitmapData support this property will always be `null`.
        */
        snapshot: Phaser.BitmapData;

        /**
        * The amount of ms allowed to elapsed before the Video.onTimeout signal is dispatched while waiting for webcam access.
        * Default: 15000
        */
        timeout: number;

        /**
        * When starting playback of a video Phaser will monitor its readyState using a setTimeout call.
        * The setTimeout happens once every `Video.retryInterval` ms. It will carry on monitoring the video
        * state in this manner until the `retryLimit` is reached and then abort.
        * Default: 20
        */
        retryLimit: number;

        /**
        * The current retry attempt.
        */
        retry: number;

        /**
        * The number of ms between each retry at monitoring the status of a downloading video.
        * Default: 500
        */
        retryInterval: number;


        /**
        * This signal is dispatched if the user allows access to their webcam.
        */
        onAccess: Phaser.Signal;

        /**
        * This signal is dispatched if an error occurs either getting permission to use the webcam (for a Video Stream) or when trying to play back a video file.
        */
        onError: Phaser.Signal;

        /**
        * This signal is dispatched when the Video starts to play. It sends 3 parameters: a reference to the Video object, if the video is set to loop or not and the playback rate.
        */
        onPlay: Phaser.Signal;

        /**
        * This signal is dispatched when the Video completes playback, i.e. enters an 'ended' state. On iOS specifically it also fires if the user hits the 'Done' button at any point during playback. Videos set to loop will never dispatch this signal.
        */
        onComplete: Phaser.Signal;
        onUpdate: Phaser.Signal;

        /**
        * This signal is dispatched if when asking for permission to use the webcam no response is given within a the Video.timeout limit.
        * This may be because the user has picked `Not now` in the permissions window, or there is a delay in establishing the LocalMediaStream.
        */
        onTimeout: Phaser.Signal;


        /**
        * true if this video is currently locked awaiting a touch event. This happens on some mobile devices, such as iOS.
        */
        touchLocked: boolean;
        complete: () => void;


        /**
        * A Video object that takes a previously loaded Video from the Phaser Cache and handles playback of it.
        * 
        * Alternatively it takes a getUserMedia feed from an active webcam and streams the contents of that to
        * the Video instead (see `startMediaStream` method)
        * 
        * The video can then be applied to a Sprite as a texture. If multiple Sprites share the same Video texture and playback
        * changes (i.e. you pause the video, or seek to a new time) then this change will be seen across all Sprites simultaneously.
        * 
        * Due to a bug in IE11 you cannot play a video texture to a Sprite in WebGL. For IE11 force Canvas mode.
        * 
        * If you need each Sprite to be able to play a video fully independently then you will need one Video object per Sprite.
        * Please understand the obvious performance implications of doing this, and the memory required to hold videos in RAM.
        * 
        * On some mobile browsers such as iOS Safari, you cannot play a video until the user has explicitly touched the screen.
        * This works in the same way as audio unlocking. Phaser will handle the touch unlocking for you, however unlike with audio
        * it's worth noting that every single Video needs to be touch unlocked, not just the first one. You can use the `changeSource`
        * method to try and work around this limitation, but see the method help for details.
        * 
        * Small screen devices, especially iPod and iPhone will launch the video in its own native video player,
        * outside of the Safari browser. There is no way to avoid this, it's a device imposed limitation.
        * 
        * Note: On iOS if you need to detect when the user presses the 'Done' button (before the video ends)
        * then you need to add your own event listener
        * 
        * @param game A reference to the currently running game.
        * @param key The key of the video file in the Phaser.Cache that this Video object will play. Set to `null` or leave undefined if you wish to use a webcam as the source. See `startMediaStream` to start webcam capture.
        * @param url If the video hasn't been loaded then you can provide a full URL to the file here (make sure to set key to null)
        */
        constructor(game: Phaser.Game, key?: string, url?: string);


        /**
        * Updates the given Display Objects so they use this Video as their texture.
        * This will replace any texture they will currently have set.
        * 
        * @param object Either a single Sprite/Image or an Array of Sprites/Images.
        * @return This Video object for method chaining.
        */
        add(object: Phaser.Sprite | Phaser.Sprite[] | Phaser.Image | Phaser.Image[]): Phaser.Video;

        /**
        * Creates a new Phaser.Image object, assigns this Video to be its texture, adds it to the world then returns it.
        * 
        * @param x The x coordinate to place the Image at.
        * @param y The y coordinate to place the Image at.
        * @param anchorX Set the x anchor point of the Image. A value between 0 and 1, where 0 is the top-left and 1 is bottom-right.
        * @param anchorY Set the y anchor point of the Image. A value between 0 and 1, where 0 is the top-left and 1 is bottom-right.
        * @param scaleX The horizontal scale factor of the Image. A value of 1 means no scaling. 2 would be twice the size, and so on. - Default: 1
        * @param scaleY The vertical scale factor of the Image. A value of 1 means no scaling. 2 would be twice the size, and so on. - Default: 1
        * @return The newly added Image object.
        */
        addToWorld(x?: number, y?: number, anchorX?: number, anchorY?: Number, scaleX?: number, scaleY?: number): Phaser.Image;

        /**
        * Creates a new Video element from the given Blob. The Blob must contain the video data in the correct encoded format.
        * This method is typically called by the Phaser.Loader and Phaser.Cache for you, but is exposed publicly for convenience.
        * 
        * @param blob The Blob containing the video data.
        * @return This Video object for method chaining.
        */
        createVideoFromBlob(blob: Blob): Phaser.Video;

        /**
        * Instead of playing a video file this method allows you to stream video data from an attached webcam.
        * 
        * As soon as this method is called the user will be prompted by their browser to "Allow" access to the webcam.
        * If they allow it the webcam feed is directed to this Video. Call `Video.play` to start the stream.
        * 
        * If they block the webcam the onError signal will be dispatched containing the NavigatorUserMediaError
        * or MediaStreamError event.
        * 
        * You can optionally set a width and height for the stream. If set the input will be cropped to these dimensions.
        * If not given then as soon as the stream has enough data the video dimensions will be changed to match the webcam device.
        * You can listen for this with the onChangeSource signal.
        * 
        * @param captureAudio Controls if audio should be captured along with video in the video stream.
        * @param width The width is used to create the video stream. If not provided the video width will be set to the width of the webcam input source.
        * @param height The height is used to create the video stream. If not provided the video height will be set to the height of the webcam input source.
        * @return This Video object for method chaining or false if the device doesn't support getUserMedia.
        */
        startMediaStream(captureAudio?: boolean, width?: number, height?: number): Phaser.Video;

        /**
        * Creates a new Video element from the given URL.
        * 
        * @param url The URL of the video.
        * @param autoplay Automatically start the video?
        * @return This Video object for method chaining.
        */
        createVideoFromURL(url: string, autoplay?: boolean): Phaser.Video;

        /**
        * On some mobile browsers you cannot play a video until the user has explicitly touched the video to allow it.
        * Phaser handles this via the `setTouchLock` method. However if you have 3 different videos, maybe an "Intro", "Start" and "Game Over"
        * split into three different Video objects, then you will need the user to touch-unlock every single one of them.
        * 
        * You can avoid this by using just one Video object and simply changing the video source. Once a Video element is unlocked it remains
        * unlocked, even if the source changes. So you can use this to your benefit to avoid forcing the user to 'touch' the video yet again.
        * 
        * As you'd expect there are limitations. So far we've found that the videos need to be in the same encoding format and bitrate.
        * This method will automatically handle a change in video dimensions, but if you try swapping to a different bitrate we've found it
        * cannot render the new video on iOS (desktop browsers cope better).
        * 
        * When the video source is changed the video file is requested over the network. Listen for the `onChangeSource` signal to know
        * when the new video has downloaded enough content to be able to be played. Previous settings such as the volume and loop state
        * are adopted automatically by the new video.
        * 
        * @param src The new URL to change the video.src to.
        * @param autoplay Should the video play automatically after the source has been updated? - Default: true
        * @return This Video object for method chaining.
        */
        changeSource(src: string, autoplay?: boolean): Phaser.Video;
        connectToMediaStram(video: any, stream: any): Phaser.Video;

        /**
        * Destroys the Video object. This calls `Video.stop` and then `Video.removeVideoElement`.
        * If any Sprites are using this Video as their texture it is up to you to manage those.
        */
        destroy(): void;

        /**
        * Starts this video playing.
        * 
        * If the video is already playing, or has been queued to play with `changeSource` then this method just returns.
        * 
        * @param loop Should the video loop automatically when it reaches the end? Please note that at present some browsers (i.e. Chrome) do not support *seamless* video looping.
        * @param playbackRate The playback rate of the video. 1 is normal speed, 2 is x2 speed, and so on. You cannot set a negative playback rate. - Default: 1
        * @return This Video object for method chaining.
        */
        play(loop?: boolean, playbackRate?: number): Phaser.Video;

        /**
        * Called when the video starts to play. Updates the texture.
        */
        playHandler(): void;

        /**
        * If the game is running in WebGL this will push the texture up to the GPU if it's dirty.
        * This is called automatically if the Video is being used by a Sprite, otherwise you need to remember to call it in your render function.
        * If you wish to suppress this functionality set Video.disableTextureUpload to `true`.
        */
        render(): void;

        /**
        * Removes the Video element from the DOM by calling parentNode.removeChild on itself.
        * Also removes the autoplay and src attributes and nulls the reference.
        */
        removeVideoElement(): void;
        resizeFrame(parent: any, width: number, height: number): void;

        /**
        * Sets the Input Manager touch callback to be Video.unlock.
        * Required for mobile video unlocking. Mostly just used internally.
        */
        setTouchLock(): void;

        /**
        * Grabs the current frame from the Video or Video Stream and renders it to the Video.snapshot BitmapData.
        * 
        * You can optionally set if the BitmapData should be cleared or not, the alpha and the blend mode of the draw.
        * 
        * If you need more advanced control over the grabbing them call `Video.snapshot.copy` directly with the same parameters as BitmapData.copy.
        * 
        * @param clear Should the BitmapData be cleared before the Video is grabbed? Unless you are using alpha or a blend mode you can usually leave this set to false.
        * @param alpha The alpha that will be set on the video before drawing. A value between 0 (fully transparent) and 1, opaque. - Default: 1
        * @param blendMode The composite blend mode that will be used when drawing. The default is no blend mode at all. This is a Canvas globalCompositeOperation value such as 'lighter' or 'xor'.
        * @return A reference to the Video.snapshot BitmapData object for further method chaining.
        */
        grab(clear?: boolean, alpha?: number, blendMode?: string): Phaser.BitmapData;

        /**
        * Stops the video playing.
        * 
        * This removes all locally set signals.
        * 
        * If you only wish to pause playback of the video, to resume at a later time, use `Video.paused = true` instead.
        * If the video hasn't finished downloading calling `Video.stop` will not abort the download. To do that you need to
        * call `Video.destroy` instead.
        * 
        * If you are using a video stream from a webcam then calling Stop will disconnect the MediaStream session and disable the webcam.
        * @return This Video object for method chaining.
        */
        stop(): void;

        /**
        * Enables the video on mobile devices, usually after the first touch.
        * If the SoundManager hasn't been unlocked then this will automatically unlock that as well.
        * Only one video can be pending unlock at any one time.
        */
        unlock(): boolean;

        /**
        * Called automatically if the video source changes and updates the internal texture dimensions.
        * Then dispatches the onChangeSource signal.
        * 
        * @param event The event which triggered the texture update.
        * @param width The new width of the video. If undefined `video.videoWidth` is used.
        * @param height The new height of the video. If undefined `video.videoHeight` is used.
        */
        updateTexture(event?: any, width?: number, height?: number): void;

    }

    module Physics {


        /**
        * The Arcade Physics world. Contains Arcade Physics related collision, overlap and motion methods.
        */
        class Arcade {


            /**
            * A constant used for the sortDirection value.
            * Use this if you don't wish to perform any pre-collision sorting at all, or will manually sort your Groups.
            */
            static SORT_NONE: number;

            /**
            * A constant used for the sortDirection value.
            * Use this if your game world is wide but short and scrolls from the left to the right (i.e. Mario)
            */
            static LEFT_RIGHT: number;

            /**
            * A constant used for the sortDirection value.
            * Use this if your game world is wide but short and scrolls from the right to the left (i.e. Mario backwards)
            */
            static RIGHT_LEFT: number;

            /**
            * A constant used for the sortDirection value.
            * Use this if your game world is narrow but tall and scrolls from the top to the bottom (i.e. Dig Dug)
            */
            static TOP_BOTTOM: number;

            /**
            * A constant used for the sortDirection value.
            * Use this if your game world is narrow but tall and scrolls from the bottom to the top (i.e. Commando or a vertically scrolling shoot-em-up)
            */
            static BOTTOM_TOP: number;

            /**
            * A value added to the delta values during collision checks.
            */
            static OVERLAP_BIAS: number;
            static TILE_BIAS: number;


            /**
            * The Arcade Physics world. Contains Arcade Physics related collision, overlap and motion methods.
            * 
            * @param game reference to the current game instance.
            */
            constructor(game: Phaser.Game);


            /**
            * The bounds inside of which the physics world exists. Defaults to match the world bounds.
            */
            bounds: Phaser.Rectangle;

            /**
            * Which edges of the World bounds Bodies can collide against when `collideWorldBounds` is `true`.
            * For example checkCollision.down = false means Bodies cannot collide with the World.bounds.bottom. An object containing allowed collision flags (up, down, left, right).
            */
            checkCollision: { up?: boolean; down?: boolean; left?: boolean; right?: boolean; };

            /**
            * If true World.separate will always separate on the X axis before Y. Otherwise it will check gravity totals first.
            */
            forceX: boolean;

            /**
            * Local reference to game.
            */
            game: Phaser.Game;

            /**
            * The World gravity setting. Defaults to x: 0, y: 0, or no gravity.
            */
            gravity: Phaser.Point;

            /**
            * The world QuadTree.
            */
            quadTree: Phaser.QuadTree;

            /**
            * Used by the QuadTree to set the maximum number of objects per quad.
            */
            maxObjects: number;

            /**
            * Used by the QuadTree to set the maximum number of iteration levels.
            */
            maxLevels: number;

            /**
            * If true the QuadTree will not be used for any collision. QuadTrees are great if objects are well spread out in your game, otherwise they are a performance hit. If you enable this you can disable on a per body basis via `Body.skipQuadTree`.
            */
            skipQuadTree: boolean;

            /**
            * Used when colliding a Sprite vs. a Group, or a Group vs. a Group, this defines the direction the sort is based on. Default is Phaser.Physics.Arcade.LEFT_RIGHT.
            */
            sortDirection: number;


            /**
            * Given the rotation (in radians) and speed calculate the acceleration and return it as a Point object, or set it to the given point object.
            * One way to use this is: accelerationFromRotation(rotation, 200, sprite.acceleration) which will set the values directly to the sprites acceleration and not create a new Point object.
            * 
            * @param rotation The angle in radians.
            * @param speed The speed it will move, in pixels per second sq. - Default: 60
            * @param point The Point object in which the x and y properties will be set to the calculated acceleration.
            * @return - A Point where point.x contains the acceleration x value and point.y contains the acceleration y value.
            */
            accelerationFromRotation(rotation: number, speed?: number, point?: Phaser.Point): Phaser.Point;

            /**
            * Sets the acceleration.x/y property on the display object so it will move towards the target at the given speed (in pixels per second sq.)
            * You must give a maximum speed value, beyond which the display object won't go any faster.
            * Note: The display object does not continuously track the target. If the target changes location during transit the display object will not modify its course.
            * Note: The display object doesn't stop moving once it reaches the destination coordinates.
            * 
            * @param displayObject The display object to move.
            * @param destination The display object to move towards. Can be any object but must have visible x/y properties.
            * @param speed The speed it will accelerate in pixels per second. - Default: 60
            * @param xSpeedMax The maximum x velocity the display object can reach. - Default: 500
            * @param ySpeedMax The maximum y velocity the display object can reach. - Default: 500
            * @return The angle (in radians) that the object should be visually set to in order to match its new trajectory.
            */
            accelerateToObject(displayObject: any, destination: any, speed?: number, xSpeedMax?: number, ySpeedMax?: number): number;

            /**
            * Sets the acceleration.x/y property on the display object so it will move towards the target at the given speed (in pixels per second sq.)
            * You must give a maximum speed value, beyond which the display object won't go any faster.
            * Note: The display object does not continuously track the target. If the target changes location during transit the display object will not modify its course.
            * Note: The display object doesn't stop moving once it reaches the destination coordinates.
            * 
            * @param displayObject The display object to move.
            * @param pointer The pointer to move towards. Defaults to Phaser.Input.activePointer.
            * @param speed The speed it will accelerate in pixels per second. - Default: 60
            * @param xSpeedMax The maximum x velocity the display object can reach. - Default: 500
            * @param ySpeedMax The maximum y velocity the display object can reach. - Default: 500
            * @return The angle (in radians) that the object should be visually set to in order to match its new trajectory.
            */
            accelerateToPointer(displayObject: any, pointer?: Phaser.Pointer, speed?: number, xSpeedMax?: number, ySpeedMax?: number): number;

            /**
            * Sets the acceleration.x/y property on the display object so it will move towards the x/y coordinates at the given speed (in pixels per second sq.)
            * You must give a maximum speed value, beyond which the display object won't go any faster.
            * Note: The display object does not continuously track the target. If the target changes location during transit the display object will not modify its course.
            * Note: The display object doesn't stop moving once it reaches the destination coordinates.
            * 
            * @param displayObject The display object to move.
            * @param x The x coordinate to accelerate towards.
            * @param y The y coordinate to accelerate towards.
            * @param speed The speed it will accelerate in pixels per second. - Default: 60
            * @param xSpeedMax The maximum x velocity the display object can reach. - Default: 500
            * @param ySpeedMax The maximum y velocity the display object can reach. - Default: 500
            * @return The angle (in radians) that the object should be visually set to in order to match its new trajectory.
            */
            accelerateToXY(displayObject: any, x: number, y: number, speed?: number, xSpeedMax?: number, ySpeedMax?: number): number;

            /**
            * Find the angle in radians between two display objects (like Sprites).
            * 
            * The optional `world` argument allows you to return the result based on the Game Objects `world` property,
            * instead of its `x` and `y` values. This is useful of the object has been nested inside an offset Group,
            * or parent Game Object.
            * 
            * @param source The Display Object to test from.
            * @param target The Display Object to test to.
            * @param world Calculate the angle using World coordinates (true), or Object coordinates (false, the default)
            * @return The angle in radians between the source and target display objects.
            */
            angleBetween(source: any, target: any, world?: boolean): number;

            /**
            * Find the angle in radians between a display object (like a Sprite) and a Pointer, taking their x/y and center into account.
            * 
            * The optional `world` argument allows you to return the result based on the Game Objects `world` property,
            * instead of its `x` and `y` values. This is useful of the object has been nested inside an offset Group,
            * or parent Game Object.
            * 
            * @param displayObject The Display Object to test from.
            * @param pointer The Phaser.Pointer to test to. If none is given then Input.activePointer is used.
            * @param world Calculate the angle using World coordinates (true), or Object coordinates (false, the default)
            * @return The angle in radians between displayObject.x/y to Pointer.x/y
            */
            angleToPointer(displayObject: any, pointer?: Phaser.Pointer, world?: boolean): number;

            /**
            * Find the angle in radians between a display object (like a Sprite) and the given x/y coordinate.
            * 
            * The optional `world` argument allows you to return the result based on the Game Objects `world` property,
            * instead of its `x` and `y` values. This is useful of the object has been nested inside an offset Group,
            * or parent Game Object.
            * 
            * @param displayObject The Display Object to test from.
            * @param x The x coordinate to get the angle to.
            * @param y The y coordinate to get the angle to.
            * @param world Calculate the angle using World coordinates (true), or Object coordinates (false, the default)
            * @return The angle in radians between displayObject.x/y to Pointer.x/y
            */
            angleToXY(displayObject: any, x: number, y: number, world?: boolean): number;

            /**
            * Checks for collision between two game objects and separates them if colliding. If you don't require separation then use {@link Phaser.Physics.Arcade#overlap overlap} instead.
            * 
            * You can perform Sprite vs. Sprite, Sprite vs. Group, Group vs. Group, Sprite vs. Tilemap Layer or Group vs. Tilemap Layer collisions.
            * Both the `object1` and `object2` can be arrays of objects, of differing types.
            * 
            * If two Groups or arrays are passed, each member of one will be tested against each member of the other.
            * 
            * If one Group **only** is passed (as `object1`), each member of the Group will be collided against the other members.
            * 
            * If either object is `null` the collision test will fail.
            * 
            * Bodies with `enable = false` and Sprites with `exists = false` are skipped (ignored).
            * 
            * An optional processCallback can be provided. If given this function will be called when two sprites are found to be colliding. It is called before any separation takes place, giving you the chance to perform additional checks. If the function returns true then the collision and separation is carried out. If it returns false it is skipped.
            * 
            * The collideCallback is an optional function that is only called if two sprites collide. If a processCallback has been set then it needs to return true for collideCallback to be called.
            * 
            * **This function is not recursive**, and will not test against children of objects passed (i.e. Groups or Tilemaps within other Groups).
            * 
            * ##### Examples
            * 
            *     collide(group);
            *     collide(group, undefined); // equivalent
            * 
            *     collide(sprite1, sprite2);
            * 
            *     collide(sprite, group);
            * 
            *     collide(group1, group2);
            * 
            *     collide([sprite1, sprite2], [sprite3, sprite4]); // 1v3, 1v4, 2v3, 2v4
            * 
            * ##### Tilemaps
            * 
            * Tiles marked via {@link Phaser.Tilemap#setCollision} (and similar methods) are "solid". If a Sprite collides with one of these tiles, the two are separated by moving the Sprite outside the tile's edges. Enable {@link Phaser.TilemapLayer#debug} to see the colliding edges of the Tilemap.
            * 
            * Tiles with a callback attached via {@link Phaser.Tilemap#setTileIndexCallback} or {@link Phaser.Tilemap#setTileLocationCallback} invoke the callback if a Sprite collides with them. If a tile has a callback attached via both methods, only the location callback is invoked. The colliding Sprite is separated from the tile only if the callback returns `true`.
            * 
            * @param object1 The first object or array of objects to check. Can be Phaser.Sprite, Phaser.Group, Phaser.Particles.Emitter, or Phaser.TilemapLayer.
            * @param object2 The second object or array of objects to check. Can be Phaser.Sprite, Phaser.Group, Phaser.Particles.Emitter or Phaser.TilemapLayer.
            * @param collideCallback An optional callback function that is called if the objects collide. The two objects will be passed to this function in the same order in which you specified them, unless you are colliding Group vs. Sprite, in which case Sprite will always be the first parameter.
            * @param processCallback A callback function that lets you perform additional checks against the two objects if they overlap. If this is set then collision will only happen if processCallback returns true. The two objects will be passed to this function in the same order in which you specified them, unless you are colliding Group vs. Sprite, in which case Sprite will always be the first parameter.
            * @param callbackContext The context in which to run the callbacks.
            * @return True if a collision occurred otherwise false.
            */
            collide(object1: any, object2?: any, collideCallback?: Function, processCallback?: Function, callbackContext?: any): boolean;

            /**
            * A tween-like function that takes a starting velocity and some other factors and returns an altered velocity.
            * Based on a function in Flixel by @ADAMATOMIC
            * 
            * @param axis 0 for nothing, 1 for horizontal, 2 for vertical.
            * @param body The Body object to be updated.
            * @param velocity Any component of velocity (e.g. 20).
            * @param acceleration Rate at which the velocity is changing.
            * @param drag Really kind of a deceleration, this is how much the velocity changes if Acceleration is not set.
            * @param max An absolute value cap for the velocity. - Default: 10000
            * @return The altered Velocity value.
            */
            computeVelocity(axis: number, body: Phaser.Physics.Arcade.Body, velocity: number, acceleration: number, drag: number, max?: number): number;

            /**
            * Find the distance between two display objects (like Sprites).
            * 
            * The optional `world` argument allows you to return the result based on the Game Objects `world` property,
            * instead of its `x` and `y` values. This is useful of the object has been nested inside an offset Group,
            * or parent Game Object.
            * 
            * @param source The Display Object to test from.
            * @param target The Display Object to test to.
            * @param world Calculate the distance using World coordinates (true), or Object coordinates (false, the default)
            * @return The distance between the source and target objects.
            */
            distanceBetween(source: any, target: any, world?: boolean): number;

            /**
            * Find the distance between a display object (like a Sprite) and a Pointer. If no Pointer is given the Input.activePointer is used.
            * The calculation is made from the display objects x/y coordinate. This may be the top-left if its anchor hasn't been changed.
            * If you need to calculate from the center of a display object instead use the method distanceBetweenCenters()
            * 
            * The optional `world` argument allows you to return the result based on the Game Objects `world` property,
            * instead of its `x` and `y` values. This is useful of the object has been nested inside an offset Group,
            * or parent Game Object.
            * 
            * @param displayObject The Display Object to test from.
            * @param pointer The Phaser.Pointer to test to. If none is given then Input.activePointer is used.
            * @param world Calculate the distance using World coordinates (true), or Object coordinates (false, the default)
            * @return The distance between the object and the Pointer.
            */
            distanceToPointer(displayObject: any, pointer?: Phaser.Pointer, world?: boolean): number;

            /**
            * Find the distance between a display object (like a Sprite) and the given x/y coordinates.
            * The calculation is made from the display objects x/y coordinate. This may be the top-left if its anchor hasn't been changed.
            * If you need to calculate from the center of a display object instead use the method distanceBetweenCenters()
            * 
            * The optional `world` argument allows you to return the result based on the Game Objects `world` property,
            * instead of its `x` and `y` values. This is useful of the object has been nested inside an offset Group,
            * or parent Game Object.
            * 
            * @param displayObject The Display Object to test from.
            * @param x The x coordinate to move towards.
            * @param y The y coordinate to move towards.
            * @param world Calculate the distance using World coordinates (true), or Object coordinates (false, the default)
            * @return The distance between the object and the x/y coordinates.
            */
            distanceToXY(displayObject: any, x: number, y: number, world?: boolean): number;

            /**
            * This will create an Arcade Physics body on the given game object or array of game objects.
            * A game object can only have 1 physics body active at any one time, and it can't be changed until the object is destroyed.
            * 
            * @param object The game object to create the physics body on. Can also be an array or Group of objects, a body will be created on every child that has a `body` property.
            * @param children Should a body be created on all children of this object? If true it will recurse down the display list as far as it can go. - Default: true
            */
            enable(object: any, children?: Boolean): void;

            /**
            * Creates an Arcade Physics body on the given game object.
            * 
            * A game object can only have 1 physics body active at any one time, and it can't be changed until the body is nulled.
            * 
            * When you add an Arcade Physics body to an object it will automatically add the object into its parent Groups hash array.
            * 
            * @param object The game object to create the physics body on. A body will only be created if this object has a null `body` property.
            */
            enableBody(object: any): void;

            /**
            * Given a Group and a location this will check to see which Group children overlap with the coordinates.
            * Each child will be sent to the given callback for further processing.
            * Note that the children are not checked for depth order, but simply if they overlap the coordinate or not.
            * 
            * @param x The x coordinate to check.
            * @param y The y coordinate to check.
            * @param group The Group to check.
            * @param callback A callback function that is called if the object overlaps the coordinates. The callback will be sent two parameters: the callbackArg and the Object that overlapped the location.
            * @param callbackContext The context in which to run the callback.
            * @param callbackArg An argument to pass to the callback.
            * @return An array of the Sprites from the Group that overlapped the coordinates.
            */
            getObjectsAtLocation(x: number, y: number, group: Phaser.Group, callback?: (callbackArg: any, object: any) => void, callbackContext?: any, callbackArg?: any): Sprite[];

            /**
            * Calculates the horizontal overlap between two Bodies and sets their properties accordingly, including:
            * `touching.left`, `touching.right` and `overlapX`.
            * 
            * @param body1 The first Body to separate.
            * @param body2 The second Body to separate.
            * @param overlapOnly Is this an overlap only check, or part of separation?
            * @return Returns the amount of horizontal overlap between the two bodies.
            */
            getOverlapX(body1: Phaser.Physics.Arcade.Body, body2: Phaser.Physics.Arcade.Body): number;

            /**
            * Calculates the vertical overlap between two Bodies and sets their properties accordingly, including:
            * `touching.up`, `touching.down` and `overlapY`.
            * 
            * @param body1 The first Body to separate.
            * @param body2 The second Body to separate.
            * @param overlapOnly Is this an overlap only check, or part of separation?
            * @return Returns the amount of vertical overlap between the two bodies.
            */
            getOverlapY(body1: Phaser.Physics.Arcade.Body, body2: Phaser.Physics.Arcade.Body): number;

            /**
            * Check for intersection against two bodies.
            * 
            * @param body1 The first Body object to check.
            * @param body2 The second Body object to check.
            * @return True if they intersect, otherwise false.
            */
            intersects(body1: Phaser.Physics.Arcade.Body, body2: Phaser.Physics.Arcade.Body): boolean;

            /**
            * Move the given display object towards the destination object at a steady velocity.
            * If you specify a maxTime then it will adjust the speed (overwriting what you set) so it arrives at the destination in that number of seconds.
            * Timings are approximate due to the way browser timers work. Allow for a variance of +- 50ms.
            * Note: The display object does not continuously track the target. If the target changes location during transit the display object will not modify its course.
            * Note: The display object doesn't stop moving once it reaches the destination coordinates.
            * Note: Doesn't take into account acceleration, maxVelocity or drag (if you've set drag or acceleration too high this object may not move at all)
            * 
            * @param displayObject The display object to move.
            * @param destination The display object to move towards. Can be any object but must have visible x/y properties.
            * @param speed The speed it will move, in pixels per second (default is 60 pixels/sec) - Default: 60
            * @param maxTime Time given in milliseconds (1000 = 1 sec). If set the speed is adjusted so the object will arrive at destination in the given number of ms.
            * @return The angle (in radians) that the object should be visually set to in order to match its new velocity.
            */
            moveToObject(displayObject: any, destination: any, speed?: number, maxTime?: number): number;

            /**
            * Move the given display object towards the pointer at a steady velocity. If no pointer is given it will use Phaser.Input.activePointer.
            * If you specify a maxTime then it will adjust the speed (over-writing what you set) so it arrives at the destination in that number of seconds.
            * Timings are approximate due to the way browser timers work. Allow for a variance of +- 50ms.
            * Note: The display object does not continuously track the target. If the target changes location during transit the display object will not modify its course.
            * Note: The display object doesn't stop moving once it reaches the destination coordinates.
            * 
            * @param displayObject The display object to move.
            * @param speed The speed it will move, in pixels per second (default is 60 pixels/sec) - Default: 60
            * @param pointer The pointer to move towards. Defaults to Phaser.Input.activePointer.
            * @param maxTime Time given in milliseconds (1000 = 1 sec). If set the speed is adjusted so the object will arrive at destination in the given number of ms.
            * @return The angle (in radians) that the object should be visually set to in order to match its new velocity.
            */
            moveToPointer(displayObject: any, speed?: number, pointer?: Phaser.Pointer, maxTime?: number): number;

            /**
            * Move the given display object towards the x/y coordinates at a steady velocity.
            * If you specify a maxTime then it will adjust the speed (over-writing what you set) so it arrives at the destination in that number of seconds.
            * Timings are approximate due to the way browser timers work. Allow for a variance of +- 50ms.
            * Note: The display object does not continuously track the target. If the target changes location during transit the display object will not modify its course.
            * Note: The display object doesn't stop moving once it reaches the destination coordinates.
            * Note: Doesn't take into account acceleration, maxVelocity or drag (if you've set drag or acceleration too high this object may not move at all)
            * 
            * @param displayObject The display object to move.
            * @param x The x coordinate to move towards.
            * @param y The y coordinate to move towards.
            * @param speed The speed it will move, in pixels per second (default is 60 pixels/sec) - Default: 60
            * @param maxTime Time given in milliseconds (1000 = 1 sec). If set the speed is adjusted so the object will arrive at destination in the given number of ms.
            * @return The angle (in radians) that the object should be visually set to in order to match its new velocity.
            */
            moveToXY(displayObject: any, x: number, y: number, speed?: number, maxTime?: number): number;

            /**
            * Checks for overlaps between two game objects. The objects can be Sprites, Groups or Emitters.
            * 
            * Unlike {@link Phaser.Physics.Arcade#collide collide} the objects are NOT automatically separated or have any physics applied, they merely test for overlap results.
            * 
            * You can perform Sprite vs. Sprite, Sprite vs. Group and Group vs. Group overlap checks.
            * Both the first and second parameter can be arrays of objects, of differing types.
            * If two arrays are passed, the contents of the first parameter will be tested against all contents of the 2nd parameter.
            * 
            * **This function is not recursive**, and will not test against children of objects passed (i.e. Groups within Groups).
            * 
            * ##### Tilemaps
            * 
            * Any overlapping tiles, including blank/null tiles, will give a positive result. Tiles marked via {@link Phaser.Tilemap#setCollision} (and similar methods) have no special status, and callbacks added via {@link Phaser.Tilemap#setTileIndexCallback} or {@link Phaser.Tilemap#setTileLocationCallback} are not invoked. So calling this method without any callbacks isn't very useful.
            * 
            * If you're interested only in whether an object overlaps a certain tile or class of tiles, filter the tiles with `processCallback` and then use the result returned by this method. Blank/null tiles can be excluded by their {@link Phaser.Tile#index index} (-1).
            * 
            * If you want to take action on certain overlaps, examine the tiles in `collideCallback` and then handle as you like.
            * 
            * @param object1 The first object or array of objects to check. Can be Phaser.Sprite, Phaser.Group or Phaser.Particles.Emitter.
            * @param object2 The second object or array of objects to check. Can be Phaser.Sprite, Phaser.Group or Phaser.Particles.Emitter.
            * @param overlapCallback An optional callback function that is called if the objects overlap. The two objects will be passed to this function in the same order in which you specified them, unless you are checking Group vs. Sprite, in which case Sprite will always be the first parameter.
            * @param processCallback A callback function that lets you perform additional checks against the two objects if they overlap. If this is set then `overlapCallback` will only be called if this callback returns `true`.
            * @param callbackContext The context in which to run the callbacks.
            * @return True if an overlap occurred otherwise false.
            */
            overlap(object1: any, object2: any, overlapCallback?: Function, processCallback?: Function, callbackContext?: any): boolean;
            processTileSeparationX(body: Phaser.Physics.Arcade.Body, x: number): boolean;
            processTileSeparationY(body: Phaser.Physics.Arcade.Body, y: number): void;

            /**
            * Updates the size of this physics world.
            * 
            * @param x Top left most corner of the world.
            * @param y Top left most corner of the world.
            * @param width New width of the world. Can never be smaller than the Game.width.
            * @param height New height of the world. Can never be smaller than the Game.height.
            */
            setBounds(x: number, y: number, width: number, height: number): void;

            /**
            * Updates the size of this physics world to match the size of the game world.
            */
            setBoundsToWorld(): void;

            /**
            * The core separation function to separate two physics bodies.
            * 
            * @param body1 The first Body object to separate.
            * @param body2 The second Body object to separate.
            * @param processCallback A callback function that lets you perform additional checks against the two objects if they overlap. If this function is set then the sprites will only be collided if it returns true.
            * @param callbackContext The context in which to run the process callback.
            * @param overlapOnly Just run an overlap or a full collision.
            * @return Returns true if the bodies collided, otherwise false.
            */
            separate(body1: Phaser.Physics.Arcade.Body, body2: Phaser.Physics.Arcade.Body, processCallback?: Function, callbackContext?: any, overlapOnly?: boolean): boolean;

            /**
            * The core separation function to separate two physics bodies on the x axis.
            * 
            * @param body1 The first Body to separate.
            * @param body2 The second Body to separate.
            * @param overlapOnly If true the bodies will only have their overlap data set, no separation or exchange of velocity will take place.
            * @return Returns true if the bodies were separated or overlap, otherwise false.
            */
            separateX(body1: Phaser.Physics.Arcade.Body, body2: Phaser.Physics.Arcade.Body, overlapOnly: boolean): boolean;

            /**
            * The core separation function to separate two physics bodies on the y axis.
            * 
            * @param body1 The first Body to separate.
            * @param body2 The second Body to separate.
            * @param overlapOnly If true the bodies will only have their overlap data set, no separation or exchange of velocity will take place.
            * @return Returns true if the bodies were separated or overlap, otherwise false.
            */
            separateY(body1: Phaser.Physics.Arcade.Body, body2: Phaser.Physics.Arcade.Body, overlapOnly: boolean): boolean;
            separateTile(i: number, body: Phaser.Physics.Arcade.Body, tile: Phaser.Tile): boolean;

            /**
            * This method will sort a Groups hash array.
            * 
            * If the Group has `physicsSortDirection` set it will use the sort direction defined.
            * 
            * Otherwise if the sortDirection parameter is undefined, or Group.physicsSortDirection is null, it will use Phaser.Physics.Arcade.sortDirection.
            * 
            * By changing Group.physicsSortDirection you can customise each Group to sort in a different order.
            * 
            * @param group The Group to sort.
            * @param sortDirection The sort direction used to sort this Group.
            */
            sort(group: Phaser.Group): void;
            tileCheckX(body: Phaser.Physics.Arcade.Body, tile: Phaser.Tile): number;
            tileCheckY(body: Phaser.Physics.Arcade.Body, tile: Phaser.Tile): number;

            /**
            * Called automatically by a Physics body, it updates all motion related values on the Body unless `World.isPaused` is `true`.
            * 
            * @param The Body object to be updated.
            */
            updateMotion(body: Phaser.Physics.Arcade.Body): void;

            /**
            * Given the angle (in degrees) and speed calculate the velocity and return it as a Point object, or set it to the given point object.
            * One way to use this is: velocityFromAngle(angle, 200, sprite.velocity) which will set the values directly to the sprites velocity and not create a new Point object.
            * 
            * @param angle The angle in degrees calculated in clockwise positive direction (down = 90 degrees positive, right = 0 degrees positive, up = 90 degrees negative)
            * @param speed The speed it will move, in pixels per second sq. - Default: 60
            * @param point The Point object in which the x and y properties will be set to the calculated velocity.
            * @return - A Point where point.x contains the velocity x value and point.y contains the velocity y value.
            */
            velocityFromAngle(angle: number, speed?: number, point?: Phaser.Point): Phaser.Point;

            /**
            * Given the rotation (in radians) and speed calculate the velocity and return it as a Point object, or set it to the given point object.
            * One way to use this is: velocityFromRotation(rotation, 200, sprite.velocity) which will set the values directly to the sprites velocity and not create a new Point object.
            * 
            * @param rotation The angle in radians.
            * @param speed The speed it will move, in pixels per second sq. - Default: 60
            * @param point The Point object in which the x and y properties will be set to the calculated velocity.
            * @return - A Point where point.x contains the velocity x value and point.y contains the velocity y value.
            */
            velocityFromRotation(rotation: number, speed?: number, point?: Phaser.Point): Phaser.Point;

        }

        module Arcade {


            /**
            * The Physics Body is linked to a single Sprite. All physics operations should be performed against the body rather than
            * the Sprite itself. For example you can set the velocity, acceleration, bounce values etc all on the Body.
            */
            class Body {


                /**
                * The Physics Body is linked to a single Sprite. All physics operations should be performed against the body rather than
                * the Sprite itself. For example you can set the velocity, acceleration, bounce values etc all on the Body.
                * 
                * @param sprite The Sprite object this physics body belongs to.
                */
                constructor(sprite: Phaser.Sprite);


                /**
                * The acceleration is the rate of change of the velocity. Measured in pixels per second squared.
                */
                acceleration: Phaser.Point;

                /**
                * Allow this Body to be influenced by {@link Phaser.Physics.Arcade.Body#drag drag}?
                * Default: true
                */
                allowDrag: boolean;

                /**
                * Allow this Body to be influenced by gravity? Either world or local.
                * Default: true
                */
                allowGravity: boolean;

                /**
                * Allow this Body to be rotated? (via angularVelocity, etc)
                * Default: true
                */
                allowRotation: boolean;

                /**
                * The angle of the Body's **velocity** in radians.
                */
                angle: number;

                /**
                * The angular acceleration is the rate of change of the angular velocity. Measured in degrees per second squared.
                */
                angularAcceleration: number;

                /**
                * The drag applied during the rotation of the Body. Measured in degrees per second squared.
                */
                angularDrag: number;

                /**
                * The angular velocity is the rate of change of the Body's rotation. It is measured in degrees per second.
                */
                angularVelocity: number;

                /**
                * This object is populated with boolean values when the Body collides with the World bounds or a Tile.
                * For example if blocked.up is true then the Body cannot move up. An object containing on which faces this Body is blocked from moving, if any (none, up, down, left, right).
                */
                blocked: FaceChoices;

                /**
                * The bottom value of this Body (same as Body.y + Body.height)
                */
                bottom: number;

                /**
                * The elasticity of the Body when colliding. bounce.x/y = 1 means full rebound, bounce.x/y = 0.5 means 50% rebound velocity.
                */
                bounce: Phaser.Point;

                /**
                * The center coordinate of the Physics Body.
                */
                center: Phaser.Point;

                /**
                * Set the checkCollision properties to control which directions collision is processed for this Body.
                * For example checkCollision.up = false means it won't collide when the collision happened while moving up.
                * If you need to disable a Body entirely, use `body.enable = false`, this will also disable motion.
                * If you need to disable just collision and/or overlap checks, but retain motion, set `checkCollision.none = true`. An object containing allowed collision (none, up, down, left, right).
                */
                checkCollision: FaceChoices;

                /**
                * A Body can be set to collide against the World bounds automatically and rebound back into the World if this is set to true. Otherwise it will leave the World. Should the Body collide with the World bounds?
                */
                collideWorldBounds: boolean;

                /**
                * This flag allows you to disable the custom x separation that takes place by Physics.Arcade.separate.
                * Used in combination with your own collision processHandler you can create whatever type of collision response you need. Use a custom separation system or the built-in one?
                */
                customSeparateX: boolean;

                /**
                * This flag allows you to disable the custom y separation that takes place by Physics.Arcade.separate.
                * Used in combination with your own collision processHandler you can create whatever type of collision response you need. Use a custom separation system or the built-in one?
                */
                customSeparateY: boolean;

                /**
                * The Sprite position is updated based on the delta x/y values. You can set a cap on those (both +-) using deltaMax.
                */
                deltaMax: Phaser.Point;

                /**
                * If this Body in a preUpdate (true) or postUpdate (false) state?
                */
                dirty: boolean;

                /**
                * The drag applied to the motion of the Body (when {@link Phaser.Physics.Arcade.Body#allowDrag allowDrag} is enabled). Measured in pixels per second squared.
                */
                drag: Phaser.Point;

                /**
                * If a body is overlapping with another body, but neither of them are moving (maybe they spawned on-top of each other?) this is set to true. Body embed value.
                */
                embedded: boolean;

                /**
                * A disabled body won't be checked for any form of collision or overlap or have its pre/post updates run.
                * Default: true
                */
                enable: boolean;

                /**
                * A const reference to the direction the Body is traveling or facing: Phaser.NONE, Phaser.LEFT, Phaser.RIGHT, Phaser.UP, or Phaser.DOWN. If the Body is moving on both axes, UP and DOWN take precedence.
                */
                facing: number;

                /**
                * If this Body is moving, and another Body is 'riding' this one, this is the amount of motion the riding Body receives on each axis.
                */
                friction: Phaser.Point;

                /**
                * Local reference to game.
                */
                game: Phaser.Game;

                /**
                * This Body's local gravity, **added** to any world gravity, unless Body.allowGravity is set to false.
                */
                gravity: Phaser.Point;

                /**
                * The calculated width / 2 of the physics body.
                */
                halfWidth: number;

                /**
                * The calculated height / 2 of the physics body.
                */
                halfHeight: number;

                /**
                * The calculated height of the physics body.
                */
                height: number;

                /**
                * An immovable Body will not receive any impacts from other bodies. **Two** immovable Bodies can't separate or exchange momentum and will pass through each other.
                */
                immovable: boolean;

                /**
                * If `true` this Body is using circular collision detection. If `false` it is using rectangular.
                * Use `Body.setCircle` to control the collision shape this Body uses.
                */
                isCircle: boolean;

                /**
                * Set by the `moveTo` and `moveFrom` methods.
                */
                isMoving: boolean;

                /**
                * The mass of the Body. When two bodies collide their mass is used in the calculation to determine the exchange of velocity.
                * Default: 1
                */
                mass: number;

                /**
                * The maximum angular velocity in degrees per second that the Body can reach.
                * Default: 1000
                */
                maxAngular: number;

                /**
                * The maximum velocity (in pixels per second squared) that the Body can reach.
                */
                maxVelocity: Phaser.Point;

                /**
                * Whether the physics system should update the Body's position and rotation based on its velocity, acceleration, drag, and gravity.
                * 
                * If you have a Body that is being moved around the world via a tween or a Group motion, but its local x/y position never
                * actually changes, then you should set Body.moves = false. Otherwise it will most likely fly off the screen.
                * If you want the physics system to move the body around, then set moves to true.
                * 
                * A Body with moves = false can still be moved slightly (but not accelerated) during collision separation unless you set {@link Phaser.Physics.Arcade.Body#immovable immovable} as well. Set to true to allow the Physics system to move this Body, otherwise false to move it manually.
                * Default: true
                */
                moves: boolean;

                /**
                * Optional callback. If set, invoked during the running of `moveTo` or `moveFrom` events.
                */
                movementCallback: any;

                /**
                * Context in which to call the movementCallback.
                */
                movementCallbackContext: any;

                /**
                * The distanced traveled during the last update, equal to `velocity * physicsElapsed`. Calculated during the Body.preUpdate and applied to its position.
                */
                newVelocity: Phaser.Point;

                /**
                * The offset of the Physics Body from the Sprite's texture.
                */
                offset: Phaser.Point;

                /**
                * A Signal that is dispatched when this Body collides with another Body.
                * 
                * You still need to call `game.physics.arcade.collide` in your `update` method in order
                * for this signal to be dispatched.
                * 
                * Usually you'd pass a callback to the `collide` method, but this signal provides for
                * a different level of notification.
                * 
                * Due to the potentially high volume of signals this could create it is disabled by default.
                * 
                * To use this feature set this property to a Phaser.Signal: `sprite.body.onCollide = new Phaser.Signal()`
                * and it will be called when a collision happens, passing two arguments: the sprites which collided.
                * The first sprite in the argument is always the owner of this Body.
                * 
                * If two Bodies with this Signal set collide, both will dispatch the Signal.
                */
                onCollide: Phaser.Signal;

                /**
                * Listen for the completion of `moveTo` or `moveFrom` events.
                */
                onMoveComplete: Phaser.Signal;

                /**
                * A Signal that is dispatched when this Body overlaps with another Body.
                * 
                * You still need to call `game.physics.arcade.overlap` in your `update` method in order
                * for this signal to be dispatched.
                * 
                * Usually you'd pass a callback to the `overlap` method, but this signal provides for
                * a different level of notification.
                * 
                * Due to the potentially high volume of signals this could create it is disabled by default.
                * 
                * To use this feature set this property to a Phaser.Signal: `sprite.body.onOverlap = new Phaser.Signal()`
                * and it will be called when a collision happens, passing two arguments: the sprites which collided.
                * The first sprite in the argument is always the owner of this Body.
                * 
                * If two Bodies with this Signal set collide, both will dispatch the Signal.
                */
                onOverlap: Phaser.Signal;

                /**
                * A Signal that is dispatched when this Body collides with the world bounds.
                * Due to the potentially high volume of signals this could create it is disabled by default.
                * To use this feature set this property to a Phaser.Signal: `sprite.body.onWorldBounds = new Phaser.Signal()`
                * and it will be called when a collision happens, passing five arguments:
                * `onWorldBounds(sprite, up, down, left, right)`
                * where the Sprite is a reference to the Sprite that owns this Body, and the other arguments are booleans
                * indicating on which side of the world the Body collided.
                */
                onWorldBounds: Phaser.Signal;

                /**
                * When this body collides with another, the amount of overlap is stored here. The amount of horizontal overlap during the collision.
                */
                overlapX: number;

                /**
                * When this body collides with another, the amount of overlap is stored here. The amount of vertical overlap during the collision.
                */
                overlapY: number;
                phase: number;

                /**
                * The position of the physics body, equivalent to ({@link Phaser.Physics.Arcade.Body#left left}, {@link Phaser.Physics.Arcade.Body#top top}).
                */
                position: Phaser.Point;

                /**
                * The previous rotation of the physics body, in degrees.
                */
                preRotation: number;

                /**
                * The previous position of the physics body.
                */
                prev: Phaser.Point;

                /**
                * The radius of the circular collision shape this Body is using if Body.setCircle has been enabled, relative to the Sprite's _texture_.
                * If you wish to change the radius then call {@link Phaser.Physics.Arcade.Body#setCircle setCircle} again with the new value.
                * If you wish to stop the Body using a circle then call {@link Phaser.Physics.Arcade.Body#setCircle setCircle} with a radius of zero (or undefined).
                * The actual radius of the Body (at any Sprite scale) is equal to {@link Phaser.Physics.Arcade.Body#halfWidth halfWidth} and the diameter is equal to {@link Phaser.Physics.Arcade.Body#width width}.
                */
                radius: number;

                /**
                * The right value of this Body (same as Body.x + Body.width)
                */
                right: number;

                /**
                * The Body's rotation in degrees, as calculated by its angularVelocity and angularAcceleration. Please understand that the collision Body
                * itself never rotates, it is always axis-aligned. However these values are passed up to the parent Sprite and updates its rotation.
                */
                rotation: number;

                /**
                * If true and you collide this Sprite against a Group, it will disable the collision check from using a QuadTree.
                */
                skipQuadTree: boolean;

                /**
                * The un-scaled original size.
                */
                sourceWidth: number;

                /**
                * The un-scaled original size.
                */
                sourceHeight: number;

                /**
                * The speed of the Body in pixels per second, equal to the magnitude of the velocity.
                */
                speed: number;

                /**
                * Reference to the parent Sprite.
                */
                sprite: Phaser.Sprite;

                /**
                * Set by the `moveTo` and `moveFrom` methods.
                */
                stopVelocityOnCollide: boolean;

                /**
                * If true the Body will check itself against the Sprite.getBounds() dimensions and adjust its width and height accordingly.
                * If false it will compare its dimensions against the Sprite scale instead, and adjust its width height if the scale has changed.
                * Typically you would need to enable syncBounds if your sprite is the child of a responsive display object such as a FlexLayer,
                * or in any situation where the Sprite scale doesn't change, but its parents scale is effecting the dimensions regardless.
                */
                syncBounds: boolean;

                /**
                * If this is an especially small or fast moving object then it can sometimes skip over tilemap collisions if it moves through a tile in a step.
                * Set this padding value to add extra padding to its bounds. tilePadding.x applied to its width, y to its height. Extra padding to be added to this sprite's dimensions when checking for tile collision.
                */
                tilePadding: Phaser.Point;

                /**
                * This object is populated with boolean values when the Body collides with another.
                * touching.up = true means the collision happened to the top of this Body for example. An object containing touching results (none, up, down, left, right).
                */
                touching: FaceChoices;

                /**
                * The type of physics system this body belongs to.
                */
                type: number;

                /**
                * This object is populated with previous touching values from the bodies previous collision. An object containing previous touching results (none, up, down, left, right).
                */
                wasTouching: FaceChoices;

                /**
                * The calculated width of the physics body.
                */
                width: number;

                /**
                * The elasticity of the Body when colliding with the World bounds.
                * By default this property is `null`, in which case `Body.bounce` is used instead. Set this property
                * to a Phaser.Point object in order to enable a World bounds specific bounce value.
                */
                worldBounce: Phaser.Point;

                /**
                * The velocity, or rate of change the Body's position. Measured in pixels per second.
                */
                velocity: Phaser.Point;

                /**
                * The x position.
                */
                x: number;

                /**
                * The y position.
                */
                y: number;


                /**
                * Internal method.
                * @return True if the Body collided with the world bounds, otherwise false.
                */
                checkWorldBounds(): void;

                /**
                * Returns the delta x value. The difference between Body.x now and in the previous step.
                * @return The delta value. Positive if the motion was to the right, negative if to the left.
                */
                deltaX(): number;

                /**
                * Returns the delta y value. The difference between Body.y now and in the previous step.
                * @return The delta value. Positive if the motion was downwards, negative if upwards.
                */
                deltaY(): number;

                /**
                * Returns the delta z value. The difference between Body.rotation now and in the previous step.
                * @return The delta value. Positive if the motion was clockwise, negative if anti-clockwise.
                */
                deltaZ(): number;

                /**
                * Returns the absolute delta x value.
                * @return The absolute delta value.
                */
                deltaAbsX(): number;

                /**
                * Returns the absolute delta y value.
                * @return The absolute delta value.
                */
                deltaAbsY(): number;

                /**
                * Destroys this Body.
                * 
                * First it calls Group.removeFromHash if the Game Object this Body belongs to is part of a Group.
                * Then it nulls the Game Objects body reference, and nulls this Body.sprite reference.
                */
                destroy(): void;

                /**
                * Returns the bounds of this physics body.
                * 
                * Only used internally by the World collision methods.
                * 
                * @param obj The object in which to set the bounds values.
                * @return The object that was given to this method.
                */
                getBounds(obj: any): any;

                /**
                * Tests if a world point lies within this Body.
                * 
                * @param x The world x coordinate to test.
                * @param y The world y coordinate to test.
                * @return True if the given coordinates are inside this Body, otherwise false.
                */
                hitTest(x: number, y: number): boolean;

                /**
                * Note: This method is experimental, and may be changed or removed in a future release.
                * 
                * This method moves the Body in the given direction, for the duration specified.
                * It works by setting the velocity on the Body, and an internal timer, and then
                * monitoring the duration each frame. When the duration is up the movement is
                * stopped and the `Body.onMoveComplete` signal is dispatched.
                * 
                * Movement also stops if the Body collides or overlaps with any other Body.
                * 
                * You can control if the velocity should be reset to zero on collision, by using
                * the property `Body.stopVelocityOnCollide`.
                * 
                * Stop the movement at any time by calling `Body.stopMovement`.
                * 
                * You can optionally set a speed in pixels per second. If not specified it
                * will use the current `Body.speed` value. If this is zero, the function will return false.
                * 
                * Please note that due to browser timings you should allow for a variance in
                * when the duration will actually expire. Depending on system it may be as much as
                * +- 50ms. Also this method doesn't take into consideration any other forces acting
                * on the Body, such as Gravity, drag or maxVelocity, all of which may impact the
                * movement.
                * 
                * @param duration The duration of the movement, in ms.
                * @param speed The speed of the movement, in pixels per second. If not provided `Body.speed` is used.
                * @param direction The angle of movement. If not provided `Body.angle` is used.
                * @return True if the movement successfully started, otherwise false.
                */
                moveFrom(duration: number, speed?: number, direction?: number): boolean;

                /**
                * Note: This method is experimental, and may be changed or removed in a future release.
                * 
                * This method moves the Body in the given direction, for the duration specified.
                * It works by setting the velocity on the Body, and an internal distance counter.
                * The distance is monitored each frame. When the distance equals the distance
                * specified in this call, the movement is stopped, and the `Body.onMoveComplete`
                * signal is dispatched.
                * 
                * Movement also stops if the Body collides or overlaps with any other Body.
                * 
                * You can control if the velocity should be reset to zero on collision, by using
                * the property `Body.stopVelocityOnCollide`.
                * 
                * Stop the movement at any time by calling `Body.stopMovement`.
                * 
                * Please note that due to browser timings you should allow for a variance in
                * when the distance will actually expire.
                * 
                * Note: This method doesn't take into consideration any other forces acting
                * on the Body, such as Gravity, drag or maxVelocity, all of which may impact the
                * movement.
                * 
                * @param duration The duration of the movement, in ms.
                * @param distance The distance, in pixels, the Body will move.
                * @param direction The angle of movement. If not provided `Body.angle` is used.
                * @return True if the movement successfully started, otherwise false.
                */
                moveTo(duration: number, distance: number, direction?: number): boolean;

                /**
                * Returns true if the bottom of this Body is in contact with either the world bounds or a tile.
                * @return True if in contact with either the world bounds or a tile.
                */
                onFloor(): boolean;

                /**
                * Returns true if either side of this Body is in contact with either the world bounds or a tile.
                * @return True if in contact with either the world bounds or a tile.
                */
                onWall(): boolean;

                /**
                * Internal method.
                */
                preUpdate(): void;

                /**
                * Internal method.
                */
                postUpdate(): void;

                /**
                * Render Sprite Body.
                * 
                * @param context The context to render to.
                * @param body The Body to render the info of.
                * @param color color of the debug info to be rendered. (format is css color string). - Default: 'rgba(0,255,0,0.4)'
                * @param filled Render the objected as a filled (default, true) or a stroked (false) - Default: true
                * @param lineWidth The width of the stroke when unfilled. - Default: 1
                */
                render(context: any, body: Phaser.Physics.Arcade.Body, color?: string, filled?: boolean, lineWidth?: number): void;

                /**
                * Render Sprite Body Physics Data as text.
                * 
                * @param body The Body to render the info of.
                * @param x X position of the debug info to be rendered.
                * @param y Y position of the debug info to be rendered.
                * @param color color of the debug info to be rendered. (format is css color string). - Default: 'rgb(255,255,255)'
                */
                renderBodyInfo(debug: Phaser.Utils.Debug, body: Phaser.Physics.Arcade.Body): void;

                /**
                * Resets all Body values (velocity, acceleration, rotation, etc)
                * 
                * @param x The new x position of the Body.
                * @param y The new y position of the Body.
                */
                reset(x: number, y: number): void;

                /**
                * Sets this Body as using a circle, of the given radius, for all collision detection instead of a rectangle.
                * The radius is given in pixels (relative to the Sprite's _texture_) and is the distance from the center of the circle to the edge.
                * 
                * You can also control the x and y offset, which is the position of the Body relative to the top-left of the Sprite's texture.
                * 
                * To change a Body back to being rectangular again call `Body.setSize`.
                * 
                * Note: Circular collision only happens with other Arcade Physics bodies, it does not
                * work against tile maps, where rectangular collision is the only method supported.
                * 
                * @param radius The radius of the Body in pixels. Pass a value of zero / undefined, to stop the Body using a circle for collision.
                * @param offsetX The X offset of the Body from the left of the Sprite's texture.
                * @param offsetY The Y offset of the Body from the top of the Sprite's texture.
                */
                setCircle(radius: number, offsetX?: number, offsetY?: number): void;

                /**
                * You can modify the size of the physics Body to be any dimension you need.
                * This allows you to make it smaller, or larger, than the parent Sprite.
                * You can also control the x and y offset of the Body. This is the position of the
                * Body relative to the top-left of the Sprite _texture_.
                * 
                * For example: If you have a Sprite with a texture that is 80x100 in size,
                * and you want the physics body to be 32x32 pixels in the middle of the texture, you would do:
                * 
                * `setSize(32 / Math.abs(this.scale.x), 32 / Math.abs(this.scale.y), 24, 34)`
                * 
                * Where the first two parameters are the new Body size (32x32 pixels) relative to the Sprite's scale.
                * 24 is the horizontal offset of the Body from the top-left of the Sprites texture, and 34
                * is the vertical offset.
                * 
                * If you've scaled a Sprite by altering its `width`, `height`, or `scale` and you want to
                * position the Body relative to the Sprite's dimensions (which will differ from its texture's
                * dimensions), you should divide these arguments by the Sprite's current scale:
                * 
                * `setSize(32 / sprite.scale.x, 32 / sprite.scale.y)`
                * 
                * Calling `setSize` on a Body that has already had `setCircle` will reset all of the Circle
                * properties, making this Body rectangular again.
                * 
                * @param width The width of the Body.
                * @param height The height of the Body.
                * @param offsetX The X offset of the Body from the left of the Sprite's texture.
                * @param offsetY The Y offset of the Body from the top of the Sprite's texture.
                */
                setSize(width: number, height: number, offsetX?: number, offsetY?: number): void;

                /**
                * Internal method.
                */
                updateBounds(): boolean;

            }

            class FaceChoices {

                none: boolean;
                up: boolean;
                down: boolean;
                left: boolean;
                right: boolean;

            }
        }


        /**
        * Ninja Physics. The Ninja Physics system was created in Flash by Metanet Software and ported to JavaScript by Richard Davey.
        * 
        * It allows for AABB and Circle to Tile collision. Tiles can be any of 34 different types, including slopes, convex and concave shapes.
        * 
        * It does what it does very well, but is ripe for expansion and optimisation. Here are some features that I'd love to see the community add:
        * 
        * * AABB to AABB collision
        * * AABB to Circle collision
        * * AABB and Circle 'immovable' property support
        * * n-way collision, so an AABB/Circle could pass through a tile from below and land upon it.
        * * QuadTree or spatial grid for faster Body vs. Tile Group look-ups.
        * * Optimise the internal vector math and reduce the quantity of temporary vars created.
        * * Expand Gravity and Bounce to allow for separate x/y axis values.
        * * Support Bodies linked to Sprites that don't have anchor set to 0.5
        * 
        * Feel free to attempt any of the above and submit a Pull Request with your code! Be sure to include test cases proving they work.
        */
        class Ninja {


            /**
            * Ninja Physics. The Ninja Physics system was created in Flash by Metanet Software and ported to JavaScript by Richard Davey.
            * 
            * It allows for AABB and Circle to Tile collision. Tiles can be any of 34 different types, including slopes, convex and concave shapes.
            * 
            * It does what it does very well, but is ripe for expansion and optimisation. Here are some features that I'd love to see the community add:
            * 
            * * AABB to AABB collision
            * * AABB to Circle collision
            * * AABB and Circle 'immovable' property support
            * * n-way collision, so an AABB/Circle could pass through a tile from below and land upon it.
            * * QuadTree or spatial grid for faster Body vs. Tile Group look-ups.
            * * Optimise the internal vector math and reduce the quantity of temporary vars created.
            * * Expand Gravity and Bounce to allow for separate x/y axis values.
            * * Support Bodies linked to Sprites that don't have anchor set to 0.5
            * 
            * Feel free to attempt any of the above and submit a Pull Request with your code! Be sure to include test cases proving they work.
            * 
            * @param game reference to the current game instance.
            */
            constructor(game: Phaser.Game);


            /**
            * Local reference to game.
            */
            game: Phaser.Game;

            /**
            * The World gravity setting.
            */
            gravity: number;

            /**
            * The bounds inside of which the physics world exists. Defaults to match the world bounds.
            */
            bounds: Phaser.Rectangle;

            /**
            * Used by the QuadTree to set the maximum number of objects per quad.
            */
            maxObjects: number;

            /**
            * Used by the QuadTree to set the maximum number of iteration levels.
            */
            maxLevels: number;

            /**
            * The world QuadTree.
            */
            quadTree: Phaser.QuadTree;

            /**
            * Local reference to game.time.
            */
            time: Phaser.Time;


            /**
            * Clears all physics bodies from the given TilemapLayer that were created with `World.convertTilemap`.
            * 
            * @param map The Tilemap to get the map data from.
            * @param layer The layer to operate on. If not given will default to map.currentLayer.
            */
            clearTilemapLayerBodies(map: Phaser.Tilemap, layer: any): void;

            /**
            * Checks for collision between two game objects. You can perform Sprite vs. Sprite, Sprite vs. Group, Group vs. Group, Sprite vs. Tilemap Layer or Group vs. Tilemap Layer collisions.
            * The second parameter can be an array of objects, of differing types.
            * The objects are also automatically separated. If you don't require separation then use ArcadePhysics.overlap instead.
            * An optional processCallback can be provided. If given this function will be called when two sprites are found to be colliding. It is called before any separation takes place,
            * giving you the chance to perform additional checks. If the function returns true then the collision and separation is carried out. If it returns false it is skipped.
            * The collideCallback is an optional function that is only called if two sprites collide. If a processCallback has been set then it needs to return true for collideCallback to be called.
            * 
            * @param object1 The first object to check. Can be an instance of Phaser.Sprite, Phaser.Group, Phaser.Particles.Emitter, or Phaser.TilemapLayer.
            * @param object2 The second object or array of objects to check. Can be Phaser.Sprite, Phaser.Group, Phaser.Particles.Emitter or Phaser.TilemapLayer.
            * @param collideCallback An optional callback function that is called if the objects collide. The two objects will be passed to this function in the same order in which you specified them.
            * @param processCallback A callback function that lets you perform additional checks against the two objects if they overlap. If this is set then collision will only happen if processCallback returns true. The two objects will be passed to this function in the same order in which you specified them.
            * @param callbackContext The context in which to run the callbacks.
            * @return True if a collision occured otherwise false.
            */
            collide(object1: any, object2: any, collideCallback?: Function, processCallback?: Function, callbackContext?: any): boolean;

            /**
            * Goes through all tiles in the given Tilemap and TilemapLayer and converts those set to collide into physics tiles.
            * Only call this *after* you have specified all of the tiles you wish to collide with calls like Tilemap.setCollisionBetween, etc.
            * Every time you call this method it will destroy any previously created bodies and remove them from the world.
            * Therefore understand it's a very expensive operation and not to be done in a core game update loop.
            * 
            * In Ninja the Tiles have an ID from 0 to 33, where 0 is 'empty', 1 is a full tile, 2 is a 45-degree slope, etc. You can find the ID
            * list either at the very bottom of `Tile.js`, or in a handy visual reference in the `resources/Ninja Physics Debug Tiles` folder in the repository.
            * The slopeMap parameter is an array that controls how the indexes of the tiles in your tilemap data will map to the Ninja Tile IDs.
            * For example if you had 6 tiles in your tileset: Imagine the first 4 should be converted into fully solid Tiles and the other 2 are 45-degree slopes.
            * Your slopeMap array would look like this: `[ 1, 1, 1, 1, 2, 3 ]`.
            * Where each element of the array is a tile in your tilemap and the resulting Ninja Tile it should create.
            * 
            * @param map The Tilemap to get the map data from.
            * @param layer The layer to operate on. If not given will default to map.currentLayer.
            * @param slopeMap The tilemap index to Tile ID map.
            * @return An array of the Phaser.Physics.Ninja.Tile objects that were created.
            */
            convertTilemap(map: Phaser.Tilemap, layer: any, slopeMap: any): Phaser.Physics.Ninja.Tile[];

            /**
            * This will create a Ninja Physics AABB body on the given game object. Its dimensions will match the width and height of the object at the point it is created.
            * A game object can only have 1 physics body active at any one time, and it can't be changed until the object is destroyed.
            * 
            * @param object The game object to create the physics body on. Can also be an array or Group of objects, a body will be created on every child that has a `body` property.
            * @param children Should a body be created on all children of this object? If true it will recurse down the display list as far as it can go. - Default: true
            */
            enableAABB(object: any, children?: boolean): void;

            /**
            * This will create a Ninja Physics Circle body on the given game object.
            * A game object can only have 1 physics body active at any one time, and it can't be changed until the object is destroyed.
            * 
            * @param object The game object to create the physics body on. Can also be an array or Group of objects, a body will be created on every child that has a `body` property.
            * @param radius The radius of the Circle.
            * @param children Should a body be created on all children of this object? If true it will recurse down the display list as far as it can go. - Default: true
            */
            enableCircle(object: any, radius: number, children?: boolean): void;

            /**
            * This will create a Ninja Physics Tile body on the given game object. There are 34 different types of tile you can create, including 45 degree slopes,
            * convex and concave circles and more. The id parameter controls which Tile type is created, but you can also change it at run-time.
            * Note that for all degree based tile types they need to have an equal width and height. If the given object doesn't have equal width and height it will use the width.
            * A game object can only have 1 physics body active at any one time, and it can't be changed until the object is destroyed.
            * 
            * @param object The game object to create the physics body on. Can also be an array or Group of objects, a body will be created on every child that has a `body` property.
            * @param id The type of Tile this will use, i.e. Phaser.Physics.Ninja.Tile.SLOPE_45DEGpn, Phaser.Physics.Ninja.Tile.CONVEXpp, etc. - Default: 1
            * @param children Should a body be created on all children of this object? If true it will recurse down the display list as far as it can go. - Default: true
            */
            enableTile(object: any, id: number, children?: boolean): void;

            /**
            * This will create a Ninja Physics body on the given game object or array of game objects.
            * A game object can only have 1 physics body active at any one time, and it can't be changed until the object is destroyed.
            * 
            * @param object The game object to create the physics body on. Can also be an array or Group of objects, a body will be created on every child that has a `body` property.
            * @param type The type of Ninja shape to create. 1 = AABB, 2 = Circle or 3 = Tile. - Default: 1
            * @param id If this body is using a Tile shape, you can set the Tile id here, i.e. Phaser.Physics.Ninja.Tile.SLOPE_45DEGpn, Phaser.Physics.Ninja.Tile.CONVEXpp, etc. - Default: 1
            * @param radius If this body is using a Circle shape this controls the radius.
            * @param children Should a body be created on all children of this object? If true it will recurse down the display list as far as it can go. - Default: true
            */
            enable(object: any, type?: number, id?: number, radius?: number, children?: boolean): void;

            /**
            * Creates a Ninja Physics body on the given game object.
            * A game object can only have 1 physics body active at any one time, and it can't be changed until the body is nulled.
            * 
            * @param object The game object to create the physics body on. A body will only be created if this object has a null `body` property.
            */
            enableBody(object: any, type?: number, id?: number, radius?: number): void;

            /**
            * Checks for overlaps between two game objects. The objects can be Sprites, Groups or Emitters.
            * You can perform Sprite vs. Sprite, Sprite vs. Group and Group vs. Group overlap checks.
            * Unlike collide the objects are NOT automatically separated or have any physics applied, they merely test for overlap results.
            * The second parameter can be an array of objects, of differing types.
            * 
            * @param object1 The first object to check. Can be an instance of Phaser.Sprite, Phaser.Group or Phaser.Particles.Emitter.
            * @param object2 The second object or array of objects to check. Can be Phaser.Sprite, Phaser.Group or Phaser.Particles.Emitter.
            * @param overlapCallback An optional callback function that is called if the objects overlap. The two objects will be passed to this function in the same order in which you specified them.
            * @param processCallback A callback function that lets you perform additional checks against the two objects if they overlap. If this is set then overlapCallback will only be called if processCallback returns true.
            * @param callbackContext The context in which to run the callbacks.
            * @return True if an overlap occured otherwise false.
            */
            overlap(object1: any, object2: any, overlapCallback?: Function, processCallback?: Function, callbackContext?: any): boolean;

            /**
            * The core separation function to separate two physics bodies.
            * 
            * @param body1 The Body object to separate.
            * @param body2 The Body object to separate.
            * @return Returns true if the bodies collided, otherwise false.
            */
            separate(body1: Phaser.Physics.Ninja.Body, body2: Phaser.Physics.Ninja.Body, processCallback?: Function, callbackContext?: any, overlapOnly?: boolean): boolean;

            /**
            * Updates the size of this physics world.
            * 
            * @param x Top left most corner of the world.
            * @param y Top left most corner of the world.
            * @param width New width of the world. Can never be smaller than the Game.width.
            * @param height New height of the world. Can never be smaller than the Game.height.
            */
            setBounds(x: number, y: number, width: number, height: number): void;

            /**
            * Updates the size of this physics world to match the size of the game world.
            */
            setBoundsToWorld(): void;
        }

        module Ninja {


            /**
            * The Physics Body is linked to a single Sprite. All physics operations should be performed against the body rather than
            * the Sprite itself. For example you can set the velocity, bounce values etc all on the Body.
            */
            class Body {


                /**
                * The Physics Body is linked to a single Sprite. All physics operations should be performed against the body rather than
                * the Sprite itself. For example you can set the velocity, bounce values etc all on the Body.
                * 
                * @param system The physics system this Body belongs to.
                * @param sprite The Sprite object this physics body belongs to.
                * @param type The type of Ninja shape to create. 1 = AABB, 2 = Circle or 3 = Tile. - Default: 1
                * @param id If this body is using a Tile shape, you can set the Tile id here, i.e. Phaser.Physics.Ninja.Tile.SLOPE_45DEGpn, Phaser.Physics.Ninja.Tile.CONVEXpp, etc. - Default: 1
                * @param radius If this body is using a Circle shape this controls the radius. - Default: 16
                * @param x The x coordinate of this Body. This is only used if a sprite is not provided.
                * @param y The y coordinate of this Body. This is only used if a sprite is not provided.
                * @param width The width of this Body. This is only used if a sprite is not provided.
                * @param height The height of this Body. This is only used if a sprite is not provided.
                */
                constructor(system: Phaser.Physics.Ninja, sprite: Phaser.Sprite, type?: number, id?: number, radius?: number, x?: number, y?: number, width?: number, height?: number);


                /**
                * The AABB object this body is using for collision.
                */
                aabb: Phaser.Physics.Ninja.AABB;

                /**
                * The angle of this Body
                */
                angle: number;

                /**
                * The bottom value of this Body (same as Body.y + Body.height)
                */
                bottom: number;

                /**
                * The bounciness of this object when it collides. A value between 0 and 1. We recommend setting it to 0.999 to avoid jittering.
                * Default: 0.3
                */
                bounce: number;

                /**
                * Set the checkCollision properties to control which directions collision is processed for this Body.
                * For example checkCollision.up = false means it won't collide when the collision happened while moving up. An object containing allowed collision.
                */
                checkCollision: Phaser.Physics.Arcade.FaceChoices;

                /**
                * The Circle object this body is using for collision.
                */
                circle: Phaser.Physics.Ninja.Circle;

                /**
                * A Body can be set to collide against the World bounds automatically and rebound back into the World if this is set to true. Otherwise it will leave the World. Should the Body collide with the World bounds?
                */
                collideWorldBounds: boolean;

                /**
                * The drag applied to this object as it moves.
                * Default: 1
                */
                drag: number;

                /**
                * A const reference to the direction the Body is traveling or facing.
                */
                facing: number;

                /**
                * The friction applied to this object as it moves.
                * Default: 0.05
                */
                friction: number;

                /**
                * Local reference to game.
                */
                game: Phaser.Game;

                /**
                * How much of the world gravity should be applied to this object? 1 = all of it, 0.5 = 50%, etc.
                * Default: 1
                */
                gravityScale: number;

                /**
                * The height of this Body
                */
                height: number;

                /**
                * An immovable Body will not receive any impacts from other bodies. Not fully implemented.
                */
                immovable: boolean;

                /**
                * The maximum speed this body can travel at (taking drag and friction into account)
                * Default: 8
                */
                maxSpeed: number;

                /**
                * The right value of this Body (same as Body.x + Body.width)
                */
                right: number;

                /**
                * Reference to the parent Sprite.
                */
                sprite: Phaser.Sprite;

                /**
                * The parent physics system.
                */
                system: Phaser.Physics.Ninja;

                /**
                * The Tile object this body is using for collision.
                */
                tile: Phaser.Physics.Ninja.Tile;

                /**
                * This object is populated with boolean values when the Body collides with another.
                * touching.up = true means the collision happened to the top of this Body for example. An object containing touching results.
                */
                touching: Phaser.Physics.Arcade.FaceChoices;

                /**
                * The type of physics system this body belongs to.
                */
                type: number;

                /**
                * A local reference to the body shape.
                */
                shape: any;

                /**
                * The speed of this Body
                */
                speed: number;

                /**
                * The velocity in pixels per second sq. of the Body.
                */
                velocity: Phaser.Point;

                /**
                * This object is populated with previous touching values from the bodies previous collision. An object containing previous touching results.
                */
                wasTouching: Phaser.Physics.Arcade.FaceChoices;

                /**
                * The width of this Body
                */
                width: number;

                /**
                * The x position.
                */
                x: number;

                /**
                * The y position.
                */
                y: number;


                /**
                * Returns the absolute delta x value.
                * @return The absolute delta value.
                */
                deltaAbsX(): number;

                /**
                * Returns the absolute delta y value.
                * @return The absolute delta value.
                */
                deltaAbsY(): number;

                /**
                * Returns the delta x value. The difference between Body.x now and in the previous step.
                * @return The delta value. Positive if the motion was to the right, negative if to the left.
                */
                deltaX(): number;

                /**
                * Returns the delta y value. The difference between Body.y now and in the previous step.
                * @return The delta value. Positive if the motion was downwards, negative if upwards.
                */
                deltaY(): number;

                /**
                * Destroys this body's reference to the sprite and system, and destroys its shape.
                */
                destroy(): void;

                /**
                * Stops all movement of this body.
                */
                setZeroVelocity(): void;
                moveTo(speed: number, angle: number): void;
                moveFrom(speed: number, angle: number): void;
                moveLeft(speed: number): void;
                moveRight(speed: number): void;
                moveUp(speed: number): void;
                moveDown(speed: number): void;
                poseUpdate(): void;

                /**
                * Internal method.
                */
                preUpdate(): void;

                /**
                * Render Sprite's Body.
                * 
                * @param context The context to render to.
                * @param body The Body to render.
                * @param color color of the debug shape to be rendered. (format is css color string). - Default: 'rgba(0,255,0,0.4)'
                * @param filled Render the shape as a filled (default, true) or a stroked (false) - Default: true
                */
                render(context: any, body: Phaser.Physics.Ninja.Body, color?: string, filled?: boolean): void;

                /**
                * Resets all Body values and repositions on the Sprite.
                */
                reset(): void;

            }


            /**
            * Ninja Physics AABB constructor.
            * Note: This class could be massively optimised and reduced in size. I leave that challenge up to you.
            */
            class AABB {


                /**
                * Ninja Physics AABB constructor.
                * Note: This class could be massively optimised and reduced in size. I leave that challenge up to you.
                * 
                * @param body The body that owns this shape.
                * @param x The x coordinate to create this shape at.
                * @param y The y coordinate to create this shape at.
                * @param width The width of this AABB.
                * @param height The height of this AABB.
                */
                constructor(body: Phaser.Physics.Ninja.Body, x: number, y: number, width: number, height: number);

                static COL_NONE: number;
                static COL_AXIS: number;
                static COL_OTHER: number;


                /**
                * All of the collision response handlers.
                */
                aabbTileProjections: any;

                /**
                * A reference to the body that owns this shape.
                */
                body: Phaser.Physics.Ninja.Body;

                /**
                * The height.
                */
                height: number;
                oldPos: Phaser.Point;

                /**
                * The position of this object.
                */
                pos: Phaser.Point;

                /**
                * A reference to the physics system.
                */
                system: Phaser.Physics.Ninja;

                /**
                * The width.
                */
                width: number;

                /**
                * The velocity of this object.
                */
                velocity: Phaser.Point;

                /**
                * Half the width.
                */
                xw: number;

                /**
                * Half the height.
                */
                yw: number;


                /**
                * Collides this AABB against the world bounds.
                */
                collideWorldBounds(): void;

                /**
                * Collides this AABB against a AABB.
                * 
                * @param aabb The AABB to collide against.
                */
                collideAABBVsAABB(aabb: Phaser.Physics.Ninja.AABB): boolean;

                /**
                * Collides this AABB against a Tile.
                * 
                * @param tile The Tile to collide against.
                */
                collideAABBVsTile(tile: Phaser.Physics.Ninja.Tile): boolean;

                /**
                * Destroys this AABB's reference to Body and System
                */
                destroy(): void;

                /**
                * Updates this AABBs position.
                */
                integrate(): void;

                /**
                * Render this AABB for debugging purposes.
                * 
                * @param context The context to render to.
                * @param xOffset X offset from AABB's position to render at.
                * @param yOffset Y offset from AABB's position to render at.
                * @param color color of the debug shape to be rendered. (format is css color string).
                * @param filled Render the shape as solid (true) or hollow (false).
                */
                render(context: any, xOffset: number, yOffset: number, color: string, filled: boolean): void;

                /**
                * Process a collision partner-agnostic collision response and apply the resulting forces.
                * 
                * @param px The tangent velocity
                * @param py The tangent velocity
                * @param dx Collision normal
                * @param dy Collision normal
                */
                reportCollision(px: number, py: number, dx: number, dy: number): void;

                /**
                * Process a world collision and apply the resulting forces.
                * 
                * @param px The tangent velocity
                * @param py The tangent velocity
                * @param dx Collision normal
                * @param dy Collision normal
                */
                reportCollisionVsWorld(px: number, py: number, dx: number, dy: number, obj: any): void;

                /**
                * Process a body collision and apply the resulting forces. Still very much WIP and doesn't work fully. Feel free to fix!
                * 
                * @param px The tangent velocity
                * @param py The tangent velocity
                * @param dx Collision normal
                * @param dy Collision normal
                * @param obj Object this AABB collided with
                */
                reportCollisionVsBody(px: number, py: number, dx: number, dy: number, obj: any): void;

                /**
                * Resolves tile collision.
                * 
                * @param x Penetration depth on the x axis.
                * @param y Penetration depth on the y axis.
                * @param body The AABB involved in the collision.
                * @param tile The Tile involved in the collision.
                * @return True if the collision was processed, otherwise false.
                */
                resolveTile(x: number, y: number, body: Phaser.Physics.Ninja.AABB, tile: Phaser.Physics.Ninja.Tile): boolean;
                reverse(): void;

            }


            /**
            * Ninja Physics Circle constructor.
            * Note: This class could be massively optimised and reduced in size. I leave that challenge up to you.
            */
            class Circle {


                /**
                * Ninja Physics Circle constructor.
                * Note: This class could be massively optimised and reduced in size. I leave that challenge up to you.
                * 
                * @param body The body that owns this shape.
                * @param x The x coordinate to create this shape at.
                * @param y The y coordinate to create this shape at.
                * @param radius The radius of this Circle.
                */
                constructor(body: Phaser.Physics.Ninja.Body, x: number, y: number, radius: number);

                COL_NONE: number;
                COL_AXIS: number;
                COL_OTHER: number;


                /**
                * A reference to the body that owns this shape.
                */
                body: Phaser.Physics.Ninja.Body;

                /**
                * All of the collision response handlers.
                */
                circleTileProjections: { [index: number]: ((x: number, y: number, oH: number, oV: number, obj: Phaser.Physics.Ninja.Circle, t: Phaser.Physics.Ninja.Tile) => number); };
                oldPos: Phaser.Point;

                /**
                * The height.
                */
                height: number;

                /**
                * The position of this object.
                */
                pos: Phaser.Point;

                /**
                * The radius of this circle shape.
                */
                radius: number;

                /**
                * A reference to the physics system.
                */
                system: Phaser.Physics.Ninja;
                type: number;

                /**
                * The velocity of this object.
                */
                velocity: Phaser.Point;

                /**
                * The width.
                */
                width: number;

                /**
                * Half the width.
                */
                xw: number;

                /**
                * Half the height.
                */
                yw: number;


                /**
                * Collides this Circle with a Tile.
                * 
                * @param t The Tile involved in the collision.
                * @return True if they collide, otherwise false.
                */
                collideCircleVsTile(tile: Phaser.Physics.Ninja.Tile): boolean;

                /**
                * Collides this Circle against the world bounds.
                */
                collideWorldBounds(): void;

                /**
                * Destroys this Circle's reference to Body and System
                */
                destroy(): void;
                distance(dest: number, round?: boolean): number;

                /**
                * Updates this Circles position.
                */
                integrate(): void;

                /**
                * Render this circle for debugging purposes.
                * 
                * @param context The context to render to.
                * @param xOffset X offset from circle's position to render at.
                * @param yOffset Y offset from circle's position to render at.
                * @param color color of the debug shape to be rendered. (format is css color string).
                * @param filled Render the shape as solid (true) or hollow (false).
                */
                render(context: any, xOffset: number, yOffset: number, color: string, filled: boolean): void;

                /**
                * Process a world collision and apply the resulting forces.
                * 
                * @param px The tangent velocity
                * @param py The tangent velocity
                * @param dx Collision normal
                * @param dy Collision normal
                * @param obj Object this Circle collided with
                */
                reportCollisionVsWorld(px: number, py: number, dx: number, dy: number, obj: any): void;
                reportCollisionVsBody(px: number, py: number, dx: number, dy: number, obj: any): void;

                /**
                * Resolves tile collision.
                * 
                * @param x Penetration depth on the x axis.
                * @param y Penetration depth on the y axis.
                * @param oH Grid / voronoi region.
                * @param oV Grid / voronoi region.
                * @param obj The Circle involved in the collision.
                * @param t The Tile involved in the collision.
                * @return The result of the collision.
                */
                resolveCircleTile(x: number, y: number, oH: number, oV: number, obj: Phaser.Physics.Ninja.Circle, t: Phaser.Physics.Ninja.Tile): boolean;

            }

            enum TileType {
                TYPE_EMPTY,
                TYPE_FULL,
                TYPE_45DEG,
                TYPE_CONCAVE,
                TYPE_CONVEX,
                TYPE_22DEGs,
                TYPE_22DEGb,
                TYPE_67DEGs,
                TYPE_67DEGb,
                TYPE_HALF
            }


            /**
            * Ninja Physics Tile constructor.
            * A Tile is defined by its width, height and type. It's type can include slope data, such as 45 degree slopes, or convex slopes.
            * Understand that for any type including a slope (types 2 to 29) the Tile must be SQUARE, i.e. have an equal width and height.
            * Also note that as Tiles are primarily used for levels they have gravity disabled and world bounds collision disabled by default.
            * 
            * Note: This class could be massively optimised and reduced in size. I leave that challenge up to you.
            */
            class Tile {


                /**
                * Ninja Physics Tile constructor.
                * A Tile is defined by its width, height and type. It's type can include slope data, such as 45 degree slopes, or convex slopes.
                * Understand that for any type including a slope (types 2 to 29) the Tile must be SQUARE, i.e. have an equal width and height.
                * Also note that as Tiles are primarily used for levels they have gravity disabled and world bounds collision disabled by default.
                * 
                * Note: This class could be massively optimised and reduced in size. I leave that challenge up to you.
                * 
                * @param body The body that owns this shape.
                * @param x The x coordinate to create this shape at.
                * @param y The y coordinate to create this shape at.
                * @param width The width of this AABB.
                * @param height The height of this AABB.
                * @param type The type of Ninja shape to create. 1 = AABB, 2 = Circle or 3 = Tile. - Default: 1
                */
                constructor(body: Phaser.Physics.Ninja.Body, x: number, y: number, width: number, height: number, type?: number);


                /**
                * A reference to the body that owns this shape.
                */
                body: Phaser.Physics.Ninja.Body;

                /**
                * The bottom value of this Body (same as Body.y + Body.height)
                */
                bottom: number;
                flipped: boolean;

                /**
                * The height.
                */
                height: number;

                /**
                * The ID of this Tile.
                */
                id: number;

                /**
                * The position of this object in the previous update.
                */
                oldpos: Phaser.Point;

                /**
                * The position of this object.
                */
                pos: Phaser.Point;

                /**
                * The right value of this Body (same as Body.x + Body.width)
                */
                right: number;
                rotation: number;

                /**
                * A reference to the physics system.
                */
                system: Phaser.Physics.Ninja;

                /**
                * The type of this Tile.
                */
                type: Phaser.Physics.Ninja.TileType;

                /**
                * The velocity of this object.
                */
                velocity: Phaser.Point;

                /**
                * The width.
                */
                width: number;

                /**
                * Half the width.
                */
                xw: number;

                /**
                * Half the height.
                */
                yw: number;

                /**
                * The x position.
                */
                x: number;

                /**
                * The y position.
                */
                y: number;


                /**
                * Sets this tile to be empty.
                */
                clear(): void;

                /**
                * Tiles cannot collide with the world bounds, it's up to you to keep them where you want them. But we need this API stub to satisfy the Body.
                */
                collideWorldBounds(): void;

                /**
                * Destroys this Tiles reference to Body and System.
                */
                destroy(): void;

                /**
                * Updates this objects position.
                */
                integrate(): void;

                /**
                * Process a world collision and apply the resulting forces.
                * 
                * @param px The tangent velocity
                * @param py The tangent velocity
                * @param dx Collision normal
                * @param dy Collision normal
                * @param obj Object this Tile collided with
                */
                reportCollisionVsWorld(px: number, py: number, dx: number, dy: number, obj: any): void;

                /**
                * Tiles cannot collide with the world bounds, it's up to you to keep them where you want them. But we need this API stub to satisfy the Body.
                * 
                * @param id The type of Tile this will use, i.e. Phaser.Physics.Ninja.Tile.SLOPE_45DEGpn, Phaser.Physics.Ninja.Tile.CONVEXpp, etc.
                */
                setType(id: number): number;

            }

        }


        /**
        * This is your main access to the P2 Physics World.
        * From here you can create materials, listen for events and add bodies into the physics simulation.
        */
        class P2 {


            /**
            * This is your main access to the P2 Physics World.
            * From here you can create materials, listen for events and add bodies into the physics simulation.
            * 
            * @param game Reference to the current game instance.
            * @param config Physics configuration object passed in from the game constructor.
            */
            constructor(game: Phaser.Game, config?: any);


            /**
            * Enable to automatically apply body damping each step.
            */
            applyDamping: boolean;

            /**
            * Enable to automatically apply gravity each step.
            */
            applyGravity: boolean;

            /**
            * Enable to automatically apply spring forces each step.
            */
            applySpringForces: boolean;

            /**
            * An array of the bodies the world bounds collides with.
            */
            boundsCollidesWith: Phaser.Physics.P2.Body[];

            /**
            * A default collision group.
            */
            boundsCollisionGroup: Phaser.Physics.P2.CollisionGroup;

            /**
            * The p2 World configuration object.
            */
            config: any;

            /**
            * The context under which the callbacks are fired.
            */
            callbackContext: any;

            /**
            * An array containing the collision groups that have been defined in the World.
            */
            collisionGroups: Phaser.Physics.P2.CollisionGroup[];

            /**
            * The default Contact Material being used by the World.
            */
            contactMaterial: Phaser.Physics.P2.ContactMaterial;

            /**
            * Set to true if you want to the world to emit the "impact" event. Turning this off could improve performance.
            */
            emitImpactEvent: boolean;

            /**
            * A default collision group.
            */
            everythingCollisionGroup: Phaser.Physics.P2.CollisionGroup;

            /**
            * The frame rate the world will be stepped at. Defaults to 1 / 60, but you can change here. Also see useElapsedTime property.
            */
            frameRate: number;

            /**
            * Friction between colliding bodies. This value is used if no matching ContactMaterial is found for a Material pair.
            */
            friction: number;

            /**
            * Local reference to game.
            */
            game: Phaser.Game;

            /**
            * The gravity applied to all bodies each step.
            */
            gravity: Phaser.Physics.P2.InversePointProxy;

            /**
            * A local array of all created Materials.
            */
            materials: Phaser.Physics.P2.Material[];

            /**
            * A default collision group.
            */
            nothingCollisionGroup: Phaser.Physics.P2.CollisionGroup;

            /**
            * This signal is dispatched when a new Body is added to the World.
            * 
            * It sends 1 argument: `body` which is the `Phaser.Physics.P2.Body` that was added to the world.
            */
            onBodyAdded: Phaser.Signal;

            /**
            * This signal is dispatched when a Body is removed to the World.
            * 
            * It sends 1 argument: `body` which is the `Phaser.Physics.P2.Body` that was removed from the world.
            */
            onBodyRemoved: Phaser.Signal;

            /**
            * This Signal is dispatched when a first contact is created between two bodies. This happens *before* the step has been done.
            * 
            * It sends 5 arguments: `bodyA`, `bodyB`, `shapeA`, `shapeB` and `contactEquations`.
            * 
            * It is possible that in certain situations the `bodyA` or `bodyB` values are `null`. You should check for this
            * in your own code to avoid processing potentially null physics bodies.
            */
            onBeginContact: Phaser.Signal;

            /**
            * This signal is dispatched when a Constraint is added to the World.
            * 
            * It sends 1 argument: `constraint` which is the `Phaser.Physics.P2.Constraint` that was added to the world.
            */
            onConstraintAdded: Phaser.Signal;

            /**
            * This signal is dispatched when a Constraint is removed from the World.
            * 
            * It sends 1 argument: `constraint` which is the `Phaser.Physics.P2.Constraint` that was removed from the world.
            */
            onConstraintRemoved: Phaser.Signal;

            /**
            * This signal is dispatched when a Contact Material is added to the World.
            * 
            * It sends 1 argument: `material` which is the `Phaser.Physics.P2.ContactMaterial` that was added to the world.
            */
            onContactMaterialAdded: Phaser.Signal;

            /**
            * This signal is dispatched when a Contact Material is removed from the World.
            * 
            * It sends 1 argument: `material` which is the `Phaser.Physics.P2.ContactMaterial` that was removed from the world.
            */
            onContactMaterialRemoved: Phaser.Signal;

            /**
            * This Signal is dispatched when final contact occurs between two bodies. This happens *before* the step has been done.
            * 
            * It sends 4 arguments: `bodyA`, `bodyB`, `shapeA` and `shapeB`.
            * 
            * It is possible that in certain situations the `bodyA` or `bodyB` values are `null`. You should check for this
            * in your own code to avoid processing potentially null physics bodies.
            */
            onEndContact: Phaser.Signal;

            /**
            * This signal is dispatched when a Spring is added to the World.
            * 
            * It sends 1 argument: `spring` which is either a `Phaser.Physics.P2.Spring`, `p2.LinearSpring` or `p2.RotationalSpring` that was added to the world.
            */
            onSpringAdded: Phaser.Signal;

            /**
            * This signal is dispatched when a Spring is removed from the World.
            * 
            * It sends 1 argument: `spring` which is either a `Phaser.Physics.P2.Spring`, `p2.LinearSpring` or `p2.RotationalSpring` that was removed from the world.
            */
            onSpringRemoved: Phaser.Signal;

            /**
            * The paused state of the P2 World.
            */
            paused: boolean;
            postBroaddphaseCallback: Function;

            /**
            * Default coefficient of restitution between colliding bodies. This value is used if no matching ContactMaterial is found for a Material pair.
            */
            restitution: number;

            /**
            * Enable/disable constraint solving in each step.
            */
            solveConstraints: boolean;

            /**
            * The World time.
            */
            time: any;

            /**
            * The total number of bodies in the world.
            */
            total: number;

            /**
            * If true the frameRate value will be ignored and instead p2 will step with the value of Game.Time.physicsElapsed, which is a delta time value.
            */
            useElapsedTime: boolean;

            /**
            * An object containing the 4 wall bodies that bound the physics world.
            */
            walls: {
                left?: Phaser.Physics.P2.Body;
                right?: Phaser.Physics.P2.Body;
                top?: Phaser.Physics.P2.Body;
                bottom?: Phaser.Physics.P2.Body;
            };

            /**
            * The p2 World in which the simulation is run.
            */
            world: p2.World;


            /**
            * Add a body to the world.
            * 
            * @param body The Body to add to the World.
            * @return True if the Body was added successfully, otherwise false.
            */
            addBody(body: Phaser.Physics.P2.Body): boolean;

            /**
            * Adds a Contact Material to the world.
            * 
            * @param material The Contact Material to be added to the World.
            * @return The Contact Material that was added.
            */
            addContactMaterial(material: Phaser.Physics.P2.ContactMaterial): Phaser.Physics.P2.ContactMaterial;

            /**
            * Adds a Constraint to the world.
            * 
            * @param constraint The Constraint to add to the World.
            * @return The Constraint that was added.
            */
            addConstraint<T>(constraint: T): T;

            /**
            * Adds a Spring to the world.
            * 
            * @param spring The Spring to add to the World.
            * @return The Spring that was added.
            */
            addSpring(spring: Phaser.Physics.P2.Spring): Phaser.Physics.P2.Spring;

            /**
            * Handles a p2 begin contact event.
            * 
            * @param event The event data.
            */
            beginContactHandler(event: any): void;

            /**
            * Clears all bodies from the simulation, resets callbacks and resets the collision bitmask.
            * 
            * The P2 world is also cleared:
            * 
            * * Removes all solver equations
            * * Removes all constraints
            * * Removes all bodies
            * * Removes all springs
            * * Removes all contact materials
            * 
            * This is called automatically when you switch state.
            */
            clear(): void;

            /**
            * Clears all physics bodies from the given TilemapLayer that were created with `World.convertTilemap`.
            * 
            * @param map The Tilemap to get the map data from.
            * @param layer The layer to operate on. If not given will default to map.currentLayer.
            */
            clearTilemapLayerBodies(map: Phaser.Tilemap, layer?: any): void;

            /**
            * Converts all of the polylines objects inside a Tiled ObjectGroup into physics bodies that are added to the world.
            * Note that the polylines must be created in such a way that they can withstand polygon decomposition.
            * 
            * @param map The Tilemap to get the map data from.
            * @param layer The layer to operate on. If not given will default to map.currentLayer.
            * @param addToWorld If true it will automatically add each body to the world. - Default: true
            * @return An array of the Phaser.Physics.Body objects that have been created.
            */
            convertCollisionObjects(map: Phaser.Tilemap, layer?: any, addToWorld?: boolean): Phaser.Physics.P2.Body[];

            /**
            * Goes through all tiles in the given Tilemap and TilemapLayer and converts those set to collide into physics bodies.
            * Only call this *after* you have specified all of the tiles you wish to collide with calls like Tilemap.setCollisionBetween, etc.
            * Every time you call this method it will destroy any previously created bodies and remove them from the world.
            * Therefore understand it's a very expensive operation and not to be done in a core game update loop.
            * 
            * @param map The Tilemap to get the map data from.
            * @param layer The layer to operate on. If not given will default to map.currentLayer.
            * @param addToWorld If true it will automatically add each body to the world, otherwise it's up to you to do so. - Default: true
            * @param optimize If true adjacent colliding tiles will be combined into a single body to save processing. However it means you cannot perform specific Tile to Body collision responses. - Default: true
            * @return An array of the Phaser.Physics.P2.Body objects that were created.
            */
            convertTilemap(map: Phaser.Tilemap, layer?: any, addToWorld?: Boolean, optimize?: boolean): Phaser.Physics.P2.Body[];

            /**
            * Creates a new Body and adds it to the World.
            * 
            * @param x The x coordinate of Body.
            * @param y The y coordinate of Body.
            * @param mass The mass of the Body. A mass of 0 means a 'static' Body is created.
            * @param addToWorld Automatically add this Body to the world? (usually false as it won't have any shapes on construction).
            * @param options An object containing the build options:
            * @param options.optimalDecomp Set to true if you need optimal decomposition. Warning: very slow for polygons with more than 10 vertices.
            * @param options.skipSimpleCheck Set to true if you already know that the path is not intersecting itself.
            * @param options.removeCollinearPoints Set to a number (angle threshold value) to remove collinear points, or false to keep all points.
            * @param points An array of 2d vectors that form the convex or concave polygon.
            *               Either [[0,0], [0,1],...] or a flat array of numbers that will be interpreted as [x,y, x,y, ...],
            *               or the arguments passed can be flat x,y values e.g. `setPolygon(options, x,y, x,y, x,y, ...)` where `x` and `y` are numbers.
            * @return The body
            */
            createBody(x: number, y: number, mass: number, addToWorld?: boolean, options?: p2.BodyOptions, data?: number[][]): Phaser.Physics.P2.Body;

            /**
            * Creates a new Body and adds it to the World.
            * 
            * @param x The x coordinate of Body.
            * @param y The y coordinate of Body.
            * @param mass The mass of the Body. A mass of 0 means a 'static' Body is created.
            * @param addToWorld Automatically add this Body to the world? (usually false as it won't have any shapes on construction).
            * @param options An object containing the build options:
            * @param options.optimalDecomp Set to true if you need optimal decomposition. Warning: very slow for polygons with more than 10 vertices.
            * @param options.skipSimpleCheck Set to true if you already know that the path is not intersecting itself.
            * @param options.removeCollinearPoints Set to a number (angle threshold value) to remove collinear points, or false to keep all points.
            * @param points An array of 2d vectors that form the convex or concave polygon.
            *               Either [[0,0], [0,1],...] or a flat array of numbers that will be interpreted as [x,y, x,y, ...],
            *               or the arguments passed can be flat x,y values e.g. `setPolygon(options, x,y, x,y, x,y, ...)` where `x` and `y` are numbers.
            * @return The body
            */
            createBody(x: number, y: number, mass: number, addToWorld?: boolean, options?: p2.BodyOptions, data?: number[]): Phaser.Physics.P2.Body;

            /**
            * Creates a new Collision Group and optionally applies it to the given object.
            * Collision Groups are handled using bitmasks, therefore you have a fixed limit you can create before you need to re-use older groups.
            * 
            * @param object An optional Sprite or Group to apply the Collision Group to. If a Group is given it will be applied to all top-level children.
            */
            createCollisionGroup(group?: Phaser.Group): Phaser.Physics.P2.CollisionGroup;

            /**
            * Creates a new Collision Group and optionally applies it to the given object.
            * Collision Groups are handled using bitmasks, therefore you have a fixed limit you can create before you need to re-use older groups.
            * 
            * @param object An optional Sprite or Group to apply the Collision Group to. If a Group is given it will be applied to all top-level children.
            */
            createCollisionGroup(group?: Phaser.Sprite): Phaser.Physics.P2.CollisionGroup;

            /**
            * Creates a Contact Material from the two given Materials. You can then edit the properties of the Contact Material directly.
            * 
            * @param materialA The first Material to create the ContactMaterial from. If undefined it will create a new Material object first.
            * @param materialB The second Material to create the ContactMaterial from. If undefined it will create a new Material object first.
            * @param options Material options object.
            * @return The Contact Material that was created.
            */
            createContactMaterial(materialA: Phaser.Physics.P2.Material, materialB: Phaser.Physics.P2.Material, options?: p2.ContactMaterialOptions): Phaser.Physics.P2.ContactMaterial;

            /**
            * Creates a constraint that tries to keep the distance between two bodies constant.
            * 
            * @param bodyA First connected body.
            * @param bodyB Second connected body.
            * @param distance The distance to keep between the bodies.
            * @param localAnchorA The anchor point for bodyA, defined locally in bodyA frame. Defaults to [0,0].
            * @param localAnchorB The anchor point for bodyB, defined locally in bodyB frame. Defaults to [0,0].
            * @param maxForce The maximum force that should be applied to constrain the bodies.
            * @return The constraint
            */
            createDistanceConstraint(bodyA: any, bodyB: any, distance: number, localAnchorA?: number[], localAnchorB?: number[], maxForce?: number): Phaser.Physics.P2.DistanceConstraint;

            /**
            * Creates a constraint that tries to keep the distance between two bodies constant.
            * 
            * @param bodyA First connected body.
            * @param bodyB Second connected body.
            * @param angle The relative angle
            * @param ratio The gear ratio. - Default: 1
            * @return The constraint
            */
            createGearConstraint(bodyA: any, bodyB: any, angle?: number, ratio?: number): Phaser.Physics.P2.GearConstraint;

            /**
            * Locks the relative position between two bodies.
            * 
            * @param bodyA First connected body.
            * @param bodyB Second connected body.
            * @param offset The offset of bodyB in bodyA's frame. The value is an array with 2 elements matching x and y, i.e: [32, 32].
            * @param angle The angle of bodyB in bodyA's frame.
            * @param maxForce The maximum force that should be applied to constrain the bodies.
            * @return The constraint
            */
            createLockConstraint(bodyA: any, bodyB: any, offset?: number[], angle?: number, maxForce?: number): Phaser.Physics.P2.LockConstraint;

            /**
            * Creates a Material. Materials are applied to Shapes owned by a Body and can be set with Body.setMaterial().
            * Materials are a way to control what happens when Shapes collide. Combine unique Materials together to create Contact Materials.
            * Contact Materials have properties such as friction and restitution that allow for fine-grained collision control between different Materials.
            * 
            * @param name Optional name of the Material. Each Material has a unique ID but string names are handy for debugging.
            * @param body Optional Body. If given it will assign the newly created Material to the Body shapes.
            * @return The Material that was created. This is also stored in Phaser.Physics.P2.materials.
            */
            createMaterial(name?: string, body?: Phaser.Physics.P2.Body): Phaser.Physics.P2.Material;

            /**
            * Creates a new Particle and adds it to the World.
            * 
            * @param x The x coordinate of Body.
            * @param y The y coordinate of Body.
            * @param mass The mass of the Body. A mass of 0 means a 'static' Body is created.
            * @param addToWorld Automatically add this Body to the world? (usually false as it won't have any shapes on construction).
            * @param options An object containing the build options:
            * @param options.optimalDecomp Set to true if you need optimal decomposition. Warning: very slow for polygons with more than 10 vertices.
            * @param options.skipSimpleCheck Set to true if you already know that the path is not intersecting itself.
            * @param options.removeCollinearPoints Set to a number (angle threshold value) to remove collinear points, or false to keep all points.
            * @param points An array of 2d vectors that form the convex or concave polygon.
            *               Either [[0,0], [0,1],...] or a flat array of numbers that will be interpreted as [x,y, x,y, ...],
            *               or the arguments passed can be flat x,y values e.g. `setPolygon(options, x,y, x,y, x,y, ...)` where `x` and `y` are numbers.
            */
            createParticle(x: number, y: number, mass: number, addToWorld?: boolean, options?: p2.BodyOptions, data?: number[][]): Phaser.Physics.P2.Body;

            /**
            * Creates a new Particle and adds it to the World.
            * 
            * @param x The x coordinate of Body.
            * @param y The y coordinate of Body.
            * @param mass The mass of the Body. A mass of 0 means a 'static' Body is created.
            * @param addToWorld Automatically add this Body to the world? (usually false as it won't have any shapes on construction).
            * @param options An object containing the build options:
            * @param options.optimalDecomp Set to true if you need optimal decomposition. Warning: very slow for polygons with more than 10 vertices.
            * @param options.skipSimpleCheck Set to true if you already know that the path is not intersecting itself.
            * @param options.removeCollinearPoints Set to a number (angle threshold value) to remove collinear points, or false to keep all points.
            * @param points An array of 2d vectors that form the convex or concave polygon.
            *               Either [[0,0], [0,1],...] or a flat array of numbers that will be interpreted as [x,y, x,y, ...],
            *               or the arguments passed can be flat x,y values e.g. `setPolygon(options, x,y, x,y, x,y, ...)` where `x` and `y` are numbers.
            */
            createParticle(x: number, y: number, mass: number, addToWorld?: boolean, options?: p2.BodyOptions, data?: number[]): Phaser.Physics.P2.Body;

            /**
            * Constraint that only allows bodies to move along a line, relative to each other.
            * See http://www.iforce2d.net/b2dtut/joints-prismatic
            * 
            * @param bodyA First connected body.
            * @param bodyB Second connected body.
            * @param lockRotation If set to false, bodyB will be free to rotate around its anchor point. - Default: true
            * @param anchorA Body A's anchor point, defined in its own local frame. The value is an array with 2 elements matching x and y, i.e: [32, 32].
            * @param anchorB Body A's anchor point, defined in its own local frame. The value is an array with 2 elements matching x and y, i.e: [32, 32].
            * @param axis An axis, defined in body A frame, that body B's anchor point may slide along. The value is an array with 2 elements matching x and y, i.e: [32, 32].
            * @param maxForce The maximum force that should be applied to constrain the bodies.
            * @return The constraint
            */
            createPrismaticConstraint(body: any, bodyB: any, lockRotation?: boolean, anchorA?: number[], anchorB?: number[], axis?: Float32Array, maxForce?: number): Phaser.Physics.P2.PrismaticConstraint;

            /**
            * Connects two bodies at given offset points, letting them rotate relative to each other around this point.
            * The pivot points are given in world (pixel) coordinates.
            * 
            * @param bodyA First connected body.
            * @param pivotA The point relative to the center of mass of bodyA which bodyA is constrained to. The value is an array with 2 elements matching x and y, i.e: [32, 32].
            * @param bodyB Second connected body.
            * @param pivotB The point relative to the center of mass of bodyB which bodyB is constrained to. The value is an array with 2 elements matching x and y, i.e: [32, 32].
            * @param maxForce The maximum force that should be applied to constrain the bodies.
            * @param worldPivot A pivot point given in world coordinates. If specified, localPivotA and localPivotB are automatically computed from this value.
            * @return The constraint
            */
            createRevoluteConstraint(bodyA: any, pivotA: number[], bodyB: any, pivotB: number[], maxForce?: number, worldPivot?: number[]): Phaser.Physics.P2.RevoluteConstraint;

            /**
            * Creates a rotational spring, connecting two bodies. A spring can have a resting length, a stiffness and damping.
            * 
            * @param bodyA First connected body.
            * @param bodyB Second connected body.
            * @param restAngle The relative angle of bodies at which the spring is at rest. If not given, it's set to the current relative angle between the bodies.
            * @param stiffness Stiffness of the spring. A number >= 0. - Default: 100
            * @param damping Damping of the spring. A number >= 0. - Default: 1
            * @return The spring
            */
            createRotationalSpring(bodyA: any, bodyB: any, restAngle?: number, stiffness?: number, damping?: number): p2.RotationalSpring;

            /**
            * Creates a linear spring, connecting two bodies. A spring can have a resting length, a stiffness and damping.
            * 
            * @param bodyA First connected body.
            * @param bodyB Second connected body.
            * @param restLength Rest length of the spring. A number > 0. - Default: 1
            * @param stiffness Stiffness of the spring. A number >= 0. - Default: 100
            * @param damping Damping of the spring. A number >= 0. - Default: 1
            * @param worldA Where to hook the spring to body A in world coordinates. This value is an array by 2 elements, x and y, i.e: [32, 32].
            * @param worldB Where to hook the spring to body B in world coordinates. This value is an array by 2 elements, x and y, i.e: [32, 32].
            * @param localA Where to hook the spring to body A in local body coordinates. This value is an array by 2 elements, x and y, i.e: [32, 32].
            * @param localB Where to hook the spring to body B in local body coordinates. This value is an array by 2 elements, x and y, i.e: [32, 32].
            * @return The spring
            */
            createSpring(bodyA: any, bodyB: any, restLength?: number, stiffness?: number, damping?: number, worldA?: number[], worldB?: number[], localA?: number[], localB?: number[]): Phaser.Physics.P2.Spring;

            /**
            * Clears all bodies from the simulation and unlinks World from Game. Should only be called on game shutdown. Call `clear` on a State change.
            */
            destroy(): void;

            /**
            * This will create a P2 Physics body on the given game object or array of game objects.
            * A game object can only have 1 physics body active at any one time, and it can't be changed until the object is destroyed.
            * Note: When the game object is enabled for P2 physics it has its anchor x/y set to 0.5 so it becomes centered.
            * 
            * @param object The game object to create the physics body on. Can also be an array or Group of objects, a body will be created on every child that has a `body` property.
            * @param debug Create a debug object to go with this body?
            * @param children Should a body be created on all children of this object? If true it will recurse down the display list as far as it can go. - Default: true
            */
            enable(object: any, debug?: boolean, children?: boolean): void;

            /**
            * Creates a P2 Physics body on the given game object.
            * A game object can only have 1 physics body active at any one time, and it can't be changed until the body is nulled.
            * 
            * @param object The game object to create the physics body on. A body will only be created if this object has a null `body` property.
            * @param debug Create a debug object to go with this body?
            */
            enableBody(object: any, debug: boolean): void;

            /**
            * Handles a p2 end contact event.
            * 
            * @param event The event data.
            */
            endContactHandler(event: any): void;

            /**
            * Populates and returns an array with references to of all current Bodies in the world.
            * @return An array containing all current Bodies in the world.
            */
            getBodies(): Phaser.Physics.P2.Body[];

            /**
            * Checks the given object to see if it has a p2.Body and if so returns it.
            * 
            * @param object The object to check for a p2.Body on.
            * @return The p2.Body, or null if not found.
            */
            getBody(object: any): Phaser.Physics.P2.Body;

            /**
            * Populates and returns an array of all current Constraints in the world.
            * You will get an array of p2 constraints back. This can be of mixed types, for example the array may contain
            * PrismaticConstraints, RevoluteConstraints or any other valid p2 constraint type.
            * @return An array containing all current Constraints in the world.
            */
            getConstraints(): p2.Constraint[];

            /**
            * Populates and returns an array of all current Springs in the world.
            * @return An array containing all current Springs in the world.
            */
            getSprings(): Phaser.Physics.P2.Spring[];

            /**
            * Gets a Contact Material based on the two given Materials.
            * 
            * @param materialA The first Material to search for.
            * @param materialB The second Material to search for.
            * @return The Contact Material or false if none was found matching the Materials given.
            */
            getContactMaterial(materialA: Phaser.Physics.P2.Material, materialB: Phaser.Physics.P2.Material): Phaser.Physics.P2.ContactMaterial;

            /**
            * Test if a world point overlaps bodies. You will get an array of actual P2 bodies back. You can find out which Sprite a Body belongs to
            * (if any) by checking the Body.parent.sprite property. Body.parent is a Phaser.Physics.P2.Body property.
            * 
            * @param worldPoint Point to use for intersection tests. The points values must be in world (pixel) coordinates.
            * @param bodies A list of objects to check for intersection. If not given it will check Phaser.Physics.P2.world.bodies (i.e. all world bodies)
            * @param precision Used for matching against particles and lines. Adds some margin to these infinitesimal objects. - Default: 5
            * @param filterStatic If true all Static objects will be removed from the results array.
            * @return Array of bodies that overlap the point.
            */
            hitTest(worldPoint: Phaser.Point, bodies?: any[], precision?: number, filterStatic?: boolean): Phaser.Physics.P2.Body[];

            /**
            * Convert p2 physics value (meters) to pixel scale.
            * By default Phaser uses a scale of 20px per meter.
            * If you need to modify this you can over-ride these functions via the Physics Configuration object.
            * 
            * @param v The value to convert.
            * @return The scaled value.
            */
            mpx(v: number): number;

            /**
            * Convert p2 physics value (meters) to pixel scale and inverses it.
            * By default Phaser uses a scale of 20px per meter.
            * If you need to modify this you can over-ride these functions via the Physics Configuration object.
            * 
            * @param v The value to convert.
            * @return The scaled value.
            */
            mpxi(v: number): number;

            /**
            * Pauses the P2 World independent of the game pause state.
            */
            pause(): void;

            /**
            * Called at the start of the core update loop. Purges flagged bodies from the world.
            */
            preUpdate(): void;

            /**
            * Convert pixel value to p2 physics scale (meters).
            * By default Phaser uses a scale of 20px per meter.
            * If you need to modify this you can over-ride these functions via the Physics Configuration object.
            * 
            * @param v The value to convert.
            * @return The scaled value.
            */
            pxm(v: number): number;

            /**
            * Convert pixel value to p2 physics scale (meters) and inverses it.
            * By default Phaser uses a scale of 20px per meter.
            * If you need to modify this you can over-ride these functions via the Physics Configuration object.
            * 
            * @param v The value to convert.
            * @return The scaled value.
            */
            pxmi(v: number): number;

            /**
            * Removes a body from the world. This will silently fail if the body wasn't part of the world to begin with.
            * 
            * @param body The Body to remove from the World.
            * @return The Body that was removed.
            */
            removeBody(body: Phaser.Physics.P2.Body): Phaser.Physics.P2.Body;

            /**
            * This will add a P2 Physics body into the removal list for the next step.
            * 
            * @param body The body to remove at the start of the next step.
            */
            removeBodyNextStep(body: Phaser.Physics.P2.Body): void;

            /**
            * Removes a Constraint from the world.
            * 
            * @param constraint The Constraint to be removed from the World.
            * @return The Constraint that was removed.
            */
            removeConstraint<T>(constraint: T): T;

            /**
            * Removes a Contact Material from the world.
            * 
            * @param material The Contact Material to be removed from the World.
            * @return The Contact Material that was removed.
            */
            removeContactMaterial(material: Phaser.Physics.P2.ContactMaterial): Phaser.Physics.P2.ContactMaterial;

            /**
            * Removes a Spring from the world.
            * 
            * @param spring The Spring to remove from the World.
            * @return The Spring that was removed.
            */
            removeSpring(spring: Phaser.Physics.P2.Spring): Phaser.Physics.P2.Spring;

            /**
            * Called by Phaser.Physics when a State swap occurs.
            * Starts the begin and end Contact listeners again.
            */
            reset(): void;

            /**
            * Resumes a paused P2 World.
            */
            resume(): void;

            /**
            * Sets the bounds of the Physics world to match the given world pixel dimensions.
            * You can optionally set which 'walls' to create: left, right, top or bottom.
            * If none of the walls are given it will default to use the walls settings it had previously.
            * I.e. if you previously told it to not have the left or right walls, and you then adjust the world size
            * the newly created bounds will also not have the left and right walls.
            * Explicitly state them in the parameters to override this.
            * 
            * @param x The x coordinate of the top-left corner of the bounds.
            * @param y The y coordinate of the top-left corner of the bounds.
            * @param width The width of the bounds.
            * @param height The height of the bounds.
            * @param left If true will create the left bounds wall. - Default: true
            * @param right If true will create the right bounds wall. - Default: true
            * @param top If true will create the top bounds wall. - Default: true
            * @param bottom If true will create the bottom bounds wall. - Default: true
            * @param setCollisionGroup If true the Bounds will be set to use its own Collision Group. - Default: true
            */
            setBounds(x: number, y: number, width: number, height: number, left?: Boolean, right?: boolean, top?: boolean, bottom?: boolean, setCollisionGroup?: boolean): void;
            setBoundsToWorld(left?: boolean, right?: boolean, top?: boolean, bottom?: boolean, setCollisionGroup?: boolean): void;
            setCollisionGroup(object: any, group: Phaser.Physics.P2.CollisionGroup): void;

            /**
            * Impact event handling is disabled by default. Enable it before any impact events will be dispatched.
            * In a busy world hundreds of impact events can be generated every step, so only enable this if you cannot do what you need via beginContact or collision masks.
            * 
            * @param state Set to true to enable impact events, or false to disable.
            */
            setImpactEvents(state: boolean): void;

            /**
            * Sets the given Material against all Shapes owned by all the Bodies in the given array.
            * 
            * @param material The Material to be applied to the given Bodies.
            * @param bodies An Array of Body objects that the given Material will be set on.
            */
            setMaterial(material: Phaser.Physics.P2.Material, bodies?: Phaser.Physics.P2.Body[]): void;

            /**
            * Sets a callback to be fired after the Broadphase has collected collision pairs in the world.
            * Just because a pair exists it doesn't mean they *will* collide, just that they potentially could do.
            * If your calback returns `false` the pair will be removed from the narrowphase. This will stop them testing for collision this step.
            * Returning `true` from the callback will ensure they are checked in the narrowphase.
            * 
            * @param callback The callback that will receive the postBroadphase event data. It must return a boolean. Set to null to disable an existing callback.
            * @param context The context under which the callback will be fired.
            */
            setPostBroadphaseCallback(callback: Function, context: any): void;
            setWorldMaterial(material: Phaser.Physics.P2.Material, left?: boolean, right?: boolean, top?: boolean, bottom?: boolean): void;

            /**
            * Converts the current world into a JSON object.
            * @return A JSON representation of the world.
            */
            toJSON(): any;

            /**
            * Internal P2 update loop.
            */
            update(): void;

            /**
            * By default the World will be set to collide everything with everything. The bounds of the world is a Body with 4 shapes, one for each face.
            * If you start to use your own collision groups then your objects will no longer collide with the bounds.
            * To fix this you need to adjust the bounds to use its own collision group first BEFORE changing your Sprites collision group.
            * 
            * @param setCollisionGroup If true the Bounds will be set to use its own Collision Group. - Default: true
            */
            updateBoundsCollisionGroup(setCollisionGroup?: boolean): void;

        }

        module P2 {


            /**
            * The Physics Body is typically linked to a single Sprite and defines properties that determine how the physics body is simulated.
            * These properties affect how the body reacts to forces, what forces it generates on itself (to simulate friction), and how it reacts to collisions in the scene.
            * In most cases, the properties are used to simulate physical effects. Each body also has its own property values that determine exactly how it reacts to forces and collisions in the scene.
            * By default a single Rectangle shape is added to the Body that matches the dimensions of the parent Sprite. See addShape, removeShape, clearShapes to add extra shapes around the Body.
            * Note: When bound to a Sprite to avoid single-pixel jitters on mobile devices we strongly recommend using Sprite sizes that are even on both axis, i.e. 128x128 not 127x127.
            * Note: When a game object is given a P2 body it has its anchor x/y set to 0.5, so it becomes centered.
            */
            class Body {


                /**
                * Dynamic body. Dynamic bodies body can move and respond to collisions and forces.
                */
                static DYNAMIC: number;

                /**
                * Static body. Static bodies do not move, and they do not respond to forces or collision.
                */
                static STATIC: number;

                /**
                * Kinematic body. Kinematic bodies only moves according to its .velocity, and does not respond to collisions or force.
                */
                static KINEMATIC: number;


                /**
                * The Physics Body is typically linked to a single Sprite and defines properties that determine how the physics body is simulated.
                * These properties affect how the body reacts to forces, what forces it generates on itself (to simulate friction), and how it reacts to collisions in the scene.
                * In most cases, the properties are used to simulate physical effects. Each body also has its own property values that determine exactly how it reacts to forces and collisions in the scene.
                * By default a single Rectangle shape is added to the Body that matches the dimensions of the parent Sprite. See addShape, removeShape, clearShapes to add extra shapes around the Body.
                * Note: When bound to a Sprite to avoid single-pixel jitters on mobile devices we strongly recommend using Sprite sizes that are even on both axis, i.e. 128x128 not 127x127.
                * Note: When a game object is given a P2 body it has its anchor x/y set to 0.5, so it becomes centered.
                * 
                * @param game Game reference to the currently running game.
                * @param sprite The Sprite object this physics body belongs to.
                * @param x The x coordinate of this Body.
                * @param y The y coordinate of this Body.
                * @param mass The default mass of this Body (0 = static). - Default: 1
                */
                constructor(game: Phaser.Game, sprite?: Phaser.Sprite, x?: number, y?: number, mass?: number);


                /**
                * -
                */
                allowSleep: boolean;

                /**
                * The angle of the Body in degrees from its original orientation. Values from 0 to 180 represent clockwise rotation; values from 0 to -180 represent counterclockwise rotation.
                * Values outside this range are added to or subtracted from 360 to obtain a value within the range. For example, the statement Body.angle = 450 is the same as Body.angle = 90.
                * If you wish to work in radians instead of degrees use the property Body.rotation instead. Working in radians is faster as it doesn't have to convert values. The angle of this Body in degrees.
                */
                angle: number;

                /**
                * Damping is specified as a value between 0 and 1, which is the proportion of velocity lost per second. The angular damping acting acting on the body.
                */
                angularDamping: number;

                /**
                * The angular force acting on the body.
                */
                angularForce: number;

                /**
                * The angular velocity of the body.
                */
                angularVelocity: number;

                /**
                * Array of CollisionGroups that this Bodies shapes collide with.
                */
                collidesWith: Phaser.Physics.P2.CollisionGroup[];

                /**
                * A Body can be set to collide against the World bounds automatically if this is set to true. Otherwise it will leave the World.
                * Note that this only applies if your World has bounds! The response to the collision should be managed via CollisionMaterials.
                * Also note that when you set this it will only affect Body shapes that already exist. If you then add further shapes to your Body
                * after setting this it will *not* proactively set them to collide with the bounds. Should the Body collide with the World bounds?
                * Default: true
                */
                collideWorldBounds: boolean;

                /**
                * Damping is specified as a value between 0 and 1, which is the proportion of velocity lost per second. The linear damping acting on the body in the velocity direction.
                */
                damping: number;

                /**
                * The p2 Body data.
                */
                data: p2.Body;

                /**
                * Enable or disable debug drawing of this body
                */
                debug: boolean;

                /**
                * Reference to the debug body.
                */
                debugBody: Phaser.Physics.P2.BodyDebug;

                /**
                * Returns true if the Body is dynamic. Setting Body.dynamic to 'false' will make it static.
                */
                dynamic: boolean;

                /**
                * -
                */
                fixedRotation: boolean;

                /**
                * The force applied to the body.
                */
                force: Phaser.Physics.P2.InversePointProxy;

                /**
                * Returns true if the Body is kinematic. Setting Body.kinematic to 'false' will make it static.
                */
                kinematic: boolean;

                /**
                * Local reference to game.
                */
                game: Phaser.Game;

                /**
                * A locally applied gravity force to the Body. Applied directly before the world step. NOTE: Not currently implemented.
                */
                gravity: Phaser.Point;

                /**
                * The Body ID. Each Body that has been added to the World has a unique ID.
                */
                id: number;

                /**
                * The inertia of the body around the Z axis..
                */
                inertia: number;

                /**
                * The mass of the body.
                */
                mass: number;

                /**
                * The type of motion this body has. Should be one of: Body.STATIC (the body does not move), Body.DYNAMIC (body can move and respond to collisions) and Body.KINEMATIC (only moves according to its .velocity).
                */
                motionState: number;

                /**
                * The offset of the Physics Body from the Sprite x/y position.
                */
                offset: Phaser.Point;

                /**
                * Dispatched when a first contact is created between shapes in two bodies.
                * This event is fired during the step, so collision has already taken place.
                * 
                * The event will be sent 5 arguments in this order:
                * 
                * The Phaser.Physics.P2.Body it is in contact with. *This might be null* if the Body was created directly in the p2 world.
                * The p2.Body this Body is in contact with.
                * The Shape from this body that caused the contact.
                * The Shape from the contact body.
                * The Contact Equation data array.
                */
                onBeginContact: Phaser.Signal;

                /**
                * Dispatched when contact ends between shapes in two bodies.
                * This event is fired during the step, so collision has already taken place.
                * 
                * The event will be sent 4 arguments in this order:
                * 
                * The Phaser.Physics.P2.Body it is in contact with. *This might be null* if the Body was created directly in the p2 world.
                * The p2.Body this Body has ended contact with.
                * The Shape from this body that caused the original contact.
                * The Shape from the contact body.
                */
                onEndContact: Phaser.Signal;

                /**
                * The angle of the Body in radians.
                * If you wish to work in degrees instead of radians use the Body.angle property instead. Working in radians is faster as it doesn't have to convert values. The angle of this Body in radians.
                */
                rotation: number;

                /**
                * To avoid deleting this body during a physics step, and causing all kinds of problems, set removeNextStep to true to have it removed in the next preUpdate.
                */
                removeNextStep: boolean;

                /**
                * Reference to the parent Sprite.
                */
                sprite: Phaser.Sprite;

                /**
                * .
                */
                sleepSpeedLimit: number;

                /**
                * Returns true if the Body is static. Setting Body.static to 'false' will make it dynamic.
                */
                static: boolean;

                /**
                * The type of physics system this body belongs to.
                */
                type: number;

                /**
                * The velocity of the body. Set velocity.x to a negative value to move to the left, position to the right. velocity.y negative values move up, positive move down.
                */
                velocity: Phaser.Physics.P2.InversePointProxy;

                /**
                * Local reference to the P2 World.
                */
                world: Phaser.Physics.P2;

                /**
                * The x coordinate of this Body.
                */
                x: number;

                /**
                * The y coordinate of this Body.
                */
                y: number;


                /**
                * Adds this physics body to the world.
                */
                addToWorld(): void;

                /**
                * Adds a Capsule shape to this Body.
                * You can control the offset from the center of the body and the rotation.
                * 
                * @param length The distance between the end points in pixels.
                * @param radius Radius of the capsule in pixels.
                * @param offsetX Local horizontal offset of the shape relative to the body center of mass.
                * @param offsetY Local vertical offset of the shape relative to the body center of mass.
                * @param rotation Local rotation of the shape relative to the body center of mass, specified in radians.
                * @return The Capsule shape that was added to the Body.
                */
                addCapsule(length: number, radius: number, offsetX?: number, offsetY?: number, rotation?: number): p2.Capsule;

                /**
                * Adds a Circle shape to this Body. You can control the offset from the center of the body and the rotation.
                * 
                * @param radius The radius of this circle (in pixels)
                * @param offsetX Local horizontal offset of the shape relative to the body center of mass.
                * @param offsetY Local vertical offset of the shape relative to the body center of mass.
                * @param rotation Local rotation of the shape relative to the body center of mass, specified in radians.
                * @return The Circle shape that was added to the Body.
                */
                addCircle(radius: number, offsetX?: number, offsetY?: number, rotation?: number): p2.Circle;

                /**
                * Add a polygon fixture. This is used during #loadPolygon.
                * 
                * @param fixtureData The data for the fixture. It contains: isSensor, filter (collision) and the actual polygon shapes.
                * @return An array containing the generated shapes for the given polygon.
                */
                addFixture(fixtureData: string): p2.Shape[];

                /**
                * Adds a Line shape to this Body.
                * The line shape is along the x direction, and stretches from [-length/2, 0] to [length/2,0].
                * You can control the offset from the center of the body and the rotation.
                * 
                * @param length The length of this line (in pixels)
                * @param offsetX Local horizontal offset of the shape relative to the body center of mass.
                * @param offsetY Local vertical offset of the shape relative to the body center of mass.
                * @param rotation Local rotation of the shape relative to the body center of mass, specified in radians.
                * @return The Line shape that was added to the Body.
                */
                addLine(length: number, offsetX?: number, offsetY?: number, rotation?: number): p2.Line;

                /**
                * Adds a Particle shape to this Body. You can control the offset from the center of the body and the rotation.
                * 
                * @param offsetX Local horizontal offset of the shape relative to the body center of mass.
                * @param offsetY Local vertical offset of the shape relative to the body center of mass.
                * @param rotation Local rotation of the shape relative to the body center of mass, specified in radians.
                * @return The Particle shape that was added to the Body.
                */
                addParticle(offsetX?: number, offsetY?: number, rotation?: number): p2.Particle;

                /**
                * Reads a polygon shape path, and assembles convex shapes from that and puts them at proper offset points. The shape must be simple and without holes.
                * This function expects the x.y values to be given in pixels. If you want to provide them at p2 world scales then call Body.data.fromPolygon directly.
                * 
                * @param options An object containing the build options:
                * @param options.optimalDecomp Set to true if you need optimal decomposition. Warning: very slow for polygons with more than 10 vertices.
                * @param options.skipSimpleCheck Set to true if you already know that the path is not intersecting itself.
                * @param options.removeCollinearPoints Set to a number (angle threshold value) to remove collinear points, or false to keep all points.
                * @param points An array of 2d vectors that form the convex or concave polygon.
                *               Either [[0,0], [0,1],...] or a flat array of numbers that will be interpreted as [x,y, x,y, ...],
                *               or the arguments passed can be flat x,y values e.g. `setPolygon(options, x,y, x,y, x,y, ...)` where `x` and `y` are numbers.
                * @return True on success, else false.
                */
                addPolygon(options: { optimalDecomp?: boolean; skipSimpleCheck?: boolean; removeCollinearPoints?: boolean; }, points: number[][]): boolean;

                /**
                * Reads the shape data from a physics data file stored in the Game.Cache and adds it as a polygon to this Body.
                * The shape data format is based on the output of the
                * {@link https://github.com/photonstorm/phaser/tree/master/resources/PhysicsEditor%20Exporter|custom phaser exporter} for
                * {@link https://www.codeandweb.com/physicseditor|PhysicsEditor}
                * 
                * @param key The key of the Physics Data file as stored in Game.Cache.
                * @param object The key of the object within the Physics data file that you wish to load the shape data from.
                * @return A list of created fixtures to be used with Phaser.Physics.P2.FixtureList
                */
                addPhaserPolygon(key: string, object: string): Phaser.Physics.P2.FixtureList;

                /**
                * Adds a Plane shape to this Body. The plane is facing in the Y direction. You can control the offset from the center of the body and the rotation.
                * 
                * @param offsetX Local horizontal offset of the shape relative to the body center of mass.
                * @param offsetY Local vertical offset of the shape relative to the body center of mass.
                * @param rotation Local rotation of the shape relative to the body center of mass, specified in radians.
                * @return The Plane shape that was added to the Body.
                */
                addPlane(offsetX?: number, offsetY?: number, rotation?: number): p2.Plane;

                /**
                * Adds a Rectangle shape to this Body. You can control the offset from the center of the body and the rotation.
                * 
                * @param width The width of the rectangle in pixels.
                * @param height The height of the rectangle in pixels.
                * @param offsetX Local horizontal offset of the shape relative to the body center of mass.
                * @param offsetY Local vertical offset of the shape relative to the body center of mass.
                * @param rotation Local rotation of the shape relative to the body center of mass, specified in radians.
                * @return The shape that was added to the Body.
                */
                addRectangle(width: number, height: number, offsetX?: number, offsetY?: number, rotation?: number): p2.Rectangle;

                /**
                * Add a shape to the body. You can pass a local transform when adding a shape, so that the shape gets an offset and an angle relative to the body center of mass.
                * Will automatically update the mass properties and bounding radius.
                * If this Body had a previously set Collision Group you will need to re-apply it to the new Shape this creates.
                * 
                * @param shape The shape to add to the body.
                * @param offsetX Local horizontal offset of the shape relative to the body center of mass.
                * @param offsetY Local vertical offset of the shape relative to the body center of mass.
                * @param rotation Local rotation of the shape relative to the body center of mass, specified in radians.
                * @return The shape that was added to the body.
                */
                addShape(shape: p2.Shape, offsetX?: number, offsetY?: number, rotation?: number): p2.Shape;

                /**
                * Moves the shape offsets so their center of mass becomes the body center of mass.
                */
                adjustCenterOfMass(): void;

                /**
                * Apply damping, see http://code.google.com/p/bullet/issues/detail?id=74 for details.
                * 
                * @param dt Current time step.
                */
                applyDamping(dt: number): void;

                /**
                * Apply force to a world point.
                * 
                * This could for example be a point on the RigidBody surface. Applying force
                * this way will add to Body.force and Body.angularForce.
                * 
                * @param force The force vector to add.
                * @param worldX The world x point to apply the force on.
                * @param worldY The world y point to apply the force on.
                */
                applyForce(force: number[], worldX: number, worldY: number): void;

                /**
                * Apply impulse to a point relative to the body.
                * This could for example be a point on the Body surface. An impulse is a force added to a body during a short
                * period of time (impulse = force * time). Impulses will be added to Body.velocity and Body.angularVelocity.
                * 
                * @param impulse The impulse vector to add, oriented in world space.
                * @param worldX A point relative to the body in world space. If not given, it is set to zero and all of the impulse will be exerted on the center of mass.
                * @param worldY A point relative to the body in world space. If not given, it is set to zero and all of the impulse will be exerted on the center of mass.
                */
                applyImpulse(impulse: number[], worldX: number, worldY: number): void;

                /**
                * Apply impulse to a point local to the body.
                * 
                * This could for example be a point on the Body surface. An impulse is a force added to a body during a short
                * period of time (impulse = force * time). Impulses will be added to Body.velocity and Body.angularVelocity.
                * 
                * @param impulse The impulse vector to add, oriented in local space.
                * @param localX A local point on the body.
                * @param localY A local point on the body.
                */
                applyImpulseLocal(impulse: number[], localX: number, localY: number): void;

                /**
                * Clears the collision data from the shapes in this Body. Optionally clears Group and/or Mask.
                * 
                * @param clearGroup Clear the collisionGroup value from the shape/s? - Default: true
                * @param clearMask Clear the collisionMask value from the shape/s? - Default: true
                * @param shape An optional Shape. If not provided the collision data will be cleared from all Shapes in this Body.
                */
                clearCollision(clearGroup?: boolean, cleanMask?: boolean, shape?: p2.Shape): void;

                /**
                * Removes all Shapes from this Body.
                */
                clearShapes(): void;

                /**
                * Adds the given CollisionGroup, or array of CollisionGroups, to the list of groups that this body will collide with and updates the collision masks.
                * 
                * @param group The Collision Group or Array of Collision Groups that this Bodies shapes will collide with.
                * @param callback Optional callback that will be triggered when this Body impacts with the given Group.
                * @param callbackContext The context under which the callback will be called.
                * @param shape An optional Shape. If not provided the collision mask will be added to all Shapes in this Body.
                */
                collides(group: any, callback?: Function, callbackContext?: any, shape?: p2.Shape): void;

                /**
                * Sets a callback to be fired any time a shape in this Body impacts with a shape in the given Body. The impact test is performed against body.id values.
                * The callback will be sent 4 parameters: This body, the body that impacted, the Shape in this body and the shape in the impacting body.
                * Note that the impact event happens after collision resolution, so it cannot be used to prevent a collision from happening.
                * It also happens mid-step. So do not destroy a Body during this callback, instead set safeDestroy to true so it will be killed on the next preUpdate.
                * 
                * @param object The object to send impact events for.
                * @param callback The callback to fire on impact. Set to null to clear a previously set callback.
                * @param callbackContext The context under which the callback will fire.
                */
                createBodyCallback(object: any, callback: Function, callbackContext: any): void;

                /**
                * Sets a callback to be fired any time this Body impacts with the given Group. The impact test is performed against shape.collisionGroup values.
                * The callback will be sent 4 parameters: This body, the body that impacted, the Shape in this body and the shape in the impacting body.
                * This callback will only fire if this Body has been assigned a collision group.
                * Note that the impact event happens after collision resolution, so it cannot be used to prevent a collision from happening.
                * It also happens mid-step. So do not destroy a Body during this callback, instead set safeDestroy to true so it will be killed on the next preUpdate.
                * 
                * @param group The Group to send impact events for.
                * @param callback The callback to fire on impact. Set to null to clear a previously set callback.
                * @param callbackContext The context under which the callback will fire.
                */
                createGroupCallback(group: Phaser.Physics.P2.CollisionGroup, callback: Function, callbackContext: any): void;

                /**
                * Destroys this Body and all references it holds to other objects.
                */
                destroy(): void;

                /**
                * Gets the collision bitmask from the groups this body collides with.
                * @return The bitmask.
                */
                getCollisionMask(): number;

                /**
                * Gets the velocity of a point in the body.
                * 
                * @param result A vector to store the result in.
                * @param relativePoint A world oriented vector, indicating the position of the point to get the velocity from.
                * @return The result vector.
                */
                getVelocityAtPoint(result: number[], relativePoint: number[]): number[];

                /**
                * Reads the shape data from a physics data file stored in the Game.Cache and adds it as a polygon to this Body.
                * 
                * As well as reading the data from the Cache you can also pass `null` as the first argument and a
                * physics data object as the second. When doing this you must ensure the structure of the object is correct in advance.
                * 
                * For more details see the format of the Lime / Corona Physics Editor export.
                * 
                * @param key The key of the Physics Data file as stored in Game.Cache. Alternatively set to `null` and pass the
                *            data as the 2nd argument.
                * @param object The key of the object within the Physics data file that you wish to load the shape data from,
                *               or if key is null pass the actual physics data object itself as this parameter.
                * @param scale Optionally resize the loaded polygon. - Default: 1
                * @return True on success, else false.
                */
                loadPolygon(key: string, object: string, scale ?: number): boolean;

                /**
                * Moves the Body backwards based on its current angle and the given speed.
                * The speed is represented in pixels per second. So a value of 100 would move 100 pixels in 1 second (1000ms).
                * 
                * @param speed The speed at which it should move backwards.
                */
                moveBackward(speed: number): void;

                /**
                * If this Body is dynamic then this will move it down by setting its y velocity to the given speed.
                * The speed is represented in pixels per second. So a value of 100 would move 100 pixels in 1 second (1000ms).
                * 
                * @param speed The speed at which it should move down, in pixels per second.
                */
                moveDown(speed: number): void;

                /**
                * Moves the Body forwards based on its current angle and the given speed.
                * The speed is represented in pixels per second. So a value of 100 would move 100 pixels in 1 second (1000ms).
                * 
                * @param speed The speed at which it should move forwards.
                */
                moveForward(speed: number): void;

                /**
                * If this Body is dynamic then this will move it to the left by setting its x velocity to the given speed.
                * The speed is represented in pixels per second. So a value of 100 would move 100 pixels in 1 second (1000ms).
                * 
                * @param speed The speed at which it should move to the left, in pixels per second.
                */
                moveLeft(speed: number): void;

                /**
                * If this Body is dynamic then this will move it to the right by setting its x velocity to the given speed.
                * The speed is represented in pixels per second. So a value of 100 would move 100 pixels in 1 second (1000ms).
                * 
                * @param speed The speed at which it should move to the right, in pixels per second.
                */
                moveRight(speed: number): void;

                /**
                * If this Body is dynamic then this will move it up by setting its y velocity to the given speed.
                * The speed is represented in pixels per second. So a value of 100 would move 100 pixels in 1 second (1000ms).
                * 
                * @param speed The speed at which it should move up, in pixels per second.
                */
                moveUp(speed: number): void;

                /**
                * Internal method. This is called directly before the sprites are sent to the renderer and after the update function has finished.
                */
                preUpdate(): void;

                /**
                * Internal method. This is called directly before the sprites are sent to the renderer and after the update function has finished.
                */
                postUpdate(): void;

                /**
                * Removes the given CollisionGroup, or array of CollisionGroups, from the list of groups that this body will collide with and updates the collision masks.
                * 
                * @param group The Collision Group or Array of Collision Groups that this Bodies shapes should not collide with anymore.
                * @param clearCallback Clear the callback that will be triggered when this Body impacts with the given Group? - Default: true
                * @param shape An optional Shape. If not provided the updated collision mask will be added to all Shapes in this Body.
                */
                removeCollisionGroup(group: any, clearCallback?: boolean, shape?: p2.Shape): void;

                /**
                * Removes this physics body from the world.
                */
                removeFromWorld(): void;

                /**
                * Remove a shape from the body. Will automatically update the mass properties and bounding radius.
                * 
                * @param shape The shape to remove from the body.
                * @return True if the shape was found and removed, else false.
                */
                removeShape(shape: p2.Shape): boolean;

                /**
                * Applies a force to the Body that causes it to 'thrust' backwards (in reverse), based on its current angle and the given speed.
                * The speed is represented in pixels per second. So a value of 100 would move 100 pixels in 1 second (1000ms).
                * 
                * @param speed The speed at which it should reverse.
                */
                reverse(speed: number): void;

                /**
                * This will rotate the Body by the given speed to the left (counter-clockwise).
                * 
                * @param speed The speed at which it should rotate.
                */
                rotateLeft(speed: number): void;

                /**
                * This will rotate the Body by the given speed to the left (clockwise).
                * 
                * @param speed The speed at which it should rotate.
                */
                rotateRight(speed: number): void;

                /**
                * Resets the Body force, velocity (linear and angular) and rotation. Optionally resets damping and mass.
                * 
                * @param x The new x position of the Body.
                * @param y The new x position of the Body.
                * @param resetDamping Resets the linear and angular damping.
                * @param resetMass Sets the Body mass back to 1.
                */
                reset(x: number, y: number, resetDamping?: boolean, resetMass?: boolean): void;

                /**
                * Updates the debug draw if any body shapes change.
                */
                shapeChanged(): void;

                /**
                * Clears any previously set shapes. Then creates a new Circle shape and adds it to this Body.
                * If this Body had a previously set Collision Group you will need to re-apply it to the new Shape this creates.
                * 
                * @param radius The radius of this circle (in pixels)
                * @param offsetX Local horizontal offset of the shape relative to the body center of mass.
                * @param offsetY Local vertical offset of the shape relative to the body center of mass.
                * @param rotation Local rotation of the shape relative to the body center of mass, specified in radians.
                */
                setCircle(radius: number, offsetX?: number, offsetY?: number, rotation?: number): p2.Circle;

                /**
                * Sets the given CollisionGroup to be the collision group for all shapes in this Body, unless a shape is specified.
                * This also resets the collisionMask.
                * 
                * @param group The Collision Group that this Bodies shapes will use.
                * @param shape An optional Shape. If not provided the collision group will be added to all Shapes in this Body.
                */
                setCollisionGroup(group: Phaser.Physics.P2.CollisionGroup, shape?: p2.Shape): void;

                /**
                * Clears any previously set shapes. The creates a new Rectangle shape at the given size and offset, and adds it to this Body.
                * If you wish to create a Rectangle to match the size of a Sprite or Image see Body.setRectangleFromSprite.
                * If this Body had a previously set Collision Group you will need to re-apply it to the new Shape this creates.
                * 
                * @param width The width of the rectangle in pixels. - Default: 16
                * @param height The height of the rectangle in pixels. - Default: 16
                * @param offsetX Local horizontal offset of the shape relative to the body center of mass.
                * @param offsetY Local vertical offset of the shape relative to the body center of mass.
                * @param rotation Local rotation of the shape relative to the body center of mass, specified in radians.
                * @return The Rectangle shape that was added to the Body.
                */
                setRectangle(width?: number, height?: number, offsetX?: number, offsetY?: number, rotation?: number): p2.Rectangle;

                /**
                * Clears any previously set shapes.
                * Then creates a Rectangle shape sized to match the dimensions and orientation of the Sprite given.
                * If no Sprite is given it defaults to using the parent of this Body.
                * If this Body had a previously set Collision Group you will need to re-apply it to the new Shape this creates.
                * 
                * @param sprite The Sprite on which the Rectangle will get its dimensions.
                * @return The Rectangle shape that was added to the Body.
                */
                setRectangleFromSprite(sprite: any): p2.Rectangle;

                /**
                * Adds the given Material to all Shapes that belong to this Body.
                * If you only wish to apply it to a specific Shape in this Body then provide that as the 2nd parameter.
                * 
                * @param material The Material that will be applied.
                * @param shape An optional Shape. If not provided the Material will be added to all Shapes in this Body.
                */
                setMaterial(material: Phaser.Physics.P2.Material, shape?: p2.Shape): void;

                /**
                * Sets the Body damping and angularDamping to zero.
                */
                setZeroDamping(): void;

                /**
                * Sets the force on the body to zero.
                */
                setZeroForce(): void;

                /**
                * If this Body is dynamic then this will zero its angular velocity.
                */
                setZeroRotation(): void;

                /**
                * If this Body is dynamic then this will zero its velocity on both axis.
                */
                setZeroVelocity(): void;

                /**
                * Transform a world point to local body frame.
                * 
                * @param out The vector to store the result in.
                * @param worldPoint The input world vector.
                */
                toLocalFrame(out: number[], worldPoint: number[]): void;

                /**
                * Applies a force to the Body that causes it to 'thrust' forwards, based on its current angle and the given speed.
                * The speed is represented in pixels per second. So a value of 100 would move 100 pixels in 1 second (1000ms).
                * 
                * @param speed The speed at which it should thrust.
                */
                thrust(speed: number): void;

                /**
                * Applies a force to the Body that causes it to 'thrust' to the left, based on its current angle and the given speed.
                * The speed is represented in pixels per second. So a value of 100 would move 100 pixels in 1 second (1000ms).
                * 
                * @param speed The speed at which it should move to the left.
                */
                thrustLeft(speed: number): void;

                /**
                * Applies a force to the Body that causes it to 'thrust' to the right, based on its current angle and the given speed.
                * The speed is represented in pixels per second. So a value of 100 would move 100 pixels in 1 second (1000ms).
                * 
                * @param speed The speed at which it should move to the right.
                */
                thrustRight(speed: number): void;

                /**
                * Transform a local point to world frame.
                * 
                * @param out The vector to store the result in.
                * @param localPoint The input local vector.
                */
                toWorldFrame(out: number[], localPoint: number[]): void;

                /**
                * Updates the collisionMask.
                * 
                * @param shape An optional Shape. If not provided the collision group will be added to all Shapes in this Body.
                */
                updateCollisionMask(shape?: p2.Shape): void;

            }


            /**
            * Draws a P2 Body to a Graphics instance for visual debugging.
            * Needless to say, for every body you enable debug drawing on, you are adding processor and graphical overhead.
            * So use sparingly and rarely (if ever) in production code.
            * 
            * Also be aware that the Debug body is only updated when the Sprite it is connected to changes position. If you
            * manipulate the sprite in any other way (such as moving it to another Group or bringToTop, etc) then you will
            * need to manually adjust its BodyDebug as well.
            */
            class BodyDebug extends Phaser.Group {


                /**
                * Draws a P2 Body to a Graphics instance for visual debugging.
                * Needless to say, for every body you enable debug drawing on, you are adding processor and graphical overhead.
                * So use sparingly and rarely (if ever) in production code.
                * 
                * Also be aware that the Debug body is only updated when the Sprite it is connected to changes position. If you
                * manipulate the sprite in any other way (such as moving it to another Group or bringToTop, etc) then you will
                * need to manually adjust its BodyDebug as well.
                * 
                * @param game Game reference to the currently running game.
                * @param body The P2 Body to display debug data for.
                * @param settings Settings object.
                */

                                                                                                                                                                    /**
                                                                                                                                                                    * The alpha value of the group container.
                                                                                                                                                                    */
                constructor(game: Phaser.Game, body: Phaser.Physics.P2.Body, settings: { pixelsPerLengthUnit?: number; debugPolygons?: boolean; lineWidth?: number; alpha?: number; });


                /**
                * The P2 Body to display debug data for.
                */
                body: Phaser.Physics.P2.Body;

                /**
                * The canvas to render the debug info to.
                */
                canvas: Phaser.Graphics;

                /**
                * Pixels per Length Unit.
                */
                ppu: number;


                /**
                * Core update.
                */
                updateSpriteTransform(): void;

                /**
                * Draws the P2 shapes to the Graphics object.
                */
                draw(): void;

            }


            /**
            * Collision Group
            */
            class CollisionGroup {


                /**
                * Collision Group
                * 
                * @param bitmask The CollisionGroup bitmask.
                */
                constructor(bitmask: number);


                /**
                * The CollisionGroup bitmask.
                */
                mask: number;

            }


            /**
            * Defines a physics material
            */
            class ContactMaterial extends p2.ContactMaterial {

            }


            /**
            * A constraint that tries to keep the distance between two bodies constant.
            */
            class DistanceConstraint extends p2.DistanceConstraint {


                /**
                * A constraint that tries to keep the distance between two bodies constant.
                * 
                * @param world A reference to the P2 World.
                * @param bodyA First connected body.
                * @param bodyB Second connected body.
                * @param distance The distance to keep between the bodies.
                * @param localAnchorA The anchor point for bodyA, defined locally in bodyA frame. Defaults to [0,0].
                * @param localAnchorB The anchor point for bodyB, defined locally in bodyB frame. Defaults to [0,0].
                * @param maxForce Maximum force to apply. - Default: Number.MAX_VALUE
                */
                constructor(world: Phaser.Physics.P2, bodyA: Phaser.Physics.P2.Body, bodyB: Phaser.Physics.P2.Body, distance: number, maxForce: number);


                /**
                * Local reference to game.
                */
                game: Phaser.Game;

                /**
                * Local reference to P2 World.
                */
                world: Phaser.Physics.P2;

            }


            /**
            * Allow to access a list of created fixture (coming from Body#addPhaserPolygon)
            * which itself parse the input from PhysicsEditor with the custom phaser exporter.
            * You can access fixtures of a Body by a group index or even by providing a fixture Key.
            * You can set the fixture key and also the group index for a fixture in PhysicsEditor.
            * This gives you the power to create a complex body built of many fixtures and modify them
            * during runtime (to remove parts, set masks, categories & sensor properties)
            */
            class FixtureList {


                /**
                * Allow to access a list of created fixture (coming from Body#addPhaserPolygon)
                * which itself parse the input from PhysicsEditor with the custom phaser exporter.
                * You can access fixtures of a Body by a group index or even by providing a fixture Key.
                * You can set the fixture key and also the group index for a fixture in PhysicsEditor.
                * This gives you the power to create a complex body built of many fixtures and modify them
                * during runtime (to remove parts, set masks, categories & sensor properties)
                * 
                * @param list A list of fixtures (from Phaser.Physics.P2.Body#addPhaserPolygon)
                */
                constructor(list: any[]);


                /**
                * A helper to flatten arrays. This is very useful as the fixtures are nested from time to time due to the way P2 creates and splits polygons.
                * 
                * @param array The array to flatten. Notice: This will happen recursive not shallow.
                */
                flatten(array: any[]): any[];

                /**
                * Accessor to get either a list of specified fixtures by key or the whole fixture list
                * 
                * @param keys A list of fixture keys
                */
                getFixtures(keys: string): any[];

                /**
                * Accessor to get either a single fixture by its key.
                * 
                * @param key The key of the fixture.
                */
                getFixtureByKey(key: string): any[];

                /**
                * Accessor to get a group of fixtures by its group index.
                * 
                * @param groupID The group index.
                */
                getGroup(groupID: number): any[];
                init(): void;

                /**
                * Parser for the output of Phaser.Physics.P2.Body#addPhaserPolygon
                */
                parse(): void;

                /**
                * 
                * 
                * @param bit The bit to set as the collision group.
                * @param fixtureKey Only apply to the fixture with the given key.
                */
                setCategory(bit: number, fictureKey: string): void;

                /**
                * 
                * 
                * @param bit The bit to set as the collision mask
                * @param fixtureKey Only apply to the fixture with the given key
                */
                setMask(bit: number, fixtureKey: string): void;

                /**
                * 
                * 
                * @param material The contact material for a fixture
                * @param fixtureKey Only apply to the fixture with the given key
                */
                setMaterial(material: any, fixtureKey: string): void;

                /**
                * 
                * 
                * @param value sensor true or false
                * @param fixtureKey Only apply to the fixture with the given key
                */
                setSensor(value: boolean, fixtureKey: string): void;

            }


            /**
            * Connects two bodies at given offset points, letting them rotate relative to each other around this point.
            */
            class GearConstraint extends p2.GearConstraint {


                /**
                * Connects two bodies at given offset points, letting them rotate relative to each other around this point.
                * 
                * @param world A reference to the P2 World.
                * @param bodyA First connected body.
                * @param bodyB Second connected body.
                * @param angle The relative angle
                * @param ratio The gear ratio. - Default: 1
                */
                constructor(world: Phaser.Physics.P2, bodyA: Phaser.Physics.P2.Body, bodyB: Phaser.Physics.P2.Body, angle?: number, ratio?: number);


                /**
                * Local reference to game.
                */
                game: Phaser.Game;

                /**
                * Local reference to P2 World.
                */
                world: Phaser.Physics.P2;

            }


            /**
            * A InversePointProxy is an internal class that allows for direct getter/setter style property access to Arrays and TypedArrays but inverses the values on set.
            */
            class InversePointProxy {


                /**
                * A InversePointProxy is an internal class that allows for direct getter/setter style property access to Arrays and TypedArrays but inverses the values on set.
                * 
                * @param world A reference to the P2 World.
                * @param destination The object to bind to.
                */
                constructor(world: Phaser.Physics.P2, destination: any);


                /**
                * The x property of this InversePointProxy get and set in pixels.
                */
                x: number;

                /**
                * The y property of this InversePointProxy get and set in pixels.
                */
                y: number;

                /**
                * The x property of this InversePointProxy get and set in meters.
                */
                mx: number;

                /**
                * The y property of this InversePointProxy get and set in meters.
                */
                my: number;

            }


            /**
            * Locks the relative position between two bodies.
            */
            class LockConstraint extends p2.LockConstraint {


                /**
                * Locks the relative position between two bodies.
                * 
                * @param world A reference to the P2 World.
                * @param bodyA First connected body.
                * @param bodyB Second connected body.
                * @param offset The offset of bodyB in bodyA's frame. The value is an array with 2 elements matching x and y, i.e: [32, 32].
                * @param angle The angle of bodyB in bodyA's frame.
                * @param maxForce The maximum force that should be applied to constrain the bodies.
                */
                constructor(world: Phaser.Physics.P2, bodyA: Phaser.Physics.P2.Body, bodyB: Phaser.Physics.P2.Body, offset?: number[], angle?: number, maxForce?: number);


                /**
                * Local reference to game.
                */
                game: Phaser.Game;

                /**
                * Local reference to P2 World.
                */
                world: Phaser.Physics.P2;
            }


            /**
            * A P2 Material.
            * 
            * \o/ ~ "Because I'm a Material girl"
            */
            class Material extends p2.Material {


                /**
                * A P2 Material.
                * 
                * \o/ ~ "Because I'm a Material girl"
                * 
                * @param name The user defined name given to this Material.
                */
                constructor(name: string);


                /**
                * The user defined name given to this Material.
                */
                name: string;

            }


            /**
            * A PointProxy is an internal class that allows for direct getter/setter style property access to Arrays and TypedArrays.
            */
            class PointProxy {


                /**
                * A PointProxy is an internal class that allows for direct getter/setter style property access to Arrays and TypedArrays.
                * 
                * @param world A reference to the P2 World.
                * @param destination The object to bind to.
                */
                constructor(world: Phaser.Physics.P2, destination: any);


                /**
                * The x property of this PointProxy get and set in pixels.
                */
                x: number;

                /**
                * The y property of this PointProxy get and set in pixels.
                */
                y: number;

                /**
                * The x property of this PointProxy get and set in meters.
                */
                mx: number;

                /**
                * The x property of this PointProxy get and set in meters.
                */
                my: number;

            }


            /**
            * Connects two bodies at given offset points, letting them rotate relative to each other around this point.
            */
            class PrismaticConstraint extends p2.PrismaticConstraint {


                /**
                * Connects two bodies at given offset points, letting them rotate relative to each other around this point.
                * 
                * @param world A reference to the P2 World.
                * @param bodyA First connected body.
                * @param bodyB Second connected body.
                * @param lockRotation If set to false, bodyB will be free to rotate around its anchor point. - Default: true
                * @param anchorA Body A's anchor point, defined in its own local frame. The value is an array with 2 elements matching x and y, i.e: [32, 32].
                * @param anchorB Body A's anchor point, defined in its own local frame. The value is an array with 2 elements matching x and y, i.e: [32, 32].
                * @param axis An axis, defined in body A frame, that body B's anchor point may slide along. The value is an array with 2 elements matching x and y, i.e: [32, 32].
                * @param maxForce The maximum force that should be applied to constrain the bodies.
                */
                constructor(world: Phaser.Physics.P2, bodyA?: Phaser.Physics.P2.Body, bodyB?: Phaser.Physics.P2.Body, lockRotation?: boolean, anchorA?: number[], anchorB?: number[], axis?: number[], maxForce?: number);


                /**
                * Local reference to game.
                */
                game: Phaser.Game;

                /**
                * Local reference to P2 World.
                */
                world: Phaser.Physics.P2;

            }


            /**
            * Connects two bodies at given offset points, letting them rotate relative to each other around this point.
            * The pivot points are given in world (pixel) coordinates.
            */
            class RevoluteConstraint extends p2.RevoluteConstraint {


                /**
                * Connects two bodies at given offset points, letting them rotate relative to each other around this point.
                * The pivot points are given in world (pixel) coordinates.
                * 
                * @param world A reference to the P2 World.
                * @param bodyA First connected body.
                * @param pivotA The point relative to the center of mass of bodyA which bodyA is constrained to. The value is an array with 2 elements matching x and y, i.e: [32, 32].
                * @param bodyB Second connected body.
                * @param pivotB The point relative to the center of mass of bodyB which bodyB is constrained to. The value is an array with 2 elements matching x and y, i.e: [32, 32].
                * @param maxForce The maximum force that should be applied to constrain the bodies.
                * @param worldPivot A pivot point given in world coordinates. If specified, localPivotA and localPivotB are automatically computed from this value.
                */
                constructor(world: Phaser.Physics.P2, bodyA: Phaser.Physics.P2.Body, pivotA: number[], bodyB: Phaser.Physics.P2.Body, pivotB: number[], maxForce?: number);


                /**
                * Local reference to game.
                */
                game: Phaser.Game;

                /**
                * Local reference to P2 World.
                */
                world: Phaser.Physics.P2;

            }


            /**
            * Creates a linear spring, connecting two bodies. A spring can have a resting length, a stiffness and damping.
            */
            class Spring {


                /**
                * Creates a linear spring, connecting two bodies. A spring can have a resting length, a stiffness and damping.
                * 
                * @param world A reference to the P2 World.
                * @param bodyA First connected body.
                * @param bodyB Second connected body.
                * @param restLength Rest length of the spring. A number > 0. - Default: 1
                * @param stiffness Stiffness of the spring. A number >= 0. - Default: 100
                * @param damping Damping of the spring. A number >= 0. - Default: 1
                * @param worldA Where to hook the spring to body A in world coordinates. This value is an array with 2 elements matching x and y, i.e: [32, 32].
                * @param worldB Where to hook the spring to body B in world coordinates. This value is an array with 2 elements matching x and y, i.e: [32, 32].
                * @param localA Where to hook the spring to body A in local body coordinates. This value is an array with 2 elements matching x and y, i.e: [32, 32].
                * @param localB Where to hook the spring to body B in local body coordinates. This value is an array with 2 elements matching x and y, i.e: [32, 32].
                */
                constructor(world: Phaser.Physics.P2, bodyA: Phaser.Physics.P2.Body, bodyB: Phaser.Physics.P2.Body, restLength?: number, stiffness?: number, damping?: number, worldA?: number[], worldB?: number[], localA?: number[], localB?: number[]);


                /**
                * The actual p2 spring object.
                */
                data: p2.LinearSpring;

                /**
                * Local reference to game.
                */
                game: Phaser.Game;

                /**
                * Local reference to P2 World.
                */
                world: Phaser.Physics.P2;

            }
        }
    }


    /**
    * This is a base Plugin template to use for any Phaser plugin development.
    */
    class Plugin implements IStateCycle {


        /**
        * This is a base Plugin template to use for any Phaser plugin development.
        * 
        * @param game A reference to the currently running game.
        * @param parent The object that owns this plugin, usually Phaser.PluginManager.
        */
        constructor(game: Phaser.Game, parent: Phaser.PluginManager);


        /**
        * A Plugin with active=true has its preUpdate and update methods called by the parent, otherwise they are skipped.
        */
        active: boolean;

        /**
        * A reference to the currently running game.
        */
        game: Phaser.Game;

        /**
        * A flag to indicate if this plugin has a postRender method.
        */
        hasPostRender: boolean;

        /**
        * A flag to indicate if this plugin has a postUpdate method.
        */
        hasPostUpdate: boolean;

        /**
        * A flag to indicate if this plugin has a preUpdate method.
        */
        hasPreUpdate: boolean;

        /**
        * A flag to indicate if this plugin has a render method.
        */
        hasRender: boolean;

        /**
        * A flag to indicate if this plugin has an update method.
        */
        hasUpdate: boolean;

        /**
        * The parent of this plugin. If added to the PluginManager the parent will be set to that, otherwise it will be null.
        */
        parent: PIXI.DisplayObject;

        /**
        * A Plugin with visible=true has its render and postRender methods called by the parent, otherwise they are skipped.
        */
        visible: boolean;


        /**
        * Clear down this Plugin and null out references
        */
        destroy(): void;

        /**
        * Post-render is called after the Game Renderer and State.render have run.
        * It is only called if visible is set to true.
        */
        postRender(): void;

        /**
        * Pre-update is called at the very start of the update cycle, before any other subsystems have been updated (including Physics).
        * It is only called if active is set to true.
        */
        preUpdate(): void;

        /**
        * Render is called right after the Game Renderer completes, but before the State.render.
        * It is only called if visible is set to true.
        */
        render(): void;

        /**
        * Update is called after all the core subsystems (Input, Tweens, Sound, etc) and the State have updated, but before the render.
        * It is only called if active is set to true.
        */
        update(): void;

    }

    module Plugin {

        class SaveCPU extends Phaser.Plugin {

            renderOnFPS: number;
            renderOnPointerChange: boolean;
            forceRender(): void;
        }


        /**
        * AStar is a phaser pathfinding plugin based on an A* kind of algorythm
        * It works with the Phaser.Tilemap
        */
        class AStar extends Phaser.Plugin {

            static VERSION: string;
            static COST_ORTHOGONAL: number;
            static COST_DIAGONAL: number;
            static DISTANCE_MANHATTAN: string;
            static DISTANCE_EUCLIDIAN: string;


            /**
            * AStar is a phaser pathfinding plugin based on an A* kind of algorythm
            * It works with the Phaser.Tilemap
            * 
            * @param parent The object that owns this plugin, usually Phaser.PluginManager.
            */
            constructor(parent: PIXI.DisplayObject);


            /**
            * The parent of this plugin. If added to the PluginManager the parent will be set to that, otherwise it will be null.
            */
            parent: PIXI.DisplayObject;
            version: string;


            /**
            * Find a path between to tiles coordinates
            * 
            * @param startPoint The start point x, y in tiles coordinates to search a path.
            * @param goalPoint The goal point x, y in tiles coordinates that you trying to reach.
            * @return The Phaser.Plugin.AStar.AStarPath that results
            */
            findPath(startPoint: Phaser.Point, goalPoint: Phaser.Point): Phaser.Plugin.AStar.AStarPath;
            isWalkable(x: number, y: number): boolean;

            /**
            * Sets the Phaser.Tilemap used to searchPath into.
            * 
            * @param map the Phaser.Tilemap used to searchPath into. It must have a tileset with tile porperties to know if tiles are walkable or not.
            * @param layerName The name of the layer that handle tiles.
            * @param tilesetName The name of the tileset that have walkable properties.
            * @return The Phaser.Plugin.AStar itself.
            */
            setAStarMap(map: Phaser.Tilemap, layerName: string, tilesetName: string): Phaser.Plugin.AStar;

        }

        module AStar {


            /**
            * AStarNode is an object that stores AStar value. Each tile have an AStarNode in their properties
            */
            class AStarNode {


                /**
                * AStarNode is an object that stores AStar value. Each tile have an AStarNode in their properties
                * 
                * @param x The x coordinate of the tile.
                * @param y The y coordinate of the tile.
                * @param isWalkable Is this tile is walkable?
                */
                constructor(x: number, y: number, isWalkable: boolean);


                /**
                * The x coordinate of the tile.
                */
                x: number;

                /**
                * The y coordinate of the tile.
                */
                y: number;

                /**
                * The total travel cost from the start point. Sum of COST_ORTHOGONAL and COST_DIAGONAL
                */
                g: number;

                /**
                * The remaing distance as the crow flies between this node and the goal.
                */
                h: number;

                /**
                * The weight. Sum of g + h.
                */
                f: number;

                /**
                * Where do we come from? It's an AStarNode reference needed to reconstruct a path backwards (from goal to start point)
                */
                parent: Phaser.Plugin.AStar.AStarNode;

                /**
                * The cost to travel to this node, COST_ORTHOGONAL or COST_DIAGONAL
                */
                travelCost: number;

                /**
                * Is this node is walkable?
                */
                walkable: boolean;

            }

            interface AStarNodeArray {
                x: number;
                y: number;
            }


            /**
            * AStarPath is an object that stores a searchPath result.
            */
            class AStarPath {


                /**
                * AStarPath is an object that stores a searchPath result.
                * 
                * @param nodes An array of nodes coordinates sorted backward from goal to start point.
                * @param start The start AStarNode used for the searchPath.
                * @param goal The goal AStarNode used for the searchPath.
                */
                constructor(nodes?: AStarNodeArray[], start?: Phaser.Plugin.AStar.AStarNode, goal?: Phaser.Plugin.AStar.AStarNode);


                /**
                * Array of AstarNodes x, y coordiantes that are the path solution from goal to start point.
                */
                nodes: AStarNodeArray[];

                /**
                * Reference to the start point used by findPath.
                */
                start: Phaser.Plugin.AStar.AStarNode;

                /**
                * Reference to the goal point used by findPath.
                */
                goal: Phaser.Plugin.AStar.AStarNode;

                /**
                * Array of AStarNodes that the findPath algorythm has visited. Used for debug only.
                */
                visited: Phaser.Plugin.AStar.AStarNode[];

            }

        }


        /**
        * A collection of methods useful for manipulating and comparing colors.
        */
        class ColorHarmony extends Phaser.Plugin {


            /**
            * Returns an Analogous Color Harmony for the given color.
            * <p>An Analogous harmony are hues adjacent to each other on the color wheel</p>
            * <p>Values returned in 0xAARRGGBB format with Alpha set to 255.</p>
            * 
            * @param color The color to base the harmony on.
            * @param threshold Control how adjacent the colors will be (default +- 30 degrees)
            * @return Object containing 3 properties: color1 (the original color), color2 (the warmer analogous color) and color3 (the colder analogous color)
            */
            getAnalogousHarmony(color: number, threshold?: number): any;

            /**
            * Returns a Complementary Color Harmony for the given color.
            * <p>A complementary hue is one directly opposite the color given on the color wheel</p>
            * <p>Value returned in 0xAARRGGBB format with Alpha set to 255.</p>
            * 
            * @param color The color to base the harmony on.
            * @return 0xAARRGGBB format color value.
            */
            getComplementHarmony(color: number): number;

            /**
            * Returns an Split Complement Color Harmony for the given color.
            * <p>A Split Complement harmony are the two hues on either side of the color's Complement</p>
            * <p>Values returned in 0xAARRGGBB format with Alpha set to 255.</p>
            * 
            * @param color The color to base the harmony on
            * @param threshold Control how adjacent the colors will be to the Complement (default +- 30 degrees)
            * @return An object containing 3 properties: color1 (the original color), color2 (the warmer analogous color) and color3 (the colder analogous color)
            */
            getSplitComplementHarmony(color: number, threshold: number): any;

            /**
            * Returns a Triadic Color Harmony for the given color.
            * <p>A Triadic harmony are 3 hues equidistant from each other on the color wheel</p>
            * <p>Values returned in 0xAARRGGBB format with Alpha set to 255.</p>
            * 
            * @param color The color to base the harmony on.
            * @return An Object containing 3 properties: color1 (the original color), color2 and color3 (the equidistant colors)
            */
            getTriadicHarmony(color: number): any;

        }


        /**
        * Phaser - Display - CSS3Filters
        * 
        * Allows for easy addition and modification of CSS3 Filters on DOM objects (typically the Game.Stage.canvas).
        */
        class CSS3Filters extends Phaser.Plugin {


            /**
            * Phaser - Display - CSS3Filters
            * 
            * Allows for easy addition and modification of CSS3 Filters on DOM objects (typically the Game.Stage.canvas).
            */
            constructor(parent: PIXI.DisplayObject);

            blur: number;
            brightness: number;
            contrast: number;
            grayscale: number;
            hueRotate: number;
            invert: number;
            opacity: number;
            saturate: number;
            sepia: number;

        }


        /**
        * Creates an object that is placed within a layer of a Phaser.Tilemap and can be moved around and rotated using the direction commands.
        */
        class TilemapWalker extends Phaser.Plugin {


            /**
            * Creates an object that is placed within a layer of a Phaser.Tilemap and can be moved around and rotated using the direction commands.
            * 
            * @param game Game reference to the currently running game.
            * @param map A reference to the Tilemap this TilemapWalker belongs to.
            * @param layer The layer to operate on. If not given will default to this.currentLayer.
            * @param x X position of the top left of the area to copy (given in tiles, not pixels)
            * @param y Y position of the top left of the area to copy (given in tiles, not pixels)
            */
            constructor(game: Phaser.Game, map: Phaser.Tilemap, layer?: any, x?: number, y?: number);


            /**
            * Does the TilemapWalker collide with the tiles in the map set for collision? If so it cannot move through them.
            * Default: true
            */
            collides: boolean;

            /**
            * A reference to the currently running Game.
            */
            game: Phaser.Game;

            /**
            * An array containing a history of movements through the map.
            */
            history: boolean;

            /**
            * The direction the location marker is facing. You can rotate it using the turn and face methods.
            */
            facing: number;

            /**
            * A reference to the Tilemap this TilemapWalker belongs to.
            */
            map: Phaser.Tilemap;

            /**
            * The current marker location. You can move the marker with the movement methods.
            */
            location: Phaser.Point;

            /**
            * The current layer of the location marker.
            */
            locationLayer: number;

            checkTile(x: number, y: number): boolean;
            getTileFromLocation(x: number, y: number): Phaser.Tile;
            getTiles(width: number, height: number, center?: boolean): any[];
            getTileBehind(distance?: number): Phaser.Tile;
            getTileBehindLeft(distance?: number): Phaser.Tile;
            getTileBehindRight(distance?: number): Phaser.Tile;
            getTileAhead(distance?: number): Phaser.Tile;
            getTileAheadLeft(distance?: number): Phaser.Tile;
            getTileAheadRight(distance?: number): Phaser.Tile;
            getTileLeft(distance: number): Phaser.Tile;
            getTileRight(distance: number): Phaser.Tile;
            moveForward(): boolean;
            moveBackward(): boolean;
            moveLeft(): boolean;
            moveRight(): boolean;
            putTile(index: number): void;
            setLocation(x: number, y: number, layer?: any): boolean;
            turnLeft(): void;
            turnRight(): void;
            updateLocation(x: number, y: number): boolean;

        }


        /**
        * A Sample Plugin demonstrating how to hook into the Phaser plugin system.
        */
        class SamplePlugin extends Phaser.Plugin {


            /**
            * A Sample Plugin demonstrating how to hook into the Phaser plugin system.
            */
            constructor(game: Phaser.Game, parent: PIXI.DisplayObject);


            /**
            * Add a Sprite reference to this Plugin.
            * All this plugin does is move the Sprite across the screen slowly.
            */
            addSprite(sprite: Phaser.Sprite): void;

            /**
            * This is run when the plugins update during the core game loop.
            */
            update(): void;

        }


        /**
        * A Virtual Joystick
        */
        class VirtualJoystick extends Phaser.Plugin {


            /**
            * A Virtual Joystick
            */
            constructor(game: Phaser.Game, parent: any);

            angle: number;
            base: Phaser.Sprite;
            baseBMD: Phaser.BitmapData;
            baseCircle: Phaser.Circle;
            deltaX: number;
            deltaY: number;
            distance: number;
            force: number;
            isDragging: boolean;
            limit: number;
            limitPoint: Phaser.Point;
            location: Phaser.Point;
            nub: Phaser.Sprite;
            nubBMD: Phaser.BitmapData;
            speed: number;
            x: number;
            y: number;

            init(x: number, y: number, diameter?: number, limit?: number): void;
            move(pointer: Phaser.Pointer, x: number, y: number): void;
            render(): void;

            /**
            * Given the speed calculate the velocity and return it as a Point object, or set it to the given point object.
      