/**
* @author       Richard Davey <rich@photonstorm.com>
* @copyright    2016 Photon Storm Ltd.
* @license      {@link https://github.com/photonstorm/phaser/blob/master/license.txt|MIT License}
*/

/**
* The Physics Body is typically linked to a single Sprite and defines properties that determine how the physics body is simulated.
* These properties affect how the body reacts to forces, what forces it generates on itself (to simulate friction), and how it reacts to collisions in the scene.
* In most cases, the properties are used to simulate physical effects. Each body also has its own property values that determine exactly how it reacts to forces and collisions in the scene.
* By default a single Rectangle shape is added to the Body that matches the dimensions of the parent Sprite. See addShape, removeShape, clearShapes to add extra shapes around the Body.
* Note: When bound to a Sprite to avoid single-pixel jitters on mobile devices we strongly recommend using Sprite sizes that are even on both axis, i.e. 128x128 not 127x127.
* Note: When a game object is given a P2 body it has its anchor x/y set to 0.5, so it becomes centered.
*
* @class Phaser.Physics.P2.Body
* @constructor
* @param {Phaser.Game} game - Game reference to the currently running game.
* @param {Phaser.Sprite} [sprite] - The Sprite object this physics body belongs to.
* @param {number} [x=0] - The x coordinate of this Body.
* @param {number} [y=0] - The y coordinate of this Body.
* @param {number} [mass=1] - The default mass of this Body (0 = static).
*/
Phaser.Physics.P2.Body = function (game, sprite, x, y, mass) {

    sprite = sprite || null;
    x = x || 0;
    y = y || 0;
    if (mass === undefined) { mass = 1; }

    /**
    * @property {Phaser.Game} game - Local reference to game.
    */
    this.game = game;

    /**
    * @property {Phaser.Physics.P2} world - Local reference to the P2 World.
    */
    this.world = game.physics.p2;

    /**
    * @property {Phaser.Sprite} sprite - Reference to the parent Sprite.
    */
    this.sprite = sprite;

    /**
    * @property {number} type - The type of physics system this body belongs to.
    */
    this.type = Phaser.Physics.P2JS;

    /**
    * @property {Phaser.Point} offset - The offset of the Physics Body from the Sprite x/y position.
    */
    this.offset = new Phaser.Point();

    /**
    * @property {p2.Body} data - The p2 Body data.
    * @protected
    */
    this.data = new p2.Body({ position: [ this.world.pxmi(x), this.world.pxmi(y) ], mass: mass });

    this.data.parent = this;

    /**
    * @property {Phaser.Physics.P2.InversePointProxy} velocity - The velocity of the body. Set velocity.x to a negative value to move to the left, position to the right. velocity.y negative values move up, positive move down.
    */
    this.velocity = new Phaser.Physics.P2.InversePointProxy(this.world, this.data.velocity);

    /**
    * @property {Phaser.Physics.P2.InversePointProxy} force - The force applied to the body.
    */
    this.force = new Phaser.Physics.P2.InversePointProxy(this.world, this.data.force);

    /**
    * @property {Phaser.Point} gravity - A locally applied gravity force to the Body. Applied directly before the world step. NOTE: Not currently implemented.
    */
    this.gravity = new Phaser.Point();

    /**
    * Dispatched when a first contact is created between shapes in two bodies.
    * This event is fired during the step, so collision has already taken place.
    *
    * The event will be sent 5 arguments in this order:
    *
    * The Phaser.Physics.P2.Body it is in contact with. *This might be null* if the Body was created directly in the p2 world.
    * The p2.Body this Body is in contact with.
    * The Shape from this body that caused the contact.
    * The Shape from the contact body.
    * The Contact Equation data array.
    *
    * @property {Phaser.Signal} onBeginContact
    */
    this.onBeginContact = new Phaser.Signal();

    /**
    * Dispatched when contact ends between shapes in two bodies.
    * This event is fired during the step, so collision has already taken place.
    *
    * The event will be sent 4 arguments in this order:
    *
    * The Phaser.Physics.P2.Body it is in contact with. *This might be null* if the Body was created directly in the p2 world.
    * The p2.Body this Body has ended contact with.
    * The Shape from this body that caused the original contact.
    * The Shape from the contact body.
    *
    * @property {Phaser.Signal} onEndContact
    */
    this.onEndContact = new Phaser.Signal();

    /**
    * @property {array} collidesWith - Array of CollisionGroups that this Bodies shapes collide with.
    */
    this.collidesWith = [];

    /**
    * @property {boolean} removeNextStep - To avoid deleting this body during a physics step, and causing all kinds of problems, set removeNextStep to true to have it removed in the next preUpdate.
    */
    this.removeNextStep = false;

    /**
    * @property {Phaser.Physics.P2.BodyDebug} debugBody - Reference to the debug body.
    */
    this.debugBody = null;

    /**
    * @property {boolean} dirty - Internally used by Sprite.x/y
    */
    this.dirty = false;

    /**
    * @property {boolean} _collideWorldBounds - Internal var that determines if this Body collides with the world bounds or not.
    * @private
    */
    this._collideWorldBounds = true;

    /**
    * @property {object} _bodyCallbacks - Array of Body callbacks.
    * @private
    */
    this._bodyCallbacks = {};

    /**
    * @property {object} _bodyCallbackContext - Array of Body callback contexts.
    * @private
    */
    this._bodyCallbackContext = {};

    /**
    * @property {object} _groupCallbacks - Array of Group callbacks.
    * @private
    */
    this._groupCallbacks = {};

    /**
    * @property {object} _bodyCallbackContext - Array of Grouo callback contexts.
    * @private
    */
    this._groupCallbackContext = {};

    /**
    * @property {boolean} _reset - Internal var.
    * @private
    */
    this._reset = false;

    //  Set-up the default shape
    if (sprite)
    {
        this.setRectangleFromSprite(sprite);

        if (sprite.exists)
        {
            this.game.physics.p2.addBody(this);
        }
    }

};

Phaser.Physics.P2.Body.prototype = {

    /**
    * Sets a callback to be fired any time a shape in this Body impacts with a shape in the given Body. The impact test is performed against body.id values.
    * The callback will be sent 4 parameters: This body, the body that impacted, the Shape in this body and the shape in the impacting body.
    * Note that the impact event happens after collision resolution, so it cannot be used to prevent a collision from happening.
    * It also happens mid-step. So do not destroy a Body during this callback, instead set safeDestroy to true so it will be killed on the next preUpdate.
    *
    * @method Phaser.Physics.P2.Body#createBodyCallback
    * @param {Phaser.Sprite|Phaser.TileSprite|Phaser.Physics.P2.Body|p2.Body} object - The object to send impact events for.
    * @param {function} callback - The callback to fire on impact. Set to null to clear a previously set callback.
    * @param {object} callbackContext - The context under which the callback will fire.
    */
    createBodyCallback: function (object, callback, callbackContext) {

        var id = -1;

        if (object['id'])
        {
            id = object.id;
        }
        else if (object['body'])
        {
            id = object.body.id;
        }

        if (id > -1)
        {
            if (callback === null)
            {
                delete (this._bodyCallbacks[id]);
                delete (this._bodyCallbackContext[id]);
            }
            else
            {
                this._bodyCallbacks[id] = callback;
                this._bodyCallbackContext[id] = callbackContext;
            }
        }

    },

    /**
    * Sets a callback to be fired any time this Body impacts with the given Group. The impact test is performed against shape.collisionGroup values.
    * The callback will be sent 4 parameters: This body, the body that impacted, the Shape in this body and the shape in the impacting body.
    * This callback will only fire if this Body has been assigned a collision group.
    * Note that the impact event happens after collision resolution, so it cannot be used to prevent a collision from happening.
    * It also happens mid-step. So do not destroy a Body during this callback, instead set safeDestroy to true so it will be killed on the next preUpdate.
    *
    * @method Phaser.Physics.P2.Body#createGroupCallback
    * @param {Phaser.Physics.CollisionGroup} group - The Group to send impact events for.
    * @param {function} callback - The callback to fire on impact. Set to null to clear a previously set callback.
    * @param {object} callbackContext - The context under which the callback will fire.
    */
    createGroupCallback: function (group, callback, callbackContext) {

        if (callback === null)
        {
            delete (this._groupCallbacks[group.mask]);
            delete (this._groupCallbackContext[group.mask]);
        }
        else
        {
            this._groupCallbacks[group.mask] = callback;
            this._groupCallbackContext[group.mask] = callbackContext;
        }

    },

    /**
    * Gets the collision bitmask from the groups this body collides with.
    *
    * @method Phaser.Physics.P2.Body#getCollisionMask
    * @return {number} The bitmask.
    */
    getCollisionMask: function () {

        var mask = 0;

        if (this._collideWorldBounds)
        {
            mask = this.game.physics.p2.boundsCollisionGroup.mask;
        }

        for (var i = 0; i < this.collidesWith.length; i++)
        {
            mask = mask | this.collidesWith[i].mask;
        }

        return mask;

    },

    /**
    * Updates the collisionMask.
    *
    * @method Phaser.Physics.P2.Body#updateCollisionMask
    * @param {p2.Shape} [shape] - An optional Shape. If not provided the collision group will be added to all Shapes in this Body.
    */
    updateCollisionMask: function (shape) {

        var mask = this.getCollisionMask();

        if (shape === undefined)
        {
            for (var i = this.data.shapes.length - 1; i >= 0; i--)
            {
                this.data.shapes[i].collisionMask = mask;
            }
        }
        else
        {
            shape.collisionMask = mask;
        }

    },

    /**
    * Sets the given CollisionGroup to be the collision group for all shapes in this Body, unless a shape is specified.
    * This also resets the collisionMask.
    *
    * @method Phaser.Physics.P2.Body#setCollisionGroup
    * @param {Phaser.Physics.CollisionGroup} group - The Collision Group that this Bodies shapes will use.
    * @param {p2.Shape} [shape] - An optional Shape. If not provided the collision group will be added to all Shapes in this Body.
    */
    setCollisionGroup: function (group, shape) {

        var mask = this.getCollisionMask();

        if (shape === undefined)
        {
            for (var i = this.data.shapes.length - 1; i >= 0; i--)
            {
                this.data.shapes[i].collisionGroup = group.mask;
                this.data.shapes[i].collisionMask = mask;
            }
        }
        else
        {
            shape.collisionGroup = group.mask;
            shape.collisionMask = mask;
        }

    },

    /**
    * Clears the collision data from the shapes in this Body. Optionally clears Group and/or Mask.
    *
    * @method Phaser.Physics.P2.Body#clearCollision
    * @param {boolean} [clearGroup=true] - Clear the collisionGroup value from the shape/s?
    * @param {boolean} [clearMask=true] - Clear the collisionMask value from the shape/s?
    * @param {p2.Shape} [shape] - An optional Shape. If not provided the collision data will be cleared from all Shapes in this Body.
    */
    clearCollision: function (clearGroup, clearMask, shape) {

        if (clearGroup === undefined) { clearGroup = true; }
        if (clearMask === undefined) { clearMask = true; }

        if (shape === undefined)
        {
            for (var i = this.data.shapes.length - 1; i >= 0; i--)
            {
                if (clearGroup)
                {
                    this.data.shapes[i].collisionGroup = null;
                }

                if (clearMask)
                {
                    this.data.shapes[i].collisionMask = null;
                }
            }
        }
        else
        {
            if (clearGroup)
            {
                shape.collisionGroup = null;
            }

            if (clearMask)
            {
                shape.collisionMask = null;
            }
        }

        if (clearGroup)
        {
            this.collidesWith.length = 0;
        }

    },

    /**
    * Removes the given CollisionGroup, or array of CollisionGroups, from the list of groups that this body will collide with and updates the collision masks.
    *
    * @method Phaser.Physics.P2.Body#removeCollisionGroup
    * @param {Phaser.Physics.CollisionGroup|array} group - The Collision Group or Array of Collision Groups that this Bodies shapes should not collide with anymore.
    * @param {boolean} [clearCallback=true] - Clear the callback that will be triggered when this Body impacts with the given Group?
    * @param {p2.Shape} [shape] - An optional Shape. If not provided the updated collision mask will be added to all Shapes in this Body.
    */
    removeCollisionGroup: function (group, clearCallback, shape) {

        if (clearCallback === undefined) { clearCallback = true; }

        var index;

        if (Array.isArray(group))
        {
            for (var i = 0; i < group.length; i++)
            {
                index = this.collidesWith.indexOf(group[i]);

                if (index > -1)
                {
                    this.collidesWith.splice(index, 1);

                    if (clearCallback)
                    {
                        delete (this._groupCallbacks[group.mask]);
                        delete (this._groupCallbackContext[group.mask]);
                    }
                }
            }
        }
        else
        {
            index = this.collidesWith.indexOf(group);

            if (index > -1)
            {
                this.collidesWith.splice(index, 1);

                if (clearCallback)
                {
                    delete (this._groupCallbacks[group.mask]);
                    delete (this._groupCallbackContext[group.mask]);
                }
            }
        }

        var mask = this.getCollisionMask();

        if (shape === undefined)
        {
            for (var i = this.data.shapes.length - 1; i >= 0; i--)
            {
                this.data.shapes[i].collisionMask = mask;
            }
        }
        else
        {
            shape.collisionMask = mask;
        }

    },

    /**
    * Adds the given CollisionGroup, or array of CollisionGroups, to the list of groups that this body will collide with and updates the collision masks.
    *
    * @method Phaser.Physics.P2.Body#collides
    * @param {Phaser.Physics.CollisionGroup|array} group - The Collision Group or Array of Collision Groups that this Bodies shapes will collide with.
    * @param {function} [callback] - Optional callback that will be triggered when this Body impacts with the given Group.
    * @param {object} [callbackContext] - The context under which the callback will be called.
    * @param {p2.Shape} [shape] - An optional Shape. If not provided the collision mask will be added to all Shapes in this Body.
    */
    collides: function (group, callback, callbackContext, shape) {

        if (Array.isArray(group))
        {
            for (var i = 0; i < group.length; i++)
            {
                if (this.collidesWith.indexOf(group[i]) === -1)
                {
                    this.collidesWith.push(group[i]);

                    if (callback)
                    {
                        this.createGroupCallback(group[i], callback, callbackContext);
                    }
                }
            }
        }
        else
        {
            if (this.collidesWith.indexOf(group) === -1)
            {
                this.collidesWith.push(group);

                if (callback)
                {
                    this.createGroupCallback(group, callback, callbackContext);
                }
            }
        }

        var mask = this.getCollisionMask();

        if (shape === undefined)
        {
            for (var i = this.data.shapes.length - 1; i >= 0; i--)
            {
                this.data.shapes[i].collisionMask = mask;
            }
        }
        else
        {
            shape.collisionMask = mask;
        }

    },

    /**
    * Moves the shape offsets so their center of mass becomes the body center of mass.
    *
    * @method Phaser.Physics.P2.Body#adjustCenterOfMass
    */
    adjustCenterOfMass: function () {

        this.data.adjustCenterOfMass();
        this.shapeChanged();

    },

    /**
    * Gets the velocity of a point in the body.
    *
    * @method Phaser.Physics.P2.Body#getVelocityAtPoint
    * @param {Array} result - A vector to store the result in.
    * @param {Array} relativePoint - A world oriented vector, indicating the position of the point to get the velocity from.
    * @return {Array} The result vector.
    */
    getVelocityAtPoint: function (result, relativePoint) {

        return this.data.getVelocityAtPoint(result, relativePoint);

    },

    /**
    * Apply damping, see http://code.google.com/p/bullet/issues/detail?id=74 for details.
    *
    * @method Phaser.Physics.P2.Body#applyDamping
    * @param {number} dt - Current time step.
    */
    applyDamping: function (dt) {

        this.data.applyDamping(dt);

    },

    /**
    * Apply impulse to a point relative to the body.
    * This could for example be a point on the Body surface. An impulse is a force added to a body during a short
    * period of time (impulse = force * time). Impulses will be added to Body.velocity and Body.angularVelocity.
    *
    * @method Phaser.Physics.P2.Body#applyImpulse
    * @param {Float32Array|Array} impulse - The impulse vector to add, oriented in world space.
    * @param {number} worldX - A point relative to the body in world space. If not given, it is set to zero and all of the impulse will be exerted on the center of mass.
    * @param {number} worldY - A point relative to the body in world space. If not given, it is set to zero and all of the impulse will be exerted on the center of mass.
    */
    applyImpulse: function (impulse, worldX, worldY) {

        this.data.applyImpulse(impulse, [this.world.pxmi(worldX), this.world.pxmi(worldY)]);

    },

    /**
    * Apply impulse to a point local to the body.
    *
    * This could for example be a point on the Body surface. An impulse is a force added to a body during a short
    * period of time (impulse = force * time). Impulses will be added to Body.velocity and Body.angularVelocity.
    *
    * @method Phaser.Physics.P2.Body#applyImpulseLocal
    * @param {Float32Array|Array} impulse - The impulse vector to add, oriented in local space.
    * @param {number} localX - A local point on the body.
    * @param {number} localY - A local point on the body.
    */
    applyImpulseLocal: function (impulse, localX, localY) {

        this.data.applyImpulseLocal(impulse, [this.world.pxmi(localX), this.world.pxmi(localY)]);

    },

    /**
    * Apply force to a world point.
    *
    * This could for example be a point on the RigidBody surface. Applying force
    * this way will add to Body.force and Body.angularForce.
    *
    * @method Phaser.Physics.P2.Body#applyForce
    * @param {Float32Array|Array} force - The force vector to add.
    * @param {number} worldX - The world x point to apply the force on.
    * @param {number} worldY - The world y point to apply the force on.
    */
    applyForce: function (force, worldX, worldY) {

        this.data.applyForce(force, [this.world.pxmi(worldX), this.world.pxmi(worldY)]);

    },

    /**
    * Sets the force on the body to zero.
    *
    * @method Phaser.Physics.P2.Body#setZeroForce
    */
    setZeroForce: function () {

        this.data.setZeroForce();

    },

    /**
    * If this Body is dynamic then this will zero its angular velocity.
    *
    * @method Phaser.Physics.P2.Body#setZeroRotation
    */
    setZeroRotation: function () {

        this.data.angularVelocity = 0;

    },

    /**
    * If this Body is dynamic then this will zero its velocity on both axis.
    *
    * @method Phaser.Physics.P2.Body#setZeroVelocity
    */
    setZeroVelocity: function () {

        this.data.velocity[0] = 0;
        this.data.velocity[1] = 0;

    },

    /**
    * Sets the Body damping and angularDamping to zero.
    *
    * @method Phaser.Physics.P2.Body#setZeroDamping
    */
    setZeroDamping: function () {

        this.data.damping = 0;
        this.data.angularDamping = 0;

    },

    /**
    * Transform a world point to local body frame.
    *
    * @method Phaser.Physics.P2.Body#toLocalFrame
    * @param {Float32Array|Array} out - The vector to store the result in.
    * @param {Float32Array|Array} worldPoint - The input world vector.
    */
    toLocalFrame: function (out, worldPoint) {

        return this.data.toLocalFrame(out, worldPoint);

    },

    /**
    * Transform a local point to world frame.
    *
    * @method Phaser.Physics.P2.Body#toWorldFrame
    * @param {Array} out - The vector to store the result in.
    * @param {Array} localPoint - The input local vector.
    */
    toWorldFrame: function (out, localPoint) {

        return this.data.toWorldFrame(out, localPoint);

    },

    /**
    * This will rotate the Body by the given speed to the left (counter-clockwise).
    *
    * @method Phaser.Physics.P2.Body#rotateLeft
    * @param {number} speed - The speed at which it should rotate.
    */
    rotateLeft: function (speed) {

        this.data.angularVelocity = this.world.pxm(-speed);

    },

    /**
    * This will rotate the Body by the given speed to the left (clockwise).
    *
    * @method Phaser.Physics.P2.Body#rotateRight
    * @param {number} speed - The speed at which it should rotate.
    */
    rotateRight: function (speed) {

        this.data.angularVelocity = this.world.pxm(speed);

    },

    /**
    * Moves the Body forwards based on its current angle and the given speed.
    * The speed is represented in pixels per second. So a value of 100 would move 100 pixels in 1 second (1000ms).
    *
    * @method Phaser.Physics.P2.Body#moveForward
    * @param {number} speed - The speed at which it should move forwards.
    */
    moveForward: function (speed) {

        var magnitude = this.world.pxmi(-speed);
        var angle = this.data.angle + Math.PI / 2;

        this.data.velocity[0] = magnitude * Math.cos(angle);
        this.data.velocity[1] = magnitude * Math.sin(angle);

    },

    /**
    * Moves the Body backwards based on its current angle and the given speed.
    * The speed is represented in pixels per second. So a value of 100 would move 100 pixels in 1 second (1000ms).
    *
    * @method Phaser.Physics.P2.Body#moveBackward
    * @param {number} speed - The speed at which it should move backwards.
    */
    moveBackward: function (speed) {

        var magnitude = this.world.pxmi(-speed);
        var angle = this.data.angle + Math.PI / 2;

        this.data.velocity[0] = -(magnitude * Math.cos(angle));
        this.data.velocity[1] = -(magnitude * Math.sin(angle));

    },

    /**
    * Applies a force to the Body that causes it to 'thrust' forwards, based on its current angle and the given speed.
    * The speed is represented in pixels per second. So a value of 100 would move 100 pixels in 1 second (1000ms).
    *
    * @method Phaser.Physics.P2.Body#thrust
    * @param {number} speed - The speed at which it should thrust.
    */
    thrust: function (speed) {

        var magnitude = this.world.pxmi(-speed);
        var angle = this.data.angle + Math.PI / 2;

        this.data.force[0] += magnitude * Math.cos(angle);
        this.data.force[1] += magnitude * Math.sin(angle);

    },

    /**
    * Applies a force to the Body that causes it to 'thrust' to the left, based on its current angle and the given speed.
    * The speed is represented in pixels per second. So a value of 100 would move 100 pixels in 1 second (1000ms).
    *
    * @method Phaser.Physics.P2.Body#thrustLeft
    * @param {number} speed - The speed at which it should move to the left.
    */
    thrustLeft: function (speed) {

        var magnitude = this.world.pxmi(-speed);
        var angle = this.data.angle;

        this.data.force[0] += magnitude * Math.cos(angle);
        this.data.force[1] += magnitude * Math.sin(angle);

    },

    /**
    * Applies a force to the Body that causes it to 'thrust' to the right, based on its current angle and the given speed.
    * The speed is represented in pixels per second. So a value of 100 would move 100 pixels in 1 second (1000ms).
    *
    * @method Phaser.Physics.P2.Body#thrustRight
    * @param {number} speed - The speed at which it should move to the right.
    */
    thrustRight: function (speed) {

        var magnitude = this.world.pxmi(-speed);
        var angle = this.data.angle;

        this.data.force[0] -= magnitude * Math.cos(angle);
        this.data.force[1] -= magnitude * Math.sin(angle);

    },

    /**
    * Applies a force to the Body that causes it to 'thrust' backwards (in reverse), based on its current angle and the given speed.
    * The speed is represented in pixels per second. So a value of 100 would move 100 pixels in 1 second (1000ms).
    *
    * @method Phaser.Physics.P2.Body#reverse
    * @param {number} speed - The speed at which it should reverse.
    */
    reverse: function (speed) {

        var magnitude = this.world.pxmi(-speed);
        var angle = this.data.angle + Math.PI / 2;

        this.data.force[0] -= magnitude * Math.cos(angle);
        this.data.force[1] -= magnitude * Math.sin(angle);

    },

    /**
    * If this Body is dynamic then this will move it to the left by setting its x velocity to the given speed.
    * The speed is represented in pixels per second. So a value of 100 would move 100 pixels in 1 second (1000ms).
    *
    * @method Phaser.Physics.P2.Body#moveLeft
    * @param {number} speed - The speed at which it should move to the left, in pixels per second.
    */
    moveLeft: function (speed) {

        this.data.velocity[0] = this.world.pxmi(-speed);

    },

    /**
    * If this Body is dynamic then this will move it to the right by setting its x velocity to the given speed.
    * The speed is represented in pixels per second. So a value of 100 would move 100 pixels in 1 second (1000ms).
    *
    * @method Phaser.Physics.P2.Body#moveRight
    * @param {number} speed - The speed at which it should move to the right, in pixels per second.
    */
    moveRight: function (speed) {

        this.data.velocity[0] = this.world.pxmi(speed);

    },

    /**
    * If this Body is dynamic then this will move it up by setting its y velocity to the given speed.
    * The speed is represented in pixels per second. So a value of 100 would move 100 pixels in 1 second (1000ms).
    *
    * @method Phaser.Physics.P2.Body#moveUp
    * @param {number} speed - The speed at which it should move up, in pixels per second.
    */
    moveUp: function (speed) {

        this.data.velocity[1] = this.world.pxmi(-speed);

    },

    /**
    * If this Body is dynamic then this will move it down by setting its y velocity to the given speed.
    * The speed is represented in pixels per second. So a value of 100 would move 100 pixels in 1 second (1000ms).
    *
    * @method Phaser.Physics.P2.Body#moveDown
    * @param {number} speed - The speed at which it should move down, in pixels per second.
    */
    moveDown: function (speed) {

        this.data.velocity[1] = this.world.pxmi(speed);

    },

    /**
    * Internal method. This is called directly before the sprites are sent to the renderer and after the update function has finished.
    *
    * @method Phaser.Physics.P2.Body#preUpdate
    * @protected
    */
    preUpdate: function () {

        this.dirty = true;

        if (this.removeNextStep)
        {
            this.removeFromWorld();
            this.removeNextStep = false;
        }

    },

    /**
    * Internal method. This is called directly before the sprites are sent to the renderer and after the update function has finished.
    *
    * @method Phaser.Physics.P2.Body#postUpdate
    * @protected
    */
    postUpdate: function () {

        this.sprite.x = this.world.mpxi(this.data.position[0]) + this.offset.x;
        this.sprite.y = this.world.mpxi(this.data.position[1]) + this.offset.y;

        if (!this.fixedRotation)
        {
            this.sprite.rotation = this.data.angle;
        }

        if (this.debugBody)
        {
            this.debugBody.updateSpriteTransform();
        }

        this.dirty = false;

    },

    /**
    * Resets the Body force, velocity (linear and angular) and rotation. Optionally resets damping and mass.
    *
    * @method Phaser.Physics.P2.Body#reset
    * @param {number} x - The new x position of the Body.
    * @param {number} y - The new x position of the Body.
    * @param {boolean} [resetDamping=false] - Resets the linear and angular damping.
    * @param {boolean} [resetMass=false] - Sets the Body mass back to 1.
    */
    reset: function (x, y, resetDamping, resetMass) {

        if (resetDamping === undefined) { resetDamping = false; }
        if (resetMass === undefined) { resetMass = false; }

        this.setZeroForce();
        this.setZeroVelocity();
        this.setZeroRotation();

        if (resetDamping)
        {
            this.setZeroDamping();
        }

        if (resetMass)
        {
            this.mass = 1;
        }

        this.x = x;
        this.y = y;

    },

    /**
    * Adds this physics body to the world.
    *
    * @method Phaser.Physics.P2.Body#addToWorld
    */
    addToWorld: function () {

        if (this.game.physics.p2._toRemove)
        {
            for (var i = 0; i < this.game.physics.p2._toRemove.length; i++)
            {
                if (this.game.physics.p2._toRemove[i] === this)
                {
                    this.game.physics.p2._toRemove.splice(i, 1);
                }
            }
        }

        if (this.data.world !== this.game.physics.p2.world)
        {
            this.game.physics.p2.addBody(this);
        }

    },

    /**
    * Removes this physics body from the world.
    *
    * @method Phaser.Physics.P2.Body#removeFromWorld
    */
    removeFromWorld: function () {

        if (this.data.world === this.game.physics.p2.world)
        {
            this.game.physics.p2.removeBodyNextStep(this);
        }

    },

    /**
    * Destroys this Body and all references it holds to other objects.
    *
    * @method Phaser.Physics.P2.Body#destroy
    */
    destroy: function () {

        this.removeFromWorld();

        this.clearShapes();

        this._bodyCallbacks = {};
        this._bodyCallbackContext = {};
        this._groupCallbacks = {};
        this._groupCallbackContext = {};

        if (this.debugBody)
        {
            this.debugBody.destroy(true, true);
        }

        this.debugBody = null;

        if (this.sprite)
        {
            this.sprite.body = null;
            this.sprite = null;
        }

    },

    /**
    * Removes all Shapes from this Body.
    *
    * @method Phaser.Physics.P2.Body#clearShapes
    */
    clearShapes: function () {

        var i = this.data.shapes.length;

        while (i--)
        {
            this.data.removeShape(this.data.shapes[i]);
        }

        this.shapeChanged();

    },

    /**
    * Add a shape to the body. You can pass a local transform when adding a shape, so that the shape gets an offset and an angle relative to the body center of mass.
    * Will automatically update the mass properties and bounding radius.
    * If this Body had a previously set Collision Group you will need to re-apply it to the new Shape this creates.
    *
    * @method Phaser.Physics.P2.Body#addShape

    * @param {p2.Shape} shape - The shape to add to the body.
    * @param {number} [offsetX=0] - Local horizontal offset of the shape relative to the body center of mass.
    * @param {number} [offsetY=0] - Local vertical offset of the shape relative to the body center of mass.
    * @param {number} [rotation=0] - Local rotation of the shape relative to the body center of mass, specified in radians.
    * @return {p2.Shape} The shape that was added to the body.
    */
    addShape: function (shape, offsetX, offsetY, rotation) {

        if (offsetX === undefined) { offsetX = 0; }
        if (offsetY === undefined) { offsetY = 0; }
        if (rotation === undefined) { rotation = 0; }

        this.data.addShape(shape, [this.world.pxmi(offsetX), this.world.pxmi(offsetY)], rotation);
        this.shapeChanged(rotation);

        return shape;

    },

    /**
    * Adds a Circle shape to this Body. You can control the offset from the center of the body and the rotation.
    *
    * @method Phaser.Physics.P2.Body#addCircle
    * @param {number} radius - The radius of this circle (in pixels)
    * @param {number} [offsetX=0] - Local horizontal offset of the shape relative to the body center of mass.
    * @param {number} [offsetY=0] - Local vertical offset of the shape relative to the body center of mass.
    * @param {number} [rotation=0] - Local rotation of the shape relative to the body center of mass, specified in radians.
    * @return {p2.Circle} The Circle shape that was added to the Body.
    */
    addCircle: function (radius, offsetX, offsetY, rotation) {

        var shape = new p2.Circle({ radius: this.world.pxm(radius) });

        return this.addShape(shape, offsetX, offsetY, rotation);

    },

    /**
    * Adds a Rectangle shape to this Body. You can control the offset from the center of the body and the rotation.
    *
    * @method Phaser.Physics.P2.Body#addRectangle
    * @param {number} width - The width of the rectangle in pixels.
    * @param {number} height - The height of the rectangle in pixels.
    * @param {number} [offsetX=0] - Local horizontal offset of the shape relative to the body center of mass.
    * @param {number} [offsetY=0] - Local vertical offset of the shape relative to the body center of mass.
    * @param {number} [rotation=0] - Local rotation of the shape relative to the body center of mass, specified in radians.
    * @return {p2.Box} The shape that was added to the Body.
    */
    addRectangle: function (width, height, offsetX, offsetY, rotation) {

        var shape = new p2.Box({ width: this.world.pxm(width), height: this.world.pxm(height)});

        return this.addShape(shape, offsetX, offsetY, rotation);

    },

    /**
    * Adds a Plane shape to this Body. The plane is facing in the Y direction. You can control the offset from the center of the body and the rotation.
    *
    * @method Phaser.Physics.P2.Body#addPlane
    * @param {number} [offsetX=0] - Local horizontal offset of the shape relative to the body center of mass.
    * @param {number} [offsetY=0] - Local vertical offset of the shape relative to the body center of mass.
    * @param {number} [rotation=0] - Local rotation of the shape relative to the body center of mass, specified in radians.
    * @return {p2.Plane} The Plane shape that was added to the Body.
    */
    addPlane: function (offsetX, offsetY, rotation) {

        var shape = new p2.Plane();

        return this.addShape(shape, offsetX, offsetY, rotation);

    },

    /**
    * Adds a Particle shape to this Body. You can control the offset from the center of the body and the rotation.
    *
    * @method Phaser.Physics.P2.Body#addParticle
    * @param {number} [offsetX=0] - Local horizontal offset of the shape relative to the body center of mass.
    * @param {number} [offsetY=0] - Local vertical offset of the shape relative to the body center of mass.
    * @param {number} [rotation=0] - Local rotation of the shape relative to the body center of mass, specified in radians.
    * @return {p2.Particle} The Particle shape that was added to the Body.
    */
    addParticle: function (offsetX, offsetY, rotation) {

        var shape = new p2.Particle();

        return this.addShape(shape, offsetX, offsetY, rotation);

    },

    /**
    * Adds a Line shape to this Body.
    * The line shape is along the x direction, and stretches from [-length/2, 0] to [length/2,0].
    * You can control the offset from the center of the body and the rotation.
    *
    * @method Phaser.Physics.P2.Body#addLine
    * @param {number} length - The length of this line (in pixels)
    * @param {number} [offsetX=0] - Local horizontal offset of the shape relative to the body center of mass.
    * @param {number} [offsetY=0] - Local vertical offset of the shape relative to the body center of mass.
    * @param {number} [rotation=0] - Local rotation of the shape relative to the body center of mass, specified in radians.
    * @return {p2.Line} The Line shape that was added to the Body.
    */
    addLine: function (length, offsetX, offsetY, rotation) {

        var shape = new p2.Line({ length: this.world.pxm(length)});

        return this.addShape(shape, offsetX, offsetY, rotation);

    },

    /**
    * Adds a Capsule shape to this Body.
    * You can control the offset from the center of the body and the rotation.
    *
    * @method Phaser.Physics.P2.Body#addCapsule
    * @param {number} length - The distance between the end points in pixels.
    * @param {number} radius - Radius of the capsule in pixels.
    * @param {number} [offsetX=0] - Local horizontal offset of the shape relative to the body center of mass.
    * @param {number} [offsetY=0] - Local vertical offset of the shape relative to the body center of mass.
    * @param {number} [rotation=0] - Local rotation of the shape relative to the body center of mass, specified in radians.
    * @return {p2.Capsule} The Capsule shape that was added to the Body.
    */
    addCapsule: function (length, radius, offsetX, offsetY, rotation) {

        var shape = new p2.Capsule({ length: this.world.pxm(length), radius: this.world.pxm(radius) });

        return this.addShape(shape, offsetX, offsetY, rotation);

    },

    /**
    * Reads a polygon shape path, and assembles convex shapes from that and puts them at proper offset points. The shape must be simple and without holes.
    * This function expects the x.y values to be given in pixels. If you want to provide them at p2 world scales then call Body.data.fromPolygon directly.
    *
    * @method Phaser.Physics.P2.Body#addPolygon
    * @param {object} options - An object containing the build options:
    * @param {boolean} [options.optimalDecomp=false] - Set to true if you need optimal decomposition. Warning: very slow for polygons with more than 10 vertices.
    * @param {boolean} [options.skipSimpleCheck=false] - Set to true if you already know that the path is not intersecting itself.
    * @param {boolean|number} [options.removeCollinearPoints=false] - Set to a number (angle threshold value) to remove collinear points, or false to keep all points.
    * @param {(number[]|...number)} points - An array of 2d vectors that form the convex or concave polygon.
    *                                       Either [[0,0], [0,1],...] or a flat array of numbers that will be interpreted as [x,y, x,y, ...]. In the first form **the array will mutate**.
    *                                       Or the arguments passed can be flat x,y values e.g. `setPolygon(options, x,y, x,y, x,y, ...)` where `x` and `y` are numbers.
    * @return {boolean} True on success, else false.
    */
    addPolygon: function (options, points) {

        options = options || {};

        if (!Array.isArray(points))
        {
            points = Array.prototype.slice.call(arguments, 1);
        }

        var path = [];

        //  Did they pass in a single array of points?
        if (points.length === 1 && Array.isArray(points[0]))
        {
            path = points[0].slice(0);
        }
        else if (Array.isArray(points[0]))
        {
            path = points.slice();
        }
        else if (typeof points[0] === 'number')
        {
            //  We've a list of numbers
            for (var i = 0, len = points.length; i < len; i += 2)
            {
                path.push([points[i], points[i + 1]]);
            }
        }

        //  top and tail
        var idx = path.length - 1;

        if (path[idx][0] === path[0][0] && path[idx][1] === path[0][1])
        {
            path.pop();
        }

        //  Now process them into p2 values
        for (var p = 0; p < path.length; p++)
        {
            path[p][0] = this.world.pxmi(path[p][0]);
            path[p][1] = this.world.pxmi(path[p][1]);
        }

        var result = this.data.fromPolygon(path, options);

        this.shapeChanged();

        return result;

    },

    /**
    * Remove a shape from the body. Will automatically update the mass properties and bounding radius.
    *
    * @method Phaser.Physics.P2.Body#removeShape
    * @param {p2.Circle|p2.Rectangle|p2.Plane|p2.Line|p2.Particle} shape - The shape to remove from the body.
    * @return {boolean} True if the shape was found and removed, else false.
    */
    removeShape: function (shape) {

		var result = this.data.removeShape(shape);

		this.shapeChanged();

        return result;
    },

    /**
    * Clears any previously set shapes. Then creates a new Circle shape and adds it to this Body.
    * If this Body had a previously set Collision Group you will need to re-apply it to the new Shape this creates.
    *
    * @method Phaser.Physics.P2.Body#setCircle
    * @param {number} radius - The radius of this circle (in pixels)
    * @param {number} [offsetX=0] - Local horizontal offset of the shape relative to the body center of mass.
    * @param {number} [offsetY=0] - Local vertical offset of the shape relative to the body center of mass.
    * @param {number} [rotation=0] - Local rotation of the shape relative to the body center of mass, specified in radians.
    */
    setCircle: function (radius, offsetX, offsetY, rotation) {

        this.clearShapes();

        return this.addCircle(radius, offsetX, offsetY, rotation);

    },

    /**
    * Clears any previously set shapes. The creates a new Rectangle shape at the given size and offset, and adds it to this Body.
    * If you wish to create a Rectangle to match the size of a Sprite or Image see Body.setRectangleFromSprite.
    * If this Body had a previously set Collision Group you will need to re-apply it to the new Shape this creates.
    *
    * @method Phaser.Physics.P2.Body#setRectangle
    * @param {number} [width=16] - The width of the rectangle in pixels.
    * @param {number} [height=16] - The height of the rectangle in pixels.
    * @param {number} [offsetX=0] - Local horizontal offset of the shape relative to the body center of mass.
    * @param {number} [offsetY=0] - Local vertical offset of the shape relative to the body center of mass.
    * @param {number} [rotation=0] - Local rotation of the shape relative to the body center of mass, specified in radians.
    * @return {p2.Rectangle} The Rectangle shape that was added to the Body.
    */
    setRectangle: function (width, height, offsetX, offsetY, rotation) {

        if (width === undefined) { width = 16; }
        if (height === undefined) { height = 16; }

        this.clearShapes();

        return this.addRectangle(width, height, offsetX, offsetY, rotation);

    },

    /**
    * Clears any previously set shapes.
    * Then creates a Rectangle shape sized to match the dimensions and orientation of the Sprite given.
    * If no Sprite is given it defaults to using the parent of this Body.
    * If this Body had a previously set Collision Group you will need to re-apply it to the new Shape this creates.
    *
    * @method Phaser.Physics.P2.Body#setRectangleFromSprite
    * @param {Phaser.Sprite|Phaser.Image} [sprite] - The Sprite on which the Rectangle will get its dimensions.
    * @return {p2.Rectangle} The Rectangle shape that was added to the Body.
    */
    setRectangleFromSprite: function (sprite) {

        if (sprite === undefined) { sprite = this.sprite; }

        this.clearShapes();

        return this.addRectangle(sprite.width, sprite.height, 0, 0, sprite.rotation);

    },

    /**
    * Adds the given Material to all Shapes that belong to this Body.
    * If you only wish to apply it to a specific Shape in this Body then provide that as the 2nd parameter.
    *
    * @method Phaser.Physics.P2.Body#setMaterial
    * @param {Phaser.Physics.P2.Material} material - The Material that will be applied.
    * @param {p2.Shape} [shape] - An optional Shape. If not provided the Material will be added to all Shapes in this Body.
    */
    setMaterial: function (material, shape) {

        if (shape === undefined)
        {
            for (var i = this.data.shapes.length - 1; i >= 0; i--)
            {
                this.data.shapes[i].material = material;
            }
        }
        else
        {
            shape.material = material;
        }

    },

    /**
    * Updates the debug draw if any body shapes change. Always update the angle data prior to debug drawing the shape.
    *
    * @method Phaser.Physics.P2.Body#shapeChanged
    * @param {number} [rotation=0] - Local rotation of the shape relative to the body center of mass, specified in radians.
    */
    shapeChanged: function(rotation) {

            this.data.angle=rotation;
        if (this.debugBody)
        {
            this.debugBody.draw();
        }

    },

    /**
    * Reads the shape data from a physics data file stored in the Game.Cache and adds it as a polygon to this Body.
    * The shape data format is based on the output of the
    * {@link https://github.com/photonstorm/phaser/tree/master/resources/PhysicsEditor%20Exporter|custom phaser exporter} for
    * {@link https://www.codeandweb.com/physicseditor|PhysicsEditor}
    *
    * @method Phaser.Physics.P2.Body#addPhaserPolygon
    * @param {string} key - The key of the Physics Data file as stored in Game.Cache.
    * @param {string} object - The key of the object within the Physics data file that you wish to load the shape data from.
    * @returns {Array} A list of created fixtures to be used with Phaser.Physics.P2.FixtureList
    */
    addPhaserPolygon: function (key, object) {

        var data = this.game.cache.getPhysicsData(key, object);
        var createdFixtures = [];

        //  Cycle through the fixtures
        for (var i = 0; i < data.length; i++)
        {
            var fixtureData = data[i];
            var shapesOfFixture = this.addFixture(fixtureData);

            //  Always add to a group
            createdFixtures[fixtureData.filter.group] = createdFixtures[fixtureData.filter.group] || [];
            createdFixtures[fixtureData.filter.group] = createdFixtures[fixtureData.filter.group].concat(shapesOfFixture);

            //  if (unique) fixture key is provided
            if (fixtureData.fixtureKey)
            {
                createdFixtures[fixtureData.fixtureKey] = shapesOfFixture;
            }
        }

        this.data.aabbNeedsUpdate = true;
        this.shapeChanged();

        return createdFixtures;

    },

    /**
    * Add a polygon fixture. This is used during #loadPolygon.
    *
    * @method Phaser.Physics.P2.Body#addFixture
    * @param {string} fixtureData - The data for the fixture. It contains: isSensor, filter (collision) and the actual polygon shapes.
    * @return {array} An array containing the generated shapes for the given polygon.
    */
    addFixture: function (fixtureData) {

        var generatedShapes = [];

        if (fixtureData.circle)
        {
            var shape = new p2.Circle({ radius: this.world.pxm(fixtureData.circle.radius) });
            shape.collisionGroup = fixtureData.filter.categoryBits;
            shape.collisionMask = fixtureData.filter.maskBits;
            shape.sensor = fixtureData.isSensor;

            var offset = p2.vec2.create();
            offset[0] = this.world.pxmi(fixtureData.circle.position[0] - this.sprite.width/2);
            offset[1] = this.world.pxmi(fixtureData.circle.position[1] - this.sprite.height/2);

            this.data.addShape(shape, offset);
            generatedShapes.push(shape);
        }
        else
        {
            var polygons = fixtureData.polygons;
            var cm = p2.vec2.create();

            for (var i = 0; i < polygons.length; i++)
            {
                var shapes = polygons[i];
                var vertices = [];

                for (var s = 0; s < shapes.length; s += 2)
                {
                    vertices.push([ this.world.pxmi(shapes[s]), this.world.pxmi(shapes[s + 1]) ]);
                }

                var shape = new p2.Convex({ vertices: vertices });

                //  Move all vertices so its center of mass is in the local center of the convex
                for (var j = 0; j !== shape.vertices.length; j++)
                {
                    var v = shape.vertices[j];
                    p2.vec2.sub(v, v, shape.centerOfMass);
                }

                p2.vec2.scale(cm, shape.centerOfMass, 1);

                cm[0] -= this.world.pxmi(this.sprite.width / 2);
                cm[1] -= this.world.pxmi(this.sprite.height / 2);

                shape.updateTriangles();
                shape.updateCenterOfMass();
                shape.updateBoundingRadius();

                shape.collisionGroup = fixtureData.filter.categoryBits;
                shape.collisionMask = fixtureData.filter.maskBits;
                shape.sensor = fixtureData.isSensor;

                this.data.addShape(shape, cm);

                generatedShapes.push(shape);
            }
        }

        return generatedShapes;

    },

    /**
    * Reads the shape data from a physics data file stored in the Game.Cache and adds it as a polygon to this Body.
    *
    * As well as reading the data from the Cache you can also pass `null` as the first argument and a
    * physics data object as the second. When doing this you must ensure the structure of the object is correct in advance.
    *
    * For more details see the format of the Lime / Corona Physics Editor export.
    *
    * @method Phaser.Physics.P2.Body#loadPolygon
    * @param {string} key - The key of the Physics Data file as stored in Game.Cache. Alternatively set to `null` and pass the
    *     data as the 2nd argument.
    * @param {string|object} object - The key of the object within the Physics data file that you wish to load the shape data from,
    *     or if key is null pass the actual physics data object itself as this parameter.
    * @param {number} [scale=1] - Optionally resize the loaded polygon.
    * @param {number} [rotation=0] - Local rotation of the shape relative to the body center of mass, specified in radians.
    * @return {boolean} True on success, else false.
    */
    loadPolygon: function (key, object, scale,rotation) {

        if (key === null)
        {
            var data = object;
        }
        else
        {
            var data = this.game.cache.getPhysicsData(key, object);
        }

        if (typeof scale !== "number")
        {
            scale = 1;
        }

        //  We've multiple Convex shapes, they should be CCW automatically
        var cm = p2.vec2.create();

        for (var i = 0; i < data.length; i++)
        {
            var vertices = [];

            for (var s = 0; s < data[i].shape.length; s += 2)
            {
                vertices.push([
                    this.world.pxmi(data[i].shape[s] * scale),
                    this.world.pxmi(data[i].shape[s + 1] * scale)
                ]);
            }

            var c = new p2.Convex({ vertices: vertices });

            // Move all vertices so its center of mass is in the local center of the convex
            for (var j = 0; j !== c.vertices.length; j++)
            {
                var v = c.vertices[j];
                p2.vec2.sub(v, v, c.centerOfMass);
            }

            p2.vec2.scale(cm, c.centerOfMass, 1);

            cm[0] -= this.world.pxmi(this.sprite.width / 2);
            cm[1] -= this.world.pxmi(this.sprite.height / 2);

            c.updateTriangles();
            c.updateCenterOfMass();
            c.updateBoundingRadius();

            this.data.addShape(c, cm);
        }

        this.data.aabbNeedsUpdate = true;
        this.shapeChanged(rotation);

        return true;

    }

};

Phaser.Physics.P2.Body.prototype.constructor = Phaser.Physics.P2.Body;

/**
 * Dynamic body. Dynamic bodies body can move and respond to collisions and forces.
 * @property DYNAMIC
 * @type {Number}
 * @static
 */
Phaser.Physics.P2.Body.DYNAMIC = 1;

/**
 * Static body. Static bodies do not move, and they do not respond to forces or collision.
 * @property STATIC
 * @type {Number}
 * @static
 */
Phaser.Physics.P2.Body.STATIC = 2;

/**
 * Kinematic body. Kinematic bodies only moves according to its .velocity, and does not respond to collisions or force.
 * @property KINEMATIC
 * @type {Number}
 * @static
 */
Phaser.Physics.P2.Body.KINEMATIC = 4;

/**
* @name Phaser.Physics.P2.Body#static
* @property {boolean} static - Returns true if the Body is static. Setting Body.static to 'false' will make it dynamic.
*/
Object.defineProperty(Phaser.Physics.P2.Body.prototype, "static", {

    get: function () {

        return (this.data.type === Phaser.Physics.P2.Body.STATIC);

    },

    set: function (value) {

        if (value && this.data.type !== Phaser.Physics.P2.Body.STATIC)
        {
            this.data.type = Phaser.Physics.P2.Body.STATIC;
            this.mass = 0;
        }
        else if (!value && this.data.type === Phaser.Physics.P2.Body.STATIC)
        {
            this.data.type = Phaser.Physics.P2.Body.DYNAMIC;
            this.mass = 1;
        }

    }

});

/**
* @name Phaser.Physics.P2.Body#dynamic
* @property {boolean} dynamic - Returns true if the Body is dynamic. Setting Body.dynamic to 'false' will make it static.
*/
Object.defineProperty(Phaser.Physics.P2.Body.prototype, "dynamic", {

    get: function () {

        return (this.data.type === Phaser.Physics.P2.Body.DYNAMIC);

    },

    set: function (value) {

        if (value && this.data.type !== Phaser.Physics.P2.Body.DYNAMIC)
        {
            this.data.type = Phaser.Physics.P2.Body.DYNAMIC;
            this.mass = 1;
        }
        else if (!value && this.data.type === Phaser.Physics.P2.Body.DYNAMIC)
        {
            this.data.type = Phaser.Physics.P2.Body.STATIC;
            this.mass = 0;
        }

    }

});

/**
* @name Phaser.Physics.P2.Body#kinematic
* @property {boolean} kinematic - Returns true if the Body is kinematic. Setting Body.kinematic to 'false' will make it static.
*/
Object.defineProperty(Phaser.Physics.P2.Body.prototype, "kinematic", {

    get: function () {

        return (this.data.type === Phaser.Physics.P2.Body.KINEMATIC);

    },

    set: function (value) {

        if (value && this.data.type !== Phaser.Physics.P2.Body.KINEMATIC)
        {
            this.data.type = Phaser.Physics.P2.Body.KINEMATIC;
            this.mass = 4;
        }
        else if (!value && this.data.type === Phaser.Physics.P2.Body.KINEMATIC)
        {
            this.data.type = Phaser.Physics.P2.Body.STATIC;
            this.mass = 0;
        }

    }

});

/**
* @name Phaser.Physics.P2.Body#allowSleep
* @property {boolean} allowSleep -
*/
Object.defineProperty(Phaser.Physics.P2.Body.prototype, "allowSleep", {

    get: function () {

        return this.data.allowSleep;

    },

    set: function (value) {

        if (value !== this.data.allowSleep)
        {
            this.data.allowSleep = value;
        }

    }

});

/**
* The angle of the Body in degrees from its original orientation. Values from 0 to 180 represent clockwise rotation; values from 0 to -180 represent counterclockwise rotation.
* Values outside this range are added to or subtracted from 360 to obtain a value within the range. For example, the statement Body.angle = 450 is the same as Body.angle = 90.
* If you wish to work in radians instead of degrees use the property Body.rotation instead. Working in radians is faster as it doesn't have to convert values.
*
* @name Phaser.Physics.P2.Body#angle
* @property {number} angle - The angle of this Body in degrees.
*/
Object.defineProperty(Phaser.Physics.P2.Body.prototype, "angle", {

    get: function() {

        return Phaser.Math.wrapAngle(Phaser.Math.radToDeg(this.data.angle));

    },

    set: function(value) {

        this.data.angle = Phaser.Math.degToRad(Phaser.Math.wrapAngle(value));

    }

});

/**
* Damping is specified as a value between 0 and 1, which is the proportion of velocity lost per second.
* @name Phaser.Physics.P2.Body#angularDamping
* @property {number} angularDamping - The angular damping acting acting on the body.
*/
Object.defineProperty(Phaser.Physics.P2.Body.prototype, "angularDamping", {

    get: function () {

        return this.data.angularDamping;

    },

    set: function (value) {

        this.data.angularDamping = value;

    }

});

/**
* @name Phaser.Physics.P2.Body#angularForce
* @property {number} angularForce - The angular force acting on the body.
*/
Object.defineProperty(Phaser.Physics.P2.Body.prototype, "angularForce", {

    get: function () {

        return this.data.angularForce;

    },

    set: function (value) {

        this.data.angularForce = value;

    }

});

/**
* @name Phaser.Physics.P2.Body#angularVelocity
* @property {number} angularVelocity - The angular velocity of the body.
*/
Object.defineProperty(Phaser.Physics.P2.Body.prototype, "angularVelocity", {

    get: function () {

        return this.data.angularVelocity;

    },

    set: function (value) {

        this.data.angularVelocity = value;

    }

});

/**
* Damping is specified as a value between 0 and 1, which is the proportion of velocity lost per second.
* @name Phaser.Physics.P2.Body#damping
* @property {number} damping - The linear damping acting on the body in the velocity direction.
*/
Object.defineProperty(Phaser.Physics.P2.Body.prototype, "damping", {

    get: function () {

        return this.data.damping;

    },

    set: function (value) {

        this.data.damping = value;

    }

});

/**
* @name Phaser.Physics.P2.Body#fixedRotation
* @property {boolean} fixedRotation -
*/
Object.defineProperty(Phaser.Physics.P2.Body.prototype, "fixedRotation", {

    get: function () {

        return this.data.fixedRotation;

    },

    set: function (value) {

        if (value !== this.data.fixedRotation)
        {
            this.data.fixedRotation = value;
        }

    }

});

/**
* @name Phaser.Physics.P2.Body#inertia
* @property {number} inertia - The inertia of the body around the Z axis..
*/
Object.defineProperty(Phaser.Physics.P2.Body.prototype, "inertia", {

    get: function () {

        return this.data.inertia;

    },

    set: function (value) {

        this.data.inertia = value;

    }

});

/**
* @name Phaser.Physics.P2.Body#mass
* @property {number} mass - The mass of the body.
*/
Object.defineProperty(Phaser.Physics.P2.Body.prototype, "mass", {

    get: function () {

        return this.data.mass;

    },

    set: function (value) {

        if (value !== this.data.mass)
        {
            this.data.mass = value;
            this.data.updateMassProperties();
        }

    }

});

/**
* @name Phaser.Physics.P2.Body#motionState
* @property {number} motionState - The type of motion this body has. Should be one of: Body.STATIC (the body does not move), Body.DYNAMIC (body can move and respond to collisions) and Body.KINEMATIC (only moves according to its .velocity).
*/
Object.defineProperty(Phaser.Physics.P2.Body.prototype, "motionState", {

    get: function () {

        return this.data.type;

    },

    set: function (value) {

        if (value !== this.data.type)
        {
            this.data.type = value;
        }

    }

});

/**
* The angle of the Body in radians.
* If you wish to work in degrees instead of radians use the Body.angle property instead. Working in radians is faster as it doesn't have to convert values.
*
* @name Phaser.Physics.P2.Body#rotation
* @property {number} rotation - The angle of this Body in radians.
*/
Object.defineProperty(Phaser.Physics.P2.Body.prototype, "rotation", {

    get: function() {

        return this.data.angle;

    },

    set: function(value) {

        this.data.angle = value;

    }

});

/**
* @name Phaser.Physics.P2.Body#sleepSpeedLimit
* @property {number} sleepSpeedLimit - .
*/
Object.defineProperty(Phaser.Physics.P2.Body.prototype, "sleepSpeedLimit", {

    get: function () {

        return this.data.sleepSpeedLimit;

    },

    set: function (value) {

        this.data.sleepSpeedLimit = value;

    }

});

/**
* @name Phaser.Physics.P2.Body#x
* @property {number} x - The x coordinate of this Body.
*/
Object.defineProperty(Phaser.Physics.P2.Body.prototype, "x", {

    get: function () {

        return this.world.mpxi(this.data.position[0]);

    },

    set: function (value) {

        this.data.position[0] = this.world.pxmi(value);

    }

});

/**
* @name Phaser.Physics.P2.Body#y
* @property {number} y - The y coordinate of this Body.
*/
Object.defineProperty(Phaser.Physics.P2.Body.prototype, "y", {

    get: function () {

        return this.world.mpxi(this.data.position[1]);

    },

    set: function (value) {

        this.data.position[1] = this.world.pxmi(value);

    }

});

/**
* @name Phaser.Physics.P2.Body#id
* @property {number} id - The Body ID. Each Body that has been added to the World has a unique ID.
* @readonly
*/
Object.defineProperty(Phaser.Physics.P2.Body.prototype, "id", {

    get: function () {

        return this.data.id;

    }

});

/**
* @name Phaser.Physics.P2.Body#debug
* @property {boolean} debug - Enable or disable debug drawing of this body
*/
Object.defineProperty(Phaser.Physics.P2.Body.prototype, "debug", {

    get: function () {

        return (this.debugBody !== null);

    },

    set: function (value) {

        if (value && !this.debugBody)
        {
            //  This will be added to the global space
            this.debugBody = new Phaser.Physics.P2.BodyDebug(this.game, this.data);
        }
        else if (!value && this.debugBody)
        {
            this.debugBody.destroy();
            this.debugBody = null;
        }

    }

});

/**
* A Body can be set to collide against the World bounds automatically if this is set to true. Otherwise it will leave the World.
* Note that this only applies if your World has bounds! The response to the collision should be managed via CollisionMaterials.
* Also note that when you set this it will only affect Body shapes that already exist. If you then add further shapes to your Body
* after setting this it will *not* proactively set them to collide with the bounds.
*
* @name Phaser.Physics.P2.Body#collideWorldBounds
* @property {boolean} collideWorldBounds - Should the Body collide with the World bounds?
* @default true
*/
Object.defineProperty(Phaser.Physics.P2.Body.prototype, "collideWorldBounds", {

    get: function () {

        return this._collideWorldBounds;

    },

    set: function (value) {

        if (value && !this._collideWorldBounds)
        {
            this._collideWorldBounds = true;
            this.updateCollisionMask();
        }
        else if (!value && this._collideWorldBounds)
        {
            this._collideWorldBounds = false;
            this.updateCollisionMask();
        }

    }

});
