/// <reference types="./matter" />

declare type CameraRotateCallback = (camera: Phaser.Cameras.Scene2D.Camera, progress: number, angle: number)=>void;

declare type DataEachCallback = (parent: any, key: string, value: any, ...args: any[])=>void;

declare type ContentLoadedCallback = ()=>void;

declare type CreateCallback = (bob: Phaser.GameObjects.Bob, index: number)=>void;

declare type EachContainerCallback<I> = (item: any, ...args: any[])=>void;

declare type LightForEach = (light: Phaser.GameObjects.Light)=>void;

/**
 * A custom function that will be responsible for wrapping the text.
 */
declare type TextStyleWordWrapCallback = (text: string, textObject: Phaser.GameObjects.Text)=>void;

declare type CenterFunction = (triangle: Phaser.Geom.Triangle)=>void;

declare namespace Phaser {
    namespace Actions {
        /**
         * Takes an array of Game Objects, or any objects that have public `x` and `y` properties, and aligns them next to each other.
         * 
         * The first item isn't moved. The second item is aligned next to the first, then the third next to the second, and so on.
         * @param items The array of items to be updated by this action.
         * @param position The position to align the items with. This is an align constant, such as `Phaser.Display.Align.LEFT_CENTER`.
         * @param offsetX Optional horizontal offset from the position. Default 0.
         * @param offsetY Optional vertical offset from the position. Default 0.
         */
        function AlignTo<G extends Phaser.GameObjects.GameObject[]>(items: G, position: number, offsetX?: number, offsetY?: number): G;

        /**
         * Takes an array of Game Objects, or any objects that have a public `angle` property,
         * and then adds the given value to each of their `angle` properties.
         * 
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         * 
         * To use this with a Group: `Angle(group.getChildren(), value, step)`
         * @param items The array of items to be updated by this action.
         * @param value The amount to be added to the `angle` property.
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function Angle<G extends Phaser.GameObjects.GameObject[]>(items: G, value: number, step?: number, index?: number, direction?: number): G;

        /**
         * Takes an array of objects and passes each of them to the given callback.
         * @param items The array of items to be updated by this action.
         * @param callback The callback to be invoked. It will be passed just one argument: the item from the array.
         * @param context The scope in which the callback will be invoked.
         */
        function Call<G extends Phaser.GameObjects.GameObject[]>(items: G, callback: Phaser.Types.Actions.CallCallback, context: any): G;

        /**
         * Takes an array of objects and returns the first element in the array that has properties which match
         * all of those specified in the `compare` object. For example, if the compare object was: `{ scaleX: 0.5, alpha: 1 }`
         * then it would return the first item which had the property `scaleX` set to 0.5 and `alpha` set to 1.
         * 
         * To use this with a Group: `GetFirst(group.getChildren(), compare, index)`
         * @param items The array of items to be searched by this action.
         * @param compare The comparison object. Each property in this object will be checked against the items of the array.
         * @param index An optional offset to start searching from within the items array. Default 0.
         */
        function GetFirst<G extends Phaser.GameObjects.GameObject[]>(items: G, compare: object, index?: number): object | Phaser.GameObjects.GameObject;

        /**
         * Takes an array of objects and returns the last element in the array that has properties which match
         * all of those specified in the `compare` object. For example, if the compare object was: `{ scaleX: 0.5, alpha: 1 }`
         * then it would return the last item which had the property `scaleX` set to 0.5 and `alpha` set to 1.
         * 
         * To use this with a Group: `GetLast(group.getChildren(), compare, index)`
         * @param items The array of items to be searched by this action.
         * @param compare The comparison object. Each property in this object will be checked against the items of the array.
         * @param index An optional offset to start searching from within the items array. Default 0.
         */
        function GetLast<G extends Phaser.GameObjects.GameObject[]>(items: G, compare: object, index?: number): object | Phaser.GameObjects.GameObject;

        /**
         * Takes an array of Game Objects, or any objects that have public `x` and `y` properties,
         * and then aligns them based on the grid configuration given to this action.
         * @param items The array of items to be updated by this action.
         * @param options The GridAlign Configuration object.
         */
        function GridAlign<G extends Phaser.GameObjects.GameObject[]>(items: G, options: Phaser.Types.Actions.GridAlignConfig): G;

        /**
         * Takes an array of Game Objects, or any objects that have a public `alpha` property,
         * and then adds the given value to each of their `alpha` properties.
         * 
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         * 
         * To use this with a Group: `IncAlpha(group.getChildren(), value, step)`
         * @param items The array of items to be updated by this action.
         * @param value The amount to be added to the `alpha` property.
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function IncAlpha<G extends Phaser.GameObjects.GameObject[]>(items: G, value: number, step?: number, index?: number, direction?: number): G;

        /**
         * Takes an array of Game Objects, or any objects that have a public `x` property,
         * and then adds the given value to each of their `x` properties.
         * 
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         * 
         * To use this with a Group: `IncX(group.getChildren(), value, step)`
         * @param items The array of items to be updated by this action.
         * @param value The amount to be added to the `x` property.
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function IncX<G extends Phaser.GameObjects.GameObject[]>(items: G, value: number, step?: number, index?: number, direction?: number): G;

        /**
         * Takes an array of Game Objects, or any objects that have public `x` and `y` properties,
         * and then adds the given value to each of them.
         * 
         * The optional `stepX` and `stepY` properties are applied incrementally, multiplied by each item in the array.
         * 
         * To use this with a Group: `IncXY(group.getChildren(), x, y, stepX, stepY)`
         * @param items The array of items to be updated by this action.
         * @param x The amount to be added to the `x` property.
         * @param y The amount to be added to the `y` property. If `undefined` or `null` it uses the `x` value. Default x.
         * @param stepX This is added to the `x` amount, multiplied by the iteration counter. Default 0.
         * @param stepY This is added to the `y` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function IncXY<G extends Phaser.GameObjects.GameObject[]>(items: G, x: number, y?: number, stepX?: number, stepY?: number, index?: number, direction?: number): G;

        /**
         * Takes an array of Game Objects, or any objects that have a public `y` property,
         * and then adds the given value to each of their `y` properties.
         * 
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         * 
         * To use this with a Group: `IncY(group.getChildren(), value, step)`
         * @param items The array of items to be updated by this action.
         * @param value The amount to be added to the `y` property.
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function IncY<G extends Phaser.GameObjects.GameObject[]>(items: G, value: number, step?: number, index?: number, direction?: number): G;

        /**
         * Takes an array of Game Objects and positions them on evenly spaced points around the perimeter of a Circle.
         * 
         * If you wish to pass a `Phaser.GameObjects.Circle` Shape to this function, you should pass its `geom` property.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param circle The Circle to position the Game Objects on.
         * @param startAngle Optional angle to start position from, in radians. Default 0.
         * @param endAngle Optional angle to stop position at, in radians. Default 6.28.
         */
        function PlaceOnCircle<G extends Phaser.GameObjects.GameObject[]>(items: G, circle: Phaser.Geom.Circle, startAngle?: number, endAngle?: number): G;

        /**
         * Takes an array of Game Objects and positions them on evenly spaced points around the perimeter of an Ellipse.
         * 
         * If you wish to pass a `Phaser.GameObjects.Ellipse` Shape to this function, you should pass its `geom` property.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param ellipse The Ellipse to position the Game Objects on.
         * @param startAngle Optional angle to start position from, in radians. Default 0.
         * @param endAngle Optional angle to stop position at, in radians. Default 6.28.
         */
        function PlaceOnEllipse<G extends Phaser.GameObjects.GameObject[]>(items: G, ellipse: Phaser.Geom.Ellipse, startAngle?: number, endAngle?: number): G;

        /**
         * Positions an array of Game Objects on evenly spaced points of a Line.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param line The Line to position the Game Objects on.
         */
        function PlaceOnLine<G extends Phaser.GameObjects.GameObject[]>(items: G, line: Phaser.Geom.Line): G;

        /**
         * Takes an array of Game Objects and positions them on evenly spaced points around the perimeter of a Rectangle.
         * 
         * Placement starts from the top-left of the rectangle, and proceeds in a clockwise direction.
         * If the `shift` parameter is given you can offset where placement begins.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param rect The Rectangle to position the Game Objects on.
         * @param shift An optional positional offset. Default 0.
         */
        function PlaceOnRectangle<G extends Phaser.GameObjects.GameObject[]>(items: G, rect: Phaser.Geom.Rectangle, shift?: number): G;

        /**
         * Takes an array of Game Objects and positions them on evenly spaced points around the edges of a Triangle.
         * 
         * If you wish to pass a `Phaser.GameObjects.Triangle` Shape to this function, you should pass its `geom` property.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param triangle The Triangle to position the Game Objects on.
         * @param stepRate An optional step rate, to increase or decrease the packing of the Game Objects on the lines. Default 1.
         */
        function PlaceOnTriangle<G extends Phaser.GameObjects.GameObject[]>(items: G, triangle: Phaser.Geom.Triangle, stepRate?: number): G;

        /**
         * Play an animation on all Game Objects in the array that have an Animation component.
         * 
         * You can pass either an animation key, or an animation configuration object for more control over the playback.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param key The string-based key of the animation to play, or an Animation instance, or a `PlayAnimationConfig` object.
         * @param ignoreIfPlaying If this animation is already playing then ignore this call. Default false.
         */
        function PlayAnimation<G extends Phaser.GameObjects.GameObject[]>(items: G, key: string | Phaser.Animations.Animation | Phaser.Types.Animations.PlayAnimationConfig, ignoreIfPlaying?: boolean): G;

        /**
         * Takes an array of Game Objects, or any objects that have a public property as defined in `key`,
         * and then adds the given value to it.
         * 
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         * 
         * To use this with a Group: `PropertyValueInc(group.getChildren(), key, value, step)`
         * @param items The array of items to be updated by this action.
         * @param key The property to be updated.
         * @param value The amount to be added to the property.
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function PropertyValueInc<G extends Phaser.GameObjects.GameObject[]>(items: G, key: string, value: number, step?: number, index?: number, direction?: number): G;

        /**
         * Takes an array of Game Objects, or any objects that have a public property as defined in `key`,
         * and then sets it to the given value.
         * 
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         * 
         * To use this with a Group: `PropertyValueSet(group.getChildren(), key, value, step)`
         * @param items The array of items to be updated by this action.
         * @param key The property to be updated.
         * @param value The amount to set the property to.
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function PropertyValueSet<G extends Phaser.GameObjects.GameObject[]>(items: G, key: string, value: number, step?: number, index?: number, direction?: number): G;

        /**
         * Takes an array of Game Objects and positions them at random locations within the Circle.
         * 
         * If you wish to pass a `Phaser.GameObjects.Circle` Shape to this function, you should pass its `geom` property.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param circle The Circle to position the Game Objects within.
         */
        function RandomCircle<G extends Phaser.GameObjects.GameObject[]>(items: G, circle: Phaser.Geom.Circle): G;

        /**
         * Takes an array of Game Objects and positions them at random locations within the Ellipse.
         * 
         * If you wish to pass a `Phaser.GameObjects.Ellipse` Shape to this function, you should pass its `geom` property.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param ellipse The Ellipse to position the Game Objects within.
         */
        function RandomEllipse<G extends Phaser.GameObjects.GameObject[]>(items: G, ellipse: Phaser.Geom.Ellipse): G;

        /**
         * Takes an array of Game Objects and positions them at random locations on the Line.
         * 
         * If you wish to pass a `Phaser.GameObjects.Line` Shape to this function, you should pass its `geom` property.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param line The Line to position the Game Objects randomly on.
         */
        function RandomLine<G extends Phaser.GameObjects.GameObject[]>(items: G, line: Phaser.Geom.Line): G;

        /**
         * Takes an array of Game Objects and positions them at random locations within the Rectangle.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param rect The Rectangle to position the Game Objects within.
         */
        function RandomRectangle<G extends Phaser.GameObjects.GameObject[]>(items: G, rect: Phaser.Geom.Rectangle): G;

        /**
         * Takes an array of Game Objects and positions them at random locations within the Triangle.
         * 
         * If you wish to pass a `Phaser.GameObjects.Triangle` Shape to this function, you should pass its `geom` property.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param triangle The Triangle to position the Game Objects within.
         */
        function RandomTriangle<G extends Phaser.GameObjects.GameObject[]>(items: G, triangle: Phaser.Geom.Triangle): G;

        /**
         * Takes an array of Game Objects, or any objects that have a public `rotation` property,
         * and then adds the given value to each of their `rotation` properties.
         * 
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         * 
         * To use this with a Group: `Rotate(group.getChildren(), value, step)`
         * @param items The array of items to be updated by this action.
         * @param value The amount to be added to the `rotation` property (in radians).
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function Rotate<G extends Phaser.GameObjects.GameObject[]>(items: G, value: number, step?: number, index?: number, direction?: number): G;

        /**
         * Rotates each item around the given point by the given angle.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param point Any object with public `x` and `y` properties.
         * @param angle The angle to rotate by, in radians.
         */
        function RotateAround<G extends Phaser.GameObjects.GameObject[]>(items: G, point: object, angle: number): G;

        /**
         * Rotates an array of Game Objects around a point by the given angle and distance.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param point Any object with public `x` and `y` properties.
         * @param angle The angle to rotate by, in radians.
         * @param distance The distance from the point of rotation in pixels.
         */
        function RotateAroundDistance<G extends Phaser.GameObjects.GameObject[]>(items: G, point: object, angle: number, distance: number): G;

        /**
         * Takes an array of Game Objects, or any objects that have a public `scaleX` property,
         * and then adds the given value to each of their `scaleX` properties.
         * 
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         * 
         * To use this with a Group: `ScaleX(group.getChildren(), value, step)`
         * @param items The array of items to be updated by this action.
         * @param value The amount to be added to the `scaleX` property.
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function ScaleX<G extends Phaser.GameObjects.GameObject[]>(items: G, value: number, step?: number, index?: number, direction?: number): G;

        /**
         * Takes an array of Game Objects, or any objects that have public `scaleX` and `scaleY` properties,
         * and then adds the given value to each of them.
         * 
         * The optional `stepX` and `stepY` properties are applied incrementally, multiplied by each item in the array.
         * 
         * To use this with a Group: `ScaleXY(group.getChildren(), scaleX, scaleY, stepX, stepY)`
         * @param items The array of items to be updated by this action.
         * @param scaleX The amount to be added to the `scaleX` property.
         * @param scaleY The amount to be added to the `scaleY` property. If `undefined` or `null` it uses the `scaleX` value.
         * @param stepX This is added to the `scaleX` amount, multiplied by the iteration counter. Default 0.
         * @param stepY This is added to the `scaleY` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function ScaleXY<G extends Phaser.GameObjects.GameObject[]>(items: G, scaleX: number, scaleY?: number, stepX?: number, stepY?: number, index?: number, direction?: number): G;

        /**
         * Takes an array of Game Objects, or any objects that have a public `scaleY` property,
         * and then adds the given value to each of their `scaleY` properties.
         * 
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         * 
         * To use this with a Group: `ScaleY(group.getChildren(), value, step)`
         * @param items The array of items to be updated by this action.
         * @param value The amount to be added to the `scaleY` property.
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function ScaleY<G extends Phaser.GameObjects.GameObject[]>(items: G, value: number, step?: number, index?: number, direction?: number): G;

        /**
         * Takes an array of Game Objects, or any objects that have the public property `alpha`
         * and then sets it to the given value.
         * 
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         * 
         * To use this with a Group: `SetAlpha(group.getChildren(), value, step)`
         * @param items The array of items to be updated by this action.
         * @param value The amount to set the property to.
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function SetAlpha<G extends Phaser.GameObjects.GameObject[]>(items: G, value: number, step?: number, index?: number, direction?: number): G;

        /**
         * Takes an array of Game Objects, or any objects that have the public property `blendMode`
         * and then sets it to the given value.
         * 
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         * 
         * To use this with a Group: `SetBlendMode(group.getChildren(), value)`
         * @param items The array of items to be updated by this action.
         * @param value The amount to set the property to.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function SetBlendMode<G extends Phaser.GameObjects.GameObject[]>(items: G, value: number, index?: number, direction?: number): G;

        /**
         * Takes an array of Game Objects, or any objects that have the public property `depth`
         * and then sets it to the given value.
         * 
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         * 
         * To use this with a Group: `SetDepth(group.getChildren(), value, step)`
         * @param items The array of items to be updated by this action.
         * @param value The amount to set the property to.
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function SetDepth<G extends Phaser.GameObjects.GameObject[]>(items: G, value: number, step?: number, index?: number, direction?: number): G;

        /**
         * Passes all provided Game Objects to the Input Manager to enable them for input with identical areas and callbacks.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param hitArea Either an input configuration object, or a geometric shape that defines the hit area for the Game Object. If not specified a Rectangle will be used.
         * @param hitAreaCallback A callback to be invoked when the Game Object is interacted with. If you provide a shape you must also provide a callback.
         */
        function SetHitArea<G extends Phaser.GameObjects.GameObject[]>(items: G, hitArea: any, hitAreaCallback: Phaser.Types.Input.HitAreaCallback): G;

        /**
         * Takes an array of Game Objects, or any objects that have the public properties `originX` and `originY`
         * and then sets them to the given values.
         * 
         * The optional `stepX` and `stepY` properties are applied incrementally, multiplied by each item in the array.
         * 
         * To use this with a Group: `SetOrigin(group.getChildren(), originX, originY, stepX, stepY)`
         * @param items The array of items to be updated by this action.
         * @param originX The amount to set the `originX` property to.
         * @param originY The amount to set the `originY` property to. If `undefined` or `null` it uses the `originX` value.
         * @param stepX This is added to the `originX` amount, multiplied by the iteration counter. Default 0.
         * @param stepY This is added to the `originY` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function SetOrigin<G extends Phaser.GameObjects.GameObject[]>(items: G, originX: number, originY?: number, stepX?: number, stepY?: number, index?: number, direction?: number): G;

        /**
         * Takes an array of Game Objects, or any objects that have the public property `rotation`
         * and then sets it to the given value.
         * 
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         * 
         * To use this with a Group: `SetRotation(group.getChildren(), value, step)`
         * @param items The array of items to be updated by this action.
         * @param value The amount to set the property to.
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function SetRotation<G extends Phaser.GameObjects.GameObject[]>(items: G, value: number, step?: number, index?: number, direction?: number): G;

        /**
         * Takes an array of Game Objects, or any objects that have the public properties `scaleX` and `scaleY`
         * and then sets them to the given values.
         * 
         * The optional `stepX` and `stepY` properties are applied incrementally, multiplied by each item in the array.
         * 
         * To use this with a Group: `SetScale(group.getChildren(), scaleX, scaleY, stepX, stepY)`
         * @param items The array of items to be updated by this action.
         * @param scaleX The amount to set the `scaleX` property to.
         * @param scaleY The amount to set the `scaleY` property to. If `undefined` or `null` it uses the `scaleX` value.
         * @param stepX This is added to the `scaleX` amount, multiplied by the iteration counter. Default 0.
         * @param stepY This is added to the `scaleY` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function SetScale<G extends Phaser.GameObjects.GameObject[]>(items: G, scaleX: number, scaleY?: number, stepX?: number, stepY?: number, index?: number, direction?: number): G;

        /**
         * Takes an array of Game Objects, or any objects that have the public property `scaleX`
         * and then sets it to the given value.
         * 
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         * 
         * To use this with a Group: `SetScaleX(group.getChildren(), value, step)`
         * @param items The array of items to be updated by this action.
         * @param value The amount to set the property to.
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function SetScaleX<G extends Phaser.GameObjects.GameObject[]>(items: G, value: number, step?: number, index?: number, direction?: number): G;

        /**
         * Takes an array of Game Objects, or any objects that have the public property `scaleY`
         * and then sets it to the given value.
         * 
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         * 
         * To use this with a Group: `SetScaleY(group.getChildren(), value, step)`
         * @param items The array of items to be updated by this action.
         * @param value The amount to set the property to.
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function SetScaleY<G extends Phaser.GameObjects.GameObject[]>(items: G, value: number, step?: number, index?: number, direction?: number): G;

        /**
         * Takes an array of Game Objects, or any objects that have the public properties `scrollFactorX` and `scrollFactorY`
         * and then sets them to the given values.
         * 
         * The optional `stepX` and `stepY` properties are applied incrementally, multiplied by each item in the array.
         * 
         * To use this with a Group: `SetScrollFactor(group.getChildren(), scrollFactorX, scrollFactorY, stepX, stepY)`
         * @param items The array of items to be updated by this action.
         * @param scrollFactorX The amount to set the `scrollFactorX` property to.
         * @param scrollFactorY The amount to set the `scrollFactorY` property to. If `undefined` or `null` it uses the `scrollFactorX` value.
         * @param stepX This is added to the `scrollFactorX` amount, multiplied by the iteration counter. Default 0.
         * @param stepY This is added to the `scrollFactorY` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function SetScrollFactor<G extends Phaser.GameObjects.GameObject[]>(items: G, scrollFactorX: number, scrollFactorY?: number, stepX?: number, stepY?: number, index?: number, direction?: number): G;

        /**
         * Takes an array of Game Objects, or any objects that have the public property `scrollFactorX`
         * and then sets it to the given value.
         * 
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         * 
         * To use this with a Group: `SetScrollFactorX(group.getChildren(), value, step)`
         * @param items The array of items to be updated by this action.
         * @param value The amount to set the property to.
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function SetScrollFactorX<G extends Phaser.GameObjects.GameObject[]>(items: G, value: number, step?: number, index?: number, direction?: number): G;

        /**
         * Takes an array of Game Objects, or any objects that have the public property `scrollFactorY`
         * and then sets it to the given value.
         * 
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         * 
         * To use this with a Group: `SetScrollFactorY(group.getChildren(), value, step)`
         * @param items The array of items to be updated by this action.
         * @param value The amount to set the property to.
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function SetScrollFactorY<G extends Phaser.GameObjects.GameObject[]>(items: G, value: number, step?: number, index?: number, direction?: number): G;

        /**
         * Takes an array of Game Objects, or any objects that have the public method setTint() and then updates it to the given value(s). You can specify tint color per corner or provide only one color value for `topLeft` parameter, in which case whole item will be tinted with that color.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param topLeft The tint being applied to top-left corner of item. If other parameters are given no value, this tint will be applied to whole item.
         * @param topRight The tint to be applied to top-right corner of item.
         * @param bottomLeft The tint to be applied to the bottom-left corner of item.
         * @param bottomRight The tint to be applied to the bottom-right corner of item.
         */
        function SetTint<G extends Phaser.GameObjects.GameObject[]>(items: G, topLeft: number, topRight?: number, bottomLeft?: number, bottomRight?: number): G;

        /**
         * Takes an array of Game Objects, or any objects that have the public property `visible`
         * and then sets it to the given value.
         * 
         * To use this with a Group: `SetVisible(group.getChildren(), value)`
         * @param items The array of items to be updated by this action.
         * @param value The value to set the property to.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function SetVisible<G extends Phaser.GameObjects.GameObject[]>(items: G, value: boolean, index?: number, direction?: number): G;

        /**
         * Takes an array of Game Objects, or any objects that have the public property `x`
         * and then sets it to the given value.
         * 
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         * 
         * To use this with a Group: `SetX(group.getChildren(), value, step)`
         * @param items The array of items to be updated by this action.
         * @param value The amount to set the property to.
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function SetX<G extends Phaser.GameObjects.GameObject[]>(items: G, value: number, step?: number, index?: number, direction?: number): G;

        /**
         * Takes an array of Game Objects, or any objects that have the public properties `x` and `y`
         * and then sets them to the given values.
         * 
         * The optional `stepX` and `stepY` properties are applied incrementally, multiplied by each item in the array.
         * 
         * To use this with a Group: `SetXY(group.getChildren(), x, y, stepX, stepY)`
         * @param items The array of items to be updated by this action.
         * @param x The amount to set the `x` property to.
         * @param y The amount to set the `y` property to. If `undefined` or `null` it uses the `x` value. Default x.
         * @param stepX This is added to the `x` amount, multiplied by the iteration counter. Default 0.
         * @param stepY This is added to the `y` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function SetXY<G extends Phaser.GameObjects.GameObject[]>(items: G, x: number, y?: number, stepX?: number, stepY?: number, index?: number, direction?: number): G;

        /**
         * Takes an array of Game Objects, or any objects that have the public property `y`
         * and then sets it to the given value.
         * 
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         * 
         * To use this with a Group: `SetY(group.getChildren(), value, step)`
         * @param items The array of items to be updated by this action.
         * @param value The amount to set the property to.
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function SetY<G extends Phaser.GameObjects.GameObject[]>(items: G, value: number, step?: number, index?: number, direction?: number): G;

        /**
         * Iterate through the items array changing the position of each element to be that of the element that came before
         * it in the array (or after it if direction = 1)
         * 
         * The first items position is set to x/y.
         * 
         * The final x/y coords are returned
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param x The x coordinate to place the first item in the array at.
         * @param y The y coordinate to place the first item in the array at.
         * @param direction The iteration direction. 0 = first to last and 1 = last to first. Default 0.
         * @param output An optional objec to store the final objects position in.
         */
        function ShiftPosition<G extends Phaser.GameObjects.GameObject[], O extends Phaser.Math.Vector2>(items: G, x: number, y: number, direction?: number, output?: O): O;

        /**
         * Shuffles the array in place. The shuffled array is both modified and returned.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         */
        function Shuffle<G extends Phaser.GameObjects.GameObject[]>(items: G): G;

        /**
         * Smootherstep is a sigmoid-like interpolation and clamping function.
         * 
         * The function depends on three parameters, the input x, the "left edge" and the "right edge", with the left edge being assumed smaller than the right edge. The function receives a real number x as an argument and returns 0 if x is less than or equal to the left edge, 1 if x is greater than or equal to the right edge, and smoothly interpolates, using a Hermite polynomial, between 0 and 1 otherwise. The slope of the smoothstep function is zero at both edges. This is convenient for creating a sequence of transitions using smoothstep to interpolate each segment as an alternative to using more sophisticated or expensive interpolation techniques.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param property The property of the Game Object to interpolate.
         * @param min The minimum interpolation value.
         * @param max The maximum interpolation value.
         * @param inc Should the values be incremented? `true` or set (`false`) Default false.
         */
        function SmootherStep<G extends Phaser.GameObjects.GameObject[]>(items: G, property: string, min: number, max: number, inc?: boolean): G;

        /**
         * Smoothstep is a sigmoid-like interpolation and clamping function.
         * 
         * The function depends on three parameters, the input x, the "left edge" and the "right edge", with the left edge being assumed smaller than the right edge. The function receives a real number x as an argument and returns 0 if x is less than or equal to the left edge, 1 if x is greater than or equal to the right edge, and smoothly interpolates, using a Hermite polynomial, between 0 and 1 otherwise. The slope of the smoothstep function is zero at both edges. This is convenient for creating a sequence of transitions using smoothstep to interpolate each segment as an alternative to using more sophisticated or expensive interpolation techniques.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param property The property of the Game Object to interpolate.
         * @param min The minimum interpolation value.
         * @param max The maximum interpolation value.
         * @param inc Should the values be incremented? `true` or set (`false`) Default false.
         */
        function SmoothStep<G extends Phaser.GameObjects.GameObject[]>(items: G, property: string, min: number, max: number, inc?: boolean): G;

        /**
         * Takes an array of Game Objects and then modifies their `property` so the value equals, or is incremented, by the
         * calculated spread value.
         * 
         * The spread value is derived from the given `min` and `max` values and the total number of items in the array.
         * 
         * For example, to cause an array of Sprites to change in alpha from 0 to 1 you could call:
         * 
         * ```javascript
         * Phaser.Actions.Spread(itemsArray, 'alpha', 0, 1);
         * ```
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param property The property of the Game Object to spread.
         * @param min The minimum value.
         * @param max The maximum value.
         * @param inc Should the values be incremented? `true` or set (`false`) Default false.
         */
        function Spread<G extends Phaser.GameObjects.GameObject[]>(items: G, property: string, min: number, max: number, inc?: boolean): G;

        /**
         * Takes an array of Game Objects and toggles the visibility of each one.
         * Those previously `visible = false` will become `visible = true`, and vice versa.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         */
        function ToggleVisible<G extends Phaser.GameObjects.GameObject[]>(items: G): G;

        /**
         * Wrap each item's coordinates within a rectangle's area.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param rect The rectangle.
         * @param padding An amount added to each side of the rectangle during the operation. Default 0.
         */
        function WrapInRectangle<G extends Phaser.GameObjects.GameObject[]>(items: G, rect: Phaser.Geom.Rectangle, padding?: number): G;

    }

    namespace Animations {
        /**
         * A Frame based Animation.
         * 
         * Animations in Phaser consist of a sequence of `AnimationFrame` objects, which are managed by
         * this class, along with properties that impact playback, such as the animations frame rate
         * or delay.
         * 
         * This class contains all of the properties and methods needed to handle playback of the animation
         * directly to an `AnimationState` instance, which is owned by a Sprite, or similar Game Object.
         * 
         * You don't typically create an instance of this class directly, but instead go via
         * either the `AnimationManager` or the `AnimationState` and use their `create` methods,
         * depending on if you need a global animation, or local to a specific Sprite.
         */
        class Animation {
            /**
             * 
             * @param manager A reference to the global Animation Manager
             * @param key The unique identifying string for this animation.
             * @param config The Animation configuration.
             */
            constructor(manager: Phaser.Animations.AnimationManager, key: string, config: Phaser.Types.Animations.Animation);

            /**
             * A reference to the global Animation Manager.
             */
            manager: Phaser.Animations.AnimationManager;

            /**
             * The unique identifying string for this animation.
             */
            key: string;

            /**
             * A frame based animation (as opposed to a bone based animation)
             */
            type: string;

            /**
             * Extract all the frame data into the frames array.
             */
            frames: Phaser.Animations.AnimationFrame[];

            /**
             * The frame rate of playback in frames per second (default 24 if duration is null)
             */
            frameRate: number;

            /**
             * How long the animation should play for, in milliseconds.
             * If the `frameRate` property has been set then it overrides this value,
             * otherwise the `frameRate` is derived from `duration`.
             */
            duration: number;

            /**
             * How many ms per frame, not including frame specific modifiers.
             */
            msPerFrame: number;

            /**
             * Skip frames if the time lags, or always advanced anyway?
             */
            skipMissedFrames: boolean;

            /**
             * The delay in ms before the playback will begin.
             */
            delay: number;

            /**
             * Number of times to repeat the animation. Set to -1 to repeat forever.
             */
            repeat: number;

            /**
             * The delay in ms before the a repeat play starts.
             */
            repeatDelay: number;

            /**
             * Should the animation yoyo (reverse back down to the start) before repeating?
             */
            yoyo: boolean;

            /**
             * Should the GameObject's `visible` property be set to `true` when the animation starts to play?
             */
            showOnStart: boolean;

            /**
             * Should the GameObject's `visible` property be set to `false` when the animation finishes?
             */
            hideOnComplete: boolean;

            /**
             * Global pause. All Game Objects using this Animation instance are impacted by this property.
             */
            paused: boolean;

            /**
             * Gets the total number of frames in this animation.
             */
            getTotalFrames(): number;

            /**
             * Calculates the duration, frame rate and msPerFrame values.
             * @param target The target to set the values on.
             * @param totalFrames The total number of frames in the animation.
             * @param duration The duration to calculate the frame rate from.
             * @param frameRate The frame ate to calculate the duration from.
             */
            calculateDuration(target: Phaser.Animations.Animation, totalFrames: number, duration: number, frameRate: number): void;

            /**
             * Add frames to the end of the animation.
             * @param config Either a string, in which case it will use all frames from a texture with the matching key, or an array of Animation Frame configuration objects.
             */
            addFrame(config: string | Phaser.Types.Animations.AnimationFrame[]): this;

            /**
             * Add frame/s into the animation.
             * @param index The index to insert the frame at within the animation.
             * @param config Either a string, in which case it will use all frames from a texture with the matching key, or an array of Animation Frame configuration objects.
             */
            addFrameAt(index: number, config: string | Phaser.Types.Animations.AnimationFrame[]): this;

            /**
             * Check if the given frame index is valid.
             * @param index The index to be checked.
             */
            checkFrame(index: number): boolean;

            /**
             * Called internally when this Animation first starts to play.
             * Sets the accumulator and nextTick properties.
             * @param state The Animation State belonging to the Game Object invoking this call.
             */
            protected getFirstTick(state: Phaser.Animations.AnimationState): void;

            /**
             * Returns the AnimationFrame at the provided index
             * @param index The index in the AnimationFrame array
             */
            protected getFrameAt(index: number): Phaser.Animations.AnimationFrame;

            /**
             * Creates AnimationFrame instances based on the given frame data.
             * @param textureManager A reference to the global Texture Manager.
             * @param frames Either a string, in which case it will use all frames from a texture with the matching key, or an array of Animation Frame configuration objects.
             * @param defaultTextureKey The key to use if no key is set in the frame configuration object.
             */
            getFrames(textureManager: Phaser.Textures.TextureManager, frames: string | Phaser.Types.Animations.AnimationFrame[], defaultTextureKey?: string): Phaser.Animations.AnimationFrame[];

            /**
             * Called internally. Sets the accumulator and nextTick values of the current Animation.
             * @param state The Animation State belonging to the Game Object invoking this call.
             */
            getNextTick(state: Phaser.Animations.AnimationState): void;

            /**
             * Returns the frame closest to the given progress value between 0 and 1.
             * @param value A value between 0 and 1.
             */
            getFrameByProgress(value: number): Phaser.Animations.AnimationFrame;

            /**
             * Advance the animation frame.
             * @param state The Animation State to advance.
             */
            nextFrame(state: Phaser.Animations.AnimationState): void;

            /**
             * Returns the animation last frame.
             */
            getLastFrame(): Phaser.Animations.AnimationFrame;

            /**
             * Called internally when the Animation is playing backwards.
             * Sets the previous frame, causing a yoyo, repeat, complete or update, accordingly.
             * @param state The Animation State belonging to the Game Object invoking this call.
             */
            previousFrame(state: Phaser.Animations.AnimationState): void;

            /**
             * Removes the given AnimationFrame from this Animation instance.
             * This is a global action. Any Game Object using this Animation will be impacted by this change.
             * @param frame The AnimationFrame to be removed.
             */
            removeFrame(frame: Phaser.Animations.AnimationFrame): this;

            /**
             * Removes a frame from the AnimationFrame array at the provided index
             * and updates the animation accordingly.
             * @param index The index in the AnimationFrame array
             */
            removeFrameAt(index: number): this;

            /**
             * Called internally during playback. Forces the animation to repeat, providing there are enough counts left
             * in the repeat counter.
             * @param state The Animation State belonging to the Game Object invoking this call.
             */
            repeatAnimation(state: Phaser.Animations.AnimationState): void;

            /**
             * Converts the animation data to JSON.
             */
            toJSON(): Phaser.Types.Animations.JSONAnimation;

            /**
             * Called internally whenever frames are added to, or removed from, this Animation.
             */
            updateFrameSequence(): this;

            /**
             * Pauses playback of this Animation. The paused state is set immediately.
             */
            pause(): this;

            /**
             * Resumes playback of this Animation. The paused state is reset immediately.
             */
            resume(): this;

            /**
             * Destroys this Animation instance. It will remove all event listeners,
             * remove this animation and its key from the global Animation Manager,
             * and then destroy all Animation Frames in turn.
             */
            destroy(): void;

        }

        /**
         * A single frame in an Animation sequence.
         * 
         * An AnimationFrame consists of a reference to the Texture it uses for rendering, references to other
         * frames in the animation, and index data. It also has the ability to modify the animation timing.
         * 
         * AnimationFrames are generated automatically by the Animation class.
         */
        class AnimationFrame {
            /**
             * 
             * @param textureKey The key of the Texture this AnimationFrame uses.
             * @param textureFrame The key of the Frame within the Texture that this AnimationFrame uses.
             * @param index The index of this AnimationFrame within the Animation sequence.
             * @param frame A reference to the Texture Frame this AnimationFrame uses for rendering.
             * @param isKeyFrame Is this Frame a Keyframe within the Animation? Default false.
             */
            constructor(textureKey: string, textureFrame: string | number, index: number, frame: Phaser.Textures.Frame, isKeyFrame?: boolean);

            /**
             * The key of the Texture this AnimationFrame uses.
             */
            textureKey: string;

            /**
             * The key of the Frame within the Texture that this AnimationFrame uses.
             */
            textureFrame: string | number;

            /**
             * The index of this AnimationFrame within the Animation sequence.
             */
            index: number;

            /**
             * A reference to the Texture Frame this AnimationFrame uses for rendering.
             */
            frame: Phaser.Textures.Frame;

            /**
             * Is this the first frame in an animation sequence?
             */
            readonly isFirst: boolean;

            /**
             * Is this the last frame in an animation sequence?
             */
            readonly isLast: boolean;

            /**
             * A reference to the AnimationFrame that comes before this one in the animation, if any.
             */
            readonly prevFrame: Phaser.Animations.AnimationFrame;

            /**
             * A reference to the AnimationFrame that comes after this one in the animation, if any.
             */
            readonly nextFrame: Phaser.Animations.AnimationFrame;

            /**
             * Additional time (in ms) that this frame should appear for during playback.
             * The value is added onto the msPerFrame set by the animation.
             */
            duration: number;

            /**
             * What % through the animation does this frame come?
             * This value is generated when the animation is created and cached here.
             */
            readonly progress: number;

            /**
             * Is this Frame a KeyFrame within the Animation?
             */
            isKeyFrame: boolean;

            /**
             * Generates a JavaScript object suitable for converting to JSON.
             */
            toJSON(): Phaser.Types.Animations.JSONAnimationFrame;

            /**
             * Destroys this object by removing references to external resources and callbacks.
             */
            destroy(): void;

        }

        /**
         * The Animation Manager.
         * 
         * Animations are managed by the global Animation Manager. This is a singleton class that is
         * responsible for creating and delivering animations and their corresponding data to all Game Objects.
         * Unlike plugins it is owned by the Game instance, not the Scene.
         * 
         * Sprites and other Game Objects get the data they need from the AnimationManager.
         */
        class AnimationManager extends Phaser.Events.EventEmitter {
            /**
             * 
             * @param game A reference to the Phaser.Game instance.
             */
            constructor(game: Phaser.Game);

            /**
             * A reference to the Phaser.Game instance.
             */
            protected game: Phaser.Game;

            /**
             * A reference to the Texture Manager.
             */
            protected textureManager: Phaser.Textures.TextureManager;

            /**
             * The global time scale of the Animation Manager.
             * 
             * This scales the time delta between two frames, thus influencing the speed of time for the Animation Manager.
             */
            globalTimeScale: number;

            /**
             * The Animations registered in the Animation Manager.
             * 
             * This map should be modified with the {@link #add} and {@link #create} methods of the Animation Manager.
             */
            protected anims: Phaser.Structs.Map<string, Phaser.Animations.Animation>;

            /**
             * A list of animation mix times.
             * 
             * See the {@link #setMix} method for more details.
             */
            mixes: Phaser.Structs.Map<string, Phaser.Animations.Animation>;

            /**
             * Whether the Animation Manager is paused along with all of its Animations.
             */
            paused: boolean;

            /**
             * The name of this Animation Manager.
             */
            name: string;

            /**
             * Registers event listeners after the Game boots.
             */
            boot(): void;

            /**
             * Adds a mix between two animations.
             * 
             * Mixing allows you to specify a unique delay between a pairing of animations.
             * 
             * When playing Animation A on a Game Object, if you then play Animation B, and a
             * mix exists, it will wait for the specified delay to be over before playing Animation B.
             * 
             * This allows you to customise smoothing between different types of animation, such
             * as blending between an idle and a walk state, or a running and a firing state.
             * 
             * Note that mixing is only applied if you use the `Sprite.play` method. If you opt to use
             * `playAfterRepeat` or `playAfterDelay` instead, those will take priority and the mix
             * delay will not be used.
             * 
             * To update an existing mix, just call this method with the new delay.
             * 
             * To remove a mix pairing, see the `removeMix` method.
             * @param animA The string-based key, or instance of, Animation A.
             * @param animB The string-based key, or instance of, Animation B.
             * @param delay The delay, in milliseconds, to wait when transitioning from Animation A to B.
             */
            addMix(animA: string | Phaser.Animations.Animation, animB: string | Phaser.Animations.Animation, delay: number): this;

            /**
             * Removes a mix between two animations.
             * 
             * Mixing allows you to specify a unique delay between a pairing of animations.
             * 
             * Calling this method lets you remove those pairings. You can either remove
             * it between `animA` and `animB`, or if you do not provide the `animB` parameter,
             * it will remove all `animA` mixes.
             * 
             * If you wish to update an existing mix instead, call the `addMix` method with the
             * new delay.
             * @param animA The string-based key, or instance of, Animation A.
             * @param animB The string-based key, or instance of, Animation B. If not given, all mixes for Animation A will be removed.
             */
            removeMix(animA: string | Phaser.Animations.Animation, animB?: string | Phaser.Animations.Animation): this;

            /**
             * Returns the mix delay between two animations.
             * 
             * If no mix has been set-up, this method will return zero.
             * 
             * If you wish to create, or update, a new mix, call the `addMix` method.
             * If you wish to remove a mix, call the `removeMix` method.
             * @param animA The string-based key, or instance of, Animation A.
             * @param animB The string-based key, or instance of, Animation B.
             */
            getMix(animA: string | Phaser.Animations.Animation, animB: string | Phaser.Animations.Animation): number;

            /**
             * Adds an existing Animation to the Animation Manager.
             * @param key The key under which the Animation should be added. The Animation will be updated with it. Must be unique.
             * @param animation The Animation which should be added to the Animation Manager.
             */
            add(key: string, animation: Phaser.Animations.Animation): this;

            /**
             * Checks to see if the given key is already in use within the Animation Manager or not.
             * 
             * Animations are global. Keys created in one scene can be used from any other Scene in your game. They are not Scene specific.
             * @param key The key of the Animation to check.
             */
            exists(key: string): boolean;

            /**
             * Create one, or more animations from a loaded Aseprite JSON file.
             * 
             * Aseprite is a powerful animated sprite editor and pixel art tool.
             * 
             * You can find more details at https://www.aseprite.org/
             * 
             * To export a compatible JSON file in Aseprite, please do the following:
             * 
             * 1. Go to "File - Export Sprite Sheet"
             * 
             * 2. On the **Layout** tab:
             * 2a. Set the "Sheet type" to "Packed"
             * 2b. Set the "Constraints" to "None"
             * 2c. Check the "Merge Duplicates" checkbox
             * 
             * 3. On the **Sprite** tab:
             * 3a. Set "Layers" to "Visible layers"
             * 3b. Set "Frames" to "All frames", unless you only wish to export a sub-set of tags
             * 
             * 4. On the **Borders** tab:
             * 4a. Check the "Trim Sprite" and "Trim Cells" options
             * 4b. Ensure "Border Padding", "Spacing" and "Inner Padding" are all > 0 (1 is usually enough)
             * 
             * 5. On the **Output** tab:
             * 5a. Check "Output File", give your image a name and make sure you choose "png files" as the file type
             * 5b. Check "JSON Data" and give your json file a name
             * 5c. The JSON Data type can be either a Hash or Array, Phaser doesn't mind.
             * 5d. Make sure "Tags" is checked in the Meta options
             * 5e. In the "Item Filename" input box, make sure it says just "{frame}" and nothing more.
             * 
             * 6. Click export
             * 
             * This was tested with Aseprite 1.2.25.
             * 
             * This will export a png and json file which you can load using the Aseprite Loader, i.e.:
             * 
             * ```javascript
             * function preload ()
             * {
             *     this.load.path = 'assets/animations/aseprite/';
             *     this.load.aseprite('paladin', 'paladin.png', 'paladin.json');
             * }
             * ```
             * 
             * Once loaded, you can call this method from within a Scene with the 'atlas' key:
             * 
             * ```javascript
             * this.anims.createFromAseprite('paladin');
             * ```
             * 
             * Any animations defined in the JSON will now be available to use in Phaser and you play them
             * via their Tag name. For example, if you have an animation called 'War Cry' on your Aseprite timeline,
             * you can play it in Phaser using that Tag name:
             * 
             * ```javascript
             * this.add.sprite(400, 300).play('War Cry');
             * ```
             * 
             * When calling this method you can optionally provide an array of tag names, and only those animations
             * will be created. For example:
             * 
             * ```javascript
             * this.anims.createFromAseprite('paladin', [ 'step', 'War Cry', 'Magnum Break' ]);
             * ```
             * 
             * This will only create the 3 animations defined. Note that the tag names are case-sensitive.
             * @param key The key of the loaded Aseprite atlas. It must have been loaded prior to calling this method.
             * @param tags An array of Tag names. If provided, only animations found in this array will be created.
             */
            createFromAseprite(key: string, tags?: string[]): Phaser.Animations.Animation[];

            /**
             * Creates a new Animation and adds it to the Animation Manager.
             * 
             * Animations are global. Once created, you can use them in any Scene in your game. They are not Scene specific.
             * 
             * If an invalid key is given this method will return `false`.
             * 
             * If you pass the key of an animation that already exists in the Animation Manager, that animation will be returned.
             * 
             * A brand new animation is only created if the key is valid and not already in use.
             * 
             * If you wish to re-use an existing key, call `AnimationManager.remove` first, then this method.
             * @param config The configuration settings for the Animation.
             */
            create(config: Phaser.Types.Animations.Animation): Phaser.Animations.Animation | false;

            /**
             * Loads this Animation Manager's Animations and settings from a JSON object.
             * @param data The JSON object to parse.
             * @param clearCurrentAnimations If set to `true`, the current animations will be removed (`anims.clear()`). If set to `false` (default), the animations in `data` will be added. Default false.
             */
            fromJSON(data: string | Phaser.Types.Animations.JSONAnimations | Phaser.Types.Animations.JSONAnimation, clearCurrentAnimations?: boolean): Phaser.Animations.Animation[];

            /**
             * Generate an array of {@link Phaser.Types.Animations.AnimationFrame} objects from a texture key and configuration object.
             * 
             * Generates objects with string based frame names, as configured by the given {@link Phaser.Types.Animations.GenerateFrameNames}.
             * 
             * It's a helper method, designed to make it easier for you to extract all of the frame names from texture atlases.
             * If you're working with a sprite sheet, see the `generateFrameNumbers` method instead.
             * 
             * Example:
             * 
             * If you have a texture atlases loaded called `gems` and it contains 6 frames called `ruby_0001`, `ruby_0002`, and so on,
             * then you can call this method using: `this.anims.generateFrameNames('gems', { prefix: 'ruby_', end: 6, zeroPad: 4 })`.
             * 
             * The `end` value tells it to look for 6 frames, incrementally numbered, all starting with the prefix `ruby_`. The `zeroPad`
             * value tells it how many zeroes pad out the numbers. To create an animation using this method, you can do:
             * 
             * ```javascript
             * this.anims.create({
             *   key: 'ruby',
             *   repeat: -1,
             *   frames: this.anims.generateFrameNames('gems', {
             *     prefix: 'ruby_',
             *     end: 6,
             *     zeroPad: 4
             *   })
             * });
             * ```
             * 
             * Please see the animation examples for further details.
             * @param key The key for the texture containing the animation frames.
             * @param config The configuration object for the animation frame names.
             */
            generateFrameNames(key: string, config?: Phaser.Types.Animations.GenerateFrameNames): Phaser.Types.Animations.AnimationFrame[];

            /**
             * Generate an array of {@link Phaser.Types.Animations.AnimationFrame} objects from a texture key and configuration object.
             * 
             * Generates objects with numbered frame names, as configured by the given {@link Phaser.Types.Animations.GenerateFrameNumbers}.
             * 
             * If you're working with a texture atlas, see the `generateFrameNames` method instead.
             * 
             * It's a helper method, designed to make it easier for you to extract frames from sprite sheets.
             * If you're working with a texture atlas, see the `generateFrameNames` method instead.
             * 
             * Example:
             * 
             * If you have a sprite sheet loaded called `explosion` and it contains 12 frames, then you can call this method using:
             * 
             * `this.anims.generateFrameNumbers('explosion', { start: 0, end: 11 })`.
             * 
             * The `end` value of 11 tells it to stop after the 12th frame has been added, because it started at zero.
             * 
             * To create an animation using this method, you can do:
             * 
             * ```javascript
             * this.anims.create({
             *   key: 'boom',
             *   frames: this.anims.generateFrameNames('explosion', {
             *     start: 0,
             *     end: 11
             *   })
             * });
             * ```
             * 
             * Note that `start` is optional and you don't need to include it if the animation starts from frame 0.
             * 
             * To specify an animation in reverse, swap the `start` and `end` values.
             * 
             * If the frames are not sequential, you may pass an array of frame numbers instead, for example:
             * 
             * `this.anims.generateFrameNumbers('explosion', { frames: [ 0, 1, 2, 1, 2, 3, 4, 0, 1, 2 ] })`
             * 
             * Please see the animation examples and `GenerateFrameNumbers` config docs for further details.
             * @param key The key for the texture containing the animation frames.
             * @param config The configuration object for the animation frames.
             */
            generateFrameNumbers(key: string, config: Phaser.Types.Animations.GenerateFrameNumbers): Phaser.Types.Animations.AnimationFrame[];

            /**
             * Get an Animation.
             * @param key The key of the Animation to retrieve.
             */
            get(key: string): Phaser.Animations.Animation;

            /**
             * Pause all animations.
             */
            pauseAll(): this;

            /**
             * Play an animation on the given Game Objects that have an Animation Component.
             * @param key The string-based key of the animation to play, or an Animation instance, or a `PlayAnimationConfig` object.
             * @param children An array of Game Objects to play the animation on. They must have an Animation Component.
             */
            play(key: string | Phaser.Animations.Animation | Phaser.Types.Animations.PlayAnimationConfig, children: Phaser.GameObjects.GameObject | Phaser.GameObjects.GameObject[]): this;

            /**
             * Takes an array of Game Objects that have an Animation Component and then
             * starts the given animation playing on them. The start time of each Game Object
             * is offset, incrementally, by the `stagger` amount.
             * 
             * For example, if you pass an array with 4 children and a stagger time of 1000,
             * the delays will be:
             * 
             * child 1: 1000ms delay
             * child 2: 2000ms delay
             * child 3: 3000ms delay
             * child 4: 4000ms delay
             * 
             * If you set the `staggerFirst` parameter to `false` they would be:
             * 
             * child 1: 0ms delay
             * child 2: 1000ms delay
             * child 3: 2000ms delay
             * child 4: 3000ms delay
             * 
             * You can also set `stagger` to be a negative value. If it was -1000, the above would be:
             * 
             * child 1: 3000ms delay
             * child 2: 2000ms delay
             * child 3: 1000ms delay
             * child 4: 0ms delay
             * @param key The string-based key of the animation to play, or an Animation instance, or a `PlayAnimationConfig` object.
             * @param children An array of Game Objects to play the animation on. They must have an Animation Component.
             * @param stagger The amount of time, in milliseconds, to offset each play time by. If a negative value is given, it's applied to the children in reverse order.
             * @param staggerFirst Should the first child be staggered as well? Default true.
             */
            staggerPlay<G extends Phaser.GameObjects.GameObject[]>(key: string | Phaser.Animations.Animation | Phaser.Types.Animations.PlayAnimationConfig, children: Phaser.GameObjects.GameObject | Phaser.GameObjects.GameObject[], stagger: number, staggerFirst?: boolean): G;

            /**
             * Removes an Animation from this Animation Manager, based on the given key.
             * 
             * This is a global action. Once an Animation has been removed, no Game Objects
             * can carry on using it.
             * @param key The key of the animation to remove.
             */
            remove(key: string): Phaser.Animations.Animation;

            /**
             * Resume all paused animations.
             */
            resumeAll(): this;

            /**
             * Returns the Animation data as JavaScript object based on the given key.
             * Or, if not key is defined, it will return the data of all animations as array of objects.
             * @param key The animation to get the JSONAnimation data from. If not provided, all animations are returned as an array.
             */
            toJSON(key?: string): Phaser.Types.Animations.JSONAnimations;

            /**
             * Destroy this Animation Manager and clean up animation definitions and references to other objects.
             * This method should not be called directly. It will be called automatically as a response to a `destroy` event from the Phaser.Game instance.
             */
            destroy(): void;

        }

        /**
         * The Animation State Component.
         * 
         * This component provides features to apply animations to Game Objects. It is responsible for
         * loading, queuing animations for later playback, mixing between animations and setting
         * the current animation frame to the Game Object that owns this component.
         * 
         * This component lives as an instance within any Game Object that has it defined, such as Sprites.
         * 
         * You can access its properties and methods via the `anims` property, i.e. `Sprite.anims`.
         * 
         * As well as playing animations stored in the global Animation Manager, this component
         * can also create animations that are stored locally within it. See the `create` method
         * for more details.
         * 
         * Prior to Phaser 3.50 this component was called just `Animation` and lived in the
         * `Phaser.GameObjects.Components` namespace. It was renamed to `AnimationState`
         * in 3.50 to help better identify its true purpose when browsing the documentation.
         */
        class AnimationState {
            /**
             * 
             * @param parent The Game Object to which this animation component belongs.
             */
            constructor(parent: Phaser.GameObjects.GameObject);

            /**
             * The Game Object to which this animation component belongs.
             * 
             * You can typically access this component from the Game Object
             * via the `this.anims` property.
             */
            parent: Phaser.GameObjects.GameObject;

            /**
             * A reference to the global Animation Manager.
             */
            animationManager: Phaser.Animations.AnimationManager;

            /**
             * A reference to the Texture Manager.
             */
            protected textureManager: Phaser.Textures.TextureManager;

            /**
             * The Animations stored locally in this Animation component.
             * 
             * Do not modify the contents of this Map directly, instead use the
             * `add`, `create` and `remove` methods of this class instead.
             */
            protected anims: Phaser.Structs.Map<string, Phaser.Animations.Animation>;

            /**
             * Is an animation currently playing or not?
             */
            isPlaying: boolean;

            /**
             * Has the current animation started playing, or is it waiting for a delay to expire?
             */
            hasStarted: boolean;

            /**
             * The current Animation loaded into this Animation component.
             * 
             * Will by `null` if no animation is yet loaded.
             */
            currentAnim: Phaser.Animations.Animation;

            /**
             * The current AnimationFrame being displayed by this Animation component.
             * 
             * Will by `null` if no animation is yet loaded.
             */
            currentFrame: Phaser.Animations.AnimationFrame;

            /**
             * The key, instance, or config of the next Animation to be loaded into this Animation component
             * when the current animation completes.
             * 
             * Will by `null` if no animation has been queued.
             */
            nextAnim: string | Phaser.Animations.Animation | Phaser.Types.Animations.PlayAnimationConfig;

            /**
             * A queue of Animations to be loaded into this Animation component when the current animation completes.
             * 
             * Populate this queue via the `chain` method.
             */
            nextAnimsQueue: any[];

            /**
             * The Time Scale factor.
             * 
             * You can adjust this value to modify the passage of time for the animation that is currently
             * playing. For example, setting it to 2 will make the animation play twice as fast. Or setting
             * it to 0.5 will slow the animation down.
             * 
             * You can change this value at run-time, or set it via the `PlayAnimationConfig`.
             * 
             * Prior to Phaser 3.50 this property was private and called `_timeScale`.
             */
            timeScale: number;

            /**
             * The frame rate of playback, of the current animation, in frames per second.
             * 
             * This value is set when a new animation is loaded into this component and should
             * be treated as read-only, as changing it once playback has started will not alter
             * the animation. To change the frame rate, provide a new value in the `PlayAnimationConfig` object.
             */
            frameRate: number;

            /**
             * The duration of the current animation, in milliseconds.
             * 
             * This value is set when a new animation is loaded into this component and should
             * be treated as read-only, as changing it once playback has started will not alter
             * the animation. To change the duration, provide a new value in the `PlayAnimationConfig` object.
             */
            duration: number;

            /**
             * The number of milliseconds per frame, not including frame specific modifiers that may be present in the
             * Animation data.
             * 
             * This value is calculated when a new animation is loaded into this component and should
             * be treated as read-only. Changing it will not alter playback speed.
             */
            msPerFrame: number;

            /**
             * Skip frames if the time lags, or always advanced anyway?
             */
            skipMissedFrames: boolean;

            /**
             * The delay before starting playback of the current animation, in milliseconds.
             * 
             * This value is set when a new animation is loaded into this component and should
             * be treated as read-only, as changing it once playback has started will not alter
             * the animation. To change the delay, provide a new value in the `PlayAnimationConfig` object.
             * 
             * Prior to Phaser 3.50 this property was private and called `_delay`.
             */
            delay: number;

            /**
             * The number of times to repeat playback of the current animation.
             * 
             * If -1, it means the animation will repeat forever.
             * 
             * This value is set when a new animation is loaded into this component and should
             * be treated as read-only, as changing it once playback has started will not alter
             * the animation. To change the number of repeats, provide a new value in the `PlayAnimationConfig` object.
             * 
             * Prior to Phaser 3.50 this property was private and called `_repeat`.
             */
            repeat: number;

            /**
             * The number of milliseconds to wait before starting the repeat playback of the current animation.
             * 
             * This value is set when a new animation is loaded into this component, but can also be modified
             * at run-time.
             * 
             * You can change the repeat delay by providing a new value in the `PlayAnimationConfig` object.
             * 
             * Prior to Phaser 3.50 this property was private and called `_repeatDelay`.
             */
            repeatDelay: number;

            /**
             * Should the current animation yoyo? An animation that yoyos will play in reverse, from the end
             * to the start, before then repeating or completing. An animation that does not yoyo will just
             * play from the start to the end.
             * 
             * This value is set when a new animation is loaded into this component, but can also be modified
             * at run-time.
             * 
             * You can change the yoyo by providing a new value in the `PlayAnimationConfig` object.
             * 
             * Prior to Phaser 3.50 this property was private and called `_yoyo`.
             */
            yoyo: boolean;

            /**
             * Should the GameObject's `visible` property be set to `true` when the animation starts to play?
             * 
             * This will happen _after_ any delay that may have been set.
             * 
             * This value is set when a new animation is loaded into this component, but can also be modified
             * at run-time, assuming the animation is currently delayed.
             */
            showOnStart: boolean;

            /**
             * Should the GameObject's `visible` property be set to `false` when the animation completes?
             * 
             * This value is set when a new animation is loaded into this component, but can also be modified
             * at run-time, assuming the animation is still actively playing.
             */
            hideOnComplete: boolean;

            /**
             * Is the playhead moving forwards (`true`) or in reverse (`false`) ?
             */
            forward: boolean;

            /**
             * An internal trigger that tells the component if it should plays the animation
             * in reverse mode ('true') or not ('false'). This is used because `forward` can
             * be changed by the `yoyo` feature.
             * 
             * Prior to Phaser 3.50 this property was private and called `_reverse`.
             */
            inReverse: boolean;

            /**
             * Internal time overflow accumulator.
             * 
             * This has the `delta` time added to it as part of the `update` step.
             */
            accumulator: number;

            /**
             * The time point at which the next animation frame will change.
             * 
             * This value is compared against the `accumulator` as part of the `update` step.
             */
            nextTick: number;

            /**
             * A counter keeping track of how much delay time, in milliseconds, is left before playback begins.
             * 
             * This is set via the `playAfterDelay` method, although it can be modified at run-time
             * if required, as long as the animation has not already started playing.
             */
            delayCounter: number;

            /**
             * A counter that keeps track of how many repeats are left to run.
             * 
             * This value is set when a new animation is loaded into this component, but can also be modified
             * at run-time.
             */
            repeatCounter: number;

            /**
             * An internal flag keeping track of pending repeats.
             */
            pendingRepeat: boolean;

            /**
             * Sets an animation, or an array of animations, to be played in the future, after the current one completes or stops.
             * 
             * The current animation must enter a 'completed' state for this to happen, i.e. finish all of its repeats, delays, etc,
             * or have one of the `stop` methods called.
             * 
             * An animation set to repeat forever will never enter a completed state unless stopped.
             * 
             * You can chain a new animation at any point, including before the current one starts playing, during it, or when it ends (via its `animationcomplete` event).
             * 
             * Chained animations are specific to a Game Object, meaning different Game Objects can have different chained animations without impacting the global animation they're playing.
             * 
             * Call this method with no arguments to reset all currently chained animations.
             * @param key The string-based key of the animation to play, or an Animation instance, or a `PlayAnimationConfig` object, or an array of them.
             */
            chain(key: string | Phaser.Animations.Animation | Phaser.Types.Animations.PlayAnimationConfig | string[] | Phaser.Animations.Animation[] | Phaser.Types.Animations.PlayAnimationConfig[]): Phaser.GameObjects.GameObject;

            /**
             * Returns the key of the animation currently loaded into this component.
             * 
             * Prior to Phaser 3.50 this method was called `getCurrentKey`.
             */
            getName(): string;

            /**
             * Returns the key of the animation frame currently displayed by this component.
             */
            getFrameName(): string;

            /**
             * Internal method used to load an animation into this component.
             * @param key The string-based key of the animation to play, or a `PlayAnimationConfig` object.
             */
            protected load(key: string | Phaser.Types.Animations.PlayAnimationConfig): Phaser.GameObjects.GameObject;

            /**
             * Pause the current animation and set the `isPlaying` property to `false`.
             * You can optionally pause it at a specific frame.
             * @param atFrame An optional frame to set after pausing the animation.
             */
            pause(atFrame?: Phaser.Animations.AnimationFrame): Phaser.GameObjects.GameObject;

            /**
             * Resumes playback of a paused animation and sets the `isPlaying` property to `true`.
             * You can optionally tell it to start playback from a specific frame.
             * @param fromFrame An optional frame to set before restarting playback.
             */
            resume(fromFrame?: Phaser.Animations.AnimationFrame): Phaser.GameObjects.GameObject;

            /**
             * Waits for the specified delay, in milliseconds, then starts playback of the given animation.
             * 
             * If the animation _also_ has a delay value set in its config, it will be **added** to the delay given here.
             * 
             * If an animation is already running and a new animation is given to this method, it will wait for
             * the given delay before starting the new animation.
             * 
             * If no animation is currently running, the given one begins after the delay.
             * 
             * Prior to Phaser 3.50 this method was called 'delayedPlay' and the parameters were in the reverse order.
             * @param key The string-based key of the animation to play, or an Animation instance, or a `PlayAnimationConfig` object.
             * @param delay The delay, in milliseconds, to wait before starting the animation playing.
             */
            playAfterDelay(key: string | Phaser.Animations.Animation | Phaser.Types.Animations.PlayAnimationConfig, delay: number): Phaser.GameObjects.GameObject;

            /**
             * Waits for the current animation to complete the `repeatCount` number of repeat cycles, then starts playback
             * of the given animation.
             * 
             * You can use this to ensure there are no harsh jumps between two sets of animations, i.e. going from an
             * idle animation to a walking animation, by making them blend smoothly into each other.
             * 
             * If no animation is currently running, the given one will start immediately.
             * @param key The string-based key of the animation to play, or an Animation instance, or a `PlayAnimationConfig` object.
             * @param repeatCount How many times should the animation repeat before the next one starts? Default 1.
             */
            playAfterRepeat(key: string | Phaser.Animations.Animation | Phaser.Types.Animations.PlayAnimationConfig, repeatCount?: number): Phaser.GameObjects.GameObject;

            /**
             * Start playing the given animation on this Sprite.
             * 
             * Animations in Phaser can either belong to the global Animation Manager, or specifically to this Sprite.
             * 
             * The benefit of a global animation is that multiple Sprites can all play the same animation, without
             * having to duplicate the data. You can just create it once and then play it on any Sprite.
             * 
             * The following code shows how to create a global repeating animation. The animation will be created
             * from all of the frames within the sprite sheet that was loaded with the key 'muybridge':
             * 
             * ```javascript
             * var config = {
             *     key: 'run',
             *     frames: 'muybridge',
             *     frameRate: 15,
             *     repeat: -1
             * };
             * 
             * //  This code should be run from within a Scene:
             * this.anims.create(config);
             * ```
             * 
             * However, if you wish to create an animation that is unique to this Sprite, and this Sprite alone,
             * you can call the `Animation.create` method instead. It accepts the exact same parameters as when
             * creating a global animation, however the resulting data is kept locally in this Sprite.
             * 
             * With the animation created, either globally or locally, you can now play it on this Sprite:
             * 
             * ```javascript
             * this.add.sprite(x, y).play('run');
             * ```
             * 
             * Alternatively, if you wish to run it at a different frame rate, for example, you can pass a config
             * object instead:
             * 
             * ```javascript
             * this.add.sprite(x, y).play({ key: 'run', frameRate: 24 });
             * ```
             * 
             * When playing an animation on a Sprite it will first check to see if it can find a matching key
             * locally within the Sprite. If it can, it will play the local animation. If not, it will then
             * search the global Animation Manager and look for it there.
             * 
             * If you need a Sprite to be able to play both local and global animations, make sure they don't
             * have conflicting keys.
             * 
             * See the documentation for the `PlayAnimationConfig` config object for more details about this.
             * 
             * Also, see the documentation in the Animation Manager for further details on creating animations.
             * @param key The string-based key of the animation to play, or an Animation instance, or a `PlayAnimationConfig` object.
             * @param ignoreIfPlaying If this animation is already playing then ignore this call. Default false.
             */
            play(key: string | Phaser.Animations.Animation | Phaser.Types.Animations.PlayAnimationConfig, ignoreIfPlaying?: boolean): Phaser.GameObjects.GameObject;

            /**
             * Start playing the given animation on this Sprite, in reverse.
             * 
             * Animations in Phaser can either belong to the global Animation Manager, or specifically to this Sprite.
             * 
             * The benefit of a global animation is that multiple Sprites can all play the same animation, without
             * having to duplicate the data. You can just create it once and then play it on any Sprite.
             * 
             * The following code shows how to create a global repeating animation. The animation will be created
             * from all of the frames within the sprite sheet that was loaded with the key 'muybridge':
             * 
             * ```javascript
             * var config = {
             *     key: 'run',
             *     frames: 'muybridge',
             *     frameRate: 15,
             *     repeat: -1
             * };
             * 
             * //  This code should be run from within a Scene:
             * this.anims.create(config);
             * ```
             * 
             * However, if you wish to create an animation that is unique to this Sprite, and this Sprite alone,
             * you can call the `Animation.create` method instead. It accepts the exact same parameters as when
             * creating a global animation, however the resulting data is kept locally in this Sprite.
             * 
             * With the animation created, either globally or locally, you can now play it on this Sprite:
             * 
             * ```javascript
             * this.add.sprite(x, y).playReverse('run');
             * ```
             * 
             * Alternatively, if you wish to run it at a different frame rate, for example, you can pass a config
             * object instead:
             * 
             * ```javascript
             * this.add.sprite(x, y).playReverse({ key: 'run', frameRate: 24 });
             * ```
             * 
             * When playing an animation on a Sprite it will first check to see if it can find a matching key
             * locally within the Sprite. If it can, it will play the local animation. If not, it will then
             * search the global Animation Manager and look for it there.
             * 
             * If you need a Sprite to be able to play both local and global animations, make sure they don't
             * have conflicting keys.
             * 
             * See the documentation for the `PlayAnimationConfig` config object for more details about this.
             * 
             * Also, see the documentation in the Animation Manager for further details on creating animations.
             * @param key The string-based key of the animation to play, or an Animation instance, or a `PlayAnimationConfig` object.
             * @param ignoreIfPlaying If an animation is already playing then ignore this call. Default false.
             */
            playReverse(key: string | Phaser.Animations.Animation | Phaser.Types.Animations.PlayAnimationConfig, ignoreIfPlaying?: boolean): Phaser.GameObjects.GameObject;

            /**
             * Load the animation based on the key and set-up all of the internal values
             * needed for playback to start. If there is no delay, it will also fire the start events.
             * @param key The string-based key of the animation to play, or a `PlayAnimationConfig` object.
             */
            startAnimation(key: string | Phaser.Types.Animations.PlayAnimationConfig): Phaser.GameObjects.GameObject;

            /**
             * Reverse the Animation that is already playing on the Game Object.
             */
            reverse(): Phaser.GameObjects.GameObject;

            /**
             * Returns a value between 0 and 1 indicating how far this animation is through, ignoring repeats and yoyos.
             * 
             * The value is based on the current frame and how far that is in the animation, it is not based on
             * the duration of the animation.
             */
            getProgress(): number;

            /**
             * Takes a value between 0 and 1 and uses it to set how far this animation is through playback.
             * 
             * Does not factor in repeats or yoyos, but does handle playing forwards or backwards.
             * 
             * The value is based on the current frame and how far that is in the animation, it is not based on
             * the duration of the animation.
             * @param value The progress value, between 0 and 1. Default 0.
             */
            setProgress(value?: number): Phaser.GameObjects.GameObject;

            /**
             * Sets the number of times that the animation should repeat after its first play through.
             * For example, if repeat is 1, the animation will play a total of twice: the initial play plus 1 repeat.
             * 
             * To repeat indefinitely, use -1.
             * The value should always be an integer.
             * 
             * Calling this method only works if the animation is already running. Otherwise, any
             * value specified here will be overwritten when the next animation loads in. To avoid this,
             * use the `repeat` property of the `PlayAnimationConfig` object instead.
             * @param value The number of times that the animation should repeat.
             */
            setRepeat(value: number): Phaser.GameObjects.GameObject;

            /**
             * Handle the removal of an animation from the Animation Manager.
             * @param key The key of the removed Animation.
             * @param animation The removed Animation.
             */
            globalRemove(key?: string, animation?: Phaser.Animations.Animation): void;

            /**
             * Restarts the current animation from its beginning.
             * 
             * You can optionally reset the delay and repeat counters as well.
             * 
             * Calling this will fire the `ANIMATION_RESTART` event immediately.
             * 
             * If you `includeDelay` then it will also fire the `ANIMATION_START` event once
             * the delay has expired, otherwise, playback will just begin immediately.
             * @param includeDelay Whether to include the delay value of the animation when restarting. Default false.
             * @param resetRepeats Whether to reset the repeat counter or not? Default false.
             */
            restart(includeDelay?: boolean, resetRepeats?: boolean): Phaser.GameObjects.GameObject;

            /**
             * The current animation has completed. This dispatches the `ANIMATION_COMPLETE` event.
             * 
             * This method is called by the Animation instance and should not usually be invoked directly.
             * 
             * If no animation is loaded, no events will be dispatched.
             * 
             * If another animation has been queued for playback, it will be started after the events fire.
             */
            complete(): Phaser.GameObjects.GameObject;

            /**
             * Immediately stops the current animation from playing and dispatches the `ANIMATION_STOP` event.
             * 
             * If no animation is running, no events will be dispatched.
             * 
             * If there is another animation in the queue (set via the `chain` method) then it will start playing.
             */
            stop(): Phaser.GameObjects.GameObject;

            /**
             * Stops the current animation from playing after the specified time delay, given in milliseconds.
             * 
             * It then dispatches the `ANIMATION_STOP` event.
             * 
             * If no animation is running, no events will be dispatched.
             * 
             * If there is another animation in the queue (set via the `chain` method) then it will start playing,
             * when the current one stops.
             * @param delay The number of milliseconds to wait before stopping this animation.
             */
            stopAfterDelay(delay: number): Phaser.GameObjects.GameObject;

            /**
             * Stops the current animation from playing when it next repeats.
             * 
             * It then dispatches the `ANIMATION_STOP` event.
             * 
             * If no animation is running, no events will be dispatched.
             * 
             * If there is another animation in the queue (set via the `chain` method) then it will start playing,
             * when the current one stops.
             * 
             * Prior to Phaser 3.50 this method was called `stopOnRepeat` and had no parameters.
             * @param repeatCount How many times should the animation repeat before stopping? Default 1.
             */
            stopAfterRepeat(repeatCount?: number): Phaser.GameObjects.GameObject;

            /**
             * Stops the current animation from playing when it next sets the given frame.
             * If this frame doesn't exist within the animation it will not stop it from playing.
             * 
             * It then dispatches the `ANIMATION_STOP` event.
             * 
             * If no animation is running, no events will be dispatched.
             * 
             * If there is another animation in the queue (set via the `chain` method) then it will start playing,
             * when the current one stops.
             * @param frame The frame to check before stopping this animation.
             */
            stopOnFrame(frame: Phaser.Animations.AnimationFrame): Phaser.GameObjects.GameObject;

            /**
             * Returns the total number of frames in this animation, or returns zero if no
             * animation has been loaded.
             */
            getTotalFrames(): number;

            /**
             * The internal update loop for the AnimationState Component.
             * 
             * This is called automatically by the `Sprite.preUpdate` method.
             * @param time The current timestamp.
             * @param delta The delta time, in ms, elapsed since the last frame.
             */
            update(time: number, delta: number): void;

            /**
             * Sets the given Animation Frame as being the current frame
             * and applies it to the parent Game Object, adjusting size and origin as needed.
             * @param animationFrame The animation frame to change to.
             */
            setCurrentFrame(animationFrame: Phaser.Animations.AnimationFrame): Phaser.GameObjects.GameObject;

            /**
             * Advances the animation to the next frame, regardless of the time or animation state.
             * If the animation is set to repeat, or yoyo, this will still take effect.
             * 
             * Calling this does not change the direction of the animation. I.e. if it was currently
             * playing in reverse, calling this method doesn't then change the direction to forwards.
             */
            nextFrame(): Phaser.GameObjects.GameObject;

            /**
             * Advances the animation to the previous frame, regardless of the time or animation state.
             * If the animation is set to repeat, or yoyo, this will still take effect.
             * 
             * Calling this does not change the direction of the animation. I.e. if it was currently
             * playing in forwards, calling this method doesn't then change the direction to backwards.
             */
            previousFrame(): Phaser.GameObjects.GameObject;

            /**
             * Get an Animation instance that has been created locally on this Sprite.
             * 
             * See the `create` method for more details.
             * @param key The key of the Animation to retrieve.
             */
            get(key: string): Phaser.Animations.Animation;

            /**
             * Checks to see if the given key is already used locally within the animations stored on this Sprite.
             * @param key The key of the Animation to check.
             */
            exists(key: string): boolean;

            /**
             * Creates a new Animation that is local specifically to this Sprite.
             * 
             * When a Sprite owns an animation, it is kept out of the global Animation Manager, which means
             * you're free to use keys that may be already defined there. Unless you specifically need a Sprite
             * to have a unique animation, you should favor using global animations instead, as they allow for
             * the same animation to be used across multiple Sprites, saving on memory. However, if this Sprite
             * is the only one to use this animation, it's sensible to create it here.
             * 
             * If an invalid key is given this method will return `false`.
             * 
             * If you pass the key of an animation that already exists locally, that animation will be returned.
             * 
             * A brand new animation is only created if the key is valid and not already in use by this Sprite.
             * 
             * If you wish to re-use an existing key, call the `remove` method first, then this method.
             * @param config The configuration settings for the Animation.
             */
            create(config: Phaser.Types.Animations.Animation): Phaser.Animations.Animation | false;

            /**
             * Generate an array of {@link Phaser.Types.Animations.AnimationFrame} objects from a texture key and configuration object.
             * 
             * Generates objects with string based frame names, as configured by the given {@link Phaser.Types.Animations.GenerateFrameNames}.
             * 
             * It's a helper method, designed to make it easier for you to extract all of the frame names from texture atlases.
             * If you're working with a sprite sheet, see the `generateFrameNumbers` method instead.
             * 
             * Example:
             * 
             * If you have a texture atlases loaded called `gems` and it contains 6 frames called `ruby_0001`, `ruby_0002`, and so on,
             * then you can call this method using: `this.anims.generateFrameNames('gems', { prefix: 'ruby_', end: 6, zeroPad: 4 })`.
             * 
             * The `end` value tells it to look for 6 frames, incrementally numbered, all starting with the prefix `ruby_`. The `zeroPad`
             * value tells it how many zeroes pad out the numbers. To create an animation using this method, you can do:
             * 
             * ```javascript
             * this.anims.create({
             *   key: 'ruby',
             *   repeat: -1,
             *   frames: this.anims.generateFrameNames('gems', {
             *     prefix: 'ruby_',
             *     end: 6,
             *     zeroPad: 4
             *   })
             * });
             * ```
             * 
             * Please see the animation examples for further details.
             * @param key The key for the texture containing the animation frames.
             * @param config The configuration object for the animation frame names.
             */
            generateFrameNames(key: string, config?: Phaser.Types.Animations.GenerateFrameNames): Phaser.Types.Animations.AnimationFrame[];

            /**
             * Generate an array of {@link Phaser.Types.Animations.AnimationFrame} objects from a texture key and configuration object.
             * 
             * Generates objects with numbered frame names, as configured by the given {@link Phaser.Types.Animations.GenerateFrameNumbers}.
             * 
             * If you're working with a texture atlas, see the `generateFrameNames` method instead.
             * 
             * It's a helper method, designed to make it easier for you to extract frames from sprite sheets.
             * If you're working with a texture atlas, see the `generateFrameNames` method instead.
             * 
             * Example:
             * 
             * If you have a sprite sheet loaded called `explosion` and it contains 12 frames, then you can call this method using:
             * `this.anims.generateFrameNumbers('explosion', { start: 0, end: 12 })`.
             * 
             * The `end` value tells it to stop after 12 frames. To create an animation using this method, you can do:
             * 
             * ```javascript
             * this.anims.create({
             *   key: 'boom',
             *   frames: this.anims.generateFrameNames('explosion', {
             *     start: 0,
             *     end: 12
             *   })
             * });
             * ```
             * 
             * Note that `start` is optional and you don't need to include it if the animation starts from frame 0.
             * 
             * To specify an animation in reverse, swap the `start` and `end` values.
             * 
             * If the frames are not sequential, you may pass an array of frame numbers instead, for example:
             * 
             * `this.anims.generateFrameNumbers('explosion', { frames: [ 0, 1, 2, 1, 2, 3, 4, 0, 1, 2 ] })`
             * 
             * Please see the animation examples and `GenerateFrameNumbers` config docs for further details.
             * @param key The key for the texture containing the animation frames.
             * @param config The configuration object for the animation frames.
             */
            generateFrameNumbers(key: string, config: Phaser.Types.Animations.GenerateFrameNumbers): Phaser.Types.Animations.AnimationFrame[];

            /**
             * Removes a locally created Animation from this Sprite, based on the given key.
             * 
             * Once an Animation has been removed, this Sprite cannot play it again without re-creating it.
             * @param key The key of the animation to remove.
             */
            remove(key: string): Phaser.Animations.Animation;

            /**
             * Destroy this Animation component.
             * 
             * Unregisters event listeners and cleans up its references.
             */
            destroy(): void;

            /**
             * `true` if the current animation is paused, otherwise `false`.
             */
            readonly isPaused: boolean;

        }

        namespace Events {
            /**
             * The Add Animation Event.
             * 
             * This event is dispatched when a new animation is added to the global Animation Manager.
             * 
             * This can happen either as a result of an animation instance being added to the Animation Manager,
             * or the Animation Manager creating a new animation directly.
             */
            const ADD_ANIMATION: any;

            /**
             * The Animation Complete Event.
             * 
             * This event is dispatched by a Sprite when an animation playing on it completes playback.
             * This happens when the animation gets to the end of its sequence, factoring in any delays
             * or repeats it may have to process.
             * 
             * An animation that is set to loop, or repeat forever, will never fire this event, because
             * it never actually completes. If you need to handle this, listen for the `ANIMATION_STOP`
             * event instead, as this is emitted when the animation is stopped directly.
             * 
             * Listen for it on the Sprite using `sprite.on('animationcomplete', listener)`
             * 
             * The animation event flow is as follows:
             * 
             * 1. `ANIMATION_START`
             * 2. `ANIMATION_UPDATE` (repeated for however many frames the animation has)
             * 3. `ANIMATION_REPEAT` (only if the animation is set to repeat, it then emits more update events after this)
             * 4. `ANIMATION_COMPLETE` (only if there is a finite, or zero, repeat count)
             * 5. `ANIMATION_COMPLETE_KEY` (only if there is a finite, or zero, repeat count)
             * 
             * If the animation is stopped directly, the `ANIMATION_STOP` event is dispatched instead of `ANIMATION_COMPLETE`.
             * 
             * If the animation is restarted while it is already playing, `ANIMATION_RESTART` is emitted.
             */
            const ANIMATION_COMPLETE: any;

            /**
             * The Animation Complete Dynamic Key Event.
             * 
             * This event is dispatched by a Sprite when an animation playing on it completes playback.
             * This happens when the animation gets to the end of its sequence, factoring in any delays
             * or repeats it may have to process.
             * 
             * An animation that is set to loop, or repeat forever, will never fire this event, because
             * it never actually completes. If you need to handle this, listen for the `ANIMATION_STOP`
             * event instead, as this is emitted when the animation is stopped directly.
             * 
             * The difference between this and the `ANIMATION_COMPLETE` event is that this one has a
             * dynamic event name that contains the name of the animation within it. For example,
             * if you had an animation called `explode` you could listen for the completion of that
             * specific animation by using: `sprite.on('animationcomplete-explode', listener)`. Or, if you
             * wish to use types: `sprite.on(Phaser.Animations.Events.ANIMATION_COMPLETE_KEY + 'explode', listener)`.
             * 
             * The animation event flow is as follows:
             * 
             * 1. `ANIMATION_START`
             * 2. `ANIMATION_UPDATE` (repeated for however many frames the animation has)
             * 3. `ANIMATION_REPEAT` (only if the animation is set to repeat, it then emits more update events after this)
             * 4. `ANIMATION_COMPLETE` (only if there is a finite, or zero, repeat count)
             * 5. `ANIMATION_COMPLETE_KEY` (only if there is a finite, or zero, repeat count)
             * 
             * If the animation is stopped directly, the `ANIMATION_STOP` event is dispatched instead of `ANIMATION_COMPLETE`.
             * 
             * If the animation is restarted while it is already playing, `ANIMATION_RESTART` is emitted.
             */
            const ANIMATION_COMPLETE_KEY: any;

            /**
             * The Animation Repeat Event.
             * 
             * This event is dispatched by a Sprite when an animation repeats playing on it.
             * This happens if the animation was created, or played, with a `repeat` value specified.
             * 
             * An animation will repeat when it reaches the end of its sequence.
             * 
             * Listen for it on the Sprite using `sprite.on('animationrepeat', listener)`
             * 
             * The animation event flow is as follows:
             * 
             * 1. `ANIMATION_START`
             * 2. `ANIMATION_UPDATE` (repeated for however many frames the animation has)
             * 3. `ANIMATION_REPEAT` (only if the animation is set to repeat, it then emits more update events after this)
             * 4. `ANIMATION_COMPLETE` (only if there is a finite, or zero, repeat count)
             * 5. `ANIMATION_COMPLETE_KEY` (only if there is a finite, or zero, repeat count)
             * 
             * If the animation is stopped directly, the `ANIMATION_STOP` event is dispatched instead of `ANIMATION_COMPLETE`.
             * 
             * If the animation is restarted while it is already playing, `ANIMATION_RESTART` is emitted.
             */
            const ANIMATION_REPEAT: any;

            /**
             * The Animation Restart Event.
             * 
             * This event is dispatched by a Sprite when an animation restarts playing on it.
             * This only happens when the `Sprite.anims.restart` method is called.
             * 
             * Listen for it on the Sprite using `sprite.on('animationrestart', listener)`
             * 
             * The animation event flow is as follows:
             * 
             * 1. `ANIMATION_START`
             * 2. `ANIMATION_UPDATE` (repeated for however many frames the animation has)
             * 3. `ANIMATION_REPEAT` (only if the animation is set to repeat, it then emits more update events after this)
             * 4. `ANIMATION_COMPLETE` (only if there is a finite, or zero, repeat count)
             * 5. `ANIMATION_COMPLETE_KEY` (only if there is a finite, or zero, repeat count)
             * 
             * If the animation is stopped directly, the `ANIMATION_STOP` event is dispatched instead of `ANIMATION_COMPLETE`.
             * 
             * If the animation is restarted while it is already playing, `ANIMATION_RESTART` is emitted.
             */
            const ANIMATION_RESTART: any;

            /**
             * The Animation Start Event.
             * 
             * This event is dispatched by a Sprite when an animation starts playing on it.
             * This happens when the animation is played, factoring in any delay that may have been specified.
             * This event happens after the delay has expired and prior to the first update event.
             * 
             * Listen for it on the Sprite using `sprite.on('animationstart', listener)`
             * 
             * The animation event flow is as follows:
             * 
             * 1. `ANIMATION_START`
             * 2. `ANIMATION_UPDATE` (repeated for however many frames the animation has)
             * 3. `ANIMATION_REPEAT` (only if the animation is set to repeat, it then emits more update events after this)
             * 4. `ANIMATION_COMPLETE` (only if there is a finite, or zero, repeat count)
             * 5. `ANIMATION_COMPLETE_KEY` (only if there is a finite, or zero, repeat count)
             * 
             * If the animation is stopped directly, the `ANIMATION_STOP` event is dispatched instead of `ANIMATION_COMPLETE`.
             * 
             * If the animation is restarted while it is already playing, `ANIMATION_RESTART` is emitted.
             */
            const ANIMATION_START: any;

            /**
             * The Animation Stop Event.
             * 
             * This event is dispatched by a Sprite when an animation is stopped on it. An animation
             * will only be stopeed if a method such as `Sprite.stop` or `Sprite.anims.stopAfterDelay`
             * is called. It can also be emitted if a new animation is started before the current one completes.
             * 
             * Listen for it on the Sprite using `sprite.on('animationstop', listener)`
             * 
             * The animation event flow is as follows:
             * 
             * 1. `ANIMATION_START`
             * 2. `ANIMATION_UPDATE` (repeated for however many frames the animation has)
             * 3. `ANIMATION_REPEAT` (only if the animation is set to repeat, it then emits more update events after this)
             * 4. `ANIMATION_COMPLETE` (only if there is a finite, or zero, repeat count)
             * 5. `ANIMATION_COMPLETE_KEY` (only if there is a finite, or zero, repeat count)
             * 
             * If the animation is stopped directly, the `ANIMATION_STOP` event is dispatched instead of `ANIMATION_COMPLETE`.
             * 
             * If the animation is restarted while it is already playing, `ANIMATION_RESTART` is emitted.
             */
            const ANIMATION_STOP: any;

            /**
             * The Animation Update Event.
             * 
             * This event is dispatched by a Sprite when an animation playing on it updates. This happens when the animation changes frame.
             * An animation will change frame based on the frme rate and other factors like `timeScale` and `delay`. It can also change
             * frame when stopped or restarted.
             * 
             * Listen for it on the Sprite using `sprite.on('animationupdate', listener)`
             * 
             * If an animation is playing faster than the game frame-rate can handle, it's entirely possible for it to emit several
             * update events in a single game frame, so please be aware of this in your code. The **final** event received that frame
             * is the one that is rendered to the game.
             * 
             * The animation event flow is as follows:
             * 
             * 1. `ANIMATION_START`
             * 2. `ANIMATION_UPDATE` (repeated for however many frames the animation has)
             * 3. `ANIMATION_REPEAT` (only if the animation is set to repeat, it then emits more update events after this)
             * 4. `ANIMATION_COMPLETE` (only if there is a finite, or zero, repeat count)
             * 5. `ANIMATION_COMPLETE_KEY` (only if there is a finite, or zero, repeat count)
             * 
             * If the animation is stopped directly, the `ANIMATION_STOP` event is dispatched instead of `ANIMATION_COMPLETE`.
             * 
             * If the animation is restarted while it is already playing, `ANIMATION_RESTART` is emitted.
             */
            const ANIMATION_UPDATE: any;

            /**
             * The Pause All Animations Event.
             * 
             * This event is dispatched when the global Animation Manager is told to pause.
             * 
             * When this happens all current animations will stop updating, although it doesn't necessarily mean
             * that the game has paused as well.
             */
            const PAUSE_ALL: any;

            /**
             * The Remove Animation Event.
             * 
             * This event is dispatched when an animation is removed from the global Animation Manager.
             */
            const REMOVE_ANIMATION: any;

            /**
             * The Resume All Animations Event.
             * 
             * This event is dispatched when the global Animation Manager resumes, having been previously paused.
             * 
             * When this happens all current animations will continue updating again.
             */
            const RESUME_ALL: any;

        }

    }

    namespace Cache {
        /**
         * The BaseCache is a base Cache class that can be used for storing references to any kind of data.
         * 
         * Data can be added, retrieved and removed based on the given keys.
         * 
         * Keys are string-based.
         */
        class BaseCache {
            /**
             * The Map in which the cache objects are stored.
             * 
             * You can query the Map directly or use the BaseCache methods.
             */
            entries: Phaser.Structs.Map<String, any>;

            /**
             * An instance of EventEmitter used by the cache to emit related events.
             */
            events: Phaser.Events.EventEmitter;

            /**
             * Adds an item to this cache. The item is referenced by a unique string, which you are responsible
             * for setting and keeping track of. The item can only be retrieved by using this string.
             * @param key The unique key by which the data added to the cache will be referenced.
             * @param data The data to be stored in the cache.
             */
            add(key: string, data: any): this;

            /**
             * Checks if this cache contains an item matching the given key.
             * This performs the same action as `BaseCache.exists`.
             * @param key The unique key of the item to be checked in this cache.
             */
            has(key: string): boolean;

            /**
             * Checks if this cache contains an item matching the given key.
             * This performs the same action as `BaseCache.has` and is called directly by the Loader.
             * @param key The unique key of the item to be checked in this cache.
             */
            exists(key: string): boolean;

            /**
             * Gets an item from this cache based on the given key.
             * @param key The unique key of the item to be retrieved from this cache.
             */
            get(key: string): any;

            /**
             * Removes and item from this cache based on the given key.
             * 
             * If an entry matching the key is found it is removed from the cache and a `remove` event emitted.
             * No additional checks are done on the item removed. If other systems or parts of your game code
             * are relying on this item, it is up to you to sever those relationships prior to removing the item.
             * @param key The unique key of the item to remove from the cache.
             */
            remove(key: string): this;

            /**
             * Returns all keys in use in this cache.
             */
            getKeys(): string[];

            /**
             * Destroys this cache and all items within it.
             */
            destroy(): void;

        }

        /**
         * The Cache Manager is the global cache owned and maintained by the Game instance.
         * 
         * Various systems, such as the file Loader, rely on this cache in order to store the files
         * it has loaded. The manager itself doesn't store any files, but instead owns multiple BaseCache
         * instances, one per type of file. You can also add your own custom caches.
         */
        class CacheManager {
            /**
             * 
             * @param game A reference to the Phaser.Game instance that owns this CacheManager.
             */
            constructor(game: Phaser.Game);

            /**
             * A reference to the Phaser.Game instance that owns this CacheManager.
             */
            protected game: Phaser.Game;

            /**
             * A Cache storing all binary files, typically added via the Loader.
             */
            binary: Phaser.Cache.BaseCache;

            /**
             * A Cache storing all bitmap font data files, typically added via the Loader.
             * Only the font data is stored in this cache, the textures are part of the Texture Manager.
             */
            bitmapFont: Phaser.Cache.BaseCache;

            /**
             * A Cache storing all JSON data files, typically added via the Loader.
             */
            json: Phaser.Cache.BaseCache;

            /**
             * A Cache storing all physics data files, typically added via the Loader.
             */
            physics: Phaser.Cache.BaseCache;

            /**
             * A Cache storing all shader source files, typically added via the Loader.
             */
            shader: Phaser.Cache.BaseCache;

            /**
             * A Cache storing all non-streaming audio files, typically added via the Loader.
             */
            audio: Phaser.Cache.BaseCache;

            /**
             * A Cache storing all non-streaming video files, typically added via the Loader.
             */
            video: Phaser.Cache.BaseCache;

            /**
             * A Cache storing all text files, typically added via the Loader.
             */
            text: Phaser.Cache.BaseCache;

            /**
             * A Cache storing all html files, typically added via the Loader.
             */
            html: Phaser.Cache.BaseCache;

            /**
             * A Cache storing all WaveFront OBJ files, typically added via the Loader.
             */
            obj: Phaser.Cache.BaseCache;

            /**
             * A Cache storing all tilemap data files, typically added via the Loader.
             * Only the data is stored in this cache, the textures are part of the Texture Manager.
             */
            tilemap: Phaser.Cache.BaseCache;

            /**
             * A Cache storing all xml data files, typically added via the Loader.
             */
            xml: Phaser.Cache.BaseCache;

            /**
             * An object that contains your own custom BaseCache entries.
             * Add to this via the `addCustom` method.
             */
            custom: {[key: string]: Phaser.Cache.BaseCache};

            /**
             * Add your own custom Cache for storing your own files.
             * The cache will be available under `Cache.custom.key`.
             * The cache will only be created if the key is not already in use.
             * @param key The unique key of your custom cache.
             */
            addCustom(key: string): Phaser.Cache.BaseCache;

            /**
             * Removes all entries from all BaseCaches and destroys all custom caches.
             */
            destroy(): void;

        }

        namespace Events {
            /**
             * The Cache Add Event.
             * 
             * This event is dispatched by any Cache that extends the BaseCache each time a new object is added to it.
             */
            const ADD: any;

            /**
             * The Cache Remove Event.
             * 
             * This event is dispatched by any Cache that extends the BaseCache each time an object is removed from it.
             */
            const REMOVE: any;

        }

    }

    namespace Cameras {
        namespace Scene2D {
            /**
             * A Base Camera class.
             * 
             * The Camera is the way in which all games are rendered in Phaser. They provide a view into your game world,
             * and can be positioned, rotated, zoomed and scrolled accordingly.
             * 
             * A Camera consists of two elements: The viewport and the scroll values.
             * 
             * The viewport is the physical position and size of the Camera within your game. Cameras, by default, are
             * created the same size as your game, but their position and size can be set to anything. This means if you
             * wanted to create a camera that was 320x200 in size, positioned in the bottom-right corner of your game,
             * you'd adjust the viewport to do that (using methods like `setViewport` and `setSize`).
             * 
             * If you wish to change where the Camera is looking in your game, then you scroll it. You can do this
             * via the properties `scrollX` and `scrollY` or the method `setScroll`. Scrolling has no impact on the
             * viewport, and changing the viewport has no impact on the scrolling.
             * 
             * By default a Camera will render all Game Objects it can see. You can change this using the `ignore` method,
             * allowing you to filter Game Objects out on a per-Camera basis.
             * 
             * The Base Camera is extended by the Camera class, which adds in special effects including Fade,
             * Flash and Camera Shake, as well as the ability to follow Game Objects.
             * 
             * The Base Camera was introduced in Phaser 3.12. It was split off from the Camera class, to allow
             * you to isolate special effects as needed. Therefore the 'since' values for properties of this class relate
             * to when they were added to the Camera class.
             */
            class BaseCamera extends Phaser.Events.EventEmitter implements Phaser.GameObjects.Components.Alpha, Phaser.GameObjects.Components.Visible {
                /**
                 * 
                 * @param x The x position of the Camera, relative to the top-left of the game canvas.
                 * @param y The y position of the Camera, relative to the top-left of the game canvas.
                 * @param width The width of the Camera, in pixels.
                 * @param height The height of the Camera, in pixels.
                 */
                constructor(x: number, y: number, width: number, height: number);

                /**
                 * A reference to the Scene this camera belongs to.
                 */
                scene: Phaser.Scene;

                /**
                 * A reference to the Game Scene Manager.
                 */
                sceneManager: Phaser.Scenes.SceneManager;

                /**
                 * A reference to the Game Scale Manager.
                 */
                scaleManager: Phaser.Scale.ScaleManager;

                /**
                 * A reference to the Scene's Camera Manager to which this Camera belongs.
                 */
                cameraManager: Phaser.Cameras.Scene2D.CameraManager;

                /**
                 * The Camera ID. Assigned by the Camera Manager and used to handle camera exclusion.
                 * This value is a bitmask.
                 */
                readonly id: number;

                /**
                 * The name of the Camera. This is left empty for your own use.
                 */
                name: string;

                /**
                 * Should this camera round its pixel values to integers?
                 */
                roundPixels: boolean;

                /**
                 * Is this Camera visible or not?
                 * 
                 * A visible camera will render and perform input tests.
                 * An invisible camera will not render anything and will skip input tests.
                 */
                visible: boolean;

                /**
                 * Is this Camera using a bounds to restrict scrolling movement?
                 * 
                 * Set this property along with the bounds via `Camera.setBounds`.
                 */
                useBounds: boolean;

                /**
                 * The World View is a Rectangle that defines the area of the 'world' the Camera is currently looking at.
                 * This factors in the Camera viewport size, zoom and scroll position and is updated in the Camera preRender step.
                 * If you have enabled Camera bounds the worldview will be clamped to those bounds accordingly.
                 * You can use it for culling or intersection checks.
                 */
                readonly worldView: Phaser.Geom.Rectangle;

                /**
                 * Is this Camera dirty?
                 * 
                 * A dirty Camera has had either its viewport size, bounds, scroll, rotation or zoom levels changed since the last frame.
                 * 
                 * This flag is cleared during the `postRenderCamera` method of the renderer.
                 */
                dirty: boolean;

                /**
                 * Does this Camera have a transparent background?
                 */
                transparent: boolean;

                /**
                 * The background color of this Camera. Only used if `transparent` is `false`.
                 */
                backgroundColor: Phaser.Display.Color;

                /**
                 * The Camera alpha value. Setting this property impacts every single object that this Camera
                 * renders. You can either set the property directly, i.e. via a Tween, to fade a Camera in or out,
                 * or via the chainable `setAlpha` method instead.
                 */
                alpha: number;

                /**
                 * Should the camera cull Game Objects before checking them for input hit tests?
                 * In some special cases it may be beneficial to disable this.
                 */
                disableCull: boolean;

                /**
                 * The mid-point of the Camera in 'world' coordinates.
                 * 
                 * Use it to obtain exactly where in the world the center of the camera is currently looking.
                 * 
                 * This value is updated in the preRender method, after the scroll values and follower
                 * have been processed.
                 */
                readonly midPoint: Phaser.Math.Vector2;

                /**
                 * The horizontal origin of rotation for this Camera.
                 * 
                 * By default the camera rotates around the center of the viewport.
                 * 
                 * Changing the origin allows you to adjust the point in the viewport from which rotation happens.
                 * A value of 0 would rotate from the top-left of the viewport. A value of 1 from the bottom right.
                 * 
                 * See `setOrigin` to set both origins in a single, chainable call.
                 */
                originX: number;

                /**
                 * The vertical origin of rotation for this Camera.
                 * 
                 * By default the camera rotates around the center of the viewport.
                 * 
                 * Changing the origin allows you to adjust the point in the viewport from which rotation happens.
                 * A value of 0 would rotate from the top-left of the viewport. A value of 1 from the bottom right.
                 * 
                 * See `setOrigin` to set both origins in a single, chainable call.
                 */
                originY: number;

                /**
                 * The Mask this Camera is using during render.
                 * Set the mask using the `setMask` method. Remove the mask using the `clearMask` method.
                 */
                mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask;

                /**
                 * This array is populated with all of the Game Objects that this Camera has rendered
                 * in the previous (or current, depending on when you inspect it) frame.
                 * 
                 * It is cleared at the start of `Camera.preUpdate`, or if the Camera is destroyed.
                 * 
                 * You should not modify this array as it is used internally by the input system,
                 * however you can read it as required. Note that Game Objects may appear in this
                 * list multiple times if they belong to multiple non-exclusive Containers.
                 */
                renderList: Phaser.GameObjects.GameObject[];

                /**
                 * Adds the given Game Object to this cameras render list.
                 * 
                 * This is invoked during the rendering stage. Only objects that are actually rendered
                 * will appear in the render list.
                 * @param child The Game Object to add to the render list.
                 */
                addToRenderList(child: Phaser.GameObjects.GameObject): void;

                /**
                 * Set the Alpha level of this Camera. The alpha controls the opacity of the Camera as it renders.
                 * Alpha values are provided as a float between 0, fully transparent, and 1, fully opaque.
                 * @param value The Camera alpha value. Default 1.
                 */
                setAlpha(value?: number): this;

                /**
                 * Sets the rotation origin of this Camera.
                 * 
                 * The values are given in the range 0 to 1 and are only used when calculating Camera rotation.
                 * 
                 * By default the camera rotates around the center of the viewport.
                 * 
                 * Changing the origin allows you to adjust the point in the viewport from which rotation happens.
                 * A value of 0 would rotate from the top-left of the viewport. A value of 1 from the bottom right.
                 * @param x The horizontal origin value. Default 0.5.
                 * @param y The vertical origin value. If not defined it will be set to the value of `x`. Default x.
                 */
                setOrigin(x?: number, y?: number): this;

                /**
                 * Calculates what the Camera.scrollX and scrollY values would need to be in order to move
                 * the Camera so it is centered on the given x and y coordinates, without actually moving
                 * the Camera there. The results are clamped based on the Camera bounds, if set.
                 * @param x The horizontal coordinate to center on.
                 * @param y The vertical coordinate to center on.
                 * @param out A Vector2 to store the values in. If not given a new Vector2 is created.
                 */
                getScroll(x: number, y: number, out?: Phaser.Math.Vector2): Phaser.Math.Vector2;

                /**
                 * Moves the Camera horizontally so that it is centered on the given x coordinate, bounds allowing.
                 * Calling this does not change the scrollY value.
                 * @param x The horizontal coordinate to center on.
                 */
                centerOnX(x: number): this;

                /**
                 * Moves the Camera vertically so that it is centered on the given y coordinate, bounds allowing.
                 * Calling this does not change the scrollX value.
                 * @param y The vertical coordinate to center on.
                 */
                centerOnY(y: number): this;

                /**
                 * Moves the Camera so that it is centered on the given coordinates, bounds allowing.
                 * @param x The horizontal coordinate to center on.
                 * @param y The vertical coordinate to center on.
                 */
                centerOn(x: number, y: number): this;

                /**
                 * Moves the Camera so that it is looking at the center of the Camera Bounds, if enabled.
                 */
                centerToBounds(): this;

                /**
                 * Moves the Camera so that it is re-centered based on its viewport size.
                 */
                centerToSize(): this;

                /**
                 * Takes an array of Game Objects and returns a new array featuring only those objects
                 * visible by this camera.
                 * @param renderableObjects An array of Game Objects to cull.
                 */
                cull<G extends Phaser.GameObjects.GameObject[]>(renderableObjects: G): G;

                /**
                 * Converts the given `x` and `y` coordinates into World space, based on this Cameras transform.
                 * You can optionally provide a Vector2, or similar object, to store the results in.
                 * @param x The x position to convert to world space.
                 * @param y The y position to convert to world space.
                 * @param output An optional object to store the results in. If not provided a new Vector2 will be created.
                 */
                getWorldPoint<O extends Phaser.Math.Vector2>(x: number, y: number, output?: O): O;

                /**
                 * Given a Game Object, or an array of Game Objects, it will update all of their camera filter settings
                 * so that they are ignored by this Camera. This means they will not be rendered by this Camera.
                 * @param entries The Game Object, or array of Game Objects, to be ignored by this Camera.
                 */
                ignore(entries: Phaser.GameObjects.GameObject | Phaser.GameObjects.GameObject[] | Phaser.GameObjects.Group): this;

                /**
                 * Internal preRender step.
                 */
                protected preRender(): void;

                /**
                 * Takes an x value and checks it's within the range of the Camera bounds, adjusting if required.
                 * Do not call this method if you are not using camera bounds.
                 * @param x The value to horizontally scroll clamp.
                 */
                clampX(x: number): number;

                /**
                 * Takes a y value and checks it's within the range of the Camera bounds, adjusting if required.
                 * Do not call this method if you are not using camera bounds.
                 * @param y The value to vertically scroll clamp.
                 */
                clampY(y: number): number;

                /**
                 * If this Camera has previously had movement bounds set on it, this will remove them.
                 */
                removeBounds(): this;

                /**
                 * Set the rotation of this Camera. This causes everything it renders to appear rotated.
                 * 
                 * Rotating a camera does not rotate the viewport itself, it is applied during rendering.
                 * @param value The cameras angle of rotation, given in degrees. Default 0.
                 */
                setAngle(value?: number): this;

                /**
                 * Sets the background color for this Camera.
                 * 
                 * By default a Camera has a transparent background but it can be given a solid color, with any level
                 * of transparency, via this method.
                 * 
                 * The color value can be specified using CSS color notation, hex or numbers.
                 * @param color The color value. In CSS, hex or numeric color notation. Default 'rgba(0,0,0,0)'.
                 */
                setBackgroundColor(color?: string | number | Phaser.Types.Display.InputColorObject): this;

                /**
                 * Set the bounds of the Camera. The bounds are an axis-aligned rectangle.
                 * 
                 * The Camera bounds controls where the Camera can scroll to, stopping it from scrolling off the
                 * edges and into blank space. It does not limit the placement of Game Objects, or where
                 * the Camera viewport can be positioned.
                 * 
                 * Temporarily disable the bounds by changing the boolean `Camera.useBounds`.
                 * 
                 * Clear the bounds entirely by calling `Camera.removeBounds`.
                 * 
                 * If you set bounds that are smaller than the viewport it will stop the Camera from being
                 * able to scroll. The bounds can be positioned where-ever you wish. By default they are from
                 * 0x0 to the canvas width x height. This means that the coordinate 0x0 is the top left of
                 * the Camera bounds. However, you can position them anywhere. So if you wanted a game world
                 * that was 2048x2048 in size, with 0x0 being the center of it, you can set the bounds x/y
                 * to be -1024, -1024, with a width and height of 2048. Depending on your game you may find
                 * it easier for 0x0 to be the top-left of the bounds, or you may wish 0x0 to be the middle.
                 * @param x The top-left x coordinate of the bounds.
                 * @param y The top-left y coordinate of the bounds.
                 * @param width The width of the bounds, in pixels.
                 * @param height The height of the bounds, in pixels.
                 * @param centerOn If `true` the Camera will automatically be centered on the new bounds. Default false.
                 */
                setBounds(x: number, y: number, width: number, height: number, centerOn?: boolean): this;

                /**
                 * Returns a rectangle containing the bounds of the Camera.
                 * 
                 * If the Camera does not have any bounds the rectangle will be empty.
                 * 
                 * The rectangle is a copy of the bounds, so is safe to modify.
                 * @param out An optional Rectangle to store the bounds in. If not given, a new Rectangle will be created.
                 */
                getBounds(out?: Phaser.Geom.Rectangle): Phaser.Geom.Rectangle;

                /**
                 * Sets the name of this Camera.
                 * This value is for your own use and isn't used internally.
                 * @param value The name of the Camera. Default ''.
                 */
                setName(value?: string): this;

                /**
                 * Set the position of the Camera viewport within the game.
                 * 
                 * This does not change where the camera is 'looking'. See `setScroll` to control that.
                 * @param x The top-left x coordinate of the Camera viewport.
                 * @param y The top-left y coordinate of the Camera viewport. Default x.
                 */
                setPosition(x: number, y?: number): this;

                /**
                 * Set the rotation of this Camera. This causes everything it renders to appear rotated.
                 * 
                 * Rotating a camera does not rotate the viewport itself, it is applied during rendering.
                 * @param value The rotation of the Camera, in radians. Default 0.
                 */
                setRotation(value?: number): this;

                /**
                 * Should the Camera round pixel values to whole integers when rendering Game Objects?
                 * 
                 * In some types of game, especially with pixel art, this is required to prevent sub-pixel aliasing.
                 * @param value `true` to round Camera pixels, `false` to not.
                 */
                setRoundPixels(value: boolean): this;

                /**
                 * Sets the Scene the Camera is bound to.
                 * @param scene The Scene the camera is bound to.
                 */
                setScene(scene: Phaser.Scene): this;

                /**
                 * Set the position of where the Camera is looking within the game.
                 * You can also modify the properties `Camera.scrollX` and `Camera.scrollY` directly.
                 * Use this method, or the scroll properties, to move your camera around the game world.
                 * 
                 * This does not change where the camera viewport is placed. See `setPosition` to control that.
                 * @param x The x coordinate of the Camera in the game world.
                 * @param y The y coordinate of the Camera in the game world. Default x.
                 */
                setScroll(x: number, y?: number): this;

                /**
                 * Set the size of the Camera viewport.
                 * 
                 * By default a Camera is the same size as the game, but can be made smaller via this method,
                 * allowing you to create mini-cam style effects by creating and positioning a smaller Camera
                 * viewport within your game.
                 * @param width The width of the Camera viewport.
                 * @param height The height of the Camera viewport. Default width.
                 */
                setSize(width: number, height?: number): this;

                /**
                 * This method sets the position and size of the Camera viewport in a single call.
                 * 
                 * If you're trying to change where the Camera is looking at in your game, then see
                 * the method `Camera.setScroll` instead. This method is for changing the viewport
                 * itself, not what the camera can see.
                 * 
                 * By default a Camera is the same size as the game, but can be made smaller via this method,
                 * allowing you to create mini-cam style effects by creating and positioning a smaller Camera
                 * viewport within your game.
                 * @param x The top-left x coordinate of the Camera viewport.
                 * @param y The top-left y coordinate of the Camera viewport.
                 * @param width The width of the Camera viewport.
                 * @param height The height of the Camera viewport. Default width.
                 */
                setViewport(x: number, y: number, width: number, height?: number): this;

                /**
                 * Set the zoom value of the Camera.
                 * 
                 * Changing to a smaller value, such as 0.5, will cause the camera to 'zoom out'.
                 * Changing to a larger value, such as 2, will cause the camera to 'zoom in'.
                 * 
                 * A value of 1 means 'no zoom' and is the default.
                 * 
                 * Changing the zoom does not impact the Camera viewport in any way, it is only applied during rendering.
                 * 
                 * As of Phaser 3.50 you can now set the horizontal and vertical zoom values independently.
                 * @param x The horizontal zoom value of the Camera. The minimum it can be is 0.001. Default 1.
                 * @param y The vertical zoom value of the Camera. The minimum it can be is 0.001. Default x.
                 */
                setZoom(x?: number, y?: number): this;

                /**
                 * Sets the mask to be applied to this Camera during rendering.
                 * 
                 * The mask must have been previously created and can be either a GeometryMask or a BitmapMask.
                 * 
                 * Bitmap Masks only work on WebGL. Geometry Masks work on both WebGL and Canvas.
                 * 
                 * If a mask is already set on this Camera it will be immediately replaced.
                 * 
                 * Masks have no impact on physics or input detection. They are purely a rendering component
                 * that allows you to limit what is visible during the render pass.
                 * @param mask The mask this Camera will use when rendering.
                 * @param fixedPosition Should the mask translate along with the Camera, or be fixed in place and not impacted by the Cameras transform? Default true.
                 */
                setMask(mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask, fixedPosition?: boolean): this;

                /**
                 * Clears the mask that this Camera was using.
                 * @param destroyMask Destroy the mask before clearing it? Default false.
                 */
                clearMask(destroyMask?: boolean): this;

                /**
                 * Sets the visibility of this Camera.
                 * 
                 * An invisible Camera will skip rendering and input tests of everything it can see.
                 * @param value The visible state of the Camera.
                 */
                setVisible(value: boolean): this;

                /**
                 * Returns an Object suitable for JSON storage containing all of the Camera viewport and rendering properties.
                 */
                toJSON(): Phaser.Types.Cameras.Scene2D.JSONCamera;

                /**
                 * Internal method called automatically by the Camera Manager.
                 * @param time The current timestamp as generated by the Request Animation Frame or SetTimeout.
                 * @param delta The delta time, in ms, elapsed since the last frame.
                 */
                protected update(time: number, delta: number): void;

                /**
                 * Destroys this Camera instance and its internal properties and references.
                 * Once destroyed you cannot use this Camera again, even if re-added to a Camera Manager.
                 * 
                 * This method is called automatically by `CameraManager.remove` if that methods `runDestroy` argument is `true`, which is the default.
                 * 
                 * Unless you have a specific reason otherwise, always use `CameraManager.remove` and allow it to handle the camera destruction,
                 * rather than calling this method directly.
                 */
                destroy(): void;

                /**
                 * The x position of the Camera viewport, relative to the top-left of the game canvas.
                 * The viewport is the area into which the camera renders.
                 * To adjust the position the camera is looking at in the game world, see the `scrollX` value.
                 */
                x: number;

                /**
                 * The y position of the Camera viewport, relative to the top-left of the game canvas.
                 * The viewport is the area into which the camera renders.
                 * To adjust the position the camera is looking at in the game world, see the `scrollY` value.
                 */
                y: number;

                /**
                 * The width of the Camera viewport, in pixels.
                 * 
                 * The viewport is the area into which the Camera renders. Setting the viewport does
                 * not restrict where the Camera can scroll to.
                 */
                width: number;

                /**
                 * The height of the Camera viewport, in pixels.
                 * 
                 * The viewport is the area into which the Camera renders. Setting the viewport does
                 * not restrict where the Camera can scroll to.
                 */
                height: number;

                /**
                 * The horizontal scroll position of this Camera.
                 * 
                 * Change this value to cause the Camera to scroll around your Scene.
                 * 
                 * Alternatively, setting the Camera to follow a Game Object, via the `startFollow` method,
                 * will automatically adjust the Camera scroll values accordingly.
                 * 
                 * You can set the bounds within which the Camera can scroll via the `setBounds` method.
                 */
                scrollX: number;

                /**
                 * The vertical scroll position of this Camera.
                 * 
                 * Change this value to cause the Camera to scroll around your Scene.
                 * 
                 * Alternatively, setting the Camera to follow a Game Object, via the `startFollow` method,
                 * will automatically adjust the Camera scroll values accordingly.
                 * 
                 * You can set the bounds within which the Camera can scroll via the `setBounds` method.
                 */
                scrollY: number;

                /**
                 * The Camera zoom value. Change this value to zoom in, or out of, a Scene.
                 * 
                 * A value of 0.5 would zoom the Camera out, so you can now see twice as much
                 * of the Scene as before. A value of 2 would zoom the Camera in, so every pixel
                 * now takes up 2 pixels when rendered.
                 * 
                 * Set to 1 to return to the default zoom level.
                 * 
                 * Be careful to never set this value to zero.
                 */
                zoom: number;

                /**
                 * The Camera horizontal zoom value. Change this value to zoom in, or out of, a Scene.
                 * 
                 * A value of 0.5 would zoom the Camera out, so you can now see twice as much
                 * of the Scene as before. A value of 2 would zoom the Camera in, so every pixel
                 * now takes up 2 pixels when rendered.
                 * 
                 * Set to 1 to return to the default zoom level.
                 * 
                 * Be careful to never set this value to zero.
                 */
                zoomX: number;

                /**
                 * The Camera vertical zoom value. Change this value to zoom in, or out of, a Scene.
                 * 
                 * A value of 0.5 would zoom the Camera out, so you can now see twice as much
                 * of the Scene as before. A value of 2 would zoom the Camera in, so every pixel
                 * now takes up 2 pixels when rendered.
                 * 
                 * Set to 1 to return to the default zoom level.
                 * 
                 * Be careful to never set this value to zero.
                 */
                zoomY: number;

                /**
                 * The horizontal position of the center of the Camera's viewport, relative to the left of the game canvas.
                 */
                readonly centerX: number;

                /**
                 * The vertical position of the center of the Camera's viewport, relative to the top of the game canvas.
                 */
                readonly centerY: number;

                /**
                 * The displayed width of the camera viewport, factoring in the camera zoom level.
                 * 
                 * If a camera has a viewport width of 800 and a zoom of 0.5 then its display width
                 * would be 1600, as it's displaying twice as many pixels as zoom level 1.
                 * 
                 * Equally, a camera with a width of 800 and zoom of 2 would have a display width
                 * of 400 pixels.
                 */
                readonly displayWidth: number;

                /**
                 * The displayed height of the camera viewport, factoring in the camera zoom level.
                 * 
                 * If a camera has a viewport height of 600 and a zoom of 0.5 then its display height
                 * would be 1200, as it's displaying twice as many pixels as zoom level 1.
                 * 
                 * Equally, a camera with a height of 600 and zoom of 2 would have a display height
                 * of 300 pixels.
                 */
                readonly displayHeight: number;

                /**
                 * Clears all alpha values associated with this Game Object.
                 * 
                 * Immediately sets the alpha levels back to 1 (fully opaque).
                 */
                clearAlpha(): this;

                /**
                 * The alpha value starting from the top-left of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 */
                alphaTopLeft: number;

                /**
                 * The alpha value starting from the top-right of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 */
                alphaTopRight: number;

                /**
                 * The alpha value starting from the bottom-left of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 */
                alphaBottomLeft: number;

                /**
                 * The alpha value starting from the bottom-right of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 */
                alphaBottomRight: number;

            }

            /**
             * A Camera.
             * 
             * The Camera is the way in which all games are rendered in Phaser. They provide a view into your game world,
             * and can be positioned, rotated, zoomed and scrolled accordingly.
             * 
             * A Camera consists of two elements: The viewport and the scroll values.
             * 
             * The viewport is the physical position and size of the Camera within your game. Cameras, by default, are
             * created the same size as your game, but their position and size can be set to anything. This means if you
             * wanted to create a camera that was 320x200 in size, positioned in the bottom-right corner of your game,
             * you'd adjust the viewport to do that (using methods like `setViewport` and `setSize`).
             * 
             * If you wish to change where the Camera is looking in your game, then you scroll it. You can do this
             * via the properties `scrollX` and `scrollY` or the method `setScroll`. Scrolling has no impact on the
             * viewport, and changing the viewport has no impact on the scrolling.
             * 
             * By default a Camera will render all Game Objects it can see. You can change this using the `ignore` method,
             * allowing you to filter Game Objects out on a per-Camera basis.
             * 
             * A Camera also has built-in special effects including Fade, Flash and Camera Shake.
             */
            class Camera extends Phaser.Cameras.Scene2D.BaseCamera implements Phaser.GameObjects.Components.Flip, Phaser.GameObjects.Components.Tint, Phaser.GameObjects.Components.Pipeline {
                /**
                 * 
                 * @param x The x position of the Camera, relative to the top-left of the game canvas.
                 * @param y The y position of the Camera, relative to the top-left of the game canvas.
                 * @param width The width of the Camera, in pixels.
                 * @param height The height of the Camera, in pixels.
                 */
                constructor(x: number, y: number, width: number, height: number);

                /**
                 * Does this Camera allow the Game Objects it renders to receive input events?
                 */
                inputEnabled: boolean;

                /**
                 * The Camera Fade effect handler.
                 * To fade this camera see the `Camera.fade` methods.
                 */
                fadeEffect: Phaser.Cameras.Scene2D.Effects.Fade;

                /**
                 * The Camera Flash effect handler.
                 * To flash this camera see the `Camera.flash` method.
                 */
                flashEffect: Phaser.Cameras.Scene2D.Effects.Flash;

                /**
                 * The Camera Shake effect handler.
                 * To shake this camera see the `Camera.shake` method.
                 */
                shakeEffect: Phaser.Cameras.Scene2D.Effects.Shake;

                /**
                 * The Camera Pan effect handler.
                 * To pan this camera see the `Camera.pan` method.
                 */
                panEffect: Phaser.Cameras.Scene2D.Effects.Pan;

                /**
                 * The Camera Rotate To effect handler.
                 * To rotate this camera see the `Camera.rotateTo` method.
                 */
                rotateToEffect: Phaser.Cameras.Scene2D.Effects.RotateTo;

                /**
                 * The Camera Zoom effect handler.
                 * To zoom this camera see the `Camera.zoom` method.
                 */
                zoomEffect: Phaser.Cameras.Scene2D.Effects.Zoom;

                /**
                 * The linear interpolation value to use when following a target.
                 * 
                 * Can also be set via `setLerp` or as part of the `startFollow` call.
                 * 
                 * The default values of 1 means the camera will instantly snap to the target coordinates.
                 * A lower value, such as 0.1 means the camera will more slowly track the target, giving
                 * a smooth transition. You can set the horizontal and vertical values independently, and also
                 * adjust this value in real-time during your game.
                 * 
                 * Be sure to keep the value between 0 and 1. A value of zero will disable tracking on that axis.
                 */
                lerp: Phaser.Math.Vector2;

                /**
                 * The values stored in this property are subtracted from the Camera targets position, allowing you to
                 * offset the camera from the actual target x/y coordinates by this amount.
                 * Can also be set via `setFollowOffset` or as part of the `startFollow` call.
                 */
                followOffset: Phaser.Math.Vector2;

                /**
                 * The Camera dead zone.
                 * 
                 * The deadzone is only used when the camera is following a target.
                 * 
                 * It defines a rectangular region within which if the target is present, the camera will not scroll.
                 * If the target moves outside of this area, the camera will begin scrolling in order to follow it.
                 * 
                 * The `lerp` values that you can set for a follower target also apply when using a deadzone.
                 * 
                 * You can directly set this property to be an instance of a Rectangle. Or, you can use the
                 * `setDeadzone` method for a chainable approach.
                 * 
                 * The rectangle you provide can have its dimensions adjusted dynamically, however, please
                 * note that its position is updated every frame, as it is constantly re-centered on the cameras mid point.
                 * 
                 * Calling `setDeadzone` with no arguments will reset an active deadzone, as will setting this property
                 * to `null`.
                 */
                deadzone: Phaser.Geom.Rectangle;

                /**
                 * Sets the Camera dead zone.
                 * 
                 * The deadzone is only used when the camera is following a target.
                 * 
                 * It defines a rectangular region within which if the target is present, the camera will not scroll.
                 * If the target moves outside of this area, the camera will begin scrolling in order to follow it.
                 * 
                 * The deadzone rectangle is re-positioned every frame so that it is centered on the mid-point
                 * of the camera. This allows you to use the object for additional game related checks, such as
                 * testing if an object is within it or not via a Rectangle.contains call.
                 * 
                 * The `lerp` values that you can set for a follower target also apply when using a deadzone.
                 * 
                 * Calling this method with no arguments will reset an active deadzone.
                 * @param width The width of the deadzone rectangle in pixels. If not specified the deadzone is removed.
                 * @param height The height of the deadzone rectangle in pixels.
                 */
                setDeadzone(width?: number, height?: number): this;

                /**
                 * Fades the Camera in from the given color over the duration specified.
                 * @param duration The duration of the effect in milliseconds. Default 1000.
                 * @param red The amount to fade the red channel towards. A value between 0 and 255. Default 0.
                 * @param green The amount to fade the green channel towards. A value between 0 and 255. Default 0.
                 * @param blue The amount to fade the blue channel towards. A value between 0 and 255. Default 0.
                 * @param callback This callback will be invoked every frame for the duration of the effect.
                 * It is sent two arguments: A reference to the camera and a progress amount between 0 and 1 indicating how complete the effect is.
                 * @param context The context in which the callback is invoked. Defaults to the Scene to which the Camera belongs.
                 */
                fadeIn(duration?: number, red?: number, green?: number, blue?: number, callback?: Function, context?: any): this;

                /**
                 * Fades the Camera out to the given color over the duration specified.
                 * This is an alias for Camera.fade that forces the fade to start, regardless of existing fades.
                 * @param duration The duration of the effect in milliseconds. Default 1000.
                 * @param red The amount to fade the red channel towards. A value between 0 and 255. Default 0.
                 * @param green The amount to fade the green channel towards. A value between 0 and 255. Default 0.
                 * @param blue The amount to fade the blue channel towards. A value between 0 and 255. Default 0.
                 * @param callback This callback will be invoked every frame for the duration of the effect.
                 * It is sent two arguments: A reference to the camera and a progress amount between 0 and 1 indicating how complete the effect is.
                 * @param context The context in which the callback is invoked. Defaults to the Scene to which the Camera belongs.
                 */
                fadeOut(duration?: number, red?: number, green?: number, blue?: number, callback?: Function, context?: any): this;

                /**
                 * Fades the Camera from the given color to transparent over the duration specified.
                 * @param duration The duration of the effect in milliseconds. Default 1000.
                 * @param red The amount to fade the red channel towards. A value between 0 and 255. Default 0.
                 * @param green The amount to fade the green channel towards. A value between 0 and 255. Default 0.
                 * @param blue The amount to fade the blue channel towards. A value between 0 and 255. Default 0.
                 * @param force Force the effect to start immediately, even if already running. Default false.
                 * @param callback This callback will be invoked every frame for the duration of the effect.
                 * It is sent two arguments: A reference to the camera and a progress amount between 0 and 1 indicating how complete the effect is.
                 * @param context The context in which the callback is invoked. Defaults to the Scene to which the Camera belongs.
                 */
                fadeFrom(duration?: number, red?: number, green?: number, blue?: number, force?: boolean, callback?: Function, context?: any): this;

                /**
                 * Fades the Camera from transparent to the given color over the duration specified.
                 * @param duration The duration of the effect in milliseconds. Default 1000.
                 * @param red The amount to fade the red channel towards. A value between 0 and 255. Default 0.
                 * @param green The amount to fade the green channel towards. A value between 0 and 255. Default 0.
                 * @param blue The amount to fade the blue channel towards. A value between 0 and 255. Default 0.
                 * @param force Force the effect to start immediately, even if already running. Default false.
                 * @param callback This callback will be invoked every frame for the duration of the effect.
                 * It is sent two arguments: A reference to the camera and a progress amount between 0 and 1 indicating how complete the effect is.
                 * @param context The context in which the callback is invoked. Defaults to the Scene to which the Camera belongs.
                 */
                fade(duration?: number, red?: number, green?: number, blue?: number, force?: boolean, callback?: Function, context?: any): this;

                /**
                 * Flashes the Camera by setting it to the given color immediately and then fading it away again quickly over the duration specified.
                 * @param duration The duration of the effect in milliseconds. Default 250.
                 * @param red The amount to fade the red channel towards. A value between 0 and 255. Default 255.
                 * @param green The amount to fade the green channel towards. A value between 0 and 255. Default 255.
                 * @param blue The amount to fade the blue channel towards. A value between 0 and 255. Default 255.
                 * @param force Force the effect to start immediately, even if already running. Default false.
                 * @param callback This callback will be invoked every frame for the duration of the effect.
                 * It is sent two arguments: A reference to the camera and a progress amount between 0 and 1 indicating how complete the effect is.
                 * @param context The context in which the callback is invoked. Defaults to the Scene to which the Camera belongs.
                 */
                flash(duration?: number, red?: number, green?: number, blue?: number, force?: boolean, callback?: Function, context?: any): this;

                /**
                 * Shakes the Camera by the given intensity over the duration specified.
                 * @param duration The duration of the effect in milliseconds. Default 100.
                 * @param intensity The intensity of the shake. Default 0.05.
                 * @param force Force the shake effect to start immediately, even if already running. Default false.
                 * @param callback This callback will be invoked every frame for the duration of the effect.
                 * It is sent two arguments: A reference to the camera and a progress amount between 0 and 1 indicating how complete the effect is.
                 * @param context The context in which the callback is invoked. Defaults to the Scene to which the Camera belongs.
                 */
                shake(duration?: number, intensity?: number | Phaser.Math.Vector2, force?: boolean, callback?: Function, context?: any): this;

                /**
                 * This effect will scroll the Camera so that the center of its viewport finishes at the given destination,
                 * over the duration and with the ease specified.
                 * @param x The destination x coordinate to scroll the center of the Camera viewport to.
                 * @param y The destination y coordinate to scroll the center of the Camera viewport to.
                 * @param duration The duration of the effect in milliseconds. Default 1000.
                 * @param ease The ease to use for the pan. Can be any of the Phaser Easing constants or a custom function. Default 'Linear'.
                 * @param force Force the pan effect to start immediately, even if already running. Default false.
                 * @param callback This callback will be invoked every frame for the duration of the effect.
                 * It is sent four arguments: A reference to the camera, a progress amount between 0 and 1 indicating how complete the effect is,
                 * the current camera scroll x coordinate and the current camera scroll y coordinate.
                 * @param context The context in which the callback is invoked. Defaults to the Scene to which the Camera belongs.
                 */
                pan(x: number, y: number, duration?: number, ease?: string | Function, force?: boolean, callback?: Phaser.Types.Cameras.Scene2D.CameraPanCallback, context?: any): this;

                /**
                 * This effect will rotate the Camera so that the viewport finishes at the given angle in radians,
                 * over the duration and with the ease specified.
                 * @param radians The destination angle in radians to rotate the Camera viewport to. If the angle is positive then the rotation is clockwise else anticlockwise
                 * @param shortestPath If shortest path is set to true the camera will rotate in the quickest direction clockwise or anti-clockwise. Default false.
                 * @param duration The duration of the effect in milliseconds. Default 1000.
                 * @param ease The ease to use for the rotation. Can be any of the Phaser Easing constants or a custom function. Default 'Linear'.
                 * @param force Force the rotation effect to start immediately, even if already running. Default false.
                 * @param callback This callback will be invoked every frame for the duration of the effect.
                 * It is sent four arguments: A reference to the camera, a progress amount between 0 and 1 indicating how complete the effect is,
                 * the current camera rotation angle in radians.
                 * @param context The context in which the callback is invoked. Defaults to the Scene to which the Camera belongs.
                 */
                rotateTo(radians: number, shortestPath?: boolean, duration?: number, ease?: string | Function, force?: boolean, callback?: CameraRotateCallback, context?: any): Phaser.Cameras.Scene2D.Camera;

                /**
                 * This effect will zoom the Camera to the given scale, over the duration and with the ease specified.
                 * @param zoom The target Camera zoom value.
                 * @param duration The duration of the effect in milliseconds. Default 1000.
                 * @param ease The ease to use for the pan. Can be any of the Phaser Easing constants or a custom function. Default 'Linear'.
                 * @param force Force the pan effect to start immediately, even if already running. Default false.
                 * @param callback This callback will be invoked every frame for the duration of the effect.
                 * It is sent four arguments: A reference to the camera, a progress amount between 0 and 1 indicating how complete the effect is,
                 * the current camera scroll x coordinate and the current camera scroll y coordinate.
                 * @param context The context in which the callback is invoked. Defaults to the Scene to which the Camera belongs.
                 */
                zoomTo(zoom: number, duration?: number, ease?: string | Function, force?: boolean, callback?: Phaser.Types.Cameras.Scene2D.CameraPanCallback, context?: any): this;

                /**
                 * Internal preRender step.
                 */
                protected preRender(): void;

                /**
                 * Sets the linear interpolation value to use when following a target.
                 * 
                 * The default values of 1 means the camera will instantly snap to the target coordinates.
                 * A lower value, such as 0.1 means the camera will more slowly track the target, giving
                 * a smooth transition. You can set the horizontal and vertical values independently, and also
                 * adjust this value in real-time during your game.
                 * 
                 * Be sure to keep the value between 0 and 1. A value of zero will disable tracking on that axis.
                 * @param x The amount added to the horizontal linear interpolation of the follow target. Default 1.
                 * @param y The amount added to the vertical linear interpolation of the follow target. Default 1.
                 */
                setLerp(x?: number, y?: number): this;

                /**
                 * Sets the horizontal and vertical offset of the camera from its follow target.
                 * The values are subtracted from the targets position during the Cameras update step.
                 * @param x The horizontal offset from the camera follow target.x position. Default 0.
                 * @param y The vertical offset from the camera follow target.y position. Default 0.
                 */
                setFollowOffset(x?: number, y?: number): this;

                /**
                 * Sets the Camera to follow a Game Object.
                 * 
                 * When enabled the Camera will automatically adjust its scroll position to keep the target Game Object
                 * in its center.
                 * 
                 * You can set the linear interpolation value used in the follow code.
                 * Use low lerp values (such as 0.1) to automatically smooth the camera motion.
                 * 
                 * If you find you're getting a slight "jitter" effect when following an object it's probably to do with sub-pixel
                 * rendering of the targets position. This can be rounded by setting the `roundPixels` argument to `true` to
                 * force full pixel rounding rendering. Note that this can still be broken if you have specified a non-integer zoom
                 * value on the camera. So be sure to keep the camera zoom to integers.
                 * @param target The target for the Camera to follow.
                 * @param roundPixels Round the camera position to whole integers to avoid sub-pixel rendering? Default false.
                 * @param lerpX A value between 0 and 1. This value specifies the amount of linear interpolation to use when horizontally tracking the target. The closer the value to 1, the faster the camera will track. Default 1.
                 * @param lerpY A value between 0 and 1. This value specifies the amount of linear interpolation to use when vertically tracking the target. The closer the value to 1, the faster the camera will track. Default 1.
                 * @param offsetX The horizontal offset from the camera follow target.x position. Default 0.
                 * @param offsetY The vertical offset from the camera follow target.y position. Default 0.
                 */
                startFollow(target: Phaser.GameObjects.GameObject | object, roundPixels?: boolean, lerpX?: number, lerpY?: number, offsetX?: number, offsetY?: number): this;

                /**
                 * Stops a Camera from following a Game Object, if previously set via `Camera.startFollow`.
                 */
                stopFollow(): this;

                /**
                 * Resets any active FX, such as a fade, flash or shake. Useful to call after a fade in order to
                 * remove the fade.
                 */
                resetFX(): this;

                /**
                 * Internal method called automatically by the Camera Manager.
                 * @param time The current timestamp as generated by the Request Animation Frame or SetTimeout.
                 * @param delta The delta time, in ms, elapsed since the last frame.
                 */
                protected update(time: number, delta: number): void;

                /**
                 * Destroys this Camera instance. You rarely need to call this directly.
                 * 
                 * Called by the Camera Manager. If you wish to destroy a Camera please use `CameraManager.remove` as
                 * cameras are stored in a pool, ready for recycling later, and calling this directly will prevent that.
                 */
                destroy(): void;

                /**
                 * Clears all alpha values associated with this Game Object.
                 * 
                 * Immediately sets the alpha levels back to 1 (fully opaque).
                 */
                clearAlpha(): this;

                /**
                 * The alpha value starting from the top-left of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 */
                alphaTopLeft: number;

                /**
                 * The alpha value starting from the top-right of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 */
                alphaTopRight: number;

                /**
                 * The alpha value starting from the bottom-left of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 */
                alphaBottomLeft: number;

                /**
                 * The alpha value starting from the bottom-right of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 */
                alphaBottomRight: number;

                /**
                 * The horizontally flipped state of the Game Object.
                 * 
                 * A Game Object that is flipped horizontally will render inversed on the horizontal axis.
                 * Flipping always takes place from the middle of the texture and does not impact the scale value.
                 * If this Game Object has a physics body, it will not change the body. This is a rendering toggle only.
                 */
                flipX: boolean;

                /**
                 * The vertically flipped state of the Game Object.
                 * 
                 * A Game Object that is flipped vertically will render inversed on the vertical axis (i.e. upside down)
                 * Flipping always takes place from the middle of the texture and does not impact the scale value.
                 * If this Game Object has a physics body, it will not change the body. This is a rendering toggle only.
                 */
                flipY: boolean;

                /**
                 * Toggles the horizontal flipped state of this Game Object.
                 * 
                 * A Game Object that is flipped horizontally will render inversed on the horizontal axis.
                 * Flipping always takes place from the middle of the texture and does not impact the scale value.
                 * If this Game Object has a physics body, it will not change the body. This is a rendering toggle only.
                 */
                toggleFlipX(): this;

                /**
                 * Toggles the vertical flipped state of this Game Object.
                 */
                toggleFlipY(): this;

                /**
                 * Sets the horizontal flipped state of this Game Object.
                 * 
                 * A Game Object that is flipped horizontally will render inversed on the horizontal axis.
                 * Flipping always takes place from the middle of the texture and does not impact the scale value.
                 * If this Game Object has a physics body, it will not change the body. This is a rendering toggle only.
                 * @param value The flipped state. `false` for no flip, or `true` to be flipped.
                 */
                setFlipX(value: boolean): this;

                /**
                 * Sets the vertical flipped state of this Game Object.
                 * @param value The flipped state. `false` for no flip, or `true` to be flipped.
                 */
                setFlipY(value: boolean): this;

                /**
                 * Sets the horizontal and vertical flipped state of this Game Object.
                 * 
                 * A Game Object that is flipped will render inversed on the flipped axis.
                 * Flipping always takes place from the middle of the texture and does not impact the scale value.
                 * If this Game Object has a physics body, it will not change the body. This is a rendering toggle only.
                 * @param x The horizontal flipped state. `false` for no flip, or `true` to be flipped.
                 * @param y The horizontal flipped state. `false` for no flip, or `true` to be flipped.
                 */
                setFlip(x: boolean, y: boolean): this;

                /**
                 * Resets the horizontal and vertical flipped state of this Game Object back to their default un-flipped state.
                 */
                resetFlip(): this;

                /**
                 * The tint value being applied to the top-left vertice of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 * The value should be set as a hex number, i.e. 0xff0000 for red, or 0xff00ff for purple.
                 */
                tintTopLeft: number;

                /**
                 * The tint value being applied to the top-right vertice of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 * The value should be set as a hex number, i.e. 0xff0000 for red, or 0xff00ff for purple.
                 */
                tintTopRight: number;

                /**
                 * The tint value being applied to the bottom-left vertice of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 * The value should be set as a hex number, i.e. 0xff0000 for red, or 0xff00ff for purple.
                 */
                tintBottomLeft: number;

                /**
                 * The tint value being applied to the bottom-right vertice of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 * The value should be set as a hex number, i.e. 0xff0000 for red, or 0xff00ff for purple.
                 */
                tintBottomRight: number;

                /**
                 * The tint fill mode.
                 * 
                 * `false` = An additive tint (the default), where vertices colors are blended with the texture.
                 * `true` = A fill tint, where the vertices colors replace the texture, but respects texture alpha.
                 */
                tintFill: boolean;

                /**
                 * Clears all tint values associated with this Game Object.
                 * 
                 * Immediately sets the color values back to 0xffffff and the tint type to 'additive',
                 * which results in no visible change to the texture.
                 */
                clearTint(): this;

                /**
                 * Sets an additive tint on this Game Object.
                 * 
                 * The tint works by taking the pixel color values from the Game Objects texture, and then
                 * multiplying it by the color value of the tint. You can provide either one color value,
                 * in which case the whole Game Object will be tinted in that color. Or you can provide a color
                 * per corner. The colors are blended together across the extent of the Game Object.
                 * 
                 * To modify the tint color once set, either call this method again with new values or use the
                 * `tint` property to set all colors at once. Or, use the properties `tintTopLeft`, `tintTopRight,
                 * `tintBottomLeft` and `tintBottomRight` to set the corner color values independently.
                 * 
                 * To remove a tint call `clearTint`.
                 * 
                 * To swap this from being an additive tint to a fill based tint set the property `tintFill` to `true`.
                 * @param topLeft The tint being applied to the top-left of the Game Object. If no other values are given this value is applied evenly, tinting the whole Game Object. Default 0xffffff.
                 * @param topRight The tint being applied to the top-right of the Game Object.
                 * @param bottomLeft The tint being applied to the bottom-left of the Game Object.
                 * @param bottomRight The tint being applied to the bottom-right of the Game Object.
                 */
                setTint(topLeft?: number, topRight?: number, bottomLeft?: number, bottomRight?: number): this;

                /**
                 * Sets a fill-based tint on this Game Object.
                 * 
                 * Unlike an additive tint, a fill-tint literally replaces the pixel colors from the texture
                 * with those in the tint. You can use this for effects such as making a player flash 'white'
                 * if hit by something. You can provide either one color value, in which case the whole
                 * Game Object will be rendered in that color. Or you can provide a color per corner. The colors
                 * are blended together across the extent of the Game Object.
                 * 
                 * To modify the tint color once set, either call this method again with new values or use the
                 * `tint` property to set all colors at once. Or, use the properties `tintTopLeft`, `tintTopRight,
                 * `tintBottomLeft` and `tintBottomRight` to set the corner color values independently.
                 * 
                 * To remove a tint call `clearTint`.
                 * 
                 * To swap this from being a fill-tint to an additive tint set the property `tintFill` to `false`.
                 * @param topLeft The tint being applied to the top-left of the Game Object. If not other values are given this value is applied evenly, tinting the whole Game Object. Default 0xffffff.
                 * @param topRight The tint being applied to the top-right of the Game Object.
                 * @param bottomLeft The tint being applied to the bottom-left of the Game Object.
                 * @param bottomRight The tint being applied to the bottom-right of the Game Object.
                 */
                setTintFill(topLeft?: number, topRight?: number, bottomLeft?: number, bottomRight?: number): this;

                /**
                 * The tint value being applied to the whole of the Game Object.
                 * This property is a setter-only. Use the properties `tintTopLeft` etc to read the current tint value.
                 */
                tint: number;

                /**
                 * Does this Game Object have a tint applied?
                 * 
                 * It checks to see if the 4 tint properties are set to the value 0xffffff
                 * and that the `tintFill` property is `false`. This indicates that a Game Object isn't tinted.
                 */
                readonly isTinted: boolean;

                /**
                 * The initial WebGL pipeline of this Game Object.
                 * 
                 * If you call `resetPipeline` on this Game Object, the pipeline is reset to this default.
                 */
                defaultPipeline: Phaser.Renderer.WebGL.WebGLPipeline;

                /**
                 * The current WebGL pipeline of this Game Object.
                 */
                pipeline: Phaser.Renderer.WebGL.WebGLPipeline;

                /**
                 * Does this Game Object have any Post Pipelines set?
                 */
                hasPostPipeline: boolean;

                /**
                 * The WebGL Post FX Pipelines this Game Object uses for post-render effects.
                 * 
                 * The pipelines are processed in the order in which they appear in this array.
                 * 
                 * If you modify this array directly, be sure to set the
                 * `hasPostPipeline` property accordingly.
                 */
                postPipelines: Phaser.Renderer.WebGL.Pipelines.PostFXPipeline[];

                /**
                 * An object to store pipeline specific data in, to be read by the pipelines this Game Object uses.
                 */
                pipelineData: object;

                /**
                 * Sets the initial WebGL Pipeline of this Game Object.
                 * 
                 * This should only be called during the instantiation of the Game Object. After that, use `setPipeline`.
                 * @param pipeline Either the string-based name of the pipeline, or a pipeline instance to set.
                 */
                initPipeline(pipeline: string | Phaser.Renderer.WebGL.WebGLPipeline): boolean;

                /**
                 * Sets the main WebGL Pipeline of this Game Object.
                 * 
                 * Also sets the `pipelineData` property, if the parameter is given.
                 * 
                 * Both the pipeline and post pipelines share the same pipeline data object.
                 * @param pipeline Either the string-based name of the pipeline, or a pipeline instance to set.
                 * @param pipelineData Optional pipeline data object that is _deep copied_ into the `pipelineData` property of this Game Object.
                 * @param copyData Should the pipeline data object be _deep copied_ into the `pipelineData` property of this Game Object? If `false` it will be set by reference instead. Default true.
                 */
                setPipeline(pipeline: string | Phaser.Renderer.WebGL.WebGLPipeline, pipelineData?: object, copyData?: boolean): this;

                /**
                 * Sets one, or more, Post Pipelines on this Game Object.
                 * 
                 * Post Pipelines are invoked after this Game Object has rendered to its target and
                 * are commonly used for post-fx.
                 * 
                 * The post pipelines are appended to the `postPipelines` array belonging to this
                 * Game Object. When the renderer processes this Game Object, it iterates through the post
                 * pipelines in the order in which they appear in the array. If you are stacking together
                 * multiple effects, be aware that the order is important.
                 * 
                 * If you call this method multiple times, the new pipelines will be appended to any existing
                 * post pipelines already set. Use the `resetPostPipeline` method to clear them first, if required.
                 * 
                 * You can optionally also sets the `pipelineData` property, if the parameter is given.
                 * 
                 * Both the pipeline and post pipelines share the pipeline data object together.
                 * @param pipelines Either the string-based name of the pipeline, or a pipeline instance, or class, or an array of them.
                 * @param pipelineData Optional pipeline data object that is _deep copied_ into the `pipelineData` property of this Game Object.
                 * @param copyData Should the pipeline data object be _deep copied_ into the `pipelineData` property of this Game Object? If `false` it will be set by reference instead. Default true.
                 */
                setPostPipeline(pipelines: string | string[] | Function | Function[] | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline[], pipelineData?: object, copyData?: boolean): this;

                /**
                 * Adds an entry to the `pipelineData` object belonging to this Game Object.
                 * 
                 * If the 'key' already exists, its value is updated. If it doesn't exist, it is created.
                 * 
                 * If `value` is undefined, and `key` exists, `key` is removed from the data object.
                 * 
                 * Both the pipeline and post pipelines share the pipeline data object together.
                 * @param key The key of the pipeline data to set, update, or delete.
                 * @param value The value to be set with the key. If `undefined` then `key` will be deleted from the object.
                 */
                setPipelineData(key: string, value?: any): this;

                /**
                 * Gets a Post Pipeline instance from this Game Object, based on the given name, and returns it.
                 * @param pipeline The string-based name of the pipeline, or a pipeline class.
                 */
                getPostPipeline(pipeline: string | Function | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline): Phaser.Renderer.WebGL.Pipelines.PostFXPipeline | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline[];

                /**
                 * Resets the WebGL Pipeline of this Game Object back to the default it was created with.
                 * @param resetPostPipelines Reset all of the post pipelines? Default false.
                 * @param resetData Reset the `pipelineData` object to being an empty object? Default false.
                 */
                resetPipeline(resetPostPipelines?: boolean, resetData?: boolean): boolean;

                /**
                 * Resets the WebGL Post Pipelines of this Game Object. It does this by calling
                 * the `destroy` method on each post pipeline and then clearing the local array.
                 * @param resetData Reset the `pipelineData` object to being an empty object? Default false.
                 */
                resetPostPipeline(resetData?: boolean): void;

                /**
                 * Removes a type of Post Pipeline instances from this Game Object, based on the given name, and destroys them.
                 * 
                 * If you wish to remove all Post Pipelines use the `resetPostPipeline` method instead.
                 * @param pipeline The string-based name of the pipeline, or a pipeline class.
                 */
                removePostPipeline(pipeline: string | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline): this;

                /**
                 * Gets the name of the WebGL Pipeline this Game Object is currently using.
                 */
                getPipelineName(): string;

            }

            /**
             * The Camera Manager is a plugin that belongs to a Scene and is responsible for managing all of the Scene Cameras.
             * 
             * By default you can access the Camera Manager from within a Scene using `this.cameras`, although this can be changed
             * in your game config.
             * 
             * Create new Cameras using the `add` method. Or extend the Camera class with your own addition code and then add
             * the new Camera in using the `addExisting` method.
             * 
             * Cameras provide a view into your game world, and can be positioned, rotated, zoomed and scrolled accordingly.
             * 
             * A Camera consists of two elements: The viewport and the scroll values.
             * 
             * The viewport is the physical position and size of the Camera within your game. Cameras, by default, are
             * created the same size as your game, but their position and size can be set to anything. This means if you
             * wanted to create a camera that was 320x200 in size, positioned in the bottom-right corner of your game,
             * you'd adjust the viewport to do that (using methods like `setViewport` and `setSize`).
             * 
             * If you wish to change where the Camera is looking in your game, then you scroll it. You can do this
             * via the properties `scrollX` and `scrollY` or the method `setScroll`. Scrolling has no impact on the
             * viewport, and changing the viewport has no impact on the scrolling.
             * 
             * By default a Camera will render all Game Objects it can see. You can change this using the `ignore` method,
             * allowing you to filter Game Objects out on a per-Camera basis. The Camera Manager can manage up to 31 unique
             * 'Game Object ignore capable' Cameras. Any Cameras beyond 31 that you create will all be given a Camera ID of
             * zero, meaning that they cannot be used for Game Object exclusion. This means if you need your Camera to ignore
             * Game Objects, make sure it's one of the first 31 created.
             * 
             * A Camera also has built-in special effects including Fade, Flash, Camera Shake, Pan and Zoom.
             */
            class CameraManager {
                /**
                 * 
                 * @param scene The Scene that owns the Camera Manager plugin.
                 */
                constructor(scene: Phaser.Scene);

                /**
                 * The Scene that owns the Camera Manager plugin.
                 */
                scene: Phaser.Scene;

                /**
                 * A reference to the Scene.Systems handler for the Scene that owns the Camera Manager.
                 */
                systems: Phaser.Scenes.Systems;

                /**
                 * All Cameras created by, or added to, this Camera Manager, will have their `roundPixels`
                 * property set to match this value. By default it is set to match the value set in the
                 * game configuration, but can be changed at any point. Equally, individual cameras can
                 * also be changed as needed.
                 */
                roundPixels: boolean;

                /**
                 * An Array of the Camera objects being managed by this Camera Manager.
                 * The Cameras are updated and rendered in the same order in which they appear in this array.
                 * Do not directly add or remove entries to this array. However, you can move the contents
                 * around the array should you wish to adjust the display order.
                 */
                cameras: Phaser.Cameras.Scene2D.Camera[];

                /**
                 * A handy reference to the 'main' camera. By default this is the first Camera the
                 * Camera Manager creates. You can also set it directly, or use the `makeMain` argument
                 * in the `add` and `addExisting` methods. It allows you to access it from your game:
                 * 
                 * ```javascript
                 * var cam = this.cameras.main;
                 * ```
                 * 
                 * Also see the properties `camera1`, `camera2` and so on.
                 */
                main: Phaser.Cameras.Scene2D.Camera;

                /**
                 * A default un-transformed Camera that doesn't exist on the camera list and doesn't
                 * count towards the total number of cameras being managed. It exists for other
                 * systems, as well as your own code, should they require a basic un-transformed
                 * camera instance from which to calculate a view matrix.
                 */
                default: Phaser.Cameras.Scene2D.Camera;

                /**
                 * Adds a new Camera into the Camera Manager. The Camera Manager can support up to 31 different Cameras.
                 * 
                 * Each Camera has its own viewport, which controls the size of the Camera and its position within the canvas.
                 * 
                 * Use the `Camera.scrollX` and `Camera.scrollY` properties to change where the Camera is looking, or the
                 * Camera methods such as `centerOn`. Cameras also have built in special effects, such as fade, flash, shake,
                 * pan and zoom.
                 * 
                 * By default Cameras are transparent and will render anything that they can see based on their `scrollX`
                 * and `scrollY` values. Game Objects can be set to be ignored by a Camera by using the `Camera.ignore` method.
                 * 
                 * The Camera will have its `roundPixels` property set to whatever `CameraManager.roundPixels` is. You can change
                 * it after creation if required.
                 * 
                 * See the Camera class documentation for more details.
                 * @param x The horizontal position of the Camera viewport. Default 0.
                 * @param y The vertical position of the Camera viewport. Default 0.
                 * @param width The width of the Camera viewport. If not given it'll be the game config size.
                 * @param height The height of the Camera viewport. If not given it'll be the game config size.
                 * @param makeMain Set this Camera as being the 'main' camera. This just makes the property `main` a reference to it. Default false.
                 * @param name The name of the Camera. Default ''.
                 */
                add(x?: number, y?: number, width?: number, height?: number, makeMain?: boolean, name?: string): Phaser.Cameras.Scene2D.Camera;

                /**
                 * Adds an existing Camera into the Camera Manager.
                 * 
                 * The Camera should either be a `Phaser.Cameras.Scene2D.Camera` instance, or a class that extends from it.
                 * 
                 * The Camera will have its `roundPixels` property set to whatever `CameraManager.roundPixels` is. You can change
                 * it after addition if required.
                 * 
                 * The Camera will be assigned an ID, which is used for Game Object exclusion and then added to the
                 * manager. As long as it doesn't already exist in the manager it will be added then returned.
                 * 
                 * If this method returns `null` then the Camera already exists in this Camera Manager.
                 * @param camera The Camera to be added to the Camera Manager.
                 * @param makeMain Set this Camera as being the 'main' camera. This just makes the property `main` a reference to it. Default false.
                 */
                addExisting(camera: Phaser.Cameras.Scene2D.Camera, makeMain?: boolean): Phaser.Cameras.Scene2D.Camera;

                /**
                 * Gets the total number of Cameras in this Camera Manager.
                 * 
                 * If the optional `isVisible` argument is set it will only count Cameras that are currently visible.
                 * @param isVisible Set the `true` to only include visible Cameras in the total. Default false.
                 */
                getTotal(isVisible?: boolean): number;

                /**
                 * Populates this Camera Manager based on the given configuration object, or an array of config objects.
                 * 
                 * See the `Phaser.Types.Cameras.Scene2D.CameraConfig` documentation for details of the object structure.
                 * @param config A Camera configuration object, or an array of them, to be added to this Camera Manager.
                 */
                fromJSON(config: Phaser.Types.Cameras.Scene2D.CameraConfig | Phaser.Types.Cameras.Scene2D.CameraConfig[]): this;

                /**
                 * Gets a Camera based on its name.
                 * 
                 * Camera names are optional and don't have to be set, so this method is only of any use if you
                 * have given your Cameras unique names.
                 * @param name The name of the Camera.
                 */
                getCamera(name: string): Phaser.Cameras.Scene2D.Camera;

                /**
                 * Returns an array of all cameras below the given Pointer.
                 * 
                 * The first camera in the array is the top-most camera in the camera list.
                 * @param pointer The Pointer to check against.
                 */
                getCamerasBelowPointer(pointer: Phaser.Input.Pointer): Phaser.Cameras.Scene2D.Camera[];

                /**
                 * Removes the given Camera, or an array of Cameras, from this Camera Manager.
                 * 
                 * If found in the Camera Manager it will be immediately removed from the local cameras array.
                 * If also currently the 'main' camera, 'main' will be reset to be camera 0.
                 * 
                 * The removed Cameras are automatically destroyed if the `runDestroy` argument is `true`, which is the default.
                 * If you wish to re-use the cameras then set this to `false`, but know that they will retain their references
                 * and internal data until destroyed or re-added to a Camera Manager.
                 * @param camera The Camera, or an array of Cameras, to be removed from this Camera Manager.
                 * @param runDestroy Automatically call `Camera.destroy` on each Camera removed from this Camera Manager. Default true.
                 */
                remove(camera: Phaser.Cameras.Scene2D.Camera | Phaser.Cameras.Scene2D.Camera[], runDestroy?: boolean): number;

                /**
                 * The internal render method. This is called automatically by the Scene and should not be invoked directly.
                 * 
                 * It will iterate through all local cameras and render them in turn, as long as they're visible and have
                 * an alpha level > 0.
                 * @param renderer The Renderer that will render the children to this camera.
                 * @param displayList The Display List for the Scene.
                 */
                protected render(renderer: Phaser.Renderer.Canvas.CanvasRenderer | Phaser.Renderer.WebGL.WebGLRenderer, displayList: Phaser.GameObjects.DisplayList): void;

                /**
                 * Takes an array of Game Objects and a Camera and returns a new array
                 * containing only those Game Objects that pass the `willRender` test
                 * against the given Camera.
                 * @param children An array of Game Objects to be checked against the camera.
                 * @param camera The camera to filte the Game Objects against.
                 */
                getVisibleChildren(children: Phaser.GameObjects.GameObject[], camera: Phaser.Cameras.Scene2D.Camera): Phaser.GameObjects.GameObject[];

                /**
                 * Resets this Camera Manager.
                 * 
                 * This will iterate through all current Cameras, destroying them all, then it will reset the
                 * cameras array, reset the ID counter and create 1 new single camera using the default values.
                 */
                resetAll(): Phaser.Cameras.Scene2D.Camera;

                /**
                 * The main update loop. Called automatically when the Scene steps.
                 * @param time The current timestamp as generated by the Request Animation Frame or SetTimeout.
                 * @param delta The delta time, in ms, elapsed since the last frame.
                 */
                protected update(time: number, delta: number): void;

                /**
                 * The event handler that manages the `resize` event dispatched by the Scale Manager.
                 * @param gameSize The default Game Size object. This is the un-modified game dimensions.
                 * @param baseSize The base Size object. The game dimensions. The canvas width / height values match this.
                 */
                onResize(gameSize: Phaser.Structs.Size, baseSize: Phaser.Structs.Size): void;

                /**
                 * Resizes all cameras to the given dimensions.
                 * @param width The new width of the camera.
                 * @param height The new height of the camera.
                 */
                resize(width: number, height: number): void;

            }

            namespace Effects {
                /**
                 * A Camera Fade effect.
                 * 
                 * This effect will fade the camera viewport to the given color, over the duration specified.
                 * 
                 * Only the camera viewport is faded. None of the objects it is displaying are impacted, i.e. their colors do
                 * not change.
                 * 
                 * The effect will dispatch several events on the Camera itself and you can also specify an `onUpdate` callback,
                 * which is invoked each frame for the duration of the effect, if required.
                 */
                class Fade {
                    /**
                     * 
                     * @param camera The camera this effect is acting upon.
                     */
                    constructor(camera: Phaser.Cameras.Scene2D.Camera);

                    /**
                     * The Camera this effect belongs to.
                     */
                    readonly camera: Phaser.Cameras.Scene2D.Camera;

                    /**
                     * Is this effect actively running?
                     */
                    readonly isRunning: boolean;

                    /**
                     * Has this effect finished running?
                     * 
                     * This is different from `isRunning` because it remains set to `true` when the effect is over,
                     * until the effect is either reset or started again.
                     */
                    readonly isComplete: boolean;

                    /**
                     * The direction of the fade.
                     * `true` = fade out (transparent to color), `false` = fade in (color to transparent)
                     */
                    readonly direction: boolean;

                    /**
                     * The duration of the effect, in milliseconds.
                     */
                    readonly duration: number;

                    /**
                     * If this effect is running this holds the current percentage of the progress, a value between 0 and 1.
                     */
                    progress: number;

                    /**
                     * Fades the Camera to or from the given color over the duration specified.
                     * @param direction The direction of the fade. `true` = fade out (transparent to color), `false` = fade in (color to transparent) Default true.
                     * @param duration The duration of the effect in milliseconds. Default 1000.
                     * @param red The amount to fade the red channel towards. A value between 0 and 255. Default 0.
                     * @param green The amount to fade the green channel towards. A value between 0 and 255. Default 0.
                     * @param blue The amount to fade the blue channel towards. A value between 0 and 255. Default 0.
                     * @param force Force the effect to start immediately, even if already running. Default false.
                     * @param callback This callback will be invoked every frame for the duration of the effect.
                     * It is sent two arguments: A reference to the camera and a progress amount between 0 and 1 indicating how complete the effect is.
                     * @param context The context in which the callback is invoked. Defaults to the Scene to which the Camera belongs.
                     */
                    start(direction?: boolean, duration?: number, red?: number, green?: number, blue?: number, force?: boolean, callback?: Phaser.Types.Cameras.Scene2D.CameraFadeCallback, context?: any): Phaser.Cameras.Scene2D.Camera;

                    /**
                     * The main update loop for this effect. Called automatically by the Camera.
                     * @param time The current timestamp as generated by the Request Animation Frame or SetTimeout.
                     * @param delta The delta time, in ms, elapsed since the last frame.
                     */
                    update(time: number, delta: number): void;

                    /**
                     * Called internally by the Canvas Renderer.
                     * @param ctx The Canvas context to render to.
                     */
                    postRenderCanvas(ctx: CanvasRenderingContext2D): boolean;

                    /**
                     * Called internally by the WebGL Renderer.
                     * @param pipeline The WebGL Pipeline to render to. Must provide the `drawFillRect` method.
                     * @param getTintFunction A function that will return the gl safe tint colors.
                     */
                    postRenderWebGL(pipeline: Phaser.Renderer.WebGL.Pipelines.MultiPipeline, getTintFunction: Function): boolean;

                    /**
                     * Called internally when the effect completes.
                     */
                    effectComplete(): void;

                    /**
                     * Resets this camera effect.
                     * If it was previously running, it stops instantly without calling its onComplete callback or emitting an event.
                     */
                    reset(): void;

                    /**
                     * Destroys this effect, releasing it from the Camera.
                     */
                    destroy(): void;

                }

                /**
                 * A Camera Flash effect.
                 * 
                 * This effect will flash the camera viewport to the given color, over the duration specified.
                 * 
                 * Only the camera viewport is flashed. None of the objects it is displaying are impacted, i.e. their colors do
                 * not change.
                 * 
                 * The effect will dispatch several events on the Camera itself and you can also specify an `onUpdate` callback,
                 * which is invoked each frame for the duration of the effect, if required.
                 */
                class Flash {
                    /**
                     * 
                     * @param camera The camera this effect is acting upon.
                     */
                    constructor(camera: Phaser.Cameras.Scene2D.Camera);

                    /**
                     * The Camera this effect belongs to.
                     */
                    readonly camera: Phaser.Cameras.Scene2D.Camera;

                    /**
                     * Is this effect actively running?
                     */
                    readonly isRunning: boolean;

                    /**
                     * The duration of the effect, in milliseconds.
                     */
                    readonly duration: number;

                    /**
                     * If this effect is running this holds the current percentage of the progress, a value between 0 and 1.
                     */
                    progress: number;

                    /**
                     * Flashes the Camera to or from the given color over the duration specified.
                     * @param duration The duration of the effect in milliseconds. Default 250.
                     * @param red The amount to flash the red channel towards. A value between 0 and 255. Default 255.
                     * @param green The amount to flash the green channel towards. A value between 0 and 255. Default 255.
                     * @param blue The amount to flash the blue channel towards. A value between 0 and 255. Default 255.
                     * @param force Force the effect to start immediately, even if already running. Default false.
                     * @param callback This callback will be invoked every frame for the duration of the effect.
                     * It is sent two arguments: A reference to the camera and a progress amount between 0 and 1 indicating how complete the effect is.
                     * @param context The context in which the callback is invoked. Defaults to the Scene to which the Camera belongs.
                     */
                    start(duration?: number, red?: number, green?: number, blue?: number, force?: boolean, callback?: Phaser.Types.Cameras.Scene2D.CameraFlashCallback, context?: any): Phaser.Cameras.Scene2D.Camera;

                    /**
                     * The main update loop for this effect. Called automatically by the Camera.
                     * @param time The current timestamp as generated by the Request Animation Frame or SetTimeout.
                     * @param delta The delta time, in ms, elapsed since the last frame.
                     */
                    update(time: number, delta: number): void;

                    /**
                     * Called internally by the Canvas Renderer.
                     * @param ctx The Canvas context to render to.
                     */
                    postRenderCanvas(ctx: CanvasRenderingContext2D): boolean;

                    /**
                     * Called internally by the WebGL Renderer.
                     * @param pipeline The WebGL Pipeline to render to. Must provide the `drawFillRect` method.
                     * @param getTintFunction A function that will return the gl safe tint colors.
                     */
                    postRenderWebGL(pipeline: Phaser.Renderer.WebGL.Pipelines.MultiPipeline, getTintFunction: Function): boolean;

                    /**
                     * Called internally when the effect completes.
                     */
                    effectComplete(): void;

                    /**
                     * Resets this camera effect.
                     * If it was previously running, it stops instantly without calling its onComplete callback or emitting an event.
                     */
                    reset(): void;

                    /**
                     * Destroys this effect, releasing it from the Camera.
                     */
                    destroy(): void;

                }

                /**
                 * A Camera Pan effect.
                 * 
                 * This effect will scroll the Camera so that the center of its viewport finishes at the given destination,
                 * over the duration and with the ease specified.
                 * 
                 * Only the camera scroll is moved. None of the objects it is displaying are impacted, i.e. their positions do
                 * not change.
                 * 
                 * The effect will dispatch several events on the Camera itself and you can also specify an `onUpdate` callback,
                 * which is invoked each frame for the duration of the effect if required.
                 */
                class Pan {
                    /**
                     * 
                     * @param camera The camera this effect is acting upon.
                     */
                    constructor(camera: Phaser.Cameras.Scene2D.Camera);

                    /**
                     * The Camera this effect belongs to.
                     */
                    readonly camera: Phaser.Cameras.Scene2D.Camera;

                    /**
                     * Is this effect actively running?
                     */
                    readonly isRunning: boolean;

                    /**
                     * The duration of the effect, in milliseconds.
                     */
                    readonly duration: number;

                    /**
                     * The starting scroll coordinates to pan the camera from.
                     */
                    source: Phaser.Math.Vector2;

                    /**
                     * The constantly updated value based on zoom.
                     */
                    current: Phaser.Math.Vector2;

                    /**
                     * The destination scroll coordinates to pan the camera to.
                     */
                    destination: Phaser.Math.Vector2;

                    /**
                     * The ease function to use during the pan.
                     */
                    ease: Function;

                    /**
                     * If this effect is running this holds the current percentage of the progress, a value between 0 and 1.
                     */
                    progress: number;

                    /**
                     * This effect will scroll the Camera so that the center of its viewport finishes at the given destination,
                     * over the duration and with the ease specified.
                     * @param x The destination x coordinate to scroll the center of the Camera viewport to.
                     * @param y The destination y coordinate to scroll the center of the Camera viewport to.
                     * @param duration The duration of the effect in milliseconds. Default 1000.
                     * @param ease The ease to use for the pan. Can be any of the Phaser Easing constants or a custom function. Default 'Linear'.
                     * @param force Force the pan effect to start immediately, even if already running. Default false.
                     * @param callback This callback will be invoked every frame for the duration of the effect.
                     * It is sent four arguments: A reference to the camera, a progress amount between 0 and 1 indicating how complete the effect is,
                     * the current camera scroll x coordinate and the current camera scroll y coordinate.
                     * @param context The context in which the callback is invoked. Defaults to the Scene to which the Camera belongs.
                     */
                    start(x: number, y: number, duration?: number, ease?: string | Function, force?: boolean, callback?: Phaser.Types.Cameras.Scene2D.CameraPanCallback, context?: any): Phaser.Cameras.Scene2D.Camera;

                    /**
                     * The main update loop for this effect. Called automatically by the Camera.
                     * @param time The current timestamp as generated by the Request Animation Frame or SetTimeout.
                     * @param delta The delta time, in ms, elapsed since the last frame.
                     */
                    update(time: number, delta: number): void;

                    /**
                     * Called internally when the effect completes.
                     */
                    effectComplete(): void;

                    /**
                     * Resets this camera effect.
                     * If it was previously running, it stops instantly without calling its onComplete callback or emitting an event.
                     */
                    reset(): void;

                    /**
                     * Destroys this effect, releasing it from the Camera.
                     */
                    destroy(): void;

                }

                /**
                 * A Camera Rotate effect.
                 * 
                 * This effect will rotate the Camera so that the its viewport finishes at the given angle in radians,
                 * over the duration and with the ease specified.
                 * 
                 * Camera rotation always takes place based on the Camera viewport. By default, rotation happens
                 * in the center of the viewport. You can adjust this with the `originX` and `originY` properties.
                 * 
                 * Rotation influences the rendering of _all_ Game Objects visible by this Camera. However, it does not
                 * rotate the Camera viewport itself, which always remains an axis-aligned rectangle.
                 * 
                 * Only the camera is rotates. None of the objects it is displaying are impacted, i.e. their positions do
                 * not change.
                 * 
                 * The effect will dispatch several events on the Camera itself and you can also specify an `onUpdate` callback,
                 * which is invoked each frame for the duration of the effect if required.
                 */
                class RotateTo {
                    /**
                     * 
                     * @param camera The camera this effect is acting upon.
                     */
                    constructor(camera: Phaser.Cameras.Scene2D.Camera);

                    /**
                     * The Camera this effect belongs to.
                     */
                    readonly camera: Phaser.Cameras.Scene2D.Camera;

                    /**
                     * Is this effect actively running?
                     */
                    readonly isRunning: boolean;

                    /**
                     * The duration of the effect, in milliseconds.
                     */
                    readonly duration: number;

                    /**
                     * The starting angle to rotate the camera from.
                     */
                    source: number;

                    /**
                     * The constantly updated value based on the force.
                     */
                    current: number;

                    /**
                     * The destination angle in radians to rotate the camera to.
                     */
                    destination: number;

                    /**
                     * The ease function to use during the Rotate.
                     */
                    ease: Function;

                    /**
                     * If this effect is running this holds the current percentage of the progress, a value between 0 and 1.
                     */
                    progress: number;

                    /**
                     * The direction of the rotation.
                     */
                    clockwise: boolean;

                    /**
                     * The shortest direction to the target rotation.
                     */
                    shortestPath: boolean;

                    /**
                     * This effect will scroll the Camera so that the center of its viewport finishes at the given angle,
                     * over the duration and with the ease specified.
                     * @param radians The destination angle in radians to rotate the Camera viewport to. If the angle is positive then the rotation is clockwise else anticlockwise
                     * @param shortestPath If shortest path is set to true the camera will rotate in the quickest direction clockwise or anti-clockwise. Default false.
                     * @param duration The duration of the effect in milliseconds. Default 1000.
                     * @param ease The ease to use for the Rotate. Can be any of the Phaser Easing constants or a custom function. Default 'Linear'.
                     * @param force Force the rotation effect to start immediately, even if already running. Default false.
                     * @param callback This callback will be invoked every frame for the duration of the effect.
                     * It is sent four arguments: A reference to the camera, a progress amount between 0 and 1 indicating how complete the effect is,
                     * the current camera scroll x coordinate and the current camera scroll y coordinate.
                     * @param context The context in which the callback is invoked. Defaults to the Scene to which the Camera belongs.
                     */
                    start(radians: number, shortestPath?: boolean, duration?: number, ease?: string | Function, force?: boolean, callback?: CameraRotateCallback, context?: any): Phaser.Cameras.Scene2D.Camera;

                    /**
                     * The main update loop for this effect. Called automatically by the Camera.
                     * @param time The current timestamp as generated by the Request Animation Frame or SetTimeout.
                     * @param delta The delta time, in ms, elapsed since the last frame.
                     */
                    update(time: number, delta: number): void;

                    /**
                     * Called internally when the effect completes.
                     */
                    effectComplete(): void;

                    /**
                     * Resets this camera effect.
                     * If it was previously running, it stops instantly without calling its onComplete callback or emitting an event.
                     */
                    reset(): void;

                    /**
                     * Destroys this effect, releasing it from the Camera.
                     */
                    destroy(): void;

                }

                /**
                 * A Camera Shake effect.
                 * 
                 * This effect will shake the camera viewport by a random amount, bounded by the specified intensity, each frame.
                 * 
                 * Only the camera viewport is moved. None of the objects it is displaying are impacted, i.e. their positions do
                 * not change.
                 * 
                 * The effect will dispatch several events on the Camera itself and you can also specify an `onUpdate` callback,
                 * which is invoked each frame for the duration of the effect if required.
                 */
                class Shake {
                    /**
                     * 
                     * @param camera The camera this effect is acting upon.
                     */
                    constructor(camera: Phaser.Cameras.Scene2D.Camera);

                    /**
                     * The Camera this effect belongs to.
                     */
                    readonly camera: Phaser.Cameras.Scene2D.Camera;

                    /**
                     * Is this effect actively running?
                     */
                    readonly isRunning: boolean;

                    /**
                     * The duration of the effect, in milliseconds.
                     */
                    readonly duration: number;

                    /**
                     * The intensity of the effect. Use small float values. The default when the effect starts is 0.05.
                     * This is a Vector2 object, allowing you to control the shake intensity independently across x and y.
                     * You can modify this value while the effect is active to create more varied shake effects.
                     */
                    intensity: Phaser.Math.Vector2;

                    /**
                     * If this effect is running this holds the current percentage of the progress, a value between 0 and 1.
                     */
                    progress: number;

                    /**
                     * Shakes the Camera by the given intensity over the duration specified.
                     * @param duration The duration of the effect in milliseconds. Default 100.
                     * @param intensity The intensity of the shake. Default 0.05.
                     * @param force Force the shake effect to start immediately, even if already running. Default false.
                     * @param callback This callback will be invoked every frame for the duration of the effect.
                     * It is sent two arguments: A reference to the camera and a progress amount between 0 and 1 indicating how complete the effect is.
                     * @param context The context in which the callback is invoked. Defaults to the Scene to which the Camera belongs.
                     */
                    start(duration?: number, intensity?: number | Phaser.Math.Vector2, force?: boolean, callback?: Phaser.Types.Cameras.Scene2D.CameraShakeCallback, context?: any): Phaser.Cameras.Scene2D.Camera;

                    /**
                     * The pre-render step for this effect. Called automatically by the Camera.
                     */
                    preRender(): void;

                    /**
                     * The main update loop for this effect. Called automatically by the Camera.
                     * @param time The current timestamp as generated by the Request Animation Frame or SetTimeout.
                     * @param delta The delta time, in ms, elapsed since the last frame.
                     */
                    update(time: number, delta: number): void;

                    /**
                     * Called internally when the effect completes.
                     */
                    effectComplete(): void;

                    /**
                     * Resets this camera effect.
                     * If it was previously running, it stops instantly without calling its onComplete callback or emitting an event.
                     */
                    reset(): void;

                    /**
                     * Destroys this effect, releasing it from the Camera.
                     */
                    destroy(): void;

                }

                /**
                 * A Camera Zoom effect.
                 * 
                 * This effect will zoom the Camera to the given scale, over the duration and with the ease specified.
                 * 
                 * The effect will dispatch several events on the Camera itself and you can also specify an `onUpdate` callback,
                 * which is invoked each frame for the duration of the effect if required.
                 */
                class Zoom {
                    /**
                     * 
                     * @param camera The camera this effect is acting upon.
                     */
                    constructor(camera: Phaser.Cameras.Scene2D.Camera);

                    /**
                     * The Camera this effect belongs to.
                     */
                    readonly camera: Phaser.Cameras.Scene2D.Camera;

                    /**
                     * Is this effect actively running?
                     */
                    readonly isRunning: boolean;

                    /**
                     * The duration of the effect, in milliseconds.
                     */
                    readonly duration: number;

                    /**
                     * The starting zoom value;
                     */
                    source: number;

                    /**
                     * The destination zoom value.
                     */
                    destination: number;

                    /**
                     * The ease function to use during the zoom.
                     */
                    ease: Function;

                    /**
                     * If this effect is running this holds the current percentage of the progress, a value between 0 and 1.
                     */
                    progress: number;

                    /**
                     * This effect will zoom the Camera to the given scale, over the duration and with the ease specified.
                     * @param zoom The target Camera zoom value.
                     * @param duration The duration of the effect in milliseconds. Default 1000.
                     * @param ease The ease to use for the Zoom. Can be any of the Phaser Easing constants or a custom function. Default 'Linear'.
                     * @param force Force the zoom effect to start immediately, even if already running. Default false.
                     * @param callback This callback will be invoked every frame for the duration of the effect.
                     * It is sent three arguments: A reference to the camera, a progress amount between 0 and 1 indicating how complete the effect is,
                     * and the current camera zoom value.
                     * @param context The context in which the callback is invoked. Defaults to the Scene to which the Camera belongs.
                     */
                    start(zoom: number, duration?: number, ease?: string | Function, force?: boolean, callback?: Phaser.Types.Cameras.Scene2D.CameraZoomCallback, context?: any): Phaser.Cameras.Scene2D.Camera;

                    /**
                     * The main update loop for this effect. Called automatically by the Camera.
                     * @param time The current timestamp as generated by the Request Animation Frame or SetTimeout.
                     * @param delta The delta time, in ms, elapsed since the last frame.
                     */
                    update(time: number, delta: number): void;

                    /**
                     * Called internally when the effect completes.
                     */
                    effectComplete(): void;

                    /**
                     * Resets this camera effect.
                     * If it was previously running, it stops instantly without calling its onComplete callback or emitting an event.
                     */
                    reset(): void;

                    /**
                     * Destroys this effect, releasing it from the Camera.
                     */
                    destroy(): void;

                }

            }

            namespace Events {
                /**
                 * The Destroy Camera Event.
                 * 
                 * This event is dispatched by a Camera instance when it is destroyed by the Camera Manager.
                 */
                const DESTROY: any;

                /**
                 * The Camera Fade In Complete Event.
                 * 
                 * This event is dispatched by a Camera instance when the Fade In Effect completes.
                 * 
                 * Listen to it from a Camera instance using `Camera.on('camerafadeincomplete', listener)`.
                 */
                const FADE_IN_COMPLETE: any;

                /**
                 * The Camera Fade In Start Event.
                 * 
                 * This event is dispatched by a Camera instance when the Fade In Effect starts.
                 * 
                 * Listen to it from a Camera instance using `Camera.on('camerafadeinstart', listener)`.
                 */
                const FADE_IN_START: any;

                /**
                 * The Camera Fade Out Complete Event.
                 * 
                 * This event is dispatched by a Camera instance when the Fade Out Effect completes.
                 * 
                 * Listen to it from a Camera instance using `Camera.on('camerafadeoutcomplete', listener)`.
                 */
                const FADE_OUT_COMPLETE: any;

                /**
                 * The Camera Fade Out Start Event.
                 * 
                 * This event is dispatched by a Camera instance when the Fade Out Effect starts.
                 * 
                 * Listen to it from a Camera instance using `Camera.on('camerafadeoutstart', listener)`.
                 */
                const FADE_OUT_START: any;

                /**
                 * The Camera Flash Complete Event.
                 * 
                 * This event is dispatched by a Camera instance when the Flash Effect completes.
                 */
                const FLASH_COMPLETE: any;

                /**
                 * The Camera Flash Start Event.
                 * 
                 * This event is dispatched by a Camera instance when the Flash Effect starts.
                 */
                const FLASH_START: any;

                /**
                 * The Camera Follower Update Event.
                 * 
                 * This event is dispatched by a Camera instance when it is following a
                 * Game Object and the Camera position has been updated as a result of
                 * that following.
                 * 
                 * Listen to it from a Camera instance using: `camera.on('followupdate', listener)`.
                 */
                const FOLLOW_UPDATE: any;

                /**
                 * The Camera Pan Complete Event.
                 * 
                 * This event is dispatched by a Camera instance when the Pan Effect completes.
                 */
                const PAN_COMPLETE: any;

                /**
                 * The Camera Pan Start Event.
                 * 
                 * This event is dispatched by a Camera instance when the Pan Effect starts.
                 */
                const PAN_START: any;

                /**
                 * The Camera Post-Render Event.
                 * 
                 * This event is dispatched by a Camera instance after is has finished rendering.
                 * It is only dispatched if the Camera is rendering to a texture.
                 * 
                 * Listen to it from a Camera instance using: `camera.on('postrender', listener)`.
                 */
                const POST_RENDER: any;

                /**
                 * The Camera Pre-Render Event.
                 * 
                 * This event is dispatched by a Camera instance when it is about to render.
                 * It is only dispatched if the Camera is rendering to a texture.
                 * 
                 * Listen to it from a Camera instance using: `camera.on('prerender', listener)`.
                 */
                const PRE_RENDER: any;

                /**
                 * The Camera Rotate Complete Event.
                 * 
                 * This event is dispatched by a Camera instance when the Rotate Effect completes.
                 */
                const ROTATE_COMPLETE: any;

                /**
                 * The Camera Rotate Start Event.
                 * 
                 * This event is dispatched by a Camera instance when the Rotate Effect starts.
                 */
                const ROTATE_START: any;

                /**
                 * The Camera Shake Complete Event.
                 * 
                 * This event is dispatched by a Camera instance when the Shake Effect completes.
                 */
                const SHAKE_COMPLETE: any;

                /**
                 * The Camera Shake Start Event.
                 * 
                 * This event is dispatched by a Camera instance when the Shake Effect starts.
                 */
                const SHAKE_START: any;

                /**
                 * The Camera Zoom Complete Event.
                 * 
                 * This event is dispatched by a Camera instance when the Zoom Effect completes.
                 */
                const ZOOM_COMPLETE: any;

                /**
                 * The Camera Zoom Start Event.
                 * 
                 * This event is dispatched by a Camera instance when the Zoom Effect starts.
                 */
                const ZOOM_START: any;

            }

        }

        namespace Controls {
            /**
             * A Fixed Key Camera Control.
             * 
             * This allows you to control the movement and zoom of a camera using the defined keys.
             * 
             * ```javascript
             * var camControl = new FixedKeyControl({
             *     camera: this.cameras.main,
             *     left: cursors.left,
             *     right: cursors.right,
             *     speed: float OR { x: 0, y: 0 }
             * });
             * ```
             * 
             * Movement is precise and has no 'smoothing' applied to it.
             * 
             * You must call the `update` method of this controller every frame.
             */
            class FixedKeyControl {
                /**
                 * 
                 * @param config The Fixed Key Control configuration object.
                 */
                constructor(config: Phaser.Types.Cameras.Controls.FixedKeyControlConfig);

                /**
                 * The Camera that this Control will update.
                 */
                camera: Phaser.Cameras.Scene2D.Camera;

                /**
                 * The Key to be pressed that will move the Camera left.
                 */
                left: Phaser.Input.Keyboard.Key;

                /**
                 * The Key to be pressed that will move the Camera right.
                 */
                right: Phaser.Input.Keyboard.Key;

                /**
                 * The Key to be pressed that will move the Camera up.
                 */
                up: Phaser.Input.Keyboard.Key;

                /**
                 * The Key to be pressed that will move the Camera down.
                 */
                down: Phaser.Input.Keyboard.Key;

                /**
                 * The Key to be pressed that will zoom the Camera in.
                 */
                zoomIn: Phaser.Input.Keyboard.Key;

                /**
                 * The Key to be pressed that will zoom the Camera out.
                 */
                zoomOut: Phaser.Input.Keyboard.Key;

                /**
                 * The speed at which the camera will zoom if the `zoomIn` or `zoomOut` keys are pressed.
                 */
                zoomSpeed: number;

                /**
                 * The smallest zoom value the camera will reach when zoomed out.
                 */
                minZoom: number;

                /**
                 * The largest zoom value the camera will reach when zoomed in.
                 */
                maxZoom: number;

                /**
                 * The horizontal speed the camera will move.
                 */
                speedX: number;

                /**
                 * The vertical speed the camera will move.
                 */
                speedY: number;

                /**
                 * A flag controlling if the Controls will update the Camera or not.
                 */
                active: boolean;

                /**
                 * Starts the Key Control running, providing it has been linked to a camera.
                 */
                start(): this;

                /**
                 * Stops this Key Control from running. Call `start` to start it again.
                 */
                stop(): this;

                /**
                 * Binds this Key Control to a camera.
                 * @param camera The camera to bind this Key Control to.
                 */
                setCamera(camera: Phaser.Cameras.Scene2D.Camera): this;

                /**
                 * Applies the results of pressing the control keys to the Camera.
                 * 
                 * You must call this every step, it is not called automatically.
                 * @param delta The delta time in ms since the last frame. This is a smoothed and capped value based on the FPS rate.
                 */
                update(delta: number): void;

                /**
                 * Destroys this Key Control.
                 */
                destroy(): void;

            }

            /**
             * A Smoothed Key Camera Control.
             * 
             * This allows you to control the movement and zoom of a camera using the defined keys.
             * Unlike the Fixed Camera Control you can also provide physics values for acceleration, drag and maxSpeed for smoothing effects.
             * 
             * ```javascript
             * var controlConfig = {
             *     camera: this.cameras.main,
             *     left: cursors.left,
             *     right: cursors.right,
             *     up: cursors.up,
             *     down: cursors.down,
             *     zoomIn: this.input.keyboard.addKey(Phaser.Input.Keyboard.KeyCodes.Q),
             *     zoomOut: this.input.keyboard.addKey(Phaser.Input.Keyboard.KeyCodes.E),
             *     zoomSpeed: 0.02,
             *     acceleration: 0.06,
             *     drag: 0.0005,
             *     maxSpeed: 1.0
             * };
             * ```
             * 
             * You must call the `update` method of this controller every frame.
             */
            class SmoothedKeyControl {
                /**
                 * 
                 * @param config The Smoothed Key Control configuration object.
                 */
                constructor(config: Phaser.Types.Cameras.Controls.SmoothedKeyControlConfig);

                /**
                 * The Camera that this Control will update.
                 */
                camera: Phaser.Cameras.Scene2D.Camera;

                /**
                 * The Key to be pressed that will move the Camera left.
                 */
                left: Phaser.Input.Keyboard.Key;

                /**
                 * The Key to be pressed that will move the Camera right.
                 */
                right: Phaser.Input.Keyboard.Key;

                /**
                 * The Key to be pressed that will move the Camera up.
                 */
                up: Phaser.Input.Keyboard.Key;

                /**
                 * The Key to be pressed that will move the Camera down.
                 */
                down: Phaser.Input.Keyboard.Key;

                /**
                 * The Key to be pressed that will zoom the Camera in.
                 */
                zoomIn: Phaser.Input.Keyboard.Key;

                /**
                 * The Key to be pressed that will zoom the Camera out.
                 */
                zoomOut: Phaser.Input.Keyboard.Key;

                /**
                 * The speed at which the camera will zoom if the `zoomIn` or `zoomOut` keys are pressed.
                 */
                zoomSpeed: number;

                /**
                 * The smallest zoom value the camera will reach when zoomed out.
                 */
                minZoom: number;

                /**
                 * The largest zoom value the camera will reach when zoomed in.
                 */
                maxZoom: number;

                /**
                 * The horizontal acceleration the camera will move.
                 */
                accelX: number;

                /**
                 * The vertical acceleration the camera will move.
                 */
                accelY: number;

                /**
                 * The horizontal drag applied to the camera when it is moving.
                 */
                dragX: number;

                /**
                 * The vertical drag applied to the camera when it is moving.
                 */
                dragY: number;

                /**
                 * The maximum horizontal speed the camera will move.
                 */
                maxSpeedX: number;

                /**
                 * The maximum vertical speed the camera will move.
                 */
                maxSpeedY: number;

                /**
                 * A flag controlling if the Controls will update the Camera or not.
                 */
                active: boolean;

                /**
                 * Starts the Key Control running, providing it has been linked to a camera.
                 */
                start(): this;

                /**
                 * Stops this Key Control from running. Call `start` to start it again.
                 */
                stop(): this;

                /**
                 * Binds this Key Control to a camera.
                 * @param camera The camera to bind this Key Control to.
                 */
                setCamera(camera: Phaser.Cameras.Scene2D.Camera): this;

                /**
                 * Applies the results of pressing the control keys to the Camera.
                 * 
                 * You must call this every step, it is not called automatically.
                 * @param delta The delta time in ms since the last frame. This is a smoothed and capped value based on the FPS rate.
                 */
                update(delta: number): void;

                /**
                 * Destroys this Key Control.
                 */
                destroy(): void;

            }

        }

    }

    /**
     * Phaser Release Version
     */
    const VERSION: string;

    /**
     * This setting will auto-detect if the browser is capable of suppporting WebGL.
     * If it is, it will use the WebGL Renderer. If not, it will fall back to the Canvas Renderer.
     */
    const AUTO: number;

    /**
     * Forces Phaser to only use the Canvas Renderer, regardless if the browser supports
     * WebGL or not.
     */
    const CANVAS: number;

    /**
     * Forces Phaser to use the WebGL Renderer. If the browser does not support it, there is
     * no fallback to Canvas with this setting, so you should trap it and display a suitable
     * message to the user.
     */
    const WEBGL: number;

    /**
     * A Headless Renderer doesn't create either a Canvas or WebGL Renderer. However, it still
     * absolutely relies on the DOM being present and available. This mode is meant for unit testing,
     * not for running Phaser on the server, which is something you really shouldn't do.
     */
    const HEADLESS: number;

    /**
     * In Phaser the value -1 means 'forever' in lots of cases, this const allows you to use it instead
     * to help you remember what the value is doing in your code.
     */
    const FOREVER: number;

    /**
     * Direction constant.
     */
    const NONE: number;

    /**
     * Direction constant.
     */
    const UP: number;

    /**
     * Direction constant.
     */
    const DOWN: number;

    /**
     * Direction constant.
     */
    const LEFT: number;

    /**
     * Direction constant.
     */
    const RIGHT: number;

    /**
     * The Phaser.Game instance is the main controller for the entire Phaser game. It is responsible
     * for handling the boot process, parsing the configuration values, creating the renderer,
     * and setting-up all of the global Phaser systems, such as sound and input.
     * Once that is complete it will start the Scene Manager and then begin the main game loop.
     * 
     * You should generally avoid accessing any of the systems created by Game, and instead use those
     * made available to you via the Phaser.Scene Systems class instead.
     */
    class Game {
        /**
         * 
         * @param GameConfig The configuration object for your Phaser Game instance.
         */
        constructor(GameConfig?: Phaser.Types.Core.GameConfig);

        /**
         * The parsed Game Configuration object.
         * 
         * The values stored within this object are read-only and should not be changed at run-time.
         */
        readonly config: Phaser.Core.Config;

        /**
         * A reference to either the Canvas or WebGL Renderer that this Game is using.
         */
        renderer: Phaser.Renderer.Canvas.CanvasRenderer | Phaser.Renderer.WebGL.WebGLRenderer;

        /**
         * A reference to an HTML Div Element used as the DOM Element Container.
         * 
         * Only set if `createDOMContainer` is `true` in the game config (by default it is `false`) and
         * if you provide a parent element to insert the Phaser Game inside.
         * 
         * See the DOM Element Game Object for more details.
         */
        domContainer: HTMLDivElement;

        /**
         * A reference to the HTML Canvas Element that Phaser uses to render the game.
         * This is created automatically by Phaser unless you provide a `canvas` property
         * in your Game Config.
         */
        canvas: HTMLCanvasElement;

        /**
         * A reference to the Rendering Context belonging to the Canvas Element this game is rendering to.
         * If the game is running under Canvas it will be a 2d Canvas Rendering Context.
         * If the game is running under WebGL it will be a WebGL Rendering Context.
         * This context is created automatically by Phaser unless you provide a `context` property
         * in your Game Config.
         */
        context: CanvasRenderingContext2D | WebGLRenderingContext;

        /**
         * A flag indicating when this Game instance has finished its boot process.
         */
        readonly isBooted: boolean;

        /**
         * A flag indicating if this Game is currently running its game step or not.
         */
        readonly isRunning: boolean;

        /**
         * An Event Emitter which is used to broadcast game-level events from the global systems.
         */
        events: Phaser.Events.EventEmitter;

        /**
         * An instance of the Animation Manager.
         * 
         * The Animation Manager is a global system responsible for managing all animations used within your game.
         */
        anims: Phaser.Animations.AnimationManager;

        /**
         * An instance of the Texture Manager.
         * 
         * The Texture Manager is a global system responsible for managing all textures being used by your game.
         */
        textures: Phaser.Textures.TextureManager;

        /**
         * An instance of the Cache Manager.
         * 
         * The Cache Manager is a global system responsible for caching, accessing and releasing external game assets.
         */
        cache: Phaser.Cache.CacheManager;

        /**
         * An instance of the Data Manager
         */
        registry: Phaser.Data.DataManager;

        /**
         * An instance of the Input Manager.
         * 
         * The Input Manager is a global system responsible for the capture of browser-level input events.
         */
        input: Phaser.Input.InputManager;

        /**
         * An instance of the Scene Manager.
         * 
         * The Scene Manager is a global system responsible for creating, modifying and updating the Scenes in your game.
         */
        scene: Phaser.Scenes.SceneManager;

        /**
         * A reference to the Device inspector.
         * 
         * Contains information about the device running this game, such as OS, browser vendor and feature support.
         * Used by various systems to determine capabilities and code paths.
         */
        device: Phaser.DeviceConf;

        /**
         * An instance of the Scale Manager.
         * 
         * The Scale Manager is a global system responsible for handling scaling of the game canvas.
         */
        scale: Phaser.Scale.ScaleManager;

        /**
         * An instance of the base Sound Manager.
         * 
         * The Sound Manager is a global system responsible for the playback and updating of all audio in your game.
         * 
         * You can disable the inclusion of the Sound Manager in your build by toggling the webpack `FEATURE_SOUND` flag.
         */
        sound: Phaser.Sound.NoAudioSoundManager | Phaser.Sound.HTML5AudioSoundManager | Phaser.Sound.WebAudioSoundManager;

        /**
         * An instance of the Time Step.
         * 
         * The Time Step is a global system responsible for setting-up and responding to the browser frame events, processing
         * them and calculating delta values. It then automatically calls the game step.
         */
        loop: Phaser.Core.TimeStep;

        /**
         * An instance of the Plugin Manager.
         * 
         * The Plugin Manager is a global system that allows plugins to register themselves with it, and can then install
         * those plugins into Scenes as required.
         */
        plugins: Phaser.Plugins.PluginManager;

        /**
         * An instance of the Facebook Instant Games Plugin.
         * 
         * This will only be available if the plugin has been built into Phaser,
         * or you're using the special Facebook Instant Games custom build.
         */
        facebook: Phaser.FacebookInstantGamesPlugin;

        /**
         * Does the window the game is running in currently have focus or not?
         * This is modified by the VisibilityHandler.
         */
        readonly hasFocus: boolean;

        /**
         * This method is called automatically when the DOM is ready. It is responsible for creating the renderer,
         * displaying the Debug Header, adding the game canvas to the DOM and emitting the 'boot' event.
         * It listens for a 'ready' event from the base systems and once received it will call `Game.start`.
         */
        protected boot(): void;

        /**
         * Called automatically by Game.boot once all of the global systems have finished setting themselves up.
         * By this point the Game is now ready to start the main loop running.
         * It will also enable the Visibility Handler.
         */
        protected start(): void;

        /**
         * The main Game Step. Called automatically by the Time Step, once per browser frame (typically as a result of
         * Request Animation Frame, or Set Timeout on very old browsers.)
         * 
         * The step will update the global managers first, then proceed to update each Scene in turn, via the Scene Manager.
         * 
         * It will then render each Scene in turn, via the Renderer. This process emits `prerender` and `postrender` events.
         * @param time The current time. Either a High Resolution Timer value if it comes from Request Animation Frame, or Date.now if using SetTimeout.
         * @param delta The delta time in ms since the last frame. This is a smoothed and capped value based on the FPS rate.
         */
        step(time: number, delta: number): void;

        /**
         * A special version of the Game Step for the HEADLESS renderer only.
         * 
         * The main Game Step. Called automatically by the Time Step, once per browser frame (typically as a result of
         * Request Animation Frame, or Set Timeout on very old browsers.)
         * 
         * The step will update the global managers first, then proceed to update each Scene in turn, via the Scene Manager.
         * 
         * This process emits `prerender` and `postrender` events, even though nothing actually displays.
         * @param time The current time. Either a High Resolution Timer value if it comes from Request Animation Frame, or Date.now if using SetTimeout.
         * @param delta The delta time in ms since the last frame. This is a smoothed and capped value based on the FPS rate.
         */
        headlessStep(time: number, delta: number): void;

        /**
         * Called automatically by the Visibility Handler.
         * This will pause the main loop and then emit a pause event.
         */
        protected onHidden(): void;

        /**
         * Called automatically by the Visibility Handler.
         * This will resume the main loop and then emit a resume event.
         */
        protected onVisible(): void;

        /**
         * Called automatically by the Visibility Handler.
         * This will set the main loop into a 'blurred' state, which pauses it.
         */
        protected onBlur(): void;

        /**
         * Called automatically by the Visibility Handler.
         * This will set the main loop into a 'focused' state, which resumes it.
         */
        protected onFocus(): void;

        /**
         * Returns the current game frame.
         * 
         * When the game starts running, the frame is incremented every time Request Animation Frame, or Set Timeout, fires.
         */
        getFrame(): number;

        /**
         * Returns the time that the current game step started at, as based on `performance.now`.
         */
        getTime(): number;

        /**
         * Flags this Game instance as needing to be destroyed on the _next frame_, making this an asynchronous operation.
         * 
         * It will wait until the current frame has completed and then call `runDestroy` internally.
         * 
         * If you need to react to the games eventual destruction, listen for the `DESTROY` event.
         * 
         * If you **do not** need to run Phaser again on the same web page you can set the `noReturn` argument to `true` and it will free-up
         * memory being held by the core Phaser plugins. If you do need to create another game instance on the same page, leave this as `false`.
         * @param removeCanvas Set to `true` if you would like the parent canvas element removed from the DOM, or `false` to leave it in place.
         * @param noReturn If `true` all the core Phaser plugins are destroyed. You cannot create another instance of Phaser on the same web page if you do this. Default false.
         */
        destroy(removeCanvas: boolean, noReturn?: boolean): void;

    }

    namespace Core {
        /**
         * The active game configuration settings, parsed from a {@link Phaser.Types.Core.GameConfig} object.
         */
        class Config {
            /**
             * 
             * @param GameConfig The configuration object for your Phaser Game instance.
             */
            constructor(GameConfig?: Phaser.Types.Core.GameConfig);

            /**
             * The width of the underlying canvas, in pixels.
             */
            readonly width: number | string;

            /**
             * The height of the underlying canvas, in pixels.
             */
            readonly height: number | string;

            /**
             * The zoom factor, as used by the Scale Manager.
             */
            readonly zoom: Phaser.Scale.ZoomType | number;

            /**
             * A parent DOM element into which the canvas created by the renderer will be injected.
             */
            readonly parent: any;

            /**
             * The scale mode as used by the Scale Manager. The default is zero, which is no scaling.
             */
            readonly scaleMode: Phaser.Scale.ScaleModeType;

            /**
             * Is the Scale Manager allowed to adjust the CSS height property of the parent to be 100%?
             */
            readonly expandParent: boolean;

            /**
             * Automatically round the display and style sizes of the canvas. This can help with performance in lower-powered devices.
             */
            readonly autoRound: boolean;

            /**
             * Automatically center the canvas within the parent?
             */
            readonly autoCenter: Phaser.Scale.CenterType;

            /**
             * How many ms should elapse before checking if the browser size has changed?
             */
            readonly resizeInterval: number;

            /**
             * The DOM element that will be sent into full screen mode, or its `id`. If undefined Phaser will create its own div and insert the canvas into it when entering fullscreen mode.
             */
            readonly fullscreenTarget: HTMLElement | string;

            /**
             * The minimum width, in pixels, the canvas will scale down to. A value of zero means no minimum.
             */
            readonly minWidth: number;

            /**
             * The maximum width, in pixels, the canvas will scale up to. A value of zero means no maximum.
             */
            readonly maxWidth: number;

            /**
             * The minimum height, in pixels, the canvas will scale down to. A value of zero means no minimum.
             */
            readonly minHeight: number;

            /**
             * The maximum height, in pixels, the canvas will scale up to. A value of zero means no maximum.
             */
            readonly maxHeight: number;

            /**
             * Force Phaser to use a specific renderer. Can be `CONST.CANVAS`, `CONST.WEBGL`, `CONST.HEADLESS` or `CONST.AUTO` (default)
             */
            readonly renderType: number;

            /**
             * Force Phaser to use your own Canvas element instead of creating one.
             */
            readonly canvas: HTMLCanvasElement;

            /**
             * Force Phaser to use your own Canvas context instead of creating one.
             */
            readonly context: CanvasRenderingContext2D | WebGLRenderingContext;

            /**
             * Optional CSS attributes to be set on the canvas object created by the renderer.
             */
            readonly canvasStyle: string;

            /**
             * Is Phaser running under a custom (non-native web) environment? If so, set this to `true` to skip internal Feature detection. If `true` the `renderType` cannot be left as `AUTO`.
             */
            readonly customEnvironment: boolean;

            /**
             * The default Scene configuration object.
             */
            readonly sceneConfig: object;

            /**
             * A seed which the Random Data Generator will use. If not given, a dynamic seed based on the time is used.
             */
            readonly seed: string[];

            /**
             * The title of the game.
             */
            readonly gameTitle: string;

            /**
             * The URL of the game.
             */
            readonly gameURL: string;

            /**
             * The version of the game.
             */
            readonly gameVersion: string;

            /**
             * If `true` the window will automatically be given focus immediately and on any future mousedown event.
             */
            readonly autoFocus: boolean;

            /**
             * Should the game create a div element to act as a DOM Container? Only enable if you're using DOM Element objects. You must provide a parent object if you use this feature.
             */
            readonly domCreateContainer: boolean;

            /**
             * Should the DOM Container that is created (if `dom.createContainer` is true) be positioned behind (true) or over the top (false, the default) of the game canvas?
             */
            readonly domBehindCanvas: boolean;

            /**
             * The default `pointerEvents` attribute set on the DOM Container.
             */
            readonly domPointerEvents: string;

            /**
             * Enable the Keyboard Plugin. This can be disabled in games that don't need keyboard input.
             */
            readonly inputKeyboard: boolean;

            /**
             * The DOM Target to listen for keyboard events on. Defaults to `window` if not specified.
             */
            readonly inputKeyboardEventTarget: any;

            /**
             * `preventDefault` will be called on every non-modified key which has a key code in this array. By default, it is empty.
             */
            readonly inputKeyboardCapture: number[];

            /**
             * Enable the Mouse Plugin. This can be disabled in games that don't need mouse input.
             */
            readonly inputMouse: boolean | object;

            /**
             * The DOM Target to listen for mouse events on. Defaults to the game canvas if not specified.
             */
            readonly inputMouseEventTarget: any;

            /**
             * Should `mousedown` DOM events have `preventDefault` called on them?
             */
            readonly inputMousePreventDefaultDown: boolean;

            /**
             * Should `mouseup` DOM events have `preventDefault` called on them?
             */
            readonly inputMousePreventDefaultUp: boolean;

            /**
             * Should `mousemove` DOM events have `preventDefault` called on them?
             */
            readonly inputMousePreventDefaultMove: boolean;

            /**
             * Should `wheel` DOM events have `preventDefault` called on them?
             */
            readonly inputMousePreventDefaultWheel: boolean;

            /**
             * Enable the Touch Plugin. This can be disabled in games that don't need touch input.
             */
            readonly inputTouch: boolean;

            /**
             * The DOM Target to listen for touch events on. Defaults to the game canvas if not specified.
             */
            readonly inputTouchEventTarget: any;

            /**
             * Should touch events be captured? I.e. have prevent default called on them.
             */
            readonly inputTouchCapture: boolean;

            /**
             * The number of Pointer objects created by default. In a mouse-only, or non-multi touch game, you can leave this as 1.
             */
            readonly inputActivePointers: number;

            /**
             * The smoothing factor to apply during Pointer movement. See {@link Phaser.Input.Pointer#smoothFactor}.
             */
            readonly inputSmoothFactor: number;

            /**
             * Should Phaser listen for input events on the Window? If you disable this, events like 'POINTER_UP_OUTSIDE' will no longer fire.
             */
            readonly inputWindowEvents: boolean;

            /**
             * Enable the Gamepad Plugin. This can be disabled in games that don't need gamepad input.
             */
            readonly inputGamepad: boolean;

            /**
             * The DOM Target to listen for gamepad events on. Defaults to `window` if not specified.
             */
            readonly inputGamepadEventTarget: any;

            /**
             * Set to `true` to disable the right-click context menu.
             */
            readonly disableContextMenu: boolean;

            /**
             * The Audio Configuration object.
             */
            readonly audio: Phaser.Types.Core.AudioConfig;

            /**
             * Don't write the banner line to the console.log.
             */
            readonly hideBanner: boolean;

            /**
             * Omit Phaser's name and version from the banner.
             */
            readonly hidePhaser: boolean;

            /**
             * The color of the banner text.
             */
            readonly bannerTextColor: string;

            /**
             * The background colors of the banner.
             */
            readonly bannerBackgroundColor: string[];

            /**
             * The Frame Rate Configuration object, as parsed by the Timestep class.
             */
            readonly fps: Phaser.Types.Core.FPSConfig;

            /**
             * An object mapping WebGL names to WebGLPipeline classes. These should be class constructors, not instances.
             */
            readonly pipeline: Phaser.Types.Core.PipelineConfig;

            /**
             * When set to `true`, WebGL uses linear interpolation to draw scaled or rotated textures, giving a smooth appearance. When set to `false`, WebGL uses nearest-neighbor interpolation, giving a crisper appearance. `false` also disables antialiasing of the game canvas itself, if the browser supports it, when the game canvas is scaled.
             */
            readonly antialias: boolean;

            /**
             * Sets the `antialias` property when the WebGL context is created. Setting this value does not impact any subsequent textures that are created, or the canvas style attributes.
             */
            readonly antialiasGL: boolean;

            /**
             * Sets the `mipmapFilter` property when the WebGL renderer is created.
             */
            readonly mipmapFilter: string;

            /**
             * When set to `true` it will create a desynchronized context for both 2D and WebGL. See https://developers.google.com/web/updates/2019/05/desynchronized for details.
             */
            readonly desynchronized: boolean;

            /**
             * Draw texture-based Game Objects at only whole-integer positions. Game Objects without textures, like Graphics, ignore this property.
             */
            readonly roundPixels: boolean;

            /**
             * Prevent pixel art from becoming blurred when scaled. It will remain crisp (tells the WebGL renderer to automatically create textures using a linear filter mode).
             */
            readonly pixelArt: boolean;

            /**
             * Whether the game canvas will have a transparent background.
             */
            readonly transparent: boolean;

            /**
             * Whether the game canvas will be cleared between each rendering frame. You can disable this if you have a full-screen background image or game object.
             */
            readonly clearBeforeRender: boolean;

            /**
             * If the value is true the WebGL buffers will not be cleared and will preserve their values until cleared or overwritten by the author.
             */
            readonly preserveDrawingBuffer: boolean;

            /**
             * In WebGL mode, sets the drawing buffer to contain colors with pre-multiplied alpha.
             */
            readonly premultipliedAlpha: boolean;

            /**
             * Let the browser abort creating a WebGL context if it judges performance would be unacceptable.
             */
            readonly failIfMajorPerformanceCaveat: boolean;

            /**
             * "high-performance", "low-power" or "default". A hint to the browser on how much device power the game might use.
             */
            readonly powerPreference: string;

            /**
             * The default WebGL Batch size. Represents the number of _quads_ that can be added to a single batch.
             */
            readonly batchSize: number;

            /**
             * When in WebGL mode, this sets the maximum number of GPU Textures to use. The default, -1, will use all available units. The WebGL1 spec says all browsers should provide a minimum of 8.
             */
            readonly maxTextures: number;

            /**
             * The maximum number of lights allowed to be visible within range of a single Camera in the LightManager.
             */
            readonly maxLights: number;

            /**
             * The background color of the game canvas. The default is black. This value is ignored if `transparent` is set to `true`.
             */
            readonly backgroundColor: Phaser.Display.Color;

            /**
             * Called before Phaser boots. Useful for initializing anything not related to Phaser that Phaser may require while booting.
             */
            readonly preBoot: Phaser.Types.Core.BootCallback;

            /**
             * A function to run at the end of the boot sequence. At this point, all the game systems have started and plugins have been loaded.
             */
            readonly postBoot: Phaser.Types.Core.BootCallback;

            /**
             * The Physics Configuration object.
             */
            readonly physics: Phaser.Types.Core.PhysicsConfig;

            /**
             * The default physics system. It will be started for each scene. Either 'arcade', 'impact' or 'matter'.
             */
            readonly defaultPhysicsSystem: boolean | string;

            /**
             * A URL used to resolve paths given to the loader. Example: 'http://labs.phaser.io/assets/'.
             */
            readonly loaderBaseURL: string;

            /**
             * A URL path used to resolve relative paths given to the loader. Example: 'images/sprites/'.
             */
            readonly loaderPath: string;

            /**
             * Maximum parallel downloads allowed for resources (Default to 32).
             */
            readonly loaderMaxParallelDownloads: number;

            /**
             * 'anonymous', 'use-credentials', or `undefined`. If you're not making cross-origin requests, leave this as `undefined`. See {@link https://developer.mozilla.org/en-US/docs/Web/HTML/CORS_settings_attributes}.
             */
            readonly loaderCrossOrigin: string | undefined;

            /**
             * The response type of the XHR request, e.g. `blob`, `text`, etc.
             */
            readonly loaderResponseType: string;

            /**
             * Should the XHR request use async or not?
             */
            readonly loaderAsync: boolean;

            /**
             * Optional username for all XHR requests.
             */
            readonly loaderUser: string;

            /**
             * Optional password for all XHR requests.
             */
            readonly loaderPassword: string;

            /**
             * Optional XHR timeout value, in ms.
             */
            readonly loaderTimeout: number;

            /**
             * Optional XHR withCredentials value.
             */
            readonly loaderWithCredentials: boolean;

            /**
             * An array of global plugins to be installed.
             */
            readonly installGlobalPlugins: any;

            /**
             * An array of Scene level plugins to be installed.
             */
            readonly installScenePlugins: any;

            /**
             * The plugins installed into every Scene (in addition to CoreScene and Global).
             */
            readonly defaultPlugins: any;

            /**
             * A base64 encoded PNG that will be used as the default blank texture.
             */
            readonly defaultImage: string;

            /**
             * A base64 encoded PNG that will be used as the default texture when a texture is assigned that is missing or not loaded.
             */
            readonly missingImage: string;

            /**
             * A base64 encoded PNG that will be used as the default texture when a texture is assigned that is white or not loaded.
             */
            readonly whiteImage: string;

        }

        /**
         * Called automatically by Phaser.Game and responsible for creating the renderer it will use.
         * 
         * Relies upon two webpack global flags to be defined: `WEBGL_RENDERER` and `CANVAS_RENDERER` during build time, but not at run-time.
         * @param game The Phaser.Game instance on which the renderer will be set.
         */
        function CreateRenderer(game: Phaser.Game): void;

        /**
         * Called automatically by Phaser.Game and responsible for creating the console.log debug header.
         * 
         * You can customize or disable the header via the Game Config object.
         * @param game The Phaser.Game instance which will output this debug header.
         */
        function DebugHeader(game: Phaser.Game): void;

        namespace Events {
            /**
             * The Game Blur Event.
             * 
             * This event is dispatched by the Game Visibility Handler when the window in which the Game instance is embedded
             * enters a blurred state. The blur event is raised when the window loses focus. This can happen if a user swaps
             * tab, or if they simply remove focus from the browser to another app.
             */
            const BLUR: any;

            /**
             * The Game Boot Event.
             * 
             * This event is dispatched when the Phaser Game instance has finished booting, but before it is ready to start running.
             * The global systems use this event to know when to set themselves up, dispatching their own `ready` events as required.
             */
            const BOOT: any;

            /**
             * The Game Context Lost Event.
             * 
             * This event is dispatched by the Game if the WebGL Renderer it is using encounters a WebGL Context Lost event from the browser.
             * 
             * The partner event is `CONTEXT_RESTORED`.
             */
            const CONTEXT_LOST: any;

            /**
             * The Game Context Restored Event.
             * 
             * This event is dispatched by the Game if the WebGL Renderer it is using encounters a WebGL Context Restored event from the browser.
             * 
             * The partner event is `CONTEXT_LOST`.
             */
            const CONTEXT_RESTORED: any;

            /**
             * The Game Destroy Event.
             * 
             * This event is dispatched when the game instance has been told to destroy itself.
             * Lots of internal systems listen to this event in order to clear themselves out.
             * Custom plugins and game code should also do the same.
             */
            const DESTROY: any;

            /**
             * The Game Focus Event.
             * 
             * This event is dispatched by the Game Visibility Handler when the window in which the Game instance is embedded
             * enters a focused state. The focus event is raised when the window re-gains focus, having previously lost it.
             */
            const FOCUS: any;

            /**
             * The Game Hidden Event.
             * 
             * This event is dispatched by the Game Visibility Handler when the document in which the Game instance is embedded
             * enters a hidden state. Only browsers that support the Visibility API will cause this event to be emitted.
             * 
             * In most modern browsers, when the document enters a hidden state, the Request Animation Frame and setTimeout, which
             * control the main game loop, will automatically pause. There is no way to stop this from happening. It is something
             * your game should account for in its own code, should the pause be an issue (i.e. for multiplayer games)
             */
            const HIDDEN: any;

            /**
             * The Game Pause Event.
             * 
             * This event is dispatched when the Game loop enters a paused state, usually as a result of the Visibility Handler.
             */
            const PAUSE: any;

            /**
             * The Game Post-Render Event.
             * 
             * This event is dispatched right at the end of the render process.
             * 
             * Every Scene will have rendered and been drawn to the canvas by the time this event is fired.
             * Use it for any last minute post-processing before the next game step begins.
             */
            const POST_RENDER: any;

            /**
             * The Game Post-Step Event.
             * 
             * This event is dispatched after the Scene Manager has updated.
             * Hook into it from plugins or systems that need to do things before the render starts.
             */
            const POST_STEP: any;

            /**
             * The Game Pre-Render Event.
             * 
             * This event is dispatched immediately before any of the Scenes have started to render.
             * 
             * The renderer will already have been initialized this frame, clearing itself and preparing to receive the Scenes for rendering, but it won't have actually drawn anything yet.
             */
            const PRE_RENDER: any;

            /**
             * The Game Pre-Step Event.
             * 
             * This event is dispatched before the main Game Step starts. By this point in the game cycle none of the Scene updates have yet happened.
             * Hook into it from plugins or systems that need to update before the Scene Manager does.
             */
            const PRE_STEP: any;

            /**
             * The Game Ready Event.
             * 
             * This event is dispatched when the Phaser Game instance has finished booting, the Texture Manager is fully ready,
             * and all local systems are now able to start.
             */
            const READY: any;

            /**
             * The Game Resume Event.
             * 
             * This event is dispatched when the game loop leaves a paused state and resumes running.
             */
            const RESUME: any;

            /**
             * The Game Step Event.
             * 
             * This event is dispatched after the Game Pre-Step and before the Scene Manager steps.
             * Hook into it from plugins or systems that need to update before the Scene Manager does, but after the core Systems have.
             */
            const STEP: any;

            /**
             * The Game Visible Event.
             * 
             * This event is dispatched by the Game Visibility Handler when the document in which the Game instance is embedded
             * enters a visible state, previously having been hidden.
             * 
             * Only browsers that support the Visibility API will cause this event to be emitted.
             */
            const VISIBLE: any;

        }

        /**
         * The core runner class that Phaser uses to handle the game loop. It can use either Request Animation Frame,
         * or SetTimeout, based on browser support and config settings, to create a continuous loop within the browser.
         * 
         * Each time the loop fires, `TimeStep.step` is called and this is then passed onto the core Game update loop,
         * it is the core heartbeat of your game. It will fire as often as Request Animation Frame is capable of handling
         * on the target device.
         * 
         * Note that there are lots of situations where a browser will stop updating your game. Such as if the player
         * switches tabs, or covers up the browser window with another application. In these cases, the 'heartbeat'
         * of your game will pause, and only resume when focus is returned to it by the player. There is no way to avoid
         * this situation, all you can do is use the visibility events the browser, and Phaser, provide to detect when
         * it has happened and then gracefully recover.
         */
        class TimeStep {
            /**
             * 
             * @param game A reference to the Phaser.Game instance that owns this Time Step.
             */
            constructor(game: Phaser.Game, config: Phaser.Types.Core.FPSConfig);

            /**
             * A reference to the Phaser.Game instance.
             */
            readonly game: Phaser.Game;

            /**
             * The Request Animation Frame DOM Event handler.
             */
            readonly raf: Phaser.DOM.RequestAnimationFrame;

            /**
             * A flag that is set once the TimeStep has started running and toggled when it stops.
             */
            readonly started: boolean;

            /**
             * A flag that is set once the TimeStep has started running and toggled when it stops.
             * The difference between this value and `started` is that `running` is toggled when
             * the TimeStep is sent to sleep, where-as `started` remains `true`, only changing if
             * the TimeStep is actually stopped, not just paused.
             */
            readonly running: boolean;

            /**
             * The minimum fps rate you want the Time Step to run at.
             */
            minFps: number;

            /**
             * The target fps rate for the Time Step to run at.
             * 
             * Setting this value will not actually change the speed at which the browser runs, that is beyond
             * the control of Phaser. Instead, it allows you to determine performance issues and if the Time Step
             * is spiraling out of control.
             */
            targetFps: number;

            /**
             * An exponential moving average of the frames per second.
             */
            readonly actualFps: number;

            /**
             * The time at which the next fps rate update will take place.
             * When an fps update happens, the `framesThisSecond` value is reset.
             */
            readonly nextFpsUpdate: number;

            /**
             * The number of frames processed this second.
             */
            readonly framesThisSecond: number;

            /**
             * A callback to be invoked each time the Time Step steps.
             */
            callback: Phaser.Types.Core.TimeStepCallback;

            /**
             * You can force the Time Step to use Set Timeout instead of Request Animation Frame by setting
             * the `forceSetTimeOut` property to `true` in the Game Configuration object. It cannot be changed at run-time.
             */
            readonly forceSetTimeOut: boolean;

            /**
             * The time, calculated at the start of the current step, as smoothed by the delta value.
             */
            time: number;

            /**
             * The time at which the game started running. This value is adjusted if the game is then
             * paused and resumes.
             */
            startTime: number;

            /**
             * The time, as returned by `performance.now` of the previous step.
             */
            lastTime: number;

            /**
             * The current frame the game is on. This counter is incremented once every game step, regardless of how much
             * time has passed and is unaffected by delta smoothing.
             */
            readonly frame: number;

            /**
             * Is the browser currently considered in focus by the Page Visibility API?
             * This value is set in the `blur` method, which is called automatically by the Game instance.
             */
            readonly inFocus: boolean;

            /**
             * The delta time, in ms, since the last game step. This is a clamped and smoothed average value.
             */
            delta: number;

            /**
             * Internal index of the delta history position.
             */
            deltaIndex: number;

            /**
             * Internal array holding the previous delta values, used for delta smoothing.
             */
            deltaHistory: number[];

            /**
             * The maximum number of delta values that are retained in order to calculate a smoothed moving average.
             * 
             * This can be changed in the Game Config via the `fps.deltaHistory` property. The default is 10.
             */
            deltaSmoothingMax: number;

            /**
             * The number of frames that the cooldown is set to after the browser panics over the FPS rate, usually
             * as a result of switching tabs and regaining focus.
             * 
             * This can be changed in the Game Config via the `fps.panicMax` property. The default is 120.
             */
            panicMax: number;

            /**
             * The actual elapsed time in ms between one update and the next.
             * 
             * Unlike with `delta`, no smoothing, capping, or averaging is applied to this value.
             * So please be careful when using this value in math calculations.
             */
            rawDelta: number;

            /**
             * The time, as returned by `performance.now` at the very start of the current step.
             * This can differ from the `time` value in that it isn't calculated based on the delta value.
             */
            now: number;

            /**
             * Apply smoothing to the delta value used within Phasers internal calculations?
             * 
             * This can be changed in the Game Config via the `fps.smoothStep` property. The default is `true`.
             * 
             * Smoothing helps settle down the delta values after browser tab switches, or other situations
             * which could cause significant delta spikes or dips. By default it has been enabled in Phaser 3
             * since the first version, but is now exposed under this property (and the corresponding game config
             * `smoothStep` value), to allow you to easily disable it, should you require.
             */
            smoothStep: boolean;

            /**
             * Called by the Game instance when the DOM window.onBlur event triggers.
             */
            blur(): void;

            /**
             * Called by the Game instance when the DOM window.onFocus event triggers.
             */
            focus(): void;

            /**
             * Called when the visibility API says the game is 'hidden' (tab switch out of view, etc)
             */
            pause(): void;

            /**
             * Called when the visibility API says the game is 'visible' again (tab switch back into view, etc)
             */
            resume(): void;

            /**
             * Resets the time, lastTime, fps averages and delta history.
             * Called automatically when a browser sleeps them resumes.
             */
            resetDelta(): void;

            /**
             * Starts the Time Step running, if it is not already doing so.
             * Called automatically by the Game Boot process.
             * @param callback The callback to be invoked each time the Time Step steps.
             */
            start(callback: Phaser.Types.Core.TimeStepCallback): void;

            /**
             * The main step method. This is called each time the browser updates, either by Request Animation Frame,
             * or by Set Timeout. It is responsible for calculating the delta values, frame totals, cool down history and more.
             * You generally should never call this method directly.
             */
            step(): void;

            /**
             * Manually calls `TimeStep.step`.
             */
            tick(): void;

            /**
             * Sends the TimeStep to sleep, stopping Request Animation Frame (or SetTimeout) and toggling the `running` flag to false.
             */
            sleep(): void;

            /**
             * Wakes-up the TimeStep, restarting Request Animation Frame (or SetTimeout) and toggling the `running` flag to true.
             * The `seamless` argument controls if the wake-up should adjust the start time or not.
             * @param seamless Adjust the startTime based on the lastTime values. Default false.
             */
            wake(seamless?: boolean): void;

            /**
             * Gets the duration which the game has been running, in seconds.
             */
            getDuration(): number;

            /**
             * Gets the duration which the game has been running, in ms.
             */
            getDurationMS(): number;

            /**
             * Stops the TimeStep running.
             */
            stop(): this;

            /**
             * Destroys the TimeStep. This will stop Request Animation Frame, stop the step, clear the callbacks and null
             * any objects.
             */
            destroy(): void;

        }

        /**
         * The Visibility Handler is responsible for listening out for document level visibility change events.
         * This includes `visibilitychange` if the browser supports it, and blur and focus events. It then uses
         * the provided Event Emitter and fires the related events.
         * @param game The Game instance this Visibility Handler is working on.
         */
        function VisibilityHandler(game: Phaser.Game): void;

    }

    namespace Create {
        /**
         * Generates a texture based on the given Create configuration object.
         * 
         * The texture is drawn using a fixed-size indexed palette of 16 colors, where the hex value in the
         * data cells map to a single color. For example, if the texture config looked like this:
         * 
         * ```javascript
         * var star = [
         *   '.....828.....',
         *   '....72227....',
         *   '....82228....',
         *   '...7222227...',
         *   '2222222222222',
         *   '8222222222228',
         *   '.72222222227.',
         *   '..787777787..',
         *   '..877777778..',
         *   '.78778887787.',
         *   '.27887.78872.',
         *   '.787.....787.'
         * ];
         * 
         * this.textures.generate('star', { data: star, pixelWidth: 4 });
         * ```
         * 
         * Then it would generate a texture that is 52 x 48 pixels in size, because each cell of the data array
         * represents 1 pixel multiplied by the `pixelWidth` value. The cell values, such as `8`, maps to color
         * number 8 in the palette. If a cell contains a period character `.` then it is transparent.
         * 
         * The default palette is Arne16, but you can specify your own using the `palette` property.
         * @param config The Generate Texture Configuration object.
         */
        function GenerateTexture(config: Phaser.Types.Create.GenerateTextureConfig): HTMLCanvasElement;

        namespace Palettes {
            /**
             * A 16 color palette by [Arne](http://androidarts.com/palette/16pal.htm)
             */
            var ARNE16: Phaser.Types.Create.Palette;

            /**
             * A 16 color palette inspired by the Commodore 64.
             */
            var C64: Phaser.Types.Create.Palette;

            /**
             * A 16 color CGA inspired palette by [Arne](http://androidarts.com/palette/16pal.htm)
             */
            var CGA: Phaser.Types.Create.Palette;

            /**
             * A 16 color JMP palette by [Arne](http://androidarts.com/palette/16pal.htm)
             */
            var JMP: Phaser.Types.Create.Palette;

            /**
             * A 16 color palette inspired by Japanese computers like the MSX.
             */
            var MSX: Phaser.Types.Create.Palette;

        }

    }

    namespace Curves {
        /**
         * A higher-order Bézier curve constructed of four points.
         */
        class CubicBezier extends Phaser.Curves.Curve {
            /**
             * 
             * @param p0 Start point, or an array of point pairs.
             * @param p1 Control Point 1.
             * @param p2 Control Point 2.
             * @param p3 End Point.
             */
            constructor(p0: Phaser.Math.Vector2 | Phaser.Math.Vector2[], p1: Phaser.Math.Vector2, p2: Phaser.Math.Vector2, p3: Phaser.Math.Vector2);

            /**
             * The start point of this curve.
             */
            p0: Phaser.Math.Vector2;

            /**
             * The first control point of this curve.
             */
            p1: Phaser.Math.Vector2;

            /**
             * The second control point of this curve.
             */
            p2: Phaser.Math.Vector2;

            /**
             * The end point of this curve.
             */
            p3: Phaser.Math.Vector2;

            /**
             * Gets the starting point on the curve.
             * @param out A Vector2 object to store the result in. If not given will be created.
             */
            getStartPoint<O extends Phaser.Math.Vector2>(out?: O): O;

            /**
             * Returns the resolution of this curve.
             * @param divisions The amount of divisions used by this curve.
             */
            getResolution(divisions: number): number;

            /**
             * Get point at relative position in curve according to length.
             * @param t The position along the curve to return. Where 0 is the start and 1 is the end.
             * @param out A Vector2 object to store the result in. If not given will be created.
             */
            getPoint<O extends Phaser.Math.Vector2>(t: number, out?: O): O;

            /**
             * Draws this curve to the specified graphics object.
             * @param graphics The graphics object this curve should be drawn to.
             * @param pointsTotal The number of intermediary points that make up this curve. A higher number of points will result in a smoother curve. Default 32.
             */
            draw<G extends Phaser.GameObjects.Graphics>(graphics: G, pointsTotal?: number): G;

            /**
             * Returns a JSON object that describes this curve.
             */
            toJSON(): Phaser.Types.Curves.JSONCurve;

            /**
             * Generates a curve from a JSON object.
             * @param data The JSON object containing this curve data.
             */
            static fromJSON(data: Phaser.Types.Curves.JSONCurve): Phaser.Curves.CubicBezier;

        }

        /**
         * A Base Curve class, which all other curve types extend.
         * 
         * Based on the three.js Curve classes created by [zz85](http://www.lab4games.net/zz85/blog)
         */
        class Curve {
            /**
             * 
             * @param type The curve type.
             */
            constructor(type: string);

            /**
             * String based identifier for the type of curve.
             */
            type: string;

            /**
             * The default number of divisions within the curve.
             */
            defaultDivisions: number;

            /**
             * The quantity of arc length divisions within the curve.
             */
            arcLengthDivisions: number;

            /**
             * An array of cached arc length values.
             */
            cacheArcLengths: number[];

            /**
             * Does the data of this curve need updating?
             */
            needsUpdate: boolean;

            /**
             * For a curve on a Path, `false` means the Path will ignore this curve.
             */
            active: boolean;

            /**
             * Draws this curve on the given Graphics object.
             * 
             * The curve is drawn using `Graphics.strokePoints` so will be drawn at whatever the present Graphics stroke color is.
             * The Graphics object is not cleared before the draw, so the curve will appear on-top of anything else already rendered to it.
             * @param graphics The Graphics instance onto which this curve will be drawn.
             * @param pointsTotal The resolution of the curve. The higher the value the smoother it will render, at the cost of rendering performance. Default 32.
             */
            draw<G extends Phaser.GameObjects.Graphics>(graphics: G, pointsTotal?: number): G;

            /**
             * Returns a Rectangle where the position and dimensions match the bounds of this Curve.
             * 
             * You can control the accuracy of the bounds. The value given is used to work out how many points
             * to plot across the curve. Higher values are more accurate at the cost of calculation speed.
             * @param out The Rectangle to store the bounds in. If falsey a new object will be created.
             * @param accuracy The accuracy of the bounds calculations. Default 16.
             */
            getBounds(out?: Phaser.Geom.Rectangle, accuracy?: number): Phaser.Geom.Rectangle;

            /**
             * Returns an array of points, spaced out X distance pixels apart.
             * The smaller the distance, the larger the array will be.
             * @param distance The distance, in pixels, between each point along the curve.
             */
            getDistancePoints(distance: number): Phaser.Geom.Point[];

            /**
             * Get a point at the end of the curve.
             * @param out Optional Vector object to store the result in.
             */
            getEndPoint(out?: Phaser.Math.Vector2): Phaser.Math.Vector2;

            /**
             * Get total curve arc length
             */
            getLength(): number;

            /**
             * Get a list of cumulative segment lengths.
             * 
             * These lengths are
             * 
             * - [0] 0
             * - [1] The first segment
             * - [2] The first and second segment
             * - ...
             * - [divisions] All segments
             * @param divisions The number of divisions or segments.
             */
            getLengths(divisions?: number): number[];

            /**
             * Get a point at a relative position on the curve, by arc length.
             * @param u The relative position, [0..1].
             * @param out A point to store the result in.
             */
            getPointAt<O extends Phaser.Math.Vector2>(u: number, out?: O): O;

            /**
             * Get a sequence of evenly spaced points from the curve.
             * 
             * You can pass `divisions`, `stepRate`, or neither.
             * 
             * The number of divisions will be
             * 
             * 1. `divisions`, if `divisions` > 0; or
             * 2. `this.getLength / stepRate`, if `stepRate` > 0; or
             * 3. `this.defaultDivisions`
             * 
             * `1 + divisions` points will be returned.
             * @param divisions The number of divisions to make.
             * @param stepRate The curve distance between points, implying `divisions`.
             * @param out An optional array to store the points in.
             */
            getPoints<O extends Phaser.Math.Vector2[]>(divisions?: number, stepRate?: number, out?: O): O;

            /**
             * Get a random point from the curve.
             * @param out A point object to store the result in.
             */
            getRandomPoint<O extends Phaser.Math.Vector2>(out?: O): O;

            /**
             * Get a sequence of equally spaced points (by arc distance) from the curve.
             * 
             * `1 + divisions` points will be returned.
             * @param divisions The number of divisions to make. Default this.defaultDivisions.
             * @param stepRate Step between points. Used to calculate the number of points to return when divisions is falsy. Ignored if divisions is positive.
             * @param out An optional array to store the points in.
             */
            getSpacedPoints(divisions?: number, stepRate?: number, out?: any[] | Phaser.Math.Vector2[]): Phaser.Math.Vector2[];

            /**
             * Get a point at the start of the curve.
             * @param out A point to store the result in.
             */
            getStartPoint<O extends Phaser.Math.Vector2>(out?: O): O;

            /**
             * Get a unit vector tangent at a relative position on the curve.
             * In case any sub curve does not implement its tangent derivation,
             * 2 points a small delta apart will be used to find its gradient
             * which seems to give a reasonable approximation
             * @param t The relative position on the curve, [0..1].
             * @param out A vector to store the result in.
             */
            getTangent<O extends Phaser.Math.Vector2>(t: number, out?: O): O;

            /**
             * Get a unit vector tangent at a relative position on the curve, by arc length.
             * @param u The relative position on the curve, [0..1].
             * @param out A vector to store the result in.
             */
            getTangentAt<O extends Phaser.Math.Vector2>(u: number, out?: O): O;

            /**
             * Given a distance in pixels, get a t to find p.
             * @param distance The distance, in pixels.
             * @param divisions Optional amount of divisions.
             */
            getTFromDistance(distance: number, divisions?: number): number;

            /**
             * Given u ( 0 .. 1 ), get a t to find p. This gives you points which are equidistant.
             * @param u A float between 0 and 1.
             * @param distance The distance, in pixels.
             * @param divisions Optional amount of divisions.
             */
            getUtoTmapping(u: number, distance: number, divisions?: number): number;

            /**
             * Calculate and cache the arc lengths.
             */
            updateArcLengths(): void;

        }

        /**
         * An Elliptical Curve derived from the Base Curve class.
         * 
         * See https://en.wikipedia.org/wiki/Elliptic_curve for more details.
         */
        class Ellipse extends Phaser.Curves.Curve {
            /**
             * 
             * @param x The x coordinate of the ellipse, or an Ellipse Curve configuration object. Default 0.
             * @param y The y coordinate of the ellipse. Default 0.
             * @param xRadius The horizontal radius of ellipse. Default 0.
             * @param yRadius The vertical radius of ellipse. Default 0.
             * @param startAngle The start angle of the ellipse, in degrees. Default 0.
             * @param endAngle The end angle of the ellipse, in degrees. Default 360.
             * @param clockwise Whether the ellipse angles are given as clockwise (`true`) or counter-clockwise (`false`). Default false.
             * @param rotation The rotation of the ellipse, in degrees. Default 0.
             */
            constructor(x?: number | Phaser.Types.Curves.EllipseCurveConfig, y?: number, xRadius?: number, yRadius?: number, startAngle?: number, endAngle?: number, clockwise?: boolean, rotation?: number);

            /**
             * The center point of the ellipse. Used for calculating rotation.
             */
            p0: Phaser.Math.Vector2;

            /**
             * Gets the starting point on the curve.
             * @param out A Vector2 object to store the result in. If not given will be created.
             */
            getStartPoint<O extends Phaser.Math.Vector2>(out?: O): O;

            /**
             * Get the resolution of the curve.
             * @param divisions Optional divisions value.
             */
            getResolution(divisions: number): number;

            /**
             * Get point at relative position in curve according to length.
             * @param t The position along the curve to return. Where 0 is the start and 1 is the end.
             * @param out A Vector2 object to store the result in. If not given will be created.
             */
            getPoint<O extends Phaser.Math.Vector2>(t: number, out?: O): O;

            /**
             * Sets the horizontal radius of this curve.
             * @param value The horizontal radius of this curve.
             */
            setXRadius(value: number): this;

            /**
             * Sets the vertical radius of this curve.
             * @param value The vertical radius of this curve.
             */
            setYRadius(value: number): this;

            /**
             * Sets the width of this curve.
             * @param value The width of this curve.
             */
            setWidth(value: number): this;

            /**
             * Sets the height of this curve.
             * @param value The height of this curve.
             */
            setHeight(value: number): this;

            /**
             * Sets the start angle of this curve.
             * @param value The start angle of this curve, in radians.
             */
            setStartAngle(value: number): this;

            /**
             * Sets the end angle of this curve.
             * @param value The end angle of this curve, in radians.
             */
            setEndAngle(value: number): this;

            /**
             * Sets if this curve extends clockwise or anti-clockwise.
             * @param value The clockwise state of this curve.
             */
            setClockwise(value: boolean): this;

            /**
             * Sets the rotation of this curve.
             * @param value The rotation of this curve, in radians.
             */
            setRotation(value: number): this;

            /**
             * The x coordinate of the center of the ellipse.
             */
            x: number;

            /**
             * The y coordinate of the center of the ellipse.
             */
            y: number;

            /**
             * The horizontal radius of the ellipse.
             */
            xRadius: number;

            /**
             * The vertical radius of the ellipse.
             */
            yRadius: number;

            /**
             * The start angle of the ellipse in degrees.
             */
            startAngle: number;

            /**
             * The end angle of the ellipse in degrees.
             */
            endAngle: number;

            /**
             * `true` if the ellipse rotation is clockwise or `false` if anti-clockwise.
             */
            clockwise: boolean;

            /**
             * The rotation of the ellipse, relative to the center, in degrees.
             */
            angle: number;

            /**
             * The rotation of the ellipse, relative to the center, in radians.
             */
            rotation: number;

            /**
             * JSON serialization of the curve.
             */
            toJSON(): Phaser.Types.Curves.JSONEllipseCurve;

            /**
             * Creates a curve from the provided Ellipse Curve Configuration object.
             * @param data The JSON object containing this curve data.
             */
            static fromJSON(data: Phaser.Types.Curves.JSONEllipseCurve): Phaser.Curves.Ellipse;

        }

        /**
         * A LineCurve is a "curve" comprising exactly two points (a line segment).
         */
        class Line extends Phaser.Curves.Curve {
            /**
             * 
             * @param p0 The first endpoint.
             * @param p1 The second endpoint.
             */
            constructor(p0: Phaser.Math.Vector2 | number[], p1?: Phaser.Math.Vector2);

            /**
             * The first endpoint.
             */
            p0: Phaser.Math.Vector2;

            /**
             * The second endpoint.
             */
            p1: Phaser.Math.Vector2;

            /**
             * The quantity of arc length divisions within the curve.
             */
            arcLengthDivisions: number;

            /**
             * Returns a Rectangle where the position and dimensions match the bounds of this Curve.
             * @param out A Rectangle object to store the bounds in. If not given a new Rectangle will be created.
             */
            getBounds<O extends Phaser.Geom.Rectangle>(out?: O): O;

            /**
             * Gets the starting point on the curve.
             * @param out A Vector2 object to store the result in. If not given will be created.
             */
            getStartPoint<O extends Phaser.Math.Vector2>(out?: O): O;

            /**
             * Gets the resolution of the line.
             * @param divisions The number of divisions to consider. Default 1.
             */
            getResolution(divisions?: number): number;

            /**
             * Get point at relative position in curve according to length.
             * @param t The position along the curve to return. Where 0 is the start and 1 is the end.
             * @param out A Vector2 object to store the result in. If not given will be created.
             */
            getPoint<O extends Phaser.Math.Vector2>(t: number, out?: O): O;

            /**
             * Gets a point at a given position on the line.
             * @param u The position along the curve to return. Where 0 is the start and 1 is the end.
             * @param out A Vector2 object to store the result in. If not given will be created.
             */
            getPointAt<O extends Phaser.Math.Vector2>(u: number, out?: O): O;

            /**
             * Gets the slope of the line as a unit vector.
             * @param t The relative position on the line, [0..1].
             * @param out A vector to store the result in.
             */
            getTangent<O extends Phaser.Math.Vector2>(t?: number, out?: O): O;

            /**
             * Given u ( 0 .. 1 ), get a t to find p. This gives you points which are equidistant.
             * @param u A float between 0 and 1.
             * @param distance The distance, in pixels.
             * @param divisions Optional amount of divisions.
             */
            getUtoTmapping(u: number, distance: number, divisions?: number): number;

            /**
             * Draws this curve on the given Graphics object.
             * 
             * The curve is drawn using `Graphics.lineBetween` so will be drawn at whatever the present Graphics line color is.
             * The Graphics object is not cleared before the draw, so the curve will appear on-top of anything else already rendered to it.
             * @param graphics The Graphics instance onto which this curve will be drawn.
             */
            draw<G extends Phaser.GameObjects.Graphics>(graphics: G): G;

            /**
             * Gets a JSON representation of the line.
             */
            toJSON(): Phaser.Types.Curves.JSONCurve;

            /**
             * Configures this line from a JSON representation.
             * @param data The JSON object containing this curve data.
             */
            static fromJSON(data: Phaser.Types.Curves.JSONCurve): Phaser.Curves.Line;

        }

        /**
         * A MoveTo Curve is a very simple curve consisting of only a single point.
         * Its intended use is to move the ending point in a Path.
         */
        class MoveTo {
            /**
             * 
             * @param x `x` pixel coordinate. Default 0.
             * @param y `y` pixel coordinate. Default 0.
             */
            constructor(x?: number, y?: number);

            /**
             * Denotes that this Curve does not influence the bounds, points, and drawing of its parent Path. Must be `false` or some methods in the parent Path will throw errors.
             */
            active: boolean;

            /**
             * The lone point which this curve consists of.
             */
            p0: Phaser.Math.Vector2;

            /**
             * Get point at relative position in curve according to length.
             * @param t The position along the curve to return. Where 0 is the start and 1 is the end.
             * @param out A Vector2 object to store the result in. If not given will be created.
             */
            getPoint<O extends Phaser.Math.Vector2>(t: number, out?: O): O;

            /**
             * Retrieves the point at given position in the curve. This will always return this curve's only point.
             * @param u The position in the path to retrieve, between 0 and 1. Not used.
             * @param out An optional vector in which to store the point.
             */
            getPointAt<O extends Phaser.Math.Vector2>(u: number, out?: O): O;

            /**
             * Gets the resolution of this curve.
             */
            getResolution(): number;

            /**
             * Gets the length of this curve.
             */
            getLength(): number;

            /**
             * Converts this curve into a JSON-serializable object.
             */
            toJSON(): Phaser.Types.Curves.JSONCurve;

        }

        /**
         * A Path combines multiple Curves into one continuous compound curve.
         * It does not matter how many Curves are in the Path or what type they are.
         * 
         * A Curve in a Path does not have to start where the previous Curve ends - that is to say, a Path does not
         * have to be an uninterrupted curve. Only the order of the Curves influences the actual points on the Path.
         */
        class Path {
            /**
             * 
             * @param x The X coordinate of the Path's starting point or a {@link Phaser.Types.Curves.JSONPath}. Default 0.
             * @param y The Y coordinate of the Path's starting point. Default 0.
             */
            constructor(x?: number, y?: number);

            /**
             * The name of this Path.
             * Empty by default and never populated by Phaser, this is left for developers to use.
             */
            name: string;

            /**
             * The list of Curves which make up this Path.
             */
            curves: Phaser.Curves.Curve[];

            /**
             * The cached length of each Curve in the Path.
             * 
             * Used internally by {@link #getCurveLengths}.
             */
            cacheLengths: number[];

            /**
             * Automatically closes the path.
             */
            autoClose: boolean;

            /**
             * The starting point of the Path.
             * 
             * This is not necessarily equivalent to the starting point of the first Curve in the Path. In an empty Path, it's also treated as the ending point.
             */
            startPoint: Phaser.Math.Vector2;

            /**
             * Appends a Curve to the end of the Path.
             * 
             * The Curve does not have to start where the Path ends or, for an empty Path, at its defined starting point.
             * @param curve The Curve to append.
             */
            add(curve: Phaser.Curves.Curve): this;

            /**
             * Creates a circular Ellipse Curve positioned at the end of the Path.
             * @param radius The radius of the circle.
             * @param clockwise `true` to create a clockwise circle as opposed to a counter-clockwise circle. Default false.
             * @param rotation The rotation of the circle in degrees. Default 0.
             */
            circleTo(radius: number, clockwise?: boolean, rotation?: number): this;

            /**
             * Ensures that the Path is closed.
             * 
             * A closed Path starts and ends at the same point. If the Path is not closed, a straight Line Curve will be created from the ending point directly to the starting point. During the check, the actual starting point of the Path, i.e. the starting point of the first Curve, will be used as opposed to the Path's defined {@link startPoint}, which could differ.
             * 
             * Calling this method on an empty Path will result in an error.
             */
            closePath(): this;

            /**
             * Creates a cubic bezier curve starting at the previous end point and ending at p3, using p1 and p2 as control points.
             * @param x The x coordinate of the end point. Or, if a Vector2, the p1 value.
             * @param y The y coordinate of the end point. Or, if a Vector2, the p2 value.
             * @param control1X The x coordinate of the first control point. Or, if a Vector2, the p3 value.
             * @param control1Y The y coordinate of the first control point. Not used if Vector2s are provided as the first 3 arguments.
             * @param control2X The x coordinate of the second control point. Not used if Vector2s are provided as the first 3 arguments.
             * @param control2Y The y coordinate of the second control point. Not used if Vector2s are provided as the first 3 arguments.
             */
            cubicBezierTo(x: number | Phaser.Math.Vector2, y: number | Phaser.Math.Vector2, control1X: number | Phaser.Math.Vector2, control1Y?: number, control2X?: number, control2Y?: number): this;

            /**
             * Creates a Quadratic Bezier Curve starting at the ending point of the Path.
             * @param x The X coordinate of the second control point or, if it's a `Vector2`, the first control point.
             * @param y The Y coordinate of the second control point or, if `x` is a `Vector2`, the second control point.
             * @param controlX If `x` is not a `Vector2`, the X coordinate of the first control point.
             * @param controlY If `x` is not a `Vector2`, the Y coordinate of the first control point.
             */
            quadraticBezierTo(x: number | Phaser.Math.Vector2[], y?: number, controlX?: number, controlY?: number): this;

            /**
             * Draws all Curves in the Path to a Graphics Game Object.
             * @param graphics The Graphics Game Object to draw to.
             * @param pointsTotal The number of points to draw for each Curve. Higher numbers result in a smoother curve but require more processing. Default 32.
             */
            draw<G extends Phaser.GameObjects.Graphics>(graphics: Phaser.GameObjects.Graphics, pointsTotal?: number): G;

            /**
             * Creates an ellipse curve positioned at the previous end point, using the given parameters.
             * @param xRadius The horizontal radius of ellipse. Default 0.
             * @param yRadius The vertical radius of ellipse. Default 0.
             * @param startAngle The start angle of the ellipse, in degrees. Default 0.
             * @param endAngle The end angle of the ellipse, in degrees. Default 360.
             * @param clockwise Whether the ellipse angles are given as clockwise (`true`) or counter-clockwise (`false`). Default false.
             * @param rotation The rotation of the ellipse, in degrees. Default 0.
             */
            ellipseTo(xRadius?: number, yRadius?: number, startAngle?: number, endAngle?: number, clockwise?: boolean, rotation?: number): this;

            /**
             * Creates a Path from a Path Configuration object.
             * 
             * The provided object should be a {@link Phaser.Types.Curves.JSONPath}, as returned by {@link #toJSON}. Providing a malformed object may cause errors.
             * @param data The JSON object containing the Path data.
             */
            fromJSON(data: Phaser.Types.Curves.JSONPath): this;

            /**
             * Returns a Rectangle with a position and size matching the bounds of this Path.
             * @param out The Rectangle to store the bounds in.
             * @param accuracy The accuracy of the bounds calculations. Higher values are more accurate at the cost of calculation speed. Default 16.
             */
            getBounds<O extends Phaser.Math.Vector2>(out?: O, accuracy?: number): O;

            /**
             * Returns an array containing the length of the Path at the end of each Curve.
             * 
             * The result of this method will be cached to avoid recalculating it in subsequent calls. The cache is only invalidated when the {@link #curves} array changes in length, leading to potential inaccuracies if a Curve in the Path is changed, or if a Curve is removed and another is added in its place.
             */
            getCurveLengths(): number[];

            /**
             * Returns the ending point of the Path.
             * 
             * A Path's ending point is equivalent to the ending point of the last Curve in the Path. For an empty Path, the ending point is at the Path's defined {@link #startPoint}.
             * @param out The object to store the point in.
             */
            getEndPoint<O extends Phaser.Math.Vector2>(out?: O): O;

            /**
             * Returns the total length of the Path.
             */
            getLength(): number;

            /**
             * Calculates the coordinates of the point at the given normalized location (between 0 and 1) on the Path.
             * 
             * The location is relative to the entire Path, not to an individual Curve. A location of 0.5 is always in the middle of the Path and is thus an equal distance away from both its starting and ending points. In a Path with one Curve, it would be in the middle of the Curve; in a Path with two Curves, it could be anywhere on either one of them depending on their lengths.
             * @param t The location of the point to return, between 0 and 1.
             * @param out The object in which to store the calculated point.
             */
            getPoint<O extends Phaser.Math.Vector2>(t: number, out?: O): O;

            /**
             * Get a sequence of points on the path.
             * @param divisions The number of divisions per resolution per curve. Default 12.
             */
            getPoints(divisions?: number): Phaser.Math.Vector2[];

            /**
             * Returns a randomly chosen point anywhere on the path. This follows the same rules as `getPoint` in that it may return a point on any Curve inside this path.
             * 
             * When calling this method multiple times, the points are not guaranteed to be equally spaced spatially.
             * @param out `Vector2` instance that should be used for storing the result. If `undefined` a new `Vector2` will be created.
             */
            getRandomPoint<O extends Phaser.Math.Vector2>(out?: O): O;

            /**
             * Divides this Path into a set of equally spaced points,
             * 
             * The resulting points are equally spaced with respect to the points' position on the path, but not necessarily equally spaced spatially.
             * @param divisions The amount of points to divide this Path into. Default 40.
             */
            getSpacedPoints(divisions?: number): Phaser.Math.Vector2[];

            /**
             * Returns the starting point of the Path.
             * @param out `Vector2` instance that should be used for storing the result. If `undefined` a new `Vector2` will be created.
             */
            getStartPoint<O extends Phaser.Math.Vector2>(out?: O): O;

            /**
             * Gets a unit vector tangent at a relative position on the path.
             * @param t The relative position on the path, [0..1].
             * @param out A vector to store the result in.
             */
            getTangent<O extends Phaser.Math.Vector2>(t: number, out?: O): O;

            /**
             * Creates a line curve from the previous end point to x/y.
             * @param x The X coordinate of the line's end point, or a `Vector2` containing the entire end point.
             * @param y The Y coordinate of the line's end point, if a number was passed as the X parameter.
             */
            lineTo(x: number | Phaser.Math.Vector2, y?: number): this;

            /**
             * Creates a spline curve starting at the previous end point, using the given points on the curve.
             * @param points The points the newly created spline curve should consist of.
             */
            splineTo(points: Phaser.Math.Vector2[]): this;

            /**
             * Creates a "gap" in this path from the path's current end point to the given coordinates.
             * 
             * After calling this function, this Path's end point will be equal to the given coordinates
             * @param x The X coordinate of the position to move the path's end point to, or a `Vector2` containing the entire new end point.
             * @param y The Y coordinate of the position to move the path's end point to, if a number was passed as the X coordinate.
             */
            moveTo(x: number | Phaser.Math.Vector2, y?: number): this;

            /**
             * Converts this Path to a JSON object containing the path information and its constituent curves.
             */
            toJSON(): Phaser.Types.Curves.JSONPath;

            /**
             * cacheLengths must be recalculated.
             */
            updateArcLengths(): void;

            /**
             * Disposes of this Path, clearing its internal references to objects so they can be garbage-collected.
             */
            destroy(): void;

        }

        /**
         * A quadratic Bézier curve constructed from two control points.
         */
        class QuadraticBezier extends Phaser.Curves.Curve {
            /**
             * 
             * @param p0 Start point, or an array of point pairs.
             * @param p1 Control Point 1.
             * @param p2 Control Point 2.
             */
            constructor(p0: Phaser.Math.Vector2 | number[], p1: Phaser.Math.Vector2, p2: Phaser.Math.Vector2);

            /**
             * The start point.
             */
            p0: Phaser.Math.Vector2;

            /**
             * The first control point.
             */
            p1: Phaser.Math.Vector2;

            /**
             * The second control point.
             */
            p2: Phaser.Math.Vector2;

            /**
             * Gets the starting point on the curve.
             * @param out A Vector2 object to store the result in. If not given will be created.
             */
            getStartPoint<O extends Phaser.Math.Vector2>(out?: O): O;

            /**
             * Get the resolution of the curve.
             * @param divisions Optional divisions value.
             */
            getResolution(divisions: number): number;

            /**
             * Get point at relative position in curve according to length.
             * @param t The position along the curve to return. Where 0 is the start and 1 is the end.
             * @param out A Vector2 object to store the result in. If not given will be created.
             */
            getPoint<O extends Phaser.Math.Vector2>(t: number, out?: O): O;

            /**
             * Draws this curve on the given Graphics object.
             * 
             * The curve is drawn using `Graphics.strokePoints` so will be drawn at whatever the present Graphics stroke color is.
             * The Graphics object is not cleared before the draw, so the curve will appear on-top of anything else already rendered to it.
             * @param graphics `Graphics` object to draw onto.
             * @param pointsTotal Number of points to be used for drawing the curve. Higher numbers result in smoother curve but require more processing. Default 32.
             */
            draw<G extends Phaser.GameObjects.Graphics>(graphics: G, pointsTotal?: number): G;

            /**
             * Converts the curve into a JSON compatible object.
             */
            toJSON(): Phaser.Types.Curves.JSONCurve;

            /**
             * Creates a curve from a JSON object, e. g. created by `toJSON`.
             * @param data The JSON object containing this curve data.
             */
            static fromJSON(data: Phaser.Types.Curves.JSONCurve): Phaser.Curves.QuadraticBezier;

        }

        /**
         * Create a smooth 2d spline curve from a series of points.
         */
        class Spline extends Phaser.Curves.Curve {
            /**
             * 
             * @param points The points that configure the curve.
             */
            constructor(points?: Phaser.Math.Vector2[] | number[] | number[][]);

            /**
             * The Vector2 points that configure the curve.
             */
            points: Phaser.Math.Vector2[];

            /**
             * Add a list of points to the current list of Vector2 points of the curve.
             * @param points The points that configure the curve.
             */
            addPoints(points: Phaser.Math.Vector2[] | number[] | number[][]): this;

            /**
             * Add a point to the current list of Vector2 points of the curve.
             * @param x The x coordinate of this curve
             * @param y The y coordinate of this curve
             */
            addPoint(x: number, y: number): Phaser.Math.Vector2;

            /**
             * Gets the starting point on the curve.
             * @param out A Vector2 object to store the result in. If not given will be created.
             */
            getStartPoint<O extends Phaser.Math.Vector2>(out?: O): O;

            /**
             * Get the resolution of the curve.
             * @param divisions Optional divisions value.
             */
            getResolution(divisions: number): number;

            /**
             * Get point at relative position in curve according to length.
             * @param t The position along the curve to return. Where 0 is the start and 1 is the end.
             * @param out A Vector2 object to store the result in. If not given will be created.
             */
            getPoint<O extends Phaser.Math.Vector2>(t: number, out?: O): O;

            /**
             * Exports a JSON object containing this curve data.
             */
            toJSON(): Phaser.Types.Curves.JSONCurve;

            /**
             * Imports a JSON object containing this curve data.
             * @param data The JSON object containing this curve data.
             */
            static fromJSON(data: Phaser.Types.Curves.JSONCurve): Phaser.Curves.Spline;

        }

    }

    namespace Data {
        /**
         * The Data Manager Component features a means to store pieces of data specific to a Game Object, System or Plugin.
         * You can then search, query it, and retrieve the data. The parent must either extend EventEmitter,
         * or have a property called `events` that is an instance of it.
         */
        class DataManager {
            /**
             * 
             * @param parent The object that this DataManager belongs to.
             * @param eventEmitter The DataManager's event emitter.
             */
            constructor(parent: object, eventEmitter?: Phaser.Events.EventEmitter);

            /**
             * The object that this DataManager belongs to.
             */
            parent: any;

            /**
             * The DataManager's event emitter.
             */
            events: Phaser.Events.EventEmitter;

            /**
             * The data list.
             */
            list: {[key: string]:  any};

            /**
             * The public values list. You can use this to access anything you have stored
             * in this Data Manager. For example, if you set a value called `gold` you can
             * access it via:
             * 
             * ```javascript
             * this.data.values.gold;
             * ```
             * 
             * You can also modify it directly:
             * 
             * ```javascript
             * this.data.values.gold += 1000;
             * ```
             * 
             * Doing so will emit a `setdata` event from the parent of this Data Manager.
             * 
             * Do not modify this object directly. Adding properties directly to this object will not
             * emit any events. Always use `DataManager.set` to create new items the first time around.
             */
            values: {[key: string]:  any};

            /**
             * Retrieves the value for the given key, or undefined if it doesn't exist.
             * 
             * You can also access values via the `values` object. For example, if you had a key called `gold` you can do either:
             * 
             * ```javascript
             * this.data.get('gold');
             * ```
             * 
             * Or access the value directly:
             * 
             * ```javascript
             * this.data.values.gold;
             * ```
             * 
             * You can also pass in an array of keys, in which case an array of values will be returned:
             * 
             * ```javascript
             * this.data.get([ 'gold', 'armor', 'health' ]);
             * ```
             * 
             * This approach is useful for destructuring arrays in ES6.
             * @param key The key of the value to retrieve, or an array of keys.
             */
            get(key: string | string[]): any;

            /**
             * Retrieves all data values in a new object.
             */
            getAll(): {[key: string]:  any};

            /**
             * Queries the DataManager for the values of keys matching the given regular expression.
             * @param search A regular expression object. If a non-RegExp object obj is passed, it is implicitly converted to a RegExp by using new RegExp(obj).
             */
            query(search: RegExp): {[key: string]:  any};

            /**
             * Sets a value for the given key. If the key doesn't already exist in the Data Manager then it is created.
             * 
             * ```javascript
             * data.set('name', 'Red Gem Stone');
             * ```
             * 
             * You can also pass in an object of key value pairs as the first argument:
             * 
             * ```javascript
             * data.set({ name: 'Red Gem Stone', level: 2, owner: 'Link', gold: 50 });
             * ```
             * 
             * To get a value back again you can call `get`:
             * 
             * ```javascript
             * data.get('gold');
             * ```
             * 
             * Or you can access the value directly via the `values` property, where it works like any other variable:
             * 
             * ```javascript
             * data.values.gold += 50;
             * ```
             * 
             * When the value is first set, a `setdata` event is emitted.
             * 
             * If the key already exists, a `changedata` event is emitted instead, along an event named after the key.
             * For example, if you updated an existing key called `PlayerLives` then it would emit the event `changedata-PlayerLives`.
             * These events will be emitted regardless if you use this method to set the value, or the direct `values` setter.
             * 
             * Please note that the data keys are case-sensitive and must be valid JavaScript Object property strings.
             * This means the keys `gold` and `Gold` are treated as two unique values within the Data Manager.
             * @param key The key to set the value for. Or an object or key value pairs. If an object the `data` argument is ignored.
             * @param data The value to set for the given key. If an object is provided as the key this argument is ignored.
             */
            set(key: string | object, data: any): this;

            /**
             * Increase a value for the given key. If the key doesn't already exist in the Data Manager then it is increased from 0.
             * 
             * When the value is first set, a `setdata` event is emitted.
             * @param key The key to increase the value for.
             * @param data The value to increase for the given key.
             */
            inc(key: string | object, data?: any): Phaser.Data.DataManager;

            /**
             * Toggle a boolean value for the given key. If the key doesn't already exist in the Data Manager then it is toggled from false.
             * 
             * When the value is first set, a `setdata` event is emitted.
             * @param key The key to toggle the value for.
             */
            toggle(key: string | object): Phaser.Data.DataManager;

            /**
             * Passes all data entries to the given callback.
             * @param callback The function to call.
             * @param context Value to use as `this` when executing callback.
             * @param args Additional arguments that will be passed to the callback, after the game object, key, and data.
             */
            each(callback: DataEachCallback, context?: any, ...args: any[]): this;

            /**
             * Merge the given object of key value pairs into this DataManager.
             * 
             * Any newly created values will emit a `setdata` event. Any updated values (see the `overwrite` argument)
             * will emit a `changedata` event.
             * @param data The data to merge.
             * @param overwrite Whether to overwrite existing data. Defaults to true. Default true.
             */
            merge(data: {[key: string]:  any}, overwrite?: boolean): this;

            /**
             * Remove the value for the given key.
             * 
             * If the key is found in this Data Manager it is removed from the internal lists and a
             * `removedata` event is emitted.
             * 
             * You can also pass in an array of keys, in which case all keys in the array will be removed:
             * 
             * ```javascript
             * this.data.remove([ 'gold', 'armor', 'health' ]);
             * ```
             * @param key The key to remove, or an array of keys to remove.
             */
            remove(key: string | string[]): this;

            /**
             * Retrieves the data associated with the given 'key', deletes it from this Data Manager, then returns it.
             * @param key The key of the value to retrieve and delete.
             */
            pop(key: string): any;

            /**
             * Determines whether the given key is set in this Data Manager.
             * 
             * Please note that the keys are case-sensitive and must be valid JavaScript Object property strings.
             * This means the keys `gold` and `Gold` are treated as two unique values within the Data Manager.
             * @param key The key to check.
             */
            has(key: string): boolean;

            /**
             * Freeze or unfreeze this Data Manager. A frozen Data Manager will block all attempts
             * to create new values or update existing ones.
             * @param value Whether to freeze or unfreeze the Data Manager.
             */
            setFreeze(value: boolean): this;

            /**
             * Delete all data in this Data Manager and unfreeze it.
             */
            reset(): this;

            /**
             * Destroy this data manager.
             */
            destroy(): void;

            /**
             * Gets or sets the frozen state of this Data Manager.
             * A frozen Data Manager will block all attempts to create new values or update existing ones.
             */
            freeze: boolean;

            /**
             * Return the total number of entries in this Data Manager.
             */
            count: number;

        }

        /**
         * The Data Component features a means to store pieces of data specific to a Game Object, System or Plugin.
         * You can then search, query it, and retrieve the data. The parent must either extend EventEmitter,
         * or have a property called `events` that is an instance of it.
         */
        class DataManagerPlugin extends Phaser.Data.DataManager {
            /**
             * 
             * @param scene A reference to the Scene that this DataManager belongs to.
             */
            constructor(scene: Phaser.Scene);

            /**
             * A reference to the Scene that this DataManager belongs to.
             */
            scene: Phaser.Scene;

            /**
             * A reference to the Scene's Systems.
             */
            systems: Phaser.Scenes.Systems;

            /**
             * The Scene that owns this plugin is being destroyed.
             * We need to shutdown and then kill off all external references.
             */
            destroy(): void;

        }

        namespace Events {
            /**
             * The Change Data Event.
             * 
             * This event is dispatched by a Data Manager when an item in the data store is changed.
             * 
             * Game Objects with data enabled have an instance of a Data Manager under the `data` property. So, to listen for
             * a change data event from a Game Object you would use: `sprite.data.on('changedata', listener)`.
             * 
             * This event is dispatched for all items that change in the Data Manager.
             * To listen for the change of a specific item, use the `CHANGE_DATA_KEY_EVENT` event.
             */
            const CHANGE_DATA: any;

            /**
             * The Change Data Key Event.
             * 
             * This event is dispatched by a Data Manager when an item in the data store is changed.
             * 
             * Game Objects with data enabled have an instance of a Data Manager under the `data` property. So, to listen for
             * the change of a specific data item from a Game Object you would use: `sprite.data.on('changedata-key', listener)`,
             * where `key` is the unique string key of the data item. For example, if you have a data item stored called `gold`
             * then you can listen for `sprite.data.on('changedata-gold')`.
             */
            const CHANGE_DATA_KEY: any;

            /**
             * The Data Manager Destroy Event.
             * 
             * The Data Manager will listen for the destroy event from its parent, and then close itself down.
             */
            const DESTROY: any;

            /**
             * The Remove Data Event.
             * 
             * This event is dispatched by a Data Manager when an item is removed from it.
             * 
             * Game Objects with data enabled have an instance of a Data Manager under the `data` property. So, to listen for
             * the removal of a data item on a Game Object you would use: `sprite.data.on('removedata', listener)`.
             */
            const REMOVE_DATA: any;

            /**
             * The Set Data Event.
             * 
             * This event is dispatched by a Data Manager when a new item is added to the data store.
             * 
             * Game Objects with data enabled have an instance of a Data Manager under the `data` property. So, to listen for
             * the addition of a new data item on a Game Object you would use: `sprite.data.on('setdata', listener)`.
             */
            const SET_DATA: any;

        }

    }

    namespace Device {
        /**
         * Determines the audio playback capabilities of the device running this Phaser Game instance.
         * These values are read-only and populated during the boot sequence of the game.
         * They are then referenced by internal game systems and are available for you to access
         * via `this.sys.game.device.audio` from within any Scene.
         */
        type Audio = {
            /**
             * Can this device play HTML Audio tags?
             */
            audioData: boolean;
            /**
             * Can this device play EC-3 Dolby Digital Plus files?
             */
            dolby: boolean;
            /**
             * Can this device can play m4a files.
             */
            m4a: boolean;
            /**
             * Can this device play mp3 files?
             */
            mp3: boolean;
            /**
             * Can this device play ogg files?
             */
            ogg: boolean;
            /**
             * Can this device play opus files?
             */
            opus: boolean;
            /**
             * Can this device play wav files?
             */
            wav: boolean;
            /**
             * Does this device have the Web Audio API?
             */
            webAudio: boolean;
            /**
             * Can this device play webm files?
             */
            webm: boolean;
        };

        /**
         * Determines the browser type and version running this Phaser Game instance.
         * These values are read-only and populated during the boot sequence of the game.
         * They are then referenced by internal game systems and are available for you to access
         * via `this.sys.game.device.browser` from within any Scene.
         */
        type Browser = {
            /**
             * Set to true if running in Chrome.
             */
            chrome: boolean;
            /**
             * Set to true if running in Microsoft Edge browser.
             */
            edge: boolean;
            /**
             * Set to true if running in Firefox.
             */
            firefox: boolean;
            /**
             * Set to true if running in Internet Explorer 11 or less (not Edge).
             */
            ie: boolean;
            /**
             * Set to true if running in Mobile Safari.
             */
            mobileSafari: boolean;
            /**
             * Set to true if running in Opera.
             */
            opera: boolean;
            /**
             * Set to true if running in Safari.
             */
            safari: boolean;
            /**
             * Set to true if running in the Silk browser (as used on the Amazon Kindle)
             */
            silk: boolean;
            /**
             * Set to true if running a Trident version of Internet Explorer (IE11+)
             */
            trident: boolean;
            /**
             * If running in Chrome this will contain the major version number.
             */
            chromeVersion: number;
            /**
             * If running in Firefox this will contain the major version number.
             */
            firefoxVersion: number;
            /**
             * If running in Internet Explorer this will contain the major version number. Beyond IE10 you should use Browser.trident and Browser.tridentVersion.
             */
            ieVersion: number;
            /**
             * If running in Safari this will contain the major version number.
             */
            safariVersion: number;
            /**
             * If running in Internet Explorer 11 this will contain the major version number. See {@link http://msdn.microsoft.com/en-us/library/ie/ms537503(v=vs.85).aspx}
             */
            tridentVersion: number;
        };

        /**
         * Determines the canvas features of the browser running this Phaser Game instance.
         * These values are read-only and populated during the boot sequence of the game.
         * They are then referenced by internal game systems and are available for you to access
         * via `this.sys.game.device.canvasFeatures` from within any Scene.
         */
        type CanvasFeatures = {
            /**
             * Set to true if the browser supports inversed alpha.
             */
            supportInverseAlpha: boolean;
            /**
             * Set to true if the browser supports new canvas blend modes.
             */
            supportNewBlendModes: boolean;
        };

        /**
         * Determines the features of the browser running this Phaser Game instance.
         * These values are read-only and populated during the boot sequence of the game.
         * They are then referenced by internal game systems and are available for you to access
         * via `this.sys.game.device.features` from within any Scene.
         */
        type Features = {
            /**
             * True if canvas supports a 'copy' bitblt onto itself when the source and destination regions overlap.
             */
            canvasBitBltShift: boolean;
            /**
             * Is canvas available?
             */
            canvas: boolean;
            /**
             * Is file available?
             */
            file: boolean;
            /**
             * Is fileSystem available?
             */
            fileSystem: boolean;
            /**
             * Does the device support the getUserMedia API?
             */
            getUserMedia: boolean;
            /**
             * Is the device big or little endian? (only detected if the browser supports TypedArrays)
             */
            littleEndian: boolean;
            /**
             * Is localStorage available?
             */
            localStorage: boolean;
            /**
             * Is Pointer Lock available?
             */
            pointerLock: boolean;
            /**
             * Does the device context support 32bit pixel manipulation using array buffer views?
             */
            support32bit: boolean;
            /**
             * Does the device support the Vibration API?
             */
            vibration: boolean;
            /**
             * Is webGL available?
             */
            webGL: boolean;
            /**
             * Is worker available?
             */
            worker: boolean;
        };

        /**
         * Determines the full screen support of the browser running this Phaser Game instance.
         * These values are read-only and populated during the boot sequence of the game.
         * They are then referenced by internal game systems and are available for you to access
         * via `this.sys.game.device.fullscreen` from within any Scene.
         */
        type Fullscreen = {
            /**
             * Does the browser support the Full Screen API?
             */
            available: boolean;
            /**
             * Does the browser support access to the Keyboard during Full Screen mode?
             */
            keyboard: boolean;
            /**
             * If the browser supports the Full Screen API this holds the call you need to use to cancel it.
             */
            cancel: string;
            /**
             * If the browser supports the Full Screen API this holds the call you need to use to activate it.
             */
            request: string;
        };

        /**
         * Determines the input support of the browser running this Phaser Game instance.
         * These values are read-only and populated during the boot sequence of the game.
         * They are then referenced by internal game systems and are available for you to access
         * via `this.sys.game.device.input` from within any Scene.
         */
        type Input = {
            /**
             * The newest type of Wheel/Scroll event supported: 'wheel', 'mousewheel', 'DOMMouseScroll'
             */
            wheelType: string;
            /**
             * Is navigator.getGamepads available?
             */
            gamepads: boolean;
            /**
             * Is mspointer available?
             */
            mspointer: boolean;
            /**
             * Is touch available?
             */
            touch: boolean;
        };

        /**
         * Determines the operating system of the device running this Phaser Game instance.
         * These values are read-only and populated during the boot sequence of the game.
         * They are then referenced by internal game systems and are available for you to access
         * via `this.sys.game.device.os` from within any Scene.
         */
        type OS = {
            /**
             * Is running on android?
             */
            android: boolean;
            /**
             * Is running on chromeOS?
             */
            chromeOS: boolean;
            /**
             * Is the game running under Apache Cordova?
             */
            cordova: boolean;
            /**
             * Is the game running under the Intel Crosswalk XDK?
             */
            crosswalk: boolean;
            /**
             * Is running on a desktop?
             */
            desktop: boolean;
            /**
             * Is the game running under Ejecta?
             */
            ejecta: boolean;
            /**
             * Is the game running under GitHub Electron?
             */
            electron: boolean;
            /**
             * Is running on iOS?
             */
            iOS: boolean;
            /**
             * Is running on iPad?
             */
            iPad: boolean;
            /**
             * Is running on iPhone?
             */
            iPhone: boolean;
            /**
             * Is running on an Amazon Kindle?
             */
            kindle: boolean;
            /**
             * Is running on linux?
             */
            linux: boolean;
            /**
             * Is running on macOS?
             */
            macOS: boolean;
            /**
             * Is the game running under Node.js?
             */
            node: boolean;
            /**
             * Is the game running under Node-Webkit?
             */
            nodeWebkit: boolean;
            /**
             * Set to true if running as a WebApp, i.e. within a WebView
             */
            webApp: boolean;
            /**
             * Is running on windows?
             */
            windows: boolean;
            /**
             * Is running on a Windows Phone?
             */
            windowsPhone: boolean;
            /**
             * If running in iOS this will contain the major version number.
             */
            iOSVersion: number;
            /**
             * PixelRatio of the host device?
             */
            pixelRatio: number;
        };

        /**
         * Determines the video support of the browser running this Phaser Game instance.
         * These values are read-only and populated during the boot sequence of the game.
         * They are then referenced by internal game systems and are available for you to access
         * via `this.sys.game.device.video` from within any Scene.
         * 
         * In Phaser 3.20 the properties were renamed to drop the 'Video' suffix.
         */
        type Video = {
            /**
             * Can this device play h264 mp4 video files?
             */
            h264: boolean;
            /**
             * Can this device play hls video files?
             */
            hls: boolean;
            /**
             * Can this device play h264 mp4 video files?
             */
            mp4: boolean;
            /**
             * Can this device play ogg video files?
             */
            ogg: boolean;
            /**
             * Can this device play vp9 video files?
             */
            vp9: boolean;
            /**
             * Can this device play webm video files?
             */
            webm: boolean;
        };

    }

    type DeviceConf = {
        /**
         * The OS Device functions.
         */
        os: Phaser.Device.OS;
        /**
         * The Browser Device functions.
         */
        browser: Phaser.Device.Browser;
        /**
         * The Features Device functions.
         */
        features: Phaser.Device.Features;
        /**
         * The Input Device functions.
         */
        input: Phaser.Device.Input;
        /**
         * The Audio Device functions.
         */
        audio: Phaser.Device.Audio;
        /**
         * The Video Device functions.
         */
        video: Phaser.Device.Video;
        /**
         * The Fullscreen Device functions.
         */
        fullscreen: Phaser.Device.Fullscreen;
        /**
         * The Canvas Device functions.
         */
        canvasFeatures: Phaser.Device.CanvasFeatures;
    };

    namespace Display {
        namespace Align {
            /**
             * A constant representing a top-left alignment or position.
             */
            const TOP_LEFT: number;

            /**
             * A constant representing a top-center alignment or position.
             */
            const TOP_CENTER: number;

            /**
             * A constant representing a top-right alignment or position.
             */
            const TOP_RIGHT: number;

            /**
             * A constant representing a left-top alignment or position.
             */
            const LEFT_TOP: number;

            /**
             * A constant representing a left-center alignment or position.
             */
            const LEFT_CENTER: number;

            /**
             * A constant representing a left-bottom alignment or position.
             */
            const LEFT_BOTTOM: number;

            /**
             * A constant representing a center alignment or position.
             */
            const CENTER: number;

            /**
             * A constant representing a right-top alignment or position.
             */
            const RIGHT_TOP: number;

            /**
             * A constant representing a right-center alignment or position.
             */
            const RIGHT_CENTER: number;

            /**
             * A constant representing a right-bottom alignment or position.
             */
            const RIGHT_BOTTOM: number;

            /**
             * A constant representing a bottom-left alignment or position.
             */
            const BOTTOM_LEFT: number;

            /**
             * A constant representing a bottom-center alignment or position.
             */
            const BOTTOM_CENTER: number;

            /**
             * A constant representing a bottom-right alignment or position.
             */
            const BOTTOM_RIGHT: number;

            namespace In {
                /**
                 * Takes given Game Object and aligns it so that it is positioned in the bottom center of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignIn The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function BottomCenter<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignIn: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned in the bottom left of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignIn The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function BottomLeft<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignIn: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned in the bottom right of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignIn The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function BottomRight<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignIn: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned in the center of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignIn The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function Center<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignIn: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned in the left center of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignIn The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function LeftCenter<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignIn: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned relative to the other.
                 * The alignment used is based on the `position` argument, which is an `ALIGN_CONST` value, such as `LEFT_CENTER` or `TOP_RIGHT`.
                 * @param child The Game Object that will be positioned.
                 * @param alignIn The Game Object to base the alignment position on.
                 * @param position The position to align the Game Object with. This is an align constant, such as `ALIGN_CONST.LEFT_CENTER`.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function QuickSet<G extends Phaser.GameObjects.GameObject>(child: G, alignIn: Phaser.GameObjects.GameObject, position: number, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned in the right center of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignIn The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function RightCenter<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignIn: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned in the top center of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignIn The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function TopCenter<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignIn: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned in the top left of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignIn The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function TopLeft<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignIn: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned in the top right of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignIn The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function TopRight<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignIn: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

            }

            namespace To {
                /**
                 * Takes given Game Object and aligns it so that it is positioned next to the bottom center position of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignTo The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function BottomCenter<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignTo: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned next to the bottom left position of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignTo The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function BottomLeft<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignTo: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned next to the bottom right position of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignTo The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function BottomRight<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignTo: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned next to the left bottom position of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignTo The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function LeftBottom<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignTo: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned next to the left center position of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignTo The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function LeftCenter<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignTo: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned next to the left top position of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignTo The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function LeftTop<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignTo: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes a Game Object and aligns it next to another, at the given position.
                 * The alignment used is based on the `position` argument, which is a `Phaser.Display.Align` property such as `LEFT_CENTER` or `TOP_RIGHT`.
                 * @param child The Game Object that will be positioned.
                 * @param alignTo The Game Object to base the alignment position on.
                 * @param position The position to align the Game Object with. This is an align constant, such as `Phaser.Display.Align.LEFT_CENTER`.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function QuickSet<G extends Phaser.GameObjects.GameObject>(child: G, alignTo: Phaser.GameObjects.GameObject, position: number, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned next to the right bottom position of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignTo The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function RightBottom<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignTo: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned next to the right center position of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignTo The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function RightCenter<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignTo: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned next to the right top position of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignTo The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function RightTop<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignTo: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned next to the top center position of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignTo The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function TopCenter<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignTo: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned next to the top left position of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignTo The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function TopLeft<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignTo: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned next to the top right position of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignTo The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function TopRight<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignTo: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

            }

        }

        namespace Bounds {
            /**
             * Positions the Game Object so that it is centered on the given coordinates.
             * @param gameObject The Game Object that will be re-positioned.
             * @param x The horizontal coordinate to position the Game Object on.
             * @param y The vertical coordinate to position the Game Object on.
             */
            function CenterOn<G extends Phaser.GameObjects.GameObject>(gameObject: G, x: number, y: number): G;

            /**
             * Returns the bottom coordinate from the bounds of the Game Object.
             * @param gameObject The Game Object to get the bounds value from.
             */
            function GetBottom(gameObject: Phaser.GameObjects.GameObject): number;

            /**
             * Returns the unrotated bounds of the Game Object as a rectangle.
             * @param gameObject The Game Object to get the bounds value from.
             * @param output An object to store the values in.
             */
            function GetBounds(gameObject: Phaser.GameObjects.GameObject, output?: Phaser.Geom.Rectangle | object): Phaser.Geom.Rectangle | object;

            /**
             * Returns the center x coordinate from the bounds of the Game Object.
             * @param gameObject The Game Object to get the bounds value from.
             */
            function GetCenterX(gameObject: Phaser.GameObjects.GameObject): number;

            /**
             * Returns the center y coordinate from the bounds of the Game Object.
             * @param gameObject The Game Object to get the bounds value from.
             */
            function GetCenterY(gameObject: Phaser.GameObjects.GameObject): number;

            /**
             * Returns the left coordinate from the bounds of the Game Object.
             * @param gameObject The Game Object to get the bounds value from.
             */
            function GetLeft(gameObject: Phaser.GameObjects.GameObject): number;

            /**
             * Returns the amount the Game Object is visually offset from its x coordinate.
             * This is the same as `width * origin.x`.
             * This value will only be > 0 if `origin.x` is not equal to zero.
             * @param gameObject The Game Object to get the bounds value from.
             */
            function GetOffsetX(gameObject: Phaser.GameObjects.GameObject): number;

            /**
             * Returns the amount the Game Object is visually offset from its y coordinate.
             * This is the same as `width * origin.y`.
             * This value will only be > 0 if `origin.y` is not equal to zero.
             * @param gameObject The Game Object to get the bounds value from.
             */
            function GetOffsetY(gameObject: Phaser.GameObjects.GameObject): number;

            /**
             * Returns the right coordinate from the bounds of the Game Object.
             * @param gameObject The Game Object to get the bounds value from.
             */
            function GetRight(gameObject: Phaser.GameObjects.GameObject): number;

            /**
             * Returns the top coordinate from the bounds of the Game Object.
             * @param gameObject The Game Object to get the bounds value from.
             */
            function GetTop(gameObject: Phaser.GameObjects.GameObject): number;

            /**
             * Positions the Game Object so that the bottom of its bounds aligns with the given coordinate.
             * @param gameObject The Game Object that will be re-positioned.
             * @param value The coordinate to position the Game Object bounds on.
             */
            function SetBottom<G extends Phaser.GameObjects.GameObject>(gameObject: G, value: number): G;

            /**
             * Positions the Game Object so that the center top of its bounds aligns with the given coordinate.
             * @param gameObject The Game Object that will be re-positioned.
             * @param x The coordinate to position the Game Object bounds on.
             */
            function SetCenterX<G extends Phaser.GameObjects.GameObject>(gameObject: G, x: number): G;

            /**
             * Positions the Game Object so that the center top of its bounds aligns with the given coordinate.
             * @param gameObject The Game Object that will be re-positioned.
             * @param y The coordinate to position the Game Object bounds on.
             */
            function SetCenterY<G extends Phaser.GameObjects.GameObject>(gameObject: G, y: number): G;

            /**
             * Positions the Game Object so that the left of its bounds aligns with the given coordinate.
             * @param gameObject The Game Object that will be re-positioned.
             * @param value The coordinate to position the Game Object bounds on.
             */
            function SetLeft<G extends Phaser.GameObjects.GameObject>(gameObject: G, value: number): G;

            /**
             * Positions the Game Object so that the left of its bounds aligns with the given coordinate.
             * @param gameObject The Game Object that will be re-positioned.
             * @param value The coordinate to position the Game Object bounds on.
             */
            function SetRight<G extends Phaser.GameObjects.GameObject>(gameObject: G, value: number): G;

            /**
             * Positions the Game Object so that the top of its bounds aligns with the given coordinate.
             * @param gameObject The Game Object that will be re-positioned.
             * @param value The coordinate to position the Game Object bounds on.
             */
            function SetTop<G extends Phaser.GameObjects.GameObject>(gameObject: G, value: number): G;

        }

        namespace Canvas {
            namespace CanvasInterpolation {
                /**
                 * Sets the CSS image-rendering property on the given canvas to be 'crisp' (aka 'optimize contrast' on webkit).
                 * @param canvas The canvas object to have the style set on.
                 */
                function setCrisp(canvas: HTMLCanvasElement): HTMLCanvasElement;

                /**
                 * Sets the CSS image-rendering property on the given canvas to be 'bicubic' (aka 'auto').
                 * @param canvas The canvas object to have the style set on.
                 */
                function setBicubic(canvas: HTMLCanvasElement): HTMLCanvasElement;

            }

            /**
             * The CanvasPool is a global static object, that allows Phaser to recycle and pool 2D Context Canvas DOM elements.
             * It does not pool WebGL Contexts, because once the context options are set they cannot be modified again, 
             * which is useless for some of the Phaser pipelines / renderer.
             * 
             * This singleton is instantiated as soon as Phaser loads, before a Phaser.Game instance has even been created.
             * Which means all instances of Phaser Games on the same page can share the one single pool.
             */
            namespace CanvasPool {
                /**
                 * Creates a new Canvas DOM element, or pulls one from the pool if free.
                 * @param parent The parent of the Canvas object.
                 * @param width The width of the Canvas. Default 1.
                 * @param height The height of the Canvas. Default 1.
                 * @param canvasType The type of the Canvas. Either `Phaser.CANVAS` or `Phaser.WEBGL`. Default Phaser.CANVAS.
                 * @param selfParent Use the generated Canvas element as the parent? Default false.
                 */
                function create(parent: any, width?: number, height?: number, canvasType?: number, selfParent?: boolean): HTMLCanvasElement;

                /**
                 * Creates a new Canvas DOM element, or pulls one from the pool if free.
                 * @param parent The parent of the Canvas object.
                 * @param width The width of the Canvas. Default 1.
                 * @param height The height of the Canvas. Default 1.
                 */
                function create2D(parent: any, width?: number, height?: number): HTMLCanvasElement;

                /**
                 * Creates a new Canvas DOM element, or pulls one from the pool if free.
                 * @param parent The parent of the Canvas object.
                 * @param width The width of the Canvas. Default 1.
                 * @param height The height of the Canvas. Default 1.
                 */
                function createWebGL(parent: any, width?: number, height?: number): HTMLCanvasElement;

                /**
                 * Gets the first free canvas index from the pool.
                 * @param canvasType The type of the Canvas. Either `Phaser.CANVAS` or `Phaser.WEBGL`. Default Phaser.CANVAS.
                 */
                function first(canvasType?: number): HTMLCanvasElement;

                /**
                 * Looks up a canvas based on its parent, and if found puts it back in the pool, freeing it up for re-use.
                 * The canvas has its width and height set to 1, and its parent attribute nulled.
                 * @param parent The canvas or the parent of the canvas to free.
                 */
                function remove(parent: any): void;

                /**
                 * Gets the total number of used canvas elements in the pool.
                 */
                function total(): number;

                /**
                 * Gets the total number of free canvas elements in the pool.
                 */
                function free(): number;

                /**
                 * Disable context smoothing on any new Canvas element created.
                 */
                function disableSmoothing(): void;

                /**
                 * Enable context smoothing on any new Canvas element created.
                 */
                function enableSmoothing(): void;

            }

            namespace Smoothing {
                /**
                 * Gets the Smoothing Enabled vendor prefix being used on the given context, or null if not set.
                 * @param context The canvas context to check.
                 */
                function getPrefix(context: CanvasRenderingContext2D | WebGLRenderingContext): string;

                /**
                 * Sets the Image Smoothing property on the given context. Set to false to disable image smoothing.
                 * By default browsers have image smoothing enabled, which isn't always what you visually want, especially
                 * when using pixel art in a game. Note that this sets the property on the context itself, so that any image
                 * drawn to the context will be affected. This sets the property across all current browsers but support is
                 * patchy on earlier browsers, especially on mobile.
                 * @param context The context on which to enable smoothing.
                 */
                function enable(context: CanvasRenderingContext2D | WebGLRenderingContext): CanvasRenderingContext2D | WebGLRenderingContext;

                /**
                 * Sets the Image Smoothing property on the given context. Set to false to disable image smoothing.
                 * By default browsers have image smoothing enabled, which isn't always what you visually want, especially
                 * when using pixel art in a game. Note that this sets the property on the context itself, so that any image
                 * drawn to the context will be affected. This sets the property across all current browsers but support is
                 * patchy on earlier browsers, especially on mobile.
                 * @param context The context on which to disable smoothing.
                 */
                function disable(context: CanvasRenderingContext2D | WebGLRenderingContext): CanvasRenderingContext2D | WebGLRenderingContext;

                /**
                 * Returns `true` if the given context has image smoothing enabled, otherwise returns `false`.
                 * Returns null if no smoothing prefix is available.
                 * @param context The context to check.
                 */
                function isEnabled(context: CanvasRenderingContext2D | WebGLRenderingContext): boolean;

            }

            /**
             * Sets the touch-action property on the canvas style. Can be used to disable default browser touch actions.
             * @param canvas The canvas element to have the style applied to.
             * @param value The touch action value to set on the canvas. Set to `none` to disable touch actions. Default 'none'.
             */
            function TouchAction(canvas: HTMLCanvasElement, value?: string): HTMLCanvasElement;

            /**
             * Sets the user-select property on the canvas style. Can be used to disable default browser selection actions.
             * @param canvas The canvas element to have the style applied to.
             * @param value The touch callout value to set on the canvas. Set to `none` to disable touch callouts. Default 'none'.
             */
            function UserSelect(canvas: HTMLCanvasElement, value?: string): HTMLCanvasElement;

        }

        namespace Color {
            namespace Interpolate {
                /**
                 * Interpolates between the two given color ranges over the length supplied.
                 * @param r1 Red value.
                 * @param g1 Blue value.
                 * @param b1 Green value.
                 * @param r2 Red value.
                 * @param g2 Blue value.
                 * @param b2 Green value.
                 * @param length Distance to interpolate over. Default 100.
                 * @param index Index to start from. Default 0.
                 */
                function RGBWithRGB(r1: number, g1: number, b1: number, r2: number, g2: number, b2: number, length?: number, index?: number): Phaser.Types.Display.ColorObject;

                /**
                 * Interpolates between the two given color objects over the length supplied.
                 * @param color1 The first Color object.
                 * @param color2 The second Color object.
                 * @param length Distance to interpolate over. Default 100.
                 * @param index Index to start from. Default 0.
                 */
                function ColorWithColor(color1: Phaser.Display.Color, color2: Phaser.Display.Color, length?: number, index?: number): Phaser.Types.Display.ColorObject;

                /**
                 * Interpolates between the Color object and color values over the length supplied.
                 * @param color1 The first Color object.
                 * @param r Red value.
                 * @param g Blue value.
                 * @param b Green value.
                 * @param length Distance to interpolate over. Default 100.
                 * @param index Index to start from. Default 0.
                 */
                function ColorWithRGB(color1: Phaser.Display.Color, r: number, g: number, b: number, length?: number, index?: number): Phaser.Types.Display.ColorObject;

            }

        }

        /**
         * The Color class holds a single color value and allows for easy modification and reading of it.
         */
        class Color {
            /**
             * 
             * @param red The red color value. A number between 0 and 255. Default 0.
             * @param green The green color value. A number between 0 and 255. Default 0.
             * @param blue The blue color value. A number between 0 and 255. Default 0.
             * @param alpha The alpha value. A number between 0 and 255. Default 255.
             */
            constructor(red?: number, green?: number, blue?: number, alpha?: number);

            /**
             * An array containing the calculated color values for WebGL use.
             */
            gl: number[];

            /**
             * Sets this color to be transparent. Sets all values to zero.
             */
            transparent(): Phaser.Display.Color;

            /**
             * Sets the color of this Color component.
             * @param red The red color value. A number between 0 and 255.
             * @param green The green color value. A number between 0 and 255.
             * @param blue The blue color value. A number between 0 and 255.
             * @param alpha The alpha value. A number between 0 and 255. Default 255.
             * @param updateHSV Update the HSV values after setting the RGB values? Default true.
             */
            setTo(red: number, green: number, blue: number, alpha?: number, updateHSV?: boolean): Phaser.Display.Color;

            /**
             * Sets the red, green, blue and alpha GL values of this Color component.
             * @param red The red color value. A number between 0 and 1.
             * @param green The green color value. A number between 0 and 1.
             * @param blue The blue color value. A number between 0 and 1.
             * @param alpha The alpha value. A number between 0 and 1. Default 1.
             */
            setGLTo(red: number, green: number, blue: number, alpha?: number): Phaser.Display.Color;

            /**
             * Sets the color based on the color object given.
             * @param color An object containing `r`, `g`, `b` and optionally `a` values in the range 0 to 255.
             */
            setFromRGB(color: Phaser.Types.Display.InputColorObject): Phaser.Display.Color;

            /**
             * Sets the color based on the hue, saturation and lightness values given.
             * @param h The hue, in the range 0 - 1. This is the base color.
             * @param s The saturation, in the range 0 - 1. This controls how much of the hue will be in the final color, where 1 is fully saturated and 0 will give you white.
             * @param v The value, in the range 0 - 1. This controls how dark the color is. Where 1 is as bright as possible and 0 is black.
             */
            setFromHSV(h: number, s: number, v: number): Phaser.Display.Color;

            /**
             * Returns a new Color component using the values from this one.
             */
            clone(): Phaser.Display.Color;

            /**
             * Sets this Color object to be grayscaled based on the shade value given.
             * @param shade A value between 0 and 255.
             */
            gray(shade: number): Phaser.Display.Color;

            /**
             * Sets this Color object to be a random color between the `min` and `max` values given.
             * @param min The minimum random color value. Between 0 and 255. Default 0.
             * @param max The maximum random color value. Between 0 and 255. Default 255.
             */
            random(min?: number, max?: number): Phaser.Display.Color;

            /**
             * Sets this Color object to be a random grayscale color between the `min` and `max` values given.
             * @param min The minimum random color value. Between 0 and 255. Default 0.
             * @param max The maximum random color value. Between 0 and 255. Default 255.
             */
            randomGray(min?: number, max?: number): Phaser.Display.Color;

            /**
             * Increase the saturation of this Color by the percentage amount given.
             * The saturation is the amount of the base color in the hue.
             * @param amount The percentage amount to change this color by. A value between 0 and 100.
             */
            saturate(amount: number): Phaser.Display.Color;

            /**
             * Decrease the saturation of this Color by the percentage amount given.
             * The saturation is the amount of the base color in the hue.
             * @param amount The percentage amount to change this color by. A value between 0 and 100.
             */
            desaturate(amount: number): Phaser.Display.Color;

            /**
             * Increase the lightness of this Color by the percentage amount given.
             * @param amount The percentage amount to change this color by. A value between 0 and 100.
             */
            lighten(amount: number): Phaser.Display.Color;

            /**
             * Decrease the lightness of this Color by the percentage amount given.
             * @param amount The percentage amount to change this color by. A value between 0 and 100.
             */
            darken(amount: number): Phaser.Display.Color;

            /**
             * Brighten this Color by the percentage amount given.
             * @param amount The percentage amount to change this color by. A value between 0 and 100.
             */
            brighten(amount: number): Phaser.Display.Color;

            /**
             * The color of this Color component, not including the alpha channel.
             */
            readonly color: number;

            /**
             * The color of this Color component, including the alpha channel.
             */
            readonly color32: number;

            /**
             * The color of this Color component as a string which can be used in CSS color values.
             */
            readonly rgba: string;

            /**
             * The red color value, normalized to the range 0 to 1.
             */
            redGL: number;

            /**
             * The green color value, normalized to the range 0 to 1.
             */
            greenGL: number;

            /**
             * The blue color value, normalized to the range 0 to 1.
             */
            blueGL: number;

            /**
             * The alpha color value, normalized to the range 0 to 1.
             */
            alphaGL: number;

            /**
             * The red color value, normalized to the range 0 to 255.
             */
            red: number;

            /**
             * The green color value, normalized to the range 0 to 255.
             */
            green: number;

            /**
             * The blue color value, normalized to the range 0 to 255.
             */
            blue: number;

            /**
             * The alpha color value, normalized to the range 0 to 255.
             */
            alpha: number;

            /**
             * The hue color value. A number between 0 and 1.
             * This is the base color.
             */
            h: number;

            /**
             * The saturation color value. A number between 0 and 1.
             * This controls how much of the hue will be in the final color, where 1 is fully saturated and 0 will give you white.
             */
            s: number;

            /**
             * The lightness color value. A number between 0 and 1.
             * This controls how dark the color is. Where 1 is as bright as possible and 0 is black.
             */
            v: number;

            /**
             * Return an array of Colors in a Color Spectrum.
             * 
             * The spectrum colors flow in the order: red, yellow, green, blue.
             * 
             * By default this function will return an array with 1024 elements in.
             * 
             * However, you can reduce this to a smaller quantity if needed, by specitying the `limit` parameter.
             * @param limit How many colors should be returned? The maximum is 1024 but you can set a smaller quantity if required. Default 1024.
             */
            static ColorSpectrum(limit?: number): Phaser.Types.Display.ColorObject[];

            /**
             * Converts the given color value into an Object containing r,g,b and a properties.
             * @param color A color value, optionally including the alpha value.
             */
            static ColorToRGBA(color: number): Phaser.Types.Display.ColorObject;

            /**
             * Returns a string containing a hex representation of the given color component.
             * @param color The color channel to get the hex value for, must be a value between 0 and 255.
             */
            static ComponentToHex(color: number): string;

            /**
             * Given 3 separate color values this will return an integer representation of it.
             * @param red The red color value. A number between 0 and 255.
             * @param green The green color value. A number between 0 and 255.
             * @param blue The blue color value. A number between 0 and 255.
             */
            static GetColor(red: number, green: number, blue: number): number;

            /**
             * Given an alpha and 3 color values this will return an integer representation of it.
             * @param red The red color value. A number between 0 and 255.
             * @param green The green color value. A number between 0 and 255.
             * @param blue The blue color value. A number between 0 and 255.
             * @param alpha The alpha color value. A number between 0 and 255.
             */
            static GetColor32(red: number, green: number, blue: number, alpha: number): number;

            /**
             * Converts a hex string into a Phaser Color object.
             * 
             * The hex string can supplied as `'#0033ff'` or the short-hand format of `'#03f'`; it can begin with an optional "#" or "0x", or be unprefixed.
             * 
             * An alpha channel is _not_ supported.
             * @param hex The hex color value to convert, such as `#0033ff` or the short-hand format: `#03f`.
             */
            static HexStringToColor(hex: string): Phaser.Display.Color;

            /**
             * Converts HSL (hue, saturation and lightness) values to a Phaser Color object.
             * @param h The hue value in the range 0 to 1.
             * @param s The saturation value in the range 0 to 1.
             * @param l The lightness value in the range 0 to 1.
             */
            static HSLToColor(h: number, s: number, l: number): Phaser.Display.Color;

            /**
             * Get HSV color wheel values in an array which will be 360 elements in size.
             * @param s The saturation, in the range 0 - 1. Default 1.
             * @param v The value, in the range 0 - 1. Default 1.
             */
            static HSVColorWheel(s?: number, v?: number): Phaser.Types.Display.ColorObject[];

            /**
             * Converts a HSV (hue, saturation and value) color set to RGB.
             * 
             * Conversion formula from https://en.wikipedia.org/wiki/HSL_and_HSV
             * 
             * Assumes HSV values are contained in the set [0, 1].
             * @param h The hue, in the range 0 - 1. This is the base color.
             * @param s The saturation, in the range 0 - 1. This controls how much of the hue will be in the final color, where 1 is fully saturated and 0 will give you white.
             * @param v The value, in the range 0 - 1. This controls how dark the color is. Where 1 is as bright as possible and 0 is black.
             * @param out A Color object to store the results in. If not given a new ColorObject will be created.
             */
            static HSVToRGB(h: number, s: number, v: number, out?: Phaser.Types.Display.ColorObject | Phaser.Display.Color): Phaser.Types.Display.ColorObject | Phaser.Display.Color;

            /**
             * Converts a hue to an RGB color.
             * Based on code by Michael Jackson (https://github.com/mjijackson)
             */
            static HueToComponent(p: number, q: number, t: number): number;

            /**
             * Converts the given color value into an instance of a Color object.
             * @param input The color value to convert into a Color object.
             */
            static IntegerToColor(input: number): Phaser.Display.Color;

            /**
             * Return the component parts of a color as an Object with the properties alpha, red, green, blue.
             * 
             * Alpha will only be set if it exists in the given color (0xAARRGGBB)
             * @param input The color value to convert into a Color object.
             */
            static IntegerToRGB(input: number): Phaser.Types.Display.ColorObject;

            /**
             * Converts an object containing `r`, `g`, `b` and `a` properties into a Color class instance.
             * @param input An object containing `r`, `g`, `b` and `a` properties in the range 0 to 255.
             */
            static ObjectToColor(input: Phaser.Types.Display.InputColorObject): Phaser.Display.Color;

            /**
             * Creates a new Color object where the r, g, and b values have been set to random values
             * based on the given min max values.
             * @param min The minimum value to set the random range from (between 0 and 255) Default 0.
             * @param max The maximum value to set the random range from (between 0 and 255) Default 255.
             */
            static RandomRGB(min?: number, max?: number): Phaser.Display.Color;

            /**
             * Converts a CSS 'web' string into a Phaser Color object.
             * 
             * The web string can be in the format `'rgb(r,g,b)'` or `'rgba(r,g,b,a)'` where r/g/b are in the range [0..255] and a is in the range [0..1].
             * @param rgb The CSS format color string, using the `rgb` or `rgba` format.
             */
            static RGBStringToColor(rgb: string): Phaser.Display.Color;

            /**
             * Converts an RGB color value to HSV (hue, saturation and value).
             * Conversion formula from http://en.wikipedia.org/wiki/HSL_color_space.
             * Assumes RGB values are contained in the set [0, 255] and returns h, s and v in the set [0, 1].
             * Based on code by Michael Jackson (https://github.com/mjijackson)
             * @param r The red color value. A number between 0 and 255.
             * @param g The green color value. A number between 0 and 255.
             * @param b The blue color value. A number between 0 and 255.
             * @param out An object to store the color values in. If not given an HSV Color Object will be created.
             */
            static RGBToHSV(r: number, g: number, b: number, out?: Phaser.Types.Display.HSVColorObject | Phaser.Display.Color): Phaser.Types.Display.HSVColorObject | Phaser.Display.Color;

            /**
             * Converts the color values into an HTML compatible color string, prefixed with either `#` or `0x`.
             * @param r The red color value. A number between 0 and 255.
             * @param g The green color value. A number between 0 and 255.
             * @param b The blue color value. A number between 0 and 255.
             * @param a The alpha value. A number between 0 and 255. Default 255.
             * @param prefix The prefix of the string. Either `#` or `0x`. Default #.
             */
            static RGBToString(r: number, g: number, b: number, a?: number, prefix?: string): string;

            /**
             * Converts the given source color value into an instance of a Color class.
             * The value can be either a string, prefixed with `rgb` or a hex string, a number or an Object.
             * @param input The source color value to convert.
             */
            static ValueToColor(input: string | number | Phaser.Types.Display.InputColorObject): Phaser.Display.Color;

        }

        /**
         * The ColorMatrix class creates a 5x4 matrix that can be used in shaders and graphics
         * operations. It provides methods required to modify the color values, such as adjusting
         * the brightness, setting a sepia tone, hue rotation and more.
         * 
         * Use the method `getData` to return a Float32Array containing the current color values.
         */
        class ColorMatrix {
            /**
             * The value that determines how much of the original color is used
             * when mixing the colors. A value between 0 (all original) and 1 (all final)
             */
            alpha: number;

            /**
             * Sets this ColorMatrix from the given array of color values.
             * @param value The ColorMatrix values to set.
             */
            set(value: number[]): this;

            /**
             * Resets the ColorMatrix.
             */
            reset(): this;

            /**
             * Gets the ColorMatrix as a Float32Array.
             * 
             * Can be used directly as a 1fv shader uniform value.
             */
            getData(): Float32Array;

            /**
             * Changes the brightness of this ColorMatrix by the given amount.
             * @param value The amount of brightness to apply to this ColorMatrix. Between 0 (black) and 1. Default 0.
             * @param multiply Multiply the resulting ColorMatrix (`true`), or set it (`false`) ? Default false.
             */
            brightness(value?: number, multiply?: boolean): this;

            /**
             * Changes the saturation of this ColorMatrix by the given amount.
             * @param value The amount of saturation to apply to this ColorMatrix. Default 0.
             * @param multiply Multiply the resulting ColorMatrix (`true`), or set it (`false`) ? Default false.
             */
            saturate(value?: number, multiply?: boolean): this;

            /**
             * Desaturates this ColorMatrix (removes color from it).
             * @param multiply Multiply the resulting ColorMatrix (`true`), or set it (`false`) ? Default false.
             */
            saturation(multiply?: boolean): this;

            /**
             * Rotates the hues of this ColorMatrix by the value given.
             * @param rotation The amount of hue rotation to apply to this ColorMatrix, in degrees. Default 0.
             * @param multiply Multiply the resulting ColorMatrix (`true`), or set it (`false`) ? Default false.
             */
            hue(rotation?: number, multiply?: boolean): this;

            /**
             * Sets this ColorMatrix to be grayscale.
             * @param value The grayscale scale (0 is black). Default 1.
             * @param multiply Multiply the resulting ColorMatrix (`true`), or set it (`false`) ? Default false.
             */
            grayscale(value?: number, multiply?: boolean): this;

            /**
             * Sets this ColorMatrix to be black and white.
             * @param multiply Multiply the resulting ColorMatrix (`true`), or set it (`false`) ? Default false.
             */
            blackWhite(multiply?: boolean): this;

            /**
             * Change the contrast of this ColorMatrix by the amount given.
             * @param value The amount of contrast to apply to this ColorMatrix. Default 0.
             * @param multiply Multiply the resulting ColorMatrix (`true`), or set it (`false`) ? Default false.
             */
            contrast(value?: number, multiply?: boolean): this;

            /**
             * Converts this ColorMatrix to have negative values.
             * @param multiply Multiply the resulting ColorMatrix (`true`), or set it (`false`) ? Default false.
             */
            negative(multiply?: boolean): this;

            /**
             * Apply a desaturated luminance to this ColorMatrix.
             * @param multiply Multiply the resulting ColorMatrix (`true`), or set it (`false`) ? Default false.
             */
            desaturateLuminance(multiply?: boolean): this;

            /**
             * Applies a sepia tone to this ColorMatrix.
             * @param multiply Multiply the resulting ColorMatrix (`true`), or set it (`false`) ? Default false.
             */
            sepia(multiply?: boolean): this;

            /**
             * Applies a night vision tone to this ColorMatrix.
             * @param intensity The intensity of this effect. Default 0.1.
             * @param multiply Multiply the resulting ColorMatrix (`true`), or set it (`false`) ? Default false.
             */
            night(intensity?: number, multiply?: boolean): this;

            /**
             * Applies a trippy color tone to this ColorMatrix.
             * @param multiply Multiply the resulting ColorMatrix (`true`), or set it (`false`) ? Default false.
             */
            lsd(multiply?: boolean): this;

            /**
             * Applies a brown tone to this ColorMatrix.
             * @param multiply Multiply the resulting ColorMatrix (`true`), or set it (`false`) ? Default false.
             */
            brown(multiply?: boolean): this;

            /**
             * Applies a vintage pinhole color effect to this ColorMatrix.
             * @param multiply Multiply the resulting ColorMatrix (`true`), or set it (`false`) ? Default false.
             */
            vintagePinhole(multiply?: boolean): this;

            /**
             * Applies a kodachrome color effect to this ColorMatrix.
             * @param multiply Multiply the resulting ColorMatrix (`true`), or set it (`false`) ? Default false.
             */
            kodachrome(multiply?: boolean): this;

            /**
             * Applies a technicolor color effect to this ColorMatrix.
             * @param multiply Multiply the resulting ColorMatrix (`true`), or set it (`false`) ? Default false.
             */
            technicolor(multiply?: boolean): this;

            /**
             * Applies a polaroid color effect to this ColorMatrix.
             * @param multiply Multiply the resulting ColorMatrix (`true`), or set it (`false`) ? Default false.
             */
            polaroid(multiply?: boolean): this;

            /**
             * Shifts the values of this ColorMatrix into BGR order.
             * @param multiply Multiply the resulting ColorMatrix (`true`), or set it (`false`) ? Default false.
             */
            shiftToBGR(multiply?: boolean): this;

            /**
             * Multiplies the two given matrices.
             * @param a The 5x4 array to multiply with ColorMatrix._matrix.
             */
            multiply(a: number[]): this;

        }

        namespace Masks {
            /**
             * A Bitmap Mask combines the alpha (opacity) of a masked pixel with the alpha of another pixel.
             * Unlike the Geometry Mask, which is a clipping path, a Bitmap Mask behaves like an alpha mask,
             * not a clipping path. It is only available when using the WebGL Renderer.
             * 
             * A Bitmap Mask can use any Game Object to determine the alpha of each pixel of the masked Game Object(s).
             * For any given point of a masked Game Object's texture, the pixel's alpha will be multiplied by the alpha
             * of the pixel at the same position in the Bitmap Mask's Game Object. The color of the pixel from the
             * Bitmap Mask doesn't matter.
             * 
             * For example, if a pure blue pixel with an alpha of 0.95 is masked with a pure red pixel with an
             * alpha of 0.5, the resulting pixel will be pure blue with an alpha of 0.475. Naturally, this means
             * that a pixel in the mask with an alpha of 0 will hide the corresponding pixel in all masked Game Objects
             *  A pixel with an alpha of 1 in the masked Game Object will receive the same alpha as the
             * corresponding pixel in the mask.
             * 
             * Note: You cannot combine Bitmap Masks and Blend Modes on the same Game Object. You can, however,
             * combine Geometry Masks and Blend Modes together.
             * 
             * The Bitmap Mask's location matches the location of its Game Object, not the location of the
             * masked objects. Moving or transforming the underlying Game Object will change the mask
             * (and affect the visibility of any masked objects), whereas moving or transforming a masked object
             * will not affect the mask.
             * 
             * The Bitmap Mask will not render its Game Object by itself. If the Game Object is not in a
             * Scene's display list, it will only be used for the mask and its full texture will not be directly
             * visible. Adding the underlying Game Object to a Scene will not cause any problems - it will
             * render as a normal Game Object and will also serve as a mask.
             */
            class BitmapMask {
                /**
                 * 
                 * @param scene The Scene which this Bitmap Mask will be used in.
                 * @param renderable A renderable Game Object that uses a texture, such as a Sprite.
                 */
                constructor(scene: Phaser.Scene, renderable: Phaser.GameObjects.GameObject);

                /**
                 * A reference to either the Canvas or WebGL Renderer that this Mask is using.
                 */
                renderer: Phaser.Renderer.Canvas.CanvasRenderer | Phaser.Renderer.WebGL.WebGLRenderer;

                /**
                 * A renderable Game Object that uses a texture, such as a Sprite.
                 */
                bitmapMask: Phaser.GameObjects.GameObject;

                /**
                 * The texture used for the masks framebuffer.
                 */
                maskTexture: WebGLTexture;

                /**
                 * The texture used for the main framebuffer.
                 */
                mainTexture: WebGLTexture;

                /**
                 * Whether the Bitmap Mask is dirty and needs to be updated.
                 */
                dirty: boolean;

                /**
                 * The framebuffer to which a masked Game Object is rendered.
                 */
                mainFramebuffer: WebGLFramebuffer;

                /**
                 * The framebuffer to which the Bitmap Mask's masking Game Object is rendered.
                 */
                maskFramebuffer: WebGLFramebuffer;

                /**
                 * Whether to invert the masks alpha.
                 * 
                 * If `true`, the alpha of the masking pixel will be inverted before it's multiplied with the masked pixel.
                 * Essentially, this means that a masked area will be visible only if the corresponding area in the mask is invisible.
                 */
                invertAlpha: boolean;

                /**
                 * Is this mask a stencil mask?
                 */
                readonly isStencil: boolean;

                /**
                 * Creates the WebGL Texture2D objects and Framebuffers required for this
                 * mask. If this mask has already been created, then `clearMask` is called first.
                 */
                createMask(): void;

                /**
                 * Deletes the `mainTexture` and `maskTexture` WebGL Textures and deletes
                 * the `mainFramebuffer` and `maskFramebuffer` too, nulling all references.
                 * 
                 * This is called when this mask is destroyed, or if you try to creat a new
                 * mask from this object when one is already set.
                 */
                clearMask(): void;

                /**
                 * Sets a new masking Game Object for the Bitmap Mask.
                 * @param renderable A renderable Game Object that uses a texture, such as a Sprite.
                 */
                setBitmap(renderable: Phaser.GameObjects.GameObject): void;

                /**
                 * Prepares the WebGL Renderer to render a Game Object with this mask applied.
                 * 
                 * This renders the masking Game Object to the mask framebuffer and switches to the main framebuffer so that the masked Game Object will be rendered to it instead of being rendered directly to the frame.
                 * @param renderer The WebGL Renderer to prepare.
                 * @param maskedObject The masked Game Object which will be drawn.
                 * @param camera The Camera to render to.
                 */
                preRenderWebGL(renderer: Phaser.Renderer.Canvas.CanvasRenderer | Phaser.Renderer.WebGL.WebGLRenderer, maskedObject: Phaser.GameObjects.GameObject, camera: Phaser.Cameras.Scene2D.Camera): void;

                /**
                 * Finalizes rendering of a masked Game Object.
                 * 
                 * This resets the previously bound framebuffer and switches the WebGL Renderer to the Bitmap Mask Pipeline, which uses a special fragment shader to apply the masking effect.
                 * @param renderer The WebGL Renderer to clean up.
                 */
                postRenderWebGL(renderer: Phaser.Renderer.Canvas.CanvasRenderer | Phaser.Renderer.WebGL.WebGLRenderer): void;

                /**
                 * This is a NOOP method. Bitmap Masks are not supported by the Canvas Renderer.
                 * @param renderer The Canvas Renderer which would be rendered to.
                 * @param mask The masked Game Object which would be rendered.
                 * @param camera The Camera to render to.
                 */
                preRenderCanvas(renderer: Phaser.Renderer.Canvas.CanvasRenderer | Phaser.Renderer.WebGL.WebGLRenderer, mask: Phaser.GameObjects.GameObject, camera: Phaser.Cameras.Scene2D.Camera): void;

                /**
                 * This is a NOOP method. Bitmap Masks are not supported by the Canvas Renderer.
                 * @param renderer The Canvas Renderer which would be rendered to.
                 */
                postRenderCanvas(renderer: Phaser.Renderer.Canvas.CanvasRenderer | Phaser.Renderer.WebGL.WebGLRenderer): void;

                /**
                 * Destroys this BitmapMask and nulls any references it holds.
                 * 
                 * Note that if a Game Object is currently using this mask it will _not_ automatically detect you have destroyed it,
                 * so be sure to call `clearMask` on any Game Object using it, before destroying it.
                 */
                destroy(): void;

            }

            /**
             * A Geometry Mask can be applied to a Game Object to hide any pixels of it which don't intersect
             * a visible pixel from the geometry mask. The mask is essentially a clipping path which can only
             * make a masked pixel fully visible or fully invisible without changing its alpha (opacity).
             * 
             * A Geometry Mask uses a Graphics Game Object to determine which pixels of the masked Game Object(s)
             * should be clipped. For any given point of a masked Game Object's texture, the pixel will only be displayed
             * if the Graphics Game Object of the Geometry Mask has a visible pixel at the same position. The color and
             * alpha of the pixel from the Geometry Mask do not matter.
             * 
             * The Geometry Mask's location matches the location of its Graphics object, not the location of the masked objects.
             * Moving or transforming the underlying Graphics object will change the mask (and affect the visibility
             * of any masked objects), whereas moving or transforming a masked object will not affect the mask.
             * You can think of the Geometry Mask (or rather, of its Graphics object) as an invisible curtain placed
             * in front of all masked objects which has its own visual properties and, naturally, respects the camera's
             * visual properties, but isn't affected by and doesn't follow the masked objects by itself.
             */
            class GeometryMask {
                /**
                 * 
                 * @param scene This parameter is not used.
                 * @param graphicsGeometry The Graphics Game Object to use for the Geometry Mask. Doesn't have to be in the Display List.
                 */
                constructor(scene: Phaser.Scene, graphicsGeometry: Phaser.GameObjects.Graphics);

                /**
                 * The Graphics object which describes the Geometry Mask.
                 */
                geometryMask: Phaser.GameObjects.Graphics;

                /**
                 * Similar to the BitmapMasks invertAlpha setting this to true will then hide all pixels
                 * drawn to the Geometry Mask.
                 * 
                 * This is a WebGL only feature.
                 */
                invertAlpha: boolean;

                /**
                 * Is this mask a stencil mask?
                 */
                readonly isStencil: boolean;

                /**
                 * Sets a new Graphics object for the Geometry Mask.
                 * @param graphicsGeometry The Graphics object which will be used for the Geometry Mask.
                 */
                setShape(graphicsGeometry: Phaser.GameObjects.Graphics): this;

                /**
                 * Sets the `invertAlpha` property of this Geometry Mask.
                 * 
                 * Inverting the alpha essentially flips the way the mask works.
                 * 
                 * This is a WebGL only feature.
                 * @param value Invert the alpha of this mask? Default true.
                 */
                setInvertAlpha(value?: boolean): this;

                /**
                 * Renders the Geometry Mask's underlying Graphics object to the OpenGL stencil buffer and enables the stencil test, which clips rendered pixels according to the mask.
                 * @param renderer The WebGL Renderer instance to draw to.
                 * @param child The Game Object being rendered.
                 * @param camera The camera the Game Object is being rendered through.
                 */
                preRenderWebGL(renderer: Phaser.Renderer.WebGL.WebGLRenderer, child: Phaser.GameObjects.GameObject, camera: Phaser.Cameras.Scene2D.Camera): void;

                /**
                 * Applies the current stencil mask to the renderer.
                 * @param renderer The WebGL Renderer instance to draw to.
                 * @param camera The camera the Game Object is being rendered through.
                 * @param inc Is this an INCR stencil or a DECR stencil?
                 */
                applyStencil(renderer: Phaser.Renderer.WebGL.WebGLRenderer, camera: Phaser.Cameras.Scene2D.Camera, inc: boolean): void;

                /**
                 * Flushes all rendered pixels and disables the stencil test of a WebGL context, thus disabling the mask for it.
                 * @param renderer The WebGL Renderer instance to draw flush.
                 */
                postRenderWebGL(renderer: Phaser.Renderer.WebGL.WebGLRenderer): void;

                /**
                 * Sets the clipping path of a 2D canvas context to the Geometry Mask's underlying Graphics object.
                 * @param renderer The Canvas Renderer instance to set the clipping path on.
                 * @param mask The Game Object being rendered.
                 * @param camera The camera the Game Object is being rendered through.
                 */
                preRenderCanvas(renderer: Phaser.Renderer.Canvas.CanvasRenderer, mask: Phaser.GameObjects.GameObject, camera: Phaser.Cameras.Scene2D.Camera): void;

                /**
                 * Restore the canvas context's previous clipping path, thus turning off the mask for it.
                 * @param renderer The Canvas Renderer instance being restored.
                 */
                postRenderCanvas(renderer: Phaser.Renderer.Canvas.CanvasRenderer): void;

                /**
                 * Destroys this GeometryMask and nulls any references it holds.
                 * 
                 * Note that if a Game Object is currently using this mask it will _not_ automatically detect you have destroyed it,
                 * so be sure to call `clearMask` on any Game Object using it, before destroying it.
                 */
                destroy(): void;

            }

        }

        /**
         * The RGB class holds a single color value and allows for easy modification and reading of it,
         * with optional on-change callback notification and a dirty flag.
         */
        class RGB {
            /**
             * 
             * @param red The red color value. A number between 0 and 1. Default 0.
             * @param green The green color value. A number between 0 and 1. Default 0.
             * @param blue The blue color value. A number between 0 and 1. Default 0.
             */
            constructor(red?: number, green?: number, blue?: number);

            /**
             * This callback will be invoked each time one of the RGB color values change.
             * 
             * The callback is sent the new color values as the parameters.
             */
            onChangeCallback: Function;

            /**
             * Is this color dirty?
             */
            dirty: boolean;

            /**
             * Sets the red, green and blue values of this RGB object, flags it as being
             * dirty and then invokes the `onChangeCallback`, if set.
             * @param red The red color value. A number between 0 and 1. Default 0.
             * @param green The green color value. A number between 0 and 1. Default 0.
             * @param blue The blue color value. A number between 0 and 1. Default 0.
             */
            set(red?: number, green?: number, blue?: number): this;

            /**
             * Compares the given rgb parameters with those in this object and returns
             * a boolean `true` value if they are equal, otherwise it returns `false`.
             * @param red The red value to compare with this object.
             * @param green The green value to compare with this object.
             * @param blue The blue value to compare with this object.
             */
            equals(red: number, green: number, blue: number): boolean;

            /**
             * Internal on change handler. Sets this object as being dirty and
             * then invokes the `onChangeCallback`, if set, passing in the
             * new RGB values.
             */
            onChange(): void;

            /**
             * The red color value. Between 0 and 1.
             * 
             * Changing this property will flag this RGB object as being dirty
             * and invoke the `onChangeCallback` , if set.
             */
            r: number;

            /**
             * The green color value. Between 0 and 1.
             * 
             * Changing this property will flag this RGB object as being dirty
             * and invoke the `onChangeCallback` , if set.
             */
            g: number;

            /**
             * The blue color value. Between 0 and 1.
             * 
             * Changing this property will flag this RGB object as being dirty
             * and invoke the `onChangeCallback` , if set.
             */
            b: number;

            /**
             * Nulls any external references this object contains.
             */
            destroy(): void;

        }

        /**
         * A BaseShader is a small resource class that contains the data required for a WebGL Shader to be created.
         * 
         * It contains the raw source code to the fragment and vertex shader, as well as an object that defines
         * the uniforms the shader requires, if any.
         * 
         * BaseShaders are stored in the Shader Cache, available in a Scene via `this.cache.shaders` and are referenced
         * by a unique key-based string. Retrieve them via `this.cache.shaders.get(key)`.
         * 
         * BaseShaders are created automatically by the GLSL File Loader when loading an external shader resource.
         * They can also be created at runtime, allowing you to use dynamically generated shader source code.
         * 
         * Default fragment and vertex source is used if not provided in the constructor, setting-up a basic shader,
         * suitable for debug rendering.
         */
        class BaseShader {
            /**
             * 
             * @param key The key of this shader. Must be unique within the shader cache.
             * @param fragmentSrc The fragment source for the shader.
             * @param vertexSrc The vertex source for the shader.
             * @param uniforms Optional object defining the uniforms the shader uses.
             */
            constructor(key: string, fragmentSrc?: string, vertexSrc?: string, uniforms?: any);

            /**
             * The key of this shader, unique within the shader cache of this Phaser game instance.
             */
            key: string;

            /**
             * The source code, as a string, of the fragment shader being used.
             */
            fragmentSrc: string;

            /**
             * The source code, as a string, of the vertex shader being used.
             */
            vertexSrc: string;

            /**
             * The default uniforms for this shader.
             */
            uniforms: any;

        }

    }

    namespace DOM {
        /**
         * Adds the given element to the DOM. If a parent is provided the element is added as a child of the parent, providing it was able to access it.
         * If no parent was given it falls back to using `document.body`.
         * @param element The element to be added to the DOM. Usually a Canvas object.
         * @param parent The parent in which to add the element. Can be a string which is passed to `getElementById` or an actual DOM object.
         */
        function AddToDOM(element: HTMLElement, parent?: string | HTMLElement): HTMLElement;

        /**
         * Inspects the readyState of the document. If the document is already complete then it invokes the given callback.
         * If not complete it sets up several event listeners such as `deviceready`, and once those fire, it invokes the callback.
         * Called automatically by the Phaser.Game instance. Should not usually be accessed directly.
         * @param callback The callback to be invoked when the device is ready and the DOM content is loaded.
         */
        function DOMContentLoaded(callback: ContentLoadedCallback): void;

        /**
         * Attempts to determine the document inner height across iOS and standard devices.
         * Based on code by @tylerjpeterson
         * @param iOS Is this running on iOS?
         */
        function GetInnerHeight(iOS: boolean): number;

        /**
         * Attempts to determine the screen orientation using the Orientation API.
         * @param width The width of the viewport.
         * @param height The height of the viewport.
         */
        function GetScreenOrientation(width: number, height: number): string;

        /**
         * Attempts to get the target DOM element based on the given value, which can be either
         * a string, in which case it will be looked-up by ID, or an element node. If nothing
         * can be found it will return a reference to the document.body.
         * @param element The DOM element to look-up.
         */
        function GetTarget(element: HTMLElement): void;

        /**
         * Takes the given data string and parses it as XML.
         * First tries to use the window.DOMParser and reverts to the Microsoft.XMLDOM if that fails.
         * The parsed XML object is returned, or `null` if there was an error while parsing the data.
         * @param data The XML source stored in a string.
         */
        function ParseXML(data: string): DOMParser | ActiveXObject;

        /**
         * Attempts to remove the element from its parentNode in the DOM.
         * @param element The DOM element to remove from its parent node.
         */
        function RemoveFromDOM(element: HTMLElement): void;

        /**
         * Abstracts away the use of RAF or setTimeOut for the core game update loop.
         * This is invoked automatically by the Phaser.Game instance.
         */
        class RequestAnimationFrame {
            /**
             * True if RequestAnimationFrame is running, otherwise false.
             */
            isRunning: boolean;

            /**
             * The callback to be invoked each step.
             */
            callback: FrameRequestCallback;

            /**
             * The most recent timestamp. Either a DOMHighResTimeStamp under RAF or `Date.now` under SetTimeout.
             */
            tick: number;

            /**
             * True if the step is using setTimeout instead of RAF.
             */
            isSetTimeOut: boolean;

            /**
             * The setTimeout or RAF callback ID used when canceling them.
             */
            timeOutID: number;

            /**
             * The previous time the step was called.
             */
            lastTime: number;

            /**
             * The target FPS rate in ms.
             * Only used when setTimeout is used instead of RAF.
             */
            target: number;

            /**
             * The RAF step function.
             * Updates the local tick value, invokes the callback and schedules another call to requestAnimationFrame.
             */
            step: FrameRequestCallback;

            /**
             * The SetTimeout step function.
             * Updates the local tick value, invokes the callback and schedules another call to setTimeout.
             */
            stepTimeout: Function;

            /**
             * Starts the requestAnimationFrame or setTimeout process running.
             * @param callback The callback to invoke each step.
             * @param forceSetTimeOut Should it use SetTimeout, even if RAF is available?
             * @param targetFPS The target fps rate (in ms). Only used when setTimeout is used.
             */
            start(callback: FrameRequestCallback, forceSetTimeOut: boolean, targetFPS: number): void;

            /**
             * Stops the requestAnimationFrame or setTimeout from running.
             */
            stop(): void;

            /**
             * Stops the step from running and clears the callback reference.
             */
            destroy(): void;

        }

    }

    namespace Events {
        /**
         * EventEmitter is a Scene Systems plugin compatible version of eventemitter3.
         */
        class EventEmitter {
            /**
             * Removes all listeners.
             */
            shutdown(): void;

            /**
             * Removes all listeners.
             */
            destroy(): void;

            /**
             * Return an array listing the events for which the emitter has registered listeners.
             */
            eventNames(): (string|symbol)[];

            /**
             * Return the listeners registered for a given event.
             * @param event The event name.
             */
            listeners(event: string | symbol): Function[];

            /**
             * Return the number of listeners listening to a given event.
             * @param event The event name.
             */
            listenerCount(event: string | symbol): number;

            /**
             * Calls each of the listeners registered for a given event.
             * @param event The event name.
             * @param args Additional arguments that will be passed to the event handler.
             */
            emit(event: string | symbol, ...args: any[]): boolean;

            /**
             * Add a listener for a given event.
             * @param event The event name.
             * @param fn The listener function.
             * @param context The context to invoke the listener with. Default this.
             */
            on(event: string | symbol, fn: Function, context?: any): this;

            /**
             * Add a listener for a given event.
             * @param event The event name.
             * @param fn The listener function.
             * @param context The context to invoke the listener with. Default this.
             */
            addListener(event: string | symbol, fn: Function, context?: any): this;

            /**
             * Add a one-time listener for a given event.
             * @param event The event name.
             * @param fn The listener function.
             * @param context The context to invoke the listener with. Default this.
             */
            once(event: string | symbol, fn: Function, context?: any): this;

            /**
             * Remove the listeners of a given event.
             * @param event The event name.
             * @param fn Only remove the listeners that match this function.
             * @param context Only remove the listeners that have this context.
             * @param once Only remove one-time listeners.
             */
            removeListener(event: string | symbol, fn?: Function, context?: any, once?: boolean): this;

            /**
             * Remove the listeners of a given event.
             * @param event The event name.
             * @param fn Only remove the listeners that match this function.
             * @param context Only remove the listeners that have this context.
             * @param once Only remove one-time listeners.
             */
            off(event: string | symbol, fn?: Function, context?: any, once?: boolean): this;

            /**
             * Remove all listeners, or those of the specified event.
             * @param event The event name.
             */
            removeAllListeners(event?: string | symbol): this;

        }

    }

    namespace GameObjects {
        /**
         * BitmapText objects work by taking a texture file and an XML or JSON file that describes the font structure.
         * 
         * During rendering for each letter of the text is rendered to the display, proportionally spaced out and aligned to
         * match the font structure.
         * 
         * Dynamic Bitmap Text objects are different from Static Bitmap Text in that they invoke a callback for each
         * letter being rendered during the render pass. This callback allows you to manipulate the properties of
         * each letter being rendered, such as its position, scale or tint, allowing you to create interesting effects
         * like jiggling text, which can't be done with Static text. This means that Dynamic Text takes more processing
         * time, so only use them if you require the callback ability they have.
         * 
         * BitmapText objects are less flexible than Text objects, in that they have less features such as shadows, fills and the ability
         * to use Web Fonts, however you trade this flexibility for rendering speed. You can also create visually compelling BitmapTexts by
         * processing the font texture in an image editor, applying fills and any other effects required.
         * 
         * To create multi-line text insert \r, \n or \r\n escape codes into the text string.
         * 
         * To create a BitmapText data files you need a 3rd party app such as:
         * 
         * BMFont (Windows, free): {@link http://www.angelcode.com/products/bmfont/|http://www.angelcode.com/products/bmfont/}
         * Glyph Designer (OS X, commercial): {@link http://www.71squared.com/en/glyphdesigner|http://www.71squared.com/en/glyphdesigner}
         * Littera (Web-based, free): {@link http://kvazars.com/littera/|http://kvazars.com/littera/}
         * 
         * For most use cases it is recommended to use XML. If you wish to use JSON, the formatting should be equal to the result of
         * converting a valid XML file through the popular X2JS library. An online tool for conversion can be found here: {@link http://codebeautify.org/xmltojson|http://codebeautify.org/xmltojson}
         */
        class DynamicBitmapText extends Phaser.GameObjects.BitmapText {
            /**
             * 
             * @param scene The Scene to which this Game Object belongs. It can only belong to one Scene at any given time.
             * @param x The x coordinate of this Game Object in world space.
             * @param y The y coordinate of this Game Object in world space.
             * @param font The key of the font to use from the Bitmap Font cache.
             * @param text The string, or array of strings, to be set as the content of this Bitmap Text.
             * @param size The font size of this Bitmap Text.
             * @param align The alignment of the text in a multi-line BitmapText object. Default 0.
             */
            constructor(scene: Phaser.Scene, x: number, y: number, font: string, text?: string | string[], size?: number, align?: number);

            /**
             * The horizontal scroll position of the Bitmap Text.
             */
            scrollX: number;

            /**
             * The vertical scroll position of the Bitmap Text.
             */
            scrollY: number;

            /**
             * The crop width of the Bitmap Text.
             */
            cropWidth: number;

            /**
             * The crop height of the Bitmap Text.
             */
            cropHeight: number;

            /**
             * A callback that alters how each character of the Bitmap Text is rendered.
             */
            displayCallback: Phaser.Types.GameObjects.BitmapText.DisplayCallback;

            /**
             * The data object that is populated during rendering, then passed to the displayCallback.
             * You should modify this object then return it back from the callback. It's updated values
             * will be used to render the specific glyph.
             * 
             * Please note that if you need a reference to this object locally in your game code then you
             * should shallow copy it, as it's updated and re-used for every glyph in the text.
             */
            callbackData: Phaser.Types.GameObjects.BitmapText.DisplayCallbackConfig;

            /**
             * Set the crop size of this Bitmap Text.
             * @param width The width of the crop.
             * @param height The height of the crop.
             */
            setSize(width: number, height: number): this;

            /**
             * Set a callback that alters how each character of the Bitmap Text is rendered.
             * 
             * The callback receives a {@link Phaser.Types.GameObjects.BitmapText.DisplayCallbackConfig} object that contains information about the character that's
             * about to be rendered.
             * 
             * It should return an object with `x`, `y`, `scale` and `rotation` properties that will be used instead of the
             * usual values when rendering.
             * @param callback The display callback to set.
             */
            setDisplayCallback(callback: Phaser.Types.GameObjects.BitmapText.DisplayCallback): this;

            /**
             * Set the horizontal scroll position of this Bitmap Text.
             * @param value The horizontal scroll position to set.
             */
            setScrollX(value: number): this;

            /**
             * Set the vertical scroll position of this Bitmap Text.
             * @param value The vertical scroll position to set.
             */
            setScrollY(value: number): this;

            /**
             * Clears all alpha values associated with this Game Object.
             * 
             * Immediately sets the alpha levels back to 1 (fully opaque).
             */
            clearAlpha(): this;

            /**
             * Set the Alpha level of this Game Object. The alpha controls the opacity of the Game Object as it renders.
             * Alpha values are provided as a float between 0, fully transparent, and 1, fully opaque.
             * 
             * If your game is running under WebGL you can optionally specify four different alpha values, each of which
             * correspond to the four corners of the Game Object. Under Canvas only the `topLeft` value given is used.
             * @param topLeft The alpha value used for the top-left of the Game Object. If this is the only value given it's applied across the whole Game Object. Default 1.
             * @param topRight The alpha value used for the top-right of the Game Object. WebGL only.
             * @param bottomLeft The alpha value used for the bottom-left of the Game Object. WebGL only.
             * @param bottomRight The alpha value used for the bottom-right of the Game Object. WebGL only.
             */
            setAlpha(topLeft?: number, topRight?: number, bottomLeft?: number, bottomRight?: number): this;

            /**
             * The alpha value of the Game Object.
             * 
             * This is a global value, impacting the entire Game Object, not just a region of it.
             */
            alpha: number;

            /**
             * The alpha value starting from the top-left of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaTopLeft: number;

            /**
             * The alpha value starting from the top-right of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaTopRight: number;

            /**
             * The alpha value starting from the bottom-left of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaBottomLeft: number;

            /**
             * The alpha value starting from the bottom-right of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaBottomRight: number;

            /**
             * Sets the Blend Mode being used by this Game Object.
             * 
             * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
             * 
             * Under WebGL only the following Blend Modes are available:
             * 
             * * ADD
             * * MULTIPLY
             * * SCREEN
             * * ERASE
             * 
             * Canvas has more available depending on browser support.
             * 
             * You can also create your own custom Blend Modes in WebGL.
             * 
             * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
             * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
             * reasons try to be careful about the construction of your Scene and the frequency of which blend modes
             * are used.
             */
            blendMode: Phaser.BlendModes | string;

            /**
             * Sets the Blend Mode being used by this Game Object.
             * 
             * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
             * 
             * Under WebGL only the following Blend Modes are available:
             * 
             * * ADD
             * * MULTIPLY
             * * SCREEN
             * * ERASE (only works when rendering to a framebuffer, like a Render Texture)
             * 
             * Canvas has more available depending on browser support.
             * 
             * You can also create your own custom Blend Modes in WebGL.
             * 
             * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
             * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
             * reasons try to be careful about the construction of your Scene and the frequency in which blend modes
             * are used.
             * @param value The BlendMode value. Either a string or a CONST.
             */
            setBlendMode(value: string | Phaser.BlendModes): this;

            /**
             * The depth of this Game Object within the Scene.
             * 
             * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
             * of Game Objects, without actually moving their position in the display list.
             * 
             * The default depth is zero. A Game Object with a higher depth
             * value will always render in front of one with a lower value.
             * 
             * Setting the depth will queue a depth sort event within the Scene.
             */
            depth: number;

            /**
             * The depth of this Game Object within the Scene.
             * 
             * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
             * of Game Objects, without actually moving their position in the display list.
             * 
             * The default depth is zero. A Game Object with a higher depth
             * value will always render in front of one with a lower value.
             * 
             * Setting the depth will queue a depth sort event within the Scene.
             * @param value The depth of this Game Object.
             */
            setDepth(value: number): this;

            /**
             * The Mask this Game Object is using during render.
             */
            mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask;

            /**
             * Sets the mask that this Game Object will use to render with.
             * 
             * The mask must have been previously created and can be either a GeometryMask or a BitmapMask.
             * Note: Bitmap Masks only work on WebGL. Geometry Masks work on both WebGL and Canvas.
             * 
             * If a mask is already set on this Game Object it will be immediately replaced.
             * 
             * Masks are positioned in global space and are not relative to the Game Object to which they
             * are applied. The reason for this is that multiple Game Objects can all share the same mask.
             * 
             * Masks have no impact on physics or input detection. They are purely a rendering component
             * that allows you to limit what is visible during the render pass.
             * @param mask The mask this Game Object will use when rendering.
             */
            setMask(mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask): this;

            /**
             * Clears the mask that this Game Object was using.
             * @param destroyMask Destroy the mask before clearing it? Default false.
             */
            clearMask(destroyMask?: boolean): this;

            /**
             * Creates and returns a Bitmap Mask. This mask can be used by any Game Object,
             * including this one.
             * 
             * Note: Bitmap Masks only work on WebGL. Geometry Masks work on both WebGL and Canvas.
             * 
             * To create the mask you need to pass in a reference to a renderable Game Object.
             * A renderable Game Object is one that uses a texture to render with, such as an
             * Image, Sprite, Render Texture or BitmapText.
             * 
             * If you do not provide a renderable object, and this Game Object has a texture,
             * it will use itself as the object. This means you can call this method to create
             * a Bitmap Mask from any renderable Game Object.
             * @param renderable A renderable Game Object that uses a texture, such as a Sprite.
             */
            createBitmapMask(renderable?: Phaser.GameObjects.GameObject): Phaser.Display.Masks.BitmapMask;

            /**
             * Creates and returns a Geometry Mask. This mask can be used by any Game Object,
             * including this one.
             * 
             * To create the mask you need to pass in a reference to a Graphics Game Object.
             * 
             * If you do not provide a graphics object, and this Game Object is an instance
             * of a Graphics object, then it will use itself to create the mask.
             * 
             * This means you can call this method to create a Geometry Mask from any Graphics Game Object.
             * @param graphics A Graphics Game Object. The geometry within it will be used as the mask.
             */
            createGeometryMask(graphics?: Phaser.GameObjects.Graphics): Phaser.Display.Masks.GeometryMask;

            /**
             * The horizontal origin of this Game Object.
             * The origin maps the relationship between the size and position of the Game Object.
             * The default value is 0.5, meaning all Game Objects are positioned based on their center.
             * Setting the value to 0 means the position now relates to the left of the Game Object.
             */
            originX: number;

            /**
             * The vertical origin of this Game Object.
             * The origin maps the relationship between the size and position of the Game Object.
             * The default value is 0.5, meaning all Game Objects are positioned based on their center.
             * Setting the value to 0 means the position now relates to the top of the Game Object.
             */
            originY: number;

            /**
             * The horizontal display origin of this Game Object.
             * The origin is a normalized value between 0 and 1.
             * The displayOrigin is a pixel value, based on the size of the Game Object combined with the origin.
             */
            displayOriginX: number;

            /**
             * The vertical display origin of this Game Object.
             * The origin is a normalized value between 0 and 1.
             * The displayOrigin is a pixel value, based on the size of the Game Object combined with the origin.
             */
            displayOriginY: number;

            /**
             * Sets the origin of this Game Object.
             * 
             * The values are given in the range 0 to 1.
             * @param x The horizontal origin value. Default 0.5.
             * @param y The vertical origin value. If not defined it will be set to the value of `x`. Default x.
             */
            setOrigin(x?: number, y?: number): this;

            /**
             * Sets the origin of this Game Object based on the Pivot values in its Frame.
             */
            setOriginFromFrame(): this;

            /**
             * Sets the display origin of this Game Object.
             * The difference between this and setting the origin is that you can use pixel values for setting the display origin.
             * @param x The horizontal display origin value. Default 0.
             * @param y The vertical display origin value. If not defined it will be set to the value of `x`. Default x.
             */
            setDisplayOrigin(x?: number, y?: number): this;

            /**
             * The initial WebGL pipeline of this Game Object.
             * 
             * If you call `resetPipeline` on this Game Object, the pipeline is reset to this default.
             */
            defaultPipeline: Phaser.Renderer.WebGL.WebGLPipeline;

            /**
             * The current WebGL pipeline of this Game Object.
             */
            pipeline: Phaser.Renderer.WebGL.WebGLPipeline;

            /**
             * Does this Game Object have any Post Pipelines set?
             */
            hasPostPipeline: boolean;

            /**
             * The WebGL Post FX Pipelines this Game Object uses for post-render effects.
             * 
             * The pipelines are processed in the order in which they appear in this array.
             * 
             * If you modify this array directly, be sure to set the
             * `hasPostPipeline` property accordingly.
             */
            postPipelines: Phaser.Renderer.WebGL.Pipelines.PostFXPipeline[];

            /**
             * An object to store pipeline specific data in, to be read by the pipelines this Game Object uses.
             */
            pipelineData: object;

            /**
             * Sets the initial WebGL Pipeline of this Game Object.
             * 
             * This should only be called during the instantiation of the Game Object. After that, use `setPipeline`.
             * @param pipeline Either the string-based name of the pipeline, or a pipeline instance to set.
             */
            initPipeline(pipeline: string | Phaser.Renderer.WebGL.WebGLPipeline): boolean;

            /**
             * Sets the main WebGL Pipeline of this Game Object.
             * 
             * Also sets the `pipelineData` property, if the parameter is given.
             * 
             * Both the pipeline and post pipelines share the same pipeline data object.
             * @param pipeline Either the string-based name of the pipeline, or a pipeline instance to set.
             * @param pipelineData Optional pipeline data object that is _deep copied_ into the `pipelineData` property of this Game Object.
             * @param copyData Should the pipeline data object be _deep copied_ into the `pipelineData` property of this Game Object? If `false` it will be set by reference instead. Default true.
             */
            setPipeline(pipeline: string | Phaser.Renderer.WebGL.WebGLPipeline, pipelineData?: object, copyData?: boolean): this;

            /**
             * Sets one, or more, Post Pipelines on this Game Object.
             * 
             * Post Pipelines are invoked after this Game Object has rendered to its target and
             * are commonly used for post-fx.
             * 
             * The post pipelines are appended to the `postPipelines` array belonging to this
             * Game Object. When the renderer processes this Game Object, it iterates through the post
             * pipelines in the order in which they appear in the array. If you are stacking together
             * multiple effects, be aware that the order is important.
             * 
             * If you call this method multiple times, the new pipelines will be appended to any existing
             * post pipelines already set. Use the `resetPostPipeline` method to clear them first, if required.
             * 
             * You can optionally also sets the `pipelineData` property, if the parameter is given.
             * 
             * Both the pipeline and post pipelines share the pipeline data object together.
             * @param pipelines Either the string-based name of the pipeline, or a pipeline instance, or class, or an array of them.
             * @param pipelineData Optional pipeline data object that is _deep copied_ into the `pipelineData` property of this Game Object.
             * @param copyData Should the pipeline data object be _deep copied_ into the `pipelineData` property of this Game Object? If `false` it will be set by reference instead. Default true.
             */
            setPostPipeline(pipelines: string | string[] | Function | Function[] | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline[], pipelineData?: object, copyData?: boolean): this;

            /**
             * Adds an entry to the `pipelineData` object belonging to this Game Object.
             * 
             * If the 'key' already exists, its value is updated. If it doesn't exist, it is created.
             * 
             * If `value` is undefined, and `key` exists, `key` is removed from the data object.
             * 
             * Both the pipeline and post pipelines share the pipeline data object together.
             * @param key The key of the pipeline data to set, update, or delete.
             * @param value The value to be set with the key. If `undefined` then `key` will be deleted from the object.
             */
            setPipelineData(key: string, value?: any): this;

            /**
             * Gets a Post Pipeline instance from this Game Object, based on the given name, and returns it.
             * @param pipeline The string-based name of the pipeline, or a pipeline class.
             */
            getPostPipeline(pipeline: string | Function | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline): Phaser.Renderer.WebGL.Pipelines.PostFXPipeline | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline[];

            /**
             * Resets the WebGL Pipeline of this Game Object back to the default it was created with.
             * @param resetPostPipelines Reset all of the post pipelines? Default false.
             * @param resetData Reset the `pipelineData` object to being an empty object? Default false.
             */
            resetPipeline(resetPostPipelines?: boolean, resetData?: boolean): boolean;

            /**
             * Resets the WebGL Post Pipelines of this Game Object. It does this by calling
             * the `destroy` method on each post pipeline and then clearing the local array.
             * @param resetData Reset the `pipelineData` object to being an empty object? Default false.
             */
            resetPostPipeline(resetData?: boolean): void;

            /**
             * Removes a type of Post Pipeline instances from this Game Object, based on the given name, and destroys them.
             * 
             * If you wish to remove all Post Pipelines use the `resetPostPipeline` method instead.
             * @param pipeline The string-based name of the pipeline, or a pipeline class.
             */
            removePostPipeline(pipeline: string | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline): this;

            /**
             * Gets the name of the WebGL Pipeline this Game Object is currently using.
             */
            getPipelineName(): string;

            /**
             * The horizontal scroll factor of this Game Object.
             * 
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             * 
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             * 
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             * 
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             */
            scrollFactorX: number;

            /**
             * The vertical scroll factor of this Game Object.
             * 
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             * 
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             * 
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             * 
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             */
            scrollFactorY: number;

            /**
             * Sets the scroll factor of this Game Object.
             * 
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             * 
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             * 
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             * 
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             * @param x The horizontal scroll factor of this Game Object.
             * @param y The vertical scroll factor of this Game Object. If not set it will use the `x` value. Default x.
             */
            setScrollFactor(x: number, y?: number): this;

            /**
             * The Texture this Game Object is using to render with.
             */
            texture: Phaser.Textures.Texture | Phaser.Textures.CanvasTexture;

            /**
             * The Texture Frame this Game Object is using to render with.
             */
            frame: Phaser.Textures.Frame;

            /**
             * Sets the texture and frame this Game Object will use to render with.
             * 
             * Textures are referenced by their string-based keys, as stored in the Texture Manager.
             * @param key The key of the texture to be used, as stored in the Texture Manager, or a Texture instance.
             * @param frame The name or index of the frame within the Texture.
             */
            setTexture(key: string | Phaser.Textures.Texture, frame?: string | number): this;

            /**
             * Sets the frame this Game Object will use to render with.
             * 
             * The Frame has to belong to the current Texture being used.
             * 
             * It can be either a string or an index.
             * 
             * Calling `setFrame` will modify the `width` and `height` properties of your Game Object.
             * It will also change the `origin` if the Frame has a custom pivot point, as exported from packages like Texture Packer.
             * @param frame The name or index of the frame within the Texture.
             * @param updateSize Should this call adjust the size of the Game Object? Default true.
             * @param updateOrigin Should this call adjust the origin of the Game Object? Default true.
             */
            setFrame(frame: string | number, updateSize?: boolean, updateOrigin?: boolean): this;

            /**
             * The tint value being applied to the top-left vertice of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             * The value should be set as a hex number, i.e. 0xff0000 for red, or 0xff00ff for purple.
             */
            tintTopLeft: number;

            /**
             * The tint value being applied to the top-right vertice of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             * The value should be set as a hex number, i.e. 0xff0000 for red, or 0xff00ff for purple.
             */
            tintTopRight: number;

            /**
             * The tint value being applied to the bottom-left vertice of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             * The value should be set as a hex number, i.e. 0xff0000 for red, or 0xff00ff for purple.
             */
            tintBottomLeft: number;

            /**
             * The tint value being applied to the bottom-right vertice of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             * The value should be set as a hex number, i.e. 0xff0000 for red, or 0xff00ff for purple.
             */
            tintBottomRight: number;

            /**
             * The tint fill mode.
             * 
             * `false` = An additive tint (the default), where vertices colors are blended with the texture.
             * `true` = A fill tint, where the vertices colors replace the texture, but respects texture alpha.
             */
            tintFill: boolean;

            /**
             * Clears all tint values associated with this Game Object.
             * 
             * Immediately sets the color values back to 0xffffff and the tint type to 'additive',
             * which results in no visible change to the texture.
             */
            clearTint(): this;

            /**
             * Sets an additive tint on this Game Object.
             * 
             * The tint works by taking the pixel color values from the Game Objects texture, and then
             * multiplying it by the color value of the tint. You can provide either one color value,
             * in which case the whole Game Object will be tinted in that color. Or you can provide a color
             * per corner. The colors are blended together across the extent of the Game Object.
             * 
             * To modify the tint color once set, either call this method again with new values or use the
             * `tint` property to set all colors at once. Or, use the properties `tintTopLeft`, `tintTopRight,
             * `tintBottomLeft` and `tintBottomRight` to set the corner color values independently.
             * 
             * To remove a tint call `clearTint`.
             * 
             * To swap this from being an additive tint to a fill based tint set the property `tintFill` to `true`.
             * @param topLeft The tint being applied to the top-left of the Game Object. If no other values are given this value is applied evenly, tinting the whole Game Object. Default 0xffffff.
             * @param topRight The tint being applied to the top-right of the Game Object.
             * @param bottomLeft The tint being applied to the bottom-left of the Game Object.
             * @param bottomRight The tint being applied to the bottom-right of the Game Object.
             */
            setTint(topLeft?: number, topRight?: number, bottomLeft?: number, bottomRight?: number): this;

            /**
             * Sets a fill-based tint on this Game Object.
             * 
             * Unlike an additive tint, a fill-tint literally replaces the pixel colors from the texture
             * with those in the tint. You can use this for effects such as making a player flash 'white'
             * if hit by something. You can provide either one color value, in which case the whole
             * Game Object will be rendered in that color. Or you can provide a color per corner. The colors
             * are blended together across the extent of the Game Object.
             * 
             * To modify the tint color once set, either call this method again with new values or use the
             * `tint` property to set all colors at once. Or, use the properties `tintTopLeft`, `tintTopRight,
             * `tintBottomLeft` and `tintBottomRight` to set the corner color values independently.
             * 
             * To remove a tint call `clearTint`.
             * 
             * To swap this from being a fill-tint to an additive tint set the property `tintFill` to `false`.
             * @param topLeft The tint being applied to the top-left of the Game Object. If not other values are given this value is applied evenly, tinting the whole Game Object. Default 0xffffff.
             * @param topRight The tint being applied to the top-right of the Game Object.
             * @param bottomLeft The tint being applied to the bottom-left of the Game Object.
             * @param bottomRight The tint being applied to the bottom-right of the Game Object.
             */
            setTintFill(topLeft?: number, topRight?: number, bottomLeft?: number, bottomRight?: number): this;

            /**
             * The tint value being applied to the whole of the Game Object.
             * This property is a setter-only. Use the properties `tintTopLeft` etc to read the current tint value.
             */
            tint: number;

            /**
             * Does this Game Object have a tint applied?
             * 
             * It checks to see if the 4 tint properties are set to the value 0xffffff
             * and that the `tintFill` property is `false`. This indicates that a Game Object isn't tinted.
             */
            readonly isTinted: boolean;

            /**
             * The x position of this Game Object.
             */
            x: number;

            /**
             * The y position of this Game Object.
             */
            y: number;

            /**
             * The z position of this Game Object.
             * 
             * Note: The z position does not control the rendering order of 2D Game Objects. Use
             * {@link Phaser.GameObjects.Components.Depth#depth} instead.
             */
            z: number;

            /**
             * The w position of this Game Object.
             */
            w: number;

            /**
             * This is a special setter that allows you to set both the horizontal and vertical scale of this Game Object
             * to the same value, at the same time. When reading this value the result returned is `(scaleX + scaleY) / 2`.
             * 
             * Use of this property implies you wish the horizontal and vertical scales to be equal to each other. If this
             * isn't the case, use the `scaleX` or `scaleY` properties instead.
             */
            scale: number;

            /**
             * The horizontal scale of this Game Object.
             */
            scaleX: number;

            /**
             * The vertical scale of this Game Object.
             */
            scaleY: number;

            /**
             * The angle of this Game Object as expressed in degrees.
             * 
             * Phaser uses a right-hand clockwise rotation system, where 0 is right, 90 is down, 180/-180 is left
             * and -90 is up.
             * 
             * If you prefer to work in radians, see the `rotation` property instead.
             */
            angle: number;

            /**
             * The angle of this Game Object in radians.
             * 
             * Phaser uses a right-hand clockwise rotation system, where 0 is right, PI/2 is down, +-PI is left
             * and -PI/2 is up.
             * 
             * If you prefer to work in degrees, see the `angle` property instead.
             */
            rotation: number;

            /**
             * Sets the position of this Game Object.
             * @param x The x position of this Game Object. Default 0.
             * @param y The y position of this Game Object. If not set it will use the `x` value. Default x.
             * @param z The z position of this Game Object. Default 0.
             * @param w The w position of this Game Object. Default 0.
             */
            setPosition(x?: number, y?: number, z?: number, w?: number): this;

            /**
             * Copies an object's coordinates to this Game Object's position.
             * @param source An object with numeric 'x', 'y', 'z', or 'w' properties. Undefined values are not copied.
             */
            copyPosition(source: Phaser.Types.Math.Vector2Like | Phaser.Types.Math.Vector3Like | Phaser.Types.Math.Vector4Like): this;

            /**
             * Sets the position of this Game Object to be a random position within the confines of
             * the given area.
             * 
             * If no area is specified a random position between 0 x 0 and the game width x height is used instead.
             * 
             * The position does not factor in the size of this Game Object, meaning that only the origin is
             * guaranteed to be within the area.
             * @param x The x position of the top-left of the random area. Default 0.
             * @param y The y position of the top-left of the random area. Default 0.
             * @param width The width of the random area.
             * @param height The height of the random area.
             */
            setRandomPosition(x?: number, y?: number, width?: number, height?: number): this;

            /**
             * Sets the rotation of this Game Object.
             * @param radians The rotation of this Game Object, in radians. Default 0.
             */
            setRotation(radians?: number): this;

            /**
             * Sets the angle of this Game Object.
             * @param degrees The rotation of this Game Object, in degrees. Default 0.
             */
            setAngle(degrees?: number): this;

            /**
             * Sets the scale of this Game Object.
             * @param x The horizontal scale of this Game Object.
             * @param y The vertical scale of this Game Object. If not set it will use the `x` value. Default x.
             */
            setScale(x: number, y?: number): this;

            /**
             * Sets the x position of this Game Object.
             * @param value The x position of this Game Object. Default 0.
             */
            setX(value?: number): this;

            /**
             * Sets the y position of this Game Object.
             * @param value The y position of this Game Object. Default 0.
             */
            setY(value?: number): this;

            /**
             * Sets the z position of this Game Object.
             * 
             * Note: The z position does not control the rendering order of 2D Game Objects. Use
             * {@link Phaser.GameObjects.Components.Depth#setDepth} instead.
             * @param value The z position of this Game Object. Default 0.
             */
            setZ(value?: number): this;

            /**
             * Sets the w position of this Game Object.
             * @param value The w position of this Game Object. Default 0.
             */
            setW(value?: number): this;

            /**
             * Gets the local transform matrix for this Game Object.
             * @param tempMatrix The matrix to populate with the values from this Game Object.
             */
            getLocalTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;

            /**
             * Gets the world transform matrix for this Game Object, factoring in any parent Containers.
             * @param tempMatrix The matrix to populate with the values from this Game Object.
             * @param parentMatrix A temporary matrix to hold parent values during the calculations.
             */
            getWorldTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix, parentMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;

            /**
             * Takes the given `x` and `y` coordinates and converts them into local space for this
             * Game Object, taking into account parent and local transforms, and the Display Origin.
             * 
             * The returned Vector2 contains the translated point in its properties.
             * 
             * A Camera needs to be provided in order to handle modified scroll factors. If no
             * camera is specified, it will use the `main` camera from the Scene to which this
             * Game Object belongs.
             * @param x The x position to translate.
             * @param y The y position to translate.
             * @param point A Vector2, or point-like object, to store the results in.
             * @param camera The Camera which is being tested against. If not given will use the Scene default camera.
             */
            getLocalPoint(x: number, y: number, point?: Phaser.Math.Vector2, camera?: Phaser.Cameras.Scene2D.Camera): Phaser.Math.Vector2;

            /**
             * Gets the sum total rotation of all of this Game Objects parent Containers.
             * 
             * The returned value is in radians and will be zero if this Game Object has no parent container.
             */
            getParentRotation(): number;

            /**
             * The visible state of the Game Object.
             * 
             * An invisible Game Object will skip rendering, but will still process update logic.
             */
            visible: boolean;

            /**
             * Sets the visibility of this Game Object.
             * 
             * An invisible Game Object will skip rendering, but will still process update logic.
             * @param value The visible state of the Game Object.
             */
            setVisible(value: boolean): this;

        }

        namespace RetroFont {
            /**
             * Text Set 1 =  !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~
             */
            var TEXT_SET1: string;

            /**
             * Text Set 2 =  !"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ
             */
            var TEXT_SET2: string;

            /**
             * Text Set 3 = ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789
             */
            var TEXT_SET3: string;

            /**
             * Text Set 4 = ABCDEFGHIJKLMNOPQRSTUVWXYZ 0123456789
             */
            var TEXT_SET4: string;

            /**
             * Text Set 5 = ABCDEFGHIJKLMNOPQRSTUVWXYZ.,/() '!?-*:0123456789
             */
            var TEXT_SET5: string;

            /**
             * Text Set 6 = ABCDEFGHIJKLMNOPQRSTUVWXYZ!?:;0123456789"(),-.'
             */
            var TEXT_SET6: string;

            /**
             * Text Set 7 = AGMSY+:4BHNTZ!;5CIOU.?06DJPV,(17EKQW")28FLRX-'39
             */
            var TEXT_SET7: string;

            /**
             * Text Set 8 = 0123456789 .ABCDEFGHIJKLMNOPQRSTUVWXYZ
             */
            var TEXT_SET8: string;

            /**
             * Text Set 9 = ABCDEFGHIJKLMNOPQRSTUVWXYZ()-0123456789.:,'"?!
             */
            var TEXT_SET9: string;

            /**
             * Text Set 10 = ABCDEFGHIJKLMNOPQRSTUVWXYZ
             */
            var TEXT_SET10: string;

            /**
             * Text Set 11 = ABCDEFGHIJKLMNOPQRSTUVWXYZ.,"-+!?()':;0123456789
             */
            var TEXT_SET11: string;

            /**
             * Parses a Retro Font configuration object so you can pass it to the BitmapText constructor
             * and create a BitmapText object using a fixed-width retro font.
             * @param scene A reference to the Phaser Scene.
             * @param config The font configuration object.
             */
            function Parse(scene: Phaser.Scene, config: Phaser.Types.GameObjects.BitmapText.RetroFontConfig): object;

        }

        /**
         * BitmapText objects work by taking a texture file and an XML or JSON file that describes the font structure.
         * 
         * During rendering for each letter of the text is rendered to the display, proportionally spaced out and aligned to
         * match the font structure.
         * 
         * BitmapText objects are less flexible than Text objects, in that they have less features such as shadows, fills and the ability
         * to use Web Fonts, however you trade this flexibility for rendering speed. You can also create visually compelling BitmapTexts by
         * processing the font texture in an image editor, applying fills and any other effects required.
         * 
         * To create multi-line text insert \r, \n or \r\n escape codes into the text string.
         * 
         * To create a BitmapText data files you need a 3rd party app such as:
         * 
         * BMFont (Windows, free): {@link http://www.angelcode.com/products/bmfont/|http://www.angelcode.com/products/bmfont/}
         * Glyph Designer (OS X, commercial): {@link http://www.71squared.com/en/glyphdesigner|http://www.71squared.com/en/glyphdesigner}
         * Littera (Web-based, free): {@link http://kvazars.com/littera/|http://kvazars.com/littera/}
         * 
         * For most use cases it is recommended to use XML. If you wish to use JSON, the formatting should be equal to the result of
         * converting a valid XML file through the popular X2JS library. An online tool for conversion can be found here: {@link http://codebeautify.org/xmltojson|http://codebeautify.org/xmltojson}
         */
        class BitmapText extends Phaser.GameObjects.GameObject implements Phaser.GameObjects.Components.Alpha, Phaser.GameObjects.Components.BlendMode, Phaser.GameObjects.Components.Depth, Phaser.GameObjects.Components.Mask, Phaser.GameObjects.Components.Origin, Phaser.GameObjects.Components.Pipeline, Phaser.GameObjects.Components.ScrollFactor, Phaser.GameObjects.Components.Texture, Phaser.GameObjects.Components.Tint, Phaser.GameObjects.Components.Transform, Phaser.GameObjects.Components.Visible {
            /**
             * 
             * @param scene The Scene to which this Game Object belongs. It can only belong to one Scene at any given time.
             * @param x The x coordinate of this Game Object in world space.
             * @param y The y coordinate of this Game Object in world space.
             * @param font The key of the font to use from the Bitmap Font cache.
             * @param text The string, or array of strings, to be set as the content of this Bitmap Text.
             * @param size The font size of this Bitmap Text.
             * @param align The alignment of the text in a multi-line BitmapText object. Default 0.
             */
            constructor(scene: Phaser.Scene, x: number, y: number, font: string, text?: string | string[], size?: number, align?: number);

            /**
             * The key of the Bitmap Font used by this Bitmap Text.
             * To change the font after creation please use `setFont`.
             */
            readonly font: string;

            /**
             * The data of the Bitmap Font used by this Bitmap Text.
             */
            readonly fontData: Phaser.Types.GameObjects.BitmapText.BitmapFontData;

            /**
             * The character code used to detect for word wrapping.
             * Defaults to 32 (a space character).
             */
            wordWrapCharCode: number;

            /**
             * The horizontal offset of the drop shadow.
             * 
             * You can set this directly, or use `Phaser.GameObjects.BitmapText#setDropShadow`.
             */
            dropShadowX: number;

            /**
             * The vertical offset of the drop shadow.
             * 
             * You can set this directly, or use `Phaser.GameObjects.BitmapText#setDropShadow`.
             */
            dropShadowY: number;

            /**
             * The color of the drop shadow.
             * 
             * You can set this directly, or use `Phaser.GameObjects.BitmapText#setDropShadow`.
             */
            dropShadowColor: number;

            /**
             * The alpha value of the drop shadow.
             * 
             * You can set this directly, or use `Phaser.GameObjects.BitmapText#setDropShadow`.
             */
            dropShadowAlpha: number;

            /**
             * Indicates whether the font texture is from an atlas or not.
             */
            readonly fromAtlas: boolean;

            /**
             * Set the lines of text in this BitmapText to be left-aligned.
             * This only has any effect if this BitmapText contains more than one line of text.
             */
            setLeftAlign(): this;

            /**
             * Set the lines of text in this BitmapText to be center-aligned.
             * This only has any effect if this BitmapText contains more than one line of text.
             */
            setCenterAlign(): this;

            /**
             * Set the lines of text in this BitmapText to be right-aligned.
             * This only has any effect if this BitmapText contains more than one line of text.
             */
            setRightAlign(): this;

            /**
             * Set the font size of this Bitmap Text.
             * @param size The font size to set.
             */
            setFontSize(size: number): this;

            /**
             * Sets the letter spacing between each character of this Bitmap Text.
             * Can be a positive value to increase the space, or negative to reduce it.
             * Spacing is applied after the kerning values have been set.
             * @param spacing The amount of horizontal space to add between each character. Default 0.
             */
            setLetterSpacing(spacing?: number): this;

            /**
             * Set the textual content of this BitmapText.
             * 
             * An array of strings will be converted into multi-line text. Use the align methods to change multi-line alignment.
             * @param value The string, or array of strings, to be set as the content of this BitmapText.
             */
            setText(value: string | string[]): this;

            /**
             * Sets a drop shadow effect on this Bitmap Text.
             * 
             * This is a WebGL only feature and only works with Static Bitmap Text, not Dynamic.
             * 
             * You can set the vertical and horizontal offset of the shadow, as well as the color and alpha.
             * 
             * Once a shadow has been enabled you can modify the `dropShadowX` and `dropShadowY` properties of this
             * Bitmap Text directly to adjust the position of the shadow in real-time.
             * 
             * If you wish to clear the shadow, call this method with no parameters specified.
             * @param x The horizontal offset of the drop shadow. Default 0.
             * @param y The vertical offset of the drop shadow. Default 0.
             * @param color The color of the drop shadow, given as a hex value, i.e. `0x000000` for black. Default 0x000000.
             * @param alpha The alpha of the drop shadow, given as a float between 0 and 1. This is combined with the Bitmap Text alpha as well. Default 0.5.
             */
            setDropShadow(x?: number, y?: number, color?: number, alpha?: number): this;

            /**
             * Sets a tint on a range of characters in this Bitmap Text, starting from the `start` parameter index
             * and running for `length` quantity of characters.
             * 
             * The `start` parameter can be negative. In this case, it starts at the end of the text and counts
             * backwards `start` places.
             * 
             * You can also pass in -1 as the `length` and it will tint all characters from `start`
             * up until the end of the string.
             * Remember that spaces and punctuation count as characters.
             * 
             * This is a WebGL only feature and only works with Static Bitmap Text, not Dynamic.
             * 
             * The tint works by taking the pixel color values from the Bitmap Text texture, and then
             * multiplying it by the color value of the tint. You can provide either one color value,
             * in which case the whole character will be tinted in that color. Or you can provide a color
             * per corner. The colors are blended together across the extent of the character range.
             * 
             * To swap this from being an additive tint to a fill based tint, set the `tintFill` parameter to `true`.
             * 
             * To modify the tint color once set, call this method again with new color values.
             * 
             * Using `setWordTint` can override tints set by this function, and vice versa.
             * 
             * To remove a tint call this method with just the `start`, and optionally, the `length` parameters defined.
             * @param start The starting character to begin the tint at. If negative, it counts back from the end of the text. Default 0.
             * @param length The number of characters to tint. Remember that spaces count as a character too. Pass -1 to tint all characters from `start` onwards. Default 1.
             * @param tintFill Use a fill-based tint (true), or an additive tint (false) Default false.
             * @param topLeft The tint being applied to the top-left of the character. If not other values are given this value is applied evenly, tinting the whole character. Default 0xffffff.
             * @param topRight The tint being applied to the top-right of the character.
             * @param bottomLeft The tint being applied to the bottom-left of the character.
             * @param bottomRight The tint being applied to the bottom-right of the character.
             */
            setCharacterTint(start?: number, length?: number, tintFill?: boolean, topLeft?: number, topRight?: number, bottomLeft?: number, bottomRight?: number): this;

            /**
             * Sets a tint on a matching word within this Bitmap Text.
             * 
             * The `word` parameter can be either a string or a number.
             * 
             * If a string, it will run a string comparison against the text contents, and if matching,
             * it will tint the whole word.
             * 
             * If a number, if till that word, based on its offset within the text contents.
             * 
             * The `count` parameter controls how many words are replaced. Pass in -1 to replace them all.
             * 
             * This parameter is ignored if you pass a number as the `word` to be searched for.
             * 
             * This is a WebGL only feature and only works with Static Bitmap Text, not Dynamic.
             * 
             * The tint works by taking the pixel color values from the Bitmap Text texture, and then
             * multiplying it by the color value of the tint. You can provide either one color value,
             * in which case the whole character will be tinted in that color. Or you can provide a color
             * per corner. The colors are blended together across the extent of the character range.
             * 
             * To swap this from being an additive tint to a fill based tint, set the `tintFill` parameter to `true`.
             * 
             * To modify the tint color once set, call this method again with new color values.
             * 
             * Using `setCharacterTint` can override tints set by this function, and vice versa.
             * @param word The word to search for. Either a string, or an index of the word in the words array.
             * @param count The number of matching words to tint. Pass -1 to tint all matching words. Default 1.
             * @param tintFill Use a fill-based tint (true), or an additive tint (false) Default false.
             * @param topLeft The tint being applied to the top-left of the word. If not other values are given this value is applied evenly, tinting the whole word. Default 0xffffff.
             * @param topRight The tint being applied to the top-right of the word.
             * @param bottomLeft The tint being applied to the bottom-left of the word.
             * @param bottomRight The tint being applied to the bottom-right of the word.
             */
            setWordTint(word: string | number, count?: number, tintFill?: boolean, topLeft?: number, topRight?: number, bottomLeft?: number, bottomRight?: number): this;

            /**
             * Calculate the bounds of this Bitmap Text.
             * 
             * An object is returned that contains the position, width and height of the Bitmap Text in local and global
             * contexts.
             * 
             * Local size is based on just the font size and a [0, 0] position.
             * 
             * Global size takes into account the Game Object's scale, world position and display origin.
             * 
             * Also in the object is data regarding the length of each line, should this be a multi-line BitmapText.
             * @param round Whether to round the results up to the nearest integer. Default false.
             */
            getTextBounds(round?: boolean): Phaser.Types.GameObjects.BitmapText.BitmapTextSize;

            /**
             * Gets the character located at the given x/y coordinate within this Bitmap Text.
             * 
             * The coordinates you pass in are translated into the local space of the
             * Bitmap Text, however, it is up to you to first translate the input coordinates to world space.
             * 
             * If you wish to use this in combination with an input event, be sure
             * to pass in `Pointer.worldX` and `worldY` so they are in world space.
             * 
             * In some cases, based on kerning, characters can overlap. When this happens,
             * the first character in the word is returned.
             * 
             * Note that this does not work for DynamicBitmapText if you have changed the
             * character positions during render. It will only scan characters in their un-translated state.
             * @param x The x position to check.
             * @param y The y position to check.
             * @param camera The Camera which is being tested against. If not given will use the Scene default camera.
             */
            getCharacterAt(x: number, y: number, camera?: Phaser.Cameras.Scene2D.Camera): Phaser.Types.GameObjects.BitmapText.BitmapTextCharacter;

            /**
             * Updates the Display Origin cached values internally stored on this Game Object.
             * You don't usually call this directly, but it is exposed for edge-cases where you may.
             */
            updateDisplayOrigin(): this;

            /**
             * Changes the font this BitmapText is using to render.
             * 
             * The new texture is loaded and applied to the BitmapText. The existing test, size and alignment are preserved,
             * unless overridden via the arguments.
             * @param font The key of the font to use from the Bitmap Font cache.
             * @param size The font size of this Bitmap Text. If not specified the current size will be used.
             * @param align The alignment of the text in a multi-line BitmapText object. If not specified the current alignment will be used. Default 0.
             */
            setFont(font: string, size?: number, align?: number): this;

            /**
             * Sets the maximum display width of this BitmapText in pixels.
             * 
             * If `BitmapText.text` is longer than `maxWidth` then the lines will be automatically wrapped
             * based on the previous whitespace character found in the line.
             * 
             * If no whitespace was found then no wrapping will take place and consequently the `maxWidth` value will not be honored.
             * 
             * Disable maxWidth by setting the value to 0.
             * 
             * You can set the whitespace character to be searched for by setting the `wordWrapCharCode` parameter or property.
             * @param value The maximum display width of this BitmapText in pixels. Set to zero to disable.
             * @param wordWrapCharCode The character code to check for when word wrapping. Defaults to 32 (the space character).
             */
            setMaxWidth(value: number, wordWrapCharCode?: number): this;

            /**
             * Controls the alignment of each line of text in this BitmapText object.
             * 
             * Only has any effect when this BitmapText contains multiple lines of text, split with carriage-returns.
             * Has no effect with single-lines of text.
             * 
             * See the methods `setLeftAlign`, `setCenterAlign` and `setRightAlign`.
             * 
             * 0 = Left aligned (default)
             * 1 = Middle aligned
             * 2 = Right aligned
             * 
             * The alignment position is based on the longest line of text.
             */
            align: number;

            /**
             * The text that this Bitmap Text object displays.
             * 
             * You can also use the method `setText` if you want a chainable way to change the text content.
             */
            text: string;

            /**
             * The font size of this Bitmap Text.
             * 
             * You can also use the method `setFontSize` if you want a chainable way to change the font size.
             */
            fontSize: number;

            /**
             * Adds / Removes spacing between characters.
             * 
             * Can be a negative or positive number.
             * 
             * You can also use the method `setLetterSpacing` if you want a chainable way to change the letter spacing.
             */
            letterSpacing: number;

            /**
             * The maximum display width of this BitmapText in pixels.
             * 
             * If BitmapText.text is longer than maxWidth then the lines will be automatically wrapped
             * based on the last whitespace character found in the line.
             * 
             * If no whitespace was found then no wrapping will take place and consequently the maxWidth value will not be honored.
             * 
             * Disable maxWidth by setting the value to 0.
             */
            maxWidth: number;

            /**
             * The width of this Bitmap Text.
             */
            readonly width: number;

            /**
             * The height of this bitmap text.
             */
            readonly height: number;

            /**
             * Build a JSON representation of this Bitmap Text.
             */
            toJSON(): Phaser.Types.GameObjects.BitmapText.JSONBitmapText;

            /**
             * Internal destroy handler, called as part of the destroy process.
             */
            protected preDestroy(): void;

            /**
             * Left align the text characters in a multi-line BitmapText object.
             */
            static ALIGN_LEFT: number;

            /**
             * Center align the text characters in a multi-line BitmapText object.
             */
            static ALIGN_CENTER: number;

            /**
             * Right align the text characters in a multi-line BitmapText object.
             */
            static ALIGN_RIGHT: number;

            /**
             * Parse an XML Bitmap Font from an Atlas.
             * 
             * Adds the parsed Bitmap Font data to the cache with the `fontName` key.
             * @param scene The Scene to parse the Bitmap Font for.
             * @param fontName The key of the font to add to the Bitmap Font cache.
             * @param textureKey The key of the BitmapFont's texture.
             * @param frameKey The key of the BitmapFont texture's frame.
             * @param xmlKey The key of the XML data of the font to parse.
             * @param xSpacing The x-axis spacing to add between each letter.
             * @param ySpacing The y-axis spacing to add to the line height.
             */
            static ParseFromAtlas(scene: Phaser.Scene, fontName: string, textureKey: string, frameKey: string, xmlKey: string, xSpacing?: number, ySpacing?: number): boolean;

            /**
             * Parse an XML font to Bitmap Font data for the Bitmap Font cache.
             * @param xml The XML Document to parse the font from.
             * @param frame The texture frame to take into account when creating the uv data.
             * @param xSpacing The x-axis spacing to add between each letter. Default 0.
             * @param ySpacing The y-axis spacing to add to the line height. Default 0.
             */
            static ParseXMLBitmapFont(xml: XMLDocument, frame: Phaser.Textures.Frame, xSpacing?: number, ySpacing?: number): Phaser.Types.GameObjects.BitmapText.BitmapFontData;

            /**
             * Clears all alpha values associated with this Game Object.
             * 
             * Immediately sets the alpha levels back to 1 (fully opaque).
             */
            clearAlpha(): this;

            /**
             * Set the Alpha level of this Game Object. The alpha controls the opacity of the Game Object as it renders.
             * Alpha values are provided as a float between 0, fully transparent, and 1, fully opaque.
             * 
             * If your game is running under WebGL you can optionally specify four different alpha values, each of which
             * correspond to the four corners of the Game Object. Under Canvas only the `topLeft` value given is used.
             * @param topLeft The alpha value used for the top-left of the Game Object. If this is the only value given it's applied across the whole Game Object. Default 1.
             * @param topRight The alpha value used for the top-right of the Game Object. WebGL only.
             * @param bottomLeft The alpha value used for the bottom-left of the Game Object. WebGL only.
             * @param bottomRight The alpha value used for the bottom-right of the Game Object. WebGL only.
             */
            setAlpha(topLeft?: number, topRight?: number, bottomLeft?: number, bottomRight?: number): this;

            /**
             * The alpha value of the Game Object.
             * 
             * This is a global value, impacting the entire Game Object, not just a region of it.
             */
            alpha: number;

            /**
             * The alpha value starting from the top-left of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaTopLeft: number;

            /**
             * The alpha value starting from the top-right of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaTopRight: number;

            /**
             * The alpha value starting from the bottom-left of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaBottomLeft: number;

            /**
             * The alpha value starting from the bottom-right of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaBottomRight: number;

            /**
             * Sets the Blend Mode being used by this Game Object.
             * 
             * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
             * 
             * Under WebGL only the following Blend Modes are available:
             * 
             * * ADD
             * * MULTIPLY
             * * SCREEN
             * * ERASE
             * 
             * Canvas has more available depending on browser support.
             * 
             * You can also create your own custom Blend Modes in WebGL.
             * 
             * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
             * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
             * reasons try to be careful about the construction of your Scene and the frequency of which blend modes
             * are used.
             */
            blendMode: Phaser.BlendModes | string;

            /**
             * Sets the Blend Mode being used by this Game Object.
             * 
             * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
             * 
             * Under WebGL only the following Blend Modes are available:
             * 
             * * ADD
             * * MULTIPLY
             * * SCREEN
             * * ERASE (only works when rendering to a framebuffer, like a Render Texture)
             * 
             * Canvas has more available depending on browser support.
             * 
             * You can also create your own custom Blend Modes in WebGL.
             * 
             * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
             * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
             * reasons try to be careful about the construction of your Scene and the frequency in which blend modes
             * are used.
             * @param value The BlendMode value. Either a string or a CONST.
             */
            setBlendMode(value: string | Phaser.BlendModes): this;

            /**
             * The depth of this Game Object within the Scene.
             * 
             * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
             * of Game Objects, without actually moving their position in the display list.
             * 
             * The default depth is zero. A Game Object with a higher depth
             * value will always render in front of one with a lower value.
             * 
             * Setting the depth will queue a depth sort event within the Scene.
             */
            depth: number;

            /**
             * The depth of this Game Object within the Scene.
             * 
             * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
             * of Game Objects, without actually moving their position in the display list.
             * 
             * The default depth is zero. A Game Object with a higher depth
             * value will always render in front of one with a lower value.
             * 
             * Setting the depth will queue a depth sort event within the Scene.
             * @param value The depth of this Game Object.
             */
            setDepth(value: number): this;

            /**
             * The Mask this Game Object is using during render.
             */
            mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask;

            /**
             * Sets the mask that this Game Object will use to render with.
             * 
             * The mask must have been previously created and can be either a GeometryMask or a BitmapMask.
             * Note: Bitmap Masks only work on WebGL. Geometry Masks work on both WebGL and Canvas.
             * 
             * If a mask is already set on this Game Object it will be immediately replaced.
             * 
             * Masks are positioned in global space and are not relative to the Game Object to which they
             * are applied. The reason for this is that multiple Game Objects can all share the same mask.
             * 
             * Masks have no impact on physics or input detection. They are purely a rendering component
             * that allows you to limit what is visible during the render pass.
             * @param mask The mask this Game Object will use when rendering.
             */
            setMask(mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask): this;

            /**
             * Clears the mask that this Game Object was using.
             * @param destroyMask Destroy the mask before clearing it? Default false.
             */
            clearMask(destroyMask?: boolean): this;

            /**
             * Creates and returns a Bitmap Mask. This mask can be used by any Game Object,
             * including this one.
             * 
             * Note: Bitmap Masks only work on WebGL. Geometry Masks work on both WebGL and Canvas.
             * 
             * To create the mask you need to pass in a reference to a renderable Game Object.
             * A renderable Game Object is one that uses a texture to render with, such as an
             * Image, Sprite, Render Texture or BitmapText.
             * 
             * If you do not provide a renderable object, and this Game Object has a texture,
             * it will use itself as the object. This means you can call this method to create
             * a Bitmap Mask from any renderable Game Object.
             * @param renderable A renderable Game Object that uses a texture, such as a Sprite.
             */
            createBitmapMask(renderable?: Phaser.GameObjects.GameObject): Phaser.Display.Masks.BitmapMask;

            /**
             * Creates and returns a Geometry Mask. This mask can be used by any Game Object,
             * including this one.
             * 
             * To create the mask you need to pass in a reference to a Graphics Game Object.
             * 
             * If you do not provide a graphics object, and this Game Object is an instance
             * of a Graphics object, then it will use itself to create the mask.
             * 
             * This means you can call this method to create a Geometry Mask from any Graphics Game Object.
             * @param graphics A Graphics Game Object. The geometry within it will be used as the mask.
             */
            createGeometryMask(graphics?: Phaser.GameObjects.Graphics): Phaser.Display.Masks.GeometryMask;

            /**
             * The horizontal origin of this Game Object.
             * The origin maps the relationship between the size and position of the Game Object.
             * The default value is 0.5, meaning all Game Objects are positioned based on their center.
             * Setting the value to 0 means the position now relates to the left of the Game Object.
             */
            originX: number;

            /**
             * The vertical origin of this Game Object.
             * The origin maps the relationship between the size and position of the Game Object.
             * The default value is 0.5, meaning all Game Objects are positioned based on their center.
             * Setting the value to 0 means the position now relates to the top of the Game Object.
             */
            originY: number;

            /**
             * The horizontal display origin of this Game Object.
             * The origin is a normalized value between 0 and 1.
             * The displayOrigin is a pixel value, based on the size of the Game Object combined with the origin.
             */
            displayOriginX: number;

            /**
             * The vertical display origin of this Game Object.
             * The origin is a normalized value between 0 and 1.
             * The displayOrigin is a pixel value, based on the size of the Game Object combined with the origin.
             */
            displayOriginY: number;

            /**
             * Sets the origin of this Game Object.
             * 
             * The values are given in the range 0 to 1.
             * @param x The horizontal origin value. Default 0.5.
             * @param y The vertical origin value. If not defined it will be set to the value of `x`. Default x.
             */
            setOrigin(x?: number, y?: number): this;

            /**
             * Sets the origin of this Game Object based on the Pivot values in its Frame.
             */
            setOriginFromFrame(): this;

            /**
             * Sets the display origin of this Game Object.
             * The difference between this and setting the origin is that you can use pixel values for setting the display origin.
             * @param x The horizontal display origin value. Default 0.
             * @param y The vertical display origin value. If not defined it will be set to the value of `x`. Default x.
             */
            setDisplayOrigin(x?: number, y?: number): this;

            /**
             * The initial WebGL pipeline of this Game Object.
             * 
             * If you call `resetPipeline` on this Game Object, the pipeline is reset to this default.
             */
            defaultPipeline: Phaser.Renderer.WebGL.WebGLPipeline;

            /**
             * The current WebGL pipeline of this Game Object.
             */
            pipeline: Phaser.Renderer.WebGL.WebGLPipeline;

            /**
             * Does this Game Object have any Post Pipelines set?
             */
            hasPostPipeline: boolean;

            /**
             * The WebGL Post FX Pipelines this Game Object uses for post-render effects.
             * 
             * The pipelines are processed in the order in which they appear in this array.
             * 
             * If you modify this array directly, be sure to set the
             * `hasPostPipeline` property accordingly.
             */
            postPipelines: Phaser.Renderer.WebGL.Pipelines.PostFXPipeline[];

            /**
             * An object to store pipeline specific data in, to be read by the pipelines this Game Object uses.
             */
            pipelineData: object;

            /**
             * Sets the initial WebGL Pipeline of this Game Object.
             * 
             * This should only be called during the instantiation of the Game Object. After that, use `setPipeline`.
             * @param pipeline Either the string-based name of the pipeline, or a pipeline instance to set.
             */
            initPipeline(pipeline: string | Phaser.Renderer.WebGL.WebGLPipeline): boolean;

            /**
             * Sets the main WebGL Pipeline of this Game Object.
             * 
             * Also sets the `pipelineData` property, if the parameter is given.
             * 
             * Both the pipeline and post pipelines share the same pipeline data object.
             * @param pipeline Either the string-based name of the pipeline, or a pipeline instance to set.
             * @param pipelineData Optional pipeline data object that is _deep copied_ into the `pipelineData` property of this Game Object.
             * @param copyData Should the pipeline data object be _deep copied_ into the `pipelineData` property of this Game Object? If `false` it will be set by reference instead. Default true.
             */
            setPipeline(pipeline: string | Phaser.Renderer.WebGL.WebGLPipeline, pipelineData?: object, copyData?: boolean): this;

            /**
             * Sets one, or more, Post Pipelines on this Game Object.
             * 
             * Post Pipelines are invoked after this Game Object has rendered to its target and
             * are commonly used for post-fx.
             * 
             * The post pipelines are appended to the `postPipelines` array belonging to this
             * Game Object. When the renderer processes this Game Object, it iterates through the post
             * pipelines in the order in which they appear in the array. If you are stacking together
             * multiple effects, be aware that the order is important.
             * 
             * If you call this method multiple times, the new pipelines will be appended to any existing
             * post pipelines already set. Use the `resetPostPipeline` method to clear them first, if required.
             * 
             * You can optionally also sets the `pipelineData` property, if the parameter is given.
             * 
             * Both the pipeline and post pipelines share the pipeline data object together.
             * @param pipelines Either the string-based name of the pipeline, or a pipeline instance, or class, or an array of them.
             * @param pipelineData Optional pipeline data object that is _deep copied_ into the `pipelineData` property of this Game Object.
             * @param copyData Should the pipeline data object be _deep copied_ into the `pipelineData` property of this Game Object? If `false` it will be set by reference instead. Default true.
             */
            setPostPipeline(pipelines: string | string[] | Function | Function[] | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline[], pipelineData?: object, copyData?: boolean): this;

            /**
             * Adds an entry to the `pipelineData` object belonging to this Game Object.
             * 
             * If the 'key' already exists, its value is updated. If it doesn't exist, it is created.
             * 
             * If `value` is undefined, and `key` exists, `key` is removed from the data object.
             * 
             * Both the pipeline and post pipelines share the pipeline data object together.
             * @param key The key of the pipeline data to set, update, or delete.
             * @param value The value to be set with the key. If `undefined` then `key` will be deleted from the object.
             */
            setPipelineData(key: string, value?: any): this;

            /**
             * Gets a Post Pipeline instance from this Game Object, based on the given name, and returns it.
             * @param pipeline The string-based name of the pipeline, or a pipeline class.
             */
            getPostPipeline(pipeline: string | Function | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline): Phaser.Renderer.WebGL.Pipelines.PostFXPipeline | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline[];

            /**
             * Resets the WebGL Pipeline of this Game Object back to the default it was created with.
             * @param resetPostPipelines Reset all of the post pipelines? Default false.
             * @param resetData Reset the `pipelineData` object to being an empty object? Default false.
             */
            resetPipeline(resetPostPipelines?: boolean, resetData?: boolean): boolean;

            /**
             * Resets the WebGL Post Pipelines of this Game Object. It does this by calling
             * the `destroy` method on each post pipeline and then clearing the local array.
             * @param resetData Reset the `pipelineData` object to being an empty object? Default false.
             */
            resetPostPipeline(resetData?: boolean): void;

            /**
             * Removes a type of Post Pipeline instances from this Game Object, based on the given name, and destroys them.
             * 
             * If you wish to remove all Post Pipelines use the `resetPostPipeline` method instead.
             * @param pipeline The string-based name of the pipeline, or a pipeline class.
             */
            removePostPipeline(pipeline: string | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline): this;

            /**
             * Gets the name of the WebGL Pipeline this Game Object is currently using.
             */
            getPipelineName(): string;

            /**
             * The horizontal scroll factor of this Game Object.
             * 
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             * 
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             * 
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             * 
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             */
            scrollFactorX: number;

            /**
             * The vertical scroll factor of this Game Object.
             * 
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             * 
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             * 
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             * 
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             */
            scrollFactorY: number;

            /**
             * Sets the scroll factor of this Game Object.
             * 
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             * 
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             * 
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             * 
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             * @param x The horizontal scroll factor of this Game Object.
             * @param y The vertical scroll factor of this Game Object. If not set it will use the `x` value. Default x.
             */
            setScrollFactor(x: number, y?: number): this;

            /**
             * The Texture this Game Object is using to render with.
             */
            texture: Phaser.Textures.Texture | Phaser.Textures.CanvasTexture;

            /**
             * The Texture Frame this Game Object is using to render with.
             */
            frame: Phaser.Textures.Frame;

            /**
             * Sets the texture and frame this Game Object will use to render with.
             * 
             * Textures are referenced by their string-based keys, as stored in the Texture Manager.
             * @param key The key of the texture to be used, as stored in the Texture Manager, or a Texture instance.
             * @param frame The name or index of the frame within the Texture.
             */
            setTexture(key: string | Phaser.Textures.Texture, frame?: string | number): this;

            /**
             * Sets the frame this Game Object will use to render with.
             * 
             * The Frame has to belong to the current Texture being used.
             * 
             * It can be either a string or an index.
             * 
             * Calling `setFrame` will modify the `width` and `height` properties of your Game Object.
             * It will also change the `origin` if the Frame has a custom pivot point, as exported from packages like Texture Packer.
             * @param frame The name or index of the frame within the Texture.
             * @param updateSize Should this call adjust the size of the Game Object? Default true.
             * @param updateOrigin Should this call adjust the origin of the Game Object? Default true.
             */
            setFrame(frame: string | number, updateSize?: boolean, updateOrigin?: boolean): this;

            /**
             * The tint value being applied to the top-left vertice of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             * The value should be set as a hex number, i.e. 0xff0000 for red, or 0xff00ff for purple.
             */
            tintTopLeft: number;

            /**
             * The tint value being applied to the top-right vertice of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             * The value should be set as a hex number, i.e. 0xff0000 for red, or 0xff00ff for purple.
             */
            tintTopRight: number;

            /**
             * The tint value being applied to the bottom-left vertice of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             * The value should be set as a hex number, i.e. 0xff0000 for red, or 0xff00ff for purple.
             */
            tintBottomLeft: number;

            /**
             * The tint value being applied to the bottom-right vertice of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             * The value should be set as a hex number, i.e. 0xff0000 for red, or 0xff00ff for purple.
             */
            tintBottomRight: number;

            /**
             * The tint fill mode.
             * 
             * `false` = An additive tint (the default), where vertices colors are blended with the texture.
             * `true` = A fill tint, where the vertices colors replace the texture, but respects texture alpha.
             */
            tintFill: boolean;

            /**
             * Clears all tint values associated with this Game Object.
             * 
             * Immediately sets the color values back to 0xffffff and the tint type to 'additive',
             * which results in no visible change to the texture.
             */
            clearTint(): this;

            /**
             * Sets an additive tint on this Game Object.
             * 
             * The tint works by taking the pixel color values from the Game Objects texture, and then
             * multiplying it by the color value of the tint. You can provide either one color value,
             * in which case the whole Game Object will be tinted in that color. Or you can provide a color
             * per corner. The colors are blended together across the extent of the Game Object.
             * 
             * To modify the tint color once set, either call this method again with new values or use the
             * `tint` property to set all colors at once. Or, use the properties `tintTopLeft`, `tintTopRight,
             * `tintBottomLeft` and `tintBottomRight` to set the corner color values independently.
             * 
             * To remove a tint call `clearTint`.
             * 
             * To swap this from being an additive tint to a fill based tint set the property `tintFill` to `true`.
             * @param topLeft The tint being applied to the top-left of the Game Object. If no other values are given this value is applied evenly, tinting the whole Game Object. Default 0xffffff.
             * @param topRight The tint being applied to the top-right of the Game Object.
             * @param bottomLeft The tint being applied to the bottom-left of the Game Object.
             * @param bottomRight The tint being applied to the bottom-right of the Game Object.
             */
            setTint(topLeft?: number, topRight?: number, bottomLeft?: number, bottomRight?: number): this;

            /**
             * Sets a fill-based tint on this Game Object.
             * 
             * Unlike an additive tint, a fill-tint literally replaces the pixel colors from the texture
             * with those in the tint. You can use this for effects such as making a player flash 'white'
             * if hit by something. You can provide either one color value, in which case the whole
             * Game Object will be rendered in that color. Or you can provide a color per corner. The colors
             * are blended together across the extent of the Game Object.
             * 
             * To modify the tint color once set, either call this method again with new values or use the
             * `tint` property to set all colors at once. Or, use the properties `tintTopLeft`, `tintTopRight,
             * `tintBottomLeft` and `tintBottomRight` to set the corner color values independently.
             * 
             * To remove a tint call `clearTint`.
             * 
             * To swap this from being a fill-tint to an additive tint set the property `tintFill` to `false`.
             * @param topLeft The tint being applied to the top-left of the Game Object. If not other values are given this value is applied evenly, tinting the whole Game Object. Default 0xffffff.
             * @param topRight The tint being applied to the top-right of the Game Object.
             * @param bottomLeft The tint being applied to the bottom-left of the Game Object.
             * @param bottomRight The tint being applied to the bottom-right of the Game Object.
             */
            setTintFill(topLeft?: number, topRight?: number, bottomLeft?: number, bottomRight?: number): this;

            /**
             * The tint value being applied to the whole of the Game Object.
             * This property is a setter-only. Use the properties `tintTopLeft` etc to read the current tint value.
             */
            tint: number;

            /**
             * Does this Game Object have a tint applied?
             * 
             * It checks to see if the 4 tint properties are set to the value 0xffffff
             * and that the `tintFill` property is `false`. This indicates that a Game Object isn't tinted.
             */
            readonly isTinted: boolean;

            /**
             * The x position of this Game Object.
             */
            x: number;

            /**
             * The y position of this Game Object.
             */
            y: number;

            /**
             * The z position of this Game Object.
             * 
             * Note: The z position does not control the rendering order of 2D Game Objects. Use
             * {@link Phaser.GameObjects.Components.Depth#depth} instead.
             */
            z: number;

            /**
             * The w position of this Game Object.
             */
            w: number;

            /**
             * This is a special setter that allows you to set both the horizontal and vertical scale of this Game Object
             * to the same value, at the same time. When reading this value the result returned is `(scaleX + scaleY) / 2`.
             * 
             * Use of this property implies you wish the horizontal and vertical scales to be equal to each other. If this
             * isn't the case, use the `scaleX` or `scaleY` properties instead.
             */
            scale: number;

            /**
             * The horizontal scale of this Game Object.
             */
            scaleX: number;

            /**
             * The vertical scale of this Game Object.
             */
            scaleY: number;

            /**
             * The angle of this Game Object as expressed in degrees.
             * 
             * Phaser uses a right-hand clockwise rotation system, where 0 is right, 90 is down, 180/-180 is left
             * and -90 is up.
             * 
             * If you prefer to work in radians, see the `rotation` property instead.
             */
            angle: number;

            /**
             * The angle of this Game Object in radians.
             * 
             * Phaser uses a right-hand clockwise rotation system, where 0 is right, PI/2 is down, +-PI is left
             * and -PI/2 is up.
             * 
             * If you prefer to work in degrees, see the `angle` property instead.
             */
            rotation: number;

            /**
             * Sets the position of this Game Object.
             * @param x The x position of this Game Object. Default 0.
             * @param y The y position of this Game Object. If not set it will use the `x` value. Default x.
             * @param z The z position of this Game Object. Default 0.
             * @param w The w position of this Game Object. Default 0.
             */
            setPosition(x?: number, y?: number, z?: number, w?: number): this;

            /**
             * Copies an object's coordinates to this Game Object's position.
             * @param source An object with numeric 'x', 'y', 'z', or 'w' properties. Undefined values are not copied.
             */
            copyPosition(source: Phaser.Types.Math.Vector2Like | Phaser.Types.Math.Vector3Like | Phaser.Types.Math.Vector4Like): this;

            /**
             * Sets the position of this Game Object to be a random position within the confines of
             * the given area.
             * 
             * If no area is specified a random position between 0 x 0 and the game width x height is used instead.
             * 
             * The position does not factor in the size of this Game Object, meaning that only the origin is
             * guaranteed to be within the area.
             * @param x The x position of the top-left of the random area. Default 0.
             * @param y The y position of the top-left of the random area. Default 0.
             * @param width The width of the random area.
             * @param height The height of the random area.
             */
            setRandomPosition(x?: number, y?: number, width?: number, height?: number): this;

            /**
             * Sets the rotation of this Game Object.
             * @param radians The rotation of this Game Object, in radians. Default 0.
             */
            setRotation(radians?: number): this;

            /**
             * Sets the angle of this Game Object.
             * @param degrees The rotation of this Game Object, in degrees. Default 0.
             */
            setAngle(degrees?: number): this;

            /**
             * Sets the scale of this Game Object.
             * @param x The horizontal scale of this Game Object.
             * @param y The vertical scale of this Game Object. If not set it will use the `x` value. Default x.
             */
            setScale(x: number, y?: number): this;

            /**
             * Sets the x position of this Game Object.
             * @param value The x position of this Game Object. Default 0.
             */
            setX(value?: number): this;

            /**
             * Sets the y position of this Game Object.
             * @param value The y position of this Game Object. Default 0.
             */
            setY(value?: number): this;

            /**
             * Sets the z position of this Game Object.
             * 
             * Note: The z position does not control the rendering order of 2D Game Objects. Use
             * {@link Phaser.GameObjects.Components.Depth#setDepth} instead.
             * @param value The z position of this Game Object. Default 0.
             */
            setZ(value?: number): this;

            /**
             * Sets the w position of this Game Object.
             * @param value The w position of this Game Object. Default 0.
             */
            setW(value?: number): this;

            /**
             * Gets the local transform matrix for this Game Object.
             * @param tempMatrix The matrix to populate with the values from this Game Object.
             */
            getLocalTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;

            /**
             * Gets the world transform matrix for this Game Object, factoring in any parent Containers.
             * @param tempMatrix The matrix to populate with the values from this Game Object.
             * @param parentMatrix A temporary matrix to hold parent values during the calculations.
             */
            getWorldTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix, parentMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;

            /**
             * Takes the given `x` and `y` coordinates and converts them into local space for this
             * Game Object, taking into account parent and local transforms, and the Display Origin.
             * 
             * The returned Vector2 contains the translated point in its properties.
             * 
             * A Camera needs to be provided in order to handle modified scroll factors. If no
             * camera is specified, it will use the `main` camera from the Scene to which this
             * Game Object belongs.
             * @param x The x position to translate.
             * @param y The y position to translate.
             * @param point A Vector2, or point-like object, to store the results in.
             * @param camera The Camera which is being tested against. If not given will use the Scene default camera.
             */
            getLocalPoint(x: number, y: number, point?: Phaser.Math.Vector2, camera?: Phaser.Cameras.Scene2D.Camera): Phaser.Math.Vector2;

            /**
             * Gets the sum total rotation of all of this Game Objects parent Containers.
             * 
             * The returned value is in radians and will be zero if this Game Object has no parent container.
             */
            getParentRotation(): number;

            /**
             * The visible state of the Game Object.
             * 
             * An invisible Game Object will skip rendering, but will still process update logic.
             */
            visible: boolean;

            /**
             * Sets the visibility of this Game Object.
             * 
             * An invisible Game Object will skip rendering, but will still process update logic.
             * @param value The visible state of the Game Object.
             */
            setVisible(value: boolean): this;

        }

        /**
         * A Blitter Game Object.
         * 
         * The Blitter Game Object is a special kind of container that creates, updates and manages Bob objects.
         * Bobs are designed for rendering speed rather than flexibility. They consist of a texture, or frame from a texture,
         * a position and an alpha value. You cannot scale or rotate them. They use a batched drawing method for speed
         * during rendering.
         * 
         * A Blitter Game Object has one texture bound to it. Bobs created by the Blitter can use any Frame from this
         * Texture to render with, but they cannot use any other Texture. It is this single texture-bind that allows
         * them their speed.
         * 
         * If you have a need to blast a large volume of frames around the screen then Blitter objects are well worth
         * investigating. They are especially useful for using as a base for your own special effects systems.
         */
        class Blitter extends Phaser.GameObjects.GameObject implements Phaser.GameObjects.Components.Alpha, Phaser.GameObjects.Components.BlendMode, Phaser.GameObjects.Components.Depth, Phaser.GameObjects.Components.Mask, Phaser.GameObjects.Components.Pipeline, Phaser.GameObjects.Components.ScrollFactor, Phaser.GameObjects.Components.Size, Phaser.GameObjects.Components.Texture, Phaser.GameObjects.Components.Transform, Phaser.GameObjects.Components.Visible {
            /**
             * 
             * @param scene The Scene to which this Game Object belongs. It can only belong to one Scene at any given time.
             * @param x The x coordinate of this Game Object in world space. Default 0.
             * @param y The y coordinate of this Game Object in world space. Default 0.
             * @param texture The key of the texture this Game Object will use for rendering. The Texture must already exist in the Texture Manager. Default '__DEFAULT'.
             * @param frame The Frame of the Texture that this Game Object will use. Only set if the Texture has multiple frames, such as a Texture Atlas or Sprite Sheet. Default 0.
             */
            constructor(scene: Phaser.Scene, x?: number, y?: number, texture?: string, frame?: string | number);

            /**
             * The children of this Blitter.
             * This List contains all of the Bob objects created by the Blitter.
             */
            children: Phaser.Structs.List<Phaser.GameObjects.Bob>;

            /**
             * Is the Blitter considered dirty?
             * A 'dirty' Blitter has had its child count changed since the last frame.
             */
            dirty: boolean;

            /**
             * Creates a new Bob in this Blitter.
             * 
             * The Bob is created at the given coordinates, relative to the Blitter and uses the given frame.
             * A Bob can use any frame belonging to the texture bound to the Blitter.
             * @param x The x position of the Bob. Bob coordinate are relative to the position of the Blitter object.
             * @param y The y position of the Bob. Bob coordinate are relative to the position of the Blitter object.
             * @param frame The Frame the Bob will use. It _must_ be part of the Texture the parent Blitter object is using.
             * @param visible Should the created Bob render or not? Default true.
             * @param index The position in the Blitters Display List to add the new Bob at. Defaults to the top of the list.
             */
            create(x: number, y: number, frame?: string | number | Phaser.Textures.Frame, visible?: boolean, index?: number): Phaser.GameObjects.Bob;

            /**
             * Creates multiple Bob objects within this Blitter and then passes each of them to the specified callback.
             * @param callback The callback to invoke after creating a bob. It will be sent two arguments: The Bob and the index of the Bob.
             * @param quantity The quantity of Bob objects to create.
             * @param frame The Frame the Bobs will use. It must be part of the Blitter Texture.
             * @param visible Should the created Bob render or not? Default true.
             */
            createFromCallback(callback: CreateCallback, quantity: number, frame?: string | number | Phaser.Textures.Frame | string[] | number[] | Phaser.Textures.Frame[], visible?: boolean): Phaser.GameObjects.Bob[];

            /**
             * Creates multiple Bobs in one call.
             * 
             * The amount created is controlled by a combination of the `quantity` argument and the number of frames provided.
             * 
             * If the quantity is set to 10 and you provide 2 frames, then 20 Bobs will be created. 10 with the first
             * frame and 10 with the second.
             * @param quantity The quantity of Bob objects to create.
             * @param frame The Frame the Bobs will use. It must be part of the Blitter Texture.
             * @param visible Should the created Bob render or not? Default true.
             */
            createMultiple(quantity: number, frame?: string | number | Phaser.Textures.Frame | string[] | number[] | Phaser.Textures.Frame[], visible?: boolean): Phaser.GameObjects.Bob[];

            /**
             * Checks if the given child can render or not, by checking its `visible` and `alpha` values.
             * @param child The Bob to check for rendering.
             */
            childCanRender(child: Phaser.GameObjects.Bob): boolean;

            /**
             * Returns an array of Bobs to be rendered.
             * If the Blitter is dirty then a new list is generated and stored in `renderList`.
             */
            getRenderList(): Phaser.GameObjects.Bob[];

            /**
             * Removes all Bobs from the children List and clears the dirty flag.
             */
            clear(): void;

            /**
             * Internal destroy handler, called as part of the destroy process.
             */
            protected preDestroy(): void;

            /**
             * Clears all alpha values associated with this Game Object.
             * 
             * Immediately sets the alpha levels back to 1 (fully opaque).
             */
            clearAlpha(): this;

            /**
             * Set the Alpha level of this Game Object. The alpha controls the opacity of the Game Object as it renders.
             * Alpha values are provided as a float between 0, fully transparent, and 1, fully opaque.
             * 
             * If your game is running under WebGL you can optionally specify four different alpha values, each of which
             * correspond to the four corners of the Game Object. Under Canvas only the `topLeft` value given is used.
             * @param topLeft The alpha value used for the top-left of the Game Object. If this is the only value given it's applied across the whole Game Object. Default 1.
             * @param topRight The alpha value used for the top-right of the Game Object. WebGL only.
             * @param bottomLeft The alpha value used for the bottom-left of the Game Object. WebGL only.
             * @param bottomRight The alpha value used for the bottom-right of the Game Object. WebGL only.
             */
            setAlpha(topLeft?: number, topRight?: number, bottomLeft?: number, bottomRight?: number): this;

            /**
             * The alpha value of the Game Object.
             * 
             * This is a global value, impacting the entire Game Object, not just a region of it.
             */
            alpha: number;

            /**
             * The alpha value starting from the top-left of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaTopLeft: number;

            /**
             * The alpha value starting from the top-right of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaTopRight: number;

            /**
             * The alpha value starting from the bottom-left of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaBottomLeft: number;

            /**
             * The alpha value starting from the bottom-right of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaBottomRight: number;

            /**
             * Sets the Blend Mode being used by this Game Object.
             * 
             * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
             * 
             * Under WebGL only the following Blend Modes are available:
             * 
             * * ADD
             * * MULTIPLY
             * * SCREEN
             * * ERASE
             * 
             * Canvas has more available depending on browser support.
             * 
             * You can also create your own custom Blend Modes in WebGL.
             * 
             * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
             * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
             * reasons try to be careful about the construction of your Scene and the frequency of which blend modes
             * are used.
             */
            blendMode: Phaser.BlendModes | string;

            /**
             * Sets the Blend Mode being used by this Game Object.
             * 
             * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
             * 
             * Under WebGL only the following Blend Modes are available:
             * 
             * * ADD
             * * MULTIPLY
             * * SCREEN
             * * ERASE (only works when rendering to a framebuffer, like a Render Texture)
             * 
             * Canvas has more available depending on browser support.
             * 
             * You can also create your own custom Blend Modes in WebGL.
             * 
             * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
             * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
             * reasons try to be careful about the construction of your Scene and the frequency in which blend modes
             * are used.
             * @param value The BlendMode value. Either a string or a CONST.
             */
            setBlendMode(value: string | Phaser.BlendModes): this;

            /**
             * The depth of this Game Object within the Scene.
             * 
             * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
             * of Game Objects, without actually moving their position in the display list.
             * 
             * The default depth is zero. A Game Object with a higher depth
             * value will always render in front of one with a lower value.
             * 
             * Setting the depth will queue a depth sort event within the Scene.
             */
            depth: number;

            /**
             * The depth of this Game Object within the Scene.
             * 
             * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
             * of Game Objects, without actually moving their position in the display list.
             * 
             * The default depth is zero. A Game Object with a higher depth
             * value will always render in front of one with a lower value.
             * 
             * Setting the depth will queue a depth sort event within the Scene.
             * @param value The depth of this Game Object.
             */
            setDepth(value: number): this;

            /**
             * The Mask this Game Object is using during render.
             */
            mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask;

            /**
             * Sets the mask that this Game Object will use to render with.
             * 
             * The mask must have been previously created and can be either a GeometryMask or a BitmapMask.
             * Note: Bitmap Masks only work on WebGL. Geometry Masks work on both WebGL and Canvas.
             * 
             * If a mask is already set on this Game Object it will be immediately replaced.
             * 
             * Masks are positioned in global space and are not relative to the Game Object to which they
             * are applied. The reason for this is that multiple Game Objects can all share the same mask.
             * 
             * Masks have no impact on physics or input detection. They are purely a rendering component
             * that allows you to limit what is visible during the render pass.
             * @param mask The mask this Game Object will use when rendering.
             */
            setMask(mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask): this;

            /**
             * Clears the mask that this Game Object was using.
             * @param destroyMask Destroy the mask before clearing it? Default false.
             */
            clearMask(destroyMask?: boolean): this;

            /**
             * Creates and returns a Bitmap Mask. This mask can be used by any Game Object,
             * including this one.
             * 
             * Note: Bitmap Masks only work on WebGL. Geometry Masks work on both WebGL and Canvas.
             * 
             * To create the mask you need to pass in a reference to a renderable Game Object.
             * A renderable Game Object is one that uses a texture to render with, such as an
             * Image, Sprite, Render Texture or BitmapText.
             * 
             * If you do not provide a renderable object, and this Game Object has a texture,
             * it will use itself as the object. This means you can call this method to create
             * a Bitmap Mask from any renderable Game Object.
             * @param renderable A renderable Game Object that uses a texture, such as a Sprite.
             */
            createBitmapMask(renderable?: Phaser.GameObjects.GameObject): Phaser.Display.Masks.BitmapMask;

            /**
             * Creates and returns a Geometry Mask. This mask can be used by any Game Object,
             * including this one.
             * 
             * To create the mask you need to pass in a reference to a Graphics Game Object.
             * 
             * If you do not provide a graphics object, and this Game Object is an instance
             * of a Graphics object, then it will use itself to create the mask.
             * 
             * This means you can call this method to create a Geometry Mask from any Graphics Game Object.
             * @param graphics A Graphics Game Object. The geometry within it will be used as the mask.
             */
            createGeometryMask(graphics?: Phaser.GameObjects.Graphics): Phaser.Display.Masks.GeometryMask;

            /**
             * The initial WebGL pipeline of this Game Object.
             * 
             * If you call `resetPipeline` on this Game Object, the pipeline is reset to this default.
             */
            defaultPipeline: Phaser.Renderer.WebGL.WebGLPipeline;

            /**
             * The current WebGL pipeline of this Game Object.
             */
            pipeline: Phaser.Renderer.WebGL.WebGLPipeline;

            /**
             * Does this Game Object have any Post Pipelines set?
             */
            hasPostPipeline: boolean;

            /**
             * The WebGL Post FX Pipelines this Game Object uses for post-render effects.
             * 
             * The pipelines are processed in the order in which they appear in this array.
             * 
             * If you modify this array directly, be sure to set the
             * `hasPostPipeline` property accordingly.
             */
            postPipelines: Phaser.Renderer.WebGL.Pipelines.PostFXPipeline[];

            /**
             * An object to store pipeline specific data in, to be read by the pipelines this Game Object uses.
             */
            pipelineData: object;

            /**
             * Sets the initial WebGL Pipeline of this Game Object.
             * 
             * This should only be called during the instantiation of the Game Object. After that, use `setPipeline`.
             * @param pipeline Either the string-based name of the pipeline, or a pipeline instance to set.
             */
            initPipeline(pipeline: string | Phaser.Renderer.WebGL.WebGLPipeline): boolean;

            /**
             * Sets the main WebGL Pipeline of this Game Object.
             * 
             * Also sets the `pipelineData` property, if the parameter is given.
             * 
             * Both the pipeline and post pipelines share the same pipeline data object.
             * @param pipeline Either the string-based name of the pipeline, or a pipeline instance to set.
             * @param pipelineData Optional pipeline data object that is _deep copied_ into the `pipelineData` property of this Game Object.
             * @param copyData Should the pipeline data object be _deep copied_ into the `pipelineData` property of this Game Object? If `false` it will be set by reference instead. Default true.
             */
            setPipeline(pipeline: string | Phaser.Renderer.WebGL.WebGLPipeline, pipelineData?: object, copyData?: boolean): this;

            /**
             * Sets one, or more, Post Pipelines on this Game Object.
             * 
             * Post Pipelines are invoked after this Game Object has rendered to its target and
             * are commonly used for post-fx.
             * 
             * The post pipelines are appended to the `postPipelines` array belonging to this
             * Game Object. When the renderer processes this Game Object, it iterates through the post
             * pipelines in the order in which they appear in the array. If you are stacking together
             * multiple effects, be aware that the order is important.
             * 
             * If you call this method multiple times, the new pipelines will be appended to any existing
             * post pipelines already set. Use the `resetPostPipeline` method to clear them first, if required.
             * 
             * You can optionally also sets the `pipelineData` property, if the parameter is given.
             * 
             * Both the pipeline and post pipelines share the pipeline data object together.
             * @param pipelines Either the string-based name of the pipeline, or a pipeline instance, or class, or an array of them.
             * @param pipelineData Optional pipeline data object that is _deep copied_ into the `pipelineData` property of this Game Object.
             * @param copyData Should the pipeline data object be _deep copied_ into the `pipelineData` property of this Game Object? If `false` it will be set by reference instead. Default true.
             */
            setPostPipeline(pipelines: string | string[] | Function | Function[] | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline[], pipelineData?: object, copyData?: boolean): this;

            /**
             * Adds an entry to the `pipelineData` object belonging to this Game Object.
             * 
             * If the 'key' already exists, its value is updated. If it doesn't exist, it is created.
             * 
             * If `value` is undefined, and `key` exists, `key` is removed from the data object.
             * 
             * Both the pipeline and post pipelines share the pipeline data object together.
             * @param key The key of the pipeline data to set, update, or delete.
             * @param value The value to be set with the key. If `undefined` then `key` will be deleted from the object.
             */
            setPipelineData(key: string, value?: any): this;

            /**
             * Gets a Post Pipeline instance from this Game Object, based on the given name, and returns it.
             * @param pipeline The string-based name of the pipeline, or a pipeline class.
             */
            getPostPipeline(pipeline: string | Function | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline): Phaser.Renderer.WebGL.Pipelines.PostFXPipeline | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline[];

            /**
             * Resets the WebGL Pipeline of this Game Object back to the default it was created with.
             * @param resetPostPipelines Reset all of the post pipelines? Default false.
             * @param resetData Reset the `pipelineData` object to being an empty object? Default false.
             */
            resetPipeline(resetPostPipelines?: boolean, resetData?: boolean): boolean;

            /**
             * Resets the WebGL Post Pipelines of this Game Object. It does this by calling
             * the `destroy` method on each post pipeline and then clearing the local array.
             * @param resetData Reset the `pipelineData` object to being an empty object? Default false.
             */
            resetPostPipeline(resetData?: boolean): void;

            /**
             * Removes a type of Post Pipeline instances from this Game Object, based on the given name, and destroys them.
             * 
             * If you wish to remove all Post Pipelines use the `resetPostPipeline` method instead.
             * @param pipeline The string-based name of the pipeline, or a pipeline class.
             */
            removePostPipeline(pipeline: string | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline): this;

            /**
             * Gets the name of the WebGL Pipeline this Game Object is currently using.
             */
            getPipelineName(): string;

            /**
             * The horizontal scroll factor of this Game Object.
             * 
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             * 
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             * 
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             * 
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             */
            scrollFactorX: number;

            /**
             * The vertical scroll factor of this Game Object.
             * 
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             * 
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             * 
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             * 
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             */
            scrollFactorY: number;

            /**
             * Sets the scroll factor of this Game Object.
             * 
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             * 
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             * 
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             * 
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             * @param x The horizontal scroll factor of this Game Object.
             * @param y The vertical scroll factor of this Game Object. If not set it will use the `x` value. Default x.
             */
            setScrollFactor(x: number, y?: number): this;

            /**
             * The native (un-scaled) width of this Game Object.
             * 
             * Changing this value will not change the size that the Game Object is rendered in-game.
             * For that you need to either set the scale of the Game Object (`setScale`) or use
             * the `displayWidth` property.
             */
            width: number;

            /**
             * The native (un-scaled) height of this Game Object.
             * 
             * Changing this value will not change the size that the Game Object is rendered in-game.
             * For that you need to either set the scale of the Game Object (`setScale`) or use
             * the `displayHeight` property.
             */
            height: number;

            /**
             * The displayed width of this Game Object.
             * 
             * This value takes into account the scale factor.
             * 
             * Setting this value will adjust the Game Object's scale property.
             */
            displayWidth: number;

            /**
             * The displayed height of this Game Object.
             * 
             * This value takes into account the scale factor.
             * 
             * Setting this value will adjust the Game Object's scale property.
             */
            displayHeight: number;

            /**
             * Sets the size of this Game Object to be that of the given Frame.
             * 
             * This will not change the size that the Game Object is rendered in-game.
             * For that you need to either set the scale of the Game Object (`setScale`) or call the
             * `setDisplaySize` method, which is the same thing as changing the scale but allows you
             * to do so by giving pixel values.
             * 
             * If you have enabled this Game Object for input, changing the size will _not_ change the
             * size of the hit area. To do this you should adjust the `input.hitArea` object directly.
             * @param frame The frame to base the size of this Game Object on.
             */
            setSizeToFrame(frame: Phaser.Textures.Frame): this;

            /**
             * Sets the internal size of this Game Object, as used for frame or physics body creation.
             * 
             * This will not change the size that the Game Object is rendered in-game.
             * For that you need to either set the scale of the Game Object (`setScale`) or call the
             * `setDisplaySize` method, which is the same thing as changing the scale but allows you
             * to do so by giving pixel values.
             * 
             * If you have enabled this Game Object for input, changing the size will _not_ change the
             * size of the hit area. To do this you should adjust the `input.hitArea` object directly.
             * @param width The width of this Game Object.
             * @param height The height of this Game Object.
             */
            setSize(width: number, height: number): this;

            /**
             * Sets the display size of this Game Object.
             * 
             * Calling this will adjust the scale.
             * @param width The width of this Game Object.
             * @param height The height of this Game Object.
             */
            setDisplaySize(width: number, height: number): this;

            /**
             * The Texture this Game Object is using to render with.
             */
            texture: Phaser.Textures.Texture | Phaser.Textures.CanvasTexture;

            /**
             * The Texture Frame this Game Object is using to render with.
             */
            frame: Phaser.Textures.Frame;

            /**
             * Sets the texture and frame this Game Object will use to render with.
             * 
             * Textures are referenced by their string-based keys, as stored in the Texture Manager.
             * @param key The key of the texture to be used, as stored in the Texture Manager, or a Texture instance.
             * @param frame The name or index of the frame within the Texture.
             */
            setTexture(key: string | Phaser.Textures.Texture, frame?: string | number): this;

            /**
             * Sets the frame this Game Object will use to render with.
             * 
             * The Frame has to belong to the current Texture being used.
             * 
             * It can be either a string or an index.
             * 
             * Calling `setFrame` will modify the `width` and `height` properties of your Game Object.
             * It will also change the `origin` if the Frame has a custom pivot point, as exported from packages like Texture Packer.
             * @param frame The name or index of the frame within the Texture.
             * @param updateSize Should this call adjust the size of the Game Object? Default true.
             * @param updateOrigin Should this call adjust the origin of the Game Object? Default true.
             */
            setFrame(frame: string | number, updateSize?: boolean, updateOrigin?: boolean): this;

            /**
             * The x position of this Game Object.
             */
            x: number;

            /**
             * The y position of this Game Object.
             */
            y: number;

            /**
             * The z position of this Game Object.
             * 
             * Note: The z position does not control the rendering order of 2D Game Objects. Use
             * {@link Phaser.GameObjects.Components.Depth#depth} instead.
             */
            z: number;

            /**
             * The w position of this Game Object.
             */
            w: number;

            /**
             * This is a special setter that allows you to set both the horizontal and vertical scale of this Game Object
             * to the same value, at the same time. When reading this value the result returned is `(scaleX + scaleY) / 2`.
             * 
             * Use of this property implies you wish the horizontal and vertical scales to be equal to each other. If this
             * isn't the case, use the `scaleX` or `scaleY` properties instead.
             */
            scale: number;

            /**
             * The horizontal scale of this Game Object.
             */
            scaleX: number;

            /**
             * The vertical scale of this Game Object.
             */
            scaleY: number;

            /**
             * The angle of this Game Object as expressed in degrees.
             * 
             * Phaser uses a right-hand clockwise rotation system, where 0 is right, 90 is down, 180/-180 is left
             * and -90 is up.
             * 
             * If you prefer to work in radians, see the `rotation` property instead.
             */
            angle: number;

            /**
             * The angle of this Game Object in radians.
             * 
             * Phaser uses a right-hand clockwise rotation system, where 0 is right, PI/2 is down, +-PI is left
             * and -PI/2 is up.
             * 
             * If you prefer to work in degrees, see the `angle` property instead.
             */
            rotation: number;

            /**
             * Sets the position of this Game Object.
             * @param x The x position of this Game Object. Default 0.
             * @param y The y position of this Game Object. If not set it will use the `x` value. Default x.
             * @param z The z position of this Game Object. Default 0.
             * @param w The w position of this Game Object. Default 0.
             */
            setPosition(x?: number, y?: number, z?: number, w?: number): this;

            /**
             * Copies an object's coordinates to this Game Object's position.
             * @param source An object with numeric 'x', 'y', 'z', or 'w' properties. Undefined values are not copied.
             */
            copyPosition(source: Phaser.Types.Math.Vector2Like | Phaser.Types.Math.Vector3Like | Phaser.Types.Math.Vector4Like): this;

            /**
             * Sets the position of this Game Object to be a random position within the confines of
             * the given area.
             * 
             * If no area is specified a random position between 0 x 0 and the game width x height is used instead.
             * 
             * The position does not factor in the size of this Game Object, meaning that only the origin is
             * guaranteed to be within the area.
             * @param x The x position of the top-left of the random area. Default 0.
             * @param y The y position of the top-left of the random area. Default 0.
             * @param width The width of the random area.
             * @param height The height of the random area.
             */
            setRandomPosition(x?: number, y?: number, width?: number, height?: number): this;

            /**
             * Sets the rotation of this Game Object.
             * @param radians The rotation of this Game Object, in radians. Default 0.
             */
            setRotation(radians?: number): this;

            /**
             * Sets the angle of this Game Object.
             * @param degrees The rotation of this Game Object, in degrees. Default 0.
             */
            setAngle(degrees?: number): this;

            /**
             * Sets the scale of this Game Object.
             * @param x The horizontal scale of this Game Object.
             * @param y The vertical scale of this Game Object. If not set it will use the `x` value. Default x.
             */
            setScale(x: number, y?: number): this;

            /**
             * Sets the x position of this Game Object.
             * @param value The x position of this Game Object. Default 0.
             */
            setX(value?: number): this;

            /**
             * Sets the y position of this Game Object.
             * @param value The y position of this Game Object. Default 0.
             */
            setY(value?: number): this;

            /**
             * Sets the z position of this Game Object.
             * 
             * Note: The z position does not control the rendering order of 2D Game Objects. Use
             * {@link Phaser.GameObjects.Components.Depth#setDepth} instead.
             * @param value The z position of this Game Object. Default 0.
             */
            setZ(value?: number): this;

            /**
             * Sets the w position of this Game Object.
             * @param value The w position of this Game Object. Default 0.
             */
            setW(value?: number): this;

            /**
             * Gets the local transform matrix for this Game Object.
             * @param tempMatrix The matrix to populate with the values from this Game Object.
             */
            getLocalTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;

            /**
             * Gets the world transform matrix for this Game Object, factoring in any parent Containers.
             * @param tempMatrix The matrix to populate with the values from this Game Object.
             * @param parentMatrix A temporary matrix to hold parent values during the calculations.
             */
            getWorldTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix, parentMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;

            /**
             * Takes the given `x` and `y` coordinates and converts them into local space for this
             * Game Object, taking into account parent and local transforms, and the Display Origin.
             * 
             * The returned Vector2 contains the translated point in its properties.
             * 
             * A Camera needs to be provided in order to handle modified scroll factors. If no
             * camera is specified, it will use the `main` camera from the Scene to which this
             * Game Object belongs.
             * @param x The x position to translate.
             * @param y The y position to translate.
             * @param point A Vector2, or point-like object, to store the results in.
             * @param camera The Camera which is being tested against. If not given will use the Scene default camera.
             */
            getLocalPoint(x: number, y: number, point?: Phaser.Math.Vector2, camera?: Phaser.Cameras.Scene2D.Camera): Phaser.Math.Vector2;

            /**
             * Gets the sum total rotation of all of this Game Objects parent Containers.
             * 
             * The returned value is in radians and will be zero if this Game Object has no parent container.
             */
            getParentRotation(): number;

            /**
             * The visible state of the Game Object.
             * 
             * An invisible Game Object will skip rendering, but will still process update logic.
             */
            visible: boolean;

            /**
             * Sets the visibility of this Game Object.
             * 
             * An invisible Game Object will skip rendering, but will still process update logic.
             * @param value The visible state of the Game Object.
             */
            setVisible(value: boolean): this;

        }

        /**
         * A Bob Game Object.
         * 
         * A Bob belongs to a Blitter Game Object. The Blitter is responsible for managing and rendering this object.
         * 
         * A Bob has a position, alpha value and a frame from a texture that it uses to render with. You can also toggle
         * the flipped and visible state of the Bob. The Frame the Bob uses to render can be changed dynamically, but it
         * must be a Frame within the Texture used by the parent Blitter.
         * 
         * Bob positions are relative to the Blitter parent. So if you move the Blitter parent, all Bob children will
         * have their positions impacted by this change as well.
         * 
         * You can manipulate Bob objects directly from your game code, but the creation and destruction of them should be
         * handled via the Blitter parent.
         */
        class Bob {
            /**
             * 
             * @param blitter The parent Blitter object is responsible for updating this Bob.
             * @param x The horizontal position of this Game Object in the world, relative to the parent Blitter position.
             * @param y The vertical position of this Game Object in the world, relative to the parent Blitter position.
             * @param frame The Frame this Bob will render with, as defined in the Texture the parent Blitter is using.
             * @param visible Should the Bob render visible or not to start with?
             */
            constructor(blitter: Phaser.GameObjects.Blitter, x: number, y: number, frame: string | number, visible: boolean);

            /**
             * The Blitter object that this Bob belongs to.
             */
            parent: Phaser.GameObjects.Blitter;

            /**
             * The x position of this Bob, relative to the x position of the Blitter.
             */
            x: number;

            /**
             * The y position of this Bob, relative to the y position of the Blitter.
             */
            y: number;

            /**
             * The frame that the Bob uses to render with.
             * To change the frame use the `Bob.setFrame` method.
             */
            protected frame: Phaser.Textures.Frame;

            /**
             * A blank object which can be used to store data related to this Bob in.
             */
            data: object;

            /**
             * The tint value of this Bob.
             */
            tint: number;

            /**
             * The horizontally flipped state of the Bob.
             * A Bob that is flipped horizontally will render inversed on the horizontal axis.
             * Flipping always takes place from the middle of the texture.
             */
            flipX: boolean;

            /**
             * The vertically flipped state of the Bob.
             * A Bob that is flipped vertically will render inversed on the vertical axis (i.e. upside down)
             * Flipping always takes place from the middle of the texture.
             */
            flipY: boolean;

            /**
             * Changes the Texture Frame being used by this Bob.
             * The frame must be part of the Texture the parent Blitter is using.
             * If no value is given it will use the default frame of the Blitter parent.
             * @param frame The frame to be used during rendering.
             */
            setFrame(frame?: string | number | Phaser.Textures.Frame): this;

            /**
             * Resets the horizontal and vertical flipped state of this Bob back to their default un-flipped state.
             */
            resetFlip(): this;

            /**
             * Resets this Bob.
             * 
             * Changes the position to the values given, and optionally changes the frame.
             * 
             * Also resets the flipX and flipY values, sets alpha back to 1 and visible to true.
             * @param x The x position of the Bob. Bob coordinate are relative to the position of the Blitter object.
             * @param y The y position of the Bob. Bob coordinate are relative to the position of the Blitter object.
             * @param frame The Frame the Bob will use. It _must_ be part of the Texture the parent Blitter object is using.
             */
            reset(x: number, y: number, frame?: string | number | Phaser.Textures.Frame): this;

            /**
             * Changes the position of this Bob to the values given.
             * @param x The x position of the Bob. Bob coordinate are relative to the position of the Blitter object.
             * @param y The y position of the Bob. Bob coordinate are relative to the position of the Blitter object.
             */
            setPosition(x: number, y: number): this;

            /**
             * Sets the horizontal flipped state of this Bob.
             * @param value The flipped state. `false` for no flip, or `true` to be flipped.
             */
            setFlipX(value: boolean): this;

            /**
             * Sets the vertical flipped state of this Bob.
             * @param value The flipped state. `false` for no flip, or `true` to be flipped.
             */
            setFlipY(value: boolean): this;

            /**
             * Sets the horizontal and vertical flipped state of this Bob.
             * @param x The horizontal flipped state. `false` for no flip, or `true` to be flipped.
             * @param y The horizontal flipped state. `false` for no flip, or `true` to be flipped.
             */
            setFlip(x: boolean, y: boolean): this;

            /**
             * Sets the visibility of this Bob.
             * 
             * An invisible Bob will skip rendering.
             * @param value The visible state of the Game Object.
             */
            setVisible(value: boolean): this;

            /**
             * Set the Alpha level of this Bob. The alpha controls the opacity of the Game Object as it renders.
             * Alpha values are provided as a float between 0, fully transparent, and 1, fully opaque.
             * 
             * A Bob with alpha 0 will skip rendering.
             * @param value The alpha value used for this Bob. Between 0 and 1.
             */
            setAlpha(value: number): this;

            /**
             * Sets the tint of this Bob.
             * @param value The tint value used for this Bob. Between 0 and 0xffffff.
             */
            setTint(value: number): this;

            /**
             * Destroys this Bob instance.
             * Removes itself from the Blitter and clears the parent, frame and data properties.
             */
            destroy(): void;

            /**
             * The visible state of the Bob.
             * 
             * An invisible Bob will skip rendering.
             */
            visible: boolean;

            /**
             * The alpha value of the Bob, between 0 and 1.
             * 
             * A Bob with alpha 0 will skip rendering.
             */
            alpha: number;

        }

        /**
         * Builds a Game Object using the provided configuration object.
         * @param scene A reference to the Scene.
         * @param gameObject The initial GameObject.
         * @param config The config to build the GameObject with.
         */
        function BuildGameObject(scene: Phaser.Scene, gameObject: Phaser.GameObjects.GameObject, config: Phaser.Types.GameObjects.GameObjectConfig): Phaser.GameObjects.GameObject;

        /**
         * Adds an Animation component to a Sprite and populates it based on the given config.
         * @param sprite The sprite to add an Animation component to.
         * @param config The animation config.
         */
        function BuildGameObjectAnimation(sprite: Phaser.GameObjects.Sprite, config: object): Phaser.GameObjects.Sprite;

        namespace Components {
            /**
             * Provides methods used for setting the alpha properties of a Game Object.
             * Should be applied as a mixin and not used directly.
             */
            interface Alpha {
                /**
                 * Clears all alpha values associated with this Game Object.
                 * 
                 * Immediately sets the alpha levels back to 1 (fully opaque).
                 */
                clearAlpha(): this;
                /**
                 * Set the Alpha level of this Game Object. The alpha controls the opacity of the Game Object as it renders.
                 * Alpha values are provided as a float between 0, fully transparent, and 1, fully opaque.
                 * 
                 * If your game is running under WebGL you can optionally specify four different alpha values, each of which
                 * correspond to the four corners of the Game Object. Under Canvas only the `topLeft` value given is used.
                 * @param topLeft The alpha value used for the top-left of the Game Object. If this is the only value given it's applied across the whole Game Object. Default 1.
                 * @param topRight The alpha value used for the top-right of the Game Object. WebGL only.
                 * @param bottomLeft The alpha value used for the bottom-left of the Game Object. WebGL only.
                 * @param bottomRight The alpha value used for the bottom-right of the Game Object. WebGL only.
                 */
                setAlpha(topLeft?: number, topRight?: number, bottomLeft?: number, bottomRight?: number): this;
                /**
                 * The alpha value of the Game Object.
                 * 
                 * This is a global value, impacting the entire Game Object, not just a region of it.
                 */
                alpha: number;
                /**
                 * The alpha value starting from the top-left of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 */
                alphaTopLeft: number;
                /**
                 * The alpha value starting from the top-right of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 */
                alphaTopRight: number;
                /**
                 * The alpha value starting from the bottom-left of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 */
                alphaBottomLeft: number;
                /**
                 * The alpha value starting from the bottom-right of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 */
                alphaBottomRight: number;
            }

            /**
             * Provides methods used for setting the alpha property of a Game Object.
             * Should be applied as a mixin and not used directly.
             */
            interface AlphaSingle {
                /**
                 * Clears all alpha values associated with this Game Object.
                 * 
                 * Immediately sets the alpha levels back to 1 (fully opaque).
                 */
                clearAlpha(): this;
                /**
                 * Set the Alpha level of this Game Object. The alpha controls the opacity of the Game Object as it renders.
                 * Alpha values are provided as a float between 0, fully transparent, and 1, fully opaque.
                 * @param value The alpha value applied across the whole Game Object. Default 1.
                 */
                setAlpha(value?: number): this;
                /**
                 * The alpha value of the Game Object.
                 * 
                 * This is a global value, impacting the entire Game Object, not just a region of it.
                 */
                alpha: number;
            }

            /**
             * Provides methods used for setting the blend mode of a Game Object.
             * Should be applied as a mixin and not used directly.
             */
            interface BlendMode {
                /**
                 * Sets the Blend Mode being used by this Game Object.
                 * 
                 * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
                 * 
                 * Under WebGL only the following Blend Modes are available:
                 * 
                 * * ADD
                 * * MULTIPLY
                 * * SCREEN
                 * * ERASE
                 * 
                 * Canvas has more available depending on browser support.
                 * 
                 * You can also create your own custom Blend Modes in WebGL.
                 * 
                 * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
                 * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
                 * reasons try to be careful about the construction of your Scene and the frequency of which blend modes
                 * are used.
                 */
                blendMode: Phaser.BlendModes | string;
                /**
                 * Sets the Blend Mode being used by this Game Object.
                 * 
                 * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
                 * 
                 * Under WebGL only the following Blend Modes are available:
                 * 
                 * * ADD
                 * * MULTIPLY
                 * * SCREEN
                 * * ERASE (only works when rendering to a framebuffer, like a Render Texture)
                 * 
                 * Canvas has more available depending on browser support.
                 * 
                 * You can also create your own custom Blend Modes in WebGL.
                 * 
                 * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
                 * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
                 * reasons try to be careful about the construction of your Scene and the frequency in which blend modes
                 * are used.
                 * @param value The BlendMode value. Either a string or a CONST.
                 */
                setBlendMode(value: string | Phaser.BlendModes): this;
            }

            /**
             * Provides methods used for calculating and setting the size of a non-Frame based Game Object.
             * Should be applied as a mixin and not used directly.
             */
            interface ComputedSize {
                /**
                 * The native (un-scaled) width of this Game Object.
                 * 
                 * Changing this value will not change the size that the Game Object is rendered in-game.
                 * For that you need to either set the scale of the Game Object (`setScale`) or use
                 * the `displayWidth` property.
                 */
                width: number;
                /**
                 * The native (un-scaled) height of this Game Object.
                 * 
                 * Changing this value will not change the size that the Game Object is rendered in-game.
                 * For that you need to either set the scale of the Game Object (`setScale`) or use
                 * the `displayHeight` property.
                 */
                height: number;
                /**
                 * The displayed width of this Game Object.
                 * 
                 * This value takes into account the scale factor.
                 * 
                 * Setting this value will adjust the Game Object's scale property.
                 */
                displayWidth: number;
                /**
                 * The displayed height of this Game Object.
                 * 
                 * This value takes into account the scale factor.
                 * 
                 * Setting this value will adjust the Game Object's scale property.
                 */
                displayHeight: number;
                /**
                 * Sets the internal size of this Game Object, as used for frame or physics body creation.
                 * 
                 * This will not change the size that the Game Object is rendered in-game.
                 * For that you need to either set the scale of the Game Object (`setScale`) or call the
                 * `setDisplaySize` method, which is the same thing as changing the scale but allows you
                 * to do so by giving pixel values.
                 * 
                 * If you have enabled this Game Object for input, changing the size will _not_ change the
                 * size of the hit area. To do this you should adjust the `input.hitArea` object directly.
                 * @param width The width of this Game Object.
                 * @param height The height of this Game Object.
                 */
                setSize(width: number, height: number): this;
                /**
                 * Sets the display size of this Game Object.
                 * 
                 * Calling this will adjust the scale.
                 * @param width The width of this Game Object.
                 * @param height The height of this Game Object.
                 */
                setDisplaySize(width: number, height: number): this;
            }

            /**
             * Provides methods used for getting and setting the texture of a Game Object.
             */
            interface Crop {
                /**
                 * The Texture this Game Object is using to render with.
                 */
                texture: Phaser.Textures.Texture | Phaser.Textures.CanvasTexture;
                /**
                 * The Texture Frame this Game Object is using to render with.
                 */
                frame: Phaser.Textures.Frame;
                /**
                 * A boolean flag indicating if this Game Object is being cropped or not.
                 * You can toggle this at any time after `setCrop` has been called, to turn cropping on or off.
                 * Equally, calling `setCrop` with no arguments will reset the crop and disable it.
                 */
                isCropped: boolean;
                /**
                 * Applies a crop to a texture based Game Object, such as a Sprite or Image.
                 * 
                 * The crop is a rectangle that limits the area of the texture frame that is visible during rendering.
                 * 
                 * Cropping a Game Object does not change its size, dimensions, physics body or hit area, it just
                 * changes what is shown when rendered.
                 * 
                 * The crop coordinates are relative to the texture frame, not the Game Object, meaning 0 x 0 is the top-left.
                 * 
                 * Therefore, if you had a Game Object that had an 800x600 sized texture, and you wanted to show only the left
                 * half of it, you could call `setCrop(0, 0, 400, 600)`.
                 * 
                 * It is also scaled to match the Game Object scale automatically. Therefore a crop rect of 100x50 would crop
                 * an area of 200x100 when applied to a Game Object that had a scale factor of 2.
                 * 
                 * You can either pass in numeric values directly, or you can provide a single Rectangle object as the first argument.
                 * 
                 * Call this method with no arguments at all to reset the crop, or toggle the property `isCropped` to `false`.
                 * 
                 * You should do this if the crop rectangle becomes the same size as the frame itself, as it will allow
                 * the renderer to skip several internal calculations.
                 * @param x The x coordinate to start the crop from. Or a Phaser.Geom.Rectangle object, in which case the rest of the arguments are ignored.
                 * @param y The y coordinate to start the crop from.
                 * @param width The width of the crop rectangle in pixels.
                 * @param height The height of the crop rectangle in pixels.
                 */
                setCrop(x?: number | Phaser.Geom.Rectangle, y?: number, width?: number, height?: number): this;
            }

            /**
             * Provides methods used for setting the depth of a Game Object.
             * Should be applied as a mixin and not used directly.
             */
            interface Depth {
                /**
                 * The depth of this Game Object within the Scene.
                 * 
                 * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
                 * of Game Objects, without actually moving their position in the display list.
                 * 
                 * The default depth is zero. A Game Object with a higher depth
                 * value will always render in front of one with a lower value.
                 * 
                 * Setting the depth will queue a depth sort event within the Scene.
                 */
                depth: number;
                /**
                 * The depth of this Game Object within the Scene.
                 * 
                 * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
                 * of Game Objects, without actually moving their position in the display list.
                 * 
                 * The default depth is zero. A Game Object with a higher depth
                 * value will always render in front of one with a lower value.
                 * 
                 * Setting the depth will queue a depth sort event within the Scene.
                 * @param value The depth of this Game Object.
                 */
                setDepth(value: number): this;
            }

            /**
             * Provides methods used for visually flipping a Game Object.
             * Should be applied as a mixin and not used directly.
             */
            interface Flip {
                /**
                 * The horizontally flipped state of the Game Object.
                 * 
                 * A Game Object that is flipped horizontally will render inversed on the horizontal axis.
                 * Flipping always takes place from the middle of the texture and does not impact the scale value.
                 * If this Game Object has a physics body, it will not change the body. This is a rendering toggle only.
                 */
                flipX: boolean;
                /**
                 * The vertically flipped state of the Game Object.
                 * 
                 * A Game Object that is flipped vertically will render inversed on the vertical axis (i.e. upside down)
                 * Flipping always takes place from the middle of the texture and does not impact the scale value.
                 * If this Game Object has a physics body, it will not change the body. This is a rendering toggle only.
                 */
                flipY: boolean;
                /**
                 * Toggles the horizontal flipped state of this Game Object.
                 * 
                 * A Game Object that is flipped horizontally will render inversed on the horizontal axis.
                 * Flipping always takes place from the middle of the texture and does not impact the scale value.
                 * If this Game Object has a physics body, it will not change the body. This is a rendering toggle only.
                 */
                toggleFlipX(): this;
                /**
                 * Toggles the vertical flipped state of this Game Object.
                 */
                toggleFlipY(): this;
                /**
                 * Sets the horizontal flipped state of this Game Object.
                 * 
                 * A Game Object that is flipped horizontally will render inversed on the horizontal axis.
                 * Flipping always takes place from the middle of the texture and does not impact the scale value.
                 * If this Game Object has a physics body, it will not change the body. This is a rendering toggle only.
                 * @param value The flipped state. `false` for no flip, or `true` to be flipped.
                 */
                setFlipX(value: boolean): this;
                /**
                 * Sets the vertical flipped state of this Game Object.
                 * @param value The flipped state. `false` for no flip, or `true` to be flipped.
                 */
                setFlipY(value: boolean): this;
                /**
                 * Sets the horizontal and vertical flipped state of this Game Object.
                 * 
                 * A Game Object that is flipped will render inversed on the flipped axis.
                 * Flipping always takes place from the middle of the texture and does not impact the scale value.
                 * If this Game Object has a physics body, it will not change the body. This is a rendering toggle only.
                 * @param x The horizontal flipped state. `false` for no flip, or `true` to be flipped.
                 * @param y The horizontal flipped state. `false` for no flip, or `true` to be flipped.
                 */
                setFlip(x: boolean, y: boolean): this;
                /**
                 * Resets the horizontal and vertical flipped state of this Game Object back to their default un-flipped state.
                 */
                resetFlip(): this;
            }

            /**
             * Provides methods used for obtaining the bounds of a Game Object.
             * Should be applied as a mixin and not used directly.
             */
            interface GetBounds {
                /**
                 * Gets the center coordinate of this Game Object, regardless of origin.
                 * The returned point is calculated in local space and does not factor in any parent containers
                 * @param output An object to store the values in. If not provided a new Vector2 will be created.
                 */
                getCenter<O extends Phaser.Math.Vector2>(output?: O): O;
                /**
                 * Gets the top-left corner coordinate of this Game Object, regardless of origin.
                 * The returned point is calculated in local space and does not factor in any parent containers
                 * @param output An object to store the values in. If not provided a new Vector2 will be created.
                 * @param includeParent If this Game Object has a parent Container, include it (and all other ancestors) in the resulting vector? Default false.
                 */
                getTopLeft<O extends Phaser.Math.Vector2>(output?: O, includeParent?: boolean): O;
                /**
                 * Gets the top-center coordinate of this Game Object, regardless of origin.
                 * The returned point is calculated in local space and does not factor in any parent containers
                 * @param output An object to store the values in. If not provided a new Vector2 will be created.
                 * @param includeParent If this Game Object has a parent Container, include it (and all other ancestors) in the resulting vector? Default false.
                 */
                getTopCenter<O extends Phaser.Math.Vector2>(output?: O, includeParent?: boolean): O;
                /**
                 * Gets the top-right corner coordinate of this Game Object, regardless of origin.
                 * The returned point is calculated in local space and does not factor in any parent containers
                 * @param output An object to store the values in. If not provided a new Vector2 will be created.
                 * @param includeParent If this Game Object has a parent Container, include it (and all other ancestors) in the resulting vector? Default false.
                 */
                getTopRight<O extends Phaser.Math.Vector2>(output?: O, includeParent?: boolean): O;
                /**
                 * Gets the left-center coordinate of this Game Object, regardless of origin.
                 * The returned point is calculated in local space and does not factor in any parent containers
                 * @param output An object to store the values in. If not provided a new Vector2 will be created.
                 * @param includeParent If this Game Object has a parent Container, include it (and all other ancestors) in the resulting vector? Default false.
                 */
                getLeftCenter<O extends Phaser.Math.Vector2>(output?: O, includeParent?: boolean): O;
                /**
                 * Gets the right-center coordinate of this Game Object, regardless of origin.
                 * The returned point is calculated in local space and does not factor in any parent containers
                 * @param output An object to store the values in. If not provided a new Vector2 will be created.
                 * @param includeParent If this Game Object has a parent Container, include it (and all other ancestors) in the resulting vector? Default false.
                 */
                getRightCenter<O extends Phaser.Math.Vector2>(output?: O, includeParent?: boolean): O;
                /**
                 * Gets the bottom-left corner coordinate of this Game Object, regardless of origin.
                 * The returned point is calculated in local space and does not factor in any parent containers
                 * @param output An object to store the values in. If not provided a new Vector2 will be created.
                 * @param includeParent If this Game Object has a parent Container, include it (and all other ancestors) in the resulting vector? Default false.
                 */
                getBottomLeft<O extends Phaser.Math.Vector2>(output?: O, includeParent?: boolean): O;
                /**
                 * Gets the bottom-center coordinate of this Game Object, regardless of origin.
                 * The returned point is calculated in local space and does not factor in any parent containers
                 * @param output An object to store the values in. If not provided a new Vector2 will be created.
                 * @param includeParent If this Game Object has a parent Container, include it (and all other ancestors) in the resulting vector? Default false.
                 */
                getBottomCenter<O extends Phaser.Math.Vector2>(output?: O, includeParent?: boolean): O;
                /**
                 * Gets the bottom-right corner coordinate of this Game Object, regardless of origin.
                 * The returned point is calculated in local space and does not factor in any parent containers
                 * @param output An object to store the values in. If not provided a new Vector2 will be created.
                 * @param includeParent If this Game Object has a parent Container, include it (and all other ancestors) in the resulting vector? Default false.
                 */
                getBottomRight<O extends Phaser.Math.Vector2>(output?: O, includeParent?: boolean): O;
                /**
                 * Gets the bounds of this Game Object, regardless of origin.
                 * The values are stored and returned in a Rectangle, or Rectangle-like, object.
                 * @param output An object to store the values in. If not provided a new Rectangle will be created.
                 */
                getBounds<O extends Phaser.Geom.Rectangle>(output?: O): O;
            }

            /**
             * Provides methods used for getting and setting the mask of a Game Object.
             */
            interface Mask {
                /**
                 * The Mask this Game Object is using during render.
                 */
                mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask;
                /**
                 * Sets the mask that this Game Object will use to render with.
                 * 
                 * The mask must have been previously created and can be either a GeometryMask or a BitmapMask.
                 * Note: Bitmap Masks only work on WebGL. Geometry Masks work on both WebGL and Canvas.
                 * 
                 * If a mask is already set on this Game Object it will be immediately replaced.
                 * 
                 * Masks are positioned in global space and are not relative to the Game Object to which they
                 * are applied. The reason for this is that multiple Game Objects can all share the same mask.
                 * 
                 * Masks have no impact on physics or input detection. They are purely a rendering component
                 * that allows you to limit what is visible during the render pass.
                 * @param mask The mask this Game Object will use when rendering.
                 */
                setMask(mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask): this;
                /**
                 * Clears the mask that this Game Object was using.
                 * @param destroyMask Destroy the mask before clearing it? Default false.
                 */
                clearMask(destroyMask?: boolean): this;
                /**
                 * Creates and returns a Bitmap Mask. This mask can be used by any Game Object,
                 * including this one.
                 * 
                 * Note: Bitmap Masks only work on WebGL. Geometry Masks work on both WebGL and Canvas.
                 * 
                 * To create the mask you need to pass in a reference to a renderable Game Object.
                 * A renderable Game Object is one that uses a texture to render with, such as an
                 * Image, Sprite, Render Texture or BitmapText.
                 * 
                 * If you do not provide a renderable object, and this Game Object has a texture,
                 * it will use itself as the object. This means you can call this method to create
                 * a Bitmap Mask from any renderable Game Object.
                 * @param renderable A renderable Game Object that uses a texture, such as a Sprite.
                 */
                createBitmapMask(renderable?: Phaser.GameObjects.GameObject): Phaser.Display.Masks.BitmapMask;
                /**
                 * Creates and returns a Geometry Mask. This mask can be used by any Game Object,
                 * including this one.
                 * 
                 * To create the mask you need to pass in a reference to a Graphics Game Object.
                 * 
                 * If you do not provide a graphics object, and this Game Object is an instance
                 * of a Graphics object, then it will use itself to create the mask.
                 * 
                 * This means you can call this method to create a Geometry Mask from any Graphics Game Object.
                 * @param graphics A Graphics Game Object. The geometry within it will be used as the mask.
                 */
                createGeometryMask(graphics?: Phaser.GameObjects.Graphics): Phaser.Display.Masks.GeometryMask;
            }

            /**
             * Provides methods used for getting and setting the origin of a Game Object.
             * Values are normalized, given in the range 0 to 1.
             * Display values contain the calculated pixel values.
             * Should be applied as a mixin and not used directly.
             */
            interface Origin {
                /**
                 * The horizontal origin of this Game Object.
                 * The origin maps the relationship between the size and position of the Game Object.
                 * The default value is 0.5, meaning all Game Objects are positioned based on their center.
                 * Setting the value to 0 means the position now relates to the left of the Game Object.
                 */
                originX: number;
                /**
                 * The vertical origin of this Game Object.
                 * The origin maps the relationship between the size and position of the Game Object.
                 * The default value is 0.5, meaning all Game Objects are positioned based on their center.
                 * Setting the value to 0 means the position now relates to the top of the Game Object.
                 */
                originY: number;
                /**
                 * The horizontal display origin of this Game Object.
                 * The origin is a normalized value between 0 and 1.
                 * The displayOrigin is a pixel value, based on the size of the Game Object combined with the origin.
                 */
                displayOriginX: number;
                /**
                 * The vertical display origin of this Game Object.
                 * The origin is a normalized value between 0 and 1.
                 * The displayOrigin is a pixel value, based on the size of the Game Object combined with the origin.
                 */
                displayOriginY: number;
                /**
                 * Sets the origin of this Game Object.
                 * 
                 * The values are given in the range 0 to 1.
                 * @param x The horizontal origin value. Default 0.5.
                 * @param y The vertical origin value. If not defined it will be set to the value of `x`. Default x.
                 */
                setOrigin(x?: number, y?: number): this;
                /**
                 * Sets the origin of this Game Object based on the Pivot values in its Frame.
                 */
                setOriginFromFrame(): this;
                /**
                 * Sets the display origin of this Game Object.
                 * The difference between this and setting the origin is that you can use pixel values for setting the display origin.
                 * @param x The horizontal display origin value. Default 0.
                 * @param y The vertical display origin value. If not defined it will be set to the value of `x`. Default x.
                 */
                setDisplayOrigin(x?: number, y?: number): this;
                /**
                 * Updates the Display Origin cached values internally stored on this Game Object.
                 * You don't usually call this directly, but it is exposed for edge-cases where you may.
                 */
                updateDisplayOrigin(): this;
            }

            /**
             * Provides methods used for managing a Game Object following a Path.
             * Should be applied as a mixin and not used directly.
             */
            interface PathFollower {
                /**
                 * The Path this PathFollower is following. It can only follow one Path at a time.
                 */
                path: Phaser.Curves.Path;
                /**
                 * Should the PathFollower automatically rotate to point in the direction of the Path?
                 */
                rotateToPath: boolean;
                /**
                 * Set the Path that this PathFollower should follow.
                 * 
                 * Optionally accepts {@link Phaser.Types.GameObjects.PathFollower.PathConfig} settings.
                 * @param path The Path this PathFollower is following. It can only follow one Path at a time.
                 * @param config Settings for the PathFollower.
                 */
                setPath(path: Phaser.Curves.Path, config?: number | Phaser.Types.GameObjects.PathFollower.PathConfig | Phaser.Types.Tweens.NumberTweenBuilderConfig): this;
                /**
                 * Set whether the PathFollower should automatically rotate to point in the direction of the Path.
                 * @param value Whether the PathFollower should automatically rotate to point in the direction of the Path.
                 * @param offset Rotation offset in degrees. Default 0.
                 */
                setRotateToPath(value: boolean, offset?: number): this;
                /**
                 * Is this PathFollower actively following a Path or not?
                 * 
                 * To be considered as `isFollowing` it must be currently moving on a Path, and not paused.
                 */
                isFollowing(): boolean;
                /**
                 * Starts this PathFollower following its given Path.
                 * @param config The duration of the follow, or a PathFollower config object. Default {}.
                 * @param startAt Optional start position of the follow, between 0 and 1. Default 0.
                 */
                startFollow(config?: number | Phaser.Types.GameObjects.PathFollower.PathConfig | Phaser.Types.Tweens.NumberTweenBuilderConfig, startAt?: number): this;
                /**
                 * Pauses this PathFollower. It will still continue to render, but it will remain motionless at the
                 * point on the Path at which you paused it.
                 */
                pauseFollow(): this;
                /**
                 * Resumes a previously paused PathFollower.
                 * 
                 * If the PathFollower was not paused this has no effect.
                 */
                resumeFollow(): this;
                /**
                 * Stops this PathFollower from following the path any longer.
                 * 
                 * This will invoke any 'stop' conditions that may exist on the Path, or for the follower.
                 */
                stopFollow(): this;
                /**
                 * Internal update handler that advances this PathFollower along the path.
                 * 
                 * Called automatically by the Scene step, should not typically be called directly.
                 */
                pathUpdate(): void;
            }

            /**
             * Provides methods used for setting the WebGL rendering pipeline of a Game Object.
             */
            interface Pipeline {
                /**
                 * The initial WebGL pipeline of this Game Object.
                 * 
                 * If you call `resetPipeline` on this Game Object, the pipeline is reset to this default.
                 */
                defaultPipeline: Phaser.Renderer.WebGL.WebGLPipeline;
                /**
                 * The current WebGL pipeline of this Game Object.
                 */
                pipeline: Phaser.Renderer.WebGL.WebGLPipeline;
                /**
                 * Does this Game Object have any Post Pipelines set?
                 */
                hasPostPipeline: boolean;
                /**
                 * The WebGL Post FX Pipelines this Game Object uses for post-render effects.
                 * 
                 * The pipelines are processed in the order in which they appear in this array.
                 * 
                 * If you modify this array directly, be sure to set the
                 * `hasPostPipeline` property accordingly.
                 */
                postPipelines: Phaser.Renderer.WebGL.Pipelines.PostFXPipeline[];
                /**
                 * An object to store pipeline specific data in, to be read by the pipelines this Game Object uses.
                 */
                pipelineData: object;
                /**
                 * Sets the initial WebGL Pipeline of this Game Object.
                 * 
                 * This should only be called during the instantiation of the Game Object. After that, use `setPipeline`.
                 * @param pipeline Either the string-based name of the pipeline, or a pipeline instance to set.
                 */
                initPipeline(pipeline: string | Phaser.Renderer.WebGL.WebGLPipeline): boolean;
                /**
                 * Sets the main WebGL Pipeline of this Game Object.
                 * 
                 * Also sets the `pipelineData` property, if the parameter is given.
                 * 
                 * Both the pipeline and post pipelines share the same pipeline data object.
                 * @param pipeline Either the string-based name of the pipeline, or a pipeline instance to set.
                 * @param pipelineData Optional pipeline data object that is _deep copied_ into the `pipelineData` property of this Game Object.
                 * @param copyData Should the pipeline data object be _deep copied_ into the `pipelineData` property of this Game Object? If `false` it will be set by reference instead. Default true.
                 */
                setPipeline(pipeline: string | Phaser.Renderer.WebGL.WebGLPipeline, pipelineData?: object, copyData?: boolean): this;
                /**
                 * Sets one, or more, Post Pipelines on this Game Object.
                 * 
                 * Post Pipelines are invoked after this Game Object has rendered to its target and
                 * are commonly used for post-fx.
                 * 
                 * The post pipelines are appended to the `postPipelines` array belonging to this
                 * Game Object. When the renderer processes this Game Object, it iterates through the post
                 * pipelines in the order in which they appear in the array. If you are stacking together
                 * multiple effects, be aware that the order is important.
                 * 
                 * If you call this method multiple times, the new pipelines will be appended to any existing
                 * post pipelines already set. Use the `resetPostPipeline` method to clear them first, if required.
                 * 
                 * You can optionally also sets the `pipelineData` property, if the parameter is given.
                 * 
                 * Both the pipeline and post pipelines share the pipeline data object together.
                 * @param pipelines Either the string-based name of the pipeline, or a pipeline instance, or class, or an array of them.
                 * @param pipelineData Optional pipeline data object that is _deep copied_ into the `pipelineData` property of this Game Object.
                 * @param copyData Should the pipeline data object be _deep copied_ into the `pipelineData` property of this Game Object? If `false` it will be set by reference instead. Default true.
                 */
                setPostPipeline(pipelines: string | string[] | Function | Function[] | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline[], pipelineData?: object, copyData?: boolean): this;
                /**
                 * Adds an entry to the `pipelineData` object belonging to this Game Object.
                 * 
                 * If the 'key' already exists, its value is updated. If it doesn't exist, it is created.
                 * 
                 * If `value` is undefined, and `key` exists, `key` is removed from the data object.
                 * 
                 * Both the pipeline and post pipelines share the pipeline data object together.
                 * @param key The key of the pipeline data to set, update, or delete.
                 * @param value The value to be set with the key. If `undefined` then `key` will be deleted from the object.
                 */
                setPipelineData(key: string, value?: any): this;
                /**
                 * Gets a Post Pipeline instance from this Game Object, based on the given name, and returns it.
                 * @param pipeline The string-based name of the pipeline, or a pipeline class.
                 */
                getPostPipeline(pipeline: string | Function | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline): Phaser.Renderer.WebGL.Pipelines.PostFXPipeline | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline[];
                /**
                 * Resets the WebGL Pipeline of this Game Object back to the default it was created with.
                 * @param resetPostPipelines Reset all of the post pipelines? Default false.
                 * @param resetData Reset the `pipelineData` object to being an empty object? Default false.
                 */
                resetPipeline(resetPostPipelines?: boolean, resetData?: boolean): boolean;
                /**
                 * Resets the WebGL Post Pipelines of this Game Object. It does this by calling
                 * the `destroy` method on each post pipeline and then clearing the local array.
                 * @param resetData Reset the `pipelineData` object to being an empty object? Default false.
                 */
                resetPostPipeline(resetData?: boolean): void;
                /**
                 * Removes a type of Post Pipeline instances from this Game Object, based on the given name, and destroys them.
                 * 
                 * If you wish to remove all Post Pipelines use the `resetPostPipeline` method instead.
                 * @param pipeline The string-based name of the pipeline, or a pipeline class.
                 */
                removePostPipeline(pipeline: string | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline): this;
                /**
                 * Gets the name of the WebGL Pipeline this Game Object is currently using.
                 */
                getPipelineName(): string;
            }

            /**
             * Provides methods used for getting and setting the Scroll Factor of a Game Object.
             */
            interface ScrollFactor {
                /**
                 * The horizontal scroll factor of this Game Object.
                 * 
                 * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
                 * 
                 * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
                 * It does not change the Game Objects actual position values.
                 * 
                 * A value of 1 means it will move exactly in sync with a camera.
                 * A value of 0 means it will not move at all, even if the camera moves.
                 * Other values control the degree to which the camera movement is mapped to this Game Object.
                 * 
                 * Please be aware that scroll factor values other than 1 are not taken in to consideration when
                 * calculating physics collisions. Bodies always collide based on their world position, but changing
                 * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
                 * them from physics bodies if not accounted for in your code.
                 */
                scrollFactorX: number;
                /**
                 * The vertical scroll factor of this Game Object.
                 * 
                 * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
                 * 
                 * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
                 * It does not change the Game Objects actual position values.
                 * 
                 * A value of 1 means it will move exactly in sync with a camera.
                 * A value of 0 means it will not move at all, even if the camera moves.
                 * Other values control the degree to which the camera movement is mapped to this Game Object.
                 * 
                 * Please be aware that scroll factor values other than 1 are not taken in to consideration when
                 * calculating physics collisions. Bodies always collide based on their world position, but changing
                 * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
                 * them from physics bodies if not accounted for in your code.
                 */
                scrollFactorY: number;
                /**
                 * Sets the scroll factor of this Game Object.
                 * 
                 * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
                 * 
                 * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
                 * It does not change the Game Objects actual position values.
                 * 
                 * A value of 1 means it will move exactly in sync with a camera.
                 * A value of 0 means it will not move at all, even if the camera moves.
                 * Other values control the degree to which the camera movement is mapped to this Game Object.
                 * 
                 * Please be aware that scroll factor values other than 1 are not taken in to consideration when
                 * calculating physics collisions. Bodies always collide based on their world position, but changing
                 * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
                 * them from physics bodies if not accounted for in your code.
                 * @param x The horizontal scroll factor of this Game Object.
                 * @param y The vertical scroll factor of this Game Object. If not set it will use the `x` value. Default x.
                 */
                setScrollFactor(x: number, y?: number): this;
            }

            /**
             * Provides methods used for getting and setting the size of a Game Object.
             */
            interface Size {
                /**
                 * The native (un-scaled) width of this Game Object.
                 * 
                 * Changing this value will not change the size that the Game Object is rendered in-game.
                 * For that you need to either set the scale of the Game Object (`setScale`) or use
                 * the `displayWidth` property.
                 */
                width: number;
                /**
                 * The native (un-scaled) height of this Game Object.
                 * 
                 * Changing this value will not change the size that the Game Object is rendered in-game.
                 * For that you need to either set the scale of the Game Object (`setScale`) or use
                 * the `displayHeight` property.
                 */
                height: number;
                /**
                 * The displayed width of this Game Object.
                 * 
                 * This value takes into account the scale factor.
                 * 
                 * Setting this value will adjust the Game Object's scale property.
                 */
                displayWidth: number;
                /**
                 * The displayed height of this Game Object.
                 * 
                 * This value takes into account the scale factor.
                 * 
                 * Setting this value will adjust the Game Object's scale property.
                 */
                displayHeight: number;
                /**
                 * Sets the size of this Game Object to be that of the given Frame.
                 * 
                 * This will not change the size that the Game Object is rendered in-game.
                 * For that you need to either set the scale of the Game Object (`setScale`) or call the
                 * `setDisplaySize` method, which is the same thing as changing the scale but allows you
                 * to do so by giving pixel values.
                 * 
                 * If you have enabled this Game Object for input, changing the size will _not_ change the
                 * size of the hit area. To do this you should adjust the `input.hitArea` object directly.
                 * @param frame The frame to base the size of this Game Object on.
                 */
                setSizeToFrame(frame: Phaser.Textures.Frame): this;
                /**
                 * Sets the internal size of this Game Object, as used for frame or physics body creation.
                 * 
                 * This will not change the size that the Game Object is rendered in-game.
                 * For that you need to either set the scale of the Game Object (`setScale`) or call the
                 * `setDisplaySize` method, which is the same thing as changing the scale but allows you
                 * to do so by giving pixel values.
                 * 
                 * If you have enabled this Game Object for input, changing the size will _not_ change the
                 * size of the hit area. To do this you should adjust the `input.hitArea` object directly.
                 * @param width The width of this Game Object.
                 * @param height The height of this Game Object.
                 */
                setSize(width: number, height: number): this;
                /**
                 * Sets the display size of this Game Object.
                 * 
                 * Calling this will adjust the scale.
                 * @param width The width of this Game Object.
                 * @param height The height of this Game Object.
                 */
                setDisplaySize(width: number, height: number): this;
            }

            /**
             * Provides methods used for getting and setting the texture of a Game Object.
             */
            interface Texture {
                /**
                 * The Texture this Game Object is using to render with.
                 */
                texture: Phaser.Textures.Texture | Phaser.Textures.CanvasTexture;
                /**
                 * The Texture Frame this Game Object is using to render with.
                 */
                frame: Phaser.Textures.Frame;
                /**
                 * Sets the texture and frame this Game Object will use to render with.
                 * 
                 * Textures are referenced by their string-based keys, as stored in the Texture Manager.
                 * @param key The key of the texture to be used, as stored in the Texture Manager, or a Texture instance.
                 * @param frame The name or index of the frame within the Texture.
                 */
                setTexture(key: string | Phaser.Textures.Texture, frame?: string | number): this;
                /**
                 * Sets the frame this Game Object will use to render with.
                 * 
                 * The Frame has to belong to the current Texture being used.
                 * 
                 * It can be either a string or an index.
                 * 
                 * Calling `setFrame` will modify the `width` and `height` properties of your Game Object.
                 * It will also change the `origin` if the Frame has a custom pivot point, as exported from packages like Texture Packer.
                 * @param frame The name or index of the frame within the Texture.
                 * @param updateSize Should this call adjust the size of the Game Object? Default true.
                 * @param updateOrigin Should this call adjust the origin of the Game Object? Default true.
                 */
                setFrame(frame: string | number, updateSize?: boolean, updateOrigin?: boolean): this;
            }

            /**
             * Provides methods used for getting and setting the texture of a Game Object.
             */
            interface TextureCrop {
                /**
                 * The Texture this Game Object is using to render with.
                 */
                texture: Phaser.Textures.Texture | Phaser.Textures.CanvasTexture;
                /**
                 * The Texture Frame this Game Object is using to render with.
                 */
                frame: Phaser.Textures.Frame;
                /**
                 * A boolean flag indicating if this Game Object is being cropped or not.
                 * You can toggle this at any time after `setCrop` has been called, to turn cropping on or off.
                 * Equally, calling `setCrop` with no arguments will reset the crop and disable it.
                 */
                isCropped: boolean;
                /**
                 * Applies a crop to a texture based Game Object, such as a Sprite or Image.
                 * 
                 * The crop is a rectangle that limits the area of the texture frame that is visible during rendering.
                 * 
                 * Cropping a Game Object does not change its size, dimensions, physics body or hit area, it just
                 * changes what is shown when rendered.
                 * 
                 * The crop coordinates are relative to the texture frame, not the Game Object, meaning 0 x 0 is the top-left.
                 * 
                 * Therefore, if you had a Game Object that had an 800x600 sized texture, and you wanted to show only the left
                 * half of it, you could call `setCrop(0, 0, 400, 600)`.
                 * 
                 * It is also scaled to match the Game Object scale automatically. Therefore a crop rect of 100x50 would crop
                 * an area of 200x100 when applied to a Game Object that had a scale factor of 2.
                 * 
                 * You can either pass in numeric values directly, or you can provide a single Rectangle object as the first argument.
                 * 
                 * Call this method with no arguments at all to reset the crop, or toggle the property `isCropped` to `false`.
                 * 
                 * You should do this if the crop rectangle becomes the same size as the frame itself, as it will allow
                 * the renderer to skip several internal calculations.
                 * @param x The x coordinate to start the crop from. Or a Phaser.Geom.Rectangle object, in which case the rest of the arguments are ignored.
                 * @param y The y coordinate to start the crop from.
                 * @param width The width of the crop rectangle in pixels.
                 * @param height The height of the crop rectangle in pixels.
                 */
                setCrop(x?: number | Phaser.Geom.Rectangle, y?: number, width?: number, height?: number): this;
                /**
                 * Sets the texture and frame this Game Object will use to render with.
                 * 
                 * Textures are referenced by their string-based keys, as stored in the Texture Manager.
                 * @param key The key of the texture to be used, as stored in the Texture Manager.
                 * @param frame The name or index of the frame within the Texture.
                 */
                setTexture(key: string, frame?: string | number): this;
                /**
                 * Sets the frame this Game Object will use to render with.
                 * 
                 * The Frame has to belong to the current Texture being used.
                 * 
                 * It can be either a string or an index.
                 * 
                 * Calling `setFrame` will modify the `width` and `height` properties of your Game Object.
                 * It will also change the `origin` if the Frame has a custom pivot point, as exported from packages like Texture Packer.
                 * @param frame The name or index of the frame within the Texture.
                 * @param updateSize Should this call adjust the size of the Game Object? Default true.
                 * @param updateOrigin Should this call adjust the origin of the Game Object? Default true.
                 */
                setFrame(frame: string | number, updateSize?: boolean, updateOrigin?: boolean): this;
            }

            /**
             * Provides methods used for setting the tint of a Game Object.
             * Should be applied as a mixin and not used directly.
             */
            interface Tint {
                /**
                 * The tint value being applied to the top-left vertice of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 * The value should be set as a hex number, i.e. 0xff0000 for red, or 0xff00ff for purple.
                 */
                tintTopLeft: number;
                /**
                 * The tint value being applied to the top-right vertice of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 * The value should be set as a hex number, i.e. 0xff0000 for red, or 0xff00ff for purple.
                 */
                tintTopRight: number;
                /**
                 * The tint value being applied to the bottom-left vertice of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 * The value should be set as a hex number, i.e. 0xff0000 for red, or 0xff00ff for purple.
                 */
                tintBottomLeft: number;
                /**
                 * The tint value being applied to the bottom-right vertice of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 * The value should be set as a hex number, i.e. 0xff0000 for red, or 0xff00ff for purple.
                 */
                tintBottomRight: number;
                /**
                 * The tint fill mode.
                 * 
                 * `false` = An additive tint (the default), where vertices colors are blended with the texture.
                 * `true` = A fill tint, where the vertices colors replace the texture, but respects texture alpha.
                 */
                tintFill: boolean;
                /**
                 * Clears all tint values associated with this Game Object.
                 * 
                 * Immediately sets the color values back to 0xffffff and the tint type to 'additive',
                 * which results in no visible change to the texture.
                 */
                clearTint(): this;
                /**
                 * Sets an additive tint on this Game Object.
                 * 
                 * The tint works by taking the pixel color values from the Game Objects texture, and then
                 * multiplying it by the color value of the tint. You can provide either one color value,
                 * in which case the whole Game Object will be tinted in that color. Or you can provide a color
                 * per corner. The colors are blended together across the extent of the Game Object.
                 * 
                 * To modify the tint color once set, either call this method again with new values or use the
                 * `tint` property to set all colors at once. Or, use the properties `tintTopLeft`, `tintTopRight,
                 * `tintBottomLeft` and `tintBottomRight` to set the corner color values independently.
                 * 
                 * To remove a tint call `clearTint`.
                 * 
                 * To swap this from being an additive tint to a fill based tint set the property `tintFill` to `true`.
                 * @param topLeft The tint being applied to the top-left of the Game Object. If no other values are given this value is applied evenly, tinting the whole Game Object. Default 0xffffff.
                 * @param topRight The tint being applied to the top-right of the Game Object.
                 * @param bottomLeft The tint being applied to the bottom-left of the Game Object.
                 * @param bottomRight The tint being applied to the bottom-right of the Game Object.
                 */
                setTint(topLeft?: number, topRight?: number, bottomLeft?: number, bottomRight?: number): this;
                /**
                 * Sets a fill-based tint on this Game Object.
                 * 
                 * Unlike an additive tint, a fill-tint literally replaces the pixel colors from the texture
                 * with those in the tint. You can use this for effects such as making a player flash 'white'
                 * if hit by something. You can provide either one color value, in which case the whole
                 * Game Object will be rendered in that color. Or you can provide a color per corner. The colors
                 * are blended together across the extent of the Game Object.
                 * 
                 * To modify the tint color once set, either call this method again with new values or use the
                 * `tint` property to set all colors at once. Or, use the properties `tintTopLeft`, `tintTopRight,
                 * `tintBottomLeft` and `tintBottomRight` to set the corner color values independently.
                 * 
                 * To remove a tint call `clearTint`.
                 * 
                 * To swap this from being a fill-tint to an additive tint set the property `tintFill` to `false`.
                 * @param topLeft The tint being applied to the top-left of the Game Object. If not other values are given this value is applied evenly, tinting the whole Game Object. Default 0xffffff.
                 * @param topRight The tint being applied to the top-right of the Game Object.
                 * @param bottomLeft The tint being applied to the bottom-left of the Game Object.
                 * @param bottomRight The tint being applied to the bottom-right of the Game Object.
                 */
                setTintFill(topLeft?: number, topRight?: number, bottomLeft?: number, bottomRight?: number): this;
                /**
                 * The tint value being applied to the whole of the Game Object.
                 * This property is a setter-only. Use the properties `tintTopLeft` etc to read the current tint value.
                 */
                tint: number;
                /**
                 * Does this Game Object have a tint applied?
                 * 
                 * It checks to see if the 4 tint properties are set to the value 0xffffff
                 * and that the `tintFill` property is `false`. This indicates that a Game Object isn't tinted.
                 */
                readonly isTinted: boolean;
            }

            /**
             * Build a JSON representation of the given Game Object.
             * 
             * This is typically extended further by Game Object specific implementations.
             */
            interface ToJSON {
            }

            /**
             * Provides methods used for getting and setting the position, scale and rotation of a Game Object.
             */
            interface Transform {
                /**
                 * The x position of this Game Object.
                 */
                x: number;
                /**
                 * The y position of this Game Object.
                 */
                y: number;
                /**
                 * The z position of this Game Object.
                 * 
                 * Note: The z position does not control the rendering order of 2D Game Objects. Use
                 * {@link Phaser.GameObjects.Components.Depth#depth} instead.
                 */
                z: number;
                /**
                 * The w position of this Game Object.
                 */
                w: number;
                /**
                 * This is a special setter that allows you to set both the horizontal and vertical scale of this Game Object
                 * to the same value, at the same time. When reading this value the result returned is `(scaleX + scaleY) / 2`.
                 * 
                 * Use of this property implies you wish the horizontal and vertical scales to be equal to each other. If this
                 * isn't the case, use the `scaleX` or `scaleY` properties instead.
                 */
                scale: number;
                /**
                 * The horizontal scale of this Game Object.
                 */
                scaleX: number;
                /**
                 * The vertical scale of this Game Object.
                 */
                scaleY: number;
                /**
                 * The angle of this Game Object as expressed in degrees.
                 * 
                 * Phaser uses a right-hand clockwise rotation system, where 0 is right, 90 is down, 180/-180 is left
                 * and -90 is up.
                 * 
                 * If you prefer to work in radians, see the `rotation` property instead.
                 */
                angle: number;
                /**
                 * The angle of this Game Object in radians.
                 * 
                 * Phaser uses a right-hand clockwise rotation system, where 0 is right, PI/2 is down, +-PI is left
                 * and -PI/2 is up.
                 * 
                 * If you prefer to work in degrees, see the `angle` property instead.
                 */
                rotation: number;
                /**
                 * Sets the position of this Game Object.
                 * @param x The x position of this Game Object. Default 0.
                 * @param y The y position of this Game Object. If not set it will use the `x` value. Default x.
                 * @param z The z position of this Game Object. Default 0.
                 * @param w The w position of this Game Object. Default 0.
                 */
                setPosition(x?: number, y?: number, z?: number, w?: number): this;
                /**
                 * Copies an object's coordinates to this Game Object's position.
                 * @param source An object with numeric 'x', 'y', 'z', or 'w' properties. Undefined values are not copied.
                 */
                copyPosition(source: Phaser.Types.Math.Vector2Like | Phaser.Types.Math.Vector3Like | Phaser.Types.Math.Vector4Like): this;
                /**
                 * Sets the position of this Game Object to be a random position within the confines of
                 * the given area.
                 * 
                 * If no area is specified a random position between 0 x 0 and the game width x height is used instead.
                 * 
                 * The position does not factor in the size of this Game Object, meaning that only the origin is
                 * guaranteed to be within the area.
                 * @param x The x position of the top-left of the random area. Default 0.
                 * @param y The y position of the top-left of the random area. Default 0.
                 * @param width The width of the random area.
                 * @param height The height of the random area.
                 */
                setRandomPosition(x?: number, y?: number, width?: number, height?: number): this;
                /**
                 * Sets the rotation of this Game Object.
                 * @param radians The rotation of this Game Object, in radians. Default 0.
                 */
                setRotation(radians?: number): this;
                /**
                 * Sets the angle of this Game Object.
                 * @param degrees The rotation of this Game Object, in degrees. Default 0.
                 */
                setAngle(degrees?: number): this;
                /**
                 * Sets the scale of this Game Object.
                 * @param x The horizontal scale of this Game Object.
                 * @param y The vertical scale of this Game Object. If not set it will use the `x` value. Default x.
                 */
                setScale(x: number, y?: number): this;
                /**
                 * Sets the x position of this Game Object.
                 * @param value The x position of this Game Object. Default 0.
                 */
                setX(value?: number): this;
                /**
                 * Sets the y position of this Game Object.
                 * @param value The y position of this Game Object. Default 0.
                 */
                setY(value?: number): this;
                /**
                 * Sets the z position of this Game Object.
                 * 
                 * Note: The z position does not control the rendering order of 2D Game Objects. Use
                 * {@link Phaser.GameObjects.Components.Depth#setDepth} instead.
                 * @param value The z position of this Game Object. Default 0.
                 */
                setZ(value?: number): this;
                /**
                 * Sets the w position of this Game Object.
                 * @param value The w position of this Game Object. Default 0.
                 */
                setW(value?: number): this;
                /**
                 * Gets the local transform matrix for this Game Object.
                 * @param tempMatrix The matrix to populate with the values from this Game Object.
                 */
                getLocalTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;
                /**
                 * Gets the world transform matrix for this Game Object, factoring in any parent Containers.
                 * @param tempMatrix The matrix to populate with the values from this Game Object.
                 * @param parentMatrix A temporary matrix to hold parent values during the calculations.
                 */
                getWorldTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix, parentMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;
                /**
                 * Takes the given `x` and `y` coordinates and converts them into local space for this
                 * Game Object, taking into account parent and local transforms, and the Display Origin.
                 * 
                 * The returned Vector2 contains the translated point in its properties.
                 * 
                 * A Camera needs to be provided in order to handle modified scroll factors. If no
                 * camera is specified, it will use the `main` camera from the Scene to which this
                 * Game Object belongs.
                 * @param x The x position to translate.
                 * @param y The y position to translate.
                 * @param point A Vector2, or point-like object, to store the results in.
                 * @param camera The Camera which is being tested against. If not given will use the Scene default camera.
                 */
                getLocalPoint(x: number, y: number, point?: Phaser.Math.Vector2, camera?: Phaser.Cameras.Scene2D.Camera): Phaser.Math.Vector2;
                /**
                 * Gets the sum total rotation of all of this Game Objects parent Containers.
                 * 
                 * The returned value is in radians and will be zero if this Game Object has no parent container.
                 */
                getParentRotation(): number;
            }

            /**
             * A Matrix used for display transformations for rendering.
             * 
             * It is represented like so:
             * 
             * ```
             * | a | c | tx |
             * | b | d | ty |
             * | 0 | 0 | 1  |
             * ```
             */
            class TransformMatrix {
                /**
                 * 
                 * @param a The Scale X value. Default 1.
                 * @param b The Skew Y value. Default 0.
                 * @param c The Skew X value. Default 0.
                 * @param d The Scale Y value. Default 1.
                 * @param tx The Translate X value. Default 0.
                 * @param ty The Translate Y value. Default 0.
                 */
                constructor(a?: number, b?: number, c?: number, d?: number, tx?: number, ty?: number);

                /**
                 * The matrix values.
                 */
                matrix: Float32Array;

                /**
                 * The decomposed matrix.
                 */
                decomposedMatrix: object;

                /**
                 * The Scale X value.
                 */
                a: number;

                /**
                 * The Skew Y value.
                 */
                b: number;

                /**
                 * The Skew X value.
                 */
                c: number;

                /**
                 * The Scale Y value.
                 */
                d: number;

                /**
                 * The Translate X value.
                 */
                e: number;

                /**
                 * The Translate Y value.
                 */
                f: number;

                /**
                 * The Translate X value.
                 */
                tx: number;

                /**
                 * The Translate Y value.
                 */
                ty: number;

                /**
                 * The rotation of the Matrix. Value is in radians.
                 */
                readonly rotation: number;

                /**
                 * The rotation of the Matrix, normalized to be within the Phaser right-handed
                 * clockwise rotation space. Value is in radians.
                 */
                readonly rotationNormalized: number;

                /**
                 * The decomposed horizontal scale of the Matrix. This value is always positive.
                 */
                readonly scaleX: number;

                /**
                 * The decomposed vertical scale of the Matrix. This value is always positive.
                 */
                readonly scaleY: number;

                /**
                 * Reset the Matrix to an identity matrix.
                 */
                loadIdentity(): this;

                /**
                 * Translate the Matrix.
                 * @param x The horizontal translation value.
                 * @param y The vertical translation value.
                 */
                translate(x: number, y: number): this;

                /**
                 * Scale the Matrix.
                 * @param x The horizontal scale value.
                 * @param y The vertical scale value.
                 */
                scale(x: number, y: number): this;

                /**
                 * Rotate the Matrix.
                 * @param angle The angle of rotation in radians.
                 */
                rotate(angle: number): this;

                /**
                 * Multiply this Matrix by the given Matrix.
                 * 
                 * If an `out` Matrix is given then the results will be stored in it.
                 * If it is not given, this matrix will be updated in place instead.
                 * Use an `out` Matrix if you do not wish to mutate this matrix.
                 * @param rhs The Matrix to multiply by.
                 * @param out An optional Matrix to store the results in.
                 */
                multiply(rhs: Phaser.GameObjects.Components.TransformMatrix, out?: Phaser.GameObjects.Components.TransformMatrix): this | Phaser.GameObjects.Components.TransformMatrix;

                /**
                 * Multiply this Matrix by the matrix given, including the offset.
                 * 
                 * The offsetX is added to the tx value: `offsetX * a + offsetY * c + tx`.
                 * The offsetY is added to the ty value: `offsetY * b + offsetY * d + ty`.
                 * @param src The source Matrix to copy from.
                 * @param offsetX Horizontal offset to factor in to the multiplication.
                 * @param offsetY Vertical offset to factor in to the multiplication.
                 */
                multiplyWithOffset(src: Phaser.GameObjects.Components.TransformMatrix, offsetX: number, offsetY: number): this;

                /**
                 * Transform the Matrix.
                 * @param a The Scale X value.
                 * @param b The Shear Y value.
                 * @param c The Shear X value.
                 * @param d The Scale Y value.
                 * @param tx The Translate X value.
                 * @param ty The Translate Y value.
                 */
                transform(a: number, b: number, c: number, d: number, tx: number, ty: number): this;

                /**
                 * Transform a point using this Matrix.
                 * @param x The x coordinate of the point to transform.
                 * @param y The y coordinate of the point to transform.
                 * @param point The Point object to store the transformed coordinates.
                 */
                transformPoint(x: number, y: number, point: Phaser.Geom.Point | Phaser.Math.Vector2 | object): Phaser.Geom.Point | Phaser.Math.Vector2 | object;

                /**
                 * Invert the Matrix.
                 */
                invert(): this;

                /**
                 * Set the values of this Matrix to copy those of the matrix given.
                 * @param src The source Matrix to copy from.
                 */
                copyFrom(src: Phaser.GameObjects.Components.TransformMatrix): this;

                /**
                 * Set the values of this Matrix to copy those of the array given.
                 * Where array indexes 0, 1, 2, 3, 4 and 5 are mapped to a, b, c, d, e and f.
                 * @param src The array of values to set into this matrix.
                 */
                copyFromArray(src: any[]): this;

                /**
                 * Copy the values from this Matrix to the given Canvas Rendering Context.
                 * This will use the Context.transform method.
                 * @param ctx The Canvas Rendering Context to copy the matrix values to.
                 */
                copyToContext(ctx: CanvasRenderingContext2D): CanvasRenderingContext2D;

                /**
                 * Copy the values from this Matrix to the given Canvas Rendering Context.
                 * This will use the Context.setTransform method.
                 * @param ctx The Canvas Rendering Context to copy the matrix values to.
                 */
                setToContext(ctx: CanvasRenderingContext2D): CanvasRenderingContext2D;

                /**
                 * Copy the values in this Matrix to the array given.
                 * 
                 * Where array indexes 0, 1, 2, 3, 4 and 5 are mapped to a, b, c, d, e and f.
                 * @param out The array to copy the matrix values in to.
                 */
                copyToArray(out?: any[]): any[];

                /**
                 * Set the values of this Matrix.
                 * @param a The Scale X value.
                 * @param b The Shear Y value.
                 * @param c The Shear X value.
                 * @param d The Scale Y value.
                 * @param tx The Translate X value.
                 * @param ty The Translate Y value.
                 */
                setTransform(a: number, b: number, c: number, d: number, tx: number, ty: number): this;

                /**
                 * Decompose this Matrix into its translation, scale and rotation values using QR decomposition.
                 * 
                 * The result must be applied in the following order to reproduce the current matrix:
                 * 
                 * translate -> rotate -> scale
                 */
                decomposeMatrix(): object;

                /**
                 * Apply the identity, translate, rotate and scale operations on the Matrix.
                 * @param x The horizontal translation.
                 * @param y The vertical translation.
                 * @param rotation The angle of rotation in radians.
                 * @param scaleX The horizontal scale.
                 * @param scaleY The vertical scale.
                 */
                applyITRS(x: number, y: number, rotation: number, scaleX: number, scaleY: number): this;

                /**
                 * Takes the `x` and `y` values and returns a new position in the `output` vector that is the inverse of
                 * the current matrix with its transformation applied.
                 * 
                 * Can be used to translate points from world to local space.
                 * @param x The x position to translate.
                 * @param y The y position to translate.
                 * @param output A Vector2, or point-like object, to store the results in.
                 */
                applyInverse(x: number, y: number, output?: Phaser.Math.Vector2): Phaser.Math.Vector2;

                /**
                 * Returns the X component of this matrix multiplied by the given values.
                 * This is the same as `x * a + y * c + e`.
                 * @param x The x value.
                 * @param y The y value.
                 */
                getX(x: number, y: number): number;

                /**
                 * Returns the Y component of this matrix multiplied by the given values.
                 * This is the same as `x * b + y * d + f`.
                 * @param x The x value.
                 * @param y The y value.
                 */
                getY(x: number, y: number): number;

                /**
                 * Returns the X component of this matrix multiplied by the given values.
                 * 
                 * This is the same as `x * a + y * c + e`, optionally passing via `Math.round`.
                 * @param x The x value.
                 * @param y The y value.
                 * @param round Math.round the resulting value? Default false.
                 */
                getXRound(x: number, y: number, round?: boolean): number;

                /**
                 * Returns the Y component of this matrix multiplied by the given values.
                 * 
                 * This is the same as `x * b + y * d + f`, optionally passing via `Math.round`.
                 * @param x The x value.
                 * @param y The y value.
                 * @param round Math.round the resulting value? Default false.
                 */
                getYRound(x: number, y: number, round?: boolean): number;

                /**
                 * Returns a string that can be used in a CSS Transform call as a `matrix` property.
                 */
                getCSSMatrix(): string;

                /**
                 * Destroys this Transform Matrix.
                 */
                destroy(): void;

            }

            /**
             * Provides methods used for setting the visibility of a Game Object.
             * Should be applied as a mixin and not used directly.
             */
            interface Visible {
                /**
                 * The visible state of the Game Object.
                 * 
                 * An invisible Game Object will skip rendering, but will still process update logic.
                 */
                visible: boolean;
                /**
                 * Sets the visibility of this Game Object.
                 * 
                 * An invisible Game Object will skip rendering, but will still process update logic.
                 * @param value The visible state of the Game Object.
                 */
                setVisible(value: boolean): this;
            }

        }

        /**
         * A Container Game Object.
         * 
         * A Container, as the name implies, can 'contain' other types of Game Object.
         * When a Game Object is added to a Container, the Container becomes responsible for the rendering of it.
         * By default it will be removed from the Display List and instead added to the Containers own internal list.
         * 
         * The position of the Game Object automatically becomes relative to the position of the Container.
         * 
         * The transform point of a Container is 0x0 (in local space) and that cannot be changed. The children you add to the
         * Container should be positioned with this value in mind. I.e. you should treat 0x0 as being the center of
         * the Container, and position children positively and negative around it as required.
         * 
         * When the Container is rendered, all of its children are rendered as well, in the order in which they exist
         * within the Container. Container children can be repositioned using methods such as `MoveUp`, `MoveDown` and `SendToBack`.
         * 
         * If you modify a transform property of the Container, such as `Container.x` or `Container.rotation` then it will
         * automatically influence all children as well.
         * 
         * Containers can include other Containers for deeply nested transforms.
         * 
         * Containers can have masks set on them and can be used as a mask too. However, Container children cannot be masked.
         * The masks do not 'stack up'. Only a Container on the root of the display list will use its mask.
         * 
         * Containers can be enabled for input. Because they do not have a texture you need to provide a shape for them
         * to use as their hit area. Container children can also be enabled for input, independent of the Container.
         * 
         * If input enabling a _child_ you should not set both the `origin` and a **negative** scale factor on the child,
         * or the input area will become misaligned.
         * 
         * Containers can be given a physics body for either Arcade Physics, Impact Physics or Matter Physics. However,
         * if Container _children_ are enabled for physics you may get unexpected results, such as offset bodies,
         * if the Container itself, or any of its ancestors, is positioned anywhere other than at 0 x 0. Container children
         * with physics do not factor in the Container due to the excessive extra calculations needed. Please structure
         * your game to work around this.
         * 
         * It's important to understand the impact of using Containers. They add additional processing overhead into
         * every one of their children. The deeper you nest them, the more the cost escalates. This is especially true
         * for input events. You also loose the ability to set the display depth of Container children in the same
         * flexible manner as those not within them. In short, don't use them for the sake of it. You pay a small cost
         * every time you create one, try to structure your game around avoiding that where possible.
         */
        class Container extends Phaser.GameObjects.GameObject implements Phaser.GameObjects.Components.AlphaSingle, Phaser.GameObjects.Components.BlendMode, Phaser.GameObjects.Components.ComputedSize, Phaser.GameObjects.Components.Depth, Phaser.GameObjects.Components.Mask, Phaser.GameObjects.Components.Pipeline, Phaser.GameObjects.Components.Transform, Phaser.GameObjects.Components.Visible {
            /**
             * 
             * @param scene The Scene to which this Game Object belongs. A Game Object can only belong to one Scene at a time.
             * @param x The horizontal position of this Game Object in the world. Default 0.
             * @param y The vertical position of this Game Object in the world. Default 0.
             * @param children An optional array of Game Objects to add to this Container.
             */
            constructor(scene: Phaser.Scene, x?: number, y?: number, children?: Phaser.GameObjects.GameObject[]);

            /**
             * An array holding the children of this Container.
             */
            list: Phaser.GameObjects.GameObject[];

            /**
             * Does this Container exclusively manage its children?
             * 
             * The default is `true` which means a child added to this Container cannot
             * belong in another Container, which includes the Scene display list.
             * 
             * If you disable this then this Container will no longer exclusively manage its children.
             * This allows you to create all kinds of interesting graphical effects, such as replicating
             * Game Objects without reparenting them all over the Scene.
             * However, doing so will prevent children from receiving any kind of input event or have
             * their physics bodies work by default, as they're no longer a single entity on the
             * display list, but are being replicated where-ever this Container is.
             */
            exclusive: boolean;

            /**
             * Containers can have an optional maximum size. If set to anything above 0 it
             * will constrict the addition of new Game Objects into the Container, capping off
             * the maximum limit the Container can grow in size to.
             */
            maxSize: number;

            /**
             * The cursor position.
             */
            position: number;

            /**
             * Internal Transform Matrix used for local space conversion.
             */
            localTransform: Phaser.GameObjects.Components.TransformMatrix;

            /**
             * The horizontal scroll factor of this Container.
             * 
             * The scroll factor controls the influence of the movement of a Camera upon this Container.
             * 
             * When a camera scrolls it will change the location at which this Container is rendered on-screen.
             * It does not change the Containers actual position values.
             * 
             * For a Container, setting this value will only update the Container itself, not its children.
             * If you wish to change the scrollFactor of the children as well, use the `setScrollFactor` method.
             * 
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Container.
             * 
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             */
            scrollFactorX: number;

            /**
             * The vertical scroll factor of this Container.
             * 
             * The scroll factor controls the influence of the movement of a Camera upon this Container.
             * 
             * When a camera scrolls it will change the location at which this Container is rendered on-screen.
             * It does not change the Containers actual position values.
             * 
             * For a Container, setting this value will only update the Container itself, not its children.
             * If you wish to change the scrollFactor of the children as well, use the `setScrollFactor` method.
             * 
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Container.
             * 
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             */
            scrollFactorY: number;

            /**
             * Internal value to allow Containers to be used for input and physics.
             * Do not change this value. It has no effect other than to break things.
             */
            readonly originX: number;

            /**
             * Internal value to allow Containers to be used for input and physics.
             * Do not change this value. It has no effect other than to break things.
             */
            readonly originY: number;

            /**
             * Internal value to allow Containers to be used for input and physics.
             * Do not change this value. It has no effect other than to break things.
             */
            readonly displayOriginX: number;

            /**
             * Internal value to allow Containers to be used for input and physics.
             * Do not change this value. It has no effect other than to break things.
             */
            readonly displayOriginY: number;

            /**
             * Does this Container exclusively manage its children?
             * 
             * The default is `true` which means a child added to this Container cannot
             * belong in another Container, which includes the Scene display list.
             * 
             * If you disable this then this Container will no longer exclusively manage its children.
             * This allows you to create all kinds of interesting graphical effects, such as replicating
             * Game Objects without reparenting them all over the Scene.
             * However, doing so will prevent children from receiving any kind of input event or have
             * their physics bodies work by default, as they're no longer a single entity on the
             * display list, but are being replicated where-ever this Container is.
             * @param value The exclusive state of this Container. Default true.
             */
            setExclusive(value?: boolean): this;

            /**
             * Gets the bounds of this Container. It works by iterating all children of the Container,
             * getting their respective bounds, and then working out a min-max rectangle from that.
             * It does not factor in if the children render or not, all are included.
             * 
             * Some children are unable to return their bounds, such as Graphics objects, in which case
             * they are skipped.
             * 
             * Depending on the quantity of children in this Container it could be a really expensive call,
             * so cache it and only poll it as needed.
             * 
             * The values are stored and returned in a Rectangle object.
             * @param output A Geom.Rectangle object to store the values in. If not provided a new Rectangle will be created.
             */
            getBounds(output?: Phaser.Geom.Rectangle): Phaser.Geom.Rectangle;

            /**
             * Takes a Point-like object, such as a Vector2, Geom.Point or object with public x and y properties,
             * and transforms it into the space of this Container, then returns it in the output object.
             * @param source The Source Point to be transformed.
             * @param output A destination object to store the transformed point in. If none given a Vector2 will be created and returned.
             */
            pointToContainer(source: object | Phaser.Geom.Point | Phaser.Math.Vector2, output?: object | Phaser.Geom.Point | Phaser.Math.Vector2): object | Phaser.Geom.Point | Phaser.Math.Vector2;

            /**
             * Returns the world transform matrix as used for Bounds checks.
             * 
             * The returned matrix is temporal and shouldn't be stored.
             */
            getBoundsTransformMatrix(): Phaser.GameObjects.Components.TransformMatrix;

            /**
             * Adds the given Game Object, or array of Game Objects, to this Container.
             * 
             * Each Game Object must be unique within the Container.
             * @param child The Game Object, or array of Game Objects, to add to the Container.
             */
            add(child: Phaser.GameObjects.GameObject | Phaser.GameObjects.GameObject[]): this;

            /**
             * Adds the given Game Object, or array of Game Objects, to this Container at the specified position.
             * 
             * Existing Game Objects in the Container are shifted up.
             * 
             * Each Game Object must be unique within the Container.
             * @param child The Game Object, or array of Game Objects, to add to the Container.
             * @param index The position to insert the Game Object/s at. Default 0.
             */
            addAt(child: Phaser.GameObjects.GameObject | Phaser.GameObjects.GameObject[], index?: number): this;

            /**
             * Returns the Game Object at the given position in this Container.
             * @param index The position to get the Game Object from.
             */
            getAt(index: number): Phaser.GameObjects.GameObject;

            /**
             * Returns the index of the given Game Object in this Container.
             * @param child The Game Object to search for in this Container.
             */
            getIndex(child: Phaser.GameObjects.GameObject): number;

            /**
             * Sort the contents of this Container so the items are in order based on the given property.
             * For example: `sort('alpha')` would sort the elements based on the value of their `alpha` property.
             * @param property The property to lexically sort by.
             * @param handler Provide your own custom handler function. Will receive 2 children which it should compare and return a boolean.
             */
            sort(property: string, handler?: Function): this;

            /**
             * Searches for the first instance of a child with its `name` property matching the given argument.
             * Should more than one child have the same name only the first is returned.
             * @param name The name to search for.
             */
            getByName(name: string): Phaser.GameObjects.GameObject;

            /**
             * Returns a random Game Object from this Container.
             * @param startIndex An optional start index. Default 0.
             * @param length An optional length, the total number of elements (from the startIndex) to choose from.
             */
            getRandom(startIndex?: number, length?: number): Phaser.GameObjects.GameObject;

            /**
             * Gets the first Game Object in this Container.
             * 
             * You can also specify a property and value to search for, in which case it will return the first
             * Game Object in this Container with a matching property and / or value.
             * 
             * For example: `getFirst('visible', true)` would return the first Game Object that had its `visible` property set.
             * 
             * You can limit the search to the `startIndex` - `endIndex` range.
             * @param property The property to test on each Game Object in the Container.
             * @param value The value to test the property against. Must pass a strict (`===`) comparison check.
             * @param startIndex An optional start index to search from. Default 0.
             * @param endIndex An optional end index to search up to (but not included) Default Container.length.
             */
            getFirst(property: string, value: any, startIndex?: number, endIndex?: number): Phaser.GameObjects.GameObject;

            /**
             * Returns all Game Objects in this Container.
             * 
             * You can optionally specify a matching criteria using the `property` and `value` arguments.
             * 
             * For example: `getAll('body')` would return only Game Objects that have a body property.
             * 
             * You can also specify a value to compare the property to:
             * 
             * `getAll('visible', true)` would return only Game Objects that have their visible property set to `true`.
             * 
             * Optionally you can specify a start and end index. For example if this Container had 100 Game Objects,
             * and you set `startIndex` to 0 and `endIndex` to 50, it would return matches from only
             * the first 50 Game Objects.
             * @param property The property to test on each Game Object in the Container.
             * @param value If property is set then the `property` must strictly equal this value to be included in the results.
             * @param startIndex An optional start index to search from. Default 0.
             * @param endIndex An optional end index to search up to (but not included) Default Container.length.
             */
            getAll(property?: string, value?: any, startIndex?: number, endIndex?: number): Phaser.GameObjects.GameObject[];

            /**
             * Returns the total number of Game Objects in this Container that have a property
             * matching the given value.
             * 
             * For example: `count('visible', true)` would count all the elements that have their visible property set.
             * 
             * You can optionally limit the operation to the `startIndex` - `endIndex` range.
             * @param property The property to check.
             * @param value The value to check.
             * @param startIndex An optional start index to search from. Default 0.
             * @param endIndex An optional end index to search up to (but not included) Default Container.length.
             */
            count(property: string, value: any, startIndex?: number, endIndex?: number): number;

            /**
             * Swaps the position of two Game Objects in this Container.
             * Both Game Objects must belong to this Container.
             * @param child1 The first Game Object to swap.
             * @param child2 The second Game Object to swap.
             */
            swap(child1: Phaser.GameObjects.GameObject, child2: Phaser.GameObjects.GameObject): this;

            /**
             * Moves a Game Object to a new position within this Container.
             * 
             * The Game Object must already be a child of this Container.
             * 
             * The Game Object is removed from its old position and inserted into the new one.
             * Therefore the Container size does not change. Other children will change position accordingly.
             * @param child The Game Object to move.
             * @param index The new position of the Game Object in this Container.
             */
            moveTo(child: Phaser.GameObjects.GameObject, index: number): this;

            /**
             * Moves a Game Object above another one within this Container.
             * 
             * These 2 Game Objects must already be children of this Container.
             * @param child1 The Game Object to move above base Game Object.
             * @param child2 The base Game Object.
             */
            moveAbove(child1: Phaser.GameObjects.GameObject, child2: Phaser.GameObjects.GameObject): this;

            /**
             * Moves a Game Object below another one within this Container.
             * 
             * These 2 Game Objects must already be children of this Container.
             * @param child1 The Game Object to move below base Game Object.
             * @param child2 The base Game Object.
             */
            moveBelow(child1: Phaser.GameObjects.GameObject, child2: Phaser.GameObjects.GameObject): this;

            /**
             * Removes the given Game Object, or array of Game Objects, from this Container.
             * 
             * The Game Objects must already be children of this Container.
             * 
             * You can also optionally call `destroy` on each Game Object that is removed from the Container.
             * @param child The Game Object, or array of Game Objects, to be removed from the Container.
             * @param destroyChild Optionally call `destroy` on each child successfully removed from this Container. Default false.
             */
            remove(child: Phaser.GameObjects.GameObject | Phaser.GameObjects.GameObject[], destroyChild?: boolean): this;

            /**
             * Removes the Game Object at the given position in this Container.
             * 
             * You can also optionally call `destroy` on the Game Object, if one is found.
             * @param index The index of the Game Object to be removed.
             * @param destroyChild Optionally call `destroy` on the Game Object if successfully removed from this Container. Default false.
             */
            removeAt(index: number, destroyChild?: boolean): this;

            /**
             * Removes the Game Objects between the given positions in this Container.
             * 
             * You can also optionally call `destroy` on each Game Object that is removed from the Container.
             * @param startIndex An optional start index to search from. Default 0.
             * @param endIndex An optional end index to search up to (but not included) Default Container.length.
             * @param destroyChild Optionally call `destroy` on each Game Object successfully removed from this Container. Default false.
             */
            removeBetween(startIndex?: number, endIndex?: number, destroyChild?: boolean): this;

            /**
             * Removes all Game Objects from this Container.
             * 
             * You can also optionally call `destroy` on each Game Object that is removed from the Container.
             * @param destroyChild Optionally call `destroy` on each Game Object successfully removed from this Container. Default false.
             */
            removeAll(destroyChild?: boolean): this;

            /**
             * Brings the given Game Object to the top of this Container.
             * This will cause it to render on-top of any other objects in the Container.
             * @param child The Game Object to bring to the top of the Container.
             */
            bringToTop(child: Phaser.GameObjects.GameObject): this;

            /**
             * Sends the given Game Object to the bottom of this Container.
             * This will cause it to render below any other objects in the Container.
             * @param child The Game Object to send to the bottom of the Container.
             */
            sendToBack(child: Phaser.GameObjects.GameObject): this;

            /**
             * Moves the given Game Object up one place in this Container, unless it's already at the top.
             * @param child The Game Object to be moved in the Container.
             */
            moveUp(child: Phaser.GameObjects.GameObject): this;

            /**
             * Moves the given Game Object down one place in this Container, unless it's already at the bottom.
             * @param child The Game Object to be moved in the Container.
             */
            moveDown(child: Phaser.GameObjects.GameObject): this;

            /**
             * Reverses the order of all Game Objects in this Container.
             */
            reverse(): this;

            /**
             * Shuffles the all Game Objects in this Container using the Fisher-Yates implementation.
             */
            shuffle(): this;

            /**
             * Replaces a Game Object in this Container with the new Game Object.
             * The new Game Object cannot already be a child of this Container.
             * @param oldChild The Game Object in this Container that will be replaced.
             * @param newChild The Game Object to be added to this Container.
             * @param destroyChild Optionally call `destroy` on the Game Object if successfully removed from this Container. Default false.
             */
            replace(oldChild: Phaser.GameObjects.GameObject, newChild: Phaser.GameObjects.GameObject, destroyChild?: boolean): this;

            /**
             * Returns `true` if the given Game Object is a direct child of this Container.
             * 
             * This check does not scan nested Containers.
             * @param child The Game Object to check for within this Container.
             */
            exists(child: Phaser.GameObjects.GameObject): boolean;

            /**
             * Sets the property to the given value on all Game Objects in this Container.
             * 
             * Optionally you can specify a start and end index. For example if this Container had 100 Game Objects,
             * and you set `startIndex` to 0 and `endIndex` to 50, it would return matches from only
             * the first 50 Game Objects.
             * @param property The property that must exist on the Game Object.
             * @param value The value to get the property to.
             * @param startIndex An optional start index to search from. Default 0.
             * @param endIndex An optional end index to search up to (but not included) Default Container.length.
             */
            setAll(property: string, value: any, startIndex?: number, endIndex?: number): this;

            /**
             * Passes all Game Objects in this Container to the given callback.
             * 
             * A copy of the Container is made before passing each entry to your callback.
             * This protects against the callback itself modifying the Container.
             * 
             * If you know for sure that the callback will not change the size of this Container
             * then you can use the more performant `Container.iterate` method instead.
             * @param callback The function to call.
             * @param context Value to use as `this` when executing callback.
             * @param args Additional arguments that will be passed to the callback, after the child.
             */
            each(callback: Function, context?: object, ...args: any[]): this;

            /**
             * Passes all Game Objects in this Container to the given callback.
             * 
             * Only use this method when you absolutely know that the Container will not be modified during
             * the iteration, i.e. by removing or adding to its contents.
             * @param callback The function to call.
             * @param context Value to use as `this` when executing callback.
             * @param args Additional arguments that will be passed to the callback, after the child.
             */
            iterate(callback: Function, context?: object, ...args: any[]): this;

            /**
             * Sets the scroll factor of this Container and optionally all of its children.
             * 
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             * 
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             * 
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             * 
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             * @param x The horizontal scroll factor of this Game Object.
             * @param y The vertical scroll factor of this Game Object. If not set it will use the `x` value. Default x.
             * @param updateChildren Apply this scrollFactor to all Container children as well? Default false.
             */
            setScrollFactor(x: number, y?: number, updateChildren?: boolean): this;

            /**
             * The number of Game Objects inside this Container.
             */
            readonly length: number;

            /**
             * Returns the first Game Object within the Container, or `null` if it is empty.
             * 
             * You can move the cursor by calling `Container.next` and `Container.previous`.
             */
            readonly first: Phaser.GameObjects.GameObject;

            /**
             * Returns the last Game Object within the Container, or `null` if it is empty.
             * 
             * You can move the cursor by calling `Container.next` and `Container.previous`.
             */
            readonly last: Phaser.GameObjects.GameObject;

            /**
             * Returns the next Game Object within the Container, or `null` if it is empty.
             * 
             * You can move the cursor by calling `Container.next` and `Container.previous`.
             */
            readonly next: Phaser.GameObjects.GameObject;

            /**
             * Returns the previous Game Object within the Container, or `null` if it is empty.
             * 
             * You can move the cursor by calling `Container.next` and `Container.previous`.
             */
            readonly previous: Phaser.GameObjects.GameObject;

            /**
             * Internal destroy handler, called as part of the destroy process.
             */
            protected preDestroy(): void;

            /**
             * Clears all alpha values associated with this Game Object.
             * 
             * Immediately sets the alpha levels back to 1 (fully opaque).
             */
            clearAlpha(): this;

            /**
             * Set the Alpha level of this Game Object. The alpha controls the opacity of the Game Object as it renders.
             * Alpha values are provided as a float between 0, fully transparent, and 1, fully opaque.
             * @param value The alpha value applied across the whole Game Object. Default 1.
             */
            setAlpha(value?: number): this;

            /**
             * The alpha value of the Game Object.
             * 
             * This is a global value, impacting the entire Game Object, not just a region of it.
             */
            alpha: number;

            /**
             * Sets the Blend Mode being used by this Game Object.
             * 
             * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
             * 
             * Under WebGL only the following Blend Modes are available:
             * 
             * * ADD
             * * MULTIPLY
             * * SCREEN
             * * ERASE
             * 
             * Canvas has more available depending on browser support.
             * 
             * You can also create your own custom Blend Modes in WebGL.
             * 
             * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
             * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
             * reasons try to be careful about the construction of your Scene and the frequency of which blend modes
             * are used.
             */
            blendMode: Phaser.BlendModes | string;

            /**
             * Sets the Blend Mode being used by this Game Object.
             * 
             * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
             * 
             * Under WebGL only the following Blend Modes are available:
             * 
             * * ADD
             * * MULTIPLY
             * * SCREEN
             * * ERASE (only works when rendering to a framebuffer, like a Render Texture)
             * 
             * Canvas has more available depending on browser support.
             * 
             * You can also create your own custom Blend Modes in WebGL.
             * 
             * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
             * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
             * reasons try to be careful about the construction of your Scene and the frequency in which blend modes
             * are used.
             * @param value The BlendMode value. Either a string or a CONST.
             */
            setBlendMode(value: string | Phaser.BlendModes): this;

            /**
             * The native (un-scaled) width of this Game Object.
             * 
             * Changing this value will not change the size that the Game Object is rendered in-game.
             * For that you need to either set the scale of the Game Object (`setScale`) or use
             * the `displayWidth` property.
             */
            width: number;

            /**
             * The native (un-scaled) height of this Game Object.
             * 
             * Changing this value will not change the size that the Game Object is rendered in-game.
             * For that you need to either set the scale of the Game Object (`setScale`) or use
             * the `displayHeight` property.
             */
            height: number;

            /**
             * The displayed width of this Game Object.
             * 
             * This value takes into account the scale factor.
             * 
             * Setting this value will adjust the Game Object's scale property.
             */
            displayWidth: number;

            /**
             * The displayed height of this Game Object.
             * 
             * This value takes into account the scale factor.
             * 
             * Setting this value will adjust the Game Object's scale property.
             */
            displayHeight: number;

            /**
             * Sets the internal size of this Game Object, as used for frame or physics body creation.
             * 
             * This will not change the size that the Game Object is rendered in-game.
             * For that you need to either set the scale of the Game Object (`setScale`) or call the
             * `setDisplaySize` method, which is the same thing as changing the scale but allows you
             * to do so by giving pixel values.
             * 
             * If you have enabled this Game Object for input, changing the size will _not_ change the
             * size of the hit area. To do this you should adjust the `input.hitArea` object directly.
             * @param width The width of this Game Object.
             * @param height The height of this Game Object.
             */
            setSize(width: number, height: number): this;

            /**
             * Sets the display size of this Game Object.
             * 
             * Calling this will adjust the scale.
             * @param width The width of this Game Object.
             * @param height The height of this Game Object.
             */
            setDisplaySize(width: number, height: number): this;

            /**
             * The depth of this Game Object within the Scene.
             * 
             * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
             * of Game Objects, without actually moving their position in the display list.
             * 
             * The default depth is zero. A Game Object with a higher depth
             * value will always render in front of one with a lower value.
             * 
             * Setting the depth will queue a depth sort event within the Scene.
             */
            depth: number;

            /**
             * The depth of this Game Object within the Scene.
             * 
             * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
             * of Game Objects, without actually moving their position in the display list.
             * 
             * The default depth is zero. A Game Object with a higher depth
             * value will always render in front of one with a lower value.
             * 
             * Setting the depth will queue a depth sort event within the Scene.
             * @param value The depth of this Game Object.
             */
            setDepth(value: number): this;

            /**
             * The Mask this Game Object is using during render.
             */
            mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask;

            /**
             * Sets the mask that this Game Object will use to render with.
             * 
             * The mask must have been previously created and can be either a GeometryMask or a BitmapMask.
             * Note: Bitmap Masks only work on WebGL. Geometry Masks work on both WebGL and Canvas.
             * 
             * If a mask is already set on this Game Object it will be immediately replaced.
             * 
             * Masks are positioned in global space and are not relative to the Game Object to which they
             * are applied. The reason for this is that multiple Game Objects can all share the same mask.
             * 
             * Masks have no impact on physics or input detection. They are purely a rendering component
             * that allows you to limit what is visible during the render pass.
             * @param mask The mask this Game Object will use when rendering.
             */
            setMask(mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask): this;

            /**
             * Clears the mask that this Game Object was using.
             * @param destroyMask Destroy the mask before clearing it? Default false.
             */
            clearMask(destroyMask?: boolean): this;

            /**
             * Creates and returns a Bitmap Mask. This mask can be used by any Game Object,
             * including this one.
             * 
             * Note: Bitmap Masks only work on WebGL. Geometry Masks work on both WebGL and Canvas.
             * 
             * To create the mask you need to pass in a reference to a renderable Game Object.
             * A renderable Game Object is one that uses a texture to render with, such as an
             * Image, Sprite, Render Texture or BitmapText.
             * 
             * If you do not provide a renderable object, and this Game Object has a texture,
             * it will use itself as the object. This means you can call this method to create
             * a Bitmap Mask from any renderable Game Object.
             * @param renderable A renderable Game Object that uses a texture, such as a Sprite.
             */
            createBitmapMask(renderable?: Phaser.GameObjects.GameObject): Phaser.Display.Masks.BitmapMask;

            /**
             * Creates and returns a Geometry Mask. This mask can be used by any Game Object,
             * including this one.
             * 
             * To create the mask you need to pass in a reference to a Graphics Game Object.
             * 
             * If you do not provide a graphics object, and this Game Object is an instance
             * of a Graphics object, then it will use itself to create the mask.
             * 
             * This means you can call this method to create a Geometry Mask from any Graphics Game Object.
             * @param graphics A Graphics Game Object. The geometry within it will be used as the mask.
             */
            createGeometryMask(graphics?: Phaser.GameObjects.Graphics): Phaser.Display.Masks.GeometryMask;

            /**
             * The initial WebGL pipeline of this Game Object.
             * 
             * If you call `resetPipeline` on this Game Object, the pipeline is reset to this default.
             */
            defaultPipeline: Phaser.Renderer.WebGL.WebGLPipeline;

            /**
             * The current WebGL pipeline of this Game Object.
             */
            pipeline: Phaser.Renderer.WebGL.WebGLPipeline;

            /**
             * Does this Game Object have any Post Pipelines set?
             */
            hasPostPipeline: boolean;

            /**
             * The WebGL Post FX Pipelines this Game Object uses for post-render effects.
             * 
             * The pipelines are processed in the order in which they appear in this array.
             * 
             * If you modify this array directly, be sure to set the
             * `hasPostPipeline` property accordingly.
             */
            postPipelines: Phaser.Renderer.WebGL.Pipelines.PostFXPipeline[];

            /**
             * An object to store pipeline specific data in, to be read by the pipelines this Game Object uses.
             */
            pipelineData: object;

            /**
             * Sets the initial WebGL Pipeline of this Game Object.
             * 
             * This should only be called during the instantiation of the Game Object. After that, use `setPipeline`.
             * @param pipeline Either the string-based name of the pipeline, or a pipeline instance to set.
             */
            initPipeline(pipeline: string | Phaser.Renderer.WebGL.WebGLPipeline): boolean;

            /**
             * Sets the main WebGL Pipeline of this Game Object.
             * 
             * Also sets the `pipelineData` property, if the parameter is given.
             * 
             * Both the pipeline and post pipelines share the same pipeline data object.
             * @param pipeline Either the string-based name of the pipeline, or a pipeline instance to set.
             * @param pipelineData Optional pipeline data object that is _deep copied_ into the `pipelineData` property of this Game Object.
             * @param copyData Should the pipeline data object be _deep copied_ into the `pipelineData` property of this Game Object? If `false` it will be set by reference instead. Default true.
             */
            setPipeline(pipeline: string | Phaser.Renderer.WebGL.WebGLPipeline, pipelineData?: object, copyData?: boolean): this;

            /**
             * Sets one, or more, Post Pipelines on this Game Object.
             * 
             * Post Pipelines are invoked after this Game Object has rendered to its target and
             * are commonly used for post-fx.
             * 
             * The post pipelines are appended to the `postPipelines` array belonging to this
             * Game Object. When the renderer processes this Game Object, it iterates through the post
             * pipelines in the order in which they appear in the array. If you are stacking together
             * multiple effects, be aware that the order is important.
             * 
             * If you call this method multiple times, the new pipelines will be appended to any existing
             * post pipelines already set. Use the `resetPostPipeline` method to clear them first, if required.
             * 
             * You can optionally also sets the `pipelineData` property, if the parameter is given.
             * 
             * Both the pipeline and post pipelines share the pipeline data object together.
             * @param pipelines Either the string-based name of the pipeline, or a pipeline instance, or class, or an array of them.
             * @param pipelineData Optional pipeline data object that is _deep copied_ into the `pipelineData` property of this Game Object.
             * @param copyData Should the pipeline data object be _deep copied_ into the `pipelineData` property of this Game Object? If `false` it will be set by reference instead. Default true.
             */
            setPostPipeline(pipelines: string | string[] | Function | Function[] | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline[], pipelineData?: object, copyData?: boolean): this;

            /**
             * Adds an entry to the `pipelineData` object belonging to this Game Object.
             * 
             * If the 'key' already exists, its value is updated. If it doesn't exist, it is created.
             * 
             * If `value` is undefined, and `key` exists, `key` is removed from the data object.
             * 
             * Both the pipeline and post pipelines share the pipeline data object together.
             * @param key The key of the pipeline data to set, update, or delete.
             * @param value The value to be set with the key. If `undefined` then `key` will be deleted from the object.
             */
            setPipelineData(key: string, value?: any): this;

            /**
             * Gets a Post Pipeline instance from this Game Object, based on the given name, and returns it.
             * @param pipeline The string-based name of the pipeline, or a pipeline class.
             */
            getPostPipeline(pipeline: string | Function | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline): Phaser.Renderer.WebGL.Pipelines.PostFXPipeline | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline[];

            /**
             * Resets the WebGL Pipeline of this Game Object back to the default it was created with.
             * @param resetPostPipelines Reset all of the post pipelines? Default false.
             * @param resetData Reset the `pipelineData` object to being an empty object? Default false.
             */
            resetPipeline(resetPostPipelines?: boolean, resetData?: boolean): boolean;

            /**
             * Resets the WebGL Post Pipelines of this Game Object. It does this by calling
             * the `destroy` method on each post pipeline and then clearing the local array.
             * @param resetData Reset the `pipelineData` object to being an empty object? Default false.
             */
            resetPostPipeline(resetData?: boolean): void;

            /**
             * Removes a type of Post Pipeline instances from this Game Object, based on the given name, and destroys them.
             * 
             * If you wish to remove all Post Pipelines use the `resetPostPipeline` method instead.
             * @param pipeline The string-based name of the pipeline, or a pipeline class.
             */
            removePostPipeline(pipeline: string | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline): this;

            /**
             * Gets the name of the WebGL Pipeline this Game Object is currently using.
             */
            getPipelineName(): string;

            /**
             * The x position of this Game Object.
             */
            x: number;

            /**
             * The y position of this Game Object.
             */
            y: number;

            /**
             * The z position of this Game Object.
             * 
             * Note: The z position does not control the rendering order of 2D Game Objects. Use
             * {@link Phaser.GameObjects.Components.Depth#depth} instead.
             */
            z: number;

            /**
             * The w position of this Game Object.
             */
            w: number;

            /**
             * This is a special setter that allows you to set both the horizontal and vertical scale of this Game Object
             * to the same value, at the same time. When reading this value the result returned is `(scaleX + scaleY) / 2`.
             * 
             * Use of this property implies you wish the horizontal and vertical scales to be equal to each other. If this
             * isn't the case, use the `scaleX` or `scaleY` properties instead.
             */
            scale: number;

            /**
             * The horizontal scale of this Game Object.
             */
            scaleX: number;

            /**
             * The vertical scale of this Game Object.
             */
            scaleY: number;

            /**
             * The angle of this Game Object as expressed in degrees.
             * 
             * Phaser uses a right-hand clockwise rotation system, where 0 is right, 90 is down, 180/-180 is left
             * and -90 is up.
             * 
             * If you prefer to work in radians, see the `rotation` property instead.
             */
            angle: number;

            /**
             * The angle of this Game Object in radians.
             * 
             * Phaser uses a right-hand clockwise rotation system, where 0 is right, PI/2 is down, +-PI is left
             * and -PI/2 is up.
             * 
             * If you prefer to work in degrees, see the `angle` property instead.
             */
            rotation: number;

            /**
             * Sets the position of this Game Object.
             * @param x The x position of this Game Object. Default 0.
             * @param y The y position of this Game Object. If not set it will use the `x` value. Default x.
             * @param z The z position of this Game Object. Default 0.
             * @param w The w position of this Game Object. Default 0.
             */
            setPosition(x?: number, y?: number, z?: number, w?: number): this;

            /**
             * Copies an object's coordinates to this Game Object's position.
             * @param source An object with numeric 'x', 'y', 'z', or 'w' properties. Undefined values are not copied.
             */
            copyPosition(source: Phaser.Types.Math.Vector2Like | Phaser.Types.Math.Vector3Like | Phaser.Types.Math.Vector4Like): this;

            /**
             * Sets the position of this Game Object to be a random position within the confines of
             * the given area.
             * 
             * If no area is specified a random position between 0 x 0 and the game width x height is used instead.
             * 
             * The position does not factor in the size of this Game Object, meaning that only the origin is
             * guaranteed to be within the area.
             * @param x The x position of the top-left of the random area. Default 0.
             * @param y The y position of the top-left of the random area. Default 0.
             * @param width The width of the random area.
             * @param height The height of the random area.
             */
            setRandomPosition(x?: number, y?: number, width?: number, height?: number): this;

            /**
             * Sets the rotation of this Game Object.
             * @param radians The rotation of this Game Object, in radians. Default 0.
             */
            setRotation(radians?: number): this;

            /**
             * Sets the angle of this Game Object.
             * @param degrees The rotation of this Game Object, in degrees. Default 0.
             */
            setAngle(degrees?: number): this;

            /**
             * Sets the scale of this Game Object.
             * @param x The horizontal scale of this Game Object.
             * @param y The vertical scale of this Game Object. If not set it will use the `x` value. Default x.
             */
            setScale(x: number, y?: number): this;

            /**
             * Sets the x position of this Game Object.
             * @param value The x position of this Game Object. Default 0.
             */
            setX(value?: number): this;

            /**
             * Sets the y position of this Game Object.
             * @param value The y position of this Game Object. Default 0.
             */
            setY(value?: number): this;

            /**
             * Sets the z position of this Game Object.
             * 
             * Note: The z position does not control the rendering order of 2D Game Objects. Use
             * {@link Phaser.GameObjects.Components.Depth#setDepth} instead.
             * @param value The z position of this Game Object. Default 0.
             */
            setZ(value?: number): this;

            /**
             * Sets the w position of this Game Object.
             * @param value The w position of this Game Object. Default 0.
             */
            setW(value?: number): this;

            /**
             * Gets the local transform matrix for this Game Object.
             * @param tempMatrix The matrix to populate with the values from this Game Object.
             */
            getLocalTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;

            /**
             * Gets the world transform matrix for this Game Object, factoring in any parent Containers.
             * @param tempMatrix The matrix to populate with the values from this Game Object.
             * @param parentMatrix A temporary matrix to hold parent values during the calculations.
             */
            getWorldTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix, parentMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;

            /**
             * Takes the given `x` and `y` coordinates and converts them into local space for this
             * Game Object, taking into account parent and local transforms, and the Display Origin.
             * 
             * The returned Vector2 contains the translated point in its properties.
             * 
             * A Camera needs to be provided in order to handle modified scroll factors. If no
             * camera is specified, it will use the `main` camera from the Scene to which this
             * Game Object belongs.
             * @param x The x position to translate.
             * @param y The y position to translate.
             * @param point A Vector2, or point-like object, to store the results in.
             * @param camera The Camera which is being tested against. If not given will use the Scene default camera.
             */
            getLocalPoint(x: number, y: number, point?: Phaser.Math.Vector2, camera?: Phaser.Cameras.Scene2D.Camera): Phaser.Math.Vector2;

            /**
             * Gets the sum total rotation of all of this Game Objects parent Containers.
             * 
             * The returned value is in radians and will be zero if this Game Object has no parent container.
             */
            getParentRotation(): number;

            /**
             * The visible state of the Game Object.
             * 
             * An invisible Game Object will skip rendering, but will still process update logic.
             */
            visible: boolean;

            /**
             * Sets the visibility of this Game Object.
             * 
             * An invisible Game Object will skip rendering, but will still process update logic.
             * @param value The visible state of the Game Object.
             */
            setVisible(value: boolean): this;

        }

        /**
         * The Display List plugin.
         * 
         * Display Lists belong to a Scene and maintain the list of Game Objects to render every frame.
         * 
         * Some of these Game Objects may also be part of the Scene's [Update List]{@link Phaser.GameObjects.UpdateList}, for updating.
         */
        class DisplayList extends Phaser.Structs.List<Phaser.GameObjects.GameObject> {
            /**
             * 
             * @param scene The Scene that this Display List belongs to.
             */
            constructor(scene: Phaser.Scene);

            /**
             * The flag the determines whether Game Objects should be sorted when `depthSort()` is called.
             */
            sortChildrenFlag: boolean;

            /**
             * The Scene that this Display List belongs to.
             */
            scene: Phaser.Scene;

            /**
             * The Scene's Systems.
             */
            systems: Phaser.Scenes.Systems;

            /**
             * The Scene's Event Emitter.
             */
            events: Phaser.Events.EventEmitter;

            /**
             * Force a sort of the display list on the next call to depthSort.
             */
            queueDepthSort(): void;

            /**
             * Immediately sorts the display list if the flag is set.
             */
            depthSort(): void;

            /**
             * Compare the depth of two Game Objects.
             * @param childA The first Game Object.
             * @param childB The second Game Object.
             */
            sortByDepth(childA: Phaser.GameObjects.GameObject, childB: Phaser.GameObjects.GameObject): number;

            /**
             * Returns an array which contains all objects currently on the Display List.
             * This is a reference to the main list array, not a copy of it, so be careful not to modify it.
             */
            getChildren(): Phaser.GameObjects.GameObject[];

        }

        /**
         * DOM Element Game Objects are a way to control and manipulate HTML Elements over the top of your game.
         * 
         * In order for DOM Elements to display you have to enable them by adding the following to your game
         * configuration object:
         * 
         * ```javascript
         * dom {
         *   createContainer: true
         * }
         * ```
         * 
         * When this is added, Phaser will automatically create a DOM Container div that is positioned over the top
         * of the game canvas. This div is sized to match the canvas, and if the canvas size changes, as a result of
         * settings within the Scale Manager, the dom container is resized accordingly.
         * 
         * If you have not already done so, you have to provide a `parent` in the Game Configuration, or the DOM
         * Container will fail to be created.
         * 
         * You can create a DOM Element by either passing in DOMStrings, or by passing in a reference to an existing
         * Element that you wish to be placed under the control of Phaser. For example:
         * 
         * ```javascript
         * this.add.dom(x, y, 'div', 'background-color: lime; width: 220px; height: 100px; font: 48px Arial', 'Phaser');
         * ```
         * 
         * The above code will insert a div element into the DOM Container at the given x/y coordinate. The DOMString in
         * the 4th argument sets the initial CSS style of the div and the final argument is the inner text. In this case,
         * it will create a lime colored div that is 220px by 100px in size with the text Phaser in it, in an Arial font.
         * 
         * You should nearly always, without exception, use explicitly sized HTML Elements, in order to fully control
         * alignment and positioning of the elements next to regular game content.
         * 
         * Rather than specify the CSS and HTML directly you can use the `load.html` File Loader to load it into the
         * cache and then use the `createFromCache` method instead. You can also use `createFromHTML` and various other
         * methods available in this class to help construct your elements.
         * 
         * Once the element has been created you can then control it like you would any other Game Object. You can set its
         * position, scale, rotation, alpha and other properties. It will move as the main Scene Camera moves and be clipped
         * at the edge of the canvas. It's important to remember some limitations of DOM Elements: The obvious one is that
         * they appear above or below your game canvas. You cannot blend them into the display list, meaning you cannot have
         * a DOM Element, then a Sprite, then another DOM Element behind it.
         * 
         * They also cannot be enabled for input. To do that, you have to use the `addListener` method to add native event
         * listeners directly. The final limitation is to do with cameras. The DOM Container is sized to match the game canvas
         * entirely and clipped accordingly. DOM Elements respect camera scrolling and scrollFactor settings, but if you
         * change the size of the camera so it no longer matches the size of the canvas, they won't be clipped accordingly.
         * 
         * Also, all DOM Elements are inserted into the same DOM Container, regardless of which Scene they are created in.
         * 
         * Note that you should only have DOM Elements in a Scene with a _single_ Camera. If you require multiple cameras,
         * use parallel scenes to achieve this.
         * 
         * DOM Elements are a powerful way to align native HTML with your Phaser Game Objects. For example, you can insert
         * a login form for a multiplayer game directly into your title screen. Or a text input box for a highscore table.
         * Or a banner ad from a 3rd party service. Or perhaps you'd like to use them for high resolution text display and
         * UI. The choice is up to you, just remember that you're dealing with standard HTML and CSS floating over the top
         * of your game, and should treat it accordingly.
         */
        class DOMElement extends Phaser.GameObjects.GameObject implements Phaser.GameObjects.Components.AlphaSingle, Phaser.GameObjects.Components.BlendMode, Phaser.GameObjects.Components.Depth, Phaser.GameObjects.Components.Origin, Phaser.GameObjects.Components.ScrollFactor, Phaser.GameObjects.Components.Transform, Phaser.GameObjects.Components.Visible {
            /**
             * 
             * @param scene The Scene to which this Game Object belongs. A Game Object can only belong to one Scene at a time.
             * @param x The horizontal position of this DOM Element in the world. Default 0.
             * @param y The vertical position of this DOM Element in the world. Default 0.
             * @param element An existing DOM element, or a string. If a string starting with a # it will do a `getElementById` look-up on the string (minus the hash). Without a hash, it represents the type of element to create, i.e. 'div'.
             * @param style If a string, will be set directly as the elements `style` property value. If a plain object, will be iterated and the values transferred. In both cases the values replacing whatever CSS styles may have been previously set.
             * @param innerText If given, will be set directly as the elements `innerText` property value, replacing whatever was there before.
             */
            constructor(scene: Phaser.Scene, x?: number, y?: number, element?: Element | string, style?: string | any, innerText?: string);

            /**
             * A reference to the parent DOM Container that the Game instance created when it started.
             */
            parent: Element;

            /**
             * A reference to the HTML Cache.
             */
            cache: Phaser.Cache.BaseCache;

            /**
             * The actual DOM Element that this Game Object is bound to. For example, if you've created a `<div>`
             * then this property is a direct reference to that element within the dom.
             */
            node: Element;

            /**
             * By default a DOM Element will have its transform, display, opacity, zIndex and blend mode properties
             * updated when its rendered. If, for some reason, you don't want any of these changed other than the
             * CSS transform, then set this flag to `true`. When `true` only the CSS Transform is applied and it's
             * up to you to keep track of and set the other properties as required.
             * 
             * This can be handy if, for example, you've a nested DOM Element and you don't want the opacity to be
             * picked-up by any of its children.
             */
            transformOnly: boolean;

            /**
             * The angle, in radians, by which to skew the DOM Element on the horizontal axis.
             * 
             * https://developer.mozilla.org/en-US/docs/Web/CSS/transform
             */
            skewX: number;

            /**
             * The angle, in radians, by which to skew the DOM Element on the vertical axis.
             * 
             * https://developer.mozilla.org/en-US/docs/Web/CSS/transform
             */
            skewY: number;

            /**
             * A Vector4 that contains the 3D rotation of this DOM Element around a fixed axis in 3D space.
             * 
             * All values in the Vector4 are treated as degrees, unless the `rotate3dAngle` property is changed.
             * 
             * For more details see the following MDN page:
             * 
             * https://developer.mozilla.org/en-US/docs/Web/CSS/transform-function/rotate3d
             */
            rotate3d: Phaser.Math.Vector4;

            /**
             * The unit that represents the 3D rotation values. By default this is `deg` for degrees, but can
             * be changed to any supported unit. See this page for further details:
             * 
             * https://developer.mozilla.org/en-US/docs/Web/CSS/transform-function/rotate3d
             */
            rotate3dAngle: string;

            /**
             * Sets the CSS `pointerEvents` attribute on the DOM Element during rendering.
             * 
             * This is 'auto' by default. Changing it may have unintended side-effects with
             * internal Phaser input handling, such as dragging, so only change this if you
             * understand the implications.
             */
            pointerEvents: string;

            /**
             * The native (un-scaled) width of this Game Object.
             * 
             * For a DOM Element this property is read-only.
             * 
             * The property `displayWidth` holds the computed bounds of this DOM Element, factoring in scaling.
             */
            readonly width: number;

            /**
             * The native (un-scaled) height of this Game Object.
             * 
             * For a DOM Element this property is read-only.
             * 
             * The property `displayHeight` holds the computed bounds of this DOM Element, factoring in scaling.
             */
            readonly height: number;

            /**
             * The computed display width of this Game Object, based on the `getBoundingClientRect` DOM call.
             * 
             * The property `width` holds the un-scaled width of this DOM Element.
             */
            readonly displayWidth: number;

            /**
             * The computed display height of this Game Object, based on the `getBoundingClientRect` DOM call.
             * 
             * The property `height` holds the un-scaled height of this DOM Element.
             */
            readonly displayHeight: number;

            /**
             * Sets the horizontal and vertical skew values of this DOM Element.
             * 
             * For more information see: https://developer.mozilla.org/en-US/docs/Web/CSS/transform
             * @param x The angle, in radians, by which to skew the DOM Element on the horizontal axis. Default 0.
             * @param y The angle, in radians, by which to skew the DOM Element on the vertical axis. Default x.
             */
            setSkew(x?: number, y?: number): this;

            /**
             * Sets the perspective CSS property of the _parent DOM Container_. This determines the distance between the z=0
             * plane and the user in order to give a 3D-positioned element some perspective. Each 3D element with
             * z > 0 becomes larger; each 3D-element with z < 0 becomes smaller. The strength of the effect is determined
             * by the value of this property.
             * 
             * For more information see: https://developer.mozilla.org/en-US/docs/Web/CSS/perspective
             * 
             * **Changing this value changes it globally for all DOM Elements, as they all share the same parent container.**
             * @param value The perspective value, in pixels, that determines the distance between the z plane and the user.
             */
            setPerspective(value: number): this;

            /**
             * The perspective CSS property value of the _parent DOM Container_. This determines the distance between the z=0
             * plane and the user in order to give a 3D-positioned element some perspective. Each 3D element with
             * z > 0 becomes larger; each 3D-element with z < 0 becomes smaller. The strength of the effect is determined
             * by the value of this property.
             * 
             * For more information see: https://developer.mozilla.org/en-US/docs/Web/CSS/perspective
             * 
             * **Changing this value changes it globally for all DOM Elements, as they all share the same parent container.**
             */
            perspective: number;

            /**
             * Adds one or more native DOM event listeners onto the underlying Element of this Game Object.
             * The event is then dispatched via this Game Objects standard event emitter.
             * 
             * For example:
             * 
             * ```javascript
             * var div = this.add.dom(x, y, element);
             * 
             * div.addListener('click');
             * 
             * div.on('click', handler);
             * ```
             * @param events The DOM event/s to listen for. You can specify multiple events by separating them with spaces.
             */
            addListener(events: string): this;

            /**
             * Removes one or more native DOM event listeners from the underlying Element of this Game Object.
             * @param events The DOM event/s to stop listening for. You can specify multiple events by separating them with spaces.
             */
            removeListener(events: string): this;

            /**
             * Creates a native DOM Element, adds it to the parent DOM Container and then binds it to this Game Object,
             * so you can control it. The `tagName` should be a string and is passed to `document.createElement`:
             * 
             * ```javascript
             * this.add.dom().createElement('div');
             * ```
             * 
             * For more details on acceptable tag names see: https://developer.mozilla.org/en-US/docs/Web/API/Document/createElement
             * 
             * You can also pass in a DOMString or style object to set the CSS on the created element, and an optional `innerText`
             * value as well. Here is an example of a DOMString:
             * 
             * ```javascript
             * this.add.dom().createElement('div', 'background-color: lime; width: 220px; height: 100px; font: 48px Arial', 'Phaser');
             * ```
             * 
             * And using a style object:
             * 
             * ```javascript
             * var style = {
             *   'background-color': 'lime';
             *   'width': '200px';
             *   'height': '100px';
             *   'font': '48px Arial';
             * };
             * 
             * this.add.dom().createElement('div', style, 'Phaser');
             * ```
             * 
             * If this Game Object already has an Element, it is removed from the DOM entirely first.
             * Any event listeners you may have previously created will need to be re-created after this call.
             * @param tagName A string that specifies the type of element to be created. The nodeName of the created element is initialized with the value of tagName. Don't use qualified names (like "html:a") with this method.
             * @param style Either a DOMString that holds the CSS styles to be applied to the created element, or an object the styles will be ready from.
             * @param innerText A DOMString that holds the text that will be set as the innerText of the created element.
             */
            createElement(tagName: string, style?: string | any, innerText?: string): this;

            /**
             * Binds a new DOM Element to this Game Object. If this Game Object already has an Element it is removed from the DOM
             * entirely first. Any event listeners you may have previously created will need to be re-created on the new element.
             * 
             * The `element` argument you pass to this method can be either a string tagName:
             * 
             * ```javascript
             * <h1 id="heading">Phaser</h1>
             * 
             * this.add.dom().setElement('heading');
             * ```
             * 
             * Or a reference to an Element instance:
             * 
             * ```javascript
             * <h1 id="heading">Phaser</h1>
             * 
             * var h1 = document.getElementById('heading');
             * 
             * this.add.dom().setElement(h1);
             * ```
             * 
             * You can also pass in a DOMString or style object to set the CSS on the created element, and an optional `innerText`
             * value as well. Here is an example of a DOMString:
             * 
             * ```javascript
             * this.add.dom().setElement(h1, 'background-color: lime; width: 220px; height: 100px; font: 48px Arial', 'Phaser');
             * ```
             * 
             * And using a style object:
             * 
             * ```javascript
             * var style = {
             *   'background-color': 'lime';
             *   'width': '200px';
             *   'height': '100px';
             *   'font': '48px Arial';
             * };
             * 
             * this.add.dom().setElement(h1, style, 'Phaser');
             * ```
             * @param element If a string it is passed to `getElementById()`, or it should be a reference to an existing Element.
             * @param style Either a DOMString that holds the CSS styles to be applied to the created element, or an object the styles will be ready from.
             * @param innerText A DOMString that holds the text that will be set as the innerText of the created element.
             */
            setElement(element: string | Element, style?: string | any, innerText?: string): this;

            /**
             * Takes a block of html from the HTML Cache, that has previously been preloaded into the game, and then
             * creates a DOM Element from it. The loaded HTML is set as the `innerHTML` property of the created
             * element.
             * 
             * Assume the following html is stored in a file called `loginform.html`:
             * 
             * ```html
             * <input type="text" name="nameField" placeholder="Enter your name" style="font-size: 32px">
             * <input type="button" name="playButton" value="Let's Play" style="font-size: 32px">
             * ```
             * 
             * Which is loaded into your game using the cache key 'login':
             * 
             * ```javascript
             * this.load.html('login', 'assets/loginform.html');
             * ```
             * 
             * You can create a DOM Element from it using the cache key:
             * 
             * ```javascript
             * this.add.dom().createFromCache('login');
             * ```
             * 
             * The optional `elementType` argument controls the container that is created, into which the loaded html is inserted.
             * The default is a plain `div` object, but any valid tagName can be given.
             * 
             * If this Game Object already has an Element, it is removed from the DOM entirely first.
             * Any event listeners you may have previously created will need to be re-created after this call.
             * @param The key of the html cache entry to use for this DOM Element.
             * @param tagName The tag name of the element into which all of the loaded html will be inserted. Defaults to a plain div tag. Default 'div'.
             */
            createFromCache(The: string, tagName?: string): this;

            /**
             * Takes a string of html and then creates a DOM Element from it. The HTML is set as the `innerHTML`
             * property of the created element.
             * 
             * ```javascript
             * let form = `
             * <input type="text" name="nameField" placeholder="Enter your name" style="font-size: 32px">
             * <input type="button" name="playButton" value="Let's Play" style="font-size: 32px">
             * `;
             * ```
             * 
             * You can create a DOM Element from it using the string:
             * 
             * ```javascript
             * this.add.dom().createFromHTML(form);
             * ```
             * 
             * The optional `elementType` argument controls the type of container that is created, into which the html is inserted.
             * The default is a plain `div` object, but any valid tagName can be given.
             * 
             * If this Game Object already has an Element, it is removed from the DOM entirely first.
             * Any event listeners you may have previously created will need to be re-created after this call.
             * @param html A string of html to be set as the `innerHTML` property of the created element.
             * @param tagName The tag name of the element into which all of the html will be inserted. Defaults to a plain div tag. Default 'div'.
             */
            createFromHTML(html: string, tagName?: string): this;

            /**
             * Removes the current DOM Element bound to this Game Object from the DOM entirely and resets the
             * `node` property of this Game Object to be `null`.
             */
            removeElement(): this;

            /**
             * Internal method that calls `getBoundingClientRect` on the `node` and then sets the bounds width
             * and height into the `displayWidth` and `displayHeight` properties, and the `clientWidth` and `clientHeight`
             * values into the `width` and `height` properties respectively.
             * 
             * This is called automatically whenever a new element is created or set.
             */
            updateSize(): this;

            /**
             * Gets all children from this DOM Elements node, using `querySelectorAll('*')` and then iterates through
             * them, looking for the first one that has a property matching the given key and value. It then returns this child
             * if found, or `null` if not.
             * @param property The property to search the children for.
             * @param value The value the property must strictly equal.
             */
            getChildByProperty(property: string, value: string): Element;

            /**
             * Gets all children from this DOM Elements node, using `querySelectorAll('*')` and then iterates through
             * them, looking for the first one that has a matching id. It then returns this child if found, or `null` if not.
             * 
             * Be aware that class and id names are case-sensitive.
             * @param id The id to search the children for.
             */
            getChildByID(id: string): Element;

            /**
             * Gets all children from this DOM Elements node, using `querySelectorAll('*')` and then iterates through
             * them, looking for the first one that has a matching name. It then returns this child if found, or `null` if not.
             * 
             * Be aware that class and id names are case-sensitive.
             * @param name The name to search the children for.
             */
            getChildByName(name: string): Element;

            /**
             * Sets the `className` property of the DOM Element node and updates the internal sizes.
             * @param className A string representing the class or space-separated classes of the element.
             */
            setClassName(className: string): this;

            /**
             * Sets the `innerText` property of the DOM Element node and updates the internal sizes.
             * 
             * Note that only certain types of Elements can have `innerText` set on them.
             * @param text A DOMString representing the rendered text content of the element.
             */
            setText(text: string): this;

            /**
             * Sets the `innerHTML` property of the DOM Element node and updates the internal sizes.
             * @param html A DOMString of html to be set as the `innerHTML` property of the element.
             */
            setHTML(html: string): this;

            /**
             * Compares the renderMask with the renderFlags to see if this Game Object will render or not.
             * 
             * DOMElements always return `true` as they need to still set values during the render pass, even if not visible.
             */
            willRender(): boolean;

            /**
             * Clears all alpha values associated with this Game Object.
             * 
             * Immediately sets the alpha levels back to 1 (fully opaque).
             */
            clearAlpha(): this;

            /**
             * Set the Alpha level of this Game Object. The alpha controls the opacity of the Game Object as it renders.
             * Alpha values are provided as a float between 0, fully transparent, and 1, fully opaque.
             * @param value The alpha value applied across the whole Game Object. Default 1.
             */
            setAlpha(value?: number): this;

            /**
             * The alpha value of the Game Object.
             * 
             * This is a global value, impacting the entire Game Object, not just a region of it.
             */
            alpha: number;

            /**
             * Sets the Blend Mode being used by this Game Object.
             * 
             * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
             * 
             * Under WebGL only the following Blend Modes are available:
             * 
             * * ADD
             * * MULTIPLY
             * * SCREEN
             * * ERASE
             * 
             * Canvas has more available depending on browser support.
             * 
             * You can also create your own custom Blend Modes in WebGL.
             * 
             * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
             * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
             * reasons try to be careful about the construction of your Scene and the frequency of which blend modes
             * are used.
             */
            blendMode: Phaser.BlendModes | string;

            /**
             * Sets the Blend Mode being used by this Game Object.
             * 
             * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
             * 
             * Under WebGL only the following Blend Modes are available:
             * 
             * * ADD
             * * MULTIPLY
             * * SCREEN
             * * ERASE (only works when rendering to a framebuffer, like a Render Texture)
             * 
             * Canvas has more available depending on browser support.
             * 
             * You can also create your own custom Blend Modes in WebGL.
             * 
             * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
             * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
             * reasons try to be careful about the construction of your Scene and the frequency in which blend modes
             * are used.
             * @param value The BlendMode value. Either a string or a CONST.
             */
            setBlendMode(value: string | Phaser.BlendModes): this;

            /**
             * The depth of this Game Object within the Scene.
             * 
             * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
             * of Game Objects, without actually moving their position in the display list.
             * 
             * The default depth is zero. A Game Object with a higher depth
             * value will always render in front of one with a lower value.
             * 
             * Setting the depth will queue a depth sort event within the Scene.
             */
            depth: number;

            /**
             * The depth of this Game Object within the Scene.
             * 
             * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
             * of Game Objects, without actually moving their position in the display list.
             * 
             * The default depth is zero. A Game Object with a higher depth
             * value will always render in front of one with a lower value.
             * 
             * Setting the depth will queue a depth sort event within the Scene.
             * @param value The depth of this Game Object.
             */
            setDepth(value: number): this;

            /**
             * The horizontal origin of this Game Object.
             * The origin maps the relationship between the size and position of the Game Object.
             * The default value is 0.5, meaning all Game Objects are positioned based on their center.
             * Setting the value to 0 means the position now relates to the left of the Game Object.
             */
            originX: number;

            /**
             * The vertical origin of this Game Object.
             * The origin maps the relationship between the size and position of the Game Object.
             * The default value is 0.5, meaning all Game Objects are positioned based on their center.
             * Setting the value to 0 means the position now relates to the top of the Game Object.
             */
            originY: number;

            /**
             * The horizontal display origin of this Game Object.
             * The origin is a normalized value between 0 and 1.
             * The displayOrigin is a pixel value, based on the size of the Game Object combined with the origin.
             */
            displayOriginX: number;

            /**
             * The vertical display origin of this Game Object.
             * The origin is a normalized value between 0 and 1.
             * The displayOrigin is a pixel value, based on the size of the Game Object combined with the origin.
             */
            displayOriginY: number;

            /**
             * Sets the origin of this Game Object.
             * 
             * The values are given in the range 0 to 1.
             * @param x The horizontal origin value. Default 0.5.
             * @param y The vertical origin value. If not defined it will be set to the value of `x`. Default x.
             */
            setOrigin(x?: number, y?: number): this;

            /**
             * Sets the origin of this Game Object based on the Pivot values in its Frame.
             */
            setOriginFromFrame(): this;

            /**
             * Sets the display origin of this Game Object.
             * The difference between this and setting the origin is that you can use pixel values for setting the display origin.
             * @param x The horizontal display origin value. Default 0.
             * @param y The vertical display origin value. If not defined it will be set to the value of `x`. Default x.
             */
            setDisplayOrigin(x?: number, y?: number): this;

            /**
             * Updates the Display Origin cached values internally stored on this Game Object.
             * You don't usually call this directly, but it is exposed for edge-cases where you may.
             */
            updateDisplayOrigin(): this;

            /**
             * The horizontal scroll factor of this Game Object.
             * 
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             * 
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             * 
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             * 
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             */
            scrollFactorX: number;

            /**
             * The vertical scroll factor of this Game Object.
             * 
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             * 
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             * 
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             * 
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             */
            scrollFactorY: number;

            /**
             * Sets the scroll factor of this Game Object.
             * 
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             * 
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             * 
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             * 
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             * @param x The horizontal scroll factor of this Game Object.
             * @param y The vertical scroll factor of this Game Object. If not set it will use the `x` value. Default x.
             */
            setScrollFactor(x: number, y?: number): this;

            /**
             * The x position of this Game Object.
             */
            x: number;

            /**
             * The y position of this Game Object.
             */
            y: number;

            /**
             * The z position of this Game Object.
             * 
             * Note: The z position does not control the rendering order of 2D Game Objects. Use
             * {@link Phaser.GameObjects.Components.Depth#depth} instead.
             */
            z: number;

            /**
             * The w position of this Game Object.
             */
            w: number;

            /**
             * This is a special setter that allows you to set both the horizontal and vertical scale of this Game Object
             * to the same value, at the same time. When reading this value the result returned is `(scaleX + scaleY) / 2`.
             * 
             * Use of this property implies you wish the horizontal and vertical scales to be equal to each other. If this
             * isn't the case, use the `scaleX` or `scaleY` properties instead.
             */
            scale: number;

            /**
             * The horizontal scale of this Game Object.
             */
            scaleX: number;

            /**
             * The vertical scale of this Game Object.
             */
            scaleY: number;

            /**
             * The angle of this Game Object as expressed in degrees.
             * 
             * Phaser uses a right-hand clockwise rotation system, where 0 is right, 90 is down, 180/-180 is left
             * and -90 is up.
             * 
             * If you prefer to work in radians, see the `rotation` property instead.
             */
            angle: number;

            /**
             * The angle of this Game Object in radians.
             * 
             * Phaser uses a right-hand clockwise rotation system, where 0 is right, PI/2 is down, +-PI is left
             * and -PI/2 is up.
             * 
             * If you prefer to work in degrees, see the `angle` property instead.
             */
            rotation: number;

            /**
             * Sets the position of this Game Object.
             * @param x The x position of this Game Object. Default 0.
             * @param y The y position of this Game Object. If not set it will use the `x` value. Default x.
             * @param z The z position of this Game Object. Default 0.
             * @param w The w position of this Game Object. Default 0.
             */
            setPosition(x?: number, y?: number, z?: number, w?: number): this;

            /**
             * Copies an object's coordinates to this Game Object's position.
             * @param source An object with numeric 'x', 'y', 'z', or 'w' properties. Undefined values are not copied.
             */
            copyPosition(source: Phaser.Types.Math.Vector2Like | Phaser.Types.Math.Vector3Like | Phaser.Types.Math.Vector4Like): this;

            /**
             * Sets the position of this Game Object to be a random position within the confines of
             * the given area.
             * 
             * If no area is specified a random position between 0 x 0 and the game width x height is used instead.
             * 
             * The position does not factor in the size of this Game Object, meaning that only the origin is
             * guaranteed to be within the area.
             * @param x The x position of the top-left of the random area. Default 0.
             * @param y The y position of the top-left of the random area. Default 0.
             * @param width The width of the random area.
             * @param height The height of the random area.
             */
            setRandomPosition(x?: number, y?: number, width?: number, height?: number): this;

            /**
             * Sets the rotation of this Game Object.
             * @param radians The rotation of this Game Object, in radians. Default 0.
             */
            setRotation(radians?: number): this;

            /**
             * Sets the angle of this Game Object.
             * @param degrees The rotation of this Game Object, in degrees. Default 0.
             */
            setAngle(degrees?: number): this;

            /**
             * Sets the scale of this Game Object.
             * @param x The horizontal scale of this Game Object.
             * @param y The vertical scale of this Game Object. If not set it will use the `x` value. Default x.
             */
            setScale(x: number, y?: number): this;

            /**
             * Sets the x position of this Game Object.
             * @param value The x position of this Game Object. Default 0.
             */
            setX(value?: number): this;

            /**
             * Sets the y position of this Game Object.
             * @param value The y position of this Game Object. Default 0.
             */
            setY(value?: number): this;

            /**
             * Sets the z position of this Game Object.
             * 
             * Note: The z position does not control the rendering order of 2D Game Objects. Use
             * {@link Phaser.GameObjects.Components.Depth#setDepth} instead.
             * @param value The z position of this Game Object. Default 0.
             */
            setZ(value?: number): this;

            /**
             * Sets the w position of this Game Object.
             * @param value The w position of this Game Object. Default 0.
             */
            setW(value?: number): this;

            /**
             * Gets the local transform matrix for this Game Object.
             * @param tempMatrix The matrix to populate with the values from this Game Object.
             */
            getLocalTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;

            /**
             * Gets the world transform matrix for this Game Object, factoring in any parent Containers.
             * @param tempMatrix The matrix to populate with the values from this Game Object.
             * @param parentMatrix A temporary matrix to hold parent values during the calculations.
             */
            getWorldTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix, parentMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;

            /**
             * Takes the given `x` and `y` coordinates and converts them into local space for this
             * Game Object, taking into account parent and local transforms, and the Display Origin.
             * 
             * The returned Vector2 contains the translated point in its properties.
             * 
             * A Camera needs to be provided in order to handle modified scroll factors. If no
             * camera is specified, it will use the `main` camera from the Scene to which this
             * Game Object belongs.
             * @param x The x position to translate.
             * @param y The y position to translate.
             * @param point A Vector2, or point-like object, to store the results in.
             * @param camera The Camera which is being tested against. If not given will use the Scene default camera.
             */
            getLocalPoint(x: number, y: number, point?: Phaser.Math.Vector2, camera?: Phaser.Cameras.Scene2D.Camera): Phaser.Math.Vector2;

            /**
             * Gets the sum total rotation of all of this Game Objects parent Containers.
             * 
             * The returned value is in radians and will be zero if this Game Object has no parent container.
             */
            getParentRotation(): number;

            /**
             * The visible state of the Game Object.
             * 
             * An invisible Game Object will skip rendering, but will still process update logic.
             */
            visible: boolean;

            /**
             * Sets the visibility of this Game Object.
             * 
             * An invisible Game Object will skip rendering, but will still process update logic.
             * @param value The visible state of the Game Object.
             */
            setVisible(value: boolean): this;

        }

        namespace Events {
            /**
             * The Game Object Added to Scene Event.
             * 
             * This event is dispatched when a Game Object is added to a Scene.
             * 
             * Listen for it on a Game Object instance using `GameObject.on('addedtoscene', listener)`.
             */
            const ADDED_TO_SCENE: any;

            /**
             * The Game Object Destroy Event.
             * 
             * This event is dispatched when a Game Object instance is being destroyed.
             * 
             * Listen for it on a Game Object instance using `GameObject.on('destroy', listener)`.
             */
            const DESTROY: any;

            /**
             * The Game Object Removed from Scene Event.
             * 
             * This event is dispatched when a Game Object is removed from a Scene.
             * 
             * Listen for it on a Game Object instance using `GameObject.on('removedfromscene', listener)`.
             */
            const REMOVED_FROM_SCENE: any;

            /**
             * The Video Game Object Complete Event.
             * 
             * This event is dispatched when a Video finishes playback by reaching the end of its duration. It
             * is also dispatched if a video marker sequence is being played and reaches the end.
             * 
             * Note that not all videos can fire this event. Live streams, for example, have no fixed duration,
             * so never technically 'complete'.
             * 
             * If a video is stopped from playback, via the `Video.stop` method, it will emit the
             * `VIDEO_STOP` event instead of this one.
             * 
             * Listen for it from a Video Game Object instance using `Video.on('complete', listener)`.
             */
            const VIDEO_COMPLETE: any;

            /**
             * The Video Game Object Created Event.
             * 
             * This event is dispatched when the texture for a Video has been created. This happens
             * when enough of the video source has been loaded that the browser is able to render a
             * frame from it.
             * 
             * Listen for it from a Video Game Object instance using `Video.on('created', listener)`.
             */
            const VIDEO_CREATED: any;

            /**
             * The Video Game Object Error Event.
             * 
             * This event is dispatched when a Video tries to play a source that does not exist, or is the wrong file type.
             * 
             * Listen for it from a Video Game Object instance using `Video.on('error', listener)`.
             */
            const VIDEO_ERROR: any;

            /**
             * The Video Game Object Loop Event.
             * 
             * This event is dispatched when a Video that is currently playing has looped. This only
             * happens if the `loop` parameter was specified, or the `setLoop` method was called,
             * and if the video has a fixed duration. Video streams, for example, cannot loop, as
             * they have no duration.
             * 
             * Looping is based on the result of the Video `timeupdate` event. This event is not
             * frame-accurate, due to the way browsers work, so please do not rely on this loop
             * event to be time or frame precise.
             * 
             * Listen for it from a Video Game Object instance using `Video.on('loop', listener)`.
             */
            const VIDEO_LOOP: any;

            /**
             * The Video Game Object Play Event.
             * 
             * This event is dispatched when a Video begins playback. For videos that do not require
             * interaction unlocking, this is usually as soon as the `Video.play` method is called.
             * However, for videos that require unlocking, it is fired once playback begins after
             * they've been unlocked.
             * 
             * Listen for it from a Video Game Object instance using `Video.on('play', listener)`.
             */
            const VIDEO_PLAY: any;

            /**
             * The Video Game Object Seeked Event.
             * 
             * This event is dispatched when a Video completes seeking to a new point in its timeline.
             * 
             * Listen for it from a Video Game Object instance using `Video.on('seeked', listener)`.
             */
            const VIDEO_SEEKED: any;

            /**
             * The Video Game Object Seeking Event.
             * 
             * This event is dispatched when a Video _begins_ seeking to a new point in its timeline.
             * When the seek is complete, it will dispatch the `VIDEO_SEEKED` event to conclude.
             * 
             * Listen for it from a Video Game Object instance using `Video.on('seeking', listener)`.
             */
            const VIDEO_SEEKING: any;

            /**
             * The Video Game Object Stopped Event.
             * 
             * This event is dispatched when a Video is stopped from playback via a call to the `Video.stop` method,
             * either directly via game code, or indirectly as the result of changing a video source or destroying it.
             * 
             * Listen for it from a Video Game Object instance using `Video.on('stop', listener)`.
             */
            const VIDEO_STOP: any;

            /**
             * The Video Game Object Timeout Event.
             * 
             * This event is dispatched when a Video has exhausted its allocated time while trying to connect to a video
             * source to start playback.
             * 
             * Listen for it from a Video Game Object instance using `Video.on('timeout', listener)`.
             */
            const VIDEO_TIMEOUT: any;

            /**
             * The Video Game Object Unlocked Event.
             * 
             * This event is dispatched when a Video that was prevented from playback due to the browsers
             * Media Engagement Interaction policy, is unlocked by a user gesture.
             * 
             * Listen for it from a Video Game Object instance using `Video.on('unlocked', listener)`.
             */
            const VIDEO_UNLOCKED: any;

        }

        /**
         * An Extern Game Object is a special type of Game Object that allows you to pass
         * rendering off to a 3rd party.
         * 
         * When you create an Extern and place it in the display list of a Scene, the renderer will
         * process the list as usual. When it finds an Extern it will flush the current batch,
         * clear down the pipeline and prepare a transform matrix which your render function can
         * take advantage of, if required.
         * 
         * The WebGL context is then left in a 'clean' state, ready for you to bind your own shaders,
         * or draw to it, whatever you wish to do. This should all take place in the `render` method.
         * The correct way to deploy an Extern object is to create a class that extends it, then
         * override the `render` (and optionally `preUpdate`) methods and pass off control to your
         * 3rd party libraries or custom WebGL code there.
         * 
         * Once you've finished, you should free-up any of your resources.
         * The Extern will then rebind the Phaser pipeline and carry on rendering the display list.
         * 
         * Although this object has lots of properties such as Alpha, Blend Mode and Tint, none of
         * them are used during rendering unless you take advantage of them in your own render code.
         */
        class Extern extends Phaser.GameObjects.GameObject implements Phaser.GameObjects.Components.Alpha, Phaser.GameObjects.Components.BlendMode, Phaser.GameObjects.Components.Depth, Phaser.GameObjects.Components.Flip, Phaser.GameObjects.Components.Origin, Phaser.GameObjects.Components.ScrollFactor, Phaser.GameObjects.Components.Size, Phaser.GameObjects.Components.Texture, Phaser.GameObjects.Components.Tint, Phaser.GameObjects.Components.Transform, Phaser.GameObjects.Components.Visible {
            /**
             * 
             * @param scene The Scene to which this Game Object belongs. A Game Object can only belong to one Scene at a time.
             */
            constructor(scene: Phaser.Scene);

            /**
             * Clears all alpha values associated with this Game Object.
             * 
             * Immediately sets the alpha levels back to 1 (fully opaque).
             */
            clearAlpha(): this;

            /**
             * Set the Alpha level of this Game Object. The alpha controls the opacity of the Game Object as it renders.
             * Alpha values are provided as a float between 0, fully transparent, and 1, fully opaque.
             * 
             * If your game is running under WebGL you can optionally specify four different alpha values, each of which
             * correspond to the four corners of the Game Object. Under Canvas only the `topLeft` value given is used.
             * @param topLeft The alpha value used for the top-left of the Game Object. If this is the only value given it's applied across the whole Game Object. Default 1.
             * @param topRight The alpha value used for the top-right of the Game Object. WebGL only.
             * @param bottomLeft The alpha value used for the bottom-left of the Game Object. WebGL only.
             * @param bottomRight The alpha value used for the bottom-right of the Game Object. WebGL only.
             */
            setAlpha(topLeft?: number, topRight?: number, bottomLeft?: number, bottomRight?: number): this;

            /**
             * The alpha value of the Game Object.
             * 
             * This is a global value, impacting the entire Game Object, not just a region of it.
             */
            alpha: number;

            /**
             * The alpha value starting from the top-left of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaTopLeft: number;

            /**
             * The alpha value starting from the top-right of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaTopRight: number;

            /**
             * The alpha value starting from the bottom-left of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaBottomLeft: number;

            /**
             * The alpha value starting from the bottom-right of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaBottomRight: number;

            /**
             * Sets the Blend Mode being used by this Game Object.
             * 
             * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
             * 
             * Under WebGL only the following Blend Modes are available:
             * 
             * * ADD
             * * MULTIPLY
             * * SCREEN
             * * ERASE
             * 
             * Canvas has more available depending on browser support.
             * 
             * You can also create your own custom Blend Modes in WebGL.
             * 
             * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
             * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
             * reasons try to be careful about the construction of your Scene and the frequency of which blend modes
             * are used.
             */
            blendMode: Phaser.BlendModes | string;

            /**
             * Sets the Blend Mode being used by this Game Object.
             * 
             * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
             * 
             * Under WebGL only the following Blend Modes are available:
             * 
             * * ADD
             * * MULTIPLY
             * * SCREEN
             * * ERASE (only works when rendering to a framebuffer, like a Render Texture)
             * 
             * Canvas has more available depending on browser support.
             * 
             * You can also create your own custom Blend Modes in WebGL.
             * 
             * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
             * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
             * reasons try to be careful about the construction of your Scene and the frequency in which blend modes
             * are used.
             * @param value The BlendMode value. Either a string or a CONST.
             */
            setBlendMode(value: string | Phaser.BlendModes): this;

            /**
             * The depth of this Game Object within the Scene.
             * 
             * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
             * of Game Objects, without actually moving their position in the display list.
             * 
             * The default depth is zero. A Game Object with a higher depth
             * value will always render in front of one with a lower value.
             * 
             * Setting the depth will queue a depth sort event within the Scene.
             */
            depth: number;

            /**
             * The depth of this Game Object within the Scene.
             * 
             * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
             * of Game Objects, without actually moving their position in the display list.
             * 
             * The default depth is zero. A Game Object with a higher depth
             * value will always render in front of one with a lower value.
             * 
             * Setting the depth will queue a depth sort event within the Scene.
             * @param value The depth of this Game Object.
             */
            setDepth(value: number): this;

            /**
             * The horizontally flipped state of the Game Object.
             * 
             * A Game Object that is flipped horizontally will render inversed on the horizontal axis.
             * Flipping always takes place from the middle of the texture and does not impact the scale value.
             * If this Game Object has a physics body, it will not change the body. This is a rendering toggle only.
             */
            flipX: boolean;

            /**
             * The vertically flipped state of the Game Object.
             * 
             * A Game Object that is flipped vertically will render inversed on the vertical axis (i.e. upside down)
             * Flipping always takes place from the middle of the texture and does not impact the scale value.
             * If this Game Object has a physics body, it will not change the body. This is a rendering toggle only.
             */
            flipY: boolean;

            /**
             * Toggles the horizontal flipped state of this Game Object.
             * 
             * A Game Object that is flipped horizontally will render inversed on the horizontal axis.
             * Flipping always takes place from the middle of the texture and does not impact the scale value.
             * If this Game Object has a physics body, it will not change the body. This is a rendering toggle only.
             */
            toggleFlipX(): this;

            /**
             * Toggles the vertical flipped state of this Game Object.
             */
            toggleFlipY(): this;

            /**
             * Sets the horizontal flipped state of this Game Object.
             * 
             * A Game Object that is flipped horizontally will render inversed on the horizontal axis.
             * Flipping always takes place from the middle of the texture and does not impact the scale value.
             * If this Game Object has a physics body, it will not change the body. This is a rendering toggle only.
             * @param value The flipped state. `false` for no flip, or `true` to be flipped.
             */
            setFlipX(value: boolean): this;

            /**
             * Sets the vertical flipped state of this Game Object.
             * @param value The flipped state. `false` for no flip, or `true` to be flipped.
             */
            setFlipY(value: boolean): this;

            /**
             * Sets the horizontal and vertical flipped state of this Game Object.
             * 
             * A Game Object that is flipped will render inversed on the flipped axis.
             * Flipping always takes place from the middle of the texture and does not impact the scale value.
             * If this Game Object has a physics body, it will not change the body. This is a rendering toggle only.
             * @param x The horizontal flipped state. `false` for no flip, or `true` to be flipped.
             * @param y The horizontal flipped state. `false` for no flip, or `true` to be flipped.
             */
            setFlip(x: boolean, y: boolean): this;

            /**
             * Resets the horizontal and vertical flipped state of this Game Object back to their default un-flipped state.
             */
            resetFlip(): this;

            /**
             * The horizontal origin of this Game Object.
             * The origin maps the relationship between the size and position of the Game Object.
             * The default value is 0.5, meaning all Game Objects are positioned based on their center.
             * Setting the value to 0 means the position now relates to the left of the Game Object.
             */
            originX: number;

            /**
             * The vertical origin of this Game Object.
             * The origin maps the relationship between the size and position of the Game Object.
             * The default value is 0.5, meaning all Game Objects are positioned based on their center.
             * Setting the value to 0 means the position now relates to the top of the Game Object.
             */
            originY: number;

            /**
             * The horizontal display origin of this Game Object.
             * The origin is a normalized value between 0 and 1.
             * The displayOrigin is a pixel value, based on the size of the Game Object combined with the origin.
             */
            displayOriginX: number;

            /**
             * The vertical display origin of this Game Object.
             * The origin is a normalized value between 0 and 1.
             * The displayOrigin is a pixel value, based on the size of the Game Object combined with the origin.
             */
            displayOriginY: number;

            /**
             * Sets the origin of this Game Object.
             * 
             * The values are given in the range 0 to 1.
             * @param x The horizontal origin value. Default 0.5.
             * @param y The vertical origin value. If not defined it will be set to the value of `x`. Default x.
             */
            setOrigin(x?: number, y?: number): this;

            /**
             * Sets the origin of this Game Object based on the Pivot values in its Frame.
             */
            setOriginFromFrame(): this;

            /**
             * Sets the display origin of this Game Object.
             * The difference between this and setting the origin is that you can use pixel values for setting the display origin.
             * @param x The horizontal display origin value. Default 0.
             * @param y The vertical display origin value. If not defined it will be set to the value of `x`. Default x.
             */
            setDisplayOrigin(x?: number, y?: number): this;

            /**
             * Updates the Display Origin cached values internally stored on this Game Object.
             * You don't usually call this directly, but it is exposed for edge-cases where you may.
             */
            updateDisplayOrigin(): this;

            /**
             * The horizontal scroll factor of this Game Object.
             * 
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             * 
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             * 
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             * 
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             */
            scrollFactorX: number;

            /**
             * The vertical scroll factor of this Game Object.
             * 
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             * 
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             * 
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             * 
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             */
            scrollFactorY: number;

            /**
             * Sets the scroll factor of this Game Object.
             * 
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             * 
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             * 
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             * 
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             * @param x The horizontal scroll factor of this Game Object.
             * @param y The vertical scroll factor of this Game Object. If not set it will use the `x` value. Default x.
             */
            setScrollFactor(x: number, y?: number): this;

            /**
             * The native (un-scaled) width of this Game Object.
             * 
             * Changing this value will not change the size that the Game Object is rendered in-game.
             * For that you need to either set the scale of the Game Object (`setScale`) or use
             * the `displayWidth` property.
             */
            width: number;

            /**
             * The native (un-scaled) height of this Game Object.
             * 
             * Changing this value will not change the size that the Game Object is rendered in-game.
             * For that you need to either set the scale of the Game Object (`setScale`) or use
             * the `displayHeight` property.
             */
            height: number;

            /**
             * The displayed width of this Game Object.
             * 
             * This value takes into account the scale factor.
             * 
             * Setting this value will adjust the Game Object's scale property.
             */
            displayWidth: number;

            /**
             * The displayed height of this Game Object.
             * 
             * This value takes into account the scale factor.
             * 
             * Setting this value will adjust the Game Object's scale property.
             */
            displayHeight: number;

            /**
             * Sets the size of this Game Object to be that of the given Frame.
             * 
             * This will not change the size that the Game Object is rendered in-game.
             * For that you need to either set the scale of the Game Object (`setScale`) or call the
             * `setDisplaySize` method, which is the same thing as changing the scale but allows you
             * to do so by giving pixel values.
             * 
             * If you have enabled this Game Object for input, changing the size will _not_ change the
             * size of the hit area. To do this you should adjust the `input.hitArea` object directly.
             * @param frame The frame to base the size of this Game Object on.
             */
            setSizeToFrame(frame: Phaser.Textures.Frame): this;

            /**
             * Sets the internal size of this Game Object, as used for frame or physics body creation.
             * 
             * This will not change the size that the Game Object is rendered in-game.
             * For that you need to either set the scale of the Game Object (`setScale`) or call the
             * `setDisplaySize` method, which is the same thing as changing the scale but allows you
             * to do so by giving pixel values.
             * 
             * If you have enabled this Game Object for input, changing the size will _not_ change the
             * size of the hit area. To do this you should adjust the `input.hitArea` object directly.
             * @param width The width of this Game Object.
             * @param height The height of this Game Object.
             */
            setSize(width: number, height: number): this;

            /**
             * Sets the display size of this Game Object.
             * 
             * Calling this will adjust the scale.
             * @param width The width of this Game Object.
             * @param height The height of this Game Object.
             */
            setDisplaySize(width: number, height: number): this;

            /**
             * The Texture this Game Object is using to render with.
             */
            texture: Phaser.Textures.Texture | Phaser.Textures.CanvasTexture;

            /**
             * The Texture Frame this Game Object is using to render with.
             */
            frame: Phaser.Textures.Frame;

            /**
             * Sets the texture and frame this Game Object will use to render with.
             * 
             * Textures are referenced by their string-based keys, as stored in the Texture Manager.
             * @param key The key of the texture to be used, as stored in the Texture Manager, or a Texture instance.
             * @param frame The name or index of the frame within the Texture.
             */
            setTexture(key: string | Phaser.Textures.Texture, frame?: string | number): this;

            /**
             * Sets the frame this Game Object will use to render with.
             * 
             * The Frame has to belong to the current Texture being used.
             * 
             * It can be either a string or an index.
             * 
             * Calling `setFrame` will modify the `width` and `height` properties of your Game Object.
             * It will also change the `origin` if the Frame has a custom pivot point, as exported from packages like Texture Packer.
             * @param frame The name or index of the frame within the Texture.
             * @param updateSize Should this call adjust the size of the Game Object? Default true.
             * @param updateOrigin Should this call adjust the origin of the Game Object? Default true.
             */
            setFrame(frame: string | number, updateSize?: boolean, updateOrigin?: boolean): this;

            /**
             * The tint value being applied to the top-left vertice of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             * The value should be set as a hex number, i.e. 0xff0000 for red, or 0xff00ff for purple.
             */
            tintTopLeft: number;

            /**
             * The tint value being applied to the top-right vertice of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             * The value should be set as a hex number, i.e. 0xff0000 for red, or 0xff00ff for purple.
             */
            tintTopRight: number;

            /**
             * The tint value being applied to the bottom-left vertice of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             * The value should be set as a hex number, i.e. 0xff0000 for red, or 0xff00ff for purple.
             */
            tintBottomLeft: number;

            /**
             * The tint value being applied to the bottom-right vertice of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             * The value should be set as a hex number, i.e. 0xff0000 for red, or 0xff00ff for purple.
             */
            tintBottomRight: number;

            /**
             * The tint fill mode.
             * 
             * `false` = An additive tint (the default), where vertices colors are blended with the texture.
             * `true` = A fill tint, where the vertices colors replace the texture, but respects texture alpha.
             */
            tintFill: boolean;

            /**
             * Clears all tint values associated with this Game Object.
             * 
             * Immediately sets the color values back to 0xffffff and the tint type to 'additive',
             * which results in no visible change to the texture.
             */
            clearTint(): this;

            /**
             * Sets an additive tint on this Game Object.
             * 
             * The tint works by taking the pixel color values from the Game Objects texture, and then
             * multiplying it by the color value of the tint. You can provide either one color value,
             * in which case the whole Game Object will be tinted in that color. Or you can provide a color
             * per corner. The colors are blended together across the extent of the Game Object.
             * 
             * To modify the tint color once set, either call this method again with new values or use the
             * `tint` property to set all colors at once. Or, use the properties `tintTopLeft`, `tintTopRight,
             * `tintBottomLeft` and `tintBottomRight` to set the corner color values independently.
             * 
             * To remove a tint call `clearTint`.
             * 
             * To swap this from being an additive tint to a fill based tint set the property `tintFill` to `true`.
             * @param topLeft The tint being applied to the top-left of the Game Object. If no other values are given this value is applied evenly, tinting the whole Game Object. Default 0xffffff.
             * @param topRight The tint being applied to the top-right of the Game Object.
             * @param bottomLeft The tint being applied to the bottom-left of the Game Object.
             * @param bottomRight The tint being applied to the bottom-right of the Game Object.
             */
            setTint(topLeft?: number, topRight?: number, bottomLeft?: number, bottomRight?: number): this;

            /**
             * Sets a fill-based tint on this Game Object.
             * 
             * Unlike an additive tint, a fill-tint literally replaces the pixel colors from the texture
             * with those in the tint. You can use this for effects such as making a player flash 'white'
             * if hit by something. You can provide either one color value, in which case the whole
             * Game Object will be rendered in that color. Or you can provide a color per corner. The colors
             * are blended together across the extent of the Game Object.
             * 
             * To modify the tint color once set, either call this method again with new values or use the
             * `tint` property to set all colors at once. Or, use the properties `tintTopLeft`, `tintTopRight,
             * `tintBottomLeft` and `tintBottomRight` to set the corner color values independently.
             * 
             * To remove a tint call `clearTint`.
             * 
             * To swap this from being a fill-tint to an additive tint set the property `tintFill` to `false`.
             * @param topLeft The tint being applied to the top-left of the Game Object. If not other values are given this value is applied evenly, tinting the whole Game Object. Default 0xffffff.
             * @param topRight The tint being applied to the top-right of the Game Object.
             * @param bottomLeft The tint being applied to the bottom-left of the Game Object.
             * @param bottomRight The tint being applied to the bottom-right of the Game Object.
             */
            setTintFill(topLeft?: number, topRight?: number, bottomLeft?: number, bottomRight?: number): this;

            /**
             * The tint value being applied to the whole of the Game Object.
             * This property is a setter-only. Use the properties `tintTopLeft` etc to read the current tint value.
             */
            tint: number;

            /**
             * Does this Game Object have a tint applied?
             * 
             * It checks to see if the 4 tint properties are set to the value 0xffffff
             * and that the `tintFill` property is `false`. This indicates that a Game Object isn't tinted.
             */
            readonly isTinted: boolean;

            /**
             * The x position of this Game Object.
             */
            x: number;

            /**
             * The y position of this Game Object.
             */
            y: number;

            /**
             * The z position of this Game Object.
             * 
             * Note: The z position does not control the rendering order of 2D Game Objects. Use
             * {@link Phaser.GameObjects.Components.Depth#depth} instead.
             */
            z: number;

            /**
             * The w position of this Game Object.
             */
            w: number;

            /**
             * This is a special setter that allows you to set both the horizontal and vertical scale of this Game Object
             * to the same value, at the same time. When reading this value the result returned is `(scaleX + scaleY) / 2`.
             * 
             * Use of this property implies you wish the horizontal and vertical scales to be equal to each other. If this
             * isn't the case, use the `scaleX` or `scaleY` properties instead.
             */
            scale: number;

            /**
             * The horizontal scale of this Game Object.
             */
            scaleX: number;

            /**
             * The vertical scale of this Game Object.
             */
            scaleY: number;

            /**
             * The angle of this Game Object as expressed in degrees.
             * 
             * Phaser uses a right-hand clockwise rotation system, where 0 is right, 90 is down, 180/-180 is left
             * and -90 is up.
             * 
             * If you prefer to work in radians, see the `rotation` property instead.
             */
            angle: number;

            /**
             * The angle of this Game Object in radians.
             * 
             * Phaser uses a right-hand clockwise rotation system, where 0 is right, PI/2 is down, +-PI is left
             * and -PI/2 is up.
             * 
             * If you prefer to work in degrees, see the `angle` property instead.
             */
            rotation: number;

            /**
             * Sets the position of this Game Object.
             * @param x The x position of this Game Object. Default 0.
             * @param y The y position of this Game Object. If not set it will use the `x` value. Default x.
             * @param z The z position of this Game Object. Default 0.
             * @param w The w position of this Game Object. Default 0.
             */
            setPosition(x?: number, y?: number, z?: number, w?: number): this;

            /**
             * Copies an object's coordinates to this Game Object's position.
             * @param source An object with numeric 'x', 'y', 'z', or 'w' properties. Undefined values are not copied.
             */
            copyPosition(source: Phaser.Types.Math.Vector2Like | Phaser.Types.Math.Vector3Like | Phaser.Types.Math.Vector4Like): this;

            /**
             * Sets the position of this Game Object to be a random position within the confines of
             * the given area.
             * 
             * If no area is specified a random position between 0 x 0 and the game width x height is used instead.
             * 
             * The position does not factor in the size of this Game Object, meaning that only the origin is
             * guaranteed to be within the area.
             * @param x The x position of the top-left of the random area. Default 0.
             * @param y The y position of the top-left of the random area. Default 0.
             * @param width The width of the random area.
             * @param height The height of the random area.
             */
            setRandomPosition(x?: number, y?: number, width?: number, height?: number): this;

            /**
             * Sets the rotation of this Game Object.
             * @param radians The rotation of this Game Object, in radians. Default 0.
             */
            setRotation(radians?: number): this;

            /**
             * Sets the angle of this Game Object.
             * @param degrees The rotation of this Game Object, in degrees. Default 0.
             */
            setAngle(degrees?: number): this;

            /**
             * Sets the scale of this Game Object.
             * @param x The horizontal scale of this Game Object.
             * @param y The vertical scale of this Game Object. If not set it will use the `x` value. Default x.
             */
            setScale(x: number, y?: number): this;

            /**
             * Sets the x position of this Game Object.
             * @param value The x position of this Game Object. Default 0.
             */
            setX(value?: number): this;

            /**
             * Sets the y position of this Game Object.
             * @param value The y position of this Game Object. Default 0.
             */
            setY(value?: number): this;

            /**
             * Sets the z position of this Game Object.
             * 
             * Note: The z position does not control the rendering order of 2D Game Objects. Use
             * {@link Phaser.GameObjects.Components.Depth#setDepth} instead.
             * @param value The z position of this Game Object. Default 0.
             */
            setZ(value?: number): this;

            /**
             * Sets the w position of this Game Object.
             * @param value The w position of this Game Object. Default 0.
             */
            setW(value?: number): this;

            /**
             * Gets the local transform matrix for this Game Object.
             * @param tempMatrix The matrix to populate with the values from this Game Object.
             */
            getLocalTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;

            /**
             * Gets the world transform matrix for this Game Object, factoring in any parent Containers.
             * @param tempMatrix The matrix to populate with the values from this Game Object.
             * @param parentMatrix A temporary matrix to hold parent values during the calculations.
             */
            getWorldTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix, parentMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;

            /**
             * Takes the given `x` and `y` coordinates and converts them into local space for this
             * Game Object, taking into account parent and local transforms, and the Display Origin.
             * 
             * The returned Vector2 contains the translated point in its properties.
             * 
             * A Camera needs to be provided in order to handle modified scroll factors. If no
             * camera is specified, it will use the `main` camera from the Scene to which this
             * Game Object belongs.
             * @param x The x position to translate.
             * @param y The y position to translate.
             * @param point A Vector2, or point-like object, to store the results in.
             * @param camera The Camera which is being tested against. If not given will use the Scene default camera.
             */
            getLocalPoint(x: number, y: number, point?: Phaser.Math.Vector2, camera?: Phaser.Cameras.Scene2D.Camera): Phaser.Math.Vector2;

            /**
             * Gets the sum total rotation of all of this Game Objects parent Containers.
             * 
             * The returned value is in radians and will be zero if this Game Object has no parent container.
             */
            getParentRotation(): number;

            /**
             * The visible state of the Game Object.
             * 
             * An invisible Game Object will skip rendering, but will still process update logic.
             */
            visible: boolean;

            /**
             * Sets the visibility of this Game Object.
             * 
             * An invisible Game Object will skip rendering, but will still process update logic.
             * @param value The visible state of the Game Object.
             */
            setVisible(value: boolean): this;

        }

        /**
         * The base class that all Game Objects extend.
         * You don't create GameObjects directly and they cannot be added to the display list.
         * Instead, use them as the base for your own custom classes.
         */
        class GameObject extends Phaser.Events.EventEmitter {
            /**
             * 
             * @param scene The Scene to which this Game Object belongs.
             * @param type A textual representation of the type of Game Object, i.e. `sprite`.
             */
            constructor(scene: Phaser.Scene, type: string);

            /**
             * A reference to the Scene to which this Game Object belongs.
             * 
             * Game Objects can only belong to one Scene.
             * 
             * You should consider this property as being read-only. You cannot move a
             * Game Object to another Scene by simply changing it.
             */
            scene: Phaser.Scene;

            /**
             * Holds a reference to the Display List that contains this Game Object.
             * 
             * This is set automatically when this Game Object is added to a Scene or Layer.
             * 
             * You should treat this property as being read-only.
             */
            displayList: Phaser.GameObjects.DisplayList | Phaser.GameObjects.Layer;

            /**
             * A textual representation of this Game Object, i.e. `sprite`.
             * Used internally by Phaser but is available for your own custom classes to populate.
             */
            type: string;

            /**
             * The current state of this Game Object.
             * 
             * Phaser itself will never modify this value, although plugins may do so.
             * 
             * Use this property to track the state of a Game Object during its lifetime. For example, it could change from
             * a state of 'moving', to 'attacking', to 'dead'. The state value should be an integer (ideally mapped to a constant
             * in your game code), or a string. These are recommended to keep it light and simple, with fast comparisons.
             * If you need to store complex data about your Game Object, look at using the Data Component instead.
             */
            state: number | string;

            /**
             * The parent Container of this Game Object, if it has one.
             */
            parentContainer: Phaser.GameObjects.Container;

            /**
             * The name of this Game Object.
             * Empty by default and never populated by Phaser, this is left for developers to use.
             */
            name: string;

            /**
             * The active state of this Game Object.
             * A Game Object with an active state of `true` is processed by the Scenes UpdateList, if added to it.
             * An active object is one which is having its logic and internal systems updated.
             */
            active: boolean;

            /**
             * The Tab Index of the Game Object.
             * Reserved for future use by plugins and the Input Manager.
             */
            tabIndex: number;

            /**
             * A Data Manager.
             * It allows you to store, query and get key/value paired information specific to this Game Object.
             * `null` by default. Automatically created if you use `getData` or `setData` or `setDataEnabled`.
             */
            data: Phaser.Data.DataManager;

            /**
             * The flags that are compared against `RENDER_MASK` to determine if this Game Object will render or not.
             * The bits are 0001 | 0010 | 0100 | 1000 set by the components Visible, Alpha, Transform and Texture respectively.
             * If those components are not used by your custom class then you can use this bitmask as you wish.
             */
            renderFlags: number;

            /**
             * A bitmask that controls if this Game Object is drawn by a Camera or not.
             * Not usually set directly, instead call `Camera.ignore`, however you can
             * set this property directly using the Camera.id property:
             */
            cameraFilter: number;

            /**
             * If this Game Object is enabled for input then this property will contain an InteractiveObject instance.
             * Not usually set directly. Instead call `GameObject.setInteractive()`.
             */
            input: Phaser.Types.Input.InteractiveObject;

            /**
             * If this Game Object is enabled for Arcade or Matter Physics then this property will contain a reference to a Physics Body.
             */
            body: Phaser.Physics.Arcade.Body | Phaser.Physics.Arcade.StaticBody | MatterJS.BodyType;

            /**
             * This Game Object will ignore all calls made to its destroy method if this flag is set to `true`.
             * This includes calls that may come from a Group, Container or the Scene itself.
             * While it allows you to persist a Game Object across Scenes, please understand you are entirely
             * responsible for managing references to and from this Game Object.
             */
            ignoreDestroy: boolean;

            /**
             * Sets the `active` property of this Game Object and returns this Game Object for further chaining.
             * A Game Object with its `active` property set to `true` will be updated by the Scenes UpdateList.
             * @param value True if this Game Object should be set as active, false if not.
             */
            setActive(value: boolean): this;

            /**
             * Sets the `name` property of this Game Object and returns this Game Object for further chaining.
             * The `name` property is not populated by Phaser and is presented for your own use.
             * @param value The name to be given to this Game Object.
             */
            setName(value: string): this;

            /**
             * Sets the current state of this Game Object.
             * 
             * Phaser itself will never modify the State of a Game Object, although plugins may do so.
             * 
             * For example, a Game Object could change from a state of 'moving', to 'attacking', to 'dead'.
             * The state value should typically be an integer (ideally mapped to a constant
             * in your game code), but could also be a string. It is recommended to keep it light and simple.
             * If you need to store complex data about your Game Object, look at using the Data Component instead.
             * @param value The state of the Game Object.
             */
            setState(value: number | string): this;

            /**
             * Adds a Data Manager component to this Game Object.
             */
            setDataEnabled(): this;

            /**
             * Allows you to store a key value pair within this Game Objects Data Manager.
             * 
             * If the Game Object has not been enabled for data (via `setDataEnabled`) then it will be enabled
             * before setting the value.
             * 
             * If the key doesn't already exist in the Data Manager then it is created.
             * 
             * ```javascript
             * sprite.setData('name', 'Red Gem Stone');
             * ```
             * 
             * You can also pass in an object of key value pairs as the first argument:
             * 
             * ```javascript
             * sprite.setData({ name: 'Red Gem Stone', level: 2, owner: 'Link', gold: 50 });
             * ```
             * 
             * To get a value back again you can call `getData`:
             * 
             * ```javascript
             * sprite.getData('gold');
             * ```
             * 
             * Or you can access the value directly via the `values` property, where it works like any other variable:
             * 
             * ```javascript
             * sprite.data.values.gold += 50;
             * ```
             * 
             * When the value is first set, a `setdata` event is emitted from this Game Object.
             * 
             * If the key already exists, a `changedata` event is emitted instead, along an event named after the key.
             * For example, if you updated an existing key called `PlayerLives` then it would emit the event `changedata-PlayerLives`.
             * These events will be emitted regardless if you use this method to set the value, or the direct `values` setter.
             * 
             * Please note that the data keys are case-sensitive and must be valid JavaScript Object property strings.
             * This means the keys `gold` and `Gold` are treated as two unique values within the Data Manager.
             * @param key The key to set the value for. Or an object of key value pairs. If an object the `data` argument is ignored.
             * @param data The value to set for the given key. If an object is provided as the key this argument is ignored.
             */
            setData(key: string | object, data?: any): this;

            /**
             * Increase a value for the given key within this Game Objects Data Manager. If the key doesn't already exist in the Data Manager then it is increased from 0.
             * 
             * If the Game Object has not been enabled for data (via `setDataEnabled`) then it will be enabled
             * before setting the value.
             * 
             * If the key doesn't already exist in the Data Manager then it is created.
             * 
             * When the value is first set, a `setdata` event is emitted from this Game Object.
             * @param key The key to increase the value for.
             * @param data The value to increase for the given key.
             */
            incData(key: string | object, data?: any): this;

            /**
             * Toggle a boolean value for the given key within this Game Objects Data Manager. If the key doesn't already exist in the Data Manager then it is toggled from false.
             * 
             * If the Game Object has not been enabled for data (via `setDataEnabled`) then it will be enabled
             * before setting the value.
             * 
             * If the key doesn't already exist in the Data Manager then it is created.
             * 
             * When the value is first set, a `setdata` event is emitted from this Game Object.
             * @param key The key to toggle the value for.
             */
            toggleData(key: string | object): this;

            /**
             * Retrieves the value for the given key in this Game Objects Data Manager, or undefined if it doesn't exist.
             * 
             * You can also access values via the `values` object. For example, if you had a key called `gold` you can do either:
             * 
             * ```javascript
             * sprite.getData('gold');
             * ```
             * 
             * Or access the value directly:
             * 
             * ```javascript
             * sprite.data.values.gold;
             * ```
             * 
             * You can also pass in an array of keys, in which case an array of values will be returned:
             * 
             * ```javascript
             * sprite.getData([ 'gold', 'armor', 'health' ]);
             * ```
             * 
             * This approach is useful for destructuring arrays in ES6.
             * @param key The key of the value to retrieve, or an array of keys.
             */
            getData(key: string | string[]): any;

            /**
             * Pass this Game Object to the Input Manager to enable it for Input.
             * 
             * Input works by using hit areas, these are nearly always geometric shapes, such as rectangles or circles, that act as the hit area
             * for the Game Object. However, you can provide your own hit area shape and callback, should you wish to handle some more advanced
             * input detection.
             * 
             * If no arguments are provided it will try and create a rectangle hit area based on the texture frame the Game Object is using. If
             * this isn't a texture-bound object, such as a Graphics or BitmapText object, this will fail, and you'll need to provide a specific
             * shape for it to use.
             * 
             * You can also provide an Input Configuration Object as the only argument to this method.
             * @param hitArea Either an input configuration object, or a geometric shape that defines the hit area for the Game Object. If not given it will try to create a Rectangle based on the texture frame.
             * @param callback The callback that determines if the pointer is within the Hit Area shape or not. If you provide a shape you must also provide a callback.
             * @param dropZone Should this Game Object be treated as a drop zone target? Default false.
             */
            setInteractive(hitArea?: Phaser.Types.Input.InputConfiguration | any, callback?: Phaser.Types.Input.HitAreaCallback, dropZone?: boolean): this;

            /**
             * If this Game Object has previously been enabled for input, this will disable it.
             * 
             * An object that is disabled for input stops processing or being considered for
             * input events, but can be turned back on again at any time by simply calling
             * `setInteractive()` with no arguments provided.
             * 
             * If want to completely remove interaction from this Game Object then use `removeInteractive` instead.
             */
            disableInteractive(): this;

            /**
             * If this Game Object has previously been enabled for input, this will queue it
             * for removal, causing it to no longer be interactive. The removal happens on
             * the next game step, it is not immediate.
             * 
             * The Interactive Object that was assigned to this Game Object will be destroyed,
             * removed from the Input Manager and cleared from this Game Object.
             * 
             * If you wish to re-enable this Game Object at a later date you will need to
             * re-create its InteractiveObject by calling `setInteractive` again.
             * 
             * If you wish to only temporarily stop an object from receiving input then use
             * `disableInteractive` instead, as that toggles the interactive state, where-as
             * this erases it completely.
             * 
             * If you wish to resize a hit area, don't remove and then set it as being
             * interactive. Instead, access the hitarea object directly and resize the shape
             * being used. I.e.: `sprite.input.hitArea.setSize(width, height)` (assuming the
             * shape is a Rectangle, which it is by default.)
             */
            removeInteractive(): this;

            /**
             * This callback is invoked when this Game Object is added to a Scene.
             * 
             * Can be overriden by custom Game Objects, but be aware of some Game Objects that
             * will use this, such as Sprites, to add themselves into the Update List.
             * 
             * You can also listen for the `ADDED_TO_SCENE` event from this Game Object.
             */
            addedToScene(): void;

            /**
             * This callback is invoked when this Game Object is removed from a Scene.
             * 
             * Can be overriden by custom Game Objects, but be aware of some Game Objects that
             * will use this, such as Sprites, to removed themselves from the Update List.
             * 
             * You can also listen for the `REMOVED_FROM_SCENE` event from this Game Object.
             */
            removedFromScene(): void;

            /**
             * To be overridden by custom GameObjects. Allows base objects to be used in a Pool.
             * @param args args
             */
            update(...args: any[]): void;

            /**
             * Returns a JSON representation of the Game Object.
             */
            toJSON(): Phaser.Types.GameObjects.JSONGameObject;

            /**
             * Compares the renderMask with the renderFlags to see if this Game Object will render or not.
             * Also checks the Game Object against the given Cameras exclusion list.
             * @param camera The Camera to check against this Game Object.
             */
            willRender(camera: Phaser.Cameras.Scene2D.Camera): boolean;

            /**
             * Returns an array containing the display list index of either this Game Object, or if it has one,
             * its parent Container. It then iterates up through all of the parent containers until it hits the
             * root of the display list (which is index 0 in the returned array).
             * 
             * Used internally by the InputPlugin but also useful if you wish to find out the display depth of
             * this Game Object and all of its ancestors.
             */
            getIndexList(): number[];

            /**
             * Adds this Game Object to the given Display List.
             * 
             * If no Display List is specified, it will default to the Display List owned by the Scene to which
             * this Game Object belongs.
             * 
             * A Game Object can only exist on one Display List at any given time, but may move freely between them.
             * 
             * If this Game Object is already on another Display List when this method is called, it will first
             * be removed from it, before being added to the new list.
             * 
             * You can query which list it is on by looking at the `Phaser.GameObjects.GameObject#displayList` property.
             * 
             * If a Game Object isn't on any display list, it will not be rendered. If you just wish to temporarly
             * disable it from rendering, consider using the `setVisible` method, instead.
             * @param displayList The Display List to add to. Defaults to the Scene Display List.
             */
            addToDisplayList(displayList?: Phaser.GameObjects.DisplayList | Phaser.GameObjects.Layer): this;

            /**
             * Adds this Game Object to the Update List belonging to the Scene.
             * 
             * When a Game Object is added to the Update List it will have its `preUpdate` method called
             * every game frame. This method is passed two parameters: `delta` and `time`.
             * 
             * If you wish to run your own logic within `preUpdate` then you should always call
             * `preUpdate.super(delta, time)` within it, or it may fail to process required operations,
             * such as Sprite animations.
             */
            addToUpdateList(): this;

            /**
             * Removes this Game Object from the Display List it is currently on.
             * 
             * A Game Object can only exist on one Display List at any given time, but may move freely removed
             * and added back at a later stage.
             * 
             * You can query which list it is on by looking at the `Phaser.GameObjects.GameObject#displayList` property.
             * 
             * If a Game Object isn't on any Display List, it will not be rendered. If you just wish to temporarly
             * disable it from rendering, consider using the `setVisible` method, instead.
             */
            removeFromDisplayList(): this;

            /**
             * Removes this Game Object from the Scene's Update List.
             * 
             * When a Game Object is on the Update List, it will have its `preUpdate` method called
             * every game frame. Calling this method will remove it from the list, preventing this.
             * 
             * Removing a Game Object from the Update List will stop most internal functions working.
             * For example, removing a Sprite from the Update List will prevent it from being able to
             * run animations.
             */
            removeFromUpdateList(): this;

            /**
             * Destroys this Game Object removing it from the Display List and Update List and
             * severing all ties to parent resources.
             * 
             * Also removes itself from the Input Manager and Physics Manager if previously enabled.
             * 
             * Use this to remove a Game Object from your game if you don't ever plan to use it again.
             * As long as no reference to it exists within your own code it should become free for
             * garbage collection by the browser.
             * 
             * If you just want to temporarily disable an object then look at using the
             * Game Object Pool instead of destroying it, as destroyed objects cannot be resurrected.
             * @param fromScene `True` if this Game Object is being destroyed by the Scene, `false` if not. Default false.
             */
            destroy(fromScene?: boolean): void;

            /**
             * The bitmask that `GameObject.renderFlags` is compared against to determine if the Game Object will render or not.
             */
            static readonly RENDER_MASK: number;

        }

        /**
         * The Game Object Creator is a Scene plugin that allows you to quickly create many common
         * types of Game Objects and return them. Unlike the Game Object Factory, they are not automatically
         * added to the Scene.
         * 
         * Game Objects directly register themselves with the Creator and inject their own creation
         * methods into the class.
         */
        class GameObjectCreator {
            /**
             * 
             * @param scene The Scene to which this Game Object Factory belongs.
             */
            constructor(scene: Phaser.Scene);

            /**
             * Creates a new Dynamic Bitmap Text Game Object and returns it.
             * 
             * Note: This method will only be available if the Dynamic Bitmap Text Game Object has been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             * @param addToScene Add this Game Object to the Scene after creating it? If set this argument overrides the `add` property in the config object.
             */
            dynamicBitmapText(config: Phaser.Types.GameObjects.BitmapText.BitmapTextConfig, addToScene?: boolean): Phaser.GameObjects.DynamicBitmapText;

            /**
             * Creates a new Bitmap Text Game Object and returns it.
             * 
             * Note: This method will only be available if the Bitmap Text Game Object has been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             * @param addToScene Add this Game Object to the Scene after creating it? If set this argument overrides the `add` property in the config object.
             */
            bitmapText(config: Phaser.Types.GameObjects.BitmapText.BitmapTextConfig, addToScene?: boolean): Phaser.GameObjects.BitmapText;

            /**
             * Creates a new Blitter Game Object and returns it.
             * 
             * Note: This method will only be available if the Blitter Game Object has been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             * @param addToScene Add this Game Object to the Scene after creating it? If set this argument overrides the `add` property in the config object.
             */
            blitter(config: object, addToScene?: boolean): Phaser.GameObjects.Blitter;

            /**
             * Creates a new Container Game Object and returns it.
             * 
             * Note: This method will only be available if the Container Game Object has been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             * @param addToScene Add this Game Object to the Scene after creating it? If set this argument overrides the `add` property in the config object.
             */
            container(config: object, addToScene?: boolean): Phaser.GameObjects.Container;

            /**
             * The Scene to which this Game Object Creator belongs.
             */
            protected scene: Phaser.Scene;

            /**
             * A reference to the Scene.Systems.
             */
            protected systems: Phaser.Scenes.Systems;

            /**
             * A reference to the Scene Event Emitter.
             */
            protected events: Phaser.Events.EventEmitter;

            /**
             * A reference to the Scene Display List.
             */
            protected displayList: Phaser.GameObjects.DisplayList;

            /**
             * A reference to the Scene Update List.
             */
            protected updateList: Phaser.GameObjects.UpdateList;

            /**
             * Static method called directly by the Game Object creator functions.
             * With this method you can register a custom GameObject factory in the GameObjectCreator,
             * providing a name (`factoryType`) and the constructor (`factoryFunction`) in order
             * to be called when you invoke Phaser.Scene.make[ factoryType ] method.
             * @param factoryType The key of the factory that you will use to call to Phaser.Scene.make[ factoryType ] method.
             * @param factoryFunction The constructor function to be called when you invoke to the Phaser.Scene.make method.
             */
            static register(factoryType: string, factoryFunction: Function): void;

            /**
             * Static method called directly by the Game Object Creator functions.
             * 
             * With this method you can remove a custom Game Object Creator that has been previously
             * registered in the Game Object Creator. Pass in its `factoryType` in order to remove it.
             * @param factoryType The key of the factory that you want to remove from the GameObjectCreator.
             */
            static remove(factoryType: string): void;

            /**
             * Creates a new Graphics Game Object and returns it.
             * 
             * Note: This method will only be available if the Graphics Game Object has been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             * @param addToScene Add this Game Object to the Scene after creating it? If set this argument overrides the `add` property in the config object.
             */
            graphics(config: object, addToScene?: boolean): Phaser.GameObjects.Graphics;

            /**
             * Creates a new Group Game Object and returns it.
             * 
             * Note: This method will only be available if the Group Game Object has been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             */
            group(config: Phaser.Types.GameObjects.Group.GroupConfig | Phaser.Types.GameObjects.Group.GroupCreateConfig): Phaser.GameObjects.Group;

            /**
             * Creates a new Image Game Object and returns it.
             * 
             * Note: This method will only be available if the Image Game Object has been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             * @param addToScene Add this Game Object to the Scene after creating it? If set this argument overrides the `add` property in the config object.
             */
            image(config: object, addToScene?: boolean): Phaser.GameObjects.Image;

            /**
             * Creates a new Layer Game Object and returns it.
             * 
             * Note: This method will only be available if the Layer Game Object has been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             * @param addToScene Add this Game Object to the Scene after creating it? If set this argument overrides the `add` property in the config object.
             */
            layer(config: object, addToScene?: boolean): Phaser.GameObjects.Layer;

            /**
             * Creates a new Mesh Game Object and returns it.
             * 
             * Note: This method will only be available if the Mesh Game Object and WebGL support have been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             * @param addToScene Add this Game Object to the Scene after creating it? If set this argument overrides the `add` property in the config object.
             */
            mesh(config: object, addToScene?: boolean): Phaser.GameObjects.Mesh;

            /**
             * Creates a new Particle Emitter Manager Game Object and returns it.
             * 
             * Note: This method will only be available if the Particles Game Object has been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             * @param addToScene Add this Game Object to the Scene after creating it? If set this argument overrides the `add` property in the config object.
             */
            particles(config: object, addToScene?: boolean): Phaser.GameObjects.Particles.ParticleEmitterManager;

            /**
             * Creates a new Point Light Game Object and returns it.
             * 
             * Note: This method will only be available if the Point Light Game Object has been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             * @param addToScene Add this Game Object to the Scene after creating it? If set this argument overrides the `add` property in the config object.
             */
            pointlight(config: object, addToScene?: boolean): Phaser.GameObjects.PointLight;

            /**
             * Creates a new Render Texture Game Object and returns it.
             * 
             * Note: This method will only be available if the Render Texture Game Object has been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             * @param addToScene Add this Game Object to the Scene after creating it? If set this argument overrides the `add` property in the config object.
             */
            renderTexture(config: Phaser.Types.GameObjects.RenderTexture.RenderTextureConfig, addToScene?: boolean): Phaser.GameObjects.RenderTexture;

            /**
             * Creates a new Rope Game Object and returns it.
             * 
             * Note: This method will only be available if the Rope Game Object and WebGL support have been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             * @param addToScene Add this Game Object to the Scene after creating it? If set this argument overrides the `add` property in the config object.
             */
            rope(config: object, addToScene?: boolean): Phaser.GameObjects.Rope;

            /**
             * Creates a new Shader Game Object and returns it.
             * 
             * Note: This method will only be available if the Shader Game Object and WebGL support have been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             * @param addToScene Add this Game Object to the Scene after creating it? If set this argument overrides the `add` property in the config object.
             */
            shader(config: object, addToScene?: boolean): Phaser.GameObjects.Shader;

            /**
             * Creates a new Sprite Game Object and returns it.
             * 
             * Note: This method will only be available if the Sprite Game Object has been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             * @param addToScene Add this Game Object to the Scene after creating it? If set this argument overrides the `add` property in the config object.
             */
            sprite(config: Phaser.Types.GameObjects.Sprite.SpriteConfig, addToScene?: boolean): Phaser.GameObjects.Sprite;

            /**
             * Creates a new Text Game Object and returns it.
             * 
             * Note: This method will only be available if the Text Game Object has been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             * @param addToScene Add this Game Object to the Scene after creating it? If set this argument overrides the `add` property in the config object.
             */
            text(config: Phaser.Types.GameObjects.Text.TextConfig, addToScene?: boolean): Phaser.GameObjects.Text;

            /**
             * Creates a new TileSprite Game Object and returns it.
             * 
             * Note: This method will only be available if the TileSprite Game Object has been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             * @param addToScene Add this Game Object to the Scene after creating it? If set this argument overrides the `add` property in the config object.
             */
            tileSprite(config: Phaser.Types.GameObjects.TileSprite.TileSpriteConfig, addToScene?: boolean): Phaser.GameObjects.TileSprite;

            /**
             * Creates a new Video Game Object and returns it.
             * 
             * Note: This method will only be available if the Video Game Object has been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             * @param addToScene Add this Game Object to the Scene after creating it? If set this argument overrides the `add` property in the config object.
             */
            video(config: object, addToScene?: boolean): Phaser.GameObjects.Video;

            /**
             * Creates a new Zone Game Object and returns it.
             * 
             * Note: This method will only be available if the Zone Game Object has been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             */
            zone(config: object): Phaser.GameObjects.Zone;

            /**
             * Creates a Tilemap from the given key or data, or creates a blank Tilemap if no key/data provided.
             * When loading from CSV or a 2D array, you should specify the tileWidth & tileHeight. When parsing
             * from a map from Tiled, the tileWidth, tileHeight, width & height will be pulled from the map
             * data. For an empty map, you should specify tileWidth, tileHeight, width & height.
             * @param config The config options for the Tilemap.
             */
            tilemap(config?: Phaser.Types.Tilemaps.TilemapConfig): Phaser.Tilemaps.Tilemap;

            /**
             * Creates a new Tween object and returns it.
             * 
             * Note: This method will only be available if Tweens have been built into Phaser.
             * @param config The Tween configuration.
             */
            tween(config: Phaser.Types.Tweens.TweenBuilderConfig | object): Phaser.Tweens.Tween;

        }

        /**
         * The Game Object Factory is a Scene plugin that allows you to quickly create many common
         * types of Game Objects and have them automatically registered with the Scene.
         * 
         * Game Objects directly register themselves with the Factory and inject their own creation
         * methods into the class.
         */
        class GameObjectFactory {
            /**
             * 
             * @param scene The Scene to which this Game Object Factory belongs.
             */
            constructor(scene: Phaser.Scene);

            /**
             * Creates a new Path Object.
             * @param x The horizontal position of this Path.
             * @param y The vertical position of this Path.
             */
            path(x: number, y: number): Phaser.Curves.Path;

            /**
             * Creates a new Dynamic Bitmap Text Game Object and adds it to the Scene.
             * 
             * BitmapText objects work by taking a texture file and an XML or JSON file that describes the font structure.
             * 
             * During rendering for each letter of the text is rendered to the display, proportionally spaced out and aligned to
             * match the font structure.
             * 
             * Dynamic Bitmap Text objects are different from Static Bitmap Text in that they invoke a callback for each
             * letter being rendered during the render pass. This callback allows you to manipulate the properties of
             * each letter being rendered, such as its position, scale or tint, allowing you to create interesting effects
             * like jiggling text, which can't be done with Static text. This means that Dynamic Text takes more processing
             * time, so only use them if you require the callback ability they have.
             * 
             * BitmapText objects are less flexible than Text objects, in that they have less features such as shadows, fills and the ability
             * to use Web Fonts, however you trade this flexibility for rendering speed. You can also create visually compelling BitmapTexts by
             * processing the font texture in an image editor, applying fills and any other effects required.
             * 
             * To create multi-line text insert \r, \n or \r\n escape codes into the text string.
             * 
             * To create a BitmapText data files you need a 3rd party app such as:
             * 
             * BMFont (Windows, free): http://www.angelcode.com/products/bmfont/
             * Glyph Designer (OS X, commercial): http://www.71squared.com/en/glyphdesigner
             * Littera (Web-based, free): http://kvazars.com/littera/
             * 
             * For most use cases it is recommended to use XML. If you wish to use JSON, the formatting should be equal to the result of
             * converting a valid XML file through the popular X2JS library. An online tool for conversion can be found here: http://codebeautify.org/xmltojson
             * 
             * Note: This method will only be available if the Dynamic Bitmap Text Game Object has been built into Phaser.
             * @param x The x position of the Game Object.
             * @param y The y position of the Game Object.
             * @param font The key of the font to use from the BitmapFont cache.
             * @param text The string, or array of strings, to be set as the content of this Bitmap Text.
             * @param size The font size to set.
             */
            dynamicBitmapText(x: number, y: number, font: string, text?: string | string[], size?: number): Phaser.GameObjects.DynamicBitmapText;

            /**
             * Creates a new Bitmap Text Game Object and adds it to the Scene.
             * 
             * BitmapText objects work by taking a texture file and an XML or JSON file that describes the font structure.
             * 
             * During rendering for each letter of the text is rendered to the display, proportionally spaced out and aligned to
             * match the font structure.
             * 
             * BitmapText objects are less flexible than Text objects, in that they have less features such as shadows, fills and the ability
             * to use Web Fonts, however you trade this flexibility for rendering speed. You can also create visually compelling BitmapTexts by
             * processing the font texture in an image editor, applying fills and any other effects required.
             * 
             * To create multi-line text insert \r, \n or \r\n escape codes into the text string.
             * 
             * To create a BitmapText data files you need a 3rd party app such as:
             * 
             * BMFont (Windows, free): http://www.angelcode.com/products/bmfont/
             * Glyph Designer (OS X, commercial): http://www.71squared.com/en/glyphdesigner
             * Littera (Web-based, free): http://kvazars.com/littera/
             * 
             * For most use cases it is recommended to use XML. If you wish to use JSON, the formatting should be equal to the result of
             * converting a valid XML file through the popular X2JS library. An online tool for conversion can be found here: http://codebeautify.org/xmltojson
             * 
             * Note: This method will only be available if the Bitmap Text Game Object has been built into Phaser.
             * @param x The x position of the Game Object.
             * @param y The y position of the Game Object.
             * @param font The key of the font to use from the BitmapFont cache.
             * @param text The string, or array of strings, to be set as the content of this Bitmap Text.
             * @param size The font size to set.
             * @param align The alignment of the text in a multi-line BitmapText object. Default 0.
             */
            bitmapText(x: number, y: number, font: string, text?: string | string[], size?: number, align?: number): Phaser.GameObjects.BitmapText;

            /**
             * Creates a new Blitter Game Object and adds it to the Scene.
             * 
             * Note: This method will only be available if the Blitter Game Object has been built into Phaser.
             * @param x The x position of the Game Object.
             * @param y The y position of the Game Object.
             * @param key The key of the Texture the Blitter object will use.
             * @param frame The default Frame children of the Blitter will use.
             */
            blitter(x: number, y: number, key: string, frame?: string | number): Phaser.GameObjects.Blitter;

            /**
             * Creates a new Container Game Object and adds it to the Scene.
             * 
             * Note: This method will only be available if the Container Game Object has been built into Phaser.
             * @param x The horizontal position of this Game Object in the world. Default 0.
             * @param y The vertical position of this Game Object in the world. Default 0.
             * @param children An optional array of Game Objects to add to this Container.
             */
            container(x?: number, y?: number, children?: Phaser.GameObjects.GameObject | Phaser.GameObjects.GameObject[]): Phaser.GameObjects.Container;

            /**
             * DOM Element Game Objects are a way to control and manipulate HTML Elements over the top of your game.
             * 
             * In order for DOM Elements to display you have to enable them by adding the following to your game
             * configuration object:
             * 
             * ```javascript
             * dom {
             *   createContainer: true
             * }
             * ```
             * 
             * When this is added, Phaser will automatically create a DOM Container div that is positioned over the top
             * of the game canvas. This div is sized to match the canvas, and if the canvas size changes, as a result of
             * settings within the Scale Manager, the dom container is resized accordingly.
             * 
             * You can create a DOM Element by either passing in DOMStrings, or by passing in a reference to an existing
             * Element that you wish to be placed under the control of Phaser. For example:
             * 
             * ```javascript
             * this.add.dom(x, y, 'div', 'background-color: lime; width: 220px; height: 100px; font: 48px Arial', 'Phaser');
             * ```
             * 
             * The above code will insert a div element into the DOM Container at the given x/y coordinate. The DOMString in
             * the 4th argument sets the initial CSS style of the div and the final argument is the inner text. In this case,
             * it will create a lime colored div that is 220px by 100px in size with the text Phaser in it, in an Arial font.
             * 
             * You should nearly always, without exception, use explicitly sized HTML Elements, in order to fully control
             * alignment and positioning of the elements next to regular game content.
             * 
             * Rather than specify the CSS and HTML directly you can use the `load.html` File Loader to load it into the
             * cache and then use the `createFromCache` method instead. You can also use `createFromHTML` and various other
             * methods available in this class to help construct your elements.
             * 
             * Once the element has been created you can then control it like you would any other Game Object. You can set its
             * position, scale, rotation, alpha and other properties. It will move as the main Scene Camera moves and be clipped
             * at the edge of the canvas. It's important to remember some limitations of DOM Elements: The obvious one is that
             * they appear above or below your game canvas. You cannot blend them into the display list, meaning you cannot have
             * a DOM Element, then a Sprite, then another DOM Element behind it.
             * 
             * They also cannot be enabled for input. To do that, you have to use the `addListener` method to add native event
             * listeners directly. The final limitation is to do with cameras. The DOM Container is sized to match the game canvas
             * entirely and clipped accordingly. DOM Elements respect camera scrolling and scrollFactor settings, but if you
             * change the size of the camera so it no longer matches the size of the canvas, they won't be clipped accordingly.
             * 
             * Also, all DOM Elements are inserted into the same DOM Container, regardless of which Scene they are created in.
             * 
             * DOM Elements are a powerful way to align native HTML with your Phaser Game Objects. For example, you can insert
             * a login form for a multiplayer game directly into your title screen. Or a text input box for a highscore table.
             * Or a banner ad from a 3rd party service. Or perhaps you'd like to use them for high resolution text display and
             * UI. The choice is up to you, just remember that you're dealing with standard HTML and CSS floating over the top
             * of your game, and should treat it accordingly.
             * 
             * Note: This method will only be available if the DOM Element Game Object has been built into Phaser.
             * @param x The horizontal position of this DOM Element in the world.
             * @param y The vertical position of this DOM Element in the world.
             * @param element An existing DOM element, or a string. If a string starting with a # it will do a `getElementById` look-up on the string (minus the hash). Without a hash, it represents the type of element to create, i.e. 'div'.
             * @param style If a string, will be set directly as the elements `style` property value. If a plain object, will be iterated and the values transferred. In both cases the values replacing whatever CSS styles may have been previously set.
             * @param innerText If given, will be set directly as the elements `innerText` property value, replacing whatever was there before.
             */
            dom(x: number, y: number, element?: HTMLElement | string, style?: string | any, innerText?: string): Phaser.GameObjects.DOMElement;

            /**
             * Creates a new Extern Game Object and adds it to the Scene.
             * 
             * Note: This method will only be available if the Extern Game Object has been built into Phaser.
             */
            extern(): Phaser.GameObjects.Extern;

            /**
             * The Scene to which this Game Object Factory belongs.
             */
            protected scene: Phaser.Scene;

            /**
             * A reference to the Scene.Systems.
             */
            protected systems: Phaser.Scenes.Systems;

            /**
             * A reference to the Scene Event Emitter.
             */
            protected events: Phaser.Events.EventEmitter;

            /**
             * A reference to the Scene Display List.
             */
            protected displayList: Phaser.GameObjects.DisplayList;

            /**
             * A reference to the Scene Update List.
             */
            protected updateList: Phaser.GameObjects.UpdateList;

            /**
             * Adds an existing Game Object to this Scene.
             * 
             * If the Game Object renders, it will be added to the Display List.
             * If it has a `preUpdate` method, it will be added to the Update List.
             * @param child The child to be added to this Scene.
             */
            existing<G extends (Phaser.GameObjects.GameObject|Phaser.GameObjects.Group|Phaser.GameObjects.Layer)>(child: G): G;

            /**
             * Static method called directly by the Game Object factory functions.
             * With this method you can register a custom GameObject factory in the GameObjectFactory,
             * providing a name (`factoryType`) and the constructor (`factoryFunction`) in order
             * to be called when you call to Phaser.Scene.add[ factoryType ] method.
             * @param factoryType The key of the factory that you will use to call to Phaser.Scene.add[ factoryType ] method.
             * @param factoryFunction The constructor function to be called when you invoke to the Phaser.Scene.add method.
             */
            static register(factoryType: string, factoryFunction: Function): void;

            /**
             * Static method called directly by the Game Object factory functions.
             * With this method you can remove a custom GameObject factory registered in the GameObjectFactory,
             * providing a its `factoryType`.
             * @param factoryType The key of the factory that you want to remove from the GameObjectFactory.
             */
            static remove(factoryType: string): void;

            /**
             * Creates a new Graphics Game Object and adds it to the Scene.
             * 
             * Note: This method will only be available if the Graphics Game Object has been built into Phaser.
             * @param config The Graphics configuration.
             */
            graphics(config?: Phaser.Types.GameObjects.Graphics.Options): Phaser.GameObjects.Graphics;

            /**
             * Creates a new Group Game Object and adds it to the Scene.
             * 
             * Note: This method will only be available if the Group Game Object has been built into Phaser.
             * @param children Game Objects to add to this Group; or the `config` argument.
             * @param config A Group Configuration object.
             */
            group(children?: Phaser.GameObjects.GameObject[] | Phaser.Types.GameObjects.Group.GroupConfig | Phaser.Types.GameObjects.Group.GroupConfig[], config?: Phaser.Types.GameObjects.Group.GroupConfig | Phaser.Types.GameObjects.Group.GroupCreateConfig): Phaser.GameObjects.Group;

            /**
             * Creates a new Image Game Object and adds it to the Scene.
             * 
             * Note: This method will only be available if the Image Game Object has been built into Phaser.
             * @param x The horizontal position of this Game Object in the world.
             * @param y The vertical position of this Game Object in the world.
             * @param texture The key, or instance of the Texture this Game Object will use to render with, as stored in the Texture Manager.
             * @param frame An optional frame from the Texture this Game Object is rendering with.
             */
            image(x: number, y: number, texture: string | Phaser.Textures.Texture, frame?: string | number): Phaser.GameObjects.Image;

            /**
             * Creates a new Layer Game Object and adds it to the Scene.
             * 
             * Note: This method will only be available if the Layer Game Object has been built into Phaser.
             * @param children An optional array of Game Objects to add to this Layer.
             */
            layer(children?: Phaser.GameObjects.GameObject | Phaser.GameObjects.GameObject[]): Phaser.GameObjects.Layer;

            /**
             * Creates a new Mesh Game Object and adds it to the Scene.
             * 
             * Note: This method will only be available if the Mesh Game Object and WebGL support have been built into Phaser.
             * @param x The horizontal position of this Game Object in the world.
             * @param y The vertical position of this Game Object in the world.
             * @param texture The key, or instance of the Texture this Game Object will use to render with, as stored in the Texture Manager.
             * @param frame An optional frame from the Texture this Game Object is rendering with.
             * @param vertices The vertices array. Either `xy` pairs, or `xyz` if the `containsZ` parameter is `true`.
             * @param uvs The UVs pairs array.
             * @param indicies Optional vertex indicies array. If you don't have one, pass `null` or an empty array.
             * @param containsZ Does the vertices data include a `z` component? Default false.
             * @param normals Optional vertex normals array. If you don't have one, pass `null` or an empty array.
             * @param colors An array of colors, one per vertex, or a single color value applied to all vertices. Default 0xffffff.
             * @param alphas An array of alpha values, one per vertex, or a single alpha value applied to all vertices. Default 1.
             */
            mesh(x?: number, y?: number, texture?: string | Phaser.Textures.Texture, frame?: string | number, vertices?: number[], uvs?: number[], indicies?: number[], containsZ?: boolean, normals?: number[], colors?: number | number[], alphas?: number | number[]): Phaser.GameObjects.Mesh;

            /**
             * Creates a new Particle Emitter Manager Game Object and adds it to the Scene.
             * 
             * Note: This method will only be available if the Particles Game Object has been built into Phaser.
             * @param texture The key, or instance of the Texture this Game Object will use to render with, as stored in the Texture Manager.
             * @param frame An optional frame from the Texture this Game Object is rendering with.
             * @param emitters Configuration settings for one or more emitters to create.
             */
            particles(texture: string | Phaser.Textures.Texture, frame?: string | number | object, emitters?: Phaser.Types.GameObjects.Particles.ParticleEmitterConfig | Phaser.Types.GameObjects.Particles.ParticleEmitterConfig[]): Phaser.GameObjects.Particles.ParticleEmitterManager;

            /**
             * Creates a new PathFollower Game Object and adds it to the Scene.
             * 
             * Note: This method will only be available if the PathFollower Game Object has been built into Phaser.
             * @param path The Path this PathFollower is connected to.
             * @param x The horizontal position of this Game Object in the world.
             * @param y The vertical position of this Game Object in the world.
             * @param texture The key, or instance of the Texture this Game Object will use to render with, as stored in the Texture Manager.
             * @param frame An optional frame from the Texture this Game Object is rendering with.
             */
            follower(path: Phaser.Curves.Path, x: number, y: number, texture: string | Phaser.Textures.Texture, frame?: string | number): Phaser.GameObjects.PathFollower;

            /**
             * Creates a new Point Light Game Object and adds it to the Scene.
             * 
             * Note: This method will only be available if the Point Light Game Object has been built into Phaser.
             * 
             * The Point Light Game Object provides a way to add a point light effect into your game,
             * without the expensive shader processing requirements of the traditional Light Game Object.
             * 
             * The difference is that the Point Light renders using a custom shader, designed to give the
             * impression of a point light source, of variable radius, intensity and color, in your game.
             * However, unlike the Light Game Object, it does not impact any other Game Objects, or use their
             * normal maps for calcuations. This makes them extremely fast to render compared to Lights
             * and perfect for special effects, such as flickering torches or muzzle flashes.
             * 
             * For maximum performance you should batch Point Light Game Objects together. This means
             * ensuring they follow each other consecutively on the display list. Ideally, use a Layer
             * Game Object and then add just Point Lights to it, so that it can batch together the rendering
             * of the lights. You don't _have_ to do this, and if you've only a handful of Point Lights in
             * your game then it's perfectly safe to mix them into the dislay list as normal. However, if
             * you're using a large number of them, please consider how they are mixed into the display list.
             * 
             * The renderer will automatically cull Point Lights. Those with a radius that does not intersect
             * with the Camera will be skipped in the rendering list. This happens automatically and the
             * culled state is refreshed every frame, for every camera.
             * 
             * The origin of a Point Light is always 0.5 and it cannot be changed.
             * 
             * Point Lights are a WebGL only feature and do not have a Canvas counterpart.
             * @param x The horizontal position of this Point Light in the world.
             * @param y The vertical position of this Point Light in the world.
             * @param color The color of the Point Light, given as a hex value. Default 0xffffff.
             * @param radius The radius of the Point Light. Default 128.
             * @param intensity The intensity, or colr blend, of the Point Light. Default 1.
             * @param attenuation The attenuation  of the Point Light. This is the reduction of light from the center point. Default 0.1.
             */
            pointlight(x: number, y: number, color?: number, radius?: number, intensity?: number, attenuation?: number): Phaser.GameObjects.PointLight;

            /**
             * Creates a new Render Texture Game Object and adds it to the Scene.
             * 
             * Note: This method will only be available if the Render Texture Game Object has been built into Phaser.
             * 
             * A Render Texture is a special texture that allows any number of Game Objects to be drawn to it. You can take many complex objects and
             * draw them all to this one texture, which can they be used as the texture for other Game Object's. It's a way to generate dynamic
             * textures at run-time that are WebGL friendly and don't invoke expensive GPU uploads.
             * @param x The horizontal position of this Game Object in the world.
             * @param y The vertical position of this Game Object in the world.
             * @param width The width of the Render Texture. Default 32.
             * @param height The height of the Render Texture. Default 32.
             */
            renderTexture(x: number, y: number, width?: number, height?: number): Phaser.GameObjects.RenderTexture;

            /**
             * Creates a new Rope Game Object and adds it to the Scene.
             * 
             * Note: This method will only be available if the Rope Game Object and WebGL support have been built into Phaser.
             * @param x The horizontal position of this Game Object in the world.
             * @param y The vertical position of this Game Object in the world.
             * @param texture The key, or instance of the Texture this Game Object will use to render with, as stored in the Texture Manager.
             * @param frame An optional frame from the Texture this Game Object is rendering with.
             * @param points An array containing the vertices data for this Rope. If none is provided a simple quad is created. See `setPoints` to set this post-creation.
             * @param horizontal Should the vertices of this Rope be aligned horizontally (`true`), or vertically (`false`)? Default true.
             * @param colors An optional array containing the color data for this Rope. You should provide one color value per pair of vertices.
             * @param alphas An optional array containing the alpha data for this Rope. You should provide one alpha value per pair of vertices.
             */
            rope(x: number, y: number, texture: string | Phaser.Textures.Texture, frame?: string | number, points?: Phaser.Types.Math.Vector2Like[], horizontal?: boolean, colors?: number[], alphas?: number[]): Phaser.GameObjects.Rope;

            /**
             * Creates a new Shader Game Object and adds it to the Scene.
             * 
             * Note: This method will only be available if the Shader Game Object and WebGL support have been built into Phaser.
             * @param key The key of the shader to use from the shader cache, or a BaseShader instance.
             * @param x The horizontal position of this Game Object in the world. Default 0.
             * @param y The vertical position of this Game Object in the world. Default 0.
             * @param width The width of the Game Object. Default 128.
             * @param height The height of the Game Object. Default 128.
             * @param textures Optional array of texture keys to bind to the iChannel0...3 uniforms. The textures must already exist in the Texture Manager.
             * @param textureData Optional additional texture data.
             */
            shader(key: string | Phaser.Display.BaseShader, x?: number, y?: number, width?: number, height?: number, textures?: string[], textureData?: object): Phaser.GameObjects.Shader;

            /**
             * Creates a new Arc Shape Game Object and adds it to the Scene.
             * 
             * Note: This method will only be available if the Arc Game Object has been built into Phaser.
             * 
             * The Arc Shape is a Game Object that can be added to a Scene, Group or Container. You can
             * treat it like any other Game Object in your game, such as tweening it, scaling it, or enabling
             * it for input or physics. It provides a quick and easy way for you to render this shape in your
             * game without using a texture, while still taking advantage of being fully batched in WebGL.
             * 
             * This shape supports both fill and stroke colors.
             * 
             * When it renders it displays an arc shape. You can control the start and end angles of the arc,
             * as well as if the angles are winding clockwise or anti-clockwise. With the default settings
             * it renders as a complete circle. By changing the angles you can create other arc shapes,
             * such as half-circles.
             * @param x The horizontal position of this Game Object in the world. Default 0.
             * @param y The vertical position of this Game Object in the world. Default 0.
             * @param radius The radius of the arc. Default 128.
             * @param startAngle The start angle of the arc, in degrees. Default 0.
             * @param endAngle The end angle of the arc, in degrees. Default 360.
             * @param anticlockwise The winding order of the start and end angles. Default false.
             * @param fillColor The color the arc will be filled with, i.e. 0xff0000 for red.
             * @param fillAlpha The alpha the arc will be filled with. You can also set the alpha of the overall Shape using its `alpha` property.
             */
            arc(x?: number, y?: number, radius?: number, startAngle?: number, endAngle?: number, anticlockwise?: boolean, fillColor?: number, fillAlpha?: number): Phaser.GameObjects.Arc;

            /**
             * Creates a new Circle Shape Game Object and adds it to the Scene.
             * 
             * A Circle is an Arc with no defined start and end angle, making it render as a complete circle.
             * 
             * Note: This method will only be available if the Arc Game Object has been built into Phaser.
             * @param x The horizontal position of this Game Object in the world. Default 0.
             * @param y The vertical position of this Game Object in the world. Default 0.
             * @param radius The radius of the circle. Default 128.
             * @param fillColor The color the circle will be filled with, i.e. 0xff0000 for red.
             * @param fillAlpha The alpha the circle will be filled with. You can also set the alpha of the overall Shape using its `alpha` property.
             */
            circle(x?: number, y?: number, radius?: number, fillColor?: number, fillAlpha?: number): Phaser.GameObjects.Arc;

            /**
             * Creates a new Curve Shape Game Object and adds it to the Scene.
             * 
             * Note: This method will only be available if the Curve Game Object has been built into Phaser.
             * 
             * The Curve Shape is a Game Object that can be added to a Scene, Group or Container. You can
             * treat it like any other Game Object in your game, such as tweening it, scaling it, or enabling
             * it for input or physics. It provides a quick and easy way for you to render this shape in your
             * game without using a texture, while still taking advantage of being fully batched in WebGL.
             * 
             * This shape supports both fill and stroke colors.
             * 
             * To render a Curve Shape you must first create a `Phaser.Curves.Curve` object, then pass it to
             * the Curve Shape in the constructor.
             * 
             * The Curve shape also has a `smoothness` property and corresponding `setSmoothness` method.
             * This allows you to control how smooth the shape renders in WebGL, by controlling the number of iterations
             * that take place during construction. Increase and decrease the default value for smoother, or more
             * jagged, shapes.
             * @param x The horizontal position of this Game Object in the world. Default 0.
             * @param y The vertical position of this Game Object in the world. Default 0.
             * @param curve The Curve object to use to create the Shape.
             * @param fillColor The color the curve will be filled with, i.e. 0xff0000 for red.
             * @param fillAlpha The alpha the curve will be filled with. You can also set the alpha of the overall Shape using its `alpha` property.
             */
            curve(x?: number, y?: number, curve?: Phaser.Curves.Curve, fillColor?: number, fillAlpha?: number): Phaser.GameObjects.Curve;

            /**
             * Creates a new Ellipse Shape Game Object and adds it to the Scene.
             * 
             * Note: This method will only be available if the Ellipse Game Object has been built into Phaser.
             * 
             * The Ellipse Shape is a Game Object that can be added to a Scene, Group or Container. You can
             * treat it like any other Game Object in your game, such as tweening it, scaling it, or enabling
             * it for input or physics. It provides a quick and easy way for you to render this shape in your
             * game without using a texture, while still taking advantage of being fully batched in WebGL.
             * 
             * This shape supports both fill and stroke colors.
             * 
             * When it renders it displays an ellipse shape. You can control the width and height of the ellipse.
             * If the width and height match it will render as a circle. If the width is less than the height,
             * it will look more like an egg shape.
             * 
             * The Ellipse shape also has a `smoothness` property and corresponding `setSmoothness` method.
             * This allows you to control how smooth the shape renders in WebGL, by controlling the number of iterations
             * that take place during construction. Increase and decrease the default value for smoother, or more
             * jagged, shapes.
             * @param x The horizontal position of this Game Object in the world. Default 0.
             * @param y The vertical position of this Game Object in the world. Default 0.
             * @param width The width of the ellipse. An ellipse with equal width and height renders as a circle. Default 128.
             * @param height The height of the ellipse. An ellipse with equal width and height renders as a circle. Default 128.
             * @param fillColor The color the ellipse will be filled with, i.e. 0xff0000 for red.
             * @param fillAlpha The alpha the ellipse will be filled with. You can also set the alpha of the overall Shape using its `alpha` property.
             */
            ellipse(x?: number, y?: number, width?: number, height?: number, fillColor?: number, fillAlpha?: number): Phaser.GameObjects.Ellipse;

            /**
             * Creates a new Grid Shape Game Object and adds it to the Scene.
             * 
             * Note: This method will only be available if the Grid Game Object has been built into Phaser.
             * 
             * The Grid Shape is a Game Object that can be added to a Scene, Group or Container. You can
             * treat it like any other Game Object in your game, such as tweening it, scaling it, or enabling
             * it for input or physics. It provides a quick and easy way for you to render this shape in your
             * game without using a texture, while still taking advantage of being fully batched in WebGL.
             * 
             * This shape supports only fill colors and cannot be stroked.
             * 
             * A Grid Shape allows you to display a grid in your game, where you can control the size of the
             * grid as well as the width and height of the grid cells. You can set a fill color for each grid
             * cell as well as an alternate fill color. When the alternate fill color is set then the grid
             * cells will alternate the fill colors as they render, creating a chess-board effect. You can
             * also optionally have an outline fill color. If set, this draws lines between the grid cells
             * in the given color. If you specify an outline color with an alpha of zero, then it will draw
             * the cells spaced out, but without the lines between them.
             * @param x The horizontal position of this Game Object in the world. Default 0.
             * @param y The vertical position of this Game Object in the world. Default 0.
             * @param width The width of the grid. Default 128.
             * @param height The height of the grid. Default 128.
             * @param cellWidth The width of one cell in the grid. Default 32.
             * @param cellHeight The height of one cell in the grid. Default 32.
             * @param fillColor The color the grid cells will be filled with, i.e. 0xff0000 for red.
             * @param fillAlpha The alpha the grid cells will be filled with. You can also set the alpha of the overall Shape using its `alpha` property.
             * @param outlineFillColor The color of the lines between the grid cells.
             * @param outlineFillAlpha The alpha of the lines between the grid cells.
             */
            grid(x?: number, y?: number, width?: number, height?: number, cellWidth?: number, cellHeight?: number, fillColor?: number, fillAlpha?: number, outlineFillColor?: number, outlineFillAlpha?: number): Phaser.GameObjects.Grid;

            /**
             * Creates a new IsoBox Shape Game Object and adds it to the Scene.
             * 
             * Note: This method will only be available if the IsoBox Game Object has been built into Phaser.
             * 
             * The IsoBox Shape is a Game Object that can be added to a Scene, Group or Container. You can
             * treat it like any other Game Object in your game, such as tweening it, scaling it, or enabling
             * it for input or physics. It provides a quick and easy way for you to render this shape in your
             * game without using a texture, while still taking advantage of being fully batched in WebGL.
             * 
             * This shape supports only fill colors and cannot be stroked.
             * 
             * An IsoBox is an 'isometric' rectangle. Each face of it has a different fill color. You can set
             * the color of the top, left and right faces of the rectangle respectively. You can also choose
             * which of the faces are rendered via the `showTop`, `showLeft` and `showRight` properties.
             * 
             * You cannot view an IsoBox from under-neath, however you can change the 'angle' by setting
             * the `projection` property.
             * @param x The horizontal position of this Game Object in the world. Default 0.
             * @param y The vertical position of this Game Object in the world. Default 0.
             * @param size The width of the iso box in pixels. The left and right faces will be exactly half this value. Default 48.
             * @param height The height of the iso box. The left and right faces will be this tall. The overall height of the isobox will be this value plus half the `size` value. Default 32.
             * @param fillTop The fill color of the top face of the iso box. Default 0xeeeeee.
             * @param fillLeft The fill color of the left face of the iso box. Default 0x999999.
             * @param fillRight The fill color of the right face of the iso box. Default 0xcccccc.
             */
            isobox(x?: number, y?: number, size?: number, height?: number, fillTop?: number, fillLeft?: number, fillRight?: number): Phaser.GameObjects.IsoBox;

            /**
             * Creates a new IsoTriangle Shape Game Object and adds it to the Scene.
             * 
             * Note: This method will only be available if the IsoTriangle Game Object has been built into Phaser.
             * 
             * The IsoTriangle Shape is a Game Object that can be added to a Scene, Group or Container. You can
             * treat it like any other Game Object in your game, such as tweening it, scaling it, or enabling
             * it for input or physics. It provides a quick and easy way for you to render this shape in your
             * game without using a texture, while still taking advantage of being fully batched in WebGL.
             * 
             * This shape supports only fill colors and cannot be stroked.
             * 
             * An IsoTriangle is an 'isometric' triangle. Think of it like a pyramid. Each face has a different
             * fill color. You can set the color of the top, left and right faces of the triangle respectively
             * You can also choose which of the faces are rendered via the `showTop`, `showLeft` and `showRight` properties.
             * 
             * You cannot view an IsoTriangle from under-neath, however you can change the 'angle' by setting
             * the `projection` property. The `reversed` property controls if the IsoTriangle is rendered upside
             * down or not.
             * @param x The horizontal position of this Game Object in the world. Default 0.
             * @param y The vertical position of this Game Object in the world. Default 0.
             * @param size The width of the iso triangle in pixels. The left and right faces will be exactly half this value. Default 48.
             * @param height The height of the iso triangle. The left and right faces will be this tall. The overall height of the iso triangle will be this value plus half the `size` value. Default 32.
             * @param reversed Is the iso triangle upside down? Default false.
             * @param fillTop The fill color of the top face of the iso triangle. Default 0xeeeeee.
             * @param fillLeft The fill color of the left face of the iso triangle. Default 0x999999.
             * @param fillRight The fill color of the right face of the iso triangle. Default 0xcccccc.
             */
            isotriangle(x?: number, y?: number, size?: number, height?: number, reversed?: boolean, fillTop?: number, fillLeft?: number, fillRight?: number): Phaser.GameObjects.IsoTriangle;

            /**
             * Creates a new Line Shape Game Object and adds it to the Scene.
             * 
             * Note: This method will only be available if the Line Game Object has been built into Phaser.
             * 
             * The Line Shape is a Game Object that can be added to a Scene, Group or Container. You can
             * treat it like any other Game Object in your game, such as tweening it, scaling it, or enabling
             * it for input or physics. It provides a quick and easy way for you to render this shape in your
             * game without using a texture, while still taking advantage of being fully batched in WebGL.
             * 
             * This shape supports only stroke colors and cannot be filled.
             * 
             * A Line Shape allows you to draw a line between two points in your game. You can control the
             * stroke color and thickness of the line. In WebGL only you can also specify a different
             * thickness for the start and end of the line, allowing you to render lines that taper-off.
             * 
             * If you need to draw multiple lines in a sequence you may wish to use the Polygon Shape instead.
             * @param x The horizontal position of this Game Object in the world. Default 0.
             * @param y The vertical position of this Game Object in the world. Default 0.
             * @param x1 The horizontal position of the start of the line. Default 0.
             * @param y1 The vertical position of the start of the line. Default 0.
             * @param x2 The horizontal position of the end of the line. Default 128.
             * @param y2 The vertical position of the end of the line. Default 0.
             * @param strokeColor The color the line will be drawn in, i.e. 0xff0000 for red.
             * @param strokeAlpha The alpha the line will be drawn in. You can also set the alpha of the overall Shape using its `alpha` property.
             */
            line(x?: number, y?: number, x1?: number, y1?: number, x2?: number, y2?: number, strokeColor?: number, strokeAlpha?: number): Phaser.GameObjects.Line;

            /**
             * Creates a new Polygon Shape Game Object and adds it to the Scene.
             * 
             * Note: This method will only be available if the Polygon Game Object has been built into Phaser.
             * 
             * The Polygon Shape is a Game Object that can be added to a Scene, Group or Container. You can
             * treat it like any other Game Object in your game, such as tweening it, scaling it, or enabling
             * it for input or physics. It provides a quick and easy way for you to render this shape in your
             * game without using a texture, while still taking advantage of being fully batched in WebGL.
             * 
             * This shape supports both fill and stroke colors.
             * 
             * The Polygon Shape is created by providing a list of points, which are then used to create an
             * internal Polygon geometry object. The points can be set from a variety of formats:
             * 
             * - An array of Point or Vector2 objects: `[new Phaser.Math.Vector2(x1, y1), ...]`
             * - An array of objects with public x/y properties: `[obj1, obj2, ...]`
             * - An array of paired numbers that represent point coordinates: `[x1,y1, x2,y2, ...]`
             * - An array of arrays with two elements representing x/y coordinates: `[[x1, y1], [x2, y2], ...]`
             * 
             * By default the `x` and `y` coordinates of this Shape refer to the center of it. However, depending
             * on the coordinates of the points provided, the final shape may be rendered offset from its origin.
             * @param x The horizontal position of this Game Object in the world. Default 0.
             * @param y The vertical position of this Game Object in the world. Default 0.
             * @param points The points that make up the polygon.
             * @param fillColor The color the polygon will be filled with, i.e. 0xff0000 for red.
             * @param fillAlpha The alpha the polygon will be filled with. You can also set the alpha of the overall Shape using its `alpha` property.
             */
            polygon(x?: number, y?: number, points?: any, fillColor?: number, fillAlpha?: number): Phaser.GameObjects.Polygon;

            /**
             * Creates a new Rectangle Shape Game Object and adds it to the Scene.
             * 
             * Note: This method will only be available if the Rectangle Game Object has been built into Phaser.
             * 
             * The Rectangle Shape is a Game Object that can be added to a Scene, Group or Container. You can
             * treat it like any other Game Object in your game, such as tweening it, scaling it, or enabling
             * it for input or physics. It provides a quick and easy way for you to render this shape in your
             * game without using a texture, while still taking advantage of being fully batched in WebGL.
             * 
             * This shape supports both fill and stroke colors.
             * 
             * You can change the size of the rectangle by changing the `width` and `height` properties.
             * @param x The horizontal position of this Game Object in the world. Default 0.
             * @param y The vertical position of this Game Object in the world. Default 0.
             * @param width The width of the rectangle. Default 128.
             * @param height The height of the rectangle. Default 128.
             * @param fillColor The color the rectangle will be filled with, i.e. 0xff0000 for red.
             * @param fillAlpha The alpha the rectangle will be filled with. You can also set the alpha of the overall Shape using its `alpha` property.
             */
            rectangle(x?: number, y?: number, width?: number, height?: number, fillColor?: number, fillAlpha?: number): Phaser.GameObjects.Rectangle;

            /**
             * Creates a new Star Shape Game Object and adds it to the Scene.
             * 
             * Note: This method will only be available if the Star Game Object has been built into Phaser.
             * 
             * The Star Shape is a Game Object that can be added to a Scene, Group or Container. You can
             * treat it like any other Game Object in your game, such as tweening it, scaling it, or enabling
             * it for input or physics. It provides a quick and easy way for you to render this shape in your
             * game without using a texture, while still taking advantage of being fully batched in WebGL.
             * 
             * This shape supports both fill and stroke colors.
             * 
             * As the name implies, the Star shape will display a star in your game. You can control several
             * aspects of it including the number of points that constitute the star. The default is 5. If
             * you change it to 4 it will render as a diamond. If you increase them, you'll get a more spiky
             * star shape.
             * 
             * You can also control the inner and outer radius, which is how 'long' each point of the star is.
             * Modify these values to create more interesting shapes.
             * @param x The horizontal position of this Game Object in the world. Default 0.
             * @param y The vertical position of this Game Object in the world. Default 0.
             * @param points The number of points on the star. Default 5.
             * @param innerRadius The inner radius of the star. Default 32.
             * @param outerRadius The outer radius of the star. Default 64.
             * @param fillColor The color the star will be filled with, i.e. 0xff0000 for red.
             * @param fillAlpha The alpha the star will be filled with. You can also set the alpha of the overall Shape using its `alpha` property.
             */
            star(x?: number, y?: number, points?: number, innerRadius?: number, outerRadius?: number, fillColor?: number, fillAlpha?: number): Phaser.GameObjects.Star;

            /**
             * Creates a new Triangle Shape Game Object and adds it to the Scene.
             * 
             * Note: This method will only be available if the Triangle Game Object has been built into Phaser.
             * 
             * The Triangle Shape is a Game Object that can be added to a Scene, Group or Container. You can
             * treat it like any other Game Object in your game, such as tweening it, scaling it, or enabling
             * it for input or physics. It provides a quick and easy way for you to render this shape in your
             * game without using a texture, while still taking advantage of being fully batched in WebGL.
             * 
             * This shape supports both fill and stroke colors.
             * 
             * The Triangle consists of 3 lines, joining up to form a triangular shape. You can control the
             * position of each point of these lines. The triangle is always closed and cannot have an open
             * face. If you require that, consider using a Polygon instead.
             * @param x The horizontal position of this Game Object in the world. Default 0.
             * @param y The vertical position of this Game Object in the world. Default 0.
             * @param x1 The horizontal position of the first point in the triangle. Default 0.
             * @param y1 The vertical position of the first point in the triangle. Default 128.
             * @param x2 The horizontal position of the second point in the triangle. Default 64.
             * @param y2 The vertical position of the second point in the triangle. Default 0.
             * @param x3 The horizontal position of the third point in the triangle. Default 128.
             * @param y3 The vertical position of the third point in the triangle. Default 128.
             * @param fillColor The color the triangle will be filled with, i.e. 0xff0000 for red.
             * @param fillAlpha The alpha the triangle will be filled with. You can also set the alpha of the overall Shape using its `alpha` property.
             */
            triangle(x?: number, y?: number, x1?: number, y1?: number, x2?: number, y2?: number, x3?: number, y3?: number, fillColor?: number, fillAlpha?: number): Phaser.GameObjects.Triangle;

            /**
             * Creates a new Sprite Game Object and adds it to the Scene.
             * 
             * Note: This method will only be available if the Sprite Game Object has been built into Phaser.
             * @param x The horizontal position of this Game Object in the world.
             * @param y The vertical position of this Game Object in the world.
             * @param texture The key, or instance of the Texture this Game Object will use to render with, as stored in the Texture Manager.
             * @param frame An optional frame from the Texture this Game Object is rendering with.
             */
            sprite(x: number, y: number, texture: string | Phaser.Textures.Texture, frame?: string | number): Phaser.GameObjects.Sprite;

            /**
             * Creates a new Text Game Object and adds it to the Scene.
             * 
             * A Text Game Object.
             * 
             * Text objects work by creating their own internal hidden Canvas and then renders text to it using
             * the standard Canvas `fillText` API. It then creates a texture from this canvas which is rendered
             * to your game during the render pass.
             * 
             * Because it uses the Canvas API you can take advantage of all the features this offers, such as
             * applying gradient fills to the text, or strokes, shadows and more. You can also use custom fonts
             * loaded externally, such as Google or TypeKit Web fonts.
             * 
             * You can only display fonts that are currently loaded and available to the browser: therefore fonts must
             * be pre-loaded. Phaser does not do ths for you, so you will require the use of a 3rd party font loader,
             * or have the fonts ready available in the CSS on the page in which your Phaser game resides.
             * 
             * See {@link http://www.jordanm.co.uk/tinytype this compatibility table} for the available default fonts
             * across mobile browsers.
             * 
             * A note on performance: Every time the contents of a Text object changes, i.e. changing the text being
             * displayed, or the style of the text, it needs to remake the Text canvas, and if on WebGL, re-upload the
             * new texture to the GPU. This can be an expensive operation if used often, or with large quantities of
             * Text objects in your game. If you run into performance issues you would be better off using Bitmap Text
             * instead, as it benefits from batching and avoids expensive Canvas API calls.
             * 
             * Note: This method will only be available if the Text Game Object has been built into Phaser.
             * @param x The horizontal position of this Game Object in the world.
             * @param y The vertical position of this Game Object in the world.
             * @param text The text this Text object will display.
             * @param style The Text style configuration object.
             */
            text(x: number, y: number, text: string | string[], style?: Phaser.Types.GameObjects.Text.TextStyle): Phaser.GameObjects.Text;

            /**
             * Creates a new TileSprite Game Object and adds it to the Scene.
             * 
             * Note: This method will only be available if the TileSprite Game Object has been built into Phaser.
             * @param x The horizontal position of this Game Object in the world.
             * @param y The vertical position of this Game Object in the world.
             * @param width The width of the Game Object. If zero it will use the size of the texture frame.
             * @param height The height of the Game Object. If zero it will use the size of the texture frame.
             * @param texture The key, or instance of the Texture this Game Object will use to render with, as stored in the Texture Manager.
             * @param frame An optional frame from the Texture this Game Object is rendering with.
             */
            tileSprite(x: number, y: number, width: number, height: number, texture: string | Phaser.Textures.Texture, frame?: string | number): Phaser.GameObjects.TileSprite;

            /**
             * Creates a new Video Game Object and adds it to the Scene.
             * 
             * Note: This method will only be available if the Video Game Object has been built into Phaser.
             * @param x The horizontal position of this Game Object in the world.
             * @param y The vertical position of this Game Object in the world.
             * @param key Optional key of the Video this Game Object will play, as stored in the Video Cache.
             */
            video(x: number, y: number, key?: string): Phaser.GameObjects.Video;

            /**
             * Creates a new Zone Game Object and adds it to the Scene.
             * 
             * Note: This method will only be available if the Zone Game Object has been built into Phaser.
             * @param x The horizontal position of this Game Object in the world.
             * @param y The vertical position of this Game Object in the world.
             * @param width The width of the Game Object.
             * @param height The height of the Game Object.
             */
            zone(x: number, y: number, width: number, height: number): Phaser.GameObjects.Zone;

            /**
             * Creates a Tilemap from the given key or data, or creates a blank Tilemap if no key/data provided.
             * When loading from CSV or a 2D array, you should specify the tileWidth & tileHeight. When parsing
             * from a map from Tiled, the tileWidth, tileHeight, width & height will be pulled from the map
             * data. For an empty map, you should specify tileWidth, tileHeight, width & height.
             * @param key The key in the Phaser cache that corresponds to the loaded tilemap data.
             * @param tileWidth The width of a tile in pixels. Pass in `null` to leave as the
             * default. Default 32.
             * @param tileHeight The height of a tile in pixels. Pass in `null` to leave as the
             * default. Default 32.
             * @param width The width of the map in tiles. Pass in `null` to leave as the
             * default. Default 10.
             * @param height The height of the map in tiles. Pass in `null` to leave as the
             * default. Default 10.
             * @param data Instead of loading from the cache, you can also load directly from
             * a 2D array of tile indexes. Pass in `null` for no data.
             * @param insertNull Controls how empty tiles, tiles with an index of -1, in the
             * map data are handled. If `true`, empty locations will get a value of `null`. If `false`, empty
             * location will get a Tile object with an index of -1. If you've a large sparsely populated map and
             * the tile data doesn't need to change then setting this value to `true` will help with memory
             * consumption. However if your map is small or you need to update the tiles dynamically, then leave
             * the default value set. Default false.
             */
            tilemap(key?: string, tileWidth?: number, tileHeight?: number, width?: number, height?: number, data?: number[][], insertNull?: boolean): Phaser.Tilemaps.Tilemap;

            /**
             * Creates a new Tween object.
             * 
             * Note: This method will only be available if Tweens have been built into Phaser.
             * @param config The Tween configuration.
             */
            tween(config: Phaser.Types.Tweens.TweenBuilderConfig | object): Phaser.Tweens.Tween;

        }

        /**
         * Calculates the Transform Matrix of the given Game Object and Camera, factoring in
         * the parent matrix if provided.
         * 
         * Note that the object this results contains _references_ to the Transform Matrices,
         * not new instances of them. Therefore, you should use their values immediately, or
         * copy them to your own matrix, as they will be replaced as soon as another Game
         * Object is rendered.
         * @param src The Game Object to calculate the transform matrix for.
         * @param camera The camera being used to render the Game Object.
         * @param parentMatrix The transform matrix of the parent container, if any.
         */
        function GetCalcMatrix(src: Phaser.GameObjects.GameObject, camera: Phaser.Cameras.Scene2D.Camera, parentMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.Types.GameObjects.GetCalcMatrixResults;

        /**
         * A Graphics object is a way to draw primitive shapes to your game. Primitives include forms of geometry, such as
         * Rectangles, Circles, and Polygons. They also include lines, arcs and curves. When you initially create a Graphics
         * object it will be empty.
         * 
         * To draw to it you must first specify a line style or fill style (or both), draw shapes using paths, and finally
         * fill or stroke them. For example:
         * 
         * ```javascript
         * graphics.lineStyle(5, 0xFF00FF, 1.0);
         * graphics.beginPath();
         * graphics.moveTo(100, 100);
         * graphics.lineTo(200, 200);
         * graphics.closePath();
         * graphics.strokePath();
         * ```
         * 
         * There are also many helpful methods that draw and fill/stroke common shapes for you.
         * 
         * ```javascript
         * graphics.lineStyle(5, 0xFF00FF, 1.0);
         * graphics.fillStyle(0xFFFFFF, 1.0);
         * graphics.fillRect(50, 50, 400, 200);
         * graphics.strokeRect(50, 50, 400, 200);
         * ```
         * 
         * When a Graphics object is rendered it will render differently based on if the game is running under Canvas or WebGL.
         * Under Canvas it will use the HTML Canvas context drawing operations to draw the path.
         * Under WebGL the graphics data is decomposed into polygons. Both of these are expensive processes, especially with
         * complex shapes.
         * 
         * If your Graphics object doesn't change much (or at all) once you've drawn your shape to it, then you will help
         * performance by calling {@link Phaser.GameObjects.Graphics#generateTexture}. This will 'bake' the Graphics object into
         * a Texture, and return it. You can then use this Texture for Sprites or other display objects. If your Graphics object
         * updates frequently then you should avoid doing this, as it will constantly generate new textures, which will consume
         * memory.
         * 
         * As you can tell, Graphics objects are a bit of a trade-off. While they are extremely useful, you need to be careful
         * in their complexity and quantity of them in your game.
         */
        class Graphics extends Phaser.GameObjects.GameObject implements Phaser.GameObjects.Components.AlphaSingle, Phaser.GameObjects.Components.BlendMode, Phaser.GameObjects.Components.Depth, Phaser.GameObjects.Components.Mask, Phaser.GameObjects.Components.Pipeline, Phaser.GameObjects.Components.Transform, Phaser.GameObjects.Components.Visible, Phaser.GameObjects.Components.ScrollFactor {
            /**
             * 
             * @param scene The Scene to which this Graphics object belongs.
             * @param options Options that set the position and default style of this Graphics object.
             */
            constructor(scene: Phaser.Scene, options?: Phaser.Types.GameObjects.Graphics.Options);

            /**
             * The horizontal display origin of the Graphics.
             */
            displayOriginX: number;

            /**
             * The vertical display origin of the Graphics.
             */
            displayOriginY: number;

            /**
             * The array of commands used to render the Graphics.
             */
            commandBuffer: any[];

            /**
             * The default fill color for shapes rendered by this Graphics object.
             */
            defaultFillColor: number;

            /**
             * The default fill alpha for shapes rendered by this Graphics object.
             */
            defaultFillAlpha: number;

            /**
             * The default stroke width for shapes rendered by this Graphics object.
             */
            defaultStrokeWidth: number;

            /**
             * The default stroke color for shapes rendered by this Graphics object.
             */
            defaultStrokeColor: number;

            /**
             * The default stroke alpha for shapes rendered by this Graphics object.
             */
            defaultStrokeAlpha: number;

            /**
             * Set the default style settings for this Graphics object.
             * @param options The styles to set as defaults.
             */
            setDefaultStyles(options: Phaser.Types.GameObjects.Graphics.Styles): this;

            /**
             * Set the current line style.
             * @param lineWidth The stroke width.
             * @param color The stroke color.
             * @param alpha The stroke alpha. Default 1.
             */
            lineStyle(lineWidth: number, color: number, alpha?: number): this;

            /**
             * Set the current fill style.
             * @param color The fill color.
             * @param alpha The fill alpha. Default 1.
             */
            fillStyle(color: number, alpha?: number): this;

            /**
             * Sets a gradient fill style. This is a WebGL only feature.
             * 
             * The gradient color values represent the 4 corners of an untransformed rectangle.
             * The gradient is used to color all filled shapes and paths drawn after calling this method.
             * If you wish to turn a gradient off, call `fillStyle` and provide a new single fill color.
             * 
             * When filling a triangle only the first 3 color values provided are used for the 3 points of a triangle.
             * 
             * This feature is best used only on rectangles and triangles. All other shapes will give strange results.
             * 
             * Note that for objects such as arcs or ellipses, or anything which is made out of triangles, each triangle used
             * will be filled with a gradient on its own. There is no ability to gradient fill a shape or path as a single
             * entity at this time.
             * @param topLeft The top left fill color.
             * @param topRight The top right fill color.
             * @param bottomLeft The bottom left fill color.
             * @param bottomRight The bottom right fill color. Not used when filling triangles.
             * @param alphaTopLeft The top left alpha value. If you give only this value, it's used for all corners. Default 1.
             * @param alphaTopRight The top right alpha value. Default 1.
             * @param alphaBottomLeft The bottom left alpha value. Default 1.
             * @param alphaBottomRight The bottom right alpha value. Default 1.
             */
            fillGradientStyle(topLeft: number, topRight: number, bottomLeft: number, bottomRight: number, alphaTopLeft?: number, alphaTopRight?: number, alphaBottomLeft?: number, alphaBottomRight?: number): this;

            /**
             * Sets a gradient line style. This is a WebGL only feature.
             * 
             * The gradient color values represent the 4 corners of an untransformed rectangle.
             * The gradient is used to color all stroked shapes and paths drawn after calling this method.
             * If you wish to turn a gradient off, call `lineStyle` and provide a new single line color.
             * 
             * This feature is best used only on single lines. All other shapes will give strange results.
             * 
             * Note that for objects such as arcs or ellipses, or anything which is made out of triangles, each triangle used
             * will be filled with a gradient on its own. There is no ability to gradient stroke a shape or path as a single
             * entity at this time.
             * @param lineWidth The stroke width.
             * @param topLeft The tint being applied to the top-left of the Game Object.
             * @param topRight The tint being applied to the top-right of the Game Object.
             * @param bottomLeft The tint being applied to the bottom-left of the Game Object.
             * @param bottomRight The tint being applied to the bottom-right of the Game Object.
             * @param alpha The fill alpha. Default 1.
             */
            lineGradientStyle(lineWidth: number, topLeft: number, topRight: number, bottomLeft: number, bottomRight: number, alpha?: number): this;

            /**
             * Start a new shape path.
             */
            beginPath(): this;

            /**
             * Close the current path.
             */
            closePath(): this;

            /**
             * Fill the current path.
             */
            fillPath(): this;

            /**
             * Fill the current path.
             * 
             * This is an alias for `Graphics.fillPath` and does the same thing.
             * It was added to match the CanvasRenderingContext 2D API.
             */
            fill(): this;

            /**
             * Stroke the current path.
             */
            strokePath(): this;

            /**
             * Stroke the current path.
             * 
             * This is an alias for `Graphics.strokePath` and does the same thing.
             * It was added to match the CanvasRenderingContext 2D API.
             */
            stroke(): this;

            /**
             * Fill the given circle.
             * @param circle The circle to fill.
             */
            fillCircleShape(circle: Phaser.Geom.Circle): this;

            /**
             * Stroke the given circle.
             * @param circle The circle to stroke.
             */
            strokeCircleShape(circle: Phaser.Geom.Circle): this;

            /**
             * Fill a circle with the given position and radius.
             * @param x The x coordinate of the center of the circle.
             * @param y The y coordinate of the center of the circle.
             * @param radius The radius of the circle.
             */
            fillCircle(x: number, y: number, radius: number): this;

            /**
             * Stroke a circle with the given position and radius.
             * @param x The x coordinate of the center of the circle.
             * @param y The y coordinate of the center of the circle.
             * @param radius The radius of the circle.
             */
            strokeCircle(x: number, y: number, radius: number): this;

            /**
             * Fill the given rectangle.
             * @param rect The rectangle to fill.
             */
            fillRectShape(rect: Phaser.Geom.Rectangle): this;

            /**
             * Stroke the given rectangle.
             * @param rect The rectangle to stroke.
             */
            strokeRectShape(rect: Phaser.Geom.Rectangle): this;

            /**
             * Fill a rectangle with the given position and size.
             * @param x The x coordinate of the top-left of the rectangle.
             * @param y The y coordinate of the top-left of the rectangle.
             * @param width The width of the rectangle.
             * @param height The height of the rectangle.
             */
            fillRect(x: number, y: number, width: number, height: number): this;

            /**
             * Stroke a rectangle with the given position and size.
             * @param x The x coordinate of the top-left of the rectangle.
             * @param y The y coordinate of the top-left of the rectangle.
             * @param width The width of the rectangle.
             * @param height The height of the rectangle.
             */
            strokeRect(x: number, y: number, width: number, height: number): this;

            /**
             * Fill a rounded rectangle with the given position, size and radius.
             * @param x The x coordinate of the top-left of the rectangle.
             * @param y The y coordinate of the top-left of the rectangle.
             * @param width The width of the rectangle.
             * @param height The height of the rectangle.
             * @param radius The corner radius; It can also be an object to specify different radii for corners. Default 20.
             */
            fillRoundedRect(x: number, y: number, width: number, height: number, radius?: Phaser.Types.GameObjects.Graphics.RoundedRectRadius | number): this;

            /**
             * Stroke a rounded rectangle with the given position, size and radius.
             * @param x The x coordinate of the top-left of the rectangle.
             * @param y The y coordinate of the top-left of the rectangle.
             * @param width The width of the rectangle.
             * @param height The height of the rectangle.
             * @param radius The corner radius; It can also be an object to specify different radii for corners. Default 20.
             */
            strokeRoundedRect(x: number, y: number, width: number, height: number, radius?: Phaser.Types.GameObjects.Graphics.RoundedRectRadius | number): this;

            /**
             * Fill the given point.
             * 
             * Draws a square at the given position, 1 pixel in size by default.
             * @param point The point to fill.
             * @param size The size of the square to draw. Default 1.
             */
            fillPointShape(point: Phaser.Geom.Point | Phaser.Math.Vector2 | object, size?: number): this;

            /**
             * Fill a point at the given position.
             * 
             * Draws a square at the given position, 1 pixel in size by default.
             * @param x The x coordinate of the point.
             * @param y The y coordinate of the point.
             * @param size The size of the square to draw. Default 1.
             */
            fillPoint(x: number, y: number, size?: number): this;

            /**
             * Fill the given triangle.
             * @param triangle The triangle to fill.
             */
            fillTriangleShape(triangle: Phaser.Geom.Triangle): this;

            /**
             * Stroke the given triangle.
             * @param triangle The triangle to stroke.
             */
            strokeTriangleShape(triangle: Phaser.Geom.Triangle): this;

            /**
             * Fill a triangle with the given points.
             * @param x0 The x coordinate of the first point.
             * @param y0 The y coordinate of the first point.
             * @param x1 The x coordinate of the second point.
             * @param y1 The y coordinate of the second point.
             * @param x2 The x coordinate of the third point.
             * @param y2 The y coordinate of the third point.
             */
            fillTriangle(x0: number, y0: number, x1: number, y1: number, x2: number, y2: number): this;

            /**
             * Stroke a triangle with the given points.
             * @param x0 The x coordinate of the first point.
             * @param y0 The y coordinate of the first point.
             * @param x1 The x coordinate of the second point.
             * @param y1 The y coordinate of the second point.
             * @param x2 The x coordinate of the third point.
             * @param y2 The y coordinate of the third point.
             */
            strokeTriangle(x0: number, y0: number, x1: number, y1: number, x2: number, y2: number): this;

            /**
             * Draw the given line.
             * @param line The line to stroke.
             */
            strokeLineShape(line: Phaser.Geom.Line): this;

            /**
             * Draw a line between the given points.
             * @param x1 The x coordinate of the start point of the line.
             * @param y1 The y coordinate of the start point of the line.
             * @param x2 The x coordinate of the end point of the line.
             * @param y2 The y coordinate of the end point of the line.
             */
            lineBetween(x1: number, y1: number, x2: number, y2: number): this;

            /**
             * Draw a line from the current drawing position to the given position.
             * 
             * Moves the current drawing position to the given position.
             * @param x The x coordinate to draw the line to.
             * @param y The y coordinate to draw the line to.
             */
            lineTo(x: number, y: number): this;

            /**
             * Move the current drawing position to the given position.
             * @param x The x coordinate to move to.
             * @param y The y coordinate to move to.
             */
            moveTo(x: number, y: number): this;

            /**
             * Stroke the shape represented by the given array of points.
             * 
             * Pass `closeShape` to automatically close the shape by joining the last to the first point.
             * 
             * Pass `closePath` to automatically close the path before it is stroked.
             * @param points The points to stroke.
             * @param closeShape When `true`, the shape is closed by joining the last point to the first point. Default false.
             * @param closePath When `true`, the path is closed before being stroked. Default false.
             * @param endIndex The index of `points` to stop drawing at. Defaults to `points.length`.
             */
            strokePoints(points: any[] | Phaser.Geom.Point[], closeShape?: boolean, closePath?: boolean, endIndex?: number): this;

            /**
             * Fill the shape represented by the given array of points.
             * 
             * Pass `closeShape` to automatically close the shape by joining the last to the first point.
             * 
             * Pass `closePath` to automatically close the path before it is filled.
             * @param points The points to fill.
             * @param closeShape When `true`, the shape is closed by joining the last point to the first point. Default false.
             * @param closePath When `true`, the path is closed before being stroked. Default false.
             * @param endIndex The index of `points` to stop at. Defaults to `points.length`.
             */
            fillPoints(points: any[] | Phaser.Geom.Point[], closeShape?: boolean, closePath?: boolean, endIndex?: number): this;

            /**
             * Stroke the given ellipse.
             * @param ellipse The ellipse to stroke.
             * @param smoothness The number of points to draw the ellipse with. Default 32.
             */
            strokeEllipseShape(ellipse: Phaser.Geom.Ellipse, smoothness?: number): this;

            /**
             * Stroke an ellipse with the given position and size.
             * @param x The x coordinate of the center of the ellipse.
             * @param y The y coordinate of the center of the ellipse.
             * @param width The width of the ellipse.
             * @param height The height of the ellipse.
             * @param smoothness The number of points to draw the ellipse with. Default 32.
             */
            strokeEllipse(x: number, y: number, width: number, height: number, smoothness?: number): this;

            /**
             * Fill the given ellipse.
             * @param ellipse The ellipse to fill.
             * @param smoothness The number of points to draw the ellipse with. Default 32.
             */
            fillEllipseShape(ellipse: Phaser.Geom.Ellipse, smoothness?: number): this;

            /**
             * Fill an ellipse with the given position and size.
             * @param x The x coordinate of the center of the ellipse.
             * @param y The y coordinate of the center of the ellipse.
             * @param width The width of the ellipse.
             * @param height The height of the ellipse.
             * @param smoothness The number of points to draw the ellipse with. Default 32.
             */
            fillEllipse(x: number, y: number, width: number, height: number, smoothness?: number): this;

            /**
             * Draw an arc.
             * 
             * This method can be used to create circles, or parts of circles.
             * 
             * Make sure you call `beginPath` before starting the arc unless you wish for the arc to automatically
             * close when filled or stroked.
             * 
             * Use the optional `overshoot` argument increase the number of iterations that take place when
             * the arc is rendered in WebGL. This is useful if you're drawing an arc with an especially thick line,
             * as it will allow the arc to fully join-up. Try small values at first, i.e. 0.01.
             * 
             * Call {@link Phaser.GameObjects.Graphics#fillPath} or {@link Phaser.GameObjects.Graphics#strokePath} after calling
             * this method to draw the arc.
             * @param x The x coordinate of the center of the circle.
             * @param y The y coordinate of the center of the circle.
             * @param radius The radius of the circle.
             * @param startAngle The starting angle, in radians.
             * @param endAngle The ending angle, in radians.
             * @param anticlockwise Whether the drawing should be anticlockwise or clockwise. Default false.
             * @param overshoot This value allows you to increase the segment iterations in WebGL rendering. Useful if the arc has a thick stroke and needs to overshoot to join-up cleanly. Use small numbers such as 0.01 to start with and increase as needed. Default 0.
             */
            arc(x: number, y: number, radius: number, startAngle: number, endAngle: number, anticlockwise?: boolean, overshoot?: number): this;

            /**
             * Creates a pie-chart slice shape centered at `x`, `y` with the given radius.
             * You must define the start and end angle of the slice.
             * 
             * Setting the `anticlockwise` argument to `true` creates a shape similar to Pacman.
             * Setting it to `false` creates a shape like a slice of pie.
             * 
             * This method will begin a new path and close the path at the end of it.
             * To display the actual slice you need to call either `strokePath` or `fillPath` after it.
             * @param x The horizontal center of the slice.
             * @param y The vertical center of the slice.
             * @param radius The radius of the slice.
             * @param startAngle The start angle of the slice, given in radians.
             * @param endAngle The end angle of the slice, given in radians.
             * @param anticlockwise Whether the drawing should be anticlockwise or clockwise. Default false.
             * @param overshoot This value allows you to overshoot the endAngle by this amount. Useful if the arc has a thick stroke and needs to overshoot to join-up cleanly. Default 0.
             */
            slice(x: number, y: number, radius: number, startAngle: number, endAngle: number, anticlockwise?: boolean, overshoot?: number): this;

            /**
             * Saves the state of the Graphics by pushing the current state onto a stack.
             * 
             * The most recently saved state can then be restored with {@link Phaser.GameObjects.Graphics#restore}.
             */
            save(): this;

            /**
             * Restores the most recently saved state of the Graphics by popping from the state stack.
             * 
             * Use {@link Phaser.GameObjects.Graphics#save} to save the current state, and call this afterwards to restore that state.
             * 
             * If there is no saved state, this command does nothing.
             */
            restore(): this;

            /**
             * Inserts a translation command into this Graphics objects command buffer.
             * 
             * All objects drawn _after_ calling this method will be translated
             * by the given amount.
             * 
             * This does not change the position of the Graphics object itself,
             * only of the objects drawn by it after calling this method.
             * @param x The horizontal translation to apply.
             * @param y The vertical translation to apply.
             */
            translateCanvas(x: number, y: number): this;

            /**
             * Inserts a scale command into this Graphics objects command buffer.
             * 
             * All objects drawn _after_ calling this method will be scaled
             * by the given amount.
             * 
             * This does not change the scale of the Graphics object itself,
             * only of the objects drawn by it after calling this method.
             * @param x The horizontal scale to apply.
             * @param y The vertical scale to apply.
             */
            scaleCanvas(x: number, y: number): this;

            /**
             * Inserts a rotation command into this Graphics objects command buffer.
             * 
             * All objects drawn _after_ calling this method will be rotated
             * by the given amount.
             * 
             * This does not change the rotation of the Graphics object itself,
             * only of the objects drawn by it after calling this method.
             * @param radians The rotation angle, in radians.
             */
            rotateCanvas(radians: number): this;

            /**
             * Clear the command buffer and reset the fill style and line style to their defaults.
             */
            clear(): this;

            /**
             * Generate a texture from this Graphics object.
             * 
             * If `key` is a string it'll generate a new texture using it and add it into the
             * Texture Manager (assuming no key conflict happens).
             * 
             * If `key` is a Canvas it will draw the texture to that canvas context. Note that it will NOT
             * automatically upload it to the GPU in WebGL mode.
             * 
             * Please understand that the texture is created via the Canvas API of the browser, therefore some
             * Graphics features, such as `fillGradientStyle`, will not appear on the resulting texture,
             * as they're unsupported by the Canvas API.
             * @param key The key to store the texture with in the Texture Manager, or a Canvas to draw to.
             * @param width The width of the graphics to generate.
             * @param height The height of the graphics to generate.
             */
            generateTexture(key: string | HTMLCanvasElement, width?: number, height?: number): this;

            /**
             * Internal destroy handler, called as part of the destroy process.
             */
            protected preDestroy(): void;

            /**
             * A Camera used specifically by the Graphics system for rendering to textures.
             */
            static TargetCamera: Phaser.Cameras.Scene2D.Camera;

            /**
             * Clears all alpha values associated with this Game Object.
             * 
             * Immediately sets the alpha levels back to 1 (fully opaque).
             */
            clearAlpha(): this;

            /**
             * Set the Alpha level of this Game Object. The alpha controls the opacity of the Game Object as it renders.
             * Alpha values are provided as a float between 0, fully transparent, and 1, fully opaque.
             * @param value The alpha value applied across the whole Game Object. Default 1.
             */
            setAlpha(value?: number): this;

            /**
             * The alpha value of the Game Object.
             * 
             * This is a global value, impacting the entire Game Object, not just a region of it.
             */
            alpha: number;

            /**
             * Sets the Blend Mode being used by this Game Object.
             * 
             * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
             * 
             * Under WebGL only the following Blend Modes are available:
             * 
             * * ADD
             * * MULTIPLY
             * * SCREEN
             * * ERASE
             * 
             * Canvas has more available depending on browser support.
             * 
             * You can also create your own custom Blend Modes in WebGL.
             * 
             * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
             * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
             * reasons try to be careful about the construction of your Scene and the frequency of which blend modes
             * are used.
             */
            blendMode: Phaser.BlendModes | string;

            /**
             * Sets the Blend Mode being used by this Game Object.
             * 
             * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
             * 
             * Under WebGL only the following Blend Modes are available:
             * 
             * * ADD
             * * MULTIPLY
             * * SCREEN
             * * ERASE (only works when rendering to a framebuffer, like a Render Texture)
             * 
             * Canvas has more available depending on browser support.
             * 
             * You can also create your own custom Blend Modes in WebGL.
             * 
             * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
             * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
             * reasons try to be careful about the construction of your Scene and the frequency in which blend modes
             * are used.
             * @param value The BlendMode value. Either a string or a CONST.
             */
            setBlendMode(value: string | Phaser.BlendModes): this;

            /**
             * The depth of this Game Object within the Scene.
             * 
             * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
             * of Game Objects, without actually moving their position in the display list.
             * 
             * The default depth is zero. A Game Object with a higher depth
             * value will always render in front of one with a lower value.
             * 
             * Setting the depth will queue a depth sort event within the Scene.
             */
            depth: number;

            /**
             * The depth of this Game Object within the Scene.
             * 
             * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
             * of Game Objects, without actually moving their position in the display list.
             * 
             * The default depth is zero. A Game Object with a higher depth
             * value will always render in front of one with a lower value.
             * 
             * Setting the depth will queue a depth sort event within the Scene.
             * @param value The depth of this Game Object.
             */
            setDepth(value: number): this;

            /**
             * The Mask this Game Object is using during render.
             */
            mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask;

            /**
             * Sets the mask that this Game Object will use to render with.
             * 
             * The mask must have been previously created and can be either a GeometryMask or a BitmapMask.
             * Note: Bitmap Masks only work on WebGL. Geometry Masks work on both WebGL and Canvas.
             * 
             * If a mask is already set on this Game Object it will be immediately replaced.
             * 
             * Masks are positioned in global space and are not relative to the Game Object to which they
             * are applied. The reason for this is that multiple Game Objects can all share the same mask.
             * 
             * Masks have no impact on physics or input detection. They are purely a rendering component
             * that allows you to limit what is visible during the render pass.
             * @param mask The mask this Game Object will use when rendering.
             */
            setMask(mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask): this;

            /**
             * Clears the mask that this Game Object was using.
             * @param destroyMask Destroy the mask before clearing it? Default false.
             */
            clearMask(destroyMask?: boolean): this;

            /**
             * Creates and returns a Bitmap Mask. This mask can be used by any Game Object,
             * including this one.
             * 
             * Note: Bitmap Masks only work on WebGL. Geometry Masks work on both WebGL and Canvas.
             * 
             * To create the mask you need to pass in a reference to a renderable Game Object.
             * A renderable Game Object is one that uses a texture to render with, such as an
             * Image, Sprite, Render Texture or BitmapText.
             * 
             * If you do not provide a renderable object, and this Game Object has a texture,
             * it will use itself as the object. This means you can call this method to create
             * a Bitmap Mask from any renderable Game Object.
             * @param renderable A renderable Game Object that uses a texture, such as a Sprite.
             */
            createBitmapMask(renderable?: Phaser.GameObjects.GameObject): Phaser.Display.Masks.BitmapMask;

            /**
             * Creates and returns a Geometry Mask. This mask can be used by any Game Object,
             * including this one.
             * 
             * To create the mask you need to pass in a reference to a Graphics Game Object.
             * 
             * If you do not provide a graphics object, and this Game Object is an instance
             * of a Graphics object, then it will use itself to create the mask.
             * 
             * This means you can call this method to create a Geometry Mask from any Graphics Game Object.
             * @param graphics A Graphics Game Object. The geometry within it will be used as the mask.
             */
            createGeometryMask(graphics?: Phaser.GameObjects.Graphics): Phaser.Display.Masks.GeometryMask;

            /**
             * The initial WebGL pipeline of this Game Object.
             * 
             * If you call `resetPipeline` on this Game Object, the pipeline is reset to this default.
             */
            defaultPipeline: Phaser.Renderer.WebGL.WebGLPipeline;

            /**
             * The current WebGL pipeline of this Game Object.
             */
            pipeline: Phaser.Renderer.WebGL.WebGLPipeline;

            /**
             * Does this Game Object have any Post Pipelines set?
             */
            hasPostPipeline: boolean;

            /**
             * The WebGL Post FX Pipelines this Game Object uses for post-render effects.
             * 
             * The pipelines are processed in the order in which they appear in this array.
             * 
             * If you modify this array directly, be sure to set the
             * `hasPostPipeline` property accordingly.
             */
            postPipelines: Phaser.Renderer.WebGL.Pipelines.PostFXPipeline[];

            /**
             * An object to store pipeline specific data in, to be read by the pipelines this Game Object uses.
             */
            pipelineData: object;

            /**
             * Sets the initial WebGL Pipeline of this Game Object.
             * 
             * This should only be called during the instantiation of the Game Object. After that, use `setPipeline`.
             * @param pipeline Either the string-based name of the pipeline, or a pipeline instance to set.
             */
            initPipeline(pipeline: string | Phaser.Renderer.WebGL.WebGLPipeline): boolean;

            /**
             * Sets the main WebGL Pipeline of this Game Object.
             * 
             * Also sets the `pipelineData` property, if the parameter is given.
             * 
             * Both the pipeline and post pipelines share the same pipeline data object.
             * @param pipeline Either the string-based name of the pipeline, or a pipeline instance to set.
             * @param pipelineData Optional pipeline data object that is _deep copied_ into the `pipelineData` property of this Game Object.
             * @param copyData Should the pipeline data object be _deep copied_ into the `pipelineData` property of this Game Object? If `false` it will be set by reference instead. Default true.
             */
            setPipeline(pipeline: string | Phaser.Renderer.WebGL.WebGLPipeline, pipelineData?: object, copyData?: boolean): this;

            /**
             * Sets one, or more, Post Pipelines on this Game Object.
             * 
             * Post Pipelines are invoked after this Game Object has rendered to its target and
             * are commonly used for post-fx.
             * 
             * The post pipelines are appended to the `postPipelines` array belonging to this
             * Game Object. When the renderer processes this Game Object, it iterates through the post
             * pipelines in the order in which they appear in the array. If you are stacking together
             * multiple effects, be aware that the order is important.
             * 
             * If you call this method multiple times, the new pipelines will be appended to any existing
             * post pipelines already set. Use the `resetPostPipeline` method to clear them first, if required.
             * 
             * You can optionally also sets the `pipelineData` property, if the parameter is given.
             * 
             * Both the pipeline and post pipelines share the pipeline data object together.
             * @param pipelines Either the string-based name of the pipeline, or a pipeline instance, or class, or an array of them.
             * @param pipelineData Optional pipeline data object that is _deep copied_ into the `pipelineData` property of this Game Object.
             * @param copyData Should the pipeline data object be _deep copied_ into the `pipelineData` property of this Game Object? If `false` it will be set by reference instead. Default true.
             */
            setPostPipeline(pipelines: string | string[] | Function | Function[] | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline[], pipelineData?: object, copyData?: boolean): this;

            /**
             * Adds an entry to the `pipelineData` object belonging to this Game Object.
             * 
             * If the 'key' already exists, its value is updated. If it doesn't exist, it is created.
             * 
             * If `value` is undefined, and `key` exists, `key` is removed from the data object.
             * 
             * Both the pipeline and post pipelines share the pipeline data object together.
             * @param key The key of the pipeline data to set, update, or delete.
             * @param value The value to be set with the key. If `undefined` then `key` will be deleted from the object.
             */
            setPipelineData(key: string, value?: any): this;

            /**
             * Gets a Post Pipeline instance from this Game Object, based on the given name, and returns it.
             * @param pipeline The string-based name of the pipeline, or a pipeline class.
             */
            getPostPipeline(pipeline: string | Function | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline): Phaser.Renderer.WebGL.Pipelines.PostFXPipeline | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline[];

            /**
             * Resets the WebGL Pipeline of this Game Object back to the default it was created with.
             * @param resetPostPipelines Reset all of the post pipelines? Default false.
             * @param resetData Reset the `pipelineData` object to being an empty object? Default false.
             */
            resetPipeline(resetPostPipelines?: boolean, resetData?: boolean): boolean;

            /**
             * Resets the WebGL Post Pipelines of this Game Object. It does this by calling
             * the `destroy` method on each post pipeline and then clearing the local array.
             * @param resetData Reset the `pipelineData` object to being an empty object? Default false.
             */
            resetPostPipeline(resetData?: boolean): void;

            /**
             * Removes a type of Post Pipeline instances from this Game Object, based on the given name, and destroys them.
             * 
             * If you wish to remove all Post Pipelines use the `resetPostPipeline` method instead.
             * @param pipeline The string-based name of the pipeline, or a pipeline class.
             */
            removePostPipeline(pipeline: string | Phaser.Renderer.WebGL.Pipelines.PostFXPipeline): this;

            /**
             * Gets the name of the WebGL Pipeline this Game Object is currently using.
             */
            getPipelineName(): string;

            /**
             * The x position of this Game Object.
             */
            x: number;

            /**
             * The y position of this Game Object.
             */
            y: number;

            /**
             * The z position of this Game Object.
             * 
             * Note: The z position does not control the rendering order of 2D Game Objects. Use
             * {@link Phaser.GameObjects.Components.Depth#depth} instead.
             */
            z: number;

            /**
             * The w position of this Game Object.
             */
            w: number;

            /**
             * This is a special setter that allows you to set both the horizontal and vertical scale of this Game Object
             * to the same value, at the same time. When reading this value the result returned is `(scaleX + scaleY) / 2`.
             * 
             * Use of this property implies you wish the horizontal and vertical scales to be equal to each other. If this
             * isn't the case, use the `scaleX` or `scaleY` properties instead.
             */
            scale: number;

            /**
             * The horizontal scale of this Game Object.
             */
            scaleX: number;

            /**
             * The vertical scale of this Game Object.
             */
            scaleY: number;

            /**
             * The angle of this Game Object as expressed in degrees.
             * 
             * Phaser uses a right-hand clockwise rotation system, where 0 is right, 90 is down, 180/-180 is left
             * and -90 is up.
             * 
             * If you prefer to work in radians, see the `rotation` property instead.
             */
            angle: number;

            /**
             * The angle of this Game Object in radians.
             * 
             * Phaser uses a right-hand clockwise rotation system, where 0 is right, PI/2 is down, +-PI is left
             * and -PI/2 is up.
             * 
             * If you prefer to work in degrees, see the `angle` property instead.
             */
            rotation: number;

            /**
             * Sets the position of this Game Object.
             * @param x The x position of this Game Object. Default 0.
             * @param y The y position of this Game Object. If not set it will use the `x` value. Default x.
             * @param z The z position of this Game Object. Default 0.
             * @param w The w position of this Game Object. Default 0.
             */
            setPosition(x?: number, y?: number, z?: number, w?: number): this;

            /**
             * Copies an object's coordinates to this Game Object's position.
             * @param source An object with numeric 'x', 'y', 'z', or 'w' properties. Undefined values are not copied.
             */
            copyPosition(source: Phaser.Types.Math.Vector2Like | Phaser.Types.Math.Vector3Like | Phaser.Types.Math.Vector4Like): this;

            /**
             * Sets the position of this Game Object to be a random position within the confines of
             * the given area.
             * 
             * If no area is specified a random position between 0 x 0 and the game width x height is used instead.
             * 
             * The position does not factor in the size of this Game Object, meaning that only the origin is
             * guaranteed to be within the area.
             * @param x The x position of the top-left of the random area. Default 0.
             * @param y The y position of the top-left of the random area. Default 0.
             * @param width The width of the random area.
             * @param height The height of the random area.
             */
            setRandomPosition(x?: number, y?: number, width?: number, height?: number): this;

            /**
             * Sets the rotation of this Game Object.
             * @param radians The rotation of this Game Object, in radians. Default 0.
             */
            setRotation(radians?: number): this;

            /**
             * Sets the angle of this Game Object.
             * @param degrees The rotation of this Game Object, in degrees. Default 0.
             */
            setAngle(degrees?: number): this;

            /**
             * Sets the scale of this Game Object.
             * @param x The horizontal scale of this Game Object.
             * @param y The vertical scale of this Game Object. If not set it will use the `x` value. Default x.
             */
            setScale(x: number, y?: number): this;

            /**
             * Sets the x position of this Game Object.
             * @param value The x position of this Game Object. Default 0.
             */
            setX(value?: number): this;

            /**
             * Sets the y position of this Game Object.
             * @param value The y position of this Game Object. Default 0.
             */
            setY(value?: number): this;

            /**
             * Sets the z position of this Game Object.
             * 
             * Note: The z position does not control the rendering order of 2D Game Objects. Use
             * {@link Phaser.GameObjects.Components.Depth#setDepth} instead.
             * @param value The z position of this Game Object. Default 0.
             */
            setZ(value?: number): this;

            /**
             * Sets the w position of this Game Object.
             * @param value The w position of this Game Object. Default 0.
             */
            setW(value?: number): this;

            /**
             * Gets the local transform matrix for this Game Object.
             * @param tempMatrix The matrix to populate with the values from this Game Object.
             */
            getLocalTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;

            /**
             * Gets the world transform matrix for this Game Object, factoring in any parent Containers.
             * @param tempMatrix The matrix to populate with the values from this Game Object.
             * @param parentMatrix A temporary matrix to hold parent values during the calculations.
             */
            getWorldTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix, parentMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;

            /**
             * Takes the given `x` and `y` coordinates and converts them into local space for this
             * Game Object, taking into account parent and local t