const WORKER_CODE = "(function () {\n    'use strict';\n\n    const converters = {\n      rgb8unorm: {\n        convertedFormat: \"rgba8unorm\",\n        convertFunction: convertRGBtoRGBA\n      },\n      \"rgb8unorm-srgb\": {\n        convertedFormat: \"rgba8unorm-srgb\",\n        convertFunction: convertRGBtoRGBA\n      }\n    };\n    function convertFormatIfRequired(textureOptions) {\n      const format = textureOptions.format;\n      if (converters[format]) {\n        const convertFunction = converters[format].convertFunction;\n        const levelBuffers = textureOptions.resource;\n        for (let i = 0; i < levelBuffers.length; i++) {\n          levelBuffers[i] = convertFunction(levelBuffers[i]);\n        }\n        textureOptions.format = converters[format].convertedFormat;\n      }\n    }\n    function convertRGBtoRGBA(levelBuffer) {\n      const pixelCount = levelBuffer.byteLength / 3;\n      const levelBufferWithAlpha = new Uint32Array(pixelCount);\n      for (let i = 0; i < pixelCount; ++i) {\n        levelBufferWithAlpha[i] = levelBuffer[i * 3] + (levelBuffer[i * 3 + 1] << 8) + (levelBuffer[i * 3 + 2] << 16) + 4278190080;\n      }\n      return new Uint8Array(levelBufferWithAlpha.buffer);\n    }\n\n    function createLevelBuffersFromKTX(ktxTexture) {\n      const levelBuffers = [];\n      for (let i = 0; i < ktxTexture.numLevels; i++) {\n        const imageData = ktxTexture.getImageData(i, 0, 0);\n        const levelBuffer = new Uint8Array(imageData.byteLength);\n        levelBuffer.set(imageData);\n        levelBuffers.push(levelBuffer);\n      }\n      return levelBuffers;\n    }\n\n    const glFormatToGPUFormatMap = {\n      6408: \"rgba8unorm\",\n      32856: \"bgra8unorm\",\n      //\n      32857: \"rgb10a2unorm\",\n      33189: \"depth16unorm\",\n      33190: \"depth24plus\",\n      33321: \"r8unorm\",\n      33323: \"rg8unorm\",\n      33325: \"r16float\",\n      33326: \"r32float\",\n      33327: \"rg16float\",\n      33328: \"rg32float\",\n      33329: \"r8sint\",\n      33330: \"r8uint\",\n      33331: \"r16sint\",\n      33332: \"r16uint\",\n      33333: \"r32sint\",\n      33334: \"r32uint\",\n      33335: \"rg8sint\",\n      33336: \"rg8uint\",\n      33337: \"rg16sint\",\n      33338: \"rg16uint\",\n      33339: \"rg32sint\",\n      33340: \"rg32uint\",\n      33778: \"bc2-rgba-unorm\",\n      33779: \"bc3-rgba-unorm\",\n      34836: \"rgba32float\",\n      34842: \"rgba16float\",\n      35056: \"depth24plus-stencil8\",\n      35898: \"rg11b10ufloat\",\n      35901: \"rgb9e5ufloat\",\n      35907: \"rgba8unorm-srgb\",\n      // bgra8unorm-srgb\n      36012: \"depth32float\",\n      36013: \"depth32float-stencil8\",\n      36168: \"stencil8\",\n      36208: \"rgba32uint\",\n      36214: \"rgba16uint\",\n      36220: \"rgba8uint\",\n      36226: \"rgba32sint\",\n      36232: \"rgba16sint\",\n      36238: \"rgba8sint\",\n      36492: \"bc7-rgba-unorm\",\n      36756: \"r8snorm\",\n      36757: \"rg8snorm\",\n      36759: \"rgba8snorm\",\n      37496: \"etc2-rgba8unorm\",\n      37808: \"astc-4x4-unorm\"\n    };\n    function glFormatToGPUFormat(glInternalFormat) {\n      const format = glFormatToGPUFormatMap[glInternalFormat];\n      if (format) {\n        return format;\n      }\n      throw new Error(`Unsupported glInternalFormat: ${glInternalFormat}`);\n    }\n\n    const vkFormatToGPUFormatMap = {\n      23: \"rgb8unorm\",\n      // VK_FORMAT_R8G8B8_UNORM\n      37: \"rgba8unorm\",\n      // VK_FORMAT_R8G8B8A8_UNORM\n      43: \"rgba8unorm-srgb\"\n      // VK_FORMAT_R8G8B8A8_SRGB\n      // TODO add more!\n    };\n    function vkFormatToGPUFormat(vkFormat) {\n      const format = vkFormatToGPUFormatMap[vkFormat];\n      if (format) {\n        return format;\n      }\n      throw new Error(`Unsupported VkFormat: ${vkFormat}`);\n    }\n\n    function getTextureFormatFromKTXTexture(ktxTexture) {\n      if (ktxTexture.classId === 2) {\n        return vkFormatToGPUFormat(ktxTexture.vkFormat);\n      }\n      return glFormatToGPUFormat(ktxTexture.glInternalformat);\n    }\n\n    const gpuFormatToBasisTranscoderFormatMap = {\n      \"bc3-rgba-unorm\": \"BC3_RGBA\",\n      \"bc7-rgba-unorm\": \"BC7_M5_RGBA\",\n      \"etc2-rgba8unorm\": \"ETC2_RGBA\",\n      \"astc-4x4-unorm\": \"ASTC_4x4_RGBA\",\n      // Uncompressed\n      rgba8unorm: \"RGBA32\",\n      rg11b10ufloat: \"R11F_G11F_B10F\"\n    };\n    function gpuFormatToKTXBasisTranscoderFormat(transcoderFormat) {\n      const format = gpuFormatToBasisTranscoderFormatMap[transcoderFormat];\n      if (format) {\n        return format;\n      }\n      throw new Error(`Unsupported transcoderFormat: ${transcoderFormat}`);\n    }\n\n    const settings = {\n      jsUrl: \"\",\n      wasmUrl: \"\"\n    };\n    let basisTranscoderFormat;\n    let basisTranscodedTextureFormat;\n    let ktxPromise;\n    async function getKTX() {\n      if (!ktxPromise) {\n        const absoluteJsUrl = new URL(settings.jsUrl, location.origin).href;\n        const absoluteWasmUrl = new URL(settings.wasmUrl, location.origin).href;\n        importScripts(absoluteJsUrl);\n        ktxPromise = new Promise((resolve) => {\n          LIBKTX({\n            locateFile: (_file) => absoluteWasmUrl\n          }).then((libktx) => {\n            resolve(libktx);\n          });\n        });\n      }\n      return ktxPromise;\n    }\n    async function fetchKTXTexture(url, ktx) {\n      const ktx2Response = await fetch(url);\n      if (ktx2Response.ok) {\n        const ktx2ArrayBuffer = await ktx2Response.arrayBuffer();\n        return new ktx.ktxTexture(new Uint8Array(ktx2ArrayBuffer));\n      }\n      throw new Error(`Failed to load KTX(2) texture: ${url}`);\n    }\n    const preferredTranscodedFormat = [\n      \"bc7-rgba-unorm\",\n      \"astc-4x4-unorm\",\n      \"etc2-rgba8unorm\",\n      \"bc3-rgba-unorm\",\n      \"rgba8unorm\"\n    ];\n    async function load(url) {\n      const ktx = await getKTX();\n      const ktxTexture = await fetchKTXTexture(url, ktx);\n      let format;\n      if (ktxTexture.needsTranscoding) {\n        format = basisTranscodedTextureFormat;\n        const transcodeFormat = ktx.TranscodeTarget[basisTranscoderFormat];\n        const result = ktxTexture.transcodeBasis(transcodeFormat, 0);\n        if (result !== ktx.ErrorCode.SUCCESS) {\n          throw new Error(\"Unable to transcode basis texture.\");\n        }\n      } else {\n        format = getTextureFormatFromKTXTexture(ktxTexture);\n      }\n      const levelBuffers = createLevelBuffersFromKTX(ktxTexture);\n      const textureOptions = {\n        width: ktxTexture.baseWidth,\n        height: ktxTexture.baseHeight,\n        format,\n        mipLevelCount: ktxTexture.numLevels,\n        resource: levelBuffers,\n        alphaMode: \"no-premultiply-alpha\"\n      };\n      convertFormatIfRequired(textureOptions);\n      return textureOptions;\n    }\n    async function init(jsUrl, wasmUrl, supportedTextures) {\n      if (jsUrl)\n        settings.jsUrl = jsUrl;\n      if (wasmUrl)\n        settings.wasmUrl = wasmUrl;\n      basisTranscodedTextureFormat = preferredTranscodedFormat.filter((format) => supportedTextures.includes(format))[0];\n      basisTranscoderFormat = gpuFormatToKTXBasisTranscoderFormat(basisTranscodedTextureFormat);\n      await getKTX();\n    }\n    const messageHandlers = {\n      init: async (data) => {\n        const { jsUrl, wasmUrl, supportedTextures } = data;\n        await init(jsUrl, wasmUrl, supportedTextures);\n      },\n      load: async (data) => {\n        try {\n          const textureOptions = await load(data.url);\n          return {\n            type: \"load\",\n            url: data.url,\n            success: true,\n            textureOptions,\n            transferables: textureOptions.resource?.map((arr) => arr.buffer)\n          };\n        } catch (e) {\n          throw e;\n        }\n      }\n    };\n    self.onmessage = async (messageEvent) => {\n      const message = messageEvent.data;\n      const response = await messageHandlers[message.type]?.(message);\n      if (response) {\n        self.postMessage(response, response.transferables);\n      }\n    };\n\n})();\n";
let WORKER_URL = null;
class WorkerInstance
{
    constructor()
    {
        if (!WORKER_URL)
        {
            WORKER_URL = URL.createObjectURL(new Blob([WORKER_CODE], { type: 'application/javascript' }));
        }
        this.worker = new Worker(WORKER_URL);
    }
}
WorkerInstance.revokeObjectURL = function revokeObjectURL()
{
    if (WORKER_URL)
    {
        URL.revokeObjectURL(WORKER_URL);
        WORKER_URL = null;
    }
};

export { WorkerInstance as default };
//# sourceMappingURL=ktx.worker.mjs.map
