'use strict';

var Extensions = require('../../../../extensions/Extensions.js');
var clean = require('../../../../utils/data/clean.js');

"use strict";
const _RenderableGCSystem = class _RenderableGCSystem {
  /** @param renderer - The renderer this System works for. */
  constructor(renderer) {
    this._managedRenderables = [];
    this._managedHashes = [];
    this._managedArrays = [];
    this._renderer = renderer;
  }
  init(options) {
    options = { ..._RenderableGCSystem.defaultOptions, ...options };
    this.maxUnusedTime = options.renderableGCMaxUnusedTime;
    this._frequency = options.renderableGCFrequency;
    this.enabled = options.renderableGCActive;
  }
  get enabled() {
    return !!this._handler;
  }
  set enabled(value) {
    if (this.enabled === value)
      return;
    if (value) {
      this._handler = this._renderer.scheduler.repeat(
        () => this.run(),
        this._frequency,
        false
      );
      this._hashHandler = this._renderer.scheduler.repeat(
        () => {
          for (const hash of this._managedHashes) {
            hash.context[hash.hash] = clean.cleanHash(hash.context[hash.hash]);
          }
        },
        this._frequency
      );
      this._arrayHandler = this._renderer.scheduler.repeat(
        () => {
          for (const array of this._managedArrays) {
            clean.cleanArray(array.context[array.hash]);
          }
        },
        this._frequency
      );
    } else {
      this._renderer.scheduler.cancel(this._handler);
      this._renderer.scheduler.cancel(this._hashHandler);
      this._renderer.scheduler.cancel(this._arrayHandler);
    }
  }
  addManagedHash(context, hash) {
    this._managedHashes.push({ context, hash });
  }
  addManagedArray(context, hash) {
    this._managedArrays.push({ context, hash });
  }
  prerender() {
    this._now = performance.now();
  }
  addRenderable(renderable, instructionSet) {
    if (!this.enabled)
      return;
    renderable._lastUsed = this._now;
    if (renderable._lastInstructionTick === -1) {
      this._managedRenderables.push(renderable);
      renderable.once("destroyed", this._removeRenderable, this);
    }
    renderable._lastInstructionTick = instructionSet.tick;
  }
  /** Runs the scheduled garbage collection */
  run() {
    const now = performance.now();
    const managedRenderables = this._managedRenderables;
    const renderPipes = this._renderer.renderPipes;
    let offset = 0;
    for (let i = 0; i < managedRenderables.length; i++) {
      const renderable = managedRenderables[i];
      if (renderable === null) {
        offset++;
        continue;
      }
      const renderGroup = renderable.renderGroup ?? renderable.parentRenderGroup;
      const currentIndex = renderGroup?.instructionSet?.tick ?? -1;
      if (renderable._lastInstructionTick !== currentIndex && now - renderable._lastUsed > this.maxUnusedTime) {
        if (!renderable.destroyed) {
          const rp = renderPipes;
          rp[renderable.renderPipeId].destroyRenderable(renderable);
        }
        renderable._lastInstructionTick = -1;
        offset++;
        renderable.off("destroyed", this._removeRenderable, this);
      } else {
        managedRenderables[i - offset] = renderable;
      }
    }
    managedRenderables.length = managedRenderables.length - offset;
  }
  destroy() {
    this.enabled = false;
    this._renderer = null;
    this._managedRenderables.length = 0;
    this._managedHashes.length = 0;
    this._managedArrays.length = 0;
  }
  _removeRenderable(renderable) {
    const index = this._managedRenderables.indexOf(renderable);
    if (index >= 0) {
      renderable.off("destroyed", this._removeRenderable, this);
      this._managedRenderables[index] = null;
    }
  }
};
/** @ignore */
_RenderableGCSystem.extension = {
  type: [
    Extensions.ExtensionType.WebGLSystem,
    Extensions.ExtensionType.WebGPUSystem
  ],
  name: "renderableGC",
  priority: 0
};
/** default options for the renderableGCSystem */
_RenderableGCSystem.defaultOptions = {
  /**
   * If set to true, this will enable the garbage collector on the GPU.
   * @default true
   */
  renderableGCActive: true,
  /**
   * The maximum idle frames before a texture is destroyed by garbage collection.
   * @default 60 * 60
   */
  renderableGCMaxUnusedTime: 6e4,
  /**
   * Frames between two garbage collections.
   * @default 600
   */
  renderableGCFrequency: 3e4
};
let RenderableGCSystem = _RenderableGCSystem;

exports.RenderableGCSystem = RenderableGCSystem;
//# sourceMappingURL=RenderableGCSystem.js.map
