import { TextStyle } from '../text/TextStyle';
import type { TextStyleOptions } from '../text/TextStyle';
import type { BitmapFont } from './BitmapFont';
import type { BitmapTextLayoutData } from './utils/getBitmapTextLayout';
/**
 *
 * The options for installing a new BitmapFont. Once installed the font will be available for use in the BitmapText.
 * It can be accessed by the `fontFamily` property of the TextStyle.
 *
 * Install a new BitmapFont will create the characters provided for the font and store them in the cache.
 * But don't worry, if a character is requested that hasn't been generated yet, it will be created on the fly.
 * @memberof text
 */
export interface BitmapFontInstallOptions {
    /** the name of the font, this will be the name you use in the fontFamily of text style to access this font */
    name?: string;
    /**
     * Characters included in the font set. You can also use ranges.
     * For example, `[['a', 'z'], ['A', 'Z'], "!@#$%^&*()~{}[] "]`.
     * Don't forget to include spaces ' ' in your character set!
     * @default BitmapFont.ALPHANUMERIC
     */
    chars?: string | (string | string[])[];
    /**
     * Render resolution for glyphs.
     * @default 1
     */
    resolution?: number;
    /**
     * Padding between glyphs on texture atlas. Lower values could mean more visual artifacts
     * and bleeding from other glyphs, larger values increase the space required on the texture.
     * @default 4
     */
    padding?: number;
    /**
     * Skip generation of kerning information for the BitmapFont.
     * If true, this could potentially increase the performance, but may impact the rendered text appearance.
     * @default false
     */
    skipKerning?: boolean;
    /** Style options to render with BitmapFont. */
    style?: TextStyle | TextStyleOptions;
}
/**
 * The BitmapFontManager is a helper that exists to install and uninstall fonts
 * into the cache for BitmapText objects.
 * @memberof text
 * @name BitmapFontManager
 * @example
 * import { BitmapFontManager, BitmapText } from 'pixi.js';
 *
 * BitmapFontManager.install({
 *   name: 'TitleFont',
 *   style: {}
 * });
 *
 * const title = new BitmapText({ text: 'This is the title', style: { fontFamily: 'TitleFont' }});
 */
declare class BitmapFontManagerClass {
    /**
     * This character set includes all the letters in the alphabet (both lower- and upper- case).
     * @type {string[][]}
     * @example
     * BitmapFont.from('ExampleFont', style, { chars: BitmapFont.ALPHA })
     */
    readonly ALPHA: (string | string[])[];
    /**
     * This character set includes all decimal digits (from 0 to 9).
     * @type {string[][]}
     * @example
     * BitmapFont.from('ExampleFont', style, { chars: BitmapFont.NUMERIC })
     */
    readonly NUMERIC: string[][];
    /**
     * This character set is the union of `BitmapFont.ALPHA` and `BitmapFont.NUMERIC`.
     * @type {string[][]}
     */
    readonly ALPHANUMERIC: (string | string[])[];
    /**
     * This character set consists of all the ASCII table.
     * @member {string[][]}
     * @see http://www.asciitable.com/
     */
    readonly ASCII: string[][];
    /** Default options for installing a new BitmapFont. */
    defaultOptions: Omit<BitmapFontInstallOptions, 'style'>;
    /**
     * Get a font for the specified text and style.
     * @param text - The text to get the font for
     * @param style - The style to use
     */
    getFont(text: string, style: TextStyle): BitmapFont;
    /**
     * Get the layout of a text for the specified style.
     * @param text - The text to get the layout for
     * @param style - The style to use
     * @param trimEnd - Whether to ignore whitespaces at the end of each line
     */
    getLayout(text: string, style: TextStyle, trimEnd?: boolean): BitmapTextLayoutData;
    /**
     * Measure the text using the specified style.
     * @param text - The text to measure
     * @param style - The style to use
     * @param trimEnd - Whether to ignore whitespaces at the end of each line
     */
    measureText(text: string, style: TextStyle, trimEnd?: boolean): {
        width: number;
        height: number;
        scale: number;
        offsetY: number;
    };
    /**
     * Generates a bitmap-font for the given style and character set
     * @param options - Setup options for font generation.
     * @returns Font generated by style options.
     * @example
     * import { BitmapFontManager, BitmapText } from 'pixi.js';
     *
     * BitmapFontManager.install('TitleFont', {
     *     fontFamily: 'Arial',
     *     fontSize: 12,
     *     strokeThickness: 2,
     *     fill: 'purple',
     * });
     *
     * const title = new BitmapText({ text: 'This is the title', fontFamily: 'TitleFont' });
     */
    install(options: BitmapFontInstallOptions): BitmapFont;
    /** @deprecated since 7.0.0 */
    install(name: string, style?: TextStyle | TextStyleOptions, options?: BitmapFontInstallOptions): BitmapFont;
    /**
     * Uninstalls a bitmap font from the cache.
     * @param {string} name - The name of the bitmap font to uninstall.
     */
    uninstall(name: string): void;
}
export declare const BitmapFontManager: BitmapFontManagerClass;
export {};
