'use strict';

var LoaderParser = require('../../../assets/loader/parsers/LoaderParser.js');
var copySearchParams = require('../../../assets/utils/copySearchParams.js');
var adapter = require('../../../environment/adapter.js');
var Extensions = require('../../../extensions/Extensions.js');
var path = require('../../../utils/path.js');
var BitmapFont = require('../BitmapFont.js');
var bitmapFontTextParser = require('./bitmapFontTextParser.js');
var bitmapFontXMLStringParser = require('./bitmapFontXMLStringParser.js');

"use strict";
const validExtensions = [".xml", ".fnt"];
const bitmapFontCachePlugin = {
  extension: {
    type: Extensions.ExtensionType.CacheParser,
    name: "cacheBitmapFont"
  },
  test: (asset) => asset instanceof BitmapFont.BitmapFont,
  getCacheableAssets(keys, asset) {
    const out = {};
    keys.forEach((key) => {
      out[key] = asset;
      out[`${key}-bitmap`] = asset;
    });
    out[`${asset.fontFamily}-bitmap`] = asset;
    return out;
  }
};
const loadBitmapFont = {
  extension: {
    type: Extensions.ExtensionType.LoadParser,
    priority: LoaderParser.LoaderParserPriority.Normal
  },
  name: "loadBitmapFont",
  test(url) {
    return validExtensions.includes(path.path.extname(url).toLowerCase());
  },
  async testParse(data) {
    return bitmapFontTextParser.bitmapFontTextParser.test(data) || bitmapFontXMLStringParser.bitmapFontXMLStringParser.test(data);
  },
  async parse(asset, data, loader) {
    const bitmapFontData = bitmapFontTextParser.bitmapFontTextParser.test(asset) ? bitmapFontTextParser.bitmapFontTextParser.parse(asset) : bitmapFontXMLStringParser.bitmapFontXMLStringParser.parse(asset);
    const { src } = data;
    const { pages } = bitmapFontData;
    const textureUrls = [];
    const textureOptions = bitmapFontData.distanceField ? {
      scaleMode: "linear",
      alphaMode: "premultiply-alpha-on-upload",
      autoGenerateMipmaps: false,
      resolution: 1
    } : {};
    for (let i = 0; i < pages.length; ++i) {
      const pageFile = pages[i].file;
      let imagePath = path.path.join(path.path.dirname(src), pageFile);
      imagePath = copySearchParams.copySearchParams(imagePath, src);
      textureUrls.push({
        src: imagePath,
        data: textureOptions
      });
    }
    const loadedTextures = await loader.load(textureUrls);
    const textures = textureUrls.map((url) => loadedTextures[url.src]);
    const bitmapFont = new BitmapFont.BitmapFont({
      data: bitmapFontData,
      textures
    }, src);
    return bitmapFont;
  },
  async load(url, _options) {
    const response = await adapter.DOMAdapter.get().fetch(url);
    return await response.text();
  },
  async unload(bitmapFont, _resolvedAsset, loader) {
    await Promise.all(bitmapFont.pages.map((page) => loader.unload(page.texture.source._sourceOrigin)));
    bitmapFont.destroy();
  }
};

exports.bitmapFontCachePlugin = bitmapFontCachePlugin;
exports.loadBitmapFont = loadBitmapFont;
//# sourceMappingURL=loadBitmapFont.js.map
