import { ExtensionType, extensions } from '@pixi/core';

class ResizePlugin {
  static init(options) {
    Object.defineProperty(this, "resizeTo", {
      set(dom) {
        globalThis.removeEventListener("resize", this.queueResize);
        this._resizeTo = dom;
        if (dom) {
          globalThis.addEventListener("resize", this.queueResize);
          this.resize();
        }
      },
      get() {
        return this._resizeTo;
      }
    });
    this.queueResize = () => {
      if (!this._resizeTo) {
        return;
      }
      this.cancelResize();
      this._resizeId = requestAnimationFrame(() => this.resize());
    };
    this.cancelResize = () => {
      if (this._resizeId) {
        cancelAnimationFrame(this._resizeId);
        this._resizeId = null;
      }
    };
    this.resize = () => {
      if (!this._resizeTo) {
        return;
      }
      this.cancelResize();
      let width;
      let height;
      if (this._resizeTo === globalThis.window) {
        width = globalThis.innerWidth;
        height = globalThis.innerHeight;
      } else {
        const { clientWidth, clientHeight } = this._resizeTo;
        width = clientWidth;
        height = clientHeight;
      }
      this.renderer.resize(width, height);
      this.render();
    };
    this._resizeId = null;
    this._resizeTo = null;
    this.resizeTo = options.resizeTo || null;
  }
  static destroy() {
    globalThis.removeEventListener("resize", this.queueResize);
    this.cancelResize();
    this.cancelResize = null;
    this.queueResize = null;
    this.resizeTo = null;
    this.resize = null;
  }
}
ResizePlugin.extension = ExtensionType.Application;
extensions.add(ResizePlugin);

export { ResizePlugin };
//# sourceMappingURL=ResizePlugin.mjs.map
