import type { LoaderParser } from './parsers/LoaderParser';
import type { PromiseAndParser, LoadAsset } from './types';
/**
 * The Loader is responsible for loading all assets, such as images, spritesheets, audio files, etc.
 * It does not do anything clever with URLs - it just loads stuff!
 * Behind the scenes all things are cached using promises. This means it's impossible to load an asset more than once.
 * Through the use of LoaderParsers, the loader can understand how to load any kind of file!
 *
 * It is not intended that this class is created by developers - its part of the Asset class
 * This is the second major system of PixiJS' main Assets class
 * @memberof PIXI
 * @class AssetLoader
 */
export declare class Loader {
    private _parsers;
    /** Cache loading promises that ae currently active */
    promiseCache: Record<string, PromiseAndParser>;
    /** function used for testing */
    reset(): void;
    /**
     * Used internally to generate a promise for the asset to be loaded.
     * @param url - The URL to be loaded
     * @param data - any custom additional information relevant to the asset being loaded
     * @returns - a promise that will resolve to an Asset for example a Texture of a JSON object
     */
    private _getLoadPromiseAndParser;
    /**
     * Loads one or more assets using the parsers added to the Loader.
     * @example
     * // Single asset:
     * const asset = await Loader.load('cool.png');
     * console.log(asset);
     *
     * // Multiple assets:
     * const assets = await Loader.load(['cool.png', 'cooler.png']);
     * console.log(assets);
     * @param assetsToLoadIn - urls that you want to load, or a single one!
     * @param onProgress - a function that gets called when the progress changes
     */
    load(assetsToLoadIn: string | string[] | LoadAsset | LoadAsset[], onProgress?: (progress: number) => void): Promise<{
        [key: string]: any;
    } | any>;
    /**
     * Unloads one or more assets. Any unloaded assets will be destroyed, freeing up memory for your app.
     * The parser that created the asset, will be the one that unloads it.
     * @example
     * // Single asset:
     * const asset = await Loader.load('cool.png');
     *
     * await Loader.unload('cool.png');
     *
     * console.log(asset.destroyed); // true
     * @param assetsToUnloadIn - urls that you want to unload, or a single one!
     */
    unload(assetsToUnloadIn: string | string[] | LoadAsset | LoadAsset[]): Promise<void>;
    /** All loader parsers registered */
    get parsers(): LoaderParser[];
}
