'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var core = require('@pixi/core');
require('../utils/index.js');
var isSingleItem = require('../utils/isSingleItem.js');
var convertToList = require('../utils/convertToList.js');

class Loader {
  constructor() {
    this._parsers = [];
    this.promiseCache = {};
  }
  reset() {
    this.promiseCache = {};
  }
  _getLoadPromiseAndParser(url, data) {
    const result = {
      promise: null,
      parser: null
    };
    result.promise = (async () => {
      let asset = null;
      for (let i = 0; i < this.parsers.length; i++) {
        const parser = this.parsers[i];
        if (parser.load && parser.test?.(url, data, this)) {
          asset = await parser.load(url, data, this);
          result.parser = parser;
          break;
        }
      }
      if (!result.parser) {
        return null;
      }
      for (let i = 0; i < this.parsers.length; i++) {
        const parser = this.parsers[i];
        if (parser.parse) {
          if (parser.parse && await parser.testParse?.(asset, data, this)) {
            asset = await parser.parse(asset, data, this) || asset;
            result.parser = parser;
          }
        }
      }
      return asset;
    })();
    return result;
  }
  async load(assetsToLoadIn, onProgress) {
    let count = 0;
    const assets = {};
    const singleAsset = isSingleItem.isSingleItem(assetsToLoadIn);
    const assetsToLoad = convertToList.convertToList(assetsToLoadIn, (item) => ({
      src: item
    }));
    const total = assetsToLoad.length;
    const promises = assetsToLoad.map(async (asset) => {
      const url = core.utils.path.toAbsolute(asset.src);
      if (!assets[asset.src]) {
        try {
          if (!this.promiseCache[url]) {
            this.promiseCache[url] = this._getLoadPromiseAndParser(url, asset);
          }
          assets[asset.src] = await this.promiseCache[url].promise;
          if (onProgress)
            onProgress(++count / total);
        } catch (e) {
          delete this.promiseCache[url];
          delete assets[asset.src];
          throw new Error(`[Loader.load] Failed to load ${url}.
${e}`);
        }
      }
    });
    await Promise.all(promises);
    return singleAsset ? assets[assetsToLoad[0].src] : assets;
  }
  async unload(assetsToUnloadIn) {
    const assetsToUnload = convertToList.convertToList(assetsToUnloadIn, (item) => ({
      src: item
    }));
    const promises = assetsToUnload.map(async (asset) => {
      const url = core.utils.path.toAbsolute(asset.src);
      const loadPromise = this.promiseCache[url];
      if (loadPromise) {
        const loadedAsset = await loadPromise.promise;
        loadPromise.parser?.unload?.(loadedAsset, asset, this);
        delete this.promiseCache[url];
      }
    });
    await Promise.all(promises);
  }
  get parsers() {
    return this._parsers;
  }
}

exports.Loader = Loader;
//# sourceMappingURL=Loader.js.map
