'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var core = require('@pixi/core');
var LoaderParser = require('../LoaderParser.js');
var WorkerManager = require('../WorkerManager.js');
var checkExtension = require('./utils/checkExtension.js');
var createTexture = require('./utils/createTexture.js');

const validImages = [".jpg", ".png", ".jpeg", ".avif", ".webp"];
async function loadImageBitmap(url) {
  const response = await core.settings.ADAPTER.fetch(url);
  if (!response.ok) {
    throw new Error(`[loadImageBitmap] Failed to fetch ${url}: ${response.status} ${response.statusText}`);
  }
  const imageBlob = await response.blob();
  const imageBitmap = await createImageBitmap(imageBlob);
  return imageBitmap;
}
const loadTextures = {
  extension: {
    type: core.ExtensionType.LoadParser,
    priority: LoaderParser.LoaderParserPriority.High
  },
  config: {
    preferWorkers: true
  },
  test(url) {
    let isValidBase64Suffix = false;
    for (let i = 0; i < validImages.length; i++) {
      if (url.startsWith(`data:image/${validImages[i].slice(1)}`)) {
        isValidBase64Suffix = true;
        break;
      }
    }
    return isValidBase64Suffix || checkExtension.checkExtension(url, validImages);
  },
  async load(url, asset, loader) {
    let src = null;
    if (globalThis.createImageBitmap) {
      if (this.config.preferWorkers && await WorkerManager.WorkerManager.isImageBitmapSupported()) {
        src = await WorkerManager.WorkerManager.loadImageBitmap(url);
      } else {
        src = await loadImageBitmap(url);
      }
    } else {
      src = await new Promise((resolve) => {
        src = new Image();
        src.crossOrigin = "anonymous";
        src.src = url;
        if (src.complete) {
          resolve(src);
        } else {
          src.onload = () => {
            resolve(src);
          };
        }
      });
    }
    const base = new core.BaseTexture(src, {
      resolution: core.utils.getResolutionOfUrl(url),
      ...asset.data
    });
    base.resource.src = url;
    return createTexture.createTexture(base, loader, url);
  },
  unload(texture) {
    texture.destroy(true);
  }
};
core.extensions.add(loadTextures);

exports.loadImageBitmap = loadImageBitmap;
exports.loadTextures = loadTextures;
//# sourceMappingURL=loadTextures.js.map
