import { settings, ExtensionType, BaseTexture, utils, extensions } from '@pixi/core';
import { LoaderParserPriority } from '../LoaderParser.mjs';
import { WorkerManager } from '../WorkerManager.mjs';
import { checkExtension } from './utils/checkExtension.mjs';
import { createTexture } from './utils/createTexture.mjs';

const validImages = [".jpg", ".png", ".jpeg", ".avif", ".webp"];
async function loadImageBitmap(url) {
  const response = await settings.ADAPTER.fetch(url);
  if (!response.ok) {
    throw new Error(`[loadImageBitmap] Failed to fetch ${url}: ${response.status} ${response.statusText}`);
  }
  const imageBlob = await response.blob();
  const imageBitmap = await createImageBitmap(imageBlob);
  return imageBitmap;
}
const loadTextures = {
  extension: {
    type: ExtensionType.LoadParser,
    priority: LoaderParserPriority.High
  },
  config: {
    preferWorkers: true
  },
  test(url) {
    let isValidBase64Suffix = false;
    for (let i = 0; i < validImages.length; i++) {
      if (url.startsWith(`data:image/${validImages[i].slice(1)}`)) {
        isValidBase64Suffix = true;
        break;
      }
    }
    return isValidBase64Suffix || checkExtension(url, validImages);
  },
  async load(url, asset, loader) {
    let src = null;
    if (globalThis.createImageBitmap) {
      if (this.config.preferWorkers && await WorkerManager.isImageBitmapSupported()) {
        src = await WorkerManager.loadImageBitmap(url);
      } else {
        src = await loadImageBitmap(url);
      }
    } else {
      src = await new Promise((resolve) => {
        src = new Image();
        src.crossOrigin = "anonymous";
        src.src = url;
        if (src.complete) {
          resolve(src);
        } else {
          src.onload = () => {
            resolve(src);
          };
        }
      });
    }
    const base = new BaseTexture(src, {
      resolution: utils.getResolutionOfUrl(url),
      ...asset.data
    });
    base.resource.src = url;
    return createTexture(base, loader, url);
  },
  unload(texture) {
    texture.destroy(true);
  }
};
extensions.add(loadTextures);

export { loadImageBitmap, loadTextures };
//# sourceMappingURL=loadTextures.mjs.map
