import { BlobResource } from './BlobResource.mjs';
import { INTERNAL_FORMAT_TO_BYTES_PER_PIXEL } from '../const.mjs';

class CompressedTextureResource extends BlobResource {
  constructor(source, options) {
    super(source, options);
    this.format = options.format;
    this.levels = options.levels || 1;
    this._width = options.width;
    this._height = options.height;
    this._extension = CompressedTextureResource._formatToExtension(this.format);
    if (options.levelBuffers || this.buffer) {
      this._levelBuffers = options.levelBuffers || CompressedTextureResource._createLevelBuffers(source instanceof Uint8Array ? source : this.buffer.uint8View, this.format, this.levels, 4, 4, this.width, this.height);
    }
  }
  upload(renderer, _texture, _glTexture) {
    const gl = renderer.gl;
    const extension = renderer.context.extensions[this._extension];
    if (!extension) {
      throw new Error(`${this._extension} textures are not supported on the current machine`);
    }
    if (!this._levelBuffers) {
      return false;
    }
    for (let i = 0, j = this.levels; i < j; i++) {
      const { levelID, levelWidth, levelHeight, levelBuffer } = this._levelBuffers[i];
      gl.compressedTexImage2D(gl.TEXTURE_2D, levelID, this.format, levelWidth, levelHeight, 0, levelBuffer);
    }
    return true;
  }
  onBlobLoaded() {
    this._levelBuffers = CompressedTextureResource._createLevelBuffers(this.buffer.uint8View, this.format, this.levels, 4, 4, this.width, this.height);
  }
  static _formatToExtension(format) {
    if (format >= 33776 && format <= 33779) {
      return "s3tc";
    } else if (format >= 37488 && format <= 37497) {
      return "etc";
    } else if (format >= 35840 && format <= 35843) {
      return "pvrtc";
    } else if (format >= 36196) {
      return "etc1";
    } else if (format >= 35986 && format <= 34798) {
      return "atc";
    }
    throw new Error("Invalid (compressed) texture format given!");
  }
  static _createLevelBuffers(buffer, format, levels, blockWidth, blockHeight, imageWidth, imageHeight) {
    const buffers = new Array(levels);
    let offset = buffer.byteOffset;
    let levelWidth = imageWidth;
    let levelHeight = imageHeight;
    let alignedLevelWidth = levelWidth + blockWidth - 1 & ~(blockWidth - 1);
    let alignedLevelHeight = levelHeight + blockHeight - 1 & ~(blockHeight - 1);
    let levelSize = alignedLevelWidth * alignedLevelHeight * INTERNAL_FORMAT_TO_BYTES_PER_PIXEL[format];
    for (let i = 0; i < levels; i++) {
      buffers[i] = {
        levelID: i,
        levelWidth: levels > 1 ? levelWidth : alignedLevelWidth,
        levelHeight: levels > 1 ? levelHeight : alignedLevelHeight,
        levelBuffer: new Uint8Array(buffer.buffer, offset, levelSize)
      };
      offset += levelSize;
      levelWidth = levelWidth >> 1 || 1;
      levelHeight = levelHeight >> 1 || 1;
      alignedLevelWidth = levelWidth + blockWidth - 1 & ~(blockWidth - 1);
      alignedLevelHeight = levelHeight + blockHeight - 1 & ~(blockHeight - 1);
      levelSize = alignedLevelWidth * alignedLevelHeight * INTERNAL_FORMAT_TO_BYTES_PER_PIXEL[format];
    }
    return buffers;
  }
}

export { CompressedTextureResource };
//# sourceMappingURL=CompressedTextureResource.mjs.map
