import { EventBoundary } from './EventBoundary';
import { FederatedWheelEvent } from './FederatedWheelEvent';
import type { IRenderableObject, ExtensionMetadata, IPointData } from '@pixi/core';
import type { ICanvas } from '@pixi/settings';
interface Renderer {
    lastObjectRendered: IRenderableObject;
    view: ICanvas;
    resolution: number;
    plugins: Record<string, any>;
}
/**
 * The system for handling UI events.
 * @memberof PIXI
 */
export declare class EventSystem {
    /** @ignore */
    static extension: ExtensionMetadata;
    /**
     * The {@link PIXI.EventBoundary} for the stage.
     *
     * The {@link PIXI.EventBoundary#rootTarget rootTarget} of this root boundary is automatically set to
     * the last rendered object before any event processing is initiated. This means the main scene
     * needs to be rendered atleast once before UI events will start propagating.
     *
     * The root boundary should only be changed during initialization. Otherwise, any state held by the
     * event boundary may be lost (like hovered & pressed DisplayObjects).
     */
    readonly rootBoundary: EventBoundary;
    /** Does the device support touch events https://www.w3.org/TR/touch-events/ */
    readonly supportsTouchEvents: boolean;
    /** Does the device support pointer events https://www.w3.org/Submission/pointer-events/ */
    readonly supportsPointerEvents: boolean;
    /**
     * Should default browser actions automatically be prevented.
     * Does not apply to pointer events for backwards compatibility
     * preventDefault on pointer events stops mouse events from firing
     * Thus, for every pointer event, there will always be either a mouse of touch event alongside it.
     * @default true
     */
    autoPreventDefault: boolean;
    /**
     * Dictionary of how different cursor modes are handled. Strings are handled as CSS cursor
     * values, objects are handled as dictionaries of CSS values for {@code domElement},
     * and functions are called instead of changing the CSS.
     * Default CSS cursor values are provided for 'default' and 'pointer' modes.
     */
    cursorStyles: Record<string, string | ((mode: string) => void) | CSSStyleDeclaration>;
    /**
     * The DOM element to which the root event listeners are bound. This is automatically set to
     * the renderer's {@link PIXI.Renderer#view view}.
     */
    domElement: HTMLElement;
    /** The resolution used to convert between the DOM client space into world space. */
    resolution: number;
    /** The renderer managing this {@link EventSystem}. */
    renderer: Renderer;
    private currentCursor;
    private rootPointerEvent;
    private rootWheelEvent;
    private eventsAdded;
    /**
     * @param {PIXI.Renderer} renderer
     */
    constructor(renderer: Renderer);
    /**
     * Runner init called, view is available at this point.
     * @ignore
     */
    init(): void;
    /**
     * Handle changing resolution.
     * @ignore
     */
    resolutionChange(resolution: number): void;
    /** Destroys all event listeners and detaches the renderer. */
    destroy(): void;
    /**
     * Sets the current cursor mode, handling any callbacks or CSS style changes.
     * @param mode - cursor mode, a key from the cursorStyles dictionary
     */
    setCursor(mode: string): void;
    /**
     * Event handler for pointer down events on {@link PIXI.EventSystem#domElement this.domElement}.
     * @param nativeEvent - The native mouse/pointer/touch event.
     */
    private onPointerDown;
    /**
     * Event handler for pointer move events on on {@link PIXI.EventSystem#domElement this.domElement}.
     * @param nativeEvent - The native mouse/pointer/touch events.
     */
    private onPointerMove;
    /**
     * Event handler for pointer up events on {@link PIXI.EventSystem#domElement this.domElement}.
     * @param nativeEvent - The native mouse/pointer/touch event.
     */
    private onPointerUp;
    /**
     * Event handler for pointer over & out events on {@link PIXI.EventSystem#domElement this.domElement}.
     * @param nativeEvent - The native mouse/pointer/touch event.
     */
    private onPointerOverOut;
    /**
     * Passive handler for `wheel` events on {@link EventSystem.domElement this.domElement}.
     * @param nativeEvent - The native wheel event.
     */
    protected onWheel(nativeEvent: WheelEvent): void;
    /**
     * Sets the {@link PIXI.EventSystem#domElement domElement} and binds event listeners.
     *
     * To deregister the current DOM element without setting a new one, pass {@code null}.
     * @param element - The new DOM element.
     */
    setTargetElement(element: HTMLElement): void;
    /** Register event listeners on {@link PIXI.Renderer#domElement this.domElement}. */
    private addEvents;
    /** Unregister event listeners on {@link PIXI.EventSystem#domElement this.domElement}. */
    private removeEvents;
    /**
     * Maps x and y coords from a DOM object and maps them correctly to the PixiJS view. The
     * resulting value is stored in the point. This takes into account the fact that the DOM
     * element could be scaled and positioned anywhere on the screen.
     * @param  {PIXI.IPointData} point - the point that the result will be stored in
     * @param  {number} x - the x coord of the position to map
     * @param  {number} y - the y coord of the position to map
     */
    mapPositionToPoint(point: IPointData, x: number, y: number): void;
    /**
     * Ensures that the original event object contains all data that a regular pointer event would have
     * @param event - The original event data from a touch or mouse event
     * @returns An array containing a single normalized pointer event, in the case of a pointer
     *  or mouse event, or a multiple normalized pointer events if there are multiple changed touches
     */
    private normalizeToPointerData;
    /**
     * Normalizes the native {@link https://w3c.github.io/uievents/#interface-wheelevent WheelEvent}.
     *
     * The returned {@link PIXI.FederatedWheelEvent} is a shared instance. It will not persist across
     * multiple native wheel events.
     * @param nativeEvent - The native wheel event that occurred on the canvas.
     * @returns A federated wheel event.
     */
    protected normalizeWheelEvent(nativeEvent: WheelEvent): FederatedWheelEvent;
    /**
     * Normalizes the {@code nativeEvent} into a federateed {@code FederatedPointerEvent}.
     * @param event
     * @param nativeEvent
     */
    private bootstrapEvent;
    /**
     * Transfers base & mouse event data from the {@code nativeEvent} to the federated event.
     * @param event
     * @param nativeEvent
     */
    private transferMouseData;
}
export {};
