import { Rectangle, RenderTexture, utils, ExtensionType, extensions } from '@pixi/core';

const TEMP_RECT = new Rectangle();
const BYTES_PER_PIXEL = 4;
const _Extract = class {
  constructor(renderer) {
    this.renderer = renderer;
  }
  async image(target, format, quality) {
    const image = new Image();
    image.src = await this.base64(target, format, quality);
    return image;
  }
  async base64(target, format, quality) {
    const canvas = this.canvas(target);
    if (canvas.toDataURL !== void 0) {
      return canvas.toDataURL(format, quality);
    }
    if (canvas.convertToBlob !== void 0) {
      const blob = await canvas.convertToBlob({ type: format, quality });
      return await new Promise((resolve) => {
        const reader = new FileReader();
        reader.onload = () => resolve(reader.result);
        reader.readAsDataURL(blob);
      });
    }
    throw new Error("Extract.base64() requires ICanvas.toDataURL or ICanvas.convertToBlob to be implemented");
  }
  canvas(target, frame) {
    const renderer = this.renderer;
    let resolution;
    let flipY = false;
    let renderTexture;
    let generated = false;
    if (target) {
      if (target instanceof RenderTexture) {
        renderTexture = target;
      } else {
        renderTexture = this.renderer.generateTexture(target);
        generated = true;
      }
    }
    if (renderTexture) {
      resolution = renderTexture.baseTexture.resolution;
      frame = frame ?? renderTexture.frame;
      flipY = false;
      renderer.renderTexture.bind(renderTexture);
    } else {
      resolution = renderer.resolution;
      if (!frame) {
        frame = TEMP_RECT;
        frame.width = renderer.width;
        frame.height = renderer.height;
      }
      flipY = true;
      renderer.renderTexture.bind(null);
    }
    const width = Math.round(frame.width * resolution);
    const height = Math.round(frame.height * resolution);
    let canvasBuffer = new utils.CanvasRenderTarget(width, height, 1);
    const webglPixels = new Uint8Array(BYTES_PER_PIXEL * width * height);
    const gl = renderer.gl;
    gl.readPixels(Math.round(frame.x * resolution), Math.round(frame.y * resolution), width, height, gl.RGBA, gl.UNSIGNED_BYTE, webglPixels);
    const canvasData = canvasBuffer.context.getImageData(0, 0, width, height);
    _Extract.arrayPostDivide(webglPixels, canvasData.data);
    canvasBuffer.context.putImageData(canvasData, 0, 0);
    if (flipY) {
      const target2 = new utils.CanvasRenderTarget(canvasBuffer.width, canvasBuffer.height, 1);
      target2.context.scale(1, -1);
      target2.context.drawImage(canvasBuffer.canvas, 0, -height);
      canvasBuffer.destroy();
      canvasBuffer = target2;
    }
    if (generated) {
      renderTexture.destroy(true);
    }
    return canvasBuffer.canvas;
  }
  pixels(target, frame) {
    const renderer = this.renderer;
    let resolution;
    let renderTexture;
    let generated = false;
    if (target) {
      if (target instanceof RenderTexture) {
        renderTexture = target;
      } else {
        renderTexture = this.renderer.generateTexture(target);
        generated = true;
      }
    }
    if (renderTexture) {
      resolution = renderTexture.baseTexture.resolution;
      frame = frame ?? renderTexture.frame;
      renderer.renderTexture.bind(renderTexture);
    } else {
      resolution = renderer.resolution;
      if (!frame) {
        frame = TEMP_RECT;
        frame.width = renderer.width;
        frame.height = renderer.height;
      }
      renderer.renderTexture.bind(null);
    }
    const width = Math.round(frame.width * resolution);
    const height = Math.round(frame.height * resolution);
    const webglPixels = new Uint8Array(BYTES_PER_PIXEL * width * height);
    const gl = renderer.gl;
    gl.readPixels(Math.round(frame.x * resolution), Math.round(frame.y * resolution), width, height, gl.RGBA, gl.UNSIGNED_BYTE, webglPixels);
    if (generated) {
      renderTexture.destroy(true);
    }
    _Extract.arrayPostDivide(webglPixels, webglPixels);
    return webglPixels;
  }
  destroy() {
    this.renderer = null;
  }
  static arrayPostDivide(pixels, out) {
    for (let i = 0; i < pixels.length; i += 4) {
      const alpha = out[i + 3] = pixels[i + 3];
      if (alpha !== 0) {
        out[i] = Math.round(Math.min(pixels[i] * 255 / alpha, 255));
        out[i + 1] = Math.round(Math.min(pixels[i + 1] * 255 / alpha, 255));
        out[i + 2] = Math.round(Math.min(pixels[i + 2] * 255 / alpha, 255));
      } else {
        out[i] = pixels[i];
        out[i + 1] = pixels[i + 1];
        out[i + 2] = pixels[i + 2];
      }
    }
  }
};
let Extract = _Extract;
Extract.extension = {
  name: "extract",
  type: ExtensionType.RendererSystem
};
extensions.add(Extract);

export { Extract };
//# sourceMappingURL=Extract.mjs.map
