import { Shader, Matrix, Program, TextureMatrix, utils } from '@pixi/core';
import fragment from './shader/mesh.mjs';
import vertex from './shader/mesh2.mjs';

class MeshMaterial extends Shader {
  constructor(uSampler, options) {
    const uniforms = {
      uSampler,
      alpha: 1,
      uTextureMatrix: Matrix.IDENTITY,
      uColor: new Float32Array([1, 1, 1, 1])
    };
    options = Object.assign({
      tint: 16777215,
      alpha: 1,
      pluginName: "batch"
    }, options);
    if (options.uniforms) {
      Object.assign(uniforms, options.uniforms);
    }
    super(options.program || Program.from(vertex, fragment), uniforms);
    this._colorDirty = false;
    this.uvMatrix = new TextureMatrix(uSampler);
    this.batchable = options.program === void 0;
    this.pluginName = options.pluginName;
    this.tint = options.tint;
    this.alpha = options.alpha;
  }
  get texture() {
    return this.uniforms.uSampler;
  }
  set texture(value) {
    if (this.uniforms.uSampler !== value) {
      if (!this.uniforms.uSampler.baseTexture.alphaMode !== !value.baseTexture.alphaMode) {
        this._colorDirty = true;
      }
      this.uniforms.uSampler = value;
      this.uvMatrix.texture = value;
    }
  }
  set alpha(value) {
    if (value === this._alpha)
      return;
    this._alpha = value;
    this._colorDirty = true;
  }
  get alpha() {
    return this._alpha;
  }
  set tint(value) {
    if (value === this._tint)
      return;
    this._tint = value;
    this._tintRGB = (value >> 16) + (value & 65280) + ((value & 255) << 16);
    this._colorDirty = true;
  }
  get tint() {
    return this._tint;
  }
  update() {
    if (this._colorDirty) {
      this._colorDirty = false;
      const baseTexture = this.texture.baseTexture;
      utils.premultiplyTintToRgba(this._tint, this._alpha, this.uniforms.uColor, baseTexture.alphaMode);
    }
    if (this.uvMatrix.update()) {
      this.uniforms.uTextureMatrix = this.uvMatrix.mapCoord;
    }
  }
}

export { MeshMaterial };
//# sourceMappingURL=MeshMaterial.mjs.map
