/*!
 * @pixi/text-bitmap - v5.0.0-alpha.2
 * Compiled Sat, 17 Mar 2018 17:48:48 UTC
 *
 * @pixi/text-bitmap is licensed under the MIT License.
 * http://www.opensource.org/licenses/mit-license
 */
'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var core = require('@pixi/core');
var display = require('@pixi/display');
var math = require('@pixi/math');
var settings = require('@pixi/settings');
var sprite = require('@pixi/sprite');
var utils = require('@pixi/utils');
var loaders = require('@pixi/loaders');

/**
 * A BitmapText object will create a line or multiple lines of text using bitmap font. To
 * split a line you can use '\n', '\r' or '\r\n' in your string. You can generate the fnt files using:
 *
 * A BitmapText can only be created when the font is loaded
 *
 * ```js
 * // in this case the font is in a file called 'desyrel.fnt'
 * let bitmapText = new PIXI.BitmapText("text using a fancy font!", {font: "35px Desyrel", align: "right"});
 * ```
 *
 * http://www.angelcode.com/products/bmfont/ for windows or
 * http://www.bmglyph.com/ for mac.
 *
 * @class
 * @extends PIXI.Container
 * @memberof PIXI
 */
var BitmapText = (function (Container$$1) {
    function BitmapText(text, style)
    {
        var this$1 = this;
        if ( style === void 0 ) style = {};

        Container$$1.call(this);

        /**
         * Private tracker for the width of the overall text
         *
         * @member {number}
         * @private
         */
        this._textWidth = 0;

        /**
         * Private tracker for the height of the overall text
         *
         * @member {number}
         * @private
         */
        this._textHeight = 0;

        /**
         * Private tracker for the letter sprite pool.
         *
         * @member {PIXI.Sprite[]}
         * @private
         */
        this._glyphs = [];

        /**
         * Private tracker for the current style.
         *
         * @member {object}
         * @private
         */
        this._font = {
            tint: style.tint !== undefined ? style.tint : 0xFFFFFF,
            align: style.align || 'left',
            name: null,
            size: 0,
        };

        /**
         * Private tracker for the current font.
         *
         * @member {object}
         * @private
         */
        this.font = style.font; // run font setter

        /**
         * Private tracker for the current text.
         *
         * @member {string}
         * @private
         */
        this._text = text;

        /**
         * The max width of this bitmap text in pixels. If the text provided is longer than the
         * value provided, line breaks will be automatically inserted in the last whitespace.
         * Disable by setting value to 0
         *
         * @member {number}
         * @private
         */
        this._maxWidth = 0;

        /**
         * The max line height. This is useful when trying to use the total height of the Text,
         * ie: when trying to vertically align.
         *
         * @member {number}
         * @private
         */
        this._maxLineHeight = 0;

        /**
         * Text anchor. read-only
         *
         * @member {PIXI.ObservablePoint}
         * @private
         */
        this._anchor = new math.ObservablePoint(function () { this$1.dirty = true; }, this, 0, 0);

        /**
         * The dirty state of this object.
         *
         * @member {boolean}
         */
        this.dirty = false;

        this.updateText();
    }

    if ( Container$$1 ) BitmapText.__proto__ = Container$$1;
    BitmapText.prototype = Object.create( Container$$1 && Container$$1.prototype );
    BitmapText.prototype.constructor = BitmapText;

    var prototypeAccessors = { tint: { configurable: true },align: { configurable: true },anchor: { configurable: true },font: { configurable: true },text: { configurable: true },maxWidth: { configurable: true },maxLineHeight: { configurable: true },textWidth: { configurable: true },textHeight: { configurable: true } };

    /**
     * Renders text and updates it when needed
     *
     * @private
     */
    BitmapText.prototype.updateText = function updateText ()
    {
        var this$1 = this;

        var data = BitmapText.fonts[this._font.name];
        var scale = this._font.size / data.size;
        var pos = new math.Point();
        var chars = [];
        var lineWidths = [];

        var prevCharCode = null;
        var lastLineWidth = 0;
        var maxLineWidth = 0;
        var line = 0;
        var lastSpace = -1;
        var lastSpaceWidth = 0;
        var spacesRemoved = 0;
        var maxLineHeight = 0;

        for (var i = 0; i < this.text.length; i++)
        {
            var charCode = this$1.text.charCodeAt(i);

            if (/(\s)/.test(this$1.text.charAt(i)))
            {
                lastSpace = i;
                lastSpaceWidth = lastLineWidth;
            }

            if (/(?:\r\n|\r|\n)/.test(this$1.text.charAt(i)))
            {
                lineWidths.push(lastLineWidth);
                maxLineWidth = Math.max(maxLineWidth, lastLineWidth);
                line++;

                pos.x = 0;
                pos.y += data.lineHeight;
                prevCharCode = null;
                continue;
            }

            if (lastSpace !== -1 && this$1._maxWidth > 0 && pos.x * scale > this$1._maxWidth)
            {
                utils.removeItems(chars, lastSpace - spacesRemoved, i - lastSpace);
                i = lastSpace;
                lastSpace = -1;
                ++spacesRemoved;

                lineWidths.push(lastSpaceWidth);
                maxLineWidth = Math.max(maxLineWidth, lastSpaceWidth);
                line++;

                pos.x = 0;
                pos.y += data.lineHeight;
                prevCharCode = null;
                continue;
            }

            var charData = data.chars[charCode];

            if (!charData)
            {
                continue;
            }

            if (prevCharCode && charData.kerning[prevCharCode])
            {
                pos.x += charData.kerning[prevCharCode];
            }

            chars.push({
                texture: charData.texture,
                line: line,
                charCode: charCode,
                position: new math.Point(pos.x + charData.xOffset, pos.y + charData.yOffset),
            });
            lastLineWidth = pos.x + (charData.texture.width + charData.xOffset);
            pos.x += charData.xAdvance;
            maxLineHeight = Math.max(maxLineHeight, (charData.yOffset + charData.texture.height));
            prevCharCode = charCode;
        }

        lineWidths.push(lastLineWidth);
        maxLineWidth = Math.max(maxLineWidth, lastLineWidth);

        var lineAlignOffsets = [];

        for (var i$1 = 0; i$1 <= line; i$1++)
        {
            var alignOffset = 0;

            if (this$1._font.align === 'right')
            {
                alignOffset = maxLineWidth - lineWidths[i$1];
            }
            else if (this$1._font.align === 'center')
            {
                alignOffset = (maxLineWidth - lineWidths[i$1]) / 2;
            }

            lineAlignOffsets.push(alignOffset);
        }

        var lenChars = chars.length;
        var tint = this.tint;

        for (var i$2 = 0; i$2 < lenChars; i$2++)
        {
            var c = this$1._glyphs[i$2]; // get the next glyph sprite

            if (c)
            {
                c.texture = chars[i$2].texture;
            }
            else
            {
                c = new sprite.Sprite(chars[i$2].texture);
                this$1._glyphs.push(c);
            }

            c.position.x = (chars[i$2].position.x + lineAlignOffsets[chars[i$2].line]) * scale;
            c.position.y = chars[i$2].position.y * scale;
            c.scale.x = c.scale.y = scale;
            c.tint = tint;

            if (!c.parent)
            {
                this$1.addChild(c);
            }
        }

        // remove unnecessary children.
        for (var i$3 = lenChars; i$3 < this._glyphs.length; ++i$3)
        {
            this$1.removeChild(this$1._glyphs[i$3]);
        }

        this._textWidth = maxLineWidth * scale;
        this._textHeight = (pos.y + data.lineHeight) * scale;

        // apply anchor
        if (this.anchor.x !== 0 || this.anchor.y !== 0)
        {
            for (var i$4 = 0; i$4 < lenChars; i$4++)
            {
                this$1._glyphs[i$4].x -= this$1._textWidth * this$1.anchor.x;
                this$1._glyphs[i$4].y -= this$1._textHeight * this$1.anchor.y;
            }
        }
        this._maxLineHeight = maxLineHeight * scale;
    };

    /**
     * Updates the transform of this object
     *
     * @private
     */
    BitmapText.prototype.updateTransform = function updateTransform ()
    {
        this.validate();
        this.containerUpdateTransform();
    };

    /**
     * Validates text before calling parent's getLocalBounds
     *
     * @return {PIXI.Rectangle} The rectangular bounding area
     */
    BitmapText.prototype.getLocalBounds = function getLocalBounds ()
    {
        this.validate();

        return Container$$1.prototype.getLocalBounds.call(this);
    };

    /**
     * Updates text when needed
     *
     * @private
     */
    BitmapText.prototype.validate = function validate ()
    {
        if (this.dirty)
        {
            this.updateText();
            this.dirty = false;
        }
    };

    /**
     * The tint of the BitmapText object
     *
     * @member {number}
     */
    prototypeAccessors.tint.get = function ()
    {
        return this._font.tint;
    };

    prototypeAccessors.tint.set = function (value) // eslint-disable-line require-jsdoc
    {
        this._font.tint = (typeof value === 'number' && value >= 0) ? value : 0xFFFFFF;

        this.dirty = true;
    };

    /**
     * The alignment of the BitmapText object
     *
     * @member {string}
     * @default 'left'
     */
    prototypeAccessors.align.get = function ()
    {
        return this._font.align;
    };

    prototypeAccessors.align.set = function (value) // eslint-disable-line require-jsdoc
    {
        this._font.align = value || 'left';

        this.dirty = true;
    };

    /**
     * The anchor sets the origin point of the text.
     * The default is 0,0 this means the text's origin is the top left
     * Setting the anchor to 0.5,0.5 means the text's origin is centered
     * Setting the anchor to 1,1 would mean the text's origin point will be the bottom right corner
     *
     * @member {PIXI.Point | number}
     */
    prototypeAccessors.anchor.get = function ()
    {
        return this._anchor;
    };

    prototypeAccessors.anchor.set = function (value) // eslint-disable-line require-jsdoc
    {
        if (typeof value === 'number')
        {
            this._anchor.set(value);
        }
        else
        {
            this._anchor.copyFrom(value);
        }
    };

    /**
     * The font descriptor of the BitmapText object
     *
     * @member {string|object}
     */
    prototypeAccessors.font.get = function ()
    {
        return this._font;
    };

    prototypeAccessors.font.set = function (value) // eslint-disable-line require-jsdoc
    {
        if (!value)
        {
            return;
        }

        if (typeof value === 'string')
        {
            value = value.split(' ');

            this._font.name = value.length === 1 ? value[0] : value.slice(1).join(' ');
            this._font.size = value.length >= 2 ? parseInt(value[0], 10) : BitmapText.fonts[this._font.name].size;
        }
        else
        {
            this._font.name = value.name;
            this._font.size = typeof value.size === 'number' ? value.size : parseInt(value.size, 10);
        }

        this.dirty = true;
    };

    /**
     * The text of the BitmapText object
     *
     * @member {string}
     */
    prototypeAccessors.text.get = function ()
    {
        return this._text;
    };

    prototypeAccessors.text.set = function (value) // eslint-disable-line require-jsdoc
    {
        value = value.toString() || ' ';
        if (this._text === value)
        {
            return;
        }
        this._text = value;
        this.dirty = true;
    };

    /**
     * The max width of this bitmap text in pixels. If the text provided is longer than the
     * value provided, line breaks will be automatically inserted in the last whitespace.
     * Disable by setting value to 0
     *
     * @member {number}
     */
    prototypeAccessors.maxWidth.get = function ()
    {
        return this._maxWidth;
    };

    prototypeAccessors.maxWidth.set = function (value) // eslint-disable-line require-jsdoc
    {
        if (this._maxWidth === value)
        {
            return;
        }
        this._maxWidth = value;
        this.dirty = true;
    };

    /**
     * The max line height. This is useful when trying to use the total height of the Text,
     * ie: when trying to vertically align.
     *
     * @member {number}
     * @readonly
     */
    prototypeAccessors.maxLineHeight.get = function ()
    {
        this.validate();

        return this._maxLineHeight;
    };

    /**
     * The width of the overall text, different from fontSize,
     * which is defined in the style object
     *
     * @member {number}
     * @readonly
     */
    prototypeAccessors.textWidth.get = function ()
    {
        this.validate();

        return this._textWidth;
    };

    /**
     * The height of the overall text, different from fontSize,
     * which is defined in the style object
     *
     * @member {number}
     * @readonly
     */
    prototypeAccessors.textHeight.get = function ()
    {
        this.validate();

        return this._textHeight;
    };

    /**
     * Register a bitmap font with data and a texture.
     *
     * @static
     * @param {XMLDocument} xml - The XML document data.
     * @param {PIXI.Texture} texture - Texture with all symbols.
     * @return {Object} Result font object with font, size, lineHeight and char fields.
     */
    BitmapText.registerFont = function registerFont (xml, texture)
    {
        var data = {};
        var info = xml.getElementsByTagName('info')[0];
        var common = xml.getElementsByTagName('common')[0];
        var fileName = xml.getElementsByTagName('page')[0].getAttribute('file');
        var res = utils.getResolutionOfUrl(fileName, settings.settings.RESOLUTION);

        data.font = info.getAttribute('face');
        data.size = parseInt(info.getAttribute('size'), 10);
        data.lineHeight = parseInt(common.getAttribute('lineHeight'), 10) / res;
        data.chars = {};

        // parse letters
        var letters = xml.getElementsByTagName('char');

        for (var i = 0; i < letters.length; i++)
        {
            var letter = letters[i];
            var charCode = parseInt(letter.getAttribute('id'), 10);

            var textureRect = new math.Rectangle(
                (parseInt(letter.getAttribute('x'), 10) / res) + (texture.frame.x / res),
                (parseInt(letter.getAttribute('y'), 10) / res) + (texture.frame.y / res),
                parseInt(letter.getAttribute('width'), 10) / res,
                parseInt(letter.getAttribute('height'), 10) / res
            );

            data.chars[charCode] = {
                xOffset: parseInt(letter.getAttribute('xoffset'), 10) / res,
                yOffset: parseInt(letter.getAttribute('yoffset'), 10) / res,
                xAdvance: parseInt(letter.getAttribute('xadvance'), 10) / res,
                kerning: {},
                texture: new core.Texture(texture.baseTexture, textureRect),

            };
        }

        // parse kernings
        var kernings = xml.getElementsByTagName('kerning');

        for (var i$1 = 0; i$1 < kernings.length; i$1++)
        {
            var kerning = kernings[i$1];
            var first = parseInt(kerning.getAttribute('first'), 10) / res;
            var second = parseInt(kerning.getAttribute('second'), 10) / res;
            var amount = parseInt(kerning.getAttribute('amount'), 10) / res;

            if (data.chars[second])
            {
                data.chars[second].kerning[first] = amount;
            }
        }

        // I'm leaving this as a temporary fix so we can test the bitmap fonts in v3
        // but it's very likely to change
        BitmapText.fonts[data.font] = data;

        return data;
    };

    Object.defineProperties( BitmapText.prototype, prototypeAccessors );

    return BitmapText;
}(display.Container));

BitmapText.fonts = {};

/**
 * {@link PIXI.Loader Loader} middleware for loading
 * bitmap-based fonts suitable for using with {@link PIXI.BitmapText}.
 * @class
 * @memberof PIXI
 * @extends PIXI.Loader~LoaderPlugin
 */
var BitmapFontLoader = function BitmapFontLoader () {};

BitmapFontLoader.parse = function parse (resource, texture)
{
    resource.bitmapFont = BitmapText.registerFont(resource.data, texture);
};

/**
 * Called when the plugin is installed.
 *
 * @see PIXI.Loader.registerPlugin
 */
BitmapFontLoader.add = function add ()
{
    loaders.LoaderResource.setExtensionXhrType('fnt', loaders.LoaderResource.XHR_RESPONSE_TYPE.DOCUMENT);
};

/**
 * Replacement for NodeJS's path.dirname
 * @private
 * @param {string} url Path to get directory for
 */
BitmapFontLoader.dirname = function dirname (url)
{
    var dir = url
        .replace(/\/$/, '') // replace trailing slash
        .replace(/\/[^\/]*$/, ''); // remove everything after the last

    // File request is relative, use current directory
    if (dir === url)
    {
        return '.';
    }
    // Started with a slash
    else if (dir === '')
    {
        return '/';
    }

    return dir;
};

/**
 * Called after a resource is loaded.
 * @see PIXI.Loader~loaderMiddleware
 * @param {PIXI.LoaderResource} resource
 * @param {function} next
 */
BitmapFontLoader.use = function use (resource, next)
{
    // skip if no data or not xml data
    if (!resource.data || resource.type !== loaders.LoaderResource.TYPE.XML)
    {
        next();

        return;
    }

    // skip if not bitmap font data, using some silly duck-typing
    if (resource.data.getElementsByTagName('page').length === 0
        || resource.data.getElementsByTagName('info').length === 0
        || resource.data.getElementsByTagName('info')[0].getAttribute('face') === null
    )
    {
        next();

        return;
    }

    var xmlUrl = !resource.isDataUrl ? BitmapFontLoader.dirname(resource.url) : '';

    if (resource.isDataUrl)
    {
        if (xmlUrl === '.')
        {
            xmlUrl = '';
        }

        if (this.baseUrl && xmlUrl)
        {
            // if baseurl has a trailing slash then add one to xmlUrl so the replace works below
            if (this.baseUrl.charAt(this.baseUrl.length - 1) === '/')
            {
                xmlUrl += '/';
            }
        }
    }

    // remove baseUrl from xmlUrl
    xmlUrl = xmlUrl.replace(this.baseUrl, '');

    // if there is an xmlUrl now, it needs a trailing slash. Ensure that it does if the string isn't empty.
    if (xmlUrl && xmlUrl.charAt(xmlUrl.length - 1) !== '/')
    {
        xmlUrl += '/';
    }

    var textureUrl = xmlUrl + resource.data.getElementsByTagName('page')[0].getAttribute('file');

    if (utils.TextureCache[textureUrl])
    {
        // reuse existing texture
        BitmapFontLoader.parse(resource, utils.TextureCache[textureUrl]);
        next();
    }
    else
    {
        var loadOptions = {
            crossOrigin: resource.crossOrigin,
            loadType: loaders.LoaderResource.LOAD_TYPE.IMAGE,
            metadata: resource.metadata.imageMetadata,
            parentResource: resource,
        };

        // load the texture for the font
        this.add(((resource.name) + "_image"), textureUrl, loadOptions, function (res) {
            BitmapFontLoader.parse(resource, res.texture);
            next();
        });
    }
};

exports.BitmapText = BitmapText;
exports.BitmapFontLoader = BitmapFontLoader;
//# sourceMappingURL=text-bitmap.js.map
