/*!
 * @pixi/text - v5.0.0-alpha.2
 * Compiled Sat, 17 Mar 2018 17:48:48 UTC
 *
 * @pixi/text is licensed under the MIT License.
 * http://www.opensource.org/licenses/mit-license
 */
'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var sprite = require('@pixi/sprite');
var core = require('@pixi/core');
var settings = require('@pixi/settings');
var math = require('@pixi/math');
var utils = require('@pixi/utils');

/**
 * Constants that define the type of gradient on text.
 *
 * @static
 * @constant
 * @name TEXT_GRADIENT
 * @memberof PIXI
 * @type {object}
 * @property {number} LINEAR_VERTICAL Vertical gradient
 * @property {number} LINEAR_HORIZONTAL Linear gradient
 */
var TEXT_GRADIENT = {
    LINEAR_VERTICAL: 0,
    LINEAR_HORIZONTAL: 1,
};

// disabling eslint for now, going to rewrite this in v5
/* eslint-disable */

var defaultStyle = {
    align: 'left',
    breakWords: false,
    dropShadow: false,
    dropShadowAlpha: 1,
    dropShadowAngle: Math.PI / 6,
    dropShadowBlur: 0,
    dropShadowColor: 'black',
    dropShadowDistance: 5,
    fill: 'black',
    fillGradientType: TEXT_GRADIENT.LINEAR_VERTICAL,
    fillGradientStops: [],
    fontFamily: 'Arial',
    fontSize: 26,
    fontStyle: 'normal',
    fontVariant: 'normal',
    fontWeight: 'normal',
    letterSpacing: 0,
    lineHeight: 0,
    lineJoin: 'miter',
    miterLimit: 10,
    padding: 0,
    stroke: 'black',
    strokeThickness: 0,
    textBaseline: 'alphabetic',
    trim: false,
    wordWrap: false,
    wordWrapWidth: 100,
    leading: 0,
};

/**
 * A TextStyle Object decorates a Text Object. It can be shared between
 * multiple Text objects. Changing the style will update all text objects using it.
 *
 * @class
 * @memberof PIXI
 */
var TextStyle = function TextStyle(style)
{
    this.styleID = 0;

    this.reset();

    deepCopyProperties(this, style, style);
};

var prototypeAccessors = { align: { configurable: true },breakWords: { configurable: true },dropShadow: { configurable: true },dropShadowAlpha: { configurable: true },dropShadowAngle: { configurable: true },dropShadowBlur: { configurable: true },dropShadowColor: { configurable: true },dropShadowDistance: { configurable: true },fill: { configurable: true },fillGradientType: { configurable: true },fillGradientStops: { configurable: true },fontFamily: { configurable: true },fontSize: { configurable: true },fontStyle: { configurable: true },fontVariant: { configurable: true },fontWeight: { configurable: true },letterSpacing: { configurable: true },lineHeight: { configurable: true },leading: { configurable: true },lineJoin: { configurable: true },miterLimit: { configurable: true },padding: { configurable: true },stroke: { configurable: true },strokeThickness: { configurable: true },textBaseline: { configurable: true },trim: { configurable: true },wordWrap: { configurable: true },wordWrapWidth: { configurable: true } };

/**
 * Creates a new TextStyle object with the same values as this one.
 * Note that the only the properties of the object are cloned.
 *
 * @return {PIXI.TextStyle} New cloned TextStyle object
 */
TextStyle.prototype.clone = function clone ()
{
    var clonedProperties = {};

    deepCopyProperties(clonedProperties, this, defaultStyle);

    return new TextStyle(clonedProperties);
};

/**
 * Resets all properties to the defaults specified in TextStyle.prototype._default
 */
TextStyle.prototype.reset = function reset ()
{
    deepCopyProperties(this, defaultStyle, defaultStyle);
};

/**
 * Alignment for multiline text ('left', 'center' or 'right'), does not affect single line text
 *
 * @member {string}
 */
prototypeAccessors.align.get = function ()
{
    return this._align;
};
prototypeAccessors.align.set = function (align) // eslint-disable-line require-jsdoc
{
    if (this._align !== align)
    {
        this._align = align;
        this.styleID++;
    }
};

/**
 * Indicates if lines can be wrapped within words, it needs wordWrap to be set to true
 *
 * @member {boolean}
 */
prototypeAccessors.breakWords.get = function ()
{
    return this._breakWords;
};
prototypeAccessors.breakWords.set = function (breakWords) // eslint-disable-line require-jsdoc
{
    if (this._breakWords !== breakWords)
    {
        this._breakWords = breakWords;
        this.styleID++;
    }
};

/**
 * Set a drop shadow for the text
 *
 * @member {boolean}
 */
prototypeAccessors.dropShadow.get = function ()
{
    return this._dropShadow;
};
prototypeAccessors.dropShadow.set = function (dropShadow) // eslint-disable-line require-jsdoc
{
    if (this._dropShadow !== dropShadow)
    {
        this._dropShadow = dropShadow;
        this.styleID++;
    }
};

/**
 * Set alpha for the drop shadow
 *
 * @member {number}
 */
prototypeAccessors.dropShadowAlpha.get = function ()
{
    return this._dropShadowAlpha;
};
prototypeAccessors.dropShadowAlpha.set = function (dropShadowAlpha) // eslint-disable-line require-jsdoc
{
    if (this._dropShadowAlpha !== dropShadowAlpha)
    {
        this._dropShadowAlpha = dropShadowAlpha;
        this.styleID++;
    }
};

/**
 * Set a angle of the drop shadow
 *
 * @member {number}
 */
prototypeAccessors.dropShadowAngle.get = function ()
{
    return this._dropShadowAngle;
};
prototypeAccessors.dropShadowAngle.set = function (dropShadowAngle) // eslint-disable-line require-jsdoc
{
    if (this._dropShadowAngle !== dropShadowAngle)
    {
        this._dropShadowAngle = dropShadowAngle;
        this.styleID++;
    }
};

/**
 * Set a shadow blur radius
 *
 * @member {number}
 */
prototypeAccessors.dropShadowBlur.get = function ()
{
    return this._dropShadowBlur;
};
prototypeAccessors.dropShadowBlur.set = function (dropShadowBlur) // eslint-disable-line require-jsdoc
{
    if (this._dropShadowBlur !== dropShadowBlur)
    {
        this._dropShadowBlur = dropShadowBlur;
        this.styleID++;
    }
};

/**
 * A fill style to be used on the dropshadow e.g 'red', '#00FF00'
 *
 * @member {string|number}
 */
prototypeAccessors.dropShadowColor.get = function ()
{
    return this._dropShadowColor;
};
prototypeAccessors.dropShadowColor.set = function (dropShadowColor) // eslint-disable-line require-jsdoc
{
    var outputColor = getColor(dropShadowColor);
    if (this._dropShadowColor !== outputColor)
    {
        this._dropShadowColor = outputColor;
        this.styleID++;
    }
};

/**
 * Set a distance of the drop shadow
 *
 * @member {number}
 */
prototypeAccessors.dropShadowDistance.get = function ()
{
    return this._dropShadowDistance;
};
prototypeAccessors.dropShadowDistance.set = function (dropShadowDistance) // eslint-disable-line require-jsdoc
{
    if (this._dropShadowDistance !== dropShadowDistance)
    {
        this._dropShadowDistance = dropShadowDistance;
        this.styleID++;
    }
};

/**
 * A canvas fillstyle that will be used on the text e.g 'red', '#00FF00'.
 * Can be an array to create a gradient eg ['#000000','#FFFFFF']
 * {@link https://developer.mozilla.org/en-US/docs/Web/API/CanvasRenderingContext2D/fillStyle|MDN}
 *
 * @member {string|string[]|number|number[]|CanvasGradient|CanvasPattern}
 */
prototypeAccessors.fill.get = function ()
{
    return this._fill;
};
prototypeAccessors.fill.set = function (fill) // eslint-disable-line require-jsdoc
{
    var outputColor = getColor(fill);
    if (this._fill !== outputColor)
    {
        this._fill = outputColor;
        this.styleID++;
    }
};

/**
 * If fill is an array of colours to create a gradient, this can change the type/direction of the gradient.
 * See {@link PIXI.TEXT_GRADIENT}
 *
 * @member {number}
 */
prototypeAccessors.fillGradientType.get = function ()
{
    return this._fillGradientType;
};
prototypeAccessors.fillGradientType.set = function (fillGradientType) // eslint-disable-line require-jsdoc
{
    if (this._fillGradientType !== fillGradientType)
    {
        this._fillGradientType = fillGradientType;
        this.styleID++;
    }
};

/**
 * If fill is an array of colours to create a gradient, this array can set the stop points
 * (numbers between 0 and 1) for the color, overriding the default behaviour of evenly spacing them.
 *
 * @member {number[]}
 */
prototypeAccessors.fillGradientStops.get = function ()
{
    return this._fillGradientStops;
};
prototypeAccessors.fillGradientStops.set = function (fillGradientStops) // eslint-disable-line require-jsdoc
{
    if (!areArraysEqual(this._fillGradientStops,fillGradientStops))
    {
        this._fillGradientStops = fillGradientStops;
        this.styleID++;
    }
};

/**
 * The font family
 *
 * @member {string|string[]}
 */
prototypeAccessors.fontFamily.get = function ()
{
    return this._fontFamily;
};
prototypeAccessors.fontFamily.set = function (fontFamily) // eslint-disable-line require-jsdoc
{
    if (this.fontFamily !== fontFamily)
    {
        this._fontFamily = fontFamily;
        this.styleID++;
    }
};

/**
 * The font size
 * (as a number it converts to px, but as a string, equivalents are '26px','20pt','160%' or '1.6em')
 *
 * @member {number|string}
 */
prototypeAccessors.fontSize.get = function ()
{
    return this._fontSize;
};
prototypeAccessors.fontSize.set = function (fontSize) // eslint-disable-line require-jsdoc
{
    if (this._fontSize !== fontSize)
    {
        this._fontSize = fontSize;
        this.styleID++;
    }
};

/**
 * The font style
 * ('normal', 'italic' or 'oblique')
 *
 * @member {string}
 */
prototypeAccessors.fontStyle.get = function ()
{
    return this._fontStyle;
};
prototypeAccessors.fontStyle.set = function (fontStyle) // eslint-disable-line require-jsdoc
{
    if (this._fontStyle !== fontStyle)
    {
        this._fontStyle = fontStyle;
        this.styleID++;
    }
};

/**
 * The font variant
 * ('normal' or 'small-caps')
 *
 * @member {string}
 */
prototypeAccessors.fontVariant.get = function ()
{
    return this._fontVariant;
};
prototypeAccessors.fontVariant.set = function (fontVariant) // eslint-disable-line require-jsdoc
{
    if (this._fontVariant !== fontVariant)
    {
        this._fontVariant = fontVariant;
        this.styleID++;
    }
};

/**
 * The font weight
 * ('normal', 'bold', 'bolder', 'lighter' and '100', '200', '300', '400', '500', '600', '700', 800' or '900')
 *
 * @member {string}
 */
prototypeAccessors.fontWeight.get = function ()
{
    return this._fontWeight;
};
prototypeAccessors.fontWeight.set = function (fontWeight) // eslint-disable-line require-jsdoc
{
    if (this._fontWeight !== fontWeight)
    {
        this._fontWeight = fontWeight;
        this.styleID++;
    }
};

/**
 * The amount of spacing between letters, default is 0
 *
 * @member {number}
 */
prototypeAccessors.letterSpacing.get = function ()
{
    return this._letterSpacing;
};
prototypeAccessors.letterSpacing.set = function (letterSpacing) // eslint-disable-line require-jsdoc
{
    if (this._letterSpacing !== letterSpacing)
    {
        this._letterSpacing = letterSpacing;
        this.styleID++;
    }
};

/**
 * The line height, a number that represents the vertical space that a letter uses
 *
 * @member {number}
 */
prototypeAccessors.lineHeight.get = function ()
{
    return this._lineHeight;
};
prototypeAccessors.lineHeight.set = function (lineHeight) // eslint-disable-line require-jsdoc
{
    if (this._lineHeight !== lineHeight)
    {
        this._lineHeight = lineHeight;
        this.styleID++;
    }
};

/**
 * The space between lines
 *
 * @member {number}
 */
prototypeAccessors.leading.get = function ()
{
    return this._leading;
};
prototypeAccessors.leading.set = function (leading) // eslint-disable-line require-jsdoc
{
    if (this._leading !== leading)
    {
        this._leading = leading;
        this.styleID++;
    }
};

/**
 * The lineJoin property sets the type of corner created, it can resolve spiked text issues.
 * Default is 'miter' (creates a sharp corner).
 *
 * @member {string}
 */
prototypeAccessors.lineJoin.get = function ()
{
    return this._lineJoin;
};
prototypeAccessors.lineJoin.set = function (lineJoin) // eslint-disable-line require-jsdoc
{
    if (this._lineJoin !== lineJoin)
    {
        this._lineJoin = lineJoin;
        this.styleID++;
    }
};

/**
 * The miter limit to use when using the 'miter' lineJoin mode
 * This can reduce or increase the spikiness of rendered text.
 *
 * @member {number}
 */
prototypeAccessors.miterLimit.get = function ()
{
    return this._miterLimit;
};
prototypeAccessors.miterLimit.set = function (miterLimit) // eslint-disable-line require-jsdoc
{
    if (this._miterLimit !== miterLimit)
    {
        this._miterLimit = miterLimit;
        this.styleID++;
    }
};

/**
 * Occasionally some fonts are cropped. Adding some padding will prevent this from happening
 * by adding padding to all sides of the text.
 *
 * @member {number}
 */
prototypeAccessors.padding.get = function ()
{
    return this._padding;
};
prototypeAccessors.padding.set = function (padding) // eslint-disable-line require-jsdoc
{
    if (this._padding !== padding)
    {
        this._padding = padding;
        this.styleID++;
    }
};

/**
 * A canvas fillstyle that will be used on the text stroke
 * e.g 'blue', '#FCFF00'
 *
 * @member {string|number}
 */
prototypeAccessors.stroke.get = function ()
{
    return this._stroke;
};
prototypeAccessors.stroke.set = function (stroke) // eslint-disable-line require-jsdoc
{
    var outputColor = getColor(stroke);
    if (this._stroke !== outputColor)
    {
        this._stroke = outputColor;
        this.styleID++;
    }
};

/**
 * A number that represents the thickness of the stroke.
 * Default is 0 (no stroke)
 *
 * @member {number}
 */
prototypeAccessors.strokeThickness.get = function ()
{
    return this._strokeThickness;
};
prototypeAccessors.strokeThickness.set = function (strokeThickness) // eslint-disable-line require-jsdoc
{
    if (this._strokeThickness !== strokeThickness)
    {
        this._strokeThickness = strokeThickness;
        this.styleID++;
    }
};

/**
 * The baseline of the text that is rendered.
 *
 * @member {string}
 */
prototypeAccessors.textBaseline.get = function ()
{
    return this._textBaseline;
};
prototypeAccessors.textBaseline.set = function (textBaseline) // eslint-disable-line require-jsdoc
{
    if (this._textBaseline !== textBaseline)
    {
        this._textBaseline = textBaseline;
        this.styleID++;
    }
};

/**
 * Trim transparent borders
 *
 * @member {boolean}
 */
prototypeAccessors.trim.get = function ()
{
    return this._trim;
};
prototypeAccessors.trim.set = function (trim) // eslint-disable-line require-jsdoc
{
    if (this._trim !== trim)
    {
        this._trim = trim;
        this.styleID++;
    }
};

/**
 * Indicates if word wrap should be used
 *
 * @member {boolean}
 */
prototypeAccessors.wordWrap.get = function ()
{
    return this._wordWrap;
};
prototypeAccessors.wordWrap.set = function (wordWrap) // eslint-disable-line require-jsdoc
{
    if (this._wordWrap !== wordWrap)
    {
        this._wordWrap = wordWrap;
        this.styleID++;
    }
};

/**
 * The width at which text will wrap, it needs wordWrap to be set to true
 *
 * @member {number}
 */
prototypeAccessors.wordWrapWidth.get = function ()
{
    return this._wordWrapWidth;
};
prototypeAccessors.wordWrapWidth.set = function (wordWrapWidth) // eslint-disable-line require-jsdoc
{
    if (this._wordWrapWidth !== wordWrapWidth)
    {
        this._wordWrapWidth = wordWrapWidth;
        this.styleID++;
    }
};

/**
 * Generates a font style string to use for `TextMetrics.measureFont()`.
 *
 * @return {string} Font style string, for passing to `TextMetrics.measureFont()`
 */
TextStyle.prototype.toFontString = function toFontString ()
{
    // build canvas api font setting from individual components. Convert a numeric this.fontSize to px
    var fontSizeString = (typeof this.fontSize === 'number') ? ((this.fontSize) + "px") : this.fontSize;

    // Clean-up fontFamily property by quoting each font name
    // this will support font names with spaces
    var fontFamilies = this.fontFamily;

    if (!Array.isArray(this.fontFamily))
    {
        fontFamilies = this.fontFamily.split(',');
    }

    for (var i = fontFamilies.length - 1; i >= 0; i--)
    {
        // Trim any extra white-space
        var fontFamily = fontFamilies[i].trim();

        // Check if font already contains strings
        if (!(/([\"\'])[^\'\"]+\1/).test(fontFamily))
        {
            fontFamily = "\"" + fontFamily + "\"";
        }
        fontFamilies[i] = fontFamily;
    }

    return ((this.fontStyle) + " " + (this.fontVariant) + " " + (this.fontWeight) + " " + fontSizeString + " " + (fontFamilies.join(',')));
};

Object.defineProperties( TextStyle.prototype, prototypeAccessors );

/**
 * Utility function to convert hexadecimal colors to strings, and simply return the color if it's a string.
 * @private
 * @param {number|number[]} color
 * @return {string} The color as a string.
 */
function getSingleColor(color)
{
    if (typeof color === 'number')
    {
        return utils.hex2string(color);
    }
    else if ( typeof color === 'string' )
    {
        if ( color.indexOf('0x') === 0 )
        {
            color = color.replace('0x', '#');
        }
    }

    return color;
}

/**
 * Utility function to convert hexadecimal colors to strings, and simply return the color if it's a string.
 * This version can also convert array of colors
 * @private
 * @param {number|number[]} color
 * @return {string} The color as a string.
 */
function getColor(color)
{
    if (!Array.isArray(color))
    {
        return getSingleColor(color);
    }
    else
    {
        for (var i = 0; i < color.length; ++i)
        {
            color[i] = getSingleColor(color[i]);
        }

        return color;
    }
}

/**
 * Utility function to convert hexadecimal colors to strings, and simply return the color if it's a string.
 * This version can also convert array of colors
 * @private
 * @param {Array} array1 First array to compare
 * @param {Array} array2 Second array to compare
 * @return {boolean} Do the arrays contain the same values in the same order
 */
function areArraysEqual(array1, array2)
{
    if (!Array.isArray(array1) || !Array.isArray(array2))
    {
        return false;
    }

    if (array1.length !== array2.length)
    {
        return false;
    }

    for (var i = 0; i < array1.length; ++i)
    {
        if (array1[i] !== array2[i])
        {
            return false;
        }
    }

    return true;
}

/**
 * Utility function to ensure that object properties are copied by value, and not by reference
 * @private
 * @param {Object} target Target object to copy properties into
 * @param {Object} source Source object for the proporties to copy
 * @param {string} propertyObj Object containing properties names we want to loop over
 */
function deepCopyProperties(target, source, propertyObj) {
    for (var prop in propertyObj) {
        if (Array.isArray(source[prop])) {
            target[prop] = source[prop].slice();
        } else {
            target[prop] = source[prop];
        }
    }
}

/**
 * The TextMetrics object represents the measurement of a block of text with a specified style.
 *
 * @class
 * @memberOf PIXI
 */
var TextMetrics = function TextMetrics(text, style, width, height, lines, lineWidths, lineHeight, maxLineWidth, fontProperties)
{
    this.text = text;
    this.style = style;
    this.width = width;
    this.height = height;
    this.lines = lines;
    this.lineWidths = lineWidths;
    this.lineHeight = lineHeight;
    this.maxLineWidth = maxLineWidth;
    this.fontProperties = fontProperties;
};

/**
 * Measures the supplied string of text and returns a Rectangle.
 *
 * @param {string} text - the text to measure.
 * @param {PIXI.TextStyle} style - the text style to use for measuring
 * @param {boolean} [wordWrap] - optional override for if word-wrap should be applied to the text.
 * @param {HTMLCanvasElement} [canvas] - optional specification of the canvas to use for measuring.
 * @return {PIXI.TextMetrics} measured width and height of the text.
 */
TextMetrics.measureText = function measureText (text, style, wordWrap, canvas)
{
        if ( canvas === void 0 ) canvas = TextMetrics._canvas;

    wordWrap = wordWrap || style.wordWrap;
    var font = style.toFontString();
    var fontProperties = TextMetrics.measureFont(font);
    var context = canvas.getContext('2d');

    context.font = font;

    var outputText = wordWrap ? TextMetrics.wordWrap(text, style, canvas) : text;
    var lines = outputText.split(/(?:\r\n|\r|\n)/);
    var lineWidths = new Array(lines.length);
    var maxLineWidth = 0;

    for (var i = 0; i < lines.length; i++)
    {
        var lineWidth = context.measureText(lines[i]).width + ((lines[i].length - 1) * style.letterSpacing);

        lineWidths[i] = lineWidth;
        maxLineWidth = Math.max(maxLineWidth, lineWidth);
    }
    var width = maxLineWidth + style.strokeThickness;

    if (style.dropShadow)
    {
        width += style.dropShadowDistance;
    }

    var lineHeight = style.lineHeight || fontProperties.fontSize + style.strokeThickness;
    var height = Math.max(lineHeight, fontProperties.fontSize + style.strokeThickness)
        + ((lines.length - 1) * (lineHeight + style.leading));

    if (style.dropShadow)
    {
        height += style.dropShadowDistance;
    }

    return new TextMetrics(
        text,
        style,
        width,
        height,
        lines,
        lineWidths,
        lineHeight + style.leading,
        maxLineWidth,
        fontProperties
    );
};

/**
 * Applies newlines to a string to have it optimally fit into the horizontal
 * bounds set by the Text object's wordWrapWidth property.
 *
 * @private
 * @param {string} text - String to apply word wrapping to
 * @param {PIXI.TextStyle} style - the style to use when wrapping
 * @param {HTMLCanvasElement} [canvas] - optional specification of the canvas to use for measuring.
 * @return {string} New string with new lines applied where required
 */
TextMetrics.wordWrap = function wordWrap (text, style, canvas)
{
        if ( canvas === void 0 ) canvas = TextMetrics._canvas;

    var context = canvas.getContext('2d');

    // Greedy wrapping algorithm that will wrap words as the line grows longer
    // than its horizontal bounds.
    var result = '';
    var lines = text.split('\n');
    var wordWrapWidth = style.wordWrapWidth;
    var characterCache = {};

    for (var i = 0; i < lines.length; i++)
    {
        var spaceLeft = wordWrapWidth;
        var words = lines[i].split(' ');

        for (var j = 0; j < words.length; j++)
        {
            var wordWidth = context.measureText(words[j]).width;

            if (style.breakWords && wordWidth > wordWrapWidth)
            {
                // Word should be split in the middle
                var characters = words[j].split('');

                for (var c = 0; c < characters.length; c++)
                {
                    var character = characters[c];
                    var characterWidth = characterCache[character];

                    if (characterWidth === undefined)
                    {
                        characterWidth = context.measureText(character).width;
                        characterCache[character] = characterWidth;
                    }

                    if (characterWidth > spaceLeft)
                    {
                        result += "\n" + character;
                        spaceLeft = wordWrapWidth - characterWidth;
                    }
                    else
                    {
                        if (c === 0)
                        {
                            result += ' ';
                        }

                        result += character;
                        spaceLeft -= characterWidth;
                    }
                }
            }
            else
            {
                var wordWidthWithSpace = wordWidth + context.measureText(' ').width;

                if (j === 0 || wordWidthWithSpace > spaceLeft)
                {
                    // Skip printing the newline if it's the first word of the line that is
                    // greater than the word wrap width.
                    if (j > 0)
                    {
                        result += '\n';
                    }
                    result += words[j];
                    spaceLeft = wordWrapWidth - wordWidth;
                }
                else
                {
                    spaceLeft -= wordWidthWithSpace;
                    result += " " + (words[j]);
                }
            }
        }

        if (i < lines.length - 1)
        {
            result += '\n';
        }
    }

    return result;
};

/**
 * Calculates the ascent, descent and fontSize of a given font-style
 *
 * @static
 * @param {string} font - String representing the style of the font
 * @return {PIXI.TextMetrics~FontMetrics} Font properties object
 */
TextMetrics.measureFont = function measureFont (font)
{
    // as this method is used for preparing assets, don't recalculate things if we don't need to
    if (TextMetrics._fonts[font])
    {
        return TextMetrics._fonts[font];
    }

    var properties = {};

    var canvas = TextMetrics._canvas;
    var context = TextMetrics._context;

    context.font = font;

    var width = Math.ceil(context.measureText('|MÉq').width);
    var baseline = Math.ceil(context.measureText('M').width);
    var height = 2 * baseline;

    baseline = baseline * 1.4 | 0;

    canvas.width = width;
    canvas.height = height;

    context.fillStyle = '#f00';
    context.fillRect(0, 0, width, height);

    context.font = font;

    context.textBaseline = 'alphabetic';
    context.fillStyle = '#000';
    context.fillText('|MÉq', 0, baseline);

    var imagedata = context.getImageData(0, 0, width, height).data;
    var pixels = imagedata.length;
    var line = width * 4;

    var i = 0;
    var idx = 0;
    var stop = false;

    // ascent. scan from top to bottom until we find a non red pixel
    for (i = 0; i < baseline; ++i)
    {
        for (var j = 0; j < line; j += 4)
        {
            if (imagedata[idx + j] !== 255)
            {
                stop = true;
                break;
            }
        }
        if (!stop)
        {
            idx += line;
        }
        else
        {
            break;
        }
    }

    properties.ascent = baseline - i;

    idx = pixels - line;
    stop = false;

    // descent. scan from bottom to top until we find a non red pixel
    for (i = height; i > baseline; --i)
    {
        for (var j$1 = 0; j$1 < line; j$1 += 4)
        {
            if (imagedata[idx + j$1] !== 255)
            {
                stop = true;
                break;
            }
        }

        if (!stop)
        {
            idx -= line;
        }
        else
        {
            break;
        }
    }

    properties.descent = i - baseline;
    properties.fontSize = properties.ascent + properties.descent;

    TextMetrics._fonts[font] = properties;

    return properties;
};

/**
 * Internal return object for {@link PIXI.TextMetrics.measureFont `TextMetrics.measureFont`}.
 * @class FontMetrics
 * @memberof PIXI.TextMetrics~
 * @property {number} ascent - The ascent distance
 * @property {number} descent - The descent distance
 * @property {number} fontSize - Font size from ascent to descent
 */

var canvas = document.createElement('canvas');

canvas.width = canvas.height = 10;

/**
 * Cached canvas element for measuring text
 * @memberof PIXI.TextMetrics
 * @type {HTMLCanvasElement}
 * @private
 */
TextMetrics._canvas = canvas;

/**
 * Cache for context to use.
 * @memberof PIXI.TextMetrics
 * @type {CanvasRenderingContext2D}
 * @private
 */
TextMetrics._context = canvas.getContext('2d');

/**
 * Cache of PIXI.TextMetrics~FontMetrics objects.
 * @memberof PIXI.TextMetrics
 * @type {Object}
 * @private
 */
TextMetrics._fonts = {};

/* eslint max-depth: [2, 8] */
var defaultDestroyOptions = {
    texture: true,
    children: false,
    baseTexture: true,
};

/**
 * A Text Object will create a line or multiple lines of text. To split a line you can use '\n' in your text string,
 * or add a wordWrap property set to true and and wordWrapWidth property with a value in the style object.
 *
 * A Text can be created directly from a string and a style object
 *
 * ```js
 * let text = new PIXI.Text('This is a PixiJS text',{fontFamily : 'Arial', fontSize: 24, fill : 0xff1010, align : 'center'});
 * ```
 *
 * @class
 * @extends PIXI.Sprite
 * @memberof PIXI
 */
var Text = (function (Sprite$$1) {
    function Text(text, style, canvas)
    {
        canvas = canvas || document.createElement('canvas');

        canvas.width = 3;
        canvas.height = 3;

        var texture = core.Texture.from(canvas, settings.settings.SCALE_MODE, 'text');

        texture.orig = new math.Rectangle();
        texture.trim = new math.Rectangle();

        Sprite$$1.call(this, texture);

        // base texture is already automatically added to the cache, now adding the actual texture
        core.Texture.addToCache(this._texture, this._texture.baseTexture.textureCacheIds[0]);

        /**
         * The canvas element that everything is drawn to
         *
         * @member {HTMLCanvasElement}
         */
        this.canvas = canvas;

        /**
         * The canvas 2d context that everything is drawn with
         * @member {CanvasRenderingContext2D}
         */
        this.context = this.canvas.getContext('2d');

        /**
         * The resolution / device pixel ratio of the canvas. This is set automatically by the renderer.
         * @member {number}
         * @default 1
         */
        this.resolution = settings.settings.RESOLUTION;

        /**
         * Private tracker for the current text.
         *
         * @member {string}
         * @private
         */
        this._text = null;

        /**
         * Private tracker for the current style.
         *
         * @member {object}
         * @private
         */
        this._style = null;
        /**
         * Private listener to track style changes.
         *
         * @member {Function}
         * @private
         */
        this._styleListener = null;

        /**
         * Private tracker for the current font.
         *
         * @member {string}
         * @private
         */
        this._font = '';

        this.text = text;
        this.style = style;

        this.localStyleID = -1;
    }

    if ( Sprite$$1 ) Text.__proto__ = Sprite$$1;
    Text.prototype = Object.create( Sprite$$1 && Sprite$$1.prototype );
    Text.prototype.constructor = Text;

    var prototypeAccessors = { width: { configurable: true },height: { configurable: true },style: { configurable: true },text: { configurable: true } };

    /**
     * Renders text and updates it when needed.
     *
     * @private
     * @param {boolean} respectDirty - Whether to abort updating the text if the Text isn't dirty and the function is called.
     */
    Text.prototype.updateText = function updateText (respectDirty)
    {
        var this$1 = this;

        var style = this._style;

        // check if style has changed..
        if (this.localStyleID !== style.styleID)
        {
            this.dirty = true;
            this.localStyleID = style.styleID;
        }

        if (!this.dirty && respectDirty)
        {
            return;
        }

        this._font = this._style.toFontString();

        var context = this.context;
        var measured = TextMetrics.measureText(this._text, this._style, this._style.wordWrap, this.canvas);
        var width = measured.width;
        var height = measured.height;
        var lines = measured.lines;
        var lineHeight = measured.lineHeight;
        var lineWidths = measured.lineWidths;
        var maxLineWidth = measured.maxLineWidth;
        var fontProperties = measured.fontProperties;

        this.canvas.width = Math.ceil((Math.max(1, width) + (style.padding * 2)) * this.resolution);
        this.canvas.height = Math.ceil((Math.max(1, height) + (style.padding * 2)) * this.resolution);

        context.scale(this.resolution, this.resolution);

        context.clearRect(0, 0, this.canvas.width, this.canvas.height);

        context.font = this._font;
        context.strokeStyle = style.stroke;
        context.lineWidth = style.strokeThickness;
        context.textBaseline = style.textBaseline;
        context.lineJoin = style.lineJoin;
        context.miterLimit = style.miterLimit;

        var linePositionX;
        var linePositionY;

        if (style.dropShadow)
        {
            context.fillStyle = style.dropShadowColor;
            context.globalAlpha = style.dropShadowAlpha;
            context.shadowBlur = style.dropShadowBlur;

            if (style.dropShadowBlur > 0)
            {
                context.shadowColor = style.dropShadowColor;
            }

            var xShadowOffset = Math.cos(style.dropShadowAngle) * style.dropShadowDistance;
            var yShadowOffset = Math.sin(style.dropShadowAngle) * style.dropShadowDistance;

            for (var i = 0; i < lines.length; i++)
            {
                linePositionX = style.strokeThickness / 2;
                linePositionY = ((style.strokeThickness / 2) + (i * lineHeight)) + fontProperties.ascent;

                if (style.align === 'right')
                {
                    linePositionX += maxLineWidth - lineWidths[i];
                }
                else if (style.align === 'center')
                {
                    linePositionX += (maxLineWidth - lineWidths[i]) / 2;
                }

                if (style.fill)
                {
                    this$1.drawLetterSpacing(
                        lines[i],
                        linePositionX + xShadowOffset + style.padding, linePositionY + yShadowOffset + style.padding
                    );

                    if (style.stroke && style.strokeThickness)
                    {
                        context.strokeStyle = style.dropShadowColor;
                        this$1.drawLetterSpacing(
                            lines[i],
                            linePositionX + xShadowOffset + style.padding, linePositionY + yShadowOffset + style.padding,
                            true
                        );
                        context.strokeStyle = style.stroke;
                    }
                }
            }
        }

        // reset the shadow blur and alpha that was set by the drop shadow, for the regular text
        context.shadowBlur = 0;
        context.globalAlpha = 1;

        // set canvas text styles
        context.fillStyle = this._generateFillStyle(style, lines);

        // draw lines line by line
        for (var i$1 = 0; i$1 < lines.length; i$1++)
        {
            linePositionX = style.strokeThickness / 2;
            linePositionY = ((style.strokeThickness / 2) + (i$1 * lineHeight)) + fontProperties.ascent;

            if (style.align === 'right')
            {
                linePositionX += maxLineWidth - lineWidths[i$1];
            }
            else if (style.align === 'center')
            {
                linePositionX += (maxLineWidth - lineWidths[i$1]) / 2;
            }

            if (style.stroke && style.strokeThickness)
            {
                this$1.drawLetterSpacing(
                    lines[i$1],
                    linePositionX + style.padding,
                    linePositionY + style.padding,
                    true
                );
            }

            if (style.fill)
            {
                this$1.drawLetterSpacing(
                    lines[i$1],
                    linePositionX + style.padding,
                    linePositionY + style.padding
                );
            }
        }

        this.updateTexture();
    };

    /**
     * Render the text with letter-spacing.
     * @param {string} text - The text to draw
     * @param {number} x - Horizontal position to draw the text
     * @param {number} y - Vertical position to draw the text
     * @param {boolean} [isStroke=false] - Is this drawing for the outside stroke of the
     *  text? If not, it's for the inside fill
     * @private
     */
    Text.prototype.drawLetterSpacing = function drawLetterSpacing (text, x, y, isStroke)
    {
        var this$1 = this;
        if ( isStroke === void 0 ) isStroke = false;

        var style = this._style;

        // letterSpacing of 0 means normal
        var letterSpacing = style.letterSpacing;

        if (letterSpacing === 0)
        {
            if (isStroke)
            {
                this.context.strokeText(text, x, y);
            }
            else
            {
                this.context.fillText(text, x, y);
            }

            return;
        }

        var characters = String.prototype.split.call(text, '');
        var currentPosition = x;
        var index = 0;
        var current = '';

        while (index < text.length)
        {
            current = characters[index++];
            if (isStroke)
            {
                this$1.context.strokeText(current, currentPosition, y);
            }
            else
            {
                this$1.context.fillText(current, currentPosition, y);
            }
            currentPosition += this$1.context.measureText(current).width + letterSpacing;
        }
    };

    /**
     * Updates texture size based on canvas size
     *
     * @private
     */
    Text.prototype.updateTexture = function updateTexture ()
    {
        var canvas = this.canvas;

        if (this._style.trim)
        {
            var trimmed = utils.trimCanvas(canvas);

            canvas.width = trimmed.width;
            canvas.height = trimmed.height;
            this.context.putImageData(trimmed.data, 0, 0);
        }

        var texture = this._texture;
        var style = this._style;
        var padding = style.trim ? 0 : style.padding;
        var baseTexture = texture.baseTexture;

        texture.trim.width = texture._frame.width = canvas.width / this.resolution;
        texture.trim.height = texture._frame.height = canvas.height / this.resolution;
        texture.trim.x = -padding;
        texture.trim.y = -padding;

        texture.orig.width = texture._frame.width - (padding * 2);
        texture.orig.height = texture._frame.height - (padding * 2);

        // call sprite onTextureUpdate to update scale if _width or _height were set
        this._onTextureUpdate();

        baseTexture.setRealSize(canvas.width, canvas.height, this.resolution);

        this.dirty = false;
    };

    /**
     * Renders the object using the WebGL renderer
     *
     * @param {PIXI.Renderer} renderer - The renderer
     */
    Text.prototype.render = function render (renderer)
    {
        if (this.resolution !== renderer.resolution)
        {
            this.resolution = renderer.resolution;
            this.dirty = true;
        }

        this.updateText(true);

        Sprite$$1.prototype.render.call(this, renderer);
    };

    /**
     * Renders the object using the Canvas renderer
     *
     * @private
     * @param {PIXI.CanvasRenderer} renderer - The renderer
     */
    Text.prototype._renderCanvas = function _renderCanvas (renderer)
    {
        if (this.resolution !== renderer.resolution)
        {
            this.resolution = renderer.resolution;
            this.dirty = true;
        }

        this.updateText(true);

        Sprite$$1.prototype._renderCanvas.call(this, renderer);
    };

    /**
     * Gets the local bounds of the text object.
     *
     * @param {Rectangle} rect - The output rectangle.
     * @return {Rectangle} The bounds.
     */
    Text.prototype.getLocalBounds = function getLocalBounds (rect)
    {
        this.updateText(true);

        return Sprite$$1.prototype.getLocalBounds.call(this, rect);
    };

    /**
     * calculates the bounds of the Text as a rectangle. The bounds calculation takes the worldTransform into account.
     */
    Text.prototype._calculateBounds = function _calculateBounds ()
    {
        this.updateText(true);
        this.calculateVertices();
        // if we have already done this on THIS frame.
        this._bounds.addQuad(this.vertexData);
    };

    /**
     * Method to be called upon a TextStyle change.
     * @private
     */
    Text.prototype._onStyleChange = function _onStyleChange ()
    {
        this.dirty = true;
    };

    /**
     * Generates the fill style. Can automatically generate a gradient based on the fill style being an array
     *
     * @private
     * @param {object} style - The style.
     * @param {string[]} lines - The lines of text.
     * @return {string|number|CanvasGradient} The fill style
     */
    Text.prototype._generateFillStyle = function _generateFillStyle (style, lines)
    {
        if (!Array.isArray(style.fill))
        {
            return style.fill;
        }

        // cocoon on canvas+ cannot generate textures, so use the first colour instead
        if (navigator.isCocoonJS)
        {
            return style.fill[0];
        }

        // the gradient will be evenly spaced out according to how large the array is.
        // ['#FF0000', '#00FF00', '#0000FF'] would created stops at 0.25, 0.5 and 0.75
        var gradient;
        var totalIterations;
        var currentIteration;
        var stop;

        var width = this.canvas.width / this.resolution;
        var height = this.canvas.height / this.resolution;

        // make a copy of the style settings, so we can manipulate them later
        var fill = style.fill.slice();
        var fillGradientStops = style.fillGradientStops.slice();

        // wanting to evenly distribute the fills. So an array of 4 colours should give fills of 0.25, 0.5 and 0.75
        if (!fillGradientStops.length)
        {
            var lengthPlus1 = fill.length + 1;

            for (var i = 1; i < lengthPlus1; ++i)
            {
                fillGradientStops.push(i / lengthPlus1);
            }
        }

        // stop the bleeding of the last gradient on the line above to the top gradient of the this line
        // by hard defining the first gradient colour at point 0, and last gradient colour at point 1
        fill.unshift(style.fill[0]);
        fillGradientStops.unshift(0);

        fill.push(style.fill[style.fill.length - 1]);
        fillGradientStops.push(1);

        if (style.fillGradientType === TEXT_GRADIENT.LINEAR_VERTICAL)
        {
            // start the gradient at the top center of the canvas, and end at the bottom middle of the canvas
            gradient = this.context.createLinearGradient(width / 2, 0, width / 2, height);

            // we need to repeat the gradient so that each individual line of text has the same vertical gradient effect
            // ['#FF0000', '#00FF00', '#0000FF'] over 2 lines would create stops at 0.125, 0.25, 0.375, 0.625, 0.75, 0.875
            totalIterations = (fill.length + 1) * lines.length;
            currentIteration = 0;
            for (var i$1 = 0; i$1 < lines.length; i$1++)
            {
                currentIteration += 1;
                for (var j = 0; j < fill.length; j++)
                {
                    if (typeof fillGradientStops[j] === 'number')
                    {
                        stop = (fillGradientStops[j] / lines.length) + (i$1 / lines.length);
                    }
                    else
                    {
                        stop = currentIteration / totalIterations;
                    }
                    gradient.addColorStop(stop, fill[j]);
                    currentIteration++;
                }
            }
        }
        else
        {
            // start the gradient at the center left of the canvas, and end at the center right of the canvas
            gradient = this.context.createLinearGradient(0, height / 2, width, height / 2);

            // can just evenly space out the gradients in this case, as multiple lines makes no difference
            // to an even left to right gradient
            totalIterations = fill.length + 1;
            currentIteration = 1;

            for (var i$2 = 0; i$2 < fill.length; i$2++)
            {
                if (typeof fillGradientStops[i$2] === 'number')
                {
                    stop = fillGradientStops[i$2];
                }
                else
                {
                    stop = currentIteration / totalIterations;
                }
                gradient.addColorStop(stop, fill[i$2]);
                currentIteration++;
            }
        }

        return gradient;
    };

    /**
     * Destroys this text object.
     * Note* Unlike a Sprite, a Text object will automatically destroy its baseTexture and texture as
     * the majority of the time the texture will not be shared with any other Sprites.
     *
     * @param {object|boolean} [options] - Options parameter. A boolean will act as if all options
     *  have been set to that value
     * @param {boolean} [options.children=false] - if set to true, all the children will have their
     *  destroy method called as well. 'options' will be passed on to those calls.
     * @param {boolean} [options.texture=true] - Should it destroy the current texture of the sprite as well
     * @param {boolean} [options.baseTexture=true] - Should it destroy the base texture of the sprite as well
     */
    Text.prototype.destroy = function destroy (options)
    {
        if (typeof options === 'boolean')
        {
            options = { children: options };
        }

        options = Object.assign({}, defaultDestroyOptions, options);

        Sprite$$1.prototype.destroy.call(this, options);

        // make sure to reset the the context and canvas.. dont want this hanging around in memory!
        this.context = null;
        this.canvas = null;

        this._style = null;
    };

    /**
     * The width of the Text, setting this will actually modify the scale to achieve the value set
     *
     * @member {number}
     */
    prototypeAccessors.width.get = function ()
    {
        this.updateText(true);

        return Math.abs(this.scale.x) * this._texture.orig.width;
    };

    prototypeAccessors.width.set = function (value) // eslint-disable-line require-jsdoc
    {
        this.updateText(true);

        var s = utils.sign(this.scale.x) || 1;

        this.scale.x = s * value / this._texture.orig.width;
        this._width = value;
    };

    /**
     * The height of the Text, setting this will actually modify the scale to achieve the value set
     *
     * @member {number}
     */
    prototypeAccessors.height.get = function ()
    {
        this.updateText(true);

        return Math.abs(this.scale.y) * this._texture.orig.height;
    };

    prototypeAccessors.height.set = function (value) // eslint-disable-line require-jsdoc
    {
        this.updateText(true);

        var s = utils.sign(this.scale.y) || 1;

        this.scale.y = s * value / this._texture.orig.height;
        this._height = value;
    };

    /**
     * Set the style of the text. Set up an event listener to listen for changes on the style
     * object and mark the text as dirty.
     *
     * @member {object|PIXI.TextStyle}
     */
    prototypeAccessors.style.get = function ()
    {
        return this._style;
    };

    prototypeAccessors.style.set = function (style) // eslint-disable-line require-jsdoc
    {
        style = style || {};

        if (style instanceof TextStyle)
        {
            this._style = style;
        }
        else
        {
            this._style = new TextStyle(style);
        }

        this.localStyleID = -1;
        this.dirty = true;
    };

    /**
     * Set the copy for the text object. To split a line you can use '\n'.
     *
     * @member {string}
     */
    prototypeAccessors.text.get = function ()
    {
        return this._text;
    };

    prototypeAccessors.text.set = function (text) // eslint-disable-line require-jsdoc
    {
        text = String(text === '' || text === null || text === undefined ? ' ' : text);

        if (this._text === text)
        {
            return;
        }
        this._text = text;
        this.dirty = true;
    };

    Object.defineProperties( Text.prototype, prototypeAccessors );

    return Text;
}(sprite.Sprite));

exports.Text = Text;
exports.TextStyle = TextStyle;
exports.TextMetrics = TextMetrics;
exports.TEXT_GRADIENT = TEXT_GRADIENT;
//# sourceMappingURL=text.js.map
