import type { ICanvas, ICanvasRenderingContext2D } from '@pixi/settings';
import type { TextStyle } from './TextStyle';
interface IFontMetrics {
    ascent: number;
    descent: number;
    fontSize: number;
}
/**
 * The TextMetrics object represents the measurement of a block of text with a specified style.
 * @example
 * import { TextMetrics, TextStyle } from 'pixi.js';
 *
 * const style = new TextStyle({
 *     fontFamily: 'Arial',
 *     fontSize: 24,
 *     fill: 0xff1010,
 *     align: 'center',
 * });
 * const textMetrics = TextMetrics.measureText('Your text', style);
 * @memberof PIXI
 */
export declare class TextMetrics {
    /** The text that was measured. */
    text: string;
    /** The style that was measured. */
    style: TextStyle;
    /** The measured width of the text. */
    width: number;
    /** The measured height of the text. */
    height: number;
    /** An array of lines of the text broken by new lines and wrapping is specified in style. */
    lines: string[];
    /** An array of the line widths for each line matched to `lines`. */
    lineWidths: number[];
    /** The measured line height for this style. */
    lineHeight: number;
    /** The maximum line width for all measured lines. */
    maxLineWidth: number;
    /**
     * The font properties object from TextMetrics.measureFont.
     * @type {PIXI.IFontMetrics}
     */
    fontProperties: IFontMetrics;
    static METRICS_STRING: string;
    static BASELINE_SYMBOL: string;
    static BASELINE_MULTIPLIER: number;
    static HEIGHT_MULTIPLIER: number;
    private static __canvas;
    private static __context;
    static _fonts: {
        [font: string]: IFontMetrics;
    };
    static _newlines: number[];
    static _breakingSpaces: number[];
    /**
     * @param text - the text that was measured
     * @param style - the style that was measured
     * @param width - the measured width of the text
     * @param height - the measured height of the text
     * @param lines - an array of the lines of text broken by new lines and wrapping if specified in style
     * @param lineWidths - an array of the line widths for each line matched to `lines`
     * @param lineHeight - the measured line height for this style
     * @param maxLineWidth - the maximum line width for all measured lines
     * @param {PIXI.IFontMetrics} fontProperties - the font properties object from TextMetrics.measureFont
     */
    constructor(text: string, style: TextStyle, width: number, height: number, lines: string[], lineWidths: number[], lineHeight: number, maxLineWidth: number, fontProperties: IFontMetrics);
    /**
     * Measures the supplied string of text and returns a Rectangle.
     * @param text - The text to measure.
     * @param style - The text style to use for measuring
     * @param wordWrap - Override for if word-wrap should be applied to the text.
     * @param canvas - optional specification of the canvas to use for measuring.
     * @returns Measured width and height of the text.
     */
    static measureText(text: string, style: TextStyle, wordWrap?: boolean, canvas?: ICanvas): TextMetrics;
    /**
     * Applies newlines to a string to have it optimally fit into the horizontal
     * bounds set by the Text object's wordWrapWidth property.
     * @param text - String to apply word wrapping to
     * @param style - the style to use when wrapping
     * @param canvas - optional specification of the canvas to use for measuring.
     * @returns New string with new lines applied where required
     */
    private static wordWrap;
    /**
     * Convienience function for logging each line added during the wordWrap method.
     * @param line    - The line of text to add
     * @param newLine - Add new line character to end
     * @returns A formatted line
     */
    private static addLine;
    /**
     * Gets & sets the widths of calculated characters in a cache object
     * @param key            - The key
     * @param letterSpacing  - The letter spacing
     * @param cache          - The cache
     * @param context        - The canvas context
     * @returns The from cache.
     */
    private static getFromCache;
    /**
     * Determines whether we should collapse breaking spaces.
     * @param whiteSpace - The TextStyle property whiteSpace
     * @returns Should collapse
     */
    private static collapseSpaces;
    /**
     * Determines whether we should collapse newLine chars.
     * @param whiteSpace - The white space
     * @returns  should collapse
     */
    private static collapseNewlines;
    /**
     * Trims breaking whitespaces from string.
     * @param  text - The text
     * @returns Trimmed string
     */
    private static trimRight;
    /**
     * Determines if char is a newline.
     * @param  char - The character
     * @returns True if newline, False otherwise.
     */
    private static isNewline;
    /**
     * Determines if char is a breaking whitespace.
     *
     * It allows one to determine whether char should be a breaking whitespace
     * For example certain characters in CJK langs or numbers.
     * It must return a boolean.
     * @param char - The character
     * @param [_nextChar] - The next character
     * @returns True if whitespace, False otherwise.
     */
    static isBreakingSpace(char: string, _nextChar?: string): boolean;
    /**
     * Splits a string into words, breaking-spaces and newLine characters
     * @param  text - The text
     * @returns  A tokenized array
     */
    private static tokenize;
    /**
     * Overridable helper method used internally by TextMetrics, exposed to allow customizing the class's behavior.
     *
     * It allows one to customise which words should break
     * Examples are if the token is CJK or numbers.
     * It must return a boolean.
     * @param _token - The token
     * @param  breakWords - The style attr break words
     * @returns Whether to break word or not
     */
    static canBreakWords(_token: string, breakWords: boolean): boolean;
    /**
     * Overridable helper method used internally by TextMetrics, exposed to allow customizing the class's behavior.
     *
     * It allows one to determine whether a pair of characters
     * should be broken by newlines
     * For example certain characters in CJK langs or numbers.
     * It must return a boolean.
     * @param _char - The character
     * @param _nextChar - The next character
     * @param _token - The token/word the characters are from
     * @param _index - The index in the token of the char
     * @param _breakWords - The style attr break words
     * @returns whether to break word or not
     */
    static canBreakChars(_char: string, _nextChar: string, _token: string, _index: number, _breakWords: boolean): boolean;
    /**
     * Overridable helper method used internally by TextMetrics, exposed to allow customizing the class's behavior.
     *
     * It is called when a token (usually a word) has to be split into separate pieces
     * in order to determine the point to break a word.
     * It must return an array of characters.
     * @example
     * // Correctly splits emojis, eg "🤪🤪" will result in two element array, each with one emoji.
     * TextMetrics.wordWrapSplit = (token) => [...token];
     * @param  token - The token to split
     * @returns The characters of the token
     */
    static wordWrapSplit(token: string): string[];
    /**
     * Calculates the ascent, descent and fontSize of a given font-style
     * @param font - String representing the style of the font
     * @returns Font properties object
     */
    static measureFont(font: string): IFontMetrics;
    /**
     * Clear font metrics in metrics cache.
     * @param {string} [font] - font name. If font name not set then clear cache for all fonts.
     */
    static clearMetrics(font?: string): void;
    /**
     * Cached canvas element for measuring text
     * TODO: this should be private, but isn't because of backward compat, will fix later.
     * @ignore
     */
    static get _canvas(): ICanvas;
    /**
     * TODO: this should be private, but isn't because of backward compat, will fix later.
     * @ignore
     */
    static get _context(): ICanvasRenderingContext2D;
}
export {};
/**
 * A number, or a string containing a number.
 * @memberof PIXI
 * @typedef {object} IFontMetrics
 * @property {number} ascent - Font ascent
 * @property {number} descent - Font descent
 * @property {number} fontSize - Font size
 */
