/**
 * @license
 * PlayCanvas Engine v1.69.2 revision 3e80480
 * Copyright 2011-2024 PlayCanvas Ltd. All rights reserved.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
import { now, math, Texture, FILTER_NEAREST, VertexFormat, SEMANTIC_POSITION, TYPE_FLOAT32, SEMANTIC_TEXCOORD0, shaderChunks, VertexBuffer, BUFFER_STREAM, IndexBuffer, INDEXFORMAT_UINT16, BUFFER_STATIC, PRIMITIVE_TRIANGLES, Mesh, Material, CULLFACE_NONE, DepthState, BlendState, BLENDEQUATION_ADD, BLENDMODE_SRC_ALPHA, BLENDMODE_ONE_MINUS_SRC_ALPHA, BLENDMODE_ONE, MeshInstance, GraphNode, LAYERID_UI, ADDRESS_REPEAT, FILTER_LINEAR, ADDRESS_CLAMP_TO_EDGE, RenderTarget, createShaderFromCode, drawQuadWithShader, SEMANTIC_NORMAL, SEMANTIC_TEXCOORD1, Color, PROJECTION_ORTHOGRAPHIC, Vec2, StandardMaterial, BasicMaterial, BLEND_NORMAL, BLEND_NONE, Vec3, Quat, BoundingBox, INDEXFORMAT_UINT32, INDEXFORMAT_UINT8, TYPE_UINT32, TYPE_INT32, TYPE_UINT16, TYPE_INT16, TYPE_UINT8, TYPE_INT8, SEMANTIC_TEXCOORD7, SEMANTIC_TEXCOORD6, SEMANTIC_TEXCOORD5, SEMANTIC_TEXCOORD4, SEMANTIC_TEXCOORD3, SEMANTIC_TEXCOORD2, SEMANTIC_BLENDWEIGHT, SEMANTIC_BLENDINDICES, SEMANTIC_COLOR, SEMANTIC_TANGENT, FILTER_LINEAR_MIPMAP_LINEAR, FILTER_NEAREST_MIPMAP_LINEAR, FILTER_LINEAR_MIPMAP_NEAREST, FILTER_NEAREST_MIPMAP_NEAREST, ADDRESS_MIRRORED_REPEAT, RenderPassShaderQuad, RenderPass, TONEMAP_ACES2, TONEMAP_ACES, TONEMAP_HEJL, TONEMAP_FILMIC, TONEMAP_LINEAR, PIXELFORMAT_RGBA32F, PIXELFORMAT_RGBA16F, LAYERID_DEPTH, SHADER_PREPASS_VELOCITY, PIXELFORMAT_RGBA8, PIXELFORMAT_DEPTH, RenderPassForward, RenderPassColorGrab, LAYERID_SKYBOX, LAYERID_IMMEDIATE, Mat4, EventHandler, Entity, PROJECTION_PERSPECTIVE, createCone, createCylinder, createBox, createTorus, createPlane, createMesh, CULLFACE_BACK } from 'playcanvas';
import { zipSync, strToU8 } from 'fflate';

class CpuTimer {
	constructor(app) {
		this._frameIndex = 0;
		this._frameTimings = [];
		this._timings = [];
		this._prevTimings = [];
		this.unitsName = 'ms';
		this.decimalPlaces = 1;
		this.enabled = true;
		app.on('frameupdate', this.begin.bind(this, 'update'));
		app.on('framerender', this.mark.bind(this, 'render'));
		app.on('frameend', this.mark.bind(this, 'other'));
	}
	begin(name) {
		if (!this.enabled) {
			return;
		}
		if (this._frameIndex < this._frameTimings.length) {
			this._frameTimings.splice(this._frameIndex);
		}
		const tmp = this._prevTimings;
		this._prevTimings = this._timings;
		this._timings = this._frameTimings;
		this._frameTimings = tmp;
		this._frameIndex = 0;
		this.mark(name);
	}
	mark(name) {
		if (!this.enabled) {
			return;
		}
		const timestamp = now();
		if (this._frameIndex > 0) {
			const prev = this._frameTimings[this._frameIndex - 1];
			prev[1] = timestamp - prev[1];
		} else if (this._timings.length > 0) {
			const prev = this._timings[this._timings.length - 1];
			prev[1] = timestamp - prev[1];
		}
		if (this._frameIndex >= this._frameTimings.length) {
			this._frameTimings.push([name, timestamp]);
		} else {
			const timing = this._frameTimings[this._frameIndex];
			timing[0] = name;
			timing[1] = timestamp;
		}
		this._frameIndex++;
	}
	get timings() {
		return this._timings.slice(0, -1).map(v => v[1]);
	}
}

class GpuTimer {
	constructor(device) {
		this.device = device;
		device.gpuProfiler.enabled = true;
		this.enabled = true;
		this.unitsName = 'ms';
		this.decimalPlaces = 1;
		this._timings = [];
	}
	get timings() {
		this._timings[0] = this.device.gpuProfiler._frameTime;
		return this._timings;
	}
}

class StatsTimer {
	constructor(app, statNames, decimalPlaces, unitsName, multiplier) {
		this.app = app;
		this.values = [];
		this.statNames = statNames;
		if (this.statNames.length > 3) this.statNames.length = 3;
		this.unitsName = unitsName;
		this.decimalPlaces = decimalPlaces;
		this.multiplier = multiplier || 1;
		const resolve = (path, obj) => {
			return path.split('.').reduce((prev, curr) => {
				return prev ? prev[curr] : null;
			}, obj || this);
		};
		app.on('frameupdate', ms => {
			for (let i = 0; i < this.statNames.length; i++) {
				this.values[i] = resolve(this.statNames[i], this.app.stats) * this.multiplier;
			}
		});
	}
	get timings() {
		return this.values;
	}
}

class Graph {
	constructor(name, app, watermark, textRefreshRate, timer) {
		this.app = app;
		this.name = name;
		this.device = app.graphicsDevice;
		this.timer = timer;
		this.watermark = watermark;
		this.enabled = false;
		this.textRefreshRate = textRefreshRate;
		this.avgTotal = 0;
		this.avgTimer = 0;
		this.avgCount = 0;
		this.timingText = '';
		this.texture = null;
		this.yOffset = 0;
		this.cursor = 0;
		this.sample = new Uint8ClampedArray(4);
		this.sample.set([0, 0, 0, 255]);
		this.counter = 0;
		this.app.on('frameupdate', this.update, this);
	}
	destroy() {
		this.app.off('frameupdate', this.update, this);
	}
	loseContext() {
		if (this.timer && typeof this.timer.loseContext === 'function') {
			this.timer.loseContext();
		}
	}
	update(ms) {
		const timings = this.timer.timings;
		const total = timings.reduce((a, v) => a + v, 0);
		this.avgTotal += total;
		this.avgTimer += ms;
		this.avgCount++;
		if (this.avgTimer > this.textRefreshRate) {
			this.timingText = (this.avgTotal / this.avgCount).toFixed(this.timer.decimalPlaces);
			this.avgTimer = 0;
			this.avgTotal = 0;
			this.avgCount = 0;
		}
		if (this.enabled) {
			let value = 0;
			const range = 1.5 * this.watermark;
			for (let i = 0; i < timings.length; ++i) {
				value += Math.floor(timings[i] / range * 255);
				this.sample[i] = value;
			}
			this.sample[3] = this.watermark / range * 255;
			const data = this.texture.lock();
			data.set(this.sample, (this.cursor + this.yOffset * this.texture.width) * 4);
			this.texture.unlock();
			this.cursor++;
			if (this.cursor === this.texture.width) {
				this.cursor = 0;
			}
		}
	}
	render(render2d, x, y, w, h) {
		render2d.quad(x + w, y, -w, h, this.enabled ? this.cursor : 0, this.enabled ? 0.5 + this.yOffset : this.texture.height - 1, -w, 0, this.texture, 0);
	}
}

class WordAtlas {
	constructor(device, words) {
		const initContext = context => {
			context.font = '10px "Lucida Console", Monaco, monospace';
			context.textAlign = 'left';
			context.textBaseline = 'alphabetic';
		};
		const isNumber = word => {
			return word === '.' || word.length === 1 && word.charCodeAt(0) >= 48 && word.charCodeAt(0) <= 57;
		};
		const canvas = document.createElement('canvas');
		const context = canvas.getContext('2d', {
			alpha: true
		});
		initContext(context);
		const placements = new Map();
		const padding = 5;
		const width = 512;
		let x = padding;
		let y = padding;
		words.forEach(word => {
			const measurement = context.measureText(word);
			const l = Math.ceil(-measurement.actualBoundingBoxLeft);
			const r = Math.ceil(measurement.actualBoundingBoxRight);
			const a = Math.ceil(measurement.actualBoundingBoxAscent);
			const d = Math.ceil(measurement.actualBoundingBoxDescent);
			const w = l + r;
			const h = a + d;
			if (x + w + padding >= width) {
				x = padding;
				y += 16;
			}
			placements.set(word, {
				l,
				r,
				a,
				d,
				w,
				h,
				x: x,
				y: y
			});
			x += w + padding;
		});
		canvas.width = 512;
		canvas.height = math.nextPowerOfTwo(y + 16 + padding);
		initContext(context);
		context.fillStyle = 'rgb(0, 0, 0)';
		context.fillRect(0, 0, canvas.width, canvas.height);
		placements.forEach((m, word) => {
			context.fillStyle = isNumber(word) ? 'rgb(255, 255, 255)' : 'rgb(170, 170, 170)';
			context.fillText(word, m.x - m.l, m.y + m.a);
		});
		this.placements = placements;
		const data = context.getImageData(0, 0, canvas.width, canvas.height).data;
		for (let i = 0; i < data.length; i += 4) {
			data[i + 3] = data[i + 0];
			data[i + 0] = 255;
			data[i + 1] = 255;
			data[i + 2] = 255;
		}
		this.texture = new Texture(device, {
			name: 'mini-stats-word-atlas',
			width: canvas.width,
			height: canvas.height,
			mipmaps: false,
			minFilter: FILTER_NEAREST,
			magFilter: FILTER_NEAREST,
			levels: [data]
		});
	}
	destroy() {
		this.texture.destroy();
		this.texture = null;
	}
	render(render2d, word, x, y) {
		const p = this.placements.get(word);
		if (p) {
			const padding = 1;
			render2d.quad(x + p.l - padding, y - p.d + padding, p.w + padding * 2, p.h + padding * 2, p.x - padding, this.texture.height - p.y - p.h - padding, undefined, undefined, this.texture, 1);
			return p.w;
		}
		return 0;
	}
}

const vertexShader = `
attribute vec3 vertex_position;         // unnormalized xy, word flag
attribute vec4 vertex_texCoord0;        // unnormalized texture space uv, normalized uv

varying vec4 uv0;
varying float wordFlag;

void main(void) {
		gl_Position = vec4(vertex_position.xy * 2.0 - 1.0, 0.5, 1.0);
		uv0 = vertex_texCoord0;
		wordFlag = vertex_position.z;
}`;
const fragmentShader$1 = `
varying vec4 uv0;
varying float wordFlag;

uniform vec4 clr;
uniform sampler2D graphTex;
uniform sampler2D wordsTex;

void main (void) {
		vec4 graphSample = texture2D(graphTex, uv0.xy);

		vec4 graph;
		if (uv0.w < graphSample.r)
				graph = vec4(0.7, 0.2, 0.2, 1.0);
		else if (uv0.w < graphSample.g)
				graph = vec4(0.2, 0.7, 0.2, 1.0);
		else if (uv0.w < graphSample.b)
				graph = vec4(0.2, 0.2, 0.7, 1.0);
		else
				graph = vec4(0.0, 0.0, 0.0, 1.0 - 0.25 * sin(uv0.w * 3.14159));

		vec4 words = texture2D(wordsTex, vec2(uv0.x, 1.0 - uv0.y));

		gl_FragColor = mix(graph, words, wordFlag) * clr;
}`;
class Render2d {
	constructor(device, maxQuads = 512) {
		const format = new VertexFormat(device, [{
			semantic: SEMANTIC_POSITION,
			components: 3,
			type: TYPE_FLOAT32
		}, {
			semantic: SEMANTIC_TEXCOORD0,
			components: 4,
			type: TYPE_FLOAT32
		}]);
		const indices = new Uint16Array(maxQuads * 6);
		for (let i = 0; i < maxQuads; ++i) {
			indices[i * 6 + 0] = i * 4;
			indices[i * 6 + 1] = i * 4 + 1;
			indices[i * 6 + 2] = i * 4 + 2;
			indices[i * 6 + 3] = i * 4;
			indices[i * 6 + 4] = i * 4 + 2;
			indices[i * 6 + 5] = i * 4 + 3;
		}
		const shader = shaderChunks.createShaderFromCode(device, vertexShader, fragmentShader$1, 'mini-stats');
		this.device = device;
		this.buffer = new VertexBuffer(device, format, maxQuads * 4, BUFFER_STREAM);
		this.data = new Float32Array(this.buffer.numBytes / 4);
		this.indexBuffer = new IndexBuffer(device, INDEXFORMAT_UINT16, maxQuads * 6, BUFFER_STATIC, indices);
		this.prim = {
			type: PRIMITIVE_TRIANGLES,
			indexed: true,
			base: 0,
			count: 0
		};
		this.quads = 0;
		this.mesh = new Mesh(device);
		this.mesh.vertexBuffer = this.buffer;
		this.mesh.indexBuffer[0] = this.indexBuffer;
		this.mesh.primitive = [this.prim];
		const material = new Material();
		this.material = material;
		material.cull = CULLFACE_NONE;
		material.shader = shader;
		material.depthState = DepthState.NODEPTH;
		material.blendState = new BlendState(true, BLENDEQUATION_ADD, BLENDMODE_SRC_ALPHA, BLENDMODE_ONE_MINUS_SRC_ALPHA, BLENDEQUATION_ADD, BLENDMODE_ONE, BLENDMODE_ONE);
		material.update();
		this.meshInstance = new MeshInstance(this.mesh, material, new GraphNode('MiniStatsMesh'));
		this.uniforms = {
			clr: new Float32Array(4)
		};
		this.targetSize = {
			width: device.width,
			height: device.height
		};
	}
	quad(x, y, w, h, u, v, uw, uh, texture, wordFlag = 0) {
		const rw = this.targetSize.width;
		const rh = this.targetSize.height;
		const x0 = x / rw;
		const y0 = y / rh;
		const x1 = (x + w) / rw;
		const y1 = (y + h) / rh;
		const tw = texture.width;
		const th = texture.height;
		const u0 = u / tw;
		const v0 = v / th;
		const u1 = (u + (uw != null ? uw : w)) / tw;
		const v1 = (v + (uh != null ? uh : h)) / th;
		this.data.set([x0, y0, wordFlag, u0, v0, 0, 0, x1, y0, wordFlag, u1, v0, 1, 0, x1, y1, wordFlag, u1, v1, 1, 1, x0, y1, wordFlag, u0, v1, 0, 1], 4 * 7 * this.quads);
		this.quads++;
		this.prim.count += 6;
	}
	startFrame() {
		this.quads = 0;
		this.prim.count = 0;
		this.targetSize.width = this.device.canvas.scrollWidth;
		this.targetSize.height = this.device.canvas.scrollHeight;
	}
	render(app, layer, graphTexture, wordsTexture, clr, height) {
		this.buffer.setData(this.data.buffer);
		this.uniforms.clr.set(clr, 0);
		this.material.setParameter('clr', this.uniforms.clr);
		this.material.setParameter('graphTex', graphTexture);
		this.material.setParameter('wordsTex', wordsTexture);
		app.drawMeshInstance(this.meshInstance, layer);
	}
}

class MiniStats {
	constructor(app, options) {
		const device = app.graphicsDevice;
		options = options || MiniStats.getDefaultOptions();
		this.initGraphs(app, device, options);
		const words = new Set(['', 'ms', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '.'].concat(this.graphs.map(graph => graph.name)).concat(options.stats ? options.stats.map(stat => stat.unitsName) : []).filter(item => !!item));
		this.wordAtlas = new WordAtlas(device, words);
		this.sizes = options.sizes;
		this._activeSizeIndex = options.startSizeIndex;
		const div = document.createElement('div');
		div.setAttribute('id', 'mini-stats');
		div.style.cssText = 'position:fixed;bottom:0;left:0;background:transparent;';
		document.body.appendChild(div);
		div.addEventListener('mouseenter', event => {
			this.opacity = 1.0;
		});
		div.addEventListener('mouseleave', event => {
			this.opacity = 0.7;
		});
		div.addEventListener('click', event => {
			event.preventDefault();
			if (this._enabled) {
				this.activeSizeIndex = (this.activeSizeIndex + 1) % this.sizes.length;
				this.resize(this.sizes[this.activeSizeIndex].width, this.sizes[this.activeSizeIndex].height, this.sizes[this.activeSizeIndex].graphs);
			}
		});
		device.on('resizecanvas', this.updateDiv, this);
		device.on('losecontext', this.loseContext, this);
		app.on('postrender', this.postRender, this);
		this.app = app;
		this.drawLayer = app.scene.layers.getLayerById(LAYERID_UI);
		this.device = device;
		this.render2d = new Render2d(device);
		this.div = div;
		this.width = 0;
		this.height = 0;
		this.gspacing = 2;
		this.clr = [1, 1, 1, 0.5];
		this._enabled = true;
		this.activeSizeIndex = this._activeSizeIndex;
	}
	destroy() {
		this.device.off('resizecanvas', this.updateDiv, this);
		this.device.off('losecontext', this.loseContext, this);
		this.app.off('postrender', this.postRender, this);
		this.graphs.forEach(graph => graph.destroy());
		this.wordAtlas.destroy();
		this.texture.destroy();
	}
	static getDefaultOptions() {
		return {
			sizes: [{
				width: 100,
				height: 16,
				spacing: 0,
				graphs: false
			}, {
				width: 128,
				height: 32,
				spacing: 2,
				graphs: true
			}, {
				width: 256,
				height: 64,
				spacing: 2,
				graphs: true
			}],
			startSizeIndex: 0,
			textRefreshRate: 500,
			cpu: {
				enabled: true,
				watermark: 33
			},
			gpu: {
				enabled: true,
				watermark: 33
			},
			stats: [{
				name: 'Frame',
				stats: ['frame.ms'],
				decimalPlaces: 1,
				unitsName: 'ms',
				watermark: 33
			}, {
				name: 'DrawCalls',
				stats: ['drawCalls.total'],
				watermark: 1000
			}]
		};
	}
	set activeSizeIndex(value) {
		this._activeSizeIndex = value;
		this.gspacing = this.sizes[value].spacing;
		this.resize(this.sizes[value].width, this.sizes[value].height, this.sizes[value].graphs);
	}
	get activeSizeIndex() {
		return this._activeSizeIndex;
	}
	set opacity(value) {
		this.clr[3] = value;
	}
	get opacity() {
		return this.clr[3];
	}
	get overallHeight() {
		const graphs = this.graphs;
		const spacing = this.gspacing;
		return this.height * graphs.length + spacing * (graphs.length - 1);
	}
	set enabled(value) {
		if (value !== this._enabled) {
			this._enabled = value;
			for (let i = 0; i < this.graphs.length; ++i) {
				this.graphs[i].enabled = value;
				this.graphs[i].timer.enabled = value;
			}
		}
	}
	get enabled() {
		return this._enabled;
	}
	initGraphs(app, device, options) {
		this.graphs = [];
		if (options.cpu.enabled) {
			const timer = new CpuTimer(app);
			const graph = new Graph('CPU', app, options.cpu.watermark, options.textRefreshRate, timer);
			this.graphs.push(graph);
		}
		if (options.gpu.enabled) {
			const timer = new GpuTimer(device);
			const graph = new Graph('GPU', app, options.gpu.watermark, options.textRefreshRate, timer);
			this.graphs.push(graph);
		}
		if (options.stats) {
			options.stats.forEach(entry => {
				const timer = new StatsTimer(app, entry.stats, entry.decimalPlaces, entry.unitsName, entry.multiplier);
				const graph = new Graph(entry.name, app, entry.watermark, options.textRefreshRate, timer);
				this.graphs.push(graph);
			});
		}
		const maxWidth = options.sizes.reduce((max, v) => {
			return v.width > max ? v.width : max;
		}, 0);
		this.texture = new Texture(device, {
			name: 'mini-stats-graph-texture',
			width: math.nextPowerOfTwo(maxWidth),
			height: math.nextPowerOfTwo(this.graphs.length),
			mipmaps: false,
			minFilter: FILTER_NEAREST,
			magFilter: FILTER_NEAREST,
			addressU: ADDRESS_REPEAT,
			addressV: ADDRESS_REPEAT
		});
		this.graphs.forEach((graph, i) => {
			graph.texture = this.texture;
			graph.yOffset = i;
		});
	}
	render() {
		const graphs = this.graphs;
		const wordAtlas = this.wordAtlas;
		const render2d = this.render2d;
		const width = this.width;
		const height = this.height;
		const gspacing = this.gspacing;
		render2d.startFrame();
		for (let i = 0; i < graphs.length; ++i) {
			const graph = graphs[i];
			let y = i * (height + gspacing);
			graph.render(render2d, 0, y, width, height);
			let x = 1;
			y += height - 13;
			x += wordAtlas.render(render2d, graph.name, x, y) + 10;
			const timingText = graph.timingText;
			for (let j = 0; j < timingText.length; ++j) {
				x += wordAtlas.render(render2d, timingText[j], x, y);
			}
			if (graph.timer.unitsName) {
				x += 3;
				wordAtlas.render(render2d, graph.timer.unitsName, x, y);
			}
		}
		render2d.render(this.app, this.drawLayer, this.texture, this.wordAtlas.texture, this.clr, height);
	}
	resize(width, height, showGraphs) {
		const graphs = this.graphs;
		for (let i = 0; i < graphs.length; ++i) {
			graphs[i].enabled = showGraphs;
		}
		this.width = width;
		this.height = height;
		this.updateDiv();
	}
	updateDiv() {
		const rect = this.device.canvas.getBoundingClientRect();
		this.div.style.left = rect.left + 'px';
		this.div.style.bottom = window.innerHeight - rect.bottom + 'px';
		this.div.style.width = this.width + 'px';
		this.div.style.height = this.overallHeight + 'px';
	}
	loseContext() {
		this.graphs.forEach(graph => graph.loseContext());
	}
	postRender() {
		if (this._enabled) {
			this.render();
		}
	}
}

const textureBlitVertexShader = `
		attribute vec2 vertex_position;
		varying vec2 uv0;
		void main(void) {
				gl_Position = vec4(vertex_position, 0.5, 1.0);
				uv0 = vertex_position.xy * 0.5 + 0.5;
		}`;
const textureBlitFragmentShader = `
		varying vec2 uv0;
		uniform sampler2D blitTexture;
		void main(void) {
				gl_FragColor = texture2D(blitTexture, uv0);
		}`;
class CoreExporter {
	constructor() {}
	textureToCanvas(texture, options = {}) {
		const image = texture.getSource();
		if (typeof HTMLImageElement !== 'undefined' && image instanceof HTMLImageElement || typeof HTMLCanvasElement !== 'undefined' && image instanceof HTMLCanvasElement || typeof OffscreenCanvas !== 'undefined' && image instanceof OffscreenCanvas || typeof ImageBitmap !== 'undefined' && image instanceof ImageBitmap) {
			const {
				width: _width,
				height: _height
			} = this.calcTextureSize(image.width, image.height, options.maxTextureSize);
			const _canvas = document.createElement('canvas');
			_canvas.width = _width;
			_canvas.height = _height;
			const context = _canvas.getContext('2d');
			context.drawImage(image, 0, 0, _canvas.width, _canvas.height);
			if (options.color) {
				const {
					r,
					g,
					b
				} = options.color;
				const imagedata = context.getImageData(0, 0, _width, _height);
				const data = imagedata.data;
				for (let i = 0; i < data.length; i += 4) {
					data[i + 0] = data[i + 0] * r;
					data[i + 1] = data[i + 1] * g;
					data[i + 2] = data[i + 2] * b;
				}
				context.putImageData(imagedata, 0, 0);
			}
			return Promise.resolve(_canvas);
		}
		const device = texture.device;
		const {
			width,
			height
		} = this.calcTextureSize(texture.width, texture.height, options.maxTextureSize);
		const dstTexture = new Texture(device, {
			name: 'ExtractedTexture',
			width,
			height,
			format: texture.format,
			cubemap: false,
			mipmaps: false,
			minFilter: FILTER_LINEAR,
			magFilter: FILTER_LINEAR,
			addressU: ADDRESS_CLAMP_TO_EDGE,
			addressV: ADDRESS_CLAMP_TO_EDGE
		});
		const renderTarget = new RenderTarget({
			colorBuffer: dstTexture,
			depth: false
		});
		const shader = createShaderFromCode(device, textureBlitVertexShader, textureBlitFragmentShader, 'ShaderCoreExporterBlit');
		device.scope.resolve('blitTexture').setValue(texture);
		device.setBlendState(BlendState.NOBLEND);
		drawQuadWithShader(device, renderTarget, shader);
		const pixels = new Uint8ClampedArray(width * height * 4);
		device.readPixels(0, 0, width, height, pixels);
		dstTexture.destroy();
		renderTarget.destroy();
		const newImage = new ImageData(pixels, width, height);
		const canvas = document.createElement('canvas');
		canvas.width = width;
		canvas.height = height;
		const newContext = canvas.getContext('2d');
		newContext.putImageData(newImage, 0, 0);
		return Promise.resolve(canvas);
	}
	calcTextureSize(width, height, maxTextureSize) {
		if (maxTextureSize) {
			const scale = Math.min(maxTextureSize / Math.max(width, height), 1);
			width = Math.round(width * scale);
			height = Math.round(height * scale);
		}
		return {
			width,
			height
		};
	}
}

const ROOT_FILE_NAME = 'root';
const header = `#usda 1.0
(
		customLayerData = {
				string creator = "PlayCanvas UsdzExporter"
		}
		metersPerUnit = 1
		upAxis = "Y"
)
`;
const materialListTemplate = materials => `
def "Materials"
{
		${materials.join('\n')}
}
`;
const meshTemplate = (faceVertexCounts, indices, normals, positions, uv0, uv1) => `
def "Mesh"
{
		def Mesh "Mesh"
		{
				int[] faceVertexCounts = [${faceVertexCounts}]
				int[] faceVertexIndices = [${indices}]
				normal3f[] normals = [${normals}] (
						interpolation = "vertex"
				)
				point3f[] points = [${positions}]
				texCoord2f[] primvars:st = [${uv0}] (
						interpolation = "vertex"
				)
				texCoord2f[] primvars:st1 = [${uv1}] (
						interpolation = "vertex"
				)
				uniform token subdivisionScheme = "none"
		}
}
`;
const meshInstanceTemplate = (nodeName, meshRefPath, worldMatrix, materialRefPath) => `
def Xform "${nodeName}" (
		prepend references = ${meshRefPath}
)
{
		matrix4d xformOp:transform = ${worldMatrix}
		uniform token[] xformOpOrder = ["xformOp:transform"]

		rel material:binding = ${materialRefPath}
}
`;
const materialValueTemplate = (type, name, value) => `                    ${type} inputs:${name} = ${value}`;
class UsdzExporter extends CoreExporter {
	constructor(...args) {
		super(...args);
		this.meshMap = void 0;
		this.materialMap = void 0;
		this.materials = void 0;
		this.textureMap = void 0;
		this.nodeNames = void 0;
		this.files = void 0;
	}
	init() {
		this.meshMap = new Map();
		this.textureMap = new Map();
		this.materialMap = new Map();
		this.materials = [];
		this.files = {};
		this.nodeNames = new Set();
	}
	done() {
		this.meshMap = null;
		this.textureMap = null;
		this.materialMap = null;
		this.materials = null;
		this.files = null;
		this.nodeNames = null;
	}
	build(entity, options = {}) {
		this.init();
		this.addFile(null, ROOT_FILE_NAME);
		const allMeshInstances = [];
		if (entity) {
			const renders = entity.findComponents("render");
			renders.forEach(render => {
				allMeshInstances.push(...render.meshInstances);
			});
		}
		let rootContent = '';
		allMeshInstances.forEach(meshInstance => {
			rootContent += this.buildMeshInstance(meshInstance);
		});
		rootContent += materialListTemplate(this.materials);
		this.addFile(null, ROOT_FILE_NAME, '', rootContent);
		const textureOptions = {
			maxTextureSize: options.maxTextureSize
		};
		const textureArray = Array.from(this.textureMap.keys());
		const promises = [];
		for (let i = 0; i < textureArray.length; i++) {
			const mimeType = 'image/png' ;
			const texture = textureArray[i];
			const texturePromise = this.textureToCanvas(texture, textureOptions).then(canvas => {
				if (canvas) {
					return new Promise(resolve => canvas.toBlob(resolve, mimeType, 1)).then(blob => blob.arrayBuffer());
				}
				console.warn(`Export of texture ${texture.name} is not currently supported.`);
				return new Promise(resolve => resolve(null));
			});
			promises.push(texturePromise);
		}
		const finalData = Promise.all(promises).then(values => {
			values.forEach((textureArrayBuffer, index) => {
				const texture = textureArray[index];
				const ids = this.getTextureFileIds(texture);
				this.files[ids.fileName] = new Uint8Array(textureArrayBuffer);
			});
			this.alignFiles();
			const arraybuffer = zipSync(this.files, {
				level: 0
			});
			this.done();
			return arraybuffer;
		});
		return finalData;
	}
	alignFiles() {
		let offset = 0;
		for (const filename in this.files) {
			const file = this.files[filename];
			const headerSize = 34 + filename.length;
			offset += headerSize;
			const offsetMod64 = offset & 63;
			if (offsetMod64 !== 4) {
				const padLength = 64 - offsetMod64;
				const padding = new Uint8Array(padLength);
				this.files[filename] = [file, {
					extra: {
						12345: padding
					}
				}];
			}
			offset = file.length;
		}
	}
	getFileIds(category, name, ref, extension = 'usda') {
		const fileName = (category ? `${category}/` : '') + `${name}.${extension}`;
		const refName = `@./${fileName}@</${ref}>`;
		return {
			name,
			fileName,
			refName
		};
	}
	getTextureFileIds(texture) {
		return this.getFileIds('texture', `Texture_${texture.id}`, 'Texture', 'png');
	}
	addFile(category, uniqueId, refName = '', content = null) {
		let contentU8 = null;
		if (content) {
			content = header + '\n' + content;
			contentU8 = strToU8(content);
		}
		const ids = this.getFileIds(category, uniqueId, refName);
		this.files[ids.fileName] = contentU8;
		return ids.refName;
	}
	getMaterialRef(material) {
		let materialRef = this.materialMap.get(material);
		if (!materialRef) {
			materialRef = this.buildMaterial(material);
			this.materialMap.set(material, materialRef);
		}
		return materialRef;
	}
	getMeshRef(mesh) {
		let meshRef = this.meshMap.get(mesh);
		if (!meshRef) {
			meshRef = this.buildMesh(mesh);
			this.meshMap.set(mesh, meshRef);
		}
		return meshRef;
	}
	buildArray2(array) {
		const components = [];
		const count = array.length;
		for (let i = 0; i < count; i += 2) {
			components.push(`(${array[i]}, ${1 - array[i + 1]})`);
		}
		return components.join(', ');
	}
	buildArray3(array) {
		const components = [];
		const count = array.length;
		for (let i = 0; i < count; i += 3) {
			components.push(`(${array[i]}, ${array[i + 1]}, ${array[i + 2]})`);
		}
		return components.join(', ');
	}
	buildMat4(mat) {
		const data = mat.data;
		const vectors = [];
		for (let i = 0; i < 16; i += 4) {
			vectors.push(`(${data[i]}, ${data[i + 1]}, ${data[i + 2]}, ${data[i + 3]})`);
		}
		return `( ${vectors.join(', ')} )`;
	}
	buildMaterial(material) {
		const materialName = `Material_${material.id}`;
		const materialPath = `/Materials/${materialName}`;
		const materialPropertyPath = property => `<${materialPath}${property}>`;
		const buildTexture = (texture, textureIds, mapType, uvChannel, tiling, offset, rotation, tintColor) => {
			return `
								def Shader "Transform2d_${mapType}" (
										sdrMetadata = {
												string role = "math"
										}
								)
								{
										uniform token info:id = "UsdTransform2d"
										float2 inputs:in.connect = ${materialPropertyPath(`/uvReader_${uvChannel}.outputs:result`)}
										float inputs:rotation = ${rotation}
										float2 inputs:scale = (${tiling.x}, ${tiling.y})
										float2 inputs:translation = (${offset.x}, ${offset.y})
										float2 outputs:result
								}

								def Shader "Texture_${texture.id}_${mapType}"
								{
										uniform token info:id = "UsdUVTexture"
										asset inputs:file = @${textureIds.fileName}@
										float2 inputs:st.connect = ${materialPropertyPath(`/Transform2d_${mapType}.outputs:result`)}
										token inputs:wrapS = "repeat"
										token inputs:wrapT = "repeat"
										float4 inputs:scale = (${tintColor.r}, ${tintColor.g}, ${tintColor.b}, ${tintColor.a})
										float outputs:r
										float outputs:g
										float outputs:b
										float3 outputs:rgb
										float outputs:a
								}
						`;
		};
		const inputs = [];
		const samplers = [];
		const addTexture = (textureSlot, uniform, propType, propName, valueName, handleOpacity = false, tintTexture = false) => {
			const texture = material[textureSlot];
			if (texture) {
				const textureIds = this.getTextureFileIds(texture);
				this.textureMap.set(texture, textureIds.refName);
				const channel = material[textureSlot + 'Channel'] || 'rgb';
				const textureValue = materialPropertyPath(`/${textureIds.name}_${valueName}.outputs:${channel}`);
				inputs.push(materialValueTemplate(propType, `${propName}.connect`, textureValue));
				if (handleOpacity) {
					if (material.alphaTest > 0.0) ;
				}
				const tiling = material[textureSlot + 'Tiling'];
				const offset = material[textureSlot + 'Offset'];
				const rotation = material[textureSlot + 'Rotation'];
				const uvChannel = material[textureSlot + 'Uv'] === 1 ? 'st1' : 'st';
				const tintColor = tintTexture && uniform ? uniform : Color.WHITE;
				samplers.push(buildTexture(texture, textureIds, valueName, uvChannel, tiling, offset, rotation, tintColor));
			} else if (uniform) {
				const value = propType === 'float' ? `${uniform}` : `(${uniform.r}, ${uniform.g}, ${uniform.b})`;
				inputs.push(materialValueTemplate(propType, propName, value));
			}
		};
		addTexture('diffuseMap', material.diffuse, 'color3f', 'diffuseColor', 'diffuse', false, true);
		if (material.transparent || material.alphaTest > 0.0) {
			addTexture('opacityMap', material.opacity, 'float', 'opacity', 'opacity', true);
		}
		addTexture('normalMap', null, 'normal3f', 'normal', 'normal');
		addTexture('emissiveMap', material.emissive, 'color3f', 'emissiveColor', 'emissive', false, true);
		addTexture('aoMap', null, 'float', 'occlusion', 'occlusion');
		addTexture('metalnessMap', material.metalness, 'float', 'metallic', 'metallic');
		addTexture('glossMap', material.gloss, 'float', 'roughness', 'roughness');
		const materialObject = `
						def Material "${materialName}"
						{
								def Shader "PreviewSurface"
								{
										uniform token info:id = "UsdPreviewSurface"
${inputs.join('\n')}
										int inputs:useSpecularWorkflow = 0
										token outputs:surface
								}

								token outputs:surface.connect = ${materialPropertyPath('/PreviewSurface.outputs:surface')}

								def Shader "uvReader_st"
								{
										uniform token info:id = "UsdPrimvarReader_float2"
										token inputs:varname = "st"
										float2 inputs:fallback = (0.0, 0.0)
										float2 outputs:result
								}

								def Shader "uvReader_st1"
								{
										uniform token info:id = "UsdPrimvarReader_float2"
										token inputs:varname = "st1"
										float2 inputs:fallback = (0.0, 0.0)
										float2 outputs:result
								}

								${samplers.join('\n')}
						}
				`;
		this.materials.push(materialObject);
		return materialPropertyPath('');
	}
	buildMesh(mesh) {
		let positions = [];
		const indices = [];
		let normals = [];
		let uv0 = [];
		let uv1 = [];
		mesh.getVertexStream(SEMANTIC_POSITION, positions);
		mesh.getVertexStream(SEMANTIC_NORMAL, normals);
		mesh.getVertexStream(SEMANTIC_TEXCOORD0, uv0);
		mesh.getVertexStream(SEMANTIC_TEXCOORD1, uv1);
		mesh.getIndices(indices);
		const indicesCount = indices.length || positions.length;
		const faceVertexCounts = Array(indicesCount / 3).fill(3).join(', ');
		if (!indices.length) {
			for (let i = 0; i < indicesCount; i++) indices[i] = i;
		}
		const numVerts = positions.length / 3;
		normals = normals.length ? normals : Array(numVerts * 3).fill(0);
		uv0 = uv0.length ? uv0 : Array(numVerts * 2).fill(0);
		uv1 = uv1.length ? uv1 : Array(numVerts * 2).fill(0);
		positions = this.buildArray3(positions);
		normals = this.buildArray3(normals);
		uv0 = this.buildArray2(uv0);
		uv1 = this.buildArray2(uv1);
		const meshObject = meshTemplate(faceVertexCounts, indices, normals, positions, uv0, uv1);
		const refPath = this.addFile('mesh', `Mesh_${mesh.id}`, 'Mesh', meshObject);
		return refPath;
	}
	buildMeshInstance(meshInstance) {
		const meshRefPath = this.getMeshRef(meshInstance.mesh);
		const materialRefPath = this.getMaterialRef(meshInstance.material);
		const worldMatrix = this.buildMat4(meshInstance.node.getWorldTransform());
		const name = meshInstance.node.name.replace(/[^a-z0-9]/gi, '_');
		let nodeName = name;
		while (this.nodeNames.has(nodeName)) {
			nodeName = `${name}_${Math.random().toString(36).slice(2, 7)}`;
		}
		this.nodeNames.add(nodeName);
		return meshInstanceTemplate(nodeName, meshRefPath, worldMatrix, materialRefPath);
	}
}

const ARRAY_BUFFER = 34962;
const ELEMENT_ARRAY_BUFFER = 34963;
const getIndexComponentType = indexFormat => {
	switch (indexFormat) {
		case INDEXFORMAT_UINT8:
			return 5121;
		case INDEXFORMAT_UINT16:
			return 5123;
		case INDEXFORMAT_UINT32:
			return 5125;
	}
	return 0;
};
const getComponentType = dataType => {
	switch (dataType) {
		case TYPE_INT8:
			return 5120;
		case TYPE_UINT8:
			return 5121;
		case TYPE_INT16:
			return 5122;
		case TYPE_UINT16:
			return 5123;
		case TYPE_INT32:
			return 5124;
		case TYPE_UINT32:
			return 5125;
		case TYPE_FLOAT32:
			return 5126;
	}
	return 0;
};
const getAccessorType = componentCount => {
	switch (componentCount) {
		case 1:
			return 'SCALAR';
		case 2:
			return 'VEC2';
		case 3:
			return 'VEC3';
		case 4:
			return 'VEC4';
	}
	return 0;
};
const getSemantic = engineSemantic => {
	switch (engineSemantic) {
		case SEMANTIC_POSITION:
			return 'POSITION';
		case SEMANTIC_NORMAL:
			return 'NORMAL';
		case SEMANTIC_TANGENT:
			return 'TANGENT';
		case SEMANTIC_COLOR:
			return 'COLOR_0';
		case SEMANTIC_BLENDINDICES:
			return 'JOINTS_0';
		case SEMANTIC_BLENDWEIGHT:
			return 'WEIGHTS_0';
		case SEMANTIC_TEXCOORD0:
			return 'TEXCOORD_0';
		case SEMANTIC_TEXCOORD1:
			return 'TEXCOORD_1';
		case SEMANTIC_TEXCOORD2:
			return 'TEXCOORD_2';
		case SEMANTIC_TEXCOORD3:
			return 'TEXCOORD_3';
		case SEMANTIC_TEXCOORD4:
			return 'TEXCOORD_4';
		case SEMANTIC_TEXCOORD5:
			return 'TEXCOORD_5';
		case SEMANTIC_TEXCOORD6:
			return 'TEXCOORD_6';
		case SEMANTIC_TEXCOORD7:
			return 'TEXCOORD_7';
	}
};
const getFilter = function getFilter(filter) {
	switch (filter) {
		case FILTER_NEAREST:
			return 9728;
		case FILTER_LINEAR:
			return 9729;
		case FILTER_NEAREST_MIPMAP_NEAREST:
			return 9984;
		case FILTER_LINEAR_MIPMAP_NEAREST:
			return 9985;
		case FILTER_NEAREST_MIPMAP_LINEAR:
			return 9986;
		case FILTER_LINEAR_MIPMAP_LINEAR:
			return 9987;
	}
};
const getWrap = function getWrap(wrap) {
	switch (wrap) {
		case ADDRESS_CLAMP_TO_EDGE:
			return 33071;
		case ADDRESS_MIRRORED_REPEAT:
			return 33648;
		case ADDRESS_REPEAT:
			return 10497;
	}
};
function isCanvasTransparent(canvas) {
	const context = canvas.getContext('2d');
	const pixelData = context.getImageData(0, 0, canvas.width, canvas.height).data;
	for (let i = 3; i < pixelData.length; i += 4) {
		if (pixelData[i] < 255) {
			return true;
		}
	}
	return false;
}
const textureSemantics = ['diffuseMap', 'colorMap', 'normalMap', 'metalnessMap', 'emissiveMap'];
class GltfExporter extends CoreExporter {
	collectResources(root) {
		const resources = {
			buffers: [],
			cameras: [],
			entities: [],
			materials: [],
			textures: [],
			entityMeshInstances: [],
			bufferViewMap: new Map(),
			compressableTexture: new Set()
		};
		const {
			materials,
			buffers,
			entityMeshInstances,
			textures
		} = resources;
		root.forEach(entity => {
			resources.entities.push(entity);
		});
		const collectMeshInstances = meshInstances => {
			meshInstances.forEach(meshInstance => {
				const material = meshInstance.material;
				if (materials.indexOf(material) < 0) {
					resources.materials.push(material);
					textureSemantics.forEach(semantic => {
						const texture = material[semantic];
						if (texture && textures.indexOf(texture) < 0) {
							if (semantic !== 'normalMap') {
								resources.compressableTexture.add(texture);
							}
							textures.push(texture);
						}
					});
				}
				const node = meshInstance.node;
				let nodeMeshInstances = entityMeshInstances.find(e => e.node === node);
				if (!nodeMeshInstances) {
					nodeMeshInstances = {
						node: node,
						meshInstances: []
					};
					entityMeshInstances.push(nodeMeshInstances);
				}
				nodeMeshInstances.meshInstances.push(meshInstance);
				const mesh = meshInstance.mesh;
				const vertexBuffer = mesh.vertexBuffer;
				if (buffers.indexOf(vertexBuffer) < 0) {
					buffers.unshift(vertexBuffer);
				}
				const indexBuffer = mesh.indexBuffer[0];
				if (buffers.indexOf(indexBuffer) < 0) {
					buffers.push(indexBuffer);
				}
			});
		};
		resources.entities.forEach(entity => {
			if (entity.camera) {
				resources.cameras.push(entity.camera);
			}
			if (entity.render && entity.render.enabled) {
				collectMeshInstances(entity.render.meshInstances);
			}
			if (entity.model && entity.model.enabled && entity.model.meshInstances) {
				collectMeshInstances(entity.model.meshInstances);
			}
		});
		return resources;
	}
	writeBufferViews(resources, json) {
		json.bufferViews = [];
		for (const buffer of resources.buffers) {
			GltfExporter.writeBufferView(resources, json, buffer);
		}
	}
	static writeBufferView(resources, json, buffer) {
		var _json$buffers, _json$buffers$;
		json.buffers = (_json$buffers = json.buffers) != null ? _json$buffers : [];
		json.buffers[0] = (_json$buffers$ = json.buffers[0]) != null ? _json$buffers$ : {
			byteLength: 0
		};
		const bufferInfo = json.buffers[0];
		bufferInfo.byteLength = math.roundUp(bufferInfo.byteLength, 4);
		const offset = bufferInfo.byteLength;
		const addBufferView = (target, byteLength, byteOffset, byteStride) => {
			const bufferView = {
				target: target,
				buffer: 0,
				byteLength: byteLength,
				byteOffset: byteOffset,
				byteStride: byteStride
			};
			return json.bufferViews.push(bufferView) - 1;
		};
		let arrayBuffer;
		if (buffer instanceof VertexBuffer) {
			arrayBuffer = buffer.lock();
			const format = buffer.getFormat();
			if (format.interleaved) {
				const bufferViewIndex = addBufferView(ARRAY_BUFFER, arrayBuffer.byteLength, offset, format.size);
				resources.bufferViewMap.set(buffer, [bufferViewIndex]);
			} else {
				const bufferViewIndices = [];
				for (const element of format.elements) {
					const bufferViewIndex = addBufferView(ARRAY_BUFFER, element.size * format.vertexCount, offset + element.offset, element.size);
					bufferViewIndices.push(bufferViewIndex);
				}
				resources.bufferViewMap.set(buffer, bufferViewIndices);
			}
		} else if (buffer instanceof IndexBuffer) {
			arrayBuffer = buffer.lock();
			const bufferViewIndex = addBufferView(ARRAY_BUFFER, arrayBuffer.byteLength, offset);
			resources.bufferViewMap.set(buffer, [bufferViewIndex]);
		} else {
			arrayBuffer = buffer;
			const bufferViewIndex = addBufferView(ELEMENT_ARRAY_BUFFER, arrayBuffer.byteLength, offset);
			resources.bufferViewMap.set(buffer, [bufferViewIndex]);
		}
		bufferInfo.byteLength += arrayBuffer.byteLength;
	}
	writeCameras(resources, json) {
		if (resources.cameras.length > 0) {
			json.cameras = resources.cameras.map(cam => {
				const projection = cam.projection;
				const nearClip = cam.nearClip;
				const farClip = cam.farClip;
				const camera = {};
				if (projection === PROJECTION_ORTHOGRAPHIC) {
					camera.type = 'orthographic';
					camera.orthographic = {
						xmag: 1,
						ymag: 1,
						znear: nearClip,
						zfar: farClip
					};
				} else {
					const fov = cam.fov;
					camera.type = 'perspective';
					camera.perspective = {
						yfov: fov * Math.PI / 180,
						znear: nearClip,
						zfar: farClip
					};
				}
				return camera;
			});
		}
	}
	attachTexture(resources, material, destination, name, textureSemantic, json) {
		const texture = material[textureSemantic];
		if (texture) {
			const textureIndex = resources.textures.indexOf(texture);
			if (textureIndex < 0) console.warn(`Texture ${texture.name} wasn't collected.`);
			destination[name] = {
				index: textureIndex
			};
			const scale = material[`${textureSemantic}Tiling`];
			const offset = material[`${textureSemantic}Offset`];
			const rotation = material[`${textureSemantic}Rotation`];
			if (scale && !scale.equals(Vec2.ONE) || offset && !offset.equals(Vec2.ZERO) || rotation !== 0) {
				var _json$extensionsUsed, _json$extensionsRequi;
				destination[name].extensions = {
					KHR_texture_transform: {}
				};
				json.extensionsUsed = (_json$extensionsUsed = json.extensionsUsed) != null ? _json$extensionsUsed : [];
				if (json.extensionsUsed.indexOf('KHR_texture_transform') < 0) {
					json.extensionsUsed.push('KHR_texture_transform');
				}
				json.extensionsRequired = (_json$extensionsRequi = json.extensionsRequired) != null ? _json$extensionsRequi : [];
				if (json.extensionsRequired.indexOf('KHR_texture_transform') < 0) {
					json.extensionsRequired.push('KHR_texture_transform');
				}
				if (scale && !scale.equals(Vec2.ONE)) {
					destination[name].extensions.KHR_texture_transform.scale = [scale.x, scale.y];
				}
				if (offset && !offset.equals(Vec2.ZERO)) {
					destination[name].extensions.KHR_texture_transform.offset = [offset.x, offset.y - 1 + scale.y];
				}
				if (rotation !== 0) {
					destination[name].extensions.KHR_texture_transform.rotation = rotation * math.DEG_TO_RAD;
				}
			}
		}
	}
	writeStandardMaterial(resources, mat, output, json) {
		const {
			diffuse,
			emissive,
			opacity,
			metalness,
			gloss,
			glossInvert
		} = mat;
		const pbr = output.pbrMetallicRoughness;
		if (!diffuse.equals(Color.WHITE) || opacity !== 1) {
			pbr.baseColorFactor = [diffuse.r, diffuse.g, diffuse.b, opacity];
		}
		if (metalness !== 1) {
			pbr.metallicFactor = metalness;
		}
		const roughness = glossInvert ? gloss : 1 - gloss;
		if (roughness !== 1) {
			pbr.roughnessFactor = roughness;
		}
		this.attachTexture(resources, mat, pbr, 'baseColorTexture', 'diffuseMap', json);
		this.attachTexture(resources, mat, pbr, 'metallicRoughnessTexture', 'metalnessMap', json);
		if (!emissive.equals(Color.BLACK)) {
			output.emissiveFactor = [emissive.r, emissive.g, emissive.b];
		}
	}
	writeBasicMaterial(resources, mat, output, json) {
		const {
			color
		} = mat;
		const pbr = output.pbrMetallicRoughness;
		if (!color.equals(Color.WHITE)) {
			pbr.baseColorFactor = [color.r, color.g, color.b, color];
		}
		this.attachTexture(resources, mat, pbr, 'baseColorTexture', 'colorMap', json);
	}
	writeMaterials(resources, json) {
		if (resources.materials.length > 0) {
			json.materials = resources.materials.map(mat => {
				const {
					name,
					blendType,
					cull,
					alphaTest
				} = mat;
				const material = {
					pbrMetallicRoughness: {}
				};
				if (name && name.length > 0) {
					material.name = name;
				}
				if (mat instanceof StandardMaterial) {
					this.writeStandardMaterial(resources, mat, material, json);
				}
				if (mat instanceof BasicMaterial) {
					this.writeBasicMaterial(resources, mat, material, json);
				}
				if (blendType === BLEND_NORMAL) {
					material.alphaMode = 'BLEND';
				} else if (blendType === BLEND_NONE) {
					if (alphaTest !== 0) {
						material.alphaMode = 'MASK';
						material.alphaCutoff = alphaTest;
					}
				}
				if (cull === CULLFACE_NONE) {
					material.doubleSided = true;
				}
				this.attachTexture(resources, mat, material, 'normalTexture', 'normalMap', json);
				this.attachTexture(resources, mat, material, 'occlusionTexture', 'aoMap', json);
				this.attachTexture(resources, mat, material, 'emissiveTexture', 'emissiveMap', json);
				return material;
			});
		}
	}
	writeNodes(resources, json) {
		if (resources.entities.length > 0) {
			json.nodes = resources.entities.map(entity => {
				const name = entity.name;
				const t = entity.getLocalPosition();
				const r = entity.getLocalRotation();
				const s = entity.getLocalScale();
				const node = {};
				if (name && name.length > 0) {
					node.name = name;
				}
				if (!t.equals(Vec3.ZERO)) {
					node.translation = [t.x, t.y, t.z];
				}
				if (!r.equals(Quat.IDENTITY)) {
					node.rotation = [r.x, r.y, r.z, r.w];
				}
				if (!s.equals(Vec3.ONE)) {
					node.scale = [s.x, s.y, s.z];
				}
				if (entity.camera && entity.camera.enabled) {
					node.camera = resources.cameras.indexOf(entity.camera);
				}
				const entityMeshInstance = resources.entityMeshInstances.find(e => e.node === entity);
				if (entityMeshInstance) {
					node.mesh = resources.entityMeshInstances.indexOf(entityMeshInstance);
				}
				if (entity.children.length > 0) {
					node.children = [];
					entity.children.forEach(child => {
						node.children.push(resources.entities.indexOf(child));
					});
				}
				return node;
			});
		}
	}
	writeMeshes(resources, json) {
		if (resources.entityMeshInstances.length > 0) {
			json.accessors = [];
			json.meshes = [];
			resources.entityMeshInstances.forEach(entityMeshInstances => {
				const mesh = {
					primitives: []
				};
				const meshInstances = entityMeshInstances.meshInstances;
				meshInstances.forEach(meshInstance => {
					const primitive = GltfExporter.createPrimitive(resources, json, meshInstance.mesh);
					primitive.material = resources.materials.indexOf(meshInstance.material);
					mesh.primitives.push(primitive);
				});
				json.meshes.push(mesh);
			});
		}
	}
	static createPrimitive(resources, json, mesh) {
		const primitive = {
			attributes: {}
		};
		const {
			vertexBuffer
		} = mesh;
		const {
			format
		} = vertexBuffer;
		const {
			interleaved,
			elements
		} = format;
		const numVertices = vertexBuffer.getNumVertices();
		elements.forEach((element, elementIndex) => {
			let bufferView = resources.bufferViewMap.get(vertexBuffer);
			if (!bufferView) {
				GltfExporter.writeBufferView(resources, json, vertexBuffer);
				resources.buffers.push(vertexBuffer);
				bufferView = resources.bufferViewMap.get(vertexBuffer);
			}
			const viewIndex = bufferView[interleaved ? 0 : elementIndex];
			const accessor = {
				bufferView: viewIndex,
				byteOffset: interleaved ? element.offset : 0,
				componentType: getComponentType(element.dataType),
				type: getAccessorType(element.numComponents),
				count: numVertices
			};
			const idx = json.accessors.push(accessor) - 1;
			primitive.attributes[getSemantic(element.name)] = idx;
			if (element.name === SEMANTIC_POSITION) {
				const positions = [];
				mesh.getPositions(positions);
				const min = new Vec3();
				const max = new Vec3();
				BoundingBox.computeMinMax(positions, min, max);
				accessor.min = [min.x, min.y, min.z];
				accessor.max = [max.x, max.y, max.z];
			}
		});
		const indexBuffer = mesh.indexBuffer[0];
		if (indexBuffer) {
			let bufferView = resources.bufferViewMap.get(indexBuffer);
			if (!bufferView) {
				GltfExporter.writeBufferView(resources, json, indexBuffer);
				resources.buffers.push(indexBuffer);
				bufferView = resources.bufferViewMap.get(indexBuffer);
			}
			const viewIndex = bufferView[0];
			const accessor = {
				bufferView: viewIndex,
				componentType: getIndexComponentType(indexBuffer.getFormat()),
				count: indexBuffer.getNumIndices(),
				type: 'SCALAR'
			};
			const idx = json.accessors.push(accessor) - 1;
			primitive.indices = idx;
		}
		return primitive;
	}
	convertTextures(srcTextures, options) {
		const textureOptions = {
			maxTextureSize: options.maxTextureSize
		};
		const promises = [];
		srcTextures.forEach(srcTexture => {
			const promise = this.textureToCanvas(srcTexture, textureOptions);
			promise.then(canvas => {
				return new Promise(resolve => resolve(canvas));
			});
			promises.push(promise);
		});
		return promises;
	}
	writeTextures(resources, textureCanvases, json, options) {
		const textures = resources.textures;
		const promises = [];
		for (let i = 0; i < textureCanvases.length; i++) {
			const texture = textures[i];
			const canvas = textureCanvases[i];
			const isRGBA = isCanvasTransparent(canvas) || !resources.compressableTexture.has(texture);
			const mimeType = isRGBA ? 'image/png' : 'image/jpeg';
			promises.push(this.getBlob(canvas, mimeType).then(blob => {
				const reader = new FileReader();
				reader.readAsArrayBuffer(blob);
				return new Promise(resolve => {
					reader.onloadend = () => {
						resolve(reader);
					};
				});
			}).then(reader => {
				const buffer = this.getPaddedArrayBuffer(reader.result);
				GltfExporter.writeBufferView(resources, json, buffer);
				resources.buffers.push(buffer);
				const bufferView = resources.bufferViewMap.get(buffer);
				json.images[i] = {
					mimeType: mimeType,
					bufferView: bufferView[0]
				};
				json.samplers[i] = {
					minFilter: getFilter(texture.minFilter),
					magFilter: getFilter(texture.magFilter),
					wrapS: getWrap(texture.addressU),
					wrapT: getWrap(texture.addressV)
				};
				json.textures[i] = {
					sampler: i,
					source: i
				};
			}));
		}
		return Promise.all(promises);
	}
	getBlob(canvas, mimeType) {
		if (canvas.toBlob !== undefined) {
			return new Promise(resolve => {
				canvas.toBlob(resolve, mimeType);
			});
		}
		let quality = 1.0;
		if (mimeType === 'image/jpeg') {
			quality = 0.92;
		}
		return canvas.convertToBlob({
			type: mimeType,
			quality: quality
		});
	}
	getPaddedArrayBuffer(arrayBuffer, paddingByte = 0) {
		const paddedLength = math.roundUp(arrayBuffer.byteLength, 4);
		if (paddedLength !== arrayBuffer.byteLength) {
			const array = new Uint8Array(paddedLength);
			array.set(new Uint8Array(arrayBuffer));
			if (paddingByte !== 0) {
				for (let i = arrayBuffer.byteLength; i < paddedLength; i++) {
					array[i] = paddingByte;
				}
			}
			return array.buffer;
		}
		return arrayBuffer;
	}
	buildJson(resources, options) {
		var _this = this;
		const promises = this.convertTextures(resources.textures, options);
		return Promise.all(promises).then(async function (textureCanvases) {
			const json = {
				asset: {
					version: '2.0',
					generator: 'PlayCanvas GltfExporter'
				},
				scenes: [{
					nodes: [0]
				}],
				images: [],
				samplers: [],
				textures: [],
				scene: 0
			};
			_this.writeBufferViews(resources, json);
			_this.writeCameras(resources, json);
			_this.writeMeshes(resources, json);
			_this.writeMaterials(resources, json);
			_this.writeNodes(resources, json, options);
			await _this.writeTextures(resources, textureCanvases, json, options);
			if (!json.images.length) delete json.images;
			if (!json.samplers.length) delete json.samplers;
			if (!json.textures.length) delete json.textures;
			return json;
		});
	}
	build(entity, options = {}) {
		const resources = this.collectResources(entity);
		return this.buildJson(resources, options).then(json => {
			const jsonText = JSON.stringify(json);
			const headerLength = 12;
			const jsonHeaderLength = 8;
			const jsonDataLength = jsonText.length;
			const jsonPaddingLength = 4 - (jsonDataLength & 3) & 3;
			const binaryHeaderLength = 8;
			const binaryDataLength = json.buffers.reduce((total, buffer) => math.roundUp(total + buffer.byteLength, 4), 0);
			let totalLength = headerLength + jsonHeaderLength + jsonDataLength + jsonPaddingLength;
			if (binaryDataLength > 0) {
				totalLength += binaryHeaderLength + binaryDataLength;
			}
			const glbBuffer = new ArrayBuffer(totalLength);
			const glbView = new DataView(glbBuffer);
			glbView.setUint32(0, 0x46546C67, true);
			glbView.setUint32(4, 2, true);
			glbView.setUint32(8, totalLength, true);
			glbView.setUint32(12, jsonDataLength + jsonPaddingLength, true);
			glbView.setUint32(16, 0x4E4F534A, true);
			let offset = headerLength + jsonHeaderLength;
			for (let i = 0; i < jsonDataLength; i++) {
				glbView.setUint8(offset + i, jsonText.charCodeAt(i));
			}
			offset += jsonDataLength;
			for (let i = 0; i < jsonPaddingLength; i++) {
				glbView.setUint8(offset + i, 0x20);
			}
			offset += jsonPaddingLength;
			if (binaryDataLength > 0) {
				glbView.setUint32(offset, binaryDataLength, true);
				glbView.setUint32(offset + 4, 0x004E4942, true);
				offset += binaryHeaderLength;
				resources.buffers.forEach(buffer => {
					let src;
					const bufferViewId = resources.bufferViewMap.get(buffer)[0];
					const bufferOffset = json.bufferViews[bufferViewId].byteOffset;
					if (buffer instanceof ArrayBuffer) {
						src = new Uint8Array(buffer);
					} else {
						const srcBuffer = buffer.lock();
						if (srcBuffer instanceof ArrayBuffer) {
							src = new Uint8Array(srcBuffer);
						} else {
							src = new Uint8Array(srcBuffer.buffer, srcBuffer.byteOffset, srcBuffer.byteLength);
						}
					}
					const dst = new Uint8Array(glbBuffer, offset + bufferOffset, src.byteLength);
					dst.set(src);
				});
			}
			return Promise.resolve(glbBuffer);
		});
	}
}

class RenderPassDownsample extends RenderPassShaderQuad {
	constructor(device, sourceTexture) {
		super(device);
		this.sourceTexture = sourceTexture;
		this.shader = this.createQuadShader('DownSampleShader', `

						uniform sampler2D sourceTexture;
						uniform vec2 sourceInvResolution;
						varying vec2 uv0;

						void main()
						{
								float x = sourceInvResolution.x;
								float y = sourceInvResolution.y;

								vec3 a = texture2D (sourceTexture, vec2 (uv0.x - 2.0 * x, uv0.y + 2.0 * y)).rgb;
								vec3 b = texture2D (sourceTexture, vec2 (uv0.x,           uv0.y + 2.0 * y)).rgb;
								vec3 c = texture2D (sourceTexture, vec2 (uv0.x + 2.0 * x, uv0.y + 2.0 * y)).rgb;

								vec3 d = texture2D (sourceTexture, vec2 (uv0.x - 2.0 * x, uv0.y)).rgb;
								vec3 e = texture2D (sourceTexture, vec2 (uv0.x,           uv0.y)).rgb;
								vec3 f = texture2D (sourceTexture, vec2 (uv0.x + 2.0 * x, uv0.y)).rgb;

								vec3 g = texture2D (sourceTexture, vec2 (uv0.x - 2.0 * x, uv0.y - 2.0 * y)).rgb;
								vec3 h = texture2D (sourceTexture, vec2 (uv0.x,           uv0.y - 2.0 * y)).rgb;
								vec3 i = texture2D (sourceTexture, vec2 (uv0.x + 2.0 * x, uv0.y - 2.0 * y)).rgb;

								vec3 j = texture2D (sourceTexture, vec2 (uv0.x - x, uv0.y + y)).rgb;
								vec3 k = texture2D (sourceTexture, vec2 (uv0.x + x, uv0.y + y)).rgb;
								vec3 l = texture2D (sourceTexture, vec2 (uv0.x - x, uv0.y - y)).rgb;
								vec3 m = texture2D (sourceTexture, vec2 (uv0.x + x, uv0.y - y)).rgb;

								vec3 value = e * 0.125;
								value += (a + c + g + i) * 0.03125;
								value += (b + d + f + h) * 0.0625;
								value += (j + k + l + m) * 0.125;

								gl_FragColor = vec4(value, 1.0);
						}`);
		this.sourceTextureId = device.scope.resolve('sourceTexture');
		this.sourceInvResolutionId = device.scope.resolve('sourceInvResolution');
		this.sourceInvResolutionValue = new Float32Array(2);
	}
	execute() {
		this.sourceTextureId.setValue(this.sourceTexture);
		this.sourceInvResolutionValue[0] = 1.0 / this.sourceTexture.width;
		this.sourceInvResolutionValue[1] = 1.0 / this.sourceTexture.height;
		this.sourceInvResolutionId.setValue(this.sourceInvResolutionValue);
		super.execute();
	}
}

class RenderPassUpsample extends RenderPassShaderQuad {
	constructor(device, sourceTexture) {
		super(device);
		this.sourceTexture = sourceTexture;
		this.shader = this.createQuadShader('UpSampleShader', `

						uniform sampler2D sourceTexture;
						uniform vec2 sourceInvResolution;
						varying vec2 uv0;

						void main()
						{
								float x = sourceInvResolution.x;
								float y = sourceInvResolution.y;

								vec3 a = texture2D (sourceTexture, vec2 (uv0.x - x, uv0.y + y)).rgb;
								vec3 b = texture2D (sourceTexture, vec2 (uv0.x,     uv0.y + y)).rgb;
								vec3 c = texture2D (sourceTexture, vec2 (uv0.x + x, uv0.y + y)).rgb;

								vec3 d = texture2D (sourceTexture, vec2 (uv0.x - x, uv0.y)).rgb;
								vec3 e = texture2D (sourceTexture, vec2 (uv0.x,     uv0.y)).rgb;
								vec3 f = texture2D (sourceTexture, vec2 (uv0.x + x, uv0.y)).rgb;

								vec3 g = texture2D (sourceTexture, vec2 (uv0.x - x, uv0.y - y)).rgb;
								vec3 h = texture2D (sourceTexture, vec2 (uv0.x,     uv0.y - y)).rgb;
								vec3 i = texture2D (sourceTexture, vec2 (uv0.x + x, uv0.y - y)).rgb;

								vec3 value = e * 4.0;
								value += (b + d + f + h) * 2.0;
								value += (a + c + g + i);
								value *= 1.0 / 16.0;

								gl_FragColor = vec4(value, 1.0);
						}`);
		this.sourceTextureId = device.scope.resolve('sourceTexture');
		this.sourceInvResolutionId = device.scope.resolve('sourceInvResolution');
		this.sourceInvResolutionValue = new Float32Array(2);
	}
	execute() {
		this.sourceTextureId.setValue(this.sourceTexture);
		this.sourceInvResolutionValue[0] = 1.0 / this.sourceTexture.width;
		this.sourceInvResolutionValue[1] = 1.0 / this.sourceTexture.height;
		this.sourceInvResolutionId.setValue(this.sourceInvResolutionValue);
		super.execute();
	}
}

class RenderPassBloom extends RenderPass {
	constructor(device, sourceTexture, format) {
		super(device);
		this.bloomTexture = void 0;
		this.lastMipLevel = 1;
		this.bloomRenderTarget = void 0;
		this.textureFormat = void 0;
		this.renderTargets = [];
		this._sourceTexture = sourceTexture;
		this.textureFormat = format;
		this.bloomRenderTarget = this.createRenderTarget(0);
		this.bloomTexture = this.bloomRenderTarget.colorBuffer;
	}
	destroy() {
		this.destroyRenderPasses();
		this.destroyRenderTargets();
	}
	destroyRenderTargets(startIndex = 0) {
		for (let i = startIndex; i < this.renderTargets.length; i++) {
			const rt = this.renderTargets[i];
			rt.destroyTextureBuffers();
			rt.destroy();
		}
		this.renderTargets.length = 0;
	}
	destroyRenderPasses() {
		for (let i = 0; i < this.beforePasses.length; i++) {
			this.beforePasses[i].destroy();
		}
		this.beforePasses.length = 0;
	}
	createRenderTarget(index) {
		return new RenderTarget({
			depth: false,
			colorBuffer: new Texture(this.device, {
				name: `BloomTexture${index}`,
				width: 1,
				height: 1,
				format: this.textureFormat,
				mipmaps: false,
				minFilter: FILTER_LINEAR,
				magFilter: FILTER_LINEAR,
				addressU: ADDRESS_CLAMP_TO_EDGE,
				addressV: ADDRESS_CLAMP_TO_EDGE
			})
		});
	}
	createRenderTargets(count) {
		for (let i = 0; i < count; i++) {
			const rt = i === 0 ? this.bloomRenderTarget : this.createRenderTarget(i);
			this.renderTargets.push(rt);
		}
	}
	calcMipLevels(width, height, minSize) {
		const min = Math.min(width, height);
		return Math.floor(Math.log2(min) - Math.log2(minSize));
	}
	createRenderPasses(numPasses) {
		const device = this.device;
		let passSourceTexture = this._sourceTexture;
		for (let i = 0; i < numPasses; i++) {
			const pass = new RenderPassDownsample(device, passSourceTexture);
			const rt = this.renderTargets[i];
			pass.init(rt, {
				resizeSource: passSourceTexture,
				scaleX: 0.5,
				scaleY: 0.5
			});
			pass.setClearColor(Color.BLACK);
			this.beforePasses.push(pass);
			passSourceTexture = rt.colorBuffer;
		}
		passSourceTexture = this.renderTargets[numPasses - 1].colorBuffer;
		for (let i = numPasses - 2; i >= 0; i--) {
			const pass = new RenderPassUpsample(device, passSourceTexture);
			const rt = this.renderTargets[i];
			pass.init(rt);
			pass.blendState = BlendState.ADDBLEND;
			this.beforePasses.push(pass);
			passSourceTexture = rt.colorBuffer;
		}
	}
	onDisable() {
		var _this$renderTargets$;
		(_this$renderTargets$ = this.renderTargets[0]) == null || _this$renderTargets$.resize(1, 1);
		this.destroyRenderPasses();
		this.destroyRenderTargets(1);
	}
	set sourceTexture(value) {
		this._sourceTexture = value;
		if (this.beforePasses.length > 0) {
			const firstPass = this.beforePasses[0];
			firstPass.options.resizeSource = value;
			firstPass.sourceTexture = value;
		}
	}
	get sourceTexture() {
		return this._sourceTexture;
	}
	frameUpdate() {
		super.frameUpdate();
		let numPasses = this.calcMipLevels(this._sourceTexture.width, this._sourceTexture.height, 2 ** this.lastMipLevel);
		numPasses = Math.max(1, numPasses);
		if (this.renderTargets.length !== numPasses) {
			this.destroyRenderPasses();
			this.destroyRenderTargets(1);
			this.createRenderTargets(numPasses);
			this.createRenderPasses(numPasses);
		}
	}
}

const fragmentShader = `
		varying vec2 uv0;
		uniform sampler2D sceneTexture;

		#ifdef BLOOM
				uniform sampler2D bloomTexture;
				uniform float bloomIntensity;
		#endif

		#ifdef GRADING
				uniform vec3 brightnessContrastSaturation;

				// for all parameters, 1.0 is the no-change value
				vec3 contrastSaturationBrightness(vec3 color, float brt, float sat, float con)
				{
						color = color * brt;
						float grey = dot(color, vec3(0.3, 0.59, 0.11));
						color  = mix(vec3(grey), color, sat);
						return max(mix(vec3(0.5), color, con), 0.0);
				}
		
		#endif

		#ifdef VIGNETTE

				uniform vec4 vignetterParams;

				float vignette(vec2 uv) {

						float inner = vignetterParams.x;
						float outer = vignetterParams.y;
						float curvature = vignetterParams.z;
						float intensity = vignetterParams.w;

						// edge curvature
						vec2 curve = pow(abs(uv * 2.0 -1.0), vec2(1.0 / curvature));

						// distance to edge
						float edge = pow(length(curve), curvature);

						// gradient and intensity
						return 1.0 - intensity * smoothstep(inner, outer, edge);
				}        

		#endif

		#ifdef FRINGING

				uniform float fringingIntensity;

				vec3 fringing(vec2 uv, vec3 color) {

						// offset depends on the direction from the center, raised to power to make it stronger away from the center
						vec2 centerDistance = uv0 - 0.5;
						vec2 offset = fringingIntensity * pow(centerDistance, vec2(2.0, 2.0));

						color.r = texture2D(sceneTexture, uv0 - offset).r;
						color.b = texture2D(sceneTexture, uv0 + offset).b;
						return color;
				}

		#endif

		void main() {

				vec2 uv = uv0;

				// TAA pass renders upside-down on WebGPU, flip it here
				#ifdef TAA
				#ifdef WEBGPU
						uv.y = 1.0 - uv.y;
				#endif
				#endif

				vec4 scene = texture2D(sceneTexture, uv);
				vec3 result = scene.rgb;

				#ifdef FRINGING
						result = fringing(uv, result);
				#endif

				#ifdef BLOOM
						vec3 bloom = texture2D(bloomTexture, uv).rgb;
						result += bloom * bloomIntensity;
				#endif

				#ifdef GRADING
						result = contrastSaturationBrightness(result, brightnessContrastSaturation.x, brightnessContrastSaturation.z, brightnessContrastSaturation.y);
				#endif

				result = toneMap(result);

				#ifdef VIGNETTE
						result *= vignette(uv);
				#endif

				result = gammaCorrectOutput(result);

				gl_FragColor = vec4(result, scene.a);
		}
`;
class RenderPassCompose extends RenderPassShaderQuad {
	constructor(graphicsDevice) {
		super(graphicsDevice);
		this.sceneTexture = null;
		this.bloomIntensity = 0.01;
		this._bloomTexture = null;
		this._toneMapping = TONEMAP_ACES2;
		this._gradingEnabled = false;
		this.gradingSaturation = 1;
		this.gradingContrast = 1;
		this.gradingBrightness = 1;
		this._shaderDirty = true;
		this._vignetteEnabled = false;
		this.vignetteInner = 0.5;
		this.vignetteOuter = 1.0;
		this.vignetteCurvature = 0.5;
		this.vignetteIntensity = 0.3;
		this._fringingEnabled = false;
		this.fringingIntensity = 10;
		this._taaEnabled = false;
		this._key = '';
		this.sceneTextureId = graphicsDevice.scope.resolve('sceneTexture');
		this.bloomTextureId = graphicsDevice.scope.resolve('bloomTexture');
		this.bloomIntensityId = graphicsDevice.scope.resolve('bloomIntensity');
		this.bcsId = graphicsDevice.scope.resolve('brightnessContrastSaturation');
		this.vignetterParamsId = graphicsDevice.scope.resolve('vignetterParams');
		this.fringingIntensityId = graphicsDevice.scope.resolve('fringingIntensity');
	}
	set bloomTexture(value) {
		if (this._bloomTexture !== value) {
			this._bloomTexture = value;
			this._shaderDirty = true;
		}
	}
	get bloomTexture() {
		return this._bloomTexture;
	}
	set taaEnabled(value) {
		if (this._taaEnabled !== value) {
			this._taaEnabled = value;
			this._shaderDirty = true;
		}
	}
	get taaEnabled() {
		return this._taaEnabled;
	}
	set gradingEnabled(value) {
		if (this._gradingEnabled !== value) {
			this._gradingEnabled = value;
			this._shaderDirty = true;
		}
	}
	get gradingEnabled() {
		return this._gradingEnabled;
	}
	set vignetteEnabled(value) {
		if (this._vignetteEnabled !== value) {
			this._vignetteEnabled = value;
			this._shaderDirty = true;
		}
	}
	get vignetteEnabled() {
		return this._vignetteEnabled;
	}
	set fringingEnabled(value) {
		if (this._fringingEnabled !== value) {
			this._fringingEnabled = value;
			this._shaderDirty = true;
		}
	}
	get fringingEnabled() {
		return this._fringingEnabled;
	}
	set toneMapping(value) {
		if (this._toneMapping !== value) {
			this._toneMapping = value;
			this._shaderDirty = true;
		}
	}
	get toneMapping() {
		return this._toneMapping;
	}
	get toneMapChunk() {
		switch (this.toneMapping) {
			case TONEMAP_LINEAR:
				return shaderChunks.tonemappingLinearPS;
			case TONEMAP_FILMIC:
				return shaderChunks.tonemappingFilmicPS;
			case TONEMAP_HEJL:
				return shaderChunks.tonemappingHejlPS;
			case TONEMAP_ACES:
				return shaderChunks.tonemappingAcesPS;
			case TONEMAP_ACES2:
				return shaderChunks.tonemappingAces2PS;
		}
		return shaderChunks.tonemappingNonePS;
	}
	postInit() {
		this.setClearColor(Color.BLACK);
		this.setClearDepth(1.0);
		this.setClearStencil(0);
	}
	frameUpdate() {
		if (this._shaderDirty) {
			this._shaderDirty = false;
			const key = `${this.toneMapping}` + `-${this.bloomTexture ? 'bloom' : 'nobloom'}` + `-${this.gradingEnabled ? 'grading' : 'nograding'}` + `-${this.vignetteEnabled ? 'vignette' : 'novignette'}` + `-${this.fringingEnabled ? 'fringing' : 'nofringing'}` + `-${this.taaEnabled ? 'taa' : 'notaa'}`;
			if (this._key !== key) {
				this._key = key;
				const defines = (this.bloomTexture ? `#define BLOOM\n` : '') + (this.gradingEnabled ? `#define GRADING\n` : '') + (this.vignetteEnabled ? `#define VIGNETTE\n` : '') + (this.fringingEnabled ? `#define FRINGING\n` : '') + (this.taaEnabled ? `#define TAA\n` : '');
				const fsChunks = shaderChunks.decodePS + shaderChunks.gamma2_2PS + this.toneMapChunk;
				this.shader = this.createQuadShader(`ComposeShader-${key}`, defines + fsChunks + fragmentShader);
			}
		}
	}
	execute() {
		this.sceneTextureId.setValue(this.sceneTexture);
		if (this._bloomTexture) {
			this.bloomTextureId.setValue(this._bloomTexture);
			this.bloomIntensityId.setValue(this.bloomIntensity);
		}
		if (this._gradingEnabled) {
			this.bcsId.setValue([this.gradingBrightness, this.gradingContrast, this.gradingSaturation]);
		}
		if (this._vignetteEnabled) {
			this.vignetterParamsId.setValue([this.vignetteInner, this.vignetteOuter, this.vignetteCurvature, this.vignetteIntensity]);
		}
		if (this._fringingEnabled) {
			this.fringingIntensityId.setValue(this.fringingIntensity / 1024);
		}
		super.execute();
	}
}

const fs = `
		uniform highp sampler2D uSceneDepthMap;
		uniform sampler2D sourceTexture;
		uniform sampler2D historyTexture;
		uniform mat4 matrix_viewProjectionPrevious;
		uniform mat4 matrix_viewProjectionInverse;
		uniform vec4 jitters;   // xy: current frame, zw: previous frame
		uniform vec2 textureSize;

		varying vec2 uv0;

		vec2 reproject(vec2 uv, float depth) {

				// fragment NDC
				#ifndef WEBGPU
						depth = depth * 2.0 - 1.0;
				#endif
				vec4 ndc = vec4(uv * 2.0 - 1.0, depth, 1.0);

				// remove jitter from the current frame
				ndc.xy -= jitters.xy;

				// Transform NDC to world space of the current frame
				vec4 worldPosition = matrix_viewProjectionInverse * ndc;
				worldPosition /= worldPosition.w;
		
				// world position to screen space of the previous frame
				vec4 screenPrevious = matrix_viewProjectionPrevious * worldPosition;

				return (screenPrevious.xy / screenPrevious.w) * 0.5 + 0.5;
		}

		vec4 colorClamp(vec2 uv, vec4 historyColor) {

				// out of range numbers
				vec3 minColor = vec3(9999.0);
				vec3 maxColor = vec3(-9999.0);
 
				// sample a 3x3 neighborhood to create a box in color space
				for(float x = -1.0; x <= 1.0; ++x)
				{
						for(float y = -1.0; y <= 1.0; ++y)
						{
								vec3 color = texture2D(sourceTexture, uv + vec2(x, y) / textureSize).rgb;
								minColor = min(minColor, color);
								maxColor = max(maxColor, color);
						}
				}
 
				// clamp the history color to min/max bounding box
				vec3 clamped = clamp(historyColor.rgb, minColor, maxColor);
				return vec4(clamped, historyColor.a);
		}

		void main()
		{
				vec2 uv = uv0;

				#ifdef WEBGPU
						// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
						// This hack is needed on webgpu, which makes TAA work but the resulting image is upside-down.
						// We could flip the image in the following pass, but ideally a better solution should be found.
						uv.y = 1.0 - uv.y;
						// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
				#endif

				// current frame
				vec4 srcColor = texture2D(sourceTexture, uv);

				// current depth
				float depth = texture2DLodEXT(uSceneDepthMap, uv, 0.0).r;

				// previous frame
				vec2 historyUv = reproject(uv0, depth);

				#ifdef QUALITY_HIGH

						// high quality history, sharper result
						vec4 historyColor = SampleTextureCatmullRom(TEXTURE_PASS(historyTexture), historyUv, textureSize);

				#else

						// single sample history, more blurry result
						vec4 historyColor = texture2D(historyTexture, historyUv);

				#endif

				// handle disocclusion by clamping the history color
				vec4 historyColorClamped = colorClamp(uv, historyColor);

				// handle history buffer outside of the frame
				float mixFactor = (historyUv.x < 0.0 || historyUv.x > 1.0 || historyUv.y < 0.0 || historyUv.y > 1.0) ?
						1.0 : 0.05;

				gl_FragColor = mix(historyColorClamped, srcColor, mixFactor);
		}
`;
class RenderPassTAA extends RenderPassShaderQuad {
	constructor(device, sourceTexture, cameraComponent) {
		super(device);
		this.historyIndex = 0;
		this.historyTexture = null;
		this.historyTextures = [];
		this.historyRenderTargets = [];
		this.sourceTexture = sourceTexture;
		this.cameraComponent = cameraComponent;
		const defines = `
						#define QUALITY_HIGH
				`;
		const fsChunks = shaderChunks.sampleCatmullRomPS;
		this.shader = this.createQuadShader('TaaResolveShader', defines + fsChunks + fs);
		const {
			scope
		} = device;
		this.sourceTextureId = scope.resolve('sourceTexture');
		this.textureSizeId = scope.resolve('textureSize');
		this.textureSize = new Float32Array(2);
		this.historyTextureId = scope.resolve('historyTexture');
		this.viewProjPrevId = scope.resolve('matrix_viewProjectionPrevious');
		this.viewProjInvId = scope.resolve('matrix_viewProjectionInverse');
		this.jittersId = scope.resolve('jitters');
		this.setup();
	}
	destroy() {
		if (this.renderTarget) {
			this.renderTarget.destroyTextureBuffers();
			this.renderTarget.destroy();
			this.renderTarget = null;
		}
	}
	setup() {
		for (let i = 0; i < 2; ++i) {
			this.historyTextures[i] = new Texture(this.device, {
				name: `TAA-History-${i}`,
				width: 4,
				height: 4,
				format: this.sourceTexture.format,
				mipmaps: false,
				minFilter: FILTER_LINEAR,
				magFilter: FILTER_LINEAR,
				addressU: ADDRESS_CLAMP_TO_EDGE,
				addressV: ADDRESS_CLAMP_TO_EDGE
			});
			this.historyRenderTargets[i] = new RenderTarget({
				colorBuffer: this.historyTextures[i],
				depth: false
			});
		}
		this.historyTexture = this.historyTextures[0];
		this.init(this.historyRenderTargets[0], {
			resizeSource: this.sourceTexture
		});
	}
	before() {
		this.sourceTextureId.setValue(this.sourceTexture);
		this.historyTextureId.setValue(this.historyTextures[1 - this.historyIndex]);
		this.textureSize[0] = this.sourceTexture.width;
		this.textureSize[1] = this.sourceTexture.height;
		this.textureSizeId.setValue(this.textureSize);
		const camera = this.cameraComponent.camera;
		this.viewProjPrevId.setValue(camera._viewProjPrevious.data);
		this.viewProjInvId.setValue(camera._viewProjInverse.data);
		this.jittersId.setValue(camera._jitters);
	}
	update() {
		this.historyIndex = 1 - this.historyIndex;
		this.historyTexture = this.historyTextures[this.historyIndex];
		this.renderTarget = this.historyRenderTargets[this.historyIndex];
		return this.historyTexture;
	}
}

const tempMeshInstances = [];
const DEPTH_UNIFORM_NAME = 'uSceneDepthMap';
const VELOCITY_UNIFORM_NAME = 'uSceneVelocityMap';
class RenderPassPrepass extends RenderPass {
	constructor(device, scene, renderer, camera, depthBuffer, options) {
		super(device);
		this.viewBindGroups = [];
		this.velocityTexture = void 0;
		this.scene = scene;
		this.renderer = renderer;
		this.camera = camera;
		this.setupRenderTarget(depthBuffer, options);
	}
	destroy() {
		var _this$renderTarget, _this$velocityTexture;
		super.destroy();
		(_this$renderTarget = this.renderTarget) == null || _this$renderTarget.destroy();
		this.renderTarget = null;
		(_this$velocityTexture = this.velocityTexture) == null || _this$velocityTexture.destroy();
		this.velocityTexture = null;
		this.viewBindGroups.forEach(bg => {
			bg.defaultUniformBuffer.destroy();
			bg.destroy();
		});
		this.viewBindGroups.length = 0;
	}
	setupRenderTarget(depthBuffer, options) {
		const {
			device
		} = this;
		const velocityFormat = device.getRenderableHdrFormat([PIXELFORMAT_RGBA32F, PIXELFORMAT_RGBA16F]);
		this.velocityTexture = new Texture(device, {
			name: 'VelocityTexture',
			width: 4,
			height: 4,
			format: velocityFormat,
			mipmaps: false,
			minFilter: FILTER_NEAREST,
			magFilter: FILTER_NEAREST,
			addressU: ADDRESS_CLAMP_TO_EDGE,
			addressV: ADDRESS_CLAMP_TO_EDGE
		});
		const renderTarget = new RenderTarget({
			name: 'PrepassRT',
			depthBuffer: depthBuffer
		});
		this.init(renderTarget, options);
		this.depthStencilOps.storeDepth = true;
	}
	after() {
		this.device.scope.resolve(DEPTH_UNIFORM_NAME).setValue(this.renderTarget.depthBuffer);
		this.device.scope.resolve(VELOCITY_UNIFORM_NAME).setValue(this.velocityTexture);
	}
	execute() {
		const {
			renderer,
			scene,
			renderTarget
		} = this;
		const camera = this.camera.camera;
		const layers = scene.layers.layerList;
		const subLayerEnabled = scene.layers.subLayerEnabled;
		const isTransparent = scene.layers.subLayerList;
		for (let i = 0; i < layers.length; i++) {
			const layer = layers[i];
			if (layer.enabled && subLayerEnabled[i]) {
				if (layer.camerasSet.has(camera)) {
					if (layer.id === LAYERID_DEPTH) break;
					const culledInstances = layer.getCulledInstances(camera);
					const meshInstances = isTransparent[i] ? culledInstances.transparent : culledInstances.opaque;
					for (let j = 0; j < meshInstances.length; j++) {
						var _meshInstance$materia;
						const meshInstance = meshInstances[j];
						if ((_meshInstance$materia = meshInstance.material) != null && _meshInstance$materia.depthWrite) {
							tempMeshInstances.push(meshInstance);
						}
					}
					renderer.renderForwardLayer(camera, renderTarget, null, undefined, SHADER_PREPASS_VELOCITY, this.viewBindGroups, {
						meshInstances: tempMeshInstances
					});
					tempMeshInstances.length = 0;
				}
			}
		}
	}
	frameUpdate() {
		super.frameUpdate();
		const {
			camera
		} = this;
		this.setClearDepth(camera.clearDepthBuffer ? 1 : undefined);
	}
}

class RenderPassCameraFrame extends RenderPass {
	constructor(app, options = {}) {
		super(app.graphicsDevice);
		this.app = void 0;
		this.prePass = void 0;
		this.scenePass = void 0;
		this.composePass = void 0;
		this.bloomPass = void 0;
		this.taaPass = void 0;
		this._bloomEnabled = true;
		this._renderTargetScale = 1;
		this._rt = null;
		this.app = app;
		this.options = this.sanitizeOptions(options);
		this.setupRenderPasses(this.options);
	}
	destroy() {
		if (this._rt) {
			this._rt.destroyTextureBuffers();
			this._rt.destroy();
			this._rt = null;
		}
		this.beforePasses.forEach(pass => pass.destroy());
		this.beforePasses = null;
	}
	sanitizeOptions(options) {
		const defaults = {
			camera: null,
			samples: 2,
			sceneColorMap: true,
			lastGrabLayerId: LAYERID_SKYBOX,
			lastGrabLayerIsTransparent: false,
			lastSceneLayerId: LAYERID_IMMEDIATE,
			lastSceneLayerIsTransparent: true,
			taaEnabled: false
		};
		return Object.assign({}, defaults, options);
	}
	set renderTargetScale(value) {
		this._renderTargetScale = value;
		if (this.scenePass) {
			this.scenePass.options.scaleX = value;
			this.scenePass.options.scaleY = value;
		}
	}
	get renderTargetScale() {
		return this._renderTargetScale;
	}
	set bloomEnabled(value) {
		if (this._bloomEnabled !== value) {
			this._bloomEnabled = value;
			this.composePass.bloomTexture = value ? this.bloomPass.bloomTexture : null;
			this.bloomPass.enabled = value;
		}
	}
	get bloomEnabled() {
		return this._bloomEnabled;
	}
	set lastMipLevel(value) {
		this.bloomPass.lastMipLevel = value;
	}
	get lastMipLevel() {
		return this.bloomPass.lastMipLevel;
	}
	setupRenderPasses(options) {
		const {
			app,
			device
		} = this;
		const {
			scene,
			renderer
		} = app;
		const composition = scene.layers;
		const cameraComponent = options.camera;
		const targetRenderTarget = cameraComponent.renderTarget;
		const format = device.getRenderableHdrFormat() || PIXELFORMAT_RGBA8;
		const sceneTexture = new Texture(device, {
			name: 'SceneColor',
			width: 4,
			height: 4,
			format: format,
			mipmaps: false,
			minFilter: FILTER_LINEAR,
			magFilter: FILTER_LINEAR,
			addressU: ADDRESS_CLAMP_TO_EDGE,
			addressV: ADDRESS_CLAMP_TO_EDGE
		});
		const sceneDepth = new Texture(device, {
			name: 'SceneDepth',
			width: 4,
			height: 4,
			format: PIXELFORMAT_DEPTH,
			mipmaps: false,
			minFilter: FILTER_NEAREST,
			magFilter: FILTER_NEAREST,
			addressU: ADDRESS_CLAMP_TO_EDGE,
			addressV: ADDRESS_CLAMP_TO_EDGE
		});
		const rt = new RenderTarget({
			colorBuffer: sceneTexture,
			depthBuffer: sceneDepth,
			samples: options.samples
		});
		this._rt = rt;
		const sceneOptions = {
			resizeSource: targetRenderTarget,
			scaleX: this.renderTargetScale,
			scaleY: this.renderTargetScale
		};
		if (options.prepassEnabled) {
			this.prePass = new RenderPassPrepass(device, scene, renderer, cameraComponent, sceneDepth, sceneOptions);
		}
		this.scenePass = new RenderPassForward(device, composition, scene, renderer);
		this.scenePass.init(rt, sceneOptions);
		if (options.prepassEnabled) {
			this.scenePass.noDepthClear = true;
			this.scenePass.depthStencilOps.storeDepth = true;
		}
		const lastLayerId = options.sceneColorMap ? options.lastGrabLayerId : options.lastSceneLayerId;
		const lastLayerIsTransparent = options.sceneColorMap ? options.lastGrabLayerIsTransparent : options.lastSceneLayerIsTransparent;
		let clearRenderTarget = true;
		let lastAddedIndex = 0;
		lastAddedIndex = this.scenePass.addLayers(composition, cameraComponent, lastAddedIndex, clearRenderTarget, lastLayerId, lastLayerIsTransparent);
		clearRenderTarget = false;
		let colorGrabPass;
		let scenePassTransparent;
		if (options.sceneColorMap) {
			colorGrabPass = new RenderPassColorGrab(device);
			colorGrabPass.source = rt;
			scenePassTransparent = new RenderPassForward(device, composition, scene, renderer);
			scenePassTransparent.init(rt);
			lastAddedIndex = scenePassTransparent.addLayers(composition, cameraComponent, lastAddedIndex, clearRenderTarget, options.lastSceneLayerId, options.lastSceneLayerIsTransparent);
			if (options.prepassEnabled) {
				scenePassTransparent.depthStencilOps.storeDepth = true;
			}
		}
		let sceneTextureWithTaa = sceneTexture;
		if (options.taaEnabled) {
			this.taaPass = new RenderPassTAA(device, sceneTexture, cameraComponent);
			sceneTextureWithTaa = this.taaPass.historyTexture;
		}
		this.bloomPass = new RenderPassBloom(app.graphicsDevice, sceneTextureWithTaa, format);
		this.composePass = new RenderPassCompose(app.graphicsDevice);
		this.composePass.bloomTexture = this.bloomPass.bloomTexture;
		this.composePass.taaEnabled = options.taaEnabled;
		this.composePass.init(targetRenderTarget);
		const afterPass = new RenderPassForward(device, composition, scene, renderer);
		afterPass.init(targetRenderTarget);
		afterPass.addLayers(composition, cameraComponent, lastAddedIndex, clearRenderTarget);
		const allPasses = [this.prePass, this.scenePass, colorGrabPass, scenePassTransparent, this.taaPass, this.bloomPass, this.composePass, afterPass];
		this.beforePasses = allPasses.filter(element => element !== undefined);
	}
	frameUpdate() {
		var _this$taaPass$update, _this$taaPass;
		super.frameUpdate();
		const sceneTexture = (_this$taaPass$update = (_this$taaPass = this.taaPass) == null ? void 0 : _this$taaPass.update()) != null ? _this$taaPass$update : this._rt.colorBuffer;
		this.composePass.sceneTexture = sceneTexture;
		if (this.bloomEnabled) {
			this.bloomPass.sourceTexture = sceneTexture;
		}
	}
}

const tmpV1$5 = new Vec3();
const tmpM1$1 = new Mat4();
const tmpM2 = new Mat4();
const xstart = new Vec3();
const xdir = new Vec3();
const MIN_GIZMO_SCALE = 1e-4;
const PERS_SCALE_RATIO = 0.3;
const PERS_CANVAS_RATIO = 1300;
const ORTHO_SCALE_RATIO = 0.32;
const GIZMO_LOCAL = 'local';
const GIZMO_WORLD = 'world';
class Gizmo extends EventHandler {
	constructor(app, camera, layer) {
		super();
		this._size = 1;
		this._scale = 1;
		this._coordSpace = GIZMO_WORLD;
		this._app = void 0;
		this._device = void 0;
		this._camera = void 0;
		this._layer = void 0;
		this.nodes = [];
		this.root = void 0;
		this.intersectData = [];
		this._app = app;
		this._device = app.graphicsDevice;
		this._camera = camera;
		this._layer = layer;
		this._createGizmo();
		this._updateScale();
		this._onPointerDown = e => {
			if (!this.root.enabled || document.pointerLockElement) {
				return;
			}
			const selection = this._getSelection(e.offsetX, e.offsetY);
			if (selection[0]) {
				e.preventDefault();
			}
			this.fire(Gizmo.EVENT_POINTERDOWN, e.offsetX, e.offsetY, selection[0]);
		};
		this._onPointerMove = e => {
			if (!this.root.enabled || document.pointerLockElement) {
				return;
			}
			const selection = this._getSelection(e.offsetX, e.offsetY);
			if (selection[0]) {
				e.preventDefault();
			}
			this.fire(Gizmo.EVENT_POINTERMOVE, e.offsetX, e.offsetY, selection[0]);
		};
		this._onPointerUp = e => {
			if (!this.root.enabled || document.pointerLockElement) {
				return;
			}
			this.fire(Gizmo.EVENT_POINTERUP);
		};
		this._device.canvas.addEventListener('pointerdown', this._onPointerDown);
		this._device.canvas.addEventListener('pointermove', this._onPointerMove);
		this._device.canvas.addEventListener('pointerup', this._onPointerUp);
		app.on('update', () => this._updateScale());
		app.on('destroy', () => this.destroy());
	}
	set coordSpace(value) {
		this._coordSpace = value != null ? value : GIZMO_WORLD;
		this._updateRotation();
	}
	get coordSpace() {
		return this._coordSpace;
	}
	set size(value) {
		this._size = value;
		this._updateScale();
	}
	get size() {
		return this._size;
	}
	_getProjFrustumWidth() {
		const gizmoPos = this.root.getPosition();
		const cameraPos = this._camera.entity.getPosition();
		const dist = tmpV1$5.copy(gizmoPos).sub(cameraPos).dot(this._camera.entity.forward);
		return dist * Math.tan(this._camera.fov * math.DEG_TO_RAD / 2);
	}
	_createGizmo() {
		this.root = new Entity('gizmo');
		this._app.root.addChild(this.root);
		this.root.enabled = false;
	}
	_updatePosition() {
		tmpV1$5.set(0, 0, 0);
		for (let i = 0; i < this.nodes.length; i++) {
			const node = this.nodes[i];
			tmpV1$5.add(node.getPosition());
		}
		tmpV1$5.mulScalar(1.0 / (this.nodes.length || 1));
		this.root.setPosition(tmpV1$5);
		this.fire(Gizmo.EVENT_POSITIONUPDATE, tmpV1$5);
	}
	_updateRotation() {
		tmpV1$5.set(0, 0, 0);
		if (this._coordSpace === GIZMO_LOCAL && this.nodes.length !== 0) {
			tmpV1$5.copy(this.nodes[this.nodes.length - 1].getEulerAngles());
		}
		this.root.setEulerAngles(tmpV1$5);
		this.fire(Gizmo.EVENT_ROTATIONUPDATE, tmpV1$5);
	}
	_updateScale() {
		if (this._camera.projection === PROJECTION_PERSPECTIVE) {
			let canvasMult = 1;
			if (this._device.width > 0 && this._device.height > 0) {
				canvasMult = PERS_CANVAS_RATIO / Math.min(this._device.width, this._device.height);
			}
			this._scale = this._getProjFrustumWidth() * canvasMult * PERS_SCALE_RATIO;
		} else {
			this._scale = this._camera.orthoHeight * ORTHO_SCALE_RATIO;
		}
		this._scale = Math.max(this._scale * this._size, MIN_GIZMO_SCALE);
		this.root.setLocalScale(this._scale, this._scale, this._scale);
		this.fire(Gizmo.EVENT_SCALEUPDATE, this._scale);
	}
	_getSelection(x, y) {
		const start = this._camera.screenToWorld(x, y, 1);
		const end = this._camera.screenToWorld(x, y, this._camera.farClip);
		const dir = end.clone().sub(start).normalize();
		const selection = [];
		for (let i = 0; i < this.intersectData.length; i++) {
			const {
				meshTriDataList,
				parent,
				meshInstances
			} = this.intersectData[i];
			const wtm = parent.getWorldTransform().clone();
			for (let j = 0; j < meshTriDataList.length; j++) {
				const {
					tris,
					ptm,
					priority
				} = meshTriDataList[j];
				tmpM1$1.copy(wtm).mul(ptm);
				tmpM2.copy(tmpM1$1).invert();
				tmpM2.transformPoint(start, xstart);
				tmpM2.transformVector(dir, xdir);
				xdir.normalize();
				for (let k = 0; k < tris.length; k++) {
					if (tris[k].intersectRay(xstart, xdir, tmpV1$5)) {
						selection.push({
							dist: tmpM1$1.transformPoint(tmpV1$5).sub(start).length(),
							meshInstances: meshInstances,
							priority: priority
						});
					}
				}
			}
		}
		if (selection.length) {
			selection.sort((s0, s1) => {
				if (s0.priority !== 0 && s1.priority !== 0) {
					return s1.priority - s0.priority;
				}
				return s0.dist - s1.dist;
			});
			return selection[0].meshInstances;
		}
		return [];
	}
	attach(nodes = []) {
		if (nodes.length === 0) {
			return;
		}
		this.nodes = nodes;
		this._updatePosition();
		this._updateRotation();
		this.fire(Gizmo.EVENT_NODESATTACH);
		this.root.enabled = true;
		this.fire(Gizmo.EVENT_RENDERUPDATE);
	}
	detach() {
		this.root.enabled = false;
		this.fire(Gizmo.EVENT_RENDERUPDATE);
		this.fire(Gizmo.EVENT_NODESDETACH);
		this.nodes = [];
	}
	destroy() {
		this.detach();
		this._device.canvas.removeEventListener('pointerdown', this._onPointerDown);
		this._device.canvas.removeEventListener('pointermove', this._onPointerMove);
		this._device.canvas.removeEventListener('pointerup', this._onPointerUp);
		this.root.destroy();
	}
}
Gizmo.EVENT_POINTERDOWN = 'pointer:down';
Gizmo.EVENT_POINTERMOVE = 'pointer:move';
Gizmo.EVENT_POINTERUP = 'pointer:up';
Gizmo.EVENT_POSITIONUPDATE = 'position:update';
Gizmo.EVENT_ROTATIONUPDATE = 'rotation:update';
Gizmo.EVENT_SCALEUPDATE = 'scale:update';
Gizmo.EVENT_NODESATTACH = 'nodes:attach';
Gizmo.EVENT_NODESDETACH = 'nodes:detach';
Gizmo.EVENT_RENDERUPDATE = 'render:update';

const COLOR_RED = Object.freeze(new Color(1, 0.3, 0.3));
const COLOR_GREEN = Object.freeze(new Color(0.3, 1, 0.3));
const COLOR_BLUE = Object.freeze(new Color(0.3, 0.3, 1));
const COLOR_YELLOW = Object.freeze(new Color(1, 1, 0.5));
const COLOR_GRAY = Object.freeze(new Color(0.5, 0.5, 0.5, 0.5));

const tmpV1$4 = new Vec3();
const tmpV2$4 = new Vec3();
const tmpV3$1 = new Vec3();
const tmpQ1$3 = new Quat();
const pointDelta = new Vec3();
const VEC3_AXES = Object.keys(tmpV1$4);
const FACING_EPSILON = 0.2;
const SPANLINE_SIZE = 1e3;
const ROTATE_SCALE = 900;
const colorSemi = color => {
	const clone = color.clone();
	clone.a = 0.6;
	return clone;
};
class TransformGizmo extends Gizmo {
	constructor(app, camera, layer) {
		super(app, camera, layer);
		this._meshColors = {
			axis: {
				x: colorSemi(COLOR_RED),
				y: colorSemi(COLOR_GREEN),
				z: colorSemi(COLOR_BLUE),
				xyz: colorSemi(Color.WHITE),
				face: colorSemi(Color.WHITE)
			},
			hover: {
				x: COLOR_RED.clone(),
				y: COLOR_GREEN.clone(),
				z: COLOR_BLUE.clone(),
				xyz: Color.WHITE.clone(),
				face: COLOR_YELLOW.clone()
			},
			disabled: COLOR_GRAY.clone()
		};
		this._guideColors = {
			x: COLOR_RED.clone(),
			y: COLOR_GREEN.clone(),
			z: COLOR_BLUE.clone(),
			face: COLOR_YELLOW.clone()
		};
		this._gizmoRotationStart = new Quat();
		this._shapes = {};
		this._shapeMap = new Map();
		this._hoverShape = null;
		this._hoverAxis = '';
		this._hoverIsPlane = false;
		this._selectedAxis = '';
		this._selectedIsPlane = false;
		this._selectionStartPoint = new Vec3();
		this._selectionStartAngle = 0;
		this._isRotation = false;
		this._useUniformScaling = false;
		this._dragging = false;
		this._snap = false;
		this.snapIncrement = 1;
		app.on('update', () => {
			if (!this.root.enabled) {
				return;
			}
			this._drawGuideLines();
		});
		this.on('pointer:down', (x, y, meshInstance) => {
			const shape = this._shapeMap.get(meshInstance);
			if (shape != null && shape.disabled) {
				return;
			}
			if (this._dragging) {
				return;
			}
			if (!meshInstance) {
				return;
			}
			this._selectedAxis = this._getAxis(meshInstance);
			this._selectedIsPlane = this._getIsPlane(meshInstance);
			this._gizmoRotationStart.copy(this.root.getRotation());
			const pointInfo = this._calcPoint(x, y);
			this._selectionStartPoint.copy(pointInfo.point);
			this._selectionStartAngle = pointInfo.angle;
			this._dragging = true;
			this.fire(TransformGizmo.EVENT_TRANSFORMSTART);
		});
		this.on('pointer:move', (x, y, meshInstance) => {
			const shape = this._shapeMap.get(meshInstance);
			if (shape != null && shape.disabled) {
				return;
			}
			this._hover(meshInstance);
			if (!this._dragging) {
				return;
			}
			const pointInfo = this._calcPoint(x, y);
			pointDelta.copy(pointInfo.point).sub(this._selectionStartPoint);
			const angleDelta = pointInfo.angle - this._selectionStartAngle;
			this.fire(TransformGizmo.EVENT_TRANSFORMMOVE, pointDelta, angleDelta);
			this._hoverAxis = '';
			this._hoverIsPlane = false;
		});
		this.on('pointer:up', () => {
			if (!this._dragging) {
				return;
			}
			this._dragging = false;
			this.fire(TransformGizmo.EVENT_TRANSFORMEND);
			this._selectedAxis = '';
			this._selectedIsPlane = false;
		});
		this.on('nodes:detach', () => {
			this.snap = false;
			this._hoverAxis = '';
			this._hoverIsPlane = false;
			this._hover(null);
			this.fire('pointer:up');
		});
	}
	set snap(value) {
		this._snap = this.root.enabled && value;
	}
	get snap() {
		return this._snap;
	}
	set xAxisColor(value) {
		this._updateAxisColor('x', value);
	}
	get xAxisColor() {
		return this._meshColors.axis.x;
	}
	set yAxisColor(value) {
		this._updateAxisColor('y', value);
	}
	get yAxisColor() {
		return this._meshColors.axis.y;
	}
	set zAxisColor(value) {
		this._updateAxisColor('z', value);
	}
	get zAxisColor() {
		return this._meshColors.axis.z;
	}
	_updateAxisColor(axis, value) {
		this._guideColors[axis].copy(value);
		this._meshColors.axis[axis].copy(colorSemi(value));
		this._meshColors.hover[axis].copy(value);
		for (const name in this._shapes) {
			this._shapes[name].hover(!!this._hoverAxis);
		}
	}
	_getAxis(meshInstance) {
		if (!meshInstance) {
			return '';
		}
		return meshInstance.node.name.split(":")[1];
	}
	_getIsPlane(meshInstance) {
		if (!meshInstance) {
			return false;
		}
		return meshInstance.node.name.indexOf('plane') !== -1;
	}
	_hover(meshInstance) {
		if (this._dragging) {
			return;
		}
		this._hoverAxis = this._getAxis(meshInstance);
		this._hoverIsPlane = this._getIsPlane(meshInstance);
		const shape = this._shapeMap.get(meshInstance) || null;
		if (shape === this._hoverShape) {
			return;
		}
		if (this._hoverShape) {
			this._hoverShape.hover(false);
			this._hoverShape = null;
		}
		if (shape) {
			shape.hover(true);
			this._hoverShape = shape;
		}
		this.fire('render:update');
	}
	_calcPoint(x, y) {
		const gizmoPos = this.root.getPosition();
		const mouseWPos = this._camera.screenToWorld(x, y, 1);
		const cameraRot = this._camera.entity.getRotation();
		const rayOrigin = this._camera.entity.getPosition();
		const rayDir = new Vec3();
		const planeNormal = new Vec3();
		const axis = this._selectedAxis;
		const isPlane = this._selectedIsPlane;
		const isRotation = this._isRotation;
		const isUniform = this._useUniformScaling && isPlane;
		const isAllAxes = axis === 'xyz';
		const isFacing = axis === 'face';
		if (this._camera.projection === PROJECTION_PERSPECTIVE) {
			rayDir.copy(mouseWPos).sub(rayOrigin).normalize();
		} else {
			rayOrigin.add(mouseWPos);
			this._camera.entity.getWorldTransform().transformVector(tmpV1$4.set(0, 0, -1), rayDir);
		}
		if (isUniform || isAllAxes || isFacing) {
			planeNormal.copy(rayOrigin).sub(gizmoPos).normalize();
		} else {
			planeNormal[axis] = 1;
			tmpQ1$3.copy(this._gizmoRotationStart).transformVector(planeNormal, planeNormal);
			if (!isPlane && !isRotation) {
				tmpV1$4.copy(rayOrigin).sub(gizmoPos).normalize();
				planeNormal.copy(tmpV1$4.sub(planeNormal.mulScalar(planeNormal.dot(tmpV1$4))).normalize());
			}
		}
		const rayPlaneDot = planeNormal.dot(rayDir);
		const planeDist = gizmoPos.dot(planeNormal);
		const pointPlaneDist = (planeNormal.dot(rayOrigin) - planeDist) / rayPlaneDot;
		const point = rayDir.mulScalar(-pointPlaneDist).add(rayOrigin);
		if (isRotation) {
			point.sub(gizmoPos);
		}
		if (isUniform) {
			tmpV1$4.copy(point).sub(gizmoPos).normalize();
			switch (axis) {
				case 'x':
					tmpV2$4.copy(this.root.up);
					tmpV3$1.copy(this.root.forward).mulScalar(-1);
					break;
				case 'y':
					tmpV2$4.copy(this.root.right);
					tmpV3$1.copy(this.root.forward).mulScalar(-1);
					break;
				case 'z':
					tmpV2$4.copy(this.root.up);
					tmpV3$1.copy(this.root.right);
					break;
				default:
					tmpV2$4.set(0, 0, 0);
					tmpV3$1.set(0, 0, 0);
					break;
			}
			tmpV2$4.add(tmpV3$1).normalize();
			const v = point.sub(gizmoPos).length() * tmpV1$4.dot(tmpV2$4);
			point.set(v, v, v);
			point[axis] = 1;
		} else if (isAllAxes) {
			tmpV1$4.copy(point).sub(gizmoPos).normalize();
			tmpV2$4.copy(this._camera.entity.up).add(this._camera.entity.right).normalize();
			const v = point.sub(gizmoPos).length() * tmpV1$4.dot(tmpV2$4);
			point.set(v, v, v);
		} else if (!isFacing) {
			if (!isPlane && !isRotation) {
				planeNormal.set(0, 0, 0);
				planeNormal[axis] = 1;
				tmpQ1$3.transformVector(planeNormal, planeNormal);
				point.copy(planeNormal.mulScalar(planeNormal.dot(point)));
			}
			tmpQ1$3.invert().transformVector(point, point);
			if (!isPlane && !isRotation) {
				const v = point[axis];
				point.set(0, 0, 0);
				point[axis] = v;
			}
		}
		let angle = 0;
		if (isRotation) {
			let isAxisFacing = isFacing;
			tmpV1$4.copy(rayOrigin).sub(gizmoPos).normalize();
			tmpV2$4.cross(planeNormal, tmpV1$4);
			isAxisFacing || (isAxisFacing = tmpV2$4.length() < FACING_EPSILON);
			if (isAxisFacing) {
				switch (axis) {
					case 'x':
						angle = Math.atan2(point.z, point.y) * math.RAD_TO_DEG;
						break;
					case 'y':
						angle = Math.atan2(point.x, point.z) * math.RAD_TO_DEG;
						break;
					case 'z':
						angle = Math.atan2(point.y, point.x) * math.RAD_TO_DEG;
						break;
					case 'face':
						cameraRot.invert().transformVector(point, tmpV1$4);
						angle = Math.atan2(tmpV1$4.y, tmpV1$4.x) * math.RAD_TO_DEG;
						break;
				}
			} else {
				angle = mouseWPos.dot(tmpV2$4.normalize()) * ROTATE_SCALE;
				if (this._camera.projection === PROJECTION_ORTHOGRAPHIC) {
					angle /= this._camera.orthoHeight || 1;
				}
			}
		}
		return {
			point,
			angle
		};
	}
	_drawGuideLines() {
		const gizmoPos = this.root.getPosition();
		const gizmoRot = tmpQ1$3.copy(this.root.getRotation());
		const checkAxis = this._hoverAxis || this._selectedAxis;
		const checkIsPlane = this._hoverIsPlane || this._selectedIsPlane;
		for (let i = 0; i < VEC3_AXES.length; i++) {
			const axis = VEC3_AXES[i];
			if (checkAxis === 'xyz') {
				this._drawSpanLine(gizmoPos, gizmoRot, axis);
				continue;
			}
			if (checkIsPlane) {
				if (axis !== checkAxis) {
					this._drawSpanLine(gizmoPos, gizmoRot, axis);
				}
			} else {
				if (axis === checkAxis) {
					this._drawSpanLine(gizmoPos, gizmoRot, axis);
				}
			}
		}
	}
	_drawSpanLine(pos, rot, axis) {
		tmpV1$4.set(0, 0, 0);
		tmpV1$4[axis] = 1;
		tmpV1$4.mulScalar(SPANLINE_SIZE);
		tmpV2$4.copy(tmpV1$4).mulScalar(-1);
		rot.transformVector(tmpV1$4, tmpV1$4);
		rot.transformVector(tmpV2$4, tmpV2$4);
		this._app.drawLine(tmpV1$4.add(pos), tmpV2$4.add(pos), this._guideColors[axis], true);
	}
	_createTransform() {
		for (const key in this._shapes) {
			const shape = this._shapes[key];
			this.root.addChild(shape.entity);
			this.intersectData.push({
				meshTriDataList: shape.meshTriDataList,
				parent: shape.entity,
				meshInstances: shape.meshInstances
			});
			for (let i = 0; i < shape.meshInstances.length; i++) {
				this._shapeMap.set(shape.meshInstances[i], shape);
			}
		}
	}
	enableShape(shapeAxis, enabled) {
		if (!this._shapes.hasOwnProperty(shapeAxis)) {
			return;
		}
		this._shapes[shapeAxis].disabled = !enabled;
	}
	isShapeEnabled(shapeAxis) {
		if (!this._shapes.hasOwnProperty(shapeAxis)) {
			return false;
		}
		return !this._shapes[shapeAxis].disabled;
	}
	destroy() {
		for (const key in this._shapes) {
			this._shapes[key].destroy();
		}
		super.destroy();
	}
}
TransformGizmo.EVENT_TRANSFORMSTART = 'transform:start';
TransformGizmo.EVENT_TRANSFORMMOVE = 'transform:move';
TransformGizmo.EVENT_TRANSFORMEND = 'transform:end';

const e1 = new Vec3();
const e2 = new Vec3();
const h = new Vec3();
const s = new Vec3();
const q = new Vec3();
const EPSILON = 1e-6;
class Tri {
	constructor(v0, v1, v2) {
		this.v0 = new Vec3();
		this.v1 = new Vec3();
		this.v2 = new Vec3();
		this.set(v0, v1, v2);
	}
	set(v0, v1, v2) {
		this.v0.copy(v0);
		this.v1.copy(v1);
		this.v2.copy(v2);
		return this;
	}
	intersectRay(origin, dir, out, epsilon = EPSILON) {
		e1.sub2(this.v1, this.v0);
		e2.sub2(this.v2, this.v0);
		h.cross(dir, e2);
		const a = e1.dot(h);
		if (a > -epsilon && a < epsilon) {
			return false;
		}
		const f = 1 / a;
		s.sub2(origin, this.v0);
		const u = f * s.dot(h);
		if (u < 0 || u > 1) {
			return false;
		}
		q.cross(s, e1);
		const v = f * dir.dot(q);
		if (v < 0 || u + v > 1) {
			return false;
		}
		const t = f * e2.dot(q);
		if (t > epsilon) {
			if (out instanceof Vec3) {
				out.copy(dir).mulScalar(t).add(origin);
			}
			return true;
		}
		return false;
	}
}

const tmpV1$3 = new Vec3();
const tmpV2$3 = new Vec3();
const tmpV3 = new Vec3();
class MeshTriData {
	constructor(mesh, priority = 0) {
		this._priority = 0;
		this._ptm = new Mat4();
		this.tris = void 0;
		this.setTris(mesh);
		this._priority = priority;
	}
	get ptm() {
		return this._ptm;
	}
	get priority() {
		return this._priority;
	}
	_trisFromMesh(mesh, destroy = true) {
		const tris = [];
		const pos = [];
		const indices = [];
		mesh.getPositions(pos);
		mesh.getIndices(indices);
		if (destroy) {
			mesh.destroy();
		}
		for (let k = 0; k < indices.length; k += 3) {
			const i1 = indices[k];
			const i2 = indices[k + 1];
			const i3 = indices[k + 2];
			tmpV1$3.set(pos[i1 * 3], pos[i1 * 3 + 1], pos[i1 * 3 + 2]);
			tmpV2$3.set(pos[i2 * 3], pos[i2 * 3 + 1], pos[i2 * 3 + 2]);
			tmpV3.set(pos[i3 * 3], pos[i3 * 3 + 1], pos[i3 * 3 + 2]);
			const tri = new Tri(tmpV1$3, tmpV2$3, tmpV3);
			tris.push(tri);
		}
		return tris;
	}
	setTransform(pos = new Vec3(), rot = new Quat(), scale = new Vec3()) {
		this.ptm.setTRS(pos, rot, scale);
	}
	setTris(mesh) {
		if (!mesh || !(mesh instanceof Mesh)) {
			throw new Error('No mesh provided.');
		}
		this.tris = this._trisFromMesh(mesh);
	}
}

const SHADOW_DAMP_SCALE = 0.25;
const SHADOW_DAMP_OFFSET = 0.75;
const SHADOW_MESH_MAP = new Map();
const TORUS_RENDER_SEGMENTS = 80;
const TORUS_INTERSECT_SEGMENTS = 20;
const LIGHT_DIR = new Vec3(1, 2, 3);
const MESH_TEMPLATES = {
	box: createBox,
	cone: createCone,
	cylinder: createCylinder,
	plane: createPlane,
	torus: createTorus
};
const SHADER = {
	vert: `
				attribute vec3 vertex_position;
				attribute vec4 vertex_color;
				varying vec4 vColor;
				varying vec2 vZW;
				uniform mat4 matrix_model;
				uniform mat4 matrix_viewProjection;
				void main(void) {
						gl_Position = matrix_viewProjection * matrix_model * vec4(vertex_position, 1.0);
						vColor = vertex_color;
						#ifdef GL2
								// store z/w for later use in fragment shader
								vZW = gl_Position.zw;
								// disable depth clipping
								gl_Position.z = 0.0;
						#endif
				}`,
	frag: `
				precision highp float;
				varying vec4 vColor;
				varying vec2 vZW;
				void main(void) {
						gl_FragColor = vColor;
						#ifdef GL2
								// clamp depth in Z to [0, 1] range
								gl_FragDepth = max(0.0, min(1.0, (vZW.x / vZW.y + 1.0) * 0.5));
						#endif
				}`
};
const tmpV1$2 = new Vec3();
const tmpV2$2 = new Vec3();
const tmpQ1$2 = new Quat();
function createShadowMesh(device, entity, type, color = Color.WHITE, templateOpts = {}) {
	const createTemplate = MESH_TEMPLATES[type];
	if (!createTemplate) {
		throw new Error('Invalid primitive type.');
	}
	const mesh = createTemplate(device, templateOpts);
	const options = {
		positions: [],
		normals: [],
		indices: [],
		colors: []
	};
	mesh.getPositions(options.positions);
	mesh.getNormals(options.normals);
	mesh.getIndices(options.indices);
	const wtm = entity.getWorldTransform().clone().invert();
	tmpV1$2.copy(LIGHT_DIR);
	wtm.transformVector(tmpV1$2, tmpV1$2);
	tmpV1$2.normalize();
	const numVertices = mesh.vertexBuffer.numVertices;
	const shadow = calculateShadow(tmpV1$2, numVertices, options.normals);
	for (let i = 0; i < shadow.length; i++) {
		options.colors.push(shadow[i] * color.r * 255, shadow[i] * color.g * 255, shadow[i] * color.b * 255, color.a * 255);
	}
	const shadowMesh = createMesh(device, options.positions, options);
	SHADOW_MESH_MAP.set(shadowMesh, shadow);
	return shadowMesh;
}
function calculateShadow(lightDir, numVertices, normals) {
	const shadow = [];
	for (let i = 0; i < numVertices; i++) {
		const x = normals[i * 3];
		const y = normals[i * 3 + 1];
		const z = normals[i * 3 + 2];
		tmpV2$2.set(x, y, z);
		const dot = lightDir.dot(tmpV2$2);
		shadow.push(dot * SHADOW_DAMP_SCALE + SHADOW_DAMP_OFFSET);
	}
	return shadow;
}
function setShadowMeshColor(mesh, color) {
	if (!SHADOW_MESH_MAP.has(mesh)) {
		return;
	}
	const shadow = SHADOW_MESH_MAP.get(mesh);
	const colors = [];
	for (let i = 0; i < shadow.length; i++) {
		colors.push(shadow[i] * color.r * 255, shadow[i] * color.g * 255, shadow[i] * color.b * 255, color.a * 255);
	}
	mesh.setColors32(colors);
	mesh.update();
}
class AxisShape {
	constructor(device, options) {
		var _options$axis, _options$position, _options$rotation, _options$scale, _options$disabled, _options$layers;
		this._position = void 0;
		this._rotation = void 0;
		this._scale = void 0;
		this._layers = [];
		this._disabled = void 0;
		this._defaultColor = Color.WHITE;
		this._hoverColor = Color.BLACK;
		this._disabledColor = COLOR_GRAY;
		this._cull = CULLFACE_BACK;
		this.device = void 0;
		this.axis = void 0;
		this.entity = void 0;
		this.meshTriDataList = [];
		this.meshInstances = [];
		this.device = device;
		this.axis = (_options$axis = options.axis) != null ? _options$axis : 'x';
		this._position = (_options$position = options.position) != null ? _options$position : new Vec3();
		this._rotation = (_options$rotation = options.rotation) != null ? _options$rotation : new Vec3();
		this._scale = (_options$scale = options.scale) != null ? _options$scale : new Vec3(1, 1, 1);
		this._disabled = (_options$disabled = options.disabled) != null ? _options$disabled : false;
		this._layers = (_options$layers = options.layers) != null ? _options$layers : this._layers;
		if (options.defaultColor instanceof Color) {
			this._defaultColor = options.defaultColor;
		}
		if (options.hoverColor instanceof Color) {
			this._hoverColor = options.hoverColor;
		}
		if (options.disabledColor instanceof Color) {
			this._disabledColor = options.disabledColor;
		}
	}
	set disabled(value) {
		for (let i = 0; i < this.meshInstances.length; i++) {
			setShadowMeshColor(this.meshInstances[i].mesh, this._disabledColor);
		}
		this._disabled = value != null ? value : false;
	}
	get disabled() {
		return this._disabled;
	}
	_createRoot(name) {
		this.entity = new Entity(name + ':' + this.axis);
		this._updateRootTransform();
	}
	_updateRootTransform() {
		this.entity.setLocalPosition(this._position);
		this.entity.setLocalEulerAngles(this._rotation);
		this.entity.setLocalScale(this._scale);
	}
	_addRenderMeshes(entity, meshes) {
		const shader = createShaderFromCode(this.device, SHADER.vert, SHADER.frag, 'axis-shape', {
			vertex_position: SEMANTIC_POSITION,
			vertex_color: SEMANTIC_COLOR
		});
		const material = new Material();
		material.shader = shader;
		material.cull = this._cull;
		material.blendType = BLEND_NORMAL;
		material.update();
		const meshInstances = [];
		for (let i = 0; i < meshes.length; i++) {
			const mi = new MeshInstance(meshes[i], material);
			meshInstances.push(mi);
			this.meshInstances.push(mi);
		}
		entity.addComponent('render', {
			meshInstances: meshInstances,
			layers: this._layers,
			castShadows: false
		});
	}
	_addRenderShadowMesh(entity, type) {
		const color = this._disabled ? this._disabledColor : this._defaultColor;
		const mesh = createShadowMesh(this.device, entity, type, color);
		this._addRenderMeshes(entity, [mesh]);
	}
	hover(state) {
		if (this._disabled) {
			return;
		}
		for (let i = 0; i < this.meshInstances.length; i++) {
			const color = state ? this._hoverColor : this._defaultColor;
			setShadowMeshColor(this.meshInstances[i].mesh, color);
		}
	}
	destroy() {
		this.entity.destroy();
	}
}
class AxisArrow extends AxisShape {
	constructor(device, options = {}) {
		super(device, options);
		this._gap = 0;
		this._lineThickness = 0.02;
		this._lineLength = 0.5;
		this._arrowThickness = 0.12;
		this._arrowLength = 0.18;
		this._tolerance = 0.1;
		this.meshTriDataList = [new MeshTriData(createCone(this.device)), new MeshTriData(createCylinder(this.device), 1)];
		this._createArrow();
	}
	set gap(value) {
		this._gap = value != null ? value : 0;
		this._updateHead();
		this._updateLine();
	}
	get gap() {
		return this._gap;
	}
	set lineThickness(value) {
		this._lineThickness = value != null ? value : 1;
		this._updateHead();
		this._updateLine();
	}
	get lineThickness() {
		return this._lineThickness;
	}
	set lineLength(value) {
		this._lineLength = value != null ? value : 1;
		this._updateHead();
		this._updateLine();
	}
	get lineLength() {
		return this._lineLength;
	}
	set arrowThickness(value) {
		this._arrowThickness = value != null ? value : 1;
		this._updateHead();
	}
	get arrowThickness() {
		return this._arrowThickness;
	}
	set arrowLength(value) {
		this._arrowLength = value != null ? value : 1;
		this._updateHead();
	}
	get arrowLength() {
		return this._arrowLength;
	}
	set tolerance(value) {
		this._tolerance = value;
		this._updateLine();
	}
	get tolerance() {
		return this._tolerance;
	}
	_createArrow() {
		this._createRoot('arrow');
		this._head = new Entity('head:' + this.axis);
		this.entity.addChild(this._head);
		this._updateHead();
		this._addRenderShadowMesh(this._head, 'cone');
		this._line = new Entity('line:' + this.axis);
		this.entity.addChild(this._line);
		this._updateLine();
		this._addRenderShadowMesh(this._line, 'cylinder');
	}
	_updateHead() {
		tmpV1$2.set(0, this._gap + this._arrowLength * 0.5 + this._lineLength, 0);
		tmpQ1$2.set(0, 0, 0, 1);
		tmpV2$2.set(this._arrowThickness, this._arrowLength, this._arrowThickness);
		this.meshTriDataList[0].setTransform(tmpV1$2, tmpQ1$2, tmpV2$2);
		this._head.setLocalPosition(0, this._gap + this._arrowLength * 0.5 + this._lineLength, 0);
		this._head.setLocalScale(this._arrowThickness, this._arrowLength, this._arrowThickness);
	}
	_updateLine() {
		tmpV1$2.set(0, this._gap + this._lineLength * 0.5, 0);
		tmpQ1$2.set(0, 0, 0, 1);
		tmpV2$2.set(this._lineThickness + this._tolerance, this._lineLength, this._lineThickness + this._tolerance);
		this.meshTriDataList[1].setTransform(tmpV1$2, tmpQ1$2, tmpV2$2);
		this._line.setLocalPosition(0, this._gap + this._lineLength * 0.5, 0);
		this._line.setLocalScale(this._lineThickness, this._lineLength, this._lineThickness);
	}
}
class AxisBoxCenter extends AxisShape {
	constructor(device, options = {}) {
		super(device, options);
		this._size = 0.12;
		this._tolerance = 0.05;
		this.meshTriDataList = [new MeshTriData(createBox(this.device), 2)];
		this._createCenter();
	}
	_createCenter() {
		this._createRoot('boxCenter');
		this._updateTransform();
		this._addRenderShadowMesh(this.entity, 'box');
	}
	set size(value) {
		this._size = value != null ? value : 1;
		this._updateTransform();
	}
	get size() {
		return this._size;
	}
	set tolerance(value) {
		this._tolerance = value;
		this._updateTransform();
	}
	get tolerance() {
		return this._tolerance;
	}
	_updateTransform() {
		this.entity.setLocalScale(this._size, this._size, this._size);
	}
}
class AxisBoxLine extends AxisShape {
	constructor(device, options = {}) {
		super(device, options);
		this._gap = 0;
		this._lineThickness = 0.02;
		this._lineLength = 0.5;
		this._boxSize = 0.12;
		this._tolerance = 0.1;
		this.meshTriDataList = [new MeshTriData(createBox(this.device)), new MeshTriData(createCylinder(this.device), 1)];
		this._createBoxLine();
	}
	set gap(value) {
		this._gap = value != null ? value : 0;
		this._updateLine();
		this._updateBox();
	}
	get gap() {
		return this._gap;
	}
	set lineThickness(value) {
		this._lineThickness = value != null ? value : 1;
		this._updateLine();
		this._updateBox();
	}
	get lineThickness() {
		return this._lineThickness;
	}
	set lineLength(value) {
		this._lineLength = value != null ? value : 1;
		this._updateLine();
		this._updateBox();
	}
	get lineLength() {
		return this._lineLength;
	}
	set boxSize(value) {
		this._boxSize = value != null ? value : 1;
		this._updateBox();
	}
	get boxSize() {
		return this._boxSize;
	}
	set tolerance(value) {
		this._tolerance = value;
		this._updateLine();
	}
	get tolerance() {
		return this._tolerance;
	}
	_createBoxLine() {
		this._createRoot('boxLine');
		this._box = new Entity('box:' + this.axis);
		this.entity.addChild(this._box);
		this._updateBox();
		this._addRenderShadowMesh(this._box, 'box');
		this._line = new Entity('line:' + this.axis);
		this.entity.addChild(this._line);
		this._updateLine();
		this._addRenderShadowMesh(this._line, 'cylinder');
	}
	_updateBox() {
		tmpV1$2.set(0, this._gap + this._boxSize * 0.5 + this._lineLength, 0);
		tmpQ1$2.set(0, 0, 0, 1);
		tmpV2$2.set(this._boxSize, this._boxSize, this._boxSize);
		this.meshTriDataList[0].setTransform(tmpV1$2, tmpQ1$2, tmpV2$2);
		this._box.setLocalPosition(0, this._gap + this._boxSize * 0.5 + this._lineLength, 0);
		this._box.setLocalScale(this._boxSize, this._boxSize, this._boxSize);
	}
	_updateLine() {
		tmpV1$2.set(0, this._gap + this._lineLength * 0.5, 0);
		tmpQ1$2.set(0, 0, 0, 1);
		tmpV2$2.set(this._lineThickness + this._tolerance, this._lineLength, this._lineThickness + this._tolerance);
		this.meshTriDataList[1].setTransform(tmpV1$2, tmpQ1$2, tmpV2$2);
		this._line.setLocalPosition(0, this._gap + this._lineLength * 0.5, 0);
		this._line.setLocalScale(this._lineThickness, this._lineLength, this._lineThickness);
	}
}
class AxisDisk extends AxisShape {
	constructor(device, options = {}) {
		var _options$tubeRadius, _options$ringRadius, _options$sectorAngle;
		super(device, options);
		this._tubeRadius = 0.01;
		this._ringRadius = 0.5;
		this._sectorAngle = void 0;
		this._lightDir = void 0;
		this._tolerance = 0.05;
		this._tubeRadius = (_options$tubeRadius = options.tubeRadius) != null ? _options$tubeRadius : this._tubeRadius;
		this._ringRadius = (_options$ringRadius = options.ringRadius) != null ? _options$ringRadius : this._ringRadius;
		this._sectorAngle = (_options$sectorAngle = options.sectorAngle) != null ? _options$sectorAngle : this._sectorAngle;
		this.meshTriDataList = [new MeshTriData(this._createIntersectTorus())];
		this._createDisk();
	}
	_createIntersectTorus() {
		return createTorus(this.device, {
			tubeRadius: this._tubeRadius + this._tolerance,
			ringRadius: this._ringRadius,
			sectorAngle: this._sectorAngle,
			segments: TORUS_INTERSECT_SEGMENTS
		});
	}
	_createRenderTorus(sectorAngle) {
		const color = this._disabled ? this._disabledColor : this._defaultColor;
		return createShadowMesh(this.device, this.entity, 'torus', color, {
			tubeRadius: this._tubeRadius,
			ringRadius: this._ringRadius,
			sectorAngle: sectorAngle,
			segments: TORUS_RENDER_SEGMENTS
		});
	}
	_createDisk() {
		this._createRoot('disk');
		this._addRenderMeshes(this.entity, [this._createRenderTorus(this._sectorAngle), this._createRenderTorus(360)]);
		this.drag(false);
	}
	set tubeRadius(value) {
		this._tubeRadius = value != null ? value : 0.1;
		this._updateTransform();
	}
	get tubeRadius() {
		return this._tubeRadius;
	}
	set ringRadius(value) {
		this._ringRadius = value != null ? value : 0.1;
		this._updateTransform();
	}
	get ringRadius() {
		return this._ringRadius;
	}
	set tolerance(value) {
		this._tolerance = value;
		this._updateTransform();
	}
	get tolerance() {
		return this._tolerance;
	}
	_updateTransform() {
		this.meshTriDataList[0].setTris(this._createIntersectTorus());
		this.meshInstances[0].mesh = this._createRenderTorus(this._sectorAngle);
		this.meshInstances[1].mesh = this._createRenderTorus(360);
	}
	drag(state) {
		this.meshInstances[0].visible = !state;
		this.meshInstances[1].visible = state;
	}
	hide(state) {
		if (state) {
			this.meshInstances[0].visible = false;
			this.meshInstances[1].visible = false;
			return;
		}
		this.drag(false);
	}
}
class AxisPlane extends AxisShape {
	constructor(device, options = {}) {
		super(device, options);
		this._cull = CULLFACE_NONE;
		this._size = 0.2;
		this._gap = 0.1;
		this.meshTriDataList = [new MeshTriData(createPlane(this.device))];
		this._createPlane();
	}
	_getPosition() {
		const offset = this._size / 2 + this._gap;
		const position = new Vec3(offset, offset, offset);
		position[this.axis] = 0;
		return position;
	}
	_createPlane() {
		this._createRoot('plane');
		this._updateTransform();
		this._addRenderShadowMesh(this.entity, 'plane');
	}
	set size(value) {
		this._size = value != null ? value : 1;
		this._updateTransform();
	}
	get size() {
		return this._size;
	}
	set gap(value) {
		this._gap = value != null ? value : 0;
		this._updateTransform();
	}
	get gap() {
		return this._gap;
	}
	_updateTransform() {
		this.entity.setLocalPosition(this._getPosition());
		this.entity.setLocalEulerAngles(this._rotation);
		this.entity.setLocalScale(this._size, this._size, this._size);
	}
}

const tmpV1$1 = new Vec3();
const tmpV2$1 = new Vec3();
const tmpQ1$1 = new Quat();
class TranslateGizmo extends TransformGizmo {
	constructor(app, camera, layer) {
		super(app, camera, layer);
		this._shapes = {
			yz: new AxisPlane(this._device, {
				axis: 'x',
				flipAxis: 'y',
				layers: [this._layer.id],
				rotation: new Vec3(0, 0, -90),
				defaultColor: this._meshColors.axis.x,
				hoverColor: this._meshColors.hover.x
			}),
			xz: new AxisPlane(this._device, {
				axis: 'y',
				flipAxis: 'z',
				layers: [this._layer.id],
				rotation: new Vec3(0, 0, 0),
				defaultColor: this._meshColors.axis.y,
				hoverColor: this._meshColors.hover.y
			}),
			xy: new AxisPlane(this._device, {
				axis: 'z',
				flipAxis: 'x',
				layers: [this._layer.id],
				rotation: new Vec3(90, 0, 0),
				defaultColor: this._meshColors.axis.z,
				hoverColor: this._meshColors.hover.z
			}),
			x: new AxisArrow(this._device, {
				axis: 'x',
				layers: [this._layer.id],
				rotation: new Vec3(0, 0, -90),
				defaultColor: this._meshColors.axis.x,
				hoverColor: this._meshColors.hover.x
			}),
			y: new AxisArrow(this._device, {
				axis: 'y',
				layers: [this._layer.id],
				rotation: new Vec3(0, 0, 0),
				defaultColor: this._meshColors.axis.y,
				hoverColor: this._meshColors.hover.y
			}),
			z: new AxisArrow(this._device, {
				axis: 'z',
				layers: [this._layer.id],
				rotation: new Vec3(90, 0, 0),
				defaultColor: this._meshColors.axis.z,
				hoverColor: this._meshColors.hover.z
			})
		};
		this._nodeLocalPositions = new Map();
		this._nodePositions = new Map();
		this.snapIncrement = 1;
		this._createTransform();
		this.on('transform:start', () => {
			this._storeNodePositions();
		});
		this.on('transform:move', pointDelta => {
			if (this.snap) {
				pointDelta.mulScalar(1 / this.snapIncrement);
				pointDelta.round();
				pointDelta.mulScalar(this.snapIncrement);
			}
			this._setNodePositions(pointDelta);
		});
		this.on('nodes:detach', () => {
			this._nodeLocalPositions.clear();
			this._nodePositions.clear();
		});
	}
	set axisGap(value) {
		this._setArrowProp('gap', value);
	}
	get axisGap() {
		return this._shapes.x.gap;
	}
	set axisLineThickness(value) {
		this._setArrowProp('lineThickness', value);
	}
	get axisLineThickness() {
		return this._shapes.x.lineThickness;
	}
	set axisLineLength(value) {
		this._setArrowProp('lineLength', value);
	}
	get axisLineLength() {
		return this._shapes.x.lineLength;
	}
	set axisLineTolerance(value) {
		this._setArrowProp('tolerance', value);
	}
	get axisLineTolerance() {
		return this._shapes.x.tolerance;
	}
	set axisArrowThickness(value) {
		this._setArrowProp('arrowThickness', value);
	}
	get axisArrowThickness() {
		return this._shapes.x.arrowThickness;
	}
	set axisArrowLength(value) {
		this._setArrowProp('arrowLength', value);
	}
	get axisArrowLength() {
		return this._shapes.x.arrowLength;
	}
	set axisPlaneSize(value) {
		this._setPlaneProp('size', value);
	}
	get axisPlaneSize() {
		return this._shapes.yz.size;
	}
	set axisPlaneGap(value) {
		this._setPlaneProp('gap', value);
	}
	get axisPlaneGap() {
		return this._shapes.yz.gap;
	}
	_setArrowProp(prop, value) {
		this._shapes.x[prop] = value;
		this._shapes.y[prop] = value;
		this._shapes.z[prop] = value;
	}
	_setPlaneProp(prop, value) {
		this._shapes.yz[prop] = value;
		this._shapes.xz[prop] = value;
		this._shapes.xy[prop] = value;
	}
	_storeNodePositions() {
		for (let i = 0; i < this.nodes.length; i++) {
			const node = this.nodes[i];
			this._nodeLocalPositions.set(node, node.getLocalPosition().clone());
			this._nodePositions.set(node, node.getPosition().clone());
		}
	}
	_setNodePositions(pointDelta) {
		for (let i = 0; i < this.nodes.length; i++) {
			const node = this.nodes[i];
			if (this._coordSpace === GIZMO_LOCAL) {
				tmpV1$1.copy(pointDelta);
				node.parent.getWorldTransform().getScale(tmpV2$1);
				tmpV2$1.x = 1 / tmpV2$1.x;
				tmpV2$1.y = 1 / tmpV2$1.y;
				tmpV2$1.z = 1 / tmpV2$1.z;
				tmpQ1$1.copy(node.getLocalRotation()).transformVector(tmpV1$1, tmpV1$1);
				tmpV1$1.mul(tmpV2$1);
				node.setLocalPosition(this._nodeLocalPositions.get(node).clone().add(tmpV1$1));
			} else {
				node.setPosition(this._nodePositions.get(node).clone().add(pointDelta));
			}
		}
		this._updatePosition();
	}
}

const tmpV1 = new Vec3();
const tmpV2 = new Vec3();
const tmpM1 = new Mat4();
const tmpQ1 = new Quat();
const tmpQ2 = new Quat();
class RotateGizmo extends TransformGizmo {
	constructor(app, camera, layer) {
		super(app, camera, layer);
		this._shapes = {
			z: new AxisDisk(this._device, {
				axis: 'z',
				layers: [this._layer.id],
				rotation: new Vec3(90, 0, 90),
				defaultColor: this._meshColors.axis.z,
				hoverColor: this._meshColors.hover.z,
				sectorAngle: 180
			}),
			x: new AxisDisk(this._device, {
				axis: 'x',
				layers: [this._layer.id],
				rotation: new Vec3(0, 0, -90),
				defaultColor: this._meshColors.axis.x,
				hoverColor: this._meshColors.hover.x,
				sectorAngle: 180
			}),
			y: new AxisDisk(this._device, {
				axis: 'y',
				layers: [this._layer.id],
				rotation: new Vec3(0, 0, 0),
				defaultColor: this._meshColors.axis.y,
				hoverColor: this._meshColors.hover.y,
				sectorAngle: 180
			}),
			face: new AxisDisk(this._device, {
				axis: 'face',
				layers: [this._layer.id],
				rotation: this._getLookAtEulerAngles(this._camera.entity.getPosition()),
				defaultColor: this._meshColors.axis.face,
				hoverColor: this._meshColors.hover.face,
				ringRadius: 0.55
			})
		};
		this._isRotation = true;
		this._nodeLocalRotations = new Map();
		this._nodeRotations = new Map();
		this._nodeOffsets = new Map();
		this._guideAngleStartColor = new Color(0, 0, 0, 0.3);
		this._guideAngleStart = new Vec3();
		this._guideAngleEnd = new Vec3();
		this.snapIncrement = 5;
		this._createTransform();
		this.on('transform:start', () => {
			const axis = this._selectedAxis;
			const isFacing = axis === 'face';
			const scale = isFacing ? this.faceRingRadius : this.xyzRingRadius;
			this._storeNodeRotations();
			this._guideAngleStart.copy(this._selectionStartPoint).normalize();
			this._guideAngleStart.mulScalar(scale);
			this._gizmoRotationStart.transformVector(this._guideAngleStart, this._guideAngleStart);
			this._guideAngleEnd.copy(this._guideAngleStart);
			this._drag(true);
		});
		this.on('transform:move', (pointDelta, angleDelta) => {
			const gizmoPos = this.root.getPosition();
			const cameraPos = this._camera.entity.getPosition();
			const axis = this._selectedAxis;
			const isFacing = axis === 'face';
			if (this.snap) {
				angleDelta = Math.round(angleDelta / this.snapIncrement) * this.snapIncrement;
			}
			this._setNodeRotations(axis, angleDelta);
			tmpV1.set(0, 0, 0);
			if (isFacing) {
				tmpV1.copy(cameraPos).sub(gizmoPos).normalize();
			} else {
				tmpV1[axis] = 1;
			}
			this._gizmoRotationStart.transformVector(tmpV1, tmpV1);
			tmpQ1.setFromAxisAngle(tmpV1, angleDelta);
			tmpQ1.transformVector(this._guideAngleStart, this._guideAngleEnd);
		});
		this.on('transform:end', () => {
			this._drag(false);
		});
		this.on('nodes:detach', () => {
			this._nodeLocalRotations.clear();
			this._nodeRotations.clear();
			this._nodeOffsets.clear();
		});
		app.on('update', () => {
			this._faceAxisLookAtCamera();
			this._xyzAxisLookAtCamera();
			if (this._dragging) {
				const gizmoPos = this.root.getPosition();
				this._drawGuideAngleLine(gizmoPos, this._selectedAxis, this._guideAngleStart, this._guideAngleStartColor);
				this._drawGuideAngleLine(gizmoPos, this._selectedAxis, this._guideAngleEnd);
			}
		});
	}
	set xyzTubeRadius(value) {
		this._setDiskProp('tubeRadius', value);
	}
	get xyzTubeRadius() {
		return this._shapes.x.tubeRadius;
	}
	set xyzRingRadius(value) {
		this._setDiskProp('ringRadius', value);
	}
	get xyzRingRadius() {
		return this._shapes.x.ringRadius;
	}
	set faceTubeRadius(value) {
		this._shapes.face.tubeRadius = value;
	}
	get faceTubeRadius() {
		return this._shapes.face.tubeRadius;
	}
	set faceRingRadius(value) {
		this._shapes.face.ringRadius = value;
	}
	get faceRingRadius() {
		return this._shapes.face.ringRadius;
	}
	set ringTolerance(value) {
		this._setDiskProp('tolerance', value);
		this._shapes.face.tolerance = value;
	}
	get ringTolerance() {
		return this._shapes.x.tolerance;
	}
	_setDiskProp(prop, value) {
		this._shapes.x[prop] = value;
		this._shapes.y[prop] = value;
		this._shapes.z[prop] = value;
	}
	_drawGuideAngleLine(pos, axis, point, color = this._guideColors[axis]) {
		tmpV1.set(0, 0, 0);
		tmpV2.copy(point).mulScalar(this._scale);
		this._app.drawLine(tmpV1.add(pos), tmpV2.add(pos), color, false, this._layer);
	}
	_getLookAtEulerAngles(position) {
		tmpV1.set(0, 0, 0);
		tmpM1.setLookAt(tmpV1, position, Vec3.UP);
		tmpQ1.setFromMat4(tmpM1);
		tmpQ1.getEulerAngles(tmpV1);
		tmpV1.x += 90;
		return tmpV1;
	}
	_faceAxisLookAtCamera() {
		if (this._camera.projection === PROJECTION_PERSPECTIVE) {
			this._shapes.face.entity.lookAt(this._camera.entity.getPosition());
			this._shapes.face.entity.rotateLocal(90, 0, 0);
		} else {
			tmpQ1.copy(this._camera.entity.getRotation());
			tmpQ1.getEulerAngles(tmpV1);
			this._shapes.face.entity.setEulerAngles(tmpV1);
			this._shapes.face.entity.rotateLocal(-90, 0, 0);
		}
	}
	_xyzAxisLookAtCamera() {
		if (this._camera.projection === PROJECTION_PERSPECTIVE) {
			tmpV1.copy(this._camera.entity.getPosition()).sub(this.root.getPosition());
			tmpQ1.copy(this.root.getRotation()).invert().transformVector(tmpV1, tmpV1);
		} else {
			tmpV1.copy(this._camera.entity.forward).mulScalar(-1);
		}
		let angle = Math.atan2(tmpV1.z, tmpV1.y) * math.RAD_TO_DEG;
		this._shapes.x.entity.setLocalEulerAngles(0, angle - 90, -90);
		angle = Math.atan2(tmpV1.x, tmpV1.z) * math.RAD_TO_DEG;
		this._shapes.y.entity.setLocalEulerAngles(0, angle, 0);
		angle = Math.atan2(tmpV1.y, tmpV1.x) * math.RAD_TO_DEG;
		this._shapes.z.entity.setLocalEulerAngles(90, 0, angle + 90);
	}
	_drag(state) {
		for (const axis in this._shapes) {
			const shape = this._shapes[axis];
			if (axis === this._selectedAxis) {
				shape.drag(state);
			} else {
				shape.hide(state);
			}
		}
		this.fire('render:update');
	}
	_storeNodeRotations() {
		const gizmoPos = this.root.getPosition();
		for (let i = 0; i < this.nodes.length; i++) {
			const node = this.nodes[i];
			this._nodeLocalRotations.set(node, node.getLocalRotation().clone());
			this._nodeRotations.set(node, node.getRotation().clone());
			this._nodeOffsets.set(node, node.getPosition().clone().sub(gizmoPos));
		}
	}
	_setNodeRotations(axis, angleDelta) {
		const gizmoPos = this.root.getPosition();
		const cameraPos = this._camera.entity.getPosition();
		const isFacing = axis === 'face';
		for (let i = 0; i < this.nodes.length; i++) {
			const node = this.nodes[i];
			if (isFacing) {
				tmpV1.copy(cameraPos).sub(gizmoPos).normalize();
			} else {
				tmpV1.set(0, 0, 0);
				tmpV1[axis] = 1;
			}
			tmpQ1.setFromAxisAngle(tmpV1, angleDelta);
			if (!isFacing && this._coordSpace === GIZMO_LOCAL) {
				tmpQ2.copy(this._nodeLocalRotations.get(node)).mul(tmpQ1);
				node.setLocalRotation(tmpQ2);
			} else {
				tmpV1.copy(this._nodeOffsets.get(node));
				tmpQ1.transformVector(tmpV1, tmpV1);
				tmpQ2.copy(tmpQ1).mul(this._nodeRotations.get(node));
				node.setEulerAngles(tmpQ2.getEulerAngles());
				node.setPosition(tmpV1.add(gizmoPos));
			}
		}
		if (this._coordSpace === GIZMO_LOCAL) {
			this._updateRotation();
		}
	}
}

class ScaleGizmo extends TransformGizmo {
	constructor(app, camera, layer) {
		super(app, camera, layer);
		this._shapes = {
			xyz: new AxisBoxCenter(this._device, {
				axis: 'xyz',
				layers: [this._layer.id],
				defaultColor: this._meshColors.axis.xyz,
				hoverColor: this._meshColors.hover.xyz
			}),
			yz: new AxisPlane(this._device, {
				axis: 'x',
				flipAxis: 'y',
				layers: [this._layer.id],
				rotation: new Vec3(0, 0, -90),
				defaultColor: this._meshColors.axis.x,
				hoverColor: this._meshColors.hover.x
			}),
			xz: new AxisPlane(this._device, {
				axis: 'y',
				flipAxis: 'z',
				layers: [this._layer.id],
				rotation: new Vec3(0, 0, 0),
				defaultColor: this._meshColors.axis.y,
				hoverColor: this._meshColors.hover.y
			}),
			xy: new AxisPlane(this._device, {
				axis: 'z',
				flipAxis: 'x',
				layers: [this._layer.id],
				rotation: new Vec3(90, 0, 0),
				defaultColor: this._meshColors.axis.z,
				hoverColor: this._meshColors.hover.z
			}),
			x: new AxisBoxLine(this._device, {
				axis: 'x',
				layers: [this._layer.id],
				rotation: new Vec3(0, 0, -90),
				defaultColor: this._meshColors.axis.x,
				hoverColor: this._meshColors.hover.x
			}),
			y: new AxisBoxLine(this._device, {
				axis: 'y',
				layers: [this._layer.id],
				rotation: new Vec3(0, 0, 0),
				defaultColor: this._meshColors.axis.y,
				hoverColor: this._meshColors.hover.y
			}),
			z: new AxisBoxLine(this._device, {
				axis: 'z',
				layers: [this._layer.id],
				rotation: new Vec3(90, 0, 0),
				defaultColor: this._meshColors.axis.z,
				hoverColor: this._meshColors.hover.z
			})
		};
		this._coordSpace = GIZMO_LOCAL;
		this._nodeScales = new Map();
		this.snapIncrement = 1;
		this._createTransform();
		this.on('transform:start', () => {
			this._selectionStartPoint.sub(Vec3.ONE);
			this._storeNodeScales();
		});
		this.on('transform:move', pointDelta => {
			if (this.snap) {
				pointDelta.mulScalar(1 / this.snapIncrement);
				pointDelta.round();
				pointDelta.mulScalar(this.snapIncrement);
			}
			this._setNodeScales(pointDelta);
		});
		this.on('nodes:detach', () => {
			this._nodeScales.clear();
		});
	}
	set coordSpace(value) {}
	get coordSpace() {
		return this._coordSpace;
	}
	set uniform(value) {
		this._useUniformScaling = value != null ? value : true;
	}
	get uniform() {
		return this._useUniformScaling;
	}
	set axisGap(value) {
		this._setArrowProp('gap', value);
	}
	get axisGap() {
		return this._shapes.x.gap;
	}
	set axisLineThickness(value) {
		this._setArrowProp('lineThickness', value);
	}
	get axisLineThickness() {
		return this._shapes.x.lineThickness;
	}
	set axisLineLength(value) {
		this._setArrowProp('lineLength', value);
	}
	get axisLineLength() {
		return this._shapes.x.lineLength;
	}
	set axisLineTolerance(value) {
		this._setArrowProp('tolerance', value);
	}
	get axisLineTolerance() {
		return this._shapes.x.tolerance;
	}
	set axisBoxSize(value) {
		this._setArrowProp('boxSize', value);
	}
	get axisBoxSize() {
		return this._shapes.x.boxSize;
	}
	set axisPlaneSize(value) {
		this._setPlaneProp('size', value);
	}
	get axisPlaneSize() {
		return this._shapes.yz.size;
	}
	set axisPlaneGap(value) {
		this._setPlaneProp('gap', value);
	}
	get axisPlaneGap() {
		return this._shapes.yz.gap;
	}
	set axisCenterSize(value) {
		this._shapes.xyz.size = value;
	}
	get axisCenterSize() {
		return this._shapes.xyz.size;
	}
	set axisCenterTolerance(value) {
		this._shapes.xyz.tolerance = value;
	}
	get axisCenterTolerance() {
		return this._shapes.xyz.tolerance;
	}
	_setArrowProp(prop, value) {
		this._shapes.x[prop] = value;
		this._shapes.y[prop] = value;
		this._shapes.z[prop] = value;
	}
	_setPlaneProp(prop, value) {
		this._shapes.yz[prop] = value;
		this._shapes.xz[prop] = value;
		this._shapes.xy[prop] = value;
	}
	_storeNodeScales() {
		for (let i = 0; i < this.nodes.length; i++) {
			const node = this.nodes[i];
			this._nodeScales.set(node, node.getLocalScale().clone());
		}
	}
	_setNodeScales(pointDelta) {
		for (let i = 0; i < this.nodes.length; i++) {
			const node = this.nodes[i];
			node.setLocalScale(this._nodeScales.get(node).clone().mul(pointDelta));
		}
	}
}

export { GIZMO_LOCAL, GIZMO_WORLD, Gizmo, GltfExporter, MiniStats, RenderPassBloom, RenderPassCameraFrame, RenderPassCompose, RenderPassDownsample, RenderPassTAA, RenderPassUpsample, RotateGizmo, ScaleGizmo, TransformGizmo, TranslateGizmo, UsdzExporter };
