/**
 * @license
 * PlayCanvas Engine v1.69.2 revision 3e80480 (DEBUG)
 * Copyright 2011-2024 PlayCanvas Ltd. All rights reserved.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
(function (global, factory) {
	typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
	typeof define === 'function' && define.amd ? define(['exports'], factory) :
	(global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.pc = {}));
})(this, (function (exports) { 'use strict';

	var _documentCurrentScript = typeof document !== 'undefined' ? document.currentScript : null;
	/**
	 * A short hand function to polyfill prototype methods which are not iterated in e.g. for-in loops.
	 * 
	 * @param {ObjectConstructor} cls 
	 * @param {string} name 
	 * @param {Function} func 
	 * @ignore
	 */
	function defineProtoFunc(cls, name, func) {
	  if (!cls.prototype[name]) {
	    Object.defineProperty(cls.prototype, name, {
	      value: func,
	      configurable: true,
	      enumerable: false,
	      writable: true
	    });
	  }
	}

	// https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array/fill#polyfill
	defineProtoFunc(Array, 'fill', function (value) {
	  // Steps 1-2.
	  if (this == null) {
	    throw new TypeError('this is null or not defined');
	  }
	  var O = Object(this);

	  // Steps 3-5.
	  var len = O.length >>> 0;

	  // Steps 6-7.
	  var start = arguments[1];
	  var relativeStart = start >> 0;

	  // Step 8.
	  var k = relativeStart < 0 ? Math.max(len + relativeStart, 0) : Math.min(relativeStart, len);

	  // Steps 9-10.
	  var end = arguments[2];
	  var relativeEnd = end === undefined ? len : end >> 0;

	  // Step 11.
	  var finalValue = relativeEnd < 0 ? Math.max(len + relativeEnd, 0) : Math.min(relativeEnd, len);

	  // Step 12.
	  while (k < finalValue) {
	    O[k] = value;
	    k++;
	  }

	  // Step 13.
	  return O;
	});

	// https://tc39.github.io/ecma262/#sec-array.prototype.find
	defineProtoFunc(Array, 'find', function (predicate) {
	  // 1. Let O be ? ToObject(this value).
	  if (this == null) {
	    throw TypeError('"this" is null or not defined');
	  }
	  var o = Object(this);

	  // 2. Let len be ? ToLength(? Get(O, "length")).
	  var len = o.length >>> 0;

	  // 3. If IsCallable(predicate) is false, throw a TypeError exception.
	  if (typeof predicate !== 'function') {
	    throw TypeError('predicate must be a function');
	  }

	  // 4. If thisArg was supplied, let T be thisArg; else let T be undefined.
	  var thisArg = arguments[1];

	  // 5. Let k be 0.
	  var k = 0;

	  // 6. Repeat, while k < len
	  while (k < len) {
	    // a. Let Pk be ! ToString(k).
	    // b. Let kValue be ? Get(O, Pk).
	    // c. Let testResult be ToBoolean(? Call(predicate, T, « kValue, k, O »)).
	    // d. If testResult is true, return kValue.
	    var kValue = o[k];
	    if (predicate.call(thisArg, kValue, k, o)) {
	      return kValue;
	    }
	    // e. Increase k by 1.
	    k++;
	  }

	  // 7. Return undefined.
	  return undefined;
	});

	// https://tc39.github.io/ecma262/#sec-array.prototype.findindex
	defineProtoFunc(Array, 'findIndex', function (predicate) {
	  // 1. Let O be ? ToObject(this value).
	  if (this == null) {
	    throw new TypeError('"this" is null or not defined');
	  }
	  var o = Object(this);

	  // 2. Let len be ? ToLength(? Get(O, "length")).
	  var len = o.length >>> 0;

	  // 3. If IsCallable(predicate) is false, throw a TypeError exception.
	  if (typeof predicate !== 'function') {
	    throw new TypeError('predicate must be a function');
	  }

	  // 4. If thisArg was supplied, let T be thisArg; else let T be undefined.
	  var thisArg = arguments[1];

	  // 5. Let k be 0.
	  var k = 0;

	  // 6. Repeat, while k < len
	  while (k < len) {
	    // a. Let Pk be ! ToString(k).
	    // b. Let kValue be ? Get(O, Pk).
	    // c. Let testResult be ToBoolean(? Call(predicate, T, « kValue, k, O »)).
	    // d. If testResult is true, return k.
	    var kValue = o[k];
	    if (predicate.call(thisArg, kValue, k, o)) {
	      return k;
	    }
	    // e. Increase k by 1.
	    k++;
	  }

	  // 7. Return -1.
	  return -1;
	});

	// https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Math/log2#Polyfill
	Math.log2 = Math.log2 || function (x) {
	  return Math.log(x) * Math.LOG2E;
	};

	// https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Math/sign#Polyfill
	if (!Math.sign) {
	  Math.sign = function (x) {
	    // If x is NaN, the result is NaN.
	    // If x is -0, the result is -0.
	    // If x is +0, the result is +0.
	    // If x is negative and not -0, the result is -1.
	    // If x is positive and not +0, the result is +1.
	    return (x > 0) - (x < 0) || +x;
	    // A more aesthetic pseudo-representation:
	    //
	    // ( (x > 0) ? 1 : 0 )  // if x is positive, then positive one
	    //          +           // else (because you can't be both - and +)
	    // ( (x < 0) ? -1 : 0 ) // if x is negative, then negative one
	    //         ||           // if x is 0, -0, or NaN, or not a number,
	    //         +x           // then the result will be x, (or) if x is
	    //                      // not a number, then x converts to number
	  };
	}

	// https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Number/isFinite#polyfill
	if (Number.isFinite === undefined) Number.isFinite = function (value) {
	  return typeof value === 'number' && isFinite(value);
	};

	// https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Object/assign#Polyfill
	if (typeof Object.assign != 'function') {
	  // Must be writable: true, enumerable: false, configurable: true
	  Object.defineProperty(Object, "assign", {
	    value: function assign(target, varArgs) {

	      if (target == null) {
	        // TypeError if undefined or null
	        throw new TypeError('Cannot convert undefined or null to object');
	      }
	      var to = Object(target);
	      for (var index = 1; index < arguments.length; index++) {
	        var nextSource = arguments[index];
	        if (nextSource != null) {
	          // Skip over if undefined or null
	          for (var nextKey in nextSource) {
	            // Avoid bugs when hasOwnProperty is shadowed
	            if (Object.prototype.hasOwnProperty.call(nextSource, nextKey)) {
	              to[nextKey] = nextSource[nextKey];
	            }
	          }
	        }
	      }
	      return to;
	    },
	    writable: true,
	    configurable: true
	  });
	}

	// https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Object/fromEntries
	// https://stackoverflow.com/questions/68654735/ie11-compatible-object-fromentries
	Object.fromEntries = Object.fromEntries || function fromEntries(entries) {
	  if (!entries || !entries[Symbol.iterator]) {
	    throw new Error('Object.fromEntries() requires a single iterable argument');
	  }
	  var res = {};
	  for (var i = 0; i < entries.length; i++) {
	    res[entries[i][0]] = entries[i][1];
	  }
	  return res;
	};
	Object.entries = Object.entries || function (obj) {
	  var ownProps = Object.keys(obj),
	    i = ownProps.length,
	    resArray = new Array(i); // preallocate the Array
	  while (i--) resArray[i] = [ownProps[i], obj[ownProps[i]]];
	  return resArray;
	};

	Object.values = Object.values || function (object) {
	  return Object.keys(object).map(function (key) {
	    return object[key];
	  });
	};

	// Apply PointerLock shims
	(function () {
	  // Old API
	  if (typeof navigator === 'undefined' || typeof document === 'undefined') {
	    // Not running in a browser
	    return;
	  }
	  navigator.pointer = navigator.pointer || navigator.webkitPointer || navigator.mozPointer;

	  // Events
	  var pointerlockchange = function pointerlockchange() {
	    var e = document.createEvent('CustomEvent');
	    e.initCustomEvent('pointerlockchange', true, false, null);
	    document.dispatchEvent(e);
	  };
	  var pointerlockerror = function pointerlockerror() {
	    var e = document.createEvent('CustomEvent');
	    e.initCustomEvent('pointerlockerror', true, false, null);
	    document.dispatchEvent(e);
	  };
	  document.addEventListener('webkitpointerlockchange', pointerlockchange, false);
	  document.addEventListener('webkitpointerlocklost', pointerlockchange, false);
	  document.addEventListener('mozpointerlockchange', pointerlockchange, false);
	  document.addEventListener('mozpointerlocklost', pointerlockchange, false);
	  document.addEventListener('webkitpointerlockerror', pointerlockerror, false);
	  document.addEventListener('mozpointerlockerror', pointerlockerror, false);

	  // requestPointerLock
	  if (Element.prototype.mozRequestPointerLock) {
	    // FF requires a new function for some reason
	    Element.prototype.requestPointerLock = function () {
	      this.mozRequestPointerLock();
	    };
	  } else {
	    Element.prototype.requestPointerLock = Element.prototype.requestPointerLock || Element.prototype.webkitRequestPointerLock || Element.prototype.mozRequestPointerLock;
	  }
	  if (!Element.prototype.requestPointerLock && navigator.pointer) {
	    Element.prototype.requestPointerLock = function () {
	      var el = this;
	      document.pointerLockElement = el;
	      navigator.pointer.lock(el, pointerlockchange, pointerlockerror);
	    };
	  }

	  // exitPointerLock
	  document.exitPointerLock = document.exitPointerLock || document.webkitExitPointerLock || document.mozExitPointerLock;
	  if (!document.exitPointerLock) {
	    document.exitPointerLock = function () {
	      if (navigator.pointer) {
	        document.pointerLockElement = null;
	        navigator.pointer.unlock();
	      }
	    };
	  }
	})();

	// https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/String/endsWith#Polyfill
	defineProtoFunc(String, 'endsWith', function (search, this_len) {
	  if (this_len === undefined || this_len > this.length) {
	    this_len = this.length;
	  }
	  return this.substring(this_len - search.length, this_len) === search;
	});

	// https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/String/includes#Polyfill
	defineProtoFunc(String, 'includes', function (search, start) {

	  if (typeof start !== 'number') {
	    start = 0;
	  }
	  if (start + search.length > this.length) {
	    return false;
	  } else {
	    return this.indexOf(search, start) !== -1;
	  }
	});

	// https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/String/startsWith#Polyfill
	defineProtoFunc(String, 'startsWith', function (search, rawPos) {
	  var pos = rawPos > 0 ? rawPos | 0 : 0;
	  return this.substring(pos, pos + search.length) === search;
	});

	// https://vanillajstoolkit.com/polyfills/stringtrimend/
	defineProtoFunc(String, 'trimEnd', function () {
	  return this.replace(new RegExp(/[\x09\x0A\x0B\x0C\x0D\x20\xA0\u1680\u2000\u2001\u2002\u2003\u2004\u2005\u2006\u2007\u2008\u2009\u200A\u202F\u205F\u3000\u2028\u2029\uFEFF]+/.source + '$', 'g'), '');
	});

	var typedArrays = [Int8Array, Uint8Array, Uint8ClampedArray, Int16Array, Uint16Array, Int32Array, Uint32Array, Float32Array];
	for (var _i = 0, _typedArrays = typedArrays; _i < _typedArrays.length; _i++) {
	  var typedArray = _typedArrays[_i];
	  // https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/TypedArray/fill#polyfill
	  defineProtoFunc(typedArray, "fill", Array.prototype.fill);
	  defineProtoFunc(typedArray, "join", Array.prototype.join);
	}

	/**
	 * Logs a frame number.
	 *
	 * @type {string}
	 * @category Debug
	 */
	var TRACEID_RENDER_FRAME = 'RenderFrame';

	/**
	 * Logs a frame time.
	 *
	 * @type {string}
	 * @category Debug
	 */
	var TRACEID_RENDER_FRAME_TIME = 'RenderFrameTime';

	/**
	 * Logs basic information about generated render passes.
	 *
	 * @type {string}
	 * @category Debug
	 */
	var TRACEID_RENDER_PASS = 'RenderPass';

	/**
	 * Logs additional detail for render passes.
	 *
	 * @type {string}
	 * @category Debug
	 */
	var TRACEID_RENDER_PASS_DETAIL = 'RenderPassDetail';

	/**
	 * Logs render actions created by the layer composition. Only executes when the
	 * layer composition changes.
	 *
	 * @type {string}
	 * @category Debug
	 */
	var TRACEID_RENDER_ACTION = 'RenderAction';

	/**
	 * Logs the allocation of render targets.
	 *
	 * @type {string}
	 * @category Debug
	 */
	var TRACEID_RENDER_TARGET_ALLOC = 'RenderTargetAlloc';

	/**
	 * Logs the allocation of textures.
	 *
	 * @type {string}
	 * @category Debug
	 */
	var TRACEID_TEXTURE_ALLOC = 'TextureAlloc';

	/**
	 * Logs the creation of shaders.
	 *
	 * @type {string}
	 * @category Debug
	 */
	var TRACEID_SHADER_ALLOC = 'ShaderAlloc';

	/**
	 * Logs the compilation time of shaders.
	 *
	 * @type {string}
	 * @category Debug
	 */
	var TRACEID_SHADER_COMPILE = 'ShaderCompile';

	/**
	 * Logs the vram use by the textures.
	 *
	 * @type {string}
	 * @category Debug
	 */
	var TRACEID_VRAM_TEXTURE = 'VRAM.Texture';

	/**
	 * Logs the vram use by the vertex buffers.
	 *
	 * @type {string}
	 * @category Debug
	 */
	var TRACEID_VRAM_VB = 'VRAM.Vb';

	/**
	 * Logs the vram use by the index buffers.
	 *
	 * @type {string}
	 * @category Debug
	 */
	var TRACEID_VRAM_IB = 'VRAM.Ib';

	/**
	 * Logs the creation of bind groups.
	 *
	 * @type {string}
	 * @category Debug
	 */
	var TRACEID_BINDGROUP_ALLOC = 'BindGroupAlloc';

	/**
	 * Logs the creation of bind group formats.
	 *
	 * @type {string}
	 * @category Debug
	 */
	var TRACEID_BINDGROUPFORMAT_ALLOC = 'BindGroupFormatAlloc';

	/**
	 * Logs the creation of render pipelines. WebBPU only.
	 *
	 * @type {string}
	 * @category Debug
	 */
	var TRACEID_RENDERPIPELINE_ALLOC = 'RenderPipelineAlloc';

	/**
	 * Logs the creation of compute pipelines. WebGPU only.
	 *
	 * @type {string}
	 * @category Debug
	 */
	var TRACEID_COMPUTEPIPELINE_ALLOC = 'ComputePipelineAlloc';

	/**
	 * Logs the creation of pipeline layouts. WebBPU only.
	 *
	 * @type {string}
	 * @category Debug
	 */
	var TRACEID_PIPELINELAYOUT_ALLOC = 'PipelineLayoutAlloc';

	/**
	 * Logs the internal debug information for Elements.
	 *
	 * @type {string}
	 * @category Debug
	 */
	var TRACE_ID_ELEMENT = "Element";

	/**
	 * Logs the vram use by all textures in memory.
	 *
	 * @type {string}
	 * @category Debug
	 */
	var TRACEID_TEXTURES = 'Textures';

	/**
	 * Logs the render queue commands.
	 *
	 * @type {string}
	 * @category Debug
	 */
	var TRACEID_RENDER_QUEUE = 'RenderQueue';

	/**
	 * Logs the GPU timings.
	 *
	 * @type {string}
	 * @category Debug
	 */
	var TRACEID_GPU_TIMINGS = 'GpuTimings';

	/**
	 * @name pc
	 * @namespace
	 * @description Root namespace for the PlayCanvas Engine.
	 */

	/**
	 * The engine version number. This is in semantic versioning format (MAJOR.MINOR.PATCH).
	 */
	var version = '1.69.2';

	/**
	 * The engine revision number. This is the Git hash of the last commit made to the branch
	 * from which the engine was built.
	 */
	var revision = '3e80480';
	var config = {};
	var common = {};
	var apps = {}; // Storage for the applications using the PlayCanvas Engine
	var data$1 = {}; // Storage for exported entity data

	var typeofs = ['undefined', 'number', 'string', 'boolean'];
	var objectTypes = {
	  '[object Array]': 'array',
	  '[object Object]': 'object',
	  '[object Function]': 'function',
	  '[object Date]': 'date',
	  '[object RegExp]': 'regexp',
	  '[object Float32Array]': 'float32array'
	};

	/**
	 * Extended typeof() function, returns the type of the object.
	 *
	 * @param {object} obj - The object to get the type of.
	 * @returns {string} The type string: "null", "undefined", "number", "string", "boolean", "array", "object", "function", "date", "regexp" or "float32array".
	 * @ignore
	 */
	function type$1(obj) {
	  if (obj === null) {
	    return 'null';
	  }
	  var typeString = typeof obj;
	  if (typeofs.includes(typeString)) {
	    return typeString;
	  }
	  return objectTypes[Object.prototype.toString.call(obj)];
	}

	/**
	 * Merge the contents of two objects into a single object.
	 *
	 * @param {object} target - The target object of the merge.
	 * @param {object} ex - The object that is merged with target.
	 * @returns {object} The target object.
	 * @example
	 * const A = {
	 *     a: function () {
	 *         console.log(this.a);
	 *     }
	 * };
	 * const B = {
	 *     b: function () {
	 *         console.log(this.b);
	 *     }
	 * };
	 *
	 * pc.extend(A, B);
	 * A.a();
	 * // logs "a"
	 * A.b();
	 * // logs "b"
	 * @ignore
	 */
	function extend(target, ex) {
	  for (var prop in ex) {
	    var copy = ex[prop];
	    if (type$1(copy) === 'object') {
	      target[prop] = extend({}, copy);
	    } else if (type$1(copy) === 'array') {
	      target[prop] = extend([], copy);
	    } else {
	      target[prop] = copy;
	    }
	  }
	  return target;
	}

	function _regeneratorRuntime() {
	  _regeneratorRuntime = function () {
	    return e;
	  };
	  var t,
	    e = {},
	    r = Object.prototype,
	    n = r.hasOwnProperty,
	    o = Object.defineProperty || function (t, e, r) {
	      t[e] = r.value;
	    },
	    i = "function" == typeof Symbol ? Symbol : {},
	    a = i.iterator || "@@iterator",
	    c = i.asyncIterator || "@@asyncIterator",
	    u = i.toStringTag || "@@toStringTag";
	  function define(t, e, r) {
	    return Object.defineProperty(t, e, {
	      value: r,
	      enumerable: !0,
	      configurable: !0,
	      writable: !0
	    }), t[e];
	  }
	  try {
	    define({}, "");
	  } catch (t) {
	    define = function (t, e, r) {
	      return t[e] = r;
	    };
	  }
	  function wrap(t, e, r, n) {
	    var i = e && e.prototype instanceof Generator ? e : Generator,
	      a = Object.create(i.prototype),
	      c = new Context(n || []);
	    return o(a, "_invoke", {
	      value: makeInvokeMethod(t, r, c)
	    }), a;
	  }
	  function tryCatch(t, e, r) {
	    try {
	      return {
	        type: "normal",
	        arg: t.call(e, r)
	      };
	    } catch (t) {
	      return {
	        type: "throw",
	        arg: t
	      };
	    }
	  }
	  e.wrap = wrap;
	  var h = "suspendedStart",
	    l = "suspendedYield",
	    f = "executing",
	    s = "completed",
	    y = {};
	  function Generator() {}
	  function GeneratorFunction() {}
	  function GeneratorFunctionPrototype() {}
	  var p = {};
	  define(p, a, function () {
	    return this;
	  });
	  var d = Object.getPrototypeOf,
	    v = d && d(d(values([])));
	  v && v !== r && n.call(v, a) && (p = v);
	  var g = GeneratorFunctionPrototype.prototype = Generator.prototype = Object.create(p);
	  function defineIteratorMethods(t) {
	    ["next", "throw", "return"].forEach(function (e) {
	      define(t, e, function (t) {
	        return this._invoke(e, t);
	      });
	    });
	  }
	  function AsyncIterator(t, e) {
	    function invoke(r, o, i, a) {
	      var c = tryCatch(t[r], t, o);
	      if ("throw" !== c.type) {
	        var u = c.arg,
	          h = u.value;
	        return h && "object" == typeof h && n.call(h, "__await") ? e.resolve(h.__await).then(function (t) {
	          invoke("next", t, i, a);
	        }, function (t) {
	          invoke("throw", t, i, a);
	        }) : e.resolve(h).then(function (t) {
	          u.value = t, i(u);
	        }, function (t) {
	          return invoke("throw", t, i, a);
	        });
	      }
	      a(c.arg);
	    }
	    var r;
	    o(this, "_invoke", {
	      value: function (t, n) {
	        function callInvokeWithMethodAndArg() {
	          return new e(function (e, r) {
	            invoke(t, n, e, r);
	          });
	        }
	        return r = r ? r.then(callInvokeWithMethodAndArg, callInvokeWithMethodAndArg) : callInvokeWithMethodAndArg();
	      }
	    });
	  }
	  function makeInvokeMethod(e, r, n) {
	    var o = h;
	    return function (i, a) {
	      if (o === f) throw new Error("Generator is already running");
	      if (o === s) {
	        if ("throw" === i) throw a;
	        return {
	          value: t,
	          done: !0
	        };
	      }
	      for (n.method = i, n.arg = a;;) {
	        var c = n.delegate;
	        if (c) {
	          var u = maybeInvokeDelegate(c, n);
	          if (u) {
	            if (u === y) continue;
	            return u;
	          }
	        }
	        if ("next" === n.method) n.sent = n._sent = n.arg;else if ("throw" === n.method) {
	          if (o === h) throw o = s, n.arg;
	          n.dispatchException(n.arg);
	        } else "return" === n.method && n.abrupt("return", n.arg);
	        o = f;
	        var p = tryCatch(e, r, n);
	        if ("normal" === p.type) {
	          if (o = n.done ? s : l, p.arg === y) continue;
	          return {
	            value: p.arg,
	            done: n.done
	          };
	        }
	        "throw" === p.type && (o = s, n.method = "throw", n.arg = p.arg);
	      }
	    };
	  }
	  function maybeInvokeDelegate(e, r) {
	    var n = r.method,
	      o = e.iterator[n];
	    if (o === t) return r.delegate = null, "throw" === n && e.iterator.return && (r.method = "return", r.arg = t, maybeInvokeDelegate(e, r), "throw" === r.method) || "return" !== n && (r.method = "throw", r.arg = new TypeError("The iterator does not provide a '" + n + "' method")), y;
	    var i = tryCatch(o, e.iterator, r.arg);
	    if ("throw" === i.type) return r.method = "throw", r.arg = i.arg, r.delegate = null, y;
	    var a = i.arg;
	    return a ? a.done ? (r[e.resultName] = a.value, r.next = e.nextLoc, "return" !== r.method && (r.method = "next", r.arg = t), r.delegate = null, y) : a : (r.method = "throw", r.arg = new TypeError("iterator result is not an object"), r.delegate = null, y);
	  }
	  function pushTryEntry(t) {
	    var e = {
	      tryLoc: t[0]
	    };
	    1 in t && (e.catchLoc = t[1]), 2 in t && (e.finallyLoc = t[2], e.afterLoc = t[3]), this.tryEntries.push(e);
	  }
	  function resetTryEntry(t) {
	    var e = t.completion || {};
	    e.type = "normal", delete e.arg, t.completion = e;
	  }
	  function Context(t) {
	    this.tryEntries = [{
	      tryLoc: "root"
	    }], t.forEach(pushTryEntry, this), this.reset(!0);
	  }
	  function values(e) {
	    if (e || "" === e) {
	      var r = e[a];
	      if (r) return r.call(e);
	      if ("function" == typeof e.next) return e;
	      if (!isNaN(e.length)) {
	        var o = -1,
	          i = function next() {
	            for (; ++o < e.length;) if (n.call(e, o)) return next.value = e[o], next.done = !1, next;
	            return next.value = t, next.done = !0, next;
	          };
	        return i.next = i;
	      }
	    }
	    throw new TypeError(typeof e + " is not iterable");
	  }
	  return GeneratorFunction.prototype = GeneratorFunctionPrototype, o(g, "constructor", {
	    value: GeneratorFunctionPrototype,
	    configurable: !0
	  }), o(GeneratorFunctionPrototype, "constructor", {
	    value: GeneratorFunction,
	    configurable: !0
	  }), GeneratorFunction.displayName = define(GeneratorFunctionPrototype, u, "GeneratorFunction"), e.isGeneratorFunction = function (t) {
	    var e = "function" == typeof t && t.constructor;
	    return !!e && (e === GeneratorFunction || "GeneratorFunction" === (e.displayName || e.name));
	  }, e.mark = function (t) {
	    return Object.setPrototypeOf ? Object.setPrototypeOf(t, GeneratorFunctionPrototype) : (t.__proto__ = GeneratorFunctionPrototype, define(t, u, "GeneratorFunction")), t.prototype = Object.create(g), t;
	  }, e.awrap = function (t) {
	    return {
	      __await: t
	    };
	  }, defineIteratorMethods(AsyncIterator.prototype), define(AsyncIterator.prototype, c, function () {
	    return this;
	  }), e.AsyncIterator = AsyncIterator, e.async = function (t, r, n, o, i) {
	    void 0 === i && (i = Promise);
	    var a = new AsyncIterator(wrap(t, r, n, o), i);
	    return e.isGeneratorFunction(r) ? a : a.next().then(function (t) {
	      return t.done ? t.value : a.next();
	    });
	  }, defineIteratorMethods(g), define(g, u, "Generator"), define(g, a, function () {
	    return this;
	  }), define(g, "toString", function () {
	    return "[object Generator]";
	  }), e.keys = function (t) {
	    var e = Object(t),
	      r = [];
	    for (var n in e) r.push(n);
	    return r.reverse(), function next() {
	      for (; r.length;) {
	        var t = r.pop();
	        if (t in e) return next.value = t, next.done = !1, next;
	      }
	      return next.done = !0, next;
	    };
	  }, e.values = values, Context.prototype = {
	    constructor: Context,
	    reset: function (e) {
	      if (this.prev = 0, this.next = 0, this.sent = this._sent = t, this.done = !1, this.delegate = null, this.method = "next", this.arg = t, this.tryEntries.forEach(resetTryEntry), !e) for (var r in this) "t" === r.charAt(0) && n.call(this, r) && !isNaN(+r.slice(1)) && (this[r] = t);
	    },
	    stop: function () {
	      this.done = !0;
	      var t = this.tryEntries[0].completion;
	      if ("throw" === t.type) throw t.arg;
	      return this.rval;
	    },
	    dispatchException: function (e) {
	      if (this.done) throw e;
	      var r = this;
	      function handle(n, o) {
	        return a.type = "throw", a.arg = e, r.next = n, o && (r.method = "next", r.arg = t), !!o;
	      }
	      for (var o = this.tryEntries.length - 1; o >= 0; --o) {
	        var i = this.tryEntries[o],
	          a = i.completion;
	        if ("root" === i.tryLoc) return handle("end");
	        if (i.tryLoc <= this.prev) {
	          var c = n.call(i, "catchLoc"),
	            u = n.call(i, "finallyLoc");
	          if (c && u) {
	            if (this.prev < i.catchLoc) return handle(i.catchLoc, !0);
	            if (this.prev < i.finallyLoc) return handle(i.finallyLoc);
	          } else if (c) {
	            if (this.prev < i.catchLoc) return handle(i.catchLoc, !0);
	          } else {
	            if (!u) throw new Error("try statement without catch or finally");
	            if (this.prev < i.finallyLoc) return handle(i.finallyLoc);
	          }
	        }
	      }
	    },
	    abrupt: function (t, e) {
	      for (var r = this.tryEntries.length - 1; r >= 0; --r) {
	        var o = this.tryEntries[r];
	        if (o.tryLoc <= this.prev && n.call(o, "finallyLoc") && this.prev < o.finallyLoc) {
	          var i = o;
	          break;
	        }
	      }
	      i && ("break" === t || "continue" === t) && i.tryLoc <= e && e <= i.finallyLoc && (i = null);
	      var a = i ? i.completion : {};
	      return a.type = t, a.arg = e, i ? (this.method = "next", this.next = i.finallyLoc, y) : this.complete(a);
	    },
	    complete: function (t, e) {
	      if ("throw" === t.type) throw t.arg;
	      return "break" === t.type || "continue" === t.type ? this.next = t.arg : "return" === t.type ? (this.rval = this.arg = t.arg, this.method = "return", this.next = "end") : "normal" === t.type && e && (this.next = e), y;
	    },
	    finish: function (t) {
	      for (var e = this.tryEntries.length - 1; e >= 0; --e) {
	        var r = this.tryEntries[e];
	        if (r.finallyLoc === t) return this.complete(r.completion, r.afterLoc), resetTryEntry(r), y;
	      }
	    },
	    catch: function (t) {
	      for (var e = this.tryEntries.length - 1; e >= 0; --e) {
	        var r = this.tryEntries[e];
	        if (r.tryLoc === t) {
	          var n = r.completion;
	          if ("throw" === n.type) {
	            var o = n.arg;
	            resetTryEntry(r);
	          }
	          return o;
	        }
	      }
	      throw new Error("illegal catch attempt");
	    },
	    delegateYield: function (e, r, n) {
	      return this.delegate = {
	        iterator: values(e),
	        resultName: r,
	        nextLoc: n
	      }, "next" === this.method && (this.arg = t), y;
	    }
	  }, e;
	}
	function asyncGeneratorStep(gen, resolve, reject, _next, _throw, key, arg) {
	  try {
	    var info = gen[key](arg);
	    var value = info.value;
	  } catch (error) {
	    reject(error);
	    return;
	  }
	  if (info.done) {
	    resolve(value);
	  } else {
	    Promise.resolve(value).then(_next, _throw);
	  }
	}
	function _asyncToGenerator(fn) {
	  return function () {
	    var self = this,
	      args = arguments;
	    return new Promise(function (resolve, reject) {
	      var gen = fn.apply(self, args);
	      function _next(value) {
	        asyncGeneratorStep(gen, resolve, reject, _next, _throw, "next", value);
	      }
	      function _throw(err) {
	        asyncGeneratorStep(gen, resolve, reject, _next, _throw, "throw", err);
	      }
	      _next(undefined);
	    });
	  };
	}
	function _defineProperties(target, props) {
	  for (var i = 0; i < props.length; i++) {
	    var descriptor = props[i];
	    descriptor.enumerable = descriptor.enumerable || false;
	    descriptor.configurable = true;
	    if ("value" in descriptor) descriptor.writable = true;
	    Object.defineProperty(target, _toPropertyKey(descriptor.key), descriptor);
	  }
	}
	function _createClass(Constructor, protoProps, staticProps) {
	  if (protoProps) _defineProperties(Constructor.prototype, protoProps);
	  if (staticProps) _defineProperties(Constructor, staticProps);
	  Object.defineProperty(Constructor, "prototype", {
	    writable: false
	  });
	  return Constructor;
	}
	function _extends() {
	  _extends = Object.assign ? Object.assign.bind() : function (target) {
	    for (var i = 1; i < arguments.length; i++) {
	      var source = arguments[i];
	      for (var key in source) {
	        if (Object.prototype.hasOwnProperty.call(source, key)) {
	          target[key] = source[key];
	        }
	      }
	    }
	    return target;
	  };
	  return _extends.apply(this, arguments);
	}
	function _inheritsLoose(subClass, superClass) {
	  subClass.prototype = Object.create(superClass.prototype);
	  subClass.prototype.constructor = subClass;
	  _setPrototypeOf(subClass, superClass);
	}
	function _setPrototypeOf(o, p) {
	  _setPrototypeOf = Object.setPrototypeOf ? Object.setPrototypeOf.bind() : function _setPrototypeOf(o, p) {
	    o.__proto__ = p;
	    return o;
	  };
	  return _setPrototypeOf(o, p);
	}
	function _assertThisInitialized(self) {
	  if (self === void 0) {
	    throw new ReferenceError("this hasn't been initialised - super() hasn't been called");
	  }
	  return self;
	}
	function _unsupportedIterableToArray(o, minLen) {
	  if (!o) return;
	  if (typeof o === "string") return _arrayLikeToArray(o, minLen);
	  var n = Object.prototype.toString.call(o).slice(8, -1);
	  if (n === "Object" && o.constructor) n = o.constructor.name;
	  if (n === "Map" || n === "Set") return Array.from(o);
	  if (n === "Arguments" || /^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(n)) return _arrayLikeToArray(o, minLen);
	}
	function _arrayLikeToArray(arr, len) {
	  if (len == null || len > arr.length) len = arr.length;
	  for (var i = 0, arr2 = new Array(len); i < len; i++) arr2[i] = arr[i];
	  return arr2;
	}
	function _createForOfIteratorHelperLoose(o, allowArrayLike) {
	  var it = typeof Symbol !== "undefined" && o[Symbol.iterator] || o["@@iterator"];
	  if (it) return (it = it.call(o)).next.bind(it);
	  if (Array.isArray(o) || (it = _unsupportedIterableToArray(o)) || allowArrayLike && o && typeof o.length === "number") {
	    if (it) o = it;
	    var i = 0;
	    return function () {
	      if (i >= o.length) return {
	        done: true
	      };
	      return {
	        done: false,
	        value: o[i++]
	      };
	    };
	  }
	  throw new TypeError("Invalid attempt to iterate non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method.");
	}
	function _toPrimitive(input, hint) {
	  if (typeof input !== "object" || input === null) return input;
	  var prim = input[Symbol.toPrimitive];
	  if (prim !== undefined) {
	    var res = prim.call(input, hint || "default");
	    if (typeof res !== "object") return res;
	    throw new TypeError("@@toPrimitive must return a primitive value.");
	  }
	  return (hint === "string" ? String : Number)(input);
	}
	function _toPropertyKey(arg) {
	  var key = _toPrimitive(arg, "string");
	  return typeof key === "symbol" ? key : String(key);
	}

	/**
	 * Log tracing functionality, allowing for tracing of the internal functionality of the engine.
	 * Note that the trace logging only takes place in the debug build of the engine and is stripped
	 * out in other builds.
	 *
	 * @category Debug
	 */
	var Tracing = /*#__PURE__*/function () {
	  function Tracing() {}
	  /**
	   * Enable or disable a trace channel.
	   *
	   * @param {string} channel - Name of the trace channel. Can be:
	   *
	   * - {@link TRACEID_RENDER_FRAME}
	   * - {@link TRACEID_RENDER_FRAME_TIME}
	   * - {@link TRACEID_RENDER_PASS}
	   * - {@link TRACEID_RENDER_PASS_DETAIL}
	   * - {@link TRACEID_RENDER_ACTION}
	   * - {@link TRACEID_RENDER_TARGET_ALLOC}
	   * - {@link TRACEID_TEXTURE_ALLOC}
	   * - {@link TRACEID_SHADER_ALLOC}
	   * - {@link TRACEID_SHADER_COMPILE}
	   * - {@link TRACEID_VRAM_TEXTURE}
	   * - {@link TRACEID_VRAM_VB}
	   * - {@link TRACEID_VRAM_IB}
	   * - {@link TRACEID_RENDERPIPELINE_ALLOC}
	   * - {@link TRACEID_COMPUTEPIPELINE_ALLOC}
	   * - {@link TRACEID_PIPELINELAYOUT_ALLOC}
	   * - {@link TRACEID_TEXTURES}
	   * - {@link TRACEID_GPU_TIMINGS}
	   *
	   * @param {boolean} enabled - New enabled state for the channel.
	   */
	  Tracing.set = function set(channel, enabled) {
	    if (enabled === void 0) {
	      enabled = true;
	    }
	    if (enabled) {
	      Tracing._traceChannels.add(channel);
	    } else {
	      Tracing._traceChannels.delete(channel);
	    }
	  }

	  /**
	   * Test if the trace channel is enabled.
	   *
	   * @param {string} channel - Name of the trace channel.
	   * @returns {boolean} - True if the trace channel is enabled.
	   */;
	  Tracing.get = function get(channel) {
	    return Tracing._traceChannels.has(channel);
	  };
	  return Tracing;
	}();
	/**
	 * Set storing the names of enabled trace channels.
	 *
	 * @type {Set<string>}
	 * @private
	 */
	Tracing._traceChannels = new Set();
	/**
	 * Enable call stack logging for trace calls. Defaults to false.
	 *
	 * @type {boolean}
	 */
	Tracing.stack = false;

	/**
	 * Engine debug log system. Note that the logging only executes in the
	 * debug build of the engine, and is stripped out in other builds.
	 *
	 * @ignore
	 */
	var Debug = /*#__PURE__*/function () {
	  function Debug() {}
	  /**
	   * Deprecated warning message.
	   *
	   * @param {string} message - The message to log.
	   */
	  Debug.deprecated = function deprecated(message) {
	    if (!Debug._loggedMessages.has(message)) {
	      Debug._loggedMessages.add(message);
	      console.warn('DEPRECATED: ' + message);
	    }
	  }

	  /**
	   * Assertion deprecated message. If the assertion is false, the deprecated message is written to the log.
	   *
	   * @param {boolean|object} assertion - The assertion to check.
	   * @param {string} message - The message to log.
	   */;
	  Debug.assertDeprecated = function assertDeprecated(assertion, message) {
	    if (!assertion) {
	      Debug.deprecated(message);
	    }
	  }

	  /**
	   * Assertion error message. If the assertion is false, the error message is written to the log.
	   *
	   * @param {boolean|object} assertion - The assertion to check.
	   * @param {...*} args - The values to be written to the log.
	   */;
	  Debug.assert = function assert(assertion) {
	    if (!assertion) {
	      var _console;
	      for (var _len = arguments.length, args = new Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
	        args[_key - 1] = arguments[_key];
	      }
	      (_console = console).error.apply(_console, ['ASSERT FAILED: '].concat(args));
	    }
	  }

	  /**
	   * Assertion error message that writes an error message to the log if the object has already
	   * been destroyed. To be used along setDestroyed.
	   *
	   * @param {object} object - The object to check.
	   */;
	  Debug.assertDestroyed = function assertDestroyed(object) {
	    if (object != null && object.__alreadyDestroyed) {
	      var _object$constructor;
	      var message = "[" + ((_object$constructor = object.constructor) == null ? void 0 : _object$constructor.name) + "] with name [" + object.name + "] has already been destroyed, and cannot be used.";
	      if (!Debug._loggedMessages.has(message)) {
	        Debug._loggedMessages.add(message);
	        console.error('ASSERT FAILED: ', message, object);
	      }
	    }
	  }

	  /**
	   * Executes a function in debug mode only.
	   *
	   * @param {Function} func - Function to call.
	   */;
	  Debug.call = function call(func) {
	    func();
	  }

	  /**
	   * Info message.
	   *
	   * @param {...*} args - The values to be written to the log.
	   */;
	  Debug.log = function log() {
	    var _console2;
	    (_console2 = console).log.apply(_console2, arguments);
	  }

	  /**
	   * Info message logged no more than once.
	   *
	   * @param {string} message - The message to log.
	   * @param {...*} args - The values to be written to the log.
	   */;
	  Debug.logOnce = function logOnce(message) {
	    if (!Debug._loggedMessages.has(message)) {
	      var _console3;
	      Debug._loggedMessages.add(message);
	      for (var _len2 = arguments.length, args = new Array(_len2 > 1 ? _len2 - 1 : 0), _key2 = 1; _key2 < _len2; _key2++) {
	        args[_key2 - 1] = arguments[_key2];
	      }
	      (_console3 = console).log.apply(_console3, [message].concat(args));
	    }
	  }

	  /**
	   * Warning message.
	   *
	   * @param {...*} args - The values to be written to the log.
	   */;
	  Debug.warn = function warn() {
	    var _console4;
	    (_console4 = console).warn.apply(_console4, arguments);
	  }

	  /**
	   * Warning message logged no more than once.
	   *
	   * @param {string} message - The message to log.
	   * @param {...*} args - The values to be written to the log.
	   */;
	  Debug.warnOnce = function warnOnce(message) {
	    if (!Debug._loggedMessages.has(message)) {
	      var _console5;
	      Debug._loggedMessages.add(message);
	      for (var _len3 = arguments.length, args = new Array(_len3 > 1 ? _len3 - 1 : 0), _key3 = 1; _key3 < _len3; _key3++) {
	        args[_key3 - 1] = arguments[_key3];
	      }
	      (_console5 = console).warn.apply(_console5, [message].concat(args));
	    }
	  }

	  /**
	   * Error message.
	   *
	   * @param {...*} args - The values to be written to the log.
	   */;
	  Debug.error = function error() {
	    var _console6;
	    (_console6 = console).error.apply(_console6, arguments);
	  }

	  /**
	   * Error message logged no more than once.
	   *
	   * @param {string} message - The message to log.
	   * @param {...*} args - The values to be written to the log.
	   */;
	  Debug.errorOnce = function errorOnce(message) {
	    if (!Debug._loggedMessages.has(message)) {
	      var _console7;
	      Debug._loggedMessages.add(message);
	      for (var _len4 = arguments.length, args = new Array(_len4 > 1 ? _len4 - 1 : 0), _key4 = 1; _key4 < _len4; _key4++) {
	        args[_key4 - 1] = arguments[_key4];
	      }
	      (_console7 = console).error.apply(_console7, [message].concat(args));
	    }
	  }

	  /**
	   * Trace message, which is logged to the console if the tracing for the channel is enabled
	   *
	   * @param {string} channel - The trace channel
	   * @param {...*} args - The values to be written to the log.
	   */;
	  Debug.trace = function trace(channel) {
	    if (Tracing.get(channel)) {
	      var _console8;
	      for (var _len5 = arguments.length, args = new Array(_len5 > 1 ? _len5 - 1 : 0), _key5 = 1; _key5 < _len5; _key5++) {
	        args[_key5 - 1] = arguments[_key5];
	      }
	      (_console8 = console).groupCollapsed.apply(_console8, [channel.padEnd(20, ' ') + "|"].concat(args));
	      if (Tracing.stack) {
	        console.trace();
	      }
	      console.groupEnd();
	    }
	  };
	  return Debug;
	}();
	/**
	 * A helper debug functionality.
	 *
	 * @ignore
	 */
	/**
	 * Set storing already logged messages, to only print each unique message one time.
	 *
	 * @type {Set<string>}
	 * @private
	 */
	Debug._loggedMessages = new Set();
	var DebugHelper = /*#__PURE__*/function () {
	  function DebugHelper() {}
	  /**
	   * Set a name to the name property of the object. Executes only in the debug build.
	   *
	   * @param {object} object - The object to assign the name to.
	   * @param {string} name - The name to assign.
	   */
	  DebugHelper.setName = function setName(object, name) {
	    if (object) {
	      object.name = name;
	    }
	  }

	  /**
	   * Set a label to the label property of the object. Executes only in the debug build.
	   *
	   * @param {object} object - The object to assign the name to.
	   * @param {string} label - The label to assign.
	   */;
	  DebugHelper.setLabel = function setLabel(object, label) {
	    if (object) {
	      object.label = label;
	    }
	  }

	  /**
	   * Marks object as destroyed. Executes only in the debug build. To be used along assertDestroyed.
	   *
	   * @param {object} object - The object to mark as destroyed.
	   */;
	  DebugHelper.setDestroyed = function setDestroyed(object) {
	    if (object) {
	      object.__alreadyDestroyed = true;
	    }
	  };
	  return DebugHelper;
	}();

	/**
	 * Event Handle that is created by {@link EventHandler} and can be used for easier event removal and management.
	 * @example
	 * const evt = obj.on('test', (a, b) => {
	 *     console.log(a + b);
	 * });
	 * obj.fire('test');
	 *
	 * evt.off(); // easy way to remove this event
	 * obj.fire('test'); // this will not trigger an event
	 * @example
	 * // store an array of event handles
	 * let events = [ ];
	 *
	 * events.push(objA.on('testA', () => { }));
	 * events.push(objB.on('testB', () => { }));
	 *
	 * // when needed, remove all events
	 * events.forEach((evt) => {
	 *     evt.off();
	 * });
	 * events = [ ];
	 */
	var EventHandle = /*#__PURE__*/function () {
	  /**
	   * @param {new Function("modulePath", "return import(modulePath)")('./event-handler.js').EventHandler} handler - source object of the event.
	   * @param {string} name - Name of the event.
	   * @param {new Function("modulePath", "return import(modulePath)")('./event-handler.js').HandleEventCallback} callback - Function that is called when event is fired.
	   * @param {object} scope - Object that is used as `this` when event is fired.
	   * @param {boolean} [once] - If this is a single event and will be removed after event is fired.
	   */
	  function EventHandle(handler, name, callback, scope, once) {
	    if (once === void 0) {
	      once = false;
	    }
	    /**
	     * @type {new Function("modulePath", "return import(modulePath)")('./event-handler.js').EventHandler}
	     * @private
	     */
	    this.handler = void 0;
	    /**
	     * @type {string}
	     * @private
	     */
	    this.name = void 0;
	    /**
	     * @type {new Function("modulePath", "return import(modulePath)")('./event-handler.js').HandleEventCallback}
	     * @ignore
	     */
	    this.callback = void 0;
	    /**
	     * @type {object}
	     * @ignore
	     */
	    this.scope = void 0;
	    /**
	     * @type {boolean}
	     * @ignore
	     */
	    this._once = void 0;
	    /**
	     * True if event has been removed.
	     * @type {boolean}
	     * @private
	     */
	    this._removed = false;
	    this.handler = handler;
	    this.name = name;
	    this.callback = callback;
	    this.scope = scope;
	    this._once = once;
	  }

	  /**
	   * Remove this event from its handler.
	   */
	  var _proto = EventHandle.prototype;
	  _proto.off = function off() {
	    if (this._removed) return;
	    this.handler.off(this.name, this.callback, this.scope);
	  };
	  _proto.on = function on(name, callback, scope) {
	    if (scope === void 0) {
	      scope = this;
	    }
	    Debug.deprecated('Using chaining with EventHandler.on is deprecated, subscribe to an event from EventHandler directly instead.');
	    return this.handler._addCallback(name, callback, scope, false);
	  };
	  _proto.once = function once(name, callback, scope) {
	    if (scope === void 0) {
	      scope = this;
	    }
	    Debug.deprecated('Using chaining with EventHandler.once is deprecated, subscribe to an event from EventHandler directly instead.');
	    return this.handler._addCallback(name, callback, scope, true);
	  }

	  /**
	   * Mark if event has been removed.
	   * @type {boolean}
	   * @ignore
	   */;
	  _createClass(EventHandle, [{
	    key: "removed",
	    get:
	    /**
	     * True if event has been removed.
	     * @type {boolean}
	     */
	    function get() {
	      return this._removed;
	    },
	    set: function set(value) {
	      if (!value) return;
	      this._removed = true;
	    }
	  }]);
	  return EventHandle;
	}();

	/**
	 * Callback used by {@link EventHandler} functions. Note the callback is limited to 8 arguments.
	 *
	 * @callback HandleEventCallback
	 * @param {*} [arg1] - First argument that is passed from caller.
	 * @param {*} [arg2] - Second argument that is passed from caller.
	 * @param {*} [arg3] - Third argument that is passed from caller.
	 * @param {*} [arg4] - Fourth argument that is passed from caller.
	 * @param {*} [arg5] - Fifth argument that is passed from caller.
	 * @param {*} [arg6] - Sixth argument that is passed from caller.
	 * @param {*} [arg7] - Seventh argument that is passed from caller.
	 * @param {*} [arg8] - Eighth argument that is passed from caller.
	 */

	/**
	 * Abstract base class that implements functionality for event handling.
	 *
	 * ```javascript
	 * const obj = new EventHandlerSubclass();
	 *
	 * // subscribe to an event
	 * obj.on('hello', function (str) {
	 *     console.log('event hello is fired', str);
	 * });
	 *
	 * // fire event
	 * obj.fire('hello', 'world');
	 * ```
	 */
	var EventHandler = /*#__PURE__*/function () {
	  function EventHandler() {
	    /**
	     * @type {Map<string,Array<EventHandle>>}
	     * @private
	     */
	    this._callbacks = new Map();
	    /**
	     * @type {Map<string,Array<EventHandle>>}
	     * @private
	     */
	    this._callbackActive = new Map();
	  }
	  var _proto = EventHandler.prototype;
	  /**
	   * Reinitialize the event handler.
	   * @ignore
	   */
	  _proto.initEventHandler = function initEventHandler() {
	    this._callbacks = new Map();
	    this._callbackActive = new Map();
	  }

	  /**
	   * Registers a new event handler.
	   *
	   * @param {string} name - Name of the event to bind the callback to.
	   * @param {HandleEventCallback} callback - Function that is called when event is fired. Note
	   * the callback is limited to 8 arguments.
	   * @param {object} scope - Object to use as 'this' when the event is fired, defaults to
	   * current this.
	   * @param {boolean} once - If true, the callback will be unbound after being fired once.
	   * @returns {EventHandle} Created {@link EventHandle}.
	   * @ignore
	   */;
	  _proto._addCallback = function _addCallback(name, callback, scope, once) {
	    if (!name || typeof name !== 'string' || !callback) console.warn("EventHandler: subscribing to an event (" + name + ") with missing arguments", callback);
	    if (!this._callbacks.has(name)) this._callbacks.set(name, []);

	    // if we are adding a callback to the list that is executing right now
	    // ensure we preserve initial list before modifications
	    if (this._callbackActive.has(name)) {
	      var callbackActive = this._callbackActive.get(name);
	      if (callbackActive && callbackActive === this._callbacks.get(name)) {
	        this._callbackActive.set(name, callbackActive.slice());
	      }
	    }
	    var evt = new EventHandle(this, name, callback, scope, once);
	    this._callbacks.get(name).push(evt);
	    return evt;
	  }

	  /**
	   * Attach an event handler to an event.
	   *
	   * @param {string} name - Name of the event to bind the callback to.
	   * @param {HandleEventCallback} callback - Function that is called when event is fired. Note
	   * the callback is limited to 8 arguments.
	   * @param {object} [scope] - Object to use as 'this' when the event is fired, defaults to
	   * current this.
	   * @returns {EventHandle} Can be used for removing event in the future.
	   * @example
	   * obj.on('test', function (a, b) {
	   *     console.log(a + b);
	   * });
	   * obj.fire('test', 1, 2); // prints 3 to the console
	   * @example
	   * const evt = obj.on('test', function (a, b) {
	   *     console.log(a + b);
	   * });
	   * // some time later
	   * evt.off();
	   */;
	  _proto.on = function on(name, callback, scope) {
	    if (scope === void 0) {
	      scope = this;
	    }
	    return this._addCallback(name, callback, scope, false);
	  }

	  /**
	   * Attach an event handler to an event. This handler will be removed after being fired once.
	   *
	   * @param {string} name - Name of the event to bind the callback to.
	   * @param {HandleEventCallback} callback - Function that is called when event is fired. Note
	   * the callback is limited to 8 arguments.
	   * @param {object} [scope] - Object to use as 'this' when the event is fired, defaults to
	   * current this.
	   * @returns {EventHandle} - can be used for removing event in the future.
	   * @example
	   * obj.once('test', function (a, b) {
	   *     console.log(a + b);
	   * });
	   * obj.fire('test', 1, 2); // prints 3 to the console
	   * obj.fire('test', 1, 2); // not going to get handled
	   */;
	  _proto.once = function once(name, callback, scope) {
	    if (scope === void 0) {
	      scope = this;
	    }
	    return this._addCallback(name, callback, scope, true);
	  }

	  /**
	   * Detach an event handler from an event. If callback is not provided then all callbacks are
	   * unbound from the event, if scope is not provided then all events with the callback will be
	   * unbound.
	   *
	   * @param {string} [name] - Name of the event to unbind.
	   * @param {HandleEventCallback} [callback] - Function to be unbound.
	   * @param {object} [scope] - Scope that was used as the this when the event is fired.
	   * @returns {EventHandler} Self for chaining.
	   * @example
	   * const handler = function () {
	   * };
	   * obj.on('test', handler);
	   *
	   * obj.off(); // Removes all events
	   * obj.off('test'); // Removes all events called 'test'
	   * obj.off('test', handler); // Removes all handler functions, called 'test'
	   * obj.off('test', handler, this); // Removes all handler functions, called 'test' with scope this
	   */;
	  _proto.off = function off(name, callback, scope) {
	    if (name) {
	      // if we are removing a callback from the list that is executing right now
	      // ensure we preserve initial list before modifications
	      if (this._callbackActive.has(name) && this._callbackActive.get(name) === this._callbacks.get(name)) this._callbackActive.set(name, this._callbackActive.get(name).slice());
	    } else {
	      // if we are removing a callback from any list that is executing right now
	      // ensure we preserve these initial lists before modifications
	      for (var _iterator = _createForOfIteratorHelperLoose(this._callbackActive), _step; !(_step = _iterator()).done;) {
	        var _step$value = _step.value,
	          key = _step$value[0],
	          callbacks = _step$value[1];
	        if (!this._callbacks.has(key)) continue;
	        if (this._callbacks.get(key) !== callbacks) continue;
	        this._callbackActive.set(key, callbacks.slice());
	      }
	    }
	    if (!name) {
	      // remove all events
	      for (var _iterator2 = _createForOfIteratorHelperLoose(this._callbacks.values()), _step2; !(_step2 = _iterator2()).done;) {
	        var _callbacks = _step2.value;
	        for (var i = 0; i < _callbacks.length; i++) {
	          _callbacks[i].removed = true;
	        }
	      }
	      this._callbacks.clear();
	    } else if (!callback) {
	      // remove all events of a specific name
	      var _callbacks2 = this._callbacks.get(name);
	      if (_callbacks2) {
	        for (var _i = 0; _i < _callbacks2.length; _i++) {
	          _callbacks2[_i].removed = true;
	        }
	        this._callbacks.delete(name);
	      }
	    } else {
	      var _callbacks3 = this._callbacks.get(name);
	      if (!_callbacks3) return this;
	      for (var _i2 = 0; _i2 < _callbacks3.length; _i2++) {
	        // remove all events with a specific name and a callback
	        if (_callbacks3[_i2].callback !== callback) continue;

	        // could be a specific scope as well
	        if (scope && _callbacks3[_i2].scope !== scope) continue;
	        _callbacks3[_i2].removed = true;
	        _callbacks3.splice(_i2, 1);
	        _i2--;
	      }
	      if (_callbacks3.length === 0) this._callbacks.delete(name);
	    }
	    return this;
	  }

	  /**
	   * Fire an event, all additional arguments are passed on to the event listener.
	   *
	   * @param {string} name - Name of event to fire.
	   * @param {*} [arg1] - First argument that is passed to the event handler.
	   * @param {*} [arg2] - Second argument that is passed to the event handler.
	   * @param {*} [arg3] - Third argument that is passed to the event handler.
	   * @param {*} [arg4] - Fourth argument that is passed to the event handler.
	   * @param {*} [arg5] - Fifth argument that is passed to the event handler.
	   * @param {*} [arg6] - Sixth argument that is passed to the event handler.
	   * @param {*} [arg7] - Seventh argument that is passed to the event handler.
	   * @param {*} [arg8] - Eighth argument that is passed to the event handler.
	   * @returns {EventHandler} Self for chaining.
	   * @example
	   * obj.fire('test', 'This is the message');
	   */;
	  _proto.fire = function fire(name, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8) {
	    if (!name) return this;
	    var callbacksInitial = this._callbacks.get(name);
	    if (!callbacksInitial) return this;
	    var callbacks;
	    if (!this._callbackActive.has(name)) {
	      // when starting callbacks execution ensure we store a list of initial callbacks
	      this._callbackActive.set(name, callbacksInitial);
	    } else if (this._callbackActive.get(name) !== callbacksInitial) {
	      // if we are trying to execute a callback while there is an active execution right now
	      // and the active list has been already modified,
	      // then we go to an unoptimized path and clone callbacks list to ensure execution consistency
	      callbacks = callbacksInitial.slice();
	    }

	    // eslint-disable-next-line no-unmodified-loop-condition
	    for (var i = 0; (callbacks || this._callbackActive.get(name)) && i < (callbacks || this._callbackActive.get(name)).length; i++) {
	      var evt = (callbacks || this._callbackActive.get(name))[i];
	      if (!evt.callback) continue;
	      evt.callback.call(evt.scope, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8);
	      if (evt._once) {
	        // check that callback still exists because user may have unsubscribed in the event handler
	        var existingCallback = this._callbacks.get(name);
	        var ind = existingCallback ? existingCallback.indexOf(evt) : -1;
	        if (ind !== -1) {
	          if (this._callbackActive.get(name) === existingCallback) this._callbackActive.set(name, this._callbackActive.get(name).slice());
	          var _callbacks4 = this._callbacks.get(name);
	          if (!_callbacks4) continue;
	          _callbacks4[ind].removed = true;
	          _callbacks4.splice(ind, 1);
	          if (_callbacks4.length === 0) this._callbacks.delete(name);
	        }
	      }
	    }
	    if (!callbacks) this._callbackActive.delete(name);
	    return this;
	  }

	  /**
	   * Test if there are any handlers bound to an event name.
	   *
	   * @param {string} name - The name of the event to test.
	   * @returns {boolean} True if the object has handlers bound to the specified event name.
	   * @example
	   * obj.on('test', function () { }); // bind an event to 'test'
	   * obj.hasEvent('test'); // returns true
	   * obj.hasEvent('hello'); // returns false
	   */;
	  _proto.hasEvent = function hasEvent(name) {
	    var _this$_callbacks$get;
	    return !!((_this$_callbacks$get = this._callbacks.get(name)) != null && _this$_callbacks$get.length);
	  };
	  return EventHandler;
	}();

	var events = {
	  /**
	   * Attach event methods 'on', 'off', 'fire', 'once' and 'hasEvent' to the target object.
	   *
	   * @param {object} target - The object to add events to.
	   * @returns {object} The target object.
	   * @example
	   * const obj = { };
	   * pc.events.attach(obj);
	   * @ignore
	   */
	  attach: function attach(target) {
	    var ev = events;
	    target._addCallback = ev._addCallback;
	    target.on = ev.on;
	    target.off = ev.off;
	    target.fire = ev.fire;
	    target.once = ev.once;
	    target.hasEvent = ev.hasEvent;
	    EventHandler.prototype.initEventHandler.call(target);
	    return target;
	  },
	  _addCallback: EventHandler.prototype._addCallback,
	  on: EventHandler.prototype.on,
	  off: EventHandler.prototype.off,
	  fire: EventHandler.prototype.fire,
	  once: EventHandler.prototype.once,
	  hasEvent: EventHandler.prototype.hasEvent
	};

	/**
	 * Basically a very large random number (128-bit) which means the probability of creating two that
	 * clash is vanishingly small. GUIDs are used as the unique identifiers for Entities.
	 *
	 * @namespace
	 */
	var guid = {
	  /**
	   * Create an RFC4122 version 4 compliant GUID.
	   *
	   * @returns {string} A new GUID.
	   */
	  create: function create() {
	    return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function (c) {
	      var r = Math.random() * 16 | 0;
	      var v = c === 'x' ? r : r & 0x3 | 0x8;
	      return v.toString(16);
	    });
	  }
	};

	/**
	 * File path API.
	 *
	 * @namespace
	 */
	var path = {
	  /**
	   * The character that separates path segments.
	   *
	   * @type {string}
	   */
	  delimiter: '/',
	  /**
	   * Join two or more sections of file path together, inserting a delimiter if needed.
	   *
	   * @param {...string} sections - Sections of the path to join.
	   * @returns {string} The joined file path.
	   * @example
	   * const path = pc.path.join('foo', 'bar');
	   * console.log(path); // Prints 'foo/bar'
	   * @example
	   * const path = pc.path.join('alpha', 'beta', 'gamma');
	   * console.log(path); // Prints 'alpha/beta/gamma'
	   */
	  join: function join() {
	    var result = arguments.length <= 0 ? undefined : arguments[0];
	    for (var i = 0; i < arguments.length - 1; i++) {
	      var one = i < 0 || arguments.length <= i ? undefined : arguments[i];
	      var two = i + 1 < 0 || arguments.length <= i + 1 ? undefined : arguments[i + 1];
	      if (two[0] === path.delimiter) {
	        result = two;
	        continue;
	      }
	      if (one && two && one[one.length - 1] !== path.delimiter && two[0] !== path.delimiter) {
	        result += path.delimiter + two;
	      } else {
	        result += two;
	      }
	    }
	    return result;
	  },
	  /**
	   * Normalize the path by removing '.' and '..' instances.
	   *
	   * @param {string} pathname - The path to normalize.
	   * @returns {string} The normalized path.
	   */
	  normalize: function normalize(pathname) {
	    var lead = pathname.startsWith(path.delimiter);
	    var trail = pathname.endsWith(path.delimiter);
	    var parts = pathname.split('/');
	    var result = '';
	    var cleaned = [];
	    for (var i = 0; i < parts.length; i++) {
	      if (parts[i] === '') continue;
	      if (parts[i] === '.') continue;
	      if (parts[i] === '..' && cleaned.length > 0) {
	        cleaned = cleaned.slice(0, cleaned.length - 2);
	        continue;
	      }
	      if (i > 0) cleaned.push(path.delimiter);
	      cleaned.push(parts[i]);
	    }
	    result = cleaned.join('');
	    if (!lead && result[0] === path.delimiter) {
	      result = result.slice(1);
	    }
	    if (trail && result[result.length - 1] !== path.delimiter) {
	      result += path.delimiter;
	    }
	    return result;
	  },
	  /**
	   * Split the pathname path into a pair [head, tail] where tail is the final part of the path
	   * after the last delimiter and head is everything leading up to that. tail will never contain
	   * a slash.
	   *
	   * @param {string} pathname - The path to split.
	   * @returns {string[]} The split path which is an array of two strings, the path and the
	   * filename.
	   */
	  split: function split(pathname) {
	    var lastDelimiterIndex = pathname.lastIndexOf(path.delimiter);
	    if (lastDelimiterIndex !== -1) {
	      return [pathname.substring(0, lastDelimiterIndex), pathname.substring(lastDelimiterIndex + 1)];
	    }
	    return ["", pathname];
	  },
	  /**
	   * Return the basename of the path. That is the second element of the pair returned by passing
	   * path into {@link path.split}.
	   *
	   * @param {string} pathname - The path to process.
	   * @returns {string} The basename.
	   * @example
	   * pc.path.getBasename("/path/to/file.txt"); // returns "file.txt"
	   * pc.path.getBasename("/path/to/dir"); // returns "dir"
	   */
	  getBasename: function getBasename(pathname) {
	    return path.split(pathname)[1];
	  },
	  /**
	   * Get the directory name from the path. This is everything up to the final instance of
	   * {@link path.delimiter}.
	   *
	   * @param {string} pathname - The path to get the directory from.
	   * @returns {string} The directory part of the path.
	   */
	  getDirectory: function getDirectory(pathname) {
	    return path.split(pathname)[0];
	  },
	  /**
	   * Return the extension of the path. Pop the last value of a list after path is split by
	   * question mark and comma.
	   *
	   * @param {string} pathname - The path to process.
	   * @returns {string} The extension.
	   * @example
	   * pc.path.getExtension("/path/to/file.txt"); // returns ".txt"
	   * pc.path.getExtension("/path/to/file.jpg"); // returns ".jpg"
	   * pc.path.getExtension("/path/to/file.txt?function=getExtension"); // returns ".txt"
	   */
	  getExtension: function getExtension(pathname) {
	    var ext = pathname.split('?')[0].split('.').pop();
	    if (ext !== pathname) {
	      return '.' + ext;
	    }
	    return '';
	  },
	  /**
	   * Check if a string s is relative path.
	   *
	   * @param {string} pathname - The path to process.
	   * @returns {boolean} True if s doesn't start with slash and doesn't include colon and double
	   * slash.
	   *
	   * @example
	   * pc.path.isRelativePath("file.txt"); // returns true
	   * pc.path.isRelativePath("path/to/file.txt"); // returns true
	   * pc.path.isRelativePath("./path/to/file.txt"); // returns true
	   * pc.path.isRelativePath("../path/to/file.jpg"); // returns true
	   * pc.path.isRelativePath("/path/to/file.jpg"); // returns false
	   * pc.path.isRelativePath("http://path/to/file.jpg"); // returns false
	   */
	  isRelativePath: function isRelativePath(pathname) {
	    return pathname.charAt(0) !== '/' && pathname.match(/:\/\//) === null;
	  },
	  /**
	   * Return the path without file name. If path is relative path, start with period.
	   *
	   * @param {string} pathname - The full path to process.
	   * @returns {string} The path without a last element from list split by slash.
	   * @example
	   * pc.path.extractPath("path/to/file.txt");    // returns "./path/to"
	   * pc.path.extractPath("./path/to/file.txt");  // returns "./path/to"
	   * pc.path.extractPath("../path/to/file.txt"); // returns "../path/to"
	   * pc.path.extractPath("/path/to/file.txt");   // returns "/path/to"
	   */
	  extractPath: function extractPath(pathname) {
	    var result = '';
	    var parts = pathname.split('/');
	    var i = 0;
	    if (parts.length > 1) {
	      if (path.isRelativePath(pathname)) {
	        if (parts[0] === '.') {
	          for (i = 0; i < parts.length - 1; ++i) {
	            result += i === 0 ? parts[i] : '/' + parts[i];
	          }
	        } else if (parts[0] === '..') {
	          for (i = 0; i < parts.length - 1; ++i) {
	            result += i === 0 ? parts[i] : '/' + parts[i];
	          }
	        } else {
	          result = '.';
	          for (i = 0; i < parts.length - 1; ++i) {
	            result += '/' + parts[i];
	          }
	        }
	      } else {
	        for (i = 0; i < parts.length - 1; ++i) {
	          result += i === 0 ? parts[i] : '/' + parts[i];
	        }
	      }
	    }
	    return result;
	  }
	};

	var _ref, _ref2, _ref3;
	// detect whether passive events are supported by the browser
	var detectPassiveEvents = function detectPassiveEvents() {
	  var result = false;
	  try {
	    var opts = Object.defineProperty({}, 'passive', {
	      get: function get() {
	        result = true;
	        return false;
	      }
	    });
	    window.addEventListener('testpassive', null, opts);
	    window.removeEventListener('testpassive', null, opts);
	  } catch (e) {}
	  return result;
	};
	var ua = typeof navigator !== 'undefined' ? navigator.userAgent : '';
	var environment = typeof window !== 'undefined' ? 'browser' : typeof global !== 'undefined' ? 'node' : 'worker';

	// detect platform
	var platformName = /android/i.test(ua) ? 'android' : /ip([ao]d|hone)/i.test(ua) ? 'ios' : /windows/i.test(ua) ? 'windows' : /mac os/i.test(ua) ? 'osx' : /linux/i.test(ua) ? 'linux' : /cros/i.test(ua) ? 'cros' : null;

	// detect browser
	var browserName = environment !== 'browser' ? null : /(Chrome\/|Chromium\/|Edg.*\/)/.test(ua) ? 'chrome' :
	// chrome, chromium, edge
	/Safari\//.test(ua) ? 'safari' :
	// safari, ios chrome/firefox
	/Firefox\//.test(ua) ? 'firefox' : 'other';
	var xbox = /xbox/i.test(ua);
	var touch = environment === 'browser' && ('ontouchstart' in window || 'maxTouchPoints' in navigator && navigator.maxTouchPoints > 0);
	var gamepads = environment === 'browser' && (!!navigator.getGamepads || !!navigator.webkitGetGamepads);
	var workers = typeof Worker !== 'undefined';
	var passiveEvents = detectPassiveEvents();

	/**
	 * Global namespace that stores flags regarding platform environment and features support.
	 *
	 * @namespace
	 * @example
	 * if (pc.platform.touch) {
	 *     // touch is supported
	 * }
	 */
	var platform = {
	  /**
	   * String identifying the current platform. Can be one of: android, ios, windows, osx, linux,
	   * cros or null.
	   *
	   * @type {'android' | 'ios' | 'windows' | 'osx' | 'linux' | 'cros' | null}
	   * @ignore
	   */
	  name: platformName,
	  /**
	   * String identifying the current runtime environment. Either 'browser', 'node' or 'worker'.
	   *
	   * @type {'browser' | 'node' | 'worker'}
	   */
	  environment: environment,
	  /**
	   * The global object. This will be the window object when running in a browser and the global
	   * object when running in nodejs and self when running in a worker.
	   *
	   * @type {object}
	   */
	  global: (_ref = (_ref2 = (_ref3 = typeof globalThis !== 'undefined' && globalThis) != null ? _ref3 : environment === 'browser' && window) != null ? _ref2 : environment === 'node' && global) != null ? _ref : environment === 'worker' && self,
	  /**
	   * Convenience boolean indicating whether we're running in the browser.
	   *
	   * @type {boolean}
	   */
	  browser: environment === 'browser',
	  /**
	   * True if running on a desktop or laptop device.
	   *
	   * @type {boolean}
	   */
	  desktop: ['windows', 'osx', 'linux', 'cros'].includes(platformName),
	  /**
	   * True if running on a mobile or tablet device.
	   *
	   * @type {boolean}
	   */
	  mobile: ['android', 'ios'].includes(platformName),
	  /**
	   * True if running on an iOS device.
	   *
	   * @type {boolean}
	   */
	  ios: platformName === 'ios',
	  /**
	   * True if running on an Android device.
	   *
	   * @type {boolean}
	   */
	  android: platformName === 'android',
	  /**
	   * True if running on an Xbox device.
	   *
	   * @type {boolean}
	   */
	  xbox: xbox,
	  /**
	   * True if the platform supports gamepads.
	   *
	   * @type {boolean}
	   */
	  gamepads: gamepads,
	  /**
	   * True if the supports touch input.
	   *
	   * @type {boolean}
	   */
	  touch: touch,
	  /**
	   * True if the platform supports Web Workers.
	   *
	   * @type {boolean}
	   */
	  workers: workers,
	  /**
	   * True if the platform supports an options object as the third parameter to
	   * `EventTarget.addEventListener()` and the passive property is supported.
	   *
	   * @type {boolean}
	   * @ignore
	   */
	  passiveEvents: passiveEvents,
	  /**
	   * Get the browser name.
	   *
	   * @type {'chrome' | 'safari' | 'firefox' | 'other' | null}
	   * @ignore
	   */
	  browserName: browserName
	};

	var ASCII_LOWERCASE = 'abcdefghijklmnopqrstuvwxyz';
	var ASCII_UPPERCASE = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
	var ASCII_LETTERS = ASCII_LOWERCASE + ASCII_UPPERCASE;
	var HIGH_SURROGATE_BEGIN = 0xD800;
	var HIGH_SURROGATE_END = 0xDBFF;
	var LOW_SURROGATE_BEGIN = 0xDC00;
	var LOW_SURROGATE_END = 0xDFFF;
	var ZERO_WIDTH_JOINER = 0x200D;

	// Flag emoji
	var REGIONAL_INDICATOR_BEGIN = 0x1F1E6;
	var REGIONAL_INDICATOR_END = 0x1F1FF;

	// Skin color modifications to emoji
	var FITZPATRICK_MODIFIER_BEGIN = 0x1F3FB;
	var FITZPATRICK_MODIFIER_END = 0x1F3FF;

	// Accent characters
	var DIACRITICAL_MARKS_BEGIN = 0x20D0;
	var DIACRITICAL_MARKS_END = 0x20FF;

	// Special emoji joins
	var VARIATION_MODIFIER_BEGIN = 0xFE00;
	var VARIATION_MODIFIER_END = 0xFE0F;
	function getCodePointData(string, i) {
	  if (i === void 0) {
	    i = 0;
	  }
	  var size = string.length;

	  // Account for out-of-bounds indices:
	  if (i < 0 || i >= size) {
	    return null;
	  }
	  var first = string.charCodeAt(i);
	  if (size > 1 && first >= HIGH_SURROGATE_BEGIN && first <= HIGH_SURROGATE_END) {
	    var second = string.charCodeAt(i + 1);
	    if (second >= LOW_SURROGATE_BEGIN && second <= LOW_SURROGATE_END) {
	      // https://mathiasbynens.be/notes/javascript-encoding#surrogate-formulae
	      return {
	        code: (first - HIGH_SURROGATE_BEGIN) * 0x400 + second - LOW_SURROGATE_BEGIN + 0x10000,
	        long: true
	      };
	    }
	  }
	  return {
	    code: first,
	    long: false
	  };
	}
	function isCodeBetween(string, begin, end) {
	  if (!string) return false;
	  var codeData = getCodePointData(string);
	  if (codeData) {
	    var code = codeData.code;
	    return code >= begin && code <= end;
	  }
	  return false;
	}
	function numCharsToTakeForNextSymbol(string, index) {
	  if (index === string.length - 1) {
	    // Last character in the string, so we can only take 1
	    return 1;
	  }
	  if (isCodeBetween(string[index], HIGH_SURROGATE_BEGIN, HIGH_SURROGATE_END)) {
	    var first = string.substring(index, index + 2);
	    var second = string.substring(index + 2, index + 4);

	    // check if second character is fitzpatrick (color) modifier
	    // or if this is a pair of regional indicators (a flag)
	    if (isCodeBetween(second, FITZPATRICK_MODIFIER_BEGIN, FITZPATRICK_MODIFIER_END) || isCodeBetween(first, REGIONAL_INDICATOR_BEGIN, REGIONAL_INDICATOR_END) && isCodeBetween(second, REGIONAL_INDICATOR_BEGIN, REGIONAL_INDICATOR_END)) {
	      return 4;
	    }

	    // check if next character is a modifier, in which case we should return it
	    if (isCodeBetween(second, VARIATION_MODIFIER_BEGIN, VARIATION_MODIFIER_END)) {
	      return 3;
	    }

	    // return surrogate pair
	    return 2;
	  }

	  // check if next character is the emoji modifier, in which case we should include it
	  if (isCodeBetween(string[index + 1], VARIATION_MODIFIER_BEGIN, VARIATION_MODIFIER_END)) {
	    return 2;
	  }

	  // just a regular character
	  return 1;
	}

	/**
	 * Extended String API.
	 *
	 * @namespace
	 */
	var string = {
	  /**
	   * All lowercase letters.
	   *
	   * @type {string}
	   */
	  ASCII_LOWERCASE: ASCII_LOWERCASE,
	  /**
	   * All uppercase letters.
	   *
	   * @type {string}
	   */
	  ASCII_UPPERCASE: ASCII_UPPERCASE,
	  /**
	   * All ASCII letters.
	   *
	   * @type {string}
	   */
	  ASCII_LETTERS: ASCII_LETTERS,
	  /**
	   * Return a string with \{n\} replaced with the n-th argument.
	   *
	   * @param {string} s - The string to format.
	   * @param {...*} args - All other arguments are substituted into the string.
	   * @returns {string} The formatted string.
	   * @example
	   * const s = pc.string.format("Hello {0}", "world");
	   * console.log(s); // Prints "Hello world"
	   */
	  format: function format(s) {
	    for (var i = 0; i < (arguments.length <= 1 ? 0 : arguments.length - 1); i++) {
	      s = s.replace("{" + i + "}", i + 1 < 1 || arguments.length <= i + 1 ? undefined : arguments[i + 1]);
	    }
	    return s;
	  },
	  /**
	   * Get the code point number for a character in a string. Polyfill for
	   * [`codePointAt`]{@link https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/String/codePointAt}.
	   *
	   * @param {string} string - The string to get the code point from.
	   * @param {number} [i] - The index in the string.
	   * @returns {number} The code point value for the character in the string.
	   */
	  getCodePoint: function getCodePoint(string, i) {
	    var codePointData = getCodePointData(string, i);
	    return codePointData && codePointData.code;
	  },
	  /**
	   * Gets an array of all code points in a string.
	   *
	   * @param {string} string - The string to get code points from.
	   * @returns {number[]} The code points in the string.
	   */
	  getCodePoints: function getCodePoints(string) {
	    if (typeof string !== 'string') {
	      throw new TypeError('Not a string');
	    }
	    var i = 0;
	    var arr = [];
	    var codePoint;
	    while (!!(codePoint = getCodePointData(string, i))) {
	      arr.push(codePoint.code);
	      i += codePoint.long ? 2 : 1;
	    }
	    return arr;
	  },
	  /**
	   * Gets an array of all grapheme clusters (visible symbols) in a string. This is needed because
	   * some symbols (such as emoji or accented characters) are actually made up of multiple
	   * character codes. See {@link https://mathiasbynens.be/notes/javascript-unicode here} for more
	   * info.
	   *
	   * @param {string} string - The string to break into symbols.
	   * @returns {string[]} The symbols in the string.
	   */
	  getSymbols: function getSymbols(string) {
	    if (typeof string !== 'string') {
	      throw new TypeError('Not a string');
	    }
	    var index = 0;
	    var length = string.length;
	    var output = [];
	    var take = 0;
	    var ch;
	    while (index < length) {
	      take += numCharsToTakeForNextSymbol(string, index + take);
	      ch = string[index + take];
	      // Handle special cases
	      if (isCodeBetween(ch, DIACRITICAL_MARKS_BEGIN, DIACRITICAL_MARKS_END)) {
	        ch = string[index + take++];
	      }
	      if (isCodeBetween(ch, VARIATION_MODIFIER_BEGIN, VARIATION_MODIFIER_END)) {
	        ch = string[index + take++];
	      }
	      if (ch && ch.charCodeAt(0) === ZERO_WIDTH_JOINER) {
	        ch = string[index + take++];
	        // Not a complete char yet
	        continue;
	      }
	      var _char = string.substring(index, index + take);
	      output.push(_char);
	      index += take;
	      take = 0;
	    }
	    return output;
	  },
	  /**
	   * Get the string for a given code point or set of code points. Polyfill for
	   * [`fromCodePoint`]{@link https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/String/fromCodePoint}.
	   *
	   * @param {...number} args - The code points to convert to a string.
	   * @returns {string} The converted string.
	   */
	  fromCodePoint: function fromCodePoint() {
	    var chars = [];
	    var current;
	    var codePoint;
	    var units;
	    for (var i = 0; i < arguments.length; ++i) {
	      current = Number(arguments[i]);
	      codePoint = current - 0x10000;
	      units = current > 0xFFFF ? [(codePoint >> 10) + 0xD800, codePoint % 0x400 + 0xDC00] : [current];
	      chars.push(String.fromCharCode.apply(null, units));
	    }
	    return chars.join('');
	  } /* ...args */
	};

	/**
	 * A ordered list-type data structure that can provide item look up by key and can also return a list.
	 *
	 * @ignore
	 */
	var IndexedList = /*#__PURE__*/function () {
	  function IndexedList() {
	    /**
	     * @type {object[]}
	     * @private
	     */
	    this._list = [];
	    /**
	     * @type {Object<string, number>}
	     * @private
	     */
	    this._index = {};
	  }
	  var _proto = IndexedList.prototype;
	  /**
	   * Add a new item into the list with a index key.
	   *
	   * @param {string} key - Key used to look up item in index.
	   * @param {object} item - Item to be stored.
	   */
	  _proto.push = function push(key, item) {
	    if (this._index[key]) {
	      throw Error('Key already in index ' + key);
	    }
	    var location = this._list.push(item) - 1;
	    this._index[key] = location;
	  }

	  /**
	   * Test whether a key has been added to the index.
	   *
	   * @param {string} key - The key to test.
	   * @returns {boolean} Returns true if key is in the index, false if not.
	   */;
	  _proto.has = function has(key) {
	    return this._index[key] !== undefined;
	  }

	  /**
	   * Return the item indexed by a key.
	   *
	   * @param {string} key - The key of the item to retrieve.
	   * @returns {object|null} The item stored at key. Returns null if key is not in the index.
	   */;
	  _proto.get = function get(key) {
	    var location = this._index[key];
	    if (location !== undefined) {
	      return this._list[location];
	    }
	    return null;
	  }

	  /**
	   * Remove the item indexed by key from the list.
	   *
	   * @param {string} key - The key at which to remove the item.
	   * @returns {boolean} Returns true if the key exists and an item was removed, returns false if
	   * no item was removed.
	   */;
	  _proto.remove = function remove(key) {
	    var location = this._index[key];
	    if (location !== undefined) {
	      this._list.splice(location, 1);
	      delete this._index[key];

	      // update index
	      for (key in this._index) {
	        var idx = this._index[key];
	        if (idx > location) {
	          this._index[key] = idx - 1;
	        }
	      }
	      return true;
	    }
	    return false;
	  }

	  /**
	   * Returns the list of items.
	   *
	   * @returns {object[]} The list of items.
	   */;
	  _proto.list = function list() {
	    return this._list;
	  }

	  /**
	   * Remove all items from the list.
	   */;
	  _proto.clear = function clear() {
	    this._list.length = 0;
	    for (var prop in this._index) {
	      delete this._index[prop];
	    }
	  };
	  return IndexedList;
	}();

	// wrapper function that caches the func result on first invocation and
	// then subsequently returns the cached value
	var cachedResult = function cachedResult(func) {
	  var uninitToken = {};
	  var result = uninitToken;
	  return function () {
	    if (result === uninitToken) {
	      result = func();
	    }
	    return result;
	  };
	};
	var Impl = /*#__PURE__*/function () {
	  function Impl() {}
	  // load a script
	  Impl.loadScript = function loadScript(url, callback) {
	    var s = document.createElement('script');
	    s.setAttribute('src', url);
	    s.onload = function () {
	      callback(null);
	    };
	    s.onerror = function () {
	      callback("Failed to load script='" + url + "'");
	    };
	    document.body.appendChild(s);
	  }

	  // load a wasm module
	  ;
	  Impl.loadWasm = function loadWasm(moduleName, config, callback) {
	    var loadUrl = Impl.wasmSupported() && config.glueUrl && config.wasmUrl ? config.glueUrl : config.fallbackUrl;
	    if (loadUrl) {
	      Impl.loadScript(loadUrl, function (err) {
	        if (err) {
	          callback(err, null);
	        } else {
	          var module = window[moduleName];

	          // clear the module from the global window since we used to store global instance here
	          window[moduleName] = undefined;

	          // instantiate the module
	          module({
	            locateFile: function locateFile() {
	              return config.wasmUrl;
	            },
	            onAbort: function onAbort() {
	              callback('wasm module aborted.');
	            }
	          }).then(function (instance) {
	            callback(null, instance);
	          });
	        }
	      });
	    } else {
	      callback('No supported wasm modules found.', null);
	    }
	  }

	  // get state object for the named module
	  ;
	  Impl.getModule = function getModule(name) {
	    if (!Impl.modules.hasOwnProperty(name)) {
	      Impl.modules[name] = {
	        config: null,
	        initializing: false,
	        instance: null,
	        callbacks: []
	      };
	    }
	    return Impl.modules[name];
	  };
	  Impl.initialize = function initialize(moduleName, module) {
	    if (module.initializing) {
	      return;
	    }
	    var config = module.config;
	    if (config.glueUrl || config.wasmUrl || config.fallbackUrl) {
	      module.initializing = true;
	      Impl.loadWasm(moduleName, config, function (err, instance) {
	        if (err) {
	          if (config.errorHandler) {
	            config.errorHandler(err);
	          } else {
	            console.error("failed to initialize module=" + moduleName + " error=" + err);
	          }
	        } else {
	          module.instance = instance;
	          module.callbacks.forEach(function (callback) {
	            callback(instance);
	          });
	        }
	      });
	    }
	  };
	  return Impl;
	}();
	/**
	 * Callback used by {@link Module#setConfig}.
	 *
	 * @callback ModuleErrorCallback
	 * @param {string} error - If the instance fails to load this will contain a description of the error.
	 */
	/**
	 * Callback used by {@link Module#getInstance}.
	 *
	 * @callback ModuleInstanceCallback
	 * @param {any} moduleInstance - The module instance.
	 */
	/**
	 * A pure static utility class which supports immediate and lazy loading of wasm modules.
	 */
	Impl.modules = {};
	// returns true if the running host supports wasm modules (all browsers except IE)
	Impl.wasmSupported = cachedResult(function () {
	  try {
	    if (typeof WebAssembly === "object" && typeof WebAssembly.instantiate === "function") {
	      var module = new WebAssembly.Module(Uint8Array.of(0x0, 0x61, 0x73, 0x6d, 0x01, 0x00, 0x00, 0x00));
	      if (module instanceof WebAssembly.Module) return new WebAssembly.Instance(module) instanceof WebAssembly.Instance;
	    }
	  } catch (e) {}
	  return false;
	});
	var WasmModule = /*#__PURE__*/function () {
	  function WasmModule() {}
	  /**
	   * Set a wasm module's configuration.
	   *
	   * @param {string} moduleName - Name of the module.
	   * @param {object} [config] - The configuration object.
	   * @param {string} [config.glueUrl] - URL of glue script.
	   * @param {string} [config.wasmUrl] - URL of the wasm script.
	   * @param {string} [config.fallbackUrl] - URL of the fallback script to use when wasm modules
	   * aren't supported.
	   * @param {number} [config.numWorkers] - For modules running on worker threads, the number of
	   * threads to use. Default value is based on module implementation.
	   * @param {ModuleErrorCallback} [config.errorHandler] - Function to be called if the module fails
	   * to download.
	   */
	  WasmModule.setConfig = function setConfig(moduleName, config) {
	    var module = Impl.getModule(moduleName);
	    module.config = config;
	    if (module.callbacks.length > 0) {
	      // start module initialize immediately since there are pending getInstance requests
	      Impl.initialize(moduleName, module);
	    }
	  }

	  /**
	   * Get a wasm module's configuration.
	   *
	   * @param {string} moduleName - Name of the module.
	   * @returns {object | undefined} The previously set configuration.
	   */;
	  WasmModule.getConfig = function getConfig(moduleName) {
	    var _Impl$modules;
	    return (_Impl$modules = Impl.modules) == null || (_Impl$modules = _Impl$modules[moduleName]) == null ? void 0 : _Impl$modules.config;
	  }

	  /**
	   * Get a wasm module instance. The instance will be created if necessary and returned
	   * in the second parameter to callback.
	   *
	   * @param {string} moduleName - Name of the module.
	   * @param {ModuleInstanceCallback} callback - The function called when the instance is
	   * available.
	   */;
	  WasmModule.getInstance = function getInstance(moduleName, callback) {
	    var module = Impl.getModule(moduleName);
	    if (module.instance) {
	      callback(module.instance);
	    } else {
	      module.callbacks.push(callback);
	      if (module.config) {
	        // config has been provided, kick off module initialize
	        Impl.initialize(moduleName, module);
	      }
	    }
	  };
	  return WasmModule;
	}();

	/**
	 * Helper class for organized reading of memory.
	 *
	 * @ignore
	 */
	var ReadStream = /*#__PURE__*/function () {
	  function ReadStream(arraybuffer) {
	    this.arraybuffer = arraybuffer;
	    this.dataView = new DataView(arraybuffer);
	    this.offset = 0;
	    this.stack = [];
	  }
	  var _proto = ReadStream.prototype;
	  _proto.reset = function reset(offset) {
	    if (offset === void 0) {
	      offset = 0;
	    }
	    this.offset = offset;
	  };
	  _proto.skip = function skip(bytes) {
	    this.offset += bytes;
	  };
	  _proto.align = function align(bytes) {
	    this.offset = this.offset + bytes - 1 & ~(bytes - 1);
	  };
	  _proto._inc = function _inc(amount) {
	    this.offset += amount;
	    return this.offset - amount;
	  };
	  _proto.readChar = function readChar() {
	    return String.fromCharCode(this.dataView.getUint8(this.offset++));
	  };
	  _proto.readChars = function readChars(numChars) {
	    var result = '';
	    for (var i = 0; i < numChars; ++i) {
	      result += this.readChar();
	    }
	    return result;
	  };
	  _proto.readU8 = function readU8() {
	    return this.dataView.getUint8(this.offset++);
	  };
	  _proto.readU16 = function readU16() {
	    return this.dataView.getUint16(this._inc(2), true);
	  };
	  _proto.readU32 = function readU32() {
	    return this.dataView.getUint32(this._inc(4), true);
	  };
	  _proto.readU64 = function readU64() {
	    return this.readU32() + Math.pow(2, 32) * this.readU32();
	  }

	  // big-endian
	  ;
	  _proto.readU32be = function readU32be() {
	    return this.dataView.getUint32(this._inc(4), false);
	  };
	  _proto.readArray = function readArray(result) {
	    for (var i = 0; i < result.length; ++i) {
	      result[i] = this.readU8();
	    }
	  };
	  _proto.readLine = function readLine() {
	    var view = this.dataView;
	    var result = '';
	    while (true) {
	      if (this.offset >= view.byteLength) {
	        break;
	      }
	      var c = String.fromCharCode(this.readU8());
	      if (c === '\n') {
	        break;
	      }
	      result += c;
	    }
	    return result;
	  };
	  _createClass(ReadStream, [{
	    key: "remainingBytes",
	    get: function get() {
	      return this.dataView.byteLength - this.offset;
	    }
	  }]);
	  return ReadStream;
	}();

	/**
	 * Helper class used to hold an array of items in a specific order. This array is safe to modify
	 * while we loop through it. The class assumes that it holds objects that need to be sorted based
	 * on one of their fields.
	 *
	 * @ignore
	 */
	var SortedLoopArray = /*#__PURE__*/function () {
	  /**
	   * Create a new SortedLoopArray instance.
	   *
	   * @param {object} args - Arguments.
	   * @param {string} args.sortBy - The name of the field that each element in the array is going
	   * to be sorted by.
	   * @example
	   * const array = new pc.SortedLoopArray({ sortBy: 'priority' });
	   * array.insert(item); // adds item to the right slot based on item.priority
	   * array.append(item); // adds item to the end of the array
	   * array.remove(item); // removes item from array
	   * for (array.loopIndex = 0; array.loopIndex < array.length; array.loopIndex++) {
	   *   // do things with array elements
	   *   // safe to remove and add elements into the array while looping
	   * }
	   */
	  function SortedLoopArray(args) {
	    /**
	     * The internal array that holds the actual array elements.
	     *
	     * @type {object[]}
	     */
	    this.items = [];
	    /**
	     * The number of elements in the array.
	     *
	     * @type {number}
	     */
	    this.length = 0;
	    /**
	     * The current index used to loop through the array. This gets modified if we add or remove
	     * elements from the array while looping. See the example to see how to loop through this
	     * array.
	     *
	     * @type {number}
	     */
	    this.loopIndex = -1;
	    /** @private */
	    this._sortBy = void 0;
	    /** @private */
	    this._sortHandler = void 0;
	    this._sortBy = args.sortBy;
	    this._sortHandler = this._doSort.bind(this);
	  }

	  /**
	   * Searches for the right spot to insert the specified item.
	   *
	   * @param {object} item - The item.
	   * @returns {number} The index where to insert the item.
	   * @private
	   */
	  var _proto = SortedLoopArray.prototype;
	  _proto._binarySearch = function _binarySearch(item) {
	    var left = 0;
	    var right = this.items.length - 1;
	    var search = item[this._sortBy];
	    var middle;
	    var current;
	    while (left <= right) {
	      middle = Math.floor((left + right) / 2);
	      current = this.items[middle][this._sortBy];
	      if (current <= search) {
	        left = middle + 1;
	      } else if (current > search) {
	        right = middle - 1;
	      }
	    }
	    return left;
	  };
	  _proto._doSort = function _doSort(a, b) {
	    var sortBy = this._sortBy;
	    return a[sortBy] - b[sortBy];
	  }

	  /**
	   * Inserts the specified item into the array at the right index based on the 'sortBy' field
	   * passed into the constructor. This also adjusts the loopIndex accordingly.
	   *
	   * @param {object} item - The item to insert.
	   */;
	  _proto.insert = function insert(item) {
	    var index = this._binarySearch(item);
	    this.items.splice(index, 0, item);
	    this.length++;
	    if (this.loopIndex >= index) {
	      this.loopIndex++;
	    }
	  }

	  /**
	   * Appends the specified item to the end of the array. Faster than insert() as it does not
	   * binary search for the right index. This also adjusts the loopIndex accordingly.
	   *
	   * @param {object} item - The item to append.
	   */;
	  _proto.append = function append(item) {
	    this.items.push(item);
	    this.length++;
	  }

	  /**
	   * Removes the specified item from the array.
	   *
	   * @param {object} item - The item to remove.
	   */;
	  _proto.remove = function remove(item) {
	    var idx = this.items.indexOf(item);
	    if (idx < 0) return;
	    this.items.splice(idx, 1);
	    this.length--;
	    if (this.loopIndex >= idx) {
	      this.loopIndex--;
	    }
	  }

	  /**
	   * Sorts elements in the array based on the 'sortBy' field passed into the constructor. This
	   * also updates the loopIndex if we are currently looping.
	   *
	   * WARNING: Be careful if you are sorting while iterating because if after sorting the array
	   * element that you are currently processing is moved behind other elements then you might end
	   * up iterating over elements more than once!
	   */;
	  _proto.sort = function sort() {
	    // get current item pointed to by loopIndex
	    var current = this.loopIndex >= 0 ? this.items[this.loopIndex] : null;
	    // sort
	    this.items.sort(this._sortHandler);
	    // find new loopIndex
	    if (current !== null) {
	      this.loopIndex = this.items.indexOf(current);
	    }
	  };
	  return SortedLoopArray;
	}();

	/**
	 * Set of tag names. Tags are automatically available on {@link Entity} and {@link Asset} as `tags`
	 * field.
	 *
	 * @augments EventHandler
	 */
	var Tags = /*#__PURE__*/function (_EventHandler) {
	  _inheritsLoose(Tags, _EventHandler);
	  /**
	   * Create an instance of a Tags.
	   *
	   * @param {object} [parent] - Parent object who tags belong to.
	   */
	  function Tags(parent) {
	    var _this;
	    _this = _EventHandler.call(this) || this;
	    /** @private */
	    _this._index = {};
	    /** @private */
	    _this._list = [];
	    _this._parent = parent;
	    return _this;
	  }

	  /**
	   * Add a tag, duplicates are ignored. Can be array or comma separated arguments for multiple tags.
	   *
	   * @param {...*} name - Name of a tag, or array of tags.
	   * @returns {boolean} True if any tag were added.
	   * @example
	   * tags.add('level-1');
	   * @example
	   * tags.add('ui', 'settings');
	   * @example
	   * tags.add(['level-2', 'mob']);
	   */
	  var _proto = Tags.prototype;
	  _proto.add = function add() {
	    var changed = false;
	    var tags = this._processArguments(arguments, true);
	    if (!tags.length) return changed;
	    for (var i = 0; i < tags.length; i++) {
	      if (this._index[tags[i]]) continue;
	      changed = true;
	      this._index[tags[i]] = true;
	      this._list.push(tags[i]);
	      this.fire('add', tags[i], this._parent);
	    }
	    if (changed) this.fire('change', this._parent);
	    return changed;
	  }

	  /**
	   * Remove tag.
	   *
	   * @param {...*} name - Name of a tag or array of tags.
	   * @returns {boolean} True if any tag were removed.
	   * @example
	   * tags.remove('level-1');
	   * @example
	   * tags.remove('ui', 'settings');
	   * @example
	   * tags.remove(['level-2', 'mob']);
	   */;
	  _proto.remove = function remove() {
	    var changed = false;
	    if (!this._list.length) return changed;
	    var tags = this._processArguments(arguments, true);
	    if (!tags.length) return changed;
	    for (var i = 0; i < tags.length; i++) {
	      if (!this._index[tags[i]]) continue;
	      changed = true;
	      delete this._index[tags[i]];
	      this._list.splice(this._list.indexOf(tags[i]), 1);
	      this.fire('remove', tags[i], this._parent);
	    }
	    if (changed) this.fire('change', this._parent);
	    return changed;
	  }

	  /**
	   * Remove all tags.
	   *
	   * @example
	   * tags.clear();
	   */;
	  _proto.clear = function clear() {
	    if (!this._list.length) return;
	    var tags = this._list.slice(0);
	    this._list = [];
	    this._index = {};
	    for (var i = 0; i < tags.length; i++) this.fire('remove', tags[i], this._parent);
	    this.fire('change', this._parent);
	  }

	  /**
	   * Check if tags satisfy filters. Filters can be provided by simple name of tag, as well as by
	   * array of tags. When an array is provided it will check if tags contain each tag within the
	   * array. If any of comma separated argument is satisfied, then it will return true. Any number
	   * of combinations are valid, and order is irrelevant.
	   *
	   * @param {...*} query - Name of a tag or array of tags.
	   * @returns {boolean} True if filters are satisfied.
	   * @example
	   * tags.has('player'); // player
	   * @example
	   * tags.has('mob', 'player'); // player OR mob
	   * @example
	   * tags.has(['level-1', 'mob']); // monster AND level-1
	   * @example
	   * tags.has(['ui', 'settings'], ['ui', 'levels']); // (ui AND settings) OR (ui AND levels)
	   */;
	  _proto.has = function has() {
	    if (!this._list.length) return false;
	    return this._has(this._processArguments(arguments));
	  }

	  /**
	   * @param {string[]|string[][]} tags - Array of tags.
	   * @returns {boolean} True if the supplied tags are present.
	   * @private
	   */;
	  _proto._has = function _has(tags) {
	    if (!this._list.length || !tags.length) return false;
	    for (var i = 0; i < tags.length; i++) {
	      if (tags[i].length === 1) {
	        // single occurrence
	        if (this._index[tags[i][0]]) return true;
	      } else {
	        // combined occurrence
	        var multiple = true;
	        for (var t = 0; t < tags[i].length; t++) {
	          if (this._index[tags[i][t]]) continue;
	          multiple = false;
	          break;
	        }
	        if (multiple) return true;
	      }
	    }
	    return false;
	  }

	  /**
	   * Returns immutable array of tags.
	   *
	   * @returns {string[]} Copy of tags array.
	   */;
	  _proto.list = function list() {
	    return this._list.slice(0);
	  }

	  /**
	   * @param {IArguments} args - Arguments to process.
	   * @param {boolean} [flat] - If true, will flatten array of tags. Defaults to false.
	   * @returns {string[]|string[][]} Array of tags.
	   * @private
	   */;
	  _proto._processArguments = function _processArguments(args, flat) {
	    var tags = [];
	    var tmp = [];
	    if (!args || !args.length) return tags;
	    for (var i = 0; i < args.length; i++) {
	      if (args[i] instanceof Array) {
	        if (!flat) tmp = [];
	        for (var t = 0; t < args[i].length; t++) {
	          if (typeof args[i][t] !== 'string') continue;
	          if (flat) {
	            tags.push(args[i][t]);
	          } else {
	            tmp.push(args[i][t]);
	          }
	        }
	        if (!flat && tmp.length) tags.push(tmp);
	      } else if (typeof args[i] === 'string') {
	        if (flat) {
	          tags.push(args[i]);
	        } else {
	          tags.push([args[i]]);
	        }
	      }
	    }
	    return tags;
	  }

	  /**
	   * Number of tags in set.
	   *
	   * @type {number}
	   */;
	  _createClass(Tags, [{
	    key: "size",
	    get: function get() {
	      return this._list.length;
	    }
	  }]);
	  return Tags;
	}(EventHandler);
	/**
	 * Fired for each individual tag that is added.
	 *
	 * @event
	 * @example
	 * tags.on('add', (tag, parent) => {
	 *    console.log(`${tag} added to ${parent.name}`);
	 * });
	 */
	Tags.EVENT_ADD = 'add';
	/**
	 * Fired for each individual tag that is removed.
	 *
	 * @event
	 * @example
	 * tags.on('remove', (tag, parent) => {
	 *   console.log(`${tag} removed from ${parent.name}`);
	 * });
	 */
	Tags.EVENT_REMOVE = 'remove';
	/**
	 * Fired when tags have been added or removed. It will fire once on bulk changes, while `add`
	 * and `remove` will fire on each tag operation.
	 *
	 * @event
	 * @example
	 * tags.on('change', (parent) => {
	 *    console.log(`Tags changed on ${parent.name}`);
	 * });
	 */
	Tags.EVENT_CHANGE = 'change';

	/**
	 * Get current time in milliseconds. Use it to measure time difference. Reference time may differ
	 * on different platforms.
	 *
	 * @returns {number} The time in milliseconds.
	 * @ignore
	 */
	var now = typeof window !== 'undefined' && window.performance && window.performance.now ? performance.now.bind(performance) : Date.now;

	/**
	 * Create a URI object from constituent parts.
	 *
	 * @param {object} options - Parts of the URI to build.
	 * @param {string} [options.scheme] - The URI scheme (e.g. http).
	 * @param {string} [options.authority] - The URI authority (e.g. `www.example.com`).
	 * @param {string} [options.host] - Combination of scheme and authority (e.g. `http://www.example.com`).
	 * @param {string} [options.path] - The URI path (e.g. /users/example).
	 * @param {string} [options.hostpath] - Combination of scheme, authority and path (e.g. `http://www.example.com/users/example`).
	 * @param {string} [options.query] - The query section, after the ?(e.g. `http://example.com?**key=value&another=123**`).
	 * @param {string} [options.fragment] - The fragment section, after the # (e.g. `http://example.com#**fragment/data**`).
	 * @returns {string} A URI string.
	 * @ignore
	 */
	function createURI(options) {
	  var s = '';
	  if ((options.authority || options.scheme) && (options.host || options.hostpath)) {
	    throw new Error('Can\'t have \'scheme\' or \'authority\' and \'host\' or \'hostpath\' option');
	  }
	  if (options.host && options.hostpath) {
	    throw new Error('Can\'t have \'host\' and \'hostpath\' option');
	  }
	  if (options.path && options.hostpath) {
	    throw new Error('Can\'t have \'path\' and \'hostpath\' option');
	  }
	  if (options.scheme) {
	    s += options.scheme + ':';
	  }
	  if (options.authority) {
	    s += '//' + options.authority;
	  }
	  if (options.host) {
	    s += options.host;
	  }
	  if (options.path) {
	    s += options.path;
	  }
	  if (options.hostpath) {
	    s += options.hostpath;
	  }
	  if (options.query) {
	    s += '?' + options.query;
	  }
	  if (options.fragment) {
	    s += '#' + options.fragment;
	  }
	  return s;
	}

	// See http://tools.ietf.org/html/rfc2396#appendix-B for details of RegExp
	var re = /^(([^:\/?#]+):)?(\/\/([^\/?#]*))?([^?#]*)(\?([^#]*))?(#(.*))?/;

	/**
	 * A URI object.
	 *
	 * @ignore
	 */
	var URI = /*#__PURE__*/function () {
	  /**
	   * Create a new URI instance.
	   *
	   * @param {string} uri - URI string.
	   */
	  function URI(uri) {
	    /**
	     * The scheme. (e.g. http).
	     *
	     * @type {string}
	     */
	    this.scheme = void 0;
	    /**
	     * The authority. (e.g. `www.example.com`).
	     *
	     * @type {string}
	     */
	    this.authority = void 0;
	    /**
	     * The path. (e.g. /users/example).
	     *
	     * @type {string}
	     */
	    this.path = void 0;
	    /**
	     * The query, the section after a ?. (e.g. search=value).
	     *
	     * @type {string}
	     */
	    this.query = void 0;
	    /**
	     * The fragment, the section after a #.
	     *
	     * @type {string}
	     */
	    this.fragment = void 0;
	    var result = uri.match(re);
	    this.scheme = result[2];
	    this.authority = result[4];
	    this.path = result[5];
	    this.query = result[7];
	    this.fragment = result[9];
	  }

	  /**
	   * Convert URI back to string.
	   *
	   * @returns {string} The URI as a string.
	   */
	  var _proto = URI.prototype;
	  _proto.toString = function toString() {
	    var s = '';
	    if (this.scheme) {
	      s += this.scheme + ':';
	    }
	    if (this.authority) {
	      s += '//' + this.authority;
	    }
	    s += this.path;
	    if (this.query) {
	      s += '?' + this.query;
	    }
	    if (this.fragment) {
	      s += '#' + this.fragment;
	    }
	    return s;
	  }

	  /**
	   * Returns the query parameters as an Object.
	   *
	   * @returns {object} The URI's query parameters converted to an Object.
	   * @example
	   * const s = "http://example.com?a=1&b=2&c=3";
	   * const uri = new pc.URI(s);
	   * const q = uri.getQuery();
	   * console.log(q.a); // logs "1"
	   * console.log(q.b); // logs "2"
	   * console.log(q.c); // logs "3"
	   */;
	  _proto.getQuery = function getQuery() {
	    var result = {};
	    if (this.query) {
	      var queryParams = decodeURIComponent(this.query).split('&');
	      for (var _iterator = _createForOfIteratorHelperLoose(queryParams), _step; !(_step = _iterator()).done;) {
	        var queryParam = _step.value;
	        var pair = queryParam.split('=');
	        result[pair[0]] = pair[1];
	      }
	    }
	    return result;
	  }

	  /**
	   * Set the query section of the URI from a Object.
	   *
	   * @param {object} params - Key-Value pairs to encode into the query string.
	   * @example
	   * const s = "http://example.com";
	   * const uri = new pc.URI(s);
	   * uri.setQuery({
	   *     "a": 1,
	   *     "b": 2
	   * });
	   * console.log(uri.toString()); // logs "http://example.com?a=1&b=2
	   */;
	  _proto.setQuery = function setQuery(params) {
	    var q = '';
	    for (var key in params) {
	      if (params.hasOwnProperty(key)) {
	        if (q !== '') {
	          q += '&';
	        }
	        q += encodeURIComponent(key) + '=' + encodeURIComponent(params[key]);
	      }
	    }
	    this.query = q;
	  };
	  return URI;
	}();

	/**
	 * A linear interpolation scheme.
	 *
	 * @type {number}
	 * @category Math
	 */
	var CURVE_LINEAR = 0;

	/**
	 * A smooth step interpolation scheme.
	 *
	 * @type {number}
	 * @category Math
	 */
	var CURVE_SMOOTHSTEP = 1;

	/**
	 * A Catmull-Rom spline interpolation scheme. This interpolation scheme is deprecated. Use
	 * CURVE_SPLINE instead.
	 *
	 * @type {number}
	 * @deprecated
	 * @ignore
	 */
	var CURVE_CATMULL = 2;

	/**
	 * A cardinal spline interpolation scheme. This interpolation scheme is deprecated. Use
	 * CURVE_SPLINE instead.
	 *
	 * @type {number}
	 * @deprecated
	 * @ignore
	 */
	var CURVE_CARDINAL = 3;

	/**
	 * Cardinal spline interpolation scheme. For Catmull-Rom, specify curve tension 0.5.
	 *
	 * @type {number}
	 * @category Math
	 */
	var CURVE_SPLINE = 4;

	/**
	 * A stepped interpolator, free from the shackles of blending.
	 *
	 * @type {number}
	 * @category Math
	 */
	var CURVE_STEP = 5;

	/**
	 * Math API.
	 *
	 * @namespace
	 * @category Math
	 */
	var math = {
	  /**
	   * Conversion factor between degrees and radians.
	   *
	   * @type {number}
	   */
	  DEG_TO_RAD: Math.PI / 180,
	  /**
	   * Conversion factor between degrees and radians.
	   *
	   * @type {number}
	   */
	  RAD_TO_DEG: 180 / Math.PI,
	  /**
	   * Clamp a number between min and max inclusive.
	   *
	   * @param {number} value - Number to clamp.
	   * @param {number} min - Min value.
	   * @param {number} max - Max value.
	   * @returns {number} The clamped value.
	   */
	  clamp: function clamp(value, min, max) {
	    if (value >= max) return max;
	    if (value <= min) return min;
	    return value;
	  },
	  /**
	   * Convert an 24 bit integer into an array of 3 bytes.
	   *
	   * @param {number} i - Number holding an integer value.
	   * @returns {number[]} An array of 3 bytes.
	   * @example
	   * // Set bytes to [0x11, 0x22, 0x33]
	   * const bytes = pc.math.intToBytes24(0x112233);
	   */
	  intToBytes24: function intToBytes24(i) {
	    var r = i >> 16 & 0xff;
	    var g = i >> 8 & 0xff;
	    var b = i & 0xff;
	    return [r, g, b];
	  },
	  /**
	   * Convert an 32 bit integer into an array of 4 bytes.
	   *
	   * @param {number} i - Number holding an integer value.
	   * @returns {number[]} An array of 4 bytes.
	   * @example
	   * // Set bytes to [0x11, 0x22, 0x33, 0x44]
	   * const bytes = pc.math.intToBytes32(0x11223344);
	   */
	  intToBytes32: function intToBytes32(i) {
	    var r = i >> 24 & 0xff;
	    var g = i >> 16 & 0xff;
	    var b = i >> 8 & 0xff;
	    var a = i & 0xff;
	    return [r, g, b, a];
	  },
	  /**
	   * Convert 3 8 bit Numbers into a single unsigned 24 bit Number.
	   *
	   * @param {number} r - A single byte (0-255).
	   * @param {number} g - A single byte (0-255).
	   * @param {number} b - A single byte (0-255).
	   * @returns {number} A single unsigned 24 bit Number.
	   * @example
	   * // Set result1 to 0x112233 from an array of 3 values
	   * const result1 = pc.math.bytesToInt24([0x11, 0x22, 0x33]);
	   *
	   * // Set result2 to 0x112233 from 3 discrete values
	   * const result2 = pc.math.bytesToInt24(0x11, 0x22, 0x33);
	   */
	  bytesToInt24: function bytesToInt24(r, g, b) {
	    if (r.length) {
	      b = r[2];
	      g = r[1];
	      r = r[0];
	    }
	    return r << 16 | g << 8 | b;
	  },
	  /**
	   * Convert 4 1-byte Numbers into a single unsigned 32bit Number.
	   *
	   * @param {number} r - A single byte (0-255).
	   * @param {number} g - A single byte (0-255).
	   * @param {number} b - A single byte (0-255).
	   * @param {number} a - A single byte (0-255).
	   * @returns {number} A single unsigned 32bit Number.
	   * @example
	   * // Set result1 to 0x11223344 from an array of 4 values
	   * const result1 = pc.math.bytesToInt32([0x11, 0x22, 0x33, 0x44]);
	   *
	   * // Set result2 to 0x11223344 from 4 discrete values
	   * const result2 = pc.math.bytesToInt32(0x11, 0x22, 0x33, 0x44);
	   */
	  bytesToInt32: function bytesToInt32(r, g, b, a) {
	    if (r.length) {
	      a = r[3];
	      b = r[2];
	      g = r[1];
	      r = r[0];
	    }

	    // Why ((r << 24)>>>0)?
	    // << operator uses signed 32 bit numbers, so 128<<24 is negative.
	    // >>> used unsigned so >>>0 converts back to an unsigned.
	    // See https://stackoverflow.com/questions/1908492/unsigned-integer-in-javascript
	    return (r << 24 | g << 16 | b << 8 | a) >>> 0;
	  },
	  /**
	   * Calculates the linear interpolation of two numbers.
	   *
	   * @param {number} a - Number to linearly interpolate from.
	   * @param {number} b - Number to linearly interpolate to.
	   * @param {number} alpha - The value controlling the result of interpolation. When alpha is 0,
	   * a is returned. When alpha is 1, b is returned. Between 0 and 1, a linear interpolation
	   * between a and b is returned. alpha is clamped between 0 and 1.
	   * @returns {number} The linear interpolation of two numbers.
	   */
	  lerp: function lerp(a, b, alpha) {
	    return a + (b - a) * math.clamp(alpha, 0, 1);
	  },
	  /**
	   * Calculates the linear interpolation of two angles ensuring that interpolation is correctly
	   * performed across the 360 to 0 degree boundary. Angles are supplied in degrees.
	   *
	   * @param {number} a - Angle (in degrees) to linearly interpolate from.
	   * @param {number} b - Angle (in degrees) to linearly interpolate to.
	   * @param {number} alpha - The value controlling the result of interpolation. When alpha is 0,
	   * a is returned. When alpha is 1, b is returned. Between 0 and 1, a linear interpolation
	   * between a and b is returned. alpha is clamped between 0 and 1.
	   * @returns {number} The linear interpolation of two angles.
	   */
	  lerpAngle: function lerpAngle(a, b, alpha) {
	    if (b - a > 180) {
	      b -= 360;
	    }
	    if (b - a < -180) {
	      b += 360;
	    }
	    return math.lerp(a, b, math.clamp(alpha, 0, 1));
	  },
	  /**
	   * Returns true if argument is a power-of-two and false otherwise.
	   *
	   * @param {number} x - Number to check for power-of-two property.
	   * @returns {boolean} true if power-of-two and false otherwise.
	   */
	  powerOfTwo: function powerOfTwo(x) {
	    return x !== 0 && !(x & x - 1);
	  },
	  /**
	   * Returns the next power of 2 for the specified value.
	   *
	   * @param {number} val - The value for which to calculate the next power of 2.
	   * @returns {number} The next power of 2.
	   */
	  nextPowerOfTwo: function nextPowerOfTwo(val) {
	    val--;
	    val |= val >> 1;
	    val |= val >> 2;
	    val |= val >> 4;
	    val |= val >> 8;
	    val |= val >> 16;
	    val++;
	    return val;
	  },
	  /**
	   * Returns the nearest (smaller or larger) power of 2 for the specified value.
	   *
	   * @param {number} val - The value for which to calculate the nearest power of 2.
	   * @returns {number} The nearest power of 2.
	   */
	  nearestPowerOfTwo: function nearestPowerOfTwo(val) {
	    return Math.pow(2, Math.round(Math.log(val) / Math.log(2)));
	  },
	  /**
	   * Return a pseudo-random number between min and max. The number generated is in the range
	   * [min, max), that is inclusive of the minimum but exclusive of the maximum.
	   *
	   * @param {number} min - Lower bound for range.
	   * @param {number} max - Upper bound for range.
	   * @returns {number} Pseudo-random number between the supplied range.
	   */
	  random: function random(min, max) {
	    var diff = max - min;
	    return Math.random() * diff + min;
	  },
	  /**
	   * The function interpolates smoothly between two input values based on a third one that should
	   * be between the first two. The returned value is clamped between 0 and 1.
	   *
	   * The slope (i.e. derivative) of the smoothstep function starts at 0 and ends at 0. This makes
	   * it easy to create a sequence of transitions using smoothstep to interpolate each segment
	   * rather than using a more sophisticated or expensive interpolation technique.
	   *
	   * See https://en.wikipedia.org/wiki/Smoothstep for more details.
	   *
	   * @param {number} min - The lower bound of the interpolation range.
	   * @param {number} max - The upper bound of the interpolation range.
	   * @param {number} x - The value to interpolate.
	   * @returns {number} The smoothly interpolated value clamped between zero and one.
	   */
	  smoothstep: function smoothstep(min, max, x) {
	    if (x <= min) return 0;
	    if (x >= max) return 1;
	    x = (x - min) / (max - min);
	    return x * x * (3 - 2 * x);
	  },
	  /**
	   * An improved version of the {@link math.smoothstep} function which has zero 1st and 2nd order
	   * derivatives at t=0 and t=1.
	   *
	   * See https://en.wikipedia.org/wiki/Smoothstep#Variations for more details.
	   *
	   * @param {number} min - The lower bound of the interpolation range.
	   * @param {number} max - The upper bound of the interpolation range.
	   * @param {number} x - The value to interpolate.
	   * @returns {number} The smoothly interpolated value clamped between zero and one.
	   */
	  smootherstep: function smootherstep(min, max, x) {
	    if (x <= min) return 0;
	    if (x >= max) return 1;
	    x = (x - min) / (max - min);
	    return x * x * x * (x * (x * 6 - 15) + 10);
	  },
	  /**
	   * Rounds a number up to nearest multiple.
	   *
	   * @param {number} numToRound - The number to round up.
	   * @param {number} multiple - The multiple to round up to.
	   * @returns {number} A number rounded up to nearest multiple.
	   */
	  roundUp: function roundUp(numToRound, multiple) {
	    if (multiple === 0) return numToRound;
	    return Math.ceil(numToRound / multiple) * multiple;
	  },
	  /**
	   * Checks whether a given number resides between two other given numbers.
	   *
	   * @param {number} num - The number to check the position of.
	   * @param {number} a - The first upper or lower threshold to check between.
	   * @param {number} b - The second upper or lower threshold to check between.
	   * @param {boolean} inclusive - If true, a num param which is equal to a or b will return true.
	   * @returns {boolean} true if between or false otherwise.
	   * @ignore
	   */
	  between: function between(num, a, b, inclusive) {
	    var min = Math.min(a, b);
	    var max = Math.max(a, b);
	    return inclusive ? num >= min && num <= max : num > min && num < max;
	  }
	};

	var _class$a;

	/**
	 * Representation of an RGBA color.
	 *
	 * @category Math
	 */
	var Color = /*#__PURE__*/function () {
	  /**
	   * Create a new Color object.
	   *
	   * @param {number|number[]} [r] - The value of the red component (0-1). Defaults to 0. If r is
	   * an array of length 3 or 4, the array will be used to populate all components.
	   * @param {number} [g] - The value of the green component (0-1). Defaults to 0.
	   * @param {number} [b] - The value of the blue component (0-1). Defaults to 0.
	   * @param {number} [a] - The value of the alpha component (0-1). Defaults to 1.
	   */
	  function Color(r, g, b, a) {
	    if (r === void 0) {
	      r = 0;
	    }
	    if (g === void 0) {
	      g = 0;
	    }
	    if (b === void 0) {
	      b = 0;
	    }
	    if (a === void 0) {
	      a = 1;
	    }
	    /**
	     * The red component of the color.
	     *
	     * @type {number}
	     */
	    this.r = void 0;
	    /**
	     * The green component of the color.
	     *
	     * @type {number}
	     */
	    this.g = void 0;
	    /**
	     * The blue component of the color.
	     *
	     * @type {number}
	     */
	    this.b = void 0;
	    /**
	     * The alpha component of the color.
	     *
	     * @type {number}
	     */
	    this.a = void 0;
	    var length = r.length;
	    if (length === 3 || length === 4) {
	      this.r = r[0];
	      this.g = r[1];
	      this.b = r[2];
	      this.a = r[3] !== undefined ? r[3] : 1;
	    } else {
	      this.r = r;
	      this.g = g;
	      this.b = b;
	      this.a = a;
	    }
	  }

	  /**
	   * Returns a clone of the specified color.
	   *
	   * @returns {this} A duplicate color object.
	   */
	  var _proto = Color.prototype;
	  _proto.clone = function clone() {
	    /** @type {this} */
	    var cstr = this.constructor;
	    return new cstr(this.r, this.g, this.b, this.a);
	  }

	  /**
	   * Copies the contents of a source color to a destination color.
	   *
	   * @param {Color} rhs - A color to copy to the specified color.
	   * @returns {Color} Self for chaining.
	   * @example
	   * const src = new pc.Color(1, 0, 0, 1);
	   * const dst = new pc.Color();
	   *
	   * dst.copy(src);
	   *
	   * console.log("The two colors are " + (dst.equals(src) ? "equal" : "different"));
	   */;
	  _proto.copy = function copy(rhs) {
	    this.r = rhs.r;
	    this.g = rhs.g;
	    this.b = rhs.b;
	    this.a = rhs.a;
	    return this;
	  }

	  /**
	   * Reports whether two colors are equal.
	   *
	   * @param {Color} rhs - The color to compare to the specified color.
	   * @returns {boolean} True if the colors are equal and false otherwise.
	   * @example
	   * const a = new pc.Color(1, 0, 0, 1);
	   * const b = new pc.Color(1, 1, 0, 1);
	   * console.log("The two colors are " + (a.equals(b) ? "equal" : "different"));
	   */;
	  _proto.equals = function equals(rhs) {
	    return this.r === rhs.r && this.g === rhs.g && this.b === rhs.b && this.a === rhs.a;
	  }

	  /**
	   * Assign values to the color components, including alpha.
	   *
	   * @param {number} r - The value for red (0-1).
	   * @param {number} g - The value for blue (0-1).
	   * @param {number} b - The value for green (0-1).
	   * @param {number} [a] - The value for the alpha (0-1), defaults to 1.
	   * @returns {Color} Self for chaining.
	   */;
	  _proto.set = function set(r, g, b, a) {
	    if (a === void 0) {
	      a = 1;
	    }
	    this.r = r;
	    this.g = g;
	    this.b = b;
	    this.a = a;
	    return this;
	  }

	  /**
	   * Returns the result of a linear interpolation between two specified colors.
	   *
	   * @param {Color} lhs - The color to interpolate from.
	   * @param {Color} rhs - The color to interpolate to.
	   * @param {number} alpha - The value controlling the point of interpolation. Between 0 and 1,
	   * the linear interpolant will occur on a straight line between lhs and rhs. Outside of this
	   * range, the linear interpolant will occur on a ray extrapolated from this line.
	   * @returns {Color} Self for chaining.
	   * @example
	   * const a = new pc.Color(0, 0, 0);
	   * const b = new pc.Color(1, 1, 0.5);
	   * const r = new pc.Color();
	   *
	   * r.lerp(a, b, 0);   // r is equal to a
	   * r.lerp(a, b, 0.5); // r is 0.5, 0.5, 0.25
	   * r.lerp(a, b, 1);   // r is equal to b
	   */;
	  _proto.lerp = function lerp(lhs, rhs, alpha) {
	    this.r = lhs.r + alpha * (rhs.r - lhs.r);
	    this.g = lhs.g + alpha * (rhs.g - lhs.g);
	    this.b = lhs.b + alpha * (rhs.b - lhs.b);
	    this.a = lhs.a + alpha * (rhs.a - lhs.a);
	    return this;
	  }

	  /**
	   * Set the values of the color from a string representation '#11223344' or '#112233'.
	   *
	   * @param {string} hex - A string representation in the format '#RRGGBBAA' or '#RRGGBB'. Where
	   * RR, GG, BB, AA are red, green, blue and alpha values. This is the same format used in
	   * HTML/CSS.
	   * @returns {Color} Self for chaining.
	   */;
	  _proto.fromString = function fromString(hex) {
	    var i = parseInt(hex.replace('#', '0x'), 16);
	    var bytes;
	    if (hex.length > 7) {
	      bytes = math.intToBytes32(i);
	    } else {
	      bytes = math.intToBytes24(i);
	      bytes[3] = 255;
	    }
	    this.set(bytes[0] / 255, bytes[1] / 255, bytes[2] / 255, bytes[3] / 255);
	    return this;
	  }

	  /**
	   * Converts the color to string form. The format is '#RRGGBBAA', where RR, GG, BB, AA are the
	   * red, green, blue and alpha values. When the alpha value is not included (the default), this
	   * is the same format as used in HTML/CSS.
	   *
	   * @param {boolean} alpha - If true, the output string will include the alpha value.
	   * @returns {string} The color in string form.
	   * @example
	   * const c = new pc.Color(1, 1, 1);
	   * // Outputs #ffffffff
	   * console.log(c.toString());
	   */;
	  _proto.toString = function toString(alpha) {
	    var s = '#' + ((1 << 24) + (Math.round(this.r * 255) << 16) + (Math.round(this.g * 255) << 8) + Math.round(this.b * 255)).toString(16).slice(1);
	    if (alpha === true) {
	      var a = Math.round(this.a * 255).toString(16);
	      if (this.a < 16 / 255) {
	        s += '0' + a;
	      } else {
	        s += a;
	      }
	    }
	    return s;
	  }

	  /**
	   * A constant color set to black [0, 0, 0, 1].
	   *
	   * @type {Color}
	   * @readonly
	   */;
	  return Color;
	}();
	_class$a = Color;
	Color.BLACK = Object.freeze(new _class$a(0, 0, 0, 1));
	/**
	 * A constant color set to blue [0, 0, 1, 1].
	 *
	 * @type {Color}
	 * @readonly
	 */
	Color.BLUE = Object.freeze(new _class$a(0, 0, 1, 1));
	/**
	 * A constant color set to cyan [0, 1, 1, 1].
	 *
	 * @type {Color}
	 * @readonly
	 */
	Color.CYAN = Object.freeze(new _class$a(0, 1, 1, 1));
	/**
	 * A constant color set to gray [0.5, 0.5, 0.5, 1].
	 *
	 * @type {Color}
	 * @readonly
	 */
	Color.GRAY = Object.freeze(new _class$a(0.5, 0.5, 0.5, 1));
	/**
	 * A constant color set to green [0, 1, 0, 1].
	 *
	 * @type {Color}
	 * @readonly
	 */
	Color.GREEN = Object.freeze(new _class$a(0, 1, 0, 1));
	/**
	 * A constant color set to magenta [1, 0, 1, 1].
	 *
	 * @type {Color}
	 * @readonly
	 */
	Color.MAGENTA = Object.freeze(new _class$a(1, 0, 1, 1));
	/**
	 * A constant color set to red [1, 0, 0, 1].
	 *
	 * @type {Color}
	 * @readonly
	 */
	Color.RED = Object.freeze(new _class$a(1, 0, 0, 1));
	/**
	 * A constant color set to white [1, 1, 1, 1].
	 *
	 * @type {Color}
	 * @readonly
	 */
	Color.WHITE = Object.freeze(new _class$a(1, 1, 1, 1));
	/**
	 * A constant color set to yellow [1, 1, 0, 1].
	 *
	 * @type {Color}
	 * @readonly
	 */
	Color.YELLOW = Object.freeze(new _class$a(1, 1, 0, 1));

	/**
	 * A class for evaluating a curve at a specific time.
	 *
	 * @ignore
	 */
	var CurveEvaluator = /*#__PURE__*/function () {
	  /**
	   * Create a new CurveEvaluator instance.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./curve.js').Curve} curve - The curve to evaluate.
	   * @param {number} time - The initial time to evaluate the curve at. Defaults to 0.
	   */
	  function CurveEvaluator(curve, time) {
	    if (time === void 0) {
	      time = 0;
	    }
	    /** @private */
	    this._curve = void 0;
	    /** @private */
	    this._left = -Infinity;
	    /** @private */
	    this._right = Infinity;
	    /** @private */
	    this._recip = 0;
	    /** @private */
	    this._p0 = 0;
	    /** @private */
	    this._p1 = 0;
	    /** @private */
	    this._m0 = 0;
	    /** @private */
	    this._m1 = 0;
	    this._curve = curve;
	    this._reset(time);
	  }

	  /**
	   * Evaluate the curve at the given time. Specify forceReset if the underlying curve keys have
	   * changed since the last evaluation.
	   *
	   * @param {number} time - Time to evaluate the curve at.
	   * @param {boolean} [forceReset] - Force reset of the curve.
	   * @returns {number} The evaluated value.
	   */
	  var _proto = CurveEvaluator.prototype;
	  _proto.evaluate = function evaluate(time, forceReset) {
	    if (forceReset === void 0) {
	      forceReset = false;
	    }
	    if (forceReset || time < this._left || time >= this._right) {
	      this._reset(time);
	    }
	    var result;
	    var type = this._curve.type;
	    if (type === CURVE_STEP) {
	      // step
	      result = this._p0;
	    } else {
	      // calculate normalized t
	      var t = this._recip === 0 ? 0 : (time - this._left) * this._recip;
	      if (type === CURVE_LINEAR) {
	        // linear
	        result = math.lerp(this._p0, this._p1, t);
	      } else if (type === CURVE_SMOOTHSTEP) {
	        // smoothstep
	        result = math.lerp(this._p0, this._p1, t * t * (3 - 2 * t));
	      } else {
	        // curve
	        result = this._evaluateHermite(this._p0, this._p1, this._m0, this._m1, t);
	      }
	    }
	    return result;
	  }

	  /**
	   * Calculate weights for the curve interval at the given time.
	   *
	   * @param {number} time - Time to evaluate the curve at.
	   * @private
	   */;
	  _proto._reset = function _reset(time) {
	    var keys = this._curve.keys;
	    var len = keys.length;
	    if (!len) {
	      // curve is empty
	      this._left = -Infinity;
	      this._right = Infinity;
	      this._recip = 0;
	      this._p0 = this._p1 = this._m0 = this._m1 = 0;
	    } else {
	      if (time < keys[0][0]) {
	        // iterator falls to the left of the start of the curve
	        this._left = -Infinity;
	        this._right = keys[0][0];
	        this._recip = 0;
	        this._p0 = this._p1 = keys[0][1];
	        this._m0 = this._m1 = 0;
	      } else if (time >= keys[len - 1][0]) {
	        // iterator falls to the right of the end of the curve
	        this._left = keys[len - 1][0];
	        this._right = Infinity;
	        this._recip = 0;
	        this._p0 = this._p1 = keys[len - 1][1];
	        this._m0 = this._m1 = 0;
	      } else {
	        // iterator falls within the bounds of the curve
	        // perform a linear search for the key just left of the current time.
	        // (TODO: for cases where the curve has more than 'n' keys it will
	        // be more efficient to perform a binary search here instead. Which is
	        // straight forward thanks to the sorted list of knots).
	        var index = 0;
	        while (time >= keys[index + 1][0]) {
	          index++;
	        }
	        this._left = keys[index][0];
	        this._right = keys[index + 1][0];
	        var diff = 1.0 / (this._right - this._left);
	        this._recip = isFinite(diff) ? diff : 0;
	        this._p0 = keys[index][1];
	        this._p1 = keys[index + 1][1];
	        if (this._isHermite()) {
	          this._calcTangents(keys, index);
	        }
	      }
	    }
	  }

	  /**
	   * Returns whether the curve is a hermite.
	   *
	   * @returns {boolean} True if the curve is a hermite and false otherwise.
	   * @private
	   */;
	  _proto._isHermite = function _isHermite() {
	    return this._curve.type === CURVE_CATMULL || this._curve.type === CURVE_CARDINAL || this._curve.type === CURVE_SPLINE;
	  }

	  /**
	   * Calculate tangents for the hermite curve.
	   *
	   * @param {number[][]} keys - The keys of the curve.
	   * @param {number} index - The key index of the key to calculate the tangents for.
	   * @private
	   */;
	  _proto._calcTangents = function _calcTangents(keys, index) {
	    var a;
	    var b = keys[index];
	    var c = keys[index + 1];
	    var d;
	    if (index === 0) {
	      a = [keys[0][0] + (keys[0][0] - keys[1][0]), keys[0][1] + (keys[0][1] - keys[1][1])];
	    } else {
	      a = keys[index - 1];
	    }
	    if (index === keys.length - 2) {
	      d = [keys[index + 1][0] + (keys[index + 1][0] - keys[index][0]), keys[index + 1][1] + (keys[index + 1][1] - keys[index][1])];
	    } else {
	      d = keys[index + 2];
	    }
	    if (this._curve.type === CURVE_SPLINE) {
	      // calculate tangent scale (due to non-uniform knot spacing)
	      var s1_ = 2 * (c[0] - b[0]) / (c[0] - a[0]);
	      var s2_ = 2 * (c[0] - b[0]) / (d[0] - b[0]);
	      this._m0 = this._curve.tension * (isFinite(s1_) ? s1_ : 0) * (c[1] - a[1]);
	      this._m1 = this._curve.tension * (isFinite(s2_) ? s2_ : 0) * (d[1] - b[1]);
	    } else {
	      // original tangent scale calc
	      var s1 = (c[0] - b[0]) / (b[0] - a[0]);
	      var s2 = (c[0] - b[0]) / (d[0] - c[0]);
	      var a_ = b[1] + (a[1] - b[1]) * (isFinite(s1) ? s1 : 0);
	      var d_ = c[1] + (d[1] - c[1]) * (isFinite(s2) ? s2 : 0);
	      var tension = this._curve.type === CURVE_CATMULL ? 0.5 : this._curve.tension;
	      this._m0 = tension * (c[1] - a_);
	      this._m1 = tension * (d_ - b[1]);
	    }
	  }

	  /**
	   * Evaluate the hermite curve at the given time.
	   *
	   * @param {number} p0 - The first key.
	   * @param {number} p1 - The second key.
	   * @param {number} m0 - The first tangent.
	   * @param {number} m1 - The second tangent.
	   * @param {number} t - Time to evaluate the curve at.
	   * @returns {number} The value of the hermite curve at the given time.
	   * @private
	   */;
	  _proto._evaluateHermite = function _evaluateHermite(p0, p1, m0, m1, t) {
	    var t2 = t * t;
	    var twot = t + t;
	    var omt = 1 - t;
	    var omt2 = omt * omt;
	    return p0 * ((1 + twot) * omt2) + m0 * (t * omt2) + p1 * (t2 * (3 - twot)) + m1 * (t2 * (t - 1));
	  };
	  return CurveEvaluator;
	}();

	/**
	 * A curve is a collection of keys (time/value pairs). The shape of the curve is defined by its
	 * type that specifies an interpolation scheme for the keys.
	 *
	 * @category Math
	 */
	var Curve = /*#__PURE__*/function () {
	  /**
	   * Creates a new Curve instance.
	   *
	   * @param {number[]} [data] - An array of keys (pairs of numbers with the time first and value
	   * second).
	   * @example
	   * const curve = new pc.Curve([
	   *     0, 0,        // At 0 time, value of 0
	   *     0.33, 2,     // At 0.33 time, value of 2
	   *     0.66, 2.6,   // At 0.66 time, value of 2.6
	   *     1, 3         // At 1 time, value of 3
	   * ]);
	   */
	  function Curve(data) {
	    this.keys = [];
	    /**
	     * The curve interpolation scheme. Can be:
	     *
	     * - {@link CURVE_LINEAR}
	     * - {@link CURVE_SMOOTHSTEP}
	     * - {@link CURVE_SPLINE}
	     * - {@link CURVE_STEP}
	     *
	     * Defaults to {@link CURVE_SMOOTHSTEP}.
	     *
	     * @type {number}
	     */
	    this.type = CURVE_SMOOTHSTEP;
	    /**
	     * Controls how {@link CURVE_SPLINE} tangents are calculated. Valid range is between 0 and 1
	     * where 0 results in a non-smooth curve (equivalent to linear interpolation) and 1 results in
	     * a very smooth curve. Use 0.5 for a Catmull-rom spline.
	     *
	     * @type {number}
	     */
	    this.tension = 0.5;
	    /**
	     * @type {CurveEvaluator}
	     * @private
	     */
	    this._eval = new CurveEvaluator(this);
	    if (data) {
	      for (var i = 0; i < data.length - 1; i += 2) {
	        this.keys.push([data[i], data[i + 1]]);
	      }
	    }
	    this.sort();
	  }

	  /**
	   * Get the number of keys in the curve.
	   *
	   * @type {number}
	   */
	  var _proto = Curve.prototype;
	  /**
	   * Add a new key to the curve.
	   *
	   * @param {number} time - Time to add new key.
	   * @param {number} value - Value of new key.
	   * @returns {number[]} [time, value] pair.
	   */
	  _proto.add = function add(time, value) {
	    var keys = this.keys;
	    var len = keys.length;
	    var i = 0;
	    for (; i < len; i++) {
	      if (keys[i][0] > time) {
	        break;
	      }
	    }
	    var key = [time, value];
	    this.keys.splice(i, 0, key);
	    return key;
	  }

	  /**
	   * Return a specific key.
	   *
	   * @param {number} index - The index of the key to return.
	   * @returns {number[]} The key at the specified index.
	   */;
	  _proto.get = function get(index) {
	    return this.keys[index];
	  }

	  /**
	   * Sort keys by time.
	   */;
	  _proto.sort = function sort() {
	    this.keys.sort(function (a, b) {
	      return a[0] - b[0];
	    });
	  }

	  /**
	   * Returns the interpolated value of the curve at specified time.
	   *
	   * @param {number} time - The time at which to calculate the value.
	   * @returns {number} The interpolated value.
	   */;
	  _proto.value = function value(time) {
	    // we force reset the evaluation because keys may have changed since the last evaluate
	    // (we can't know)
	    return this._eval.evaluate(time, true);
	  };
	  _proto.closest = function closest(time) {
	    var keys = this.keys;
	    var length = keys.length;
	    var min = 2;
	    var result = null;
	    for (var i = 0; i < length; i++) {
	      var diff = Math.abs(time - keys[i][0]);
	      if (min >= diff) {
	        min = diff;
	        result = keys[i];
	      } else {
	        break;
	      }
	    }
	    return result;
	  }

	  /**
	   * Returns a clone of the specified curve object.
	   *
	   * @returns {this} A clone of the specified curve.
	   */;
	  _proto.clone = function clone() {
	    /** @type {this} */
	    var result = new this.constructor();
	    result.keys = extend(result.keys, this.keys);
	    result.type = this.type;
	    result.tension = this.tension;
	    return result;
	  }

	  /**
	   * Sample the curve at regular intervals over the range [0..1].
	   *
	   * @param {number} precision - The number of samples to return.
	   * @returns {Float32Array} The set of quantized values.
	   * @ignore
	   */;
	  _proto.quantize = function quantize(precision) {
	    precision = Math.max(precision, 2);
	    var values = new Float32Array(precision);
	    var step = 1.0 / (precision - 1);

	    // quantize graph to table of interpolated values
	    values[0] = this._eval.evaluate(0, true);
	    for (var i = 1; i < precision; i++) {
	      values[i] = this._eval.evaluate(step * i);
	    }
	    return values;
	  }

	  /**
	   * Sample the curve at regular intervals over the range [0..1] and clamp the resulting samples
	   * to [min..max].
	   *
	   * @param {number} precision - The number of samples to return.
	   * @param {number} min - The minimum output value.
	   * @param {number} max - The maximum output value.
	   * @returns {Float32Array} The set of quantized values.
	   * @ignore
	   */;
	  _proto.quantizeClamped = function quantizeClamped(precision, min, max) {
	    var result = this.quantize(precision);
	    for (var i = 0; i < result.length; ++i) {
	      result[i] = Math.min(max, Math.max(min, result[i]));
	    }
	    return result;
	  };
	  _createClass(Curve, [{
	    key: "length",
	    get: function get() {
	      return this.keys.length;
	    }
	  }]);
	  return Curve;
	}();

	/**
	 * A curve set is a collection of curves.
	 *
	 * @category Math
	 */
	var CurveSet = /*#__PURE__*/function () {
	  /**
	   * Creates a new CurveSet instance.
	   *
	   * @param {Array<number[]>} curveKeys - An array of arrays of keys (pairs of numbers with the
	   * time first and value second).
	   * @example
	   * const curveSet = new pc.CurveSet([
	   *     [
	   *         0, 0,        // At 0 time, value of 0
	   *         0.33, 2,     // At 0.33 time, value of 2
	   *         0.66, 2.6,   // At 0.66 time, value of 2.6
	   *         1, 3         // At 1 time, value of 3
	   *     ],
	   *     [
	   *         0, 34,
	   *         0.33, 35,
	   *         0.66, 36,
	   *         1, 37
	   *     ]
	   * ]);
	   */
	  function CurveSet() {
	    this.curves = [];
	    /**
	     * @type {number}
	     * @private
	     */
	    this._type = CURVE_SMOOTHSTEP;
	    if (arguments.length > 1) {
	      for (var i = 0; i < arguments.length; i++) {
	        this.curves.push(new Curve(arguments[i]));
	      }
	    } else {
	      if (arguments.length === 0) {
	        this.curves.push(new Curve());
	      } else {
	        var arg = arguments[0];
	        if (typeof arg === 'number') {
	          for (var _i = 0; _i < arg; _i++) {
	            this.curves.push(new Curve());
	          }
	        } else {
	          for (var _i2 = 0; _i2 < arg.length; _i2++) {
	            this.curves.push(new Curve(arg[_i2]));
	          }
	        }
	      }
	    }
	  }

	  /**
	   * The number of curves in the curve set.
	   *
	   * @type {number}
	   */
	  var _proto = CurveSet.prototype;
	  /**
	   * Return a specific curve in the curve set.
	   *
	   * @param {number} index - The index of the curve to return.
	   * @returns {Curve} The curve at the specified index.
	   */
	  _proto.get = function get(index) {
	    return this.curves[index];
	  }

	  /**
	   * Returns the interpolated value of all curves in the curve set at the specified time.
	   *
	   * @param {number} time - The time at which to calculate the value.
	   * @param {number[]} [result] - The interpolated curve values at the specified time. If this
	   * parameter is not supplied, the function allocates a new array internally to return the
	   * result.
	   * @returns {number[]} The interpolated curve values at the specified time.
	   */;
	  _proto.value = function value(time, result) {
	    if (result === void 0) {
	      result = [];
	    }
	    var length = this.curves.length;
	    result.length = length;
	    for (var i = 0; i < length; i++) {
	      result[i] = this.curves[i].value(time);
	    }
	    return result;
	  }

	  /**
	   * Returns a clone of the specified curve set object.
	   *
	   * @returns {this} A clone of the specified curve set.
	   */;
	  _proto.clone = function clone() {
	    /** @type {this} */
	    var result = new this.constructor();
	    result.curves = [];
	    for (var i = 0; i < this.curves.length; i++) {
	      result.curves.push(this.curves[i].clone());
	    }
	    result._type = this._type;
	    return result;
	  }

	  /**
	   * Sample the curveset at regular intervals over the range [0..1].
	   *
	   * @param {number} precision - The number of samples to return.
	   * @returns {Float32Array} The set of quantized values.
	   * @ignore
	   */;
	  _proto.quantize = function quantize(precision) {
	    precision = Math.max(precision, 2);
	    var numCurves = this.curves.length;
	    var values = new Float32Array(precision * numCurves);
	    var step = 1.0 / (precision - 1);
	    for (var c = 0; c < numCurves; c++) {
	      var ev = new CurveEvaluator(this.curves[c]);
	      for (var i = 0; i < precision; i++) {
	        // quantize graph to table of interpolated values
	        values[i * numCurves + c] = ev.evaluate(step * i);
	      }
	    }
	    return values;
	  }

	  /**
	   * Sample the curveset at regular intervals over the range [0..1] and clamp the result to min
	   * and max.
	   *
	   * @param {number} precision - The number of samples to return.
	   * @param {number} min - The minimum output value.
	   * @param {number} max - The maximum output value.
	   * @returns {Float32Array} The set of quantized values.
	   * @ignore
	   */;
	  _proto.quantizeClamped = function quantizeClamped(precision, min, max) {
	    var result = this.quantize(precision);
	    for (var i = 0; i < result.length; ++i) {
	      result[i] = Math.min(max, Math.max(min, result[i]));
	    }
	    return result;
	  };
	  _createClass(CurveSet, [{
	    key: "length",
	    get: function get() {
	      return this.curves.length;
	    }

	    /**
	     * The interpolation scheme applied to all curves in the curve set. Can be:
	     *
	     * - {@link CURVE_LINEAR}
	     * - {@link CURVE_SMOOTHSTEP}
	     * - {@link CURVE_SPLINE}
	     * - {@link CURVE_STEP}
	     *
	     * Defaults to {@link CURVE_SMOOTHSTEP}.
	     *
	     * @type {number}
	     */
	  }, {
	    key: "type",
	    get: function get() {
	      return this._type;
	    },
	    set: function set(value) {
	      this._type = value;
	      for (var i = 0; i < this.curves.length; i++) {
	        this.curves[i].type = value;
	      }
	    }
	  }]);
	  return CurveSet;
	}();

	var checkRange = 5;
	var oneDiv255 = 1 / 255;
	var floatView = new Float32Array(1);
	var int32View = new Int32Array(floatView.buffer);

	/**
	 * Utility static class providing functionality to pack float values to various storage
	 * representations.
	 *
	 * @category Math
	 */
	var FloatPacking = /*#__PURE__*/function () {
	  function FloatPacking() {}
	  /**
	   * Packs a float to a 16-bit half-float representation used by the GPU.
	   *
	   * @param {number} value - The float value to pack.
	   * @returns {number} The packed value.
	   */
	  FloatPacking.float2Half = function float2Half(value) {
	    // based on https://esdiscuss.org/topic/float16array
	    // This method is faster than the OpenEXR implementation (very often
	    // used, eg. in Ogre), with the additional benefit of rounding, inspired
	    // by James Tursa?s half-precision code.
	    floatView[0] = value;
	    var x = int32View[0];
	    var bits = x >> 16 & 0x8000; // Get the sign
	    var m = x >> 12 & 0x07ff; // Keep one extra bit for rounding
	    var e = x >> 23 & 0xff; // Using int is faster here

	    // If zero, or denormal, or exponent underflows too much for a denormal half, return signed zero.
	    if (e < 103) {
	      return bits;
	    }

	    // If NaN, return NaN. If Inf or exponent overflow, return Inf.
	    if (e > 142) {
	      bits |= 0x7c00;

	      // If exponent was 0xff and one mantissa bit was set, it means NaN,
	      // not Inf, so make sure we set one mantissa bit too.
	      bits |= (e === 255 ? 0 : 1) && x & 0x007fffff;
	      return bits;
	    }

	    // If exponent underflows but not too much, return a denormal
	    if (e < 113) {
	      m |= 0x0800;

	      // Extra rounding may overflow and set mantissa to 0 and exponent to 1, which is OK.
	      bits |= (m >> 114 - e) + (m >> 113 - e & 1);
	      return bits;
	    }
	    bits |= e - 112 << 10 | m >> 1;

	    // Extra rounding. An overflow will set mantissa to 0 and increment the exponent, which is OK.
	    bits += m & 1;
	    return bits;
	  }

	  /**
	   * Packs a float value in [0..1) range to specified number of bytes and stores them in an array
	   * with start offset. Based on: https://aras-p.info/blog/2009/07/30/encoding-floats-to-rgba-the-final/
	   * Note: calls to Math.round are only needed on iOS. Precision is somehow really bad without
	   * it. Looks like an issue with their implementation of Uint8ClampedArray.
	   *
	   * @param {number} value - The float value to pack.
	   * @param {Uint8ClampedArray} array - The array to store the packed value in.
	   * @param {number} offset - The start offset in the array to store the packed value at.
	   * @param {number} numBytes - The number of bytes to pack the value to.
	   *
	   * @ignore
	   */;
	  FloatPacking.float2Bytes = function float2Bytes(value, array, offset, numBytes) {
	    var enc1 = 255.0 * value % 1;
	    array[offset + 0] = Math.round((value % 1 - oneDiv255 * enc1) * 255);
	    if (numBytes > 1) {
	      var enc2 = 65025.0 * value % 1;
	      array[offset + 1] = Math.round((enc1 - oneDiv255 * enc2) * 255);
	      if (numBytes > 2) {
	        var enc3 = 16581375.0 * value % 1;
	        array[offset + 2] = Math.round((enc2 - oneDiv255 * enc3) * 255);
	        if (numBytes > 3) {
	          array[offset + 3] = Math.round(enc3 * 255);
	        }
	      }
	    }
	  }

	  /**
	   * Packs a float into specified number of bytes. Min and max range for the float is specified,
	   * allowing the float to be normalized to 0..1 range.
	   *
	   * @param {number} value - The float value to pack.
	   * @param {Uint8ClampedArray} array - The array to store the packed value in.
	   * @param {number} offset - The start offset in the array to store the packed value at.
	   * @param {number} min - Range minimum.
	   * @param {number} max - Range maximum.
	   * @param {number} numBytes - The number of bytes to pack the value to.
	   *
	   * @ignore
	   */;
	  FloatPacking.float2BytesRange = function float2BytesRange(value, array, offset, min, max, numBytes) {
	    if (value < min || value > max) {
	      if (checkRange) {
	        checkRange--;
	        console.warn('float2BytesRange - value to pack is out of specified range.');
	      }
	    }
	    value = math.clamp((value - min) / (max - min), 0, 1);
	    FloatPacking.float2Bytes(value, array, offset, numBytes);
	  }

	  /**
	   * Packs a float into specified number of bytes, using 1 byte for exponent and the remaining
	   * bytes for the mantissa.
	   *
	   * @param {number} value - The float value to pack.
	   * @param {Uint8ClampedArray} array - The array to store the packed value in.
	   * @param {number} offset - The start offset in the array to store the packed value at.
	   * @param {number} numBytes - The number of bytes to pack the value to.
	   *
	   * @ignore
	   */;
	  FloatPacking.float2MantissaExponent = function float2MantissaExponent(value, array, offset, numBytes) {
	    // exponent is increased by one, so that 2^exponent is larger than the value
	    var exponent = Math.floor(Math.log2(Math.abs(value))) + 1;
	    value /= Math.pow(2, exponent);

	    // value is now in -1..1 range, store it using specified number of bytes less one
	    FloatPacking.float2BytesRange(value, array, offset, -1, 1, numBytes - 1);

	    // last byte for the exponent (positive or negative)
	    array[offset + numBytes - 1] = Math.round(exponent + 127);
	  };
	  return FloatPacking;
	}();

	var _class$9;
	/**
	 * 3-dimensional vector.
	 *
	 * @category Math
	 */
	var Vec3 = /*#__PURE__*/function () {
	  /**
	   * Creates a new Vec3 object.
	   *
	   * @param {number|number[]} [x] - The x value. Defaults to 0. If x is an array of length 3, the
	   * array will be used to populate all components.
	   * @param {number} [y] - The y value. Defaults to 0.
	   * @param {number} [z] - The z value. Defaults to 0.
	   * @example
	   * const v = new pc.Vec3(1, 2, 3);
	   */
	  function Vec3(x, y, z) {
	    if (x === void 0) {
	      x = 0;
	    }
	    if (y === void 0) {
	      y = 0;
	    }
	    if (z === void 0) {
	      z = 0;
	    }
	    /**
	     * The first component of the vector.
	     *
	     * @type {number}
	     */
	    this.x = void 0;
	    /**
	     * The second component of the vector.
	     *
	     * @type {number}
	     */
	    this.y = void 0;
	    /**
	     * The third component of the vector.
	     *
	     * @type {number}
	     */
	    this.z = void 0;
	    if (x.length === 3) {
	      this.x = x[0];
	      this.y = x[1];
	      this.z = x[2];
	    } else {
	      this.x = x;
	      this.y = y;
	      this.z = z;
	    }
	  }

	  /**
	   * Adds a 3-dimensional vector to another in place.
	   *
	   * @param {Vec3} rhs - The vector to add to the specified vector.
	   * @returns {Vec3} Self for chaining.
	   * @example
	   * const a = new pc.Vec3(10, 10, 10);
	   * const b = new pc.Vec3(20, 20, 20);
	   *
	   * a.add(b);
	   *
	   * // Outputs [30, 30, 30]
	   * console.log("The result of the addition is: " + a.toString());
	   */
	  var _proto = Vec3.prototype;
	  _proto.add = function add(rhs) {
	    this.x += rhs.x;
	    this.y += rhs.y;
	    this.z += rhs.z;
	    return this;
	  }

	  /**
	   * Adds two 3-dimensional vectors together and returns the result.
	   *
	   * @param {Vec3} lhs - The first vector operand for the addition.
	   * @param {Vec3} rhs - The second vector operand for the addition.
	   * @returns {Vec3} Self for chaining.
	   * @example
	   * const a = new pc.Vec3(10, 10, 10);
	   * const b = new pc.Vec3(20, 20, 20);
	   * const r = new pc.Vec3();
	   *
	   * r.add2(a, b);
	   * // Outputs [30, 30, 30]
	   *
	   * console.log("The result of the addition is: " + r.toString());
	   */;
	  _proto.add2 = function add2(lhs, rhs) {
	    this.x = lhs.x + rhs.x;
	    this.y = lhs.y + rhs.y;
	    this.z = lhs.z + rhs.z;
	    return this;
	  }

	  /**
	   * Adds a number to each element of a vector.
	   *
	   * @param {number} scalar - The number to add.
	   * @returns {Vec3} Self for chaining.
	   * @example
	   * const vec = new pc.Vec3(3, 4, 5);
	   *
	   * vec.addScalar(2);
	   *
	   * // Outputs [5, 6, 7]
	   * console.log("The result of the addition is: " + vec.toString());
	   */;
	  _proto.addScalar = function addScalar(scalar) {
	    this.x += scalar;
	    this.y += scalar;
	    this.z += scalar;
	    return this;
	  }

	  /**
	   * Adds a 3-dimensional vector scaled by scalar value. Does not modify the vector being added.
	   *
	   * @param {Vec3} rhs - The vector to add to the specified vector.
	   * @param {number} scalar - The number to multiply the added vector with.
	   * @returns {Vec3} Self for chaining.
	   * @example
	   * const vec = new pc.Vec3(1, 2, 3);
	   *
	   * vec.addScaled(pc.Vec3.UP, 2);
	   *
	   * // Outputs [1, 4, 3]
	   * console.log("The result of the addition is: " + vec.toString());
	   */;
	  _proto.addScaled = function addScaled(rhs, scalar) {
	    this.x += rhs.x * scalar;
	    this.y += rhs.y * scalar;
	    this.z += rhs.z * scalar;
	    return this;
	  }

	  /**
	   * Returns an identical copy of the specified 3-dimensional vector.
	   *
	   * @returns {this} A 3-dimensional vector containing the result of the cloning.
	   * @example
	   * const v = new pc.Vec3(10, 20, 30);
	   * const vclone = v.clone();
	   * console.log("The result of the cloning is: " + vclone.toString());
	   */;
	  _proto.clone = function clone() {
	    /** @type {this} */
	    var cstr = this.constructor;
	    return new cstr(this.x, this.y, this.z);
	  }

	  /**
	   * Copies the contents of a source 3-dimensional vector to a destination 3-dimensional vector.
	   *
	   * @param {Vec3} rhs - A vector to copy to the specified vector.
	   * @returns {Vec3} Self for chaining.
	   * @example
	   * const src = new pc.Vec3(10, 20, 30);
	   * const dst = new pc.Vec3();
	   *
	   * dst.copy(src);
	   *
	   * console.log("The two vectors are " + (dst.equals(src) ? "equal" : "different"));
	   */;
	  _proto.copy = function copy(rhs) {
	    this.x = rhs.x;
	    this.y = rhs.y;
	    this.z = rhs.z;
	    return this;
	  }

	  /**
	   * Returns the result of a cross product operation performed on the two specified 3-dimensional
	   * vectors.
	   *
	   * @param {Vec3} lhs - The first 3-dimensional vector operand of the cross product.
	   * @param {Vec3} rhs - The second 3-dimensional vector operand of the cross product.
	   * @returns {Vec3} Self for chaining.
	   * @example
	   * const back = new pc.Vec3().cross(pc.Vec3.RIGHT, pc.Vec3.UP);
	   *
	   * // Prints the Z axis (i.e. [0, 0, 1])
	   * console.log("The result of the cross product is: " + back.toString());
	   */;
	  _proto.cross = function cross(lhs, rhs) {
	    // Create temporary variables in case lhs or rhs are 'this'
	    var lx = lhs.x;
	    var ly = lhs.y;
	    var lz = lhs.z;
	    var rx = rhs.x;
	    var ry = rhs.y;
	    var rz = rhs.z;
	    this.x = ly * rz - ry * lz;
	    this.y = lz * rx - rz * lx;
	    this.z = lx * ry - rx * ly;
	    return this;
	  }

	  /**
	   * Returns the distance between the two specified 3-dimensional vectors.
	   *
	   * @param {Vec3} rhs - The second 3-dimensional vector to test.
	   * @returns {number} The distance between the two vectors.
	   * @example
	   * const v1 = new pc.Vec3(5, 10, 20);
	   * const v2 = new pc.Vec3(10, 20, 40);
	   * const d = v1.distance(v2);
	   * console.log("The distance between v1 and v2 is: " + d);
	   */;
	  _proto.distance = function distance(rhs) {
	    var x = this.x - rhs.x;
	    var y = this.y - rhs.y;
	    var z = this.z - rhs.z;
	    return Math.sqrt(x * x + y * y + z * z);
	  }

	  /**
	   * Divides a 3-dimensional vector by another in place.
	   *
	   * @param {Vec3} rhs - The vector to divide the specified vector by.
	   * @returns {Vec3} Self for chaining.
	   * @example
	   * const a = new pc.Vec3(4, 9, 16);
	   * const b = new pc.Vec3(2, 3, 4);
	   *
	   * a.div(b);
	   *
	   * // Outputs [2, 3, 4]
	   * console.log("The result of the division is: " + a.toString());
	   */;
	  _proto.div = function div(rhs) {
	    this.x /= rhs.x;
	    this.y /= rhs.y;
	    this.z /= rhs.z;
	    return this;
	  }

	  /**
	   * Divides one 3-dimensional vector by another and writes the result to the specified vector.
	   *
	   * @param {Vec3} lhs - The dividend vector (the vector being divided).
	   * @param {Vec3} rhs - The divisor vector (the vector dividing the dividend).
	   * @returns {Vec3} Self for chaining.
	   * @example
	   * const a = new pc.Vec3(4, 9, 16);
	   * const b = new pc.Vec3(2, 3, 4);
	   * const r = new pc.Vec3();
	   *
	   * r.div2(a, b);
	   * // Outputs [2, 3, 4]
	   *
	   * console.log("The result of the division is: " + r.toString());
	   */;
	  _proto.div2 = function div2(lhs, rhs) {
	    this.x = lhs.x / rhs.x;
	    this.y = lhs.y / rhs.y;
	    this.z = lhs.z / rhs.z;
	    return this;
	  }

	  /**
	   * Divides each element of a vector by a number.
	   *
	   * @param {number} scalar - The number to divide by.
	   * @returns {Vec3} Self for chaining.
	   * @example
	   * const vec = new pc.Vec3(3, 6, 9);
	   *
	   * vec.divScalar(3);
	   *
	   * // Outputs [1, 2, 3]
	   * console.log("The result of the division is: " + vec.toString());
	   */;
	  _proto.divScalar = function divScalar(scalar) {
	    this.x /= scalar;
	    this.y /= scalar;
	    this.z /= scalar;
	    return this;
	  }

	  /**
	   * Returns the result of a dot product operation performed on the two specified 3-dimensional
	   * vectors.
	   *
	   * @param {Vec3} rhs - The second 3-dimensional vector operand of the dot product.
	   * @returns {number} The result of the dot product operation.
	   * @example
	   * const v1 = new pc.Vec3(5, 10, 20);
	   * const v2 = new pc.Vec3(10, 20, 40);
	   * const v1dotv2 = v1.dot(v2);
	   * console.log("The result of the dot product is: " + v1dotv2);
	   */;
	  _proto.dot = function dot(rhs) {
	    return this.x * rhs.x + this.y * rhs.y + this.z * rhs.z;
	  }

	  /**
	   * Reports whether two vectors are equal.
	   *
	   * @param {Vec3} rhs - The vector to compare to the specified vector.
	   * @returns {boolean} True if the vectors are equal and false otherwise.
	   * @example
	   * const a = new pc.Vec3(1, 2, 3);
	   * const b = new pc.Vec3(4, 5, 6);
	   * console.log("The two vectors are " + (a.equals(b) ? "equal" : "different"));
	   */;
	  _proto.equals = function equals(rhs) {
	    return this.x === rhs.x && this.y === rhs.y && this.z === rhs.z;
	  }

	  /**
	   * Reports whether two vectors are equal using an absolute error tolerance.
	   *
	   * @param {Vec3} rhs - The vector to be compared against.
	   * @param {number} [epsilon] - The maximum difference between each component of the two
	   * vectors. Defaults to 1e-6.
	   * @returns {boolean} True if the vectors are equal and false otherwise.
	   * @example
	   * const a = new pc.Vec3();
	   * const b = new pc.Vec3();
	   * console.log("The two vectors are approximately " + (a.equalsApprox(b, 1e-9) ? "equal" : "different"));
	   */;
	  _proto.equalsApprox = function equalsApprox(rhs, epsilon) {
	    if (epsilon === void 0) {
	      epsilon = 1e-6;
	    }
	    return Math.abs(this.x - rhs.x) < epsilon && Math.abs(this.y - rhs.y) < epsilon && Math.abs(this.z - rhs.z) < epsilon;
	  }

	  /**
	   * Returns the magnitude of the specified 3-dimensional vector.
	   *
	   * @returns {number} The magnitude of the specified 3-dimensional vector.
	   * @example
	   * const vec = new pc.Vec3(3, 4, 0);
	   * const len = vec.length();
	   * // Outputs 5
	   * console.log("The length of the vector is: " + len);
	   */;
	  _proto.length = function length() {
	    return Math.sqrt(this.x * this.x + this.y * this.y + this.z * this.z);
	  }

	  /**
	   * Returns the magnitude squared of the specified 3-dimensional vector.
	   *
	   * @returns {number} The magnitude of the specified 3-dimensional vector.
	   * @example
	   * const vec = new pc.Vec3(3, 4, 0);
	   * const len = vec.lengthSq();
	   * // Outputs 25
	   * console.log("The length squared of the vector is: " + len);
	   */;
	  _proto.lengthSq = function lengthSq() {
	    return this.x * this.x + this.y * this.y + this.z * this.z;
	  }

	  /**
	   * Returns the result of a linear interpolation between two specified 3-dimensional vectors.
	   *
	   * @param {Vec3} lhs - The 3-dimensional to interpolate from.
	   * @param {Vec3} rhs - The 3-dimensional to interpolate to.
	   * @param {number} alpha - The value controlling the point of interpolation. Between 0 and 1,
	   * the linear interpolant will occur on a straight line between lhs and rhs. Outside of this
	   * range, the linear interpolant will occur on a ray extrapolated from this line.
	   * @returns {Vec3} Self for chaining.
	   * @example
	   * const a = new pc.Vec3(0, 0, 0);
	   * const b = new pc.Vec3(10, 10, 10);
	   * const r = new pc.Vec3();
	   *
	   * r.lerp(a, b, 0);   // r is equal to a
	   * r.lerp(a, b, 0.5); // r is 5, 5, 5
	   * r.lerp(a, b, 1);   // r is equal to b
	   */;
	  _proto.lerp = function lerp(lhs, rhs, alpha) {
	    this.x = lhs.x + alpha * (rhs.x - lhs.x);
	    this.y = lhs.y + alpha * (rhs.y - lhs.y);
	    this.z = lhs.z + alpha * (rhs.z - lhs.z);
	    return this;
	  }

	  /**
	   * Multiplies a 3-dimensional vector to another in place.
	   *
	   * @param {Vec3} rhs - The 3-dimensional vector used as the second multiplicand of the operation.
	   * @returns {Vec3} Self for chaining.
	   * @example
	   * const a = new pc.Vec3(2, 3, 4);
	   * const b = new pc.Vec3(4, 5, 6);
	   *
	   * a.mul(b);
	   *
	   * // Outputs 8, 15, 24
	   * console.log("The result of the multiplication is: " + a.toString());
	   */;
	  _proto.mul = function mul(rhs) {
	    this.x *= rhs.x;
	    this.y *= rhs.y;
	    this.z *= rhs.z;
	    return this;
	  }

	  /**
	   * Returns the result of multiplying the specified 3-dimensional vectors together.
	   *
	   * @param {Vec3} lhs - The 3-dimensional vector used as the first multiplicand of the operation.
	   * @param {Vec3} rhs - The 3-dimensional vector used as the second multiplicand of the operation.
	   * @returns {Vec3} Self for chaining.
	   * @example
	   * const a = new pc.Vec3(2, 3, 4);
	   * const b = new pc.Vec3(4, 5, 6);
	   * const r = new pc.Vec3();
	   *
	   * r.mul2(a, b);
	   *
	   * // Outputs 8, 15, 24
	   * console.log("The result of the multiplication is: " + r.toString());
	   */;
	  _proto.mul2 = function mul2(lhs, rhs) {
	    this.x = lhs.x * rhs.x;
	    this.y = lhs.y * rhs.y;
	    this.z = lhs.z * rhs.z;
	    return this;
	  }

	  /**
	   * Multiplies each element of a vector by a number.
	   *
	   * @param {number} scalar - The number to multiply by.
	   * @returns {Vec3} Self for chaining.
	   * @example
	   * const vec = new pc.Vec3(3, 6, 9);
	   *
	   * vec.mulScalar(3);
	   *
	   * // Outputs [9, 18, 27]
	   * console.log("The result of the multiplication is: " + vec.toString());
	   */;
	  _proto.mulScalar = function mulScalar(scalar) {
	    this.x *= scalar;
	    this.y *= scalar;
	    this.z *= scalar;
	    return this;
	  }

	  /**
	   * Returns this 3-dimensional vector converted to a unit vector in place. If the vector has a
	   * length of zero, the vector's elements will be set to zero.
	   *
	   * @param {Vec3} [src] - The vector to normalize. If not set, the operation is done in place.
	   * @returns {Vec3} Self for chaining.
	   * @example
	   * const v = new pc.Vec3(25, 0, 0);
	   *
	   * v.normalize();
	   *
	   * // Outputs 1, 0, 0
	   * console.log("The result of the vector normalization is: " + v.toString());
	   */;
	  _proto.normalize = function normalize(src) {
	    if (src === void 0) {
	      src = this;
	    }
	    var lengthSq = src.x * src.x + src.y * src.y + src.z * src.z;
	    if (lengthSq > 0) {
	      var invLength = 1 / Math.sqrt(lengthSq);
	      this.x = src.x * invLength;
	      this.y = src.y * invLength;
	      this.z = src.z * invLength;
	    }
	    return this;
	  }

	  /**
	   * Each element is set to the largest integer less than or equal to its value.
	   *
	   * @param {Vec3} [src] - The vector to floor. If not set, the operation is done in place.
	   * @returns {Vec3} Self for chaining.
	   */;
	  _proto.floor = function floor(src) {
	    if (src === void 0) {
	      src = this;
	    }
	    this.x = Math.floor(src.x);
	    this.y = Math.floor(src.y);
	    this.z = Math.floor(src.z);
	    return this;
	  }

	  /**
	   * Each element is rounded up to the next largest integer.
	   *
	   * @param {Vec3} [src] - The vector to ceil. If not set, the operation is done in place.
	   * @returns {Vec3} Self for chaining.
	   */;
	  _proto.ceil = function ceil(src) {
	    if (src === void 0) {
	      src = this;
	    }
	    this.x = Math.ceil(src.x);
	    this.y = Math.ceil(src.y);
	    this.z = Math.ceil(src.z);
	    return this;
	  }

	  /**
	   * Each element is rounded up or down to the nearest integer.
	   *
	   * @param {Vec3} [src] - The vector to round. If not set, the operation is done in place.
	   * @returns {Vec3} Self for chaining.
	   */;
	  _proto.round = function round(src) {
	    if (src === void 0) {
	      src = this;
	    }
	    this.x = Math.round(src.x);
	    this.y = Math.round(src.y);
	    this.z = Math.round(src.z);
	    return this;
	  }

	  /**
	   * Each element is assigned a value from rhs parameter if it is smaller.
	   *
	   * @param {Vec3} rhs - The 3-dimensional vector used as the source of elements to compare to.
	   * @returns {Vec3} Self for chaining.
	   */;
	  _proto.min = function min(rhs) {
	    if (rhs.x < this.x) this.x = rhs.x;
	    if (rhs.y < this.y) this.y = rhs.y;
	    if (rhs.z < this.z) this.z = rhs.z;
	    return this;
	  }

	  /**
	   * Each element is assigned a value from rhs parameter if it is larger.
	   *
	   * @param {Vec3} rhs - The 3-dimensional vector used as the source of elements to compare to.
	   * @returns {Vec3} Self for chaining.
	   */;
	  _proto.max = function max(rhs) {
	    if (rhs.x > this.x) this.x = rhs.x;
	    if (rhs.y > this.y) this.y = rhs.y;
	    if (rhs.z > this.z) this.z = rhs.z;
	    return this;
	  }

	  /**
	   * Projects this 3-dimensional vector onto the specified vector.
	   *
	   * @param {Vec3} rhs - The vector onto which the original vector will be projected on.
	   * @returns {Vec3} Self for chaining.
	   * @example
	   * const v = new pc.Vec3(5, 5, 5);
	   * const normal = new pc.Vec3(1, 0, 0);
	   *
	   * v.project(normal);
	   *
	   * // Outputs 5, 0, 0
	   * console.log("The result of the vector projection is: " + v.toString());
	   */;
	  _proto.project = function project(rhs) {
	    var a_dot_b = this.x * rhs.x + this.y * rhs.y + this.z * rhs.z;
	    var b_dot_b = rhs.x * rhs.x + rhs.y * rhs.y + rhs.z * rhs.z;
	    var s = a_dot_b / b_dot_b;
	    this.x = rhs.x * s;
	    this.y = rhs.y * s;
	    this.z = rhs.z * s;
	    return this;
	  }

	  /**
	   * Sets the specified 3-dimensional vector to the supplied numerical values.
	   *
	   * @param {number} x - The value to set on the first component of the vector.
	   * @param {number} y - The value to set on the second component of the vector.
	   * @param {number} z - The value to set on the third component of the vector.
	   * @returns {Vec3} Self for chaining.
	   * @example
	   * const v = new pc.Vec3();
	   * v.set(5, 10, 20);
	   *
	   * // Outputs 5, 10, 20
	   * console.log("The result of the vector set is: " + v.toString());
	   */;
	  _proto.set = function set(x, y, z) {
	    this.x = x;
	    this.y = y;
	    this.z = z;
	    return this;
	  }

	  /**
	   * Subtracts a 3-dimensional vector from another in place.
	   *
	   * @param {Vec3} rhs - The vector to subtract from the specified vector.
	   * @returns {Vec3} Self for chaining.
	   * @example
	   * const a = new pc.Vec3(10, 10, 10);
	   * const b = new pc.Vec3(20, 20, 20);
	   *
	   * a.sub(b);
	   *
	   * // Outputs [-10, -10, -10]
	   * console.log("The result of the subtraction is: " + a.toString());
	   */;
	  _proto.sub = function sub(rhs) {
	    this.x -= rhs.x;
	    this.y -= rhs.y;
	    this.z -= rhs.z;
	    return this;
	  }

	  /**
	   * Subtracts two 3-dimensional vectors from one another and returns the result.
	   *
	   * @param {Vec3} lhs - The first vector operand for the subtraction.
	   * @param {Vec3} rhs - The second vector operand for the subtraction.
	   * @returns {Vec3} Self for chaining.
	   * @example
	   * const a = new pc.Vec3(10, 10, 10);
	   * const b = new pc.Vec3(20, 20, 20);
	   * const r = new pc.Vec3();
	   *
	   * r.sub2(a, b);
	   *
	   * // Outputs [-10, -10, -10]
	   * console.log("The result of the subtraction is: " + r.toString());
	   */;
	  _proto.sub2 = function sub2(lhs, rhs) {
	    this.x = lhs.x - rhs.x;
	    this.y = lhs.y - rhs.y;
	    this.z = lhs.z - rhs.z;
	    return this;
	  }

	  /**
	   * Subtracts a number from each element of a vector.
	   *
	   * @param {number} scalar - The number to subtract.
	   * @returns {Vec3} Self for chaining.
	   * @example
	   * const vec = new pc.Vec3(3, 4, 5);
	   *
	   * vec.subScalar(2);
	   *
	   * // Outputs [1, 2, 3]
	   * console.log("The result of the subtraction is: " + vec.toString());
	   */;
	  _proto.subScalar = function subScalar(scalar) {
	    this.x -= scalar;
	    this.y -= scalar;
	    this.z -= scalar;
	    return this;
	  }

	  /**
	   * Converts the vector to string form.
	   *
	   * @returns {string} The vector in string form.
	   * @example
	   * const v = new pc.Vec3(20, 10, 5);
	   * // Outputs [20, 10, 5]
	   * console.log(v.toString());
	   */;
	  _proto.toString = function toString() {
	    return "[" + this.x + ", " + this.y + ", " + this.z + "]";
	  }

	  /**
	   * A constant vector set to [0, 0, 0].
	   *
	   * @type {Vec3}
	   * @readonly
	   */;
	  return Vec3;
	}();
	_class$9 = Vec3;
	Vec3.ZERO = Object.freeze(new _class$9(0, 0, 0));
	/**
	 * A constant vector set to [1, 1, 1].
	 *
	 * @type {Vec3}
	 * @readonly
	 */
	Vec3.ONE = Object.freeze(new _class$9(1, 1, 1));
	/**
	 * A constant vector set to [0, 1, 0].
	 *
	 * @type {Vec3}
	 * @readonly
	 */
	Vec3.UP = Object.freeze(new _class$9(0, 1, 0));
	/**
	 * A constant vector set to [0, -1, 0].
	 *
	 * @type {Vec3}
	 * @readonly
	 */
	Vec3.DOWN = Object.freeze(new _class$9(0, -1, 0));
	/**
	 * A constant vector set to [1, 0, 0].
	 *
	 * @type {Vec3}
	 * @readonly
	 */
	Vec3.RIGHT = Object.freeze(new _class$9(1, 0, 0));
	/**
	 * A constant vector set to [-1, 0, 0].
	 *
	 * @type {Vec3}
	 * @readonly
	 */
	Vec3.LEFT = Object.freeze(new _class$9(-1, 0, 0));
	/**
	 * A constant vector set to [0, 0, -1].
	 *
	 * @type {Vec3}
	 * @readonly
	 */
	Vec3.FORWARD = Object.freeze(new _class$9(0, 0, -1));
	/**
	 * A constant vector set to [0, 0, 1].
	 *
	 * @type {Vec3}
	 * @readonly
	 */
	Vec3.BACK = Object.freeze(new _class$9(0, 0, 1));

	var _class$8;

	/**
	 * A 3x3 matrix.
	 *
	 * @category Math
	 */
	var Mat3 = /*#__PURE__*/function () {
	  /**
	   * Create a new Mat3 instance. It is initialized to the identity matrix.
	   */
	  function Mat3() {
	    /**
	     * Matrix elements in the form of a flat array.
	     *
	     * @type {Float32Array}
	     */
	    this.data = new Float32Array(9);
	    // Create an identity matrix. Note that a new Float32Array has all elements set
	    // to zero by default, so we only need to set the relevant elements to one.
	    this.data[0] = this.data[4] = this.data[8] = 1;
	  }

	  /**
	   * Creates a duplicate of the specified matrix.
	   *
	   * @returns {this} A duplicate matrix.
	   * @example
	   * const src = new pc.Mat3().translate(10, 20, 30);
	   * const dst = src.clone();
	   * console.log("The two matrices are " + (src.equals(dst) ? "equal" : "different"));
	   */
	  var _proto = Mat3.prototype;
	  _proto.clone = function clone() {
	    /** @type {this} */
	    var cstr = this.constructor;
	    return new cstr().copy(this);
	  }

	  /**
	   * Copies the contents of a source 3x3 matrix to a destination 3x3 matrix.
	   *
	   * @param {Mat3} rhs - A 3x3 matrix to be copied.
	   * @returns {Mat3} Self for chaining.
	   * @example
	   * const src = new pc.Mat3().translate(10, 20, 30);
	   * const dst = new pc.Mat3();
	   * dst.copy(src);
	   * console.log("The two matrices are " + (src.equals(dst) ? "equal" : "different"));
	   */;
	  _proto.copy = function copy(rhs) {
	    var src = rhs.data;
	    var dst = this.data;
	    dst[0] = src[0];
	    dst[1] = src[1];
	    dst[2] = src[2];
	    dst[3] = src[3];
	    dst[4] = src[4];
	    dst[5] = src[5];
	    dst[6] = src[6];
	    dst[7] = src[7];
	    dst[8] = src[8];
	    return this;
	  }

	  /**
	   * Copies the contents of a source array[9] to a destination 3x3 matrix.
	   *
	   * @param {number[]} src - An array[9] to be copied.
	   * @returns {Mat3} Self for chaining.
	   * @example
	   * const dst = new pc.Mat3();
	   * dst.set([0, 1, 2, 3, 4, 5, 6, 7, 8]);
	   */;
	  _proto.set = function set(src) {
	    var dst = this.data;
	    dst[0] = src[0];
	    dst[1] = src[1];
	    dst[2] = src[2];
	    dst[3] = src[3];
	    dst[4] = src[4];
	    dst[5] = src[5];
	    dst[6] = src[6];
	    dst[7] = src[7];
	    dst[8] = src[8];
	    return this;
	  }

	  /**
	   * Extracts the x-axis from the specified matrix.
	   *
	   * @param {Vec3} [x] - The vector to receive the x axis of the matrix.
	   * @returns {Vec3} The x-axis of the specified matrix.
	   */;
	  _proto.getX = function getX(x) {
	    if (x === void 0) {
	      x = new Vec3();
	    }
	    return x.set(this.data[0], this.data[1], this.data[2]);
	  }

	  /**
	   * Extracts the y-axis from the specified matrix.
	   *
	   * @param {Vec3} [y] - The vector to receive the y axis of the matrix.
	   * @returns {Vec3} The y-axis of the specified matrix.
	   */;
	  _proto.getY = function getY(y) {
	    if (y === void 0) {
	      y = new Vec3();
	    }
	    return y.set(this.data[3], this.data[4], this.data[5]);
	  }

	  /**
	   * Extracts the z-axis from the specified matrix.
	   *
	   * @param {Vec3} [z] - The vector to receive the z axis of the matrix.
	   * @returns {Vec3} The z-axis of the specified matrix.
	   */;
	  _proto.getZ = function getZ(z) {
	    if (z === void 0) {
	      z = new Vec3();
	    }
	    return z.set(this.data[6], this.data[7], this.data[8]);
	  }

	  /**
	   * Reports whether two matrices are equal.
	   *
	   * @param {Mat3} rhs - The other matrix.
	   * @returns {boolean} True if the matrices are equal and false otherwise.
	   * @example
	   * const a = new pc.Mat3().translate(10, 20, 30);
	   * const b = new pc.Mat3();
	   * console.log("The two matrices are " + (a.equals(b) ? "equal" : "different"));
	   */;
	  _proto.equals = function equals(rhs) {
	    var l = this.data;
	    var r = rhs.data;
	    return l[0] === r[0] && l[1] === r[1] && l[2] === r[2] && l[3] === r[3] && l[4] === r[4] && l[5] === r[5] && l[6] === r[6] && l[7] === r[7] && l[8] === r[8];
	  }

	  /**
	   * Reports whether the specified matrix is the identity matrix.
	   *
	   * @returns {boolean} True if the matrix is identity and false otherwise.
	   * @example
	   * const m = new pc.Mat3();
	   * console.log("The matrix is " + (m.isIdentity() ? "identity" : "not identity"));
	   */;
	  _proto.isIdentity = function isIdentity() {
	    var m = this.data;
	    return m[0] === 1 && m[1] === 0 && m[2] === 0 && m[3] === 0 && m[4] === 1 && m[5] === 0 && m[6] === 0 && m[7] === 0 && m[8] === 1;
	  }

	  /**
	   * Sets the matrix to the identity matrix.
	   *
	   * @returns {Mat3} Self for chaining.
	   * @example
	   * m.setIdentity();
	   * console.log("The matrix is " + (m.isIdentity() ? "identity" : "not identity"));
	   */;
	  _proto.setIdentity = function setIdentity() {
	    var m = this.data;
	    m[0] = 1;
	    m[1] = 0;
	    m[2] = 0;
	    m[3] = 0;
	    m[4] = 1;
	    m[5] = 0;
	    m[6] = 0;
	    m[7] = 0;
	    m[8] = 1;
	    return this;
	  }

	  /**
	   * Converts the matrix to string form.
	   *
	   * @returns {string} The matrix in string form.
	   * @example
	   * const m = new pc.Mat3();
	   * // Outputs [1, 0, 0, 0, 1, 0, 0, 0, 1]
	   * console.log(m.toString());
	   */;
	  _proto.toString = function toString() {
	    return '[' + this.data.join(', ') + ']';
	  }

	  /**
	   * Generates the transpose of the specified 3x3 matrix.
	   *
	   * @param {Mat3} [src] - The matrix to transpose. If not set, the matrix is transposed in-place.
	   * @returns {Mat3} Self for chaining.
	   * @example
	   * const m = new pc.Mat3();
	   *
	   * // Transpose in place
	   * m.transpose();
	   */;
	  _proto.transpose = function transpose(src) {
	    if (src === void 0) {
	      src = this;
	    }
	    var s = src.data;
	    var t = this.data;
	    if (s === t) {
	      var tmp;
	      tmp = s[1];
	      t[1] = s[3];
	      t[3] = tmp;
	      tmp = s[2];
	      t[2] = s[6];
	      t[6] = tmp;
	      tmp = s[5];
	      t[5] = s[7];
	      t[7] = tmp;
	    } else {
	      t[0] = s[0];
	      t[1] = s[3];
	      t[2] = s[6];
	      t[3] = s[1];
	      t[4] = s[4];
	      t[5] = s[7];
	      t[6] = s[2];
	      t[7] = s[5];
	      t[8] = s[8];
	    }
	    return this;
	  }

	  /**
	   * Converts the specified 4x4 matrix to a Mat3.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./mat4.js').Mat4} m - The 4x4 matrix to convert.
	   * @returns {Mat3} Self for chaining.
	   */;
	  _proto.setFromMat4 = function setFromMat4(m) {
	    var src = m.data;
	    var dst = this.data;
	    dst[0] = src[0];
	    dst[1] = src[1];
	    dst[2] = src[2];
	    dst[3] = src[4];
	    dst[4] = src[5];
	    dst[5] = src[6];
	    dst[6] = src[8];
	    dst[7] = src[9];
	    dst[8] = src[10];
	    return this;
	  }

	  /**
	   * Set the matrix to the inverse of the specified 4x4 matrix.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./mat4.js').Mat4} src - The 4x4 matrix to invert.
	   * @returns {Mat3} Self for chaining.
	   *
	   * @ignore
	   */;
	  _proto.invertMat4 = function invertMat4(src) {
	    var s = src.data;
	    var a0 = s[0];
	    var a1 = s[1];
	    var a2 = s[2];
	    var a4 = s[4];
	    var a5 = s[5];
	    var a6 = s[6];
	    var a8 = s[8];
	    var a9 = s[9];
	    var a10 = s[10];
	    var b11 = a10 * a5 - a6 * a9;
	    var b21 = -a10 * a1 + a2 * a9;
	    var b31 = a6 * a1 - a2 * a5;
	    var b12 = -a10 * a4 + a6 * a8;
	    var b22 = a10 * a0 - a2 * a8;
	    var b32 = -a6 * a0 + a2 * a4;
	    var b13 = a9 * a4 - a5 * a8;
	    var b23 = -a9 * a0 + a1 * a8;
	    var b33 = a5 * a0 - a1 * a4;
	    var det = a0 * b11 + a1 * b12 + a2 * b13;
	    if (det === 0) {
	      this.setIdentity();
	    } else {
	      var invDet = 1 / det;
	      var t = this.data;
	      t[0] = b11 * invDet;
	      t[1] = b21 * invDet;
	      t[2] = b31 * invDet;
	      t[3] = b12 * invDet;
	      t[4] = b22 * invDet;
	      t[5] = b32 * invDet;
	      t[6] = b13 * invDet;
	      t[7] = b23 * invDet;
	      t[8] = b33 * invDet;
	    }
	    return this;
	  }

	  /**
	   * Transforms a 3-dimensional vector by a 3x3 matrix.
	   *
	   * @param {Vec3} vec - The 3-dimensional vector to be transformed.
	   * @param {Vec3} [res] - An optional 3-dimensional vector to receive the result of the
	   * transformation.
	   * @returns {Vec3} The input vector v transformed by the current instance.
	   */;
	  _proto.transformVector = function transformVector(vec, res) {
	    if (res === void 0) {
	      res = new Vec3();
	    }
	    var m = this.data;
	    var x = vec.x;
	    var y = vec.y;
	    var z = vec.z;
	    res.x = x * m[0] + y * m[3] + z * m[6];
	    res.y = x * m[1] + y * m[4] + z * m[7];
	    res.z = x * m[2] + y * m[5] + z * m[8];
	    return res;
	  }

	  /**
	   * A constant matrix set to the identity.
	   *
	   * @type {Mat3}
	   * @readonly
	   */;
	  return Mat3;
	}();
	_class$8 = Mat3;
	Mat3.IDENTITY = Object.freeze(new _class$8());
	/**
	 * A constant matrix with all elements set to 0.
	 *
	 * @type {Mat3}
	 * @readonly
	 */
	Mat3.ZERO = Object.freeze(new _class$8().set([0, 0, 0, 0, 0, 0, 0, 0, 0]));

	var _class$7;

	/**
	 * A 2-dimensional vector.
	 *
	 * @category Math
	 */
	var Vec2 = /*#__PURE__*/function () {
	  /**
	   * Create a new Vec2 instance.
	   *
	   * @param {number|number[]} [x] - The x value. Defaults to 0. If x is an array of length 2, the
	   * array will be used to populate all components.
	   * @param {number} [y] - The y value. Defaults to 0.
	   * @example
	   * const v = new pc.Vec2(1, 2);
	   */
	  function Vec2(x, y) {
	    if (x === void 0) {
	      x = 0;
	    }
	    if (y === void 0) {
	      y = 0;
	    }
	    /**
	     * The first component of the vector.
	     *
	     * @type {number}
	     */
	    this.x = void 0;
	    /**
	     * The second component of the vector.
	     *
	     * @type {number}
	     */
	    this.y = void 0;
	    if (x.length === 2) {
	      this.x = x[0];
	      this.y = x[1];
	    } else {
	      this.x = x;
	      this.y = y;
	    }
	  }

	  /**
	   * Adds a 2-dimensional vector to another in place.
	   *
	   * @param {Vec2} rhs - The vector to add to the specified vector.
	   * @returns {Vec2} Self for chaining.
	   * @example
	   * const a = new pc.Vec2(10, 10);
	   * const b = new pc.Vec2(20, 20);
	   *
	   * a.add(b);
	   *
	   * // Outputs [30, 30]
	   * console.log("The result of the addition is: " + a.toString());
	   */
	  var _proto = Vec2.prototype;
	  _proto.add = function add(rhs) {
	    this.x += rhs.x;
	    this.y += rhs.y;
	    return this;
	  }

	  /**
	   * Adds two 2-dimensional vectors together and returns the result.
	   *
	   * @param {Vec2} lhs - The first vector operand for the addition.
	   * @param {Vec2} rhs - The second vector operand for the addition.
	   * @returns {Vec2} Self for chaining.
	   * @example
	   * const a = new pc.Vec2(10, 10);
	   * const b = new pc.Vec2(20, 20);
	   * const r = new pc.Vec2();
	   *
	   * r.add2(a, b);
	   * // Outputs [30, 30]
	   *
	   * console.log("The result of the addition is: " + r.toString());
	   */;
	  _proto.add2 = function add2(lhs, rhs) {
	    this.x = lhs.x + rhs.x;
	    this.y = lhs.y + rhs.y;
	    return this;
	  }

	  /**
	   * Adds a number to each element of a vector.
	   *
	   * @param {number} scalar - The number to add.
	   * @returns {Vec2} Self for chaining.
	   * @example
	   * const vec = new pc.Vec2(3, 4);
	   *
	   * vec.addScalar(2);
	   *
	   * // Outputs [5, 6]
	   * console.log("The result of the addition is: " + vec.toString());
	   */;
	  _proto.addScalar = function addScalar(scalar) {
	    this.x += scalar;
	    this.y += scalar;
	    return this;
	  }

	  /**
	   * Adds a 2-dimensional vector scaled by scalar value. Does not modify the vector being added.
	   *
	   * @param {Vec2} rhs - The vector to add to the specified vector.
	   * @param {number} scalar - The number to multiply the added vector with.
	   * @returns {Vec2} Self for chaining.
	   * @example
	   * const vec = new pc.Vec2(1, 2);
	   *
	   * vec.addScaled(pc.Vec2.UP, 2);
	   *
	   * // Outputs [1, 4]
	   * console.log("The result of the addition is: " + vec.toString());
	   */;
	  _proto.addScaled = function addScaled(rhs, scalar) {
	    this.x += rhs.x * scalar;
	    this.y += rhs.y * scalar;
	    return this;
	  }

	  /**
	   * Returns an identical copy of the specified 2-dimensional vector.
	   *
	   * @returns {this} A 2-dimensional vector containing the result of the cloning.
	   * @example
	   * const v = new pc.Vec2(10, 20);
	   * const vclone = v.clone();
	   * console.log("The result of the cloning is: " + vclone.toString());
	   */;
	  _proto.clone = function clone() {
	    /** @type {this} */
	    var cstr = this.constructor;
	    return new cstr(this.x, this.y);
	  }

	  /**
	   * Copies the contents of a source 2-dimensional vector to a destination 2-dimensional vector.
	   *
	   * @param {Vec2} rhs - A vector to copy to the specified vector.
	   * @returns {Vec2} Self for chaining.
	   * @example
	   * const src = new pc.Vec2(10, 20);
	   * const dst = new pc.Vec2();
	   *
	   * dst.copy(src);
	   *
	   * console.log("The two vectors are " + (dst.equals(src) ? "equal" : "different"));
	   */;
	  _proto.copy = function copy(rhs) {
	    this.x = rhs.x;
	    this.y = rhs.y;
	    return this;
	  }

	  /**
	   * Returns the result of a cross product operation performed on the two specified 2-dimensional
	   * vectors.
	   *
	   * @param {Vec2} rhs - The second 2-dimensional vector operand of the cross product.
	   * @returns {number} The cross product of the two vectors.
	   * @example
	   * const right = new pc.Vec2(1, 0);
	   * const up = new pc.Vec2(0, 1);
	   * const crossProduct = right.cross(up);
	   *
	   * // Prints 1
	   * console.log("The result of the cross product is: " + crossProduct);
	   */;
	  _proto.cross = function cross(rhs) {
	    return this.x * rhs.y - this.y * rhs.x;
	  }

	  /**
	   * Returns the distance between the two specified 2-dimensional vectors.
	   *
	   * @param {Vec2} rhs - The second 2-dimensional vector to test.
	   * @returns {number} The distance between the two vectors.
	   * @example
	   * const v1 = new pc.Vec2(5, 10);
	   * const v2 = new pc.Vec2(10, 20);
	   * const d = v1.distance(v2);
	   * console.log("The distance between v1 and v2 is: " + d);
	   */;
	  _proto.distance = function distance(rhs) {
	    var x = this.x - rhs.x;
	    var y = this.y - rhs.y;
	    return Math.sqrt(x * x + y * y);
	  }

	  /**
	   * Divides a 2-dimensional vector by another in place.
	   *
	   * @param {Vec2} rhs - The vector to divide the specified vector by.
	   * @returns {Vec2} Self for chaining.
	   * @example
	   * const a = new pc.Vec2(4, 9);
	   * const b = new pc.Vec2(2, 3);
	   *
	   * a.div(b);
	   *
	   * // Outputs [2, 3]
	   * console.log("The result of the division is: " + a.toString());
	   */;
	  _proto.div = function div(rhs) {
	    this.x /= rhs.x;
	    this.y /= rhs.y;
	    return this;
	  }

	  /**
	   * Divides one 2-dimensional vector by another and writes the result to the specified vector.
	   *
	   * @param {Vec2} lhs - The dividend vector (the vector being divided).
	   * @param {Vec2} rhs - The divisor vector (the vector dividing the dividend).
	   * @returns {Vec2} Self for chaining.
	   * @example
	   * const a = new pc.Vec2(4, 9);
	   * const b = new pc.Vec2(2, 3);
	   * const r = new pc.Vec2();
	   *
	   * r.div2(a, b);
	   * // Outputs [2, 3]
	   *
	   * console.log("The result of the division is: " + r.toString());
	   */;
	  _proto.div2 = function div2(lhs, rhs) {
	    this.x = lhs.x / rhs.x;
	    this.y = lhs.y / rhs.y;
	    return this;
	  }

	  /**
	   * Divides each element of a vector by a number.
	   *
	   * @param {number} scalar - The number to divide by.
	   * @returns {Vec2} Self for chaining.
	   * @example
	   * const vec = new pc.Vec2(3, 6);
	   *
	   * vec.divScalar(3);
	   *
	   * // Outputs [1, 2]
	   * console.log("The result of the division is: " + vec.toString());
	   */;
	  _proto.divScalar = function divScalar(scalar) {
	    this.x /= scalar;
	    this.y /= scalar;
	    return this;
	  }

	  /**
	   * Returns the result of a dot product operation performed on the two specified 2-dimensional
	   * vectors.
	   *
	   * @param {Vec2} rhs - The second 2-dimensional vector operand of the dot product.
	   * @returns {number} The result of the dot product operation.
	   * @example
	   * const v1 = new pc.Vec2(5, 10);
	   * const v2 = new pc.Vec2(10, 20);
	   * const v1dotv2 = v1.dot(v2);
	   * console.log("The result of the dot product is: " + v1dotv2);
	   */;
	  _proto.dot = function dot(rhs) {
	    return this.x * rhs.x + this.y * rhs.y;
	  }

	  /**
	   * Reports whether two vectors are equal.
	   *
	   * @param {Vec2} rhs - The vector to compare to the specified vector.
	   * @returns {boolean} True if the vectors are equal and false otherwise.
	   * @example
	   * const a = new pc.Vec2(1, 2);
	   * const b = new pc.Vec2(4, 5);
	   * console.log("The two vectors are " + (a.equals(b) ? "equal" : "different"));
	   */;
	  _proto.equals = function equals(rhs) {
	    return this.x === rhs.x && this.y === rhs.y;
	  }

	  /**
	   * Reports whether two vectors are equal using an absolute error tolerance.
	   *
	   * @param {Vec2} rhs - The vector to be compared against.
	   * @param {number} [epsilon] - The maximum difference between each component of the two
	   * vectors. Defaults to 1e-6.
	   * @returns {boolean} True if the vectors are equal and false otherwise.
	   * @example
	   * const a = new pc.Vec2();
	   * const b = new pc.Vec2();
	   * console.log("The two vectors are approximately " + (a.equalsApprox(b, 1e-9) ? "equal" : "different"));
	   */;
	  _proto.equalsApprox = function equalsApprox(rhs, epsilon) {
	    if (epsilon === void 0) {
	      epsilon = 1e-6;
	    }
	    return Math.abs(this.x - rhs.x) < epsilon && Math.abs(this.y - rhs.y) < epsilon;
	  }

	  /**
	   * Returns the magnitude of the specified 2-dimensional vector.
	   *
	   * @returns {number} The magnitude of the specified 2-dimensional vector.
	   * @example
	   * const vec = new pc.Vec2(3, 4);
	   * const len = vec.length();
	   * // Outputs 5
	   * console.log("The length of the vector is: " + len);
	   */;
	  _proto.length = function length() {
	    return Math.sqrt(this.x * this.x + this.y * this.y);
	  }

	  /**
	   * Returns the magnitude squared of the specified 2-dimensional vector.
	   *
	   * @returns {number} The magnitude of the specified 2-dimensional vector.
	   * @example
	   * const vec = new pc.Vec2(3, 4);
	   * const len = vec.lengthSq();
	   * // Outputs 25
	   * console.log("The length squared of the vector is: " + len);
	   */;
	  _proto.lengthSq = function lengthSq() {
	    return this.x * this.x + this.y * this.y;
	  }

	  /**
	   * Returns the result of a linear interpolation between two specified 2-dimensional vectors.
	   *
	   * @param {Vec2} lhs - The 2-dimensional to interpolate from.
	   * @param {Vec2} rhs - The 2-dimensional to interpolate to.
	   * @param {number} alpha - The value controlling the point of interpolation. Between 0 and 1,
	   * the linear interpolant will occur on a straight line between lhs and rhs. Outside of this
	   * range, the linear interpolant will occur on a ray extrapolated from this line.
	   * @returns {Vec2} Self for chaining.
	   * @example
	   * const a = new pc.Vec2(0, 0);
	   * const b = new pc.Vec2(10, 10);
	   * const r = new pc.Vec2();
	   *
	   * r.lerp(a, b, 0);   // r is equal to a
	   * r.lerp(a, b, 0.5); // r is 5, 5
	   * r.lerp(a, b, 1);   // r is equal to b
	   */;
	  _proto.lerp = function lerp(lhs, rhs, alpha) {
	    this.x = lhs.x + alpha * (rhs.x - lhs.x);
	    this.y = lhs.y + alpha * (rhs.y - lhs.y);
	    return this;
	  }

	  /**
	   * Multiplies a 2-dimensional vector to another in place.
	   *
	   * @param {Vec2} rhs - The 2-dimensional vector used as the second multiplicand of the operation.
	   * @returns {Vec2} Self for chaining.
	   * @example
	   * const a = new pc.Vec2(2, 3);
	   * const b = new pc.Vec2(4, 5);
	   *
	   * a.mul(b);
	   *
	   * // Outputs 8, 15
	   * console.log("The result of the multiplication is: " + a.toString());
	   */;
	  _proto.mul = function mul(rhs) {
	    this.x *= rhs.x;
	    this.y *= rhs.y;
	    return this;
	  }

	  /**
	   * Returns the result of multiplying the specified 2-dimensional vectors together.
	   *
	   * @param {Vec2} lhs - The 2-dimensional vector used as the first multiplicand of the operation.
	   * @param {Vec2} rhs - The 2-dimensional vector used as the second multiplicand of the operation.
	   * @returns {Vec2} Self for chaining.
	   * @example
	   * const a = new pc.Vec2(2, 3);
	   * const b = new pc.Vec2(4, 5);
	   * const r = new pc.Vec2();
	   *
	   * r.mul2(a, b);
	   *
	   * // Outputs 8, 15
	   * console.log("The result of the multiplication is: " + r.toString());
	   */;
	  _proto.mul2 = function mul2(lhs, rhs) {
	    this.x = lhs.x * rhs.x;
	    this.y = lhs.y * rhs.y;
	    return this;
	  }

	  /**
	   * Multiplies each element of a vector by a number.
	   *
	   * @param {number} scalar - The number to multiply by.
	   * @returns {Vec2} Self for chaining.
	   * @example
	   * const vec = new pc.Vec2(3, 6);
	   *
	   * vec.mulScalar(3);
	   *
	   * // Outputs [9, 18]
	   * console.log("The result of the multiplication is: " + vec.toString());
	   */;
	  _proto.mulScalar = function mulScalar(scalar) {
	    this.x *= scalar;
	    this.y *= scalar;
	    return this;
	  }

	  /**
	   * Returns this 2-dimensional vector converted to a unit vector in place. If the vector has a
	   * length of zero, the vector's elements will be set to zero.
	   *
	   * @param {Vec2} [src] - The vector to normalize. If not set, the operation is done in place.
	   * @returns {Vec2} Self for chaining.
	   * @example
	   * const v = new pc.Vec2(25, 0);
	   *
	   * v.normalize();
	   *
	   * // Outputs 1, 0
	   * console.log("The result of the vector normalization is: " + v.toString());
	   */;
	  _proto.normalize = function normalize(src) {
	    if (src === void 0) {
	      src = this;
	    }
	    var lengthSq = src.x * src.x + src.y * src.y;
	    if (lengthSq > 0) {
	      var invLength = 1 / Math.sqrt(lengthSq);
	      this.x = src.x * invLength;
	      this.y = src.y * invLength;
	    }
	    return this;
	  }

	  /**
	   * Rotate a vector by an angle in degrees.
	   *
	   * @param {number} degrees - The number to degrees to rotate the vector by.
	   * @returns {Vec2} Self for chaining.
	   * @example
	   * const v = new pc.Vec2(0, 10);
	   *
	   * v.rotate(45); // rotates by 45 degrees
	   *
	   * // Outputs [7.071068.., 7.071068..]
	   * console.log("Vector after rotation is: " + v.toString());
	   */;
	  _proto.rotate = function rotate(degrees) {
	    var angle = Math.atan2(this.x, this.y) + degrees * math.DEG_TO_RAD;
	    var len = Math.sqrt(this.x * this.x + this.y * this.y);
	    this.x = Math.sin(angle) * len;
	    this.y = Math.cos(angle) * len;
	    return this;
	  }

	  /**
	   * Returns the angle in degrees of the specified 2-dimensional vector.
	   *
	   * @returns {number} The angle in degrees of the specified 2-dimensional vector.
	   * @example
	   * const v = new pc.Vec2(6, 0);
	   * const angle = v.angle();
	   * // Outputs 90..
	   * console.log("The angle of the vector is: " + angle);
	   */;
	  _proto.angle = function angle() {
	    return Math.atan2(this.x, this.y) * math.RAD_TO_DEG;
	  }

	  /**
	   * Returns the shortest Euler angle between two 2-dimensional vectors.
	   *
	   * @param {Vec2} rhs - The 2-dimensional vector to calculate angle to.
	   * @returns {number} The shortest angle in degrees between two 2-dimensional vectors.
	   * @example
	   * const a = new pc.Vec2(0, 10); // up
	   * const b = new pc.Vec2(1, -1); // down-right
	   * const angle = a.angleTo(b);
	   * // Outputs 135..
	   * console.log("The angle between vectors a and b: " + angle);
	   */;
	  _proto.angleTo = function angleTo(rhs) {
	    return Math.atan2(this.x * rhs.y + this.y * rhs.x, this.x * rhs.x + this.y * rhs.y) * math.RAD_TO_DEG;
	  }

	  /**
	   * Each element is set to the largest integer less than or equal to its value.
	   *
	   * @param {Vec2} [src] - The vector to floor. If not set, the operation is done in place.
	   * @returns {Vec2} Self for chaining.
	   */;
	  _proto.floor = function floor(src) {
	    if (src === void 0) {
	      src = this;
	    }
	    this.x = Math.floor(src.x);
	    this.y = Math.floor(src.y);
	    return this;
	  }

	  /**
	   * Each element is rounded up to the next largest integer.
	   *
	   * @param {Vec2} [src] - The vector to ceil. If not set, the operation is done in place.
	   * @returns {Vec2} Self for chaining.
	   */;
	  _proto.ceil = function ceil(src) {
	    if (src === void 0) {
	      src = this;
	    }
	    this.x = Math.ceil(src.x);
	    this.y = Math.ceil(src.y);
	    return this;
	  }

	  /**
	   * Each element is rounded up or down to the nearest integer.
	   *
	   * @param {Vec2} [src] - The vector to round. If not set, the operation is done in place.
	   * @returns {Vec2} Self for chaining.
	   */;
	  _proto.round = function round(src) {
	    if (src === void 0) {
	      src = this;
	    }
	    this.x = Math.round(src.x);
	    this.y = Math.round(src.y);
	    return this;
	  }

	  /**
	   * Each element is assigned a value from rhs parameter if it is smaller.
	   *
	   * @param {Vec2} rhs - The 2-dimensional vector used as the source of elements to compare to.
	   * @returns {Vec2} Self for chaining.
	   */;
	  _proto.min = function min(rhs) {
	    if (rhs.x < this.x) this.x = rhs.x;
	    if (rhs.y < this.y) this.y = rhs.y;
	    return this;
	  }

	  /**
	   * Each element is assigned a value from rhs parameter if it is larger.
	   *
	   * @param {Vec2} rhs - The 2-dimensional vector used as the source of elements to compare to.
	   * @returns {Vec2} Self for chaining.
	   */;
	  _proto.max = function max(rhs) {
	    if (rhs.x > this.x) this.x = rhs.x;
	    if (rhs.y > this.y) this.y = rhs.y;
	    return this;
	  }

	  /**
	   * Sets the specified 2-dimensional vector to the supplied numerical values.
	   *
	   * @param {number} x - The value to set on the first component of the vector.
	   * @param {number} y - The value to set on the second component of the vector.
	   * @returns {Vec2} Self for chaining.
	   * @example
	   * const v = new pc.Vec2();
	   * v.set(5, 10);
	   *
	   * // Outputs 5, 10
	   * console.log("The result of the vector set is: " + v.toString());
	   */;
	  _proto.set = function set(x, y) {
	    this.x = x;
	    this.y = y;
	    return this;
	  }

	  /**
	   * Subtracts a 2-dimensional vector from another in place.
	   *
	   * @param {Vec2} rhs - The vector to subtract from the specified vector.
	   * @returns {Vec2} Self for chaining.
	   * @example
	   * const a = new pc.Vec2(10, 10);
	   * const b = new pc.Vec2(20, 20);
	   *
	   * a.sub(b);
	   *
	   * // Outputs [-10, -10]
	   * console.log("The result of the subtraction is: " + a.toString());
	   */;
	  _proto.sub = function sub(rhs) {
	    this.x -= rhs.x;
	    this.y -= rhs.y;
	    return this;
	  }

	  /**
	   * Subtracts two 2-dimensional vectors from one another and returns the result.
	   *
	   * @param {Vec2} lhs - The first vector operand for the subtraction.
	   * @param {Vec2} rhs - The second vector operand for the subtraction.
	   * @returns {Vec2} Self for chaining.
	   * @example
	   * const a = new pc.Vec2(10, 10);
	   * const b = new pc.Vec2(20, 20);
	   * const r = new pc.Vec2();
	   *
	   * r.sub2(a, b);
	   *
	   * // Outputs [-10, -10]
	   * console.log("The result of the subtraction is: " + r.toString());
	   */;
	  _proto.sub2 = function sub2(lhs, rhs) {
	    this.x = lhs.x - rhs.x;
	    this.y = lhs.y - rhs.y;
	    return this;
	  }

	  /**
	   * Subtracts a number from each element of a vector.
	   *
	   * @param {number} scalar - The number to subtract.
	   * @returns {Vec2} Self for chaining.
	   * @example
	   * const vec = new pc.Vec2(3, 4);
	   *
	   * vec.subScalar(2);
	   *
	   * // Outputs [1, 2]
	   * console.log("The result of the subtraction is: " + vec.toString());
	   */;
	  _proto.subScalar = function subScalar(scalar) {
	    this.x -= scalar;
	    this.y -= scalar;
	    return this;
	  }

	  /**
	   * Converts the vector to string form.
	   *
	   * @returns {string} The vector in string form.
	   * @example
	   * const v = new pc.Vec2(20, 10);
	   * // Outputs [20, 10]
	   * console.log(v.toString());
	   */;
	  _proto.toString = function toString() {
	    return "[" + this.x + ", " + this.y + "]";
	  }

	  /**
	   * Calculates the angle between two Vec2's in radians.
	   *
	   * @param {Vec2} lhs - The first vector operand for the calculation.
	   * @param {Vec2} rhs - The second vector operand for the calculation.
	   * @returns {number} The calculated angle in radians.
	   * @ignore
	   */;
	  Vec2.angleRad = function angleRad(lhs, rhs) {
	    return Math.atan2(lhs.x * rhs.y - lhs.y * rhs.x, lhs.x * rhs.x + lhs.y * rhs.y);
	  }

	  /**
	   * A constant vector set to [0, 0].
	   *
	   * @type {Vec2}
	   * @readonly
	   */;
	  return Vec2;
	}();
	_class$7 = Vec2;
	Vec2.ZERO = Object.freeze(new _class$7(0, 0));
	/**
	 * A constant vector set to [1, 1].
	 *
	 * @type {Vec2}
	 * @readonly
	 */
	Vec2.ONE = Object.freeze(new _class$7(1, 1));
	/**
	 * A constant vector set to [0, 1].
	 *
	 * @type {Vec2}
	 * @readonly
	 */
	Vec2.UP = Object.freeze(new _class$7(0, 1));
	/**
	 * A constant vector set to [0, -1].
	 *
	 * @type {Vec2}
	 * @readonly
	 */
	Vec2.DOWN = Object.freeze(new _class$7(0, -1));
	/**
	 * A constant vector set to [1, 0].
	 *
	 * @type {Vec2}
	 * @readonly
	 */
	Vec2.RIGHT = Object.freeze(new _class$7(1, 0));
	/**
	 * A constant vector set to [-1, 0].
	 *
	 * @type {Vec2}
	 * @readonly
	 */
	Vec2.LEFT = Object.freeze(new _class$7(-1, 0));

	var _class$6;
	/**
	 * A 4-dimensional vector.
	 *
	 * @category Math
	 */
	var Vec4 = /*#__PURE__*/function () {
	  /**
	   * Creates a new Vec4 object.
	   *
	   * @param {number|number[]} [x] - The x value. Defaults to 0. If x is an array of length 4, the
	   * array will be used to populate all components.
	   * @param {number} [y] - The y value. Defaults to 0.
	   * @param {number} [z] - The z value. Defaults to 0.
	   * @param {number} [w] - The w value. Defaults to 0.
	   * @example
	   * const v = new pc.Vec4(1, 2, 3, 4);
	   */
	  function Vec4(x, y, z, w) {
	    if (x === void 0) {
	      x = 0;
	    }
	    if (y === void 0) {
	      y = 0;
	    }
	    if (z === void 0) {
	      z = 0;
	    }
	    if (w === void 0) {
	      w = 0;
	    }
	    /**
	     * The first component of the vector.
	     *
	     * @type {number}
	     */
	    this.x = void 0;
	    /**
	     * The second component of the vector.
	     *
	     * @type {number}
	     */
	    this.y = void 0;
	    /**
	     * The third component of the vector.
	     *
	     * @type {number}
	     */
	    this.z = void 0;
	    /**
	     * The fourth component of the vector.
	     *
	     * @type {number}
	     */
	    this.w = void 0;
	    if (x.length === 4) {
	      this.x = x[0];
	      this.y = x[1];
	      this.z = x[2];
	      this.w = x[3];
	    } else {
	      this.x = x;
	      this.y = y;
	      this.z = z;
	      this.w = w;
	    }
	  }

	  /**
	   * Adds a 4-dimensional vector to another in place.
	   *
	   * @param {Vec4} rhs - The vector to add to the specified vector.
	   * @returns {Vec4} Self for chaining.
	   * @example
	   * const a = new pc.Vec4(10, 10, 10, 10);
	   * const b = new pc.Vec4(20, 20, 20, 20);
	   *
	   * a.add(b);
	   *
	   * // Outputs [30, 30, 30]
	   * console.log("The result of the addition is: " + a.toString());
	   */
	  var _proto = Vec4.prototype;
	  _proto.add = function add(rhs) {
	    this.x += rhs.x;
	    this.y += rhs.y;
	    this.z += rhs.z;
	    this.w += rhs.w;
	    return this;
	  }

	  /**
	   * Adds two 4-dimensional vectors together and returns the result.
	   *
	   * @param {Vec4} lhs - The first vector operand for the addition.
	   * @param {Vec4} rhs - The second vector operand for the addition.
	   * @returns {Vec4} Self for chaining.
	   * @example
	   * const a = new pc.Vec4(10, 10, 10, 10);
	   * const b = new pc.Vec4(20, 20, 20, 20);
	   * const r = new pc.Vec4();
	   *
	   * r.add2(a, b);
	   * // Outputs [30, 30, 30]
	   *
	   * console.log("The result of the addition is: " + r.toString());
	   */;
	  _proto.add2 = function add2(lhs, rhs) {
	    this.x = lhs.x + rhs.x;
	    this.y = lhs.y + rhs.y;
	    this.z = lhs.z + rhs.z;
	    this.w = lhs.w + rhs.w;
	    return this;
	  }

	  /**
	   * Adds a number to each element of a vector.
	   *
	   * @param {number} scalar - The number to add.
	   * @returns {Vec4} Self for chaining.
	   * @example
	   * const vec = new pc.Vec4(3, 4, 5, 6);
	   *
	   * vec.addScalar(2);
	   *
	   * // Outputs [5, 6, 7, 8]
	   * console.log("The result of the addition is: " + vec.toString());
	   */;
	  _proto.addScalar = function addScalar(scalar) {
	    this.x += scalar;
	    this.y += scalar;
	    this.z += scalar;
	    this.w += scalar;
	    return this;
	  }

	  /**
	   * Adds a 4-dimensional vector scaled by scalar value. Does not modify the vector being added.
	   *
	   * @param {Vec4} rhs - The vector to add to the specified vector.
	   * @param {number} scalar - The number to multiply the added vector with.
	   * @returns {Vec4} Self for chaining.
	   * @example
	   * const vec = new pc.Vec4(1, 2, 3, 4);
	   *
	   * vec.addScaled(pc.Vec4.ONE, 2);
	   *
	   * // Outputs [3, 4, 5, 6]
	   * console.log("The result of the addition is: " + vec.toString());
	   */;
	  _proto.addScaled = function addScaled(rhs, scalar) {
	    this.x += rhs.x * scalar;
	    this.y += rhs.y * scalar;
	    this.z += rhs.z * scalar;
	    this.w += rhs.w * scalar;
	    return this;
	  }

	  /**
	   * Returns an identical copy of the specified 4-dimensional vector.
	   *
	   * @returns {this} A 4-dimensional vector containing the result of the cloning.
	   * @example
	   * const v = new pc.Vec4(10, 20, 30, 40);
	   * const vclone = v.clone();
	   * console.log("The result of the cloning is: " + vclone.toString());
	   */;
	  _proto.clone = function clone() {
	    /** @type {this} */
	    var cstr = this.constructor;
	    return new cstr(this.x, this.y, this.z, this.w);
	  }

	  /**
	   * Copies the contents of a source 4-dimensional vector to a destination 4-dimensional vector.
	   *
	   * @param {Vec4} rhs - A vector to copy to the specified vector.
	   * @returns {Vec4} Self for chaining.
	   * @example
	   * const src = new pc.Vec4(10, 20, 30, 40);
	   * const dst = new pc.Vec4();
	   *
	   * dst.copy(src);
	   *
	   * console.log("The two vectors are " + (dst.equals(src) ? "equal" : "different"));
	   */;
	  _proto.copy = function copy(rhs) {
	    this.x = rhs.x;
	    this.y = rhs.y;
	    this.z = rhs.z;
	    this.w = rhs.w;
	    return this;
	  }

	  /**
	   * Divides a 4-dimensional vector by another in place.
	   *
	   * @param {Vec4} rhs - The vector to divide the specified vector by.
	   * @returns {Vec4} Self for chaining.
	   * @example
	   * const a = new pc.Vec4(4, 9, 16, 25);
	   * const b = new pc.Vec4(2, 3, 4, 5);
	   *
	   * a.div(b);
	   *
	   * // Outputs [2, 3, 4, 5]
	   * console.log("The result of the division is: " + a.toString());
	   */;
	  _proto.div = function div(rhs) {
	    this.x /= rhs.x;
	    this.y /= rhs.y;
	    this.z /= rhs.z;
	    this.w /= rhs.w;
	    return this;
	  }

	  /**
	   * Divides one 4-dimensional vector by another and writes the result to the specified vector.
	   *
	   * @param {Vec4} lhs - The dividend vector (the vector being divided).
	   * @param {Vec4} rhs - The divisor vector (the vector dividing the dividend).
	   * @returns {Vec4} Self for chaining.
	   * @example
	   * const a = new pc.Vec4(4, 9, 16, 25);
	   * const b = new pc.Vec4(2, 3, 4, 5);
	   * const r = new pc.Vec4();
	   *
	   * r.div2(a, b);
	   * // Outputs [2, 3, 4, 5]
	   *
	   * console.log("The result of the division is: " + r.toString());
	   */;
	  _proto.div2 = function div2(lhs, rhs) {
	    this.x = lhs.x / rhs.x;
	    this.y = lhs.y / rhs.y;
	    this.z = lhs.z / rhs.z;
	    this.w = lhs.w / rhs.w;
	    return this;
	  }

	  /**
	   * Divides each element of a vector by a number.
	   *
	   * @param {number} scalar - The number to divide by.
	   * @returns {Vec4} Self for chaining.
	   * @example
	   * const vec = new pc.Vec4(3, 6, 9, 12);
	   *
	   * vec.divScalar(3);
	   *
	   * // Outputs [1, 2, 3, 4]
	   * console.log("The result of the division is: " + vec.toString());
	   */;
	  _proto.divScalar = function divScalar(scalar) {
	    this.x /= scalar;
	    this.y /= scalar;
	    this.z /= scalar;
	    this.w /= scalar;
	    return this;
	  }

	  /**
	   * Returns the result of a dot product operation performed on the two specified 4-dimensional
	   * vectors.
	   *
	   * @param {Vec4} rhs - The second 4-dimensional vector operand of the dot product.
	   * @returns {number} The result of the dot product operation.
	   * @example
	   * const v1 = new pc.Vec4(5, 10, 20, 40);
	   * const v2 = new pc.Vec4(10, 20, 40, 80);
	   * const v1dotv2 = v1.dot(v2);
	   * console.log("The result of the dot product is: " + v1dotv2);
	   */;
	  _proto.dot = function dot(rhs) {
	    return this.x * rhs.x + this.y * rhs.y + this.z * rhs.z + this.w * rhs.w;
	  }

	  /**
	   * Reports whether two vectors are equal.
	   *
	   * @param {Vec4} rhs - The vector to compare to the specified vector.
	   * @returns {boolean} True if the vectors are equal and false otherwise.
	   * @example
	   * const a = new pc.Vec4(1, 2, 3, 4);
	   * const b = new pc.Vec4(5, 6, 7, 8);
	   * console.log("The two vectors are " + (a.equals(b) ? "equal" : "different"));
	   */;
	  _proto.equals = function equals(rhs) {
	    return this.x === rhs.x && this.y === rhs.y && this.z === rhs.z && this.w === rhs.w;
	  }

	  /**
	   * Reports whether two vectors are equal using an absolute error tolerance.
	   *
	   * @param {Vec4} rhs - The vector to be compared against.
	   * @param {number} [epsilon] - The maximum difference between each component of the two
	   * vectors. Defaults to 1e-6.
	   * @returns {boolean} True if the vectors are equal and false otherwise.
	   * @example
	   * const a = new pc.Vec4();
	   * const b = new pc.Vec4();
	   * console.log("The two vectors are approximately " + (a.equalsApprox(b, 1e-9) ? "equal" : "different"));
	   */;
	  _proto.equalsApprox = function equalsApprox(rhs, epsilon) {
	    if (epsilon === void 0) {
	      epsilon = 1e-6;
	    }
	    return Math.abs(this.x - rhs.x) < epsilon && Math.abs(this.y - rhs.y) < epsilon && Math.abs(this.z - rhs.z) < epsilon && Math.abs(this.w - rhs.w) < epsilon;
	  }

	  /**
	   * Returns the magnitude of the specified 4-dimensional vector.
	   *
	   * @returns {number} The magnitude of the specified 4-dimensional vector.
	   * @example
	   * const vec = new pc.Vec4(3, 4, 0, 0);
	   * const len = vec.length();
	   * // Outputs 5
	   * console.log("The length of the vector is: " + len);
	   */;
	  _proto.length = function length() {
	    return Math.sqrt(this.x * this.x + this.y * this.y + this.z * this.z + this.w * this.w);
	  }

	  /**
	   * Returns the magnitude squared of the specified 4-dimensional vector.
	   *
	   * @returns {number} The magnitude of the specified 4-dimensional vector.
	   * @example
	   * const vec = new pc.Vec4(3, 4, 0);
	   * const len = vec.lengthSq();
	   * // Outputs 25
	   * console.log("The length squared of the vector is: " + len);
	   */;
	  _proto.lengthSq = function lengthSq() {
	    return this.x * this.x + this.y * this.y + this.z * this.z + this.w * this.w;
	  }

	  /**
	   * Returns the result of a linear interpolation between two specified 4-dimensional vectors.
	   *
	   * @param {Vec4} lhs - The 4-dimensional to interpolate from.
	   * @param {Vec4} rhs - The 4-dimensional to interpolate to.
	   * @param {number} alpha - The value controlling the point of interpolation. Between 0 and 1,
	   * the linear interpolant will occur on a straight line between lhs and rhs. Outside of this
	   * range, the linear interpolant will occur on a ray extrapolated from this line.
	   * @returns {Vec4} Self for chaining.
	   * @example
	   * const a = new pc.Vec4(0, 0, 0, 0);
	   * const b = new pc.Vec4(10, 10, 10, 10);
	   * const r = new pc.Vec4();
	   *
	   * r.lerp(a, b, 0);   // r is equal to a
	   * r.lerp(a, b, 0.5); // r is 5, 5, 5, 5
	   * r.lerp(a, b, 1);   // r is equal to b
	   */;
	  _proto.lerp = function lerp(lhs, rhs, alpha) {
	    this.x = lhs.x + alpha * (rhs.x - lhs.x);
	    this.y = lhs.y + alpha * (rhs.y - lhs.y);
	    this.z = lhs.z + alpha * (rhs.z - lhs.z);
	    this.w = lhs.w + alpha * (rhs.w - lhs.w);
	    return this;
	  }

	  /**
	   * Multiplies a 4-dimensional vector to another in place.
	   *
	   * @param {Vec4} rhs - The 4-dimensional vector used as the second multiplicand of the operation.
	   * @returns {Vec4} Self for chaining.
	   * @example
	   * const a = new pc.Vec4(2, 3, 4, 5);
	   * const b = new pc.Vec4(4, 5, 6, 7);
	   *
	   * a.mul(b);
	   *
	   * // Outputs 8, 15, 24, 35
	   * console.log("The result of the multiplication is: " + a.toString());
	   */;
	  _proto.mul = function mul(rhs) {
	    this.x *= rhs.x;
	    this.y *= rhs.y;
	    this.z *= rhs.z;
	    this.w *= rhs.w;
	    return this;
	  }

	  /**
	   * Returns the result of multiplying the specified 4-dimensional vectors together.
	   *
	   * @param {Vec4} lhs - The 4-dimensional vector used as the first multiplicand of the operation.
	   * @param {Vec4} rhs - The 4-dimensional vector used as the second multiplicand of the operation.
	   * @returns {Vec4} Self for chaining.
	   * @example
	   * const a = new pc.Vec4(2, 3, 4, 5);
	   * const b = new pc.Vec4(4, 5, 6, 7);
	   * const r = new pc.Vec4();
	   *
	   * r.mul2(a, b);
	   *
	   * // Outputs 8, 15, 24, 35
	   * console.log("The result of the multiplication is: " + r.toString());
	   */;
	  _proto.mul2 = function mul2(lhs, rhs) {
	    this.x = lhs.x * rhs.x;
	    this.y = lhs.y * rhs.y;
	    this.z = lhs.z * rhs.z;
	    this.w = lhs.w * rhs.w;
	    return this;
	  }

	  /**
	   * Multiplies each element of a vector by a number.
	   *
	   * @param {number} scalar - The number to multiply by.
	   * @returns {Vec4} Self for chaining.
	   * @example
	   * const vec = new pc.Vec4(3, 6, 9, 12);
	   *
	   * vec.mulScalar(3);
	   *
	   * // Outputs [9, 18, 27, 36]
	   * console.log("The result of the multiplication is: " + vec.toString());
	   */;
	  _proto.mulScalar = function mulScalar(scalar) {
	    this.x *= scalar;
	    this.y *= scalar;
	    this.z *= scalar;
	    this.w *= scalar;
	    return this;
	  }

	  /**
	   * Returns this 4-dimensional vector converted to a unit vector in place. If the vector has a
	   * length of zero, the vector's elements will be set to zero.
	   *
	   * @param {Vec4} [src] - The vector to normalize. If not set, the operation is done in place.
	   * @returns {Vec4} Self for chaining.
	   * @example
	   * const v = new pc.Vec4(25, 0, 0, 0);
	   *
	   * v.normalize();
	   *
	   * // Outputs 1, 0, 0, 0
	   * console.log("The result of the vector normalization is: " + v.toString());
	   */;
	  _proto.normalize = function normalize(src) {
	    if (src === void 0) {
	      src = this;
	    }
	    var lengthSq = src.x * src.x + src.y * src.y + src.z * src.z + src.w * src.w;
	    if (lengthSq > 0) {
	      var invLength = 1 / Math.sqrt(lengthSq);
	      this.x = src.x * invLength;
	      this.y = src.y * invLength;
	      this.z = src.z * invLength;
	      this.w = src.w * invLength;
	    }
	    return this;
	  }

	  /**
	   * Each element is set to the largest integer less than or equal to its value.
	   *
	   * @param {Vec4} [src] - The vector to floor. If not set, the operation is done in place.
	   * @returns {Vec4} Self for chaining.
	   */;
	  _proto.floor = function floor(src) {
	    if (src === void 0) {
	      src = this;
	    }
	    this.x = Math.floor(src.x);
	    this.y = Math.floor(src.y);
	    this.z = Math.floor(src.z);
	    this.w = Math.floor(src.w);
	    return this;
	  }

	  /**
	   * Each element is rounded up to the next largest integer.
	   *
	   * @param {Vec4} [src] - The vector to ceil. If not set, the operation is done in place.
	   * @returns {Vec4} Self for chaining.
	   */;
	  _proto.ceil = function ceil(src) {
	    if (src === void 0) {
	      src = this;
	    }
	    this.x = Math.ceil(src.x);
	    this.y = Math.ceil(src.y);
	    this.z = Math.ceil(src.z);
	    this.w = Math.ceil(src.w);
	    return this;
	  }

	  /**
	   * Each element is rounded up or down to the nearest integer.
	   *
	   * @param {Vec4} [src] - The vector to round. If not set, the operation is done in place.
	   * @returns {Vec4} Self for chaining.
	   */;
	  _proto.round = function round(src) {
	    if (src === void 0) {
	      src = this;
	    }
	    this.x = Math.round(src.x);
	    this.y = Math.round(src.y);
	    this.z = Math.round(src.z);
	    this.w = Math.round(src.w);
	    return this;
	  }

	  /**
	   * Each element is assigned a value from rhs parameter if it is smaller.
	   *
	   * @param {Vec4} rhs - The 4-dimensional vector used as the source of elements to compare to.
	   * @returns {Vec4} Self for chaining.
	   */;
	  _proto.min = function min(rhs) {
	    if (rhs.x < this.x) this.x = rhs.x;
	    if (rhs.y < this.y) this.y = rhs.y;
	    if (rhs.z < this.z) this.z = rhs.z;
	    if (rhs.w < this.w) this.w = rhs.w;
	    return this;
	  }

	  /**
	   * Each element is assigned a value from rhs parameter if it is larger.
	   *
	   * @param {Vec4} rhs - The 4-dimensional vector used as the source of elements to compare to.
	   * @returns {Vec4} Self for chaining.
	   */;
	  _proto.max = function max(rhs) {
	    if (rhs.x > this.x) this.x = rhs.x;
	    if (rhs.y > this.y) this.y = rhs.y;
	    if (rhs.z > this.z) this.z = rhs.z;
	    if (rhs.w > this.w) this.w = rhs.w;
	    return this;
	  }

	  /**
	   * Sets the specified 4-dimensional vector to the supplied numerical values.
	   *
	   * @param {number} x - The value to set on the first component of the vector.
	   * @param {number} y - The value to set on the second component of the vector.
	   * @param {number} z - The value to set on the third component of the vector.
	   * @param {number} w - The value to set on the fourth component of the vector.
	   * @returns {Vec4} Self for chaining.
	   * @example
	   * const v = new pc.Vec4();
	   * v.set(5, 10, 20, 40);
	   *
	   * // Outputs 5, 10, 20, 40
	   * console.log("The result of the vector set is: " + v.toString());
	   */;
	  _proto.set = function set(x, y, z, w) {
	    this.x = x;
	    this.y = y;
	    this.z = z;
	    this.w = w;
	    return this;
	  }

	  /**
	   * Subtracts a 4-dimensional vector from another in place.
	   *
	   * @param {Vec4} rhs - The vector to add to the specified vector.
	   * @returns {Vec4} Self for chaining.
	   * @example
	   * const a = new pc.Vec4(10, 10, 10, 10);
	   * const b = new pc.Vec4(20, 20, 20, 20);
	   *
	   * a.sub(b);
	   *
	   * // Outputs [-10, -10, -10, -10]
	   * console.log("The result of the subtraction is: " + a.toString());
	   */;
	  _proto.sub = function sub(rhs) {
	    this.x -= rhs.x;
	    this.y -= rhs.y;
	    this.z -= rhs.z;
	    this.w -= rhs.w;
	    return this;
	  }

	  /**
	   * Subtracts two 4-dimensional vectors from one another and returns the result.
	   *
	   * @param {Vec4} lhs - The first vector operand for the subtraction.
	   * @param {Vec4} rhs - The second vector operand for the subtraction.
	   * @returns {Vec4} Self for chaining.
	   * @example
	   * const a = new pc.Vec4(10, 10, 10, 10);
	   * const b = new pc.Vec4(20, 20, 20, 20);
	   * const r = new pc.Vec4();
	   *
	   * r.sub2(a, b);
	   *
	   * // Outputs [-10, -10, -10, -10]
	   * console.log("The result of the subtraction is: " + r.toString());
	   */;
	  _proto.sub2 = function sub2(lhs, rhs) {
	    this.x = lhs.x - rhs.x;
	    this.y = lhs.y - rhs.y;
	    this.z = lhs.z - rhs.z;
	    this.w = lhs.w - rhs.w;
	    return this;
	  }

	  /**
	   * Subtracts a number from each element of a vector.
	   *
	   * @param {number} scalar - The number to subtract.
	   * @returns {Vec4} Self for chaining.
	   * @example
	   * const vec = new pc.Vec4(3, 4, 5, 6);
	   *
	   * vec.subScalar(2);
	   *
	   * // Outputs [1, 2, 3, 4]
	   * console.log("The result of the subtraction is: " + vec.toString());
	   */;
	  _proto.subScalar = function subScalar(scalar) {
	    this.x -= scalar;
	    this.y -= scalar;
	    this.z -= scalar;
	    this.w -= scalar;
	    return this;
	  }

	  /**
	   * Converts the vector to string form.
	   *
	   * @returns {string} The vector in string form.
	   * @example
	   * const v = new pc.Vec4(20, 10, 5, 0);
	   * // Outputs [20, 10, 5, 0]
	   * console.log(v.toString());
	   */;
	  _proto.toString = function toString() {
	    return "[" + this.x + ", " + this.y + ", " + this.z + ", " + this.w + "]";
	  }

	  /**
	   * A constant vector set to [0, 0, 0, 0].
	   *
	   * @type {Vec4}
	   * @readonly
	   */;
	  return Vec4;
	}();
	_class$6 = Vec4;
	Vec4.ZERO = Object.freeze(new _class$6(0, 0, 0, 0));
	/**
	 * A constant vector set to [1, 1, 1, 1].
	 *
	 * @type {Vec4}
	 * @readonly
	 */
	Vec4.ONE = Object.freeze(new _class$6(1, 1, 1, 1));

	var _class$5;
	var _halfSize$1 = new Vec2();
	var x = new Vec3();
	var y = new Vec3();
	var z = new Vec3();
	var scale = new Vec3();

	/**
	 * A 4x4 matrix.
	 *
	 * @category Math
	 */
	var Mat4 = /*#__PURE__*/function () {
	  /**
	   * Create a new Mat4 instance. It is initialized to the identity matrix.
	   */
	  function Mat4() {
	    /**
	     * Matrix elements in the form of a flat array.
	     *
	     * @type {Float32Array}
	     */
	    this.data = new Float32Array(16);
	    // Create an identity matrix. Note that a new Float32Array has all elements set
	    // to zero by default, so we only need to set the relevant elements to one.
	    this.data[0] = this.data[5] = this.data[10] = this.data[15] = 1;
	  }

	  // Static function which evaluates perspective projection matrix half size at the near plane
	  Mat4._getPerspectiveHalfSize = function _getPerspectiveHalfSize(halfSize, fov, aspect, znear, fovIsHorizontal) {
	    if (fovIsHorizontal) {
	      halfSize.x = znear * Math.tan(fov * Math.PI / 360);
	      halfSize.y = halfSize.x / aspect;
	    } else {
	      halfSize.y = znear * Math.tan(fov * Math.PI / 360);
	      halfSize.x = halfSize.y * aspect;
	    }
	  }

	  /**
	   * Adds the specified 4x4 matrices together and stores the result in the current instance.
	   *
	   * @param {Mat4} lhs - The 4x4 matrix used as the first operand of the addition.
	   * @param {Mat4} rhs - The 4x4 matrix used as the second operand of the addition.
	   * @returns {Mat4} Self for chaining.
	   * @example
	   * const m = new pc.Mat4();
	   *
	   * m.add2(pc.Mat4.IDENTITY, pc.Mat4.ONE);
	   *
	   * console.log("The result of the addition is: " + m.toString());
	   */;
	  var _proto = Mat4.prototype;
	  _proto.add2 = function add2(lhs, rhs) {
	    var a = lhs.data,
	      b = rhs.data,
	      r = this.data;
	    r[0] = a[0] + b[0];
	    r[1] = a[1] + b[1];
	    r[2] = a[2] + b[2];
	    r[3] = a[3] + b[3];
	    r[4] = a[4] + b[4];
	    r[5] = a[5] + b[5];
	    r[6] = a[6] + b[6];
	    r[7] = a[7] + b[7];
	    r[8] = a[8] + b[8];
	    r[9] = a[9] + b[9];
	    r[10] = a[10] + b[10];
	    r[11] = a[11] + b[11];
	    r[12] = a[12] + b[12];
	    r[13] = a[13] + b[13];
	    r[14] = a[14] + b[14];
	    r[15] = a[15] + b[15];
	    return this;
	  }

	  /**
	   * Adds the specified 4x4 matrix to the current instance.
	   *
	   * @param {Mat4} rhs - The 4x4 matrix used as the second operand of the addition.
	   * @returns {Mat4} Self for chaining.
	   * @example
	   * const m = new pc.Mat4();
	   *
	   * m.add(pc.Mat4.ONE);
	   *
	   * console.log("The result of the addition is: " + m.toString());
	   */;
	  _proto.add = function add(rhs) {
	    return this.add2(this, rhs);
	  }

	  /**
	   * Creates a duplicate of the specified matrix.
	   *
	   * @returns {this} A duplicate matrix.
	   * @example
	   * const src = new pc.Mat4().setFromEulerAngles(10, 20, 30);
	   * const dst = src.clone();
	   * console.log("The two matrices are " + (src.equals(dst) ? "equal" : "different"));
	   */;
	  _proto.clone = function clone() {
	    /** @type {this} */
	    var cstr = this.constructor;
	    return new cstr().copy(this);
	  }

	  /**
	   * Copies the contents of a source 4x4 matrix to a destination 4x4 matrix.
	   *
	   * @param {Mat4} rhs - A 4x4 matrix to be copied.
	   * @returns {Mat4} Self for chaining.
	   * @example
	   * const src = new pc.Mat4().setFromEulerAngles(10, 20, 30);
	   * const dst = new pc.Mat4();
	   * dst.copy(src);
	   * console.log("The two matrices are " + (src.equals(dst) ? "equal" : "different"));
	   */;
	  _proto.copy = function copy(rhs) {
	    var src = rhs.data,
	      dst = this.data;
	    dst[0] = src[0];
	    dst[1] = src[1];
	    dst[2] = src[2];
	    dst[3] = src[3];
	    dst[4] = src[4];
	    dst[5] = src[5];
	    dst[6] = src[6];
	    dst[7] = src[7];
	    dst[8] = src[8];
	    dst[9] = src[9];
	    dst[10] = src[10];
	    dst[11] = src[11];
	    dst[12] = src[12];
	    dst[13] = src[13];
	    dst[14] = src[14];
	    dst[15] = src[15];
	    return this;
	  }

	  /**
	   * Reports whether two matrices are equal.
	   *
	   * @param {Mat4} rhs - The other matrix.
	   * @returns {boolean} True if the matrices are equal and false otherwise.
	   * @example
	   * const a = new pc.Mat4().setFromEulerAngles(10, 20, 30);
	   * const b = new pc.Mat4();
	   * console.log("The two matrices are " + (a.equals(b) ? "equal" : "different"));
	   */;
	  _proto.equals = function equals(rhs) {
	    var l = this.data,
	      r = rhs.data;
	    return l[0] === r[0] && l[1] === r[1] && l[2] === r[2] && l[3] === r[3] && l[4] === r[4] && l[5] === r[5] && l[6] === r[6] && l[7] === r[7] && l[8] === r[8] && l[9] === r[9] && l[10] === r[10] && l[11] === r[11] && l[12] === r[12] && l[13] === r[13] && l[14] === r[14] && l[15] === r[15];
	  }

	  /**
	   * Reports whether the specified matrix is the identity matrix.
	   *
	   * @returns {boolean} True if the matrix is identity and false otherwise.
	   * @example
	   * const m = new pc.Mat4();
	   * console.log("The matrix is " + (m.isIdentity() ? "identity" : "not identity"));
	   */;
	  _proto.isIdentity = function isIdentity() {
	    var m = this.data;
	    return m[0] === 1 && m[1] === 0 && m[2] === 0 && m[3] === 0 && m[4] === 0 && m[5] === 1 && m[6] === 0 && m[7] === 0 && m[8] === 0 && m[9] === 0 && m[10] === 1 && m[11] === 0 && m[12] === 0 && m[13] === 0 && m[14] === 0 && m[15] === 1;
	  }

	  /**
	   * Multiplies the specified 4x4 matrices together and stores the result in the current
	   * instance.
	   *
	   * @param {Mat4} lhs - The 4x4 matrix used as the first multiplicand of the operation.
	   * @param {Mat4} rhs - The 4x4 matrix used as the second multiplicand of the operation.
	   * @returns {Mat4} Self for chaining.
	   * @example
	   * const a = new pc.Mat4().setFromEulerAngles(10, 20, 30);
	   * const b = new pc.Mat4().setFromAxisAngle(pc.Vec3.UP, 180);
	   * const r = new pc.Mat4();
	   *
	   * // r = a * b
	   * r.mul2(a, b);
	   *
	   * console.log("The result of the multiplication is: " + r.toString());
	   */;
	  _proto.mul2 = function mul2(lhs, rhs) {
	    var a = lhs.data;
	    var b = rhs.data;
	    var r = this.data;
	    var a00 = a[0];
	    var a01 = a[1];
	    var a02 = a[2];
	    var a03 = a[3];
	    var a10 = a[4];
	    var a11 = a[5];
	    var a12 = a[6];
	    var a13 = a[7];
	    var a20 = a[8];
	    var a21 = a[9];
	    var a22 = a[10];
	    var a23 = a[11];
	    var a30 = a[12];
	    var a31 = a[13];
	    var a32 = a[14];
	    var a33 = a[15];
	    var b0, b1, b2, b3;
	    b0 = b[0];
	    b1 = b[1];
	    b2 = b[2];
	    b3 = b[3];
	    r[0] = a00 * b0 + a10 * b1 + a20 * b2 + a30 * b3;
	    r[1] = a01 * b0 + a11 * b1 + a21 * b2 + a31 * b3;
	    r[2] = a02 * b0 + a12 * b1 + a22 * b2 + a32 * b3;
	    r[3] = a03 * b0 + a13 * b1 + a23 * b2 + a33 * b3;
	    b0 = b[4];
	    b1 = b[5];
	    b2 = b[6];
	    b3 = b[7];
	    r[4] = a00 * b0 + a10 * b1 + a20 * b2 + a30 * b3;
	    r[5] = a01 * b0 + a11 * b1 + a21 * b2 + a31 * b3;
	    r[6] = a02 * b0 + a12 * b1 + a22 * b2 + a32 * b3;
	    r[7] = a03 * b0 + a13 * b1 + a23 * b2 + a33 * b3;
	    b0 = b[8];
	    b1 = b[9];
	    b2 = b[10];
	    b3 = b[11];
	    r[8] = a00 * b0 + a10 * b1 + a20 * b2 + a30 * b3;
	    r[9] = a01 * b0 + a11 * b1 + a21 * b2 + a31 * b3;
	    r[10] = a02 * b0 + a12 * b1 + a22 * b2 + a32 * b3;
	    r[11] = a03 * b0 + a13 * b1 + a23 * b2 + a33 * b3;
	    b0 = b[12];
	    b1 = b[13];
	    b2 = b[14];
	    b3 = b[15];
	    r[12] = a00 * b0 + a10 * b1 + a20 * b2 + a30 * b3;
	    r[13] = a01 * b0 + a11 * b1 + a21 * b2 + a31 * b3;
	    r[14] = a02 * b0 + a12 * b1 + a22 * b2 + a32 * b3;
	    r[15] = a03 * b0 + a13 * b1 + a23 * b2 + a33 * b3;
	    return this;
	  }

	  /**
	   * Multiplies the specified 4x4 matrices together and stores the result in the current
	   * instance. This function assumes the matrices are affine transformation matrices, where the
	   * upper left 3x3 elements are a rotation matrix, and the bottom left 3 elements are
	   * translation. The rightmost column is assumed to be [0, 0, 0, 1]. The parameters are not
	   * verified to be in the expected format. This function is faster than general
	   * {@link Mat4#mul2}.
	   *
	   * @param {Mat4} lhs - The affine transformation 4x4 matrix used as the first multiplicand of
	   * the operation.
	   * @param {Mat4} rhs - The affine transformation 4x4 matrix used as the second multiplicand of
	   * the operation.
	   * @returns {Mat4} Self for chaining.
	   */;
	  _proto.mulAffine2 = function mulAffine2(lhs, rhs) {
	    var a = lhs.data;
	    var b = rhs.data;
	    var r = this.data;
	    var a00 = a[0];
	    var a01 = a[1];
	    var a02 = a[2];
	    var a10 = a[4];
	    var a11 = a[5];
	    var a12 = a[6];
	    var a20 = a[8];
	    var a21 = a[9];
	    var a22 = a[10];
	    var a30 = a[12];
	    var a31 = a[13];
	    var a32 = a[14];
	    var b0, b1, b2;
	    b0 = b[0];
	    b1 = b[1];
	    b2 = b[2];
	    r[0] = a00 * b0 + a10 * b1 + a20 * b2;
	    r[1] = a01 * b0 + a11 * b1 + a21 * b2;
	    r[2] = a02 * b0 + a12 * b1 + a22 * b2;
	    r[3] = 0;
	    b0 = b[4];
	    b1 = b[5];
	    b2 = b[6];
	    r[4] = a00 * b0 + a10 * b1 + a20 * b2;
	    r[5] = a01 * b0 + a11 * b1 + a21 * b2;
	    r[6] = a02 * b0 + a12 * b1 + a22 * b2;
	    r[7] = 0;
	    b0 = b[8];
	    b1 = b[9];
	    b2 = b[10];
	    r[8] = a00 * b0 + a10 * b1 + a20 * b2;
	    r[9] = a01 * b0 + a11 * b1 + a21 * b2;
	    r[10] = a02 * b0 + a12 * b1 + a22 * b2;
	    r[11] = 0;
	    b0 = b[12];
	    b1 = b[13];
	    b2 = b[14];
	    r[12] = a00 * b0 + a10 * b1 + a20 * b2 + a30;
	    r[13] = a01 * b0 + a11 * b1 + a21 * b2 + a31;
	    r[14] = a02 * b0 + a12 * b1 + a22 * b2 + a32;
	    r[15] = 1;
	    return this;
	  }

	  /**
	   * Multiplies the current instance by the specified 4x4 matrix.
	   *
	   * @param {Mat4} rhs - The 4x4 matrix used as the second multiplicand of the operation.
	   * @returns {Mat4} Self for chaining.
	   * @example
	   * const a = new pc.Mat4().setFromEulerAngles(10, 20, 30);
	   * const b = new pc.Mat4().setFromAxisAngle(pc.Vec3.UP, 180);
	   *
	   * // a = a * b
	   * a.mul(b);
	   *
	   * console.log("The result of the multiplication is: " + a.toString());
	   */;
	  _proto.mul = function mul(rhs) {
	    return this.mul2(this, rhs);
	  }

	  /**
	   * Transforms a 3-dimensional point by a 4x4 matrix.
	   *
	   * @param {Vec3} vec - The 3-dimensional point to be transformed.
	   * @param {Vec3} [res] - An optional 3-dimensional point to receive the result of the
	   * transformation.
	   * @returns {Vec3} The input point v transformed by the current instance.
	   * @example
	   * // Create a 3-dimensional point
	   * const v = new pc.Vec3(1, 2, 3);
	   *
	   * // Create a 4x4 rotation matrix
	   * const m = new pc.Mat4().setFromEulerAngles(10, 20, 30);
	   *
	   * const tv = m.transformPoint(v);
	   */;
	  _proto.transformPoint = function transformPoint(vec, res) {
	    if (res === void 0) {
	      res = new Vec3();
	    }
	    var m = this.data;
	    var x = vec.x;
	    var y = vec.y;
	    var z = vec.z;
	    res.x = x * m[0] + y * m[4] + z * m[8] + m[12];
	    res.y = x * m[1] + y * m[5] + z * m[9] + m[13];
	    res.z = x * m[2] + y * m[6] + z * m[10] + m[14];
	    return res;
	  }

	  /**
	   * Transforms a 3-dimensional vector by a 4x4 matrix.
	   *
	   * @param {Vec3} vec - The 3-dimensional vector to be transformed.
	   * @param {Vec3} [res] - An optional 3-dimensional vector to receive the result of the
	   * transformation.
	   * @returns {Vec3} The input vector v transformed by the current instance.
	   * @example
	   * // Create a 3-dimensional vector
	   * const v = new pc.Vec3(1, 2, 3);
	   *
	   * // Create a 4x4 rotation matrix
	   * const m = new pc.Mat4().setFromEulerAngles(10, 20, 30);
	   *
	   * const tv = m.transformVector(v);
	   */;
	  _proto.transformVector = function transformVector(vec, res) {
	    if (res === void 0) {
	      res = new Vec3();
	    }
	    var m = this.data;
	    var x = vec.x;
	    var y = vec.y;
	    var z = vec.z;
	    res.x = x * m[0] + y * m[4] + z * m[8];
	    res.y = x * m[1] + y * m[5] + z * m[9];
	    res.z = x * m[2] + y * m[6] + z * m[10];
	    return res;
	  }

	  /**
	   * Transforms a 4-dimensional vector by a 4x4 matrix.
	   *
	   * @param {Vec4} vec - The 4-dimensional vector to be transformed.
	   * @param {Vec4} [res] - An optional 4-dimensional vector to receive the result of the
	   * transformation.
	   * @returns {Vec4} The input vector v transformed by the current instance.
	   * @example
	   * // Create an input 4-dimensional vector
	   * const v = new pc.Vec4(1, 2, 3, 4);
	   *
	   * // Create an output 4-dimensional vector
	   * const result = new pc.Vec4();
	   *
	   * // Create a 4x4 rotation matrix
	   * const m = new pc.Mat4().setFromEulerAngles(10, 20, 30);
	   *
	   * m.transformVec4(v, result);
	   */;
	  _proto.transformVec4 = function transformVec4(vec, res) {
	    if (res === void 0) {
	      res = new Vec4();
	    }
	    var m = this.data;
	    var x = vec.x;
	    var y = vec.y;
	    var z = vec.z;
	    var w = vec.w;
	    res.x = x * m[0] + y * m[4] + z * m[8] + w * m[12];
	    res.y = x * m[1] + y * m[5] + z * m[9] + w * m[13];
	    res.z = x * m[2] + y * m[6] + z * m[10] + w * m[14];
	    res.w = x * m[3] + y * m[7] + z * m[11] + w * m[15];
	    return res;
	  }

	  /**
	   * Sets the specified matrix to a viewing matrix derived from an eye point, a target point and
	   * an up vector. The matrix maps the target point to the negative z-axis and the eye point to
	   * the origin, so that when you use a typical projection matrix, the center of the scene maps
	   * to the center of the viewport. Similarly, the direction described by the up vector projected
	   * onto the viewing plane is mapped to the positive y-axis so that it points upward in the
	   * viewport. The up vector must not be parallel to the line of sight from the eye to the
	   * reference point.
	   *
	   * @param {Vec3} position - 3-d vector holding view position.
	   * @param {Vec3} target - 3-d vector holding reference point.
	   * @param {Vec3} up - 3-d vector holding the up direction.
	   * @returns {Mat4} Self for chaining.
	   * @example
	   * const position = new pc.Vec3(10, 10, 10);
	   * const target = new pc.Vec3(0, 0, 0);
	   * const up = new pc.Vec3(0, 1, 0);
	   * const m = new pc.Mat4().setLookAt(position, target, up);
	   */;
	  _proto.setLookAt = function setLookAt(position, target, up) {
	    z.sub2(position, target).normalize();
	    y.copy(up).normalize();
	    x.cross(y, z).normalize();
	    y.cross(z, x);
	    var r = this.data;
	    r[0] = x.x;
	    r[1] = x.y;
	    r[2] = x.z;
	    r[3] = 0;
	    r[4] = y.x;
	    r[5] = y.y;
	    r[6] = y.z;
	    r[7] = 0;
	    r[8] = z.x;
	    r[9] = z.y;
	    r[10] = z.z;
	    r[11] = 0;
	    r[12] = position.x;
	    r[13] = position.y;
	    r[14] = position.z;
	    r[15] = 1;
	    return this;
	  }

	  /**
	   * Sets the specified matrix to a perspective projection matrix. The function's parameters
	   * define the shape of a frustum.
	   *
	   * @param {number} left - The x-coordinate for the left edge of the camera's projection plane
	   * in eye space.
	   * @param {number} right - The x-coordinate for the right edge of the camera's projection plane
	   * in eye space.
	   * @param {number} bottom - The y-coordinate for the bottom edge of the camera's projection
	   * plane in eye space.
	   * @param {number} top - The y-coordinate for the top edge of the camera's projection plane in
	   * eye space.
	   * @param {number} znear - The near clip plane in eye coordinates.
	   * @param {number} zfar - The far clip plane in eye coordinates.
	   * @returns {Mat4} Self for chaining.
	   * @example
	   * // Create a 4x4 perspective projection matrix
	   * const f = pc.Mat4().setFrustum(-2, 2, -1, 1, 1, 1000);
	   * @ignore
	   */;
	  _proto.setFrustum = function setFrustum(left, right, bottom, top, znear, zfar) {
	    var temp1 = 2 * znear;
	    var temp2 = right - left;
	    var temp3 = top - bottom;
	    var temp4 = zfar - znear;
	    var r = this.data;
	    r[0] = temp1 / temp2;
	    r[1] = 0;
	    r[2] = 0;
	    r[3] = 0;
	    r[4] = 0;
	    r[5] = temp1 / temp3;
	    r[6] = 0;
	    r[7] = 0;
	    r[8] = (right + left) / temp2;
	    r[9] = (top + bottom) / temp3;
	    r[10] = (-zfar - znear) / temp4;
	    r[11] = -1;
	    r[12] = 0;
	    r[13] = 0;
	    r[14] = -temp1 * zfar / temp4;
	    r[15] = 0;
	    return this;
	  }

	  /**
	   * Sets the specified matrix to a perspective projection matrix. The function's parameters
	   * define the shape of a frustum.
	   *
	   * @param {number} fov - The frustum's field of view in degrees. The fovIsHorizontal parameter
	   * controls whether this is a vertical or horizontal field of view. By default, it's a vertical
	   * field of view.
	   * @param {number} aspect - The aspect ratio of the frustum's projection plane
	   * (width / height).
	   * @param {number} znear - The near clip plane in eye coordinates.
	   * @param {number} zfar - The far clip plane in eye coordinates.
	   * @param {boolean} [fovIsHorizontal] - Set to true to treat the fov as horizontal (x-axis) and
	   * false for vertical (y-axis). Defaults to false.
	   * @returns {Mat4} Self for chaining.
	   * @example
	   * // Create a 4x4 perspective projection matrix
	   * const persp = pc.Mat4().setPerspective(45, 16 / 9, 1, 1000);
	   */;
	  _proto.setPerspective = function setPerspective(fov, aspect, znear, zfar, fovIsHorizontal) {
	    Mat4._getPerspectiveHalfSize(_halfSize$1, fov, aspect, znear, fovIsHorizontal);
	    return this.setFrustum(-_halfSize$1.x, _halfSize$1.x, -_halfSize$1.y, _halfSize$1.y, znear, zfar);
	  }

	  /**
	   * Sets the specified matrix to an orthographic projection matrix. The function's parameters
	   * define the shape of a cuboid-shaped frustum.
	   *
	   * @param {number} left - The x-coordinate for the left edge of the camera's projection plane
	   * in eye space.
	   * @param {number} right - The x-coordinate for the right edge of the camera's projection plane
	   * in eye space.
	   * @param {number} bottom - The y-coordinate for the bottom edge of the camera's projection
	   * plane in eye space.
	   * @param {number} top - The y-coordinate for the top edge of the camera's projection plane in
	   * eye space.
	   * @param {number} near - The near clip plane in eye coordinates.
	   * @param {number} far - The far clip plane in eye coordinates.
	   * @returns {Mat4} Self for chaining.
	   * @example
	   * // Create a 4x4 orthographic projection matrix
	   * const ortho = pc.Mat4().ortho(-2, 2, -2, 2, 1, 1000);
	   */;
	  _proto.setOrtho = function setOrtho(left, right, bottom, top, near, far) {
	    var r = this.data;
	    r[0] = 2 / (right - left);
	    r[1] = 0;
	    r[2] = 0;
	    r[3] = 0;
	    r[4] = 0;
	    r[5] = 2 / (top - bottom);
	    r[6] = 0;
	    r[7] = 0;
	    r[8] = 0;
	    r[9] = 0;
	    r[10] = -2 / (far - near);
	    r[11] = 0;
	    r[12] = -(right + left) / (right - left);
	    r[13] = -(top + bottom) / (top - bottom);
	    r[14] = -(far + near) / (far - near);
	    r[15] = 1;
	    return this;
	  }

	  /**
	   * Sets the specified matrix to a rotation matrix equivalent to a rotation around an axis. The
	   * axis must be normalized (unit length) and the angle must be specified in degrees.
	   *
	   * @param {Vec3} axis - The normalized axis vector around which to rotate.
	   * @param {number} angle - The angle of rotation in degrees.
	   * @returns {Mat4} Self for chaining.
	   * @example
	   * // Create a 4x4 rotation matrix
	   * const rm = new pc.Mat4().setFromAxisAngle(pc.Vec3.UP, 90);
	   */;
	  _proto.setFromAxisAngle = function setFromAxisAngle(axis, angle) {
	    angle *= math.DEG_TO_RAD;
	    var x = axis.x;
	    var y = axis.y;
	    var z = axis.z;
	    var c = Math.cos(angle);
	    var s = Math.sin(angle);
	    var t = 1 - c;
	    var tx = t * x;
	    var ty = t * y;
	    var m = this.data;
	    m[0] = tx * x + c;
	    m[1] = tx * y + s * z;
	    m[2] = tx * z - s * y;
	    m[3] = 0;
	    m[4] = tx * y - s * z;
	    m[5] = ty * y + c;
	    m[6] = ty * z + s * x;
	    m[7] = 0;
	    m[8] = tx * z + s * y;
	    m[9] = ty * z - x * s;
	    m[10] = t * z * z + c;
	    m[11] = 0;
	    m[12] = 0;
	    m[13] = 0;
	    m[14] = 0;
	    m[15] = 1;
	    return this;
	  }

	  /**
	   * Sets the specified matrix to a translation matrix.
	   *
	   * @param {number} x - The x-component of the translation.
	   * @param {number} y - The y-component of the translation.
	   * @param {number} z - The z-component of the translation.
	   * @returns {Mat4} Self for chaining.
	   * @example
	   * // Create a 4x4 translation matrix
	   * const tm = new pc.Mat4().setTranslate(10, 10, 10);
	   * @ignore
	   */;
	  _proto.setTranslate = function setTranslate(x, y, z) {
	    var m = this.data;
	    m[0] = 1;
	    m[1] = 0;
	    m[2] = 0;
	    m[3] = 0;
	    m[4] = 0;
	    m[5] = 1;
	    m[6] = 0;
	    m[7] = 0;
	    m[8] = 0;
	    m[9] = 0;
	    m[10] = 1;
	    m[11] = 0;
	    m[12] = x;
	    m[13] = y;
	    m[14] = z;
	    m[15] = 1;
	    return this;
	  }

	  /**
	   * Sets the specified matrix to a scale matrix.
	   *
	   * @param {number} x - The x-component of the scale.
	   * @param {number} y - The y-component of the scale.
	   * @param {number} z - The z-component of the scale.
	   * @returns {Mat4} Self for chaining.
	   * @example
	   * // Create a 4x4 scale matrix
	   * const sm = new pc.Mat4().setScale(10, 10, 10);
	   * @ignore
	   */;
	  _proto.setScale = function setScale(x, y, z) {
	    var m = this.data;
	    m[0] = x;
	    m[1] = 0;
	    m[2] = 0;
	    m[3] = 0;
	    m[4] = 0;
	    m[5] = y;
	    m[6] = 0;
	    m[7] = 0;
	    m[8] = 0;
	    m[9] = 0;
	    m[10] = z;
	    m[11] = 0;
	    m[12] = 0;
	    m[13] = 0;
	    m[14] = 0;
	    m[15] = 1;
	    return this;
	  }

	  /**
	   * Sets the specified matrix to a matrix transforming a normalized view volume (in range of
	   * -1 .. 1) to their position inside a viewport (in range of 0 .. 1). This encapsulates a
	   * scaling to the size of the viewport and a translation to the position of the viewport.
	   *
	   * @param {number} x - The x-component of the position of the viewport (in 0..1 range).
	   * @param {number} y - The y-component of the position of the viewport (in 0..1 range).
	   * @param {number} width - The width of the viewport (in 0..1 range).
	   * @param {number} height - The height of the viewport (in 0..1 range).
	   * @returns {Mat4} Self for chaining.
	   * @example
	   * // Create a 4x4 viewport matrix which scales normalized view volume to full texture viewport
	   * const vm = new pc.Mat4().setViewport(0, 0, 1, 1);
	   * @ignore
	   */;
	  _proto.setViewport = function setViewport(x, y, width, height) {
	    var m = this.data;
	    m[0] = width * 0.5;
	    m[1] = 0;
	    m[2] = 0;
	    m[3] = 0;
	    m[4] = 0;
	    m[5] = height * 0.5;
	    m[6] = 0;
	    m[7] = 0;
	    m[8] = 0;
	    m[9] = 0;
	    m[10] = 0.5;
	    m[11] = 0;
	    m[12] = x + width * 0.5;
	    m[13] = y + height * 0.5;
	    m[14] = 0.5;
	    m[15] = 1;
	    return this;
	  }

	  /**
	   * Sets the matrix to a reflection matrix, which can be used as a mirror transformation by the
	   * plane.
	   *
	   * @param {Vec3} normal - The normal of the plane to reflect by.
	   * @param {number} distance - The distance of plane to reflect by.
	   * @returns {Mat4} Self for chaining.
	   */;
	  _proto.setReflection = function setReflection(normal, distance) {
	    var a = normal.x;
	    var b = normal.y;
	    var c = normal.z;
	    var data = this.data;
	    data[0] = 1.0 - 2 * a * a;
	    data[1] = -2 * a * b;
	    data[2] = -2 * a * c;
	    data[3] = 0;
	    data[4] = -2 * a * b;
	    data[5] = 1.0 - 2 * b * b;
	    data[6] = -2 * b * c;
	    data[7] = 0;
	    data[8] = -2 * a * c;
	    data[9] = -2 * b * c;
	    data[10] = 1.0 - 2 * c * c;
	    data[11] = 0;
	    data[12] = -2 * a * distance;
	    data[13] = -2 * b * distance;
	    data[14] = -2 * c * distance;
	    data[15] = 1;
	    return this;
	  }

	  /**
	   * Sets the matrix to the inverse of a source matrix.
	   *
	   * @param {Mat4} [src] - The matrix to invert. If not set, the matrix is inverted in-place.
	   * @returns {Mat4} Self for chaining.
	   * @example
	   * // Create a 4x4 rotation matrix of 180 degrees around the y-axis
	   * const rot = new pc.Mat4().setFromAxisAngle(pc.Vec3.UP, 180);
	   *
	   * // Invert in place
	   * rot.invert();
	   */;
	  _proto.invert = function invert(src) {
	    if (src === void 0) {
	      src = this;
	    }
	    var s = src.data;
	    var a00 = s[0];
	    var a01 = s[1];
	    var a02 = s[2];
	    var a03 = s[3];
	    var a10 = s[4];
	    var a11 = s[5];
	    var a12 = s[6];
	    var a13 = s[7];
	    var a20 = s[8];
	    var a21 = s[9];
	    var a22 = s[10];
	    var a23 = s[11];
	    var a30 = s[12];
	    var a31 = s[13];
	    var a32 = s[14];
	    var a33 = s[15];
	    var b00 = a00 * a11 - a01 * a10;
	    var b01 = a00 * a12 - a02 * a10;
	    var b02 = a00 * a13 - a03 * a10;
	    var b03 = a01 * a12 - a02 * a11;
	    var b04 = a01 * a13 - a03 * a11;
	    var b05 = a02 * a13 - a03 * a12;
	    var b06 = a20 * a31 - a21 * a30;
	    var b07 = a20 * a32 - a22 * a30;
	    var b08 = a20 * a33 - a23 * a30;
	    var b09 = a21 * a32 - a22 * a31;
	    var b10 = a21 * a33 - a23 * a31;
	    var b11 = a22 * a33 - a23 * a32;
	    var det = b00 * b11 - b01 * b10 + b02 * b09 + b03 * b08 - b04 * b07 + b05 * b06;
	    if (det === 0) {
	      this.setIdentity();
	    } else {
	      var invDet = 1 / det;
	      var t = this.data;
	      t[0] = (a11 * b11 - a12 * b10 + a13 * b09) * invDet;
	      t[1] = (-a01 * b11 + a02 * b10 - a03 * b09) * invDet;
	      t[2] = (a31 * b05 - a32 * b04 + a33 * b03) * invDet;
	      t[3] = (-a21 * b05 + a22 * b04 - a23 * b03) * invDet;
	      t[4] = (-a10 * b11 + a12 * b08 - a13 * b07) * invDet;
	      t[5] = (a00 * b11 - a02 * b08 + a03 * b07) * invDet;
	      t[6] = (-a30 * b05 + a32 * b02 - a33 * b01) * invDet;
	      t[7] = (a20 * b05 - a22 * b02 + a23 * b01) * invDet;
	      t[8] = (a10 * b10 - a11 * b08 + a13 * b06) * invDet;
	      t[9] = (-a00 * b10 + a01 * b08 - a03 * b06) * invDet;
	      t[10] = (a30 * b04 - a31 * b02 + a33 * b00) * invDet;
	      t[11] = (-a20 * b04 + a21 * b02 - a23 * b00) * invDet;
	      t[12] = (-a10 * b09 + a11 * b07 - a12 * b06) * invDet;
	      t[13] = (a00 * b09 - a01 * b07 + a02 * b06) * invDet;
	      t[14] = (-a30 * b03 + a31 * b01 - a32 * b00) * invDet;
	      t[15] = (a20 * b03 - a21 * b01 + a22 * b00) * invDet;
	    }
	    return this;
	  }

	  /**
	   * Sets matrix data from an array.
	   *
	   * @param {number[]} src - Source array. Must have 16 values.
	   * @returns {Mat4} Self for chaining.
	   */;
	  _proto.set = function set(src) {
	    var dst = this.data;
	    dst[0] = src[0];
	    dst[1] = src[1];
	    dst[2] = src[2];
	    dst[3] = src[3];
	    dst[4] = src[4];
	    dst[5] = src[5];
	    dst[6] = src[6];
	    dst[7] = src[7];
	    dst[8] = src[8];
	    dst[9] = src[9];
	    dst[10] = src[10];
	    dst[11] = src[11];
	    dst[12] = src[12];
	    dst[13] = src[13];
	    dst[14] = src[14];
	    dst[15] = src[15];
	    return this;
	  }

	  /**
	   * Sets the specified matrix to the identity matrix.
	   *
	   * @returns {Mat4} Self for chaining.
	   * @example
	   * m.setIdentity();
	   * console.log("The matrix is " + (m.isIdentity() ? "identity" : "not identity"));
	   */;
	  _proto.setIdentity = function setIdentity() {
	    var m = this.data;
	    m[0] = 1;
	    m[1] = 0;
	    m[2] = 0;
	    m[3] = 0;
	    m[4] = 0;
	    m[5] = 1;
	    m[6] = 0;
	    m[7] = 0;
	    m[8] = 0;
	    m[9] = 0;
	    m[10] = 1;
	    m[11] = 0;
	    m[12] = 0;
	    m[13] = 0;
	    m[14] = 0;
	    m[15] = 1;
	    return this;
	  }

	  /**
	   * Sets the specified matrix to the concatenation of a translation, a quaternion rotation and a
	   * scale.
	   *
	   * @param {Vec3} t - A 3-d vector translation.
	   * @param {new Function("modulePath", "return import(modulePath)")('./quat.js').Quat} r - A quaternion rotation.
	   * @param {Vec3} s - A 3-d vector scale.
	   * @returns {Mat4} Self for chaining.
	   * @example
	   * const t = new pc.Vec3(10, 20, 30);
	   * const r = new pc.Quat();
	   * const s = new pc.Vec3(2, 2, 2);
	   *
	   * const m = new pc.Mat4();
	   * m.setTRS(t, r, s);
	   */;
	  _proto.setTRS = function setTRS(t, r, s) {
	    var qx = r.x;
	    var qy = r.y;
	    var qz = r.z;
	    var qw = r.w;
	    var sx = s.x;
	    var sy = s.y;
	    var sz = s.z;
	    var x2 = qx + qx;
	    var y2 = qy + qy;
	    var z2 = qz + qz;
	    var xx = qx * x2;
	    var xy = qx * y2;
	    var xz = qx * z2;
	    var yy = qy * y2;
	    var yz = qy * z2;
	    var zz = qz * z2;
	    var wx = qw * x2;
	    var wy = qw * y2;
	    var wz = qw * z2;
	    var m = this.data;
	    m[0] = (1 - (yy + zz)) * sx;
	    m[1] = (xy + wz) * sx;
	    m[2] = (xz - wy) * sx;
	    m[3] = 0;
	    m[4] = (xy - wz) * sy;
	    m[5] = (1 - (xx + zz)) * sy;
	    m[6] = (yz + wx) * sy;
	    m[7] = 0;
	    m[8] = (xz + wy) * sz;
	    m[9] = (yz - wx) * sz;
	    m[10] = (1 - (xx + yy)) * sz;
	    m[11] = 0;
	    m[12] = t.x;
	    m[13] = t.y;
	    m[14] = t.z;
	    m[15] = 1;
	    return this;
	  }

	  /**
	   * Sets the matrix to the transpose of a source matrix.
	   *
	   * @param {Mat4} [src] - The matrix to transpose. If not set, the matrix is transposed in-place.
	   * @returns {Mat4} Self for chaining.
	   * @example
	   * const m = new pc.Mat4();
	   *
	   * // Transpose in place
	   * m.transpose();
	   */;
	  _proto.transpose = function transpose(src) {
	    if (src === void 0) {
	      src = this;
	    }
	    var s = src.data;
	    var t = this.data;
	    if (s === t) {
	      var tmp;
	      tmp = s[1];
	      t[1] = s[4];
	      t[4] = tmp;
	      tmp = s[2];
	      t[2] = s[8];
	      t[8] = tmp;
	      tmp = s[3];
	      t[3] = s[12];
	      t[12] = tmp;
	      tmp = s[6];
	      t[6] = s[9];
	      t[9] = tmp;
	      tmp = s[7];
	      t[7] = s[13];
	      t[13] = tmp;
	      tmp = s[11];
	      t[11] = s[14];
	      t[14] = tmp;
	    } else {
	      t[0] = s[0];
	      t[1] = s[4];
	      t[2] = s[8];
	      t[3] = s[12];
	      t[4] = s[1];
	      t[5] = s[5];
	      t[6] = s[9];
	      t[7] = s[13];
	      t[8] = s[2];
	      t[9] = s[6];
	      t[10] = s[10];
	      t[11] = s[14];
	      t[12] = s[3];
	      t[13] = s[7];
	      t[14] = s[11];
	      t[15] = s[15];
	    }
	    return this;
	  }

	  /**
	   * Extracts the translational component from the specified 4x4 matrix.
	   *
	   * @param {Vec3} [t] - The vector to receive the translation of the matrix.
	   * @returns {Vec3} The translation of the specified 4x4 matrix.
	   * @example
	   * // Create a 4x4 matrix
	   * const m = new pc.Mat4();
	   *
	   * // Query the translation component
	   * const t = new pc.Vec3();
	   * m.getTranslation(t);
	   */;
	  _proto.getTranslation = function getTranslation(t) {
	    if (t === void 0) {
	      t = new Vec3();
	    }
	    return t.set(this.data[12], this.data[13], this.data[14]);
	  }

	  /**
	   * Extracts the x-axis from the specified 4x4 matrix.
	   *
	   * @param {Vec3} [x] - The vector to receive the x axis of the matrix.
	   * @returns {Vec3} The x-axis of the specified 4x4 matrix.
	   * @example
	   * // Create a 4x4 matrix
	   * const m = new pc.Mat4();
	   *
	   * // Query the x-axis component
	   * const x = new pc.Vec3();
	   * m.getX(x);
	   */;
	  _proto.getX = function getX(x) {
	    if (x === void 0) {
	      x = new Vec3();
	    }
	    return x.set(this.data[0], this.data[1], this.data[2]);
	  }

	  /**
	   * Extracts the y-axis from the specified 4x4 matrix.
	   *
	   * @param {Vec3} [y] - The vector to receive the y axis of the matrix.
	   * @returns {Vec3} The y-axis of the specified 4x4 matrix.
	   * @example
	   * // Create a 4x4 matrix
	   * const m = new pc.Mat4();
	   *
	   * // Query the y-axis component
	   * const y = new pc.Vec3();
	   * m.getY(y);
	   */;
	  _proto.getY = function getY(y) {
	    if (y === void 0) {
	      y = new Vec3();
	    }
	    return y.set(this.data[4], this.data[5], this.data[6]);
	  }

	  /**
	   * Extracts the z-axis from the specified 4x4 matrix.
	   *
	   * @param {Vec3} [z] - The vector to receive the z axis of the matrix.
	   * @returns {Vec3} The z-axis of the specified 4x4 matrix.
	   * @example
	   * // Create a 4x4 matrix
	   * const m = new pc.Mat4();
	   *
	   * // Query the z-axis component
	   * const z = new pc.Vec3();
	   * m.getZ(z);
	   */;
	  _proto.getZ = function getZ(z) {
	    if (z === void 0) {
	      z = new Vec3();
	    }
	    return z.set(this.data[8], this.data[9], this.data[10]);
	  }

	  /**
	   * Extracts the scale component from the specified 4x4 matrix.
	   *
	   * @param {Vec3} [scale] - Vector to receive the scale.
	   * @returns {Vec3} The scale in X, Y and Z of the specified 4x4 matrix.
	   * @example
	   * // Query the scale component
	   * const scale = m.getScale();
	   */;
	  _proto.getScale = function getScale(scale) {
	    if (scale === void 0) {
	      scale = new Vec3();
	    }
	    this.getX(x);
	    this.getY(y);
	    this.getZ(z);
	    scale.set(x.length(), y.length(), z.length());
	    return scale;
	  }

	  /**
	   * -1 if the the matrix has an odd number of negative scales (mirrored); 1 otherwise.
	   *
	   * @type {number}
	   * @ignore
	   */;
	  /**
	   * Sets the specified matrix to a rotation matrix defined by Euler angles. The Euler angles are
	   * specified in XYZ order and in degrees.
	   *
	   * @param {number} ex - Angle to rotate around X axis in degrees.
	   * @param {number} ey - Angle to rotate around Y axis in degrees.
	   * @param {number} ez - Angle to rotate around Z axis in degrees.
	   * @returns {Mat4} Self for chaining.
	   * @example
	   * const m = new pc.Mat4();
	   * m.setFromEulerAngles(45, 90, 180);
	   */
	  _proto.setFromEulerAngles = function setFromEulerAngles(ex, ey, ez) {
	    // http://en.wikipedia.org/wiki/Rotation_matrix#Conversion_from_and_to_axis-angle
	    // The 3D space is right-handed, so the rotation around each axis will be counterclockwise
	    // for an observer placed so that the axis goes in his or her direction (Right-hand rule).
	    ex *= math.DEG_TO_RAD;
	    ey *= math.DEG_TO_RAD;
	    ez *= math.DEG_TO_RAD;

	    // Solution taken from http://en.wikipedia.org/wiki/Euler_angles#Matrix_orientation
	    var s1 = Math.sin(-ex);
	    var c1 = Math.cos(-ex);
	    var s2 = Math.sin(-ey);
	    var c2 = Math.cos(-ey);
	    var s3 = Math.sin(-ez);
	    var c3 = Math.cos(-ez);
	    var m = this.data;

	    // Set rotation elements
	    m[0] = c2 * c3;
	    m[1] = -c2 * s3;
	    m[2] = s2;
	    m[3] = 0;
	    m[4] = c1 * s3 + c3 * s1 * s2;
	    m[5] = c1 * c3 - s1 * s2 * s3;
	    m[6] = -c2 * s1;
	    m[7] = 0;
	    m[8] = s1 * s3 - c1 * c3 * s2;
	    m[9] = c3 * s1 + c1 * s2 * s3;
	    m[10] = c1 * c2;
	    m[11] = 0;
	    m[12] = 0;
	    m[13] = 0;
	    m[14] = 0;
	    m[15] = 1;
	    return this;
	  }

	  /**
	   * Extracts the Euler angles equivalent to the rotational portion of the specified matrix. The
	   * returned Euler angles are in XYZ order an in degrees.
	   *
	   * @param {Vec3} [eulers] - A 3-d vector to receive the Euler angles.
	   * @returns {Vec3} A 3-d vector containing the Euler angles.
	   * @example
	   * // Create a 4x4 rotation matrix of 45 degrees around the y-axis
	   * const m = new pc.Mat4().setFromAxisAngle(pc.Vec3.UP, 45);
	   *
	   * const eulers = m.getEulerAngles();
	   */;
	  _proto.getEulerAngles = function getEulerAngles(eulers) {
	    if (eulers === void 0) {
	      eulers = new Vec3();
	    }
	    this.getScale(scale);
	    var sx = scale.x;
	    var sy = scale.y;
	    var sz = scale.z;
	    if (sx === 0 || sy === 0 || sz === 0) return eulers.set(0, 0, 0);
	    var m = this.data;
	    var y = Math.asin(-m[2] / sx);
	    var halfPi = Math.PI * 0.5;
	    var x, z;
	    if (y < halfPi) {
	      if (y > -halfPi) {
	        x = Math.atan2(m[6] / sy, m[10] / sz);
	        z = Math.atan2(m[1] / sx, m[0] / sx);
	      } else {
	        // Not a unique solution
	        z = 0;
	        x = -Math.atan2(m[4] / sy, m[5] / sy);
	      }
	    } else {
	      // Not a unique solution
	      z = 0;
	      x = Math.atan2(m[4] / sy, m[5] / sy);
	    }
	    return eulers.set(x, y, z).mulScalar(math.RAD_TO_DEG);
	  }

	  /**
	   * Converts the specified matrix to string form.
	   *
	   * @returns {string} The matrix in string form.
	   * @example
	   * const m = new pc.Mat4();
	   * // Outputs [1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1]
	   * console.log(m.toString());
	   */;
	  _proto.toString = function toString() {
	    return '[' + this.data.join(', ') + ']';
	  }

	  /**
	   * A constant matrix set to the identity.
	   *
	   * @type {Mat4}
	   * @readonly
	   */;
	  _createClass(Mat4, [{
	    key: "scaleSign",
	    get: function get() {
	      this.getX(x);
	      this.getY(y);
	      this.getZ(z);
	      x.cross(x, y);
	      return x.dot(z) < 0 ? -1 : 1;
	    }
	  }]);
	  return Mat4;
	}();
	_class$5 = Mat4;
	Mat4.IDENTITY = Object.freeze(new _class$5());
	/**
	 * A constant matrix with all elements set to 0.
	 *
	 * @type {Mat4}
	 * @readonly
	 */
	Mat4.ZERO = Object.freeze(new _class$5().set([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]));

	var _class$4;

	/**
	 * A quaternion.
	 *
	 * @category Math
	 */
	var Quat = /*#__PURE__*/function () {
	  /**
	   * Create a new Quat instance.
	   *
	   * @param {number|number[]} [x] - The quaternion's x component. Defaults to 0. If x is an array
	   * of length 4, the array will be used to populate all components.
	   * @param {number} [y] - The quaternion's y component. Defaults to 0.
	   * @param {number} [z] - The quaternion's z component. Defaults to 0.
	   * @param {number} [w] - The quaternion's w component. Defaults to 1.
	   */
	  function Quat(x, y, z, w) {
	    if (x === void 0) {
	      x = 0;
	    }
	    if (y === void 0) {
	      y = 0;
	    }
	    if (z === void 0) {
	      z = 0;
	    }
	    if (w === void 0) {
	      w = 1;
	    }
	    /**
	     * The x component of the quaternion.
	     *
	     * @type {number}
	     */
	    this.x = void 0;
	    /**
	     * The y component of the quaternion.
	     *
	     * @type {number}
	     */
	    this.y = void 0;
	    /**
	     * The z component of the quaternion.
	     *
	     * @type {number}
	     */
	    this.z = void 0;
	    /**
	     * The w component of the quaternion.
	     *
	     * @type {number}
	     */
	    this.w = void 0;
	    if (x.length === 4) {
	      this.x = x[0];
	      this.y = x[1];
	      this.z = x[2];
	      this.w = x[3];
	    } else {
	      this.x = x;
	      this.y = y;
	      this.z = z;
	      this.w = w;
	    }
	  }

	  /**
	   * Returns an identical copy of the specified quaternion.
	   *
	   * @returns {this} A quaternion containing the result of the cloning.
	   * @example
	   * const q = new pc.Quat(-0.11, -0.15, -0.46, 0.87);
	   * const qclone = q.clone();
	   *
	   * console.log("The result of the cloning is: " + q.toString());
	   */
	  var _proto = Quat.prototype;
	  _proto.clone = function clone() {
	    /** @type {this} */
	    var cstr = this.constructor;
	    return new cstr(this.x, this.y, this.z, this.w);
	  };
	  _proto.conjugate = function conjugate(src) {
	    if (src === void 0) {
	      src = this;
	    }
	    this.x = src.x * -1;
	    this.y = src.y * -1;
	    this.z = src.z * -1;
	    this.w = src.w;
	    return this;
	  }

	  /**
	   * Copies the contents of a source quaternion to a destination quaternion.
	   *
	   * @param {Quat} rhs - The quaternion to be copied.
	   * @returns {Quat} Self for chaining.
	   * @example
	   * const src = new pc.Quat();
	   * const dst = new pc.Quat();
	   * dst.copy(src, src);
	   * console.log("The two quaternions are " + (src.equals(dst) ? "equal" : "different"));
	   */;
	  _proto.copy = function copy(rhs) {
	    this.x = rhs.x;
	    this.y = rhs.y;
	    this.z = rhs.z;
	    this.w = rhs.w;
	    return this;
	  }

	  /**
	   * Reports whether two quaternions are equal.
	   *
	   * @param {Quat} rhs - The quaternion to be compared against.
	   * @returns {boolean} True if the quaternions are equal and false otherwise.
	   * @example
	   * const a = new pc.Quat();
	   * const b = new pc.Quat();
	   * console.log("The two quaternions are " + (a.equals(b) ? "equal" : "different"));
	   */;
	  _proto.equals = function equals(rhs) {
	    return this.x === rhs.x && this.y === rhs.y && this.z === rhs.z && this.w === rhs.w;
	  }

	  /**
	   * Reports whether two quaternions are equal using an absolute error tolerance.
	   *
	   * @param {Quat} rhs - The quaternion to be compared against.
	   * @param {number} [epsilon] - The maximum difference between each component of the two
	   * quaternions. Defaults to 1e-6.
	   * @returns {boolean} True if the quaternions are equal and false otherwise.
	   * @example
	   * const a = new pc.Quat();
	   * const b = new pc.Quat();
	   * console.log("The two quaternions are approximately " + (a.equalsApprox(b, 1e-9) ? "equal" : "different"));
	   */;
	  _proto.equalsApprox = function equalsApprox(rhs, epsilon) {
	    if (epsilon === void 0) {
	      epsilon = 1e-6;
	    }
	    return Math.abs(this.x - rhs.x) < epsilon && Math.abs(this.y - rhs.y) < epsilon && Math.abs(this.z - rhs.z) < epsilon && Math.abs(this.w - rhs.w) < epsilon;
	  }

	  /**
	   * Gets the rotation axis and angle for a given quaternion. If a quaternion is created with
	   * `setFromAxisAngle`, this method will return the same values as provided in the original
	   * parameter list OR functionally equivalent values.
	   *
	   * @param {Vec3} axis - The 3-dimensional vector to receive the axis of rotation.
	   * @returns {number} Angle, in degrees, of the rotation.
	   * @example
	   * const q = new pc.Quat();
	   * q.setFromAxisAngle(new pc.Vec3(0, 1, 0), 90);
	   * const v = new pc.Vec3();
	   * const angle = q.getAxisAngle(v);
	   * // Outputs 90
	   * console.log(angle);
	   * // Outputs [0, 1, 0]
	   * console.log(v.toString());
	   */;
	  _proto.getAxisAngle = function getAxisAngle(axis) {
	    var rad = Math.acos(this.w) * 2;
	    var s = Math.sin(rad / 2);
	    if (s !== 0) {
	      axis.x = this.x / s;
	      axis.y = this.y / s;
	      axis.z = this.z / s;
	      if (axis.x < 0 || axis.y < 0 || axis.z < 0) {
	        // Flip the sign
	        axis.x *= -1;
	        axis.y *= -1;
	        axis.z *= -1;
	        rad *= -1;
	      }
	    } else {
	      // If s is zero, return any axis (no rotation - axis does not matter)
	      axis.x = 1;
	      axis.y = 0;
	      axis.z = 0;
	    }
	    return rad * math.RAD_TO_DEG;
	  }

	  /**
	   * Converts the supplied quaternion to Euler angles.
	   *
	   * @param {Vec3} [eulers] - The 3-dimensional vector to receive the Euler angles.
	   * @returns {Vec3} The 3-dimensional vector holding the Euler angles that
	   * correspond to the supplied quaternion.
	   */;
	  _proto.getEulerAngles = function getEulerAngles(eulers) {
	    if (eulers === void 0) {
	      eulers = new Vec3();
	    }
	    var x, y, z;
	    var qx = this.x;
	    var qy = this.y;
	    var qz = this.z;
	    var qw = this.w;
	    var a2 = 2 * (qw * qy - qx * qz);
	    if (a2 <= -0.99999) {
	      x = 2 * Math.atan2(qx, qw);
	      y = -Math.PI / 2;
	      z = 0;
	    } else if (a2 >= 0.99999) {
	      x = 2 * Math.atan2(qx, qw);
	      y = Math.PI / 2;
	      z = 0;
	    } else {
	      x = Math.atan2(2 * (qw * qx + qy * qz), 1 - 2 * (qx * qx + qy * qy));
	      y = Math.asin(a2);
	      z = Math.atan2(2 * (qw * qz + qx * qy), 1 - 2 * (qy * qy + qz * qz));
	    }
	    return eulers.set(x, y, z).mulScalar(math.RAD_TO_DEG);
	  }

	  /**
	   * Generates the inverse of the specified quaternion.
	   *
	   * @param {Quat} [src] - The quaternion to invert. If not set, the operation is done in place.
	   * @returns {Quat} Self for chaining.
	   * @example
	   * // Create a quaternion rotated 180 degrees around the y-axis
	   * const rot = new pc.Quat().setFromEulerAngles(0, 180, 0);
	   *
	   * // Invert in place
	   * rot.invert();
	   */;
	  _proto.invert = function invert(src) {
	    if (src === void 0) {
	      src = this;
	    }
	    return this.conjugate(src).normalize();
	  }

	  /**
	   * Returns the magnitude of the specified quaternion.
	   *
	   * @returns {number} The magnitude of the specified quaternion.
	   * @example
	   * const q = new pc.Quat(0, 0, 0, 5);
	   * const len = q.length();
	   * // Outputs 5
	   * console.log("The length of the quaternion is: " + len);
	   */;
	  _proto.length = function length() {
	    return Math.sqrt(this.x * this.x + this.y * this.y + this.z * this.z + this.w * this.w);
	  }

	  /**
	   * Returns the magnitude squared of the specified quaternion.
	   *
	   * @returns {number} The magnitude of the specified quaternion.
	   * @example
	   * const q = new pc.Quat(3, 4, 0);
	   * const lenSq = q.lengthSq();
	   * // Outputs 25
	   * console.log("The length squared of the quaternion is: " + lenSq);
	   */;
	  _proto.lengthSq = function lengthSq() {
	    return this.x * this.x + this.y * this.y + this.z * this.z + this.w * this.w;
	  }

	  /**
	   * Returns the result of multiplying the specified quaternions together.
	   *
	   * @param {Quat} rhs - The quaternion used as the second multiplicand of the operation.
	   * @returns {Quat} Self for chaining.
	   * @example
	   * const a = new pc.Quat().setFromEulerAngles(0, 30, 0);
	   * const b = new pc.Quat().setFromEulerAngles(0, 60, 0);
	   *
	   * // a becomes a 90 degree rotation around the Y axis
	   * // In other words, a = a * b
	   * a.mul(b);
	   *
	   * console.log("The result of the multiplication is: " + a.toString());
	   */;
	  _proto.mul = function mul(rhs) {
	    var q1x = this.x;
	    var q1y = this.y;
	    var q1z = this.z;
	    var q1w = this.w;
	    var q2x = rhs.x;
	    var q2y = rhs.y;
	    var q2z = rhs.z;
	    var q2w = rhs.w;
	    this.x = q1w * q2x + q1x * q2w + q1y * q2z - q1z * q2y;
	    this.y = q1w * q2y + q1y * q2w + q1z * q2x - q1x * q2z;
	    this.z = q1w * q2z + q1z * q2w + q1x * q2y - q1y * q2x;
	    this.w = q1w * q2w - q1x * q2x - q1y * q2y - q1z * q2z;
	    return this;
	  }

	  /**
	   * Returns the result of multiplying the specified quaternions together.
	   *
	   * @param {Quat} lhs - The quaternion used as the first multiplicand of the operation.
	   * @param {Quat} rhs - The quaternion used as the second multiplicand of the operation.
	   * @returns {Quat} Self for chaining.
	   * @example
	   * const a = new pc.Quat().setFromEulerAngles(0, 30, 0);
	   * const b = new pc.Quat().setFromEulerAngles(0, 60, 0);
	   * const r = new pc.Quat();
	   *
	   * // r is set to a 90 degree rotation around the Y axis
	   * // In other words, r = a * b
	   * r.mul2(a, b);
	   *
	   * console.log("The result of the multiplication is: " + r.toString());
	   */;
	  _proto.mul2 = function mul2(lhs, rhs) {
	    var q1x = lhs.x;
	    var q1y = lhs.y;
	    var q1z = lhs.z;
	    var q1w = lhs.w;
	    var q2x = rhs.x;
	    var q2y = rhs.y;
	    var q2z = rhs.z;
	    var q2w = rhs.w;
	    this.x = q1w * q2x + q1x * q2w + q1y * q2z - q1z * q2y;
	    this.y = q1w * q2y + q1y * q2w + q1z * q2x - q1x * q2z;
	    this.z = q1w * q2z + q1z * q2w + q1x * q2y - q1y * q2x;
	    this.w = q1w * q2w - q1x * q2x - q1y * q2y - q1z * q2z;
	    return this;
	  }

	  /**
	   * Returns the specified quaternion converted in place to a unit quaternion.
	   *
	   * @param {Quat} [src] - The quaternion to normalize. If not set, the operation is done in place.
	   * @returns {Quat} The result of the normalization.
	   * @example
	   * const v = new pc.Quat(0, 0, 0, 5);
	   *
	   * v.normalize();
	   *
	   * // Outputs 0, 0, 0, 1
	   * console.log("The result of the vector normalization is: " + v.toString());
	   */;
	  _proto.normalize = function normalize(src) {
	    if (src === void 0) {
	      src = this;
	    }
	    var len = src.length();
	    if (len === 0) {
	      this.x = this.y = this.z = 0;
	      this.w = 1;
	    } else {
	      len = 1 / len;
	      this.x = src.x * len;
	      this.y = src.y * len;
	      this.z = src.z * len;
	      this.w = src.w * len;
	    }
	    return this;
	  }

	  /**
	   * Sets the specified quaternion to the supplied numerical values.
	   *
	   * @param {number} x - The x component of the quaternion.
	   * @param {number} y - The y component of the quaternion.
	   * @param {number} z - The z component of the quaternion.
	   * @param {number} w - The w component of the quaternion.
	   * @returns {Quat} Self for chaining.
	   * @example
	   * const q = new pc.Quat();
	   * q.set(1, 0, 0, 0);
	   *
	   * // Outputs 1, 0, 0, 0
	   * console.log("The result of the vector set is: " + q.toString());
	   */;
	  _proto.set = function set(x, y, z, w) {
	    this.x = x;
	    this.y = y;
	    this.z = z;
	    this.w = w;
	    return this;
	  }

	  /**
	   * Sets a quaternion from an angular rotation around an axis.
	   *
	   * @param {Vec3} axis - World space axis around which to rotate.
	   * @param {number} angle - Angle to rotate around the given axis in degrees.
	   * @returns {Quat} Self for chaining.
	   * @example
	   * const q = new pc.Quat();
	   * q.setFromAxisAngle(pc.Vec3.UP, 90);
	   */;
	  _proto.setFromAxisAngle = function setFromAxisAngle(axis, angle) {
	    angle *= 0.5 * math.DEG_TO_RAD;
	    var sa = Math.sin(angle);
	    var ca = Math.cos(angle);
	    this.x = sa * axis.x;
	    this.y = sa * axis.y;
	    this.z = sa * axis.z;
	    this.w = ca;
	    return this;
	  }

	  /**
	   * Sets a quaternion from Euler angles specified in XYZ order.
	   *
	   * @param {number|Vec3} ex - Angle to rotate around X axis in degrees. If ex is a Vec3, the
	   * three angles will be read from it instead.
	   * @param {number} [ey] - Angle to rotate around Y axis in degrees.
	   * @param {number} [ez] - Angle to rotate around Z axis in degrees.
	   * @returns {Quat} Self for chaining.
	   * @example
	   * // Create a quaternion from 3 euler angles
	   * const q = new pc.Quat();
	   * q.setFromEulerAngles(45, 90, 180);
	   *
	   * // Create the same quaternion from a vector containing the same 3 euler angles
	   * const v = new pc.Vec3(45, 90, 180);
	   * const r = new pc.Quat();
	   * r.setFromEulerAngles(v);
	   */;
	  _proto.setFromEulerAngles = function setFromEulerAngles(ex, ey, ez) {
	    if (ex instanceof Vec3) {
	      var vec = ex;
	      ex = vec.x;
	      ey = vec.y;
	      ez = vec.z;
	    }
	    var halfToRad = 0.5 * math.DEG_TO_RAD;
	    ex *= halfToRad;
	    ey *= halfToRad;
	    ez *= halfToRad;
	    var sx = Math.sin(ex);
	    var cx = Math.cos(ex);
	    var sy = Math.sin(ey);
	    var cy = Math.cos(ey);
	    var sz = Math.sin(ez);
	    var cz = Math.cos(ez);
	    this.x = sx * cy * cz - cx * sy * sz;
	    this.y = cx * sy * cz + sx * cy * sz;
	    this.z = cx * cy * sz - sx * sy * cz;
	    this.w = cx * cy * cz + sx * sy * sz;
	    return this;
	  }

	  /**
	   * Converts the specified 4x4 matrix to a quaternion. Note that since a quaternion is purely a
	   * representation for orientation, only the translational part of the matrix is lost.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./mat4.js').Mat4} m - The 4x4 matrix to convert.
	   * @returns {Quat} Self for chaining.
	   * @example
	   * // Create a 4x4 rotation matrix of 180 degrees around the y-axis
	   * const rot = new pc.Mat4().setFromAxisAngle(pc.Vec3.UP, 180);
	   *
	   * // Convert to a quaternion
	   * const q = new pc.Quat().setFromMat4(rot);
	   */;
	  _proto.setFromMat4 = function setFromMat4(m) {
	    var m00, m01, m02, m10, m11, m12, m20, m21, m22, s, rs, lx, ly, lz;
	    m = m.data;

	    // Cache matrix values for super-speed
	    m00 = m[0];
	    m01 = m[1];
	    m02 = m[2];
	    m10 = m[4];
	    m11 = m[5];
	    m12 = m[6];
	    m20 = m[8];
	    m21 = m[9];
	    m22 = m[10];

	    // Remove the scale from the matrix
	    lx = m00 * m00 + m01 * m01 + m02 * m02;
	    if (lx === 0) return this;
	    lx = 1 / Math.sqrt(lx);
	    ly = m10 * m10 + m11 * m11 + m12 * m12;
	    if (ly === 0) return this;
	    ly = 1 / Math.sqrt(ly);
	    lz = m20 * m20 + m21 * m21 + m22 * m22;
	    if (lz === 0) return this;
	    lz = 1 / Math.sqrt(lz);
	    m00 *= lx;
	    m01 *= lx;
	    m02 *= lx;
	    m10 *= ly;
	    m11 *= ly;
	    m12 *= ly;
	    m20 *= lz;
	    m21 *= lz;
	    m22 *= lz;

	    // http://www.cs.ucr.edu/~vbz/resources/quatut.pdf

	    var tr = m00 + m11 + m22;
	    if (tr >= 0) {
	      s = Math.sqrt(tr + 1);
	      this.w = s * 0.5;
	      s = 0.5 / s;
	      this.x = (m12 - m21) * s;
	      this.y = (m20 - m02) * s;
	      this.z = (m01 - m10) * s;
	    } else {
	      if (m00 > m11) {
	        if (m00 > m22) {
	          // XDiagDomMatrix
	          rs = m00 - (m11 + m22) + 1;
	          rs = Math.sqrt(rs);
	          this.x = rs * 0.5;
	          rs = 0.5 / rs;
	          this.w = (m12 - m21) * rs;
	          this.y = (m01 + m10) * rs;
	          this.z = (m02 + m20) * rs;
	        } else {
	          // ZDiagDomMatrix
	          rs = m22 - (m00 + m11) + 1;
	          rs = Math.sqrt(rs);
	          this.z = rs * 0.5;
	          rs = 0.5 / rs;
	          this.w = (m01 - m10) * rs;
	          this.x = (m20 + m02) * rs;
	          this.y = (m21 + m12) * rs;
	        }
	      } else if (m11 > m22) {
	        // YDiagDomMatrix
	        rs = m11 - (m22 + m00) + 1;
	        rs = Math.sqrt(rs);
	        this.y = rs * 0.5;
	        rs = 0.5 / rs;
	        this.w = (m20 - m02) * rs;
	        this.z = (m12 + m21) * rs;
	        this.x = (m10 + m01) * rs;
	      } else {
	        // ZDiagDomMatrix
	        rs = m22 - (m00 + m11) + 1;
	        rs = Math.sqrt(rs);
	        this.z = rs * 0.5;
	        rs = 0.5 / rs;
	        this.w = (m01 - m10) * rs;
	        this.x = (m20 + m02) * rs;
	        this.y = (m21 + m12) * rs;
	      }
	    }
	    return this;
	  }

	  /**
	   * Set the quaternion that represents the shortest rotation from one direction to another.
	   *
	   * @param {Vec3} from - The direction to rotate from. It should be normalized.
	   * @param {Vec3} to - The direction to rotate to. It should be normalized.
	   * @returns {Quat} Self for chaining.
	   *
	   * {@link https://www.xarg.org/proof/quaternion-from-two-vectors/ Proof of correctness}
	   */;
	  _proto.setFromDirections = function setFromDirections(from, to) {
	    var dotProduct = 1 + from.dot(to);
	    if (dotProduct < Number.EPSILON) {
	      // the vectors point in opposite directions
	      // so we need to rotate 180 degrees around an arbitrary orthogonal axis
	      if (Math.abs(from.x) > Math.abs(from.y)) {
	        this.x = -from.z;
	        this.y = 0;
	        this.z = from.x;
	        this.w = 0;
	      } else {
	        this.x = 0;
	        this.y = -from.z;
	        this.z = from.y;
	        this.w = 0;
	      }
	    } else {
	      // cross product between the two vectors
	      this.x = from.y * to.z - from.z * to.y;
	      this.y = from.z * to.x - from.x * to.z;
	      this.z = from.x * to.y - from.y * to.x;
	      this.w = dotProduct;
	    }
	    return this.normalize();
	  }

	  /**
	   * Performs a spherical interpolation between two quaternions. The result of the interpolation
	   * is written to the quaternion calling the function.
	   *
	   * @param {Quat} lhs - The quaternion to interpolate from.
	   * @param {Quat} rhs - The quaternion to interpolate to.
	   * @param {number} alpha - The value controlling the interpolation in relation to the two input
	   * quaternions. The value is in the range 0 to 1, 0 generating q1, 1 generating q2 and anything
	   * in between generating a spherical interpolation between the two.
	   * @returns {Quat} Self for chaining.
	   * @example
	   * const q1 = new pc.Quat(-0.11, -0.15, -0.46, 0.87);
	   * const q2 = new pc.Quat(-0.21, -0.21, -0.67, 0.68);
	   *
	   * const result;
	   * result = new pc.Quat().slerp(q1, q2, 0);   // Return q1
	   * result = new pc.Quat().slerp(q1, q2, 0.5); // Return the midpoint interpolant
	   * result = new pc.Quat().slerp(q1, q2, 1);   // Return q2
	   */;
	  _proto.slerp = function slerp(lhs, rhs, alpha) {
	    // Algorithm sourced from:
	    // http://www.euclideanspace.com/maths/algebra/realNormedAlgebra/quaternions/slerp/
	    var lx = lhs.x;
	    var ly = lhs.y;
	    var lz = lhs.z;
	    var lw = lhs.w;
	    var rx = rhs.x;
	    var ry = rhs.y;
	    var rz = rhs.z;
	    var rw = rhs.w;

	    // Calculate angle between them.
	    var cosHalfTheta = lw * rw + lx * rx + ly * ry + lz * rz;
	    if (cosHalfTheta < 0) {
	      rw = -rw;
	      rx = -rx;
	      ry = -ry;
	      rz = -rz;
	      cosHalfTheta = -cosHalfTheta;
	    }

	    // If lhs == rhs or lhs == -rhs then theta == 0 and we can return lhs
	    if (Math.abs(cosHalfTheta) >= 1) {
	      this.w = lw;
	      this.x = lx;
	      this.y = ly;
	      this.z = lz;
	      return this;
	    }

	    // Calculate temporary values.
	    var halfTheta = Math.acos(cosHalfTheta);
	    var sinHalfTheta = Math.sqrt(1 - cosHalfTheta * cosHalfTheta);

	    // If theta = 180 degrees then result is not fully defined
	    // we could rotate around any axis normal to qa or qb
	    if (Math.abs(sinHalfTheta) < 0.001) {
	      this.w = lw * 0.5 + rw * 0.5;
	      this.x = lx * 0.5 + rx * 0.5;
	      this.y = ly * 0.5 + ry * 0.5;
	      this.z = lz * 0.5 + rz * 0.5;
	      return this;
	    }
	    var ratioA = Math.sin((1 - alpha) * halfTheta) / sinHalfTheta;
	    var ratioB = Math.sin(alpha * halfTheta) / sinHalfTheta;

	    // Calculate Quaternion.
	    this.w = lw * ratioA + rw * ratioB;
	    this.x = lx * ratioA + rx * ratioB;
	    this.y = ly * ratioA + ry * ratioB;
	    this.z = lz * ratioA + rz * ratioB;
	    return this;
	  }

	  /**
	   * Transforms a 3-dimensional vector by the specified quaternion.
	   *
	   * @param {Vec3} vec - The 3-dimensional vector to be transformed.
	   * @param {Vec3} [res] - An optional 3-dimensional vector to receive the result of the transformation.
	   * @returns {Vec3} The input vector v transformed by the current instance.
	   * @example
	   * // Create a 3-dimensional vector
	   * const v = new pc.Vec3(1, 2, 3);
	   *
	   * // Create a 4x4 rotation matrix
	   * const q = new pc.Quat().setFromEulerAngles(10, 20, 30);
	   *
	   * const tv = q.transformVector(v);
	   */;
	  _proto.transformVector = function transformVector(vec, res) {
	    if (res === void 0) {
	      res = new Vec3();
	    }
	    var x = vec.x,
	      y = vec.y,
	      z = vec.z;
	    var qx = this.x,
	      qy = this.y,
	      qz = this.z,
	      qw = this.w;

	    // calculate quat * vec
	    var ix = qw * x + qy * z - qz * y;
	    var iy = qw * y + qz * x - qx * z;
	    var iz = qw * z + qx * y - qy * x;
	    var iw = -qx * x - qy * y - qz * z;

	    // calculate result * inverse quat
	    res.x = ix * qw + iw * -qx + iy * -qz - iz * -qy;
	    res.y = iy * qw + iw * -qy + iz * -qx - ix * -qz;
	    res.z = iz * qw + iw * -qz + ix * -qy - iy * -qx;
	    return res;
	  }

	  /**
	   * Converts the quaternion to string form.
	   *
	   * @returns {string} The quaternion in string form.
	   * @example
	   * const v = new pc.Quat(0, 0, 0, 1);
	   * // Outputs [0, 0, 0, 1]
	   * console.log(v.toString());
	   */;
	  _proto.toString = function toString() {
	    return "[" + this.x + ", " + this.y + ", " + this.z + ", " + this.w + "]";
	  }

	  /**
	   * A constant quaternion set to [0, 0, 0, 1] (the identity).
	   *
	   * @type {Quat}
	   * @readonly
	   */;
	  return Quat;
	}();
	_class$4 = Quat;
	Quat.IDENTITY = Object.freeze(new _class$4(0, 0, 0, 1));
	/**
	 * A constant quaternion set to [0, 0, 0, 0].
	 *
	 * @type {Quat}
	 * @readonly
	 */
	Quat.ZERO = Object.freeze(new _class$4(0, 0, 0, 0));

	var tmpVecA$1 = new Vec3();
	var tmpVecB$1 = new Vec3();
	var tmpVecC = new Vec3();
	var tmpVecD = new Vec3();
	var tmpVecE = new Vec3();

	/**
	 * Axis-Aligned Bounding Box.
	 *
	 * @category Math
	 */
	var BoundingBox = /*#__PURE__*/function () {
	  /**
	   * Create a new BoundingBox instance. The bounding box is axis-aligned.
	   *
	   * @param {Vec3} [center] - Center of box. The constructor takes a reference of this parameter.
	   * @param {Vec3} [halfExtents] - Half the distance across the box in each axis. The constructor
	   * takes a reference of this parameter. Defaults to 0.5 on each axis.
	   */
	  function BoundingBox(center, halfExtents) {
	    if (center === void 0) {
	      center = new Vec3();
	    }
	    if (halfExtents === void 0) {
	      halfExtents = new Vec3(0.5, 0.5, 0.5);
	    }
	    /**
	     * Center of box.
	     *
	     * @type {Vec3}
	     */
	    this.center = void 0;
	    /**
	     * Half the distance across the box in each axis.
	     *
	     * @type {Vec3}
	     */
	    this.halfExtents = void 0;
	    /**
	     * @type {Vec3}
	     * @private
	     */
	    this._min = new Vec3();
	    /**
	     * @type {Vec3}
	     * @private
	     */
	    this._max = new Vec3();
	    Debug.assert(!Object.isFrozen(center), 'The constructor of \'BoundingBox\' does not accept a constant (frozen) object as a \'center\' parameter');
	    Debug.assert(!Object.isFrozen(halfExtents), 'The constructor of \'BoundingBox\' does not accept a constant (frozen) object as a \'halfExtents\' parameter');
	    this.center = center;
	    this.halfExtents = halfExtents;
	  }

	  /**
	   * Combines two bounding boxes into one, enclosing both.
	   *
	   * @param {BoundingBox} other - Bounding box to add.
	   */
	  var _proto = BoundingBox.prototype;
	  _proto.add = function add(other) {
	    var tc = this.center;
	    var tcx = tc.x;
	    var tcy = tc.y;
	    var tcz = tc.z;
	    var th = this.halfExtents;
	    var thx = th.x;
	    var thy = th.y;
	    var thz = th.z;
	    var tminx = tcx - thx;
	    var tmaxx = tcx + thx;
	    var tminy = tcy - thy;
	    var tmaxy = tcy + thy;
	    var tminz = tcz - thz;
	    var tmaxz = tcz + thz;
	    var oc = other.center;
	    var ocx = oc.x;
	    var ocy = oc.y;
	    var ocz = oc.z;
	    var oh = other.halfExtents;
	    var ohx = oh.x;
	    var ohy = oh.y;
	    var ohz = oh.z;
	    var ominx = ocx - ohx;
	    var omaxx = ocx + ohx;
	    var ominy = ocy - ohy;
	    var omaxy = ocy + ohy;
	    var ominz = ocz - ohz;
	    var omaxz = ocz + ohz;
	    if (ominx < tminx) tminx = ominx;
	    if (omaxx > tmaxx) tmaxx = omaxx;
	    if (ominy < tminy) tminy = ominy;
	    if (omaxy > tmaxy) tmaxy = omaxy;
	    if (ominz < tminz) tminz = ominz;
	    if (omaxz > tmaxz) tmaxz = omaxz;
	    tc.x = (tminx + tmaxx) * 0.5;
	    tc.y = (tminy + tmaxy) * 0.5;
	    tc.z = (tminz + tmaxz) * 0.5;
	    th.x = (tmaxx - tminx) * 0.5;
	    th.y = (tmaxy - tminy) * 0.5;
	    th.z = (tmaxz - tminz) * 0.5;
	  }

	  /**
	   * Copies the contents of a source AABB.
	   *
	   * @param {BoundingBox} src - The AABB to copy from.
	   */;
	  _proto.copy = function copy(src) {
	    this.center.copy(src.center);
	    this.halfExtents.copy(src.halfExtents);
	  }

	  /**
	   * Returns a clone of the AABB.
	   *
	   * @returns {BoundingBox} A duplicate AABB.
	   */;
	  _proto.clone = function clone() {
	    return new BoundingBox(this.center.clone(), this.halfExtents.clone());
	  }

	  /**
	   * Test whether two axis-aligned bounding boxes intersect.
	   *
	   * @param {BoundingBox} other - Bounding box to test against.
	   * @returns {boolean} True if there is an intersection.
	   */;
	  _proto.intersects = function intersects(other) {
	    var aMax = this.getMax();
	    var aMin = this.getMin();
	    var bMax = other.getMax();
	    var bMin = other.getMin();
	    return aMin.x <= bMax.x && aMax.x >= bMin.x && aMin.y <= bMax.y && aMax.y >= bMin.y && aMin.z <= bMax.z && aMax.z >= bMin.z;
	  };
	  _proto._intersectsRay = function _intersectsRay(ray, point) {
	    var tMin = tmpVecA$1.copy(this.getMin()).sub(ray.origin);
	    var tMax = tmpVecB$1.copy(this.getMax()).sub(ray.origin);
	    var dir = ray.direction;

	    // Ensure that we are not dividing it by zero
	    if (dir.x === 0) {
	      tMin.x = tMin.x < 0 ? -Number.MAX_VALUE : Number.MAX_VALUE;
	      tMax.x = tMax.x < 0 ? -Number.MAX_VALUE : Number.MAX_VALUE;
	    } else {
	      tMin.x /= dir.x;
	      tMax.x /= dir.x;
	    }
	    if (dir.y === 0) {
	      tMin.y = tMin.y < 0 ? -Number.MAX_VALUE : Number.MAX_VALUE;
	      tMax.y = tMax.y < 0 ? -Number.MAX_VALUE : Number.MAX_VALUE;
	    } else {
	      tMin.y /= dir.y;
	      tMax.y /= dir.y;
	    }
	    if (dir.z === 0) {
	      tMin.z = tMin.z < 0 ? -Number.MAX_VALUE : Number.MAX_VALUE;
	      tMax.z = tMax.z < 0 ? -Number.MAX_VALUE : Number.MAX_VALUE;
	    } else {
	      tMin.z /= dir.z;
	      tMax.z /= dir.z;
	    }
	    var realMin = tmpVecC.set(Math.min(tMin.x, tMax.x), Math.min(tMin.y, tMax.y), Math.min(tMin.z, tMax.z));
	    var realMax = tmpVecD.set(Math.max(tMin.x, tMax.x), Math.max(tMin.y, tMax.y), Math.max(tMin.z, tMax.z));
	    var minMax = Math.min(Math.min(realMax.x, realMax.y), realMax.z);
	    var maxMin = Math.max(Math.max(realMin.x, realMin.y), realMin.z);
	    var intersects = minMax >= maxMin && maxMin >= 0;
	    if (intersects) point.copy(ray.direction).mulScalar(maxMin).add(ray.origin);
	    return intersects;
	  };
	  _proto._fastIntersectsRay = function _fastIntersectsRay(ray) {
	    var diff = tmpVecA$1;
	    var cross = tmpVecB$1;
	    var prod = tmpVecC;
	    var absDiff = tmpVecD;
	    var absDir = tmpVecE;
	    var rayDir = ray.direction;
	    diff.sub2(ray.origin, this.center);
	    absDiff.set(Math.abs(diff.x), Math.abs(diff.y), Math.abs(diff.z));
	    prod.mul2(diff, rayDir);
	    if (absDiff.x > this.halfExtents.x && prod.x >= 0) return false;
	    if (absDiff.y > this.halfExtents.y && prod.y >= 0) return false;
	    if (absDiff.z > this.halfExtents.z && prod.z >= 0) return false;
	    absDir.set(Math.abs(rayDir.x), Math.abs(rayDir.y), Math.abs(rayDir.z));
	    cross.cross(rayDir, diff);
	    cross.set(Math.abs(cross.x), Math.abs(cross.y), Math.abs(cross.z));
	    if (cross.x > this.halfExtents.y * absDir.z + this.halfExtents.z * absDir.y) return false;
	    if (cross.y > this.halfExtents.x * absDir.z + this.halfExtents.z * absDir.x) return false;
	    if (cross.z > this.halfExtents.x * absDir.y + this.halfExtents.y * absDir.x) return false;
	    return true;
	  }

	  /**
	   * Test if a ray intersects with the AABB.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./ray.js').Ray} ray - Ray to test against (direction must be normalized).
	   * @param {Vec3} [point] - If there is an intersection, the intersection point will be copied
	   * into here.
	   * @returns {boolean} True if there is an intersection.
	   */;
	  _proto.intersectsRay = function intersectsRay(ray, point) {
	    if (point) {
	      return this._intersectsRay(ray, point);
	    }
	    return this._fastIntersectsRay(ray);
	  }

	  /**
	   * Sets the minimum and maximum corner of the AABB. Using this function is faster than
	   * assigning min and max separately.
	   *
	   * @param {Vec3} min - The minimum corner of the AABB.
	   * @param {Vec3} max - The maximum corner of the AABB.
	   */;
	  _proto.setMinMax = function setMinMax(min, max) {
	    this.center.add2(max, min).mulScalar(0.5);
	    this.halfExtents.sub2(max, min).mulScalar(0.5);
	  }

	  /**
	   * Return the minimum corner of the AABB.
	   *
	   * @returns {Vec3} Minimum corner.
	   */;
	  _proto.getMin = function getMin() {
	    return this._min.copy(this.center).sub(this.halfExtents);
	  }

	  /**
	   * Return the maximum corner of the AABB.
	   *
	   * @returns {Vec3} Maximum corner.
	   */;
	  _proto.getMax = function getMax() {
	    return this._max.copy(this.center).add(this.halfExtents);
	  }

	  /**
	   * Test if a point is inside a AABB.
	   *
	   * @param {Vec3} point - Point to test.
	   * @returns {boolean} True if the point is inside the AABB and false otherwise.
	   */;
	  _proto.containsPoint = function containsPoint(point) {
	    var min = this.getMin();
	    var max = this.getMax();
	    if (point.x < min.x || point.x > max.x || point.y < min.y || point.y > max.y || point.z < min.z || point.z > max.z) {
	      return false;
	    }
	    return true;
	  }

	  /**
	   * Set an AABB to enclose the specified AABB if it were to be transformed by the specified 4x4
	   * matrix.
	   *
	   * @param {BoundingBox} aabb - Box to transform and enclose.
	   * @param {new Function("modulePath", "return import(modulePath)")('../math/mat4.js').Mat4} m - Transformation matrix to apply to source AABB.
	   * @param {boolean} ignoreScale - If true is specified, a scale from the matrix is ignored. Defaults to false.
	   */;
	  _proto.setFromTransformedAabb = function setFromTransformedAabb(aabb, m, ignoreScale) {
	    if (ignoreScale === void 0) {
	      ignoreScale = false;
	    }
	    var ac = aabb.center;
	    var ar = aabb.halfExtents;
	    var d = m.data;
	    var mx0 = d[0];
	    var mx1 = d[4];
	    var mx2 = d[8];
	    var my0 = d[1];
	    var my1 = d[5];
	    var my2 = d[9];
	    var mz0 = d[2];
	    var mz1 = d[6];
	    var mz2 = d[10];

	    // renormalize axis if scale is to be ignored
	    if (ignoreScale) {
	      var lengthSq = mx0 * mx0 + mx1 * mx1 + mx2 * mx2;
	      if (lengthSq > 0) {
	        var invLength = 1 / Math.sqrt(lengthSq);
	        mx0 *= invLength;
	        mx1 *= invLength;
	        mx2 *= invLength;
	      }
	      lengthSq = my0 * my0 + my1 * my1 + my2 * my2;
	      if (lengthSq > 0) {
	        var _invLength = 1 / Math.sqrt(lengthSq);
	        my0 *= _invLength;
	        my1 *= _invLength;
	        my2 *= _invLength;
	      }
	      lengthSq = mz0 * mz0 + mz1 * mz1 + mz2 * mz2;
	      if (lengthSq > 0) {
	        var _invLength2 = 1 / Math.sqrt(lengthSq);
	        mz0 *= _invLength2;
	        mz1 *= _invLength2;
	        mz2 *= _invLength2;
	      }
	    }
	    this.center.set(d[12] + mx0 * ac.x + mx1 * ac.y + mx2 * ac.z, d[13] + my0 * ac.x + my1 * ac.y + my2 * ac.z, d[14] + mz0 * ac.x + mz1 * ac.y + mz2 * ac.z);
	    this.halfExtents.set(Math.abs(mx0) * ar.x + Math.abs(mx1) * ar.y + Math.abs(mx2) * ar.z, Math.abs(my0) * ar.x + Math.abs(my1) * ar.y + Math.abs(my2) * ar.z, Math.abs(mz0) * ar.x + Math.abs(mz1) * ar.y + Math.abs(mz2) * ar.z);
	  }

	  /**
	   * Compute the min and max bounding values to encapsulate all specified vertices.
	   *
	   * @param {number[]|Float32Array} vertices - The vertices used to compute the new size for the
	   * AABB.
	   * @param {Vec3} min - Stored computed min value.
	   * @param {Vec3} max - Stored computed max value.
	   * @param {number} [numVerts] - Number of vertices to use from the beginning of vertices array.
	   * All vertices are used if not specified.
	   */;
	  BoundingBox.computeMinMax = function computeMinMax(vertices, min, max, numVerts) {
	    if (numVerts === void 0) {
	      numVerts = vertices.length / 3;
	    }
	    if (numVerts > 0) {
	      var minx = vertices[0];
	      var miny = vertices[1];
	      var minz = vertices[2];
	      var maxx = minx;
	      var maxy = miny;
	      var maxz = minz;
	      var n = numVerts * 3;
	      for (var i = 3; i < n; i += 3) {
	        var x = vertices[i];
	        var y = vertices[i + 1];
	        var z = vertices[i + 2];
	        if (x < minx) minx = x;
	        if (y < miny) miny = y;
	        if (z < minz) minz = z;
	        if (x > maxx) maxx = x;
	        if (y > maxy) maxy = y;
	        if (z > maxz) maxz = z;
	      }
	      min.set(minx, miny, minz);
	      max.set(maxx, maxy, maxz);
	    }
	  }

	  /**
	   * Compute the size of the AABB to encapsulate all specified vertices.
	   *
	   * @param {number[]|Float32Array} vertices - The vertices used to compute the new size for the
	   * AABB.
	   * @param {number} [numVerts] - Number of vertices to use from the beginning of vertices array.
	   * All vertices are used if not specified.
	   */;
	  _proto.compute = function compute(vertices, numVerts) {
	    BoundingBox.computeMinMax(vertices, tmpVecA$1, tmpVecB$1, numVerts);
	    this.setMinMax(tmpVecA$1, tmpVecB$1);
	  }

	  /**
	   * Test if a Bounding Sphere is overlapping, enveloping, or inside this AABB.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./bounding-sphere.js').BoundingSphere} sphere - Bounding Sphere to test.
	   * @returns {boolean} True if the Bounding Sphere is overlapping, enveloping, or inside the
	   * AABB and false otherwise.
	   */;
	  _proto.intersectsBoundingSphere = function intersectsBoundingSphere(sphere) {
	    var sq = this._distanceToBoundingSphereSq(sphere);
	    if (sq <= sphere.radius * sphere.radius) {
	      return true;
	    }
	    return false;
	  };
	  _proto._distanceToBoundingSphereSq = function _distanceToBoundingSphereSq(sphere) {
	    var boxMin = this.getMin();
	    var boxMax = this.getMax();
	    var sq = 0;
	    var axis = ['x', 'y', 'z'];
	    for (var i = 0; i < 3; ++i) {
	      var out = 0;
	      var pn = sphere.center[axis[i]];
	      var bMin = boxMin[axis[i]];
	      var bMax = boxMax[axis[i]];
	      var val = 0;
	      if (pn < bMin) {
	        val = bMin - pn;
	        out += val * val;
	      }
	      if (pn > bMax) {
	        val = pn - bMax;
	        out += val * val;
	      }
	      sq += out;
	    }
	    return sq;
	  };
	  _proto._expand = function _expand(expandMin, expandMax) {
	    tmpVecA$1.add2(this.getMin(), expandMin);
	    tmpVecB$1.add2(this.getMax(), expandMax);
	    this.setMinMax(tmpVecA$1, tmpVecB$1);
	  };
	  return BoundingBox;
	}();

	var tmpVecA = new Vec3();
	var tmpVecB = new Vec3();

	/**
	 * A bounding sphere is a volume for facilitating fast intersection testing.
	 *
	 * @category Math
	 */
	var BoundingSphere = /*#__PURE__*/function () {
	  /**
	   * Creates a new BoundingSphere instance.
	   *
	   * @param {Vec3} [center] - The world space coordinate marking the center of the sphere. The
	   * constructor takes a reference of this parameter.
	   * @param {number} [radius] - The radius of the bounding sphere. Defaults to 0.5.
	   * @example
	   * // Create a new bounding sphere centered on the origin with a radius of 0.5
	   * const sphere = new pc.BoundingSphere();
	   */
	  function BoundingSphere(center, radius) {
	    if (center === void 0) {
	      center = new Vec3();
	    }
	    if (radius === void 0) {
	      radius = 0.5;
	    }
	    /**
	     * Center of sphere.
	     *
	     * @type {Vec3}
	     */
	    this.center = void 0;
	    /**
	     * The radius of the bounding sphere.
	     *
	     * @type {number}
	     */
	    this.radius = void 0;
	    Debug.assert(!Object.isFrozen(center), 'The constructor of \'BoundingSphere\' does not accept a constant (frozen) object as a \'center\' parameter');
	    this.center = center;
	    this.radius = radius;
	  }
	  var _proto = BoundingSphere.prototype;
	  _proto.containsPoint = function containsPoint(point) {
	    var lenSq = tmpVecA.sub2(point, this.center).lengthSq();
	    var r = this.radius;
	    return lenSq < r * r;
	  }

	  /**
	   * Test if a ray intersects with the sphere.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./ray.js').Ray} ray - Ray to test against (direction must be normalized).
	   * @param {Vec3} [point] - If there is an intersection, the intersection point will be copied
	   * into here.
	   * @returns {boolean} True if there is an intersection.
	   */;
	  _proto.intersectsRay = function intersectsRay(ray, point) {
	    var m = tmpVecA.copy(ray.origin).sub(this.center);
	    var b = m.dot(tmpVecB.copy(ray.direction).normalize());
	    var c = m.dot(m) - this.radius * this.radius;

	    // exit if ray's origin outside of sphere (c > 0) and ray pointing away from s (b > 0)
	    if (c > 0 && b > 0) return false;
	    var discr = b * b - c;
	    // a negative discriminant corresponds to ray missing sphere
	    if (discr < 0) return false;

	    // ray intersects sphere, compute smallest t value of intersection
	    var t = Math.abs(-b - Math.sqrt(discr));

	    // if t is negative, ray started inside sphere so clamp t to zero
	    if (point) point.copy(ray.direction).mulScalar(t).add(ray.origin);
	    return true;
	  }

	  /**
	   * Test if a Bounding Sphere is overlapping, enveloping, or inside this Bounding Sphere.
	   *
	   * @param {BoundingSphere} sphere - Bounding Sphere to test.
	   * @returns {boolean} True if the Bounding Sphere is overlapping, enveloping, or inside this Bounding Sphere and false otherwise.
	   */;
	  _proto.intersectsBoundingSphere = function intersectsBoundingSphere(sphere) {
	    tmpVecA.sub2(sphere.center, this.center);
	    var totalRadius = sphere.radius + this.radius;
	    if (tmpVecA.lengthSq() <= totalRadius * totalRadius) {
	      return true;
	    }
	    return false;
	  };
	  return BoundingSphere;
	}();

	/**
	 * A frustum is a shape that defines the viewing space of a camera. It can be used to determine
	 * visibility of points and bounding spheres. Typically, you would not create a Frustum shape
	 * directly, but instead query {@link CameraComponent#frustum}.
	 *
	 * @category Math
	 */
	var Frustum = /*#__PURE__*/function () {
	  /**
	   * Create a new Frustum instance.
	   *
	   * @example
	   * const frustum = new pc.Frustum();
	   */
	  function Frustum() {
	    this.planes = [];
	    for (var i = 0; i < 6; i++) this.planes[i] = [];
	  }

	  /**
	   * Updates the frustum shape based on the supplied 4x4 matrix.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('../math/mat4.js').Mat4} matrix - The matrix describing the shape of the
	   * frustum.
	   * @example
	   * // Create a perspective projection matrix
	   * const projMat = pc.Mat4();
	   * projMat.setPerspective(45, 16 / 9, 1, 1000);
	   *
	   * // Create a frustum shape that is represented by the matrix
	   * const frustum = new pc.Frustum();
	   * frustum.setFromMat4(projMat);
	   */
	  var _proto = Frustum.prototype;
	  _proto.setFromMat4 = function setFromMat4(matrix) {
	    var vpm = matrix.data;
	    var plane;
	    var planes = this.planes;

	    // Extract the numbers for the RIGHT plane
	    plane = planes[0];
	    plane[0] = vpm[3] - vpm[0];
	    plane[1] = vpm[7] - vpm[4];
	    plane[2] = vpm[11] - vpm[8];
	    plane[3] = vpm[15] - vpm[12];
	    // Normalize the result
	    var t = Math.sqrt(plane[0] * plane[0] + plane[1] * plane[1] + plane[2] * plane[2]);
	    plane[0] /= t;
	    plane[1] /= t;
	    plane[2] /= t;
	    plane[3] /= t;

	    // Extract the numbers for the LEFT plane
	    plane = planes[1];
	    plane[0] = vpm[3] + vpm[0];
	    plane[1] = vpm[7] + vpm[4];
	    plane[2] = vpm[11] + vpm[8];
	    plane[3] = vpm[15] + vpm[12];
	    // Normalize the result
	    t = Math.sqrt(plane[0] * plane[0] + plane[1] * plane[1] + plane[2] * plane[2]);
	    plane[0] /= t;
	    plane[1] /= t;
	    plane[2] /= t;
	    plane[3] /= t;

	    // Extract the BOTTOM plane
	    plane = planes[2];
	    plane[0] = vpm[3] + vpm[1];
	    plane[1] = vpm[7] + vpm[5];
	    plane[2] = vpm[11] + vpm[9];
	    plane[3] = vpm[15] + vpm[13];
	    // Normalize the result
	    t = Math.sqrt(plane[0] * plane[0] + plane[1] * plane[1] + plane[2] * plane[2]);
	    plane[0] /= t;
	    plane[1] /= t;
	    plane[2] /= t;
	    plane[3] /= t;

	    // Extract the TOP plane
	    plane = planes[3];
	    plane[0] = vpm[3] - vpm[1];
	    plane[1] = vpm[7] - vpm[5];
	    plane[2] = vpm[11] - vpm[9];
	    plane[3] = vpm[15] - vpm[13];
	    // Normalize the result
	    t = Math.sqrt(plane[0] * plane[0] + plane[1] * plane[1] + plane[2] * plane[2]);
	    plane[0] /= t;
	    plane[1] /= t;
	    plane[2] /= t;
	    plane[3] /= t;

	    // Extract the FAR plane
	    plane = planes[4];
	    plane[0] = vpm[3] - vpm[2];
	    plane[1] = vpm[7] - vpm[6];
	    plane[2] = vpm[11] - vpm[10];
	    plane[3] = vpm[15] - vpm[14];
	    // Normalize the result
	    t = Math.sqrt(plane[0] * plane[0] + plane[1] * plane[1] + plane[2] * plane[2]);
	    plane[0] /= t;
	    plane[1] /= t;
	    plane[2] /= t;
	    plane[3] /= t;

	    // Extract the NEAR plane
	    plane = planes[5];
	    plane[0] = vpm[3] + vpm[2];
	    plane[1] = vpm[7] + vpm[6];
	    plane[2] = vpm[11] + vpm[10];
	    plane[3] = vpm[15] + vpm[14];
	    // Normalize the result
	    t = Math.sqrt(plane[0] * plane[0] + plane[1] * plane[1] + plane[2] * plane[2]);
	    plane[0] /= t;
	    plane[1] /= t;
	    plane[2] /= t;
	    plane[3] /= t;
	  }

	  /**
	   * Tests whether a point is inside the frustum. Note that points lying in a frustum plane are
	   * considered to be outside the frustum.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('../math/vec3.js').Vec3} point - The point to test.
	   * @returns {boolean} True if the point is inside the frustum, false otherwise.
	   */;
	  _proto.containsPoint = function containsPoint(point) {
	    var p, plane;
	    for (p = 0; p < 6; p++) {
	      plane = this.planes[p];
	      if (plane[0] * point.x + plane[1] * point.y + plane[2] * point.z + plane[3] <= 0) {
	        return false;
	      }
	    }
	    return true;
	  }

	  /**
	   * Tests whether a bounding sphere intersects the frustum. If the sphere is outside the
	   * frustum, zero is returned. If the sphere intersects the frustum, 1 is returned. If the
	   * sphere is completely inside the frustum, 2 is returned. Note that a sphere touching a
	   * frustum plane from the outside is considered to be outside the frustum.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./bounding-sphere.js').BoundingSphere} sphere - The sphere to test.
	   * @returns {number} 0 if the bounding sphere is outside the frustum, 1 if it intersects the
	   * frustum and 2 if it is contained by the frustum.
	   */;
	  _proto.containsSphere = function containsSphere(sphere) {
	    var c = 0;
	    var d;
	    var p;
	    var sr = sphere.radius;
	    var sc = sphere.center;
	    var scx = sc.x;
	    var scy = sc.y;
	    var scz = sc.z;
	    var planes = this.planes;
	    var plane;
	    for (p = 0; p < 6; p++) {
	      plane = planes[p];
	      d = plane[0] * scx + plane[1] * scy + plane[2] * scz + plane[3];
	      if (d <= -sr) return 0;
	      if (d > sr) c++;
	    }
	    return c === 6 ? 2 : 1;
	  };
	  return Frustum;
	}();

	/**
	 * An infinite ray.
	 *
	 * @category Math
	 */
	var Ray = /*#__PURE__*/function () {
	  /**
	   * Creates a new Ray instance. The ray is infinite, starting at a given origin and pointing in
	   * a given direction.
	   *
	   * @param {Vec3} [origin] - The starting point of the ray. The constructor copies
	   * this parameter. Defaults to the origin (0, 0, 0).
	   * @param {Vec3} [direction] - The direction of the ray. The constructor copies
	   * this parameter. Defaults to a direction down the world negative Z axis (0, 0, -1).
	   * @example
	   * // Create a new ray starting at the position of this entity and pointing down
	   * // the entity's negative Z axis
	   * const ray = new pc.Ray(this.entity.getPosition(), this.entity.forward);
	   */
	  function Ray(origin, direction) {
	    /**
	     * The starting point of the ray.
	     *
	     * @readonly
	     * @type {Vec3}
	     */
	    this.origin = new Vec3();
	    /**
	     * The direction of the ray.
	     *
	     * @readonly
	     * @type {Vec3}
	     */
	    this.direction = Vec3.FORWARD.clone();
	    if (origin) {
	      this.origin.copy(origin);
	    }
	    if (direction) {
	      this.direction.copy(direction);
	    }
	  }

	  /**
	   * Sets origin and direction to the supplied vector values.
	   *
	   * @param {Vec3} origin - The starting point of the ray.
	   * @param {Vec3} direction - The direction of the ray.
	   * @returns {Ray} Self for chaining.
	   */
	  var _proto = Ray.prototype;
	  _proto.set = function set(origin, direction) {
	    this.origin.copy(origin);
	    this.direction.copy(direction);
	    return this;
	  }

	  /**
	   * Copies the contents of a source Ray.
	   *
	   * @param {Ray} src - The Ray to copy from.
	   * @returns {Ray} Self for chaining.
	   */;
	  _proto.copy = function copy(src) {
	    return this.set(src.origin, src.direction);
	  }

	  /**
	   * Returns a clone of the Ray.
	   *
	   * @returns {this} A duplicate Ray.
	   */;
	  _proto.clone = function clone() {
	    return new this.constructor(this.origin, this.direction);
	  };
	  return Ray;
	}();

	var tmpRay = new Ray();
	var tmpVec3$2 = new Vec3();
	var tmpSphere = new BoundingSphere();
	var tmpMat4$1 = new Mat4();

	/**
	 * Oriented Box.
	 *
	 * @category Math
	 */
	var OrientedBox = /*#__PURE__*/function () {
	  /**
	   * Create a new OrientedBox instance.
	   *
	   * @param {Mat4} [worldTransform] - Transform that has the orientation and position of the box.
	   * Scale is assumed to be one.
	   * @param {Vec3} [halfExtents] - Half the distance across the box in each local axis. The
	   * constructor takes a reference of this parameter.
	   */
	  function OrientedBox(worldTransform, halfExtents) {
	    if (worldTransform === void 0) {
	      worldTransform = new Mat4();
	    }
	    if (halfExtents === void 0) {
	      halfExtents = new Vec3(0.5, 0.5, 0.5);
	    }
	    this.halfExtents = void 0;
	    /**
	     * @type {Mat4}
	     * @private
	     */
	    this._modelTransform = void 0;
	    /**
	     * @type {Mat4}
	     * @private
	     */
	    this._worldTransform = void 0;
	    /**
	     * @type {BoundingBox}
	     * @private
	     */
	    this._aabb = void 0;
	    Debug.assert(!Object.isFrozen(worldTransform), 'The constructor of \'OrientedBox\' does not accept a constant (frozen) object as a \'worldTransform\' parameter');
	    Debug.assert(!Object.isFrozen(halfExtents), 'The constructor of \'OrientedBox\' does not accept a constant (frozen) object as a \'halfExtents\' parameter');
	    this.halfExtents = halfExtents;
	    this._modelTransform = worldTransform.clone().invert();
	    this._worldTransform = worldTransform.clone();
	    this._aabb = new BoundingBox(new Vec3(), this.halfExtents);
	  }

	  /**
	   * The world transform of the OBB.
	   *
	   * @type {Mat4}
	   */
	  var _proto = OrientedBox.prototype;
	  /**
	   * Test if a ray intersects with the OBB.
	   *
	   * @param {Ray} ray - Ray to test against (direction must be normalized).
	   * @param {Vec3} [point] - If there is an intersection, the intersection point will be copied
	   * into here.
	   * @returns {boolean} True if there is an intersection.
	   */
	  _proto.intersectsRay = function intersectsRay(ray, point) {
	    this._modelTransform.transformPoint(ray.origin, tmpRay.origin);
	    this._modelTransform.transformVector(ray.direction, tmpRay.direction);
	    if (point) {
	      var result = this._aabb._intersectsRay(tmpRay, point);
	      tmpMat4$1.copy(this._modelTransform).invert().transformPoint(point, point);
	      return result;
	    }
	    return this._aabb._fastIntersectsRay(tmpRay);
	  }

	  /**
	   * Test if a point is inside a OBB.
	   *
	   * @param {Vec3} point - Point to test.
	   * @returns {boolean} True if the point is inside the OBB and false otherwise.
	   */;
	  _proto.containsPoint = function containsPoint(point) {
	    this._modelTransform.transformPoint(point, tmpVec3$2);
	    return this._aabb.containsPoint(tmpVec3$2);
	  }

	  /**
	   * Test if a Bounding Sphere is overlapping, enveloping, or inside this OBB.
	   *
	   * @param {BoundingSphere} sphere - Bounding Sphere to test.
	   * @returns {boolean} True if the Bounding Sphere is overlapping, enveloping or inside this OBB
	   * and false otherwise.
	   */;
	  _proto.intersectsBoundingSphere = function intersectsBoundingSphere(sphere) {
	    this._modelTransform.transformPoint(sphere.center, tmpSphere.center);
	    tmpSphere.radius = sphere.radius;
	    if (this._aabb.intersectsBoundingSphere(tmpSphere)) {
	      return true;
	    }
	    return false;
	  };
	  _createClass(OrientedBox, [{
	    key: "worldTransform",
	    get: function get() {
	      return this._worldTransform;
	    },
	    set: function set(value) {
	      this._worldTransform.copy(value);
	      this._modelTransform.copy(value).invert();
	    }
	  }]);
	  return OrientedBox;
	}();

	/**
	 * An infinite plane. Internally it's represented in a parametric equation form:
	 * ax + by + cz + distance = 0.
	 *
	 * @category Math
	 */
	var Plane = /*#__PURE__*/function () {
	  /**
	   * Create a new Plane instance.
	   *
	   * @param {Vec3} [normal] - Normal of the plane. The constructor copies this parameter. Defaults
	   * to {@link Vec3.UP}.
	   * @param {number} [distance] - The distance from the plane to the origin, along its normal.
	   * Defaults to 0.
	   */
	  function Plane(normal, distance) {
	    if (normal === void 0) {
	      normal = Vec3.UP;
	    }
	    if (distance === void 0) {
	      distance = 0;
	    }
	    /**
	     * The normal of the plane.
	     *
	     * @readonly
	     * @type {Vec3}
	     */
	    this.normal = new Vec3();
	    /**
	     * The distance from the plane to the origin, along its normal.
	     *
	     * @readonly
	     * @type {number}
	     */
	    this.distance = void 0;
	    this.normal.copy(normal);
	    this.distance = distance;
	  }

	  /**
	   * Sets the plane based on a specified normal and a point on the plane.
	   *
	   * @param {Vec3} point - The point on the plane.
	   * @param {Vec3} normal - The normal of the plane.
	   * @returns {Plane} Self for chaining.
	   */
	  var _proto = Plane.prototype;
	  _proto.setFromPointNormal = function setFromPointNormal(point, normal) {
	    this.normal.copy(normal);
	    this.distance = -this.normal.dot(point);
	    return this;
	  }

	  /**
	   * Test if the plane intersects between two points.
	   *
	   * @param {Vec3} start - Start position of line.
	   * @param {Vec3} end - End position of line.
	   * @param {Vec3} [point] - If there is an intersection, the intersection point will be copied
	   * into here.
	   * @returns {boolean} True if there is an intersection.
	   */;
	  _proto.intersectsLine = function intersectsLine(start, end, point) {
	    var d = this.distance;
	    var d0 = this.normal.dot(start) + d;
	    var d1 = this.normal.dot(end) + d;
	    var t = d0 / (d0 - d1);
	    var intersects = t >= 0 && t <= 1;
	    if (intersects && point) point.lerp(start, end, t);
	    return intersects;
	  }

	  /**
	   * Test if a ray intersects with the infinite plane.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./ray.js').Ray} ray - Ray to test against (direction must be normalized).
	   * @param {Vec3} [point] - If there is an intersection, the intersection point will be copied
	   * into here.
	   * @returns {boolean} True if there is an intersection.
	   */;
	  _proto.intersectsRay = function intersectsRay(ray, point) {
	    var denominator = this.normal.dot(ray.direction);
	    if (denominator === 0) return false;
	    var t = -(this.normal.dot(ray.origin) + this.distance) / denominator;
	    if (t >= 0 && point) {
	      point.copy(ray.direction).mulScalar(t).add(ray.origin);
	    }
	    return t >= 0;
	  }

	  /**
	   * Copies the contents of a source Plane.
	   *
	   * @param {Plane} src - The Plane to copy from.
	   * @returns {Plane} Self for chaining.
	   */;
	  _proto.copy = function copy(src) {
	    this.normal.copy(src.normal);
	    this.distance = src.distance;
	    return this;
	  }

	  /**
	   * Returns a clone of the Plane.
	   *
	   * @returns {this} A duplicate Plane.
	   */;
	  _proto.clone = function clone() {
	    /** @type {this} */
	    var cstr = this.constructor;
	    return new cstr().copy(this);
	  };
	  return Plane;
	}();

	/**
	 * Linear distance model.
	 *
	 * @type {string}
	 * @category Sound
	 */
	var DISTANCE_LINEAR = 'linear';

	/**
	 * Inverse distance model.
	 *
	 * @type {string}
	 * @category Sound
	 */
	var DISTANCE_INVERSE = 'inverse';

	/**
	 * Exponential distance model.
	 *
	 * @type {string}
	 * @category Sound
	 */
	var DISTANCE_EXPONENTIAL = 'exponential';

	/**
	 * Ignores the integer part of texture coordinates, using only the fractional part.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var ADDRESS_REPEAT = 0;

	/**
	 * Clamps texture coordinate to the range 0 to 1.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var ADDRESS_CLAMP_TO_EDGE = 1;

	/**
	 * Texture coordinate to be set to the fractional part if the integer part is even. If the integer
	 * part is odd, then the texture coordinate is set to 1 minus the fractional part.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var ADDRESS_MIRRORED_REPEAT = 2;

	/**
	 * Multiply all fragment components by zero.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var BLENDMODE_ZERO = 0;

	/**
	 * Multiply all fragment components by one.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var BLENDMODE_ONE = 1;

	/**
	 * Multiply all fragment components by the components of the source fragment.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var BLENDMODE_SRC_COLOR = 2;

	/**
	 * Multiply all fragment components by one minus the components of the source fragment.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var BLENDMODE_ONE_MINUS_SRC_COLOR = 3;

	/**
	 * Multiply all fragment components by the components of the destination fragment.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var BLENDMODE_DST_COLOR = 4;

	/**
	 * Multiply all fragment components by one minus the components of the destination fragment.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var BLENDMODE_ONE_MINUS_DST_COLOR = 5;

	/**
	 * Multiply all fragment components by the alpha value of the source fragment.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var BLENDMODE_SRC_ALPHA = 6;

	/**
	 * Multiply all fragment components by the alpha value of the source fragment.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var BLENDMODE_SRC_ALPHA_SATURATE = 7;

	/**
	 * Multiply all fragment components by one minus the alpha value of the source fragment.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var BLENDMODE_ONE_MINUS_SRC_ALPHA = 8;

	/**
	 * Multiply all fragment components by the alpha value of the destination fragment.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var BLENDMODE_DST_ALPHA = 9;

	/**
	 * Multiply all fragment components by one minus the alpha value of the destination fragment.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var BLENDMODE_ONE_MINUS_DST_ALPHA = 10;

	/**
	 * Multiplies all fragment components by a constant.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var BLENDMODE_CONSTANT = 11;

	/**
	 * Multiplies all fragment components by 1 minus a constant.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var BLENDMODE_ONE_MINUS_CONSTANT = 12;

	/**
	 * Add the results of the source and destination fragment multiplies.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var BLENDEQUATION_ADD = 0;

	/**
	 * Subtract the results of the source and destination fragment multiplies.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var BLENDEQUATION_SUBTRACT = 1;

	/**
	 * Reverse and subtract the results of the source and destination fragment multiplies.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var BLENDEQUATION_REVERSE_SUBTRACT = 2;

	/**
	 * Use the smallest value. Check app.graphicsDevice.extBlendMinmax for support.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var BLENDEQUATION_MIN = 3;

	/**
	 * Use the largest value. Check app.graphicsDevice.extBlendMinmax for support.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var BLENDEQUATION_MAX = 4;

	/**
	 * The data store contents will be modified once and used many times.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var BUFFER_STATIC = 0;

	/**
	 * The data store contents will be modified repeatedly and used many times.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var BUFFER_DYNAMIC = 1;

	/**
	 * The data store contents will be modified once and used at most a few times.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var BUFFER_STREAM = 2;

	/**
	 * The data store contents will be modified repeatedly on the GPU and used many times. Optimal for
	 * transform feedback usage (WebGL2 only).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var BUFFER_GPUDYNAMIC = 3;

	/**
	 * Clear the color buffer.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var CLEARFLAG_COLOR = 1;

	/**
	 * Clear the depth buffer.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var CLEARFLAG_DEPTH = 2;

	/**
	 * Clear the stencil buffer.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var CLEARFLAG_STENCIL = 4;

	/**
	 * The positive X face of a cubemap.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var CUBEFACE_POSX = 0;

	/**
	 * The negative X face of a cubemap.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var CUBEFACE_NEGX = 1;

	/**
	 * The positive Y face of a cubemap.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var CUBEFACE_POSY = 2;

	/**
	 * The negative Y face of a cubemap.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var CUBEFACE_NEGY = 3;

	/**
	 * The positive Z face of a cubemap.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var CUBEFACE_POSZ = 4;

	/**
	 * The negative Z face of a cubemap.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var CUBEFACE_NEGZ = 5;

	/**
	 * No triangles are culled.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var CULLFACE_NONE = 0;

	/**
	 * Triangles facing away from the view direction are culled.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var CULLFACE_BACK = 1;

	/**
	 * Triangles facing the view direction are culled.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var CULLFACE_FRONT = 2;

	/**
	 * Triangles are culled regardless of their orientation with respect to the view direction. Note
	 * that point or line primitives are unaffected by this render state.
	 *
	 * @type {number}
	 * @ignore
	 * @category Graphics
	 */
	var CULLFACE_FRONTANDBACK = 3;

	/**
	 * Point sample filtering.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var FILTER_NEAREST = 0;

	/**
	 * Bilinear filtering.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var FILTER_LINEAR = 1;

	/**
	 * Use the nearest neighbor in the nearest mipmap level.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var FILTER_NEAREST_MIPMAP_NEAREST = 2;

	/**
	 * Linearly interpolate in the nearest mipmap level.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var FILTER_NEAREST_MIPMAP_LINEAR = 3;

	/**
	 * Use the nearest neighbor after linearly interpolating between mipmap levels.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var FILTER_LINEAR_MIPMAP_NEAREST = 4;

	/**
	 * Linearly interpolate both the mipmap levels and between texels.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var FILTER_LINEAR_MIPMAP_LINEAR = 5;

	/**
	 * Never pass.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var FUNC_NEVER = 0;

	/**
	 * Pass if (ref & mask) < (stencil & mask).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var FUNC_LESS = 1;

	/**
	 * Pass if (ref & mask) == (stencil & mask).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var FUNC_EQUAL = 2;

	/**
	 * Pass if (ref & mask) <= (stencil & mask).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var FUNC_LESSEQUAL = 3;

	/**
	 * Pass if (ref & mask) > (stencil & mask).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var FUNC_GREATER = 4;

	/**
	 * Pass if (ref & mask) != (stencil & mask).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var FUNC_NOTEQUAL = 5;

	/**
	 * Pass if (ref & mask) >= (stencil & mask).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var FUNC_GREATEREQUAL = 6;

	/**
	 * Always pass.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var FUNC_ALWAYS = 7;

	/**
	 * 8-bit unsigned vertex indices (0 to 255).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var INDEXFORMAT_UINT8 = 0;

	/**
	 * 16-bit unsigned vertex indices (0 to 65,535).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var INDEXFORMAT_UINT16 = 1;

	/**
	 * 32-bit unsigned vertex indices (0 to 4,294,967,295).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var INDEXFORMAT_UINT32 = 2;

	/**
	 * 8-bit alpha.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_A8 = 0;

	/**
	 * 8-bit luminance.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_L8 = 1;

	/**
	 * 8-bit luminance with 8-bit alpha.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_LA8 = 2;

	/**
	 * 16-bit RGB (5-bits for red channel, 6 for green and 5 for blue).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_RGB565 = 3;

	/**
	 * 16-bit RGBA (5-bits for red channel, 5 for green, 5 for blue with 1-bit alpha).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_RGBA5551 = 4;

	/**
	 * 16-bit RGBA (4-bits for red channel, 4 for green, 4 for blue with 4-bit alpha).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_RGBA4 = 5;

	/**
	 * 24-bit RGB (8-bits for red channel, 8 for green and 8 for blue).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_RGB8 = 6;

	/**
	 * 32-bit RGBA (8-bits for red channel, 8 for green, 8 for blue with 8-bit alpha).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_RGBA8 = 7;

	/**
	 * Block compressed format storing 16 input pixels in 64 bits of output, consisting of two 16-bit
	 * RGB 5:6:5 color values and a 4x4 two bit lookup table.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_DXT1 = 8;

	/**
	 * Block compressed format storing 16 input pixels (corresponding to a 4x4 pixel block) into 128
	 * bits of output, consisting of 64 bits of alpha channel data (4 bits for each pixel) followed by
	 * 64 bits of color data; encoded the same way as DXT1.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_DXT3 = 9;

	/**
	 * Block compressed format storing 16 input pixels into 128 bits of output, consisting of 64 bits
	 * of alpha channel data (two 8 bit alpha values and a 4x4 3 bit lookup table) followed by 64 bits
	 * of color data (encoded the same way as DXT1).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_DXT5 = 10;

	/**
	 * 16-bit floating point RGB (16-bit float for each red, green and blue channels).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_RGB16F = 11;

	/**
	 * 16-bit floating point RGBA (16-bit float for each red, green, blue and alpha channels).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_RGBA16F = 12;

	/**
	 * 32-bit floating point RGB (32-bit float for each red, green and blue channels).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_RGB32F = 13;

	/**
	 * 32-bit floating point RGBA (32-bit float for each red, green, blue and alpha channels).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_RGBA32F = 14;

	/**
	 * 32-bit floating point single channel format (WebGL2 only).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_R32F = 15;

	/**
	 * A readable depth buffer format.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_DEPTH = 16;

	/**
	 * A readable depth/stencil buffer format (WebGL2 only).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_DEPTHSTENCIL = 17;

	/**
	 * A floating-point color-only format with 11 bits for red and green channels and 10 bits for the
	 * blue channel (WebGL2 only).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_111110F = 18;

	/**
	 * Color-only sRGB format (WebGL2 only).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_SRGB = 19;

	/**
	 * Color sRGB format with additional alpha channel (WebGL2 only).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_SRGBA = 20;

	/**
	 * ETC1 compressed format.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_ETC1 = 21;

	/**
	 * ETC2 (RGB) compressed format.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_ETC2_RGB = 22;

	/**
	 * ETC2 (RGBA) compressed format.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_ETC2_RGBA = 23;

	/**
	 * PVRTC (2BPP RGB) compressed format.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_PVRTC_2BPP_RGB_1 = 24;

	/**
	 * PVRTC (2BPP RGBA) compressed format.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_PVRTC_2BPP_RGBA_1 = 25;

	/**
	 * PVRTC (4BPP RGB) compressed format.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_PVRTC_4BPP_RGB_1 = 26;

	/**
	 * PVRTC (4BPP RGBA) compressed format.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_PVRTC_4BPP_RGBA_1 = 27;

	/**
	 * ATC compressed format with alpha channel in blocks of 4x4.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_ASTC_4x4 = 28;

	/**
	 * ATC compressed format with no alpha channel.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_ATC_RGB = 29;

	/**
	 * ATC compressed format with alpha channel.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_ATC_RGBA = 30;

	/**
	 * 32-bit BGRA (8-bits for blue channel, 8 for green, 8 for red with 8-bit alpha).
	 *
	 * @type {number}
	 * @ignore
	 * @category Graphics
	 */
	var PIXELFORMAT_BGRA8 = 31;

	/**
	 * 8-bit signed integer single-channel (R) format (Not supported by WebGL1).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_R8I = 32;

	/**
	 * 8-bit unsigned integer single-channel (R) format (Not supported by WebGL1).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_R8U = 33;

	/**
	 * 16-bit signed integer single-channel (R) format (Not supported by WebGL1).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_R16I = 34;

	/**
	 * 16-bit unsigned integer single-channel (R) format (Not supported by WebGL1).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_R16U = 35;

	/**
	 * 32-bit signed integer single-channel (R) format (Not supported by WebGL1).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_R32I = 36;

	/**
	 * 32-bit unsigned integer single-channel (R) format (Not supported by WebGL1).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_R32U = 37;

	/**
	 * 8-bit per-channel signed integer (RG) format (Not supported by WebGL1).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_RG8I = 38;

	/**
	 * 8-bit per-channel unsigned integer (RG) format (Not supported by WebGL1).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_RG8U = 39;

	/**
	 * 16-bit per-channel signed integer (RG) format (Not supported by WebGL1).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_RG16I = 40;

	/**
	 * 16-bit per-channel unsigned integer (RG) format (Not supported by WebGL1).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_RG16U = 41;

	/**
	 * 32-bit per-channel signed integer (RG) format (Not supported by WebGL1).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_RG32I = 42;

	/**
	 * 32-bit per-channel unsigned integer (RG) format (Not supported by WebGL1).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_RG32U = 43;

	/**
	 * 8-bit per-channel signed integer (RGBA) format (Not supported by WebGL1).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_RGBA8I = 44;

	/**
	 * 8-bit per-channel unsigned integer (RGBA) format (Not supported by WebGL1).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_RGBA8U = 45;

	/**
	 * 16-bit per-channel signed integer (RGBA) format (Not supported by WebGL1).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_RGBA16I = 46;

	/**
	 * 16-bit per-channel unsigned integer (RGBA) format (Not supported by WebGL1).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_RGBA16U = 47;

	/**
	 * 32-bit per-channel signed integer (RGBA) format (Not supported by WebGL1).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_RGBA32I = 48;

	/**
	 * 32-bit per-channel unsigned integer (RGBA) format (Not supported by WebGL1).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_RGBA32U = 49;

	/**
	 * 16-bit floating point R (16-bit float for red channel).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_R16F = 50;

	/**
	 * 16-bit floating point RG (16-bit float for each red and green channels).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PIXELFORMAT_RG16F = 51;

	// map of engine PIXELFORMAT_*** enums to information about the format
	var pixelFormatInfo = new Map([
	// uncompressed formats
	[PIXELFORMAT_A8, {
	  name: 'A8',
	  size: 1
	}], [PIXELFORMAT_L8, {
	  name: 'L8',
	  size: 1
	}], [PIXELFORMAT_LA8, {
	  name: 'LA8',
	  size: 2
	}], [PIXELFORMAT_RGB565, {
	  name: 'RGB565',
	  size: 2
	}], [PIXELFORMAT_RGBA5551, {
	  name: 'RGBA5551',
	  size: 2
	}], [PIXELFORMAT_RGBA4, {
	  name: 'RGBA4',
	  size: 2
	}], [PIXELFORMAT_RGB8, {
	  name: 'RGB8',
	  size: 4
	}], [PIXELFORMAT_RGBA8, {
	  name: 'RGBA8',
	  size: 4
	}], [PIXELFORMAT_R16F, {
	  name: 'R16F',
	  size: 2
	}], [PIXELFORMAT_RG16F, {
	  name: 'RG16F',
	  size: 4
	}], [PIXELFORMAT_RGB16F, {
	  name: 'RGB16F',
	  size: 8
	}], [PIXELFORMAT_RGBA16F, {
	  name: 'RGBA16F',
	  size: 8
	}], [PIXELFORMAT_RGB32F, {
	  name: 'RGB32F',
	  size: 16
	}], [PIXELFORMAT_RGBA32F, {
	  name: 'RGBA32F',
	  size: 16
	}], [PIXELFORMAT_R32F, {
	  name: 'R32F',
	  size: 4
	}], [PIXELFORMAT_DEPTH, {
	  name: 'DEPTH',
	  size: 4
	}], [PIXELFORMAT_DEPTHSTENCIL, {
	  name: 'DEPTHSTENCIL',
	  size: 4
	}], [PIXELFORMAT_111110F, {
	  name: '111110F',
	  size: 4
	}], [PIXELFORMAT_SRGB, {
	  name: 'SRGB',
	  size: 4
	}], [PIXELFORMAT_SRGBA, {
	  name: 'SRGBA',
	  size: 4
	}], [PIXELFORMAT_BGRA8, {
	  name: 'BGRA8',
	  size: 4
	}],
	// compressed formats
	[PIXELFORMAT_DXT1, {
	  name: 'DXT1',
	  blockSize: 8
	}], [PIXELFORMAT_DXT3, {
	  name: 'DXT3',
	  blockSize: 16
	}], [PIXELFORMAT_DXT5, {
	  name: 'DXT5',
	  blockSize: 16
	}], [PIXELFORMAT_ETC1, {
	  name: 'ETC1',
	  blockSize: 8
	}], [PIXELFORMAT_ETC2_RGB, {
	  name: 'ETC2_RGB',
	  blockSize: 8
	}], [PIXELFORMAT_ETC2_RGBA, {
	  name: 'ETC2_RGBA',
	  blockSize: 16
	}], [PIXELFORMAT_PVRTC_2BPP_RGB_1, {
	  name: 'PVRTC_2BPP_RGB_1',
	  blockSize: 8
	}], [PIXELFORMAT_PVRTC_2BPP_RGBA_1, {
	  name: 'PVRTC_2BPP_RGBA_1',
	  blockSize: 8
	}], [PIXELFORMAT_PVRTC_4BPP_RGB_1, {
	  name: 'PVRTC_4BPP_RGB_1',
	  blockSize: 8
	}], [PIXELFORMAT_PVRTC_4BPP_RGBA_1, {
	  name: 'PVRTC_4BPP_RGBA_1',
	  blockSize: 8
	}], [PIXELFORMAT_ASTC_4x4, {
	  name: 'ASTC_4x4',
	  blockSize: 16
	}], [PIXELFORMAT_ATC_RGB, {
	  name: 'ATC_RGB',
	  blockSize: 8
	}], [PIXELFORMAT_ATC_RGBA, {
	  name: 'ATC_RGBA',
	  blockSize: 16
	}],
	// uncompressed integer formats (Not supported on WebGL1)
	[PIXELFORMAT_R8I, {
	  name: 'R8I',
	  size: 1,
	  isInt: true
	}], [PIXELFORMAT_R8U, {
	  name: 'R8U',
	  size: 1,
	  isInt: true
	}], [PIXELFORMAT_R16I, {
	  name: 'R16I',
	  size: 2,
	  isInt: true
	}], [PIXELFORMAT_R16U, {
	  name: 'R16U',
	  size: 2,
	  isInt: true
	}], [PIXELFORMAT_R32I, {
	  name: 'R32I',
	  size: 4,
	  isInt: true
	}], [PIXELFORMAT_R32U, {
	  name: 'R32U',
	  size: 4,
	  isInt: true
	}], [PIXELFORMAT_RG8I, {
	  name: 'RG8I',
	  size: 2,
	  isInt: true
	}], [PIXELFORMAT_RG8U, {
	  name: 'RG8U',
	  size: 2,
	  isInt: true
	}], [PIXELFORMAT_RG16I, {
	  name: 'RG16I',
	  size: 4,
	  isInt: true
	}], [PIXELFORMAT_RG16U, {
	  name: 'RG16U',
	  size: 4,
	  isInt: true
	}], [PIXELFORMAT_RG32I, {
	  name: 'RG32I',
	  size: 8,
	  isInt: true
	}], [PIXELFORMAT_RG32U, {
	  name: 'RG32U',
	  size: 8,
	  isInt: true
	}], [PIXELFORMAT_RGBA8I, {
	  name: 'RGBA8I',
	  size: 4,
	  isInt: true
	}], [PIXELFORMAT_RGBA8U, {
	  name: 'RGBA8U',
	  size: 4,
	  isInt: true
	}], [PIXELFORMAT_RGBA16I, {
	  name: 'RGBA16I',
	  size: 8,
	  isInt: true
	}], [PIXELFORMAT_RGBA16U, {
	  name: 'RGBA16U',
	  size: 8,
	  isInt: true
	}], [PIXELFORMAT_RGBA32I, {
	  name: 'RGBA32I',
	  size: 16,
	  isInt: true
	}], [PIXELFORMAT_RGBA32U, {
	  name: 'RGBA32U',
	  size: 16,
	  isInt: true
	}]]);

	// update this function when exposing additional compressed pixel formats
	var isCompressedPixelFormat = function isCompressedPixelFormat(format) {
	  var _pixelFormatInfo$get;
	  return ((_pixelFormatInfo$get = pixelFormatInfo.get(format)) == null ? void 0 : _pixelFormatInfo$get.blockSize) !== undefined;
	};
	var isIntegerPixelFormat = function isIntegerPixelFormat(format) {
	  var _pixelFormatInfo$get2;
	  return ((_pixelFormatInfo$get2 = pixelFormatInfo.get(format)) == null ? void 0 : _pixelFormatInfo$get2.isInt) === true;
	};

	// get the pixel format array type
	var getPixelFormatArrayType = function getPixelFormatArrayType(format) {
	  switch (format) {
	    case PIXELFORMAT_R32F:
	    case PIXELFORMAT_RGB32F:
	    case PIXELFORMAT_RGBA32F:
	      return Float32Array;
	    case PIXELFORMAT_R32I:
	    case PIXELFORMAT_RG32I:
	    case PIXELFORMAT_RGBA32I:
	      return Int32Array;
	    case PIXELFORMAT_R32U:
	    case PIXELFORMAT_RG32U:
	    case PIXELFORMAT_RGBA32U:
	      return Uint32Array;
	    case PIXELFORMAT_R16I:
	    case PIXELFORMAT_RG16I:
	    case PIXELFORMAT_RGBA16I:
	      return Int16Array;
	    case PIXELFORMAT_R16U:
	    case PIXELFORMAT_RG16U:
	    case PIXELFORMAT_RGBA16U:
	    case PIXELFORMAT_RGB565:
	    case PIXELFORMAT_RGBA5551:
	    case PIXELFORMAT_RGBA4:
	    case PIXELFORMAT_R16F:
	    case PIXELFORMAT_RG16F:
	    case PIXELFORMAT_RGB16F:
	    case PIXELFORMAT_RGBA16F:
	      return Uint16Array;
	    case PIXELFORMAT_R8I:
	    case PIXELFORMAT_RG8I:
	    case PIXELFORMAT_RGBA8I:
	      return Int8Array;
	    default:
	      return Uint8Array;
	  }
	};

	/**
	 * List of distinct points.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PRIMITIVE_POINTS = 0;

	/**
	 * Discrete list of line segments.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PRIMITIVE_LINES = 1;

	/**
	 * List of points that are linked sequentially by line segments, with a closing line segment
	 * between the last and first points.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PRIMITIVE_LINELOOP = 2;

	/**
	 * List of points that are linked sequentially by line segments.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PRIMITIVE_LINESTRIP = 3;

	/**
	 * Discrete list of triangles.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PRIMITIVE_TRIANGLES = 4;

	/**
	 * Connected strip of triangles where a specified vertex forms a triangle using the previous two.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PRIMITIVE_TRISTRIP = 5;

	/**
	 * Connected fan of triangles where the first vertex forms triangles with the following pairs of vertices.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var PRIMITIVE_TRIFAN = 6;

	/**
	 * Vertex attribute to be treated as a position.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var SEMANTIC_POSITION = "POSITION";

	/**
	 * Vertex attribute to be treated as a normal.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var SEMANTIC_NORMAL = "NORMAL";

	/**
	 * Vertex attribute to be treated as a tangent.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var SEMANTIC_TANGENT = "TANGENT";

	/**
	 * Vertex attribute to be treated as skin blend weights.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var SEMANTIC_BLENDWEIGHT = "BLENDWEIGHT";

	/**
	 * Vertex attribute to be treated as skin blend indices.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var SEMANTIC_BLENDINDICES = "BLENDINDICES";

	/**
	 * Vertex attribute to be treated as a color.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var SEMANTIC_COLOR = "COLOR";

	// private semantic used for programmatic construction of individual texcoord semantics
	var SEMANTIC_TEXCOORD = "TEXCOORD";

	/**
	 * Vertex attribute to be treated as a texture coordinate (set 0).
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var SEMANTIC_TEXCOORD0 = "TEXCOORD0";

	/**
	 * Vertex attribute to be treated as a texture coordinate (set 1).
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var SEMANTIC_TEXCOORD1 = "TEXCOORD1";

	/**
	 * Vertex attribute to be treated as a texture coordinate (set 2).
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var SEMANTIC_TEXCOORD2 = "TEXCOORD2";

	/**
	 * Vertex attribute to be treated as a texture coordinate (set 3).
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var SEMANTIC_TEXCOORD3 = "TEXCOORD3";

	/**
	 * Vertex attribute to be treated as a texture coordinate (set 4).
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var SEMANTIC_TEXCOORD4 = "TEXCOORD4";

	/**
	 * Vertex attribute to be treated as a texture coordinate (set 5).
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var SEMANTIC_TEXCOORD5 = "TEXCOORD5";

	/**
	 * Vertex attribute to be treated as a texture coordinate (set 6).
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var SEMANTIC_TEXCOORD6 = "TEXCOORD6";

	/**
	 * Vertex attribute to be treated as a texture coordinate (set 7).
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var SEMANTIC_TEXCOORD7 = "TEXCOORD7";

	// private semantic used for programmatic construction of individual attr semantics
	var SEMANTIC_ATTR = "ATTR";

	/**
	 * Vertex attribute with a user defined semantic.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var SEMANTIC_ATTR0 = "ATTR0";

	/**
	 * Vertex attribute with a user defined semantic.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var SEMANTIC_ATTR1 = "ATTR1";

	/**
	 * Vertex attribute with a user defined semantic.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var SEMANTIC_ATTR2 = "ATTR2";

	/**
	 * Vertex attribute with a user defined semantic.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var SEMANTIC_ATTR3 = "ATTR3";

	/**
	 * Vertex attribute with a user defined semantic.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var SEMANTIC_ATTR4 = "ATTR4";

	/**
	 * Vertex attribute with a user defined semantic.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var SEMANTIC_ATTR5 = "ATTR5";

	/**
	 * Vertex attribute with a user defined semantic.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var SEMANTIC_ATTR6 = "ATTR6";

	/**
	 * Vertex attribute with a user defined semantic.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var SEMANTIC_ATTR7 = "ATTR7";

	/**
	 * Vertex attribute with a user defined semantic.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var SEMANTIC_ATTR8 = "ATTR8";

	/**
	 * Vertex attribute with a user defined semantic.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var SEMANTIC_ATTR9 = "ATTR9";

	/**
	 * Vertex attribute with a user defined semantic.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var SEMANTIC_ATTR10 = "ATTR10";

	/**
	 * Vertex attribute with a user defined semantic.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var SEMANTIC_ATTR11 = "ATTR11";

	/**
	 * Vertex attribute with a user defined semantic.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var SEMANTIC_ATTR12 = "ATTR12";

	/**
	 * Vertex attribute with a user defined semantic.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var SEMANTIC_ATTR13 = "ATTR13";

	/**
	 * Vertex attribute with a user defined semantic.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var SEMANTIC_ATTR14 = "ATTR14";

	/**
	 * Vertex attribute with a user defined semantic.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var SEMANTIC_ATTR15 = "ATTR15";
	var SHADERTAG_MATERIAL = 1;

	/**
	 * Don't change the stencil buffer value.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var STENCILOP_KEEP = 0;

	/**
	 * Set value to zero.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var STENCILOP_ZERO = 1;

	/**
	 * Replace value with the reference value (see {@link StencilParameters}).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var STENCILOP_REPLACE = 2;

	/**
	 * Increment the value.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var STENCILOP_INCREMENT = 3;

	/**
	 * Increment the value but wrap it to zero when it's larger than a maximum representable value.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var STENCILOP_INCREMENTWRAP = 4;

	/**
	 * Decrement the value.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var STENCILOP_DECREMENT = 5;

	/**
	 * Decrement the value but wrap it to a maximum representable value if the current value is 0.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var STENCILOP_DECREMENTWRAP = 6;

	/**
	 * Invert the value bitwise.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var STENCILOP_INVERT = 7;

	/**
	 * The texture is not in a locked state.
	 *
	 * @type {number}
	 */
	var TEXTURELOCK_NONE = 0;

	/**
	 * Read only. Any changes to the locked mip level's pixels will not update the texture.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var TEXTURELOCK_READ = 1;

	/**
	 * Write only. The contents of the specified mip level will be entirely replaced.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var TEXTURELOCK_WRITE = 2;

	/**
	 * Texture is a default type.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var TEXTURETYPE_DEFAULT = 'default';

	/**
	 * Texture stores high dynamic range data in RGBM format.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var TEXTURETYPE_RGBM = 'rgbm';

	/**
	 * Texture stores high dynamic range data in RGBE format.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var TEXTURETYPE_RGBE = 'rgbe';

	/**
	 * Texture stores high dynamic range data in RGBP encoding.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var TEXTURETYPE_RGBP = 'rgbp';

	/**
	 * Texture stores normalmap data swizzled in GGGR format. This is used for tangent space normal
	 * maps. The R component is stored in alpha and G is stored in RGB. This packing can result in
	 * higher quality when the texture data is compressed.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var TEXTURETYPE_SWIZZLEGGGR = 'swizzleGGGR';
	var TEXHINT_NONE = 0;
	var TEXHINT_SHADOWMAP = 1;
	var TEXHINT_ASSET = 2;
	var TEXHINT_LIGHTMAP = 3;
	var TEXTUREDIMENSION_1D = '1d';
	var TEXTUREDIMENSION_2D = '2d';
	var TEXTUREDIMENSION_2D_ARRAY = '2d-array';
	var TEXTUREDIMENSION_CUBE = 'cube';
	var TEXTUREDIMENSION_CUBE_ARRAY = 'cube-array';
	var TEXTUREDIMENSION_3D = '3d';
	var SAMPLETYPE_FLOAT = 0;
	var SAMPLETYPE_UNFILTERABLE_FLOAT = 1;
	var SAMPLETYPE_DEPTH = 2;
	var SAMPLETYPE_INT = 3;
	var SAMPLETYPE_UINT = 4;

	/**
	 * Texture data is not stored a specific projection format.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var TEXTUREPROJECTION_NONE = "none";

	/**
	 * Texture data is stored in cubemap projection format.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var TEXTUREPROJECTION_CUBE = "cube";

	/**
	 * Texture data is stored in equirectangular projection format.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var TEXTUREPROJECTION_EQUIRECT = "equirect";

	/**
	 * Texture data is stored in octahedral projection format.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var TEXTUREPROJECTION_OCTAHEDRAL = "octahedral";

	/**
	 * Shader source code uses GLSL language.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var SHADERLANGUAGE_GLSL = 'glsl';

	/**
	 * Shader source code uses WGSL language.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var SHADERLANGUAGE_WGSL = 'wgsl';

	/**
	 * Signed byte vertex element type.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var TYPE_INT8 = 0;

	/**
	 * Unsigned byte vertex element type.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var TYPE_UINT8 = 1;

	/**
	 * Signed short vertex element type.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var TYPE_INT16 = 2;

	/**
	 * Unsigned short vertex element type.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var TYPE_UINT16 = 3;

	/**
	 * Signed integer vertex element type.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var TYPE_INT32 = 4;

	/**
	 * Unsigned integer vertex element type.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var TYPE_UINT32 = 5;

	/**
	 * Floating point vertex element type.
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var TYPE_FLOAT32 = 6;

	/**
	 * 16-bit floating point vertex element type (not supported by WebGL1).
	 *
	 * @type {number}
	 * @category Graphics
	 */
	var TYPE_FLOAT16 = 7;

	// Uniform types
	var UNIFORMTYPE_BOOL = 0;
	var UNIFORMTYPE_INT = 1;
	var UNIFORMTYPE_FLOAT = 2;
	var UNIFORMTYPE_VEC2 = 3;
	var UNIFORMTYPE_VEC3 = 4;
	var UNIFORMTYPE_VEC4 = 5;
	var UNIFORMTYPE_IVEC2 = 6;
	var UNIFORMTYPE_IVEC3 = 7;
	var UNIFORMTYPE_IVEC4 = 8;
	var UNIFORMTYPE_BVEC2 = 9;
	var UNIFORMTYPE_BVEC3 = 10;
	var UNIFORMTYPE_BVEC4 = 11;
	var UNIFORMTYPE_MAT2 = 12;
	var UNIFORMTYPE_MAT3 = 13;
	var UNIFORMTYPE_MAT4 = 14;
	var UNIFORMTYPE_TEXTURE2D = 15;
	var UNIFORMTYPE_TEXTURECUBE = 16;
	var UNIFORMTYPE_FLOATARRAY = 17;
	var UNIFORMTYPE_TEXTURE2D_SHADOW = 18;
	var UNIFORMTYPE_TEXTURECUBE_SHADOW = 19;
	var UNIFORMTYPE_TEXTURE3D = 20;
	var UNIFORMTYPE_VEC2ARRAY = 21;
	var UNIFORMTYPE_VEC3ARRAY = 22;
	var UNIFORMTYPE_VEC4ARRAY = 23;
	var UNIFORMTYPE_MAT4ARRAY = 24;
	var UNIFORMTYPE_TEXTURE2D_ARRAY = 25;

	// Unsigned uniform types
	var UNIFORMTYPE_UINT = 26;
	var UNIFORMTYPE_UVEC2 = 27;
	var UNIFORMTYPE_UVEC3 = 28;
	var UNIFORMTYPE_UVEC4 = 29;

	// Integer uniform array types
	var UNIFORMTYPE_INTARRAY = 30;
	var UNIFORMTYPE_UINTARRAY = 31;
	var UNIFORMTYPE_BOOLARRAY = 32;
	var UNIFORMTYPE_IVEC2ARRAY = 33;
	var UNIFORMTYPE_UVEC2ARRAY = 34;
	var UNIFORMTYPE_BVEC2ARRAY = 35;
	var UNIFORMTYPE_IVEC3ARRAY = 36;
	var UNIFORMTYPE_UVEC3ARRAY = 37;
	var UNIFORMTYPE_BVEC3ARRAY = 38;
	var UNIFORMTYPE_IVEC4ARRAY = 39;
	var UNIFORMTYPE_UVEC4ARRAY = 40;
	var UNIFORMTYPE_BVEC4ARRAY = 41;

	// Integer texture types
	var UNIFORMTYPE_ITEXTURE2D = 42;
	var UNIFORMTYPE_UTEXTURE2D = 43;
	var UNIFORMTYPE_ITEXTURECUBE = 44;
	var UNIFORMTYPE_UTEXTURECUBE = 45;
	var UNIFORMTYPE_ITEXTURE3D = 46;
	var UNIFORMTYPE_UTEXTURE3D = 47;
	var UNIFORMTYPE_ITEXTURE2D_ARRAY = 48;
	var UNIFORMTYPE_UTEXTURE2D_ARRAY = 49;
	var uniformTypeToName = [
	// Uniforms
	'bool', 'int', 'float', 'vec2', 'vec3', 'vec4', 'ivec2', 'ivec3', 'ivec4', 'bvec2', 'bvec3', 'bvec4', 'mat2', 'mat3', 'mat4', 'sampler2D', 'samplerCube', '',
	// not directly handled: UNIFORMTYPE_FLOATARRAY
	'sampler2DShadow', 'samplerCubeShadow', 'sampler3D', '',
	// not directly handled: UNIFORMTYPE_VEC2ARRAY
	'',
	// not directly handled: UNIFORMTYPE_VEC3ARRAY
	'',
	// not directly handled: UNIFORMTYPE_VEC4ARRAY
	'',
	// not directly handled: UNIFORMTYPE_MAT4ARRAY
	'sampler2DArray', 'uint', 'uvec2', 'uvec3', 'uvec4', '',
	// not directly handled: UNIFORMTYPE_INTARRAY
	'',
	// not directly handled: UNIFORMTYPE_UINTARRAY
	'',
	// not directly handled: UNIFORMTYPE_BOOLARRAY
	'',
	// not directly handled: UNIFORMTYPE_IVEC2ARRAY
	'',
	// not directly handled: UNIFORMTYPE_UVEC2ARRAY
	'',
	// not directly handled: UNIFORMTYPE_BVEC2ARRAY
	'',
	// not directly handled: UNIFORMTYPE_IVEC3ARRAY
	'',
	// not directly handled: UNIFORMTYPE_UVEC3ARRAY
	'',
	// not directly handled: UNIFORMTYPE_BVEC3ARRAY
	'',
	// not directly handled: UNIFORMTYPE_IVEC4ARRAY
	'',
	// not directly handled: UNIFORMTYPE_UVEC4ARRAY
	'',
	// not directly handled: UNIFORMTYPE_BVEC4ARRAY
	'isampler2D', 'usampler2D', 'isamplerCube', 'usamplerCube', 'isampler3D', 'usampler3D', 'isampler2DArray', 'usampler2DArray'];

	// Map to convert uniform type to storage type, used in uniform-buffer.js
	var uniformTypeToStorage = new Uint8Array([TYPE_INT32,
	// UNIFORMTYPE_BOOL
	TYPE_INT32,
	// UNIFORMTYPE_INT
	TYPE_FLOAT32,
	// UNIFORMTYPE_FLOAT
	TYPE_FLOAT32,
	// UNIFORMTYPE_VEC2
	TYPE_FLOAT32,
	// UNIFORMTYPE_VEC3
	TYPE_FLOAT32,
	// UNIFORMTYPE_VEC4
	TYPE_INT32,
	// UNIFORMTYPE_IVEC2
	TYPE_INT32,
	// UNIFORMTYPE_IVEC3
	TYPE_INT32,
	// UNIFORMTYPE_IVEC4
	TYPE_INT32,
	// UNIFORMTYPE_BVEC2
	TYPE_INT32,
	// UNIFORMTYPE_BVEC3
	TYPE_INT32,
	// UNIFORMTYPE_BVEC4
	TYPE_FLOAT32,
	// UNIFORMTYPE_MAT2
	TYPE_FLOAT32,
	// UNIFORMTYPE_MAT3
	TYPE_FLOAT32,
	// UNIFORMTYPE_MAT4
	TYPE_INT32,
	// UNIFORMTYPE_TEXTURE2D
	TYPE_INT32,
	// UNIFORMTYPE_TEXTURECUBE
	TYPE_FLOAT32,
	// UNIFORMTYPE_FLOATARRAY
	TYPE_INT32,
	// UNIFORMTYPE_TEXTURE2D_SHADOW
	TYPE_INT32,
	// UNIFORMTYPE_TEXTURECUBE_SHADOW
	TYPE_INT32,
	// UNIFORMTYPE_TEXTURE3D
	TYPE_FLOAT32,
	// UNIFORMTYPE_VEC2ARRAY
	TYPE_FLOAT32,
	// UNIFORMTYPE_VEC3ARRAY
	TYPE_FLOAT32,
	// UNIFORMTYPE_VEC4ARRAY
	TYPE_FLOAT32,
	// UNIFORMTYPE_MAT4ARRAY
	TYPE_INT32,
	// UNIFORMTYPE_TEXTURE2D_ARRAY
	TYPE_UINT32,
	// UNIFORMTYPE_UINT
	TYPE_UINT32,
	// UNIFORMTYPE_UVEC2
	TYPE_UINT32,
	// UNIFORMTYPE_UVEC3
	TYPE_UINT32,
	// UNIFORMTYPE_UVEC4
	TYPE_INT32,
	// UNIFORMTYPE_INTARRAY
	TYPE_UINT32,
	// UNIFORMTYPE_UINTARRAY
	TYPE_INT32,
	// UNIFORMTYPE_BOOLARRAY
	TYPE_INT32,
	// UNIFORMTYPE_IVEC2ARRAY
	TYPE_UINT32,
	// UNIFORMTYPE_UVEC2ARRAY
	TYPE_INT32,
	// UNIFORMTYPE_BVEC2ARRAY
	TYPE_INT32,
	// UNIFORMTYPE_IVEC3ARRAY
	TYPE_UINT32,
	// UNIFORMTYPE_UVEC3ARRAY
	TYPE_INT32,
	// UNIFORMTYPE_BVEC3ARRAY
	TYPE_INT32,
	// UNIFORMTYPE_IVEC4ARRAY
	TYPE_UINT32,
	// UNIFORMTYPE_UVEC4ARRAY
	TYPE_INT32,
	// UNIFORMTYPE_BVEC4ARRAY
	TYPE_INT32,
	// UNIFORMTYPE_ITEXTURE2D
	TYPE_UINT32,
	// UNIFORMTYPE_UTEXTURE2D
	TYPE_INT32,
	// UNIFORMTYPE_ITEXTURECUBE
	TYPE_UINT32,
	// UNIFORMTYPE_UTEXTURECUBE
	TYPE_INT32,
	// UNIFORMTYPE_ITEXTURE3D
	TYPE_UINT32,
	// UNIFORMTYPE_UTEXTURE3D
	TYPE_INT32,
	// UNIFORMTYPE_ITEXTURE2D_ARRAY
	TYPE_UINT32 // UNIFORMTYPE_UTEXTURE2D_ARRAY
	]);

	/**
	 * A WebGL 1 device type.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var DEVICETYPE_WEBGL1 = 'webgl1';

	/**
	 * A WebGL 2 device type.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var DEVICETYPE_WEBGL2 = 'webgl2';

	/**
	 * A WebGPU device type.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var DEVICETYPE_WEBGPU = 'webgpu';

	/**
	 * A Null device type.
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var DEVICETYPE_NULL = 'null';

	// (bit-flags) shader stages for resource visibility on the GPU
	var SHADERSTAGE_VERTEX = 1;
	var SHADERSTAGE_FRAGMENT = 2;
	var SHADERSTAGE_COMPUTE = 4;

	// indices of commonly used bind groups
	// sorted in a way that any trailing bind groups can be unused in any render pass
	var BINDGROUP_MESH = 0;
	var BINDGROUP_VIEW = 1;

	// names of bind groups
	var bindGroupNames = ['mesh', 'view'];

	// name of the default uniform buffer slot in a bind group
	var UNIFORM_BUFFER_DEFAULT_SLOT_NAME = 'default';

	// map of engine TYPE_*** enums to their corresponding typed array constructors and byte sizes
	var typedArrayTypes = [Int8Array, Uint8Array, Int16Array, Uint16Array, Int32Array, Uint32Array, Float32Array, Uint16Array];
	var typedArrayTypesByteSize = [1, 1, 2, 2, 4, 4, 4, 2];
	var vertexTypesNames = ['INT8', 'UINT8', 'INT16', 'UINT16', 'INT32', 'UINT32', 'FLOAT32', 'FLOAT16'];

	// map of typed array to engine TYPE_***
	var typedArrayToType = {
	  "Int8Array": TYPE_INT8,
	  "Uint8Array": TYPE_UINT8,
	  "Int16Array": TYPE_INT16,
	  "Uint16Array": TYPE_UINT16,
	  "Int32Array": TYPE_INT32,
	  "Uint32Array": TYPE_UINT32,
	  "Float32Array": TYPE_FLOAT32
	};

	// map of engine INDEXFORMAT_*** to their corresponding typed array constructors and byte sizes
	var typedArrayIndexFormats = [Uint8Array, Uint16Array, Uint32Array];
	var typedArrayIndexFormatsByteSize = [1, 2, 4];

	/**
	 * Map of engine semantics into location on device in range 0..15 (note - semantics mapping to the
	 * same location cannot be used at the same time) organized in a way that ATTR0-ATTR7 do not
	 * overlap with common important semantics.
	 *
	 * @type {object}
	 * @ignore
	 * @category Graphics
	 */
	var semanticToLocation = {};
	semanticToLocation[SEMANTIC_POSITION] = 0;
	semanticToLocation[SEMANTIC_NORMAL] = 1;
	semanticToLocation[SEMANTIC_BLENDWEIGHT] = 2;
	semanticToLocation[SEMANTIC_BLENDINDICES] = 3;
	semanticToLocation[SEMANTIC_COLOR] = 4;
	semanticToLocation[SEMANTIC_TEXCOORD0] = 5;
	semanticToLocation[SEMANTIC_TEXCOORD1] = 6;
	semanticToLocation[SEMANTIC_TEXCOORD2] = 7;
	semanticToLocation[SEMANTIC_TEXCOORD3] = 8;
	semanticToLocation[SEMANTIC_TEXCOORD4] = 9;
	semanticToLocation[SEMANTIC_TEXCOORD5] = 10;
	semanticToLocation[SEMANTIC_TEXCOORD6] = 11;
	semanticToLocation[SEMANTIC_TEXCOORD7] = 12;
	semanticToLocation[SEMANTIC_TANGENT] = 13;
	semanticToLocation[SEMANTIC_ATTR0] = 0;
	semanticToLocation[SEMANTIC_ATTR1] = 1;
	semanticToLocation[SEMANTIC_ATTR2] = 2;
	semanticToLocation[SEMANTIC_ATTR3] = 3;
	semanticToLocation[SEMANTIC_ATTR4] = 4;
	semanticToLocation[SEMANTIC_ATTR5] = 5;
	semanticToLocation[SEMANTIC_ATTR6] = 6;
	semanticToLocation[SEMANTIC_ATTR7] = 7;
	semanticToLocation[SEMANTIC_ATTR8] = 8;
	semanticToLocation[SEMANTIC_ATTR9] = 9;
	semanticToLocation[SEMANTIC_ATTR10] = 10;
	semanticToLocation[SEMANTIC_ATTR11] = 11;
	semanticToLocation[SEMANTIC_ATTR12] = 12;
	semanticToLocation[SEMANTIC_ATTR13] = 13;
	semanticToLocation[SEMANTIC_ATTR14] = 14;
	semanticToLocation[SEMANTIC_ATTR15] = 15;

	/**
	 * Chunk API versions
	 *
	 * @type {string}
	 * @category Graphics
	 */
	var CHUNKAPI_1_51 = '1.51';
	var CHUNKAPI_1_55 = '1.55';
	var CHUNKAPI_1_56 = '1.56';
	var CHUNKAPI_1_57 = '1.57';
	var CHUNKAPI_1_58 = '1.58';
	var CHUNKAPI_1_60 = '1.60';
	var CHUNKAPI_1_62 = '1.62';
	var CHUNKAPI_1_65 = '1.65';

	/**
	 * BitPacking API - functionality for operating on values stored as bits in a number.
	 *
	 * @namespace
	 * @ignore
	 */
	var BitPacking = {
	  /**
	   * Sets a value to specified bits of a number.
	   *
	   * @param {number} storage - Number to store the bits into.
	   * @param {number} value - Value to store.
	   * @param {number} shift - Number of bits to shift the value.
	   * @param {number} [mask] - Mask for the value to limit the number of storage bits. Defaults to 1.
	   * @returns {number} Returns the storage updated with the value.
	   */
	  set: function set(storage, value, shift, mask) {
	    if (mask === void 0) {
	      mask = 1;
	    }
	    // clear the space
	    var data = storage & ~(mask << shift);

	    // set the bits
	    return data | value << shift;
	  },
	  /**
	   * Gets the value of specified bits from a number.
	   *
	   * @param {number} storage - Number to extract the bits from.
	   * @param {number} shift - Number of bits to shift the mask.
	   * @param {number} [mask] - Mask for the value to limit the number of storage bits. Defaults to 1.
	   * @returns {number} Returns the extracted value.
	   */
	  get: function get(storage, shift, mask) {
	    if (mask === void 0) {
	      mask = 1;
	    }
	    return storage >> shift & mask;
	  },
	  /**
	   * Tests if all specified bits are set.
	   *
	   * @param {number} storage - Number to test.
	   * @param {number} shift - Number of bits to shift the mask.
	   * @param {number} [mask] - Mask to limit the number of storage bits. Defaults to 1.
	   * @returns {boolean} Returns true if all bits in the mask are set in the storage.
	   */
	  all: function all(storage, shift, mask) {
	    if (mask === void 0) {
	      mask = 1;
	    }
	    var shifted = mask << shift;
	    return (storage & shifted) === shifted;
	  },
	  /**
	   * Tests if any specified bits are set.
	   *
	   * @param {number} storage - Number to test.
	   * @param {number} shift - Number of bits to shift the mask.
	   * @param {number} [mask] - Mask to limit the number of storage bits. Defaults to 1.
	   * @returns {boolean} Returns true if any bits in the mask are set in the storage.
	   */
	  any: function any(storage, shift, mask) {
	    if (mask === void 0) {
	      mask = 1;
	    }
	    return (storage & mask << shift) !== 0;
	  }
	};

	var _class$3;

	// masks (to only keep relevant bits)
	var opMask = 7;
	var factorMask = 15;

	// shifts values to where individual parts are stored
	var colorOpShift = 0; // 00 - 02 (3bits)
	var colorSrcFactorShift = 3; // 03 - 06 (4bits)
	var colorDstFactorShift = 7; // 07 - 10 (4bits)
	var alphaOpShift = 11; // 11 - 13 (3bits)
	var alphaSrcFactorShift = 14; // 14 - 17 (4bits)
	var alphaDstFactorShift = 18; // 18 - 21 (4bits)
	var redWriteShift = 22; // 22 (1 bit)
	var greenWriteShift = 23; // 23 (1 bit)
	var blueWriteShift = 24; // 24 (1 bit)
	var alphaWriteShift = 25; // 25 (1 bit)
	var blendShift = 26; // 26 (1 bit)

	// combined values access
	var allWriteMasks = 15;
	var allWriteShift = redWriteShift;
	/**
	 * BlendState is a descriptor that defines how output of fragment shader is written and blended
	 * into render target. A blend state can be set on a material using {@link Material#blendState},
	 * or in some cases on the graphics device using {@link GraphicsDevice#setBlendState}.
	 *
	 * For the best performance, do not modify blend state after it has been created, but create
	 * multiple blend states and assign them to the material or graphics device as needed.
	 *
	 * @category Graphics
	 */
	var BlendState = /*#__PURE__*/function () {
	  /**
	   * Create a new BlendState instance.
	   *
	   * All factor parameters can take the following values:
	   *
	   * - {@link BLENDMODE_ZERO}
	   * - {@link BLENDMODE_ONE}
	   * - {@link BLENDMODE_SRC_COLOR}
	   * - {@link BLENDMODE_ONE_MINUS_SRC_COLOR}
	   * - {@link BLENDMODE_DST_COLOR}
	   * - {@link BLENDMODE_ONE_MINUS_DST_COLOR}
	   * - {@link BLENDMODE_SRC_ALPHA}
	   * - {@link BLENDMODE_SRC_ALPHA_SATURATE}
	   * - {@link BLENDMODE_ONE_MINUS_SRC_ALPHA}
	   * - {@link BLENDMODE_DST_ALPHA}
	   * - {@link BLENDMODE_ONE_MINUS_DST_ALPHA}
	   * - {@link BLENDMODE_CONSTANT}
	   * - {@link BLENDMODE_ONE_MINUS_CONSTANT}
	   *
	   * All op parameters can take the following values:
	   *
	   * - {@link BLENDEQUATION_ADD}
	   * - {@link BLENDEQUATION_SUBTRACT}
	   * - {@link BLENDEQUATION_REVERSE_SUBTRACT}
	   * - {@link BLENDEQUATION_MIN}
	   * - {@link BLENDEQUATION_MAX}
	   *
	   * Note that MIN and MAX operations on WebGL platform require either EXT_blend_minmax or WebGL2
	   * to work (check device.extBlendMinmax).
	   *
	   * @param {boolean} [blend] - Enables or disables blending. Defaults to false.
	   * @param {number} [colorOp] - Configures color blending operation. Defaults to
	   * {@link BLENDEQUATION_ADD}.
	   * @param {number} [colorSrcFactor] - Configures source color blending factor. Defaults to
	   * {@link BLENDMODE_ONE}.
	   * @param {number} [colorDstFactor] - Configures destination color blending factor. Defaults to
	   * {@link BLENDMODE_ZERO}.
	   * @param {number} [alphaOp] - Configures alpha blending operation. Defaults to
	   * {@link BLENDEQUATION_ADD}.
	   * @param {number} [alphaSrcFactor] - Configures source alpha blending factor. Defaults to
	   * {@link BLENDMODE_ONE}.
	   * @param {number} [alphaDstFactor] - Configures destination alpha blending factor. Defaults to
	   * {@link BLENDMODE_ZERO}.
	   * @param {boolean} [redWrite] - True to enable writing of the red channel and false otherwise.
	   * Defaults to true.
	   * @param {boolean} [greenWrite] - True to enable writing of the green channel and false
	   * otherwise. Defaults to true.
	   * @param {boolean} [blueWrite] - True to enable writing of the blue channel and false otherwise.
	   * Defaults to true.
	   * @param {boolean} [alphaWrite] - True to enable writing of the alpha channel and false
	   * otherwise. Defaults to true.
	   */
	  function BlendState(blend, colorOp, colorSrcFactor, colorDstFactor, alphaOp, alphaSrcFactor, alphaDstFactor, redWrite, greenWrite, blueWrite, alphaWrite) {
	    if (blend === void 0) {
	      blend = false;
	    }
	    if (colorOp === void 0) {
	      colorOp = BLENDEQUATION_ADD;
	    }
	    if (colorSrcFactor === void 0) {
	      colorSrcFactor = BLENDMODE_ONE;
	    }
	    if (colorDstFactor === void 0) {
	      colorDstFactor = BLENDMODE_ZERO;
	    }
	    if (redWrite === void 0) {
	      redWrite = true;
	    }
	    if (greenWrite === void 0) {
	      greenWrite = true;
	    }
	    if (blueWrite === void 0) {
	      blueWrite = true;
	    }
	    if (alphaWrite === void 0) {
	      alphaWrite = true;
	    }
	    /**
	     * Bitfield representing the blend state for render target 0.
	     *
	     * @private
	     */
	    this.target0 = 0;
	    this.setColorBlend(colorOp, colorSrcFactor, colorDstFactor);
	    this.setAlphaBlend(alphaOp != null ? alphaOp : colorOp, alphaSrcFactor != null ? alphaSrcFactor : colorSrcFactor, alphaDstFactor != null ? alphaDstFactor : colorDstFactor);
	    this.setColorWrite(redWrite, greenWrite, blueWrite, alphaWrite);
	    this.blend = blend;
	  }

	  /**
	   * Enables or disables blending.
	   *
	   * @type {boolean}
	   */
	  var _proto = BlendState.prototype;
	  _proto.setColorBlend = function setColorBlend(op, srcFactor, dstFactor) {
	    this.target0 = BitPacking.set(this.target0, op, colorOpShift, opMask);
	    this.target0 = BitPacking.set(this.target0, srcFactor, colorSrcFactorShift, factorMask);
	    this.target0 = BitPacking.set(this.target0, dstFactor, colorDstFactorShift, factorMask);
	  };
	  _proto.setAlphaBlend = function setAlphaBlend(op, srcFactor, dstFactor) {
	    this.target0 = BitPacking.set(this.target0, op, alphaOpShift, opMask);
	    this.target0 = BitPacking.set(this.target0, srcFactor, alphaSrcFactorShift, factorMask);
	    this.target0 = BitPacking.set(this.target0, dstFactor, alphaDstFactorShift, factorMask);
	  };
	  _proto.setColorWrite = function setColorWrite(redWrite, greenWrite, blueWrite, alphaWrite) {
	    this.redWrite = redWrite;
	    this.greenWrite = greenWrite;
	    this.blueWrite = blueWrite;
	    this.alphaWrite = alphaWrite;
	  };
	  /**
	   * Copies the contents of a source blend state to this blend state.
	   *
	   * @param {BlendState} rhs - A blend state to copy from.
	   * @returns {BlendState} Self for chaining.
	   */
	  _proto.copy = function copy(rhs) {
	    this.target0 = rhs.target0;
	    return this;
	  }

	  /**
	   * Returns an identical copy of the specified blend state.
	   *
	   * @returns {this} The result of the cloning.
	   */;
	  _proto.clone = function clone() {
	    var clone = new this.constructor();
	    return clone.copy(this);
	  };
	  /**
	   * Reports whether two BlendStates are equal.
	   *
	   * @param {BlendState} rhs - The blend state to compare to.
	   * @returns {boolean} True if the blend states are equal and false otherwise.
	   */
	  _proto.equals = function equals(rhs) {
	    return this.target0 === rhs.target0;
	  }

	  /**
	   * A blend state that has blending disabled and writes to all color channels.
	   *
	   * @type {BlendState}
	   * @readonly
	   */;
	  _createClass(BlendState, [{
	    key: "blend",
	    get: function get() {
	      return BitPacking.all(this.target0, blendShift);
	    },
	    set: function set(value) {
	      this.target0 = BitPacking.set(this.target0, value ? 1 : 0, blendShift);
	    }
	  }, {
	    key: "colorOp",
	    get: function get() {
	      return BitPacking.get(this.target0, colorOpShift, opMask);
	    }
	  }, {
	    key: "colorSrcFactor",
	    get: function get() {
	      return BitPacking.get(this.target0, colorSrcFactorShift, factorMask);
	    }
	  }, {
	    key: "colorDstFactor",
	    get: function get() {
	      return BitPacking.get(this.target0, colorDstFactorShift, factorMask);
	    }
	  }, {
	    key: "alphaOp",
	    get: function get() {
	      return BitPacking.get(this.target0, alphaOpShift, opMask);
	    }
	  }, {
	    key: "alphaSrcFactor",
	    get: function get() {
	      return BitPacking.get(this.target0, alphaSrcFactorShift, factorMask);
	    }
	  }, {
	    key: "alphaDstFactor",
	    get: function get() {
	      return BitPacking.get(this.target0, alphaDstFactorShift, factorMask);
	    }
	  }, {
	    key: "redWrite",
	    get: function get() {
	      return BitPacking.all(this.target0, redWriteShift);
	    },
	    set: function set(value) {
	      this.target0 = BitPacking.set(this.target0, value ? 1 : 0, redWriteShift);
	    }
	  }, {
	    key: "greenWrite",
	    get: function get() {
	      return BitPacking.all(this.target0, greenWriteShift);
	    },
	    set: function set(value) {
	      this.target0 = BitPacking.set(this.target0, value ? 1 : 0, greenWriteShift);
	    }
	  }, {
	    key: "blueWrite",
	    get: function get() {
	      return BitPacking.all(this.target0, blueWriteShift);
	    },
	    set: function set(value) {
	      this.target0 = BitPacking.set(this.target0, value ? 1 : 0, blueWriteShift);
	    }
	  }, {
	    key: "alphaWrite",
	    get: function get() {
	      return BitPacking.all(this.target0, alphaWriteShift);
	    },
	    set: function set(value) {
	      this.target0 = BitPacking.set(this.target0, value ? 1 : 0, alphaWriteShift);
	    }
	  }, {
	    key: "allWrite",
	    get: function get() {
	      // return a number with all 4 bits, for fast compare
	      return BitPacking.get(this.target0, allWriteShift, allWriteMasks);
	    }
	  }, {
	    key: "key",
	    get: function get() {
	      return this.target0;
	    }
	  }]);
	  return BlendState;
	}();
	_class$3 = BlendState;
	BlendState.NOBLEND = Object.freeze(new _class$3());
	/**
	 * A blend state that does not write to color channels.
	 *
	 * @type {BlendState}
	 * @readonly
	 */
	BlendState.NOWRITE = Object.freeze(new _class$3(undefined, undefined, undefined, undefined, undefined, undefined, undefined, false, false, false, false));
	/**
	 * A blend state that does simple translucency using alpha channel.
	 *
	 * @type {BlendState}
	 * @readonly
	 */
	BlendState.ALPHABLEND = Object.freeze(new _class$3(true, BLENDEQUATION_ADD, BLENDMODE_SRC_ALPHA, BLENDMODE_ONE_MINUS_SRC_ALPHA));
	/**
	 * A blend state that does simple additive blending.
	 *
	 * @type {BlendState}
	 * @readonly
	 */
	BlendState.ADDBLEND = Object.freeze(new _class$3(true, BLENDEQUATION_ADD, BLENDMODE_ONE, BLENDMODE_ONE));

	/**
	 * A cache for assigning unique numerical ids to strings.
	 *
	 * @ignore
	 */
	var StringIds = /*#__PURE__*/function () {
	  function StringIds() {
	    /** @type {Map<string, number>} */
	    this.map = new Map();
	    /** @type {number} */
	    this.id = 0;
	  }
	  var _proto = StringIds.prototype;
	  /**
	   * Get the id for the given name. If the name has not been seen before, it will be assigned a new
	   * id.
	   *
	   * @param {string} name - The name to get the id for.
	   * @returns {number} The id for the given name.
	   */
	  _proto.get = function get(name) {
	    var value = this.map.get(name);
	    if (value === undefined) {
	      value = this.id++;
	      this.map.set(name, value);
	    }
	    return value;
	  };
	  return StringIds;
	}();

	var _class$2;
	var stringIds$4 = new StringIds();

	// masks (to only keep relevant bits)
	var funcMask = 7;

	// shifts values to where individual parts are stored
	var funcShift = 0; // 00 - 02 (3bits)
	var writeShift = 3; // 03 - 03 (1bit)

	/**
	 * DepthState is a descriptor that defines how the depth value of the fragment is used by the
	 * rendering pipeline. A depth state can be set on a material using {@link Material#depthState},
	 * or in some cases on the graphics device using {@link GraphicsDevice#setDepthState}.
	 *
	 * For the best performance, do not modify depth state after it has been created, but create
	 * multiple depth states and assign them to the material or graphics device as needed.
	 *
	 * @category Graphics
	 */
	var DepthState = /*#__PURE__*/function () {
	  /**
	   * Create a new Depth State instance.
	   *
	   * @param {number} func - Controls how the depth of the fragment is compared against the
	   * current depth contained in the depth buffer. See {@link DepthState#func} for details.
	   * Defaults to {@link FUNC_LESSEQUAL}.
	   * @param {boolean} write - If true, depth values are written to the depth buffer of the
	   * currently active render target. Defaults to true.
	   */
	  function DepthState(func, write) {
	    if (func === void 0) {
	      func = FUNC_LESSEQUAL;
	    }
	    if (write === void 0) {
	      write = true;
	    }
	    /**
	     * Bitfield representing the depth state.
	     *
	     * @private
	     */
	    this.data = 0;
	    this._depthBias = 0;
	    this._depthBiasSlope = 0;
	    /**
	     * A unique number representing the depth state. You can use this number to quickly compare
	     * two depth states for equality. The key is always maintained valid without a dirty flag,
	     * to avoid condition check at runtime, considering these change rarely.
	     *
	     * @type {number}
	     */
	    this.key = 0;
	    this.func = func;
	    this.write = write;
	  }

	  /**
	   * If true, a shader fragment is only written to the current render target if it passes the depth
	   * test. If false, it is written regardless of what is in the depth buffer. Note that when depth
	   * testing is disabled, writes to the depth buffer are also disabled. Defaults to true.
	   *
	   * @type {boolean}
	   */
	  var _proto = DepthState.prototype;
	  /**
	   * Copies the contents of a source depth state to this depth state.
	   *
	   * @param {DepthState} rhs - A depth state to copy from.
	   * @returns {DepthState} Self for chaining.
	   */
	  _proto.copy = function copy(rhs) {
	    this.data = rhs.data;
	    this._depthBias = rhs._depthBias;
	    this._depthBiasSlope = rhs._depthBiasSlope;
	    this.key = rhs.key;
	    return this;
	  }

	  /**
	   * Returns an identical copy of the specified depth state.
	   *
	   * @returns {this} The result of the cloning.
	   */;
	  _proto.clone = function clone() {
	    var clone = new this.constructor();
	    return clone.copy(this);
	  };
	  _proto.updateKey = function updateKey() {
	    var data = this.data,
	      _depthBias = this._depthBias,
	      _depthBiasSlope = this._depthBiasSlope;
	    var key = data + "-" + _depthBias + "-" + _depthBiasSlope;

	    // convert string to a unique number
	    this.key = stringIds$4.get(key);
	  }

	  /**
	   * Reports whether two DepthStates are equal.
	   *
	   * @param {DepthState} rhs - The depth state to compare to.
	   * @returns {boolean} True if the depth states are equal and false otherwise.
	   */;
	  _proto.equals = function equals(rhs) {
	    return this.key === rhs.key;
	  }

	  /**
	   * A default depth state that has the depth testing function set to {@link FUNC_LESSEQUAL} and depth writes enabled.
	   *
	   * @type {DepthState}
	   * @readonly
	   */;
	  _createClass(DepthState, [{
	    key: "test",
	    get: function get() {
	      return this.func !== FUNC_ALWAYS;
	    }

	    /**
	     * If true, shader write a depth value to the depth buffer of the currently active render
	     * target. If false, no depth value is written.
	     *
	     * @type {boolean}
	     */,
	    set: function set(value) {
	      this.func = value ? FUNC_LESSEQUAL : FUNC_ALWAYS;
	      this.updateKey();
	    }
	  }, {
	    key: "write",
	    get: function get() {
	      return BitPacking.all(this.data, writeShift);
	    }

	    /**
	     * Controls how the depth of the fragment is compared against the current depth contained in
	     * the depth buffer. Can be:
	     *
	     * - {@link FUNC_NEVER}: don't draw
	     * - {@link FUNC_LESS}: draw if new depth < depth buffer
	     * - {@link FUNC_EQUAL}: draw if new depth == depth buffer
	     * - {@link FUNC_LESSEQUAL}: draw if new depth <= depth buffer
	     * - {@link FUNC_GREATER}: draw if new depth > depth buffer
	     * - {@link FUNC_NOTEQUAL}: draw if new depth != depth buffer
	     * - {@link FUNC_GREATEREQUAL}: draw if new depth >= depth buffer
	     * - {@link FUNC_ALWAYS}: always draw
	     *
	     * @type {number}
	     */,
	    set: function set(value) {
	      this.data = BitPacking.set(this.data, value ? 1 : 0, writeShift);
	      this.updateKey();
	    }
	  }, {
	    key: "func",
	    get: function get() {
	      return BitPacking.get(this.data, funcShift, funcMask);
	    }

	    /**
	     * Constant depth bias added to each fragment's depth. Useful for decals to prevent z-fighting.
	     * Typically a small negative value (-0.1) is used to render the mesh slightly closer to the
	     * camera. Defaults to 0.
	     *
	     * @type {number}
	     */,
	    set: function set(value) {
	      this.data = BitPacking.set(this.data, value, funcShift, funcMask);
	      this.updateKey();
	    }
	  }, {
	    key: "depthBias",
	    get: function get() {
	      return this._depthBias;
	    }

	    /**
	     * Depth bias that scales with the fragment’s slope. Defaults to 0.
	     *
	     * @type {number}
	     */,
	    set: function set(value) {
	      this._depthBias = value;
	      this.updateKey();
	    }
	  }, {
	    key: "depthBiasSlope",
	    get: function get() {
	      return this._depthBiasSlope;
	    },
	    set: function set(value) {
	      this._depthBiasSlope = value;
	      this.updateKey();
	    }
	  }]);
	  return DepthState;
	}();
	_class$2 = DepthState;
	DepthState.DEFAULT = Object.freeze(new _class$2());
	/**
	 * A depth state that always passes the fragment but does not write depth to the depth buffer.
	 *
	 * @type {DepthState}
	 * @readonly
	 */
	DepthState.NODEPTH = Object.freeze(new _class$2(FUNC_ALWAYS, false));
	/**
	 * A depth state that always passes the fragment and writes depth to the depth buffer.
	 *
	 * @type {DepthState}
	 * @readonly
	 */
	DepthState.WRITEDEPTH = Object.freeze(new _class$2(FUNC_ALWAYS, true));

	var Version = /*#__PURE__*/function () {
	  function Version() {
	    // Set the variables
	    this.globalId = 0;
	    this.revision = 0;
	  }
	  var _proto = Version.prototype;
	  _proto.equals = function equals(other) {
	    return this.globalId === other.globalId && this.revision === other.revision;
	  };
	  _proto.copy = function copy(other) {
	    this.globalId = other.globalId;
	    this.revision = other.revision;
	  };
	  _proto.reset = function reset() {
	    this.globalId = 0;
	    this.revision = 0;
	  };
	  return Version;
	}();

	var idCounter = 0;
	var VersionedObject = /*#__PURE__*/function () {
	  function VersionedObject() {
	    // Increment the global object ID counter
	    idCounter++;

	    // Create a version for this object
	    this.version = new Version();

	    // Set the unique object ID
	    this.version.globalId = idCounter;
	  }
	  var _proto = VersionedObject.prototype;
	  _proto.increment = function increment() {
	    // Increment the revision number
	    this.version.revision++;
	  };
	  return VersionedObject;
	}();

	/**
	 * The scope for a variable.
	 *
	 * @category Graphics
	 */
	var ScopeId = /*#__PURE__*/function () {
	  /**
	   * Create a new ScopeId instance.
	   *
	   * @param {string} name - The variable name.
	   */
	  function ScopeId(name) {
	    /**
	     * The variable name.
	     *
	     * @type {string}
	     */
	    this.name = name;

	    // Set the default value
	    this.value = null;

	    // Create the version object
	    this.versionObject = new VersionedObject();
	  }

	  // Don't stringify ScopeId to JSON by JSON.stringify, as this stores 'value'
	  // which is not needed. This is used when stringifying a uniform buffer format, which
	  // internally stores the scope.
	  var _proto = ScopeId.prototype;
	  _proto.toJSON = function toJSON(key) {
	    return undefined;
	  }

	  /**
	   * Set variable value.
	   *
	   * @param {*} value - The value.
	   */;
	  _proto.setValue = function setValue(value) {
	    // Set the new value
	    this.value = value;

	    // Increment the revision
	    this.versionObject.increment();
	  }

	  /**
	   * Get variable value.
	   *
	   * @returns {*} The value.
	   */;
	  _proto.getValue = function getValue() {
	    return this.value;
	  };
	  return ScopeId;
	}();

	/**
	 * The scope for variables.
	 *
	 * @category Graphics
	 */
	var ScopeSpace = /*#__PURE__*/function () {
	  /**
	   * Create a new ScopeSpace instance.
	   *
	   * @param {string} name - The scope name.
	   */
	  function ScopeSpace(name) {
	    /**
	     * The scope name.
	     *
	     * @type {string}
	     */
	    this.name = name;

	    // Create map which maps a uniform name into ScopeId
	    this.variables = new Map();
	  }

	  /**
	   * Get (or create, if it doesn't already exist) a variable in the scope.
	   *
	   * @param {string} name - The variable name.
	   * @returns {ScopeId} The variable instance.
	   */
	  var _proto = ScopeSpace.prototype;
	  _proto.resolve = function resolve(name) {
	    // add new ScopeId if it does not exist yet
	    if (!this.variables.has(name)) {
	      this.variables.set(name, new ScopeId(name));
	    }

	    // return the ScopeId instance
	    return this.variables.get(name);
	  }

	  /**
	   * Clears value for any uniform with matching value (used to remove deleted textures).
	   *
	   * @param {*} value - The value to clear.
	   * @ignore
	   */;
	  _proto.removeValue = function removeValue(value) {
	    for (var uniformName in this.variables) {
	      var uniform = this.variables[uniformName];
	      if (uniform.value === value) {
	        uniform.value = null;
	      }
	    }
	  };
	  return ScopeSpace;
	}();

	var id$a = 0;

	/**
	 * A vertex buffer is the mechanism via which the application specifies vertex data to the graphics
	 * hardware.
	 *
	 * @category Graphics
	 */
	var VertexBuffer = /*#__PURE__*/function () {
	  /**
	   * Create a new VertexBuffer instance.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./graphics-device.js').GraphicsDevice} graphicsDevice - The graphics device
	   * used to manage this vertex buffer.
	   * @param {new Function("modulePath", "return import(modulePath)")('./vertex-format.js').VertexFormat} format - The vertex format of this vertex
	   * buffer.
	   * @param {number} numVertices - The number of vertices that this vertex buffer will hold.
	   * @param {number} [usage] - The usage type of the vertex buffer (see BUFFER_*). Defaults to BUFFER_STATIC.
	   * @param {ArrayBuffer} [initialData] - Initial data.
	   */
	  function VertexBuffer(graphicsDevice, format, numVertices, usage, initialData) {
	    if (usage === void 0) {
	      usage = BUFFER_STATIC;
	    }
	    // By default, vertex buffers are static (better for performance since buffer data can be cached in VRAM)
	    this.device = graphicsDevice;
	    this.format = format;
	    this.numVertices = numVertices;
	    this.usage = usage;
	    this.id = id$a++;
	    this.impl = graphicsDevice.createVertexBufferImpl(this, format);

	    // Calculate the size. If format contains verticesByteSize (non-interleaved format), use it
	    this.numBytes = format.verticesByteSize ? format.verticesByteSize : format.size * numVertices;
	    this.adjustVramSizeTracking(graphicsDevice._vram, this.numBytes);

	    // Allocate the storage
	    if (initialData) {
	      this.setData(initialData);
	    } else {
	      this.storage = new ArrayBuffer(this.numBytes);
	    }
	    this.device.buffers.push(this);
	  }

	  /**
	   * Frees resources associated with this vertex buffer.
	   */
	  var _proto = VertexBuffer.prototype;
	  _proto.destroy = function destroy() {
	    // stop tracking the vertex buffer
	    var device = this.device;
	    var idx = device.buffers.indexOf(this);
	    if (idx !== -1) {
	      device.buffers.splice(idx, 1);
	    }
	    if (this.impl.initialized) {
	      this.impl.destroy(device);
	      this.adjustVramSizeTracking(device._vram, -this.storage.byteLength);
	    }
	  };
	  _proto.adjustVramSizeTracking = function adjustVramSizeTracking(vram, size) {
	    Debug.trace(TRACEID_VRAM_VB, this.id + " size: " + size + " vram.vb: " + vram.vb + " => " + (vram.vb + size));
	    vram.vb += size;
	  }

	  /**
	   * Called when the rendering context was lost. It releases all context related resources.
	   *
	   * @ignore
	   */;
	  _proto.loseContext = function loseContext() {
	    this.impl.loseContext();
	  }

	  /**
	   * Returns the data format of the specified vertex buffer.
	   *
	   * @returns {new Function("modulePath", "return import(modulePath)")('./vertex-format.js').VertexFormat} The data format of the specified vertex
	   * buffer.
	   */;
	  _proto.getFormat = function getFormat() {
	    return this.format;
	  }

	  /**
	   * Returns the usage type of the specified vertex buffer. This indicates whether the buffer can
	   * be modified once and used many times {@link BUFFER_STATIC}, modified repeatedly and used
	   * many times {@link BUFFER_DYNAMIC} or modified once and used at most a few times
	   * {@link BUFFER_STREAM}.
	   *
	   * @returns {number} The usage type of the vertex buffer (see BUFFER_*).
	   */;
	  _proto.getUsage = function getUsage() {
	    return this.usage;
	  }

	  /**
	   * Returns the number of vertices stored in the specified vertex buffer.
	   *
	   * @returns {number} The number of vertices stored in the vertex buffer.
	   */;
	  _proto.getNumVertices = function getNumVertices() {
	    return this.numVertices;
	  }

	  /**
	   * Returns a mapped memory block representing the content of the vertex buffer.
	   *
	   * @returns {ArrayBuffer} An array containing the byte data stored in the vertex buffer.
	   */;
	  _proto.lock = function lock() {
	    return this.storage;
	  }

	  /**
	   * Notifies the graphics engine that the client side copy of the vertex buffer's memory can be
	   * returned to the control of the graphics driver.
	   */;
	  _proto.unlock = function unlock() {
	    // Upload the new vertex data
	    this.impl.unlock(this);
	  }

	  /**
	   * Copies data into vertex buffer's memory.
	   *
	   * @param {ArrayBuffer} [data] - Source data to copy.
	   * @returns {boolean} True if function finished successfully, false otherwise.
	   */;
	  _proto.setData = function setData(data) {
	    if (data.byteLength !== this.numBytes) {
	      Debug.error("VertexBuffer: wrong initial data size: expected " + this.numBytes + ", got " + data.byteLength);
	      return false;
	    }
	    this.storage = data;
	    this.unlock();
	    return true;
	  };
	  return VertexBuffer;
	}();

	/**
	 * Calculates simple hash value of a string. Designed for performance, not perfect.
	 *
	 * @param {string} str - String.
	 * @returns {number} Hash value.
	 * @ignore
	 */
	function hashCode(str) {
	  var hash = 0;
	  for (var i = 0, len = str.length; i < len; i++) {
	    hash = (hash << 5) - hash + str.charCodeAt(i);
	    // Convert to 32bit integer
	    hash |= 0;
	  }
	  return hash;
	}

	/**
	 * Calculates simple 32bit hash value of an array of 32bit integer numbers. Designed for
	 * performance, but provides good distribution with small number of collisions. Based on
	 * FNV-1a non-cryptographic hash function.
	 *.
	 * @param {number[]|Uint32Array} array - Array of 32bit integer numbers to hash.
	 * @returns {number} 32bit unsigned integer hash value.
	 * @ignore
	 */
	function hash32Fnv1a(array) {
	  var prime = 16777619;
	  var hash = 2166136261;
	  for (var i = 0; i < array.length; i++) {
	    hash ^= array[i];
	    hash *= prime;
	  }
	  return hash >>> 0; // Ensure non-negative integer
	}

	/**
	 * A cache storing shared resources associated with a device. The resources are removed
	 * from the cache when the device is destroyed.
	 *
	 * @ignore
	 */
	var DeviceCache = /*#__PURE__*/function () {
	  function DeviceCache() {
	    /**
	     * Cache storing the resource for each GraphicsDevice
	     *
	     * @type {Map<new Function("modulePath", "return import(modulePath)")('./graphics-device.js').GraphicsDevice, any>}
	     */
	    this._cache = new Map();
	  }
	  var _proto = DeviceCache.prototype;
	  /**
	   * Returns the resources for the supplied device.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./graphics-device.js').GraphicsDevice} device - The graphics device.
	   * @returns {any} The resource for the device.
	   */
	  _proto.get = function get(device, onCreate) {
	    var _this = this;
	    if (!this._cache.has(device)) {
	      this._cache.set(device, onCreate());

	      // when the device is destroyed, destroy and remove its entry
	      device.on('destroy', function () {
	        _this.remove(device);
	      });

	      // when the context is lost, call optional loseContext on its entry
	      device.on('devicelost', function () {
	        var _this$_cache$get;
	        (_this$_cache$get = _this._cache.get(device)) == null || _this$_cache$get.loseContext == null || _this$_cache$get.loseContext(device);
	      });
	    }
	    return this._cache.get(device);
	  }

	  /**
	   * Destroys and removes the content of the cache associated with the device
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./graphics-device.js').GraphicsDevice} device - The graphics device.
	   */;
	  _proto.remove = function remove(device) {
	    var _this$_cache$get2;
	    (_this$_cache$get2 = this._cache.get(device)) == null || _this$_cache$get2.destroy == null || _this$_cache$get2.destroy(device);
	    this._cache.delete(device);
	  };
	  return DeviceCache;
	}();

	var stringIds$3 = new StringIds();
	var webgpuValidElementSizes = [2, 4, 8, 12, 16];

	// device cache storing the default instancing format per device
	var deviceCache$3 = new DeviceCache();

	/**
	 * A vertex format is a descriptor that defines the layout of vertex data inside a
	 * {@link VertexBuffer}.
	 *
	 * @property {object[]} elements The vertex attribute elements.
	 * @property {string} elements[].name The meaning of the vertex element. This is used to link the
	 * vertex data to a shader input. Can be:
	 *
	 * - {@link SEMANTIC_POSITION}
	 * - {@link SEMANTIC_NORMAL}
	 * - {@link SEMANTIC_TANGENT}
	 * - {@link SEMANTIC_BLENDWEIGHT}
	 * - {@link SEMANTIC_BLENDINDICES}
	 * - {@link SEMANTIC_COLOR}
	 * - {@link SEMANTIC_TEXCOORD0}
	 * - {@link SEMANTIC_TEXCOORD1}
	 * - {@link SEMANTIC_TEXCOORD2}
	 * - {@link SEMANTIC_TEXCOORD3}
	 * - {@link SEMANTIC_TEXCOORD4}
	 * - {@link SEMANTIC_TEXCOORD5}
	 * - {@link SEMANTIC_TEXCOORD6}
	 * - {@link SEMANTIC_TEXCOORD7}
	 *
	 * If vertex data has a meaning other that one of those listed above, use the user-defined
	 * semantics: {@link SEMANTIC_ATTR0} to {@link SEMANTIC_ATTR15}.
	 * @property {number} elements[].numComponents The number of components of the vertex attribute.
	 * Can be 1, 2, 3 or 4.
	 * @property {number} elements[].dataType The data type of the attribute. Can be:
	 *
	 * - {@link TYPE_INT8}
	 * - {@link TYPE_UINT8}
	 * - {@link TYPE_INT16}
	 * - {@link TYPE_UINT16}
	 * - {@link TYPE_INT32}
	 * - {@link TYPE_UINT32}
	 * - {@link TYPE_FLOAT32}
	 * - {@link TYPE_FLOAT16}
	 * @property {boolean} elements[].normalize If true, vertex attribute data will be mapped from a 0
	 * to 255 range down to 0 to 1 when fed to a shader. If false, vertex attribute data is left
	 * unchanged. If this property is unspecified, false is assumed.
	 * @property {number} elements[].offset The number of initial bytes at the start of a vertex that
	 * are not relevant to this attribute.
	 * @property {number} elements[].stride The number of total bytes that are between the start of one
	 * vertex, and the start of the next.
	 * @property {number} elements[].size The size of the attribute in bytes.
	 * @category Graphics
	 */
	var VertexFormat = /*#__PURE__*/function () {
	  /**
	   * Create a new VertexFormat instance.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./graphics-device.js').GraphicsDevice} graphicsDevice - The graphics device
	   * used to manage this vertex format.
	   * @param {object[]} description - An array of vertex attribute descriptions.
	   * @param {string} description[].semantic - The meaning of the vertex element. This is used to
	   * link the vertex data to a shader input. Can be:
	   *
	   * - {@link SEMANTIC_POSITION}
	   * - {@link SEMANTIC_NORMAL}
	   * - {@link SEMANTIC_TANGENT}
	   * - {@link SEMANTIC_BLENDWEIGHT}
	   * - {@link SEMANTIC_BLENDINDICES}
	   * - {@link SEMANTIC_COLOR}
	   * - {@link SEMANTIC_TEXCOORD0}
	   * - {@link SEMANTIC_TEXCOORD1}
	   * - {@link SEMANTIC_TEXCOORD2}
	   * - {@link SEMANTIC_TEXCOORD3}
	   * - {@link SEMANTIC_TEXCOORD4}
	   * - {@link SEMANTIC_TEXCOORD5}
	   * - {@link SEMANTIC_TEXCOORD6}
	   * - {@link SEMANTIC_TEXCOORD7}
	   *
	   * If vertex data has a meaning other that one of those listed above, use the user-defined
	   * semantics: {@link SEMANTIC_ATTR0} to {@link SEMANTIC_ATTR15}.
	   * @param {number} description[].components - The number of components of the vertex attribute.
	   * Can be 1, 2, 3 or 4.
	   * @param {number} description[].type - The data type of the attribute. Can be:
	   *
	   * - {@link TYPE_INT8}
	   * - {@link TYPE_UINT8}
	   * - {@link TYPE_INT16}
	   * - {@link TYPE_UINT16}
	   * - {@link TYPE_INT32}
	   * - {@link TYPE_UINT32}
	   * - {@link TYPE_FLOAT16}
	   * - {@link TYPE_FLOAT32}
	   *
	   * @param {boolean} [description[].normalize] - If true, vertex attribute data will be mapped
	   * from a 0 to 255 range down to 0 to 1 when fed to a shader. If false, vertex attribute data
	   * is left unchanged. If this property is unspecified, false is assumed. This property is
	   * ignored when asInt is true.
	   * @param {boolean} [description[].asInt] - If true, vertex attribute data will be accessible
	   * as integer numbers in shader code. Defaults to false, which means that vertex attribute data
	   * will be accessible as floating point numbers. Can be only used with INT and UINT data types.
	   * @param {number} [vertexCount] - When specified, vertex format will be set up for
	   * non-interleaved format with a specified number of vertices. (example: PPPPNNNNCCCC), where
	   * arrays of individual attributes will be stored one right after the other (subject to
	   * alignment requirements). Note that in this case, the format depends on the number of
	   * vertices, and needs to change when the number of vertices changes. When not specified,
	   * vertex format will be interleaved. (example: PNCPNCPNCPNC).
	   * @example
	   * // Specify 3-component positions (x, y, z)
	   * const vertexFormat = new pc.VertexFormat(graphicsDevice, [
	   *     { semantic: pc.SEMANTIC_POSITION, components: 3, type: pc.TYPE_FLOAT32 }
	   * ]);
	   * @example
	   * // Specify 2-component positions (x, y), a texture coordinate (u, v) and a vertex color (r, g, b, a)
	   * const vertexFormat = new pc.VertexFormat(graphicsDevice, [
	   *     { semantic: pc.SEMANTIC_POSITION, components: 2, type: pc.TYPE_FLOAT32 },
	   *     { semantic: pc.SEMANTIC_TEXCOORD0, components: 2, type: pc.TYPE_FLOAT32 },
	   *     { semantic: pc.SEMANTIC_COLOR, components: 4, type: pc.TYPE_UINT8, normalize: true }
	   * ]);
	   */
	  function VertexFormat(graphicsDevice, description, vertexCount) {
	    this.device = graphicsDevice;
	    this._elements = [];
	    this.hasUv0 = false;
	    this.hasUv1 = false;
	    this.hasColor = false;
	    this.hasTangents = false;
	    this.verticesByteSize = 0;
	    this.vertexCount = vertexCount;
	    this.interleaved = vertexCount === undefined;

	    // true if the vertex format represents an instancing vertex buffer
	    this.instancing = false;

	    // calculate total size of the vertex
	    this.size = description.reduce(function (total, desc) {
	      return total + Math.ceil(desc.components * typedArrayTypesByteSize[desc.type] / 4) * 4;
	    }, 0);
	    var offset = 0,
	      elementSize;
	    for (var i = 0, len = description.length; i < len; i++) {
	      var _elementDesc$asInt, _elementDesc$normaliz;
	      var elementDesc = description[i];
	      elementSize = elementDesc.components * typedArrayTypesByteSize[elementDesc.type];

	      // WebGPU has limited element size support (for example uint16x3 is not supported)
	      Debug.assert(VertexFormat.isElementValid(graphicsDevice, elementDesc), "WebGPU does not support the format of vertex element " + elementDesc.semantic + " : " + vertexTypesNames[elementDesc.type] + " x " + elementDesc.components);

	      // align up the offset to elementSize (when vertexCount is specified only - case of non-interleaved format)
	      if (vertexCount) {
	        offset = math.roundUp(offset, elementSize);

	        // non-interleaved format with elementSize not multiple of 4 might be slower on some platforms - padding is recommended to align its size
	        // example: use 4 x TYPE_UINT8 instead of 3 x TYPE_UINT8
	        Debug.assert(elementSize % 4 === 0, "Non-interleaved vertex format with element size not multiple of 4 can have performance impact on some platforms. Element size: " + elementSize);
	      }
	      var asInt = (_elementDesc$asInt = elementDesc.asInt) != null ? _elementDesc$asInt : false;
	      var normalize = asInt ? false : (_elementDesc$normaliz = elementDesc.normalize) != null ? _elementDesc$normaliz : false;
	      var element = {
	        name: elementDesc.semantic,
	        offset: vertexCount ? offset : elementDesc.hasOwnProperty('offset') ? elementDesc.offset : offset,
	        stride: vertexCount ? elementSize : elementDesc.hasOwnProperty('stride') ? elementDesc.stride : this.size,
	        dataType: elementDesc.type,
	        numComponents: elementDesc.components,
	        normalize: normalize,
	        size: elementSize,
	        asInt: asInt
	      };
	      this._elements.push(element);
	      if (vertexCount) {
	        offset += elementSize * vertexCount;
	      } else {
	        offset += Math.ceil(elementSize / 4) * 4;
	      }
	      if (elementDesc.semantic === SEMANTIC_TEXCOORD0) {
	        this.hasUv0 = true;
	      } else if (elementDesc.semantic === SEMANTIC_TEXCOORD1) {
	        this.hasUv1 = true;
	      } else if (elementDesc.semantic === SEMANTIC_COLOR) {
	        this.hasColor = true;
	      } else if (elementDesc.semantic === SEMANTIC_TANGENT) {
	        this.hasTangents = true;
	      }
	    }
	    if (vertexCount) {
	      this.verticesByteSize = offset;
	    }
	    this._evaluateHash();
	  }
	  /**
	   * The {@link VertexFormat} used to store matrices of type {@link Mat4} for hardware instancing.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./graphics-device.js').GraphicsDevice} graphicsDevice - The graphics device
	   * used to create this vertex format.
	   *
	   * @returns {VertexFormat} The default instancing vertex format.
	   */
	  VertexFormat.getDefaultInstancingFormat = function getDefaultInstancingFormat(graphicsDevice) {
	    // get it from the device cache, or create a new one if not cached yet
	    return deviceCache$3.get(graphicsDevice, function () {
	      return new VertexFormat(graphicsDevice, [{
	        semantic: SEMANTIC_ATTR12,
	        components: 4,
	        type: TYPE_FLOAT32
	      }, {
	        semantic: SEMANTIC_ATTR13,
	        components: 4,
	        type: TYPE_FLOAT32
	      }, {
	        semantic: SEMANTIC_ATTR14,
	        components: 4,
	        type: TYPE_FLOAT32
	      }, {
	        semantic: SEMANTIC_ATTR15,
	        components: 4,
	        type: TYPE_FLOAT32
	      }]);
	    });
	  };
	  VertexFormat.isElementValid = function isElementValid(graphicsDevice, elementDesc) {
	    var elementSize = elementDesc.components * typedArrayTypesByteSize[elementDesc.type];
	    if (graphicsDevice.isWebGPU && !webgpuValidElementSizes.includes(elementSize)) return false;
	    return true;
	  }

	  /**
	   * Applies any changes made to the VertexFormat's properties.
	   *
	   * @private
	   */;
	  var _proto = VertexFormat.prototype;
	  _proto.update = function update() {
	    // Note that this is used only by vertex attribute morphing on the WebGL.
	    Debug.assert(!this.device.isWebGPU, "VertexFormat#update is not supported on WebGPU and VertexFormat cannot be modified.");
	    this._evaluateHash();
	  }

	  /**
	   * Evaluates hash values for the format allowing fast compare of batching / rendering compatibility.
	   *
	   * @private
	   */;
	  _proto._evaluateHash = function _evaluateHash() {
	    var stringElementsBatch = [];
	    var stringElementsRender = [];
	    var len = this._elements.length;
	    for (var i = 0; i < len; i++) {
	      var _this$_elements$i = this._elements[i],
	        name = _this$_elements$i.name,
	        dataType = _this$_elements$i.dataType,
	        numComponents = _this$_elements$i.numComponents,
	        normalize = _this$_elements$i.normalize,
	        offset = _this$_elements$i.offset,
	        stride = _this$_elements$i.stride,
	        size = _this$_elements$i.size,
	        asInt = _this$_elements$i.asInt;

	      // create string description of each element that is relevant for batching
	      var stringElementBatch = name + dataType + numComponents + normalize + asInt;
	      stringElementsBatch.push(stringElementBatch);

	      // create string description of each element that is relevant for rendering
	      var stringElementRender = stringElementBatch + offset + stride + size;
	      stringElementsRender.push(stringElementRender);
	    }

	    // sort batching ones alphabetically to make the hash order independent
	    stringElementsBatch.sort();
	    var batchingString = stringElementsBatch.join();
	    this.batchingHash = hashCode(batchingString);

	    // shader processing hash - all elements that are used by the ShaderProcessor processing attributes
	    // at the moment this matches the batching hash
	    this.shaderProcessingHashString = batchingString;

	    // rendering hash
	    this.renderingHashString = stringElementsRender.join('_');
	    this.renderingHash = stringIds$3.get(this.renderingHashString);
	  };
	  _createClass(VertexFormat, [{
	    key: "elements",
	    get: function get() {
	      return this._elements;
	    }
	  }]);
	  return VertexFormat;
	}();

	var _class$1;
	var stringIds$2 = new StringIds();

	/**
	 * Holds stencil test settings.
	 *
	 * @category Graphics
	 */
	var StencilParameters = /*#__PURE__*/function () {
	  /**
	   * Create a new StencilParameters instance.
	   *
	   * @param {object} [options] - Options object to configure the stencil parameters.
	   */
	  function StencilParameters(options) {
	    var _options$func, _options$ref, _options$readMask, _options$writeMask, _options$fail, _options$zfail, _options$zpass;
	    if (options === void 0) {
	      options = {};
	    }
	    /**
	     * @type {number}
	     * @private
	     */
	    this._func = void 0;
	    /**
	     * @type {number}
	     * @private
	     */
	    this._ref = void 0;
	    /**
	     * @type {number}
	     * @private
	     */
	    this._fail = void 0;
	    /**
	     * @type {number}
	     * @private
	     */
	    this._zfail = void 0;
	    /**
	     * @type {number}
	     * @private
	     */
	    this._zpass = void 0;
	    /**
	     * @type {number}
	     * @private
	     */
	    this._readMask = void 0;
	    /**
	     * @type {number}
	     * @private
	     */
	    this._writeMask = void 0;
	    /**
	     * @type {boolean}
	     * @private
	     */
	    this._dirty = true;
	    /**
	     * @type {number}
	     * @private
	     */
	    this._key = void 0;
	    this._func = (_options$func = options.func) != null ? _options$func : FUNC_ALWAYS;
	    this._ref = (_options$ref = options.ref) != null ? _options$ref : 0;
	    this._readMask = (_options$readMask = options.readMask) != null ? _options$readMask : 0xFF;
	    this._writeMask = (_options$writeMask = options.writeMask) != null ? _options$writeMask : 0xFF;
	    this._fail = (_options$fail = options.fail) != null ? _options$fail : STENCILOP_KEEP; // keep == 0
	    this._zfail = (_options$zfail = options.zfail) != null ? _options$zfail : STENCILOP_KEEP;
	    this._zpass = (_options$zpass = options.zpass) != null ? _options$zpass : STENCILOP_KEEP;

	    // Evaluate key here. This evaluates the key for the DEFAULT instance, which is important,
	    // as during rendering it gets copied and the key would get evaluated each time.
	    this._evalKey();
	  }
	  var _proto = StencilParameters.prototype;
	  _proto._evalKey = function _evalKey() {
	    var _func = this._func,
	      _ref = this._ref,
	      _fail = this._fail,
	      _zfail = this._zfail,
	      _zpass = this._zpass,
	      _readMask = this._readMask,
	      _writeMask = this._writeMask;
	    var key = _func + "," + _ref + "," + _fail + "," + _zfail + "," + _zpass + "," + _readMask + "," + _writeMask;
	    this._key = stringIds$2.get(key);
	    this._dirty = false;
	  };
	  /**
	   * Copies the contents of a source stencil parameters to this stencil parameters.
	   *
	   * @param {StencilParameters} rhs - A stencil parameters to copy from.
	   * @returns {StencilParameters} Self for chaining.
	   */
	  _proto.copy = function copy(rhs) {
	    this._func = rhs._func;
	    this._ref = rhs._ref;
	    this._readMask = rhs._readMask;
	    this._writeMask = rhs._writeMask;
	    this._fail = rhs._fail;
	    this._zfail = rhs._zfail;
	    this._zpass = rhs._zpass;
	    this._dirty = rhs._dirty;
	    this._key = rhs._key;
	    return this;
	  }

	  /**
	   * Clone the stencil parameters.
	   *
	   * @returns {StencilParameters} A cloned StencilParameters object.
	   */;
	  _proto.clone = function clone() {
	    var clone = new this.constructor();
	    return clone.copy(this);
	  }

	  /**
	   * A default stencil state.
	   *
	   * @type {StencilParameters}
	   * @readonly
	   */;
	  _createClass(StencilParameters, [{
	    key: "func",
	    get: function get() {
	      return this._func;
	    }

	    /**
	     * Sets stencil test reference value used in comparisons.
	     *
	     * @type {number}
	     */,
	    set:
	    /**
	     * A comparison function that decides if the pixel should be written, based on the current
	     * stencil buffer value, reference value, and mask value. Can be:
	     *
	     * - {@link FUNC_NEVER}: never pass
	     * - {@link FUNC_LESS}: pass if (ref & mask) < (stencil & mask)
	     * - {@link FUNC_EQUAL}: pass if (ref & mask) == (stencil & mask)
	     * - {@link FUNC_LESSEQUAL}: pass if (ref & mask) <= (stencil & mask)
	     * - {@link FUNC_GREATER}: pass if (ref & mask) > (stencil & mask)
	     * - {@link FUNC_NOTEQUAL}: pass if (ref & mask) != (stencil & mask)
	     * - {@link FUNC_GREATEREQUAL}: pass if (ref & mask) >= (stencil & mask)
	     * - {@link FUNC_ALWAYS}: always pass
	     *
	     * @type {number}
	     */
	    function set(value) {
	      this._func = value;
	      this._dirty = true;
	    }
	  }, {
	    key: "ref",
	    get: function get() {
	      return this._ref;
	    }

	    /**
	     * Operation to perform if stencil test is failed. Can be:
	     *
	     * - {@link STENCILOP_KEEP}: don't change the stencil buffer value
	     * - {@link STENCILOP_ZERO}: set value to zero
	     * - {@link STENCILOP_REPLACE}: replace value with the reference value.
	     * - {@link STENCILOP_INCREMENT}: increment the value
	     * - {@link STENCILOP_INCREMENTWRAP}: increment the value, but wrap it to zero when it's larger
	     * than a maximum representable value
	     * - {@link STENCILOP_DECREMENT}: decrement the value
	     * - {@link STENCILOP_DECREMENTWRAP}: decrement the value, but wrap it to a maximum
	     * representable value, if the current value is 0
	     * - {@link STENCILOP_INVERT}: invert the value bitwise
	     *
	     * @type {number}
	     */,
	    set: function set(value) {
	      this._ref = value;
	      this._dirty = true;
	    }
	  }, {
	    key: "fail",
	    get: function get() {
	      return this._fail;
	    }

	    /**
	     * Operation to perform if depth test is failed. Accepts the same values as `fail`.
	     *
	     * @type {number}
	     */,
	    set: function set(value) {
	      this._fail = value;
	      this._dirty = true;
	    }
	  }, {
	    key: "zfail",
	    get: function get() {
	      return this._zfail;
	    }

	    /**
	     * Operation to perform if both stencil and depth test are passed. Accepts the same values as
	     * `fail`.
	     *
	     * @type {number}
	     */,
	    set: function set(value) {
	      this._zfail = value;
	      this._dirty = true;
	    }
	  }, {
	    key: "zpass",
	    get: function get() {
	      return this._zpass;
	    }

	    /**
	     * Mask applied to stencil buffer value and reference value before comparison.
	     *
	     * @type {number}
	     */,
	    set: function set(value) {
	      this._zpass = value;
	      this._dirty = true;
	    }
	  }, {
	    key: "readMask",
	    get: function get() {
	      return this._readMask;
	    }

	    /**
	     * A bit mask applied to the stencil value, when written.
	     *
	     * @type {number}
	     */,
	    set: function set(value) {
	      this._readMask = value;
	      this._dirty = true;
	    }
	  }, {
	    key: "writeMask",
	    get: function get() {
	      return this._writeMask;
	    },
	    set: function set(value) {
	      this._writeMask = value;
	      this._dirty = true;
	    }
	  }, {
	    key: "key",
	    get: function get() {
	      if (this._dirty) {
	        this._evalKey();
	      }
	      return this._key;
	    }
	  }]);
	  return StencilParameters;
	}();
	_class$1 = StencilParameters;
	StencilParameters.DEFAULT = Object.freeze(new _class$1());

	/**
	 * The graphics device manages the underlying graphics context. It is responsible for submitting
	 * render state changes and graphics primitives to the hardware. A graphics device is tied to a
	 * specific canvas HTML element. It is valid to have more than one canvas element per page and
	 * create a new graphics device against each.
	 *
	 * @augments EventHandler
	 * @category Graphics
	 */
	var GraphicsDevice = /*#__PURE__*/function (_EventHandler) {
	  _inheritsLoose(GraphicsDevice, _EventHandler);
	  function GraphicsDevice(canvas, options) {
	    var _this$initOptions, _this$initOptions$dep, _this$initOptions2, _this$initOptions2$st, _this$initOptions3, _this$initOptions3$an, _this$initOptions4, _this$initOptions4$po;
	    var _this;
	    _this = _EventHandler.call(this) || this;
	    /**
	     * Fired when the canvas is resized. The handler is passed the new width and height as number
	     * parameters.
	     *
	     * @event
	     * @example
	     * graphicsDevice.on('resizecanvas', (width, height) => {
	     *     console.log(`The canvas was resized to ${width}x${height}`);
	     * });
	     */
	    /**
	     * The canvas DOM element that provides the underlying WebGL context used by the graphics device.
	     *
	     * @type {HTMLCanvasElement}
	     * @readonly
	     */
	    _this.canvas = void 0;
	    /**
	     * The render target representing the main back-buffer.
	     *
	     * @type {new Function("modulePath", "return import(modulePath)")('./render-target.js').RenderTarget|null}
	     * @ignore
	     */
	    _this.backBuffer = null;
	    /**
	     * The dimensions of the back buffer.
	     *
	     * @ignore
	     */
	    _this.backBufferSize = new Vec2();
	    /**
	     * The pixel format of the back buffer. Typically PIXELFORMAT_RGBA8, PIXELFORMAT_BGRA8 or
	     * PIXELFORMAT_RGB8.
	     *
	     * @ignore
	     */
	    _this.backBufferFormat = void 0;
	    /**
	     * True if the back buffer should use anti-aliasing.
	     *
	     * @type {boolean}
	     */
	    _this.backBufferAntialias = false;
	    /**
	     * True if the deviceType is WebGPU
	     *
	     * @type {boolean}
	     * @readonly
	     */
	    _this.isWebGPU = false;
	    /**
	     * True if the deviceType is WebGL1
	     *
	     * @type {boolean}
	     * @readonly
	     */
	    _this.isWebGL1 = false;
	    /**
	     * True if the deviceType is WebGL2
	     *
	     * @type {boolean}
	     * @readonly
	     */
	    _this.isWebGL2 = false;
	    /**
	     * The scope namespace for shader attributes and variables.
	     *
	     * @type {ScopeSpace}
	     * @readonly
	     */
	    _this.scope = void 0;
	    /**
	     * The maximum number of supported bones using uniform buffers.
	     *
	     * @type {number}
	     * @readonly
	     */
	    _this.boneLimit = void 0;
	    /**
	     * The maximum supported texture anisotropy setting.
	     *
	     * @type {number}
	     * @readonly
	     */
	    _this.maxAnisotropy = void 0;
	    /**
	     * The maximum supported dimension of a cube map.
	     *
	     * @type {number}
	     * @readonly
	     */
	    _this.maxCubeMapSize = void 0;
	    /**
	     * The maximum supported dimension of a texture.
	     *
	     * @type {number}
	     * @readonly
	     */
	    _this.maxTextureSize = void 0;
	    /**
	     * The maximum supported dimension of a 3D texture (any axis).
	     *
	     * @type {number}
	     * @readonly
	     */
	    _this.maxVolumeSize = void 0;
	    /**
	     * The maximum supported number of color buffers attached to a render target.
	     *
	     * @type {number}
	     * @readonly
	     */
	    _this.maxColorAttachments = 1;
	    /**
	     * The highest shader precision supported by this graphics device. Can be 'hiphp', 'mediump' or
	     * 'lowp'.
	     *
	     * @type {string}
	     * @readonly
	     */
	    _this.precision = void 0;
	    /**
	     * The number of hardware anti-aliasing samples used by the frame buffer.
	     *
	     * @readonly
	     * @type {number}
	     */
	    _this.samples = void 0;
	    /**
	     * True if the main framebuffer contains stencil attachment.
	     *
	     * @ignore
	     * @type {boolean}
	     */
	    _this.supportsStencil = void 0;
	    /**
	     * True if Multiple Render Targets feature is supported. This refers to the ability to render to
	     * multiple color textures with a single draw call.
	     *
	     * @readonly
	     * @type {boolean}
	     */
	    _this.supportsMrt = false;
	    /**
	     * True if the device supports volume textures.
	     *
	     * @readonly
	     * @type {boolean}
	     */
	    _this.supportsVolumeTextures = false;
	    /**
	     * True if the device supports compute shaders.
	     *
	     * @readonly
	     * @type {boolean}
	     */
	    _this.supportsCompute = false;
	    /**
	     * Currently active render target.
	     *
	     * @type {new Function("modulePath", "return import(modulePath)")('./render-target.js').RenderTarget|null}
	     * @ignore
	     */
	    _this.renderTarget = null;
	    /**
	     * Array of objects that need to be re-initialized after a context restore event
	     *
	     * @type {new Function("modulePath", "return import(modulePath)")('./shader.js').Shader[]}
	     * @ignore
	     */
	    _this.shaders = [];
	    /**
	     * An array of currently created textures.
	     *
	     * @type {new Function("modulePath", "return import(modulePath)")('./texture.js').Texture[]}
	     * @ignore
	     */
	    _this.textures = [];
	    /**
	     * A set of currently created render targets.
	     *
	     * @type {Set<new Function("modulePath", "return import(modulePath)")('./render-target.js').RenderTarget>}
	     * @ignore
	     */
	    _this.targets = new Set();
	    /**
	     * A version number that is incremented every frame. This is used to detect if some object were
	     * invalidated.
	     *
	     * @type {number}
	     * @ignore
	     */
	    _this.renderVersion = 0;
	    /**
	     * Index of the currently active render pass.
	     *
	     * @type {number}
	     * @ignore
	     */
	    _this.renderPassIndex = void 0;
	    /** @type {boolean} */
	    _this.insideRenderPass = false;
	    /**
	     * True if hardware instancing is supported.
	     *
	     * @type {boolean}
	     * @readonly
	     */
	    _this.supportsInstancing = void 0;
	    /**
	     * True if the device supports uniform buffers.
	     *
	     * @type {boolean}
	     * @ignore
	     */
	    _this.supportsUniformBuffers = false;
	    /**
	     * True if 32-bit floating-point textures can be used as a frame buffer.
	     *
	     * @type {boolean}
	     * @readonly
	     */
	    _this.textureFloatRenderable = void 0;
	    /**
	     * True if 16-bit floating-point textures can be used as a frame buffer.
	     *
	     * @type {boolean}
	     * @readonly
	     */
	    _this.textureHalfFloatRenderable = void 0;
	    /**
	     * True if filtering can be applied when sampling float textures.
	     *
	     * @type {boolean}
	     * @readonly
	     */
	    _this.textureFloatFilterable = false;
	    /**
	     * True if filtering can be applied when sampling 16-bit float textures.
	     *
	     * @type {boolean}
	     * @readonly
	     */
	    _this.textureHalfFloatFilterable = false;
	    /**
	     * A vertex buffer representing a quad.
	     *
	     * @type {VertexBuffer}
	     * @ignore
	     */
	    _this.quadVertexBuffer = void 0;
	    /**
	     * An object representing current blend state
	     *
	     * @ignore
	     */
	    _this.blendState = new BlendState();
	    /**
	     * The current depth state.
	     *
	     * @ignore
	     */
	    _this.depthState = new DepthState();
	    /**
	     * True if stencil is enabled and stencilFront and stencilBack are used
	     *
	     * @ignore
	     */
	    _this.stencilEnabled = false;
	    /**
	     * The current front stencil parameters.
	     *
	     * @ignore
	     */
	    _this.stencilFront = new StencilParameters();
	    /**
	     * The current back stencil parameters.
	     *
	     * @ignore
	     */
	    _this.stencilBack = new StencilParameters();
	    /**
	     * The dynamic buffer manager.
	     *
	     * @type {new Function("modulePath", "return import(modulePath)")('./dynamic-buffers.js').DynamicBuffers}
	     * @ignore
	     */
	    _this.dynamicBuffers = void 0;
	    /**
	     * The GPU profiler.
	     *
	     * @type {new Function("modulePath", "return import(modulePath)")('./gpu-profiler.js').GpuProfiler}
	     */
	    _this.gpuProfiler = void 0;
	    _this.defaultClearOptions = {
	      color: [0, 0, 0, 1],
	      depth: 1,
	      stencil: 0,
	      flags: CLEARFLAG_COLOR | CLEARFLAG_DEPTH
	    };
	    _this.canvas = canvas;

	    // copy options and handle defaults
	    _this.initOptions = _extends({}, options);
	    (_this$initOptions$dep = (_this$initOptions = _this.initOptions).depth) != null ? _this$initOptions$dep : _this$initOptions.depth = true;
	    (_this$initOptions2$st = (_this$initOptions2 = _this.initOptions).stencil) != null ? _this$initOptions2$st : _this$initOptions2.stencil = true;
	    (_this$initOptions3$an = (_this$initOptions3 = _this.initOptions).antialias) != null ? _this$initOptions3$an : _this$initOptions3.antialias = true;
	    (_this$initOptions4$po = (_this$initOptions4 = _this.initOptions).powerPreference) != null ? _this$initOptions4$po : _this$initOptions4.powerPreference = 'high-performance';

	    // Some devices window.devicePixelRatio can be less than one
	    // eg Oculus Quest 1 which returns a window.devicePixelRatio of 0.8
	    _this._maxPixelRatio = platform.browser ? Math.min(1, window.devicePixelRatio) : 1;
	    _this.buffers = [];
	    _this._vram = {
	      texShadow: 0,
	      texAsset: 0,
	      texLightmap: 0,
	      tex: 0,
	      vb: 0,
	      ib: 0,
	      ub: 0
	    };
	    _this._shaderStats = {
	      vsCompiled: 0,
	      fsCompiled: 0,
	      linked: 0,
	      materialShaders: 0,
	      compileTime: 0
	    };
	    _this.initializeContextCaches();

	    // Profiler stats
	    _this._drawCallsPerFrame = 0;
	    _this._shaderSwitchesPerFrame = 0;
	    _this._primsPerFrame = [];
	    for (var i = PRIMITIVE_POINTS; i <= PRIMITIVE_TRIFAN; i++) {
	      _this._primsPerFrame[i] = 0;
	    }
	    _this._renderTargetCreationTime = 0;

	    // Create the ScopeNamespace for shader attributes and variables
	    _this.scope = new ScopeSpace("Device");
	    _this.textureBias = _this.scope.resolve("textureBias");
	    _this.textureBias.setValue(0.0);
	    return _this;
	  }

	  /**
	   * Function that executes after the device has been created.
	   */
	  var _proto = GraphicsDevice.prototype;
	  _proto.postInit = function postInit() {
	    // create quad vertex buffer
	    var vertexFormat = new VertexFormat(this, [{
	      semantic: SEMANTIC_POSITION,
	      components: 2,
	      type: TYPE_FLOAT32
	    }]);
	    var positions = new Float32Array([-1, -1, 1, -1, -1, 1, 1, 1]);
	    this.quadVertexBuffer = new VertexBuffer(this, vertexFormat, 4, BUFFER_STATIC, positions);
	  }

	  /**
	   * Destroy the graphics device.
	   */;
	  _proto.destroy = function destroy() {
	    var _this$quadVertexBuffe, _this$dynamicBuffers, _this$gpuProfiler;
	    // fire the destroy event.
	    // textures and other device resources may destroy themselves in response.
	    this.fire('destroy');
	    (_this$quadVertexBuffe = this.quadVertexBuffer) == null || _this$quadVertexBuffe.destroy();
	    this.quadVertexBuffer = null;
	    (_this$dynamicBuffers = this.dynamicBuffers) == null || _this$dynamicBuffers.destroy();
	    this.dynamicBuffers = null;
	    (_this$gpuProfiler = this.gpuProfiler) == null || _this$gpuProfiler.destroy();
	    this.gpuProfiler = null;
	  };
	  _proto.onDestroyShader = function onDestroyShader(shader) {
	    this.fire('destroy:shader', shader);
	    var idx = this.shaders.indexOf(shader);
	    if (idx !== -1) {
	      this.shaders.splice(idx, 1);
	    }
	  }

	  // executes after the extended classes have executed their destroy function
	  ;
	  _proto.postDestroy = function postDestroy() {
	    this.scope = null;
	    this.canvas = null;
	  }

	  // don't stringify GraphicsDevice to JSON by JSON.stringify
	  ;
	  _proto.toJSON = function toJSON(key) {
	    return undefined;
	  };
	  _proto.initializeContextCaches = function initializeContextCaches() {
	    this.indexBuffer = null;
	    this.vertexBuffers = [];
	    this.shader = null;
	    this.shaderValid = undefined;
	    this.shaderAsyncCompile = false;
	    this.renderTarget = null;
	  };
	  _proto.initializeRenderState = function initializeRenderState() {
	    this.blendState = new BlendState();
	    this.depthState = new DepthState();
	    this.cullMode = CULLFACE_BACK;

	    // Cached viewport and scissor dimensions
	    this.vx = this.vy = this.vw = this.vh = 0;
	    this.sx = this.sy = this.sw = this.sh = 0;
	    this.blendColor = new Color(0, 0, 0, 0);
	  }

	  /**
	   * Sets the specified stencil state. If both stencilFront and stencilBack are null, stencil
	   * operation is disabled.
	   *
	   * @param {StencilParameters} [stencilFront] - The front stencil parameters. Defaults to
	   * {@link StencilParameters.DEFAULT} if not specified.
	   * @param {StencilParameters} [stencilBack] - The back stencil parameters. Defaults to
	   * {@link StencilParameters.DEFAULT} if not specified.
	   */;
	  _proto.setStencilState = function setStencilState(stencilFront, stencilBack) {
	    Debug.assert(false);
	  }

	  /**
	   * Sets the specified blend state.
	   *
	   * @param {BlendState} blendState - New blend state.
	   */;
	  _proto.setBlendState = function setBlendState(blendState) {
	    Debug.assert(false);
	  }

	  /**
	   * Sets the constant blend color and alpha values used with {@link BLENDMODE_CONSTANT} and
	   * {@link BLENDMODE_ONE_MINUS_CONSTANT} factors specified in {@link BlendState}. Defaults to
	   * [0, 0, 0, 0].
	   *
	   * @param {number} r - The value for red.
	   * @param {number} g - The value for green.
	   * @param {number} b - The value for blue.
	   * @param {number} a - The value for alpha.
	   */;
	  _proto.setBlendColor = function setBlendColor(r, g, b, a) {
	    Debug.assert(false);
	  }

	  /**
	   * Sets the specified depth state.
	   *
	   * @param {DepthState} depthState - New depth state.
	   */;
	  _proto.setDepthState = function setDepthState(depthState) {
	    Debug.assert(false);
	  }

	  /**
	   * Controls how triangles are culled based on their face direction. The default cull mode is
	   * {@link CULLFACE_BACK}.
	   *
	   * @param {number} cullMode - The cull mode to set. Can be:
	   *
	   * - {@link CULLFACE_NONE}
	   * - {@link CULLFACE_BACK}
	   * - {@link CULLFACE_FRONT}
	   */;
	  _proto.setCullMode = function setCullMode(cullMode) {
	    Debug.assert(false);
	  }

	  /**
	   * Sets the specified render target on the device. If null is passed as a parameter, the back
	   * buffer becomes the current target for all rendering operations.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./render-target.js').RenderTarget|null} renderTarget - The render target to
	   * activate.
	   * @example
	   * // Set a render target to receive all rendering output
	   * device.setRenderTarget(renderTarget);
	   *
	   * // Set the back buffer to receive all rendering output
	   * device.setRenderTarget(null);
	   */;
	  _proto.setRenderTarget = function setRenderTarget(renderTarget) {
	    this.renderTarget = renderTarget;
	  }

	  /**
	   * Sets the current index buffer on the graphics device. On subsequent calls to
	   * {@link GraphicsDevice#draw}, the specified index buffer will be used to provide index data
	   * for any indexed primitives.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./index-buffer.js').IndexBuffer} indexBuffer - The index buffer to assign to
	   * the device.
	   */;
	  _proto.setIndexBuffer = function setIndexBuffer(indexBuffer) {
	    // Store the index buffer
	    this.indexBuffer = indexBuffer;
	  }

	  /**
	   * Sets the current vertex buffer on the graphics device. On subsequent calls to
	   * {@link GraphicsDevice#draw}, the specified vertex buffer(s) will be used to provide vertex
	   * data for any primitives.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./vertex-buffer.js').VertexBuffer} vertexBuffer - The vertex buffer to
	   * assign to the device.
	   */;
	  _proto.setVertexBuffer = function setVertexBuffer(vertexBuffer) {
	    if (vertexBuffer) {
	      this.vertexBuffers.push(vertexBuffer);
	    }
	  }

	  /**
	   * Queries the currently set render target on the device.
	   *
	   * @returns {new Function("modulePath", "return import(modulePath)")('./render-target.js').RenderTarget} The current render target.
	   * @example
	   * // Get the current render target
	   * const renderTarget = device.getRenderTarget();
	   */;
	  _proto.getRenderTarget = function getRenderTarget() {
	    return this.renderTarget;
	  }

	  /**
	   * Initialize render target before it can be used.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./render-target.js').RenderTarget} target - The render target to be
	   * initialized.
	   * @ignore
	   */;
	  _proto.initRenderTarget = function initRenderTarget(target) {
	    if (target.initialized) return;
	    var startTime = now();
	    this.fire('fbo:create', {
	      timestamp: startTime,
	      target: this
	    });
	    target.init();
	    this.targets.add(target);
	    this._renderTargetCreationTime += now() - startTime;
	  }

	  /**
	   * Reports whether a texture source is a canvas, image, video or ImageBitmap.
	   *
	   * @param {*} texture - Texture source data.
	   * @returns {boolean} True if the texture is a canvas, image, video or ImageBitmap and false
	   * otherwise.
	   * @ignore
	   */;
	  _proto._isBrowserInterface = function _isBrowserInterface(texture) {
	    return this._isImageBrowserInterface(texture) || this._isImageCanvasInterface(texture) || this._isImageVideoInterface(texture);
	  };
	  _proto._isImageBrowserInterface = function _isImageBrowserInterface(texture) {
	    return typeof ImageBitmap !== 'undefined' && texture instanceof ImageBitmap || typeof HTMLImageElement !== 'undefined' && texture instanceof HTMLImageElement;
	  };
	  _proto._isImageCanvasInterface = function _isImageCanvasInterface(texture) {
	    return typeof HTMLCanvasElement !== 'undefined' && texture instanceof HTMLCanvasElement;
	  };
	  _proto._isImageVideoInterface = function _isImageVideoInterface(texture) {
	    return typeof HTMLVideoElement !== 'undefined' && texture instanceof HTMLVideoElement;
	  }

	  /**
	   * Sets the width and height of the canvas, then fires the `resizecanvas` event. Note that the
	   * specified width and height values will be multiplied by the value of
	   * {@link GraphicsDevice#maxPixelRatio} to give the final resultant width and height for the
	   * canvas.
	   *
	   * @param {number} width - The new width of the canvas.
	   * @param {number} height - The new height of the canvas.
	   * @ignore
	   */;
	  _proto.resizeCanvas = function resizeCanvas(width, height) {
	    var pixelRatio = Math.min(this._maxPixelRatio, platform.browser ? window.devicePixelRatio : 1);
	    var w = Math.floor(width * pixelRatio);
	    var h = Math.floor(height * pixelRatio);
	    if (w !== this.canvas.width || h !== this.canvas.height) {
	      this.setResolution(w, h);
	    }
	  }

	  /**
	   * Sets the width and height of the canvas, then fires the `resizecanvas` event. Note that the
	   * value of {@link GraphicsDevice#maxPixelRatio} is ignored.
	   *
	   * @param {number} width - The new width of the canvas.
	   * @param {number} height - The new height of the canvas.
	   * @ignore
	   */;
	  _proto.setResolution = function setResolution(width, height) {
	    this.canvas.width = width;
	    this.canvas.height = height;
	    this.fire(GraphicsDevice.EVENT_RESIZE, width, height);
	  };
	  _proto.updateClientRect = function updateClientRect() {
	    this.clientRect = this.canvas.getBoundingClientRect();
	  }

	  /**
	   * Width of the back buffer in pixels.
	   *
	   * @type {number}
	   */;
	  /**
	   * Queries the maximum number of bones that can be referenced by a shader. The shader
	   * generators (programlib) use this number to specify the matrix array size of the uniform
	   * 'matrix_pose[0]'. The value is calculated based on the number of available uniform vectors
	   * available after subtracting the number taken by a typical heavyweight shader. If a different
	   * number is required, it can be tuned via {@link GraphicsDevice#setBoneLimit}.
	   *
	   * @returns {number} The maximum number of bones that can be supported by the host hardware.
	   * @ignore
	   */
	  _proto.getBoneLimit = function getBoneLimit() {
	    return this.boneLimit;
	  }

	  /**
	   * Specifies the maximum number of bones that the device can support on the current hardware.
	   * This function allows the default calculated value based on available vector uniforms to be
	   * overridden.
	   *
	   * @param {number} maxBones - The maximum number of bones supported by the host hardware.
	   * @ignore
	   */;
	  _proto.setBoneLimit = function setBoneLimit(maxBones) {
	    this.boneLimit = maxBones;
	  };
	  _proto.startRenderPass = function startRenderPass(renderPass) {};
	  _proto.endRenderPass = function endRenderPass(renderPass) {};
	  _proto.startComputePass = function startComputePass() {};
	  _proto.endComputePass = function endComputePass() {}

	  /**
	   * Function which executes at the start of the frame. This should not be called manually, as
	   * it is handled by the AppBase instance.
	   *
	   * @ignore
	   */;
	  _proto.frameStart = function frameStart() {
	    var _this2 = this;
	    this.renderPassIndex = 0;
	    this.renderVersion++;
	    Debug.call(function () {
	      // log out all loaded textures, sorted by gpu memory size
	      if (Tracing.get(TRACEID_TEXTURES)) {
	        var textures = _this2.textures.slice();
	        textures.sort(function (a, b) {
	          return b.gpuSize - a.gpuSize;
	        });
	        Debug.log("Textures: " + textures.length);
	        var textureTotal = 0;
	        textures.forEach(function (texture, index) {
	          var textureSize = texture.gpuSize;
	          textureTotal += textureSize;
	          Debug.log(index + ". " + texture.name + " " + texture.width + "x" + texture.height + " VRAM: " + (textureSize / 1024 / 1024).toFixed(2) + " MB");
	        });
	        Debug.log("Total: " + (textureTotal / 1024 / 1024).toFixed(2) + "MB");
	      }
	    });
	  }

	  /**
	   * Function which executes at the end of the frame. This should not be called manually, as it is
	   * handled by the AppBase instance.
	   *
	   * @ignore
	   */;
	  _proto.frameEnd = function frameEnd() {}

	  /**
	   * Get a renderable HDR pixel format supported by the graphics device.
	   *
	   * @param {number[]} [formats] - An array of pixel formats to check for support. Can contain:
	   *
	   * - {@link PIXELFORMAT_111110F}
	   * - {@link PIXELFORMAT_RGBA16F}
	   * - {@link PIXELFORMAT_RGBA32F}
	   *
	   * @param {boolean} [filterable] - If true, the format also needs to be filterable. Defaults to
	   * true.
	   * @returns {number|undefined} The first supported renderable HDR format or undefined if none is
	   * supported.
	   */;
	  _proto.getRenderableHdrFormat = function getRenderableHdrFormat(formats, filterable) {
	    if (formats === void 0) {
	      formats = [PIXELFORMAT_111110F, PIXELFORMAT_RGBA16F, PIXELFORMAT_RGBA32F];
	    }
	    if (filterable === void 0) {
	      filterable = true;
	    }
	    for (var i = 0; i < formats.length; i++) {
	      var format = formats[i];
	      switch (format) {
	        case PIXELFORMAT_111110F:
	          {
	            if (this.textureRG11B10Renderable) return format;
	            break;
	          }
	        case PIXELFORMAT_RGBA16F:
	          if (this.textureHalfFloatRenderable && (!filterable || this.textureHalfFloatFilterable)) {
	            return format;
	          }
	          break;
	        case PIXELFORMAT_RGBA32F:
	          if (this.textureFloatRenderable && (!filterable || this.textureFloatFilterable)) {
	            return format;
	          }
	          break;
	      }
	    }
	    return undefined;
	  };
	  _createClass(GraphicsDevice, [{
	    key: "width",
	    get: function get() {
	      return this.canvas.width;
	    }

	    /**
	     * Height of the back buffer in pixels.
	     *
	     * @type {number}
	     */
	  }, {
	    key: "height",
	    get: function get() {
	      return this.canvas.height;
	    }

	    /**
	     * Fullscreen mode.
	     *
	     * @type {boolean}
	     */
	  }, {
	    key: "fullscreen",
	    get: function get() {
	      Debug.error("GraphicsDevice.fullscreen is not implemented on current device.");
	      return false;
	    }

	    /**
	     * Maximum pixel ratio.
	     *
	     * @type {number}
	     */,
	    set: function set(fullscreen) {
	      Debug.error("GraphicsDevice.fullscreen is not implemented on current device.");
	    }
	  }, {
	    key: "maxPixelRatio",
	    get: function get() {
	      return this._maxPixelRatio;
	    }

	    /**
	     * The type of the device. Can be one of pc.DEVICETYPE_WEBGL1, pc.DEVICETYPE_WEBGL2 or pc.DEVICETYPE_WEBGPU.
	     *
	     * @type {new Function("modulePath", "return import(modulePath)")('./constants.js').DEVICETYPE_WEBGL1 | new Function("modulePath", "return import(modulePath)")('./constants.js').DEVICETYPE_WEBGL2 | new Function("modulePath", "return import(modulePath)")('./constants.js').DEVICETYPE_WEBGPU}
	     */,
	    set: function set(ratio) {
	      this._maxPixelRatio = ratio;
	    }
	  }, {
	    key: "deviceType",
	    get: function get() {
	      return this._deviceType;
	    }
	  }]);
	  return GraphicsDevice;
	}(EventHandler);
	GraphicsDevice.EVENT_RESIZE = 'resizecanvas';

	/**
	 * Internal graphics debug system - gpu markers and similar. Note that the functions only execute in the
	 * debug build, and are stripped out in other builds.
	 *
	 * @ignore
	 */
	var DebugGraphics = /*#__PURE__*/function () {
	  function DebugGraphics() {}
	  /**
	   * Clear internal stack of the GPU markers. It should be called at the start of the frame to
	   * prevent the array growing if there are exceptions during the rendering.
	   */
	  DebugGraphics.clearGpuMarkers = function clearGpuMarkers() {
	    DebugGraphics.markers.length = 0;
	  }

	  /**
	   * Push GPU marker to the stack on the device.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./graphics-device.js').GraphicsDevice} device - The graphics device.
	   * @param {string} name - The name of the marker.
	   */;
	  DebugGraphics.pushGpuMarker = function pushGpuMarker(device, name) {
	    DebugGraphics.markers.push(name);
	    device.pushMarker(name);
	  }

	  /**
	   * Pop GPU marker from the stack on the device.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./graphics-device.js').GraphicsDevice} device - The graphics device.
	   */;
	  DebugGraphics.popGpuMarker = function popGpuMarker(device) {
	    if (DebugGraphics.markers.length) {
	      DebugGraphics.markers.pop();
	    }
	    device.popMarker();
	  }

	  /**
	   * Converts current markers into a single string format.
	   *
	   * @returns {string} String representation of current markers.
	   */;
	  DebugGraphics.toString = function toString() {
	    return DebugGraphics.markers.join(" | ");
	  };
	  return DebugGraphics;
	}();
	/**
	 * An array of markers, representing a stack.
	 *
	 * @type {string[]}
	 * @private
	 */
	DebugGraphics.markers = [];

	var id$9 = 0;

	/**
	 * A render target is a rectangular rendering surface.
	 *
	 * @category Graphics
	 */
	var RenderTarget = /*#__PURE__*/function () {
	  /**
	   * Creates a new RenderTarget instance. A color buffer or a depth buffer must be set.
	   *
	   * @param {object} [options] - Object for passing optional arguments.
	   * @param {boolean} [options.autoResolve] - If samples > 1, enables or disables automatic MSAA
	   * resolve after rendering to this RT (see {@link RenderTarget#resolve}). Defaults to true.
	   * @param {new Function("modulePath", "return import(modulePath)")('./texture.js').Texture} [options.colorBuffer] - The texture that this render
	   * target will treat as a rendering surface.
	   * @param {new Function("modulePath", "return import(modulePath)")('./texture.js').Texture[]} [options.colorBuffers] - The textures that this
	   * render target will treat as a rendering surfaces. If this option is set, the colorBuffer
	   * option is ignored. This option can be used only when {@link GraphicsDevice#supportsMrt} is
	   * true.
	   * @param {boolean} [options.depth] - If set to true, depth buffer will be created. Defaults to
	   * true. Ignored if depthBuffer is defined.
	   * @param {new Function("modulePath", "return import(modulePath)")('./texture.js').Texture} [options.depthBuffer] - The texture that this render
	   * target will treat as a depth/stencil surface (WebGL2 only). If set, the 'depth' and
	   * 'stencil' properties are ignored. Texture must have {@link PIXELFORMAT_DEPTH} or
	   * {@link PIXELFORMAT_DEPTHSTENCIL} format.
	   * @param {number} [options.face] - If the colorBuffer parameter is a cubemap, use this option
	   * to specify the face of the cubemap to render to. Can be:
	   *
	   * - {@link CUBEFACE_POSX}
	   * - {@link CUBEFACE_NEGX}
	   * - {@link CUBEFACE_POSY}
	   * - {@link CUBEFACE_NEGY}
	   * - {@link CUBEFACE_POSZ}
	   * - {@link CUBEFACE_NEGZ}
	   *
	   * Defaults to {@link CUBEFACE_POSX}.
	   * @param {boolean} [options.flipY] - When set to true the image will be flipped in Y. Default
	   * is false.
	   * @param {string} [options.name] - The name of the render target.
	   * @param {number} [options.samples] - Number of hardware anti-aliasing samples (not supported
	   * on WebGL1). Default is 1.
	   * @param {boolean} [options.stencil] - If set to true, depth buffer will include stencil.
	   * Defaults to false. Ignored if depthBuffer is defined or depth is false.
	   * @example
	   * // Create a 512x512x24-bit render target with a depth buffer
	   * const colorBuffer = new pc.Texture(graphicsDevice, {
	   *     width: 512,
	   *     height: 512,
	   *     format: pc.PIXELFORMAT_RGB8
	   * });
	   * const renderTarget = new pc.RenderTarget({
	   *     colorBuffer: colorBuffer,
	   *     depth: true
	   * });
	   *
	   * // Set the render target on a camera component
	   * camera.renderTarget = renderTarget;
	   *
	   * // Destroy render target at a later stage. Note that the color buffer needs
	   * // to be destroyed separately.
	   * renderTarget.colorBuffer.destroy();
	   * renderTarget.destroy();
	   * camera.renderTarget = null;
	   */
	  function RenderTarget(options) {
	    var _options$face,
	      _this$_colorBuffer,
	      _this$_depthBuffer,
	      _this = this,
	      _options$samples,
	      _options$autoResolve,
	      _options$flipY,
	      _this$_colorBuffers;
	    if (options === void 0) {
	      options = {};
	    }
	    /**
	     * The name of the render target.
	     *
	     * @type {string}
	     */
	    this.name = void 0;
	    /**
	     * @type {new Function("modulePath", "return import(modulePath)")('./graphics-device.js').GraphicsDevice}
	     * @private
	     */
	    this._device = void 0;
	    /**
	     * @type {new Function("modulePath", "return import(modulePath)")('./texture.js').Texture}
	     * @private
	     */
	    this._colorBuffer = void 0;
	    /**
	     * @type {new Function("modulePath", "return import(modulePath)")('./texture.js').Texture[]}
	     * @private
	     */
	    this._colorBuffers = void 0;
	    /**
	     * @type {new Function("modulePath", "return import(modulePath)")('./texture.js').Texture}
	     * @private
	     */
	    this._depthBuffer = void 0;
	    /**
	     * @type {boolean}
	     * @private
	     */
	    this._depth = void 0;
	    /**
	     * @type {boolean}
	     * @private
	     */
	    this._stencil = void 0;
	    /**
	     * @type {number}
	     * @private
	     */
	    this._samples = void 0;
	    /** @type {boolean} */
	    this.autoResolve = void 0;
	    /**
	     * @type {number}
	     * @private
	     */
	    this._face = void 0;
	    /** @type {boolean} */
	    this.flipY = void 0;
	    this.id = id$9++;
	    var _arg2 = arguments[1];
	    var _arg3 = arguments[2];
	    if (options instanceof GraphicsDevice) {
	      // old constructor
	      this._colorBuffer = _arg2;
	      options = _arg3;
	      Debug.deprecated('pc.RenderTarget constructor no longer accepts GraphicsDevice parameter.');
	    } else {
	      // new constructor
	      this._colorBuffer = options.colorBuffer;
	    }

	    // Use the single colorBuffer in the colorBuffers array. This allows us to always just use the array internally.
	    if (this._colorBuffer) {
	      this._colorBuffers = [this._colorBuffer];
	    }

	    // Process optional arguments
	    this._depthBuffer = options.depthBuffer;
	    this._face = (_options$face = options.face) != null ? _options$face : 0;
	    if (this._depthBuffer) {
	      var format = this._depthBuffer._format;
	      if (format === PIXELFORMAT_DEPTH) {
	        this._depth = true;
	        this._stencil = false;
	      } else if (format === PIXELFORMAT_DEPTHSTENCIL) {
	        this._depth = true;
	        this._stencil = true;
	      } else {
	        Debug.warn('Incorrect depthBuffer format. Must be pc.PIXELFORMAT_DEPTH or pc.PIXELFORMAT_DEPTHSTENCIL');
	        this._depth = false;
	        this._stencil = false;
	      }
	    } else {
	      var _options$depth, _options$stencil;
	      this._depth = (_options$depth = options.depth) != null ? _options$depth : true;
	      this._stencil = (_options$stencil = options.stencil) != null ? _options$stencil : false;
	    }

	    // MRT
	    if (options.colorBuffers) {
	      Debug.assert(!this._colorBuffers, 'When constructing RenderTarget and options.colorBuffers is used, options.colorBuffer must not be used.');
	      if (!this._colorBuffers) {
	        this._colorBuffers = [].concat(options.colorBuffers);

	        // set the main color buffer to point to 0 index
	        this._colorBuffer = options.colorBuffers[0];
	      }
	    }

	    // device, from one of the buffers
	    var device = ((_this$_colorBuffer = this._colorBuffer) == null ? void 0 : _this$_colorBuffer.device) || ((_this$_depthBuffer = this._depthBuffer) == null ? void 0 : _this$_depthBuffer.device) || options.graphicsDevice;
	    Debug.assert(device, "Failed to obtain the device, colorBuffer nor depthBuffer store it.");
	    this._device = device;
	    Debug.call(function () {
	      if (_this._colorBuffers) {
	        Debug.assert(_this._colorBuffers.length <= 1 || device.supportsMrt, 'Multiple render targets are not supported on this device');
	      }
	    });
	    var maxSamples = this._device.maxSamples;
	    this._samples = Math.min((_options$samples = options.samples) != null ? _options$samples : 1, maxSamples);

	    // WebGPU only supports values of 1 or 4 for samples
	    if (device.isWebGPU) {
	      this._samples = this._samples > 1 ? maxSamples : 1;
	    }
	    this.autoResolve = (_options$autoResolve = options.autoResolve) != null ? _options$autoResolve : true;

	    // use specified name, otherwise get one from color or depth buffer
	    this.name = options.name;
	    if (!this.name) {
	      var _this$_colorBuffer2;
	      this.name = (_this$_colorBuffer2 = this._colorBuffer) == null ? void 0 : _this$_colorBuffer2.name;
	    }
	    if (!this.name) {
	      var _this$_depthBuffer2;
	      this.name = (_this$_depthBuffer2 = this._depthBuffer) == null ? void 0 : _this$_depthBuffer2.name;
	    }
	    if (!this.name) {
	      this.name = "Untitled";
	    }

	    // render image flipped in Y
	    this.flipY = (_options$flipY = options.flipY) != null ? _options$flipY : false;
	    this.validateMrt();

	    // device specific implementation
	    this.impl = device.createRenderTargetImpl(this);
	    Debug.trace(TRACEID_RENDER_TARGET_ALLOC, "Alloc: Id " + this.id + " " + this.name + ": " + this.width + "x" + this.height + " " + ("[samples: " + this.samples + "]") + ("" + ((_this$_colorBuffers = this._colorBuffers) != null && _this$_colorBuffers.length ? "[MRT: " + this._colorBuffers.length + "]" : '')) + ("" + (this.colorBuffer ? '[Color]' : '')) + ("" + (this.depth ? '[Depth]' : '')) + ("" + (this.stencil ? '[Stencil]' : '')) + ("[Face:" + this.face + "]"));
	  }

	  /**
	   * Frees resources associated with this render target.
	   */
	  var _proto = RenderTarget.prototype;
	  _proto.destroy = function destroy() {
	    Debug.trace(TRACEID_RENDER_TARGET_ALLOC, "DeAlloc: Id " + this.id + " " + this.name);
	    var device = this._device;
	    if (device) {
	      device.targets.delete(this);
	      if (device.renderTarget === this) {
	        device.setRenderTarget(null);
	      }
	      this.destroyFrameBuffers();
	    }
	  }

	  /**
	   * Free device resources associated with this render target.
	   *
	   * @ignore
	   */;
	  _proto.destroyFrameBuffers = function destroyFrameBuffers() {
	    var device = this._device;
	    if (device) {
	      this.impl.destroy(device);
	    }
	  }

	  /**
	   * Free textures associated with this render target.
	   *
	   * @ignore
	   */;
	  _proto.destroyTextureBuffers = function destroyTextureBuffers() {
	    var _this$_depthBuffer3, _this$_colorBuffers2;
	    (_this$_depthBuffer3 = this._depthBuffer) == null || _this$_depthBuffer3.destroy();
	    this._depthBuffer = null;
	    (_this$_colorBuffers2 = this._colorBuffers) == null || _this$_colorBuffers2.forEach(function (colorBuffer) {
	      colorBuffer.destroy();
	    });
	    this._colorBuffers = null;
	    this._colorBuffer = null;
	  }

	  /**
	   * Resizes the render target to the specified width and height. Internally this resizes all the
	   * assigned texture color and depth buffers.
	   *
	   * @param {number} width - The width of the render target in pixels.
	   * @param {number} height - The height of the render target in pixels.
	   */;
	  _proto.resize = function resize(width, height) {
	    if (this.width !== width || this.height !== height) {
	      var _this$_depthBuffer4, _this$_colorBuffers3;
	      // release existing
	      var device = this._device;
	      this.destroyFrameBuffers();
	      if (device.renderTarget === this) {
	        device.setRenderTarget(null);
	      }

	      // resize textures
	      (_this$_depthBuffer4 = this._depthBuffer) == null || _this$_depthBuffer4.resize(width, height);
	      (_this$_colorBuffers3 = this._colorBuffers) == null || _this$_colorBuffers3.forEach(function (colorBuffer) {
	        colorBuffer.resize(width, height);
	      });

	      // initialize again
	      this.validateMrt();
	      this.impl = device.createRenderTargetImpl(this);
	    }
	  };
	  _proto.validateMrt = function validateMrt() {
	    var _this2 = this;
	    Debug.call(function () {
	      if (_this2._colorBuffers) {
	        var _this2$_colorBuffers$ = _this2._colorBuffers[0],
	          width = _this2$_colorBuffers$.width,
	          height = _this2$_colorBuffers$.height,
	          cubemap = _this2$_colorBuffers$.cubemap,
	          volume = _this2$_colorBuffers$.volume;
	        for (var i = 1; i < _this2._colorBuffers.length; i++) {
	          var colorBuffer = _this2._colorBuffers[i];
	          Debug.assert(colorBuffer.width === width, 'All render target color buffers must have the same width', _this2);
	          Debug.assert(colorBuffer.height === height, 'All render target color buffers must have the same height', _this2);
	          Debug.assert(colorBuffer.cubemap === cubemap, 'All render target color buffers must have the same cubemap setting', _this2);
	          Debug.assert(colorBuffer.volume === volume, 'All render target color buffers must have the same volume setting', _this2);
	        }
	      }
	    });
	  }

	  /**
	   * Initializes the resources associated with this render target.
	   *
	   * @ignore
	   */;
	  _proto.init = function init() {
	    this.impl.init(this._device, this);
	  }

	  /** @ignore */;
	  /**
	   * Called when the device context was lost. It releases all context related resources.
	   *
	   * @ignore
	   */
	  _proto.loseContext = function loseContext() {
	    this.impl.loseContext();
	  }

	  /**
	   * If samples > 1, resolves the anti-aliased render target (WebGL2 only). When you're rendering
	   * to an anti-aliased render target, pixels aren't written directly to the readable texture.
	   * Instead, they're first written to a MSAA buffer, where each sample for each pixel is stored
	   * independently. In order to read the results, you first need to 'resolve' the buffer - to
	   * average all samples and create a simple texture with one color per pixel. This function
	   * performs this averaging and updates the colorBuffer and the depthBuffer. If autoResolve is
	   * set to true, the resolve will happen after every rendering to this render target, otherwise
	   * you can do it manually, during the app update or inside a {@link Command}.
	   *
	   * @param {boolean} [color] - Resolve color buffer. Defaults to true.
	   * @param {boolean} [depth] - Resolve depth buffer. Defaults to true if the render target has a
	   * depth buffer.
	   */;
	  _proto.resolve = function resolve(color, depth) {
	    if (color === void 0) {
	      color = true;
	    }
	    if (depth === void 0) {
	      depth = !!this._depthBuffer;
	    }
	    // TODO: consider adding support for MRT to this function.

	    if (this._device && this._samples > 1) {
	      DebugGraphics.pushGpuMarker(this._device, "RESOLVE-RT:" + this.name);
	      this.impl.resolve(this._device, this, color, depth);
	      DebugGraphics.popGpuMarker(this._device);
	    }
	  }

	  /**
	   * Copies color and/or depth contents of source render target to this one. Formats, sizes and
	   * anti-aliasing samples must match. Depth buffer can only be copied on WebGL 2.0.
	   *
	   * @param {RenderTarget} source - Source render target to copy from.
	   * @param {boolean} [color] - If true will copy the color buffer. Defaults to false.
	   * @param {boolean} [depth] - If true will copy the depth buffer. Defaults to false.
	   * @returns {boolean} True if the copy was successful, false otherwise.
	   */;
	  _proto.copy = function copy(source, color, depth) {
	    // TODO: consider adding support for MRT to this function.

	    if (!this._device) {
	      if (source._device) {
	        this._device = source._device;
	      } else {
	        Debug.error("Render targets are not initialized");
	        return false;
	      }
	    }
	    DebugGraphics.pushGpuMarker(this._device, "COPY-RT:" + source.name + "->" + this.name);
	    var success = this._device.copyRenderTarget(source, this, color, depth);
	    DebugGraphics.popGpuMarker(this._device);
	    return success;
	  }

	  /**
	   * Number of antialiasing samples the render target uses.
	   *
	   * @type {number}
	   */;
	  /**
	   * Accessor for multiple render target color buffers.
	   *
	   * @param {*} index - Index of the color buffer to get.
	   * @returns {new Function("modulePath", "return import(modulePath)")('./texture.js').Texture} - Color buffer at the specified index.
	   */
	  _proto.getColorBuffer = function getColorBuffer(index) {
	    var _this$_colorBuffers4;
	    return (_this$_colorBuffers4 = this._colorBuffers) == null ? void 0 : _this$_colorBuffers4[index];
	  }

	  /**
	   * Depth buffer set up on the render target. Only available, if depthBuffer was set in
	   * constructor. Not available if depth property was used instead.
	   *
	   * @type {new Function("modulePath", "return import(modulePath)")('./texture.js').Texture}
	   */;
	  _createClass(RenderTarget, [{
	    key: "initialized",
	    get: function get() {
	      return this.impl.initialized;
	    }

	    /** @ignore */
	  }, {
	    key: "device",
	    get: function get() {
	      return this._device;
	    }
	  }, {
	    key: "samples",
	    get: function get() {
	      return this._samples;
	    }

	    /**
	     * True if the render target contains the depth attachment.
	     *
	     * @type {boolean}
	     */
	  }, {
	    key: "depth",
	    get: function get() {
	      return this._depth;
	    }

	    /**
	     * True if the render target contains the stencil attachment.
	     *
	     * @type {boolean}
	     */
	  }, {
	    key: "stencil",
	    get: function get() {
	      return this._stencil;
	    }

	    /**
	     * Color buffer set up on the render target.
	     *
	     * @type {new Function("modulePath", "return import(modulePath)")('./texture.js').Texture}
	     */
	  }, {
	    key: "colorBuffer",
	    get: function get() {
	      return this._colorBuffer;
	    }
	  }, {
	    key: "depthBuffer",
	    get: function get() {
	      return this._depthBuffer;
	    }

	    /**
	     * If the render target is bound to a cubemap, this property specifies which face of the
	     * cubemap is rendered to. Can be:
	     *
	     * - {@link CUBEFACE_POSX}
	     * - {@link CUBEFACE_NEGX}
	     * - {@link CUBEFACE_POSY}
	     * - {@link CUBEFACE_NEGY}
	     * - {@link CUBEFACE_POSZ}
	     * - {@link CUBEFACE_NEGZ}
	     *
	     * @type {number}
	     */
	  }, {
	    key: "face",
	    get: function get() {
	      return this._face;
	    }

	    /**
	     * Width of the render target in pixels.
	     *
	     * @type {number}
	     */
	  }, {
	    key: "width",
	    get: function get() {
	      var _this$_colorBuffer3, _this$_depthBuffer5;
	      return ((_this$_colorBuffer3 = this._colorBuffer) == null ? void 0 : _this$_colorBuffer3.width) || ((_this$_depthBuffer5 = this._depthBuffer) == null ? void 0 : _this$_depthBuffer5.width) || this._device.width;
	    }

	    /**
	     * Height of the render target in pixels.
	     *
	     * @type {number}
	     */
	  }, {
	    key: "height",
	    get: function get() {
	      var _this$_colorBuffer4, _this$_depthBuffer6;
	      return ((_this$_colorBuffer4 = this._colorBuffer) == null ? void 0 : _this$_colorBuffer4.height) || ((_this$_depthBuffer6 = this._depthBuffer) == null ? void 0 : _this$_depthBuffer6.height) || this._device.height;
	    }
	  }]);
	  return RenderTarget;
	}();

	// Maximum number of times a duplicate error message is logged.
	var MAX_DUPLICATES = 5;

	/**
	 * Internal WebGPU debug system. Note that the functions only execute in the debug build, and are
	 * stripped out in other builds.
	 *
	 * @ignore
	 */
	var WebgpuDebug = /*#__PURE__*/function () {
	  function WebgpuDebug() {}
	  /**
	   * Start a validation error scope.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./webgpu-graphics-device.js').WebgpuGraphicsDevice} device - The graphics
	   * device.
	   */
	  WebgpuDebug.validate = function validate(device) {
	    device.wgpu.pushErrorScope('validation');
	    WebgpuDebug._scopes.push('validation');
	    WebgpuDebug._markers.push(DebugGraphics.toString());
	  }

	  /**
	   * Start an out-of-memory error scope.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./webgpu-graphics-device.js').WebgpuGraphicsDevice} device - The graphics
	   * device.
	   */;
	  WebgpuDebug.memory = function memory(device) {
	    device.wgpu.pushErrorScope('out-of-memory');
	    WebgpuDebug._scopes.push('out-of-memory');
	    WebgpuDebug._markers.push(DebugGraphics.toString());
	  }

	  /**
	   * Start an internal error scope.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./webgpu-graphics-device.js').WebgpuGraphicsDevice} device - The graphics
	   * device.
	   */;
	  WebgpuDebug.internal = function internal(device) {
	    device.wgpu.pushErrorScope('internal');
	    WebgpuDebug._scopes.push('internal');
	    WebgpuDebug._markers.push(DebugGraphics.toString());
	  }

	  /**
	   * End the previous error scope, and print errors if any.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./webgpu-graphics-device.js').WebgpuGraphicsDevice} device - The graphics
	   * device.
	   * @param {...any} args - Additional parameters that form the error message.
	   */;
	  WebgpuDebug.end = function end(device) {
	    for (var _len = arguments.length, args = new Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
	      args[_key - 1] = arguments[_key];
	    }
	    var header = WebgpuDebug._scopes.pop();
	    var marker = WebgpuDebug._markers.pop();
	    Debug.assert(header, 'Non matching end.');
	    device.wgpu.popErrorScope().then(function (error) {
	      if (error) {
	        var _WebgpuDebug$_loggedM;
	        var count = (_WebgpuDebug$_loggedM = WebgpuDebug._loggedMessages.get(error.message)) != null ? _WebgpuDebug$_loggedM : 0;
	        if (count < MAX_DUPLICATES) {
	          var _console;
	          var tooMany = count === MAX_DUPLICATES - 1 ? ' (Too many errors, ignoring this one from now)' : '';
	          WebgpuDebug._loggedMessages.set(error.message, count + 1);
	          (_console = console).error.apply(_console, ["WebGPU " + header + " error: " + error.message, tooMany, "while rendering", marker].concat(args));
	        }
	      }
	    });
	  };
	  return WebgpuDebug;
	}();
	WebgpuDebug._scopes = [];
	WebgpuDebug._markers = [];
	/** @type {Map<string,number>} */
	WebgpuDebug._loggedMessages = new Map();

	/**
	 * A WebGPU implementation of the BindGroup, which is a wrapper over GPUBindGroup.
	 *
	 * @ignore
	 */
	var WebgpuBindGroup = /*#__PURE__*/function () {
	  function WebgpuBindGroup() {
	    /**
	     * @type {GPUBindGroup}
	     * @private
	     */
	    this.bindGroup = void 0;
	  }
	  var _proto = WebgpuBindGroup.prototype;
	  _proto.update = function update(bindGroup) {
	    this.destroy();
	    var device = bindGroup.device;

	    /** @type {GPUBindGroupDescriptor} */
	    var descr = this.createDescriptor(device, bindGroup);
	    WebgpuDebug.validate(device);
	    this.bindGroup = device.wgpu.createBindGroup(descr);
	    WebgpuDebug.end(device, {
	      debugFormat: this.debugFormat,
	      descr: descr,
	      format: bindGroup.format,
	      bindGroup: bindGroup
	    });
	  };
	  _proto.destroy = function destroy() {
	    // this.bindGroup?.destroy();
	    this.bindGroup = null;
	  }

	  /**
	   * Creates a bind group descriptor in WebGPU format
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./webgpu-graphics-device.js').WebgpuGraphicsDevice} device - Graphics device.
	   * @param {new Function("modulePath", "return import(modulePath)")('../bind-group.js').BindGroup} bindGroup - Bind group to create the
	   * descriptor for.
	   * @returns {object} - Returns the generated descriptor of type
	   * GPUBindGroupDescriptor, which can be used to create a GPUBindGroup
	   */;
	  _proto.createDescriptor = function createDescriptor(device, bindGroup) {
	    var _this = this;
	    // Note: This needs to match WebgpuBindGroupFormat.createDescriptor
	    var entries = [];
	    var format = bindGroup.format;
	    Debug.call(function () {
	      _this.debugFormat = '';
	    });

	    // uniform buffers
	    var index = 0;
	    bindGroup.uniformBuffers.forEach(function (ub) {
	      var buffer = ub.persistent ? ub.impl.buffer : ub.allocation.gpuBuffer.buffer;
	      Debug.assert(buffer, 'NULL uniform buffer cannot be used by the bind group');
	      Debug.call(function () {
	        _this.debugFormat += index + ": UB\n";
	      });
	      entries.push({
	        binding: index++,
	        resource: {
	          buffer: buffer,
	          offset: 0,
	          size: ub.format.byteSize
	        }
	      });
	    });

	    // textures
	    bindGroup.textures.forEach(function (tex, textureIndex) {
	      /** @type {new Function("modulePath", "return import(modulePath)")('./webgpu-texture.js').WebgpuTexture} */
	      var wgpuTexture = tex.impl;
	      var textureFormat = format.textureFormats[textureIndex];

	      // texture
	      var view = wgpuTexture.getView(device);
	      Debug.assert(view, 'NULL texture view cannot be used by the bind group');
	      Debug.call(function () {
	        _this.debugFormat += index + ": " + bindGroup.format.textureFormats[textureIndex].name + "\n";
	      });
	      entries.push({
	        binding: index++,
	        resource: view
	      });

	      // sampler
	      var sampler = wgpuTexture.getSampler(device, textureFormat.sampleType);
	      Debug.assert(sampler, 'NULL sampler cannot be used by the bind group');
	      Debug.call(function () {
	        _this.debugFormat += index + ": " + sampler.label + "\n";
	      });
	      entries.push({
	        binding: index++,
	        resource: sampler
	      });
	    });

	    // storage textures
	    bindGroup.storageTextures.forEach(function (tex, textureIndex) {
	      /** @type {new Function("modulePath", "return import(modulePath)")('./webgpu-texture.js').WebgpuTexture} */
	      var wgpuTexture = tex.impl;

	      // texture
	      var view = wgpuTexture.getView(device);
	      Debug.assert(view, 'NULL texture view cannot be used by the bind group');
	      Debug.call(function () {
	        _this.debugFormat += index + ": " + bindGroup.format.storageTextureFormats[textureIndex].name + "\n";
	      });
	      entries.push({
	        binding: index++,
	        resource: view
	      });
	    });
	    var descr = {
	      layout: bindGroup.format.impl.bindGroupLayout,
	      entries: entries
	    };
	    DebugHelper.setLabel(descr, bindGroup.name);
	    return descr;
	  };
	  return WebgpuBindGroup;
	}();

	var WebgpuUtils = /*#__PURE__*/function () {
	  function WebgpuUtils() {}
	  // converts a combination of SHADER_STAGE_* into GPUShaderStage.*
	  WebgpuUtils.shaderStage = function shaderStage(stage) {
	    var ret = 0;
	    if (stage & SHADERSTAGE_VERTEX) ret |= GPUShaderStage.VERTEX;
	    if (stage & SHADERSTAGE_FRAGMENT) ret |= GPUShaderStage.FRAGMENT;
	    if (stage & SHADERSTAGE_COMPUTE) ret |= GPUShaderStage.COMPUTE;
	    return ret;
	  };
	  return WebgpuUtils;
	}();

	// map of PIXELFORMAT_*** to GPUTextureFormat
	var gpuTextureFormats = [];
	gpuTextureFormats[PIXELFORMAT_A8] = '';
	gpuTextureFormats[PIXELFORMAT_L8] = 'r8unorm';
	gpuTextureFormats[PIXELFORMAT_LA8] = 'rg8unorm';
	gpuTextureFormats[PIXELFORMAT_RGB565] = '';
	gpuTextureFormats[PIXELFORMAT_RGBA5551] = '';
	gpuTextureFormats[PIXELFORMAT_RGBA4] = '';
	gpuTextureFormats[PIXELFORMAT_RGB8] = 'rgba8unorm';
	gpuTextureFormats[PIXELFORMAT_RGBA8] = 'rgba8unorm';
	gpuTextureFormats[PIXELFORMAT_DXT1] = 'bc1-rgba-unorm';
	gpuTextureFormats[PIXELFORMAT_DXT3] = 'bc2-rgba-unorm';
	gpuTextureFormats[PIXELFORMAT_DXT5] = 'bc3-rgba-unorm';
	gpuTextureFormats[PIXELFORMAT_RGB16F] = '';
	gpuTextureFormats[PIXELFORMAT_RGBA16F] = 'rgba16float';
	gpuTextureFormats[PIXELFORMAT_R16F] = 'r16float';
	gpuTextureFormats[PIXELFORMAT_RG16F] = 'rg16float';
	gpuTextureFormats[PIXELFORMAT_RGB32F] = '';
	gpuTextureFormats[PIXELFORMAT_RGBA32F] = 'rgba32float';
	gpuTextureFormats[PIXELFORMAT_R32F] = 'r32float';
	gpuTextureFormats[PIXELFORMAT_DEPTH] = 'depth32float';
	gpuTextureFormats[PIXELFORMAT_DEPTHSTENCIL] = 'depth24plus-stencil8';
	gpuTextureFormats[PIXELFORMAT_111110F] = 'rg11b10ufloat';
	gpuTextureFormats[PIXELFORMAT_SRGB] = '';
	gpuTextureFormats[PIXELFORMAT_SRGBA] = '';
	gpuTextureFormats[PIXELFORMAT_ETC1] = '';
	gpuTextureFormats[PIXELFORMAT_ETC2_RGB] = 'etc2-rgb8unorm';
	gpuTextureFormats[PIXELFORMAT_ETC2_RGBA] = 'etc2-rgba8unorm';
	gpuTextureFormats[PIXELFORMAT_PVRTC_2BPP_RGB_1] = '';
	gpuTextureFormats[PIXELFORMAT_PVRTC_2BPP_RGBA_1] = '';
	gpuTextureFormats[PIXELFORMAT_PVRTC_4BPP_RGB_1] = '';
	gpuTextureFormats[PIXELFORMAT_PVRTC_4BPP_RGBA_1] = '';
	gpuTextureFormats[PIXELFORMAT_ASTC_4x4] = 'astc-4x4-unorm';
	gpuTextureFormats[PIXELFORMAT_ATC_RGB] = '';
	gpuTextureFormats[PIXELFORMAT_ATC_RGBA] = '';
	gpuTextureFormats[PIXELFORMAT_BGRA8] = 'bgra8unorm';
	gpuTextureFormats[PIXELFORMAT_R8I] = 'r8sint';
	gpuTextureFormats[PIXELFORMAT_R8U] = 'r8uint';
	gpuTextureFormats[PIXELFORMAT_R16I] = 'r16sint';
	gpuTextureFormats[PIXELFORMAT_R16U] = 'r16uint';
	gpuTextureFormats[PIXELFORMAT_R32I] = 'r32sint';
	gpuTextureFormats[PIXELFORMAT_R32U] = 'r32uint';
	gpuTextureFormats[PIXELFORMAT_RG8I] = 'rg8sint';
	gpuTextureFormats[PIXELFORMAT_RG8U] = 'rg8uint';
	gpuTextureFormats[PIXELFORMAT_RG16I] = 'rg16sint';
	gpuTextureFormats[PIXELFORMAT_RG16U] = 'rg16uint';
	gpuTextureFormats[PIXELFORMAT_RG32I] = 'rg32sint';
	gpuTextureFormats[PIXELFORMAT_RG32U] = 'rg32uint';
	gpuTextureFormats[PIXELFORMAT_RGBA8I] = 'rgba8sint';
	gpuTextureFormats[PIXELFORMAT_RGBA8U] = 'rgba8uint';
	gpuTextureFormats[PIXELFORMAT_RGBA16I] = 'rgba16sint';
	gpuTextureFormats[PIXELFORMAT_RGBA16U] = 'rgba16uint';
	gpuTextureFormats[PIXELFORMAT_RGBA32I] = 'rgba32sint';
	gpuTextureFormats[PIXELFORMAT_RGBA32U] = 'rgba32uint';

	var samplerTypes = [];
	samplerTypes[SAMPLETYPE_FLOAT] = 'filtering';
	samplerTypes[SAMPLETYPE_UNFILTERABLE_FLOAT] = 'non-filtering';
	samplerTypes[SAMPLETYPE_DEPTH] = 'comparison';

	// Using 'comparison' instead of 'non-filtering' may seem unusual, but currently we will get a
	// validation error if we use 'non-filtering' along with texelFetch/textureLoad. 'comparison' works
	// very well for the most common use-case of integer textures, texelFetch. We may be able to change
	// how we initialize the sampler elsewhere to support 'non-filtering' in the future.
	samplerTypes[SAMPLETYPE_INT] = 'comparison';
	samplerTypes[SAMPLETYPE_UINT] = 'comparison';
	var sampleTypes = [];
	sampleTypes[SAMPLETYPE_FLOAT] = 'float';
	sampleTypes[SAMPLETYPE_UNFILTERABLE_FLOAT] = 'unfilterable-float';
	sampleTypes[SAMPLETYPE_DEPTH] = 'depth';
	sampleTypes[SAMPLETYPE_INT] = 'sint';
	sampleTypes[SAMPLETYPE_UINT] = 'uint';
	var stringIds$1 = new StringIds();

	/**
	 * A WebGPU implementation of the BindGroupFormat, which is a wrapper over GPUBindGroupLayout.
	 *
	 * @ignore
	 */
	var WebgpuBindGroupFormat = /*#__PURE__*/function () {
	  /**
	   * @param {new Function("modulePath", "return import(modulePath)")('../bind-group-format.js').BindGroupFormat} bindGroupFormat - Bind group format.
	   */
	  function WebgpuBindGroupFormat(bindGroupFormat) {
	    var _this = this;
	    /** @type {new Function("modulePath", "return import(modulePath)")('./webgpu-graphics-device.js').WebgpuGraphicsDevice} */
	    var device = bindGroupFormat.device;
	    var _this$createDescripto = this.createDescriptor(bindGroupFormat),
	      key = _this$createDescripto.key,
	      descr = _this$createDescripto.descr;

	    /**
	     * Unique key, used for caching
	     *
	     * @type {number}
	     */
	    this.key = stringIds$1.get(key);

	    // keep descr in debug mode
	    Debug.call(function () {
	      _this.descr = descr;
	    });

	    /**
	     * @type {GPUBindGroupLayout}
	     * @private
	     */
	    this.bindGroupLayout = device.wgpu.createBindGroupLayout(descr);
	    DebugHelper.setLabel(this.bindGroupLayout, bindGroupFormat.name);
	  }
	  var _proto = WebgpuBindGroupFormat.prototype;
	  _proto.destroy = function destroy() {
	    this.bindGroupLayout = null;
	  };
	  _proto.loseContext = function loseContext() {
	    // this.bindGroupLayout = null;
	  }

	  /**
	   * Returns texture binding slot.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('../bind-group-format.js').BindGroupFormat} bindGroupFormat - Bind group format.
	   * @param {number} index - The index of the texture.
	   * @returns {number} - The slot index.
	   */;
	  _proto.getTextureSlot = function getTextureSlot(bindGroupFormat, index) {
	    // each texture takes 2 slots (texture, sampler) and those are added after uniform buffers
	    return bindGroupFormat.bufferFormats.length + index * 2;
	  }

	  /**
	   * @param {any} bindGroupFormat - The format of the bind group.
	   * @returns {any} Returns the bind group descriptor.
	   */;
	  _proto.createDescriptor = function createDescriptor(bindGroupFormat) {
	    // all WebGPU bindings:
	    // - buffer: GPUBufferBindingLayout, resource type is GPUBufferBinding
	    // - sampler: GPUSamplerBindingLayout, resource type is GPUSampler
	    // - texture: GPUTextureBindingLayout, resource type is GPUTextureView
	    // - storageTexture: GPUStorageTextureBindingLayout, resource type is GPUTextureView
	    // - externalTexture: GPUExternalTextureBindingLayout, resource type is GPUExternalTexture
	    var entries = [];

	    // generate unique key
	    var key = '';
	    var index = 0;

	    // buffers
	    bindGroupFormat.bufferFormats.forEach(function (bufferFormat) {
	      var visibility = WebgpuUtils.shaderStage(bufferFormat.visibility);
	      key += "#" + index + "U:" + visibility;
	      entries.push({
	        binding: index++,
	        visibility: visibility,
	        buffer: {
	          type: 'uniform',
	          // "uniform", "storage", "read-only-storage"

	          // whether this binding requires a dynamic offset
	          // currently all UBs are dynamic and need the offset
	          hasDynamicOffset: true

	          // defaults to 0 meaning no validation, can do early size validation using it
	          // minBindingSize
	        }
	      });
	    });

	    // textures
	    bindGroupFormat.textureFormats.forEach(function (textureFormat) {
	      var visibility = WebgpuUtils.shaderStage(textureFormat.visibility);

	      // texture
	      var sampleType = textureFormat.sampleType;
	      var viewDimension = textureFormat.textureDimension;
	      var multisampled = false;
	      var gpuSampleType = sampleTypes[sampleType];
	      Debug.assert(gpuSampleType);
	      key += "#" + index + "T:" + visibility + "-" + gpuSampleType + "-" + viewDimension + "-" + multisampled;

	      // texture
	      entries.push({
	        binding: index++,
	        visibility: visibility,
	        texture: {
	          // Indicates the type required for texture views bound to this binding.
	          // "float", "unfilterable-float", "depth", "sint", "uint",
	          sampleType: gpuSampleType,
	          // Indicates the required dimension for texture views bound to this binding.
	          // "1d", "2d", "2d-array", "cube", "cube-array", "3d"
	          viewDimension: viewDimension,
	          // Indicates whether or not texture views bound to this binding must be multisampled
	          multisampled: multisampled
	        }
	      });

	      // sampler
	      var gpuSamplerType = samplerTypes[sampleType];
	      Debug.assert(gpuSamplerType);
	      key += "#" + index + "S:" + visibility + "-" + gpuSamplerType;
	      entries.push({
	        binding: index++,
	        visibility: visibility,
	        sampler: {
	          // Indicates the required type of a sampler bound to this bindings
	          // 'filtering', 'non-filtering', 'comparison'
	          type: gpuSamplerType
	        }
	      });
	    });

	    // storage textures
	    bindGroupFormat.storageTextureFormats.forEach(function (textureFormat) {
	      var format = textureFormat.format,
	        textureDimension = textureFormat.textureDimension;
	      key += "#" + index + "ST:" + format + "-" + textureDimension;

	      // storage texture
	      entries.push({
	        binding: index++,
	        visibility: GPUShaderStage.COMPUTE,
	        storageTexture: {
	          // The access mode for this binding, indicating readability and writability.
	          access: 'write-only',
	          // only single option currently, more in the future

	          // The required format of texture views bound to this binding.
	          format: gpuTextureFormats[format],
	          // Indicates the required dimension for texture views bound to this binding.
	          // "1d", "2d", "2d-array", "cube", "cube-array", "3d"
	          viewDimension: textureDimension
	        }
	      });
	    });

	    /** @type {GPUBindGroupLayoutDescriptor} */
	    var descr = {
	      entries: entries
	    };
	    return {
	      key: key,
	      descr: descr
	    };
	  };
	  return WebgpuBindGroupFormat;
	}();

	/**
	 * A WebGPU implementation of the Buffer.
	 *
	 * @ignore
	 */
	var WebgpuBuffer = /*#__PURE__*/function () {
	  function WebgpuBuffer() {
	    /**
	     * @type {GPUBuffer}
	     * @private
	     */
	    this.buffer = null;
	  }
	  var _proto = WebgpuBuffer.prototype;
	  _proto.destroy = function destroy(device) {
	    if (this.buffer) {
	      this.buffer.destroy();
	      this.buffer = null;
	    }
	  };
	  _proto.loseContext = function loseContext() {}

	  /**
	   * @param {new Function("modulePath", "return import(modulePath)")('./webgpu-graphics-device.js').WebgpuGraphicsDevice} device - Graphics device.
	   * @param {*} usage -
	   * @param {*} target -
	   * @param {*} storage -
	   */;
	  _proto.unlock = function unlock(device, usage, target, storage) {
	    var _storage$byteOffset, _storage$buffer;
	    var wgpu = device.wgpu;

	    // offset of getMappedRange must me a multiple of 8
	    // size of getMappedRange must be a multiple of 4

	    if (!this.buffer) {
	      // size needs to be a multiple of 4
	      var size = storage.byteLength + 3 & ~3;
	      this.buffer = device.wgpu.createBuffer({
	        size: size,
	        usage: target | GPUBufferUsage.COPY_DST
	      });
	      DebugHelper.setLabel(this.buffer, target & GPUBufferUsage.VERTEX ? 'VertexBuffer' : target & GPUBufferUsage.INDEX ? 'IndexBuffer' : target & GPUBufferUsage.UNIFORM ? "UniformBuffer" : '');

	      // mappedAtCreation path - this could be used when the data is provided

	      // this.buffer = device.wgpu.createBuffer({
	      //     size: size,
	      //     usage: target,
	      //     mappedAtCreation: true
	      // });

	      // const dest = new Uint8Array(this.buffer.getMappedRange());
	      // const src = new Uint8Array(storage.buffer ? storage.buffer : storage);
	      // dest.set(src);
	      // this.buffer.unmap();
	    }

	    // src size needs to be a multiple of 4 as well
	    var srcOffset = (_storage$byteOffset = storage.byteOffset) != null ? _storage$byteOffset : 0;
	    var srcData = new Uint8Array((_storage$buffer = storage.buffer) != null ? _storage$buffer : storage, srcOffset, storage.byteLength);
	    var data = new Uint8Array(this.buffer.size);
	    data.set(srcData);

	    // copy data to the gpu buffer
	    Debug.trace(TRACEID_RENDER_QUEUE, "writeBuffer: " + this.buffer.label);
	    wgpu.queue.writeBuffer(this.buffer, 0, data, 0, data.length);

	    // TODO: handle usage types:
	    // - BUFFER_STATIC, BUFFER_DYNAMIC, BUFFER_STREAM, BUFFER_GPUDYNAMIC
	  };
	  _createClass(WebgpuBuffer, [{
	    key: "initialized",
	    get: function get() {
	      return !!this.buffer;
	    }
	  }]);
	  return WebgpuBuffer;
	}();

	/**
	 * A WebGPU implementation of the IndexBuffer.
	 *
	 * @ignore
	 */
	var WebgpuIndexBuffer = /*#__PURE__*/function (_WebgpuBuffer) {
	  _inheritsLoose(WebgpuIndexBuffer, _WebgpuBuffer);
	  function WebgpuIndexBuffer(indexBuffer) {
	    var _this;
	    _this = _WebgpuBuffer.call(this) || this;
	    _this.format = null;
	    Debug.assert(indexBuffer.format !== INDEXFORMAT_UINT8, "WebGPU does not support 8-bit index buffer format");
	    _this.format = indexBuffer.format === INDEXFORMAT_UINT16 ? "uint16" : "uint32";
	    return _this;
	  }
	  var _proto = WebgpuIndexBuffer.prototype;
	  _proto.unlock = function unlock(indexBuffer) {
	    var device = indexBuffer.device;
	    _WebgpuBuffer.prototype.unlock.call(this, device, indexBuffer.usage, GPUBufferUsage.INDEX, indexBuffer.storage);
	  };
	  return WebgpuIndexBuffer;
	}(WebgpuBuffer);

	var array = {
	  // helper function to compare two arrays for equality
	  equals: function equals(arr1, arr2) {
	    if (arr1.size !== arr2.size) {
	      return false;
	    }
	    for (var i = 0; i < arr1.length; i++) {
	      if (arr1[i] !== arr2[i]) {
	        return false;
	      }
	    }
	    return true;
	  }
	};

	// map of TYPE_*** to GPUVertexFormat
	var gpuVertexFormats = [];
	gpuVertexFormats[TYPE_INT8] = 'sint8';
	gpuVertexFormats[TYPE_UINT8] = 'uint8';
	gpuVertexFormats[TYPE_INT16] = 'sint16';
	gpuVertexFormats[TYPE_UINT16] = 'uint16';
	gpuVertexFormats[TYPE_INT32] = 'sint32';
	gpuVertexFormats[TYPE_UINT32] = 'uint32';
	gpuVertexFormats[TYPE_FLOAT32] = 'float32';
	gpuVertexFormats[TYPE_FLOAT16] = 'float16';
	var gpuVertexFormatsNormalized = [];
	gpuVertexFormatsNormalized[TYPE_INT8] = 'snorm8';
	gpuVertexFormatsNormalized[TYPE_UINT8] = 'unorm8';
	gpuVertexFormatsNormalized[TYPE_INT16] = 'snorm16';
	gpuVertexFormatsNormalized[TYPE_UINT16] = 'unorm16';
	gpuVertexFormatsNormalized[TYPE_INT32] = 'sint32'; // there is no 32bit normalized signed int
	gpuVertexFormatsNormalized[TYPE_UINT32] = 'uint32'; // there is no 32bit normalized unsigned int
	gpuVertexFormatsNormalized[TYPE_FLOAT32] = 'float32'; // there is no 32bit normalized float
	gpuVertexFormatsNormalized[TYPE_FLOAT16] = 'float16'; // there is no 16bit normalized half-float

	/**
	 * @ignore
	 */
	var WebgpuVertexBufferLayout = /*#__PURE__*/function () {
	  function WebgpuVertexBufferLayout() {
	    /**
	     * @type {Map<string, GPUVertexBufferLayout[]>}
	     * @private
	     */
	    this.cache = new Map();
	  }
	  var _proto = WebgpuVertexBufferLayout.prototype;
	  /**
	   * Obtain a vertex layout of one or two vertex formats.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('../vertex-format.js').VertexFormat} vertexFormat0 - The first vertex format.
	   * @param {new Function("modulePath", "return import(modulePath)")('../vertex-format.js').VertexFormat} [vertexFormat1] - The second vertex format.
	   * @returns {any[]} - The vertex layout.
	   */
	  _proto.get = function get(vertexFormat0, vertexFormat1) {
	    if (vertexFormat1 === void 0) {
	      vertexFormat1 = null;
	    }
	    var key = this.getKey(vertexFormat0, vertexFormat1);
	    var layout = this.cache.get(key);
	    if (!layout) {
	      layout = this.create(vertexFormat0, vertexFormat1);
	      this.cache.set(key, layout);
	    }
	    return layout;
	  };
	  _proto.getKey = function getKey(vertexFormat0, vertexFormat1) {
	    var _vertexFormat;
	    if (vertexFormat1 === void 0) {
	      vertexFormat1 = null;
	    }
	    return (vertexFormat0 == null ? void 0 : vertexFormat0.renderingHashString) + "-" + ((_vertexFormat = vertexFormat1) == null ? void 0 : _vertexFormat.renderingHashString);
	  }

	  /**
	   * @param {new Function("modulePath", "return import(modulePath)")('../vertex-format.js').VertexFormat} vertexFormat0 - The first vertex format.
	   * @param {new Function("modulePath", "return import(modulePath)")('../vertex-format.js').VertexFormat} vertexFormat1 - The second vertex format.
	   * @returns {any[]} - The vertex buffer layout.
	   */;
	  _proto.create = function create(vertexFormat0, vertexFormat1) {
	    // type  {GPUVertexBufferLayout[]}
	    var layout = [];
	    var addFormat = function addFormat(format) {
	      var interleaved = format.interleaved;
	      var stepMode = format.instancing ? 'instance' : 'vertex';
	      var attributes = [];
	      var elementCount = format.elements.length;
	      for (var i = 0; i < elementCount; i++) {
	        var element = format.elements[i];
	        var location = semanticToLocation[element.name];
	        var formatTable = element.normalize ? gpuVertexFormatsNormalized : gpuVertexFormats;
	        attributes.push({
	          shaderLocation: location,
	          offset: interleaved ? element.offset : 0,
	          format: "" + formatTable[element.dataType] + (element.numComponents > 1 ? 'x' + element.numComponents : '')
	        });
	        if (!interleaved || i === elementCount - 1) {
	          layout.push({
	            attributes: attributes,
	            arrayStride: element.stride,
	            stepMode: stepMode
	          });
	          attributes = [];
	        }
	      }
	    };
	    if (vertexFormat0) addFormat(vertexFormat0);
	    if (vertexFormat1) addFormat(vertexFormat1);
	    return layout;
	  };
	  return WebgpuVertexBufferLayout;
	}();

	var _layoutId = 0;

	/**
	 * Base class for render and compute pipelines.
	 *
	 * @ignore
	 */
	var WebgpuPipeline = /*#__PURE__*/function () {
	  function WebgpuPipeline(device) {
	    /** @type {new Function("modulePath", "return import(modulePath)")('./webgpu-graphics-device.js').WebgpuGraphicsDevice} */
	    this.device = device;
	  }

	  // TODO: this could be cached using bindGroupKey

	  /**
	   * @param {new Function("modulePath", "return import(modulePath)")('../bind-group-format.js').BindGroupFormat[]} bindGroupFormats - An array
	   * of bind group formats.
	   * @returns {any} Returns the pipeline layout.
	   */
	  var _proto = WebgpuPipeline.prototype;
	  _proto.getPipelineLayout = function getPipelineLayout(bindGroupFormats) {
	    var bindGroupLayouts = [];
	    bindGroupFormats.forEach(function (format) {
	      bindGroupLayouts.push(format.bindGroupLayout);
	    });
	    var descr = {
	      bindGroupLayouts: bindGroupLayouts
	    };
	    _layoutId++;
	    DebugHelper.setLabel(descr, "PipelineLayoutDescr-" + _layoutId);

	    /** @type {GPUPipelineLayout} */
	    var pipelineLayout = this.device.wgpu.createPipelineLayout(descr);
	    DebugHelper.setLabel(pipelineLayout, "PipelineLayout-" + _layoutId);
	    Debug.trace(TRACEID_PIPELINELAYOUT_ALLOC, "Alloc: Id " + _layoutId, {
	      descr: descr,
	      bindGroupFormats: bindGroupFormats
	    });
	    return pipelineLayout;
	  };
	  return WebgpuPipeline;
	}();

	var _pipelineId$1 = 0;
	var _primitiveTopology = ['point-list',
	// PRIMITIVE_POINTS
	'line-list',
	// PRIMITIVE_LINES
	undefined,
	// PRIMITIVE_LINELOOP
	'line-strip',
	// PRIMITIVE_LINESTRIP
	'triangle-list',
	// PRIMITIVE_TRIANGLES
	'triangle-strip',
	// PRIMITIVE_TRISTRIP
	undefined // PRIMITIVE_TRIFAN
	];

	var _blendOperation = ['add',
	// BLENDEQUATION_ADD
	'subtract',
	// BLENDEQUATION_SUBTRACT
	'reverse-subtract',
	// BLENDEQUATION_REVERSE_SUBTRACT
	'min',
	// BLENDEQUATION_MIN
	'max' // BLENDEQUATION_MAX
	];

	var _blendFactor = ['zero',
	// BLENDMODE_ZERO
	'one',
	// BLENDMODE_ONE
	'src',
	// BLENDMODE_SRC_COLOR
	'one-minus-src',
	// BLENDMODE_ONE_MINUS_SRC_COLOR
	'dst',
	// BLENDMODE_DST_COLOR
	'one-minus-dst',
	// BLENDMODE_ONE_MINUS_DST_COLOR
	'src-alpha',
	// BLENDMODE_SRC_ALPHA
	'src-alpha-saturated',
	// BLENDMODE_SRC_ALPHA_SATURATE
	'one-minus-src-alpha',
	// BLENDMODE_ONE_MINUS_SRC_ALPHA
	'dst-alpha',
	// BLENDMODE_DST_ALPHA
	'one-minus-dst-alpha',
	// BLENDMODE_ONE_MINUS_DST_ALPHA
	'constant',
	// BLENDMODE_CONSTANT
	'one-minus-constant' // BLENDMODE_ONE_MINUS_CONSTANT
	];

	var _compareFunction = ['never',
	// FUNC_NEVER
	'less',
	// FUNC_LESS
	'equal',
	// FUNC_EQUAL
	'less-equal',
	// FUNC_LESSEQUAL
	'greater',
	// FUNC_GREATER
	'not-equal',
	// FUNC_NOTEQUAL
	'greater-equal',
	// FUNC_GREATEREQUAL
	'always' // FUNC_ALWAYS
	];

	var _cullModes = ['none',
	// CULLFACE_NONE
	'back',
	// CULLFACE_BACK
	'front' // CULLFACE_FRONT
	];

	var _stencilOps = ['keep',
	// STENCILOP_KEEP
	'zero',
	// STENCILOP_ZERO
	'replace',
	// STENCILOP_REPLACE
	'increment-clamp',
	// STENCILOP_INCREMENT
	'increment-wrap',
	// STENCILOP_INCREMENTWRAP
	'decrement-clamp',
	// STENCILOP_DECREMENT
	'decrement-wrap',
	// STENCILOP_DECREMENTWRAP
	'invert' // STENCILOP_INVERT
	];

	/** @ignore */
	var CacheEntry = function CacheEntry() {
	  /**
	   * Render pipeline
	   *
	   * @type {GPURenderPipeline}
	   * @private
	   */
	  this.pipeline = void 0;
	  /**
	   * The full array of hashes used to lookup the pipeline, used in case of hash collision.
	   *
	   * @type {Uint32Array}
	   */
	  this.hashes = void 0;
	};
	/**
	 * @ignore
	 */
	var WebgpuRenderPipeline = /*#__PURE__*/function (_WebgpuPipeline) {
	  _inheritsLoose(WebgpuRenderPipeline, _WebgpuPipeline);
	  function WebgpuRenderPipeline(device) {
	    var _this;
	    _this = _WebgpuPipeline.call(this, device) || this;

	    /**
	     * The cache of vertex buffer layouts
	     *
	     * @type {WebgpuVertexBufferLayout}
	     */
	    _this.lookupHashes = new Uint32Array(13);
	    _this.vertexBufferLayout = new WebgpuVertexBufferLayout();

	    /**
	     * The cache of render pipelines
	     *
	     * @type {Map<number, CacheEntry[]>}
	     */
	    _this.cache = new Map();
	    return _this;
	  }

	  /** @private */
	  var _proto = WebgpuRenderPipeline.prototype;
	  _proto.get = function get(primitive, vertexFormat0, vertexFormat1, shader, renderTarget, bindGroupFormats, blendState, depthState, cullMode, stencilEnabled, stencilFront, stencilBack) {
	    var _vertexFormat0$render, _vertexFormat1$render, _bindGroupFormats$0$k, _bindGroupFormats$, _bindGroupFormats$1$k, _bindGroupFormats$2, _bindGroupFormats$2$k, _bindGroupFormats$3;
	    Debug.assert(bindGroupFormats.length <= 3);

	    // render pipeline unique hash
	    var lookupHashes = this.lookupHashes;
	    lookupHashes[0] = primitive.type;
	    lookupHashes[1] = shader.id;
	    lookupHashes[2] = cullMode;
	    lookupHashes[3] = depthState.key;
	    lookupHashes[4] = blendState.key;
	    lookupHashes[5] = (_vertexFormat0$render = vertexFormat0 == null ? void 0 : vertexFormat0.renderingHash) != null ? _vertexFormat0$render : 0;
	    lookupHashes[6] = (_vertexFormat1$render = vertexFormat1 == null ? void 0 : vertexFormat1.renderingHash) != null ? _vertexFormat1$render : 0;
	    lookupHashes[7] = renderTarget.impl.key;
	    lookupHashes[8] = (_bindGroupFormats$0$k = (_bindGroupFormats$ = bindGroupFormats[0]) == null ? void 0 : _bindGroupFormats$.key) != null ? _bindGroupFormats$0$k : 0;
	    lookupHashes[9] = (_bindGroupFormats$1$k = (_bindGroupFormats$2 = bindGroupFormats[1]) == null ? void 0 : _bindGroupFormats$2.key) != null ? _bindGroupFormats$1$k : 0;
	    lookupHashes[10] = (_bindGroupFormats$2$k = (_bindGroupFormats$3 = bindGroupFormats[2]) == null ? void 0 : _bindGroupFormats$3.key) != null ? _bindGroupFormats$2$k : 0;
	    lookupHashes[11] = stencilEnabled ? stencilFront.key : 0;
	    lookupHashes[12] = stencilEnabled ? stencilBack.key : 0;
	    var hash = hash32Fnv1a(lookupHashes);

	    // cached pipeline
	    var cacheEntries = this.cache.get(hash);

	    // if we have cache entries, find the exact match, as hash collision can occur
	    if (cacheEntries) {
	      for (var i = 0; i < cacheEntries.length; i++) {
	        var entry = cacheEntries[i];
	        if (array.equals(entry.hashes, lookupHashes)) {
	          return entry.pipeline;
	        }
	      }
	    }

	    // no match or a hash collision, so create a new pipeline
	    var primitiveTopology = _primitiveTopology[primitive.type];
	    Debug.assert(primitiveTopology, "Unsupported primitive topology", primitive);

	    // pipeline layout
	    var pipelineLayout = this.getPipelineLayout(bindGroupFormats);

	    // vertex buffer layout
	    var vertexBufferLayout = this.vertexBufferLayout.get(vertexFormat0, vertexFormat1);

	    // pipeline
	    var cacheEntry = new CacheEntry();
	    cacheEntry.hashes = new Uint32Array(lookupHashes);
	    cacheEntry.pipeline = this.create(primitiveTopology, shader, renderTarget, pipelineLayout, blendState, depthState, vertexBufferLayout, cullMode, stencilEnabled, stencilFront, stencilBack);

	    // add to cache
	    if (cacheEntries) {
	      cacheEntries.push(cacheEntry);
	    } else {
	      cacheEntries = [cacheEntry];
	    }
	    this.cache.set(hash, cacheEntries);
	    return cacheEntry.pipeline;
	  };
	  _proto.getBlend = function getBlend(blendState) {
	    // blend needs to be undefined when blending is disabled
	    var blend;
	    if (blendState.blend) {
	      /** @type {GPUBlendState} */
	      blend = {
	        color: {
	          operation: _blendOperation[blendState.colorOp],
	          srcFactor: _blendFactor[blendState.colorSrcFactor],
	          dstFactor: _blendFactor[blendState.colorDstFactor]
	        },
	        alpha: {
	          operation: _blendOperation[blendState.alphaOp],
	          srcFactor: _blendFactor[blendState.alphaSrcFactor],
	          dstFactor: _blendFactor[blendState.alphaDstFactor]
	        }
	      };

	      // unsupported blend factors
	      Debug.assert(blend.color.srcFactor !== undefined);
	      Debug.assert(blend.color.dstFactor !== undefined);
	      Debug.assert(blend.alpha.srcFactor !== undefined);
	      Debug.assert(blend.alpha.dstFactor !== undefined);
	    }
	    return blend;
	  }

	  /** @private */;
	  _proto.getDepthStencil = function getDepthStencil(depthState, renderTarget, stencilEnabled, stencilFront, stencilBack) {
	    /** @type {GPUDepthStencilState} */
	    var depthStencil;
	    var depth = renderTarget.depth,
	      stencil = renderTarget.stencil;
	    if (depth || stencil) {
	      // format of depth-stencil attachment
	      depthStencil = {
	        format: renderTarget.impl.depthFormat
	      };

	      // depth
	      if (depth) {
	        depthStencil.depthWriteEnabled = depthState.write;
	        depthStencil.depthCompare = _compareFunction[depthState.func];
	        depthStencil.depthBias = depthState.depthBias;
	        depthStencil.depthBiasSlopeScale = depthState.depthBiasSlope;
	      } else {
	        // if render target does not have depth buffer
	        depthStencil.depthWriteEnabled = false;
	        depthStencil.depthCompare = 'always';
	      }

	      // stencil
	      if (stencil && stencilEnabled) {
	        // Note that WebGPU only supports a single mask, we use the one from front, but not from back.
	        depthStencil.stencilReadMas = stencilFront.readMask;
	        depthStencil.stencilWriteMask = stencilFront.writeMask;
	        depthStencil.stencilFront = {
	          compare: _compareFunction[stencilFront.func],
	          failOp: _stencilOps[stencilFront.fail],
	          passOp: _stencilOps[stencilFront.zpass],
	          depthFailOp: _stencilOps[stencilFront.zfail]
	        };
	        depthStencil.stencilBack = {
	          compare: _compareFunction[stencilBack.func],
	          failOp: _stencilOps[stencilBack.fail],
	          passOp: _stencilOps[stencilBack.zpass],
	          depthFailOp: _stencilOps[stencilBack.zfail]
	        };
	      }
	    }
	    return depthStencil;
	  };
	  _proto.create = function create(primitiveTopology, shader, renderTarget, pipelineLayout, blendState, depthState, vertexBufferLayout, cullMode, stencilEnabled, stencilFront, stencilBack) {
	    var wgpu = this.device.wgpu;

	    /** @type {new Function("modulePath", "return import(modulePath)")('./webgpu-shader.js').WebgpuShader} */
	    var webgpuShader = shader.impl;

	    /** @type {GPURenderPipelineDescriptor} */
	    var descr = {
	      vertex: {
	        module: webgpuShader.getVertexShaderModule(),
	        entryPoint: webgpuShader.vertexEntryPoint,
	        buffers: vertexBufferLayout
	      },
	      primitive: {
	        topology: primitiveTopology,
	        frontFace: 'ccw',
	        cullMode: _cullModes[cullMode]
	      },
	      depthStencil: this.getDepthStencil(depthState, renderTarget, stencilEnabled, stencilFront, stencilBack),
	      multisample: {
	        count: renderTarget.samples
	      },
	      // uniform / texture binding layout
	      layout: pipelineLayout
	    };
	    descr.fragment = {
	      module: webgpuShader.getFragmentShaderModule(),
	      entryPoint: webgpuShader.fragmentEntryPoint,
	      targets: []
	    };
	    var colorAttachments = renderTarget.impl.colorAttachments;
	    if (colorAttachments.length > 0) {
	      // the same write mask is used by all color buffers, to match the WebGL behavior
	      var writeMask = 0;
	      if (blendState.redWrite) writeMask |= GPUColorWrite.RED;
	      if (blendState.greenWrite) writeMask |= GPUColorWrite.GREEN;
	      if (blendState.blueWrite) writeMask |= GPUColorWrite.BLUE;
	      if (blendState.alphaWrite) writeMask |= GPUColorWrite.ALPHA;

	      // the same blend state is used by all color buffers, to match the WebGL behavior
	      var blend = this.getBlend(blendState);
	      colorAttachments.forEach(function (attachment) {
	        descr.fragment.targets.push({
	          format: attachment.format,
	          writeMask: writeMask,
	          blend: blend
	        });
	      });
	    }
	    WebgpuDebug.validate(this.device);
	    _pipelineId$1++;
	    DebugHelper.setLabel(descr, "RenderPipelineDescr-" + _pipelineId$1);
	    var pipeline = wgpu.createRenderPipeline(descr);
	    DebugHelper.setLabel(pipeline, "RenderPipeline-" + _pipelineId$1);
	    Debug.trace(TRACEID_RENDERPIPELINE_ALLOC, "Alloc: Id " + _pipelineId$1 + ", stack: " + DebugGraphics.toString(), descr);
	    WebgpuDebug.end(this.device, {
	      renderPipeline: this,
	      descr: descr,
	      shader: shader
	    });
	    return pipeline;
	  };
	  return WebgpuRenderPipeline;
	}(WebgpuPipeline);

	var _pipelineId = 0;

	/**
	 * @ignore
	 */
	var WebgpuComputePipeline = /*#__PURE__*/function (_WebgpuPipeline) {
	  _inheritsLoose(WebgpuComputePipeline, _WebgpuPipeline);
	  function WebgpuComputePipeline() {
	    return _WebgpuPipeline.apply(this, arguments) || this;
	  }
	  var _proto = WebgpuComputePipeline.prototype;
	  /** @private */
	  _proto.get = function get(shader, bindGroupFormat) {
	    // pipeline layout
	    var pipelineLayout = this.getPipelineLayout([bindGroupFormat.impl]);

	    // TODO: this could be cached

	    var pipeline = this.create(shader, pipelineLayout);
	    return pipeline;
	  };
	  _proto.create = function create(shader, pipelineLayout) {
	    var wgpu = this.device.wgpu;

	    /** @type {new Function("modulePath", "return import(modulePath)")('./webgpu-shader.js').WebgpuShader} */
	    var webgpuShader = shader.impl;

	    /** @type {GPUComputePipelineDescriptor} */
	    var descr = {
	      compute: {
	        module: webgpuShader.getComputeShaderModule(),
	        entryPoint: webgpuShader.computeEntryPoint
	      },
	      // uniform / texture binding layout
	      layout: pipelineLayout
	    };
	    WebgpuDebug.validate(this.device);
	    _pipelineId++;
	    DebugHelper.setLabel(descr, "ComputePipelineDescr-" + _pipelineId);
	    var pipeline = wgpu.createComputePipeline(descr);
	    DebugHelper.setLabel(pipeline, "ComputePipeline-" + _pipelineId);
	    Debug.trace(TRACEID_COMPUTEPIPELINE_ALLOC, "Alloc: Id " + _pipelineId, descr);
	    WebgpuDebug.end(this.device, {
	      computePipeline: this,
	      descr: descr,
	      shader: shader
	    });
	    return pipeline;
	  };
	  return WebgpuComputePipeline;
	}(WebgpuPipeline);

	var stringIds = new StringIds();

	/**
	 * Private class storing info about color buffer.
	 *
	 * @ignore
	 */
	var ColorAttachment = /*#__PURE__*/function () {
	  function ColorAttachment() {
	    /**
	     * @type {GPUTextureFormat}
	     * @private
	     */
	    this.format = void 0;
	    /**
	     * @type {GPUTexture}
	     * @private
	     */
	    this.multisampledBuffer = void 0;
	  }
	  var _proto = ColorAttachment.prototype;
	  _proto.destroy = function destroy() {
	    var _this$multisampledBuf;
	    (_this$multisampledBuf = this.multisampledBuffer) == null || _this$multisampledBuf.destroy();
	    this.multisampledBuffer = null;
	  };
	  return ColorAttachment;
	}();
	/**
	 * A WebGPU implementation of the RenderTarget.
	 *
	 * @ignore
	 */
	var WebgpuRenderTarget = /*#__PURE__*/function () {
	  /**
	   * @param {new Function("modulePath", "return import(modulePath)")('../render-target.js').RenderTarget} renderTarget - The render target owning
	   * this implementation.
	   */
	  function WebgpuRenderTarget(renderTarget) {
	    var _this = this;
	    /** @type {boolean} */
	    this.initialized = false;
	    /**
	     * Unique key used by render pipeline creation
	     *
	     * @type {number}
	     */
	    this.key = void 0;
	    /** @type {ColorAttachment[]} */
	    this.colorAttachments = [];
	    /**
	     * @type {GPUTextureFormat}
	     * @private
	     */
	    this.depthFormat = void 0;
	    /** @type {boolean} */
	    this.hasStencil = void 0;
	    /**
	     * @type {GPUTexture}
	     * @private
	     */
	    this.depthTexture = null;
	    /**
	     * True if the depthTexture is internally allocated / owned
	     *
	     * @type {boolean}
	     */
	    this.depthTextureInternal = false;
	    /**
	     * Texture assigned each frame, and not owned by this render target. This is used on the
	     * framebuffer to assign per frame texture obtained from the context.
	     *
	     * @type {GPUTexture}
	     * @private
	     */
	    this.assignedColorTexture = null;
	    /**
	     * Render pass descriptor used when starting a render pass for this render target.
	     *
	     * @type {GPURenderPassDescriptor}
	     * @private
	     */
	    this.renderPassDescriptor = {};
	    this.renderTarget = renderTarget;

	    // color formats are based on the textures
	    if (renderTarget._colorBuffers) {
	      renderTarget._colorBuffers.forEach(function (colorBuffer, index) {
	        _this.setColorAttachment(index, undefined, colorBuffer.impl.format);
	      });
	    }
	    this.updateKey();
	  }

	  /**
	   * Release associated resources. Note that this needs to leave this instance in a state where
	   * it can be re-initialized again, which is used by render target resizing.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('../webgpu/webgpu-graphics-device.js').WebgpuGraphicsDevice} device - The
	   * graphics device.
	   */
	  var _proto2 = WebgpuRenderTarget.prototype;
	  _proto2.destroy = function destroy(device) {
	    this.initialized = false;
	    if (this.depthTextureInternal) {
	      var _this$depthTexture;
	      (_this$depthTexture = this.depthTexture) == null || _this$depthTexture.destroy();
	      this.depthTexture = null;
	    }
	    this.assignedColorTexture = null;
	    this.colorAttachments.forEach(function (colorAttachment) {
	      colorAttachment.destroy();
	    });
	    this.colorAttachments.length = 0;
	  };
	  _proto2.updateKey = function updateKey() {
	    var rt = this.renderTarget;

	    // key used by render pipeline creation
	    var key = rt.samples + ":" + (rt.depth ? this.depthFormat : 'nodepth');
	    this.colorAttachments.forEach(function (colorAttachment) {
	      key += ":" + colorAttachment.format;
	    });

	    // convert string to a unique number
	    this.key = stringIds.get(key);
	  };
	  _proto2.setDepthFormat = function setDepthFormat(depthFormat) {
	    Debug.assert(depthFormat);
	    this.depthFormat = depthFormat;
	    this.hasStencil = depthFormat === 'depth24plus-stencil8';
	  }

	  /**
	   * Assign a color buffer. This allows the color buffer of the main framebuffer
	   * to be swapped each frame to a buffer provided by the context.
	   *
	   * @param {any} gpuTexture - The color buffer.
	   */;
	  _proto2.assignColorTexture = function assignColorTexture(gpuTexture) {
	    Debug.assert(gpuTexture);
	    this.assignedColorTexture = gpuTexture;
	    var view = gpuTexture.createView();
	    DebugHelper.setLabel(view, 'Framebuffer.assignedColor');

	    // use it as render buffer or resolve target
	    var colorAttachment = this.renderPassDescriptor.colorAttachments[0];
	    var samples = this.renderTarget.samples;
	    if (samples > 1) {
	      colorAttachment.resolveTarget = view;
	    } else {
	      colorAttachment.view = view;
	    }

	    // for main framebuffer, this is how the format is obtained
	    this.setColorAttachment(0, undefined, gpuTexture.format);
	    this.updateKey();
	  };
	  _proto2.setColorAttachment = function setColorAttachment(index, multisampledBuffer, format) {
	    if (!this.colorAttachments[index]) {
	      this.colorAttachments[index] = new ColorAttachment();
	    }
	    if (multisampledBuffer) {
	      this.colorAttachments[index].multisampledBuffer = multisampledBuffer;
	    }
	    if (format) {
	      this.colorAttachments[index].format = format;
	    }
	  }

	  /**
	   * Initialize render target for rendering one time.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('../webgpu/webgpu-graphics-device.js').WebgpuGraphicsDevice} device - The
	   * graphics device.
	   * @param {new Function("modulePath", "return import(modulePath)")('../render-target.js').RenderTarget} renderTarget - The render target.
	   */;
	  _proto2.init = function init(device, renderTarget) {
	    var _renderTarget$_colorB, _renderTarget$_colorB2;
	    var wgpu = device.wgpu;
	    Debug.assert(!this.initialized);
	    WebgpuDebug.memory(device);
	    WebgpuDebug.validate(device);

	    // initialize depth/stencil
	    this.initDepthStencil(wgpu, renderTarget);

	    // initialize color attachments
	    this.renderPassDescriptor.colorAttachments = [];
	    var count = (_renderTarget$_colorB = (_renderTarget$_colorB2 = renderTarget._colorBuffers) == null ? void 0 : _renderTarget$_colorB2.length) != null ? _renderTarget$_colorB : 1;
	    for (var i = 0; i < count; ++i) {
	      var _this$colorAttachment;
	      var colorAttachment = this.initColor(wgpu, renderTarget, i);

	      // default framebuffer, buffer gets assigned later
	      var isDefaultFramebuffer = i === 0 && ((_this$colorAttachment = this.colorAttachments[0]) == null ? void 0 : _this$colorAttachment.format);

	      // if we have a color buffer, or is the default framebuffer
	      if (colorAttachment.view || isDefaultFramebuffer) {
	        this.renderPassDescriptor.colorAttachments.push(colorAttachment);
	      }
	    }
	    this.initialized = true;
	    WebgpuDebug.end(device, {
	      renderTarget: renderTarget
	    });
	    WebgpuDebug.end(device, {
	      renderTarget: renderTarget
	    });
	  };
	  _proto2.initDepthStencil = function initDepthStencil(wgpu, renderTarget) {
	    var samples = renderTarget.samples,
	      width = renderTarget.width,
	      height = renderTarget.height,
	      depth = renderTarget.depth,
	      depthBuffer = renderTarget.depthBuffer;

	    // depth buffer that we render to (single or multi-sampled). We don't create resolve
	    // depth buffer as we don't currently resolve it. This might need to change in the future.
	    if (depth || depthBuffer) {
	      // allocate depth buffer if not provided
	      if (!depthBuffer) {
	        // TODO: support rendering to 32bit depth without a stencil as well
	        this.setDepthFormat('depth24plus-stencil8');

	        /** @type {GPUTextureDescriptor} */
	        var depthTextureDesc = {
	          size: [width, height, 1],
	          dimension: '2d',
	          sampleCount: samples,
	          format: this.depthFormat,
	          usage: GPUTextureUsage.RENDER_ATTACHMENT
	        };
	        if (samples > 1) {
	          // enable multi-sampled depth texture to be a source of our shader based resolver in WebgpuResolver
	          // TODO: we do not always need to resolve it, and so might consider this flag to be optional
	          depthTextureDesc.usage |= GPUTextureUsage.TEXTURE_BINDING;
	        } else {
	          // single sampled depth buffer can be copied out (grab pass)
	          // TODO: we should not enable this for shadow maps, as it is not needed
	          depthTextureDesc.usage |= GPUTextureUsage.COPY_SRC;
	        }

	        // allocate depth buffer
	        this.depthTexture = wgpu.createTexture(depthTextureDesc);
	        this.depthTextureInternal = true;
	      } else {
	        // use provided depth buffer
	        this.depthTexture = depthBuffer.impl.gpuTexture;
	        this.setDepthFormat(depthBuffer.impl.format);
	      }
	      Debug.assert(this.depthTexture);
	      DebugHelper.setLabel(this.depthTexture, renderTarget.name + ".depthTexture");

	      // @type {GPURenderPassDepthStencilAttachment}
	      this.renderPassDescriptor.depthStencilAttachment = {
	        view: this.depthTexture.createView()
	      };
	    }
	  }

	  /**
	   * @private
	   */;
	  _proto2.initColor = function initColor(wgpu, renderTarget, index) {
	    // Single-sampled color buffer gets passed in:
	    // - for normal render target, constructor takes the color buffer as an option
	    // - for the main framebuffer, the device supplies the buffer each frame
	    // And so we only need to create multi-sampled color buffer if needed here.
	    /** @type {GPURenderPassColorAttachment} */
	    var colorAttachment = {};
	    var samples = renderTarget.samples,
	      width = renderTarget.width,
	      height = renderTarget.height;
	    var colorBuffer = renderTarget.getColorBuffer(index);

	    // view used to write to the color buffer (either by rendering to it, or resolving to it)
	    var colorView = null;
	    if (colorBuffer) {
	      // render to top mip level in case of mip-mapped buffer
	      var mipLevelCount = 1;

	      // cubemap face view - face is a single 2d array layer in order [+X, -X, +Y, -Y, +Z, -Z]
	      if (colorBuffer.cubemap) {
	        colorView = colorBuffer.impl.createView({
	          dimension: '2d',
	          baseArrayLayer: renderTarget.face,
	          arrayLayerCount: 1,
	          mipLevelCount: mipLevelCount
	        });
	      } else {
	        colorView = colorBuffer.impl.createView({
	          mipLevelCount: mipLevelCount
	        });
	      }
	    }

	    // multi-sampled color buffer
	    if (samples > 1) {
	      var _this$colorAttachment2, _this$colorAttachment3;
	      /** @type {GPUTextureDescriptor} */
	      var multisampledTextureDesc = {
	        size: [width, height, 1],
	        dimension: '2d',
	        sampleCount: samples,
	        format: (_this$colorAttachment2 = (_this$colorAttachment3 = this.colorAttachments[index]) == null ? void 0 : _this$colorAttachment3.format) != null ? _this$colorAttachment2 : colorBuffer.impl.format,
	        usage: GPUTextureUsage.RENDER_ATTACHMENT
	      };

	      // allocate multi-sampled color buffer
	      var multisampledColorBuffer = wgpu.createTexture(multisampledTextureDesc);
	      DebugHelper.setLabel(multisampledColorBuffer, renderTarget.name + ".multisampledColor");
	      this.setColorAttachment(index, multisampledColorBuffer, multisampledTextureDesc.format);
	      colorAttachment.view = multisampledColorBuffer.createView();
	      DebugHelper.setLabel(colorAttachment.view, renderTarget.name + ".multisampledColorView");
	      colorAttachment.resolveTarget = colorView;
	    } else {
	      colorAttachment.view = colorView;
	    }
	    return colorAttachment;
	  }

	  /**
	   * Update WebGPU render pass descriptor by RenderPass settings.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('../render-pass.js').RenderPass} renderPass - The render pass to start.
	   */;
	  _proto2.setupForRenderPass = function setupForRenderPass(renderPass) {
	    var _this$renderPassDescr, _this$renderPassDescr2;
	    Debug.assert(this.renderPassDescriptor);
	    var count = (_this$renderPassDescr = (_this$renderPassDescr2 = this.renderPassDescriptor.colorAttachments) == null ? void 0 : _this$renderPassDescr2.length) != null ? _this$renderPassDescr : 0;
	    for (var i = 0; i < count; ++i) {
	      var colorAttachment = this.renderPassDescriptor.colorAttachments[i];
	      var colorOps = renderPass.colorArrayOps[i];
	      colorAttachment.clearValue = colorOps.clearValue;
	      colorAttachment.loadOp = colorOps.clear ? 'clear' : 'load';
	      colorAttachment.storeOp = colorOps.store ? 'store' : 'discard';
	    }
	    var depthAttachment = this.renderPassDescriptor.depthStencilAttachment;
	    if (depthAttachment) {
	      depthAttachment.depthClearValue = renderPass.depthStencilOps.clearDepthValue;
	      depthAttachment.depthLoadOp = renderPass.depthStencilOps.clearDepth ? 'clear' : 'load';
	      depthAttachment.depthStoreOp = renderPass.depthStencilOps.storeDepth ? 'store' : 'discard';
	      depthAttachment.depthReadOnly = false;
	      if (this.hasStencil) {
	        depthAttachment.stencilClearValue = renderPass.depthStencilOps.clearStencilValue;
	        depthAttachment.stencilLoadOp = renderPass.depthStencilOps.clearStencil ? 'clear' : 'load';
	        depthAttachment.stencilStoreOp = renderPass.depthStencilOps.storeStencil ? 'store' : 'discard';
	        depthAttachment.stencilReadOnly = false;
	      }
	    }
	  };
	  _proto2.loseContext = function loseContext() {
	    this.initialized = false;
	  };
	  _proto2.resolve = function resolve(device, target, color, depth) {};
	  return WebgpuRenderTarget;
	}();

	// map of UNIFORMTYPE_*** to number of 32bit components
	var uniformTypeToNumComponents = [];
	uniformTypeToNumComponents[UNIFORMTYPE_FLOAT] = 1;
	uniformTypeToNumComponents[UNIFORMTYPE_VEC2] = 2;
	uniformTypeToNumComponents[UNIFORMTYPE_VEC3] = 3;
	uniformTypeToNumComponents[UNIFORMTYPE_VEC4] = 4;
	uniformTypeToNumComponents[UNIFORMTYPE_INT] = 1;
	uniformTypeToNumComponents[UNIFORMTYPE_IVEC2] = 2;
	uniformTypeToNumComponents[UNIFORMTYPE_IVEC3] = 3;
	uniformTypeToNumComponents[UNIFORMTYPE_IVEC4] = 4;
	uniformTypeToNumComponents[UNIFORMTYPE_BOOL] = 1;
	uniformTypeToNumComponents[UNIFORMTYPE_BVEC2] = 2;
	uniformTypeToNumComponents[UNIFORMTYPE_BVEC3] = 3;
	uniformTypeToNumComponents[UNIFORMTYPE_BVEC4] = 4;
	uniformTypeToNumComponents[UNIFORMTYPE_MAT2] = 8; // 2 x vec4
	uniformTypeToNumComponents[UNIFORMTYPE_MAT3] = 12; // 3 x vec4
	uniformTypeToNumComponents[UNIFORMTYPE_MAT4] = 16; // 4 x vec4
	uniformTypeToNumComponents[UNIFORMTYPE_UINT] = 1;
	uniformTypeToNumComponents[UNIFORMTYPE_UVEC2] = 2;
	uniformTypeToNumComponents[UNIFORMTYPE_UVEC3] = 3;
	uniformTypeToNumComponents[UNIFORMTYPE_UVEC4] = 4;

	/**
	 * A class storing description of an individual uniform, stored inside a uniform buffer.
	 *
	 * @ignore
	 */
	var UniformFormat = /*#__PURE__*/function () {
	  function UniformFormat(name, type, count) {
	    var _this = this;
	    if (count === void 0) {
	      count = 0;
	    }
	    /** @type {string} */
	    this.name = void 0;
	    // UNIFORMTYPE_***
	    /** @type {number} */
	    this.type = void 0;
	    /** @type {number} */
	    this.byteSize = void 0;
	    /**
	     * Index of the uniform in an array of 32bit values (Float32Array and similar)
	     *
	     * @type {number}
	     */
	    this.offset = void 0;
	    /** @type {new Function("modulePath", "return import(modulePath)")('./scope-id.js').ScopeId} */
	    this.scopeId = void 0;
	    /**
	     * Count of elements for arrays, otherwise 0.
	     *
	     * @type {number}
	     */
	    this.count = void 0;
	    /**
	     * Number of components in each element (e.g. vec2 has 2 components, mat4 has 16 components)
	     *
	     * @type {number}
	     */
	    this.numComponents = void 0;
	    // just a name
	    this.shortName = name;

	    // name with [0] if this is an array
	    this.name = count ? name + "[0]" : name;
	    this.type = type;
	    this.numComponents = uniformTypeToNumComponents[type];
	    Debug.assert(this.numComponents, "Unhandled uniform format " + type + " used for " + name);
	    this.updateType = type;
	    if (count > 0) {
	      switch (type) {
	        case UNIFORMTYPE_FLOAT:
	          this.updateType = UNIFORMTYPE_FLOATARRAY;
	          break;
	        case UNIFORMTYPE_INT:
	          this.updateType = UNIFORMTYPE_INTARRAY;
	          break;
	        case UNIFORMTYPE_UINT:
	          this.updateType = UNIFORMTYPE_UINTARRAY;
	          break;
	        case UNIFORMTYPE_BOOL:
	          this.updateType = UNIFORMTYPE_BOOLARRAY;
	          break;
	        case UNIFORMTYPE_VEC2:
	          this.updateType = UNIFORMTYPE_VEC2ARRAY;
	          break;
	        case UNIFORMTYPE_IVEC2:
	          this.updateType = UNIFORMTYPE_IVEC2ARRAY;
	          break;
	        case UNIFORMTYPE_UVEC2:
	          this.updateType = UNIFORMTYPE_UVEC2ARRAY;
	          break;
	        case UNIFORMTYPE_BVEC2:
	          this.updateType = UNIFORMTYPE_BVEC2ARRAY;
	          break;
	        case UNIFORMTYPE_VEC3:
	          this.updateType = UNIFORMTYPE_VEC3ARRAY;
	          break;
	        case UNIFORMTYPE_IVEC3:
	          this.updateType = UNIFORMTYPE_IVEC3ARRAY;
	          break;
	        case UNIFORMTYPE_UVEC3:
	          this.updateType = UNIFORMTYPE_UVEC3ARRAY;
	          break;
	        case UNIFORMTYPE_BVEC3:
	          this.updateType = UNIFORMTYPE_BVEC3ARRAY;
	          break;
	        case UNIFORMTYPE_VEC4:
	          this.updateType = UNIFORMTYPE_VEC4ARRAY;
	          break;
	        case UNIFORMTYPE_IVEC4:
	          this.updateType = UNIFORMTYPE_IVEC4ARRAY;
	          break;
	        case UNIFORMTYPE_UVEC4:
	          this.updateType = UNIFORMTYPE_UVEC4ARRAY;
	          break;
	        case UNIFORMTYPE_BVEC4:
	          this.updateType = UNIFORMTYPE_BVEC4ARRAY;
	          break;
	        case UNIFORMTYPE_MAT4:
	          this.updateType = UNIFORMTYPE_MAT4ARRAY;
	          break;
	        default:
	          Debug.error("Uniform array of type " + uniformTypeToName[type] + " is not supported when processing uniform '" + name + "'.");
	          Debug.call(function () {
	            _this.invalid = true;
	          });
	          break;
	      }
	    }
	    this.count = count;
	    Debug.assert(!isNaN(count), "Unsupported uniform: " + name + "[" + count + "]");
	    Debug.call(function () {
	      if (isNaN(count)) _this.invalid = true;
	    });
	    var componentSize = this.numComponents;

	    // component size for arrays is aligned up to vec4
	    if (count) {
	      componentSize = math.roundUp(componentSize, 4);
	    }
	    this.byteSize = componentSize * 4;
	    if (count) this.byteSize *= count;
	    Debug.assert(this.byteSize, "Unknown byte size for uniform format " + type + " used for " + name);
	  }

	  // std140 rules: https://registry.khronos.org/OpenGL/specs/gl/glspec45.core.pdf#page=159
	  // TODO: this support limited subset of functionality, arrays and structs are not supported.
	  var _proto = UniformFormat.prototype;
	  _proto.calculateOffset = function calculateOffset(offset) {
	    // Note: vec3 has the same alignment as vec4
	    var alignment = this.byteSize <= 8 ? this.byteSize : 16;

	    // arrays have vec4 alignments
	    if (this.count) alignment = 16;

	    // align the start offset
	    offset = math.roundUp(offset, alignment);
	    this.offset = offset / 4;
	  };
	  _createClass(UniformFormat, [{
	    key: "isArrayType",
	    get:
	    /**
	     * True if this is an array of elements (i.e. count > 0)
	     *
	     * @type {number}
	     */
	    function get() {
	      return this.count > 0;
	    }
	  }]);
	  return UniformFormat;
	}();
	/**
	 * A descriptor that defines the layout of of data inside the {@link UniformBuffer}.
	 *
	 * @ignore
	 */
	var UniformBufferFormat = /*#__PURE__*/function () {
	  /**
	   * Create a new UniformBufferFormat instance.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./graphics-device.js').GraphicsDevice} graphicsDevice - The graphics device.
	   * @param {UniformFormat[]} uniforms - An array of uniforms to be stored in the buffer
	   */
	  function UniformBufferFormat(graphicsDevice, uniforms) {
	    /** @type {number} */
	    this.byteSize = 0;
	    /** @type {Map<string,UniformFormat>} */
	    this.map = new Map();
	    this.scope = graphicsDevice.scope;

	    /** @type {UniformFormat[]} */
	    this.uniforms = uniforms;

	    // TODO: optimize uniforms ordering

	    var offset = 0;
	    for (var i = 0; i < uniforms.length; i++) {
	      var uniform = uniforms[i];
	      uniform.calculateOffset(offset);
	      offset = uniform.offset * 4 + uniform.byteSize;
	      uniform.scopeId = this.scope.resolve(uniform.name);
	      this.map.set(uniform.name, uniform);
	    }

	    // round up buffer size
	    this.byteSize = math.roundUp(offset, 16);
	  }

	  /**
	   * Returns format of a uniform with specified name.
	   *
	   * @param {string} name - The name of the uniform.
	   * @returns {UniformFormat} - The format of the uniform.
	   */
	  var _proto2 = UniformBufferFormat.prototype;
	  _proto2.get = function get(name) {
	    return this.map.get(name);
	  };
	  _proto2.getShaderDeclaration = function getShaderDeclaration(bindGroup, bindIndex) {
	    var name = bindGroupNames[bindGroup];
	    var code = "layout(set = " + bindGroup + ", binding = " + bindIndex + ", std140) uniform ub_" + name + " {\n";
	    this.uniforms.forEach(function (uniform) {
	      var typeString = uniformTypeToName[uniform.type];
	      Debug.assert(typeString.length > 0, "Uniform type " + uniform.type + " is not handled.");
	      code += "    " + typeString + " " + uniform.shortName + (uniform.count ? "[" + uniform.count + "]" : '') + ";\n";
	    });
	    return code + '};\n';
	  };
	  return UniformBufferFormat;
	}();

	var _textureDimensionInfo;
	var id$8 = 0;
	var textureDimensionInfo = (_textureDimensionInfo = {}, _textureDimensionInfo[TEXTUREDIMENSION_2D] = 'texture2D', _textureDimensionInfo[TEXTUREDIMENSION_CUBE] = 'textureCube', _textureDimensionInfo[TEXTUREDIMENSION_3D] = 'texture3D', _textureDimensionInfo[TEXTUREDIMENSION_2D_ARRAY] = 'texture2DArray', _textureDimensionInfo);

	/**
	 * @ignore
	 */
	var BindBufferFormat = function BindBufferFormat(name, visibility) {
	  /** @type {string} */
	  this.name = name;

	  // SHADERSTAGE_VERTEX, SHADERSTAGE_FRAGMENT, SHADERSTAGE_COMPUTE
	  this.visibility = visibility;
	};
	/**
	 * @ignore
	 */
	var BindTextureFormat = function BindTextureFormat(name, visibility, textureDimension, sampleType) {
	  if (textureDimension === void 0) {
	    textureDimension = TEXTUREDIMENSION_2D;
	  }
	  if (sampleType === void 0) {
	    sampleType = SAMPLETYPE_FLOAT;
	  }
	  /** @type {new Function("modulePath", "return import(modulePath)")('./scope-id.js').ScopeId} */
	  this.scopeId = void 0;
	  /** @type {string} */
	  this.name = name;

	  // SHADERSTAGE_VERTEX, SHADERSTAGE_FRAGMENT, SHADERSTAGE_COMPUTE
	  this.visibility = visibility;

	  // TEXTUREDIMENSION_***
	  this.textureDimension = textureDimension;

	  // SAMPLETYPE_***
	  this.sampleType = sampleType;
	};
	/**
	 * @ignore
	 */
	var BindStorageTextureFormat = function BindStorageTextureFormat(name, format, textureDimension) {
	  if (format === void 0) {
	    format = PIXELFORMAT_RGBA8;
	  }
	  if (textureDimension === void 0) {
	    textureDimension = TEXTUREDIMENSION_2D;
	  }
	  /** @type {new Function("modulePath", "return import(modulePath)")('./scope-id.js').ScopeId} */
	  this.scopeId = void 0;
	  /** @type {string} */
	  this.name = name;

	  // PIXELFORMAT_***
	  this.format = format;

	  // TEXTUREDIMENSION_***
	  this.textureDimension = textureDimension;
	};
	/**
	 * @ignore
	 */
	var BindGroupFormat = /*#__PURE__*/function () {
	  /**
	   * @param {new Function("modulePath", "return import(modulePath)")('./graphics-device.js').GraphicsDevice} graphicsDevice - The graphics device
	   * used to manage this vertex format.
	   * @param {BindBufferFormat[]} [bufferFormats] - An array of bind buffer formats (uniform
	   * buffers). Defaults to an empty array.
	   * @param {BindTextureFormat[]} [textureFormats] - An array of bind texture formats (textures).
	   * Defaults to an empty array.
	   * @param {BindStorageTextureFormat[]} [storageTextureFormats] - An array of bind storage texture
	   * formats (storage textures), used by the compute shader. Defaults to an empty array.
	   * @param {object} [options] - Object for passing optional arguments.
	   * @param {boolean} [options.compute] - If true, this bind group format is used by the compute
	   * shader.
	   */
	  function BindGroupFormat(graphicsDevice, bufferFormats, textureFormats, storageTextureFormats, options) {
	    var _options$compute,
	      _this = this;
	    if (bufferFormats === void 0) {
	      bufferFormats = [];
	    }
	    if (textureFormats === void 0) {
	      textureFormats = [];
	    }
	    if (storageTextureFormats === void 0) {
	      storageTextureFormats = [];
	    }
	    if (options === void 0) {
	      options = {};
	    }
	    this.compute = false;
	    this.id = id$8++;
	    DebugHelper.setName(this, "BindGroupFormat_" + this.id);
	    this.compute = (_options$compute = options.compute) != null ? _options$compute : false;
	    Debug.assert(this.compute || storageTextureFormats.length === 0, "Storage textures can be specified only for compute");

	    /** @type {new Function("modulePath", "return import(modulePath)")('./graphics-device.js').GraphicsDevice} */
	    this.device = graphicsDevice;
	    var scope = graphicsDevice.scope;

	    /** @type {BindBufferFormat[]} */
	    this.bufferFormats = bufferFormats;

	    // maps a buffer format name to an index
	    /** @type {Map<string, number>} */
	    this.bufferFormatsMap = new Map();
	    bufferFormats.forEach(function (bf, i) {
	      return _this.bufferFormatsMap.set(bf.name, i);
	    });

	    /** @type {BindTextureFormat[]} */
	    this.textureFormats = textureFormats;

	    // maps a texture format name to a slot index
	    /** @type {Map<string, number>} */
	    this.textureFormatsMap = new Map();
	    textureFormats.forEach(function (tf, i) {
	      _this.textureFormatsMap.set(tf.name, i);

	      // resolve scope id
	      tf.scopeId = scope.resolve(tf.name);
	    });

	    /** @type {BindStorageTextureFormat[]} */
	    this.storageTextureFormats = storageTextureFormats;

	    // maps a storage texture format name to a slot index
	    /** @type {Map<string, number>} */
	    this.storageTextureFormatsMap = new Map();
	    storageTextureFormats.forEach(function (tf, i) {
	      _this.storageTextureFormatsMap.set(tf.name, i);

	      // resolve scope id
	      tf.scopeId = scope.resolve(tf.name);
	    });
	    this.impl = graphicsDevice.createBindGroupFormatImpl(this);
	    Debug.trace(TRACEID_BINDGROUPFORMAT_ALLOC, "Alloc: Id " + this.id, this);
	  }

	  /**
	   * Frees resources associated with this bind group.
	   */
	  var _proto = BindGroupFormat.prototype;
	  _proto.destroy = function destroy() {
	    this.impl.destroy();
	  }

	  /**
	   * Returns format of texture with specified name.
	   *
	   * @param {string} name - The name of the texture slot.
	   * @returns {BindTextureFormat|null} - The format.
	   */;
	  _proto.getTexture = function getTexture(name) {
	    var index = this.textureFormatsMap.get(name);
	    if (index !== undefined) {
	      return this.textureFormats[index];
	    }
	    return null;
	  }

	  /**
	   * Returns format of storage texture with specified name.
	   *
	   * @param {string} name - The name of the texture slot.
	   * @returns {BindStorageTextureFormat|null} - The format.
	   */;
	  _proto.getStorageTexture = function getStorageTexture(name) {
	    var index = this.storageTextureFormatsMap.get(name);
	    if (index !== undefined) {
	      return this.storageTextureFormats[index];
	    }
	    return null;
	  };
	  _proto.getShaderDeclarationTextures = function getShaderDeclarationTextures(bindGroup) {
	    var code = '';
	    var bindIndex = this.bufferFormats.length;
	    this.textureFormats.forEach(function (format) {
	      var textureType = textureDimensionInfo[format.textureDimension];
	      Debug.assert(textureType, "Unsupported texture type", format.textureDimension);

	      // handle texture2DArray by renaming the texture object and defining a replacement macro
	      var namePostfix = '';
	      var extraCode = '';
	      if (textureType === 'texture2DArray') {
	        namePostfix = '_texture';
	        extraCode = "#define " + format.name + " sampler2DArray(" + format.name + namePostfix + ", " + format.name + "_sampler)\n";
	      }
	      if (format.sampleType === SAMPLETYPE_INT) {
	        textureType = "i" + textureType;
	      } else if (format.sampleType === SAMPLETYPE_UINT) {
	        textureType = "u" + textureType;
	      }
	      code += "layout(set = " + bindGroup + ", binding = " + bindIndex++ + ") uniform " + textureType + " " + format.name + namePostfix + ";\n" + ("layout(set = " + bindGroup + ", binding = " + bindIndex++ + ") uniform sampler " + format.name + "_sampler;\n") + extraCode;
	    });
	    return code;
	  };
	  _proto.loseContext = function loseContext() {
	    // TODO: implement
	  };
	  return BindGroupFormat;
	}();

	// accepted keywords
	// TODO: 'out' keyword is not in the list, as handling it is more complicated due
	// to 'out' keyword also being used to mark output only function parameters.
	var KEYWORD$1 = /[ \t]*(\battribute\b|\bvarying\b|\buniform\b)/g;

	// match 'attribute' and anything else till ';'
	var KEYWORD_LINE = /(\battribute\b|\bvarying\b|\bout\b|\buniform\b)[ \t]*([^;]+)([;]+)/g;

	// marker for a place in the source code to be replaced by code
	var MARKER = '@@@';

	// an array identifier, for example 'data[4]' - group 1 is 'data', group 2 is everything in brackets: '4'
	var ARRAY_IDENTIFIER = /([\w-]+)\[(.*?)\]/;
	var precisionQualifiers = new Set(['highp', 'mediump', 'lowp']);
	var shadowSamplers = new Set(['sampler2DShadow', 'samplerCubeShadow', 'sampler2DArrayShadow']);
	var textureDimensions = {
	  sampler2D: TEXTUREDIMENSION_2D,
	  sampler3D: TEXTUREDIMENSION_3D,
	  samplerCube: TEXTUREDIMENSION_CUBE,
	  samplerCubeShadow: TEXTUREDIMENSION_CUBE,
	  sampler2DShadow: TEXTUREDIMENSION_2D,
	  sampler2DArray: TEXTUREDIMENSION_2D_ARRAY,
	  sampler2DArrayShadow: TEXTUREDIMENSION_2D_ARRAY,
	  isampler2D: TEXTUREDIMENSION_2D,
	  usampler2D: TEXTUREDIMENSION_2D,
	  isampler3D: TEXTUREDIMENSION_3D,
	  usampler3D: TEXTUREDIMENSION_3D,
	  isamplerCube: TEXTUREDIMENSION_CUBE,
	  usamplerCube: TEXTUREDIMENSION_CUBE,
	  isampler2DArray: TEXTUREDIMENSION_2D_ARRAY,
	  usampler2DArray: TEXTUREDIMENSION_2D_ARRAY
	};
	var UniformLine = function UniformLine(line, shader) {
	  // example: `lowp vec4 tints[2 * 4]`
	  this.line = line;

	  // split to words handling any number of spaces
	  var words = line.trim().split(/\s+/);

	  // optional precision
	  if (precisionQualifiers.has(words[0])) {
	    this.precision = words.shift();
	  }

	  // type
	  this.type = words.shift();
	  if (line.includes(',')) {
	    Debug.error("A comma on a uniform line is not supported, split it into multiple uniforms: " + line, shader);
	  }

	  // array of uniforms
	  if (line.includes('[')) {
	    var rest = words.join(' ');
	    var match = ARRAY_IDENTIFIER.exec(rest);
	    Debug.assert(match);
	    this.name = match[1];
	    this.arraySize = Number(match[2]);
	    if (isNaN(this.arraySize)) {
	      shader.failed = true;
	      Debug.error("Only numerically specified uniform array sizes are supported, this uniform is not supported: '" + line + "'", shader);
	    }
	  } else {
	    // simple uniform
	    this.name = words.shift();
	    this.arraySize = 0;
	  }
	  this.isSampler = this.type.indexOf('sampler') !== -1;
	  this.isSignedInt = this.type.indexOf('isampler') !== -1;
	  this.isUnsignedInt = this.type.indexOf('usampler') !== -1;
	};
	/**
	 * Pure static class implementing processing of GLSL shaders. It allocates fixed locations for
	 * attributes, and handles conversion of uniforms to uniform buffers.
	 *
	 * @ignore
	 */
	var ShaderProcessor = /*#__PURE__*/function () {
	  function ShaderProcessor() {}
	  /**
	   * Process the shader.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./graphics-device.js').GraphicsDevice} device - The graphics device.
	   * @param {object} shaderDefinition - The shader definition.
	   * @param {new Function("modulePath", "return import(modulePath)")('./shader.js').Shader} shader - The shader definition.
	   * @returns {object} - The processed shader data.
	   */
	  ShaderProcessor.run = function run(device, shaderDefinition, shader) {
	    /** @type {Map<string, number>} */
	    var varyingMap = new Map();

	    // extract lines of interests from both shaders
	    var vertexExtracted = ShaderProcessor.extract(shaderDefinition.vshader);
	    var fragmentExtracted = ShaderProcessor.extract(shaderDefinition.fshader);

	    // VS - convert a list of attributes to a shader block with fixed locations
	    var attributesBlock = ShaderProcessor.processAttributes(vertexExtracted.attributes, shaderDefinition.attributes, shaderDefinition.processingOptions);

	    // VS - convert a list of varyings to a shader block
	    var vertexVaryingsBlock = ShaderProcessor.processVaryings(vertexExtracted.varyings, varyingMap, true);

	    // FS - convert a list of varyings to a shader block
	    var fragmentVaryingsBlock = ShaderProcessor.processVaryings(fragmentExtracted.varyings, varyingMap, false);

	    // FS - convert a list of outputs to a shader block
	    var outBlock = ShaderProcessor.processOuts(fragmentExtracted.outs);

	    // uniforms - merge vertex and fragment uniforms, and create shared uniform buffers
	    // Note that as both vertex and fragment can declare the same uniform, we need to remove duplicates
	    var concatUniforms = vertexExtracted.uniforms.concat(fragmentExtracted.uniforms);
	    var uniforms = Array.from(new Set(concatUniforms));

	    // parse uniform lines
	    var parsedUniforms = uniforms.map(function (line) {
	      return new UniformLine(line, shader);
	    });

	    // validation - as uniforms go to a shared uniform buffer, vertex and fragment versions need to match
	    Debug.call(function () {
	      var map = new Map();
	      parsedUniforms.forEach(function (uni) {
	        var existing = map.get(uni.name);
	        Debug.assert(!existing, "Vertex and fragment shaders cannot use the same uniform name with different types: '" + existing + "' and '" + uni.line + "'", shader);
	        map.set(uni.name, uni.line);
	      });
	    });
	    var uniformsData = ShaderProcessor.processUniforms(device, parsedUniforms, shaderDefinition.processingOptions, shader);

	    // VS - insert the blocks to the source
	    var vBlock = attributesBlock + '\n' + vertexVaryingsBlock + '\n' + uniformsData.code;
	    var vshader = vertexExtracted.src.replace(MARKER, vBlock);

	    // FS - insert the blocks to the source
	    var fBlock = fragmentVaryingsBlock + '\n' + outBlock + '\n' + uniformsData.code;
	    var fshader = fragmentExtracted.src.replace(MARKER, fBlock);
	    return {
	      vshader: vshader,
	      fshader: fshader,
	      meshUniformBufferFormat: uniformsData.meshUniformBufferFormat,
	      meshBindGroupFormat: uniformsData.meshBindGroupFormat
	    };
	  }

	  // Extract required information from the shader source code.
	  ;
	  ShaderProcessor.extract = function extract(src) {
	    // collected data
	    var attributes = [];
	    var varyings = [];
	    var outs = [];
	    var uniforms = [];

	    // replacement marker - mark a first replacement place, this is where code
	    // blocks are injected later
	    var replacement = MARKER + "\n";

	    // extract relevant parts of the shader
	    var match;
	    while ((match = KEYWORD$1.exec(src)) !== null) {
	      var keyword = match[1];
	      switch (keyword) {
	        case 'attribute':
	        case 'varying':
	        case 'uniform':
	        case 'out':
	          {
	            // read the line
	            KEYWORD_LINE.lastIndex = match.index;
	            var lineMatch = KEYWORD_LINE.exec(src);
	            if (keyword === 'attribute') {
	              attributes.push(lineMatch[2]);
	            } else if (keyword === 'varying') {
	              varyings.push(lineMatch[2]);
	            } else if (keyword === 'out') {
	              outs.push(lineMatch[2]);
	            } else if (keyword === 'uniform') {
	              uniforms.push(lineMatch[2]);
	            }

	            // cut it out
	            src = ShaderProcessor.cutOut(src, match.index, KEYWORD_LINE.lastIndex, replacement);
	            KEYWORD$1.lastIndex = match.index + replacement.length;

	            // only place a single replacement marker
	            replacement = '';
	            break;
	          }
	      }
	    }
	    return {
	      src: src,
	      attributes: attributes,
	      varyings: varyings,
	      outs: outs,
	      uniforms: uniforms
	    };
	  }

	  /**
	   * Process the lines with uniforms. The function receives the lines containing all uniforms,
	   * both numerical as well as textures/samplers. The function also receives the format of uniform
	   * buffers (numerical) and bind groups (textures) for view and material level. All uniforms that
	   * match any of those are ignored, as those would be supplied by view / material level buffers.
	   * All leftover uniforms create uniform buffer and bind group for the mesh itself, containing
	   * uniforms that change on the level of the mesh.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./graphics-device.js').GraphicsDevice} device - The graphics device.
	   * @param {Array<UniformLine>} uniforms - Lines containing uniforms.
	   * @param {new Function("modulePath", "return import(modulePath)")('./shader-processor-options.js').ShaderProcessorOptions} processingOptions -
	   * Uniform formats.
	   * @param {new Function("modulePath", "return import(modulePath)")('./shader.js').Shader} shader - The shader definition.
	   * @returns {object} - The uniform data. Returns a shader code block containing uniforms, to be
	   * inserted into the shader, as well as generated uniform format structures for the mesh level.
	   */;
	  ShaderProcessor.processUniforms = function processUniforms(device, uniforms, processingOptions, shader) {
	    // split uniform lines into samplers and the rest
	    /** @type {Array<UniformLine>} */
	    var uniformLinesSamplers = [];
	    /** @type {Array<UniformLine>} */
	    var uniformLinesNonSamplers = [];
	    uniforms.forEach(function (uniform) {
	      if (uniform.isSampler) {
	        uniformLinesSamplers.push(uniform);
	      } else {
	        uniformLinesNonSamplers.push(uniform);
	      }
	    });

	    // build mesh uniform buffer format
	    var meshUniforms = [];
	    uniformLinesNonSamplers.forEach(function (uniform) {
	      // uniforms not already in supplied uniform buffers go to the mesh buffer
	      if (!processingOptions.hasUniform(uniform.name)) {
	        var uniformType = uniformTypeToName.indexOf(uniform.type);
	        Debug.assert(uniformType >= 0, "Uniform type " + uniform.type + " is not recognized on line [" + uniform.line + "]");
	        var uniformFormat = new UniformFormat(uniform.name, uniformType, uniform.arraySize);
	        Debug.assert(!uniformFormat.invalid, "Invalid uniform line: " + uniform.line, shader);
	        meshUniforms.push(uniformFormat);
	      }

	      // validate types in else
	    });

	    var meshUniformBufferFormat = meshUniforms.length ? new UniformBufferFormat(device, meshUniforms) : null;

	    // build mesh bind group format - start with uniform buffer
	    var bufferFormats = [];
	    if (meshUniformBufferFormat) {
	      // TODO: we could optimize visibility to only stages that use any of the data
	      bufferFormats.push(new BindBufferFormat(UNIFORM_BUFFER_DEFAULT_SLOT_NAME, SHADERSTAGE_VERTEX | SHADERSTAGE_FRAGMENT));
	    }

	    // add textures uniforms
	    var textureFormats = [];
	    uniformLinesSamplers.forEach(function (uniform) {
	      // unmatched texture uniforms go to mesh block
	      if (!processingOptions.hasTexture(uniform.name)) {
	        // sample type
	        // WebGpu does not currently support filtered float format textures, and so we map them to unfilterable type
	        // as we sample them without filtering anyways
	        var sampleType = SAMPLETYPE_FLOAT;
	        if (uniform.isSignedInt) {
	          sampleType = SAMPLETYPE_INT;
	        } else if (uniform.isUnsignedInt) {
	          sampleType = SAMPLETYPE_UINT;
	        } else {
	          if (uniform.precision === 'highp') sampleType = SAMPLETYPE_UNFILTERABLE_FLOAT;
	          if (shadowSamplers.has(uniform.type)) sampleType = SAMPLETYPE_DEPTH;
	        }

	        // dimension
	        var dimension = textureDimensions[uniform.type];

	        // TODO: we could optimize visibility to only stages that use any of the data
	        textureFormats.push(new BindTextureFormat(uniform.name, SHADERSTAGE_VERTEX | SHADERSTAGE_FRAGMENT, dimension, sampleType));
	      }

	      // validate types in else
	    });

	    var meshBindGroupFormat = new BindGroupFormat(device, bufferFormats, textureFormats);

	    // generate code for uniform buffers
	    var code = '';
	    processingOptions.uniformFormats.forEach(function (format, bindGroupIndex) {
	      if (format) {
	        code += format.getShaderDeclaration(bindGroupIndex, 0);
	      }
	    });

	    // and also for generated mesh format, which is at the slot 0 of the bind group
	    if (meshUniformBufferFormat) {
	      code += meshUniformBufferFormat.getShaderDeclaration(BINDGROUP_MESH, 0);
	    }

	    // generate code for textures
	    processingOptions.bindGroupFormats.forEach(function (format, bindGroupIndex) {
	      if (format) {
	        code += format.getShaderDeclarationTextures(bindGroupIndex);
	      }
	    });

	    // and also for generated mesh format
	    code += meshBindGroupFormat.getShaderDeclarationTextures(BINDGROUP_MESH);
	    return {
	      code: code,
	      meshUniformBufferFormat: meshUniformBufferFormat,
	      meshBindGroupFormat: meshBindGroupFormat
	    };
	  };
	  ShaderProcessor.processVaryings = function processVaryings(varyingLines, varyingMap, isVertex) {
	    var block = '';
	    var op = isVertex ? 'out' : 'in';
	    varyingLines.forEach(function (line, index) {
	      var words = ShaderProcessor.splitToWords(line);
	      var type = words[0];
	      var name = words[1];
	      if (isVertex) {
	        // store it in the map
	        varyingMap.set(name, index);
	      } else {
	        Debug.assert(varyingMap.has(name), "Fragment shader requires varying [" + name + "] but vertex shader does not generate it.");
	        index = varyingMap.get(name);
	      }

	      // generates: 'layout(location = 0) in vec4 position;'
	      block += "layout(location = " + index + ") " + op + " " + type + " " + name + ";\n";
	    });
	    return block;
	  };
	  ShaderProcessor.processOuts = function processOuts(outsLines) {
	    var block = '';
	    outsLines.forEach(function (line, index) {
	      // generates: 'layout(location = 0) out vec4 gl_FragColor;'
	      block += "layout(location = " + index + ") out " + line + ";\n";
	    });
	    return block;
	  }

	  // extract count from type ('vec3' => 3, 'float' => 1)
	  ;
	  ShaderProcessor.getTypeCount = function getTypeCount(type) {
	    var lastChar = type.substring(type.length - 1);
	    var num = parseInt(lastChar, 10);
	    return isNaN(num) ? 1 : num;
	  };
	  ShaderProcessor.processAttributes = function processAttributes(attributeLines, shaderDefinitionAttributes, processingOptions) {
	    var block = '';
	    var usedLocations = {};
	    attributeLines.forEach(function (line) {
	      var words = ShaderProcessor.splitToWords(line);
	      var type = words[0];
	      var name = words[1];
	      if (shaderDefinitionAttributes.hasOwnProperty(name)) {
	        var semantic = shaderDefinitionAttributes[name];
	        var location = semanticToLocation[semantic];
	        Debug.assert(!usedLocations.hasOwnProperty(location), "WARNING: Two vertex attributes are mapped to the same location in a shader: " + usedLocations[location] + " and " + semantic);
	        usedLocations[location] = semantic;

	        // if vertex format for this attribute is not of a float type, we need to adjust the attribute format, for example we convert
	        //      attribute vec4 vertex_position;
	        // to
	        //      attribute ivec4 _private_vertex_position;
	        //      vec4 vertex_position = vec4(_private_vertex_position);
	        // Note that we skip normalized elements, as shader receives them as floats already.
	        var copyCode;
	        var element = processingOptions.getVertexElement(semantic);
	        if (element) {
	          var dataType = element.dataType;
	          if (dataType !== TYPE_FLOAT32 && dataType !== TYPE_FLOAT16 && !element.normalize && !element.asInt) {
	            var attribNumElements = ShaderProcessor.getTypeCount(type);
	            var newName = "_private_" + name;

	            // second line of new code, copy private (u)int type into vec type
	            copyCode = "vec" + attribNumElements + " " + name + " = vec" + attribNumElements + "(" + newName + ");\n";
	            name = newName;

	            // new attribute type, based on the vertex format element type, example: vec3 -> ivec3
	            var isSignedType = dataType === TYPE_INT8 || dataType === TYPE_INT16 || dataType === TYPE_INT32;
	            if (attribNumElements === 1) {
	              type = isSignedType ? 'int' : 'uint';
	            } else {
	              type = isSignedType ? "ivec" + attribNumElements : "uvec" + attribNumElements;
	            }
	          }
	        }

	        // generates: 'layout(location = 0) in vec4 position;'
	        block += "layout(location = " + location + ") in " + type + " " + name + ";\n";
	        if (copyCode) {
	          block += copyCode;
	        }
	      }
	    });
	    return block;
	  };
	  ShaderProcessor.splitToWords = function splitToWords(line) {
	    // remove any double spaces
	    line = line.replace(/\s+/g, ' ').trim();
	    return line.split(' ');
	  };
	  ShaderProcessor.cutOut = function cutOut(src, start, end, replacement) {
	    return src.substring(0, start) + replacement + src.substring(end);
	  };
	  return ShaderProcessor;
	}();

	/**
	 * A WebGPU implementation of the Shader.
	 *
	 * @ignore
	 */
	var WebgpuShader = /*#__PURE__*/function () {
	  /**
	   * @param {new Function("modulePath", "return import(modulePath)")('../shader.js').Shader} shader - The shader.
	   */
	  function WebgpuShader(shader) {
	    /**
	     * Transpiled vertex shader code.
	     *
	     * @type {string|null}
	     */
	    this._vertexCode = null;
	    /**
	     * Transpiled fragment shader code.
	     *
	     * @type {string|null}
	     */
	    this._fragmentCode = null;
	    /**
	     * Compute shader code.
	     *
	     * @type {string|null}
	     */
	    this._computeCode = null;
	    /**
	     * Name of the vertex entry point function.
	     */
	    this.vertexEntryPoint = 'main';
	    /**
	     * Name of the fragment entry point function.
	     */
	    this.fragmentEntryPoint = 'main';
	    /**
	     * Name of the compute entry point function.
	     */
	    this.computeEntryPoint = 'main';
	    /** @type {new Function("modulePath", "return import(modulePath)")('../shader.js').Shader} */
	    this.shader = shader;
	    var definition = shader.definition;
	    Debug.assert(definition);
	    if (definition.shaderLanguage === SHADERLANGUAGE_WGSL) {
	      var _definition$vshader, _definition$fshader, _definition$cshader;
	      this._vertexCode = (_definition$vshader = definition.vshader) != null ? _definition$vshader : null;
	      this._fragmentCode = (_definition$fshader = definition.fshader) != null ? _definition$fshader : null;
	      this._computeCode = (_definition$cshader = definition.cshader) != null ? _definition$cshader : null;
	      this.meshUniformBufferFormat = definition.meshUniformBufferFormat;
	      this.meshBindGroupFormat = definition.meshBindGroupFormat;
	      this.vertexEntryPoint = 'vertexMain';
	      this.fragmentEntryPoint = 'fragmentMain';
	      shader.ready = true;
	    } else {
	      if (definition.processingOptions) {
	        this.process();
	      }
	    }
	  }

	  /**
	   * Free the WebGPU resources associated with a shader.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('../shader.js').Shader} shader - The shader to free.
	   */
	  var _proto = WebgpuShader.prototype;
	  _proto.destroy = function destroy(shader) {
	    this._vertexCode = null;
	    this._fragmentCode = null;
	  };
	  _proto.createShaderModule = function createShaderModule(code, shaderType) {
	    var device = this.shader.device;
	    var wgpu = device.wgpu;
	    WebgpuDebug.validate(device);
	    var shaderModule = wgpu.createShaderModule({
	      code: code
	    });
	    DebugHelper.setLabel(shaderModule, shaderType + ":" + this.shader.label);
	    WebgpuDebug.end(device, {
	      shaderType: shaderType,
	      source: code,
	      shader: this.shader
	    });
	    return shaderModule;
	  };
	  _proto.getVertexShaderModule = function getVertexShaderModule() {
	    return this.createShaderModule(this._vertexCode, 'Vertex');
	  };
	  _proto.getFragmentShaderModule = function getFragmentShaderModule() {
	    return this.createShaderModule(this._fragmentCode, 'Fragment');
	  };
	  _proto.getComputeShaderModule = function getComputeShaderModule() {
	    return this.createShaderModule(this._computeCode, 'Compute');
	  };
	  _proto.process = function process() {
	    var _this = this;
	    var shader = this.shader;

	    // process the shader source to allow for uniforms
	    var processed = ShaderProcessor.run(shader.device, shader.definition, shader);

	    // keep reference to processed shaders in debug mode
	    Debug.call(function () {
	      _this.processed = processed;
	    });
	    this._vertexCode = this.transpile(processed.vshader, 'vertex', shader.definition.vshader);
	    this._fragmentCode = this.transpile(processed.fshader, 'fragment', shader.definition.fshader);
	    if (!(this._vertexCode && this._fragmentCode)) {
	      shader.failed = true;
	    } else {
	      shader.ready = true;
	    }
	    shader.meshUniformBufferFormat = processed.meshUniformBufferFormat;
	    shader.meshBindGroupFormat = processed.meshBindGroupFormat;
	  };
	  _proto.transpile = function transpile(src, shaderType, originalSrc) {
	    try {
	      var spirv = this.shader.device.glslang.compileGLSL(src, shaderType);
	      return this.shader.device.twgsl.convertSpirV2WGSL(spirv);
	    } catch (err) {
	      console.error("Failed to transpile webgl " + shaderType + " shader [" + this.shader.label + "] to WebGPU: [" + err.message + "] while rendering " + DebugGraphics.toString(), {
	        processed: src,
	        original: originalSrc,
	        shader: this.shader
	      });
	    }
	  };
	  /**
	   * Dispose the shader when the context has been lost.
	   */
	  _proto.loseContext = function loseContext() {}

	  /**
	   * Restore shader after the context has been obtained.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('../graphics-device.js').GraphicsDevice} device - The graphics device.
	   * @param {new Function("modulePath", "return import(modulePath)")('../shader.js').Shader} shader - The shader to restore.
	   */;
	  _proto.restoreContext = function restoreContext(device, shader) {};
	  _createClass(WebgpuShader, [{
	    key: "vertexCode",
	    get: function get() {
	      Debug.assert(this._vertexCode);
	      return this._vertexCode;
	    }
	  }, {
	    key: "fragmentCode",
	    get: function get() {
	      Debug.assert(this._fragmentCode);
	      return this._fragmentCode;
	    }
	  }]);
	  return WebgpuShader;
	}();

	/**
	 * A class providing utility functions for textures.
	 *
	 * @ignore
	 */
	var TextureUtils = /*#__PURE__*/function () {
	  function TextureUtils() {}
	  /**
	   * Calculate the dimension of a texture at a specific mip level.
	   *
	   * @param {number} dimension - Texture dimension at level 0.
	   * @param {number} mipLevel - Mip level.
	   * @returns {number} The dimension of the texture at the specified mip level.
	   */
	  TextureUtils.calcLevelDimension = function calcLevelDimension(dimension, mipLevel) {
	    return Math.max(dimension >> mipLevel, 1);
	  }

	  /**
	   * Calculate the number of mip levels for a texture with the specified dimensions.
	   *
	   * @param {number} width - Texture's width.
	   * @param {number} height - Texture's height.
	   * @param {number} [depth] - Texture's depth. Defaults to 1.
	   * @returns {number} The number of mip levels required for the texture.
	   */;
	  TextureUtils.calcMipLevelsCount = function calcMipLevelsCount(width, height, depth) {
	    if (depth === void 0) {
	      depth = 1;
	    }
	    return 1 + Math.floor(Math.log2(Math.max(width, height, depth)));
	  }

	  /**
	   * Calculate the size in bytes of the texture level given its format and dimensions.
	   *
	   * @param {number} width - Texture's width.
	   * @param {number} height - Texture's height.
	   * @param {number} depth - Texture's depth.
	   * @param {number} format - Texture's pixel format PIXELFORMAT_***.
	   * @returns {number} The number of bytes of GPU memory required for the texture.
	   * @ignore
	   */;
	  TextureUtils.calcLevelGpuSize = function calcLevelGpuSize(width, height, depth, format) {
	    var _pixelFormatInfo$get$, _pixelFormatInfo$get, _formatInfo$blockSize;
	    var formatInfo = pixelFormatInfo.get(format);
	    Debug.assert(formatInfo !== undefined, "Invalid pixel format " + format);
	    var pixelSize = (_pixelFormatInfo$get$ = (_pixelFormatInfo$get = pixelFormatInfo.get(format)) == null ? void 0 : _pixelFormatInfo$get.size) != null ? _pixelFormatInfo$get$ : 0;
	    if (pixelSize > 0) {
	      return width * height * depth * pixelSize;
	    }
	    var blockSize = (_formatInfo$blockSize = formatInfo.blockSize) != null ? _formatInfo$blockSize : 0;
	    var blockWidth = Math.floor((width + 3) / 4);
	    var blockHeight = Math.floor((height + 3) / 4);
	    var blockDepth = Math.floor((depth + 3) / 4);
	    if (format === PIXELFORMAT_PVRTC_2BPP_RGB_1 || format === PIXELFORMAT_PVRTC_2BPP_RGBA_1) {
	      blockWidth = Math.max(Math.floor(blockWidth / 2), 1);
	    }
	    return blockWidth * blockHeight * blockDepth * blockSize;
	  }

	  /**
	   * Calculate the GPU memory required for a texture.
	   *
	   * @param {number} width - Texture's width.
	   * @param {number} height - Texture's height.
	   * @param {number} depth - Texture's depth.
	   * @param {number} format - Texture's pixel format PIXELFORMAT_***.
	   * @param {boolean} mipmaps - True if the texture includes mipmaps, false otherwise.
	   * @param {boolean} cubemap - True is the texture is a cubemap, false otherwise.
	   * @returns {number} The number of bytes of GPU memory required for the texture.
	   * @ignore
	   */;
	  TextureUtils.calcGpuSize = function calcGpuSize(width, height, depth, format, mipmaps, cubemap) {
	    var result = 0;
	    while (1) {
	      result += TextureUtils.calcLevelGpuSize(width, height, depth, format);

	      // we're done if mipmaps aren't required or we've calculated the smallest mipmap level
	      if (!mipmaps || width === 1 && height === 1 && depth === 1) {
	        break;
	      }
	      width = Math.max(width >> 1, 1);
	      height = Math.max(height >> 1, 1);
	      depth = Math.max(depth >> 1, 1);
	    }
	    return result * (cubemap ? 6 : 1);
	  };
	  return TextureUtils;
	}();

	// map of ADDRESS_*** to GPUAddressMode
	var gpuAddressModes = [];
	gpuAddressModes[ADDRESS_REPEAT] = 'repeat';
	gpuAddressModes[ADDRESS_CLAMP_TO_EDGE] = 'clamp-to-edge';
	gpuAddressModes[ADDRESS_MIRRORED_REPEAT] = 'mirror-repeat';

	// map of FILTER_*** to GPUFilterMode for level and mip sampling
	var gpuFilterModes = [];
	gpuFilterModes[FILTER_NEAREST] = {
	  level: 'nearest',
	  mip: 'nearest'
	};
	gpuFilterModes[FILTER_LINEAR] = {
	  level: 'linear',
	  mip: 'nearest'
	};
	gpuFilterModes[FILTER_NEAREST_MIPMAP_NEAREST] = {
	  level: 'nearest',
	  mip: 'nearest'
	};
	gpuFilterModes[FILTER_NEAREST_MIPMAP_LINEAR] = {
	  level: 'nearest',
	  mip: 'linear'
	};
	gpuFilterModes[FILTER_LINEAR_MIPMAP_NEAREST] = {
	  level: 'linear',
	  mip: 'nearest'
	};
	gpuFilterModes[FILTER_LINEAR_MIPMAP_LINEAR] = {
	  level: 'linear',
	  mip: 'linear'
	};
	var dummyUse = function dummyUse(thingOne) {
	  // so lint thinks we're doing something with thingOne
	};

	/**
	 * A WebGPU implementation of the Texture.
	 *
	 * @ignore
	 */
	var WebgpuTexture = /*#__PURE__*/function () {
	  function WebgpuTexture(texture) {
	    /**
	     * @type {GPUTexture}
	     * @private
	     */
	    this.gpuTexture = void 0;
	    /**
	     * @type {GPUTextureView}
	     * @private
	     */
	    this.view = void 0;
	    /**
	     * An array of samplers, addressed by SAMPLETYPE_*** constant, allowing texture to be sampled
	     * using different samplers. Most textures are sampled as interpolated floats, but some can
	     * additionally be sampled using non-interpolated floats (raw data) or compare sampling
	     * (shadow maps).
	     *
	     * @type {GPUSampler[]}
	     * @private
	     */
	    this.samplers = [];
	    /**
	     * @type {GPUTextureDescriptor}
	     * @private
	     */
	    this.descr = void 0;
	    /**
	     * @type {GPUTextureFormat}
	     * @private
	     */
	    this.format = void 0;
	    /** @type {new Function("modulePath", "return import(modulePath)")('../texture.js').Texture} */
	    this.texture = texture;
	    this.format = gpuTextureFormats[texture.format];
	    Debug.assert(this.format !== '', "WebGPU does not support texture format " + texture.format + " for texture " + texture.name, texture);
	    this.create(texture.device);
	  }
	  var _proto = WebgpuTexture.prototype;
	  _proto.create = function create(device) {
	    var texture = this.texture;
	    var wgpu = device.wgpu;
	    var mipLevelCount = texture.requiredMipLevels;
	    Debug.assert(texture.width > 0 && texture.height > 0, "Invalid texture dimensions " + texture.width + "x" + texture.height + " for texture " + texture.name, texture);
	    this.descr = {
	      size: {
	        width: texture.width,
	        height: texture.height,
	        depthOrArrayLayers: texture.cubemap ? 6 : texture.array ? texture.arrayLength : 1
	      },
	      format: this.format,
	      mipLevelCount: mipLevelCount,
	      sampleCount: 1,
	      dimension: texture.volume ? '3d' : '2d',
	      // TODO: use only required usage flags
	      // COPY_SRC - probably only needed on render target textures, to support copyRenderTarget (grab pass needs it)
	      // RENDER_ATTACHMENT - needed for mipmap generation
	      usage: GPUTextureUsage.TEXTURE_BINDING | GPUTextureUsage.COPY_DST | GPUTextureUsage.COPY_SRC | (isCompressedPixelFormat(texture.format) ? 0 : GPUTextureUsage.RENDER_ATTACHMENT) | (texture.storage ? GPUTextureUsage.STORAGE_BINDING : 0)
	    };
	    WebgpuDebug.validate(device);
	    this.gpuTexture = wgpu.createTexture(this.descr);
	    DebugHelper.setLabel(this.gpuTexture, "" + texture.name + (texture.cubemap ? '[cubemap]' : '') + (texture.volume ? '[3d]' : ''));
	    WebgpuDebug.end(device, {
	      descr: this.descr,
	      texture: texture
	    });

	    // default texture view descriptor
	    var viewDescr;

	    // some format require custom default texture view
	    if (this.texture.format === PIXELFORMAT_DEPTHSTENCIL) {
	      // we expose the depth part of the format
	      viewDescr = {
	        format: 'depth24plus',
	        aspect: 'depth-only'
	      };
	    }
	    this.view = this.createView(viewDescr);
	  };
	  _proto.destroy = function destroy(device) {};
	  _proto.propertyChanged = function propertyChanged(flag) {
	    // samplers need to be recreated
	    this.samplers.length = 0;
	  }

	  /**
	   * @param {any} device - The Graphics Device.
	   * @returns {any} - Returns the view.
	   */;
	  _proto.getView = function getView(device) {
	    this.uploadImmediate(device, this.texture);
	    Debug.assert(this.view);
	    return this.view;
	  };
	  _proto.createView = function createView(viewDescr) {
	    var _options$format, _options$dimension, _options$aspect, _options$baseMipLevel, _options$mipLevelCoun, _options$baseArrayLay, _options$arrayLayerCo;
	    var options = viewDescr != null ? viewDescr : {};
	    var textureDescr = this.descr;
	    var texture = this.texture;

	    // '1d', '2d', '2d-array', 'cube', 'cube-array', '3d'
	    var defaultViewDimension = function defaultViewDimension() {
	      if (texture.cubemap) return 'cube';
	      if (texture.volume) return '3d';
	      if (texture.array) return '2d-array';
	      return '2d';
	    };

	    /** @type {GPUTextureViewDescriptor} */
	    var descr = {
	      format: (_options$format = options.format) != null ? _options$format : textureDescr.format,
	      dimension: (_options$dimension = options.dimension) != null ? _options$dimension : defaultViewDimension(),
	      aspect: (_options$aspect = options.aspect) != null ? _options$aspect : 'all',
	      baseMipLevel: (_options$baseMipLevel = options.baseMipLevel) != null ? _options$baseMipLevel : 0,
	      mipLevelCount: (_options$mipLevelCoun = options.mipLevelCount) != null ? _options$mipLevelCoun : textureDescr.mipLevelCount,
	      baseArrayLayer: (_options$baseArrayLay = options.baseArrayLayer) != null ? _options$baseArrayLay : 0,
	      arrayLayerCount: (_options$arrayLayerCo = options.arrayLayerCount) != null ? _options$arrayLayerCo : textureDescr.depthOrArrayLayers
	    };
	    var view = this.gpuTexture.createView(descr);
	    DebugHelper.setLabel(view, (viewDescr ? "CustomView" + JSON.stringify(viewDescr) : 'DefaultView') + ":" + this.texture.name);
	    return view;
	  }

	  // TODO: share a global map of samplers. Possibly even use shared samplers for bind group,
	  // or maybe even have some attached in view bind group and use globally

	  /**
	   * @param {any} device - The Graphics Device.
	   * @param {number} [sampleType] - A sample type for the sampler, SAMPLETYPE_*** constant. If not
	   * specified, the sampler type is based on the texture format / texture sampling type.
	   * @returns {any} - Returns the sampler.
	   */;
	  _proto.getSampler = function getSampler(device, sampleType) {
	    var sampler = this.samplers[sampleType];
	    if (!sampler) {
	      var texture = this.texture;
	      var label;

	      /** @type GPUSamplerDescriptor */
	      var descr = {
	        addressModeU: gpuAddressModes[texture.addressU],
	        addressModeV: gpuAddressModes[texture.addressV],
	        addressModeW: gpuAddressModes[texture.addressW]
	      };

	      // default for compare sampling of texture
	      if (!sampleType && texture.compareOnRead) {
	        sampleType = SAMPLETYPE_DEPTH;
	      }
	      if (sampleType === SAMPLETYPE_DEPTH || sampleType === SAMPLETYPE_INT || sampleType === SAMPLETYPE_UINT) {
	        // depth compare sampling
	        descr.compare = 'less';
	        descr.magFilter = 'linear';
	        descr.minFilter = 'linear';
	        label = 'Compare';
	      } else if (sampleType === SAMPLETYPE_UNFILTERABLE_FLOAT) {
	        // webgpu cannot currently filter float / half float textures, or integer textures
	        descr.magFilter = 'nearest';
	        descr.minFilter = 'nearest';
	        descr.mipmapFilter = 'nearest';
	        label = 'Unfilterable';
	      } else {
	        // TODO: this is temporary and needs to be made generic
	        if (this.texture.format === PIXELFORMAT_RGBA32F || this.texture.format === PIXELFORMAT_DEPTHSTENCIL || this.texture.format === PIXELFORMAT_RGBA16F || isIntegerPixelFormat(this.texture.format)) {
	          descr.magFilter = 'nearest';
	          descr.minFilter = 'nearest';
	          descr.mipmapFilter = 'nearest';
	          label = 'Nearest';
	        } else {
	          descr.magFilter = gpuFilterModes[texture.magFilter].level;
	          descr.minFilter = gpuFilterModes[texture.minFilter].level;
	          descr.mipmapFilter = gpuFilterModes[texture.minFilter].mip;
	          Debug.call(function () {
	            label = "Texture:" + texture.magFilter + "-" + texture.minFilter + "-" + descr.mipmapFilter;
	          });
	        }
	      }

	      // ensure anisotropic filtering is only set when filtering is correctly
	      // set up
	      var allLinear = descr.minFilter === 'linear' && descr.magFilter === 'linear' && descr.mipmapFilter === 'linear';
	      descr.maxAnisotropy = allLinear ? math.clamp(Math.round(texture._anisotropy), 1, device.maxTextureAnisotropy) : 1;
	      sampler = device.wgpu.createSampler(descr);
	      DebugHelper.setLabel(sampler, label);
	      this.samplers[sampleType] = sampler;
	    }
	    return sampler;
	  };
	  _proto.loseContext = function loseContext() {}

	  /**
	   * @param {new Function("modulePath", "return import(modulePath)")('./webgpu-graphics-device.js').WebgpuGraphicsDevice} device - The graphics
	   * device.
	   * @param {new Function("modulePath", "return import(modulePath)")('../texture.js').Texture} texture - The texture.
	   */;
	  _proto.uploadImmediate = function uploadImmediate(device, texture) {
	    if (texture._needsUpload || texture._needsMipmapsUpload) {
	      this.uploadData(device);
	      texture._needsUpload = false;
	      texture._needsMipmapsUpload = false;
	    }
	  }

	  /**
	   * @param {new Function("modulePath", "return import(modulePath)")('./webgpu-graphics-device.js').WebgpuGraphicsDevice} device - The graphics
	   * device.
	   */;
	  _proto.uploadData = function uploadData(device) {
	    var texture = this.texture;
	    if (texture._levels) {
	      // upload texture data if any
	      var anyUploads = false;
	      var anyLevelMissing = false;
	      var requiredMipLevels = texture.requiredMipLevels;
	      for (var mipLevel = 0; mipLevel < requiredMipLevels; mipLevel++) {
	        var mipObject = texture._levels[mipLevel];
	        if (mipObject) {
	          if (texture.cubemap) {
	            for (var face = 0; face < 6; face++) {
	              var faceSource = mipObject[face];
	              if (faceSource) {
	                if (this.isExternalImage(faceSource)) {
	                  this.uploadExternalImage(device, faceSource, mipLevel, face);
	                  anyUploads = true;
	                } else if (ArrayBuffer.isView(faceSource)) {
	                  // typed array

	                  this.uploadTypedArrayData(device, faceSource, mipLevel, face);
	                  anyUploads = true;
	                } else {
	                  Debug.error('Unsupported texture source data for cubemap face', faceSource);
	                }
	              } else {
	                anyLevelMissing = true;
	              }
	            }
	          } else if (texture._volume) {
	            Debug.warn('Volume texture data upload is not supported yet', this.texture);
	          } else if (texture.array) {
	            // texture array

	            if (texture.arrayLength === mipObject.length) {
	              for (var index = 0; index < texture._arrayLength; index++) {
	                var arraySource = mipObject[index];
	                if (this.isExternalImage(arraySource)) {
	                  this.uploadExternalImage(device, arraySource, mipLevel, index);
	                  anyUploads = true;
	                } else if (ArrayBuffer.isView(arraySource)) {
	                  // typed array

	                  this.uploadTypedArrayData(device, arraySource, mipLevel, index);
	                  anyUploads = true;
	                } else {
	                  Debug.error('Unsupported texture source data for texture array entry', arraySource);
	                }
	              }
	            } else {
	              anyLevelMissing = true;
	            }
	          } else {
	            // 2d texture

	            if (this.isExternalImage(mipObject)) {
	              this.uploadExternalImage(device, mipObject, mipLevel, 0);
	              anyUploads = true;
	            } else if (ArrayBuffer.isView(mipObject)) {
	              // typed array

	              this.uploadTypedArrayData(device, mipObject, mipLevel, 0);
	              anyUploads = true;
	            } else {
	              Debug.error('Unsupported texture source data', mipObject);
	            }
	          }
	        } else {
	          anyLevelMissing = true;
	        }
	      }
	      if (anyUploads && anyLevelMissing && texture.mipmaps && !isCompressedPixelFormat(texture.format)) {
	        device.mipmapRenderer.generate(this);
	      }

	      // update vram stats
	      if (texture._gpuSize) {
	        texture.adjustVramSizeTracking(device._vram, -texture._gpuSize);
	      }
	      texture._gpuSize = texture.gpuSize;
	      texture.adjustVramSizeTracking(device._vram, texture._gpuSize);
	    }
	  }

	  // image types supported by copyExternalImageToTexture
	  ;
	  _proto.isExternalImage = function isExternalImage(image) {
	    return image instanceof ImageBitmap || image instanceof HTMLVideoElement || image instanceof HTMLCanvasElement || image instanceof OffscreenCanvas;
	  };
	  _proto.uploadExternalImage = function uploadExternalImage(device, image, mipLevel, index) {
	    Debug.assert(mipLevel < this.descr.mipLevelCount, "Accessing mip level " + mipLevel + " of texture with " + this.descr.mipLevelCount + " mip levels", this);
	    var src = {
	      source: image,
	      origin: [0, 0],
	      flipY: false
	    };
	    var dst = {
	      texture: this.gpuTexture,
	      mipLevel: mipLevel,
	      origin: [0, 0, index],
	      aspect: 'all' // can be: "all", "stencil-only", "depth-only"
	    };

	    var copySize = {
	      width: this.descr.size.width,
	      height: this.descr.size.height,
	      depthOrArrayLayers: 1 // single layer
	    };

	    // submit existing scheduled commands to the queue before copying to preserve the order
	    device.submit();

	    // create 2d context so webgpu can upload the texture
	    dummyUse(image instanceof HTMLCanvasElement && image.getContext('2d'));
	    Debug.trace(TRACEID_RENDER_QUEUE, "IMAGE-TO-TEX: mip:" + mipLevel + " index:" + index + " " + this.texture.name);
	    device.wgpu.queue.copyExternalImageToTexture(src, dst, copySize);
	  };
	  _proto.uploadTypedArrayData = function uploadTypedArrayData(device, data, mipLevel, index) {
	    var texture = this.texture;
	    var wgpu = device.wgpu;

	    /** @type {GPUImageCopyTexture} */
	    var dest = {
	      texture: this.gpuTexture,
	      origin: [0, 0, index],
	      mipLevel: mipLevel
	    };

	    // texture dimensions at the specified mip level
	    var width = TextureUtils.calcLevelDimension(texture.width, mipLevel);
	    var height = TextureUtils.calcLevelDimension(texture.height, mipLevel);

	    // data sizes
	    var byteSize = TextureUtils.calcLevelGpuSize(width, height, 1, texture.format);
	    Debug.assert(byteSize === data.byteLength, "Error uploading data to texture, the data byte size of " + data.byteLength + " does not match required " + byteSize, texture);
	    var formatInfo = pixelFormatInfo.get(texture.format);
	    Debug.assert(formatInfo);

	    /** @type {GPUImageDataLayout} */
	    var dataLayout;
	    var size;
	    if (formatInfo.size) {
	      // uncompressed format
	      dataLayout = {
	        offset: 0,
	        bytesPerRow: formatInfo.size * width,
	        rowsPerImage: height
	      };
	      size = {
	        width: width,
	        height: height
	      };
	    } else if (formatInfo.blockSize) {
	      // compressed format
	      var blockDim = function blockDim(size) {
	        return Math.floor((size + 3) / 4);
	      };
	      dataLayout = {
	        offset: 0,
	        bytesPerRow: formatInfo.blockSize * blockDim(width),
	        rowsPerImage: blockDim(height)
	      };
	      size = {
	        width: Math.max(4, width),
	        height: Math.max(4, height)
	      };
	    } else {
	      Debug.assert(false, "WebGPU does not yet support texture format " + formatInfo.name + " for texture " + texture.name, texture);
	    }

	    // submit existing scheduled commands to the queue before copying to preserve the order
	    device.submit();
	    Debug.trace(TRACEID_RENDER_QUEUE, "WRITE-TEX: mip:" + mipLevel + " index:" + index + " " + this.texture.name);
	    wgpu.queue.writeTexture(dest, data, dataLayout, size);
	  };
	  return WebgpuTexture;
	}();

	/**
	 * A WebGPU implementation of the UniformBuffer.
	 *
	 * @ignore
	 */
	var WebgpuUniformBuffer = /*#__PURE__*/function (_WebgpuBuffer) {
	  _inheritsLoose(WebgpuUniformBuffer, _WebgpuBuffer);
	  function WebgpuUniformBuffer(uniformBuffer) {
	    return _WebgpuBuffer.call(this) || this;
	  }
	  var _proto = WebgpuUniformBuffer.prototype;
	  _proto.destroy = function destroy(device) {
	    _WebgpuBuffer.prototype.destroy.call(this, device);

	    // TODO: clear up bound uniform buffers
	  };
	  _proto.unlock = function unlock(uniformBuffer) {
	    var device = uniformBuffer.device;
	    _WebgpuBuffer.prototype.unlock.call(this, device, undefined, GPUBufferUsage.UNIFORM, uniformBuffer.storage);
	  };
	  return WebgpuUniformBuffer;
	}(WebgpuBuffer);

	/**
	 * A WebGPU implementation of the VertexBuffer.
	 *
	 * @ignore
	 */
	var WebgpuVertexBuffer = /*#__PURE__*/function (_WebgpuBuffer) {
	  _inheritsLoose(WebgpuVertexBuffer, _WebgpuBuffer);
	  function WebgpuVertexBuffer(vertexBuffer, format) {
	    return _WebgpuBuffer.call(this) || this;
	  }
	  var _proto = WebgpuVertexBuffer.prototype;
	  _proto.destroy = function destroy(device) {
	    _WebgpuBuffer.prototype.destroy.call(this, device);

	    // TODO: clear up bound vertex buffers
	  };
	  _proto.unlock = function unlock(vertexBuffer) {
	    var device = vertexBuffer.device;
	    _WebgpuBuffer.prototype.unlock.call(this, device, vertexBuffer.usage, GPUBufferUsage.VERTEX, vertexBuffer.storage);
	  };
	  return WebgpuVertexBuffer;
	}(WebgpuBuffer);

	// id for debug tracing
	var TRACEID = 'Preprocessor';

	// accepted keywords
	var KEYWORD = /[ \t]*#(ifn?def|if|endif|else|elif|define|undef|extension)/g;

	// #define EXPRESSION
	var DEFINE = /define[ \t]+([^\n]+)\r?(?:\n|$)/g;

	// #extension IDENTIFIER : enabled
	var EXTENSION = /extension[ \t]+([\w-]+)[ \t]*:[ \t]*(enable|require)/g;

	// #undef EXPRESSION
	var UNDEF = /undef[ \t]+([^\n]+)\r?(?:\n|$)/g;

	// #ifdef/#ifndef SOMEDEFINE, #if EXPRESSION
	var IF = /(ifdef|ifndef|if)[ \t]*([^\r\n]+)\r?\n/g;

	// #endif/#else or #elif EXPRESSION
	var ENDIF = /(endif|else|elif)([ \t]+[^\r\n]+)?\r?(?:\n|$)/g;

	// identifier
	var IDENTIFIER$1 = /([\w-]+)/;

	// [!]defined(EXPRESSION)
	var DEFINED = /(!|\s)?defined\(([\w-]+)\)/;

	// currently unsupported characters in the expression: | & < > = + -
	var INVALID = /[><=|&+-]/g;

	/**
	 * Pure static class implementing subset of C-style preprocessor.
	 * inspired by: https://github.com/dcodeIO/Preprocessor.js
	 *
	 * @ignore
	 */
	var Preprocessor = /*#__PURE__*/function () {
	  function Preprocessor() {}
	  /**
	   * Run c-like preprocessor on the source code, and resolves the code based on the defines and ifdefs
	   *
	   * @param {string} source - The source code to work on.
	   * @param {boolean} [stripUnusedColorAttachments] - If true, strips unused color attachments.
	   * @returns {string|null} Returns preprocessed source code, or null in case of error.
	   */
	  Preprocessor.run = function run(source, stripUnusedColorAttachments) {
	    if (stripUnusedColorAttachments === void 0) {
	      stripUnusedColorAttachments = false;
	    }
	    // strips comments, handles // and many cases of /*
	    source = source.replace(/\/\*[\s\S]*?\*\/|([^\\:]|^)\/\/.*$/gm, '$1');

	    // right trim each line
	    source = source.split(/\r?\n/).map(function (line) {
	      return line.trimEnd();
	    }).join('\n');

	    // generate defines to remove unused color attachments
	    var defines = new Map();
	    if (stripUnusedColorAttachments) {
	      // find out how many times pcFragColorX is used (see gles3.js)
	      var counts = new Map();
	      var regex = /(pcFragColor[1-8])\b/g;
	      var matches = source.match(regex);
	      matches == null || matches.forEach(function (match) {
	        var _counts$get;
	        var index = parseInt(match.charAt(match.length - 1), 10);
	        counts.set(index, ((_counts$get = counts.get(index)) != null ? _counts$get : 0) + 1);
	      });

	      // if pcFragColorX is used only once, remove it
	      counts.forEach(function (count, index) {
	        if (count === 1) {
	          defines.set("REMOVE_COLOR_ATTACHMENT_" + index, '');
	        }
	      });
	    }

	    // preprocess defines / ifdefs ..
	    source = this._preprocess(source, defines);

	    // extract defines that evaluate to an integer number
	    var intDefines = new Map();
	    defines.forEach(function (value, key) {
	      if (Number.isInteger(parseFloat(value)) && !value.includes('.')) {
	        intDefines.set(key, value);
	      }
	    });

	    // remove empty lines
	    source = this.RemoveEmptyLines(source);

	    // process array sizes
	    source = this.processArraySize(source, intDefines);
	    return source;
	  };
	  Preprocessor.processArraySize = function processArraySize(source, intDefines) {
	    if (source !== null) {
	      // replace lines containing "[intDefine]" with their values, so that we know the array size for WebGPU uniform buffer
	      // example: weight[SAMPLES] => float weight[11] in case there was a "define SAMPLES 11" in the source code
	      intDefines.forEach(function (value, key) {
	        source = source.replace(new RegExp("\\[" + key + "\\]", 'g'), "[" + value + "]");
	      });
	    }
	    return source;
	  };
	  Preprocessor.RemoveEmptyLines = function RemoveEmptyLines(source) {
	    if (source !== null) {
	      source = source.split(/\r?\n/)

	      // convert lines with only white space into empty string
	      .map(function (line) {
	        return line.trim() === '' ? '' : line;
	      }).join('\n');

	      // remove more than 1 consecutive empty lines
	      source = source.replace(/(\n\n){3,}/gm, '\n\n');
	    }
	    return source;
	  }

	  /**
	   * Process source code, and resolves the code based on the defines and ifdefs.
	   *
	   * @param {string} source - The source code to work on.
	   * @param {Map<string, string>} defines - Supplied defines which are used in addition to those
	   * defined in the source code. Maps a define name to its value. Note that the map is modified
	   * by the function.
	   * @returns {string} Returns preprocessed source code.
	   */;
	  Preprocessor._preprocess = function _preprocess(source, defines) {
	    if (defines === void 0) {
	      defines = new Map();
	    }
	    var originalSource = source;

	    // stack, storing info about ifdef blocks
	    var stack = [];

	    // true if the function encounter a problem
	    var error = false;
	    var match;
	    while ((match = KEYWORD.exec(source)) !== null) {
	      var keyword = match[1];
	      switch (keyword) {
	        case 'define':
	          {
	            // read the rest of the define line
	            DEFINE.lastIndex = match.index;
	            var define = DEFINE.exec(source);
	            Debug.assert(define, "Invalid [" + keyword + "]: " + source.substring(match.index, match.index + 100) + "...");
	            error || (error = define === null);
	            var expression = define[1];

	            // split it to identifier name and a value
	            IDENTIFIER$1.lastIndex = define.index;
	            var identifierValue = IDENTIFIER$1.exec(expression);
	            var identifier = identifierValue[1];
	            var value = expression.substring(identifier.length).trim();
	            if (value === "") value = "true";

	            // are we inside if-blocks that are accepted
	            var keep = Preprocessor._keep(stack);
	            if (keep) {
	              defines.set(identifier, value);
	            }
	            Debug.trace(TRACEID, keyword + ": [" + identifier + "] " + value + " " + (keep ? "" : "IGNORED"));

	            // continue on the next line
	            KEYWORD.lastIndex = define.index + define[0].length;
	            break;
	          }
	        case 'undef':
	          {
	            // read the rest of the define line
	            UNDEF.lastIndex = match.index;
	            var undef = UNDEF.exec(source);
	            var _identifier = undef[1].trim();

	            // are we inside if-blocks that are accepted
	            var _keep2 = Preprocessor._keep(stack);

	            // remove it from defines
	            if (_keep2) {
	              defines.delete(_identifier);
	            }
	            Debug.trace(TRACEID, keyword + ": [" + _identifier + "] " + (_keep2 ? "" : "IGNORED"));

	            // continue on the next line
	            KEYWORD.lastIndex = undef.index + undef[0].length;
	            break;
	          }
	        case 'extension':
	          {
	            EXTENSION.lastIndex = match.index;
	            var extension = EXTENSION.exec(source);
	            Debug.assert(extension, "Invalid [" + keyword + "]: " + source.substring(match.index, match.index + 100) + "...");
	            error || (error = extension === null);
	            if (extension) {
	              var _identifier2 = extension[1];

	              // are we inside if-blocks that are accepted
	              var _keep3 = Preprocessor._keep(stack);
	              if (_keep3) {
	                defines.set(_identifier2, "true");
	              }
	              Debug.trace(TRACEID, keyword + ": [" + _identifier2 + "] " + (_keep3 ? "" : "IGNORED"));
	            }

	            // continue on the next line
	            KEYWORD.lastIndex = extension.index + extension[0].length;
	            break;
	          }
	        case 'ifdef':
	        case 'ifndef':
	        case 'if':
	          {
	            // read the if line
	            IF.lastIndex = match.index;
	            var iff = IF.exec(source);
	            var _expression = iff[2];

	            // evaluate expression
	            var evaluated = Preprocessor.evaluate(_expression, defines);
	            error || (error = evaluated.error);
	            var result = evaluated.result;
	            if (keyword === 'ifndef') {
	              result = !result;
	            }

	            // add info to the stack (to be handled later)
	            stack.push({
	              anyKeep: result,
	              // true if any branch was already accepted
	              keep: result,
	              // true if this branch is being taken
	              start: match.index,
	              // start index if IF line
	              end: IF.lastIndex // end index of IF line
	            });

	            Debug.trace(TRACEID, keyword + ": [" + _expression + "] => " + result);

	            // continue on the next line
	            KEYWORD.lastIndex = iff.index + iff[0].length;
	            break;
	          }
	        case 'endif':
	        case 'else':
	        case 'elif':
	          {
	            // match the endif
	            ENDIF.lastIndex = match.index;
	            var endif = ENDIF.exec(source);
	            var blockInfo = stack.pop();

	            // code between if and endif
	            var blockCode = blockInfo.keep ? source.substring(blockInfo.end, match.index) : "";
	            Debug.trace(TRACEID, keyword + ": [previous block] => " + (blockCode !== ""));

	            // cut out the IF and ENDIF lines, leave block if required
	            source = source.substring(0, blockInfo.start) + blockCode + source.substring(ENDIF.lastIndex);
	            KEYWORD.lastIndex = blockInfo.start + blockCode.length;

	            // handle else if
	            var endifCommand = endif[1];
	            if (endifCommand === 'else' || endifCommand === 'elif') {
	              // if any branch was already accepted, all else branches need to fail regardless of the result
	              var _result = false;
	              if (!blockInfo.anyKeep) {
	                if (endifCommand === 'else') {
	                  _result = !blockInfo.keep;
	                } else {
	                  var _evaluated = Preprocessor.evaluate(endif[2], defines);
	                  _result = _evaluated.result;
	                  error || (error = _evaluated.error);
	                }
	              }

	              // add back to stack
	              stack.push({
	                anyKeep: blockInfo.anyKeep || _result,
	                keep: _result,
	                start: KEYWORD.lastIndex,
	                end: KEYWORD.lastIndex
	              });
	              Debug.trace(TRACEID, keyword + ": [" + endif[2] + "] => " + _result);
	            }
	            break;
	          }
	      }
	    }
	    if (error) {
	      console.warn("Failed to preprocess shader: ", {
	        source: originalSource
	      });
	      return originalSource;
	    }
	    return source;
	  }

	  // function returns true if the evaluation is inside keep branches
	  ;
	  Preprocessor._keep = function _keep(stack) {
	    for (var i = 0; i < stack.length; i++) {
	      if (!stack[i].keep) return false;
	    }
	    return true;
	  }

	  /**
	   * Very simple expression evaluation, handles cases:
	   * expression
	   * defined(expression)
	   * !defined(expression)
	   *
	   * But does not handle more complex cases, which would require more complex system:
	   * defined(A) || defined(B)
	   */;
	  Preprocessor.evaluate = function evaluate(expression, defines) {
	    var correct = INVALID.exec(expression) === null;
	    Debug.assert(correct, "Resolving expression like this is not supported: " + expression);

	    // if the format is defined(expression), extract expression
	    var invert = false;
	    var defined = DEFINED.exec(expression);
	    if (defined) {
	      invert = defined[1] === '!';
	      expression = defined[2];
	    }

	    // test if expression define exists
	    expression = expression.trim();
	    var exists = defines.has(expression);

	    // handle inversion
	    if (invert) {
	      exists = !exists;
	    }
	    return {
	      result: exists,
	      error: !correct
	    };
	  };
	  return Preprocessor;
	}();

	var id$7 = 0;

	/**
	 * A shader is a program that is responsible for rendering graphical primitives on a device's
	 * graphics processor. The shader is generated from a shader definition. This shader definition
	 * specifies the code for processing vertices and fragments processed by the GPU. The language of
	 * the code is GLSL (or more specifically ESSL, the OpenGL ES Shading Language). The shader
	 * definition also describes how the PlayCanvas engine should map vertex buffer elements onto the
	 * attributes specified in the vertex shader code.
	 *
	 * @category Graphics
	 */
	var Shader = /*#__PURE__*/function () {
	  /**
	   * Creates a new Shader instance.
	   *
	   * Consider {@link createShaderFromCode} as a simpler and more powerful way to create
	   * a shader.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./graphics-device.js').GraphicsDevice} graphicsDevice - The graphics device
	   * used to manage this shader.
	   * @param {object} definition - The shader definition from which to build the shader.
	   * @param {string} [definition.name] - The name of the shader.
	   * @param {Object<string, string>} [definition.attributes] - Object detailing the mapping of
	   * vertex shader attribute names to semantics SEMANTIC_*. This enables the engine to match
	   * vertex buffer data as inputs to the shader. When not specified, rendering without
	   * vertex buffer is assumed.
	   * @param {string} [definition.vshader] - Vertex shader source (GLSL code). Optional when
	   * compute shader is specified.
	   * @param {string} [definition.fshader] - Fragment shader source (GLSL code). Optional when
	   * useTransformFeedback or compute shader is specified.
	   * @param {string} [definition.cshader] - Compute shader source (WGSL code). Only supported on
	   * WebGPU platform.
	   * @param {boolean} [definition.useTransformFeedback] - Specifies that this shader outputs
	   * post-VS data to a buffer.
	   * @param {string | string[]} [definition.fragmentOutputTypes] - Fragment shader output types,
	   * which default to vec4. Passing a string will set the output type for all color attachments.
	   * Passing an array will set the output type for each color attachment.
	   * @param {string} [definition.shaderLanguage] - Specifies the shader language of vertex and
	   * fragment shaders. Defaults to {@link SHADERLANGUAGE_GLSL}.
	   * @example
	   * // Create a shader that renders primitives with a solid red color
	   *
	   * // Vertex shader
	   * const vshader = `
	   * attribute vec3 aPosition;
	   *
	   * void main(void) {
	   *     gl_Position = vec4(aPosition, 1.0);
	   * }
	   * `;
	   *
	   * // Fragment shader
	   * const fshader = `
	   * precision ${graphicsDevice.precision} float;
	   *
	   * void main(void) {
	   *     gl_FragColor = vec4(1.0, 0.0, 0.0, 1.0);
	   * }
	   * `;
	   *
	   * const shaderDefinition = {
	   *     attributes: {
	   *         aPosition: pc.SEMANTIC_POSITION
	   *     },
	   *     vshader,
	   *     fshader
	   * };
	   *
	   * const shader = new pc.Shader(graphicsDevice, shaderDefinition);
	   */
	  function Shader(graphicsDevice, definition) {
	    /**
	     * Format of the uniform buffer for mesh bind group.
	     *
	     * @type {new Function("modulePath", "return import(modulePath)")('./uniform-buffer-format.js').UniformBufferFormat}
	     * @ignore
	     */
	    this.meshUniformBufferFormat = void 0;
	    /**
	     * Format of the bind group for the mesh bind group.
	     *
	     * @type {new Function("modulePath", "return import(modulePath)")('./bind-group-format.js').BindGroupFormat}
	     * @ignore
	     */
	    this.meshBindGroupFormat = void 0;
	    this.id = id$7++;
	    this.device = graphicsDevice;
	    this.definition = definition;
	    this.name = definition.name || 'Untitled';
	    this.init();
	    if (definition.cshader) {
	      Debug.assert(graphicsDevice.supportsCompute, 'Compute shaders are not supported on this device.');
	      Debug.assert(!definition.vshader && !definition.fshader, 'Vertex and fragment shaders are not supported when creating a compute shader.');
	    } else {
	      Debug.assert(definition.vshader, 'No vertex shader has been specified when creating a shader.');
	      Debug.assert(definition.fshader, 'No fragment shader has been specified when creating a shader.');

	      // pre-process shader sources
	      definition.vshader = Preprocessor.run(definition.vshader);

	      // Strip unused color attachments from fragment shader.
	      // Note: this is only needed for iOS 15 on WebGL2 where there seems to be a bug where color attachments that are not
	      // written to generate metal linking errors. This is fixed on iOS 16, and iOS 14 does not support WebGL2.
	      var stripUnusedColorAttachments = graphicsDevice.isWebGL2 && (platform.name === 'osx' || platform.name === 'ios');
	      definition.fshader = Preprocessor.run(definition.fshader, stripUnusedColorAttachments);
	    }
	    this.impl = graphicsDevice.createShaderImpl(this);
	    Debug.trace(TRACEID_SHADER_ALLOC, "Alloc: " + this.label + ", stack: " + DebugGraphics.toString(), {
	      instance: this
	    });
	  }

	  /**
	   * Initialize a shader back to its default state.
	   *
	   * @private
	   */
	  var _proto = Shader.prototype;
	  _proto.init = function init() {
	    this.ready = false;
	    this.failed = false;
	  }

	  /** @ignore */;
	  /**
	   * Frees resources associated with this shader.
	   */
	  _proto.destroy = function destroy() {
	    Debug.trace(TRACEID_SHADER_ALLOC, "DeAlloc: Id " + this.id + " " + this.name);
	    this.device.onDestroyShader(this);
	    this.impl.destroy(this);
	  }

	  /**
	   * Called when the WebGL context was lost. It releases all context related resources.
	   *
	   * @ignore
	   */;
	  _proto.loseContext = function loseContext() {
	    this.init();
	    this.impl.loseContext();
	  }

	  /** @ignore */;
	  _proto.restoreContext = function restoreContext() {
	    this.impl.restoreContext(this.device, this);
	  };
	  _createClass(Shader, [{
	    key: "label",
	    get: function get() {
	      return "Shader Id " + this.id + " " + this.name;
	    }
	  }]);
	  return Shader;
	}();

	var id$6 = 0;

	/**
	 * A bind group represents an collection of {@link UniformBuffer} and {@link Texture} instance,
	 * which can be bind on a GPU for rendering.
	 *
	 * @ignore
	 */
	var BindGroup = /*#__PURE__*/function () {
	  /**
	   * Create a new Bind Group.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./graphics-device.js').GraphicsDevice} graphicsDevice - The graphics device
	   * used to manage this uniform buffer.
	   * @param {new Function("modulePath", "return import(modulePath)")('./bind-group-format.js').BindGroupFormat} format - Format of the bind group.
	   * @param {new Function("modulePath", "return import(modulePath)")('./uniform-buffer.js').UniformBuffer} [defaultUniformBuffer] - The default
	   * uniform buffer. Typically a bind group only has a single uniform buffer, and this allows
	   * easier access.
	   */
	  function BindGroup(graphicsDevice, format, defaultUniformBuffer) {
	    /**
	     * A render version the bind group was last updated on.
	     *
	     * @type {number}
	     * @ignore
	     */
	    this.renderVersionUpdated = -1;
	    /** @type {new Function("modulePath", "return import(modulePath)")('./uniform-buffer.js').UniformBuffer[]} */
	    this.uniformBuffers = void 0;
	    /**
	     * An array of offsets for each uniform buffer in the bind group. This is the offset in the
	     * buffer where the uniform buffer data starts.
	     *
	     * @type {number[]}
	     */
	    this.uniformBufferOffsets = [];
	    this.id = id$6++;
	    this.device = graphicsDevice;
	    this.format = format;
	    this.dirty = true;
	    this.impl = graphicsDevice.createBindGroupImpl(this);
	    this.textures = [];
	    this.storageTextures = [];
	    this.uniformBuffers = [];

	    /** @type {new Function("modulePath", "return import(modulePath)")('./uniform-buffer.js').UniformBuffer} */
	    this.defaultUniformBuffer = defaultUniformBuffer;
	    if (defaultUniformBuffer) {
	      this.setUniformBuffer(UNIFORM_BUFFER_DEFAULT_SLOT_NAME, defaultUniformBuffer);
	    }
	    Debug.trace(TRACEID_BINDGROUP_ALLOC, "Alloc: Id " + this.id, this, format);
	  }

	  /**
	   * Frees resources associated with this bind group.
	   */
	  var _proto = BindGroup.prototype;
	  _proto.destroy = function destroy() {
	    this.impl.destroy();
	    this.impl = null;
	    this.format = null;
	    this.defaultUniformBuffer = null;
	  }

	  /**
	   * Assign a uniform buffer to a slot.
	   *
	   * @param {string} name - The name of the uniform buffer slot
	   * @param {new Function("modulePath", "return import(modulePath)")('./uniform-buffer.js').UniformBuffer} uniformBuffer - The Uniform buffer to
	   * assign to the slot.
	   */;
	  _proto.setUniformBuffer = function setUniformBuffer(name, uniformBuffer) {
	    var index = this.format.bufferFormatsMap.get(name);
	    Debug.assert(index !== undefined, "Setting a uniform [" + name + "] on a bind group with id " + this.id + " which does not contain in, while rendering [" + DebugGraphics.toString() + "]", this);
	    if (this.uniformBuffers[index] !== uniformBuffer) {
	      this.uniformBuffers[index] = uniformBuffer;
	      this.dirty = true;
	    }
	  }

	  /**
	   * Assign a texture to a named slot.
	   *
	   * @param {string} name - The name of the texture slot.
	   * @param {new Function("modulePath", "return import(modulePath)")('./texture.js').Texture} texture - Texture to assign to the slot.
	   */;
	  _proto.setTexture = function setTexture(name, texture) {
	    var index = this.format.textureFormatsMap.get(name);
	    Debug.assert(index !== undefined, "Setting a texture [" + name + "] on a bind group with id: " + this.id + " which does not contain in, while rendering [" + DebugGraphics.toString() + "]", this);
	    if (this.textures[index] !== texture) {
	      this.textures[index] = texture;
	      this.dirty = true;
	    } else if (this.renderVersionUpdated < texture.renderVersionDirty) {
	      // if the texture properties have changed
	      this.dirty = true;
	    }
	  }

	  /**
	   * Assign a storage texture to a named slot.
	   *
	   * @param {string} name - The name of the texture slot.
	   * @param {new Function("modulePath", "return import(modulePath)")('./texture.js').Texture} texture - Texture to assign to the slot.
	   */;
	  _proto.setStorageTexture = function setStorageTexture(name, texture) {
	    var index = this.format.storageTextureFormatsMap.get(name);
	    Debug.assert(index !== undefined, "Setting a storage texture [" + name + "] on a bind group with id: " + this.id + " which does not contain in, while rendering [" + DebugGraphics.toString() + "]", this);
	    if (this.storageTextures[index] !== texture) {
	      this.storageTextures[index] = texture;
	      this.dirty = true;
	    } else if (this.renderVersionUpdated < texture.renderVersionDirty) {
	      // if the texture properties have changed
	      this.dirty = true;
	    }
	  }

	  /**
	   * Applies any changes made to the bind group's properties.
	   */;
	  _proto.update = function update() {
	    // TODO: implement faster version of this, which does not call SetTexture, which does a map lookup
	    var _this$format = this.format,
	      textureFormats = _this$format.textureFormats,
	      storageTextureFormats = _this$format.storageTextureFormats;
	    for (var i = 0; i < textureFormats.length; i++) {
	      var textureFormat = textureFormats[i];
	      var value = textureFormat.scopeId.value;
	      Debug.assert(value, "Value was not set when assigning texture slot [" + textureFormat.name + "] to a bind group, while rendering [" + DebugGraphics.toString() + "]", this);
	      this.setTexture(textureFormat.name, value);
	    }
	    for (var _i = 0; _i < storageTextureFormats.length; _i++) {
	      var storageTextureFormat = storageTextureFormats[_i];
	      var _value = storageTextureFormat.scopeId.value;
	      Debug.assert(_value, "Value was not set when assigning storage texture slot [" + storageTextureFormat.name + "] to a bind group, while rendering [" + DebugGraphics.toString() + "]", this);
	      this.setStorageTexture(storageTextureFormat.name, _value);
	    }

	    // update uniform buffer offsets
	    this.uniformBufferOffsets.length = this.uniformBuffers.length;
	    for (var _i2 = 0; _i2 < this.uniformBuffers.length; _i2++) {
	      var uniformBuffer = this.uniformBuffers[_i2];

	      // offset
	      this.uniformBufferOffsets[_i2] = uniformBuffer.offset;

	      // test if any of the uniform buffers have changed (not their content, but the buffer container itself)
	      if (this.renderVersionUpdated < uniformBuffer.renderVersionDirty) {
	        this.dirty = true;
	      }
	    }
	    if (this.dirty) {
	      this.dirty = false;
	      this.renderVersionUpdated = this.device.renderVersion;
	      this.impl.update(this);
	    }
	  };
	  return BindGroup;
	}();

	/**
	 * A base class representing a single per platform buffer.
	 *
	 * @ignore
	 */
	var DynamicBuffer = function DynamicBuffer(device) {
	  /** @type {new Function("modulePath", "return import(modulePath)")('./graphics-device.js').GraphicsDevice} */
	  this.device = void 0;
	  this.device = device;
	};
	/**
	 * A container for storing the used areas of a pair of staging and gpu buffers.
	 *
	 * @ignore
	 */
	var UsedBuffer = function UsedBuffer() {
	  /** @type {DynamicBuffer} */
	  this.gpuBuffer = void 0;
	  /** @type {DynamicBuffer} */
	  this.stagingBuffer = void 0;
	  /**
	   * The beginning position of the used area that needs to be copied from staging to to the GPU
	   * buffer.
	   *
	   * @type {number}
	   */
	  this.offset = void 0;
	  /**
	   * Used byte size of the buffer, from the offset.
	   *
	   * @type {number}
	   */
	  this.size = void 0;
	};
	/**
	 * A container for storing the return values of an allocation function.
	 *
	 * @ignore
	 */
	var DynamicBufferAllocation = function DynamicBufferAllocation() {
	  /**
	   * The storage access to the allocated data in the staging buffer.
	   *
	   * @type {Int32Array}
	   */
	  this.storage = void 0;
	  /**
	   * The gpu buffer this allocation will be copied to.
	   *
	   * @type {DynamicBuffer}
	   */
	  this.gpuBuffer = void 0;
	  /**
	   * Offset in the gpuBuffer where the data will be copied to.
	   *
	   * @type {number}
	   */
	  this.offset = void 0;
	};
	/**
	 * The DynamicBuffers class provides a dynamic memory allocation system for uniform buffer data,
	 * particularly for non-persistent uniform buffers. This class utilizes a bump allocator to
	 * efficiently allocate aligned memory space from a set of large buffers managed internally. To
	 * utilize this system, the user writes data to CPU-accessible staging buffers. When submitting
	 * command buffers that require these buffers, the system automatically uploads the data to the GPU
	 * buffers. This approach ensures efficient memory management and smooth data transfer between the
	 * CPU and GPU.
	 *
	 * @ignore
	 */
	var DynamicBuffers = /*#__PURE__*/function () {
	  /**
	   * Create the system of dynamic buffers.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./graphics-device.js').GraphicsDevice} device - The graphics device.
	   * @param {number} bufferSize - The size of the underlying large buffers.
	   * @param {number} bufferAlignment - Alignment of each allocation.
	   */
	  function DynamicBuffers(device, bufferSize, bufferAlignment) {
	    /**
	     * Allocation size of the underlying buffers.
	     *
	     * @type {number}
	     */
	    this.bufferSize = void 0;
	    /**
	     * Internally allocated gpu buffers.
	     *
	     * @type {DynamicBuffer[]}
	     */
	    this.gpuBuffers = [];
	    /**
	     * Internally allocated staging buffers (CPU writable)
	     *
	     * @type {DynamicBuffer[]}
	     */
	    this.stagingBuffers = [];
	    /**
	     * @type {UsedBuffer[]}
	     */
	    this.usedBuffers = [];
	    /**
	     * @type {UsedBuffer}
	     */
	    this.activeBuffer = null;
	    this.device = device;
	    this.bufferSize = bufferSize;
	    this.bufferAlignment = bufferAlignment;
	  }

	  /**
	   * Destroy the system of dynamic buffers.
	   */
	  var _proto = DynamicBuffers.prototype;
	  _proto.destroy = function destroy() {
	    var _this = this;
	    this.gpuBuffers.forEach(function (gpuBuffer) {
	      gpuBuffer.destroy(_this.device);
	    });
	    this.gpuBuffers = null;
	    this.stagingBuffers.forEach(function (stagingBuffer) {
	      stagingBuffer.destroy(_this.device);
	    });
	    this.stagingBuffers = null;
	    this.usedBuffers = null;
	    this.activeBuffer = null;
	  }

	  /**
	   * Allocate an aligned space of the given size from a dynamic buffer.
	   *
	   * @param {DynamicBufferAllocation} allocation - The allocation info to fill.
	   * @param {number} size - The size of the allocation.
	   */;
	  _proto.alloc = function alloc(allocation, size) {
	    // if we have active buffer without enough space
	    if (this.activeBuffer) {
	      var _alignedStart = math.roundUp(this.activeBuffer.size, this.bufferAlignment);
	      var space = this.bufferSize - _alignedStart;
	      if (space < size) {
	        // we're done with this buffer, schedule it for submit
	        this.scheduleSubmit();
	      }
	    }

	    // if we don't have an active buffer, allocate new one
	    if (!this.activeBuffer) {
	      // gpu buffer
	      var gpuBuffer = this.gpuBuffers.pop();
	      if (!gpuBuffer) {
	        gpuBuffer = this.createBuffer(this.device, this.bufferSize, false);
	      }

	      // staging buffer
	      var stagingBuffer = this.stagingBuffers.pop();
	      if (!stagingBuffer) {
	        stagingBuffer = this.createBuffer(this.device, this.bufferSize, true);
	      }
	      this.activeBuffer = new UsedBuffer();
	      this.activeBuffer.stagingBuffer = stagingBuffer;
	      this.activeBuffer.gpuBuffer = gpuBuffer;
	      this.activeBuffer.offset = 0;
	      this.activeBuffer.size = 0;
	    }

	    // allocate from active buffer
	    var activeBuffer = this.activeBuffer;
	    var alignedStart = math.roundUp(activeBuffer.size, this.bufferAlignment);
	    Debug.assert(alignedStart + size <= this.bufferSize, "The allocation size of " + size + " is larger than the buffer size of " + this.bufferSize);
	    allocation.gpuBuffer = activeBuffer.gpuBuffer;
	    allocation.offset = alignedStart;
	    allocation.storage = activeBuffer.stagingBuffer.alloc(alignedStart, size);

	    // take the allocation from the buffer
	    activeBuffer.size = alignedStart + size;
	  };
	  _proto.scheduleSubmit = function scheduleSubmit() {
	    if (this.activeBuffer) {
	      this.usedBuffers.push(this.activeBuffer);
	      this.activeBuffer = null;
	    }
	  };
	  _proto.submit = function submit() {
	    // schedule currently active buffer for submit
	    this.scheduleSubmit();
	  };
	  return DynamicBuffers;
	}();

	// Uniform buffer set functions - only implemented for types for which the default
	// array to buffer copy does not work, or could be slower.
	var _updateFunctions = [];
	_updateFunctions[UNIFORMTYPE_FLOAT] = function (uniformBuffer, value, offset) {
	  var dst = uniformBuffer.storageFloat32;
	  dst[offset] = value;
	};
	_updateFunctions[UNIFORMTYPE_VEC2] = function (uniformBuffer, value, offset) {
	  var dst = uniformBuffer.storageFloat32;
	  dst[offset] = value[0];
	  dst[offset + 1] = value[1];
	};
	_updateFunctions[UNIFORMTYPE_VEC3] = function (uniformBuffer, value, offset) {
	  var dst = uniformBuffer.storageFloat32;
	  dst[offset] = value[0];
	  dst[offset + 1] = value[1];
	  dst[offset + 2] = value[2];
	};
	_updateFunctions[UNIFORMTYPE_VEC4] = function (uniformBuffer, value, offset) {
	  var dst = uniformBuffer.storageFloat32;
	  dst[offset] = value[0];
	  dst[offset + 1] = value[1];
	  dst[offset + 2] = value[2];
	  dst[offset + 3] = value[3];
	};
	_updateFunctions[UNIFORMTYPE_INT] = function (uniformBuffer, value, offset) {
	  var dst = uniformBuffer.storageInt32;
	  dst[offset] = value;
	};
	_updateFunctions[UNIFORMTYPE_IVEC2] = function (uniformBuffer, value, offset) {
	  var dst = uniformBuffer.storageInt32;
	  dst[offset] = value[0];
	  dst[offset + 1] = value[1];
	};
	_updateFunctions[UNIFORMTYPE_IVEC3] = function (uniformBuffer, value, offset) {
	  var dst = uniformBuffer.storageInt32;
	  dst[offset] = value[0];
	  dst[offset + 1] = value[1];
	  dst[offset + 2] = value[2];
	};
	_updateFunctions[UNIFORMTYPE_IVEC4] = function (uniformBuffer, value, offset) {
	  var dst = uniformBuffer.storageInt32;
	  dst[offset] = value[0];
	  dst[offset + 1] = value[1];
	  dst[offset + 2] = value[2];
	  dst[offset + 3] = value[3];
	};

	// convert from continuous array to vec2[3] with padding to vec4[2]
	_updateFunctions[UNIFORMTYPE_MAT2] = function (uniformBuffer, value, offset) {
	  var dst = uniformBuffer.storageFloat32;
	  dst[offset] = value[0];
	  dst[offset + 1] = value[1];
	  dst[offset + 4] = value[2];
	  dst[offset + 5] = value[3];
	  dst[offset + 8] = value[4];
	  dst[offset + 9] = value[5];
	};

	// convert from continuous array to vec3[3] with padding to vec4[3]
	_updateFunctions[UNIFORMTYPE_MAT3] = function (uniformBuffer, value, offset) {
	  var dst = uniformBuffer.storageFloat32;
	  dst[offset] = value[0];
	  dst[offset + 1] = value[1];
	  dst[offset + 2] = value[2];
	  dst[offset + 4] = value[3];
	  dst[offset + 5] = value[4];
	  dst[offset + 6] = value[5];
	  dst[offset + 8] = value[6];
	  dst[offset + 9] = value[7];
	  dst[offset + 10] = value[8];
	};
	_updateFunctions[UNIFORMTYPE_FLOATARRAY] = function (uniformBuffer, value, offset, count) {
	  var dst = uniformBuffer.storageFloat32;
	  for (var i = 0; i < count; i++) {
	    dst[offset + i * 4] = value[i];
	  }
	};
	_updateFunctions[UNIFORMTYPE_VEC2ARRAY] = function (uniformBuffer, value, offset, count) {
	  var dst = uniformBuffer.storageFloat32;
	  for (var i = 0; i < count; i++) {
	    dst[offset + i * 4] = value[i * 2];
	    dst[offset + i * 4 + 1] = value[i * 2 + 1];
	  }
	};
	_updateFunctions[UNIFORMTYPE_VEC3ARRAY] = function (uniformBuffer, value, offset, count) {
	  var dst = uniformBuffer.storageFloat32;
	  for (var i = 0; i < count; i++) {
	    dst[offset + i * 4] = value[i * 3];
	    dst[offset + i * 4 + 1] = value[i * 3 + 1];
	    dst[offset + i * 4 + 2] = value[i * 3 + 2];
	  }
	};
	_updateFunctions[UNIFORMTYPE_UINT] = function (uniformBuffer, value, offset, count) {
	  var dst = uniformBuffer.storageUint32;
	  dst[offset] = value;
	};
	_updateFunctions[UNIFORMTYPE_UVEC2] = function (uniformBuffer, value, offset, count) {
	  var dst = uniformBuffer.storageUint32;
	  dst[offset] = value[0];
	  dst[offset + 1] = value[1];
	};
	_updateFunctions[UNIFORMTYPE_UVEC3] = function (uniformBuffer, value, offset, count) {
	  var dst = uniformBuffer.storageUint32;
	  dst[offset] = value[0];
	  dst[offset + 1] = value[1];
	  dst[offset + 2] = value[2];
	};
	_updateFunctions[UNIFORMTYPE_UVEC4] = function (uniformBuffer, value, offset, count) {
	  var dst = uniformBuffer.storageUint32;
	  dst[offset] = value[0];
	  dst[offset + 1] = value[1];
	  dst[offset + 2] = value[2];
	  dst[offset + 3] = value[3];
	};
	_updateFunctions[UNIFORMTYPE_INTARRAY] = function (uniformBuffer, value, offset, count) {
	  var dst = uniformBuffer.storageInt32;
	  for (var i = 0; i < count; i++) {
	    dst[offset + i * 4] = value[i];
	  }
	};
	_updateFunctions[UNIFORMTYPE_BOOLARRAY] = _updateFunctions[UNIFORMTYPE_INTARRAY];
	_updateFunctions[UNIFORMTYPE_UINTARRAY] = function (uniformBuffer, value, offset, count) {
	  var dst = uniformBuffer.storageUint32;
	  for (var i = 0; i < count; i++) {
	    dst[offset + i * 4] = value[i];
	  }
	};
	_updateFunctions[UNIFORMTYPE_IVEC2ARRAY] = function (uniformBuffer, value, offset, count) {
	  var dst = uniformBuffer.storageInt32;
	  for (var i = 0; i < count; i++) {
	    dst[offset + i * 4] = value[i * 2];
	    dst[offset + i * 4 + 1] = value[i * 2 + 1];
	  }
	};
	_updateFunctions[UNIFORMTYPE_BVEC2ARRAY] = _updateFunctions[UNIFORMTYPE_IVEC2ARRAY];
	_updateFunctions[UNIFORMTYPE_UVEC2ARRAY] = function (uniformBuffer, value, offset, count) {
	  var dst = uniformBuffer.storageUint32;
	  for (var i = 0; i < count; i++) {
	    dst[offset + i * 4] = value[i * 2];
	    dst[offset + i * 4 + 1] = value[i * 2 + 1];
	  }
	};
	_updateFunctions[UNIFORMTYPE_IVEC3ARRAY] = function (uniformBuffer, value, offset, count) {
	  var dst = uniformBuffer.storageInt32;
	  for (var i = 0; i < count; i++) {
	    dst[offset + i * 4] = value[i * 3];
	    dst[offset + i * 4 + 1] = value[i * 3 + 1];
	    dst[offset + i * 4 + 2] = value[i * 3 + 2];
	  }
	};
	_updateFunctions[UNIFORMTYPE_BVEC3ARRAY] = _updateFunctions[UNIFORMTYPE_IVEC3ARRAY];
	_updateFunctions[UNIFORMTYPE_UVEC3ARRAY] = function (uniformBuffer, value, offset, count) {
	  var dst = uniformBuffer.storageUint32;
	  for (var i = 0; i < count; i++) {
	    dst[offset + i * 4] = value[i * 3];
	    dst[offset + i * 4 + 1] = value[i * 3 + 1];
	    dst[offset + i * 4 + 2] = value[i * 3 + 2];
	  }
	};

	/**
	 * A uniform buffer represents a GPU memory buffer storing the uniforms.
	 *
	 * @ignore
	 */
	var UniformBuffer = /*#__PURE__*/function () {
	  /**
	   * Create a new UniformBuffer instance.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./graphics-device.js').GraphicsDevice} graphicsDevice - The graphics device
	   * used to manage this uniform buffer.
	   * @param {new Function("modulePath", "return import(modulePath)")('./uniform-buffer-format.js').UniformBufferFormat} format - Format of the
	   * uniform buffer.
	   * @param {boolean} [persistent] - Whether the buffer is persistent. Defaults to true.
	   */
	  function UniformBuffer(graphicsDevice, format, persistent) {
	    if (persistent === void 0) {
	      persistent = true;
	    }
	    this.device = void 0;
	    /** @type {boolean} */
	    this.persistent = void 0;
	    /** @type {DynamicBufferAllocation} */
	    this.allocation = void 0;
	    /** @type {Float32Array} */
	    this.storageFloat32 = void 0;
	    /** @type {Int32Array} */
	    this.storageInt32 = void 0;
	    /** @type {Uint32Array} */
	    this.storageUint32 = void 0;
	    /**
	     * A render version used to track the last time the properties requiring bind group to be
	     * updated were changed.
	     *
	     * @type {number}
	     */
	    this.renderVersionDirty = 0;
	    this.device = graphicsDevice;
	    this.format = format;
	    this.persistent = persistent;
	    Debug.assert(format);
	    if (persistent) {
	      this.impl = graphicsDevice.createUniformBufferImpl(this);
	      var storage = new ArrayBuffer(format.byteSize);
	      this.assignStorage(new Int32Array(storage));
	      graphicsDevice._vram.ub += this.format.byteSize;

	      // TODO: register with the device and handle lost context
	      // this.device.buffers.push(this);
	    } else {
	      this.allocation = new DynamicBufferAllocation();
	    }
	  }

	  /**
	   * Frees resources associated with this uniform buffer.
	   */
	  var _proto = UniformBuffer.prototype;
	  _proto.destroy = function destroy() {
	    if (this.persistent) {
	      // stop tracking the vertex buffer
	      // TODO: remove the buffer from the list on the device (lost context handling)
	      var device = this.device;
	      this.impl.destroy(device);
	      device._vram.ub -= this.format.byteSize;
	    }
	  };
	  /**
	   * Assign a storage to this uniform buffer.
	   *
	   * @param {Int32Array} storage - The storage to assign to this uniform buffer.
	   */
	  _proto.assignStorage = function assignStorage(storage) {
	    this.storageInt32 = storage;
	    this.storageUint32 = new Uint32Array(storage.buffer, storage.byteOffset, storage.byteLength / 4);
	    this.storageFloat32 = new Float32Array(storage.buffer, storage.byteOffset, storage.byteLength / 4);
	  }

	  /**
	   * Called when the rendering context was lost. It releases all context related resources.
	   *
	   * @ignore
	   */;
	  _proto.loseContext = function loseContext() {
	    var _this$impl;
	    (_this$impl = this.impl) == null || _this$impl.loseContext();
	  }

	  /**
	   * Assign a value to the uniform specified by its format. This is the fast version of assigning
	   * a value to a uniform, avoiding any lookups.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./uniform-buffer-format.js').UniformFormat} uniformFormat - The format of
	   * the uniform.
	   */;
	  _proto.setUniform = function setUniform(uniformFormat) {
	    Debug.assert(uniformFormat);
	    var offset = uniformFormat.offset;
	    var value = uniformFormat.scopeId.value;
	    if (value !== null && value !== undefined) {
	      var updateFunction = _updateFunctions[uniformFormat.updateType];
	      if (updateFunction) {
	        updateFunction(this, value, offset, uniformFormat.count);
	      } else {
	        this.storageFloat32.set(value, offset);
	      }
	    } else {
	      Debug.warnOnce("Value was not set when assigning to uniform [" + uniformFormat.name + "]" + (", expected type " + uniformTypeToName[uniformFormat.type] + " while rendering " + DebugGraphics.toString()));
	    }
	  }

	  /**
	   * Assign a value to the uniform specified by name.
	   *
	   * @param {string} name - The name of the uniform.
	   */;
	  _proto.set = function set(name) {
	    var uniformFormat = this.format.map.get(name);
	    Debug.assert(uniformFormat, "Uniform name [" + name + "] is not part of the Uniform buffer.");
	    if (uniformFormat) {
	      this.setUniform(uniformFormat);
	    }
	  };
	  _proto.update = function update() {
	    var persistent = this.persistent;
	    if (!persistent) {
	      // allocate memory from dynamic buffer for this frame
	      var allocation = this.allocation;
	      var oldGpuBuffer = allocation.gpuBuffer;
	      this.device.dynamicBuffers.alloc(allocation, this.format.byteSize);
	      this.assignStorage(allocation.storage);

	      // buffer has changed, update the render version to force bind group to be updated
	      if (oldGpuBuffer !== allocation.gpuBuffer) {
	        this.renderVersionDirty = this.device.renderVersion;
	      }
	    }

	    // set new values
	    var uniforms = this.format.uniforms;
	    for (var i = 0; i < uniforms.length; i++) {
	      this.setUniform(uniforms[i]);
	    }
	    if (persistent) {
	      // Upload the new data
	      this.impl.unlock(this);
	    } else {
	      this.storageFloat32 = null;
	      this.storageInt32 = null;
	    }
	  };
	  _createClass(UniformBuffer, [{
	    key: "offset",
	    get: function get() {
	      return this.persistent ? 0 : this.allocation.offset;
	    }
	  }]);
	  return UniformBuffer;
	}();

	var primitive = {
	  type: PRIMITIVE_TRISTRIP,
	  base: 0,
	  count: 4,
	  indexed: false
	};

	/**
	 * A WebGPU helper class implementing a viewport clear operation. When rendering to a texture,
	 * the whole surface can be cleared using loadOp, but if only a viewport needs to be cleared, or if
	 * it needs to be cleared later during the rendering, this need to be archieved by rendering a quad.
	 * This class renders a full-screen quad, and expects the viewport / scissor to be set up to clip
	 * it to only required area.
	 *
	 * @ignore
	 */
	var WebgpuClearRenderer = /*#__PURE__*/function () {
	  function WebgpuClearRenderer(device) {
	    // shader that can write out color and depth values
	    var code = "\n\n            struct ub_mesh {\n                color : vec4f,\n                depth: f32\n            }\n\n            @group(0) @binding(0) var<uniform> ubMesh : ub_mesh;\n\n            var<private> pos : array<vec2f, 4> = array<vec2f, 4>(\n                vec2(-1.0, 1.0), vec2(1.0, 1.0),\n                vec2(-1.0, -1.0), vec2(1.0, -1.0)\n            );\n\n            struct VertexOutput {\n                @builtin(position) position : vec4f\n            }\n\n            @vertex\n            fn vertexMain(@builtin(vertex_index) vertexIndex : u32) -> VertexOutput {\n                var output : VertexOutput;\n                output.position = vec4(pos[vertexIndex], ubMesh.depth, 1.0);\n                return output;\n            }\n\n            @fragment\n            fn fragmentMain() -> @location(0) vec4f {\n                return ubMesh.color;\n            }\n        ";
	    this.shader = new Shader(device, {
	      name: 'WebGPUClearRendererShader',
	      shaderLanguage: SHADERLANGUAGE_WGSL,
	      vshader: code,
	      fshader: code
	    });

	    // uniforms
	    this.uniformBuffer = new UniformBuffer(device, new UniformBufferFormat(device, [new UniformFormat('color', UNIFORMTYPE_VEC4), new UniformFormat('depth', UNIFORMTYPE_FLOAT)]), false);

	    // format of the bind group
	    var bindGroupFormat = new BindGroupFormat(device, [new BindBufferFormat(UNIFORM_BUFFER_DEFAULT_SLOT_NAME, SHADERSTAGE_VERTEX | SHADERSTAGE_FRAGMENT)]);

	    // bind group
	    this.bindGroup = new BindGroup(device, bindGroupFormat, this.uniformBuffer);
	    DebugHelper.setName(this.bindGroup, "ClearRenderer-BindGroup_" + this.bindGroup.id);

	    // uniform data
	    this.colorData = new Float32Array(4);
	    this.colorId = device.scope.resolve('color');
	    this.depthId = device.scope.resolve('depth');
	  }
	  var _proto = WebgpuClearRenderer.prototype;
	  _proto.destroy = function destroy() {
	    this.shader.destroy();
	    this.shader = null;
	    this.uniformBuffer.destroy();
	    this.uniformBuffer = null;
	    this.bindGroup.destroy();
	    this.bindGroup = null;
	  };
	  _proto.clear = function clear(device, renderTarget, options, defaultOptions) {
	    var _options$flags;
	    options = options || defaultOptions;
	    var flags = (_options$flags = options.flags) != null ? _options$flags : defaultOptions.flags;
	    if (flags !== 0) {
	      DebugGraphics.pushGpuMarker(device, 'CLEAR-RENDERER');

	      // setup clear color
	      if (flags & CLEARFLAG_COLOR && renderTarget.colorBuffer) {
	        var _options$color;
	        var color = (_options$color = options.color) != null ? _options$color : defaultOptions.color;
	        this.colorData.set(color);
	        device.setBlendState(BlendState.NOBLEND);
	      } else {
	        device.setBlendState(BlendState.NOWRITE);
	      }
	      this.colorId.setValue(this.colorData);

	      // setup depth clear
	      if (flags & CLEARFLAG_DEPTH && renderTarget.depth) {
	        var _options$depth;
	        var depth = (_options$depth = options.depth) != null ? _options$depth : defaultOptions.depth;
	        this.depthId.setValue(depth);
	        device.setDepthState(DepthState.WRITEDEPTH);
	      } else {
	        this.depthId.setValue(1);
	        device.setDepthState(DepthState.NODEPTH);
	      }

	      // setup stencil clear
	      if (flags & CLEARFLAG_STENCIL && renderTarget.stencil) {
	        Debug.warnOnce("ClearRenderer does not support stencil clear at the moment");
	      }
	      device.setCullMode(CULLFACE_NONE);

	      // render 4 vertices without vertex buffer
	      device.setShader(this.shader);
	      var bindGroup = this.bindGroup;
	      bindGroup.defaultUniformBuffer.update();
	      bindGroup.update();
	      device.setBindGroup(BINDGROUP_MESH, bindGroup);
	      device.draw(primitive);
	      DebugGraphics.popGpuMarker(device);
	    }
	  };
	  return WebgpuClearRenderer;
	}();

	/**
	 * A WebGPU helper class implementing texture mipmap generation.
	 *
	 * @ignore
	 */
	var WebgpuMipmapRenderer = /*#__PURE__*/function () {
	  function WebgpuMipmapRenderer(device) {
	    /** @type {new Function("modulePath", "return import(modulePath)")('./webgpu-graphics-device.js').WebgpuGraphicsDevice} */
	    this.device = void 0;
	    this.device = device;

	    // Shader that renders a fullscreen textured quad
	    var code = "\n \n            var<private> pos : array<vec2f, 4> = array<vec2f, 4>(\n                vec2(-1.0, 1.0), vec2(1.0, 1.0),\n                vec2(-1.0, -1.0), vec2(1.0, -1.0)\n            );\n\n            struct VertexOutput {\n                @builtin(position) position : vec4f,\n                @location(0) texCoord : vec2f\n            };\n\n            @vertex\n            fn vertexMain(@builtin(vertex_index) vertexIndex : u32) -> VertexOutput {\n              var output : VertexOutput;\n              output.texCoord = pos[vertexIndex] * vec2f(0.5, -0.5) + vec2f(0.5);\n              output.position = vec4f(pos[vertexIndex], 0, 1);\n              return output;\n            }\n\n            @group(0) @binding(0) var imgSampler : sampler;\n            @group(0) @binding(1) var img : texture_2d<f32>;\n\n            @fragment\n            fn fragmentMain(@location(0) texCoord : vec2f) -> @location(0) vec4f {\n              return textureSample(img, imgSampler, texCoord);\n            }\n        ";
	    this.shader = new Shader(device, {
	      name: 'WebGPUMipmapRendererShader',
	      shaderLanguage: SHADERLANGUAGE_WGSL,
	      vshader: code,
	      fshader: code
	    });

	    // using minified rendering, so that's the only filter mode we need to set.
	    this.minSampler = device.wgpu.createSampler({
	      minFilter: 'linear'
	    });
	  }
	  var _proto = WebgpuMipmapRenderer.prototype;
	  _proto.destroy = function destroy() {
	    this.shader.destroy();
	    this.shader = null;
	  }

	  /**
	   * Generates mipmaps for the specified WebGPU texture.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./webgpu-texture.js').WebgpuTexture} webgpuTexture - The texture to generate mipmaps for.
	   */;
	  _proto.generate = function generate(webgpuTexture) {
	    var _device$commandEncode;
	    // ignore texture with no mipmaps
	    var textureDescr = webgpuTexture.descr;
	    if (textureDescr.mipLevelCount <= 1) {
	      return;
	    }

	    // not all types are currently supported
	    if (webgpuTexture.texture.volume) {
	      Debug.warnOnce('WebGPU mipmap generation is not supported volume texture.', webgpuTexture.texture);
	      return;
	    }
	    var device = this.device;
	    var wgpu = device.wgpu;

	    /** @type {new Function("modulePath", "return import(modulePath)")('./webgpu-shader.js').WebgpuShader} */
	    var webgpuShader = this.shader.impl;
	    var pipeline = wgpu.createRenderPipeline({
	      layout: 'auto',
	      vertex: {
	        module: webgpuShader.getVertexShaderModule(),
	        entryPoint: webgpuShader.vertexEntryPoint
	      },
	      fragment: {
	        module: webgpuShader.getFragmentShaderModule(),
	        entryPoint: webgpuShader.fragmentEntryPoint,
	        targets: [{
	          format: textureDescr.format // use the same format as the texture
	        }]
	      },

	      primitive: {
	        topology: 'triangle-strip'
	      }
	    });
	    DebugHelper.setLabel(pipeline, 'RenderPipeline-MipmapRenderer');
	    var texture = webgpuTexture.texture;
	    var numFaces = texture.cubemap ? 6 : texture.array ? texture.arrayLength : 1;
	    var srcViews = [];
	    for (var face = 0; face < numFaces; face++) {
	      srcViews.push(webgpuTexture.createView({
	        dimension: '2d',
	        baseMipLevel: 0,
	        mipLevelCount: 1,
	        baseArrayLayer: face
	      }));
	    }

	    // loop through each mip level and render the previous level's contents into it.
	    var commandEncoder = (_device$commandEncode = device.commandEncoder) != null ? _device$commandEncode : wgpu.createCommandEncoder();
	    DebugHelper.setLabel(commandEncoder, 'MipmapRendererEncoder');
	    DebugGraphics.pushGpuMarker(device, 'MIPMAP-RENDERER');
	    for (var i = 1; i < textureDescr.mipLevelCount; i++) {
	      for (var _face = 0; _face < numFaces; _face++) {
	        var dstView = webgpuTexture.createView({
	          dimension: '2d',
	          baseMipLevel: i,
	          mipLevelCount: 1,
	          baseArrayLayer: _face
	        });
	        var passEncoder = commandEncoder.beginRenderPass({
	          colorAttachments: [{
	            view: dstView,
	            loadOp: 'clear',
	            storeOp: 'store'
	          }]
	        });
	        DebugHelper.setLabel(passEncoder, "MipmapRenderer-PassEncoder_" + i);
	        var bindGroup = wgpu.createBindGroup({
	          layout: pipeline.getBindGroupLayout(0),
	          entries: [{
	            binding: 0,
	            resource: this.minSampler
	          }, {
	            binding: 1,
	            resource: srcViews[_face]
	          }]
	        });
	        passEncoder.setPipeline(pipeline);
	        passEncoder.setBindGroup(0, bindGroup);
	        passEncoder.draw(4);
	        passEncoder.end();

	        // next iteration
	        srcViews[_face] = dstView;
	      }
	    }
	    DebugGraphics.popGpuMarker(device);

	    // submit the encoded commands if we created the encoder
	    if (!device.commandEncoder) {
	      var cb = commandEncoder.finish();
	      DebugHelper.setLabel(cb, 'MipmapRenderer-CommandBuffer');
	      device.addCommandBuffer(cb);
	    }

	    // clear invalidated state
	    device.pipeline = null;
	  };
	  return WebgpuMipmapRenderer;
	}();

	/**
	 * @ignore
	 */
	var WebgpuDynamicBuffer = /*#__PURE__*/function (_DynamicBuffer) {
	  _inheritsLoose(WebgpuDynamicBuffer, _DynamicBuffer);
	  function WebgpuDynamicBuffer(device, size, isStaging) {
	    var _this;
	    _this = _DynamicBuffer.call(this, device) || this;
	    /**
	     * @type {GPUBuffer}
	     * @private
	     */
	    _this.buffer = null;
	    /**
	     * CPU access over the whole buffer.
	     *
	     * @type {ArrayBuffer}
	     */
	    _this.mappedRange = null;
	    _this.buffer = device.wgpu.createBuffer({
	      size: size,
	      usage: isStaging ? GPUBufferUsage.MAP_WRITE | GPUBufferUsage.COPY_SRC : GPUBufferUsage.UNIFORM | GPUBufferUsage.COPY_DST,
	      mappedAtCreation: isStaging
	    });
	    if (isStaging) {
	      _this.onAvailable();
	    }

	    // staging buffers are not stored in vram, but add them for tracking purposes anyways
	    device._vram.ub += size;
	    DebugHelper.setLabel(_this.buffer, "DynamicBuffer-" + (isStaging ? 'Staging' : 'Gpu'));
	    return _this;
	  }
	  var _proto = WebgpuDynamicBuffer.prototype;
	  _proto.destroy = function destroy(device) {
	    device._vram.ub -= this.buffer.size;
	    this.buffer.destroy();
	    this.buffer = null;
	  }

	  /**
	   * Called when the staging buffer is mapped for writing.
	   */;
	  _proto.onAvailable = function onAvailable() {
	    // map the whole buffer
	    this.mappedRange = this.buffer.getMappedRange();
	  };
	  _proto.alloc = function alloc(offset, size) {
	    return new Int32Array(this.mappedRange, offset, size / 4);
	  };
	  return WebgpuDynamicBuffer;
	}(DynamicBuffer);

	var WebgpuDynamicBuffers = /*#__PURE__*/function (_DynamicBuffers) {
	  _inheritsLoose(WebgpuDynamicBuffers, _DynamicBuffers);
	  function WebgpuDynamicBuffers() {
	    var _this;
	    for (var _len = arguments.length, args = new Array(_len), _key = 0; _key < _len; _key++) {
	      args[_key] = arguments[_key];
	    }
	    _this = _DynamicBuffers.call.apply(_DynamicBuffers, [this].concat(args)) || this;
	    /**
	     * Staging buffers which are getting copied over to gpu buffers in the the command buffer waiting
	     * to be submitted. When those command buffers are submitted, we can mapAsync these staging
	     * buffers for reuse.
	     *
	     * @type {WebgpuDynamicBuffer[]}
	     */
	    _this.pendingStagingBuffers = [];
	    return _this;
	  }
	  var _proto = WebgpuDynamicBuffers.prototype;
	  _proto.createBuffer = function createBuffer(device, size, isStaging) {
	    return new WebgpuDynamicBuffer(device, size, isStaging);
	  }

	  /**
	   * Submit all used buffers to the device.
	   */;
	  _proto.submit = function submit() {
	    _DynamicBuffers.prototype.submit.call(this);

	    // submit all used buffers
	    var count = this.usedBuffers.length;
	    if (count) {
	      var device = this.device;
	      var gpuBuffers = this.gpuBuffers;

	      // new command encoder, as buffer copies need to be submitted before the currently recorded
	      // rendering encoder is submitted
	      var commandEncoder = device.wgpu.createCommandEncoder();
	      DebugHelper.setLabel(commandEncoder, 'DynamicBuffersSubmit');
	      DebugGraphics.pushGpuMarker(device, 'DynamicBuffersSubmit');

	      // run this loop backwards to preserve the order of buffers in gpuBuffers array
	      for (var i = count - 1; i >= 0; i--) {
	        var usedBuffer = this.usedBuffers[i];
	        var stagingBuffer = usedBuffer.stagingBuffer,
	          gpuBuffer = usedBuffer.gpuBuffer,
	          offset = usedBuffer.offset,
	          size = usedBuffer.size;

	        // unmap staging buffer (we're done writing to it on CPU)
	        var src = stagingBuffer.buffer;
	        src.unmap();

	        // schedule data copy from staging to gpu buffer
	        commandEncoder.copyBufferToBuffer(src, offset, gpuBuffer.buffer, offset, size);

	        // gpu buffer can be reused immediately
	        gpuBuffers.push(gpuBuffer);
	      }
	      DebugGraphics.popGpuMarker(device);

	      // schedule the command buffer to run before all currently scheduled command buffers
	      var cb = commandEncoder.finish();
	      DebugHelper.setLabel(cb, 'DynamicBuffers');
	      device.addCommandBuffer(cb, true);

	      // keep the used staging buffers in the pending list
	      for (var _i = 0; _i < count; _i++) {
	        var _stagingBuffer = this.usedBuffers[_i].stagingBuffer;
	        this.pendingStagingBuffers.push(_stagingBuffer);
	      }
	      this.usedBuffers.length = 0;
	    }
	  }

	  /**
	   * Called when all scheduled command buffers are submitted to the device.
	   */;
	  _proto.onCommandBuffersSubmitted = function onCommandBuffersSubmitted() {
	    var _this2 = this;
	    // map the staging buffers for write to alow them to be reused - this resolves when the CBs
	    // using them are done on the GPU
	    var count = this.pendingStagingBuffers.length;
	    if (count) {
	      var _loop = function _loop() {
	        var stagingBuffer = _this2.pendingStagingBuffers[i];
	        stagingBuffer.buffer.mapAsync(GPUMapMode.WRITE).then(function () {
	          // the buffer can be mapped after the device has been destroyed, so test for that
	          if (_this2.stagingBuffers) {
	            stagingBuffer.onAvailable();
	            _this2.stagingBuffers.push(stagingBuffer);
	          }
	        });
	      };
	      for (var i = 0; i < count; i++) {
	        _loop();
	      }
	      this.pendingStagingBuffers.length = 0;
	    }
	  };
	  return WebgpuDynamicBuffers;
	}(DynamicBuffers);

	/**
	 * Base class of a simple GPU profiler.
	 *
	 * @ignore
	 */
	var GpuProfiler = /*#__PURE__*/function () {
	  function GpuProfiler() {
	    /**
	     * Profiling slots allocated for the current frame, storing the names of the slots.
	     *
	     * @type {string[]}
	     * @ignore
	     */
	    this.frameAllocations = [];
	    /**
	     * Map of past frame allocations, indexed by renderVersion
	     *
	     * @type {Map<number, string[]>}
	     * @ignore
	     */
	    this.pastFrameAllocations = new Map();
	    /**
	     * The if enabled in the current frame.
	     * @ignore
	     */
	    this._enabled = false;
	    /**
	     * The enable request for the next frame.
	     * @ignore
	     */
	    this._enableRequest = false;
	    /**
	     * The time it took to render the last frame on GPU, or 0 if the profiler is not enabled
	     * @ignore
	     */
	    this._frameTime = 0;
	  }
	  var _proto = GpuProfiler.prototype;
	  _proto.loseContext = function loseContext() {
	    this.pastFrameAllocations.clear();
	  }

	  /**
	   * True to enable the profiler.
	   *
	   * @type {boolean}
	   */;
	  _proto.processEnableRequest = function processEnableRequest() {
	    if (this._enableRequest !== this._enabled) {
	      this._enabled = this._enableRequest;
	      if (!this._enabled) {
	        this._frameTime = 0;
	      }
	    }
	  };
	  _proto.request = function request(renderVersion) {
	    this.pastFrameAllocations.set(renderVersion, this.frameAllocations);
	    this.frameAllocations = [];
	  };
	  _proto.report = function report(renderVersion, timings) {
	    if (timings) {
	      var allocations = this.pastFrameAllocations.get(renderVersion);
	      Debug.assert(allocations.length === timings.length);

	      // store frame duration
	      if (timings.length > 0) {
	        this._frameTime = timings[0];
	      }

	      // log out timings
	      if (Tracing.get(TRACEID_GPU_TIMINGS)) {
	        for (var i = 0; i < allocations.length; ++i) {
	          var name = allocations[i];
	          Debug.trace(TRACEID_GPU_TIMINGS, timings[i].toFixed(2) + " ms " + name);
	        }
	      }
	    }

	    // remove frame info
	    this.pastFrameAllocations.delete(renderVersion);
	  }

	  /**
	   * Allocate a slot for GPU timing during the frame. This slot is valid only for the current
	   * frame. This allows multiple timers to be used during the frame, each with a unique name.
	   * @param {string} name - The name of the slot.
	   * @returns {number} The assigned slot index.
	   * @ignore
	   */;
	  _proto.getSlot = function getSlot(name) {
	    var slot = this.frameAllocations.length;
	    this.frameAllocations.push(name);
	    return slot;
	  }

	  /**
	   * Number of slots allocated during the frame.
	   *
	   * @ignore
	   */;
	  _createClass(GpuProfiler, [{
	    key: "enabled",
	    get: function get() {
	      return this._enableRequest;
	    },
	    set: function set(value) {
	      this._enableRequest = value;
	    }
	  }, {
	    key: "slotCount",
	    get: function get() {
	      return this.frameAllocations.length;
	    }
	  }]);
	  return GpuProfiler;
	}();

	/**
	 * A wrapper over the GpuQuerySet object, allowing timestamp and occlusion queries. The results
	 * are copied back using staging buffers to avoid blocking.
	 *
	 * @ignore
	 */
	var WebgpuQuerySet = /*#__PURE__*/function () {
	  function WebgpuQuerySet(device, isTimestamp, capacity) {
	    /**
	     * @type {GPUQuerySet}
	     */
	    this.querySet = void 0;
	    this.stagingBuffers = [];
	    this.activeStagingBuffer = null;
	    /** @type {number} */
	    this.bytesPerSlot = void 0;
	    this.device = device;
	    this.capacity = capacity;
	    this.bytesPerSlot = isTimestamp ? 8 : 4;

	    // query set
	    var wgpu = device.wgpu;
	    this.querySet = wgpu.createQuerySet({
	      type: isTimestamp ? 'timestamp' : 'occlusion',
	      count: capacity
	    });
	    DebugHelper.setLabel(this.querySet, "QuerySet-" + (isTimestamp ? 'Timestamp' : 'Occlusion'));

	    // gpu buffer for query results GPU writes to
	    this.queryBuffer = wgpu.createBuffer({
	      size: this.bytesPerSlot * capacity,
	      usage: GPUBufferUsage.QUERY_RESOLVE | GPUBufferUsage.STORAGE | GPUBufferUsage.COPY_SRC | GPUBufferUsage.COPY_DST
	    });
	    DebugHelper.setLabel(this.queryBuffer, 'QueryGpuBuffer');
	  }
	  var _proto = WebgpuQuerySet.prototype;
	  _proto.destroy = function destroy() {
	    var _this$querySet, _this$queryBuffer;
	    (_this$querySet = this.querySet) == null || _this$querySet.destroy();
	    this.querySet = null;
	    (_this$queryBuffer = this.queryBuffer) == null || _this$queryBuffer.destroy();
	    this.queryBuffer = null;
	    this.activeStagingBuffer = null;
	    this.stagingBuffers.forEach(function (stagingBuffer) {
	      stagingBuffer.destroy();
	    });
	    this.stagingBuffers = null;
	  };
	  _proto.getStagingBuffer = function getStagingBuffer() {
	    var stagingBuffer = this.stagingBuffers.pop();
	    if (!stagingBuffer) {
	      stagingBuffer = this.device.wgpu.createBuffer({
	        size: this.queryBuffer.size,
	        usage: GPUBufferUsage.COPY_DST | GPUBufferUsage.MAP_READ
	      });
	      DebugHelper.setLabel(this.queryBuffer, 'QueryStagingBuffer');
	    }
	    return stagingBuffer;
	  };
	  _proto.resolve = function resolve(count) {
	    var device = this.device;
	    var commandEncoder = device.wgpu.createCommandEncoder();
	    DebugHelper.setLabel(commandEncoder, 'ResolveQuerySet-Encoder');

	    // copy times to the gpu buffer
	    commandEncoder.resolveQuerySet(this.querySet, 0, count, this.queryBuffer, 0);

	    // copy the gpu buffer to the staging buffer
	    var activeStagingBuffer = this.getStagingBuffer();
	    this.activeStagingBuffer = activeStagingBuffer;
	    commandEncoder.copyBufferToBuffer(this.queryBuffer, 0, activeStagingBuffer, 0, this.bytesPerSlot * count);
	    var cb = commandEncoder.finish();
	    DebugHelper.setLabel(cb, 'ResolveQuerySet');
	    device.addCommandBuffer(cb);
	  };
	  _proto.request = function request(count, renderVersion) {
	    var _this = this;
	    var stagingBuffer = this.activeStagingBuffer;
	    this.activeStagingBuffer = null;
	    return stagingBuffer.mapAsync(GPUMapMode.READ).then(function () {
	      var _this$stagingBuffers;
	      // timestamps in nanoseconds. Note that this array is valid only till we unmap the staging buffer.
	      var srcTimings = new BigInt64Array(stagingBuffer.getMappedRange());

	      // convert to ms per sample pair
	      var timings = [];
	      for (var i = 0; i < count; i++) {
	        timings.push(Number(srcTimings[i * 2 + 1] - srcTimings[i * 2]) * 0.000001);
	      }
	      stagingBuffer.unmap();
	      (_this$stagingBuffers = _this.stagingBuffers) == null || _this$stagingBuffers.push(stagingBuffer);
	      return {
	        renderVersion: renderVersion,
	        timings: timings
	      };
	    });
	  };
	  return WebgpuQuerySet;
	}();

	var WebgpuGpuProfiler = /*#__PURE__*/function (_GpuProfiler) {
	  _inheritsLoose(WebgpuGpuProfiler, _GpuProfiler);
	  function WebgpuGpuProfiler(device) {
	    var _this;
	    _this = _GpuProfiler.call(this) || this;
	    _this.device = void 0;
	    /** @type {number} */
	    _this.frameGPUMarkerSlot = void 0;
	    _this.device = device;

	    // gpu timing queries
	    _this.timestampQueriesSet = device.supportsTimestampQuery ? new WebgpuQuerySet(device, true, 512) : null;
	    return _this;
	  }
	  var _proto = WebgpuGpuProfiler.prototype;
	  _proto.destroy = function destroy() {
	    var _this$timestampQuerie;
	    (_this$timestampQuerie = this.timestampQueriesSet) == null || _this$timestampQuerie.destroy();
	    this.timestampQueriesSet = null;
	  };
	  _proto.frameStart = function frameStart() {
	    this.processEnableRequest();
	  };
	  _proto.frameEnd = function frameEnd() {
	    if (this._enabled) {
	      var _this$timestampQuerie2;
	      // schedule command buffer where timestamps are copied to CPU
	      (_this$timestampQuerie2 = this.timestampQueriesSet) == null || _this$timestampQuerie2.resolve(this.slotCount * 2);
	    }
	  };
	  _proto.request = function request() {
	    var _this2 = this;
	    if (this._enabled) {
	      var _this$timestampQuerie3;
	      // request results
	      var renderVersion = this.device.renderVersion;
	      (_this$timestampQuerie3 = this.timestampQueriesSet) == null || _this$timestampQuerie3.request(this.slotCount, renderVersion).then(function (results) {
	        _this2.report(results.renderVersion, results.timings);
	      });
	      _GpuProfiler.prototype.request.call(this, renderVersion);
	    }
	  };
	  return WebgpuGpuProfiler;
	}(GpuProfiler);

	/**
	 * A WebGPU helper class implementing custom resolve of multi-sampled textures.
	 *
	 * @ignore
	 */
	var WebgpuResolver = /*#__PURE__*/function () {
	  function WebgpuResolver(device) {
	    /** @type {new Function("modulePath", "return import(modulePath)")('./webgpu-graphics-device.js').WebgpuGraphicsDevice} */
	    this.device = void 0;
	    /**
	     * Cache of render pipelines for each texture format, to avoid their per frame creation.
	     *
	     * @type {Map<GPUTextureFormat, GPURenderPipeline>}
	     * @private
	     */
	    this.pipelineCache = new Map();
	    this.device = device;

	    // Shader that renders a fullscreen textured quad and copies the depth value from sample index 0
	    // TODO: could handle all sample indices and use min/max as needed
	    var code = "\n \n            var<private> pos : array<vec2f, 4> = array<vec2f, 4>(\n                vec2(-1.0, 1.0), vec2(1.0, 1.0), vec2(-1.0, -1.0), vec2(1.0, -1.0)\n            );\n\n            struct VertexOutput {\n                @builtin(position) position : vec4f,\n            };\n\n            @vertex\n            fn vertexMain(@builtin(vertex_index) vertexIndex : u32) -> VertexOutput {\n              var output : VertexOutput;\n              output.position = vec4f(pos[vertexIndex], 0, 1);\n              return output;\n            }\n\n            @group(0) @binding(0) var img : texture_depth_multisampled_2d;\n\n            @fragment\n            fn fragmentMain(@builtin(position) fragColor: vec4f) -> @location(0) vec4f {\n                // load th depth value from sample index 0\n                var depth = textureLoad(img, vec2i(fragColor.xy), 0u);\n                return vec4<f32>(depth, 0.0, 0.0, 0.0);\n            }\n        ";
	    this.shader = new Shader(device, {
	      name: 'WebGPUResolverDepthShader',
	      shaderLanguage: SHADERLANGUAGE_WGSL,
	      vshader: code,
	      fshader: code
	    });
	  }
	  var _proto = WebgpuResolver.prototype;
	  _proto.destroy = function destroy() {
	    this.shader.destroy();
	    this.shader = null;
	    this.pipelineCache = null;
	  }

	  /** @private */;
	  _proto.getPipeline = function getPipeline(format) {
	    var pipeline = this.pipelineCache.get(format);
	    if (!pipeline) {
	      pipeline = this.createPipeline(format);
	      this.pipelineCache.set(format, pipeline);
	    }
	    return pipeline;
	  }

	  /** @private */;
	  _proto.createPipeline = function createPipeline(format) {
	    /** @type {new Function("modulePath", "return import(modulePath)")('./webgpu-shader.js').WebgpuShader} */
	    var webgpuShader = this.shader.impl;
	    var pipeline = this.device.wgpu.createRenderPipeline({
	      layout: 'auto',
	      vertex: {
	        module: webgpuShader.getVertexShaderModule(),
	        entryPoint: webgpuShader.vertexEntryPoint
	      },
	      fragment: {
	        module: webgpuShader.getFragmentShaderModule(),
	        entryPoint: webgpuShader.fragmentEntryPoint,
	        targets: [{
	          format: format
	        }]
	      },
	      primitive: {
	        topology: 'triangle-strip'
	      }
	    });
	    DebugHelper.setLabel(pipeline, "RenderPipeline-DepthResolver-" + format);
	    return pipeline;
	  }

	  /**
	   * @param {GPUCommandEncoder} commandEncoder - Command encoder to use for the resolve.
	   * @param {GPUTexture} sourceTexture - Source multi-sampled depth texture to resolve.
	   * @param {GPUTexture} destinationTexture - Destination depth texture to resolve to.
	   * @private
	   */;
	  _proto.resolveDepth = function resolveDepth(commandEncoder, sourceTexture, destinationTexture) {
	    Debug.assert(sourceTexture.sampleCount > 1);
	    Debug.assert(destinationTexture.sampleCount === 1);
	    Debug.assert(sourceTexture.depthOrArrayLayers === destinationTexture.depthOrArrayLayers);
	    var device = this.device;
	    var wgpu = device.wgpu;

	    // pipeline depends on the format
	    var pipeline = this.getPipeline(destinationTexture.format);
	    DebugGraphics.pushGpuMarker(device, 'DEPTH_RESOLVE-RENDERER');
	    var numFaces = sourceTexture.depthOrArrayLayers;
	    for (var face = 0; face < numFaces; face++) {
	      // copy depth only (not stencil)
	      var srcView = sourceTexture.createView({
	        dimension: '2d',
	        aspect: 'depth-only',
	        baseMipLevel: 0,
	        mipLevelCount: 1,
	        baseArrayLayer: face
	      });
	      var dstView = destinationTexture.createView({
	        dimension: '2d',
	        baseMipLevel: 0,
	        mipLevelCount: 1,
	        baseArrayLayer: face
	      });
	      var passEncoder = commandEncoder.beginRenderPass({
	        colorAttachments: [{
	          view: dstView,
	          loadOp: 'clear',
	          storeOp: 'store'
	        }]
	      });
	      DebugHelper.setLabel(passEncoder, "DepthResolve-PassEncoder");

	      // no need for a sampler when using textureLoad
	      var bindGroup = wgpu.createBindGroup({
	        layout: pipeline.getBindGroupLayout(0),
	        entries: [{
	          binding: 0,
	          resource: srcView
	        }]
	      });
	      passEncoder.setPipeline(pipeline);
	      passEncoder.setBindGroup(0, bindGroup);
	      passEncoder.draw(4);
	      passEncoder.end();
	    }
	    DebugGraphics.popGpuMarker(device);

	    // clear invalidated state
	    device.pipeline = null;
	  };
	  return WebgpuResolver;
	}();

	/**
	 * A WebGPU implementation of the Compute.
	 *
	 * @ignore
	 */
	var WebgpuCompute = /*#__PURE__*/function () {
	  function WebgpuCompute(compute) {
	    this.compute = compute;
	    var device = compute.device,
	      shader = compute.shader;

	    // create bind group
	    var computeBindGroupFormat = shader.impl.computeBindGroupFormat;
	    Debug.assert(computeBindGroupFormat, 'Compute shader does not have computeBindGroupFormat specified', shader);
	    this.bindGroup = new BindGroup(device, computeBindGroupFormat);
	    DebugHelper.setName(this.bindGroup, "Compute-BindGroup_" + this.bindGroup.id);

	    // pipeline
	    this.pipeline = device.computePipeline.get(shader, computeBindGroupFormat);
	  }
	  var _proto = WebgpuCompute.prototype;
	  _proto.dispatch = function dispatch(x, y, z) {
	    // TODO: currently each dispatch is a separate compute pass, which is not optimal, and we should
	    // batch multiple dispatches into a single compute pass
	    var device = this.compute.device;
	    device.startComputePass();

	    // bind group data
	    var bindGroup = this.bindGroup;
	    bindGroup.update();
	    device.setBindGroup(0, bindGroup);

	    // dispatch
	    var passEncoder = device.passEncoder;
	    passEncoder.setPipeline(this.pipeline);
	    passEncoder.dispatchWorkgroups(x, y, z);
	    device.endComputePass();
	  };
	  return WebgpuCompute;
	}();

	var WebgpuGraphicsDevice = /*#__PURE__*/function (_GraphicsDevice) {
	  _inheritsLoose(WebgpuGraphicsDevice, _GraphicsDevice);
	  function WebgpuGraphicsDevice(canvas, options) {
	    var _options$alpha, _options$antialias;
	    var _this;
	    if (options === void 0) {
	      options = {};
	    }
	    _this = _GraphicsDevice.call(this, canvas, options) || this;
	    /**
	     * Object responsible for caching and creation of render pipelines.
	     */
	    _this.renderPipeline = new WebgpuRenderPipeline(_assertThisInitialized(_this));
	    /**
	     * Object responsible for caching and creation of compute pipelines.
	     */
	    _this.computePipeline = new WebgpuComputePipeline(_assertThisInitialized(_this));
	    /**
	     * Object responsible for clearing the rendering surface by rendering a quad.
	     *
	     * @type { WebgpuClearRenderer }
	     */
	    _this.clearRenderer = void 0;
	    /**
	     * Object responsible for mipmap generation.
	     *
	     * @type { WebgpuMipmapRenderer }
	     */
	    _this.mipmapRenderer = void 0;
	    /**
	     * Render pipeline currently set on the device.
	     *
	     * @type {GPURenderPipeline}
	     * @private
	     */
	    _this.pipeline = void 0;
	    /**
	     * An array of bind group formats, based on currently assigned bind groups
	     *
	     * @type {WebgpuBindGroupFormat[]}
	     */
	    _this.bindGroupFormats = [];
	    /**
	     * Current command buffer encoder.
	     *
	     * @type {GPUCommandEncoder|null}
	     * @private
	     */
	    _this.commandEncoder = null;
	    /**
	     * Command buffers scheduled for execution on the GPU.
	     *
	     * @type {GPUCommandBuffer[]}
	     * @private
	     */
	    _this.commandBuffers = [];
	    /**
	     * @type {GPUSupportedLimits}
	     * @private
	     */
	    _this.limits = void 0;
	    options = _this.initOptions;

	    // alpha defaults to true
	    options.alpha = (_options$alpha = options.alpha) != null ? _options$alpha : true;
	    _this.backBufferAntialias = (_options$antialias = options.antialias) != null ? _options$antialias : false;
	    _this.isWebGPU = true;
	    _this._deviceType = DEVICETYPE_WEBGPU;
	    return _this;
	  }

	  /**
	   * Destroy the graphics device.
	   */
	  var _proto = WebgpuGraphicsDevice.prototype;
	  _proto.destroy = function destroy() {
	    this.clearRenderer.destroy();
	    this.clearRenderer = null;
	    this.mipmapRenderer.destroy();
	    this.mipmapRenderer = null;
	    this.resolver.destroy();
	    this.resolver = null;
	    _GraphicsDevice.prototype.destroy.call(this);
	  };
	  _proto.initDeviceCaps = function initDeviceCaps() {
	    // temporarily disabled functionality which is not supported to avoid errors
	    this.disableParticleSystem = true;
	    var limits = this.gpuAdapter.limits;
	    this.limits = limits;
	    this.precision = 'highp';
	    this.maxPrecision = 'highp';
	    this.maxSamples = 4;
	    this.maxTextures = 16;
	    this.maxTextureSize = limits.maxTextureDimension2D;
	    this.maxCubeMapSize = limits.maxTextureDimension2D;
	    this.maxVolumeSize = limits.maxTextureDimension3D;
	    this.maxColorAttachments = limits.maxColorAttachments;
	    this.maxPixelRatio = 1;
	    this.maxAnisotropy = 16;
	    this.fragmentUniformsCount = limits.maxUniformBufferBindingSize / 16;
	    this.vertexUniformsCount = limits.maxUniformBufferBindingSize / 16;
	    this.supportsInstancing = true;
	    this.supportsUniformBuffers = true;
	    this.supportsVolumeTextures = true;
	    this.supportsBoneTextures = true;
	    this.supportsMorphTargetTexturesCore = true;
	    this.supportsAreaLights = true;
	    this.supportsDepthShadow = true;
	    this.supportsGpuParticles = false;
	    this.supportsMrt = true;
	    this.supportsCompute = true;
	    this.extUintElement = true;
	    this.extTextureFloat = true;
	    this.textureFloatRenderable = true;
	    this.textureHalfFloatFilterable = true;
	    this.extTextureHalfFloat = true;
	    this.textureHalfFloatRenderable = true;
	    this.textureHalfFloatUpdatable = true;
	    this.boneLimit = 1024;
	    this.supportsImageBitmap = true;
	    this.extStandardDerivatives = true;
	    this.extBlendMinmax = true;
	    this.areaLightLutFormat = this.textureFloatFilterable ? PIXELFORMAT_RGBA32F : PIXELFORMAT_RGBA8;
	    this.supportsTextureFetch = true;

	    // WebGPU currently only supports 1 and 4 samples
	    this.samples = this.backBufferAntialias ? 4 : 1;
	  };
	  _proto.initWebGpu = /*#__PURE__*/function () {
	    var _initWebGpu = _asyncToGenerator( /*#__PURE__*/_regeneratorRuntime().mark(function _callee(glslangUrl, twgslUrl) {
	      var _this2 = this;
	      var buildUrl, results, adapterOptions, requiredFeatures, requireFeature, deviceDescr, preferredCanvasFormat;
	      return _regeneratorRuntime().wrap(function _callee$(_context) {
	        while (1) switch (_context.prev = _context.next) {
	          case 0:
	            if (window.navigator.gpu) {
	              _context.next = 2;
	              break;
	            }
	            throw new Error('Unable to retrieve GPU. Ensure you are using a browser that supports WebGPU rendering.');
	          case 2:
	            // temporary message to confirm Webgpu is being used
	            Debug.log("WebgpuGraphicsDevice initialization ..");

	            // build a full URL from a relative path
	            buildUrl = function buildUrl(srcPath) {
	              if (!path.isRelativePath(srcPath)) {
	                return srcPath;
	              }
	              var url = new URL(window.location.href);
	              url.pathname = srcPath;
	              url.search = '';
	              return url.toString();
	            };
	            _context.next = 6;
	            return Promise.all([new Function("modulePath", "return import(modulePath)")("" + buildUrl(twgslUrl)).then(function (module) {
	              return twgsl(twgslUrl.replace('.js', '.wasm'));
	            }), new Function("modulePath", "return import(modulePath)")("" + buildUrl(glslangUrl)).then(function (module) {
	              return module.default();
	            })]);
	          case 6:
	            results = _context.sent;
	            this.twgsl = results[0];
	            this.glslang = results[1];

	            /** @type {GPURequestAdapterOptions} */
	            adapterOptions = {
	              powerPreference: this.initOptions.powerPreference !== 'default' ? this.initOptions.powerPreference : undefined
	            };
	            /**
	             * @type {GPUAdapter}
	             * @private
	             */
	            _context.next = 12;
	            return window.navigator.gpu.requestAdapter(adapterOptions);
	          case 12:
	            this.gpuAdapter = _context.sent;
	            // optional features:
	            //      "depth-clip-control",
	            //      "depth32float-stencil8",
	            //      "indirect-first-instance",
	            //      "shader-f16",
	            //      "bgra8unorm-storage",
	            // request optional features
	            requiredFeatures = [];
	            requireFeature = function requireFeature(feature) {
	              var supported = _this2.gpuAdapter.features.has(feature);
	              if (supported) {
	                requiredFeatures.push(feature);
	              }
	              return supported;
	            };
	            this.textureFloatFilterable = requireFeature('float32-filterable');
	            this.extCompressedTextureS3TC = requireFeature('texture-compression-bc');
	            this.extCompressedTextureETC = requireFeature('texture-compression-etc2');
	            this.extCompressedTextureASTC = requireFeature('texture-compression-astc');
	            this.supportsTimestampQuery = requireFeature('timestamp-query');
	            this.textureRG11B10Renderable = requireFeature('rg11b10ufloat-renderable');
	            Debug.log("WEBGPU features: " + requiredFeatures.join(', '));

	            /** @type {GPUDeviceDescriptor} */
	            deviceDescr = {
	              requiredFeatures: requiredFeatures,
	              // Note that we can request limits, but it does not seem to be supported at the moment
	              requiredLimits: {},
	              defaultQueue: {
	                label: 'Default Queue'
	              }
	            };
	            /**
	             * @type {GPUDevice}
	             * @private
	             */
	            _context.next = 25;
	            return this.gpuAdapter.requestDevice(deviceDescr);
	          case 25:
	            this.wgpu = _context.sent;
	            this.initDeviceCaps();
	            this.gpuContext = this.canvas.getContext('webgpu');

	            // pixel format of the framebuffer is the most efficient one on the system
	            preferredCanvasFormat = navigator.gpu.getPreferredCanvasFormat();
	            this.backBufferFormat = preferredCanvasFormat === 'rgba8unorm' ? PIXELFORMAT_RGBA8 : PIXELFORMAT_BGRA8;

	            /**
	             * Configuration of the main colorframebuffer we obtain using getCurrentTexture
	             *
	             * @type {GPUCanvasConfiguration}
	             * @private
	             */
	            this.canvasConfig = {
	              device: this.wgpu,
	              colorSpace: 'srgb',
	              alphaMode: this.initOptions.alpha ? 'premultiplied' : 'opaque',
	              // use preferred format for optimal performance on mobile
	              format: preferredCanvasFormat,
	              // RENDER_ATTACHMENT is required, COPY_SRC allows scene grab to copy out from it
	              usage: GPUTextureUsage.RENDER_ATTACHMENT | GPUTextureUsage.COPY_SRC | GPUTextureUsage.COPY_DST,
	              // formats that views created from textures returned by getCurrentTexture may use
	              viewFormats: []
	            };
	            this.gpuContext.configure(this.canvasConfig);
	            this.createBackbuffer();
	            this.clearRenderer = new WebgpuClearRenderer(this);
	            this.mipmapRenderer = new WebgpuMipmapRenderer(this);
	            this.resolver = new WebgpuResolver(this);
	            this.postInit();
	            return _context.abrupt("return", this);
	          case 38:
	          case "end":
	            return _context.stop();
	        }
	      }, _callee, this);
	    }));
	    function initWebGpu(_x, _x2) {
	      return _initWebGpu.apply(this, arguments);
	    }
	    return initWebGpu;
	  }();
	  _proto.postInit = function postInit() {
	    _GraphicsDevice.prototype.postInit.call(this);
	    this.initializeRenderState();
	    this.setupPassEncoderDefaults();
	    this.gpuProfiler = new WebgpuGpuProfiler(this);

	    // init dynamic buffer using 1MB allocation
	    this.dynamicBuffers = new WebgpuDynamicBuffers(this, 1024 * 1024, this.limits.minUniformBufferOffsetAlignment);
	  };
	  _proto.createBackbuffer = function createBackbuffer() {
	    this.supportsStencil = this.initOptions.stencil;
	    this.backBuffer = new RenderTarget({
	      name: 'WebgpuFramebuffer',
	      graphicsDevice: this,
	      depth: this.initOptions.depth,
	      stencil: this.supportsStencil,
	      samples: this.samples
	    });
	  };
	  _proto.frameStart = function frameStart() {
	    _GraphicsDevice.prototype.frameStart.call(this);
	    this.gpuProfiler.frameStart();

	    // submit any commands collected before the frame rendering
	    this.submit();
	    WebgpuDebug.memory(this);
	    WebgpuDebug.validate(this);

	    // current frame color output buffer
	    var outColorBuffer = this.gpuContext.getCurrentTexture();
	    DebugHelper.setLabel(outColorBuffer, "" + this.backBuffer.name);

	    // reallocate framebuffer if dimensions change, to match the output texture
	    if (this.backBufferSize.x !== outColorBuffer.width || this.backBufferSize.y !== outColorBuffer.height) {
	      this.backBufferSize.set(outColorBuffer.width, outColorBuffer.height);
	      this.backBuffer.destroy();
	      this.backBuffer = null;
	      this.createBackbuffer();
	    }
	    var rt = this.backBuffer;
	    var wrt = rt.impl;

	    // assign the format, allowing following init call to use it to allocate matching multisampled buffer
	    wrt.setColorAttachment(0, undefined, outColorBuffer.format);
	    this.initRenderTarget(rt);

	    // assign current frame's render texture
	    wrt.assignColorTexture(outColorBuffer);
	    WebgpuDebug.end(this);
	    WebgpuDebug.end(this);
	  };
	  _proto.frameEnd = function frameEnd() {
	    _GraphicsDevice.prototype.frameEnd.call(this);
	    this.gpuProfiler.frameEnd();

	    // submit scheduled command buffers
	    this.submit();
	    this.gpuProfiler.request();
	  };
	  _proto.createUniformBufferImpl = function createUniformBufferImpl(uniformBuffer) {
	    return new WebgpuUniformBuffer(uniformBuffer);
	  };
	  _proto.createVertexBufferImpl = function createVertexBufferImpl(vertexBuffer, format) {
	    return new WebgpuVertexBuffer(vertexBuffer, format);
	  };
	  _proto.createIndexBufferImpl = function createIndexBufferImpl(indexBuffer) {
	    return new WebgpuIndexBuffer(indexBuffer);
	  };
	  _proto.createShaderImpl = function createShaderImpl(shader) {
	    return new WebgpuShader(shader);
	  };
	  _proto.createTextureImpl = function createTextureImpl(texture) {
	    return new WebgpuTexture(texture);
	  };
	  _proto.createRenderTargetImpl = function createRenderTargetImpl(renderTarget) {
	    return new WebgpuRenderTarget(renderTarget);
	  };
	  _proto.createBindGroupFormatImpl = function createBindGroupFormatImpl(bindGroupFormat) {
	    return new WebgpuBindGroupFormat(bindGroupFormat);
	  };
	  _proto.createBindGroupImpl = function createBindGroupImpl(bindGroup) {
	    return new WebgpuBindGroup();
	  };
	  _proto.createComputeImpl = function createComputeImpl(compute) {
	    return new WebgpuCompute(compute);
	  }

	  /**
	   * @param {number} index - Index of the bind group slot
	   * @param {new Function("modulePath", "return import(modulePath)")('../bind-group.js').BindGroup} bindGroup - Bind group to attach
	   */;
	  _proto.setBindGroup = function setBindGroup(index, bindGroup) {
	    // TODO: this condition should be removed, it's here to handle fake grab pass, which should be refactored instead
	    if (this.passEncoder) {
	      // set it on the device
	      this.passEncoder.setBindGroup(index, bindGroup.impl.bindGroup, bindGroup.uniformBufferOffsets);

	      // store the active formats, used by the pipeline creation
	      this.bindGroupFormats[index] = bindGroup.format.impl;
	    }
	  };
	  _proto.submitVertexBuffer = function submitVertexBuffer(vertexBuffer, slot) {
	    var elements = vertexBuffer.format.elements;
	    var elementCount = elements.length;
	    var vbBuffer = vertexBuffer.impl.buffer;
	    for (var i = 0; i < elementCount; i++) {
	      this.passEncoder.setVertexBuffer(slot + i, vbBuffer, elements[i].offset);
	    }
	    return elementCount;
	  };
	  _proto.draw = function draw(primitive, numInstances, keepBuffers) {
	    if (numInstances === void 0) {
	      numInstances = 1;
	    }
	    if (this.shader.ready && !this.shader.failed) {
	      WebgpuDebug.validate(this);
	      var passEncoder = this.passEncoder;
	      Debug.assert(passEncoder);

	      // vertex buffers
	      var vb0 = this.vertexBuffers[0];
	      var vb1 = this.vertexBuffers[1];
	      this.vertexBuffers.length = 0;
	      if (vb0) {
	        var vbSlot = this.submitVertexBuffer(vb0, 0);
	        if (vb1) {
	          this.submitVertexBuffer(vb1, vbSlot);
	        }
	      }

	      // render pipeline
	      var pipeline = this.renderPipeline.get(primitive, vb0 == null ? void 0 : vb0.format, vb1 == null ? void 0 : vb1.format, this.shader, this.renderTarget, this.bindGroupFormats, this.blendState, this.depthState, this.cullMode, this.stencilEnabled, this.stencilFront, this.stencilBack);
	      Debug.assert(pipeline);
	      if (this.pipeline !== pipeline) {
	        this.pipeline = pipeline;
	        passEncoder.setPipeline(pipeline);
	      }

	      // draw
	      var ib = this.indexBuffer;
	      if (ib) {
	        this.indexBuffer = null;
	        passEncoder.setIndexBuffer(ib.impl.buffer, ib.impl.format);
	        passEncoder.drawIndexed(primitive.count, numInstances, primitive.base, 0, 0);
	      } else {
	        passEncoder.draw(primitive.count, numInstances, primitive.base, 0);
	      }
	      WebgpuDebug.end(this, {
	        vb0: vb0,
	        vb1: vb1,
	        ib: ib,
	        primitive: primitive,
	        numInstances: numInstances,
	        pipeline: pipeline
	      });
	    }
	  };
	  _proto.setShader = function setShader(shader, asyncCompile) {
	    if (shader !== this.shader) {
	      this.shader = shader;

	      // TODO: we should probably track other stats instead, like pipeline switches
	      this._shaderSwitchesPerFrame++;
	    }
	  };
	  _proto.setBlendState = function setBlendState(blendState) {
	    this.blendState.copy(blendState);
	  };
	  _proto.setDepthState = function setDepthState(depthState) {
	    this.depthState.copy(depthState);
	  };
	  _proto.setStencilState = function setStencilState(stencilFront, stencilBack) {
	    if (stencilFront || stencilBack) {
	      this.stencilEnabled = true;
	      this.stencilFront.copy(stencilFront != null ? stencilFront : StencilParameters.DEFAULT);
	      this.stencilBack.copy(stencilBack != null ? stencilBack : StencilParameters.DEFAULT);

	      // ref value - based on stencil front
	      var ref = this.stencilFront.ref;
	      if (this.stencilRef !== ref) {
	        this.stencilRef = ref;
	        this.passEncoder.setStencilReference(ref);
	      }
	    } else {
	      this.stencilEnabled = false;
	    }
	  };
	  _proto.setBlendColor = function setBlendColor(r, g, b, a) {
	    var c = this.blendColor;
	    if (r !== c.r || g !== c.g || b !== c.b || a !== c.a) {
	      c.set(r, g, b, a);
	      this.passEncoder.setBlendConstant(c);
	    }
	  };
	  _proto.setCullMode = function setCullMode(cullMode) {
	    this.cullMode = cullMode;
	  };
	  _proto.setAlphaToCoverage = function setAlphaToCoverage(state) {};
	  _proto.initializeContextCaches = function initializeContextCaches() {
	    _GraphicsDevice.prototype.initializeContextCaches.call(this);
	  }

	  /**
	   * Set up default values for the render pass encoder.
	   */;
	  _proto.setupPassEncoderDefaults = function setupPassEncoderDefaults() {
	    this.pipeline = null;
	    this.stencilRef = 0;
	    this.blendColor.set(0, 0, 0, 0);
	  };
	  _proto._uploadDirtyTextures = function _uploadDirtyTextures() {
	    this.textures.forEach(function (texture) {
	      if (texture._needsUpload || texture._needsMipmaps) {
	        texture.upload();
	      }
	    });
	  }

	  /**
	   * Start a render pass.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('../render-pass.js').RenderPass} renderPass - The render pass to start.
	   * @ignore
	   */;
	  _proto.startRenderPass = function startRenderPass(renderPass) {
	    // upload textures that need it, to avoid them being uploaded / their mips generated during the pass
	    // TODO: this needs a better solution
	    this._uploadDirtyTextures();
	    WebgpuDebug.internal(this);
	    WebgpuDebug.validate(this);
	    var rt = renderPass.renderTarget || this.backBuffer;
	    this.renderTarget = rt;
	    Debug.assert(rt);

	    /** @type {WebgpuRenderTarget} */
	    var wrt = rt.impl;

	    // create a new encoder for each pass
	    this.commandEncoder = this.wgpu.createCommandEncoder();
	    DebugHelper.setLabel(this.commandEncoder, renderPass.name + "-Encoder");

	    // framebuffer is initialized at the start of the frame
	    if (rt !== this.backBuffer) {
	      this.initRenderTarget(rt);
	    }

	    // set up clear / store / load settings
	    wrt.setupForRenderPass(renderPass);
	    var renderPassDesc = wrt.renderPassDescriptor;

	    // timestamp
	    if (this.gpuProfiler._enabled) {
	      if (this.gpuProfiler.timestampQueriesSet) {
	        var slot = this.gpuProfiler.getSlot(renderPass.name);
	        renderPassDesc.timestampWrites = {
	          querySet: this.gpuProfiler.timestampQueriesSet.querySet,
	          beginningOfPassWriteIndex: slot * 2,
	          endOfPassWriteIndex: slot * 2 + 1
	        };
	      }
	    }

	    // start the pass
	    this.passEncoder = this.commandEncoder.beginRenderPass(renderPassDesc);
	    DebugHelper.setLabel(this.passEncoder, renderPass.name);

	    // push marker to the passEncoder
	    DebugGraphics.pushGpuMarker(this, "Pass:" + renderPass.name);
	    this.setupPassEncoderDefaults();

	    // the pass always clears full target
	    // TODO: avoid this setting the actual viewport/scissor on webgpu as those are automatically reset to full
	    // render target. We just need to update internal state, for the get functionality to return it.
	    var width = rt.width,
	      height = rt.height;
	    this.setViewport(0, 0, width, height);
	    this.setScissor(0, 0, width, height);
	    Debug.assert(!this.insideRenderPass, 'RenderPass cannot be started while inside another render pass.');
	    this.insideRenderPass = true;
	  }

	  /**
	   * End a render pass.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('../render-pass.js').RenderPass} renderPass - The render pass to end.
	   * @ignore
	   */;
	  _proto.endRenderPass = function endRenderPass(renderPass) {
	    // pop the marker from the passEncoder
	    DebugGraphics.popGpuMarker(this);

	    // end the render pass
	    this.passEncoder.end();
	    this.passEncoder = null;
	    this.insideRenderPass = false;

	    // each render pass can use different number of bind groups
	    this.bindGroupFormats.length = 0;

	    // generate mipmaps using the same command buffer encoder
	    for (var i = 0; i < renderPass.colorArrayOps.length; i++) {
	      var colorOps = renderPass.colorArrayOps[i];
	      if (colorOps.mipmaps) {
	        this.mipmapRenderer.generate(renderPass.renderTarget._colorBuffers[i].impl);
	      }
	    }

	    // schedule command buffer submission
	    var cb = this.commandEncoder.finish();
	    DebugHelper.setLabel(cb, renderPass.name + "-CommandBuffer");
	    this.addCommandBuffer(cb);
	    this.commandEncoder = null;
	    WebgpuDebug.end(this, {
	      renderPass: renderPass
	    });
	    WebgpuDebug.end(this, {
	      renderPass: renderPass
	    });
	  };
	  _proto.startComputePass = function startComputePass() {
	    WebgpuDebug.internal(this);
	    WebgpuDebug.validate(this);

	    // create a new encoder for each pass
	    this.commandEncoder = this.wgpu.createCommandEncoder();
	    // DebugHelper.setLabel(this.commandEncoder, `${renderPass.name}-Encoder`);
	    DebugHelper.setLabel(this.commandEncoder, 'ComputePass-Encoder');

	    // clear cached encoder state
	    this.pipeline = null;

	    // TODO: add performance queries to compute passes

	    // start the pass
	    this.passEncoder = this.commandEncoder.beginComputePass();
	    DebugHelper.setLabel(this.passEncoder, 'ComputePass');
	    Debug.assert(!this.insideRenderPass, 'ComputePass cannot be started while inside another pass.');
	    this.insideRenderPass = true;
	  };
	  _proto.endComputePass = function endComputePass() {
	    // end the compute pass
	    this.passEncoder.end();
	    this.passEncoder = null;
	    this.insideRenderPass = false;

	    // each render pass can use different number of bind groups
	    this.bindGroupFormats.length = 0;

	    // schedule command buffer submission
	    var cb = this.commandEncoder.finish();
	    // DebugHelper.setLabel(cb, `${renderPass.name}-CommandBuffer`);
	    DebugHelper.setLabel(cb, 'ComputePass-CommandBuffer');
	    this.addCommandBuffer(cb);
	    this.commandEncoder = null;
	    WebgpuDebug.end(this);
	    WebgpuDebug.end(this);
	  };
	  _proto.addCommandBuffer = function addCommandBuffer(commandBuffer, front) {
	    if (front === void 0) {
	      front = false;
	    }
	    if (front) {
	      this.commandBuffers.unshift(commandBuffer);
	    } else {
	      this.commandBuffers.push(commandBuffer);
	    }
	  };
	  _proto.submit = function submit() {
	    var _this3 = this;
	    if (this.commandBuffers.length > 0) {
	      // copy dynamic buffers data to the GPU (this schedules the copy CB to run before all other CBs)
	      this.dynamicBuffers.submit();

	      // trace all scheduled command buffers
	      Debug.call(function () {
	        if (_this3.commandBuffers.length > 0) {
	          Debug.trace(TRACEID_RENDER_QUEUE, "SUBMIT (" + _this3.commandBuffers.length + ")");
	          for (var i = 0; i < _this3.commandBuffers.length; i++) {
	            Debug.trace(TRACEID_RENDER_QUEUE, "  CB: " + _this3.commandBuffers[i].label);
	          }
	        }
	      });
	      this.wgpu.queue.submit(this.commandBuffers);
	      this.commandBuffers.length = 0;

	      // notify dynamic buffers
	      this.dynamicBuffers.onCommandBuffersSubmitted();
	    }
	  };
	  _proto.clear = function clear(options) {
	    if (options.flags) {
	      this.clearRenderer.clear(this, this.renderTarget, options, this.defaultClearOptions);
	    }
	  };
	  _proto.setViewport = function setViewport(x, y, w, h) {
	    // TODO: only execute when it changes. Also, the viewport of encoder  matches the rendering attachments,
	    // so we can skip this if fullscreen
	    // TODO: this condition should be removed, it's here to handle fake grab pass, which should be refactored instead
	    if (this.passEncoder) {
	      if (!this.renderTarget.flipY) {
	        y = this.renderTarget.height - y - h;
	      }
	      this.vx = x;
	      this.vy = y;
	      this.vw = w;
	      this.vh = h;
	      this.passEncoder.setViewport(x, y, w, h, 0, 1);
	    }
	  };
	  _proto.setScissor = function setScissor(x, y, w, h) {
	    // TODO: only execute when it changes. Also, the viewport of encoder  matches the rendering attachments,
	    // so we can skip this if fullscreen
	    // TODO: this condition should be removed, it's here to handle fake grab pass, which should be refactored instead
	    if (this.passEncoder) {
	      if (!this.renderTarget.flipY) {
	        y = this.renderTarget.height - y - h;
	      }
	      this.sx = x;
	      this.sy = y;
	      this.sw = w;
	      this.sh = h;
	      this.passEncoder.setScissorRect(x, y, w, h);
	    }
	  }

	  /**
	   * Copies source render target into destination render target. Mostly used by post-effects.
	   *
	   * @param {RenderTarget} [source] - The source render target. Defaults to frame buffer.
	   * @param {RenderTarget} [dest] - The destination render target. Defaults to frame buffer.
	   * @param {boolean} [color] - If true will copy the color buffer. Defaults to false.
	   * @param {boolean} [depth] - If true will copy the depth buffer. Defaults to false.
	   * @returns {boolean} True if the copy was successful, false otherwise.
	   */;
	  _proto.copyRenderTarget = function copyRenderTarget(source, dest, color, depth) {
	    var _this$commandEncoder;
	    /** @type {GPUExtent3D} */
	    var copySize = {
	      width: source ? source.width : dest.width,
	      height: source ? source.height : dest.height,
	      depthOrArrayLayers: 1
	    };

	    // use existing or create new encoder if not in a render pass
	    var commandEncoder = (_this$commandEncoder = this.commandEncoder) != null ? _this$commandEncoder : this.wgpu.createCommandEncoder();
	    DebugHelper.setLabel(commandEncoder, 'CopyRenderTarget-Encoder');
	    DebugGraphics.pushGpuMarker(this, 'COPY-RT');
	    if (color) {
	      // read from supplied render target, or from the framebuffer
	      /** @type {GPUImageCopyTexture} */
	      var copySrc = {
	        texture: source ? source.colorBuffer.impl.gpuTexture : this.renderTarget.impl.assignedColorTexture,
	        mipLevel: 0
	      };

	      // write to supplied render target, or to the framebuffer
	      /** @type {GPUImageCopyTexture} */
	      var copyDst = {
	        texture: dest ? dest.colorBuffer.impl.gpuTexture : this.renderTarget.impl.assignedColorTexture,
	        mipLevel: 0
	      };
	      Debug.assert(copySrc.texture !== null && copyDst.texture !== null);
	      commandEncoder.copyTextureToTexture(copySrc, copyDst, copySize);
	    }
	    if (depth) {
	      // read from supplied render target, or from the framebuffer
	      var sourceRT = source ? source : this.renderTarget;
	      var sourceTexture = sourceRT.impl.depthTexture;
	      if (source.samples > 1) {
	        // resolve the depth to a color buffer of destination render target
	        var destTexture = dest.colorBuffer.impl.gpuTexture;
	        this.resolver.resolveDepth(commandEncoder, sourceTexture, destTexture);
	      } else {
	        // write to supplied render target, or to the framebuffer
	        var _destTexture = dest ? dest.depthBuffer.impl.gpuTexture : this.renderTarget.impl.depthTexture;

	        /** @type {GPUImageCopyTexture} */
	        var _copySrc = {
	          texture: sourceTexture,
	          mipLevel: 0
	        };

	        /** @type {GPUImageCopyTexture} */
	        var _copyDst = {
	          texture: _destTexture,
	          mipLevel: 0
	        };
	        Debug.assert(_copySrc.texture !== null && _copyDst.texture !== null);
	        commandEncoder.copyTextureToTexture(_copySrc, _copyDst, copySize);
	      }
	    }
	    DebugGraphics.popGpuMarker(this);

	    // if we created the encoder
	    if (!this.commandEncoder) {
	      // copy operation runs next
	      var cb = commandEncoder.finish();
	      DebugHelper.setLabel(cb, 'CopyRenderTarget-CommandBuffer');
	      this.addCommandBuffer(cb);
	    }
	    return true;
	  };
	  _proto.pushMarker = function pushMarker(name) {
	    var _this$passEncoder;
	    (_this$passEncoder = this.passEncoder) == null || _this$passEncoder.pushDebugGroup(name);
	  };
	  _proto.popMarker = function popMarker() {
	    var _this$passEncoder2;
	    (_this$passEncoder2 = this.passEncoder) == null || _this$passEncoder2.popDebugGroup();
	  };
	  return WebgpuGraphicsDevice;
	}(GraphicsDevice);

	var id$5 = 0;

	/**
	 * A texture is a container for texel data that can be utilized in a fragment shader. Typically,
	 * the texel data represents an image that is mapped over geometry.
	 *
	 * @category Graphics
	 */
	var Texture = /*#__PURE__*/function () {
	  /**
	   * Create a new Texture instance.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./graphics-device.js').GraphicsDevice} graphicsDevice - The graphics device
	   * used to manage this texture.
	   * @param {object} [options] - Object for passing optional arguments.
	   * @param {string} [options.name] - The name of the texture. Defaults to null.
	   * @param {number} [options.width] - The width of the texture in pixels. Defaults to 4.
	   * @param {number} [options.height] - The height of the texture in pixels. Defaults to 4.
	   * @param {number} [options.depth] - The number of depth slices in a 3D texture (not supported by WebGl1).
	   * @param {number} [options.format] - The pixel format of the texture. Can be:
	   *
	   * - {@link PIXELFORMAT_A8}
	   * - {@link PIXELFORMAT_L8}
	   * - {@link PIXELFORMAT_LA8}
	   * - {@link PIXELFORMAT_RGB565}
	   * - {@link PIXELFORMAT_RGBA5551}
	   * - {@link PIXELFORMAT_RGBA4}
	   * - {@link PIXELFORMAT_RGB8}
	   * - {@link PIXELFORMAT_RGBA8}
	   * - {@link PIXELFORMAT_DXT1}
	   * - {@link PIXELFORMAT_DXT3}
	   * - {@link PIXELFORMAT_DXT5}
	   * - {@link PIXELFORMAT_RGB16F}
	   * - {@link PIXELFORMAT_RGBA16F}
	   * - {@link PIXELFORMAT_RGB32F}
	   * - {@link PIXELFORMAT_RGBA32F}
	   * - {@link PIXELFORMAT_ETC1}
	   * - {@link PIXELFORMAT_PVRTC_2BPP_RGB_1}
	   * - {@link PIXELFORMAT_PVRTC_2BPP_RGBA_1}
	   * - {@link PIXELFORMAT_PVRTC_4BPP_RGB_1}
	   * - {@link PIXELFORMAT_PVRTC_4BPP_RGBA_1}
	   * - {@link PIXELFORMAT_111110F}
	   * - {@link PIXELFORMAT_ASTC_4x4}
	   * - {@link PIXELFORMAT_ATC_RGB}
	   * - {@link PIXELFORMAT_ATC_RGBA}
	   *
	   * Defaults to {@link PIXELFORMAT_RGBA8}.
	   * @param {string} [options.projection] - The projection type of the texture, used when the
	   * texture represents an environment. Can be:
	   *
	   * - {@link TEXTUREPROJECTION_NONE}
	   * - {@link TEXTUREPROJECTION_CUBE}
	   * - {@link TEXTUREPROJECTION_EQUIRECT}
	   * - {@link TEXTUREPROJECTION_OCTAHEDRAL}
	   *
	   * Defaults to {@link TEXTUREPROJECTION_CUBE} if options.cubemap is true, otherwise
	   * {@link TEXTUREPROJECTION_NONE}.
	   * @param {number} [options.minFilter] - The minification filter type to use. Defaults to
	   * {@link FILTER_LINEAR_MIPMAP_LINEAR}.
	   * @param {number} [options.magFilter] - The magnification filter type to use. Defaults to
	   * {@link FILTER_LINEAR}.
	   * @param {number} [options.anisotropy] - The level of anisotropic filtering to use. Defaults
	   * to 1.
	   * @param {number} [options.addressU] - The repeat mode to use in the U direction. Defaults to
	   * {@link ADDRESS_REPEAT}.
	   * @param {number} [options.addressV] - The repeat mode to use in the V direction. Defaults to
	   * {@link ADDRESS_REPEAT}.
	   * @param {number} [options.addressW] - The repeat mode to use in the W direction. Defaults to
	   * {@link ADDRESS_REPEAT}.
	   * @param {boolean} [options.mipmaps] - When enabled try to generate or use mipmaps for this
	   * texture. Default is true.
	   * @param {boolean} [options.cubemap] - Specifies whether the texture is to be a cubemap.
	   * Defaults to false.
	   * @param {number} [options.arrayLength] - Specifies whether the texture is to be a 2D texture array.
	   * When passed in as undefined or < 1, this is not an array texture. If >= 1, this is an array texture.
	   * (not supported by WebGL1). Defaults to undefined.
	   * @param {boolean} [options.volume] - Specifies whether the texture is to be a 3D volume
	   * (not supported by WebGL1). Defaults to false.
	   * @param {string} [options.type] - Specifies the texture type.  Can be:
	   *
	   * - {@link TEXTURETYPE_DEFAULT}
	   * - {@link TEXTURETYPE_RGBM}
	   * - {@link TEXTURETYPE_RGBE}
	   * - {@link TEXTURETYPE_RGBP}
	   * - {@link TEXTURETYPE_SWIZZLEGGGR}
	   *
	   * Defaults to {@link TEXTURETYPE_DEFAULT}.
	   * @param {boolean} [options.fixCubemapSeams] - Specifies whether this cubemap texture requires
	   * special seam fixing shader code to look right. Defaults to false.
	   * @param {boolean} [options.flipY] - Specifies whether the texture should be flipped in the
	   * Y-direction. Only affects textures with a source that is an image, canvas or video element.
	   * Does not affect cubemaps, compressed textures or textures set from raw pixel data. Defaults
	   * to false.
	   * @param {boolean} [options.premultiplyAlpha] - If true, the alpha channel of the texture (if
	   * present) is multiplied into the color channels. Defaults to false.
	   * @param {boolean} [options.compareOnRead] - When enabled, and if texture format is
	   * {@link PIXELFORMAT_DEPTH} or {@link PIXELFORMAT_DEPTHSTENCIL}, hardware PCF is enabled for
	   * this texture, and you can get filtered results of comparison using texture() in your shader
	   * (not supported by WebGL1). Defaults to false.
	   * @param {number} [options.compareFunc] - Comparison function when compareOnRead is enabled
	   * (not supported by WebGL1). Can be:
	   *
	   * - {@link FUNC_LESS}
	   * - {@link FUNC_LESSEQUAL}
	   * - {@link FUNC_GREATER}
	   * - {@link FUNC_GREATEREQUAL}
	   * - {@link FUNC_EQUAL}
	   * - {@link FUNC_NOTEQUAL}
	   *
	   * Defaults to {@link FUNC_LESS}.
	   * @param {Uint8Array[]|HTMLCanvasElement[]|HTMLImageElement[]|HTMLVideoElement[]|Uint8Array[][]} [options.levels]
	   * - Array of Uint8Array or other supported browser interface; or a two-dimensional array
	   * of Uint8Array if options.arrayLength is defined and greater than zero.
	   * @param {boolean} [options.storage] - Defines if texture can be used as a storage texture by
	   * a compute shader. Defaults to false.
	   * @example
	   * // Create a 8x8x24-bit texture
	   * const texture = new pc.Texture(graphicsDevice, {
	   *     width: 8,
	   *     height: 8,
	   *     format: pc.PIXELFORMAT_RGB8
	   * });
	   *
	   * // Fill the texture with a gradient
	   * const pixels = texture.lock();
	   * const count = 0;
	   * for (let i = 0; i < 8; i++) {
	   *     for (let j = 0; j < 8; j++) {
	   *         pixels[count++] = i * 32;
	   *         pixels[count++] = j * 32;
	   *         pixels[count++] = 255;
	   *     }
	   * }
	   * texture.unlock();
	   */
	  function Texture(graphicsDevice, options) {
	    var _options$name, _options$width, _options$height, _options$format, _options$storage, _options$cubemap, _options$fixCubemapSe, _options$flipY, _options$premultiplyA, _ref, _options$mipmaps, _options$minFilter, _options$magFilter, _options$anisotropy, _options$addressU, _options$addressV, _options$addressW, _options$compareOnRea, _options$compareFunc, _options$profilerHint;
	    if (options === void 0) {
	      options = {};
	    }
	    /**
	     * The name of the texture.
	     *
	     * @type {string}
	     */
	    this.name = void 0;
	    /** @ignore */
	    this._gpuSize = 0;
	    /** @protected */
	    this.id = id$5++;
	    /** @protected */
	    this._invalid = false;
	    /** @protected */
	    this._lockedLevel = -1;
	    /** @protected */
	    this._lockedMode = TEXTURELOCK_NONE;
	    /**
	     * A render version used to track the last time the texture properties requiring bind group
	     * to be updated were changed.
	     *
	     * @type {number}
	     * @ignore
	     */
	    this.renderVersionDirty = 0;
	    /** @protected */
	    this._storage = false;
	    this.device = graphicsDevice;
	    Debug.assert(this.device, "Texture constructor requires a graphicsDevice to be valid");
	    Debug.assert(!options.width || Number.isInteger(options.width), "Texture width must be an integer number, got", options);
	    Debug.assert(!options.height || Number.isInteger(options.height), "Texture height must be an integer number, got", options);
	    Debug.assert(!options.depth || Number.isInteger(options.depth), "Texture depth must be an integer number, got", options);
	    this.name = (_options$name = options.name) != null ? _options$name : '';
	    this._width = Math.floor((_options$width = options.width) != null ? _options$width : 4);
	    this._height = Math.floor((_options$height = options.height) != null ? _options$height : 4);
	    this._format = (_options$format = options.format) != null ? _options$format : PIXELFORMAT_RGBA8;
	    this._compressed = isCompressedPixelFormat(this._format);
	    this._integerFormat = isIntegerPixelFormat(this._format);
	    if (this._integerFormat) {
	      options.mipmaps = false;
	      options.minFilter = FILTER_NEAREST;
	      options.magFilter = FILTER_NEAREST;
	    }
	    if (graphicsDevice.supportsVolumeTextures) {
	      var _options$volume, _options$depth, _options$arrayLength;
	      this._volume = (_options$volume = options.volume) != null ? _options$volume : false;
	      this._depth = Math.floor((_options$depth = options.depth) != null ? _options$depth : 1);
	      this._arrayLength = Math.floor((_options$arrayLength = options.arrayLength) != null ? _options$arrayLength : 0);
	    } else {
	      this._volume = false;
	      this._depth = 1;
	      this._arrayLength = 0;
	    }
	    this._storage = (_options$storage = options.storage) != null ? _options$storage : false;
	    this._cubemap = (_options$cubemap = options.cubemap) != null ? _options$cubemap : false;
	    this.fixCubemapSeams = (_options$fixCubemapSe = options.fixCubemapSeams) != null ? _options$fixCubemapSe : false;
	    this._flipY = (_options$flipY = options.flipY) != null ? _options$flipY : false;
	    this._premultiplyAlpha = (_options$premultiplyA = options.premultiplyAlpha) != null ? _options$premultiplyA : false;
	    this._mipmaps = (_ref = (_options$mipmaps = options.mipmaps) != null ? _options$mipmaps : options.autoMipmap) != null ? _ref : true;
	    this._minFilter = (_options$minFilter = options.minFilter) != null ? _options$minFilter : FILTER_LINEAR_MIPMAP_LINEAR;
	    this._magFilter = (_options$magFilter = options.magFilter) != null ? _options$magFilter : FILTER_LINEAR;
	    this._anisotropy = (_options$anisotropy = options.anisotropy) != null ? _options$anisotropy : 1;
	    this._addressU = (_options$addressU = options.addressU) != null ? _options$addressU : ADDRESS_REPEAT;
	    this._addressV = (_options$addressV = options.addressV) != null ? _options$addressV : ADDRESS_REPEAT;
	    this._addressW = (_options$addressW = options.addressW) != null ? _options$addressW : ADDRESS_REPEAT;
	    this._compareOnRead = (_options$compareOnRea = options.compareOnRead) != null ? _options$compareOnRea : false;
	    this._compareFunc = (_options$compareFunc = options.compareFunc) != null ? _options$compareFunc : FUNC_LESS;
	    this.type = TEXTURETYPE_DEFAULT;
	    if (options.hasOwnProperty('type')) {
	      this.type = options.type;
	    } else if (options.hasOwnProperty('rgbm')) {
	      Debug.deprecated("options.rgbm is deprecated. Use options.type instead.");
	      this.type = options.rgbm ? TEXTURETYPE_RGBM : TEXTURETYPE_DEFAULT;
	    } else if (options.hasOwnProperty('swizzleGGGR')) {
	      Debug.deprecated("options.swizzleGGGR is deprecated. Use options.type instead.");
	      this.type = options.swizzleGGGR ? TEXTURETYPE_SWIZZLEGGGR : TEXTURETYPE_DEFAULT;
	    }
	    this.projection = TEXTUREPROJECTION_NONE;
	    if (this._cubemap) {
	      this.projection = TEXTUREPROJECTION_CUBE;
	    } else if (options.projection && options.projection !== TEXTUREPROJECTION_CUBE) {
	      this.projection = options.projection;
	    }
	    this.impl = graphicsDevice.createTextureImpl(this);
	    this.profilerHint = (_options$profilerHint = options.profilerHint) != null ? _options$profilerHint : 0;
	    this.dirtyAll();
	    this._levels = options.levels;
	    if (this._levels) {
	      this.upload();
	    } else {
	      this._levels = this._cubemap ? [[null, null, null, null, null, null]] : [null];
	    }

	    // track the texture
	    graphicsDevice.textures.push(this);
	    Debug.trace(TRACEID_TEXTURE_ALLOC, "Alloc: Id " + this.id + " " + this.name + ": " + this.width + "x" + this.height + " " + ("" + (this.cubemap ? '[Cubemap]' : '')) + ("" + (this.volume ? '[Volume]' : '')) + ("" + (this.array ? '[Array]' : '')) + ("" + (this.mipmaps ? '[Mipmaps]' : '')), this);
	  }

	  /**
	   * Frees resources associated with this texture.
	   */
	  var _proto = Texture.prototype;
	  _proto.destroy = function destroy() {
	    Debug.trace(TRACEID_TEXTURE_ALLOC, "DeAlloc: Id " + this.id + " " + this.name);
	    var device = this.device;
	    if (device) {
	      // stop tracking the texture
	      var idx = device.textures.indexOf(this);
	      if (idx !== -1) {
	        device.textures.splice(idx, 1);
	      }

	      // Remove texture from any uniforms
	      device.scope.removeValue(this);

	      // destroy implementation
	      this.impl.destroy(device);

	      // Update texture stats
	      this.adjustVramSizeTracking(device._vram, -this._gpuSize);
	      this._levels = null;
	      this.device = null;
	    }
	  }

	  /**
	   * Resizes the texture. Only supported for render target textures, as it does not resize the
	   * existing content of the texture, but only the allocated buffer for rendering into.
	   *
	   * @param {number} width - The new width of the texture.
	   * @param {number} height - The new height of the texture.
	   * @param {number} [depth] - The new depth of the texture. Defaults to 1.
	   * @ignore
	   */;
	  _proto.resize = function resize(width, height, depth) {
	    if (depth === void 0) {
	      depth = 1;
	    }
	    // destroy texture impl
	    var device = this.device;
	    this.adjustVramSizeTracking(device._vram, -this._gpuSize);
	    this.impl.destroy(device);
	    this._width = Math.floor(width);
	    this._height = Math.floor(height);
	    this._depth = Math.floor(depth);

	    // re-create the implementation
	    this.impl = device.createTextureImpl(this);
	    this.dirtyAll();
	  }

	  /**
	   * Called when the rendering context was lost. It releases all context related resources.
	   *
	   * @ignore
	   */;
	  _proto.loseContext = function loseContext() {
	    this.impl.loseContext();
	    this.dirtyAll();
	  }

	  /**
	   * Updates vram size tracking for the texture, size can be positive to add or negative to subtract
	   *
	   * @ignore
	   */;
	  _proto.adjustVramSizeTracking = function adjustVramSizeTracking(vram, size) {
	    Debug.trace(TRACEID_VRAM_TEXTURE, this.id + " " + this.name + " size: " + size + " vram.texture: " + vram.tex + " => " + (vram.tex + size));
	    vram.tex += size;
	    if (this.profilerHint === TEXHINT_SHADOWMAP) {
	      vram.texShadow += size;
	    } else if (this.profilerHint === TEXHINT_ASSET) {
	      vram.texAsset += size;
	    } else if (this.profilerHint === TEXHINT_LIGHTMAP) {
	      vram.texLightmap += size;
	    }
	  };
	  _proto.propertyChanged = function propertyChanged(flag) {
	    this.impl.propertyChanged(flag);
	    this.renderVersionDirty = this.device.renderVersion;
	  }

	  /**
	   * Returns number of required mip levels for the texture based on its dimensions and parameters.
	   *
	   * @ignore
	   * @type {number}
	   */;
	  // Force a full resubmission of the texture to the GPU (used on a context restore event)
	  _proto.dirtyAll = function dirtyAll() {
	    this._levelsUpdated = this._cubemap ? [[true, true, true, true, true, true]] : [true];
	    this._needsUpload = true;
	    this._needsMipmapsUpload = this._mipmaps;
	    this._mipmapsUploaded = false;
	    this.propertyChanged(255); // 1 | 2 | 4 | 8 | 16 | 32 | 64 | 128
	  }

	  /**
	   * Locks a miplevel of the texture, returning a typed array to be filled with pixel data.
	   *
	   * @param {object} [options] - Optional options object. Valid properties are as follows:
	   * @param {number} [options.level] - The mip level to lock with 0 being the top level. Defaults
	   * to 0.
	   * @param {number} [options.face] - If the texture is a cubemap, this is the index of the face
	   * to lock.
	   * @param {number} [options.mode] - The lock mode. Can be:
	   * - {@link TEXTURELOCK_READ}
	   * - {@link TEXTURELOCK_WRITE}
	   * Defaults to {@link TEXTURELOCK_WRITE}.
	   * @returns {Uint8Array|Uint16Array|Float32Array} A typed array containing the pixel data of
	   * the locked mip level.
	   */;
	  _proto.lock = function lock(options) {
	    var _options, _options$level, _options2, _options2$face, _options3, _options3$mode;
	    if (options === void 0) {
	      options = {};
	    }
	    // Initialize options to some sensible defaults
	    (_options$level = (_options = options).level) != null ? _options$level : _options.level = 0;
	    (_options2$face = (_options2 = options).face) != null ? _options2$face : _options2.face = 0;
	    (_options3$mode = (_options3 = options).mode) != null ? _options3$mode : _options3.mode = TEXTURELOCK_WRITE;
	    Debug.assert(this._lockedMode === TEXTURELOCK_NONE, 'The texture is already locked. Call `texture.unlock()` before attempting to lock again.', this);
	    Debug.assert(options.mode === TEXTURELOCK_READ || options.mode === TEXTURELOCK_WRITE, 'Cannot lock a texture with TEXTURELOCK_NONE. To unlock a texture, call `texture.unlock()`.', this);
	    this._lockedMode = options.mode;
	    this._lockedLevel = options.level;
	    var levels = this.cubemap ? this._levels[options.face] : this._levels;
	    if (levels[options.level] === null) {
	      // allocate storage for this mip level
	      var width = Math.max(1, this._width >> options.level);
	      var height = Math.max(1, this._height >> options.level);
	      var depth = Math.max(1, this._depth >> options.level);
	      var data = new ArrayBuffer(TextureUtils.calcLevelGpuSize(width, height, depth, this._format));
	      levels[options.level] = new (getPixelFormatArrayType(this._format))(data);
	    }
	    return levels[options.level];
	  }

	  /**
	   * Set the pixel data of the texture from a canvas, image, video DOM element. If the texture is
	   * a cubemap, the supplied source must be an array of 6 canvases, images or videos.
	   *
	   * @param {HTMLCanvasElement|HTMLImageElement|HTMLVideoElement|HTMLCanvasElement[]|HTMLImageElement[]|HTMLVideoElement[]} source - A
	   * canvas, image or video element, or an array of 6 canvas, image or video elements.
	   * @param {number} [mipLevel] - A non-negative integer specifying the image level of detail.
	   * Defaults to 0, which represents the base image source. A level value of N, that is greater
	   * than 0, represents the image source for the Nth mipmap reduction level.
	   */;
	  _proto.setSource = function setSource(source, mipLevel) {
	    if (mipLevel === void 0) {
	      mipLevel = 0;
	    }
	    var invalid = false;
	    var width, height;
	    if (this._cubemap) {
	      if (source[0]) {
	        // rely on first face sizes
	        width = source[0].width || 0;
	        height = source[0].height || 0;
	        for (var i = 0; i < 6; i++) {
	          var face = source[i];
	          // cubemap becomes invalid if any condition is not satisfied
	          if (!face ||
	          // face is missing
	          face.width !== width ||
	          // face is different width
	          face.height !== height ||
	          // face is different height
	          !this.device._isBrowserInterface(face)) {
	            // new image bitmap
	            invalid = true;
	            break;
	          }
	        }
	      } else {
	        // first face is missing
	        invalid = true;
	      }
	      if (!invalid) {
	        // mark levels as updated
	        for (var _i = 0; _i < 6; _i++) {
	          if (this._levels[mipLevel][_i] !== source[_i]) this._levelsUpdated[mipLevel][_i] = true;
	        }
	      }
	    } else {
	      // check if source is valid type of element
	      if (!this.device._isBrowserInterface(source)) invalid = true;
	      if (!invalid) {
	        // mark level as updated
	        if (source !== this._levels[mipLevel]) this._levelsUpdated[mipLevel] = true;
	        width = source.width;
	        height = source.height;
	      }
	    }
	    if (invalid) {
	      // invalid texture

	      // default sizes
	      this._width = 4;
	      this._height = 4;

	      // remove levels
	      if (this._cubemap) {
	        for (var _i2 = 0; _i2 < 6; _i2++) {
	          this._levels[mipLevel][_i2] = null;
	          this._levelsUpdated[mipLevel][_i2] = true;
	        }
	      } else {
	        this._levels[mipLevel] = null;
	        this._levelsUpdated[mipLevel] = true;
	      }
	    } else {
	      // valid texture
	      if (mipLevel === 0) {
	        this._width = width;
	        this._height = height;
	      }
	      this._levels[mipLevel] = source;
	    }

	    // valid or changed state of validity
	    if (this._invalid !== invalid || !invalid) {
	      this._invalid = invalid;

	      // reupload
	      this.upload();
	    }
	  }

	  /**
	   * Get the pixel data of the texture. If this is a cubemap then an array of 6 images will be
	   * returned otherwise a single image.
	   *
	   * @param {number} [mipLevel] - A non-negative integer specifying the image level of detail.
	   * Defaults to 0, which represents the base image source. A level value of N, that is greater
	   * than 0, represents the image source for the Nth mipmap reduction level.
	   * @returns {HTMLImageElement} The source image of this texture. Can be null if source not
	   * assigned for specific image level.
	   */;
	  _proto.getSource = function getSource(mipLevel) {
	    if (mipLevel === void 0) {
	      mipLevel = 0;
	    }
	    return this._levels[mipLevel];
	  }

	  /**
	   * Unlocks the currently locked mip level and uploads it to VRAM.
	   */;
	  _proto.unlock = function unlock() {
	    if (this._lockedMode === TEXTURELOCK_NONE) {
	      Debug.warn("pc.Texture#unlock: Attempting to unlock a texture that is not locked.", this);
	    }

	    // Upload the new pixel data if locked in write mode (default)
	    if (this._lockedMode === TEXTURELOCK_WRITE) {
	      this.upload();
	    }
	    this._lockedLevel = -1;
	    this._lockedMode = TEXTURELOCK_NONE;
	  }

	  /**
	   * Forces a reupload of the textures pixel data to graphics memory. Ordinarily, this function
	   * is called by internally by {@link Texture#setSource} and {@link Texture#unlock}. However, it
	   * still needs to be called explicitly in the case where an HTMLVideoElement is set as the
	   * source of the texture.  Normally, this is done once every frame before video textured
	   * geometry is rendered.
	   */;
	  _proto.upload = function upload() {
	    var _this$impl$uploadImme, _this$impl;
	    this._needsUpload = true;
	    this._needsMipmapsUpload = this._mipmaps;
	    (_this$impl$uploadImme = (_this$impl = this.impl).uploadImmediate) == null || _this$impl$uploadImme.call(_this$impl, this.device, this);
	  }

	  /**
	   * Download texture's top level data from graphics memory to local memory.
	   *
	   * @ignore
	   */;
	  _proto.downloadAsync =
	  /*#__PURE__*/
	  function () {
	    var _downloadAsync = _asyncToGenerator( /*#__PURE__*/_regeneratorRuntime().mark(function _callee() {
	      var _this = this;
	      var promises, _loop, i;
	      return _regeneratorRuntime().wrap(function _callee$(_context2) {
	        while (1) switch (_context2.prev = _context2.next) {
	          case 0:
	            promises = [];
	            _loop = /*#__PURE__*/_regeneratorRuntime().mark(function _loop() {
	              var renderTarget, levels, level, promise;
	              return _regeneratorRuntime().wrap(function _loop$(_context) {
	                while (1) switch (_context.prev = _context.next) {
	                  case 0:
	                    renderTarget = new RenderTarget({
	                      colorBuffer: _this,
	                      depth: false,
	                      face: i
	                    });
	                    _this.device.setRenderTarget(renderTarget);
	                    _this.device.initRenderTarget(renderTarget);
	                    levels = _this.cubemap ? _this._levels[i] : _this._levels;
	                    level = levels[0];
	                    if (levels[0] && _this.device._isBrowserInterface(levels[0])) {
	                      levels[0] = null;
	                    }
	                    level = _this.lock({
	                      face: i
	                    });
	                    promise = _this.device.readPixelsAsync == null ? void 0 : _this.device.readPixelsAsync(0, 0, _this.width, _this.height, level).then(function () {
	                      return renderTarget.destroy();
	                    });
	                    promises.push(promise);
	                  case 9:
	                  case "end":
	                    return _context.stop();
	                }
	              }, _loop);
	            });
	            i = 0;
	          case 3:
	            if (!(i < (this.cubemap ? 6 : 1))) {
	              _context2.next = 8;
	              break;
	            }
	            return _context2.delegateYield(_loop(), "t0", 5);
	          case 5:
	            i++;
	            _context2.next = 3;
	            break;
	          case 8:
	            _context2.next = 10;
	            return Promise.all(promises);
	          case 10:
	          case "end":
	            return _context2.stop();
	        }
	      }, _callee, this);
	    }));
	    function downloadAsync() {
	      return _downloadAsync.apply(this, arguments);
	    }
	    return downloadAsync;
	  }();
	  _createClass(Texture, [{
	    key: "requiredMipLevels",
	    get: function get() {
	      return this.mipmaps ? TextureUtils.calcMipLevelsCount(this.width, this.height) : 1;
	    }

	    /**
	     * Returns the current lock mode. One of:
	     *
	     * - {@link TEXTURELOCK_NONE}
	     * - {@link TEXTURELOCK_READ}
	     * - {@link TEXTURELOCK_WRITE}
	     *
	     * @ignore
	     * @type {number}
	     */
	  }, {
	    key: "lockedMode",
	    get: function get() {
	      return this._lockedMode;
	    }

	    /**
	     * The minification filter to be applied to the texture. Can be:
	     *
	     * - {@link FILTER_NEAREST}
	     * - {@link FILTER_LINEAR}
	     * - {@link FILTER_NEAREST_MIPMAP_NEAREST}
	     * - {@link FILTER_NEAREST_MIPMAP_LINEAR}
	     * - {@link FILTER_LINEAR_MIPMAP_NEAREST}
	     * - {@link FILTER_LINEAR_MIPMAP_LINEAR}
	     *
	     * @type {number}
	     */
	  }, {
	    key: "minFilter",
	    get: function get() {
	      return this._minFilter;
	    }

	    /**
	     * The magnification filter to be applied to the texture. Can be:
	     *
	     * - {@link FILTER_NEAREST}
	     * - {@link FILTER_LINEAR}
	     *
	     * @type {number}
	     */,
	    set: function set(v) {
	      if (this._minFilter !== v) {
	        if (isIntegerPixelFormat(this._format)) {
	          Debug.warn("Texture#minFilter: minFilter property cannot be changed on an integer texture, will remain FILTER_NEAREST", this);
	        } else {
	          this._minFilter = v;
	          this.propertyChanged(1);
	        }
	      }
	    }
	  }, {
	    key: "magFilter",
	    get: function get() {
	      return this._magFilter;
	    }

	    /**
	     * The addressing mode to be applied to the texture horizontally. Can be:
	     *
	     * - {@link ADDRESS_REPEAT}
	     * - {@link ADDRESS_CLAMP_TO_EDGE}
	     * - {@link ADDRESS_MIRRORED_REPEAT}
	     *
	     * @type {number}
	     */,
	    set: function set(v) {
	      if (this._magFilter !== v) {
	        if (isIntegerPixelFormat(this._format)) {
	          Debug.warn("Texture#magFilter: magFilter property cannot be changed on an integer texture, will remain FILTER_NEAREST", this);
	        } else {
	          this._magFilter = v;
	          this.propertyChanged(2);
	        }
	      }
	    }
	  }, {
	    key: "addressU",
	    get: function get() {
	      return this._addressU;
	    }

	    /**
	     * The addressing mode to be applied to the texture vertically. Can be:
	     *
	     * - {@link ADDRESS_REPEAT}
	     * - {@link ADDRESS_CLAMP_TO_EDGE}
	     * - {@link ADDRESS_MIRRORED_REPEAT}
	     *
	     * @type {number}
	     */,
	    set: function set(v) {
	      if (this._addressU !== v) {
	        this._addressU = v;
	        this.propertyChanged(4);
	      }
	    }
	  }, {
	    key: "addressV",
	    get: function get() {
	      return this._addressV;
	    }

	    /**
	     * The addressing mode to be applied to the 3D texture depth (not supported on WebGL1). Can be:
	     *
	     * - {@link ADDRESS_REPEAT}
	     * - {@link ADDRESS_CLAMP_TO_EDGE}
	     * - {@link ADDRESS_MIRRORED_REPEAT}
	     *
	     * @type {number}
	     */,
	    set: function set(v) {
	      if (this._addressV !== v) {
	        this._addressV = v;
	        this.propertyChanged(8);
	      }
	    }
	  }, {
	    key: "addressW",
	    get: function get() {
	      return this._addressW;
	    }

	    /**
	     * When enabled, and if texture format is {@link PIXELFORMAT_DEPTH} or
	     * {@link PIXELFORMAT_DEPTHSTENCIL}, hardware PCF is enabled for this texture, and you can get
	     * filtered results of comparison using texture() in your shader (not supported on WebGL1).
	     *
	     * @type {boolean}
	     */,
	    set: function set(addressW) {
	      if (!this.device.supportsVolumeTextures) return;
	      if (!this._volume) {
	        Debug.warn("pc.Texture#addressW: Can't set W addressing mode for a non-3D texture.");
	        return;
	      }
	      if (addressW !== this._addressW) {
	        this._addressW = addressW;
	        this.propertyChanged(16);
	      }
	    }
	  }, {
	    key: "compareOnRead",
	    get: function get() {
	      return this._compareOnRead;
	    }

	    /**
	     * Comparison function when compareOnRead is enabled (not supported on WebGL1). Possible values:
	     *
	     * - {@link FUNC_LESS}
	     * - {@link FUNC_LESSEQUAL}
	     * - {@link FUNC_GREATER}
	     * - {@link FUNC_GREATEREQUAL}
	     * - {@link FUNC_EQUAL}
	     * - {@link FUNC_NOTEQUAL}
	     *
	     * @type {number}
	     */,
	    set: function set(v) {
	      if (this._compareOnRead !== v) {
	        this._compareOnRead = v;
	        this.propertyChanged(32);
	      }
	    }
	  }, {
	    key: "compareFunc",
	    get: function get() {
	      return this._compareFunc;
	    }

	    /**
	     * Integer value specifying the level of anisotropic to apply to the texture ranging from 1 (no
	     * anisotropic filtering) to the {@link GraphicsDevice} property maxAnisotropy.
	     *
	     * @type {number}
	     */,
	    set: function set(v) {
	      if (this._compareFunc !== v) {
	        this._compareFunc = v;
	        this.propertyChanged(64);
	      }
	    }
	  }, {
	    key: "anisotropy",
	    get: function get() {
	      return this._anisotropy;
	    }

	    /**
	     * Defines if texture should generate/upload mipmaps if possible.
	     *
	     * @type {boolean}
	     */,
	    set: function set(v) {
	      if (this._anisotropy !== v) {
	        this._anisotropy = v;
	        this.propertyChanged(128);
	      }
	    }
	  }, {
	    key: "mipmaps",
	    get: function get() {
	      return this._mipmaps;
	    }

	    /**
	     * Defines if texture can be used as a storage texture by a compute shader.
	     *
	     * @type {boolean}
	     */,
	    set: function set(v) {
	      if (this._mipmaps !== v) {
	        if (this.device.isWebGPU) {
	          Debug.warn("Texture#mipmaps: mipmap property is currently not allowed to be changed on WebGPU, create the texture appropriately.", this);
	        } else if (isIntegerPixelFormat(this._format)) {
	          Debug.warn("Texture#mipmaps: mipmap property cannot be changed on an integer texture, will remain false", this);
	        } else {
	          this._mipmaps = v;
	        }
	        if (v) this._needsMipmapsUpload = true;
	      }
	    }
	  }, {
	    key: "storage",
	    get: function get() {
	      return this._storage;
	    }

	    /**
	     * The width of the texture in pixels.
	     *
	     * @type {number}
	     */
	  }, {
	    key: "width",
	    get: function get() {
	      return this._width;
	    }

	    /**
	     * The height of the texture in pixels.
	     *
	     * @type {number}
	     */
	  }, {
	    key: "height",
	    get: function get() {
	      return this._height;
	    }

	    /**
	     * The number of depth slices in a 3D texture.
	     *
	     * @type {number}
	     */
	  }, {
	    key: "depth",
	    get: function get() {
	      return this._depth;
	    }

	    /**
	     * The pixel format of the texture. Can be:
	     *
	     * - {@link PIXELFORMAT_A8}
	     * - {@link PIXELFORMAT_L8}
	     * - {@link PIXELFORMAT_LA8}
	     * - {@link PIXELFORMAT_RGB565}
	     * - {@link PIXELFORMAT_RGBA5551}
	     * - {@link PIXELFORMAT_RGBA4}
	     * - {@link PIXELFORMAT_RGB8}
	     * - {@link PIXELFORMAT_RGBA8}
	     * - {@link PIXELFORMAT_DXT1}
	     * - {@link PIXELFORMAT_DXT3}
	     * - {@link PIXELFORMAT_DXT5}
	     * - {@link PIXELFORMAT_RGB16F}
	     * - {@link PIXELFORMAT_RGBA16F}
	     * - {@link PIXELFORMAT_RGB32F}
	     * - {@link PIXELFORMAT_RGBA32F}
	     * - {@link PIXELFORMAT_ETC1}
	     * - {@link PIXELFORMAT_PVRTC_2BPP_RGB_1}
	     * - {@link PIXELFORMAT_PVRTC_2BPP_RGBA_1}
	     * - {@link PIXELFORMAT_PVRTC_4BPP_RGB_1}
	     * - {@link PIXELFORMAT_PVRTC_4BPP_RGBA_1}
	     * - {@link PIXELFORMAT_111110F}
	     * - {@link PIXELFORMAT_ASTC_4x4}>/li>
	     * - {@link PIXELFORMAT_ATC_RGB}
	     * - {@link PIXELFORMAT_ATC_RGBA}
	     *
	     * @type {number}
	     */
	  }, {
	    key: "format",
	    get: function get() {
	      return this._format;
	    }

	    /**
	     * Returns true if this texture is a cube map and false otherwise.
	     *
	     * @type {boolean}
	     */
	  }, {
	    key: "cubemap",
	    get: function get() {
	      return this._cubemap;
	    }
	  }, {
	    key: "gpuSize",
	    get: function get() {
	      var mips = this.pot && this._mipmaps && !(this._compressed && this._levels.length === 1);
	      return TextureUtils.calcGpuSize(this._width, this._height, this._depth, this._format, mips, this._cubemap);
	    }

	    /**
	     * Returns true if this texture is a 2D texture array and false otherwise.
	     *
	     * @type {boolean}
	     */
	  }, {
	    key: "array",
	    get: function get() {
	      return this._arrayLength > 0;
	    }

	    /**
	     * Returns the number of textures inside this texture if this is a 2D array texture or 0 otherwise.
	     *
	     * @type {number}
	     */
	  }, {
	    key: "arrayLength",
	    get: function get() {
	      return this._arrayLength;
	    }

	    /**
	     * Returns true if this texture is a 3D volume and false otherwise.
	     *
	     * @type {boolean}
	     */
	  }, {
	    key: "volume",
	    get: function get() {
	      return this._volume;
	    }

	    /**
	     * Specifies whether the texture should be flipped in the Y-direction. Only affects textures
	     * with a source that is an image, canvas or video element. Does not affect cubemaps,
	     * compressed textures or textures set from raw pixel data. Defaults to true.
	     *
	     * @type {boolean}
	     */
	  }, {
	    key: "flipY",
	    get: function get() {
	      return this._flipY;
	    },
	    set: function set(flipY) {
	      if (this._flipY !== flipY) {
	        this._flipY = flipY;
	        this._needsUpload = true;
	      }
	    }
	  }, {
	    key: "premultiplyAlpha",
	    get: function get() {
	      return this._premultiplyAlpha;
	    }

	    /**
	     * Returns true if all dimensions of the texture are power of two, and false otherwise.
	     *
	     * @type {boolean}
	     */,
	    set: function set(premultiplyAlpha) {
	      if (this._premultiplyAlpha !== premultiplyAlpha) {
	        this._premultiplyAlpha = premultiplyAlpha;
	        this._needsUpload = true;
	      }
	    }
	  }, {
	    key: "pot",
	    get: function get() {
	      return math.powerOfTwo(this._width) && math.powerOfTwo(this._height);
	    }

	    // get the texture's encoding type
	  }, {
	    key: "encoding",
	    get: function get() {
	      switch (this.type) {
	        case TEXTURETYPE_RGBM:
	          return 'rgbm';
	        case TEXTURETYPE_RGBE:
	          return 'rgbe';
	        case TEXTURETYPE_RGBP:
	          return 'rgbp';
	        default:
	          return this.format === PIXELFORMAT_RGB16F || this.format === PIXELFORMAT_RGB32F || this.format === PIXELFORMAT_RGBA16F || this.format === PIXELFORMAT_RGBA32F || isIntegerPixelFormat(this.format) ? 'linear' : 'srgb';
	      }
	    }
	  }]);
	  return Texture;
	}();

	/**
	 * A WebGL implementation of the Buffer.
	 *
	 * @ignore
	 */
	var WebglBuffer = /*#__PURE__*/function () {
	  function WebglBuffer() {
	    this.bufferId = null;
	  }
	  var _proto = WebglBuffer.prototype;
	  _proto.destroy = function destroy(device) {
	    if (this.bufferId) {
	      device.gl.deleteBuffer(this.bufferId);
	      this.bufferId = null;
	    }
	  };
	  _proto.loseContext = function loseContext() {
	    this.bufferId = null;
	  };
	  _proto.unlock = function unlock(device, usage, target, storage) {
	    var gl = device.gl;
	    if (!this.bufferId) {
	      var glUsage;
	      switch (usage) {
	        case BUFFER_STATIC:
	          glUsage = gl.STATIC_DRAW;
	          break;
	        case BUFFER_DYNAMIC:
	          glUsage = gl.DYNAMIC_DRAW;
	          break;
	        case BUFFER_STREAM:
	          glUsage = gl.STREAM_DRAW;
	          break;
	        case BUFFER_GPUDYNAMIC:
	          glUsage = device.isWebGL2 ? gl.DYNAMIC_COPY : gl.STATIC_DRAW;
	          break;
	      }
	      this.bufferId = gl.createBuffer();
	      gl.bindBuffer(target, this.bufferId);
	      gl.bufferData(target, storage, glUsage);
	    } else {
	      gl.bindBuffer(target, this.bufferId);
	      gl.bufferSubData(target, 0, storage);
	    }
	  };
	  _createClass(WebglBuffer, [{
	    key: "initialized",
	    get: function get() {
	      return !!this.bufferId;
	    }
	  }]);
	  return WebglBuffer;
	}();

	/**
	 * A WebGL implementation of the VertexBuffer.
	 *
	 * @ignore
	 */
	var WebglVertexBuffer = /*#__PURE__*/function (_WebglBuffer) {
	  _inheritsLoose(WebglVertexBuffer, _WebglBuffer);
	  function WebglVertexBuffer() {
	    var _this;
	    for (var _len = arguments.length, args = new Array(_len), _key = 0; _key < _len; _key++) {
	      args[_key] = arguments[_key];
	    }
	    _this = _WebglBuffer.call.apply(_WebglBuffer, [this].concat(args)) || this;
	    // vertex array object
	    _this.vao = null;
	    return _this;
	  }
	  var _proto = WebglVertexBuffer.prototype;
	  _proto.destroy = function destroy(device) {
	    _WebglBuffer.prototype.destroy.call(this, device);

	    // clear up bound vertex buffers
	    device.unbindVertexArray();
	  };
	  _proto.loseContext = function loseContext() {
	    _WebglBuffer.prototype.loseContext.call(this);
	    this.vao = null;
	  };
	  _proto.unlock = function unlock(vertexBuffer) {
	    var device = vertexBuffer.device;
	    _WebglBuffer.prototype.unlock.call(this, device, vertexBuffer.usage, device.gl.ARRAY_BUFFER, vertexBuffer.storage);
	  };
	  return WebglVertexBuffer;
	}(WebglBuffer);

	/**
	 * A WebGL implementation of the IndexBuffer.
	 *
	 * @ignore
	 */
	var WebglIndexBuffer = /*#__PURE__*/function (_WebglBuffer) {
	  _inheritsLoose(WebglIndexBuffer, _WebglBuffer);
	  function WebglIndexBuffer(indexBuffer) {
	    var _this;
	    _this = _WebglBuffer.call(this) || this;
	    var gl = indexBuffer.device.gl;
	    var format = indexBuffer.format;
	    if (format === INDEXFORMAT_UINT8) {
	      _this.glFormat = gl.UNSIGNED_BYTE;
	    } else if (format === INDEXFORMAT_UINT16) {
	      _this.glFormat = gl.UNSIGNED_SHORT;
	    } else if (format === INDEXFORMAT_UINT32) {
	      _this.glFormat = gl.UNSIGNED_INT;
	    }
	    return _this;
	  }
	  var _proto = WebglIndexBuffer.prototype;
	  _proto.unlock = function unlock(indexBuffer) {
	    var device = indexBuffer.device;
	    _WebglBuffer.prototype.unlock.call(this, device, indexBuffer.usage, device.gl.ELEMENT_ARRAY_BUFFER, indexBuffer.storage);
	  };
	  return WebglIndexBuffer;
	}(WebglBuffer);

	/**
	 * Representation of a shader uniform.
	 *
	 * @ignore
	 */
	var WebglShaderInput =
	/**
	 * Create a new WebglShaderInput instance.
	 *
	 * @param {new Function("modulePath", "return import(modulePath)")('../graphics-device.js').GraphicsDevice} graphicsDevice - The graphics device
	 * used to manage this shader input.
	 * @param {string} name - The name of the shader input.
	 * @param {number} type - The type of the shader input.
	 * @param {number | WebGLUniformLocation} locationId - The location id of the shader input.
	 */
	function WebglShaderInput(graphicsDevice, name, type, locationId) {
	  // Set the shader attribute location
	  this.locationId = locationId;

	  // Resolve the ScopeId for the attribute name
	  this.scopeId = graphicsDevice.scope.resolve(name);

	  // Create the version
	  this.version = new Version();

	  // custom data type for arrays
	  if (name.substring(name.length - 3) === "[0]") {
	    switch (type) {
	      case UNIFORMTYPE_FLOAT:
	        type = UNIFORMTYPE_FLOATARRAY;
	        break;
	      case UNIFORMTYPE_INT:
	        type = UNIFORMTYPE_INTARRAY;
	        break;
	      case UNIFORMTYPE_UINT:
	        type = UNIFORMTYPE_UINTARRAY;
	        break;
	      case UNIFORMTYPE_BOOL:
	        type = UNIFORMTYPE_BOOLARRAY;
	        break;
	      case UNIFORMTYPE_VEC2:
	        type = UNIFORMTYPE_VEC2ARRAY;
	        break;
	      case UNIFORMTYPE_IVEC2:
	        type = UNIFORMTYPE_IVEC2ARRAY;
	        break;
	      case UNIFORMTYPE_UVEC2:
	        type = UNIFORMTYPE_UVEC2ARRAY;
	        break;
	      case UNIFORMTYPE_BVEC2:
	        type = UNIFORMTYPE_BVEC2ARRAY;
	        break;
	      case UNIFORMTYPE_VEC3:
	        type = UNIFORMTYPE_VEC3ARRAY;
	        break;
	      case UNIFORMTYPE_IVEC3:
	        type = UNIFORMTYPE_IVEC3ARRAY;
	        break;
	      case UNIFORMTYPE_UVEC3:
	        type = UNIFORMTYPE_UVEC3ARRAY;
	        break;
	      case UNIFORMTYPE_BVEC3:
	        type = UNIFORMTYPE_BVEC3ARRAY;
	        break;
	      case UNIFORMTYPE_VEC4:
	        type = UNIFORMTYPE_VEC4ARRAY;
	        break;
	      case UNIFORMTYPE_IVEC4:
	        type = UNIFORMTYPE_IVEC4ARRAY;
	        break;
	      case UNIFORMTYPE_UVEC4:
	        type = UNIFORMTYPE_UVEC4ARRAY;
	        break;
	      case UNIFORMTYPE_BVEC4:
	        type = UNIFORMTYPE_BVEC4ARRAY;
	        break;
	    }
	  }

	  // Set the data dataType
	  this.dataType = type;
	  this.value = [null, null, null, null];

	  // Array to hold texture unit ids
	  this.array = [];
	};

	var _totalCompileTime = 0;
	var _vertexShaderBuiltins = new Set(['gl_VertexID', 'gl_InstanceID', 'gl_DrawID', 'gl_BaseVertex', 'gl_BaseInstance']);

	// class used to hold compiled WebGL vertex or fragment shaders in the device cache
	var CompiledShaderCache = /*#__PURE__*/function () {
	  function CompiledShaderCache() {
	    // maps shader source to a compiled WebGL shader
	    this.map = new Map();
	  }
	  var _proto = CompiledShaderCache.prototype;
	  // destroy all created shaders when the device is destroyed
	  _proto.destroy = function destroy(device) {
	    this.map.forEach(function (shader) {
	      device.gl.deleteShader(shader);
	    });
	  }

	  // just empty the cache when the context is lost
	  ;
	  _proto.loseContext = function loseContext(device) {
	    this.map.clear();
	  };
	  return CompiledShaderCache;
	}();
	var _vertexShaderCache = new DeviceCache();
	var _fragmentShaderCache = new DeviceCache();

	/**
	 * A WebGL implementation of the Shader.
	 *
	 * @ignore
	 */
	var WebglShader = /*#__PURE__*/function () {
	  function WebglShader(shader) {
	    this.compileDuration = 0;
	    this.init();

	    // kick off vertex and fragment shader compilation
	    this.compile(shader.device, shader);

	    // kick off linking, as this is non-blocking too
	    this.link(shader.device, shader);

	    // add it to a device list of all shaders
	    shader.device.shaders.push(shader);
	  }

	  /**
	   * Free the WebGL resources associated with a shader.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('../shader.js').Shader} shader - The shader to free.
	   */
	  var _proto2 = WebglShader.prototype;
	  _proto2.destroy = function destroy(shader) {
	    if (this.glProgram) {
	      shader.device.gl.deleteProgram(this.glProgram);
	      this.glProgram = null;
	    }
	  };
	  _proto2.init = function init() {
	    this.uniforms = [];
	    this.samplers = [];
	    this.attributes = [];
	    this.glProgram = null;
	    this.glVertexShader = null;
	    this.glFragmentShader = null;
	  }

	  /**
	   * Dispose the shader when the context has been lost.
	   */;
	  _proto2.loseContext = function loseContext() {
	    this.init();
	  }

	  /**
	   * Restore shader after the context has been obtained.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./webgl-graphics-device.js').WebglGraphicsDevice} device - The graphics device.
	   * @param {new Function("modulePath", "return import(modulePath)")('../shader.js').Shader} shader - The shader to restore.
	   */;
	  _proto2.restoreContext = function restoreContext(device, shader) {
	    this.compile(device, shader);
	    this.link(device, shader);
	  }

	  /**
	   * Compile shader programs.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./webgl-graphics-device.js').WebglGraphicsDevice} device - The graphics device.
	   * @param {new Function("modulePath", "return import(modulePath)")('../shader.js').Shader} shader - The shader to compile.
	   */;
	  _proto2.compile = function compile(device, shader) {
	    var definition = shader.definition;
	    this.glVertexShader = this._compileShaderSource(device, definition.vshader, true);
	    this.glFragmentShader = this._compileShaderSource(device, definition.fshader, false);
	  }

	  /**
	   * Link shader programs. This is called at a later stage, to allow many shaders to compile in parallel.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./webgl-graphics-device.js').WebglGraphicsDevice} device - The graphics device.
	   * @param {new Function("modulePath", "return import(modulePath)")('../shader.js').Shader} shader - The shader to compile.
	   */;
	  _proto2.link = function link(device, shader) {
	    var _this = this;
	    // if the shader was already linked
	    if (this.glProgram) return;

	    // if the device is lost, silently ignore
	    var gl = device.gl;
	    if (gl.isContextLost()) {
	      return;
	    }
	    var startTime = 0;
	    Debug.call(function () {
	      _this.compileDuration = 0;
	      startTime = now();
	    });
	    var glProgram = gl.createProgram();
	    this.glProgram = glProgram;
	    gl.attachShader(glProgram, this.glVertexShader);
	    gl.attachShader(glProgram, this.glFragmentShader);
	    var definition = shader.definition;
	    var attrs = definition.attributes;
	    if (device.isWebGL2 && definition.useTransformFeedback) {
	      // Collect all "out_" attributes and use them for output
	      var outNames = [];
	      for (var attr in attrs) {
	        if (attrs.hasOwnProperty(attr)) {
	          outNames.push("out_" + attr);
	        }
	      }
	      gl.transformFeedbackVaryings(glProgram, outNames, gl.INTERLEAVED_ATTRIBS);
	    }

	    // map all vertex input attributes to fixed locations
	    var locations = {};
	    for (var _attr in attrs) {
	      if (attrs.hasOwnProperty(_attr)) {
	        var semantic = attrs[_attr];
	        var loc = semanticToLocation[semantic];
	        Debug.assert(!locations.hasOwnProperty(loc), "WARNING: Two attributes are mapped to the same location in a shader: " + locations[loc] + " and " + _attr);
	        locations[loc] = _attr;
	        gl.bindAttribLocation(glProgram, loc, _attr);
	      }
	    }
	    gl.linkProgram(glProgram);
	    Debug.call(function () {
	      _this.compileDuration = now() - startTime;
	    });
	    device._shaderStats.linked++;
	    if (definition.tag === SHADERTAG_MATERIAL) {
	      device._shaderStats.materialShaders++;
	    }
	  }

	  /**
	   * Compiles an individual shader.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./webgl-graphics-device.js').WebglGraphicsDevice} device - The graphics device.
	   * @param {string} src - The shader source code.
	   * @param {boolean} isVertexShader - True if the shader is a vertex shader, false if it is a
	   * fragment shader.
	   * @returns {WebGLShader} The compiled shader.
	   * @private
	   */;
	  _proto2._compileShaderSource = function _compileShaderSource(device, src, isVertexShader) {
	    var gl = device.gl;

	    // device cache for current device, containing cache of compiled shaders
	    var shaderDeviceCache = isVertexShader ? _vertexShaderCache : _fragmentShaderCache;
	    var shaderCache = shaderDeviceCache.get(device, function () {
	      return new CompiledShaderCache();
	    });

	    // try to get compiled shader from the cache
	    var glShader = shaderCache.map.get(src);
	    if (!glShader) {
	      var startTime = now();
	      device.fire('shader:compile:start', {
	        timestamp: startTime,
	        target: device
	      });
	      glShader = gl.createShader(isVertexShader ? gl.VERTEX_SHADER : gl.FRAGMENT_SHADER);

	      // if the device is lost, silently ignore
	      if (!glShader && gl.isContextLost()) {
	        return glShader;
	      }
	      gl.shaderSource(glShader, src);
	      gl.compileShader(glShader);
	      shaderCache.map.set(src, glShader);
	      var endTime = now();
	      device.fire('shader:compile:end', {
	        timestamp: endTime,
	        target: device
	      });
	      device._shaderStats.compileTime += endTime - startTime;
	      if (isVertexShader) {
	        device._shaderStats.vsCompiled++;
	      } else {
	        device._shaderStats.fsCompiled++;
	      }
	    }
	    return glShader;
	  }

	  /**
	   * Link the shader, and extract its attributes and uniform information.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./webgl-graphics-device.js').WebglGraphicsDevice} device - The graphics device.
	   * @param {new Function("modulePath", "return import(modulePath)")('../shader.js').Shader} shader - The shader to query.
	   * @returns {boolean} True if the shader was successfully queried and false otherwise.
	   */;
	  _proto2.finalize = function finalize(device, shader) {
	    var _this2 = this;
	    // if the device is lost, silently ignore
	    var gl = device.gl;
	    if (gl.isContextLost()) {
	      return true;
	    }
	    var glProgram = this.glProgram;
	    var definition = shader.definition;
	    var startTime = now();
	    device.fire('shader:link:start', {
	      timestamp: startTime,
	      target: device
	    });

	    // this is the main thead blocking part of the shader compilation, time it
	    var linkStartTime = 0;
	    Debug.call(function () {
	      linkStartTime = now();
	    });

	    // check the link status of a shader - this is a blocking operation waiting for the shader
	    // to finish compiling and linking
	    var linkStatus = gl.getProgramParameter(glProgram, gl.LINK_STATUS);
	    if (!linkStatus) {
	      var _gl$getExtension, _gl$getExtension2;
	      // Check for compilation errors
	      if (!this._isCompiled(device, shader, this.glVertexShader, definition.vshader, "vertex")) return false;
	      if (!this._isCompiled(device, shader, this.glFragmentShader, definition.fshader, "fragment")) return false;
	      var message = "Failed to link shader program. Error: " + gl.getProgramInfoLog(glProgram);

	      // log translated shaders
	      definition.translatedFrag = (_gl$getExtension = gl.getExtension('WEBGL_debug_shaders')) == null ? void 0 : _gl$getExtension.getTranslatedShaderSource(this.glFragmentShader);
	      definition.translatedVert = (_gl$getExtension2 = gl.getExtension('WEBGL_debug_shaders')) == null ? void 0 : _gl$getExtension2.getTranslatedShaderSource(this.glVertexShader);
	      console.error(message, definition);
	      return false;
	    }

	    // Query the program for each vertex buffer input (GLSL 'attribute')
	    var numAttributes = gl.getProgramParameter(glProgram, gl.ACTIVE_ATTRIBUTES);
	    for (var i = 0; i < numAttributes; i++) {
	      var info = gl.getActiveAttrib(glProgram, i);
	      var location = gl.getAttribLocation(glProgram, info.name);

	      // a built-in attributes for which we do not need to provide any data
	      if (_vertexShaderBuiltins.has(info.name)) continue;

	      // Check attributes are correctly linked up
	      if (definition.attributes[info.name] === undefined) {
	        console.error("Vertex shader attribute \"" + info.name + "\" is not mapped to a semantic in shader definition, shader [" + shader.label + "]", shader);
	        shader.failed = true;
	      } else {
	        var shaderInput = new WebglShaderInput(device, definition.attributes[info.name], device.pcUniformType[info.type], location);
	        this.attributes.push(shaderInput);
	      }
	    }

	    // Query the program for each shader state (GLSL 'uniform')
	    var samplerTypes = device._samplerTypes;
	    var numUniforms = gl.getProgramParameter(glProgram, gl.ACTIVE_UNIFORMS);
	    for (var _i = 0; _i < numUniforms; _i++) {
	      var _info = gl.getActiveUniform(glProgram, _i);
	      var _location = gl.getUniformLocation(glProgram, _info.name);
	      var _shaderInput = new WebglShaderInput(device, _info.name, device.pcUniformType[_info.type], _location);
	      if (samplerTypes.has(_info.type)) {
	        this.samplers.push(_shaderInput);
	      } else {
	        this.uniforms.push(_shaderInput);
	      }
	    }
	    shader.ready = true;
	    var endTime = now();
	    device.fire('shader:link:end', {
	      timestamp: endTime,
	      target: device
	    });
	    device._shaderStats.compileTime += endTime - startTime;
	    Debug.call(function () {
	      var duration = now() - linkStartTime;
	      _this2.compileDuration += duration;
	      _totalCompileTime += _this2.compileDuration;
	      Debug.trace(TRACEID_SHADER_COMPILE, "[id: " + shader.id + "] " + shader.name + ": " + _this2.compileDuration.toFixed(1) + "ms, TOTAL: " + _totalCompileTime.toFixed(1) + "ms");
	    });
	    return true;
	  }

	  /**
	   * Check the compilation status of a shader.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./webgl-graphics-device.js').WebglGraphicsDevice} device - The graphics device.
	   * @param {new Function("modulePath", "return import(modulePath)")('../shader.js').Shader} shader - The shader to query.
	   * @param {WebGLShader} glShader - The WebGL shader.
	   * @param {string} source - The shader source code.
	   * @param {string} shaderType - The shader type. Can be 'vertex' or 'fragment'.
	   * @returns {boolean} True if the shader compiled successfully, false otherwise.
	   * @private
	   */;
	  _proto2._isCompiled = function _isCompiled(device, shader, glShader, source, shaderType) {
	    var gl = device.gl;
	    if (!gl.getShaderParameter(glShader, gl.COMPILE_STATUS)) {
	      var infoLog = gl.getShaderInfoLog(glShader);
	      var _this$_processError = this._processError(source, infoLog),
	        code = _this$_processError[0],
	        error = _this$_processError[1];
	      var message = "Failed to compile " + shaderType + " shader:\n\n" + infoLog + "\n" + code + " while rendering " + DebugGraphics.toString();
	      error.shader = shader;
	      console.error(message, error);
	      return false;
	    }
	    return true;
	  }

	  /**
	   * Check the linking status of a shader.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./webgl-graphics-device.js').WebglGraphicsDevice} device - The graphics device.
	   * @returns {boolean} True if the shader is already linked, false otherwise. Note that unless the
	   * device supports the KHR_parallel_shader_compile extension, this will always return true.
	   */;
	  _proto2.isLinked = function isLinked(device) {
	    var extParallelShaderCompile = device.extParallelShaderCompile;
	    if (extParallelShaderCompile) {
	      return device.gl.getProgramParameter(this.glProgram, extParallelShaderCompile.COMPLETION_STATUS_KHR);
	    }
	    return true;
	  }

	  /**
	   * Truncate the WebGL shader compilation log to just include the error line plus the 5 lines
	   * before and after it.
	   *
	   * @param {string} src - The shader source code.
	   * @param {string} infoLog - The info log returned from WebGL on a failed shader compilation.
	   * @returns {Array} An array where the first element is the 10 lines of code around the first
	   * detected error, and the second element an object storing the error message, line number and
	   * complete shader source.
	   * @private
	   */;
	  _proto2._processError = function _processError(src, infoLog) {
	    var error = {};
	    var code = '';
	    if (src) {
	      var lines = src.split('\n');
	      var from = 0;
	      var to = lines.length;

	      // if error is in the code, only show nearby lines instead of whole shader code
	      if (infoLog && infoLog.startsWith('ERROR:')) {
	        var match = infoLog.match(/^ERROR:\s([0-9]+):([0-9]+):\s*(.+)/);
	        if (match) {
	          error.message = match[3];
	          error.line = parseInt(match[2], 10);
	          from = Math.max(0, error.line - 6);
	          to = Math.min(lines.length, error.line + 5);
	        }
	      }

	      // Chrome reports shader errors on lines indexed from 1
	      for (var i = from; i < to; i++) {
	        code += i + 1 + ":\t" + lines[i] + '\n';
	      }
	      error.source = src;
	    }
	    return [code, error];
	  };
	  return WebglShader;
	}();

	/**
	 * Checks that an image's width and height do not exceed the max texture size. If they do, it will
	 * be scaled down to that maximum size and returned as a canvas element.
	 *
	 * @param {HTMLImageElement} image - The image to downsample.
	 * @param {number} size - The maximum allowed size of the image.
	 * @returns {HTMLImageElement|HTMLCanvasElement} The downsampled image.
	 * @ignore
	 */
	function downsampleImage(image, size) {
	  var srcW = image.width;
	  var srcH = image.height;
	  if (srcW > size || srcH > size) {
	    var scale = size / Math.max(srcW, srcH);
	    var dstW = Math.floor(srcW * scale);
	    var dstH = Math.floor(srcH * scale);
	    Debug.warn("Image dimensions larger than max supported texture size of " + size + ". Resizing from " + srcW + ", " + srcH + " to " + dstW + ", " + dstH + ".");
	    var canvas = document.createElement('canvas');
	    canvas.width = dstW;
	    canvas.height = dstH;
	    var context = canvas.getContext('2d');
	    context.drawImage(image, 0, 0, srcW, srcH, 0, 0, dstW, dstH);
	    return canvas;
	  }
	  return image;
	}

	/**
	 * A WebGL implementation of the Texture.
	 *
	 * @ignore
	 */
	var WebglTexture = /*#__PURE__*/function () {
	  function WebglTexture() {
	    this._glTexture = null;
	    this._glTarget = void 0;
	    this._glFormat = void 0;
	    this._glInternalFormat = void 0;
	    this._glPixelType = void 0;
	    this._glCreated = void 0;
	    this.dirtyParameterFlags = 0;
	  }
	  var _proto = WebglTexture.prototype;
	  _proto.destroy = function destroy(device) {
	    if (this._glTexture) {
	      // Update shadowed texture unit state to remove texture from any units
	      for (var i = 0; i < device.textureUnits.length; i++) {
	        var textureUnit = device.textureUnits[i];
	        for (var j = 0; j < textureUnit.length; j++) {
	          if (textureUnit[j] === this._glTexture) {
	            textureUnit[j] = null;
	          }
	        }
	      }

	      // release WebGL texture resource
	      device.gl.deleteTexture(this._glTexture);
	      this._glTexture = null;
	    }
	  };
	  _proto.loseContext = function loseContext() {
	    this._glTexture = null;
	  };
	  _proto.propertyChanged = function propertyChanged(flag) {
	    this.dirtyParameterFlags |= flag;
	  };
	  _proto.initialize = function initialize(device, texture) {
	    var gl = device.gl;
	    this._glTexture = gl.createTexture();
	    this._glTarget = texture._cubemap ? gl.TEXTURE_CUBE_MAP : texture._volume ? gl.TEXTURE_3D : texture.array ? gl.TEXTURE_2D_ARRAY : gl.TEXTURE_2D;
	    switch (texture._format) {
	      case PIXELFORMAT_A8:
	        this._glFormat = gl.ALPHA;
	        this._glInternalFormat = gl.ALPHA;
	        this._glPixelType = gl.UNSIGNED_BYTE;
	        break;
	      case PIXELFORMAT_L8:
	        this._glFormat = gl.LUMINANCE;
	        this._glInternalFormat = gl.LUMINANCE;
	        this._glPixelType = gl.UNSIGNED_BYTE;
	        break;
	      case PIXELFORMAT_LA8:
	        this._glFormat = gl.LUMINANCE_ALPHA;
	        this._glInternalFormat = gl.LUMINANCE_ALPHA;
	        this._glPixelType = gl.UNSIGNED_BYTE;
	        break;
	      case PIXELFORMAT_RGB565:
	        this._glFormat = gl.RGB;
	        this._glInternalFormat = gl.RGB;
	        this._glPixelType = gl.UNSIGNED_SHORT_5_6_5;
	        break;
	      case PIXELFORMAT_RGBA5551:
	        this._glFormat = gl.RGBA;
	        this._glInternalFormat = gl.RGBA;
	        this._glPixelType = gl.UNSIGNED_SHORT_5_5_5_1;
	        break;
	      case PIXELFORMAT_RGBA4:
	        this._glFormat = gl.RGBA;
	        this._glInternalFormat = gl.RGBA;
	        this._glPixelType = gl.UNSIGNED_SHORT_4_4_4_4;
	        break;
	      case PIXELFORMAT_RGB8:
	        this._glFormat = gl.RGB;
	        this._glInternalFormat = device.isWebGL2 ? gl.RGB8 : gl.RGB;
	        this._glPixelType = gl.UNSIGNED_BYTE;
	        break;
	      case PIXELFORMAT_RGBA8:
	        this._glFormat = gl.RGBA;
	        this._glInternalFormat = device.isWebGL2 ? gl.RGBA8 : gl.RGBA;
	        this._glPixelType = gl.UNSIGNED_BYTE;
	        break;
	      case PIXELFORMAT_DXT1:
	        this._glFormat = gl.RGB;
	        this._glInternalFormat = device.extCompressedTextureS3TC.COMPRESSED_RGB_S3TC_DXT1_EXT;
	        break;
	      case PIXELFORMAT_DXT3:
	        this._glFormat = gl.RGBA;
	        this._glInternalFormat = device.extCompressedTextureS3TC.COMPRESSED_RGBA_S3TC_DXT3_EXT;
	        break;
	      case PIXELFORMAT_DXT5:
	        this._glFormat = gl.RGBA;
	        this._glInternalFormat = device.extCompressedTextureS3TC.COMPRESSED_RGBA_S3TC_DXT5_EXT;
	        break;
	      case PIXELFORMAT_ETC1:
	        this._glFormat = gl.RGB;
	        this._glInternalFormat = device.extCompressedTextureETC1.COMPRESSED_RGB_ETC1_WEBGL;
	        break;
	      case PIXELFORMAT_PVRTC_2BPP_RGB_1:
	        this._glFormat = gl.RGB;
	        this._glInternalFormat = device.extCompressedTexturePVRTC.COMPRESSED_RGB_PVRTC_2BPPV1_IMG;
	        break;
	      case PIXELFORMAT_PVRTC_2BPP_RGBA_1:
	        this._glFormat = gl.RGBA;
	        this._glInternalFormat = device.extCompressedTexturePVRTC.COMPRESSED_RGBA_PVRTC_2BPPV1_IMG;
	        break;
	      case PIXELFORMAT_PVRTC_4BPP_RGB_1:
	        this._glFormat = gl.RGB;
	        this._glInternalFormat = device.extCompressedTexturePVRTC.COMPRESSED_RGB_PVRTC_4BPPV1_IMG;
	        break;
	      case PIXELFORMAT_PVRTC_4BPP_RGBA_1:
	        this._glFormat = gl.RGBA;
	        this._glInternalFormat = device.extCompressedTexturePVRTC.COMPRESSED_RGBA_PVRTC_4BPPV1_IMG;
	        break;
	      case PIXELFORMAT_ETC2_RGB:
	        this._glFormat = gl.RGB;
	        this._glInternalFormat = device.extCompressedTextureETC.COMPRESSED_RGB8_ETC2;
	        break;
	      case PIXELFORMAT_ETC2_RGBA:
	        this._glFormat = gl.RGBA;
	        this._glInternalFormat = device.extCompressedTextureETC.COMPRESSED_RGBA8_ETC2_EAC;
	        break;
	      case PIXELFORMAT_ASTC_4x4:
	        this._glFormat = gl.RGBA;
	        this._glInternalFormat = device.extCompressedTextureASTC.COMPRESSED_RGBA_ASTC_4x4_KHR;
	        break;
	      case PIXELFORMAT_ATC_RGB:
	        this._glFormat = gl.RGB;
	        this._glInternalFormat = device.extCompressedTextureATC.COMPRESSED_RGB_ATC_WEBGL;
	        break;
	      case PIXELFORMAT_ATC_RGBA:
	        this._glFormat = gl.RGBA;
	        this._glInternalFormat = device.extCompressedTextureATC.COMPRESSED_RGBA_ATC_INTERPOLATED_ALPHA_WEBGL;
	        break;
	      case PIXELFORMAT_R16F:
	        if (device.isWebGL2) {
	          this._glFormat = gl.RED;
	          this._glInternalFormat = gl.R16F;
	          this._glPixelType = gl.HALF_FLOAT;
	        } else {
	          this._glFormat = gl.LUMINANCE;
	          this._glInternalFormat = gl.LUMINANCE;
	          this._glPixelType = device.extTextureHalfFloat.HALF_FLOAT_OES;
	        }
	        break;
	      case PIXELFORMAT_RG16F:
	        if (device.isWebGL2) {
	          this._glFormat = gl.RG;
	          this._glInternalFormat = gl.RG16F;
	          this._glPixelType = gl.HALF_FLOAT;
	        } else {
	          this._glFormat = gl.RG;
	          this._glInternalFormat = gl.RG;
	          this._glPixelType = device.extTextureHalfFloat.HALF_FLOAT_OES;
	        }
	        break;
	      case PIXELFORMAT_RGB16F:
	        // definition varies between WebGL1 and 2
	        this._glFormat = gl.RGB;
	        if (device.isWebGL2) {
	          this._glInternalFormat = gl.RGB16F;
	          this._glPixelType = gl.HALF_FLOAT;
	        } else {
	          this._glInternalFormat = gl.RGB;
	          this._glPixelType = device.extTextureHalfFloat.HALF_FLOAT_OES;
	        }
	        break;
	      case PIXELFORMAT_RGBA16F:
	        // definition varies between WebGL1 and 2
	        this._glFormat = gl.RGBA;
	        if (device.isWebGL2) {
	          this._glInternalFormat = gl.RGBA16F;
	          this._glPixelType = gl.HALF_FLOAT;
	        } else {
	          this._glInternalFormat = gl.RGBA;
	          this._glPixelType = device.extTextureHalfFloat.HALF_FLOAT_OES;
	        }
	        break;
	      case PIXELFORMAT_RGB32F:
	        // definition varies between WebGL1 and 2
	        this._glFormat = gl.RGB;
	        if (device.isWebGL2) {
	          this._glInternalFormat = gl.RGB32F;
	        } else {
	          this._glInternalFormat = gl.RGB;
	        }
	        this._glPixelType = gl.FLOAT;
	        break;
	      case PIXELFORMAT_RGBA32F:
	        // definition varies between WebGL1 and 2
	        this._glFormat = gl.RGBA;
	        if (device.isWebGL2) {
	          this._glInternalFormat = gl.RGBA32F;
	        } else {
	          this._glInternalFormat = gl.RGBA;
	        }
	        this._glPixelType = gl.FLOAT;
	        break;
	      case PIXELFORMAT_R32F:
	        // WebGL2 only
	        this._glFormat = gl.RED;
	        this._glInternalFormat = gl.R32F;
	        this._glPixelType = gl.FLOAT;
	        break;
	      case PIXELFORMAT_DEPTH:
	        if (device.isWebGL2) {
	          // native WebGL2
	          this._glFormat = gl.DEPTH_COMPONENT;
	          this._glInternalFormat = gl.DEPTH_COMPONENT32F; // should allow 16/24 bits?
	          this._glPixelType = gl.FLOAT;
	        } else {
	          // using WebGL1 extension
	          this._glFormat = gl.DEPTH_COMPONENT;
	          this._glInternalFormat = gl.DEPTH_COMPONENT;
	          this._glPixelType = gl.UNSIGNED_SHORT; // the only acceptable value?
	        }

	        break;
	      case PIXELFORMAT_DEPTHSTENCIL:
	        this._glFormat = gl.DEPTH_STENCIL;
	        if (device.isWebGL2) {
	          this._glInternalFormat = gl.DEPTH24_STENCIL8;
	          this._glPixelType = gl.UNSIGNED_INT_24_8;
	        } else {
	          this._glInternalFormat = gl.DEPTH_STENCIL;
	          this._glPixelType = device.extDepthTexture.UNSIGNED_INT_24_8_WEBGL;
	        }
	        break;
	      case PIXELFORMAT_111110F:
	        // WebGL2 only
	        Debug.assert(device.isWebGL2, "PIXELFORMAT_111110F texture format is not supported by WebGL1.");
	        this._glFormat = gl.RGB;
	        this._glInternalFormat = gl.R11F_G11F_B10F;
	        this._glPixelType = gl.UNSIGNED_INT_10F_11F_11F_REV;
	        break;
	      case PIXELFORMAT_SRGB:
	        // WebGL2 only
	        this._glFormat = gl.RGB;
	        this._glInternalFormat = gl.SRGB8;
	        this._glPixelType = gl.UNSIGNED_BYTE;
	        break;
	      case PIXELFORMAT_SRGBA:
	        // WebGL2 only
	        this._glFormat = gl.RGBA;
	        this._glInternalFormat = gl.SRGB8_ALPHA8;
	        this._glPixelType = gl.UNSIGNED_BYTE;
	        break;
	      // Integer texture formats (R) (WebGL2 only)
	      case PIXELFORMAT_R8I:
	        // WebGL2 only
	        this._glFormat = gl.RED_INTEGER;
	        this._glInternalFormat = gl.R8I;
	        this._glPixelType = gl.BYTE;
	        break;
	      case PIXELFORMAT_R8U:
	        // WebGL2 only
	        this._glFormat = gl.RED_INTEGER;
	        this._glInternalFormat = gl.R8UI;
	        this._glPixelType = gl.UNSIGNED_BYTE;
	        break;
	      case PIXELFORMAT_R16I:
	        // WebGL2 only
	        this._glFormat = gl.RED_INTEGER;
	        this._glInternalFormat = gl.R16I;
	        this._glPixelType = gl.SHORT;
	        break;
	      case PIXELFORMAT_R16U:
	        // WebGL2 only
	        this._glFormat = gl.RED_INTEGER;
	        this._glInternalFormat = gl.R16UI;
	        this._glPixelType = gl.UNSIGNED_SHORT;
	        break;
	      case PIXELFORMAT_R32I:
	        // WebGL2 only
	        this._glFormat = gl.RED_INTEGER;
	        this._glInternalFormat = gl.R32I;
	        this._glPixelType = gl.INT;
	        break;
	      case PIXELFORMAT_R32U:
	        // WebGL2 only
	        this._glFormat = gl.RED_INTEGER;
	        this._glInternalFormat = gl.R32UI;
	        this._glPixelType = gl.UNSIGNED_INT;
	        break;
	      // Integer texture formats (RG) (WebGL2 only)
	      case PIXELFORMAT_RG8I:
	        // WebGL2 only
	        this._glFormat = gl.RG_INTEGER;
	        this._glInternalFormat = gl.RG8I;
	        this._glPixelType = gl.BYTE;
	        break;
	      case PIXELFORMAT_RG8U:
	        // WebGL2 only
	        this._glFormat = gl.RG_INTEGER;
	        this._glInternalFormat = gl.RG8UI;
	        this._glPixelType = gl.UNSIGNED_BYTE;
	        break;
	      case PIXELFORMAT_RG16I:
	        // WebGL2 only
	        this._glFormat = gl.RG_INTEGER;
	        this._glInternalFormat = gl.RG16I;
	        this._glPixelType = gl.SHORT;
	        break;
	      case PIXELFORMAT_RG16U:
	        // WebGL2 only
	        this._glFormat = gl.RG_INTEGER;
	        this._glInternalFormat = gl.RG16UI;
	        this._glPixelType = gl.UNSIGNED_SHORT;
	        break;
	      case PIXELFORMAT_RG32I:
	        // WebGL2 only
	        this._glFormat = gl.RG_INTEGER;
	        this._glInternalFormat = gl.RG32I;
	        this._glPixelType = gl.INT;
	        break;
	      case PIXELFORMAT_RG32U:
	        // WebGL2 only
	        this._glFormat = gl.RG_INTEGER;
	        this._glInternalFormat = gl.RG32UI;
	        this._glPixelType = gl.UNSIGNED_INT;
	        break;
	      // Integer texture formats (RGBA) (WebGL2 only)
	      case PIXELFORMAT_RGBA8I:
	        // WebGL2 only
	        this._glFormat = gl.RGBA_INTEGER;
	        this._glInternalFormat = gl.RGBA8I;
	        this._glPixelType = gl.BYTE;
	        break;
	      case PIXELFORMAT_RGBA8U:
	        // WebGL2 only
	        this._glFormat = gl.RGBA_INTEGER;
	        this._glInternalFormat = gl.RGBA8UI;
	        this._glPixelType = gl.UNSIGNED_BYTE;
	        break;
	      case PIXELFORMAT_RGBA16I:
	        // WebGL2 only
	        this._glFormat = gl.RGBA_INTEGER;
	        this._glInternalFormat = gl.RGBA16I;
	        this._glPixelType = gl.SHORT;
	        break;
	      case PIXELFORMAT_RGBA16U:
	        // WebGL2 only
	        this._glFormat = gl.RGBA_INTEGER;
	        this._glInternalFormat = gl.RGBA16UI;
	        this._glPixelType = gl.UNSIGNED_SHORT;
	        break;
	      case PIXELFORMAT_RGBA32I:
	        // WebGL2 only
	        this._glFormat = gl.RGBA_INTEGER;
	        this._glInternalFormat = gl.RGBA32I;
	        this._glPixelType = gl.INT;
	        break;
	      case PIXELFORMAT_RGBA32U:
	        // WebGL2 only
	        this._glFormat = gl.RGBA_INTEGER;
	        this._glInternalFormat = gl.RGBA32UI;
	        this._glPixelType = gl.UNSIGNED_INT;
	        break;
	      case PIXELFORMAT_BGRA8:
	        Debug.error("BGRA8 texture format is not supported by WebGL.");
	        break;
	    }
	    this._glCreated = false;
	  }

	  /**
	   * @param {new Function("modulePath", "return import(modulePath)")('./webgl-graphics-device.js').WebglGraphicsDevice} device - The device.
	   * @param {new Function("modulePath", "return import(modulePath)")('../texture.js').Texture} texture - The texture to update.
	   */;
	  _proto.upload = function upload(device, texture) {
	    Debug.assert(texture.device, "Attempting to use a texture that has been destroyed.", texture);
	    var gl = device.gl;
	    if (!texture._needsUpload && (texture._needsMipmapsUpload && texture._mipmapsUploaded || !texture.pot)) return;
	    var mipLevel = 0;
	    var mipObject;
	    var resMult;
	    var requiredMipLevels = texture.requiredMipLevels;
	    if (texture.array) {
	      // for texture arrays we reserve the space in advance
	      gl.texStorage3D(gl.TEXTURE_2D_ARRAY, requiredMipLevels, this._glInternalFormat, texture._width, texture._height, texture._arrayLength);
	    }

	    // Upload all existing mip levels. Initialize 0 mip anyway.
	    while (texture._levels[mipLevel] || mipLevel === 0) {
	      if (!texture._needsUpload && mipLevel === 0) {
	        mipLevel++;
	        continue;
	      } else if (mipLevel && (!texture._needsMipmapsUpload || !texture._mipmaps)) {
	        break;
	      }
	      mipObject = texture._levels[mipLevel];
	      resMult = 1 / Math.pow(2, mipLevel);
	      if (mipLevel === 1 && !texture._compressed && !texture._integerFormat && texture._levels.length < requiredMipLevels) {
	        // We have more than one mip levels we want to assign, but we need all mips to make
	        // the texture complete. Therefore first generate all mip chain from 0, then assign custom mips.
	        // (this implies the call to _completePartialMipLevels above was unsuccessful)
	        gl.generateMipmap(this._glTarget);
	        texture._mipmapsUploaded = true;
	      }
	      if (texture._cubemap) {
	        // ----- CUBEMAP -----
	        var face = void 0;
	        if (device._isBrowserInterface(mipObject[0])) {
	          // Upload the image, canvas or video
	          for (face = 0; face < 6; face++) {
	            if (!texture._levelsUpdated[0][face]) continue;
	            var src = mipObject[face];
	            // Downsize images that are too large to be used as cube maps
	            if (device._isImageBrowserInterface(src)) {
	              if (src.width > device.maxCubeMapSize || src.height > device.maxCubeMapSize) {
	                src = downsampleImage(src, device.maxCubeMapSize);
	                if (mipLevel === 0) {
	                  texture._width = src.width;
	                  texture._height = src.height;
	                }
	              }
	            }
	            device.setUnpackFlipY(false);
	            device.setUnpackPremultiplyAlpha(texture._premultiplyAlpha);
	            if (this._glCreated) {
	              gl.texSubImage2D(gl.TEXTURE_CUBE_MAP_POSITIVE_X + face, mipLevel, 0, 0, this._glFormat, this._glPixelType, src);
	            } else {
	              gl.texImage2D(gl.TEXTURE_CUBE_MAP_POSITIVE_X + face, mipLevel, this._glInternalFormat, this._glFormat, this._glPixelType, src);
	            }
	          }
	        } else {
	          // Upload the byte array
	          resMult = 1 / Math.pow(2, mipLevel);
	          for (face = 0; face < 6; face++) {
	            if (!texture._levelsUpdated[0][face]) continue;
	            var texData = mipObject[face];
	            if (texture._compressed) {
	              if (this._glCreated && texData) {
	                gl.compressedTexSubImage2D(gl.TEXTURE_CUBE_MAP_POSITIVE_X + face, mipLevel, 0, 0, Math.max(texture._width * resMult, 1), Math.max(texture._height * resMult, 1), this._glInternalFormat, texData);
	              } else {
	                gl.compressedTexImage2D(gl.TEXTURE_CUBE_MAP_POSITIVE_X + face, mipLevel, this._glInternalFormat, Math.max(texture._width * resMult, 1), Math.max(texture._height * resMult, 1), 0, texData);
	              }
	            } else {
	              device.setUnpackFlipY(false);
	              device.setUnpackPremultiplyAlpha(texture._premultiplyAlpha);
	              if (this._glCreated && texData) {
	                gl.texSubImage2D(gl.TEXTURE_CUBE_MAP_POSITIVE_X + face, mipLevel, 0, 0, Math.max(texture._width * resMult, 1), Math.max(texture._height * resMult, 1), this._glFormat, this._glPixelType, texData);
	              } else {
	                gl.texImage2D(gl.TEXTURE_CUBE_MAP_POSITIVE_X + face, mipLevel, this._glInternalFormat, Math.max(texture._width * resMult, 1), Math.max(texture._height * resMult, 1), 0, this._glFormat, this._glPixelType, texData);
	              }
	            }
	          }
	        }
	      } else if (texture._volume) {
	        // ----- 3D -----
	        // Image/canvas/video not supported (yet?)
	        // Upload the byte array
	        if (texture._compressed) {
	          gl.compressedTexImage3D(gl.TEXTURE_3D, mipLevel, this._glInternalFormat, Math.max(texture._width * resMult, 1), Math.max(texture._height * resMult, 1), Math.max(texture._depth * resMult, 1), 0, mipObject);
	        } else {
	          device.setUnpackFlipY(false);
	          device.setUnpackPremultiplyAlpha(texture._premultiplyAlpha);
	          gl.texImage3D(gl.TEXTURE_3D, mipLevel, this._glInternalFormat, Math.max(texture._width * resMult, 1), Math.max(texture._height * resMult, 1), Math.max(texture._depth * resMult, 1), 0, this._glFormat, this._glPixelType, mipObject);
	        }
	      } else if (texture.array && typeof mipObject === "object") {
	        if (texture._arrayLength === mipObject.length) {
	          if (texture._compressed) {
	            for (var index = 0; index < texture._arrayLength; index++) {
	              gl.compressedTexSubImage3D(gl.TEXTURE_2D_ARRAY, mipLevel, 0, 0, index, Math.max(Math.floor(texture._width * resMult), 1), Math.max(Math.floor(texture._height * resMult), 1), 1, this._glFormat, mipObject[index]);
	            }
	          } else {
	            for (var _index = 0; _index < texture._arrayLength; _index++) {
	              gl.texSubImage3D(gl.TEXTURE_2D_ARRAY, mipLevel, 0, 0, _index, Math.max(Math.floor(texture._width * resMult), 1), Math.max(Math.floor(texture._height * resMult), 1), 1, this._glFormat, this._glPixelType, mipObject[_index]);
	            }
	          }
	        }
	      } else {
	        // ----- 2D -----
	        if (device._isBrowserInterface(mipObject)) {
	          // Downsize images that are too large to be used as textures
	          if (device._isImageBrowserInterface(mipObject)) {
	            if (mipObject.width > device.maxTextureSize || mipObject.height > device.maxTextureSize) {
	              mipObject = downsampleImage(mipObject, device.maxTextureSize);
	              if (mipLevel === 0) {
	                texture._width = mipObject.width;
	                texture._height = mipObject.height;
	              }
	            }
	          }
	          var w = mipObject.width || mipObject.videoWidth;
	          var h = mipObject.height || mipObject.videoHeight;

	          // Upload the image, canvas or video
	          device.setUnpackFlipY(texture._flipY);
	          device.setUnpackPremultiplyAlpha(texture._premultiplyAlpha);

	          // TEMP: disable fast path for video updates until
	          // https://bugs.chromium.org/p/chromium/issues/detail?id=1511207 is resolved
	          if (this._glCreated && texture._width === w && texture._height === h && !device._isImageVideoInterface(mipObject)) {
	            gl.texSubImage2D(gl.TEXTURE_2D, mipLevel, 0, 0, this._glFormat, this._glPixelType, mipObject);
	          } else {
	            gl.texImage2D(gl.TEXTURE_2D, mipLevel, this._glInternalFormat, this._glFormat, this._glPixelType, mipObject);
	            if (mipLevel === 0) {
	              texture._width = w;
	              texture._height = h;
	            }
	          }
	        } else {
	          // Upload the byte array
	          resMult = 1 / Math.pow(2, mipLevel);
	          if (texture._compressed) {
	            if (this._glCreated && mipObject) {
	              gl.compressedTexSubImage2D(gl.TEXTURE_2D, mipLevel, 0, 0, Math.max(Math.floor(texture._width * resMult), 1), Math.max(Math.floor(texture._height * resMult), 1), this._glInternalFormat, mipObject);
	            } else {
	              gl.compressedTexImage2D(gl.TEXTURE_2D, mipLevel, this._glInternalFormat, Math.max(Math.floor(texture._width * resMult), 1), Math.max(Math.floor(texture._height * resMult), 1), 0, mipObject);
	            }
	          } else {
	            device.setUnpackFlipY(false);
	            device.setUnpackPremultiplyAlpha(texture._premultiplyAlpha);
	            if (this._glCreated && mipObject) {
	              gl.texSubImage2D(gl.TEXTURE_2D, mipLevel, 0, 0, Math.max(texture._width * resMult, 1), Math.max(texture._height * resMult, 1), this._glFormat, this._glPixelType, mipObject);
	            } else {
	              gl.texImage2D(gl.TEXTURE_2D, mipLevel, this._glInternalFormat, Math.max(texture._width * resMult, 1), Math.max(texture._height * resMult, 1), 0, this._glFormat, this._glPixelType, mipObject);
	            }
	          }
	        }
	        if (mipLevel === 0) {
	          texture._mipmapsUploaded = false;
	        } else {
	          texture._mipmapsUploaded = true;
	        }
	      }
	      mipLevel++;
	    }
	    if (texture._needsUpload) {
	      if (texture._cubemap) {
	        for (var i = 0; i < 6; i++) texture._levelsUpdated[0][i] = false;
	      } else {
	        texture._levelsUpdated[0] = false;
	      }
	    }
	    if (!texture._compressed && !texture._integerFormat && texture._mipmaps && texture._needsMipmapsUpload && (texture.pot || device.isWebGL2) && texture._levels.length === 1) {
	      gl.generateMipmap(this._glTarget);
	      texture._mipmapsUploaded = true;
	    }

	    // update vram stats
	    if (texture._gpuSize) {
	      texture.adjustVramSizeTracking(device._vram, -texture._gpuSize);
	    }
	    texture._gpuSize = texture.gpuSize;
	    texture.adjustVramSizeTracking(device._vram, texture._gpuSize);
	    this._glCreated = true;
	  };
	  return WebglTexture;
	}();

	/**
	 * A private class representing a pair of framebuffers, when MSAA is used.
	 *
	 * @ignore
	 */
	var FramebufferPair = /*#__PURE__*/function () {
	  function FramebufferPair(msaaFB, resolveFB) {
	    /** Multi-sampled rendering framebuffer */
	    this.msaaFB = void 0;
	    /** Single-sampled resolve framebuffer */
	    this.resolveFB = void 0;
	    this.msaaFB = msaaFB;
	    this.resolveFB = resolveFB;
	  }
	  var _proto = FramebufferPair.prototype;
	  _proto.destroy = function destroy(gl) {
	    if (this.msaaFB) {
	      gl.deleteRenderbuffer(this.msaaFB);
	      this.msaaFB = null;
	    }
	    if (this.resolveFB) {
	      gl.deleteRenderbuffer(this.resolveFB);
	      this.resolveFB = null;
	    }
	  };
	  return FramebufferPair;
	}();
	/**
	 * A WebGL implementation of the RenderTarget.
	 *
	 * @ignore
	 */
	var WebglRenderTarget = /*#__PURE__*/function () {
	  function WebglRenderTarget() {
	    this._glFrameBuffer = null;
	    this._glDepthBuffer = null;
	    this._glResolveFrameBuffer = null;
	    /**
	     * A list of framebuffers created When MSAA and MRT are used together, one for each color buffer.
	     * This allows color buffers to be resolved separately.
	     *
	     * @type {FramebufferPair[]}
	     */
	    this.colorMrtFramebuffers = null;
	    this._glMsaaColorBuffers = [];
	    this._glMsaaDepthBuffer = null;
	    /**
	     * The supplied single-sampled framebuffer for rendering. Undefined represents no supplied
	     * framebuffer. Null represents the default framebuffer. A value represents a user-supplied
	     * framebuffer.
	     */
	    this.suppliedColorFramebuffer = void 0;
	    this._isInitialized = false;
	  }
	  var _proto2 = WebglRenderTarget.prototype;
	  _proto2.destroy = function destroy(device) {
	    var _this$colorMrtFramebu;
	    var gl = device.gl;
	    this._isInitialized = false;
	    if (this._glFrameBuffer) {
	      if (this._glFrameBuffer !== this.suppliedColorFramebuffer) gl.deleteFramebuffer(this._glFrameBuffer);
	      this._glFrameBuffer = null;
	    }
	    if (this._glDepthBuffer) {
	      gl.deleteRenderbuffer(this._glDepthBuffer);
	      this._glDepthBuffer = null;
	    }
	    if (this._glResolveFrameBuffer) {
	      if (this._glResolveFrameBuffer !== this.suppliedColorFramebuffer) gl.deleteFramebuffer(this._glResolveFrameBuffer);
	      this._glResolveFrameBuffer = null;
	    }
	    this._glMsaaColorBuffers.forEach(function (buffer) {
	      gl.deleteRenderbuffer(buffer);
	    });
	    this._glMsaaColorBuffers.length = 0;
	    (_this$colorMrtFramebu = this.colorMrtFramebuffers) == null || _this$colorMrtFramebu.forEach(function (framebuffer) {
	      framebuffer.destroy(gl);
	    });
	    this.colorMrtFramebuffers = null;
	    if (this._glMsaaDepthBuffer) {
	      gl.deleteRenderbuffer(this._glMsaaDepthBuffer);
	      this._glMsaaDepthBuffer = null;
	    }
	    this.suppliedColorFramebuffer = undefined;
	  };
	  _proto2.init = function init(device, target) {
	    var _this = this;
	    var gl = device.gl;
	    this._isInitialized = true;
	    var buffers = [];
	    if (this.suppliedColorFramebuffer !== undefined) {
	      this._glFrameBuffer = this.suppliedColorFramebuffer;
	    } else {
	      var _target$_colorBuffers, _target$_colorBuffers2, _device$extDrawBuffer, _device$extDrawBuffer2;
	      // ##### Create main FBO #####
	      this._glFrameBuffer = gl.createFramebuffer();
	      device.setFramebuffer(this._glFrameBuffer);

	      // --- Init the provided color buffer (optional) ---
	      var colorBufferCount = (_target$_colorBuffers = (_target$_colorBuffers2 = target._colorBuffers) == null ? void 0 : _target$_colorBuffers2.length) != null ? _target$_colorBuffers : 0;
	      var attachmentBaseConstant = device.isWebGL2 ? gl.COLOR_ATTACHMENT0 : (_device$extDrawBuffer = (_device$extDrawBuffer2 = device.extDrawBuffers) == null ? void 0 : _device$extDrawBuffer2.COLOR_ATTACHMENT0_WEBGL) != null ? _device$extDrawBuffer : gl.COLOR_ATTACHMENT0;
	      for (var i = 0; i < colorBufferCount; ++i) {
	        var colorBuffer = target.getColorBuffer(i);
	        if (colorBuffer) {
	          if (!colorBuffer.impl._glTexture) {
	            // Clamp the render buffer size to the maximum supported by the device
	            colorBuffer._width = Math.min(colorBuffer.width, device.maxRenderBufferSize);
	            colorBuffer._height = Math.min(colorBuffer.height, device.maxRenderBufferSize);
	            device.setTexture(colorBuffer, 0);
	          }
	          // Attach the color buffer
	          gl.framebufferTexture2D(gl.FRAMEBUFFER, attachmentBaseConstant + i, colorBuffer._cubemap ? gl.TEXTURE_CUBE_MAP_POSITIVE_X + target._face : gl.TEXTURE_2D, colorBuffer.impl._glTexture, 0);
	          buffers.push(attachmentBaseConstant + i);
	        }
	      }
	      if (device.drawBuffers) {
	        device.drawBuffers(buffers);
	      }
	      var depthBuffer = target._depthBuffer;
	      if (depthBuffer) {
	        // --- Init the provided depth/stencil buffer (optional, WebGL2 only) ---
	        if (!depthBuffer.impl._glTexture) {
	          // Clamp the render buffer size to the maximum supported by the device
	          depthBuffer._width = Math.min(depthBuffer.width, device.maxRenderBufferSize);
	          depthBuffer._height = Math.min(depthBuffer.height, device.maxRenderBufferSize);
	          device.setTexture(depthBuffer, 0);
	        }
	        // Attach
	        if (target._stencil) {
	          gl.framebufferTexture2D(gl.FRAMEBUFFER, gl.DEPTH_STENCIL_ATTACHMENT, depthBuffer._cubemap ? gl.TEXTURE_CUBE_MAP_POSITIVE_X + target._face : gl.TEXTURE_2D, target._depthBuffer.impl._glTexture, 0);
	        } else {
	          gl.framebufferTexture2D(gl.FRAMEBUFFER, gl.DEPTH_ATTACHMENT, depthBuffer._cubemap ? gl.TEXTURE_CUBE_MAP_POSITIVE_X + target._face : gl.TEXTURE_2D, target._depthBuffer.impl._glTexture, 0);
	        }
	      } else if (target._depth) {
	        // --- Init a new depth/stencil buffer (optional) ---
	        // if device is a MSAA RT, and no buffer to resolve to, skip creating non-MSAA depth
	        var willRenderMsaa = target._samples > 1 && device.isWebGL2;
	        if (!willRenderMsaa) {
	          if (!this._glDepthBuffer) {
	            this._glDepthBuffer = gl.createRenderbuffer();
	          }
	          gl.bindRenderbuffer(gl.RENDERBUFFER, this._glDepthBuffer);
	          if (target._stencil) {
	            gl.renderbufferStorage(gl.RENDERBUFFER, gl.DEPTH_STENCIL, target.width, target.height);
	            gl.framebufferRenderbuffer(gl.FRAMEBUFFER, gl.DEPTH_STENCIL_ATTACHMENT, gl.RENDERBUFFER, this._glDepthBuffer);
	          } else {
	            var depthFormat = device.isWebGL2 ? gl.DEPTH_COMPONENT32F : gl.DEPTH_COMPONENT16;
	            gl.renderbufferStorage(gl.RENDERBUFFER, depthFormat, target.width, target.height);
	            gl.framebufferRenderbuffer(gl.FRAMEBUFFER, gl.DEPTH_ATTACHMENT, gl.RENDERBUFFER, this._glDepthBuffer);
	          }
	          gl.bindRenderbuffer(gl.RENDERBUFFER, null);
	        }
	      }
	      Debug.call(function () {
	        return _this._checkFbo(device, target);
	      });
	    }

	    // ##### Create MSAA FBO (WebGL2 only) #####
	    if (device.isWebGL2 && target._samples > 1) {
	      var _target$_colorBuffers3, _target$_colorBuffers4;
	      Debug.call(function () {
	        if (target.width <= 0 || target.height <= 0) {
	          Debug.warnOnce("Invalid render target size: " + target.width + " x " + target.height, target);
	        }
	      });

	      // Use previous FBO for resolves
	      this._glResolveFrameBuffer = this._glFrameBuffer;

	      // Actual FBO will be MSAA
	      this._glFrameBuffer = gl.createFramebuffer();
	      device.setFramebuffer(this._glFrameBuffer);

	      // Create an optional MSAA color buffers
	      var _colorBufferCount = (_target$_colorBuffers3 = (_target$_colorBuffers4 = target._colorBuffers) == null ? void 0 : _target$_colorBuffers4.length) != null ? _target$_colorBuffers3 : 0;
	      if (this.suppliedColorFramebuffer !== undefined) {
	        var buffer = gl.createRenderbuffer();
	        this._glMsaaColorBuffers.push(buffer);
	        var internalFormat = device.backBufferFormat === PIXELFORMAT_RGBA8 ? gl.RGBA8 : gl.RGB8;
	        gl.bindRenderbuffer(gl.RENDERBUFFER, buffer);
	        gl.renderbufferStorageMultisample(gl.RENDERBUFFER, target._samples, internalFormat, target.width, target.height);
	        gl.framebufferRenderbuffer(gl.FRAMEBUFFER, gl.COLOR_ATTACHMENT0, gl.RENDERBUFFER, buffer);
	      } else {
	        for (var _i = 0; _i < _colorBufferCount; ++_i) {
	          var _colorBuffer = target.getColorBuffer(_i);
	          if (_colorBuffer) {
	            var _buffer = gl.createRenderbuffer();
	            this._glMsaaColorBuffers.push(_buffer);
	            gl.bindRenderbuffer(gl.RENDERBUFFER, _buffer);
	            gl.renderbufferStorageMultisample(gl.RENDERBUFFER, target._samples, _colorBuffer.impl._glInternalFormat, target.width, target.height);
	            gl.framebufferRenderbuffer(gl.FRAMEBUFFER, gl.COLOR_ATTACHMENT0 + _i, gl.RENDERBUFFER, _buffer);
	          }
	        }
	      }

	      // Optionally add a MSAA depth/stencil buffer
	      if (target._depth) {
	        if (!this._glMsaaDepthBuffer) {
	          this._glMsaaDepthBuffer = gl.createRenderbuffer();
	        }
	        gl.bindRenderbuffer(gl.RENDERBUFFER, this._glMsaaDepthBuffer);
	        if (target._stencil) {
	          gl.renderbufferStorageMultisample(gl.RENDERBUFFER, target._samples, gl.DEPTH24_STENCIL8, target.width, target.height);
	          gl.framebufferRenderbuffer(gl.FRAMEBUFFER, gl.DEPTH_STENCIL_ATTACHMENT, gl.RENDERBUFFER, this._glMsaaDepthBuffer);
	        } else {
	          gl.renderbufferStorageMultisample(gl.RENDERBUFFER, target._samples, gl.DEPTH_COMPONENT32F, target.width, target.height);
	          gl.framebufferRenderbuffer(gl.FRAMEBUFFER, gl.DEPTH_ATTACHMENT, gl.RENDERBUFFER, this._glMsaaDepthBuffer);
	        }
	      }
	      Debug.call(function () {
	        return _this._checkFbo(device, target, 'MSAA');
	      });
	      if (_colorBufferCount > 1) {
	        // create framebuffers allowing us to individually resolve each color buffer
	        this._createMsaaMrtFramebuffers(device, target, _colorBufferCount);

	        // restore rendering back to the main framebuffer
	        device.setFramebuffer(this._glFrameBuffer);
	        device.drawBuffers(buffers);
	      }
	    }
	  };
	  _proto2._createMsaaMrtFramebuffers = function _createMsaaMrtFramebuffers(device, target, colorBufferCount) {
	    var _this2 = this;
	    var gl = device.gl;
	    this.colorMrtFramebuffers = [];
	    var _loop = function _loop(i) {
	      var colorBuffer = target.getColorBuffer(i);

	      // src
	      var srcFramebuffer = gl.createFramebuffer();
	      device.setFramebuffer(srcFramebuffer);
	      var buffer = _this2._glMsaaColorBuffers[i];
	      gl.bindRenderbuffer(gl.RENDERBUFFER, buffer);
	      gl.renderbufferStorageMultisample(gl.RENDERBUFFER, target._samples, colorBuffer.impl._glInternalFormat, target.width, target.height);
	      gl.framebufferRenderbuffer(gl.FRAMEBUFFER, gl.COLOR_ATTACHMENT0, gl.RENDERBUFFER, buffer);
	      device.drawBuffers([gl.COLOR_ATTACHMENT0]);
	      Debug.call(function () {
	        return _this2._checkFbo(device, target, "MSAA-MRT-src" + i);
	      });

	      // dst
	      var dstFramebuffer = gl.createFramebuffer();
	      device.setFramebuffer(dstFramebuffer);
	      gl.framebufferTexture2D(gl.FRAMEBUFFER, gl.COLOR_ATTACHMENT0, colorBuffer._cubemap ? gl.TEXTURE_CUBE_MAP_POSITIVE_X + target._face : gl.TEXTURE_2D, colorBuffer.impl._glTexture, 0);
	      _this2.colorMrtFramebuffers[i] = new FramebufferPair(srcFramebuffer, dstFramebuffer);
	      Debug.call(function () {
	        return _this2._checkFbo(device, target, "MSAA-MRT-dst" + i);
	      });
	    };
	    for (var i = 0; i < colorBufferCount; ++i) {
	      _loop(i);
	    }
	  }

	  /**
	   * Checks the completeness status of the currently bound WebGLFramebuffer object.
	   *
	   * @private
	   */;
	  _proto2._checkFbo = function _checkFbo(device, target, type) {
	    if (type === void 0) {
	      type = '';
	    }
	    var gl = device.gl;
	    var status = gl.checkFramebufferStatus(gl.FRAMEBUFFER);
	    var errorCode;
	    switch (status) {
	      case gl.FRAMEBUFFER_INCOMPLETE_ATTACHMENT:
	        errorCode = 'FRAMEBUFFER_INCOMPLETE_ATTACHMENT';
	        break;
	      case gl.FRAMEBUFFER_INCOMPLETE_MISSING_ATTACHMENT:
	        errorCode = 'FRAMEBUFFER_INCOMPLETE_MISSING_ATTACHMENT';
	        break;
	      case gl.FRAMEBUFFER_INCOMPLETE_DIMENSIONS:
	        errorCode = 'FRAMEBUFFER_INCOMPLETE_DIMENSIONS';
	        break;
	      case gl.FRAMEBUFFER_UNSUPPORTED:
	        errorCode = 'FRAMEBUFFER_UNSUPPORTED';
	        break;
	    }
	    Debug.assert(!errorCode, "Framebuffer creation failed with error code " + errorCode + ", render target: " + target.name + " " + type, target);
	  };
	  _proto2.loseContext = function loseContext() {
	    this._glFrameBuffer = null;
	    this._glDepthBuffer = null;
	    this._glResolveFrameBuffer = null;
	    this._glMsaaColorBuffers.length = 0;
	    this._glMsaaDepthBuffer = null;
	    this.colorMrtFramebuffers = null;
	    this.suppliedColorFramebuffer = undefined;
	    this._isInitialized = false;
	  };
	  _proto2.internalResolve = function internalResolve(device, src, dst, target, mask) {
	    Debug.assert(src !== dst, 'Source and destination framebuffers must be different when blitting.');

	    // blit is affected by scissor test, so make it full size
	    device.setScissor(0, 0, target.width, target.height);
	    var gl = device.gl;
	    gl.bindFramebuffer(gl.READ_FRAMEBUFFER, src);
	    gl.bindFramebuffer(gl.DRAW_FRAMEBUFFER, dst);
	    gl.blitFramebuffer(0, 0, target.width, target.height, 0, 0, target.width, target.height, mask, gl.NEAREST);
	  };
	  _proto2.resolve = function resolve(device, target, color, depth) {
	    if (device.isWebGL2) {
	      var gl = device.gl;

	      // if MRT is used, we need to resolve each buffer individually
	      if (this.colorMrtFramebuffers) {
	        // color
	        if (color) {
	          for (var i = 0; i < this.colorMrtFramebuffers.length; i++) {
	            var fbPair = this.colorMrtFramebuffers[i];
	            DebugGraphics.pushGpuMarker(device, "RESOLVE-MRT" + i);
	            this.internalResolve(device, fbPair.msaaFB, fbPair.resolveFB, target, gl.COLOR_BUFFER_BIT);
	            DebugGraphics.popGpuMarker(device);
	          }
	        }

	        // depth
	        if (depth) {
	          DebugGraphics.pushGpuMarker(device, "RESOLVE-MRT-DEPTH");
	          this.internalResolve(device, this._glFrameBuffer, this._glResolveFrameBuffer, target, gl.DEPTH_BUFFER_BIT);
	          DebugGraphics.popGpuMarker(device);
	        }
	      } else {
	        DebugGraphics.pushGpuMarker(device, "RESOLVE");
	        this.internalResolve(device, this._glFrameBuffer, this._glResolveFrameBuffer, target, (color ? gl.COLOR_BUFFER_BIT : 0) | (depth ? gl.DEPTH_BUFFER_BIT : 0));
	        DebugGraphics.popGpuMarker(device);
	      }
	      gl.bindFramebuffer(gl.FRAMEBUFFER, this._glFrameBuffer);
	    }
	  };
	  _createClass(WebglRenderTarget, [{
	    key: "initialized",
	    get: function get() {
	      return this._isInitialized;
	    }
	  }]);
	  return WebglRenderTarget;
	}();

	var gles2PS = /* glsl */"\n\n#define pcFragColor0 gl_FragData[0]\n\n#if COLOR_ATTACHMENT_1\n#define pcFragColor1 gl_FragData[1]\n#endif\n\n#if COLOR_ATTACHMENT_2\n#define pcFragColor2 gl_FragData[2]\n#endif\n\n#if COLOR_ATTACHMENT_3\n#define pcFragColor3 gl_FragData[3]\n#endif\n\n#if COLOR_ATTACHMENT_4\n#define pcFragColor4 gl_FragData[4]\n#endif\n\n#if COLOR_ATTACHMENT_5\n#define pcFragColor5 gl_FragData[5]\n#endif\n\n#if COLOR_ATTACHMENT_6\n#define pcFragColor6 gl_FragData[6]\n#endif\n\n#if COLOR_ATTACHMENT_7\n#define pcFragColor7 gl_FragData[7]\n#endif\n\n#define texture2DBias texture2D\n#define itexture2D texture2D\n#define utexture2D texture2D\n\n// pass / accept shadow map or texture as a function parameter, on webgl this is simply passed as is\n// but this is needed for WebGPU\n#define SHADOWMAP_PASS(name) name\n#define SHADOWMAP_ACCEPT(name) sampler2D name\n#define TEXTURE_PASS(name) name\n#define TEXTURE_ACCEPT(name) sampler2D name\n\n#ifndef SUPPORTS_TEXLOD\n\n    // fallback for lod instructions\n    #define texture2DLodEXT texture2D\n    #define texture2DProjLodEXT textureProj\n    #define textureCubeLodEXT textureCube\n    #define textureShadow texture2D\n\n#else\n\n    #define textureShadow(res, uv) texture2DGradEXT(res, uv, vec2(1, 1), vec2(1, 1))\n\n#endif\n\n#ifdef SUPPORTS_MRT\n    #define gl_FragColor pcFragColor0\n#endif\n\n";

	var gles3PS = /* glsl */"\n\n#ifndef outType_0\n#define outType_0 vec4\n#endif\n\nlayout(location = 0) out highp outType_0 pc_fragColor;\n\n#ifndef REMOVE_COLOR_ATTACHMENT_1\n#if COLOR_ATTACHMENT_1\nlayout(location = 1) out highp outType_1 pc_fragColor1;\n#endif\n#endif\n\n#ifndef REMOVE_COLOR_ATTACHMENT_2\n#if COLOR_ATTACHMENT_2\nlayout(location = 2) out highp outType_2 pc_fragColor2;\n#endif\n#endif\n\n#ifndef REMOVE_COLOR_ATTACHMENT_3\n#if COLOR_ATTACHMENT_3\nlayout(location = 3) out highp outType_3 pc_fragColor3;\n#endif\n#endif\n\n#ifndef REMOVE_COLOR_ATTACHMENT_4\n#if COLOR_ATTACHMENT_4\nlayout(location = 4) out highp outType_4 pc_fragColor4;\n#endif\n#endif\n\n#ifndef REMOVE_COLOR_ATTACHMENT_5\n#if COLOR_ATTACHMENT_5\nlayout(location = 5) out highp outType_5 pc_fragColor5;\n#endif\n#endif\n\n#ifndef REMOVE_COLOR_ATTACHMENT_6\n#if COLOR_ATTACHMENT_6\nlayout(location = 6) out highp outType_6 pc_fragColor6;\n#endif\n#endif\n\n#ifndef REMOVE_COLOR_ATTACHMENT_7\n#if COLOR_ATTACHMENT_7\nlayout(location = 7) out highp outType_7 pc_fragColor7;\n#endif\n#endif\n\n#define gl_FragColor pc_fragColor\n\n#define pcFragColor0 pc_fragColor\n#define pcFragColor1 pc_fragColor1\n#define pcFragColor2 pc_fragColor2\n#define pcFragColor3 pc_fragColor3\n#define pcFragColor4 pc_fragColor4\n#define pcFragColor5 pc_fragColor5\n#define pcFragColor6 pc_fragColor6\n#define pcFragColor7 pc_fragColor7\n\n#define varying in\n\n#define texture2D texture\n#define texture2DBias texture\n#define textureCube texture\n#define texture2DProj textureProj\n#define texture2DLodEXT textureLod\n#define texture2DProjLodEXT textureProjLod\n#define textureCubeLodEXT textureLod\n#define texture2DGradEXT textureGrad\n#define texture2DProjGradEXT textureProjGrad\n#define textureCubeGradEXT textureGrad\n#define utexture2D texture\n#define itexture2D texture\n\n// sample shadows using textureGrad to remove derivatives in the dynamic loops (which are used by\n// clustered lighting) - as DirectX shader compiler tries to unroll the loops and takes long time\n// to compile the shader. Using textureLod would be even better, but WebGl does not translate it to\n// lod instruction for DirectX correctly and uses SampleCmp instead of SampleCmpLevelZero or similar.\n#define textureShadow(res, uv) textureGrad(res, uv, vec2(1, 1), vec2(1, 1))\n\n// pass / accept shadow map or texture as a function parameter, on webgl this is simply passed as is\n// but this is needed for WebGPU\n#define SHADOWMAP_PASS(name) name\n#define SHADOWMAP_ACCEPT(name) sampler2DShadow name\n#define TEXTURE_PASS(name) name\n#define TEXTURE_ACCEPT(name) sampler2D name\n\n#define GL2\n#define SUPPORTS_TEXLOD\n#define SUPPORTS_MRT\n";

	var gles3VS = /* glsl */"\n#define attribute in\n#define varying out\n#define texture2D texture\n#define utexture2D texture\n#define itexture2D texture\n#define GL2\n#define VERTEXSHADER\n";

	var webgpuPS = /* glsl */"\n\n// texelFetch support and others\n#extension GL_EXT_samplerless_texture_functions : require\n\n#ifndef outType_0\n#define outType_0 vec4\n#endif\n#ifndef outType_1\n#define outType_1 vec4\n#endif\n#ifndef outType_2\n#define outType_2 vec4\n#endif\n#ifndef outType_3\n#define outType_3 vec4\n#endif\n#ifndef outType_4\n#define outType_4 vec4\n#endif\n#ifndef outType_5\n#define outType_5 vec4\n#endif\n#ifndef outType_6\n#define outType_6 vec4\n#endif\n#ifndef outType_7\n#define outType_7 vec4\n#endif\n\nlayout(location = 0) out highp outType_0 pc_fragColor;\nlayout(location = 1) out highp outType_1 pc_fragColor1;\nlayout(location = 2) out highp outType_2 pc_fragColor2;\nlayout(location = 3) out highp outType_3 pc_fragColor3;\nlayout(location = 4) out highp outType_4 pc_fragColor4;\nlayout(location = 5) out highp outType_5 pc_fragColor5;\nlayout(location = 6) out highp outType_6 pc_fragColor6;\nlayout(location = 7) out highp outType_7 pc_fragColor7;\n\n#define gl_FragColor pc_fragColor\n\n#define pcFragColor0 pc_fragColor\n#define pcFragColor1 pc_fragColor1\n#define pcFragColor2 pc_fragColor2\n#define pcFragColor3 pc_fragColor3\n#define pcFragColor4 pc_fragColor4\n#define pcFragColor5 pc_fragColor5\n#define pcFragColor6 pc_fragColor6\n#define pcFragColor7 pc_fragColor7\n\n#define texture2D(res, uv) texture(sampler2D(res, res ## _sampler), uv)\n#define texture2DBias(res, uv, bias) texture(sampler2D(res, res ## _sampler), uv, bias)\n#define texture2DLodEXT(res, uv, lod) textureLod(sampler2D(res, res ## _sampler), uv, lod)\n#define textureCube(res, uv) texture(samplerCube(res, res ## _sampler), uv)\n#define textureCubeLodEXT(res, uv, lod) textureLod(samplerCube(res, res ## _sampler), uv, lod)\n#define textureShadow(res, uv) textureLod(sampler2DShadow(res, res ## _sampler), uv, 0.0)\n#define itexture2D(res, uv) texture(isampler2D(res, res ## _sampler), uv)\n#define utexture2D(res, uv) texture(usampler2D(res, res ## _sampler), uv)\n\n// TODO: implement other texture sampling macros\n// #define texture2DProj textureProj\n// #define texture2DProjLodEXT textureProjLod\n// #define texture2DGradEXT textureGrad\n// #define texture2DProjGradEXT textureProjGrad\n// #define textureCubeGradEXT textureGrad\n\n// pass / accept shadow map as a function parameter, passes both the texture as well as sampler\n// as the combined sampler can be only created at a point of use\n#define SHADOWMAP_PASS(name) name, name ## _sampler\n#define SHADOWMAP_ACCEPT(name) texture2D name, sampler name ## _sampler\n#define TEXTURE_PASS(name) name, name ## _sampler\n#define TEXTURE_ACCEPT(name) texture2D name, sampler name ## _sampler\n\n#define GL2\n#define WEBGPU\n#define SUPPORTS_TEXLOD\n#define SUPPORTS_MRT\n";

	var webgpuVS = /* glsl */"\n\n// texelFetch support and others\n#extension GL_EXT_samplerless_texture_functions : require\n\n#define texture2D(res, uv) texture(sampler2D(res, res ## _sampler), uv)\n#define itexture2D(res, uv) texture(isampler2D(res, res ## _sampler), uv)\n#define utexture2D(res, uv) texture(usampler2D(res, res ## _sampler), uv)\n\n#define GL2\n#define WEBGPU\n#define VERTEXSHADER\n";

	var sharedFS = /* glsl */"\n\n// convert clip space position into texture coordinates to sample scene grab textures\nvec2 getGrabScreenPos(vec4 clipPos) {\n    vec2 uv = (clipPos.xy / clipPos.w) * 0.5 + 0.5;\n\n    #ifdef WEBGPU\n        uv.y = 1.0 - uv.y;\n    #endif\n\n    return uv;\n}\n\n// convert uv coordinates to sample image effect texture (render target texture rendered without\n// forward renderer which does the flip in the projection matrix)\nvec2 getImageEffectUV(vec2 uv) {\n    #ifdef WEBGPU\n        uv.y = 1.0 - uv.y;\n    #endif\n\n    return uv;\n}\n";

	var _attrib2Semantic = {
	  vertex_position: SEMANTIC_POSITION,
	  vertex_normal: SEMANTIC_NORMAL,
	  vertex_tangent: SEMANTIC_TANGENT,
	  vertex_texCoord0: SEMANTIC_TEXCOORD0,
	  vertex_texCoord1: SEMANTIC_TEXCOORD1,
	  vertex_texCoord2: SEMANTIC_TEXCOORD2,
	  vertex_texCoord3: SEMANTIC_TEXCOORD3,
	  vertex_texCoord4: SEMANTIC_TEXCOORD4,
	  vertex_texCoord5: SEMANTIC_TEXCOORD5,
	  vertex_texCoord6: SEMANTIC_TEXCOORD6,
	  vertex_texCoord7: SEMANTIC_TEXCOORD7,
	  vertex_color: SEMANTIC_COLOR,
	  vertex_boneIndices: SEMANTIC_BLENDINDICES,
	  vertex_boneWeights: SEMANTIC_BLENDWEIGHT
	};

	/**
	 * A class providing utility functions for shader creation.
	 *
	 * @ignore
	 */
	var ShaderUtils = /*#__PURE__*/function () {
	  function ShaderUtils() {}
	  /**
	   * Creates a shader definition.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./graphics-device.js').GraphicsDevice} device - The graphics device.
	   * @param {object} options - Object for passing optional arguments.
	   * @param {string} [options.name] - A name of the shader.
	   * @param {object} [options.attributes] - Attributes. Will be extracted from the vertexCode if
	   * not provided.
	   * @param {string} options.vertexCode - The vertex shader code.
	   * @param {string} [options.vertexDefines] - The vertex shader defines.
	   * @param {string} [options.vertexExtensions] - The vertex shader extensions code.
	   * @param {string} [options.fragmentCode] - The fragment shader code.
	   * @param {string} [options.fragmentDefines] - The fragment shader defines.
	   * @param {string} [options.fragmentExtensions] - The fragment shader extensions code.
	   * @param {string} [options.fragmentPreamble] - The preamble string for the fragment shader.
	   * @param {boolean} [options.useTransformFeedback] - Whether to use transform feedback. Defaults to false.
	   * @param {string | string[]} [options.fragmentOutputTypes] - Fragment shader output types,
	   * which default to vec4. Passing a string will set the output type for all color attachments.
	   * Passing an array will set the output type for each color attachment.
	   * @returns {object} Returns the created shader definition.
	   */
	  ShaderUtils.createDefinition = function createDefinition(device, options) {
	    var _options$name, _options$attributes;
	    Debug.assert(options);
	    var getDefines = function getDefines(gpu, gl2, gl1, isVertex, options) {
	      var deviceIntro = device.isWebGPU ? gpu : device.isWebGL2 ? gl2 : ShaderUtils.gl1Extensions(device, options) + gl1;

	      // a define per supported color attachment, which strips out unsupported output definitions in the deviceIntro
	      var attachmentsDefine = '';

	      // Define the fragment shader output type, vec4 by default
	      if (!isVertex) {
	        var _options$fragmentOutp;
	        // Normalize fragmentOutputTypes to an array
	        var fragmentOutputTypes = (_options$fragmentOutp = options.fragmentOutputTypes) != null ? _options$fragmentOutp : 'vec4';
	        if (!Array.isArray(fragmentOutputTypes)) {
	          fragmentOutputTypes = [fragmentOutputTypes];
	        }
	        for (var i = 0; i < device.maxColorAttachments; i++) {
	          var _fragmentOutputTypes$;
	          attachmentsDefine += "#define COLOR_ATTACHMENT_" + i + "\n";
	          var outType = (_fragmentOutputTypes$ = fragmentOutputTypes[i]) != null ? _fragmentOutputTypes$ : 'vec4';
	          attachmentsDefine += "#define outType_" + i + " " + outType + "\n";
	        }
	      }
	      return attachmentsDefine + deviceIntro;
	    };
	    var name = (_options$name = options.name) != null ? _options$name : 'Untitled';

	    // vertex code
	    var vertDefines = options.vertexDefines || getDefines(webgpuVS, gles3VS, '', true, options);
	    var vertCode = ShaderUtils.versionCode(device) + vertDefines + sharedFS + ShaderUtils.getShaderNameCode(name) + options.vertexCode;

	    // fragment code
	    var fragDefines = options.fragmentDefines || getDefines(webgpuPS, gles3PS, gles2PS, false, options);
	    var fragCode = (options.fragmentPreamble || '') + ShaderUtils.versionCode(device) + fragDefines + ShaderUtils.precisionCode(device) + '\n' + sharedFS + ShaderUtils.getShaderNameCode(name) + (options.fragmentCode || ShaderUtils.dummyFragmentCode());

	    // attributes
	    var attribs = (_options$attributes = options.attributes) != null ? _options$attributes : ShaderUtils.collectAttributes(options.vertexCode);
	    return {
	      name: name,
	      attributes: attribs,
	      vshader: vertCode,
	      fshader: fragCode,
	      useTransformFeedback: options.useTransformFeedback
	    };
	  }

	  // SpectorJS integration
	  ;
	  ShaderUtils.getShaderNameCode = function getShaderNameCode(name) {
	    return "#define SHADER_NAME " + name + "\n";
	  };
	  ShaderUtils.gl1Extensions = function gl1Extensions(device, options, isVertex) {
	    var code;
	    if (isVertex) {
	      code = options.vertexExtensions ? options.vertexExtensions + "\n" : '';
	    } else {
	      code = options.fragmentExtensions ? options.fragmentExtensions + "\n" : '';

	      // extensions used by default
	      if (device.extStandardDerivatives) {
	        code += "#extension GL_OES_standard_derivatives : enable\n";
	      }
	      if (device.extTextureLod) {
	        code += "#extension GL_EXT_shader_texture_lod : enable\n";
	        code += "#define SUPPORTS_TEXLOD\n";
	      }
	      if (device.extDrawBuffers) {
	        code += "#extension GL_EXT_draw_buffers : require\n";
	        code += "#define SUPPORTS_MRT\n";
	      }
	    }
	    return code;
	  };
	  ShaderUtils.dummyFragmentCode = function dummyFragmentCode() {
	    return "void main(void) {gl_FragColor = vec4(0.0);}";
	  };
	  ShaderUtils.versionCode = function versionCode(device) {
	    if (device.isWebGPU) {
	      return '#version 450\n';
	    }
	    return device.isWebGL2 ? "#version 300 es\n" : "";
	  };
	  ShaderUtils.precisionCode = function precisionCode(device, forcePrecision) {
	    var code = '';
	    if (forcePrecision && forcePrecision !== 'highp' && forcePrecision !== 'mediump' && forcePrecision !== 'lowp') {
	      forcePrecision = null;
	    }
	    if (forcePrecision) {
	      if (forcePrecision === 'highp' && device.maxPrecision !== 'highp') {
	        forcePrecision = 'mediump';
	      }
	      if (forcePrecision === 'mediump' && device.maxPrecision === 'lowp') {
	        forcePrecision = 'lowp';
	      }
	    }
	    var precision = forcePrecision ? forcePrecision : device.precision;
	    if (!device.isWebGPU) {
	      code = "precision " + precision + " float;\n";
	      if (device.isWebGL2) {
	        code += "precision " + precision + " sampler2DShadow;\n";
	      }
	    } else {
	      // WebGPU

	      code = "precision " + precision + " float;\nprecision " + precision + " int;\n";
	    }
	    return code;
	  }

	  /**
	   * Extract the attributes specified in a vertex shader.
	   *
	   * @param {string} vsCode - The vertex shader code.
	   * @returns {Object<string, string>} The attribute name to semantic map.
	   * @ignore
	   */;
	  ShaderUtils.collectAttributes = function collectAttributes(vsCode) {
	    var attribs = {};
	    var attrs = 0;
	    var found = vsCode.indexOf("attribute");
	    while (found >= 0) {
	      if (found > 0 && vsCode[found - 1] === "/") break;
	      var endOfLine = vsCode.indexOf(';', found);
	      var startOfAttribName = vsCode.lastIndexOf(' ', endOfLine);
	      var attribName = vsCode.substring(startOfAttribName + 1, endOfLine);
	      var semantic = _attrib2Semantic[attribName];
	      if (semantic !== undefined) {
	        attribs[attribName] = semantic;
	      } else {
	        attribs[attribName] = "ATTR" + attrs;
	        attrs++;
	      }
	      found = vsCode.indexOf("attribute", found + 1);
	    }
	    return attribs;
	  };
	  return ShaderUtils;
	}();

	/**
	 * Class holding information about the queries for a single frame.
	 *
	 * @ignore
	 */
	var FrameQueriesInfo = /*#__PURE__*/function () {
	  function FrameQueriesInfo() {
	    /**
	     * The render version of the frame.
	     *
	     * @type {number[]}
	     */
	    this.renderVersion = void 0;
	    /**
	     * The queries for the frame.
	     *
	     * @type {WebGLQuery[]}
	     */
	    this.queries = [];
	  }
	  var _proto = FrameQueriesInfo.prototype;
	  _proto.destroy = function destroy(gl) {
	    this.queries.forEach(function (query) {
	      return gl.deleteQuery(query);
	    });
	    this.queries = null;
	  };
	  return FrameQueriesInfo;
	}();
	/**
	 * @ignore
	 */
	var WebglGpuProfiler = /*#__PURE__*/function (_GpuProfiler) {
	  _inheritsLoose(WebglGpuProfiler, _GpuProfiler);
	  function WebglGpuProfiler(device) {
	    var _this;
	    _this = _GpuProfiler.call(this) || this;
	    _this.device = void 0;
	    /**
	     * The pool of unused queries.
	     *
	     * @type {WebGLQuery[]}
	     */
	    _this.freeQueries = [];
	    /**
	     * The pool of queries for the current frame.
	     *
	     * @type {WebGLQuery[]}
	     */
	    _this.frameQueries = [];
	    /**
	     * A list of queries from the previous frames which are waiting for results.
	     *
	     * @type {FrameQueriesInfo[]}
	     */
	    _this.previousFrameQueries = [];
	    /**
	     * Temporary array to storing the timings.
	     *
	     * @type {number[]}
	     */
	    _this.timings = [];
	    _this.device = device;
	    _this.ext = device.extDisjointTimerQuery;
	    return _this;
	  }
	  var _proto2 = WebglGpuProfiler.prototype;
	  _proto2.destroy = function destroy() {
	    var _this2 = this;
	    this.freeQueries.forEach(function (query) {
	      return _this2.device.gl.deleteQuery(query);
	    });
	    this.frameQueries.forEach(function (query) {
	      return _this2.device.gl.deleteQuery(query);
	    });
	    this.previousFrameQueries.forEach(function (frameQueriesInfo) {
	      return frameQueriesInfo.destroy(_this2.device.gl);
	    });
	    this.freeQueries = null;
	    this.frameQueries = null;
	    this.previousFrameQueries = null;
	  }

	  /**
	   * Called when the WebGL context was lost. It releases all context related resources.
	   *
	   * @ignore
	   */;
	  _proto2.loseContext = function loseContext() {
	    _GpuProfiler.prototype.loseContext.call(this);
	    this.freeQueries = [];
	    this.frameQueries = [];
	    this.previousFrameQueries = [];
	  };
	  _proto2.restoreContext = function restoreContext() {
	    this.ext = this.device.extDisjointTimerQuery;
	  };
	  _proto2.getQuery = function getQuery() {
	    var _this$freeQueries$pop;
	    return (_this$freeQueries$pop = this.freeQueries.pop()) != null ? _this$freeQueries$pop : this.device.gl.createQuery();
	  };
	  _proto2.start = function start(name) {
	    if (this.ext) {
	      var slot = this.getSlot(name);
	      var query = this.getQuery();
	      this.frameQueries[slot] = query;
	      this.device.gl.beginQuery(this.ext.TIME_ELAPSED_EXT, query);
	      return slot;
	    }
	    return undefined;
	  };
	  _proto2.end = function end(slot) {
	    if (slot !== undefined) {
	      this.device.gl.endQuery(this.ext.TIME_ELAPSED_EXT);
	    }
	  };
	  _proto2.frameStart = function frameStart() {
	    this.processEnableRequest();
	    if (this._enabled) {
	      this.frameGPUMarkerSlot = this.start('GpuFrame');
	    }
	  };
	  _proto2.frameEnd = function frameEnd() {
	    if (this._enabled) {
	      this.end(this.frameGPUMarkerSlot);
	    }
	  };
	  _proto2.request = function request() {
	    var _this3 = this;
	    if (this._enabled) {
	      var ext = this.ext;
	      var gl = this.device.gl;
	      var renderVersion = this.device.renderVersion;

	      // add current frame queries to the end of frames list
	      var frameQueries = this.frameQueries;
	      if (frameQueries.length > 0) {
	        this.frameQueries = [];
	        var frameQueriesInfo = new FrameQueriesInfo();
	        frameQueriesInfo.queries = frameQueries;
	        frameQueriesInfo.renderVersion = renderVersion;
	        this.previousFrameQueries.push(frameQueriesInfo);
	      }

	      // try to resolve the oldest frame
	      if (this.previousFrameQueries.length > 0) {
	        var previousQueriesInfo = this.previousFrameQueries[0];
	        var previousQueries = previousQueriesInfo.queries;
	        var lastQuery = previousQueries[previousQueries.length - 1];
	        var available = gl.getQueryParameter(lastQuery, gl.QUERY_RESULT_AVAILABLE);
	        var disjoint = gl.getParameter(ext.GPU_DISJOINT_EXT);

	        // valid results
	        if (available && !disjoint) {
	          // remove the oldest frame from the list
	          this.previousFrameQueries.shift();

	          // get timings
	          var timings = this.timings;
	          timings.length = 0;
	          for (var i = 0; i < previousQueries.length; i++) {
	            var query = previousQueries[i];
	            var duration = gl.getQueryParameter(query, gl.QUERY_RESULT);
	            timings[i] = duration * 0.000001;

	            // return queries to the pool
	            this.freeQueries.push(query);
	          }

	          // report timings
	          this.report(previousQueriesInfo.renderVersion, timings);
	        }

	        // GPU was interrupted, discard all in-flight queries
	        if (disjoint) {
	          this.previousFrameQueries.forEach(function (frameQueriesInfo) {
	            _this3.report(frameQueriesInfo.renderVersion, null);
	            frameQueriesInfo.destroy(gl);
	          });
	          this.previousFrameQueries.length = 0;
	        }
	      }
	      _GpuProfiler.prototype.request.call(this, renderVersion);
	    }
	  };
	  return WebglGpuProfiler;
	}(GpuProfiler);

	var invalidateAttachments = [];
	var _fullScreenQuadVS = /* glsl */"\nattribute vec2 vertex_position;\nvarying vec2 vUv0;\nvoid main(void)\n{\n    gl_Position = vec4(vertex_position, 0.5, 1.0);\n    vUv0 = vertex_position.xy*0.5+0.5;\n}\n";
	var _precisionTest1PS = /* glsl */"\nvoid main(void) { \n    gl_FragColor = vec4(2147483648.0);\n}\n";
	var _precisionTest2PS = /* glsl */"\nuniform sampler2D source;\nvec4 packFloat(float depth) {\n    const vec4 bit_shift = vec4(256.0 * 256.0 * 256.0, 256.0 * 256.0, 256.0, 1.0);\n    const vec4 bit_mask  = vec4(0.0, 1.0 / 256.0, 1.0 / 256.0, 1.0 / 256.0);\n    vec4 res = mod(depth * bit_shift * vec4(255), vec4(256) ) / vec4(255);\n    res -= res.xxyz * bit_mask;\n    return res;\n}\nvoid main(void) {\n    float c = texture2D(source, vec2(0.0)).r;\n    float diff = abs(c - 2147483648.0) / 2147483648.0;\n    gl_FragColor = packFloat(diff);\n}\n";
	var _outputTexture2D = /* glsl */"\nvarying vec2 vUv0;\nuniform sampler2D source;\nvoid main(void) {\n    gl_FragColor = texture2D(source, vUv0);\n}\n";
	function quadWithShader(device, target, shader) {
	  DebugGraphics.pushGpuMarker(device, "QuadWithShader");
	  var oldRt = device.renderTarget;
	  device.setRenderTarget(target);
	  device.updateBegin();
	  device.setCullMode(CULLFACE_NONE);
	  device.setBlendState(BlendState.NOBLEND);
	  device.setDepthState(DepthState.NODEPTH);
	  device.setStencilState(null, null);
	  device.setVertexBuffer(device.quadVertexBuffer, 0);
	  device.setShader(shader);
	  device.draw({
	    type: PRIMITIVE_TRISTRIP,
	    base: 0,
	    count: 4,
	    indexed: false
	  });
	  device.updateEnd();
	  device.setRenderTarget(oldRt);
	  device.updateBegin();
	  DebugGraphics.popGpuMarker(device);
	}
	function testRenderable(gl, pixelFormat) {
	  var result = true;

	  // Create a 2x2 texture
	  var texture = gl.createTexture();
	  gl.bindTexture(gl.TEXTURE_2D, texture);
	  gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MIN_FILTER, gl.NEAREST);
	  gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MAG_FILTER, gl.NEAREST);
	  gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_S, gl.CLAMP_TO_EDGE);
	  gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_T, gl.CLAMP_TO_EDGE);
	  gl.texImage2D(gl.TEXTURE_2D, 0, gl.RGBA, 2, 2, 0, gl.RGBA, pixelFormat, null);

	  // Try to use this texture as a render target
	  var framebuffer = gl.createFramebuffer();
	  gl.bindFramebuffer(gl.FRAMEBUFFER, framebuffer);
	  gl.framebufferTexture2D(gl.FRAMEBUFFER, gl.COLOR_ATTACHMENT0, gl.TEXTURE_2D, texture, 0);

	  // It is legal for a WebGL implementation exposing the OES_texture_float extension to
	  // support floating-point textures but not as attachments to framebuffer objects.
	  if (gl.checkFramebufferStatus(gl.FRAMEBUFFER) !== gl.FRAMEBUFFER_COMPLETE) {
	    result = false;
	  }

	  // Clean up
	  gl.bindTexture(gl.TEXTURE_2D, null);
	  gl.deleteTexture(texture);
	  gl.bindFramebuffer(gl.FRAMEBUFFER, null);
	  gl.deleteFramebuffer(framebuffer);
	  return result;
	}
	function testTextureHalfFloatUpdatable(gl, pixelFormat) {
	  var result = true;

	  // Create a 2x2 texture
	  var texture = gl.createTexture();
	  gl.bindTexture(gl.TEXTURE_2D, texture);
	  gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MIN_FILTER, gl.NEAREST);
	  gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MAG_FILTER, gl.NEAREST);
	  gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_S, gl.CLAMP_TO_EDGE);
	  gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_T, gl.CLAMP_TO_EDGE);

	  // upload some data - on iOS prior to about November 2019, passing data to half texture would fail here
	  // see details here: https://bugs.webkit.org/show_bug.cgi?id=169999
	  // note that if not supported, this prints an error to console, the error can be safely ignored as it's handled
	  var data = new Uint16Array(4 * 2 * 2);
	  gl.texImage2D(gl.TEXTURE_2D, 0, gl.RGBA, 2, 2, 0, gl.RGBA, pixelFormat, data);
	  if (gl.getError() !== gl.NO_ERROR) {
	    result = false;
	    console.log("Above error related to HALF_FLOAT_OES can be ignored, it was triggered by testing half float texture support");
	  }

	  // Clean up
	  gl.bindTexture(gl.TEXTURE_2D, null);
	  gl.deleteTexture(texture);
	  return result;
	}
	function testTextureFloatHighPrecision(device) {
	  if (!device.textureFloatRenderable) return false;
	  var shader1 = new Shader(device, ShaderUtils.createDefinition(device, {
	    name: 'ptest1',
	    vertexCode: _fullScreenQuadVS,
	    fragmentCode: _precisionTest1PS
	  }));
	  var shader2 = new Shader(device, ShaderUtils.createDefinition(device, {
	    name: 'ptest2',
	    vertexCode: _fullScreenQuadVS,
	    fragmentCode: _precisionTest2PS
	  }));
	  var textureOptions = {
	    format: PIXELFORMAT_RGBA32F,
	    width: 1,
	    height: 1,
	    mipmaps: false,
	    minFilter: FILTER_NEAREST,
	    magFilter: FILTER_NEAREST,
	    name: 'testFHP'
	  };
	  var tex1 = new Texture(device, textureOptions);
	  var targ1 = new RenderTarget({
	    colorBuffer: tex1,
	    depth: false
	  });
	  quadWithShader(device, targ1, shader1);
	  textureOptions.format = PIXELFORMAT_RGBA8;
	  var tex2 = new Texture(device, textureOptions);
	  var targ2 = new RenderTarget({
	    colorBuffer: tex2,
	    depth: false
	  });
	  device.constantTexSource.setValue(tex1);
	  quadWithShader(device, targ2, shader2);
	  var prevFramebuffer = device.activeFramebuffer;
	  device.setFramebuffer(targ2.impl._glFrameBuffer);
	  var pixels = new Uint8Array(4);
	  device.readPixels(0, 0, 1, 1, pixels);
	  device.setFramebuffer(prevFramebuffer);
	  var x = pixels[0] / 255;
	  var y = pixels[1] / 255;
	  var z = pixels[2] / 255;
	  var w = pixels[3] / 255;
	  var f = x / (256 * 256 * 256) + y / (256 * 256) + z / 256 + w;
	  tex1.destroy();
	  targ1.destroy();
	  tex2.destroy();
	  targ2.destroy();
	  shader1.destroy();
	  shader2.destroy();
	  return f === 0;
	}

	/**
	 * The graphics device manages the underlying graphics context. It is responsible for submitting
	 * render state changes and graphics primitives to the hardware. A graphics device is tied to a
	 * specific canvas HTML element. It is valid to have more than one canvas element per page and
	 * create a new graphics device against each.
	 *
	 * @augments GraphicsDevice
	 * @category Graphics
	 */
	var WebglGraphicsDevice = /*#__PURE__*/function (_GraphicsDevice) {
	  _inheritsLoose(WebglGraphicsDevice, _GraphicsDevice);
	  /**
	   * Creates a new WebglGraphicsDevice instance.
	   *
	   * @param {HTMLCanvasElement} canvas - The canvas to which the graphics device will render.
	   * @param {object} [options] - Options passed when creating the WebGL context.
	   * @param {boolean} [options.alpha] - Boolean that indicates if the canvas contains an
	   * alpha buffer. Defaults to true.
	   * @param {boolean} [options.depth] - Boolean that indicates that the drawing buffer is
	   * requested to have a depth buffer of at least 16 bits. Defaults to true.
	   * @param {boolean} [options.stencil] - Boolean that indicates that the drawing buffer is
	   * requested to have a stencil buffer of at least 8 bits. Defaults to true.
	   * @param {boolean} [options.antialias] - Boolean that indicates whether or not to perform
	   * anti-aliasing if possible. Defaults to true.
	   * @param {boolean} [options.premultipliedAlpha] - Boolean that indicates that the page
	   * compositor will assume the drawing buffer contains colors with pre-multiplied alpha.
	   * Defaults to true.
	   * @param {boolean} [options.preserveDrawingBuffer] - If the value is true the buffers will not
	   * be cleared and will preserve their values until cleared or overwritten by the author.
	   * Defaults to false.
	   * @param {'default'|'high-performance'|'low-power'} [options.powerPreference] - A hint to the
	   * user agent indicating what configuration of GPU is suitable for the WebGL context. Possible
	   * values are:
	   *
	   * - 'default': Let the user agent decide which GPU configuration is most suitable. This is the
	   * default value.
	   * - 'high-performance': Prioritizes rendering performance over power consumption.
	   * - 'low-power': Prioritizes power saving over rendering performance.
	   *
	   * Defaults to 'default'.
	   * @param {boolean} [options.failIfMajorPerformanceCaveat] - Boolean that indicates if a
	   * context will be created if the system performance is low or if no hardware GPU is available.
	   * Defaults to false.
	   * @param {boolean} [options.preferWebGl2] - Boolean that indicates if a WebGl2 context should
	   * be preferred. Defaults to true.
	   * @param {boolean} [options.desynchronized] - Boolean that hints the user agent to reduce the
	   * latency by desynchronizing the canvas paint cycle from the event loop. Defaults to false.
	   * @param {boolean} [options.xrCompatible] - Boolean that hints to the user agent to use a
	   * compatible graphics adapter for an immersive XR device.
	   * @param {WebGLRenderingContext | WebGL2RenderingContext} [options.gl] - The rendering context
	   * to use. If not specified, a new context will be created.
	   */
	  function WebglGraphicsDevice(canvas, options) {
	    var _options$antialias;
	    var _this;
	    if (options === void 0) {
	      options = {};
	    }
	    _this = _GraphicsDevice.call(this, canvas, options) || this;
	    /**
	     * The WebGL context managed by the graphics device. The type could also technically be
	     * `WebGLRenderingContext` if WebGL 2.0 is not available. But in order for IntelliSense to be
	     * able to function for all WebGL calls in the codebase, we specify `WebGL2RenderingContext`
	     * here instead.
	     *
	     * @type {WebGL2RenderingContext}
	     * @ignore
	     */
	    _this.gl = void 0;
	    /**
	     * WebGLFramebuffer object that represents the backbuffer of the device for a rendering frame.
	     * When null, this is a framebuffer created when the device was created, otherwise it is a
	     * framebuffer supplied by the XR session.
	     *
	     * @ignore
	     */
	    _this._defaultFramebuffer = null;
	    /**
	     * True if the default framebuffer has changed since the last frame.
	     *
	     * @ignore
	     */
	    _this._defaultFramebufferChanged = false;
	    options = _this.initOptions;
	    _this.updateClientRect();

	    // initialize this before registering lost context handlers to avoid undefined access when the device is created lost.
	    _this.initTextureUnits();

	    // Add handlers for when the WebGL context is lost or restored
	    _this.contextLost = false;
	    _this._contextLostHandler = function (event) {
	      event.preventDefault();
	      _this.loseContext();
	      Debug.log('pc.GraphicsDevice: WebGL context lost.');
	      _this.fire('devicelost');
	    };
	    _this._contextRestoredHandler = function () {
	      Debug.log('pc.GraphicsDevice: WebGL context restored.');
	      _this.restoreContext();
	      _this.fire('devicerestored');
	    };

	    // #4136 - turn off antialiasing on AppleWebKit browsers 15.4
	    var ua = typeof navigator !== 'undefined' && navigator.userAgent;
	    _this.forceDisableMultisampling = ua && ua.includes('AppleWebKit') && (ua.includes('15.4') || ua.includes('15_4'));
	    if (_this.forceDisableMultisampling) {
	      options.antialias = false;
	      Debug.log("Antialiasing has been turned off due to rendering issues on AppleWebKit 15.4");
	    }

	    // #5856 - turn off antialiasing on Windows Firefox
	    if (platform.browserName === 'firefox' && platform.name === 'windows') {
	      var _ua = typeof navigator !== 'undefined' ? navigator.userAgent : '';
	      var match = _ua.match(/Firefox\/(\d+(\.\d+)*)/);
	      var firefoxVersion = match ? match[1] : null;
	      if (firefoxVersion) {
	        var version = parseFloat(firefoxVersion);
	        if (version >= 120 || version === 115) {
	          options.antialias = false;
	          Debug.log("Antialiasing has been turned off due to rendering issues on Windows Firefox esr115 and 120+. Current version: " + firefoxVersion);
	        }
	      }
	    }

	    /** @type {WebGL2RenderingContext} */
	    var gl = null;

	    // we always allocate the default framebuffer without antialiasing, so remove that option
	    _this.backBufferAntialias = (_options$antialias = options.antialias) != null ? _options$antialias : false;
	    options.antialias = false;

	    // Retrieve the WebGL context
	    if (options.gl) {
	      gl = options.gl;
	    } else {
	      var preferWebGl2 = options.preferWebGl2 !== undefined ? options.preferWebGl2 : true;
	      var names = preferWebGl2 ? ["webgl2", "webgl", "experimental-webgl"] : ["webgl", "experimental-webgl"];
	      for (var i = 0; i < names.length; i++) {
	        gl = canvas.getContext(names[i], options);
	        if (gl) {
	          break;
	        }
	      }
	    }
	    if (!gl) {
	      throw new Error("WebGL not supported");
	    }
	    _this.gl = gl;
	    _this.isWebGL2 = typeof WebGL2RenderingContext !== 'undefined' && gl instanceof WebGL2RenderingContext;
	    _this.isWebGL1 = !_this.isWebGL2;
	    _this._deviceType = _this.isWebGL2 ? DEVICETYPE_WEBGL2 : DEVICETYPE_WEBGL1;

	    // pixel format of the framebuffer
	    _this.updateBackbufferFormat(null);
	    var isChrome = platform.browserName === 'chrome';
	    var isSafari = platform.browserName === 'safari';
	    var isMac = platform.browser && navigator.appVersion.indexOf("Mac") !== -1;

	    // enable temporary texture unit workaround on desktop safari
	    _this._tempEnableSafariTextureUnitWorkaround = isSafari;

	    // enable temporary workaround for glBlitFramebuffer failing on Mac Chrome (#2504)
	    _this._tempMacChromeBlitFramebufferWorkaround = isMac && isChrome && !options.alpha;
	    canvas.addEventListener("webglcontextlost", _this._contextLostHandler, false);
	    canvas.addEventListener("webglcontextrestored", _this._contextRestoredHandler, false);
	    _this.initializeExtensions();
	    _this.initializeCapabilities();
	    _this.initializeRenderState();
	    _this.initializeContextCaches();
	    _this.createBackbuffer(null);

	    // only enable ImageBitmap on chrome
	    _this.supportsImageBitmap = !isSafari && typeof ImageBitmap !== 'undefined';

	    // supported sampler types
	    _this._samplerTypes = new Set([gl.SAMPLER_2D, gl.SAMPLER_CUBE].concat(_this.isWebGL2 ? [gl.UNSIGNED_INT_SAMPLER_2D, gl.INT_SAMPLER_2D, gl.SAMPLER_2D_SHADOW, gl.SAMPLER_CUBE_SHADOW, gl.SAMPLER_3D, gl.INT_SAMPLER_3D, gl.UNSIGNED_INT_SAMPLER_3D, gl.SAMPLER_2D_ARRAY, gl.INT_SAMPLER_2D_ARRAY, gl.UNSIGNED_INT_SAMPLER_2D_ARRAY] : []));
	    _this.glAddress = [gl.REPEAT, gl.CLAMP_TO_EDGE, gl.MIRRORED_REPEAT];
	    _this.glBlendEquation = [gl.FUNC_ADD, gl.FUNC_SUBTRACT, gl.FUNC_REVERSE_SUBTRACT, _this.isWebGL2 ? gl.MIN : _this.extBlendMinmax ? _this.extBlendMinmax.MIN_EXT : gl.FUNC_ADD, _this.isWebGL2 ? gl.MAX : _this.extBlendMinmax ? _this.extBlendMinmax.MAX_EXT : gl.FUNC_ADD];
	    _this.glBlendFunctionColor = [gl.ZERO, gl.ONE, gl.SRC_COLOR, gl.ONE_MINUS_SRC_COLOR, gl.DST_COLOR, gl.ONE_MINUS_DST_COLOR, gl.SRC_ALPHA, gl.SRC_ALPHA_SATURATE, gl.ONE_MINUS_SRC_ALPHA, gl.DST_ALPHA, gl.ONE_MINUS_DST_ALPHA, gl.CONSTANT_COLOR, gl.ONE_MINUS_CONSTANT_COLOR];
	    _this.glBlendFunctionAlpha = [gl.ZERO, gl.ONE, gl.SRC_COLOR, gl.ONE_MINUS_SRC_COLOR, gl.DST_COLOR, gl.ONE_MINUS_DST_COLOR, gl.SRC_ALPHA, gl.SRC_ALPHA_SATURATE, gl.ONE_MINUS_SRC_ALPHA, gl.DST_ALPHA, gl.ONE_MINUS_DST_ALPHA, gl.CONSTANT_ALPHA, gl.ONE_MINUS_CONSTANT_ALPHA];
	    _this.glComparison = [gl.NEVER, gl.LESS, gl.EQUAL, gl.LEQUAL, gl.GREATER, gl.NOTEQUAL, gl.GEQUAL, gl.ALWAYS];
	    _this.glStencilOp = [gl.KEEP, gl.ZERO, gl.REPLACE, gl.INCR, gl.INCR_WRAP, gl.DECR, gl.DECR_WRAP, gl.INVERT];
	    _this.glClearFlag = [0, gl.COLOR_BUFFER_BIT, gl.DEPTH_BUFFER_BIT, gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT, gl.STENCIL_BUFFER_BIT, gl.STENCIL_BUFFER_BIT | gl.COLOR_BUFFER_BIT, gl.STENCIL_BUFFER_BIT | gl.DEPTH_BUFFER_BIT, gl.STENCIL_BUFFER_BIT | gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT];
	    _this.glCull = [0, gl.BACK, gl.FRONT, gl.FRONT_AND_BACK];
	    _this.glFilter = [gl.NEAREST, gl.LINEAR, gl.NEAREST_MIPMAP_NEAREST, gl.NEAREST_MIPMAP_LINEAR, gl.LINEAR_MIPMAP_NEAREST, gl.LINEAR_MIPMAP_LINEAR];
	    _this.glPrimitive = [gl.POINTS, gl.LINES, gl.LINE_LOOP, gl.LINE_STRIP, gl.TRIANGLES, gl.TRIANGLE_STRIP, gl.TRIANGLE_FAN];
	    _this.glType = [gl.BYTE, gl.UNSIGNED_BYTE, gl.SHORT, gl.UNSIGNED_SHORT, gl.INT, gl.UNSIGNED_INT, gl.FLOAT, gl.HALF_FLOAT];
	    _this.pcUniformType = {};
	    _this.pcUniformType[gl.BOOL] = UNIFORMTYPE_BOOL;
	    _this.pcUniformType[gl.INT] = UNIFORMTYPE_INT;
	    _this.pcUniformType[gl.FLOAT] = UNIFORMTYPE_FLOAT;
	    _this.pcUniformType[gl.FLOAT_VEC2] = UNIFORMTYPE_VEC2;
	    _this.pcUniformType[gl.FLOAT_VEC3] = UNIFORMTYPE_VEC3;
	    _this.pcUniformType[gl.FLOAT_VEC4] = UNIFORMTYPE_VEC4;
	    _this.pcUniformType[gl.INT_VEC2] = UNIFORMTYPE_IVEC2;
	    _this.pcUniformType[gl.INT_VEC3] = UNIFORMTYPE_IVEC3;
	    _this.pcUniformType[gl.INT_VEC4] = UNIFORMTYPE_IVEC4;
	    _this.pcUniformType[gl.BOOL_VEC2] = UNIFORMTYPE_BVEC2;
	    _this.pcUniformType[gl.BOOL_VEC3] = UNIFORMTYPE_BVEC3;
	    _this.pcUniformType[gl.BOOL_VEC4] = UNIFORMTYPE_BVEC4;
	    _this.pcUniformType[gl.FLOAT_MAT2] = UNIFORMTYPE_MAT2;
	    _this.pcUniformType[gl.FLOAT_MAT3] = UNIFORMTYPE_MAT3;
	    _this.pcUniformType[gl.FLOAT_MAT4] = UNIFORMTYPE_MAT4;
	    _this.pcUniformType[gl.SAMPLER_2D] = UNIFORMTYPE_TEXTURE2D;
	    _this.pcUniformType[gl.SAMPLER_CUBE] = UNIFORMTYPE_TEXTURECUBE;
	    _this.pcUniformType[gl.UNSIGNED_INT] = UNIFORMTYPE_UINT;
	    _this.pcUniformType[gl.UNSIGNED_INT_VEC2] = UNIFORMTYPE_UVEC2;
	    _this.pcUniformType[gl.UNSIGNED_INT_VEC3] = UNIFORMTYPE_UVEC3;
	    _this.pcUniformType[gl.UNSIGNED_INT_VEC4] = UNIFORMTYPE_UVEC4;
	    if (_this.isWebGL2) {
	      _this.pcUniformType[gl.SAMPLER_2D_SHADOW] = UNIFORMTYPE_TEXTURE2D_SHADOW;
	      _this.pcUniformType[gl.SAMPLER_CUBE_SHADOW] = UNIFORMTYPE_TEXTURECUBE_SHADOW;
	      _this.pcUniformType[gl.SAMPLER_2D_ARRAY] = UNIFORMTYPE_TEXTURE2D_ARRAY;
	      _this.pcUniformType[gl.SAMPLER_3D] = UNIFORMTYPE_TEXTURE3D;
	      _this.pcUniformType[gl.INT_SAMPLER_2D] = UNIFORMTYPE_ITEXTURE2D;
	      _this.pcUniformType[gl.UNSIGNED_INT_SAMPLER_2D] = UNIFORMTYPE_UTEXTURE2D;
	      _this.pcUniformType[gl.INT_SAMPLER_CUBE] = UNIFORMTYPE_ITEXTURECUBE;
	      _this.pcUniformType[gl.UNSIGNED_INT_SAMPLER_2D] = UNIFORMTYPE_UTEXTURECUBE;
	      _this.pcUniformType[gl.INT_SAMPLER_3D] = UNIFORMTYPE_ITEXTURE3D;
	      _this.pcUniformType[gl.UNSIGNED_INT_SAMPLER_3D] = UNIFORMTYPE_UTEXTURE3D;
	      _this.pcUniformType[gl.INT_SAMPLER_2D_ARRAY] = UNIFORMTYPE_ITEXTURE2D_ARRAY;
	      _this.pcUniformType[gl.UNSIGNED_INT_SAMPLER_2D_ARRAY] = UNIFORMTYPE_UTEXTURE2D_ARRAY;
	    }
	    _this.targetToSlot = {};
	    _this.targetToSlot[gl.TEXTURE_2D] = 0;
	    _this.targetToSlot[gl.TEXTURE_CUBE_MAP] = 1;
	    _this.targetToSlot[gl.TEXTURE_3D] = 2;

	    // Define the uniform commit functions
	    var scopeX, scopeY, scopeZ, scopeW;
	    var uniformValue;
	    _this.commitFunction = [];
	    _this.commitFunction[UNIFORMTYPE_BOOL] = function (uniform, value) {
	      if (uniform.value !== value) {
	        gl.uniform1i(uniform.locationId, value);
	        uniform.value = value;
	      }
	    };
	    _this.commitFunction[UNIFORMTYPE_INT] = _this.commitFunction[UNIFORMTYPE_BOOL];
	    _this.commitFunction[UNIFORMTYPE_FLOAT] = function (uniform, value) {
	      if (uniform.value !== value) {
	        gl.uniform1f(uniform.locationId, value);
	        uniform.value = value;
	      }
	    };
	    _this.commitFunction[UNIFORMTYPE_VEC2] = function (uniform, value) {
	      uniformValue = uniform.value;
	      scopeX = value[0];
	      scopeY = value[1];
	      if (uniformValue[0] !== scopeX || uniformValue[1] !== scopeY) {
	        gl.uniform2fv(uniform.locationId, value);
	        uniformValue[0] = scopeX;
	        uniformValue[1] = scopeY;
	      }
	    };
	    _this.commitFunction[UNIFORMTYPE_VEC3] = function (uniform, value) {
	      uniformValue = uniform.value;
	      scopeX = value[0];
	      scopeY = value[1];
	      scopeZ = value[2];
	      if (uniformValue[0] !== scopeX || uniformValue[1] !== scopeY || uniformValue[2] !== scopeZ) {
	        gl.uniform3fv(uniform.locationId, value);
	        uniformValue[0] = scopeX;
	        uniformValue[1] = scopeY;
	        uniformValue[2] = scopeZ;
	      }
	    };
	    _this.commitFunction[UNIFORMTYPE_VEC4] = function (uniform, value) {
	      uniformValue = uniform.value;
	      scopeX = value[0];
	      scopeY = value[1];
	      scopeZ = value[2];
	      scopeW = value[3];
	      if (uniformValue[0] !== scopeX || uniformValue[1] !== scopeY || uniformValue[2] !== scopeZ || uniformValue[3] !== scopeW) {
	        gl.uniform4fv(uniform.locationId, value);
	        uniformValue[0] = scopeX;
	        uniformValue[1] = scopeY;
	        uniformValue[2] = scopeZ;
	        uniformValue[3] = scopeW;
	      }
	    };
	    _this.commitFunction[UNIFORMTYPE_IVEC2] = function (uniform, value) {
	      uniformValue = uniform.value;
	      scopeX = value[0];
	      scopeY = value[1];
	      if (uniformValue[0] !== scopeX || uniformValue[1] !== scopeY) {
	        gl.uniform2iv(uniform.locationId, value);
	        uniformValue[0] = scopeX;
	        uniformValue[1] = scopeY;
	      }
	    };
	    _this.commitFunction[UNIFORMTYPE_BVEC2] = _this.commitFunction[UNIFORMTYPE_IVEC2];
	    _this.commitFunction[UNIFORMTYPE_IVEC3] = function (uniform, value) {
	      uniformValue = uniform.value;
	      scopeX = value[0];
	      scopeY = value[1];
	      scopeZ = value[2];
	      if (uniformValue[0] !== scopeX || uniformValue[1] !== scopeY || uniformValue[2] !== scopeZ) {
	        gl.uniform3iv(uniform.locationId, value);
	        uniformValue[0] = scopeX;
	        uniformValue[1] = scopeY;
	        uniformValue[2] = scopeZ;
	      }
	    };
	    _this.commitFunction[UNIFORMTYPE_BVEC3] = _this.commitFunction[UNIFORMTYPE_IVEC3];
	    _this.commitFunction[UNIFORMTYPE_IVEC4] = function (uniform, value) {
	      uniformValue = uniform.value;
	      scopeX = value[0];
	      scopeY = value[1];
	      scopeZ = value[2];
	      scopeW = value[3];
	      if (uniformValue[0] !== scopeX || uniformValue[1] !== scopeY || uniformValue[2] !== scopeZ || uniformValue[3] !== scopeW) {
	        gl.uniform4iv(uniform.locationId, value);
	        uniformValue[0] = scopeX;
	        uniformValue[1] = scopeY;
	        uniformValue[2] = scopeZ;
	        uniformValue[3] = scopeW;
	      }
	    };
	    _this.commitFunction[UNIFORMTYPE_BVEC4] = _this.commitFunction[UNIFORMTYPE_IVEC4];
	    _this.commitFunction[UNIFORMTYPE_MAT2] = function (uniform, value) {
	      gl.uniformMatrix2fv(uniform.locationId, false, value);
	    };
	    _this.commitFunction[UNIFORMTYPE_MAT3] = function (uniform, value) {
	      gl.uniformMatrix3fv(uniform.locationId, false, value);
	    };
	    _this.commitFunction[UNIFORMTYPE_MAT4] = function (uniform, value) {
	      gl.uniformMatrix4fv(uniform.locationId, false, value);
	    };
	    _this.commitFunction[UNIFORMTYPE_FLOATARRAY] = function (uniform, value) {
	      gl.uniform1fv(uniform.locationId, value);
	    };
	    _this.commitFunction[UNIFORMTYPE_VEC2ARRAY] = function (uniform, value) {
	      gl.uniform2fv(uniform.locationId, value);
	    };
	    _this.commitFunction[UNIFORMTYPE_VEC3ARRAY] = function (uniform, value) {
	      gl.uniform3fv(uniform.locationId, value);
	    };
	    _this.commitFunction[UNIFORMTYPE_VEC4ARRAY] = function (uniform, value) {
	      gl.uniform4fv(uniform.locationId, value);
	    };
	    _this.commitFunction[UNIFORMTYPE_UINT] = function (uniform, value) {
	      if (uniform.value !== value) {
	        gl.uniform1ui(uniform.locationId, value);
	        uniform.value = value;
	      }
	    };
	    _this.commitFunction[UNIFORMTYPE_UVEC2] = function (uniform, value) {
	      uniformValue = uniform.value;
	      scopeX = value[0];
	      scopeY = value[1];
	      if (uniformValue[0] !== scopeX || uniformValue[1] !== scopeY) {
	        gl.uniform2uiv(uniform.locationId, value);
	        uniformValue[0] = scopeX;
	        uniformValue[1] = scopeY;
	      }
	    };
	    _this.commitFunction[UNIFORMTYPE_UVEC3] = function (uniform, value) {
	      uniformValue = uniform.value;
	      scopeX = value[0];
	      scopeY = value[1];
	      scopeZ = value[2];
	      if (uniformValue[0] !== scopeX || uniformValue[1] !== scopeY || uniformValue[2] !== scopeZ) {
	        gl.uniform3uiv(uniform.locationId, value);
	        uniformValue[0] = scopeX;
	        uniformValue[1] = scopeY;
	        uniformValue[2] = scopeZ;
	      }
	    };
	    _this.commitFunction[UNIFORMTYPE_UVEC4] = function (uniform, value) {
	      uniformValue = uniform.value;
	      scopeX = value[0];
	      scopeY = value[1];
	      scopeZ = value[2];
	      scopeW = value[3];
	      if (uniformValue[0] !== scopeX || uniformValue[1] !== scopeY || uniformValue[2] !== scopeZ || uniformValue[3] !== scopeW) {
	        gl.uniform4uiv(uniform.locationId, value);
	        uniformValue[0] = scopeX;
	        uniformValue[1] = scopeY;
	        uniformValue[2] = scopeZ;
	        uniformValue[3] = scopeW;
	      }
	    };
	    _this.commitFunction[UNIFORMTYPE_INTARRAY] = function (uniform, value) {
	      gl.uniform1iv(uniform.locationId, value);
	    };
	    _this.commitFunction[UNIFORMTYPE_UINTARRAY] = function (uniform, value) {
	      gl.uniform1uiv(uniform.locationId, value);
	    };
	    _this.commitFunction[UNIFORMTYPE_BOOLARRAY] = _this.commitFunction[UNIFORMTYPE_INTARRAY];
	    _this.commitFunction[UNIFORMTYPE_IVEC2ARRAY] = function (uniform, value) {
	      gl.uniform2iv(uniform.locationId, value);
	    };
	    _this.commitFunction[UNIFORMTYPE_UVEC2ARRAY] = function (uniform, value) {
	      gl.uniform2uiv(uniform.locationId, value);
	    };
	    _this.commitFunction[UNIFORMTYPE_BVEC2ARRAY] = _this.commitFunction[UNIFORMTYPE_IVEC2ARRAY];
	    _this.commitFunction[UNIFORMTYPE_IVEC3ARRAY] = function (uniform, value) {
	      gl.uniform3iv(uniform.locationId, value);
	    };
	    _this.commitFunction[UNIFORMTYPE_UVEC3ARRAY] = function (uniform, value) {
	      gl.uniform3uiv(uniform.locationId, value);
	    };
	    _this.commitFunction[UNIFORMTYPE_BVEC3ARRAY] = _this.commitFunction[UNIFORMTYPE_IVEC3ARRAY];
	    _this.commitFunction[UNIFORMTYPE_IVEC4ARRAY] = function (uniform, value) {
	      gl.uniform4iv(uniform.locationId, value);
	    };
	    _this.commitFunction[UNIFORMTYPE_UVEC4ARRAY] = function (uniform, value) {
	      gl.uniform4uiv(uniform.locationId, value);
	    };
	    _this.commitFunction[UNIFORMTYPE_BVEC4ARRAY] = _this.commitFunction[UNIFORMTYPE_IVEC4ARRAY];
	    _this.commitFunction[UNIFORMTYPE_MAT4ARRAY] = function (uniform, value) {
	      gl.uniformMatrix4fv(uniform.locationId, false, value);
	    };
	    _this.supportsBoneTextures = _this.extTextureFloat && _this.maxVertexTextures > 0;

	    // Calculate an estimate of the maximum number of bones that can be uploaded to the GPU
	    // based on the number of available uniforms and the number of uniforms required for non-
	    // bone data.  This is based off of the Standard shader.  A user defined shader may have
	    // even less space available for bones so this calculated value can be overridden via
	    // pc.GraphicsDevice.setBoneLimit.
	    var numUniforms = _this.vertexUniformsCount;
	    numUniforms -= 4 * 4; // Model, view, projection and shadow matrices
	    numUniforms -= 8; // 8 lights max, each specifying a position vector
	    numUniforms -= 1; // Eye position
	    numUniforms -= 4 * 4; // Up to 4 texture transforms
	    _this.boneLimit = Math.floor(numUniforms / 3); // each bone uses 3 uniforms

	    // Put a limit on the number of supported bones before skin partitioning must be performed
	    // Some GPUs have demonstrated performance issues if the number of vectors allocated to the
	    // skin matrix palette is left unbounded
	    _this.boneLimit = Math.min(_this.boneLimit, 128);
	    if (_this.unmaskedRenderer === 'Mali-450 MP') {
	      _this.boneLimit = 34;
	    }
	    _this.constantTexSource = _this.scope.resolve("source");
	    if (_this.extTextureFloat) {
	      if (_this.isWebGL2) {
	        // In WebGL2 float texture renderability is dictated by the EXT_color_buffer_float extension
	        _this.textureFloatRenderable = !!_this.extColorBufferFloat;
	      } else {
	        // In WebGL1 we should just try rendering into a float texture
	        _this.textureFloatRenderable = testRenderable(gl, gl.FLOAT);
	      }
	    } else {
	      _this.textureFloatRenderable = false;
	    }

	    // two extensions allow us to render to half float buffers
	    if (_this.extColorBufferHalfFloat) {
	      _this.textureHalfFloatRenderable = !!_this.extColorBufferHalfFloat;
	    } else if (_this.extTextureHalfFloat) {
	      if (_this.isWebGL2) {
	        // EXT_color_buffer_float should affect both float and halffloat formats
	        _this.textureHalfFloatRenderable = !!_this.extColorBufferFloat;
	      } else {
	        // Manual render check for half float
	        _this.textureHalfFloatRenderable = testRenderable(gl, _this.extTextureHalfFloat.HALF_FLOAT_OES);
	      }
	    } else {
	      _this.textureHalfFloatRenderable = false;
	    }
	    _this.supportsMorphTargetTexturesCore = _this.maxPrecision === "highp" && _this.maxVertexTextures >= 2;
	    _this.supportsDepthShadow = _this.isWebGL2;
	    _this._textureFloatHighPrecision = undefined;
	    _this._textureHalfFloatUpdatable = undefined;

	    // area light LUT format - order of preference: half, float, 8bit
	    _this.areaLightLutFormat = PIXELFORMAT_RGBA8;
	    if (_this.extTextureHalfFloat && _this.textureHalfFloatUpdatable && _this.extTextureHalfFloatLinear) {
	      _this.areaLightLutFormat = PIXELFORMAT_RGBA16F;
	    } else if (_this.extTextureFloat && _this.extTextureFloatLinear) {
	      _this.areaLightLutFormat = PIXELFORMAT_RGBA32F;
	    }
	    _this.postInit();
	    return _this;
	  }
	  var _proto = WebglGraphicsDevice.prototype;
	  _proto.postInit = function postInit() {
	    _GraphicsDevice.prototype.postInit.call(this);
	    this.gpuProfiler = new WebglGpuProfiler(this);
	  }

	  /**
	   * Destroy the graphics device.
	   */;
	  _proto.destroy = function destroy() {
	    _GraphicsDevice.prototype.destroy.call(this);
	    var gl = this.gl;
	    if (this.isWebGL2 && this.feedback) {
	      gl.deleteTransformFeedback(this.feedback);
	    }
	    this.clearVertexArrayObjectCache();
	    this.canvas.removeEventListener('webglcontextlost', this._contextLostHandler, false);
	    this.canvas.removeEventListener('webglcontextrestored', this._contextRestoredHandler, false);
	    this._contextLostHandler = null;
	    this._contextRestoredHandler = null;
	    this.gl = null;
	    _GraphicsDevice.prototype.postDestroy.call(this);
	  };
	  _proto.createBackbuffer = function createBackbuffer(frameBuffer) {
	    this.supportsStencil = this.initOptions.stencil;
	    this.backBuffer = new RenderTarget({
	      name: 'WebglFramebuffer',
	      graphicsDevice: this,
	      depth: this.initOptions.depth,
	      stencil: this.supportsStencil,
	      samples: this.samples
	    });

	    // use the default WebGL framebuffer for rendering
	    this.backBuffer.impl.suppliedColorFramebuffer = frameBuffer;
	  }

	  // Update framebuffer format based on the current framebuffer, as this is use to create matching multi-sampled framebuffer
	  ;
	  _proto.updateBackbufferFormat = function updateBackbufferFormat(framebuffer) {
	    var gl = this.gl;
	    gl.bindFramebuffer(gl.FRAMEBUFFER, framebuffer);
	    var alphaBits = this.gl.getParameter(this.gl.ALPHA_BITS);
	    this.backBufferFormat = alphaBits ? PIXELFORMAT_RGBA8 : PIXELFORMAT_RGB8;
	  };
	  _proto.updateBackbuffer = function updateBackbuffer() {
	    var resolutionChanged = this.canvas.width !== this.backBufferSize.x || this.canvas.height !== this.backBufferSize.y;
	    if (this._defaultFramebufferChanged || resolutionChanged) {
	      // if the default framebuffer changes (entering or exiting XR for example)
	      if (this._defaultFramebufferChanged) {
	        this.updateBackbufferFormat(this._defaultFramebuffer);
	      }
	      this._defaultFramebufferChanged = false;
	      this.backBufferSize.set(this.canvas.width, this.canvas.height);

	      // recreate the backbuffer with newly supplied framebuffer
	      this.backBuffer.destroy();
	      this.createBackbuffer(this._defaultFramebuffer);
	    }
	  }

	  // provide webgl implementation for the vertex buffer
	  ;
	  _proto.createVertexBufferImpl = function createVertexBufferImpl(vertexBuffer, format) {
	    return new WebglVertexBuffer();
	  }

	  // provide webgl implementation for the index buffer
	  ;
	  _proto.createIndexBufferImpl = function createIndexBufferImpl(indexBuffer) {
	    return new WebglIndexBuffer(indexBuffer);
	  };
	  _proto.createShaderImpl = function createShaderImpl(shader) {
	    return new WebglShader(shader);
	  };
	  _proto.createTextureImpl = function createTextureImpl(texture) {
	    return new WebglTexture();
	  };
	  _proto.createRenderTargetImpl = function createRenderTargetImpl(renderTarget) {
	    return new WebglRenderTarget();
	  };
	  _proto.pushMarker = function pushMarker(name) {
	    if (platform.browser && window.spector) {
	      var label = DebugGraphics.toString();
	      window.spector.setMarker(label + " #");
	    }
	  };
	  _proto.popMarker = function popMarker() {
	    if (platform.browser && window.spector) {
	      var label = DebugGraphics.toString();
	      if (label.length) window.spector.setMarker(label + " #");else window.spector.clearMarker();
	    }
	  }

	  /**
	   * Query the precision supported by ints and floats in vertex and fragment shaders. Note that
	   * getShaderPrecisionFormat is not guaranteed to be present (such as some instances of the
	   * default Android browser). In this case, assume highp is available.
	   *
	   * @returns {string} "highp", "mediump" or "lowp"
	   * @ignore
	   */;
	  _proto.getPrecision = function getPrecision() {
	    var gl = this.gl;
	    var precision = "highp";
	    if (gl.getShaderPrecisionFormat) {
	      var vertexShaderPrecisionHighpFloat = gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.HIGH_FLOAT);
	      var vertexShaderPrecisionMediumpFloat = gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.MEDIUM_FLOAT);
	      var fragmentShaderPrecisionHighpFloat = gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.HIGH_FLOAT);
	      var fragmentShaderPrecisionMediumpFloat = gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.MEDIUM_FLOAT);
	      if (vertexShaderPrecisionHighpFloat && vertexShaderPrecisionMediumpFloat && fragmentShaderPrecisionHighpFloat && fragmentShaderPrecisionMediumpFloat) {
	        var highpAvailable = vertexShaderPrecisionHighpFloat.precision > 0 && fragmentShaderPrecisionHighpFloat.precision > 0;
	        var mediumpAvailable = vertexShaderPrecisionMediumpFloat.precision > 0 && fragmentShaderPrecisionMediumpFloat.precision > 0;
	        if (!highpAvailable) {
	          if (mediumpAvailable) {
	            precision = "mediump";
	            Debug.warn("WARNING: highp not supported, using mediump");
	          } else {
	            precision = "lowp";
	            Debug.warn("WARNING: highp and mediump not supported, using lowp");
	          }
	        }
	      }
	    }
	    return precision;
	  };
	  _proto.getExtension = function getExtension() {
	    for (var i = 0; i < arguments.length; i++) {
	      if (this.supportedExtensions.indexOf(arguments[i]) !== -1) {
	        return this.gl.getExtension(arguments[i]);
	      }
	    }
	    return null;
	  }

	  /** @ignore */;
	  /**
	   * Initialize the extensions provided by the WebGL context.
	   *
	   * @ignore
	   */
	  _proto.initializeExtensions = function initializeExtensions() {
	    var _gl$getSupportedExten;
	    var gl = this.gl;
	    this.supportedExtensions = (_gl$getSupportedExten = gl.getSupportedExtensions()) != null ? _gl$getSupportedExten : [];
	    this._extDisjointTimerQuery = null;
	    if (this.isWebGL2) {
	      this.extBlendMinmax = true;
	      this.extDrawBuffers = true;
	      this.drawBuffers = gl.drawBuffers.bind(gl);
	      this.extInstancing = true;
	      this.extStandardDerivatives = true;
	      this.extTextureFloat = true;
	      this.extTextureHalfFloat = true;
	      this.textureHalfFloatFilterable = true;
	      this.extTextureLod = true;
	      this.extUintElement = true;
	      this.extVertexArrayObject = true;
	      this.extColorBufferFloat = this.getExtension('EXT_color_buffer_float');
	      this.extDepthTexture = true;
	      this.textureRG11B10Renderable = true;
	    } else {
	      var _this$extDrawBuffers;
	      this.extBlendMinmax = this.getExtension("EXT_blend_minmax");
	      this.extDrawBuffers = this.getExtension('WEBGL_draw_buffers');
	      this.extInstancing = this.getExtension("ANGLE_instanced_arrays");
	      this.drawBuffers = (_this$extDrawBuffers = this.extDrawBuffers) == null ? void 0 : _this$extDrawBuffers.drawBuffersWEBGL.bind(this.extDrawBuffers);
	      if (this.extInstancing) {
	        // Install the WebGL 2 Instancing API for WebGL 1.0
	        var ext = this.extInstancing;
	        gl.drawArraysInstanced = ext.drawArraysInstancedANGLE.bind(ext);
	        gl.drawElementsInstanced = ext.drawElementsInstancedANGLE.bind(ext);
	        gl.vertexAttribDivisor = ext.vertexAttribDivisorANGLE.bind(ext);
	      }
	      this.extStandardDerivatives = this.getExtension("OES_standard_derivatives");
	      this.extTextureFloat = this.getExtension("OES_texture_float");
	      this.extTextureLod = this.getExtension('EXT_shader_texture_lod');
	      this.extUintElement = this.getExtension("OES_element_index_uint");
	      this.extVertexArrayObject = this.getExtension("OES_vertex_array_object");
	      if (this.extVertexArrayObject) {
	        // Install the WebGL 2 VAO API for WebGL 1.0
	        var _ext = this.extVertexArrayObject;
	        gl.createVertexArray = _ext.createVertexArrayOES.bind(_ext);
	        gl.deleteVertexArray = _ext.deleteVertexArrayOES.bind(_ext);
	        gl.isVertexArray = _ext.isVertexArrayOES.bind(_ext);
	        gl.bindVertexArray = _ext.bindVertexArrayOES.bind(_ext);
	      }
	      this.extColorBufferFloat = null;
	      this.extDepthTexture = gl.getExtension('WEBGL_depth_texture');
	      this.extTextureHalfFloat = this.getExtension("OES_texture_half_float");
	      this.extTextureHalfFloatLinear = this.getExtension("OES_texture_half_float_linear");
	      this.textureHalfFloatFilterable = !!this.extTextureHalfFloatLinear;
	    }
	    this.extDebugRendererInfo = this.getExtension('WEBGL_debug_renderer_info');
	    this.extTextureFloatLinear = this.getExtension("OES_texture_float_linear");
	    this.textureFloatFilterable = !!this.extTextureFloatLinear;
	    this.extFloatBlend = this.getExtension("EXT_float_blend");
	    this.extTextureFilterAnisotropic = this.getExtension('EXT_texture_filter_anisotropic', 'WEBKIT_EXT_texture_filter_anisotropic');
	    this.extCompressedTextureETC1 = this.getExtension('WEBGL_compressed_texture_etc1');
	    this.extCompressedTextureETC = this.getExtension('WEBGL_compressed_texture_etc');
	    this.extCompressedTexturePVRTC = this.getExtension('WEBGL_compressed_texture_pvrtc', 'WEBKIT_WEBGL_compressed_texture_pvrtc');
	    this.extCompressedTextureS3TC = this.getExtension('WEBGL_compressed_texture_s3tc', 'WEBKIT_WEBGL_compressed_texture_s3tc');
	    this.extCompressedTextureATC = this.getExtension('WEBGL_compressed_texture_atc');
	    this.extCompressedTextureASTC = this.getExtension('WEBGL_compressed_texture_astc');
	    this.extParallelShaderCompile = this.getExtension('KHR_parallel_shader_compile');

	    // iOS exposes this for half precision render targets on both Webgl1 and 2 from iOS v 14.5beta
	    this.extColorBufferHalfFloat = this.getExtension("EXT_color_buffer_half_float");
	  }

	  /**
	   * Query the capabilities of the WebGL context.
	   *
	   * @ignore
	   */;
	  _proto.initializeCapabilities = function initializeCapabilities() {
	    var _contextAttribs$antia, _contextAttribs$stenc;
	    var gl = this.gl;
	    var ext;
	    var userAgent = typeof navigator !== 'undefined' ? navigator.userAgent : "";
	    this.maxPrecision = this.precision = this.getPrecision();
	    var contextAttribs = gl.getContextAttributes();
	    this.supportsMsaa = (_contextAttribs$antia = contextAttribs == null ? void 0 : contextAttribs.antialias) != null ? _contextAttribs$antia : false;
	    this.supportsStencil = (_contextAttribs$stenc = contextAttribs == null ? void 0 : contextAttribs.stencil) != null ? _contextAttribs$stenc : false;
	    this.supportsInstancing = !!this.extInstancing;

	    // Query parameter values from the WebGL context
	    this.maxTextureSize = gl.getParameter(gl.MAX_TEXTURE_SIZE);
	    this.maxCubeMapSize = gl.getParameter(gl.MAX_CUBE_MAP_TEXTURE_SIZE);
	    this.maxRenderBufferSize = gl.getParameter(gl.MAX_RENDERBUFFER_SIZE);
	    this.maxTextures = gl.getParameter(gl.MAX_TEXTURE_IMAGE_UNITS);
	    this.maxCombinedTextures = gl.getParameter(gl.MAX_COMBINED_TEXTURE_IMAGE_UNITS);
	    this.maxVertexTextures = gl.getParameter(gl.MAX_VERTEX_TEXTURE_IMAGE_UNITS);
	    this.vertexUniformsCount = gl.getParameter(gl.MAX_VERTEX_UNIFORM_VECTORS);
	    this.fragmentUniformsCount = gl.getParameter(gl.MAX_FRAGMENT_UNIFORM_VECTORS);
	    if (this.isWebGL2) {
	      this.maxDrawBuffers = gl.getParameter(gl.MAX_DRAW_BUFFERS);
	      this.maxColorAttachments = gl.getParameter(gl.MAX_COLOR_ATTACHMENTS);
	      this.maxVolumeSize = gl.getParameter(gl.MAX_3D_TEXTURE_SIZE);
	      this.supportsMrt = true;
	      this.supportsVolumeTextures = true;
	    } else {
	      ext = this.extDrawBuffers;
	      this.supportsMrt = !!ext;
	      this.maxDrawBuffers = ext ? gl.getParameter(ext.MAX_DRAW_BUFFERS_WEBGL) : 1;
	      this.maxColorAttachments = ext ? gl.getParameter(ext.MAX_COLOR_ATTACHMENTS_WEBGL) : 1;
	      this.maxVolumeSize = 1;
	    }
	    ext = this.extDebugRendererInfo;
	    this.unmaskedRenderer = ext ? gl.getParameter(ext.UNMASKED_RENDERER_WEBGL) : '';
	    this.unmaskedVendor = ext ? gl.getParameter(ext.UNMASKED_VENDOR_WEBGL) : '';

	    // Mali-G52 has rendering issues with GPU particles including
	    // SM-A225M, M2003J15SC and KFRAWI (Amazon Fire HD 8 2022)
	    var maliRendererRegex = /\bMali-G52+/;

	    // Samsung devices with Exynos (ARM) either crash or render incorrectly when using GPU for particles. See:
	    // https://github.com/playcanvas/engine/issues/3967
	    // https://github.com/playcanvas/engine/issues/3415
	    // https://github.com/playcanvas/engine/issues/4514
	    // Example UA matches: Starting 'SM' and any combination of letters or numbers:
	    // Mozilla/5.0 (Linux, Android 12; SM-G970F Build/SP1A.210812.016; wv)
	    var samsungModelRegex = /SM-[a-zA-Z0-9]+/;
	    this.supportsGpuParticles = !(this.unmaskedVendor === 'ARM' && userAgent.match(samsungModelRegex)) && !this.unmaskedRenderer.match(maliRendererRegex);
	    ext = this.extTextureFilterAnisotropic;
	    this.maxAnisotropy = ext ? gl.getParameter(ext.MAX_TEXTURE_MAX_ANISOTROPY_EXT) : 1;
	    var antialiasSupported = this.isWebGL2 && !this.forceDisableMultisampling;
	    this.maxSamples = antialiasSupported ? gl.getParameter(gl.MAX_SAMPLES) : 1;

	    // some devices incorrectly report max samples larger than 4
	    this.maxSamples = Math.min(this.maxSamples, 4);

	    // we handle anti-aliasing internally by allocating multi-sampled backbuffer
	    this.samples = antialiasSupported && this.backBufferAntialias ? this.maxSamples : 1;

	    // Don't allow area lights on old android devices, they often fail to compile the shader, run it incorrectly or are very slow.
	    this.supportsAreaLights = this.isWebGL2 || !platform.android;

	    // supports texture fetch instruction
	    this.supportsTextureFetch = this.isWebGL2;

	    // Also do not allow them when we only have small number of texture units
	    if (this.maxTextures <= 8) {
	      this.supportsAreaLights = false;
	    }
	  }

	  /**
	   * Set the initial render state on the WebGL context.
	   *
	   * @ignore
	   */;
	  _proto.initializeRenderState = function initializeRenderState() {
	    _GraphicsDevice.prototype.initializeRenderState.call(this);
	    var gl = this.gl;

	    // Initialize render state to a known start state

	    // default blend state
	    gl.disable(gl.BLEND);
	    gl.blendFunc(gl.ONE, gl.ZERO);
	    gl.blendEquation(gl.FUNC_ADD);
	    gl.colorMask(true, true, true, true);
	    gl.blendColor(0, 0, 0, 0);
	    gl.enable(gl.CULL_FACE);
	    this.cullFace = gl.BACK;
	    gl.cullFace(gl.BACK);

	    // default depth state
	    gl.enable(gl.DEPTH_TEST);
	    gl.depthFunc(gl.LEQUAL);
	    gl.depthMask(true);
	    this.stencil = false;
	    gl.disable(gl.STENCIL_TEST);
	    this.stencilFuncFront = this.stencilFuncBack = FUNC_ALWAYS;
	    this.stencilRefFront = this.stencilRefBack = 0;
	    this.stencilMaskFront = this.stencilMaskBack = 0xFF;
	    gl.stencilFunc(gl.ALWAYS, 0, 0xFF);
	    this.stencilFailFront = this.stencilFailBack = STENCILOP_KEEP;
	    this.stencilZfailFront = this.stencilZfailBack = STENCILOP_KEEP;
	    this.stencilZpassFront = this.stencilZpassBack = STENCILOP_KEEP;
	    this.stencilWriteMaskFront = 0xFF;
	    this.stencilWriteMaskBack = 0xFF;
	    gl.stencilOp(gl.KEEP, gl.KEEP, gl.KEEP);
	    gl.stencilMask(0xFF);
	    this.alphaToCoverage = false;
	    this.raster = true;
	    if (this.isWebGL2) {
	      gl.disable(gl.SAMPLE_ALPHA_TO_COVERAGE);
	      gl.disable(gl.RASTERIZER_DISCARD);
	    }
	    this.depthBiasEnabled = false;
	    gl.disable(gl.POLYGON_OFFSET_FILL);
	    this.clearDepth = 1;
	    gl.clearDepth(1);
	    this.clearColor = new Color(0, 0, 0, 0);
	    gl.clearColor(0, 0, 0, 0);
	    this.clearStencil = 0;
	    gl.clearStencil(0);
	    if (this.isWebGL2) {
	      gl.hint(gl.FRAGMENT_SHADER_DERIVATIVE_HINT, gl.NICEST);
	    } else {
	      if (this.extStandardDerivatives) {
	        gl.hint(this.extStandardDerivatives.FRAGMENT_SHADER_DERIVATIVE_HINT_OES, gl.NICEST);
	      }
	    }
	    gl.enable(gl.SCISSOR_TEST);
	    gl.pixelStorei(gl.UNPACK_COLORSPACE_CONVERSION_WEBGL, gl.NONE);
	    this.unpackFlipY = false;
	    gl.pixelStorei(gl.UNPACK_FLIP_Y_WEBGL, false);
	    this.unpackPremultiplyAlpha = false;
	    gl.pixelStorei(gl.UNPACK_PREMULTIPLY_ALPHA_WEBGL, false);
	    gl.pixelStorei(gl.UNPACK_ALIGNMENT, 1);
	  };
	  _proto.initTextureUnits = function initTextureUnits(count) {
	    if (count === void 0) {
	      count = 16;
	    }
	    this.textureUnits = [];
	    for (var i = 0; i < count; i++) {
	      this.textureUnits.push([null, null, null]);
	    }
	  };
	  _proto.initializeContextCaches = function initializeContextCaches() {
	    _GraphicsDevice.prototype.initializeContextCaches.call(this);

	    // cache of VAOs
	    this._vaoMap = new Map();
	    this.boundVao = null;
	    this.activeFramebuffer = null;
	    this.feedback = null;
	    this.transformFeedbackBuffer = null;
	    this.textureUnit = 0;
	    this.initTextureUnits(this.maxCombinedTextures);
	  }

	  /**
	   * Called when the WebGL context was lost. It releases all context related resources.
	   *
	   * @ignore
	   */;
	  _proto.loseContext = function loseContext() {
	    var _this$gpuProfiler;
	    this.contextLost = true;

	    // force the backbuffer to be recreated on restore
	    this.backBufferSize.set(-1, -1);

	    // release shaders
	    for (var _iterator = _createForOfIteratorHelperLoose(this.shaders), _step; !(_step = _iterator()).done;) {
	      var shader = _step.value;
	      shader.loseContext();
	    }

	    // release textures
	    for (var _iterator2 = _createForOfIteratorHelperLoose(this.textures), _step2; !(_step2 = _iterator2()).done;) {
	      var texture = _step2.value;
	      texture.loseContext();
	    }

	    // release vertex and index buffers
	    for (var _iterator3 = _createForOfIteratorHelperLoose(this.buffers), _step3; !(_step3 = _iterator3()).done;) {
	      var buffer = _step3.value;
	      buffer.loseContext();
	    }

	    // Reset all render targets so they'll be recreated as required.
	    // TODO: a solution for the case where a render target contains something
	    // that was previously generated that needs to be re-rendered.
	    for (var _iterator4 = _createForOfIteratorHelperLoose(this.targets), _step4; !(_step4 = _iterator4()).done;) {
	      var target = _step4.value;
	      target.loseContext();
	    }
	    (_this$gpuProfiler = this.gpuProfiler) == null || _this$gpuProfiler.loseContext();
	  }

	  /**
	   * Called when the WebGL context is restored. It reinitializes all context related resources.
	   *
	   * @ignore
	   */;
	  _proto.restoreContext = function restoreContext() {
	    var _this$gpuProfiler2;
	    this.contextLost = false;
	    this.initializeExtensions();
	    this.initializeCapabilities();
	    this.initializeRenderState();
	    this.initializeContextCaches();

	    // Recompile all shaders
	    for (var _iterator5 = _createForOfIteratorHelperLoose(this.shaders), _step5; !(_step5 = _iterator5()).done;) {
	      var shader = _step5.value;
	      shader.restoreContext();
	    }

	    // Recreate buffer objects and reupload buffer data to the GPU
	    for (var _iterator6 = _createForOfIteratorHelperLoose(this.buffers), _step6; !(_step6 = _iterator6()).done;) {
	      var buffer = _step6.value;
	      buffer.unlock();
	    }
	    (_this$gpuProfiler2 = this.gpuProfiler) == null || _this$gpuProfiler2.restoreContext();
	  }

	  /**
	   * Set the active rectangle for rendering on the specified device.
	   *
	   * @param {number} x - The pixel space x-coordinate of the bottom left corner of the viewport.
	   * @param {number} y - The pixel space y-coordinate of the bottom left corner of the viewport.
	   * @param {number} w - The width of the viewport in pixels.
	   * @param {number} h - The height of the viewport in pixels.
	   */;
	  _proto.setViewport = function setViewport(x, y, w, h) {
	    if (this.vx !== x || this.vy !== y || this.vw !== w || this.vh !== h) {
	      this.gl.viewport(x, y, w, h);
	      this.vx = x;
	      this.vy = y;
	      this.vw = w;
	      this.vh = h;
	    }
	  }

	  /**
	   * Set the active scissor rectangle on the specified device.
	   *
	   * @param {number} x - The pixel space x-coordinate of the bottom left corner of the scissor rectangle.
	   * @param {number} y - The pixel space y-coordinate of the bottom left corner of the scissor rectangle.
	   * @param {number} w - The width of the scissor rectangle in pixels.
	   * @param {number} h - The height of the scissor rectangle in pixels.
	   */;
	  _proto.setScissor = function setScissor(x, y, w, h) {
	    if (this.sx !== x || this.sy !== y || this.sw !== w || this.sh !== h) {
	      this.gl.scissor(x, y, w, h);
	      this.sx = x;
	      this.sy = y;
	      this.sw = w;
	      this.sh = h;
	    }
	  }

	  /**
	   * Binds the specified framebuffer object.
	   *
	   * @param {WebGLFramebuffer | null} fb - The framebuffer to bind.
	   * @ignore
	   */;
	  _proto.setFramebuffer = function setFramebuffer(fb) {
	    if (this.activeFramebuffer !== fb) {
	      var gl = this.gl;
	      gl.bindFramebuffer(gl.FRAMEBUFFER, fb);
	      this.activeFramebuffer = fb;
	    }
	  }

	  /**
	   * Copies source render target into destination render target. Mostly used by post-effects.
	   *
	   * @param {RenderTarget} [source] - The source render target. Defaults to frame buffer.
	   * @param {RenderTarget} [dest] - The destination render target. Defaults to frame buffer.
	   * @param {boolean} [color] - If true will copy the color buffer. Defaults to false.
	   * @param {boolean} [depth] - If true will copy the depth buffer. Defaults to false.
	   * @returns {boolean} True if the copy was successful, false otherwise.
	   */;
	  _proto.copyRenderTarget = function copyRenderTarget(source, dest, color, depth) {
	    var gl = this.gl;

	    // if copying from the backbuffer
	    if (source === this.backBuffer) {
	      source = null;
	    }
	    if (!this.isWebGL2 && depth) {
	      Debug.error("Depth is not copyable on WebGL 1.0");
	      return false;
	    }
	    if (color) {
	      if (!dest) {
	        // copying to backbuffer
	        if (!source._colorBuffer) {
	          Debug.error("Can't copy empty color buffer to backbuffer");
	          return false;
	        }
	      } else if (source) {
	        // copying to render target
	        if (!source._colorBuffer || !dest._colorBuffer) {
	          Debug.error("Can't copy color buffer, because one of the render targets doesn't have it");
	          return false;
	        }
	        if (source._colorBuffer._format !== dest._colorBuffer._format) {
	          Debug.error("Can't copy render targets of different color formats");
	          return false;
	        }
	      }
	    }
	    if (depth && source) {
	      if (!source._depth) {
	        // when depth is automatic, we cannot test the buffer nor its format
	        if (!source._depthBuffer || !dest._depthBuffer) {
	          Debug.error("Can't copy depth buffer, because one of the render targets doesn't have it");
	          return false;
	        }
	        if (source._depthBuffer._format !== dest._depthBuffer._format) {
	          Debug.error("Can't copy render targets of different depth formats");
	          return false;
	        }
	      }
	    }
	    DebugGraphics.pushGpuMarker(this, 'COPY-RT');
	    if (this.isWebGL2 && dest) {
	      var _this$backBuffer;
	      var prevRt = this.renderTarget;
	      this.renderTarget = dest;
	      this.updateBegin();

	      // copy from single sampled framebuffer
	      var src = source ? source.impl._glFrameBuffer : (_this$backBuffer = this.backBuffer) == null ? void 0 : _this$backBuffer.impl._glFrameBuffer;
	      var dst = dest.impl._glFrameBuffer;
	      Debug.assert(src !== dst, 'Source and destination framebuffers must be different when blitting.');
	      gl.bindFramebuffer(gl.READ_FRAMEBUFFER, src);
	      gl.bindFramebuffer(gl.DRAW_FRAMEBUFFER, dst);
	      var w = source ? source.width : dest.width;
	      var h = source ? source.height : dest.height;
	      gl.blitFramebuffer(0, 0, w, h, 0, 0, w, h, (color ? gl.COLOR_BUFFER_BIT : 0) | (depth ? gl.DEPTH_BUFFER_BIT : 0), gl.NEAREST);

	      // TODO: not sure we need to restore the prev target, as this only should run in-between render passes
	      this.renderTarget = prevRt;
	      gl.bindFramebuffer(gl.FRAMEBUFFER, prevRt ? prevRt.impl._glFrameBuffer : null);
	    } else {
	      var shader = this.getCopyShader();
	      this.constantTexSource.setValue(source._colorBuffer);
	      quadWithShader(this, dest, shader);
	    }
	    DebugGraphics.popGpuMarker(this);
	    return true;
	  }

	  /**
	   * Get copy shader for efficient rendering of fullscreen-quad with texture.
	   *
	   * @returns {Shader} The copy shader (based on `fullscreenQuadVS` and `outputTex2DPS` in
	   * `shaderChunks`).
	   * @ignore
	   */;
	  _proto.getCopyShader = function getCopyShader() {
	    if (!this._copyShader) {
	      this._copyShader = new Shader(this, ShaderUtils.createDefinition(this, {
	        name: 'outputTex2D',
	        vertexCode: _fullScreenQuadVS,
	        fragmentCode: _outputTexture2D
	      }));
	    }
	    return this._copyShader;
	  };
	  _proto.frameStart = function frameStart() {
	    _GraphicsDevice.prototype.frameStart.call(this);
	    this.updateBackbuffer();
	    this.gpuProfiler.frameStart();
	  };
	  _proto.frameEnd = function frameEnd() {
	    _GraphicsDevice.prototype.frameEnd.call(this);
	    this.gpuProfiler.frameEnd();
	    this.gpuProfiler.request();
	  }

	  /**
	   * Start a render pass.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('../render-pass.js').RenderPass} renderPass - The render pass to start.
	   * @ignore
	   */;
	  _proto.startRenderPass = function startRenderPass(renderPass) {
	    var _renderPass$renderTar,
	      _this2 = this;
	    DebugGraphics.pushGpuMarker(this, "Pass:" + renderPass.name);
	    DebugGraphics.pushGpuMarker(this, "START-PASS");

	    // set up render target
	    var rt = (_renderPass$renderTar = renderPass.renderTarget) != null ? _renderPass$renderTar : this.backBuffer;
	    this.renderTarget = rt;
	    Debug.assert(rt);
	    this.updateBegin();

	    // the pass always start using full size of the target
	    var width = rt.width,
	      height = rt.height;
	    this.setViewport(0, 0, width, height);
	    this.setScissor(0, 0, width, height);

	    // clear the render target
	    var colorOps = renderPass.colorOps;
	    var depthStencilOps = renderPass.depthStencilOps;
	    if (colorOps != null && colorOps.clear || depthStencilOps.clearDepth || depthStencilOps.clearStencil) {
	      var clearFlags = 0;
	      var clearOptions = {};
	      if (colorOps != null && colorOps.clear) {
	        clearFlags |= CLEARFLAG_COLOR;
	        clearOptions.color = [colorOps.clearValue.r, colorOps.clearValue.g, colorOps.clearValue.b, colorOps.clearValue.a];
	      }
	      if (depthStencilOps.clearDepth) {
	        clearFlags |= CLEARFLAG_DEPTH;
	        clearOptions.depth = depthStencilOps.clearDepthValue;
	      }
	      if (depthStencilOps.clearStencil) {
	        clearFlags |= CLEARFLAG_STENCIL;
	        clearOptions.stencil = depthStencilOps.clearStencilValue;
	      }

	      // clear it
	      clearOptions.flags = clearFlags;
	      this.clear(clearOptions);
	    }
	    Debug.call(function () {
	      if (_this2.insideRenderPass) {
	        Debug.errorOnce('RenderPass cannot be started while inside another render pass.');
	      }
	    });
	    this.insideRenderPass = true;
	    DebugGraphics.popGpuMarker(this);
	  }

	  /**
	   * End a render pass.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('../render-pass.js').RenderPass} renderPass - The render pass to end.
	   * @ignore
	   */;
	  _proto.endRenderPass = function endRenderPass(renderPass) {
	    DebugGraphics.pushGpuMarker(this, "END-PASS");
	    this.unbindVertexArray();
	    var target = this.renderTarget;
	    var colorBufferCount = renderPass.colorArrayOps.length;
	    if (target) {
	      var _renderPass$colorOps;
	      // invalidate buffers to stop them being written to on tiled architectures
	      if (this.isWebGL2) {
	        invalidateAttachments.length = 0;
	        var gl = this.gl;

	        // color buffers
	        for (var i = 0; i < colorBufferCount; i++) {
	          var colorOps = renderPass.colorArrayOps[i];

	          // invalidate color only if we don't need to resolve it
	          if (!(colorOps.store || colorOps.resolve)) {
	            invalidateAttachments.push(gl.COLOR_ATTACHMENT0 + i);
	          }
	        }

	        // we cannot invalidate depth/stencil buffers of the backbuffer
	        if (target !== this.backBuffer) {
	          if (!renderPass.depthStencilOps.storeDepth) {
	            invalidateAttachments.push(gl.DEPTH_ATTACHMENT);
	          }
	          if (!renderPass.depthStencilOps.storeStencil) {
	            invalidateAttachments.push(gl.STENCIL_ATTACHMENT);
	          }
	        }
	        if (invalidateAttachments.length > 0) {
	          // invalidate the whole buffer
	          // TODO: we could handle viewport invalidation as well
	          if (renderPass.fullSizeClearRect) {
	            gl.invalidateFramebuffer(gl.DRAW_FRAMEBUFFER, invalidateAttachments);
	          }
	        }
	      }

	      // resolve the color buffer (this resolves all MRT color buffers at once)
	      if ((_renderPass$colorOps = renderPass.colorOps) != null && _renderPass$colorOps.resolve) {
	        if (this.isWebGL2 && renderPass.samples > 1 && target.autoResolve) {
	          target.resolve(true, false);
	        }
	      }

	      // generate mipmaps
	      for (var _i = 0; _i < colorBufferCount; _i++) {
	        var _colorOps = renderPass.colorArrayOps[_i];
	        if (_colorOps.mipmaps) {
	          var colorBuffer = target._colorBuffers[_i];
	          if (colorBuffer && colorBuffer.impl._glTexture && colorBuffer.mipmaps && (colorBuffer.pot || this.isWebGL2)) {
	            DebugGraphics.pushGpuMarker(this, "MIPS" + _i);
	            this.activeTexture(this.maxCombinedTextures - 1);
	            this.bindTexture(colorBuffer);
	            this.gl.generateMipmap(colorBuffer.impl._glTarget);
	            DebugGraphics.popGpuMarker(this);
	          }
	        }
	      }
	    }
	    this.insideRenderPass = false;
	    DebugGraphics.popGpuMarker(this);
	    DebugGraphics.popGpuMarker(this); // pop the pass-start marker
	  };
	  /**
	   * Marks the beginning of a block of rendering. Internally, this function binds the render
	   * target currently set on the device. This function should be matched with a call to
	   * {@link GraphicsDevice#updateEnd}. Calls to {@link GraphicsDevice#updateBegin} and
	   * {@link GraphicsDevice#updateEnd} must not be nested.
	   *
	   * @ignore
	   */
	  _proto.updateBegin = function updateBegin() {
	    var _this$renderTarget;
	    DebugGraphics.pushGpuMarker(this, 'UPDATE-BEGIN');
	    this.boundVao = null;

	    // clear texture units once a frame on desktop safari
	    if (this._tempEnableSafariTextureUnitWorkaround) {
	      for (var unit = 0; unit < this.textureUnits.length; ++unit) {
	        for (var slot = 0; slot < 3; ++slot) {
	          this.textureUnits[unit][slot] = null;
	        }
	      }
	    }

	    // Set the render target
	    var target = (_this$renderTarget = this.renderTarget) != null ? _this$renderTarget : this.backBuffer;
	    Debug.assert(target);

	    // Initialize the framebuffer
	    var targetImpl = target.impl;
	    if (!targetImpl.initialized) {
	      this.initRenderTarget(target);
	    }

	    // Bind the framebuffer
	    this.setFramebuffer(targetImpl._glFrameBuffer);
	    DebugGraphics.popGpuMarker(this);
	  }

	  /**
	   * Marks the end of a block of rendering. This function should be called after a matching call
	   * to {@link GraphicsDevice#updateBegin}. Calls to {@link GraphicsDevice#updateBegin} and
	   * {@link GraphicsDevice#updateEnd} must not be nested.
	   *
	   * @ignore
	   */;
	  _proto.updateEnd = function updateEnd() {
	    DebugGraphics.pushGpuMarker(this, "UPDATE-END");
	    this.unbindVertexArray();

	    // Unset the render target
	    var target = this.renderTarget;
	    if (target && target !== this.backBuffer) {
	      // Resolve MSAA if needed
	      if (this.isWebGL2 && target._samples > 1 && target.autoResolve) {
	        target.resolve();
	      }

	      // If the active render target is auto-mipmapped, generate its mip chain
	      var colorBuffer = target._colorBuffer;
	      if (colorBuffer && colorBuffer.impl._glTexture && colorBuffer.mipmaps && (colorBuffer.pot || this.isWebGL2)) {
	        // FIXME: if colorBuffer is a cubemap currently we're re-generating mipmaps after
	        // updating each face!
	        this.activeTexture(this.maxCombinedTextures - 1);
	        this.bindTexture(colorBuffer);
	        this.gl.generateMipmap(colorBuffer.impl._glTarget);
	      }
	    }
	    DebugGraphics.popGpuMarker(this);
	  }

	  /**
	   * Updates a texture's vertical flip.
	   *
	   * @param {boolean} flipY - True to flip the texture vertically.
	   * @ignore
	   */;
	  _proto.setUnpackFlipY = function setUnpackFlipY(flipY) {
	    if (this.unpackFlipY !== flipY) {
	      this.unpackFlipY = flipY;

	      // Note: the WebGL spec states that UNPACK_FLIP_Y_WEBGL only affects
	      // texImage2D and texSubImage2D, not compressedTexImage2D
	      var gl = this.gl;
	      gl.pixelStorei(gl.UNPACK_FLIP_Y_WEBGL, flipY);
	    }
	  }

	  /**
	   * Updates a texture to have its RGB channels premultiplied by its alpha channel or not.
	   *
	   * @param {boolean} premultiplyAlpha - True to premultiply the alpha channel against the RGB
	   * channels.
	   * @ignore
	   */;
	  _proto.setUnpackPremultiplyAlpha = function setUnpackPremultiplyAlpha(premultiplyAlpha) {
	    if (this.unpackPremultiplyAlpha !== premultiplyAlpha) {
	      this.unpackPremultiplyAlpha = premultiplyAlpha;

	      // Note: the WebGL spec states that UNPACK_PREMULTIPLY_ALPHA_WEBGL only affects
	      // texImage2D and texSubImage2D, not compressedTexImage2D
	      var gl = this.gl;
	      gl.pixelStorei(gl.UNPACK_PREMULTIPLY_ALPHA_WEBGL, premultiplyAlpha);
	    }
	  }

	  /**
	   * Activate the specified texture unit.
	   *
	   * @param {number} textureUnit - The texture unit to activate.
	   * @ignore
	   */;
	  _proto.activeTexture = function activeTexture(textureUnit) {
	    if (this.textureUnit !== textureUnit) {
	      this.gl.activeTexture(this.gl.TEXTURE0 + textureUnit);
	      this.textureUnit = textureUnit;
	    }
	  }

	  /**
	   * If the texture is not already bound on the currently active texture unit, bind it.
	   *
	   * @param {Texture} texture - The texture to bind.
	   * @ignore
	   */;
	  _proto.bindTexture = function bindTexture(texture) {
	    var impl = texture.impl;
	    var textureTarget = impl._glTarget;
	    var textureObject = impl._glTexture;
	    var textureUnit = this.textureUnit;
	    var slot = this.targetToSlot[textureTarget];
	    if (this.textureUnits[textureUnit][slot] !== textureObject) {
	      this.gl.bindTexture(textureTarget, textureObject);
	      this.textureUnits[textureUnit][slot] = textureObject;
	    }
	  }

	  /**
	   * If the texture is not bound on the specified texture unit, active the texture unit and bind
	   * the texture to it.
	   *
	   * @param {Texture} texture - The texture to bind.
	   * @param {number} textureUnit - The texture unit to activate and bind the texture to.
	   * @ignore
	   */;
	  _proto.bindTextureOnUnit = function bindTextureOnUnit(texture, textureUnit) {
	    var impl = texture.impl;
	    var textureTarget = impl._glTarget;
	    var textureObject = impl._glTexture;
	    var slot = this.targetToSlot[textureTarget];
	    if (this.textureUnits[textureUnit][slot] !== textureObject) {
	      this.activeTexture(textureUnit);
	      this.gl.bindTexture(textureTarget, textureObject);
	      this.textureUnits[textureUnit][slot] = textureObject;
	    }
	  }

	  /**
	   * Update the texture parameters for a given texture if they have changed.
	   *
	   * @param {Texture} texture - The texture to update.
	   * @ignore
	   */;
	  _proto.setTextureParameters = function setTextureParameters(texture) {
	    var gl = this.gl;
	    var flags = texture.impl.dirtyParameterFlags;
	    var target = texture.impl._glTarget;
	    if (flags & 1) {
	      var filter = texture._minFilter;
	      if (!texture.pot && !this.isWebGL2 || !texture._mipmaps || texture._compressed && texture._levels.length === 1) {
	        if (filter === FILTER_NEAREST_MIPMAP_NEAREST || filter === FILTER_NEAREST_MIPMAP_LINEAR) {
	          filter = FILTER_NEAREST;
	        } else if (filter === FILTER_LINEAR_MIPMAP_NEAREST || filter === FILTER_LINEAR_MIPMAP_LINEAR) {
	          filter = FILTER_LINEAR;
	        }
	      }
	      gl.texParameteri(target, gl.TEXTURE_MIN_FILTER, this.glFilter[filter]);
	    }
	    if (flags & 2) {
	      gl.texParameteri(target, gl.TEXTURE_MAG_FILTER, this.glFilter[texture._magFilter]);
	    }
	    if (flags & 4) {
	      if (this.isWebGL2) {
	        gl.texParameteri(target, gl.TEXTURE_WRAP_S, this.glAddress[texture._addressU]);
	      } else {
	        // WebGL1 doesn't support all addressing modes with NPOT textures
	        gl.texParameteri(target, gl.TEXTURE_WRAP_S, this.glAddress[texture.pot ? texture._addressU : ADDRESS_CLAMP_TO_EDGE]);
	      }
	    }
	    if (flags & 8) {
	      if (this.isWebGL2) {
	        gl.texParameteri(target, gl.TEXTURE_WRAP_T, this.glAddress[texture._addressV]);
	      } else {
	        // WebGL1 doesn't support all addressing modes with NPOT textures
	        gl.texParameteri(target, gl.TEXTURE_WRAP_T, this.glAddress[texture.pot ? texture._addressV : ADDRESS_CLAMP_TO_EDGE]);
	      }
	    }
	    if (flags & 16) {
	      if (this.isWebGL2) {
	        gl.texParameteri(target, gl.TEXTURE_WRAP_R, this.glAddress[texture._addressW]);
	      }
	    }
	    if (flags & 32) {
	      if (this.isWebGL2) {
	        gl.texParameteri(target, gl.TEXTURE_COMPARE_MODE, texture._compareOnRead ? gl.COMPARE_REF_TO_TEXTURE : gl.NONE);
	      }
	    }
	    if (flags & 64) {
	      if (this.isWebGL2) {
	        gl.texParameteri(target, gl.TEXTURE_COMPARE_FUNC, this.glComparison[texture._compareFunc]);
	      }
	    }
	    if (flags & 128) {
	      var ext = this.extTextureFilterAnisotropic;
	      if (ext) {
	        gl.texParameterf(target, ext.TEXTURE_MAX_ANISOTROPY_EXT, math.clamp(Math.round(texture._anisotropy), 1, this.maxAnisotropy));
	      }
	    }
	  }

	  /**
	   * Sets the specified texture on the specified texture unit.
	   *
	   * @param {Texture} texture - The texture to set.
	   * @param {number} textureUnit - The texture unit to set the texture on.
	   * @ignore
	   */;
	  _proto.setTexture = function setTexture(texture, textureUnit) {
	    var impl = texture.impl;
	    if (!impl._glTexture) impl.initialize(this, texture);
	    if (impl.dirtyParameterFlags > 0 || texture._needsUpload || texture._needsMipmapsUpload) {
	      // Ensure the specified texture unit is active
	      this.activeTexture(textureUnit);

	      // Ensure the texture is bound on correct target of the specified texture unit
	      this.bindTexture(texture);
	      if (impl.dirtyParameterFlags) {
	        this.setTextureParameters(texture);
	        impl.dirtyParameterFlags = 0;
	      }
	      if (texture._needsUpload || texture._needsMipmapsUpload) {
	        impl.upload(this, texture);
	        texture._needsUpload = false;
	        texture._needsMipmapsUpload = false;
	      }
	    } else {
	      // Ensure the texture is currently bound to the correct target on the specified texture unit.
	      // If the texture is already bound to the correct target on the specified unit, there's no need
	      // to actually make the specified texture unit active because the texture itself does not need
	      // to be updated.
	      this.bindTextureOnUnit(texture, textureUnit);
	    }
	  }

	  // function creates VertexArrayObject from list of vertex buffers
	  ;
	  _proto.createVertexArray = function createVertexArray(vertexBuffers) {
	    var key, vao;

	    // only use cache when more than 1 vertex buffer, otherwise it's unique
	    var useCache = vertexBuffers.length > 1;
	    if (useCache) {
	      // generate unique key for the vertex buffers
	      key = "";
	      for (var i = 0; i < vertexBuffers.length; i++) {
	        var vertexBuffer = vertexBuffers[i];
	        key += vertexBuffer.id + vertexBuffer.format.renderingHash;
	      }

	      // try to get VAO from cache
	      vao = this._vaoMap.get(key);
	    }

	    // need to create new vao
	    if (!vao) {
	      // create VA object
	      var gl = this.gl;
	      vao = gl.createVertexArray();
	      gl.bindVertexArray(vao);

	      // don't capture index buffer in VAO
	      gl.bindBuffer(gl.ELEMENT_ARRAY_BUFFER, null);
	      var locZero = false;
	      for (var _i2 = 0; _i2 < vertexBuffers.length; _i2++) {
	        // bind buffer
	        var _vertexBuffer = vertexBuffers[_i2];
	        gl.bindBuffer(gl.ARRAY_BUFFER, _vertexBuffer.impl.bufferId);

	        // for each attribute
	        var elements = _vertexBuffer.format.elements;
	        for (var j = 0; j < elements.length; j++) {
	          var e = elements[j];
	          var loc = semanticToLocation[e.name];
	          if (loc === 0) {
	            locZero = true;
	          }
	          if (e.asInt) {
	            gl.vertexAttribIPointer(loc, e.numComponents, this.glType[e.dataType], e.stride, e.offset);
	          } else {
	            gl.vertexAttribPointer(loc, e.numComponents, this.glType[e.dataType], e.normalize, e.stride, e.offset);
	          }
	          gl.enableVertexAttribArray(loc);
	          if (_vertexBuffer.format.instancing) {
	            gl.vertexAttribDivisor(loc, 1);
	          }
	        }
	      }

	      // end of VA object
	      gl.bindVertexArray(null);

	      // unbind any array buffer
	      gl.bindBuffer(gl.ARRAY_BUFFER, null);

	      // add it to cache
	      if (useCache) {
	        this._vaoMap.set(key, vao);
	      }
	      if (!locZero) {
	        Debug.warn("No vertex attribute is mapped to location 0, which might cause compatibility issues on Safari on MacOS - please use attribute SEMANTIC_POSITION or SEMANTIC_ATTR15");
	      }
	    }
	    return vao;
	  };
	  _proto.unbindVertexArray = function unbindVertexArray() {
	    // unbind VAO from device to protect it from being changed
	    if (this.boundVao) {
	      this.boundVao = null;
	      this.gl.bindVertexArray(null);
	    }
	  };
	  _proto.setBuffers = function setBuffers() {
	    var gl = this.gl;
	    var vao;

	    // create VAO for specified vertex buffers
	    if (this.vertexBuffers.length === 1) {
	      // single VB keeps its VAO
	      var vertexBuffer = this.vertexBuffers[0];
	      Debug.assert(vertexBuffer.device === this, "The VertexBuffer was not created using current GraphicsDevice");
	      if (!vertexBuffer.impl.vao) {
	        vertexBuffer.impl.vao = this.createVertexArray(this.vertexBuffers);
	      }
	      vao = vertexBuffer.impl.vao;
	    } else {
	      // obtain temporary VAO for multiple vertex buffers
	      vao = this.createVertexArray(this.vertexBuffers);
	    }

	    // set active VAO
	    if (this.boundVao !== vao) {
	      this.boundVao = vao;
	      gl.bindVertexArray(vao);
	    }

	    // empty array of vertex buffers
	    this.vertexBuffers.length = 0;

	    // Set the active index buffer object
	    // Note: we don't cache this state and set it only when it changes, as VAO captures last bind buffer in it
	    // and so we don't know what VAO sets it to.
	    var bufferId = this.indexBuffer ? this.indexBuffer.impl.bufferId : null;
	    gl.bindBuffer(gl.ELEMENT_ARRAY_BUFFER, bufferId);
	  }

	  /**
	   * Submits a graphical primitive to the hardware for immediate rendering.
	   *
	   * @param {object} primitive - Primitive object describing how to submit current vertex/index
	   * buffers.
	   * @param {number} primitive.type - The type of primitive to render. Can be:
	   *
	   * - {@link PRIMITIVE_POINTS}
	   * - {@link PRIMITIVE_LINES}
	   * - {@link PRIMITIVE_LINELOOP}
	   * - {@link PRIMITIVE_LINESTRIP}
	   * - {@link PRIMITIVE_TRIANGLES}
	   * - {@link PRIMITIVE_TRISTRIP}
	   * - {@link PRIMITIVE_TRIFAN}
	   *
	   * @param {number} primitive.base - The offset of the first index or vertex to dispatch in the
	   * draw call.
	   * @param {number} primitive.count - The number of indices or vertices to dispatch in the draw
	   * call.
	   * @param {boolean} [primitive.indexed] - True to interpret the primitive as indexed, thereby
	   * using the currently set index buffer and false otherwise.
	   * @param {number} [numInstances] - The number of instances to render when using
	   * ANGLE_instanced_arrays. Defaults to 1.
	   * @param {boolean} [keepBuffers] - Optionally keep the current set of vertex / index buffers /
	   * VAO. This is used when rendering of multiple views, for example under WebXR.
	   * @example
	   * // Render a single, unindexed triangle
	   * device.draw({
	   *     type: pc.PRIMITIVE_TRIANGLES,
	   *     base: 0,
	   *     count: 3,
	   *     indexed: false
	   * });
	   */;
	  _proto.draw = function draw(primitive, numInstances, keepBuffers) {
	    var gl = this.gl;
	    this.activateShader(this);
	    if (!this.shaderValid) return;
	    var sampler, samplerValue, texture, numTextures; // Samplers
	    var uniform, scopeId, uniformVersion, programVersion; // Uniforms
	    var shader = this.shader;
	    if (!shader) return;
	    var samplers = shader.impl.samplers;
	    var uniforms = shader.impl.uniforms;

	    // vertex buffers
	    if (!keepBuffers) {
	      this.setBuffers();
	    }

	    // Commit the shader program variables
	    var textureUnit = 0;
	    for (var i = 0, len = samplers.length; i < len; i++) {
	      sampler = samplers[i];
	      samplerValue = sampler.scopeId.value;
	      if (!samplerValue) {
	        var samplerName = sampler.scopeId.name;
	        if (samplerName === 'uSceneDepthMap' || samplerName === 'uDepthMap') {
	          Debug.warnOnce("A sampler " + samplerName + " is used by the shader but a scene depth texture is not available. Use CameraComponent.requestSceneDepthMap / enable Depth Grabpass on the Camera Component to enable it.");
	        }
	        if (samplerName === 'uSceneColorMap' || samplerName === 'texture_grabPass') {
	          Debug.warnOnce("A sampler " + samplerName + " is used by the shader but a scene color texture is not available. Use CameraComponent.requestSceneColorMap / enable Color Grabpass on the Camera Component to enable it.");
	        }
	        Debug.errorOnce("Shader [" + shader.label + "] requires texture sampler [" + samplerName + "] which has not been set, while rendering [" + DebugGraphics.toString() + "]");

	        // skip this draw call to avoid incorrect rendering / webgl errors
	        return;
	      }
	      if (samplerValue instanceof Texture) {
	        texture = samplerValue;
	        this.setTexture(texture, textureUnit);
	        if (this.renderTarget) {
	          // Set breakpoint here to debug "Source and destination textures of the draw are the same" errors
	          if (this.renderTarget._samples < 2) {
	            if (this.renderTarget.colorBuffer && this.renderTarget.colorBuffer === texture) {
	              Debug.error("Trying to bind current color buffer as a texture", {
	                renderTarget: this.renderTarget,
	                texture: texture
	              });
	            } else if (this.renderTarget.depthBuffer && this.renderTarget.depthBuffer === texture) {
	              Debug.error("Trying to bind current depth buffer as a texture", {
	                texture: texture
	              });
	            }
	          }
	        }
	        if (sampler.slot !== textureUnit) {
	          gl.uniform1i(sampler.locationId, textureUnit);
	          sampler.slot = textureUnit;
	        }
	        textureUnit++;
	      } else {
	        // Array
	        sampler.array.length = 0;
	        numTextures = samplerValue.length;
	        for (var j = 0; j < numTextures; j++) {
	          texture = samplerValue[j];
	          this.setTexture(texture, textureUnit);
	          sampler.array[j] = textureUnit;
	          textureUnit++;
	        }
	        gl.uniform1iv(sampler.locationId, sampler.array);
	      }
	    }

	    // Commit any updated uniforms
	    for (var _i3 = 0, _len = uniforms.length; _i3 < _len; _i3++) {
	      uniform = uniforms[_i3];
	      scopeId = uniform.scopeId;
	      uniformVersion = uniform.version;
	      programVersion = scopeId.versionObject.version;

	      // Check the value is valid
	      if (uniformVersion.globalId !== programVersion.globalId || uniformVersion.revision !== programVersion.revision) {
	        uniformVersion.globalId = programVersion.globalId;
	        uniformVersion.revision = programVersion.revision;

	        // Call the function to commit the uniform value
	        if (scopeId.value !== null) {
	          this.commitFunction[uniform.dataType](uniform, scopeId.value);
	        }
	      }
	    }
	    if (this.isWebGL2 && this.transformFeedbackBuffer) {
	      // Enable TF, start writing to out buffer
	      gl.bindBufferBase(gl.TRANSFORM_FEEDBACK_BUFFER, 0, this.transformFeedbackBuffer.impl.bufferId);
	      gl.beginTransformFeedback(gl.POINTS);
	    }
	    var mode = this.glPrimitive[primitive.type];
	    var count = primitive.count;
	    if (primitive.indexed) {
	      var indexBuffer = this.indexBuffer;
	      Debug.assert(indexBuffer.device === this, "The IndexBuffer was not created using current GraphicsDevice");
	      var format = indexBuffer.impl.glFormat;
	      var offset = primitive.base * indexBuffer.bytesPerIndex;
	      if (numInstances > 0) {
	        gl.drawElementsInstanced(mode, count, format, offset, numInstances);
	      } else {
	        gl.drawElements(mode, count, format, offset);
	      }
	    } else {
	      var first = primitive.base;
	      if (numInstances > 0) {
	        gl.drawArraysInstanced(mode, first, count, numInstances);
	      } else {
	        gl.drawArrays(mode, first, count);
	      }
	    }
	    if (this.isWebGL2 && this.transformFeedbackBuffer) {
	      // disable TF
	      gl.endTransformFeedback();
	      gl.bindBufferBase(gl.TRANSFORM_FEEDBACK_BUFFER, 0, null);
	    }
	    this._drawCallsPerFrame++;
	    this._primsPerFrame[primitive.type] += primitive.count * (numInstances > 1 ? numInstances : 1);
	  }

	  /**
	   * Clears the frame buffer of the currently set render target.
	   *
	   * @param {object} [options] - Optional options object that controls the behavior of the clear
	   * operation defined as follows:
	   * @param {number[]} [options.color] - The color to clear the color buffer to in the range 0 to
	   * 1 for each component.
	   * @param {number} [options.depth] - The depth value to clear the depth buffer to in the
	   * range 0 to 1. Defaults to 1.
	   * @param {number} [options.flags] - The buffers to clear (the types being color, depth and
	   * stencil). Can be any bitwise combination of:
	   *
	   * - {@link CLEARFLAG_COLOR}
	   * - {@link CLEARFLAG_DEPTH}
	   * - {@link CLEARFLAG_STENCIL}
	   *
	   * @param {number} [options.stencil] - The stencil value to clear the stencil buffer to.
	   * Defaults to 0.
	   * @example
	   * // Clear color buffer to black and depth buffer to 1
	   * device.clear();
	   *
	   * // Clear just the color buffer to red
	   * device.clear({
	   *     color: [1, 0, 0, 1],
	   *     flags: pc.CLEARFLAG_COLOR
	   * });
	   *
	   * // Clear color buffer to yellow and depth to 1.0
	   * device.clear({
	   *     color: [1, 1, 0, 1],
	   *     depth: 1,
	   *     flags: pc.CLEARFLAG_COLOR | pc.CLEARFLAG_DEPTH
	   * });
	   */;
	  _proto.clear = function clear(options) {
	    var _options$flags;
	    var defaultOptions = this.defaultClearOptions;
	    options = options || defaultOptions;
	    var flags = (_options$flags = options.flags) != null ? _options$flags : defaultOptions.flags;
	    if (flags !== 0) {
	      var gl = this.gl;

	      // Set the clear color
	      if (flags & CLEARFLAG_COLOR) {
	        var _options$color;
	        var color = (_options$color = options.color) != null ? _options$color : defaultOptions.color;
	        var r = color[0];
	        var g = color[1];
	        var b = color[2];
	        var a = color[3];
	        var c = this.clearColor;
	        if (r !== c.r || g !== c.g || b !== c.b || a !== c.a) {
	          this.gl.clearColor(r, g, b, a);
	          this.clearColor.set(r, g, b, a);
	        }
	        this.setBlendState(BlendState.NOBLEND);
	      }
	      if (flags & CLEARFLAG_DEPTH) {
	        var _options$depth;
	        // Set the clear depth
	        var depth = (_options$depth = options.depth) != null ? _options$depth : defaultOptions.depth;
	        if (depth !== this.clearDepth) {
	          this.gl.clearDepth(depth);
	          this.clearDepth = depth;
	        }
	        this.setDepthState(DepthState.WRITEDEPTH);
	      }
	      if (flags & CLEARFLAG_STENCIL) {
	        var _options$stencil;
	        // Set the clear stencil
	        var stencil = (_options$stencil = options.stencil) != null ? _options$stencil : defaultOptions.stencil;
	        if (stencil !== this.clearStencil) {
	          this.gl.clearStencil(stencil);
	          this.clearStencil = stencil;
	        }
	      }

	      // Clear the frame buffer
	      gl.clear(this.glClearFlag[flags]);
	    }
	  };
	  _proto.submit = function submit() {
	    this.gl.flush();
	  }

	  /**
	   * Reads a block of pixels from a specified rectangle of the current color framebuffer into an
	   * ArrayBufferView object.
	   *
	   * @param {number} x - The x-coordinate of the rectangle's lower-left corner.
	   * @param {number} y - The y-coordinate of the rectangle's lower-left corner.
	   * @param {number} w - The width of the rectangle, in pixels.
	   * @param {number} h - The height of the rectangle, in pixels.
	   * @param {ArrayBufferView} pixels - The ArrayBufferView object that holds the returned pixel
	   * data.
	   * @ignore
	   */;
	  _proto.readPixels = function readPixels(x, y, w, h, pixels) {
	    var gl = this.gl;
	    gl.readPixels(x, y, w, h, gl.RGBA, gl.UNSIGNED_BYTE, pixels);
	  }

	  /**
	   * Asynchronously reads a block of pixels from a specified rectangle of the current color framebuffer
	   * into an ArrayBufferView object.
	   *
	   * @param {number} x - The x-coordinate of the rectangle's lower-left corner.
	   * @param {number} y - The y-coordinate of the rectangle's lower-left corner.
	   * @param {number} w - The width of the rectangle, in pixels.
	   * @param {number} h - The height of the rectangle, in pixels.
	   * @param {ArrayBufferView} pixels - The ArrayBufferView object that holds the returned pixel
	   * data.
	   * @ignore
	   */;
	  _proto.readPixelsAsync =
	  /*#__PURE__*/
	  function () {
	    var _readPixelsAsync = _asyncToGenerator( /*#__PURE__*/_regeneratorRuntime().mark(function _callee(x, y, w, h, pixels) {
	      var _this3 = this,
	        _this$renderTarget$co,
	        _impl$_glFormat,
	        _impl$_glPixelType;
	      var gl, clientWaitAsync, impl, format, pixelType, buf;
	      return _regeneratorRuntime().wrap(function _callee$(_context) {
	        while (1) switch (_context.prev = _context.next) {
	          case 0:
	            gl = this.gl;
	            if (this.isWebGL2) {
	              _context.next = 4;
	              break;
	            }
	            // async fences aren't supported on webgl1
	            this.readPixels(x, y, w, h, pixels);
	            return _context.abrupt("return");
	          case 4:
	            clientWaitAsync = function clientWaitAsync(flags, interval_ms) {
	              var sync = gl.fenceSync(gl.SYNC_GPU_COMMANDS_COMPLETE, 0);
	              _this3.submit();
	              return new Promise(function (resolve, reject) {
	                function test() {
	                  var res = gl.clientWaitSync(sync, flags, 0);
	                  if (res === gl.WAIT_FAILED) {
	                    gl.deleteSync(sync);
	                    reject(new Error('webgl clientWaitSync sync failed'));
	                  } else if (res === gl.TIMEOUT_EXPIRED) {
	                    setTimeout(test, interval_ms);
	                  } else {
	                    gl.deleteSync(sync);
	                    resolve();
	                  }
	                }
	                test();
	              });
	            };
	            impl = (_this$renderTarget$co = this.renderTarget.colorBuffer) == null ? void 0 : _this$renderTarget$co.impl;
	            format = (_impl$_glFormat = impl == null ? void 0 : impl._glFormat) != null ? _impl$_glFormat : gl.RGBA;
	            pixelType = (_impl$_glPixelType = impl == null ? void 0 : impl._glPixelType) != null ? _impl$_glPixelType : gl.UNSIGNED_BYTE; // create temporary (gpu-side) buffer and copy data into it
	            buf = gl.createBuffer();
	            gl.bindBuffer(gl.PIXEL_PACK_BUFFER, buf);
	            gl.bufferData(gl.PIXEL_PACK_BUFFER, pixels.byteLength, gl.STREAM_READ);
	            gl.readPixels(x, y, w, h, format, pixelType, 0);
	            gl.bindBuffer(gl.PIXEL_PACK_BUFFER, null);

	            // async wait for previous read to finish
	            _context.next = 15;
	            return clientWaitAsync(0, 20);
	          case 15:
	            // copy the resulting data once it's arrived
	            gl.bindBuffer(gl.PIXEL_PACK_BUFFER, buf);
	            gl.getBufferSubData(gl.PIXEL_PACK_BUFFER, 0, pixels);
	            gl.bindBuffer(gl.PIXEL_PACK_BUFFER, null);
	            gl.deleteBuffer(buf);
	          case 19:
	          case "end":
	            return _context.stop();
	        }
	      }, _callee, this);
	    }));
	    function readPixelsAsync(_x, _x2, _x3, _x4, _x5) {
	      return _readPixelsAsync.apply(this, arguments);
	    }
	    return readPixelsAsync;
	  }()
	  /**
	   * Enables or disables alpha to coverage (WebGL2 only).
	   *
	   * @param {boolean} state - True to enable alpha to coverage and false to disable it.
	   * @ignore
	   */
	  ;
	  _proto.setAlphaToCoverage = function setAlphaToCoverage(state) {
	    if (this.isWebGL1) return;
	    if (this.alphaToCoverage === state) return;
	    this.alphaToCoverage = state;
	    if (state) {
	      this.gl.enable(this.gl.SAMPLE_ALPHA_TO_COVERAGE);
	    } else {
	      this.gl.disable(this.gl.SAMPLE_ALPHA_TO_COVERAGE);
	    }
	  }

	  /**
	   * Sets the output vertex buffer. It will be written to by a shader with transform feedback
	   * varyings.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('../vertex-buffer.js').VertexBuffer} tf - The output vertex buffer.
	   * @ignore
	   */;
	  _proto.setTransformFeedbackBuffer = function setTransformFeedbackBuffer(tf) {
	    if (this.transformFeedbackBuffer === tf) return;
	    this.transformFeedbackBuffer = tf;
	    if (this.isWebGL2) {
	      var _gl = this.gl;
	      if (tf) {
	        if (!this.feedback) {
	          this.feedback = _gl.createTransformFeedback();
	        }
	        _gl.bindTransformFeedback(_gl.TRANSFORM_FEEDBACK, this.feedback);
	      } else {
	        _gl.bindTransformFeedback(_gl.TRANSFORM_FEEDBACK, null);
	      }
	    }
	  }

	  /**
	   * Toggles the rasterization render state. Useful with transform feedback, when you only need
	   * to process the data without drawing.
	   *
	   * @param {boolean} on - True to enable rasterization and false to disable it.
	   * @ignore
	   */;
	  _proto.setRaster = function setRaster(on) {
	    if (this.raster === on) return;
	    this.raster = on;
	    if (this.isWebGL2) {
	      if (on) {
	        this.gl.disable(this.gl.RASTERIZER_DISCARD);
	      } else {
	        this.gl.enable(this.gl.RASTERIZER_DISCARD);
	      }
	    }
	  };
	  _proto.setStencilTest = function setStencilTest(enable) {
	    if (this.stencil !== enable) {
	      var _gl2 = this.gl;
	      if (enable) {
	        _gl2.enable(_gl2.STENCIL_TEST);
	      } else {
	        _gl2.disable(_gl2.STENCIL_TEST);
	      }
	      this.stencil = enable;
	    }
	  };
	  _proto.setStencilFunc = function setStencilFunc(func, ref, mask) {
	    if (this.stencilFuncFront !== func || this.stencilRefFront !== ref || this.stencilMaskFront !== mask || this.stencilFuncBack !== func || this.stencilRefBack !== ref || this.stencilMaskBack !== mask) {
	      this.gl.stencilFunc(this.glComparison[func], ref, mask);
	      this.stencilFuncFront = this.stencilFuncBack = func;
	      this.stencilRefFront = this.stencilRefBack = ref;
	      this.stencilMaskFront = this.stencilMaskBack = mask;
	    }
	  };
	  _proto.setStencilFuncFront = function setStencilFuncFront(func, ref, mask) {
	    if (this.stencilFuncFront !== func || this.stencilRefFront !== ref || this.stencilMaskFront !== mask) {
	      var _gl3 = this.gl;
	      _gl3.stencilFuncSeparate(_gl3.FRONT, this.glComparison[func], ref, mask);
	      this.stencilFuncFront = func;
	      this.stencilRefFront = ref;
	      this.stencilMaskFront = mask;
	    }
	  };
	  _proto.setStencilFuncBack = function setStencilFuncBack(func, ref, mask) {
	    if (this.stencilFuncBack !== func || this.stencilRefBack !== ref || this.stencilMaskBack !== mask) {
	      var _gl4 = this.gl;
	      _gl4.stencilFuncSeparate(_gl4.BACK, this.glComparison[func], ref, mask);
	      this.stencilFuncBack = func;
	      this.stencilRefBack = ref;
	      this.stencilMaskBack = mask;
	    }
	  };
	  _proto.setStencilOperation = function setStencilOperation(fail, zfail, zpass, writeMask) {
	    if (this.stencilFailFront !== fail || this.stencilZfailFront !== zfail || this.stencilZpassFront !== zpass || this.stencilFailBack !== fail || this.stencilZfailBack !== zfail || this.stencilZpassBack !== zpass) {
	      this.gl.stencilOp(this.glStencilOp[fail], this.glStencilOp[zfail], this.glStencilOp[zpass]);
	      this.stencilFailFront = this.stencilFailBack = fail;
	      this.stencilZfailFront = this.stencilZfailBack = zfail;
	      this.stencilZpassFront = this.stencilZpassBack = zpass;
	    }
	    if (this.stencilWriteMaskFront !== writeMask || this.stencilWriteMaskBack !== writeMask) {
	      this.gl.stencilMask(writeMask);
	      this.stencilWriteMaskFront = writeMask;
	      this.stencilWriteMaskBack = writeMask;
	    }
	  };
	  _proto.setStencilOperationFront = function setStencilOperationFront(fail, zfail, zpass, writeMask) {
	    if (this.stencilFailFront !== fail || this.stencilZfailFront !== zfail || this.stencilZpassFront !== zpass) {
	      this.gl.stencilOpSeparate(this.gl.FRONT, this.glStencilOp[fail], this.glStencilOp[zfail], this.glStencilOp[zpass]);
	      this.stencilFailFront = fail;
	      this.stencilZfailFront = zfail;
	      this.stencilZpassFront = zpass;
	    }
	    if (this.stencilWriteMaskFront !== writeMask) {
	      this.gl.stencilMaskSeparate(this.gl.FRONT, writeMask);
	      this.stencilWriteMaskFront = writeMask;
	    }
	  };
	  _proto.setStencilOperationBack = function setStencilOperationBack(fail, zfail, zpass, writeMask) {
	    if (this.stencilFailBack !== fail || this.stencilZfailBack !== zfail || this.stencilZpassBack !== zpass) {
	      this.gl.stencilOpSeparate(this.gl.BACK, this.glStencilOp[fail], this.glStencilOp[zfail], this.glStencilOp[zpass]);
	      this.stencilFailBack = fail;
	      this.stencilZfailBack = zfail;
	      this.stencilZpassBack = zpass;
	    }
	    if (this.stencilWriteMaskBack !== writeMask) {
	      this.gl.stencilMaskSeparate(this.gl.BACK, writeMask);
	      this.stencilWriteMaskBack = writeMask;
	    }
	  };
	  _proto.setBlendState = function setBlendState(blendState) {
	    var currentBlendState = this.blendState;
	    if (!currentBlendState.equals(blendState)) {
	      var _gl5 = this.gl;

	      // state values to set
	      var blend = blendState.blend,
	        colorOp = blendState.colorOp,
	        alphaOp = blendState.alphaOp,
	        colorSrcFactor = blendState.colorSrcFactor,
	        colorDstFactor = blendState.colorDstFactor,
	        alphaSrcFactor = blendState.alphaSrcFactor,
	        alphaDstFactor = blendState.alphaDstFactor;

	      // enable blend
	      if (currentBlendState.blend !== blend) {
	        if (blend) {
	          _gl5.enable(_gl5.BLEND);
	        } else {
	          _gl5.disable(_gl5.BLEND);
	        }
	      }

	      // blend ops
	      if (currentBlendState.colorOp !== colorOp || currentBlendState.alphaOp !== alphaOp) {
	        var glBlendEquation = this.glBlendEquation;
	        _gl5.blendEquationSeparate(glBlendEquation[colorOp], glBlendEquation[alphaOp]);
	      }

	      // blend factors
	      if (currentBlendState.colorSrcFactor !== colorSrcFactor || currentBlendState.colorDstFactor !== colorDstFactor || currentBlendState.alphaSrcFactor !== alphaSrcFactor || currentBlendState.alphaDstFactor !== alphaDstFactor) {
	        _gl5.blendFuncSeparate(this.glBlendFunctionColor[colorSrcFactor], this.glBlendFunctionColor[colorDstFactor], this.glBlendFunctionAlpha[alphaSrcFactor], this.glBlendFunctionAlpha[alphaDstFactor]);
	      }

	      // color write
	      if (currentBlendState.allWrite !== blendState.allWrite) {
	        this.gl.colorMask(blendState.redWrite, blendState.greenWrite, blendState.blueWrite, blendState.alphaWrite);
	      }

	      // update internal state
	      currentBlendState.copy(blendState);
	    }
	  }

	  /**
	   * Set the source and destination blending factors.
	   *
	   * @param {number} r - The red component in the range of 0 to 1. Default value is 0.
	   * @param {number} g - The green component in the range of 0 to 1. Default value is 0.
	   * @param {number} b - The blue component in the range of 0 to 1. Default value is 0.
	   * @param {number} a - The alpha component in the range of 0 to 1. Default value is 0.
	   * @ignore
	   */;
	  _proto.setBlendColor = function setBlendColor(r, g, b, a) {
	    var c = this.blendColor;
	    if (r !== c.r || g !== c.g || b !== c.b || a !== c.a) {
	      this.gl.blendColor(r, g, b, a);
	      c.set(r, g, b, a);
	    }
	  };
	  _proto.setStencilState = function setStencilState(stencilFront, stencilBack) {
	    if (stencilFront || stencilBack) {
	      this.setStencilTest(true);
	      if (stencilFront === stencilBack) {
	        // identical front/back stencil
	        this.setStencilFunc(stencilFront.func, stencilFront.ref, stencilFront.readMask);
	        this.setStencilOperation(stencilFront.fail, stencilFront.zfail, stencilFront.zpass, stencilFront.writeMask);
	      } else {
	        var _stencilFront, _stencilBack;
	        // front
	        (_stencilFront = stencilFront) != null ? _stencilFront : stencilFront = StencilParameters.DEFAULT;
	        this.setStencilFuncFront(stencilFront.func, stencilFront.ref, stencilFront.readMask);
	        this.setStencilOperationFront(stencilFront.fail, stencilFront.zfail, stencilFront.zpass, stencilFront.writeMask);

	        // back
	        (_stencilBack = stencilBack) != null ? _stencilBack : stencilBack = StencilParameters.DEFAULT;
	        this.setStencilFuncBack(stencilBack.func, stencilBack.ref, stencilBack.readMask);
	        this.setStencilOperationBack(stencilBack.fail, stencilBack.zfail, stencilBack.zpass, stencilBack.writeMask);
	      }
	    } else {
	      this.setStencilTest(false);
	    }
	  };
	  _proto.setDepthState = function setDepthState(depthState) {
	    var currentDepthState = this.depthState;
	    if (!currentDepthState.equals(depthState)) {
	      var _gl6 = this.gl;

	      // write
	      var write = depthState.write;
	      if (currentDepthState.write !== write) {
	        _gl6.depthMask(write);
	      }

	      // handle case where depth testing is off, but depth write is on => enable always test to depth write
	      // Note on WebGL API behavior: When depth testing is disabled, writes to the depth buffer are also disabled.
	      var func = depthState.func,
	        test = depthState.test;
	      if (!test && write) {
	        test = true;
	        func = FUNC_ALWAYS;
	      }
	      if (currentDepthState.func !== func) {
	        _gl6.depthFunc(this.glComparison[func]);
	      }
	      if (currentDepthState.test !== test) {
	        if (test) {
	          _gl6.enable(_gl6.DEPTH_TEST);
	        } else {
	          _gl6.disable(_gl6.DEPTH_TEST);
	        }
	      }

	      // depth bias
	      var depthBias = depthState.depthBias,
	        depthBiasSlope = depthState.depthBiasSlope;
	      if (depthBias || depthBiasSlope) {
	        // enable bias
	        if (!this.depthBiasEnabled) {
	          this.depthBiasEnabled = true;
	          this.gl.enable(this.gl.POLYGON_OFFSET_FILL);
	        }

	        // values
	        _gl6.polygonOffset(depthBiasSlope, depthBias);
	      } else {
	        // disable bias
	        if (this.depthBiasEnabled) {
	          this.depthBiasEnabled = false;
	          this.gl.disable(this.gl.POLYGON_OFFSET_FILL);
	        }
	      }

	      // update internal state
	      currentDepthState.copy(depthState);
	    }
	  };
	  _proto.setCullMode = function setCullMode(cullMode) {
	    if (this.cullMode !== cullMode) {
	      if (cullMode === CULLFACE_NONE) {
	        this.gl.disable(this.gl.CULL_FACE);
	      } else {
	        if (this.cullMode === CULLFACE_NONE) {
	          this.gl.enable(this.gl.CULL_FACE);
	        }
	        var mode = this.glCull[cullMode];
	        if (this.cullFace !== mode) {
	          this.gl.cullFace(mode);
	          this.cullFace = mode;
	        }
	      }
	      this.cullMode = cullMode;
	    }
	  }

	  /**
	   * Sets the active shader to be used during subsequent draw calls.
	   *
	   * @param {Shader} shader - The shader to assign to the device.
	   */

	  /**
	   * Sets the active shader to be used during subsequent draw calls.
	   *
	   * @param {Shader} shader - The shader to assign to the device.
	   * @param {boolean} asyncCompile - If true, rendering will be skipped until the shader is
	   * compiled, otherwise the rendering will wait for the shader compilation to finish. Defaults to
	   * false.
	   */;
	  _proto.setShader = function setShader(shader, asyncCompile) {
	    if (asyncCompile === void 0) {
	      asyncCompile = false;
	    }
	    if (shader !== this.shader) {
	      this.shader = shader;
	      this.shaderAsyncCompile = asyncCompile;
	      this.shaderValid = undefined; // need to run activation / validation

	      this._shaderSwitchesPerFrame++;
	    }
	  };
	  _proto.activateShader = function activateShader(device) {
	    var shader = this.shader;
	    var impl = shader.impl;
	    if (this.shaderValid === undefined) {
	      if (shader.failed) {
	        this.shaderValid = false;
	      } else if (!shader.ready) {
	        // if the shader is async compiled and can be skipped if not ready
	        if (this.shaderAsyncCompile) {
	          // if the shader is linked, finalize it
	          if (impl.isLinked(device)) {
	            if (!impl.finalize(this, shader)) {
	              shader.failed = true;
	              this.shaderValid = false;
	            }
	          } else {
	            // skip the async shader rendering
	            this.shaderValid = false;
	          }
	        } else {
	          // this cannot be skipped, wait for the shader to be ready
	          if (!impl.finalize(this, shader)) {
	            shader.failed = true;
	            this.shaderValid = false;
	          }
	        }
	      }
	    }
	    if (this.shaderValid === undefined) {
	      // Set the active shader
	      this.gl.useProgram(impl.glProgram);
	      this.shaderValid = true;
	    }
	  }

	  /**
	   * Frees memory from all vertex array objects ever allocated with this device.
	   *
	   * @ignore
	   */;
	  _proto.clearVertexArrayObjectCache = function clearVertexArrayObjectCache() {
	    var gl = this.gl;
	    this._vaoMap.forEach(function (item, key, mapObj) {
	      gl.deleteVertexArray(item);
	    });
	    this._vaoMap.clear();
	  }

	  /**
	   * Fullscreen mode.
	   *
	   * @type {boolean}
	   */;
	  // debug helper to force lost context
	  _proto.debugLoseContext = function debugLoseContext(sleep) {
	    if (sleep === void 0) {
	      sleep = 100;
	    }
	    var context = this.gl.getExtension('WEBGL_lose_context');
	    context.loseContext();
	    setTimeout(function () {
	      return context.restoreContext();
	    }, sleep);
	  };
	  _createClass(WebglGraphicsDevice, [{
	    key: "extDisjointTimerQuery",
	    get: function get() {
	      // lazy evaluation as this is not typically used
	      if (!this._extDisjointTimerQuery) {
	        if (this.isWebGL2) {
	          // Note that Firefox exposes EXT_disjoint_timer_query under WebGL2 rather than EXT_disjoint_timer_query_webgl2
	          this._extDisjointTimerQuery = this.getExtension('EXT_disjoint_timer_query_webgl2', 'EXT_disjoint_timer_query');
	        }
	      }
	      return this._extDisjointTimerQuery;
	    }
	  }, {
	    key: "defaultFramebuffer",
	    get: function get() {
	      return this._defaultFramebuffer;
	    },
	    set: function set(value) {
	      if (this._defaultFramebuffer !== value) {
	        this._defaultFramebuffer = value;
	        this._defaultFramebufferChanged = true;
	      }
	    }
	  }, {
	    key: "fullscreen",
	    get: function get() {
	      return !!document.fullscreenElement;
	    }

	    /**
	     * Check if high precision floating-point textures are supported.
	     *
	     * @type {boolean}
	     */,
	    set: function set(fullscreen) {
	      if (fullscreen) {
	        var canvas = this.gl.canvas;
	        canvas.requestFullscreen();
	      } else {
	        document.exitFullscreen();
	      }
	    }
	  }, {
	    key: "textureFloatHighPrecision",
	    get: function get() {
	      if (this._textureFloatHighPrecision === undefined) {
	        this._textureFloatHighPrecision = testTextureFloatHighPrecision(this);
	      }
	      return this._textureFloatHighPrecision;
	    }

	    /**
	     * Check if texture with half float format can be updated with data.
	     *
	     * @type {boolean}
	     */
	  }, {
	    key: "textureHalfFloatUpdatable",
	    get: function get() {
	      if (this._textureHalfFloatUpdatable === undefined) {
	        if (this.isWebGL2) {
	          this._textureHalfFloatUpdatable = true;
	        } else {
	          this._textureHalfFloatUpdatable = testTextureHalfFloatUpdatable(this.gl, this.extTextureHalfFloat.HALF_FLOAT_OES);
	        }
	      }
	      return this._textureHalfFloatUpdatable;
	    }
	  }]);
	  return WebglGraphicsDevice;
	}(GraphicsDevice);

	/**
	 * A Null implementation of the IndexBuffer.
	 *
	 * @ignore
	 */
	var NullIndexBuffer = /*#__PURE__*/function () {
	  function NullIndexBuffer() {}
	  var _proto = NullIndexBuffer.prototype;
	  _proto.unlock = function unlock(indexBuffer) {};
	  return NullIndexBuffer;
	}();

	/**
	 * A Null implementation of the RenderTarget.
	 *
	 * @ignore
	 */
	var NullRenderTarget = /*#__PURE__*/function () {
	  function NullRenderTarget() {}
	  var _proto = NullRenderTarget.prototype;
	  _proto.destroy = function destroy(device) {};
	  _proto.init = function init(device, renderTarget) {};
	  _proto.loseContext = function loseContext() {};
	  _proto.resolve = function resolve(device, target, color, depth) {};
	  return NullRenderTarget;
	}();

	/**
	 * A Null implementation of the Shader.
	 *
	 * @ignore
	 */
	var NullShader = /*#__PURE__*/function () {
	  function NullShader() {}
	  var _proto = NullShader.prototype;
	  _proto.destroy = function destroy(shader) {};
	  _proto.loseContext = function loseContext() {};
	  _proto.restoreContext = function restoreContext(device, shader) {};
	  return NullShader;
	}();

	/**
	 * A NULL implementation of the Texture.
	 *
	 * @ignore
	 */
	var NullTexture = /*#__PURE__*/function () {
	  function NullTexture() {}
	  var _proto = NullTexture.prototype;
	  _proto.destroy = function destroy(device) {};
	  _proto.propertyChanged = function propertyChanged(flag) {};
	  _proto.loseContext = function loseContext() {};
	  return NullTexture;
	}();

	/**
	 * A Null implementation of the VertexBuffer.
	 *
	 * @ignore
	 */
	var NullVertexBuffer = /*#__PURE__*/function () {
	  function NullVertexBuffer() {}
	  var _proto = NullVertexBuffer.prototype;
	  _proto.destroy = function destroy(device) {};
	  _proto.unlock = function unlock(vertexBuffer) {};
	  return NullVertexBuffer;
	}();

	var NullGraphicsDevice = /*#__PURE__*/function (_GraphicsDevice) {
	  _inheritsLoose(NullGraphicsDevice, _GraphicsDevice);
	  function NullGraphicsDevice(canvas, options) {
	    var _this;
	    if (options === void 0) {
	      options = {};
	    }
	    _this = _GraphicsDevice.call(this, canvas, options) || this;
	    options = _this.initOptions;
	    _this.isNull = true;
	    _this._deviceType = DEVICETYPE_NULL;
	    _this.samples = 1;
	    Debug.log('NullGraphicsDevice');
	    return _this;
	  }
	  var _proto = NullGraphicsDevice.prototype;
	  _proto.destroy = function destroy() {
	    _GraphicsDevice.prototype.destroy.call(this);
	  };
	  _proto.initDeviceCaps = function initDeviceCaps() {
	    this.disableParticleSystem = true;
	    this.precision = 'highp';
	    this.maxPrecision = 'highp';
	    this.maxSamples = 4;
	    this.maxTextures = 16;
	    this.maxTextureSize = 4096;
	    this.maxCubeMapSize = 4096;
	    this.maxVolumeSize = 4096;
	    this.maxColorAttachments = 8;
	    this.maxPixelRatio = 1;
	    this.maxAnisotropy = 16;
	    this.supportsInstancing = true;
	    this.supportsUniformBuffers = false;
	    this.supportsVolumeTextures = true;
	    this.supportsBoneTextures = true;
	    this.supportsMorphTargetTexturesCore = true;
	    this.supportsAreaLights = true;
	    this.supportsDepthShadow = true;
	    this.supportsGpuParticles = false;
	    this.supportsMrt = true;
	    this.extUintElement = true;
	    this.extTextureFloat = true;
	    this.textureFloatRenderable = true;
	    this.extTextureHalfFloat = true;
	    this.textureHalfFloatRenderable = true;
	    this.textureHalfFloatUpdatable = true;
	    this.boneLimit = 1024;
	    this.supportsImageBitmap = true;
	    this.extStandardDerivatives = true;
	    this.extBlendMinmax = true;
	    this.areaLightLutFormat = PIXELFORMAT_RGBA8;
	    this.supportsTextureFetch = true;
	  };
	  _proto.postInit = function postInit() {
	    _GraphicsDevice.prototype.postInit.call(this);
	  };
	  _proto.frameStart = function frameStart() {
	    _GraphicsDevice.prototype.frameStart.call(this);
	  };
	  _proto.frameEnd = function frameEnd() {
	    _GraphicsDevice.prototype.frameEnd.call(this);
	  };
	  _proto.updateBegin = function updateBegin() {};
	  _proto.updateEnd = function updateEnd() {};
	  _proto.readPixels = function readPixels(x, y, w, h, pixels) {};
	  _proto.createVertexBufferImpl = function createVertexBufferImpl(vertexBuffer, format) {
	    return new NullVertexBuffer(vertexBuffer, format);
	  };
	  _proto.createIndexBufferImpl = function createIndexBufferImpl(indexBuffer) {
	    return new NullIndexBuffer(indexBuffer);
	  };
	  _proto.createShaderImpl = function createShaderImpl(shader) {
	    return new NullShader(shader);
	  };
	  _proto.createTextureImpl = function createTextureImpl(texture) {
	    return new NullTexture(texture);
	  };
	  _proto.createRenderTargetImpl = function createRenderTargetImpl(renderTarget) {
	    return new NullRenderTarget(renderTarget);
	  };
	  _proto.draw = function draw(primitive, numInstances, keepBuffers) {
	  };
	  _proto.setShader = function setShader(shader, asyncCompile) {
	  };
	  _proto.setBlendState = function setBlendState(blendState) {};
	  _proto.setDepthState = function setDepthState(depthState) {};
	  _proto.setStencilState = function setStencilState(stencilFront, stencilBack) {};
	  _proto.setBlendColor = function setBlendColor(r, g, b, a) {};
	  _proto.setCullMode = function setCullMode(cullMode) {};
	  _proto.setAlphaToCoverage = function setAlphaToCoverage(state) {};
	  _proto.initializeContextCaches = function initializeContextCaches() {
	    _GraphicsDevice.prototype.initializeContextCaches.call(this);
	  };
	  _proto.clear = function clear(options) {};
	  _proto.setViewport = function setViewport(x, y, w, h) {};
	  _proto.setScissor = function setScissor(x, y, w, h) {};
	  _proto.copyRenderTarget = function copyRenderTarget(source, dest, color, depth) {
	    return true;
	  };
	  _proto.pushMarker = function pushMarker(name) {};
	  _proto.popMarker = function popMarker() {};
	  return NullGraphicsDevice;
	}(GraphicsDevice);

	/**
	 * Creates a graphics device.
	 *
	 * @param {HTMLCanvasElement} canvas - The canvas element.
	 * @param {object} options - Graphics device options.
	 * @param {string[]} [options.deviceTypes] - An array of DEVICETYPE_*** constants, defining the
	 * order in which the devices are attempted to get created. Defaults to an empty array. If the
	 * specified array does not contain [{@link DEVICETYPE_WEBGL2} or {@link DEVICETYPE_WEBGL1}], those
	 * are internally added to its end in this order. Typically, you'd only specify
	 * {@link DEVICETYPE_WEBGPU}, or leave it empty.
	 * @param {boolean} [options.antialias] - Boolean that indicates whether or not to perform
	 * anti-aliasing if possible. Defaults to true.
	 * @param {boolean} [options.depth] - Boolean that indicates that the drawing buffer is
	 * requested to have a depth buffer of at least 16 bits. Defaults to true.
	 * @param {boolean} [options.stencil] - Boolean that indicates that the drawing buffer is
	 * requested to have a stencil buffer of at least 8 bits. Defaults to true.
	 * @param {string} [options.glslangUrl] - The URL to the glslang script. Required if the
	 * {@link DEVICETYPE_WEBGPU} type is added to deviceTypes array. Not used for
	 * {@link DEVICETYPE_WEBGL1} or {@link DEVICETYPE_WEBGL2} device type creation.
	 * @param {string} [options.twgslUrl] - An url to twgsl script, required if glslangUrl was specified.
	 * @param {boolean} [options.xrCompatible] - Boolean that hints to the user agent to use a
	 * compatible graphics adapter for an immersive XR device.
	 * @param {'default'|'high-performance'|'low-power'} [options.powerPreference] - A hint indicating
	 * what configuration of GPU would be selected. Possible values are:
	 *
	 * - 'default': Let the user agent decide which GPU configuration is most suitable. This is the
	 * default value.
	 * - 'high-performance': Prioritizes rendering performance over power consumption.
	 * - 'low-power': Prioritizes power saving over rendering performance.
	 *
	 * Defaults to 'default'.
	 * @returns {Promise} - Promise object representing the created graphics device.
	 * @category Graphics
	 */
	function createGraphicsDevice(canvas, options) {
	  var _options$deviceTypes;
	  if (options === void 0) {
	    options = {};
	  }
	  var deviceTypes = (_options$deviceTypes = options.deviceTypes) != null ? _options$deviceTypes : [];

	  // automatically added fallbacks
	  if (!deviceTypes.includes(DEVICETYPE_WEBGL2)) {
	    deviceTypes.push(DEVICETYPE_WEBGL2);
	  }
	  if (!deviceTypes.includes(DEVICETYPE_WEBGL1)) {
	    deviceTypes.push(DEVICETYPE_WEBGL1);
	  }
	  if (!deviceTypes.includes(DEVICETYPE_NULL)) {
	    deviceTypes.push(DEVICETYPE_NULL);
	  }

	  // XR compatibility if not specified
	  if (platform.browser && !!navigator.xr) {
	    var _options, _options$xrCompatible;
	    (_options$xrCompatible = (_options = options).xrCompatible) != null ? _options$xrCompatible : _options.xrCompatible = true;
	  }

	  // make a list of device creation functions in priority order
	  var deviceCreateFuncs = [];
	  var _loop = function _loop() {
	    var _window;
	    var deviceType = deviceTypes[i];
	    if (deviceType === DEVICETYPE_WEBGPU && (_window = window) != null && (_window = _window.navigator) != null && _window.gpu) {
	      deviceCreateFuncs.push(function () {
	        var device = new WebgpuGraphicsDevice(canvas, options);
	        return device.initWebGpu(options.glslangUrl, options.twgslUrl);
	      });
	    }
	    if (deviceType === DEVICETYPE_WEBGL1 || deviceType === DEVICETYPE_WEBGL2) {
	      deviceCreateFuncs.push(function () {
	        options.preferWebGl2 = deviceType === DEVICETYPE_WEBGL2;
	        return new WebglGraphicsDevice(canvas, options);
	      });
	    }
	    if (deviceType === DEVICETYPE_NULL) {
	      deviceCreateFuncs.push(function () {
	        return new NullGraphicsDevice(canvas, options);
	      });
	    }
	  };
	  for (var i = 0; i < deviceTypes.length; i++) {
	    _loop();
	  }

	  // execute each device creation function returning the first successful result
	  return new Promise(function (resolve, reject) {
	    var attempt = 0;
	    var next = function next() {
	      if (attempt >= deviceCreateFuncs.length) {
	        reject(new Error('Failed to create a graphics device'));
	      } else {
	        Promise.resolve(deviceCreateFuncs[attempt++]()).then(function (device) {
	          if (device) {
	            resolve(device);
	          } else {
	            next();
	          }
	        }).catch(function (err) {
	          console.log(err);
	          next();
	        });
	      }
	    };
	    next();
	  });
	}

	/**
	 * A representation of a compute shader with the associated data, that can be executed on the GPU.
	 *
	 * @ignore
	 */
	var Compute = /*#__PURE__*/function () {
	  /**
	   * Create a compute instance. Note that this is supported on WebGPU only and is a no-op on
	   * other platforms.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./graphics-device.js').GraphicsDevice} graphicsDevice -
	   * The graphics device.
	   * @param {new Function("modulePath", "return import(modulePath)")('./shader.js').Shader} shader - The compute shader.
	   */
	  function Compute(graphicsDevice, shader) {
	    /**
	     * A compute shader.
	     *
	     * @type {new Function("modulePath", "return import(modulePath)")('./shader.js').Shader|null}
	     * @ignore
	     */
	    this.shader = null;
	    this.device = graphicsDevice;
	    this.shader = shader;
	    if (graphicsDevice.supportsCompute) {
	      this.impl = graphicsDevice.createComputeImpl(this);
	    }
	  }

	  /**
	   * Dispatch the compute work.
	   *
	   * @param {number} x - X dimension of the grid of work-groups to dispatch.
	   * @param {number} [y] - Y dimension of the grid of work-groups to dispatch.
	   * @param {number} [z] - Z dimension of the grid of work-groups to dispatch.
	   */
	  var _proto = Compute.prototype;
	  _proto.dispatch = function dispatch(x, y, z) {
	    var _this$impl;
	    (_this$impl = this.impl) == null || _this$impl.dispatch(x, y, z);
	  };
	  return Compute;
	}();

	var id$4 = 0;

	/**
	 * An index buffer stores index values into a {@link VertexBuffer}. Indexed graphical primitives
	 * can normally utilize less memory that unindexed primitives (if vertices are shared).
	 *
	 * Typically, index buffers are set on {@link Mesh} objects.
	 *
	 * @category Graphics
	 */
	var IndexBuffer = /*#__PURE__*/function () {
	  /**
	   * Create a new IndexBuffer instance.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./graphics-device.js').GraphicsDevice} graphicsDevice - The graphics device
	   * used to manage this index buffer.
	   * @param {number} format - The type of each index to be stored in the index buffer. Can be:
	   *
	   * - {@link INDEXFORMAT_UINT8}
	   * - {@link INDEXFORMAT_UINT16}
	   * - {@link INDEXFORMAT_UINT32}
	   * @param {number} numIndices - The number of indices to be stored in the index buffer.
	   * @param {number} [usage] - The usage type of the vertex buffer. Can be:
	   *
	   * - {@link BUFFER_DYNAMIC}
	   * - {@link BUFFER_STATIC}
	   * - {@link BUFFER_STREAM}
	   *
	   * Defaults to {@link BUFFER_STATIC}.
	   * @param {ArrayBuffer} [initialData] - Initial data. If left unspecified, the index buffer
	   * will be initialized to zeros.
	   * @example
	   * // Create an index buffer holding 3 16-bit indices. The buffer is marked as
	   * // static, hinting that the buffer will never be modified.
	   * const indices = new UInt16Array([0, 1, 2]);
	   * const indexBuffer = new pc.IndexBuffer(graphicsDevice,
	   *                                        pc.INDEXFORMAT_UINT16,
	   *                                        3,
	   *                                        pc.BUFFER_STATIC,
	   *                                        indices);
	   */
	  function IndexBuffer(graphicsDevice, format, numIndices, usage, initialData) {
	    if (usage === void 0) {
	      usage = BUFFER_STATIC;
	    }
	    // By default, index buffers are static (better for performance since buffer data can be cached in VRAM)
	    this.device = graphicsDevice;
	    this.format = format;
	    this.numIndices = numIndices;
	    this.usage = usage;
	    this.id = id$4++;
	    this.impl = graphicsDevice.createIndexBufferImpl(this);

	    // Allocate the storage
	    var bytesPerIndex = typedArrayIndexFormatsByteSize[format];
	    this.bytesPerIndex = bytesPerIndex;
	    this.numBytes = this.numIndices * bytesPerIndex;
	    if (initialData) {
	      this.setData(initialData);
	    } else {
	      this.storage = new ArrayBuffer(this.numBytes);
	    }
	    this.adjustVramSizeTracking(graphicsDevice._vram, this.numBytes);
	    this.device.buffers.push(this);
	  }

	  /**
	   * Frees resources associated with this index buffer.
	   */
	  var _proto = IndexBuffer.prototype;
	  _proto.destroy = function destroy() {
	    // stop tracking the index buffer
	    var device = this.device;
	    var idx = device.buffers.indexOf(this);
	    if (idx !== -1) {
	      device.buffers.splice(idx, 1);
	    }
	    if (this.device.indexBuffer === this) {
	      this.device.indexBuffer = null;
	    }
	    if (this.impl.initialized) {
	      this.impl.destroy(device);
	      this.adjustVramSizeTracking(device._vram, -this.storage.byteLength);
	    }
	  };
	  _proto.adjustVramSizeTracking = function adjustVramSizeTracking(vram, size) {
	    Debug.trace(TRACEID_VRAM_IB, this.id + " size: " + size + " vram.ib: " + vram.ib + " => " + (vram.ib + size));
	    vram.ib += size;
	  }

	  /**
	   * Called when the rendering context was lost. It releases all context related resources.
	   *
	   * @ignore
	   */;
	  _proto.loseContext = function loseContext() {
	    this.impl.loseContext();
	  }

	  /**
	   * Returns the data format of the specified index buffer.
	   *
	   * @returns {number} The data format of the specified index buffer. Can be:
	   *
	   * - {@link INDEXFORMAT_UINT8}
	   * - {@link INDEXFORMAT_UINT16}
	   * - {@link INDEXFORMAT_UINT32}
	   */;
	  _proto.getFormat = function getFormat() {
	    return this.format;
	  }

	  /**
	   * Returns the number of indices stored in the specified index buffer.
	   *
	   * @returns {number} The number of indices stored in the specified index buffer.
	   */;
	  _proto.getNumIndices = function getNumIndices() {
	    return this.numIndices;
	  }

	  /**
	   * Gives access to the block of memory that stores the buffer's indices.
	   *
	   * @returns {ArrayBuffer} A contiguous block of memory where index data can be written to.
	   */;
	  _proto.lock = function lock() {
	    return this.storage;
	  }

	  /**
	   * Signals that the block of memory returned by a call to the lock function is ready to be
	   * given to the graphics hardware. Only unlocked index buffers can be set on the currently
	   * active device.
	   */;
	  _proto.unlock = function unlock() {
	    // Upload the new index data
	    this.impl.unlock(this);
	  }

	  /**
	   * Set preallocated data on the index buffer.
	   *
	   * @param {ArrayBuffer} data - The index data to set.
	   * @returns {boolean} True if the data was set successfully, false otherwise.
	   * @ignore
	   */;
	  _proto.setData = function setData(data) {
	    if (data.byteLength !== this.numBytes) {
	      Debug.error("IndexBuffer: wrong initial data size: expected " + this.numBytes + ", got " + data.byteLength);
	      return false;
	    }
	    this.storage = data;
	    this.unlock();
	    return true;
	  }

	  /**
	   * Get the appropriate typed array from an index buffer.
	   *
	   * @returns {Uint8Array|Uint16Array|Uint32Array} The typed array containing the index data.
	   * @private
	   */;
	  _proto._lockTypedArray = function _lockTypedArray() {
	    var lock = this.lock();
	    var indices = this.format === INDEXFORMAT_UINT32 ? new Uint32Array(lock) : this.format === INDEXFORMAT_UINT16 ? new Uint16Array(lock) : new Uint8Array(lock);
	    return indices;
	  }

	  /**
	   * Copies the specified number of elements from data into index buffer. Optimized for
	   * performance from both typed array as well as array.
	   *
	   * @param {Uint8Array|Uint16Array|Uint32Array|number[]} data - The data to write.
	   * @param {number} count - The number of indices to write.
	   * @ignore
	   */;
	  _proto.writeData = function writeData(data, count) {
	    var indices = this._lockTypedArray();

	    // if data contains more indices than needed, copy from its subarray
	    if (data.length > count) {
	      // if data is typed array
	      if (ArrayBuffer.isView(data)) {
	        data = data.subarray(0, count);
	        indices.set(data);
	      } else {
	        // data is array, copy right amount manually
	        for (var i = 0; i < count; i++) indices[i] = data[i];
	      }
	    } else {
	      // copy whole data
	      indices.set(data);
	    }
	    this.unlock();
	  }

	  /**
	   * Copies index data from index buffer into provided data array.
	   *
	   * @param {Uint8Array|Uint16Array|Uint32Array|number[]} data - The data array to write to.
	   * @returns {number} The number of indices read.
	   * @ignore
	   */;
	  _proto.readData = function readData(data) {
	    // note: there is no need to unlock this buffer, as we are only reading from it
	    var indices = this._lockTypedArray();
	    var count = this.numIndices;
	    if (ArrayBuffer.isView(data)) {
	      // destination data is typed array
	      data.set(indices);
	    } else {
	      // data is array, copy right amount manually
	      data.length = 0;
	      for (var i = 0; i < count; i++) data[i] = indices[i];
	    }
	    return count;
	  };
	  return IndexBuffer;
	}();

	var ColorAttachmentOps = function ColorAttachmentOps() {
	  /**
	   * A color used to clear the color attachment when the clear is enabled.
	   */
	  this.clearValue = new Color(0, 0, 0, 1);
	  /**
	   * True if the attachment should be cleared before rendering, false to preserve
	   * the existing content.
	   */
	  this.clear = false;
	  /**
	   * True if the attachment needs to be stored after the render pass. False
	   * if it can be discarded.
	   * Note: This relates to the surface that is getting rendered to, and can be either
	   * single or multi-sampled. Further, if a multi-sampled surface is used, the resolve
	   * flag further specifies if this gets resolved to a single-sampled surface. This
	   * behavior matches the WebGPU specification.
	   *
	   * @type {boolean}
	   */
	  this.store = false;
	  /**
	   * True if the attachment needs to be resolved.
	   *
	   * @type {boolean}
	   */
	  this.resolve = true;
	  /**
	   * True if the attachment needs to have mipmaps generated.
	   *
	   * @type {boolean}
	   */
	  this.mipmaps = false;
	};
	var DepthStencilAttachmentOps = function DepthStencilAttachmentOps() {
	  /**
	   * A depth value used to clear the depth attachment when the clear is enabled.
	   */
	  this.clearDepthValue = 1;
	  /**
	   * A stencil value used to clear the stencil attachment when the clear is enabled.
	   */
	  this.clearStencilValue = 0;
	  /**
	   * True if the depth attachment should be cleared before rendering, false to preserve
	   * the existing content.
	   */
	  this.clearDepth = false;
	  /**
	   * True if the stencil attachment should be cleared before rendering, false to preserve
	   * the existing content.
	   */
	  this.clearStencil = false;
	  /**
	   * True if the depth attachment needs to be stored after the render pass. False
	   * if it can be discarded.
	   *
	   * @type {boolean}
	   */
	  this.storeDepth = false;
	  /**
	   * True if the stencil attachment needs to be stored after the render pass. False
	   * if it can be discarded.
	   *
	   * @type {boolean}
	   */
	  this.storeStencil = false;
	};
	/**
	 * A render pass represents a node in the frame graph, and encapsulates a system which
	 * renders to a render target using an execution callback.
	 *
	 * @ignore
	 */
	var RenderPass = /*#__PURE__*/function () {
	  /**
	   * Creates an instance of the RenderPass.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('../graphics/graphics-device.js').GraphicsDevice} graphicsDevice - The
	   * graphics device.
	   */
	  function RenderPass(graphicsDevice) {
	    /** @type {string} */
	    this._name = void 0;
	    /**
	     * The graphics device.
	     *
	     * @type {new Function("modulePath", "return import(modulePath)")('../graphics/graphics-device.js').GraphicsDevice}
	     */
	    this.device = void 0;
	    /**
	     * True if the render pass is enabled.
	     *
	     * @type {boolean}
	     * @private
	     */
	    this._enabled = true;
	    /**
	     * True if the render pass is enabled and execute function will be called. Note that before and
	     * after functions are called regardless of this flag.
	     */
	    this.executeEnabled = true;
	    /**
	     * The render target for this render pass:
	     *  - `undefined`: render pass does not render to any render target
	     *  - `null`: render pass renders to the backbuffer
	     *  - Otherwise, renders to the provided RT.
	     * @type {new Function("modulePath", "return import(modulePath)")('../graphics/render-target.js').RenderTarget|null|undefined}
	     */
	    this.renderTarget = void 0;
	    /**
	     * The options specified when the render target was initialized.
	     */
	    this._options = void 0;
	    /**
	     * Number of samples. 0 if no render target, otherwise number of samples from the render target,
	     * or the main framebuffer if render target is null.
	     *
	     * @type {number}
	     */
	    this.samples = 0;
	    /**
	     * Array of color attachment operations. The first element corresponds to the color attachment
	     * 0, and so on.
	     *
	     * @type {Array<ColorAttachmentOps>}
	     */
	    this.colorArrayOps = [];
	    /** @type {DepthStencilAttachmentOps} */
	    this.depthStencilOps = void 0;
	    /**
	     * If true, this pass might use dynamically rendered cubemaps. Use for a case where rendering to cubemap
	     * faces is interleaved with rendering to shadows, to avoid generating cubemap mipmaps. This will likely
	     * be retired when render target dependency tracking gets implemented.
	     *
	     * @type {boolean}
	     */
	    this.requiresCubemaps = true;
	    /**
	     * True if the render pass uses the full viewport / scissor for rendering into the render target.
	     *
	     * @type {boolean}
	     */
	    this.fullSizeClearRect = true;
	    /**
	     * Render passes which need to be executed before this pass.
	     *
	     * @type {RenderPass[]}
	     */
	    this.beforePasses = [];
	    /**
	     * Render passes which need to be executed after this pass.
	     *
	     * @type {RenderPass[]}
	     */
	    this.afterPasses = [];
	    Debug.assert(graphicsDevice);
	    this.device = graphicsDevice;
	  }
	  var _proto = RenderPass.prototype;
	  /**
	   * @param {new Function("modulePath", "return import(modulePath)")('../graphics/render-target.js').RenderTarget|null} [renderTarget] - The render
	   * target to render into (output). This function should be called only for render passes which
	   * use render target, or passes which render directly into the default framebuffer, in which
	   * case a null or undefined render target is expected.
	   */
	  _proto.init = function init(renderTarget, options) {
	    var _renderTarget$_colorB;
	    if (renderTarget === void 0) {
	      renderTarget = null;
	    }
	    if (options === void 0) {
	      options = null;
	    }
	    this.options = options;

	    // null represents the default framebuffer
	    this.renderTarget = renderTarget;

	    // defaults depend on multisampling
	    this.samples = Math.max(this.renderTarget ? this.renderTarget.samples : this.device.samples, 1);

	    // allocate ops only when render target is used
	    this.depthStencilOps = new DepthStencilAttachmentOps();
	    var numColorOps = renderTarget ? (_renderTarget$_colorB = renderTarget._colorBuffers) == null ? void 0 : _renderTarget$_colorB.length : 1;
	    this.colorArrayOps.length = 0;
	    for (var i = 0; i < numColorOps; i++) {
	      var _this$renderTarget;
	      var colorOps = new ColorAttachmentOps();
	      this.colorArrayOps[i] = colorOps;

	      // if rendering to single-sampled buffer, this buffer needs to be stored
	      if (this.samples === 1) {
	        colorOps.store = true;
	        colorOps.resolve = false;
	      }

	      // if render target needs mipmaps
	      if ((_this$renderTarget = this.renderTarget) != null && (_this$renderTarget = _this$renderTarget._colorBuffers) != null && _this$renderTarget[i].mipmaps) {
	        colorOps.mipmaps = true;
	      }
	    }
	    this.postInit();
	  };
	  _proto.destroy = function destroy() {};
	  _proto.postInit = function postInit() {};
	  _proto.frameUpdate = function frameUpdate() {
	    // resize the render target if needed
	    if (this._options && this.renderTarget) {
	      var _this$_options$resize;
	      var resizeSource = (_this$_options$resize = this._options.resizeSource) != null ? _this$_options$resize : this.device.backBuffer;
	      var width = Math.floor(resizeSource.width * this._options.scaleX);
	      var height = Math.floor(resizeSource.height * this._options.scaleY);
	      this.renderTarget.resize(width, height);
	    }
	  };
	  _proto.before = function before() {};
	  _proto.execute = function execute() {};
	  _proto.after = function after() {};
	  _proto.onEnable = function onEnable() {};
	  _proto.onDisable = function onDisable() {};
	  /**
	   * Mark render pass as clearing the full color buffer.
	   *
	   * @param {Color|undefined} color - The color to clear to, or undefined to preserve the existing
	   * content.
	   */
	  _proto.setClearColor = function setClearColor(color) {
	    // in case of MRT, we clear all color buffers.
	    // TODO: expose per color buffer clear parameters on the camera, and copy them here.
	    var count = this.colorArrayOps.length;
	    for (var i = 0; i < count; i++) {
	      var colorOps = this.colorArrayOps[i];
	      if (color) colorOps.clearValue.copy(color);
	      colorOps.clear = !!color;
	    }
	  }

	  /**
	   * Mark render pass as clearing the full depth buffer.
	   *
	   * @param {number|undefined} depthValue - The depth value to clear to, or undefined to preserve
	   * the existing content.
	   */;
	  _proto.setClearDepth = function setClearDepth(depthValue) {
	    if (depthValue) this.depthStencilOps.clearDepthValue = depthValue;
	    this.depthStencilOps.clearDepth = depthValue !== undefined;
	  }

	  /**
	   * Mark render pass as clearing the full stencil buffer.
	   *
	   * @param {number|undefined} stencilValue - The stencil value to clear to, or undefined to preserve the
	   * existing content.
	   */;
	  _proto.setClearStencil = function setClearStencil(stencilValue) {
	    if (stencilValue) this.depthStencilOps.clearStencilValue = stencilValue;
	    this.depthStencilOps.clearStencil = stencilValue !== undefined;
	  }

	  /**
	   * Render the render pass
	   */;
	  _proto.render = function render() {
	    var _this = this;
	    if (this.enabled) {
	      var device = this.device;
	      var realPass = this.renderTarget !== undefined;
	      Debug.call(function () {
	        _this.log(device, device.renderPassIndex);
	      });
	      this.before();
	      if (this.executeEnabled) {
	        if (realPass) {
	          device.startRenderPass(this);
	        }
	        this.execute();
	        if (realPass) {
	          device.endRenderPass(this);
	        }
	      }
	      this.after();
	      device.renderPassIndex++;
	    }
	  };
	  _proto.log = function log(device, index) {
	    if (Tracing.get(TRACEID_RENDER_PASS) || Tracing.get(TRACEID_RENDER_PASS_DETAIL)) {
	      var _this$renderTarget2, _rt$_colorBuffers$len, _rt$_colorBuffers;
	      var rt = (_this$renderTarget2 = this.renderTarget) != null ? _this$renderTarget2 : this.renderTarget === null ? device.backBuffer : null;
	      var isBackBuffer = !!(rt != null && rt.impl.assignedColorTexture) || (rt == null ? void 0 : rt.impl.suppliedColorFramebuffer) !== undefined;
	      var numColor = (_rt$_colorBuffers$len = rt == null || (_rt$_colorBuffers = rt._colorBuffers) == null ? void 0 : _rt$_colorBuffers.length) != null ? _rt$_colorBuffers$len : isBackBuffer ? 1 : 0;
	      var hasDepth = rt == null ? void 0 : rt.depth;
	      var hasStencil = rt == null ? void 0 : rt.stencil;
	      var rtInfo = !rt ? '' : " RT: " + (rt ? rt.name : 'NULL') + " " + ("" + (numColor > 0 ? "[Color" + (numColor > 1 ? " x " + numColor : '') + "]" : '')) + ("" + (hasDepth ? '[Depth]' : '')) + ("" + (hasStencil ? '[Stencil]' : '')) + (" " + rt.width + " x " + rt.height) + ("" + (this.samples > 0 ? ' samples: ' + this.samples : ''));
	      Debug.trace(TRACEID_RENDER_PASS, index.toString().padEnd(2, ' ') + ": " + this.name.padEnd(20, ' ') + ("" + (this.executeEnabled ? '' : ' DISABLED ')) + rtInfo.padEnd(30));
	      for (var i = 0; i < numColor; i++) {
	        var colorOps = this.colorArrayOps[i];
	        Debug.trace(TRACEID_RENDER_PASS_DETAIL, "    color[" + i + "]: " + ((colorOps.clear ? 'clear' : 'load') + "->") + ((colorOps.store ? 'store' : 'discard') + " ") + ("" + (colorOps.resolve ? 'resolve ' : '')) + ("" + (colorOps.mipmaps ? 'mipmaps ' : '')));
	      }
	      if (this.depthStencilOps) {
	        if (hasDepth) {
	          Debug.trace(TRACEID_RENDER_PASS_DETAIL, "    depthOps: " + ((this.depthStencilOps.clearDepth ? 'clear' : 'load') + "->") + ("" + (this.depthStencilOps.storeDepth ? 'store' : 'discard')));
	        }
	        if (hasStencil) {
	          Debug.trace(TRACEID_RENDER_PASS_DETAIL, "    stencOps: " + ((this.depthStencilOps.clearStencil ? 'clear' : 'load') + "->") + ("" + (this.depthStencilOps.storeStencil ? 'store' : 'discard')));
	        }
	      }
	    }
	  };
	  _createClass(RenderPass, [{
	    key: "colorOps",
	    get:
	    /**
	     * Color attachment operations for the first color attachment.
	     *
	     * @type {ColorAttachmentOps}
	     */
	    function get() {
	      return this.colorArrayOps[0];
	    }
	  }, {
	    key: "name",
	    get: function get() {
	      if (!this._name) this._name = this.constructor.name;
	      return this._name;
	    },
	    set: function set(value) {
	      this._name = value;
	    }
	  }, {
	    key: "options",
	    get: function get() {
	      return this._options;
	    },
	    set: function set(value) {
	      this._options = value;

	      // sanitize options
	      if (value) {
	        var _this$_options$scaleX, _this$_options$scaleY;
	        this._options.scaleX = (_this$_options$scaleX = this._options.scaleX) != null ? _this$_options$scaleX : 1;
	        this._options.scaleY = (_this$_options$scaleY = this._options.scaleY) != null ? _this$_options$scaleY : 1;
	      }
	    }
	  }, {
	    key: "enabled",
	    get: function get() {
	      return this._enabled;
	    },
	    set: function set(value) {
	      if (this._enabled !== value) {
	        this._enabled = value;
	        if (value) {
	          this.onEnable();
	        } else {
	          this.onDisable();
	        }
	      }
	    }
	  }]);
	  return RenderPass;
	}();

	/**
	 * Options to drive shader processing to add support for bind groups and uniform buffers.
	 *
	 * @ignore
	 */
	var ShaderProcessorOptions = /*#__PURE__*/function () {
	  /**
	   * Constructs shader processing options, used to process the shader for uniform buffer support.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./uniform-buffer-format.js').UniformBufferFormat} [viewUniformFormat] - Format
	   * of the uniform buffer.
	   * @param {new Function("modulePath", "return import(modulePath)")('./bind-group-format.js').BindGroupFormat} [viewBindGroupFormat] - Format of
	   * the bind group.
	   * @param {new Function("modulePath", "return import(modulePath)")('./vertex-format.js').VertexFormat} [vertexFormat] - Format of the vertex
	   * buffer.
	   */
	  function ShaderProcessorOptions(viewUniformFormat, viewBindGroupFormat, vertexFormat) {
	    /** @type {new Function("modulePath", "return import(modulePath)")('./uniform-buffer-format.js').UniformBufferFormat[]} */
	    this.uniformFormats = [];
	    /** @type {new Function("modulePath", "return import(modulePath)")('./bind-group-format.js').BindGroupFormat[]} */
	    this.bindGroupFormats = [];
	    /** @type {new Function("modulePath", "return import(modulePath)")('./vertex-format.js').VertexFormat[]} */
	    this.vertexFormat = void 0;
	    // construct a sparse array
	    this.uniformFormats[BINDGROUP_VIEW] = viewUniformFormat;
	    this.bindGroupFormats[BINDGROUP_VIEW] = viewBindGroupFormat;
	    this.vertexFormat = vertexFormat;
	  }

	  /**
	   * Get the bind group index for the uniform name.
	   *
	   * @param {string} name - The name of the uniform.
	   * @returns {boolean} - Returns true if the uniform exists, false otherwise.
	   */
	  var _proto = ShaderProcessorOptions.prototype;
	  _proto.hasUniform = function hasUniform(name) {
	    for (var i = 0; i < this.uniformFormats.length; i++) {
	      var uniformFormat = this.uniformFormats[i];
	      if (uniformFormat != null && uniformFormat.get(name)) {
	        return true;
	      }
	    }
	    return false;
	  }

	  /**
	   * Get the bind group texture slot for the texture uniform name.
	   *
	   * @param {string} name - The name of the texture uniform.
	   * @returns {boolean} - Returns true if the texture uniform exists, false otherwise.
	   */;
	  _proto.hasTexture = function hasTexture(name) {
	    for (var i = 0; i < this.bindGroupFormats.length; i++) {
	      var groupFormat = this.bindGroupFormats[i];
	      if (groupFormat != null && groupFormat.getTexture(name)) {
	        return true;
	      }
	    }
	    return false;
	  };
	  _proto.getVertexElement = function getVertexElement(semantic) {
	    var _this$vertexFormat;
	    return (_this$vertexFormat = this.vertexFormat) == null ? void 0 : _this$vertexFormat.elements.find(function (element) {
	      return element.name === semantic;
	    });
	  }

	  /**
	   * Generate unique key representing the processing options.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./graphics-device.js').GraphicsDevice} device - The device.
	   * @returns {string} - Returns the key.
	   */;
	  _proto.generateKey = function generateKey(device) {
	    // TODO: Optimize. Uniform and BindGroup formats should have their keys evaluated in their
	    // constructors, and here we should simply concatenate those.
	    var key = JSON.stringify(this.uniformFormats) + JSON.stringify(this.bindGroupFormats);

	    // WebGPU shaders are processed per vertex format
	    if (device.isWebGPU) {
	      var _this$vertexFormat2;
	      key += (_this$vertexFormat2 = this.vertexFormat) == null ? void 0 : _this$vertexFormat2.shaderProcessingHashString;
	    }
	    return key;
	  };
	  return ShaderProcessorOptions;
	}();

	/**
	 * This object allows you to configure and use the transform feedback feature (WebGL2 only). How to
	 * use:
	 *
	 * 1. First, check that you're on WebGL2, by looking at the `app.graphicsDevice.isWebGL2`` value.
	 * 2. Define the outputs in your vertex shader. The syntax is `out vec3 out_vertex_position`,
	 * note that there must be out_ in the name. You can then simply assign values to these outputs in
	 * VS. The order and size of shader outputs must match the output buffer layout.
	 * 3. Create the shader using `TransformFeedback.createShader(device, vsCode, yourShaderName)`.
	 * 4. Create/acquire the input vertex buffer. Can be any VertexBuffer, either manually created, or
	 * from a Mesh.
	 * 5. Create the TransformFeedback object: `const tf = new TransformFeedback(inputBuffer)`. This
	 * object will internally create an output buffer.
	 * 6. Run the shader: `tf.process(shader)`. Shader will take the input buffer, process it and write
	 * to the output buffer, then the input/output buffers will be automatically swapped, so you'll
	 * immediately see the result.
	 *
	 * ```javascript
	 * // *** shader asset ***
	 * attribute vec3 vertex_position;
	 * attribute vec3 vertex_normal;
	 * attribute vec2 vertex_texCoord0;
	 * out vec3 out_vertex_position;
	 * out vec3 out_vertex_normal;
	 * out vec2 out_vertex_texCoord0;
	 * void main(void) {
	 *     // read position and normal, write new position (push away)
	 *     out_vertex_position = vertex_position + vertex_normal * 0.01;
	 *     // pass other attributes unchanged
	 *     out_vertex_normal = vertex_normal;
	 *     out_vertex_texCoord0 = vertex_texCoord0;
	 * }
	 * ```
	 *
	 * ```javascript
	 * // *** script asset ***
	 * var TransformExample = pc.createScript('transformExample');
	 *
	 * // attribute that references shader asset and material
	 * TransformExample.attributes.add('shaderCode', { type: 'asset', assetType: 'shader' });
	 * TransformExample.attributes.add('material', { type: 'asset', assetType: 'material' });
	 *
	 * TransformExample.prototype.initialize = function() {
	 *     const device = this.app.graphicsDevice;
	 *     const mesh = pc.createTorus(device, { tubeRadius: 0.01, ringRadius: 3 });
	 *     const meshInstance = new pc.MeshInstance(mesh, this.material.resource);
	 *     const entity = new pc.Entity();
	 *     entity.addComponent('render', {
	 *         type: 'asset',
	 *         meshInstances: [meshInstance]
	 *     });
	 *     app.root.addChild(entity);
	 *
	 *     // if webgl2 is not supported, transform-feedback is not available
	 *     if (!device.isWebGL2) return;
	 *     const inputBuffer = mesh.vertexBuffer;
	 *     this.tf = new pc.TransformFeedback(inputBuffer);
	 *     this.shader = pc.TransformFeedback.createShader(device, this.shaderCode.resource, "tfMoveUp");
	 * };
	 *
	 * TransformExample.prototype.update = function(dt) {
	 *     if (!this.app.graphicsDevice.isWebGL2) return;
	 *     this.tf.process(this.shader);
	 * };
	 * ```
	 *
	 * @category Graphics
	 */
	var TransformFeedback = /*#__PURE__*/function () {
	  /**
	   * Create a new TransformFeedback instance.
	   *
	   * @param {VertexBuffer} inputBuffer - The input vertex buffer.
	   * @param {number} [usage] - The optional usage type of the output vertex buffer. Can be:
	   *
	   * - {@link BUFFER_STATIC}
	   * - {@link BUFFER_DYNAMIC}
	   * - {@link BUFFER_STREAM}
	   * - {@link BUFFER_GPUDYNAMIC}
	   *
	   * Defaults to {@link BUFFER_GPUDYNAMIC} (which is recommended for continuous update).
	   */
	  function TransformFeedback(inputBuffer, usage) {
	    if (usage === void 0) {
	      usage = BUFFER_GPUDYNAMIC;
	    }
	    this.device = inputBuffer.device;
	    var gl = this.device.gl;
	    Debug.assert(inputBuffer.format.interleaved || inputBuffer.format.elements.length <= 1, "Vertex buffer used by TransformFeedback needs to be interleaved.");
	    this._inputBuffer = inputBuffer;
	    if (usage === BUFFER_GPUDYNAMIC && inputBuffer.usage !== usage) {
	      // have to recreate input buffer with other usage
	      gl.bindBuffer(gl.ARRAY_BUFFER, inputBuffer.impl.bufferId);
	      gl.bufferData(gl.ARRAY_BUFFER, inputBuffer.storage, gl.DYNAMIC_COPY);
	    }
	    this._outputBuffer = new VertexBuffer(inputBuffer.device, inputBuffer.format, inputBuffer.numVertices, usage, inputBuffer.storage);
	  }

	  /**
	   * Creates a transform feedback ready vertex shader from code.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./graphics-device.js').GraphicsDevice} graphicsDevice - The graphics device
	   * used by the renderer.
	   * @param {string} vertexCode - Vertex shader code. Should contain output variables starting with "out_".
	   * @param {string} name - Unique name for caching the shader.
	   * @returns {Shader} A shader to use in the process() function.
	   */
	  TransformFeedback.createShader = function createShader(graphicsDevice, vertexCode, name) {
	    return new Shader(graphicsDevice, ShaderUtils.createDefinition(graphicsDevice, {
	      name: name,
	      vertexCode: vertexCode,
	      useTransformFeedback: true
	    }));
	  }

	  /**
	   * Destroys the transform feedback helper object.
	   */;
	  var _proto = TransformFeedback.prototype;
	  _proto.destroy = function destroy() {
	    this._outputBuffer.destroy();
	  }

	  /**
	   * Runs the specified shader on the input buffer, writes results into the new buffer, then
	   * optionally swaps input/output.
	   *
	   * @param {Shader} shader - A vertex shader to run. Should be created with
	   * {@link TransformFeedback.createShader}.
	   * @param {boolean} [swap] - Swap input/output buffer data. Useful for continuous buffer
	   * processing. Default is true.
	   */;
	  _proto.process = function process(shader, swap) {
	    if (swap === void 0) {
	      swap = true;
	    }
	    var device = this.device;
	    DebugGraphics.pushGpuMarker(device, "TransformFeedback");
	    var oldRt = device.getRenderTarget();
	    device.setRenderTarget(null);
	    device.updateBegin();
	    device.setVertexBuffer(this._inputBuffer, 0);
	    device.setRaster(false);
	    device.setTransformFeedbackBuffer(this._outputBuffer);
	    device.setShader(shader);
	    device.draw({
	      type: PRIMITIVE_POINTS,
	      base: 0,
	      count: this._inputBuffer.numVertices,
	      indexed: false
	    });
	    device.setTransformFeedbackBuffer(null);
	    device.setRaster(true);
	    device.updateEnd();
	    device.setRenderTarget(oldRt);
	    DebugGraphics.popGpuMarker(device);

	    // swap buffers
	    if (swap) {
	      var tmp = this._inputBuffer.impl.bufferId;
	      this._inputBuffer.impl.bufferId = this._outputBuffer.impl.bufferId;
	      this._outputBuffer.impl.bufferId = tmp;

	      // swap VAO
	      tmp = this._inputBuffer.impl.vao;
	      this._inputBuffer.impl.vao = this._outputBuffer.impl.vao;
	      this._outputBuffer.impl.vao = tmp;
	    }
	  }

	  /**
	   * The current input buffer.
	   *
	   * @type {VertexBuffer}
	   */;
	  _createClass(TransformFeedback, [{
	    key: "inputBuffer",
	    get: function get() {
	      return this._inputBuffer;
	    }

	    /**
	     * The current output buffer.
	     *
	     * @type {VertexBuffer}
	     */
	  }, {
	    key: "outputBuffer",
	    get: function get() {
	      return this._outputBuffer;
	    }
	  }]);
	  return TransformFeedback;
	}();

	function set1(a) {
	  this.array[this.index] = a;
	}
	function set2(a, b) {
	  this.array[this.index] = a;
	  this.array[this.index + 1] = b;
	}
	function set3(a, b, c) {
	  this.array[this.index] = a;
	  this.array[this.index + 1] = b;
	  this.array[this.index + 2] = c;
	}
	function set4(a, b, c, d) {
	  this.array[this.index] = a;
	  this.array[this.index + 1] = b;
	  this.array[this.index + 2] = c;
	  this.array[this.index + 3] = d;
	}
	function arraySet1(index, inputArray, inputIndex) {
	  this.array[index] = inputArray[inputIndex];
	}
	function arraySet2(index, inputArray, inputIndex) {
	  this.array[index] = inputArray[inputIndex];
	  this.array[index + 1] = inputArray[inputIndex + 1];
	}
	function arraySet3(index, inputArray, inputIndex) {
	  this.array[index] = inputArray[inputIndex];
	  this.array[index + 1] = inputArray[inputIndex + 1];
	  this.array[index + 2] = inputArray[inputIndex + 2];
	}
	function arraySet4(index, inputArray, inputIndex) {
	  this.array[index] = inputArray[inputIndex];
	  this.array[index + 1] = inputArray[inputIndex + 1];
	  this.array[index + 2] = inputArray[inputIndex + 2];
	  this.array[index + 3] = inputArray[inputIndex + 3];
	}
	function arrayGet1(offset, outputArray, outputIndex) {
	  outputArray[outputIndex] = this.array[offset];
	}
	function arrayGet2(offset, outputArray, outputIndex) {
	  outputArray[outputIndex] = this.array[offset];
	  outputArray[outputIndex + 1] = this.array[offset + 1];
	}
	function arrayGet3(offset, outputArray, outputIndex) {
	  outputArray[outputIndex] = this.array[offset];
	  outputArray[outputIndex + 1] = this.array[offset + 1];
	  outputArray[outputIndex + 2] = this.array[offset + 2];
	}
	function arrayGet4(offset, outputArray, outputIndex) {
	  outputArray[outputIndex] = this.array[offset];
	  outputArray[outputIndex + 1] = this.array[offset + 1];
	  outputArray[outputIndex + 2] = this.array[offset + 2];
	  outputArray[outputIndex + 3] = this.array[offset + 3];
	}

	/**
	 * Helps with accessing a specific vertex attribute.
	 *
	 * @category Graphics
	 * @ignore
	 */
	var VertexIteratorAccessor = /*#__PURE__*/function () {
	  /**
	   * Create a new VertexIteratorAccessor instance.
	   *
	   * @param {ArrayBuffer} buffer - The vertex buffer containing the attribute to be accessed.
	   * @param {object} vertexElement - The vertex attribute to be accessed.
	   * @param {string} vertexElement.name - The meaning of the vertex element. This is used to link
	   * the vertex data to a shader input. Can be:
	   *
	   * - {@link SEMANTIC_POSITION}
	   * - {@link SEMANTIC_NORMAL}
	   * - {@link SEMANTIC_TANGENT}
	   * - {@link SEMANTIC_BLENDWEIGHT}
	   * - {@link SEMANTIC_BLENDINDICES}
	   * - {@link SEMANTIC_COLOR}
	   * - {@link SEMANTIC_TEXCOORD0}
	   * - {@link SEMANTIC_TEXCOORD1}
	   * - {@link SEMANTIC_TEXCOORD2}
	   * - {@link SEMANTIC_TEXCOORD3}
	   * - {@link SEMANTIC_TEXCOORD4}
	   * - {@link SEMANTIC_TEXCOORD5}
	   * - {@link SEMANTIC_TEXCOORD6}
	   * - {@link SEMANTIC_TEXCOORD7}
	   *
	   * If vertex data has a meaning other that one of those listed above, use the user-defined
	   * semantics: {@link SEMANTIC_ATTR0} to {@link SEMANTIC_ATTR15}.
	   * @param {number} vertexElement.numComponents - The number of components of the vertex
	   * attribute. Can be 1, 2, 3 or 4.
	   * @param {number} vertexElement.dataType - The data type of the attribute. Can be:
	   *
	   * - {@link TYPE_INT8}
	   * - {@link TYPE_UINT8}
	   * - {@link TYPE_INT16}
	   * - {@link TYPE_UINT16}
	   * - {@link TYPE_INT32}
	   * - {@link TYPE_UINT32}
	   * - {@link TYPE_FLOAT32}
	   * @param {boolean} vertexElement.normalize - If true, vertex attribute data will be mapped
	   * from a 0 to 255 range down to 0 to 1 when fed to a shader. If false, vertex attribute data
	   * is left unchanged. If this property is unspecified, false is assumed.
	   * @param {number} vertexElement.offset - The number of initial bytes at the start of a vertex
	   * that are not relevant to this attribute.
	   * @param {number} vertexElement.stride - The number of total bytes that are between the start
	   * of one vertex, and the start of the next.
	   * @param {new Function("modulePath", "return import(modulePath)")('./scope-id.js').ScopeId} vertexElement.scopeId - The shader input variable
	   * corresponding to the attribute.
	   * @param {number} vertexElement.size - The size of the attribute in bytes.
	   * @param {new Function("modulePath", "return import(modulePath)")('./vertex-format.js').VertexFormat} vertexFormat - A vertex format that
	   * defines the layout of vertex data inside the buffer.
	   */
	  function VertexIteratorAccessor(buffer, vertexElement, vertexFormat) {
	    this.index = 0;
	    this.numComponents = vertexElement.numComponents;

	    // create the typed array based on the element data type
	    if (vertexFormat.interleaved) {
	      this.array = new typedArrayTypes[vertexElement.dataType](buffer, vertexElement.offset);
	    } else {
	      this.array = new typedArrayTypes[vertexElement.dataType](buffer, vertexElement.offset, vertexFormat.vertexCount * vertexElement.numComponents);
	    }

	    // BYTES_PER_ELEMENT is on the instance and constructor for Chrome, Safari and Firefox, but just the constructor for Opera
	    this.stride = vertexElement.stride / this.array.constructor.BYTES_PER_ELEMENT;

	    // Methods
	    switch (vertexElement.numComponents) {
	      case 1:
	        this.set = set1;
	        this.getToArray = arrayGet1;
	        this.setFromArray = arraySet1;
	        break;
	      case 2:
	        this.set = set2;
	        this.getToArray = arrayGet2;
	        this.setFromArray = arraySet2;
	        break;
	      case 3:
	        this.set = set3;
	        this.getToArray = arrayGet3;
	        this.setFromArray = arraySet3;
	        break;
	      case 4:
	        this.set = set4;
	        this.getToArray = arrayGet4;
	        this.setFromArray = arraySet4;
	        break;
	    }
	  }

	  /**
	   * Get a attribute component at the iterator's current index.
	   *
	   * @param {number} offset - The component offset. Should be either 0, 1, 2, or 3.
	   * @returns {number} The value of a attribute component.
	   */
	  var _proto = VertexIteratorAccessor.prototype;
	  _proto.get = function get(offset) {
	    return this.array[this.index + offset];
	  }

	  /**
	   * Set all the attribute components at the iterator's current index.
	   *
	   * @param {number} a - The first component value.
	   * @param {number} [b] - The second component value (if applicable).
	   * @param {number} [c] - The third component value (if applicable).
	   * @param {number} [d] - The fourth component value (if applicable).
	   */;
	  _proto.set = function set(a, b, c, d) {
	    // Will be replaced with specialized implementation based on number of components
	  }

	  /**
	   * Read attribute components to an output array.
	   *
	   * @param {number} offset - The component offset at which to read data from the buffer. Will be
	   * used instead of the iterator's current index.
	   * @param {number[]|Int8Array|Uint8Array|Uint8ClampedArray|Int16Array|Uint16Array|Int32Array|Uint32Array|Float32Array} outputArray - The output array to write data into.
	   * @param {number} outputIndex - The output index at which to write into the output array.
	   */;
	  _proto.getToArray = function getToArray(offset, outputArray, outputIndex) {
	    // Will be replaced with specialized implementation based on number of components
	  }

	  /**
	   * Write attribute components from an input array.
	   *
	   * @param {number} index - The starting index at which to write data into the buffer. Will be
	   * used instead of the iterator's current index.
	   * @param {number[]|Int8Array|Uint8Array|Uint8ClampedArray|Int16Array|Uint16Array|Int32Array|Uint32Array|Float32Array} inputArray - The input array to read data from.
	   * @param {number} inputIndex - The input index at which to read from the input array.
	   */;
	  _proto.setFromArray = function setFromArray(index, inputArray, inputIndex) {
	    // Will be replaced with specialized implementation based on number of components
	  };
	  return VertexIteratorAccessor;
	}();
	/**
	 * A vertex iterator simplifies the process of writing vertex data to a vertex buffer.
	 *
	 * @category Graphics
	 */
	var VertexIterator = /*#__PURE__*/function () {
	  /**
	   * Create a new VertexIterator instance.
	   *
	   * @param {new Function("modulePath", "return import(modulePath)")('./vertex-buffer.js').VertexBuffer} vertexBuffer - The vertex buffer to be
	   * iterated.
	   */
	  function VertexIterator(vertexBuffer) {
	    // Store the vertex buffer
	    this.vertexBuffer = vertexBuffer;
	    this.vertexFormatSize = vertexBuffer.getFormat().size;

	    // Lock the vertex buffer
	    this.buffer = this.vertexBuffer.lock();

	    // Create an empty list
	    this.accessors = [];

	    /**
	     * The vertex buffer elements.
	     *
	     * @type {Object<string, VertexIteratorAccessor>}
	     */
	    this.element = {};

	    // Add a new 'setter' function for each element
	    var vertexFormat = this.vertexBuffer.getFormat();
	    for (var i = 0; i < vertexFormat.elements.length; i++) {
	      var vertexElement = vertexFormat.elements[i];
	      this.accessors[i] = new VertexIteratorAccessor(this.buffer, vertexElement, vertexFormat);
	      this.element[vertexElement.name] = this.accessors[i];
	    }
	  }

	  /**
	   * Moves the vertex iterator on to the next vertex.
	   *
	   * @param {number} [count] - Optional number of steps to move on when calling next. Defaults to
	   * 1.
	   * @example
	   * const iterator = new pc.VertexIterator(vertexBuffer);
	   * iterator.element[pc.SEMANTIC_POSITION].set(-0.9, -0.9, 0.0);
	   * iterator.element[pc.SEMANTIC_COLOR].set(255, 0, 0, 255);
	   * iterator.next();
	   * iterator.element[pc.SEMANTIC_POSITION].set(0.9, -0.9, 0.0);
	   * iterator.element[pc.SEMANTIC_COLOR].set(0, 255, 0, 255);
	   * iterator.next();
	   * iterator.element[pc.SEMANTIC_POSITION].set(0.0, 0.9, 0.0);
	   * iterator.element[pc.SEMANTIC_COLOR].set(0, 0, 255, 255);
	   * iterator.end();
	   */
	  var _proto2 = VertexIterator.prototype;
	  _proto2.next = function next(count) {
	    if (count === void 0) {
	      count = 1;
	    }
	    var i = 0;
	    var accessors = this.accessors;
	    var numAccessors = this.accessors.length;
	    while (i < numAccessors) {
	      var accessor = accessors[i++];
	      accessor.index += count * accessor.stride;
	    }
	  }

	  /**
	   * Notifies the vertex buffer being iterated that writes are complete. Internally the vertex
	   * buffer is unlocked and vertex data is uploaded to video memory.
	   *
	   * @example
	   * const iterator = new pc.VertexIterator(vertexBuffer);
	   * iterator.element[pc.SEMANTIC_POSITION].set(-0.9, -0.9, 0.0);
	   * iterator.element[pc.SEMANTIC_COLOR].set(255, 0, 0, 255);
	   * iterator.next();
	   * iterator.element[pc.SEMANTIC_POSITION].set(0.9, -0.9, 0.0);
	   * iterator.element[pc.SEMANTIC_COLOR].set(0, 255, 0, 255);
	   * iterator.next();
	   * iterator.element[pc.SEMANTIC_POSITION].set(0.0, 0.9, 0.0);
	   * iterator.element[pc.SEMANTIC_COLOR].set(0, 0, 255, 255);
	   * iterator.end();
	   */;
	  _proto2.end = function end() {
	    // Unlock the vertex buffer
	    this.vertexBuffer.unlock();
	  }

	  /**
	   * Copies data for specified semantic into vertex buffer. Works with both interleaved (slower)
	   * and non-interleaved (fast) vertex buffers.
	   *
	   * @param {string} semantic - The semantic of the vertex element to set.
	   * @param {number[]|Int8Array|Uint8Array|Uint8ClampedArray|Int16Array|Uint16Array|Int32Array|Uint32Array|Float32Array} data - The data to set.
	   * @param {number} numVertices - The number of vertices to write.
	   * @ignore
	   */;
	  _proto2.writeData = function writeData(semantic, data, numVertices) {
	    var element = this.element[semantic];
	    if (element) {
	      if (numVertices > this.vertexBuffer.numVertices) {
	        Debug.error("NumVertices provided to setData: " + numVertices + " is larger than space in VertexBuffer: " + this.vertexBuffer.numVertices);

	        // avoid overwrite
	        numVertices = this.vertexBuffer.numVertices;
	      }
	      var numComponents = element.numComponents;

	      // copy data to interleaved buffer by looping over vertices and copying them manually
	      if (this.vertexBuffer.getFormat().interleaved) {
	        var index = 0;
	        for (var i = 0; i < numVertices; i++) {
	          element.setFromArray(index, data, i * numComponents);
	          index += element.stride;
	        }
	      } else {
	        // non-interleaved copy

	        // if data contains more  data than needed, copy from its subarray
	        if (data.length > numVertices * numComponents) {
	          var copyCount = numVertices * numComponents;

	          // if data is typed array
	          if (ArrayBuffer.isView(data)) {
	            data = data.subarray(0, copyCount);
	            element.array.set(data);
	          } else {
	            // data is array, copy right amount manually
	            for (var _i = 0; _i < copyCount; _i++) element.array[_i] = data[_i];
	          }
	        } else {
	          // copy whole data
	          element.array.set(data);
	        }
	      }
	    }
	  }

	  /**
	   * Function to extract elements of a specified semantic from vertex buffer into flat array
	   * (data). Works with both interleaved (slower) and non-interleaved (fast) vertex buffers.
	   * Returns number of vertices. Note: when data is a typed array and is smaller than needed,
	   * only part of the data gets copied out (typed arrays ignore read/write out of range).
	   *
	   * @param {string} semantic - The semantic of the vertex element to read.
	   * @param {number[]|Int8Array|Uint8Array|Uint8ClampedArray|Int16Array|Uint16Array|Int32Array|Uint32Array|Float32Array} data - The
	   * array to receive the data.
	   * @returns {number} The number of vertices read.
	   * @ignore
	   */;
	  _proto2.readData = function readData(semantic, data) {
	    var element = this.element[semantic];
	    var count = 0;
	    if (element) {
	      count = this.vertexBuffer.numVertices;
	      var i;
	      var numComponents = element.numComponents;
	      if (this.vertexBuffer.getFormat().interleaved) {
	        // extract data from interleaved buffer by looping over vertices and copying them manually
	        if (Array.isArray(data)) data.length = 0;
	        element.index = 0;
	        var offset = 0;
	        for (i = 0; i < count; i++) {
	          element.getToArray(offset, data, i * numComponents);
	          offset += element.stride;
	        }
	      } else {
	        if (ArrayBuffer.isView(data)) {
	          // destination data is typed array
	          data.set(element.array);
	        } else {
	          // destination data is array
	          data.length = 0;
	          var copyCount = count * numComponents;
	          for (i = 0; i < copyCount; i++) data[i] = element.array[i];
	        }
	      }
	    }
	    return count;
	  };
	  return VertexIterator;
	}();

	var ACTION_MOUSE = 'mouse';
	var ACTION_KEYBOARD = 'keyboard';
	var ACTION_GAMEPAD = 'gamepad';
	var AXIS_MOUSE_X = 'mousex';
	var AXIS_MOUSE_Y = 'mousey';
	var AXIS_PAD_L_X = 'padlx';
	var AXIS_PAD_L_Y = 'padly';
	var AXIS_PAD_R_X = 'padrx';
	var AXIS_PAD_R_Y = 'padry';
	var AXIS_KEY = 'key';

	/**
	 * Name of event fired when a key is pressed.
	 *
	 * @type {string}
	 * @category Input
	 */
	var EVENT_KEYDOWN = 'keydown';

	/**
	 * Name of event fired when a key is released.
	 *
	 * @type {string}
	 * @category Input
	 */
	var EVENT_KEYUP = 'keyup';

	/**
	 * Name of event fired when a mouse button is pressed.
	 *
	 * @type {string}
	 * @category Input
	 */
	var EVENT_MOUSEDOWN = 'mousedown';

	/**
	 * Name of event fired when the mouse is moved.
	 *
	 * @type {string}
	 * @category Input
	 */
	var EVENT_MOUSEMOVE = 'mousemove';

	/**
	 * Name of event fired when a mouse button is released.
	 *
	 * @type {string}
	 * @category Input
	 */
	var EVENT_MOUSEUP = 'mouseup';

	/**
	 * Name of event fired when the mouse wheel is rotated.
	 *
	 * @type {string}
	 * @category Input
	 */
	var EVENT_MOUSEWHEEL = 'mousewheel';

	/**
	 * Name of event fired when a new touch occurs. For example, a finger is placed on the device.
	 *
	 * @type {string}
	 * @category Input
	 */
	var EVENT_TOUCHSTART = 'touchstart';

	/**
	 * Name of event fired when touch ends. For example, a finger is lifted off the device.
	 *
	 * @type {string}
	 * @category Input
	 */
	var EVENT_TOUCHEND = 'touchend';

	/**
	 * Name of event fired when a touch moves.
	 *
	 * @type {string}
	 * @category Input
	 */
	var EVENT_TOUCHMOVE = 'touchmove';

	/**
	 * Name of event fired when a touch point is interrupted in some way. The exact reasons for
	 * canceling a touch can vary from device to device. For example, a modal alert pops up during the
	 * interaction; the touch point leaves the document area, or there are more touch points than the
	 * device supports, in which case the earliest touch point is canceled.
	 *
	 * @type {string}
	 * @category Input
	 */
	var EVENT_TOUCHCANCEL = 'touchcancel';

	/**
	 * Name of event fired when a new xr select occurs. For example, primary trigger was pressed.
	 *
	 * @type {string}
	 * @category Input
	 */
	var EVENT_SELECT = 'select';

	/**
	 * Name of event fired when a new xr select starts. For example, primary trigger is now pressed.
	 *
	 * @type {string}
	 * @category Input
	 */
	var EVENT_SELECTSTART = 'selectstart';

	/**
	 * Name of event fired when xr select ends. For example, a primary trigger is now released.
	 *
	 * @type {string}
	 * @category Input
	 */
	var EVENT_SELECTEND = 'selectend';

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_BACKSPACE = 8;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_TAB = 9;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_RETURN = 13;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_ENTER = 13;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_SHIFT = 16;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_CONTROL = 17;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_ALT = 18;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_PAUSE = 19;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_CAPS_LOCK = 20;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_ESCAPE = 27;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_SPACE = 32;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_PAGE_UP = 33;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_PAGE_DOWN = 34;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_END = 35;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_HOME = 36;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_LEFT = 37;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_UP = 38;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_RIGHT = 39;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_DOWN = 40;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_PRINT_SCREEN = 44;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_INSERT = 45;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_DELETE = 46;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_0 = 48;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_1 = 49;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_2 = 50;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_3 = 51;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_4 = 52;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_5 = 53;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_6 = 54;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_7 = 55;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_8 = 56;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_9 = 57;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_SEMICOLON = 59;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_EQUAL = 61;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_A = 65;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_B = 66;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_C = 67;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_D = 68;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_E = 69;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_F = 70;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_G = 71;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_H = 72;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_I = 73;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_J = 74;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_K = 75;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_L = 76;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_M = 77;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_N = 78;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_O = 79;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_P = 80;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_Q = 81;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_R = 82;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_S = 83;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_T = 84;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_U = 85;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_V = 86;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_W = 87;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_X = 88;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_Y = 89;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_Z = 90;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_WINDOWS = 91;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_CONTEXT_MENU = 93;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_NUMPAD_0 = 96;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_NUMPAD_1 = 97;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_NUMPAD_2 = 98;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_NUMPAD_3 = 99;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_NUMPAD_4 = 100;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_NUMPAD_5 = 101;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_NUMPAD_6 = 102;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_NUMPAD_7 = 103;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_NUMPAD_8 = 104;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_NUMPAD_9 = 105;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_MULTIPLY = 106;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_ADD = 107;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_SEPARATOR = 108;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_SUBTRACT = 109;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_DECIMAL = 110;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_DIVIDE = 111;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_F1 = 112;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_F2 = 113;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_F3 = 114;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_F4 = 115;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_F5 = 116;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_F6 = 117;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_F7 = 118;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_F8 = 119;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_F9 = 120;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_F10 = 121;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_F11 = 122;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_F12 = 123;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_COMMA = 188;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_PERIOD = 190;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_SLASH = 191;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_OPEN_BRACKET = 219;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_BACK_SLASH = 220;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_CLOSE_BRACKET = 221;

	/**
	 * @type {number}
	 * @category Input
	 */
	var KEY_META = 224;

	/**
	 * No mouse buttons pressed.
	 *
	 * @type {number}
	 * @category Input
	 */
	var MOUSEBUTTON_NONE = -1;

	/**
	 * The left mouse button.
	 *
	 * @type {number}
	 * @category Input
	 */
	var MOUSEBUTTON_LEFT = 0;

	/**
	 * The middle mouse button.
	 *
	 * @type {number}
	 * @category Input
	 */
	var MOUSEBUTTON_MIDDLE = 1;

	/**
	 * The right mouse button.
	 *
	 * @type {number}
	 * @category Input
	 */
	var MOUSEBUTTON_RIGHT = 2;

	/**
	 * Index for pad 1.
	 *
	 * @type {number}
	 * @category Input
	 */
	var PAD_1 = 0;

	/**
	 * Index for pad 2.
	 *
	 * @type {number}
	 * @category Input
	 */
	var PAD_2 = 1;

	/**
	 * Index for pad 3.
	 *
	 * @type {number}
	 * @category Input
	 */
	var PAD_3 = 2;

	/**
	 * Index for pad 4.
	 *
	 * @type {number}
	 * @category Input
	 */
	var PAD_4 = 3;

	/**
	 * The first face button, from bottom going clockwise.
	 *
	 * @type {number}
	 * @category Input
	 */
	var PAD_FACE_1 = 0;

	/**
	 * The second face button, from bottom going clockwise.
	 *
	 * @type {number}
	 * @category Input
	 */
	var PAD_FACE_2 = 1;

	/**
	 * The third face button, from bottom going clockwise.
	 *
	 * @type {number}
	 * @category Input
	 */
	var PAD_FACE_3 = 2;

	/**
	 * The fourth face button, from bottom going clockwise.
	 *
	 * @type {number}
	 * @category Input
	 */
	var PAD_FACE_4 = 3;

	/**
	 * The first shoulder button on the left.
	 *
	 * @type {number}
	 * @category Input
	 */
	var PAD_L_SHOULDER_1 = 4;

	/**
	 * The first shoulder button on the right.
	 *
	 * @type {number}
	 * @category Input
	 */
	var PAD_R_SHOULDER_1 = 5;

	/**
	 * The second shoulder button on the left.
	 *
	 * @type {number}
	 * @category Input
	 */
	var PAD_L_SHOULDER_2 = 6;

	/**
	 * The second shoulder button on the right.
	 *
	 * @type {number}
	 * @category Input
	 */
	var PAD_R_SHOULDER_2 = 7;

	/**
	 * The select button.
	 *
	 * @type {number}
	 * @category Input
	 */
	var PAD_SELECT = 8;

	/**
	 * The start button.
	 *
	 * @type {number}
	 * @category Input
	 */
	var PAD_START = 9;

	/**
	 * The button when depressing the left analogue stick.
	 *
	 * @type {number}
	 * @category Input
	 */
	var PAD_L_STICK_BUTTON = 10;

	/**
	 * The button when depressing the right analogue stick.
	 *
	 * @type {number}
	 * @category Input
	 */
	var PAD_R_STICK_BUTTON = 11;

	/**
	 * Direction pad up.
	 *
	 * @type {number}
	 * @category Input
	 */
	var PAD_UP = 12;

	/**
	 * Direction pad down.
	 *
	 * @type {number}
	 * @category Input
	 */
	var PAD_DOWN = 13;

	/**
	 * Direction pad left.
	 *
	 * @type {number}
	 * @category Input
	 */
	var PAD_LEFT = 14;

	/**
	 * Direction pad right.
	 *
	 * @type {number}
	 * @category Input
	 */
	var PAD_RIGHT = 15;

	/**
	 * Vendor specific button.
	 *
	 * @type {number}
	 * @category Input
	 */
	var PAD_VENDOR = 16;

	/**
	 * Horizontal axis on the left analogue stick.
	 *
	 * @type {number}
	 * @category Input
	 */
	var PAD_L_STICK_X = 0;

	/**
	 * Vertical axis on the left analogue stick.
	 *
	 * @type {number}
	 * @category Input
	 */
	var PAD_L_STICK_Y = 1;

	/**
	 * Horizontal axis on the right analogue stick.
	 *
	 * @type {number}
	 * @category Input
	 */
	var PAD_R_STICK_X = 2;

	/**
	 * Vertical axis on the right analogue stick.
	 *
	 * @type {number}
	 * @category Input
	 */
	var PAD_R_STICK_Y = 3;

	/**
	 * Name of event fired when a gamepad connects.
	 *
	 * @type {string}
	 * @category Input
	 */
	var EVENT_GAMEPADCONNECTED = 'gamepadconnected';

	/**
	 * Name of event fired when a gamepad disconnects.
	 *
	 * @type {string}
	 * @category Input
	 */
	var EVENT_GAMEPADDISCONNECTED = 'gamepaddisconnected';

	/**
	 * Horizontal axis on the touchpad of a XR pad.
	 *
	 * @type {number}
	 * @category Input
	 */
	var XRPAD_TOUCHPAD_X = 0;

	/**
	 * Vertical axis on the thouchpad of a XR pad.
	 *
	 * @type {number}
	 * @category Input
	 */
	var XRPAD_TOUCHPAD_Y = 1;

	/**
	 * Horizontal axis on the stick of a XR pad.
	 *
	 * @type {number}
	 * @category Input
	 */
	var XRPAD_STICK_X = 2;

	/**
	 * Vertical axis on the stick of a XR pad.
	 *
	 * @type {number}
	 * @category Input
	 */
	var XRPAD_STICK_Y = 3;

	/**
	 * The button when pressing the XR pad's touchpad.
	 *
	 * @type {number}
	 * @category Input
	 */
	var XRPAD_TOUCHPAD_BUTTON = 2;

	/**
	 * The trigger button from XR pad.
	 *
	 * @type {number}
	 * @category Input
	 */
	var XRPAD_TRIGGER = 0;

	/**
	 * The squeeze button from XR pad.
	 *
	 * @type {number}
	 * @category Input
	 */
	var XRPAD_SQUEEZE = 1;

	/**
	 * The button when pressing the XR pad's stick.
	 *
	 * @type {number}
	 * @category Input
	 */
	var XRPAD_STICK_BUTTON = 3;

	/**
	 * The A button from XR pad.
	 *
	 * @type {number}
	 * @category Input
	 */
	var XRPAD_A = 4;

	/**
	 * The B button from XR pad.
	 *
	 * @type {number}
	 * @category Input
	 */
	var XRPAD_B = 5;

	/**
	 * The KeyboardEvent is passed into all event callbacks from the {@link Keyboard}. It corresponds
	 * to a key press or release.
	 *
	 * @category Input
	 */
	var KeyboardEvent =
	/**
	 * Create a new KeyboardEvent.
	 *
	 * @param {new Function("modulePath", "return import(modulePath)")('./keyboard.js').Keyboard} keyboard - The keyboard object which is firing the
	 * event.
	 * @param {globalThis.KeyboardEvent} event - The original browser event that was fired.
	 * @example
	 * const onKeyDown = function (e) {
	 *     if (e.key === pc.KEY_SPACE) {
	 *         // space key pressed
	 *     }
	 *     e.event.preventDefault(); // Use original browser event to prevent browser action.
	 * };
	 * app.keyboard.on("keydown", onKeyDown, this);
	 */
	function KeyboardEvent(keyboard, event) {
	  if (event) {
	    /**
	     * The keyCode of the key that has changed. See the KEY_* constants.
	     *
	     * @type {number}
	     */
	    this.key = event.keyCode;
	    /**
	     * The element that fired the keyboard event.
	     *
	     * @type {Element}
	     */
	    this.element = event.target;
	    /**
	     * The original browser event which was fired.
	     *
	     * @type {globalThis.KeyboardEvent}
	     */
	    this.event = event;
	  } else {
	    this.key = null;
	    this.element = null;
	    this.event = null;
	  }
	};

	// internal global keyboard events
	var _keyboardEvent = new KeyboardEvent();

	/**
	 * Convert a browser keyboard event to a PlayCanvas keyboard event.
	 *
	 * @param {globalThis.KeyboardEvent} event - A browser keyboard event.
	 * @returns {KeyboardEvent} A PlayCanvas keyboard event.
	 * @ignore
	 */
	function makeKeyboardEvent(event) {
	  _keyboardEvent.key = event.keyCode;
	  _keyboardEvent.element = event.target;
	  _keyboardEvent.event = event;
	  return _keyboardEvent;
	}

	/**
	 * Convert a string or keycode to a keycode.
	 *
	 * @param {string|number} s - Either a character code or the key character.
	 * @returns {number} The character code.
	 * @ignore
	 */
	function toKeyCode(s) {
	  if (typeof s === 'string') {
	    return s.toUpperCase().charCodeAt(0);
	  }
	  return s;
	}
	var _keyCodeToKeyIdentifier = {
	  '9': 'Tab',
	  '13': 'Enter',
	  '16': 'Shift',
	  '17': 'Control',
	  '18': 'Alt',
	  '27': 'Escape',
	  '37': 'Left',
	  '38': 'Up',
	  '39': 'Right',
	  '40': 'Down',
	  '46': 'Delete',
	  '91': 'Win'
	};

	/**
	 * A Keyboard device bound to an Element. Allows you to detect the state of the key presses. Note
	 * that the Keyboard object must be attached to an Element before it can detect any key presses.
	 *
	 * @augments EventHandler
	 * @category Input
	 */
	var Keyboard = /*#__PURE__*/function (_EventHandler) {
	  _inheritsLoose(Keyboard, _EventHandler);
	  /**
	   * Create a new Keyboard instance.
	   *
	   * @param {Element|Window} [element] - Element to attach Keyboard to. Note that elements like
	   * &lt;div&gt; can't accept focus by default. To use keyboard events on an element like this it
	   * must have a value of 'tabindex' e.g. tabindex="0". See
	   * [here](https://www.w3.org/WAI/GL/WCAG20/WD-WCAG20-TECHS/SCR29.html) for more details.
	   * @param {object} [options] - Optional options object.
	   * @param {boolean} [options.preventDefault] - Call preventDefault() in key event handlers.
	   * This stops the default action of the event occurring. e.g. Ctrl+T will not open a new
	   * browser tab.
	   * @param {boolean} [options.stopPropagation] - Call stopPropagation() in key event handlers.
	   * This stops the event bubbling up the DOM so no parent handlers will be notified of the
	   * event.
	   * @example
	   * // attach keyboard listeners to the window
	   * const keyboard = new pc.Keyboard(window);
	   */
	  function Keyboard(element, options) {
	    var _this;
	    if (options === void 0) {
	      options = {};
	    }
	    _this = _EventHandler.call(this) || this;
	    _this._element = null;
	    _this._keyDownHandler = _this._handleKeyDown.bind(_assertThisInitialized(_this));
	    _this._keyUpHandler = _this._handleKeyUp.bind(_assertThisInitialized(_this));
	    _this._keyPressHandler = _this._handleKeyPress.bind(_assertThisInitialized(_this));
	    _this._visibilityChangeHandler = _this._handleVisibilityChange.bind(_assertThisInitialized(_this));
	    _this._windowBlurHandler = _this._handleWindowBlur.bind(_assertThisInitialized(_this));
	    _this._keymap = {};
	    _this._lastmap = {};
	    if (element) {
	      _this.attach(element);
	    }
	    _this.preventDefault = options.preventDefault || false;
	    _this.stopPropagation = options.stopPropagation || false;
	    return _this;
	  }

	  /**
	   * Attach the keyboard event handlers to an Element.
	   *
	   * @param {Element|Window} element - The element to listen for keyboard events on.
	   */
	  var _proto = Keyboard.prototype;
	  _proto.attach = function attach(element) {
	    if (this._element) {
	      // remove previous attached element
	      this.detach();
	    }
	    this._element = element;
	    this._element.addEventListener('keydown', this._keyDownHandler, false);
	    this._element.addEventListener('keypress', this._keyPressHandler, false);
	    this._element.addEventListener('keyup', this._keyUpHandler, false);
	    document.addEventListener('visibilitychange', this._visibilityChangeHandler, false);
	    window.addEventListener('blur', this._windowBlurHandler, false);
	  }

	  /**
	   * Detach the keyboard event handlers from the element it is attached to.
	   */;
	  _proto.detach = function detach() {
	    if (!this._element) {
	      Debug.warn('Unable to detach keyboard. It is not attached to an element.');
	      return;
	    }
	    this._element.removeEventListener('keydown', this._keyDownHandler);
	    this._element.removeEventListener('keypress', this._keyPressHandler);
	    this._element.removeEventListener('keyup', this._keyUpHandler);
	    this._element = null;
	    document.removeEventListener('visibilitychange', this._visibilityChangeHandler, false);
	    window.removeEventListener('blur', this._windowBlurHandler, false);
	  }

	  /**
	   * Convert a key code into a key identifier.
	   *
	   * @param {number} keyCode - The key code.
	   * @returns {string} The key identifier.
	   * @private
	   */;
	  _proto.toKeyIdentifier = function toKeyIdentifier(keyCode) {
	    keyCode = toKeyCode(keyCode);
	    var id = _keyCodeToKeyIdentifier[keyCode.toString()];
	    if (id) {
	      return id;
	    }

	    // Convert to hex and add leading 0's
	    var hex = keyCode.toString(16).toUpperCase();
	    var length = hex.length;
	    for (var count = 0; count < 4 - length; count++) {
	      hex = '0' + hex;
	    }
	    return 'U+' + hex;
	  }

	  /**
	   * Process the browser keydown event.
	   *
	   * @param {globalThis.KeyboardEvent} event - The browser keyboard event.
	   * @private
	   */;
	  _proto._handleKeyDown = function _handleKeyDown(event) {
	    var code = event.keyCode || event.charCode;

	    // Google Chrome auto-filling of login forms could raise a malformed event
	    if (code === undefined) return;
	    var id = this.toKeyIdentifier(code);
	    this._keymap[id] = true;
	    this.fire('keydown', makeKeyboardEvent(event));
	    if (this.preventDefault) {
	      event.preventDefault();
	    }
	    if (this.stopPropagation) {
	      event.stopPropagation();
	    }
	  }

	  /**
	   * Process the browser keyup event.
	   *
	   * @param {globalThis.KeyboardEvent} event - The browser keyboard event.
	   * @private
	   */;
	  _proto._handleKeyUp = function _handleKeyUp(event) {
	    var code = event.keyCode || event.charCode;

	    // Google Chrome auto-filling of login forms could raise a malformed event
	    if (code === undefined) return;
	    var id = this.toKeyIdentifier(code);
	    delete this._keymap[id];
	    this.fire('keyup', makeKeyboardEvent(event));
	    if (this.preventDefault) {
	      event.preventDefault();
	    }
	    if (this.stopPropagation) {
	      event.stopPropagation();
	    }
	  }

	  /**
	   * Process the browser keypress event.
	   *
	   * @param {globalThis.KeyboardEvent} event - The browser keyboard event.
	   * @private
	   */;
	  _proto._handleKeyPress = function _handleKeyPress(event) {
	    this.fire('keypress', makeKeyboardEvent(event));
	    if (this.preventDefault) {
	      event.preventDefault();
	    }
	    if (this.stopPropagation) {
	      event.stopPropagation();
	    }
	  }

	  /**
	   * Handle the browser visibilitychange event.
	   *
	   * @private
	   */;
	  _proto._handleVisibilityChange = function _handleVisibilityChange() {
	    if (document.visibilityState === 'hidden') {
	      this._handleWindowBlur();
	    }
	  }

	  /**
	   * Handle the browser blur event.
	   *
	   * @private
	   */;
	  _proto._handleWindowBlur = function _handleWindowBlur() {
	    this._keymap = {};
	    this._lastmap = {};
	  }

	  /**
	   * Called once per frame to update internal state.
	   *
	   * @ignore
	   */;
	  _proto.update = function update() {
	    // clear all keys
	    for (var prop in this._lastmap) {
	      delete this._lastmap[prop];
	    }
	    for (var _prop in this._keymap) {
	      if (this._keymap.hasOwnProperty(_prop)) {
	        this._lastmap[_prop] = this._keymap[_prop];
	      }
	    }
	  }

	  /**
	   * Return true if the key is currently down.
	   *
	   * @param {number} key - The keyCode of the key to test. See the KEY_* constants.
	   * @returns {boolean} True if the key was pressed, false if not.
	   */;
	  _proto.isPressed = function isPressed(key) {
	    var keyCode = toKeyCode(key);
	    var id = this.toKeyIdentifier(keyCode);
	    return !!this._keymap[id];
	  }

	  /**
	   * Returns true if the key was pressed since the last update.
	   *
	   * @param {number} key - The keyCode of the key to test. See the KEY_* constants.
	   * @returns {boolean} True if the key was pressed.
	   */;
	  _proto.wasPressed = function wasPressed(key) {
	    var keyCode = toKeyCode(key);
	    var id = this.toKeyIdentifier(keyCode);
	    return !!this._keymap[id] && !!!this._lastmap[id];
	  }

	  /**
	   * Returns true if the key was released since the last update.
	   *
	   * @param {number} key - The keyCode of the key to test. See the KEY_* constants.
	   * @returns {boolean} True if the key was pressed.
	   */;
	  _proto.wasReleased = function wasReleased(key) {
	    var keyCode = toKeyCode(key);
	    var id = this.toKeyIdentifier(keyCode);
	    return !!!this._keymap[id] && !!this._lastmap[id];
	  };
	  return Keyboard;
	}(EventHandler);
	/**
	 * Fired when a key is pressed. The handler is passed a {@link KeyboardEvent}.
	 *
	 * @event
	 * @example
	 * const onKeyDown = (e) => {
	 *     if (e.key === pc.KEY_SPACE) {
	 *         // space key pressed
	 *     }
	 *     e.event.preventDefault(); // Use original browser event to prevent browser action.
	 * };
	 *
	 * app.keyboard.on("keydown", onKeyDown, this);
	 */
	Keyboard.EVENT_KEYDOWN = 'keydown';
	/**
	 * Fired when a key is released. The handler is passed a {@link KeyboardEvent}.
	 *
	 * @event
	 * @example
	 * const onKeyUp = (e) => {
	 *     if (e.key === pc.KEY_SPACE) {
	 *         // space key released
	 *     }
	 *     e.event.preventDefault(); // Use original browser event to prevent browser action.
	 * };
	 *
	 * app.keyboard.on("keyup", onKeyUp, this);
	 */
	Keyboard.EVENT_KEYUP = 'keyup';

	/**
	 * Returns true if pointer lock is currently enabled.
	 *
	 * @returns {boolean} True if pointer lock is currently enabled.
	 * @ignore
	 */
	function isMousePointerLocked() {
	  return !!(document.pointerLockElement || document.mozPointerLockElement || document.webkitPointerLockElement);
	}

	/**
	 * MouseEvent object that is passed to events 'mousemove', 'mouseup', 'mousedown' and 'mousewheel'.
	 *
	 * @category Input
	 */
	var MouseEvent =
	/**
	 * Create a new MouseEvent instance.
	 *
	 * @param {new Function("modulePath", "return import(modulePath)")('./mouse.js').Mouse} mouse - The Mouse device that is firing this event.
	 * @param {globalThis.MouseEvent} event - The original browser event that fired.
	 */
	function MouseEvent(mouse, event) {
	  var coords = {
	    x: 0,
	    y: 0
	  };
	  if (event) {
	    if (event instanceof MouseEvent) {
	      throw Error('Expected MouseEvent');
	    }
	    coords = mouse._getTargetCoords(event);
	  } else {
	    event = {};
	  }
	  if (coords) {
	    /**
	     * The x coordinate of the mouse pointer relative to the element {@link Mouse} is
	     * attached to.
	     *
	     * @type {number}
	     */
	    this.x = coords.x;
	    /**
	     * The y coordinate of the mouse pointer relative to the element {@link Mouse} is
	     * attached to.
	     *
	     * @type {number}
	     */
	    this.y = coords.y;
	  } else if (isMousePointerLocked()) {
	    this.x = 0;
	    this.y = 0;
	  } else {
	    return;
	  }

	  /**
	   * A value representing the amount the mouse wheel has moved, only valid for
	   * {@link EVENT_MOUSEWHEEL} events.
	   *
	   * @type {number}
	   */
	  this.wheelDelta = 0;
	  // deltaY is in a different range across different browsers. The only thing
	  // that is consistent is the sign of the value so snap to -1/+1.
	  if (event.type === 'wheel') {
	    if (event.deltaY > 0) {
	      this.wheelDelta = 1;
	    } else if (event.deltaY < 0) {
	      this.wheelDelta = -1;
	    }
	  }

	  // Get the movement delta in this event
	  if (isMousePointerLocked()) {
	    /**
	     * The change in x coordinate since the last mouse event.
	     *
	     * @type {number}
	     */
	    this.dx = event.movementX || event.webkitMovementX || event.mozMovementX || 0;
	    /**
	     * The change in y coordinate since the last mouse event.
	     *
	     * @type {number}
	     */
	    this.dy = event.movementY || event.webkitMovementY || event.mozMovementY || 0;
	  } else {
	    this.dx = this.x - mouse._lastX;
	    this.dy = this.y - mouse._lastY;
	  }
	  if (event.type === 'mousedown' || event.type === 'mouseup') {
	    /**
	     * The mouse button associated with this event. Can be:
	     *
	     * - {@link MOUSEBUTTON_LEFT}
	     * - {@link MOUSEBUTTON_MIDDLE}
	     * - {@link MOUSEBUTTON_RIGHT}
	     *
	     * @type {number}
	     */
	    this.button = event.button;
	  } else {
	    this.button = MOUSEBUTTON_NONE;
	  }
	  this.buttons = mouse._buttons.slice(0);

	  /**
	   * The element that the mouse was fired from.
	   *
	   * @type {Element}
	   */
	  this.element = event.target;

	  /**
	   * True if the ctrl key was pressed when this event was fired.
	   *
	   * @type {boolean}
	   */
	  this.ctrlKey = event.ctrlKey || false;
	  /**
	   * True if the alt key was pressed when this event was fired.
	   *
	   * @type {boolean}
	   */
	  this.altKey = event.altKey || false;
	  /**
	   * True if the shift key was pressed when this event was fired.
	   *
	   * @type {boolean}
	   */
	  this.shiftKey = event.shiftKey || false;
	  /**
	   * True if the meta key was pressed when this event was fired.
	   *
	   * @type {boolean}
	   */
	  this.metaKey = event.metaKey || false;

	  /**
	   * The original browser event.
	   *
	   * @type {globalThis.MouseEvent}
	   */
	  this.event = event;
	};

	/**
	 * Callback used by {@link Mouse#enablePointerLock} and {@link Application#disablePointerLock}.
	 *
	 * @callback LockMouseCallback
	 */

	/**
	 * A Mouse Device, bound to a DOM Element.
	 *
	 * @augments EventHandler
	 * @category Input
	 */
	var Mouse = /*#__PURE__*/function (_EventHandler) {
	  _inheritsLoose(Mouse, _EventHandler);
	  /**
	   * Create a new Mouse instance.
	   *
	   * @param {Element} [element] - The Element that the mouse events are attached to.
	   */
	  function Mouse(element) {
	    var _this;
	    _this = _EventHandler.call(this) || this;

	    // Clear the mouse state
	    _this._lastX = 0;
	    _this._lastY = 0;
	    _this._buttons = [false, false, false];
	    _this._lastbuttons = [false, false, false];

	    // Setup event handlers so they are bound to the correct 'this'
	    _this._upHandler = _this._handleUp.bind(_assertThisInitialized(_this));
	    _this._downHandler = _this._handleDown.bind(_assertThisInitialized(_this));
	    _this._moveHandler = _this._handleMove.bind(_assertThisInitialized(_this));
	    _this._wheelHandler = _this._handleWheel.bind(_assertThisInitialized(_this));
	    _this._contextMenuHandler = function (event) {
	      event.preventDefault();
	    };
	    _this._target = null;
	    _this._attached = false;
	    _this.attach(element);
	    return _this;
	  }

	  /**
	   * Check if the mouse pointer has been locked, using {@link Mouse#enablePointerLock}.
	   *
	   * @returns {boolean} True if locked.
	   */
	  Mouse.isPointerLocked = function isPointerLocked() {
	    return isMousePointerLocked();
	  }

	  /**
	   * Attach mouse events to an Element.
	   *
	   * @param {Element} element - The DOM element to attach the mouse to.
	   */;
	  var _proto = Mouse.prototype;
	  _proto.attach = function attach(element) {
	    this._target = element;
	    if (this._attached) return;
	    this._attached = true;
	    var opts = platform.passiveEvents ? {
	      passive: false
	    } : false;
	    window.addEventListener('mouseup', this._upHandler, opts);
	    window.addEventListener('mousedown', this._downHandler, opts);
	    window.addEventListener('mousemove', this._moveHandler, opts);
	    window.addEventListener('wheel', this._wheelHandler, opts);
	  }

	  /**
	   * Remove mouse events from the element that it is attached to.
	   */;
	  _proto.detach = function detach() {
	    if (!this._attached) return;
	    this._attached = false;
	    this._target = null;
	    var opts = platform.passiveEvents ? {
	      passive: false
	    } : false;
	    window.removeEventListener('mouseup', this._upHandler, opts);
	    window.removeEventListener('mousedown', this._downHandler, opts);
	    window.removeEventListener('mousemove', this._moveHandler, opts);
	    window.removeEventListener('wheel', this._wheelHandler, opts);
	  }

	  /**
	   * Disable the context menu usually activated with right-click.
	   */;
	  _proto.disableContextMenu = function disableContextMenu() {
	    if (!this._target) return;
	    this._target.addEventListener('contextmenu', this._contextMenuHandler);
	  }

	  /**
	   * Enable the context menu usually activated with right-click. This option is active by
	   * default.
	   */;
	  _proto.enableContextMenu = function enableContextMenu() {
	    if (!this._target) return;
	    this._target.removeEventListener('contextmenu', this._contextMenuHandler);
	  }

	  /**
	   * Request that the browser hides the mouse cursor and locks the mouse to the element. Allowing
	   * raw access to mouse movement input without risking the mouse exiting the element. Notes:
	   *
	   * - In some browsers this will only work when the browser is running in fullscreen mode. See
	   * {@link https://developer.mozilla.org/en-US/docs/Web/API/Fullscreen_API Fullscreen API} for
	   * more details.
	   * - Enabling pointer lock can only be initiated by a user action e.g. in the event handler for
	   * a mouse or keyboard input.
	   *
	   * @param {LockMouseCallback} [success] - Function called if the request for mouse lock is
	   * successful.
	   * @param {LockMouseCallback} [error] - Function called if the request for mouse lock is
	   * unsuccessful.
	   */;
	  _proto.enablePointerLock = function enablePointerLock(success, error) {
	    if (!document.body.requestPointerLock) {
	      if (error) error();
	      return;
	    }
	    var s = function s() {
	      success();
	      document.removeEventListener('pointerlockchange', s);
	    };
	    var e = function e() {
	      error();
	      document.removeEventListener('pointerlockerror', e);
	    };
	    if (success) {
	      document.addEventListener('pointerlockchange', s, false);
	    }
	    if (error) {
	      document.addEventListener('pointerlockerror', e, false);
	    }
	    document.body.requestPointerLock();
	  }

	  /**
	   * Return control of the mouse cursor to the user.
	   *
	   * @param {LockMouseCallback} [success] - Function called when the mouse lock is disabled.
	   */;
	  _proto.disablePointerLock = function disablePointerLock(success) {
	    if (!document.exitPointerLock) {
	      return;
	    }
	    var s = function s() {
	      success();
	      document.removeEventListener('pointerlockchange', s);
	    };
	    if (success) {
	      document.addEventListener('pointerlockchange', s, false);
	    }
	    document.exitPointerLock();
	  }

	  /**
	   * Update method, should be called once per frame.
	   */;
	  _proto.update = function update() {
	    // Copy current button state
	    this._lastbuttons[0] = this._buttons[0];
	    this._lastbuttons[1] = this._buttons[1];
	    this._lastbuttons[2] = this._buttons[2];
	  }

	  /**
	   * Returns true if the mouse button is currently pressed.
	   *
	   * @param {number} button - The mouse button to test. Can be:
	   *
	   * - {@link MOUSEBUTTON_LEFT}
	   * - {@link MOUSEBUTTON_MIDDLE}
	   * - {@link MOUSEBUTTON_RIGHT}
	   *
	   * @returns {boolean} True if the mouse button is current pressed.
	   */;
	  _proto.isPressed = function isPressed(button) {
	    return this._buttons[button];
	  }

	  /**
	   * Returns true if the mouse button was pressed this frame (since the last call to update).
	   *
	   * @param {number} button - The mouse button to test. Can be:
	   *
	   * - {@link MOUSEBUTTON_LEFT}
	   * - {@link MOUSEBUTTON_MIDDLE}
	   * - {@link MOUSEBUTTON_RIGHT}
	   *
	   * @returns {boolean} True if the mouse button was pressed since the last update.
	   */;
	  _proto.wasPressed = function wasPressed(button) {
	    return this._buttons[button] && !this._lastbuttons[button];
	  }

	  /**
	   * Returns true if the mouse button was released this frame (since the last call to update).
	   *
	   * @param {number} button - The mouse button to test. Can be:
	   *
	   * - {@link MOUSEBUTTON_LEFT}
	   * - {@link MOUSEBUTTON_MIDDLE}
	   * - {@link MOUSEBUTTON_RIGHT}
	   *
	   * @returns {boolean} True if the mouse button was released since the last update.
	   */;
	  _proto.wasReleased = function wasReleased(button) {
	    return !this._buttons[button] && this._lastbuttons[button];
	  };
	  _proto._handleUp = function _handleUp(event) {
	    // disable released button
	    this._buttons[event.button] = false;
	    var e = new MouseEvent(this, event);
	    if (!e.event) return;

	    // send 'mouseup' event
	    this.fire(EVENT_MOUSEUP, e);
	  };
	  _proto._handleDown = function _handleDown(event) {
	    // Store which button has affected
	    this._buttons[event.button] = true;
	    var e = new MouseEvent(this, event);
	    if (!e.event) return;
	    this.fire(EVENT_MOUSEDOWN, e);
	  };
	  _proto._handleMove = function _handleMove(event) {
	    var e = new MouseEvent(this, event);
	    if (!e.event) return;
	    this.fire(EVENT_MOUSEMOVE, e);

	    // Store the last offset position to calculate deltas
	    this._lastX = e.x;
	    this._lastY = e.y;
	  };
	  _proto._handleWheel = function _handleWheel(event) {
	    var e = new MouseEvent(this, event);
	    if (!e.event) return;
	    this.fire(EVENT_MOUSEWHEEL, e);
	  };
	  _proto._getTargetCoords = function _getTargetCoords(event) {
	    var rect = this._target.getBoundingClientRect();
	    var left = Math.floor(rect.left);
	    var top = Math.floor(rect.top);

	    // mouse is outside of canvas
	    if (event.clientX < left || event.clientX >= left + this._target.clientWidth || event.clientY < top || event.clientY >= top + this._target.clientHeight) {
	      return null;
	    }
	    return {
	      x: event.clientX - left,
	      y: event.clientY - top
	    };
	  };
	  return Mouse;
	}(EventHandler);
	/**
	 * Fired when the mouse is moved. The handler is passed a {@link MouseEvent}.
	 *
	 * @event
	 * @example
	 * app.mouse.on('mousemove', (e) => {
	 *     console.log(`Current mouse position is: ${e.x}, ${e.y}`);
	 * });
	 */
	Mouse.EVENT_MOUSEMOVE = EVENT_MOUSEMOVE;
	/**
	 * Fired when a mouse button is pressed. The handler is passed a {@link MouseEvent}.
	 *
	 * @event
	 * @example
	 * app.mouse.on('mousedown', (e) => {
	 *     console.log(`The ${e.button} button was pressed at position: ${e.x}, ${e.y}`);
	 * });
	 */
	Mouse.EVENT_MOUSEDOWN = EVENT_MOUSEDOWN;
	/**
	 * Fired when a mouse button is released. The handler is passed a {@link MouseEvent}.
	 *
	 * @event
	 * @example
	 * app.mouse.on('mouseup', (e) => {
	 *     console.log(`The ${e.button} button was released at position: ${e.x}, ${e.y}`);
	 * });
	 */
	Mouse.EVENT_MOUSEUP = EVENT_MOUSEUP;
	/**
	 * Fired when a mouse wheel is moved. The handler is passed a {@link MouseEvent}.
	 *
	 * @event
	 * @example
	 * app.mouse.on('mousewheel', (e) => {
	 *     console.log(`The mouse wheel was moved by ${e.wheelDelta}`);
	 * });
	 */
	Mouse.EVENT_MOUSEWHEEL = EVENT_MOUSEWHEEL;

	/**
	 * A general input handler which handles both mouse and keyboard input assigned to named actions.
	 * This allows you to define input handlers separately to defining keyboard/mouse configurations.
	 *
	 * @category Input
	 */
	var Controller = /*#__PURE__*/function () {
	  /**
	   * Create a new instance of a Controller.
	   *
	   * @param {Element} [element] - Element to attach Controller to.
	   * @param {object} [options] - Optional arguments.
	   * @param {Keyboard} [options.keyboard] - A Keyboard object to use.
	   * @param {Mouse} [options.mouse] - A Mouse object to use.
	   * @param {new Function("modulePath", "return import(modulePath)")('./game-pads.js').GamePads} [options.gamepads] - A Gamepads object to use.
	   * @example
	   * const c = new pc.Controller(document);
	   *
	   * // Register the "fire" action and assign it to both the Enter key and the space bar.
	   * c.registerKeys("fire", [pc.KEY_ENTER, pc.KEY_SPACE]);
	   */
	  function Controller(element, options) {
	    if (options === void 0) {
	      options = {};
	    }
	    this._keyboard = options.keyboard || null;
	    this._mouse = options.mouse || null;
	    this._gamepads = options.gamepads || null;
	    this._element = null;
	    this._actions = {};
	    this._axes = {};
	    this._axesValues = {};
	    if (element) {
	      this.attach(element);
	    }
	  }

	  /**
	   * Attach Controller to an Element. This is required before you can monitor for key/mouse
	   * inputs.
	   *
	   * @param {Element} element - The element to attach mouse and keyboard event handler too.
	   */
	  var _proto = Controller.prototype;
	  _proto.attach = function attach(element) {
	    this._element = element;
	    if (this._keyboard) {
	      this._keyboard.attach(element);
	    }
	    if (this._mouse) {
	      this._mouse.attach(element);
	    }
	  }

	  /**
	   * Detach Controller from an Element. This should be done before the Controller is destroyed.
	   */;
	  _proto.detach = function detach() {
	    if (this._keyboard) {
	      this._keyboard.detach();
	    }
	    if (this._mouse) {
	      this._mouse.detach();
	    }
	    this._element = null;
	  }

	  /**
	   * Disable the context menu usually activated with the right mouse button.
	   */;
	  _proto.disableContextMenu = function disableContextMenu() {
	    if (!this._mouse) {
	      this._enableMouse();
	    }
	    this._mouse.disableContextMenu();
	  }

	  /**
	   * Enable the context menu usually activated with the right mouse button. This is enabled by
	   * default.
	   */;
	  _proto.enableContextMenu = function enableContextMenu() {
	    if (!this._mouse) {
	      this._enableMouse();
	    }
	    this._mouse.enableContextMenu();
	  }

	  /**
	   * Update the Keyboard and Mouse handlers.
	   *
	   * @param {object} dt - The time since the last frame.
	   */;
	  _proto.update = function update(dt) {
	    if (this._keyboard) {
	      this._keyboard.update();
	    }
	    if (this._mouse) {
	      this._mouse.update();
	    }
	    if (this._gamepads) {
	      this._gamepads.update();
	    }

	    // clear axes values
	    this._axesValues = {};
	    for (var key in this._axes) {
	      this._axesValues[key] = [];
	    }
	  }

	  /**
	   * Helper function to append an action.
	   *
	   * @param {string} action_name - The name of the action.
	   * @param {object} action - An action object to add.
	   * @param {ACTION_KEYBOARD | ACTION_MOUSE | ACTION_GAMEPAD} action.type - The name of the action.
	   * @param {number[]} [action.keys] - Keyboard: A list of keycodes e.g. `[pc.KEY_A, pc.KEY_ENTER]`.
	   * @param {number} [action.button] - Mouse: e.g. `pc.MOUSEBUTTON_LEFT` - Gamepad: e.g. `pc.PAD_FACE_1`
	   * @param {number} [action.pad] - Gamepad: An index of the pad to register (use {@link PAD_1}, etc).
	   */;
	  _proto.appendAction = function appendAction(action_name, action) {
	    this._actions[action_name] = this._actions[action_name] || [];
	    this._actions[action_name].push(action);
	  }

	  /**
	   * Create or update a action which is enabled when the supplied keys are pressed.
	   *
	   * @param {string} action - The name of the action.
	   * @param {number[]} keys - A list of keycodes.
	   */;
	  _proto.registerKeys = function registerKeys(action, keys) {
	    if (!this._keyboard) {
	      this._enableKeyboard();
	    }
	    if (this._actions[action]) {
	      throw new Error("Action: " + action + " already registered");
	    }
	    if (keys === undefined) {
	      throw new Error('Invalid button');
	    }

	    // convert to an array
	    if (!keys.length) {
	      keys = [keys];
	    }

	    // add keys to actions
	    this.appendAction(action, {
	      type: ACTION_KEYBOARD,
	      keys: keys
	    });
	  }

	  /**
	   * Create or update an action which is enabled when the supplied mouse button is pressed.
	   *
	   * @param {string} action - The name of the action.
	   * @param {number} button - The mouse button.
	   */;
	  _proto.registerMouse = function registerMouse(action, button) {
	    if (!this._mouse) {
	      this._enableMouse();
	    }
	    if (button === undefined) {
	      throw new Error('Invalid button');
	    }

	    // add mouse button to actions
	    this.appendAction(action, {
	      type: ACTION_MOUSE,
	      button: button
	    });
	  }

	  /**
	   * Create or update an action which is enabled when the gamepad button is pressed.
	   *
	   * @param {string} action - The name of the action.
	   * @param {number} pad - The index of the pad to register (use {@link PAD_1}, etc).
	   * @param {number} button - The pad button.
	   */;
	  _proto.registerPadButton = function registerPadButton(action, pad, button) {
	    if (button === undefined) {
	      throw new Error('Invalid button');
	    }
	    // add gamepad button and pad to actions
	    this.appendAction(action, {
	      type: ACTION_GAMEPAD,
	      button: button,
	      pad: pad
	    });
	  }

	  /**
	   * Register an action against a controller axis.
	   *
	   * @param {object} [options] - Optional options object.
	   * @param {number} [options.pad] - The index of the game pad to register for (use {@link PAD_1}, etc).
	   */;
	  _proto.registerAxis = function registerAxis(options) {
	    var name = options.name;
	    if (!this._axes[name]) {
	      this._axes[name] = [];
	    }
	    var i = this._axes[name].push(name);
	    options = options || {};
	    options.pad = options.pad || PAD_1;
	    var bind = function bind(controller, source, value, key) {
	      switch (source) {
	        case 'mousex':
	          controller._mouse.on(EVENT_MOUSEMOVE, function (e) {
	            controller._axesValues[name][i] = e.dx / 10;
	          });
	          break;
	        case 'mousey':
	          controller._mouse.on(EVENT_MOUSEMOVE, function (e) {
	            controller._axesValues[name][i] = e.dy / 10;
	          });
	          break;
	        case 'key':
	          controller._axes[name].push(function () {
	            return controller._keyboard.isPressed(key) ? value : 0;
	          });
	          break;
	        case 'padrx':
	          controller._axes[name].push(function () {
	            return controller._gamepads.getAxis(options.pad, PAD_R_STICK_X);
	          });
	          break;
	        case 'padry':
	          controller._axes[name].push(function () {
	            return controller._gamepads.getAxis(options.pad, PAD_R_STICK_Y);
	          });
	          break;
	        case 'padlx':
	          controller._axes[name].push(function () {
	            return controller._gamepads.getAxis(options.pad, PAD_L_STICK_X);
	          });
	          break;
	        case 'padly':
	          controller._axes[name].push(function () {
	            return controller._gamepads.getAxis(options.pad, PAD_L_STICK_Y);
	          });
	          break;
	        default:
	          throw new Error('Unknown axis');
	      }
	    };
	    bind(this, options.positive, 1, options.positiveKey);
	    if (options.negativeKey || options.negative !== options.positive) {
	      bind(this, options.negative, -1, options.negativeKey);
	    }
	  }

	  /**
	   * Returns true if the current action is enabled.
	   *
	   * @param {string} actionName - The name of the action.
	   * @returns {boolean} True if the action is enabled.
	   */;
	  _proto.isPressed = function isPressed(actionName) {
	    if (!this._actions[actionName]) {
	      return false;
	    }
	    var length = this._actions[actionName].length;
	    for (var index = 0; index < length; ++index) {
	      var action = this._actions[actionName][index];
	      switch (action.type) {
	        case ACTION_KEYBOARD:
	          if (this._keyboard) {
	            var len = action.keys.length;
	            for (var i = 0; i < len; i++) {
	              if (this._keyboard.isPressed(action.keys[i])) {
	                return true;
	              }
	            }
	          }
	          break;
	        case ACTION_MOUSE:
	          if (this._mouse && this._mouse.isPressed(action.button)) {
	            return true;
	          }
	          break;
	        case ACTION_GAMEPAD:
	          if (this._gamepads && this._gamepads.isPressed(action.pad, action.button)) {
	            return true;
	          }
	          break;
	      }
	    }
	    return false;
	  }

	  /**
	   * Returns true if the action was enabled this since the last update.
	   *
	   * @param {string} actionName - The name of the action.
	   * @returns {boolean} True if the action was enabled this since the last update.
	   */;
	  _proto.wasPressed = function wasPressed(actionName) {
	    if (!this._actions[actionName]) {
	      return false;
	    }
	    var length = this._actions[actionName].length;
	    for (var index = 0; index < length; ++index) {
	      var action = this._actions[actionName][index];
	      switch (action.type) {
	        case ACTION_KEYBOARD:
	          if (this._keyboard) {
	            var len = action.keys.length;
	            for (var i = 0; i < len; i++) {
	              if (this._keyboard.wasPressed(action.keys[i])) {
	                return true;
	              }
	            }
	          }
	          break;
	        case ACTION_MOUSE:
	          if (this._mouse && this._mouse.wasPressed(action.button)) {
	            return true;
	          }
	          break;
	        case ACTION_GAMEPAD:
	          if (this._gamepads && this._gamepads.wasPressed(action.pad, action.button)) {
	            return true;
	          }
	          break;
	      }
	    }
	    return false;
	  };
	  _proto.getAxis = function getAxis(name) {
	    var value = 0;
	    if (this._axes[name]) {
	      var len = this._axes[name].length;
	      for (var i = 0; i < len; i++) {
	        if (type$1(this._axes[name][i]) === 'function') {
	          var v = this._axes[name][i]();
	          if (Math.abs(v) > Math.abs(value)) {
	            value = v;
	          }
	        } else if (this._axesValues[name]) {
	          if (Math.abs(this._axesValues[name][i]) > Math.abs(value)) {
	            value = this._axesValues[name][i];
	          }
	        }
	      }
	    }
	    return value;
	  };
	  _proto._enableMouse = function _enableMouse() {
	    this._mouse = new Mouse();
	    if (!this._element) {
	      throw new Error('Controller must be attached to an Element');
	    }
	    this._mouse.attach(this._element);
	  };
	  _proto._enableKeyboard = function _enableKeyboard() {
	    this._keyboard = new Keyboard();
	    if (!this._element) {
	      throw new Error('Controller must be attached to an Element');
	    }
	    this._keyboard.attach(this._element);
	  };
	  return Controller;
	}();

	var dummyArray = Object.freeze([]);

	/**
	 * Get Gamepads from API.
	 *
	 * @type {Function}
	 * @returns {Gamepad[]} Retrieved gamepads from the device.
	 * @ignore
	 */
	var getGamepads = function getGamepads() {
	  return dummyArray;
	};
	if (typeof navigator !== 'undefined') {
	  getGamepads = (navigator.getGamepads || navigator.webkitGetGamepads || getGamepads).bind(navigator);
	}
	var MAPS_INDEXES = {
	  buttons: {
	    PAD_FACE_1: PAD_FACE_1,
	    PAD_FACE_2: PAD_FACE_2,
	    PAD_FACE_3: 