/**
 * @license
 * PlayCanvas Engine v1.69.2 revision 3e80480 (RELEASE)
 * Copyright 2011-2024 PlayCanvas Ltd. All rights reserved.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
const TRACEID_RENDER_FRAME = 'RenderFrame';
const TRACEID_RENDER_FRAME_TIME = 'RenderFrameTime';
const TRACEID_RENDER_PASS = 'RenderPass';
const TRACEID_RENDER_PASS_DETAIL = 'RenderPassDetail';
const TRACEID_RENDER_ACTION = 'RenderAction';
const TRACEID_RENDER_TARGET_ALLOC = 'RenderTargetAlloc';
const TRACEID_TEXTURE_ALLOC = 'TextureAlloc';
const TRACEID_SHADER_ALLOC = 'ShaderAlloc';
const TRACEID_SHADER_COMPILE = 'ShaderCompile';
const TRACEID_VRAM_TEXTURE = 'VRAM.Texture';
const TRACEID_VRAM_VB = 'VRAM.Vb';
const TRACEID_VRAM_IB = 'VRAM.Ib';
const TRACEID_BINDGROUP_ALLOC = 'BindGroupAlloc';
const TRACEID_BINDGROUPFORMAT_ALLOC = 'BindGroupFormatAlloc';
const TRACEID_RENDERPIPELINE_ALLOC = 'RenderPipelineAlloc';
const TRACEID_COMPUTEPIPELINE_ALLOC = 'ComputePipelineAlloc';
const TRACEID_PIPELINELAYOUT_ALLOC = 'PipelineLayoutAlloc';
const TRACE_ID_ELEMENT = "Element";
const TRACEID_TEXTURES = 'Textures';
const TRACEID_RENDER_QUEUE = 'RenderQueue';
const TRACEID_GPU_TIMINGS = 'GpuTimings';

const version = '1.69.2';
const revision = '3e80480';
const config = {};
const common = {};
const apps = {};
const data$1 = {};
const typeofs = ['undefined', 'number', 'string', 'boolean'];
const objectTypes = {
	'[object Array]': 'array',
	'[object Object]': 'object',
	'[object Function]': 'function',
	'[object Date]': 'date',
	'[object RegExp]': 'regexp',
	'[object Float32Array]': 'float32array'
};
function type(obj) {
	if (obj === null) {
		return 'null';
	}
	const typeString = typeof obj;
	if (typeofs.includes(typeString)) {
		return typeString;
	}
	return objectTypes[Object.prototype.toString.call(obj)];
}
function extend(target, ex) {
	for (const prop in ex) {
		const copy = ex[prop];
		if (type(copy) === 'object') {
			target[prop] = extend({}, copy);
		} else if (type(copy) === 'array') {
			target[prop] = extend([], copy);
		} else {
			target[prop] = copy;
		}
	}
	return target;
}

class Tracing {
	static set(channel, enabled = true) {}
	static get(channel) {
		return Tracing._traceChannels.has(channel);
	}
}
Tracing._traceChannels = new Set();
Tracing.stack = false;

class EventHandle {
	constructor(handler, name, callback, scope, once = false) {
		this.handler = void 0;
		this.name = void 0;
		this.callback = void 0;
		this.scope = void 0;
		this._once = void 0;
		this._removed = false;
		this.handler = handler;
		this.name = name;
		this.callback = callback;
		this.scope = scope;
		this._once = once;
	}
	off() {
		if (this._removed) return;
		this.handler.off(this.name, this.callback, this.scope);
	}
	on(name, callback, scope = this) {
		return this.handler._addCallback(name, callback, scope, false);
	}
	once(name, callback, scope = this) {
		return this.handler._addCallback(name, callback, scope, true);
	}
	set removed(value) {
		if (!value) return;
		this._removed = true;
	}
	get removed() {
		return this._removed;
	}
}

class EventHandler {
	constructor() {
		this._callbacks = new Map();
		this._callbackActive = new Map();
	}
	initEventHandler() {
		this._callbacks = new Map();
		this._callbackActive = new Map();
	}
	_addCallback(name, callback, scope, once) {
		if (!this._callbacks.has(name)) this._callbacks.set(name, []);
		if (this._callbackActive.has(name)) {
			const callbackActive = this._callbackActive.get(name);
			if (callbackActive && callbackActive === this._callbacks.get(name)) {
				this._callbackActive.set(name, callbackActive.slice());
			}
		}
		const evt = new EventHandle(this, name, callback, scope, once);
		this._callbacks.get(name).push(evt);
		return evt;
	}
	on(name, callback, scope = this) {
		return this._addCallback(name, callback, scope, false);
	}
	once(name, callback, scope = this) {
		return this._addCallback(name, callback, scope, true);
	}
	off(name, callback, scope) {
		if (name) {
			if (this._callbackActive.has(name) && this._callbackActive.get(name) === this._callbacks.get(name)) this._callbackActive.set(name, this._callbackActive.get(name).slice());
		} else {
			for (const [key, callbacks] of this._callbackActive) {
				if (!this._callbacks.has(key)) continue;
				if (this._callbacks.get(key) !== callbacks) continue;
				this._callbackActive.set(key, callbacks.slice());
			}
		}
		if (!name) {
			for (const callbacks of this._callbacks.values()) {
				for (let i = 0; i < callbacks.length; i++) {
					callbacks[i].removed = true;
				}
			}
			this._callbacks.clear();
		} else if (!callback) {
			const callbacks = this._callbacks.get(name);
			if (callbacks) {
				for (let i = 0; i < callbacks.length; i++) {
					callbacks[i].removed = true;
				}
				this._callbacks.delete(name);
			}
		} else {
			const callbacks = this._callbacks.get(name);
			if (!callbacks) return this;
			for (let i = 0; i < callbacks.length; i++) {
				if (callbacks[i].callback !== callback) continue;
				if (scope && callbacks[i].scope !== scope) continue;
				callbacks[i].removed = true;
				callbacks.splice(i, 1);
				i--;
			}
			if (callbacks.length === 0) this._callbacks.delete(name);
		}
		return this;
	}
	fire(name, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8) {
		if (!name) return this;
		const callbacksInitial = this._callbacks.get(name);
		if (!callbacksInitial) return this;
		let callbacks;
		if (!this._callbackActive.has(name)) {
			this._callbackActive.set(name, callbacksInitial);
		} else if (this._callbackActive.get(name) !== callbacksInitial) {
			callbacks = callbacksInitial.slice();
		}
		for (let i = 0; (callbacks || this._callbackActive.get(name)) && i < (callbacks || this._callbackActive.get(name)).length; i++) {
			const evt = (callbacks || this._callbackActive.get(name))[i];
			if (!evt.callback) continue;
			evt.callback.call(evt.scope, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8);
			if (evt._once) {
				const existingCallback = this._callbacks.get(name);
				const ind = existingCallback ? existingCallback.indexOf(evt) : -1;
				if (ind !== -1) {
					if (this._callbackActive.get(name) === existingCallback) this._callbackActive.set(name, this._callbackActive.get(name).slice());
					const _callbacks = this._callbacks.get(name);
					if (!_callbacks) continue;
					_callbacks[ind].removed = true;
					_callbacks.splice(ind, 1);
					if (_callbacks.length === 0) this._callbacks.delete(name);
				}
			}
		}
		if (!callbacks) this._callbackActive.delete(name);
		return this;
	}
	hasEvent(name) {
		var _this$_callbacks$get;
		return !!((_this$_callbacks$get = this._callbacks.get(name)) != null && _this$_callbacks$get.length);
	}
}

const events = {
	attach(target) {
		const ev = events;
		target._addCallback = ev._addCallback;
		target.on = ev.on;
		target.off = ev.off;
		target.fire = ev.fire;
		target.once = ev.once;
		target.hasEvent = ev.hasEvent;
		EventHandler.prototype.initEventHandler.call(target);
		return target;
	},
	_addCallback: EventHandler.prototype._addCallback,
	on: EventHandler.prototype.on,
	off: EventHandler.prototype.off,
	fire: EventHandler.prototype.fire,
	once: EventHandler.prototype.once,
	hasEvent: EventHandler.prototype.hasEvent
};

const guid = {
	create() {
		return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function (c) {
			const r = Math.random() * 16 | 0;
			const v = c === 'x' ? r : r & 0x3 | 0x8;
			return v.toString(16);
		});
	}
};

const path = {
	delimiter: '/',
	join(...sections) {
		let result = sections[0];
		for (let i = 0; i < sections.length - 1; i++) {
			const one = sections[i];
			const two = sections[i + 1];
			if (two[0] === path.delimiter) {
				result = two;
				continue;
			}
			if (one && two && one[one.length - 1] !== path.delimiter && two[0] !== path.delimiter) {
				result += path.delimiter + two;
			} else {
				result += two;
			}
		}
		return result;
	},
	normalize(pathname) {
		const lead = pathname.startsWith(path.delimiter);
		const trail = pathname.endsWith(path.delimiter);
		const parts = pathname.split('/');
		let result = '';
		let cleaned = [];
		for (let i = 0; i < parts.length; i++) {
			if (parts[i] === '') continue;
			if (parts[i] === '.') continue;
			if (parts[i] === '..' && cleaned.length > 0) {
				cleaned = cleaned.slice(0, cleaned.length - 2);
				continue;
			}
			if (i > 0) cleaned.push(path.delimiter);
			cleaned.push(parts[i]);
		}
		result = cleaned.join('');
		if (!lead && result[0] === path.delimiter) {
			result = result.slice(1);
		}
		if (trail && result[result.length - 1] !== path.delimiter) {
			result += path.delimiter;
		}
		return result;
	},
	split(pathname) {
		const lastDelimiterIndex = pathname.lastIndexOf(path.delimiter);
		if (lastDelimiterIndex !== -1) {
			return [pathname.substring(0, lastDelimiterIndex), pathname.substring(lastDelimiterIndex + 1)];
		}
		return ["", pathname];
	},
	getBasename(pathname) {
		return path.split(pathname)[1];
	},
	getDirectory(pathname) {
		return path.split(pathname)[0];
	},
	getExtension(pathname) {
		const ext = pathname.split('?')[0].split('.').pop();
		if (ext !== pathname) {
			return '.' + ext;
		}
		return '';
	},
	isRelativePath(pathname) {
		return pathname.charAt(0) !== '/' && pathname.match(/:\/\//) === null;
	},
	extractPath(pathname) {
		let result = '';
		const parts = pathname.split('/');
		let i = 0;
		if (parts.length > 1) {
			if (path.isRelativePath(pathname)) {
				if (parts[0] === '.') {
					for (i = 0; i < parts.length - 1; ++i) {
						result += i === 0 ? parts[i] : '/' + parts[i];
					}
				} else if (parts[0] === '..') {
					for (i = 0; i < parts.length - 1; ++i) {
						result += i === 0 ? parts[i] : '/' + parts[i];
					}
				} else {
					result = '.';
					for (i = 0; i < parts.length - 1; ++i) {
						result += '/' + parts[i];
					}
				}
			} else {
				for (i = 0; i < parts.length - 1; ++i) {
					result += i === 0 ? parts[i] : '/' + parts[i];
				}
			}
		}
		return result;
	}
};

var _ref, _ref2, _ref3;
const detectPassiveEvents = () => {
	let result = false;
	try {
		const opts = Object.defineProperty({}, 'passive', {
			get: function () {
				result = true;
				return false;
			}
		});
		window.addEventListener('testpassive', null, opts);
		window.removeEventListener('testpassive', null, opts);
	} catch (e) {}
	return result;
};
const ua = typeof navigator !== 'undefined' ? navigator.userAgent : '';
const environment = typeof window !== 'undefined' ? 'browser' : typeof global !== 'undefined' ? 'node' : 'worker';
const platformName = /android/i.test(ua) ? 'android' : /ip([ao]d|hone)/i.test(ua) ? 'ios' : /windows/i.test(ua) ? 'windows' : /mac os/i.test(ua) ? 'osx' : /linux/i.test(ua) ? 'linux' : /cros/i.test(ua) ? 'cros' : null;
const browserName = environment !== 'browser' ? null : /(Chrome\/|Chromium\/|Edg.*\/)/.test(ua) ? 'chrome' : /Safari\//.test(ua) ? 'safari' : /Firefox\//.test(ua) ? 'firefox' : 'other';
const xbox = /xbox/i.test(ua);
const touch = environment === 'browser' && ('ontouchstart' in window || 'maxTouchPoints' in navigator && navigator.maxTouchPoints > 0);
const gamepads = environment === 'browser' && (!!navigator.getGamepads || !!navigator.webkitGetGamepads);
const workers = typeof Worker !== 'undefined';
const passiveEvents = detectPassiveEvents();
const platform = {
	name: platformName,
	environment: environment,
	global: (_ref = (_ref2 = (_ref3 = typeof globalThis !== 'undefined' && globalThis) != null ? _ref3 : environment === 'browser' && window) != null ? _ref2 : environment === 'node' && global) != null ? _ref : environment === 'worker' && self,
	browser: environment === 'browser',
	desktop: ['windows', 'osx', 'linux', 'cros'].includes(platformName),
	mobile: ['android', 'ios'].includes(platformName),
	ios: platformName === 'ios',
	android: platformName === 'android',
	xbox: xbox,
	gamepads: gamepads,
	touch: touch,
	workers: workers,
	passiveEvents: passiveEvents,
	browserName: browserName
};

const ASCII_LOWERCASE = 'abcdefghijklmnopqrstuvwxyz';
const ASCII_UPPERCASE = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
const ASCII_LETTERS = ASCII_LOWERCASE + ASCII_UPPERCASE;
const HIGH_SURROGATE_BEGIN = 0xD800;
const HIGH_SURROGATE_END = 0xDBFF;
const LOW_SURROGATE_BEGIN = 0xDC00;
const LOW_SURROGATE_END = 0xDFFF;
const ZERO_WIDTH_JOINER = 0x200D;
const REGIONAL_INDICATOR_BEGIN = 0x1F1E6;
const REGIONAL_INDICATOR_END = 0x1F1FF;
const FITZPATRICK_MODIFIER_BEGIN = 0x1F3FB;
const FITZPATRICK_MODIFIER_END = 0x1F3FF;
const DIACRITICAL_MARKS_BEGIN = 0x20D0;
const DIACRITICAL_MARKS_END = 0x20FF;
const VARIATION_MODIFIER_BEGIN = 0xFE00;
const VARIATION_MODIFIER_END = 0xFE0F;
function getCodePointData(string, i = 0) {
	const size = string.length;
	if (i < 0 || i >= size) {
		return null;
	}
	const first = string.charCodeAt(i);
	if (size > 1 && first >= HIGH_SURROGATE_BEGIN && first <= HIGH_SURROGATE_END) {
		const second = string.charCodeAt(i + 1);
		if (second >= LOW_SURROGATE_BEGIN && second <= LOW_SURROGATE_END) {
			return {
				code: (first - HIGH_SURROGATE_BEGIN) * 0x400 + second - LOW_SURROGATE_BEGIN + 0x10000,
				long: true
			};
		}
	}
	return {
		code: first,
		long: false
	};
}
function isCodeBetween(string, begin, end) {
	if (!string) return false;
	const codeData = getCodePointData(string);
	if (codeData) {
		const code = codeData.code;
		return code >= begin && code <= end;
	}
	return false;
}
function numCharsToTakeForNextSymbol(string, index) {
	if (index === string.length - 1) {
		return 1;
	}
	if (isCodeBetween(string[index], HIGH_SURROGATE_BEGIN, HIGH_SURROGATE_END)) {
		const first = string.substring(index, index + 2);
		const second = string.substring(index + 2, index + 4);
		if (isCodeBetween(second, FITZPATRICK_MODIFIER_BEGIN, FITZPATRICK_MODIFIER_END) || isCodeBetween(first, REGIONAL_INDICATOR_BEGIN, REGIONAL_INDICATOR_END) && isCodeBetween(second, REGIONAL_INDICATOR_BEGIN, REGIONAL_INDICATOR_END)) {
			return 4;
		}
		if (isCodeBetween(second, VARIATION_MODIFIER_BEGIN, VARIATION_MODIFIER_END)) {
			return 3;
		}
		return 2;
	}
	if (isCodeBetween(string[index + 1], VARIATION_MODIFIER_BEGIN, VARIATION_MODIFIER_END)) {
		return 2;
	}
	return 1;
}
const string = {
	ASCII_LOWERCASE: ASCII_LOWERCASE,
	ASCII_UPPERCASE: ASCII_UPPERCASE,
	ASCII_LETTERS: ASCII_LETTERS,
	format(s, ...args) {
		for (let i = 0; i < args.length; i++) {
			s = s.replace(`{${i}}`, args[i]);
		}
		return s;
	},
	getCodePoint(string, i) {
		const codePointData = getCodePointData(string, i);
		return codePointData && codePointData.code;
	},
	getCodePoints(string) {
		if (typeof string !== 'string') {
			throw new TypeError('Not a string');
		}
		let i = 0;
		const arr = [];
		let codePoint;
		while (!!(codePoint = getCodePointData(string, i))) {
			arr.push(codePoint.code);
			i += codePoint.long ? 2 : 1;
		}
		return arr;
	},
	getSymbols(string) {
		if (typeof string !== 'string') {
			throw new TypeError('Not a string');
		}
		let index = 0;
		const length = string.length;
		const output = [];
		let take = 0;
		let ch;
		while (index < length) {
			take += numCharsToTakeForNextSymbol(string, index + take);
			ch = string[index + take];
			if (isCodeBetween(ch, DIACRITICAL_MARKS_BEGIN, DIACRITICAL_MARKS_END)) {
				ch = string[index + take++];
			}
			if (isCodeBetween(ch, VARIATION_MODIFIER_BEGIN, VARIATION_MODIFIER_END)) {
				ch = string[index + take++];
			}
			if (ch && ch.charCodeAt(0) === ZERO_WIDTH_JOINER) {
				ch = string[index + take++];
				continue;
			}
			const char = string.substring(index, index + take);
			output.push(char);
			index += take;
			take = 0;
		}
		return output;
	},
	fromCodePoint() {
		const chars = [];
		let current;
		let codePoint;
		let units;
		for (let i = 0; i < arguments.length; ++i) {
			current = Number(arguments[i]);
			codePoint = current - 0x10000;
			units = current > 0xFFFF ? [(codePoint >> 10) + 0xD800, codePoint % 0x400 + 0xDC00] : [current];
			chars.push(String.fromCharCode.apply(null, units));
		}
		return chars.join('');
	}
};

class IndexedList {
	constructor() {
		this._list = [];
		this._index = {};
	}
	push(key, item) {
		if (this._index[key]) {
			throw Error('Key already in index ' + key);
		}
		const location = this._list.push(item) - 1;
		this._index[key] = location;
	}
	has(key) {
		return this._index[key] !== undefined;
	}
	get(key) {
		const location = this._index[key];
		if (location !== undefined) {
			return this._list[location];
		}
		return null;
	}
	remove(key) {
		const location = this._index[key];
		if (location !== undefined) {
			this._list.splice(location, 1);
			delete this._index[key];
			for (key in this._index) {
				const idx = this._index[key];
				if (idx > location) {
					this._index[key] = idx - 1;
				}
			}
			return true;
		}
		return false;
	}
	list() {
		return this._list;
	}
	clear() {
		this._list.length = 0;
		for (const prop in this._index) {
			delete this._index[prop];
		}
	}
}

const cachedResult = func => {
	const uninitToken = {};
	let result = uninitToken;
	return () => {
		if (result === uninitToken) {
			result = func();
		}
		return result;
	};
};
class Impl {
	static loadScript(url, callback) {
		const s = document.createElement('script');
		s.setAttribute('src', url);
		s.onload = () => {
			callback(null);
		};
		s.onerror = () => {
			callback(`Failed to load script='${url}'`);
		};
		document.body.appendChild(s);
	}
	static loadWasm(moduleName, config, callback) {
		const loadUrl = Impl.wasmSupported() && config.glueUrl && config.wasmUrl ? config.glueUrl : config.fallbackUrl;
		if (loadUrl) {
			Impl.loadScript(loadUrl, err => {
				if (err) {
					callback(err, null);
				} else {
					const module = window[moduleName];
					window[moduleName] = undefined;
					module({
						locateFile: () => config.wasmUrl,
						onAbort: () => {
							callback('wasm module aborted.');
						}
					}).then(instance => {
						callback(null, instance);
					});
				}
			});
		} else {
			callback('No supported wasm modules found.', null);
		}
	}
	static getModule(name) {
		if (!Impl.modules.hasOwnProperty(name)) {
			Impl.modules[name] = {
				config: null,
				initializing: false,
				instance: null,
				callbacks: []
			};
		}
		return Impl.modules[name];
	}
	static initialize(moduleName, module) {
		if (module.initializing) {
			return;
		}
		const config = module.config;
		if (config.glueUrl || config.wasmUrl || config.fallbackUrl) {
			module.initializing = true;
			Impl.loadWasm(moduleName, config, (err, instance) => {
				if (err) {
					if (config.errorHandler) {
						config.errorHandler(err);
					} else {
						console.error(`failed to initialize module=${moduleName} error=${err}`);
					}
				} else {
					module.instance = instance;
					module.callbacks.forEach(callback => {
						callback(instance);
					});
				}
			});
		}
	}
}
Impl.modules = {};
Impl.wasmSupported = cachedResult(() => {
	try {
		if (typeof WebAssembly === "object" && typeof WebAssembly.instantiate === "function") {
			const module = new WebAssembly.Module(Uint8Array.of(0x0, 0x61, 0x73, 0x6d, 0x01, 0x00, 0x00, 0x00));
			if (module instanceof WebAssembly.Module) return new WebAssembly.Instance(module) instanceof WebAssembly.Instance;
		}
	} catch (e) {}
	return false;
});
class WasmModule {
	static setConfig(moduleName, config) {
		const module = Impl.getModule(moduleName);
		module.config = config;
		if (module.callbacks.length > 0) {
			Impl.initialize(moduleName, module);
		}
	}
	static getConfig(moduleName) {
		var _Impl$modules;
		return (_Impl$modules = Impl.modules) == null || (_Impl$modules = _Impl$modules[moduleName]) == null ? void 0 : _Impl$modules.config;
	}
	static getInstance(moduleName, callback) {
		const module = Impl.getModule(moduleName);
		if (module.instance) {
			callback(module.instance);
		} else {
			module.callbacks.push(callback);
			if (module.config) {
				Impl.initialize(moduleName, module);
			}
		}
	}
}

class ReadStream {
	constructor(arraybuffer) {
		this.arraybuffer = arraybuffer;
		this.dataView = new DataView(arraybuffer);
		this.offset = 0;
		this.stack = [];
	}
	get remainingBytes() {
		return this.dataView.byteLength - this.offset;
	}
	reset(offset = 0) {
		this.offset = offset;
	}
	skip(bytes) {
		this.offset += bytes;
	}
	align(bytes) {
		this.offset = this.offset + bytes - 1 & ~(bytes - 1);
	}
	_inc(amount) {
		this.offset += amount;
		return this.offset - amount;
	}
	readChar() {
		return String.fromCharCode(this.dataView.getUint8(this.offset++));
	}
	readChars(numChars) {
		let result = '';
		for (let i = 0; i < numChars; ++i) {
			result += this.readChar();
		}
		return result;
	}
	readU8() {
		return this.dataView.getUint8(this.offset++);
	}
	readU16() {
		return this.dataView.getUint16(this._inc(2), true);
	}
	readU32() {
		return this.dataView.getUint32(this._inc(4), true);
	}
	readU64() {
		return this.readU32() + 2 ** 32 * this.readU32();
	}
	readU32be() {
		return this.dataView.getUint32(this._inc(4), false);
	}
	readArray(result) {
		for (let i = 0; i < result.length; ++i) {
			result[i] = this.readU8();
		}
	}
	readLine() {
		const view = this.dataView;
		let result = '';
		while (true) {
			if (this.offset >= view.byteLength) {
				break;
			}
			const c = String.fromCharCode(this.readU8());
			if (c === '\n') {
				break;
			}
			result += c;
		}
		return result;
	}
}

class SortedLoopArray {
	constructor(args) {
		this.items = [];
		this.length = 0;
		this.loopIndex = -1;
		this._sortBy = void 0;
		this._sortHandler = void 0;
		this._sortBy = args.sortBy;
		this._sortHandler = this._doSort.bind(this);
	}
	_binarySearch(item) {
		let left = 0;
		let right = this.items.length - 1;
		const search = item[this._sortBy];
		let middle;
		let current;
		while (left <= right) {
			middle = Math.floor((left + right) / 2);
			current = this.items[middle][this._sortBy];
			if (current <= search) {
				left = middle + 1;
			} else if (current > search) {
				right = middle - 1;
			}
		}
		return left;
	}
	_doSort(a, b) {
		const sortBy = this._sortBy;
		return a[sortBy] - b[sortBy];
	}
	insert(item) {
		const index = this._binarySearch(item);
		this.items.splice(index, 0, item);
		this.length++;
		if (this.loopIndex >= index) {
			this.loopIndex++;
		}
	}
	append(item) {
		this.items.push(item);
		this.length++;
	}
	remove(item) {
		const idx = this.items.indexOf(item);
		if (idx < 0) return;
		this.items.splice(idx, 1);
		this.length--;
		if (this.loopIndex >= idx) {
			this.loopIndex--;
		}
	}
	sort() {
		const current = this.loopIndex >= 0 ? this.items[this.loopIndex] : null;
		this.items.sort(this._sortHandler);
		if (current !== null) {
			this.loopIndex = this.items.indexOf(current);
		}
	}
}

class Tags extends EventHandler {
	constructor(parent) {
		super();
		this._index = {};
		this._list = [];
		this._parent = parent;
	}
	add() {
		let changed = false;
		const tags = this._processArguments(arguments, true);
		if (!tags.length) return changed;
		for (let i = 0; i < tags.length; i++) {
			if (this._index[tags[i]]) continue;
			changed = true;
			this._index[tags[i]] = true;
			this._list.push(tags[i]);
			this.fire('add', tags[i], this._parent);
		}
		if (changed) this.fire('change', this._parent);
		return changed;
	}
	remove() {
		let changed = false;
		if (!this._list.length) return changed;
		const tags = this._processArguments(arguments, true);
		if (!tags.length) return changed;
		for (let i = 0; i < tags.length; i++) {
			if (!this._index[tags[i]]) continue;
			changed = true;
			delete this._index[tags[i]];
			this._list.splice(this._list.indexOf(tags[i]), 1);
			this.fire('remove', tags[i], this._parent);
		}
		if (changed) this.fire('change', this._parent);
		return changed;
	}
	clear() {
		if (!this._list.length) return;
		const tags = this._list.slice(0);
		this._list = [];
		this._index = {};
		for (let i = 0; i < tags.length; i++) this.fire('remove', tags[i], this._parent);
		this.fire('change', this._parent);
	}
	has() {
		if (!this._list.length) return false;
		return this._has(this._processArguments(arguments));
	}
	_has(tags) {
		if (!this._list.length || !tags.length) return false;
		for (let i = 0; i < tags.length; i++) {
			if (tags[i].length === 1) {
				if (this._index[tags[i][0]]) return true;
			} else {
				let multiple = true;
				for (let t = 0; t < tags[i].length; t++) {
					if (this._index[tags[i][t]]) continue;
					multiple = false;
					break;
				}
				if (multiple) return true;
			}
		}
		return false;
	}
	list() {
		return this._list.slice(0);
	}
	_processArguments(args, flat) {
		const tags = [];
		let tmp = [];
		if (!args || !args.length) return tags;
		for (let i = 0; i < args.length; i++) {
			if (args[i] instanceof Array) {
				if (!flat) tmp = [];
				for (let t = 0; t < args[i].length; t++) {
					if (typeof args[i][t] !== 'string') continue;
					if (flat) {
						tags.push(args[i][t]);
					} else {
						tmp.push(args[i][t]);
					}
				}
				if (!flat && tmp.length) tags.push(tmp);
			} else if (typeof args[i] === 'string') {
				if (flat) {
					tags.push(args[i]);
				} else {
					tags.push([args[i]]);
				}
			}
		}
		return tags;
	}
	get size() {
		return this._list.length;
	}
}
Tags.EVENT_ADD = 'add';
Tags.EVENT_REMOVE = 'remove';
Tags.EVENT_CHANGE = 'change';

const now = typeof window !== 'undefined' && window.performance && window.performance.now ? performance.now.bind(performance) : Date.now;

function createURI(options) {
	let s = '';
	if ((options.authority || options.scheme) && (options.host || options.hostpath)) {
		throw new Error('Can\'t have \'scheme\' or \'authority\' and \'host\' or \'hostpath\' option');
	}
	if (options.host && options.hostpath) {
		throw new Error('Can\'t have \'host\' and \'hostpath\' option');
	}
	if (options.path && options.hostpath) {
		throw new Error('Can\'t have \'path\' and \'hostpath\' option');
	}
	if (options.scheme) {
		s += options.scheme + ':';
	}
	if (options.authority) {
		s += '//' + options.authority;
	}
	if (options.host) {
		s += options.host;
	}
	if (options.path) {
		s += options.path;
	}
	if (options.hostpath) {
		s += options.hostpath;
	}
	if (options.query) {
		s += '?' + options.query;
	}
	if (options.fragment) {
		s += '#' + options.fragment;
	}
	return s;
}
const re = /^(([^:\/?#]+):)?(\/\/([^\/?#]*))?([^?#]*)(\?([^#]*))?(#(.*))?/;
class URI {
	constructor(uri) {
		this.scheme = void 0;
		this.authority = void 0;
		this.path = void 0;
		this.query = void 0;
		this.fragment = void 0;
		const result = uri.match(re);
		this.scheme = result[2];
		this.authority = result[4];
		this.path = result[5];
		this.query = result[7];
		this.fragment = result[9];
	}
	toString() {
		let s = '';
		if (this.scheme) {
			s += this.scheme + ':';
		}
		if (this.authority) {
			s += '//' + this.authority;
		}
		s += this.path;
		if (this.query) {
			s += '?' + this.query;
		}
		if (this.fragment) {
			s += '#' + this.fragment;
		}
		return s;
	}
	getQuery() {
		const result = {};
		if (this.query) {
			const queryParams = decodeURIComponent(this.query).split('&');
			for (const queryParam of queryParams) {
				const pair = queryParam.split('=');
				result[pair[0]] = pair[1];
			}
		}
		return result;
	}
	setQuery(params) {
		let q = '';
		for (const key in params) {
			if (params.hasOwnProperty(key)) {
				if (q !== '') {
					q += '&';
				}
				q += encodeURIComponent(key) + '=' + encodeURIComponent(params[key]);
			}
		}
		this.query = q;
	}
}

const CURVE_LINEAR = 0;
const CURVE_SMOOTHSTEP = 1;
const CURVE_CATMULL = 2;
const CURVE_CARDINAL = 3;
const CURVE_SPLINE = 4;
const CURVE_STEP = 5;

const math = {
	DEG_TO_RAD: Math.PI / 180,
	RAD_TO_DEG: 180 / Math.PI,
	clamp(value, min, max) {
		if (value >= max) return max;
		if (value <= min) return min;
		return value;
	},
	intToBytes24(i) {
		const r = i >> 16 & 0xff;
		const g = i >> 8 & 0xff;
		const b = i & 0xff;
		return [r, g, b];
	},
	intToBytes32(i) {
		const r = i >> 24 & 0xff;
		const g = i >> 16 & 0xff;
		const b = i >> 8 & 0xff;
		const a = i & 0xff;
		return [r, g, b, a];
	},
	bytesToInt24(r, g, b) {
		if (r.length) {
			b = r[2];
			g = r[1];
			r = r[0];
		}
		return r << 16 | g << 8 | b;
	},
	bytesToInt32(r, g, b, a) {
		if (r.length) {
			a = r[3];
			b = r[2];
			g = r[1];
			r = r[0];
		}
		return (r << 24 | g << 16 | b << 8 | a) >>> 0;
	},
	lerp(a, b, alpha) {
		return a + (b - a) * math.clamp(alpha, 0, 1);
	},
	lerpAngle(a, b, alpha) {
		if (b - a > 180) {
			b -= 360;
		}
		if (b - a < -180) {
			b += 360;
		}
		return math.lerp(a, b, math.clamp(alpha, 0, 1));
	},
	powerOfTwo(x) {
		return x !== 0 && !(x & x - 1);
	},
	nextPowerOfTwo(val) {
		val--;
		val |= val >> 1;
		val |= val >> 2;
		val |= val >> 4;
		val |= val >> 8;
		val |= val >> 16;
		val++;
		return val;
	},
	nearestPowerOfTwo(val) {
		return Math.pow(2, Math.round(Math.log(val) / Math.log(2)));
	},
	random(min, max) {
		const diff = max - min;
		return Math.random() * diff + min;
	},
	smoothstep(min, max, x) {
		if (x <= min) return 0;
		if (x >= max) return 1;
		x = (x - min) / (max - min);
		return x * x * (3 - 2 * x);
	},
	smootherstep(min, max, x) {
		if (x <= min) return 0;
		if (x >= max) return 1;
		x = (x - min) / (max - min);
		return x * x * x * (x * (x * 6 - 15) + 10);
	},
	roundUp(numToRound, multiple) {
		if (multiple === 0) return numToRound;
		return Math.ceil(numToRound / multiple) * multiple;
	},
	between(num, a, b, inclusive) {
		const min = Math.min(a, b);
		const max = Math.max(a, b);
		return inclusive ? num >= min && num <= max : num > min && num < max;
	}
};

var _class$a;
class Color {
	constructor(r = 0, g = 0, b = 0, a = 1) {
		this.r = void 0;
		this.g = void 0;
		this.b = void 0;
		this.a = void 0;
		const length = r.length;
		if (length === 3 || length === 4) {
			this.r = r[0];
			this.g = r[1];
			this.b = r[2];
			this.a = r[3] !== undefined ? r[3] : 1;
		} else {
			this.r = r;
			this.g = g;
			this.b = b;
			this.a = a;
		}
	}
	clone() {
		const cstr = this.constructor;
		return new cstr(this.r, this.g, this.b, this.a);
	}
	copy(rhs) {
		this.r = rhs.r;
		this.g = rhs.g;
		this.b = rhs.b;
		this.a = rhs.a;
		return this;
	}
	equals(rhs) {
		return this.r === rhs.r && this.g === rhs.g && this.b === rhs.b && this.a === rhs.a;
	}
	set(r, g, b, a = 1) {
		this.r = r;
		this.g = g;
		this.b = b;
		this.a = a;
		return this;
	}
	lerp(lhs, rhs, alpha) {
		this.r = lhs.r + alpha * (rhs.r - lhs.r);
		this.g = lhs.g + alpha * (rhs.g - lhs.g);
		this.b = lhs.b + alpha * (rhs.b - lhs.b);
		this.a = lhs.a + alpha * (rhs.a - lhs.a);
		return this;
	}
	fromString(hex) {
		const i = parseInt(hex.replace('#', '0x'), 16);
		let bytes;
		if (hex.length > 7) {
			bytes = math.intToBytes32(i);
		} else {
			bytes = math.intToBytes24(i);
			bytes[3] = 255;
		}
		this.set(bytes[0] / 255, bytes[1] / 255, bytes[2] / 255, bytes[3] / 255);
		return this;
	}
	toString(alpha) {
		let s = '#' + ((1 << 24) + (Math.round(this.r * 255) << 16) + (Math.round(this.g * 255) << 8) + Math.round(this.b * 255)).toString(16).slice(1);
		if (alpha === true) {
			const a = Math.round(this.a * 255).toString(16);
			if (this.a < 16 / 255) {
				s += '0' + a;
			} else {
				s += a;
			}
		}
		return s;
	}
}
_class$a = Color;
Color.BLACK = Object.freeze(new _class$a(0, 0, 0, 1));
Color.BLUE = Object.freeze(new _class$a(0, 0, 1, 1));
Color.CYAN = Object.freeze(new _class$a(0, 1, 1, 1));
Color.GRAY = Object.freeze(new _class$a(0.5, 0.5, 0.5, 1));
Color.GREEN = Object.freeze(new _class$a(0, 1, 0, 1));
Color.MAGENTA = Object.freeze(new _class$a(1, 0, 1, 1));
Color.RED = Object.freeze(new _class$a(1, 0, 0, 1));
Color.WHITE = Object.freeze(new _class$a(1, 1, 1, 1));
Color.YELLOW = Object.freeze(new _class$a(1, 1, 0, 1));

class CurveEvaluator {
	constructor(curve, time = 0) {
		this._curve = void 0;
		this._left = -Infinity;
		this._right = Infinity;
		this._recip = 0;
		this._p0 = 0;
		this._p1 = 0;
		this._m0 = 0;
		this._m1 = 0;
		this._curve = curve;
		this._reset(time);
	}
	evaluate(time, forceReset = false) {
		if (forceReset || time < this._left || time >= this._right) {
			this._reset(time);
		}
		let result;
		const type = this._curve.type;
		if (type === CURVE_STEP) {
			result = this._p0;
		} else {
			const t = this._recip === 0 ? 0 : (time - this._left) * this._recip;
			if (type === CURVE_LINEAR) {
				result = math.lerp(this._p0, this._p1, t);
			} else if (type === CURVE_SMOOTHSTEP) {
				result = math.lerp(this._p0, this._p1, t * t * (3 - 2 * t));
			} else {
				result = this._evaluateHermite(this._p0, this._p1, this._m0, this._m1, t);
			}
		}
		return result;
	}
	_reset(time) {
		const keys = this._curve.keys;
		const len = keys.length;
		if (!len) {
			this._left = -Infinity;
			this._right = Infinity;
			this._recip = 0;
			this._p0 = this._p1 = this._m0 = this._m1 = 0;
		} else {
			if (time < keys[0][0]) {
				this._left = -Infinity;
				this._right = keys[0][0];
				this._recip = 0;
				this._p0 = this._p1 = keys[0][1];
				this._m0 = this._m1 = 0;
			} else if (time >= keys[len - 1][0]) {
				this._left = keys[len - 1][0];
				this._right = Infinity;
				this._recip = 0;
				this._p0 = this._p1 = keys[len - 1][1];
				this._m0 = this._m1 = 0;
			} else {
				let index = 0;
				while (time >= keys[index + 1][0]) {
					index++;
				}
				this._left = keys[index][0];
				this._right = keys[index + 1][0];
				const diff = 1.0 / (this._right - this._left);
				this._recip = isFinite(diff) ? diff : 0;
				this._p0 = keys[index][1];
				this._p1 = keys[index + 1][1];
				if (this._isHermite()) {
					this._calcTangents(keys, index);
				}
			}
		}
	}
	_isHermite() {
		return this._curve.type === CURVE_CATMULL || this._curve.type === CURVE_CARDINAL || this._curve.type === CURVE_SPLINE;
	}
	_calcTangents(keys, index) {
		let a;
		const b = keys[index];
		const c = keys[index + 1];
		let d;
		if (index === 0) {
			a = [keys[0][0] + (keys[0][0] - keys[1][0]), keys[0][1] + (keys[0][1] - keys[1][1])];
		} else {
			a = keys[index - 1];
		}
		if (index === keys.length - 2) {
			d = [keys[index + 1][0] + (keys[index + 1][0] - keys[index][0]), keys[index + 1][1] + (keys[index + 1][1] - keys[index][1])];
		} else {
			d = keys[index + 2];
		}
		if (this._curve.type === CURVE_SPLINE) {
			const s1_ = 2 * (c[0] - b[0]) / (c[0] - a[0]);
			const s2_ = 2 * (c[0] - b[0]) / (d[0] - b[0]);
			this._m0 = this._curve.tension * (isFinite(s1_) ? s1_ : 0) * (c[1] - a[1]);
			this._m1 = this._curve.tension * (isFinite(s2_) ? s2_ : 0) * (d[1] - b[1]);
		} else {
			const s1 = (c[0] - b[0]) / (b[0] - a[0]);
			const s2 = (c[0] - b[0]) / (d[0] - c[0]);
			const a_ = b[1] + (a[1] - b[1]) * (isFinite(s1) ? s1 : 0);
			const d_ = c[1] + (d[1] - c[1]) * (isFinite(s2) ? s2 : 0);
			const tension = this._curve.type === CURVE_CATMULL ? 0.5 : this._curve.tension;
			this._m0 = tension * (c[1] - a_);
			this._m1 = tension * (d_ - b[1]);
		}
	}
	_evaluateHermite(p0, p1, m0, m1, t) {
		const t2 = t * t;
		const twot = t + t;
		const omt = 1 - t;
		const omt2 = omt * omt;
		return p0 * ((1 + twot) * omt2) + m0 * (t * omt2) + p1 * (t2 * (3 - twot)) + m1 * (t2 * (t - 1));
	}
}

class Curve {
	constructor(data) {
		this.keys = [];
		this.type = CURVE_SMOOTHSTEP;
		this.tension = 0.5;
		this._eval = new CurveEvaluator(this);
		if (data) {
			for (let i = 0; i < data.length - 1; i += 2) {
				this.keys.push([data[i], data[i + 1]]);
			}
		}
		this.sort();
	}
	get length() {
		return this.keys.length;
	}
	add(time, value) {
		const keys = this.keys;
		const len = keys.length;
		let i = 0;
		for (; i < len; i++) {
			if (keys[i][0] > time) {
				break;
			}
		}
		const key = [time, value];
		this.keys.splice(i, 0, key);
		return key;
	}
	get(index) {
		return this.keys[index];
	}
	sort() {
		this.keys.sort(function (a, b) {
			return a[0] - b[0];
		});
	}
	value(time) {
		return this._eval.evaluate(time, true);
	}
	closest(time) {
		const keys = this.keys;
		const length = keys.length;
		let min = 2;
		let result = null;
		for (let i = 0; i < length; i++) {
			const diff = Math.abs(time - keys[i][0]);
			if (min >= diff) {
				min = diff;
				result = keys[i];
			} else {
				break;
			}
		}
		return result;
	}
	clone() {
		const result = new this.constructor();
		result.keys = extend(result.keys, this.keys);
		result.type = this.type;
		result.tension = this.tension;
		return result;
	}
	quantize(precision) {
		precision = Math.max(precision, 2);
		const values = new Float32Array(precision);
		const step = 1.0 / (precision - 1);
		values[0] = this._eval.evaluate(0, true);
		for (let i = 1; i < precision; i++) {
			values[i] = this._eval.evaluate(step * i);
		}
		return values;
	}
	quantizeClamped(precision, min, max) {
		const result = this.quantize(precision);
		for (let i = 0; i < result.length; ++i) {
			result[i] = Math.min(max, Math.max(min, result[i]));
		}
		return result;
	}
}

class CurveSet {
	constructor() {
		this.curves = [];
		this._type = CURVE_SMOOTHSTEP;
		if (arguments.length > 1) {
			for (let i = 0; i < arguments.length; i++) {
				this.curves.push(new Curve(arguments[i]));
			}
		} else {
			if (arguments.length === 0) {
				this.curves.push(new Curve());
			} else {
				const arg = arguments[0];
				if (typeof arg === 'number') {
					for (let i = 0; i < arg; i++) {
						this.curves.push(new Curve());
					}
				} else {
					for (let i = 0; i < arg.length; i++) {
						this.curves.push(new Curve(arg[i]));
					}
				}
			}
		}
	}
	get length() {
		return this.curves.length;
	}
	set type(value) {
		this._type = value;
		for (let i = 0; i < this.curves.length; i++) {
			this.curves[i].type = value;
		}
	}
	get type() {
		return this._type;
	}
	get(index) {
		return this.curves[index];
	}
	value(time, result = []) {
		const length = this.curves.length;
		result.length = length;
		for (let i = 0; i < length; i++) {
			result[i] = this.curves[i].value(time);
		}
		return result;
	}
	clone() {
		const result = new this.constructor();
		result.curves = [];
		for (let i = 0; i < this.curves.length; i++) {
			result.curves.push(this.curves[i].clone());
		}
		result._type = this._type;
		return result;
	}
	quantize(precision) {
		precision = Math.max(precision, 2);
		const numCurves = this.curves.length;
		const values = new Float32Array(precision * numCurves);
		const step = 1.0 / (precision - 1);
		for (let c = 0; c < numCurves; c++) {
			const ev = new CurveEvaluator(this.curves[c]);
			for (let i = 0; i < precision; i++) {
				values[i * numCurves + c] = ev.evaluate(step * i);
			}
		}
		return values;
	}
	quantizeClamped(precision, min, max) {
		const result = this.quantize(precision);
		for (let i = 0; i < result.length; ++i) {
			result[i] = Math.min(max, Math.max(min, result[i]));
		}
		return result;
	}
}

const oneDiv255 = 1 / 255;
const floatView = new Float32Array(1);
const int32View = new Int32Array(floatView.buffer);
class FloatPacking {
	static float2Half(value) {
		floatView[0] = value;
		const x = int32View[0];
		let bits = x >> 16 & 0x8000;
		let m = x >> 12 & 0x07ff;
		const e = x >> 23 & 0xff;
		if (e < 103) {
			return bits;
		}
		if (e > 142) {
			bits |= 0x7c00;
			bits |= (e === 255 ? 0 : 1) && x & 0x007fffff;
			return bits;
		}
		if (e < 113) {
			m |= 0x0800;
			bits |= (m >> 114 - e) + (m >> 113 - e & 1);
			return bits;
		}
		bits |= e - 112 << 10 | m >> 1;
		bits += m & 1;
		return bits;
	}
	static float2Bytes(value, array, offset, numBytes) {
		const enc1 = 255.0 * value % 1;
		array[offset + 0] = Math.round((value % 1 - oneDiv255 * enc1) * 255);
		if (numBytes > 1) {
			const enc2 = 65025.0 * value % 1;
			array[offset + 1] = Math.round((enc1 - oneDiv255 * enc2) * 255);
			if (numBytes > 2) {
				const enc3 = 16581375.0 * value % 1;
				array[offset + 2] = Math.round((enc2 - oneDiv255 * enc3) * 255);
				if (numBytes > 3) {
					array[offset + 3] = Math.round(enc3 * 255);
				}
			}
		}
	}
	static float2BytesRange(value, array, offset, min, max, numBytes) {
		value = math.clamp((value - min) / (max - min), 0, 1);
		FloatPacking.float2Bytes(value, array, offset, numBytes);
	}
	static float2MantissaExponent(value, array, offset, numBytes) {
		const exponent = Math.floor(Math.log2(Math.abs(value))) + 1;
		value /= Math.pow(2, exponent);
		FloatPacking.float2BytesRange(value, array, offset, -1, 1, numBytes - 1);
		array[offset + numBytes - 1] = Math.round(exponent + 127);
	}
}

var _class$9;
class Vec3 {
	constructor(x = 0, y = 0, z = 0) {
		this.x = void 0;
		this.y = void 0;
		this.z = void 0;
		if (x.length === 3) {
			this.x = x[0];
			this.y = x[1];
			this.z = x[2];
		} else {
			this.x = x;
			this.y = y;
			this.z = z;
		}
	}
	add(rhs) {
		this.x += rhs.x;
		this.y += rhs.y;
		this.z += rhs.z;
		return this;
	}
	add2(lhs, rhs) {
		this.x = lhs.x + rhs.x;
		this.y = lhs.y + rhs.y;
		this.z = lhs.z + rhs.z;
		return this;
	}
	addScalar(scalar) {
		this.x += scalar;
		this.y += scalar;
		this.z += scalar;
		return this;
	}
	addScaled(rhs, scalar) {
		this.x += rhs.x * scalar;
		this.y += rhs.y * scalar;
		this.z += rhs.z * scalar;
		return this;
	}
	clone() {
		const cstr = this.constructor;
		return new cstr(this.x, this.y, this.z);
	}
	copy(rhs) {
		this.x = rhs.x;
		this.y = rhs.y;
		this.z = rhs.z;
		return this;
	}
	cross(lhs, rhs) {
		const lx = lhs.x;
		const ly = lhs.y;
		const lz = lhs.z;
		const rx = rhs.x;
		const ry = rhs.y;
		const rz = rhs.z;
		this.x = ly * rz - ry * lz;
		this.y = lz * rx - rz * lx;
		this.z = lx * ry - rx * ly;
		return this;
	}
	distance(rhs) {
		const x = this.x - rhs.x;
		const y = this.y - rhs.y;
		const z = this.z - rhs.z;
		return Math.sqrt(x * x + y * y + z * z);
	}
	div(rhs) {
		this.x /= rhs.x;
		this.y /= rhs.y;
		this.z /= rhs.z;
		return this;
	}
	div2(lhs, rhs) {
		this.x = lhs.x / rhs.x;
		this.y = lhs.y / rhs.y;
		this.z = lhs.z / rhs.z;
		return this;
	}
	divScalar(scalar) {
		this.x /= scalar;
		this.y /= scalar;
		this.z /= scalar;
		return this;
	}
	dot(rhs) {
		return this.x * rhs.x + this.y * rhs.y + this.z * rhs.z;
	}
	equals(rhs) {
		return this.x === rhs.x && this.y === rhs.y && this.z === rhs.z;
	}
	equalsApprox(rhs, epsilon = 1e-6) {
		return Math.abs(this.x - rhs.x) < epsilon && Math.abs(this.y - rhs.y) < epsilon && Math.abs(this.z - rhs.z) < epsilon;
	}
	length() {
		return Math.sqrt(this.x * this.x + this.y * this.y + this.z * this.z);
	}
	lengthSq() {
		return this.x * this.x + this.y * this.y + this.z * this.z;
	}
	lerp(lhs, rhs, alpha) {
		this.x = lhs.x + alpha * (rhs.x - lhs.x);
		this.y = lhs.y + alpha * (rhs.y - lhs.y);
		this.z = lhs.z + alpha * (rhs.z - lhs.z);
		return this;
	}
	mul(rhs) {
		this.x *= rhs.x;
		this.y *= rhs.y;
		this.z *= rhs.z;
		return this;
	}
	mul2(lhs, rhs) {
		this.x = lhs.x * rhs.x;
		this.y = lhs.y * rhs.y;
		this.z = lhs.z * rhs.z;
		return this;
	}
	mulScalar(scalar) {
		this.x *= scalar;
		this.y *= scalar;
		this.z *= scalar;
		return this;
	}
	normalize(src = this) {
		const lengthSq = src.x * src.x + src.y * src.y + src.z * src.z;
		if (lengthSq > 0) {
			const invLength = 1 / Math.sqrt(lengthSq);
			this.x = src.x * invLength;
			this.y = src.y * invLength;
			this.z = src.z * invLength;
		}
		return this;
	}
	floor(src = this) {
		this.x = Math.floor(src.x);
		this.y = Math.floor(src.y);
		this.z = Math.floor(src.z);
		return this;
	}
	ceil(src = this) {
		this.x = Math.ceil(src.x);
		this.y = Math.ceil(src.y);
		this.z = Math.ceil(src.z);
		return this;
	}
	round(src = this) {
		this.x = Math.round(src.x);
		this.y = Math.round(src.y);
		this.z = Math.round(src.z);
		return this;
	}
	min(rhs) {
		if (rhs.x < this.x) this.x = rhs.x;
		if (rhs.y < this.y) this.y = rhs.y;
		if (rhs.z < this.z) this.z = rhs.z;
		return this;
	}
	max(rhs) {
		if (rhs.x > this.x) this.x = rhs.x;
		if (rhs.y > this.y) this.y = rhs.y;
		if (rhs.z > this.z) this.z = rhs.z;
		return this;
	}
	project(rhs) {
		const a_dot_b = this.x * rhs.x + this.y * rhs.y + this.z * rhs.z;
		const b_dot_b = rhs.x * rhs.x + rhs.y * rhs.y + rhs.z * rhs.z;
		const s = a_dot_b / b_dot_b;
		this.x = rhs.x * s;
		this.y = rhs.y * s;
		this.z = rhs.z * s;
		return this;
	}
	set(x, y, z) {
		this.x = x;
		this.y = y;
		this.z = z;
		return this;
	}
	sub(rhs) {
		this.x -= rhs.x;
		this.y -= rhs.y;
		this.z -= rhs.z;
		return this;
	}
	sub2(lhs, rhs) {
		this.x = lhs.x - rhs.x;
		this.y = lhs.y - rhs.y;
		this.z = lhs.z - rhs.z;
		return this;
	}
	subScalar(scalar) {
		this.x -= scalar;
		this.y -= scalar;
		this.z -= scalar;
		return this;
	}
	toString() {
		return `[${this.x}, ${this.y}, ${this.z}]`;
	}
}
_class$9 = Vec3;
Vec3.ZERO = Object.freeze(new _class$9(0, 0, 0));
Vec3.ONE = Object.freeze(new _class$9(1, 1, 1));
Vec3.UP = Object.freeze(new _class$9(0, 1, 0));
Vec3.DOWN = Object.freeze(new _class$9(0, -1, 0));
Vec3.RIGHT = Object.freeze(new _class$9(1, 0, 0));
Vec3.LEFT = Object.freeze(new _class$9(-1, 0, 0));
Vec3.FORWARD = Object.freeze(new _class$9(0, 0, -1));
Vec3.BACK = Object.freeze(new _class$9(0, 0, 1));

var _class$8;
class Mat3 {
	constructor() {
		this.data = new Float32Array(9);
		this.data[0] = this.data[4] = this.data[8] = 1;
	}
	clone() {
		const cstr = this.constructor;
		return new cstr().copy(this);
	}
	copy(rhs) {
		const src = rhs.data;
		const dst = this.data;
		dst[0] = src[0];
		dst[1] = src[1];
		dst[2] = src[2];
		dst[3] = src[3];
		dst[4] = src[4];
		dst[5] = src[5];
		dst[6] = src[6];
		dst[7] = src[7];
		dst[8] = src[8];
		return this;
	}
	set(src) {
		const dst = this.data;
		dst[0] = src[0];
		dst[1] = src[1];
		dst[2] = src[2];
		dst[3] = src[3];
		dst[4] = src[4];
		dst[5] = src[5];
		dst[6] = src[6];
		dst[7] = src[7];
		dst[8] = src[8];
		return this;
	}
	getX(x = new Vec3()) {
		return x.set(this.data[0], this.data[1], this.data[2]);
	}
	getY(y = new Vec3()) {
		return y.set(this.data[3], this.data[4], this.data[5]);
	}
	getZ(z = new Vec3()) {
		return z.set(this.data[6], this.data[7], this.data[8]);
	}
	equals(rhs) {
		const l = this.data;
		const r = rhs.data;
		return l[0] === r[0] && l[1] === r[1] && l[2] === r[2] && l[3] === r[3] && l[4] === r[4] && l[5] === r[5] && l[6] === r[6] && l[7] === r[7] && l[8] === r[8];
	}
	isIdentity() {
		const m = this.data;
		return m[0] === 1 && m[1] === 0 && m[2] === 0 && m[3] === 0 && m[4] === 1 && m[5] === 0 && m[6] === 0 && m[7] === 0 && m[8] === 1;
	}
	setIdentity() {
		const m = this.data;
		m[0] = 1;
		m[1] = 0;
		m[2] = 0;
		m[3] = 0;
		m[4] = 1;
		m[5] = 0;
		m[6] = 0;
		m[7] = 0;
		m[8] = 1;
		return this;
	}
	toString() {
		return '[' + this.data.join(', ') + ']';
	}
	transpose(src = this) {
		const s = src.data;
		const t = this.data;
		if (s === t) {
			let tmp;
			tmp = s[1];
			t[1] = s[3];
			t[3] = tmp;
			tmp = s[2];
			t[2] = s[6];
			t[6] = tmp;
			tmp = s[5];
			t[5] = s[7];
			t[7] = tmp;
		} else {
			t[0] = s[0];
			t[1] = s[3];
			t[2] = s[6];
			t[3] = s[1];
			t[4] = s[4];
			t[5] = s[7];
			t[6] = s[2];
			t[7] = s[5];
			t[8] = s[8];
		}
		return this;
	}
	setFromMat4(m) {
		const src = m.data;
		const dst = this.data;
		dst[0] = src[0];
		dst[1] = src[1];
		dst[2] = src[2];
		dst[3] = src[4];
		dst[4] = src[5];
		dst[5] = src[6];
		dst[6] = src[8];
		dst[7] = src[9];
		dst[8] = src[10];
		return this;
	}
	invertMat4(src) {
		const s = src.data;
		const a0 = s[0];
		const a1 = s[1];
		const a2 = s[2];
		const a4 = s[4];
		const a5 = s[5];
		const a6 = s[6];
		const a8 = s[8];
		const a9 = s[9];
		const a10 = s[10];
		const b11 = a10 * a5 - a6 * a9;
		const b21 = -a10 * a1 + a2 * a9;
		const b31 = a6 * a1 - a2 * a5;
		const b12 = -a10 * a4 + a6 * a8;
		const b22 = a10 * a0 - a2 * a8;
		const b32 = -a6 * a0 + a2 * a4;
		const b13 = a9 * a4 - a5 * a8;
		const b23 = -a9 * a0 + a1 * a8;
		const b33 = a5 * a0 - a1 * a4;
		const det = a0 * b11 + a1 * b12 + a2 * b13;
		if (det === 0) {
			this.setIdentity();
		} else {
			const invDet = 1 / det;
			const t = this.data;
			t[0] = b11 * invDet;
			t[1] = b21 * invDet;
			t[2] = b31 * invDet;
			t[3] = b12 * invDet;
			t[4] = b22 * invDet;
			t[5] = b32 * invDet;
			t[6] = b13 * invDet;
			t[7] = b23 * invDet;
			t[8] = b33 * invDet;
		}
		return this;
	}
	transformVector(vec, res = new Vec3()) {
		const m = this.data;
		const x = vec.x;
		const y = vec.y;
		const z = vec.z;
		res.x = x * m[0] + y * m[3] + z * m[6];
		res.y = x * m[1] + y * m[4] + z * m[7];
		res.z = x * m[2] + y * m[5] + z * m[8];
		return res;
	}
}
_class$8 = Mat3;
Mat3.IDENTITY = Object.freeze(new _class$8());
Mat3.ZERO = Object.freeze(new _class$8().set([0, 0, 0, 0, 0, 0, 0, 0, 0]));

var _class$7;
class Vec2 {
	constructor(x = 0, y = 0) {
		this.x = void 0;
		this.y = void 0;
		if (x.length === 2) {
			this.x = x[0];
			this.y = x[1];
		} else {
			this.x = x;
			this.y = y;
		}
	}
	add(rhs) {
		this.x += rhs.x;
		this.y += rhs.y;
		return this;
	}
	add2(lhs, rhs) {
		this.x = lhs.x + rhs.x;
		this.y = lhs.y + rhs.y;
		return this;
	}
	addScalar(scalar) {
		this.x += scalar;
		this.y += scalar;
		return this;
	}
	addScaled(rhs, scalar) {
		this.x += rhs.x * scalar;
		this.y += rhs.y * scalar;
		return this;
	}
	clone() {
		const cstr = this.constructor;
		return new cstr(this.x, this.y);
	}
	copy(rhs) {
		this.x = rhs.x;
		this.y = rhs.y;
		return this;
	}
	cross(rhs) {
		return this.x * rhs.y - this.y * rhs.x;
	}
	distance(rhs) {
		const x = this.x - rhs.x;
		const y = this.y - rhs.y;
		return Math.sqrt(x * x + y * y);
	}
	div(rhs) {
		this.x /= rhs.x;
		this.y /= rhs.y;
		return this;
	}
	div2(lhs, rhs) {
		this.x = lhs.x / rhs.x;
		this.y = lhs.y / rhs.y;
		return this;
	}
	divScalar(scalar) {
		this.x /= scalar;
		this.y /= scalar;
		return this;
	}
	dot(rhs) {
		return this.x * rhs.x + this.y * rhs.y;
	}
	equals(rhs) {
		return this.x === rhs.x && this.y === rhs.y;
	}
	equalsApprox(rhs, epsilon = 1e-6) {
		return Math.abs(this.x - rhs.x) < epsilon && Math.abs(this.y - rhs.y) < epsilon;
	}
	length() {
		return Math.sqrt(this.x * this.x + this.y * this.y);
	}
	lengthSq() {
		return this.x * this.x + this.y * this.y;
	}
	lerp(lhs, rhs, alpha) {
		this.x = lhs.x + alpha * (rhs.x - lhs.x);
		this.y = lhs.y + alpha * (rhs.y - lhs.y);
		return this;
	}
	mul(rhs) {
		this.x *= rhs.x;
		this.y *= rhs.y;
		return this;
	}
	mul2(lhs, rhs) {
		this.x = lhs.x * rhs.x;
		this.y = lhs.y * rhs.y;
		return this;
	}
	mulScalar(scalar) {
		this.x *= scalar;
		this.y *= scalar;
		return this;
	}
	normalize(src = this) {
		const lengthSq = src.x * src.x + src.y * src.y;
		if (lengthSq > 0) {
			const invLength = 1 / Math.sqrt(lengthSq);
			this.x = src.x * invLength;
			this.y = src.y * invLength;
		}
		return this;
	}
	rotate(degrees) {
		const angle = Math.atan2(this.x, this.y) + degrees * math.DEG_TO_RAD;
		const len = Math.sqrt(this.x * this.x + this.y * this.y);
		this.x = Math.sin(angle) * len;
		this.y = Math.cos(angle) * len;
		return this;
	}
	angle() {
		return Math.atan2(this.x, this.y) * math.RAD_TO_DEG;
	}
	angleTo(rhs) {
		return Math.atan2(this.x * rhs.y + this.y * rhs.x, this.x * rhs.x + this.y * rhs.y) * math.RAD_TO_DEG;
	}
	floor(src = this) {
		this.x = Math.floor(src.x);
		this.y = Math.floor(src.y);
		return this;
	}
	ceil(src = this) {
		this.x = Math.ceil(src.x);
		this.y = Math.ceil(src.y);
		return this;
	}
	round(src = this) {
		this.x = Math.round(src.x);
		this.y = Math.round(src.y);
		return this;
	}
	min(rhs) {
		if (rhs.x < this.x) this.x = rhs.x;
		if (rhs.y < this.y) this.y = rhs.y;
		return this;
	}
	max(rhs) {
		if (rhs.x > this.x) this.x = rhs.x;
		if (rhs.y > this.y) this.y = rhs.y;
		return this;
	}
	set(x, y) {
		this.x = x;
		this.y = y;
		return this;
	}
	sub(rhs) {
		this.x -= rhs.x;
		this.y -= rhs.y;
		return this;
	}
	sub2(lhs, rhs) {
		this.x = lhs.x - rhs.x;
		this.y = lhs.y - rhs.y;
		return this;
	}
	subScalar(scalar) {
		this.x -= scalar;
		this.y -= scalar;
		return this;
	}
	toString() {
		return `[${this.x}, ${this.y}]`;
	}
	static angleRad(lhs, rhs) {
		return Math.atan2(lhs.x * rhs.y - lhs.y * rhs.x, lhs.x * rhs.x + lhs.y * rhs.y);
	}
}
_class$7 = Vec2;
Vec2.ZERO = Object.freeze(new _class$7(0, 0));
Vec2.ONE = Object.freeze(new _class$7(1, 1));
Vec2.UP = Object.freeze(new _class$7(0, 1));
Vec2.DOWN = Object.freeze(new _class$7(0, -1));
Vec2.RIGHT = Object.freeze(new _class$7(1, 0));
Vec2.LEFT = Object.freeze(new _class$7(-1, 0));

var _class$6;
class Vec4 {
	constructor(x = 0, y = 0, z = 0, w = 0) {
		this.x = void 0;
		this.y = void 0;
		this.z = void 0;
		this.w = void 0;
		if (x.length === 4) {
			this.x = x[0];
			this.y = x[1];
			this.z = x[2];
			this.w = x[3];
		} else {
			this.x = x;
			this.y = y;
			this.z = z;
			this.w = w;
		}
	}
	add(rhs) {
		this.x += rhs.x;
		this.y += rhs.y;
		this.z += rhs.z;
		this.w += rhs.w;
		return this;
	}
	add2(lhs, rhs) {
		this.x = lhs.x + rhs.x;
		this.y = lhs.y + rhs.y;
		this.z = lhs.z + rhs.z;
		this.w = lhs.w + rhs.w;
		return this;
	}
	addScalar(scalar) {
		this.x += scalar;
		this.y += scalar;
		this.z += scalar;
		this.w += scalar;
		return this;
	}
	addScaled(rhs, scalar) {
		this.x += rhs.x * scalar;
		this.y += rhs.y * scalar;
		this.z += rhs.z * scalar;
		this.w += rhs.w * scalar;
		return this;
	}
	clone() {
		const cstr = this.constructor;
		return new cstr(this.x, this.y, this.z, this.w);
	}
	copy(rhs) {
		this.x = rhs.x;
		this.y = rhs.y;
		this.z = rhs.z;
		this.w = rhs.w;
		return this;
	}
	div(rhs) {
		this.x /= rhs.x;
		this.y /= rhs.y;
		this.z /= rhs.z;
		this.w /= rhs.w;
		return this;
	}
	div2(lhs, rhs) {
		this.x = lhs.x / rhs.x;
		this.y = lhs.y / rhs.y;
		this.z = lhs.z / rhs.z;
		this.w = lhs.w / rhs.w;
		return this;
	}
	divScalar(scalar) {
		this.x /= scalar;
		this.y /= scalar;
		this.z /= scalar;
		this.w /= scalar;
		return this;
	}
	dot(rhs) {
		return this.x * rhs.x + this.y * rhs.y + this.z * rhs.z + this.w * rhs.w;
	}
	equals(rhs) {
		return this.x === rhs.x && this.y === rhs.y && this.z === rhs.z && this.w === rhs.w;
	}
	equalsApprox(rhs, epsilon = 1e-6) {
		return Math.abs(this.x - rhs.x) < epsilon && Math.abs(this.y - rhs.y) < epsilon && Math.abs(this.z - rhs.z) < epsilon && Math.abs(this.w - rhs.w) < epsilon;
	}
	length() {
		return Math.sqrt(this.x * this.x + this.y * this.y + this.z * this.z + this.w * this.w);
	}
	lengthSq() {
		return this.x * this.x + this.y * this.y + this.z * this.z + this.w * this.w;
	}
	lerp(lhs, rhs, alpha) {
		this.x = lhs.x + alpha * (rhs.x - lhs.x);
		this.y = lhs.y + alpha * (rhs.y - lhs.y);
		this.z = lhs.z + alpha * (rhs.z - lhs.z);
		this.w = lhs.w + alpha * (rhs.w - lhs.w);
		return this;
	}
	mul(rhs) {
		this.x *= rhs.x;
		this.y *= rhs.y;
		this.z *= rhs.z;
		this.w *= rhs.w;
		return this;
	}
	mul2(lhs, rhs) {
		this.x = lhs.x * rhs.x;
		this.y = lhs.y * rhs.y;
		this.z = lhs.z * rhs.z;
		this.w = lhs.w * rhs.w;
		return this;
	}
	mulScalar(scalar) {
		this.x *= scalar;
		this.y *= scalar;
		this.z *= scalar;
		this.w *= scalar;
		return this;
	}
	normalize(src = this) {
		const lengthSq = src.x * src.x + src.y * src.y + src.z * src.z + src.w * src.w;
		if (lengthSq > 0) {
			const invLength = 1 / Math.sqrt(lengthSq);
			this.x = src.x * invLength;
			this.y = src.y * invLength;
			this.z = src.z * invLength;
			this.w = src.w * invLength;
		}
		return this;
	}
	floor(src = this) {
		this.x = Math.floor(src.x);
		this.y = Math.floor(src.y);
		this.z = Math.floor(src.z);
		this.w = Math.floor(src.w);
		return this;
	}
	ceil(src = this) {
		this.x = Math.ceil(src.x);
		this.y = Math.ceil(src.y);
		this.z = Math.ceil(src.z);
		this.w = Math.ceil(src.w);
		return this;
	}
	round(src = this) {
		this.x = Math.round(src.x);
		this.y = Math.round(src.y);
		this.z = Math.round(src.z);
		this.w = Math.round(src.w);
		return this;
	}
	min(rhs) {
		if (rhs.x < this.x) this.x = rhs.x;
		if (rhs.y < this.y) this.y = rhs.y;
		if (rhs.z < this.z) this.z = rhs.z;
		if (rhs.w < this.w) this.w = rhs.w;
		return this;
	}
	max(rhs) {
		if (rhs.x > this.x) this.x = rhs.x;
		if (rhs.y > this.y) this.y = rhs.y;
		if (rhs.z > this.z) this.z = rhs.z;
		if (rhs.w > this.w) this.w = rhs.w;
		return this;
	}
	set(x, y, z, w) {
		this.x = x;
		this.y = y;
		this.z = z;
		this.w = w;
		return this;
	}
	sub(rhs) {
		this.x -= rhs.x;
		this.y -= rhs.y;
		this.z -= rhs.z;
		this.w -= rhs.w;
		return this;
	}
	sub2(lhs, rhs) {
		this.x = lhs.x - rhs.x;
		this.y = lhs.y - rhs.y;
		this.z = lhs.z - rhs.z;
		this.w = lhs.w - rhs.w;
		return this;
	}
	subScalar(scalar) {
		this.x -= scalar;
		this.y -= scalar;
		this.z -= scalar;
		this.w -= scalar;
		return this;
	}
	toString() {
		return `[${this.x}, ${this.y}, ${this.z}, ${this.w}]`;
	}
}
_class$6 = Vec4;
Vec4.ZERO = Object.freeze(new _class$6(0, 0, 0, 0));
Vec4.ONE = Object.freeze(new _class$6(1, 1, 1, 1));

var _class$5;
const _halfSize$1 = new Vec2();
const x = new Vec3();
const y = new Vec3();
const z = new Vec3();
const scale = new Vec3();
class Mat4 {
	constructor() {
		this.data = new Float32Array(16);
		this.data[0] = this.data[5] = this.data[10] = this.data[15] = 1;
	}
	static _getPerspectiveHalfSize(halfSize, fov, aspect, znear, fovIsHorizontal) {
		if (fovIsHorizontal) {
			halfSize.x = znear * Math.tan(fov * Math.PI / 360);
			halfSize.y = halfSize.x / aspect;
		} else {
			halfSize.y = znear * Math.tan(fov * Math.PI / 360);
			halfSize.x = halfSize.y * aspect;
		}
	}
	add2(lhs, rhs) {
		const a = lhs.data,
			b = rhs.data,
			r = this.data;
		r[0] = a[0] + b[0];
		r[1] = a[1] + b[1];
		r[2] = a[2] + b[2];
		r[3] = a[3] + b[3];
		r[4] = a[4] + b[4];
		r[5] = a[5] + b[5];
		r[6] = a[6] + b[6];
		r[7] = a[7] + b[7];
		r[8] = a[8] + b[8];
		r[9] = a[9] + b[9];
		r[10] = a[10] + b[10];
		r[11] = a[11] + b[11];
		r[12] = a[12] + b[12];
		r[13] = a[13] + b[13];
		r[14] = a[14] + b[14];
		r[15] = a[15] + b[15];
		return this;
	}
	add(rhs) {
		return this.add2(this, rhs);
	}
	clone() {
		const cstr = this.constructor;
		return new cstr().copy(this);
	}
	copy(rhs) {
		const src = rhs.data,
			dst = this.data;
		dst[0] = src[0];
		dst[1] = src[1];
		dst[2] = src[2];
		dst[3] = src[3];
		dst[4] = src[4];
		dst[5] = src[5];
		dst[6] = src[6];
		dst[7] = src[7];
		dst[8] = src[8];
		dst[9] = src[9];
		dst[10] = src[10];
		dst[11] = src[11];
		dst[12] = src[12];
		dst[13] = src[13];
		dst[14] = src[14];
		dst[15] = src[15];
		return this;
	}
	equals(rhs) {
		const l = this.data,
			r = rhs.data;
		return l[0] === r[0] && l[1] === r[1] && l[2] === r[2] && l[3] === r[3] && l[4] === r[4] && l[5] === r[5] && l[6] === r[6] && l[7] === r[7] && l[8] === r[8] && l[9] === r[9] && l[10] === r[10] && l[11] === r[11] && l[12] === r[12] && l[13] === r[13] && l[14] === r[14] && l[15] === r[15];
	}
	isIdentity() {
		const m = this.data;
		return m[0] === 1 && m[1] === 0 && m[2] === 0 && m[3] === 0 && m[4] === 0 && m[5] === 1 && m[6] === 0 && m[7] === 0 && m[8] === 0 && m[9] === 0 && m[10] === 1 && m[11] === 0 && m[12] === 0 && m[13] === 0 && m[14] === 0 && m[15] === 1;
	}
	mul2(lhs, rhs) {
		const a = lhs.data;
		const b = rhs.data;
		const r = this.data;
		const a00 = a[0];
		const a01 = a[1];
		const a02 = a[2];
		const a03 = a[3];
		const a10 = a[4];
		const a11 = a[5];
		const a12 = a[6];
		const a13 = a[7];
		const a20 = a[8];
		const a21 = a[9];
		const a22 = a[10];
		const a23 = a[11];
		const a30 = a[12];
		const a31 = a[13];
		const a32 = a[14];
		const a33 = a[15];
		let b0, b1, b2, b3;
		b0 = b[0];
		b1 = b[1];
		b2 = b[2];
		b3 = b[3];
		r[0] = a00 * b0 + a10 * b1 + a20 * b2 + a30 * b3;
		r[1] = a01 * b0 + a11 * b1 + a21 * b2 + a31 * b3;
		r[2] = a02 * b0 + a12 * b1 + a22 * b2 + a32 * b3;
		r[3] = a03 * b0 + a13 * b1 + a23 * b2 + a33 * b3;
		b0 = b[4];
		b1 = b[5];
		b2 = b[6];
		b3 = b[7];
		r[4] = a00 * b0 + a10 * b1 + a20 * b2 + a30 * b3;
		r[5] = a01 * b0 + a11 * b1 + a21 * b2 + a31 * b3;
		r[6] = a02 * b0 + a12 * b1 + a22 * b2 + a32 * b3;
		r[7] = a03 * b0 + a13 * b1 + a23 * b2 + a33 * b3;
		b0 = b[8];
		b1 = b[9];
		b2 = b[10];
		b3 = b[11];
		r[8] = a00 * b0 + a10 * b1 + a20 * b2 + a30 * b3;
		r[9] = a01 * b0 + a11 * b1 + a21 * b2 + a31 * b3;
		r[10] = a02 * b0 + a12 * b1 + a22 * b2 + a32 * b3;
		r[11] = a03 * b0 + a13 * b1 + a23 * b2 + a33 * b3;
		b0 = b[12];
		b1 = b[13];
		b2 = b[14];
		b3 = b[15];
		r[12] = a00 * b0 + a10 * b1 + a20 * b2 + a30 * b3;
		r[13] = a01 * b0 + a11 * b1 + a21 * b2 + a31 * b3;
		r[14] = a02 * b0 + a12 * b1 + a22 * b2 + a32 * b3;
		r[15] = a03 * b0 + a13 * b1 + a23 * b2 + a33 * b3;
		return this;
	}
	mulAffine2(lhs, rhs) {
		const a = lhs.data;
		const b = rhs.data;
		const r = this.data;
		const a00 = a[0];
		const a01 = a[1];
		const a02 = a[2];
		const a10 = a[4];
		const a11 = a[5];
		const a12 = a[6];
		const a20 = a[8];
		const a21 = a[9];
		const a22 = a[10];
		const a30 = a[12];
		const a31 = a[13];
		const a32 = a[14];
		let b0, b1, b2;
		b0 = b[0];
		b1 = b[1];
		b2 = b[2];
		r[0] = a00 * b0 + a10 * b1 + a20 * b2;
		r[1] = a01 * b0 + a11 * b1 + a21 * b2;
		r[2] = a02 * b0 + a12 * b1 + a22 * b2;
		r[3] = 0;
		b0 = b[4];
		b1 = b[5];
		b2 = b[6];
		r[4] = a00 * b0 + a10 * b1 + a20 * b2;
		r[5] = a01 * b0 + a11 * b1 + a21 * b2;
		r[6] = a02 * b0 + a12 * b1 + a22 * b2;
		r[7] = 0;
		b0 = b[8];
		b1 = b[9];
		b2 = b[10];
		r[8] = a00 * b0 + a10 * b1 + a20 * b2;
		r[9] = a01 * b0 + a11 * b1 + a21 * b2;
		r[10] = a02 * b0 + a12 * b1 + a22 * b2;
		r[11] = 0;
		b0 = b[12];
		b1 = b[13];
		b2 = b[14];
		r[12] = a00 * b0 + a10 * b1 + a20 * b2 + a30;
		r[13] = a01 * b0 + a11 * b1 + a21 * b2 + a31;
		r[14] = a02 * b0 + a12 * b1 + a22 * b2 + a32;
		r[15] = 1;
		return this;
	}
	mul(rhs) {
		return this.mul2(this, rhs);
	}
	transformPoint(vec, res = new Vec3()) {
		const m = this.data;
		const x = vec.x;
		const y = vec.y;
		const z = vec.z;
		res.x = x * m[0] + y * m[4] + z * m[8] + m[12];
		res.y = x * m[1] + y * m[5] + z * m[9] + m[13];
		res.z = x * m[2] + y * m[6] + z * m[10] + m[14];
		return res;
	}
	transformVector(vec, res = new Vec3()) {
		const m = this.data;
		const x = vec.x;
		const y = vec.y;
		const z = vec.z;
		res.x = x * m[0] + y * m[4] + z * m[8];
		res.y = x * m[1] + y * m[5] + z * m[9];
		res.z = x * m[2] + y * m[6] + z * m[10];
		return res;
	}
	transformVec4(vec, res = new Vec4()) {
		const m = this.data;
		const x = vec.x;
		const y = vec.y;
		const z = vec.z;
		const w = vec.w;
		res.x = x * m[0] + y * m[4] + z * m[8] + w * m[12];
		res.y = x * m[1] + y * m[5] + z * m[9] + w * m[13];
		res.z = x * m[2] + y * m[6] + z * m[10] + w * m[14];
		res.w = x * m[3] + y * m[7] + z * m[11] + w * m[15];
		return res;
	}
	setLookAt(position, target, up) {
		z.sub2(position, target).normalize();
		y.copy(up).normalize();
		x.cross(y, z).normalize();
		y.cross(z, x);
		const r = this.data;
		r[0] = x.x;
		r[1] = x.y;
		r[2] = x.z;
		r[3] = 0;
		r[4] = y.x;
		r[5] = y.y;
		r[6] = y.z;
		r[7] = 0;
		r[8] = z.x;
		r[9] = z.y;
		r[10] = z.z;
		r[11] = 0;
		r[12] = position.x;
		r[13] = position.y;
		r[14] = position.z;
		r[15] = 1;
		return this;
	}
	setFrustum(left, right, bottom, top, znear, zfar) {
		const temp1 = 2 * znear;
		const temp2 = right - left;
		const temp3 = top - bottom;
		const temp4 = zfar - znear;
		const r = this.data;
		r[0] = temp1 / temp2;
		r[1] = 0;
		r[2] = 0;
		r[3] = 0;
		r[4] = 0;
		r[5] = temp1 / temp3;
		r[6] = 0;
		r[7] = 0;
		r[8] = (right + left) / temp2;
		r[9] = (top + bottom) / temp3;
		r[10] = (-zfar - znear) / temp4;
		r[11] = -1;
		r[12] = 0;
		r[13] = 0;
		r[14] = -temp1 * zfar / temp4;
		r[15] = 0;
		return this;
	}
	setPerspective(fov, aspect, znear, zfar, fovIsHorizontal) {
		Mat4._getPerspectiveHalfSize(_halfSize$1, fov, aspect, znear, fovIsHorizontal);
		return this.setFrustum(-_halfSize$1.x, _halfSize$1.x, -_halfSize$1.y, _halfSize$1.y, znear, zfar);
	}
	setOrtho(left, right, bottom, top, near, far) {
		const r = this.data;
		r[0] = 2 / (right - left);
		r[1] = 0;
		r[2] = 0;
		r[3] = 0;
		r[4] = 0;
		r[5] = 2 / (top - bottom);
		r[6] = 0;
		r[7] = 0;
		r[8] = 0;
		r[9] = 0;
		r[10] = -2 / (far - near);
		r[11] = 0;
		r[12] = -(right + left) / (right - left);
		r[13] = -(top + bottom) / (top - bottom);
		r[14] = -(far + near) / (far - near);
		r[15] = 1;
		return this;
	}
	setFromAxisAngle(axis, angle) {
		angle *= math.DEG_TO_RAD;
		const x = axis.x;
		const y = axis.y;
		const z = axis.z;
		const c = Math.cos(angle);
		const s = Math.sin(angle);
		const t = 1 - c;
		const tx = t * x;
		const ty = t * y;
		const m = this.data;
		m[0] = tx * x + c;
		m[1] = tx * y + s * z;
		m[2] = tx * z - s * y;
		m[3] = 0;
		m[4] = tx * y - s * z;
		m[5] = ty * y + c;
		m[6] = ty * z + s * x;
		m[7] = 0;
		m[8] = tx * z + s * y;
		m[9] = ty * z - x * s;
		m[10] = t * z * z + c;
		m[11] = 0;
		m[12] = 0;
		m[13] = 0;
		m[14] = 0;
		m[15] = 1;
		return this;
	}
	setTranslate(x, y, z) {
		const m = this.data;
		m[0] = 1;
		m[1] = 0;
		m[2] = 0;
		m[3] = 0;
		m[4] = 0;
		m[5] = 1;
		m[6] = 0;
		m[7] = 0;
		m[8] = 0;
		m[9] = 0;
		m[10] = 1;
		m[11] = 0;
		m[12] = x;
		m[13] = y;
		m[14] = z;
		m[15] = 1;
		return this;
	}
	setScale(x, y, z) {
		const m = this.data;
		m[0] = x;
		m[1] = 0;
		m[2] = 0;
		m[3] = 0;
		m[4] = 0;
		m[5] = y;
		m[6] = 0;
		m[7] = 0;
		m[8] = 0;
		m[9] = 0;
		m[10] = z;
		m[11] = 0;
		m[12] = 0;
		m[13] = 0;
		m[14] = 0;
		m[15] = 1;
		return this;
	}
	setViewport(x, y, width, height) {
		const m = this.data;
		m[0] = width * 0.5;
		m[1] = 0;
		m[2] = 0;
		m[3] = 0;
		m[4] = 0;
		m[5] = height * 0.5;
		m[6] = 0;
		m[7] = 0;
		m[8] = 0;
		m[9] = 0;
		m[10] = 0.5;
		m[11] = 0;
		m[12] = x + width * 0.5;
		m[13] = y + height * 0.5;
		m[14] = 0.5;
		m[15] = 1;
		return this;
	}
	setReflection(normal, distance) {
		const a = normal.x;
		const b = normal.y;
		const c = normal.z;
		const data = this.data;
		data[0] = 1.0 - 2 * a * a;
		data[1] = -2 * a * b;
		data[2] = -2 * a * c;
		data[3] = 0;
		data[4] = -2 * a * b;
		data[5] = 1.0 - 2 * b * b;
		data[6] = -2 * b * c;
		data[7] = 0;
		data[8] = -2 * a * c;
		data[9] = -2 * b * c;
		data[10] = 1.0 - 2 * c * c;
		data[11] = 0;
		data[12] = -2 * a * distance;
		data[13] = -2 * b * distance;
		data[14] = -2 * c * distance;
		data[15] = 1;
		return this;
	}
	invert(src = this) {
		const s = src.data;
		const a00 = s[0];
		const a01 = s[1];
		const a02 = s[2];
		const a03 = s[3];
		const a10 = s[4];
		const a11 = s[5];
		const a12 = s[6];
		const a13 = s[7];
		const a20 = s[8];
		const a21 = s[9];
		const a22 = s[10];
		const a23 = s[11];
		const a30 = s[12];
		const a31 = s[13];
		const a32 = s[14];
		const a33 = s[15];
		const b00 = a00 * a11 - a01 * a10;
		const b01 = a00 * a12 - a02 * a10;
		const b02 = a00 * a13 - a03 * a10;
		const b03 = a01 * a12 - a02 * a11;
		const b04 = a01 * a13 - a03 * a11;
		const b05 = a02 * a13 - a03 * a12;
		const b06 = a20 * a31 - a21 * a30;
		const b07 = a20 * a32 - a22 * a30;
		const b08 = a20 * a33 - a23 * a30;
		const b09 = a21 * a32 - a22 * a31;
		const b10 = a21 * a33 - a23 * a31;
		const b11 = a22 * a33 - a23 * a32;
		const det = b00 * b11 - b01 * b10 + b02 * b09 + b03 * b08 - b04 * b07 + b05 * b06;
		if (det === 0) {
			this.setIdentity();
		} else {
			const invDet = 1 / det;
			const t = this.data;
			t[0] = (a11 * b11 - a12 * b10 + a13 * b09) * invDet;
			t[1] = (-a01 * b11 + a02 * b10 - a03 * b09) * invDet;
			t[2] = (a31 * b05 - a32 * b04 + a33 * b03) * invDet;
			t[3] = (-a21 * b05 + a22 * b04 - a23 * b03) * invDet;
			t[4] = (-a10 * b11 + a12 * b08 - a13 * b07) * invDet;
			t[5] = (a00 * b11 - a02 * b08 + a03 * b07) * invDet;
			t[6] = (-a30 * b05 + a32 * b02 - a33 * b01) * invDet;
			t[7] = (a20 * b05 - a22 * b02 + a23 * b01) * invDet;
			t[8] = (a10 * b10 - a11 * b08 + a13 * b06) * invDet;
			t[9] = (-a00 * b10 + a01 * b08 - a03 * b06) * invDet;
			t[10] = (a30 * b04 - a31 * b02 + a33 * b00) * invDet;
			t[11] = (-a20 * b04 + a21 * b02 - a23 * b00) * invDet;
			t[12] = (-a10 * b09 + a11 * b07 - a12 * b06) * invDet;
			t[13] = (a00 * b09 - a01 * b07 + a02 * b06) * invDet;
			t[14] = (-a30 * b03 + a31 * b01 - a32 * b00) * invDet;
			t[15] = (a20 * b03 - a21 * b01 + a22 * b00) * invDet;
		}
		return this;
	}
	set(src) {
		const dst = this.data;
		dst[0] = src[0];
		dst[1] = src[1];
		dst[2] = src[2];
		dst[3] = src[3];
		dst[4] = src[4];
		dst[5] = src[5];
		dst[6] = src[6];
		dst[7] = src[7];
		dst[8] = src[8];
		dst[9] = src[9];
		dst[10] = src[10];
		dst[11] = src[11];
		dst[12] = src[12];
		dst[13] = src[13];
		dst[14] = src[14];
		dst[15] = src[15];
		return this;
	}
	setIdentity() {
		const m = this.data;
		m[0] = 1;
		m[1] = 0;
		m[2] = 0;
		m[3] = 0;
		m[4] = 0;
		m[5] = 1;
		m[6] = 0;
		m[7] = 0;
		m[8] = 0;
		m[9] = 0;
		m[10] = 1;
		m[11] = 0;
		m[12] = 0;
		m[13] = 0;
		m[14] = 0;
		m[15] = 1;
		return this;
	}
	setTRS(t, r, s) {
		const qx = r.x;
		const qy = r.y;
		const qz = r.z;
		const qw = r.w;
		const sx = s.x;
		const sy = s.y;
		const sz = s.z;
		const x2 = qx + qx;
		const y2 = qy + qy;
		const z2 = qz + qz;
		const xx = qx * x2;
		const xy = qx * y2;
		const xz = qx * z2;
		const yy = qy * y2;
		const yz = qy * z2;
		const zz = qz * z2;
		const wx = qw * x2;
		const wy = qw * y2;
		const wz = qw * z2;
		const m = this.data;
		m[0] = (1 - (yy + zz)) * sx;
		m[1] = (xy + wz) * sx;
		m[2] = (xz - wy) * sx;
		m[3] = 0;
		m[4] = (xy - wz) * sy;
		m[5] = (1 - (xx + zz)) * sy;
		m[6] = (yz + wx) * sy;
		m[7] = 0;
		m[8] = (xz + wy) * sz;
		m[9] = (yz - wx) * sz;
		m[10] = (1 - (xx + yy)) * sz;
		m[11] = 0;
		m[12] = t.x;
		m[13] = t.y;
		m[14] = t.z;
		m[15] = 1;
		return this;
	}
	transpose(src = this) {
		const s = src.data;
		const t = this.data;
		if (s === t) {
			let tmp;
			tmp = s[1];
			t[1] = s[4];
			t[4] = tmp;
			tmp = s[2];
			t[2] = s[8];
			t[8] = tmp;
			tmp = s[3];
			t[3] = s[12];
			t[12] = tmp;
			tmp = s[6];
			t[6] = s[9];
			t[9] = tmp;
			tmp = s[7];
			t[7] = s[13];
			t[13] = tmp;
			tmp = s[11];
			t[11] = s[14];
			t[14] = tmp;
		} else {
			t[0] = s[0];
			t[1] = s[4];
			t[2] = s[8];
			t[3] = s[12];
			t[4] = s[1];
			t[5] = s[5];
			t[6] = s[9];
			t[7] = s[13];
			t[8] = s[2];
			t[9] = s[6];
			t[10] = s[10];
			t[11] = s[14];
			t[12] = s[3];
			t[13] = s[7];
			t[14] = s[11];
			t[15] = s[15];
		}
		return this;
	}
	getTranslation(t = new Vec3()) {
		return t.set(this.data[12], this.data[13], this.data[14]);
	}
	getX(x = new Vec3()) {
		return x.set(this.data[0], this.data[1], this.data[2]);
	}
	getY(y = new Vec3()) {
		return y.set(this.data[4], this.data[5], this.data[6]);
	}
	getZ(z = new Vec3()) {
		return z.set(this.data[8], this.data[9], this.data[10]);
	}
	getScale(scale = new Vec3()) {
		this.getX(x);
		this.getY(y);
		this.getZ(z);
		scale.set(x.length(), y.length(), z.length());
		return scale;
	}
	get scaleSign() {
		this.getX(x);
		this.getY(y);
		this.getZ(z);
		x.cross(x, y);
		return x.dot(z) < 0 ? -1 : 1;
	}
	setFromEulerAngles(ex, ey, ez) {
		ex *= math.DEG_TO_RAD;
		ey *= math.DEG_TO_RAD;
		ez *= math.DEG_TO_RAD;
		const s1 = Math.sin(-ex);
		const c1 = Math.cos(-ex);
		const s2 = Math.sin(-ey);
		const c2 = Math.cos(-ey);
		const s3 = Math.sin(-ez);
		const c3 = Math.cos(-ez);
		const m = this.data;
		m[0] = c2 * c3;
		m[1] = -c2 * s3;
		m[2] = s2;
		m[3] = 0;
		m[4] = c1 * s3 + c3 * s1 * s2;
		m[5] = c1 * c3 - s1 * s2 * s3;
		m[6] = -c2 * s1;
		m[7] = 0;
		m[8] = s1 * s3 - c1 * c3 * s2;
		m[9] = c3 * s1 + c1 * s2 * s3;
		m[10] = c1 * c2;
		m[11] = 0;
		m[12] = 0;
		m[13] = 0;
		m[14] = 0;
		m[15] = 1;
		return this;
	}
	getEulerAngles(eulers = new Vec3()) {
		this.getScale(scale);
		const sx = scale.x;
		const sy = scale.y;
		const sz = scale.z;
		if (sx === 0 || sy === 0 || sz === 0) return eulers.set(0, 0, 0);
		const m = this.data;
		const y = Math.asin(-m[2] / sx);
		const halfPi = Math.PI * 0.5;
		let x, z;
		if (y < halfPi) {
			if (y > -halfPi) {
				x = Math.atan2(m[6] / sy, m[10] / sz);
				z = Math.atan2(m[1] / sx, m[0] / sx);
			} else {
				z = 0;
				x = -Math.atan2(m[4] / sy, m[5] / sy);
			}
		} else {
			z = 0;
			x = Math.atan2(m[4] / sy, m[5] / sy);
		}
		return eulers.set(x, y, z).mulScalar(math.RAD_TO_DEG);
	}
	toString() {
		return '[' + this.data.join(', ') + ']';
	}
}
_class$5 = Mat4;
Mat4.IDENTITY = Object.freeze(new _class$5());
Mat4.ZERO = Object.freeze(new _class$5().set([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]));

var _class$4;
class Quat {
	constructor(x = 0, y = 0, z = 0, w = 1) {
		this.x = void 0;
		this.y = void 0;
		this.z = void 0;
		this.w = void 0;
		if (x.length === 4) {
			this.x = x[0];
			this.y = x[1];
			this.z = x[2];
			this.w = x[3];
		} else {
			this.x = x;
			this.y = y;
			this.z = z;
			this.w = w;
		}
	}
	clone() {
		const cstr = this.constructor;
		return new cstr(this.x, this.y, this.z, this.w);
	}
	conjugate(src = this) {
		this.x = src.x * -1;
		this.y = src.y * -1;
		this.z = src.z * -1;
		this.w = src.w;
		return this;
	}
	copy(rhs) {
		this.x = rhs.x;
		this.y = rhs.y;
		this.z = rhs.z;
		this.w = rhs.w;
		return this;
	}
	equals(rhs) {
		return this.x === rhs.x && this.y === rhs.y && this.z === rhs.z && this.w === rhs.w;
	}
	equalsApprox(rhs, epsilon = 1e-6) {
		return Math.abs(this.x - rhs.x) < epsilon && Math.abs(this.y - rhs.y) < epsilon && Math.abs(this.z - rhs.z) < epsilon && Math.abs(this.w - rhs.w) < epsilon;
	}
	getAxisAngle(axis) {
		let rad = Math.acos(this.w) * 2;
		const s = Math.sin(rad / 2);
		if (s !== 0) {
			axis.x = this.x / s;
			axis.y = this.y / s;
			axis.z = this.z / s;
			if (axis.x < 0 || axis.y < 0 || axis.z < 0) {
				axis.x *= -1;
				axis.y *= -1;
				axis.z *= -1;
				rad *= -1;
			}
		} else {
			axis.x = 1;
			axis.y = 0;
			axis.z = 0;
		}
		return rad * math.RAD_TO_DEG;
	}
	getEulerAngles(eulers = new Vec3()) {
		let x, y, z;
		const qx = this.x;
		const qy = this.y;
		const qz = this.z;
		const qw = this.w;
		const a2 = 2 * (qw * qy - qx * qz);
		if (a2 <= -0.99999) {
			x = 2 * Math.atan2(qx, qw);
			y = -Math.PI / 2;
			z = 0;
		} else if (a2 >= 0.99999) {
			x = 2 * Math.atan2(qx, qw);
			y = Math.PI / 2;
			z = 0;
		} else {
			x = Math.atan2(2 * (qw * qx + qy * qz), 1 - 2 * (qx * qx + qy * qy));
			y = Math.asin(a2);
			z = Math.atan2(2 * (qw * qz + qx * qy), 1 - 2 * (qy * qy + qz * qz));
		}
		return eulers.set(x, y, z).mulScalar(math.RAD_TO_DEG);
	}
	invert(src = this) {
		return this.conjugate(src).normalize();
	}
	length() {
		return Math.sqrt(this.x * this.x + this.y * this.y + this.z * this.z + this.w * this.w);
	}
	lengthSq() {
		return this.x * this.x + this.y * this.y + this.z * this.z + this.w * this.w;
	}
	mul(rhs) {
		const q1x = this.x;
		const q1y = this.y;
		const q1z = this.z;
		const q1w = this.w;
		const q2x = rhs.x;
		const q2y = rhs.y;
		const q2z = rhs.z;
		const q2w = rhs.w;
		this.x = q1w * q2x + q1x * q2w + q1y * q2z - q1z * q2y;
		this.y = q1w * q2y + q1y * q2w + q1z * q2x - q1x * q2z;
		this.z = q1w * q2z + q1z * q2w + q1x * q2y - q1y * q2x;
		this.w = q1w * q2w - q1x * q2x - q1y * q2y - q1z * q2z;
		return this;
	}
	mul2(lhs, rhs) {
		const q1x = lhs.x;
		const q1y = lhs.y;
		const q1z = lhs.z;
		const q1w = lhs.w;
		const q2x = rhs.x;
		const q2y = rhs.y;
		const q2z = rhs.z;
		const q2w = rhs.w;
		this.x = q1w * q2x + q1x * q2w + q1y * q2z - q1z * q2y;
		this.y = q1w * q2y + q1y * q2w + q1z * q2x - q1x * q2z;
		this.z = q1w * q2z + q1z * q2w + q1x * q2y - q1y * q2x;
		this.w = q1w * q2w - q1x * q2x - q1y * q2y - q1z * q2z;
		return this;
	}
	normalize(src = this) {
		let len = src.length();
		if (len === 0) {
			this.x = this.y = this.z = 0;
			this.w = 1;
		} else {
			len = 1 / len;
			this.x = src.x * len;
			this.y = src.y * len;
			this.z = src.z * len;
			this.w = src.w * len;
		}
		return this;
	}
	set(x, y, z, w) {
		this.x = x;
		this.y = y;
		this.z = z;
		this.w = w;
		return this;
	}
	setFromAxisAngle(axis, angle) {
		angle *= 0.5 * math.DEG_TO_RAD;
		const sa = Math.sin(angle);
		const ca = Math.cos(angle);
		this.x = sa * axis.x;
		this.y = sa * axis.y;
		this.z = sa * axis.z;
		this.w = ca;
		return this;
	}
	setFromEulerAngles(ex, ey, ez) {
		if (ex instanceof Vec3) {
			const vec = ex;
			ex = vec.x;
			ey = vec.y;
			ez = vec.z;
		}
		const halfToRad = 0.5 * math.DEG_TO_RAD;
		ex *= halfToRad;
		ey *= halfToRad;
		ez *= halfToRad;
		const sx = Math.sin(ex);
		const cx = Math.cos(ex);
		const sy = Math.sin(ey);
		const cy = Math.cos(ey);
		const sz = Math.sin(ez);
		const cz = Math.cos(ez);
		this.x = sx * cy * cz - cx * sy * sz;
		this.y = cx * sy * cz + sx * cy * sz;
		this.z = cx * cy * sz - sx * sy * cz;
		this.w = cx * cy * cz + sx * sy * sz;
		return this;
	}
	setFromMat4(m) {
		let m00, m01, m02, m10, m11, m12, m20, m21, m22, s, rs, lx, ly, lz;
		m = m.data;
		m00 = m[0];
		m01 = m[1];
		m02 = m[2];
		m10 = m[4];
		m11 = m[5];
		m12 = m[6];
		m20 = m[8];
		m21 = m[9];
		m22 = m[10];
		lx = m00 * m00 + m01 * m01 + m02 * m02;
		if (lx === 0) return this;
		lx = 1 / Math.sqrt(lx);
		ly = m10 * m10 + m11 * m11 + m12 * m12;
		if (ly === 0) return this;
		ly = 1 / Math.sqrt(ly);
		lz = m20 * m20 + m21 * m21 + m22 * m22;
		if (lz === 0) return this;
		lz = 1 / Math.sqrt(lz);
		m00 *= lx;
		m01 *= lx;
		m02 *= lx;
		m10 *= ly;
		m11 *= ly;
		m12 *= ly;
		m20 *= lz;
		m21 *= lz;
		m22 *= lz;
		const tr = m00 + m11 + m22;
		if (tr >= 0) {
			s = Math.sqrt(tr + 1);
			this.w = s * 0.5;
			s = 0.5 / s;
			this.x = (m12 - m21) * s;
			this.y = (m20 - m02) * s;
			this.z = (m01 - m10) * s;
		} else {
			if (m00 > m11) {
				if (m00 > m22) {
					rs = m00 - (m11 + m22) + 1;
					rs = Math.sqrt(rs);
					this.x = rs * 0.5;
					rs = 0.5 / rs;
					this.w = (m12 - m21) * rs;
					this.y = (m01 + m10) * rs;
					this.z = (m02 + m20) * rs;
				} else {
					rs = m22 - (m00 + m11) + 1;
					rs = Math.sqrt(rs);
					this.z = rs * 0.5;
					rs = 0.5 / rs;
					this.w = (m01 - m10) * rs;
					this.x = (m20 + m02) * rs;
					this.y = (m21 + m12) * rs;
				}
			} else if (m11 > m22) {
				rs = m11 - (m22 + m00) + 1;
				rs = Math.sqrt(rs);
				this.y = rs * 0.5;
				rs = 0.5 / rs;
				this.w = (m20 - m02) * rs;
				this.z = (m12 + m21) * rs;
				this.x = (m10 + m01) * rs;
			} else {
				rs = m22 - (m00 + m11) + 1;
				rs = Math.sqrt(rs);
				this.z = rs * 0.5;
				rs = 0.5 / rs;
				this.w = (m01 - m10) * rs;
				this.x = (m20 + m02) * rs;
				this.y = (m21 + m12) * rs;
			}
		}
		return this;
	}
	setFromDirections(from, to) {
		const dotProduct = 1 + from.dot(to);
		if (dotProduct < Number.EPSILON) {
			if (Math.abs(from.x) > Math.abs(from.y)) {
				this.x = -from.z;
				this.y = 0;
				this.z = from.x;
				this.w = 0;
			} else {
				this.x = 0;
				this.y = -from.z;
				this.z = from.y;
				this.w = 0;
			}
		} else {
			this.x = from.y * to.z - from.z * to.y;
			this.y = from.z * to.x - from.x * to.z;
			this.z = from.x * to.y - from.y * to.x;
			this.w = dotProduct;
		}
		return this.normalize();
	}
	slerp(lhs, rhs, alpha) {
		const lx = lhs.x;
		const ly = lhs.y;
		const lz = lhs.z;
		const lw = lhs.w;
		let rx = rhs.x;
		let ry = rhs.y;
		let rz = rhs.z;
		let rw = rhs.w;
		let cosHalfTheta = lw * rw + lx * rx + ly * ry + lz * rz;
		if (cosHalfTheta < 0) {
			rw = -rw;
			rx = -rx;
			ry = -ry;
			rz = -rz;
			cosHalfTheta = -cosHalfTheta;
		}
		if (Math.abs(cosHalfTheta) >= 1) {
			this.w = lw;
			this.x = lx;
			this.y = ly;
			this.z = lz;
			return this;
		}
		const halfTheta = Math.acos(cosHalfTheta);
		const sinHalfTheta = Math.sqrt(1 - cosHalfTheta * cosHalfTheta);
		if (Math.abs(sinHalfTheta) < 0.001) {
			this.w = lw * 0.5 + rw * 0.5;
			this.x = lx * 0.5 + rx * 0.5;
			this.y = ly * 0.5 + ry * 0.5;
			this.z = lz * 0.5 + rz * 0.5;
			return this;
		}
		const ratioA = Math.sin((1 - alpha) * halfTheta) / sinHalfTheta;
		const ratioB = Math.sin(alpha * halfTheta) / sinHalfTheta;
		this.w = lw * ratioA + rw * ratioB;
		this.x = lx * ratioA + rx * ratioB;
		this.y = ly * ratioA + ry * ratioB;
		this.z = lz * ratioA + rz * ratioB;
		return this;
	}
	transformVector(vec, res = new Vec3()) {
		const x = vec.x,
			y = vec.y,
			z = vec.z;
		const qx = this.x,
			qy = this.y,
			qz = this.z,
			qw = this.w;
		const ix = qw * x + qy * z - qz * y;
		const iy = qw * y + qz * x - qx * z;
		const iz = qw * z + qx * y - qy * x;
		const iw = -qx * x - qy * y - qz * z;
		res.x = ix * qw + iw * -qx + iy * -qz - iz * -qy;
		res.y = iy * qw + iw * -qy + iz * -qx - ix * -qz;
		res.z = iz * qw + iw * -qz + ix * -qy - iy * -qx;
		return res;
	}
	toString() {
		return `[${this.x}, ${this.y}, ${this.z}, ${this.w}]`;
	}
}
_class$4 = Quat;
Quat.IDENTITY = Object.freeze(new _class$4(0, 0, 0, 1));
Quat.ZERO = Object.freeze(new _class$4(0, 0, 0, 0));

const tmpVecA$1 = new Vec3();
const tmpVecB$1 = new Vec3();
const tmpVecC = new Vec3();
const tmpVecD = new Vec3();
const tmpVecE = new Vec3();
class BoundingBox {
	constructor(center = new Vec3(), halfExtents = new Vec3(0.5, 0.5, 0.5)) {
		this.center = void 0;
		this.halfExtents = void 0;
		this._min = new Vec3();
		this._max = new Vec3();
		this.center = center;
		this.halfExtents = halfExtents;
	}
	add(other) {
		const tc = this.center;
		const tcx = tc.x;
		const tcy = tc.y;
		const tcz = tc.z;
		const th = this.halfExtents;
		const thx = th.x;
		const thy = th.y;
		const thz = th.z;
		let tminx = tcx - thx;
		let tmaxx = tcx + thx;
		let tminy = tcy - thy;
		let tmaxy = tcy + thy;
		let tminz = tcz - thz;
		let tmaxz = tcz + thz;
		const oc = other.center;
		const ocx = oc.x;
		const ocy = oc.y;
		const ocz = oc.z;
		const oh = other.halfExtents;
		const ohx = oh.x;
		const ohy = oh.y;
		const ohz = oh.z;
		const ominx = ocx - ohx;
		const omaxx = ocx + ohx;
		const ominy = ocy - ohy;
		const omaxy = ocy + ohy;
		const ominz = ocz - ohz;
		const omaxz = ocz + ohz;
		if (ominx < tminx) tminx = ominx;
		if (omaxx > tmaxx) tmaxx = omaxx;
		if (ominy < tminy) tminy = ominy;
		if (omaxy > tmaxy) tmaxy = omaxy;
		if (ominz < tminz) tminz = ominz;
		if (omaxz > tmaxz) tmaxz = omaxz;
		tc.x = (tminx + tmaxx) * 0.5;
		tc.y = (tminy + tmaxy) * 0.5;
		tc.z = (tminz + tmaxz) * 0.5;
		th.x = (tmaxx - tminx) * 0.5;
		th.y = (tmaxy - tminy) * 0.5;
		th.z = (tmaxz - tminz) * 0.5;
	}
	copy(src) {
		this.center.copy(src.center);
		this.halfExtents.copy(src.halfExtents);
	}
	clone() {
		return new BoundingBox(this.center.clone(), this.halfExtents.clone());
	}
	intersects(other) {
		const aMax = this.getMax();
		const aMin = this.getMin();
		const bMax = other.getMax();
		const bMin = other.getMin();
		return aMin.x <= bMax.x && aMax.x >= bMin.x && aMin.y <= bMax.y && aMax.y >= bMin.y && aMin.z <= bMax.z && aMax.z >= bMin.z;
	}
	_intersectsRay(ray, point) {
		const tMin = tmpVecA$1.copy(this.getMin()).sub(ray.origin);
		const tMax = tmpVecB$1.copy(this.getMax()).sub(ray.origin);
		const dir = ray.direction;
		if (dir.x === 0) {
			tMin.x = tMin.x < 0 ? -Number.MAX_VALUE : Number.MAX_VALUE;
			tMax.x = tMax.x < 0 ? -Number.MAX_VALUE : Number.MAX_VALUE;
		} else {
			tMin.x /= dir.x;
			tMax.x /= dir.x;
		}
		if (dir.y === 0) {
			tMin.y = tMin.y < 0 ? -Number.MAX_VALUE : Number.MAX_VALUE;
			tMax.y = tMax.y < 0 ? -Number.MAX_VALUE : Number.MAX_VALUE;
		} else {
			tMin.y /= dir.y;
			tMax.y /= dir.y;
		}
		if (dir.z === 0) {
			tMin.z = tMin.z < 0 ? -Number.MAX_VALUE : Number.MAX_VALUE;
			tMax.z = tMax.z < 0 ? -Number.MAX_VALUE : Number.MAX_VALUE;
		} else {
			tMin.z /= dir.z;
			tMax.z /= dir.z;
		}
		const realMin = tmpVecC.set(Math.min(tMin.x, tMax.x), Math.min(tMin.y, tMax.y), Math.min(tMin.z, tMax.z));
		const realMax = tmpVecD.set(Math.max(tMin.x, tMax.x), Math.max(tMin.y, tMax.y), Math.max(tMin.z, tMax.z));
		const minMax = Math.min(Math.min(realMax.x, realMax.y), realMax.z);
		const maxMin = Math.max(Math.max(realMin.x, realMin.y), realMin.z);
		const intersects = minMax >= maxMin && maxMin >= 0;
		if (intersects) point.copy(ray.direction).mulScalar(maxMin).add(ray.origin);
		return intersects;
	}
	_fastIntersectsRay(ray) {
		const diff = tmpVecA$1;
		const cross = tmpVecB$1;
		const prod = tmpVecC;
		const absDiff = tmpVecD;
		const absDir = tmpVecE;
		const rayDir = ray.direction;
		diff.sub2(ray.origin, this.center);
		absDiff.set(Math.abs(diff.x), Math.abs(diff.y), Math.abs(diff.z));
		prod.mul2(diff, rayDir);
		if (absDiff.x > this.halfExtents.x && prod.x >= 0) return false;
		if (absDiff.y > this.halfExtents.y && prod.y >= 0) return false;
		if (absDiff.z > this.halfExtents.z && prod.z >= 0) return false;
		absDir.set(Math.abs(rayDir.x), Math.abs(rayDir.y), Math.abs(rayDir.z));
		cross.cross(rayDir, diff);
		cross.set(Math.abs(cross.x), Math.abs(cross.y), Math.abs(cross.z));
		if (cross.x > this.halfExtents.y * absDir.z + this.halfExtents.z * absDir.y) return false;
		if (cross.y > this.halfExtents.x * absDir.z + this.halfExtents.z * absDir.x) return false;
		if (cross.z > this.halfExtents.x * absDir.y + this.halfExtents.y * absDir.x) return false;
		return true;
	}
	intersectsRay(ray, point) {
		if (point) {
			return this._intersectsRay(ray, point);
		}
		return this._fastIntersectsRay(ray);
	}
	setMinMax(min, max) {
		this.center.add2(max, min).mulScalar(0.5);
		this.halfExtents.sub2(max, min).mulScalar(0.5);
	}
	getMin() {
		return this._min.copy(this.center).sub(this.halfExtents);
	}
	getMax() {
		return this._max.copy(this.center).add(this.halfExtents);
	}
	containsPoint(point) {
		const min = this.getMin();
		const max = this.getMax();
		if (point.x < min.x || point.x > max.x || point.y < min.y || point.y > max.y || point.z < min.z || point.z > max.z) {
			return false;
		}
		return true;
	}
	setFromTransformedAabb(aabb, m, ignoreScale = false) {
		const ac = aabb.center;
		const ar = aabb.halfExtents;
		const d = m.data;
		let mx0 = d[0];
		let mx1 = d[4];
		let mx2 = d[8];
		let my0 = d[1];
		let my1 = d[5];
		let my2 = d[9];
		let mz0 = d[2];
		let mz1 = d[6];
		let mz2 = d[10];
		if (ignoreScale) {
			let lengthSq = mx0 * mx0 + mx1 * mx1 + mx2 * mx2;
			if (lengthSq > 0) {
				const invLength = 1 / Math.sqrt(lengthSq);
				mx0 *= invLength;
				mx1 *= invLength;
				mx2 *= invLength;
			}
			lengthSq = my0 * my0 + my1 * my1 + my2 * my2;
			if (lengthSq > 0) {
				const invLength = 1 / Math.sqrt(lengthSq);
				my0 *= invLength;
				my1 *= invLength;
				my2 *= invLength;
			}
			lengthSq = mz0 * mz0 + mz1 * mz1 + mz2 * mz2;
			if (lengthSq > 0) {
				const invLength = 1 / Math.sqrt(lengthSq);
				mz0 *= invLength;
				mz1 *= invLength;
				mz2 *= invLength;
			}
		}
		this.center.set(d[12] + mx0 * ac.x + mx1 * ac.y + mx2 * ac.z, d[13] + my0 * ac.x + my1 * ac.y + my2 * ac.z, d[14] + mz0 * ac.x + mz1 * ac.y + mz2 * ac.z);
		this.halfExtents.set(Math.abs(mx0) * ar.x + Math.abs(mx1) * ar.y + Math.abs(mx2) * ar.z, Math.abs(my0) * ar.x + Math.abs(my1) * ar.y + Math.abs(my2) * ar.z, Math.abs(mz0) * ar.x + Math.abs(mz1) * ar.y + Math.abs(mz2) * ar.z);
	}
	static computeMinMax(vertices, min, max, numVerts = vertices.length / 3) {
		if (numVerts > 0) {
			let minx = vertices[0];
			let miny = vertices[1];
			let minz = vertices[2];
			let maxx = minx;
			let maxy = miny;
			let maxz = minz;
			const n = numVerts * 3;
			for (let i = 3; i < n; i += 3) {
				const x = vertices[i];
				const y = vertices[i + 1];
				const z = vertices[i + 2];
				if (x < minx) minx = x;
				if (y < miny) miny = y;
				if (z < minz) minz = z;
				if (x > maxx) maxx = x;
				if (y > maxy) maxy = y;
				if (z > maxz) maxz = z;
			}
			min.set(minx, miny, minz);
			max.set(maxx, maxy, maxz);
		}
	}
	compute(vertices, numVerts) {
		BoundingBox.computeMinMax(vertices, tmpVecA$1, tmpVecB$1, numVerts);
		this.setMinMax(tmpVecA$1, tmpVecB$1);
	}
	intersectsBoundingSphere(sphere) {
		const sq = this._distanceToBoundingSphereSq(sphere);
		if (sq <= sphere.radius * sphere.radius) {
			return true;
		}
		return false;
	}
	_distanceToBoundingSphereSq(sphere) {
		const boxMin = this.getMin();
		const boxMax = this.getMax();
		let sq = 0;
		const axis = ['x', 'y', 'z'];
		for (let i = 0; i < 3; ++i) {
			let out = 0;
			const pn = sphere.center[axis[i]];
			const bMin = boxMin[axis[i]];
			const bMax = boxMax[axis[i]];
			let val = 0;
			if (pn < bMin) {
				val = bMin - pn;
				out += val * val;
			}
			if (pn > bMax) {
				val = pn - bMax;
				out += val * val;
			}
			sq += out;
		}
		return sq;
	}
	_expand(expandMin, expandMax) {
		tmpVecA$1.add2(this.getMin(), expandMin);
		tmpVecB$1.add2(this.getMax(), expandMax);
		this.setMinMax(tmpVecA$1, tmpVecB$1);
	}
}

const tmpVecA = new Vec3();
const tmpVecB = new Vec3();
class BoundingSphere {
	constructor(center = new Vec3(), radius = 0.5) {
		this.center = void 0;
		this.radius = void 0;
		this.center = center;
		this.radius = radius;
	}
	containsPoint(point) {
		const lenSq = tmpVecA.sub2(point, this.center).lengthSq();
		const r = this.radius;
		return lenSq < r * r;
	}
	intersectsRay(ray, point) {
		const m = tmpVecA.copy(ray.origin).sub(this.center);
		const b = m.dot(tmpVecB.copy(ray.direction).normalize());
		const c = m.dot(m) - this.radius * this.radius;
		if (c > 0 && b > 0) return false;
		const discr = b * b - c;
		if (discr < 0) return false;
		const t = Math.abs(-b - Math.sqrt(discr));
		if (point) point.copy(ray.direction).mulScalar(t).add(ray.origin);
		return true;
	}
	intersectsBoundingSphere(sphere) {
		tmpVecA.sub2(sphere.center, this.center);
		const totalRadius = sphere.radius + this.radius;
		if (tmpVecA.lengthSq() <= totalRadius * totalRadius) {
			return true;
		}
		return false;
	}
}

class Frustum {
	constructor() {
		this.planes = [];
		for (let i = 0; i < 6; i++) this.planes[i] = [];
	}
	setFromMat4(matrix) {
		const vpm = matrix.data;
		let plane;
		const planes = this.planes;
		plane = planes[0];
		plane[0] = vpm[3] - vpm[0];
		plane[1] = vpm[7] - vpm[4];
		plane[2] = vpm[11] - vpm[8];
		plane[3] = vpm[15] - vpm[12];
		let t = Math.sqrt(plane[0] * plane[0] + plane[1] * plane[1] + plane[2] * plane[2]);
		plane[0] /= t;
		plane[1] /= t;
		plane[2] /= t;
		plane[3] /= t;
		plane = planes[1];
		plane[0] = vpm[3] + vpm[0];
		plane[1] = vpm[7] + vpm[4];
		plane[2] = vpm[11] + vpm[8];
		plane[3] = vpm[15] + vpm[12];
		t = Math.sqrt(plane[0] * plane[0] + plane[1] * plane[1] + plane[2] * plane[2]);
		plane[0] /= t;
		plane[1] /= t;
		plane[2] /= t;
		plane[3] /= t;
		plane = planes[2];
		plane[0] = vpm[3] + vpm[1];
		plane[1] = vpm[7] + vpm[5];
		plane[2] = vpm[11] + vpm[9];
		plane[3] = vpm[15] + vpm[13];
		t = Math.sqrt(plane[0] * plane[0] + plane[1] * plane[1] + plane[2] * plane[2]);
		plane[0] /= t;
		plane[1] /= t;
		plane[2] /= t;
		plane[3] /= t;
		plane = planes[3];
		plane[0] = vpm[3] - vpm[1];
		plane[1] = vpm[7] - vpm[5];
		plane[2] = vpm[11] - vpm[9];
		plane[3] = vpm[15] - vpm[13];
		t = Math.sqrt(plane[0] * plane[0] + plane[1] * plane[1] + plane[2] * plane[2]);
		plane[0] /= t;
		plane[1] /= t;
		plane[2] /= t;
		plane[3] /= t;
		plane = planes[4];
		plane[0] = vpm[3] - vpm[2];
		plane[1] = vpm[7] - vpm[6];
		plane[2] = vpm[11] - vpm[10];
		plane[3] = vpm[15] - vpm[14];
		t = Math.sqrt(plane[0] * plane[0] + plane[1] * plane[1] + plane[2] * plane[2]);
		plane[0] /= t;
		plane[1] /= t;
		plane[2] /= t;
		plane[3] /= t;
		plane = planes[5];
		plane[0] = vpm[3] + vpm[2];
		plane[1] = vpm[7] + vpm[6];
		plane[2] = vpm[11] + vpm[10];
		plane[3] = vpm[15] + vpm[14];
		t = Math.sqrt(plane[0] * plane[0] + plane[1] * plane[1] + plane[2] * plane[2]);
		plane[0] /= t;
		plane[1] /= t;
		plane[2] /= t;
		plane[3] /= t;
	}
	containsPoint(point) {
		let p, plane;
		for (p = 0; p < 6; p++) {
			plane = this.planes[p];
			if (plane[0] * point.x + plane[1] * point.y + plane[2] * point.z + plane[3] <= 0) {
				return false;
			}
		}
		return true;
	}
	containsSphere(sphere) {
		let c = 0;
		let d;
		let p;
		const sr = sphere.radius;
		const sc = sphere.center;
		const scx = sc.x;
		const scy = sc.y;
		const scz = sc.z;
		const planes = this.planes;
		let plane;
		for (p = 0; p < 6; p++) {
			plane = planes[p];
			d = plane[0] * scx + plane[1] * scy + plane[2] * scz + plane[3];
			if (d <= -sr) return 0;
			if (d > sr) c++;
		}
		return c === 6 ? 2 : 1;
	}
}

class Ray {
	constructor(origin, direction) {
		this.origin = new Vec3();
		this.direction = Vec3.FORWARD.clone();
		if (origin) {
			this.origin.copy(origin);
		}
		if (direction) {
			this.direction.copy(direction);
		}
	}
	set(origin, direction) {
		this.origin.copy(origin);
		this.direction.copy(direction);
		return this;
	}
	copy(src) {
		return this.set(src.origin, src.direction);
	}
	clone() {
		return new this.constructor(this.origin, this.direction);
	}
}

const tmpRay = new Ray();
const tmpVec3$2 = new Vec3();
const tmpSphere = new BoundingSphere();
const tmpMat4$1 = new Mat4();
class OrientedBox {
	constructor(worldTransform = new Mat4(), halfExtents = new Vec3(0.5, 0.5, 0.5)) {
		this.halfExtents = void 0;
		this._modelTransform = void 0;
		this._worldTransform = void 0;
		this._aabb = void 0;
		this.halfExtents = halfExtents;
		this._modelTransform = worldTransform.clone().invert();
		this._worldTransform = worldTransform.clone();
		this._aabb = new BoundingBox(new Vec3(), this.halfExtents);
	}
	set worldTransform(value) {
		this._worldTransform.copy(value);
		this._modelTransform.copy(value).invert();
	}
	get worldTransform() {
		return this._worldTransform;
	}
	intersectsRay(ray, point) {
		this._modelTransform.transformPoint(ray.origin, tmpRay.origin);
		this._modelTransform.transformVector(ray.direction, tmpRay.direction);
		if (point) {
			const result = this._aabb._intersectsRay(tmpRay, point);
			tmpMat4$1.copy(this._modelTransform).invert().transformPoint(point, point);
			return result;
		}
		return this._aabb._fastIntersectsRay(tmpRay);
	}
	containsPoint(point) {
		this._modelTransform.transformPoint(point, tmpVec3$2);
		return this._aabb.containsPoint(tmpVec3$2);
	}
	intersectsBoundingSphere(sphere) {
		this._modelTransform.transformPoint(sphere.center, tmpSphere.center);
		tmpSphere.radius = sphere.radius;
		if (this._aabb.intersectsBoundingSphere(tmpSphere)) {
			return true;
		}
		return false;
	}
}

class Plane {
	constructor(normal = Vec3.UP, distance = 0) {
		this.normal = new Vec3();
		this.distance = void 0;
		this.normal.copy(normal);
		this.distance = distance;
	}
	setFromPointNormal(point, normal) {
		this.normal.copy(normal);
		this.distance = -this.normal.dot(point);
		return this;
	}
	intersectsLine(start, end, point) {
		const d = this.distance;
		const d0 = this.normal.dot(start) + d;
		const d1 = this.normal.dot(end) + d;
		const t = d0 / (d0 - d1);
		const intersects = t >= 0 && t <= 1;
		if (intersects && point) point.lerp(start, end, t);
		return intersects;
	}
	intersectsRay(ray, point) {
		const denominator = this.normal.dot(ray.direction);
		if (denominator === 0) return false;
		const t = -(this.normal.dot(ray.origin) + this.distance) / denominator;
		if (t >= 0 && point) {
			point.copy(ray.direction).mulScalar(t).add(ray.origin);
		}
		return t >= 0;
	}
	copy(src) {
		this.normal.copy(src.normal);
		this.distance = src.distance;
		return this;
	}
	clone() {
		const cstr = this.constructor;
		return new cstr().copy(this);
	}
}

const DISTANCE_LINEAR = 'linear';
const DISTANCE_INVERSE = 'inverse';
const DISTANCE_EXPONENTIAL = 'exponential';

const ADDRESS_REPEAT = 0;
const ADDRESS_CLAMP_TO_EDGE = 1;
const ADDRESS_MIRRORED_REPEAT = 2;
const BLENDMODE_ZERO = 0;
const BLENDMODE_ONE = 1;
const BLENDMODE_SRC_COLOR = 2;
const BLENDMODE_ONE_MINUS_SRC_COLOR = 3;
const BLENDMODE_DST_COLOR = 4;
const BLENDMODE_ONE_MINUS_DST_COLOR = 5;
const BLENDMODE_SRC_ALPHA = 6;
const BLENDMODE_SRC_ALPHA_SATURATE = 7;
const BLENDMODE_ONE_MINUS_SRC_ALPHA = 8;
const BLENDMODE_DST_ALPHA = 9;
const BLENDMODE_ONE_MINUS_DST_ALPHA = 10;
const BLENDMODE_CONSTANT = 11;
const BLENDMODE_ONE_MINUS_CONSTANT = 12;
const BLENDEQUATION_ADD = 0;
const BLENDEQUATION_SUBTRACT = 1;
const BLENDEQUATION_REVERSE_SUBTRACT = 2;
const BLENDEQUATION_MIN = 3;
const BLENDEQUATION_MAX = 4;
const BUFFER_STATIC = 0;
const BUFFER_DYNAMIC = 1;
const BUFFER_STREAM = 2;
const BUFFER_GPUDYNAMIC = 3;
const CLEARFLAG_COLOR = 1;
const CLEARFLAG_DEPTH = 2;
const CLEARFLAG_STENCIL = 4;
const CUBEFACE_POSX = 0;
const CUBEFACE_NEGX = 1;
const CUBEFACE_POSY = 2;
const CUBEFACE_NEGY = 3;
const CUBEFACE_POSZ = 4;
const CUBEFACE_NEGZ = 5;
const CULLFACE_NONE = 0;
const CULLFACE_BACK = 1;
const CULLFACE_FRONT = 2;
const CULLFACE_FRONTANDBACK = 3;
const FILTER_NEAREST = 0;
const FILTER_LINEAR = 1;
const FILTER_NEAREST_MIPMAP_NEAREST = 2;
const FILTER_NEAREST_MIPMAP_LINEAR = 3;
const FILTER_LINEAR_MIPMAP_NEAREST = 4;
const FILTER_LINEAR_MIPMAP_LINEAR = 5;
const FUNC_NEVER = 0;
const FUNC_LESS = 1;
const FUNC_EQUAL = 2;
const FUNC_LESSEQUAL = 3;
const FUNC_GREATER = 4;
const FUNC_NOTEQUAL = 5;
const FUNC_GREATEREQUAL = 6;
const FUNC_ALWAYS = 7;
const INDEXFORMAT_UINT8 = 0;
const INDEXFORMAT_UINT16 = 1;
const INDEXFORMAT_UINT32 = 2;
const PIXELFORMAT_A8 = 0;
const PIXELFORMAT_L8 = 1;
const PIXELFORMAT_LA8 = 2;
const PIXELFORMAT_RGB565 = 3;
const PIXELFORMAT_RGBA5551 = 4;
const PIXELFORMAT_RGBA4 = 5;
const PIXELFORMAT_RGB8 = 6;
const PIXELFORMAT_RGBA8 = 7;
const PIXELFORMAT_DXT1 = 8;
const PIXELFORMAT_DXT3 = 9;
const PIXELFORMAT_DXT5 = 10;
const PIXELFORMAT_RGB16F = 11;
const PIXELFORMAT_RGBA16F = 12;
const PIXELFORMAT_RGB32F = 13;
const PIXELFORMAT_RGBA32F = 14;
const PIXELFORMAT_R32F = 15;
const PIXELFORMAT_DEPTH = 16;
const PIXELFORMAT_DEPTHSTENCIL = 17;
const PIXELFORMAT_111110F = 18;
const PIXELFORMAT_SRGB = 19;
const PIXELFORMAT_SRGBA = 20;
const PIXELFORMAT_ETC1 = 21;
const PIXELFORMAT_ETC2_RGB = 22;
const PIXELFORMAT_ETC2_RGBA = 23;
const PIXELFORMAT_PVRTC_2BPP_RGB_1 = 24;
const PIXELFORMAT_PVRTC_2BPP_RGBA_1 = 25;
const PIXELFORMAT_PVRTC_4BPP_RGB_1 = 26;
const PIXELFORMAT_PVRTC_4BPP_RGBA_1 = 27;
const PIXELFORMAT_ASTC_4x4 = 28;
const PIXELFORMAT_ATC_RGB = 29;
const PIXELFORMAT_ATC_RGBA = 30;
const PIXELFORMAT_BGRA8 = 31;
const PIXELFORMAT_R8I = 32;
const PIXELFORMAT_R8U = 33;
const PIXELFORMAT_R16I = 34;
const PIXELFORMAT_R16U = 35;
const PIXELFORMAT_R32I = 36;
const PIXELFORMAT_R32U = 37;
const PIXELFORMAT_RG8I = 38;
const PIXELFORMAT_RG8U = 39;
const PIXELFORMAT_RG16I = 40;
const PIXELFORMAT_RG16U = 41;
const PIXELFORMAT_RG32I = 42;
const PIXELFORMAT_RG32U = 43;
const PIXELFORMAT_RGBA8I = 44;
const PIXELFORMAT_RGBA8U = 45;
const PIXELFORMAT_RGBA16I = 46;
const PIXELFORMAT_RGBA16U = 47;
const PIXELFORMAT_RGBA32I = 48;
const PIXELFORMAT_RGBA32U = 49;
const PIXELFORMAT_R16F = 50;
const PIXELFORMAT_RG16F = 51;
const pixelFormatInfo = new Map([[PIXELFORMAT_A8, {
	name: 'A8',
	size: 1
}], [PIXELFORMAT_L8, {
	name: 'L8',
	size: 1
}], [PIXELFORMAT_LA8, {
	name: 'LA8',
	size: 2
}], [PIXELFORMAT_RGB565, {
	name: 'RGB565',
	size: 2
}], [PIXELFORMAT_RGBA5551, {
	name: 'RGBA5551',
	size: 2
}], [PIXELFORMAT_RGBA4, {
	name: 'RGBA4',
	size: 2
}], [PIXELFORMAT_RGB8, {
	name: 'RGB8',
	size: 4
}], [PIXELFORMAT_RGBA8, {
	name: 'RGBA8',
	size: 4
}], [PIXELFORMAT_R16F, {
	name: 'R16F',
	size: 2
}], [PIXELFORMAT_RG16F, {
	name: 'RG16F',
	size: 4
}], [PIXELFORMAT_RGB16F, {
	name: 'RGB16F',
	size: 8
}], [PIXELFORMAT_RGBA16F, {
	name: 'RGBA16F',
	size: 8
}], [PIXELFORMAT_RGB32F, {
	name: 'RGB32F',
	size: 16
}], [PIXELFORMAT_RGBA32F, {
	name: 'RGBA32F',
	size: 16
}], [PIXELFORMAT_R32F, {
	name: 'R32F',
	size: 4
}], [PIXELFORMAT_DEPTH, {
	name: 'DEPTH',
	size: 4
}], [PIXELFORMAT_DEPTHSTENCIL, {
	name: 'DEPTHSTENCIL',
	size: 4
}], [PIXELFORMAT_111110F, {
	name: '111110F',
	size: 4
}], [PIXELFORMAT_SRGB, {
	name: 'SRGB',
	size: 4
}], [PIXELFORMAT_SRGBA, {
	name: 'SRGBA',
	size: 4
}], [PIXELFORMAT_BGRA8, {
	name: 'BGRA8',
	size: 4
}], [PIXELFORMAT_DXT1, {
	name: 'DXT1',
	blockSize: 8
}], [PIXELFORMAT_DXT3, {
	name: 'DXT3',
	blockSize: 16
}], [PIXELFORMAT_DXT5, {
	name: 'DXT5',
	blockSize: 16
}], [PIXELFORMAT_ETC1, {
	name: 'ETC1',
	blockSize: 8
}], [PIXELFORMAT_ETC2_RGB, {
	name: 'ETC2_RGB',
	blockSize: 8
}], [PIXELFORMAT_ETC2_RGBA, {
	name: 'ETC2_RGBA',
	blockSize: 16
}], [PIXELFORMAT_PVRTC_2BPP_RGB_1, {
	name: 'PVRTC_2BPP_RGB_1',
	blockSize: 8
}], [PIXELFORMAT_PVRTC_2BPP_RGBA_1, {
	name: 'PVRTC_2BPP_RGBA_1',
	blockSize: 8
}], [PIXELFORMAT_PVRTC_4BPP_RGB_1, {
	name: 'PVRTC_4BPP_RGB_1',
	blockSize: 8
}], [PIXELFORMAT_PVRTC_4BPP_RGBA_1, {
	name: 'PVRTC_4BPP_RGBA_1',
	blockSize: 8
}], [PIXELFORMAT_ASTC_4x4, {
	name: 'ASTC_4x4',
	blockSize: 16
}], [PIXELFORMAT_ATC_RGB, {
	name: 'ATC_RGB',
	blockSize: 8
}], [PIXELFORMAT_ATC_RGBA, {
	name: 'ATC_RGBA',
	blockSize: 16
}], [PIXELFORMAT_R8I, {
	name: 'R8I',
	size: 1,
	isInt: true
}], [PIXELFORMAT_R8U, {
	name: 'R8U',
	size: 1,
	isInt: true
}], [PIXELFORMAT_R16I, {
	name: 'R16I',
	size: 2,
	isInt: true
}], [PIXELFORMAT_R16U, {
	name: 'R16U',
	size: 2,
	isInt: true
}], [PIXELFORMAT_R32I, {
	name: 'R32I',
	size: 4,
	isInt: true
}], [PIXELFORMAT_R32U, {
	name: 'R32U',
	size: 4,
	isInt: true
}], [PIXELFORMAT_RG8I, {
	name: 'RG8I',
	size: 2,
	isInt: true
}], [PIXELFORMAT_RG8U, {
	name: 'RG8U',
	size: 2,
	isInt: true
}], [PIXELFORMAT_RG16I, {
	name: 'RG16I',
	size: 4,
	isInt: true
}], [PIXELFORMAT_RG16U, {
	name: 'RG16U',
	size: 4,
	isInt: true
}], [PIXELFORMAT_RG32I, {
	name: 'RG32I',
	size: 8,
	isInt: true
}], [PIXELFORMAT_RG32U, {
	name: 'RG32U',
	size: 8,
	isInt: true
}], [PIXELFORMAT_RGBA8I, {
	name: 'RGBA8I',
	size: 4,
	isInt: true
}], [PIXELFORMAT_RGBA8U, {
	name: 'RGBA8U',
	size: 4,
	isInt: true
}], [PIXELFORMAT_RGBA16I, {
	name: 'RGBA16I',
	size: 8,
	isInt: true
}], [PIXELFORMAT_RGBA16U, {
	name: 'RGBA16U',
	size: 8,
	isInt: true
}], [PIXELFORMAT_RGBA32I, {
	name: 'RGBA32I',
	size: 16,
	isInt: true
}], [PIXELFORMAT_RGBA32U, {
	name: 'RGBA32U',
	size: 16,
	isInt: true
}]]);
const isCompressedPixelFormat = format => {
	var _pixelFormatInfo$get;
	return ((_pixelFormatInfo$get = pixelFormatInfo.get(format)) == null ? void 0 : _pixelFormatInfo$get.blockSize) !== undefined;
};
const isIntegerPixelFormat = format => {
	var _pixelFormatInfo$get2;
	return ((_pixelFormatInfo$get2 = pixelFormatInfo.get(format)) == null ? void 0 : _pixelFormatInfo$get2.isInt) === true;
};
const getPixelFormatArrayType = format => {
	switch (format) {
		case PIXELFORMAT_R32F:
		case PIXELFORMAT_RGB32F:
		case PIXELFORMAT_RGBA32F:
			return Float32Array;
		case PIXELFORMAT_R32I:
		case PIXELFORMAT_RG32I:
		case PIXELFORMAT_RGBA32I:
			return Int32Array;
		case PIXELFORMAT_R32U:
		case PIXELFORMAT_RG32U:
		case PIXELFORMAT_RGBA32U:
			return Uint32Array;
		case PIXELFORMAT_R16I:
		case PIXELFORMAT_RG16I:
		case PIXELFORMAT_RGBA16I:
			return Int16Array;
		case PIXELFORMAT_R16U:
		case PIXELFORMAT_RG16U:
		case PIXELFORMAT_RGBA16U:
		case PIXELFORMAT_RGB565:
		case PIXELFORMAT_RGBA5551:
		case PIXELFORMAT_RGBA4:
		case PIXELFORMAT_R16F:
		case PIXELFORMAT_RG16F:
		case PIXELFORMAT_RGB16F:
		case PIXELFORMAT_RGBA16F:
			return Uint16Array;
		case PIXELFORMAT_R8I:
		case PIXELFORMAT_RG8I:
		case PIXELFORMAT_RGBA8I:
			return Int8Array;
		default:
			return Uint8Array;
	}
};
const PRIMITIVE_POINTS = 0;
const PRIMITIVE_LINES = 1;
const PRIMITIVE_LINELOOP = 2;
const PRIMITIVE_LINESTRIP = 3;
const PRIMITIVE_TRIANGLES = 4;
const PRIMITIVE_TRISTRIP = 5;
const PRIMITIVE_TRIFAN = 6;
const SEMANTIC_POSITION = "POSITION";
const SEMANTIC_NORMAL = "NORMAL";
const SEMANTIC_TANGENT = "TANGENT";
const SEMANTIC_BLENDWEIGHT = "BLENDWEIGHT";
const SEMANTIC_BLENDINDICES = "BLENDINDICES";
const SEMANTIC_COLOR = "COLOR";
const SEMANTIC_TEXCOORD = "TEXCOORD";
const SEMANTIC_TEXCOORD0 = "TEXCOORD0";
const SEMANTIC_TEXCOORD1 = "TEXCOORD1";
const SEMANTIC_TEXCOORD2 = "TEXCOORD2";
const SEMANTIC_TEXCOORD3 = "TEXCOORD3";
const SEMANTIC_TEXCOORD4 = "TEXCOORD4";
const SEMANTIC_TEXCOORD5 = "TEXCOORD5";
const SEMANTIC_TEXCOORD6 = "TEXCOORD6";
const SEMANTIC_TEXCOORD7 = "TEXCOORD7";
const SEMANTIC_ATTR = "ATTR";
const SEMANTIC_ATTR0 = "ATTR0";
const SEMANTIC_ATTR1 = "ATTR1";
const SEMANTIC_ATTR2 = "ATTR2";
const SEMANTIC_ATTR3 = "ATTR3";
const SEMANTIC_ATTR4 = "ATTR4";
const SEMANTIC_ATTR5 = "ATTR5";
const SEMANTIC_ATTR6 = "ATTR6";
const SEMANTIC_ATTR7 = "ATTR7";
const SEMANTIC_ATTR8 = "ATTR8";
const SEMANTIC_ATTR9 = "ATTR9";
const SEMANTIC_ATTR10 = "ATTR10";
const SEMANTIC_ATTR11 = "ATTR11";
const SEMANTIC_ATTR12 = "ATTR12";
const SEMANTIC_ATTR13 = "ATTR13";
const SEMANTIC_ATTR14 = "ATTR14";
const SEMANTIC_ATTR15 = "ATTR15";
const SHADERTAG_MATERIAL = 1;
const STENCILOP_KEEP = 0;
const STENCILOP_ZERO = 1;
const STENCILOP_REPLACE = 2;
const STENCILOP_INCREMENT = 3;
const STENCILOP_INCREMENTWRAP = 4;
const STENCILOP_DECREMENT = 5;
const STENCILOP_DECREMENTWRAP = 6;
const STENCILOP_INVERT = 7;
const TEXTURELOCK_NONE = 0;
const TEXTURELOCK_READ = 1;
const TEXTURELOCK_WRITE = 2;
const TEXTURETYPE_DEFAULT = 'default';
const TEXTURETYPE_RGBM = 'rgbm';
const TEXTURETYPE_RGBE = 'rgbe';
const TEXTURETYPE_RGBP = 'rgbp';
const TEXTURETYPE_SWIZZLEGGGR = 'swizzleGGGR';
const TEXHINT_NONE = 0;
const TEXHINT_SHADOWMAP = 1;
const TEXHINT_ASSET = 2;
const TEXHINT_LIGHTMAP = 3;
const TEXTUREDIMENSION_1D = '1d';
const TEXTUREDIMENSION_2D = '2d';
const TEXTUREDIMENSION_2D_ARRAY = '2d-array';
const TEXTUREDIMENSION_CUBE = 'cube';
const TEXTUREDIMENSION_CUBE_ARRAY = 'cube-array';
const TEXTUREDIMENSION_3D = '3d';
const SAMPLETYPE_FLOAT = 0;
const SAMPLETYPE_UNFILTERABLE_FLOAT = 1;
const SAMPLETYPE_DEPTH = 2;
const SAMPLETYPE_INT = 3;
const SAMPLETYPE_UINT = 4;
const TEXTUREPROJECTION_NONE = "none";
const TEXTUREPROJECTION_CUBE = "cube";
const TEXTUREPROJECTION_EQUIRECT = "equirect";
const TEXTUREPROJECTION_OCTAHEDRAL = "octahedral";
const SHADERLANGUAGE_GLSL = 'glsl';
const SHADERLANGUAGE_WGSL = 'wgsl';
const TYPE_INT8 = 0;
const TYPE_UINT8 = 1;
const TYPE_INT16 = 2;
const TYPE_UINT16 = 3;
const TYPE_INT32 = 4;
const TYPE_UINT32 = 5;
const TYPE_FLOAT32 = 6;
const TYPE_FLOAT16 = 7;
const UNIFORMTYPE_BOOL = 0;
const UNIFORMTYPE_INT = 1;
const UNIFORMTYPE_FLOAT = 2;
const UNIFORMTYPE_VEC2 = 3;
const UNIFORMTYPE_VEC3 = 4;
const UNIFORMTYPE_VEC4 = 5;
const UNIFORMTYPE_IVEC2 = 6;
const UNIFORMTYPE_IVEC3 = 7;
const UNIFORMTYPE_IVEC4 = 8;
const UNIFORMTYPE_BVEC2 = 9;
const UNIFORMTYPE_BVEC3 = 10;
const UNIFORMTYPE_BVEC4 = 11;
const UNIFORMTYPE_MAT2 = 12;
const UNIFORMTYPE_MAT3 = 13;
const UNIFORMTYPE_MAT4 = 14;
const UNIFORMTYPE_TEXTURE2D = 15;
const UNIFORMTYPE_TEXTURECUBE = 16;
const UNIFORMTYPE_FLOATARRAY = 17;
const UNIFORMTYPE_TEXTURE2D_SHADOW = 18;
const UNIFORMTYPE_TEXTURECUBE_SHADOW = 19;
const UNIFORMTYPE_TEXTURE3D = 20;
const UNIFORMTYPE_VEC2ARRAY = 21;
const UNIFORMTYPE_VEC3ARRAY = 22;
const UNIFORMTYPE_VEC4ARRAY = 23;
const UNIFORMTYPE_MAT4ARRAY = 24;
const UNIFORMTYPE_TEXTURE2D_ARRAY = 25;
const UNIFORMTYPE_UINT = 26;
const UNIFORMTYPE_UVEC2 = 27;
const UNIFORMTYPE_UVEC3 = 28;
const UNIFORMTYPE_UVEC4 = 29;
const UNIFORMTYPE_INTARRAY = 30;
const UNIFORMTYPE_UINTARRAY = 31;
const UNIFORMTYPE_BOOLARRAY = 32;
const UNIFORMTYPE_IVEC2ARRAY = 33;
const UNIFORMTYPE_UVEC2ARRAY = 34;
const UNIFORMTYPE_BVEC2ARRAY = 35;
const UNIFORMTYPE_IVEC3ARRAY = 36;
const UNIFORMTYPE_UVEC3ARRAY = 37;
const UNIFORMTYPE_BVEC3ARRAY = 38;
const UNIFORMTYPE_IVEC4ARRAY = 39;
const UNIFORMTYPE_UVEC4ARRAY = 40;
const UNIFORMTYPE_BVEC4ARRAY = 41;
const UNIFORMTYPE_ITEXTURE2D = 42;
const UNIFORMTYPE_UTEXTURE2D = 43;
const UNIFORMTYPE_ITEXTURECUBE = 44;
const UNIFORMTYPE_UTEXTURECUBE = 45;
const UNIFORMTYPE_ITEXTURE3D = 46;
const UNIFORMTYPE_UTEXTURE3D = 47;
const UNIFORMTYPE_ITEXTURE2D_ARRAY = 48;
const UNIFORMTYPE_UTEXTURE2D_ARRAY = 49;
const uniformTypeToName = ['bool', 'int', 'float', 'vec2', 'vec3', 'vec4', 'ivec2', 'ivec3', 'ivec4', 'bvec2', 'bvec3', 'bvec4', 'mat2', 'mat3', 'mat4', 'sampler2D', 'samplerCube', '', 'sampler2DShadow', 'samplerCubeShadow', 'sampler3D', '', '', '', '', 'sampler2DArray', 'uint', 'uvec2', 'uvec3', 'uvec4', '', '', '', '', '', '', '', '', '', '', '', '', 'isampler2D', 'usampler2D', 'isamplerCube', 'usamplerCube', 'isampler3D', 'usampler3D', 'isampler2DArray', 'usampler2DArray'];
const uniformTypeToStorage = new Uint8Array([TYPE_INT32, TYPE_INT32, TYPE_FLOAT32, TYPE_FLOAT32, TYPE_FLOAT32, TYPE_FLOAT32, TYPE_INT32, TYPE_INT32, TYPE_INT32, TYPE_INT32, TYPE_INT32, TYPE_INT32, TYPE_FLOAT32, TYPE_FLOAT32, TYPE_FLOAT32, TYPE_INT32, TYPE_INT32, TYPE_FLOAT32, TYPE_INT32, TYPE_INT32, TYPE_INT32, TYPE_FLOAT32, TYPE_FLOAT32, TYPE_FLOAT32, TYPE_FLOAT32, TYPE_INT32, TYPE_UINT32, TYPE_UINT32, TYPE_UINT32, TYPE_UINT32, TYPE_INT32, TYPE_UINT32, TYPE_INT32, TYPE_INT32, TYPE_UINT32, TYPE_INT32, TYPE_INT32, TYPE_UINT32, TYPE_INT32, TYPE_INT32, TYPE_UINT32, TYPE_INT32, TYPE_INT32, TYPE_UINT32, TYPE_INT32, TYPE_UINT32, TYPE_INT32, TYPE_UINT32, TYPE_INT32, TYPE_UINT32]);
const DEVICETYPE_WEBGL1 = 'webgl1';
const DEVICETYPE_WEBGL2 = 'webgl2';
const DEVICETYPE_WEBGPU = 'webgpu';
const DEVICETYPE_NULL = 'null';
const SHADERSTAGE_VERTEX = 1;
const SHADERSTAGE_FRAGMENT = 2;
const SHADERSTAGE_COMPUTE = 4;
const BINDGROUP_MESH = 0;
const BINDGROUP_VIEW = 1;
const bindGroupNames = ['mesh', 'view'];
const UNIFORM_BUFFER_DEFAULT_SLOT_NAME = 'default';
const typedArrayTypes = [Int8Array, Uint8Array, Int16Array, Uint16Array, Int32Array, Uint32Array, Float32Array, Uint16Array];
const typedArrayTypesByteSize = [1, 1, 2, 2, 4, 4, 4, 2];
const vertexTypesNames = ['INT8', 'UINT8', 'INT16', 'UINT16', 'INT32', 'UINT32', 'FLOAT32', 'FLOAT16'];
const typedArrayToType = {
	"Int8Array": TYPE_INT8,
	"Uint8Array": TYPE_UINT8,
	"Int16Array": TYPE_INT16,
	"Uint16Array": TYPE_UINT16,
	"Int32Array": TYPE_INT32,
	"Uint32Array": TYPE_UINT32,
	"Float32Array": TYPE_FLOAT32
};
const typedArrayIndexFormats = [Uint8Array, Uint16Array, Uint32Array];
const typedArrayIndexFormatsByteSize = [1, 2, 4];
const semanticToLocation = {};
semanticToLocation[SEMANTIC_POSITION] = 0;
semanticToLocation[SEMANTIC_NORMAL] = 1;
semanticToLocation[SEMANTIC_BLENDWEIGHT] = 2;
semanticToLocation[SEMANTIC_BLENDINDICES] = 3;
semanticToLocation[SEMANTIC_COLOR] = 4;
semanticToLocation[SEMANTIC_TEXCOORD0] = 5;
semanticToLocation[SEMANTIC_TEXCOORD1] = 6;
semanticToLocation[SEMANTIC_TEXCOORD2] = 7;
semanticToLocation[SEMANTIC_TEXCOORD3] = 8;
semanticToLocation[SEMANTIC_TEXCOORD4] = 9;
semanticToLocation[SEMANTIC_TEXCOORD5] = 10;
semanticToLocation[SEMANTIC_TEXCOORD6] = 11;
semanticToLocation[SEMANTIC_TEXCOORD7] = 12;
semanticToLocation[SEMANTIC_TANGENT] = 13;
semanticToLocation[SEMANTIC_ATTR0] = 0;
semanticToLocation[SEMANTIC_ATTR1] = 1;
semanticToLocation[SEMANTIC_ATTR2] = 2;
semanticToLocation[SEMANTIC_ATTR3] = 3;
semanticToLocation[SEMANTIC_ATTR4] = 4;
semanticToLocation[SEMANTIC_ATTR5] = 5;
semanticToLocation[SEMANTIC_ATTR6] = 6;
semanticToLocation[SEMANTIC_ATTR7] = 7;
semanticToLocation[SEMANTIC_ATTR8] = 8;
semanticToLocation[SEMANTIC_ATTR9] = 9;
semanticToLocation[SEMANTIC_ATTR10] = 10;
semanticToLocation[SEMANTIC_ATTR11] = 11;
semanticToLocation[SEMANTIC_ATTR12] = 12;
semanticToLocation[SEMANTIC_ATTR13] = 13;
semanticToLocation[SEMANTIC_ATTR14] = 14;
semanticToLocation[SEMANTIC_ATTR15] = 15;
const CHUNKAPI_1_51 = '1.51';
const CHUNKAPI_1_55 = '1.55';
const CHUNKAPI_1_56 = '1.56';
const CHUNKAPI_1_57 = '1.57';
const CHUNKAPI_1_58 = '1.58';
const CHUNKAPI_1_60 = '1.60';
const CHUNKAPI_1_62 = '1.62';
const CHUNKAPI_1_65 = '1.65';

function _extends() {
  _extends = Object.assign ? Object.assign.bind() : function (target) {
    for (var i = 1; i < arguments.length; i++) {
      var source = arguments[i];
      for (var key in source) {
        if (Object.prototype.hasOwnProperty.call(source, key)) {
          target[key] = source[key];
        }
      }
    }
    return target;
  };
  return _extends.apply(this, arguments);
}

const BitPacking = {
	set(storage, value, shift, mask = 1) {
		const data = storage & ~(mask << shift);
		return data | value << shift;
	},
	get(storage, shift, mask = 1) {
		return storage >> shift & mask;
	},
	all(storage, shift, mask = 1) {
		const shifted = mask << shift;
		return (storage & shifted) === shifted;
	},
	any(storage, shift, mask = 1) {
		return (storage & mask << shift) !== 0;
	}
};

var _class$3;
const opMask = 0b111;
const factorMask = 0b1111;
const colorOpShift = 0;
const colorSrcFactorShift = 3;
const colorDstFactorShift = 7;
const alphaOpShift = 11;
const alphaSrcFactorShift = 14;
const alphaDstFactorShift = 18;
const redWriteShift = 22;
const greenWriteShift = 23;
const blueWriteShift = 24;
const alphaWriteShift = 25;
const blendShift = 26;
const allWriteMasks = 0b1111;
const allWriteShift = redWriteShift;
class BlendState {
	constructor(blend = false, colorOp = BLENDEQUATION_ADD, colorSrcFactor = BLENDMODE_ONE, colorDstFactor = BLENDMODE_ZERO, alphaOp, alphaSrcFactor, alphaDstFactor, redWrite = true, greenWrite = true, blueWrite = true, alphaWrite = true) {
		this.target0 = 0;
		this.setColorBlend(colorOp, colorSrcFactor, colorDstFactor);
		this.setAlphaBlend(alphaOp != null ? alphaOp : colorOp, alphaSrcFactor != null ? alphaSrcFactor : colorSrcFactor, alphaDstFactor != null ? alphaDstFactor : colorDstFactor);
		this.setColorWrite(redWrite, greenWrite, blueWrite, alphaWrite);
		this.blend = blend;
	}
	set blend(value) {
		this.target0 = BitPacking.set(this.target0, value ? 1 : 0, blendShift);
	}
	get blend() {
		return BitPacking.all(this.target0, blendShift);
	}
	setColorBlend(op, srcFactor, dstFactor) {
		this.target0 = BitPacking.set(this.target0, op, colorOpShift, opMask);
		this.target0 = BitPacking.set(this.target0, srcFactor, colorSrcFactorShift, factorMask);
		this.target0 = BitPacking.set(this.target0, dstFactor, colorDstFactorShift, factorMask);
	}
	setAlphaBlend(op, srcFactor, dstFactor) {
		this.target0 = BitPacking.set(this.target0, op, alphaOpShift, opMask);
		this.target0 = BitPacking.set(this.target0, srcFactor, alphaSrcFactorShift, factorMask);
		this.target0 = BitPacking.set(this.target0, dstFactor, alphaDstFactorShift, factorMask);
	}
	setColorWrite(redWrite, greenWrite, blueWrite, alphaWrite) {
		this.redWrite = redWrite;
		this.greenWrite = greenWrite;
		this.blueWrite = blueWrite;
		this.alphaWrite = alphaWrite;
	}
	get colorOp() {
		return BitPacking.get(this.target0, colorOpShift, opMask);
	}
	get colorSrcFactor() {
		return BitPacking.get(this.target0, colorSrcFactorShift, factorMask);
	}
	get colorDstFactor() {
		return BitPacking.get(this.target0, colorDstFactorShift, factorMask);
	}
	get alphaOp() {
		return BitPacking.get(this.target0, alphaOpShift, opMask);
	}
	get alphaSrcFactor() {
		return BitPacking.get(this.target0, alphaSrcFactorShift, factorMask);
	}
	get alphaDstFactor() {
		return BitPacking.get(this.target0, alphaDstFactorShift, factorMask);
	}
	set redWrite(value) {
		this.target0 = BitPacking.set(this.target0, value ? 1 : 0, redWriteShift);
	}
	get redWrite() {
		return BitPacking.all(this.target0, redWriteShift);
	}
	set greenWrite(value) {
		this.target0 = BitPacking.set(this.target0, value ? 1 : 0, greenWriteShift);
	}
	get greenWrite() {
		return BitPacking.all(this.target0, greenWriteShift);
	}
	set blueWrite(value) {
		this.target0 = BitPacking.set(this.target0, value ? 1 : 0, blueWriteShift);
	}
	get blueWrite() {
		return BitPacking.all(this.target0, blueWriteShift);
	}
	set alphaWrite(value) {
		this.target0 = BitPacking.set(this.target0, value ? 1 : 0, alphaWriteShift);
	}
	get alphaWrite() {
		return BitPacking.all(this.target0, alphaWriteShift);
	}
	get allWrite() {
		return BitPacking.get(this.target0, allWriteShift, allWriteMasks);
	}
	copy(rhs) {
		this.target0 = rhs.target0;
		return this;
	}
	clone() {
		const clone = new this.constructor();
		return clone.copy(this);
	}
	get key() {
		return this.target0;
	}
	equals(rhs) {
		return this.target0 === rhs.target0;
	}
}
_class$3 = BlendState;
BlendState.NOBLEND = Object.freeze(new _class$3());
BlendState.NOWRITE = Object.freeze(new _class$3(undefined, undefined, undefined, undefined, undefined, undefined, undefined, false, false, false, false));
BlendState.ALPHABLEND = Object.freeze(new _class$3(true, BLENDEQUATION_ADD, BLENDMODE_SRC_ALPHA, BLENDMODE_ONE_MINUS_SRC_ALPHA));
BlendState.ADDBLEND = Object.freeze(new _class$3(true, BLENDEQUATION_ADD, BLENDMODE_ONE, BLENDMODE_ONE));

class StringIds {
	constructor() {
		this.map = new Map();
		this.id = 0;
	}
	get(name) {
		let value = this.map.get(name);
		if (value === undefined) {
			value = this.id++;
			this.map.set(name, value);
		}
		return value;
	}
}

var _class$2;
const stringIds$4 = new StringIds();
const funcMask = 0b111;
const funcShift = 0;
const writeShift = 3;
class DepthState {
	constructor(func = FUNC_LESSEQUAL, write = true) {
		this.data = 0;
		this._depthBias = 0;
		this._depthBiasSlope = 0;
		this.key = 0;
		this.func = func;
		this.write = write;
	}
	set test(value) {
		this.func = value ? FUNC_LESSEQUAL : FUNC_ALWAYS;
		this.updateKey();
	}
	get test() {
		return this.func !== FUNC_ALWAYS;
	}
	set write(value) {
		this.data = BitPacking.set(this.data, value ? 1 : 0, writeShift);
		this.updateKey();
	}
	get write() {
		return BitPacking.all(this.data, writeShift);
	}
	set func(value) {
		this.data = BitPacking.set(this.data, value, funcShift, funcMask);
		this.updateKey();
	}
	get func() {
		return BitPacking.get(this.data, funcShift, funcMask);
	}
	set depthBias(value) {
		this._depthBias = value;
		this.updateKey();
	}
	get depthBias() {
		return this._depthBias;
	}
	set depthBiasSlope(value) {
		this._depthBiasSlope = value;
		this.updateKey();
	}
	get depthBiasSlope() {
		return this._depthBiasSlope;
	}
	copy(rhs) {
		this.data = rhs.data;
		this._depthBias = rhs._depthBias;
		this._depthBiasSlope = rhs._depthBiasSlope;
		this.key = rhs.key;
		return this;
	}
	clone() {
		const clone = new this.constructor();
		return clone.copy(this);
	}
	updateKey() {
		const {
			data,
			_depthBias,
			_depthBiasSlope
		} = this;
		const key = `${data}-${_depthBias}-${_depthBiasSlope}`;
		this.key = stringIds$4.get(key);
	}
	equals(rhs) {
		return this.key === rhs.key;
	}
}
_class$2 = DepthState;
DepthState.DEFAULT = Object.freeze(new _class$2());
DepthState.NODEPTH = Object.freeze(new _class$2(FUNC_ALWAYS, false));
DepthState.WRITEDEPTH = Object.freeze(new _class$2(FUNC_ALWAYS, true));

class Version {
	constructor() {
		this.globalId = 0;
		this.revision = 0;
	}
	equals(other) {
		return this.globalId === other.globalId && this.revision === other.revision;
	}
	copy(other) {
		this.globalId = other.globalId;
		this.revision = other.revision;
	}
	reset() {
		this.globalId = 0;
		this.revision = 0;
	}
}

let idCounter = 0;
class VersionedObject {
	constructor() {
		idCounter++;
		this.version = new Version();
		this.version.globalId = idCounter;
	}
	increment() {
		this.version.revision++;
	}
}

class ScopeId {
	constructor(name) {
		this.name = name;
		this.value = null;
		this.versionObject = new VersionedObject();
	}
	toJSON(key) {
		return undefined;
	}
	setValue(value) {
		this.value = value;
		this.versionObject.increment();
	}
	getValue() {
		return this.value;
	}
}

class ScopeSpace {
	constructor(name) {
		this.name = name;
		this.variables = new Map();
	}
	resolve(name) {
		if (!this.variables.has(name)) {
			this.variables.set(name, new ScopeId(name));
		}
		return this.variables.get(name);
	}
	removeValue(value) {
		for (const uniformName in this.variables) {
			const uniform = this.variables[uniformName];
			if (uniform.value === value) {
				uniform.value = null;
			}
		}
	}
}

let id$a = 0;
class VertexBuffer {
	constructor(graphicsDevice, format, numVertices, usage = BUFFER_STATIC, initialData) {
		this.device = graphicsDevice;
		this.format = format;
		this.numVertices = numVertices;
		this.usage = usage;
		this.id = id$a++;
		this.impl = graphicsDevice.createVertexBufferImpl(this, format);
		this.numBytes = format.verticesByteSize ? format.verticesByteSize : format.size * numVertices;
		this.adjustVramSizeTracking(graphicsDevice._vram, this.numBytes);
		if (initialData) {
			this.setData(initialData);
		} else {
			this.storage = new ArrayBuffer(this.numBytes);
		}
		this.device.buffers.push(this);
	}
	destroy() {
		const device = this.device;
		const idx = device.buffers.indexOf(this);
		if (idx !== -1) {
			device.buffers.splice(idx, 1);
		}
		if (this.impl.initialized) {
			this.impl.destroy(device);
			this.adjustVramSizeTracking(device._vram, -this.storage.byteLength);
		}
	}
	adjustVramSizeTracking(vram, size) {
		vram.vb += size;
	}
	loseContext() {
		this.impl.loseContext();
	}
	getFormat() {
		return this.format;
	}
	getUsage() {
		return this.usage;
	}
	getNumVertices() {
		return this.numVertices;
	}
	lock() {
		return this.storage;
	}
	unlock() {
		this.impl.unlock(this);
	}
	setData(data) {
		if (data.byteLength !== this.numBytes) {
			return false;
		}
		this.storage = data;
		this.unlock();
		return true;
	}
}

function hashCode(str) {
	let hash = 0;
	for (let i = 0, len = str.length; i < len; i++) {
		hash = (hash << 5) - hash + str.charCodeAt(i);
		hash |= 0;
	}
	return hash;
}
function hash32Fnv1a(array) {
	const prime = 16777619;
	let hash = 2166136261;
	for (let i = 0; i < array.length; i++) {
		hash ^= array[i];
		hash *= prime;
	}
	return hash >>> 0;
}

class DeviceCache {
	constructor() {
		this._cache = new Map();
	}
	get(device, onCreate) {
		if (!this._cache.has(device)) {
			this._cache.set(device, onCreate());
			device.on('destroy', () => {
				this.remove(device);
			});
			device.on('devicelost', () => {
				var _this$_cache$get;
				(_this$_cache$get = this._cache.get(device)) == null || _this$_cache$get.loseContext == null || _this$_cache$get.loseContext(device);
			});
		}
		return this._cache.get(device);
	}
	remove(device) {
		var _this$_cache$get2;
		(_this$_cache$get2 = this._cache.get(device)) == null || _this$_cache$get2.destroy == null || _this$_cache$get2.destroy(device);
		this._cache.delete(device);
	}
}

const stringIds$3 = new StringIds();
const webgpuValidElementSizes = [2, 4, 8, 12, 16];
const deviceCache$3 = new DeviceCache();
class VertexFormat {
	constructor(graphicsDevice, description, vertexCount) {
		this.device = graphicsDevice;
		this._elements = [];
		this.hasUv0 = false;
		this.hasUv1 = false;
		this.hasColor = false;
		this.hasTangents = false;
		this.verticesByteSize = 0;
		this.vertexCount = vertexCount;
		this.interleaved = vertexCount === undefined;
		this.instancing = false;
		this.size = description.reduce((total, desc) => {
			return total + Math.ceil(desc.components * typedArrayTypesByteSize[desc.type] / 4) * 4;
		}, 0);
		let offset = 0,
			elementSize;
		for (let i = 0, len = description.length; i < len; i++) {
			var _elementDesc$asInt, _elementDesc$normaliz;
			const elementDesc = description[i];
			elementSize = elementDesc.components * typedArrayTypesByteSize[elementDesc.type];
			if (vertexCount) {
				offset = math.roundUp(offset, elementSize);
			}
			const asInt = (_elementDesc$asInt = elementDesc.asInt) != null ? _elementDesc$asInt : false;
			const normalize = asInt ? false : (_elementDesc$normaliz = elementDesc.normalize) != null ? _elementDesc$normaliz : false;
			const element = {
				name: elementDesc.semantic,
				offset: vertexCount ? offset : elementDesc.hasOwnProperty('offset') ? elementDesc.offset : offset,
				stride: vertexCount ? elementSize : elementDesc.hasOwnProperty('stride') ? elementDesc.stride : this.size,
				dataType: elementDesc.type,
				numComponents: elementDesc.components,
				normalize: normalize,
				size: elementSize,
				asInt: asInt
			};
			this._elements.push(element);
			if (vertexCount) {
				offset += elementSize * vertexCount;
			} else {
				offset += Math.ceil(elementSize / 4) * 4;
			}
			if (elementDesc.semantic === SEMANTIC_TEXCOORD0) {
				this.hasUv0 = true;
			} else if (elementDesc.semantic === SEMANTIC_TEXCOORD1) {
				this.hasUv1 = true;
			} else if (elementDesc.semantic === SEMANTIC_COLOR) {
				this.hasColor = true;
			} else if (elementDesc.semantic === SEMANTIC_TANGENT) {
				this.hasTangents = true;
			}
		}
		if (vertexCount) {
			this.verticesByteSize = offset;
		}
		this._evaluateHash();
	}
	get elements() {
		return this._elements;
	}
	static getDefaultInstancingFormat(graphicsDevice) {
		return deviceCache$3.get(graphicsDevice, () => {
			return new VertexFormat(graphicsDevice, [{
				semantic: SEMANTIC_ATTR12,
				components: 4,
				type: TYPE_FLOAT32
			}, {
				semantic: SEMANTIC_ATTR13,
				components: 4,
				type: TYPE_FLOAT32
			}, {
				semantic: SEMANTIC_ATTR14,
				components: 4,
				type: TYPE_FLOAT32
			}, {
				semantic: SEMANTIC_ATTR15,
				components: 4,
				type: TYPE_FLOAT32
			}]);
		});
	}
	static isElementValid(graphicsDevice, elementDesc) {
		const elementSize = elementDesc.components * typedArrayTypesByteSize[elementDesc.type];
		if (graphicsDevice.isWebGPU && !webgpuValidElementSizes.includes(elementSize)) return false;
		return true;
	}
	update() {
		this._evaluateHash();
	}
	_evaluateHash() {
		const stringElementsBatch = [];
		const stringElementsRender = [];
		const len = this._elements.length;
		for (let i = 0; i < len; i++) {
			const {
				name,
				dataType,
				numComponents,
				normalize,
				offset,
				stride,
				size,
				asInt
			} = this._elements[i];
			const stringElementBatch = name + dataType + numComponents + normalize + asInt;
			stringElementsBatch.push(stringElementBatch);
			const stringElementRender = stringElementBatch + offset + stride + size;
			stringElementsRender.push(stringElementRender);
		}
		stringElementsBatch.sort();
		const batchingString = stringElementsBatch.join();
		this.batchingHash = hashCode(batchingString);
		this.shaderProcessingHashString = batchingString;
		this.renderingHashString = stringElementsRender.join('_');
		this.renderingHash = stringIds$3.get(this.renderingHashString);
	}
}

var _class$1;
const stringIds$2 = new StringIds();
class StencilParameters {
	set func(value) {
		this._func = value;
		this._dirty = true;
	}
	get func() {
		return this._func;
	}
	set ref(value) {
		this._ref = value;
		this._dirty = true;
	}
	get ref() {
		return this._ref;
	}
	set fail(value) {
		this._fail = value;
		this._dirty = true;
	}
	get fail() {
		return this._fail;
	}
	set zfail(value) {
		this._zfail = value;
		this._dirty = true;
	}
	get zfail() {
		return this._zfail;
	}
	set zpass(value) {
		this._zpass = value;
		this._dirty = true;
	}
	get zpass() {
		return this._zpass;
	}
	set readMask(value) {
		this._readMask = value;
		this._dirty = true;
	}
	get readMask() {
		return this._readMask;
	}
	set writeMask(value) {
		this._writeMask = value;
		this._dirty = true;
	}
	get writeMask() {
		return this._writeMask;
	}
	constructor(options = {}) {
		var _options$func, _options$ref, _options$readMask, _options$writeMask, _options$fail, _options$zfail, _options$zpass;
		this._func = void 0;
		this._ref = void 0;
		this._fail = void 0;
		this._zfail = void 0;
		this._zpass = void 0;
		this._readMask = void 0;
		this._writeMask = void 0;
		this._dirty = true;
		this._key = void 0;
		this._func = (_options$func = options.func) != null ? _options$func : FUNC_ALWAYS;
		this._ref = (_options$ref = options.ref) != null ? _options$ref : 0;
		this._readMask = (_options$readMask = options.readMask) != null ? _options$readMask : 0xFF;
		this._writeMask = (_options$writeMask = options.writeMask) != null ? _options$writeMask : 0xFF;
		this._fail = (_options$fail = options.fail) != null ? _options$fail : STENCILOP_KEEP;
		this._zfail = (_options$zfail = options.zfail) != null ? _options$zfail : STENCILOP_KEEP;
		this._zpass = (_options$zpass = options.zpass) != null ? _options$zpass : STENCILOP_KEEP;
		this._evalKey();
	}
	_evalKey() {
		const {
			_func,
			_ref,
			_fail,
			_zfail,
			_zpass,
			_readMask,
			_writeMask
		} = this;
		const key = `${_func},${_ref},${_fail},${_zfail},${_zpass},${_readMask},${_writeMask}`;
		this._key = stringIds$2.get(key);
		this._dirty = false;
	}
	get key() {
		if (this._dirty) {
			this._evalKey();
		}
		return this._key;
	}
	copy(rhs) {
		this._func = rhs._func;
		this._ref = rhs._ref;
		this._readMask = rhs._readMask;
		this._writeMask = rhs._writeMask;
		this._fail = rhs._fail;
		this._zfail = rhs._zfail;
		this._zpass = rhs._zpass;
		this._dirty = rhs._dirty;
		this._key = rhs._key;
		return this;
	}
	clone() {
		const clone = new this.constructor();
		return clone.copy(this);
	}
}
_class$1 = StencilParameters;
StencilParameters.DEFAULT = Object.freeze(new _class$1());

class GraphicsDevice extends EventHandler {
	constructor(canvas, options) {
		var _this$initOptions, _this$initOptions$dep, _this$initOptions2, _this$initOptions2$st, _this$initOptions3, _this$initOptions3$an, _this$initOptions4, _this$initOptions4$po;
		super();
		this.canvas = void 0;
		this.backBuffer = null;
		this.backBufferSize = new Vec2();
		this.backBufferFormat = void 0;
		this.backBufferAntialias = false;
		this.isWebGPU = false;
		this.isWebGL1 = false;
		this.isWebGL2 = false;
		this.scope = void 0;
		this.boneLimit = void 0;
		this.maxAnisotropy = void 0;
		this.maxCubeMapSize = void 0;
		this.maxTextureSize = void 0;
		this.maxVolumeSize = void 0;
		this.maxColorAttachments = 1;
		this.precision = void 0;
		this.samples = void 0;
		this.supportsStencil = void 0;
		this.supportsMrt = false;
		this.supportsVolumeTextures = false;
		this.supportsCompute = false;
		this.renderTarget = null;
		this.shaders = [];
		this.textures = [];
		this.targets = new Set();
		this.renderVersion = 0;
		this.renderPassIndex = void 0;
		this.insideRenderPass = false;
		this.supportsInstancing = void 0;
		this.supportsUniformBuffers = false;
		this.textureFloatRenderable = void 0;
		this.textureHalfFloatRenderable = void 0;
		this.textureFloatFilterable = false;
		this.textureHalfFloatFilterable = false;
		this.quadVertexBuffer = void 0;
		this.blendState = new BlendState();
		this.depthState = new DepthState();
		this.stencilEnabled = false;
		this.stencilFront = new StencilParameters();
		this.stencilBack = new StencilParameters();
		this.dynamicBuffers = void 0;
		this.gpuProfiler = void 0;
		this.defaultClearOptions = {
			color: [0, 0, 0, 1],
			depth: 1,
			stencil: 0,
			flags: CLEARFLAG_COLOR | CLEARFLAG_DEPTH
		};
		this.canvas = canvas;
		this.initOptions = _extends({}, options);
		(_this$initOptions$dep = (_this$initOptions = this.initOptions).depth) != null ? _this$initOptions$dep : _this$initOptions.depth = true;
		(_this$initOptions2$st = (_this$initOptions2 = this.initOptions).stencil) != null ? _this$initOptions2$st : _this$initOptions2.stencil = true;
		(_this$initOptions3$an = (_this$initOptions3 = this.initOptions).antialias) != null ? _this$initOptions3$an : _this$initOptions3.antialias = true;
		(_this$initOptions4$po = (_this$initOptions4 = this.initOptions).powerPreference) != null ? _this$initOptions4$po : _this$initOptions4.powerPreference = 'high-performance';
		this._maxPixelRatio = platform.browser ? Math.min(1, window.devicePixelRatio) : 1;
		this.buffers = [];
		this._vram = {
			tex: 0,
			vb: 0,
			ib: 0,
			ub: 0
		};
		this._shaderStats = {
			vsCompiled: 0,
			fsCompiled: 0,
			linked: 0,
			materialShaders: 0,
			compileTime: 0
		};
		this.initializeContextCaches();
		this._drawCallsPerFrame = 0;
		this._shaderSwitchesPerFrame = 0;
		this._primsPerFrame = [];
		for (let i = PRIMITIVE_POINTS; i <= PRIMITIVE_TRIFAN; i++) {
			this._primsPerFrame[i] = 0;
		}
		this._renderTargetCreationTime = 0;
		this.scope = new ScopeSpace("Device");
		this.textureBias = this.scope.resolve("textureBias");
		this.textureBias.setValue(0.0);
	}
	postInit() {
		const vertexFormat = new VertexFormat(this, [{
			semantic: SEMANTIC_POSITION,
			components: 2,
			type: TYPE_FLOAT32
		}]);
		const positions = new Float32Array([-1, -1, 1, -1, -1, 1, 1, 1]);
		this.quadVertexBuffer = new VertexBuffer(this, vertexFormat, 4, BUFFER_STATIC, positions);
	}
	destroy() {
		var _this$quadVertexBuffe, _this$dynamicBuffers, _this$gpuProfiler;
		this.fire('destroy');
		(_this$quadVertexBuffe = this.quadVertexBuffer) == null || _this$quadVertexBuffe.destroy();
		this.quadVertexBuffer = null;
		(_this$dynamicBuffers = this.dynamicBuffers) == null || _this$dynamicBuffers.destroy();
		this.dynamicBuffers = null;
		(_this$gpuProfiler = this.gpuProfiler) == null || _this$gpuProfiler.destroy();
		this.gpuProfiler = null;
	}
	onDestroyShader(shader) {
		this.fire('destroy:shader', shader);
		const idx = this.shaders.indexOf(shader);
		if (idx !== -1) {
			this.shaders.splice(idx, 1);
		}
	}
	postDestroy() {
		this.scope = null;
		this.canvas = null;
	}
	toJSON(key) {
		return undefined;
	}
	initializeContextCaches() {
		this.indexBuffer = null;
		this.vertexBuffers = [];
		this.shader = null;
		this.shaderValid = undefined;
		this.shaderAsyncCompile = false;
		this.renderTarget = null;
	}
	initializeRenderState() {
		this.blendState = new BlendState();
		this.depthState = new DepthState();
		this.cullMode = CULLFACE_BACK;
		this.vx = this.vy = this.vw = this.vh = 0;
		this.sx = this.sy = this.sw = this.sh = 0;
		this.blendColor = new Color(0, 0, 0, 0);
	}
	setStencilState(stencilFront, stencilBack) {}
	setBlendState(blendState) {}
	setBlendColor(r, g, b, a) {}
	setDepthState(depthState) {}
	setCullMode(cullMode) {}
	setRenderTarget(renderTarget) {
		this.renderTarget = renderTarget;
	}
	setIndexBuffer(indexBuffer) {
		this.indexBuffer = indexBuffer;
	}
	setVertexBuffer(vertexBuffer) {
		if (vertexBuffer) {
			this.vertexBuffers.push(vertexBuffer);
		}
	}
	getRenderTarget() {
		return this.renderTarget;
	}
	initRenderTarget(target) {
		if (target.initialized) return;
		target.init();
		this.targets.add(target);
	}
	_isBrowserInterface(texture) {
		return this._isImageBrowserInterface(texture) || this._isImageCanvasInterface(texture) || this._isImageVideoInterface(texture);
	}
	_isImageBrowserInterface(texture) {
		return typeof ImageBitmap !== 'undefined' && texture instanceof ImageBitmap || typeof HTMLImageElement !== 'undefined' && texture instanceof HTMLImageElement;
	}
	_isImageCanvasInterface(texture) {
		return typeof HTMLCanvasElement !== 'undefined' && texture instanceof HTMLCanvasElement;
	}
	_isImageVideoInterface(texture) {
		return typeof HTMLVideoElement !== 'undefined' && texture instanceof HTMLVideoElement;
	}
	resizeCanvas(width, height) {
		const pixelRatio = Math.min(this._maxPixelRatio, platform.browser ? window.devicePixelRatio : 1);
		const w = Math.floor(width * pixelRatio);
		const h = Math.floor(height * pixelRatio);
		if (w !== this.canvas.width || h !== this.canvas.height) {
			this.setResolution(w, h);
		}
	}
	setResolution(width, height) {
		this.canvas.width = width;
		this.canvas.height = height;
		this.fire(GraphicsDevice.EVENT_RESIZE, width, height);
	}
	updateClientRect() {
		this.clientRect = this.canvas.getBoundingClientRect();
	}
	get width() {
		return this.canvas.width;
	}
	get height() {
		return this.canvas.height;
	}
	set fullscreen(fullscreen) {}
	get fullscreen() {
		return false;
	}
	set maxPixelRatio(ratio) {
		this._maxPixelRatio = ratio;
	}
	get maxPixelRatio() {
		return this._maxPixelRatio;
	}
	get deviceType() {
		return this._deviceType;
	}
	getBoneLimit() {
		return this.boneLimit;
	}
	setBoneLimit(maxBones) {
		this.boneLimit = maxBones;
	}
	startRenderPass(renderPass) {}
	endRenderPass(renderPass) {}
	startComputePass() {}
	endComputePass() {}
	frameStart() {
		this.renderPassIndex = 0;
		this.renderVersion++;
	}
	frameEnd() {}
	getRenderableHdrFormat(formats = [PIXELFORMAT_111110F, PIXELFORMAT_RGBA16F, PIXELFORMAT_RGBA32F], filterable = true) {
		for (let i = 0; i < formats.length; i++) {
			const format = formats[i];
			switch (format) {
				case PIXELFORMAT_111110F:
					{
						if (this.textureRG11B10Renderable) return format;
						break;
					}
				case PIXELFORMAT_RGBA16F:
					if (this.textureHalfFloatRenderable && (!filterable || this.textureHalfFloatFilterable)) {
						return format;
					}
					break;
				case PIXELFORMAT_RGBA32F:
					if (this.textureFloatRenderable && (!filterable || this.textureFloatFilterable)) {
						return format;
					}
					break;
			}
		}
		return undefined;
	}
}
GraphicsDevice.EVENT_RESIZE = 'resizecanvas';

let id$9 = 0;
class RenderTarget {
	constructor(options = {}) {
		var _options$face, _this$_colorBuffer, _this$_depthBuffer, _options$samples, _options$autoResolve, _options$flipY;
		this.name = void 0;
		this._device = void 0;
		this._colorBuffer = void 0;
		this._colorBuffers = void 0;
		this._depthBuffer = void 0;
		this._depth = void 0;
		this._stencil = void 0;
		this._samples = void 0;
		this.autoResolve = void 0;
		this._face = void 0;
		this.flipY = void 0;
		this.id = id$9++;
		const _arg2 = arguments[1];
		const _arg3 = arguments[2];
		if (options instanceof GraphicsDevice) {
			this._colorBuffer = _arg2;
			options = _arg3;
		} else {
			this._colorBuffer = options.colorBuffer;
		}
		if (this._colorBuffer) {
			this._colorBuffers = [this._colorBuffer];
		}
		this._depthBuffer = options.depthBuffer;
		this._face = (_options$face = options.face) != null ? _options$face : 0;
		if (this._depthBuffer) {
			const format = this._depthBuffer._format;
			if (format === PIXELFORMAT_DEPTH) {
				this._depth = true;
				this._stencil = false;
			} else if (format === PIXELFORMAT_DEPTHSTENCIL) {
				this._depth = true;
				this._stencil = true;
			} else {
				this._depth = false;
				this._stencil = false;
			}
		} else {
			var _options$depth, _options$stencil;
			this._depth = (_options$depth = options.depth) != null ? _options$depth : true;
			this._stencil = (_options$stencil = options.stencil) != null ? _options$stencil : false;
		}
		if (options.colorBuffers) {
			if (!this._colorBuffers) {
				this._colorBuffers = [...options.colorBuffers];
				this._colorBuffer = options.colorBuffers[0];
			}
		}
		const device = ((_this$_colorBuffer = this._colorBuffer) == null ? void 0 : _this$_colorBuffer.device) || ((_this$_depthBuffer = this._depthBuffer) == null ? void 0 : _this$_depthBuffer.device) || options.graphicsDevice;
		this._device = device;
		const {
			maxSamples
		} = this._device;
		this._samples = Math.min((_options$samples = options.samples) != null ? _options$samples : 1, maxSamples);
		if (device.isWebGPU) {
			this._samples = this._samples > 1 ? maxSamples : 1;
		}
		this.autoResolve = (_options$autoResolve = options.autoResolve) != null ? _options$autoResolve : true;
		this.name = options.name;
		if (!this.name) {
			var _this$_colorBuffer2;
			this.name = (_this$_colorBuffer2 = this._colorBuffer) == null ? void 0 : _this$_colorBuffer2.name;
		}
		if (!this.name) {
			var _this$_depthBuffer2;
			this.name = (_this$_depthBuffer2 = this._depthBuffer) == null ? void 0 : _this$_depthBuffer2.name;
		}
		if (!this.name) {
			this.name = "Untitled";
		}
		this.flipY = (_options$flipY = options.flipY) != null ? _options$flipY : false;
		this.validateMrt();
		this.impl = device.createRenderTargetImpl(this);
	}
	destroy() {
		const device = this._device;
		if (device) {
			device.targets.delete(this);
			if (device.renderTarget === this) {
				device.setRenderTarget(null);
			}
			this.destroyFrameBuffers();
		}
	}
	destroyFrameBuffers() {
		const device = this._device;
		if (device) {
			this.impl.destroy(device);
		}
	}
	destroyTextureBuffers() {
		var _this$_depthBuffer3, _this$_colorBuffers;
		(_this$_depthBuffer3 = this._depthBuffer) == null || _this$_depthBuffer3.destroy();
		this._depthBuffer = null;
		(_this$_colorBuffers = this._colorBuffers) == null || _this$_colorBuffers.forEach(colorBuffer => {
			colorBuffer.destroy();
		});
		this._colorBuffers = null;
		this._colorBuffer = null;
	}
	resize(width, height) {
		if (this.width !== width || this.height !== height) {
			var _this$_depthBuffer4, _this$_colorBuffers2;
			const device = this._device;
			this.destroyFrameBuffers();
			if (device.renderTarget === this) {
				device.setRenderTarget(null);
			}
			(_this$_depthBuffer4 = this._depthBuffer) == null || _this$_depthBuffer4.resize(width, height);
			(_this$_colorBuffers2 = this._colorBuffers) == null || _this$_colorBuffers2.forEach(colorBuffer => {
				colorBuffer.resize(width, height);
			});
			this.validateMrt();
			this.impl = device.createRenderTargetImpl(this);
		}
	}
	validateMrt() {}
	init() {
		this.impl.init(this._device, this);
	}
	get initialized() {
		return this.impl.initialized;
	}
	get device() {
		return this._device;
	}
	loseContext() {
		this.impl.loseContext();
	}
	resolve(color = true, depth = !!this._depthBuffer) {
		if (this._device && this._samples > 1) {
			this.impl.resolve(this._device, this, color, depth);
		}
	}
	copy(source, color, depth) {
		if (!this._device) {
			if (source._device) {
				this._device = source._device;
			} else {
				return false;
			}
		}
		const success = this._device.copyRenderTarget(source, this, color, depth);
		return success;
	}
	get samples() {
		return this._samples;
	}
	get depth() {
		return this._depth;
	}
	get stencil() {
		return this._stencil;
	}
	get colorBuffer() {
		return this._colorBuffer;
	}
	getColorBuffer(index) {
		var _this$_colorBuffers3;
		return (_this$_colorBuffers3 = this._colorBuffers) == null ? void 0 : _this$_colorBuffers3[index];
	}
	get depthBuffer() {
		return this._depthBuffer;
	}
	get face() {
		return this._face;
	}
	get width() {
		var _this$_colorBuffer3, _this$_depthBuffer5;
		return ((_this$_colorBuffer3 = this._colorBuffer) == null ? void 0 : _this$_colorBuffer3.width) || ((_this$_depthBuffer5 = this._depthBuffer) == null ? void 0 : _this$_depthBuffer5.width) || this._device.width;
	}
	get height() {
		var _this$_colorBuffer4, _this$_depthBuffer6;
		return ((_this$_colorBuffer4 = this._colorBuffer) == null ? void 0 : _this$_colorBuffer4.height) || ((_this$_depthBuffer6 = this._depthBuffer) == null ? void 0 : _this$_depthBuffer6.height) || this._device.height;
	}
}

class WebgpuBindGroup {
	constructor() {
		this.bindGroup = void 0;
	}
	update(bindGroup) {
		this.destroy();
		const device = bindGroup.device;
		const descr = this.createDescriptor(device, bindGroup);
		this.bindGroup = device.wgpu.createBindGroup(descr);
	}
	destroy() {
		this.bindGroup = null;
	}
	createDescriptor(device, bindGroup) {
		const entries = [];
		const format = bindGroup.format;
		let index = 0;
		bindGroup.uniformBuffers.forEach(ub => {
			const buffer = ub.persistent ? ub.impl.buffer : ub.allocation.gpuBuffer.buffer;
			entries.push({
				binding: index++,
				resource: {
					buffer: buffer,
					offset: 0,
					size: ub.format.byteSize
				}
			});
		});
		bindGroup.textures.forEach((tex, textureIndex) => {
			const wgpuTexture = tex.impl;
			const textureFormat = format.textureFormats[textureIndex];
			const view = wgpuTexture.getView(device);
			entries.push({
				binding: index++,
				resource: view
			});
			const sampler = wgpuTexture.getSampler(device, textureFormat.sampleType);
			entries.push({
				binding: index++,
				resource: sampler
			});
		});
		bindGroup.storageTextures.forEach((tex, textureIndex) => {
			const wgpuTexture = tex.impl;
			const view = wgpuTexture.getView(device);
			entries.push({
				binding: index++,
				resource: view
			});
		});
		const descr = {
			layout: bindGroup.format.impl.bindGroupLayout,
			entries: entries
		};
		return descr;
	}
}

class WebgpuUtils {
	static shaderStage(stage) {
		let ret = 0;
		if (stage & SHADERSTAGE_VERTEX) ret |= GPUShaderStage.VERTEX;
		if (stage & SHADERSTAGE_FRAGMENT) ret |= GPUShaderStage.FRAGMENT;
		if (stage & SHADERSTAGE_COMPUTE) ret |= GPUShaderStage.COMPUTE;
		return ret;
	}
}

const gpuTextureFormats = [];
gpuTextureFormats[PIXELFORMAT_A8] = '';
gpuTextureFormats[PIXELFORMAT_L8] = 'r8unorm';
gpuTextureFormats[PIXELFORMAT_LA8] = 'rg8unorm';
gpuTextureFormats[PIXELFORMAT_RGB565] = '';
gpuTextureFormats[PIXELFORMAT_RGBA5551] = '';
gpuTextureFormats[PIXELFORMAT_RGBA4] = '';
gpuTextureFormats[PIXELFORMAT_RGB8] = 'rgba8unorm';
gpuTextureFormats[PIXELFORMAT_RGBA8] = 'rgba8unorm';
gpuTextureFormats[PIXELFORMAT_DXT1] = 'bc1-rgba-unorm';
gpuTextureFormats[PIXELFORMAT_DXT3] = 'bc2-rgba-unorm';
gpuTextureFormats[PIXELFORMAT_DXT5] = 'bc3-rgba-unorm';
gpuTextureFormats[PIXELFORMAT_RGB16F] = '';
gpuTextureFormats[PIXELFORMAT_RGBA16F] = 'rgba16float';
gpuTextureFormats[PIXELFORMAT_R16F] = 'r16float';
gpuTextureFormats[PIXELFORMAT_RG16F] = 'rg16float';
gpuTextureFormats[PIXELFORMAT_RGB32F] = '';
gpuTextureFormats[PIXELFORMAT_RGBA32F] = 'rgba32float';
gpuTextureFormats[PIXELFORMAT_R32F] = 'r32float';
gpuTextureFormats[PIXELFORMAT_DEPTH] = 'depth32float';
gpuTextureFormats[PIXELFORMAT_DEPTHSTENCIL] = 'depth24plus-stencil8';
gpuTextureFormats[PIXELFORMAT_111110F] = 'rg11b10ufloat';
gpuTextureFormats[PIXELFORMAT_SRGB] = '';
gpuTextureFormats[PIXELFORMAT_SRGBA] = '';
gpuTextureFormats[PIXELFORMAT_ETC1] = '';
gpuTextureFormats[PIXELFORMAT_ETC2_RGB] = 'etc2-rgb8unorm';
gpuTextureFormats[PIXELFORMAT_ETC2_RGBA] = 'etc2-rgba8unorm';
gpuTextureFormats[PIXELFORMAT_PVRTC_2BPP_RGB_1] = '';
gpuTextureFormats[PIXELFORMAT_PVRTC_2BPP_RGBA_1] = '';
gpuTextureFormats[PIXELFORMAT_PVRTC_4BPP_RGB_1] = '';
gpuTextureFormats[PIXELFORMAT_PVRTC_4BPP_RGBA_1] = '';
gpuTextureFormats[PIXELFORMAT_ASTC_4x4] = 'astc-4x4-unorm';
gpuTextureFormats[PIXELFORMAT_ATC_RGB] = '';
gpuTextureFormats[PIXELFORMAT_ATC_RGBA] = '';
gpuTextureFormats[PIXELFORMAT_BGRA8] = 'bgra8unorm';
gpuTextureFormats[PIXELFORMAT_R8I] = 'r8sint';
gpuTextureFormats[PIXELFORMAT_R8U] = 'r8uint';
gpuTextureFormats[PIXELFORMAT_R16I] = 'r16sint';
gpuTextureFormats[PIXELFORMAT_R16U] = 'r16uint';
gpuTextureFormats[PIXELFORMAT_R32I] = 'r32sint';
gpuTextureFormats[PIXELFORMAT_R32U] = 'r32uint';
gpuTextureFormats[PIXELFORMAT_RG8I] = 'rg8sint';
gpuTextureFormats[PIXELFORMAT_RG8U] = 'rg8uint';
gpuTextureFormats[PIXELFORMAT_RG16I] = 'rg16sint';
gpuTextureFormats[PIXELFORMAT_RG16U] = 'rg16uint';
gpuTextureFormats[PIXELFORMAT_RG32I] = 'rg32sint';
gpuTextureFormats[PIXELFORMAT_RG32U] = 'rg32uint';
gpuTextureFormats[PIXELFORMAT_RGBA8I] = 'rgba8sint';
gpuTextureFormats[PIXELFORMAT_RGBA8U] = 'rgba8uint';
gpuTextureFormats[PIXELFORMAT_RGBA16I] = 'rgba16sint';
gpuTextureFormats[PIXELFORMAT_RGBA16U] = 'rgba16uint';
gpuTextureFormats[PIXELFORMAT_RGBA32I] = 'rgba32sint';
gpuTextureFormats[PIXELFORMAT_RGBA32U] = 'rgba32uint';

const samplerTypes = [];
samplerTypes[SAMPLETYPE_FLOAT] = 'filtering';
samplerTypes[SAMPLETYPE_UNFILTERABLE_FLOAT] = 'non-filtering';
samplerTypes[SAMPLETYPE_DEPTH] = 'comparison';
samplerTypes[SAMPLETYPE_INT] = 'comparison';
samplerTypes[SAMPLETYPE_UINT] = 'comparison';
const sampleTypes = [];
sampleTypes[SAMPLETYPE_FLOAT] = 'float';
sampleTypes[SAMPLETYPE_UNFILTERABLE_FLOAT] = 'unfilterable-float';
sampleTypes[SAMPLETYPE_DEPTH] = 'depth';
sampleTypes[SAMPLETYPE_INT] = 'sint';
sampleTypes[SAMPLETYPE_UINT] = 'uint';
const stringIds$1 = new StringIds();
class WebgpuBindGroupFormat {
	constructor(bindGroupFormat) {
		const device = bindGroupFormat.device;
		const {
			key,
			descr
		} = this.createDescriptor(bindGroupFormat);
		this.key = stringIds$1.get(key);
		this.bindGroupLayout = device.wgpu.createBindGroupLayout(descr);
	}
	destroy() {
		this.bindGroupLayout = null;
	}
	loseContext() {}
	getTextureSlot(bindGroupFormat, index) {
		return bindGroupFormat.bufferFormats.length + index * 2;
	}
	createDescriptor(bindGroupFormat) {
		const entries = [];
		let key = '';
		let index = 0;
		bindGroupFormat.bufferFormats.forEach(bufferFormat => {
			const visibility = WebgpuUtils.shaderStage(bufferFormat.visibility);
			key += `#${index}U:${visibility}`;
			entries.push({
				binding: index++,
				visibility: visibility,
				buffer: {
					type: 'uniform',
					hasDynamicOffset: true
				}
			});
		});
		bindGroupFormat.textureFormats.forEach(textureFormat => {
			const visibility = WebgpuUtils.shaderStage(textureFormat.visibility);
			const sampleType = textureFormat.sampleType;
			const viewDimension = textureFormat.textureDimension;
			const multisampled = false;
			const gpuSampleType = sampleTypes[sampleType];
			key += `#${index}T:${visibility}-${gpuSampleType}-${viewDimension}-${multisampled}`;
			entries.push({
				binding: index++,
				visibility: visibility,
				texture: {
					sampleType: gpuSampleType,
					viewDimension: viewDimension,
					multisampled: multisampled
				}
			});
			const gpuSamplerType = samplerTypes[sampleType];
			key += `#${index}S:${visibility}-${gpuSamplerType}`;
			entries.push({
				binding: index++,
				visibility: visibility,
				sampler: {
					type: gpuSamplerType
				}
			});
		});
		bindGroupFormat.storageTextureFormats.forEach(textureFormat => {
			const {
				format,
				textureDimension
			} = textureFormat;
			key += `#${index}ST:${format}-${textureDimension}`;
			entries.push({
				binding: index++,
				visibility: GPUShaderStage.COMPUTE,
				storageTexture: {
					access: 'write-only',
					format: gpuTextureFormats[format],
					viewDimension: textureDimension
				}
			});
		});
		const descr = {
			entries: entries
		};
		return {
			key,
			descr
		};
	}
}

class WebgpuBuffer {
	constructor() {
		this.buffer = null;
	}
	destroy(device) {
		if (this.buffer) {
			this.buffer.destroy();
			this.buffer = null;
		}
	}
	get initialized() {
		return !!this.buffer;
	}
	loseContext() {}
	unlock(device, usage, target, storage) {
		var _storage$byteOffset, _storage$buffer;
		const wgpu = device.wgpu;
		if (!this.buffer) {
			const size = storage.byteLength + 3 & ~3;
			this.buffer = device.wgpu.createBuffer({
				size: size,
				usage: target | GPUBufferUsage.COPY_DST
			});
		}
		const srcOffset = (_storage$byteOffset = storage.byteOffset) != null ? _storage$byteOffset : 0;
		const srcData = new Uint8Array((_storage$buffer = storage.buffer) != null ? _storage$buffer : storage, srcOffset, storage.byteLength);
		const data = new Uint8Array(this.buffer.size);
		data.set(srcData);
		wgpu.queue.writeBuffer(this.buffer, 0, data, 0, data.length);
	}
}

class WebgpuIndexBuffer extends WebgpuBuffer {
	constructor(indexBuffer) {
		super();
		this.format = null;
		this.format = indexBuffer.format === INDEXFORMAT_UINT16 ? "uint16" : "uint32";
	}
	unlock(indexBuffer) {
		const device = indexBuffer.device;
		super.unlock(device, indexBuffer.usage, GPUBufferUsage.INDEX, indexBuffer.storage);
	}
}

const array = {
	equals(arr1, arr2) {
		if (arr1.size !== arr2.size) {
			return false;
		}
		for (let i = 0; i < arr1.length; i++) {
			if (arr1[i] !== arr2[i]) {
				return false;
			}
		}
		return true;
	}
};

const gpuVertexFormats = [];
gpuVertexFormats[TYPE_INT8] = 'sint8';
gpuVertexFormats[TYPE_UINT8] = 'uint8';
gpuVertexFormats[TYPE_INT16] = 'sint16';
gpuVertexFormats[TYPE_UINT16] = 'uint16';
gpuVertexFormats[TYPE_INT32] = 'sint32';
gpuVertexFormats[TYPE_UINT32] = 'uint32';
gpuVertexFormats[TYPE_FLOAT32] = 'float32';
gpuVertexFormats[TYPE_FLOAT16] = 'float16';
const gpuVertexFormatsNormalized = [];
gpuVertexFormatsNormalized[TYPE_INT8] = 'snorm8';
gpuVertexFormatsNormalized[TYPE_UINT8] = 'unorm8';
gpuVertexFormatsNormalized[TYPE_INT16] = 'snorm16';
gpuVertexFormatsNormalized[TYPE_UINT16] = 'unorm16';
gpuVertexFormatsNormalized[TYPE_INT32] = 'sint32';
gpuVertexFormatsNormalized[TYPE_UINT32] = 'uint32';
gpuVertexFormatsNormalized[TYPE_FLOAT32] = 'float32';
gpuVertexFormatsNormalized[TYPE_FLOAT16] = 'float16';
class WebgpuVertexBufferLayout {
	constructor() {
		this.cache = new Map();
	}
	get(vertexFormat0, vertexFormat1 = null) {
		const key = this.getKey(vertexFormat0, vertexFormat1);
		let layout = this.cache.get(key);
		if (!layout) {
			layout = this.create(vertexFormat0, vertexFormat1);
			this.cache.set(key, layout);
		}
		return layout;
	}
	getKey(vertexFormat0, vertexFormat1 = null) {
		return `${vertexFormat0 == null ? void 0 : vertexFormat0.renderingHashString}-${vertexFormat1 == null ? void 0 : vertexFormat1.renderingHashString}`;
	}
	create(vertexFormat0, vertexFormat1) {
		const layout = [];
		const addFormat = format => {
			const interleaved = format.interleaved;
			const stepMode = format.instancing ? 'instance' : 'vertex';
			let attributes = [];
			const elementCount = format.elements.length;
			for (let i = 0; i < elementCount; i++) {
				const element = format.elements[i];
				const location = semanticToLocation[element.name];
				const formatTable = element.normalize ? gpuVertexFormatsNormalized : gpuVertexFormats;
				attributes.push({
					shaderLocation: location,
					offset: interleaved ? element.offset : 0,
					format: `${formatTable[element.dataType]}${element.numComponents > 1 ? 'x' + element.numComponents : ''}`
				});
				if (!interleaved || i === elementCount - 1) {
					layout.push({
						attributes: attributes,
						arrayStride: element.stride,
						stepMode: stepMode
					});
					attributes = [];
				}
			}
		};
		if (vertexFormat0) addFormat(vertexFormat0);
		if (vertexFormat1) addFormat(vertexFormat1);
		return layout;
	}
}

class WebgpuPipeline {
	constructor(device) {
		this.device = device;
	}
	getPipelineLayout(bindGroupFormats) {
		const bindGroupLayouts = [];
		bindGroupFormats.forEach(format => {
			bindGroupLayouts.push(format.bindGroupLayout);
		});
		const descr = {
			bindGroupLayouts: bindGroupLayouts
		};
		const pipelineLayout = this.device.wgpu.createPipelineLayout(descr);
		return pipelineLayout;
	}
}

const _primitiveTopology = ['point-list', 'line-list', undefined, 'line-strip', 'triangle-list', 'triangle-strip', undefined];
const _blendOperation = ['add', 'subtract', 'reverse-subtract', 'min', 'max'];
const _blendFactor = ['zero', 'one', 'src', 'one-minus-src', 'dst', 'one-minus-dst', 'src-alpha', 'src-alpha-saturated', 'one-minus-src-alpha', 'dst-alpha', 'one-minus-dst-alpha', 'constant', 'one-minus-constant'];
const _compareFunction = ['never', 'less', 'equal', 'less-equal', 'greater', 'not-equal', 'greater-equal', 'always'];
const _cullModes = ['none', 'back', 'front'];
const _stencilOps = ['keep', 'zero', 'replace', 'increment-clamp', 'increment-wrap', 'decrement-clamp', 'decrement-wrap', 'invert'];
class CacheEntry {
	constructor() {
		this.pipeline = void 0;
		this.hashes = void 0;
	}
}
class WebgpuRenderPipeline extends WebgpuPipeline {
	constructor(device) {
		super(device);
		this.lookupHashes = new Uint32Array(13);
		this.vertexBufferLayout = new WebgpuVertexBufferLayout();
		this.cache = new Map();
	}
	get(primitive, vertexFormat0, vertexFormat1, shader, renderTarget, bindGroupFormats, blendState, depthState, cullMode, stencilEnabled, stencilFront, stencilBack) {
		var _vertexFormat0$render, _vertexFormat1$render, _bindGroupFormats$0$k, _bindGroupFormats$, _bindGroupFormats$1$k, _bindGroupFormats$2, _bindGroupFormats$2$k, _bindGroupFormats$3;
		const lookupHashes = this.lookupHashes;
		lookupHashes[0] = primitive.type;
		lookupHashes[1] = shader.id;
		lookupHashes[2] = cullMode;
		lookupHashes[3] = depthState.key;
		lookupHashes[4] = blendState.key;
		lookupHashes[5] = (_vertexFormat0$render = vertexFormat0 == null ? void 0 : vertexFormat0.renderingHash) != null ? _vertexFormat0$render : 0;
		lookupHashes[6] = (_vertexFormat1$render = vertexFormat1 == null ? void 0 : vertexFormat1.renderingHash) != null ? _vertexFormat1$render : 0;
		lookupHashes[7] = renderTarget.impl.key;
		lookupHashes[8] = (_bindGroupFormats$0$k = (_bindGroupFormats$ = bindGroupFormats[0]) == null ? void 0 : _bindGroupFormats$.key) != null ? _bindGroupFormats$0$k : 0;
		lookupHashes[9] = (_bindGroupFormats$1$k = (_bindGroupFormats$2 = bindGroupFormats[1]) == null ? void 0 : _bindGroupFormats$2.key) != null ? _bindGroupFormats$1$k : 0;
		lookupHashes[10] = (_bindGroupFormats$2$k = (_bindGroupFormats$3 = bindGroupFormats[2]) == null ? void 0 : _bindGroupFormats$3.key) != null ? _bindGroupFormats$2$k : 0;
		lookupHashes[11] = stencilEnabled ? stencilFront.key : 0;
		lookupHashes[12] = stencilEnabled ? stencilBack.key : 0;
		const hash = hash32Fnv1a(lookupHashes);
		let cacheEntries = this.cache.get(hash);
		if (cacheEntries) {
			for (let i = 0; i < cacheEntries.length; i++) {
				const entry = cacheEntries[i];
				if (array.equals(entry.hashes, lookupHashes)) {
					return entry.pipeline;
				}
			}
		}
		const primitiveTopology = _primitiveTopology[primitive.type];
		const pipelineLayout = this.getPipelineLayout(bindGroupFormats);
		const vertexBufferLayout = this.vertexBufferLayout.get(vertexFormat0, vertexFormat1);
		const cacheEntry = new CacheEntry();
		cacheEntry.hashes = new Uint32Array(lookupHashes);
		cacheEntry.pipeline = this.create(primitiveTopology, shader, renderTarget, pipelineLayout, blendState, depthState, vertexBufferLayout, cullMode, stencilEnabled, stencilFront, stencilBack);
		if (cacheEntries) {
			cacheEntries.push(cacheEntry);
		} else {
			cacheEntries = [cacheEntry];
		}
		this.cache.set(hash, cacheEntries);
		return cacheEntry.pipeline;
	}
	getBlend(blendState) {
		let blend;
		if (blendState.blend) {
			blend = {
				color: {
					operation: _blendOperation[blendState.colorOp],
					srcFactor: _blendFactor[blendState.colorSrcFactor],
					dstFactor: _blendFactor[blendState.colorDstFactor]
				},
				alpha: {
					operation: _blendOperation[blendState.alphaOp],
					srcFactor: _blendFactor[blendState.alphaSrcFactor],
					dstFactor: _blendFactor[blendState.alphaDstFactor]
				}
			};
		}
		return blend;
	}
	getDepthStencil(depthState, renderTarget, stencilEnabled, stencilFront, stencilBack) {
		let depthStencil;
		const {
			depth,
			stencil
		} = renderTarget;
		if (depth || stencil) {
			depthStencil = {
				format: renderTarget.impl.depthFormat
			};
			if (depth) {
				depthStencil.depthWriteEnabled = depthState.write;
				depthStencil.depthCompare = _compareFunction[depthState.func];
				depthStencil.depthBias = depthState.depthBias;
				depthStencil.depthBiasSlopeScale = depthState.depthBiasSlope;
			} else {
				depthStencil.depthWriteEnabled = false;
				depthStencil.depthCompare = 'always';
			}
			if (stencil && stencilEnabled) {
				depthStencil.stencilReadMas = stencilFront.readMask;
				depthStencil.stencilWriteMask = stencilFront.writeMask;
				depthStencil.stencilFront = {
					compare: _compareFunction[stencilFront.func],
					failOp: _stencilOps[stencilFront.fail],
					passOp: _stencilOps[stencilFront.zpass],
					depthFailOp: _stencilOps[stencilFront.zfail]
				};
				depthStencil.stencilBack = {
					compare: _compareFunction[stencilBack.func],
					failOp: _stencilOps[stencilBack.fail],
					passOp: _stencilOps[stencilBack.zpass],
					depthFailOp: _stencilOps[stencilBack.zfail]
				};
			}
		}
		return depthStencil;
	}
	create(primitiveTopology, shader, renderTarget, pipelineLayout, blendState, depthState, vertexBufferLayout, cullMode, stencilEnabled, stencilFront, stencilBack) {
		const wgpu = this.device.wgpu;
		const webgpuShader = shader.impl;
		const descr = {
			vertex: {
				module: webgpuShader.getVertexShaderModule(),
				entryPoint: webgpuShader.vertexEntryPoint,
				buffers: vertexBufferLayout
			},
			primitive: {
				topology: primitiveTopology,
				frontFace: 'ccw',
				cullMode: _cullModes[cullMode]
			},
			depthStencil: this.getDepthStencil(depthState, renderTarget, stencilEnabled, stencilFront, stencilBack),
			multisample: {
				count: renderTarget.samples
			},
			layout: pipelineLayout
		};
		descr.fragment = {
			module: webgpuShader.getFragmentShaderModule(),
			entryPoint: webgpuShader.fragmentEntryPoint,
			targets: []
		};
		const colorAttachments = renderTarget.impl.colorAttachments;
		if (colorAttachments.length > 0) {
			let writeMask = 0;
			if (blendState.redWrite) writeMask |= GPUColorWrite.RED;
			if (blendState.greenWrite) writeMask |= GPUColorWrite.GREEN;
			if (blendState.blueWrite) writeMask |= GPUColorWrite.BLUE;
			if (blendState.alphaWrite) writeMask |= GPUColorWrite.ALPHA;
			const blend = this.getBlend(blendState);
			colorAttachments.forEach(attachment => {
				descr.fragment.targets.push({
					format: attachment.format,
					writeMask: writeMask,
					blend: blend
				});
			});
		}
		const pipeline = wgpu.createRenderPipeline(descr);
		return pipeline;
	}
}

class WebgpuComputePipeline extends WebgpuPipeline {
	get(shader, bindGroupFormat) {
		const pipelineLayout = this.getPipelineLayout([bindGroupFormat.impl]);
		const pipeline = this.create(shader, pipelineLayout);
		return pipeline;
	}
	create(shader, pipelineLayout) {
		const wgpu = this.device.wgpu;
		const webgpuShader = shader.impl;
		const descr = {
			compute: {
				module: webgpuShader.getComputeShaderModule(),
				entryPoint: webgpuShader.computeEntryPoint
			},
			layout: pipelineLayout
		};
		const pipeline = wgpu.createComputePipeline(descr);
		return pipeline;
	}
}

const stringIds = new StringIds();
class ColorAttachment {
	constructor() {
		this.format = void 0;
		this.multisampledBuffer = void 0;
	}
	destroy() {
		var _this$multisampledBuf;
		(_this$multisampledBuf = this.multisampledBuffer) == null || _this$multisampledBuf.destroy();
		this.multisampledBuffer = null;
	}
}
class WebgpuRenderTarget {
	constructor(renderTarget) {
		this.initialized = false;
		this.key = void 0;
		this.colorAttachments = [];
		this.depthFormat = void 0;
		this.hasStencil = void 0;
		this.depthTexture = null;
		this.depthTextureInternal = false;
		this.assignedColorTexture = null;
		this.renderPassDescriptor = {};
		this.renderTarget = renderTarget;
		if (renderTarget._colorBuffers) {
			renderTarget._colorBuffers.forEach((colorBuffer, index) => {
				this.setColorAttachment(index, undefined, colorBuffer.impl.format);
			});
		}
		this.updateKey();
	}
	destroy(device) {
		this.initialized = false;
		if (this.depthTextureInternal) {
			var _this$depthTexture;
			(_this$depthTexture = this.depthTexture) == null || _this$depthTexture.destroy();
			this.depthTexture = null;
		}
		this.assignedColorTexture = null;
		this.colorAttachments.forEach(colorAttachment => {
			colorAttachment.destroy();
		});
		this.colorAttachments.length = 0;
	}
	updateKey() {
		const rt = this.renderTarget;
		let key = `${rt.samples}:${rt.depth ? this.depthFormat : 'nodepth'}`;
		this.colorAttachments.forEach(colorAttachment => {
			key += `:${colorAttachment.format}`;
		});
		this.key = stringIds.get(key);
	}
	setDepthFormat(depthFormat) {
		this.depthFormat = depthFormat;
		this.hasStencil = depthFormat === 'depth24plus-stencil8';
	}
	assignColorTexture(gpuTexture) {
		this.assignedColorTexture = gpuTexture;
		const view = gpuTexture.createView();
		const colorAttachment = this.renderPassDescriptor.colorAttachments[0];
		const samples = this.renderTarget.samples;
		if (samples > 1) {
			colorAttachment.resolveTarget = view;
		} else {
			colorAttachment.view = view;
		}
		this.setColorAttachment(0, undefined, gpuTexture.format);
		this.updateKey();
	}
	setColorAttachment(index, multisampledBuffer, format) {
		if (!this.colorAttachments[index]) {
			this.colorAttachments[index] = new ColorAttachment();
		}
		if (multisampledBuffer) {
			this.colorAttachments[index].multisampledBuffer = multisampledBuffer;
		}
		if (format) {
			this.colorAttachments[index].format = format;
		}
	}
	init(device, renderTarget) {
		var _renderTarget$_colorB, _renderTarget$_colorB2;
		const wgpu = device.wgpu;
		this.initDepthStencil(wgpu, renderTarget);
		this.renderPassDescriptor.colorAttachments = [];
		const count = (_renderTarget$_colorB = (_renderTarget$_colorB2 = renderTarget._colorBuffers) == null ? void 0 : _renderTarget$_colorB2.length) != null ? _renderTarget$_colorB : 1;
		for (let i = 0; i < count; ++i) {
			var _this$colorAttachment;
			const colorAttachment = this.initColor(wgpu, renderTarget, i);
			const isDefaultFramebuffer = i === 0 && ((_this$colorAttachment = this.colorAttachments[0]) == null ? void 0 : _this$colorAttachment.format);
			if (colorAttachment.view || isDefaultFramebuffer) {
				this.renderPassDescriptor.colorAttachments.push(colorAttachment);
			}
		}
		this.initialized = true;
	}
	initDepthStencil(wgpu, renderTarget) {
		const {
			samples,
			width,
			height,
			depth,
			depthBuffer
		} = renderTarget;
		if (depth || depthBuffer) {
			if (!depthBuffer) {
				this.setDepthFormat('depth24plus-stencil8');
				const depthTextureDesc = {
					size: [width, height, 1],
					dimension: '2d',
					sampleCount: samples,
					format: this.depthFormat,
					usage: GPUTextureUsage.RENDER_ATTACHMENT
				};
				if (samples > 1) {
					depthTextureDesc.usage |= GPUTextureUsage.TEXTURE_BINDING;
				} else {
					depthTextureDesc.usage |= GPUTextureUsage.COPY_SRC;
				}
				this.depthTexture = wgpu.createTexture(depthTextureDesc);
				this.depthTextureInternal = true;
			} else {
				this.depthTexture = depthBuffer.impl.gpuTexture;
				this.setDepthFormat(depthBuffer.impl.format);
			}
			this.renderPassDescriptor.depthStencilAttachment = {
				view: this.depthTexture.createView()
			};
		}
	}
	initColor(wgpu, renderTarget, index) {
		const colorAttachment = {};
		const {
			samples,
			width,
			height
		} = renderTarget;
		const colorBuffer = renderTarget.getColorBuffer(index);
		let colorView = null;
		if (colorBuffer) {
			const mipLevelCount = 1;
			if (colorBuffer.cubemap) {
				colorView = colorBuffer.impl.createView({
					dimension: '2d',
					baseArrayLayer: renderTarget.face,
					arrayLayerCount: 1,
					mipLevelCount
				});
			} else {
				colorView = colorBuffer.impl.createView({
					mipLevelCount
				});
			}
		}
		if (samples > 1) {
			var _this$colorAttachment2, _this$colorAttachment3;
			const multisampledTextureDesc = {
				size: [width, height, 1],
				dimension: '2d',
				sampleCount: samples,
				format: (_this$colorAttachment2 = (_this$colorAttachment3 = this.colorAttachments[index]) == null ? void 0 : _this$colorAttachment3.format) != null ? _this$colorAttachment2 : colorBuffer.impl.format,
				usage: GPUTextureUsage.RENDER_ATTACHMENT
			};
			const multisampledColorBuffer = wgpu.createTexture(multisampledTextureDesc);
			this.setColorAttachment(index, multisampledColorBuffer, multisampledTextureDesc.format);
			colorAttachment.view = multisampledColorBuffer.createView();
			colorAttachment.resolveTarget = colorView;
		} else {
			colorAttachment.view = colorView;
		}
		return colorAttachment;
	}
	setupForRenderPass(renderPass) {
		var _this$renderPassDescr, _this$renderPassDescr2;
		const count = (_this$renderPassDescr = (_this$renderPassDescr2 = this.renderPassDescriptor.colorAttachments) == null ? void 0 : _this$renderPassDescr2.length) != null ? _this$renderPassDescr : 0;
		for (let i = 0; i < count; ++i) {
			const colorAttachment = this.renderPassDescriptor.colorAttachments[i];
			const colorOps = renderPass.colorArrayOps[i];
			colorAttachment.clearValue = colorOps.clearValue;
			colorAttachment.loadOp = colorOps.clear ? 'clear' : 'load';
			colorAttachment.storeOp = colorOps.store ? 'store' : 'discard';
		}
		const depthAttachment = this.renderPassDescriptor.depthStencilAttachment;
		if (depthAttachment) {
			depthAttachment.depthClearValue = renderPass.depthStencilOps.clearDepthValue;
			depthAttachment.depthLoadOp = renderPass.depthStencilOps.clearDepth ? 'clear' : 'load';
			depthAttachment.depthStoreOp = renderPass.depthStencilOps.storeDepth ? 'store' : 'discard';
			depthAttachment.depthReadOnly = false;
			if (this.hasStencil) {
				depthAttachment.stencilClearValue = renderPass.depthStencilOps.clearStencilValue;
				depthAttachment.stencilLoadOp = renderPass.depthStencilOps.clearStencil ? 'clear' : 'load';
				depthAttachment.stencilStoreOp = renderPass.depthStencilOps.storeStencil ? 'store' : 'discard';
				depthAttachment.stencilReadOnly = false;
			}
		}
	}
	loseContext() {
		this.initialized = false;
	}
	resolve(device, target, color, depth) {}
}

const uniformTypeToNumComponents = [];
uniformTypeToNumComponents[UNIFORMTYPE_FLOAT] = 1;
uniformTypeToNumComponents[UNIFORMTYPE_VEC2] = 2;
uniformTypeToNumComponents[UNIFORMTYPE_VEC3] = 3;
uniformTypeToNumComponents[UNIFORMTYPE_VEC4] = 4;
uniformTypeToNumComponents[UNIFORMTYPE_INT] = 1;
uniformTypeToNumComponents[UNIFORMTYPE_IVEC2] = 2;
uniformTypeToNumComponents[UNIFORMTYPE_IVEC3] = 3;
uniformTypeToNumComponents[UNIFORMTYPE_IVEC4] = 4;
uniformTypeToNumComponents[UNIFORMTYPE_BOOL] = 1;
uniformTypeToNumComponents[UNIFORMTYPE_BVEC2] = 2;
uniformTypeToNumComponents[UNIFORMTYPE_BVEC3] = 3;
uniformTypeToNumComponents[UNIFORMTYPE_BVEC4] = 4;
uniformTypeToNumComponents[UNIFORMTYPE_MAT2] = 8;
uniformTypeToNumComponents[UNIFORMTYPE_MAT3] = 12;
uniformTypeToNumComponents[UNIFORMTYPE_MAT4] = 16;
uniformTypeToNumComponents[UNIFORMTYPE_UINT] = 1;
uniformTypeToNumComponents[UNIFORMTYPE_UVEC2] = 2;
uniformTypeToNumComponents[UNIFORMTYPE_UVEC3] = 3;
uniformTypeToNumComponents[UNIFORMTYPE_UVEC4] = 4;
class UniformFormat {
	get isArrayType() {
		return this.count > 0;
	}
	constructor(name, type, count = 0) {
		this.name = void 0;
		this.type = void 0;
		this.byteSize = void 0;
		this.offset = void 0;
		this.scopeId = void 0;
		this.count = void 0;
		this.numComponents = void 0;
		this.shortName = name;
		this.name = count ? `${name}[0]` : name;
		this.type = type;
		this.numComponents = uniformTypeToNumComponents[type];
		this.updateType = type;
		if (count > 0) {
			switch (type) {
				case UNIFORMTYPE_FLOAT:
					this.updateType = UNIFORMTYPE_FLOATARRAY;
					break;
				case UNIFORMTYPE_INT:
					this.updateType = UNIFORMTYPE_INTARRAY;
					break;
				case UNIFORMTYPE_UINT:
					this.updateType = UNIFORMTYPE_UINTARRAY;
					break;
				case UNIFORMTYPE_BOOL:
					this.updateType = UNIFORMTYPE_BOOLARRAY;
					break;
				case UNIFORMTYPE_VEC2:
					this.updateType = UNIFORMTYPE_VEC2ARRAY;
					break;
				case UNIFORMTYPE_IVEC2:
					this.updateType = UNIFORMTYPE_IVEC2ARRAY;
					break;
				case UNIFORMTYPE_UVEC2:
					this.updateType = UNIFORMTYPE_UVEC2ARRAY;
					break;
				case UNIFORMTYPE_BVEC2:
					this.updateType = UNIFORMTYPE_BVEC2ARRAY;
					break;
				case UNIFORMTYPE_VEC3:
					this.updateType = UNIFORMTYPE_VEC3ARRAY;
					break;
				case UNIFORMTYPE_IVEC3:
					this.updateType = UNIFORMTYPE_IVEC3ARRAY;
					break;
				case UNIFORMTYPE_UVEC3:
					this.updateType = UNIFORMTYPE_UVEC3ARRAY;
					break;
				case UNIFORMTYPE_BVEC3:
					this.updateType = UNIFORMTYPE_BVEC3ARRAY;
					break;
				case UNIFORMTYPE_VEC4:
					this.updateType = UNIFORMTYPE_VEC4ARRAY;
					break;
				case UNIFORMTYPE_IVEC4:
					this.updateType = UNIFORMTYPE_IVEC4ARRAY;
					break;
				case UNIFORMTYPE_UVEC4:
					this.updateType = UNIFORMTYPE_UVEC4ARRAY;
					break;
				case UNIFORMTYPE_BVEC4:
					this.updateType = UNIFORMTYPE_BVEC4ARRAY;
					break;
				case UNIFORMTYPE_MAT4:
					this.updateType = UNIFORMTYPE_MAT4ARRAY;
					break;
			}
		}
		this.count = count;
		let componentSize = this.numComponents;
		if (count) {
			componentSize = math.roundUp(componentSize, 4);
		}
		this.byteSize = componentSize * 4;
		if (count) this.byteSize *= count;
	}
	calculateOffset(offset) {
		let alignment = this.byteSize <= 8 ? this.byteSize : 16;
		if (this.count) alignment = 16;
		offset = math.roundUp(offset, alignment);
		this.offset = offset / 4;
	}
}
class UniformBufferFormat {
	constructor(graphicsDevice, uniforms) {
		this.byteSize = 0;
		this.map = new Map();
		this.scope = graphicsDevice.scope;
		this.uniforms = uniforms;
		let offset = 0;
		for (let i = 0; i < uniforms.length; i++) {
			const uniform = uniforms[i];
			uniform.calculateOffset(offset);
			offset = uniform.offset * 4 + uniform.byteSize;
			uniform.scopeId = this.scope.resolve(uniform.name);
			this.map.set(uniform.name, uniform);
		}
		this.byteSize = math.roundUp(offset, 16);
	}
	get(name) {
		return this.map.get(name);
	}
	getShaderDeclaration(bindGroup, bindIndex) {
		const name = bindGroupNames[bindGroup];
		let code = `layout(set = ${bindGroup}, binding = ${bindIndex}, std140) uniform ub_${name} {\n`;
		this.uniforms.forEach(uniform => {
			const typeString = uniformTypeToName[uniform.type];
			code += `    ${typeString} ${uniform.shortName}${uniform.count ? `[${uniform.count}]` : ''};\n`;
		});
		return code + '};\n';
	}
}

let id$8 = 0;
const textureDimensionInfo = {
	[TEXTUREDIMENSION_2D]: 'texture2D',
	[TEXTUREDIMENSION_CUBE]: 'textureCube',
	[TEXTUREDIMENSION_3D]: 'texture3D',
	[TEXTUREDIMENSION_2D_ARRAY]: 'texture2DArray'
};
class BindBufferFormat {
	constructor(name, visibility) {
		this.name = name;
		this.visibility = visibility;
	}
}
class BindTextureFormat {
	constructor(name, visibility, textureDimension = TEXTUREDIMENSION_2D, sampleType = SAMPLETYPE_FLOAT) {
		this.scopeId = void 0;
		this.name = name;
		this.visibility = visibility;
		this.textureDimension = textureDimension;
		this.sampleType = sampleType;
	}
}
class BindStorageTextureFormat {
	constructor(name, format = PIXELFORMAT_RGBA8, textureDimension = TEXTUREDIMENSION_2D) {
		this.scopeId = void 0;
		this.name = name;
		this.format = format;
		this.textureDimension = textureDimension;
	}
}
class BindGroupFormat {
	constructor(graphicsDevice, bufferFormats = [], textureFormats = [], storageTextureFormats = [], options = {}) {
		var _options$compute;
		this.compute = false;
		this.id = id$8++;
		this.compute = (_options$compute = options.compute) != null ? _options$compute : false;
		this.device = graphicsDevice;
		const scope = graphicsDevice.scope;
		this.bufferFormats = bufferFormats;
		this.bufferFormatsMap = new Map();
		bufferFormats.forEach((bf, i) => this.bufferFormatsMap.set(bf.name, i));
		this.textureFormats = textureFormats;
		this.textureFormatsMap = new Map();
		textureFormats.forEach((tf, i) => {
			this.textureFormatsMap.set(tf.name, i);
			tf.scopeId = scope.resolve(tf.name);
		});
		this.storageTextureFormats = storageTextureFormats;
		this.storageTextureFormatsMap = new Map();
		storageTextureFormats.forEach((tf, i) => {
			this.storageTextureFormatsMap.set(tf.name, i);
			tf.scopeId = scope.resolve(tf.name);
		});
		this.impl = graphicsDevice.createBindGroupFormatImpl(this);
	}
	destroy() {
		this.impl.destroy();
	}
	getTexture(name) {
		const index = this.textureFormatsMap.get(name);
		if (index !== undefined) {
			return this.textureFormats[index];
		}
		return null;
	}
	getStorageTexture(name) {
		const index = this.storageTextureFormatsMap.get(name);
		if (index !== undefined) {
			return this.storageTextureFormats[index];
		}
		return null;
	}
	getShaderDeclarationTextures(bindGroup) {
		let code = '';
		let bindIndex = this.bufferFormats.length;
		this.textureFormats.forEach(format => {
			let textureType = textureDimensionInfo[format.textureDimension];
			let namePostfix = '';
			let extraCode = '';
			if (textureType === 'texture2DArray') {
				namePostfix = '_texture';
				extraCode = `#define ${format.name} sampler2DArray(${format.name}${namePostfix}, ${format.name}_sampler)\n`;
			}
			if (format.sampleType === SAMPLETYPE_INT) {
				textureType = `i${textureType}`;
			} else if (format.sampleType === SAMPLETYPE_UINT) {
				textureType = `u${textureType}`;
			}
			code += `layout(set = ${bindGroup}, binding = ${bindIndex++}) uniform ${textureType} ${format.name}${namePostfix};\n` + `layout(set = ${bindGroup}, binding = ${bindIndex++}) uniform sampler ${format.name}_sampler;\n` + extraCode;
		});
		return code;
	}
	loseContext() {}
}

const KEYWORD$1 = /[ \t]*(\battribute\b|\bvarying\b|\buniform\b)/g;
const KEYWORD_LINE = /(\battribute\b|\bvarying\b|\bout\b|\buniform\b)[ \t]*([^;]+)([;]+)/g;
const MARKER = '@@@';
const ARRAY_IDENTIFIER = /([\w-]+)\[(.*?)\]/;
const precisionQualifiers = new Set(['highp', 'mediump', 'lowp']);
const shadowSamplers = new Set(['sampler2DShadow', 'samplerCubeShadow', 'sampler2DArrayShadow']);
const textureDimensions = {
	sampler2D: TEXTUREDIMENSION_2D,
	sampler3D: TEXTUREDIMENSION_3D,
	samplerCube: TEXTUREDIMENSION_CUBE,
	samplerCubeShadow: TEXTUREDIMENSION_CUBE,
	sampler2DShadow: TEXTUREDIMENSION_2D,
	sampler2DArray: TEXTUREDIMENSION_2D_ARRAY,
	sampler2DArrayShadow: TEXTUREDIMENSION_2D_ARRAY,
	isampler2D: TEXTUREDIMENSION_2D,
	usampler2D: TEXTUREDIMENSION_2D,
	isampler3D: TEXTUREDIMENSION_3D,
	usampler3D: TEXTUREDIMENSION_3D,
	isamplerCube: TEXTUREDIMENSION_CUBE,
	usamplerCube: TEXTUREDIMENSION_CUBE,
	isampler2DArray: TEXTUREDIMENSION_2D_ARRAY,
	usampler2DArray: TEXTUREDIMENSION_2D_ARRAY
};
class UniformLine {
	constructor(line, shader) {
		this.line = line;
		const words = line.trim().split(/\s+/);
		if (precisionQualifiers.has(words[0])) {
			this.precision = words.shift();
		}
		this.type = words.shift();
		if (line.includes(',')) ;
		if (line.includes('[')) {
			const rest = words.join(' ');
			const match = ARRAY_IDENTIFIER.exec(rest);
			this.name = match[1];
			this.arraySize = Number(match[2]);
			if (isNaN(this.arraySize)) {
				shader.failed = true;
			}
		} else {
			this.name = words.shift();
			this.arraySize = 0;
		}
		this.isSampler = this.type.indexOf('sampler') !== -1;
		this.isSignedInt = this.type.indexOf('isampler') !== -1;
		this.isUnsignedInt = this.type.indexOf('usampler') !== -1;
	}
}
class ShaderProcessor {
	static run(device, shaderDefinition, shader) {
		const varyingMap = new Map();
		const vertexExtracted = ShaderProcessor.extract(shaderDefinition.vshader);
		const fragmentExtracted = ShaderProcessor.extract(shaderDefinition.fshader);
		const attributesBlock = ShaderProcessor.processAttributes(vertexExtracted.attributes, shaderDefinition.attributes, shaderDefinition.processingOptions);
		const vertexVaryingsBlock = ShaderProcessor.processVaryings(vertexExtracted.varyings, varyingMap, true);
		const fragmentVaryingsBlock = ShaderProcessor.processVaryings(fragmentExtracted.varyings, varyingMap, false);
		const outBlock = ShaderProcessor.processOuts(fragmentExtracted.outs);
		const concatUniforms = vertexExtracted.uniforms.concat(fragmentExtracted.uniforms);
		const uniforms = Array.from(new Set(concatUniforms));
		const parsedUniforms = uniforms.map(line => new UniformLine(line, shader));
		const uniformsData = ShaderProcessor.processUniforms(device, parsedUniforms, shaderDefinition.processingOptions, shader);
		const vBlock = attributesBlock + '\n' + vertexVaryingsBlock + '\n' + uniformsData.code;
		const vshader = vertexExtracted.src.replace(MARKER, vBlock);
		const fBlock = fragmentVaryingsBlock + '\n' + outBlock + '\n' + uniformsData.code;
		const fshader = fragmentExtracted.src.replace(MARKER, fBlock);
		return {
			vshader: vshader,
			fshader: fshader,
			meshUniformBufferFormat: uniformsData.meshUniformBufferFormat,
			meshBindGroupFormat: uniformsData.meshBindGroupFormat
		};
	}
	static extract(src) {
		const attributes = [];
		const varyings = [];
		const outs = [];
		const uniforms = [];
		let replacement = `${MARKER}\n`;
		let match;
		while ((match = KEYWORD$1.exec(src)) !== null) {
			const keyword = match[1];
			switch (keyword) {
				case 'attribute':
				case 'varying':
				case 'uniform':
				case 'out':
					{
						KEYWORD_LINE.lastIndex = match.index;
						const lineMatch = KEYWORD_LINE.exec(src);
						if (keyword === 'attribute') {
							attributes.push(lineMatch[2]);
						} else if (keyword === 'varying') {
							varyings.push(lineMatch[2]);
						} else if (keyword === 'out') {
							outs.push(lineMatch[2]);
						} else if (keyword === 'uniform') {
							uniforms.push(lineMatch[2]);
						}
						src = ShaderProcessor.cutOut(src, match.index, KEYWORD_LINE.lastIndex, replacement);
						KEYWORD$1.lastIndex = match.index + replacement.length;
						replacement = '';
						break;
					}
			}
		}
		return {
			src,
			attributes,
			varyings,
			outs,
			uniforms
		};
	}
	static processUniforms(device, uniforms, processingOptions, shader) {
		const uniformLinesSamplers = [];
		const uniformLinesNonSamplers = [];
		uniforms.forEach(uniform => {
			if (uniform.isSampler) {
				uniformLinesSamplers.push(uniform);
			} else {
				uniformLinesNonSamplers.push(uniform);
			}
		});
		const meshUniforms = [];
		uniformLinesNonSamplers.forEach(uniform => {
			if (!processingOptions.hasUniform(uniform.name)) {
				const uniformType = uniformTypeToName.indexOf(uniform.type);
				const uniformFormat = new UniformFormat(uniform.name, uniformType, uniform.arraySize);
				meshUniforms.push(uniformFormat);
			}
		});
		const meshUniformBufferFormat = meshUniforms.length ? new UniformBufferFormat(device, meshUniforms) : null;
		const bufferFormats = [];
		if (meshUniformBufferFormat) {
			bufferFormats.push(new BindBufferFormat(UNIFORM_BUFFER_DEFAULT_SLOT_NAME, SHADERSTAGE_VERTEX | SHADERSTAGE_FRAGMENT));
		}
		const textureFormats = [];
		uniformLinesSamplers.forEach(uniform => {
			if (!processingOptions.hasTexture(uniform.name)) {
				let sampleType = SAMPLETYPE_FLOAT;
				if (uniform.isSignedInt) {
					sampleType = SAMPLETYPE_INT;
				} else if (uniform.isUnsignedInt) {
					sampleType = SAMPLETYPE_UINT;
				} else {
					if (uniform.precision === 'highp') sampleType = SAMPLETYPE_UNFILTERABLE_FLOAT;
					if (shadowSamplers.has(uniform.type)) sampleType = SAMPLETYPE_DEPTH;
				}
				const dimension = textureDimensions[uniform.type];
				textureFormats.push(new BindTextureFormat(uniform.name, SHADERSTAGE_VERTEX | SHADERSTAGE_FRAGMENT, dimension, sampleType));
			}
		});
		const meshBindGroupFormat = new BindGroupFormat(device, bufferFormats, textureFormats);
		let code = '';
		processingOptions.uniformFormats.forEach((format, bindGroupIndex) => {
			if (format) {
				code += format.getShaderDeclaration(bindGroupIndex, 0);
			}
		});
		if (meshUniformBufferFormat) {
			code += meshUniformBufferFormat.getShaderDeclaration(BINDGROUP_MESH, 0);
		}
		processingOptions.bindGroupFormats.forEach((format, bindGroupIndex) => {
			if (format) {
				code += format.getShaderDeclarationTextures(bindGroupIndex);
			}
		});
		code += meshBindGroupFormat.getShaderDeclarationTextures(BINDGROUP_MESH);
		return {
			code,
			meshUniformBufferFormat,
			meshBindGroupFormat
		};
	}
	static processVaryings(varyingLines, varyingMap, isVertex) {
		let block = '';
		const op = isVertex ? 'out' : 'in';
		varyingLines.forEach((line, index) => {
			const words = ShaderProcessor.splitToWords(line);
			const type = words[0];
			const name = words[1];
			if (isVertex) {
				varyingMap.set(name, index);
			} else {
				index = varyingMap.get(name);
			}
			block += `layout(location = ${index}) ${op} ${type} ${name};\n`;
		});
		return block;
	}
	static processOuts(outsLines) {
		let block = '';
		outsLines.forEach((line, index) => {
			block += `layout(location = ${index}) out ${line};\n`;
		});
		return block;
	}
	static getTypeCount(type) {
		const lastChar = type.substring(type.length - 1);
		const num = parseInt(lastChar, 10);
		return isNaN(num) ? 1 : num;
	}
	static processAttributes(attributeLines, shaderDefinitionAttributes, processingOptions) {
		let block = '';
		attributeLines.forEach(line => {
			const words = ShaderProcessor.splitToWords(line);
			let type = words[0];
			let name = words[1];
			if (shaderDefinitionAttributes.hasOwnProperty(name)) {
				const semantic = shaderDefinitionAttributes[name];
				const location = semanticToLocation[semantic];
				let copyCode;
				const element = processingOptions.getVertexElement(semantic);
				if (element) {
					const dataType = element.dataType;
					if (dataType !== TYPE_FLOAT32 && dataType !== TYPE_FLOAT16 && !element.normalize && !element.asInt) {
						const attribNumElements = ShaderProcessor.getTypeCount(type);
						const newName = `_private_${name}`;
						copyCode = `vec${attribNumElements} ${name} = vec${attribNumElements}(${newName});\n`;
						name = newName;
						const isSignedType = dataType === TYPE_INT8 || dataType === TYPE_INT16 || dataType === TYPE_INT32;
						if (attribNumElements === 1) {
							type = isSignedType ? 'int' : 'uint';
						} else {
							type = isSignedType ? `ivec${attribNumElements}` : `uvec${attribNumElements}`;
						}
					}
				}
				block += `layout(location = ${location}) in ${type} ${name};\n`;
				if (copyCode) {
					block += copyCode;
				}
			}
		});
		return block;
	}
	static splitToWords(line) {
		line = line.replace(/\s+/g, ' ').trim();
		return line.split(' ');
	}
	static cutOut(src, start, end, replacement) {
		return src.substring(0, start) + replacement + src.substring(end);
	}
}

class WebgpuShader {
	constructor(shader) {
		this._vertexCode = null;
		this._fragmentCode = null;
		this._computeCode = null;
		this.vertexEntryPoint = 'main';
		this.fragmentEntryPoint = 'main';
		this.computeEntryPoint = 'main';
		this.shader = shader;
		const definition = shader.definition;
		if (definition.shaderLanguage === SHADERLANGUAGE_WGSL) {
			var _definition$vshader, _definition$fshader, _definition$cshader;
			this._vertexCode = (_definition$vshader = definition.vshader) != null ? _definition$vshader : null;
			this._fragmentCode = (_definition$fshader = definition.fshader) != null ? _definition$fshader : null;
			this._computeCode = (_definition$cshader = definition.cshader) != null ? _definition$cshader : null;
			this.meshUniformBufferFormat = definition.meshUniformBufferFormat;
			this.meshBindGroupFormat = definition.meshBindGroupFormat;
			this.vertexEntryPoint = 'vertexMain';
			this.fragmentEntryPoint = 'fragmentMain';
			shader.ready = true;
		} else {
			if (definition.processingOptions) {
				this.process();
			}
		}
	}
	destroy(shader) {
		this._vertexCode = null;
		this._fragmentCode = null;
	}
	createShaderModule(code, shaderType) {
		const device = this.shader.device;
		const wgpu = device.wgpu;
		const shaderModule = wgpu.createShaderModule({
			code: code
		});
		return shaderModule;
	}
	getVertexShaderModule() {
		return this.createShaderModule(this._vertexCode, 'Vertex');
	}
	getFragmentShaderModule() {
		return this.createShaderModule(this._fragmentCode, 'Fragment');
	}
	getComputeShaderModule() {
		return this.createShaderModule(this._computeCode, 'Compute');
	}
	process() {
		const shader = this.shader;
		const processed = ShaderProcessor.run(shader.device, shader.definition, shader);
		this._vertexCode = this.transpile(processed.vshader, 'vertex', shader.definition.vshader);
		this._fragmentCode = this.transpile(processed.fshader, 'fragment', shader.definition.fshader);
		if (!(this._vertexCode && this._fragmentCode)) {
			shader.failed = true;
		} else {
			shader.ready = true;
		}
		shader.meshUniformBufferFormat = processed.meshUniformBufferFormat;
		shader.meshBindGroupFormat = processed.meshBindGroupFormat;
	}
	transpile(src, shaderType, originalSrc) {
		try {
			const spirv = this.shader.device.glslang.compileGLSL(src, shaderType);
			return this.shader.device.twgsl.convertSpirV2WGSL(spirv);
		} catch (err) {
			console.error(`Failed to transpile webgl ${shaderType} shader [${this.shader.label}] to WebGPU: [${err.message}] while rendering ${void 0}`, {
				processed: src,
				original: originalSrc,
				shader: this.shader
			});
		}
	}
	get vertexCode() {
		return this._vertexCode;
	}
	get fragmentCode() {
		return this._fragmentCode;
	}
	loseContext() {}
	restoreContext(device, shader) {}
}

class TextureUtils {
	static calcLevelDimension(dimension, mipLevel) {
		return Math.max(dimension >> mipLevel, 1);
	}
	static calcMipLevelsCount(width, height, depth = 1) {
		return 1 + Math.floor(Math.log2(Math.max(width, height, depth)));
	}
	static calcLevelGpuSize(width, height, depth, format) {
		var _pixelFormatInfo$get$, _pixelFormatInfo$get, _formatInfo$blockSize;
		const formatInfo = pixelFormatInfo.get(format);
		const pixelSize = (_pixelFormatInfo$get$ = (_pixelFormatInfo$get = pixelFormatInfo.get(format)) == null ? void 0 : _pixelFormatInfo$get.size) != null ? _pixelFormatInfo$get$ : 0;
		if (pixelSize > 0) {
			return width * height * depth * pixelSize;
		}
		const blockSize = (_formatInfo$blockSize = formatInfo.blockSize) != null ? _formatInfo$blockSize : 0;
		let blockWidth = Math.floor((width + 3) / 4);
		const blockHeight = Math.floor((height + 3) / 4);
		const blockDepth = Math.floor((depth + 3) / 4);
		if (format === PIXELFORMAT_PVRTC_2BPP_RGB_1 || format === PIXELFORMAT_PVRTC_2BPP_RGBA_1) {
			blockWidth = Math.max(Math.floor(blockWidth / 2), 1);
		}
		return blockWidth * blockHeight * blockDepth * blockSize;
	}
	static calcGpuSize(width, height, depth, format, mipmaps, cubemap) {
		let result = 0;
		while (1) {
			result += TextureUtils.calcLevelGpuSize(width, height, depth, format);
			if (!mipmaps || width === 1 && height === 1 && depth === 1) {
				break;
			}
			width = Math.max(width >> 1, 1);
			height = Math.max(height >> 1, 1);
			depth = Math.max(depth >> 1, 1);
		}
		return result * (cubemap ? 6 : 1);
	}
}

const gpuAddressModes = [];
gpuAddressModes[ADDRESS_REPEAT] = 'repeat';
gpuAddressModes[ADDRESS_CLAMP_TO_EDGE] = 'clamp-to-edge';
gpuAddressModes[ADDRESS_MIRRORED_REPEAT] = 'mirror-repeat';
const gpuFilterModes = [];
gpuFilterModes[FILTER_NEAREST] = {
	level: 'nearest',
	mip: 'nearest'
};
gpuFilterModes[FILTER_LINEAR] = {
	level: 'linear',
	mip: 'nearest'
};
gpuFilterModes[FILTER_NEAREST_MIPMAP_NEAREST] = {
	level: 'nearest',
	mip: 'nearest'
};
gpuFilterModes[FILTER_NEAREST_MIPMAP_LINEAR] = {
	level: 'nearest',
	mip: 'linear'
};
gpuFilterModes[FILTER_LINEAR_MIPMAP_NEAREST] = {
	level: 'linear',
	mip: 'nearest'
};
gpuFilterModes[FILTER_LINEAR_MIPMAP_LINEAR] = {
	level: 'linear',
	mip: 'linear'
};
const dummyUse = thingOne => {};
class WebgpuTexture {
	constructor(texture) {
		this.gpuTexture = void 0;
		this.view = void 0;
		this.samplers = [];
		this.descr = void 0;
		this.format = void 0;
		this.texture = texture;
		this.format = gpuTextureFormats[texture.format];
		this.create(texture.device);
	}
	create(device) {
		const texture = this.texture;
		const wgpu = device.wgpu;
		const mipLevelCount = texture.requiredMipLevels;
		this.descr = {
			size: {
				width: texture.width,
				height: texture.height,
				depthOrArrayLayers: texture.cubemap ? 6 : texture.array ? texture.arrayLength : 1
			},
			format: this.format,
			mipLevelCount: mipLevelCount,
			sampleCount: 1,
			dimension: texture.volume ? '3d' : '2d',
			usage: GPUTextureUsage.TEXTURE_BINDING | GPUTextureUsage.COPY_DST | GPUTextureUsage.COPY_SRC | (isCompressedPixelFormat(texture.format) ? 0 : GPUTextureUsage.RENDER_ATTACHMENT) | (texture.storage ? GPUTextureUsage.STORAGE_BINDING : 0)
		};
		this.gpuTexture = wgpu.createTexture(this.descr);
		let viewDescr;
		if (this.texture.format === PIXELFORMAT_DEPTHSTENCIL) {
			viewDescr = {
				format: 'depth24plus',
				aspect: 'depth-only'
			};
		}
		this.view = this.createView(viewDescr);
	}
	destroy(device) {}
	propertyChanged(flag) {
		this.samplers.length = 0;
	}
	getView(device) {
		this.uploadImmediate(device, this.texture);
		return this.view;
	}
	createView(viewDescr) {
		var _options$format, _options$dimension, _options$aspect, _options$baseMipLevel, _options$mipLevelCoun, _options$baseArrayLay, _options$arrayLayerCo;
		const options = viewDescr != null ? viewDescr : {};
		const textureDescr = this.descr;
		const texture = this.texture;
		const defaultViewDimension = () => {
			if (texture.cubemap) return 'cube';
			if (texture.volume) return '3d';
			if (texture.array) return '2d-array';
			return '2d';
		};
		const descr = {
			format: (_options$format = options.format) != null ? _options$format : textureDescr.format,
			dimension: (_options$dimension = options.dimension) != null ? _options$dimension : defaultViewDimension(),
			aspect: (_options$aspect = options.aspect) != null ? _options$aspect : 'all',
			baseMipLevel: (_options$baseMipLevel = options.baseMipLevel) != null ? _options$baseMipLevel : 0,
			mipLevelCount: (_options$mipLevelCoun = options.mipLevelCount) != null ? _options$mipLevelCoun : textureDescr.mipLevelCount,
			baseArrayLayer: (_options$baseArrayLay = options.baseArrayLayer) != null ? _options$baseArrayLay : 0,
			arrayLayerCount: (_options$arrayLayerCo = options.arrayLayerCount) != null ? _options$arrayLayerCo : textureDescr.depthOrArrayLayers
		};
		const view = this.gpuTexture.createView(descr);
		return view;
	}
	getSampler(device, sampleType) {
		let sampler = this.samplers[sampleType];
		if (!sampler) {
			const texture = this.texture;
			const descr = {
				addressModeU: gpuAddressModes[texture.addressU],
				addressModeV: gpuAddressModes[texture.addressV],
				addressModeW: gpuAddressModes[texture.addressW]
			};
			if (!sampleType && texture.compareOnRead) {
				sampleType = SAMPLETYPE_DEPTH;
			}
			if (sampleType === SAMPLETYPE_DEPTH || sampleType === SAMPLETYPE_INT || sampleType === SAMPLETYPE_UINT) {
				descr.compare = 'less';
				descr.magFilter = 'linear';
				descr.minFilter = 'linear';
			} else if (sampleType === SAMPLETYPE_UNFILTERABLE_FLOAT) {
				descr.magFilter = 'nearest';
				descr.minFilter = 'nearest';
				descr.mipmapFilter = 'nearest';
			} else {
				if (this.texture.format === PIXELFORMAT_RGBA32F || this.texture.format === PIXELFORMAT_DEPTHSTENCIL || this.texture.format === PIXELFORMAT_RGBA16F || isIntegerPixelFormat(this.texture.format)) {
					descr.magFilter = 'nearest';
					descr.minFilter = 'nearest';
					descr.mipmapFilter = 'nearest';
				} else {
					descr.magFilter = gpuFilterModes[texture.magFilter].level;
					descr.minFilter = gpuFilterModes[texture.minFilter].level;
					descr.mipmapFilter = gpuFilterModes[texture.minFilter].mip;
				}
			}
			const allLinear = descr.minFilter === 'linear' && descr.magFilter === 'linear' && descr.mipmapFilter === 'linear';
			descr.maxAnisotropy = allLinear ? math.clamp(Math.round(texture._anisotropy), 1, device.maxTextureAnisotropy) : 1;
			sampler = device.wgpu.createSampler(descr);
			this.samplers[sampleType] = sampler;
		}
		return sampler;
	}
	loseContext() {}
	uploadImmediate(device, texture) {
		if (texture._needsUpload || texture._needsMipmapsUpload) {
			this.uploadData(device);
			texture._needsUpload = false;
			texture._needsMipmapsUpload = false;
		}
	}
	uploadData(device) {
		const texture = this.texture;
		if (texture._levels) {
			let anyUploads = false;
			let anyLevelMissing = false;
			const requiredMipLevels = texture.requiredMipLevels;
			for (let mipLevel = 0; mipLevel < requiredMipLevels; mipLevel++) {
				const mipObject = texture._levels[mipLevel];
				if (mipObject) {
					if (texture.cubemap) {
						for (let face = 0; face < 6; face++) {
							const faceSource = mipObject[face];
							if (faceSource) {
								if (this.isExternalImage(faceSource)) {
									this.uploadExternalImage(device, faceSource, mipLevel, face);
									anyUploads = true;
								} else if (ArrayBuffer.isView(faceSource)) {
									this.uploadTypedArrayData(device, faceSource, mipLevel, face);
									anyUploads = true;
								} else ;
							} else {
								anyLevelMissing = true;
							}
						}
					} else if (texture._volume) ; else if (texture.array) {
						if (texture.arrayLength === mipObject.length) {
							for (let index = 0; index < texture._arrayLength; index++) {
								const arraySource = mipObject[index];
								if (this.isExternalImage(arraySource)) {
									this.uploadExternalImage(device, arraySource, mipLevel, index);
									anyUploads = true;
								} else if (ArrayBuffer.isView(arraySource)) {
									this.uploadTypedArrayData(device, arraySource, mipLevel, index);
									anyUploads = true;
								} else ;
							}
						} else {
							anyLevelMissing = true;
						}
					} else {
						if (this.isExternalImage(mipObject)) {
							this.uploadExternalImage(device, mipObject, mipLevel, 0);
							anyUploads = true;
						} else if (ArrayBuffer.isView(mipObject)) {
							this.uploadTypedArrayData(device, mipObject, mipLevel, 0);
							anyUploads = true;
						} else ;
					}
				} else {
					anyLevelMissing = true;
				}
			}
			if (anyUploads && anyLevelMissing && texture.mipmaps && !isCompressedPixelFormat(texture.format)) {
				device.mipmapRenderer.generate(this);
			}
			if (texture._gpuSize) {
				texture.adjustVramSizeTracking(device._vram, -texture._gpuSize);
			}
			texture._gpuSize = texture.gpuSize;
			texture.adjustVramSizeTracking(device._vram, texture._gpuSize);
		}
	}
	isExternalImage(image) {
		return image instanceof ImageBitmap || image instanceof HTMLVideoElement || image instanceof HTMLCanvasElement || image instanceof OffscreenCanvas;
	}
	uploadExternalImage(device, image, mipLevel, index) {
		const src = {
			source: image,
			origin: [0, 0],
			flipY: false
		};
		const dst = {
			texture: this.gpuTexture,
			mipLevel: mipLevel,
			origin: [0, 0, index],
			aspect: 'all'
		};
		const copySize = {
			width: this.descr.size.width,
			height: this.descr.size.height,
			depthOrArrayLayers: 1
		};
		device.submit();
		dummyUse(image instanceof HTMLCanvasElement && image.getContext('2d'));
		device.wgpu.queue.copyExternalImageToTexture(src, dst, copySize);
	}
	uploadTypedArrayData(device, data, mipLevel, index) {
		const texture = this.texture;
		const wgpu = device.wgpu;
		const dest = {
			texture: this.gpuTexture,
			origin: [0, 0, index],
			mipLevel: mipLevel
		};
		const width = TextureUtils.calcLevelDimension(texture.width, mipLevel);
		const height = TextureUtils.calcLevelDimension(texture.height, mipLevel);
		TextureUtils.calcLevelGpuSize(width, height, 1, texture.format);
		const formatInfo = pixelFormatInfo.get(texture.format);
		let dataLayout;
		let size;
		if (formatInfo.size) {
			dataLayout = {
				offset: 0,
				bytesPerRow: formatInfo.size * width,
				rowsPerImage: height
			};
			size = {
				width: width,
				height: height
			};
		} else if (formatInfo.blockSize) {
			const blockDim = size => {
				return Math.floor((size + 3) / 4);
			};
			dataLayout = {
				offset: 0,
				bytesPerRow: formatInfo.blockSize * blockDim(width),
				rowsPerImage: blockDim(height)
			};
			size = {
				width: Math.max(4, width),
				height: Math.max(4, height)
			};
		} else ;
		device.submit();
		wgpu.queue.writeTexture(dest, data, dataLayout, size);
	}
}

class WebgpuUniformBuffer extends WebgpuBuffer {
	constructor(uniformBuffer) {
		super();
	}
	destroy(device) {
		super.destroy(device);
	}
	unlock(uniformBuffer) {
		const device = uniformBuffer.device;
		super.unlock(device, undefined, GPUBufferUsage.UNIFORM, uniformBuffer.storage);
	}
}

class WebgpuVertexBuffer extends WebgpuBuffer {
	constructor(vertexBuffer, format) {
		super();
	}
	destroy(device) {
		super.destroy(device);
	}
	unlock(vertexBuffer) {
		const device = vertexBuffer.device;
		super.unlock(device, vertexBuffer.usage, GPUBufferUsage.VERTEX, vertexBuffer.storage);
	}
}

const KEYWORD = /[ \t]*#(ifn?def|if|endif|else|elif|define|undef|extension)/g;
const DEFINE = /define[ \t]+([^\n]+)\r?(?:\n|$)/g;
const EXTENSION = /extension[ \t]+([\w-]+)[ \t]*:[ \t]*(enable|require)/g;
const UNDEF = /undef[ \t]+([^\n]+)\r?(?:\n|$)/g;
const IF = /(ifdef|ifndef|if)[ \t]*([^\r\n]+)\r?\n/g;
const ENDIF = /(endif|else|elif)([ \t]+[^\r\n]+)?\r?(?:\n|$)/g;
const IDENTIFIER$1 = /([\w-]+)/;
const DEFINED = /(!|\s)?defined\(([\w-]+)\)/;
const INVALID = /[><=|&+-]/g;
class Preprocessor {
	static run(source, stripUnusedColorAttachments = false) {
		source = source.replace(/\/\*[\s\S]*?\*\/|([^\\:]|^)\/\/.*$/gm, '$1');
		source = source.split(/\r?\n/).map(line => line.trimEnd()).join('\n');
		const defines = new Map();
		if (stripUnusedColorAttachments) {
			const counts = new Map();
			const regex = /(pcFragColor[1-8])\b/g;
			const matches = source.match(regex);
			matches == null || matches.forEach(match => {
				var _counts$get;
				const index = parseInt(match.charAt(match.length - 1), 10);
				counts.set(index, ((_counts$get = counts.get(index)) != null ? _counts$get : 0) + 1);
			});
			counts.forEach((count, index) => {
				if (count === 1) {
					defines.set(`REMOVE_COLOR_ATTACHMENT_${index}`, '');
				}
			});
		}
		source = this._preprocess(source, defines);
		const intDefines = new Map();
		defines.forEach((value, key) => {
			if (Number.isInteger(parseFloat(value)) && !value.includes('.')) {
				intDefines.set(key, value);
			}
		});
		source = this.RemoveEmptyLines(source);
		source = this.processArraySize(source, intDefines);
		return source;
	}
	static processArraySize(source, intDefines) {
		if (source !== null) {
			intDefines.forEach((value, key) => {
				source = source.replace(new RegExp(`\\[${key}\\]`, 'g'), `[${value}]`);
			});
		}
		return source;
	}
	static RemoveEmptyLines(source) {
		if (source !== null) {
			source = source.split(/\r?\n/).map(line => line.trim() === '' ? '' : line).join('\n');
			source = source.replace(/(\n\n){3,}/gm, '\n\n');
		}
		return source;
	}
	static _preprocess(source, defines = new Map()) {
		const originalSource = source;
		const stack = [];
		let error = false;
		let match;
		while ((match = KEYWORD.exec(source)) !== null) {
			const keyword = match[1];
			switch (keyword) {
				case 'define':
					{
						DEFINE.lastIndex = match.index;
						const define = DEFINE.exec(source);
						error || (error = define === null);
						const expression = define[1];
						IDENTIFIER$1.lastIndex = define.index;
						const identifierValue = IDENTIFIER$1.exec(expression);
						const identifier = identifierValue[1];
						let value = expression.substring(identifier.length).trim();
						if (value === "") value = "true";
						const keep = Preprocessor._keep(stack);
						if (keep) {
							defines.set(identifier, value);
						}
						KEYWORD.lastIndex = define.index + define[0].length;
						break;
					}
				case 'undef':
					{
						UNDEF.lastIndex = match.index;
						const undef = UNDEF.exec(source);
						const identifier = undef[1].trim();
						const keep = Preprocessor._keep(stack);
						if (keep) {
							defines.delete(identifier);
						}
						KEYWORD.lastIndex = undef.index + undef[0].length;
						break;
					}
				case 'extension':
					{
						EXTENSION.lastIndex = match.index;
						const extension = EXTENSION.exec(source);
						error || (error = extension === null);
						if (extension) {
							const identifier = extension[1];
							const keep = Preprocessor._keep(stack);
							if (keep) {
								defines.set(identifier, "true");
							}
						}
						KEYWORD.lastIndex = extension.index + extension[0].length;
						break;
					}
				case 'ifdef':
				case 'ifndef':
				case 'if':
					{
						IF.lastIndex = match.index;
						const iff = IF.exec(source);
						const expression = iff[2];
						const evaluated = Preprocessor.evaluate(expression, defines);
						error || (error = evaluated.error);
						let result = evaluated.result;
						if (keyword === 'ifndef') {
							result = !result;
						}
						stack.push({
							anyKeep: result,
							keep: result,
							start: match.index,
							end: IF.lastIndex
						});
						KEYWORD.lastIndex = iff.index + iff[0].length;
						break;
					}
				case 'endif':
				case 'else':
				case 'elif':
					{
						ENDIF.lastIndex = match.index;
						const endif = ENDIF.exec(source);
						const blockInfo = stack.pop();
						const blockCode = blockInfo.keep ? source.substring(blockInfo.end, match.index) : "";
						source = source.substring(0, blockInfo.start) + blockCode + source.substring(ENDIF.lastIndex);
						KEYWORD.lastIndex = blockInfo.start + blockCode.length;
						const endifCommand = endif[1];
						if (endifCommand === 'else' || endifCommand === 'elif') {
							let result = false;
							if (!blockInfo.anyKeep) {
								if (endifCommand === 'else') {
									result = !blockInfo.keep;
								} else {
									const evaluated = Preprocessor.evaluate(endif[2], defines);
									result = evaluated.result;
									error || (error = evaluated.error);
								}
							}
							stack.push({
								anyKeep: blockInfo.anyKeep || result,
								keep: result,
								start: KEYWORD.lastIndex,
								end: KEYWORD.lastIndex
							});
						}
						break;
					}
			}
		}
		if (error) {
			console.warn("Failed to preprocess shader: ", {
				source: originalSource
			});
			return originalSource;
		}
		return source;
	}
	static _keep(stack) {
		for (let i = 0; i < stack.length; i++) {
			if (!stack[i].keep) return false;
		}
		return true;
	}
	static evaluate(expression, defines) {
		const correct = INVALID.exec(expression) === null;
		let invert = false;
		const defined = DEFINED.exec(expression);
		if (defined) {
			invert = defined[1] === '!';
			expression = defined[2];
		}
		expression = expression.trim();
		let exists = defines.has(expression);
		if (invert) {
			exists = !exists;
		}
		return {
			result: exists,
			error: !correct
		};
	}
}

let id$7 = 0;
class Shader {
	constructor(graphicsDevice, definition) {
		this.meshUniformBufferFormat = void 0;
		this.meshBindGroupFormat = void 0;
		this.id = id$7++;
		this.device = graphicsDevice;
		this.definition = definition;
		this.name = definition.name || 'Untitled';
		this.init();
		if (definition.cshader) ; else {
			definition.vshader = Preprocessor.run(definition.vshader);
			const stripUnusedColorAttachments = graphicsDevice.isWebGL2 && (platform.name === 'osx' || platform.name === 'ios');
			definition.fshader = Preprocessor.run(definition.fshader, stripUnusedColorAttachments);
		}
		this.impl = graphicsDevice.createShaderImpl(this);
	}
	init() {
		this.ready = false;
		this.failed = false;
	}
	get label() {
		return `Shader Id ${this.id} ${this.name}`;
	}
	destroy() {
		this.device.onDestroyShader(this);
		this.impl.destroy(this);
	}
	loseContext() {
		this.init();
		this.impl.loseContext();
	}
	restoreContext() {
		this.impl.restoreContext(this.device, this);
	}
}

let id$6 = 0;
class BindGroup {
	constructor(graphicsDevice, format, defaultUniformBuffer) {
		this.renderVersionUpdated = -1;
		this.uniformBuffers = void 0;
		this.uniformBufferOffsets = [];
		this.id = id$6++;
		this.device = graphicsDevice;
		this.format = format;
		this.dirty = true;
		this.impl = graphicsDevice.createBindGroupImpl(this);
		this.textures = [];
		this.storageTextures = [];
		this.uniformBuffers = [];
		this.defaultUniformBuffer = defaultUniformBuffer;
		if (defaultUniformBuffer) {
			this.setUniformBuffer(UNIFORM_BUFFER_DEFAULT_SLOT_NAME, defaultUniformBuffer);
		}
	}
	destroy() {
		this.impl.destroy();
		this.impl = null;
		this.format = null;
		this.defaultUniformBuffer = null;
	}
	setUniformBuffer(name, uniformBuffer) {
		const index = this.format.bufferFormatsMap.get(name);
		if (this.uniformBuffers[index] !== uniformBuffer) {
			this.uniformBuffers[index] = uniformBuffer;
			this.dirty = true;
		}
	}
	setTexture(name, texture) {
		const index = this.format.textureFormatsMap.get(name);
		if (this.textures[index] !== texture) {
			this.textures[index] = texture;
			this.dirty = true;
		} else if (this.renderVersionUpdated < texture.renderVersionDirty) {
			this.dirty = true;
		}
	}
	setStorageTexture(name, texture) {
		const index = this.format.storageTextureFormatsMap.get(name);
		if (this.storageTextures[index] !== texture) {
			this.storageTextures[index] = texture;
			this.dirty = true;
		} else if (this.renderVersionUpdated < texture.renderVersionDirty) {
			this.dirty = true;
		}
	}
	update() {
		const {
			textureFormats,
			storageTextureFormats
		} = this.format;
		for (let i = 0; i < textureFormats.length; i++) {
			const textureFormat = textureFormats[i];
			const value = textureFormat.scopeId.value;
			this.setTexture(textureFormat.name, value);
		}
		for (let i = 0; i < storageTextureFormats.length; i++) {
			const storageTextureFormat = storageTextureFormats[i];
			const value = storageTextureFormat.scopeId.value;
			this.setStorageTexture(storageTextureFormat.name, value);
		}
		this.uniformBufferOffsets.length = this.uniformBuffers.length;
		for (let i = 0; i < this.uniformBuffers.length; i++) {
			const uniformBuffer = this.uniformBuffers[i];
			this.uniformBufferOffsets[i] = uniformBuffer.offset;
			if (this.renderVersionUpdated < uniformBuffer.renderVersionDirty) {
				this.dirty = true;
			}
		}
		if (this.dirty) {
			this.dirty = false;
			this.renderVersionUpdated = this.device.renderVersion;
			this.impl.update(this);
		}
	}
}

class DynamicBuffer {
	constructor(device) {
		this.device = void 0;
		this.device = device;
	}
}
class UsedBuffer {
	constructor() {
		this.gpuBuffer = void 0;
		this.stagingBuffer = void 0;
		this.offset = void 0;
		this.size = void 0;
	}
}
class DynamicBufferAllocation {
	constructor() {
		this.storage = void 0;
		this.gpuBuffer = void 0;
		this.offset = void 0;
	}
}
class DynamicBuffers {
	constructor(device, bufferSize, bufferAlignment) {
		this.bufferSize = void 0;
		this.gpuBuffers = [];
		this.stagingBuffers = [];
		this.usedBuffers = [];
		this.activeBuffer = null;
		this.device = device;
		this.bufferSize = bufferSize;
		this.bufferAlignment = bufferAlignment;
	}
	destroy() {
		this.gpuBuffers.forEach(gpuBuffer => {
			gpuBuffer.destroy(this.device);
		});
		this.gpuBuffers = null;
		this.stagingBuffers.forEach(stagingBuffer => {
			stagingBuffer.destroy(this.device);
		});
		this.stagingBuffers = null;
		this.usedBuffers = null;
		this.activeBuffer = null;
	}
	alloc(allocation, size) {
		if (this.activeBuffer) {
			const _alignedStart = math.roundUp(this.activeBuffer.size, this.bufferAlignment);
			const space = this.bufferSize - _alignedStart;
			if (space < size) {
				this.scheduleSubmit();
			}
		}
		if (!this.activeBuffer) {
			let gpuBuffer = this.gpuBuffers.pop();
			if (!gpuBuffer) {
				gpuBuffer = this.createBuffer(this.device, this.bufferSize, false);
			}
			let stagingBuffer = this.stagingBuffers.pop();
			if (!stagingBuffer) {
				stagingBuffer = this.createBuffer(this.device, this.bufferSize, true);
			}
			this.activeBuffer = new UsedBuffer();
			this.activeBuffer.stagingBuffer = stagingBuffer;
			this.activeBuffer.gpuBuffer = gpuBuffer;
			this.activeBuffer.offset = 0;
			this.activeBuffer.size = 0;
		}
		const activeBuffer = this.activeBuffer;
		const alignedStart = math.roundUp(activeBuffer.size, this.bufferAlignment);
		allocation.gpuBuffer = activeBuffer.gpuBuffer;
		allocation.offset = alignedStart;
		allocation.storage = activeBuffer.stagingBuffer.alloc(alignedStart, size);
		activeBuffer.size = alignedStart + size;
	}
	scheduleSubmit() {
		if (this.activeBuffer) {
			this.usedBuffers.push(this.activeBuffer);
			this.activeBuffer = null;
		}
	}
	submit() {
		this.scheduleSubmit();
	}
}

const _updateFunctions = [];
_updateFunctions[UNIFORMTYPE_FLOAT] = function (uniformBuffer, value, offset) {
	const dst = uniformBuffer.storageFloat32;
	dst[offset] = value;
};
_updateFunctions[UNIFORMTYPE_VEC2] = (uniformBuffer, value, offset) => {
	const dst = uniformBuffer.storageFloat32;
	dst[offset] = value[0];
	dst[offset + 1] = value[1];
};
_updateFunctions[UNIFORMTYPE_VEC3] = (uniformBuffer, value, offset) => {
	const dst = uniformBuffer.storageFloat32;
	dst[offset] = value[0];
	dst[offset + 1] = value[1];
	dst[offset + 2] = value[2];
};
_updateFunctions[UNIFORMTYPE_VEC4] = (uniformBuffer, value, offset) => {
	const dst = uniformBuffer.storageFloat32;
	dst[offset] = value[0];
	dst[offset + 1] = value[1];
	dst[offset + 2] = value[2];
	dst[offset + 3] = value[3];
};
_updateFunctions[UNIFORMTYPE_INT] = function (uniformBuffer, value, offset) {
	const dst = uniformBuffer.storageInt32;
	dst[offset] = value;
};
_updateFunctions[UNIFORMTYPE_IVEC2] = function (uniformBuffer, value, offset) {
	const dst = uniformBuffer.storageInt32;
	dst[offset] = value[0];
	dst[offset + 1] = value[1];
};
_updateFunctions[UNIFORMTYPE_IVEC3] = function (uniformBuffer, value, offset) {
	const dst = uniformBuffer.storageInt32;
	dst[offset] = value[0];
	dst[offset + 1] = value[1];
	dst[offset + 2] = value[2];
};
_updateFunctions[UNIFORMTYPE_IVEC4] = function (uniformBuffer, value, offset) {
	const dst = uniformBuffer.storageInt32;
	dst[offset] = value[0];
	dst[offset + 1] = value[1];
	dst[offset + 2] = value[2];
	dst[offset + 3] = value[3];
};
_updateFunctions[UNIFORMTYPE_MAT2] = (uniformBuffer, value, offset) => {
	const dst = uniformBuffer.storageFloat32;
	dst[offset] = value[0];
	dst[offset + 1] = value[1];
	dst[offset + 4] = value[2];
	dst[offset + 5] = value[3];
	dst[offset + 8] = value[4];
	dst[offset + 9] = value[5];
};
_updateFunctions[UNIFORMTYPE_MAT3] = (uniformBuffer, value, offset) => {
	const dst = uniformBuffer.storageFloat32;
	dst[offset] = value[0];
	dst[offset + 1] = value[1];
	dst[offset + 2] = value[2];
	dst[offset + 4] = value[3];
	dst[offset + 5] = value[4];
	dst[offset + 6] = value[5];
	dst[offset + 8] = value[6];
	dst[offset + 9] = value[7];
	dst[offset + 10] = value[8];
};
_updateFunctions[UNIFORMTYPE_FLOATARRAY] = function (uniformBuffer, value, offset, count) {
	const dst = uniformBuffer.storageFloat32;
	for (let i = 0; i < count; i++) {
		dst[offset + i * 4] = value[i];
	}
};
_updateFunctions[UNIFORMTYPE_VEC2ARRAY] = (uniformBuffer, value, offset, count) => {
	const dst = uniformBuffer.storageFloat32;
	for (let i = 0; i < count; i++) {
		dst[offset + i * 4] = value[i * 2];
		dst[offset + i * 4 + 1] = value[i * 2 + 1];
	}
};
_updateFunctions[UNIFORMTYPE_VEC3ARRAY] = (uniformBuffer, value, offset, count) => {
	const dst = uniformBuffer.storageFloat32;
	for (let i = 0; i < count; i++) {
		dst[offset + i * 4] = value[i * 3];
		dst[offset + i * 4 + 1] = value[i * 3 + 1];
		dst[offset + i * 4 + 2] = value[i * 3 + 2];
	}
};
_updateFunctions[UNIFORMTYPE_UINT] = (uniformBuffer, value, offset, count) => {
	const dst = uniformBuffer.storageUint32;
	dst[offset] = value;
};
_updateFunctions[UNIFORMTYPE_UVEC2] = (uniformBuffer, value, offset, count) => {
	const dst = uniformBuffer.storageUint32;
	dst[offset] = value[0];
	dst[offset + 1] = value[1];
};
_updateFunctions[UNIFORMTYPE_UVEC3] = (uniformBuffer, value, offset, count) => {
	const dst = uniformBuffer.storageUint32;
	dst[offset] = value[0];
	dst[offset + 1] = value[1];
	dst[offset + 2] = value[2];
};
_updateFunctions[UNIFORMTYPE_UVEC4] = (uniformBuffer, value, offset, count) => {
	const dst = uniformBuffer.storageUint32;
	dst[offset] = value[0];
	dst[offset + 1] = value[1];
	dst[offset + 2] = value[2];
	dst[offset + 3] = value[3];
};
_updateFunctions[UNIFORMTYPE_INTARRAY] = function (uniformBuffer, value, offset, count) {
	const dst = uniformBuffer.storageInt32;
	for (let i = 0; i < count; i++) {
		dst[offset + i * 4] = value[i];
	}
};
_updateFunctions[UNIFORMTYPE_BOOLARRAY] = _updateFunctions[UNIFORMTYPE_INTARRAY];
_updateFunctions[UNIFORMTYPE_UINTARRAY] = function (uniformBuffer, value, offset, count) {
	const dst = uniformBuffer.storageUint32;
	for (let i = 0; i < count; i++) {
		dst[offset + i * 4] = value[i];
	}
};
_updateFunctions[UNIFORMTYPE_IVEC2ARRAY] = (uniformBuffer, value, offset, count) => {
	const dst = uniformBuffer.storageInt32;
	for (let i = 0; i < count; i++) {
		dst[offset + i * 4] = value[i * 2];
		dst[offset + i * 4 + 1] = value[i * 2 + 1];
	}
};
_updateFunctions[UNIFORMTYPE_BVEC2ARRAY] = _updateFunctions[UNIFORMTYPE_IVEC2ARRAY];
_updateFunctions[UNIFORMTYPE_UVEC2ARRAY] = (uniformBuffer, value, offset, count) => {
	const dst = uniformBuffer.storageUint32;
	for (let i = 0; i < count; i++) {
		dst[offset + i * 4] = value[i * 2];
		dst[offset + i * 4 + 1] = value[i * 2 + 1];
	}
};
_updateFunctions[UNIFORMTYPE_IVEC3ARRAY] = (uniformBuffer, value, offset, count) => {
	const dst = uniformBuffer.storageInt32;
	for (let i = 0; i < count; i++) {
		dst[offset + i * 4] = value[i * 3];
		dst[offset + i * 4 + 1] = value[i * 3 + 1];
		dst[offset + i * 4 + 2] = value[i * 3 + 2];
	}
};
_updateFunctions[UNIFORMTYPE_BVEC3ARRAY] = _updateFunctions[UNIFORMTYPE_IVEC3ARRAY];
_updateFunctions[UNIFORMTYPE_UVEC3ARRAY] = (uniformBuffer, value, offset, count) => {
	const dst = uniformBuffer.storageUint32;
	for (let i = 0; i < count; i++) {
		dst[offset + i * 4] = value[i * 3];
		dst[offset + i * 4 + 1] = value[i * 3 + 1];
		dst[offset + i * 4 + 2] = value[i * 3 + 2];
	}
};
class UniformBuffer {
	constructor(graphicsDevice, format, persistent = true) {
		this.device = void 0;
		this.persistent = void 0;
		this.allocation = void 0;
		this.storageFloat32 = void 0;
		this.storageInt32 = void 0;
		this.storageUint32 = void 0;
		this.renderVersionDirty = 0;
		this.device = graphicsDevice;
		this.format = format;
		this.persistent = persistent;
		if (persistent) {
			this.impl = graphicsDevice.createUniformBufferImpl(this);
			const storage = new ArrayBuffer(format.byteSize);
			this.assignStorage(new Int32Array(storage));
			graphicsDevice._vram.ub += this.format.byteSize;
		} else {
			this.allocation = new DynamicBufferAllocation();
		}
	}
	destroy() {
		if (this.persistent) {
			const device = this.device;
			this.impl.destroy(device);
			device._vram.ub -= this.format.byteSize;
		}
	}
	get offset() {
		return this.persistent ? 0 : this.allocation.offset;
	}
	assignStorage(storage) {
		this.storageInt32 = storage;
		this.storageUint32 = new Uint32Array(storage.buffer, storage.byteOffset, storage.byteLength / 4);
		this.storageFloat32 = new Float32Array(storage.buffer, storage.byteOffset, storage.byteLength / 4);
	}
	loseContext() {
		var _this$impl;
		(_this$impl = this.impl) == null || _this$impl.loseContext();
	}
	setUniform(uniformFormat) {
		const offset = uniformFormat.offset;
		const value = uniformFormat.scopeId.value;
		if (value !== null && value !== undefined) {
			const updateFunction = _updateFunctions[uniformFormat.updateType];
			if (updateFunction) {
				updateFunction(this, value, offset, uniformFormat.count);
			} else {
				this.storageFloat32.set(value, offset);
			}
		}
	}
	set(name) {
		const uniformFormat = this.format.map.get(name);
		if (uniformFormat) {
			this.setUniform(uniformFormat);
		}
	}
	update() {
		const persistent = this.persistent;
		if (!persistent) {
			const allocation = this.allocation;
			const oldGpuBuffer = allocation.gpuBuffer;
			this.device.dynamicBuffers.alloc(allocation, this.format.byteSize);
			this.assignStorage(allocation.storage);
			if (oldGpuBuffer !== allocation.gpuBuffer) {
				this.renderVersionDirty = this.device.renderVersion;
			}
		}
		const uniforms = this.format.uniforms;
		for (let i = 0; i < uniforms.length; i++) {
			this.setUniform(uniforms[i]);
		}
		if (persistent) {
			this.impl.unlock(this);
		} else {
			this.storageFloat32 = null;
			this.storageInt32 = null;
		}
	}
}

const primitive = {
	type: PRIMITIVE_TRISTRIP,
	base: 0,
	count: 4,
	indexed: false
};
class WebgpuClearRenderer {
	constructor(device) {
		const code = `

						struct ub_mesh {
								color : vec4f,
								depth: f32
						}

						@group(0) @binding(0) var<uniform> ubMesh : ub_mesh;

						var<private> pos : array<vec2f, 4> = array<vec2f, 4>(
								vec2(-1.0, 1.0), vec2(1.0, 1.0),
								vec2(-1.0, -1.0), vec2(1.0, -1.0)
						);

						struct VertexOutput {
								@builtin(position) position : vec4f
						}

						@vertex
						fn vertexMain(@builtin(vertex_index) vertexIndex : u32) -> VertexOutput {
								var output : VertexOutput;
								output.position = vec4(pos[vertexIndex], ubMesh.depth, 1.0);
								return output;
						}

						@fragment
						fn fragmentMain() -> @location(0) vec4f {
								return ubMesh.color;
						}
				`;
		this.shader = new Shader(device, {
			name: 'WebGPUClearRendererShader',
			shaderLanguage: SHADERLANGUAGE_WGSL,
			vshader: code,
			fshader: code
		});
		this.uniformBuffer = new UniformBuffer(device, new UniformBufferFormat(device, [new UniformFormat('color', UNIFORMTYPE_VEC4), new UniformFormat('depth', UNIFORMTYPE_FLOAT)]), false);
		const bindGroupFormat = new BindGroupFormat(device, [new BindBufferFormat(UNIFORM_BUFFER_DEFAULT_SLOT_NAME, SHADERSTAGE_VERTEX | SHADERSTAGE_FRAGMENT)]);
		this.bindGroup = new BindGroup(device, bindGroupFormat, this.uniformBuffer);
		this.colorData = new Float32Array(4);
		this.colorId = device.scope.resolve('color');
		this.depthId = device.scope.resolve('depth');
	}
	destroy() {
		this.shader.destroy();
		this.shader = null;
		this.uniformBuffer.destroy();
		this.uniformBuffer = null;
		this.bindGroup.destroy();
		this.bindGroup = null;
	}
	clear(device, renderTarget, options, defaultOptions) {
		var _options$flags;
		options = options || defaultOptions;
		const flags = (_options$flags = options.flags) != null ? _options$flags : defaultOptions.flags;
		if (flags !== 0) {
			if (flags & CLEARFLAG_COLOR && renderTarget.colorBuffer) {
				var _options$color;
				const color = (_options$color = options.color) != null ? _options$color : defaultOptions.color;
				this.colorData.set(color);
				device.setBlendState(BlendState.NOBLEND);
			} else {
				device.setBlendState(BlendState.NOWRITE);
			}
			this.colorId.setValue(this.colorData);
			if (flags & CLEARFLAG_DEPTH && renderTarget.depth) {
				var _options$depth;
				const depth = (_options$depth = options.depth) != null ? _options$depth : defaultOptions.depth;
				this.depthId.setValue(depth);
				device.setDepthState(DepthState.WRITEDEPTH);
			} else {
				this.depthId.setValue(1);
				device.setDepthState(DepthState.NODEPTH);
			}
			if (flags & CLEARFLAG_STENCIL && renderTarget.stencil) ;
			device.setCullMode(CULLFACE_NONE);
			device.setShader(this.shader);
			const bindGroup = this.bindGroup;
			bindGroup.defaultUniformBuffer.update();
			bindGroup.update();
			device.setBindGroup(BINDGROUP_MESH, bindGroup);
			device.draw(primitive);
		}
	}
}

class WebgpuMipmapRenderer {
	constructor(device) {
		this.device = void 0;
		this.device = device;
		const code = `
 
						var<private> pos : array<vec2f, 4> = array<vec2f, 4>(
								vec2(-1.0, 1.0), vec2(1.0, 1.0),
								vec2(-1.0, -1.0), vec2(1.0, -1.0)
						);

						struct VertexOutput {
								@builtin(position) position : vec4f,
								@location(0) texCoord : vec2f
						};

						@vertex
						fn vertexMain(@builtin(vertex_index) vertexIndex : u32) -> VertexOutput {
							var output : VertexOutput;
							output.texCoord = pos[vertexIndex] * vec2f(0.5, -0.5) + vec2f(0.5);
							output.position = vec4f(pos[vertexIndex], 0, 1);
							return output;
						}

						@group(0) @binding(0) var imgSampler : sampler;
						@group(0) @binding(1) var img : texture_2d<f32>;

						@fragment
						fn fragmentMain(@location(0) texCoord : vec2f) -> @location(0) vec4f {
							return textureSample(img, imgSampler, texCoord);
						}
				`;
		this.shader = new Shader(device, {
			name: 'WebGPUMipmapRendererShader',
			shaderLanguage: SHADERLANGUAGE_WGSL,
			vshader: code,
			fshader: code
		});
		this.minSampler = device.wgpu.createSampler({
			minFilter: 'linear'
		});
	}
	destroy() {
		this.shader.destroy();
		this.shader = null;
	}
	generate(webgpuTexture) {
		var _device$commandEncode;
		const textureDescr = webgpuTexture.descr;
		if (textureDescr.mipLevelCount <= 1) {
			return;
		}
		if (webgpuTexture.texture.volume) {
			return;
		}
		const device = this.device;
		const wgpu = device.wgpu;
		const webgpuShader = this.shader.impl;
		const pipeline = wgpu.createRenderPipeline({
			layout: 'auto',
			vertex: {
				module: webgpuShader.getVertexShaderModule(),
				entryPoint: webgpuShader.vertexEntryPoint
			},
			fragment: {
				module: webgpuShader.getFragmentShaderModule(),
				entryPoint: webgpuShader.fragmentEntryPoint,
				targets: [{
					format: textureDescr.format
				}]
			},
			primitive: {
				topology: 'triangle-strip'
			}
		});
		const texture = webgpuTexture.texture;
		const numFaces = texture.cubemap ? 6 : texture.array ? texture.arrayLength : 1;
		const srcViews = [];
		for (let face = 0; face < numFaces; face++) {
			srcViews.push(webgpuTexture.createView({
				dimension: '2d',
				baseMipLevel: 0,
				mipLevelCount: 1,
				baseArrayLayer: face
			}));
		}
		const commandEncoder = (_device$commandEncode = device.commandEncoder) != null ? _device$commandEncode : wgpu.createCommandEncoder();
		for (let i = 1; i < textureDescr.mipLevelCount; i++) {
			for (let face = 0; face < numFaces; face++) {
				const dstView = webgpuTexture.createView({
					dimension: '2d',
					baseMipLevel: i,
					mipLevelCount: 1,
					baseArrayLayer: face
				});
				const passEncoder = commandEncoder.beginRenderPass({
					colorAttachments: [{
						view: dstView,
						loadOp: 'clear',
						storeOp: 'store'
					}]
				});
				const bindGroup = wgpu.createBindGroup({
					layout: pipeline.getBindGroupLayout(0),
					entries: [{
						binding: 0,
						resource: this.minSampler
					}, {
						binding: 1,
						resource: srcViews[face]
					}]
				});
				passEncoder.setPipeline(pipeline);
				passEncoder.setBindGroup(0, bindGroup);
				passEncoder.draw(4);
				passEncoder.end();
				srcViews[face] = dstView;
			}
		}
		if (!device.commandEncoder) {
			const cb = commandEncoder.finish();
			device.addCommandBuffer(cb);
		}
		device.pipeline = null;
	}
}

class WebgpuDynamicBuffer extends DynamicBuffer {
	constructor(device, size, isStaging) {
		super(device);
		this.buffer = null;
		this.mappedRange = null;
		this.buffer = device.wgpu.createBuffer({
			size: size,
			usage: isStaging ? GPUBufferUsage.MAP_WRITE | GPUBufferUsage.COPY_SRC : GPUBufferUsage.UNIFORM | GPUBufferUsage.COPY_DST,
			mappedAtCreation: isStaging
		});
		if (isStaging) {
			this.onAvailable();
		}
		device._vram.ub += size;
	}
	destroy(device) {
		device._vram.ub -= this.buffer.size;
		this.buffer.destroy();
		this.buffer = null;
	}
	onAvailable() {
		this.mappedRange = this.buffer.getMappedRange();
	}
	alloc(offset, size) {
		return new Int32Array(this.mappedRange, offset, size / 4);
	}
}

class WebgpuDynamicBuffers extends DynamicBuffers {
	constructor(...args) {
		super(...args);
		this.pendingStagingBuffers = [];
	}
	createBuffer(device, size, isStaging) {
		return new WebgpuDynamicBuffer(device, size, isStaging);
	}
	submit() {
		super.submit();
		const count = this.usedBuffers.length;
		if (count) {
			const device = this.device;
			const gpuBuffers = this.gpuBuffers;
			const commandEncoder = device.wgpu.createCommandEncoder();
			for (let i = count - 1; i >= 0; i--) {
				const usedBuffer = this.usedBuffers[i];
				const {
					stagingBuffer,
					gpuBuffer,
					offset,
					size
				} = usedBuffer;
				const src = stagingBuffer.buffer;
				src.unmap();
				commandEncoder.copyBufferToBuffer(src, offset, gpuBuffer.buffer, offset, size);
				gpuBuffers.push(gpuBuffer);
			}
			const cb = commandEncoder.finish();
			device.addCommandBuffer(cb, true);
			for (let i = 0; i < count; i++) {
				const stagingBuffer = this.usedBuffers[i].stagingBuffer;
				this.pendingStagingBuffers.push(stagingBuffer);
			}
			this.usedBuffers.length = 0;
		}
	}
	onCommandBuffersSubmitted() {
		const count = this.pendingStagingBuffers.length;
		if (count) {
			for (let i = 0; i < count; i++) {
				const stagingBuffer = this.pendingStagingBuffers[i];
				stagingBuffer.buffer.mapAsync(GPUMapMode.WRITE).then(() => {
					if (this.stagingBuffers) {
						stagingBuffer.onAvailable();
						this.stagingBuffers.push(stagingBuffer);
					}
				});
			}
			this.pendingStagingBuffers.length = 0;
		}
	}
}

class GpuProfiler {
	constructor() {
		this.frameAllocations = [];
		this.pastFrameAllocations = new Map();
		this._enabled = false;
		this._enableRequest = false;
		this._frameTime = 0;
	}
	loseContext() {
		this.pastFrameAllocations.clear();
	}
	set enabled(value) {
		this._enableRequest = value;
	}
	get enabled() {
		return this._enableRequest;
	}
	processEnableRequest() {
		if (this._enableRequest !== this._enabled) {
			this._enabled = this._enableRequest;
			if (!this._enabled) {
				this._frameTime = 0;
			}
		}
	}
	request(renderVersion) {
		this.pastFrameAllocations.set(renderVersion, this.frameAllocations);
		this.frameAllocations = [];
	}
	report(renderVersion, timings) {
		if (timings) {
			const allocations = this.pastFrameAllocations.get(renderVersion);
			if (timings.length > 0) {
				this._frameTime = timings[0];
			}
			if (Tracing.get(TRACEID_GPU_TIMINGS)) {
				for (let i = 0; i < allocations.length; ++i) {
					allocations[i];
				}
			}
		}
		this.pastFrameAllocations.delete(renderVersion);
	}
	getSlot(name) {
		const slot = this.frameAllocations.length;
		this.frameAllocations.push(name);
		return slot;
	}
	get slotCount() {
		return this.frameAllocations.length;
	}
}

class WebgpuQuerySet {
	constructor(device, isTimestamp, capacity) {
		this.querySet = void 0;
		this.stagingBuffers = [];
		this.activeStagingBuffer = null;
		this.bytesPerSlot = void 0;
		this.device = device;
		this.capacity = capacity;
		this.bytesPerSlot = isTimestamp ? 8 : 4;
		const wgpu = device.wgpu;
		this.querySet = wgpu.createQuerySet({
			type: isTimestamp ? 'timestamp' : 'occlusion',
			count: capacity
		});
		this.queryBuffer = wgpu.createBuffer({
			size: this.bytesPerSlot * capacity,
			usage: GPUBufferUsage.QUERY_RESOLVE | GPUBufferUsage.STORAGE | GPUBufferUsage.COPY_SRC | GPUBufferUsage.COPY_DST
		});
	}
	destroy() {
		var _this$querySet, _this$queryBuffer;
		(_this$querySet = this.querySet) == null || _this$querySet.destroy();
		this.querySet = null;
		(_this$queryBuffer = this.queryBuffer) == null || _this$queryBuffer.destroy();
		this.queryBuffer = null;
		this.activeStagingBuffer = null;
		this.stagingBuffers.forEach(stagingBuffer => {
			stagingBuffer.destroy();
		});
		this.stagingBuffers = null;
	}
	getStagingBuffer() {
		let stagingBuffer = this.stagingBuffers.pop();
		if (!stagingBuffer) {
			stagingBuffer = this.device.wgpu.createBuffer({
				size: this.queryBuffer.size,
				usage: GPUBufferUsage.COPY_DST | GPUBufferUsage.MAP_READ
			});
		}
		return stagingBuffer;
	}
	resolve(count) {
		const device = this.device;
		const commandEncoder = device.wgpu.createCommandEncoder();
		commandEncoder.resolveQuerySet(this.querySet, 0, count, this.queryBuffer, 0);
		const activeStagingBuffer = this.getStagingBuffer();
		this.activeStagingBuffer = activeStagingBuffer;
		commandEncoder.copyBufferToBuffer(this.queryBuffer, 0, activeStagingBuffer, 0, this.bytesPerSlot * count);
		const cb = commandEncoder.finish();
		device.addCommandBuffer(cb);
	}
	request(count, renderVersion) {
		const stagingBuffer = this.activeStagingBuffer;
		this.activeStagingBuffer = null;
		return stagingBuffer.mapAsync(GPUMapMode.READ).then(() => {
			var _this$stagingBuffers;
			const srcTimings = new BigInt64Array(stagingBuffer.getMappedRange());
			const timings = [];
			for (let i = 0; i < count; i++) {
				timings.push(Number(srcTimings[i * 2 + 1] - srcTimings[i * 2]) * 0.000001);
			}
			stagingBuffer.unmap();
			(_this$stagingBuffers = this.stagingBuffers) == null || _this$stagingBuffers.push(stagingBuffer);
			return {
				renderVersion,
				timings
			};
		});
	}
}

class WebgpuGpuProfiler extends GpuProfiler {
	constructor(device) {
		super();
		this.device = void 0;
		this.frameGPUMarkerSlot = void 0;
		this.device = device;
		this.timestampQueriesSet = device.supportsTimestampQuery ? new WebgpuQuerySet(device, true, 512) : null;
	}
	destroy() {
		var _this$timestampQuerie;
		(_this$timestampQuerie = this.timestampQueriesSet) == null || _this$timestampQuerie.destroy();
		this.timestampQueriesSet = null;
	}
	frameStart() {
		this.processEnableRequest();
	}
	frameEnd() {
		if (this._enabled) {
			var _this$timestampQuerie2;
			(_this$timestampQuerie2 = this.timestampQueriesSet) == null || _this$timestampQuerie2.resolve(this.slotCount * 2);
		}
	}
	request() {
		if (this._enabled) {
			var _this$timestampQuerie3;
			const renderVersion = this.device.renderVersion;
			(_this$timestampQuerie3 = this.timestampQueriesSet) == null || _this$timestampQuerie3.request(this.slotCount, renderVersion).then(results => {
				this.report(results.renderVersion, results.timings);
			});
			super.request(renderVersion);
		}
	}
}

class WebgpuResolver {
	constructor(device) {
		this.device = void 0;
		this.pipelineCache = new Map();
		this.device = device;
		const code = `
 
						var<private> pos : array<vec2f, 4> = array<vec2f, 4>(
								vec2(-1.0, 1.0), vec2(1.0, 1.0), vec2(-1.0, -1.0), vec2(1.0, -1.0)
						);

						struct VertexOutput {
								@builtin(position) position : vec4f,
						};

						@vertex
						fn vertexMain(@builtin(vertex_index) vertexIndex : u32) -> VertexOutput {
							var output : VertexOutput;
							output.position = vec4f(pos[vertexIndex], 0, 1);
							return output;
						}

						@group(0) @binding(0) var img : texture_depth_multisampled_2d;

						@fragment
						fn fragmentMain(@builtin(position) fragColor: vec4f) -> @location(0) vec4f {
								// load th depth value from sample index 0
								var depth = textureLoad(img, vec2i(fragColor.xy), 0u);
								return vec4<f32>(depth, 0.0, 0.0, 0.0);
						}
				`;
		this.shader = new Shader(device, {
			name: 'WebGPUResolverDepthShader',
			shaderLanguage: SHADERLANGUAGE_WGSL,
			vshader: code,
			fshader: code
		});
	}
	destroy() {
		this.shader.destroy();
		this.shader = null;
		this.pipelineCache = null;
	}
	getPipeline(format) {
		let pipeline = this.pipelineCache.get(format);
		if (!pipeline) {
			pipeline = this.createPipeline(format);
			this.pipelineCache.set(format, pipeline);
		}
		return pipeline;
	}
	createPipeline(format) {
		const webgpuShader = this.shader.impl;
		const pipeline = this.device.wgpu.createRenderPipeline({
			layout: 'auto',
			vertex: {
				module: webgpuShader.getVertexShaderModule(),
				entryPoint: webgpuShader.vertexEntryPoint
			},
			fragment: {
				module: webgpuShader.getFragmentShaderModule(),
				entryPoint: webgpuShader.fragmentEntryPoint,
				targets: [{
					format: format
				}]
			},
			primitive: {
				topology: 'triangle-strip'
			}
		});
		return pipeline;
	}
	resolveDepth(commandEncoder, sourceTexture, destinationTexture) {
		const device = this.device;
		const wgpu = device.wgpu;
		const pipeline = this.getPipeline(destinationTexture.format);
		const numFaces = sourceTexture.depthOrArrayLayers;
		for (let face = 0; face < numFaces; face++) {
			const srcView = sourceTexture.createView({
				dimension: '2d',
				aspect: 'depth-only',
				baseMipLevel: 0,
				mipLevelCount: 1,
				baseArrayLayer: face
			});
			const dstView = destinationTexture.createView({
				dimension: '2d',
				baseMipLevel: 0,
				mipLevelCount: 1,
				baseArrayLayer: face
			});
			const passEncoder = commandEncoder.beginRenderPass({
				colorAttachments: [{
					view: dstView,
					loadOp: 'clear',
					storeOp: 'store'
				}]
			});
			const bindGroup = wgpu.createBindGroup({
				layout: pipeline.getBindGroupLayout(0),
				entries: [{
					binding: 0,
					resource: srcView
				}]
			});
			passEncoder.setPipeline(pipeline);
			passEncoder.setBindGroup(0, bindGroup);
			passEncoder.draw(4);
			passEncoder.end();
		}
		device.pipeline = null;
	}
}

class WebgpuCompute {
	constructor(compute) {
		this.compute = compute;
		const {
			device,
			shader
		} = compute;
		const {
			computeBindGroupFormat
		} = shader.impl;
		this.bindGroup = new BindGroup(device, computeBindGroupFormat);
		this.pipeline = device.computePipeline.get(shader, computeBindGroupFormat);
	}
	dispatch(x, y, z) {
		const device = this.compute.device;
		device.startComputePass();
		const {
			bindGroup
		} = this;
		bindGroup.update();
		device.setBindGroup(0, bindGroup);
		const passEncoder = device.passEncoder;
		passEncoder.setPipeline(this.pipeline);
		passEncoder.dispatchWorkgroups(x, y, z);
		device.endComputePass();
	}
}

class WebgpuGraphicsDevice extends GraphicsDevice {
	constructor(canvas, options = {}) {
		var _options$alpha, _options$antialias;
		super(canvas, options);
		this.renderPipeline = new WebgpuRenderPipeline(this);
		this.computePipeline = new WebgpuComputePipeline(this);
		this.clearRenderer = void 0;
		this.mipmapRenderer = void 0;
		this.pipeline = void 0;
		this.bindGroupFormats = [];
		this.commandEncoder = null;
		this.commandBuffers = [];
		this.limits = void 0;
		options = this.initOptions;
		options.alpha = (_options$alpha = options.alpha) != null ? _options$alpha : true;
		this.backBufferAntialias = (_options$antialias = options.antialias) != null ? _options$antialias : false;
		this.isWebGPU = true;
		this._deviceType = DEVICETYPE_WEBGPU;
	}
	destroy() {
		this.clearRenderer.destroy();
		this.clearRenderer = null;
		this.mipmapRenderer.destroy();
		this.mipmapRenderer = null;
		this.resolver.destroy();
		this.resolver = null;
		super.destroy();
	}
	initDeviceCaps() {
		this.disableParticleSystem = true;
		const limits = this.gpuAdapter.limits;
		this.limits = limits;
		this.precision = 'highp';
		this.maxPrecision = 'highp';
		this.maxSamples = 4;
		this.maxTextures = 16;
		this.maxTextureSize = limits.maxTextureDimension2D;
		this.maxCubeMapSize = limits.maxTextureDimension2D;
		this.maxVolumeSize = limits.maxTextureDimension3D;
		this.maxColorAttachments = limits.maxColorAttachments;
		this.maxPixelRatio = 1;
		this.maxAnisotropy = 16;
		this.fragmentUniformsCount = limits.maxUniformBufferBindingSize / 16;
		this.vertexUniformsCount = limits.maxUniformBufferBindingSize / 16;
		this.supportsInstancing = true;
		this.supportsUniformBuffers = true;
		this.supportsVolumeTextures = true;
		this.supportsBoneTextures = true;
		this.supportsMorphTargetTexturesCore = true;
		this.supportsAreaLights = true;
		this.supportsDepthShadow = true;
		this.supportsGpuParticles = false;
		this.supportsMrt = true;
		this.supportsCompute = true;
		this.extUintElement = true;
		this.extTextureFloat = true;
		this.textureFloatRenderable = true;
		this.textureHalfFloatFilterable = true;
		this.extTextureHalfFloat = true;
		this.textureHalfFloatRenderable = true;
		this.textureHalfFloatUpdatable = true;
		this.boneLimit = 1024;
		this.supportsImageBitmap = true;
		this.extStandardDerivatives = true;
		this.extBlendMinmax = true;
		this.areaLightLutFormat = this.textureFloatFilterable ? PIXELFORMAT_RGBA32F : PIXELFORMAT_RGBA8;
		this.supportsTextureFetch = true;
		this.samples = this.backBufferAntialias ? 4 : 1;
	}
	async initWebGpu(glslangUrl, twgslUrl) {
		if (!window.navigator.gpu) {
			throw new Error('Unable to retrieve GPU. Ensure you are using a browser that supports WebGPU rendering.');
		}
		const buildUrl = srcPath => {
			if (!path.isRelativePath(srcPath)) {
				return srcPath;
			}
			const url = new URL(window.location.href);
			url.pathname = srcPath;
			url.search = '';
			return url.toString();
		};
		const results = await Promise.all([import(/* @vite-ignore */`${buildUrl(twgslUrl)}`).then(module => twgsl(twgslUrl.replace('.js', '.wasm'))), import(/* @vite-ignore */`${buildUrl(glslangUrl)}`).then(module => module.default())]);
		this.twgsl = results[0];
		this.glslang = results[1];
		const adapterOptions = {
			powerPreference: this.initOptions.powerPreference !== 'default' ? this.initOptions.powerPreference : undefined
		};
		this.gpuAdapter = await window.navigator.gpu.requestAdapter(adapterOptions);
		const requiredFeatures = [];
		const requireFeature = feature => {
			const supported = this.gpuAdapter.features.has(feature);
			if (supported) {
				requiredFeatures.push(feature);
			}
			return supported;
		};
		this.textureFloatFilterable = requireFeature('float32-filterable');
		this.extCompressedTextureS3TC = requireFeature('texture-compression-bc');
		this.extCompressedTextureETC = requireFeature('texture-compression-etc2');
		this.extCompressedTextureASTC = requireFeature('texture-compression-astc');
		this.supportsTimestampQuery = requireFeature('timestamp-query');
		this.textureRG11B10Renderable = requireFeature('rg11b10ufloat-renderable');
		const deviceDescr = {
			requiredFeatures,
			requiredLimits: {},
			defaultQueue: {
				label: 'Default Queue'
			}
		};
		this.wgpu = await this.gpuAdapter.requestDevice(deviceDescr);
		this.initDeviceCaps();
		this.gpuContext = this.canvas.getContext('webgpu');
		const preferredCanvasFormat = navigator.gpu.getPreferredCanvasFormat();
		this.backBufferFormat = preferredCanvasFormat === 'rgba8unorm' ? PIXELFORMAT_RGBA8 : PIXELFORMAT_BGRA8;
		this.canvasConfig = {
			device: this.wgpu,
			colorSpace: 'srgb',
			alphaMode: this.initOptions.alpha ? 'premultiplied' : 'opaque',
			format: preferredCanvasFormat,
			usage: GPUTextureUsage.RENDER_ATTACHMENT | GPUTextureUsage.COPY_SRC | GPUTextureUsage.COPY_DST,
			viewFormats: []
		};
		this.gpuContext.configure(this.canvasConfig);
		this.createBackbuffer();
		this.clearRenderer = new WebgpuClearRenderer(this);
		this.mipmapRenderer = new WebgpuMipmapRenderer(this);
		this.resolver = new WebgpuResolver(this);
		this.postInit();
		return this;
	}
	postInit() {
		super.postInit();
		this.initializeRenderState();
		this.setupPassEncoderDefaults();
		this.gpuProfiler = new WebgpuGpuProfiler(this);
		this.dynamicBuffers = new WebgpuDynamicBuffers(this, 1024 * 1024, this.limits.minUniformBufferOffsetAlignment);
	}
	createBackbuffer() {
		this.supportsStencil = this.initOptions.stencil;
		this.backBuffer = new RenderTarget({
			name: 'WebgpuFramebuffer',
			graphicsDevice: this,
			depth: this.initOptions.depth,
			stencil: this.supportsStencil,
			samples: this.samples
		});
	}
	frameStart() {
		super.frameStart();
		this.gpuProfiler.frameStart();
		this.submit();
		const outColorBuffer = this.gpuContext.getCurrentTexture();
		if (this.backBufferSize.x !== outColorBuffer.width || this.backBufferSize.y !== outColorBuffer.height) {
			this.backBufferSize.set(outColorBuffer.width, outColorBuffer.height);
			this.backBuffer.destroy();
			this.backBuffer = null;
			this.createBackbuffer();
		}
		const rt = this.backBuffer;
		const wrt = rt.impl;
		wrt.setColorAttachment(0, undefined, outColorBuffer.format);
		this.initRenderTarget(rt);
		wrt.assignColorTexture(outColorBuffer);
	}
	frameEnd() {
		super.frameEnd();
		this.gpuProfiler.frameEnd();
		this.submit();
		this.gpuProfiler.request();
	}
	createUniformBufferImpl(uniformBuffer) {
		return new WebgpuUniformBuffer(uniformBuffer);
	}
	createVertexBufferImpl(vertexBuffer, format) {
		return new WebgpuVertexBuffer(vertexBuffer, format);
	}
	createIndexBufferImpl(indexBuffer) {
		return new WebgpuIndexBuffer(indexBuffer);
	}
	createShaderImpl(shader) {
		return new WebgpuShader(shader);
	}
	createTextureImpl(texture) {
		return new WebgpuTexture(texture);
	}
	createRenderTargetImpl(renderTarget) {
		return new WebgpuRenderTarget(renderTarget);
	}
	createBindGroupFormatImpl(bindGroupFormat) {
		return new WebgpuBindGroupFormat(bindGroupFormat);
	}
	createBindGroupImpl(bindGroup) {
		return new WebgpuBindGroup();
	}
	createComputeImpl(compute) {
		return new WebgpuCompute(compute);
	}
	setBindGroup(index, bindGroup) {
		if (this.passEncoder) {
			this.passEncoder.setBindGroup(index, bindGroup.impl.bindGroup, bindGroup.uniformBufferOffsets);
			this.bindGroupFormats[index] = bindGroup.format.impl;
		}
	}
	submitVertexBuffer(vertexBuffer, slot) {
		const elements = vertexBuffer.format.elements;
		const elementCount = elements.length;
		const vbBuffer = vertexBuffer.impl.buffer;
		for (let i = 0; i < elementCount; i++) {
			this.passEncoder.setVertexBuffer(slot + i, vbBuffer, elements[i].offset);
		}
		return elementCount;
	}
	draw(primitive, numInstances = 1, keepBuffers) {
		if (this.shader.ready && !this.shader.failed) {
			const passEncoder = this.passEncoder;
			const vb0 = this.vertexBuffers[0];
			const vb1 = this.vertexBuffers[1];
			this.vertexBuffers.length = 0;
			if (vb0) {
				const vbSlot = this.submitVertexBuffer(vb0, 0);
				if (vb1) {
					this.submitVertexBuffer(vb1, vbSlot);
				}
			}
			const pipeline = this.renderPipeline.get(primitive, vb0 == null ? void 0 : vb0.format, vb1 == null ? void 0 : vb1.format, this.shader, this.renderTarget, this.bindGroupFormats, this.blendState, this.depthState, this.cullMode, this.stencilEnabled, this.stencilFront, this.stencilBack);
			if (this.pipeline !== pipeline) {
				this.pipeline = pipeline;
				passEncoder.setPipeline(pipeline);
			}
			const ib = this.indexBuffer;
			if (ib) {
				this.indexBuffer = null;
				passEncoder.setIndexBuffer(ib.impl.buffer, ib.impl.format);
				passEncoder.drawIndexed(primitive.count, numInstances, primitive.base, 0, 0);
			} else {
				passEncoder.draw(primitive.count, numInstances, primitive.base, 0);
			}
		}
	}
	setShader(shader, asyncCompile = false) {
		if (shader !== this.shader) {
			this.shader = shader;
		}
	}
	setBlendState(blendState) {
		this.blendState.copy(blendState);
	}
	setDepthState(depthState) {
		this.depthState.copy(depthState);
	}
	setStencilState(stencilFront, stencilBack) {
		if (stencilFront || stencilBack) {
			this.stencilEnabled = true;
			this.stencilFront.copy(stencilFront != null ? stencilFront : StencilParameters.DEFAULT);
			this.stencilBack.copy(stencilBack != null ? stencilBack : StencilParameters.DEFAULT);
			const ref = this.stencilFront.ref;
			if (this.stencilRef !== ref) {
				this.stencilRef = ref;
				this.passEncoder.setStencilReference(ref);
			}
		} else {
			this.stencilEnabled = false;
		}
	}
	setBlendColor(r, g, b, a) {
		const c = this.blendColor;
		if (r !== c.r || g !== c.g || b !== c.b || a !== c.a) {
			c.set(r, g, b, a);
			this.passEncoder.setBlendConstant(c);
		}
	}
	setCullMode(cullMode) {
		this.cullMode = cullMode;
	}
	setAlphaToCoverage(state) {}
	initializeContextCaches() {
		super.initializeContextCaches();
	}
	setupPassEncoderDefaults() {
		this.pipeline = null;
		this.stencilRef = 0;
		this.blendColor.set(0, 0, 0, 0);
	}
	_uploadDirtyTextures() {
		this.textures.forEach(texture => {
			if (texture._needsUpload || texture._needsMipmaps) {
				texture.upload();
			}
		});
	}
	startRenderPass(renderPass) {
		this._uploadDirtyTextures();
		const rt = renderPass.renderTarget || this.backBuffer;
		this.renderTarget = rt;
		const wrt = rt.impl;
		this.commandEncoder = this.wgpu.createCommandEncoder();
		if (rt !== this.backBuffer) {
			this.initRenderTarget(rt);
		}
		wrt.setupForRenderPass(renderPass);
		const renderPassDesc = wrt.renderPassDescriptor;
		if (this.gpuProfiler._enabled) {
			if (this.gpuProfiler.timestampQueriesSet) {
				const slot = this.gpuProfiler.getSlot(renderPass.name);
				renderPassDesc.timestampWrites = {
					querySet: this.gpuProfiler.timestampQueriesSet.querySet,
					beginningOfPassWriteIndex: slot * 2,
					endOfPassWriteIndex: slot * 2 + 1
				};
			}
		}
		this.passEncoder = this.commandEncoder.beginRenderPass(renderPassDesc);
		this.setupPassEncoderDefaults();
		const {
			width,
			height
		} = rt;
		this.setViewport(0, 0, width, height);
		this.setScissor(0, 0, width, height);
		this.insideRenderPass = true;
	}
	endRenderPass(renderPass) {
		this.passEncoder.end();
		this.passEncoder = null;
		this.insideRenderPass = false;
		this.bindGroupFormats.length = 0;
		for (let i = 0; i < renderPass.colorArrayOps.length; i++) {
			const colorOps = renderPass.colorArrayOps[i];
			if (colorOps.mipmaps) {
				this.mipmapRenderer.generate(renderPass.renderTarget._colorBuffers[i].impl);
			}
		}
		const cb = this.commandEncoder.finish();
		this.addCommandBuffer(cb);
		this.commandEncoder = null;
	}
	startComputePass() {
		this.commandEncoder = this.wgpu.createCommandEncoder();
		this.pipeline = null;
		this.passEncoder = this.commandEncoder.beginComputePass();
		this.insideRenderPass = true;
	}
	endComputePass() {
		this.passEncoder.end();
		this.passEncoder = null;
		this.insideRenderPass = false;
		this.bindGroupFormats.length = 0;
		const cb = this.commandEncoder.finish();
		this.addCommandBuffer(cb);
		this.commandEncoder = null;
	}
	addCommandBuffer(commandBuffer, front = false) {
		if (front) {
			this.commandBuffers.unshift(commandBuffer);
		} else {
			this.commandBuffers.push(commandBuffer);
		}
	}
	submit() {
		if (this.commandBuffers.length > 0) {
			this.dynamicBuffers.submit();
			this.wgpu.queue.submit(this.commandBuffers);
			this.commandBuffers.length = 0;
			this.dynamicBuffers.onCommandBuffersSubmitted();
		}
	}
	clear(options) {
		if (options.flags) {
			this.clearRenderer.clear(this, this.renderTarget, options, this.defaultClearOptions);
		}
	}
	setViewport(x, y, w, h) {
		if (this.passEncoder) {
			if (!this.renderTarget.flipY) {
				y = this.renderTarget.height - y - h;
			}
			this.vx = x;
			this.vy = y;
			this.vw = w;
			this.vh = h;
			this.passEncoder.setViewport(x, y, w, h, 0, 1);
		}
	}
	setScissor(x, y, w, h) {
		if (this.passEncoder) {
			if (!this.renderTarget.flipY) {
				y = this.renderTarget.height - y - h;
			}
			this.sx = x;
			this.sy = y;
			this.sw = w;
			this.sh = h;
			this.passEncoder.setScissorRect(x, y, w, h);
		}
	}
	copyRenderTarget(source, dest, color, depth) {
		var _this$commandEncoder;
		const copySize = {
			width: source ? source.width : dest.width,
			height: source ? source.height : dest.height,
			depthOrArrayLayers: 1
		};
		const commandEncoder = (_this$commandEncoder = this.commandEncoder) != null ? _this$commandEncoder : this.wgpu.createCommandEncoder();
		if (color) {
			const copySrc = {
				texture: source ? source.colorBuffer.impl.gpuTexture : this.renderTarget.impl.assignedColorTexture,
				mipLevel: 0
			};
			const copyDst = {
				texture: dest ? dest.colorBuffer.impl.gpuTexture : this.renderTarget.impl.assignedColorTexture,
				mipLevel: 0
			};
			commandEncoder.copyTextureToTexture(copySrc, copyDst, copySize);
		}
		if (depth) {
			const sourceRT = source ? source : this.renderTarget;
			const sourceTexture = sourceRT.impl.depthTexture;
			if (source.samples > 1) {
				const destTexture = dest.colorBuffer.impl.gpuTexture;
				this.resolver.resolveDepth(commandEncoder, sourceTexture, destTexture);
			} else {
				const destTexture = dest ? dest.depthBuffer.impl.gpuTexture : this.renderTarget.impl.depthTexture;
				const copySrc = {
					texture: sourceTexture,
					mipLevel: 0
				};
				const copyDst = {
					texture: destTexture,
					mipLevel: 0
				};
				commandEncoder.copyTextureToTexture(copySrc, copyDst, copySize);
			}
		}
		if (!this.commandEncoder) {
			const cb = commandEncoder.finish();
			this.addCommandBuffer(cb);
		}
		return true;
	}
}

let id$5 = 0;
class Texture {
	constructor(graphicsDevice, options = {}) {
		var _options$name, _options$width, _options$height, _options$format, _options$storage, _options$cubemap, _options$fixCubemapSe, _options$flipY, _options$premultiplyA, _ref, _options$mipmaps, _options$minFilter, _options$magFilter, _options$anisotropy, _options$addressU, _options$addressV, _options$addressW, _options$compareOnRea, _options$compareFunc;
		this.name = void 0;
		this._gpuSize = 0;
		this.id = id$5++;
		this._invalid = false;
		this._lockedLevel = -1;
		this._lockedMode = TEXTURELOCK_NONE;
		this.renderVersionDirty = 0;
		this._storage = false;
		this.device = graphicsDevice;
		this.name = (_options$name = options.name) != null ? _options$name : '';
		this._width = Math.floor((_options$width = options.width) != null ? _options$width : 4);
		this._height = Math.floor((_options$height = options.height) != null ? _options$height : 4);
		this._format = (_options$format = options.format) != null ? _options$format : PIXELFORMAT_RGBA8;
		this._compressed = isCompressedPixelFormat(this._format);
		this._integerFormat = isIntegerPixelFormat(this._format);
		if (this._integerFormat) {
			options.mipmaps = false;
			options.minFilter = FILTER_NEAREST;
			options.magFilter = FILTER_NEAREST;
		}
		if (graphicsDevice.supportsVolumeTextures) {
			var _options$volume, _options$depth, _options$arrayLength;
			this._volume = (_options$volume = options.volume) != null ? _options$volume : false;
			this._depth = Math.floor((_options$depth = options.depth) != null ? _options$depth : 1);
			this._arrayLength = Math.floor((_options$arrayLength = options.arrayLength) != null ? _options$arrayLength : 0);
		} else {
			this._volume = false;
			this._depth = 1;
			this._arrayLength = 0;
		}
		this._storage = (_options$storage = options.storage) != null ? _options$storage : false;
		this._cubemap = (_options$cubemap = options.cubemap) != null ? _options$cubemap : false;
		this.fixCubemapSeams = (_options$fixCubemapSe = options.fixCubemapSeams) != null ? _options$fixCubemapSe : false;
		this._flipY = (_options$flipY = options.flipY) != null ? _options$flipY : false;
		this._premultiplyAlpha = (_options$premultiplyA = options.premultiplyAlpha) != null ? _options$premultiplyA : false;
		this._mipmaps = (_ref = (_options$mipmaps = options.mipmaps) != null ? _options$mipmaps : options.autoMipmap) != null ? _ref : true;
		this._minFilter = (_options$minFilter = options.minFilter) != null ? _options$minFilter : FILTER_LINEAR_MIPMAP_LINEAR;
		this._magFilter = (_options$magFilter = options.magFilter) != null ? _options$magFilter : FILTER_LINEAR;
		this._anisotropy = (_options$anisotropy = options.anisotropy) != null ? _options$anisotropy : 1;
		this._addressU = (_options$addressU = options.addressU) != null ? _options$addressU : ADDRESS_REPEAT;
		this._addressV = (_options$addressV = options.addressV) != null ? _options$addressV : ADDRESS_REPEAT;
		this._addressW = (_options$addressW = options.addressW) != null ? _options$addressW : ADDRESS_REPEAT;
		this._compareOnRead = (_options$compareOnRea = options.compareOnRead) != null ? _options$compareOnRea : false;
		this._compareFunc = (_options$compareFunc = options.compareFunc) != null ? _options$compareFunc : FUNC_LESS;
		this.type = TEXTURETYPE_DEFAULT;
		if (options.hasOwnProperty('type')) {
			this.type = options.type;
		} else if (options.hasOwnProperty('rgbm')) {
			this.type = options.rgbm ? TEXTURETYPE_RGBM : TEXTURETYPE_DEFAULT;
		} else if (options.hasOwnProperty('swizzleGGGR')) {
			this.type = options.swizzleGGGR ? TEXTURETYPE_SWIZZLEGGGR : TEXTURETYPE_DEFAULT;
		}
		this.projection = TEXTUREPROJECTION_NONE;
		if (this._cubemap) {
			this.projection = TEXTUREPROJECTION_CUBE;
		} else if (options.projection && options.projection !== TEXTUREPROJECTION_CUBE) {
			this.projection = options.projection;
		}
		this.impl = graphicsDevice.createTextureImpl(this);
		this.dirtyAll();
		this._levels = options.levels;
		if (this._levels) {
			this.upload();
		} else {
			this._levels = this._cubemap ? [[null, null, null, null, null, null]] : [null];
		}
		graphicsDevice.textures.push(this);
	}
	destroy() {
		const device = this.device;
		if (device) {
			const idx = device.textures.indexOf(this);
			if (idx !== -1) {
				device.textures.splice(idx, 1);
			}
			device.scope.removeValue(this);
			this.impl.destroy(device);
			this.adjustVramSizeTracking(device._vram, -this._gpuSize);
			this._levels = null;
			this.device = null;
		}
	}
	resize(width, height, depth = 1) {
		const device = this.device;
		this.adjustVramSizeTracking(device._vram, -this._gpuSize);
		this.impl.destroy(device);
		this._width = Math.floor(width);
		this._height = Math.floor(height);
		this._depth = Math.floor(depth);
		this.impl = device.createTextureImpl(this);
		this.dirtyAll();
	}
	loseContext() {
		this.impl.loseContext();
		this.dirtyAll();
	}
	adjustVramSizeTracking(vram, size) {
		vram.tex += size;
	}
	propertyChanged(flag) {
		this.impl.propertyChanged(flag);
		this.renderVersionDirty = this.device.renderVersion;
	}
	get requiredMipLevels() {
		return this.mipmaps ? TextureUtils.calcMipLevelsCount(this.width, this.height) : 1;
	}
	get lockedMode() {
		return this._lockedMode;
	}
	set minFilter(v) {
		if (this._minFilter !== v) {
			if (isIntegerPixelFormat(this._format)) ; else {
				this._minFilter = v;
				this.propertyChanged(1);
			}
		}
	}
	get minFilter() {
		return this._minFilter;
	}
	set magFilter(v) {
		if (this._magFilter !== v) {
			if (isIntegerPixelFormat(this._format)) ; else {
				this._magFilter = v;
				this.propertyChanged(2);
			}
		}
	}
	get magFilter() {
		return this._magFilter;
	}
	set addressU(v) {
		if (this._addressU !== v) {
			this._addressU = v;
			this.propertyChanged(4);
		}
	}
	get addressU() {
		return this._addressU;
	}
	set addressV(v) {
		if (this._addressV !== v) {
			this._addressV = v;
			this.propertyChanged(8);
		}
	}
	get addressV() {
		return this._addressV;
	}
	set addressW(addressW) {
		if (!this.device.supportsVolumeTextures) return;
		if (!this._volume) {
			return;
		}
		if (addressW !== this._addressW) {
			this._addressW = addressW;
			this.propertyChanged(16);
		}
	}
	get addressW() {
		return this._addressW;
	}
	set compareOnRead(v) {
		if (this._compareOnRead !== v) {
			this._compareOnRead = v;
			this.propertyChanged(32);
		}
	}
	get compareOnRead() {
		return this._compareOnRead;
	}
	set compareFunc(v) {
		if (this._compareFunc !== v) {
			this._compareFunc = v;
			this.propertyChanged(64);
		}
	}
	get compareFunc() {
		return this._compareFunc;
	}
	set anisotropy(v) {
		if (this._anisotropy !== v) {
			this._anisotropy = v;
			this.propertyChanged(128);
		}
	}
	get anisotropy() {
		return this._anisotropy;
	}
	set mipmaps(v) {
		if (this._mipmaps !== v) {
			if (this.device.isWebGPU) ; else if (isIntegerPixelFormat(this._format)) ; else {
				this._mipmaps = v;
			}
			if (v) this._needsMipmapsUpload = true;
		}
	}
	get mipmaps() {
		return this._mipmaps;
	}
	get storage() {
		return this._storage;
	}
	get width() {
		return this._width;
	}
	get height() {
		return this._height;
	}
	get depth() {
		return this._depth;
	}
	get format() {
		return this._format;
	}
	get cubemap() {
		return this._cubemap;
	}
	get gpuSize() {
		const mips = this.pot && this._mipmaps && !(this._compressed && this._levels.length === 1);
		return TextureUtils.calcGpuSize(this._width, this._height, this._depth, this._format, mips, this._cubemap);
	}
	get array() {
		return this._arrayLength > 0;
	}
	get arrayLength() {
		return this._arrayLength;
	}
	get volume() {
		return this._volume;
	}
	set flipY(flipY) {
		if (this._flipY !== flipY) {
			this._flipY = flipY;
			this._needsUpload = true;
		}
	}
	get flipY() {
		return this._flipY;
	}
	set premultiplyAlpha(premultiplyAlpha) {
		if (this._premultiplyAlpha !== premultiplyAlpha) {
			this._premultiplyAlpha = premultiplyAlpha;
			this._needsUpload = true;
		}
	}
	get premultiplyAlpha() {
		return this._premultiplyAlpha;
	}
	get pot() {
		return math.powerOfTwo(this._width) && math.powerOfTwo(this._height);
	}
	get encoding() {
		switch (this.type) {
			case TEXTURETYPE_RGBM:
				return 'rgbm';
			case TEXTURETYPE_RGBE:
				return 'rgbe';
			case TEXTURETYPE_RGBP:
				return 'rgbp';
			default:
				return this.format === PIXELFORMAT_RGB16F || this.format === PIXELFORMAT_RGB32F || this.format === PIXELFORMAT_RGBA16F || this.format === PIXELFORMAT_RGBA32F || isIntegerPixelFormat(this.format) ? 'linear' : 'srgb';
		}
	}
	dirtyAll() {
		this._levelsUpdated = this._cubemap ? [[true, true, true, true, true, true]] : [true];
		this._needsUpload = true;
		this._needsMipmapsUpload = this._mipmaps;
		this._mipmapsUploaded = false;
		this.propertyChanged(255);
	}
	lock(options = {}) {
		var _options$level, _options$face, _options$mode;
		(_options$level = options.level) != null ? _options$level : options.level = 0;
		(_options$face = options.face) != null ? _options$face : options.face = 0;
		(_options$mode = options.mode) != null ? _options$mode : options.mode = TEXTURELOCK_WRITE;
		this._lockedMode = options.mode;
		this._lockedLevel = options.level;
		const levels = this.cubemap ? this._levels[options.face] : this._levels;
		if (levels[options.level] === null) {
			const width = Math.max(1, this._width >> options.level);
			const height = Math.max(1, this._height >> options.level);
			const depth = Math.max(1, this._depth >> options.level);
			const data = new ArrayBuffer(TextureUtils.calcLevelGpuSize(width, height, depth, this._format));
			levels[options.level] = new (getPixelFormatArrayType(this._format))(data);
		}
		return levels[options.level];
	}
	setSource(source, mipLevel = 0) {
		let invalid = false;
		let width, height;
		if (this._cubemap) {
			if (source[0]) {
				width = source[0].width || 0;
				height = source[0].height || 0;
				for (let i = 0; i < 6; i++) {
					const face = source[i];
					if (!face || face.width !== width || face.height !== height || !this.device._isBrowserInterface(face)) {
						invalid = true;
						break;
					}
				}
			} else {
				invalid = true;
			}
			if (!invalid) {
				for (let i = 0; i < 6; i++) {
					if (this._levels[mipLevel][i] !== source[i]) this._levelsUpdated[mipLevel][i] = true;
				}
			}
		} else {
			if (!this.device._isBrowserInterface(source)) invalid = true;
			if (!invalid) {
				if (source !== this._levels[mipLevel]) this._levelsUpdated[mipLevel] = true;
				width = source.width;
				height = source.height;
			}
		}
		if (invalid) {
			this._width = 4;
			this._height = 4;
			if (this._cubemap) {
				for (let i = 0; i < 6; i++) {
					this._levels[mipLevel][i] = null;
					this._levelsUpdated[mipLevel][i] = true;
				}
			} else {
				this._levels[mipLevel] = null;
				this._levelsUpdated[mipLevel] = true;
			}
		} else {
			if (mipLevel === 0) {
				this._width = width;
				this._height = height;
			}
			this._levels[mipLevel] = source;
		}
		if (this._invalid !== invalid || !invalid) {
			this._invalid = invalid;
			this.upload();
		}
	}
	getSource(mipLevel = 0) {
		return this._levels[mipLevel];
	}
	unlock() {
		if (this._lockedMode === TEXTURELOCK_NONE) ;
		if (this._lockedMode === TEXTURELOCK_WRITE) {
			this.upload();
		}
		this._lockedLevel = -1;
		this._lockedMode = TEXTURELOCK_NONE;
	}
	upload() {
		var _this$impl$uploadImme, _this$impl;
		this._needsUpload = true;
		this._needsMipmapsUpload = this._mipmaps;
		(_this$impl$uploadImme = (_this$impl = this.impl).uploadImmediate) == null || _this$impl$uploadImme.call(_this$impl, this.device, this);
	}
	async downloadAsync() {
		const promises = [];
		for (let i = 0; i < (this.cubemap ? 6 : 1); i++) {
			var _this$device$readPixe, _this$device;
			const renderTarget = new RenderTarget({
				colorBuffer: this,
				depth: false,
				face: i
			});
			this.device.setRenderTarget(renderTarget);
			this.device.initRenderTarget(renderTarget);
			const levels = this.cubemap ? this._levels[i] : this._levels;
			let level = levels[0];
			if (levels[0] && this.device._isBrowserInterface(levels[0])) {
				levels[0] = null;
			}
			level = this.lock({
				face: i
			});
			const promise = (_this$device$readPixe = (_this$device = this.device).readPixelsAsync) == null ? void 0 : _this$device$readPixe.call(_this$device, 0, 0, this.width, this.height, level).then(() => renderTarget.destroy());
			promises.push(promise);
		}
		await Promise.all(promises);
	}
}

class WebglBuffer {
	constructor() {
		this.bufferId = null;
	}
	destroy(device) {
		if (this.bufferId) {
			device.gl.deleteBuffer(this.bufferId);
			this.bufferId = null;
		}
	}
	get initialized() {
		return !!this.bufferId;
	}
	loseContext() {
		this.bufferId = null;
	}
	unlock(device, usage, target, storage) {
		const gl = device.gl;
		if (!this.bufferId) {
			let glUsage;
			switch (usage) {
				case BUFFER_STATIC:
					glUsage = gl.STATIC_DRAW;
					break;
				case BUFFER_DYNAMIC:
					glUsage = gl.DYNAMIC_DRAW;
					break;
				case BUFFER_STREAM:
					glUsage = gl.STREAM_DRAW;
					break;
				case BUFFER_GPUDYNAMIC:
					glUsage = device.isWebGL2 ? gl.DYNAMIC_COPY : gl.STATIC_DRAW;
					break;
			}
			this.bufferId = gl.createBuffer();
			gl.bindBuffer(target, this.bufferId);
			gl.bufferData(target, storage, glUsage);
		} else {
			gl.bindBuffer(target, this.bufferId);
			gl.bufferSubData(target, 0, storage);
		}
	}
}

class WebglVertexBuffer extends WebglBuffer {
	constructor(...args) {
		super(...args);
		this.vao = null;
	}
	destroy(device) {
		super.destroy(device);
		device.unbindVertexArray();
	}
	loseContext() {
		super.loseContext();
		this.vao = null;
	}
	unlock(vertexBuffer) {
		const device = vertexBuffer.device;
		super.unlock(device, vertexBuffer.usage, device.gl.ARRAY_BUFFER, vertexBuffer.storage);
	}
}

class WebglIndexBuffer extends WebglBuffer {
	constructor(indexBuffer) {
		super();
		const gl = indexBuffer.device.gl;
		const format = indexBuffer.format;
		if (format === INDEXFORMAT_UINT8) {
			this.glFormat = gl.UNSIGNED_BYTE;
		} else if (format === INDEXFORMAT_UINT16) {
			this.glFormat = gl.UNSIGNED_SHORT;
		} else if (format === INDEXFORMAT_UINT32) {
			this.glFormat = gl.UNSIGNED_INT;
		}
	}
	unlock(indexBuffer) {
		const device = indexBuffer.device;
		super.unlock(device, indexBuffer.usage, device.gl.ELEMENT_ARRAY_BUFFER, indexBuffer.storage);
	}
}

class WebglShaderInput {
	constructor(graphicsDevice, name, type, locationId) {
		this.locationId = locationId;
		this.scopeId = graphicsDevice.scope.resolve(name);
		this.version = new Version();
		if (name.substring(name.length - 3) === "[0]") {
			switch (type) {
				case UNIFORMTYPE_FLOAT:
					type = UNIFORMTYPE_FLOATARRAY;
					break;
				case UNIFORMTYPE_INT:
					type = UNIFORMTYPE_INTARRAY;
					break;
				case UNIFORMTYPE_UINT:
					type = UNIFORMTYPE_UINTARRAY;
					break;
				case UNIFORMTYPE_BOOL:
					type = UNIFORMTYPE_BOOLARRAY;
					break;
				case UNIFORMTYPE_VEC2:
					type = UNIFORMTYPE_VEC2ARRAY;
					break;
				case UNIFORMTYPE_IVEC2:
					type = UNIFORMTYPE_IVEC2ARRAY;
					break;
				case UNIFORMTYPE_UVEC2:
					type = UNIFORMTYPE_UVEC2ARRAY;
					break;
				case UNIFORMTYPE_BVEC2:
					type = UNIFORMTYPE_BVEC2ARRAY;
					break;
				case UNIFORMTYPE_VEC3:
					type = UNIFORMTYPE_VEC3ARRAY;
					break;
				case UNIFORMTYPE_IVEC3:
					type = UNIFORMTYPE_IVEC3ARRAY;
					break;
				case UNIFORMTYPE_UVEC3:
					type = UNIFORMTYPE_UVEC3ARRAY;
					break;
				case UNIFORMTYPE_BVEC3:
					type = UNIFORMTYPE_BVEC3ARRAY;
					break;
				case UNIFORMTYPE_VEC4:
					type = UNIFORMTYPE_VEC4ARRAY;
					break;
				case UNIFORMTYPE_IVEC4:
					type = UNIFORMTYPE_IVEC4ARRAY;
					break;
				case UNIFORMTYPE_UVEC4:
					type = UNIFORMTYPE_UVEC4ARRAY;
					break;
				case UNIFORMTYPE_BVEC4:
					type = UNIFORMTYPE_BVEC4ARRAY;
					break;
			}
		}
		this.dataType = type;
		this.value = [null, null, null, null];
		this.array = [];
	}
}

const _vertexShaderBuiltins = new Set(['gl_VertexID', 'gl_InstanceID', 'gl_DrawID', 'gl_BaseVertex', 'gl_BaseInstance']);
class CompiledShaderCache {
	constructor() {
		this.map = new Map();
	}
	destroy(device) {
		this.map.forEach(shader => {
			device.gl.deleteShader(shader);
		});
	}
	loseContext(device) {
		this.map.clear();
	}
}
const _vertexShaderCache = new DeviceCache();
const _fragmentShaderCache = new DeviceCache();
class WebglShader {
	constructor(shader) {
		this.compileDuration = 0;
		this.init();
		this.compile(shader.device, shader);
		this.link(shader.device, shader);
		shader.device.shaders.push(shader);
	}
	destroy(shader) {
		if (this.glProgram) {
			shader.device.gl.deleteProgram(this.glProgram);
			this.glProgram = null;
		}
	}
	init() {
		this.uniforms = [];
		this.samplers = [];
		this.attributes = [];
		this.glProgram = null;
		this.glVertexShader = null;
		this.glFragmentShader = null;
	}
	loseContext() {
		this.init();
	}
	restoreContext(device, shader) {
		this.compile(device, shader);
		this.link(device, shader);
	}
	compile(device, shader) {
		const definition = shader.definition;
		this.glVertexShader = this._compileShaderSource(device, definition.vshader, true);
		this.glFragmentShader = this._compileShaderSource(device, definition.fshader, false);
	}
	link(device, shader) {
		if (this.glProgram) return;
		const gl = device.gl;
		if (gl.isContextLost()) {
			return;
		}
		const glProgram = gl.createProgram();
		this.glProgram = glProgram;
		gl.attachShader(glProgram, this.glVertexShader);
		gl.attachShader(glProgram, this.glFragmentShader);
		const definition = shader.definition;
		const attrs = definition.attributes;
		if (device.isWebGL2 && definition.useTransformFeedback) {
			const outNames = [];
			for (const attr in attrs) {
				if (attrs.hasOwnProperty(attr)) {
					outNames.push("out_" + attr);
				}
			}
			gl.transformFeedbackVaryings(glProgram, outNames, gl.INTERLEAVED_ATTRIBS);
		}
		for (const attr in attrs) {
			if (attrs.hasOwnProperty(attr)) {
				const semantic = attrs[attr];
				const loc = semanticToLocation[semantic];
				gl.bindAttribLocation(glProgram, loc, attr);
			}
		}
		gl.linkProgram(glProgram);
	}
	_compileShaderSource(device, src, isVertexShader) {
		const gl = device.gl;
		const shaderDeviceCache = isVertexShader ? _vertexShaderCache : _fragmentShaderCache;
		const shaderCache = shaderDeviceCache.get(device, () => {
			return new CompiledShaderCache();
		});
		let glShader = shaderCache.map.get(src);
		if (!glShader) {
			glShader = gl.createShader(isVertexShader ? gl.VERTEX_SHADER : gl.FRAGMENT_SHADER);
			if (!glShader && gl.isContextLost()) {
				return glShader;
			}
			gl.shaderSource(glShader, src);
			gl.compileShader(glShader);
			shaderCache.map.set(src, glShader);
		}
		return glShader;
	}
	finalize(device, shader) {
		const gl = device.gl;
		if (gl.isContextLost()) {
			return true;
		}
		const glProgram = this.glProgram;
		const definition = shader.definition;
		const linkStatus = gl.getProgramParameter(glProgram, gl.LINK_STATUS);
		if (!linkStatus) {
			if (!this._isCompiled(device, shader, this.glVertexShader, definition.vshader, "vertex")) return false;
			if (!this._isCompiled(device, shader, this.glFragmentShader, definition.fshader, "fragment")) return false;
			const message = "Failed to link shader program. Error: " + gl.getProgramInfoLog(glProgram);
			console.error(message);
			return false;
		}
		const numAttributes = gl.getProgramParameter(glProgram, gl.ACTIVE_ATTRIBUTES);
		for (let i = 0; i < numAttributes; i++) {
			const info = gl.getActiveAttrib(glProgram, i);
			const location = gl.getAttribLocation(glProgram, info.name);
			if (_vertexShaderBuiltins.has(info.name)) continue;
			if (definition.attributes[info.name] === undefined) {
				console.error(`Vertex shader attribute "${info.name}" is not mapped to a semantic in shader definition, shader [${shader.label}]`, shader);
				shader.failed = true;
			} else {
				const shaderInput = new WebglShaderInput(device, definition.attributes[info.name], device.pcUniformType[info.type], location);
				this.attributes.push(shaderInput);
			}
		}
		const samplerTypes = device._samplerTypes;
		const numUniforms = gl.getProgramParameter(glProgram, gl.ACTIVE_UNIFORMS);
		for (let i = 0; i < numUniforms; i++) {
			const info = gl.getActiveUniform(glProgram, i);
			const location = gl.getUniformLocation(glProgram, info.name);
			const shaderInput = new WebglShaderInput(device, info.name, device.pcUniformType[info.type], location);
			if (samplerTypes.has(info.type)) {
				this.samplers.push(shaderInput);
			} else {
				this.uniforms.push(shaderInput);
			}
		}
		shader.ready = true;
		return true;
	}
	_isCompiled(device, shader, glShader, source, shaderType) {
		const gl = device.gl;
		if (!gl.getShaderParameter(glShader, gl.COMPILE_STATUS)) {
			const infoLog = gl.getShaderInfoLog(glShader);
			const [code, error] = this._processError(source, infoLog);
			const message = `Failed to compile ${shaderType} shader:\n\n${infoLog}\n${code} while rendering ${void 0}`;
			console.error(message);
			return false;
		}
		return true;
	}
	isLinked(device) {
		const {
			extParallelShaderCompile
		} = device;
		if (extParallelShaderCompile) {
			return device.gl.getProgramParameter(this.glProgram, extParallelShaderCompile.COMPLETION_STATUS_KHR);
		}
		return true;
	}
	_processError(src, infoLog) {
		const error = {};
		let code = '';
		if (src) {
			const lines = src.split('\n');
			let from = 0;
			let to = lines.length;
			if (infoLog && infoLog.startsWith('ERROR:')) {
				const match = infoLog.match(/^ERROR:\s([0-9]+):([0-9]+):\s*(.+)/);
				if (match) {
					error.message = match[3];
					error.line = parseInt(match[2], 10);
					from = Math.max(0, error.line - 6);
					to = Math.min(lines.length, error.line + 5);
				}
			}
			for (let i = from; i < to; i++) {
				code += i + 1 + ":\t" + lines[i] + '\n';
			}
			error.source = src;
		}
		return [code, error];
	}
}

function downsampleImage(image, size) {
	const srcW = image.width;
	const srcH = image.height;
	if (srcW > size || srcH > size) {
		const scale = size / Math.max(srcW, srcH);
		const dstW = Math.floor(srcW * scale);
		const dstH = Math.floor(srcH * scale);
		const canvas = document.createElement('canvas');
		canvas.width = dstW;
		canvas.height = dstH;
		const context = canvas.getContext('2d');
		context.drawImage(image, 0, 0, srcW, srcH, 0, 0, dstW, dstH);
		return canvas;
	}
	return image;
}
class WebglTexture {
	constructor() {
		this._glTexture = null;
		this._glTarget = void 0;
		this._glFormat = void 0;
		this._glInternalFormat = void 0;
		this._glPixelType = void 0;
		this._glCreated = void 0;
		this.dirtyParameterFlags = 0;
	}
	destroy(device) {
		if (this._glTexture) {
			for (let i = 0; i < device.textureUnits.length; i++) {
				const textureUnit = device.textureUnits[i];
				for (let j = 0; j < textureUnit.length; j++) {
					if (textureUnit[j] === this._glTexture) {
						textureUnit[j] = null;
					}
				}
			}
			device.gl.deleteTexture(this._glTexture);
			this._glTexture = null;
		}
	}
	loseContext() {
		this._glTexture = null;
	}
	propertyChanged(flag) {
		this.dirtyParameterFlags |= flag;
	}
	initialize(device, texture) {
		const gl = device.gl;
		this._glTexture = gl.createTexture();
		this._glTarget = texture._cubemap ? gl.TEXTURE_CUBE_MAP : texture._volume ? gl.TEXTURE_3D : texture.array ? gl.TEXTURE_2D_ARRAY : gl.TEXTURE_2D;
		switch (texture._format) {
			case PIXELFORMAT_A8:
				this._glFormat = gl.ALPHA;
				this._glInternalFormat = gl.ALPHA;
				this._glPixelType = gl.UNSIGNED_BYTE;
				break;
			case PIXELFORMAT_L8:
				this._glFormat = gl.LUMINANCE;
				this._glInternalFormat = gl.LUMINANCE;
				this._glPixelType = gl.UNSIGNED_BYTE;
				break;
			case PIXELFORMAT_LA8:
				this._glFormat = gl.LUMINANCE_ALPHA;
				this._glInternalFormat = gl.LUMINANCE_ALPHA;
				this._glPixelType = gl.UNSIGNED_BYTE;
				break;
			case PIXELFORMAT_RGB565:
				this._glFormat = gl.RGB;
				this._glInternalFormat = gl.RGB;
				this._glPixelType = gl.UNSIGNED_SHORT_5_6_5;
				break;
			case PIXELFORMAT_RGBA5551:
				this._glFormat = gl.RGBA;
				this._glInternalFormat = gl.RGBA;
				this._glPixelType = gl.UNSIGNED_SHORT_5_5_5_1;
				break;
			case PIXELFORMAT_RGBA4:
				this._glFormat = gl.RGBA;
				this._glInternalFormat = gl.RGBA;
				this._glPixelType = gl.UNSIGNED_SHORT_4_4_4_4;
				break;
			case PIXELFORMAT_RGB8:
				this._glFormat = gl.RGB;
				this._glInternalFormat = device.isWebGL2 ? gl.RGB8 : gl.RGB;
				this._glPixelType = gl.UNSIGNED_BYTE;
				break;
			case PIXELFORMAT_RGBA8:
				this._glFormat = gl.RGBA;
				this._glInternalFormat = device.isWebGL2 ? gl.RGBA8 : gl.RGBA;
				this._glPixelType = gl.UNSIGNED_BYTE;
				break;
			case PIXELFORMAT_DXT1:
				this._glFormat = gl.RGB;
				this._glInternalFormat = device.extCompressedTextureS3TC.COMPRESSED_RGB_S3TC_DXT1_EXT;
				break;
			case PIXELFORMAT_DXT3:
				this._glFormat = gl.RGBA;
				this._glInternalFormat = device.extCompressedTextureS3TC.COMPRESSED_RGBA_S3TC_DXT3_EXT;
				break;
			case PIXELFORMAT_DXT5:
				this._glFormat = gl.RGBA;
				this._glInternalFormat = device.extCompressedTextureS3TC.COMPRESSED_RGBA_S3TC_DXT5_EXT;
				break;
			case PIXELFORMAT_ETC1:
				this._glFormat = gl.RGB;
				this._glInternalFormat = device.extCompressedTextureETC1.COMPRESSED_RGB_ETC1_WEBGL;
				break;
			case PIXELFORMAT_PVRTC_2BPP_RGB_1:
				this._glFormat = gl.RGB;
				this._glInternalFormat = device.extCompressedTexturePVRTC.COMPRESSED_RGB_PVRTC_2BPPV1_IMG;
				break;
			case PIXELFORMAT_PVRTC_2BPP_RGBA_1:
				this._glFormat = gl.RGBA;
				this._glInternalFormat = device.extCompressedTexturePVRTC.COMPRESSED_RGBA_PVRTC_2BPPV1_IMG;
				break;
			case PIXELFORMAT_PVRTC_4BPP_RGB_1:
				this._glFormat = gl.RGB;
				this._glInternalFormat = device.extCompressedTexturePVRTC.COMPRESSED_RGB_PVRTC_4BPPV1_IMG;
				break;
			case PIXELFORMAT_PVRTC_4BPP_RGBA_1:
				this._glFormat = gl.RGBA;
				this._glInternalFormat = device.extCompressedTexturePVRTC.COMPRESSED_RGBA_PVRTC_4BPPV1_IMG;
				break;
			case PIXELFORMAT_ETC2_RGB:
				this._glFormat = gl.RGB;
				this._glInternalFormat = device.extCompressedTextureETC.COMPRESSED_RGB8_ETC2;
				break;
			case PIXELFORMAT_ETC2_RGBA:
				this._glFormat = gl.RGBA;
				this._glInternalFormat = device.extCompressedTextureETC.COMPRESSED_RGBA8_ETC2_EAC;
				break;
			case PIXELFORMAT_ASTC_4x4:
				this._glFormat = gl.RGBA;
				this._glInternalFormat = device.extCompressedTextureASTC.COMPRESSED_RGBA_ASTC_4x4_KHR;
				break;
			case PIXELFORMAT_ATC_RGB:
				this._glFormat = gl.RGB;
				this._glInternalFormat = device.extCompressedTextureATC.COMPRESSED_RGB_ATC_WEBGL;
				break;
			case PIXELFORMAT_ATC_RGBA:
				this._glFormat = gl.RGBA;
				this._glInternalFormat = device.extCompressedTextureATC.COMPRESSED_RGBA_ATC_INTERPOLATED_ALPHA_WEBGL;
				break;
			case PIXELFORMAT_R16F:
				if (device.isWebGL2) {
					this._glFormat = gl.RED;
					this._glInternalFormat = gl.R16F;
					this._glPixelType = gl.HALF_FLOAT;
				} else {
					this._glFormat = gl.LUMINANCE;
					this._glInternalFormat = gl.LUMINANCE;
					this._glPixelType = device.extTextureHalfFloat.HALF_FLOAT_OES;
				}
				break;
			case PIXELFORMAT_RG16F:
				if (device.isWebGL2) {
					this._glFormat = gl.RG;
					this._glInternalFormat = gl.RG16F;
					this._glPixelType = gl.HALF_FLOAT;
				} else {
					this._glFormat = gl.RG;
					this._glInternalFormat = gl.RG;
					this._glPixelType = device.extTextureHalfFloat.HALF_FLOAT_OES;
				}
				break;
			case PIXELFORMAT_RGB16F:
				this._glFormat = gl.RGB;
				if (device.isWebGL2) {
					this._glInternalFormat = gl.RGB16F;
					this._glPixelType = gl.HALF_FLOAT;
				} else {
					this._glInternalFormat = gl.RGB;
					this._glPixelType = device.extTextureHalfFloat.HALF_FLOAT_OES;
				}
				break;
			case PIXELFORMAT_RGBA16F:
				this._glFormat = gl.RGBA;
				if (device.isWebGL2) {
					this._glInternalFormat = gl.RGBA16F;
					this._glPixelType = gl.HALF_FLOAT;
				} else {
					this._glInternalFormat = gl.RGBA;
					this._glPixelType = device.extTextureHalfFloat.HALF_FLOAT_OES;
				}
				break;
			case PIXELFORMAT_RGB32F:
				this._glFormat = gl.RGB;
				if (device.isWebGL2) {
					this._glInternalFormat = gl.RGB32F;
				} else {
					this._glInternalFormat = gl.RGB;
				}
				this._glPixelType = gl.FLOAT;
				break;
			case PIXELFORMAT_RGBA32F:
				this._glFormat = gl.RGBA;
				if (device.isWebGL2) {
					this._glInternalFormat = gl.RGBA32F;
				} else {
					this._glInternalFormat = gl.RGBA;
				}
				this._glPixelType = gl.FLOAT;
				break;
			case PIXELFORMAT_R32F:
				this._glFormat = gl.RED;
				this._glInternalFormat = gl.R32F;
				this._glPixelType = gl.FLOAT;
				break;
			case PIXELFORMAT_DEPTH:
				if (device.isWebGL2) {
					this._glFormat = gl.DEPTH_COMPONENT;
					this._glInternalFormat = gl.DEPTH_COMPONENT32F;
					this._glPixelType = gl.FLOAT;
				} else {
					this._glFormat = gl.DEPTH_COMPONENT;
					this._glInternalFormat = gl.DEPTH_COMPONENT;
					this._glPixelType = gl.UNSIGNED_SHORT;
				}
				break;
			case PIXELFORMAT_DEPTHSTENCIL:
				this._glFormat = gl.DEPTH_STENCIL;
				if (device.isWebGL2) {
					this._glInternalFormat = gl.DEPTH24_STENCIL8;
					this._glPixelType = gl.UNSIGNED_INT_24_8;
				} else {
					this._glInternalFormat = gl.DEPTH_STENCIL;
					this._glPixelType = device.extDepthTexture.UNSIGNED_INT_24_8_WEBGL;
				}
				break;
			case PIXELFORMAT_111110F:
				this._glFormat = gl.RGB;
				this._glInternalFormat = gl.R11F_G11F_B10F;
				this._glPixelType = gl.UNSIGNED_INT_10F_11F_11F_REV;
				break;
			case PIXELFORMAT_SRGB:
				this._glFormat = gl.RGB;
				this._glInternalFormat = gl.SRGB8;
				this._glPixelType = gl.UNSIGNED_BYTE;
				break;
			case PIXELFORMAT_SRGBA:
				this._glFormat = gl.RGBA;
				this._glInternalFormat = gl.SRGB8_ALPHA8;
				this._glPixelType = gl.UNSIGNED_BYTE;
				break;
			case PIXELFORMAT_R8I:
				this._glFormat = gl.RED_INTEGER;
				this._glInternalFormat = gl.R8I;
				this._glPixelType = gl.BYTE;
				break;
			case PIXELFORMAT_R8U:
				this._glFormat = gl.RED_INTEGER;
				this._glInternalFormat = gl.R8UI;
				this._glPixelType = gl.UNSIGNED_BYTE;
				break;
			case PIXELFORMAT_R16I:
				this._glFormat = gl.RED_INTEGER;
				this._glInternalFormat = gl.R16I;
				this._glPixelType = gl.SHORT;
				break;
			case PIXELFORMAT_R16U:
				this._glFormat = gl.RED_INTEGER;
				this._glInternalFormat = gl.R16UI;
				this._glPixelType = gl.UNSIGNED_SHORT;
				break;
			case PIXELFORMAT_R32I:
				this._glFormat = gl.RED_INTEGER;
				this._glInternalFormat = gl.R32I;
				this._glPixelType = gl.INT;
				break;
			case PIXELFORMAT_R32U:
				this._glFormat = gl.RED_INTEGER;
				this._glInternalFormat = gl.R32UI;
				this._glPixelType = gl.UNSIGNED_INT;
				break;
			case PIXELFORMAT_RG8I:
				this._glFormat = gl.RG_INTEGER;
				this._glInternalFormat = gl.RG8I;
				this._glPixelType = gl.BYTE;
				break;
			case PIXELFORMAT_RG8U:
				this._glFormat = gl.RG_INTEGER;
				this._glInternalFormat = gl.RG8UI;
				this._glPixelType = gl.UNSIGNED_BYTE;
				break;
			case PIXELFORMAT_RG16I:
				this._glFormat = gl.RG_INTEGER;
				this._glInternalFormat = gl.RG16I;
				this._glPixelType = gl.SHORT;
				break;
			case PIXELFORMAT_RG16U:
				this._glFormat = gl.RG_INTEGER;
				this._glInternalFormat = gl.RG16UI;
				this._glPixelType = gl.UNSIGNED_SHORT;
				break;
			case PIXELFORMAT_RG32I:
				this._glFormat = gl.RG_INTEGER;
				this._glInternalFormat = gl.RG32I;
				this._glPixelType = gl.INT;
				break;
			case PIXELFORMAT_RG32U:
				this._glFormat = gl.RG_INTEGER;
				this._glInternalFormat = gl.RG32UI;
				this._glPixelType = gl.UNSIGNED_INT;
				break;
			case PIXELFORMAT_RGBA8I:
				this._glFormat = gl.RGBA_INTEGER;
				this._glInternalFormat = gl.RGBA8I;
				this._glPixelType = gl.BYTE;
				break;
			case PIXELFORMAT_RGBA8U:
				this._glFormat = gl.RGBA_INTEGER;
				this._glInternalFormat = gl.RGBA8UI;
				this._glPixelType = gl.UNSIGNED_BYTE;
				break;
			case PIXELFORMAT_RGBA16I:
				this._glFormat = gl.RGBA_INTEGER;
				this._glInternalFormat = gl.RGBA16I;
				this._glPixelType = gl.SHORT;
				break;
			case PIXELFORMAT_RGBA16U:
				this._glFormat = gl.RGBA_INTEGER;
				this._glInternalFormat = gl.RGBA16UI;
				this._glPixelType = gl.UNSIGNED_SHORT;
				break;
			case PIXELFORMAT_RGBA32I:
				this._glFormat = gl.RGBA_INTEGER;
				this._glInternalFormat = gl.RGBA32I;
				this._glPixelType = gl.INT;
				break;
			case PIXELFORMAT_RGBA32U:
				this._glFormat = gl.RGBA_INTEGER;
				this._glInternalFormat = gl.RGBA32UI;
				this._glPixelType = gl.UNSIGNED_INT;
				break;
		}
		this._glCreated = false;
	}
	upload(device, texture) {
		const gl = device.gl;
		if (!texture._needsUpload && (texture._needsMipmapsUpload && texture._mipmapsUploaded || !texture.pot)) return;
		let mipLevel = 0;
		let mipObject;
		let resMult;
		const requiredMipLevels = texture.requiredMipLevels;
		if (texture.array) {
			gl.texStorage3D(gl.TEXTURE_2D_ARRAY, requiredMipLevels, this._glInternalFormat, texture._width, texture._height, texture._arrayLength);
		}
		while (texture._levels[mipLevel] || mipLevel === 0) {
			if (!texture._needsUpload && mipLevel === 0) {
				mipLevel++;
				continue;
			} else if (mipLevel && (!texture._needsMipmapsUpload || !texture._mipmaps)) {
				break;
			}
			mipObject = texture._levels[mipLevel];
			resMult = 1 / Math.pow(2, mipLevel);
			if (mipLevel === 1 && !texture._compressed && !texture._integerFormat && texture._levels.length < requiredMipLevels) {
				gl.generateMipmap(this._glTarget);
				texture._mipmapsUploaded = true;
			}
			if (texture._cubemap) {
				let face;
				if (device._isBrowserInterface(mipObject[0])) {
					for (face = 0; face < 6; face++) {
						if (!texture._levelsUpdated[0][face]) continue;
						let src = mipObject[face];
						if (device._isImageBrowserInterface(src)) {
							if (src.width > device.maxCubeMapSize || src.height > device.maxCubeMapSize) {
								src = downsampleImage(src, device.maxCubeMapSize);
								if (mipLevel === 0) {
									texture._width = src.width;
									texture._height = src.height;
								}
							}
						}
						device.setUnpackFlipY(false);
						device.setUnpackPremultiplyAlpha(texture._premultiplyAlpha);
						if (this._glCreated) {
							gl.texSubImage2D(gl.TEXTURE_CUBE_MAP_POSITIVE_X + face, mipLevel, 0, 0, this._glFormat, this._glPixelType, src);
						} else {
							gl.texImage2D(gl.TEXTURE_CUBE_MAP_POSITIVE_X + face, mipLevel, this._glInternalFormat, this._glFormat, this._glPixelType, src);
						}
					}
				} else {
					resMult = 1 / Math.pow(2, mipLevel);
					for (face = 0; face < 6; face++) {
						if (!texture._levelsUpdated[0][face]) continue;
						const texData = mipObject[face];
						if (texture._compressed) {
							if (this._glCreated && texData) {
								gl.compressedTexSubImage2D(gl.TEXTURE_CUBE_MAP_POSITIVE_X + face, mipLevel, 0, 0, Math.max(texture._width * resMult, 1), Math.max(texture._height * resMult, 1), this._glInternalFormat, texData);
							} else {
								gl.compressedTexImage2D(gl.TEXTURE_CUBE_MAP_POSITIVE_X + face, mipLevel, this._glInternalFormat, Math.max(texture._width * resMult, 1), Math.max(texture._height * resMult, 1), 0, texData);
							}
						} else {
							device.setUnpackFlipY(false);
							device.setUnpackPremultiplyAlpha(texture._premultiplyAlpha);
							if (this._glCreated && texData) {
								gl.texSubImage2D(gl.TEXTURE_CUBE_MAP_POSITIVE_X + face, mipLevel, 0, 0, Math.max(texture._width * resMult, 1), Math.max(texture._height * resMult, 1), this._glFormat, this._glPixelType, texData);
							} else {
								gl.texImage2D(gl.TEXTURE_CUBE_MAP_POSITIVE_X + face, mipLevel, this._glInternalFormat, Math.max(texture._width * resMult, 1), Math.max(texture._height * resMult, 1), 0, this._glFormat, this._glPixelType, texData);
							}
						}
					}
				}
			} else if (texture._volume) {
				if (texture._compressed) {
					gl.compressedTexImage3D(gl.TEXTURE_3D, mipLevel, this._glInternalFormat, Math.max(texture._width * resMult, 1), Math.max(texture._height * resMult, 1), Math.max(texture._depth * resMult, 1), 0, mipObject);
				} else {
					device.setUnpackFlipY(false);
					device.setUnpackPremultiplyAlpha(texture._premultiplyAlpha);
					gl.texImage3D(gl.TEXTURE_3D, mipLevel, this._glInternalFormat, Math.max(texture._width * resMult, 1), Math.max(texture._height * resMult, 1), Math.max(texture._depth * resMult, 1), 0, this._glFormat, this._glPixelType, mipObject);
				}
			} else if (texture.array && typeof mipObject === "object") {
				if (texture._arrayLength === mipObject.length) {
					if (texture._compressed) {
						for (let index = 0; index < texture._arrayLength; index++) {
							gl.compressedTexSubImage3D(gl.TEXTURE_2D_ARRAY, mipLevel, 0, 0, index, Math.max(Math.floor(texture._width * resMult), 1), Math.max(Math.floor(texture._height * resMult), 1), 1, this._glFormat, mipObject[index]);
						}
					} else {
						for (let index = 0; index < texture._arrayLength; index++) {
							gl.texSubImage3D(gl.TEXTURE_2D_ARRAY, mipLevel, 0, 0, index, Math.max(Math.floor(texture._width * resMult), 1), Math.max(Math.floor(texture._height * resMult), 1), 1, this._glFormat, this._glPixelType, mipObject[index]);
						}
					}
				}
			} else {
				if (device._isBrowserInterface(mipObject)) {
					if (device._isImageBrowserInterface(mipObject)) {
						if (mipObject.width > device.maxTextureSize || mipObject.height > device.maxTextureSize) {
							mipObject = downsampleImage(mipObject, device.maxTextureSize);
							if (mipLevel === 0) {
								texture._width = mipObject.width;
								texture._height = mipObject.height;
							}
						}
					}
					const w = mipObject.width || mipObject.videoWidth;
					const h = mipObject.height || mipObject.videoHeight;
					device.setUnpackFlipY(texture._flipY);
					device.setUnpackPremultiplyAlpha(texture._premultiplyAlpha);
					if (this._glCreated && texture._width === w && texture._height === h && !device._isImageVideoInterface(mipObject)) {
						gl.texSubImage2D(gl.TEXTURE_2D, mipLevel, 0, 0, this._glFormat, this._glPixelType, mipObject);
					} else {
						gl.texImage2D(gl.TEXTURE_2D, mipLevel, this._glInternalFormat, this._glFormat, this._glPixelType, mipObject);
						if (mipLevel === 0) {
							texture._width = w;
							texture._height = h;
						}
					}
				} else {
					resMult = 1 / Math.pow(2, mipLevel);
					if (texture._compressed) {
						if (this._glCreated && mipObject) {
							gl.compressedTexSubImage2D(gl.TEXTURE_2D, mipLevel, 0, 0, Math.max(Math.floor(texture._width * resMult), 1), Math.max(Math.floor(texture._height * resMult), 1), this._glInternalFormat, mipObject);
						} else {
							gl.compressedTexImage2D(gl.TEXTURE_2D, mipLevel, this._glInternalFormat, Math.max(Math.floor(texture._width * resMult), 1), Math.max(Math.floor(texture._height * resMult), 1), 0, mipObject);
						}
					} else {
						device.setUnpackFlipY(false);
						device.setUnpackPremultiplyAlpha(texture._premultiplyAlpha);
						if (this._glCreated && mipObject) {
							gl.texSubImage2D(gl.TEXTURE_2D, mipLevel, 0, 0, Math.max(texture._width * resMult, 1), Math.max(texture._height * resMult, 1), this._glFormat, this._glPixelType, mipObject);
						} else {
							gl.texImage2D(gl.TEXTURE_2D, mipLevel, this._glInternalFormat, Math.max(texture._width * resMult, 1), Math.max(texture._height * resMult, 1), 0, this._glFormat, this._glPixelType, mipObject);
						}
					}
				}
				if (mipLevel === 0) {
					texture._mipmapsUploaded = false;
				} else {
					texture._mipmapsUploaded = true;
				}
			}
			mipLevel++;
		}
		if (texture._needsUpload) {
			if (texture._cubemap) {
				for (let i = 0; i < 6; i++) texture._levelsUpdated[0][i] = false;
			} else {
				texture._levelsUpdated[0] = false;
			}
		}
		if (!texture._compressed && !texture._integerFormat && texture._mipmaps && texture._needsMipmapsUpload && (texture.pot || device.isWebGL2) && texture._levels.length === 1) {
			gl.generateMipmap(this._glTarget);
			texture._mipmapsUploaded = true;
		}
		if (texture._gpuSize) {
			texture.adjustVramSizeTracking(device._vram, -texture._gpuSize);
		}
		texture._gpuSize = texture.gpuSize;
		texture.adjustVramSizeTracking(device._vram, texture._gpuSize);
		this._glCreated = true;
	}
}

class FramebufferPair {
	constructor(msaaFB, resolveFB) {
		this.msaaFB = void 0;
		this.resolveFB = void 0;
		this.msaaFB = msaaFB;
		this.resolveFB = resolveFB;
	}
	destroy(gl) {
		if (this.msaaFB) {
			gl.deleteRenderbuffer(this.msaaFB);
			this.msaaFB = null;
		}
		if (this.resolveFB) {
			gl.deleteRenderbuffer(this.resolveFB);
			this.resolveFB = null;
		}
	}
}
class WebglRenderTarget {
	constructor() {
		this._glFrameBuffer = null;
		this._glDepthBuffer = null;
		this._glResolveFrameBuffer = null;
		this.colorMrtFramebuffers = null;
		this._glMsaaColorBuffers = [];
		this._glMsaaDepthBuffer = null;
		this.suppliedColorFramebuffer = void 0;
		this._isInitialized = false;
	}
	destroy(device) {
		var _this$colorMrtFramebu;
		const gl = device.gl;
		this._isInitialized = false;
		if (this._glFrameBuffer) {
			if (this._glFrameBuffer !== this.suppliedColorFramebuffer) gl.deleteFramebuffer(this._glFrameBuffer);
			this._glFrameBuffer = null;
		}
		if (this._glDepthBuffer) {
			gl.deleteRenderbuffer(this._glDepthBuffer);
			this._glDepthBuffer = null;
		}
		if (this._glResolveFrameBuffer) {
			if (this._glResolveFrameBuffer !== this.suppliedColorFramebuffer) gl.deleteFramebuffer(this._glResolveFrameBuffer);
			this._glResolveFrameBuffer = null;
		}
		this._glMsaaColorBuffers.forEach(buffer => {
			gl.deleteRenderbuffer(buffer);
		});
		this._glMsaaColorBuffers.length = 0;
		(_this$colorMrtFramebu = this.colorMrtFramebuffers) == null || _this$colorMrtFramebu.forEach(framebuffer => {
			framebuffer.destroy(gl);
		});
		this.colorMrtFramebuffers = null;
		if (this._glMsaaDepthBuffer) {
			gl.deleteRenderbuffer(this._glMsaaDepthBuffer);
			this._glMsaaDepthBuffer = null;
		}
		this.suppliedColorFramebuffer = undefined;
	}
	get initialized() {
		return this._isInitialized;
	}
	init(device, target) {
		const gl = device.gl;
		this._isInitialized = true;
		const buffers = [];
		if (this.suppliedColorFramebuffer !== undefined) {
			this._glFrameBuffer = this.suppliedColorFramebuffer;
		} else {
			var _target$_colorBuffers, _target$_colorBuffers2, _device$extDrawBuffer, _device$extDrawBuffer2;
			this._glFrameBuffer = gl.createFramebuffer();
			device.setFramebuffer(this._glFrameBuffer);
			const colorBufferCount = (_target$_colorBuffers = (_target$_colorBuffers2 = target._colorBuffers) == null ? void 0 : _target$_colorBuffers2.length) != null ? _target$_colorBuffers : 0;
			const attachmentBaseConstant = device.isWebGL2 ? gl.COLOR_ATTACHMENT0 : (_device$extDrawBuffer = (_device$extDrawBuffer2 = device.extDrawBuffers) == null ? void 0 : _device$extDrawBuffer2.COLOR_ATTACHMENT0_WEBGL) != null ? _device$extDrawBuffer : gl.COLOR_ATTACHMENT0;
			for (let i = 0; i < colorBufferCount; ++i) {
				const colorBuffer = target.getColorBuffer(i);
				if (colorBuffer) {
					if (!colorBuffer.impl._glTexture) {
						colorBuffer._width = Math.min(colorBuffer.width, device.maxRenderBufferSize);
						colorBuffer._height = Math.min(colorBuffer.height, device.maxRenderBufferSize);
						device.setTexture(colorBuffer, 0);
					}
					gl.framebufferTexture2D(gl.FRAMEBUFFER, attachmentBaseConstant + i, colorBuffer._cubemap ? gl.TEXTURE_CUBE_MAP_POSITIVE_X + target._face : gl.TEXTURE_2D, colorBuffer.impl._glTexture, 0);
					buffers.push(attachmentBaseConstant + i);
				}
			}
			if (device.drawBuffers) {
				device.drawBuffers(buffers);
			}
			const depthBuffer = target._depthBuffer;
			if (depthBuffer) {
				if (!depthBuffer.impl._glTexture) {
					depthBuffer._width = Math.min(depthBuffer.width, device.maxRenderBufferSize);
					depthBuffer._height = Math.min(depthBuffer.height, device.maxRenderBufferSize);
					device.setTexture(depthBuffer, 0);
				}
				if (target._stencil) {
					gl.framebufferTexture2D(gl.FRAMEBUFFER, gl.DEPTH_STENCIL_ATTACHMENT, depthBuffer._cubemap ? gl.TEXTURE_CUBE_MAP_POSITIVE_X + target._face : gl.TEXTURE_2D, target._depthBuffer.impl._glTexture, 0);
				} else {
					gl.framebufferTexture2D(gl.FRAMEBUFFER, gl.DEPTH_ATTACHMENT, depthBuffer._cubemap ? gl.TEXTURE_CUBE_MAP_POSITIVE_X + target._face : gl.TEXTURE_2D, target._depthBuffer.impl._glTexture, 0);
				}
			} else if (target._depth) {
				const willRenderMsaa = target._samples > 1 && device.isWebGL2;
				if (!willRenderMsaa) {
					if (!this._glDepthBuffer) {
						this._glDepthBuffer = gl.createRenderbuffer();
					}
					gl.bindRenderbuffer(gl.RENDERBUFFER, this._glDepthBuffer);
					if (target._stencil) {
						gl.renderbufferStorage(gl.RENDERBUFFER, gl.DEPTH_STENCIL, target.width, target.height);
						gl.framebufferRenderbuffer(gl.FRAMEBUFFER, gl.DEPTH_STENCIL_ATTACHMENT, gl.RENDERBUFFER, this._glDepthBuffer);
					} else {
						const depthFormat = device.isWebGL2 ? gl.DEPTH_COMPONENT32F : gl.DEPTH_COMPONENT16;
						gl.renderbufferStorage(gl.RENDERBUFFER, depthFormat, target.width, target.height);
						gl.framebufferRenderbuffer(gl.FRAMEBUFFER, gl.DEPTH_ATTACHMENT, gl.RENDERBUFFER, this._glDepthBuffer);
					}
					gl.bindRenderbuffer(gl.RENDERBUFFER, null);
				}
			}
		}
		if (device.isWebGL2 && target._samples > 1) {
			var _target$_colorBuffers3, _target$_colorBuffers4;
			this._glResolveFrameBuffer = this._glFrameBuffer;
			this._glFrameBuffer = gl.createFramebuffer();
			device.setFramebuffer(this._glFrameBuffer);
			const colorBufferCount = (_target$_colorBuffers3 = (_target$_colorBuffers4 = target._colorBuffers) == null ? void 0 : _target$_colorBuffers4.length) != null ? _target$_colorBuffers3 : 0;
			if (this.suppliedColorFramebuffer !== undefined) {
				const buffer = gl.createRenderbuffer();
				this._glMsaaColorBuffers.push(buffer);
				const internalFormat = device.backBufferFormat === PIXELFORMAT_RGBA8 ? gl.RGBA8 : gl.RGB8;
				gl.bindRenderbuffer(gl.RENDERBUFFER, buffer);
				gl.renderbufferStorageMultisample(gl.RENDERBUFFER, target._samples, internalFormat, target.width, target.height);
				gl.framebufferRenderbuffer(gl.FRAMEBUFFER, gl.COLOR_ATTACHMENT0, gl.RENDERBUFFER, buffer);
			} else {
				for (let i = 0; i < colorBufferCount; ++i) {
					const colorBuffer = target.getColorBuffer(i);
					if (colorBuffer) {
						const buffer = gl.createRenderbuffer();
						this._glMsaaColorBuffers.push(buffer);
						gl.bindRenderbuffer(gl.RENDERBUFFER, buffer);
						gl.renderbufferStorageMultisample(gl.RENDERBUFFER, target._samples, colorBuffer.impl._glInternalFormat, target.width, target.height);
						gl.framebufferRenderbuffer(gl.FRAMEBUFFER, gl.COLOR_ATTACHMENT0 + i, gl.RENDERBUFFER, buffer);
					}
				}
			}
			if (target._depth) {
				if (!this._glMsaaDepthBuffer) {
					this._glMsaaDepthBuffer = gl.createRenderbuffer();
				}
				gl.bindRenderbuffer(gl.RENDERBUFFER, this._glMsaaDepthBuffer);
				if (target._stencil) {
					gl.renderbufferStorageMultisample(gl.RENDERBUFFER, target._samples, gl.DEPTH24_STENCIL8, target.width, target.height);
					gl.framebufferRenderbuffer(gl.FRAMEBUFFER, gl.DEPTH_STENCIL_ATTACHMENT, gl.RENDERBUFFER, this._glMsaaDepthBuffer);
				} else {
					gl.renderbufferStorageMultisample(gl.RENDERBUFFER, target._samples, gl.DEPTH_COMPONENT32F, target.width, target.height);
					gl.framebufferRenderbuffer(gl.FRAMEBUFFER, gl.DEPTH_ATTACHMENT, gl.RENDERBUFFER, this._glMsaaDepthBuffer);
				}
			}
			if (colorBufferCount > 1) {
				this._createMsaaMrtFramebuffers(device, target, colorBufferCount);
				device.setFramebuffer(this._glFrameBuffer);
				device.drawBuffers(buffers);
			}
		}
	}
	_createMsaaMrtFramebuffers(device, target, colorBufferCount) {
		const gl = device.gl;
		this.colorMrtFramebuffers = [];
		for (let i = 0; i < colorBufferCount; ++i) {
			const colorBuffer = target.getColorBuffer(i);
			const srcFramebuffer = gl.createFramebuffer();
			device.setFramebuffer(srcFramebuffer);
			const buffer = this._glMsaaColorBuffers[i];
			gl.bindRenderbuffer(gl.RENDERBUFFER, buffer);
			gl.renderbufferStorageMultisample(gl.RENDERBUFFER, target._samples, colorBuffer.impl._glInternalFormat, target.width, target.height);
			gl.framebufferRenderbuffer(gl.FRAMEBUFFER, gl.COLOR_ATTACHMENT0, gl.RENDERBUFFER, buffer);
			device.drawBuffers([gl.COLOR_ATTACHMENT0]);
			const dstFramebuffer = gl.createFramebuffer();
			device.setFramebuffer(dstFramebuffer);
			gl.framebufferTexture2D(gl.FRAMEBUFFER, gl.COLOR_ATTACHMENT0, colorBuffer._cubemap ? gl.TEXTURE_CUBE_MAP_POSITIVE_X + target._face : gl.TEXTURE_2D, colorBuffer.impl._glTexture, 0);
			this.colorMrtFramebuffers[i] = new FramebufferPair(srcFramebuffer, dstFramebuffer);
		}
	}
	_checkFbo(device, target, type = '') {
		const gl = device.gl;
		const status = gl.checkFramebufferStatus(gl.FRAMEBUFFER);
		switch (status) {
			case gl.FRAMEBUFFER_INCOMPLETE_ATTACHMENT:
				break;
			case gl.FRAMEBUFFER_INCOMPLETE_MISSING_ATTACHMENT:
				break;
			case gl.FRAMEBUFFER_INCOMPLETE_DIMENSIONS:
				break;
			case gl.FRAMEBUFFER_UNSUPPORTED:
				break;
		}
	}
	loseContext() {
		this._glFrameBuffer = null;
		this._glDepthBuffer = null;
		this._glResolveFrameBuffer = null;
		this._glMsaaColorBuffers.length = 0;
		this._glMsaaDepthBuffer = null;
		this.colorMrtFramebuffers = null;
		this.suppliedColorFramebuffer = undefined;
		this._isInitialized = false;
	}
	internalResolve(device, src, dst, target, mask) {
		device.setScissor(0, 0, target.width, target.height);
		const gl = device.gl;
		gl.bindFramebuffer(gl.READ_FRAMEBUFFER, src);
		gl.bindFramebuffer(gl.DRAW_FRAMEBUFFER, dst);
		gl.blitFramebuffer(0, 0, target.width, target.height, 0, 0, target.width, target.height, mask, gl.NEAREST);
	}
	resolve(device, target, color, depth) {
		if (device.isWebGL2) {
			const gl = device.gl;
			if (this.colorMrtFramebuffers) {
				if (color) {
					for (let i = 0; i < this.colorMrtFramebuffers.length; i++) {
						const fbPair = this.colorMrtFramebuffers[i];
						this.internalResolve(device, fbPair.msaaFB, fbPair.resolveFB, target, gl.COLOR_BUFFER_BIT);
					}
				}
				if (depth) {
					this.internalResolve(device, this._glFrameBuffer, this._glResolveFrameBuffer, target, gl.DEPTH_BUFFER_BIT);
				}
			} else {
				this.internalResolve(device, this._glFrameBuffer, this._glResolveFrameBuffer, target, (color ? gl.COLOR_BUFFER_BIT : 0) | (depth ? gl.DEPTH_BUFFER_BIT : 0));
			}
			gl.bindFramebuffer(gl.FRAMEBUFFER, this._glFrameBuffer);
		}
	}
}

var gles2PS = `
#define pcFragColor0 gl_FragData[0]
#if COLOR_ATTACHMENT_1
#define pcFragColor1 gl_FragData[1]
#endif
#if COLOR_ATTACHMENT_2
#define pcFragColor2 gl_FragData[2]
#endif
#if COLOR_ATTACHMENT_3
#define pcFragColor3 gl_FragData[3]
#endif
#if COLOR_ATTACHMENT_4
#define pcFragColor4 gl_FragData[4]
#endif
#if COLOR_ATTACHMENT_5
#define pcFragColor5 gl_FragData[5]
#endif
#if COLOR_ATTACHMENT_6
#define pcFragColor6 gl_FragData[6]
#endif
#if COLOR_ATTACHMENT_7
#define pcFragColor7 gl_FragData[7]
#endif
#define texture2DBias texture2D
#define itexture2D texture2D
#define utexture2D texture2D
#define SHADOWMAP_PASS(name) name
#define SHADOWMAP_ACCEPT(name) sampler2D name
#define TEXTURE_PASS(name) name
#define TEXTURE_ACCEPT(name) sampler2D name
#ifndef SUPPORTS_TEXLOD
	#define texture2DLodEXT texture2D
	#define texture2DProjLodEXT textureProj
	#define textureCubeLodEXT textureCube
	#define textureShadow texture2D
#else
	#define textureShadow(res, uv) texture2DGradEXT(res, uv, vec2(1, 1), vec2(1, 1))
#endif
#ifdef SUPPORTS_MRT
	#define gl_FragColor pcFragColor0
#endif
`;

var gles3PS = `
#ifndef outType_0
#define outType_0 vec4
#endif
layout(location = 0) out highp outType_0 pc_fragColor;
#ifndef REMOVE_COLOR_ATTACHMENT_1
#if COLOR_ATTACHMENT_1
layout(location = 1) out highp outType_1 pc_fragColor1;
#endif
#endif
#ifndef REMOVE_COLOR_ATTACHMENT_2
#if COLOR_ATTACHMENT_2
layout(location = 2) out highp outType_2 pc_fragColor2;
#endif
#endif
#ifndef REMOVE_COLOR_ATTACHMENT_3
#if COLOR_ATTACHMENT_3
layout(location = 3) out highp outType_3 pc_fragColor3;
#endif
#endif
#ifndef REMOVE_COLOR_ATTACHMENT_4
#if COLOR_ATTACHMENT_4
layout(location = 4) out highp outType_4 pc_fragColor4;
#endif
#endif
#ifndef REMOVE_COLOR_ATTACHMENT_5
#if COLOR_ATTACHMENT_5
layout(location = 5) out highp outType_5 pc_fragColor5;
#endif
#endif
#ifndef REMOVE_COLOR_ATTACHMENT_6
#if COLOR_ATTACHMENT_6
layout(location = 6) out highp outType_6 pc_fragColor6;
#endif
#endif
#ifndef REMOVE_COLOR_ATTACHMENT_7
#if COLOR_ATTACHMENT_7
layout(location = 7) out highp outType_7 pc_fragColor7;
#endif
#endif
#define gl_FragColor pc_fragColor
#define pcFragColor0 pc_fragColor
#define pcFragColor1 pc_fragColor1
#define pcFragColor2 pc_fragColor2
#define pcFragColor3 pc_fragColor3
#define pcFragColor4 pc_fragColor4
#define pcFragColor5 pc_fragColor5
#define pcFragColor6 pc_fragColor6
#define pcFragColor7 pc_fragColor7
#define varying in
#define texture2D texture
#define texture2DBias texture
#define textureCube texture
#define texture2DProj textureProj
#define texture2DLodEXT textureLod
#define texture2DProjLodEXT textureProjLod
#define textureCubeLodEXT textureLod
#define texture2DGradEXT textureGrad
#define texture2DProjGradEXT textureProjGrad
#define textureCubeGradEXT textureGrad
#define utexture2D texture
#define itexture2D texture
#define textureShadow(res, uv) textureGrad(res, uv, vec2(1, 1), vec2(1, 1))
#define SHADOWMAP_PASS(name) name
#define SHADOWMAP_ACCEPT(name) sampler2DShadow name
#define TEXTURE_PASS(name) name
#define TEXTURE_ACCEPT(name) sampler2D name
#define GL2
#define SUPPORTS_TEXLOD
#define SUPPORTS_MRT
`;

var gles3VS = `
#define attribute in
#define varying out
#define texture2D texture
#define utexture2D texture
#define itexture2D texture
#define GL2
#define VERTEXSHADER
`;

var webgpuPS = `
#extension GL_EXT_samplerless_texture_functions : require
#ifndef outType_0
#define outType_0 vec4
#endif
#ifndef outType_1
#define outType_1 vec4
#endif
#ifndef outType_2
#define outType_2 vec4
#endif
#ifndef outType_3
#define outType_3 vec4
#endif
#ifndef outType_4
#define outType_4 vec4
#endif
#ifndef outType_5
#define outType_5 vec4
#endif
#ifndef outType_6
#define outType_6 vec4
#endif
#ifndef outType_7
#define outType_7 vec4
#endif
layout(location = 0) out highp outType_0 pc_fragColor;
layout(location = 1) out highp outType_1 pc_fragColor1;
layout(location = 2) out highp outType_2 pc_fragColor2;
layout(location = 3) out highp outType_3 pc_fragColor3;
layout(location = 4) out highp outType_4 pc_fragColor4;
layout(location = 5) out highp outType_5 pc_fragColor5;
layout(location = 6) out highp outType_6 pc_fragColor6;
layout(location = 7) out highp outType_7 pc_fragColor7;
#define gl_FragColor pc_fragColor
#define pcFragColor0 pc_fragColor
#define pcFragColor1 pc_fragColor1
#define pcFragColor2 pc_fragColor2
#define pcFragColor3 pc_fragColor3
#define pcFragColor4 pc_fragColor4
#define pcFragColor5 pc_fragColor5
#define pcFragColor6 pc_fragColor6
#define pcFragColor7 pc_fragColor7
#define texture2D(res, uv) texture(sampler2D(res, res ## _sampler), uv)
#define texture2DBias(res, uv, bias) texture(sampler2D(res, res ## _sampler), uv, bias)
#define texture2DLodEXT(res, uv, lod) textureLod(sampler2D(res, res ## _sampler), uv, lod)
#define textureCube(res, uv) texture(samplerCube(res, res ## _sampler), uv)
#define textureCubeLodEXT(res, uv, lod) textureLod(samplerCube(res, res ## _sampler), uv, lod)
#define textureShadow(res, uv) textureLod(sampler2DShadow(res, res ## _sampler), uv, 0.0)
#define itexture2D(res, uv) texture(isampler2D(res, res ## _sampler), uv)
#define utexture2D(res, uv) texture(usampler2D(res, res ## _sampler), uv)
#define SHADOWMAP_PASS(name) name, name ## _sampler
#define SHADOWMAP_ACCEPT(name) texture2D name, sampler name ## _sampler
#define TEXTURE_PASS(name) name, name ## _sampler
#define TEXTURE_ACCEPT(name) texture2D name, sampler name ## _sampler
#define GL2
#define WEBGPU
#define SUPPORTS_TEXLOD
#define SUPPORTS_MRT
`;

var webgpuVS = `
#extension GL_EXT_samplerless_texture_functions : require
#define texture2D(res, uv) texture(sampler2D(res, res ## _sampler), uv)
#define itexture2D(res, uv) texture(isampler2D(res, res ## _sampler), uv)
#define utexture2D(res, uv) texture(usampler2D(res, res ## _sampler), uv)
#define GL2
#define WEBGPU
#define VERTEXSHADER
`;

var sharedFS = `
vec2 getGrabScreenPos(vec4 clipPos) {
	vec2 uv = (clipPos.xy / clipPos.w) * 0.5 + 0.5;
	#ifdef WEBGPU
		uv.y = 1.0 - uv.y;
	#endif
	return uv;
}
vec2 getImageEffectUV(vec2 uv) {
	#ifdef WEBGPU
		uv.y = 1.0 - uv.y;
	#endif
	return uv;
}
`;

const _attrib2Semantic = {
	vertex_position: SEMANTIC_POSITION,
	vertex_normal: SEMANTIC_NORMAL,
	vertex_tangent: SEMANTIC_TANGENT,
	vertex_texCoord0: SEMANTIC_TEXCOORD0,
	vertex_texCoord1: SEMANTIC_TEXCOORD1,
	vertex_texCoord2: SEMANTIC_TEXCOORD2,
	vertex_texCoord3: SEMANTIC_TEXCOORD3,
	vertex_texCoord4: SEMANTIC_TEXCOORD4,
	vertex_texCoord5: SEMANTIC_TEXCOORD5,
	vertex_texCoord6: SEMANTIC_TEXCOORD6,
	vertex_texCoord7: SEMANTIC_TEXCOORD7,
	vertex_color: SEMANTIC_COLOR,
	vertex_boneIndices: SEMANTIC_BLENDINDICES,
	vertex_boneWeights: SEMANTIC_BLENDWEIGHT
};
class ShaderUtils {
	static createDefinition(device, options) {
		var _options$name, _options$attributes;
		const getDefines = (gpu, gl2, gl1, isVertex, options) => {
			const deviceIntro = device.isWebGPU ? gpu : device.isWebGL2 ? gl2 : ShaderUtils.gl1Extensions(device, options) + gl1;
			let attachmentsDefine = '';
			if (!isVertex) {
				var _options$fragmentOutp;
				let fragmentOutputTypes = (_options$fragmentOutp = options.fragmentOutputTypes) != null ? _options$fragmentOutp : 'vec4';
				if (!Array.isArray(fragmentOutputTypes)) {
					fragmentOutputTypes = [fragmentOutputTypes];
				}
				for (let i = 0; i < device.maxColorAttachments; i++) {
					var _fragmentOutputTypes$;
					attachmentsDefine += `#define COLOR_ATTACHMENT_${i}\n`;
					const outType = (_fragmentOutputTypes$ = fragmentOutputTypes[i]) != null ? _fragmentOutputTypes$ : 'vec4';
					attachmentsDefine += `#define outType_${i} ${outType}\n`;
				}
			}
			return attachmentsDefine + deviceIntro;
		};
		const name = (_options$name = options.name) != null ? _options$name : 'Untitled';
		const vertDefines = options.vertexDefines || getDefines(webgpuVS, gles3VS, '', true, options);
		const vertCode = ShaderUtils.versionCode(device) + vertDefines + sharedFS + ShaderUtils.getShaderNameCode(name) + options.vertexCode;
		const fragDefines = options.fragmentDefines || getDefines(webgpuPS, gles3PS, gles2PS, false, options);
		const fragCode = (options.fragmentPreamble || '') + ShaderUtils.versionCode(device) + fragDefines + ShaderUtils.precisionCode(device) + '\n' + sharedFS + ShaderUtils.getShaderNameCode(name) + (options.fragmentCode || ShaderUtils.dummyFragmentCode());
		const attribs = (_options$attributes = options.attributes) != null ? _options$attributes : ShaderUtils.collectAttributes(options.vertexCode);
		return {
			name: name,
			attributes: attribs,
			vshader: vertCode,
			fshader: fragCode,
			useTransformFeedback: options.useTransformFeedback
		};
	}
	static getShaderNameCode(name) {
		return `#define SHADER_NAME ${name}\n`;
	}
	static gl1Extensions(device, options, isVertex) {
		let code;
		if (isVertex) {
			code = options.vertexExtensions ? `${options.vertexExtensions}\n` : '';
		} else {
			code = options.fragmentExtensions ? `${options.fragmentExtensions}\n` : '';
			if (device.extStandardDerivatives) {
				code += "#extension GL_OES_standard_derivatives : enable\n";
			}
			if (device.extTextureLod) {
				code += "#extension GL_EXT_shader_texture_lod : enable\n";
				code += "#define SUPPORTS_TEXLOD\n";
			}
			if (device.extDrawBuffers) {
				code += "#extension GL_EXT_draw_buffers : require\n";
				code += "#define SUPPORTS_MRT\n";
			}
		}
		return code;
	}
	static dummyFragmentCode() {
		return "void main(void) {gl_FragColor = vec4(0.0);}";
	}
	static versionCode(device) {
		if (device.isWebGPU) {
			return '#version 450\n';
		}
		return device.isWebGL2 ? "#version 300 es\n" : "";
	}
	static precisionCode(device, forcePrecision) {
		let code = '';
		if (forcePrecision && forcePrecision !== 'highp' && forcePrecision !== 'mediump' && forcePrecision !== 'lowp') {
			forcePrecision = null;
		}
		if (forcePrecision) {
			if (forcePrecision === 'highp' && device.maxPrecision !== 'highp') {
				forcePrecision = 'mediump';
			}
			if (forcePrecision === 'mediump' && device.maxPrecision === 'lowp') {
				forcePrecision = 'lowp';
			}
		}
		const precision = forcePrecision ? forcePrecision : device.precision;
		if (!device.isWebGPU) {
			code = `precision ${precision} float;\n`;
			if (device.isWebGL2) {
				code += `precision ${precision} sampler2DShadow;\n`;
			}
		} else {
			code = `precision ${precision} float;\nprecision ${precision} int;\n`;
		}
		return code;
	}
	static collectAttributes(vsCode) {
		const attribs = {};
		let attrs = 0;
		let found = vsCode.indexOf("attribute");
		while (found >= 0) {
			if (found > 0 && vsCode[found - 1] === "/") break;
			const endOfLine = vsCode.indexOf(';', found);
			const startOfAttribName = vsCode.lastIndexOf(' ', endOfLine);
			const attribName = vsCode.substring(startOfAttribName + 1, endOfLine);
			const semantic = _attrib2Semantic[attribName];
			if (semantic !== undefined) {
				attribs[attribName] = semantic;
			} else {
				attribs[attribName] = "ATTR" + attrs;
				attrs++;
			}
			found = vsCode.indexOf("attribute", found + 1);
		}
		return attribs;
	}
}

class FrameQueriesInfo {
	constructor() {
		this.renderVersion = void 0;
		this.queries = [];
	}
	destroy(gl) {
		this.queries.forEach(query => gl.deleteQuery(query));
		this.queries = null;
	}
}
class WebglGpuProfiler extends GpuProfiler {
	constructor(device) {
		super();
		this.device = void 0;
		this.freeQueries = [];
		this.frameQueries = [];
		this.previousFrameQueries = [];
		this.timings = [];
		this.device = device;
		this.ext = device.extDisjointTimerQuery;
	}
	destroy() {
		this.freeQueries.forEach(query => this.device.gl.deleteQuery(query));
		this.frameQueries.forEach(query => this.device.gl.deleteQuery(query));
		this.previousFrameQueries.forEach(frameQueriesInfo => frameQueriesInfo.destroy(this.device.gl));
		this.freeQueries = null;
		this.frameQueries = null;
		this.previousFrameQueries = null;
	}
	loseContext() {
		super.loseContext();
		this.freeQueries = [];
		this.frameQueries = [];
		this.previousFrameQueries = [];
	}
	restoreContext() {
		this.ext = this.device.extDisjointTimerQuery;
	}
	getQuery() {
		var _this$freeQueries$pop;
		return (_this$freeQueries$pop = this.freeQueries.pop()) != null ? _this$freeQueries$pop : this.device.gl.createQuery();
	}
	start(name) {
		if (this.ext) {
			const slot = this.getSlot(name);
			const query = this.getQuery();
			this.frameQueries[slot] = query;
			this.device.gl.beginQuery(this.ext.TIME_ELAPSED_EXT, query);
			return slot;
		}
		return undefined;
	}
	end(slot) {
		if (slot !== undefined) {
			this.device.gl.endQuery(this.ext.TIME_ELAPSED_EXT);
		}
	}
	frameStart() {
		this.processEnableRequest();
		if (this._enabled) {
			this.frameGPUMarkerSlot = this.start('GpuFrame');
		}
	}
	frameEnd() {
		if (this._enabled) {
			this.end(this.frameGPUMarkerSlot);
		}
	}
	request() {
		if (this._enabled) {
			const ext = this.ext;
			const gl = this.device.gl;
			const renderVersion = this.device.renderVersion;
			const frameQueries = this.frameQueries;
			if (frameQueries.length > 0) {
				this.frameQueries = [];
				const frameQueriesInfo = new FrameQueriesInfo();
				frameQueriesInfo.queries = frameQueries;
				frameQueriesInfo.renderVersion = renderVersion;
				this.previousFrameQueries.push(frameQueriesInfo);
			}
			if (this.previousFrameQueries.length > 0) {
				const previousQueriesInfo = this.previousFrameQueries[0];
				const previousQueries = previousQueriesInfo.queries;
				const lastQuery = previousQueries[previousQueries.length - 1];
				const available = gl.getQueryParameter(lastQuery, gl.QUERY_RESULT_AVAILABLE);
				const disjoint = gl.getParameter(ext.GPU_DISJOINT_EXT);
				if (available && !disjoint) {
					this.previousFrameQueries.shift();
					const timings = this.timings;
					timings.length = 0;
					for (let i = 0; i < previousQueries.length; i++) {
						const query = previousQueries[i];
						const duration = gl.getQueryParameter(query, gl.QUERY_RESULT);
						timings[i] = duration * 0.000001;
						this.freeQueries.push(query);
					}
					this.report(previousQueriesInfo.renderVersion, timings);
				}
				if (disjoint) {
					this.previousFrameQueries.forEach(frameQueriesInfo => {
						this.report(frameQueriesInfo.renderVersion, null);
						frameQueriesInfo.destroy(gl);
					});
					this.previousFrameQueries.length = 0;
				}
			}
			super.request(renderVersion);
		}
	}
}

const invalidateAttachments = [];
const _fullScreenQuadVS = `
attribute vec2 vertex_position;
varying vec2 vUv0;
void main(void)
{
	gl_Position = vec4(vertex_position, 0.5, 1.0);
	vUv0 = vertex_position.xy*0.5+0.5;
}
`;
const _precisionTest1PS = `
void main(void) { 
	gl_FragColor = vec4(2147483648.0);
}
`;
const _precisionTest2PS = `
uniform sampler2D source;
vec4 packFloat(float depth) {
	const vec4 bit_shift = vec4(256.0 * 256.0 * 256.0, 256.0 * 256.0, 256.0, 1.0);
	const vec4 bit_mask  = vec4(0.0, 1.0 / 256.0, 1.0 / 256.0, 1.0 / 256.0);
	vec4 res = mod(depth * bit_shift * vec4(255), vec4(256) ) / vec4(255);
	res -= res.xxyz * bit_mask;
	return res;
}
void main(void) {
	float c = texture2D(source, vec2(0.0)).r;
	float diff = abs(c - 2147483648.0) / 2147483648.0;
	gl_FragColor = packFloat(diff);
}
`;
const _outputTexture2D = `
varying vec2 vUv0;
uniform sampler2D source;
void main(void) {
	gl_FragColor = texture2D(source, vUv0);
}
`;
function quadWithShader(device, target, shader) {
	const oldRt = device.renderTarget;
	device.setRenderTarget(target);
	device.updateBegin();
	device.setCullMode(CULLFACE_NONE);
	device.setBlendState(BlendState.NOBLEND);
	device.setDepthState(DepthState.NODEPTH);
	device.setStencilState(null, null);
	device.setVertexBuffer(device.quadVertexBuffer, 0);
	device.setShader(shader);
	device.draw({
		type: PRIMITIVE_TRISTRIP,
		base: 0,
		count: 4,
		indexed: false
	});
	device.updateEnd();
	device.setRenderTarget(oldRt);
	device.updateBegin();
}
function testRenderable(gl, pixelFormat) {
	let result = true;
	const texture = gl.createTexture();
	gl.bindTexture(gl.TEXTURE_2D, texture);
	gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MIN_FILTER, gl.NEAREST);
	gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MAG_FILTER, gl.NEAREST);
	gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_S, gl.CLAMP_TO_EDGE);
	gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_T, gl.CLAMP_TO_EDGE);
	gl.texImage2D(gl.TEXTURE_2D, 0, gl.RGBA, 2, 2, 0, gl.RGBA, pixelFormat, null);
	const framebuffer = gl.createFramebuffer();
	gl.bindFramebuffer(gl.FRAMEBUFFER, framebuffer);
	gl.framebufferTexture2D(gl.FRAMEBUFFER, gl.COLOR_ATTACHMENT0, gl.TEXTURE_2D, texture, 0);
	if (gl.checkFramebufferStatus(gl.FRAMEBUFFER) !== gl.FRAMEBUFFER_COMPLETE) {
		result = false;
	}
	gl.bindTexture(gl.TEXTURE_2D, null);
	gl.deleteTexture(texture);
	gl.bindFramebuffer(gl.FRAMEBUFFER, null);
	gl.deleteFramebuffer(framebuffer);
	return result;
}
function testTextureHalfFloatUpdatable(gl, pixelFormat) {
	let result = true;
	const texture = gl.createTexture();
	gl.bindTexture(gl.TEXTURE_2D, texture);
	gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MIN_FILTER, gl.NEAREST);
	gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MAG_FILTER, gl.NEAREST);
	gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_S, gl.CLAMP_TO_EDGE);
	gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_T, gl.CLAMP_TO_EDGE);
	const data = new Uint16Array(4 * 2 * 2);
	gl.texImage2D(gl.TEXTURE_2D, 0, gl.RGBA, 2, 2, 0, gl.RGBA, pixelFormat, data);
	if (gl.getError() !== gl.NO_ERROR) {
		result = false;
		console.log("Above error related to HALF_FLOAT_OES can be ignored, it was triggered by testing half float texture support");
	}
	gl.bindTexture(gl.TEXTURE_2D, null);
	gl.deleteTexture(texture);
	return result;
}
function testTextureFloatHighPrecision(device) {
	if (!device.textureFloatRenderable) return false;
	const shader1 = new Shader(device, ShaderUtils.createDefinition(device, {
		name: 'ptest1',
		vertexCode: _fullScreenQuadVS,
		fragmentCode: _precisionTest1PS
	}));
	const shader2 = new Shader(device, ShaderUtils.createDefinition(device, {
		name: 'ptest2',
		vertexCode: _fullScreenQuadVS,
		fragmentCode: _precisionTest2PS
	}));
	const textureOptions = {
		format: PIXELFORMAT_RGBA32F,
		width: 1,
		height: 1,
		mipmaps: false,
		minFilter: FILTER_NEAREST,
		magFilter: FILTER_NEAREST,
		name: 'testFHP'
	};
	const tex1 = new Texture(device, textureOptions);
	const targ1 = new RenderTarget({
		colorBuffer: tex1,
		depth: false
	});
	quadWithShader(device, targ1, shader1);
	textureOptions.format = PIXELFORMAT_RGBA8;
	const tex2 = new Texture(device, textureOptions);
	const targ2 = new RenderTarget({
		colorBuffer: tex2,
		depth: false
	});
	device.constantTexSource.setValue(tex1);
	quadWithShader(device, targ2, shader2);
	const prevFramebuffer = device.activeFramebuffer;
	device.setFramebuffer(targ2.impl._glFrameBuffer);
	const pixels = new Uint8Array(4);
	device.readPixels(0, 0, 1, 1, pixels);
	device.setFramebuffer(prevFramebuffer);
	const x = pixels[0] / 255;
	const y = pixels[1] / 255;
	const z = pixels[2] / 255;
	const w = pixels[3] / 255;
	const f = x / (256 * 256 * 256) + y / (256 * 256) + z / 256 + w;
	tex1.destroy();
	targ1.destroy();
	tex2.destroy();
	targ2.destroy();
	shader1.destroy();
	shader2.destroy();
	return f === 0;
}
class WebglGraphicsDevice extends GraphicsDevice {
	constructor(canvas, options = {}) {
		var _options$antialias;
		super(canvas, options);
		this.gl = void 0;
		this._defaultFramebuffer = null;
		this._defaultFramebufferChanged = false;
		options = this.initOptions;
		this.updateClientRect();
		this.initTextureUnits();
		this.contextLost = false;
		this._contextLostHandler = event => {
			event.preventDefault();
			this.loseContext();
			this.fire('devicelost');
		};
		this._contextRestoredHandler = () => {
			this.restoreContext();
			this.fire('devicerestored');
		};
		const ua = typeof navigator !== 'undefined' && navigator.userAgent;
		this.forceDisableMultisampling = ua && ua.includes('AppleWebKit') && (ua.includes('15.4') || ua.includes('15_4'));
		if (this.forceDisableMultisampling) {
			options.antialias = false;
		}
		if (platform.browserName === 'firefox' && platform.name === 'windows') {
			const _ua = typeof navigator !== 'undefined' ? navigator.userAgent : '';
			const match = _ua.match(/Firefox\/(\d+(\.\d+)*)/);
			const firefoxVersion = match ? match[1] : null;
			if (firefoxVersion) {
				const version = parseFloat(firefoxVersion);
				if (version >= 120 || version === 115) {
					options.antialias = false;
				}
			}
		}
		let gl = null;
		this.backBufferAntialias = (_options$antialias = options.antialias) != null ? _options$antialias : false;
		options.antialias = false;
		if (options.gl) {
			gl = options.gl;
		} else {
			const preferWebGl2 = options.preferWebGl2 !== undefined ? options.preferWebGl2 : true;
			const names = preferWebGl2 ? ["webgl2", "webgl", "experimental-webgl"] : ["webgl", "experimental-webgl"];
			for (let i = 0; i < names.length; i++) {
				gl = canvas.getContext(names[i], options);
				if (gl) {
					break;
				}
			}
		}
		if (!gl) {
			throw new Error("WebGL not supported");
		}
		this.gl = gl;
		this.isWebGL2 = typeof WebGL2RenderingContext !== 'undefined' && gl instanceof WebGL2RenderingContext;
		this.isWebGL1 = !this.isWebGL2;
		this._deviceType = this.isWebGL2 ? DEVICETYPE_WEBGL2 : DEVICETYPE_WEBGL1;
		this.updateBackbufferFormat(null);
		const isChrome = platform.browserName === 'chrome';
		const isSafari = platform.browserName === 'safari';
		const isMac = platform.browser && navigator.appVersion.indexOf("Mac") !== -1;
		this._tempEnableSafariTextureUnitWorkaround = isSafari;
		this._tempMacChromeBlitFramebufferWorkaround = isMac && isChrome && !options.alpha;
		canvas.addEventListener("webglcontextlost", this._contextLostHandler, false);
		canvas.addEventListener("webglcontextrestored", this._contextRestoredHandler, false);
		this.initializeExtensions();
		this.initializeCapabilities();
		this.initializeRenderState();
		this.initializeContextCaches();
		this.createBackbuffer(null);
		this.supportsImageBitmap = !isSafari && typeof ImageBitmap !== 'undefined';
		this._samplerTypes = new Set([...[gl.SAMPLER_2D, gl.SAMPLER_CUBE], ...(this.isWebGL2 ? [gl.UNSIGNED_INT_SAMPLER_2D, gl.INT_SAMPLER_2D, gl.SAMPLER_2D_SHADOW, gl.SAMPLER_CUBE_SHADOW, gl.SAMPLER_3D, gl.INT_SAMPLER_3D, gl.UNSIGNED_INT_SAMPLER_3D, gl.SAMPLER_2D_ARRAY, gl.INT_SAMPLER_2D_ARRAY, gl.UNSIGNED_INT_SAMPLER_2D_ARRAY] : [])]);
		this.glAddress = [gl.REPEAT, gl.CLAMP_TO_EDGE, gl.MIRRORED_REPEAT];
		this.glBlendEquation = [gl.FUNC_ADD, gl.FUNC_SUBTRACT, gl.FUNC_REVERSE_SUBTRACT, this.isWebGL2 ? gl.MIN : this.extBlendMinmax ? this.extBlendMinmax.MIN_EXT : gl.FUNC_ADD, this.isWebGL2 ? gl.MAX : this.extBlendMinmax ? this.extBlendMinmax.MAX_EXT : gl.FUNC_ADD];
		this.glBlendFunctionColor = [gl.ZERO, gl.ONE, gl.SRC_COLOR, gl.ONE_MINUS_SRC_COLOR, gl.DST_COLOR, gl.ONE_MINUS_DST_COLOR, gl.SRC_ALPHA, gl.SRC_ALPHA_SATURATE, gl.ONE_MINUS_SRC_ALPHA, gl.DST_ALPHA, gl.ONE_MINUS_DST_ALPHA, gl.CONSTANT_COLOR, gl.ONE_MINUS_CONSTANT_COLOR];
		this.glBlendFunctionAlpha = [gl.ZERO, gl.ONE, gl.SRC_COLOR, gl.ONE_MINUS_SRC_COLOR, gl.DST_COLOR, gl.ONE_MINUS_DST_COLOR, gl.SRC_ALPHA, gl.SRC_ALPHA_SATURATE, gl.ONE_MINUS_SRC_ALPHA, gl.DST_ALPHA, gl.ONE_MINUS_DST_ALPHA, gl.CONSTANT_ALPHA, gl.ONE_MINUS_CONSTANT_ALPHA];
		this.glComparison = [gl.NEVER, gl.LESS, gl.EQUAL, gl.LEQUAL, gl.GREATER, gl.NOTEQUAL, gl.GEQUAL, gl.ALWAYS];
		this.glStencilOp = [gl.KEEP, gl.ZERO, gl.REPLACE, gl.INCR, gl.INCR_WRAP, gl.DECR, gl.DECR_WRAP, gl.INVERT];
		this.glClearFlag = [0, gl.COLOR_BUFFER_BIT, gl.DEPTH_BUFFER_BIT, gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT, gl.STENCIL_BUFFER_BIT, gl.STENCIL_BUFFER_BIT | gl.COLOR_BUFFER_BIT, gl.STENCIL_BUFFER_BIT | gl.DEPTH_BUFFER_BIT, gl.STENCIL_BUFFER_BIT | gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT];
		this.glCull = [0, gl.BACK, gl.FRONT, gl.FRONT_AND_BACK];
		this.glFilter = [gl.NEAREST, gl.LINEAR, gl.NEAREST_MIPMAP_NEAREST, gl.NEAREST_MIPMAP_LINEAR, gl.LINEAR_MIPMAP_NEAREST, gl.LINEAR_MIPMAP_LINEAR];
		this.glPrimitive = [gl.POINTS, gl.LINES, gl.LINE_LOOP, gl.LINE_STRIP, gl.TRIANGLES, gl.TRIANGLE_STRIP, gl.TRIANGLE_FAN];
		this.glType = [gl.BYTE, gl.UNSIGNED_BYTE, gl.SHORT, gl.UNSIGNED_SHORT, gl.INT, gl.UNSIGNED_INT, gl.FLOAT, gl.HALF_FLOAT];
		this.pcUniformType = {};
		this.pcUniformType[gl.BOOL] = UNIFORMTYPE_BOOL;
		this.pcUniformType[gl.INT] = UNIFORMTYPE_INT;
		this.pcUniformType[gl.FLOAT] = UNIFORMTYPE_FLOAT;
		this.pcUniformType[gl.FLOAT_VEC2] = UNIFORMTYPE_VEC2;
		this.pcUniformType[gl.FLOAT_VEC3] = UNIFORMTYPE_VEC3;
		this.pcUniformType[gl.FLOAT_VEC4] = UNIFORMTYPE_VEC4;
		this.pcUniformType[gl.INT_VEC2] = UNIFORMTYPE_IVEC2;
		this.pcUniformType[gl.INT_VEC3] = UNIFORMTYPE_IVEC3;
		this.pcUniformType[gl.INT_VEC4] = UNIFORMTYPE_IVEC4;
		this.pcUniformType[gl.BOOL_VEC2] = UNIFORMTYPE_BVEC2;
		this.pcUniformType[gl.BOOL_VEC3] = UNIFORMTYPE_BVEC3;
		this.pcUniformType[gl.BOOL_VEC4] = UNIFORMTYPE_BVEC4;
		this.pcUniformType[gl.FLOAT_MAT2] = UNIFORMTYPE_MAT2;
		this.pcUniformType[gl.FLOAT_MAT3] = UNIFORMTYPE_MAT3;
		this.pcUniformType[gl.FLOAT_MAT4] = UNIFORMTYPE_MAT4;
		this.pcUniformType[gl.SAMPLER_2D] = UNIFORMTYPE_TEXTURE2D;
		this.pcUniformType[gl.SAMPLER_CUBE] = UNIFORMTYPE_TEXTURECUBE;
		this.pcUniformType[gl.UNSIGNED_INT] = UNIFORMTYPE_UINT;
		this.pcUniformType[gl.UNSIGNED_INT_VEC2] = UNIFORMTYPE_UVEC2;
		this.pcUniformType[gl.UNSIGNED_INT_VEC3] = UNIFORMTYPE_UVEC3;
		this.pcUniformType[gl.UNSIGNED_INT_VEC4] = UNIFORMTYPE_UVEC4;
		if (this.isWebGL2) {
			this.pcUniformType[gl.SAMPLER_2D_SHADOW] = UNIFORMTYPE_TEXTURE2D_SHADOW;
			this.pcUniformType[gl.SAMPLER_CUBE_SHADOW] = UNIFORMTYPE_TEXTURECUBE_SHADOW;
			this.pcUniformType[gl.SAMPLER_2D_ARRAY] = UNIFORMTYPE_TEXTURE2D_ARRAY;
			this.pcUniformType[gl.SAMPLER_3D] = UNIFORMTYPE_TEXTURE3D;
			this.pcUniformType[gl.INT_SAMPLER_2D] = UNIFORMTYPE_ITEXTURE2D;
			this.pcUniformType[gl.UNSIGNED_INT_SAMPLER_2D] = UNIFORMTYPE_UTEXTURE2D;
			this.pcUniformType[gl.INT_SAMPLER_CUBE] = UNIFORMTYPE_ITEXTURECUBE;
			this.pcUniformType[gl.UNSIGNED_INT_SAMPLER_2D] = UNIFORMTYPE_UTEXTURECUBE;
			this.pcUniformType[gl.INT_SAMPLER_3D] = UNIFORMTYPE_ITEXTURE3D;
			this.pcUniformType[gl.UNSIGNED_INT_SAMPLER_3D] = UNIFORMTYPE_UTEXTURE3D;
			this.pcUniformType[gl.INT_SAMPLER_2D_ARRAY] = UNIFORMTYPE_ITEXTURE2D_ARRAY;
			this.pcUniformType[gl.UNSIGNED_INT_SAMPLER_2D_ARRAY] = UNIFORMTYPE_UTEXTURE2D_ARRAY;
		}
		this.targetToSlot = {};
		this.targetToSlot[gl.TEXTURE_2D] = 0;
		this.targetToSlot[gl.TEXTURE_CUBE_MAP] = 1;
		this.targetToSlot[gl.TEXTURE_3D] = 2;
		let scopeX, scopeY, scopeZ, scopeW;
		let uniformValue;
		this.commitFunction = [];
		this.commitFunction[UNIFORMTYPE_BOOL] = function (uniform, value) {
			if (uniform.value !== value) {
				gl.uniform1i(uniform.locationId, value);
				uniform.value = value;
			}
		};
		this.commitFunction[UNIFORMTYPE_INT] = this.commitFunction[UNIFORMTYPE_BOOL];
		this.commitFunction[UNIFORMTYPE_FLOAT] = function (uniform, value) {
			if (uniform.value !== value) {
				gl.uniform1f(uniform.locationId, value);
				uniform.value = value;
			}
		};
		this.commitFunction[UNIFORMTYPE_VEC2] = function (uniform, value) {
			uniformValue = uniform.value;
			scopeX = value[0];
			scopeY = value[1];
			if (uniformValue[0] !== scopeX || uniformValue[1] !== scopeY) {
				gl.uniform2fv(uniform.locationId, value);
				uniformValue[0] = scopeX;
				uniformValue[1] = scopeY;
			}
		};
		this.commitFunction[UNIFORMTYPE_VEC3] = function (uniform, value) {
			uniformValue = uniform.value;
			scopeX = value[0];
			scopeY = value[1];
			scopeZ = value[2];
			if (uniformValue[0] !== scopeX || uniformValue[1] !== scopeY || uniformValue[2] !== scopeZ) {
				gl.uniform3fv(uniform.locationId, value);
				uniformValue[0] = scopeX;
				uniformValue[1] = scopeY;
				uniformValue[2] = scopeZ;
			}
		};
		this.commitFunction[UNIFORMTYPE_VEC4] = function (uniform, value) {
			uniformValue = uniform.value;
			scopeX = value[0];
			scopeY = value[1];
			scopeZ = value[2];
			scopeW = value[3];
			if (uniformValue[0] !== scopeX || uniformValue[1] !== scopeY || uniformValue[2] !== scopeZ || uniformValue[3] !== scopeW) {
				gl.uniform4fv(uniform.locationId, value);
				uniformValue[0] = scopeX;
				uniformValue[1] = scopeY;
				uniformValue[2] = scopeZ;
				uniformValue[3] = scopeW;
			}
		};
		this.commitFunction[UNIFORMTYPE_IVEC2] = function (uniform, value) {
			uniformValue = uniform.value;
			scopeX = value[0];
			scopeY = value[1];
			if (uniformValue[0] !== scopeX || uniformValue[1] !== scopeY) {
				gl.uniform2iv(uniform.locationId, value);
				uniformValue[0] = scopeX;
				uniformValue[1] = scopeY;
			}
		};
		this.commitFunction[UNIFORMTYPE_BVEC2] = this.commitFunction[UNIFORMTYPE_IVEC2];
		this.commitFunction[UNIFORMTYPE_IVEC3] = function (uniform, value) {
			uniformValue = uniform.value;
			scopeX = value[0];
			scopeY = value[1];
			scopeZ = value[2];
			if (uniformValue[0] !== scopeX || uniformValue[1] !== scopeY || uniformValue[2] !== scopeZ) {
				gl.uniform3iv(uniform.locationId, value);
				uniformValue[0] = scopeX;
				uniformValue[1] = scopeY;
				uniformValue[2] = scopeZ;
			}
		};
		this.commitFunction[UNIFORMTYPE_BVEC3] = this.commitFunction[UNIFORMTYPE_IVEC3];
		this.commitFunction[UNIFORMTYPE_IVEC4] = function (uniform, value) {
			uniformValue = uniform.value;
			scopeX = value[0];
			scopeY = value[1];
			scopeZ = value[2];
			scopeW = value[3];
			if (uniformValue[0] !== scopeX || uniformValue[1] !== scopeY || uniformValue[2] !== scopeZ || uniformValue[3] !== scopeW) {
				gl.uniform4iv(uniform.locationId, value);
				uniformValue[0] = scopeX;
				uniformValue[1] = scopeY;
				uniformValue[2] = scopeZ;
				uniformValue[3] = scopeW;
			}
		};
		this.commitFunction[UNIFORMTYPE_BVEC4] = this.commitFunction[UNIFORMTYPE_IVEC4];
		this.commitFunction[UNIFORMTYPE_MAT2] = function (uniform, value) {
			gl.uniformMatrix2fv(uniform.locationId, false, value);
		};
		this.commitFunction[UNIFORMTYPE_MAT3] = function (uniform, value) {
			gl.uniformMatrix3fv(uniform.locationId, false, value);
		};
		this.commitFunction[UNIFORMTYPE_MAT4] = function (uniform, value) {
			gl.uniformMatrix4fv(uniform.locationId, false, value);
		};
		this.commitFunction[UNIFORMTYPE_FLOATARRAY] = function (uniform, value) {
			gl.uniform1fv(uniform.locationId, value);
		};
		this.commitFunction[UNIFORMTYPE_VEC2ARRAY] = function (uniform, value) {
			gl.uniform2fv(uniform.locationId, value);
		};
		this.commitFunction[UNIFORMTYPE_VEC3ARRAY] = function (uniform, value) {
			gl.uniform3fv(uniform.locationId, value);
		};
		this.commitFunction[UNIFORMTYPE_VEC4ARRAY] = function (uniform, value) {
			gl.uniform4fv(uniform.locationId, value);
		};
		this.commitFunction[UNIFORMTYPE_UINT] = function (uniform, value) {
			if (uniform.value !== value) {
				gl.uniform1ui(uniform.locationId, value);
				uniform.value = value;
			}
		};
		this.commitFunction[UNIFORMTYPE_UVEC2] = function (uniform, value) {
			uniformValue = uniform.value;
			scopeX = value[0];
			scopeY = value[1];
			if (uniformValue[0] !== scopeX || uniformValue[1] !== scopeY) {
				gl.uniform2uiv(uniform.locationId, value);
				uniformValue[0] = scopeX;
				uniformValue[1] = scopeY;
			}
		};
		this.commitFunction[UNIFORMTYPE_UVEC3] = function (uniform, value) {
			uniformValue = uniform.value;
			scopeX = value[0];
			scopeY = value[1];
			scopeZ = value[2];
			if (uniformValue[0] !== scopeX || uniformValue[1] !== scopeY || uniformValue[2] !== scopeZ) {
				gl.uniform3uiv(uniform.locationId, value);
				uniformValue[0] = scopeX;
				uniformValue[1] = scopeY;
				uniformValue[2] = scopeZ;
			}
		};
		this.commitFunction[UNIFORMTYPE_UVEC4] = function (uniform, value) {
			uniformValue = uniform.value;
			scopeX = value[0];
			scopeY = value[1];
			scopeZ = value[2];
			scopeW = value[3];
			if (uniformValue[0] !== scopeX || uniformValue[1] !== scopeY || uniformValue[2] !== scopeZ || uniformValue[3] !== scopeW) {
				gl.uniform4uiv(uniform.locationId, value);
				uniformValue[0] = scopeX;
				uniformValue[1] = scopeY;
				uniformValue[2] = scopeZ;
				uniformValue[3] = scopeW;
			}
		};
		this.commitFunction[UNIFORMTYPE_INTARRAY] = function (uniform, value) {
			gl.uniform1iv(uniform.locationId, value);
		};
		this.commitFunction[UNIFORMTYPE_UINTARRAY] = function (uniform, value) {
			gl.uniform1uiv(uniform.locationId, value);
		};
		this.commitFunction[UNIFORMTYPE_BOOLARRAY] = this.commitFunction[UNIFORMTYPE_INTARRAY];
		this.commitFunction[UNIFORMTYPE_IVEC2ARRAY] = function (uniform, value) {
			gl.uniform2iv(uniform.locationId, value);
		};
		this.commitFunction[UNIFORMTYPE_UVEC2ARRAY] = function (uniform, value) {
			gl.uniform2uiv(uniform.locationId, value);
		};
		this.commitFunction[UNIFORMTYPE_BVEC2ARRAY] = this.commitFunction[UNIFORMTYPE_IVEC2ARRAY];
		this.commitFunction[UNIFORMTYPE_IVEC3ARRAY] = function (uniform, value) {
			gl.uniform3iv(uniform.locationId, value);
		};
		this.commitFunction[UNIFORMTYPE_UVEC3ARRAY] = function (uniform, value) {
			gl.uniform3uiv(uniform.locationId, value);
		};
		this.commitFunction[UNIFORMTYPE_BVEC3ARRAY] = this.commitFunction[UNIFORMTYPE_IVEC3ARRAY];
		this.commitFunction[UNIFORMTYPE_IVEC4ARRAY] = function (uniform, value) {
			gl.uniform4iv(uniform.locationId, value);
		};
		this.commitFunction[UNIFORMTYPE_UVEC4ARRAY] = function (uniform, value) {
			gl.uniform4uiv(uniform.locationId, value);
		};
		this.commitFunction[UNIFORMTYPE_BVEC4ARRAY] = this.commitFunction[UNIFORMTYPE_IVEC4ARRAY];
		this.commitFunction[UNIFORMTYPE_MAT4ARRAY] = function (uniform, value) {
			gl.uniformMatrix4fv(uniform.locationId, false, value);
		};
		this.supportsBoneTextures = this.extTextureFloat && this.maxVertexTextures > 0;
		let numUniforms = this.vertexUniformsCount;
		numUniforms -= 4 * 4;
		numUniforms -= 8;
		numUniforms -= 1;
		numUniforms -= 4 * 4;
		this.boneLimit = Math.floor(numUniforms / 3);
		this.boneLimit = Math.min(this.boneLimit, 128);
		if (this.unmaskedRenderer === 'Mali-450 MP') {
			this.boneLimit = 34;
		}
		this.constantTexSource = this.scope.resolve("source");
		if (this.extTextureFloat) {
			if (this.isWebGL2) {
				this.textureFloatRenderable = !!this.extColorBufferFloat;
			} else {
				this.textureFloatRenderable = testRenderable(gl, gl.FLOAT);
			}
		} else {
			this.textureFloatRenderable = false;
		}
		if (this.extColorBufferHalfFloat) {
			this.textureHalfFloatRenderable = !!this.extColorBufferHalfFloat;
		} else if (this.extTextureHalfFloat) {
			if (this.isWebGL2) {
				this.textureHalfFloatRenderable = !!this.extColorBufferFloat;
			} else {
				this.textureHalfFloatRenderable = testRenderable(gl, this.extTextureHalfFloat.HALF_FLOAT_OES);
			}
		} else {
			this.textureHalfFloatRenderable = false;
		}
		this.supportsMorphTargetTexturesCore = this.maxPrecision === "highp" && this.maxVertexTextures >= 2;
		this.supportsDepthShadow = this.isWebGL2;
		this._textureFloatHighPrecision = undefined;
		this._textureHalfFloatUpdatable = undefined;
		this.areaLightLutFormat = PIXELFORMAT_RGBA8;
		if (this.extTextureHalfFloat && this.textureHalfFloatUpdatable && this.extTextureHalfFloatLinear) {
			this.areaLightLutFormat = PIXELFORMAT_RGBA16F;
		} else if (this.extTextureFloat && this.extTextureFloatLinear) {
			this.areaLightLutFormat = PIXELFORMAT_RGBA32F;
		}
		this.postInit();
	}
	postInit() {
		super.postInit();
		this.gpuProfiler = new WebglGpuProfiler(this);
	}
	destroy() {
		super.destroy();
		const gl = this.gl;
		if (this.isWebGL2 && this.feedback) {
			gl.deleteTransformFeedback(this.feedback);
		}
		this.clearVertexArrayObjectCache();
		this.canvas.removeEventListener('webglcontextlost', this._contextLostHandler, false);
		this.canvas.removeEventListener('webglcontextrestored', this._contextRestoredHandler, false);
		this._contextLostHandler = null;
		this._contextRestoredHandler = null;
		this.gl = null;
		super.postDestroy();
	}
	createBackbuffer(frameBuffer) {
		this.supportsStencil = this.initOptions.stencil;
		this.backBuffer = new RenderTarget({
			name: 'WebglFramebuffer',
			graphicsDevice: this,
			depth: this.initOptions.depth,
			stencil: this.supportsStencil,
			samples: this.samples
		});
		this.backBuffer.impl.suppliedColorFramebuffer = frameBuffer;
	}
	updateBackbufferFormat(framebuffer) {
		const gl = this.gl;
		gl.bindFramebuffer(gl.FRAMEBUFFER, framebuffer);
		const alphaBits = this.gl.getParameter(this.gl.ALPHA_BITS);
		this.backBufferFormat = alphaBits ? PIXELFORMAT_RGBA8 : PIXELFORMAT_RGB8;
	}
	updateBackbuffer() {
		const resolutionChanged = this.canvas.width !== this.backBufferSize.x || this.canvas.height !== this.backBufferSize.y;
		if (this._defaultFramebufferChanged || resolutionChanged) {
			if (this._defaultFramebufferChanged) {
				this.updateBackbufferFormat(this._defaultFramebuffer);
			}
			this._defaultFramebufferChanged = false;
			this.backBufferSize.set(this.canvas.width, this.canvas.height);
			this.backBuffer.destroy();
			this.createBackbuffer(this._defaultFramebuffer);
		}
	}
	createVertexBufferImpl(vertexBuffer, format) {
		return new WebglVertexBuffer();
	}
	createIndexBufferImpl(indexBuffer) {
		return new WebglIndexBuffer(indexBuffer);
	}
	createShaderImpl(shader) {
		return new WebglShader(shader);
	}
	createTextureImpl(texture) {
		return new WebglTexture();
	}
	createRenderTargetImpl(renderTarget) {
		return new WebglRenderTarget();
	}
	getPrecision() {
		const gl = this.gl;
		let precision = "highp";
		if (gl.getShaderPrecisionFormat) {
			const vertexShaderPrecisionHighpFloat = gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.HIGH_FLOAT);
			const vertexShaderPrecisionMediumpFloat = gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.MEDIUM_FLOAT);
			const fragmentShaderPrecisionHighpFloat = gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.HIGH_FLOAT);
			const fragmentShaderPrecisionMediumpFloat = gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.MEDIUM_FLOAT);
			if (vertexShaderPrecisionHighpFloat && vertexShaderPrecisionMediumpFloat && fragmentShaderPrecisionHighpFloat && fragmentShaderPrecisionMediumpFloat) {
				const highpAvailable = vertexShaderPrecisionHighpFloat.precision > 0 && fragmentShaderPrecisionHighpFloat.precision > 0;
				const mediumpAvailable = vertexShaderPrecisionMediumpFloat.precision > 0 && fragmentShaderPrecisionMediumpFloat.precision > 0;
				if (!highpAvailable) {
					if (mediumpAvailable) {
						precision = "mediump";
					} else {
						precision = "lowp";
					}
				}
			}
		}
		return precision;
	}
	getExtension() {
		for (let i = 0; i < arguments.length; i++) {
			if (this.supportedExtensions.indexOf(arguments[i]) !== -1) {
				return this.gl.getExtension(arguments[i]);
			}
		}
		return null;
	}
	get extDisjointTimerQuery() {
		if (!this._extDisjointTimerQuery) {
			if (this.isWebGL2) {
				this._extDisjointTimerQuery = this.getExtension('EXT_disjoint_timer_query_webgl2', 'EXT_disjoint_timer_query');
			}
		}
		return this._extDisjointTimerQuery;
	}
	initializeExtensions() {
		var _gl$getSupportedExten;
		const gl = this.gl;
		this.supportedExtensions = (_gl$getSupportedExten = gl.getSupportedExtensions()) != null ? _gl$getSupportedExten : [];
		this._extDisjointTimerQuery = null;
		if (this.isWebGL2) {
			this.extBlendMinmax = true;
			this.extDrawBuffers = true;
			this.drawBuffers = gl.drawBuffers.bind(gl);
			this.extInstancing = true;
			this.extStandardDerivatives = true;
			this.extTextureFloat = true;
			this.extTextureHalfFloat = true;
			this.textureHalfFloatFilterable = true;
			this.extTextureLod = true;
			this.extUintElement = true;
			this.extVertexArrayObject = true;
			this.extColorBufferFloat = this.getExtension('EXT_color_buffer_float');
			this.extDepthTexture = true;
			this.textureRG11B10Renderable = true;
		} else {
			var _this$extDrawBuffers;
			this.extBlendMinmax = this.getExtension("EXT_blend_minmax");
			this.extDrawBuffers = this.getExtension('WEBGL_draw_buffers');
			this.extInstancing = this.getExtension("ANGLE_instanced_arrays");
			this.drawBuffers = (_this$extDrawBuffers = this.extDrawBuffers) == null ? void 0 : _this$extDrawBuffers.drawBuffersWEBGL.bind(this.extDrawBuffers);
			if (this.extInstancing) {
				const ext = this.extInstancing;
				gl.drawArraysInstanced = ext.drawArraysInstancedANGLE.bind(ext);
				gl.drawElementsInstanced = ext.drawElementsInstancedANGLE.bind(ext);
				gl.vertexAttribDivisor = ext.vertexAttribDivisorANGLE.bind(ext);
			}
			this.extStandardDerivatives = this.getExtension("OES_standard_derivatives");
			this.extTextureFloat = this.getExtension("OES_texture_float");
			this.extTextureLod = this.getExtension('EXT_shader_texture_lod');
			this.extUintElement = this.getExtension("OES_element_index_uint");
			this.extVertexArrayObject = this.getExtension("OES_vertex_array_object");
			if (this.extVertexArrayObject) {
				const ext = this.extVertexArrayObject;
				gl.createVertexArray = ext.createVertexArrayOES.bind(ext);
				gl.deleteVertexArray = ext.deleteVertexArrayOES.bind(ext);
				gl.isVertexArray = ext.isVertexArrayOES.bind(ext);
				gl.bindVertexArray = ext.bindVertexArrayOES.bind(ext);
			}
			this.extColorBufferFloat = null;
			this.extDepthTexture = gl.getExtension('WEBGL_depth_texture');
			this.extTextureHalfFloat = this.getExtension("OES_texture_half_float");
			this.extTextureHalfFloatLinear = this.getExtension("OES_texture_half_float_linear");
			this.textureHalfFloatFilterable = !!this.extTextureHalfFloatLinear;
		}
		this.extDebugRendererInfo = this.getExtension('WEBGL_debug_renderer_info');
		this.extTextureFloatLinear = this.getExtension("OES_texture_float_linear");
		this.textureFloatFilterable = !!this.extTextureFloatLinear;
		this.extFloatBlend = this.getExtension("EXT_float_blend");
		this.extTextureFilterAnisotropic = this.getExtension('EXT_texture_filter_anisotropic', 'WEBKIT_EXT_texture_filter_anisotropic');
		this.extCompressedTextureETC1 = this.getExtension('WEBGL_compressed_texture_etc1');
		this.extCompressedTextureETC = this.getExtension('WEBGL_compressed_texture_etc');
		this.extCompressedTexturePVRTC = this.getExtension('WEBGL_compressed_texture_pvrtc', 'WEBKIT_WEBGL_compressed_texture_pvrtc');
		this.extCompressedTextureS3TC = this.getExtension('WEBGL_compressed_texture_s3tc', 'WEBKIT_WEBGL_compressed_texture_s3tc');
		this.extCompressedTextureATC = this.getExtension('WEBGL_compressed_texture_atc');
		this.extCompressedTextureASTC = this.getExtension('WEBGL_compressed_texture_astc');
		this.extParallelShaderCompile = this.getExtension('KHR_parallel_shader_compile');
		this.extColorBufferHalfFloat = this.getExtension("EXT_color_buffer_half_float");
	}
	initializeCapabilities() {
		var _contextAttribs$antia, _contextAttribs$stenc;
		const gl = this.gl;
		let ext;
		const userAgent = typeof navigator !== 'undefined' ? navigator.userAgent : "";
		this.maxPrecision = this.precision = this.getPrecision();
		const contextAttribs = gl.getContextAttributes();
		this.supportsMsaa = (_contextAttribs$antia = contextAttribs == null ? void 0 : contextAttribs.antialias) != null ? _contextAttribs$antia : false;
		this.supportsStencil = (_contextAttribs$stenc = contextAttribs == null ? void 0 : contextAttribs.stencil) != null ? _contextAttribs$stenc : false;
		this.supportsInstancing = !!this.extInstancing;
		this.maxTextureSize = gl.getParameter(gl.MAX_TEXTURE_SIZE);
		this.maxCubeMapSize = gl.getParameter(gl.MAX_CUBE_MAP_TEXTURE_SIZE);
		this.maxRenderBufferSize = gl.getParameter(gl.MAX_RENDERBUFFER_SIZE);
		this.maxTextures = gl.getParameter(gl.MAX_TEXTURE_IMAGE_UNITS);
		this.maxCombinedTextures = gl.getParameter(gl.MAX_COMBINED_TEXTURE_IMAGE_UNITS);
		this.maxVertexTextures = gl.getParameter(gl.MAX_VERTEX_TEXTURE_IMAGE_UNITS);
		this.vertexUniformsCount = gl.getParameter(gl.MAX_VERTEX_UNIFORM_VECTORS);
		this.fragmentUniformsCount = gl.getParameter(gl.MAX_FRAGMENT_UNIFORM_VECTORS);
		if (this.isWebGL2) {
			this.maxDrawBuffers = gl.getParameter(gl.MAX_DRAW_BUFFERS);
			this.maxColorAttachments = gl.getParameter(gl.MAX_COLOR_ATTACHMENTS);
			this.maxVolumeSize = gl.getParameter(gl.MAX_3D_TEXTURE_SIZE);
			this.supportsMrt = true;
			this.supportsVolumeTextures = true;
		} else {
			ext = this.extDrawBuffers;
			this.supportsMrt = !!ext;
			this.maxDrawBuffers = ext ? gl.getParameter(ext.MAX_DRAW_BUFFERS_WEBGL) : 1;
			this.maxColorAttachments = ext ? gl.getParameter(ext.MAX_COLOR_ATTACHMENTS_WEBGL) : 1;
			this.maxVolumeSize = 1;
		}
		ext = this.extDebugRendererInfo;
		this.unmaskedRenderer = ext ? gl.getParameter(ext.UNMASKED_RENDERER_WEBGL) : '';
		this.unmaskedVendor = ext ? gl.getParameter(ext.UNMASKED_VENDOR_WEBGL) : '';
		const maliRendererRegex = /\bMali-G52+/;
		const samsungModelRegex = /SM-[a-zA-Z0-9]+/;
		this.supportsGpuParticles = !(this.unmaskedVendor === 'ARM' && userAgent.match(samsungModelRegex)) && !this.unmaskedRenderer.match(maliRendererRegex);
		ext = this.extTextureFilterAnisotropic;
		this.maxAnisotropy = ext ? gl.getParameter(ext.MAX_TEXTURE_MAX_ANISOTROPY_EXT) : 1;
		const antialiasSupported = this.isWebGL2 && !this.forceDisableMultisampling;
		this.maxSamples = antialiasSupported ? gl.getParameter(gl.MAX_SAMPLES) : 1;
		this.maxSamples = Math.min(this.maxSamples, 4);
		this.samples = antialiasSupported && this.backBufferAntialias ? this.maxSamples : 1;
		this.supportsAreaLights = this.isWebGL2 || !platform.android;
		this.supportsTextureFetch = this.isWebGL2;
		if (this.maxTextures <= 8) {
			this.supportsAreaLights = false;
		}
	}
	initializeRenderState() {
		super.initializeRenderState();
		const gl = this.gl;
		gl.disable(gl.BLEND);
		gl.blendFunc(gl.ONE, gl.ZERO);
		gl.blendEquation(gl.FUNC_ADD);
		gl.colorMask(true, true, true, true);
		gl.blendColor(0, 0, 0, 0);
		gl.enable(gl.CULL_FACE);
		this.cullFace = gl.BACK;
		gl.cullFace(gl.BACK);
		gl.enable(gl.DEPTH_TEST);
		gl.depthFunc(gl.LEQUAL);
		gl.depthMask(true);
		this.stencil = false;
		gl.disable(gl.STENCIL_TEST);
		this.stencilFuncFront = this.stencilFuncBack = FUNC_ALWAYS;
		this.stencilRefFront = this.stencilRefBack = 0;
		this.stencilMaskFront = this.stencilMaskBack = 0xFF;
		gl.stencilFunc(gl.ALWAYS, 0, 0xFF);
		this.stencilFailFront = this.stencilFailBack = STENCILOP_KEEP;
		this.stencilZfailFront = this.stencilZfailBack = STENCILOP_KEEP;
		this.stencilZpassFront = this.stencilZpassBack = STENCILOP_KEEP;
		this.stencilWriteMaskFront = 0xFF;
		this.stencilWriteMaskBack = 0xFF;
		gl.stencilOp(gl.KEEP, gl.KEEP, gl.KEEP);
		gl.stencilMask(0xFF);
		this.alphaToCoverage = false;
		this.raster = true;
		if (this.isWebGL2) {
			gl.disable(gl.SAMPLE_ALPHA_TO_COVERAGE);
			gl.disable(gl.RASTERIZER_DISCARD);
		}
		this.depthBiasEnabled = false;
		gl.disable(gl.POLYGON_OFFSET_FILL);
		this.clearDepth = 1;
		gl.clearDepth(1);
		this.clearColor = new Color(0, 0, 0, 0);
		gl.clearColor(0, 0, 0, 0);
		this.clearStencil = 0;
		gl.clearStencil(0);
		if (this.isWebGL2) {
			gl.hint(gl.FRAGMENT_SHADER_DERIVATIVE_HINT, gl.NICEST);
		} else {
			if (this.extStandardDerivatives) {
				gl.hint(this.extStandardDerivatives.FRAGMENT_SHADER_DERIVATIVE_HINT_OES, gl.NICEST);
			}
		}
		gl.enable(gl.SCISSOR_TEST);
		gl.pixelStorei(gl.UNPACK_COLORSPACE_CONVERSION_WEBGL, gl.NONE);
		this.unpackFlipY = false;
		gl.pixelStorei(gl.UNPACK_FLIP_Y_WEBGL, false);
		this.unpackPremultiplyAlpha = false;
		gl.pixelStorei(gl.UNPACK_PREMULTIPLY_ALPHA_WEBGL, false);
		gl.pixelStorei(gl.UNPACK_ALIGNMENT, 1);
	}
	initTextureUnits(count = 16) {
		this.textureUnits = [];
		for (let i = 0; i < count; i++) {
			this.textureUnits.push([null, null, null]);
		}
	}
	initializeContextCaches() {
		super.initializeContextCaches();
		this._vaoMap = new Map();
		this.boundVao = null;
		this.activeFramebuffer = null;
		this.feedback = null;
		this.transformFeedbackBuffer = null;
		this.textureUnit = 0;
		this.initTextureUnits(this.maxCombinedTextures);
	}
	loseContext() {
		var _this$gpuProfiler;
		this.contextLost = true;
		this.backBufferSize.set(-1, -1);
		for (const shader of this.shaders) {
			shader.loseContext();
		}
		for (const texture of this.textures) {
			texture.loseContext();
		}
		for (const buffer of this.buffers) {
			buffer.loseContext();
		}
		for (const target of this.targets) {
			target.loseContext();
		}
		(_this$gpuProfiler = this.gpuProfiler) == null || _this$gpuProfiler.loseContext();
	}
	restoreContext() {
		var _this$gpuProfiler2;
		this.contextLost = false;
		this.initializeExtensions();
		this.initializeCapabilities();
		this.initializeRenderState();
		this.initializeContextCaches();
		for (const shader of this.shaders) {
			shader.restoreContext();
		}
		for (const buffer of this.buffers) {
			buffer.unlock();
		}
		(_this$gpuProfiler2 = this.gpuProfiler) == null || _this$gpuProfiler2.restoreContext();
	}
	setViewport(x, y, w, h) {
		if (this.vx !== x || this.vy !== y || this.vw !== w || this.vh !== h) {
			this.gl.viewport(x, y, w, h);
			this.vx = x;
			this.vy = y;
			this.vw = w;
			this.vh = h;
		}
	}
	setScissor(x, y, w, h) {
		if (this.sx !== x || this.sy !== y || this.sw !== w || this.sh !== h) {
			this.gl.scissor(x, y, w, h);
			this.sx = x;
			this.sy = y;
			this.sw = w;
			this.sh = h;
		}
	}
	setFramebuffer(fb) {
		if (this.activeFramebuffer !== fb) {
			const gl = this.gl;
			gl.bindFramebuffer(gl.FRAMEBUFFER, fb);
			this.activeFramebuffer = fb;
		}
	}
	copyRenderTarget(source, dest, color, depth) {
		const gl = this.gl;
		if (source === this.backBuffer) {
			source = null;
		}
		if (!this.isWebGL2 && depth) {
			return false;
		}
		if (color) {
			if (!dest) {
				if (!source._colorBuffer) {
					return false;
				}
			} else if (source) {
				if (!source._colorBuffer || !dest._colorBuffer) {
					return false;
				}
				if (source._colorBuffer._format !== dest._colorBuffer._format) {
					return false;
				}
			}
		}
		if (depth && source) {
			if (!source._depth) {
				if (!source._depthBuffer || !dest._depthBuffer) {
					return false;
				}
				if (source._depthBuffer._format !== dest._depthBuffer._format) {
					return false;
				}
			}
		}
		if (this.isWebGL2 && dest) {
			var _this$backBuffer;
			const prevRt = this.renderTarget;
			this.renderTarget = dest;
			this.updateBegin();
			const src = source ? source.impl._glFrameBuffer : (_this$backBuffer = this.backBuffer) == null ? void 0 : _this$backBuffer.impl._glFrameBuffer;
			const dst = dest.impl._glFrameBuffer;
			gl.bindFramebuffer(gl.READ_FRAMEBUFFER, src);
			gl.bindFramebuffer(gl.DRAW_FRAMEBUFFER, dst);
			const w = source ? source.width : dest.width;
			const h = source ? source.height : dest.height;
			gl.blitFramebuffer(0, 0, w, h, 0, 0, w, h, (color ? gl.COLOR_BUFFER_BIT : 0) | (depth ? gl.DEPTH_BUFFER_BIT : 0), gl.NEAREST);
			this.renderTarget = prevRt;
			gl.bindFramebuffer(gl.FRAMEBUFFER, prevRt ? prevRt.impl._glFrameBuffer : null);
		} else {
			const shader = this.getCopyShader();
			this.constantTexSource.setValue(source._colorBuffer);
			quadWithShader(this, dest, shader);
		}
		return true;
	}
	getCopyShader() {
		if (!this._copyShader) {
			this._copyShader = new Shader(this, ShaderUtils.createDefinition(this, {
				name: 'outputTex2D',
				vertexCode: _fullScreenQuadVS,
				fragmentCode: _outputTexture2D
			}));
		}
		return this._copyShader;
	}
	frameStart() {
		super.frameStart();
		this.updateBackbuffer();
		this.gpuProfiler.frameStart();
	}
	frameEnd() {
		super.frameEnd();
		this.gpuProfiler.frameEnd();
		this.gpuProfiler.request();
	}
	startRenderPass(renderPass) {
		var _renderPass$renderTar;
		const rt = (_renderPass$renderTar = renderPass.renderTarget) != null ? _renderPass$renderTar : this.backBuffer;
		this.renderTarget = rt;
		this.updateBegin();
		const {
			width,
			height
		} = rt;
		this.setViewport(0, 0, width, height);
		this.setScissor(0, 0, width, height);
		const colorOps = renderPass.colorOps;
		const depthStencilOps = renderPass.depthStencilOps;
		if (colorOps != null && colorOps.clear || depthStencilOps.clearDepth || depthStencilOps.clearStencil) {
			let clearFlags = 0;
			const clearOptions = {};
			if (colorOps != null && colorOps.clear) {
				clearFlags |= CLEARFLAG_COLOR;
				clearOptions.color = [colorOps.clearValue.r, colorOps.clearValue.g, colorOps.clearValue.b, colorOps.clearValue.a];
			}
			if (depthStencilOps.clearDepth) {
				clearFlags |= CLEARFLAG_DEPTH;
				clearOptions.depth = depthStencilOps.clearDepthValue;
			}
			if (depthStencilOps.clearStencil) {
				clearFlags |= CLEARFLAG_STENCIL;
				clearOptions.stencil = depthStencilOps.clearStencilValue;
			}
			clearOptions.flags = clearFlags;
			this.clear(clearOptions);
		}
		this.insideRenderPass = true;
	}
	endRenderPass(renderPass) {
		this.unbindVertexArray();
		const target = this.renderTarget;
		const colorBufferCount = renderPass.colorArrayOps.length;
		if (target) {
			var _renderPass$colorOps;
			if (this.isWebGL2) {
				invalidateAttachments.length = 0;
				const gl = this.gl;
				for (let i = 0; i < colorBufferCount; i++) {
					const colorOps = renderPass.colorArrayOps[i];
					if (!(colorOps.store || colorOps.resolve)) {
						invalidateAttachments.push(gl.COLOR_ATTACHMENT0 + i);
					}
				}
				if (target !== this.backBuffer) {
					if (!renderPass.depthStencilOps.storeDepth) {
						invalidateAttachments.push(gl.DEPTH_ATTACHMENT);
					}
					if (!renderPass.depthStencilOps.storeStencil) {
						invalidateAttachments.push(gl.STENCIL_ATTACHMENT);
					}
				}
				if (invalidateAttachments.length > 0) {
					if (renderPass.fullSizeClearRect) {
						gl.invalidateFramebuffer(gl.DRAW_FRAMEBUFFER, invalidateAttachments);
					}
				}
			}
			if ((_renderPass$colorOps = renderPass.colorOps) != null && _renderPass$colorOps.resolve) {
				if (this.isWebGL2 && renderPass.samples > 1 && target.autoResolve) {
					target.resolve(true, false);
				}
			}
			for (let i = 0; i < colorBufferCount; i++) {
				const colorOps = renderPass.colorArrayOps[i];
				if (colorOps.mipmaps) {
					const colorBuffer = target._colorBuffers[i];
					if (colorBuffer && colorBuffer.impl._glTexture && colorBuffer.mipmaps && (colorBuffer.pot || this.isWebGL2)) {
						this.activeTexture(this.maxCombinedTextures - 1);
						this.bindTexture(colorBuffer);
						this.gl.generateMipmap(colorBuffer.impl._glTarget);
					}
				}
			}
		}
		this.insideRenderPass = false;
	}
	set defaultFramebuffer(value) {
		if (this._defaultFramebuffer !== value) {
			this._defaultFramebuffer = value;
			this._defaultFramebufferChanged = true;
		}
	}
	get defaultFramebuffer() {
		return this._defaultFramebuffer;
	}
	updateBegin() {
		var _this$renderTarget;
		this.boundVao = null;
		if (this._tempEnableSafariTextureUnitWorkaround) {
			for (let unit = 0; unit < this.textureUnits.length; ++unit) {
				for (let slot = 0; slot < 3; ++slot) {
					this.textureUnits[unit][slot] = null;
				}
			}
		}
		const target = (_this$renderTarget = this.renderTarget) != null ? _this$renderTarget : this.backBuffer;
		const targetImpl = target.impl;
		if (!targetImpl.initialized) {
			this.initRenderTarget(target);
		}
		this.setFramebuffer(targetImpl._glFrameBuffer);
	}
	updateEnd() {
		this.unbindVertexArray();
		const target = this.renderTarget;
		if (target && target !== this.backBuffer) {
			if (this.isWebGL2 && target._samples > 1 && target.autoResolve) {
				target.resolve();
			}
			const colorBuffer = target._colorBuffer;
			if (colorBuffer && colorBuffer.impl._glTexture && colorBuffer.mipmaps && (colorBuffer.pot || this.isWebGL2)) {
				this.activeTexture(this.maxCombinedTextures - 1);
				this.bindTexture(colorBuffer);
				this.gl.generateMipmap(colorBuffer.impl._glTarget);
			}
		}
	}
	setUnpackFlipY(flipY) {
		if (this.unpackFlipY !== flipY) {
			this.unpackFlipY = flipY;
			const gl = this.gl;
			gl.pixelStorei(gl.UNPACK_FLIP_Y_WEBGL, flipY);
		}
	}
	setUnpackPremultiplyAlpha(premultiplyAlpha) {
		if (this.unpackPremultiplyAlpha !== premultiplyAlpha) {
			this.unpackPremultiplyAlpha = premultiplyAlpha;
			const gl = this.gl;
			gl.pixelStorei(gl.UNPACK_PREMULTIPLY_ALPHA_WEBGL, premultiplyAlpha);
		}
	}
	activeTexture(textureUnit) {
		if (this.textureUnit !== textureUnit) {
			this.gl.activeTexture(this.gl.TEXTURE0 + textureUnit);
			this.textureUnit = textureUnit;
		}
	}
	bindTexture(texture) {
		const impl = texture.impl;
		const textureTarget = impl._glTarget;
		const textureObject = impl._glTexture;
		const textureUnit = this.textureUnit;
		const slot = this.targetToSlot[textureTarget];
		if (this.textureUnits[textureUnit][slot] !== textureObject) {
			this.gl.bindTexture(textureTarget, textureObject);
			this.textureUnits[textureUnit][slot] = textureObject;
		}
	}
	bindTextureOnUnit(texture, textureUnit) {
		const impl = texture.impl;
		const textureTarget = impl._glTarget;
		const textureObject = impl._glTexture;
		const slot = this.targetToSlot[textureTarget];
		if (this.textureUnits[textureUnit][slot] !== textureObject) {
			this.activeTexture(textureUnit);
			this.gl.bindTexture(textureTarget, textureObject);
			this.textureUnits[textureUnit][slot] = textureObject;
		}
	}
	setTextureParameters(texture) {
		const gl = this.gl;
		const flags = texture.impl.dirtyParameterFlags;
		const target = texture.impl._glTarget;
		if (flags & 1) {
			let filter = texture._minFilter;
			if (!texture.pot && !this.isWebGL2 || !texture._mipmaps || texture._compressed && texture._levels.length === 1) {
				if (filter === FILTER_NEAREST_MIPMAP_NEAREST || filter === FILTER_NEAREST_MIPMAP_LINEAR) {
					filter = FILTER_NEAREST;
				} else if (filter === FILTER_LINEAR_MIPMAP_NEAREST || filter === FILTER_LINEAR_MIPMAP_LINEAR) {
					filter = FILTER_LINEAR;
				}
			}
			gl.texParameteri(target, gl.TEXTURE_MIN_FILTER, this.glFilter[filter]);
		}
		if (flags & 2) {
			gl.texParameteri(target, gl.TEXTURE_MAG_FILTER, this.glFilter[texture._magFilter]);
		}
		if (flags & 4) {
			if (this.isWebGL2) {
				gl.texParameteri(target, gl.TEXTURE_WRAP_S, this.glAddress[texture._addressU]);
			} else {
				gl.texParameteri(target, gl.TEXTURE_WRAP_S, this.glAddress[texture.pot ? texture._addressU : ADDRESS_CLAMP_TO_EDGE]);
			}
		}
		if (flags & 8) {
			if (this.isWebGL2) {
				gl.texParameteri(target, gl.TEXTURE_WRAP_T, this.glAddress[texture._addressV]);
			} else {
				gl.texParameteri(target, gl.TEXTURE_WRAP_T, this.glAddress[texture.pot ? texture._addressV : ADDRESS_CLAMP_TO_EDGE]);
			}
		}
		if (flags & 16) {
			if (this.isWebGL2) {
				gl.texParameteri(target, gl.TEXTURE_WRAP_R, this.glAddress[texture._addressW]);
			}
		}
		if (flags & 32) {
			if (this.isWebGL2) {
				gl.texParameteri(target, gl.TEXTURE_COMPARE_MODE, texture._compareOnRead ? gl.COMPARE_REF_TO_TEXTURE : gl.NONE);
			}
		}
		if (flags & 64) {
			if (this.isWebGL2) {
				gl.texParameteri(target, gl.TEXTURE_COMPARE_FUNC, this.glComparison[texture._compareFunc]);
			}
		}
		if (flags & 128) {
			const ext = this.extTextureFilterAnisotropic;
			if (ext) {
				gl.texParameterf(target, ext.TEXTURE_MAX_ANISOTROPY_EXT, math.clamp(Math.round(texture._anisotropy), 1, this.maxAnisotropy));
			}
		}
	}
	setTexture(texture, textureUnit) {
		const impl = texture.impl;
		if (!impl._glTexture) impl.initialize(this, texture);
		if (impl.dirtyParameterFlags > 0 || texture._needsUpload || texture._needsMipmapsUpload) {
			this.activeTexture(textureUnit);
			this.bindTexture(texture);
			if (impl.dirtyParameterFlags) {
				this.setTextureParameters(texture);
				impl.dirtyParameterFlags = 0;
			}
			if (texture._needsUpload || texture._needsMipmapsUpload) {
				impl.upload(this, texture);
				texture._needsUpload = false;
				texture._needsMipmapsUpload = false;
			}
		} else {
			this.bindTextureOnUnit(texture, textureUnit);
		}
	}
	createVertexArray(vertexBuffers) {
		let key, vao;
		const useCache = vertexBuffers.length > 1;
		if (useCache) {
			key = "";
			for (let i = 0; i < vertexBuffers.length; i++) {
				const vertexBuffer = vertexBuffers[i];
				key += vertexBuffer.id + vertexBuffer.format.renderingHash;
			}
			vao = this._vaoMap.get(key);
		}
		if (!vao) {
			const gl = this.gl;
			vao = gl.createVertexArray();
			gl.bindVertexArray(vao);
			gl.bindBuffer(gl.ELEMENT_ARRAY_BUFFER, null);
			for (let i = 0; i < vertexBuffers.length; i++) {
				const vertexBuffer = vertexBuffers[i];
				gl.bindBuffer(gl.ARRAY_BUFFER, vertexBuffer.impl.bufferId);
				const elements = vertexBuffer.format.elements;
				for (let j = 0; j < elements.length; j++) {
					const e = elements[j];
					const loc = semanticToLocation[e.name];
					if (e.asInt) {
						gl.vertexAttribIPointer(loc, e.numComponents, this.glType[e.dataType], e.stride, e.offset);
					} else {
						gl.vertexAttribPointer(loc, e.numComponents, this.glType[e.dataType], e.normalize, e.stride, e.offset);
					}
					gl.enableVertexAttribArray(loc);
					if (vertexBuffer.format.instancing) {
						gl.vertexAttribDivisor(loc, 1);
					}
				}
			}
			gl.bindVertexArray(null);
			gl.bindBuffer(gl.ARRAY_BUFFER, null);
			if (useCache) {
				this._vaoMap.set(key, vao);
			}
		}
		return vao;
	}
	unbindVertexArray() {
		if (this.boundVao) {
			this.boundVao = null;
			this.gl.bindVertexArray(null);
		}
	}
	setBuffers() {
		const gl = this.gl;
		let vao;
		if (this.vertexBuffers.length === 1) {
			const vertexBuffer = this.vertexBuffers[0];
			if (!vertexBuffer.impl.vao) {
				vertexBuffer.impl.vao = this.createVertexArray(this.vertexBuffers);
			}
			vao = vertexBuffer.impl.vao;
		} else {
			vao = this.createVertexArray(this.vertexBuffers);
		}
		if (this.boundVao !== vao) {
			this.boundVao = vao;
			gl.bindVertexArray(vao);
		}
		this.vertexBuffers.length = 0;
		const bufferId = this.indexBuffer ? this.indexBuffer.impl.bufferId : null;
		gl.bindBuffer(gl.ELEMENT_ARRAY_BUFFER, bufferId);
	}
	draw(primitive, numInstances, keepBuffers) {
		const gl = this.gl;
		this.activateShader(this);
		if (!this.shaderValid) return;
		let sampler, samplerValue, texture, numTextures;
		let uniform, scopeId, uniformVersion, programVersion;
		const shader = this.shader;
		if (!shader) return;
		const samplers = shader.impl.samplers;
		const uniforms = shader.impl.uniforms;
		if (!keepBuffers) {
			this.setBuffers();
		}
		let textureUnit = 0;
		for (let i = 0, len = samplers.length; i < len; i++) {
			sampler = samplers[i];
			samplerValue = sampler.scopeId.value;
			if (!samplerValue) {
				return;
			}
			if (samplerValue instanceof Texture) {
				texture = samplerValue;
				this.setTexture(texture, textureUnit);
				if (sampler.slot !== textureUnit) {
					gl.uniform1i(sampler.locationId, textureUnit);
					sampler.slot = textureUnit;
				}
				textureUnit++;
			} else {
				sampler.array.length = 0;
				numTextures = samplerValue.length;
				for (let j = 0; j < numTextures; j++) {
					texture = samplerValue[j];
					this.setTexture(texture, textureUnit);
					sampler.array[j] = textureUnit;
					textureUnit++;
				}
				gl.uniform1iv(sampler.locationId, sampler.array);
			}
		}
		for (let i = 0, len = uniforms.length; i < len; i++) {
			uniform = uniforms[i];
			scopeId = uniform.scopeId;
			uniformVersion = uniform.version;
			programVersion = scopeId.versionObject.version;
			if (uniformVersion.globalId !== programVersion.globalId || uniformVersion.revision !== programVersion.revision) {
				uniformVersion.globalId = programVersion.globalId;
				uniformVersion.revision = programVersion.revision;
				if (scopeId.value !== null) {
					this.commitFunction[uniform.dataType](uniform, scopeId.value);
				}
			}
		}
		if (this.isWebGL2 && this.transformFeedbackBuffer) {
			gl.bindBufferBase(gl.TRANSFORM_FEEDBACK_BUFFER, 0, this.transformFeedbackBuffer.impl.bufferId);
			gl.beginTransformFeedback(gl.POINTS);
		}
		const mode = this.glPrimitive[primitive.type];
		const count = primitive.count;
		if (primitive.indexed) {
			const indexBuffer = this.indexBuffer;
			const format = indexBuffer.impl.glFormat;
			const offset = primitive.base * indexBuffer.bytesPerIndex;
			if (numInstances > 0) {
				gl.drawElementsInstanced(mode, count, format, offset, numInstances);
			} else {
				gl.drawElements(mode, count, format, offset);
			}
		} else {
			const first = primitive.base;
			if (numInstances > 0) {
				gl.drawArraysInstanced(mode, first, count, numInstances);
			} else {
				gl.drawArrays(mode, first, count);
			}
		}
		if (this.isWebGL2 && this.transformFeedbackBuffer) {
			gl.endTransformFeedback();
			gl.bindBufferBase(gl.TRANSFORM_FEEDBACK_BUFFER, 0, null);
		}
		this._drawCallsPerFrame++;
	}
	clear(options) {
		var _options$flags;
		const defaultOptions = this.defaultClearOptions;
		options = options || defaultOptions;
		const flags = (_options$flags = options.flags) != null ? _options$flags : defaultOptions.flags;
		if (flags !== 0) {
			const gl = this.gl;
			if (flags & CLEARFLAG_COLOR) {
				var _options$color;
				const color = (_options$color = options.color) != null ? _options$color : defaultOptions.color;
				const r = color[0];
				const g = color[1];
				const b = color[2];
				const a = color[3];
				const c = this.clearColor;
				if (r !== c.r || g !== c.g || b !== c.b || a !== c.a) {
					this.gl.clearColor(r, g, b, a);
					this.clearColor.set(r, g, b, a);
				}
				this.setBlendState(BlendState.NOBLEND);
			}
			if (flags & CLEARFLAG_DEPTH) {
				var _options$depth;
				const depth = (_options$depth = options.depth) != null ? _options$depth : defaultOptions.depth;
				if (depth !== this.clearDepth) {
					this.gl.clearDepth(depth);
					this.clearDepth = depth;
				}
				this.setDepthState(DepthState.WRITEDEPTH);
			}
			if (flags & CLEARFLAG_STENCIL) {
				var _options$stencil;
				const stencil = (_options$stencil = options.stencil) != null ? _options$stencil : defaultOptions.stencil;
				if (stencil !== this.clearStencil) {
					this.gl.clearStencil(stencil);
					this.clearStencil = stencil;
				}
			}
			gl.clear(this.glClearFlag[flags]);
		}
	}
	submit() {
		this.gl.flush();
	}
	readPixels(x, y, w, h, pixels) {
		const gl = this.gl;
		gl.readPixels(x, y, w, h, gl.RGBA, gl.UNSIGNED_BYTE, pixels);
	}
	async readPixelsAsync(x, y, w, h, pixels) {
		var _this$renderTarget$co, _impl$_glFormat, _impl$_glPixelType;
		const gl = this.gl;
		if (!this.isWebGL2) {
			this.readPixels(x, y, w, h, pixels);
			return;
		}
		const clientWaitAsync = (flags, interval_ms) => {
			const sync = gl.fenceSync(gl.SYNC_GPU_COMMANDS_COMPLETE, 0);
			this.submit();
			return new Promise((resolve, reject) => {
				function test() {
					const res = gl.clientWaitSync(sync, flags, 0);
					if (res === gl.WAIT_FAILED) {
						gl.deleteSync(sync);
						reject(new Error('webgl clientWaitSync sync failed'));
					} else if (res === gl.TIMEOUT_EXPIRED) {
						setTimeout(test, interval_ms);
					} else {
						gl.deleteSync(sync);
						resolve();
					}
				}
				test();
			});
		};
		const impl = (_this$renderTarget$co = this.renderTarget.colorBuffer) == null ? void 0 : _this$renderTarget$co.impl;
		const format = (_impl$_glFormat = impl == null ? void 0 : impl._glFormat) != null ? _impl$_glFormat : gl.RGBA;
		const pixelType = (_impl$_glPixelType = impl == null ? void 0 : impl._glPixelType) != null ? _impl$_glPixelType : gl.UNSIGNED_BYTE;
		const buf = gl.createBuffer();
		gl.bindBuffer(gl.PIXEL_PACK_BUFFER, buf);
		gl.bufferData(gl.PIXEL_PACK_BUFFER, pixels.byteLength, gl.STREAM_READ);
		gl.readPixels(x, y, w, h, format, pixelType, 0);
		gl.bindBuffer(gl.PIXEL_PACK_BUFFER, null);
		await clientWaitAsync(0, 20);
		gl.bindBuffer(gl.PIXEL_PACK_BUFFER, buf);
		gl.getBufferSubData(gl.PIXEL_PACK_BUFFER, 0, pixels);
		gl.bindBuffer(gl.PIXEL_PACK_BUFFER, null);
		gl.deleteBuffer(buf);
	}
	setAlphaToCoverage(state) {
		if (this.isWebGL1) return;
		if (this.alphaToCoverage === state) return;
		this.alphaToCoverage = state;
		if (state) {
			this.gl.enable(this.gl.SAMPLE_ALPHA_TO_COVERAGE);
		} else {
			this.gl.disable(this.gl.SAMPLE_ALPHA_TO_COVERAGE);
		}
	}
	setTransformFeedbackBuffer(tf) {
		if (this.transformFeedbackBuffer === tf) return;
		this.transformFeedbackBuffer = tf;
		if (this.isWebGL2) {
			const gl = this.gl;
			if (tf) {
				if (!this.feedback) {
					this.feedback = gl.createTransformFeedback();
				}
				gl.bindTransformFeedback(gl.TRANSFORM_FEEDBACK, this.feedback);
			} else {
				gl.bindTransformFeedback(gl.TRANSFORM_FEEDBACK, null);
			}
		}
	}
	setRaster(on) {
		if (this.raster === on) return;
		this.raster = on;
		if (this.isWebGL2) {
			if (on) {
				this.gl.disable(this.gl.RASTERIZER_DISCARD);
			} else {
				this.gl.enable(this.gl.RASTERIZER_DISCARD);
			}
		}
	}
	setStencilTest(enable) {
		if (this.stencil !== enable) {
			const gl = this.gl;
			if (enable) {
				gl.enable(gl.STENCIL_TEST);
			} else {
				gl.disable(gl.STENCIL_TEST);
			}
			this.stencil = enable;
		}
	}
	setStencilFunc(func, ref, mask) {
		if (this.stencilFuncFront !== func || this.stencilRefFront !== ref || this.stencilMaskFront !== mask || this.stencilFuncBack !== func || this.stencilRefBack !== ref || this.stencilMaskBack !== mask) {
			this.gl.stencilFunc(this.glComparison[func], ref, mask);
			this.stencilFuncFront = this.stencilFuncBack = func;
			this.stencilRefFront = this.stencilRefBack = ref;
			this.stencilMaskFront = this.stencilMaskBack = mask;
		}
	}
	setStencilFuncFront(func, ref, mask) {
		if (this.stencilFuncFront !== func || this.stencilRefFront !== ref || this.stencilMaskFront !== mask) {
			const gl = this.gl;
			gl.stencilFuncSeparate(gl.FRONT, this.glComparison[func], ref, mask);
			this.stencilFuncFront = func;
			this.stencilRefFront = ref;
			this.stencilMaskFront = mask;
		}
	}
	setStencilFuncBack(func, ref, mask) {
		if (this.stencilFuncBack !== func || this.stencilRefBack !== ref || this.stencilMaskBack !== mask) {
			const gl = this.gl;
			gl.stencilFuncSeparate(gl.BACK, this.glComparison[func], ref, mask);
			this.stencilFuncBack = func;
			this.stencilRefBack = ref;
			this.stencilMaskBack = mask;
		}
	}
	setStencilOperation(fail, zfail, zpass, writeMask) {
		if (this.stencilFailFront !== fail || this.stencilZfailFront !== zfail || this.stencilZpassFront !== zpass || this.stencilFailBack !== fail || this.stencilZfailBack !== zfail || this.stencilZpassBack !== zpass) {
			this.gl.stencilOp(this.glStencilOp[fail], this.glStencilOp[zfail], this.glStencilOp[zpass]);
			this.stencilFailFront = this.stencilFailBack = fail;
			this.stencilZfailFront = this.stencilZfailBack = zfail;
			this.stencilZpassFront = this.stencilZpassBack = zpass;
		}
		if (this.stencilWriteMaskFront !== writeMask || this.stencilWriteMaskBack !== writeMask) {
			this.gl.stencilMask(writeMask);
			this.stencilWriteMaskFront = writeMask;
			this.stencilWriteMaskBack = writeMask;
		}
	}
	setStencilOperationFront(fail, zfail, zpass, writeMask) {
		if (this.stencilFailFront !== fail || this.stencilZfailFront !== zfail || this.stencilZpassFront !== zpass) {
			this.gl.stencilOpSeparate(this.gl.FRONT, this.glStencilOp[fail], this.glStencilOp[zfail], this.glStencilOp[zpass]);
			this.stencilFailFront = fail;
			this.stencilZfailFront = zfail;
			this.stencilZpassFront = zpass;
		}
		if (this.stencilWriteMaskFront !== writeMask) {
			this.gl.stencilMaskSeparate(this.gl.FRONT, writeMask);
			this.stencilWriteMaskFront = writeMask;
		}
	}
	setStencilOperationBack(fail, zfail, zpass, writeMask) {
		if (this.stencilFailBack !== fail || this.stencilZfailBack !== zfail || this.stencilZpassBack !== zpass) {
			this.gl.stencilOpSeparate(this.gl.BACK, this.glStencilOp[fail], this.glStencilOp[zfail], this.glStencilOp[zpass]);
			this.stencilFailBack = fail;
			this.stencilZfailBack = zfail;
			this.stencilZpassBack = zpass;
		}
		if (this.stencilWriteMaskBack !== writeMask) {
			this.gl.stencilMaskSeparate(this.gl.BACK, writeMask);
			this.stencilWriteMaskBack = writeMask;
		}
	}
	setBlendState(blendState) {
		const currentBlendState = this.blendState;
		if (!currentBlendState.equals(blendState)) {
			const gl = this.gl;
			const {
				blend,
				colorOp,
				alphaOp,
				colorSrcFactor,
				colorDstFactor,
				alphaSrcFactor,
				alphaDstFactor
			} = blendState;
			if (currentBlendState.blend !== blend) {
				if (blend) {
					gl.enable(gl.BLEND);
				} else {
					gl.disable(gl.BLEND);
				}
			}
			if (currentBlendState.colorOp !== colorOp || currentBlendState.alphaOp !== alphaOp) {
				const glBlendEquation = this.glBlendEquation;
				gl.blendEquationSeparate(glBlendEquation[colorOp], glBlendEquation[alphaOp]);
			}
			if (currentBlendState.colorSrcFactor !== colorSrcFactor || currentBlendState.colorDstFactor !== colorDstFactor || currentBlendState.alphaSrcFactor !== alphaSrcFactor || currentBlendState.alphaDstFactor !== alphaDstFactor) {
				gl.blendFuncSeparate(this.glBlendFunctionColor[colorSrcFactor], this.glBlendFunctionColor[colorDstFactor], this.glBlendFunctionAlpha[alphaSrcFactor], this.glBlendFunctionAlpha[alphaDstFactor]);
			}
			if (currentBlendState.allWrite !== blendState.allWrite) {
				this.gl.colorMask(blendState.redWrite, blendState.greenWrite, blendState.blueWrite, blendState.alphaWrite);
			}
			currentBlendState.copy(blendState);
		}
	}
	setBlendColor(r, g, b, a) {
		const c = this.blendColor;
		if (r !== c.r || g !== c.g || b !== c.b || a !== c.a) {
			this.gl.blendColor(r, g, b, a);
			c.set(r, g, b, a);
		}
	}
	setStencilState(stencilFront, stencilBack) {
		if (stencilFront || stencilBack) {
			this.setStencilTest(true);
			if (stencilFront === stencilBack) {
				this.setStencilFunc(stencilFront.func, stencilFront.ref, stencilFront.readMask);
				this.setStencilOperation(stencilFront.fail, stencilFront.zfail, stencilFront.zpass, stencilFront.writeMask);
			} else {
				var _stencilFront, _stencilBack;
				(_stencilFront = stencilFront) != null ? _stencilFront : stencilFront = StencilParameters.DEFAULT;
				this.setStencilFuncFront(stencilFront.func, stencilFront.ref, stencilFront.readMask);
				this.setStencilOperationFront(stencilFront.fail, stencilFront.zfail, stencilFront.zpass, stencilFront.writeMask);
				(_stencilBack = stencilBack) != null ? _stencilBack : stencilBack = StencilParameters.DEFAULT;
				this.setStencilFuncBack(stencilBack.func, stencilBack.ref, stencilBack.readMask);
				this.setStencilOperationBack(stencilBack.fail, stencilBack.zfail, stencilBack.zpass, stencilBack.writeMask);
			}
		} else {
			this.setStencilTest(false);
		}
	}
	setDepthState(depthState) {
		const currentDepthState = this.depthState;
		if (!currentDepthState.equals(depthState)) {
			const gl = this.gl;
			const write = depthState.write;
			if (currentDepthState.write !== write) {
				gl.depthMask(write);
			}
			let {
				func,
				test
			} = depthState;
			if (!test && write) {
				test = true;
				func = FUNC_ALWAYS;
			}
			if (currentDepthState.func !== func) {
				gl.depthFunc(this.glComparison[func]);
			}
			if (currentDepthState.test !== test) {
				if (test) {
					gl.enable(gl.DEPTH_TEST);
				} else {
					gl.disable(gl.DEPTH_TEST);
				}
			}
			const {
				depthBias,
				depthBiasSlope
			} = depthState;
			if (depthBias || depthBiasSlope) {
				if (!this.depthBiasEnabled) {
					this.depthBiasEnabled = true;
					this.gl.enable(this.gl.POLYGON_OFFSET_FILL);
				}
				gl.polygonOffset(depthBiasSlope, depthBias);
			} else {
				if (this.depthBiasEnabled) {
					this.depthBiasEnabled = false;
					this.gl.disable(this.gl.POLYGON_OFFSET_FILL);
				}
			}
			currentDepthState.copy(depthState);
		}
	}
	setCullMode(cullMode) {
		if (this.cullMode !== cullMode) {
			if (cullMode === CULLFACE_NONE) {
				this.gl.disable(this.gl.CULL_FACE);
			} else {
				if (this.cullMode === CULLFACE_NONE) {
					this.gl.enable(this.gl.CULL_FACE);
				}
				const mode = this.glCull[cullMode];
				if (this.cullFace !== mode) {
					this.gl.cullFace(mode);
					this.cullFace = mode;
				}
			}
			this.cullMode = cullMode;
		}
	}
	setShader(shader, asyncCompile = false) {
		if (shader !== this.shader) {
			this.shader = shader;
			this.shaderAsyncCompile = asyncCompile;
			this.shaderValid = undefined;
		}
	}
	activateShader(device) {
		const {
			shader
		} = this;
		const {
			impl
		} = shader;
		if (this.shaderValid === undefined) {
			if (shader.failed) {
				this.shaderValid = false;
			} else if (!shader.ready) {
				if (this.shaderAsyncCompile) {
					if (impl.isLinked(device)) {
						if (!impl.finalize(this, shader)) {
							shader.failed = true;
							this.shaderValid = false;
						}
					} else {
						this.shaderValid = false;
					}
				} else {
					if (!impl.finalize(this, shader)) {
						shader.failed = true;
						this.shaderValid = false;
					}
				}
			}
		}
		if (this.shaderValid === undefined) {
			this.gl.useProgram(impl.glProgram);
			this.shaderValid = true;
		}
	}
	clearVertexArrayObjectCache() {
		const gl = this.gl;
		this._vaoMap.forEach((item, key, mapObj) => {
			gl.deleteVertexArray(item);
		});
		this._vaoMap.clear();
	}
	set fullscreen(fullscreen) {
		if (fullscreen) {
			const canvas = this.gl.canvas;
			canvas.requestFullscreen();
		} else {
			document.exitFullscreen();
		}
	}
	get fullscreen() {
		return !!document.fullscreenElement;
	}
	get textureFloatHighPrecision() {
		if (this._textureFloatHighPrecision === undefined) {
			this._textureFloatHighPrecision = testTextureFloatHighPrecision(this);
		}
		return this._textureFloatHighPrecision;
	}
	get textureHalfFloatUpdatable() {
		if (this._textureHalfFloatUpdatable === undefined) {
			if (this.isWebGL2) {
				this._textureHalfFloatUpdatable = true;
			} else {
				this._textureHalfFloatUpdatable = testTextureHalfFloatUpdatable(this.gl, this.extTextureHalfFloat.HALF_FLOAT_OES);
			}
		}
		return this._textureHalfFloatUpdatable;
	}
}

class NullIndexBuffer {
	unlock(indexBuffer) {}
}

class NullRenderTarget {
	destroy(device) {}
	init(device, renderTarget) {}
	loseContext() {}
	resolve(device, target, color, depth) {}
}

class NullShader {
	destroy(shader) {}
	loseContext() {}
	restoreContext(device, shader) {}
}

class NullTexture {
	destroy(device) {}
	propertyChanged(flag) {}
	loseContext() {}
}

class NullVertexBuffer {
	destroy(device) {}
	unlock(vertexBuffer) {}
}

class NullGraphicsDevice extends GraphicsDevice {
	constructor(canvas, options = {}) {
		super(canvas, options);
		options = this.initOptions;
		this.isNull = true;
		this._deviceType = DEVICETYPE_NULL;
		this.samples = 1;
	}
	destroy() {
		super.destroy();
	}
	initDeviceCaps() {
		this.disableParticleSystem = true;
		this.precision = 'highp';
		this.maxPrecision = 'highp';
		this.maxSamples = 4;
		this.maxTextures = 16;
		this.maxTextureSize = 4096;
		this.maxCubeMapSize = 4096;
		this.maxVolumeSize = 4096;
		this.maxColorAttachments = 8;
		this.maxPixelRatio = 1;
		this.maxAnisotropy = 16;
		this.supportsInstancing = true;
		this.supportsUniformBuffers = false;
		this.supportsVolumeTextures = true;
		this.supportsBoneTextures = true;
		this.supportsMorphTargetTexturesCore = true;
		this.supportsAreaLights = true;
		this.supportsDepthShadow = true;
		this.supportsGpuParticles = false;
		this.supportsMrt = true;
		this.extUintElement = true;
		this.extTextureFloat = true;
		this.textureFloatRenderable = true;
		this.extTextureHalfFloat = true;
		this.textureHalfFloatRenderable = true;
		this.textureHalfFloatUpdatable = true;
		this.boneLimit = 1024;
		this.supportsImageBitmap = true;
		this.extStandardDerivatives = true;
		this.extBlendMinmax = true;
		this.areaLightLutFormat = PIXELFORMAT_RGBA8;
		this.supportsTextureFetch = true;
	}
	postInit() {
		super.postInit();
	}
	frameStart() {
		super.frameStart();
	}
	frameEnd() {
		super.frameEnd();
	}
	updateBegin() {}
	updateEnd() {}
	readPixels(x, y, w, h, pixels) {}
	createVertexBufferImpl(vertexBuffer, format) {
		return new NullVertexBuffer(vertexBuffer, format);
	}
	createIndexBufferImpl(indexBuffer) {
		return new NullIndexBuffer(indexBuffer);
	}
	createShaderImpl(shader) {
		return new NullShader(shader);
	}
	createTextureImpl(texture) {
		return new NullTexture(texture);
	}
	createRenderTargetImpl(renderTarget) {
		return new NullRenderTarget(renderTarget);
	}
	draw(primitive, numInstances = 1, keepBuffers) {}
	setShader(shader, asyncCompile = false) {}
	setBlendState(blendState) {}
	setDepthState(depthState) {}
	setStencilState(stencilFront, stencilBack) {}
	setBlendColor(r, g, b, a) {}
	setCullMode(cullMode) {}
	setAlphaToCoverage(state) {}
	initializeContextCaches() {
		super.initializeContextCaches();
	}
	clear(options) {}
	setViewport(x, y, w, h) {}
	setScissor(x, y, w, h) {}
	copyRenderTarget(source, dest, color, depth) {
		return true;
	}
}

function createGraphicsDevice(canvas, options = {}) {
	var _options$deviceTypes;
	const deviceTypes = (_options$deviceTypes = options.deviceTypes) != null ? _options$deviceTypes : [];
	if (!deviceTypes.includes(DEVICETYPE_WEBGL2)) {
		deviceTypes.push(DEVICETYPE_WEBGL2);
	}
	if (!deviceTypes.includes(DEVICETYPE_WEBGL1)) {
		deviceTypes.push(DEVICETYPE_WEBGL1);
	}
	if (!deviceTypes.includes(DEVICETYPE_NULL)) {
		deviceTypes.push(DEVICETYPE_NULL);
	}
	if (platform.browser && !!navigator.xr) {
		var _options$xrCompatible;
		(_options$xrCompatible = options.xrCompatible) != null ? _options$xrCompatible : options.xrCompatible = true;
	}
	const deviceCreateFuncs = [];
	for (let i = 0; i < deviceTypes.length; i++) {
		var _window;
		const deviceType = deviceTypes[i];
		if (deviceType === DEVICETYPE_WEBGPU && (_window = window) != null && (_window = _window.navigator) != null && _window.gpu) {
			deviceCreateFuncs.push(() => {
				const device = new WebgpuGraphicsDevice(canvas, options);
				return device.initWebGpu(options.glslangUrl, options.twgslUrl);
			});
		}
		if (deviceType === DEVICETYPE_WEBGL1 || deviceType === DEVICETYPE_WEBGL2) {
			deviceCreateFuncs.push(() => {
				options.preferWebGl2 = deviceType === DEVICETYPE_WEBGL2;
				return new WebglGraphicsDevice(canvas, options);
			});
		}
		if (deviceType === DEVICETYPE_NULL) {
			deviceCreateFuncs.push(() => {
				return new NullGraphicsDevice(canvas, options);
			});
		}
	}
	return new Promise((resolve, reject) => {
		let attempt = 0;
		const next = () => {
			if (attempt >= deviceCreateFuncs.length) {
				reject(new Error('Failed to create a graphics device'));
			} else {
				Promise.resolve(deviceCreateFuncs[attempt++]()).then(device => {
					if (device) {
						resolve(device);
					} else {
						next();
					}
				}).catch(err => {
					console.log(err);
					next();
				});
			}
		};
		next();
	});
}

class Compute {
	constructor(graphicsDevice, shader) {
		this.shader = null;
		this.device = graphicsDevice;
		this.shader = shader;
		if (graphicsDevice.supportsCompute) {
			this.impl = graphicsDevice.createComputeImpl(this);
		}
	}
	dispatch(x, y, z) {
		var _this$impl;
		(_this$impl = this.impl) == null || _this$impl.dispatch(x, y, z);
	}
}

let id$4 = 0;
class IndexBuffer {
	constructor(graphicsDevice, format, numIndices, usage = BUFFER_STATIC, initialData) {
		this.device = graphicsDevice;
		this.format = format;
		this.numIndices = numIndices;
		this.usage = usage;
		this.id = id$4++;
		this.impl = graphicsDevice.createIndexBufferImpl(this);
		const bytesPerIndex = typedArrayIndexFormatsByteSize[format];
		this.bytesPerIndex = bytesPerIndex;
		this.numBytes = this.numIndices * bytesPerIndex;
		if (initialData) {
			this.setData(initialData);
		} else {
			this.storage = new ArrayBuffer(this.numBytes);
		}
		this.adjustVramSizeTracking(graphicsDevice._vram, this.numBytes);
		this.device.buffers.push(this);
	}
	destroy() {
		const device = this.device;
		const idx = device.buffers.indexOf(this);
		if (idx !== -1) {
			device.buffers.splice(idx, 1);
		}
		if (this.device.indexBuffer === this) {
			this.device.indexBuffer = null;
		}
		if (this.impl.initialized) {
			this.impl.destroy(device);
			this.adjustVramSizeTracking(device._vram, -this.storage.byteLength);
		}
	}
	adjustVramSizeTracking(vram, size) {
		vram.ib += size;
	}
	loseContext() {
		this.impl.loseContext();
	}
	getFormat() {
		return this.format;
	}
	getNumIndices() {
		return this.numIndices;
	}
	lock() {
		return this.storage;
	}
	unlock() {
		this.impl.unlock(this);
	}
	setData(data) {
		if (data.byteLength !== this.numBytes) {
			return false;
		}
		this.storage = data;
		this.unlock();
		return true;
	}
	_lockTypedArray() {
		const lock = this.lock();
		const indices = this.format === INDEXFORMAT_UINT32 ? new Uint32Array(lock) : this.format === INDEXFORMAT_UINT16 ? new Uint16Array(lock) : new Uint8Array(lock);
		return indices;
	}
	writeData(data, count) {
		const indices = this._lockTypedArray();
		if (data.length > count) {
			if (ArrayBuffer.isView(data)) {
				data = data.subarray(0, count);
				indices.set(data);
			} else {
				for (let i = 0; i < count; i++) indices[i] = data[i];
			}
		} else {
			indices.set(data);
		}
		this.unlock();
	}
	readData(data) {
		const indices = this._lockTypedArray();
		const count = this.numIndices;
		if (ArrayBuffer.isView(data)) {
			data.set(indices);
		} else {
			data.length = 0;
			for (let i = 0; i < count; i++) data[i] = indices[i];
		}
		return count;
	}
}

class ColorAttachmentOps {
	constructor() {
		this.clearValue = new Color(0, 0, 0, 1);
		this.clear = false;
		this.store = false;
		this.resolve = true;
		this.mipmaps = false;
	}
}
class DepthStencilAttachmentOps {
	constructor() {
		this.clearDepthValue = 1;
		this.clearStencilValue = 0;
		this.clearDepth = false;
		this.clearStencil = false;
		this.storeDepth = false;
		this.storeStencil = false;
	}
}
class RenderPass {
	get colorOps() {
		return this.colorArrayOps[0];
	}
	constructor(graphicsDevice) {
		this._name = void 0;
		this.device = void 0;
		this._enabled = true;
		this.executeEnabled = true;
		this.renderTarget = void 0;
		this._options = void 0;
		this.samples = 0;
		this.colorArrayOps = [];
		this.depthStencilOps = void 0;
		this.requiresCubemaps = true;
		this.fullSizeClearRect = true;
		this.beforePasses = [];
		this.afterPasses = [];
		this.device = graphicsDevice;
	}
	set name(value) {
		this._name = value;
	}
	get name() {
		if (!this._name) this._name = this.constructor.name;
		return this._name;
	}
	set options(value) {
		this._options = value;
		if (value) {
			var _this$_options$scaleX, _this$_options$scaleY;
			this._options.scaleX = (_this$_options$scaleX = this._options.scaleX) != null ? _this$_options$scaleX : 1;
			this._options.scaleY = (_this$_options$scaleY = this._options.scaleY) != null ? _this$_options$scaleY : 1;
		}
	}
	get options() {
		return this._options;
	}
	init(renderTarget = null, options = null) {
		var _renderTarget$_colorB;
		this.options = options;
		this.renderTarget = renderTarget;
		this.samples = Math.max(this.renderTarget ? this.renderTarget.samples : this.device.samples, 1);
		this.depthStencilOps = new DepthStencilAttachmentOps();
		const numColorOps = renderTarget ? (_renderTarget$_colorB = renderTarget._colorBuffers) == null ? void 0 : _renderTarget$_colorB.length : 1;
		this.colorArrayOps.length = 0;
		for (let i = 0; i < numColorOps; i++) {
			var _this$renderTarget;
			const colorOps = new ColorAttachmentOps();
			this.colorArrayOps[i] = colorOps;
			if (this.samples === 1) {
				colorOps.store = true;
				colorOps.resolve = false;
			}
			if ((_this$renderTarget = this.renderTarget) != null && (_this$renderTarget = _this$renderTarget._colorBuffers) != null && _this$renderTarget[i].mipmaps) {
				colorOps.mipmaps = true;
			}
		}
		this.postInit();
	}
	destroy() {}
	postInit() {}
	frameUpdate() {
		if (this._options && this.renderTarget) {
			var _this$_options$resize;
			const resizeSource = (_this$_options$resize = this._options.resizeSource) != null ? _this$_options$resize : this.device.backBuffer;
			const width = Math.floor(resizeSource.width * this._options.scaleX);
			const height = Math.floor(resizeSource.height * this._options.scaleY);
			this.renderTarget.resize(width, height);
		}
	}
	before() {}
	execute() {}
	after() {}
	onEnable() {}
	onDisable() {}
	set enabled(value) {
		if (this._enabled !== value) {
			this._enabled = value;
			if (value) {
				this.onEnable();
			} else {
				this.onDisable();
			}
		}
	}
	get enabled() {
		return this._enabled;
	}
	setClearColor(color) {
		const count = this.colorArrayOps.length;
		for (let i = 0; i < count; i++) {
			const colorOps = this.colorArrayOps[i];
			if (color) colorOps.clearValue.copy(color);
			colorOps.clear = !!color;
		}
	}
	setClearDepth(depthValue) {
		if (depthValue) this.depthStencilOps.clearDepthValue = depthValue;
		this.depthStencilOps.clearDepth = depthValue !== undefined;
	}
	setClearStencil(stencilValue) {
		if (stencilValue) this.depthStencilOps.clearStencilValue = stencilValue;
		this.depthStencilOps.clearStencil = stencilValue !== undefined;
	}
	render() {
		if (this.enabled) {
			const device = this.device;
			const realPass = this.renderTarget !== undefined;
			this.before();
			if (this.executeEnabled) {
				if (realPass) {
					device.startRenderPass(this);
				}
				this.execute();
				if (realPass) {
					device.endRenderPass(this);
				}
			}
			this.after();
			device.renderPassIndex++;
		}
	}
}

class ShaderProcessorOptions {
	constructor(viewUniformFormat, viewBindGroupFormat, vertexFormat) {
		this.uniformFormats = [];
		this.bindGroupFormats = [];
		this.vertexFormat = void 0;
		this.uniformFormats[BINDGROUP_VIEW] = viewUniformFormat;
		this.bindGroupFormats[BINDGROUP_VIEW] = viewBindGroupFormat;
		this.vertexFormat = vertexFormat;
	}
	hasUniform(name) {
		for (let i = 0; i < this.uniformFormats.length; i++) {
			const uniformFormat = this.uniformFormats[i];
			if (uniformFormat != null && uniformFormat.get(name)) {
				return true;
			}
		}
		return false;
	}
	hasTexture(name) {
		for (let i = 0; i < this.bindGroupFormats.length; i++) {
			const groupFormat = this.bindGroupFormats[i];
			if (groupFormat != null && groupFormat.getTexture(name)) {
				return true;
			}
		}
		return false;
	}
	getVertexElement(semantic) {
		var _this$vertexFormat;
		return (_this$vertexFormat = this.vertexFormat) == null ? void 0 : _this$vertexFormat.elements.find(element => element.name === semantic);
	}
	generateKey(device) {
		let key = JSON.stringify(this.uniformFormats) + JSON.stringify(this.bindGroupFormats);
		if (device.isWebGPU) {
			var _this$vertexFormat2;
			key += (_this$vertexFormat2 = this.vertexFormat) == null ? void 0 : _this$vertexFormat2.shaderProcessingHashString;
		}
		return key;
	}
}

class TransformFeedback {
	constructor(inputBuffer, usage = BUFFER_GPUDYNAMIC) {
		this.device = inputBuffer.device;
		const gl = this.device.gl;
		this._inputBuffer = inputBuffer;
		if (usage === BUFFER_GPUDYNAMIC && inputBuffer.usage !== usage) {
			gl.bindBuffer(gl.ARRAY_BUFFER, inputBuffer.impl.bufferId);
			gl.bufferData(gl.ARRAY_BUFFER, inputBuffer.storage, gl.DYNAMIC_COPY);
		}
		this._outputBuffer = new VertexBuffer(inputBuffer.device, inputBuffer.format, inputBuffer.numVertices, usage, inputBuffer.storage);
	}
	static createShader(graphicsDevice, vertexCode, name) {
		return new Shader(graphicsDevice, ShaderUtils.createDefinition(graphicsDevice, {
			name,
			vertexCode,
			useTransformFeedback: true
		}));
	}
	destroy() {
		this._outputBuffer.destroy();
	}
	process(shader, swap = true) {
		const device = this.device;
		const oldRt = device.getRenderTarget();
		device.setRenderTarget(null);
		device.updateBegin();
		device.setVertexBuffer(this._inputBuffer, 0);
		device.setRaster(false);
		device.setTransformFeedbackBuffer(this._outputBuffer);
		device.setShader(shader);
		device.draw({
			type: PRIMITIVE_POINTS,
			base: 0,
			count: this._inputBuffer.numVertices,
			indexed: false
		});
		device.setTransformFeedbackBuffer(null);
		device.setRaster(true);
		device.updateEnd();
		device.setRenderTarget(oldRt);
		if (swap) {
			let tmp = this._inputBuffer.impl.bufferId;
			this._inputBuffer.impl.bufferId = this._outputBuffer.impl.bufferId;
			this._outputBuffer.impl.bufferId = tmp;
			tmp = this._inputBuffer.impl.vao;
			this._inputBuffer.impl.vao = this._outputBuffer.impl.vao;
			this._outputBuffer.impl.vao = tmp;
		}
	}
	get inputBuffer() {
		return this._inputBuffer;
	}
	get outputBuffer() {
		return this._outputBuffer;
	}
}

function set1(a) {
	this.array[this.index] = a;
}
function set2(a, b) {
	this.array[this.index] = a;
	this.array[this.index + 1] = b;
}
function set3(a, b, c) {
	this.array[this.index] = a;
	this.array[this.index + 1] = b;
	this.array[this.index + 2] = c;
}
function set4(a, b, c, d) {
	this.array[this.index] = a;
	this.array[this.index + 1] = b;
	this.array[this.index + 2] = c;
	this.array[this.index + 3] = d;
}
function arraySet1(index, inputArray, inputIndex) {
	this.array[index] = inputArray[inputIndex];
}
function arraySet2(index, inputArray, inputIndex) {
	this.array[index] = inputArray[inputIndex];
	this.array[index + 1] = inputArray[inputIndex + 1];
}
function arraySet3(index, inputArray, inputIndex) {
	this.array[index] = inputArray[inputIndex];
	this.array[index + 1] = inputArray[inputIndex + 1];
	this.array[index + 2] = inputArray[inputIndex + 2];
}
function arraySet4(index, inputArray, inputIndex) {
	this.array[index] = inputArray[inputIndex];
	this.array[index + 1] = inputArray[inputIndex + 1];
	this.array[index + 2] = inputArray[inputIndex + 2];
	this.array[index + 3] = inputArray[inputIndex + 3];
}
function arrayGet1(offset, outputArray, outputIndex) {
	outputArray[outputIndex] = this.array[offset];
}
function arrayGet2(offset, outputArray, outputIndex) {
	outputArray[outputIndex] = this.array[offset];
	outputArray[outputIndex + 1] = this.array[offset + 1];
}
function arrayGet3(offset, outputArray, outputIndex) {
	outputArray[outputIndex] = this.array[offset];
	outputArray[outputIndex + 1] = this.array[offset + 1];
	outputArray[outputIndex + 2] = this.array[offset + 2];
}
function arrayGet4(offset, outputArray, outputIndex) {
	outputArray[outputIndex] = this.array[offset];
	outputArray[outputIndex + 1] = this.array[offset + 1];
	outputArray[outputIndex + 2] = this.array[offset + 2];
	outputArray[outputIndex + 3] = this.array[offset + 3];
}
class VertexIteratorAccessor {
	constructor(buffer, vertexElement, vertexFormat) {
		this.index = 0;
		this.numComponents = vertexElement.numComponents;
		if (vertexFormat.interleaved) {
			this.array = new typedArrayTypes[vertexElement.dataType](buffer, vertexElement.offset);
		} else {
			this.array = new typedArrayTypes[vertexElement.dataType](buffer, vertexElement.offset, vertexFormat.vertexCount * vertexElement.numComponents);
		}
		this.stride = vertexElement.stride / this.array.constructor.BYTES_PER_ELEMENT;
		switch (vertexElement.numComponents) {
			case 1:
				this.set = set1;
				this.getToArray = arrayGet1;
				this.setFromArray = arraySet1;
				break;
			case 2:
				this.set = set2;
				this.getToArray = arrayGet2;
				this.setFromArray = arraySet2;
				break;
			case 3:
				this.set = set3;
				this.getToArray = arrayGet3;
				this.setFromArray = arraySet3;
				break;
			case 4:
				this.set = set4;
				this.getToArray = arrayGet4;
				this.setFromArray = arraySet4;
				break;
		}
	}
	get(offset) {
		return this.array[this.index + offset];
	}
	set(a, b, c, d) {}
	getToArray(offset, outputArray, outputIndex) {}
	setFromArray(index, inputArray, inputIndex) {}
}
class VertexIterator {
	constructor(vertexBuffer) {
		this.vertexBuffer = vertexBuffer;
		this.vertexFormatSize = vertexBuffer.getFormat().size;
		this.buffer = this.vertexBuffer.lock();
		this.accessors = [];
		this.element = {};
		const vertexFormat = this.vertexBuffer.getFormat();
		for (let i = 0; i < vertexFormat.elements.length; i++) {
			const vertexElement = vertexFormat.elements[i];
			this.accessors[i] = new VertexIteratorAccessor(this.buffer, vertexElement, vertexFormat);
			this.element[vertexElement.name] = this.accessors[i];
		}
	}
	next(count = 1) {
		let i = 0;
		const accessors = this.accessors;
		const numAccessors = this.accessors.length;
		while (i < numAccessors) {
			const accessor = accessors[i++];
			accessor.index += count * accessor.stride;
		}
	}
	end() {
		this.vertexBuffer.unlock();
	}
	writeData(semantic, data, numVertices) {
		const element = this.element[semantic];
		if (element) {
			if (numVertices > this.vertexBuffer.numVertices) {
				numVertices = this.vertexBuffer.numVertices;
			}
			const numComponents = element.numComponents;
			if (this.vertexBuffer.getFormat().interleaved) {
				let index = 0;
				for (let i = 0; i < numVertices; i++) {
					element.setFromArray(index, data, i * numComponents);
					index += element.stride;
				}
			} else {
				if (data.length > numVertices * numComponents) {
					const copyCount = numVertices * numComponents;
					if (ArrayBuffer.isView(data)) {
						data = data.subarray(0, copyCount);
						element.array.set(data);
					} else {
						for (let i = 0; i < copyCount; i++) element.array[i] = data[i];
					}
				} else {
					element.array.set(data);
				}
			}
		}
	}
	readData(semantic, data) {
		const element = this.element[semantic];
		let count = 0;
		if (element) {
			count = this.vertexBuffer.numVertices;
			let i;
			const numComponents = element.numComponents;
			if (this.vertexBuffer.getFormat().interleaved) {
				if (Array.isArray(data)) data.length = 0;
				element.index = 0;
				let offset = 0;
				for (i = 0; i < count; i++) {
					element.getToArray(offset, data, i * numComponents);
					offset += element.stride;
				}
			} else {
				if (ArrayBuffer.isView(data)) {
					data.set(element.array);
				} else {
					data.length = 0;
					const copyCount = count * numComponents;
					for (i = 0; i < copyCount; i++) data[i] = element.array[i];
				}
			}
		}
		return count;
	}
}

const ACTION_MOUSE = 'mouse';
const ACTION_KEYBOARD = 'keyboard';
const ACTION_GAMEPAD = 'gamepad';
const AXIS_MOUSE_X = 'mousex';
const AXIS_MOUSE_Y = 'mousey';
const AXIS_PAD_L_X = 'padlx';
const AXIS_PAD_L_Y = 'padly';
const AXIS_PAD_R_X = 'padrx';
const AXIS_PAD_R_Y = 'padry';
const AXIS_KEY = 'key';
const EVENT_KEYDOWN = 'keydown';
const EVENT_KEYUP = 'keyup';
const EVENT_MOUSEDOWN = 'mousedown';
const EVENT_MOUSEMOVE = 'mousemove';
const EVENT_MOUSEUP = 'mouseup';
const EVENT_MOUSEWHEEL = 'mousewheel';
const EVENT_TOUCHSTART = 'touchstart';
const EVENT_TOUCHEND = 'touchend';
const EVENT_TOUCHMOVE = 'touchmove';
const EVENT_TOUCHCANCEL = 'touchcancel';
const EVENT_SELECT = 'select';
const EVENT_SELECTSTART = 'selectstart';
const EVENT_SELECTEND = 'selectend';
const KEY_BACKSPACE = 8;
const KEY_TAB = 9;
const KEY_RETURN = 13;
const KEY_ENTER = 13;
const KEY_SHIFT = 16;
const KEY_CONTROL = 17;
const KEY_ALT = 18;
const KEY_PAUSE = 19;
const KEY_CAPS_LOCK = 20;
const KEY_ESCAPE = 27;
const KEY_SPACE = 32;
const KEY_PAGE_UP = 33;
const KEY_PAGE_DOWN = 34;
const KEY_END = 35;
const KEY_HOME = 36;
const KEY_LEFT = 37;
const KEY_UP = 38;
const KEY_RIGHT = 39;
const KEY_DOWN = 40;
const KEY_PRINT_SCREEN = 44;
const KEY_INSERT = 45;
const KEY_DELETE = 46;
const KEY_0 = 48;
const KEY_1 = 49;
const KEY_2 = 50;
const KEY_3 = 51;
const KEY_4 = 52;
const KEY_5 = 53;
const KEY_6 = 54;
const KEY_7 = 55;
const KEY_8 = 56;
const KEY_9 = 57;
const KEY_SEMICOLON = 59;
const KEY_EQUAL = 61;
const KEY_A = 65;
const KEY_B = 66;
const KEY_C = 67;
const KEY_D = 68;
const KEY_E = 69;
const KEY_F = 70;
const KEY_G = 71;
const KEY_H = 72;
const KEY_I = 73;
const KEY_J = 74;
const KEY_K = 75;
const KEY_L = 76;
const KEY_M = 77;
const KEY_N = 78;
const KEY_O = 79;
const KEY_P = 80;
const KEY_Q = 81;
const KEY_R = 82;
const KEY_S = 83;
const KEY_T = 84;
const KEY_U = 85;
const KEY_V = 86;
const KEY_W = 87;
const KEY_X = 88;
const KEY_Y = 89;
const KEY_Z = 90;
const KEY_WINDOWS = 91;
const KEY_CONTEXT_MENU = 93;
const KEY_NUMPAD_0 = 96;
const KEY_NUMPAD_1 = 97;
const KEY_NUMPAD_2 = 98;
const KEY_NUMPAD_3 = 99;
const KEY_NUMPAD_4 = 100;
const KEY_NUMPAD_5 = 101;
const KEY_NUMPAD_6 = 102;
const KEY_NUMPAD_7 = 103;
const KEY_NUMPAD_8 = 104;
const KEY_NUMPAD_9 = 105;
const KEY_MULTIPLY = 106;
const KEY_ADD = 107;
const KEY_SEPARATOR = 108;
const KEY_SUBTRACT = 109;
const KEY_DECIMAL = 110;
const KEY_DIVIDE = 111;
const KEY_F1 = 112;
const KEY_F2 = 113;
const KEY_F3 = 114;
const KEY_F4 = 115;
const KEY_F5 = 116;
const KEY_F6 = 117;
const KEY_F7 = 118;
const KEY_F8 = 119;
const KEY_F9 = 120;
const KEY_F10 = 121;
const KEY_F11 = 122;
const KEY_F12 = 123;
const KEY_COMMA = 188;
const KEY_PERIOD = 190;
const KEY_SLASH = 191;
const KEY_OPEN_BRACKET = 219;
const KEY_BACK_SLASH = 220;
const KEY_CLOSE_BRACKET = 221;
const KEY_META = 224;
const MOUSEBUTTON_NONE = -1;
const MOUSEBUTTON_LEFT = 0;
const MOUSEBUTTON_MIDDLE = 1;
const MOUSEBUTTON_RIGHT = 2;
const PAD_1 = 0;
const PAD_2 = 1;
const PAD_3 = 2;
const PAD_4 = 3;
const PAD_FACE_1 = 0;
const PAD_FACE_2 = 1;
const PAD_FACE_3 = 2;
const PAD_FACE_4 = 3;
const PAD_L_SHOULDER_1 = 4;
const PAD_R_SHOULDER_1 = 5;
const PAD_L_SHOULDER_2 = 6;
const PAD_R_SHOULDER_2 = 7;
const PAD_SELECT = 8;
const PAD_START = 9;
const PAD_L_STICK_BUTTON = 10;
const PAD_R_STICK_BUTTON = 11;
const PAD_UP = 12;
const PAD_DOWN = 13;
const PAD_LEFT = 14;
const PAD_RIGHT = 15;
const PAD_VENDOR = 16;
const PAD_L_STICK_X = 0;
const PAD_L_STICK_Y = 1;
const PAD_R_STICK_X = 2;
const PAD_R_STICK_Y = 3;
const EVENT_GAMEPADCONNECTED = 'gamepadconnected';
const EVENT_GAMEPADDISCONNECTED = 'gamepaddisconnected';
const XRPAD_TOUCHPAD_X = 0;
const XRPAD_TOUCHPAD_Y = 1;
const XRPAD_STICK_X = 2;
const XRPAD_STICK_Y = 3;
const XRPAD_TOUCHPAD_BUTTON = 2;
const XRPAD_TRIGGER = 0;
const XRPAD_SQUEEZE = 1;
const XRPAD_STICK_BUTTON = 3;
const XRPAD_A = 4;
const XRPAD_B = 5;

class KeyboardEvent {
	constructor(keyboard, event) {
		if (event) {
			this.key = event.keyCode;
			this.element = event.target;
			this.event = event;
		} else {
			this.key = null;
			this.element = null;
			this.event = null;
		}
	}
}

const _keyboardEvent = new KeyboardEvent();
function makeKeyboardEvent(event) {
	_keyboardEvent.key = event.keyCode;
	_keyboardEvent.element = event.target;
	_keyboardEvent.event = event;
	return _keyboardEvent;
}
function toKeyCode(s) {
	if (typeof s === 'string') {
		return s.toUpperCase().charCodeAt(0);
	}
	return s;
}
const _keyCodeToKeyIdentifier = {
	'9': 'Tab',
	'13': 'Enter',
	'16': 'Shift',
	'17': 'Control',
	'18': 'Alt',
	'27': 'Escape',
	'37': 'Left',
	'38': 'Up',
	'39': 'Right',
	'40': 'Down',
	'46': 'Delete',
	'91': 'Win'
};
class Keyboard extends EventHandler {
	constructor(element, options = {}) {
		super();
		this._element = null;
		this._keyDownHandler = this._handleKeyDown.bind(this);
		this._keyUpHandler = this._handleKeyUp.bind(this);
		this._keyPressHandler = this._handleKeyPress.bind(this);
		this._visibilityChangeHandler = this._handleVisibilityChange.bind(this);
		this._windowBlurHandler = this._handleWindowBlur.bind(this);
		this._keymap = {};
		this._lastmap = {};
		if (element) {
			this.attach(element);
		}
		this.preventDefault = options.preventDefault || false;
		this.stopPropagation = options.stopPropagation || false;
	}
	attach(element) {
		if (this._element) {
			this.detach();
		}
		this._element = element;
		this._element.addEventListener('keydown', this._keyDownHandler, false);
		this._element.addEventListener('keypress', this._keyPressHandler, false);
		this._element.addEventListener('keyup', this._keyUpHandler, false);
		document.addEventListener('visibilitychange', this._visibilityChangeHandler, false);
		window.addEventListener('blur', this._windowBlurHandler, false);
	}
	detach() {
		if (!this._element) {
			return;
		}
		this._element.removeEventListener('keydown', this._keyDownHandler);
		this._element.removeEventListener('keypress', this._keyPressHandler);
		this._element.removeEventListener('keyup', this._keyUpHandler);
		this._element = null;
		document.removeEventListener('visibilitychange', this._visibilityChangeHandler, false);
		window.removeEventListener('blur', this._windowBlurHandler, false);
	}
	toKeyIdentifier(keyCode) {
		keyCode = toKeyCode(keyCode);
		const id = _keyCodeToKeyIdentifier[keyCode.toString()];
		if (id) {
			return id;
		}
		let hex = keyCode.toString(16).toUpperCase();
		const length = hex.length;
		for (let count = 0; count < 4 - length; count++) {
			hex = '0' + hex;
		}
		return 'U+' + hex;
	}
	_handleKeyDown(event) {
		const code = event.keyCode || event.charCode;
		if (code === undefined) return;
		const id = this.toKeyIdentifier(code);
		this._keymap[id] = true;
		this.fire('keydown', makeKeyboardEvent(event));
		if (this.preventDefault) {
			event.preventDefault();
		}
		if (this.stopPropagation) {
			event.stopPropagation();
		}
	}
	_handleKeyUp(event) {
		const code = event.keyCode || event.charCode;
		if (code === undefined) return;
		const id = this.toKeyIdentifier(code);
		delete this._keymap[id];
		this.fire('keyup', makeKeyboardEvent(event));
		if (this.preventDefault) {
			event.preventDefault();
		}
		if (this.stopPropagation) {
			event.stopPropagation();
		}
	}
	_handleKeyPress(event) {
		this.fire('keypress', makeKeyboardEvent(event));
		if (this.preventDefault) {
			event.preventDefault();
		}
		if (this.stopPropagation) {
			event.stopPropagation();
		}
	}
	_handleVisibilityChange() {
		if (document.visibilityState === 'hidden') {
			this._handleWindowBlur();
		}
	}
	_handleWindowBlur() {
		this._keymap = {};
		this._lastmap = {};
	}
	update() {
		for (const prop in this._lastmap) {
			delete this._lastmap[prop];
		}
		for (const prop in this._keymap) {
			if (this._keymap.hasOwnProperty(prop)) {
				this._lastmap[prop] = this._keymap[prop];
			}
		}
	}
	isPressed(key) {
		const keyCode = toKeyCode(key);
		const id = this.toKeyIdentifier(keyCode);
		return !!this._keymap[id];
	}
	wasPressed(key) {
		const keyCode = toKeyCode(key);
		const id = this.toKeyIdentifier(keyCode);
		return !!this._keymap[id] && !!!this._lastmap[id];
	}
	wasReleased(key) {
		const keyCode = toKeyCode(key);
		const id = this.toKeyIdentifier(keyCode);
		return !!!this._keymap[id] && !!this._lastmap[id];
	}
}
Keyboard.EVENT_KEYDOWN = 'keydown';
Keyboard.EVENT_KEYUP = 'keyup';

function isMousePointerLocked() {
	return !!(document.pointerLockElement || document.mozPointerLockElement || document.webkitPointerLockElement);
}
class MouseEvent {
	constructor(mouse, event) {
		let coords = {
			x: 0,
			y: 0
		};
		if (event) {
			if (event instanceof MouseEvent) {
				throw Error('Expected MouseEvent');
			}
			coords = mouse._getTargetCoords(event);
		} else {
			event = {};
		}
		if (coords) {
			this.x = coords.x;
			this.y = coords.y;
		} else if (isMousePointerLocked()) {
			this.x = 0;
			this.y = 0;
		} else {
			return;
		}
		this.wheelDelta = 0;
		if (event.type === 'wheel') {
			if (event.deltaY > 0) {
				this.wheelDelta = 1;
			} else if (event.deltaY < 0) {
				this.wheelDelta = -1;
			}
		}
		if (isMousePointerLocked()) {
			this.dx = event.movementX || event.webkitMovementX || event.mozMovementX || 0;
			this.dy = event.movementY || event.webkitMovementY || event.mozMovementY || 0;
		} else {
			this.dx = this.x - mouse._lastX;
			this.dy = this.y - mouse._lastY;
		}
		if (event.type === 'mousedown' || event.type === 'mouseup') {
			this.button = event.button;
		} else {
			this.button = MOUSEBUTTON_NONE;
		}
		this.buttons = mouse._buttons.slice(0);
		this.element = event.target;
		this.ctrlKey = event.ctrlKey || false;
		this.altKey = event.altKey || false;
		this.shiftKey = event.shiftKey || false;
		this.metaKey = event.metaKey || false;
		this.event = event;
	}
}

class Mouse extends EventHandler {
	constructor(element) {
		super();
		this._lastX = 0;
		this._lastY = 0;
		this._buttons = [false, false, false];
		this._lastbuttons = [false, false, false];
		this._upHandler = this._handleUp.bind(this);
		this._downHandler = this._handleDown.bind(this);
		this._moveHandler = this._handleMove.bind(this);
		this._wheelHandler = this._handleWheel.bind(this);
		this._contextMenuHandler = event => {
			event.preventDefault();
		};
		this._target = null;
		this._attached = false;
		this.attach(element);
	}
	static isPointerLocked() {
		return isMousePointerLocked();
	}
	attach(element) {
		this._target = element;
		if (this._attached) return;
		this._attached = true;
		const opts = platform.passiveEvents ? {
			passive: false
		} : false;
		window.addEventListener('mouseup', this._upHandler, opts);
		window.addEventListener('mousedown', this._downHandler, opts);
		window.addEventListener('mousemove', this._moveHandler, opts);
		window.addEventListener('wheel', this._wheelHandler, opts);
	}
	detach() {
		if (!this._attached) return;
		this._attached = false;
		this._target = null;
		const opts = platform.passiveEvents ? {
			passive: false
		} : false;
		window.removeEventListener('mouseup', this._upHandler, opts);
		window.removeEventListener('mousedown', this._downHandler, opts);
		window.removeEventListener('mousemove', this._moveHandler, opts);
		window.removeEventListener('wheel', this._wheelHandler, opts);
	}
	disableContextMenu() {
		if (!this._target) return;
		this._target.addEventListener('contextmenu', this._contextMenuHandler);
	}
	enableContextMenu() {
		if (!this._target) return;
		this._target.removeEventListener('contextmenu', this._contextMenuHandler);
	}
	enablePointerLock(success, error) {
		if (!document.body.requestPointerLock) {
			if (error) error();
			return;
		}
		const s = () => {
			success();
			document.removeEventListener('pointerlockchange', s);
		};
		const e = () => {
			error();
			document.removeEventListener('pointerlockerror', e);
		};
		if (success) {
			document.addEventListener('pointerlockchange', s, false);
		}
		if (error) {
			document.addEventListener('pointerlockerror', e, false);
		}
		document.body.requestPointerLock();
	}
	disablePointerLock(success) {
		if (!document.exitPointerLock) {
			return;
		}
		const s = () => {
			success();
			document.removeEventListener('pointerlockchange', s);
		};
		if (success) {
			document.addEventListener('pointerlockchange', s, false);
		}
		document.exitPointerLock();
	}
	update() {
		this._lastbuttons[0] = this._buttons[0];
		this._lastbuttons[1] = this._buttons[1];
		this._lastbuttons[2] = this._buttons[2];
	}
	isPressed(button) {
		return this._buttons[button];
	}
	wasPressed(button) {
		return this._buttons[button] && !this._lastbuttons[button];
	}
	wasReleased(button) {
		return !this._buttons[button] && this._lastbuttons[button];
	}
	_handleUp(event) {
		this._buttons[event.button] = false;
		const e = new MouseEvent(this, event);
		if (!e.event) return;
		this.fire(EVENT_MOUSEUP, e);
	}
	_handleDown(event) {
		this._buttons[event.button] = true;
		const e = new MouseEvent(this, event);
		if (!e.event) return;
		this.fire(EVENT_MOUSEDOWN, e);
	}
	_handleMove(event) {
		const e = new MouseEvent(this, event);
		if (!e.event) return;
		this.fire(EVENT_MOUSEMOVE, e);
		this._lastX = e.x;
		this._lastY = e.y;
	}
	_handleWheel(event) {
		const e = new MouseEvent(this, event);
		if (!e.event) return;
		this.fire(EVENT_MOUSEWHEEL, e);
	}
	_getTargetCoords(event) {
		const rect = this._target.getBoundingClientRect();
		const left = Math.floor(rect.left);
		const top = Math.floor(rect.top);
		if (event.clientX < left || event.clientX >= left + this._target.clientWidth || event.clientY < top || event.clientY >= top + this._target.clientHeight) {
			return null;
		}
		return {
			x: event.clientX - left,
			y: event.clientY - top
		};
	}
}
Mouse.EVENT_MOUSEMOVE = EVENT_MOUSEMOVE;
Mouse.EVENT_MOUSEDOWN = EVENT_MOUSEDOWN;
Mouse.EVENT_MOUSEUP = EVENT_MOUSEUP;
Mouse.EVENT_MOUSEWHEEL = EVENT_MOUSEWHEEL;

class Controller {
	constructor(element, options = {}) {
		this._keyboard = options.keyboard || null;
		this._mouse = options.mouse || null;
		this._gamepads = options.gamepads || null;
		this._element = null;
		this._actions = {};
		this._axes = {};
		this._axesValues = {};
		if (element) {
			this.attach(element);
		}
	}
	attach(element) {
		this._element = element;
		if (this._keyboard) {
			this._keyboard.attach(element);
		}
		if (this._mouse) {
			this._mouse.attach(element);
		}
	}
	detach() {
		if (this._keyboard) {
			this._keyboard.detach();
		}
		if (this._mouse) {
			this._mouse.detach();
		}
		this._element = null;
	}
	disableContextMenu() {
		if (!this._mouse) {
			this._enableMouse();
		}
		this._mouse.disableContextMenu();
	}
	enableContextMenu() {
		if (!this._mouse) {
			this._enableMouse();
		}
		this._mouse.enableContextMenu();
	}
	update(dt) {
		if (this._keyboard) {
			this._keyboard.update();
		}
		if (this._mouse) {
			this._mouse.update();
		}
		if (this._gamepads) {
			this._gamepads.update();
		}
		this._axesValues = {};
		for (const key in this._axes) {
			this._axesValues[key] = [];
		}
	}
	appendAction(action_name, action) {
		this._actions[action_name] = this._actions[action_name] || [];
		this._actions[action_name].push(action);
	}
	registerKeys(action, keys) {
		if (!this._keyboard) {
			this._enableKeyboard();
		}
		if (this._actions[action]) {
			throw new Error(`Action: ${action} already registered`);
		}
		if (keys === undefined) {
			throw new Error('Invalid button');
		}
		if (!keys.length) {
			keys = [keys];
		}
		this.appendAction(action, {
			type: ACTION_KEYBOARD,
			keys
		});
	}
	registerMouse(action, button) {
		if (!this._mouse) {
			this._enableMouse();
		}
		if (button === undefined) {
			throw new Error('Invalid button');
		}
		this.appendAction(action, {
			type: ACTION_MOUSE,
			button
		});
	}
	registerPadButton(action, pad, button) {
		if (button === undefined) {
			throw new Error('Invalid button');
		}
		this.appendAction(action, {
			type: ACTION_GAMEPAD,
			button,
			pad
		});
	}
	registerAxis(options) {
		const name = options.name;
		if (!this._axes[name]) {
			this._axes[name] = [];
		}
		const i = this._axes[name].push(name);
		options = options || {};
		options.pad = options.pad || PAD_1;
		const bind = function bind(controller, source, value, key) {
			switch (source) {
				case 'mousex':
					controller._mouse.on(EVENT_MOUSEMOVE, function (e) {
						controller._axesValues[name][i] = e.dx / 10;
					});
					break;
				case 'mousey':
					controller._mouse.on(EVENT_MOUSEMOVE, function (e) {
						controller._axesValues[name][i] = e.dy / 10;
					});
					break;
				case 'key':
					controller._axes[name].push(function () {
						return controller._keyboard.isPressed(key) ? value : 0;
					});
					break;
				case 'padrx':
					controller._axes[name].push(function () {
						return controller._gamepads.getAxis(options.pad, PAD_R_STICK_X);
					});
					break;
				case 'padry':
					controller._axes[name].push(function () {
						return controller._gamepads.getAxis(options.pad, PAD_R_STICK_Y);
					});
					break;
				case 'padlx':
					controller._axes[name].push(function () {
						return controller._gamepads.getAxis(options.pad, PAD_L_STICK_X);
					});
					break;
				case 'padly':
					controller._axes[name].push(function () {
						return controller._gamepads.getAxis(options.pad, PAD_L_STICK_Y);
					});
					break;
				default:
					throw new Error('Unknown axis');
			}
		};
		bind(this, options.positive, 1, options.positiveKey);
		if (options.negativeKey || options.negative !== options.positive) {
			bind(this, options.negative, -1, options.negativeKey);
		}
	}
	isPressed(actionName) {
		if (!this._actions[actionName]) {
			return false;
		}
		const length = this._actions[actionName].length;
		for (let index = 0; index < length; ++index) {
			const action = this._actions[actionName][index];
			switch (action.type) {
				case ACTION_KEYBOARD:
					if (this._keyboard) {
						const len = action.keys.length;
						for (let i = 0; i < len; i++) {
							if (this._keyboard.isPressed(action.keys[i])) {
								return true;
							}
						}
					}
					break;
				case ACTION_MOUSE:
					if (this._mouse && this._mouse.isPressed(action.button)) {
						return true;
					}
					break;
				case ACTION_GAMEPAD:
					if (this._gamepads && this._gamepads.isPressed(action.pad, action.button)) {
						return true;
					}
					break;
			}
		}
		return false;
	}
	wasPressed(actionName) {
		if (!this._actions[actionName]) {
			return false;
		}
		const length = this._actions[actionName].length;
		for (let index = 0; index < length; ++index) {
			const action = this._actions[actionName][index];
			switch (action.type) {
				case ACTION_KEYBOARD:
					if (this._keyboard) {
						const len = action.keys.length;
						for (let i = 0; i < len; i++) {
							if (this._keyboard.wasPressed(action.keys[i])) {
								return true;
							}
						}
					}
					break;
				case ACTION_MOUSE:
					if (this._mouse && this._mouse.wasPressed(action.button)) {
						return true;
					}
					break;
				case ACTION_GAMEPAD:
					if (this._gamepads && this._gamepads.wasPressed(action.pad, action.button)) {
						return true;
					}
					break;
			}
		}
		return false;
	}
	getAxis(name) {
		let value = 0;
		if (this._axes[name]) {
			const len = this._axes[name].length;
			for (let i = 0; i < len; i++) {
				if (type(this._axes[name][i]) === 'function') {
					const v = this._axes[name][i]();
					if (Math.abs(v) > Math.abs(value)) {
						value = v;
					}
				} else if (this._axesValues[name]) {
					if (Math.abs(this._axesValues[name][i]) > Math.abs(value)) {
						value = this._axesValues[name][i];
					}
				}
			}
		}
		return value;
	}
	_enableMouse() {
		this._mouse = new Mouse();
		if (!this._element) {
			throw new Error('Controller must be attached to an Element');
		}
		this._mouse.attach(this._element);
	}
	_enableKeyboard() {
		this._keyboard = new Keyboard();
		if (!this._element) {
			throw new Error('Controller must be attached to an Element');
		}
		this._keyboard.attach(this._element);
	}
}

const dummyArray = Object.freeze([]);
let _getGamepads = function getGamepads() {
	return dummyArray;
};
if (typeof navigator !== 'undefined') {
	_getGamepads = (navigator.getGamepads || navigator.webkitGetGamepads || _getGamepads).bind(navigator);
}
const MAPS_INDEXES = {
	buttons: {
		PAD_FACE_1,
		PAD_FACE_2,
		PAD_FACE_3,
		PAD_FACE_4,
		PAD_L_SHOULDER_1,
		PAD_R_SHOULDER_1,
		PAD_L_SHOULDER_2,
		PAD_R_SHOULDER_2,
		PAD_SELECT,
		PAD_START,
		PAD_L_STICK_BUTTON,
		PAD_R_STICK_BUTTON,
		PAD_UP,
		PAD_DOWN,
		PAD_LEFT,
		PAD_RIGHT,
		PAD_VENDOR,
		XRPAD_TRIGGER,
		XRPAD_SQUEEZE,
		XRPAD_TOUCHPAD_BUTTON,
		XRPAD_STICK_BUTTON,
		XRPAD_A,
		XRPAD_B
	},
	axes: {
		PAD_L_STICK_X,
		PAD_L_STICK_Y,
		PAD_R_STICK_X,
		PAD_R_STICK_Y,
		XRPAD_TOUCHPAD_X,
		XRPAD_TOUCHPAD_Y,
		XRPAD_STICK_X,
		XRPAD_STICK_Y
	}
};
const MAPS = {
	DEFAULT: {
		buttons: ['PAD_FACE_1', 'PAD_FACE_2', 'PAD_FACE_3', 'PAD_FACE_4', 'PAD_L_SHOULDER_1', 'PAD_R_SHOULDER_1', 'PAD_L_SHOULDER_2', 'PAD_R_SHOULDER_2', 'PAD_SELECT', 'PAD_START', 'PAD_L_STICK_BUTTON', 'PAD_R_STICK_BUTTON', 'PAD_UP', 'PAD_DOWN', 'PAD_LEFT', 'PAD_RIGHT', 'PAD_VENDOR'],
		axes: ['PAD_L_STICK_X', 'PAD_L_STICK_Y', 'PAD_R_STICK_X', 'PAD_R_STICK_Y']
	},
	DEFAULT_DUAL: {
		buttons: ['PAD_FACE_1', 'PAD_FACE_2', 'PAD_FACE_3', 'PAD_FACE_4', 'PAD_L_SHOULDER_1', 'PAD_R_SHOULDER_1', 'PAD_L_SHOULDER_2', 'PAD_R_SHOULDER_2', 'PAD_SELECT', 'PAD_START', 'PAD_L_STICK_BUTTON', 'PAD_R_STICK_BUTTON', 'PAD_VENDOR'],
		axes: ['PAD_L_STICK_X', 'PAD_L_STICK_Y', 'PAD_R_STICK_X', 'PAD_R_STICK_Y'],
		synthesizedButtons: {
			PAD_UP: {
				axis: 0,
				min: 0,
				max: 1
			},
			PAD_DOWN: {
				axis: 0,
				min: -1,
				max: 0
			},
			PAD_LEFT: {
				axis: 0,
				min: -1,
				max: 0
			},
			PAD_RIGHT: {
				axis: 0,
				min: 0,
				max: 1
			}
		}
	},
	PS3: {
		buttons: ['PAD_FACE_1', 'PAD_FACE_2', 'PAD_FACE_4', 'PAD_FACE_3', 'PAD_L_SHOULDER_1', 'PAD_R_SHOULDER_1', 'PAD_L_SHOULDER_2', 'PAD_R_SHOULDER_2', 'PAD_SELECT', 'PAD_START', 'PAD_L_STICK_BUTTON', 'PAD_R_STICK_BUTTON', 'PAD_UP', 'PAD_DOWN', 'PAD_LEFT', 'PAD_RIGHT', 'PAD_VENDOR'],
		axes: ['PAD_L_STICK_X', 'PAD_L_STICK_Y', 'PAD_R_STICK_X', 'PAD_R_STICK_Y'],
		mapping: 'standard'
	},
	DEFAULT_XR: {
		buttons: ['XRPAD_TRIGGER', 'XRPAD_SQUEEZE', 'XRPAD_TOUCHPAD_BUTTON', 'XRPAD_STICK_BUTTON', 'XRPAD_A', 'XRPAD_B'],
		axes: ['XRPAD_TOUCHPAD_X', 'XRPAD_TOUCHPAD_Y', 'XRPAD_STICK_X', 'XRPAD_STICK_Y'],
		mapping: 'xr-standard'
	}
};
const PRODUCT_CODES = {
	'Product: 0268': 'PS3'
};
const custom_maps = {};
function getMap(pad) {
	const custom = custom_maps[pad.id];
	if (custom) {
		return custom;
	}
	for (const code in PRODUCT_CODES) {
		if (pad.id.indexOf(code) !== -1) {
			const product = PRODUCT_CODES[code];
			if (!pad.mapping) {
				const raw = MAPS['RAW_' + product];
				if (raw) {
					return raw;
				}
			}
			return MAPS[product];
		}
	}
	if (pad.mapping === 'xr-standard') {
		return MAPS.DEFAULT_XR;
	}
	const defaultmap = MAPS.DEFAULT;
	const map = pad.buttons.length < defaultmap.buttons.length ? MAPS.DEFAULT_DUAL : defaultmap;
	map.mapping = pad.mapping;
	return map;
}
let deadZone = 0.25;
function sleep(ms) {
	return new Promise(resolve => {
		setTimeout(resolve, ms);
	});
}
class GamePadButton {
	constructor(current, previous) {
		this.value = 0;
		this.pressed = false;
		this.touched = false;
		this.wasPressed = false;
		this.wasReleased = false;
		this.wasTouched = false;
		if (typeof current === 'number') {
			this.value = current;
			this.pressed = current === 1;
			this.touched = current > 0;
		} else {
			var _current$touched;
			this.value = current.value;
			this.pressed = current.pressed;
			this.touched = (_current$touched = current.touched) != null ? _current$touched : current.value > 0;
		}
		if (previous) {
			if (typeof previous === 'number') {
				this.wasPressed = previous !== 1 && this.pressed;
				this.wasReleased = previous === 1 && !this.pressed;
				this.wasTouched = previous === 0 && this.touched;
			} else {
				var _previous$touched;
				this.wasPressed = !previous.pressed && this.pressed;
				this.wasReleased = previous.pressed && !this.pressed;
				this.wasTouched = !((_previous$touched = previous.touched) != null ? _previous$touched : previous.value > 0) && this.touched;
			}
		}
	}
	update(button) {
		var _button$touched;
		const {
			value,
			pressed
		} = button;
		const touched = (_button$touched = button.touched) != null ? _button$touched : value > 0;
		this.wasPressed = !this.pressed && pressed;
		this.wasReleased = this.pressed && !pressed;
		this.wasTouched = !this.touched && touched;
		this.value = value;
		this.pressed = pressed;
		this.touched = touched;
	}
}
const dummyButton = Object.freeze(new GamePadButton(0));
class GamePad {
	constructor(gamepad, map) {
		this._compiledMapping = {
			buttons: [],
			axes: []
		};
		this.id = gamepad.id;
		this.index = gamepad.index;
		this._buttons = gamepad.buttons.map(b => new GamePadButton(b));
		this._axes = [...gamepad.axes];
		this._previousAxes = [...gamepad.axes];
		this.mapping = map.mapping;
		this.map = map;
		this.hand = gamepad.hand || 'none';
		this.pad = gamepad;
		this._compileMapping();
	}
	get connected() {
		return this.pad.connected;
	}
	_compileMapping() {
		const {
			axes,
			buttons
		} = this._compiledMapping;
		const axesIndexes = MAPS_INDEXES.axes;
		const buttonsIndexes = MAPS_INDEXES.buttons;
		axes.length = 0;
		buttons.length = 0;
		const axesMap = this.map.axes;
		if (axesMap) {
			this.map.axes.forEach((axis, i) => {
				axes[axesIndexes[axis]] = () => this.pad.axes[i] || 0;
			});
		}
		for (let i = 0, l = axes.length; i < l; i++) {
			if (!axes[i]) {
				axes[i] = () => 0;
			}
		}
		const buttonsMap = this.map.buttons;
		if (buttonsMap) {
			buttonsMap.forEach((button, i) => {
				buttons[buttonsIndexes[button]] = () => this._buttons[i] || dummyButton;
			});
		}
		const synthesizedButtonsMap = this.map.synthesizedButtons;
		if (synthesizedButtonsMap) {
			Object.entries(synthesizedButtonsMap).forEach(button => {
				const {
					axis,
					max,
					min
				} = button[1];
				buttons[buttonsIndexes[button[0]]] = () => {
					var _this$_axes$axis, _this$_previousAxes$a;
					return new GamePadButton(Math.abs(math.clamp((_this$_axes$axis = this._axes[axis]) != null ? _this$_axes$axis : 0, min, max)), Math.abs(math.clamp((_this$_previousAxes$a = this._previousAxes[axis]) != null ? _this$_previousAxes$a : 0, min, max)));
				};
			});
		}
		for (let i = 0, l = buttons.length; i < l; i++) {
			if (!buttons[i]) {
				buttons[i] = () => dummyButton;
			}
		}
	}
	update(gamepad) {
		this.pad = gamepad;
		const previousAxes = this._previousAxes;
		const axes = this._axes;
		previousAxes.length = 0;
		previousAxes.push(...axes);
		axes.length = 0;
		axes.push(...gamepad.axes);
		const buttons = this._buttons;
		for (let i = 0, l = buttons.length; i < l; i++) {
			buttons[i].update(gamepad.buttons[i]);
		}
		return this;
	}
	updateMap(map) {
		map.mapping = 'custom';
		custom_maps[this.id] = map;
		this.map = map;
		this.mapping = 'custom';
		this._compileMapping();
	}
	resetMap() {
		if (custom_maps[this.id]) {
			delete custom_maps[this.id];
			const map = getMap(this.pad);
			this.map = map;
			this.mapping = map.mapping;
			this._compileMapping();
		}
	}
	get axes() {
		return this._compiledMapping.axes.map(a => a());
	}
	get buttons() {
		return this._compiledMapping.buttons.map(b => b());
	}
	async pulse(intensity, duration, options) {
		const actuators = this.pad.vibrationActuator ? [this.pad.vibrationActuator] : this.pad.hapticActuators || dummyArray;
		if (actuators.length) {
			var _options$startDelay, _options$strongMagnit, _options$weakMagnitud;
			const startDelay = (_options$startDelay = options == null ? void 0 : options.startDelay) != null ? _options$startDelay : 0;
			const strongMagnitude = (_options$strongMagnit = options == null ? void 0 : options.strongMagnitude) != null ? _options$strongMagnit : intensity;
			const weakMagnitude = (_options$weakMagnitud = options == null ? void 0 : options.weakMagnitude) != null ? _options$weakMagnitud : intensity;
			const results = await Promise.all(actuators.map(async function (actuator) {
				if (!actuator) {
					return true;
				}
				if (actuator.playEffect) {
					return actuator.playEffect(actuator.type, {
						duration,
						startDelay,
						strongMagnitude,
						weakMagnitude
					});
				} else if (actuator.pulse) {
					await sleep(startDelay);
					return actuator.pulse(intensity, duration);
				}
				return false;
			}));
			return results.some(r => r === true || r === 'complete');
		}
		return false;
	}
	getButton(index) {
		const button = this._compiledMapping.buttons[index];
		return button ? button() : dummyButton;
	}
	isPressed(button) {
		return this.getButton(button).pressed;
	}
	wasPressed(button) {
		return this.getButton(button).wasPressed;
	}
	wasReleased(button) {
		return this.getButton(button).wasReleased;
	}
	isTouched(button) {
		return this.getButton(button).touched;
	}
	wasTouched(button) {
		return this.getButton(button).wasTouched;
	}
	getValue(button) {
		return this.getButton(button).value;
	}
	getAxis(axis) {
		const a = this.axes[axis];
		return a && Math.abs(a) > deadZone ? a : 0;
	}
}
class GamePads extends EventHandler {
	constructor() {
		super();
		this.gamepadsSupported = platform.gamepads;
		this.current = [];
		this._previous = [];
		this._ongamepadconnectedHandler = this._ongamepadconnected.bind(this);
		this._ongamepaddisconnectedHandler = this._ongamepaddisconnected.bind(this);
		window.addEventListener('gamepadconnected', this._ongamepadconnectedHandler, false);
		window.addEventListener('gamepaddisconnected', this._ongamepaddisconnectedHandler, false);
		this.poll();
	}
	set deadZone(value) {
		deadZone = value;
	}
	get deadZone() {
		return deadZone;
	}
	get previous() {
		const current = this.current;
		for (let i = 0, l = current.length; i < l; i++) {
			const buttons = current[i]._buttons;
			if (!this._previous[i]) {
				this._previous[i] = [];
			}
			for (let j = 0, m = buttons.length; j < m; j++) {
				const button = buttons[i];
				this.previous[i][j] = button ? !button.wasPressed && button.pressed || button.wasReleased : false;
			}
		}
		this._previous.length = this.current.length;
		return this._previous;
	}
	_ongamepadconnected(event) {
		const pad = new GamePad(event.gamepad, this.getMap(event.gamepad));
		const current = this.current;
		let padIndex = current.findIndex(gp => gp.index === pad.index);
		while (padIndex !== -1) {
			current.splice(padIndex, 1);
			padIndex = current.findIndex(gp => gp.index === pad.index);
		}
		current.push(pad);
		this.fire(EVENT_GAMEPADCONNECTED, pad);
	}
	_ongamepaddisconnected(event) {
		const current = this.current;
		const padIndex = current.findIndex(gp => gp.index === event.gamepad.index);
		if (padIndex !== -1) {
			this.fire(EVENT_GAMEPADDISCONNECTED, current[padIndex]);
			current.splice(padIndex, 1);
		}
	}
	update() {
		this.poll();
	}
	poll(pads = []) {
		if (pads.length > 0) {
			pads.length = 0;
		}
		const padDevices = _getGamepads();
		for (let i = 0, len = padDevices.length; i < len; i++) {
			if (padDevices[i]) {
				const pad = this.findByIndex(padDevices[i].index);
				if (pad) {
					pads.push(pad.update(padDevices[i]));
				} else {
					const nPad = new GamePad(padDevices[i], this.getMap(padDevices[i]));
					this.current.push(nPad);
					pads.push(nPad);
				}
			}
		}
		return pads;
	}
	destroy() {
		window.removeEventListener('gamepadconnected', this._ongamepadconnectedHandler, false);
		window.removeEventListener('gamepaddisconnected', this._ongamepaddisconnectedHandler, false);
	}
	getMap(pad) {
		return getMap(pad);
	}
	isPressed(orderIndex, button) {
		var _this$current$orderIn;
		return ((_this$current$orderIn = this.current[orderIndex]) == null ? void 0 : _this$current$orderIn.isPressed(button)) || false;
	}
	wasPressed(orderIndex, button) {
		var _this$current$orderIn2;
		return ((_this$current$orderIn2 = this.current[orderIndex]) == null ? void 0 : _this$current$orderIn2.wasPressed(button)) || false;
	}
	wasReleased(orderIndex, button) {
		var _this$current$orderIn3;
		return ((_this$current$orderIn3 = this.current[orderIndex]) == null ? void 0 : _this$current$orderIn3.wasReleased(button)) || false;
	}
	getAxis(orderIndex, axis) {
		var _this$current$orderIn4;
		return ((_this$current$orderIn4 = this.current[orderIndex]) == null ? void 0 : _this$current$orderIn4.getAxis(axis)) || 0;
	}
	pulse(orderIndex, intensity, duration, options) {
		const pad = this.current[orderIndex];
		return pad ? pad.pulse(intensity, duration, options) : Promise.resolve(false);
	}
	pulseAll(intensity, duration, options) {
		return Promise.all(this.current.map(pad => pad.pulse(intensity, duration, options)));
	}
	findById(id) {
		return this.current.find(gp => gp && gp.id === id) || null;
	}
	findByIndex(index) {
		return this.current.find(gp => gp && gp.index === index) || null;
	}
}
GamePads.EVENT_GAMEPADCONNECTED = 'gamepadconnected';
GamePads.EVENT_GAMEPADDISCONNECTED = 'gamepaddisconnected';

function getTouchTargetCoords(touch) {
	let totalOffsetX = 0;
	let totalOffsetY = 0;
	let target = touch.target;
	while (!(target instanceof HTMLElement)) {
		target = target.parentNode;
	}
	let currentElement = target;
	do {
		totalOffsetX += currentElement.offsetLeft - currentElement.scrollLeft;
		totalOffsetY += currentElement.offsetTop - currentElement.scrollTop;
		currentElement = currentElement.offsetParent;
	} while (currentElement);
	return {
		x: touch.pageX - totalOffsetX,
		y: touch.pageY - totalOffsetY
	};
}
class Touch {
	constructor(touch) {
		const coords = getTouchTargetCoords(touch);
		this.id = touch.identifier;
		this.x = coords.x;
		this.y = coords.y;
		this.target = touch.target;
		this.touch = touch;
	}
}
class TouchEvent {
	constructor(device, event) {
		this.element = event.target;
		this.event = event;
		this.touches = [];
		this.changedTouches = [];
		if (event) {
			for (let i = 0, l = event.touches.length; i < l; i++) {
				this.touches.push(new Touch(event.touches[i]));
			}
			for (let i = 0, l = event.changedTouches.length; i < l; i++) {
				this.changedTouches.push(new Touch(event.changedTouches[i]));
			}
		}
	}
	getTouchById(id, list) {
		for (let i = 0, l = list.length; i < l; i++) {
			if (list[i].id === id) {
				return list[i];
			}
		}
		return null;
	}
}

class TouchDevice extends EventHandler {
	constructor(element) {
		super();
		this._element = null;
		this._startHandler = this._handleTouchStart.bind(this);
		this._endHandler = this._handleTouchEnd.bind(this);
		this._moveHandler = this._handleTouchMove.bind(this);
		this._cancelHandler = this._handleTouchCancel.bind(this);
		this.attach(element);
	}
	attach(element) {
		if (this._element) {
			this.detach();
		}
		this._element = element;
		this._element.addEventListener('touchstart', this._startHandler, false);
		this._element.addEventListener('touchend', this._endHandler, false);
		this._element.addEventListener('touchmove', this._moveHandler, false);
		this._element.addEventListener('touchcancel', this._cancelHandler, false);
	}
	detach() {
		if (this._element) {
			this._element.removeEventListener('touchstart', this._startHandler, false);
			this._element.removeEventListener('touchend', this._endHandler, false);
			this._element.removeEventListener('touchmove', this._moveHandler, false);
			this._element.removeEventListener('touchcancel', this._cancelHandler, false);
		}
		this._element = null;
	}
	_handleTouchStart(e) {
		this.fire('touchstart', new TouchEvent(this, e));
	}
	_handleTouchEnd(e) {
		this.fire('touchend', new TouchEvent(this, e));
	}
	_handleTouchMove(e) {
		e.preventDefault();
		this.fire('touchmove', new TouchEvent(this, e));
	}
	_handleTouchCancel(e) {
		this.fire('touchcancel', new TouchEvent(this, e));
	}
}

class Http {
	get(url, options, callback) {
		if (typeof options === 'function') {
			callback = options;
			options = {};
		}
		return this.request('GET', url, options, callback);
	}
	post(url, data, options, callback) {
		if (typeof options === 'function') {
			callback = options;
			options = {};
		}
		options.postdata = data;
		return this.request('POST', url, options, callback);
	}
	put(url, data, options, callback) {
		if (typeof options === 'function') {
			callback = options;
			options = {};
		}
		options.postdata = data;
		return this.request('PUT', url, options, callback);
	}
	del(url, options, callback) {
		if (typeof options === 'function') {
			callback = options;
			options = {};
		}
		return this.request('DELETE', url, options, callback);
	}
	request(method, url, options, callback) {
		let uri, query, postdata;
		let errored = false;
		if (typeof options === 'function') {
			callback = options;
			options = {};
		}
		if (options.retry) {
			options = Object.assign({
				retries: 0,
				maxRetries: 5
			}, options);
		}
		options.callback = callback;
		if (options.async == null) {
			options.async = true;
		}
		if (options.headers == null) {
			options.headers = {};
		}
		if (options.postdata != null) {
			if (options.postdata instanceof Document) {
				postdata = options.postdata;
			} else if (options.postdata instanceof FormData) {
				postdata = options.postdata;
			} else if (options.postdata instanceof Object) {
				let contentType = options.headers['Content-Type'];
				if (contentType === undefined) {
					options.headers['Content-Type'] = Http.ContentType.FORM_URLENCODED;
					contentType = options.headers['Content-Type'];
				}
				switch (contentType) {
					case Http.ContentType.FORM_URLENCODED:
						{
							postdata = '';
							let bFirstItem = true;
							for (const key in options.postdata) {
								if (options.postdata.hasOwnProperty(key)) {
									if (bFirstItem) {
										bFirstItem = false;
									} else {
										postdata += '&';
									}
									const encodedKey = encodeURIComponent(key);
									const encodedValue = encodeURIComponent(options.postdata[key]);
									postdata += `${encodedKey}=${encodedValue}`;
								}
							}
							break;
						}
					default:
					case Http.ContentType.JSON:
						if (contentType == null) {
							options.headers['Content-Type'] = Http.ContentType.JSON;
						}
						postdata = JSON.stringify(options.postdata);
						break;
				}
			} else {
				postdata = options.postdata;
			}
		}
		if (options.cache === false) {
			const timestamp = now();
			uri = new URI(url);
			if (!uri.query) {
				uri.query = 'ts=' + timestamp;
			} else {
				uri.query = uri.query + '&ts=' + timestamp;
			}
			url = uri.toString();
		}
		if (options.query) {
			uri = new URI(url);
			query = extend(uri.getQuery(), options.query);
			uri.setQuery(query);
			url = uri.toString();
		}
		const xhr = new XMLHttpRequest();
		xhr.open(method, url, options.async);
		xhr.withCredentials = options.withCredentials !== undefined ? options.withCredentials : false;
		xhr.responseType = options.responseType || this._guessResponseType(url);
		for (const header in options.headers) {
			if (options.headers.hasOwnProperty(header)) {
				xhr.setRequestHeader(header, options.headers[header]);
			}
		}
		xhr.onreadystatechange = () => {
			this._onReadyStateChange(method, url, options, xhr);
		};
		xhr.onerror = () => {
			this._onError(method, url, options, xhr);
			errored = true;
		};
		try {
			xhr.send(postdata);
		} catch (e) {
			if (!errored) {
				options.error(xhr.status, xhr, e);
			}
		}
		return xhr;
	}
	_guessResponseType(url) {
		const uri = new URI(url);
		const ext = path.getExtension(uri.path).toLowerCase();
		if (Http.binaryExtensions.indexOf(ext) >= 0) {
			return Http.ResponseType.ARRAY_BUFFER;
		} else if (ext === '.json') {
			return Http.ResponseType.JSON;
		} else if (ext === '.xml') {
			return Http.ResponseType.DOCUMENT;
		}
		return Http.ResponseType.TEXT;
	}
	_isBinaryContentType(contentType) {
		const binTypes = [Http.ContentType.BASIS, Http.ContentType.BIN, Http.ContentType.DDS, Http.ContentType.GLB, Http.ContentType.MP3, Http.ContentType.MP4, Http.ContentType.OGG, Http.ContentType.OPUS, Http.ContentType.WAV];
		if (binTypes.indexOf(contentType) >= 0) {
			return true;
		}
		return false;
	}
	_isBinaryResponseType(responseType) {
		return responseType === Http.ResponseType.ARRAY_BUFFER || responseType === Http.ResponseType.BLOB || responseType === Http.ResponseType.JSON;
	}
	_onReadyStateChange(method, url, options, xhr) {
		if (xhr.readyState === 4) {
			switch (xhr.status) {
				case 0:
					{
						if (xhr.responseURL && xhr.responseURL.startsWith('file:///')) {
							this._onSuccess(method, url, options, xhr);
						} else {
							this._onError(method, url, options, xhr);
						}
						break;
					}
				case 200:
				case 201:
				case 206:
				case 304:
					{
						this._onSuccess(method, url, options, xhr);
						break;
					}
				default:
					{
						this._onError(method, url, options, xhr);
						break;
					}
			}
		}
	}
	_onSuccess(method, url, options, xhr) {
		let response;
		let contentType;
		const header = xhr.getResponseHeader('Content-Type');
		if (header) {
			const parts = header.split(';');
			contentType = parts[0].trim();
		}
		try {
			if (this._isBinaryContentType(contentType) || this._isBinaryResponseType(xhr.responseType)) {
				response = xhr.response;
			} else if (contentType === Http.ContentType.JSON || url.split('?')[0].endsWith('.json')) {
				response = JSON.parse(xhr.responseText);
			} else if (xhr.responseType === Http.ResponseType.DOCUMENT || contentType === Http.ContentType.XML) {
				response = xhr.responseXML;
			} else {
				response = xhr.responseText;
			}
			options.callback(null, response);
		} catch (err) {
			options.callback(err);
		}
	}
	_onError(method, url, options, xhr) {
		if (options.retrying) {
			return;
		}
		if (options.retry && options.retries < options.maxRetries) {
			options.retries++;
			options.retrying = true;
			const retryDelay = math.clamp(Math.pow(2, options.retries) * Http.retryDelay, 0, options.maxRetryDelay || 5000);
			console.log(`${method}: ${url} - Error ${xhr.status}. Retrying in ${retryDelay} ms`);
			setTimeout(() => {
				options.retrying = false;
				this.request(method, url, options, options.callback);
			}, retryDelay);
		} else {
			options.callback(xhr.status === 0 ? 'Network error' : xhr.status, null);
		}
	}
}
Http.ContentType = {
	AAC: 'audio/aac',
	BASIS: 'image/basis',
	BIN: 'application/octet-stream',
	DDS: 'image/dds',
	FORM_URLENCODED: 'application/x-www-form-urlencoded',
	GIF: 'image/gif',
	GLB: 'model/gltf-binary',
	JPEG: 'image/jpeg',
	JSON: 'application/json',
	MP3: 'audio/mpeg',
	MP4: 'audio/mp4',
	OGG: 'audio/ogg',
	OPUS: 'audio/ogg; codecs="opus"',
	PNG: 'image/png',
	TEXT: 'text/plain',
	WAV: 'audio/x-wav',
	XML: 'application/xml'
};
Http.ResponseType = {
	TEXT: 'text',
	ARRAY_BUFFER: 'arraybuffer',
	BLOB: 'blob',
	DOCUMENT: 'document',
	JSON: 'json'
};
Http.binaryExtensions = ['.model', '.wav', '.ogg', '.mp3', '.mp4', '.m4a', '.aac', '.dds', '.basis', '.glb', '.opus'];
Http.retryDelay = 100;
const http = new Http();

function hasAudioContext() {
	return !!(typeof AudioContext !== 'undefined' || typeof webkitAudioContext !== 'undefined');
}

class Channel {
	constructor(manager, sound, options = {}) {
		var _options$volume, _options$loop, _options$pitch;
		this.volume = (_options$volume = options.volume) != null ? _options$volume : 1;
		this.loop = (_options$loop = options.loop) != null ? _options$loop : false;
		this.pitch = (_options$pitch = options.pitch) != null ? _options$pitch : 1;
		this.sound = sound;
		this.paused = false;
		this.suspended = false;
		this.manager = manager;
		this.source = null;
		if (hasAudioContext()) {
			this.startTime = 0;
			this.startOffset = 0;
			const context = manager.context;
			this.gain = context.createGain();
		} else if (sound.audio) {
			this.source = sound.audio.cloneNode(false);
			this.source.pause();
		}
	}
	getVolume() {
		return this.volume;
	}
	getLoop() {
		return this.loop;
	}
	setLoop(loop) {
		this.loop = loop;
		if (this.source) {
			this.source.loop = loop;
		}
	}
	getPitch() {
		return this.pitch;
	}
	onManagerVolumeChange() {
		this.setVolume(this.getVolume());
	}
	onManagerSuspend() {
		if (this.isPlaying() && !this.suspended) {
			this.suspended = true;
			this.pause();
		}
	}
	onManagerResume() {
		if (this.suspended) {
			this.suspended = false;
			this.unpause();
		}
	}
	play() {
		if (this.source) {
			throw new Error('Call stop() before calling play()');
		}
		this._createSource();
		if (!this.source) {
			return;
		}
		this.startTime = this.manager.context.currentTime;
		this.source.start(0, this.startOffset % this.source.buffer.duration);
		this.setVolume(this.volume);
		this.setLoop(this.loop);
		this.setPitch(this.pitch);
		this.manager.on('volumechange', this.onManagerVolumeChange, this);
		this.manager.on('suspend', this.onManagerSuspend, this);
		this.manager.on('resume', this.onManagerResume, this);
		if (this.manager.suspended) this.onManagerSuspend();
	}
	pause() {
		if (this.source) {
			this.paused = true;
			this.startOffset += this.manager.context.currentTime - this.startTime;
			this.source.stop(0);
			this.source = null;
		}
	}
	unpause() {
		if (this.source || !this.paused) {
			console.warn('Call pause() before unpausing.');
			return;
		}
		this._createSource();
		if (!this.source) {
			return;
		}
		this.startTime = this.manager.context.currentTime;
		this.source.start(0, this.startOffset % this.source.buffer.duration);
		this.setVolume(this.volume);
		this.setLoop(this.loop);
		this.setPitch(this.pitch);
		this.paused = false;
	}
	stop() {
		if (this.source) {
			this.source.stop(0);
			this.source = null;
		}
		this.manager.off('volumechange', this.onManagerVolumeChange, this);
		this.manager.off('suspend', this.onManagerSuspend, this);
		this.manager.off('resume', this.onManagerResume, this);
	}
	setVolume(volume) {
		volume = math.clamp(volume, 0, 1);
		this.volume = volume;
		if (this.gain) {
			this.gain.gain.value = volume * this.manager.volume;
		}
	}
	setPitch(pitch) {
		this.pitch = pitch;
		if (this.source) {
			this.source.playbackRate.value = pitch;
		}
	}
	isPlaying() {
		return !this.paused && this.source.playbackState === this.source.PLAYING_STATE;
	}
	getDuration() {
		return this.source ? this.source.buffer.duration : 0;
	}
	_createSource() {
		const context = this.manager.context;
		if (this.sound.buffer) {
			this.source = context.createBufferSource();
			this.source.buffer = this.sound.buffer;
			this.source.connect(this.gain);
			this.gain.connect(context.destination);
			if (!this.loop) {
				this.source.onended = this.pause.bind(this);
			}
		}
	}
}
if (!hasAudioContext()) {
	Object.assign(Channel.prototype, {
		play: function () {
			if (this.source) {
				this.paused = false;
				this.setVolume(this.volume);
				this.setLoop(this.loop);
				this.setPitch(this.pitch);
				this.source.play();
			}
			this.manager.on('volumechange', this.onManagerVolumeChange, this);
			this.manager.on('suspend', this.onManagerSuspend, this);
			this.manager.on('resume', this.onManagerResume, this);
			if (this.manager.suspended) this.onManagerSuspend();
		},
		pause: function () {
			if (this.source) {
				this.paused = true;
				this.source.pause();
			}
		},
		unpause: function () {
			if (this.source) {
				this.paused = false;
				this.source.play();
			}
		},
		stop: function () {
			if (this.source) {
				this.source.pause();
			}
			this.manager.off('volumechange', this.onManagerVolumeChange, this);
			this.manager.off('suspend', this.onManagerSuspend, this);
			this.manager.off('resume', this.onManagerResume, this);
		},
		setVolume: function (volume) {
			volume = math.clamp(volume, 0, 1);
			this.volume = volume;
			if (this.source) {
				this.source.volume = volume * this.manager.volume;
			}
		},
		setPitch: function (pitch) {
			this.pitch = pitch;
			if (this.source) {
				this.source.playbackRate = pitch;
			}
		},
		getDuration: function () {
			return this.source && !isNaN(this.source.duration) ? this.source.duration : 0;
		},
		isPlaying: function () {
			return !this.source.paused;
		}
	});
}

const MAX_DISTANCE$1 = 10000;
class Channel3d extends Channel {
	constructor(manager, sound, options) {
		super(manager, sound, options);
		this.position = new Vec3();
		this.velocity = new Vec3();
		if (hasAudioContext()) {
			this.panner = manager.context.createPanner();
		} else {
			this.maxDistance = MAX_DISTANCE$1;
			this.minDistance = 1;
			this.rollOffFactor = 1;
			this.distanceModel = DISTANCE_INVERSE;
		}
	}
	getPosition() {
		return this.position;
	}
	setPosition(position) {
		this.position.copy(position);
		const panner = this.panner;
		if ('positionX' in panner) {
			panner.positionX.value = position.x;
			panner.positionY.value = position.y;
			panner.positionZ.value = position.z;
		} else if (panner.setPosition) {
			panner.setPosition(position.x, position.y, position.z);
		}
	}
	getVelocity() {
		return this.velocity;
	}
	setVelocity(velocity) {
		this.velocity.copy(velocity);
	}
	getMaxDistance() {
		return this.panner.maxDistance;
	}
	setMaxDistance(max) {
		this.panner.maxDistance = max;
	}
	getMinDistance() {
		return this.panner.refDistance;
	}
	setMinDistance(min) {
		this.panner.refDistance = min;
	}
	getRollOffFactor() {
		return this.panner.rolloffFactor;
	}
	setRollOffFactor(factor) {
		this.panner.rolloffFactor = factor;
	}
	getDistanceModel() {
		return this.panner.distanceModel;
	}
	setDistanceModel(distanceModel) {
		this.panner.distanceModel = distanceModel;
	}
	_createSource() {
		const context = this.manager.context;
		this.source = context.createBufferSource();
		this.source.buffer = this.sound.buffer;
		this.source.connect(this.panner);
		this.panner.connect(this.gain);
		this.gain.connect(context.destination);
		if (!this.loop) {
			this.source.onended = this.pause.bind(this);
		}
	}
}
if (!hasAudioContext()) {
	let offset = new Vec3();
	const fallOff = function fallOff(posOne, posTwo, refDistance, maxDistance, rolloffFactor, distanceModel) {
		offset = offset.sub2(posOne, posTwo);
		const distance = offset.length();
		if (distance < refDistance) {
			return 1;
		} else if (distance > maxDistance) {
			return 0;
		}
		let result = 0;
		if (distanceModel === DISTANCE_LINEAR) {
			result = 1 - rolloffFactor * (distance - refDistance) / (maxDistance - refDistance);
		} else if (distanceModel === DISTANCE_INVERSE) {
			result = refDistance / (refDistance + rolloffFactor * (distance - refDistance));
		} else if (distanceModel === DISTANCE_EXPONENTIAL) {
			result = Math.pow(distance / refDistance, -rolloffFactor);
		}
		return math.clamp(result, 0, 1);
	};
	Object.assign(Channel3d.prototype, {
		setPosition: function (position) {
			this.position.copy(position);
			if (this.source) {
				const listener = this.manager.listener;
				const lpos = listener.getPosition();
				const factor = fallOff(lpos, this.position, this.minDistance, this.maxDistance, this.rollOffFactor, this.distanceModel);
				const v = this.getVolume();
				this.source.volume = v * factor;
			}
		},
		getMaxDistance: function () {
			return this.maxDistance;
		},
		setMaxDistance: function (max) {
			this.maxDistance = max;
		},
		getMinDistance: function () {
			return this.minDistance;
		},
		setMinDistance: function (min) {
			this.minDistance = min;
		},
		getRollOffFactor: function () {
			return this.rollOffFactor;
		},
		setRollOffFactor: function (factor) {
			this.rollOffFactor = factor;
		},
		getDistanceModel: function () {
			return this.distanceModel;
		},
		setDistanceModel: function (distanceModel) {
			this.distanceModel = distanceModel;
		}
	});
}

class Listener {
	constructor(manager) {
		this._manager = manager;
		this.position = new Vec3();
		this.velocity = new Vec3();
		this.orientation = new Mat4();
	}
	getPosition() {
		return this.position;
	}
	setPosition(position) {
		this.position.copy(position);
		const listener = this.listener;
		if (listener) {
			if ('positionX' in listener) {
				listener.positionX.value = position.x;
				listener.positionY.value = position.y;
				listener.positionZ.value = position.z;
			} else if (listener.setPosition) {
				listener.setPosition(position.x, position.y, position.z);
			}
		}
	}
	getVelocity() {
		return this.velocity;
	}
	setVelocity(velocity) {}
	setOrientation(orientation) {
		this.orientation.copy(orientation);
		const listener = this.listener;
		if (listener) {
			const m = orientation.data;
			if ('forwardX' in listener) {
				listener.forwardX.value = -m[8];
				listener.forwardY.value = -m[9];
				listener.forwardZ.value = -m[10];
				listener.upX.value = m[4];
				listener.upY.value = m[5];
				listener.upZ.value = m[6];
			} else if (listener.setOrientation) {
				listener.setOrientation(-m[8], -m[9], -m[10], m[4], m[5], m[6]);
			}
		}
	}
	getOrientation() {
		return this.orientation;
	}
	get listener() {
		const context = this._manager.context;
		return context ? context.listener : null;
	}
}

const CONTEXT_STATE_RUNNING = 'running';
const USER_INPUT_EVENTS = ['click', 'touchstart', 'mousedown'];
class SoundManager extends EventHandler {
	constructor() {
		super();
		this._context = null;
		this.AudioContext = typeof AudioContext !== 'undefined' && AudioContext || typeof webkitAudioContext !== 'undefined' && webkitAudioContext;
		if (!this.AudioContext) ;
		this._unlockHandlerFunc = this._unlockHandler.bind(this);
		this._userSuspended = false;
		this.listener = new Listener(this);
		this._volume = 1;
	}
	set volume(volume) {
		volume = math.clamp(volume, 0, 1);
		this._volume = volume;
		this.fire('volumechange', volume);
	}
	get volume() {
		return this._volume;
	}
	get suspended() {
		return this._userSuspended;
	}
	get context() {
		if (!this._context && this.AudioContext) {
			this._context = new this.AudioContext();
			if (this._context.state !== CONTEXT_STATE_RUNNING) {
				this._registerUnlockListeners();
			}
		}
		return this._context;
	}
	suspend() {
		if (!this._userSuspended) {
			this._userSuspended = true;
			if (this._context && this._context.state === CONTEXT_STATE_RUNNING) {
				this._suspend();
			}
		}
	}
	resume() {
		if (this._userSuspended) {
			this._userSuspended = false;
			if (this._context && this._context.state !== CONTEXT_STATE_RUNNING) {
				this._resume();
			}
		}
	}
	destroy() {
		this.fire('destroy');
		if (this._context) {
			var _this$_context;
			this._removeUnlockListeners();
			(_this$_context = this._context) == null || _this$_context.close();
			this._context = null;
		}
	}
	playSound(sound, options = {}) {
		let channel = null;
		if (Channel) {
			channel = new Channel(this, sound, options);
			channel.play();
		}
		return channel;
	}
	playSound3d(sound, position, options = {}) {
		let channel = null;
		if (Channel3d) {
			channel = new Channel3d(this, sound, options);
			channel.setPosition(position);
			if (options.volume) {
				channel.setVolume(options.volume);
			}
			if (options.loop) {
				channel.setLoop(options.loop);
			}
			if (options.maxDistance) {
				channel.setMaxDistance(options.maxDistance);
			}
			if (options.minDistance) {
				channel.setMinDistance(options.minDistance);
			}
			if (options.rollOffFactor) {
				channel.setRollOffFactor(options.rollOffFactor);
			}
			if (options.distanceModel) {
				channel.setDistanceModel(options.distanceModel);
			}
			channel.play();
		}
		return channel;
	}
	_resume() {
		this._context.resume().then(() => {
			const source = this._context.createBufferSource();
			source.buffer = this._context.createBuffer(1, 1, this._context.sampleRate);
			source.connect(this._context.destination);
			source.start(0);
			source.onended = event => {
				source.disconnect(0);
				this.fire('resume');
			};
		}, e => {}).catch(e => {});
	}
	_suspend() {
		this._context.suspend().then(() => {
			this.fire('suspend');
		}, e => {}).catch(e => {});
	}
	_unlockHandler() {
		this._removeUnlockListeners();
		if (!this._userSuspended && this._context.state !== CONTEXT_STATE_RUNNING) {
			this._resume();
		}
	}
	_registerUnlockListeners() {
		USER_INPUT_EVENTS.forEach(eventName => {
			window.addEventListener(eventName, this._unlockHandlerFunc, false);
		});
	}
	_removeUnlockListeners() {
		USER_INPUT_EVENTS.forEach(eventName => {
			window.removeEventListener(eventName, this._unlockHandlerFunc, false);
		});
	}
}

class Sound {
	constructor(resource) {
		this.audio = void 0;
		this.buffer = void 0;
		if (resource instanceof Audio) {
			this.audio = resource;
		} else {
			this.buffer = resource;
		}
	}
	get duration() {
		let duration = 0;
		if (this.buffer) {
			duration = this.buffer.duration;
		} else if (this.audio) {
			duration = this.audio.duration;
		}
		return duration || 0;
	}
}

const STATE_PLAYING = 0;
const STATE_PAUSED = 1;
const STATE_STOPPED = 2;
function capTime(time, duration) {
	return time % duration || 0;
}
class SoundInstance extends EventHandler {
	constructor(manager, sound, options) {
		super();
		this.source = null;
		this._manager = manager;
		this._volume = options.volume !== undefined ? math.clamp(Number(options.volume) || 0, 0, 1) : 1;
		this._pitch = options.pitch !== undefined ? Math.max(0.01, Number(options.pitch) || 0) : 1;
		this._loop = !!(options.loop !== undefined ? options.loop : false);
		this._sound = sound;
		this._state = STATE_STOPPED;
		this._suspended = false;
		this._suspendEndEvent = 0;
		this._suspendInstanceEvents = false;
		this._playWhenLoaded = true;
		this._startTime = Math.max(0, Number(options.startTime) || 0);
		this._duration = Math.max(0, Number(options.duration) || 0);
		this._startOffset = null;
		this._onPlayCallback = options.onPlay;
		this._onPauseCallback = options.onPause;
		this._onResumeCallback = options.onResume;
		this._onStopCallback = options.onStop;
		this._onEndCallback = options.onEnd;
		if (hasAudioContext()) {
			this._startedAt = 0;
			this._currentTime = 0;
			this._currentOffset = 0;
			this._inputNode = null;
			this._connectorNode = null;
			this._firstNode = null;
			this._lastNode = null;
			this._waitingContextSuspension = false;
			this._initializeNodes();
			this._endedHandler = this._onEnded.bind(this);
		} else {
			this._isReady = false;
			this._loadedMetadataHandler = this._onLoadedMetadata.bind(this);
			this._timeUpdateHandler = this._onTimeUpdate.bind(this);
			this._endedHandler = this._onEnded.bind(this);
			this._createSource();
		}
	}
	set currentTime(value) {
		if (value < 0) return;
		if (this._state === STATE_PLAYING) {
			const suspend = this._suspendInstanceEvents;
			this._suspendInstanceEvents = true;
			this.stop();
			this._startOffset = value;
			this.play();
			this._suspendInstanceEvents = suspend;
		} else {
			this._startOffset = value;
			this._currentTime = value;
		}
	}
	get currentTime() {
		if (this._startOffset !== null) {
			return this._startOffset;
		}
		if (this._state === STATE_PAUSED) {
			return this._currentTime;
		}
		if (this._state === STATE_STOPPED || !this.source) {
			return 0;
		}
		this._updateCurrentTime();
		return this._currentTime;
	}
	set duration(value) {
		this._duration = Math.max(0, Number(value) || 0);
		const isPlaying = this._state === STATE_PLAYING;
		this.stop();
		if (isPlaying) {
			this.play();
		}
	}
	get duration() {
		if (!this._sound) {
			return 0;
		}
		if (this._duration) {
			return capTime(this._duration, this._sound.duration);
		}
		return this._sound.duration;
	}
	get isPaused() {
		return this._state === STATE_PAUSED;
	}
	get isPlaying() {
		return this._state === STATE_PLAYING;
	}
	get isStopped() {
		return this._state === STATE_STOPPED;
	}
	get isSuspended() {
		return this._suspended;
	}
	set loop(value) {
		this._loop = !!value;
		if (this.source) {
			this.source.loop = this._loop;
		}
	}
	get loop() {
		return this._loop;
	}
	set pitch(pitch) {
		this._currentOffset = this.currentTime;
		this._startedAt = this._manager.context.currentTime;
		this._pitch = Math.max(Number(pitch) || 0, 0.01);
		if (this.source) {
			this.source.playbackRate.value = this._pitch;
		}
	}
	get pitch() {
		return this._pitch;
	}
	set sound(value) {
		this._sound = value;
		if (this._state !== STATE_STOPPED) {
			this.stop();
		} else {
			this._createSource();
		}
	}
	get sound() {
		return this._sound;
	}
	set startTime(value) {
		this._startTime = Math.max(0, Number(value) || 0);
		const isPlaying = this._state === STATE_PLAYING;
		this.stop();
		if (isPlaying) {
			this.play();
		}
	}
	get startTime() {
		return this._startTime;
	}
	set volume(volume) {
		volume = math.clamp(volume, 0, 1);
		this._volume = volume;
		if (this.gain) {
			this.gain.gain.value = volume * this._manager.volume;
		}
	}
	get volume() {
		return this._volume;
	}
	_onPlay() {
		this.fire('play');
		if (this._onPlayCallback) this._onPlayCallback(this);
	}
	_onPause() {
		this.fire('pause');
		if (this._onPauseCallback) this._onPauseCallback(this);
	}
	_onResume() {
		this.fire('resume');
		if (this._onResumeCallback) this._onResumeCallback(this);
	}
	_onStop() {
		this.fire('stop');
		if (this._onStopCallback) this._onStopCallback(this);
	}
	_onEnded() {
		if (this._suspendEndEvent > 0) {
			this._suspendEndEvent--;
			return;
		}
		this.fire('end');
		if (this._onEndCallback) this._onEndCallback(this);
		this.stop();
	}
	_onManagerVolumeChange() {
		this.volume = this._volume;
	}
	_onManagerSuspend() {
		if (this._state === STATE_PLAYING && !this._suspended) {
			this._suspended = true;
			this.pause();
		}
	}
	_onManagerResume() {
		if (this._suspended) {
			this._suspended = false;
			this.resume();
		}
	}
	_initializeNodes() {
		this.gain = this._manager.context.createGain();
		this._inputNode = this.gain;
		this._connectorNode = this.gain;
		this._connectorNode.connect(this._manager.context.destination);
	}
	play() {
		if (this._state !== STATE_STOPPED) {
			this.stop();
		}
		this._state = STATE_PLAYING;
		this._playWhenLoaded = false;
		if (this._waitingContextSuspension) {
			return false;
		}
		if (this._manager.suspended) {
			this._manager.once('resume', this._playAudioImmediate, this);
			this._waitingContextSuspension = true;
			return false;
		}
		this._playAudioImmediate();
		return true;
	}
	_playAudioImmediate() {
		this._waitingContextSuspension = false;
		if (this._state !== STATE_PLAYING) {
			return;
		}
		if (!this.source) {
			this._createSource();
		}
		let offset = capTime(this._startOffset, this.duration);
		offset = capTime(this._startTime + offset, this._sound.duration);
		this._startOffset = null;
		if (this._duration) {
			this.source.start(0, offset, this._duration);
		} else {
			this.source.start(0, offset);
		}
		this._startedAt = this._manager.context.currentTime;
		this._currentTime = 0;
		this._currentOffset = offset;
		this.volume = this._volume;
		this.loop = this._loop;
		this.pitch = this._pitch;
		this._manager.on('volumechange', this._onManagerVolumeChange, this);
		this._manager.on('suspend', this._onManagerSuspend, this);
		this._manager.on('resume', this._onManagerResume, this);
		this._manager.on('destroy', this._onManagerDestroy, this);
		if (!this._suspendInstanceEvents) {
			this._onPlay();
		}
	}
	pause() {
		this._playWhenLoaded = false;
		if (this._state !== STATE_PLAYING) return false;
		this._state = STATE_PAUSED;
		if (this._waitingContextSuspension) {
			return true;
		}
		this._updateCurrentTime();
		this._suspendEndEvent++;
		this.source.stop(0);
		this.source = null;
		this._startOffset = null;
		if (!this._suspendInstanceEvents) this._onPause();
		return true;
	}
	resume() {
		if (this._state !== STATE_PAUSED) {
			return false;
		}
		let offset = this.currentTime;
		this._state = STATE_PLAYING;
		if (this._waitingContextSuspension) {
			return true;
		}
		if (!this.source) {
			this._createSource();
		}
		if (this._startOffset !== null) {
			offset = capTime(this._startOffset, this.duration);
			offset = capTime(this._startTime + offset, this._sound.duration);
			this._startOffset = null;
		}
		if (this._duration) {
			this.source.start(0, offset, this._duration);
		} else {
			this.source.start(0, offset);
		}
		this._startedAt = this._manager.context.currentTime;
		this._currentOffset = offset;
		this.volume = this._volume;
		this.loop = this._loop;
		this.pitch = this._pitch;
		this._playWhenLoaded = false;
		if (!this._suspendInstanceEvents) this._onResume();
		return true;
	}
	stop() {
		this._playWhenLoaded = false;
		if (this._state === STATE_STOPPED) return false;
		const wasPlaying = this._state === STATE_PLAYING;
		this._state = STATE_STOPPED;
		if (this._waitingContextSuspension) {
			return true;
		}
		this._manager.off('volumechange', this._onManagerVolumeChange, this);
		this._manager.off('suspend', this._onManagerSuspend, this);
		this._manager.off('resume', this._onManagerResume, this);
		this._manager.off('destroy', this._onManagerDestroy, this);
		this._startedAt = 0;
		this._currentTime = 0;
		this._currentOffset = 0;
		this._startOffset = null;
		this._suspendEndEvent++;
		if (wasPlaying && this.source) {
			this.source.stop(0);
		}
		this.source = null;
		if (!this._suspendInstanceEvents) this._onStop();
		return true;
	}
	setExternalNodes(firstNode, lastNode) {
		if (!firstNode) {
			console.error('The firstNode must be a valid Audio Node');
			return;
		}
		if (!lastNode) {
			lastNode = firstNode;
		}
		const speakers = this._manager.context.destination;
		if (this._firstNode !== firstNode) {
			if (this._firstNode) {
				this._connectorNode.disconnect(this._firstNode);
			} else {
				this._connectorNode.disconnect(speakers);
			}
			this._firstNode = firstNode;
			this._connectorNode.connect(firstNode);
		}
		if (this._lastNode !== lastNode) {
			if (this._lastNode) {
				this._lastNode.disconnect(speakers);
			}
			this._lastNode = lastNode;
			this._lastNode.connect(speakers);
		}
	}
	clearExternalNodes() {
		const speakers = this._manager.context.destination;
		if (this._firstNode) {
			this._connectorNode.disconnect(this._firstNode);
			this._firstNode = null;
		}
		if (this._lastNode) {
			this._lastNode.disconnect(speakers);
			this._lastNode = null;
		}
		this._connectorNode.connect(speakers);
	}
	getExternalNodes() {
		return [this._firstNode, this._lastNode];
	}
	_createSource() {
		if (!this._sound) {
			return null;
		}
		const context = this._manager.context;
		if (this._sound.buffer) {
			this.source = context.createBufferSource();
			this.source.buffer = this._sound.buffer;
			this.source.connect(this._inputNode);
			this.source.onended = this._endedHandler;
			this.source.loopStart = capTime(this._startTime, this.source.buffer.duration);
			if (this._duration) {
				this.source.loopEnd = Math.max(this.source.loopStart, capTime(this._startTime + this._duration, this.source.buffer.duration));
			}
		}
		return this.source;
	}
	_updateCurrentTime() {
		this._currentTime = capTime((this._manager.context.currentTime - this._startedAt) * this._pitch + this._currentOffset, this.duration);
	}
	_onManagerDestroy() {
		if (this.source && this._state === STATE_PLAYING) {
			this.source.stop(0);
			this.source = null;
		}
	}
}
SoundInstance.EVENT_PLAY = 'play';
SoundInstance.EVENT_PAUSE = 'pause';
SoundInstance.EVENT_RESUME = 'resume';
SoundInstance.EVENT_STOP = 'stop';
SoundInstance.EVENT_END = 'end';
if (!hasAudioContext()) {
	Object.assign(SoundInstance.prototype, {
		play: function () {
			if (this._state !== STATE_STOPPED) {
				this.stop();
			}
			if (!this.source) {
				if (!this._createSource()) {
					return false;
				}
			}
			this.volume = this._volume;
			this.pitch = this._pitch;
			this.loop = this._loop;
			this.source.play();
			this._state = STATE_PLAYING;
			this._playWhenLoaded = false;
			this._manager.on('volumechange', this._onManagerVolumeChange, this);
			this._manager.on('suspend', this._onManagerSuspend, this);
			this._manager.on('resume', this._onManagerResume, this);
			this._manager.on('destroy', this._onManagerDestroy, this);
			if (this._manager.suspended) this._onManagerSuspend();
			if (!this._suspendInstanceEvents) this._onPlay();
			return true;
		},
		pause: function () {
			if (!this.source || this._state !== STATE_PLAYING) return false;
			this._suspendEndEvent++;
			this.source.pause();
			this._playWhenLoaded = false;
			this._state = STATE_PAUSED;
			this._startOffset = null;
			if (!this._suspendInstanceEvents) this._onPause();
			return true;
		},
		resume: function () {
			if (!this.source || this._state !== STATE_PAUSED) return false;
			this._state = STATE_PLAYING;
			this._playWhenLoaded = false;
			if (this.source.paused) {
				this.source.play();
				if (!this._suspendInstanceEvents) this._onResume();
			}
			return true;
		},
		stop: function () {
			if (!this.source || this._state === STATE_STOPPED) return false;
			this._manager.off('volumechange', this._onManagerVolumeChange, this);
			this._manager.off('suspend', this._onManagerSuspend, this);
			this._manager.off('resume', this._onManagerResume, this);
			this._manager.off('destroy', this._onManagerDestroy, this);
			this._suspendEndEvent++;
			this.source.pause();
			this._playWhenLoaded = false;
			this._state = STATE_STOPPED;
			this._startOffset = null;
			if (!this._suspendInstanceEvents) this._onStop();
			return true;
		},
		setExternalNodes: function () {},
		clearExternalNodes: function () {},
		getExternalNodes: function () {
			return [null, null];
		},
		_onLoadedMetadata: function () {
			this.source.removeEventListener('loadedmetadata', this._loadedMetadataHandler);
			this._isReady = true;
			let offset = capTime(this._startOffset, this.duration);
			offset = capTime(this._startTime + offset, this._sound.duration);
			this._startOffset = null;
			this.source.currentTime = offset;
		},
		_createSource: function () {
			if (this._sound && this._sound.audio) {
				this._isReady = false;
				this.source = this._sound.audio.cloneNode(true);
				this.source.addEventListener('loadedmetadata', this._loadedMetadataHandler);
				this.source.addEventListener('timeupdate', this._timeUpdateHandler);
				this.source.onended = this._endedHandler;
			}
			return this.source;
		},
		_onTimeUpdate: function () {
			if (!this._duration) return;
			if (this.source.currentTime > capTime(this._startTime + this._duration, this.source.duration)) {
				if (this.loop) {
					this.source.currentTime = capTime(this._startTime, this.source.duration);
				} else {
					this.source.removeEventListener('timeupdate', this._timeUpdateHandler);
					this.source.pause();
					this._onEnded();
				}
			}
		},
		_onManagerDestroy: function () {
			if (this.source) {
				this.source.pause();
			}
		}
	});
	Object.defineProperty(SoundInstance.prototype, 'volume', {
		get: function () {
			return this._volume;
		},
		set: function (volume) {
			volume = math.clamp(volume, 0, 1);
			this._volume = volume;
			if (this.source) {
				this.source.volume = volume * this._manager.volume;
			}
		}
	});
	Object.defineProperty(SoundInstance.prototype, 'pitch', {
		get: function () {
			return this._pitch;
		},
		set: function (pitch) {
			this._pitch = Math.max(Number(pitch) || 0, 0.01);
			if (this.source) {
				this.source.playbackRate = this._pitch;
			}
		}
	});
	Object.defineProperty(SoundInstance.prototype, 'sound', {
		get: function () {
			return this._sound;
		},
		set: function (value) {
			this.stop();
			this._sound = value;
		}
	});
	Object.defineProperty(SoundInstance.prototype, 'currentTime', {
		get: function () {
			if (this._startOffset !== null) {
				return this._startOffset;
			}
			if (this._state === STATE_STOPPED || !this.source) {
				return 0;
			}
			return this.source.currentTime - this._startTime;
		},
		set: function (value) {
			if (value < 0) return;
			this._startOffset = value;
			if (this.source && this._isReady) {
				this.source.currentTime = capTime(this._startTime + capTime(value, this.duration), this._sound.duration);
				this._startOffset = null;
			}
		}
	});
}

const MAX_DISTANCE = 10000;
class SoundInstance3d extends SoundInstance {
	constructor(manager, sound, options = {}) {
		super(manager, sound, options);
		this._position = new Vec3();
		this._velocity = new Vec3();
		if (options.position) this.position = options.position;
		this.maxDistance = options.maxDistance !== undefined ? Number(options.maxDistance) : MAX_DISTANCE;
		this.refDistance = options.refDistance !== undefined ? Number(options.refDistance) : 1;
		this.rollOffFactor = options.rollOffFactor !== undefined ? Number(options.rollOffFactor) : 1;
		this.distanceModel = options.distanceModel !== undefined ? options.distanceModel : DISTANCE_LINEAR;
	}
	_initializeNodes() {
		this.gain = this._manager.context.createGain();
		this.panner = this._manager.context.createPanner();
		this.panner.connect(this.gain);
		this._inputNode = this.panner;
		this._connectorNode = this.gain;
		this._connectorNode.connect(this._manager.context.destination);
	}
	set position(value) {
		this._position.copy(value);
		const panner = this.panner;
		if ('positionX' in panner) {
			panner.positionX.value = value.x;
			panner.positionY.value = value.y;
			panner.positionZ.value = value.z;
		} else if (panner.setPosition) {
			panner.setPosition(value.x, value.y, value.z);
		}
	}
	get position() {
		return this._position;
	}
	set velocity(velocity) {
		this._velocity.copy(velocity);
	}
	get velocity() {
		return this._velocity;
	}
	set maxDistance(value) {
		this.panner.maxDistance = value;
	}
	get maxDistance() {
		return this.panner.maxDistance;
	}
	set refDistance(value) {
		this.panner.refDistance = value;
	}
	get refDistance() {
		return this.panner.refDistance;
	}
	set rollOffFactor(value) {
		this.panner.rolloffFactor = value;
	}
	get rollOffFactor() {
		return this.panner.rolloffFactor;
	}
	set distanceModel(value) {
		this.panner.distanceModel = value;
	}
	get distanceModel() {
		return this.panner.distanceModel;
	}
}
if (!hasAudioContext()) {
	let offset = new Vec3();
	const fallOff = function fallOff(posOne, posTwo, refDistance, maxDistance, rollOffFactor, distanceModel) {
		offset = offset.sub2(posOne, posTwo);
		const distance = offset.length();
		if (distance < refDistance) {
			return 1;
		} else if (distance > maxDistance) {
			return 0;
		}
		let result = 0;
		if (distanceModel === DISTANCE_LINEAR) {
			result = 1 - rollOffFactor * (distance - refDistance) / (maxDistance - refDistance);
		} else if (distanceModel === DISTANCE_INVERSE) {
			result = refDistance / (refDistance + rollOffFactor * (distance - refDistance));
		} else if (distanceModel === DISTANCE_EXPONENTIAL) {
			result = Math.pow(distance / refDistance, -rollOffFactor);
		}
		return math.clamp(result, 0, 1);
	};
	Object.defineProperty(SoundInstance3d.prototype, 'position', {
		get: function () {
			return this._position;
		},
		set: function (position) {
			this._position.copy(position);
			if (this.source) {
				const listener = this._manager.listener;
				const lpos = listener.getPosition();
				const factor = fallOff(lpos, this._position, this.refDistance, this.maxDistance, this.rollOffFactor, this.distanceModel);
				const v = this.volume;
				this.source.volume = v * factor * this._manager.volume;
			}
		}
	});
	Object.defineProperty(SoundInstance3d.prototype, 'maxDistance', {
		get: function () {
			return this._maxDistance;
		},
		set: function (value) {
			this._maxDistance = value;
		}
	});
	Object.defineProperty(SoundInstance3d.prototype, 'refDistance', {
		get: function () {
			return this._refDistance;
		},
		set: function (value) {
			this._refDistance = value;
		}
	});
	Object.defineProperty(SoundInstance3d.prototype, 'rollOffFactor', {
		get: function () {
			return this._rollOffFactor;
		},
		set: function (value) {
			this._rollOffFactor = value;
		}
	});
	Object.defineProperty(SoundInstance3d.prototype, 'distanceModel', {
		get: function () {
			return this._distanceModel;
		},
		set: function (value) {
			this._distanceModel = value;
		}
	});
}

const BLEND_SUBTRACTIVE = 0;
const BLEND_ADDITIVE = 1;
const BLEND_NORMAL = 2;
const BLEND_NONE = 3;
const BLEND_PREMULTIPLIED = 4;
const BLEND_MULTIPLICATIVE = 5;
const BLEND_ADDITIVEALPHA = 6;
const BLEND_MULTIPLICATIVE2X = 7;
const BLEND_SCREEN = 8;
const BLEND_MIN = 9;
const BLEND_MAX = 10;
const FOG_NONE = 'none';
const FOG_LINEAR = 'linear';
const FOG_EXP = 'exp';
const FOG_EXP2 = 'exp2';
const FRESNEL_NONE = 0;
const FRESNEL_SCHLICK = 2;
const LAYER_HUD = 0;
const LAYER_GIZMO = 1;
const LAYER_FX = 2;
const LAYER_WORLD = 15;
const LAYERID_WORLD = 0;
const LAYERID_DEPTH = 1;
const LAYERID_SKYBOX = 2;
const LAYERID_IMMEDIATE = 3;
const LAYERID_UI = 4;
const LIGHTTYPE_DIRECTIONAL = 0;
const LIGHTTYPE_OMNI = 1;
const LIGHTTYPE_POINT = LIGHTTYPE_OMNI;
const LIGHTTYPE_SPOT = 2;
const LIGHTTYPE_COUNT = 3;
const LIGHTSHAPE_PUNCTUAL = 0;
const LIGHTSHAPE_RECT = 1;
const LIGHTSHAPE_DISK = 2;
const LIGHTSHAPE_SPHERE = 3;
const LIGHTFALLOFF_LINEAR = 0;
const LIGHTFALLOFF_INVERSESQUARED = 1;
const SHADOW_PCF3 = 0;
const SHADOW_DEPTH = 0;
const SHADOW_VSM8 = 1;
const SHADOW_VSM16 = 2;
const SHADOW_VSM32 = 3;
const SHADOW_PCF5 = 4;
const SHADOW_PCF1 = 5;
const SHADOW_PCSS = 6;
const shadowTypeToString = {};
shadowTypeToString[SHADOW_PCF3] = 'PCF3';
shadowTypeToString[SHADOW_VSM8] = 'VSM8';
shadowTypeToString[SHADOW_VSM16] = 'VSM16';
shadowTypeToString[SHADOW_VSM32] = 'VSM32';
shadowTypeToString[SHADOW_PCF5] = 'PCF5';
shadowTypeToString[SHADOW_PCF1] = 'PCF1';
shadowTypeToString[SHADOW_PCSS] = 'PCSS';
const BLUR_BOX = 0;
const BLUR_GAUSSIAN = 1;
const PARTICLESORT_NONE = 0;
const PARTICLESORT_DISTANCE = 1;
const PARTICLESORT_NEWER_FIRST = 2;
const PARTICLESORT_OLDER_FIRST = 3;
const PARTICLEMODE_GPU = 0;
const PARTICLEMODE_CPU = 1;
const EMITTERSHAPE_BOX = 0;
const EMITTERSHAPE_SPHERE = 1;
const PARTICLEORIENTATION_SCREEN = 0;
const PARTICLEORIENTATION_WORLD = 1;
const PARTICLEORIENTATION_EMITTER = 2;
const PROJECTION_PERSPECTIVE = 0;
const PROJECTION_ORTHOGRAPHIC = 1;
const RENDERSTYLE_SOLID = 0;
const RENDERSTYLE_WIREFRAME = 1;
const RENDERSTYLE_POINTS = 2;
const CUBEPROJ_NONE = 0;
const CUBEPROJ_BOX = 1;
const SPECULAR_PHONG = 0;
const SPECULAR_BLINN = 1;
const DETAILMODE_MUL = 'mul';
const DETAILMODE_ADD = 'add';
const DETAILMODE_SCREEN = 'screen';
const DETAILMODE_OVERLAY = 'overlay';
const DETAILMODE_MIN = 'min';
const DETAILMODE_MAX = 'max';
const GAMMA_NONE = 0;
const GAMMA_SRGB = 1;
const GAMMA_SRGBFAST = 2;
const GAMMA_SRGBHDR = 3;
const TONEMAP_LINEAR = 0;
const TONEMAP_FILMIC = 1;
const TONEMAP_HEJL = 2;
const TONEMAP_ACES = 3;
const TONEMAP_ACES2 = 4;
const SPECOCC_NONE = 0;
const SPECOCC_AO = 1;
const SPECOCC_GLOSSDEPENDENT = 2;
const SHADERDEF_NOSHADOW = 1;
const SHADERDEF_SKIN = 2;
const SHADERDEF_UV0 = 4;
const SHADERDEF_UV1 = 8;
const SHADERDEF_VCOLOR = 16;
const SHADERDEF_INSTANCING = 32;
const SHADERDEF_LM = 64;
const SHADERDEF_DIRLM = 128;
const SHADERDEF_SCREENSPACE = 256;
const SHADERDEF_TANGENTS = 512;
const SHADERDEF_MORPH_POSITION = 1024;
const SHADERDEF_MORPH_NORMAL = 2048;
const SHADERDEF_MORPH_TEXTURE_BASED = 4096;
const SHADERDEF_LMAMBIENT = 8192;
const SHADOWUPDATE_NONE = 0;
const SHADOWUPDATE_THISFRAME = 1;
const SHADOWUPDATE_REALTIME = 2;
const SORTKEY_FORWARD = 0;
const SORTKEY_DEPTH = 1;
const MASK_AFFECT_DYNAMIC = 1;
const MASK_AFFECT_LIGHTMAPPED = 2;
const MASK_BAKE = 4;
const SHADER_FORWARD = 0;
const SHADER_FORWARDHDR = 1;
const SHADER_DEPTH = 2;
const SHADER_PICK = 3;
const SHADER_SHADOW = 4;
const SHADER_PREPASS_VELOCITY = 5;
const SHADERPASS_FORWARD = 'forward';
const SHADERPASS_ALBEDO = 'debug_albedo';
const SHADERPASS_WORLDNORMAL = 'debug_world_normal';
const SHADERPASS_OPACITY = 'debug_opacity';
const SHADERPASS_SPECULARITY = 'debug_specularity';
const SHADERPASS_GLOSS = 'debug_gloss';
const SHADERPASS_METALNESS = 'debug_metalness';
const SHADERPASS_AO = 'debug_ao';
const SHADERPASS_EMISSION = 'debug_emission';
const SHADERPASS_LIGHTING = 'debug_lighting';
const SHADERPASS_UV0 = 'debug_uv0';
const SPRITE_RENDERMODE_SIMPLE = 0;
const SPRITE_RENDERMODE_SLICED = 1;
const SPRITE_RENDERMODE_TILED = 2;
const BAKE_COLOR = 0;
const BAKE_COLORDIR = 1;
const VIEW_CENTER = 0;
const VIEW_LEFT = 1;
const VIEW_RIGHT = 2;
const SORTMODE_NONE = 0;
const SORTMODE_MANUAL = 1;
const SORTMODE_MATERIALMESH = 2;
const SORTMODE_BACK2FRONT = 3;
const SORTMODE_FRONT2BACK = 4;
const SORTMODE_CUSTOM = 5;
const ASPECT_AUTO = 0;
const ASPECT_MANUAL = 1;
const ORIENTATION_HORIZONTAL = 0;
const ORIENTATION_VERTICAL = 1;
const SKYTYPE_INFINITE = 'infinite';
const SKYTYPE_BOX = 'box';
const SKYTYPE_DOME = 'dome';
const DITHER_NONE = 'none';
const DITHER_BAYER8 = 'bayer8';
const DITHER_BLUENOISE = 'bluenoise';

class RefCountedObject {
	constructor() {
		this._refCount = 0;
	}
	incRefCount() {
		this._refCount++;
	}
	decRefCount() {
		this._refCount--;
	}
	get refCount() {
		return this._refCount;
	}
}

let id$3 = 0;
class GeometryData {
	constructor() {
		this.initDefaults();
	}
	initDefaults() {
		this.recreate = false;
		this.verticesUsage = BUFFER_STATIC;
		this.indicesUsage = BUFFER_STATIC;
		this.maxVertices = 0;
		this.maxIndices = 0;
		this.vertexCount = 0;
		this.indexCount = 0;
		this.vertexStreamsUpdated = false;
		this.indexStreamUpdated = false;
		this.vertexStreamDictionary = {};
		this.indices = null;
	}
	_changeVertexCount(count, semantic) {
		if (!this.vertexCount) {
			this.vertexCount = count;
		}
	}
}
GeometryData.DEFAULT_COMPONENTS_POSITION = 3;
GeometryData.DEFAULT_COMPONENTS_NORMAL = 3;
GeometryData.DEFAULT_COMPONENTS_UV = 2;
GeometryData.DEFAULT_COMPONENTS_COLORS = 4;
class GeometryVertexStream {
	constructor(data, componentCount, dataType, dataTypeNormalize) {
		this.data = data;
		this.componentCount = componentCount;
		this.dataType = dataType;
		this.dataTypeNormalize = dataTypeNormalize;
	}
}
class Mesh extends RefCountedObject {
	constructor(graphicsDevice) {
		super();
		this._aabbVer = 0;
		this._aabb = new BoundingBox();
		this.id = id$3++;
		this.device = graphicsDevice;
		this.vertexBuffer = null;
		this.indexBuffer = [null];
		this.primitive = [{
			type: 0,
			base: 0,
			count: 0
		}];
		this.skin = null;
		this._morph = null;
		this._geometryData = null;
		this.boneAabb = null;
	}
	set morph(morph) {
		if (morph !== this._morph) {
			if (this._morph) {
				this._morph.decRefCount();
			}
			this._morph = morph;
			if (morph) {
				morph.incRefCount();
			}
		}
	}
	get morph() {
		return this._morph;
	}
	set aabb(aabb) {
		this._aabb = aabb;
		this._aabbVer++;
	}
	get aabb() {
		return this._aabb;
	}
	destroy() {
		const morph = this.morph;
		if (morph) {
			this.morph = null;
			if (morph.refCount < 1) {
				morph.destroy();
			}
		}
		if (this.vertexBuffer) {
			this.vertexBuffer.destroy();
			this.vertexBuffer = null;
		}
		for (let j = 0; j < this.indexBuffer.length; j++) {
			this._destroyIndexBuffer(j);
		}
		this.indexBuffer.length = 0;
		this._geometryData = null;
	}
	_destroyIndexBuffer(index) {
		if (this.indexBuffer[index]) {
			this.indexBuffer[index].destroy();
			this.indexBuffer[index] = null;
		}
	}
	_initBoneAabbs(morphTargets) {
		this.boneAabb = [];
		this.boneUsed = [];
		let x, y, z;
		let bMax, bMin;
		const boneMin = [];
		const boneMax = [];
		const boneUsed = this.boneUsed;
		const numBones = this.skin.boneNames.length;
		let maxMorphX, maxMorphY, maxMorphZ;
		for (let i = 0; i < numBones; i++) {
			boneMin[i] = new Vec3(Number.MAX_VALUE, Number.MAX_VALUE, Number.MAX_VALUE);
			boneMax[i] = new Vec3(-Number.MAX_VALUE, -Number.MAX_VALUE, -Number.MAX_VALUE);
		}
		const iterator = new VertexIterator(this.vertexBuffer);
		const posElement = iterator.element[SEMANTIC_POSITION];
		const weightsElement = iterator.element[SEMANTIC_BLENDWEIGHT];
		const indicesElement = iterator.element[SEMANTIC_BLENDINDICES];
		const numVerts = this.vertexBuffer.numVertices;
		for (let j = 0; j < numVerts; j++) {
			for (let k = 0; k < 4; k++) {
				const boneWeight = weightsElement.array[weightsElement.index + k];
				if (boneWeight > 0) {
					const boneIndex = indicesElement.array[indicesElement.index + k];
					boneUsed[boneIndex] = true;
					x = posElement.array[posElement.index];
					y = posElement.array[posElement.index + 1];
					z = posElement.array[posElement.index + 2];
					bMax = boneMax[boneIndex];
					bMin = boneMin[boneIndex];
					if (bMin.x > x) bMin.x = x;
					if (bMin.y > y) bMin.y = y;
					if (bMin.z > z) bMin.z = z;
					if (bMax.x < x) bMax.x = x;
					if (bMax.y < y) bMax.y = y;
					if (bMax.z < z) bMax.z = z;
					if (morphTargets) {
						let minMorphX = maxMorphX = x;
						let minMorphY = maxMorphY = y;
						let minMorphZ = maxMorphZ = z;
						for (let l = 0; l < morphTargets.length; l++) {
							const target = morphTargets[l];
							const dx = target.deltaPositions[j * 3];
							const dy = target.deltaPositions[j * 3 + 1];
							const dz = target.deltaPositions[j * 3 + 2];
							if (dx < 0) {
								minMorphX += dx;
							} else {
								maxMorphX += dx;
							}
							if (dy < 0) {
								minMorphY += dy;
							} else {
								maxMorphY += dy;
							}
							if (dz < 0) {
								minMorphZ += dz;
							} else {
								maxMorphZ += dz;
							}
						}
						if (bMin.x > minMorphX) bMin.x = minMorphX;
						if (bMin.y > minMorphY) bMin.y = minMorphY;
						if (bMin.z > minMorphZ) bMin.z = minMorphZ;
						if (bMax.x < maxMorphX) bMax.x = maxMorphX;
						if (bMax.y < maxMorphY) bMax.y = maxMorphY;
						if (bMax.z < maxMorphZ) bMax.z = maxMorphZ;
					}
				}
			}
			iterator.next();
		}
		const positionElement = this.vertexBuffer.getFormat().elements.find(e => e.name === SEMANTIC_POSITION);
		if (positionElement && positionElement.normalize) {
			const func = (() => {
				switch (positionElement.dataType) {
					case TYPE_INT8:
						return x => Math.max(x / 127.0, -1.0);
					case TYPE_UINT8:
						return x => x / 255.0;
					case TYPE_INT16:
						return x => Math.max(x / 32767.0, -1.0);
					case TYPE_UINT16:
						return x => x / 65535.0;
					default:
						return x => x;
				}
			})();
			for (let i = 0; i < numBones; i++) {
				if (boneUsed[i]) {
					const min = boneMin[i];
					const max = boneMax[i];
					min.set(func(min.x), func(min.y), func(min.z));
					max.set(func(max.x), func(max.y), func(max.z));
				}
			}
		}
		for (let i = 0; i < numBones; i++) {
			const aabb = new BoundingBox();
			aabb.setMinMax(boneMin[i], boneMax[i]);
			this.boneAabb.push(aabb);
		}
	}
	_initGeometryData() {
		if (!this._geometryData) {
			this._geometryData = new GeometryData();
			if (this.vertexBuffer) {
				this._geometryData.vertexCount = this.vertexBuffer.numVertices;
				this._geometryData.maxVertices = this.vertexBuffer.numVertices;
			}
			if (this.indexBuffer.length > 0 && this.indexBuffer[0]) {
				this._geometryData.indexCount = this.indexBuffer[0].numIndices;
				this._geometryData.maxIndices = this.indexBuffer[0].numIndices;
			}
		}
	}
	clear(verticesDynamic, indicesDynamic, maxVertices = 0, maxIndices = 0) {
		this._initGeometryData();
		this._geometryData.initDefaults();
		this._geometryData.recreate = true;
		this._geometryData.maxVertices = maxVertices;
		this._geometryData.maxIndices = maxIndices;
		this._geometryData.verticesUsage = verticesDynamic ? BUFFER_STATIC : BUFFER_DYNAMIC;
		this._geometryData.indicesUsage = indicesDynamic ? BUFFER_STATIC : BUFFER_DYNAMIC;
	}
	setVertexStream(semantic, data, componentCount, numVertices, dataType = TYPE_FLOAT32, dataTypeNormalize = false) {
		this._initGeometryData();
		const vertexCount = numVertices || data.length / componentCount;
		this._geometryData._changeVertexCount(vertexCount, semantic);
		this._geometryData.vertexStreamsUpdated = true;
		this._geometryData.vertexStreamDictionary[semantic] = new GeometryVertexStream(data, componentCount, dataType, dataTypeNormalize);
	}
	getVertexStream(semantic, data) {
		let count = 0;
		let done = false;
		if (this._geometryData) {
			const stream = this._geometryData.vertexStreamDictionary[semantic];
			if (stream) {
				done = true;
				count = this._geometryData.vertexCount;
				if (ArrayBuffer.isView(data)) {
					data.set(stream.data);
				} else {
					data.length = 0;
					data.push(stream.data);
				}
			}
		}
		if (!done) {
			if (this.vertexBuffer) {
				const iterator = new VertexIterator(this.vertexBuffer);
				count = iterator.readData(semantic, data);
			}
		}
		return count;
	}
	setPositions(positions, componentCount = GeometryData.DEFAULT_COMPONENTS_POSITION, numVertices) {
		this.setVertexStream(SEMANTIC_POSITION, positions, componentCount, numVertices, TYPE_FLOAT32, false);
	}
	setNormals(normals, componentCount = GeometryData.DEFAULT_COMPONENTS_NORMAL, numVertices) {
		this.setVertexStream(SEMANTIC_NORMAL, normals, componentCount, numVertices, TYPE_FLOAT32, false);
	}
	setUvs(channel, uvs, componentCount = GeometryData.DEFAULT_COMPONENTS_UV, numVertices) {
		this.setVertexStream(SEMANTIC_TEXCOORD + channel, uvs, componentCount, numVertices, TYPE_FLOAT32, false);
	}
	setColors(colors, componentCount = GeometryData.DEFAULT_COMPONENTS_COLORS, numVertices) {
		this.setVertexStream(SEMANTIC_COLOR, colors, componentCount, numVertices, TYPE_FLOAT32, false);
	}
	setColors32(colors, numVertices) {
		this.setVertexStream(SEMANTIC_COLOR, colors, GeometryData.DEFAULT_COMPONENTS_COLORS, numVertices, TYPE_UINT8, true);
	}
	setIndices(indices, numIndices) {
		this._initGeometryData();
		this._geometryData.indexStreamUpdated = true;
		this._geometryData.indices = indices;
		this._geometryData.indexCount = numIndices || indices.length;
	}
	getPositions(positions) {
		return this.getVertexStream(SEMANTIC_POSITION, positions);
	}
	getNormals(normals) {
		return this.getVertexStream(SEMANTIC_NORMAL, normals);
	}
	getUvs(channel, uvs) {
		return this.getVertexStream(SEMANTIC_TEXCOORD + channel, uvs);
	}
	getColors(colors) {
		return this.getVertexStream(SEMANTIC_COLOR, colors);
	}
	getIndices(indices) {
		let count = 0;
		if (this._geometryData && this._geometryData.indices) {
			const streamIndices = this._geometryData.indices;
			count = this._geometryData.indexCount;
			if (ArrayBuffer.isView(indices)) {
				indices.set(streamIndices);
			} else {
				indices.length = 0;
				for (let i = 0, il = streamIndices.length; i < il; i++) {
					indices.push(streamIndices[i]);
				}
			}
		} else {
			if (this.indexBuffer.length > 0 && this.indexBuffer[0]) {
				const indexBuffer = this.indexBuffer[0];
				count = indexBuffer.readData(indices);
			}
		}
		return count;
	}
	update(primitiveType = PRIMITIVE_TRIANGLES, updateBoundingBox = true) {
		if (this._geometryData) {
			if (updateBoundingBox) {
				const stream = this._geometryData.vertexStreamDictionary[SEMANTIC_POSITION];
				if (stream) {
					if (stream.componentCount === 3) {
						this._aabb.compute(stream.data, this._geometryData.vertexCount);
						this._aabbVer++;
					}
				}
			}
			let destroyVB = this._geometryData.recreate;
			if (this._geometryData.vertexCount > this._geometryData.maxVertices) {
				destroyVB = true;
				this._geometryData.maxVertices = this._geometryData.vertexCount;
			}
			if (destroyVB) {
				if (this.vertexBuffer) {
					this.vertexBuffer.destroy();
					this.vertexBuffer = null;
				}
			}
			let destroyIB = this._geometryData.recreate;
			if (this._geometryData.indexCount > this._geometryData.maxIndices) {
				destroyIB = true;
				this._geometryData.maxIndices = this._geometryData.indexCount;
			}
			if (destroyIB) {
				if (this.indexBuffer.length > 0 && this.indexBuffer[0]) {
					this.indexBuffer[0].destroy();
					this.indexBuffer[0] = null;
				}
			}
			if (this._geometryData.vertexStreamsUpdated) {
				this._updateVertexBuffer();
			}
			if (this._geometryData.indexStreamUpdated) {
				this._updateIndexBuffer();
			}
			this.primitive[0].type = primitiveType;
			if (this.indexBuffer.length > 0 && this.indexBuffer[0]) {
				if (this._geometryData.indexStreamUpdated) {
					this.primitive[0].count = this._geometryData.indexCount;
					this.primitive[0].indexed = true;
				}
			} else {
				if (this._geometryData.vertexStreamsUpdated) {
					this.primitive[0].count = this._geometryData.vertexCount;
					this.primitive[0].indexed = false;
				}
			}
			this._geometryData.vertexCount = 0;
			this._geometryData.indexCount = 0;
			this._geometryData.vertexStreamsUpdated = false;
			this._geometryData.indexStreamUpdated = false;
			this._geometryData.recreate = false;
			this.updateRenderStates();
		}
	}
	_buildVertexFormat(vertexCount) {
		const vertexDesc = [];
		for (const semantic in this._geometryData.vertexStreamDictionary) {
			const stream = this._geometryData.vertexStreamDictionary[semantic];
			vertexDesc.push({
				semantic: semantic,
				components: stream.componentCount,
				type: stream.dataType,
				normalize: stream.dataTypeNormalize
			});
		}
		return new VertexFormat(this.device, vertexDesc, vertexCount);
	}
	_updateVertexBuffer() {
		if (!this.vertexBuffer) {
			const allocateVertexCount = this._geometryData.maxVertices;
			const format = this._buildVertexFormat(allocateVertexCount);
			this.vertexBuffer = new VertexBuffer(this.device, format, allocateVertexCount, this._geometryData.verticesUsage);
		}
		const iterator = new VertexIterator(this.vertexBuffer);
		const numVertices = this._geometryData.vertexCount;
		for (const semantic in this._geometryData.vertexStreamDictionary) {
			const stream = this._geometryData.vertexStreamDictionary[semantic];
			iterator.writeData(semantic, stream.data, numVertices);
			delete this._geometryData.vertexStreamDictionary[semantic];
		}
		iterator.end();
	}
	_updateIndexBuffer() {
		if (this.indexBuffer.length <= 0 || !this.indexBuffer[0]) {
			const createFormat = this._geometryData.maxVertices > 0xffff ? INDEXFORMAT_UINT32 : INDEXFORMAT_UINT16;
			this.indexBuffer[0] = new IndexBuffer(this.device, createFormat, this._geometryData.maxIndices, this._geometryData.indicesUsage);
		}
		const srcIndices = this._geometryData.indices;
		if (srcIndices) {
			const indexBuffer = this.indexBuffer[0];
			indexBuffer.writeData(srcIndices, this._geometryData.indexCount);
			this._geometryData.indices = null;
		}
	}
	prepareRenderState(renderStyle) {
		if (renderStyle === RENDERSTYLE_WIREFRAME) {
			this.generateWireframe();
		} else if (renderStyle === RENDERSTYLE_POINTS) {
			this.primitive[RENDERSTYLE_POINTS] = {
				type: PRIMITIVE_POINTS,
				base: 0,
				count: this.vertexBuffer ? this.vertexBuffer.numVertices : 0,
				indexed: false
			};
		}
	}
	updateRenderStates() {
		if (this.primitive[RENDERSTYLE_POINTS]) {
			this.prepareRenderState(RENDERSTYLE_POINTS);
		}
		if (this.primitive[RENDERSTYLE_WIREFRAME]) {
			this.prepareRenderState(RENDERSTYLE_WIREFRAME);
		}
	}
	generateWireframe() {
		this._destroyIndexBuffer(RENDERSTYLE_WIREFRAME);
		const numVertices = this.vertexBuffer.numVertices;
		const lines = [];
		let format;
		if (this.indexBuffer.length > 0 && this.indexBuffer[0]) {
			const offsets = [[0, 1], [1, 2], [2, 0]];
			const base = this.primitive[RENDERSTYLE_SOLID].base;
			const count = this.primitive[RENDERSTYLE_SOLID].count;
			const indexBuffer = this.indexBuffer[RENDERSTYLE_SOLID];
			const srcIndices = new typedArrayIndexFormats[indexBuffer.format](indexBuffer.storage);
			const seen = new Set();
			for (let j = base; j < base + count; j += 3) {
				for (let k = 0; k < 3; k++) {
					const i1 = srcIndices[j + offsets[k][0]];
					const i2 = srcIndices[j + offsets[k][1]];
					const hash = i1 > i2 ? i2 * numVertices + i1 : i1 * numVertices + i2;
					if (!seen.has(hash)) {
						seen.add(hash);
						lines.push(i1, i2);
					}
				}
			}
			format = indexBuffer.format;
		} else {
			for (let i = 0; i < numVertices; i += 3) {
				lines.push(i, i + 1, i + 1, i + 2, i + 2, i);
			}
			format = lines.length > 65535 ? INDEXFORMAT_UINT32 : INDEXFORMAT_UINT16;
		}
		const wireBuffer = new IndexBuffer(this.vertexBuffer.device, format, lines.length);
		const dstIndices = new typedArrayIndexFormats[wireBuffer.format](wireBuffer.storage);
		dstIndices.set(lines);
		wireBuffer.unlock();
		this.primitive[RENDERSTYLE_WIREFRAME] = {
			type: PRIMITIVE_LINES,
			base: 0,
			count: lines.length,
			indexed: true
		};
		this.indexBuffer[RENDERSTYLE_WIREFRAME] = wireBuffer;
	}
}

const primitiveUv1Padding = 4.0 / 64;
const primitiveUv1PaddingScale = 1.0 - primitiveUv1Padding * 2;
const shapePrimitives = [];
function calculateNormals(positions, indices) {
	const triangleCount = indices.length / 3;
	const vertexCount = positions.length / 3;
	const p1 = new Vec3();
	const p2 = new Vec3();
	const p3 = new Vec3();
	const p1p2 = new Vec3();
	const p1p3 = new Vec3();
	const faceNormal = new Vec3();
	const normals = [];
	for (let i = 0; i < positions.length; i++) {
		normals[i] = 0;
	}
	for (let i = 0; i < triangleCount; i++) {
		const i1 = indices[i * 3];
		const i2 = indices[i * 3 + 1];
		const i3 = indices[i * 3 + 2];
		p1.set(positions[i1 * 3], positions[i1 * 3 + 1], positions[i1 * 3 + 2]);
		p2.set(positions[i2 * 3], positions[i2 * 3 + 1], positions[i2 * 3 + 2]);
		p3.set(positions[i3 * 3], positions[i3 * 3 + 1], positions[i3 * 3 + 2]);
		p1p2.sub2(p2, p1);
		p1p3.sub2(p3, p1);
		faceNormal.cross(p1p2, p1p3).normalize();
		normals[i1 * 3] += faceNormal.x;
		normals[i1 * 3 + 1] += faceNormal.y;
		normals[i1 * 3 + 2] += faceNormal.z;
		normals[i2 * 3] += faceNormal.x;
		normals[i2 * 3 + 1] += faceNormal.y;
		normals[i2 * 3 + 2] += faceNormal.z;
		normals[i3 * 3] += faceNormal.x;
		normals[i3 * 3 + 1] += faceNormal.y;
		normals[i3 * 3 + 2] += faceNormal.z;
	}
	for (let i = 0; i < vertexCount; i++) {
		const nx = normals[i * 3];
		const ny = normals[i * 3 + 1];
		const nz = normals[i * 3 + 2];
		const invLen = 1 / Math.sqrt(nx * nx + ny * ny + nz * nz);
		normals[i * 3] *= invLen;
		normals[i * 3 + 1] *= invLen;
		normals[i * 3 + 2] *= invLen;
	}
	return normals;
}
function calculateTangents(positions, normals, uvs, indices) {
	const triangleCount = indices.length / 3;
	const vertexCount = positions.length / 3;
	const v1 = new Vec3();
	const v2 = new Vec3();
	const v3 = new Vec3();
	const w1 = new Vec2();
	const w2 = new Vec2();
	const w3 = new Vec2();
	const sdir = new Vec3();
	const tdir = new Vec3();
	const tan1 = new Float32Array(vertexCount * 3);
	const tan2 = new Float32Array(vertexCount * 3);
	const tangents = [];
	for (let i = 0; i < triangleCount; i++) {
		const i1 = indices[i * 3];
		const i2 = indices[i * 3 + 1];
		const i3 = indices[i * 3 + 2];
		v1.set(positions[i1 * 3], positions[i1 * 3 + 1], positions[i1 * 3 + 2]);
		v2.set(positions[i2 * 3], positions[i2 * 3 + 1], positions[i2 * 3 + 2]);
		v3.set(positions[i3 * 3], positions[i3 * 3 + 1], positions[i3 * 3 + 2]);
		w1.set(uvs[i1 * 2], uvs[i1 * 2 + 1]);
		w2.set(uvs[i2 * 2], uvs[i2 * 2 + 1]);
		w3.set(uvs[i3 * 2], uvs[i3 * 2 + 1]);
		const x1 = v2.x - v1.x;
		const x2 = v3.x - v1.x;
		const y1 = v2.y - v1.y;
		const y2 = v3.y - v1.y;
		const z1 = v2.z - v1.z;
		const z2 = v3.z - v1.z;
		const s1 = w2.x - w1.x;
		const s2 = w3.x - w1.x;
		const _t = w2.y - w1.y;
		const _t2 = w3.y - w1.y;
		const area = s1 * _t2 - s2 * _t;
		if (area === 0) {
			sdir.set(0, 1, 0);
			tdir.set(1, 0, 0);
		} else {
			const r = 1 / area;
			sdir.set((_t2 * x1 - _t * x2) * r, (_t2 * y1 - _t * y2) * r, (_t2 * z1 - _t * z2) * r);
			tdir.set((s1 * x2 - s2 * x1) * r, (s1 * y2 - s2 * y1) * r, (s1 * z2 - s2 * z1) * r);
		}
		tan1[i1 * 3 + 0] += sdir.x;
		tan1[i1 * 3 + 1] += sdir.y;
		tan1[i1 * 3 + 2] += sdir.z;
		tan1[i2 * 3 + 0] += sdir.x;
		tan1[i2 * 3 + 1] += sdir.y;
		tan1[i2 * 3 + 2] += sdir.z;
		tan1[i3 * 3 + 0] += sdir.x;
		tan1[i3 * 3 + 1] += sdir.y;
		tan1[i3 * 3 + 2] += sdir.z;
		tan2[i1 * 3 + 0] += tdir.x;
		tan2[i1 * 3 + 1] += tdir.y;
		tan2[i1 * 3 + 2] += tdir.z;
		tan2[i2 * 3 + 0] += tdir.x;
		tan2[i2 * 3 + 1] += tdir.y;
		tan2[i2 * 3 + 2] += tdir.z;
		tan2[i3 * 3 + 0] += tdir.x;
		tan2[i3 * 3 + 1] += tdir.y;
		tan2[i3 * 3 + 2] += tdir.z;
	}
	const t1 = new Vec3();
	const t2 = new Vec3();
	const n = new Vec3();
	const temp = new Vec3();
	for (let i = 0; i < vertexCount; i++) {
		n.set(normals[i * 3], normals[i * 3 + 1], normals[i * 3 + 2]);
		t1.set(tan1[i * 3], tan1[i * 3 + 1], tan1[i * 3 + 2]);
		t2.set(tan2[i * 3], tan2[i * 3 + 1], tan2[i * 3 + 2]);
		const ndott = n.dot(t1);
		temp.copy(n).mulScalar(ndott);
		temp.sub2(t1, temp).normalize();
		tangents[i * 4] = temp.x;
		tangents[i * 4 + 1] = temp.y;
		tangents[i * 4 + 2] = temp.z;
		temp.cross(n, t1);
		tangents[i * 4 + 3] = temp.dot(t2) < 0.0 ? -1.0 : 1.0;
	}
	return tangents;
}
function createMesh$1(device, positions, opts) {
	const mesh = new Mesh(device);
	mesh.setPositions(positions);
	if (opts) {
		if (opts.normals) {
			mesh.setNormals(opts.normals);
		}
		if (opts.tangents) {
			mesh.setVertexStream(SEMANTIC_TANGENT, opts.tangents, 4);
		}
		if (opts.colors) {
			mesh.setColors32(opts.colors);
		}
		if (opts.uvs) {
			mesh.setUvs(0, opts.uvs);
		}
		if (opts.uvs1) {
			mesh.setUvs(1, opts.uvs1);
		}
		if (opts.blendIndices) {
			mesh.setVertexStream(SEMANTIC_BLENDINDICES, opts.blendIndices, 4, opts.blendIndices.length / 4, TYPE_UINT8);
		}
		if (opts.blendWeights) {
			mesh.setVertexStream(SEMANTIC_BLENDWEIGHT, opts.blendWeights, 4);
		}
		if (opts.indices) {
			mesh.setIndices(opts.indices);
		}
	}
	mesh.update();
	return mesh;
}
function createTorus(device, opts = {}) {
	var _opts$tubeRadius, _opts$ringRadius, _opts$sectorAngle, _opts$segments, _opts$sides, _opts$calculateTangen;
	const rc = (_opts$tubeRadius = opts.tubeRadius) != null ? _opts$tubeRadius : 0.2;
	const rt = (_opts$ringRadius = opts.ringRadius) != null ? _opts$ringRadius : 0.3;
	const sectorAngle = ((_opts$sectorAngle = opts.sectorAngle) != null ? _opts$sectorAngle : 360) * math.DEG_TO_RAD;
	const segments = (_opts$segments = opts.segments) != null ? _opts$segments : 30;
	const sides = (_opts$sides = opts.sides) != null ? _opts$sides : 20;
	const calcTangents = (_opts$calculateTangen = opts.calculateTangents) != null ? _opts$calculateTangen : false;
	const positions = [];
	const normals = [];
	const uvs = [];
	const indices = [];
	for (let i = 0; i <= sides; i++) {
		for (let j = 0; j <= segments; j++) {
			const x = Math.cos(sectorAngle * j / segments) * (rt + rc * Math.cos(2 * Math.PI * i / sides));
			const y = Math.sin(2 * Math.PI * i / sides) * rc;
			const z = Math.sin(sectorAngle * j / segments) * (rt + rc * Math.cos(2 * Math.PI * i / sides));
			const nx = Math.cos(sectorAngle * j / segments) * Math.cos(2 * Math.PI * i / sides);
			const ny = Math.sin(2 * Math.PI * i / sides);
			const nz = Math.sin(sectorAngle * j / segments) * Math.cos(2 * Math.PI * i / sides);
			const u = i / sides;
			const v = 1 - j / segments;
			positions.push(x, y, z);
			normals.push(nx, ny, nz);
			uvs.push(u, 1.0 - v);
			if (i < sides && j < segments) {
				const first = i * (segments + 1) + j;
				const second = (i + 1) * (segments + 1) + j;
				const third = i * (segments + 1) + (j + 1);
				const fourth = (i + 1) * (segments + 1) + (j + 1);
				indices.push(first, second, third);
				indices.push(second, fourth, third);
			}
		}
	}
	const options = {
		normals: normals,
		uvs: uvs,
		uvs1: uvs,
		indices: indices
	};
	if (calcTangents) {
		options.tangents = calculateTangents(positions, normals, uvs, indices);
	}
	return createMesh$1(device, positions, options);
}
function _createConeData(baseRadius, peakRadius, height, heightSegments, capSegments, roundedCaps) {
	const pos = new Vec3();
	const bottomToTop = new Vec3();
	const norm = new Vec3();
	const top = new Vec3();
	const bottom = new Vec3();
	const tangent = new Vec3();
	const positions = [];
	const normals = [];
	const uvs = [];
	const uvs1 = [];
	const indices = [];
	let offset;
	if (height > 0) {
		for (let i = 0; i <= heightSegments; i++) {
			for (let j = 0; j <= capSegments; j++) {
				const theta = j / capSegments * 2 * Math.PI - Math.PI;
				const sinTheta = Math.sin(theta);
				const cosTheta = Math.cos(theta);
				bottom.set(sinTheta * baseRadius, -height / 2, cosTheta * baseRadius);
				top.set(sinTheta * peakRadius, height / 2, cosTheta * peakRadius);
				pos.lerp(bottom, top, i / heightSegments);
				bottomToTop.sub2(top, bottom).normalize();
				tangent.set(cosTheta, 0, -sinTheta);
				norm.cross(tangent, bottomToTop).normalize();
				positions.push(pos.x, pos.y, pos.z);
				normals.push(norm.x, norm.y, norm.z);
				let u = j / capSegments;
				let v = i / heightSegments;
				uvs.push(u, 1 - v);
				const _v = v;
				v = u;
				u = _v;
				u = u * primitiveUv1PaddingScale + primitiveUv1Padding;
				v = v * primitiveUv1PaddingScale + primitiveUv1Padding;
				u /= 3;
				uvs1.push(u, 1 - v);
				if (i < heightSegments && j < capSegments) {
					const first = i * (capSegments + 1) + j;
					const second = i * (capSegments + 1) + (j + 1);
					const third = (i + 1) * (capSegments + 1) + j;
					const fourth = (i + 1) * (capSegments + 1) + (j + 1);
					indices.push(first, second, third);
					indices.push(second, fourth, third);
				}
			}
		}
	}
	if (roundedCaps) {
		const latitudeBands = Math.floor(capSegments / 2);
		const longitudeBands = capSegments;
		const capOffset = height / 2;
		for (let lat = 0; lat <= latitudeBands; lat++) {
			const theta = lat * Math.PI * 0.5 / latitudeBands;
			const sinTheta = Math.sin(theta);
			const cosTheta = Math.cos(theta);
			for (let lon = 0; lon <= longitudeBands; lon++) {
				const phi = lon * 2 * Math.PI / longitudeBands - Math.PI / 2;
				const sinPhi = Math.sin(phi);
				const cosPhi = Math.cos(phi);
				const x = cosPhi * sinTheta;
				const y = cosTheta;
				const z = sinPhi * sinTheta;
				let u = 1 - lon / longitudeBands;
				let v = 1 - lat / latitudeBands;
				positions.push(x * peakRadius, y * peakRadius + capOffset, z * peakRadius);
				normals.push(x, y, z);
				uvs.push(u, 1 - v);
				u = u * primitiveUv1PaddingScale + primitiveUv1Padding;
				v = v * primitiveUv1PaddingScale + primitiveUv1Padding;
				u /= 3;
				v /= 3;
				u += 1.0 / 3;
				uvs1.push(u, 1 - v);
			}
		}
		offset = (heightSegments + 1) * (capSegments + 1);
		for (let lat = 0; lat < latitudeBands; ++lat) {
			for (let lon = 0; lon < longitudeBands; ++lon) {
				const first = lat * (longitudeBands + 1) + lon;
				const second = first + longitudeBands + 1;
				indices.push(offset + first + 1, offset + second, offset + first);
				indices.push(offset + first + 1, offset + second + 1, offset + second);
			}
		}
		for (let lat = 0; lat <= latitudeBands; lat++) {
			const theta = Math.PI * 0.5 + lat * Math.PI * 0.5 / latitudeBands;
			const sinTheta = Math.sin(theta);
			const cosTheta = Math.cos(theta);
			for (let lon = 0; lon <= longitudeBands; lon++) {
				const phi = lon * 2 * Math.PI / longitudeBands - Math.PI / 2;
				const sinPhi = Math.sin(phi);
				const cosPhi = Math.cos(phi);
				const x = cosPhi * sinTheta;
				const y = cosTheta;
				const z = sinPhi * sinTheta;
				let u = 1 - lon / longitudeBands;
				let v = 1 - lat / latitudeBands;
				positions.push(x * peakRadius, y * peakRadius - capOffset, z * peakRadius);
				normals.push(x, y, z);
				uvs.push(u, 1 - v);
				u = u * primitiveUv1PaddingScale + primitiveUv1Padding;
				v = v * primitiveUv1PaddingScale + primitiveUv1Padding;
				u /= 3;
				v /= 3;
				u += 2.0 / 3;
				uvs1.push(u, 1 - v);
			}
		}
		offset = (heightSegments + 1) * (capSegments + 1) + (longitudeBands + 1) * (latitudeBands + 1);
		for (let lat = 0; lat < latitudeBands; ++lat) {
			for (let lon = 0; lon < longitudeBands; ++lon) {
				const first = lat * (longitudeBands + 1) + lon;
				const second = first + longitudeBands + 1;
				indices.push(offset + first + 1, offset + second, offset + first);
				indices.push(offset + first + 1, offset + second + 1, offset + second);
			}
		}
	} else {
		offset = (heightSegments + 1) * (capSegments + 1);
		if (baseRadius > 0) {
			for (let i = 0; i < capSegments; i++) {
				const theta = i / capSegments * 2 * Math.PI;
				const x = Math.sin(theta);
				const y = -height / 2;
				const z = Math.cos(theta);
				let u = 1 - (x + 1) / 2;
				let v = (z + 1) / 2;
				positions.push(x * baseRadius, y, z * baseRadius);
				normals.push(0, -1, 0);
				uvs.push(u, 1 - v);
				u = u * primitiveUv1PaddingScale + primitiveUv1Padding;
				v = v * primitiveUv1PaddingScale + primitiveUv1Padding;
				u /= 3;
				v /= 3;
				u += 1 / 3;
				uvs1.push(u, 1 - v);
				if (i > 1) {
					indices.push(offset, offset + i, offset + i - 1);
				}
			}
		}
		offset += capSegments;
		if (peakRadius > 0) {
			for (let i = 0; i < capSegments; i++) {
				const theta = i / capSegments * 2 * Math.PI;
				const x = Math.sin(theta);
				const y = height / 2;
				const z = Math.cos(theta);
				let u = 1 - (x + 1) / 2;
				let v = (z + 1) / 2;
				positions.push(x * peakRadius, y, z * peakRadius);
				normals.push(0, 1, 0);
				uvs.push(u, 1 - v);
				u = u * primitiveUv1PaddingScale + primitiveUv1Padding;
				v = v * primitiveUv1PaddingScale + primitiveUv1Padding;
				u /= 3;
				v /= 3;
				u += 2 / 3;
				uvs1.push(u, 1 - v);
				if (i > 1) {
					indices.push(offset, offset + i - 1, offset + i);
				}
			}
		}
	}
	return {
		positions: positions,
		normals: normals,
		uvs: uvs,
		uvs1: uvs1,
		indices: indices
	};
}
function createCylinder(device, opts = {}) {
	var _opts$radius, _opts$height, _opts$heightSegments, _opts$capSegments, _opts$calculateTangen2;
	const radius = (_opts$radius = opts.radius) != null ? _opts$radius : 0.5;
	const height = (_opts$height = opts.height) != null ? _opts$height : 1;
	const heightSegments = (_opts$heightSegments = opts.heightSegments) != null ? _opts$heightSegments : 5;
	const capSegments = (_opts$capSegments = opts.capSegments) != null ? _opts$capSegments : 20;
	const calcTangents = (_opts$calculateTangen2 = opts.calculateTangents) != null ? _opts$calculateTangen2 : false;
	const options = _createConeData(radius, radius, height, heightSegments, capSegments, false);
	if (calcTangents) {
		options.tangents = calculateTangents(options.positions, options.normals, options.uvs, options.indices);
	}
	return createMesh$1(device, options.positions, options);
}
function createCapsule(device, opts = {}) {
	var _opts$radius2, _opts$height2, _opts$heightSegments2, _opts$sides2, _opts$calculateTangen3;
	const radius = (_opts$radius2 = opts.radius) != null ? _opts$radius2 : 0.3;
	const height = (_opts$height2 = opts.height) != null ? _opts$height2 : 1;
	const heightSegments = (_opts$heightSegments2 = opts.heightSegments) != null ? _opts$heightSegments2 : 1;
	const sides = (_opts$sides2 = opts.sides) != null ? _opts$sides2 : 20;
	const calcTangents = (_opts$calculateTangen3 = opts.calculateTangents) != null ? _opts$calculateTangen3 : false;
	const options = _createConeData(radius, radius, height - 2 * radius, heightSegments, sides, true);
	if (calcTangents) {
		options.tangents = calculateTangents(options.positions, options.normals, options.uvs, options.indices);
	}
	return createMesh$1(device, options.positions, options);
}
function createCone(device, opts = {}) {
	var _opts$baseRadius, _opts$peakRadius, _opts$height3, _opts$heightSegments3, _opts$capSegments2, _opts$calculateTangen4;
	const baseRadius = (_opts$baseRadius = opts.baseRadius) != null ? _opts$baseRadius : 0.5;
	const peakRadius = (_opts$peakRadius = opts.peakRadius) != null ? _opts$peakRadius : 0;
	const height = (_opts$height3 = opts.height) != null ? _opts$height3 : 1;
	const heightSegments = (_opts$heightSegments3 = opts.heightSegments) != null ? _opts$heightSegments3 : 5;
	const capSegments = (_opts$capSegments2 = opts.capSegments) != null ? _opts$capSegments2 : 18;
	const calcTangents = (_opts$calculateTangen4 = opts.calculateTangents) != null ? _opts$calculateTangen4 : false;
	const options = _createConeData(baseRadius, peakRadius, height, heightSegments, capSegments, false);
	if (calcTangents) {
		options.tangents = calculateTangents(options.positions, options.normals, options.uvs, options.indices);
	}
	return createMesh$1(device, options.positions, options);
}
function createSphere(device, opts = {}) {
	var _opts$radius3, _opts$latitudeBands, _opts$longitudeBands, _opts$calculateTangen5;
	const radius = (_opts$radius3 = opts.radius) != null ? _opts$radius3 : 0.5;
	const latitudeBands = (_opts$latitudeBands = opts.latitudeBands) != null ? _opts$latitudeBands : 16;
	const longitudeBands = (_opts$longitudeBands = opts.longitudeBands) != null ? _opts$longitudeBands : 16;
	const calcTangents = (_opts$calculateTangen5 = opts.calculateTangents) != null ? _opts$calculateTangen5 : false;
	const positions = [];
	const normals = [];
	const uvs = [];
	const indices = [];
	for (let lat = 0; lat <= latitudeBands; lat++) {
		const theta = lat * Math.PI / latitudeBands;
		const sinTheta = Math.sin(theta);
		const cosTheta = Math.cos(theta);
		for (let lon = 0; lon <= longitudeBands; lon++) {
			const phi = lon * 2 * Math.PI / longitudeBands - Math.PI / 2;
			const sinPhi = Math.sin(phi);
			const cosPhi = Math.cos(phi);
			const x = cosPhi * sinTheta;
			const y = cosTheta;
			const z = sinPhi * sinTheta;
			const u = 1 - lon / longitudeBands;
			const v = 1 - lat / latitudeBands;
			positions.push(x * radius, y * radius, z * radius);
			normals.push(x, y, z);
			uvs.push(u, 1 - v);
		}
	}
	for (let lat = 0; lat < latitudeBands; ++lat) {
		for (let lon = 0; lon < longitudeBands; ++lon) {
			const first = lat * (longitudeBands + 1) + lon;
			const second = first + longitudeBands + 1;
			indices.push(first + 1, second, first);
			indices.push(first + 1, second + 1, second);
		}
	}
	const options = {
		normals: normals,
		uvs: uvs,
		uvs1: uvs,
		indices: indices
	};
	if (calcTangents) {
		options.tangents = calculateTangents(positions, normals, uvs, indices);
	}
	return createMesh$1(device, positions, options);
}
function createPlane(device, opts = {}) {
	var _opts$halfExtents, _opts$widthSegments, _opts$lengthSegments, _opts$calculateTangen6;
	const he = (_opts$halfExtents = opts.halfExtents) != null ? _opts$halfExtents : new Vec2(0.5, 0.5);
	const ws = (_opts$widthSegments = opts.widthSegments) != null ? _opts$widthSegments : 5;
	const ls = (_opts$lengthSegments = opts.lengthSegments) != null ? _opts$lengthSegments : 5;
	const calcTangents = (_opts$calculateTangen6 = opts.calculateTangents) != null ? _opts$calculateTangen6 : false;
	const positions = [];
	const normals = [];
	const uvs = [];
	const indices = [];
	let vcounter = 0;
	for (let i = 0; i <= ws; i++) {
		for (let j = 0; j <= ls; j++) {
			const x = -he.x + 2 * he.x * i / ws;
			const y = 0.0;
			const z = -(-he.y + 2 * he.y * j / ls);
			const u = i / ws;
			const v = j / ls;
			positions.push(x, y, z);
			normals.push(0, 1, 0);
			uvs.push(u, 1 - v);
			if (i < ws && j < ls) {
				indices.push(vcounter + ls + 1, vcounter + 1, vcounter);
				indices.push(vcounter + ls + 1, vcounter + ls + 2, vcounter + 1);
			}
			vcounter++;
		}
	}
	const options = {
		normals: normals,
		uvs: uvs,
		uvs1: uvs,
		indices: indices
	};
	if (calcTangents) {
		options.tangents = calculateTangents(positions, normals, uvs, indices);
	}
	return createMesh$1(device, positions, options);
}
function createBox(device, opts = {}) {
	var _opts$halfExtents2, _opts$widthSegments2, _opts$lengthSegments2, _opts$heightSegments4, _opts$calculateTangen7, _opts$yOffset;
	const he = (_opts$halfExtents2 = opts.halfExtents) != null ? _opts$halfExtents2 : new Vec3(0.5, 0.5, 0.5);
	const ws = (_opts$widthSegments2 = opts.widthSegments) != null ? _opts$widthSegments2 : 1;
	const ls = (_opts$lengthSegments2 = opts.lengthSegments) != null ? _opts$lengthSegments2 : 1;
	const hs = (_opts$heightSegments4 = opts.heightSegments) != null ? _opts$heightSegments4 : 1;
	const calcTangents = (_opts$calculateTangen7 = opts.calculateTangents) != null ? _opts$calculateTangen7 : false;
	const yOffset = (_opts$yOffset = opts.yOffset) != null ? _opts$yOffset : 0;
	const minY = -he.y + yOffset;
	const maxY = he.y + yOffset;
	const corners = [new Vec3(-he.x, minY, he.z), new Vec3(he.x, minY, he.z), new Vec3(he.x, maxY, he.z), new Vec3(-he.x, maxY, he.z), new Vec3(he.x, minY, -he.z), new Vec3(-he.x, minY, -he.z), new Vec3(-he.x, maxY, -he.z), new Vec3(he.x, maxY, -he.z)];
	const faceAxes = [[0, 1, 3], [4, 5, 7], [3, 2, 6], [1, 0, 4], [1, 4, 2], [5, 0, 6]];
	const faceNormals = [[0, 0, 1], [0, 0, -1], [0, 1, 0], [0, -1, 0], [1, 0, 0], [-1, 0, 0]];
	const sides = {
		FRONT: 0,
		BACK: 1,
		TOP: 2,
		BOTTOM: 3,
		RIGHT: 4,
		LEFT: 5
	};
	const positions = [];
	const normals = [];
	const uvs = [];
	const uvs1 = [];
	const indices = [];
	let vcounter = 0;
	const generateFace = (side, uSegments, vSegments) => {
		const temp1 = new Vec3();
		const temp2 = new Vec3();
		const temp3 = new Vec3();
		const r = new Vec3();
		for (let i = 0; i <= uSegments; i++) {
			for (let j = 0; j <= vSegments; j++) {
				temp1.lerp(corners[faceAxes[side][0]], corners[faceAxes[side][1]], i / uSegments);
				temp2.lerp(corners[faceAxes[side][0]], corners[faceAxes[side][2]], j / vSegments);
				temp3.sub2(temp2, corners[faceAxes[side][0]]);
				r.add2(temp1, temp3);
				let u = i / uSegments;
				let v = j / vSegments;
				positions.push(r.x, r.y, r.z);
				normals.push(faceNormals[side][0], faceNormals[side][1], faceNormals[side][2]);
				uvs.push(u, 1 - v);
				u = u * primitiveUv1PaddingScale + primitiveUv1Padding;
				v = v * primitiveUv1PaddingScale + primitiveUv1Padding;
				u /= 3;
				v /= 3;
				u += side % 3 / 3;
				v += Math.floor(side / 3) / 3;
				uvs1.push(u, 1 - v);
				if (i < uSegments && j < vSegments) {
					indices.push(vcounter + vSegments + 1, vcounter + 1, vcounter);
					indices.push(vcounter + vSegments + 1, vcounter + vSegments + 2, vcounter + 1);
				}
				vcounter++;
			}
		}
	};
	generateFace(sides.FRONT, ws, hs);
	generateFace(sides.BACK, ws, hs);
	generateFace(sides.TOP, ws, ls);
	generateFace(sides.BOTTOM, ws, ls);
	generateFace(sides.RIGHT, ls, hs);
	generateFace(sides.LEFT, ls, hs);
	const options = {
		normals: normals,
		uvs: uvs,
		uvs1: uvs1,
		indices: indices
	};
	if (calcTangents) {
		options.tangents = calculateTangents(positions, normals, uvs, indices);
	}
	return createMesh$1(device, positions, options);
}
function getShapePrimitive(device, type) {
	let primData = null;
	for (let i = 0; i < shapePrimitives.length; i++) {
		if (shapePrimitives[i].type === type && shapePrimitives[i].device === device) {
			primData = shapePrimitives[i].primData;
		}
	}
	if (!primData) {
		let mesh, area;
		switch (type) {
			case 'box':
				mesh = createBox(device);
				area = {
					x: 2,
					y: 2,
					z: 2,
					uv: 2.0 / 3
				};
				break;
			case 'capsule':
				mesh = createCapsule(device, {
					radius: 0.5,
					height: 2
				});
				area = {
					x: Math.PI * 2,
					y: Math.PI,
					z: Math.PI * 2,
					uv: 1.0 / 3 + 1.0 / 3 / 3 * 2
				};
				break;
			case 'cone':
				mesh = createCone(device, {
					baseRadius: 0.5,
					peakRadius: 0,
					height: 1
				});
				area = {
					x: 2.54,
					y: 2.54,
					z: 2.54,
					uv: 1.0 / 3 + 1.0 / 3 / 3
				};
				break;
			case 'cylinder':
				mesh = createCylinder(device, {
					radius: 0.5,
					height: 1
				});
				area = {
					x: Math.PI,
					y: 0.79 * 2,
					z: Math.PI,
					uv: 1.0 / 3 + 1.0 / 3 / 3 * 2
				};
				break;
			case 'plane':
				mesh = createPlane(device, {
					halfExtents: new Vec2(0.5, 0.5),
					widthSegments: 1,
					lengthSegments: 1
				});
				area = {
					x: 0,
					y: 1,
					z: 0,
					uv: 1
				};
				break;
			case 'sphere':
				mesh = createSphere(device, {
					radius: 0.5
				});
				area = {
					x: Math.PI,
					y: Math.PI,
					z: Math.PI,
					uv: 1
				};
				break;
			case 'torus':
				mesh = createTorus(device, {
					tubeRadius: 0.2,
					ringRadius: 0.3
				});
				area = {
					x: Math.PI * 0.5 * 0.5 - Math.PI * 0.1 * 0.1,
					y: 0.4,
					z: 0.4,
					uv: 1
				};
				break;
			default:
				throw new Error('Invalid primitive type: ' + type);
		}
		mesh.incRefCount();
		primData = {
			mesh: mesh,
			area: area
		};
		shapePrimitives.push({
			type: type,
			device: device,
			primData: primData
		});
	}
	return primData;
}

var alphaTestPS = `
uniform float alpha_ref;
void alphaTest(float a) {
	if (a < alpha_ref) discard;
}
`;

var ambientConstantPS = `
void addAmbient(vec3 worldNormal) {
	dDiffuseLight += light_globalAmbient;
}
`;

var ambientEnvPS = `
#ifndef ENV_ATLAS
#define ENV_ATLAS
uniform sampler2D texture_envAtlas;
#endif
void addAmbient(vec3 worldNormal) {
	vec3 dir = normalize(cubeMapRotate(worldNormal) * vec3(-1.0, 1.0, 1.0));
	vec2 uv = mapUv(toSphericalUv(dir), vec4(128.0, 256.0 + 128.0, 64.0, 32.0) / atlasSize);
	vec4 raw = texture2D(texture_envAtlas, uv);
	vec3 linear = $DECODE(raw);
	dDiffuseLight += processEnvironment(linear);
}
`;

var ambientSHPS = `
uniform vec3 ambientSH[9];
void addAmbient(vec3 worldNormal) {
	vec3 n = cubeMapRotate(worldNormal);
	vec3 color =
		ambientSH[0] +
		ambientSH[1] * n.x +
		ambientSH[2] * n.y +
		ambientSH[3] * n.z +
		ambientSH[4] * n.x * n.z +
		ambientSH[5] * n.z * n.y +
		ambientSH[6] * n.y * n.x +
		ambientSH[7] * (3.0 * n.z * n.z - 1.0) +
		ambientSH[8] * (n.x * n.x - n.y * n.y);
	dDiffuseLight += processEnvironment(max(color, vec3(0.0)));
}
`;

var aoPS = `
void getAO() {
	dAo = 1.0;
	#ifdef MAPTEXTURE
	float aoBase = texture2DBias($SAMPLER, $UV, textureBias).$CH;
	dAo *= addAoDetail(aoBase);
	#endif
	#ifdef MAPVERTEX
	dAo *= saturate(vVertexColor.$VC);
	#endif
}
`;

var aoDetailMapPS = `
float addAoDetail(float ao) {
#ifdef MAPTEXTURE
	float aoDetail = texture2DBias($SAMPLER, $UV, textureBias).$CH;
	return detailMode_$DETAILMODE(vec3(ao), vec3(aoDetail)).r;
#else
	return ao;
#endif
}
`;

var aoDiffuseOccPS = `
void occludeDiffuse(float ao) {
	dDiffuseLight *= ao;
}
`;

var aoSpecOccPS = `
uniform float material_occludeSpecularIntensity;
void occludeSpecular(float gloss, float ao, vec3 worldNormal, vec3 viewDir) {
	float specPow = exp2(gloss * 11.0);
	float specOcc = saturate(pow(dot(worldNormal, viewDir) + ao, 0.01*specPow) - 1.0 + ao);
	specOcc = mix(1.0, specOcc, material_occludeSpecularIntensity);
	dSpecularLight *= specOcc;
	dReflection *= specOcc;
	
#ifdef LIT_SHEEN
	sSpecularLight *= specOcc;
	sReflection *= specOcc;
#endif
}
`;

var aoSpecOccConstPS = `
void occludeSpecular(float gloss, float ao, vec3 worldNormal, vec3 viewDir) {
	float specPow = exp2(gloss * 11.0);
	float specOcc = saturate(pow(dot(worldNormal, viewDir) + ao, 0.01*specPow) - 1.0 + ao);
	dSpecularLight *= specOcc;
	dReflection *= specOcc;
	
#ifdef LIT_SHEEN
	sSpecularLight *= specOcc;
	sReflection *= specOcc;
#endif
}
`;

var aoSpecOccConstSimplePS = `
void occludeSpecular(float gloss, float ao, vec3 worldNormal, vec3 viewDir) {
	dSpecularLight *= ao;
	dReflection *= ao;
#ifdef LIT_SHEEN
	sSpecularLight *= ao;
	sReflection *= ao;
#endif
}
`;

var aoSpecOccSimplePS = `
uniform float material_occludeSpecularIntensity;
void occludeSpecular(float gloss, float ao, vec3 worldNormal, vec3 viewDir) {
	float specOcc = mix(1.0, ao, material_occludeSpecularIntensity);
	dSpecularLight *= specOcc;
	dReflection *= specOcc;
#ifdef LIT_SHEEN
	sSpecularLight *= specOcc;
	sReflection *= specOcc;
#endif
}
`;

var basePS = `
uniform vec3 view_position;
uniform vec3 light_globalAmbient;
float square(float x) {
	return x*x;
}
float saturate(float x) {
	return clamp(x, 0.0, 1.0);
}
vec3 saturate(vec3 x) {
	return clamp(x, vec3(0.0), vec3(1.0));
}
`;

var baseVS = `
attribute vec3 vertex_position;
attribute vec3 vertex_normal;
attribute vec4 vertex_tangent;
attribute vec2 vertex_texCoord0;
attribute vec2 vertex_texCoord1;
attribute vec4 vertex_color;
uniform mat4 matrix_viewProjection;
uniform mat4 matrix_model;
uniform mat3 matrix_normal;
vec3 dPositionW;
mat4 dModelMatrix;
mat3 dNormalMatrix;
`;

var baseNineSlicedPS = `
#define NINESLICED
varying vec2 vMask;
varying vec2 vTiledUv;
uniform mediump vec4 innerOffset;
uniform mediump vec2 outerScale;
uniform mediump vec4 atlasRect;
vec2 nineSlicedUv;
`;

var baseNineSlicedVS = `
#define NINESLICED
varying vec2 vMask;
varying vec2 vTiledUv;
uniform mediump vec4 innerOffset;
uniform mediump vec2 outerScale;
uniform mediump vec4 atlasRect;
`;

var baseNineSlicedTiledPS = `
#define NINESLICED
#define NINESLICETILED
varying vec2 vMask;
varying vec2 vTiledUv;
uniform mediump vec4 innerOffset;
uniform mediump vec2 outerScale;
uniform mediump vec4 atlasRect;
vec2 nineSlicedUv;
`;

var bayerPS = `
float bayer2(vec2 p) {
	return mod(2.0 * p.y + p.x + 1.0, 4.0);
}
float bayer4(vec2 p) {
	vec2 p1 = mod(p, 2.0);
	vec2 p2 = floor(0.5 * mod(p, 4.0));
	return 4.0 * bayer2(p1) + bayer2(p2);
}
float bayer8(vec2 p) {
	vec2 p1 = mod(p, 2.0);
	vec2 p2 = floor(0.5 * mod(p, 4.0));
	vec2 p4 = floor(0.25 * mod(p, 8.0));
	return 4.0 * (4.0 * bayer2(p1) + bayer2(p2)) + bayer2(p4);
}
`;

var biasConstPS = `
#define SHADOWBIAS
#define SHADOW_SAMPLE_Z_BIAS
float getShadowBias(float resolution, float maxBias) {
	return maxBias;
}
`;

var blurVSMPS = `
varying vec2 vUv0;
uniform sampler2D source;
uniform vec2 pixelOffset;
#ifdef GAUSS
uniform float weight[SAMPLES];
#endif
#ifdef PACKED
float decodeFloatRG(vec2 rg) {
	return rg.y*(1.0/255.0) + rg.x;
}
vec2 encodeFloatRG( float v ) {
	vec2 enc = vec2(1.0, 255.0) * v;
	enc = fract(enc);
	enc -= enc.yy * vec2(1.0/255.0, 1.0/255.0);
	return enc;
}
#endif
void main(void) {
	vec3 moments = vec3(0.0);
	vec2 uv = vUv0 - pixelOffset * (float(SAMPLES) * 0.5);
	for (int i=0; i<SAMPLES; i++) {
		vec4 c = texture2D(source, uv + pixelOffset * float(i));
		#ifdef PACKED
		c.xy = vec2(decodeFloatRG(c.xy), decodeFloatRG(c.zw));
		#endif
		#ifdef GAUSS
		moments += c.xyz * weight[i];
		#else
		moments += c.xyz;
		#endif
	}
	#ifndef GAUSS
	moments /= float(SAMPLES);
	#endif
	#ifdef PACKED
	gl_FragColor = vec4(encodeFloatRG(moments.x), encodeFloatRG(moments.y));
	#else
	gl_FragColor = vec4(moments.x, moments.y, moments.z, 1.0);
	#endif
}
`;

var clearCoatPS = `
#ifdef MAPFLOAT
uniform float material_clearCoat;
#endif
void getClearCoat() {
	ccSpecularity = 1.0;
	#ifdef MAPFLOAT
	ccSpecularity *= material_clearCoat;
	#endif
	#ifdef MAPTEXTURE
	ccSpecularity *= texture2DBias($SAMPLER, $UV, textureBias).$CH;
	#endif
	#ifdef MAPVERTEX
	ccSpecularity *= saturate(vVertexColor.$VC);
	#endif
}
`;

var clearCoatGlossPS = `
#ifdef MAPFLOAT
uniform float material_clearCoatGloss;
#endif
void getClearCoatGlossiness() {
	ccGlossiness = 1.0;
	#ifdef MAPFLOAT
	ccGlossiness *= material_clearCoatGloss;
	#endif
	#ifdef MAPTEXTURE
	ccGlossiness *= texture2DBias($SAMPLER, $UV, textureBias).$CH;
	#endif
	#ifdef MAPVERTEX
	ccGlossiness *= saturate(vVertexColor.$VC);
	#endif
	#ifdef MAPINVERT
	ccGlossiness = 1.0 - ccGlossiness;
	#endif
	ccGlossiness += 0.0000001;
}
`;

var clearCoatNormalPS = `
#ifdef MAPTEXTURE
uniform float material_clearCoatBumpiness;
#endif
void getClearCoatNormal() {
#ifdef MAPTEXTURE
	vec3 normalMap = unpackNormal(texture2DBias($SAMPLER, $UV, textureBias));
	normalMap = mix(vec3(0.0, 0.0, 1.0), normalMap, material_clearCoatBumpiness);
	ccNormalW = normalize(dTBN * normalMap);
#else
	ccNormalW = dVertexNormalW;
#endif
}
`;

var clusteredLightUtilsPS = `
vec2 getCubemapFaceCoordinates(const vec3 dir, out float faceIndex, out vec2 tileOffset)
{
	vec3 vAbs = abs(dir);
	float ma;
	vec2 uv;
	if (vAbs.z >= vAbs.x && vAbs.z >= vAbs.y) {
		faceIndex = dir.z < 0.0 ? 5.0 : 4.0;
		ma = 0.5 / vAbs.z;
		uv = vec2(dir.z < 0.0 ? -dir.x : dir.x, -dir.y);
		tileOffset.x = 2.0;
		tileOffset.y = dir.z < 0.0 ? 1.0 : 0.0;
	} else if(vAbs.y >= vAbs.x) {
		faceIndex = dir.y < 0.0 ? 3.0 : 2.0;
		ma = 0.5 / vAbs.y;
		uv = vec2(dir.x, dir.y < 0.0 ? -dir.z : dir.z);
		tileOffset.x = 1.0;
		tileOffset.y = dir.y < 0.0 ? 1.0 : 0.0;
	} else {
		faceIndex = dir.x < 0.0 ? 1.0 : 0.0;
		ma = 0.5 / vAbs.x;
		uv = vec2(dir.x < 0.0 ? dir.z : -dir.z, -dir.y);
		tileOffset.x = 0.0;
		tileOffset.y = dir.x < 0.0 ? 1.0 : 0.0;
	}
	return uv * ma + 0.5;
}
vec2 getCubemapAtlasCoordinates(const vec3 omniAtlasViewport, float shadowEdgePixels, float shadowTextureResolution, const vec3 dir) {
	float faceIndex;
	vec2 tileOffset;
	vec2 uv = getCubemapFaceCoordinates(dir, faceIndex, tileOffset);
	float atlasFaceSize = omniAtlasViewport.z;
	float tileSize = shadowTextureResolution * atlasFaceSize;
	float offset = shadowEdgePixels / tileSize;
	uv = uv * vec2(1.0 - offset * 2.0) + vec2(offset * 1.0);
	uv *= atlasFaceSize;
	uv += tileOffset * atlasFaceSize;
	uv += omniAtlasViewport.xy;
	return uv;
}
`;

var clusteredLightCookiesPS = `
vec3 _getCookieClustered(TEXTURE_ACCEPT(tex), vec2 uv, float intensity, bool isRgb, vec4 cookieChannel) {
	vec4 pixel = mix(vec4(1.0), texture2DLodEXT(tex, uv, 0.0), intensity);
	return isRgb == true ? pixel.rgb : vec3(dot(pixel, cookieChannel));
}
vec3 getCookie2DClustered(TEXTURE_ACCEPT(tex), mat4 transform, vec3 worldPosition, float intensity, bool isRgb, vec4 cookieChannel) {
	vec4 projPos = transform * vec4(worldPosition, 1.0);
	return _getCookieClustered(TEXTURE_PASS(tex), projPos.xy / projPos.w, intensity, isRgb, cookieChannel);
}
vec3 getCookieCubeClustered(TEXTURE_ACCEPT(tex), vec3 dir, float intensity, bool isRgb, vec4 cookieChannel, float shadowTextureResolution, float shadowEdgePixels, vec3 omniAtlasViewport) {
	vec2 uv = getCubemapAtlasCoordinates(omniAtlasViewport, shadowEdgePixels, shadowTextureResolution, dir);
	return _getCookieClustered(TEXTURE_PASS(tex), uv, intensity, isRgb, cookieChannel);
}
`;

var clusteredLightShadowsPS = `
void _getShadowCoordPerspZbuffer(mat4 shadowMatrix, vec4 shadowParams, vec3 wPos) {
	vec4 projPos = shadowMatrix * vec4(wPos, 1.0);
	projPos.xyz /= projPos.w;
	dShadowCoord = projPos.xyz;
}
void getShadowCoordPerspZbufferNormalOffset(mat4 shadowMatrix, vec4 shadowParams, vec3 normal) {
	vec3 wPos = vPositionW + normal * shadowParams.y;
	_getShadowCoordPerspZbuffer(shadowMatrix, shadowParams, wPos);
}
vec3 normalOffsetPointShadow(vec4 shadowParams, vec3 lightPos, inout vec3 lightDir, vec3 lightDirNorm, vec3 normal) {
	float distScale = length(lightDir);
	vec3 wPos = vPositionW + normal * shadowParams.y * clamp(1.0 - dot(normal, -lightDirNorm), 0.0, 1.0) * distScale;
	vec3 dir = wPos - lightPos;
	return dir;
}
#ifdef GL2
	#if defined(CLUSTER_SHADOW_TYPE_PCF1)
	float getShadowOmniClusteredPCF1(SHADOWMAP_ACCEPT(shadowMap), vec4 shadowParams, vec3 omniAtlasViewport, float shadowEdgePixels, vec3 lightDir) {
		float shadowTextureResolution = shadowParams.x;
		vec2 uv = getCubemapAtlasCoordinates(omniAtlasViewport, shadowEdgePixels, shadowTextureResolution, lightDir);
		float shadowZ = length(lightDir) * shadowParams.w + shadowParams.z;
		return textureShadow(shadowMap, vec3(uv, shadowZ));
	}
	#endif
	#if defined(CLUSTER_SHADOW_TYPE_PCF3)
	float getShadowOmniClusteredPCF3(SHADOWMAP_ACCEPT(shadowMap), vec4 shadowParams, vec3 omniAtlasViewport, float shadowEdgePixels, vec3 lightDir) {
		float shadowTextureResolution = shadowParams.x;
		vec2 uv = getCubemapAtlasCoordinates(omniAtlasViewport, shadowEdgePixels, shadowTextureResolution, lightDir);
		float shadowZ = length(lightDir) * shadowParams.w + shadowParams.z;
		vec3 shadowCoord = vec3(uv, shadowZ);
		return getShadowPCF3x3(SHADOWMAP_PASS(shadowMap), shadowCoord, shadowParams);
	}
	#endif
	#if defined(CLUSTER_SHADOW_TYPE_PCF5)
	float getShadowOmniClusteredPCF5(SHADOWMAP_ACCEPT(shadowMap), vec4 shadowParams, vec3 omniAtlasViewport, float shadowEdgePixels, vec3 lightDir) {
		float shadowTextureResolution = shadowParams.x;
		vec2 uv = getCubemapAtlasCoordinates(omniAtlasViewport, shadowEdgePixels, shadowTextureResolution, lightDir);
		float shadowZ = length(lightDir) * shadowParams.w + shadowParams.z;
		vec3 shadowCoord = vec3(uv, shadowZ);
		return getShadowPCF5x5(SHADOWMAP_PASS(shadowMap), shadowCoord, shadowParams);
	}
	#endif
#else
	#if defined(CLUSTER_SHADOW_TYPE_PCF1)
	float getShadowOmniClusteredPCF1(sampler2D shadowMap, vec4 shadowParams, vec3 omniAtlasViewport, float shadowEdgePixels, vec3 lightDir) {
		float shadowTextureResolution = shadowParams.x;
		vec2 uv = getCubemapAtlasCoordinates(omniAtlasViewport, shadowEdgePixels, shadowTextureResolution, lightDir);
		float depth = unpackFloat(textureShadow(shadowMap, uv));
		float shadowZ = length(lightDir) * shadowParams.w + shadowParams.z;
		return depth > shadowZ ? 1.0 : 0.0;
	}
	#endif
	#if defined(CLUSTER_SHADOW_TYPE_PCF3)
	float getShadowOmniClusteredPCF3(sampler2D shadowMap, vec4 shadowParams, vec3 omniAtlasViewport, float shadowEdgePixels, vec3 lightDir) {
		float shadowTextureResolution = shadowParams.x;
		vec2 uv = getCubemapAtlasCoordinates(omniAtlasViewport, shadowEdgePixels, shadowTextureResolution, lightDir);
		float shadowZ = length(lightDir) * shadowParams.w + shadowParams.z;
		vec3 shadowCoord = vec3(uv, shadowZ);
		return getShadowPCF3x3(shadowMap, shadowCoord, shadowParams);
	}
	#endif
	#if defined(CLUSTER_SHADOW_TYPE_PCF5)
	float getShadowOmniClusteredPCF5(sampler2D shadowMap, vec4 shadowParams, vec3 omniAtlasViewport, float shadowEdgePixels, vec3 lightDir) {
		float shadowTextureResolution = shadowParams.x;
		vec2 uv = getCubemapAtlasCoordinates(omniAtlasViewport, shadowEdgePixels, shadowTextureResolution, lightDir);
		float shadowZ = length(lightDir) * shadowParams.w + shadowParams.z;
		vec3 shadowCoord = vec3(uv, shadowZ);
		return getShadowPCF3x3(shadowMap, shadowCoord, shadowParams);
	}
	#endif
#endif
#ifdef GL2
	#if defined(CLUSTER_SHADOW_TYPE_PCF1)
	float getShadowSpotClusteredPCF1(SHADOWMAP_ACCEPT(shadowMap), vec3 shadowCoord, vec4 shadowParams) {
		return textureShadow(shadowMap, shadowCoord);
	}
	#endif
	#if defined(CLUSTER_SHADOW_TYPE_PCF3)
	float getShadowSpotClusteredPCF3(SHADOWMAP_ACCEPT(shadowMap), vec3 shadowCoord, vec4 shadowParams) {
		return getShadowSpotPCF3x3(SHADOWMAP_PASS(shadowMap), shadowCoord, shadowParams);
	}
	#endif
	#if defined(CLUSTER_SHADOW_TYPE_PCF5)
	float getShadowSpotClusteredPCF5(SHADOWMAP_ACCEPT(shadowMap), vec3 shadowCoord, vec4 shadowParams) {
		return getShadowPCF5x5(SHADOWMAP_PASS(shadowMap), shadowCoord, shadowParams);
	}
	#endif
#else
	#if defined(CLUSTER_SHADOW_TYPE_PCF1)
	float getShadowSpotClusteredPCF1(sampler2D shadowMap, vec3 shadowCoord, vec4 shadowParams) {
		float depth = unpackFloat(textureShadow(shadowMap, shadowCoord.xy));
		return depth > shadowCoord.z ? 1.0 : 0.0;
	}
	#endif
	#if defined(CLUSTER_SHADOW_TYPE_PCF3)
	float getShadowSpotClusteredPCF3(sampler2D shadowMap, vec3 shadowCoord, vec4 shadowParams) {
		return getShadowSpotPCF3x3(shadowMap, shadowCoord, shadowParams);
	}
	#endif
	#if defined(CLUSTER_SHADOW_TYPE_PCF5)
	float getShadowSpotClusteredPCF5(sampler2D shadowMap, vec3 shadowCoord, vec4 shadowParams) {
		return getShadowSpotPCF3x3(shadowMap, shadowCoord, shadowParams);
	}
	#endif
#endif
`;

var clusteredLightPS = `
uniform highp sampler2D clusterWorldTexture;
uniform highp sampler2D lightsTexture8;
uniform highp sampler2D lightsTextureFloat;
#if defined(CLUSTER_COOKIES)
	#define CLUSTER_COOKIES_OR_SHADOWS
#endif
#if defined(CLUSTER_SHADOWS)
	#define CLUSTER_COOKIES_OR_SHADOWS
#endif
#ifdef CLUSTER_SHADOWS
	#ifdef GL2
		uniform sampler2DShadow shadowAtlasTexture;
	#else
		uniform sampler2D shadowAtlasTexture;
	#endif
#endif
#ifdef CLUSTER_COOKIES
	uniform sampler2D cookieAtlasTexture;
#endif
#ifdef GL2
	uniform int clusterMaxCells;
#else
	uniform float clusterMaxCells;
	uniform vec4 lightsTextureInvSize;
#endif
uniform float clusterSkip;
uniform vec3 clusterCellsCountByBoundsSize;
uniform vec3 clusterTextureSize;
uniform vec3 clusterBoundsMin;
uniform vec3 clusterBoundsDelta;
uniform vec3 clusterCellsDot;
uniform vec3 clusterCellsMax;
uniform vec2 clusterCompressionLimit0;
uniform vec2 shadowAtlasParams;
struct ClusterLightData {
	vec3 halfWidth;
	float lightType;
	vec3 halfHeight;
	#ifdef GL2
		int lightIndex;
	#else
		float lightV;
	#endif
	vec3 position;
	float shape;
	vec3 direction;
	float falloffMode;
	vec3 color;
	float shadowIntensity;
	vec3 omniAtlasViewport;
	float range;
	vec4 cookieChannelMask;
	float shadowBias;
	float shadowNormalBias;
	float innerConeAngleCos;
	float outerConeAngleCos;
	float cookie;
	float cookieRgb;
	float cookieIntensity;
	float mask;
};
mat4 lightProjectionMatrix;
#define isClusteredLightCastShadow(light) ( light.shadowIntensity > 0.0 )
#define isClusteredLightCookie(light) (light.cookie > 0.5 )
#define isClusteredLightCookieRgb(light) (light.cookieRgb > 0.5 )
#define isClusteredLightSpot(light) ( light.lightType > 0.5 )
#define isClusteredLightFalloffLinear(light) ( light.falloffMode < 0.5 )
#define isClusteredLightArea(light) ( light.shape > 0.1 )
#define isClusteredLightRect(light) ( light.shape < 0.3 )
#define isClusteredLightDisk(light) ( light.shape < 0.6 )
#ifdef CLUSTER_MESH_DYNAMIC_LIGHTS
	#define acceptLightMask(light) ( light.mask < 0.75)
#else
	#define acceptLightMask(light) ( light.mask > 0.25)
#endif
vec4 decodeClusterLowRange4Vec4(vec4 d0, vec4 d1, vec4 d2, vec4 d3) {
	return vec4(
		bytes2floatRange4(d0, -2.0, 2.0),
		bytes2floatRange4(d1, -2.0, 2.0),
		bytes2floatRange4(d2, -2.0, 2.0),
		bytes2floatRange4(d3, -2.0, 2.0)
	);
}
#ifdef GL2
	vec4 sampleLightsTexture8(const ClusterLightData clusterLightData, int index) {
		return texelFetch(lightsTexture8, ivec2(index, clusterLightData.lightIndex), 0);
	}
	vec4 sampleLightTextureF(const ClusterLightData clusterLightData, int index) {
		return texelFetch(lightsTextureFloat, ivec2(index, clusterLightData.lightIndex), 0);
	}
#else
	vec4 sampleLightsTexture8(const ClusterLightData clusterLightData, float index) {
		return texture2DLodEXT(lightsTexture8, vec2(index * lightsTextureInvSize.z, clusterLightData.lightV), 0.0);
	}
	vec4 sampleLightTextureF(const ClusterLightData clusterLightData, float index) {
		return texture2DLodEXT(lightsTextureFloat, vec2(index * lightsTextureInvSize.x, clusterLightData.lightV), 0.0);
	}
#endif
void decodeClusterLightCore(inout ClusterLightData clusterLightData, float lightIndex) {
	#ifdef GL2
		clusterLightData.lightIndex = int(lightIndex);
	#else
		clusterLightData.lightV = (lightIndex + 0.5) * lightsTextureInvSize.w;
	#endif
	vec4 lightInfo = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_FLAGS);
	clusterLightData.lightType = lightInfo.x;
	clusterLightData.shape = lightInfo.y;
	clusterLightData.falloffMode = lightInfo.z;
	clusterLightData.shadowIntensity = lightInfo.w;
	vec4 colorA = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_COLOR_A);
	vec4 colorB = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_COLOR_B);
	clusterLightData.color = vec3(bytes2float2(colorA.xy), bytes2float2(colorA.zw), bytes2float2(colorB.xy)) * clusterCompressionLimit0.y;
	clusterLightData.cookie = colorB.z;
	clusterLightData.mask = colorB.w;
	#ifdef CLUSTER_TEXTURE_FLOAT
		vec4 lightPosRange = sampleLightTextureF(clusterLightData, CLUSTER_TEXTURE_F_POSITION_RANGE);
		clusterLightData.position = lightPosRange.xyz;
		clusterLightData.range = lightPosRange.w;
		vec4 lightDir_Unused = sampleLightTextureF(clusterLightData, CLUSTER_TEXTURE_F_SPOT_DIRECTION);
		clusterLightData.direction = lightDir_Unused.xyz;
	#else
		vec4 encPosX = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_POSITION_X);
		vec4 encPosY = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_POSITION_Y);
		vec4 encPosZ = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_POSITION_Z);
		clusterLightData.position = vec3(bytes2float4(encPosX), bytes2float4(encPosY), bytes2float4(encPosZ)) * clusterBoundsDelta + clusterBoundsMin;
		vec4 encRange = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_RANGE);
		clusterLightData.range = bytes2float4(encRange) * clusterCompressionLimit0.x;
		vec4 encDirX = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_SPOT_DIRECTION_X);
		vec4 encDirY = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_SPOT_DIRECTION_Y);
		vec4 encDirZ = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_SPOT_DIRECTION_Z);
		clusterLightData.direction = vec3(bytes2float4(encDirX), bytes2float4(encDirY), bytes2float4(encDirZ)) * 2.0 - 1.0;
	#endif
}
void decodeClusterLightSpot(inout ClusterLightData clusterLightData) {
	vec4 coneAngle = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_SPOT_ANGLES);
	clusterLightData.innerConeAngleCos = bytes2float2(coneAngle.xy) * 2.0 - 1.0;
	clusterLightData.outerConeAngleCos = bytes2float2(coneAngle.zw) * 2.0 - 1.0;
}
void decodeClusterLightOmniAtlasViewport(inout ClusterLightData clusterLightData) {
	#ifdef CLUSTER_TEXTURE_FLOAT
		clusterLightData.omniAtlasViewport = sampleLightTextureF(clusterLightData, CLUSTER_TEXTURE_F_PROJ_MAT_0).xyz;
	#else
		vec4 viewportA = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_ATLAS_VIEWPORT_A);
		vec4 viewportB = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_ATLAS_VIEWPORT_B);
		clusterLightData.omniAtlasViewport = vec3(bytes2float2(viewportA.xy), bytes2float2(viewportA.zw), bytes2float2(viewportB.xy));
	#endif
}
void decodeClusterLightAreaData(inout ClusterLightData clusterLightData) {
	#ifdef CLUSTER_TEXTURE_FLOAT
		clusterLightData.halfWidth = sampleLightTextureF(clusterLightData, CLUSTER_TEXTURE_F_AREA_DATA_WIDTH).xyz;
		clusterLightData.halfHeight = sampleLightTextureF(clusterLightData, CLUSTER_TEXTURE_F_AREA_DATA_HEIGHT).xyz;
	#else
		vec4 areaWidthX = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_AREA_DATA_WIDTH_X);
		vec4 areaWidthY = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_AREA_DATA_WIDTH_Y);
		vec4 areaWidthZ = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_AREA_DATA_WIDTH_Z);
		clusterLightData.halfWidth = vec3(mantissaExponent2Float(areaWidthX), mantissaExponent2Float(areaWidthY), mantissaExponent2Float(areaWidthZ));
		vec4 areaHeightX = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_AREA_DATA_HEIGHT_X);
		vec4 areaHeightY = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_AREA_DATA_HEIGHT_Y);
		vec4 areaHeightZ = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_AREA_DATA_HEIGHT_Z);
		clusterLightData.halfHeight = vec3(mantissaExponent2Float(areaHeightX), mantissaExponent2Float(areaHeightY), mantissaExponent2Float(areaHeightZ));
	#endif
}
void decodeClusterLightProjectionMatrixData(inout ClusterLightData clusterLightData) {
	
	#ifdef CLUSTER_TEXTURE_FLOAT
		vec4 m0 = sampleLightTextureF(clusterLightData, CLUSTER_TEXTURE_F_PROJ_MAT_0);
		vec4 m1 = sampleLightTextureF(clusterLightData, CLUSTER_TEXTURE_F_PROJ_MAT_1);
		vec4 m2 = sampleLightTextureF(clusterLightData, CLUSTER_TEXTURE_F_PROJ_MAT_2);
		vec4 m3 = sampleLightTextureF(clusterLightData, CLUSTER_TEXTURE_F_PROJ_MAT_3);
	#else
		vec4 m00 = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_PROJ_MAT_00);
		vec4 m01 = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_PROJ_MAT_01);
		vec4 m02 = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_PROJ_MAT_02);
		vec4 m03 = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_PROJ_MAT_03);
		vec4 m0 = decodeClusterLowRange4Vec4(m00, m01, m02, m03);
		vec4 m10 = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_PROJ_MAT_10);
		vec4 m11 = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_PROJ_MAT_11);
		vec4 m12 = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_PROJ_MAT_12);
		vec4 m13 = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_PROJ_MAT_13);
		vec4 m1 = decodeClusterLowRange4Vec4(m10, m11, m12, m13);
		vec4 m20 = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_PROJ_MAT_20);
		vec4 m21 = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_PROJ_MAT_21);
		vec4 m22 = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_PROJ_MAT_22);
		vec4 m23 = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_PROJ_MAT_23);
		vec4 m2 = decodeClusterLowRange4Vec4(m20, m21, m22, m23);
		vec4 m30 = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_PROJ_MAT_30);
		vec4 m31 = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_PROJ_MAT_31);
		vec4 m32 = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_PROJ_MAT_32);
		vec4 m33 = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_PROJ_MAT_33);
		vec4 m3 = vec4(mantissaExponent2Float(m30), mantissaExponent2Float(m31), mantissaExponent2Float(m32), mantissaExponent2Float(m33));
	#endif
	
	lightProjectionMatrix = mat4(m0, m1, m2, m3);
}
void decodeClusterLightShadowData(inout ClusterLightData clusterLightData) {
	
	vec4 biases = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_SHADOW_BIAS);
	clusterLightData.shadowBias = bytes2floatRange2(biases.xy, -1.0, 20.0),
	clusterLightData.shadowNormalBias = bytes2float2(biases.zw);
}
void decodeClusterLightCookieData(inout ClusterLightData clusterLightData) {
	vec4 cookieA = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_COOKIE_A);
	clusterLightData.cookieIntensity = cookieA.x;
	clusterLightData.cookieRgb = cookieA.y;
	clusterLightData.cookieChannelMask = sampleLightsTexture8(clusterLightData, CLUSTER_TEXTURE_8_COOKIE_B);
}
void evaluateLight(
	ClusterLightData light, 
	vec3 worldNormal, 
	vec3 viewDir, 
	vec3 reflectionDir,
#if defined(LIT_CLEARCOAT)
	vec3 clearcoatReflectionDir,
#endif
	float gloss, 
	vec3 specularity, 
	vec3 geometricNormal, 
	mat3 tbn, 
#if defined(LIT_IRIDESCENCE)
	vec3 iridescenceFresnel,
#endif
	vec3 clearcoat_worldNormal,
	float clearcoat_gloss,
	float sheen_gloss,
	float iridescence_intensity
) {
	vec3 cookieAttenuation = vec3(1.0);
	float diffuseAttenuation = 1.0;
	float falloffAttenuation = 1.0;
	getLightDirPoint(light.position);
	#ifdef CLUSTER_AREALIGHTS
	if (isClusteredLightArea(light)) {
		decodeClusterLightAreaData(light);
		if (isClusteredLightRect(light)) {
			calcRectLightValues(light.position, light.halfWidth, light.halfHeight);
		} else if (isClusteredLightDisk(light)) {
			calcDiskLightValues(light.position, light.halfWidth, light.halfHeight);
		} else {
			calcSphereLightValues(light.position, light.halfWidth, light.halfHeight);
		}
		falloffAttenuation = getFalloffWindow(light.range, dLightDirW);
	} else
	#endif
	{
		if (isClusteredLightFalloffLinear(light))
			falloffAttenuation = getFalloffLinear(light.range, dLightDirW);
		else
			falloffAttenuation = getFalloffInvSquared(light.range, dLightDirW);
	}
	if (falloffAttenuation > 0.00001) {
		#ifdef CLUSTER_AREALIGHTS
		if (isClusteredLightArea(light)) {
			if (isClusteredLightRect(light)) {
				diffuseAttenuation = getRectLightDiffuse(worldNormal, viewDir, dLightDirW, dLightDirNormW) * 16.0;
			} else if (isClusteredLightDisk(light)) {
				diffuseAttenuation = getDiskLightDiffuse(worldNormal, viewDir, dLightDirW, dLightDirNormW) * 16.0;
			} else {
				diffuseAttenuation = getSphereLightDiffuse(worldNormal, viewDir, dLightDirW, dLightDirNormW) * 16.0;
			}
		} else
		#endif
		{
			falloffAttenuation *= getLightDiffuse(worldNormal, viewDir, dLightDirW, dLightDirNormW); 
		}
		if (isClusteredLightSpot(light)) {
			decodeClusterLightSpot(light);
			falloffAttenuation *= getSpotEffect(light.direction, light.innerConeAngleCos, light.outerConeAngleCos, dLightDirNormW);
		}
		#if defined(CLUSTER_COOKIES_OR_SHADOWS)
		if (falloffAttenuation > 0.00001) {
			if (isClusteredLightCastShadow(light) || isClusteredLightCookie(light)) {
				if (isClusteredLightSpot(light)) {
					decodeClusterLightProjectionMatrixData(light);
				} else {
					decodeClusterLightOmniAtlasViewport(light);
				}
				float shadowTextureResolution = shadowAtlasParams.x;
				float shadowEdgePixels = shadowAtlasParams.y;
				#ifdef CLUSTER_COOKIES
				if (isClusteredLightCookie(light)) {
					decodeClusterLightCookieData(light);
					if (isClusteredLightSpot(light)) {
						cookieAttenuation = getCookie2DClustered(TEXTURE_PASS(cookieAtlasTexture), lightProjectionMatrix, vPositionW, light.cookieIntensity, isClusteredLightCookieRgb(light), light.cookieChannelMask);
					} else {
						cookieAttenuation = getCookieCubeClustered(TEXTURE_PASS(cookieAtlasTexture), dLightDirW, light.cookieIntensity, isClusteredLightCookieRgb(light), light.cookieChannelMask, shadowTextureResolution, shadowEdgePixels, light.omniAtlasViewport);
					}
				}
				#endif
				#ifdef CLUSTER_SHADOWS
				if (isClusteredLightCastShadow(light)) {
					decodeClusterLightShadowData(light);
					vec4 shadowParams = vec4(shadowTextureResolution, light.shadowNormalBias, light.shadowBias, 1.0 / light.range);
					if (isClusteredLightSpot(light)) {
						getShadowCoordPerspZbufferNormalOffset(lightProjectionMatrix, shadowParams, geometricNormal);
						
						#if defined(CLUSTER_SHADOW_TYPE_PCF1)
							float shadow = getShadowSpotClusteredPCF1(SHADOWMAP_PASS(shadowAtlasTexture), dShadowCoord, shadowParams);
						#elif defined(CLUSTER_SHADOW_TYPE_PCF3)
							float shadow = getShadowSpotClusteredPCF3(SHADOWMAP_PASS(shadowAtlasTexture), dShadowCoord, shadowParams);
						#elif defined(CLUSTER_SHADOW_TYPE_PCF5)
							float shadow = getShadowSpotClusteredPCF5(SHADOWMAP_PASS(shadowAtlasTexture), dShadowCoord, shadowParams);
						#elif defined(CLUSTER_SHADOW_TYPE_PCSS)
							float shadow = getShadowSpotClusteredPCSS(SHADOWMAP_PASS(shadowAtlasTexture), dShadowCoord, shadowParams);
						#endif
						falloffAttenuation *= mix(1.0, shadow, light.shadowIntensity);
					} else {
						vec3 dir = normalOffsetPointShadow(shadowParams, dLightPosW, dLightDirW, dLightDirNormW, geometricNormal);
						#if defined(CLUSTER_SHADOW_TYPE_PCF1)
							float shadow = getShadowOmniClusteredPCF1(SHADOWMAP_PASS(shadowAtlasTexture), shadowParams, light.omniAtlasViewport, shadowEdgePixels, dir);
						#elif defined(CLUSTER_SHADOW_TYPE_PCF3)
							float shadow = getShadowOmniClusteredPCF3(SHADOWMAP_PASS(shadowAtlasTexture), shadowParams, light.omniAtlasViewport, shadowEdgePixels, dir);
						#elif defined(CLUSTER_SHADOW_TYPE_PCF5)
							float shadow = getShadowOmniClusteredPCF5(SHADOWMAP_PASS(shadowAtlasTexture), shadowParams, light.omniAtlasViewport, shadowEdgePixels, dir);
						#endif
						falloffAttenuation *= mix(1.0, shadow, light.shadowIntensity);
					}
				}
				#endif
			}
		}
		#endif
		#ifdef CLUSTER_AREALIGHTS
		if (isClusteredLightArea(light)) {
			{
				vec3 areaDiffuse = (diffuseAttenuation * falloffAttenuation) * light.color * cookieAttenuation;
				#if defined(LIT_SPECULAR)
					#if defined(LIT_CONSERVE_ENERGY)
						areaDiffuse = mix(areaDiffuse, vec3(0), dLTCSpecFres);
					#endif
				#endif
				dDiffuseLight += areaDiffuse;
			}
			#ifdef LIT_SPECULAR
				float areaLightSpecular;
				if (isClusteredLightRect(light)) {
					areaLightSpecular = getRectLightSpecular(worldNormal, viewDir);
				} else if (isClusteredLightDisk(light)) {
					areaLightSpecular = getDiskLightSpecular(worldNormal, viewDir);
				} else {
					areaLightSpecular = getSphereLightSpecular(worldNormal, viewDir);
				}
				dSpecularLight += dLTCSpecFres * areaLightSpecular * falloffAttenuation * light.color * cookieAttenuation;
				#ifdef LIT_CLEARCOAT
					float areaLightSpecularCC;
					if (isClusteredLightRect(light)) {
						areaLightSpecularCC = getRectLightSpecular(clearcoat_worldNormal, viewDir);
					} else if (isClusteredLightDisk(light)) {
						areaLightSpecularCC = getDiskLightSpecular(clearcoat_worldNormal, viewDir);
					} else {
						areaLightSpecularCC = getSphereLightSpecular(clearcoat_worldNormal, viewDir);
					}
					ccSpecularLight += ccLTCSpecFres * areaLightSpecularCC * falloffAttenuation * light.color  * cookieAttenuation;
				#endif
			#endif
		} else
		#endif
		{
			{
				vec3 punctualDiffuse = falloffAttenuation * light.color * cookieAttenuation;
				#if defined(CLUSTER_AREALIGHTS)
				#if defined(LIT_SPECULAR)
				#if defined(LIT_CONSERVE_ENERGY)
					punctualDiffuse = mix(punctualDiffuse, vec3(0), specularity);
				#endif
				#endif
				#endif
				dDiffuseLight += punctualDiffuse;
			}
	 
			#ifdef LIT_SPECULAR
				vec3 halfDir = normalize(-dLightDirNormW + viewDir);
				
				#ifdef LIT_SPECULAR_FRESNEL
					dSpecularLight += 
						getLightSpecular(halfDir, reflectionDir, worldNormal, viewDir, dLightDirNormW, gloss, tbn) * falloffAttenuation * light.color * cookieAttenuation * 
						getFresnel(
							dot(viewDir, halfDir), 
							gloss, 
							specularity
						#if defined(LIT_IRIDESCENCE)
							, iridescenceFresnel,
							iridescence_intensity
						#endif
							);
				#else
					dSpecularLight += getLightSpecular(halfDir, reflectionDir, worldNormal, viewDir, dLightDirNormW, gloss, tbn) * falloffAttenuation * light.color * cookieAttenuation * specularity;
				#endif
				#ifdef LIT_CLEARCOAT
					#ifdef LIT_SPECULAR_FRESNEL
						ccSpecularLight += getLightSpecular(halfDir, clearcoatReflectionDir, clearcoat_worldNormal, viewDir, dLightDirNormW, clearcoat_gloss, tbn) * falloffAttenuation * light.color * cookieAttenuation * getFresnelCC(dot(viewDir, halfDir));
					#else
						ccSpecularLight += getLightSpecular(halfDir, clearcoatReflectionDir, clearcoat_worldNormal, viewDir, dLightDirNormW, clearcoat_gloss, tbn) * falloffAttenuation * light.color * cookieAttenuation; 
					#endif
				#endif
				#ifdef LIT_SHEEN
					sSpecularLight += getLightSpecularSheen(halfDir, worldNormal, viewDir, dLightDirNormW, sheen_gloss) * falloffAttenuation * light.color * cookieAttenuation;
				#endif
			#endif
		}
	}
	dAtten = falloffAttenuation;
	dAttenD = diffuseAttenuation;
	dAtten3 = cookieAttenuation;
}
void evaluateClusterLight(
	float lightIndex, 
	vec3 worldNormal, 
	vec3 viewDir, 
	vec3 reflectionDir, 
#if defined(LIT_CLEARCOAT)
	vec3 clearcoatReflectionDir,
#endif
	float gloss, 
	vec3 specularity, 
	vec3 geometricNormal, 
	mat3 tbn, 
#if defined(LIT_IRIDESCENCE)
	vec3 iridescenceFresnel,
#endif
	vec3 clearcoat_worldNormal,
	float clearcoat_gloss,
	float sheen_gloss,
	float iridescence_intensity
) {
	ClusterLightData clusterLightData;
	decodeClusterLightCore(clusterLightData, lightIndex);
	if (acceptLightMask(clusterLightData))
		evaluateLight(
			clusterLightData, 
			worldNormal, 
			viewDir, 
			reflectionDir, 
#if defined(LIT_CLEARCOAT)
			clearcoatReflectionDir, 
#endif
			gloss, 
			specularity, 
			geometricNormal, 
			tbn, 
#if defined(LIT_IRIDESCENCE)
			iridescenceFresnel,
#endif
			clearcoat_worldNormal,
			clearcoat_gloss,
			sheen_gloss,
			iridescence_intensity
		);
}
void addClusteredLights(
	vec3 worldNormal, 
	vec3 viewDir, 
	vec3 reflectionDir, 
#if defined(LIT_CLEARCOAT)
	vec3 clearcoatReflectionDir,
#endif
	float gloss, 
	vec3 specularity, 
	vec3 geometricNormal, 
	mat3 tbn, 
#if defined(LIT_IRIDESCENCE)
	vec3 iridescenceFresnel,
#endif
	vec3 clearcoat_worldNormal,
	float clearcoat_gloss,
	float sheen_gloss,
	float iridescence_intensity
) {
	if (clusterSkip > 0.5)
		return;
	vec3 cellCoords = floor((vPositionW - clusterBoundsMin) * clusterCellsCountByBoundsSize);
	if (!(any(lessThan(cellCoords, vec3(0.0))) || any(greaterThanEqual(cellCoords, clusterCellsMax)))) {
		float cellIndex = dot(clusterCellsDot, cellCoords);
		float clusterV = floor(cellIndex * clusterTextureSize.y);
		float clusterU = cellIndex - (clusterV * clusterTextureSize.x);
		#ifdef GL2
			for (int lightCellIndex = 0; lightCellIndex < clusterMaxCells; lightCellIndex++) {
				float lightIndex = texelFetch(clusterWorldTexture, ivec2(int(clusterU) + lightCellIndex, clusterV), 0).x;
				if (lightIndex <= 0.0)
						return;
				evaluateClusterLight(
					lightIndex * 255.0, 
					worldNormal, 
					viewDir, 
					reflectionDir,
#if defined(LIT_CLEARCOAT)
					clearcoatReflectionDir,
#endif
					gloss, 
					specularity, 
					geometricNormal, 
					tbn, 
#if defined(LIT_IRIDESCENCE)
					iridescenceFresnel,
#endif
					clearcoat_worldNormal,
					clearcoat_gloss,
					sheen_gloss,
					iridescence_intensity
				); 
			}
		#else
			clusterV = (clusterV + 0.5) * clusterTextureSize.z;
			const float maxLightCells = 256.0;
			for (float lightCellIndex = 0.5; lightCellIndex < maxLightCells; lightCellIndex++) {
				float lightIndex = texture2DLodEXT(clusterWorldTexture, vec2(clusterTextureSize.y * (clusterU + lightCellIndex), clusterV), 0.0).x;
				if (lightIndex <= 0.0)
					return;
				
				evaluateClusterLight(
					lightIndex * 255.0, 
					worldNormal, 
					viewDir, 
					reflectionDir,
#if defined(LIT_CLEARCOAT)
					clearcoatReflectionDir,
#endif
					gloss, 
					specularity, 
					geometricNormal, 
					tbn, 
#if defined(LIT_IRIDESCENCE)
					iridescenceFresnel,
#endif
					clearcoat_worldNormal,
					clearcoat_gloss,
					sheen_gloss,
					iridescence_intensity
				); 
				if (lightCellIndex >= clusterMaxCells) {
					break;
				}
			}
		#endif
	}
}
`;

var combinePS = `
vec3 combineColor(vec3 albedo, vec3 sheenSpecularity, float clearcoatSpecularity) {
	vec3 ret = vec3(0);
#ifdef LIT_OLD_AMBIENT
	ret += (dDiffuseLight - light_globalAmbient) * albedo + material_ambient * light_globalAmbient;
#else
	ret += albedo * dDiffuseLight;
#endif
#ifdef LIT_SPECULAR
	ret += dSpecularLight;
#endif
#ifdef LIT_REFLECTIONS
	ret += dReflection.rgb * dReflection.a;
#endif
#ifdef LIT_SHEEN
	float sheenScaling = 1.0 - max(max(sheenSpecularity.r, sheenSpecularity.g), sheenSpecularity.b) * 0.157;
	ret = ret * sheenScaling + (sSpecularLight + sReflection.rgb) * sheenSpecularity;
#endif
#ifdef LIT_CLEARCOAT
	float clearCoatScaling = 1.0 - ccFresnel * clearcoatSpecularity;
	ret = ret * clearCoatScaling + (ccSpecularLight + ccReflection.rgb) * clearcoatSpecularity;
#endif
	return ret;
}
`;

var cookiePS = `
vec4 getCookie2D(sampler2D tex, mat4 transform, float intensity) {
	vec4 projPos = transform * vec4(vPositionW, 1.0);
	projPos.xy /= projPos.w;
	return mix(vec4(1.0), texture2D(tex, projPos.xy), intensity);
}
vec4 getCookie2DClip(sampler2D tex, mat4 transform, float intensity) {
	vec4 projPos = transform * vec4(vPositionW, 1.0);
	projPos.xy /= projPos.w;
	if (projPos.x < 0.0 || projPos.x > 1.0 || projPos.y < 0.0 || projPos.y > 1.0 || projPos.z < 0.0) return vec4(0.0);
	return mix(vec4(1.0), texture2D(tex, projPos.xy), intensity);
}
vec4 getCookie2DXform(sampler2D tex, mat4 transform, float intensity, vec4 cookieMatrix, vec2 cookieOffset) {
	vec4 projPos = transform * vec4(vPositionW, 1.0);
	projPos.xy /= projPos.w;
	projPos.xy += cookieOffset;
	vec2 uv = mat2(cookieMatrix) * (projPos.xy-vec2(0.5)) + vec2(0.5);
	return mix(vec4(1.0), texture2D(tex, uv), intensity);
}
vec4 getCookie2DClipXform(sampler2D tex, mat4 transform, float intensity, vec4 cookieMatrix, vec2 cookieOffset) {
	vec4 projPos = transform * vec4(vPositionW, 1.0);
	projPos.xy /= projPos.w;
	projPos.xy += cookieOffset;
	if (projPos.x < 0.0 || projPos.x > 1.0 || projPos.y < 0.0 || projPos.y > 1.0 || projPos.z < 0.0) return vec4(0.0);
	vec2 uv = mat2(cookieMatrix) * (projPos.xy-vec2(0.5)) + vec2(0.5);
	return mix(vec4(1.0), texture2D(tex, uv), intensity);
}
vec4 getCookieCube(samplerCube tex, mat4 transform, float intensity) {
	return mix(vec4(1.0), textureCube(tex, dLightDirNormW * mat3(transform)), intensity);
}
`;

var cubeMapProjectBoxPS = `
uniform vec3 envBoxMin;
uniform vec3 envBoxMax;
vec3 cubeMapProject(vec3 nrdir) {
	nrdir = cubeMapRotate(nrdir);
	vec3 rbmax = (envBoxMax - vPositionW) / nrdir;
	vec3 rbmin = (envBoxMin - vPositionW) / nrdir;
	vec3 rbminmax;
	rbminmax.x = nrdir.x>0.0? rbmax.x : rbmin.x;
	rbminmax.y = nrdir.y>0.0? rbmax.y : rbmin.y;
	rbminmax.z = nrdir.z>0.0? rbmax.z : rbmin.z;
	float fa = min(min(rbminmax.x, rbminmax.y), rbminmax.z);
	vec3 posonbox = vPositionW + nrdir * fa;
	vec3 envBoxPos = (envBoxMin + envBoxMax) * 0.5;
	return normalize(posonbox - envBoxPos);
}
`;

var cubeMapProjectNonePS = `
vec3 cubeMapProject(vec3 dir) {
	return cubeMapRotate(dir);
}
`;

var cubeMapRotatePS = `
#ifdef CUBEMAP_ROTATION
uniform mat3 cubeMapRotationMatrix;
#endif
vec3 cubeMapRotate(vec3 refDir) {
#ifdef CUBEMAP_ROTATION
	return refDir * cubeMapRotationMatrix;
#else
	return refDir;
#endif
}
`;

var debugOutputPS = `
#ifdef DEBUG_ALBEDO_PASS
gl_FragColor = vec4(gammaCorrectOutput(litArgs_albedo), 1.0);
#endif
#ifdef DEBUG_UV0_PASS
gl_FragColor = vec4(litArgs_albedo , 1.0);
#endif
#ifdef DEBUG_WORLD_NORMAL_PASS
gl_FragColor = vec4(litArgs_worldNormal * 0.5 + 0.5, 1.0);
#endif
#ifdef DEBUG_OPACITY_PASS
gl_FragColor = vec4(vec3(litArgs_opacity) , 1.0);
#endif
#ifdef DEBUG_SPECULARITY_PASS
gl_FragColor = vec4(litArgs_specularity, 1.0);
#endif
#ifdef DEBUG_GLOSS_PASS
gl_FragColor = vec4(vec3(litArgs_gloss) , 1.0);
#endif
#ifdef DEBUG_METALNESS_PASS
gl_FragColor = vec4(vec3(litArgs_metalness) , 1.0);
#endif
#ifdef DEBUG_AO_PASS
gl_FragColor = vec4(vec3(litArgs_ao) , 1.0);
#endif
#ifdef DEBUG_EMISSION_PASS
gl_FragColor = vec4(gammaCorrectOutput(litArgs_emission), 1.0);
#endif
`;

var debugProcessFrontendPS = `
#ifdef DEBUG_LIGHTING_PASS
litArgs_albedo = vec3(0.5);
#endif
#ifdef DEBUG_UV0_PASS
#ifdef VARYING_VUV0
litArgs_albedo = vec3(vUv0, 0);
#else
litArgs_albedo = vec3(0);
#endif
#endif
`;

var decodePS = `
vec3 decodeLinear(vec4 raw) {
	return raw.rgb;
}
float decodeGamma(float raw) {
	return pow(raw, 2.2);
}
vec3 decodeGamma(vec3 raw) {
	return pow(raw, vec3(2.2));
}
vec3 decodeGamma(vec4 raw) {
	return pow(raw.xyz, vec3(2.2));
}
vec3 decodeRGBM(vec4 raw) {
	vec3 color = (8.0 * raw.a) * raw.rgb;
	return color * color;
}
vec3 decodeRGBP(vec4 raw) {
	vec3 color = raw.rgb * (-raw.a * 7.0 + 8.0);
	return color * color;
}
vec3 decodeRGBE(vec4 raw) {
	if (raw.a == 0.0) {
		return vec3(0.0, 0.0, 0.0);
	} else {
		return raw.xyz * pow(2.0, raw.w * 255.0 - 128.0);
	}
}
vec4 passThrough(vec4 raw) {
	return raw;
}
`;

var detailModesPS = `
vec3 detailMode_mul(vec3 c1, vec3 c2) {
	return c1 * c2;
}
vec3 detailMode_add(vec3 c1, vec3 c2) {
	return c1 + c2;
}
vec3 detailMode_screen(vec3 c1, vec3 c2) {
	return 1.0 - (1.0 - c1)*(1.0 - c2);
}
vec3 detailMode_overlay(vec3 c1, vec3 c2) {
	return mix(1.0 - 2.0*(1.0 - c1)*(1.0 - c2), 2.0*c1*c2, step(c1, vec3(0.5)));
}
vec3 detailMode_min(vec3 c1, vec3 c2) {
	return min(c1, c2);
}
vec3 detailMode_max(vec3 c1, vec3 c2) {
	return max(c1, c2);
}
`;

var diffusePS = `
#ifdef MAPCOLOR
uniform vec3 material_diffuse;
#endif
void getAlbedo() {
	dAlbedo = vec3(1.0);
#ifdef MAPCOLOR
	dAlbedo *= material_diffuse.rgb;
#endif
#ifdef MAPTEXTURE
	vec3 albedoBase = $DECODE(texture2DBias($SAMPLER, $UV, textureBias)).$CH;
	dAlbedo *= addAlbedoDetail(albedoBase);
#endif
#ifdef MAPVERTEX
	dAlbedo *= gammaCorrectInput(saturate(vVertexColor.$VC));
#endif
}
`;

var diffuseDetailMapPS = `
vec3 addAlbedoDetail(vec3 albedo) {
#ifdef MAPTEXTURE
	vec3 albedoDetail = $DECODE(texture2DBias($SAMPLER, $UV, textureBias)).$CH;
	return detailMode_$DETAILMODE(albedo, albedoDetail);
#else
	return albedo;
#endif
}
`;

var emissivePS = `
#ifdef MAPCOLOR
uniform vec3 material_emissive;
#endif
#ifdef MAPFLOAT
uniform float material_emissiveIntensity;
#endif
void getEmission() {
	dEmission = vec3(1.0);
	#ifdef MAPFLOAT
	dEmission *= material_emissiveIntensity;
	#endif
	#ifdef MAPCOLOR
	dEmission *= material_emissive;
	#endif
	#ifdef MAPTEXTURE
	dEmission *= $DECODE(texture2DBias($SAMPLER, $UV, textureBias)).$CH;
	#endif
	#ifdef MAPVERTEX
	dEmission *= gammaCorrectInput(saturate(vVertexColor.$VC));
	#endif
}
`;

var encodePS = `
vec4 encodeLinear(vec3 source) {
	return vec4(source, 1.0);
}
vec4 encodeGamma(vec3 source) {
	return vec4(pow(source + 0.0000001, vec3(1.0 / 2.2)), 1.0);
}
vec4 encodeRGBM(vec3 source) {
	vec4 result;
	result.rgb = pow(source.rgb, vec3(0.5));
	result.rgb *= 1.0 / 8.0;
	result.a = saturate( max( max( result.r, result.g ), max( result.b, 1.0 / 255.0 ) ) );
	result.a = ceil(result.a * 255.0) / 255.0;
	result.rgb /= result.a;
	return result;
}
vec4 encodeRGBP(vec3 source) {
	vec3 gamma = pow(source, vec3(0.5));
	float maxVal = min(8.0, max(1.0, max(gamma.x, max(gamma.y, gamma.z))));
	float v = 1.0 - ((maxVal - 1.0) / 7.0);
	v = ceil(v * 255.0) / 255.0;
	return vec4(gamma / (-v * 7.0 + 8.0), v);	
}
vec4 encodeRGBE(vec3 source) {
	float maxVal = max(source.x, max(source.y, source.z));
	if (maxVal < 1e-32) {
		return vec4(0, 0, 0, 0);
	} else {
		float e = ceil(log2(maxVal));
		return vec4(source / pow(2.0, e), (e + 128.0) / 255.0);
	}
}
`;

var endPS = `
	gl_FragColor.rgb = combineColor(litArgs_albedo, litArgs_sheen_specularity, litArgs_clearcoat_specularity);
	gl_FragColor.rgb += litArgs_emission;
	gl_FragColor.rgb = addFog(gl_FragColor.rgb);
	#ifndef HDR
	gl_FragColor.rgb = toneMap(gl_FragColor.rgb);
	gl_FragColor.rgb = gammaCorrectOutput(gl_FragColor.rgb);
	#endif
`;

var endVS = `
`;

var envAtlasPS = `
const float atlasSize = 512.0;
const float seamSize = 1.0 / atlasSize;
vec2 mapUv(vec2 uv, vec4 rect) {
	return vec2(mix(rect.x + seamSize, rect.x + rect.z - seamSize, uv.x),
				mix(rect.y + seamSize, rect.y + rect.w - seamSize, uv.y));
}
vec2 mapRoughnessUv(vec2 uv, float level) {
	float t = 1.0 / exp2(level);
	return mapUv(uv, vec4(0, 1.0 - t, t, t * 0.5));
}
vec2 mapShinyUv(vec2 uv, float level) {
	float t = 1.0 / exp2(level);
	return mapUv(uv, vec4(1.0 - t, 1.0 - t, t, t * 0.5));
}
`;

var envConstPS = `
vec3 processEnvironment(vec3 color) {
	return color;
}
`;

var envMultiplyPS = `
uniform float skyboxIntensity;
vec3 processEnvironment(vec3 color) {
	return color * skyboxIntensity;
}
`;

var extensionPS = `
`;

var extensionVS = `
`;

var falloffInvSquaredPS = `
float getFalloffWindow(float lightRadius, vec3 lightDir) {
	float sqrDist = dot(lightDir, lightDir);
	float invRadius = 1.0 / lightRadius;
	return square( saturate( 1.0 - square( sqrDist * square(invRadius) ) ) );
}
float getFalloffInvSquared(float lightRadius, vec3 lightDir) {
	float sqrDist = dot(lightDir, lightDir);
	float falloff = 1.0 / (sqrDist + 1.0);
	float invRadius = 1.0 / lightRadius;
	falloff *= 16.0;
	falloff *= square( saturate( 1.0 - square( sqrDist * square(invRadius) ) ) );
	return falloff;
}
`;

var falloffLinearPS = `
float getFalloffLinear(float lightRadius, vec3 lightDir) {
	float d = length(lightDir);
	return max(((lightRadius - d) / lightRadius), 0.0);
}
`;

var fixCubemapSeamsNonePS = `
vec3 fixSeams(vec3 vec, float mipmapIndex) {
	return vec;
}
vec3 fixSeams(vec3 vec) {
	return vec;
}
vec3 fixSeamsStatic(vec3 vec, float invRecMipSize) {
	return vec;
}
vec3 calcSeam(vec3 vec) {
	return vec3(0);
}
vec3 applySeam(vec3 vec, vec3 seam, float scale) {
	return vec;
}
`;

var fixCubemapSeamsStretchPS = `
vec3 fixSeams(vec3 vec, float mipmapIndex) {
	vec3 avec = abs(vec);
	float scale = 1.0 - exp2(mipmapIndex) / 128.0;
	float M = max(max(avec.x, avec.y), avec.z);
	if (avec.x != M) vec.x *= scale;
	if (avec.y != M) vec.y *= scale;
	if (avec.z != M) vec.z *= scale;
	return vec;
}
vec3 fixSeams(vec3 vec) {
	vec3 avec = abs(vec);
	float scale = 1.0 - 1.0 / 128.0;
	float M = max(max(avec.x, avec.y), avec.z);
	if (avec.x != M) vec.x *= scale;
	if (avec.y != M) vec.y *= scale;
	if (avec.z != M) vec.z *= scale;
	return vec;
}
vec3 fixSeamsStatic(vec3 vec, float invRecMipSize) {
	vec3 avec = abs(vec);
	float scale = invRecMipSize;
	float M = max(max(avec.x, avec.y), avec.z);
	if (avec.x != M) vec.x *= scale;
	if (avec.y != M) vec.y *= scale;
	if (avec.z != M) vec.z *= scale;
	return vec;
}
vec3 calcSeam(vec3 vec) {
	vec3 avec = abs(vec);
	float M = max(avec.x, max(avec.y, avec.z));
	return vec3(avec.x != M ? 1.0 : 0.0,
				avec.y != M ? 1.0 : 0.0,
				avec.z != M ? 1.0 : 0.0);
}
vec3 applySeam(vec3 vec, vec3 seam, float scale) {
	return vec * (seam * -scale + vec3(1.0));
}
`;

var floatUnpackingPS = `
float bytes2float2(vec2 data) {
	return dot(data, vec2(1.0, 1.0 / 255.0));
}
float bytes2float3(vec3 data) {
	return dot(data, vec3(1.0, 1.0 / 255.0, 1.0 / 65025.0));
}
float bytes2float4(vec4 data) {
	return dot(data, vec4(1.0, 1.0 / 255.0, 1.0 / 65025.0, 1.0 / 16581375.0));
}
float bytes2floatRange2(vec2 data, float min, float max) {
	return mix(min, max, bytes2float2(data));
}
float bytes2floatRange3(vec3 data, float min, float max) {
	return mix(min, max, bytes2float3(data));
}
float bytes2floatRange4(vec4 data, float min, float max) {
	return mix(min, max, bytes2float4(data));
}
float mantissaExponent2Float(vec4 pack)
{
	float value = bytes2floatRange3(pack.xyz, -1.0, 1.0);
	float exponent = floor(pack.w * 255.0 - 127.0);
	return value * exp2(exponent);
}
`;

var fogExpPS = `
uniform vec3 fog_color;
uniform float fog_density;
float dBlendModeFogFactor = 1.0;
vec3 addFog(vec3 color) {
	float depth = gl_FragCoord.z / gl_FragCoord.w;
	float fogFactor = exp(-depth * fog_density);
	fogFactor = clamp(fogFactor, 0.0, 1.0);
	return mix(fog_color * dBlendModeFogFactor, color, fogFactor);
}
`;

var fogExp2PS = `
uniform vec3 fog_color;
uniform float fog_density;
float dBlendModeFogFactor = 1.0;
vec3 addFog(vec3 color) {
	float depth = gl_FragCoord.z / gl_FragCoord.w;
	float fogFactor = exp(-depth * depth * fog_density * fog_density);
	fogFactor = clamp(fogFactor, 0.0, 1.0);
	return mix(fog_color * dBlendModeFogFactor, color, fogFactor);
}
`;

var fogLinearPS = `
uniform vec3 fog_color;
uniform float fog_start;
uniform float fog_end;
float dBlendModeFogFactor = 1.0;
vec3 addFog(vec3 color) {
	float depth = gl_FragCoord.z / gl_FragCoord.w;
	float fogFactor = (fog_end - depth) / (fog_end - fog_start);
	fogFactor = clamp(fogFactor, 0.0, 1.0);
	return mix(fog_color * dBlendModeFogFactor, color, fogFactor);
}
`;

var fogNonePS = `
float dBlendModeFogFactor = 1.0;
vec3 addFog(vec3 color) {
	return color;
}
`;

var fresnelSchlickPS = `
vec3 getFresnel(
		float cosTheta, 
		float gloss, 
		vec3 specularity
#if defined(LIT_IRIDESCENCE)
		, vec3 iridescenceFresnel, 
		float iridescenceIntensity
#endif
	) {
	float fresnel = pow(1.0 - max(cosTheta, 0.0), 5.0);
	float glossSq = gloss * gloss;
	vec3 ret = specularity + (max(vec3(glossSq), specularity) - specularity) * fresnel;
#if defined(LIT_IRIDESCENCE)
	return mix(ret, iridescenceFresnel, iridescenceIntensity);
#else
	return ret;
#endif	
}
float getFresnelCC(float cosTheta) {
	float fresnel = pow(1.0 - max(cosTheta, 0.0), 5.0);
	return 0.04 + (1.0 - 0.04) * fresnel;
}
`;

var fullscreenQuadPS = `
varying vec2 vUv0;
uniform sampler2D source;
void main(void) {
	gl_FragColor = texture2D(source, vUv0);
}
`;

var fullscreenQuadVS = `
attribute vec2 vertex_position;
varying vec2 vUv0;
void main(void)
{
	gl_Position = vec4(vertex_position, 0.5, 1.0);
	vUv0 = vertex_position.xy*0.5+0.5;
}
`;

var gamma1_0PS = `
float gammaCorrectInput(float color) {
	return color;
}
vec3 gammaCorrectInput(vec3 color) {
	return color;
}
vec4 gammaCorrectInput(vec4 color) {
	return color;
}
vec3 gammaCorrectOutput(vec3 color) {
	return color;
}
`;

var gamma2_2PS = `
float gammaCorrectInput(float color) {
	return decodeGamma(color);
}
vec3 gammaCorrectInput(vec3 color) {
	return decodeGamma(color);
}
vec4 gammaCorrectInput(vec4 color) {
	return vec4(decodeGamma(color.xyz), color.w);
}
vec3 gammaCorrectOutput(vec3 color) {
#ifdef HDR
	return color;
#else
	return pow(color + 0.0000001, vec3(1.0 / 2.2));
#endif
}
`;

var glossPS = `
#ifdef MAPFLOAT
uniform float material_gloss;
#endif
void getGlossiness() {
	dGlossiness = 1.0;
	#ifdef MAPFLOAT
	dGlossiness *= material_gloss;
	#endif
	#ifdef MAPTEXTURE
	dGlossiness *= texture2DBias($SAMPLER, $UV, textureBias).$CH;
	#endif
	#ifdef MAPVERTEX
	dGlossiness *= saturate(vVertexColor.$VC);
	#endif
	#ifdef MAPINVERT
	dGlossiness = 1.0 - dGlossiness;
	#endif
	dGlossiness += 0.0000001;
}
`;

var iridescenceDiffractionPS = `
uniform float material_iridescenceRefractionIndex;
#ifndef PI
#define PI 3.14159265
#endif
float iridescence_iorToFresnel(float transmittedIor, float incidentIor) {
	return pow((transmittedIor - incidentIor) / (transmittedIor + incidentIor), 2.0);
}
vec3 iridescence_iorToFresnel(vec3 transmittedIor, float incidentIor) {
	return pow((transmittedIor - vec3(incidentIor)) / (transmittedIor + vec3(incidentIor)), vec3(2.0));
}
vec3 iridescence_fresnelToIor(vec3 f0) {
	vec3 sqrtF0 = sqrt(f0);
	return (vec3(1.0) + sqrtF0) / (vec3(1.0) - sqrtF0);
}
vec3 iridescence_sensitivity(float opd, vec3 shift) {
	float phase = 2.0 * PI * opd * 1.0e-9;
	const vec3 val = vec3(5.4856e-13, 4.4201e-13, 5.2481e-13);
	const vec3 pos = vec3(1.6810e+06, 1.7953e+06, 2.2084e+06);
	const vec3 var = vec3(4.3278e+09, 9.3046e+09, 6.6121e+09);
	vec3 xyz = val * sqrt(2.0 * PI * var) * cos(pos * phase + shift) * exp(-pow(phase, 2.0) * var);
	xyz.x += 9.7470e-14 * sqrt(2.0 * PI * 4.5282e+09) * cos(2.2399e+06 * phase + shift[0]) * exp(-4.5282e+09 * pow(phase, 2.0));
	xyz /= vec3(1.0685e-07);
	const mat3 XYZ_TO_REC709 = mat3(
		3.2404542, -0.9692660,  0.0556434,
	   -1.5371385,  1.8760108, -0.2040259,
	   -0.4985314,  0.0415560,  1.0572252
	);
	return XYZ_TO_REC709 * xyz;
}
float iridescence_fresnel(float cosTheta, float f0) {
	float x = clamp(1.0 - cosTheta, 0.0, 1.0);
	float x2 = x * x;
	float x5 = x * x2 * x2;
	return f0 + (1.0 - f0) * x5;
} 
vec3 iridescence_fresnel(float cosTheta, vec3 f0) {
	float x = clamp(1.0 - cosTheta, 0.0, 1.0);
	float x2 = x * x;
	float x5 = x * x2 * x2; 
	return f0 + (vec3(1.0) - f0) * x5;
}
vec3 calcIridescence(float outsideIor, float cosTheta, vec3 base_f0, float iridescenceThickness) {
	float iridescenceIor = mix(outsideIor, material_iridescenceRefractionIndex, smoothstep(0.0, 0.03, iridescenceThickness));
	float sinTheta2Sq = pow(outsideIor / iridescenceIor, 2.0) * (1.0 - pow(cosTheta, 2.0));
	float cosTheta2Sq = 1.0 - sinTheta2Sq;
	if (cosTheta2Sq < 0.0) {
		return vec3(1.0);
	}
	float cosTheta2 = sqrt(cosTheta2Sq);
	float r0 = iridescence_iorToFresnel(iridescenceIor, outsideIor);
	float r12 = iridescence_fresnel(cosTheta, r0);
	float r21 = r12;
	float t121 = 1.0 - r12;
	float phi12 = iridescenceIor < outsideIor ? PI : 0.0;
	float phi21 = PI - phi12;
	vec3 baseIor = iridescence_fresnelToIor(base_f0 + vec3(0.0001));
	vec3 r1 = iridescence_iorToFresnel(baseIor, iridescenceIor);
	vec3 r23 = iridescence_fresnel(cosTheta2, r1);
	vec3 phi23 = vec3(0.0);
	if (baseIor[0] < iridescenceIor) phi23[0] = PI;
	if (baseIor[1] < iridescenceIor) phi23[1] = PI;
	if (baseIor[2] < iridescenceIor) phi23[2] = PI;
	float opd = 2.0 * iridescenceIor * iridescenceThickness * cosTheta2;
	vec3 phi = vec3(phi21) + phi23; 
	vec3 r123Sq = clamp(r12 * r23, 1e-5, 0.9999);
	vec3 r123 = sqrt(r123Sq);
	vec3 rs = pow(t121, 2.0) * r23 / (1.0 - r123Sq);
	vec3 c0 = r12 + rs;
	vec3 i = c0;
	vec3 cm = rs - t121;
	for (int m = 1; m <= 2; m++) {
		cm *= r123;
		vec3 sm = 2.0 * iridescence_sensitivity(float(m) * opd, float(m) * phi);
		i += cm * sm;
	}
	return max(i, vec3(0.0));
}
vec3 getIridescence(float cosTheta, vec3 specularity, float iridescenceThickness) {
	return calcIridescence(1.0, cosTheta, specularity, iridescenceThickness);
}
`;

var iridescencePS = `
#ifdef MAPFLOAT
uniform float material_iridescence;
#endif
void getIridescence() {
	float iridescence = 1.0;
	#ifdef MAPFLOAT
	iridescence *= material_iridescence;
	#endif
	#ifdef MAPTEXTURE
	iridescence *= texture2DBias($SAMPLER, $UV, textureBias).$CH;
	#endif
	dIridescence = iridescence; 
}
`;

var iridescenceThicknessPS = `
uniform float material_iridescenceThicknessMax;
#ifdef MAPTEXTURE
uniform float material_iridescenceThicknessMin;
#endif
void getIridescenceThickness() {
	#ifdef MAPTEXTURE
	float blend = texture2DBias($SAMPLER, $UV, textureBias).$CH;
	float iridescenceThickness = mix(material_iridescenceThicknessMin, material_iridescenceThicknessMax, blend);
	#else
	float iridescenceThickness = material_iridescenceThicknessMax;
	#endif
	dIridescenceThickness = iridescenceThickness; 
}
`;

var instancingVS = `
attribute vec4 instance_line1;
attribute vec4 instance_line2;
attribute vec4 instance_line3;
attribute vec4 instance_line4;
`;

var iorPS = `
#ifdef MAPFLOAT
uniform float material_refractionIndex;
#endif
void getIor() {
#ifdef MAPFLOAT
	dIor = material_refractionIndex;
#else
	dIor = 1.0 / 1.5;
#endif
}
`;

var lightDiffuseLambertPS = `
float getLightDiffuse(vec3 worldNormal, vec3 viewDir, vec3 lightDir, vec3 lightDirNorm) {
	return max(dot(worldNormal, -lightDirNorm), 0.0);
}
`;

var lightDirPointPS = `
void getLightDirPoint(vec3 lightPosW) {
	dLightDirW = vPositionW - lightPosW;
	dLightDirNormW = normalize(dLightDirW);
	dLightPosW = lightPosW;
}
`;

var lightmapAddPS = `
void addLightMap(
	vec3 lightmap, 
	vec3 dir, 
	vec3 worldNormal, 
	vec3 viewDir, 
	vec3 reflectionDir, 
	float gloss, 
	vec3 specularity, 
	vec3 vertexNormal, 
	mat3 tbn
#if defined(LIT_IRIDESCENCE)
	vec3 iridescenceFresnel, 
	float iridescenceIntensity
#endif
) {
	dDiffuseLight += lightmap;
}
`;

var lightmapDirAddPS = `
void addLightMap(
	vec3 lightmap, 
	vec3 dir, 
	vec3 worldNormal, 
	vec3 viewDir, 
	vec3 reflectionDir, 
	float gloss, 
	vec3 specularity, 
	vec3 vertexNormal, 
	mat3 tbn
#if defined(LIT_IRIDESCENCE)
	vec3 iridescenceFresnel, 
	float iridescenceIntensity
#endif
) {
	if (dot(dir, dir) < 0.0001) {
		dDiffuseLight += lightmap;
	} else {
		float vlight = saturate(dot(dir, -vertexNormal));
		float flight = saturate(dot(dir, -worldNormal));
		float nlight = (flight / max(vlight, 0.01)) * 0.5;
		dDiffuseLight += lightmap * nlight * 2.0;
		vec3 halfDir = normalize(-dir + viewDir);
		vec3 specularLight = lightmap * getLightSpecular(halfDir, reflectionDir, worldNormal, viewDir, dir, gloss, tbn);
#ifdef LIT_SPECULAR_FRESNEL
		specularLight *= 
			getFresnel(dot(viewDir, halfDir), 
			gloss, 
			specularity
		#if defined(LIT_IRIDESCENCE)
			, iridescenceFresnel,
			iridescenceIntensity
		#endif
			);
#endif
		dSpecularLight += specularLight;
	}
}
`;

var lightmapDirPS = `
uniform sampler2D texture_lightMap;
uniform sampler2D texture_dirLightMap;
void getLightMap() {
	dLightmap = $DECODE(texture2DBias(texture_lightMap, $UV, textureBias)).$CH;
	vec3 dir = texture2DBias(texture_dirLightMap, $UV, textureBias).xyz * 2.0 - 1.0;
	float dirDot = dot(dir, dir);
	dLightmapDir = (dirDot > 0.001) ? dir / sqrt(dirDot) : vec3(0.0);
}
`;

var lightmapSinglePS = `
void getLightMap() {
	dLightmap = vec3(1.0);
	#ifdef MAPTEXTURE
	dLightmap *= $DECODE(texture2DBias($SAMPLER, $UV, textureBias)).$CH;
	#endif
	#ifdef MAPVERTEX
	dLightmap *= saturate(vVertexColor.$VC);
	#endif
}
`;

var lightSpecularAnisoGGXPS = `
float calcLightSpecular(float gloss, vec3 worldNormal, vec3 viewDir, vec3 h, vec3 lightDirNorm, mat3 tbn) {
	float PI = 3.141592653589793;
	float roughness = max((1.0 - gloss) * (1.0 - gloss), 0.001);
	float anisotropy = material_anisotropy * roughness;
 
	float at = max((roughness + anisotropy), roughness / 4.0);
	float ab = max((roughness - anisotropy), roughness / 4.0);
	float NoH = dot(worldNormal, h);
	float ToH = dot(tbn[0], h);
	float BoH = dot(tbn[1], h);
	float a2 = at * ab;
	vec3 v = vec3(ab * ToH, at * BoH, a2 * NoH);
	float v2 = dot(v, v);
	float w2 = a2 / v2;
	float D = a2 * w2 * w2 * (1.0 / PI);
	float ToV = dot(tbn[0], viewDir);
	float BoV = dot(tbn[1], viewDir);
	float ToL = dot(tbn[0], -lightDirNorm);
	float BoL = dot(tbn[1], -lightDirNorm);
	float NoV = dot(worldNormal, viewDir);
	float NoL = dot(worldNormal, -lightDirNorm);
	float lambdaV = NoL * length(vec3(at * ToV, ab * BoV, NoV));
	float lambdaL = NoV * length(vec3(at * ToL, ab * BoL, NoL));
	float G = 0.5 / (lambdaV + lambdaL);
	return D * G;
}
float getLightSpecular(vec3 h, vec3 reflDir, vec3 worldNormal, vec3 viewDir, vec3 lightDirNorm, float gloss, mat3 tbn) {
	return calcLightSpecular(gloss, worldNormal, viewDir, h, lightDirNorm, tbn);
}
`;

var lightSpecularBlinnPS = `
float calcLightSpecular(float gloss, vec3 worldNormal, vec3 h) {
	float nh = max( dot( h, worldNormal ), 0.0 );
	float specPow = exp2(gloss * 11.0);
	specPow = max(specPow, 0.0001);
	return pow(nh, specPow) * (specPow + 2.0) / 8.0;
}
float getLightSpecular(vec3 h, vec3 reflDir, vec3 worldNormal, vec3 viewDir, vec3 lightDirNorm, float gloss, mat3 tbn) {
	return calcLightSpecular(gloss, worldNormal, h);
}
`;

var lightSpecularPhongPS = `
float calcLightSpecular(float gloss, vec3 reflDir, vec3 lightDirNorm) {
	float specPow = gloss;
	return pow(max(dot(reflDir, -lightDirNorm), 0.0), specPow + 0.0001);
}
float getLightSpecular(vec3 h, vec3 reflDir, vec3 worldNormal, vec3 viewDir, vec3 lightDirNorm, float gloss, mat3 tbn) {
	return calcLightSpecular(gloss, reflDir, lightDirNorm);
}
`;

var lightSheenPS = `
float sheenD(vec3 normal, vec3 h, float roughness) {
	float invR = 1.0 / (roughness * roughness);
	float cos2h = max(dot(normal, h), 0.0);
	cos2h *= cos2h;
	float sin2h = max(1.0 - cos2h, 0.0078125);
	return (2.0 + invR) * pow(sin2h, invR * 0.5) / (2.0 * PI);
}
float sheenV(vec3 normal, vec3 viewDir, vec3 light) {
	float NoV = max(dot(normal, viewDir), 0.000001);
	float NoL = max(dot(normal, light), 0.000001);
	return 1.0 / (4.0 * (NoL + NoV - NoL * NoV));
}
float getLightSpecularSheen(vec3 h, vec3 worldNormal, vec3 viewDir, vec3 lightDirNorm, float sheenGloss) {
	float D = sheenD(worldNormal, h, sheenGloss);
	float V = sheenV(worldNormal, viewDir, -lightDirNorm);
	return D * V;
}
`;

var linearizeDepthPS = `
#ifndef LINEARIZE_DEPTH
#define LINEARIZE_DEPTH
float linearizeDepth(float z, vec4 cameraParams) {
	if (cameraParams.w == 0.0)
		return (cameraParams.z * cameraParams.y) / (cameraParams.y + z * (cameraParams.z - cameraParams.y));
	else
		return cameraParams.z + z * (cameraParams.y - cameraParams.z);
}
#ifndef CAMERAPLANES
#define CAMERAPLANES
uniform vec4 camera_params;
#endif
#ifdef GL2
float linearizeDepth(float z) {
	return linearizeDepth(z, camera_params);
}
#else
#ifndef UNPACKFLOAT
#define UNPACKFLOAT
float unpackFloat(vec4 rgbaDepth) {
	const vec4 bitShift = vec4(1.0 / (256.0 * 256.0 * 256.0), 1.0 / (256.0 * 256.0), 1.0 / 256.0, 1.0);
	return dot(rgbaDepth, bitShift);
}
#endif
#endif
#endif
`;

var litShaderArgsPS = `
vec3 litArgs_albedo;
float litArgs_opacity;
vec3 litArgs_emission;
vec3 litArgs_worldNormal;
float litArgs_ao;
vec3 litArgs_lightmap;
vec3 litArgs_lightmapDir;
float litArgs_metalness;
vec3 litArgs_specularity;
float litArgs_specularityFactor;
float litArgs_gloss;
float litArgs_sheen_gloss;
vec3 litArgs_sheen_specularity;
float litArgs_transmission;
float litArgs_thickness;
float litArgs_ior;
float litArgs_iridescence_intensity;
float litArgs_iridescence_thickness;
vec3 litArgs_clearcoat_worldNormal;
float litArgs_clearcoat_specularity;
float litArgs_clearcoat_gloss;
`;

var ltcPS = `
mat3 transposeMat3( const in mat3 m ) {
	mat3 tmp;
	tmp[ 0 ] = vec3( m[ 0 ].x, m[ 1 ].x, m[ 2 ].x );
	tmp[ 1 ] = vec3( m[ 0 ].y, m[ 1 ].y, m[ 2 ].y );
	tmp[ 2 ] = vec3( m[ 0 ].z, m[ 1 ].z, m[ 2 ].z );
	return tmp;
}
vec2 LTC_Uv( const in vec3 N, const in vec3 V, const in float roughness ) {
	const float LUT_SIZE = 64.0;
	const float LUT_SCALE = ( LUT_SIZE - 1.0 ) / LUT_SIZE;
	const float LUT_BIAS = 0.5 / LUT_SIZE;
	float dotNV = saturate( dot( N, V ) );
	vec2 uv = vec2( roughness, sqrt( 1.0 - dotNV ) );
	uv = uv * LUT_SCALE + LUT_BIAS;
	return uv;
}
float LTC_ClippedSphereFormFactor( const in vec3 f ) {
	float l = length( f );
	return max( ( l * l + f.z ) / ( l + 1.0 ), 0.0 );
}
vec3 LTC_EdgeVectorFormFactor( const in vec3 v1, const in vec3 v2 ) {
	float x = dot( v1, v2 );
	float y = abs( x );
	float a = 0.8543985 + ( 0.4965155 + 0.0145206 * y ) * y;
	float b = 3.4175940 + ( 4.1616724 + y ) * y;
	float v = a / b;
	float theta_sintheta = ( x > 0.0 ) ? v : 0.5 * inversesqrt( max( 1.0 - x * x, 1e-7 ) ) - v;
	return cross( v1, v2 ) * theta_sintheta;
}
struct Coords {
	vec3 coord0;
	vec3 coord1;
	vec3 coord2;
	vec3 coord3;
};
float LTC_EvaluateRect( const in vec3 N, const in vec3 V, const in vec3 P, const in mat3 mInv, const in Coords rectCoords) {
	vec3 v1 = rectCoords.coord1 - rectCoords.coord0;
	vec3 v2 = rectCoords.coord3 - rectCoords.coord0;
	
	vec3 lightNormal = cross( v1, v2 );
	float factor = sign(-dot( lightNormal, P - rectCoords.coord0 ));
	vec3 T1, T2;
	T1 = normalize( V - N * dot( V, N ) );
	T2 =  factor * cross( N, T1 );
	mat3 mat = mInv * transposeMat3( mat3( T1, T2, N ) );
	vec3 coords[ 4 ];
	coords[ 0 ] = mat * ( rectCoords.coord0 - P );
	coords[ 1 ] = mat * ( rectCoords.coord1 - P );
	coords[ 2 ] = mat * ( rectCoords.coord2 - P );
	coords[ 3 ] = mat * ( rectCoords.coord3 - P );
	coords[ 0 ] = normalize( coords[ 0 ] );
	coords[ 1 ] = normalize( coords[ 1 ] );
	coords[ 2 ] = normalize( coords[ 2 ] );
	coords[ 3 ] = normalize( coords[ 3 ] );
	vec3 vectorFormFactor = vec3( 0.0 );
	vectorFormFactor += LTC_EdgeVectorFormFactor( coords[ 0 ], coords[ 1 ] );
	vectorFormFactor += LTC_EdgeVectorFormFactor( coords[ 1 ], coords[ 2 ] );
	vectorFormFactor += LTC_EdgeVectorFormFactor( coords[ 2 ], coords[ 3 ] );
	vectorFormFactor += LTC_EdgeVectorFormFactor( coords[ 3 ], coords[ 0 ] );
	float result = LTC_ClippedSphereFormFactor( vectorFormFactor );
	return result;
}
Coords dLTCCoords;
Coords getLTCLightCoords(vec3 lightPos, vec3 halfWidth, vec3 halfHeight){
	Coords coords;
	coords.coord0 = lightPos + halfWidth - halfHeight;
	coords.coord1 = lightPos - halfWidth - halfHeight;
	coords.coord2 = lightPos - halfWidth + halfHeight;
	coords.coord3 = lightPos + halfWidth + halfHeight;
	return coords;
}
float dSphereRadius;
Coords getSphereLightCoords(vec3 lightPos, vec3 halfWidth, vec3 halfHeight){
	dSphereRadius = max(length(halfWidth), length(halfHeight));
	vec3 f = reflect(normalize(lightPos - view_position), vNormalW);
	vec3 w = normalize(cross(f, halfHeight));
	vec3 h = normalize(cross(f, w));
	return getLTCLightCoords(lightPos, w * dSphereRadius, h * dSphereRadius);
}
vec2 dLTCUV;
#ifdef LIT_CLEARCOAT
vec2 ccLTCUV;
#endif
vec2 getLTCLightUV(float gloss, vec3 worldNormal, vec3 viewDir)
{
	float roughness = max((1.0 - gloss) * (1.0 - gloss), 0.001);
	return LTC_Uv( worldNormal, viewDir, roughness );
}
vec3 dLTCSpecFres;
#ifdef LIT_CLEARCOAT
vec3 ccLTCSpecFres;
#endif
vec3 getLTCLightSpecFres(vec2 uv, vec3 specularity)
{
	vec4 t2 = texture2DLodEXT(areaLightsLutTex2, uv, 0.0);
	#ifdef AREA_R8_G8_B8_A8_LUTS
	t2 *= vec4(0.693103,1,1,1);
	t2 += vec4(0.306897,0,0,0);
	#endif
	return specularity * t2.x + ( vec3( 1.0 ) - specularity) * t2.y;
}
void calcLTCLightValues(float gloss, vec3 worldNormal, vec3 viewDir, vec3 specularity, float clearcoatGloss, vec3 clearcoatWorldNormal, float clearcoatSpecularity)
{
	dLTCUV = getLTCLightUV(gloss, worldNormal, viewDir);
	dLTCSpecFres = getLTCLightSpecFres(dLTCUV, specularity); 
#ifdef LIT_CLEARCOAT
	ccLTCUV = getLTCLightUV(clearcoatGloss, clearcoatWorldNormal, viewDir);
	ccLTCSpecFres = getLTCLightSpecFres(ccLTCUV, vec3(clearcoatSpecularity));
#endif
}
void calcRectLightValues(vec3 lightPos, vec3 halfWidth, vec3 halfHeight)
{
	dLTCCoords = getLTCLightCoords(lightPos, halfWidth, halfHeight);
}
void calcDiskLightValues(vec3 lightPos, vec3 halfWidth, vec3 halfHeight)
{
	calcRectLightValues(lightPos, halfWidth, halfHeight);
}
void calcSphereLightValues(vec3 lightPos, vec3 halfWidth, vec3 halfHeight)
{
	dLTCCoords = getSphereLightCoords(lightPos, halfWidth, halfHeight);
}
vec3 SolveCubic(vec4 Coefficient)
{
	float pi = 3.14159;
	Coefficient.xyz /= Coefficient.w;
	Coefficient.yz /= 3.0;
	float A = Coefficient.w;
	float B = Coefficient.z;
	float C = Coefficient.y;
	float D = Coefficient.x;
	vec3 Delta = vec3(
		-Coefficient.z * Coefficient.z + Coefficient.y,
		-Coefficient.y * Coefficient.z + Coefficient.x,
		dot(vec2(Coefficient.z, -Coefficient.y), Coefficient.xy)
	);
	float Discriminant = dot(vec2(4.0 * Delta.x, -Delta.y), Delta.zy);
	vec3 RootsA, RootsD;
	vec2 xlc, xsc;
	{
		float A_a = 1.0;
		float C_a = Delta.x;
		float D_a = -2.0 * B * Delta.x + Delta.y;
		float Theta = atan(sqrt(Discriminant), -D_a) / 3.0;
		float x_1a = 2.0 * sqrt(-C_a) * cos(Theta);
		float x_3a = 2.0 * sqrt(-C_a) * cos(Theta + (2.0 / 3.0) * pi);
		float xl;
		if ((x_1a + x_3a) > 2.0 * B)
			xl = x_1a;
		else
			xl = x_3a;
		xlc = vec2(xl - B, A);
	}
	{
		float A_d = D;
		float C_d = Delta.z;
		float D_d = -D * Delta.y + 2.0 * C * Delta.z;
		float Theta = atan(D * sqrt(Discriminant), -D_d) / 3.0;
		float x_1d = 2.0 * sqrt(-C_d) * cos(Theta);
		float x_3d = 2.0 * sqrt(-C_d) * cos(Theta + (2.0 / 3.0) * pi);
		float xs;
		if (x_1d + x_3d < 2.0 * C)
			xs = x_1d;
		else
			xs = x_3d;
		xsc = vec2(-D, xs + C);
	}
	float E =  xlc.y * xsc.y;
	float F = -xlc.x * xsc.y - xlc.y * xsc.x;
	float G =  xlc.x * xsc.x;
	vec2 xmc = vec2(C * F - B * G, -B * F + C * E);
	vec3 Root = vec3(xsc.x / xsc.y, xmc.x / xmc.y, xlc.x / xlc.y);
	if (Root.x < Root.y && Root.x < Root.z)
		Root.xyz = Root.yxz;
	else if (Root.z < Root.x && Root.z < Root.y)
		Root.xyz = Root.xzy;
	return Root;
}
float LTC_EvaluateDisk(vec3 N, vec3 V, vec3 P, mat3 Minv, Coords points)
{
	vec3 T1, T2;
	T1 = normalize(V - N * dot(V, N));
	T2 = cross(N, T1);
	mat3 R = transposeMat3( mat3( T1, T2, N ) );
	vec3 L_[ 3 ];
	L_[ 0 ] = R * ( points.coord0 - P );
	L_[ 1 ] = R * ( points.coord1 - P );
	L_[ 2 ] = R * ( points.coord2 - P );
	vec3 Lo_i = vec3(0);
	vec3 C  = 0.5 * (L_[0] + L_[2]);
	vec3 V1 = 0.5 * (L_[1] - L_[2]);
	vec3 V2 = 0.5 * (L_[1] - L_[0]);
	C  = Minv * C;
	V1 = Minv * V1;
	V2 = Minv * V2;
	float a, b;
	float d11 = dot(V1, V1);
	float d22 = dot(V2, V2);
	float d12 = dot(V1, V2);
	if (abs(d12) / sqrt(d11 * d22) > 0.0001)
	{
		float tr = d11 + d22;
		float det = -d12 * d12 + d11 * d22;
		det = sqrt(det);
		float u = 0.5 * sqrt(tr - 2.0 * det);
		float v = 0.5 * sqrt(tr + 2.0 * det);
		float e_max = (u + v) * (u + v);
		float e_min = (u - v) * (u - v);
		vec3 V1_, V2_;
		if (d11 > d22)
		{
			V1_ = d12 * V1 + (e_max - d11) * V2;
			V2_ = d12 * V1 + (e_min - d11) * V2;
		}
		else
		{
			V1_ = d12*V2 + (e_max - d22)*V1;
			V2_ = d12*V2 + (e_min - d22)*V1;
		}
		a = 1.0 / e_max;
		b = 1.0 / e_min;
		V1 = normalize(V1_);
		V2 = normalize(V2_);
	}
	else
	{
		a = 1.0 / dot(V1, V1);
		b = 1.0 / dot(V2, V2);
		V1 *= sqrt(a);
		V2 *= sqrt(b);
	}
	vec3 V3 = cross(V1, V2);
	if (dot(C, V3) < 0.0)
		V3 *= -1.0;
	float L  = dot(V3, C);
	float x0 = dot(V1, C) / L;
	float y0 = dot(V2, C) / L;
	float E1 = inversesqrt(a);
	float E2 = inversesqrt(b);
	a *= L * L;
	b *= L * L;
	float c0 = a * b;
	float c1 = a * b * (1.0 + x0 * x0 + y0 * y0) - a - b;
	float c2 = 1.0 - a * (1.0 + x0 * x0) - b * (1.0 + y0 * y0);
	float c3 = 1.0;
	vec3 roots = SolveCubic(vec4(c0, c1, c2, c3));
	float e1 = roots.x;
	float e2 = roots.y;
	float e3 = roots.z;
	vec3 avgDir = vec3(a * x0 / (a - e2), b * y0 / (b - e2), 1.0);
	mat3 rotate = mat3(V1, V2, V3);
	avgDir = rotate * avgDir;
	avgDir = normalize(avgDir);
	float L1 = sqrt(-e2 / e3);
	float L2 = sqrt(-e2 / e1);
	float formFactor = L1 * L2 * inversesqrt((1.0 + L1 * L1) * (1.0 + L2 * L2));
	
	const float LUT_SIZE = 64.0;
	const float LUT_SCALE = ( LUT_SIZE - 1.0 ) / LUT_SIZE;
	const float LUT_BIAS = 0.5 / LUT_SIZE;
	vec2 uv = vec2(avgDir.z * 0.5 + 0.5, formFactor);
	uv = uv*LUT_SCALE + LUT_BIAS;
	float scale = texture2DLodEXT(areaLightsLutTex2, uv, 0.0).w;
	return formFactor*scale;
}
float getRectLightDiffuse(vec3 worldNormal, vec3 viewDir, vec3 lightDir, vec3 lightDirNorm) {
	return LTC_EvaluateRect( worldNormal, viewDir, vPositionW, mat3( 1.0 ), dLTCCoords );
}
float getDiskLightDiffuse(vec3 worldNormal, vec3 viewDir, vec3 lightDir, vec3 lightDirNorm) {
	return LTC_EvaluateDisk( worldNormal, viewDir, vPositionW, mat3( 1.0 ), dLTCCoords );
}
float getSphereLightDiffuse(vec3 worldNormal, vec3 viewDir, vec3 lightDir, vec3 lightDirNorm) {
	float falloff = dSphereRadius / (dot(lightDir, lightDir) + dSphereRadius);
	return getLightDiffuse(worldNormal, viewDir, lightDir, lightDirNorm) * falloff;
}
mat3 getLTCLightInvMat(vec2 uv)
{
	vec4 t1 = texture2DLodEXT(areaLightsLutTex1, uv, 0.0);
	#ifdef AREA_R8_G8_B8_A8_LUTS
	t1 *= vec4(1.001, 0.3239, 0.60437568, 1.0);
	t1 += vec4(0.0, -0.2976, -0.01381, 0.0);
	#endif
	return mat3(
		vec3( t1.x, 0, t1.y ),
		vec3(	0, 1,	0 ),
		vec3( t1.z, 0, t1.w )
	);
}
float calcRectLightSpecular(vec3 worldNormal, vec3 viewDir, vec2 uv) {
	mat3 mInv = getLTCLightInvMat(uv);
	return LTC_EvaluateRect( worldNormal, viewDir, vPositionW, mInv, dLTCCoords );
}
float getRectLightSpecular(vec3 worldNormal, vec3 viewDir) {
	return calcRectLightSpecular(worldNormal, viewDir, dLTCUV);
}
float calcDiskLightSpecular(vec3 worldNormal, vec3 viewDir, vec2 uv) {
	mat3 mInv = getLTCLightInvMat(uv);
	return LTC_EvaluateDisk( worldNormal, viewDir, vPositionW, mInv, dLTCCoords );
}
float getDiskLightSpecular(vec3 worldNormal, vec3 viewDir) {
	return calcDiskLightSpecular(worldNormal, viewDir, dLTCUV);
}
float getSphereLightSpecular(vec3 worldNormal, vec3 viewDir) {
	return calcDiskLightSpecular(worldNormal, viewDir, dLTCUV);
}
`;

var metalnessPS = `
#ifdef MAPFLOAT
uniform float material_metalness;
#endif
void getMetalness() {
	float metalness = 1.0;
	#ifdef MAPFLOAT
	metalness *= material_metalness;
	#endif
	#ifdef MAPTEXTURE
	metalness *= texture2DBias($SAMPLER, $UV, textureBias).$CH;
	#endif
	#ifdef MAPVERTEX
	metalness *= saturate(vVertexColor.$VC);
	#endif
	dMetalness = metalness;
}
`;

var msdfPS = `
uniform sampler2D texture_msdfMap;
#ifdef GL_OES_standard_derivatives
#define USE_FWIDTH
#endif
#ifdef GL2
#define USE_FWIDTH
#endif
float median(float r, float g, float b) {
	return max(min(r, g), min(max(r, g), b));
}
float map (float min, float max, float v) {
	return (v - min) / (max - min);
}
uniform float font_sdfIntensity;
uniform float font_pxrange;
uniform float font_textureWidth;
#ifdef UNIFORM_TEXT_PARAMETERS
uniform vec4 outline_color;
uniform float outline_thickness;
uniform vec4 shadow_color;
uniform vec2 shadow_offset;
#else
varying vec4 outline_color;
varying float outline_thickness;
varying vec4 shadow_color;
varying vec2 shadow_offset;
#endif
vec4 applyMsdf(vec4 color) {
	vec3 tsample = texture2D(texture_msdfMap, vUv0).rgb;
	vec2 uvShdw = vUv0 - shadow_offset;
	vec3 ssample = texture2D(texture_msdfMap, uvShdw).rgb;
	float sigDist = median(tsample.r, tsample.g, tsample.b);
	float sigDistShdw = median(ssample.r, ssample.g, ssample.b);
	float smoothingMax = 0.2;
	#ifdef USE_FWIDTH
	vec2 w = fwidth(vUv0);
	float smoothing = clamp(w.x * font_textureWidth / font_pxrange, 0.0, smoothingMax);
	#else
	float font_size = 16.0;
	float smoothing = clamp(font_pxrange / font_size, 0.0, smoothingMax);
	#endif
	float mapMin = 0.05;
	float mapMax = clamp(1.0 - font_sdfIntensity, mapMin, 1.0);
	float sigDistInner = map(mapMin, mapMax, sigDist);
	float sigDistOutline = map(mapMin, mapMax, sigDist + outline_thickness);
	sigDistShdw = map(mapMin, mapMax, sigDistShdw + outline_thickness);
	float center = 0.5;
	float inside = smoothstep(center-smoothing, center+smoothing, sigDistInner);
	float outline = smoothstep(center-smoothing, center+smoothing, sigDistOutline);
	float shadow = smoothstep(center-smoothing, center+smoothing, sigDistShdw);
	vec4 tcolor = (outline > inside) ? outline * vec4(outline_color.a * outline_color.rgb, outline_color.a) : vec4(0.0);
	tcolor = mix(tcolor, color, inside);
	vec4 scolor = (shadow > outline) ? shadow * vec4(shadow_color.a * shadow_color.rgb, shadow_color.a) : tcolor;
	tcolor = mix(scolor, tcolor, outline);
	
	return tcolor;
}
`;

var metalnessModulatePS = `
vec3 getSpecularModulate(in vec3 specularity, in vec3 albedo, in float metalness, in float f0) {
	vec3 dielectricF0 = f0 * specularity;
	return mix(dielectricF0, albedo, metalness);
}
vec3 getAlbedoModulate(in vec3 albedo, in float metalness) {
	return albedo * (1.0 - metalness);
}
`;

var msdfVS = `
attribute vec3 vertex_outlineParameters;
attribute vec3 vertex_shadowParameters;
varying vec4 outline_color;
varying float outline_thickness;
varying vec4 shadow_color;
varying vec2 shadow_offset;
void unpackMsdfParams() {
	vec3 little = mod(vertex_outlineParameters, 256.);
	vec3 big = (vertex_outlineParameters - little) / 256.;
	outline_color.rb = little.xy / 255.;
	outline_color.ga = big.xy / 255.;
	outline_thickness = little.z / 255. * 0.2;
	little = mod(vertex_shadowParameters, 256.);
	big = (vertex_shadowParameters - little) / 256.;
	shadow_color.rb = little.xy / 255.;
	shadow_color.ga = big.xy / 255.;
	shadow_offset = (vec2(little.z, big.z) / 127. - 1.) * 0.005;
}
`;

var normalVS = `
#ifdef MORPHING_TEXTURE_BASED_NORMAL
uniform highp sampler2D morphNormalTex;
#endif
vec3 getNormal() {
	#ifdef SKIN
	dNormalMatrix = mat3(dModelMatrix[0].xyz, dModelMatrix[1].xyz, dModelMatrix[2].xyz);
	#elif defined(INSTANCING)
	dNormalMatrix = mat3(instance_line1.xyz, instance_line2.xyz, instance_line3.xyz);
	#else
	dNormalMatrix = matrix_normal;
	#endif
	vec3 tempNormal = vertex_normal;
	#ifdef MORPHING
	#ifdef MORPHING_NRM03
	tempNormal += morph_weights_a[0] * morph_nrm0;
	tempNormal += morph_weights_a[1] * morph_nrm1;
	tempNormal += morph_weights_a[2] * morph_nrm2;
	tempNormal += morph_weights_a[3] * morph_nrm3;
	#endif
	#ifdef MORPHING_NRM47
	tempNormal += morph_weights_b[0] * morph_nrm4;
	tempNormal += morph_weights_b[1] * morph_nrm5;
	tempNormal += morph_weights_b[2] * morph_nrm6;
	tempNormal += morph_weights_b[3] * morph_nrm7;
	#endif
	#endif
	#ifdef MORPHING_TEXTURE_BASED_NORMAL
		#ifdef WEBGPU
			ivec2 morphUV = getTextureMorphCoords();
			vec3 morphNormal = texelFetch(morphNormalTex, ivec2(morphUV), 0).xyz;
		#else
			vec2 morphUV = getTextureMorphCoords();
			vec3 morphNormal = texture2D(morphNormalTex, morphUV).xyz;
		#endif
	tempNormal += morphNormal;
	#endif
	return normalize(dNormalMatrix * tempNormal);
}
`;

var normalDetailMapPS = `
#ifdef MAPTEXTURE
uniform float material_normalDetailMapBumpiness;
vec3 blendNormals(vec3 n1, vec3 n2) {
	n1 += vec3(0, 0, 1);
	n2 *= vec3(-1, -1, 1);
	return n1 * dot(n1, n2) / n1.z - n2;
}
#endif
vec3 addNormalDetail(vec3 normalMap) {
#ifdef MAPTEXTURE
	vec3 normalDetailMap = unpackNormal(texture2DBias($SAMPLER, $UV, textureBias));
	normalDetailMap = mix(vec3(0.0, 0.0, 1.0), normalDetailMap, material_normalDetailMapBumpiness);
	return blendNormals(normalMap, normalDetailMap);
#else
	return normalMap;
#endif
}
`;

var normalInstancedVS = `
vec3 getNormal() {
	dNormalMatrix = mat3(instance_line1.xyz, instance_line2.xyz, instance_line3.xyz);
	return normalize(dNormalMatrix * vertex_normal);
}
`;

var normalMapPS = `
#ifdef MAPTEXTURE
uniform float material_bumpiness;
#endif
void getNormal() {
#ifdef MAPTEXTURE
	vec3 normalMap = unpackNormal(texture2DBias($SAMPLER, $UV, textureBias));
	normalMap = mix(vec3(0.0, 0.0, 1.0), normalMap, material_bumpiness);
	dNormalW = normalize(dTBN * addNormalDetail(normalMap));
#else
	dNormalW = dVertexNormalW;
#endif
}
`;

var normalSkinnedVS = `
vec3 getNormal() {
	dNormalMatrix = mat3(dModelMatrix[0].xyz, dModelMatrix[1].xyz, dModelMatrix[2].xyz);
	return normalize(dNormalMatrix * vertex_normal);
}
`;

var normalXYPS = `
vec3 unpackNormal(vec4 nmap) {
	vec3 normal;
	normal.xy = nmap.wy * 2.0 - 1.0;
	normal.z = sqrt(1.0 - saturate(dot(normal.xy, normal.xy)));
	return normal;
}
`;

var normalXYZPS = `
vec3 unpackNormal(vec4 nmap) {
	return nmap.xyz * 2.0 - 1.0;
}
`;

var opacityPS = `
#ifdef MAPFLOAT
uniform float material_opacity;
#endif
void getOpacity() {
	dAlpha = 1.0;
	#ifdef MAPFLOAT
	dAlpha *= material_opacity;
	#endif
	#ifdef MAPTEXTURE
	dAlpha *= texture2DBias($SAMPLER, $UV, textureBias).$CH;
	#endif
	#ifdef MAPVERTEX
	dAlpha *= clamp(vVertexColor.$VC, 0.0, 1.0);
	#endif
}
`;

var opacityDitherPS = `
uniform vec4 blueNoiseJitter;
#ifndef DITHER_BAYER8
	uniform sampler2D blueNoiseTex32;
#endif
void opacityDither(float alpha, float id) {
	#ifdef DITHER_BAYER8
		float noise = bayer8(floor(mod(gl_FragCoord.xy + blueNoiseJitter.xy + id, 8.0))) / 64.0;
	#else
		vec2 uv = fract(gl_FragCoord.xy / 32.0 + blueNoiseJitter.xy + id);
		float noise = texture2DLodEXT(blueNoiseTex32, uv, 0.0).y;
	#endif
	if (alpha < noise)
		discard;
}
`;

var outputPS = `
`;

var outputAlphaPS = `
gl_FragColor.a = litArgs_opacity;
`;

var outputAlphaOpaquePS = `
	gl_FragColor.a = 1.0;
`;

var outputAlphaPremulPS = `
gl_FragColor.rgb *= litArgs_opacity;
gl_FragColor.a = litArgs_opacity;
`;

var outputTex2DPS = `
varying vec2 vUv0;
uniform sampler2D source;
void main(void) {
	gl_FragColor = texture2D(source, vUv0);
}
`;

var packDepthPS = `
vec4 packFloat(float depth) {
	const vec4 bit_shift = vec4(256.0 * 256.0 * 256.0, 256.0 * 256.0, 256.0, 1.0);
	const vec4 bit_mask  = vec4(0.0, 1.0 / 256.0, 1.0 / 256.0, 1.0 / 256.0);
	vec4 res = mod(depth * bit_shift * vec4(255), vec4(256) ) / vec4(255);
	res -= res.xxyz * bit_mask;
	return res;
}
`;

var sheenPS = `
#ifdef MAPCOLOR
uniform vec3 material_sheen;
#endif
void getSheen() {
	vec3 sheenColor = vec3(1, 1, 1);
	#ifdef MAPCOLOR
	sheenColor *= material_sheen;
	#endif
	#ifdef MAPTEXTURE
	sheenColor *= $DECODE(texture2DBias($SAMPLER, $UV, textureBias)).$CH;
	#endif
	#ifdef MAPVERTEX
	sheenColor *= saturate(vVertexColor.$VC);
	#endif
	sSpecularity = sheenColor;
}
`;

var sheenGlossPS = `
#ifdef MAPFLOAT
uniform float material_sheenGloss;
#endif
void getSheenGlossiness() {
	float sheenGlossiness = 1.0;
	#ifdef MAPFLOAT
	sheenGlossiness *= material_sheenGloss;
	#endif
	#ifdef MAPTEXTURE
	sheenGlossiness *= texture2DBias($SAMPLER, $UV, textureBias).$CH;
	#endif
	#ifdef MAPVERTEX
	sheenGlossiness *= saturate(vVertexColor.$VC);
	#endif
	#ifdef MAPINVERT
	sheenGlossiness = 1.0 - sheenGlossiness;
	#endif
	sheenGlossiness += 0.0000001;
	sGlossiness = sheenGlossiness;
}
`;

var parallaxPS = `
uniform float material_heightMapFactor;
void getParallax() {
	float parallaxScale = material_heightMapFactor;
	float height = texture2DBias($SAMPLER, $UV, textureBias).$CH;
	height = height * parallaxScale - parallaxScale*0.5;
	vec3 viewDirT = dViewDirW * dTBN;
	viewDirT.z += 0.42;
	dUvOffset = height * (viewDirT.xy / viewDirT.z);
}
`;

var particlePS = `
varying vec4 texCoordsAlphaLife;
uniform sampler2D colorMap;
uniform sampler2D colorParam;
uniform float graphSampleSize;
uniform float graphNumSamples;
#ifndef CAMERAPLANES
#define CAMERAPLANES
uniform vec4 camera_params;
#endif
uniform float softening;
uniform float colorMult;
float saturate(float x) {
	return clamp(x, 0.0, 1.0);
}
#ifndef UNPACKFLOAT
#define UNPACKFLOAT
float unpackFloat(vec4 rgbaDepth) {
	const vec4 bitShift = vec4(1.0 / (256.0 * 256.0 * 256.0), 1.0 / (256.0 * 256.0), 1.0 / 256.0, 1.0);
	float depth = dot(rgbaDepth, bitShift);
	return depth;
}
#endif
void main(void) {
	vec4 tex  = gammaCorrectInput(texture2D(colorMap, vec2(texCoordsAlphaLife.x, 1.0 - texCoordsAlphaLife.y)));
	vec4 ramp = gammaCorrectInput(texture2D(colorParam, vec2(texCoordsAlphaLife.w, 0.0)));
	ramp.rgb *= colorMult;
	ramp.a += texCoordsAlphaLife.z;
	vec3 rgb = tex.rgb * ramp.rgb;
	float a  = tex.a * ramp.a;
`;

var particleVS = `
vec3 unpack3NFloats(float src) {
	float r = fract(src);
	float g = fract(src * 256.0);
	float b = fract(src * 65536.0);
	return vec3(r, g, b);
}
float saturate(float x) {
	return clamp(x, 0.0, 1.0);
}
vec4 tex1Dlod_lerp(highp sampler2D tex, vec2 tc) {
	return mix( texture2D(tex,tc), texture2D(tex,tc + graphSampleSize), fract(tc.x*graphNumSamples) );
}
vec4 tex1Dlod_lerp(highp sampler2D tex, vec2 tc, out vec3 w) {
	vec4 a = texture2D(tex,tc);
	vec4 b = texture2D(tex,tc + graphSampleSize);
	float c = fract(tc.x*graphNumSamples);
	vec3 unpackedA = unpack3NFloats(a.w);
	vec3 unpackedB = unpack3NFloats(b.w);
	w = mix(unpackedA, unpackedB, c);
	return mix(a, b, c);
}
vec2 rotate(vec2 quadXY, float pRotation, out mat2 rotMatrix) {
	float c = cos(pRotation);
	float s = sin(pRotation);
	mat2 m = mat2(c, -s, s, c);
	rotMatrix = m;
	return m * quadXY;
}
vec3 billboard(vec3 InstanceCoords, vec2 quadXY) {
	#ifdef SCREEN_SPACE
		vec3 pos = vec3(-1, 0, 0) * quadXY.x + vec3(0, -1, 0) * quadXY.y;
	#else
		vec3 pos = -matrix_viewInverse[0].xyz * quadXY.x + -matrix_viewInverse[1].xyz * quadXY.y;
	#endif
	return pos;
}
vec3 customFace(vec3 InstanceCoords, vec2 quadXY) {
	vec3 pos = faceTangent * quadXY.x + faceBinorm * quadXY.y;
	return pos;
}
vec2 safeNormalize(vec2 v) {
	float l = length(v);
	return (l > 1e-06) ? v / l : v;
}
void main(void) {
	vec3 meshLocalPos = particle_vertexData.xyz;
	float id = floor(particle_vertexData.w);
	float rndFactor = fract(sin(id + 1.0 + seed));
	vec3 rndFactor3 = vec3(rndFactor, fract(rndFactor*10.0), fract(rndFactor*100.0));
	float uv = id / numParticlesPot;
	readInput(uv);
#ifdef LOCAL_SPACE
	inVel = mat3(matrix_model) * inVel;
#endif
	vec2 velocityV = safeNormalize((mat3(matrix_view) * inVel).xy);
	float particleLifetime = lifetime;
	if (inLife <= 0.0 || inLife > particleLifetime || !inShow) meshLocalPos = vec3(0.0);
	vec2 quadXY = meshLocalPos.xy;
	float nlife = clamp(inLife / particleLifetime, 0.0, 1.0);
	vec3 paramDiv;
	vec4 params = tex1Dlod_lerp(internalTex2, vec2(nlife, 0), paramDiv);
	float scale = params.y;
	float scaleDiv = paramDiv.x;
	float alphaDiv = paramDiv.z;
	scale += (scaleDiv * 2.0 - 1.0) * scaleDivMult * fract(rndFactor*10000.0);
#ifndef USE_MESH
	texCoordsAlphaLife = vec4(quadXY * -0.5 + 0.5, (alphaDiv * 2.0 - 1.0) * alphaDivMult * fract(rndFactor*1000.0), nlife);
#else
	texCoordsAlphaLife = vec4(particle_uv, (alphaDiv * 2.0 - 1.0) * alphaDivMult * fract(rndFactor*1000.0), nlife);
#endif
	vec3 particlePos = inPos;
	vec3 particlePosMoved = vec3(0.0);
	mat2 rotMatrix;
`;

var particleAnimFrameClampVS = `
	float animFrame = min(floor(texCoordsAlphaLife.w * animTexParams.y) + animTexParams.x, animTexParams.z);
`;

var particleAnimFrameLoopVS = `
	float animFrame = floor(mod(texCoordsAlphaLife.w * animTexParams.y + animTexParams.x, animTexParams.z + 1.0));
`;

var particleAnimTexVS = `
	float animationIndex;
	if (animTexIndexParams.y == 1.0) {
		animationIndex = floor((animTexParams.w + 1.0) * rndFactor3.z) * (animTexParams.z + 1.0);
	} else {
		animationIndex = animTexIndexParams.x * (animTexParams.z + 1.0);
	}
	float atlasX = (animationIndex + animFrame) * animTexTilesParams.x;
	float atlasY = 1.0 - floor(atlasX + 1.0) * animTexTilesParams.y;
	atlasX = fract(atlasX);
	texCoordsAlphaLife.xy *= animTexTilesParams.xy;
	texCoordsAlphaLife.xy += vec2(atlasX, atlasY);
`;

var particleInputFloatPS = `
void readInput(float uv) {
	vec4 tex = texture2D(particleTexIN, vec2(uv, 0.25));
	vec4 tex2 = texture2D(particleTexIN, vec2(uv, 0.75));
	inPos = tex.xyz;
	inVel = tex2.xyz;
	inAngle = (tex.w < 0.0? -tex.w : tex.w) - 1000.0;
	inShow = tex.w >= 0.0;
	inLife = tex2.w;
}
`;

var particleInputRgba8PS = `
#define PI2 6.283185307179586
uniform vec3 inBoundsSize;
uniform vec3 inBoundsCenter;
uniform float maxVel;
float decodeFloatRG(vec2 rg) {
	return rg.y*(1.0/255.0) + rg.x;
}
float decodeFloatRGBA( vec4 rgba ) {
	return dot( rgba, vec4(1.0, 1.0/255.0, 1.0/65025.0, 1.0/160581375.0) );
}
void readInput(float uv) {
	vec4 tex0 = texture2D(particleTexIN, vec2(uv, 0.125));
	vec4 tex1 = texture2D(particleTexIN, vec2(uv, 0.375));
	vec4 tex2 = texture2D(particleTexIN, vec2(uv, 0.625));
	vec4 tex3 = texture2D(particleTexIN, vec2(uv, 0.875));
	inPos = vec3(decodeFloatRG(tex0.rg), decodeFloatRG(tex0.ba), decodeFloatRG(tex1.rg));
	inPos = (inPos - vec3(0.5)) * inBoundsSize + inBoundsCenter;
	inVel = tex2.xyz;
	inVel = (inVel - vec3(0.5)) * maxVel;
	inAngle = decodeFloatRG(tex1.ba) * PI2;
	inShow = tex2.a > 0.5;
	inLife = decodeFloatRGBA(tex3);
	float maxNegLife = max(lifetime, (numParticles - 1.0) * (rate+rateDiv));
	float maxPosLife = lifetime+1.0;
	inLife = inLife * (maxNegLife + maxPosLife) - maxNegLife;
}
`;

var particleOutputFloatPS = `
void writeOutput() {
	if (gl_FragCoord.y<1.0) {
		gl_FragColor = vec4(outPos, (outAngle + 1000.0) * visMode);
	} else {
		gl_FragColor = vec4(outVel, outLife);
	}
}
`;

var particleOutputRgba8PS = `
uniform vec3 outBoundsMul;
uniform vec3 outBoundsAdd;
vec2 encodeFloatRG( float v ) {
	vec2 enc = vec2(1.0, 255.0) * v;
	enc = fract(enc);
	enc -= enc.yy * vec2(1.0/255.0, 1.0/255.0);
	return enc;
}
vec4 encodeFloatRGBA( float v ) {
	vec4 enc = vec4(1.0, 255.0, 65025.0, 160581375.0) * v;
	enc = fract(enc);
	enc -= enc.yzww * vec4(1.0/255.0,1.0/255.0,1.0/255.0,0.0);
	return enc;
}
void writeOutput() {
	outPos = outPos * outBoundsMul + outBoundsAdd;
	outAngle = fract(outAngle / PI2);
	outVel = (outVel / maxVel) + vec3(0.5);
	float maxNegLife = max(lifetime, (numParticles - 1.0) * (rate+rateDiv));
	float maxPosLife = lifetime+1.0;
	outLife = (outLife + maxNegLife) / (maxNegLife + maxPosLife);
	if (gl_FragCoord.y < 1.0) {
		gl_FragColor = vec4(encodeFloatRG(outPos.x), encodeFloatRG(outPos.y));
	} else if (gl_FragCoord.y < 2.0) {
		gl_FragColor = vec4(encodeFloatRG(outPos.z), encodeFloatRG(outAngle));
	} else if (gl_FragCoord.y < 3.0) {
		gl_FragColor = vec4(outVel, visMode*0.5+0.5);
	} else {
		gl_FragColor = encodeFloatRGBA(outLife);
	}
}
`;

var particleUpdaterAABBPS = `
uniform mat3 spawnBounds;
uniform vec3 spawnPosInnerRatio;
vec3 calcSpawnPosition(vec3 inBounds, float rndFactor) {
	vec3 pos = inBounds - vec3(0.5);
	vec3 posAbs = abs(pos);
	vec3 maxPos = vec3(max(posAbs.x, max(posAbs.y, posAbs.z)));
	vec3 edge = maxPos + (vec3(0.5) - maxPos) * spawnPosInnerRatio;
	pos.x = edge.x * (maxPos.x == posAbs.x ? sign(pos.x) : 2.0 * pos.x);
	pos.y = edge.y * (maxPos.y == posAbs.y ? sign(pos.y) : 2.0 * pos.y);
	pos.z = edge.z * (maxPos.z == posAbs.z ? sign(pos.z) : 2.0 * pos.z);
#ifndef LOCAL_SPACE
	return emitterPos + spawnBounds * pos;
#else
	return spawnBounds * pos;
#endif
}
void addInitialVelocity(inout vec3 localVelocity, vec3 inBounds) {
	localVelocity -= vec3(0, 0, initialVelocity);
}
`;

var particleUpdaterEndPS = `
	writeOutput();
}
`;

var particleUpdaterInitPS = `
varying vec2 vUv0;
uniform highp sampler2D particleTexIN;
uniform highp sampler2D internalTex0;
uniform highp sampler2D internalTex1;
uniform highp sampler2D internalTex2;
uniform highp sampler2D internalTex3;
uniform mat3 emitterMatrix, emitterMatrixInv;
uniform vec3 emitterScale;
uniform vec3 emitterPos, frameRandom, localVelocityDivMult, velocityDivMult;
uniform float delta, rate, rateDiv, lifetime, numParticles, rotSpeedDivMult, radialSpeedDivMult, seed;
uniform float startAngle, startAngle2;
uniform float initialVelocity;
uniform float graphSampleSize;
uniform float graphNumSamples;
vec3 inPos;
vec3 inVel;
float inAngle;
bool inShow;
float inLife;
float visMode;
vec3 outPos;
vec3 outVel;
float outAngle;
bool outShow;
float outLife;
`;

var particleUpdaterNoRespawnPS = `
	if (outLife >= lifetime) {
		outLife -= max(lifetime, (numParticles - 1.0) * particleRate);
		visMode = -1.0;
	}
`;

var particleUpdaterOnStopPS = `
	visMode = outLife < 0.0? -1.0: visMode;
`;

var particleUpdaterRespawnPS = `
	if (outLife >= lifetime) {
		outLife -= max(lifetime, (numParticles - 1.0) * particleRate);
		visMode = 1.0;
	}
	visMode = outLife < 0.0? 1.0: visMode;
`;

var particleUpdaterSpherePS = `
uniform float spawnBoundsSphere;
uniform float spawnBoundsSphereInnerRatio;
vec3 calcSpawnPosition(vec3 inBounds, float rndFactor) {
	float rnd4 = fract(rndFactor * 1000.0);
	vec3 norm = normalize(inBounds.xyz - vec3(0.5));
	float r = rnd4 * (1.0 - spawnBoundsSphereInnerRatio) + spawnBoundsSphereInnerRatio;
#ifndef LOCAL_SPACE
	return emitterPos + norm * r * spawnBoundsSphere;
#else
	return norm * r * spawnBoundsSphere;
#endif
}
void addInitialVelocity(inout vec3 localVelocity, vec3 inBounds) {
	localVelocity += normalize(inBounds - vec3(0.5)) * initialVelocity;
}
`;

var particleUpdaterStartPS = `
float saturate(float x) {
	return clamp(x, 0.0, 1.0);
}
vec3 unpack3NFloats(float src) {
	float r = fract(src);
	float g = fract(src * 256.0);
	float b = fract(src * 65536.0);
	return vec3(r, g, b);
}
vec3 tex1Dlod_lerp(highp sampler2D tex, vec2 tc, out vec3 w) {
	vec4 a = texture2D(tex, tc);
	vec4 b = texture2D(tex, tc + graphSampleSize);
	float c = fract(tc.x * graphNumSamples);
	vec3 unpackedA = unpack3NFloats(a.w);
	vec3 unpackedB = unpack3NFloats(b.w);
	w = mix(unpackedA, unpackedB, c);
	return mix(a.xyz, b.xyz, c);
}
#define HASHSCALE4 vec4(1031, .1030, .0973, .1099)
vec4 hash41(float p) {
	vec4 p4 = fract(vec4(p) * HASHSCALE4);
	p4 += dot(p4, p4.wzxy+19.19);
	return fract(vec4((p4.x + p4.y)*p4.z, (p4.x + p4.z)*p4.y, (p4.y + p4.z)*p4.w, (p4.z + p4.w)*p4.x));
}
void main(void) {
	if (gl_FragCoord.x > numParticles) discard;
	readInput(vUv0.x);
	visMode = inShow? 1.0 : -1.0;
	vec4 rndFactor = hash41(gl_FragCoord.x + seed);
	float particleRate = rate + rateDiv * rndFactor.x;
	outLife = inLife + delta;
	float nlife = clamp(outLife / lifetime, 0.0, 1.0);
	vec3 localVelocityDiv;
	vec3 velocityDiv;
	vec3 paramDiv;
	vec3 localVelocity = tex1Dlod_lerp(internalTex0, vec2(nlife, 0), localVelocityDiv);
	vec3 velocity =	  tex1Dlod_lerp(internalTex1, vec2(nlife, 0), velocityDiv);
	vec3 params =		tex1Dlod_lerp(internalTex2, vec2(nlife, 0), paramDiv);
	float rotSpeed = params.x;
	float rotSpeedDiv = paramDiv.y;
	vec3 radialParams = tex1Dlod_lerp(internalTex3, vec2(nlife, 0), paramDiv);
	float radialSpeed = radialParams.x;
	float radialSpeedDiv = radialParams.y;
	bool respawn = inLife <= 0.0 || outLife >= lifetime;
	inPos = respawn ? calcSpawnPosition(rndFactor.xyz, rndFactor.x) : inPos;
	inAngle = respawn ? mix(startAngle, startAngle2, rndFactor.x) : inAngle;
#ifndef LOCAL_SPACE
	vec3 radialVel = inPos - emitterPos;
#else
	vec3 radialVel = inPos;
#endif
	radialVel = (dot(radialVel, radialVel) > 1.0E-8) ? radialSpeed * normalize(radialVel) : vec3(0.0);
	radialVel += (radialSpeedDiv * vec3(2.0) - vec3(1.0)) * radialSpeedDivMult * rndFactor.xyz;
	localVelocity +=	(localVelocityDiv * vec3(2.0) - vec3(1.0)) * localVelocityDivMult * rndFactor.xyz;
	velocity +=		 (velocityDiv * vec3(2.0) - vec3(1.0)) * velocityDivMult * rndFactor.xyz;
	rotSpeed +=		 (rotSpeedDiv * 2.0 - 1.0) * rotSpeedDivMult * rndFactor.y;
	addInitialVelocity(localVelocity, rndFactor.xyz);
#ifndef LOCAL_SPACE
	outVel = emitterMatrix * localVelocity + (radialVel + velocity) * emitterScale;
#else
	outVel = (localVelocity + radialVel) / emitterScale + emitterMatrixInv * velocity;
#endif
	outPos = inPos + outVel * delta;
	outAngle = inAngle + rotSpeed * delta;
`;

var particle_billboardVS = `
	quadXY = rotate(quadXY, inAngle, rotMatrix);
	vec3 localPos = billboard(particlePos, quadXY);
`;

var particle_blendAddPS = `
	dBlendModeFogFactor = 0.0;
	rgb *= saturate(gammaCorrectInput(max(a, 0.0)));
	if ((rgb.r + rgb.g + rgb.b) < 0.000001) discard;
`;

var particle_blendMultiplyPS = `
	rgb = mix(vec3(1.0), rgb, vec3(a));
	if (rgb.r + rgb.g + rgb.b > 2.99) discard;
`;

var particle_blendNormalPS = `
	if (a < 0.01) discard;
`;

var particle_cpuVS = `
attribute vec4 particle_vertexData;
attribute vec4 particle_vertexData2;
attribute vec4 particle_vertexData3;
attribute float particle_vertexData4;
#ifndef USE_MESH
attribute vec2 particle_vertexData5;
#else
attribute vec4 particle_vertexData5;
#endif
uniform mat4 matrix_viewProjection;
uniform mat4 matrix_model;
#ifndef VIEWMATRIX
#define VIEWMATRIX
uniform mat4 matrix_view;
#endif
uniform mat3 matrix_normal;
uniform mat4 matrix_viewInverse;
uniform float numParticles;
uniform float lifetime;
uniform float stretch;
uniform float seed;
uniform vec3 wrapBounds;
uniform vec3 emitterScale;
uniform vec3 faceTangent;
uniform vec3 faceBinorm;
uniform highp sampler2D internalTex0;
uniform highp sampler2D internalTex1;
uniform highp sampler2D internalTex2;
uniform vec3 emitterPos;
varying vec4 texCoordsAlphaLife;
vec2 rotate(vec2 quadXY, float pRotation, out mat2 rotMatrix)
{
	float c = cos(pRotation);
	float s = sin(pRotation);
	mat2 m = mat2(c, -s, s, c);
	rotMatrix = m;
	return m * quadXY;
}
vec3 billboard(vec3 InstanceCoords, vec2 quadXY)
{
	vec3 pos = -matrix_viewInverse[0].xyz * quadXY.x + -matrix_viewInverse[1].xyz * quadXY.y;
	return pos;
}
vec3 customFace(vec3 InstanceCoords, vec2 quadXY)
{
	vec3 pos = faceTangent * quadXY.x + faceBinorm * quadXY.y;
	return pos;
}
void main(void)
{
	vec3 particlePos = particle_vertexData.xyz;
	vec3 inPos = particlePos;
	vec3 vertPos = particle_vertexData3.xyz;
	vec3 inVel = vec3(particle_vertexData2.w, particle_vertexData3.w, particle_vertexData5.x);
	float id = floor(particle_vertexData4);
	float rndFactor = fract(sin(id + 1.0 + seed));
	vec3 rndFactor3 = vec3(rndFactor, fract(rndFactor*10.0), fract(rndFactor*100.0));
#ifdef LOCAL_SPACE
	inVel = mat3(matrix_model) * inVel;
#endif
	vec2 velocityV = normalize((mat3(matrix_view) * inVel).xy);
	vec2 quadXY = vertPos.xy;
#ifdef USE_MESH
	texCoordsAlphaLife = vec4(particle_vertexData5.zw, particle_vertexData2.z, particle_vertexData.w);
#else
	texCoordsAlphaLife = vec4(quadXY * -0.5 + 0.5, particle_vertexData2.z, particle_vertexData.w);
#endif
	mat2 rotMatrix;
	float inAngle = particle_vertexData2.x;
	vec3 particlePosMoved = vec3(0.0);
	vec3 meshLocalPos = particle_vertexData3.xyz;
`;

var particle_cpu_endVS = `
	localPos *= particle_vertexData2.y * emitterScale;
	localPos += particlePos;
	gl_Position = matrix_viewProjection * vec4(localPos, 1.0);
`;

var particle_customFaceVS = `
	quadXY = rotate(quadXY, inAngle, rotMatrix);
	vec3 localPos = customFace(particlePos, quadXY);
`;

var particle_endPS = `
	rgb = addFog(rgb);
	rgb = toneMap(rgb);
	rgb = gammaCorrectOutput(rgb);
	gl_FragColor = vec4(rgb, a);
}
`;

var particle_endVS = `
	localPos *= scale * emitterScale;
	localPos += particlePos;
	#ifdef SCREEN_SPACE
	gl_Position = vec4(localPos.x, localPos.y, 0.0, 1.0);
	#else
	gl_Position = matrix_viewProjection * vec4(localPos.xyz, 1.0);
	#endif
`;

var particle_halflambertPS = `
	vec3 negNormal = normal*0.5+0.5;
	vec3 posNormal = -normal*0.5+0.5;
	negNormal *= negNormal;
	posNormal *= posNormal;
`;

var particle_initVS = `
attribute vec4 particle_vertexData;
#ifdef USE_MESH
attribute vec2 particle_uv;
#endif
uniform mat4 matrix_viewProjection;
uniform mat4 matrix_model;
uniform mat3 matrix_normal;
uniform mat4 matrix_viewInverse;
#ifndef VIEWMATRIX
#define VIEWMATRIX
uniform mat4 matrix_view;
#endif
uniform float numParticles, numParticlesPot;
uniform float graphSampleSize;
uniform float graphNumSamples;
uniform float stretch;
uniform vec3 wrapBounds;
uniform vec3 emitterScale, emitterPos, faceTangent, faceBinorm;
uniform float rate, rateDiv, lifetime, deltaRandomnessStatic, scaleDivMult, alphaDivMult, seed, delta;
uniform sampler2D particleTexOUT, particleTexIN;
uniform highp sampler2D internalTex0;
uniform highp sampler2D internalTex1;
uniform highp sampler2D internalTex2;
#ifndef CAMERAPLANES
#define CAMERAPLANES
uniform vec4 camera_params;
#endif
varying vec4 texCoordsAlphaLife;
vec3 inPos;
vec3 inVel;
float inAngle;
bool inShow;
float inLife;
`;

var particle_lambertPS = `
	vec3 negNormal = max(normal, vec3(0.0));
	vec3 posNormal = max(-normal, vec3(0.0));
`;

var particle_lightingPS = `
	vec3 light = negNormal.x*lightCube[0] + posNormal.x*lightCube[1] +
						negNormal.y*lightCube[2] + posNormal.y*lightCube[3] +
						negNormal.z*lightCube[4] + posNormal.z*lightCube[5];
	rgb *= light;
`;

var particle_localShiftVS = `
	particlePos = (matrix_model * vec4(particlePos, 1.0)).xyz;
`;

var particle_meshVS = `
	vec3 localPos = meshLocalPos;
	localPos.xy = rotate(localPos.xy, inAngle, rotMatrix);
	localPos.yz = rotate(localPos.yz, inAngle, rotMatrix);
	billboard(particlePos, quadXY);
`;

var particle_normalVS = `
	Normal = normalize(localPos + matrix_viewInverse[2].xyz);
`;

var particle_normalMapPS = `
	vec3 normalMap = normalize(texture2D(normalMap, vec2(texCoordsAlphaLife.x, 1.0 - texCoordsAlphaLife.y)).xyz * 2.0 - 1.0);
	vec3 normal = ParticleMat * normalMap;
`;

var particle_pointAlongVS = `
	inAngle = atan(velocityV.x, velocityV.y);
`;

var particle_softPS = `
	float depth = getLinearScreenDepth();
	float particleDepth = vDepth;
	float depthDiff = saturate(abs(particleDepth - depth) * softening);
	a *= depthDiff;
`;

var particle_softVS = `
	vDepth = getLinearDepth(localPos);
`;

var particle_stretchVS = `
	vec3 moveDir = inVel * stretch;
	vec3 posPrev = particlePos - moveDir;
	posPrev += particlePosMoved;
	vec2 centerToVertexV = normalize((mat3(matrix_view) * localPos).xy);
	float interpolation = dot(-velocityV, centerToVertexV) * 0.5 + 0.5;
	particlePos = mix(particlePos, posPrev, interpolation);
`;

var particle_TBNVS = `
	mat3 rot3 = mat3(rotMatrix[0][0], rotMatrix[0][1], 0.0, rotMatrix[1][0], rotMatrix[1][1], 0.0, 0.0, 0.0, 1.0);
	ParticleMat = mat3(-matrix_viewInverse[0].xyz, -matrix_viewInverse[1].xyz, matrix_viewInverse[2].xyz) * rot3;
`;

var particle_wrapVS = `
	vec3 origParticlePos = particlePos;
	particlePos -= matrix_model[3].xyz;
	particlePos = mod(particlePos, wrapBounds) - wrapBounds * 0.5;
	particlePos += matrix_model[3].xyz;
	particlePosMoved = particlePos - origParticlePos;
`;

var reflDirPS = `
void getReflDir(vec3 worldNormal, vec3 viewDir, float gloss, mat3 tbn) {
	dReflDirW = normalize(-reflect(viewDir, worldNormal));
}
`;

var reflDirAnisoPS = `
void getReflDir(vec3 worldNormal, vec3 viewDir, float gloss, mat3 tbn) {
	float roughness = sqrt(1.0 - min(gloss, 1.0));
	float anisotropy = material_anisotropy * roughness;
	vec3 anisotropicDirection = anisotropy >= 0.0 ? tbn[1] : tbn[0];
	vec3 anisotropicTangent = cross(anisotropicDirection, viewDir);
	vec3 anisotropicNormal = cross(anisotropicTangent, anisotropicDirection);
	vec3 bentNormal = normalize(mix(normalize(worldNormal), normalize(anisotropicNormal), anisotropy));
	dReflDirW = reflect(-viewDir, bentNormal);
}
`;

var reflectionCCPS = `
#ifdef LIT_CLEARCOAT
void addReflectionCC(vec3 reflDir, float gloss) {
	ccReflection += calcReflection(reflDir, gloss);
}
#endif
`;

var reflectionCubePS = `
uniform samplerCube texture_cubeMap;
uniform float material_reflectivity;
vec3 calcReflection(vec3 reflDir, float gloss) {
	vec3 lookupVec = fixSeams(cubeMapProject(reflDir));
	lookupVec.x *= -1.0;
	return $DECODE(textureCube(texture_cubeMap, lookupVec));
}
void addReflection(vec3 reflDir, float gloss) {   
	dReflection += vec4(calcReflection(reflDir, gloss), material_reflectivity);
}
`;

var reflectionEnvHQPS = `
#ifndef ENV_ATLAS
#define ENV_ATLAS
uniform sampler2D texture_envAtlas;
#endif
uniform samplerCube texture_cubeMap;
uniform float material_reflectivity;
vec3 calcReflection(vec3 reflDir, float gloss) {
	vec3 dir = cubeMapProject(reflDir) * vec3(-1.0, 1.0, 1.0);
	vec2 uv = toSphericalUv(dir);
	float level = saturate(1.0 - gloss) * 5.0;
	float ilevel = floor(level);
	float flevel = level - ilevel;
	vec3 sharp = $DECODE_CUBEMAP(textureCube(texture_cubeMap, fixSeams(dir)));
	vec3 roughA = $DECODE(texture2D(texture_envAtlas, mapRoughnessUv(uv, ilevel)));
	vec3 roughB = $DECODE(texture2D(texture_envAtlas, mapRoughnessUv(uv, ilevel + 1.0)));
	return processEnvironment(mix(sharp, mix(roughA, roughB, flevel), min(level, 1.0)));
}
void addReflection(vec3 reflDir, float gloss) {   
	dReflection += vec4(calcReflection(reflDir, gloss), material_reflectivity);
}
`;

var reflectionEnvPS = `
#ifndef ENV_ATLAS
#define ENV_ATLAS
uniform sampler2D texture_envAtlas;
#endif
uniform float material_reflectivity;
float shinyMipLevel(vec2 uv) {
	vec2 dx = dFdx(uv);
	vec2 dy = dFdy(uv);
	vec2 uv2 = vec2(fract(uv.x + 0.5), uv.y);
	vec2 dx2 = dFdx(uv2);
	vec2 dy2 = dFdy(uv2);
	float maxd = min(max(dot(dx, dx), dot(dy, dy)), max(dot(dx2, dx2), dot(dy2, dy2)));
	return clamp(0.5 * log2(maxd) - 1.0 + textureBias, 0.0, 5.0);
}
vec3 calcReflection(vec3 reflDir, float gloss) {
	vec3 dir = cubeMapProject(reflDir) * vec3(-1.0, 1.0, 1.0);
	vec2 uv = toSphericalUv(dir);
	float level = saturate(1.0 - gloss) * 5.0;
	float ilevel = floor(level);
	float level2 = shinyMipLevel(uv * atlasSize);
	float ilevel2 = floor(level2);
	vec2 uv0, uv1;
	float weight;
	if (ilevel == 0.0) {
		uv0 = mapShinyUv(uv, ilevel2);
		uv1 = mapShinyUv(uv, ilevel2 + 1.0);
		weight = level2 - ilevel2;
	} else {
		uv0 = uv1 = mapRoughnessUv(uv, ilevel);
		weight = 0.0;
	}
	vec3 linearA = $DECODE(texture2D(texture_envAtlas, uv0));
	vec3 linearB = $DECODE(texture2D(texture_envAtlas, uv1));
	vec3 linear0 = mix(linearA, linearB, weight);
	vec3 linear1 = $DECODE(texture2D(texture_envAtlas, mapRoughnessUv(uv, ilevel + 1.0)));
	return processEnvironment(mix(linear0, linear1, level - ilevel));
}
void addReflection(vec3 reflDir, float gloss) {   
	dReflection += vec4(calcReflection(reflDir, gloss), material_reflectivity);
}
`;

var reflectionSpherePS = `
#ifndef VIEWMATRIX
#define VIEWMATRIX
uniform mat4 matrix_view;
#endif
uniform sampler2D texture_sphereMap;
uniform float material_reflectivity;
vec3 calcReflection(vec3 reflDir, float gloss) {
	vec3 reflDirV = (mat3(matrix_view) * reflDir).xyz;
	float m = 2.0 * sqrt( dot(reflDirV.xy, reflDirV.xy) + (reflDirV.z+1.0)*(reflDirV.z+1.0) );
	vec2 sphereMapUv = reflDirV.xy / m + 0.5;
	return $DECODE(texture2D(texture_sphereMap, sphereMapUv));
}
void addReflection(vec3 reflDir, float gloss) {   
	dReflection += vec4(calcReflection(reflDir, gloss), material_reflectivity);
}
`;

var reflectionSheenPS = `
void addReflectionSheen(vec3 worldNormal, vec3 viewDir, float gloss) {
	float NoV = dot(worldNormal, viewDir);
	float alphaG = gloss * gloss;
	float a = gloss < 0.25 ? -339.2 * alphaG + 161.4 * gloss - 25.9 : -8.48 * alphaG + 14.3 * gloss - 9.95;
	float b = gloss < 0.25 ? 44.0 * alphaG - 23.7 * gloss + 3.26 : 1.97 * alphaG - 3.27 * gloss + 0.72;
	float DG = exp( a * NoV + b ) + ( gloss < 0.25 ? 0.0 : 0.1 * ( gloss - 0.25 ) );
	sReflection += calcReflection(worldNormal, 0.0) * saturate(DG);
}
`;

var refractionCubePS = `
vec3 refract2(vec3 viewVec, vec3 normal, float IOR) {
	float vn = dot(viewVec, normal);
	float k = 1.0 - IOR * IOR * (1.0 - vn * vn);
	vec3 refrVec = IOR * viewVec - (IOR * vn + sqrt(k)) * normal;
	return refrVec;
}
void addRefraction(
	vec3 worldNormal, 
	vec3 viewDir, 
	float thickness, 
	float gloss, 
	vec3 specularity, 
	vec3 albedo, 
	float transmission,
	float refractionIndex
#if defined(LIT_IRIDESCENCE)
	, vec3 iridescenceFresnel,
	float iridescenceIntensity
#endif 
) {
	vec4 tmpRefl = dReflection;
	vec3 reflectionDir = refract2(-viewDir, worldNormal, refractionIndex);
	dReflection = vec4(0);
	addReflection(reflectionDir, gloss);
	dDiffuseLight = mix(dDiffuseLight, dReflection.rgb * albedo, transmission);
	dReflection = tmpRefl;
}
`;

var refractionDynamicPS = `
uniform float material_invAttenuationDistance;
uniform vec3 material_attenuation;
void addRefraction(
	vec3 worldNormal, 
	vec3 viewDir, 
	float thickness, 
	float gloss, 
	vec3 specularity, 
	vec3 albedo, 
	float transmission,
	float refractionIndex
#if defined(LIT_IRIDESCENCE)
	, vec3 iridescenceFresnel,
	float iridescenceIntensity
#endif
) {
	vec3 modelScale;
	modelScale.x = length(vec3(matrix_model[0].xyz));
	modelScale.y = length(vec3(matrix_model[1].xyz));
	modelScale.z = length(vec3(matrix_model[2].xyz));
	vec3 refractionVector = normalize(refract(-viewDir, worldNormal, refractionIndex)) * thickness * modelScale;
	vec4 pointOfRefraction = vec4(vPositionW + refractionVector, 1.0);
	vec4 projectionPoint = matrix_viewProjection * pointOfRefraction;
	vec2 uv = getGrabScreenPos(projectionPoint);
	#ifdef SUPPORTS_TEXLOD
		float iorToRoughness = (1.0 - gloss) * clamp((1.0 / refractionIndex) * 2.0 - 2.0, 0.0, 1.0);
		float refractionLod = log2(uScreenSize.x) * iorToRoughness;
		vec3 refraction = texture2DLodEXT(uSceneColorMap, uv, refractionLod).rgb;
	#else
		vec3 refraction = texture2D(uSceneColorMap, uv).rgb;
	#endif
	vec3 transmittance;
	if (material_invAttenuationDistance != 0.0)
	{
		vec3 attenuation = -log(material_attenuation) * material_invAttenuationDistance;
		transmittance = exp(-attenuation * length(refractionVector));
	}
	else
	{
		transmittance = refraction;
	}
	vec3 fresnel = vec3(1.0) - 
		getFresnel(
			dot(viewDir, worldNormal), 
			gloss, 
			specularity
		#if defined(LIT_IRIDESCENCE)
			, iridescenceFresnel,
			iridescenceIntensity
		#endif
		);
	dDiffuseLight = mix(dDiffuseLight, refraction * transmittance * fresnel, transmission);
}
`;

var reprojectPS = `
varying vec2 vUv0;
#ifdef CUBEMAP_SOURCE
	uniform samplerCube sourceCube;
#else
	uniform sampler2D sourceTex;
#endif
#ifdef USE_SAMPLES_TEX
	uniform sampler2D samplesTex;
	uniform vec2 samplesTexInverseSize;
#endif
uniform vec4 params;
uniform vec2 params2;
float targetFace() { return params.x; }
float specularPower() { return params.y; }
float sourceCubeSeamScale() { return params.z; }
float targetCubeSeamScale() { return params.w; }
float targetTotalPixels() { return params2.x; }
float sourceTotalPixels() { return params2.y; }
float PI = 3.141592653589793;
float saturate(float x) {
	return clamp(x, 0.0, 1.0);
}
${decodePS}
${encodePS}
vec3 modifySeams(vec3 dir, float scale) {
	vec3 adir = abs(dir);
	float M = max(max(adir.x, adir.y), adir.z);
	return dir / M * vec3(
		adir.x == M ? 1.0 : scale,
		adir.y == M ? 1.0 : scale,
		adir.z == M ? 1.0 : scale
	);
}
vec2 toSpherical(vec3 dir) {
	return vec2(dir.xz == vec2(0.0) ? 0.0 : atan(dir.x, dir.z), asin(dir.y));
}
vec3 fromSpherical(vec2 uv) {
	return vec3(cos(uv.y) * sin(uv.x),
				sin(uv.y),
				cos(uv.y) * cos(uv.x));
}
vec3 getDirectionEquirect() {
	return fromSpherical((vec2(vUv0.x, 1.0 - vUv0.y) * 2.0 - 1.0) * vec2(PI, PI * 0.5));
}
float signNotZero(float k){
	return(k >= 0.0) ? 1.0 : -1.0;
}
vec2 signNotZero(vec2 v) {
	return vec2(signNotZero(v.x), signNotZero(v.y));
}
vec3 octDecode(vec2 o) {
	vec3 v = vec3(o.x, 1.0 - abs(o.x) - abs(o.y), o.y);
	if (v.y < 0.0) {
		v.xz = (1.0 - abs(v.zx)) * signNotZero(v.xz);
	}
	return normalize(v);
}
vec3 getDirectionOctahedral() {
	return octDecode(vec2(vUv0.x, 1.0 - vUv0.y) * 2.0 - 1.0);
}
vec2 octEncode(in vec3 v) {
	float l1norm = abs(v.x) + abs(v.y) + abs(v.z);
	vec2 result = v.xz * (1.0 / l1norm);
	if (v.y < 0.0) {
		result = (1.0 - abs(result.yx)) * signNotZero(result.xy);
	}
	return result;
}
#ifdef CUBEMAP_SOURCE
	vec4 sampleCubemap(vec3 dir) {
		return textureCube(sourceCube, modifySeams(dir, 1.0 - sourceCubeSeamScale()));
	}
	vec4 sampleCubemap(vec2 sph) {
	return sampleCubemap(fromSpherical(sph));
}
	vec4 sampleCubemap(vec3 dir, float mipLevel) {
		return textureCubeLodEXT(sourceCube, modifySeams(dir, 1.0 - exp2(mipLevel) * sourceCubeSeamScale()), mipLevel);
	}
	vec4 sampleCubemap(vec2 sph, float mipLevel) {
		return sampleCubemap(fromSpherical(sph), mipLevel);
	}
#else
	vec4 sampleEquirect(vec2 sph) {
		vec2 uv = sph / vec2(PI * 2.0, PI) + 0.5;
		return texture2D(sourceTex, vec2(uv.x, 1.0 - uv.y));
	}
	vec4 sampleEquirect(vec3 dir) {
		return sampleEquirect(toSpherical(dir));
	}
	vec4 sampleEquirect(vec2 sph, float mipLevel) {
		vec2 uv = sph / vec2(PI * 2.0, PI) + 0.5;
		return texture2DLodEXT(sourceTex, vec2(uv.x, 1.0 - uv.y), mipLevel);
	}
	vec4 sampleEquirect(vec3 dir, float mipLevel) {
		return sampleEquirect(toSpherical(dir), mipLevel);
	}
	vec4 sampleOctahedral(vec3 dir) {
		vec2 uv = octEncode(dir) * 0.5 + 0.5;
		return texture2D(sourceTex, vec2(uv.x, 1.0 - uv.y));
	}
	vec4 sampleOctahedral(vec2 sph) {
		return sampleOctahedral(fromSpherical(sph));
	}
	vec4 sampleOctahedral(vec3 dir, float mipLevel) {
		vec2 uv = octEncode(dir) * 0.5 + 0.5;
		return texture2DLodEXT(sourceTex, vec2(uv.x, 1.0 - uv.y), mipLevel);
	}
	vec4 sampleOctahedral(vec2 sph, float mipLevel) {
		return sampleOctahedral(fromSpherical(sph), mipLevel);
	}
#endif
vec3 getDirectionCubemap() {
	vec2 st = vUv0 * 2.0 - 1.0;
	float face = targetFace();
	vec3 vec;
	if (face == 0.0) {
		vec = vec3(1, -st.y, -st.x);
	} else if (face == 1.0) {
		vec = vec3(-1, -st.y, st.x);
	} else if (face == 2.0) {
		vec = vec3(st.x, 1, st.y);
	} else if (face == 3.0) {
		vec = vec3(st.x, -1, -st.y);
	} else if (face == 4.0) {
		vec = vec3(st.x, -st.y, 1);
	} else {
		vec = vec3(-st.x, -st.y, -1);
	}
	return normalize(modifySeams(vec, 1.0 / (1.0 - targetCubeSeamScale())));
}
mat3 matrixFromVector(vec3 n) {
	float a = 1.0 / (1.0 + n.z);
	float b = -n.x * n.y * a;
	vec3 b1 = vec3(1.0 - n.x * n.x * a, b, -n.x);
	vec3 b2 = vec3(b, 1.0 - n.y * n.y * a, -n.y);
	return mat3(b1, b2, n);
}
mat3 matrixFromVectorSlow(vec3 n) {
	vec3 up = (1.0 - abs(n.y) <= 0.0000001) ? vec3(0.0, 0.0, n.y > 0.0 ? 1.0 : -1.0) : vec3(0.0, 1.0, 0.0);
	vec3 x = normalize(cross(up, n));
	vec3 y = cross(n, x);
	return mat3(x, y, n);
}
vec4 reproject() {
	if (NUM_SAMPLES <= 1) {
		return ENCODE_FUNC(DECODE_FUNC(SOURCE_FUNC(TARGET_FUNC())));
	} else {
		vec3 t = TARGET_FUNC();
		vec3 tu = dFdx(t);
		vec3 tv = dFdy(t);
		vec3 result = vec3(0.0);
		for (float u = 0.0; u < NUM_SAMPLES_SQRT; ++u) {
			for (float v = 0.0; v < NUM_SAMPLES_SQRT; ++v) {
				result += DECODE_FUNC(SOURCE_FUNC(normalize(t +
															tu * (u / NUM_SAMPLES_SQRT - 0.5) +
															tv * (v / NUM_SAMPLES_SQRT - 0.5))));
			}
		}
		return ENCODE_FUNC(result / (NUM_SAMPLES_SQRT * NUM_SAMPLES_SQRT));
	}
}
vec4 unpackFloat = vec4(1.0, 1.0 / 255.0, 1.0 / 65025.0, 1.0 / 16581375.0);
#ifdef USE_SAMPLES_TEX
	void unpackSample(int i, out vec3 L, out float mipLevel) {
		float u = (float(i * 4) + 0.5) * samplesTexInverseSize.x;
		float v = (floor(u) + 0.5) * samplesTexInverseSize.y;
		vec4 raw;
		raw.x = dot(texture2D(samplesTex, vec2(u, v)), unpackFloat); u += samplesTexInverseSize.x;
		raw.y = dot(texture2D(samplesTex, vec2(u, v)), unpackFloat); u += samplesTexInverseSize.x;
		raw.z = dot(texture2D(samplesTex, vec2(u, v)), unpackFloat); u += samplesTexInverseSize.x;
		raw.w = dot(texture2D(samplesTex, vec2(u, v)), unpackFloat);
		L.xyz = raw.xyz * 2.0 - 1.0;
		mipLevel = raw.w * 8.0;
	}
	vec4 prefilterSamples() {
		mat3 vecSpace = matrixFromVectorSlow(TARGET_FUNC());
		vec3 L;
		float mipLevel;
		vec3 result = vec3(0.0);
		float totalWeight = 0.0;
		for (int i = 0; i < NUM_SAMPLES; ++i) {
			unpackSample(i, L, mipLevel);
			result += DECODE_FUNC(SOURCE_FUNC(vecSpace * L, mipLevel)) * L.z;
			totalWeight += L.z;
		}
		return ENCODE_FUNC(result / totalWeight);
	}
	vec4 prefilterSamplesUnweighted() {
		mat3 vecSpace = matrixFromVectorSlow(TARGET_FUNC());
		vec3 L;
		float mipLevel;
		vec3 result = vec3(0.0);
		float totalWeight = 0.0;
		for (int i = 0; i < NUM_SAMPLES; ++i) {
			unpackSample(i, L, mipLevel);
			result += DECODE_FUNC(SOURCE_FUNC(vecSpace * L, mipLevel));
		}
		return ENCODE_FUNC(result / float(NUM_SAMPLES));
	}
#endif
void main(void) {
	gl_FragColor = PROCESS_FUNC();
}
`;

var sampleCatmullRomPS = `
vec4 SampleTextureCatmullRom(TEXTURE_ACCEPT(tex), vec2 uv, vec2 texSize) {
	vec2 samplePos = uv * texSize;
	vec2 texPos1 = floor(samplePos - 0.5) + 0.5;
	vec2 f = samplePos - texPos1;
	vec2 w0 = f * (-0.5 + f * (1.0 - 0.5 * f));
	vec2 w1 = 1.0 + f * f * (-2.5 + 1.5 * f);
	vec2 w2 = f * (0.5 + f * (2.0 - 1.5 * f));
	vec2 w3 = f * f * (-0.5 + 0.5 * f);
	vec2 w12 = w1 + w2;
	vec2 offset12 = w2 / (w1 + w2);
	vec2 texPos0 = (texPos1 - 1.0) / texSize;
	vec2 texPos3 = (texPos1 + 2.0) / texSize;
	vec2 texPos12 = (texPos1 + offset12) / texSize;
	vec4 result = vec4(0.0);
	result += texture2DLodEXT(tex, vec2(texPos0.x, texPos0.y), 0.0) * w0.x * w0.y;
	result += texture2DLodEXT(tex, vec2(texPos12.x, texPos0.y), 0.0) * w12.x * w0.y;
	result += texture2DLodEXT(tex, vec2(texPos3.x, texPos0.y), 0.0) * w3.x * w0.y;
	result += texture2DLodEXT(tex, vec2(texPos0.x, texPos12.y), 0.0) * w0.x * w12.y;
	result += texture2DLodEXT(tex, vec2(texPos12.x, texPos12.y), 0.0) * w12.x * w12.y;
	result += texture2DLodEXT(tex, vec2(texPos3.x, texPos12.y), 0.0) * w3.x * w12.y;
	result += texture2DLodEXT(tex, vec2(texPos0.x, texPos3.y), 0.0) * w0.x * w3.y;
	result += texture2DLodEXT(tex, vec2(texPos12.x, texPos3.y), 0.0) * w12.x * w3.y;
	result += texture2DLodEXT(tex, vec2(texPos3.x, texPos3.y), 0.0) * w3.x * w3.y;
	return result;
}
`;

var screenDepthPS = `
uniform highp sampler2D uSceneDepthMap;
#ifndef SCREENSIZE
#define SCREENSIZE
uniform vec4 uScreenSize;
#endif
#ifndef VIEWMATRIX
#define VIEWMATRIX
uniform mat4 matrix_view;
#endif
#ifndef LINEARIZE_DEPTH
#ifndef CAMERAPLANES
#define CAMERAPLANES
uniform vec4 camera_params;
#endif
#define LINEARIZE_DEPTH
#ifdef GL2
float linearizeDepth(float z) {
	if (camera_params.w == 0.0)
		return (camera_params.z * camera_params.y) / (camera_params.y + z * (camera_params.z - camera_params.y));
	else
		return camera_params.z + z * (camera_params.y - camera_params.z);
}
#else
#ifndef UNPACKFLOAT
#define UNPACKFLOAT
float unpackFloat(vec4 rgbaDepth) {
	const vec4 bitShift = vec4(1.0 / (256.0 * 256.0 * 256.0), 1.0 / (256.0 * 256.0), 1.0 / 256.0, 1.0);
	return dot(rgbaDepth, bitShift);
}
#endif
#endif
#endif
float getLinearScreenDepth(vec2 uv) {
	#ifdef GL2
		return linearizeDepth(texture2D(uSceneDepthMap, uv).r);
	#else
		return unpackFloat(texture2D(uSceneDepthMap, uv)) * camera_params.y;
	#endif
}
#ifndef VERTEXSHADER
float getLinearScreenDepth() {
	vec2 uv = gl_FragCoord.xy * uScreenSize.zw;
	return getLinearScreenDepth(uv);
}
#endif
float getLinearDepth(vec3 pos) {
	return -(matrix_view * vec4(pos, 1.0)).z;
}
`;

var shadowCascadesPS = `
const float maxCascades = 4.0;
mat4 cascadeShadowMat;
void getShadowCascadeMatrix(mat4 shadowMatrixPalette[4], float shadowCascadeDistances[4], float shadowCascadeCount) {
	float depth = 1.0 / gl_FragCoord.w;
	float cascadeIndex = 0.0;
	for (float i = 0.0; i < maxCascades; i++) {
		if (depth < shadowCascadeDistances[int(i)]) {
			cascadeIndex = i;
			break;
		}
	}
	cascadeIndex = min(cascadeIndex, shadowCascadeCount - 1.0);
	#ifdef GL2
		cascadeShadowMat = shadowMatrixPalette[int(cascadeIndex)];
	#else
		if (cascadeIndex == 0.0) {
			cascadeShadowMat = shadowMatrixPalette[0];
		}
		else if (cascadeIndex == 1.0) {
			cascadeShadowMat = shadowMatrixPalette[1];
		}
		else if (cascadeIndex == 2.0) {
			cascadeShadowMat = shadowMatrixPalette[2];
		}
		else {
			cascadeShadowMat = shadowMatrixPalette[3];
		}
	#endif
}
void fadeShadow(float shadowCascadeDistances[4]) {				  
	float depth = 1.0 / gl_FragCoord.w;
	if (depth > shadowCascadeDistances[int(maxCascades - 1.0)]) {
		dShadowCoord.z = -9999999.0;
	}
}
`;

var shadowEVSMPS = `
float VSM$(sampler2D tex, vec2 texCoords, float resolution, float Z, float vsmBias, float exponent) {
	vec3 moments = texture2D(tex, texCoords).xyz;
	return calculateEVSM(moments, Z, vsmBias, exponent);
}
float getShadowVSM$(sampler2D shadowMap, vec3 shadowCoord, vec4 shadowParams, float exponent, vec3 lightDir) {
	return VSM$(shadowMap, shadowCoord.xy, shadowParams.x, shadowCoord.z, shadowParams.y, exponent);
}
float getShadowSpotVSM$(sampler2D shadowMap, vec3 shadowCoord, vec4 shadowParams, float exponent, vec3 lightDir) {
	return VSM$(shadowMap, shadowCoord.xy, shadowParams.x, length(lightDir) * shadowParams.w + shadowParams.z, shadowParams.y, exponent);
}
`;

var shadowEVSMnPS = `
float VSM$(TEXTURE_ACCEPT(tex), vec2 texCoords, float resolution, float Z, float vsmBias, float exponent) {
	float pixelSize = 1.0 / resolution;
	texCoords -= vec2(pixelSize);
	vec3 s00 = texture2D(tex, texCoords).xyz;
	vec3 s10 = texture2D(tex, texCoords + vec2(pixelSize, 0)).xyz;
	vec3 s01 = texture2D(tex, texCoords + vec2(0, pixelSize)).xyz;
	vec3 s11 = texture2D(tex, texCoords + vec2(pixelSize)).xyz;
	vec2 fr = fract(texCoords * resolution);
	vec3 h0 = mix(s00, s10, fr.x);
	vec3 h1 = mix(s01, s11, fr.x);
	vec3 moments = mix(h0, h1, fr.y);
	return calculateEVSM(moments, Z, vsmBias, exponent);
}
float getShadowVSM$(TEXTURE_ACCEPT(shadowMap), vec3 shadowCoord, vec4 shadowParams, float exponent, vec3 lightDir) {
	return VSM$(TEXTURE_PASS(shadowMap), shadowCoord.xy, shadowParams.x, shadowCoord.z, shadowParams.y, exponent);
}
float getShadowSpotVSM$(TEXTURE_ACCEPT(shadowMap), vec3 shadowCoord, vec4 shadowParams, float exponent, vec3 lightDir) {
	return VSM$(TEXTURE_PASS(shadowMap), shadowCoord.xy, shadowParams.x, length(lightDir) * shadowParams.w + shadowParams.z, shadowParams.y, exponent);
}
`;

var shadowPCSSPS = `
#define PCSS_SAMPLE_COUNT 16
uniform float pcssDiskSamples[PCSS_SAMPLE_COUNT];
uniform float pcssSphereSamples[PCSS_SAMPLE_COUNT];
vec2 vogelDisk(int sampleIndex, float count, float phi, float r) {
	const float GoldenAngle = 2.4;
	float theta = float(sampleIndex) * GoldenAngle + phi;
	float sine = sin(theta);
	float cosine = cos(theta);
	return vec2(r * cosine, r * sine);
}
vec3 vogelSphere(int sampleIndex, float count, float phi, float r) {
	const float GoldenAngle = 2.4;
	float theta = float(sampleIndex) * GoldenAngle + phi;
	float weight = float(sampleIndex) / count;
	return vec3(cos(theta) * r, weight, sin(theta) * r);
}
float noise(vec2 screenPos) {
	const float PHI = 1.61803398874989484820459;
	return fract(sin(dot(screenPos * PHI, screenPos)) * screenPos.x);
}
#ifndef UNPACKFLOAT
#define UNPACKFLOAT
float unpackFloat(vec4 rgbaDepth) {
	const vec4 bitShift = vec4(1.0 / (256.0 * 256.0 * 256.0), 1.0 / (256.0 * 256.0), 1.0 / 256.0, 1.0);
	return dot(rgbaDepth, bitShift);
}
#endif
float viewSpaceDepth(float depth, mat4 invProjection) {
	float z = depth * 2.0 - 1.0;
	vec4 clipSpace = vec4(0.0, 0.0, z, 1.0);
	vec4 viewSpace = invProjection * clipSpace;
	return viewSpace.z;
}
float PCSSBlockerDistance(TEXTURE_ACCEPT(shadowMap), vec2 sampleCoords[PCSS_SAMPLE_COUNT], vec2 shadowCoords, vec2 searchSize, float z) {
	float blockers = 0.0;
	float averageBlocker = 0.0;
	for (int i = 0; i < PCSS_SAMPLE_COUNT; i++) {
		vec2 offset = sampleCoords[i] * searchSize;
		vec2 sampleUV = shadowCoords + offset;
	#ifdef GL2
		float blocker = textureLod(shadowMap, sampleUV, 0.0).r;
	#else
		float blocker = unpackFloat(texture2D(shadowMap, sampleUV));
	#endif		
		float isBlocking = step(blocker, z);
		blockers += isBlocking;
		averageBlocker += blocker * isBlocking;
	}
	if (blockers > 0.0)
		return averageBlocker /= blockers;
	return -1.0;
}
float PCSS(TEXTURE_ACCEPT(shadowMap), vec3 shadowCoords, vec4 cameraParams, vec2 shadowSearchArea) {
	float receiverDepth = shadowCoords.z;
#ifndef GL2
	receiverDepth *= 1.0 / (cameraParams.y - cameraParams.z);
#endif
	vec2 samplePoints[PCSS_SAMPLE_COUNT];
	float noise = noise( gl_FragCoord.xy ) * 2.0 * PI;
	for (int i = 0; i < PCSS_SAMPLE_COUNT; i++) {
		float pcssPresample = pcssDiskSamples[i];
		samplePoints[i] = vogelDisk(i, float(PCSS_SAMPLE_COUNT), noise, pcssPresample);
	}
	float averageBlocker = PCSSBlockerDistance(TEXTURE_PASS(shadowMap), samplePoints, shadowCoords.xy, shadowSearchArea, receiverDepth);
	if (averageBlocker == -1.0) {
		return 1.0;
	} else {
		vec2 filterRadius = ((receiverDepth - averageBlocker) / averageBlocker) * shadowSearchArea * cameraParams.x;
		float shadow = 0.0;
		for (int i = 0; i < PCSS_SAMPLE_COUNT; i ++)
		{
			vec2 sampleUV = samplePoints[i] * filterRadius;
			sampleUV = shadowCoords.xy + sampleUV;
		#ifdef GL2
			float depth = textureLod(shadowMap, sampleUV, 0.0).r;
		#else
			float depth = unpackFloat(texture2D(shadowMap, sampleUV));
		#endif
			shadow += step(receiverDepth, depth);
		}
		return shadow / float(PCSS_SAMPLE_COUNT);
	} 
}
float PCSSCubeBlockerDistance(samplerCube shadowMap, vec3 lightDirNorm, vec3 samplePoints[PCSS_SAMPLE_COUNT], float z, float shadowSearchArea) {
	float blockers = 0.0;
	float averageBlocker = 0.0;
	for (int i = 0; i < PCSS_SAMPLE_COUNT; i++) {
		vec3 sampleDir = lightDirNorm + samplePoints[i] * shadowSearchArea;
		sampleDir = normalize(sampleDir);
	#ifdef GL2
		float blocker = textureCubeLodEXT(shadowMap, sampleDir, 0.0).r;
	#else
		float blocker = unpackFloat(textureCube(shadowMap, sampleDir));
	#endif
		float isBlocking = step(blocker, z);
		blockers += isBlocking;
		averageBlocker += blocker * isBlocking;
	}
	if (blockers > 0.0)
		return averageBlocker /= float(blockers);
	return -1.0;
}
float PCSSCube(samplerCube shadowMap, vec4 shadowParams, vec3 shadowCoords, vec4 cameraParams, float shadowSearchArea, vec3 lightDir) {
	
	vec3 samplePoints[PCSS_SAMPLE_COUNT];
	float noise = noise( gl_FragCoord.xy ) * 2.0 * PI;
	for (int i = 0; i < PCSS_SAMPLE_COUNT; i++) {
		float r = pcssSphereSamples[i];
		samplePoints[i] = vogelSphere(i, float(PCSS_SAMPLE_COUNT), noise, r);
	}
	float receiverDepth = length(lightDir) * shadowParams.w + shadowParams.z;
	vec3 lightDirNorm = normalize(lightDir);
	
	float averageBlocker = PCSSCubeBlockerDistance(shadowMap, lightDirNorm, samplePoints, receiverDepth, shadowSearchArea);
	if (averageBlocker == -1.0) {
		return 1.0;
	} else {
		float filterRadius = ((receiverDepth - averageBlocker) / averageBlocker) * shadowSearchArea;
		float shadow = 0.0;
		for (int i = 0; i < PCSS_SAMPLE_COUNT; i++)
		{
			vec3 offset = samplePoints[i] * filterRadius;
			vec3 sampleDir = lightDirNorm + offset;
			sampleDir = normalize(sampleDir);
			#ifdef GL2
				float depth = textureCubeLodEXT(shadowMap, sampleDir, 0.0).r;
			#else
				float depth = unpackFloat(textureCube(shadowMap, sampleDir));
			#endif
			shadow += step(receiverDepth, depth);
		}
		return shadow / float(PCSS_SAMPLE_COUNT);
	}
}
float getShadowPointPCSS(samplerCube shadowMap, vec3 shadowCoord, vec4 shadowParams, vec4 cameraParams, vec2 shadowSearchArea, vec3 lightDir) {
	return PCSSCube(shadowMap, shadowParams, shadowCoord, cameraParams, shadowSearchArea.x, lightDir);
}
float getShadowSpotPCSS(TEXTURE_ACCEPT(shadowMap), vec3 shadowCoord, vec4 shadowParams, vec4 cameraParams, vec2 shadowSearchArea, vec3 lightDir) {
	return PCSS(TEXTURE_PASS(shadowMap), shadowCoord, cameraParams, shadowSearchArea);
}
float getShadowPCSS(TEXTURE_ACCEPT(shadowMap), vec3 shadowCoord, vec4 shadowParams, vec4 cameraParams, vec2 shadowSearchArea, vec3 lightDir) {
	return PCSS(TEXTURE_PASS(shadowMap), shadowCoord, cameraParams, shadowSearchArea);
}
`;

var shadowSampleCoordPS = `
vec3 getShadowSampleCoord$LIGHT(mat4 shadowTransform, vec4 shadowParams, vec3 worldPosition, vec3 lightPos, inout vec3 lightDir, vec3 lightDirNorm, vec3 normal) {
	vec3 surfacePosition = worldPosition;
#ifdef SHADOW_SAMPLE_POINT
	#ifdef SHADOW_SAMPLE_NORMAL_OFFSET
		float distScale = length(lightDir);
		surfacePosition = worldPosition + normal * shadowParams.y * clamp(1.0 - dot(normal, -lightDirNorm), 0.0, 1.0) * distScale;
		lightDir = surfacePosition - lightPos;
		return lightDir;
	#endif
#else
	#ifdef SHADOW_SAMPLE_SOURCE_ZBUFFER
		#ifdef SHADOW_SAMPLE_NORMAL_OFFSET
			surfacePosition = worldPosition + normal * shadowParams.y;
		#endif
	#else
		#ifdef SHADOW_SAMPLE_NORMAL_OFFSET
			#ifdef SHADOW_SAMPLE_ORTHO
				float distScale = 1.0;
			#else
				float distScale = abs(dot(vPositionW - lightPos, lightDirNorm));
			#endif
			surfacePosition = worldPosition + normal * shadowParams.y * clamp(1.0 - dot(normal, -lightDirNorm), 0.0, 1.0) * distScale;
		#endif
	#endif
	vec4 positionInShadowSpace = shadowTransform * vec4(surfacePosition, 1.0);
	#ifdef SHADOW_SAMPLE_ORTHO
		positionInShadowSpace.z = saturate(positionInShadowSpace.z) - 0.0001;
	#else
		#ifdef SHADOW_SAMPLE_SOURCE_ZBUFFER
			positionInShadowSpace.xyz /= positionInShadowSpace.w;
		#else
			positionInShadowSpace.xy /= positionInShadowSpace.w;
			positionInShadowSpace.z = length(lightDir) * shadowParams.w;
		#endif
	#endif
	#ifdef SHADOW_SAMPLE_Z_BIAS
		positionInShadowSpace.z += getShadowBias(shadowParams.x, shadowParams.z);
	#endif
	surfacePosition = positionInShadowSpace.xyz;
#endif
	return surfacePosition;
}
`;

var shadowStandardPS = `
vec3 lessThan2(vec3 a, vec3 b) {
	return clamp((b - a)*1000.0, 0.0, 1.0);
}
#ifndef UNPACKFLOAT
#define UNPACKFLOAT
	float unpackFloat(vec4 rgbaDepth) {
		const vec4 bitShift = vec4(1.0 / (256.0 * 256.0 * 256.0), 1.0 / (256.0 * 256.0), 1.0 / 256.0, 1.0);
		return dot(rgbaDepth, bitShift);
	}
#endif
#ifdef GL2
float _getShadowPCF3x3(SHADOWMAP_ACCEPT(shadowMap), vec3 shadowCoord, vec3 shadowParams) {
	float z = shadowCoord.z;
	vec2 uv = shadowCoord.xy * shadowParams.x;
	float shadowMapSizeInv = 1.0 / shadowParams.x;
	vec2 base_uv = floor(uv + 0.5);
	float s = (uv.x + 0.5 - base_uv.x);
	float t = (uv.y + 0.5 - base_uv.y);
	base_uv -= vec2(0.5);
	base_uv *= shadowMapSizeInv;
	float sum = 0.0;
	float uw0 = (3.0 - 2.0 * s);
	float uw1 = (1.0 + 2.0 * s);
	float u0 = (2.0 - s) / uw0 - 1.0;
	float u1 = s / uw1 + 1.0;
	float vw0 = (3.0 - 2.0 * t);
	float vw1 = (1.0 + 2.0 * t);
	float v0 = (2.0 - t) / vw0 - 1.0;
	float v1 = t / vw1 + 1.0;
	u0 = u0 * shadowMapSizeInv + base_uv.x;
	v0 = v0 * shadowMapSizeInv + base_uv.y;
	u1 = u1 * shadowMapSizeInv + base_uv.x;
	v1 = v1 * shadowMapSizeInv + base_uv.y;
	sum += uw0 * vw0 * textureShadow(shadowMap, vec3(u0, v0, z));
	sum += uw1 * vw0 * textureShadow(shadowMap, vec3(u1, v0, z));
	sum += uw0 * vw1 * textureShadow(shadowMap, vec3(u0, v1, z));
	sum += uw1 * vw1 * textureShadow(shadowMap, vec3(u1, v1, z));
	sum *= 1.0f / 16.0;
	return sum;
}
float getShadowPCF3x3(SHADOWMAP_ACCEPT(shadowMap), vec3 shadowCoord, vec4 shadowParams) {
	return _getShadowPCF3x3(SHADOWMAP_PASS(shadowMap), shadowCoord, shadowParams.xyz);
}
float getShadowSpotPCF3x3(SHADOWMAP_ACCEPT(shadowMap), vec3 shadowCoord, vec4 shadowParams) {
	return _getShadowPCF3x3(SHADOWMAP_PASS(shadowMap), shadowCoord, shadowParams.xyz);
}
float getShadowPCF1x1(SHADOWMAP_ACCEPT(shadowMap), vec3 shadowCoord, vec4 shadowParams) {
	return textureShadow(shadowMap, shadowCoord);
}
float getShadowSpotPCF1x1(SHADOWMAP_ACCEPT(shadowMap), vec3 shadowCoord, vec4 shadowParams) {
	return textureShadow(shadowMap, shadowCoord);
}
#else
float _xgetShadowPCF3x3(mat3 depthKernel, vec3 shadowCoord, sampler2D shadowMap, vec3 shadowParams) {
	mat3 shadowKernel;
	vec3 shadowZ = vec3(shadowCoord.z);
	shadowKernel[0] = vec3(greaterThan(depthKernel[0], shadowZ));
	shadowKernel[1] = vec3(greaterThan(depthKernel[1], shadowZ));
	shadowKernel[2] = vec3(greaterThan(depthKernel[2], shadowZ));
	vec2 fractionalCoord = fract( shadowCoord.xy * shadowParams.x );
	shadowKernel[0] = mix(shadowKernel[0], shadowKernel[1], fractionalCoord.x);
	shadowKernel[1] = mix(shadowKernel[1], shadowKernel[2], fractionalCoord.x);
	vec4 shadowValues;
	shadowValues.x = mix(shadowKernel[0][0], shadowKernel[0][1], fractionalCoord.y);
	shadowValues.y = mix(shadowKernel[0][1], shadowKernel[0][2], fractionalCoord.y);
	shadowValues.z = mix(shadowKernel[1][0], shadowKernel[1][1], fractionalCoord.y);
	shadowValues.w = mix(shadowKernel[1][1], shadowKernel[1][2], fractionalCoord.y);
	return dot( shadowValues, vec4( 1.0 ) ) * 0.25;
}
float _getShadowPCF3x3(sampler2D shadowMap, vec3 shadowCoord, vec3 shadowParams) {
	float xoffset = 1.0 / shadowParams.x;
	float dx0 = -xoffset;
	float dx1 = xoffset;
	mat3 depthKernel;
	depthKernel[0][0] = unpackFloat(textureShadow(shadowMap, shadowCoord.xy + vec2(dx0, dx0)));
	depthKernel[0][1] = unpackFloat(textureShadow(shadowMap, shadowCoord.xy + vec2(dx0, 0.0)));
	depthKernel[0][2] = unpackFloat(textureShadow(shadowMap, shadowCoord.xy + vec2(dx0, dx1)));
	depthKernel[1][0] = unpackFloat(textureShadow(shadowMap, shadowCoord.xy + vec2(0.0, dx0)));
	depthKernel[1][1] = unpackFloat(textureShadow(shadowMap, shadowCoord.xy));
	depthKernel[1][2] = unpackFloat(textureShadow(shadowMap, shadowCoord.xy + vec2(0.0, dx1)));
	depthKernel[2][0] = unpackFloat(textureShadow(shadowMap, shadowCoord.xy + vec2(dx1, dx0)));
	depthKernel[2][1] = unpackFloat(textureShadow(shadowMap, shadowCoord.xy + vec2(dx1, 0.0)));
	depthKernel[2][2] = unpackFloat(textureShadow(shadowMap, shadowCoord.xy + vec2(dx1, dx1)));
	return _xgetShadowPCF3x3(depthKernel, shadowCoord, shadowMap, shadowParams);
}
float getShadowPCF3x3(sampler2D shadowMap, vec3 shadowCoord, vec4 shadowParams) {
	return _getShadowPCF3x3(shadowMap, shadowCoord, shadowParams.xyz);
}
float getShadowSpotPCF3x3(sampler2D shadowMap, vec3 shadowCoord, vec4 shadowParams) {
	return _getShadowPCF3x3(shadowMap, shadowCoord, shadowParams.xyz);
}
float _getShadowPCF1x1(sampler2D shadowMap, vec3 shadowCoord) {
	float shadowSample = unpackFloat(textureShadow(shadowMap, shadowCoord.xy));
	return shadowSample > shadowCoord.z ? 1.0 : 0.0;
}
float getShadowPCF1x1(sampler2D shadowMap, vec3 shadowCoord, vec4 shadowParams) {
	return _getShadowPCF1x1(shadowMap, shadowCoord);
}
float getShadowSpotPCF1x1(sampler2D shadowMap, vec3 shadowCoord, vec4 shadowParams) {
	return _getShadowPCF1x1(shadowMap, shadowCoord);
}
#endif
#ifndef WEBGPU
float _getShadowPoint(samplerCube shadowMap, vec4 shadowParams, vec3 dir) {
	vec3 tc = normalize(dir);
	vec3 tcAbs = abs(tc);
	vec4 dirX = vec4(1,0,0, tc.x);
	vec4 dirY = vec4(0,1,0, tc.y);
	float majorAxisLength = tc.z;
	if ((tcAbs.x > tcAbs.y) && (tcAbs.x > tcAbs.z)) {
		dirX = vec4(0,0,1, tc.z);
		dirY = vec4(0,1,0, tc.y);
		majorAxisLength = tc.x;
	} else if ((tcAbs.y > tcAbs.x) && (tcAbs.y > tcAbs.z)) {
		dirX = vec4(1,0,0, tc.x);
		dirY = vec4(0,0,1, tc.z);
		majorAxisLength = tc.y;
	}
	float shadowParamsInFaceSpace = ((1.0/shadowParams.x) * 2.0) * abs(majorAxisLength);
	vec3 xoffset = (dirX.xyz * shadowParamsInFaceSpace);
	vec3 yoffset = (dirY.xyz * shadowParamsInFaceSpace);
	vec3 dx0 = -xoffset;
	vec3 dy0 = -yoffset;
	vec3 dx1 = xoffset;
	vec3 dy1 = yoffset;
	mat3 shadowKernel;
	mat3 depthKernel;
	depthKernel[0][0] = unpackFloat(textureCube(shadowMap, tc + dx0 + dy0));
	depthKernel[0][1] = unpackFloat(textureCube(shadowMap, tc + dx0));
	depthKernel[0][2] = unpackFloat(textureCube(shadowMap, tc + dx0 + dy1));
	depthKernel[1][0] = unpackFloat(textureCube(shadowMap, tc + dy0));
	depthKernel[1][1] = unpackFloat(textureCube(shadowMap, tc));
	depthKernel[1][2] = unpackFloat(textureCube(shadowMap, tc + dy1));
	depthKernel[2][0] = unpackFloat(textureCube(shadowMap, tc + dx1 + dy0));
	depthKernel[2][1] = unpackFloat(textureCube(shadowMap, tc + dx1));
	depthKernel[2][2] = unpackFloat(textureCube(shadowMap, tc + dx1 + dy1));
	vec3 shadowZ = vec3(length(dir) * shadowParams.w + shadowParams.z);
	shadowKernel[0] = vec3(lessThan2(depthKernel[0], shadowZ));
	shadowKernel[1] = vec3(lessThan2(depthKernel[1], shadowZ));
	shadowKernel[2] = vec3(lessThan2(depthKernel[2], shadowZ));
	vec2 uv = (vec2(dirX.w, dirY.w) / abs(majorAxisLength)) * 0.5;
	vec2 fractionalCoord = fract( uv * shadowParams.x );
	shadowKernel[0] = mix(shadowKernel[0], shadowKernel[1], fractionalCoord.x);
	shadowKernel[1] = mix(shadowKernel[1], shadowKernel[2], fractionalCoord.x);
	vec4 shadowValues;
	shadowValues.x = mix(shadowKernel[0][0], shadowKernel[0][1], fractionalCoord.y);
	shadowValues.y = mix(shadowKernel[0][1], shadowKernel[0][2], fractionalCoord.y);
	shadowValues.z = mix(shadowKernel[1][0], shadowKernel[1][1], fractionalCoord.y);
	shadowValues.w = mix(shadowKernel[1][1], shadowKernel[1][2], fractionalCoord.y);
	return 1.0 - dot( shadowValues, vec4( 1.0 ) ) * 0.25;
}
float getShadowPointPCF3x3(samplerCube shadowMap, vec3 shadowCoord, vec4 shadowParams, vec3 lightDir) {
	return _getShadowPoint(shadowMap, shadowParams, lightDir);
}
#endif
`;

var shadowStandardGL2PS = `
float _getShadowPCF5x5(SHADOWMAP_ACCEPT(shadowMap), vec3 shadowCoord, vec3 shadowParams) {
	float z = shadowCoord.z;
	vec2 uv = shadowCoord.xy * shadowParams.x;
	float shadowMapSizeInv = 1.0 / shadowParams.x;
	vec2 base_uv = floor(uv + 0.5);
	float s = (uv.x + 0.5 - base_uv.x);
	float t = (uv.y + 0.5 - base_uv.y);
	base_uv -= vec2(0.5);
	base_uv *= shadowMapSizeInv;
	float uw0 = (4.0 - 3.0 * s);
	float uw1 = 7.0;
	float uw2 = (1.0 + 3.0 * s);
	float u0 = (3.0 - 2.0 * s) / uw0 - 2.0;
	float u1 = (3.0 + s) / uw1;
	float u2 = s / uw2 + 2.0;
	float vw0 = (4.0 - 3.0 * t);
	float vw1 = 7.0;
	float vw2 = (1.0 + 3.0 * t);
	float v0 = (3.0 - 2.0 * t) / vw0 - 2.0;
	float v1 = (3.0 + t) / vw1;
	float v2 = t / vw2 + 2.0;
	float sum = 0.0;
	u0 = u0 * shadowMapSizeInv + base_uv.x;
	v0 = v0 * shadowMapSizeInv + base_uv.y;
	u1 = u1 * shadowMapSizeInv + base_uv.x;
	v1 = v1 * shadowMapSizeInv + base_uv.y;
	u2 = u2 * shadowMapSizeInv + base_uv.x;
	v2 = v2 * shadowMapSizeInv + base_uv.y;
	sum += uw0 * vw0 * textureShadow(shadowMap, vec3(u0, v0, z));
	sum += uw1 * vw0 * textureShadow(shadowMap, vec3(u1, v0, z));
	sum += uw2 * vw0 * textureShadow(shadowMap, vec3(u2, v0, z));
	sum += uw0 * vw1 * textureShadow(shadowMap, vec3(u0, v1, z));
	sum += uw1 * vw1 * textureShadow(shadowMap, vec3(u1, v1, z));
	sum += uw2 * vw1 * textureShadow(shadowMap, vec3(u2, v1, z));
	sum += uw0 * vw2 * textureShadow(shadowMap, vec3(u0, v2, z));
	sum += uw1 * vw2 * textureShadow(shadowMap, vec3(u1, v2, z));
	sum += uw2 * vw2 * textureShadow(shadowMap, vec3(u2, v2, z));
	sum *= 1.0f / 144.0;
	sum = saturate(sum);
	return sum;
}
float getShadowPCF5x5(SHADOWMAP_ACCEPT(shadowMap), vec3 shadowCoord, vec4 shadowParams) {
	return _getShadowPCF5x5(SHADOWMAP_PASS(shadowMap), shadowCoord, shadowParams.xyz);
}
float getShadowSpotPCF5x5(SHADOWMAP_ACCEPT(shadowMap), vec3 shadowCoord, vec4 shadowParams) {
	return _getShadowPCF5x5(SHADOWMAP_PASS(shadowMap), shadowCoord, shadowParams.xyz);
}
`;

var shadowVSM8PS = `
float calculateVSM8(vec3 moments, float Z, float vsmBias) {
	float VSMBias = vsmBias;
	float depthScale = VSMBias * Z;
	float minVariance1 = depthScale * depthScale;
	return chebyshevUpperBound(moments.xy, Z, minVariance1, 0.1);
}
float decodeFloatRG(vec2 rg) {
	return rg.y*(1.0/255.0) + rg.x;
}
float VSM8(TEXTURE_ACCEPT(tex), vec2 texCoords, float resolution, float Z, float vsmBias, float exponent) {
	vec4 c = texture2D(tex, texCoords);
	vec3 moments = vec3(decodeFloatRG(c.xy), decodeFloatRG(c.zw), 0.0);
	return calculateVSM8(moments, Z, vsmBias);
}
float getShadowVSM8(TEXTURE_ACCEPT(shadowMap), vec3 shadowCoord, vec4 shadowParams, float exponent, vec3 lightDir) {
	return VSM8(TEXTURE_PASS(shadowMap), shadowCoord.xy, shadowParams.x, shadowCoord.z, shadowParams.y, 0.0);
}
float getShadowSpotVSM8(TEXTURE_ACCEPT(shadowMap), vec3 shadowCoord, vec4 shadowParams, float exponent, vec3 lightDir) {
	return VSM8(TEXTURE_PASS(shadowMap), shadowCoord.xy, shadowParams.x, length(lightDir) * shadowParams.w + shadowParams.z, shadowParams.y, 0.0);
}
`;

var shadowVSM_commonPS = `
float linstep(float a, float b, float v) {
	return saturate((v - a) / (b - a));
}
float reduceLightBleeding(float pMax, float amount) {
	 return linstep(amount, 1.0, pMax);
}
float chebyshevUpperBound(vec2 moments, float mean, float minVariance, float lightBleedingReduction) {
	float variance = moments.y - (moments.x * moments.x);
	variance = max(variance, minVariance);
	float d = mean - moments.x;
	float pMax = variance / (variance + (d * d));
	pMax = reduceLightBleeding(pMax, lightBleedingReduction);
	return (mean <= moments.x ? 1.0 : pMax);
}
float calculateEVSM(vec3 moments, float Z, float vsmBias, float exponent) {
	Z = 2.0 * Z - 1.0;
	float warpedDepth = exp(exponent * Z);
	moments.xy += vec2(warpedDepth, warpedDepth*warpedDepth) * (1.0 - moments.z);
	float VSMBias = vsmBias;
	float depthScale = VSMBias * exponent * warpedDepth;
	float minVariance1 = depthScale * depthScale;
	return chebyshevUpperBound(moments.xy, warpedDepth, minVariance1, 0.1);
}
`;

var skinBatchConstVS = `
attribute float vertex_boneIndices;
uniform vec4 matrix_pose[BONE_LIMIT * 3];
mat4 getBoneMatrix(const in float i) {
	vec4 v1 = matrix_pose[int(3.0 * i)];
	vec4 v2 = matrix_pose[int(3.0 * i + 1.0)];
	vec4 v3 = matrix_pose[int(3.0 * i + 2.0)];
	return mat4(
		v1.x, v2.x, v3.x, 0,
		v1.y, v2.y, v3.y, 0,
		v1.z, v2.z, v3.z, 0,
		v1.w, v2.w, v3.w, 1
	);
}
`;

var skinBatchTexVS = `
attribute float vertex_boneIndices;
uniform highp sampler2D texture_poseMap;
uniform vec4 texture_poseMapSize;
mat4 getBoneMatrix(const in float i) {
	float j = i * 3.0;
	float dx = texture_poseMapSize.z;
	float dy = texture_poseMapSize.w;
	float y = floor(j * dx);
	float x = j - (y * texture_poseMapSize.x);
	y = dy * (y + 0.5);
	vec4 v1 = texture2D(texture_poseMap, vec2(dx * (x + 0.5), y));
	vec4 v2 = texture2D(texture_poseMap, vec2(dx * (x + 1.5), y));
	vec4 v3 = texture2D(texture_poseMap, vec2(dx * (x + 2.5), y));
	return mat4(
		v1.x, v2.x, v3.x, 0,
		v1.y, v2.y, v3.y, 0,
		v1.z, v2.z, v3.z, 0,
		v1.w, v2.w, v3.w, 1
	);
}
`;

var skinConstVS = `
attribute vec4 vertex_boneWeights;
attribute vec4 vertex_boneIndices;
uniform vec4 matrix_pose[BONE_LIMIT * 3];
void getBoneMatrix(const in float i, out vec4 v1, out vec4 v2, out vec4 v3) {
	v1 = matrix_pose[int(3.0 * i)];
	v2 = matrix_pose[int(3.0 * i + 1.0)];
	v3 = matrix_pose[int(3.0 * i + 2.0)];
}
mat4 getSkinMatrix(const in vec4 indices, const in vec4 weights) {
	vec4 a1, a2, a3;
	getBoneMatrix(indices.x, a1, a2, a3);
	vec4 b1, b2, b3;
	getBoneMatrix(indices.y, b1, b2, b3);
	vec4 c1, c2, c3;
	getBoneMatrix(indices.z, c1, c2, c3);
	vec4 d1, d2, d3;
	getBoneMatrix(indices.w, d1, d2, d3);
	vec4 v1 = a1 * weights.x + b1 * weights.y + c1 * weights.z + d1 * weights.w;
	vec4 v2 = a2 * weights.x + b2 * weights.y + c2 * weights.z + d2 * weights.w;
	vec4 v3 = a3 * weights.x + b3 * weights.y + c3 * weights.z + d3 * weights.w;
	float one = dot(weights, vec4(1.0));
	return mat4(
		v1.x, v2.x, v3.x, 0,
		v1.y, v2.y, v3.y, 0,
		v1.z, v2.z, v3.z, 0,
		v1.w, v2.w, v3.w, one
	);
}
`;

var skinTexVS = `
attribute vec4 vertex_boneWeights;
attribute vec4 vertex_boneIndices;
uniform highp sampler2D texture_poseMap;
uniform vec4 texture_poseMapSize;
void getBoneMatrix(const in float index, out vec4 v1, out vec4 v2, out vec4 v3) {
	float i = float(index);
	float j = i * 3.0;
	float dx = texture_poseMapSize.z;
	float dy = texture_poseMapSize.w;
	
	float y = floor(j * dx);
	float x = j - (y * texture_poseMapSize.x);
	y = dy * (y + 0.5);
	v1 = texture2D(texture_poseMap, vec2(dx * (x + 0.5), y));
	v2 = texture2D(texture_poseMap, vec2(dx * (x + 1.5), y));
	v3 = texture2D(texture_poseMap, vec2(dx * (x + 2.5), y));
}
mat4 getSkinMatrix(const in vec4 indices, const in vec4 weights) {
	vec4 a1, a2, a3;
	getBoneMatrix(indices.x, a1, a2, a3);
	vec4 b1, b2, b3;
	getBoneMatrix(indices.y, b1, b2, b3);
	vec4 c1, c2, c3;
	getBoneMatrix(indices.z, c1, c2, c3);
	vec4 d1, d2, d3;
	getBoneMatrix(indices.w, d1, d2, d3);
	vec4 v1 = a1 * weights.x + b1 * weights.y + c1 * weights.z + d1 * weights.w;
	vec4 v2 = a2 * weights.x + b2 * weights.y + c2 * weights.z + d2 * weights.w;
	vec4 v3 = a3 * weights.x + b3 * weights.y + c3 * weights.z + d3 * weights.w;
	float one = dot(weights, vec4(1.0));
	return mat4(
		v1.x, v2.x, v3.x, 0,
		v1.y, v2.y, v3.y, 0,
		v1.z, v2.z, v3.z, 0,
		v1.w, v2.w, v3.w, one
	);
}
`;

var skyboxEnvPS = `
varying vec3 vViewDir;
uniform sampler2D texture_envAtlas;
uniform float mipLevel;
void main(void) {
	vec3 dir = vViewDir * vec3(-1.0, 1.0, 1.0);
	vec2 uv = toSphericalUv(normalize(dir));
	vec3 linear = $DECODE(texture2D(texture_envAtlas, mapRoughnessUv(uv, mipLevel)));
	gl_FragColor = vec4(gammaCorrectOutput(toneMap(processEnvironment(linear))), 1.0);
}
`;

var skyboxHDRPS = `
varying vec3 vViewDir;
uniform samplerCube texture_cubeMap;
#ifdef SKYMESH
	varying vec3 vWorldPos;
	uniform mat3 cubeMapRotationMatrix;
	uniform vec3 projectedSkydomeCenter;
#endif
void main(void) {
	#ifdef SKYMESH
		vec3 envDir = normalize(vWorldPos - projectedSkydomeCenter);
		vec3 dir = envDir * cubeMapRotationMatrix;
	#else
		vec3 dir = vViewDir;
	#endif
	dir.x *= -1.0;
	vec3 linear = $DECODE(textureCube(texture_cubeMap, fixSeamsStatic(dir, $FIXCONST)));
	gl_FragColor = vec4(gammaCorrectOutput(toneMap(processEnvironment(linear))), 1.0);
}
`;

var skyboxVS = `
attribute vec3 aPosition;
#ifndef VIEWMATRIX
#define VIEWMATRIX
uniform mat4 matrix_view;
#endif
uniform mat4 matrix_projectionSkybox;
uniform mat3 cubeMapRotationMatrix;
varying vec3 vViewDir;
#ifdef SKYMESH
	uniform mat4 matrix_model;
	varying vec3 vWorldPos;
#endif
void main(void) {
	mat4 view = matrix_view;
	#ifdef SKYMESH
		vec4 worldPos = matrix_model * vec4(aPosition, 1.0);
		vWorldPos = worldPos.xyz;
		gl_Position = matrix_projectionSkybox * view * worldPos;
	#else
		view[3][0] = view[3][1] = view[3][2] = 0.0;
		gl_Position = matrix_projectionSkybox * view * vec4(aPosition, 1.0);
		vViewDir = aPosition * cubeMapRotationMatrix;
	#endif
	gl_Position.z = gl_Position.w - 0.00001;
}
`;

var specularPS = `
#ifdef MAPCOLOR
uniform vec3 material_specular;
#endif
void getSpecularity() {
	vec3 specularColor = vec3(1,1,1);
	#ifdef MAPCOLOR
	specularColor *= material_specular;
	#endif
	#ifdef MAPTEXTURE
	specularColor *= $DECODE(texture2DBias($SAMPLER, $UV, textureBias)).$CH;
	#endif
	#ifdef MAPVERTEX
	specularColor *= saturate(vVertexColor.$VC);
	#endif
	dSpecularity = specularColor;
}
`;

var sphericalPS = `
const float PI = 3.141592653589793;
vec2 toSpherical(vec3 dir) {
	return vec2(dir.xz == vec2(0.0) ? 0.0 : atan(dir.x, dir.z), asin(dir.y));
}
vec2 toSphericalUv(vec3 dir) {
	vec2 uv = toSpherical(dir) / vec2(PI * 2.0, PI) + 0.5;
	return vec2(uv.x, 1.0 - uv.y);
}
`;

var specularityFactorPS = `
#ifdef MAPFLOAT
uniform float material_specularityFactor;
#endif
void getSpecularityFactor() {
	float specularityFactor = 1.0;
	#ifdef MAPFLOAT
	specularityFactor *= material_specularityFactor;
	#endif
	#ifdef MAPTEXTURE
	specularityFactor *= texture2DBias($SAMPLER, $UV, textureBias).$CH;
	#endif
	#ifdef MAPVERTEX
	specularityFactor *= saturate(vVertexColor.$VC);
	#endif
	dSpecularityFactor = specularityFactor;
}
`;

var spotPS = `
float getSpotEffect(vec3 lightSpotDir, float lightInnerConeAngle, float lightOuterConeAngle, vec3 lightDirNorm) {
	float cosAngle = dot(lightDirNorm, lightSpotDir);
	return smoothstep(lightOuterConeAngle, lightInnerConeAngle, cosAngle);
}
`;

var startPS = `
void main(void) {
	dReflection = vec4(0);
	#ifdef LIT_CLEARCOAT
	ccSpecularLight = vec3(0);
	ccReflection = vec3(0);
	#endif
`;

var startVS = `
void main(void) {
	gl_Position = getPosition();
`;

var startNineSlicedPS = `
	nineSlicedUv = vUv0;
	nineSlicedUv.y = 1.0 - nineSlicedUv.y;
`;

var startNineSlicedTiledPS = `
	vec2 tileMask = step(vMask, vec2(0.99999));
	vec2 tileSize = 0.5 * (innerOffset.xy + innerOffset.zw);
	vec2 tileScale = vec2(1.0) / (vec2(1.0) - tileSize);
	vec2 clampedUv = mix(innerOffset.xy * 0.5, vec2(1.0) - innerOffset.zw * 0.5, fract((vTiledUv - tileSize) * tileScale));
	clampedUv = clampedUv * atlasRect.zw + atlasRect.xy;
	nineSlicedUv = vUv0 * tileMask + clampedUv * (vec2(1.0) - tileMask);
	nineSlicedUv.y = 1.0 - nineSlicedUv.y;
	
`;

var storeEVSMPS = `
float exponent = VSM_EXPONENT;
depth = 2.0 * depth - 1.0;
depth =  exp(exponent * depth);
gl_FragColor = vec4(depth, depth*depth, 1.0, 1.0);
`;

var tangentBinormalVS = `
vec3 getTangent() {
	return normalize(dNormalMatrix * vertex_tangent.xyz);
}
vec3 getBinormal() {
	return cross(vNormalW, vTangentW) * vertex_tangent.w;
}
`;

var TBNPS = `
void getTBN(vec3 tangent, vec3 binormal, vec3 normal) {
	dTBN = mat3(normalize(tangent), normalize(binormal), normalize(normal));
}
`;

var TBNderivativePS = `
uniform float tbnBasis;
void getTBN(vec3 tangent, vec3 binormal, vec3 normal) {
	vec2 uv = $UV;
	vec3 dp1 = dFdx( vPositionW );
	vec3 dp2 = dFdy( vPositionW );
	vec2 duv1 = dFdx( uv );
	vec2 duv2 = dFdy( uv );
	vec3 dp2perp = cross( dp2, normal );
	vec3 dp1perp = cross( normal, dp1 );
	vec3 T = dp2perp * duv1.x + dp1perp * duv2.x;
	vec3 B = dp2perp * duv1.y + dp1perp * duv2.y;
	float denom = max( dot(T,T), dot(B,B) );
	float invmax = (denom == 0.0) ? 0.0 : tbnBasis / sqrt( denom );
	dTBN = mat3(T * invmax, -B * invmax, normal );
}
`;

var TBNfastPS = `
void getTBN(vec3 tangent, vec3 binormal, vec3 normal) {
	dTBN = mat3(tangent, binormal, normal);
}
`;

var TBNObjectSpacePS = `
void getTBN(vec3 tangent, vec3 binormal, vec3 normal) {
	vec3 B = cross(normal, vObjectSpaceUpW);
	vec3 T = cross(normal, B);
	if (dot(B,B)==0.0)
	{
		float major=max(max(normal.x, normal.y), normal.z);
		if (normal.x == major)
		{
			B=cross(normal, vec3(0,1,0));
			T=cross(normal, B);
		}
		else if (normal.y == major)
		{
			B=cross(normal, vec3(0,0,1));
			T=cross(normal, B);
		}
		else if (normal.z == major)
		{
			B=cross(normal, vec3(1,0,0));
			T=cross(normal, B);
		}
	}
	dTBN = mat3(normalize(T), normalize(B), normalize(normal));
}
`;

var textureSamplePS = `
vec4 texture2DSRGB(sampler2D tex, vec2 uv) {
	return gammaCorrectInput(texture2D(tex, uv));
}
vec4 texture2DSRGB(sampler2D tex, vec2 uv, float bias) {
	return gammaCorrectInput(texture2D(tex, uv, bias));
}
vec3 texture2DRGBM(sampler2D tex, vec2 uv) {
	return decodeRGBM(texture2D(tex, uv));
}
vec3 texture2DRGBM(sampler2D tex, vec2 uv, float bias) {
	return decodeRGBM(texture2D(tex, uv, bias));
}
vec3 texture2DRGBE(sampler2D tex, vec2 uv) {
	return decodeRGBM(texture2D(tex, uv));
}
vec3 texture2DRGBE(sampler2D tex, vec2 uv, float bias) {
	return decodeRGBM(texture2D(tex, uv, bias));
}
`;

var thicknessPS = `
#ifdef MAPFLOAT
uniform float material_thickness;
#endif
void getThickness() {
	dThickness = 1.0;
	#ifdef MAPFLOAT
	dThickness *= material_thickness;
	#endif
	#ifdef MAPTEXTURE
	dThickness *= texture2DBias($SAMPLER, $UV, textureBias).$CH;
	#endif
	#ifdef MAPVERTEX
	dThickness *= saturate(vVertexColor.$VC);
	#endif
}
`;

var tonemappingAcesPS = `
uniform float exposure;
vec3 toneMap(vec3 color) {
	float tA = 2.51;
	float tB = 0.03;
	float tC = 2.43;
	float tD = 0.59;
	float tE = 0.14;
	vec3 x = color * exposure;
	return (x*(tA*x+tB))/(x*(tC*x+tD)+tE);
}
`;

var tonemappingAces2PS = `
uniform float exposure;
const mat3 ACESInputMat = mat3(
	0.59719, 0.35458, 0.04823,
	0.07600, 0.90834, 0.01566,
	0.02840, 0.13383, 0.83777
);
const mat3 ACESOutputMat = mat3(
	 1.60475, -0.53108, -0.07367,
	-0.10208,  1.10813, -0.00605,
	-0.00327, -0.07276,  1.07602
);
vec3 RRTAndODTFit(vec3 v) {
	vec3 a = v * (v + 0.0245786) - 0.000090537;
	vec3 b = v * (0.983729 * v + 0.4329510) + 0.238081;
	return a / b;
}
vec3 toneMap(vec3 color) {
	color *= exposure / 0.6;
	color = color * ACESInputMat;
	color = RRTAndODTFit(color);
	color = color * ACESOutputMat;
	color = clamp(color, 0.0, 1.0);
	return color;
}
`;

var tonemappingFilmicPS = `
const float A =  0.15;
const float B =  0.50;
const float C =  0.10;
const float D =  0.20;
const float E =  0.02;
const float F =  0.30;
const float W =  11.2;
uniform float exposure;
vec3 uncharted2Tonemap(vec3 x) {
	 return ((x*(A*x+C*B)+D*E)/(x*(A*x+B)+D*F))-E/F;
}
vec3 toneMap(vec3 color) {
	color = uncharted2Tonemap(color * exposure);
	vec3 whiteScale = 1.0 / uncharted2Tonemap(vec3(W,W,W));
	color = color * whiteScale;
	return color;
}
`;

var tonemappingHejlPS = `
uniform float exposure;
vec3 toneMap(vec3 color) {
	color *= exposure;
	const float  A = 0.22, B = 0.3, C = .1, D = 0.2, E = .01, F = 0.3;
	const float Scl = 1.25;
	vec3 h = max( vec3(0.0), color - vec3(0.004) );
	return (h*((Scl*A)*h+Scl*vec3(C*B,C*B,C*B))+Scl*vec3(D*E,D*E,D*E)) / (h*(A*h+vec3(B,B,B))+vec3(D*F,D*F,D*F)) - Scl*vec3(E/F,E/F,E/F);
}
`;

var tonemappingLinearPS = `
uniform float exposure;
vec3 toneMap(vec3 color) {
	return color * exposure;
}
`;

var tonemappingNonePS = `
vec3 toneMap(vec3 color) {
	return color;
}
`;

var transformVS = `
#ifdef PIXELSNAP
uniform vec4 uScreenSize;
#endif
#ifdef SCREENSPACE
uniform float projectionFlipY;
#endif
#ifdef MORPHING
uniform vec4 morph_weights_a;
uniform vec4 morph_weights_b;
#endif
#ifdef MORPHING_TEXTURE_BASED
	uniform vec4 morph_tex_params;
	#ifdef WEBGPU
		ivec2 getTextureMorphCoords() {
			ivec2 textureSize = ivec2(morph_tex_params.xy);
			int morphGridV = int(morph_vertex_id / textureSize.x);
			int morphGridU = int(morph_vertex_id - (morphGridV * textureSize.x));
			morphGridV = textureSize.y - morphGridV - 1;
			return ivec2(morphGridU, morphGridV);
		}
	#else
		vec2 getTextureMorphCoords() {
			vec2 textureSize = morph_tex_params.xy;
			vec2 invTextureSize = morph_tex_params.zw;
			float morphGridV = floor(morph_vertex_id * invTextureSize.x);
			float morphGridU = morph_vertex_id - (morphGridV * textureSize.x);
			return vec2(morphGridU, morphGridV) * invTextureSize + (0.5 * invTextureSize);
		}
	#endif
#endif
#ifdef MORPHING_TEXTURE_BASED_POSITION
uniform highp sampler2D morphPositionTex;
#endif
mat4 getModelMatrix() {
	#ifdef DYNAMICBATCH
	return getBoneMatrix(vertex_boneIndices);
	#elif defined(SKIN)
	return matrix_model * getSkinMatrix(vertex_boneIndices, vertex_boneWeights);
	#elif defined(INSTANCING)
	return mat4(instance_line1, instance_line2, instance_line3, instance_line4);
	#else
	return matrix_model;
	#endif
}
vec4 getPosition() {
	dModelMatrix = getModelMatrix();
	vec3 localPos = vertex_position;
	#ifdef NINESLICED
	localPos.xz *= outerScale;
	vec2 positiveUnitOffset = clamp(vertex_position.xz, vec2(0.0), vec2(1.0));
	vec2 negativeUnitOffset = clamp(-vertex_position.xz, vec2(0.0), vec2(1.0));
	localPos.xz += (-positiveUnitOffset * innerOffset.xy + negativeUnitOffset * innerOffset.zw) * vertex_texCoord0.xy;
	vTiledUv = (localPos.xz - outerScale + innerOffset.xy) * -0.5 + 1.0;
	localPos.xz *= -0.5;
	localPos = localPos.xzy;
	#endif
	#ifdef MORPHING
	#ifdef MORPHING_POS03
	localPos.xyz += morph_weights_a[0] * morph_pos0;
	localPos.xyz += morph_weights_a[1] * morph_pos1;
	localPos.xyz += morph_weights_a[2] * morph_pos2;
	localPos.xyz += morph_weights_a[3] * morph_pos3;
	#endif
	#ifdef MORPHING_POS47
	localPos.xyz += morph_weights_b[0] * morph_pos4;
	localPos.xyz += morph_weights_b[1] * morph_pos5;
	localPos.xyz += morph_weights_b[2] * morph_pos6;
	localPos.xyz += morph_weights_b[3] * morph_pos7;
	#endif
	#endif
	#ifdef MORPHING_TEXTURE_BASED_POSITION
		#ifdef WEBGPU
			ivec2 morphUV = getTextureMorphCoords();
			vec3 morphPos = texelFetch(morphPositionTex, ivec2(morphUV), 0).xyz;
		#else
			vec2 morphUV = getTextureMorphCoords();
			vec3 morphPos = texture2D(morphPositionTex, morphUV).xyz;
		#endif
		localPos += morphPos;
	#endif
	vec4 posW = dModelMatrix * vec4(localPos, 1.0);
	#ifdef SCREENSPACE
	posW.zw = vec2(0.0, 1.0);
	#endif
	dPositionW = posW.xyz;
	vec4 screenPos;
	#ifdef UV1LAYOUT
	screenPos = vec4(vertex_texCoord1.xy * 2.0 - 1.0, 0.5, 1);
		#ifdef WEBGPU
		screenPos.y *= -1.0;
		#endif
	#else
	#ifdef SCREENSPACE
	screenPos = posW;
	screenPos.y *= projectionFlipY;
	#else
	screenPos = matrix_viewProjection * posW;
	#endif
	#ifdef PIXELSNAP
	screenPos.xy = (screenPos.xy * 0.5) + 0.5;
	screenPos.xy *= uScreenSize.xy;
	screenPos.xy = floor(screenPos.xy);
	screenPos.xy *= uScreenSize.zw;
	screenPos.xy = (screenPos.xy * 2.0) - 1.0;
	#endif
	#endif
	return screenPos;
}
vec3 getWorldPosition() {
	return dPositionW;
}
`;

var transformDeclVS = `
attribute vec3 vertex_position;
uniform mat4 matrix_model;
uniform mat4 matrix_viewProjection;
vec3 dPositionW;
mat4 dModelMatrix;
`;

var transmissionPS = `
#ifdef MAPFLOAT
uniform float material_refraction;
#endif
void getRefraction() {
	float refraction = 1.0;
	#ifdef MAPFLOAT
	refraction = material_refraction;
	#endif
	#ifdef MAPTEXTURE
	refraction *= texture2DBias($SAMPLER, $UV, textureBias).$CH;
	#endif
	#ifdef MAPVERTEX
	refraction *= saturate(vVertexColor.$VC);
	#endif
	dTransmission = refraction;
}
`;

var uv0VS = `
#ifdef NINESLICED
vec2 getUv0() {
	vec2 uv = vertex_position.xz;
	vec2 positiveUnitOffset = clamp(vertex_position.xz, vec2(0.0), vec2(1.0));
	vec2 negativeUnitOffset = clamp(-vertex_position.xz, vec2(0.0), vec2(1.0));
	uv += (-positiveUnitOffset * innerOffset.xy + negativeUnitOffset * innerOffset.zw) * vertex_texCoord0.xy;
	uv = uv * -0.5 + 0.5;
	uv = uv * atlasRect.zw + atlasRect.xy;
	vMask = vertex_texCoord0.xy;
	return uv;
}
#else
vec2 getUv0() {
	return vertex_texCoord0;
}
#endif
`;

var uv1VS = `
vec2 getUv1() {
	return vertex_texCoord1;
}
`;

var viewDirPS = `
void getViewDir() {
	dViewDirW = normalize(view_position - vPositionW);
}
`;

var viewNormalVS = `
#ifndef VIEWMATRIX
#define VIEWMATRIX
uniform mat4 matrix_view;
#endif
vec3 getViewNormal() {
	return mat3(matrix_view) * vNormalW;
}
`;

const shaderChunks = {
	alphaTestPS,
	ambientConstantPS,
	ambientEnvPS,
	ambientSHPS,
	aoPS,
	aoDetailMapPS,
	aoDiffuseOccPS,
	aoSpecOccPS,
	aoSpecOccConstPS,
	aoSpecOccConstSimplePS,
	aoSpecOccSimplePS,
	basePS,
	baseVS,
	baseNineSlicedPS,
	baseNineSlicedVS,
	baseNineSlicedTiledPS,
	bayerPS,
	biasConstPS,
	blurVSMPS,
	clearCoatPS,
	clearCoatGlossPS,
	clearCoatNormalPS,
	clusteredLightCookiesPS,
	clusteredLightShadowsPS,
	clusteredLightUtilsPS,
	clusteredLightPS,
	combinePS,
	cookiePS,
	cubeMapProjectBoxPS,
	cubeMapProjectNonePS,
	cubeMapRotatePS,
	debugOutputPS,
	debugProcessFrontendPS,
	detailModesPS,
	diffusePS,
	diffuseDetailMapPS,
	decodePS,
	emissivePS,
	encodePS,
	endPS,
	endVS,
	envAtlasPS,
	envConstPS,
	envMultiplyPS,
	extensionPS,
	extensionVS,
	falloffInvSquaredPS,
	falloffLinearPS,
	fixCubemapSeamsNonePS,
	fixCubemapSeamsStretchPS,
	floatUnpackingPS,
	fogExpPS,
	fogExp2PS,
	fogLinearPS,
	fogNonePS,
	fresnelSchlickPS,
	fullscreenQuadPS,
	fullscreenQuadVS,
	gamma1_0PS,
	gamma2_2PS,
	gles2PS,
	gles3PS,
	gles3VS,
	glossPS,
	iridescenceDiffractionPS,
	iridescencePS,
	iridescenceThicknessPS,
	instancingVS,
	iorPS,
	lightDiffuseLambertPS,
	lightDirPointPS,
	lightmapAddPS,
	lightmapDirAddPS,
	lightmapDirPS,
	lightmapSinglePS,
	lightSpecularAnisoGGXPS,
	lightSpecularBlinnPS,
	lightSpecularPhongPS,
	lightSheenPS,
	linearizeDepthPS,
	litShaderArgsPS,
	ltcPS,
	metalnessPS,
	metalnessModulatePS,
	msdfPS,
	msdfVS,
	normalVS,
	normalDetailMapPS,
	normalInstancedVS,
	normalMapPS,
	normalSkinnedVS,
	normalXYPS,
	normalXYZPS,
	opacityPS,
	opacityDitherPS,
	outputPS,
	outputAlphaPS,
	outputAlphaOpaquePS,
	outputAlphaPremulPS,
	outputTex2DPS,
	packDepthPS,
	sheenPS,
	sheenGlossPS,
	parallaxPS,
	particlePS,
	particleVS,
	particleAnimFrameClampVS,
	particleAnimFrameLoopVS,
	particleAnimTexVS,
	particleInputFloatPS,
	particleInputRgba8PS,
	particleOutputFloatPS,
	particleOutputRgba8PS,
	particleUpdaterAABBPS,
	particleUpdaterEndPS,
	particleUpdaterInitPS,
	particleUpdaterNoRespawnPS,
	particleUpdaterOnStopPS,
	particleUpdaterRespawnPS,
	particleUpdaterSpherePS,
	particleUpdaterStartPS,
	particle_billboardVS,
	particle_blendAddPS,
	particle_blendMultiplyPS,
	particle_blendNormalPS,
	particle_cpuVS,
	particle_cpu_endVS,
	particle_customFaceVS,
	particle_endPS,
	particle_endVS,
	particle_halflambertPS,
	particle_initVS,
	particle_lambertPS,
	particle_lightingPS,
	particle_localShiftVS,
	particle_meshVS,
	particle_normalVS,
	particle_normalMapPS,
	particle_pointAlongVS,
	particle_softPS,
	particle_softVS,
	particle_stretchVS,
	particle_TBNVS,
	particle_wrapVS,
	reflDirPS,
	reflDirAnisoPS,
	reflectionCCPS,
	reflectionCubePS,
	reflectionEnvHQPS,
	reflectionEnvPS,
	reflectionSpherePS,
	reflectionSheenPS,
	refractionCubePS,
	refractionDynamicPS,
	reprojectPS,
	sampleCatmullRomPS,
	screenDepthPS,
	shadowCascadesPS,
	shadowEVSMPS,
	shadowEVSMnPS,
	shadowPCSSPS,
	shadowSampleCoordPS,
	shadowStandardPS,
	shadowStandardGL2PS,
	shadowVSM8PS,
	shadowVSM_commonPS,
	skinBatchConstVS,
	skinBatchTexVS,
	skinConstVS,
	skinTexVS,
	skyboxEnvPS,
	skyboxHDRPS,
	skyboxVS,
	specularPS,
	sphericalPS,
	specularityFactorPS,
	spotPS,
	startPS,
	startVS,
	startNineSlicedPS,
	startNineSlicedTiledPS,
	storeEVSMPS,
	tangentBinormalVS,
	TBNPS,
	TBNderivativePS,
	TBNfastPS,
	TBNObjectSpacePS,
	textureSamplePS,
	thicknessPS,
	tonemappingAcesPS,
	tonemappingAces2PS,
	tonemappingFilmicPS,
	tonemappingHejlPS,
	tonemappingLinearPS,
	tonemappingNonePS,
	transformVS,
	transformDeclVS,
	transmissionPS,
	uv0VS,
	uv1VS,
	viewDirPS,
	viewNormalVS,
	webgpuPS,
	webgpuVS
};

const programLibraryDeviceCache = new DeviceCache();
function getProgramLibrary(device) {
	const library = programLibraryDeviceCache.get(device);
	return library;
}
function setProgramLibrary(device, library) {
	programLibraryDeviceCache.get(device, () => {
		return library;
	});
}

class ShaderGenerator {
	static begin() {
		return 'void main(void)\n{\n';
	}
	static end() {
		return '}\n';
	}
	static skinCode(device, chunks = shaderChunks) {
		if (device.supportsBoneTextures) {
			return chunks.skinTexVS;
		}
		return "#define BONE_LIMIT " + device.getBoneLimit() + "\n" + chunks.skinConstVS;
	}
	static fogCode(value, chunks = shaderChunks) {
		if (value === 'linear') {
			return chunks.fogLinearPS ? chunks.fogLinearPS : shaderChunks.fogLinearPS;
		} else if (value === 'exp') {
			return chunks.fogExpPS ? chunks.fogExpPS : shaderChunks.fogExpPS;
		} else if (value === 'exp2') {
			return chunks.fogExp2PS ? chunks.fogExp2PS : shaderChunks.fogExp2PS;
		}
		return chunks.fogNonePS ? chunks.fogNonePS : shaderChunks.fogNonePS;
	}
	static gammaCode(value, chunks = shaderChunks) {
		if (value === GAMMA_SRGB || value === GAMMA_SRGBFAST) {
			return chunks.gamma2_2PS ? chunks.gamma2_2PS : shaderChunks.gamma2_2PS;
		} else if (value === GAMMA_SRGBHDR) {
			return "#define HDR\n" + (chunks.gamma2_2PS ? chunks.gamma2_2PS : shaderChunks.gamma2_2PS);
		}
		return chunks.gamma1_0PS ? chunks.gamma1_0PS : shaderChunks.gamma1_0PS;
	}
	static tonemapCode(value, chunks = shaderChunks) {
		if (value === TONEMAP_FILMIC) {
			return chunks.tonemappingFilmicPS ? chunks.tonemappingFilmicPS : shaderChunks.tonemappingFilmicPS;
		} else if (value === TONEMAP_LINEAR) {
			return chunks.tonemappingLinearPS ? chunks.tonemappingLinearPS : shaderChunks.tonemappingLinearPS;
		} else if (value === TONEMAP_HEJL) {
			return chunks.tonemappingHejlPS ? chunks.tonemappingHejlPS : shaderChunks.tonemappingHejlPS;
		} else if (value === TONEMAP_ACES) {
			return chunks.tonemappingAcesPS ? chunks.tonemappingAcesPS : shaderChunks.tonemappingAcesPS;
		} else if (value === TONEMAP_ACES2) {
			return chunks.tonemappingAces2PS ? chunks.tonemappingAces2PS : shaderChunks.tonemappingAces2PS;
		}
		return chunks.tonemapingNonePS ? chunks.tonemapingNonePS : shaderChunks.tonemappingNonePS;
	}
}

function createShader(device, vsName, fsName, useTransformFeedback = false, shaderDefinitionOptions = {}) {
	if (typeof useTransformFeedback === 'boolean') {
		shaderDefinitionOptions.useTransformFeedback = useTransformFeedback;
	} else if (typeof useTransformFeedback === 'object') {
		shaderDefinitionOptions = _extends({}, shaderDefinitionOptions, useTransformFeedback);
	}
	return new Shader(device, ShaderUtils.createDefinition(device, _extends({}, shaderDefinitionOptions, {
		name: `${vsName}_${fsName}`,
		vertexCode: shaderChunks[vsName],
		fragmentCode: shaderChunks[fsName]
	})));
}
function createShaderFromCode(device, vsCode, fsCode, uniqueName, attributes, useTransformFeedback = false, shaderDefinitionOptions = {}) {
	if (typeof useTransformFeedback === 'boolean') {
		shaderDefinitionOptions.useTransformFeedback = useTransformFeedback;
	} else if (typeof useTransformFeedback === 'object') {
		shaderDefinitionOptions = _extends({}, shaderDefinitionOptions, useTransformFeedback);
	}
	const programLibrary = getProgramLibrary(device);
	let shader = programLibrary.getCachedShader(uniqueName);
	if (!shader) {
		shader = new Shader(device, ShaderUtils.createDefinition(device, _extends({}, shaderDefinitionOptions, {
			name: uniqueName,
			vertexCode: vsCode,
			fragmentCode: fsCode,
			attributes: attributes
		})));
		programLibrary.setCachedShader(uniqueName, shader);
	}
	return shader;
}
class ShaderGeneratorPassThrough extends ShaderGenerator {
	constructor(key, shaderDefinition) {
		super();
		this.key = key;
		this.shaderDefinition = shaderDefinition;
	}
	generateKey(options) {
		return this.key;
	}
	createShaderDefinition(device, options) {
		return this.shaderDefinition;
	}
}
function processShader(shader, processingOptions) {
	var _shaderDefinition$nam;
	const shaderDefinition = shader.definition;
	const name = (_shaderDefinition$nam = shaderDefinition.name) != null ? _shaderDefinition$nam : 'shader';
	const key = `${name}-id-${shader.id}`;
	const materialGenerator = new ShaderGeneratorPassThrough(key, shaderDefinition);
	const libraryModuleName = 'shader';
	const library = getProgramLibrary(shader.device);
	library.register(libraryModuleName, materialGenerator);
	const variant = library.getProgram(libraryModuleName, {}, processingOptions);
	if (shader.definition.shaderLanguage === SHADERLANGUAGE_WGSL) {
		variant.meshUniformBufferFormat = shaderDefinition.meshUniformBufferFormat;
		variant.meshBindGroupFormat = shaderDefinition.meshBindGroupFormat;
	}
	library.unregister(libraryModuleName);
	return variant;
}
shaderChunks.createShader = createShader;
shaderChunks.createShaderFromCode = createShaderFromCode;

const _quadPrimitive = {
	type: PRIMITIVE_TRISTRIP,
	base: 0,
	count: 4,
	indexed: false
};
const _tempViewport = new Vec4();
const _tempScissor = new Vec4();
class QuadRender {
	constructor(shader) {
		this.uniformBuffer = void 0;
		this.bindGroup = void 0;
		const device = shader.device;
		this.shader = shader;
		if (device.supportsUniformBuffers) {
			const processingOptions = new ShaderProcessorOptions();
			this.shader = processShader(shader, processingOptions);
			const ubFormat = this.shader.meshUniformBufferFormat;
			if (ubFormat) {
				this.uniformBuffer = new UniformBuffer(device, ubFormat, false);
			}
			const bindGroupFormat = this.shader.meshBindGroupFormat;
			this.bindGroup = new BindGroup(device, bindGroupFormat, this.uniformBuffer);
		}
	}
	destroy() {
		var _this$uniformBuffer, _this$bindGroup;
		(_this$uniformBuffer = this.uniformBuffer) == null || _this$uniformBuffer.destroy();
		this.uniformBuffer = null;
		(_this$bindGroup = this.bindGroup) == null || _this$bindGroup.destroy();
		this.bindGroup = null;
	}
	render(viewport, scissor) {
		const device = this.shader.device;
		if (viewport) {
			var _scissor;
			_tempViewport.set(device.vx, device.vy, device.vw, device.vh);
			_tempScissor.set(device.sx, device.sy, device.sw, device.sh);
			scissor = (_scissor = scissor) != null ? _scissor : viewport;
			device.setViewport(viewport.x, viewport.y, viewport.z, viewport.w);
			device.setScissor(scissor.x, scissor.y, scissor.z, scissor.w);
		}
		device.setVertexBuffer(device.quadVertexBuffer, 0);
		const shader = this.shader;
		device.setShader(shader);
		if (device.supportsUniformBuffers) {
			var _bindGroup$defaultUni;
			const bindGroup = this.bindGroup;
			(_bindGroup$defaultUni = bindGroup.defaultUniformBuffer) == null || _bindGroup$defaultUni.update();
			bindGroup.update();
			device.setBindGroup(BINDGROUP_MESH, bindGroup);
		}
		device.draw(_quadPrimitive);
		if (viewport) {
			device.setViewport(_tempViewport.x, _tempViewport.y, _tempViewport.z, _tempViewport.w);
			device.setScissor(_tempScissor.x, _tempScissor.y, _tempScissor.z, _tempScissor.w);
		}
	}
}

class RenderPassQuad extends RenderPass {
	constructor(device, quad, rect, scissorRect) {
		super(device);
		this.quad = quad;
		this.rect = rect;
		this.scissorRect = scissorRect;
	}
	execute() {
		const {
			device
		} = this;
		device.setCullMode(CULLFACE_NONE);
		device.setDepthState(DepthState.NODEPTH);
		device.setStencilState(null, null);
		this.quad.render(this.rect, this.scissorRect);
	}
}

const _tempRect = new Vec4();
function drawQuadWithShader(device, target, shader, rect, scissorRect) {
	const quad = new QuadRender(shader);
	if (!rect) {
		rect = _tempRect;
		rect.x = 0;
		rect.y = 0;
		rect.z = target ? target.width : device.width;
		rect.w = target ? target.height : device.height;
	}
	const renderPass = new RenderPassQuad(device, quad, rect, scissorRect);
	renderPass.init(target);
	renderPass.colorOps.clear = false;
	renderPass.depthStencilOps.clearDepth = false;
	if (device.isWebGPU && target === null && device.samples > 1) {
		renderPass.colorOps.store = true;
	}
	renderPass.render();
	quad.destroy();
}
function drawTexture(device, texture, target, shader, rect, scissorRect) {
	shader = shader || device.getCopyShader();
	device.constantTexSource.setValue(texture);
	drawQuadWithShader(device, target, shader, rect, scissorRect);
}

const shaderPassDeviceCache = new DeviceCache();
class ShaderPassInfo {
	constructor(name, index, options = {}) {
		this.index = void 0;
		this.name = void 0;
		this.shaderDefines = void 0;
		this.name = name;
		this.index = index;
		Object.assign(this, options);
		this.shaderDefines = this.buildShaderDefines();
	}
	buildShaderDefines() {
		let keyword;
		if (this.isShadow) {
			keyword = 'SHADOW';
		} else if (this.isForward) {
			keyword = 'FORWARD';
		} else if (this.index === SHADER_DEPTH) {
			keyword = 'DEPTH';
		} else if (this.index === SHADER_PICK) {
			keyword = 'PICK';
		}
		const define1 = keyword ? `#define ${keyword}_PASS\n` : '';
		const define2 = `#define ${this.name.toUpperCase()}_PASS\n`;
		return define1 + define2;
	}
}
class ShaderPass {
	constructor() {
		this.passesNamed = new Map();
		this.passesIndexed = [];
		this.nextIndex = 0;
		const add = (name, index, options) => {
			this.allocate(name, options);
		};
		add('forward', SHADER_FORWARD, {
			isForward: true
		});
		add('forward_hdr', SHADER_FORWARDHDR, {
			isForward: true
		});
		add('depth');
		add('pick');
		add('shadow');
		add('prepass');
	}
	static get(device) {
		return shaderPassDeviceCache.get(device, () => {
			return new ShaderPass();
		});
	}
	allocate(name, options) {
		let info = this.passesNamed.get(name);
		if (info === undefined) {
			info = new ShaderPassInfo(name, this.nextIndex, options);
			this.passesNamed.set(info.name, info);
			this.passesIndexed[info.index] = info;
			this.nextIndex++;
		}
		return info;
	}
	getByIndex(index) {
		const info = this.passesIndexed[index];
		return info;
	}
	getByName(name) {
		return this.passesNamed.get(name);
	}
}

class ShaderGeneratorBasic extends ShaderGenerator {
	generateKey(options) {
		let key = 'basic';
		if (options.fog) key += '_fog';
		if (options.alphaTest) key += '_atst';
		if (options.vertexColors) key += '_vcol';
		if (options.diffuseMap) key += '_diff';
		if (options.skin) key += '_skin';
		if (options.screenSpace) key += '_ss';
		if (options.useInstancing) key += '_inst';
		if (options.useMorphPosition) key += '_morphp';
		if (options.useMorphNormal) key += '_morphn';
		if (options.useMorphTextureBased) key += '_morpht';
		key += '_' + options.pass;
		return key;
	}
	createShaderDefinition(device, options) {
		const attributes = {
			vertex_position: SEMANTIC_POSITION
		};
		if (options.skin) {
			attributes.vertex_boneWeights = SEMANTIC_BLENDWEIGHT;
			attributes.vertex_boneIndices = SEMANTIC_BLENDINDICES;
		}
		if (options.vertexColors) {
			attributes.vertex_color = SEMANTIC_COLOR;
		}
		if (options.diffuseMap) {
			attributes.vertex_texCoord0 = SEMANTIC_TEXCOORD0;
		}
		const shaderPassInfo = ShaderPass.get(device).getByIndex(options.pass);
		const shaderPassDefines = shaderPassInfo.shaderDefines;
		let vshader = shaderPassDefines;
		vshader += shaderChunks.transformDeclVS;
		if (options.skin) {
			vshader += ShaderGenerator.skinCode(device);
			vshader += shaderChunks.transformSkinnedVS;
		} else {
			vshader += shaderChunks.transformVS;
		}
		if (options.vertexColors) {
			vshader += 'attribute vec4 vertex_color;\n';
			vshader += 'varying vec4 vColor;\n';
		}
		if (options.diffuseMap) {
			vshader += 'attribute vec2 vertex_texCoord0;\n';
			vshader += 'varying vec2 vUv0;\n';
		}
		if (options.pass === SHADER_DEPTH) {
			vshader += 'varying float vDepth;\n';
			vshader += '#ifndef VIEWMATRIX\n';
			vshader += '#define VIEWMATRIX\n';
			vshader += 'uniform mat4 matrix_view;\n';
			vshader += '#endif\n';
			vshader += '#ifndef CAMERAPLANES\n';
			vshader += '#define CAMERAPLANES\n';
			vshader += 'uniform vec4 camera_params;\n\n';
			vshader += '#endif\n';
		}
		vshader += ShaderGenerator.begin();
		vshader += "   gl_Position = getPosition();\n";
		if (options.pass === SHADER_DEPTH) {
			vshader += "    vDepth = -(matrix_view * vec4(getWorldPosition(),1.0)).z * camera_params.x;\n";
		}
		if (options.vertexColors) {
			vshader += '    vColor = vertex_color;\n';
		}
		if (options.diffuseMap) {
			vshader += '    vUv0 = vertex_texCoord0;\n';
		}
		vshader += ShaderGenerator.end();
		let fshader = shaderPassDefines;
		if (options.vertexColors) {
			fshader += 'varying vec4 vColor;\n';
		} else {
			fshader += 'uniform vec4 uColor;\n';
		}
		if (options.diffuseMap) {
			fshader += 'varying vec2 vUv0;\n';
			fshader += 'uniform sampler2D texture_diffuseMap;\n';
		}
		if (options.fog) {
			fshader += ShaderGenerator.fogCode(options.fog);
		}
		if (options.alphaTest) {
			fshader += shaderChunks.alphaTestPS;
		}
		if (options.pass === SHADER_DEPTH) {
			fshader += 'varying float vDepth;\n';
			fshader += shaderChunks.packDepthPS;
		}
		fshader += ShaderGenerator.begin();
		if (options.vertexColors) {
			fshader += '    gl_FragColor = vColor;\n';
		} else {
			fshader += '    gl_FragColor = uColor;\n';
		}
		if (options.diffuseMap) {
			fshader += '    gl_FragColor *= texture2D(texture_diffuseMap, vUv0);\n';
		}
		if (options.alphaTest) {
			fshader += "   alphaTest(gl_FragColor.a);\n";
		}
		if (options.pass !== SHADER_PICK) {
			if (options.pass === SHADER_DEPTH) {
				fshader += "    gl_FragColor = packFloat(vDepth);\n";
			} else {
				if (options.fog) {
					fshader += "   glFragColor.rgb = addFog(gl_FragColor.rgb);\n";
				}
			}
		}
		fshader += ShaderGenerator.end();
		return ShaderUtils.createDefinition(device, {
			name: 'BasicShader',
			attributes: attributes,
			vertexCode: vshader,
			fragmentCode: fshader
		});
	}
}
const basic = new ShaderGeneratorBasic();

const defaultMaterialDeviceCache = new DeviceCache();
function getDefaultMaterial(device) {
	const material = defaultMaterialDeviceCache.get(device);
	return material;
}
function setDefaultMaterial(device, material) {
	defaultMaterialDeviceCache.get(device, () => {
		return material;
	});
}

const blendModes = [];
blendModes[BLEND_SUBTRACTIVE] = {
	src: BLENDMODE_ONE,
	dst: BLENDMODE_ONE,
	op: BLENDEQUATION_REVERSE_SUBTRACT
};
blendModes[BLEND_NONE] = {
	src: BLENDMODE_ONE,
	dst: BLENDMODE_ZERO,
	op: BLENDEQUATION_ADD
};
blendModes[BLEND_NORMAL] = {
	src: BLENDMODE_SRC_ALPHA,
	dst: BLENDMODE_ONE_MINUS_SRC_ALPHA,
	op: BLENDEQUATION_ADD
};
blendModes[BLEND_PREMULTIPLIED] = {
	src: BLENDMODE_ONE,
	dst: BLENDMODE_ONE_MINUS_SRC_ALPHA,
	op: BLENDEQUATION_ADD
};
blendModes[BLEND_ADDITIVE] = {
	src: BLENDMODE_ONE,
	dst: BLENDMODE_ONE,
	op: BLENDEQUATION_ADD
};
blendModes[BLEND_ADDITIVEALPHA] = {
	src: BLENDMODE_SRC_ALPHA,
	dst: BLENDMODE_ONE,
	op: BLENDEQUATION_ADD
};
blendModes[BLEND_MULTIPLICATIVE2X] = {
	src: BLENDMODE_DST_COLOR,
	dst: BLENDMODE_SRC_COLOR,
	op: BLENDEQUATION_ADD
};
blendModes[BLEND_SCREEN] = {
	src: BLENDMODE_ONE_MINUS_DST_COLOR,
	dst: BLENDMODE_ONE,
	op: BLENDEQUATION_ADD
};
blendModes[BLEND_MULTIPLICATIVE] = {
	src: BLENDMODE_DST_COLOR,
	dst: BLENDMODE_ZERO,
	op: BLENDEQUATION_ADD
};
blendModes[BLEND_MIN] = {
	src: BLENDMODE_ONE,
	dst: BLENDMODE_ONE,
	op: BLENDEQUATION_MIN
};
blendModes[BLEND_MAX] = {
	src: BLENDMODE_ONE,
	dst: BLENDMODE_ONE,
	op: BLENDEQUATION_MAX
};
let id$2 = 0;
class Material {
	constructor() {
		this._shader = null;
		this.meshInstances = [];
		this.name = 'Untitled';
		this.userId = '';
		this.id = id$2++;
		this.variants = new Map();
		this.parameters = {};
		this.alphaTest = 0;
		this.alphaToCoverage = false;
		this._blendState = new BlendState();
		this._depthState = new DepthState();
		this.cull = CULLFACE_BACK;
		this.stencilFront = null;
		this.stencilBack = null;
		this._shaderVersion = 0;
		this._scene = null;
		this.dirty = true;
	}
	set depthBias(value) {
		this._depthState.depthBias = value;
	}
	get depthBias() {
		return this._depthState.depthBias;
	}
	set slopeDepthBias(value) {
		this._depthState.depthBiasSlope = value;
	}
	get slopeDepthBias() {
		return this._depthState.depthBiasSlope;
	}
	set redWrite(value) {
		this._blendState.redWrite = value;
	}
	get redWrite() {
		return this._blendState.redWrite;
	}
	set greenWrite(value) {
		this._blendState.greenWrite = value;
	}
	get greenWrite() {
		return this._blendState.greenWrite;
	}
	set blueWrite(value) {
		this._blendState.blueWrite = value;
	}
	get blueWrite() {
		return this._blendState.blueWrite;
	}
	set alphaWrite(value) {
		this._blendState.alphaWrite = value;
	}
	get alphaWrite() {
		return this._blendState.alphaWrite;
	}
	set shader(shader) {
		this._shader = shader;
	}
	get shader() {
		return this._shader;
	}
	get transparent() {
		return this._blendState.blend;
	}
	_updateTransparency() {
		const transparent = this.transparent;
		const meshInstances = this.meshInstances;
		for (let i = 0; i < meshInstances.length; i++) {
			meshInstances[i].transparent = transparent;
		}
	}
	set blendState(value) {
		this._blendState.copy(value);
		this._updateTransparency();
	}
	get blendState() {
		return this._blendState;
	}
	set blendType(type) {
		const blendMode = blendModes[type];
		this._blendState.setColorBlend(blendMode.op, blendMode.src, blendMode.dst);
		this._blendState.setAlphaBlend(blendMode.op, blendMode.src, blendMode.dst);
		const blend = type !== BLEND_NONE;
		if (this._blendState.blend !== blend) {
			this._blendState.blend = blend;
			this._updateTransparency();
		}
		this._updateMeshInstanceKeys();
	}
	get blendType() {
		if (!this.transparent) {
			return BLEND_NONE;
		}
		const {
			colorOp,
			colorSrcFactor,
			colorDstFactor,
			alphaOp,
			alphaSrcFactor,
			alphaDstFactor
		} = this._blendState;
		for (let i = 0; i < blendModes.length; i++) {
			const blendMode = blendModes[i];
			if (blendMode.src === colorSrcFactor && blendMode.dst === colorDstFactor && blendMode.op === colorOp && blendMode.src === alphaSrcFactor && blendMode.dst === alphaDstFactor && blendMode.op === alphaOp) {
				return i;
			}
		}
		return BLEND_NORMAL;
	}
	set depthState(value) {
		this._depthState.copy(value);
	}
	get depthState() {
		return this._depthState;
	}
	set depthTest(value) {
		this._depthState.test = value;
	}
	get depthTest() {
		return this._depthState.test;
	}
	set depthFunc(value) {
		this._depthState.func = value;
	}
	get depthFunc() {
		return this._depthState.func;
	}
	set depthWrite(value) {
		this._depthState.write = value;
	}
	get depthWrite() {
		return this._depthState.write;
	}
	copy(source) {
		var _source$stencilFront;
		this.name = source.name;
		this._shader = source._shader;
		this.alphaTest = source.alphaTest;
		this.alphaToCoverage = source.alphaToCoverage;
		this._blendState.copy(source._blendState);
		this._depthState.copy(source._depthState);
		this.cull = source.cull;
		this.stencilFront = (_source$stencilFront = source.stencilFront) == null ? void 0 : _source$stencilFront.clone();
		if (source.stencilBack) {
			this.stencilBack = source.stencilFront === source.stencilBack ? this.stencilFront : source.stencilBack.clone();
		}
		return this;
	}
	clone() {
		const clone = new this.constructor();
		return clone.copy(this);
	}
	_updateMeshInstanceKeys() {
		const meshInstances = this.meshInstances;
		for (let i = 0; i < meshInstances.length; i++) {
			meshInstances[i].updateKey();
		}
	}
	updateUniforms(device, scene) {}
	getShaderVariant(device, scene, objDefs, unused, pass, sortedLights, viewUniformFormat, viewBindGroupFormat, vertexFormat) {
		const processingOptions = new ShaderProcessorOptions(viewUniformFormat, viewBindGroupFormat, vertexFormat);
		return processShader(this._shader, processingOptions);
	}
	update() {
		this.dirty = true;
		if (this._shader) this._shader.failed = false;
	}
	clearParameters() {
		this.parameters = {};
	}
	getParameters() {
		return this.parameters;
	}
	clearVariants() {
		this.variants.clear();
		const meshInstances = this.meshInstances;
		const count = meshInstances.length;
		for (let i = 0; i < count; i++) {
			meshInstances[i].clearShaders();
		}
	}
	getParameter(name) {
		return this.parameters[name];
	}
	setParameter(name, data) {
		if (data === undefined && typeof name === 'object') {
			const uniformObject = name;
			if (uniformObject.length) {
				for (let i = 0; i < uniformObject.length; i++) {
					this.setParameter(uniformObject[i]);
				}
				return;
			}
			name = uniformObject.name;
			data = uniformObject.value;
		}
		const param = this.parameters[name];
		if (param) {
			param.data = data;
		} else {
			this.parameters[name] = {
				scopeId: null,
				data: data
			};
		}
	}
	deleteParameter(name) {
		if (this.parameters[name]) {
			delete this.parameters[name];
		}
	}
	setParameters(device, names) {
		const parameters = this.parameters;
		if (names === undefined) names = parameters;
		for (const paramName in names) {
			const parameter = parameters[paramName];
			if (parameter) {
				if (!parameter.scopeId) {
					parameter.scopeId = device.scope.resolve(paramName);
				}
				parameter.scopeId.setValue(parameter.data);
			}
		}
	}
	destroy() {
		this.variants.clear();
		this._shader = null;
		for (let i = 0; i < this.meshInstances.length; i++) {
			const meshInstance = this.meshInstances[i];
			meshInstance.clearShaders();
			meshInstance._material = null;
			if (meshInstance.mesh) {
				const defaultMaterial = getDefaultMaterial(meshInstance.mesh.device);
				if (this !== defaultMaterial) {
					meshInstance.material = defaultMaterial;
				}
			}
		}
		this.meshInstances.length = 0;
	}
	addMeshInstanceRef(meshInstance) {
		this.meshInstances.push(meshInstance);
	}
	removeMeshInstanceRef(meshInstance) {
		const meshInstances = this.meshInstances;
		const i = meshInstances.indexOf(meshInstance);
		if (i !== -1) {
			meshInstances.splice(i, 1);
		}
	}
}

class BasicMaterial extends Material {
	constructor(...args) {
		super(...args);
		this.color = new Color(1, 1, 1, 1);
		this.colorUniform = new Float32Array(4);
		this.colorMap = null;
		this.vertexColors = false;
	}
	copy(source) {
		super.copy(source);
		this.color.copy(source.color);
		this.colorMap = source.colorMap;
		this.vertexColors = source.vertexColors;
		return this;
	}
	updateUniforms(device, scene) {
		this.clearParameters();
		this.colorUniform[0] = this.color.r;
		this.colorUniform[1] = this.color.g;
		this.colorUniform[2] = this.color.b;
		this.colorUniform[3] = this.color.a;
		this.setParameter('uColor', this.colorUniform);
		if (this.colorMap) {
			this.setParameter('texture_diffuseMap', this.colorMap);
		}
	}
	getShaderVariant(device, scene, objDefs, unused, pass, sortedLights, viewUniformFormat, viewBindGroupFormat, vertexFormat) {
		const options = {
			skin: objDefs && (objDefs & SHADERDEF_SKIN) !== 0,
			screenSpace: objDefs && (objDefs & SHADERDEF_SCREENSPACE) !== 0,
			useInstancing: objDefs && (objDefs & SHADERDEF_INSTANCING) !== 0,
			useMorphPosition: objDefs && (objDefs & SHADERDEF_MORPH_POSITION) !== 0,
			useMorphNormal: objDefs && (objDefs & SHADERDEF_MORPH_NORMAL) !== 0,
			useMorphTextureBased: objDefs && (objDefs & SHADERDEF_MORPH_TEXTURE_BASED) !== 0,
			alphaTest: this.alphaTest > 0,
			vertexColors: this.vertexColors,
			diffuseMap: !!this.colorMap,
			pass: pass
		};
		const processingOptions = new ShaderProcessorOptions(viewUniformFormat, viewBindGroupFormat, vertexFormat);
		const library = getProgramLibrary(device);
		library.register('basic', basic);
		return library.getProgram('basic', options, processingOptions, this.userId);
	}
}

class Batch {
	constructor(meshInstances, dynamic, batchGroupId) {
		this._aabb = new BoundingBox();
		this.origMeshInstances = void 0;
		this.meshInstance = null;
		this.dynamic = void 0;
		this.batchGroupId = void 0;
		this.origMeshInstances = meshInstances;
		this.dynamic = dynamic;
		this.batchGroupId = batchGroupId;
	}
	destroy(scene, layers) {
		if (this.meshInstance) {
			this.removeFromLayers(scene, layers);
			this.meshInstance.destroy();
			this.meshInstance = null;
		}
	}
	addToLayers(scene, layers) {
		for (let i = 0; i < layers.length; i++) {
			const layer = scene.layers.getLayerById(layers[i]);
			if (layer) {
				layer.addMeshInstances([this.meshInstance]);
			}
		}
	}
	removeFromLayers(scene, layers) {
		for (let i = 0; i < layers.length; i++) {
			const layer = scene.layers.getLayerById(layers[i]);
			if (layer) {
				layer.removeMeshInstances([this.meshInstance]);
			}
		}
	}
	updateBoundingBox() {
		this._aabb.copy(this.origMeshInstances[0].aabb);
		for (let i = 1; i < this.origMeshInstances.length; i++) {
			this._aabb.add(this.origMeshInstances[i].aabb);
		}
		this.meshInstance.aabb = this._aabb;
		this.meshInstance._aabbVer = 0;
	}
}

class BatchGroup {
	constructor(id, name, dynamic, maxAabbSize, layers = [LAYERID_WORLD]) {
		this._ui = false;
		this._sprite = false;
		this._obj = {
			model: [],
			element: [],
			sprite: [],
			render: []
		};
		this.id = void 0;
		this.name = void 0;
		this.dynamic = void 0;
		this.maxAabbSize = void 0;
		this.layers = void 0;
		this.id = id;
		this.name = name;
		this.dynamic = dynamic;
		this.maxAabbSize = maxAabbSize;
		this.layers = layers;
	}
}
BatchGroup.MODEL = 'model';
BatchGroup.ELEMENT = 'element';
BatchGroup.SPRITE = 'sprite';
BatchGroup.RENDER = 'render';

const _invMatrix = new Mat4();
class SkinInstance {
	constructor(skin) {
		this.bones = void 0;
		this.boneTextureSize = void 0;
		this._dirty = true;
		this._rootBone = null;
		this._skinUpdateIndex = -1;
		this._updateBeforeCull = true;
		if (skin) {
			this.initSkin(skin);
		}
	}
	set rootBone(rootBone) {
		this._rootBone = rootBone;
	}
	get rootBone() {
		return this._rootBone;
	}
	init(device, numBones) {
		if (device.supportsBoneTextures) {
			const numPixels = numBones * 3;
			let width = Math.ceil(Math.sqrt(numPixels));
			width = math.roundUp(width, 3);
			const height = Math.ceil(numPixels / width);
			this.boneTexture = new Texture(device, {
				width: width,
				height: height,
				format: PIXELFORMAT_RGBA32F,
				mipmaps: false,
				minFilter: FILTER_NEAREST,
				magFilter: FILTER_NEAREST,
				name: 'skin'
			});
			this.boneTextureSize = [width, height, 1.0 / width, 1.0 / height];
			this.matrixPalette = this.boneTexture.lock({
				mode: TEXTURELOCK_READ
			});
			this.boneTexture.unlock();
		} else {
			this.matrixPalette = new Float32Array(numBones * 12);
		}
	}
	destroy() {
		if (this.boneTexture) {
			this.boneTexture.destroy();
			this.boneTexture = null;
		}
	}
	resolve(rootBone, entity) {
		this.rootBone = rootBone;
		const skin = this.skin;
		const bones = [];
		for (let j = 0; j < skin.boneNames.length; j++) {
			const boneName = skin.boneNames[j];
			let bone = rootBone.findByName(boneName);
			if (!bone) {
				bone = entity;
			}
			bones.push(bone);
		}
		this.bones = bones;
	}
	initSkin(skin) {
		this.skin = skin;
		this.bones = [];
		const numBones = skin.inverseBindPose.length;
		this.init(skin.device, numBones);
		this.matrices = [];
		for (let i = 0; i < numBones; i++) {
			this.matrices[i] = new Mat4();
		}
	}
	uploadBones(device) {
		if (device.supportsBoneTextures) {
			this.boneTexture.lock();
			this.boneTexture.unlock();
		}
	}
	_updateMatrices(rootNode, skinUpdateIndex) {
		if (this._skinUpdateIndex !== skinUpdateIndex) {
			this._skinUpdateIndex = skinUpdateIndex;
			_invMatrix.copy(rootNode.getWorldTransform()).invert();
			for (let i = this.bones.length - 1; i >= 0; i--) {
				this.matrices[i].mulAffine2(_invMatrix, this.bones[i].getWorldTransform());
				this.matrices[i].mulAffine2(this.matrices[i], this.skin.inverseBindPose[i]);
			}
		}
	}
	updateMatrices(rootNode, skinUpdateIndex) {
		if (this._updateBeforeCull) {
			this._updateMatrices(rootNode, skinUpdateIndex);
		}
	}
	updateMatrixPalette(rootNode, skinUpdateIndex) {
		this._updateMatrices(rootNode, skinUpdateIndex);
		const mp = this.matrixPalette;
		const count = this.bones.length;
		for (let i = 0; i < count; i++) {
			const pe = this.matrices[i].data;
			const base = i * 12;
			mp[base] = pe[0];
			mp[base + 1] = pe[4];
			mp[base + 2] = pe[8];
			mp[base + 3] = pe[12];
			mp[base + 4] = pe[1];
			mp[base + 5] = pe[5];
			mp[base + 6] = pe[9];
			mp[base + 7] = pe[13];
			mp[base + 8] = pe[2];
			mp[base + 9] = pe[6];
			mp[base + 10] = pe[10];
			mp[base + 11] = pe[14];
		}
		this.uploadBones(this.skin.device);
	}
}

class SkinBatchInstance extends SkinInstance {
	constructor(device, nodes, rootNode) {
		super();
		const numBones = nodes.length;
		this.init(device, numBones);
		this.device = device;
		this.rootNode = rootNode;
		this.bones = nodes;
	}
	updateMatrices(rootNode, skinUpdateIndex) {}
	updateMatrixPalette(rootNode, skinUpdateIndex) {
		const mp = this.matrixPalette;
		const count = this.bones.length;
		for (let i = 0; i < count; i++) {
			const pe = this.bones[i].getWorldTransform().data;
			const base = i * 12;
			mp[base] = pe[0];
			mp[base + 1] = pe[4];
			mp[base + 2] = pe[8];
			mp[base + 3] = pe[12];
			mp[base + 4] = pe[1];
			mp[base + 5] = pe[5];
			mp[base + 6] = pe[9];
			mp[base + 7] = pe[13];
			mp[base + 8] = pe[2];
			mp[base + 9] = pe[6];
			mp[base + 10] = pe[10];
			mp[base + 11] = pe[14];
		}
		this.uploadBones(this.device);
	}
}

const scaleCompensatePosTransform = new Mat4();
const scaleCompensatePos = new Vec3();
const scaleCompensateRot = new Quat();
const scaleCompensateRot2 = new Quat();
const scaleCompensateScale = new Vec3();
const scaleCompensateScaleForParent = new Vec3();
const tmpMat4 = new Mat4();
const tmpQuat = new Quat();
const position$1 = new Vec3();
const invParentWtm$1 = new Mat4();
const rotation = new Quat();
const invParentRot = new Quat();
const matrix = new Mat4();
const target = new Vec3();
const up = new Vec3();
function createTest(attr, value) {
	if (attr instanceof Function) {
		return attr;
	}
	return node => {
		let x = node[attr];
		if (x instanceof Function) {
			x = x();
		}
		return x === value;
	};
}
function findNode(node, test) {
	if (test(node)) return node;
	const children = node._children;
	const len = children.length;
	for (let i = 0; i < len; ++i) {
		const result = findNode(children[i], test);
		if (result) return result;
	}
	return null;
}
class GraphNode extends EventHandler {
	constructor(name = 'Untitled') {
		super();
		this.name = void 0;
		this.tags = new Tags(this);
		this._labels = {};
		this.localPosition = new Vec3();
		this.localRotation = new Quat();
		this.localScale = new Vec3(1, 1, 1);
		this.localEulerAngles = new Vec3();
		this.position = new Vec3();
		this.rotation = new Quat();
		this.eulerAngles = new Vec3();
		this._scale = null;
		this.localTransform = new Mat4();
		this._dirtyLocal = false;
		this._aabbVer = 0;
		this._frozen = false;
		this.worldTransform = new Mat4();
		this._dirtyWorld = false;
		this._worldScaleSign = 0;
		this._normalMatrix = new Mat3();
		this._dirtyNormal = true;
		this._right = null;
		this._up = null;
		this._forward = null;
		this._parent = null;
		this._children = [];
		this._graphDepth = 0;
		this._enabled = true;
		this._enabledInHierarchy = false;
		this.scaleCompensation = false;
		this.name = name;
	}
	get right() {
		if (!this._right) {
			this._right = new Vec3();
		}
		return this.getWorldTransform().getX(this._right).normalize();
	}
	get up() {
		if (!this._up) {
			this._up = new Vec3();
		}
		return this.getWorldTransform().getY(this._up).normalize();
	}
	get forward() {
		if (!this._forward) {
			this._forward = new Vec3();
		}
		return this.getWorldTransform().getZ(this._forward).normalize().mulScalar(-1);
	}
	get normalMatrix() {
		const normalMat = this._normalMatrix;
		if (this._dirtyNormal) {
			normalMat.invertMat4(this.getWorldTransform()).transpose();
			this._dirtyNormal = false;
		}
		return normalMat;
	}
	set enabled(enabled) {
		if (this._enabled !== enabled) {
			var _this$_parent;
			this._enabled = enabled;
			if (enabled && (_this$_parent = this._parent) != null && _this$_parent.enabled || !enabled) {
				this._notifyHierarchyStateChanged(this, enabled);
			}
		}
	}
	get enabled() {
		return this._enabled && this._enabledInHierarchy;
	}
	get parent() {
		return this._parent;
	}
	get path() {
		let node = this._parent;
		if (!node) {
			return '';
		}
		let result = this.name;
		while (node && node._parent) {
			result = `${node.name}/${result}`;
			node = node._parent;
		}
		return result;
	}
	get root() {
		let result = this;
		while (result._parent) {
			result = result._parent;
		}
		return result;
	}
	get children() {
		return this._children;
	}
	get graphDepth() {
		return this._graphDepth;
	}
	_notifyHierarchyStateChanged(node, enabled) {
		node._onHierarchyStateChanged(enabled);
		const c = node._children;
		for (let i = 0, len = c.length; i < len; i++) {
			if (c[i]._enabled) this._notifyHierarchyStateChanged(c[i], enabled);
		}
	}
	_onHierarchyStateChanged(enabled) {
		this._enabledInHierarchy = enabled;
		if (enabled && !this._frozen) this._unfreezeParentToRoot();
	}
	_cloneInternal(clone) {
		clone.name = this.name;
		const tags = this.tags._list;
		clone.tags.clear();
		for (let i = 0; i < tags.length; i++) clone.tags.add(tags[i]);
		clone._labels = Object.assign({}, this._labels);
		clone.localPosition.copy(this.localPosition);
		clone.localRotation.copy(this.localRotation);
		clone.localScale.copy(this.localScale);
		clone.localEulerAngles.copy(this.localEulerAngles);
		clone.position.copy(this.position);
		clone.rotation.copy(this.rotation);
		clone.eulerAngles.copy(this.eulerAngles);
		clone.localTransform.copy(this.localTransform);
		clone._dirtyLocal = this._dirtyLocal;
		clone.worldTransform.copy(this.worldTransform);
		clone._dirtyWorld = this._dirtyWorld;
		clone._dirtyNormal = this._dirtyNormal;
		clone._aabbVer = this._aabbVer + 1;
		clone._enabled = this._enabled;
		clone.scaleCompensation = this.scaleCompensation;
		clone._enabledInHierarchy = false;
	}
	clone() {
		const clone = new this.constructor();
		this._cloneInternal(clone);
		return clone;
	}
	copy(source) {
		source._cloneInternal(this);
		return this;
	}
	destroy() {
		this.remove();
		const children = this._children;
		while (children.length) {
			const child = children.pop();
			child._parent = null;
			child.destroy();
		}
		this.fire('destroy', this);
		this.off();
	}
	find(attr, value) {
		const results = [];
		const test = createTest(attr, value);
		this.forEach(node => {
			if (test(node)) results.push(node);
		});
		return results;
	}
	findOne(attr, value) {
		const test = createTest(attr, value);
		return findNode(this, test);
	}
	findByTag() {
		const query = arguments;
		const results = [];
		const queryNode = (node, checkNode) => {
			if (checkNode && node.tags.has(...query)) {
				results.push(node);
			}
			for (let i = 0; i < node._children.length; i++) {
				queryNode(node._children[i], true);
			}
		};
		queryNode(this, false);
		return results;
	}
	findByName(name) {
		return this.findOne('name', name);
	}
	findByPath(path) {
		const parts = Array.isArray(path) ? path : path.split('/');
		let result = this;
		for (let i = 0, imax = parts.length; i < imax; ++i) {
			result = result.children.find(c => c.name === parts[i]);
			if (!result) {
				return null;
			}
		}
		return result;
	}
	forEach(callback, thisArg) {
		callback.call(thisArg, this);
		const children = this._children;
		const len = children.length;
		for (let i = 0; i < len; ++i) {
			children[i].forEach(callback, thisArg);
		}
	}
	isDescendantOf(node) {
		let parent = this._parent;
		while (parent) {
			if (parent === node) return true;
			parent = parent._parent;
		}
		return false;
	}
	isAncestorOf(node) {
		return node.isDescendantOf(this);
	}
	getEulerAngles() {
		this.getWorldTransform().getEulerAngles(this.eulerAngles);
		return this.eulerAngles;
	}
	getLocalEulerAngles() {
		this.localRotation.getEulerAngles(this.localEulerAngles);
		return this.localEulerAngles;
	}
	getLocalPosition() {
		return this.localPosition;
	}
	getLocalRotation() {
		return this.localRotation;
	}
	getLocalScale() {
		return this.localScale;
	}
	getLocalTransform() {
		if (this._dirtyLocal) {
			this.localTransform.setTRS(this.localPosition, this.localRotation, this.localScale);
			this._dirtyLocal = false;
		}
		return this.localTransform;
	}
	getPosition() {
		this.getWorldTransform().getTranslation(this.position);
		return this.position;
	}
	getRotation() {
		this.rotation.setFromMat4(this.getWorldTransform());
		return this.rotation;
	}
	getScale() {
		if (!this._scale) {
			this._scale = new Vec3();
		}
		return this.getWorldTransform().getScale(this._scale);
	}
	getWorldTransform() {
		if (!this._dirtyLocal && !this._dirtyWorld) return this.worldTransform;
		if (this._parent) this._parent.getWorldTransform();
		this._sync();
		return this.worldTransform;
	}
	get worldScaleSign() {
		if (this._worldScaleSign === 0) {
			this._worldScaleSign = this.getWorldTransform().scaleSign;
		}
		return this._worldScaleSign;
	}
	remove() {
		var _this$_parent2;
		(_this$_parent2 = this._parent) == null || _this$_parent2.removeChild(this);
	}
	reparent(parent, index) {
		this.remove();
		if (parent) {
			if (index >= 0) {
				parent.insertChild(this, index);
			} else {
				parent.addChild(this);
			}
		}
	}
	setLocalEulerAngles(x, y, z) {
		this.localRotation.setFromEulerAngles(x, y, z);
		if (!this._dirtyLocal) this._dirtifyLocal();
	}
	setLocalPosition(x, y, z) {
		if (x instanceof Vec3) {
			this.localPosition.copy(x);
		} else {
			this.localPosition.set(x, y, z);
		}
		if (!this._dirtyLocal) this._dirtifyLocal();
	}
	setLocalRotation(x, y, z, w) {
		if (x instanceof Quat) {
			this.localRotation.copy(x);
		} else {
			this.localRotation.set(x, y, z, w);
		}
		if (!this._dirtyLocal) this._dirtifyLocal();
	}
	setLocalScale(x, y, z) {
		if (x instanceof Vec3) {
			this.localScale.copy(x);
		} else {
			this.localScale.set(x, y, z);
		}
		if (!this._dirtyLocal) this._dirtifyLocal();
	}
	_dirtifyLocal() {
		if (!this._dirtyLocal) {
			this._dirtyLocal = true;
			if (!this._dirtyWorld) this._dirtifyWorld();
		}
	}
	_unfreezeParentToRoot() {
		let p = this._parent;
		while (p) {
			p._frozen = false;
			p = p._parent;
		}
	}
	_dirtifyWorld() {
		if (!this._dirtyWorld) this._unfreezeParentToRoot();
		this._dirtifyWorldInternal();
	}
	_dirtifyWorldInternal() {
		if (!this._dirtyWorld) {
			this._frozen = false;
			this._dirtyWorld = true;
			for (let i = 0; i < this._children.length; i++) {
				if (!this._children[i]._dirtyWorld) this._children[i]._dirtifyWorldInternal();
			}
		}
		this._dirtyNormal = true;
		this._worldScaleSign = 0;
		this._aabbVer++;
	}
	setPosition(x, y, z) {
		if (x instanceof Vec3) {
			position$1.copy(x);
		} else {
			position$1.set(x, y, z);
		}
		if (this._parent === null) {
			this.localPosition.copy(position$1);
		} else {
			invParentWtm$1.copy(this._parent.getWorldTransform()).invert();
			invParentWtm$1.transformPoint(position$1, this.localPosition);
		}
		if (!this._dirtyLocal) this._dirtifyLocal();
	}
	setRotation(x, y, z, w) {
		if (x instanceof Quat) {
			rotation.copy(x);
		} else {
			rotation.set(x, y, z, w);
		}
		if (this._parent === null) {
			this.localRotation.copy(rotation);
		} else {
			const parentRot = this._parent.getRotation();
			invParentRot.copy(parentRot).invert();
			this.localRotation.copy(invParentRot).mul(rotation);
		}
		if (!this._dirtyLocal) this._dirtifyLocal();
	}
	setEulerAngles(x, y, z) {
		this.localRotation.setFromEulerAngles(x, y, z);
		if (this._parent !== null) {
			const parentRot = this._parent.getRotation();
			invParentRot.copy(parentRot).invert();
			this.localRotation.mul2(invParentRot, this.localRotation);
		}
		if (!this._dirtyLocal) this._dirtifyLocal();
	}
	addChild(node) {
		this._prepareInsertChild(node);
		this._children.push(node);
		this._onInsertChild(node);
	}
	addChildAndSaveTransform(node) {
		const wPos = node.getPosition();
		const wRot = node.getRotation();
		this._prepareInsertChild(node);
		node.setPosition(tmpMat4.copy(this.worldTransform).invert().transformPoint(wPos));
		node.setRotation(tmpQuat.copy(this.getRotation()).invert().mul(wRot));
		this._children.push(node);
		this._onInsertChild(node);
	}
	insertChild(node, index) {
		this._prepareInsertChild(node);
		this._children.splice(index, 0, node);
		this._onInsertChild(node);
	}
	_prepareInsertChild(node) {
		node.remove();
	}
	_fireOnHierarchy(name, nameHierarchy, parent) {
		this.fire(name, parent);
		for (let i = 0; i < this._children.length; i++) {
			this._children[i]._fireOnHierarchy(nameHierarchy, nameHierarchy, parent);
		}
	}
	_onInsertChild(node) {
		node._parent = this;
		const enabledInHierarchy = node._enabled && this.enabled;
		if (node._enabledInHierarchy !== enabledInHierarchy) {
			node._enabledInHierarchy = enabledInHierarchy;
			node._notifyHierarchyStateChanged(node, enabledInHierarchy);
		}
		node._updateGraphDepth();
		node._dirtifyWorld();
		if (this._frozen) node._unfreezeParentToRoot();
		node._fireOnHierarchy('insert', 'inserthierarchy', this);
		if (this.fire) this.fire('childinsert', node);
	}
	_updateGraphDepth() {
		this._graphDepth = this._parent ? this._parent._graphDepth + 1 : 0;
		for (let i = 0, len = this._children.length; i < len; i++) {
			this._children[i]._updateGraphDepth();
		}
	}
	removeChild(child) {
		const index = this._children.indexOf(child);
		if (index === -1) {
			return;
		}
		this._children.splice(index, 1);
		child._parent = null;
		child._fireOnHierarchy('remove', 'removehierarchy', this);
		this.fire('childremove', child);
	}
	_sync() {
		if (this._dirtyLocal) {
			this.localTransform.setTRS(this.localPosition, this.localRotation, this.localScale);
			this._dirtyLocal = false;
		}
		if (this._dirtyWorld) {
			if (this._parent === null) {
				this.worldTransform.copy(this.localTransform);
			} else {
				if (this.scaleCompensation) {
					let parentWorldScale;
					const parent = this._parent;
					let scale = this.localScale;
					let parentToUseScaleFrom = parent;
					if (parentToUseScaleFrom) {
						while (parentToUseScaleFrom && parentToUseScaleFrom.scaleCompensation) {
							parentToUseScaleFrom = parentToUseScaleFrom._parent;
						}
						if (parentToUseScaleFrom) {
							parentToUseScaleFrom = parentToUseScaleFrom._parent;
							if (parentToUseScaleFrom) {
								parentWorldScale = parentToUseScaleFrom.worldTransform.getScale();
								scaleCompensateScale.mul2(parentWorldScale, this.localScale);
								scale = scaleCompensateScale;
							}
						}
					}
					scaleCompensateRot2.setFromMat4(parent.worldTransform);
					scaleCompensateRot.mul2(scaleCompensateRot2, this.localRotation);
					let tmatrix = parent.worldTransform;
					if (parent.scaleCompensation) {
						scaleCompensateScaleForParent.mul2(parentWorldScale, parent.getLocalScale());
						scaleCompensatePosTransform.setTRS(parent.worldTransform.getTranslation(scaleCompensatePos), scaleCompensateRot2, scaleCompensateScaleForParent);
						tmatrix = scaleCompensatePosTransform;
					}
					tmatrix.transformPoint(this.localPosition, scaleCompensatePos);
					this.worldTransform.setTRS(scaleCompensatePos, scaleCompensateRot, scale);
				} else {
					this.worldTransform.mulAffine2(this._parent.worldTransform, this.localTransform);
				}
			}
			this._dirtyWorld = false;
		}
	}
	syncHierarchy() {
		if (!this._enabled) return;
		if (this._frozen) return;
		this._frozen = true;
		if (this._dirtyLocal || this._dirtyWorld) {
			this._sync();
		}
		const children = this._children;
		for (let i = 0, len = children.length; i < len; i++) {
			children[i].syncHierarchy();
		}
	}
	lookAt(x, y, z, ux = 0, uy = 1, uz = 0) {
		if (x instanceof Vec3) {
			target.copy(x);
			if (y instanceof Vec3) {
				up.copy(y);
			} else {
				up.copy(Vec3.UP);
			}
		} else if (z === undefined) {
			return;
		} else {
			target.set(x, y, z);
			up.set(ux, uy, uz);
		}
		matrix.setLookAt(this.getPosition(), target, up);
		rotation.setFromMat4(matrix);
		this.setRotation(rotation);
	}
	translate(x, y, z) {
		if (x instanceof Vec3) {
			position$1.copy(x);
		} else {
			position$1.set(x, y, z);
		}
		position$1.add(this.getPosition());
		this.setPosition(position$1);
	}
	translateLocal(x, y, z) {
		if (x instanceof Vec3) {
			position$1.copy(x);
		} else {
			position$1.set(x, y, z);
		}
		this.localRotation.transformVector(position$1, position$1);
		this.localPosition.add(position$1);
		if (!this._dirtyLocal) this._dirtifyLocal();
	}
	rotate(x, y, z) {
		rotation.setFromEulerAngles(x, y, z);
		if (this._parent === null) {
			this.localRotation.mul2(rotation, this.localRotation);
		} else {
			const rot = this.getRotation();
			const parentRot = this._parent.getRotation();
			invParentRot.copy(parentRot).invert();
			rotation.mul2(invParentRot, rotation);
			this.localRotation.mul2(rotation, rot);
		}
		if (!this._dirtyLocal) this._dirtifyLocal();
	}
	rotateLocal(x, y, z) {
		rotation.setFromEulerAngles(x, y, z);
		this.localRotation.mul(rotation);
		if (!this._dirtyLocal) this._dirtifyLocal();
	}
}

class RefCountedCache {
	constructor() {
		this.cache = new Map();
	}
	destroy() {
		this.cache.forEach((refCount, object) => {
			object.destroy();
		});
		this.cache.clear();
	}
	incRef(object) {
		const refCount = (this.cache.get(object) || 0) + 1;
		this.cache.set(object, refCount);
	}
	decRef(object) {
		if (object) {
			let refCount = this.cache.get(object);
			if (refCount) {
				refCount--;
				if (refCount === 0) {
					this.cache.delete(object);
					object.destroy();
				} else {
					this.cache.set(object, refCount);
				}
			}
		}
	}
}

class LightmapCache {
	static incRef(texture) {
		this.cache.incRef(texture);
	}
	static decRef(texture) {
		this.cache.decRef(texture);
	}
	static destroy() {
		this.cache.destroy();
	}
}
LightmapCache.cache = new RefCountedCache();

let id$1 = 0;
const _tmpAabb = new BoundingBox();
const _tempBoneAabb = new BoundingBox();
const _tempSphere = new BoundingSphere();
const _meshSet = new Set();
class InstancingData {
	constructor(numObjects) {
		this.vertexBuffer = null;
		this.count = numObjects;
	}
}
class ShaderInstance {
	constructor() {
		this.shader = void 0;
		this.bindGroup = null;
	}
	getBindGroup(device) {
		if (!this.bindGroup) {
			const shader = this.shader;
			const ubFormat = shader.meshUniformBufferFormat;
			const uniformBuffer = new UniformBuffer(device, ubFormat, false);
			const bindGroupFormat = shader.meshBindGroupFormat;
			this.bindGroup = new BindGroup(device, bindGroupFormat, uniformBuffer);
		}
		return this.bindGroup;
	}
	destroy() {
		const group = this.bindGroup;
		if (group) {
			var _group$defaultUniform;
			(_group$defaultUniform = group.defaultUniformBuffer) == null || _group$defaultUniform.destroy();
			group.destroy();
			this.bindGroup = null;
		}
	}
}
class ShaderCacheEntry {
	constructor() {
		this.shaderInstances = new Map();
	}
	destroy() {
		this.shaderInstances.forEach(instance => instance.destroy());
		this.shaderInstances.clear();
	}
}
class MeshInstance {
	constructor(mesh, material, node = null) {
		this.visible = true;
		this.castShadow = false;
		this.transparent = false;
		this._material = null;
		this._shaderCache = [];
		this.id = id$1++;
		this.pick = true;
		if (mesh instanceof GraphNode) {
			const temp = mesh;
			mesh = material;
			material = node;
			node = temp;
		}
		this._key = [0, 0];
		this.node = node;
		this._mesh = mesh;
		mesh.incRefCount();
		this.material = material;
		this._shaderDefs = MASK_AFFECT_DYNAMIC << 16;
		this._shaderDefs |= mesh.vertexBuffer.format.hasUv0 ? SHADERDEF_UV0 : 0;
		this._shaderDefs |= mesh.vertexBuffer.format.hasUv1 ? SHADERDEF_UV1 : 0;
		this._shaderDefs |= mesh.vertexBuffer.format.hasColor ? SHADERDEF_VCOLOR : 0;
		this._shaderDefs |= mesh.vertexBuffer.format.hasTangents ? SHADERDEF_TANGENTS : 0;
		this.layer = LAYER_WORLD;
		this._renderStyle = RENDERSTYLE_SOLID;
		this._receiveShadow = true;
		this._screenSpace = false;
		this.cull = true;
		this._updateAabb = true;
		this._updateAabbFunc = null;
		this._calculateSortDistance = null;
		this.updateKey();
		this._skinInstance = null;
		this._morphInstance = null;
		this.gsplatInstance = null;
		this.instancingData = null;
		this._customAabb = null;
		this.aabb = new BoundingBox();
		this._aabbVer = -1;
		this._aabbMeshVer = -1;
		this.drawOrder = 0;
		this.visibleThisFrame = false;
		this.isVisibleFunc = null;
		this.parameters = {};
		this.stencilFront = null;
		this.stencilBack = null;
		this.flipFacesFactor = 1;
	}
	set renderStyle(renderStyle) {
		this._renderStyle = renderStyle;
		this.mesh.prepareRenderState(renderStyle);
	}
	get renderStyle() {
		return this._renderStyle;
	}
	set mesh(mesh) {
		if (mesh === this._mesh) return;
		if (this._mesh) {
			this._mesh.decRefCount();
		}
		this._mesh = mesh;
		if (mesh) {
			mesh.incRefCount();
		}
	}
	get mesh() {
		return this._mesh;
	}
	set aabb(aabb) {
		this._aabb = aabb;
	}
	get aabb() {
		if (!this._updateAabb) {
			return this._aabb;
		}
		if (this._updateAabbFunc) {
			return this._updateAabbFunc(this._aabb);
		}
		let localAabb = this._customAabb;
		let toWorldSpace = !!localAabb;
		if (!localAabb) {
			localAabb = _tmpAabb;
			if (this.skinInstance) {
				if (!this.mesh.boneAabb) {
					const morphTargets = this._morphInstance ? this._morphInstance.morph._targets : null;
					this.mesh._initBoneAabbs(morphTargets);
				}
				const boneUsed = this.mesh.boneUsed;
				let first = true;
				for (let i = 0; i < this.mesh.boneAabb.length; i++) {
					if (boneUsed[i]) {
						_tempBoneAabb.setFromTransformedAabb(this.mesh.boneAabb[i], this.skinInstance.matrices[i]);
						if (first) {
							first = false;
							localAabb.center.copy(_tempBoneAabb.center);
							localAabb.halfExtents.copy(_tempBoneAabb.halfExtents);
						} else {
							localAabb.add(_tempBoneAabb);
						}
					}
				}
				toWorldSpace = true;
			} else if (this.node._aabbVer !== this._aabbVer || this.mesh._aabbVer !== this._aabbMeshVer) {
				if (this.mesh) {
					localAabb.center.copy(this.mesh.aabb.center);
					localAabb.halfExtents.copy(this.mesh.aabb.halfExtents);
				} else {
					localAabb.center.set(0, 0, 0);
					localAabb.halfExtents.set(0, 0, 0);
				}
				if (this.mesh && this.mesh.morph) {
					const morphAabb = this.mesh.morph.aabb;
					localAabb._expand(morphAabb.getMin(), morphAabb.getMax());
				}
				toWorldSpace = true;
				this._aabbVer = this.node._aabbVer;
				this._aabbMeshVer = this.mesh._aabbVer;
			}
		}
		if (toWorldSpace) {
			this._aabb.setFromTransformedAabb(localAabb, this.node.getWorldTransform());
		}
		return this._aabb;
	}
	clearShaders() {
		const shaderCache = this._shaderCache;
		for (let i = 0; i < shaderCache.length; i++) {
			var _shaderCache$i;
			(_shaderCache$i = shaderCache[i]) == null || _shaderCache$i.destroy();
			shaderCache[i] = null;
		}
	}
	getShaderInstance(shaderPass, lightHash, scene, viewUniformFormat, viewBindGroupFormat, sortedLights) {
		let shaderInstance;
		let passEntry = this._shaderCache[shaderPass];
		if (passEntry) {
			shaderInstance = passEntry.shaderInstances.get(lightHash);
		} else {
			passEntry = new ShaderCacheEntry();
			this._shaderCache[shaderPass] = passEntry;
		}
		if (!shaderInstance) {
			const mat = this._material;
			const shaderDefs = this._shaderDefs;
			const variantKey = shaderPass + '_' + shaderDefs + '_' + lightHash;
			shaderInstance = new ShaderInstance();
			shaderInstance.shader = mat.variants.get(variantKey);
			if (!shaderInstance.shader) {
				const shader = mat.getShaderVariant(this.mesh.device, scene, shaderDefs, null, shaderPass, sortedLights, viewUniformFormat, viewBindGroupFormat, this._mesh.vertexBuffer.format);
				mat.variants.set(variantKey, shader);
				shaderInstance.shader = shader;
			}
			passEntry.shaderInstances.set(lightHash, shaderInstance);
		}
		return shaderInstance;
	}
	set material(material) {
		this.clearShaders();
		const prevMat = this._material;
		if (prevMat) {
			prevMat.removeMeshInstanceRef(this);
		}
		this._material = material;
		if (material) {
			material.addMeshInstanceRef(this);
			this.transparent = material.transparent;
			this.updateKey();
		}
	}
	get material() {
		return this._material;
	}
	set layer(layer) {
		this._layer = layer;
		this.updateKey();
	}
	get layer() {
		return this._layer;
	}
	_updateShaderDefs(shaderDefs) {
		if (shaderDefs !== this._shaderDefs) {
			this._shaderDefs = shaderDefs;
			this.clearShaders();
		}
	}
	set calculateSortDistance(calculateSortDistance) {
		this._calculateSortDistance = calculateSortDistance;
	}
	get calculateSortDistance() {
		return this._calculateSortDistance;
	}
	set receiveShadow(val) {
		if (this._receiveShadow !== val) {
			this._receiveShadow = val;
			this._updateShaderDefs(val ? this._shaderDefs & ~SHADERDEF_NOSHADOW : this._shaderDefs | SHADERDEF_NOSHADOW);
		}
	}
	get receiveShadow() {
		return this._receiveShadow;
	}
	set skinInstance(val) {
		this._skinInstance = val;
		this._updateShaderDefs(val ? this._shaderDefs | SHADERDEF_SKIN : this._shaderDefs & ~SHADERDEF_SKIN);
		this._setupSkinUpdate();
	}
	get skinInstance() {
		return this._skinInstance;
	}
	set morphInstance(val) {
		var _this$_morphInstance;
		(_this$_morphInstance = this._morphInstance) == null || _this$_morphInstance.destroy();
		this._morphInstance = val;
		let shaderDefs = this._shaderDefs;
		shaderDefs = val && val.morph.useTextureMorph ? shaderDefs | SHADERDEF_MORPH_TEXTURE_BASED : shaderDefs & ~SHADERDEF_MORPH_TEXTURE_BASED;
		shaderDefs = val && val.morph.morphPositions ? shaderDefs | SHADERDEF_MORPH_POSITION : shaderDefs & ~SHADERDEF_MORPH_POSITION;
		shaderDefs = val && val.morph.morphNormals ? shaderDefs | SHADERDEF_MORPH_NORMAL : shaderDefs & ~SHADERDEF_MORPH_NORMAL;
		this._updateShaderDefs(shaderDefs);
	}
	get morphInstance() {
		return this._morphInstance;
	}
	set screenSpace(val) {
		if (this._screenSpace !== val) {
			this._screenSpace = val;
			this._updateShaderDefs(val ? this._shaderDefs | SHADERDEF_SCREENSPACE : this._shaderDefs & ~SHADERDEF_SCREENSPACE);
		}
	}
	get screenSpace() {
		return this._screenSpace;
	}
	set key(val) {
		this._key[SORTKEY_FORWARD] = val;
	}
	get key() {
		return this._key[SORTKEY_FORWARD];
	}
	set mask(val) {
		const toggles = this._shaderDefs & 0x0000FFFF;
		this._updateShaderDefs(toggles | val << 16);
	}
	get mask() {
		return this._shaderDefs >> 16;
	}
	set instancingCount(value) {
		if (this.instancingData) this.instancingData.count = value;
	}
	get instancingCount() {
		return this.instancingData ? this.instancingData.count : 0;
	}
	destroy() {
		var _this$_skinInstance, _this$morphInstance;
		const mesh = this.mesh;
		if (mesh) {
			this.mesh = null;
			if (mesh.refCount < 1) {
				mesh.destroy();
			}
		}
		this.setRealtimeLightmap(MeshInstance.lightmapParamNames[0], null);
		this.setRealtimeLightmap(MeshInstance.lightmapParamNames[1], null);
		(_this$_skinInstance = this._skinInstance) == null || _this$_skinInstance.destroy();
		this._skinInstance = null;
		(_this$morphInstance = this.morphInstance) == null || _this$morphInstance.destroy();
		this.morphInstance = null;
		this.clearShaders();
		this.material = null;
	}
	static _prepareRenderStyleForArray(meshInstances, renderStyle) {
		if (meshInstances) {
			for (let i = 0; i < meshInstances.length; i++) {
				meshInstances[i]._renderStyle = renderStyle;
				const mesh = meshInstances[i].mesh;
				if (!_meshSet.has(mesh)) {
					_meshSet.add(mesh);
					mesh.prepareRenderState(renderStyle);
				}
			}
			_meshSet.clear();
		}
	}
	_isVisible(camera) {
		if (this.visible) {
			if (this.isVisibleFunc) {
				return this.isVisibleFunc(camera);
			}
			_tempSphere.center = this.aabb.center;
			_tempSphere.radius = this._aabb.halfExtents.length();
			return camera.frustum.containsSphere(_tempSphere);
		}
		return false;
	}
	updateKey() {
		const material = this.material;
		const blendType = material.alphaToCoverage || material.alphaTest ? BLEND_NORMAL : material.blendType;
		this._key[SORTKEY_FORWARD] = (this.layer & 0x0f) << 27 | (blendType === BLEND_NONE ? 1 : 0) << 26 | (material.id & 0x1ffffff) << 0;
	}
	setInstancing(vertexBuffer, cull = false) {
		if (vertexBuffer) {
			this.instancingData = new InstancingData(vertexBuffer.numVertices);
			this.instancingData.vertexBuffer = vertexBuffer;
			vertexBuffer.format.instancing = true;
			this.cull = cull;
		} else {
			this.instancingData = null;
			this.cull = true;
		}
		this._updateShaderDefs(vertexBuffer ? this._shaderDefs | SHADERDEF_INSTANCING : this._shaderDefs & ~SHADERDEF_INSTANCING);
	}
	ensureMaterial(device) {
		if (!this.material) {
			this.material = getDefaultMaterial(device);
		}
	}
	clearParameters() {
		this.parameters = {};
	}
	getParameters() {
		return this.parameters;
	}
	getParameter(name) {
		return this.parameters[name];
	}
	setParameter(name, data, passFlags = -262141) {
		if (data === undefined && typeof name === 'object') {
			const uniformObject = name;
			if (uniformObject.length) {
				for (let i = 0; i < uniformObject.length; i++) {
					this.setParameter(uniformObject[i]);
				}
				return;
			}
			name = uniformObject.name;
			data = uniformObject.value;
		}
		const param = this.parameters[name];
		if (param) {
			param.data = data;
			param.passFlags = passFlags;
		} else {
			this.parameters[name] = {
				scopeId: null,
				data: data,
				passFlags: passFlags
			};
		}
	}
	setRealtimeLightmap(name, texture) {
		const old = this.getParameter(name);
		if (old === texture) return;
		if (old) {
			LightmapCache.decRef(old.data);
		}
		if (texture) {
			LightmapCache.incRef(texture);
			this.setParameter(name, texture);
		} else {
			this.deleteParameter(name);
		}
	}
	deleteParameter(name) {
		if (this.parameters[name]) {
			delete this.parameters[name];
		}
	}
	setParameters(device, passFlag) {
		const parameters = this.parameters;
		for (const paramName in parameters) {
			const parameter = parameters[paramName];
			if (parameter.passFlags & passFlag) {
				if (!parameter.scopeId) {
					parameter.scopeId = device.scope.resolve(paramName);
				}
				parameter.scopeId.setValue(parameter.data);
			}
		}
	}
	setLightmapped(value) {
		if (value) {
			this.mask = (this.mask | MASK_AFFECT_LIGHTMAPPED) & ~(MASK_AFFECT_DYNAMIC | MASK_BAKE);
		} else {
			this.setRealtimeLightmap(MeshInstance.lightmapParamNames[0], null);
			this.setRealtimeLightmap(MeshInstance.lightmapParamNames[1], null);
			this._shaderDefs &= ~(SHADERDEF_LM | SHADERDEF_DIRLM | SHADERDEF_LMAMBIENT);
			this.mask = (this.mask | MASK_AFFECT_DYNAMIC) & ~(MASK_AFFECT_LIGHTMAPPED | MASK_BAKE);
		}
	}
	setCustomAabb(aabb) {
		if (aabb) {
			if (this._customAabb) {
				this._customAabb.copy(aabb);
			} else {
				this._customAabb = aabb.clone();
			}
		} else {
			this._customAabb = null;
			this._aabbVer = -1;
		}
		this._setupSkinUpdate();
	}
	_setupSkinUpdate() {
		if (this._skinInstance) {
			this._skinInstance._updateBeforeCull = !this._customAabb;
		}
	}
}
MeshInstance.lightmapParamNames = ['texture_lightMap', 'texture_dirLightMap'];

function paramsIdentical(a, b) {
	if (a && !b) return false;
	if (!a && b) return false;
	a = a.data;
	b = b.data;
	if (a === b) return true;
	if (a instanceof Float32Array && b instanceof Float32Array) {
		if (a.length !== b.length) return false;
		for (let i = 0; i < a.length; i++) {
			if (a[i] !== b[i]) return false;
		}
		return true;
	}
	return false;
}
function equalParamSets(params1, params2) {
	for (const param in params1) {
		if (params1.hasOwnProperty(param) && !paramsIdentical(params1[param], params2[param])) return false;
	}
	for (const param in params2) {
		if (params2.hasOwnProperty(param) && !paramsIdentical(params2[param], params1[param])) return false;
	}
	return true;
}
const _triFanIndices = [0, 1, 3, 2, 3, 1];
const _triStripIndices = [0, 1, 3, 0, 3, 2];
const mat3 = new Mat3();
function getScaleSign(mi) {
	return mi.node.worldTransform.scaleSign;
}
class BatchManager {
	constructor(device, root, scene) {
		this.device = device;
		this.rootNode = root;
		this.scene = scene;
		this._init = false;
		this._batchGroups = {};
		this._batchGroupCounter = 0;
		this._batchList = [];
		this._dirtyGroups = [];
	}
	destroy() {
		this.device = null;
		this.rootNode = null;
		this.scene = null;
		this._batchGroups = {};
		this._batchList = [];
		this._dirtyGroups = [];
	}
	addGroup(name, dynamic, maxAabbSize, id, layers) {
		if (id === undefined) {
			id = this._batchGroupCounter;
			this._batchGroupCounter++;
		}
		if (this._batchGroups[id]) {
			return undefined;
		}
		const group = new BatchGroup(id, name, dynamic, maxAabbSize, layers);
		this._batchGroups[id] = group;
		return group;
	}
	removeGroup(id) {
		if (!this._batchGroups[id]) {
			return;
		}
		const newBatchList = [];
		for (let i = 0; i < this._batchList.length; i++) {
			if (this._batchList[i].batchGroupId === id) {
				this.destroyBatch(this._batchList[i]);
			} else {
				newBatchList.push(this._batchList[i]);
			}
		}
		this._batchList = newBatchList;
		this._removeModelsFromBatchGroup(this.rootNode, id);
		delete this._batchGroups[id];
	}
	markGroupDirty(id) {
		if (this._dirtyGroups.indexOf(id) < 0) {
			this._dirtyGroups.push(id);
		}
	}
	getGroupByName(name) {
		const groups = this._batchGroups;
		for (const group in groups) {
			if (!groups.hasOwnProperty(group)) continue;
			if (groups[group].name === name) {
				return groups[group];
			}
		}
		return null;
	}
	getBatches(batchGroupId) {
		const results = [];
		const len = this._batchList.length;
		for (let i = 0; i < len; i++) {
			const batch = this._batchList[i];
			if (batch.batchGroupId === batchGroupId) {
				results.push(batch);
			}
		}
		return results;
	}
	_removeModelsFromBatchGroup(node, id) {
		if (!node.enabled) return;
		if (node.model && node.model.batchGroupId === id) {
			node.model.batchGroupId = -1;
		}
		if (node.render && node.render.batchGroupId === id) {
			node.render.batchGroupId = -1;
		}
		if (node.element && node.element.batchGroupId === id) {
			node.element.batchGroupId = -1;
		}
		if (node.sprite && node.sprite.batchGroupId === id) {
			node.sprite.batchGroupId = -1;
		}
		for (let i = 0; i < node._children.length; i++) {
			this._removeModelsFromBatchGroup(node._children[i], id);
		}
	}
	insert(type, groupId, node) {
		const group = this._batchGroups[groupId];
		if (group) {
			if (group._obj[type].indexOf(node) < 0) {
				group._obj[type].push(node);
				this.markGroupDirty(groupId);
			}
		}
	}
	remove(type, groupId, node) {
		const group = this._batchGroups[groupId];
		if (group) {
			const idx = group._obj[type].indexOf(node);
			if (idx >= 0) {
				group._obj[type].splice(idx, 1);
				this.markGroupDirty(groupId);
			}
		}
	}
	_extractRender(node, arr, group, groupMeshInstances) {
		if (node.render) {
			arr = groupMeshInstances[node.render.batchGroupId] = arr.concat(node.render.meshInstances);
			node.render.removeFromLayers();
		}
		return arr;
	}
	_extractModel(node, arr, group, groupMeshInstances) {
		if (node.model && node.model.model) {
			arr = groupMeshInstances[node.model.batchGroupId] = arr.concat(node.model.meshInstances);
			node.model.removeModelFromLayers();
		}
		return arr;
	}
	_extractElement(node, arr, group) {
		if (!node.element) return;
		let valid = false;
		if (node.element._text && node.element._text._model.meshInstances.length > 0) {
			arr.push(node.element._text._model.meshInstances[0]);
			node.element.removeModelFromLayers(node.element._text._model);
			valid = true;
		} else if (node.element._image) {
			arr.push(node.element._image._renderable.meshInstance);
			node.element.removeModelFromLayers(node.element._image._renderable.model);
			if (node.element._image._renderable.unmaskMeshInstance) {
				arr.push(node.element._image._renderable.unmaskMeshInstance);
				if (!node.element._image._renderable.unmaskMeshInstance.stencilFront || !node.element._image._renderable.unmaskMeshInstance.stencilBack) {
					node.element._dirtifyMask();
					node.element._onPrerender();
				}
			}
			valid = true;
		}
		if (valid) {
			group._ui = true;
		}
	}
	_collectAndRemoveMeshInstances(groupMeshInstances, groupIds) {
		for (let g = 0; g < groupIds.length; g++) {
			const id = groupIds[g];
			const group = this._batchGroups[id];
			if (!group) continue;
			let arr = groupMeshInstances[id];
			if (!arr) arr = groupMeshInstances[id] = [];
			for (let m = 0; m < group._obj.model.length; m++) {
				arr = this._extractModel(group._obj.model[m], arr, group, groupMeshInstances);
			}
			for (let r = 0; r < group._obj.render.length; r++) {
				arr = this._extractRender(group._obj.render[r], arr, group, groupMeshInstances);
			}
			for (let e = 0; e < group._obj.element.length; e++) {
				this._extractElement(group._obj.element[e], arr, group);
			}
			for (let s = 0; s < group._obj.sprite.length; s++) {
				const node = group._obj.sprite[s];
				if (node.sprite && node.sprite._meshInstance && (group.dynamic || node.sprite.sprite._renderMode === SPRITE_RENDERMODE_SIMPLE)) {
					arr.push(node.sprite._meshInstance);
					node.sprite.removeModelFromLayers();
					group._sprite = true;
					node.sprite._batchGroup = group;
				}
			}
		}
	}
	generate(groupIds) {
		const groupMeshInstances = {};
		if (!groupIds) {
			groupIds = Object.keys(this._batchGroups);
		}
		const newBatchList = [];
		for (let i = 0; i < this._batchList.length; i++) {
			if (groupIds.indexOf(this._batchList[i].batchGroupId) < 0) {
				newBatchList.push(this._batchList[i]);
				continue;
			}
			this.destroyBatch(this._batchList[i]);
		}
		this._batchList = newBatchList;
		this._collectAndRemoveMeshInstances(groupMeshInstances, groupIds);
		if (groupIds === this._dirtyGroups) {
			this._dirtyGroups.length = 0;
		} else {
			const newDirtyGroups = [];
			for (let i = 0; i < this._dirtyGroups.length; i++) {
				if (groupIds.indexOf(this._dirtyGroups[i]) < 0) newDirtyGroups.push(this._dirtyGroups[i]);
			}
			this._dirtyGroups = newDirtyGroups;
		}
		let group, lists, groupData, batch;
		for (const groupId in groupMeshInstances) {
			if (!groupMeshInstances.hasOwnProperty(groupId)) continue;
			group = groupMeshInstances[groupId];
			groupData = this._batchGroups[groupId];
			if (!groupData) {
				continue;
			}
			lists = this.prepare(group, groupData.dynamic, groupData.maxAabbSize, groupData._ui || groupData._sprite);
			for (let i = 0; i < lists.length; i++) {
				batch = this.create(lists[i], groupData.dynamic, parseInt(groupId, 10));
				if (batch) {
					batch.addToLayers(this.scene, groupData.layers);
				}
			}
		}
	}
	prepare(meshInstances, dynamic, maxAabbSize = Number.POSITIVE_INFINITY, translucent) {
		if (meshInstances.length === 0) return [];
		const halfMaxAabbSize = maxAabbSize * 0.5;
		const maxInstanceCount = this.device.supportsBoneTextures ? 1024 : this.device.boneLimit;
		const maxNumVertices = this.device.extUintElement ? 0xffffffff : 0xffff;
		const aabb = new BoundingBox();
		const testAabb = new BoundingBox();
		let skipTranslucentAabb = null;
		let sf;
		const lists = [];
		let j = 0;
		if (translucent) {
			meshInstances.sort(function (a, b) {
				return a.drawOrder - b.drawOrder;
			});
		}
		let meshInstancesLeftA = meshInstances;
		let meshInstancesLeftB;
		const skipMesh = translucent ? function (mi) {
			if (skipTranslucentAabb) {
				skipTranslucentAabb.add(mi.aabb);
			} else {
				skipTranslucentAabb = mi.aabb.clone();
			}
			meshInstancesLeftB.push(mi);
		} : function (mi) {
			meshInstancesLeftB.push(mi);
		};
		while (meshInstancesLeftA.length > 0) {
			lists[j] = [meshInstancesLeftA[0]];
			meshInstancesLeftB = [];
			const material = meshInstancesLeftA[0].material;
			const layer = meshInstancesLeftA[0].layer;
			const defs = meshInstancesLeftA[0]._shaderDefs;
			const params = meshInstancesLeftA[0].parameters;
			const stencil = meshInstancesLeftA[0].stencilFront;
			let vertCount = meshInstancesLeftA[0].mesh.vertexBuffer.getNumVertices();
			const drawOrder = meshInstancesLeftA[0].drawOrder;
			aabb.copy(meshInstancesLeftA[0].aabb);
			const scaleSign = getScaleSign(meshInstancesLeftA[0]);
			const vertexFormatBatchingHash = meshInstancesLeftA[0].mesh.vertexBuffer.format.batchingHash;
			const indexed = meshInstancesLeftA[0].mesh.primitive[0].indexed;
			skipTranslucentAabb = null;
			for (let i = 1; i < meshInstancesLeftA.length; i++) {
				const mi = meshInstancesLeftA[i];
				if (dynamic && lists[j].length >= maxInstanceCount) {
					meshInstancesLeftB = meshInstancesLeftB.concat(meshInstancesLeftA.slice(i));
					break;
				}
				if (material !== mi.material || layer !== mi.layer || vertexFormatBatchingHash !== mi.mesh.vertexBuffer.format.batchingHash || indexed !== mi.mesh.primitive[0].indexed || defs !== mi._shaderDefs || vertCount + mi.mesh.vertexBuffer.getNumVertices() > maxNumVertices) {
					skipMesh(mi);
					continue;
				}
				testAabb.copy(aabb);
				testAabb.add(mi.aabb);
				if (testAabb.halfExtents.x > halfMaxAabbSize || testAabb.halfExtents.y > halfMaxAabbSize || testAabb.halfExtents.z > halfMaxAabbSize) {
					skipMesh(mi);
					continue;
				}
				if (stencil) {
					if (!(sf = mi.stencilFront) || stencil.func !== sf.func || stencil.zpass !== sf.zpass) {
						skipMesh(mi);
						continue;
					}
				}
				if (scaleSign !== getScaleSign(mi)) {
					skipMesh(mi);
					continue;
				}
				if (!equalParamSets(params, mi.parameters)) {
					skipMesh(mi);
					continue;
				}
				if (translucent && skipTranslucentAabb && skipTranslucentAabb.intersects(mi.aabb) && mi.drawOrder !== drawOrder) {
					skipMesh(mi);
					continue;
				}
				aabb.add(mi.aabb);
				vertCount += mi.mesh.vertexBuffer.getNumVertices();
				lists[j].push(mi);
			}
			j++;
			meshInstancesLeftA = meshInstancesLeftB;
		}
		return lists;
	}
	collectBatchedMeshData(meshInstances, dynamic) {
		let streams = null;
		let batchNumVerts = 0;
		let batchNumIndices = 0;
		let material = null;
		for (let i = 0; i < meshInstances.length; i++) {
			if (meshInstances[i].visible) {
				const mesh = meshInstances[i].mesh;
				const numVerts = mesh.vertexBuffer.numVertices;
				batchNumVerts += numVerts;
				if (mesh.primitive[0].indexed) {
					batchNumIndices += mesh.primitive[0].count;
				} else {
					const primitiveType = mesh.primitive[0].type;
					if (primitiveType === PRIMITIVE_TRIFAN || primitiveType === PRIMITIVE_TRISTRIP) {
						if (mesh.primitive[0].count === 4) batchNumIndices += 6;
					}
				}
				if (!streams) {
					material = meshInstances[i].material;
					streams = {};
					const elems = mesh.vertexBuffer.format.elements;
					for (let j = 0; j < elems.length; j++) {
						const semantic = elems[j].name;
						streams[semantic] = {
							numComponents: elems[j].numComponents,
							dataType: elems[j].dataType,
							normalize: elems[j].normalize,
							count: 0
						};
					}
					if (dynamic) {
						streams[SEMANTIC_BLENDINDICES] = {
							numComponents: 1,
							dataType: TYPE_FLOAT32,
							normalize: false,
							count: 0
						};
					}
				}
			}
		}
		return {
			streams: streams,
			batchNumVerts: batchNumVerts,
			batchNumIndices: batchNumIndices,
			material: material
		};
	}
	create(meshInstances, dynamic, batchGroupId) {
		if (!this._init) {
			const boneLimit = '#define BONE_LIMIT ' + this.device.getBoneLimit() + '\n';
			this.transformVS = boneLimit + '#define DYNAMICBATCH\n' + shaderChunks.transformVS;
			this.skinTexVS = shaderChunks.skinBatchTexVS;
			this.skinConstVS = shaderChunks.skinBatchConstVS;
			this.vertexFormats = {};
			this._init = true;
		}
		let stream = null;
		let semantic;
		let mesh, numVerts;
		let batch = null;
		const batchData = this.collectBatchedMeshData(meshInstances, dynamic);
		if (batchData.streams) {
			const streams = batchData.streams;
			let material = batchData.material;
			const batchNumVerts = batchData.batchNumVerts;
			const batchNumIndices = batchData.batchNumIndices;
			batch = new Batch(meshInstances, dynamic, batchGroupId);
			this._batchList.push(batch);
			let indexBase, numIndices, indexData;
			let verticesOffset = 0;
			let indexOffset = 0;
			let transform;
			const vec = new Vec3();
			const indexArrayType = batchNumVerts <= 0xffff ? Uint16Array : Uint32Array;
			const indices = new indexArrayType(batchNumIndices);
			for (semantic in streams) {
				stream = streams[semantic];
				stream.typeArrayType = typedArrayTypes[stream.dataType];
				stream.elementByteSize = typedArrayTypesByteSize[stream.dataType];
				stream.buffer = new stream.typeArrayType(batchNumVerts * stream.numComponents);
			}
			for (let i = 0; i < meshInstances.length; i++) {
				if (!meshInstances[i].visible) continue;
				mesh = meshInstances[i].mesh;
				numVerts = mesh.vertexBuffer.numVertices;
				if (!dynamic) {
					transform = meshInstances[i].node.getWorldTransform();
				}
				for (semantic in streams) {
					if (semantic !== SEMANTIC_BLENDINDICES) {
						stream = streams[semantic];
						const subarray = new stream.typeArrayType(stream.buffer.buffer, stream.elementByteSize * stream.count);
						const totalComponents = mesh.getVertexStream(semantic, subarray) * stream.numComponents;
						stream.count += totalComponents;
						if (!dynamic && stream.numComponents >= 3) {
							if (semantic === SEMANTIC_POSITION) {
								for (let j = 0; j < totalComponents; j += stream.numComponents) {
									vec.set(subarray[j], subarray[j + 1], subarray[j + 2]);
									transform.transformPoint(vec, vec);
									subarray[j] = vec.x;
									subarray[j + 1] = vec.y;
									subarray[j + 2] = vec.z;
								}
							} else if (semantic === SEMANTIC_NORMAL || semantic === SEMANTIC_TANGENT) {
								mat3.invertMat4(transform).transpose();
								for (let j = 0; j < totalComponents; j += stream.numComponents) {
									vec.set(subarray[j], subarray[j + 1], subarray[j + 2]);
									mat3.transformVector(vec, vec);
									subarray[j] = vec.x;
									subarray[j + 1] = vec.y;
									subarray[j + 2] = vec.z;
								}
							}
						}
					}
				}
				if (dynamic) {
					stream = streams[SEMANTIC_BLENDINDICES];
					for (let j = 0; j < numVerts; j++) stream.buffer[stream.count++] = i;
				}
				if (mesh.primitive[0].indexed) {
					indexBase = mesh.primitive[0].base;
					numIndices = mesh.primitive[0].count;
					const srcFormat = mesh.indexBuffer[0].getFormat();
					indexData = new typedArrayIndexFormats[srcFormat](mesh.indexBuffer[0].storage);
				} else {
					const primitiveType = mesh.primitive[0].type;
					if (primitiveType === PRIMITIVE_TRIFAN || primitiveType === PRIMITIVE_TRISTRIP) {
						if (mesh.primitive[0].count === 4) {
							indexBase = 0;
							numIndices = 6;
							indexData = primitiveType === PRIMITIVE_TRIFAN ? _triFanIndices : _triStripIndices;
						} else {
							numIndices = 0;
							continue;
						}
					}
				}
				for (let j = 0; j < numIndices; j++) {
					indices[j + indexOffset] = indexData[indexBase + j] + verticesOffset;
				}
				indexOffset += numIndices;
				verticesOffset += numVerts;
			}
			mesh = new Mesh(this.device);
			for (semantic in streams) {
				stream = streams[semantic];
				mesh.setVertexStream(semantic, stream.buffer, stream.numComponents, undefined, stream.dataType, stream.normalize);
			}
			if (indices.length > 0) mesh.setIndices(indices);
			mesh.update(PRIMITIVE_TRIANGLES, false);
			if (dynamic) {
				material = material.clone();
				material.chunks.transformVS = this.transformVS;
				material.chunks.skinTexVS = this.skinTexVS;
				material.chunks.skinConstVS = this.skinConstVS;
				material.update();
			}
			const meshInstance = new MeshInstance(mesh, material, this.rootNode);
			meshInstance.castShadow = batch.origMeshInstances[0].castShadow;
			meshInstance.parameters = batch.origMeshInstances[0].parameters;
			meshInstance.layer = batch.origMeshInstances[0].layer;
			meshInstance._shaderDefs = batch.origMeshInstances[0]._shaderDefs;
			meshInstance.cull = batch.origMeshInstances[0].cull;
			const batchGroup = this._batchGroups[batchGroupId];
			if (batchGroup && batchGroup._ui) meshInstance.cull = false;
			if (dynamic) {
				const nodes = [];
				for (let i = 0; i < batch.origMeshInstances.length; i++) {
					nodes.push(batch.origMeshInstances[i].node);
				}
				meshInstance.skinInstance = new SkinBatchInstance(this.device, nodes, this.rootNode);
			}
			meshInstance._updateAabb = false;
			meshInstance.drawOrder = batch.origMeshInstances[0].drawOrder;
			meshInstance.stencilFront = batch.origMeshInstances[0].stencilFront;
			meshInstance.stencilBack = batch.origMeshInstances[0].stencilBack;
			meshInstance.flipFacesFactor = getScaleSign(batch.origMeshInstances[0]);
			meshInstance.castShadow = batch.origMeshInstances[0].castShadow;
			batch.meshInstance = meshInstance;
			batch.updateBoundingBox();
		}
		return batch;
	}
	updateAll() {
		if (this._dirtyGroups.length > 0) {
			this.generate(this._dirtyGroups);
		}
		for (let i = 0; i < this._batchList.length; i++) {
			if (!this._batchList[i].dynamic) continue;
			this._batchList[i].updateBoundingBox();
		}
	}
	clone(batch, clonedMeshInstances) {
		const batch2 = new Batch(clonedMeshInstances, batch.dynamic, batch.batchGroupId);
		this._batchList.push(batch2);
		const nodes = [];
		for (let i = 0; i < clonedMeshInstances.length; i++) {
			nodes.push(clonedMeshInstances[i].node);
		}
		batch2.meshInstance = new MeshInstance(batch.meshInstance.mesh, batch.meshInstance.material, batch.meshInstance.node);
		batch2.meshInstance._updateAabb = false;
		batch2.meshInstance.parameters = clonedMeshInstances[0].parameters;
		batch2.meshInstance.cull = clonedMeshInstances[0].cull;
		batch2.meshInstance.layer = clonedMeshInstances[0].layer;
		if (batch.dynamic) {
			batch2.meshInstance.skinInstance = new SkinBatchInstance(this.device, nodes, this.rootNode);
		}
		batch2.meshInstance.castShadow = batch.meshInstance.castShadow;
		batch2.meshInstance._shader = batch.meshInstance._shader.slice();
		batch2.meshInstance.castShadow = batch.meshInstance.castShadow;
		return batch2;
	}
	destroyBatch(batch) {
		batch.destroy(this.scene, this._batchGroups[batch.batchGroupId].layers);
	}
}

const _colorUniformNames = ['uSceneColorMap', 'texture_grabPass'];
class RenderPassColorGrab extends RenderPass {
	constructor(...args) {
		super(...args);
		this.colorRenderTarget = null;
		this.source = null;
	}
	destroy() {
		super.destroy();
		this.releaseRenderTarget(this.colorRenderTarget);
	}
	shouldReallocate(targetRT, sourceTexture, sourceFormat) {
		const targetFormat = targetRT == null ? void 0 : targetRT.colorBuffer.format;
		if (targetFormat !== sourceFormat) return true;
		const width = (sourceTexture == null ? void 0 : sourceTexture.width) || this.device.width;
		const height = (sourceTexture == null ? void 0 : sourceTexture.height) || this.device.height;
		return !targetRT || width !== targetRT.width || height !== targetRT.height;
	}
	allocateRenderTarget(renderTarget, sourceRenderTarget, device, format) {
		const mipmaps = device.isWebGL2;
		const texture = new Texture(device, {
			name: _colorUniformNames[0],
			format,
			width: sourceRenderTarget ? sourceRenderTarget.colorBuffer.width : device.width,
			height: sourceRenderTarget ? sourceRenderTarget.colorBuffer.height : device.height,
			mipmaps,
			minFilter: mipmaps ? FILTER_LINEAR_MIPMAP_LINEAR : FILTER_LINEAR,
			magFilter: FILTER_LINEAR,
			addressU: ADDRESS_CLAMP_TO_EDGE,
			addressV: ADDRESS_CLAMP_TO_EDGE
		});
		if (renderTarget) {
			renderTarget.destroyFrameBuffers();
			renderTarget._colorBuffer = texture;
			renderTarget._colorBuffers = [texture];
		} else {
			renderTarget = new RenderTarget({
				name: 'ColorGrabRT',
				colorBuffer: texture,
				depth: false,
				stencil: false,
				autoResolve: false
			});
		}
		return renderTarget;
	}
	releaseRenderTarget(rt) {
		if (rt) {
			rt.destroyTextureBuffers();
			rt.destroy();
		}
	}
	frameUpdate() {
		var _sourceRt$colorBuffer;
		const device = this.device;
		const sourceRt = this.source;
		const sourceFormat = (_sourceRt$colorBuffer = sourceRt == null ? void 0 : sourceRt.colorBuffer.format) != null ? _sourceRt$colorBuffer : this.device.backBufferFormat;
		if (this.shouldReallocate(this.colorRenderTarget, sourceRt == null ? void 0 : sourceRt.colorBuffer, sourceFormat)) {
			this.releaseRenderTarget(this.colorRenderTarget);
			this.colorRenderTarget = this.allocateRenderTarget(this.colorRenderTarget, sourceRt, device, sourceFormat);
		}
		const colorBuffer = this.colorRenderTarget.colorBuffer;
		_colorUniformNames.forEach(name => device.scope.resolve(name).setValue(colorBuffer));
	}
	execute() {
		const device = this.device;
		const sourceRt = this.source;
		const colorBuffer = this.colorRenderTarget.colorBuffer;
		if (device.isWebGPU) {
			device.copyRenderTarget(sourceRt, this.colorRenderTarget, true, false);
			device.mipmapRenderer.generate(this.colorRenderTarget.colorBuffer.impl);
		} else if (device.isWebGL2) {
			device.copyRenderTarget(sourceRt, this.colorRenderTarget, true, false);
			device.activeTexture(device.maxCombinedTextures - 1);
			device.bindTexture(colorBuffer);
			device.gl.generateMipmap(colorBuffer.impl._glTarget);
		} else {
			if (!colorBuffer.impl._glTexture) {
				colorBuffer.impl.initialize(device, colorBuffer);
			}
			device.bindTexture(colorBuffer);
			const gl = device.gl;
			gl.copyTexImage2D(gl.TEXTURE_2D, 0, colorBuffer.impl._glFormat, 0, 0, colorBuffer.width, colorBuffer.height, 0);
			colorBuffer._needsUpload = false;
			colorBuffer._needsMipmapsUpload = false;
		}
	}
}

const _depthUniformNames$1 = ['uSceneDepthMap', 'uDepthMap'];
class RenderPassDepthGrab extends RenderPass {
	constructor(device, camera) {
		super(device);
		this.depthRenderTarget = null;
		this.camera = null;
		this.camera = camera;
	}
	destroy() {
		super.destroy();
		this.releaseRenderTarget(this.depthRenderTarget);
	}
	shouldReallocate(targetRT, sourceTexture) {
		const width = (sourceTexture == null ? void 0 : sourceTexture.width) || this.device.width;
		const height = (sourceTexture == null ? void 0 : sourceTexture.height) || this.device.height;
		return !targetRT || width !== targetRT.width || height !== targetRT.height;
	}
	allocateRenderTarget(renderTarget, sourceRenderTarget, device, format, isDepth) {
		const texture = new Texture(device, {
			name: _depthUniformNames$1[0],
			format,
			width: sourceRenderTarget ? sourceRenderTarget.colorBuffer.width : device.width,
			height: sourceRenderTarget ? sourceRenderTarget.colorBuffer.height : device.height,
			mipmaps: false,
			minFilter: FILTER_NEAREST,
			magFilter: FILTER_NEAREST,
			addressU: ADDRESS_CLAMP_TO_EDGE,
			addressV: ADDRESS_CLAMP_TO_EDGE
		});
		if (renderTarget) {
			renderTarget.destroyFrameBuffers();
			if (isDepth) {
				renderTarget._depthBuffer = texture;
			} else {
				renderTarget._colorBuffer = texture;
				renderTarget._colorBuffers = [texture];
			}
		} else {
			renderTarget = new RenderTarget({
				name: 'DepthGrabRT',
				colorBuffer: isDepth ? null : texture,
				depthBuffer: isDepth ? texture : null,
				depth: !isDepth,
				stencil: device.supportsStencil,
				autoResolve: false
			});
		}
		return renderTarget;
	}
	releaseRenderTarget(rt) {
		if (rt) {
			rt.destroyTextureBuffers();
			rt.destroy();
		}
	}
	before() {
		var _camera$renderTarget, _camera$renderTarget$, _camera$renderTarget2, _camera$renderTarget3;
		const camera = this.camera;
		const device = this.device;
		const destinationRt = (_camera$renderTarget = camera == null ? void 0 : camera.renderTarget) != null ? _camera$renderTarget : device.backBuffer;
		let useDepthBuffer = true;
		let format = destinationRt.stencil ? PIXELFORMAT_DEPTHSTENCIL : PIXELFORMAT_DEPTH;
		if (device.isWebGPU) {
			const numSamples = destinationRt.samples;
			if (numSamples > 1) {
				format = PIXELFORMAT_R32F;
				useDepthBuffer = false;
			}
		}
		const sourceTexture = (_camera$renderTarget$ = (_camera$renderTarget2 = camera.renderTarget) == null ? void 0 : _camera$renderTarget2.depthBuffer) != null ? _camera$renderTarget$ : (_camera$renderTarget3 = camera.renderTarget) == null ? void 0 : _camera$renderTarget3.colorBuffer;
		if (this.shouldReallocate(this.depthRenderTarget, sourceTexture)) {
			this.releaseRenderTarget(this.depthRenderTarget);
			this.depthRenderTarget = this.allocateRenderTarget(this.depthRenderTarget, camera.renderTarget, device, format, useDepthBuffer);
		}
		const colorBuffer = useDepthBuffer ? this.depthRenderTarget.depthBuffer : this.depthRenderTarget.colorBuffer;
		_depthUniformNames$1.forEach(name => device.scope.resolve(name).setValue(colorBuffer));
	}
	execute() {
		const device = this.device;
		if (device.isWebGL2 && device.renderTarget.samples > 1) {
			const src = device.renderTarget.impl._glFrameBuffer;
			const dest = this.depthRenderTarget;
			device.renderTarget = dest;
			device.updateBegin();
			this.depthRenderTarget.impl.internalResolve(device, src, dest.impl._glFrameBuffer, this.depthRenderTarget, device.gl.DEPTH_BUFFER_BIT);
		} else {
			device.copyRenderTarget(device.renderTarget, this.depthRenderTarget, false, true);
		}
	}
}

const webgl1DepthClearColor = new Color(254.0 / 255, 254.0 / 255, 254.0 / 255, 254.0 / 255);
const tempMeshInstances$1 = [];
const lights$1 = [[], [], []];
const _depthUniformNames = ['uSceneDepthMap', 'uDepthMap'];
class RenderPassDepth extends RenderPass {
	constructor(device, renderer, camera) {
		super(device);
		this.renderer = renderer;
		this.camera = camera;
		this.setupRenderTarget();
	}
	destroy() {
		super.destroy();
		if (this.renderTarget) {
			this.renderTarget.destroyTextureBuffers();
			this.renderTarget.destroy();
			this.renderTarget = null;
		}
	}
	update(scene) {
		this.scene = scene;
	}
	setupRenderTarget() {
		const texture = new Texture(this.device, {
			name: _depthUniformNames[0],
			format: PIXELFORMAT_RGBA8,
			width: 4,
			height: 4,
			mipmaps: false,
			minFilter: FILTER_NEAREST,
			magFilter: FILTER_NEAREST,
			addressU: ADDRESS_CLAMP_TO_EDGE,
			addressV: ADDRESS_CLAMP_TO_EDGE
		});
		const renderTarget = new RenderTarget({
			name: `${_depthUniformNames[0]}RT}`,
			colorBuffer: texture,
			depth: true,
			stencil: false
		});
		this.init(renderTarget, {});
		this.setClearColor(webgl1DepthClearColor);
		this.setClearDepth(1.0);
	}
	before() {
		const device = this.device;
		const colorBuffer = this.renderTarget.colorBuffer;
		_depthUniformNames.forEach(name => device.scope.resolve(name).setValue(colorBuffer));
	}
	execute() {
		const {
			device,
			renderer,
			camera,
			scene,
			renderTarget
		} = this;
		const layers = scene.layers.layerList;
		const subLayerEnabled = scene.layers.subLayerEnabled;
		const isTransparent = scene.layers.subLayerList;
		for (let i = 0; i < layers.length; i++) {
			const layer = layers[i];
			if (layer.enabled && subLayerEnabled[i]) {
				if (layer.camerasSet.has(camera)) {
					if (layer.id === LAYERID_DEPTH) break;
					const culledInstances = layer.getCulledInstances(camera);
					const meshInstances = isTransparent[i] ? culledInstances.transparent : culledInstances.opaque;
					for (let j = 0; j < meshInstances.length; j++) {
						var _meshInstance$materia;
						const meshInstance = meshInstances[j];
						if ((_meshInstance$materia = meshInstance.material) != null && _meshInstance$materia.depthWrite) {
							tempMeshInstances$1.push(meshInstance);
						}
					}
					renderer.setCameraUniforms(camera, renderTarget);
					renderer.renderForward(camera, tempMeshInstances$1, lights$1, SHADER_DEPTH, meshInstance => {
						device.setBlendState(BlendState.NOBLEND);
					}, layer);
					tempMeshInstances$1.length = 0;
				}
			}
		}
	}
}

const _deviceCoord = new Vec3();
const _halfSize = new Vec3();
const _point$1 = new Vec3();
const _invViewProjMat = new Mat4();
const _frustumPoints = [new Vec3(), new Vec3(), new Vec3(), new Vec3(), new Vec3(), new Vec3(), new Vec3(), new Vec3()];
class Camera {
	constructor() {
		this.shaderPassInfo = null;
		this.renderPassColorGrab = null;
		this.renderPassDepthGrab = null;
		this.renderPasses = [];
		this.jitter = 0;
		this._aspectRatio = 16 / 9;
		this._aspectRatioMode = ASPECT_AUTO;
		this._calculateProjection = null;
		this._calculateTransform = null;
		this._clearColor = new Color(0.75, 0.75, 0.75, 1);
		this._clearColorBuffer = true;
		this._clearDepth = 1;
		this._clearDepthBuffer = true;
		this._clearStencil = 0;
		this._clearStencilBuffer = true;
		this._cullFaces = true;
		this._farClip = 1000;
		this._flipFaces = false;
		this._fov = 45;
		this._frustumCulling = true;
		this._horizontalFov = false;
		this._layers = [LAYERID_WORLD, LAYERID_DEPTH, LAYERID_SKYBOX, LAYERID_UI, LAYERID_IMMEDIATE];
		this._layersSet = new Set(this._layers);
		this._nearClip = 0.1;
		this._node = null;
		this._orthoHeight = 10;
		this._projection = PROJECTION_PERSPECTIVE;
		this._rect = new Vec4(0, 0, 1, 1);
		this._renderTarget = null;
		this._scissorRect = new Vec4(0, 0, 1, 1);
		this._scissorRectClear = false;
		this._aperture = 16.0;
		this._shutter = 1.0 / 1000.0;
		this._sensitivity = 1000;
		this._projMat = new Mat4();
		this._projMatDirty = true;
		this._projMatSkybox = new Mat4();
		this._viewMat = new Mat4();
		this._viewMatDirty = true;
		this._viewProjMat = new Mat4();
		this._viewProjMatDirty = true;
		this._shaderMatricesVersion = 0;
		this._viewProjInverse = new Mat4();
		this._viewProjCurrent = null;
		this._viewProjPrevious = new Mat4();
		this._jitters = [0, 0, 0, 0];
		this.frustum = new Frustum();
		this._xr = null;
		this._xrProperties = {
			horizontalFov: this._horizontalFov,
			fov: this._fov,
			aspectRatio: this._aspectRatio,
			farClip: this._farClip,
			nearClip: this._nearClip
		};
	}
	destroy() {
		var _this$renderPassColor, _this$renderPassDepth;
		(_this$renderPassColor = this.renderPassColorGrab) == null || _this$renderPassColor.destroy();
		this.renderPassColorGrab = null;
		(_this$renderPassDepth = this.renderPassDepthGrab) == null || _this$renderPassDepth.destroy();
		this.renderPassDepthGrab = null;
		this.renderPasses.length = 0;
	}
	_storeShaderMatrices(viewProjMat, jitterX, jitterY, renderVersion) {
		if (this._shaderMatricesVersion !== renderVersion) {
			var _this$_viewProjCurren, _this$_viewProjCurren2;
			this._shaderMatricesVersion = renderVersion;
			this._viewProjPrevious.copy((_this$_viewProjCurren = this._viewProjCurrent) != null ? _this$_viewProjCurren : viewProjMat);
			(_this$_viewProjCurren2 = this._viewProjCurrent) != null ? _this$_viewProjCurren2 : this._viewProjCurrent = new Mat4();
			this._viewProjCurrent.copy(viewProjMat);
			this._viewProjInverse.invert(viewProjMat);
			this._jitters[2] = this._jitters[0];
			this._jitters[3] = this._jitters[1];
			this._jitters[0] = jitterX;
			this._jitters[1] = jitterY;
		}
	}
	get fullSizeClearRect() {
		const rect = this._scissorRectClear ? this.scissorRect : this._rect;
		return rect.x === 0 && rect.y === 0 && rect.z === 1 && rect.w === 1;
	}
	set aspectRatio(newValue) {
		if (this._aspectRatio !== newValue) {
			this._aspectRatio = newValue;
			this._projMatDirty = true;
		}
	}
	get aspectRatio() {
		var _this$xr;
		return (_this$xr = this.xr) != null && _this$xr.active ? this._xrProperties.aspectRatio : this._aspectRatio;
	}
	set aspectRatioMode(newValue) {
		if (this._aspectRatioMode !== newValue) {
			this._aspectRatioMode = newValue;
			this._projMatDirty = true;
		}
	}
	get aspectRatioMode() {
		return this._aspectRatioMode;
	}
	set calculateProjection(newValue) {
		this._calculateProjection = newValue;
		this._projMatDirty = true;
	}
	get calculateProjection() {
		return this._calculateProjection;
	}
	set calculateTransform(newValue) {
		this._calculateTransform = newValue;
	}
	get calculateTransform() {
		return this._calculateTransform;
	}
	set clearColor(newValue) {
		this._clearColor.copy(newValue);
	}
	get clearColor() {
		return this._clearColor;
	}
	set clearColorBuffer(newValue) {
		this._clearColorBuffer = newValue;
	}
	get clearColorBuffer() {
		return this._clearColorBuffer;
	}
	set clearDepth(newValue) {
		this._clearDepth = newValue;
	}
	get clearDepth() {
		return this._clearDepth;
	}
	set clearDepthBuffer(newValue) {
		this._clearDepthBuffer = newValue;
	}
	get clearDepthBuffer() {
		return this._clearDepthBuffer;
	}
	set clearStencil(newValue) {
		this._clearStencil = newValue;
	}
	get clearStencil() {
		return this._clearStencil;
	}
	set clearStencilBuffer(newValue) {
		this._clearStencilBuffer = newValue;
	}
	get clearStencilBuffer() {
		return this._clearStencilBuffer;
	}
	set cullFaces(newValue) {
		this._cullFaces = newValue;
	}
	get cullFaces() {
		return this._cullFaces;
	}
	set farClip(newValue) {
		if (this._farClip !== newValue) {
			this._farClip = newValue;
			this._projMatDirty = true;
		}
	}
	get farClip() {
		var _this$xr2;
		return (_this$xr2 = this.xr) != null && _this$xr2.active ? this._xrProperties.farClip : this._farClip;
	}
	set flipFaces(newValue) {
		this._flipFaces = newValue;
	}
	get flipFaces() {
		return this._flipFaces;
	}
	set fov(newValue) {
		if (this._fov !== newValue) {
			this._fov = newValue;
			this._projMatDirty = true;
		}
	}
	get fov() {
		var _this$xr3;
		return (_this$xr3 = this.xr) != null && _this$xr3.active ? this._xrProperties.fov : this._fov;
	}
	set frustumCulling(newValue) {
		this._frustumCulling = newValue;
	}
	get frustumCulling() {
		return this._frustumCulling;
	}
	set horizontalFov(newValue) {
		if (this._horizontalFov !== newValue) {
			this._horizontalFov = newValue;
			this._projMatDirty = true;
		}
	}
	get horizontalFov() {
		var _this$xr4;
		return (_this$xr4 = this.xr) != null && _this$xr4.active ? this._xrProperties.horizontalFov : this._horizontalFov;
	}
	set layers(newValue) {
		this._layers = newValue.slice(0);
		this._layersSet = new Set(this._layers);
	}
	get layers() {
		return this._layers;
	}
	get layersSet() {
		return this._layersSet;
	}
	set nearClip(newValue) {
		if (this._nearClip !== newValue) {
			this._nearClip = newValue;
			this._projMatDirty = true;
		}
	}
	get nearClip() {
		var _this$xr5;
		return (_this$xr5 = this.xr) != null && _this$xr5.active ? this._xrProperties.nearClip : this._nearClip;
	}
	set node(newValue) {
		this._node = newValue;
	}
	get node() {
		return this._node;
	}
	set orthoHeight(newValue) {
		if (this._orthoHeight !== newValue) {
			this._orthoHeight = newValue;
			this._projMatDirty = true;
		}
	}
	get orthoHeight() {
		return this._orthoHeight;
	}
	set projection(newValue) {
		if (this._projection !== newValue) {
			this._projection = newValue;
			this._projMatDirty = true;
		}
	}
	get projection() {
		return this._projection;
	}
	get projectionMatrix() {
		this._evaluateProjectionMatrix();
		return this._projMat;
	}
	set rect(newValue) {
		this._rect.copy(newValue);
	}
	get rect() {
		return this._rect;
	}
	set renderTarget(newValue) {
		this._renderTarget = newValue;
	}
	get renderTarget() {
		return this._renderTarget;
	}
	set scissorRect(newValue) {
		this._scissorRect.copy(newValue);
	}
	get scissorRect() {
		return this._scissorRect;
	}
	get viewMatrix() {
		if (this._viewMatDirty) {
			const wtm = this._node.getWorldTransform();
			this._viewMat.copy(wtm).invert();
			this._viewMatDirty = false;
		}
		return this._viewMat;
	}
	set aperture(newValue) {
		this._aperture = newValue;
	}
	get aperture() {
		return this._aperture;
	}
	set sensitivity(newValue) {
		this._sensitivity = newValue;
	}
	get sensitivity() {
		return this._sensitivity;
	}
	set shutter(newValue) {
		this._shutter = newValue;
	}
	get shutter() {
		return this._shutter;
	}
	set xr(newValue) {
		if (this._xr !== newValue) {
			this._xr = newValue;
			this._projMatDirty = true;
		}
	}
	get xr() {
		return this._xr;
	}
	clone() {
		return new Camera().copy(this);
	}
	copy(other) {
		this._aspectRatio = other._aspectRatio;
		this._farClip = other._farClip;
		this._fov = other._fov;
		this._horizontalFov = other._horizontalFov;
		this._nearClip = other._nearClip;
		this._xrProperties.aspectRatio = other._xrProperties.aspectRatio;
		this._xrProperties.farClip = other._xrProperties.farClip;
		this._xrProperties.fov = other._xrProperties.fov;
		this._xrProperties.horizontalFov = other._xrProperties.horizontalFov;
		this._xrProperties.nearClip = other._xrProperties.nearClip;
		this.aspectRatioMode = other.aspectRatioMode;
		this.calculateProjection = other.calculateProjection;
		this.calculateTransform = other.calculateTransform;
		this.clearColor = other.clearColor;
		this.clearColorBuffer = other.clearColorBuffer;
		this.clearDepth = other.clearDepth;
		this.clearDepthBuffer = other.clearDepthBuffer;
		this.clearStencil = other.clearStencil;
		this.clearStencilBuffer = other.clearStencilBuffer;
		this.cullFaces = other.cullFaces;
		this.flipFaces = other.flipFaces;
		this.frustumCulling = other.frustumCulling;
		this.layers = other.layers;
		this.orthoHeight = other.orthoHeight;
		this.projection = other.projection;
		this.rect = other.rect;
		this.renderTarget = other.renderTarget;
		this.scissorRect = other.scissorRect;
		this.aperture = other.aperture;
		this.shutter = other.shutter;
		this.sensitivity = other.sensitivity;
		this.shaderPassInfo = other.shaderPassInfo;
		this.jitter = other.jitter;
		this._projMatDirty = true;
		return this;
	}
	_enableRenderPassColorGrab(device, enable) {
		if (enable) {
			if (!this.renderPassColorGrab) {
				this.renderPassColorGrab = new RenderPassColorGrab(device);
			}
		} else {
			var _this$renderPassColor2;
			(_this$renderPassColor2 = this.renderPassColorGrab) == null || _this$renderPassColor2.destroy();
			this.renderPassColorGrab = null;
		}
	}
	_enableRenderPassDepthGrab(device, renderer, enable) {
		if (enable) {
			if (!this.renderPassDepthGrab) {
				this.renderPassDepthGrab = device.isWebGL1 ? new RenderPassDepth(device, renderer, this) : new RenderPassDepthGrab(device, this);
			}
		} else {
			var _this$renderPassDepth2;
			(_this$renderPassDepth2 = this.renderPassDepthGrab) == null || _this$renderPassDepth2.destroy();
			this.renderPassDepthGrab = null;
		}
	}
	_updateViewProjMat() {
		if (this._projMatDirty || this._viewMatDirty || this._viewProjMatDirty) {
			this._viewProjMat.mul2(this.projectionMatrix, this.viewMatrix);
			this._viewProjMatDirty = false;
		}
	}
	worldToScreen(worldCoord, cw, ch, screenCoord = new Vec3()) {
		this._updateViewProjMat();
		this._viewProjMat.transformPoint(worldCoord, screenCoord);
		const vpm = this._viewProjMat.data;
		const w = worldCoord.x * vpm[3] + worldCoord.y * vpm[7] + worldCoord.z * vpm[11] + 1 * vpm[15];
		screenCoord.x = (screenCoord.x / w + 1) * 0.5 * cw;
		screenCoord.y = (1 - screenCoord.y / w) * 0.5 * ch;
		return screenCoord;
	}
	screenToWorld(x, y, z, cw, ch, worldCoord = new Vec3()) {
		const range = this.farClip - this.nearClip;
		_deviceCoord.set(x / cw, (ch - y) / ch, z / range);
		_deviceCoord.mulScalar(2);
		_deviceCoord.sub(Vec3.ONE);
		if (this._projection === PROJECTION_PERSPECTIVE) {
			Mat4._getPerspectiveHalfSize(_halfSize, this.fov, this.aspectRatio, this.nearClip, this.horizontalFov);
			_halfSize.x *= _deviceCoord.x;
			_halfSize.y *= _deviceCoord.y;
			const invView = this._node.getWorldTransform();
			_halfSize.z = -this.nearClip;
			invView.transformPoint(_halfSize, _point$1);
			const cameraPos = this._node.getPosition();
			worldCoord.sub2(_point$1, cameraPos);
			worldCoord.normalize();
			worldCoord.mulScalar(z);
			worldCoord.add(cameraPos);
		} else {
			this._updateViewProjMat();
			_invViewProjMat.copy(this._viewProjMat).invert();
			_invViewProjMat.transformPoint(_deviceCoord, worldCoord);
		}
		return worldCoord;
	}
	_evaluateProjectionMatrix() {
		if (this._projMatDirty) {
			if (this._projection === PROJECTION_PERSPECTIVE) {
				this._projMat.setPerspective(this.fov, this.aspectRatio, this.nearClip, this.farClip, this.horizontalFov);
				this._projMatSkybox.copy(this._projMat);
			} else {
				const y = this._orthoHeight;
				const x = y * this.aspectRatio;
				this._projMat.setOrtho(-x, x, -y, y, this.nearClip, this.farClip);
				this._projMatSkybox.setPerspective(this.fov, this.aspectRatio, this.nearClip, this.farClip);
			}
			this._projMatDirty = false;
		}
	}
	getProjectionMatrixSkybox() {
		this._evaluateProjectionMatrix();
		return this._projMatSkybox;
	}
	getExposure() {
		const ev100 = Math.log2(this._aperture * this._aperture / this._shutter * 100.0 / this._sensitivity);
		return 1.0 / (Math.pow(2.0, ev100) * 1.2);
	}
	getScreenSize(sphere) {
		if (this._projection === PROJECTION_PERSPECTIVE) {
			const distance = this._node.getPosition().distance(sphere.center);
			if (distance < sphere.radius) {
				return 1;
			}
			const viewAngle = Math.asin(sphere.radius / distance);
			const sphereViewHeight = Math.tan(viewAngle);
			const screenViewHeight = Math.tan(this.fov / 2 * math.DEG_TO_RAD);
			return Math.min(sphereViewHeight / screenViewHeight, 1);
		}
		return math.clamp(sphere.radius / this._orthoHeight, 0, 1);
	}
	getFrustumCorners(near = this.nearClip, far = this.farClip) {
		const fov = this.fov * Math.PI / 180.0;
		let y = this._projection === PROJECTION_PERSPECTIVE ? Math.tan(fov / 2.0) * near : this._orthoHeight;
		let x = y * this.aspectRatio;
		const points = _frustumPoints;
		points[0].x = x;
		points[0].y = -y;
		points[0].z = -near;
		points[1].x = x;
		points[1].y = y;
		points[1].z = -near;
		points[2].x = -x;
		points[2].y = y;
		points[2].z = -near;
		points[3].x = -x;
		points[3].y = -y;
		points[3].z = -near;
		if (this._projection === PROJECTION_PERSPECTIVE) {
			y = Math.tan(fov / 2.0) * far;
			x = y * this.aspectRatio;
		}
		points[4].x = x;
		points[4].y = -y;
		points[4].z = -far;
		points[5].x = x;
		points[5].y = y;
		points[5].z = -far;
		points[6].x = -x;
		points[6].y = y;
		points[6].z = -far;
		points[7].x = -x;
		points[7].y = -y;
		points[7].z = -far;
		return points;
	}
	setXrProperties(properties) {
		Object.assign(this._xrProperties, properties);
		this._projMatDirty = true;
	}
}

class LitShaderOptions {
	constructor() {
		this.hasTangents = false;
		this.chunks = {};
		this.pass = 0;
		this.alphaTest = false;
		this.blendType = BLEND_NONE;
		this.separateAmbient = false;
		this.screenSpace = false;
		this.skin = false;
		this.useInstancing = false;
		this.useMorphPosition = false;
		this.useMorphNormal = false;
		this.useMorphTextureBased = false;
		this.nineSlicedMode = 0;
		this.clusteredLightingEnabled = true;
		this.clusteredLightingCookiesEnabled = false;
		this.clusteredLightingShadowsEnabled = false;
		this.clusteredLightingShadowType = 0;
		this.clusteredLightingAreaLightsEnabled = false;
		this.vertexColors = false;
		this.lightMapEnabled = false;
		this.dirLightMapEnabled = false;
		this.useHeights = false;
		this.useNormals = false;
		this.useClearCoatNormals = false;
		this.useAo = false;
		this.diffuseMapEnabled = false;
		this.useAmbientTint = false;
		this.customFragmentShader = null;
		this.pixelSnap = false;
		this.shadingModel = 0;
		this.ambientSH = false;
		this.fastTbn = false;
		this.twoSidedLighting = false;
		this.occludeDirect = false;
		this.occludeSpecular = 0;
		this.occludeSpecularFloat = false;
		this.useMsdf = false;
		this.msdfTextAttribute = false;
		this.alphaToCoverage = false;
		this.opacityFadesSpecular = false;
		this.opacityDither = DITHER_NONE;
		this.opacityShadowDither = DITHER_NONE;
		this.cubeMapProjection = 0;
		this.conserveEnergy = false;
		this.useSpecular = false;
		this.useSpecularityFactor = false;
		this.enableGGXSpecular = false;
		this.fresnelModel = 0;
		this.useRefraction = false;
		this.useClearCoat = false;
		this.useSheen = false;
		this.useIridescence = false;
		this.useMetalness = false;
		this.useDynamicRefraction = false;
		this.fog = FOG_NONE;
		this.gamma = GAMMA_NONE;
		this.toneMap = -1;
		this.fixSeams = false;
		this.reflectionSource = null;
		this.reflectionEncoding = null;
		this.reflectionCubemapEncoding = null;
		this.ambientSource = 'constant';
		this.ambientEncoding = null;
		this.skyboxIntensity = 1.0;
		this.useCubeMapRotation = false;
		this.lightMapWithoutAmbient = false;
		this.lights = [];
		this.noShadow = false;
		this.lightMaskDynamic = 0x0;
		this.userAttributes = {};
	}
}

class LitMaterialOptions {
	constructor() {
		this.usedUvs = void 0;
		this.shaderChunk = void 0;
		this.litOptions = new LitShaderOptions();
	}
}

class LitMaterialOptionsBuilder {
	static update(litOptions, material, scene, objDefs, pass, sortedLights) {
		LitMaterialOptionsBuilder.updateSharedOptions(litOptions, material, scene, objDefs, pass);
		LitMaterialOptionsBuilder.updateMaterialOptions(litOptions, material);
		LitMaterialOptionsBuilder.updateEnvOptions(litOptions, material, scene);
		LitMaterialOptionsBuilder.updateLightingOptions(litOptions, material, objDefs, sortedLights);
		if (pass === SHADER_FORWARDHDR) {
			litOptions.gamma = GAMMA_SRGBHDR;
			litOptions.toneMap = TONEMAP_LINEAR;
		}
	}
	static updateSharedOptions(litOptions, material, scene, objDefs, pass) {
		litOptions.chunks = material.chunks;
		litOptions.pass = pass;
		litOptions.alphaTest = material.alphaTest > 0;
		litOptions.blendType = material.blendType;
		litOptions.screenSpace = objDefs && (objDefs & SHADERDEF_SCREENSPACE) !== 0;
		litOptions.skin = objDefs && (objDefs & SHADERDEF_SKIN) !== 0;
		litOptions.useInstancing = objDefs && (objDefs & SHADERDEF_INSTANCING) !== 0;
		litOptions.useMorphPosition = objDefs && (objDefs & SHADERDEF_MORPH_POSITION) !== 0;
		litOptions.useMorphNormal = objDefs && (objDefs & SHADERDEF_MORPH_NORMAL) !== 0;
		litOptions.useMorphTextureBased = objDefs && (objDefs & SHADERDEF_MORPH_TEXTURE_BASED) !== 0;
		litOptions.hasTangents = objDefs && (objDefs & SHADERDEF_TANGENTS) !== 0;
		litOptions.nineSlicedMode = material.nineSlicedMode || SPRITE_RENDERMODE_SIMPLE;
		if (material.useLighting && scene.clusteredLightingEnabled) {
			litOptions.clusteredLightingEnabled = true;
			litOptions.clusteredLightingCookiesEnabled = scene.lighting.cookiesEnabled;
			litOptions.clusteredLightingShadowsEnabled = scene.lighting.shadowsEnabled;
			litOptions.clusteredLightingShadowType = scene.lighting.shadowType;
			litOptions.clusteredLightingAreaLightsEnabled = scene.lighting.areaLightsEnabled;
		} else {
			litOptions.clusteredLightingEnabled = false;
			litOptions.clusteredLightingCookiesEnabled = false;
			litOptions.clusteredLightingShadowsEnabled = false;
			litOptions.clusteredLightingAreaLightsEnabled = false;
		}
	}
	static updateMaterialOptions(litOptions, material) {
		litOptions.useAmbientTint = false;
		litOptions.separateAmbient = false;
		litOptions.customFragmentShader = null;
		litOptions.pixelSnap = material.pixelSnap;
		litOptions.shadingModel = material.shadingModel;
		litOptions.ambientSH = material.ambientSH;
		litOptions.fastTbn = material.fastTbn;
		litOptions.twoSidedLighting = material.twoSidedLighting;
		litOptions.occludeDirect = material.occludeDirect;
		litOptions.occludeSpecular = material.occludeSpecular;
		litOptions.occludeSpecularFloat = material.occludeSpecularIntensity !== 1.0;
		litOptions.useMsdf = false;
		litOptions.msdfTextAttribute = false;
		litOptions.alphaToCoverage = material.alphaToCoverage;
		litOptions.opacityFadesSpecular = material.opacityFadesSpecular;
		litOptions.opacityDither = material.opacityDither;
		litOptions.opacityShadowDither = material.opacityShadowDither;
		litOptions.cubeMapProjection = CUBEPROJ_NONE;
		litOptions.conserveEnergy = material.conserveEnergy && material.shadingModel === SPECULAR_BLINN;
		litOptions.useSpecular = material.hasSpecular;
		litOptions.useSpecularityFactor = material.hasSpecularityFactor;
		litOptions.enableGGXSpecular = material.ggxSpecular;
		litOptions.fresnelModel = material.fresnelModel;
		litOptions.useRefraction = material.hasRefraction;
		litOptions.useClearCoat = material.hasClearCoat;
		litOptions.useSheen = material.hasSheen;
		litOptions.useIridescence = material.hasIrridescence;
		litOptions.useMetalness = material.hasMetalness;
		litOptions.useDynamicRefraction = material.dynamicRefraction;
		litOptions.vertexColors = false;
		litOptions.lightMapEnabled = material.hasLighting;
		litOptions.dirLightMapEnabled = material.dirLightMap;
		litOptions.useHeights = material.hasHeights;
		litOptions.useNormals = material.hasNormals;
		litOptions.useClearCoatNormals = material.hasClearCoatNormals;
		litOptions.useAo = material.hasAo;
		litOptions.diffuseMapEnabled = material.hasDiffuseMap;
	}
	static updateEnvOptions(litOptions, material, scene) {
		litOptions.fog = material.useFog ? scene.fog : 'none';
		litOptions.gamma = material.useGammaTonemap ? scene.gammaCorrection : GAMMA_NONE;
		litOptions.toneMap = material.useGammaTonemap ? scene.toneMapping : -1;
		litOptions.fixSeams = false;
		if (material.useSkybox && scene.envAtlas && scene.skybox) {
			litOptions.reflectionSource = 'envAtlasHQ';
			litOptions.reflectionEncoding = scene.envAtlas.encoding;
			litOptions.reflectionCubemapEncoding = scene.skybox.encoding;
		} else if (material.useSkybox && scene.envAtlas) {
			litOptions.reflectionSource = 'envAtlas';
			litOptions.reflectionEncoding = scene.envAtlas.encoding;
		} else if (material.useSkybox && scene.skybox) {
			litOptions.reflectionSource = 'cubeMap';
			litOptions.reflectionEncoding = scene.skybox.encoding;
		} else {
			litOptions.reflectionSource = null;
			litOptions.reflectionEncoding = null;
		}
		if (material.ambientSH) {
			litOptions.ambientSource = 'ambientSH';
			litOptions.ambientEncoding = null;
		} else if (litOptions.reflectionSource && scene.envAtlas) {
			litOptions.ambientSource = 'envAtlas';
			litOptions.ambientEncoding = scene.envAtlas.encoding;
		} else {
			litOptions.ambientSource = 'constant';
			litOptions.ambientEncoding = null;
		}
		const hasSkybox = !!litOptions.reflectionSource;
		litOptions.skyboxIntensity = hasSkybox && (scene.skyboxIntensity !== 1 || scene.physicalUnits);
		litOptions.useCubeMapRotation = hasSkybox && scene._skyboxRotationShaderInclude;
	}
	static updateLightingOptions(litOptions, material, objDefs, sortedLights) {
		litOptions.lightMapWithoutAmbient = false;
		if (material.useLighting) {
			const lightsFiltered = [];
			const mask = objDefs ? objDefs >> 16 : MASK_AFFECT_DYNAMIC;
			litOptions.lightMaskDynamic = !!(mask & MASK_AFFECT_DYNAMIC);
			litOptions.lightMapWithoutAmbient = false;
			if (sortedLights) {
				LitMaterialOptionsBuilder.collectLights(LIGHTTYPE_DIRECTIONAL, sortedLights[LIGHTTYPE_DIRECTIONAL], lightsFiltered, mask);
				LitMaterialOptionsBuilder.collectLights(LIGHTTYPE_OMNI, sortedLights[LIGHTTYPE_OMNI], lightsFiltered, mask);
				LitMaterialOptionsBuilder.collectLights(LIGHTTYPE_SPOT, sortedLights[LIGHTTYPE_SPOT], lightsFiltered, mask);
			}
			litOptions.lights = lightsFiltered;
		} else {
			litOptions.lights = [];
		}
		if (litOptions.lights.length === 0 || (objDefs & SHADERDEF_NOSHADOW) !== 0) {
			litOptions.noShadow = true;
		}
	}
	static collectLights(lType, lights, lightsFiltered, mask) {
		for (let i = 0; i < lights.length; i++) {
			const light = lights[i];
			if (light.enabled) {
				if (light.mask & mask) {
					lightsFiltered.push(light);
				}
			}
		}
	}
}

class ChunkBuilder {
	constructor() {
		this.code = '';
	}
	append(...chunks) {
		chunks.forEach(chunk => {
			if (chunk.endsWith('\n')) {
				this.code += chunk;
			} else {
				this.code += chunk + '\n';
			}
		});
	}
	prepend(...chunks) {
		chunks.forEach(chunk => {
			if (chunk.endsWith('\n')) {
				this.code = chunk + this.code;
			} else {
				this.code = chunk + '\n' + this.code;
			}
		});
	}
}

const decodeTable = {
	'linear': 'decodeLinear',
	'srgb': 'decodeGamma',
	'rgbm': 'decodeRGBM',
	'rgbe': 'decodeRGBE',
	'rgbp': 'decodeRGBP'
};
const encodeTable = {
	'linear': 'encodeLinear',
	'srgb': 'encodeGamma',
	'rgbm': 'encodeRGBM',
	'rgbe': 'encodeRGBE',
	'rgbp': 'encodeRGBP'
};
class ChunkUtils {
	static decodeFunc(encoding) {
		return decodeTable[encoding] || 'decodeGamma';
	}
	static encodeFunc(encoding) {
		return encodeTable[encoding] || 'encodeGamma';
	}
}

const _viewMat = new Mat4();
const _viewProjMat = new Mat4();
const _viewportMatrix = new Mat4();
class LightCamera {
	static create(name, lightType, face) {
		const camera = new Camera();
		camera.node = new GraphNode(name);
		camera.aspectRatio = 1;
		camera.aspectRatioMode = ASPECT_MANUAL;
		camera._scissorRectClear = true;
		switch (lightType) {
			case LIGHTTYPE_OMNI:
				camera.node.setRotation(LightCamera.pointLightRotations[face]);
				camera.fov = 90;
				camera.projection = PROJECTION_PERSPECTIVE;
				break;
			case LIGHTTYPE_SPOT:
				camera.projection = PROJECTION_PERSPECTIVE;
				break;
			case LIGHTTYPE_DIRECTIONAL:
				camera.projection = PROJECTION_ORTHOGRAPHIC;
				break;
		}
		return camera;
	}
	static evalSpotCookieMatrix(light) {
		let cookieCamera = LightCamera._spotCookieCamera;
		if (!cookieCamera) {
			cookieCamera = LightCamera.create('SpotCookieCamera', LIGHTTYPE_SPOT);
			LightCamera._spotCookieCamera = cookieCamera;
		}
		cookieCamera.fov = light._outerConeAngle * 2;
		const cookieNode = cookieCamera._node;
		cookieNode.setPosition(light._node.getPosition());
		cookieNode.setRotation(light._node.getRotation());
		cookieNode.rotateLocal(-90, 0, 0);
		_viewMat.setTRS(cookieNode.getPosition(), cookieNode.getRotation(), Vec3.ONE).invert();
		_viewProjMat.mul2(cookieCamera.projectionMatrix, _viewMat);
		const cookieMatrix = light.cookieMatrix;
		const rectViewport = light.atlasViewport;
		_viewportMatrix.setViewport(rectViewport.x, rectViewport.y, rectViewport.z, rectViewport.w);
		cookieMatrix.mul2(_viewportMatrix, _viewProjMat);
		return cookieMatrix;
	}
}
LightCamera.pointLightRotations = [new Quat().setFromEulerAngles(0, 90, 180), new Quat().setFromEulerAngles(0, -90, 180), new Quat().setFromEulerAngles(90, 0, 0), new Quat().setFromEulerAngles(-90, 0, 0), new Quat().setFromEulerAngles(0, 180, 180), new Quat().setFromEulerAngles(0, 0, 180)];
LightCamera._spotCookieCamera = null;

const epsilon$1 = 0.000001;
const tempVec3$1 = new Vec3();
const tempAreaLightSizes = new Float32Array(6);
const areaHalfAxisWidth = new Vec3(-0.5, 0, 0);
const areaHalfAxisHeight = new Vec3(0, 0, 0.5);
const TextureIndex8 = {
	FLAGS: 0,
	COLOR_A: 1,
	COLOR_B: 2,
	SPOT_ANGLES: 3,
	SHADOW_BIAS: 4,
	COOKIE_A: 5,
	COOKIE_B: 6,
	COUNT_ALWAYS: 7,
	POSITION_X: 7,
	POSITION_Y: 8,
	POSITION_Z: 9,
	RANGE: 10,
	SPOT_DIRECTION_X: 11,
	SPOT_DIRECTION_Y: 12,
	SPOT_DIRECTION_Z: 13,
	PROJ_MAT_00: 14,
	ATLAS_VIEWPORT_A: 14,
	PROJ_MAT_01: 15,
	ATLAS_VIEWPORT_B: 15,
	PROJ_MAT_02: 16,
	PROJ_MAT_03: 17,
	PROJ_MAT_10: 18,
	PROJ_MAT_11: 19,
	PROJ_MAT_12: 20,
	PROJ_MAT_13: 21,
	PROJ_MAT_20: 22,
	PROJ_MAT_21: 23,
	PROJ_MAT_22: 24,
	PROJ_MAT_23: 25,
	PROJ_MAT_30: 26,
	PROJ_MAT_31: 27,
	PROJ_MAT_32: 28,
	PROJ_MAT_33: 29,
	AREA_DATA_WIDTH_X: 30,
	AREA_DATA_WIDTH_Y: 31,
	AREA_DATA_WIDTH_Z: 32,
	AREA_DATA_HEIGHT_X: 33,
	AREA_DATA_HEIGHT_Y: 34,
	AREA_DATA_HEIGHT_Z: 35,
	COUNT: 36
};
const TextureIndexFloat = {
	POSITION_RANGE: 0,
	SPOT_DIRECTION: 1,
	PROJ_MAT_0: 2,
	ATLAS_VIEWPORT: 2,
	PROJ_MAT_1: 3,
	PROJ_MAT_2: 4,
	PROJ_MAT_3: 5,
	AREA_DATA_WIDTH: 6,
	AREA_DATA_HEIGHT: 7,
	COUNT: 8
};
const FORMAT_FLOAT = 0;
const FORMAT_8BIT = 1;
const shaderDefinesDeviceCache = new DeviceCache();
class LightsBuffer {
	static getLightTextureFormat(device) {
		return device.extTextureFloat && device.maxTextures > 8 ? FORMAT_FLOAT : FORMAT_8BIT;
	}
	static getShaderDefines(device) {
		return shaderDefinesDeviceCache.get(device, () => {
			const buildShaderDefines = (device, object, prefix, floatOffset) => {
				return Object.keys(object).map(key => `#define ${prefix}${key} ${object[key]}${floatOffset}`).join('\n');
			};
			const lightTextureFormat = LightsBuffer.getLightTextureFormat(device);
			const clusterTextureFormat = lightTextureFormat === FORMAT_FLOAT ? 'FLOAT' : '8BIT';
			const floatOffset = device.supportsTextureFetch ? '' : '.5';
			return `
								\n#define CLUSTER_TEXTURE_${clusterTextureFormat}
								${buildShaderDefines(device, TextureIndex8, 'CLUSTER_TEXTURE_8_', floatOffset)}
								${buildShaderDefines(device, TextureIndexFloat, 'CLUSTER_TEXTURE_F_', floatOffset)}
						`;
		});
	}
	constructor(device) {
		this.device = device;
		this.cookiesEnabled = false;
		this.shadowsEnabled = false;
		this.areaLightsEnabled = false;
		this.maxLights = 255;
		let pixelsPerLight8 = TextureIndex8.COUNT_ALWAYS;
		let pixelsPerLightFloat = 0;
		this.lightTextureFormat = LightsBuffer.getLightTextureFormat(device);
		if (this.lightTextureFormat === FORMAT_FLOAT) {
			pixelsPerLightFloat = TextureIndexFloat.COUNT;
		} else {
			pixelsPerLight8 = TextureIndex8.COUNT;
		}
		this.lights8 = new Uint8ClampedArray(4 * pixelsPerLight8 * this.maxLights);
		this.lightsTexture8 = this.createTexture(this.device, pixelsPerLight8, this.maxLights, PIXELFORMAT_RGBA8, 'LightsTexture8');
		this._lightsTexture8Id = this.device.scope.resolve('lightsTexture8');
		if (pixelsPerLightFloat) {
			this.lightsFloat = new Float32Array(4 * pixelsPerLightFloat * this.maxLights);
			this.lightsTextureFloat = this.createTexture(this.device, pixelsPerLightFloat, this.maxLights, PIXELFORMAT_RGBA32F, 'LightsTextureFloat');
			this._lightsTextureFloatId = this.device.scope.resolve('lightsTextureFloat');
		} else {
			this.lightsFloat = null;
			this.lightsTextureFloat = null;
			this._lightsTextureFloatId = undefined;
		}
		this._lightsTextureInvSizeId = this.device.scope.resolve('lightsTextureInvSize');
		this._lightsTextureInvSizeData = new Float32Array(4);
		this._lightsTextureInvSizeData[0] = pixelsPerLightFloat ? 1.0 / this.lightsTextureFloat.width : 0;
		this._lightsTextureInvSizeData[1] = pixelsPerLightFloat ? 1.0 / this.lightsTextureFloat.height : 0;
		this._lightsTextureInvSizeData[2] = 1.0 / this.lightsTexture8.width;
		this._lightsTextureInvSizeData[3] = 1.0 / this.lightsTexture8.height;
		this.invMaxColorValue = 0;
		this.invMaxAttenuation = 0;
		this.boundsMin = new Vec3();
		this.boundsDelta = new Vec3();
	}
	destroy() {
		if (this.lightsTexture8) {
			this.lightsTexture8.destroy();
			this.lightsTexture8 = null;
		}
		if (this.lightsTextureFloat) {
			this.lightsTextureFloat.destroy();
			this.lightsTextureFloat = null;
		}
	}
	createTexture(device, width, height, format, name) {
		const tex = new Texture(device, {
			name: name,
			width: width,
			height: height,
			mipmaps: false,
			format: format,
			addressU: ADDRESS_CLAMP_TO_EDGE,
			addressV: ADDRESS_CLAMP_TO_EDGE,
			type: TEXTURETYPE_DEFAULT,
			magFilter: FILTER_NEAREST,
			minFilter: FILTER_NEAREST,
			anisotropy: 1
		});
		return tex;
	}
	setCompressionRanges(maxAttenuation, maxColorValue) {
		this.invMaxColorValue = 1 / maxColorValue;
		this.invMaxAttenuation = 1 / maxAttenuation;
	}
	setBounds(min, delta) {
		this.boundsMin.copy(min);
		this.boundsDelta.copy(delta);
	}
	uploadTextures() {
		if (this.lightsTextureFloat) {
			this.lightsTextureFloat.lock().set(this.lightsFloat);
			this.lightsTextureFloat.unlock();
		}
		this.lightsTexture8.lock().set(this.lights8);
		this.lightsTexture8.unlock();
	}
	updateUniforms() {
		this._lightsTexture8Id.setValue(this.lightsTexture8);
		if (this.lightTextureFormat === FORMAT_FLOAT) {
			this._lightsTextureFloatId.setValue(this.lightsTextureFloat);
		}
		this._lightsTextureInvSizeId.setValue(this._lightsTextureInvSizeData);
	}
	getSpotDirection(direction, spot) {
		const mat = spot._node.getWorldTransform();
		mat.getY(direction).mulScalar(-1);
		direction.normalize();
	}
	getLightAreaSizes(light) {
		const mat = light._node.getWorldTransform();
		mat.transformVector(areaHalfAxisWidth, tempVec3$1);
		tempAreaLightSizes[0] = tempVec3$1.x;
		tempAreaLightSizes[1] = tempVec3$1.y;
		tempAreaLightSizes[2] = tempVec3$1.z;
		mat.transformVector(areaHalfAxisHeight, tempVec3$1);
		tempAreaLightSizes[3] = tempVec3$1.x;
		tempAreaLightSizes[4] = tempVec3$1.y;
		tempAreaLightSizes[5] = tempVec3$1.z;
		return tempAreaLightSizes;
	}
	addLightDataFlags(data8, index, light, isSpot, castShadows, shadowIntensity) {
		data8[index + 0] = isSpot ? 255 : 0;
		data8[index + 1] = light._shape * 64;
		data8[index + 2] = light._falloffMode * 255;
		data8[index + 3] = castShadows ? shadowIntensity * 255 : 0;
	}
	addLightDataColor(data8, index, light, gammaCorrection, isCookie) {
		const invMaxColorValue = this.invMaxColorValue;
		const color = gammaCorrection ? light._linearFinalColor : light._finalColor;
		FloatPacking.float2Bytes(color[0] * invMaxColorValue, data8, index + 0, 2);
		FloatPacking.float2Bytes(color[1] * invMaxColorValue, data8, index + 2, 2);
		FloatPacking.float2Bytes(color[2] * invMaxColorValue, data8, index + 4, 2);
		data8[index + 6] = isCookie ? 255 : 0;
		const isDynamic = !!(light.mask & MASK_AFFECT_DYNAMIC);
		const isLightmapped = !!(light.mask & MASK_AFFECT_LIGHTMAPPED);
		data8[index + 7] = isDynamic && isLightmapped ? 127 : isLightmapped ? 255 : 0;
	}
	addLightDataSpotAngles(data8, index, light) {
		FloatPacking.float2Bytes(light._innerConeAngleCos * (0.5 - epsilon$1) + 0.5, data8, index + 0, 2);
		FloatPacking.float2Bytes(light._outerConeAngleCos * (0.5 - epsilon$1) + 0.5, data8, index + 2, 2);
	}
	addLightDataShadowBias(data8, index, light) {
		const lightRenderData = light.getRenderData(null, 0);
		const biases = light._getUniformBiasValues(lightRenderData);
		FloatPacking.float2BytesRange(biases.bias, data8, index, -1, 20, 2);
		FloatPacking.float2Bytes(biases.normalBias, data8, index + 2, 2);
	}
	addLightDataPositionRange(data8, index, light, pos) {
		const normPos = tempVec3$1.sub2(pos, this.boundsMin).div(this.boundsDelta);
		FloatPacking.float2Bytes(normPos.x, data8, index + 0, 4);
		FloatPacking.float2Bytes(normPos.y, data8, index + 4, 4);
		FloatPacking.float2Bytes(normPos.z, data8, index + 8, 4);
		FloatPacking.float2Bytes(light.attenuationEnd * this.invMaxAttenuation, data8, index + 12, 4);
	}
	addLightDataSpotDirection(data8, index, light) {
		this.getSpotDirection(tempVec3$1, light);
		FloatPacking.float2Bytes(tempVec3$1.x * (0.5 - epsilon$1) + 0.5, data8, index + 0, 4);
		FloatPacking.float2Bytes(tempVec3$1.y * (0.5 - epsilon$1) + 0.5, data8, index + 4, 4);
		FloatPacking.float2Bytes(tempVec3$1.z * (0.5 - epsilon$1) + 0.5, data8, index + 8, 4);
	}
	addLightDataLightProjMatrix(data8, index, lightProjectionMatrix) {
		const matData = lightProjectionMatrix.data;
		for (let m = 0; m < 12; m++) FloatPacking.float2BytesRange(matData[m], data8, index + 4 * m, -2, 2, 4);
		for (let m = 12; m < 16; m++) {
			FloatPacking.float2MantissaExponent(matData[m], data8, index + 4 * m, 4);
		}
	}
	addLightDataCookies(data8, index, light) {
		const isRgb = light._cookieChannel === 'rgb';
		data8[index + 0] = Math.floor(light.cookieIntensity * 255);
		data8[index + 1] = isRgb ? 255 : 0;
		if (!isRgb) {
			const channel = light._cookieChannel;
			data8[index + 4] = channel === 'rrr' ? 255 : 0;
			data8[index + 5] = channel === 'ggg' ? 255 : 0;
			data8[index + 6] = channel === 'bbb' ? 255 : 0;
			data8[index + 7] = channel === 'aaa' ? 255 : 0;
		}
	}
	addLightAtlasViewport(data8, index, atlasViewport) {
		FloatPacking.float2Bytes(atlasViewport.x, data8, index + 0, 2);
		FloatPacking.float2Bytes(atlasViewport.y, data8, index + 2, 2);
		FloatPacking.float2Bytes(atlasViewport.z / 3, data8, index + 4, 2);
	}
	addLightAreaSizes(data8, index, light) {
		const areaSizes = this.getLightAreaSizes(light);
		for (let i = 0; i < 6; i++) {
			FloatPacking.float2MantissaExponent(areaSizes[i], data8, index + 4 * i, 4);
		}
	}
	addLightData(light, lightIndex, gammaCorrection) {
		const isSpot = light._type === LIGHTTYPE_SPOT;
		const hasAtlasViewport = light.atlasViewportAllocated;
		const isCookie = this.cookiesEnabled && !!light._cookie && hasAtlasViewport;
		const isArea = this.areaLightsEnabled && light.shape !== LIGHTSHAPE_PUNCTUAL;
		const castShadows = this.shadowsEnabled && light.castShadows && hasAtlasViewport;
		const pos = light._node.getPosition();
		let lightProjectionMatrix = null;
		let atlasViewport = null;
		if (isSpot) {
			if (castShadows) {
				const lightRenderData = light.getRenderData(null, 0);
				lightProjectionMatrix = lightRenderData.shadowMatrix;
			} else if (isCookie) {
				lightProjectionMatrix = LightCamera.evalSpotCookieMatrix(light);
			}
		} else {
			if (castShadows || isCookie) {
				atlasViewport = light.atlasViewport;
			}
		}
		const data8 = this.lights8;
		const data8Start = lightIndex * this.lightsTexture8.width * 4;
		this.addLightDataFlags(data8, data8Start + 4 * TextureIndex8.FLAGS, light, isSpot, castShadows, light.shadowIntensity);
		this.addLightDataColor(data8, data8Start + 4 * TextureIndex8.COLOR_A, light, gammaCorrection, isCookie);
		if (isSpot) {
			this.addLightDataSpotAngles(data8, data8Start + 4 * TextureIndex8.SPOT_ANGLES, light);
		}
		if (light.castShadows) {
			this.addLightDataShadowBias(data8, data8Start + 4 * TextureIndex8.SHADOW_BIAS, light);
		}
		if (isCookie) {
			this.addLightDataCookies(data8, data8Start + 4 * TextureIndex8.COOKIE_A, light);
		}
		if (this.lightTextureFormat === FORMAT_FLOAT) {
			const dataFloat = this.lightsFloat;
			const dataFloatStart = lightIndex * this.lightsTextureFloat.width * 4;
			dataFloat[dataFloatStart + 4 * TextureIndexFloat.POSITION_RANGE + 0] = pos.x;
			dataFloat[dataFloatStart + 4 * TextureIndexFloat.POSITION_RANGE + 1] = pos.y;
			dataFloat[dataFloatStart + 4 * TextureIndexFloat.POSITION_RANGE + 2] = pos.z;
			dataFloat[dataFloatStart + 4 * TextureIndexFloat.POSITION_RANGE + 3] = light.attenuationEnd;
			if (isSpot) {
				this.getSpotDirection(tempVec3$1, light);
				dataFloat[dataFloatStart + 4 * TextureIndexFloat.SPOT_DIRECTION + 0] = tempVec3$1.x;
				dataFloat[dataFloatStart + 4 * TextureIndexFloat.SPOT_DIRECTION + 1] = tempVec3$1.y;
				dataFloat[dataFloatStart + 4 * TextureIndexFloat.SPOT_DIRECTION + 2] = tempVec3$1.z;
			}
			if (lightProjectionMatrix) {
				const matData = lightProjectionMatrix.data;
				for (let m = 0; m < 16; m++) dataFloat[dataFloatStart + 4 * TextureIndexFloat.PROJ_MAT_0 + m] = matData[m];
			}
			if (atlasViewport) {
				dataFloat[dataFloatStart + 4 * TextureIndexFloat.ATLAS_VIEWPORT + 0] = atlasViewport.x;
				dataFloat[dataFloatStart + 4 * TextureIndexFloat.ATLAS_VIEWPORT + 1] = atlasViewport.y;
				dataFloat[dataFloatStart + 4 * TextureIndexFloat.ATLAS_VIEWPORT + 2] = atlasViewport.z / 3;
			}
			if (isArea) {
				const areaSizes = this.getLightAreaSizes(light);
				dataFloat[dataFloatStart + 4 * TextureIndexFloat.AREA_DATA_WIDTH + 0] = areaSizes[0];
				dataFloat[dataFloatStart + 4 * TextureIndexFloat.AREA_DATA_WIDTH + 1] = areaSizes[1];
				dataFloat[dataFloatStart + 4 * TextureIndexFloat.AREA_DATA_WIDTH + 2] = areaSizes[2];
				dataFloat[dataFloatStart + 4 * TextureIndexFloat.AREA_DATA_HEIGHT + 0] = areaSizes[3];
				dataFloat[dataFloatStart + 4 * TextureIndexFloat.AREA_DATA_HEIGHT + 1] = areaSizes[4];
				dataFloat[dataFloatStart + 4 * TextureIndexFloat.AREA_DATA_HEIGHT + 2] = areaSizes[5];
			}
		} else {
			this.addLightDataPositionRange(data8, data8Start + 4 * TextureIndex8.POSITION_X, light, pos);
			if (isSpot) {
				this.addLightDataSpotDirection(data8, data8Start + 4 * TextureIndex8.SPOT_DIRECTION_X, light);
			}
			if (lightProjectionMatrix) {
				this.addLightDataLightProjMatrix(data8, data8Start + 4 * TextureIndex8.PROJ_MAT_00, lightProjectionMatrix);
			}
			if (atlasViewport) {
				this.addLightAtlasViewport(data8, data8Start + 4 * TextureIndex8.ATLAS_VIEWPORT_A, atlasViewport);
			}
			if (isArea) {
				this.addLightAreaSizes(data8, data8Start + 4 * TextureIndex8.AREA_DATA_WIDTH_X, light);
			}
		}
	}
}

const builtinAttributes = {
	vertex_normal: SEMANTIC_NORMAL,
	vertex_tangent: SEMANTIC_TANGENT,
	vertex_texCoord0: SEMANTIC_TEXCOORD0,
	vertex_texCoord1: SEMANTIC_TEXCOORD1,
	vertex_color: SEMANTIC_COLOR,
	vertex_boneWeights: SEMANTIC_BLENDWEIGHT,
	vertex_boneIndices: SEMANTIC_BLENDINDICES
};
const builtinVaryings = {
	vVertexColor: "vec4",
	vPositionW: "vec3",
	vNormalV: "vec3",
	vNormalW: "vec3",
	vTangentW: "vec3",
	vBinormalW: "vec3",
	vObjectSpaceUpW: "vec3",
	vUv0: "vec2",
	vUv1: "vec2"
};
class LitShader {
	constructor(device, options) {
		this.device = device;
		this.options = options;
		this.attributes = {
			vertex_position: SEMANTIC_POSITION
		};
		if (options.userAttributes) {
			for (const [semantic, name] of Object.entries(options.userAttributes)) {
				this.attributes[name] = semantic;
			}
		}
		if (options.chunks) {
			const userChunks = options.chunks;
			this.chunks = Object.create(shaderChunks);
			for (const chunkName in shaderChunks) {
				if (userChunks.hasOwnProperty(chunkName)) {
					const chunk = userChunks[chunkName];
					for (const a in builtinAttributes) {
						if (builtinAttributes.hasOwnProperty(a) && chunk.indexOf(a) >= 0) {
							this.attributes[a] = builtinAttributes[a];
						}
					}
					this.chunks[chunkName] = chunk;
				}
			}
		} else {
			this.chunks = shaderChunks;
		}
		this.shaderPassInfo = ShaderPass.get(this.device).getByIndex(options.pass);
		this.shadowPass = this.shaderPassInfo.isShadow;
		this.lighting = options.lights.length > 0 || options.dirLightMapEnabled || options.clusteredLightingEnabled;
		this.reflections = !!options.reflectionSource;
		this.needsNormal = this.lighting || this.reflections || options.useSpecular || options.ambientSH || options.useHeights || options.enableGGXSpecular || options.clusteredLightingEnabled && !this.shadowPass || options.useClearCoatNormals;
		this.needsNormal = this.needsNormal && !this.shadowPass;
		this.needsSceneColor = options.useDynamicRefraction;
		this.needsScreenSize = options.useDynamicRefraction;
		this.needsTransforms = options.useDynamicRefraction;
		this.varyings = "";
		this.varyingDefines = "";
		this.vshader = null;
		this.frontendDecl = null;
		this.frontendCode = null;
		this.frontendFunc = null;
		this.lightingUv = null;
		this.defines = [];
		this.fshader = null;
	}
	_vsAddBaseCode(code, chunks, options) {
		code += chunks.baseVS;
		if (options.nineSlicedMode === SPRITE_RENDERMODE_SLICED || options.nineSlicedMode === SPRITE_RENDERMODE_TILED) {
			code += chunks.baseNineSlicedVS;
		}
		return code;
	}
	_vsAddTransformCode(code, device, chunks, options) {
		code += this.chunks.transformVS;
		return code;
	}
	_setMapTransform(codes, name, id, uv) {
		const checkId = id + uv * 100;
		if (!codes[3][checkId]) {
			const varName = `texture_${name}MapTransform`;
			codes[0] += `uniform vec3 ${varName}0;\n`;
			codes[0] += `uniform vec3 ${varName}1;\n`;
			codes[1] += `varying vec2 vUV${uv}_${id};\n`;
			codes[2] += `   vUV${uv}_${id} = vec2(dot(vec3(uv${uv}, 1), ${varName}0), dot(vec3(uv${uv}, 1), ${varName}1));\n`;
			codes[3][checkId] = true;
		}
		return codes;
	}
	_fsGetBaseCode() {
		const options = this.options;
		const chunks = this.chunks;
		let result = this.chunks.basePS;
		if (options.nineSlicedMode === SPRITE_RENDERMODE_SLICED) {
			result += chunks.baseNineSlicedPS;
		} else if (options.nineSlicedMode === SPRITE_RENDERMODE_TILED) {
			result += chunks.baseNineSlicedTiledPS;
		}
		return result;
	}
	_fsGetStartCode(code, device, chunks, options) {
		let result = chunks.startPS;
		if (options.nineSlicedMode === SPRITE_RENDERMODE_SLICED) {
			result += chunks.startNineSlicedPS;
		} else if (options.nineSlicedMode === SPRITE_RENDERMODE_TILED) {
			result += chunks.startNineSlicedTiledPS;
		}
		return result;
	}
	_getLightSourceShapeString(shape) {
		switch (shape) {
			case LIGHTSHAPE_RECT:
				return 'Rect';
			case LIGHTSHAPE_DISK:
				return 'Disk';
			case LIGHTSHAPE_SPHERE:
				return 'Sphere';
			default:
				return '';
		}
	}
	generateVertexShader(useUv, useUnmodifiedUv, mapTransforms) {
		const device = this.device;
		const options = this.options;
		const chunks = this.chunks;
		let code = '';
		let codeBody = '';
		code = this._vsAddBaseCode(code, chunks, options);
		codeBody += "   vPositionW    = getWorldPosition();\n";
		if (this.options.pass === SHADER_DEPTH) {
			code += 'varying float vDepth;\n';
			code += '#ifndef VIEWMATRIX\n';
			code += '#define VIEWMATRIX\n';
			code += 'uniform mat4 matrix_view;\n';
			code += '#endif\n';
			code += '#ifndef CAMERAPLANES\n';
			code += '#define CAMERAPLANES\n';
			code += 'uniform vec4 camera_params;\n\n';
			code += '#endif\n';
			codeBody += "    vDepth = -(matrix_view * vec4(vPositionW,1.0)).z * camera_params.x;\n";
		}
		if (this.options.pass === SHADER_PREPASS_VELOCITY) ;
		if (this.options.useInstancing) {
			this.attributes.instance_line1 = SEMANTIC_ATTR12;
			this.attributes.instance_line2 = SEMANTIC_ATTR13;
			this.attributes.instance_line3 = SEMANTIC_ATTR14;
			this.attributes.instance_line4 = SEMANTIC_ATTR15;
			code += chunks.instancingVS;
		}
		if (this.needsNormal) {
			this.attributes.vertex_normal = SEMANTIC_NORMAL;
			codeBody += "   vNormalW = getNormal();\n";
			if (options.reflectionSource === 'sphereMap' && device.fragmentUniformsCount <= 16) {
				code += chunks.viewNormalVS;
				codeBody += "   vNormalV    = getViewNormal();\n";
			}
			if (options.hasTangents && (options.useHeights || options.useNormals || options.enableGGXSpecular)) {
				this.attributes.vertex_tangent = SEMANTIC_TANGENT;
				code += chunks.tangentBinormalVS;
				codeBody += "   vTangentW   = getTangent();\n";
				codeBody += "   vBinormalW  = getBinormal();\n";
			} else if (options.enableGGXSpecular || !device.extStandardDerivatives) {
				codeBody += "   vObjectSpaceUpW = normalize(dNormalMatrix * vec3(0, 1, 0));\n";
			}
		}
		const maxUvSets = 2;
		for (let i = 0; i < maxUvSets; i++) {
			if (useUv[i]) {
				this.attributes["vertex_texCoord" + i] = "TEXCOORD" + i;
				code += chunks["uv" + i + "VS"];
				codeBody += "   vec2 uv" + i + " = getUv" + i + "();\n";
			}
			if (useUnmodifiedUv[i]) {
				codeBody += "   vUv" + i + " = uv" + i + ";\n";
			}
		}
		const codes = [code, this.varyings, codeBody, []];
		mapTransforms.forEach(mapTransform => {
			this._setMapTransform(codes, mapTransform.name, mapTransform.id, mapTransform.uv);
		});
		code = codes[0];
		this.varyings = codes[1];
		codeBody = codes[2];
		if (options.vertexColors) {
			this.attributes.vertex_color = SEMANTIC_COLOR;
			codeBody += "   vVertexColor = vertex_color;\n";
		}
		if (options.useMsdf && options.msdfTextAttribute) {
			this.attributes.vertex_outlineParameters = SEMANTIC_ATTR8;
			this.attributes.vertex_shadowParameters = SEMANTIC_ATTR9;
			codeBody += "    unpackMsdfParams();\n";
			code += chunks.msdfVS;
		}
		if (options.useMorphPosition || options.useMorphNormal) {
			if (options.useMorphTextureBased) {
				code += "#define MORPHING_TEXTURE_BASED\n";
				if (options.useMorphPosition) {
					code += "#define MORPHING_TEXTURE_BASED_POSITION\n";
				}
				if (options.useMorphNormal) {
					code += "#define MORPHING_TEXTURE_BASED_NORMAL\n";
				}
				this.attributes.morph_vertex_id = SEMANTIC_ATTR15;
				const morphIdType = device.isWebGPU ? 'uint' : 'float';
				code += `attribute ${morphIdType} morph_vertex_id;\n`;
			} else {
				code += "#define MORPHING\n";
				if (options.useMorphPosition) {
					this.attributes.morph_pos0 = SEMANTIC_ATTR8;
					this.attributes.morph_pos1 = SEMANTIC_ATTR9;
					this.attributes.morph_pos2 = SEMANTIC_ATTR10;
					this.attributes.morph_pos3 = SEMANTIC_ATTR11;
					code += "#define MORPHING_POS03\n";
					code += "attribute vec3 morph_pos0;\n";
					code += "attribute vec3 morph_pos1;\n";
					code += "attribute vec3 morph_pos2;\n";
					code += "attribute vec3 morph_pos3;\n";
				} else if (options.useMorphNormal) {
					this.attributes.morph_nrm0 = SEMANTIC_ATTR8;
					this.attributes.morph_nrm1 = SEMANTIC_ATTR9;
					this.attributes.morph_nrm2 = SEMANTIC_ATTR10;
					this.attributes.morph_nrm3 = SEMANTIC_ATTR11;
					code += "#define MORPHING_NRM03\n";
					code += "attribute vec3 morph_nrm0;\n";
					code += "attribute vec3 morph_nrm1;\n";
					code += "attribute vec3 morph_nrm2;\n";
					code += "attribute vec3 morph_nrm3;\n";
				}
				if (!options.useMorphNormal) {
					this.attributes.morph_pos4 = SEMANTIC_ATTR12;
					this.attributes.morph_pos5 = SEMANTIC_ATTR13;
					this.attributes.morph_pos6 = SEMANTIC_ATTR14;
					this.attributes.morph_pos7 = SEMANTIC_ATTR15;
					code += "#define MORPHING_POS47\n";
					code += "attribute vec3 morph_pos4;\n";
					code += "attribute vec3 morph_pos5;\n";
					code += "attribute vec3 morph_pos6;\n";
					code += "attribute vec3 morph_pos7;\n";
				} else {
					this.attributes.morph_nrm4 = SEMANTIC_ATTR12;
					this.attributes.morph_nrm5 = SEMANTIC_ATTR13;
					this.attributes.morph_nrm6 = SEMANTIC_ATTR14;
					this.attributes.morph_nrm7 = SEMANTIC_ATTR15;
					code += "#define MORPHING_NRM47\n";
					code += "attribute vec3 morph_nrm4;\n";
					code += "attribute vec3 morph_nrm5;\n";
					code += "attribute vec3 morph_nrm6;\n";
					code += "attribute vec3 morph_nrm7;\n";
				}
			}
		}
		if (options.skin) {
			this.attributes.vertex_boneWeights = SEMANTIC_BLENDWEIGHT;
			this.attributes.vertex_boneIndices = SEMANTIC_BLENDINDICES;
			code += ShaderGenerator.skinCode(device, chunks);
			code += "#define SKIN\n";
		} else if (options.useInstancing) {
			code += "#define INSTANCING\n";
		}
		if (options.screenSpace) {
			code += "#define SCREENSPACE\n";
		}
		if (options.pixelSnap) {
			code += "#define PIXELSNAP\n";
		}
		code = this._vsAddTransformCode(code, device, chunks, options);
		if (this.needsNormal) {
			code += chunks.normalVS;
		}
		code += "\n";
		code += chunks.startVS;
		code += codeBody;
		code += chunks.endVS;
		code += "}";
		Object.keys(builtinVaryings).forEach(v => {
			if (code.indexOf(v) >= 0) {
				this.varyings += `varying ${builtinVaryings[v]} ${v};\n`;
				this.varyingDefines += `#define VARYING_${v.toUpperCase()}\n`;
			}
		});
		const shaderPassDefines = this.shaderPassInfo.shaderDefines;
		this.vshader = shaderPassDefines + this.varyings + code;
	}
	_fsGetBeginCode() {
		let code = this.shaderPassInfo.shaderDefines;
		for (let i = 0; i < this.defines.length; i++) {
			code += `#define ${this.defines[i]}\n`;
		}
		return code;
	}
	_fsGetPickPassCode() {
		let code = this._fsGetBeginCode();
		code += "uniform vec4 uColor;\n";
		code += this.varyings;
		code += this.varyingDefines;
		code += this.frontendDecl;
		code += this.frontendCode;
		code += ShaderGenerator.begin();
		code += this.frontendFunc;
		code += "    gl_FragColor = uColor;\n";
		code += ShaderGenerator.end();
		return code;
	}
	_fsGetDepthPassCode() {
		const chunks = this.chunks;
		let code = this._fsGetBeginCode();
		code += 'varying float vDepth;\n';
		code += this.varyings;
		code += this.varyingDefines;
		code += chunks.packDepthPS;
		code += this.frontendDecl;
		code += this.frontendCode;
		code += ShaderGenerator.begin();
		code += this.frontendFunc;
		code += "    gl_FragColor = packFloat(vDepth);\n";
		code += ShaderGenerator.end();
		return code;
	}
	_fsGetPrePassVelocityCode() {
		const code = `
						void main(void)
						{
								gl_FragColor = vec4(1, 0, 0, 1);
						}
						`;
		return code;
	}
	_fsGetShadowPassCode() {
		const device = this.device;
		const options = this.options;
		const chunks = this.chunks;
		const varyings = this.varyings;
		const lightType = this.shaderPassInfo.lightType;
		let shadowType = this.shaderPassInfo.shadowType;
		if (lightType !== LIGHTTYPE_DIRECTIONAL && options.clusteredLightingEnabled) {
			if (shadowType === SHADOW_VSM8 || shadowType === SHADOW_VSM16 || shadowType === SHADOW_VSM32 || shadowType === SHADOW_PCSS) {
				shadowType = SHADOW_PCF3;
			}
		}
		let code = this._fsGetBeginCode();
		if (device.extStandardDerivatives && device.isWebGL1) {
			code += 'uniform vec2 polygonOffset;\n';
		}
		if (shadowType === SHADOW_VSM32) {
			if (device.textureFloatHighPrecision) {
				code += '#define VSM_EXPONENT 15.0\n\n';
			} else {
				code += '#define VSM_EXPONENT 5.54\n\n';
			}
		} else if (shadowType === SHADOW_VSM16) {
			code += '#define VSM_EXPONENT 5.54\n\n';
		}
		if (lightType !== LIGHTTYPE_DIRECTIONAL) {
			code += 'uniform vec3 view_position;\n';
			code += 'uniform float light_radius;\n';
		}
		code += varyings;
		code += this.varyingDefines;
		code += this.frontendDecl;
		code += this.frontendCode;
		const mayPackDepth = shadowType === SHADOW_PCF1 || shadowType === SHADOW_PCF3 || shadowType === SHADOW_PCF5 || shadowType === SHADOW_PCSS;
		const mustPackDepth = lightType === LIGHTTYPE_OMNI && shadowType !== SHADOW_PCSS && !options.clusteredLightingEnabled;
		const usePackedDepth = mayPackDepth && !device.supportsDepthShadow || mustPackDepth;
		if (usePackedDepth) {
			code += chunks.packDepthPS;
		} else if (shadowType === SHADOW_VSM8) {
			code += "vec2 encodeFloatRG( float v ) {\n";
			code += "    vec2 enc = vec2(1.0, 255.0) * v;\n";
			code += "    enc = fract(enc);\n";
			code += "    enc -= enc.yy * vec2(1.0/255.0, 1.0/255.0);\n";
			code += "    return enc;\n";
			code += "}\n\n";
		}
		if (shadowType === SHADOW_PCSS) {
			code += shaderChunks.linearizeDepthPS;
		}
		code += ShaderGenerator.begin();
		code += this.frontendFunc;
		const isVsm = shadowType === SHADOW_VSM8 || shadowType === SHADOW_VSM16 || shadowType === SHADOW_VSM32;
		const applySlopeScaleBias = device.isWebGL1 && device.extStandardDerivatives;
		const usePerspectiveDepth = lightType === LIGHTTYPE_DIRECTIONAL || !isVsm && lightType === LIGHTTYPE_SPOT;
		let hasModifiedDepth = false;
		if (usePerspectiveDepth) {
			code += "    float depth = gl_FragCoord.z;\n";
		} else {
			code += "    float depth = min(distance(view_position, vPositionW) / light_radius, 0.99999);\n";
			hasModifiedDepth = true;
		}
		if (applySlopeScaleBias) {
			code += "    float minValue = 2.3374370500153186e-10; //(1.0 / 255.0) / (256.0 * 256.0 * 256.0);\n";
			code += "    depth += polygonOffset.x * max(abs(dFdx(depth)), abs(dFdy(depth))) + minValue * polygonOffset.y;\n";
			hasModifiedDepth = true;
		}
		if (usePackedDepth) {
			code += "    gl_FragColor = packFloat(depth);\n";
		} else if (!isVsm) {
			const exportR32 = shadowType === SHADOW_PCSS;
			if (exportR32) {
				code += "    gl_FragColor.r = depth;\n";
			} else {
				if (hasModifiedDepth) {
					code += "    gl_FragDepth = depth;\n";
				}
				code += "    gl_FragColor = vec4(1.0);\n";
			}
		} else if (shadowType === SHADOW_VSM8) {
			code += "    gl_FragColor = vec4(encodeFloatRG(depth), encodeFloatRG(depth*depth));\n";
		} else {
			code += chunks.storeEVSMPS;
		}
		code += ShaderGenerator.end();
		return code;
	}
	_fsGetLitPassCode() {
		const device = this.device;
		const options = this.options;
		const chunks = this.chunks;
		const decl = new ChunkBuilder();
		const func = new ChunkBuilder();
		const backend = new ChunkBuilder();
		const code = new ChunkBuilder();
		if (options.opacityFadesSpecular === false) {
			decl.append('uniform float material_alphaFade;');
		}
		if (options.useSpecular) {
			this.defines.push("LIT_SPECULAR");
			if (this.reflections) {
				this.defines.push("LIT_REFLECTIONS");
			}
			if (options.useClearCoat) {
				this.defines.push("LIT_CLEARCOAT");
			}
			if (options.fresnelModel > 0) {
				this.defines.push("LIT_SPECULAR_FRESNEL");
			}
			if (options.conserveEnergy) {
				this.defines.push("LIT_CONSERVE_ENERGY");
			}
			if (options.useSheen) {
				this.defines.push("LIT_SHEEN");
			}
			if (options.useIridescence) {
				this.defines.push("LIT_IRIDESCENCE");
			}
		}
		const shadowTypeUsed = [];
		let numShadowLights = 0;
		let shadowedDirectionalLightUsed = false;
		let useVsm = false;
		let usePcss = false;
		let hasAreaLights = options.lights.some(function (light) {
			return light._shape && light._shape !== LIGHTSHAPE_PUNCTUAL;
		});
		if (options.clusteredLightingEnabled && options.clusteredLightingAreaLightsEnabled) {
			hasAreaLights = true;
		}
		let areaLutsPrecision = 'highp';
		if (device.areaLightLutFormat === PIXELFORMAT_RGBA8) {
			decl.append("#define AREA_R8_G8_B8_A8_LUTS");
			areaLutsPrecision = 'lowp';
		}
		if (hasAreaLights || options.clusteredLightingEnabled) {
			decl.append("#define AREA_LIGHTS");
			decl.append(`uniform ${areaLutsPrecision} sampler2D areaLightsLutTex1;`);
			decl.append(`uniform ${areaLutsPrecision} sampler2D areaLightsLutTex2;`);
		}
		for (let i = 0; i < options.lights.length; i++) {
			const light = options.lights[i];
			const lightType = light._type;
			if (options.clusteredLightingEnabled && lightType !== LIGHTTYPE_DIRECTIONAL) continue;
			const lightShape = hasAreaLights && light._shape ? light._shape : LIGHTSHAPE_PUNCTUAL;
			decl.append("uniform vec3 light" + i + "_color;");
			if (light._shadowType === SHADOW_PCSS && light.castShadows && !options.noShadow) {
				decl.append(`uniform float light${i}_shadowSearchArea;`);
				decl.append(`uniform vec4 light${i}_cameraParams;`);
			}
			if (lightType === LIGHTTYPE_DIRECTIONAL) {
				decl.append("uniform vec3 light" + i + "_direction;");
			} else {
				decl.append("uniform vec3 light" + i + "_position;");
				decl.append("uniform float light" + i + "_radius;");
				if (lightType === LIGHTTYPE_SPOT) {
					decl.append("uniform vec3 light" + i + "_direction;");
					decl.append("uniform float light" + i + "_innerConeAngle;");
					decl.append("uniform float light" + i + "_outerConeAngle;");
				}
			}
			if (lightShape !== LIGHTSHAPE_PUNCTUAL) {
				if (lightType === LIGHTTYPE_DIRECTIONAL) {
					decl.append("uniform vec3 light" + i + "_position;");
				}
				decl.append("uniform vec3 light" + i + "_halfWidth;");
				decl.append("uniform vec3 light" + i + "_halfHeight;");
			}
			if (light.castShadows && !options.noShadow) {
				decl.append("uniform mat4 light" + i + "_shadowMatrix;");
				decl.append("uniform float light" + i + "_shadowIntensity;");
				if (lightType === LIGHTTYPE_DIRECTIONAL) {
					decl.append("uniform mat4 light" + i + "_shadowMatrixPalette[4];");
					decl.append("uniform float light" + i + "_shadowCascadeDistances[4];");
					decl.append("uniform float light" + i + "_shadowCascadeCount;");
				}
				decl.append("uniform vec4 light" + i + "_shadowParams;");
				if (lightType === LIGHTTYPE_DIRECTIONAL) {
					shadowedDirectionalLightUsed = true;
				}
				if (lightType === LIGHTTYPE_OMNI) {
					decl.append("uniform samplerCube light" + i + "_shadowMap;");
				} else {
					if (light._isPcf && device.supportsDepthShadow) {
						decl.append("uniform sampler2DShadow light" + i + "_shadowMap;");
					} else {
						decl.append("uniform sampler2D light" + i + "_shadowMap;");
					}
				}
				numShadowLights++;
				shadowTypeUsed[light._shadowType] = true;
				if (light._isVsm) useVsm = true;
				if (light._shadowType === SHADOW_PCSS) usePcss = true;
			}
			if (light._cookie) {
				if (light._cookie._cubemap) {
					if (lightType === LIGHTTYPE_OMNI) {
						decl.append("uniform samplerCube light" + i + "_cookie;");
						decl.append("uniform float light" + i + "_cookieIntensity;");
						if (!light.castShadows || options.noShadow) {
							decl.append("uniform mat4 light" + i + "_shadowMatrix;");
						}
					}
				} else {
					if (lightType === LIGHTTYPE_SPOT) {
						decl.append("uniform sampler2D light" + i + "_cookie;");
						decl.append("uniform float light" + i + "_cookieIntensity;");
						if (!light.castShadows || options.noShadow) {
							decl.append("uniform mat4 light" + i + "_shadowMatrix;");
						}
						if (light._cookieTransform) {
							decl.append("uniform vec4 light" + i + "_cookieMatrix;");
							decl.append("uniform vec2 light" + i + "_cookieOffset;");
						}
					}
				}
			}
		}
		const hasTBN = this.needsNormal && (options.useNormals || options.useClearCoatNormals || options.enableGGXSpecular && !options.useHeights);
		if (hasTBN) {
			if (options.hasTangents) {
				func.append(options.fastTbn ? chunks.TBNfastPS : chunks.TBNPS);
			} else {
				if (device.extStandardDerivatives && (options.useNormals || options.useClearCoatNormals)) {
					func.append(chunks.TBNderivativePS.replace(/\$UV/g, this.lightingUv));
				} else {
					func.append(chunks.TBNObjectSpacePS);
				}
			}
		}
		func.append(chunks.sphericalPS);
		func.append(chunks.decodePS);
		func.append(ShaderGenerator.gammaCode(options.gamma, chunks));
		func.append(ShaderGenerator.tonemapCode(options.toneMap, chunks));
		func.append(ShaderGenerator.fogCode(options.fog, chunks));
		func.append(this.frontendCode);
		if (options.useCubeMapRotation) {
			decl.append("#define CUBEMAP_ROTATION");
		}
		if (this.needsNormal) {
			func.append(chunks.cubeMapRotatePS);
			func.append(options.cubeMapProjection > 0 ? chunks.cubeMapProjectBoxPS : chunks.cubeMapProjectNonePS);
			func.append(options.skyboxIntensity ? chunks.envMultiplyPS : chunks.envConstPS);
		}
		if (this.lighting && options.useSpecular || this.reflections) {
			if (options.useMetalness) {
				func.append(chunks.metalnessModulatePS);
			}
			if (options.fresnelModel === FRESNEL_SCHLICK) {
				func.append(chunks.fresnelSchlickPS);
			}
			if (options.useIridescence) {
				func.append(chunks.iridescenceDiffractionPS);
			}
		}
		if (options.useAo) {
			func.append(chunks.aoDiffuseOccPS);
			switch (options.occludeSpecular) {
				case SPECOCC_AO:
					func.append(options.occludeSpecularFloat ? chunks.aoSpecOccSimplePS : chunks.aoSpecOccConstSimplePS);
					break;
				case SPECOCC_GLOSSDEPENDENT:
					func.append(options.occludeSpecularFloat ? chunks.aoSpecOccPS : chunks.aoSpecOccConstPS);
					break;
			}
		}
		if (options.reflectionSource === 'envAtlasHQ') {
			func.append(options.fixSeams ? chunks.fixCubemapSeamsStretchPS : chunks.fixCubemapSeamsNonePS);
			func.append(chunks.envAtlasPS);
			func.append(chunks.reflectionEnvHQPS.replace(/\$DECODE_CUBEMAP/g, ChunkUtils.decodeFunc(options.reflectionCubemapEncoding)).replace(/\$DECODE/g, ChunkUtils.decodeFunc(options.reflectionEncoding)));
		} else if (options.reflectionSource === 'envAtlas') {
			func.append(chunks.envAtlasPS);
			func.append(chunks.reflectionEnvPS.replace(/\$DECODE/g, ChunkUtils.decodeFunc(options.reflectionEncoding)));
		} else if (options.reflectionSource === 'cubeMap') {
			func.append(options.fixSeams ? chunks.fixCubemapSeamsStretchPS : chunks.fixCubemapSeamsNonePS);
			func.append(chunks.reflectionCubePS.replace(/\$DECODE/g, ChunkUtils.decodeFunc(options.reflectionEncoding)));
		} else if (options.reflectionSource === 'sphereMap') {
			func.append(chunks.reflectionSpherePS.replace(/\$DECODE/g, ChunkUtils.decodeFunc(options.reflectionEncoding)));
		}
		if (this.reflections) {
			if (options.useClearCoat) {
				func.append(chunks.reflectionCCPS);
			}
			if (options.useSheen) {
				func.append(chunks.reflectionSheenPS);
			}
		}
		if (options.useRefraction) {
			if (options.useDynamicRefraction) {
				func.append(chunks.refractionDynamicPS);
			} else if (this.reflections) {
				func.append(chunks.refractionCubePS);
			}
		}
		if (options.useSheen) {
			func.append(chunks.lightSheenPS);
		}
		if (options.clusteredLightingEnabled) {
			func.append(chunks.clusteredLightUtilsPS);
			if (options.clusteredLightingCookiesEnabled) func.append(chunks.clusteredLightCookiesPS);
			if (options.clusteredLightingShadowsEnabled && !options.noShadow) {
				shadowTypeUsed[SHADOW_PCF3] = true;
				shadowTypeUsed[SHADOW_PCF5] = true;
				shadowTypeUsed[SHADOW_PCSS] = true;
			}
		}
		if (numShadowLights > 0 || options.clusteredLightingEnabled) {
			if (shadowedDirectionalLightUsed) {
				func.append(chunks.shadowCascadesPS);
			}
			if (shadowTypeUsed[SHADOW_PCF1] || shadowTypeUsed[SHADOW_PCF3]) {
				func.append(chunks.shadowStandardPS);
			}
			if (shadowTypeUsed[SHADOW_PCF5] && !device.isWebGL1) {
				func.append(chunks.shadowStandardGL2PS);
			}
			if (useVsm) {
				func.append(chunks.shadowVSM_commonPS);
				if (shadowTypeUsed[SHADOW_VSM8]) {
					func.append(chunks.shadowVSM8PS);
				}
				if (shadowTypeUsed[SHADOW_VSM16]) {
					func.append(device.extTextureHalfFloatLinear ? chunks.shadowEVSMPS.replace(/\$/g, "16") : chunks.shadowEVSMnPS.replace(/\$/g, "16"));
				}
				if (shadowTypeUsed[SHADOW_VSM32]) {
					func.append(device.extTextureFloatLinear ? chunks.shadowEVSMPS.replace(/\$/g, "32") : chunks.shadowEVSMnPS.replace(/\$/g, "32"));
				}
			}
			if (usePcss) {
				func.append(chunks.linearizeDepthPS);
				func.append(chunks.shadowPCSSPS);
			}
			if (!(device.isWebGL2 || device.isWebGPU || device.extStandardDerivatives)) {
				func.append(chunks.biasConstPS);
			}
		}
		if (options.enableGGXSpecular) func.append("uniform float material_anisotropy;");
		if (this.lighting) {
			func.append(chunks.lightDiffuseLambertPS);
			if (hasAreaLights || options.clusteredLightingAreaLightsEnabled) {
				func.append(chunks.ltcPS);
			}
		}
		let useOldAmbient = false;
		if (options.useSpecular) {
			if (this.lighting) {
				func.append(options.shadingModel === SPECULAR_PHONG ? chunks.lightSpecularPhongPS : options.enableGGXSpecular ? chunks.lightSpecularAnisoGGXPS : chunks.lightSpecularBlinnPS);
			}
			if (!options.fresnelModel && !this.reflections && !options.diffuseMapEnabled) {
				decl.append("uniform vec3 material_ambient;");
				decl.append("#define LIT_OLD_AMBIENT");
				useOldAmbient = true;
			}
		}
		func.append(chunks.combinePS);
		if (options.lightMapEnabled) {
			func.append(options.useSpecular && options.dirLightMapEnabled ? chunks.lightmapDirAddPS : chunks.lightmapAddPS);
		}
		const addAmbient = !options.lightMapEnabled || options.lightMapWithoutAmbient;
		if (addAmbient) {
			if (options.ambientSource === 'ambientSH') {
				func.append(chunks.ambientSHPS);
			} else if (options.ambientSource === 'envAtlas') {
				if (options.reflectionSource !== 'envAtlas' && options.reflectionSource !== 'envAtlasHQ') {
					func.append(chunks.envAtlasPS);
				}
				func.append(chunks.ambientEnvPS.replace(/\$DECODE/g, ChunkUtils.decodeFunc(options.ambientEncoding)));
			} else {
				func.append(chunks.ambientConstantPS);
			}
		}
		if (options.useAmbientTint && !useOldAmbient) {
			decl.append("uniform vec3 material_ambient;");
		}
		if (options.useMsdf) {
			if (!options.msdfTextAttribute) {
				decl.append("#define UNIFORM_TEXT_PARAMETERS");
			}
			func.append(chunks.msdfPS);
		}
		if (this.needsNormal) {
			func.append(chunks.viewDirPS);
			if (options.useSpecular) {
				func.append(options.enableGGXSpecular ? chunks.reflDirAnisoPS : chunks.reflDirPS);
			}
		}
		let hasPointLights = false;
		let usesLinearFalloff = false;
		let usesInvSquaredFalloff = false;
		let usesSpot = false;
		let usesCookie = false;
		let usesCookieNow;
		if (options.clusteredLightingEnabled && this.lighting) {
			usesSpot = true;
			hasPointLights = true;
			usesLinearFalloff = true;
			usesCookie = true;
			func.append(chunks.floatUnpackingPS);
			if (options.lightMaskDynamic) decl.append("#define CLUSTER_MESH_DYNAMIC_LIGHTS");
			if (options.clusteredLightingCookiesEnabled) decl.append("#define CLUSTER_COOKIES");
			if (options.clusteredLightingShadowsEnabled && !options.noShadow) {
				decl.append("#define CLUSTER_SHADOWS");
				decl.append("#define CLUSTER_SHADOW_TYPE_" + shadowTypeToString[options.clusteredLightingShadowType]);
			}
			if (options.clusteredLightingAreaLightsEnabled) decl.append("#define CLUSTER_AREALIGHTS");
			decl.append(LightsBuffer.getShaderDefines(device));
			if (options.clusteredLightingShadowsEnabled && !options.noShadow) {
				func.append(chunks.clusteredLightShadowsPS);
			}
			func.append(chunks.clusteredLightPS);
		}
		if (options.twoSidedLighting) {
			decl.append("uniform float twoSidedLightingNegScaleFactor;");
		}
		code.append(this._fsGetStartCode(code, device, chunks, options));
		if (this.needsNormal) {
			if (options.twoSidedLighting) {
				code.append("    dVertexNormalW = normalize(gl_FrontFacing ? vNormalW * twoSidedLightingNegScaleFactor : -vNormalW * twoSidedLightingNegScaleFactor);");
			} else {
				code.append("    dVertexNormalW = normalize(vNormalW);");
			}
			if ((options.useHeights || options.useNormals) && options.hasTangents) {
				if (options.twoSidedLighting) {
					code.append("    dTangentW = gl_FrontFacing ? vTangentW * twoSidedLightingNegScaleFactor : -vTangentW * twoSidedLightingNegScaleFactor;");
					code.append("    dBinormalW = gl_FrontFacing ? vBinormalW * twoSidedLightingNegScaleFactor : -vBinormalW * twoSidedLightingNegScaleFactor;");
				} else {
					code.append("    dTangentW = vTangentW;");
					code.append("    dBinormalW = vBinormalW;");
				}
			}
			code.append("    getViewDir();");
			if (hasTBN) {
				code.append("    getTBN(dTangentW, dBinormalW, dVertexNormalW);");
			}
		}
		code.append(this.frontendFunc);
		if (this.needsNormal) {
			if (options.useSpecular) {
				backend.append("    getReflDir(litArgs_worldNormal, dViewDirW, litArgs_gloss, dTBN);");
			}
			if (options.useClearCoat) {
				backend.append("    ccReflDirW = normalize(-reflect(dViewDirW, litArgs_clearcoat_worldNormal));");
			}
		}
		if (this.lighting && options.useSpecular || this.reflections) {
			if (options.useMetalness) {
				backend.append("    float f0 = 1.0 / litArgs_ior; f0 = (f0 - 1.0) / (f0 + 1.0); f0 *= f0;");
				backend.append("    litArgs_specularity = getSpecularModulate(litArgs_specularity, litArgs_albedo, litArgs_metalness, f0);");
				backend.append("    litArgs_albedo = getAlbedoModulate(litArgs_albedo, litArgs_metalness);");
			}
			if (options.useIridescence) {
				backend.append("    vec3 iridescenceFresnel = getIridescence(saturate(dot(dViewDirW, litArgs_worldNormal)), litArgs_specularity, litArgs_iridescence_thickness);");
			}
		}
		if (addAmbient) {
			backend.append("    addAmbient(litArgs_worldNormal);");
			if (options.conserveEnergy && options.useSpecular) {
				backend.append(`   dDiffuseLight = dDiffuseLight * (1.0 - litArgs_specularity);`);
			}
			if (options.separateAmbient) {
				backend.append(`
										vec3 dAmbientLight = dDiffuseLight;
										dDiffuseLight = vec3(0);
								`);
			}
		}
		if (options.useAmbientTint && !useOldAmbient) {
			backend.append("    dDiffuseLight *= material_ambient;");
		}
		if (options.useAo && !options.occludeDirect) {
			backend.append("    occludeDiffuse(litArgs_ao);");
		}
		if (options.lightMapEnabled) {
			backend.append(`    addLightMap(
								litArgs_lightmap, 
								litArgs_lightmapDir, 
								litArgs_worldNormal, 
								dViewDirW, 
								dReflDirW, 
								litArgs_gloss, 
								litArgs_specularity, 
								dVertexNormalW,
								dTBN
						#if defined(LIT_IRIDESCENCE)
								, iridescenceFresnel,
								litArgs_iridescence_intensity
						#endif
								);`);
		}
		if (this.lighting || this.reflections) {
			if (this.reflections) {
				if (options.useClearCoat) {
					backend.append("    addReflectionCC(ccReflDirW, litArgs_clearcoat_gloss);");
					if (options.fresnelModel > 0) {
						backend.append("    ccFresnel = getFresnelCC(dot(dViewDirW, litArgs_clearcoat_worldNormal));");
						backend.append("    ccReflection.rgb *= ccFresnel;");
					} else {
						backend.append("    ccFresnel = 0.0;");
					}
				}
				if (options.useSpecularityFactor) {
					backend.append("    ccReflection.rgb *= litArgs_specularityFactor;");
				}
				if (options.useSheen) {
					backend.append("    addReflectionSheen(litArgs_worldNormal, dViewDirW, litArgs_sheen_gloss);");
				}
				backend.append("    addReflection(dReflDirW, litArgs_gloss);");
				if (options.fresnelModel > 0) {
					backend.append(`    dReflection.rgb *= 
												getFresnel(
														dot(dViewDirW, litArgs_worldNormal), 
														litArgs_gloss, 
														litArgs_specularity
												#if defined(LIT_IRIDESCENCE)
														, iridescenceFresnel,
														litArgs_iridescence_intensity
												#endif
														);`);
				} else {
					backend.append("    dReflection.rgb *= litArgs_specularity;");
				}
				if (options.useSpecularityFactor) {
					backend.append("    dReflection.rgb *= litArgs_specularityFactor;");
				}
			}
			if (hasAreaLights) {
				backend.append("    dSpecularLight *= litArgs_specularity;");
				if (options.useSpecular) {
					backend.append("    calcLTCLightValues(litArgs_gloss, litArgs_worldNormal, dViewDirW, litArgs_specularity, litArgs_clearcoat_gloss, litArgs_clearcoat_worldNormal, litArgs_clearcoat_specularity);");
				}
			}
			for (let i = 0; i < options.lights.length; i++) {
				const light = options.lights[i];
				const lightType = light._type;
				if (options.clusteredLightingEnabled && lightType !== LIGHTTYPE_DIRECTIONAL) {
					continue;
				}
				usesCookieNow = false;
				const lightShape = hasAreaLights && light._shape ? light.shape : LIGHTSHAPE_PUNCTUAL;
				const shapeString = hasAreaLights && light._shape ? this._getLightSourceShapeString(lightShape) : '';
				if (lightShape !== LIGHTSHAPE_PUNCTUAL) {
					backend.append("    calc" + shapeString + "LightValues(light" + i + "_position, light" + i + "_halfWidth, light" + i + "_halfHeight);");
				}
				if (lightType === LIGHTTYPE_DIRECTIONAL) {
					backend.append("    dLightDirNormW = light" + i + "_direction;");
					backend.append("    dAtten = 1.0;");
				} else {
					if (light._cookie) {
						if (lightType === LIGHTTYPE_SPOT && !light._cookie._cubemap) {
							usesCookie = true;
							usesCookieNow = true;
						} else if (lightType === LIGHTTYPE_OMNI && light._cookie._cubemap) {
							usesCookie = true;
							usesCookieNow = true;
						}
					}
					backend.append("    getLightDirPoint(light" + i + "_position);");
					hasPointLights = true;
					if (usesCookieNow) {
						if (lightType === LIGHTTYPE_SPOT) {
							backend.append("    dAtten3 = getCookie2D" + (light._cookieFalloff ? "" : "Clip") + (light._cookieTransform ? "Xform" : "") + "(light" + i + "_cookie, light" + i + "_shadowMatrix, light" + i + "_cookieIntensity" + (light._cookieTransform ? ", light" + i + "_cookieMatrix, light" + i + "_cookieOffset" : "") + ")." + light._cookieChannel + ";");
						} else {
							backend.append("    dAtten3 = getCookieCube(light" + i + "_cookie, light" + i + "_shadowMatrix, light" + i + "_cookieIntensity)." + light._cookieChannel + ";");
						}
					}
					if (lightShape === LIGHTSHAPE_PUNCTUAL) {
						if (light._falloffMode === LIGHTFALLOFF_LINEAR) {
							backend.append("    dAtten = getFalloffLinear(light" + i + "_radius, dLightDirW);");
							usesLinearFalloff = true;
						} else {
							backend.append("    dAtten = getFalloffInvSquared(light" + i + "_radius, dLightDirW);");
							usesInvSquaredFalloff = true;
						}
					} else {
						backend.append("    dAtten = getFalloffWindow(light" + i + "_radius, dLightDirW);");
						usesInvSquaredFalloff = true;
					}
					backend.append("    if (dAtten > 0.00001) {");
					if (lightType === LIGHTTYPE_SPOT) {
						if (!(usesCookieNow && !light._cookieFalloff)) {
							backend.append("    dAtten *= getSpotEffect(light" + i + "_direction, light" + i + "_innerConeAngle, light" + i + "_outerConeAngle, dLightDirNormW);");
							usesSpot = true;
						}
					}
				}
				if (lightShape !== LIGHTSHAPE_PUNCTUAL) {
					if (lightType === LIGHTTYPE_DIRECTIONAL) {
						backend.append("    dAttenD = getLightDiffuse(litArgs_worldNormal, dViewDirW, dLightDirW, dLightDirNormW);");
					} else {
						backend.append("    dAttenD = get" + shapeString + "LightDiffuse(litArgs_worldNormal, dViewDirW, dLightDirW, dLightDirNormW) * 16.0;");
					}
				} else {
					backend.append("    dAtten *= getLightDiffuse(litArgs_worldNormal, dViewDirW, dLightDirW, dLightDirNormW);");
				}
				if (light.castShadows && !options.noShadow) {
					const pcssShadows = light._shadowType === SHADOW_PCSS;
					const vsmShadows = light._shadowType === SHADOW_VSM8 || light._shadowType === SHADOW_VSM16 || light._shadowType === SHADOW_VSM32;
					const pcfShadows = light._shadowType === SHADOW_PCF1 || light._shadowType === SHADOW_PCF3 || light._shadowType === SHADOW_PCF5;
					let shadowReadMode = null;
					let evsmExp;
					switch (light._shadowType) {
						case SHADOW_VSM8:
							shadowReadMode = "VSM8";
							evsmExp = "0.0";
							break;
						case SHADOW_VSM16:
							shadowReadMode = "VSM16";
							evsmExp = "5.54";
							break;
						case SHADOW_VSM32:
							shadowReadMode = "VSM32";
							if (device.textureFloatHighPrecision) {
								evsmExp = "15.0";
							} else {
								evsmExp = "5.54";
							}
							break;
						case SHADOW_PCF1:
							shadowReadMode = "PCF1x1";
							break;
						case SHADOW_PCF5:
							shadowReadMode = "PCF5x5";
							break;
						case SHADOW_PCSS:
							shadowReadMode = "PCSS";
							break;
						case SHADOW_PCF3:
						default:
							shadowReadMode = "PCF3x3";
							break;
					}
					if (shadowReadMode !== null) {
						if (light._normalOffsetBias && !light._isVsm) {
							func.append("#define SHADOW_SAMPLE_NORMAL_OFFSET");
						}
						if (lightType === LIGHTTYPE_DIRECTIONAL) {
							func.append("#define SHADOW_SAMPLE_ORTHO");
						}
						if ((pcfShadows || pcssShadows) && device.isWebGL2 || device.isWebGPU || device.extStandardDerivatives) {
							func.append("#define SHADOW_SAMPLE_SOURCE_ZBUFFER");
						}
						if (lightType === LIGHTTYPE_OMNI) {
							func.append("#define SHADOW_SAMPLE_POINT");
						}
						const coordCode = chunks.shadowSampleCoordPS;
						func.append(coordCode.replace("$LIGHT", i));
						func.append("#undef SHADOW_SAMPLE_NORMAL_OFFSET");
						func.append("#undef SHADOW_SAMPLE_ORTHO");
						func.append("#undef SHADOW_SAMPLE_SOURCE_ZBUFFER");
						func.append("#undef SHADOW_SAMPLE_POINT");
						let shadowMatrix = `light${i}_shadowMatrix`;
						if (lightType === LIGHTTYPE_DIRECTIONAL && light.numCascades > 1) {
							backend.append(`    getShadowCascadeMatrix(light${i}_shadowMatrixPalette, light${i}_shadowCascadeDistances, light${i}_shadowCascadeCount);`);
							shadowMatrix = `cascadeShadowMat`;
						}
						backend.append(`    dShadowCoord = getShadowSampleCoord${i}(${shadowMatrix}, light${i}_shadowParams, vPositionW, dLightPosW, dLightDirW, dLightDirNormW, dVertexNormalW);`);
						if (lightType === LIGHTTYPE_DIRECTIONAL) {
							backend.append(`    fadeShadow(light${i}_shadowCascadeDistances);`);
						}
						var shadowCoordArgs = `SHADOWMAP_PASS(light${i}_shadowMap), dShadowCoord, light${i}_shadowParams`;
						if (vsmShadows) {
							shadowCoordArgs = `${shadowCoordArgs}, ${evsmExp}, dLightDirW`;
						} else if (pcssShadows) {
							let penumbraSizeArg = `vec2(light${i}_shadowSearchArea)`;
							if (lightShape !== LIGHTSHAPE_PUNCTUAL) {
								penumbraSizeArg = `vec2(length(light${i}_halfWidth), length(light${i}_halfHeight)) * light${i}_shadowSearchArea`;
							}
							shadowCoordArgs = `${shadowCoordArgs}, light${i}_cameraParams, ${penumbraSizeArg}, dLightDirW`;
						}
						if (lightType === LIGHTTYPE_OMNI) {
							shadowReadMode = `Point${shadowReadMode}`;
							if (!pcssShadows) {
								shadowCoordArgs = `${shadowCoordArgs}, dLightDirW`;
							}
						} else if (lightType === LIGHTTYPE_SPOT) {
							shadowReadMode = `Spot${shadowReadMode}`;
						}
						backend.append(`    float shadow${i} = getShadow${shadowReadMode}(${shadowCoordArgs});`);
						backend.append(`    dAtten *= mix(1.0, shadow${i}, light${i}_shadowIntensity);`);
					}
				}
				if (lightShape !== LIGHTSHAPE_PUNCTUAL) {
					if (options.conserveEnergy && options.useSpecular) {
						backend.append("    dDiffuseLight += ((dAttenD * dAtten) * light" + i + "_color" + (usesCookieNow ? " * dAtten3" : "") + ") * (1.0 - dLTCSpecFres);");
					} else {
						backend.append("    dDiffuseLight += (dAttenD * dAtten) * light" + i + "_color" + (usesCookieNow ? " * dAtten3" : "") + ";");
					}
				} else {
					if (hasAreaLights && options.conserveEnergy && options.useSpecular) {
						backend.append("    dDiffuseLight += (dAtten * light" + i + "_color" + (usesCookieNow ? " * dAtten3" : "") + ") * (1.0 - litArgs_specularity);");
					} else {
						backend.append("    dDiffuseLight += dAtten * light" + i + "_color" + (usesCookieNow ? " * dAtten3" : "") + ";");
					}
				}
				if (options.useSpecular) {
					backend.append("    dHalfDirW = normalize(-dLightDirNormW + dViewDirW);");
				}
				if (light.affectSpecularity) {
					if (lightShape !== LIGHTSHAPE_PUNCTUAL) {
						if (options.useClearCoat) {
							backend.append(`    ccSpecularLight += ccLTCSpecFres * get${shapeString}LightSpecular(litArgs_clearcoat_worldNormal, dViewDirW) * dAtten * light${i}_color` + (usesCookieNow ? " * dAtten3" : "") + ";");
						}
						if (options.useSpecular) {
							backend.append(`    dSpecularLight += dLTCSpecFres * get${shapeString}LightSpecular(litArgs_worldNormal, dViewDirW) * dAtten * light${i}_color` + (usesCookieNow ? " * dAtten3" : "") + ";");
						}
					} else {
						var calcFresnel = false;
						if (lightType === LIGHTTYPE_DIRECTIONAL && options.fresnelModel > 0) {
							calcFresnel = true;
						}
						if (options.useClearCoat) {
							backend.append(`    ccSpecularLight += getLightSpecular(dHalfDirW, ccReflDirW, litArgs_clearcoat_worldNormal, dViewDirW, dLightDirNormW, litArgs_clearcoat_gloss, dTBN) * dAtten * light${i}_color` + (usesCookieNow ? " * dAtten3" : "") + (calcFresnel ? " * getFresnelCC(dot(dViewDirW, dHalfDirW));" : ";"));
						}
						if (options.useSheen) {
							backend.append(`    sSpecularLight += getLightSpecularSheen(dHalfDirW, litArgs_worldNormal, dViewDirW, dLightDirNormW, litArgs_sheen_gloss) * dAtten * light${i}_color` + (usesCookieNow ? " * dAtten3;" : ";"));
						}
						if (options.useSpecular) {
							backend.append(`    dSpecularLight += getLightSpecular(dHalfDirW, dReflDirW, litArgs_worldNormal, dViewDirW, dLightDirNormW, litArgs_gloss, dTBN) * dAtten * light${i}_color` + (usesCookieNow ? " * dAtten3" : "") + (calcFresnel ? ` 
																		* getFresnel(
																				dot(dViewDirW, dHalfDirW), 
																				litArgs_gloss, 
																				litArgs_specularity
																		#if defined(LIT_IRIDESCENCE)
																				, iridescenceFresnel, 
																				litArgs_iridescence_intensity
																		#endif
																		);` : `* litArgs_specularity;`));
						}
					}
				}
				if (lightType !== LIGHTTYPE_DIRECTIONAL) {
					backend.append("    }");
				}
			}
			if (options.clusteredLightingEnabled && this.lighting) {
				usesLinearFalloff = true;
				usesInvSquaredFalloff = true;
				hasPointLights = true;
				backend.append(`    addClusteredLights(
																				litArgs_worldNormal, 
																				dViewDirW, 
																				dReflDirW,
																#if defined(LIT_CLEARCOAT)
																				ccReflDirW,
																#endif
																				litArgs_gloss, 
																				litArgs_specularity, 
																				dVertexNormalW, 
																				dTBN, 
																#if defined(LIT_IRIDESCENCE)
																				iridescenceFresnel,
																#endif
																				litArgs_clearcoat_worldNormal, 
																				litArgs_clearcoat_gloss,
																				litArgs_sheen_gloss,
																				litArgs_iridescence_intensity
																		);`);
			}
			if (hasAreaLights) {
				if (options.useClearCoat) {
					backend.append("    litArgs_clearcoat_specularity = 1.0;");
				}
				if (options.useSpecular) {
					backend.append("    litArgs_specularity = vec3(1);");
				}
			}
			if (options.useRefraction) {
				backend.append(`    addRefraction(
												litArgs_worldNormal, 
												dViewDirW, 
												litArgs_thickness, 
												litArgs_gloss, 
												litArgs_specularity, 
												litArgs_albedo, 
												litArgs_transmission,
												litArgs_ior
										#if defined(LIT_IRIDESCENCE)
												, iridescenceFresnel, 
												litArgs_iridescence_intensity
										#endif
										);`);
			}
		}
		if (options.useAo) {
			if (options.occludeDirect) {
				backend.append("    occludeDiffuse(litArgs_ao);");
			}
			if (options.occludeSpecular === SPECOCC_AO || options.occludeSpecular === SPECOCC_GLOSSDEPENDENT) {
				backend.append("    occludeSpecular(litArgs_gloss, litArgs_ao, litArgs_worldNormal, dViewDirW);");
			}
		}
		if (options.useSpecularityFactor) {
			backend.append("    dSpecularLight *= litArgs_specularityFactor;");
		}
		if (options.opacityFadesSpecular === false) {
			if (options.blendType === BLEND_NORMAL || options.blendType === BLEND_PREMULTIPLIED) {
				backend.append("float specLum = dot((dSpecularLight + dReflection.rgb * dReflection.a), vec3( 0.2126, 0.7152, 0.0722 ));");
				backend.append("#ifdef LIT_CLEARCOAT\n specLum += dot(ccSpecularLight * litArgs_clearcoat_specularity + ccReflection.rgb * litArgs_clearcoat_specularity, vec3( 0.2126, 0.7152, 0.0722 ));\n#endif");
				backend.append("litArgs_opacity = clamp(litArgs_opacity + gammaCorrectInput(specLum), 0.0, 1.0);");
			}
			backend.append("litArgs_opacity *= material_alphaFade;");
		}
		backend.append(chunks.endPS);
		if (options.blendType === BLEND_NORMAL || options.blendType === BLEND_ADDITIVEALPHA || options.alphaToCoverage) {
			backend.append(chunks.outputAlphaPS);
		} else if (options.blendType === BLEND_PREMULTIPLIED) {
			backend.append(chunks.outputAlphaPremulPS);
		} else {
			backend.append(chunks.outputAlphaOpaquePS);
		}
		if (options.useMsdf) {
			backend.append("    gl_FragColor = applyMsdf(gl_FragColor);");
		}
		backend.append(chunks.outputPS);
		backend.append(chunks.debugOutputPS);
		if (hasPointLights) {
			func.prepend(chunks.lightDirPointPS);
		}
		if (usesLinearFalloff) {
			func.prepend(chunks.falloffLinearPS);
		}
		if (usesInvSquaredFalloff) {
			func.prepend(chunks.falloffInvSquaredPS);
		}
		if (usesSpot) {
			func.prepend(chunks.spotPS);
		}
		if (usesCookie && !options.clusteredLightingEnabled) {
			func.prepend(chunks.cookiePS);
		}
		let structCode = "";
		const backendCode = `void evaluateBackend() {\n${backend.code}\n}`;
		func.append(backendCode);
		code.append(chunks.debugProcessFrontendPS);
		code.append("    evaluateBackend();");
		code.append(ShaderGenerator.end());
		const mergedCode = decl.code + func.code + code.code;
		if (mergedCode.includes("dTBN")) structCode += "mat3 dTBN;\n";
		if (mergedCode.includes("dVertexNormalW")) structCode += "vec3 dVertexNormalW;\n";
		if (mergedCode.includes("dTangentW")) structCode += "vec3 dTangentW;\n";
		if (mergedCode.includes("dBinormalW")) structCode += "vec3 dBinormalW;\n";
		if (mergedCode.includes("dViewDirW")) structCode += "vec3 dViewDirW;\n";
		if (mergedCode.includes("dReflDirW")) structCode += "vec3 dReflDirW;\n";
		if (mergedCode.includes("dHalfDirW")) structCode += "vec3 dHalfDirW;\n";
		if (mergedCode.includes("ccReflDirW")) structCode += "vec3 ccReflDirW;\n";
		if (mergedCode.includes("dLightDirNormW")) structCode += "vec3 dLightDirNormW;\n";
		if (mergedCode.includes("dLightDirW")) structCode += "vec3 dLightDirW;\n";
		if (mergedCode.includes("dLightPosW")) structCode += "vec3 dLightPosW;\n";
		if (mergedCode.includes("dShadowCoord")) structCode += "vec3 dShadowCoord;\n";
		if (mergedCode.includes("dReflection")) structCode += "vec4 dReflection;\n";
		if (mergedCode.includes("dDiffuseLight")) structCode += "vec3 dDiffuseLight;\n";
		if (mergedCode.includes("dSpecularLight")) structCode += "vec3 dSpecularLight;\n";
		if (mergedCode.includes("dAtten")) structCode += "float dAtten;\n";
		if (mergedCode.includes("dAttenD")) structCode += "float dAttenD;\n";
		if (mergedCode.includes("dAtten3")) structCode += "vec3 dAtten3;\n";
		if (mergedCode.includes("dMsdf")) structCode += "vec4 dMsdf;\n";
		if (mergedCode.includes("ccFresnel")) structCode += "float ccFresnel;\n";
		if (mergedCode.includes("ccReflection")) structCode += "vec3 ccReflection;\n";
		if (mergedCode.includes("ccSpecularLight")) structCode += "vec3 ccSpecularLight;\n";
		if (mergedCode.includes("ccSpecularityNoFres")) structCode += "float ccSpecularityNoFres;\n";
		if (mergedCode.includes("sSpecularLight")) structCode += "vec3 sSpecularLight;\n";
		if (mergedCode.includes("sReflection")) structCode += "vec3 sReflection;\n";
		const result = this._fsGetBeginCode() + this.varyings + this.varyingDefines + this._fsGetBaseCode() + structCode + this.frontendDecl + mergedCode;
		return result;
	}
	generateFragmentShader(frontendDecl, frontendCode, frontendFunc, lightingUv) {
		var _this$handleCompatibi;
		const options = this.options;
		this.frontendDecl = frontendDecl;
		this.frontendCode = frontendCode;
		this.frontendFunc = frontendFunc;
		this.lightingUv = lightingUv;
		if (options.pass === SHADER_PICK) {
			this.fshader = this._fsGetPickPassCode();
		} else if (options.pass === SHADER_DEPTH) {
			this.fshader = this._fsGetDepthPassCode();
		} else if (options.pass === SHADER_PREPASS_VELOCITY) {
			this.fshader = this._fsGetPrePassVelocityCode();
		} else if (this.shadowPass) {
			this.fshader = this._fsGetShadowPassCode();
		} else if (options.customFragmentShader) {
			this.fshader = this._fsGetBeginCode() + options.customFragmentShader;
		} else {
			this.fshader = this._fsGetLitPassCode();
		}
		(_this$handleCompatibi = this.handleCompatibility) == null || _this$handleCompatibi.call(this);
	}
	getDefinition() {
		const definition = ShaderUtils.createDefinition(this.device, {
			name: 'LitShader',
			attributes: this.attributes,
			vertexCode: this.vshader,
			fragmentCode: this.fshader
		});
		if (this.shaderPassInfo.isForward) {
			definition.tag = SHADERTAG_MATERIAL;
		}
		return definition;
	}
}

const LitOptionsUtils = {
	generateKey(options) {
		return "lit" + Object.keys(options).sort().map(key => {
			if (key === "chunks") {
				return LitOptionsUtils.generateChunksKey(options);
			} else if (key === "lights") {
				return LitOptionsUtils.generateLightsKey(options);
			}
			return key + options[key];
		}).join("\n");
	},
	generateLightsKey(options) {
		return 'lights:' + options.lights.map(light => {
			return !options.clusteredLightingEnabled || light._type === LIGHTTYPE_DIRECTIONAL ? `${light.key},` : '';
		}).join("");
	},
	generateChunksKey(options) {
		var _options$chunks;
		return 'chunks:\n' + Object.keys((_options$chunks = options.chunks) != null ? _options$chunks : {}).sort().map(key => key + options.chunks[key]).join("");
	}
};

const dummyUvs = [0, 1, 2, 3, 4, 5, 6, 7];
class ShaderGeneratorLit extends ShaderGenerator {
	generateKey(options) {
		const key = "lit" + dummyUvs.map((dummy, index) => {
			return options.usedUvs[index] ? "1" : "0";
		}).join("") + options.shaderChunk + LitOptionsUtils.generateKey(options.litOptions);
		return key;
	}
	createShaderDefinition(device, options) {
		const litShader = new LitShader(device, options.litOptions);
		const decl = new ChunkBuilder();
		const code = new ChunkBuilder();
		const func = new ChunkBuilder();
		decl.append(`uniform float textureBias;`);
		decl.append(litShader.chunks.litShaderArgsPS);
		code.append(options.shaderChunk);
		func.code = `evaluateFrontend();`;
		func.code = `\n${func.code.split('\n').map(l => `    ${l}`).join('\n')}\n\n`;
		const usedUvSets = options.usedUvs || [true];
		const mapTransforms = [];
		litShader.generateVertexShader(usedUvSets, usedUvSets, mapTransforms);
		litShader.generateFragmentShader(decl.code, code.code, func.code, "vUv0");
		return litShader.getDefinition();
	}
}
const lit = new ShaderGeneratorLit();

const options = new LitMaterialOptions();
class LitMaterial extends Material {
	constructor(...args) {
		super(...args);
		this.usedUvs = [true];
		this.shaderChunk = 'void evaluateFrontend() {}\n';
		this.chunks = null;
		this.useLighting = true;
		this.useFog = true;
		this.useGammaTonemap = true;
		this.useSkybox = true;
		this.shadingModel = SPECULAR_BLINN;
		this.ambientSH = null;
		this.pixelSnap = false;
		this.nineSlicedMode = null;
		this.fastTbn = false;
		this.twoSidedLighting = false;
		this.occludeDirect = false;
		this.occludeSpecular = SPECOCC_AO;
		this.occludeSpecularIntensity = 1;
		this.opacityFadesSpecular = true;
		this.opacityDither = DITHER_NONE;
		this.opacityShadowDither = DITHER_NONE;
		this.conserveEnergy = true;
		this.ggxSpecular = false;
		this.fresnelModel = FRESNEL_SCHLICK;
		this.dynamicRefraction = false;
		this.hasAo = false;
		this.hasSpecular = false;
		this.hasSpecularityFactor = false;
		this.hasLighting = false;
		this.hasHeights = false;
		this.hasNormals = false;
		this.hasSheen = false;
		this.hasRefraction = false;
		this.hasIrridescence = false;
		this.hasMetalness = false;
		this.hasClearCoat = false;
		this.hasClearCoatNormals = false;
	}
	getShaderVariant(device, scene, objDefs, unused, pass, sortedLights, viewUniformFormat, viewBindGroupFormat, vertexFormat) {
		options.usedUvs = this.usedUvs.slice();
		options.shaderChunk = this.shaderChunk;
		LitMaterialOptionsBuilder.update(options.litOptions, this, scene, objDefs, pass, sortedLights);
		const processingOptions = new ShaderProcessorOptions(viewUniformFormat, viewBindGroupFormat, vertexFormat);
		const library = getProgramLibrary(device);
		library.register('lit', lit);
		const shader = library.getProgram('lit', options, processingOptions, this.userId);
		return shader;
	}
}

const tempVec3 = new Vec3();
const tempMin3 = new Vec3();
const tempMax3 = new Vec3();
const tempBox = new BoundingBox();
const epsilon = 0.000001;
class ClusterLight {
	constructor() {
		this.light = null;
		this.min = new Vec3();
		this.max = new Vec3();
	}
}
class WorldClusters {
	constructor(device) {
		this.clusterTexture = void 0;
		this.device = device;
		this.name = 'Untitled';
		this.reportCount = 0;
		this.boundsMin = new Vec3();
		this.boundsMax = new Vec3();
		this.boundsDelta = new Vec3();
		this._cells = new Vec3(1, 1, 1);
		this._cellsLimit = new Vec3();
		this.cells = this._cells;
		this.maxCellLightCount = 4;
		this._maxAttenuation = 0;
		this._maxColorValue = 0;
		this._usedLights = [];
		this._usedLights.push(new ClusterLight());
		this.lightsBuffer = new LightsBuffer(device);
		this.registerUniforms(device);
	}
	set maxCellLightCount(count) {
		if (count !== this._maxCellLightCount) {
			this._maxCellLightCount = count;
			this._cellsDirty = true;
		}
	}
	get maxCellLightCount() {
		return this._maxCellLightCount;
	}
	set cells(value) {
		tempVec3.copy(value).floor();
		if (!this._cells.equals(tempVec3)) {
			this._cells.copy(tempVec3);
			this._cellsLimit.copy(tempVec3).sub(Vec3.ONE);
			this._cellsDirty = true;
		}
	}
	get cells() {
		return this._cells;
	}
	destroy() {
		this.lightsBuffer.destroy();
		this.releaseClusterTexture();
	}
	releaseClusterTexture() {
		if (this.clusterTexture) {
			this.clusterTexture.destroy();
			this.clusterTexture = null;
		}
	}
	registerUniforms(device) {
		this._clusterSkipId = device.scope.resolve('clusterSkip');
		this._clusterMaxCellsId = device.scope.resolve('clusterMaxCells');
		this._clusterWorldTextureId = device.scope.resolve('clusterWorldTexture');
		this._clusterTextureSizeId = device.scope.resolve('clusterTextureSize');
		this._clusterTextureSizeData = new Float32Array(3);
		this._clusterBoundsMinId = device.scope.resolve('clusterBoundsMin');
		this._clusterBoundsMinData = new Float32Array(3);
		this._clusterBoundsDeltaId = device.scope.resolve('clusterBoundsDelta');
		this._clusterBoundsDeltaData = new Float32Array(3);
		this._clusterCellsCountByBoundsSizeId = device.scope.resolve('clusterCellsCountByBoundsSize');
		this._clusterCellsCountByBoundsSizeData = new Float32Array(3);
		this._clusterCellsDotId = device.scope.resolve('clusterCellsDot');
		this._clusterCellsDotData = new Float32Array(3);
		this._clusterCellsMaxId = device.scope.resolve('clusterCellsMax');
		this._clusterCellsMaxData = new Float32Array(3);
		this._clusterCompressionLimit0Id = device.scope.resolve('clusterCompressionLimit0');
		this._clusterCompressionLimit0Data = new Float32Array(2);
	}
	updateParams(lightingParams) {
		if (lightingParams) {
			this.cells = lightingParams.cells;
			this.maxCellLightCount = lightingParams.maxLightsPerCell;
			this.lightsBuffer.cookiesEnabled = lightingParams.cookiesEnabled;
			this.lightsBuffer.shadowsEnabled = lightingParams.shadowsEnabled;
			this.lightsBuffer.areaLightsEnabled = lightingParams.areaLightsEnabled;
		}
	}
	updateCells() {
		if (this._cellsDirty) {
			this._cellsDirty = false;
			const cx = this._cells.x;
			const cy = this._cells.y;
			const cz = this._cells.z;
			const numCells = cx * cy * cz;
			const totalPixels = this.maxCellLightCount * numCells;
			let width = Math.ceil(Math.sqrt(totalPixels));
			width = math.roundUp(width, this.maxCellLightCount);
			const height = Math.ceil(totalPixels / width);
			this._clusterCellsMaxData[0] = cx;
			this._clusterCellsMaxData[1] = cy;
			this._clusterCellsMaxData[2] = cz;
			this._clusterCellsDotData[0] = this.maxCellLightCount;
			this._clusterCellsDotData[1] = cx * cz * this.maxCellLightCount;
			this._clusterCellsDotData[2] = cx * this.maxCellLightCount;
			this.clusters = new Uint8ClampedArray(totalPixels);
			this.counts = new Int32Array(numCells);
			this._clusterTextureSizeData[0] = width;
			this._clusterTextureSizeData[1] = 1.0 / width;
			this._clusterTextureSizeData[2] = 1.0 / height;
			this.releaseClusterTexture();
			this.clusterTexture = this.lightsBuffer.createTexture(this.device, width, height, PIXELFORMAT_L8, 'ClusterTexture');
		}
	}
	uploadTextures() {
		this.clusterTexture.lock().set(this.clusters);
		this.clusterTexture.unlock();
		this.lightsBuffer.uploadTextures();
	}
	updateUniforms() {
		this._clusterSkipId.setValue(this._usedLights.length > 1 ? 0 : 1);
		this.lightsBuffer.updateUniforms();
		this._clusterWorldTextureId.setValue(this.clusterTexture);
		this._clusterMaxCellsId.setValue(this.maxCellLightCount);
		const boundsDelta = this.boundsDelta;
		this._clusterCellsCountByBoundsSizeData[0] = this._cells.x / boundsDelta.x;
		this._clusterCellsCountByBoundsSizeData[1] = this._cells.y / boundsDelta.y;
		this._clusterCellsCountByBoundsSizeData[2] = this._cells.z / boundsDelta.z;
		this._clusterCellsCountByBoundsSizeId.setValue(this._clusterCellsCountByBoundsSizeData);
		this._clusterBoundsMinData[0] = this.boundsMin.x;
		this._clusterBoundsMinData[1] = this.boundsMin.y;
		this._clusterBoundsMinData[2] = this.boundsMin.z;
		this._clusterBoundsDeltaData[0] = boundsDelta.x;
		this._clusterBoundsDeltaData[1] = boundsDelta.y;
		this._clusterBoundsDeltaData[2] = boundsDelta.z;
		this._clusterCompressionLimit0Data[0] = this._maxAttenuation;
		this._clusterCompressionLimit0Data[1] = this._maxColorValue;
		this._clusterTextureSizeId.setValue(this._clusterTextureSizeData);
		this._clusterBoundsMinId.setValue(this._clusterBoundsMinData);
		this._clusterBoundsDeltaId.setValue(this._clusterBoundsDeltaData);
		this._clusterCellsDotId.setValue(this._clusterCellsDotData);
		this._clusterCellsMaxId.setValue(this._clusterCellsMaxData);
		this._clusterCompressionLimit0Id.setValue(this._clusterCompressionLimit0Data);
	}
	evalLightCellMinMax(clusteredLight, min, max) {
		min.copy(clusteredLight.min);
		min.sub(this.boundsMin);
		min.div(this.boundsDelta);
		min.mul2(min, this.cells);
		min.floor();
		max.copy(clusteredLight.max);
		max.sub(this.boundsMin);
		max.div(this.boundsDelta);
		max.mul2(max, this.cells);
		max.ceil();
		min.max(Vec3.ZERO);
		max.min(this._cellsLimit);
	}
	collectLights(lights) {
		const maxLights = this.lightsBuffer.maxLights;
		const usedLights = this._usedLights;
		let lightIndex = 1;
		lights.forEach(light => {
			const runtimeLight = !!(light.mask & (MASK_AFFECT_DYNAMIC | MASK_AFFECT_LIGHTMAPPED));
			const zeroAngleSpotlight = light.type === LIGHTTYPE_SPOT && light._outerConeAngle === 0;
			if (light.enabled && light.type !== LIGHTTYPE_DIRECTIONAL && light.visibleThisFrame && light.intensity > 0 && runtimeLight && !zeroAngleSpotlight) {
				if (lightIndex < maxLights) {
					let clusteredLight;
					if (lightIndex < usedLights.length) {
						clusteredLight = usedLights[lightIndex];
					} else {
						clusteredLight = new ClusterLight();
						usedLights.push(clusteredLight);
					}
					clusteredLight.light = light;
					light.getBoundingBox(tempBox);
					clusteredLight.min.copy(tempBox.getMin());
					clusteredLight.max.copy(tempBox.getMax());
					lightIndex++;
				}
			}
		});
		usedLights.length = lightIndex;
	}
	evaluateBounds() {
		const usedLights = this._usedLights;
		const min = this.boundsMin;
		const max = this.boundsMax;
		if (usedLights.length > 1) {
			min.copy(usedLights[1].min);
			max.copy(usedLights[1].max);
			for (let i = 2; i < usedLights.length; i++) {
				min.min(usedLights[i].min);
				max.max(usedLights[i].max);
			}
		} else {
			min.set(0, 0, 0);
			max.set(1, 1, 1);
		}
		this.boundsDelta.sub2(max, min);
		this.lightsBuffer.setBounds(min, this.boundsDelta);
	}
	evaluateCompressionLimits(gammaCorrection) {
		let maxAttenuation = 0;
		let maxColorValue = 0;
		const usedLights = this._usedLights;
		for (let i = 1; i < usedLights.length; i++) {
			const light = usedLights[i].light;
			maxAttenuation = Math.max(light.attenuationEnd, maxAttenuation);
			const color = gammaCorrection ? light._linearFinalColor : light._finalColor;
			maxColorValue = Math.max(color[0], maxColorValue);
			maxColorValue = Math.max(color[1], maxColorValue);
			maxColorValue = Math.max(color[2], maxColorValue);
		}
		this._maxAttenuation = maxAttenuation + epsilon;
		this._maxColorValue = maxColorValue + epsilon;
		this.lightsBuffer.setCompressionRanges(this._maxAttenuation, this._maxColorValue);
	}
	updateClusters(gammaCorrection) {
		this.counts.fill(0);
		this.clusters.fill(0);
		const divX = this._cells.x;
		const divZ = this._cells.z;
		const counts = this.counts;
		const limit = this._maxCellLightCount;
		const clusters = this.clusters;
		const pixelsPerCellCount = this.maxCellLightCount;
		const usedLights = this._usedLights;
		for (let i = 1; i < usedLights.length; i++) {
			const clusteredLight = usedLights[i];
			const light = clusteredLight.light;
			this.lightsBuffer.addLightData(light, i, gammaCorrection);
			this.evalLightCellMinMax(clusteredLight, tempMin3, tempMax3);
			const xStart = tempMin3.x;
			const xEnd = tempMax3.x;
			const yStart = tempMin3.y;
			const yEnd = tempMax3.y;
			const zStart = tempMin3.z;
			const zEnd = tempMax3.z;
			for (let x = xStart; x <= xEnd; x++) {
				for (let z = zStart; z <= zEnd; z++) {
					for (let y = yStart; y <= yEnd; y++) {
						const clusterIndex = x + divX * (z + y * divZ);
						const count = counts[clusterIndex];
						if (count < limit) {
							clusters[pixelsPerCellCount * clusterIndex + count] = i;
							counts[clusterIndex] = count + 1;
						}
					}
				}
			}
		}
	}
	update(lights, gammaCorrection, lightingParams) {
		this.updateParams(lightingParams);
		this.updateCells();
		this.collectLights(lights);
		this.evaluateBounds();
		this.evaluateCompressionLimits(gammaCorrection);
		this.updateClusters(gammaCorrection);
		this.uploadTextures();
	}
	activate() {
		this.updateUniforms();
	}
}

const _goldenAngle = 2.399963229728653;
const random = {
	circlePoint(point) {
		const r = Math.sqrt(Math.random());
		const theta = Math.random() * 2 * Math.PI;
		point.x = r * Math.cos(theta);
		point.y = r * Math.sin(theta);
	},
	circlePointDeterministic(point, index, numPoints) {
		const theta = index * _goldenAngle;
		const r = Math.sqrt(index) / Math.sqrt(numPoints);
		point.x = r * Math.cos(theta);
		point.y = r * Math.sin(theta);
	},
	spherePointDeterministic(point, index, numPoints, start = 0, end = 1) {
		start = 1 - 2 * start;
		end = 1 - 2 * end;
		const y = math.lerp(start, end, index / numPoints);
		const radius = Math.sqrt(1 - y * y);
		const theta = _goldenAngle * index;
		point.x = Math.cos(theta) * radius;
		point.y = y;
		point.z = Math.sin(theta) * radius;
	},
	radicalInverse(i) {
		let bits = (i << 16 | i >>> 16) >>> 0;
		bits = ((bits & 0x55555555) << 1 | (bits & 0xAAAAAAAA) >>> 1) >>> 0;
		bits = ((bits & 0x33333333) << 2 | (bits & 0xCCCCCCCC) >>> 2) >>> 0;
		bits = ((bits & 0x0F0F0F0F) << 4 | (bits & 0xF0F0F0F0) >>> 4) >>> 0;
		bits = ((bits & 0x00FF00FF) << 8 | (bits & 0xFF00FF00) >>> 8) >>> 0;
		return bits * 2.3283064365386963e-10;
	}
};

const getProjectionName = projection => {
	switch (projection) {
		case TEXTUREPROJECTION_CUBE:
			return "Cubemap";
		case TEXTUREPROJECTION_OCTAHEDRAL:
			return "Octahedral";
		default:
			return "Equirect";
	}
};
const packFloat32ToRGBA8 = (value, array, offset) => {
	if (value <= 0) {
		array[offset + 0] = 0;
		array[offset + 1] = 0;
		array[offset + 2] = 0;
		array[offset + 3] = 0;
	} else if (value >= 1.0) {
		array[offset + 0] = 255;
		array[offset + 1] = 0;
		array[offset + 2] = 0;
		array[offset + 3] = 0;
	} else {
		let encX = 1 * value % 1;
		let encY = 255 * value % 1;
		let encZ = 65025 * value % 1;
		const encW = 16581375.0 * value % 1;
		encX -= encY / 255;
		encY -= encZ / 255;
		encZ -= encW / 255;
		array[offset + 0] = Math.min(255, Math.floor(encX * 256));
		array[offset + 1] = Math.min(255, Math.floor(encY * 256));
		array[offset + 2] = Math.min(255, Math.floor(encZ * 256));
		array[offset + 3] = Math.min(255, Math.floor(encW * 256));
	}
};
const packSamples = samples => {
	const numSamples = samples.length;
	const w = Math.min(numSamples, 512);
	const h = Math.ceil(numSamples / w);
	const data = new Uint8Array(w * h * 4);
	let off = 0;
	for (let i = 0; i < numSamples; i += 4) {
		packFloat32ToRGBA8(samples[i + 0] * 0.5 + 0.5, data, off + 0);
		packFloat32ToRGBA8(samples[i + 1] * 0.5 + 0.5, data, off + 4);
		packFloat32ToRGBA8(samples[i + 2] * 0.5 + 0.5, data, off + 8);
		packFloat32ToRGBA8(samples[i + 3] / 8, data, off + 12);
		off += 16;
	}
	return {
		width: w,
		height: h,
		data: data
	};
};
const hemisphereSamplePhong = (dstVec, x, y, specularPower) => {
	const phi = y * 2 * Math.PI;
	const cosTheta = Math.pow(1 - x, 1 / (specularPower + 1));
	const sinTheta = Math.sqrt(1 - cosTheta * cosTheta);
	dstVec.set(Math.cos(phi) * sinTheta, Math.sin(phi) * sinTheta, cosTheta).normalize();
};
const hemisphereSampleLambert = (dstVec, x, y) => {
	const phi = y * 2 * Math.PI;
	const cosTheta = Math.sqrt(1 - x);
	const sinTheta = Math.sqrt(x);
	dstVec.set(Math.cos(phi) * sinTheta, Math.sin(phi) * sinTheta, cosTheta).normalize();
};
const hemisphereSampleGGX = (dstVec, x, y, a) => {
	const phi = y * 2 * Math.PI;
	const cosTheta = Math.sqrt((1 - x) / (1 + (a * a - 1) * x));
	const sinTheta = Math.sqrt(1 - cosTheta * cosTheta);
	dstVec.set(Math.cos(phi) * sinTheta, Math.sin(phi) * sinTheta, cosTheta).normalize();
};
const D_GGX = (NoH, linearRoughness) => {
	const a = NoH * linearRoughness;
	const k = linearRoughness / (1.0 - NoH * NoH + a * a);
	return k * k * (1 / Math.PI);
};
const generatePhongSamples = (numSamples, specularPower) => {
	const H = new Vec3();
	const result = [];
	for (let i = 0; i < numSamples; ++i) {
		hemisphereSamplePhong(H, i / numSamples, random.radicalInverse(i), specularPower);
		result.push(H.x, H.y, H.z, 0);
	}
	return result;
};
const generateLambertSamples = (numSamples, sourceTotalPixels) => {
	const pixelsPerSample = sourceTotalPixels / numSamples;
	const H = new Vec3();
	const result = [];
	for (let i = 0; i < numSamples; ++i) {
		hemisphereSampleLambert(H, i / numSamples, random.radicalInverse(i));
		const pdf = H.z / Math.PI;
		const mipLevel = 0.5 * Math.log2(pixelsPerSample / pdf);
		result.push(H.x, H.y, H.z, mipLevel);
	}
	return result;
};
const requiredSamplesGGX = {
	"16": {
		"2": 26,
		"8": 20,
		"32": 17,
		"128": 16,
		"512": 16
	},
	"32": {
		"2": 53,
		"8": 40,
		"32": 34,
		"128": 32,
		"512": 32
	},
	"128": {
		"2": 214,
		"8": 163,
		"32": 139,
		"128": 130,
		"512": 128
	},
	"1024": {
		"2": 1722,
		"8": 1310,
		"32": 1114,
		"128": 1041,
		"512": 1025
	}
};
const getRequiredSamplesGGX = (numSamples, specularPower) => {
	const table = requiredSamplesGGX[numSamples];
	return table && table[specularPower] || numSamples;
};
const generateGGXSamples = (numSamples, specularPower, sourceTotalPixels) => {
	const pixelsPerSample = sourceTotalPixels / numSamples;
	const roughness = 1 - Math.log2(specularPower) / 11.0;
	const a = roughness * roughness;
	const H = new Vec3();
	const L = new Vec3();
	const N = new Vec3(0, 0, 1);
	const result = [];
	const requiredSamples = getRequiredSamplesGGX(numSamples, specularPower);
	for (let i = 0; i < requiredSamples; ++i) {
		hemisphereSampleGGX(H, i / requiredSamples, random.radicalInverse(i), a);
		const NoH = H.z;
		L.set(H.x, H.y, H.z).mulScalar(2 * NoH).sub(N);
		if (L.z > 0) {
			const pdf = D_GGX(Math.min(1, NoH), a) / 4 + 0.001;
			const mipLevel = 0.5 * Math.log2(pixelsPerSample / pdf);
			result.push(L.x, L.y, L.z, mipLevel);
		}
	}
	while (result.length < numSamples * 4) {
		result.push(0, 0, 0, 0);
	}
	return result;
};
const createSamplesTex = (device, name, samples) => {
	const packedSamples = packSamples(samples);
	return new Texture(device, {
		name: name,
		width: packedSamples.width,
		height: packedSamples.height,
		mipmaps: false,
		minFilter: FILTER_NEAREST,
		magFilter: FILTER_NEAREST,
		levels: [packedSamples.data]
	});
};
class SimpleCache {
	constructor(destroyContent = true) {
		this.map = new Map();
		this.destroyContent = destroyContent;
	}
	destroy() {
		if (this.destroyContent) {
			this.map.forEach((value, key) => {
				value.destroy();
			});
		}
	}
	get(key, missFunc) {
		if (!this.map.has(key)) {
			const result = missFunc();
			this.map.set(key, result);
			return result;
		}
		return this.map.get(key);
	}
}
const samplesCache = new SimpleCache(false);
const deviceCache$2 = new DeviceCache();
const getCachedTexture = (device, key, getSamplesFnc) => {
	const cache = deviceCache$2.get(device, () => {
		return new SimpleCache();
	});
	return cache.get(key, () => {
		return createSamplesTex(device, key, samplesCache.get(key, getSamplesFnc));
	});
};
const generateLambertSamplesTex = (device, numSamples, sourceTotalPixels) => {
	const key = `lambert-samples-${numSamples}-${sourceTotalPixels}`;
	return getCachedTexture(device, key, () => {
		return generateLambertSamples(numSamples, sourceTotalPixels);
	});
};
const generatePhongSamplesTex = (device, numSamples, specularPower) => {
	const key = `phong-samples-${numSamples}-${specularPower}`;
	return getCachedTexture(device, key, () => {
		return generatePhongSamples(numSamples, specularPower);
	});
};
const generateGGXSamplesTex = (device, numSamples, specularPower, sourceTotalPixels) => {
	const key = `ggx-samples-${numSamples}-${specularPower}-${sourceTotalPixels}`;
	return getCachedTexture(device, key, () => {
		return generateGGXSamples(numSamples, specularPower, sourceTotalPixels);
	});
};
const vsCode = `
attribute vec2 vertex_position;

uniform vec4 uvMod;

varying vec2 vUv0;

void main(void) {
		gl_Position = vec4(vertex_position, 0.5, 1.0);
		vUv0 = getImageEffectUV((vertex_position.xy * 0.5 + 0.5) * uvMod.xy + uvMod.zw);
}
`;
function reprojectTexture(source, target, options = {}) {
	var _options$seamPixels, _options$rect$z, _options$rect, _options$rect$w, _options$rect2;
	if (source instanceof GraphicsDevice) {
		source = arguments[1];
		target = arguments[2];
		options = {};
		if (arguments[3] !== undefined) {
			options.specularPower = arguments[3];
		}
		if (arguments[4] !== undefined) {
			options.numSamples = arguments[4];
		}
	}
	const seamPixels = (_options$seamPixels = options.seamPixels) != null ? _options$seamPixels : 0;
	const innerWidth = ((_options$rect$z = (_options$rect = options.rect) == null ? void 0 : _options$rect.z) != null ? _options$rect$z : target.width) - seamPixels * 2;
	const innerHeight = ((_options$rect$w = (_options$rect2 = options.rect) == null ? void 0 : _options$rect2.w) != null ? _options$rect$w : target.height) - seamPixels * 2;
	if (innerWidth < 1 || innerHeight < 1) {
		return false;
	}
	const funcNames = {
		'none': 'reproject',
		'lambert': 'prefilterSamplesUnweighted',
		'phong': 'prefilterSamplesUnweighted',
		'ggx': 'prefilterSamples'
	};
	const specularPower = options.hasOwnProperty('specularPower') ? options.specularPower : 1;
	const face = options.hasOwnProperty('face') ? options.face : null;
	const distribution = options.hasOwnProperty('distribution') ? options.distribution : specularPower === 1 ? 'none' : 'phong';
	const processFunc = funcNames[distribution] || 'reproject';
	const prefilterSamples = processFunc.startsWith('prefilterSamples');
	const decodeFunc = ChunkUtils.decodeFunc(source.encoding);
	const encodeFunc = ChunkUtils.encodeFunc(target.encoding);
	const sourceFunc = `sample${getProjectionName(source.projection)}`;
	const targetFunc = `getDirection${getProjectionName(target.projection)}`;
	const numSamples = options.hasOwnProperty('numSamples') ? options.numSamples : 1024;
	const shaderKey = `${processFunc}_${decodeFunc}_${encodeFunc}_${sourceFunc}_${targetFunc}_${numSamples}`;
	const device = source.device;
	let shader = getProgramLibrary(device).getCachedShader(shaderKey);
	if (!shader) {
		const defines = `#define PROCESS_FUNC ${processFunc}\n` + (prefilterSamples ? `#define USE_SAMPLES_TEX\n` : '') + (source.cubemap ? `#define CUBEMAP_SOURCE\n` : '') + `#define DECODE_FUNC ${decodeFunc}\n` + `#define ENCODE_FUNC ${encodeFunc}\n` + `#define SOURCE_FUNC ${sourceFunc}\n` + `#define TARGET_FUNC ${targetFunc}\n` + `#define NUM_SAMPLES ${numSamples}\n` + `#define NUM_SAMPLES_SQRT ${Math.round(Math.sqrt(numSamples)).toFixed(1)}\n`;
		shader = createShaderFromCode(device, vsCode, `${defines}\n${shaderChunks.reprojectPS}`, shaderKey);
	}
	device.setBlendState(BlendState.NOBLEND);
	const constantSource = device.scope.resolve(source.cubemap ? "sourceCube" : "sourceTex");
	constantSource.setValue(source);
	const constantParams = device.scope.resolve("params");
	const constantParams2 = device.scope.resolve("params2");
	const uvModParam = device.scope.resolve("uvMod");
	if (seamPixels > 0) {
		uvModParam.setValue([(innerWidth + seamPixels * 2) / innerWidth, (innerHeight + seamPixels * 2) / innerHeight, -seamPixels / innerWidth, -seamPixels / innerHeight]);
	} else {
		uvModParam.setValue([1, 1, 0, 0]);
	}
	const params = [0, specularPower, source.fixCubemapSeams ? 1.0 / source.width : 0.0, target.fixCubemapSeams ? 1.0 / target.width : 0.0];
	const params2 = [target.width * target.height * (target.cubemap ? 6 : 1), source.width * source.height * (source.cubemap ? 6 : 1)];
	if (prefilterSamples) {
		const sourceTotalPixels = source.width * source.height * (source.cubemap ? 6 : 1);
		const samplesTex = distribution === 'ggx' ? generateGGXSamplesTex(device, numSamples, specularPower, sourceTotalPixels) : distribution === 'lambert' ? generateLambertSamplesTex(device, numSamples, sourceTotalPixels) : generatePhongSamplesTex(device, numSamples, specularPower);
		device.scope.resolve("samplesTex").setValue(samplesTex);
		device.scope.resolve("samplesTexInverseSize").setValue([1.0 / samplesTex.width, 1.0 / samplesTex.height]);
	}
	for (let f = 0; f < (target.cubemap ? 6 : 1); f++) {
		if (face === null || f === face) {
			var _options;
			const renderTarget = new RenderTarget({
				colorBuffer: target,
				face: f,
				depth: false,
				flipY: device.isWebGPU
			});
			params[0] = f;
			constantParams.setValue(params);
			constantParams2.setValue(params2);
			drawQuadWithShader(device, renderTarget, shader, (_options = options) == null ? void 0 : _options.rect);
			renderTarget.destroy();
		}
	}
	return true;
}

const fixCubemapSeams = true;
const calcLevels = (width, height = 0) => {
	return 1 + Math.floor(Math.log2(Math.max(width, height)));
};
const supportsFloat16 = device => {
	return device.extTextureHalfFloat && device.textureHalfFloatRenderable;
};
const supportsFloat32 = device => {
	return device.extTextureFloat && device.textureFloatRenderable;
};
const lightingSourcePixelFormat = device => {
	return supportsFloat16(device) ? PIXELFORMAT_RGBA16F : supportsFloat32(device) ? PIXELFORMAT_RGBA32F : PIXELFORMAT_RGBA8;
};
const lightingPixelFormat = device => {
	return PIXELFORMAT_RGBA8;
};
const createCubemap = (device, size, format, mipmaps) => {
	return new Texture(device, {
		name: `lighting-${size}`,
		cubemap: true,
		width: size,
		height: size,
		format: format,
		type: format === PIXELFORMAT_RGBA8 ? TEXTURETYPE_RGBP : TEXTURETYPE_DEFAULT,
		addressU: ADDRESS_CLAMP_TO_EDGE,
		addressV: ADDRESS_CLAMP_TO_EDGE,
		fixCubemapSeams: fixCubemapSeams,
		mipmaps: !!mipmaps
	});
};
class EnvLighting {
	static generateSkyboxCubemap(source, size) {
		const device = source.device;
		const result = createCubemap(device, size || (source.cubemap ? source.width : source.width / 4), PIXELFORMAT_RGBA8, false);
		reprojectTexture(source, result, {
			numSamples: 1024
		});
		return result;
	}
	static generateLightingSource(source, options) {
		const device = source.device;
		const format = lightingSourcePixelFormat(device);
		const result = (options == null ? void 0 : options.target) || new Texture(device, {
			name: `lighting-source`,
			cubemap: true,
			width: (options == null ? void 0 : options.size) || 128,
			height: (options == null ? void 0 : options.size) || 128,
			format: format,
			type: format === PIXELFORMAT_RGBA8 ? TEXTURETYPE_RGBP : TEXTURETYPE_DEFAULT,
			addressU: ADDRESS_CLAMP_TO_EDGE,
			addressV: ADDRESS_CLAMP_TO_EDGE,
			fixCubemapSeams: false,
			mipmaps: true
		});
		reprojectTexture(source, result, {
			numSamples: source.mipmaps ? 1 : 1024
		});
		return result;
	}
	static generateAtlas(source, options) {
		const device = source.device;
		const format = lightingPixelFormat();
		const result = (options == null ? void 0 : options.target) || new Texture(device, {
			name: 'envAtlas',
			width: (options == null ? void 0 : options.size) || 512,
			height: (options == null ? void 0 : options.size) || 512,
			format: format,
			type: TEXTURETYPE_RGBP ,
			projection: TEXTUREPROJECTION_EQUIRECT,
			addressU: ADDRESS_CLAMP_TO_EDGE,
			addressV: ADDRESS_CLAMP_TO_EDGE,
			mipmaps: false
		});
		const s = result.width / 512;
		const rect = new Vec4(0, 0, 512 * s, 256 * s);
		const levels = calcLevels(256) - calcLevels(4);
		for (let i = 0; i < levels; ++i) {
			reprojectTexture(source, result, {
				numSamples: 1,
				rect: rect,
				seamPixels: s
			});
			rect.x += rect.w;
			rect.y += rect.w;
			rect.z = Math.max(1, Math.floor(rect.z * 0.5));
			rect.w = Math.max(1, Math.floor(rect.w * 0.5));
		}
		rect.set(0, 256 * s, 256 * s, 128 * s);
		for (let i = 1; i < 7; ++i) {
			reprojectTexture(source, result, {
				numSamples: (options == null ? void 0 : options.numReflectionSamples) || 1024,
				distribution: (options == null ? void 0 : options.distribution) || 'ggx',
				specularPower: Math.max(1, 2048 >> i * 2),
				rect: rect,
				seamPixels: s
			});
			rect.y += rect.w;
			rect.z = Math.max(1, Math.floor(rect.z * 0.5));
			rect.w = Math.max(1, Math.floor(rect.w * 0.5));
		}
		rect.set(128 * s, (256 + 128) * s, 64 * s, 32 * s);
		reprojectTexture(source, result, {
			numSamples: (options == null ? void 0 : options.numAmbientSamples) || 2048,
			distribution: 'lambert',
			rect: rect,
			seamPixels: s
		});
		return result;
	}
	static generatePrefilteredAtlas(sources, options) {
		const device = sources[0].device;
		const format = sources[0].format;
		const type = sources[0].type;
		const result = (options == null ? void 0 : options.target) || new Texture(device, {
			name: 'envPrefilteredAtlas',
			width: (options == null ? void 0 : options.size) || 512,
			height: (options == null ? void 0 : options.size) || 512,
			format: format,
			type: type,
			projection: TEXTUREPROJECTION_EQUIRECT,
			addressU: ADDRESS_CLAMP_TO_EDGE,
			addressV: ADDRESS_CLAMP_TO_EDGE,
			mipmaps: false
		});
		const s = result.width / 512;
		const rect = new Vec4(0, 0, 512 * s, 256 * s);
		const levels = calcLevels(512);
		for (let i = 0; i < levels; ++i) {
			reprojectTexture(sources[0], result, {
				numSamples: 1,
				rect: rect,
				seamPixels: s
			});
			rect.x += rect.w;
			rect.y += rect.w;
			rect.z = Math.max(1, Math.floor(rect.z * 0.5));
			rect.w = Math.max(1, Math.floor(rect.w * 0.5));
		}
		rect.set(0, 256 * s, 256 * s, 128 * s);
		for (let i = 1; i < sources.length; ++i) {
			reprojectTexture(sources[i], result, {
				numSamples: 1,
				rect: rect,
				seamPixels: s
			});
			rect.y += rect.w;
			rect.z = Math.max(1, Math.floor(rect.z * 0.5));
			rect.w = Math.max(1, Math.floor(rect.w * 0.5));
		}
		rect.set(128 * s, (256 + 128) * s, 64 * s, 32 * s);
		if (options != null && options.legacyAmbient) {
			reprojectTexture(sources[5], result, {
				numSamples: 1,
				rect: rect,
				seamPixels: s
			});
		} else {
			reprojectTexture(sources[0], result, {
				numSamples: (options == null ? void 0 : options.numSamples) || 2048,
				distribution: 'lambert',
				rect: rect,
				seamPixels: s
			});
		}
		return result;
	}
}

class StandardMaterialOptions {
	constructor() {
		this.forceUv1 = false;
		this.ambientTint = false;
		this.diffuseTint = false;
		this.specularTint = false;
		this.metalnessTint = false;
		this.glossTint = false;
		this.emissiveTint = false;
		this.opacityTint = false;
		this.emissiveEncoding = 'linear';
		this.lightMapEncoding = 'linear';
		this.packedNormal = false;
		this.glossInvert = false;
		this.sheenGlossInvert = false;
		this.clearCoatGlossInvert = false;
		this.litOptions = new LitShaderOptions();
	}
	get pass() {
		return this.litOptions.pass;
	}
}

const _matTex2D = [];
const buildPropertiesList = options => {
	return Object.keys(options).filter(key => key !== "litOptions").sort();
};
class ShaderGeneratorStandard extends ShaderGenerator {
	constructor(...args) {
		super(...args);
		this.optionsContext = new StandardMaterialOptions();
		this.optionsContextMin = new StandardMaterialOptions();
	}
	generateKey(options) {
		let props;
		if (options === this.optionsContextMin) {
			if (!this.propsMin) this.propsMin = buildPropertiesList(options);
			props = this.propsMin;
		} else if (options === this.optionsContext) {
			if (!this.props) this.props = buildPropertiesList(options);
			props = this.props;
		} else {
			props = buildPropertiesList(options);
		}
		const key = "standard:\n" + props.map(prop => prop + options[prop]).join('\n') + LitOptionsUtils.generateKey(options.litOptions);
		return key;
	}
	_getUvSourceExpression(transformPropName, uVPropName, options) {
		const transformId = options[transformPropName];
		const uvChannel = options[uVPropName];
		const isMainPass = options.litOptions.pass === SHADER_FORWARD || options.litOptions.pass === SHADER_FORWARDHDR;
		let expression;
		if (isMainPass && options.litOptions.nineSlicedMode === SPRITE_RENDERMODE_SLICED) {
			expression = "nineSlicedUv";
		} else if (isMainPass && options.litOptions.nineSlicedMode === SPRITE_RENDERMODE_TILED) {
			expression = "nineSlicedUv";
		} else {
			if (transformId === 0) {
				expression = "vUv" + uvChannel;
			} else {
				expression = "vUV" + uvChannel + "_" + transformId;
			}
			if (options.heightMap && transformPropName !== "heightMapTransform") {
				expression += " + dUvOffset";
			}
		}
		return expression;
	}
	_addMapDef(name, enabled) {
		return enabled ? `#define ${name}\n` : `#undef ${name}\n`;
	}
	_addMapDefs(float, color, vertex, map, invert) {
		return this._addMapDef("MAPFLOAT", float) + this._addMapDef("MAPCOLOR", color) + this._addMapDef("MAPVERTEX", vertex) + this._addMapDef("MAPTEXTURE", map) + this._addMapDef("MAPINVERT", invert);
	}
	_addMap(propName, chunkName, options, chunks, mapping, encoding = null) {
		const mapPropName = propName + "Map";
		const uVPropName = mapPropName + "Uv";
		const identifierPropName = mapPropName + "Identifier";
		const transformPropName = mapPropName + "Transform";
		const channelPropName = mapPropName + "Channel";
		const vertexColorChannelPropName = propName + "VertexColorChannel";
		const tintPropName = propName + "Tint";
		const vertexColorPropName = propName + "VertexColor";
		const detailModePropName = propName + "Mode";
		const invertName = propName + "Invert";
		const tintOption = options[tintPropName];
		const vertexColorOption = options[vertexColorPropName];
		const textureOption = options[mapPropName];
		const textureIdentifier = options[identifierPropName];
		const detailModeOption = options[detailModePropName];
		let subCode = chunks[chunkName];
		if (textureOption) {
			const uv = this._getUvSourceExpression(transformPropName, uVPropName, options);
			subCode = subCode.replace(/\$UV/g, uv).replace(/\$CH/g, options[channelPropName]);
			if (mapping && subCode.search(/\$SAMPLER/g) !== -1) {
				let samplerName = "texture_" + mapPropName;
				const alias = mapping[textureIdentifier];
				if (alias) {
					samplerName = alias;
				} else {
					mapping[textureIdentifier] = samplerName;
				}
				subCode = subCode.replace(/\$SAMPLER/g, samplerName);
			}
			if (encoding) {
				if (options[channelPropName] === 'aaa') {
					subCode = subCode.replace(/\$DECODE/g, 'passThrough');
				} else {
					subCode = subCode.replace(/\$DECODE/g, ChunkUtils.decodeFunc(!options.litOptions.gamma && encoding === 'srgb' ? 'linear' : encoding));
				}
				if (subCode.indexOf('$texture2DSAMPLE')) {
					const decodeTable = {
						linear: 'texture2D',
						srgb: 'texture2DSRGB',
						rgbm: 'texture2DRGBM',
						rgbe: 'texture2DRGBE'
					};
					subCode = subCode.replace(/\$texture2DSAMPLE/g, decodeTable[encoding] || 'texture2D');
				}
			}
		}
		if (vertexColorOption) {
			subCode = subCode.replace(/\$VC/g, options[vertexColorChannelPropName]);
		}
		if (detailModeOption) {
			subCode = subCode.replace(/\$DETAILMODE/g, detailModeOption);
		}
		const isFloatTint = !!(tintOption & 1);
		const isVecTint = !!(tintOption & 2);
		const invertOption = !!options[invertName];
		subCode = this._addMapDefs(isFloatTint, isVecTint, vertexColorOption, textureOption, invertOption) + subCode;
		return subCode.replace(/\$/g, "");
	}
	_correctChannel(p, chan, _matTex2D) {
		if (_matTex2D[p] > 0) {
			if (_matTex2D[p] < chan.length) {
				return chan.substring(0, _matTex2D[p]);
			} else if (_matTex2D[p] > chan.length) {
				let str = chan;
				const chr = str.charAt(str.length - 1);
				const addLen = _matTex2D[p] - str.length;
				for (let i = 0; i < addLen; i++) str += chr;
				return str;
			}
			return chan;
		}
	}
	createShaderDefinition(device, options) {
		const shaderPassInfo = ShaderPass.get(device).getByIndex(options.litOptions.pass);
		const isForwardPass = shaderPassInfo.isForward;
		const litShader = new LitShader(device, options.litOptions);
		const useUv = [];
		const useUnmodifiedUv = [];
		const mapTransforms = [];
		const maxUvSets = 2;
		const textureMapping = {};
		for (const p in _matTex2D) {
			const mname = p + "Map";
			if (options[p + "VertexColor"]) {
				const cname = p + "VertexColorChannel";
				options[cname] = this._correctChannel(p, options[cname], _matTex2D);
			}
			if (options[mname]) {
				const cname = mname + "Channel";
				const tname = mname + "Transform";
				const uname = mname + "Uv";
				options[uname] = Math.min(options[uname], maxUvSets - 1);
				options[cname] = this._correctChannel(p, options[cname], _matTex2D);
				const uvSet = options[uname];
				useUv[uvSet] = true;
				useUnmodifiedUv[uvSet] = useUnmodifiedUv[uvSet] || options[mname] && !options[tname];
				if (options[tname]) {
					mapTransforms.push({
						name: p,
						id: options[tname],
						uv: options[uname]
					});
				}
			}
		}
		if (options.forceUv1) {
			useUv[1] = true;
			useUnmodifiedUv[1] = useUnmodifiedUv[1] !== undefined ? useUnmodifiedUv[1] : true;
		}
		litShader.generateVertexShader(useUv, useUnmodifiedUv, mapTransforms);
		if (options.litOptions.shadingModel === SPECULAR_PHONG) {
			options.litOptions.fresnelModel = 0;
			options.litOptions.ambientSH = false;
		} else {
			options.litOptions.fresnelModel = options.litOptions.fresnelModel === 0 ? FRESNEL_SCHLICK : options.litOptions.fresnelModel;
		}
		const decl = new ChunkBuilder();
		const code = new ChunkBuilder();
		const func = new ChunkBuilder();
		const args = new ChunkBuilder();
		let lightingUv = "";
		if (options.litOptions.nineSlicedMode === SPRITE_RENDERMODE_TILED) {
			decl.append(`const float textureBias = -1000.0;`);
		} else {
			decl.append(`uniform float textureBias;`);
		}
		if (isForwardPass) {
			if (options.heightMap) {
				decl.append("vec2 dUvOffset;");
				code.append(this._addMap("height", "parallaxPS", options, litShader.chunks, textureMapping));
				func.append("getParallax();");
			}
			if (options.litOptions.blendType !== BLEND_NONE || options.litOptions.alphaTest || options.litOptions.alphaToCoverage || options.litOptions.opacityDither !== DITHER_NONE) {
				decl.append("float dAlpha;");
				code.append(this._addMap("opacity", "opacityPS", options, litShader.chunks, textureMapping));
				func.append("getOpacity();");
				args.append("litArgs_opacity = dAlpha;");
				if (options.litOptions.alphaTest) {
					code.append(litShader.chunks.alphaTestPS);
					func.append("alphaTest(dAlpha);");
				}
				const opacityDither = options.litOptions.opacityDither;
				if (opacityDither !== DITHER_NONE) {
					if (opacityDither === DITHER_BAYER8) decl.append(litShader.chunks.bayerPS);
					decl.append(`#define DITHER_${opacityDither.toUpperCase()}\n`);
					decl.append(litShader.chunks.opacityDitherPS);
					func.append("opacityDither(dAlpha, 0.0);");
				}
			} else {
				decl.append("float dAlpha = 1.0;");
			}
			if (litShader.needsNormal) {
				if (options.normalMap || options.clearCoatNormalMap) {
					code.append(options.packedNormal ? litShader.chunks.normalXYPS : litShader.chunks.normalXYZPS);
					if (!options.litOptions.hasTangents) {
						const baseName = options.normalMap ? "normalMap" : "clearCoatNormalMap";
						lightingUv = this._getUvSourceExpression(`${baseName}Transform`, `${baseName}Uv`, options);
					}
				}
				decl.append("vec3 dNormalW;");
				code.append(this._addMap("normalDetail", "normalDetailMapPS", options, litShader.chunks, textureMapping));
				code.append(this._addMap("normal", "normalMapPS", options, litShader.chunks, textureMapping));
				func.append("getNormal();");
				args.append("litArgs_worldNormal = dNormalW;");
			}
			if (litShader.needsSceneColor) {
				decl.append("uniform sampler2D uSceneColorMap;");
			}
			if (litShader.needsScreenSize) {
				decl.append("uniform vec4 uScreenSize;");
			}
			if (litShader.needsTransforms) {
				decl.append("uniform mat4 matrix_viewProjection;");
				decl.append("uniform mat4 matrix_model;");
			}
			if (options.diffuseDetail || options.aoDetail) {
				code.append(litShader.chunks.detailModesPS);
			}
			decl.append("vec3 dAlbedo;");
			if (options.diffuseDetail) {
				code.append(this._addMap("diffuseDetail", "diffuseDetailMapPS", options, litShader.chunks, textureMapping, options.diffuseDetailEncoding));
			}
			code.append(this._addMap("diffuse", "diffusePS", options, litShader.chunks, textureMapping, options.diffuseEncoding));
			func.append("getAlbedo();");
			args.append("litArgs_albedo = dAlbedo;");
			if (options.litOptions.useRefraction) {
				decl.append("float dTransmission;");
				code.append(this._addMap("refraction", "transmissionPS", options, litShader.chunks, textureMapping));
				func.append("getRefraction();");
				args.append("litArgs_transmission = dTransmission;");
				decl.append("float dThickness;");
				code.append(this._addMap("thickness", "thicknessPS", options, litShader.chunks, textureMapping));
				func.append("getThickness();");
				args.append("litArgs_thickness = dThickness;");
			}
			if (options.litOptions.useIridescence) {
				decl.append("float dIridescence;");
				code.append(this._addMap("iridescence", "iridescencePS", options, litShader.chunks, textureMapping));
				func.append("getIridescence();");
				args.append("litArgs_iridescence_intensity = dIridescence;");
				decl.append("float dIridescenceThickness;");
				code.append(this._addMap("iridescenceThickness", "iridescenceThicknessPS", options, litShader.chunks, textureMapping));
				func.append("getIridescenceThickness();");
				args.append("litArgs_iridescence_thickness = dIridescenceThickness;");
			}
			if (litShader.lighting && options.litOptions.useSpecular || litShader.reflections) {
				decl.append("vec3 dSpecularity;");
				decl.append("float dGlossiness;");
				if (options.litOptions.useSheen) {
					decl.append("vec3 sSpecularity;");
					code.append(this._addMap("sheen", "sheenPS", options, litShader.chunks, textureMapping, options.sheenEncoding));
					func.append("getSheen();");
					args.append("litArgs_sheen_specularity = sSpecularity;");
					decl.append("float sGlossiness;");
					code.append(this._addMap("sheenGloss", "sheenGlossPS", options, litShader.chunks, textureMapping));
					func.append("getSheenGlossiness();");
					args.append("litArgs_sheen_gloss = sGlossiness;");
				}
				if (options.litOptions.useMetalness) {
					decl.append("float dMetalness;");
					code.append(this._addMap("metalness", "metalnessPS", options, litShader.chunks, textureMapping));
					func.append("getMetalness();");
					args.append("litArgs_metalness = dMetalness;");
					decl.append("float dIor;");
					code.append(this._addMap("ior", "iorPS", options, litShader.chunks, textureMapping));
					func.append("getIor();");
					args.append("litArgs_ior = dIor;");
				}
				if (options.litOptions.useSpecularityFactor) {
					decl.append("float dSpecularityFactor;");
					code.append(this._addMap("specularityFactor", "specularityFactorPS", options, litShader.chunks, textureMapping));
					func.append("getSpecularityFactor();");
					args.append("litArgs_specularityFactor = dSpecularityFactor;");
				}
				if (options.useSpecularColor) {
					code.append(this._addMap("specular", "specularPS", options, litShader.chunks, textureMapping, options.specularEncoding));
				} else {
					code.append("void getSpecularity() { dSpecularity = vec3(1); }");
				}
				code.append(this._addMap("gloss", "glossPS", options, litShader.chunks, textureMapping));
				func.append("getGlossiness();");
				func.append("getSpecularity();");
				args.append("litArgs_specularity = dSpecularity;");
				args.append("litArgs_gloss = dGlossiness;");
			} else {
				decl.append("vec3 dSpecularity = vec3(0.0);");
				decl.append("float dGlossiness = 0.0;");
			}
			if (options.aoDetail) {
				code.append(this._addMap("aoDetail", "aoDetailMapPS", options, litShader.chunks, textureMapping));
			}
			if (options.aoMap || options.aoVertexColor) {
				decl.append("float dAo;");
				code.append(this._addMap("ao", "aoPS", options, litShader.chunks, textureMapping));
				func.append("getAO();");
				args.append("litArgs_ao = dAo;");
			}
			decl.append("vec3 dEmission;");
			code.append(this._addMap("emissive", "emissivePS", options, litShader.chunks, textureMapping, options.emissiveEncoding));
			func.append("getEmission();");
			args.append("litArgs_emission = dEmission;");
			if (options.litOptions.useClearCoat) {
				decl.append("float ccSpecularity;");
				decl.append("float ccGlossiness;");
				decl.append("vec3 ccNormalW;");
				code.append(this._addMap("clearCoat", "clearCoatPS", options, litShader.chunks, textureMapping));
				code.append(this._addMap("clearCoatGloss", "clearCoatGlossPS", options, litShader.chunks, textureMapping));
				code.append(this._addMap("clearCoatNormal", "clearCoatNormalPS", options, litShader.chunks, textureMapping));
				func.append("getClearCoat();");
				func.append("getClearCoatGlossiness();");
				func.append("getClearCoatNormal();");
				args.append("litArgs_clearcoat_specularity = ccSpecularity;");
				args.append("litArgs_clearcoat_gloss = ccGlossiness;");
				args.append("litArgs_clearcoat_worldNormal = ccNormalW;");
			}
			if (options.lightMap || options.lightVertexColor) {
				const lightmapDir = options.dirLightMap && options.litOptions.useSpecular;
				const lightmapChunkPropName = lightmapDir ? 'lightmapDirPS' : 'lightmapSinglePS';
				decl.append("vec3 dLightmap;");
				if (lightmapDir) {
					decl.append("vec3 dLightmapDir;");
				}
				code.append(this._addMap("light", lightmapChunkPropName, options, litShader.chunks, textureMapping, options.lightMapEncoding));
				func.append("getLightMap();");
				args.append("litArgs_lightmap = dLightmap;");
				if (lightmapDir) {
					args.append("litArgs_lightmapDir = dLightmapDir;");
				}
			}
			if (code.code.indexOf('texture2DSRGB') !== -1 || code.code.indexOf('texture2DRGBM') !== -1 || code.code.indexOf('texture2DRGBE') !== -1) {
				code.prepend(litShader.chunks.textureSamplePS);
			}
		} else {
			const opacityShadowDither = options.litOptions.opacityShadowDither;
			if (options.litOptions.alphaTest || opacityShadowDither) {
				decl.append("float dAlpha;");
				code.append(this._addMap("opacity", "opacityPS", options, litShader.chunks, textureMapping));
				func.append("getOpacity();");
				args.append("litArgs_opacity = dAlpha;");
				if (options.litOptions.alphaTest) {
					code.append(litShader.chunks.alphaTestPS);
					func.append("alphaTest(dAlpha);");
				}
				if (opacityShadowDither !== DITHER_NONE) {
					if (opacityShadowDither === DITHER_BAYER8) decl.append(litShader.chunks.bayerPS);
					decl.append(`#define DITHER_${opacityShadowDither.toUpperCase()}\n`);
					decl.append(litShader.chunks.opacityDitherPS);
					func.append("opacityDither(dAlpha, 0.0);");
				}
			}
		}
		decl.append(litShader.chunks.litShaderArgsPS);
		code.append(`void evaluateFrontend() { \n${func.code}\n${args.code}\n }\n`);
		func.code = `evaluateFrontend();`;
		for (const texture in textureMapping) {
			decl.append(`uniform sampler2D ${textureMapping[texture]};`);
		}
		func.code = `\n${func.code.split('\n').map(l => `    ${l}`).join('\n')}\n\n`;
		litShader.generateFragmentShader(decl.code, code.code, func.code, lightingUv);
		return litShader.getDefinition();
	}
}
const standard = new ShaderGeneratorStandard();

const arraysEqual = (a, b) => {
	if (a.length !== b.length) {
		return false;
	}
	for (let i = 0; i < a.length; ++i) {
		if (a[i] !== b[i]) {
			return false;
		}
	}
	return true;
};
const notWhite = color => {
	return color.r !== 1 || color.g !== 1 || color.b !== 1;
};
const notBlack = color => {
	return color.r !== 0 || color.g !== 0 || color.b !== 0;
};
class StandardMaterialOptionsBuilder {
	constructor() {
		this._mapXForms = null;
	}
	updateMinRef(options, scene, stdMat, objDefs, pass, sortedLights) {
		this._updateSharedOptions(options, scene, stdMat, objDefs, pass);
		this._updateMinOptions(options, stdMat);
		this._updateUVOptions(options, stdMat, objDefs, true);
	}
	updateRef(options, scene, stdMat, objDefs, pass, sortedLights) {
		this._updateSharedOptions(options, scene, stdMat, objDefs, pass);
		this._updateEnvOptions(options, stdMat, scene);
		this._updateMaterialOptions(options, stdMat);
		if (pass === SHADER_FORWARDHDR) {
			if (options.litOptions.gamma) options.litOptions.gamma = GAMMA_SRGBHDR;
			options.litOptions.toneMap = TONEMAP_LINEAR;
		}
		options.litOptions.hasTangents = objDefs && (objDefs & SHADERDEF_TANGENTS) !== 0;
		this._updateLightOptions(options, scene, stdMat, objDefs, sortedLights);
		this._updateUVOptions(options, stdMat, objDefs, false);
	}
	_updateSharedOptions(options, scene, stdMat, objDefs, pass) {
		options.forceUv1 = stdMat.forceUv1;
		if (stdMat.userAttributes) {
			options.litOptions.userAttributes = Object.fromEntries(stdMat.userAttributes.entries());
		}
		options.litOptions.chunks = stdMat.chunks || {};
		options.litOptions.pass = pass;
		options.litOptions.alphaTest = stdMat.alphaTest > 0;
		options.litOptions.blendType = stdMat.blendType;
		options.litOptions.screenSpace = objDefs && (objDefs & SHADERDEF_SCREENSPACE) !== 0;
		options.litOptions.skin = objDefs && (objDefs & SHADERDEF_SKIN) !== 0;
		options.litOptions.useInstancing = objDefs && (objDefs & SHADERDEF_INSTANCING) !== 0;
		options.litOptions.useMorphPosition = objDefs && (objDefs & SHADERDEF_MORPH_POSITION) !== 0;
		options.litOptions.useMorphNormal = objDefs && (objDefs & SHADERDEF_MORPH_NORMAL) !== 0;
		options.litOptions.useMorphTextureBased = objDefs && (objDefs & SHADERDEF_MORPH_TEXTURE_BASED) !== 0;
		options.litOptions.nineSlicedMode = stdMat.nineSlicedMode || 0;
		if (scene.clusteredLightingEnabled && stdMat.useLighting) {
			options.litOptions.clusteredLightingEnabled = true;
			options.litOptions.clusteredLightingCookiesEnabled = scene.lighting.cookiesEnabled;
			options.litOptions.clusteredLightingShadowsEnabled = scene.lighting.shadowsEnabled;
			options.litOptions.clusteredLightingShadowType = scene.lighting.shadowType;
			options.litOptions.clusteredLightingAreaLightsEnabled = scene.lighting.areaLightsEnabled;
		} else {
			options.litOptions.clusteredLightingEnabled = false;
			options.litOptions.clusteredLightingCookiesEnabled = false;
			options.litOptions.clusteredLightingShadowsEnabled = false;
			options.litOptions.clusteredLightingAreaLightsEnabled = false;
		}
	}
	_updateUVOptions(options, stdMat, objDefs, minimalOptions) {
		let hasUv0 = false;
		let hasUv1 = false;
		let hasVcolor = false;
		if (objDefs) {
			hasUv0 = (objDefs & SHADERDEF_UV0) !== 0;
			hasUv1 = (objDefs & SHADERDEF_UV1) !== 0;
			hasVcolor = (objDefs & SHADERDEF_VCOLOR) !== 0;
		}
		options.litOptions.vertexColors = false;
		this._mapXForms = [];
		const uniqueTextureMap = {};
		for (const p in _matTex2D) {
			this._updateTexOptions(options, stdMat, p, hasUv0, hasUv1, hasVcolor, minimalOptions, uniqueTextureMap);
		}
		this._mapXForms = null;
		options.litOptions.lightMapEnabled = options.lightMap;
		options.litOptions.dirLightMapEnabled = options.dirLightMap;
		options.litOptions.useHeights = options.heightMap;
		options.litOptions.useNormals = options.normalMap;
		options.litOptions.useClearCoatNormals = options.clearCoatNormalMap;
		options.litOptions.useAo = options.aoMap || options.aoVertexColor;
		options.litOptions.diffuseMapEnabled = options.diffuseMap;
	}
	_updateTexOptions(options, stdMat, p, hasUv0, hasUv1, hasVcolor, minimalOptions, uniqueTextureMap) {
		const isOpacity = p === 'opacity';
		if (!minimalOptions || isOpacity) {
			const mname = p + 'Map';
			const vname = p + 'VertexColor';
			const vcname = p + 'VertexColorChannel';
			const cname = mname + 'Channel';
			const tname = mname + 'Transform';
			const uname = mname + 'Uv';
			const iname = mname + 'Identifier';
			if (p !== 'light') {
				options[mname] = false;
				options[iname] = undefined;
				options[cname] = '';
				options[tname] = 0;
				options[uname] = 0;
			}
			options[vname] = false;
			options[vcname] = '';
			if (isOpacity && stdMat.blendType === BLEND_NONE && stdMat.alphaTest === 0.0 && !stdMat.alphaToCoverage && stdMat.opacityDither === DITHER_NONE) {
				return;
			}
			if (p !== 'height' && stdMat[vname]) {
				if (hasVcolor) {
					options[vname] = stdMat[vname];
					options[vcname] = stdMat[vcname];
					options.litOptions.vertexColors = true;
				}
			}
			if (stdMat[mname]) {
				let allow = true;
				if (stdMat[uname] === 0 && !hasUv0) allow = false;
				if (stdMat[uname] === 1 && !hasUv1) allow = false;
				if (allow) {
					const mapId = stdMat[mname].id;
					let identifier = uniqueTextureMap[mapId];
					if (identifier === undefined) {
						uniqueTextureMap[mapId] = p;
						identifier = p;
					}
					options[mname] = !!stdMat[mname];
					options[iname] = identifier;
					options[tname] = this._getMapTransformID(stdMat.getUniform(tname), stdMat[uname]);
					options[cname] = stdMat[cname];
					options[uname] = stdMat[uname];
				}
			}
		}
	}
	_updateMinOptions(options, stdMat) {
		options.opacityTint = stdMat.opacity !== 1 && (stdMat.blendType !== BLEND_NONE || stdMat.opacityShadowDither !== DITHER_NONE);
		options.litOptions.opacityShadowDither = stdMat.opacityShadowDither;
		options.litOptions.lights = [];
	}
	_updateMaterialOptions(options, stdMat) {
		var _stdMat$diffuseMap, _stdMat$diffuseDetail, _stdMat$emissiveMap, _stdMat$lightMap;
		const diffuseTint = (stdMat.diffuseTint || !stdMat.diffuseMap && !stdMat.diffuseVertexColor) && notWhite(stdMat.diffuse);
		const useSpecular = !!(stdMat.useMetalness || stdMat.specularMap || stdMat.sphereMap || stdMat.cubeMap || notBlack(stdMat.specular) || stdMat.specularityFactor > 0 && stdMat.useMetalness || stdMat.enableGGXSpecular || stdMat.clearCoat > 0);
		const useSpecularColor = !stdMat.useMetalness || stdMat.useMetalnessSpecularColor;
		const specularTint = useSpecular && (stdMat.specularTint || !stdMat.specularMap && !stdMat.specularVertexColor) && notWhite(stdMat.specular);
		const specularityFactorTint = useSpecular && stdMat.useMetalnessSpecularColor && (stdMat.specularityFactorTint || stdMat.specularityFactor < 1 && !stdMat.specularityFactorMap);
		const emissiveTintColor = !stdMat.emissiveMap || notWhite(stdMat.emissive) && stdMat.emissiveTint;
		const emissiveTintIntensity = stdMat.emissiveIntensity !== 1;
		const isPackedNormalMap = stdMat.normalMap ? stdMat.normalMap.format === PIXELFORMAT_DXT5 || stdMat.normalMap.type === TEXTURETYPE_SWIZZLEGGGR : false;
		options.opacityTint = stdMat.opacity !== 1 && (stdMat.blendType !== BLEND_NONE || stdMat.alphaTest > 0 || stdMat.opacityDither !== DITHER_NONE) ? 1 : 0;
		options.ambientTint = stdMat.ambientTint;
		options.diffuseTint = diffuseTint ? 2 : 0;
		options.specularTint = specularTint ? 2 : 0;
		options.specularityFactorTint = specularityFactorTint ? 1 : 0;
		options.metalnessTint = stdMat.useMetalness && stdMat.metalness < 1 ? 1 : 0;
		options.glossTint = 1;
		options.emissiveTint = (emissiveTintColor ? 2 : 0) + (emissiveTintIntensity ? 1 : 0);
		options.diffuseEncoding = (_stdMat$diffuseMap = stdMat.diffuseMap) == null ? void 0 : _stdMat$diffuseMap.encoding;
		options.diffuseDetailEncoding = (_stdMat$diffuseDetail = stdMat.diffuseDetailMap) == null ? void 0 : _stdMat$diffuseDetail.encoding;
		options.emissiveEncoding = (_stdMat$emissiveMap = stdMat.emissiveMap) == null ? void 0 : _stdMat$emissiveMap.encoding;
		options.lightMapEncoding = (_stdMat$lightMap = stdMat.lightMap) == null ? void 0 : _stdMat$lightMap.encoding;
		options.packedNormal = isPackedNormalMap;
		options.refractionTint = stdMat.refraction !== 1.0 ? 1 : 0;
		options.refractionIndexTint = stdMat.refractionIndex !== 1.0 / 1.5 ? 1 : 0;
		options.thicknessTint = stdMat.useDynamicRefraction && stdMat.thickness !== 1.0 ? 1 : 0;
		options.specularEncoding = stdMat.specularEncoding || 'linear';
		options.sheenEncoding = stdMat.sheenEncoding || 'linear';
		options.aoMapUv = stdMat.aoUvSet;
		options.aoDetail = !!stdMat.aoMap;
		options.diffuseDetail = !!stdMat.diffuseMap;
		options.normalDetail = !!stdMat.normalMap;
		options.diffuseDetailMode = stdMat.diffuseDetailMode;
		options.aoDetailMode = stdMat.aoDetailMode;
		options.clearCoatTint = stdMat.clearCoat !== 1.0 ? 1 : 0;
		options.clearCoatGloss = !!stdMat.clearCoatGloss;
		options.clearCoatGlossTint = stdMat.clearCoatGloss !== 1.0 ? 1 : 0;
		options.iorTint = stdMat.refractionIndex !== 1.0 / 1.5 ? 1 : 0;
		options.iridescenceTint = stdMat.iridescence !== 1.0 ? 1 : 0;
		options.sheenTint = stdMat.useSheen && notWhite(stdMat.sheen) ? 2 : 0;
		options.sheenGlossTint = 1;
		options.glossInvert = stdMat.glossInvert;
		options.sheenGlossInvert = stdMat.sheenGlossInvert;
		options.clearCoatGlossInvert = stdMat.clearCoatGlossInvert;
		options.useSpecularColor = useSpecularColor;
		options.litOptions.separateAmbient = false;
		options.litOptions.useAmbientTint = stdMat.ambientTint;
		options.litOptions.customFragmentShader = stdMat.customFragmentShader;
		options.litOptions.pixelSnap = stdMat.pixelSnap;
		options.litOptions.shadingModel = stdMat.shadingModel;
		options.litOptions.ambientSH = !!stdMat.ambientSH;
		options.litOptions.fastTbn = stdMat.fastTbn;
		options.litOptions.twoSidedLighting = stdMat.twoSidedLighting;
		options.litOptions.occludeSpecular = stdMat.occludeSpecular;
		options.litOptions.occludeSpecularFloat = stdMat.occludeSpecularIntensity !== 1.0;
		options.litOptions.useMsdf = !!stdMat.msdfMap;
		options.litOptions.msdfTextAttribute = !!stdMat.msdfTextAttribute;
		options.litOptions.alphaToCoverage = stdMat.alphaToCoverage;
		options.litOptions.opacityFadesSpecular = stdMat.opacityFadesSpecular;
		options.litOptions.opacityDither = stdMat.opacityDither;
		options.litOptions.cubeMapProjection = stdMat.cubeMapProjection;
		options.litOptions.occludeDirect = stdMat.occludeDirect;
		options.litOptions.conserveEnergy = stdMat.conserveEnergy && stdMat.shadingModel !== SPECULAR_PHONG;
		options.litOptions.useSpecular = useSpecular;
		options.litOptions.useSpecularityFactor = (specularityFactorTint || !!stdMat.specularityFactorMap) && stdMat.useMetalnessSpecularColor;
		options.litOptions.enableGGXSpecular = stdMat.enableGGXSpecular;
		options.litOptions.fresnelModel = stdMat.fresnelModel;
		options.litOptions.useRefraction = (stdMat.refraction || !!stdMat.refractionMap) && (stdMat.useDynamicRefraction || !!options.litOptions.reflectionSource);
		options.litOptions.useClearCoat = !!stdMat.clearCoat;
		options.litOptions.useSheen = stdMat.useSheen;
		options.litOptions.useIridescence = stdMat.useIridescence && stdMat.iridescence !== 0.0;
		options.litOptions.useMetalness = stdMat.useMetalness;
		options.litOptions.useDynamicRefraction = stdMat.useDynamicRefraction;
	}
	_updateEnvOptions(options, stdMat, scene) {
		options.litOptions.fog = stdMat.useFog ? scene.fog : 'none';
		options.litOptions.gamma = stdMat.useGammaTonemap ? scene.gammaCorrection : GAMMA_NONE;
		options.litOptions.toneMap = stdMat.useGammaTonemap ? scene.toneMapping : -1;
		options.litOptions.fixSeams = stdMat.cubeMap ? stdMat.cubeMap.fixCubemapSeams : false;
		const isPhong = stdMat.shadingModel === SPECULAR_PHONG;
		let usingSceneEnv = false;
		if (stdMat.envAtlas && stdMat.cubeMap && !isPhong) {
			options.litOptions.reflectionSource = 'envAtlasHQ';
			options.litOptions.reflectionEncoding = stdMat.envAtlas.encoding;
			options.litOptions.reflectionCubemapEncoding = stdMat.cubeMap.encoding;
		} else if (stdMat.envAtlas && !isPhong) {
			options.litOptions.reflectionSource = 'envAtlas';
			options.litOptions.reflectionEncoding = stdMat.envAtlas.encoding;
		} else if (stdMat.cubeMap) {
			options.litOptions.reflectionSource = 'cubeMap';
			options.litOptions.reflectionEncoding = stdMat.cubeMap.encoding;
		} else if (stdMat.sphereMap) {
			options.litOptions.reflectionSource = 'sphereMap';
			options.litOptions.reflectionEncoding = stdMat.sphereMap.encoding;
		} else if (stdMat.useSkybox && scene.envAtlas && scene.skybox && !isPhong) {
			options.litOptions.reflectionSource = 'envAtlasHQ';
			options.litOptions.reflectionEncoding = scene.envAtlas.encoding;
			options.litOptions.reflectionCubemapEncoding = scene.skybox.encoding;
			usingSceneEnv = true;
		} else if (stdMat.useSkybox && scene.envAtlas && !isPhong) {
			options.litOptions.reflectionSource = 'envAtlas';
			options.litOptions.reflectionEncoding = scene.envAtlas.encoding;
			usingSceneEnv = true;
		} else if (stdMat.useSkybox && scene.skybox) {
			options.litOptions.reflectionSource = 'cubeMap';
			options.litOptions.reflectionEncoding = scene.skybox.encoding;
			usingSceneEnv = true;
		} else {
			options.litOptions.reflectionSource = null;
			options.litOptions.reflectionEncoding = null;
		}
		if (stdMat.ambientSH && !isPhong) {
			options.litOptions.ambientSource = 'ambientSH';
			options.litOptions.ambientEncoding = null;
		} else {
			const envAtlas = stdMat.envAtlas || (stdMat.useSkybox && scene.envAtlas ? scene.envAtlas : null);
			if (envAtlas && !isPhong) {
				options.litOptions.ambientSource = 'envAtlas';
				options.litOptions.ambientEncoding = envAtlas.encoding;
			} else {
				options.litOptions.ambientSource = 'constant';
				options.litOptions.ambientEncoding = null;
			}
		}
		options.litOptions.skyboxIntensity = usingSceneEnv && (scene.skyboxIntensity !== 1 || scene.physicalUnits);
		options.litOptions.useCubeMapRotation = usingSceneEnv && scene._skyboxRotationShaderInclude;
	}
	_updateLightOptions(options, scene, stdMat, objDefs, sortedLights) {
		options.lightMap = false;
		options.lightMapChannel = '';
		options.lightMapUv = 0;
		options.lightMapTransform = 0;
		options.litOptions.lightMapWithoutAmbient = false;
		options.dirLightMap = false;
		if (objDefs) {
			options.litOptions.noShadow = (objDefs & SHADERDEF_NOSHADOW) !== 0;
			if ((objDefs & SHADERDEF_LM) !== 0) {
				options.lightMapEncoding = scene.lightmapPixelFormat === PIXELFORMAT_RGBA8 ? 'rgbm' : 'linear';
				options.lightMap = true;
				options.lightMapChannel = 'rgb';
				options.lightMapUv = 1;
				options.lightMapTransform = 0;
				options.litOptions.lightMapWithoutAmbient = !stdMat.lightMap;
				if ((objDefs & SHADERDEF_DIRLM) !== 0) {
					options.dirLightMap = true;
				}
				if ((objDefs & SHADERDEF_LMAMBIENT) !== 0) {
					options.litOptions.lightMapWithoutAmbient = false;
				}
			}
		}
		if (stdMat.useLighting) {
			const lightsFiltered = [];
			const mask = objDefs ? objDefs >> 16 : MASK_AFFECT_DYNAMIC;
			options.litOptions.lightMaskDynamic = !!(mask & MASK_AFFECT_DYNAMIC);
			if (sortedLights) {
				LitMaterialOptionsBuilder.collectLights(LIGHTTYPE_DIRECTIONAL, sortedLights[LIGHTTYPE_DIRECTIONAL], lightsFiltered, mask);
				LitMaterialOptionsBuilder.collectLights(LIGHTTYPE_OMNI, sortedLights[LIGHTTYPE_OMNI], lightsFiltered, mask);
				LitMaterialOptionsBuilder.collectLights(LIGHTTYPE_SPOT, sortedLights[LIGHTTYPE_SPOT], lightsFiltered, mask);
			}
			options.litOptions.lights = lightsFiltered;
		} else {
			options.litOptions.lights = [];
		}
		if (options.litOptions.lights.length === 0) {
			options.litOptions.noShadow = true;
		}
	}
	_getMapTransformID(xform, uv) {
		if (!xform) return 0;
		let xforms = this._mapXForms[uv];
		if (!xforms) {
			xforms = [];
			this._mapXForms[uv] = xforms;
		}
		for (let i = 0; i < xforms.length; i++) {
			if (arraysEqual(xforms[i][0].value, xform[0].value) && arraysEqual(xforms[i][1].value, xform[1].value)) {
				return i + 1;
			}
		}
		return xforms.push(xform);
	}
}

function _textureParameter(name, channel = true, vertexColor = true) {
	const result = {};
	result[`${name}Map`] = 'texture';
	result[`${name}MapTiling`] = 'vec2';
	result[`${name}MapOffset`] = 'vec2';
	result[`${name}MapRotation`] = 'number';
	result[`${name}MapUv`] = 'number';
	if (channel) {
		result[`${name}MapChannel`] = 'string';
		if (vertexColor) {
			result[`${name}VertexColor`] = 'boolean';
			result[`${name}VertexColorChannel`] = 'string';
		}
	}
	return result;
}
const standardMaterialParameterTypes = _extends({
	name: 'string',
	chunks: 'chunks',
	mappingFormat: 'string',
	_engine: 'boolean',
	ambient: 'rgb',
	ambientTint: 'boolean'
}, _textureParameter('ao'), _textureParameter('aoDetail', true, false), {
	aoDetailMode: 'string',
	diffuse: 'rgb',
	diffuseTint: 'boolean'
}, _textureParameter('diffuse'), _textureParameter('diffuseDetail', true, false), {
	diffuseDetailMode: 'string',
	specular: 'rgb',
	specularTint: 'boolean'
}, _textureParameter('specular'), {
	occludeSpecular: 'enum:occludeSpecular',
	specularityFactor: 'number',
	specularityFactorTint: 'boolean'
}, _textureParameter('specularityFactor'), {
	useMetalness: 'boolean',
	metalness: 'number',
	enableGGXSpecular: 'boolean',
	anisotropy: 'number',
	metalnessTint: 'boolean'
}, _textureParameter('metalness'), {
	useMetalnessSpecularColor: 'boolean',
	conserveEnergy: 'boolean',
	shininess: 'number',
	gloss: 'number',
	glossInvert: 'boolean'
}, _textureParameter('gloss'), {
	clearCoat: 'number'
}, _textureParameter('clearCoat'), {
	clearCoatGloss: 'number',
	clearCoatGlossInvert: 'boolean'
}, _textureParameter('clearCoatGloss'), {
	clearCoatBumpiness: 'number'
}, _textureParameter('clearCoatNormal', false), {
	useSheen: 'boolean',
	sheen: 'rgb',
	sheenTint: 'boolean'
}, _textureParameter('sheen'), {
	sheenGloss: 'number',
	sheenGlossTint: 'boolean',
	sheenGlossInvert: 'boolean'
}, _textureParameter('sheenGloss'), {
	fresnelModel: 'number',
	emissive: 'rgb',
	emissiveTint: 'boolean'
}, _textureParameter('emissive'), {
	emissiveIntensity: 'number'
}, _textureParameter('normal', false), {
	bumpiness: 'number'
}, _textureParameter('normalDetail', false), {
	normalDetailMapBumpiness: 'number'
}, _textureParameter('height', true, false), {
	heightMapFactor: 'number',
	alphaToCoverage: 'boolean',
	alphaTest: 'number',
	alphaFade: 'number',
	opacity: 'number'
}, _textureParameter('opacity'), {
	opacityFadesSpecular: 'boolean',
	opacityDither: 'string',
	opacityShadowDither: 'string',
	reflectivity: 'number',
	refraction: 'number',
	refractionTint: 'boolean'
}, _textureParameter('refraction'), {
	refractionIndex: 'number',
	thickness: 'number',
	thicknessTint: 'boolean'
}, _textureParameter('thickness'), {
	attenuation: 'rgb',
	attenuationDistance: 'number',
	useDynamicRefraction: 'boolean',
	sphereMap: 'texture',
	cubeMap: 'cubemap',
	cubeMapProjection: 'number',
	cubeMapProjectionBox: 'boundingbox',
	useIridescence: 'boolean',
	iridescence: 'number',
	iridescenceTint: 'boolean'
}, _textureParameter('iridescence'), {
	iridescenceThicknessTint: 'boolean',
	iridescenceThicknessMin: 'number',
	iridescenceThicknessMax: 'number',
	iridescenceRefractionIndex: 'number'
}, _textureParameter('iridescenceThickness'), _textureParameter('light'), {
	depthTest: 'boolean',
	depthFunc: 'enum:depthFunc',
	depthWrite: 'boolean',
	depthBias: 'number',
	slopeDepthBias: 'number',
	cull: 'enum:cull',
	blendType: 'enum:blendType',
	shadingModel: 'enum:shadingModel',
	useFog: 'boolean',
	useLighting: 'boolean',
	useSkybox: 'boolean',
	useGammaTonemap: 'boolean',
	envAtlas: 'texture',
	twoSidedLighting: 'boolean'
});
const standardMaterialTextureParameters = [];
for (const key in standardMaterialParameterTypes) {
	const type = standardMaterialParameterTypes[key];
	if (type === 'texture') {
		standardMaterialTextureParameters.push(key);
	}
}
const standardMaterialCubemapParameters = [];
for (const key in standardMaterialParameterTypes) {
	const type = standardMaterialParameterTypes[key];
	if (type === 'cubemap') {
		standardMaterialCubemapParameters.push(key);
	}
}
const standardMaterialRemovedParameters = {
	aoMapVertexColor: 'boolean',
	diffuseMapTint: 'boolean',
	diffuseMapVertexColor: 'boolean',
	emissiveMapTint: 'boolean',
	emissiveMapVertexColor: 'boolean',
	glossMapVertexColor: 'boolean',
	metalnessMapVertexColor: 'boolean',
	opacityMapVertexColor: 'boolean',
	specularAntialias: 'boolean',
	specularMapTint: 'boolean',
	specularMapVertexColor: 'boolean'
};

const _props = {};
const _uniforms = {};
let _params = new Set();
class StandardMaterial extends Material {
	constructor() {
		super();
		this.userAttributes = new Map();
		this._dirtyShader = true;
		this._assetReferences = {};
		this._activeParams = new Set();
		this._activeLightingParams = new Set();
		this.shaderOptBuilder = new StandardMaterialOptionsBuilder();
		this.reset();
	}
	reset() {
		Object.keys(_props).forEach(name => {
			this[`_${name}`] = _props[name].value();
		});
		this._chunks = {};
		this._uniformCache = {};
	}
	set shader(shader) {}
	get shader() {
		return null;
	}
	set chunks(value) {
		this._dirtyShader = true;
		this._chunks = value;
	}
	get chunks() {
		this._dirtyShader = true;
		return this._chunks;
	}
	copy(source) {
		super.copy(source);
		Object.keys(_props).forEach(k => {
			this[k] = source[k];
		});
		for (const p in source._chunks) {
			if (source._chunks.hasOwnProperty(p)) this._chunks[p] = source._chunks[p];
		}
		return this;
	}
	setAttribute(name, semantic) {
		this.userAttributes.set(semantic, name);
	}
	_setParameter(name, value) {
		_params.add(name);
		this.setParameter(name, value);
	}
	_setParameters(parameters) {
		parameters.forEach(v => {
			this._setParameter(v.name, v.value);
		});
	}
	_processParameters(paramsName) {
		const prevParams = this[paramsName];
		prevParams.forEach(param => {
			if (!_params.has(param)) {
				delete this.parameters[param];
			}
		});
		this[paramsName] = _params;
		_params = prevParams;
		_params.clear();
	}
	_updateMap(p) {
		const mname = p + 'Map';
		const map = this[mname];
		if (map) {
			this._setParameter('texture_' + mname, map);
			const tname = mname + 'Transform';
			const uniform = this.getUniform(tname);
			if (uniform) {
				this._setParameters(uniform);
			}
		}
	}
	_allocUniform(name, allocFunc) {
		let uniform = this._uniformCache[name];
		if (!uniform) {
			uniform = allocFunc();
			this._uniformCache[name] = uniform;
		}
		return uniform;
	}
	getUniform(name, device, scene) {
		return _uniforms[name](this, device, scene);
	}
	updateUniforms(device, scene) {
		const getUniform = name => {
			return this.getUniform(name, device, scene);
		};
		this._setParameter('material_ambient', getUniform('ambient'));
		if (!this.diffuseMap || this.diffuseTint) {
			this._setParameter('material_diffuse', getUniform('diffuse'));
		}
		if (this.useMetalness) {
			if (!this.metalnessMap || this.metalness < 1) {
				this._setParameter('material_metalness', this.metalness);
			}
			if (!this.specularMap || this.specularTint) {
				this._setParameter('material_specular', getUniform('specular'));
			}
			if (!this.specularityFactorMap || this.specularityFactorTint) {
				this._setParameter('material_specularityFactor', this.specularityFactor);
			}
			if (!this.sheenMap || this.sheenTint) {
				this._setParameter('material_sheen', getUniform('sheen'));
			}
			if (!this.sheenGlossMap || this.sheenGlossTint) {
				this._setParameter('material_sheenGloss', this.sheenGloss);
			}
			this._setParameter('material_refractionIndex', this.refractionIndex);
		} else {
			if (!this.specularMap || this.specularTint) {
				this._setParameter('material_specular', getUniform('specular'));
			}
		}
		if (this.enableGGXSpecular) {
			this._setParameter('material_anisotropy', this.anisotropy);
		}
		if (this.clearCoat > 0) {
			this._setParameter('material_clearCoat', this.clearCoat);
			this._setParameter('material_clearCoatGloss', this.clearCoatGloss);
			this._setParameter('material_clearCoatBumpiness', this.clearCoatBumpiness);
		}
		this._setParameter('material_gloss', getUniform('gloss'));
		if (!this.emissiveMap || this.emissiveTint) {
			this._setParameter('material_emissive', getUniform('emissive'));
		}
		if (this.emissiveIntensity !== 1) {
			this._setParameter('material_emissiveIntensity', this.emissiveIntensity);
		}
		if (this.refraction > 0) {
			this._setParameter('material_refraction', this.refraction);
		}
		if (this.useDynamicRefraction) {
			this._setParameter('material_thickness', this.thickness);
			this._setParameter('material_attenuation', getUniform('attenuation'));
			this._setParameter('material_invAttenuationDistance', this.attenuationDistance === 0 ? 0 : 1.0 / this.attenuationDistance);
		}
		if (this.useIridescence) {
			this._setParameter('material_iridescence', this.iridescence);
			this._setParameter('material_iridescenceRefractionIndex', this.iridescenceRefractionIndex);
			this._setParameter('material_iridescenceThicknessMin', this.iridescenceThicknessMin);
			this._setParameter('material_iridescenceThicknessMax', this.iridescenceThicknessMax);
		}
		this._setParameter('material_opacity', this.opacity);
		if (this.opacityFadesSpecular === false) {
			this._setParameter('material_alphaFade', this.alphaFade);
		}
		if (this.occludeSpecular) {
			this._setParameter('material_occludeSpecularIntensity', this.occludeSpecularIntensity);
		}
		if (this.cubeMapProjection === CUBEPROJ_BOX) {
			this._setParameter(getUniform('cubeMapProjectionBox'));
		}
		for (const p in _matTex2D) {
			this._updateMap(p);
		}
		if (this.ambientSH) {
			this._setParameter('ambientSH[0]', this.ambientSH);
		}
		if (this.normalMap) {
			this._setParameter('material_bumpiness', this.bumpiness);
		}
		if (this.normalMap && this.normalDetailMap) {
			this._setParameter('material_normalDetailMapBumpiness', this.normalDetailMapBumpiness);
		}
		if (this.heightMap) {
			this._setParameter('material_heightMapFactor', getUniform('heightMapFactor'));
		}
		const isPhong = this.shadingModel === SPECULAR_PHONG;
		if (this.envAtlas && this.cubeMap && !isPhong) {
			this._setParameter('texture_envAtlas', this.envAtlas);
			this._setParameter('texture_cubeMap', this.cubeMap);
		} else if (this.envAtlas && !isPhong) {
			this._setParameter('texture_envAtlas', this.envAtlas);
		} else if (this.cubeMap) {
			this._setParameter('texture_cubeMap', this.cubeMap);
		} else if (this.sphereMap) {
			this._setParameter('texture_sphereMap', this.sphereMap);
		}
		this._setParameter('material_reflectivity', this.reflectivity);
		this._processParameters('_activeParams');
		if (this._dirtyShader) {
			this.clearVariants();
		}
	}
	updateEnvUniforms(device, scene) {
		const isPhong = this.shadingModel === SPECULAR_PHONG;
		const hasLocalEnvOverride = this.envAtlas && !isPhong || this.cubeMap || this.sphereMap;
		if (!hasLocalEnvOverride && this.useSkybox) {
			if (scene.envAtlas && scene.skybox && !isPhong) {
				this._setParameter('texture_envAtlas', scene.envAtlas);
				this._setParameter('texture_cubeMap', scene.skybox);
			} else if (scene.envAtlas && !isPhong) {
				this._setParameter('texture_envAtlas', scene.envAtlas);
			} else if (scene.skybox) {
				this._setParameter('texture_cubeMap', scene.skybox);
			}
		}
		this._processParameters('_activeLightingParams');
	}
	getShaderVariant(device, scene, objDefs, unused, pass, sortedLights, viewUniformFormat, viewBindGroupFormat, vertexFormat) {
		this.updateEnvUniforms(device, scene);
		const shaderPassInfo = ShaderPass.get(device).getByIndex(pass);
		const minimalOptions = pass === SHADER_DEPTH || pass === SHADER_PICK || pass === SHADER_PREPASS_VELOCITY || shaderPassInfo.isShadow;
		let options = minimalOptions ? standard.optionsContextMin : standard.optionsContext;
		if (minimalOptions) this.shaderOptBuilder.updateMinRef(options, scene, this, objDefs, pass, sortedLights);else this.shaderOptBuilder.updateRef(options, scene, this, objDefs, pass, sortedLights);
		if (this.onUpdateShader) {
			options = this.onUpdateShader(options);
		}
		const processingOptions = new ShaderProcessorOptions(viewUniformFormat, viewBindGroupFormat, vertexFormat);
		const library = getProgramLibrary(device);
		library.register('standard', standard);
		const shader = library.getProgram('standard', options, processingOptions, this.userId);
		this._dirtyShader = false;
		return shader;
	}
	destroy() {
		for (const asset in this._assetReferences) {
			this._assetReferences[asset]._unbind();
		}
		this._assetReferences = null;
		super.destroy();
	}
}
StandardMaterial.TEXTURE_PARAMETERS = standardMaterialTextureParameters;
StandardMaterial.CUBEMAP_PARAMETERS = standardMaterialCubemapParameters;
const defineUniform = (name, getUniformFunc) => {
	_uniforms[name] = getUniformFunc;
};
const definePropInternal = (name, constructorFunc, setterFunc, getterFunc) => {
	Object.defineProperty(StandardMaterial.prototype, name, {
		get: getterFunc || function () {
			return this[`_${name}`];
		},
		set: setterFunc
	});
	_props[name] = {
		value: constructorFunc
	};
};
const defineValueProp = prop => {
	const internalName = `_${prop.name}`;
	const dirtyShaderFunc = prop.dirtyShaderFunc || (() => true);
	const setterFunc = function setterFunc(value) {
		const oldValue = this[internalName];
		if (oldValue !== value) {
			this._dirtyShader = this._dirtyShader || dirtyShaderFunc(oldValue, value);
			this[internalName] = value;
		}
	};
	definePropInternal(prop.name, () => prop.defaultValue, setterFunc, prop.getterFunc);
};
const defineAggProp = prop => {
	const internalName = `_${prop.name}`;
	const dirtyShaderFunc = prop.dirtyShaderFunc || (() => true);
	const setterFunc = function setterFunc(value) {
		const oldValue = this[internalName];
		if (!oldValue.equals(value)) {
			this._dirtyShader = this._dirtyShader || dirtyShaderFunc(oldValue, value);
			this[internalName] = oldValue.copy(value);
		}
	};
	definePropInternal(prop.name, () => prop.defaultValue.clone(), setterFunc, prop.getterFunc);
};
const defineProp = prop => {
	return prop.defaultValue && prop.defaultValue.clone ? defineAggProp(prop) : defineValueProp(prop);
};
function _defineTex2D(name, channel = "rgb", vertexColor = true, uv = 0) {
	_matTex2D[name] = channel.length || -1;
	defineProp({
		name: `${name}Map`,
		defaultValue: null,
		dirtyShaderFunc: (oldValue, newValue) => {
			return !!oldValue !== !!newValue || oldValue && (oldValue.type !== newValue.type || oldValue.fixCubemapSeams !== newValue.fixCubemapSeams || oldValue.format !== newValue.format);
		}
	});
	defineProp({
		name: `${name}MapTiling`,
		defaultValue: new Vec2(1, 1)
	});
	defineProp({
		name: `${name}MapOffset`,
		defaultValue: new Vec2(0, 0)
	});
	defineProp({
		name: `${name}MapRotation`,
		defaultValue: 0
	});
	defineProp({
		name: `${name}MapUv`,
		defaultValue: uv
	});
	if (channel) {
		defineProp({
			name: `${name}MapChannel`,
			defaultValue: channel
		});
		if (vertexColor) {
			defineProp({
				name: `${name}VertexColor`,
				defaultValue: false
			});
			defineProp({
				name: `${name}VertexColorChannel`,
				defaultValue: channel
			});
		}
	}
	const mapTiling = `${name}MapTiling`;
	const mapOffset = `${name}MapOffset`;
	const mapRotation = `${name}MapRotation`;
	const mapTransform = `${name}MapTransform`;
	defineUniform(mapTransform, (material, device, scene) => {
		const tiling = material[mapTiling];
		const offset = material[mapOffset];
		const rotation = material[mapRotation];
		if (tiling.x === 1 && tiling.y === 1 && offset.x === 0 && offset.y === 0 && rotation === 0) {
			return null;
		}
		const uniform = material._allocUniform(mapTransform, () => {
			return [{
				name: `texture_${mapTransform}0`,
				value: new Float32Array(3)
			}, {
				name: `texture_${mapTransform}1`,
				value: new Float32Array(3)
			}];
		});
		const cr = Math.cos(rotation * math.DEG_TO_RAD);
		const sr = Math.sin(rotation * math.DEG_TO_RAD);
		const uniform0 = uniform[0].value;
		uniform0[0] = cr * tiling.x;
		uniform0[1] = -sr * tiling.y;
		uniform0[2] = offset.x;
		const uniform1 = uniform[1].value;
		uniform1[0] = sr * tiling.x;
		uniform1[1] = cr * tiling.y;
		uniform1[2] = 1.0 - tiling.y - offset.y;
		return uniform;
	});
}
function _defineColor(name, defaultValue) {
	defineProp({
		name: name,
		defaultValue: defaultValue,
		getterFunc: function () {
			this._dirtyShader = true;
			return this[`_${name}`];
		}
	});
	defineUniform(name, (material, device, scene) => {
		const uniform = material._allocUniform(name, () => new Float32Array(3));
		const color = material[name];
		const gamma = material.useGammaTonemap && scene.gammaCorrection;
		if (gamma) {
			uniform[0] = Math.pow(color.r, 2.2);
			uniform[1] = Math.pow(color.g, 2.2);
			uniform[2] = Math.pow(color.b, 2.2);
		} else {
			uniform[0] = color.r;
			uniform[1] = color.g;
			uniform[2] = color.b;
		}
		return uniform;
	});
}
function _defineFloat(name, defaultValue, getUniformFunc) {
	defineProp({
		name: name,
		defaultValue: defaultValue,
		dirtyShaderFunc: (oldValue, newValue) => {
			return (oldValue === 0 || oldValue === 1) !== (newValue === 0 || newValue === 1);
		}
	});
	defineUniform(name, getUniformFunc);
}
function _defineObject(name, getUniformFunc) {
	defineProp({
		name: name,
		defaultValue: null,
		dirtyShaderFunc: (oldValue, newValue) => {
			return !!oldValue === !!newValue;
		}
	});
	defineUniform(name, getUniformFunc);
}
function _defineFlag(name, defaultValue) {
	defineProp({
		name: name,
		defaultValue: defaultValue
	});
}
function _defineMaterialProps() {
	_defineColor('ambient', new Color(0.7, 0.7, 0.7));
	_defineColor('diffuse', new Color(1, 1, 1));
	_defineColor('specular', new Color(0, 0, 0));
	_defineColor('emissive', new Color(0, 0, 0));
	_defineColor('sheen', new Color(1, 1, 1));
	_defineColor('attenuation', new Color(1, 1, 1));
	_defineFloat('emissiveIntensity', 1);
	_defineFloat('specularityFactor', 1);
	_defineFloat('sheenGloss', 0.0);
	_defineFloat('gloss', 0.25, (material, device, scene) => {
		return material.shadingModel === SPECULAR_PHONG ? Math.pow(2, material.gloss * 11) : material.gloss;
	});
	_defineFloat('heightMapFactor', 1, (material, device, scene) => {
		return material.heightMapFactor * 0.025;
	});
	_defineFloat('opacity', 1);
	_defineFloat('alphaFade', 1);
	_defineFloat('alphaTest', 0);
	_defineFloat('bumpiness', 1);
	_defineFloat('normalDetailMapBumpiness', 1);
	_defineFloat('reflectivity', 1);
	_defineFloat('occludeSpecularIntensity', 1);
	_defineFloat('refraction', 0);
	_defineFloat('refractionIndex', 1.0 / 1.5);
	_defineFloat('thickness', 0);
	_defineFloat('attenuationDistance', 0);
	_defineFloat('metalness', 1);
	_defineFloat('anisotropy', 0);
	_defineFloat('clearCoat', 0);
	_defineFloat('clearCoatGloss', 1);
	_defineFloat('clearCoatBumpiness', 1);
	_defineFloat('aoUvSet', 0, null);
	_defineFloat('iridescence', 0);
	_defineFloat('iridescenceRefractionIndex', 1.0 / 1.5);
	_defineFloat('iridescenceThicknessMin', 0);
	_defineFloat('iridescenceThicknessMax', 0);
	_defineObject('ambientSH');
	_defineObject('cubeMapProjectionBox', (material, device, scene) => {
		const uniform = material._allocUniform('cubeMapProjectionBox', () => {
			return [{
				name: 'envBoxMin',
				value: new Float32Array(3)
			}, {
				name: 'envBoxMax',
				value: new Float32Array(3)
			}];
		});
		const bboxMin = material.cubeMapProjectionBox.getMin();
		const minUniform = uniform[0].value;
		minUniform[0] = bboxMin.x;
		minUniform[1] = bboxMin.y;
		minUniform[2] = bboxMin.z;
		const bboxMax = material.cubeMapProjectionBox.getMax();
		const maxUniform = uniform[1].value;
		maxUniform[0] = bboxMax.x;
		maxUniform[1] = bboxMax.y;
		maxUniform[2] = bboxMax.z;
		return uniform;
	});
	_defineFlag('ambientTint', false);
	_defineFlag('diffuseTint', false);
	_defineFlag('specularTint', false);
	_defineFlag('specularityFactorTint', false);
	_defineFlag('emissiveTint', false);
	_defineFlag('fastTbn', false);
	_defineFlag('useMetalness', false);
	_defineFlag('useMetalnessSpecularColor', false);
	_defineFlag('useSheen', false);
	_defineFlag('enableGGXSpecular', false);
	_defineFlag('occludeDirect', false);
	_defineFlag('normalizeNormalMap', true);
	_defineFlag('conserveEnergy', true);
	_defineFlag('opacityFadesSpecular', true);
	_defineFlag('occludeSpecular', SPECOCC_AO);
	_defineFlag('shadingModel', SPECULAR_BLINN);
	_defineFlag('fresnelModel', FRESNEL_SCHLICK);
	_defineFlag('useDynamicRefraction', false);
	_defineFlag('cubeMapProjection', CUBEPROJ_NONE);
	_defineFlag('customFragmentShader', null);
	_defineFlag('useFog', true);
	_defineFlag('useLighting', true);
	_defineFlag('useGammaTonemap', true);
	_defineFlag('useSkybox', true);
	_defineFlag('forceUv1', false);
	_defineFlag('pixelSnap', false);
	_defineFlag('twoSidedLighting', false);
	_defineFlag('nineSlicedMode', undefined);
	_defineFlag('msdfTextAttribute', false);
	_defineFlag('useIridescence', false);
	_defineFlag('glossInvert', false);
	_defineFlag('sheenGlossInvert', false);
	_defineFlag('clearCoatGlossInvert', false);
	_defineFlag('opacityDither', DITHER_NONE);
	_defineFlag('opacityShadowDither', DITHER_NONE);
	_defineTex2D('diffuse');
	_defineTex2D('specular');
	_defineTex2D('emissive');
	_defineTex2D('thickness', 'g');
	_defineTex2D('specularityFactor', 'g');
	_defineTex2D('normal', '');
	_defineTex2D('metalness', 'g');
	_defineTex2D('gloss', 'g');
	_defineTex2D('opacity', 'a');
	_defineTex2D('refraction', 'g');
	_defineTex2D('height', 'g', false);
	_defineTex2D('ao', 'g');
	_defineTex2D('light', 'rgb', true, 1);
	_defineTex2D('msdf', '');
	_defineTex2D('diffuseDetail', 'rgb', false);
	_defineTex2D('normalDetail', '');
	_defineTex2D('aoDetail', 'g', false);
	_defineTex2D('clearCoat', 'g');
	_defineTex2D('clearCoatGloss', 'g');
	_defineTex2D('clearCoatNormal', '');
	_defineTex2D('sheen', 'rgb');
	_defineTex2D('sheenGloss', 'g');
	_defineTex2D('iridescence', 'g');
	_defineTex2D('iridescenceThickness', 'g');
	_defineFlag('diffuseDetailMode', DETAILMODE_MUL);
	_defineFlag('aoDetailMode', DETAILMODE_MUL);
	_defineObject('cubeMap');
	_defineObject('sphereMap');
	_defineObject('envAtlas');
	const getterFunc = function getterFunc() {
		return this._prefilteredCubemaps;
	};
	const setterFunc = function setterFunc(value) {
		const cubemaps = this._prefilteredCubemaps;
		value = value || [];
		let changed = false;
		let complete = true;
		for (let i = 0; i < 6; ++i) {
			const v = value[i] || null;
			if (cubemaps[i] !== v) {
				cubemaps[i] = v;
				changed = true;
			}
			complete = complete && !!cubemaps[i];
		}
		if (changed) {
			if (complete) {
				this.envAtlas = EnvLighting.generatePrefilteredAtlas(cubemaps, {
					target: this.envAtlas
				});
			} else {
				if (this.envAtlas) {
					this.envAtlas.destroy();
					this.envAtlas = null;
				}
			}
			this._dirtyShader = true;
		}
	};
	const empty = [null, null, null, null, null, null];
	definePropInternal('prefilteredCubemaps', () => empty.slice(), setterFunc, getterFunc);
}
_defineMaterialProps();

new Vec3(1, 1, 1);
new Vec3(40, 0, 0);

class ShadowMap {
	constructor(texture, targets) {
		this.texture = texture;
		this.cached = false;
		this.renderTargets = targets;
	}
	destroy() {
		if (this.texture) {
			this.texture.destroy();
			this.texture = null;
		}
		const targets = this.renderTargets;
		for (let i = 0; i < targets.length; i++) {
			targets[i].destroy();
		}
		this.renderTargets.length = 0;
	}
	static getShadowFormat(device, shadowType) {
		if (shadowType === SHADOW_VSM32) {
			return PIXELFORMAT_RGBA32F;
		} else if (shadowType === SHADOW_VSM16) {
			return PIXELFORMAT_RGBA16F;
		} else if (shadowType === SHADOW_PCF5) {
			return PIXELFORMAT_DEPTH;
		} else if ((shadowType === SHADOW_PCF1 || shadowType === SHADOW_PCF3) && device.supportsDepthShadow) {
			return PIXELFORMAT_DEPTH;
		} else if (shadowType === SHADOW_PCSS && !device.isWebGL1) {
			return PIXELFORMAT_R32F;
		}
		return PIXELFORMAT_RGBA8;
	}
	static getShadowFiltering(device, shadowType) {
		if ((shadowType === SHADOW_PCF1 || shadowType === SHADOW_PCF3 || shadowType === SHADOW_PCSS) && !device.supportsDepthShadow) {
			return FILTER_NEAREST;
		} else if (shadowType === SHADOW_VSM32) {
			return device.extTextureFloatLinear ? FILTER_LINEAR : FILTER_NEAREST;
		} else if (shadowType === SHADOW_VSM16) {
			return device.extTextureHalfFloatLinear ? FILTER_LINEAR : FILTER_NEAREST;
		}
		return FILTER_LINEAR;
	}
	static create(device, light) {
		let shadowMap = null;
		if (light._type === LIGHTTYPE_OMNI) {
			shadowMap = this.createCubemap(device, light._shadowResolution, light._shadowType);
		} else {
			shadowMap = this.create2dMap(device, light._shadowResolution, light._shadowType);
		}
		return shadowMap;
	}
	static createAtlas(device, resolution, shadowType) {
		const shadowMap = this.create2dMap(device, resolution, shadowType);
		const targets = shadowMap.renderTargets;
		const rt = targets[0];
		for (let i = 0; i < 5; i++) {
			targets.push(rt);
		}
		return shadowMap;
	}
	static create2dMap(device, size, shadowType) {
		const format = this.getShadowFormat(device, shadowType);
		const filter = this.getShadowFiltering(device, shadowType);
		const texture = new Texture(device, {
			format: format,
			width: size,
			height: size,
			mipmaps: false,
			minFilter: filter,
			magFilter: filter,
			addressU: ADDRESS_CLAMP_TO_EDGE,
			addressV: ADDRESS_CLAMP_TO_EDGE,
			name: 'ShadowMap2D'
		});
		let target = null;
		if (shadowType === SHADOW_PCF5 || (shadowType === SHADOW_PCF1 || shadowType === SHADOW_PCF3) && device.supportsDepthShadow) {
			texture.compareOnRead = true;
			texture.compareFunc = FUNC_LESS;
			target = new RenderTarget({
				depthBuffer: texture
			});
		} else {
			target = new RenderTarget({
				colorBuffer: texture,
				depth: true
			});
		}
		if (device.isWebGPU) {
			target.flipY = true;
		}
		return new ShadowMap(texture, [target]);
	}
	static createCubemap(device, size, shadowType) {
		const format = shadowType === SHADOW_PCSS && !device.isWebGL1 ? PIXELFORMAT_R32F : PIXELFORMAT_RGBA8;
		const cubemap = new Texture(device, {
			format: format,
			width: size,
			height: size,
			cubemap: true,
			mipmaps: false,
			minFilter: FILTER_NEAREST,
			magFilter: FILTER_NEAREST,
			addressU: ADDRESS_CLAMP_TO_EDGE,
			addressV: ADDRESS_CLAMP_TO_EDGE,
			name: 'ShadowMapCube'
		});
		const targets = [];
		for (let i = 0; i < 6; i++) {
			const target = new RenderTarget({
				colorBuffer: cubemap,
				face: i,
				depth: true
			});
			targets.push(target);
		}
		return new ShadowMap(cubemap, targets);
	}
}

const _tempArray = [];
const _tempArray2 = [];
const _viewport$3 = new Vec4();
const _scissor = new Vec4();
class Slot {
	constructor(rect) {
		this.size = Math.floor(rect.w * 1024);
		this.used = false;
		this.lightId = -1;
		this.rect = rect;
	}
}
class LightTextureAtlas {
	constructor(device) {
		this.device = device;
		this.version = 1;
		this.shadowAtlasResolution = 2048;
		this.shadowAtlas = null;
		this.shadowEdgePixels = 3;
		this.cookieAtlasResolution = 4;
		this.cookieAtlas = new Texture(this.device, {
			name: 'CookieAtlas',
			width: this.cookieAtlasResolution,
			height: this.cookieAtlasResolution,
			format: PIXELFORMAT_RGBA8,
			cubemap: false,
			mipmaps: false,
			minFilter: FILTER_NEAREST,
			magFilter: FILTER_NEAREST,
			addressU: ADDRESS_CLAMP_TO_EDGE,
			addressV: ADDRESS_CLAMP_TO_EDGE
		});
		this.cookieRenderTarget = new RenderTarget({
			colorBuffer: this.cookieAtlas,
			depth: false,
			flipY: true
		});
		this.slots = [];
		this.atlasSplit = [];
		this.cubeSlotsOffsets = [new Vec2(0, 0), new Vec2(0, 1), new Vec2(1, 0), new Vec2(1, 1), new Vec2(2, 0), new Vec2(2, 1)];
		this.scissorVec = new Vec4();
		this.allocateShadowAtlas(1);
		this.allocateCookieAtlas(1);
		this.allocateUniforms();
	}
	destroy() {
		this.destroyShadowAtlas();
		this.destroyCookieAtlas();
	}
	destroyShadowAtlas() {
		var _this$shadowAtlas;
		(_this$shadowAtlas = this.shadowAtlas) == null || _this$shadowAtlas.destroy();
		this.shadowAtlas = null;
	}
	destroyCookieAtlas() {
		var _this$cookieAtlas, _this$cookieRenderTar;
		(_this$cookieAtlas = this.cookieAtlas) == null || _this$cookieAtlas.destroy();
		this.cookieAtlas = null;
		(_this$cookieRenderTar = this.cookieRenderTarget) == null || _this$cookieRenderTar.destroy();
		this.cookieRenderTarget = null;
	}
	allocateShadowAtlas(resolution) {
		if (!this.shadowAtlas || this.shadowAtlas.texture.width !== resolution) {
			this.version++;
			this.destroyShadowAtlas();
			this.shadowAtlas = ShadowMap.createAtlas(this.device, resolution, SHADOW_PCF3);
			this.shadowAtlas.cached = true;
			const scissorOffset = 4 / this.shadowAtlasResolution;
			this.scissorVec.set(scissorOffset, scissorOffset, -2 * scissorOffset, -2 * scissorOffset);
		}
	}
	allocateCookieAtlas(resolution) {
		if (this.cookieAtlas.width !== resolution) {
			this.cookieRenderTarget.resize(resolution, resolution);
			this.version++;
		}
	}
	allocateUniforms() {
		this._shadowAtlasTextureId = this.device.scope.resolve('shadowAtlasTexture');
		this._shadowAtlasParamsId = this.device.scope.resolve('shadowAtlasParams');
		this._shadowAtlasParams = new Float32Array(2);
		this._cookieAtlasTextureId = this.device.scope.resolve('cookieAtlasTexture');
	}
	updateUniforms() {
		const isShadowFilterPcf = true;
		const rt = this.shadowAtlas.renderTargets[0];
		const isDepthShadow = !this.device.isWebGL1 && isShadowFilterPcf;
		const shadowBuffer = isDepthShadow ? rt.depthBuffer : rt.colorBuffer;
		this._shadowAtlasTextureId.setValue(shadowBuffer);
		this._shadowAtlasParams[0] = this.shadowAtlasResolution;
		this._shadowAtlasParams[1] = this.shadowEdgePixels;
		this._shadowAtlasParamsId.setValue(this._shadowAtlasParams);
		this._cookieAtlasTextureId.setValue(this.cookieAtlas);
	}
	subdivide(numLights, lightingParams) {
		let atlasSplit = lightingParams.atlasSplit;
		if (!atlasSplit) {
			const gridSize = Math.ceil(Math.sqrt(numLights));
			atlasSplit = _tempArray2;
			atlasSplit[0] = gridSize;
			atlasSplit.length = 1;
		}
		const arraysEqual = (a, b) => a.length === b.length && a.every((v, i) => v === b[i]);
		if (!arraysEqual(atlasSplit, this.atlasSplit)) {
			this.version++;
			this.slots.length = 0;
			this.atlasSplit.length = 0;
			this.atlasSplit.push(...atlasSplit);
			const splitCount = this.atlasSplit[0];
			if (splitCount > 1) {
				const invSize = 1 / splitCount;
				for (let i = 0; i < splitCount; i++) {
					for (let j = 0; j < splitCount; j++) {
						const rect = new Vec4(i * invSize, j * invSize, invSize, invSize);
						const nextLevelSplit = this.atlasSplit[1 + i * splitCount + j];
						if (nextLevelSplit > 1) {
							for (let x = 0; x < nextLevelSplit; x++) {
								for (let y = 0; y < nextLevelSplit; y++) {
									const invSizeNext = invSize / nextLevelSplit;
									const rectNext = new Vec4(rect.x + x * invSizeNext, rect.y + y * invSizeNext, invSizeNext, invSizeNext);
									this.slots.push(new Slot(rectNext));
								}
							}
						} else {
							this.slots.push(new Slot(rect));
						}
					}
				}
			} else {
				this.slots.push(new Slot(new Vec4(0, 0, 1, 1)));
			}
			this.slots.sort((a, b) => {
				return b.size - a.size;
			});
		}
	}
	collectLights(localLights, lightingParams) {
		const cookiesEnabled = lightingParams.cookiesEnabled;
		const shadowsEnabled = lightingParams.shadowsEnabled;
		let needsShadowAtlas = false;
		let needsCookieAtlas = false;
		const lights = _tempArray;
		lights.length = 0;
		const processLights = list => {
			for (let i = 0; i < list.length; i++) {
				const light = list[i];
				if (light.visibleThisFrame) {
					const lightShadow = shadowsEnabled && light.castShadows;
					const lightCookie = cookiesEnabled && !!light.cookie;
					needsShadowAtlas || (needsShadowAtlas = lightShadow);
					needsCookieAtlas || (needsCookieAtlas = lightCookie);
					if (lightShadow || lightCookie) {
						lights.push(light);
					}
				}
			}
		};
		if (cookiesEnabled || shadowsEnabled) {
			processLights(localLights);
		}
		lights.sort((a, b) => {
			return b.maxScreenSize - a.maxScreenSize;
		});
		if (needsShadowAtlas) {
			this.allocateShadowAtlas(this.shadowAtlasResolution);
		}
		if (needsCookieAtlas) {
			this.allocateCookieAtlas(this.cookieAtlasResolution);
		}
		if (needsShadowAtlas || needsCookieAtlas) {
			this.subdivide(lights.length, lightingParams);
		}
		return lights;
	}
	setupSlot(light, rect) {
		light.atlasViewport.copy(rect);
		const faceCount = light.numShadowFaces;
		for (let face = 0; face < faceCount; face++) {
			if (light.castShadows || light._cookie) {
				_viewport$3.copy(rect);
				_scissor.copy(rect);
				if (light._type === LIGHTTYPE_SPOT) {
					_viewport$3.add(this.scissorVec);
				}
				if (light._type === LIGHTTYPE_OMNI) {
					const smallSize = _viewport$3.z / 3;
					const offset = this.cubeSlotsOffsets[face];
					_viewport$3.x += smallSize * offset.x;
					_viewport$3.y += smallSize * offset.y;
					_viewport$3.z = smallSize;
					_viewport$3.w = smallSize;
					_scissor.copy(_viewport$3);
				}
				if (light.castShadows) {
					const lightRenderData = light.getRenderData(null, face);
					lightRenderData.shadowViewport.copy(_viewport$3);
					lightRenderData.shadowScissor.copy(_scissor);
				}
			}
		}
	}
	assignSlot(light, slotIndex, slotReassigned) {
		light.atlasViewportAllocated = true;
		const slot = this.slots[slotIndex];
		slot.lightId = light.id;
		slot.used = true;
		if (slotReassigned) {
			light.atlasSlotUpdated = true;
			light.atlasVersion = this.version;
			light.atlasSlotIndex = slotIndex;
		}
	}
	update(localLights, lightingParams) {
		this.shadowAtlasResolution = lightingParams.shadowAtlasResolution;
		this.cookieAtlasResolution = lightingParams.cookieAtlasResolution;
		const lights = this.collectLights(localLights, lightingParams);
		if (lights.length > 0) {
			const slots = this.slots;
			for (let i = 0; i < slots.length; i++) {
				slots[i].used = false;
			}
			const assignCount = Math.min(lights.length, slots.length);
			for (let i = 0; i < assignCount; i++) {
				const light = lights[i];
				if (light.castShadows) light._shadowMap = this.shadowAtlas;
				const previousSlot = slots[light.atlasSlotIndex];
				if (light.atlasVersion === this.version && light.id === (previousSlot == null ? void 0 : previousSlot.lightId)) {
					const _previousSlot = slots[light.atlasSlotIndex];
					if (_previousSlot.size === slots[i].size && !_previousSlot.used) {
						this.assignSlot(light, light.atlasSlotIndex, false);
					}
				}
			}
			let usedCount = 0;
			for (let i = 0; i < assignCount; i++) {
				while (usedCount < slots.length && slots[usedCount].used) usedCount++;
				const light = lights[i];
				if (!light.atlasViewportAllocated) {
					this.assignSlot(light, usedCount, true);
				}
				const slot = slots[light.atlasSlotIndex];
				this.setupSlot(light, slot.rect);
			}
		}
		this.updateUniforms();
	}
}

const lightCubeDir = [new Vec3(-1, 0, 0), new Vec3(1, 0, 0), new Vec3(0, -1, 0), new Vec3(0, 1, 0), new Vec3(0, 0, -1), new Vec3(0, 0, 1)];
class LightCube {
	constructor() {
		this.colors = new Float32Array(6 * 3);
	}
	update(ambientLight, lights) {
		const colors = this.colors;
		const {
			r,
			g,
			b
		} = ambientLight;
		for (let j = 0; j < 6; j++) {
			colors[j * 3] = r;
			colors[j * 3 + 1] = g;
			colors[j * 3 + 2] = b;
		}
		for (let j = 0; j < lights.length; j++) {
			const light = lights[j];
			if (light._type === LIGHTTYPE_DIRECTIONAL) {
				for (let c = 0; c < 6; c++) {
					const weight = Math.max(lightCubeDir[c].dot(light._direction), 0) * light._intensity;
					const lightColor = light._color;
					colors[c * 3] += lightColor.r * weight;
					colors[c * 3 + 1] += lightColor.g * weight;
					colors[c * 3 + 2] += lightColor.b * weight;
				}
			}
		}
	}
}

class ShadowMapCache {
	constructor() {
		this.cache = new Map();
	}
	destroy() {
		this.clear();
		this.cache = null;
	}
	clear() {
		this.cache.forEach(shadowMaps => {
			shadowMaps.forEach(shadowMap => {
				shadowMap.destroy();
			});
		});
		this.cache.clear();
	}
	getKey(light) {
		const isCubeMap = light._type === LIGHTTYPE_OMNI;
		const shadowType = light._shadowType;
		const resolution = light._shadowResolution;
		return `${isCubeMap}-${shadowType}-${resolution}`;
	}
	get(device, light) {
		const key = this.getKey(light);
		const shadowMaps = this.cache.get(key);
		if (shadowMaps && shadowMaps.length) {
			return shadowMaps.pop();
		}
		const shadowMap = ShadowMap.create(device, light);
		shadowMap.cached = true;
		return shadowMap;
	}
	add(light, shadowMap) {
		const key = this.getKey(light);
		const shadowMaps = this.cache.get(key);
		if (shadowMaps) {
			shadowMaps.push(shadowMap);
		} else {
			this.cache.set(key, [shadowMap]);
		}
	}
}

class RenderPassShadowLocalNonClustered extends RenderPass {
	constructor(device, shadowRenderer, light, face, applyVsm) {
		super(device);
		this.requiresCubemaps = false;
		this.shadowRenderer = shadowRenderer;
		this.light = light;
		this.face = face;
		this.applyVsm = applyVsm;
		this.shadowCamera = shadowRenderer.prepareFace(light, null, face);
		shadowRenderer.setupRenderPass(this, this.shadowCamera, true);
	}
	execute() {
		this.shadowRenderer.renderFace(this.light, null, this.face, false);
	}
	after() {
		if (this.applyVsm) {
			this.shadowRenderer.renderVsm(this.light, this.shadowCamera);
		}
	}
}

class ShadowRendererLocal {
	constructor(renderer, shadowRenderer) {
		this.shadowLights = [];
		this.renderer = void 0;
		this.shadowRenderer = void 0;
		this.device = void 0;
		this.renderer = renderer;
		this.shadowRenderer = shadowRenderer;
		this.device = renderer.device;
	}
	cull(light, comp, casters = null) {
		const isClustered = this.renderer.scene.clusteredLightingEnabled;
		light.visibleThisFrame = true;
		if (!isClustered) {
			if (!light._shadowMap) {
				light._shadowMap = ShadowMap.create(this.device, light);
			}
		}
		const type = light._type;
		const faceCount = type === LIGHTTYPE_SPOT ? 1 : 6;
		for (let face = 0; face < faceCount; face++) {
			const lightRenderData = light.getRenderData(null, face);
			const shadowCam = lightRenderData.shadowCamera;
			shadowCam.nearClip = light.attenuationEnd / 1000;
			shadowCam.farClip = light.attenuationEnd;
			lightRenderData.depthRangeCompensation = shadowCam.farClip - shadowCam.nearClip;
			const shadowCamNode = shadowCam._node;
			const lightNode = light._node;
			shadowCamNode.setPosition(lightNode.getPosition());
			if (type === LIGHTTYPE_SPOT) {
				shadowCam.fov = light._outerConeAngle * 2;
				shadowCamNode.setRotation(lightNode.getRotation());
				shadowCamNode.rotateLocal(-90, 0, 0);
			} else if (type === LIGHTTYPE_OMNI) {
				if (isClustered) {
					const tileSize = this.shadowRenderer.lightTextureAtlas.shadowAtlasResolution * light.atlasViewport.z / 3;
					const texelSize = 2 / tileSize;
					const filterSize = texelSize * this.shadowRenderer.lightTextureAtlas.shadowEdgePixels;
					shadowCam.fov = Math.atan(1 + filterSize) * math.RAD_TO_DEG * 2;
				} else {
					shadowCam.fov = 90;
				}
			}
			this.renderer.updateCameraFrustum(shadowCam);
			this.shadowRenderer.cullShadowCasters(comp, light, lightRenderData.visibleCasters, shadowCam, casters);
		}
	}
	prepareLights(shadowLights, lights) {
		let shadowCamera;
		for (let i = 0; i < lights.length; i++) {
			const light = lights[i];
			if (this.shadowRenderer.needsShadowRendering(light) && light.atlasViewportAllocated) {
				shadowLights.push(light);
				for (let face = 0; face < light.numShadowFaces; face++) {
					shadowCamera = this.shadowRenderer.prepareFace(light, null, face);
				}
			}
		}
		return shadowCamera;
	}
	buildNonClusteredRenderPasses(frameGraph, localLights) {
		for (let i = 0; i < localLights.length; i++) {
			const light = localLights[i];
			if (this.shadowRenderer.needsShadowRendering(light)) {
				const applyVsm = light._type === LIGHTTYPE_SPOT;
				const faceCount = light.numShadowFaces;
				for (let face = 0; face < faceCount; face++) {
					const renderPass = new RenderPassShadowLocalNonClustered(this.device, this.shadowRenderer, light, face, applyVsm);
					frameGraph.addRenderPass(renderPass);
				}
			}
		}
	}
}

class RenderPassShadowDirectional extends RenderPass {
	constructor(device, shadowRenderer, light, camera, allCascadesRendering) {
		super(device);
		this.shadowRenderer = shadowRenderer;
		this.light = light;
		this.camera = camera;
		this.allCascadesRendering = allCascadesRendering;
	}
	execute() {
		const {
			light,
			camera,
			shadowRenderer,
			allCascadesRendering
		} = this;
		const faceCount = light.numShadowFaces;
		const shadowUpdateOverrides = light.shadowUpdateOverrides;
		for (let face = 0; face < faceCount; face++) {
			if ((shadowUpdateOverrides == null ? void 0 : shadowUpdateOverrides[face]) !== SHADOWUPDATE_NONE) {
				shadowRenderer.renderFace(light, camera, face, !allCascadesRendering);
			}
			if ((shadowUpdateOverrides == null ? void 0 : shadowUpdateOverrides[face]) === SHADOWUPDATE_THISFRAME) {
				shadowUpdateOverrides[face] = SHADOWUPDATE_NONE;
			}
		}
	}
	after() {
		this.shadowRenderer.renderVsm(this.light, this.camera);
	}
}

const visibleSceneAabb = new BoundingBox();
const center = new Vec3();
const shadowCamView$1 = new Mat4();
const aabbPoints = [new Vec3(), new Vec3(), new Vec3(), new Vec3(), new Vec3(), new Vec3(), new Vec3(), new Vec3()];
const _depthRange = {
	min: 0,
	max: 0
};
function getDepthRange(cameraViewMatrix, aabbMin, aabbMax) {
	aabbPoints[0].x = aabbPoints[1].x = aabbPoints[2].x = aabbPoints[3].x = aabbMin.x;
	aabbPoints[1].y = aabbPoints[3].y = aabbPoints[7].y = aabbPoints[5].y = aabbMin.y;
	aabbPoints[2].z = aabbPoints[3].z = aabbPoints[6].z = aabbPoints[7].z = aabbMin.z;
	aabbPoints[4].x = aabbPoints[5].x = aabbPoints[6].x = aabbPoints[7].x = aabbMax.x;
	aabbPoints[0].y = aabbPoints[2].y = aabbPoints[4].y = aabbPoints[6].y = aabbMax.y;
	aabbPoints[0].z = aabbPoints[1].z = aabbPoints[4].z = aabbPoints[5].z = aabbMax.z;
	let minz = 9999999999;
	let maxz = -9999999999;
	for (let i = 0; i < 8; ++i) {
		cameraViewMatrix.transformPoint(aabbPoints[i], aabbPoints[i]);
		const z = aabbPoints[i].z;
		if (z < minz) minz = z;
		if (z > maxz) maxz = z;
	}
	_depthRange.min = minz;
	_depthRange.max = maxz;
	return _depthRange;
}
class ShadowRendererDirectional {
	constructor(renderer, shadowRenderer) {
		this.renderer = void 0;
		this.shadowRenderer = void 0;
		this.device = void 0;
		this.renderer = renderer;
		this.shadowRenderer = shadowRenderer;
		this.device = renderer.device;
	}
	cull(light, comp, camera, casters = null) {
		light.visibleThisFrame = true;
		if (!light._shadowMap) {
			light._shadowMap = ShadowMap.create(this.device, light);
		}
		const nearDist = camera._nearClip;
		this.generateSplitDistances(light, nearDist, Math.min(camera._farClip, light.shadowDistance));
		const shadowUpdateOverrides = light.shadowUpdateOverrides;
		for (let cascade = 0; cascade < light.numCascades; cascade++) {
			if ((shadowUpdateOverrides == null ? void 0 : shadowUpdateOverrides[cascade]) === SHADOWUPDATE_NONE) {
				break;
			}
			const lightRenderData = light.getRenderData(camera, cascade);
			const shadowCam = lightRenderData.shadowCamera;
			shadowCam.renderTarget = light._shadowMap.renderTargets[0];
			lightRenderData.shadowViewport.copy(light.cascades[cascade]);
			lightRenderData.shadowScissor.copy(light.cascades[cascade]);
			const shadowCamNode = shadowCam._node;
			const lightNode = light._node;
			shadowCamNode.setPosition(lightNode.getPosition());
			shadowCamNode.setRotation(lightNode.getRotation());
			shadowCamNode.rotateLocal(-90, 0, 0);
			const frustumNearDist = cascade === 0 ? nearDist : light._shadowCascadeDistances[cascade - 1];
			const frustumFarDist = light._shadowCascadeDistances[cascade];
			const frustumPoints = camera.getFrustumCorners(frustumNearDist, frustumFarDist);
			center.set(0, 0, 0);
			const cameraWorldMat = camera.node.getWorldTransform();
			for (let i = 0; i < 8; i++) {
				cameraWorldMat.transformPoint(frustumPoints[i], frustumPoints[i]);
				center.add(frustumPoints[i]);
			}
			center.mulScalar(1 / 8);
			let radius = 0;
			for (let i = 0; i < 8; i++) {
				const dist = frustumPoints[i].sub(center).length();
				if (dist > radius) radius = dist;
			}
			const right = shadowCamNode.right;
			const up = shadowCamNode.up;
			const lightDir = shadowCamNode.forward;
			const sizeRatio = 0.25 * light._shadowResolution / radius;
			const x = Math.ceil(center.dot(up) * sizeRatio) / sizeRatio;
			const y = Math.ceil(center.dot(right) * sizeRatio) / sizeRatio;
			const scaledUp = up.mulScalar(x);
			const scaledRight = right.mulScalar(y);
			const dot = center.dot(lightDir);
			const scaledDir = lightDir.mulScalar(dot);
			center.add2(scaledUp, scaledRight).add(scaledDir);
			shadowCamNode.setPosition(center);
			shadowCamNode.translateLocal(0, 0, 1000000);
			shadowCam.nearClip = 0.01;
			shadowCam.farClip = 2000000;
			shadowCam.orthoHeight = radius;
			this.renderer.updateCameraFrustum(shadowCam);
			this.shadowRenderer.cullShadowCasters(comp, light, lightRenderData.visibleCasters, shadowCam, casters);
			let emptyAabb = true;
			const visibleCasters = lightRenderData.visibleCasters;
			for (let i = 0; i < visibleCasters.length; i++) {
				const meshInstance = visibleCasters[i];
				if (emptyAabb) {
					emptyAabb = false;
					visibleSceneAabb.copy(meshInstance.aabb);
				} else {
					visibleSceneAabb.add(meshInstance.aabb);
				}
			}
			shadowCamView$1.copy(shadowCamNode.getWorldTransform()).invert();
			const depthRange = getDepthRange(shadowCamView$1, visibleSceneAabb.getMin(), visibleSceneAabb.getMax());
			shadowCamNode.translateLocal(0, 0, depthRange.max + 0.1);
			shadowCam.farClip = depthRange.max - depthRange.min + 0.2;
			lightRenderData.depthRangeCompensation = shadowCam.farClip;
			lightRenderData.projectionCompensation = radius;
		}
	}
	generateSplitDistances(light, nearDist, farDist) {
		light._shadowCascadeDistances.fill(farDist);
		for (let i = 1; i < light.numCascades; i++) {
			const fraction = i / light.numCascades;
			const linearDist = nearDist + (farDist - nearDist) * fraction;
			const logDist = nearDist * (farDist / nearDist) ** fraction;
			const dist = math.lerp(linearDist, logDist, light.cascadeDistribution);
			light._shadowCascadeDistances[i - 1] = dist;
		}
	}
	getLightRenderPass(light, camera) {
		let renderPass = null;
		if (this.shadowRenderer.needsShadowRendering(light)) {
			const faceCount = light.numShadowFaces;
			const shadowUpdateOverrides = light.shadowUpdateOverrides;
			let allCascadesRendering = true;
			let shadowCamera;
			for (let face = 0; face < faceCount; face++) {
				if ((shadowUpdateOverrides == null ? void 0 : shadowUpdateOverrides[face]) === SHADOWUPDATE_NONE) allCascadesRendering = false;
				shadowCamera = this.shadowRenderer.prepareFace(light, camera, face);
			}
			renderPass = new RenderPassShadowDirectional(this.device, this.shadowRenderer, light, camera, allCascadesRendering);
			this.shadowRenderer.setupRenderPass(renderPass, shadowCamera, allCascadesRendering);
		}
		return renderPass;
	}
}

function gauss(x, sigma) {
	return Math.exp(-(x * x) / (2.0 * sigma * sigma));
}
function gaussWeights(kernelSize) {
	const sigma = (kernelSize - 1) / (2 * 3);
	const halfWidth = (kernelSize - 1) * 0.5;
	const values = new Array(kernelSize);
	let sum = 0.0;
	for (let i = 0; i < kernelSize; ++i) {
		values[i] = gauss(i - halfWidth, sigma);
		sum += values[i];
	}
	for (let i = 0; i < kernelSize; ++i) {
		values[i] /= sum;
	}
	return values;
}
const tempSet$1 = new Set();
const shadowCamView = new Mat4();
const shadowCamViewProj = new Mat4();
const pixelOffset = new Float32Array(2);
const blurScissorRect = new Vec4(1, 1, 0, 0);
const viewportMatrix = new Mat4();
class ShadowRenderer {
	constructor(renderer, lightTextureAtlas) {
		this.shadowPassCache = [];
		this.device = renderer.device;
		this.renderer = renderer;
		this.lightTextureAtlas = lightTextureAtlas;
		const scope = this.device.scope;
		this.polygonOffsetId = scope.resolve('polygonOffset');
		this.polygonOffset = new Float32Array(2);
		this.sourceId = scope.resolve('source');
		this.pixelOffsetId = scope.resolve('pixelOffset');
		this.weightId = scope.resolve('weight[0]');
		this.blurVsmShaderCode = [shaderChunks.blurVSMPS, '#define GAUSS\n' + shaderChunks.blurVSMPS];
		const packed = '#define PACKED\n';
		this.blurPackedVsmShaderCode = [packed + this.blurVsmShaderCode[0], packed + this.blurVsmShaderCode[1]];
		this.blurVsmShader = [{}, {}];
		this.blurPackedVsmShader = [{}, {}];
		this.blurVsmWeights = {};
		this.shadowMapLightRadiusId = scope.resolve('light_radius');
		this.viewUniformFormat = null;
		this.viewBindGroupFormat = null;
		this.blendStateWrite = new BlendState();
		this.blendStateNoWrite = new BlendState();
		this.blendStateNoWrite.setColorWrite(false, false, false, false);
	}
	static createShadowCamera(device, shadowType, type, face) {
		const shadowCam = LightCamera.create('ShadowCamera', type, face);
		if (shadowType >= SHADOW_VSM8 && shadowType <= SHADOW_VSM32) {
			shadowCam.clearColor = new Color(0, 0, 0, 0);
		} else {
			shadowCam.clearColor = new Color(1, 1, 1, 1);
		}
		shadowCam.clearDepthBuffer = true;
		shadowCam.clearStencilBuffer = false;
		return shadowCam;
	}
	static setShadowCameraSettings(shadowCam, device, shadowType, type, isClustered) {
		let hwPcf = shadowType === SHADOW_PCF5 || (shadowType === SHADOW_PCF1 || shadowType === SHADOW_PCF3) && device.supportsDepthShadow;
		if (type === LIGHTTYPE_OMNI && !isClustered) {
			hwPcf = false;
		}
		shadowCam.clearColorBuffer = !hwPcf;
	}
	_cullShadowCastersInternal(meshInstances, visible, camera) {
		const numInstances = meshInstances.length;
		for (let i = 0; i < numInstances; i++) {
			const meshInstance = meshInstances[i];
			if (meshInstance.castShadow) {
				if (!meshInstance.cull || meshInstance._isVisible(camera)) {
					meshInstance.visibleThisFrame = true;
					visible.push(meshInstance);
				}
			}
		}
	}
	cullShadowCasters(comp, light, visible, camera, casters) {
		visible.length = 0;
		if (casters) {
			this._cullShadowCastersInternal(casters, visible, camera);
		} else {
			const layers = comp.layerList;
			const len = layers.length;
			for (let i = 0; i < len; i++) {
				const layer = layers[i];
				if (layer._lightsSet.has(light)) {
					if (!tempSet$1.has(layer)) {
						tempSet$1.add(layer);
						this._cullShadowCastersInternal(layer.shadowCasters, visible, camera);
					}
				}
			}
			tempSet$1.clear();
		}
		visible.sort(this.renderer.sortCompareDepth);
	}
	setupRenderState(device, light) {
		if (device.isWebGL1 && device.extStandardDerivatives) {
			if (light._type === LIGHTTYPE_OMNI) {
				this.polygonOffset[0] = 0;
				this.polygonOffset[1] = 0;
				this.polygonOffsetId.setValue(this.polygonOffset);
			} else {
				this.polygonOffset[0] = light.shadowBias * -1000.0;
				this.polygonOffset[1] = light.shadowBias * -1000.0;
				this.polygonOffsetId.setValue(this.polygonOffset);
			}
		}
		const isClustered = this.renderer.scene.clusteredLightingEnabled;
		const gpuOrGl2 = device.isWebGL2 || device.isWebGPU;
		const useShadowSampler = isClustered ? light._isPcf && gpuOrGl2 : light._isPcf && gpuOrGl2 && light._type !== LIGHTTYPE_OMNI;
		device.setBlendState(useShadowSampler ? this.blendStateNoWrite : this.blendStateWrite);
		device.setDepthState(light.shadowDepthState);
		device.setStencilState(null, null);
	}
	dispatchUniforms(light, shadowCam, lightRenderData, face) {
		const shadowCamNode = shadowCam._node;
		if (light._type !== LIGHTTYPE_DIRECTIONAL) {
			this.renderer.dispatchViewPos(shadowCamNode.getPosition());
			this.shadowMapLightRadiusId.setValue(light.attenuationEnd);
		}
		shadowCamView.setTRS(shadowCamNode.getPosition(), shadowCamNode.getRotation(), Vec3.ONE).invert();
		shadowCamViewProj.mul2(shadowCam.projectionMatrix, shadowCamView);
		const rectViewport = lightRenderData.shadowViewport;
		shadowCam.rect = rectViewport;
		shadowCam.scissorRect = lightRenderData.shadowScissor;
		viewportMatrix.setViewport(rectViewport.x, rectViewport.y, rectViewport.z, rectViewport.w);
		lightRenderData.shadowMatrix.mul2(viewportMatrix, shadowCamViewProj);
		if (light._type === LIGHTTYPE_DIRECTIONAL) {
			light._shadowMatrixPalette.set(lightRenderData.shadowMatrix.data, face * 16);
		}
	}
	getShadowPass(light) {
		var _this$shadowPassCache;
		const lightType = light._type;
		const shadowType = light._shadowType;
		let shadowPassInfo = (_this$shadowPassCache = this.shadowPassCache[lightType]) == null ? void 0 : _this$shadowPassCache[shadowType];
		if (!shadowPassInfo) {
			const shadowPassName = `ShadowPass_${lightType}_${shadowType}`;
			shadowPassInfo = ShaderPass.get(this.device).allocate(shadowPassName, {
				isShadow: true,
				lightType: lightType,
				shadowType: shadowType
			});
			if (!this.shadowPassCache[lightType]) this.shadowPassCache[lightType] = [];
			this.shadowPassCache[lightType][shadowType] = shadowPassInfo;
		}
		return shadowPassInfo.index;
	}
	submitCasters(visibleCasters, light) {
		const device = this.device;
		const renderer = this.renderer;
		const scene = renderer.scene;
		const passFlags = 1 << SHADER_SHADOW;
		const shadowPass = this.getShadowPass(light);
		const count = visibleCasters.length;
		for (let i = 0; i < count; i++) {
			const meshInstance = visibleCasters[i];
			const mesh = meshInstance.mesh;
			meshInstance.ensureMaterial(device);
			const material = meshInstance.material;
			renderer.setBaseConstants(device, material);
			renderer.setSkinning(device, meshInstance);
			if (material.dirty) {
				material.updateUniforms(device, scene);
				material.dirty = false;
			}
			if (material.chunks) {
				renderer.setupCullMode(true, 1, meshInstance);
				material.setParameters(device);
				meshInstance.setParameters(device, passFlags);
			}
			const shaderInstance = meshInstance.getShaderInstance(shadowPass, 0, scene, this.viewUniformFormat, this.viewBindGroupFormat);
			const shadowShader = shaderInstance.shader;
			meshInstance._key[SORTKEY_DEPTH] = shadowShader.id;
			device.setShader(shadowShader);
			renderer.setVertexBuffers(device, mesh);
			renderer.setMorphing(device, meshInstance.morphInstance);
			this.renderer.setupMeshUniformBuffers(shaderInstance, meshInstance);
			const style = meshInstance.renderStyle;
			device.setIndexBuffer(mesh.indexBuffer[style]);
			renderer.drawInstance(device, meshInstance, mesh, style);
			renderer._shadowDrawCalls++;
		}
	}
	needsShadowRendering(light) {
		const needs = light.enabled && light.castShadows && light.shadowUpdateMode !== SHADOWUPDATE_NONE && light.visibleThisFrame;
		if (light.shadowUpdateMode === SHADOWUPDATE_THISFRAME) {
			light.shadowUpdateMode = SHADOWUPDATE_NONE;
		}
		if (needs) {
			this.renderer._shadowMapUpdates += light.numShadowFaces;
		}
		return needs;
	}
	getLightRenderData(light, camera, face) {
		return light.getRenderData(light._type === LIGHTTYPE_DIRECTIONAL ? camera : null, face);
	}
	setupRenderPass(renderPass, shadowCamera, clearRenderTarget) {
		const rt = shadowCamera.renderTarget;
		renderPass.init(rt);
		renderPass.depthStencilOps.clearDepthValue = 1;
		renderPass.depthStencilOps.clearDepth = clearRenderTarget;
		if (rt.depthBuffer) {
			renderPass.depthStencilOps.storeDepth = true;
		} else {
			renderPass.colorOps.clearValue.copy(shadowCamera.clearColor);
			renderPass.colorOps.clear = clearRenderTarget;
			renderPass.depthStencilOps.storeDepth = false;
		}
		renderPass.requiresCubemaps = false;
	}
	prepareFace(light, camera, face) {
		const type = light._type;
		const shadowType = light._shadowType;
		const isClustered = this.renderer.scene.clusteredLightingEnabled;
		const lightRenderData = this.getLightRenderData(light, camera, face);
		const shadowCam = lightRenderData.shadowCamera;
		ShadowRenderer.setShadowCameraSettings(shadowCam, this.device, shadowType, type, isClustered);
		const renderTargetIndex = type === LIGHTTYPE_DIRECTIONAL ? 0 : face;
		shadowCam.renderTarget = light._shadowMap.renderTargets[renderTargetIndex];
		return shadowCam;
	}
	renderFace(light, camera, face, clear, insideRenderPass = true) {
		const device = this.device;
		const lightRenderData = this.getLightRenderData(light, camera, face);
		const shadowCam = lightRenderData.shadowCamera;
		this.dispatchUniforms(light, shadowCam, lightRenderData, face);
		const rt = shadowCam.renderTarget;
		const renderer = this.renderer;
		renderer.setCameraUniforms(shadowCam, rt);
		if (device.supportsUniformBuffers) {
			renderer.setupViewUniformBuffers(lightRenderData.viewBindGroups, this.viewUniformFormat, this.viewBindGroupFormat, 1);
		}
		if (insideRenderPass) {
			renderer.setupViewport(shadowCam, rt);
			if (clear) {
				renderer.clear(shadowCam);
			}
		} else {
			renderer.clearView(shadowCam, rt, true, false);
		}
		this.setupRenderState(device, light);
		this.submitCasters(lightRenderData.visibleCasters, light);
	}
	render(light, camera, insideRenderPass = true) {
		if (this.needsShadowRendering(light)) {
			const faceCount = light.numShadowFaces;
			for (let face = 0; face < faceCount; face++) {
				this.prepareFace(light, camera, face);
				this.renderFace(light, camera, face, true, insideRenderPass);
			}
			this.renderVsm(light, camera);
		}
	}
	renderVsm(light, camera) {
		if (light._isVsm && light._vsmBlurSize > 1) {
			const isClustered = this.renderer.scene.clusteredLightingEnabled;
			if (!isClustered || light._type === LIGHTTYPE_DIRECTIONAL) {
				this.applyVsmBlur(light, camera);
			}
		}
	}
	getVsmBlurShader(isVsm8, blurMode, filterSize) {
		let blurShader = (isVsm8 ? this.blurPackedVsmShader : this.blurVsmShader)[blurMode][filterSize];
		if (!blurShader) {
			this.blurVsmWeights[filterSize] = gaussWeights(filterSize);
			const blurVS = shaderChunks.fullscreenQuadVS;
			let blurFS = '#define SAMPLES ' + filterSize + '\n';
			if (isVsm8) {
				blurFS += this.blurPackedVsmShaderCode[blurMode];
			} else {
				blurFS += this.blurVsmShaderCode[blurMode];
			}
			const blurShaderName = 'blurVsm' + blurMode + '' + filterSize + '' + isVsm8;
			blurShader = createShaderFromCode(this.device, blurVS, blurFS, blurShaderName);
			if (isVsm8) {
				this.blurPackedVsmShader[blurMode][filterSize] = blurShader;
			} else {
				this.blurVsmShader[blurMode][filterSize] = blurShader;
			}
		}
		return blurShader;
	}
	applyVsmBlur(light, camera) {
		const device = this.device;
		device.setBlendState(BlendState.NOBLEND);
		const lightRenderData = light.getRenderData(light._type === LIGHTTYPE_DIRECTIONAL ? camera : null, 0);
		const shadowCam = lightRenderData.shadowCamera;
		const origShadowMap = shadowCam.renderTarget;
		const tempShadowMap = this.renderer.shadowMapCache.get(device, light);
		const tempRt = tempShadowMap.renderTargets[0];
		const isVsm8 = light._shadowType === SHADOW_VSM8;
		const blurMode = light.vsmBlurMode;
		const filterSize = light._vsmBlurSize;
		const blurShader = this.getVsmBlurShader(isVsm8, blurMode, filterSize);
		blurScissorRect.z = light._shadowResolution - 2;
		blurScissorRect.w = blurScissorRect.z;
		this.sourceId.setValue(origShadowMap.colorBuffer);
		pixelOffset[0] = 1 / light._shadowResolution;
		pixelOffset[1] = 0;
		this.pixelOffsetId.setValue(pixelOffset);
		if (blurMode === BLUR_GAUSSIAN) this.weightId.setValue(this.blurVsmWeights[filterSize]);
		drawQuadWithShader(device, tempRt, blurShader, null, blurScissorRect);
		this.sourceId.setValue(tempRt.colorBuffer);
		pixelOffset[1] = pixelOffset[0];
		pixelOffset[0] = 0;
		this.pixelOffsetId.setValue(pixelOffset);
		drawQuadWithShader(device, origShadowMap, blurShader, null, blurScissorRect);
		this.renderer.shadowMapCache.add(light, tempShadowMap);
	}
	initViewBindGroupFormat() {
		if (this.device.supportsUniformBuffers && !this.viewUniformFormat) {
			this.viewUniformFormat = new UniformBufferFormat(this.device, [new UniformFormat("matrix_viewProjection", UNIFORMTYPE_MAT4)]);
			this.viewBindGroupFormat = new BindGroupFormat(this.device, [new BindBufferFormat(UNIFORM_BUFFER_DEFAULT_SLOT_NAME, SHADERSTAGE_VERTEX | SHADERSTAGE_FRAGMENT)], []);
		}
	}
	frameUpdate() {
		this.initViewBindGroupFormat();
	}
}

const tempClusterArray = [];
class WorldClustersAllocator {
	constructor(graphicsDevice) {
		this._empty = null;
		this._allocated = [];
		this._clusters = new Map();
		this.device = graphicsDevice;
	}
	destroy() {
		if (this._empty) {
			this._empty.destroy();
			this._empty = null;
		}
		this._allocated.forEach(cluster => {
			cluster.destroy();
		});
		this._allocated.length = 0;
	}
	get count() {
		return this._allocated.length;
	}
	get empty() {
		if (!this._empty) {
			const empty = new WorldClusters(this.device);
			empty.name = 'ClusterEmpty';
			empty.update([], false, null);
			this._empty = empty;
		}
		return this._empty;
	}
	assign(renderPasses) {
		const empty = this.empty;
		tempClusterArray.push(...this._allocated);
		this._allocated.length = 0;
		this._clusters.clear();
		const passCount = renderPasses.length;
		for (let p = 0; p < passCount; p++) {
			const renderPass = renderPasses[p];
			const renderActions = renderPass.renderActions;
			if (renderActions) {
				const count = renderActions.length;
				for (let i = 0; i < count; i++) {
					const ra = renderActions[i];
					ra.lightClusters = null;
					const layer = ra.layer;
					if (layer.hasClusteredLights && layer.meshInstances.length) {
						const hash = layer.getLightIdHash();
						const existingRenderAction = this._clusters.get(hash);
						let clusters = existingRenderAction == null ? void 0 : existingRenderAction.lightClusters;
						if (!clusters) {
							var _tempClusterArray$pop;
							clusters = (_tempClusterArray$pop = tempClusterArray.pop()) != null ? _tempClusterArray$pop : new WorldClusters(this.device);
							this._allocated.push(clusters);
							this._clusters.set(hash, ra);
						}
						ra.lightClusters = clusters;
					}
					if (!ra.lightClusters) {
						ra.lightClusters = empty;
					}
				}
			}
		}
		tempClusterArray.forEach(item => item.destroy());
		tempClusterArray.length = 0;
	}
	update(renderPasses, gammaCorrection, lighting) {
		this.assign(renderPasses);
		this._clusters.forEach(renderAction => {
			const layer = renderAction.layer;
			const cluster = renderAction.lightClusters;
			cluster.update(layer.clusteredLightsSet, gammaCorrection, lighting);
		});
	}
}

const textureBlitVertexShader = `
	attribute vec2 vertex_position;
	varying vec2 uv0;
	void main(void) {
		gl_Position = vec4(vertex_position, 0.5, 1.0);
		uv0 = vertex_position.xy * 0.5 + 0.5;
		#ifndef WEBGPU
			uv0.y = 1.0 - uv0.y;
		#endif
	}`;
const textureBlitFragmentShader = `
	varying vec2 uv0;
	uniform sampler2D blitTexture;
	void main(void) {
		gl_FragColor = texture2D(blitTexture, uv0);
	}`;
const textureCubeBlitFragmentShader = `
	varying vec2 uv0;
	uniform samplerCube blitTexture;
	uniform mat4 invViewProj;
	void main(void) {
		vec4 projPos = vec4(uv0 * 2.0 - 1.0, 0.5, 1.0);
		vec4 worldPos = invViewProj * projPos;
		gl_FragColor = textureCube(blitTexture, worldPos.xyz);
	}`;
const _viewport$2 = new Vec4();
const _invViewProjMatrices = [];
class RenderPassCookieRenderer extends RenderPass {
	constructor(device, cubeSlotsOffsets) {
		super(device);
		this._quadRenderer2D = null;
		this._quadRendererCube = null;
		this._filteredLights = [];
		this._cubeSlotsOffsets = cubeSlotsOffsets;
		this.requiresCubemaps = false;
		this.blitTextureId = device.scope.resolve('blitTexture');
		this.invViewProjId = device.scope.resolve('invViewProj');
	}
	destroy() {
		var _this$_quadRenderer2D, _this$_quadRendererCu;
		(_this$_quadRenderer2D = this._quadRenderer2D) == null || _this$_quadRenderer2D.destroy();
		this._quadRenderer2D = null;
		(_this$_quadRendererCu = this._quadRendererCube) == null || _this$_quadRendererCu.destroy();
		this._quadRendererCube = null;
	}
	static create(renderTarget, cubeSlotsOffsets) {
		const renderPass = new RenderPassCookieRenderer(renderTarget.device, cubeSlotsOffsets);
		renderPass.init(renderTarget);
		renderPass.colorOps.clear = false;
		renderPass.depthStencilOps.clearDepth = false;
		return renderPass;
	}
	update(lights) {
		const filteredLights = this._filteredLights;
		this.filter(lights, filteredLights);
		this.executeEnabled = filteredLights.length > 0;
	}
	filter(lights, filteredLights) {
		for (let i = 0; i < lights.length; i++) {
			const light = lights[i];
			if (light._type === LIGHTTYPE_DIRECTIONAL) continue;
			if (!light.atlasViewportAllocated) continue;
			if (!light.atlasSlotUpdated) continue;
			if (light.enabled && light.cookie && light.visibleThisFrame) {
				filteredLights.push(light);
			}
		}
	}
	initInvViewProjMatrices() {
		if (!_invViewProjMatrices.length) {
			for (let face = 0; face < 6; face++) {
				const camera = LightCamera.create(null, LIGHTTYPE_OMNI, face);
				const projMat = camera.projectionMatrix;
				const viewMat = camera.node.getLocalTransform().clone().invert();
				_invViewProjMatrices[face] = new Mat4().mul2(projMat, viewMat).invert();
			}
		}
	}
	get quadRenderer2D() {
		if (!this._quadRenderer2D) {
			const shader = createShaderFromCode(this.device, textureBlitVertexShader, textureBlitFragmentShader, `cookieRenderer2d`);
			this._quadRenderer2D = new QuadRender(shader);
		}
		return this._quadRenderer2D;
	}
	get quadRendererCube() {
		if (!this._quadRendererCube) {
			const shader = createShaderFromCode(this.device, textureBlitVertexShader, textureCubeBlitFragmentShader, `cookieRendererCube`);
			this._quadRendererCube = new QuadRender(shader);
		}
		return this._quadRendererCube;
	}
	execute() {
		const device = this.device;
		device.setBlendState(BlendState.NOBLEND);
		device.setCullMode(CULLFACE_NONE);
		device.setDepthState(DepthState.NODEPTH);
		device.setStencilState();
		const renderTargetWidth = this.renderTarget.colorBuffer.width;
		const cubeSlotsOffsets = this._cubeSlotsOffsets;
		const filteredLights = this._filteredLights;
		for (let i = 0; i < filteredLights.length; i++) {
			const light = filteredLights[i];
			const faceCount = light.numShadowFaces;
			const quad = faceCount > 1 ? this.quadRendererCube : this.quadRenderer2D;
			if (faceCount > 1) {
				this.initInvViewProjMatrices();
			}
			this.blitTextureId.setValue(light.cookie);
			for (let face = 0; face < faceCount; face++) {
				_viewport$2.copy(light.atlasViewport);
				if (faceCount > 1) {
					const smallSize = _viewport$2.z / 3;
					const offset = cubeSlotsOffsets[face];
					_viewport$2.x += smallSize * offset.x;
					_viewport$2.y += smallSize * offset.y;
					_viewport$2.z = smallSize;
					_viewport$2.w = smallSize;
					this.invViewProjId.setValue(_invViewProjMatrices[face].data);
				}
				_viewport$2.mulScalar(renderTargetWidth);
				quad.render(_viewport$2);
			}
		}
		filteredLights.length = 0;
	}
}

class RenderPassShadowLocalClustered extends RenderPass {
	constructor(device, shadowRenderer, shadowRendererLocal) {
		super(device);
		this.requiresCubemaps = false;
		this.shadowRenderer = shadowRenderer;
		this.shadowRendererLocal = shadowRendererLocal;
	}
	update(localLights) {
		const shadowLights = this.shadowRendererLocal.shadowLights;
		const shadowCamera = this.shadowRendererLocal.prepareLights(shadowLights, localLights);
		const count = shadowLights.length;
		this.enabled = count > 0;
		if (count) {
			this.shadowRenderer.setupRenderPass(this, shadowCamera, false);
		}
	}
	execute() {
		const shadowLights = this.shadowRendererLocal.shadowLights;
		const count = shadowLights.length;
		for (let i = 0; i < count; i++) {
			const light = shadowLights[i];
			for (let face = 0; face < light.numShadowFaces; face++) {
				this.shadowRenderer.renderFace(light, null, face, true);
			}
		}
		shadowLights.length = 0;
	}
}

class RenderPassUpdateClustered extends RenderPass {
	constructor(device, renderer, shadowRenderer, shadowRendererLocal, lightTextureAtlas) {
		super(device);
		this.renderer = renderer;
		this.frameGraph = null;
		this.cookiesRenderPass = RenderPassCookieRenderer.create(lightTextureAtlas.cookieRenderTarget, lightTextureAtlas.cubeSlotsOffsets);
		this.beforePasses.push(this.cookiesRenderPass);
		this.shadowRenderPass = new RenderPassShadowLocalClustered(device, shadowRenderer, shadowRendererLocal);
		this.beforePasses.push(this.shadowRenderPass);
	}
	update(frameGraph, shadowsEnabled, cookiesEnabled, lights, localLights) {
		this.frameGraph = frameGraph;
		this.cookiesRenderPass.enabled = cookiesEnabled;
		if (cookiesEnabled) {
			this.cookiesRenderPass.update(lights);
		}
		this.shadowRenderPass.enabled = shadowsEnabled;
		if (shadowsEnabled) {
			this.shadowRenderPass.update(localLights);
		}
	}
	destroy() {
		this.cookiesRenderPass.destroy();
		this.cookiesRenderPass = null;
	}
	execute() {
		const {
			renderer
		} = this;
		const {
			scene
		} = renderer;
		renderer.worldClustersAllocator.update(this.frameGraph.renderPasses, scene.gammaCorrection, scene.lighting);
	}
}

const base64String = "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";
let data = null;
const initData = () => {
	if (!data) {
		const binaryString = atob(base64String);
		data = Uint8Array.from(binaryString, char => char.charCodeAt(0));
	}
};
const blueNoiseData = () => {
	initData();
	return data;
};
class BlueNoise {
	constructor(seed = 0) {
		this.seed = 0;
		this.seed = seed * 4;
		initData();
	}
	_next() {
		this.seed = (this.seed + 4) % data.length;
	}
	value() {
		this._next();
		return data[this.seed] / 255;
	}
	vec4(dest = new Vec4()) {
		this._next();
		return dest.set(data[this.seed], data[this.seed + 1], data[this.seed + 2], data[this.seed + 3]).mulScalar(1 / 255);
	}
}

const deviceCache$1 = new DeviceCache();
function getBlueNoiseTexture(device) {
	return deviceCache$1.get(device, () => {
		const data = blueNoiseData();
		const size = Math.sqrt(data.length / 4);
		const texture = new Texture(device, {
			name: `BlueNoise${size}`,
			width: size,
			height: size,
			format: PIXELFORMAT_RGBA8,
			addressU: ADDRESS_REPEAT,
			addressV: ADDRESS_REPEAT,
			type: TEXTURETYPE_DEFAULT,
			magFilter: FILTER_NEAREST,
			minFilter: FILTER_NEAREST,
			anisotropy: 1,
			mipmaps: false
		});
		texture.lock().set(data);
		texture.unlock();
		return texture;
	});
}

let _skinUpdateIndex = 0;
const viewProjMat = new Mat4();
const viewInvMat = new Mat4();
const viewMat = new Mat4();
const viewMat3 = new Mat3();
const tempSphere$1 = new BoundingSphere();
const _flipYMat = new Mat4().setScale(1, -1, 1);
const _tempLightSet = new Set();
const _tempLayerSet = new Set();
const _tempVec4 = new Vec4();
const _fixProjRangeMat = new Mat4().set([1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0.5, 0, 0, 0, 0.5, 1]);
const _haltonSequence = [new Vec2(0.5, 0.333333), new Vec2(0.25, 0.666667), new Vec2(0.75, 0.111111), new Vec2(0.125, 0.444444), new Vec2(0.625, 0.777778), new Vec2(0.375, 0.222222), new Vec2(0.875, 0.555556), new Vec2(0.0625, 0.888889), new Vec2(0.5625, 0.037037), new Vec2(0.3125, 0.370370), new Vec2(0.8125, 0.703704), new Vec2(0.1875, 0.148148), new Vec2(0.6875, 0.481481), new Vec2(0.4375, 0.814815), new Vec2(0.9375, 0.259259), new Vec2(0.03125, 0.592593)];
const _tempProjMat0 = new Mat4();
const _tempProjMat1 = new Mat4();
const _tempProjMat2 = new Mat4();
const _tempProjMat3 = new Mat4();
const _tempProjMat4 = new Mat4();
const _tempProjMat5 = new Mat4();
const _tempSet = new Set();
const _tempMeshInstances = [];
const _tempMeshInstancesSkinned = [];
class Renderer {
	constructor(graphicsDevice) {
		this.clustersDebugRendered = false;
		this.processingMeshInstances = new Set();
		this.worldClustersAllocator = void 0;
		this.lights = [];
		this.localLights = [];
		this.cameraDirShadowLights = new Map();
		this.dirLightShadows = new Map();
		this.blueNoise = new BlueNoise(123);
		this.device = graphicsDevice;
		this.scene = null;
		this.worldClustersAllocator = new WorldClustersAllocator(graphicsDevice);
		this.lightTextureAtlas = new LightTextureAtlas(graphicsDevice);
		this.shadowMapCache = new ShadowMapCache();
		this.shadowRenderer = new ShadowRenderer(this, this.lightTextureAtlas);
		this._shadowRendererLocal = new ShadowRendererLocal(this, this.shadowRenderer);
		this._shadowRendererDirectional = new ShadowRendererDirectional(this, this.shadowRenderer);
		this._renderPassUpdateClustered = new RenderPassUpdateClustered(this.device, this, this.shadowRenderer, this._shadowRendererLocal, this.lightTextureAtlas);
		this.viewUniformFormat = null;
		this.viewBindGroupFormat = null;
		this._skinTime = 0;
		this._morphTime = 0;
		this._cullTime = 0;
		this._shadowMapTime = 0;
		this._lightClustersTime = 0;
		this._layerCompositionUpdateTime = 0;
		this._shadowDrawCalls = 0;
		this._skinDrawCalls = 0;
		this._instancedDrawCalls = 0;
		this._shadowMapUpdates = 0;
		this._numDrawCallsCulled = 0;
		this._camerasRendered = 0;
		this._lightClusters = 0;
		const scope = graphicsDevice.scope;
		this.boneTextureId = scope.resolve('texture_poseMap');
		this.boneTextureSizeId = scope.resolve('texture_poseMapSize');
		this.poseMatrixId = scope.resolve('matrix_pose[0]');
		this.modelMatrixId = scope.resolve('matrix_model');
		this.normalMatrixId = scope.resolve('matrix_normal');
		this.viewInvId = scope.resolve('matrix_viewInverse');
		this.viewPos = new Float32Array(3);
		this.viewPosId = scope.resolve('view_position');
		this.projId = scope.resolve('matrix_projection');
		this.projSkyboxId = scope.resolve('matrix_projectionSkybox');
		this.viewId = scope.resolve('matrix_view');
		this.viewId3 = scope.resolve('matrix_view3');
		this.viewProjId = scope.resolve('matrix_viewProjection');
		this.flipYId = scope.resolve('projectionFlipY');
		this.tbnBasis = scope.resolve('tbnBasis');
		this.nearClipId = scope.resolve('camera_near');
		this.farClipId = scope.resolve('camera_far');
		this.cameraParams = new Float32Array(4);
		this.cameraParamsId = scope.resolve('camera_params');
		this.viewIndexId = scope.resolve('view_index');
		this.blueNoiseJitterId = scope.resolve('blueNoiseJitter');
		this.blueNoiseTextureId = scope.resolve('blueNoiseTex32');
		this.alphaTestId = scope.resolve('alpha_ref');
		this.opacityMapId = scope.resolve('texture_opacityMap');
		this.exposureId = scope.resolve('exposure');
		this.twoSidedLightingNegScaleFactorId = scope.resolve('twoSidedLightingNegScaleFactor');
		this.twoSidedLightingNegScaleFactorId.setValue(0);
		this.morphWeightsA = scope.resolve('morph_weights_a');
		this.morphWeightsB = scope.resolve('morph_weights_b');
		this.morphPositionTex = scope.resolve('morphPositionTex');
		this.morphNormalTex = scope.resolve('morphNormalTex');
		this.morphTexParams = scope.resolve('morph_tex_params');
		this.lightCube = new LightCube();
		this.constantLightCube = scope.resolve('lightCube[0]');
	}
	destroy() {
		this.shadowRenderer = null;
		this._shadowRendererLocal = null;
		this._shadowRendererDirectional = null;
		this.shadowMapCache.destroy();
		this.shadowMapCache = null;
		this._renderPassUpdateClustered.destroy();
		this._renderPassUpdateClustered = null;
		this.lightTextureAtlas.destroy();
		this.lightTextureAtlas = null;
	}
	sortCompare(drawCallA, drawCallB) {
		if (drawCallA.layer === drawCallB.layer) {
			if (drawCallA.drawOrder && drawCallB.drawOrder) {
				return drawCallA.drawOrder - drawCallB.drawOrder;
			} else if (drawCallA.zdist && drawCallB.zdist) {
				return drawCallB.zdist - drawCallA.zdist;
			} else if (drawCallA.zdist2 && drawCallB.zdist2) {
				return drawCallA.zdist2 - drawCallB.zdist2;
			}
		}
		return drawCallB._key[SORTKEY_FORWARD] - drawCallA._key[SORTKEY_FORWARD];
	}
	sortCompareMesh(drawCallA, drawCallB) {
		if (drawCallA.layer === drawCallB.layer) {
			if (drawCallA.drawOrder && drawCallB.drawOrder) {
				return drawCallA.drawOrder - drawCallB.drawOrder;
			} else if (drawCallA.zdist && drawCallB.zdist) {
				return drawCallB.zdist - drawCallA.zdist;
			}
		}
		const keyA = drawCallA._key[SORTKEY_FORWARD];
		const keyB = drawCallB._key[SORTKEY_FORWARD];
		if (keyA === keyB && drawCallA.mesh && drawCallB.mesh) {
			return drawCallB.mesh.id - drawCallA.mesh.id;
		}
		return keyB - keyA;
	}
	sortCompareDepth(drawCallA, drawCallB) {
		const keyA = drawCallA._key[SORTKEY_DEPTH];
		const keyB = drawCallB._key[SORTKEY_DEPTH];
		if (keyA === keyB && drawCallA.mesh && drawCallB.mesh) {
			return drawCallB.mesh.id - drawCallA.mesh.id;
		}
		return keyB - keyA;
	}
	setupViewport(camera, renderTarget) {
		const device = this.device;
		const pixelWidth = renderTarget ? renderTarget.width : device.width;
		const pixelHeight = renderTarget ? renderTarget.height : device.height;
		const rect = camera.rect;
		let x = Math.floor(rect.x * pixelWidth);
		let y = Math.floor(rect.y * pixel