/**
 * Copyright 2014-present Palantir Technologies
 * @license MIT
 */
"use strict";
var __extends = (this && this.__extends) || function (d, b) {
    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
    function __() { this.constructor = d; }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
};
var component_1 = require("./component");
/**
 * Returns next grid position based on tick value
 *
 * @param scale Scale used by the grid
 * @param between Value denoting whether the grid renders between ticks or on ticks
 * @param orderedTicks All ticks in order. only needed when rendering lines between ticks
 */
function gridPositionFactory(scale, between, orderedTicks) {
    var previousTick = {};
    if (orderedTicks !== undefined) {
        for (var i = 0; i < orderedTicks.length; i++) {
            var previous = orderedTicks[i - 1];
            var current = orderedTicks[i];
            previousTick[current] = previous;
        }
    }
    return function (tickVal) {
        var position = scale.scale(tickVal);
        if (!between) {
            return position;
        }
        var gridPosition;
        var previousPosition = previousTick[tickVal] === undefined
            ? undefined
            : scale.scale(previousTick[tickVal]);
        if (previousPosition !== undefined) {
            gridPosition = previousPosition + (position - previousPosition) / 2;
        }
        return gridPosition;
    };
}
var Gridlines = (function (_super) {
    __extends(Gridlines, _super);
    /**
     * @constructor
     * @param {Scale} xScale The scale to base the x gridlines on. Pass null if no gridlines are desired.
     * @param {Scale} yScale The scale to base the y gridlines on. Pass null if no gridlines are desired.
     */
    function Gridlines(xScale, yScale) {
        var _this = _super.call(this) || this;
        _this.addClass("gridlines");
        _this._xScale = xScale;
        _this._yScale = yScale;
        _this._renderCallback = function (scale) { return _this.render(); };
        if (_this._xScale) {
            _this._xScale.onUpdate(_this._renderCallback);
        }
        if (_this._yScale) {
            _this._yScale.onUpdate(_this._renderCallback);
        }
        return _this;
    }
    Gridlines.prototype.betweenX = function (_betweenX) {
        if (_betweenX === undefined) {
            return this._betweenX;
        }
        if (_betweenX !== this._betweenX) {
            this._betweenX = _betweenX;
            this.render();
        }
        return this;
    };
    Gridlines.prototype.betweenY = function (_betweenY) {
        if (_betweenY === undefined) {
            return this._betweenY;
        }
        if (_betweenY !== this._betweenY) {
            this._betweenY = _betweenY;
            this.render();
        }
        return this;
    };
    Gridlines.prototype.destroy = function () {
        _super.prototype.destroy.call(this);
        if (this._xScale) {
            this._xScale.offUpdate(this._renderCallback);
        }
        if (this._yScale) {
            this._yScale.offUpdate(this._renderCallback);
        }
        return this;
    };
    Gridlines.prototype._setup = function () {
        _super.prototype._setup.call(this);
        this._xLinesContainer = this.content().append("g").classed("x-gridlines", true);
        this._yLinesContainer = this.content().append("g").classed("y-gridlines", true);
    };
    Gridlines.prototype.renderImmediately = function () {
        _super.prototype.renderImmediately.call(this);
        this._redrawXLines();
        this._redrawYLines();
        return this;
    };
    Gridlines.prototype.computeLayout = function (origin, availableWidth, availableHeight) {
        _super.prototype.computeLayout.call(this, origin, availableWidth, availableHeight);
        if (this._xScale != null) {
            this._xScale.range([0, this.width()]);
        }
        if (this._yScale != null) {
            this._yScale.range([this.height(), 0]);
        }
        return this;
    };
    Gridlines.prototype._redrawXLines = function () {
        if (this._xScale) {
            var between = this.betweenX();
            var xTicks = this._xScale.ticks().slice(between ? 1 : 0);
            var xLinesUpdate = this._xLinesContainer.selectAll("line").data(xTicks);
            var xLines = xLinesUpdate.enter().append("line").merge(xLinesUpdate);
            xLines.attr("x1", gridPositionFactory(this._xScale, between, this._xScale.ticks()))
                .attr("y1", 0)
                .attr("x2", gridPositionFactory(this._xScale, between, this._xScale.ticks()))
                .attr("y2", this.height())
                .classed("betweenline", between)
                .classed("zeroline", function (t) { return t === 0; });
            xLinesUpdate.exit().remove();
        }
    };
    Gridlines.prototype._redrawYLines = function () {
        if (this._yScale) {
            var between = this.betweenY();
            var yTicks = this._yScale.ticks().slice(between ? 1 : 0);
            var yLinesUpdate = this._yLinesContainer.selectAll("line").data(yTicks);
            var yLines = yLinesUpdate.enter().append("line").merge(yLinesUpdate);
            yLines.attr("x1", 0)
                .attr("y1", gridPositionFactory(this._yScale, between, this._yScale.ticks()))
                .attr("x2", this.width())
                .attr("y2", gridPositionFactory(this._yScale, between, this._yScale.ticks()))
                .classed("betweenline", between)
                .classed("zeroline", function (t) { return t === 0; });
            yLinesUpdate.exit().remove();
        }
    };
    return Gridlines;
}(component_1.Component));
exports.Gridlines = Gridlines;
