/**
 * Copyright 2014-present Palantir Technologies
 * @license MIT
 */
"use strict";
var d3 = require("d3");
/**
 * A CanvasDrawer draws data onto a supplied Canvas Context.
 *
 * This class is immutable (but has internal state) and shouldn't be extended.
 */
var CanvasDrawer = (function () {
    /**
     * @param _context The context for a canvas that this drawer will draw to.
     * @param _drawStep The draw step logic that actually draws.
     */
    function CanvasDrawer(_context, _drawStep) {
        this._context = _context;
        this._drawStep = _drawStep;
    }
    // public for testing
    CanvasDrawer.prototype.getDrawStep = function () {
        return this._drawStep;
    };
    CanvasDrawer.prototype.draw = function (data, appliedDrawSteps) {
        var projector = appliedDrawSteps[appliedDrawSteps.length - 1].attrToAppliedProjector;
        // don't support animations for now; just draw the last draw step immediately
        this._context.save();
        this._drawStep(this._context, data, projector);
        this._context.restore();
    };
    CanvasDrawer.prototype.getVisualPrimitives = function () {
        return [];
    };
    CanvasDrawer.prototype.getVisualPrimitiveAtIndex = function (index) {
        return null;
    };
    CanvasDrawer.prototype.remove = function () {
        // NO op - canvas element owns the canvas; context is free
    };
    return CanvasDrawer;
}());
exports.CanvasDrawer = CanvasDrawer;
exports.ContextStyleAttrs = [
    "fill-opacity",
    "fill",
    "opacity",
    "stroke-opacity",
    "stroke-width",
    "stroke",
];
function resolveAttributesSubsetWithStyles(projector, extraKeys, datum, index) {
    var attrKeys = exports.ContextStyleAttrs.concat(extraKeys);
    return resolveAttributes(projector, attrKeys, datum, index);
}
exports.resolveAttributesSubsetWithStyles = resolveAttributesSubsetWithStyles;
function resolveAttributes(projector, attrKeys, datum, index) {
    var attrs = {};
    for (var _i = 0, attrKeys_1 = attrKeys; _i < attrKeys_1.length; _i++) {
        var attrKey = attrKeys_1[_i];
        if (projector.hasOwnProperty(attrKey)) {
            attrs[attrKey] = projector[attrKey](datum, index);
        }
    }
    return attrs;
}
exports.resolveAttributes = resolveAttributes;
function getStrokeOpacity(style) {
    var baseOpacity = style["opacity"] != null ? parseFloat(style["opacity"]) : 1;
    var strokeOpacity = style["stroke-opacity"] != null ? parseFloat(style["stroke-opacity"]) : 1;
    return strokeOpacity * baseOpacity;
}
function getFillOpacity(style) {
    var baseOpacity = style["opacity"] != null ? parseFloat(style["opacity"]) : 1;
    var fillOpacity = style["fill-opacity"] != null ? parseFloat(style["fill-opacity"]) : 1;
    return fillOpacity * baseOpacity;
}
function getStrokeWidth(style) {
    return style["stroke-width"] != null ? parseFloat(style["stroke-width"]) : 1;
}
function renderArea(context, d3Area, data, style) {
    context.save();
    context.beginPath();
    d3Area.context(context);
    d3Area(data);
    context.lineJoin = "round";
    renderPathWithStyle(context, style);
    context.restore();
}
exports.renderArea = renderArea;
function renderLine(context, d3Line, data, style) {
    context.save();
    context.beginPath();
    d3Line.context(context);
    d3Line(data);
    context.lineJoin = "round";
    renderPathWithStyle(context, style);
    context.restore();
}
exports.renderLine = renderLine;
function renderPathWithStyle(context, style) {
    if (style["stroke"]) {
        context.lineWidth = getStrokeWidth(style);
        var strokeColor = d3.color(style["stroke"]);
        strokeColor.opacity = getStrokeOpacity(style);
        context.strokeStyle = strokeColor.toString();
        context.stroke();
    }
    if (style["fill"]) {
        var fillColor = d3.color(style["fill"]);
        fillColor.opacity = getFillOpacity(style);
        context.fillStyle = fillColor.toString();
        context.fill();
    }
}
exports.renderPathWithStyle = renderPathWithStyle;
