import { Dataset } from "../core/dataset";
import { DatumFormatter } from "../core/formatters";
import { AttributeToProjector, Bounds, IAccessor, Point, Range } from "../core/interfaces";
import * as Drawers from "../drawers";
import { ProxyDrawer } from "../drawers/drawer";
import { Scale } from "../scales/scale";
import * as Utils from "../utils";
import * as Plots from "./";
import { IPlotEntity } from "./";
import { XYPlot } from "./xyPlot";
export declare const BarOrientation: {
    vertical: "vertical";
    horizontal: "horizontal";
};
export declare type BarOrientation = keyof typeof BarOrientation;
export declare const LabelsPosition: {
    start: "start";
    end: "end";
    middle: "middle";
    outside: "outside";
};
export declare type LabelsPosition = keyof typeof LabelsPosition;
export declare const BarAlignment: {
    start: "start";
    end: "end";
    middle: "middle";
};
export declare type BarAlignment = keyof typeof BarAlignment;
export declare class Bar<X, Y> extends XYPlot<X, Y> {
    static _BAR_THICKNESS_RATIO: number;
    static _SINGLE_BAR_DIMENSION_RATIO: number;
    private static _BAR_AREA_CLASS;
    private static _BAR_END_KEY;
    protected static _BAR_THICKNESS_KEY: string;
    protected static _LABEL_AREA_CLASS: string;
    protected static _LABEL_PADDING: number;
    private _baseline;
    private _baselineValue;
    protected _isVertical: boolean;
    private _labelFormatter;
    private _labelsEnabled;
    private _labelsPosition;
    private _hideBarsIfAnyAreTooWide;
    private _labelConfig;
    private _baselineValueProvider;
    private _barAlignment;
    /**
     * Whether all the bars in this barPlot have the same pixel thickness.
     * If so, use the _barPixelThickness property to access the thickness.
     */
    private _fixedBarPixelThickness;
    private _barPixelThickness;
    private _updateBarPixelThicknessCallback;
    /**
     * A Bar Plot draws bars growing out from a baseline to some value
     *
     * @constructor
     * @param {string} [orientation="vertical"] One of "vertical"/"horizontal".
     */
    constructor(orientation?: BarOrientation);
    computeLayout(origin?: Point, availableWidth?: number, availableHeight?: number): this;
    x(): Plots.ITransformableAccessorScaleBinding<X, number>;
    x(x: number | IAccessor<number>): this;
    x(x: X | IAccessor<X>, xScale: Scale<X, number>): this;
    y(): Plots.ITransformableAccessorScaleBinding<Y, number>;
    y(y: number | IAccessor<number>): this;
    y(y: Y | IAccessor<Y>, yScale: Scale<Y, number>): this;
    /**
     * The binding associated with bar length. Length is the count or value the bar is trying to show.
     * This is the .y() for a vertical plot and .x() for a horizontal plot.
     */
    protected length(): Plots.ITransformableAccessorScaleBinding<any, number>;
    /**
     * The binding associated with bar position. Position separates the different bar categories.
     * This is the .x() for a vertical plot and .y() for a horizontal plot.
     */
    protected position(): Plots.ITransformableAccessorScaleBinding<any, number>;
    /**
     * Gets the accessor for the bar "end", which is used to compute the width of
     * each bar on the independent axis.
     */
    barEnd(): Plots.ITransformableAccessorScaleBinding<X, number>;
    /**
     * Sets the accessor for the bar "end", which is used to compute the width of
     * each bar on the x axis (y axis if horizontal).
     *
     * If a `Scale` has been set for the independent axis via the `x` method (`y`
     * if horizontal), it will also be used to scale `barEnd`.
     *
     * Additionally, calling this setter will set `barAlignment` to `"start"`,
     * indicating that `x` and `barEnd` now define the left and right x
     * coordinates of a bar (bottom/top if horizontal).
     *
     * Normally, a single bar width for all bars is determined by how many bars
     * can fit in the given space (minus some padding). Settings this accessor
     * will override this behavior and manually set the start and end coordinates
     * for each bar.
     *
     * This means it will totally ignore the range band width of category scales,
     * so this probably doesn't make sense to use with category axes.
     */
    barEnd(end: number | IAccessor<number> | X | IAccessor<X>): this;
    /**
     * Gets the current bar alignment
     */
    barAlignment(): BarAlignment;
    /**
     * Sets the bar alignment. Valid values are `"start"`, `"middle"`, and
     * `"end"`, which determines the meaning of the accessor of the bar's scale
     * coordinate (e.g. "x" for vertical bars).
     *
     * For example, a value of "start" means the x coordinate accessor sets the
     * left hand side of the rect.
     *
     * The default value is "middle", which aligns to rect so that it centered on
     * the x coordinate
     */
    barAlignment(align: BarAlignment): this;
    /**
     * Gets the orientation of the plot
     *
     * @return "vertical" | "horizontal"
     */
    orientation(): BarOrientation;
    protected _createDrawer(): ProxyDrawer;
    protected _setup(): void;
    /**
     * Gets the baseline value.
     * The baseline is the line that the bars are drawn from.
     *
     * @returns {X|Y}
     */
    baselineValue(): X | Y;
    /**
     * Sets the baseline value.
     * The baseline is the line that the bars are drawn from.
     *
     * @param {X|Y} value
     * @returns {Bar} The calling Bar Plot.
     */
    baselineValue(value: X | Y): this;
    addDataset(dataset: Dataset): this;
    protected _addDataset(dataset: Dataset): this;
    removeDataset(dataset: Dataset): this;
    protected _removeDataset(dataset: Dataset): this;
    datasets(): Dataset[];
    datasets(datasets: Dataset[]): this;
    /**
     * Get whether bar labels are enabled.
     *
     * @returns {boolean} Whether bars should display labels or not.
     */
    labelsEnabled(): boolean;
    /**
     * Sets whether labels are enabled. If enabled, also sets their position relative to the baseline.
     *
     * @param {boolean} labelsEnabled
     * @param {LabelsPosition} labelsPosition
     * @returns {Bar} The calling Bar Plot.
     */
    labelsEnabled(enabled: boolean): this;
    labelsEnabled(enabled: boolean, labelsPosition: LabelsPosition): this;
    /**
     * Gets the Formatter for the labels.
     */
    labelFormatter(): DatumFormatter;
    /**
     * Sets the Formatter for the labels. The labelFormatter will be fed each bar's
     * computed height as defined by the `.y()` accessor for vertical bars, or the
     * width as defined by the `.x()` accessor for horizontal bars, as well as the
     * datum, datum index, and dataset associated with that bar.
     *
     * @param {Formatter} formatter
     * @returns {Bar} The calling Bar Plot.
     */
    labelFormatter(formatter: DatumFormatter): this;
    protected _createNodesForDataset(dataset: Dataset): ProxyDrawer;
    protected _removeDatasetNodes(dataset: Dataset): void;
    /**
     * Returns the PlotEntity nearest to the query point according to the following algorithm:
     *   - If the query point is inside a bar, returns the PlotEntity for that bar.
     *   - Otherwise, gets the nearest PlotEntity by the primary direction (X for vertical, Y for horizontal),
     *     breaking ties with the secondary direction.
     * Returns undefined if no PlotEntity can be found.
     *
     * @param {Point} queryPoint
     * @returns {PlotEntity} The nearest PlotEntity, or undefined if no PlotEntity can be found.
     */
    entityNearest(queryPoint: Point): IPlotEntity;
    /**
     * Gets the Entities at a particular Point.
     *
     * @param {Point} p
     * @returns {PlotEntity[]}
     */
    entitiesAt(p: Point): IPlotEntity[];
    /**
     * Gets the Entities that intersect the Bounds.
     *
     * @param {Bounds} bounds
     * @returns {PlotEntity[]}
     */
    entitiesIn(bounds: Bounds): IPlotEntity[];
    /**
     * Gets the Entities that intersect the area defined by the ranges.
     *
     * @param {Range} xRange
     * @param {Range} yRange
     * @returns {PlotEntity[]}
     */
    entitiesIn(xRange: Range, yRange: Range): IPlotEntity[];
    private _entitiesIntersecting(xValOrRange, yValOrRange);
    private _updateLengthScale();
    protected _additionalPaint(time: number): void;
    /**
     * Makes sure the extent takes into account the widths of the bars
     */
    protected _extentsForProperty(property: string): any[];
    /**
     * Return the <rect>'s x or y attr value given the position and thickness of
     * that bar. This method is responsible for account for barAlignment, in particular.
     */
    protected _getPositionAttr(position: number, thickness: number): number;
    protected _drawLabels(): void;
    private _drawLabel(datum, index, dataset, attrToProjector);
    private _getShowLabelOnBar(barCoordinates, barDimensions, measurement);
    private _calculateLabelProperties(barCoordinates, barDimensions, measurement, showLabelOnBar, aboveOrLeftOfBaseline);
    private _createLabelContainer(labelArea, labelContainerOrigin, labelOrigin, measurement, showLabelOnBar, color);
    protected _generateDrawSteps(): Drawers.DrawStep[];
    protected _generateAttrToProjector(): AttributeToProjector;
    protected _updateThicknessAttr(): void;
    private _updateBarPixelWidth();
    entities(datasets?: Dataset[]): IPlotEntity[];
    protected _entityBounds(entity: Plots.IPlotEntity | Plots.ILightweightPlotEntity): {
        x: any;
        y: any;
        width: any;
        height: any;
    };
    /**
     * The rectangular bounds of a bar. Note that the x/y coordinates are not the
     * same as the "pixel point" because they are always at the top/left of the
     * bar.
     */
    protected _pixelBounds(datum: any, index: number, dataset: Dataset): {
        x: any;
        y: any;
        width: any;
        height: any;
    };
    /**
     * The "pixel point" of a bar is the farthest point from the baseline.
     *
     * For example, in a vertical bar chart with positive bar values, the pixel
     * point will be at the top of the bar. For negative bar values, the pixel
     * point will be at the bottom of the bar.
     */
    protected _pixelPoint(datum: any, index: number, dataset: Dataset): Point;
    private _pixelPointBar(originalPosition, scaledBaseline, rect);
    protected _uninstallScaleForKey(scale: Scale<any, number>, key: string): void;
    protected _getDataToDraw(): Utils.Map<Dataset, any[]>;
}
