/**
 * Copyright 2014-present Palantir Technologies
 * @license MIT
 */
"use strict";
var __extends = (this && this.__extends) || function (d, b) {
    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
    function __() { this.constructor = d; }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
};
var d3 = require("d3");
var Typesettable = require("typesettable");
var Animators = require("../animators");
var Formatters = require("../core/formatters");
var Drawers = require("../drawers");
var drawer_1 = require("../drawers/drawer");
var rectangleDrawer_1 = require("../drawers/rectangleDrawer");
var Scales = require("../scales");
var quantitativeScale_1 = require("../scales/quantitativeScale");
var Utils = require("../utils");
var makeEnum_1 = require("../utils/makeEnum");
var Plots = require("./");
var plot_1 = require("./plot");
var xyPlot_1 = require("./xyPlot");
exports.BarOrientation = makeEnum_1.makeEnum(["vertical", "horizontal"]);
exports.LabelsPosition = makeEnum_1.makeEnum(["start", "middle", "end", "outside"]);
exports.BarAlignment = makeEnum_1.makeEnum(["start", "middle", "end"]);
var Bar = (function (_super) {
    __extends(Bar, _super);
    /**
     * A Bar Plot draws bars growing out from a baseline to some value
     *
     * @constructor
     * @param {string} [orientation="vertical"] One of "vertical"/"horizontal".
     */
    function Bar(orientation) {
        if (orientation === void 0) { orientation = "vertical"; }
        var _this = _super.call(this) || this;
        _this._labelFormatter = Formatters.identity();
        _this._labelsEnabled = false;
        _this._labelsPosition = exports.LabelsPosition.end;
        _this._hideBarsIfAnyAreTooWide = true;
        _this._barAlignment = "middle";
        /**
         * Whether all the bars in this barPlot have the same pixel thickness.
         * If so, use the _barPixelThickness property to access the thickness.
         */
        _this._fixedBarPixelThickness = true;
        _this._barPixelThickness = 0;
        _this.addClass("bar-plot");
        if (orientation !== "vertical" && orientation !== "horizontal") {
            throw new Error(orientation + " is not a valid orientation for Plots.Bar");
        }
        _this._isVertical = orientation === "vertical";
        _this.animator("baseline", new Animators.Null());
        _this.attr("fill", new Scales.Color().range()[0]);
        _this.attr(Bar._BAR_THICKNESS_KEY, function () { return _this._barPixelThickness; });
        _this._labelConfig = new Utils.Map();
        _this._baselineValueProvider = function () { return [_this.baselineValue()]; };
        _this._updateBarPixelThicknessCallback = function () { return _this._updateBarPixelWidth(); };
        return _this;
    }
    Bar.prototype.computeLayout = function (origin, availableWidth, availableHeight) {
        _super.prototype.computeLayout.call(this, origin, availableWidth, availableHeight);
        this._updateBarPixelWidth();
        this._updateExtents();
        return this;
    };
    Bar.prototype.x = function (x, xScale) {
        if (x == null) {
            return _super.prototype.x.call(this);
        }
        if (xScale == null) {
            _super.prototype.x.call(this, x);
        }
        else {
            _super.prototype.x.call(this, x, xScale);
            xScale.onUpdate(this._updateBarPixelThicknessCallback);
        }
        this._updateThicknessAttr();
        this._updateLengthScale();
        return this;
    };
    Bar.prototype.y = function (y, yScale) {
        if (y == null) {
            return _super.prototype.y.call(this);
        }
        if (yScale == null) {
            _super.prototype.y.call(this, y);
        }
        else {
            _super.prototype.y.call(this, y, yScale);
            yScale.onUpdate(this._updateBarPixelThicknessCallback);
        }
        this._updateLengthScale();
        return this;
    };
    /**
     * The binding associated with bar length. Length is the count or value the bar is trying to show.
     * This is the .y() for a vertical plot and .x() for a horizontal plot.
     */
    Bar.prototype.length = function () {
        return this._isVertical ? this.y() : this.x();
    };
    /**
     * The binding associated with bar position. Position separates the different bar categories.
     * This is the .x() for a vertical plot and .y() for a horizontal plot.
     */
    Bar.prototype.position = function () {
        return this._isVertical ? this.x() : this.y();
    };
    Bar.prototype.barEnd = function (end) {
        if (end == null) {
            return this._propertyBindings.get(Bar._BAR_END_KEY);
        }
        var binding = this.position();
        var scale = binding && binding.scale;
        this._bindProperty(Bar._BAR_END_KEY, end, scale);
        this._updateThicknessAttr();
        this._updateLengthScale();
        this.render();
        return this;
    };
    Bar.prototype.barAlignment = function (align) {
        if (align == null) {
            return this._barAlignment;
        }
        this._barAlignment = align;
        this._clearAttrToProjectorCache();
        this.render();
        return this;
    };
    /**
     * Gets the orientation of the plot
     *
     * @return "vertical" | "horizontal"
     */
    Bar.prototype.orientation = function () {
        return this._isVertical ? "vertical" : "horizontal";
    };
    Bar.prototype._createDrawer = function () {
        return new drawer_1.ProxyDrawer(function () { return new rectangleDrawer_1.RectangleSVGDrawer(Bar._BAR_AREA_CLASS); }, function (ctx) { return new Drawers.RectangleCanvasDrawer(ctx); });
    };
    Bar.prototype._setup = function () {
        _super.prototype._setup.call(this);
        this._baseline = this._renderArea.append("line").classed("baseline", true);
    };
    Bar.prototype.baselineValue = function (value) {
        if (value == null) {
            if (this._baselineValue != null) {
                return this._baselineValue;
            }
            if (!this._projectorsReady()) {
                return 0;
            }
            var lengthScale = this.length().scale;
            if (!lengthScale) {
                return 0;
            }
            if (lengthScale instanceof Scales.Time) {
                return new Date(0);
            }
            return 0;
        }
        this._baselineValue = value;
        this._updateLengthScale();
        this._clearAttrToProjectorCache();
        this.render();
        return this;
    };
    Bar.prototype.addDataset = function (dataset) {
        _super.prototype.addDataset.call(this, dataset);
        this._updateBarPixelWidth();
        return this;
    };
    Bar.prototype._addDataset = function (dataset) {
        dataset.onUpdate(this._updateBarPixelThicknessCallback);
        _super.prototype._addDataset.call(this, dataset);
        return this;
    };
    Bar.prototype.removeDataset = function (dataset) {
        dataset.offUpdate(this._updateBarPixelThicknessCallback);
        _super.prototype.removeDataset.call(this, dataset);
        this._updateBarPixelWidth();
        return this;
    };
    Bar.prototype._removeDataset = function (dataset) {
        dataset.offUpdate(this._updateBarPixelThicknessCallback);
        _super.prototype._removeDataset.call(this, dataset);
        return this;
    };
    Bar.prototype.datasets = function (datasets) {
        if (datasets == null) {
            return _super.prototype.datasets.call(this);
        }
        _super.prototype.datasets.call(this, datasets);
        this._updateBarPixelWidth();
        return this;
    };
    Bar.prototype.labelsEnabled = function (enabled, labelsPosition) {
        if (enabled == null) {
            return this._labelsEnabled;
        }
        else {
            this._labelsEnabled = enabled;
            if (labelsPosition != null) {
                this._labelsPosition = labelsPosition;
            }
            this._clearAttrToProjectorCache();
            this.render();
            return this;
        }
    };
    Bar.prototype.labelFormatter = function (formatter) {
        if (formatter == null) {
            return this._labelFormatter;
        }
        else {
            this._labelFormatter = formatter;
            this._clearAttrToProjectorCache();
            this.render();
            return this;
        }
    };
    Bar.prototype._createNodesForDataset = function (dataset) {
        var drawer = _super.prototype._createNodesForDataset.call(this, dataset);
        var labelArea = this._renderArea.append("g").classed(Bar._LABEL_AREA_CLASS, true);
        var context = new Typesettable.SvgContext(labelArea.node());
        var measurer = new Typesettable.CacheMeasurer(context);
        var writer = new Typesettable.Writer(measurer, context);
        this._labelConfig.set(dataset, { labelArea: labelArea, measurer: measurer, writer: writer });
        return drawer;
    };
    Bar.prototype._removeDatasetNodes = function (dataset) {
        _super.prototype._removeDatasetNodes.call(this, dataset);
        var labelConfig = this._labelConfig.get(dataset);
        if (labelConfig != null) {
            labelConfig.labelArea.remove();
            this._labelConfig.delete(dataset);
        }
    };
    /**
     * Returns the PlotEntity nearest to the query point according to the following algorithm:
     *   - If the query point is inside a bar, returns the PlotEntity for that bar.
     *   - Otherwise, gets the nearest PlotEntity by the primary direction (X for vertical, Y for horizontal),
     *     breaking ties with the secondary direction.
     * Returns undefined if no PlotEntity can be found.
     *
     * @param {Point} queryPoint
     * @returns {PlotEntity} The nearest PlotEntity, or undefined if no PlotEntity can be found.
     */
    Bar.prototype.entityNearest = function (queryPoint) {
        var _this = this;
        var queryPtPrimary = this._isVertical ? queryPoint.x : queryPoint.y;
        var queryPtSecondary = this._isVertical ? queryPoint.y : queryPoint.x;
        // SVGRects are positioned with sub-pixel accuracy (the default unit
        // for the x, y, height & width attributes), but user selections (e.g. via
        // mouse events) usually have pixel accuracy. We add a tolerance of 0.5 pixels.
        var tolerance = 0.5;
        // PERF: precompute all these values to prevent recomputing for each entity
        var chartBounds = this.bounds();
        var chartWidth = chartBounds.bottomRight.x - chartBounds.topLeft.x;
        var chartHeight = chartBounds.bottomRight.y - chartBounds.topLeft.y;
        var xRange = { min: 0, max: chartWidth };
        var yRange = { min: 0, max: chartHeight };
        var originalPositionProjector = (this._isVertical ? plot_1.Plot._scaledAccessor(this.y()) : plot_1.Plot._scaledAccessor(this.x()));
        var scaledBaseline = (this._isVertical ? this.y().scale : this.x().scale).scale(this.baselineValue());
        var plotPointFactory = function (datum, index, dataset, rect) {
            return _this._pixelPointBar(originalPositionProjector(datum, index, dataset), scaledBaseline, rect);
        };
        var minPrimaryDist = Infinity;
        var minSecondaryDist = Infinity;
        var closest;
        this._getEntityStore().entities().forEach(function (entity) {
            var barBBox = _this._entityBounds(entity);
            if (!Utils.DOM.intersectsBBox(xRange, yRange, barBBox)) {
                return;
            }
            var primaryDist = 0;
            var secondaryDist = 0;
            // if we're inside a bar, distance in both directions should stay 0
            if (!Utils.DOM.intersectsBBox(queryPoint.x, queryPoint.y, barBBox, tolerance)) {
                var plotPt = plotPointFactory(entity.datum, entity.index, entity.dataset, barBBox);
                var plotPtPrimary = _this._isVertical ? plotPt.x : plotPt.y;
                primaryDist = Math.abs(queryPtPrimary - plotPtPrimary);
                // compute this bar's min and max along the secondary axis
                var barMinSecondary = _this._isVertical ? barBBox.y : barBBox.x;
                var barMaxSecondary = barMinSecondary + (_this._isVertical ? barBBox.height : barBBox.width);
                if (queryPtSecondary >= barMinSecondary - tolerance && queryPtSecondary <= barMaxSecondary + tolerance) {
                    // if we're within a bar's secondary axis span, it is closest in that direction
                    secondaryDist = 0;
                }
                else {
                    var plotPtSecondary = _this._isVertical ? plotPt.y : plotPt.x;
                    secondaryDist = Math.abs(queryPtSecondary - plotPtSecondary);
                }
            }
            // if we find a closer bar, record its distance and start new closest lists
            if (primaryDist < minPrimaryDist
                || primaryDist === minPrimaryDist && secondaryDist < minSecondaryDist) {
                closest = entity;
                minPrimaryDist = primaryDist;
                minSecondaryDist = secondaryDist;
            }
        });
        if (closest !== undefined) {
            return this._lightweightPlotEntityToPlotEntity(closest);
        }
        else {
            return undefined;
        }
    };
    /**
     * Gets the Entities at a particular Point.
     *
     * @param {Point} p
     * @returns {PlotEntity[]}
     */
    Bar.prototype.entitiesAt = function (p) {
        return this._entitiesIntersecting(p.x, p.y);
    };
    Bar.prototype.entitiesIn = function (xRangeOrBounds, yRange) {
        var dataXRange;
        var dataYRange;
        if (yRange == null) {
            var bounds = xRangeOrBounds;
            dataXRange = { min: bounds.topLeft.x, max: bounds.bottomRight.x };
            dataYRange = { min: bounds.topLeft.y, max: bounds.bottomRight.y };
        }
        else {
            dataXRange = xRangeOrBounds;
            dataYRange = yRange;
        }
        return this._entitiesIntersecting(dataXRange, dataYRange);
    };
    Bar.prototype._entitiesIntersecting = function (xValOrRange, yValOrRange) {
        var _this = this;
        var intersected = [];
        this._getEntityStore().entities().forEach(function (entity) {
            if (Utils.DOM.intersectsBBox(xValOrRange, yValOrRange, _this._entityBounds(entity))) {
                intersected.push(_this._lightweightPlotEntityToPlotEntity(entity));
            }
        });
        return intersected;
    };
    Bar.prototype._updateLengthScale = function () {
        if (!this._projectorsReady()) {
            return;
        }
        var lengthScale = this.length().scale;
        if (lengthScale instanceof quantitativeScale_1.QuantitativeScale) {
            lengthScale.addPaddingExceptionsProvider(this._baselineValueProvider);
            lengthScale.addIncludedValuesProvider(this._baselineValueProvider);
        }
    };
    Bar.prototype._additionalPaint = function (time) {
        var _this = this;
        var lengthScale = this.length().scale;
        var scaledBaseline = lengthScale.scale(this.baselineValue());
        var baselineAttr = {
            "x1": this._isVertical ? 0 : scaledBaseline,
            "y1": this._isVertical ? scaledBaseline : 0,
            "x2": this._isVertical ? this.width() : scaledBaseline,
            "y2": this._isVertical ? scaledBaseline : this.height(),
        };
        this._getAnimator("baseline").animate(this._baseline, baselineAttr);
        this.datasets().forEach(function (dataset) { return _this._labelConfig.get(dataset).labelArea.selectAll("g").remove(); });
        if (this._labelsEnabled) {
            Utils.Window.setTimeout(function () { return _this._drawLabels(); }, time);
        }
    };
    /**
     * Makes sure the extent takes into account the widths of the bars
     */
    Bar.prototype._extentsForProperty = function (property) {
        var _this = this;
        var extents = _super.prototype._extentsForProperty.call(this, property);
        var accScaleBinding;
        if (property === "x" && this._isVertical) {
            accScaleBinding = this.x();
        }
        else if (property === "y" && !this._isVertical) {
            accScaleBinding = this.y();
        }
        else {
            return extents;
        }
        if (!(accScaleBinding && accScaleBinding.scale && accScaleBinding.scale instanceof quantitativeScale_1.QuantitativeScale)) {
            return extents;
        }
        var scale = accScaleBinding.scale;
        var width = this._barPixelThickness;
        // To account for inverted domains
        extents = extents.map(function (extent) { return d3.extent([
            scale.invert(_this._getPositionAttr(scale.scale(extent[0]), width)),
            scale.invert(_this._getPositionAttr(scale.scale(extent[0]), width) + width),
            scale.invert(_this._getPositionAttr(scale.scale(extent[1]), width)),
            scale.invert(_this._getPositionAttr(scale.scale(extent[1]), width) + width),
        ]); });
        return extents;
    };
    /**
     * Return the <rect>'s x or y attr value given the position and thickness of
     * that bar. This method is responsible for account for barAlignment, in particular.
     */
    Bar.prototype._getPositionAttr = function (position, thickness) {
        // account for flipped vertical axis
        if (!this._isVertical) {
            position -= thickness;
            thickness *= -1;
        }
        switch (this._barAlignment) {
            case "start":
                return position;
            case "end":
                return position - thickness;
            case "middle":
            default:
                return position - thickness / 2;
        }
    };
    Bar.prototype._drawLabels = function () {
        var _this = this;
        var dataToDraw = this._getDataToDraw();
        var attrToProjector = this._getAttrToProjector();
        var anyLabelTooWide = this.datasets().some(function (dataset) {
            return dataToDraw.get(dataset).some(function (datum, index) {
                return _this._drawLabel(datum, index, dataset, attrToProjector);
            });
        });
        if (this._hideBarsIfAnyAreTooWide && anyLabelTooWide) {
            this.datasets().forEach(function (dataset) { return _this._labelConfig.get(dataset).labelArea.selectAll("g").remove(); });
        }
    };
    Bar.prototype._drawLabel = function (datum, index, dataset, attrToProjector) {
        var _a = this._labelConfig.get(dataset), labelArea = _a.labelArea, measurer = _a.measurer, writer = _a.writer;
        var lengthAccessor = this.length().accessor;
        var length = lengthAccessor(datum, index, dataset);
        var lengthScale = this.length().scale;
        var scaledLength = lengthScale != null ? lengthScale.scale(length) : length;
        var scaledBaseline = lengthScale != null ? lengthScale.scale(this.baselineValue()) : this.baselineValue();
        var barCoordinates = { x: attrToProjector["x"](datum, index, dataset), y: attrToProjector["y"](datum, index, dataset) };
        var barDimensions = { width: attrToProjector["width"](datum, index, dataset), height: attrToProjector["height"](datum, index, dataset) };
        var text = this._labelFormatter(length, datum, index, dataset);
        var measurement = measurer.measure(text);
        var showLabelOnBar = this._getShowLabelOnBar(barCoordinates, barDimensions, measurement);
        // show label on right when value === baseline for horizontal plots
        var aboveOrLeftOfBaseline = this._isVertical ? scaledLength <= scaledBaseline : scaledLength < scaledBaseline;
        var _b = this._calculateLabelProperties(barCoordinates, barDimensions, measurement, showLabelOnBar, aboveOrLeftOfBaseline), containerDimensions = _b.containerDimensions, labelContainerOrigin = _b.labelContainerOrigin, labelOrigin = _b.labelOrigin, alignment = _b.alignment;
        var color = attrToProjector["fill"](datum, index, dataset);
        var labelContainer = this._createLabelContainer(labelArea, labelContainerOrigin, labelOrigin, measurement, showLabelOnBar, color);
        var writeOptions = { xAlign: alignment.x, yAlign: alignment.y };
        writer.write(text, containerDimensions.width, containerDimensions.height, writeOptions, labelContainer.node());
        var tooWide = this._isVertical
            ? barDimensions.width < (measurement.width + Bar._LABEL_PADDING * 2)
            : barDimensions.height < (measurement.height + Bar._LABEL_PADDING * 2);
        return tooWide;
    };
    Bar.prototype._getShowLabelOnBar = function (barCoordinates, barDimensions, measurement) {
        if (this._labelsPosition === exports.LabelsPosition.outside) {
            return false;
        }
        var barCoordinate = this._isVertical ? barCoordinates.y : barCoordinates.x;
        var barDimension = this._isVertical ? barDimensions.height : barDimensions.width;
        var plotDimension = this._isVertical ? this.height() : this.width();
        var measurementDimension = this._isVertical ? measurement.height : measurement.width;
        var effectiveBarDimension = barDimension;
        if (barCoordinate + barDimension > plotDimension) {
            effectiveBarDimension = plotDimension - barCoordinate;
        }
        else if (barCoordinate < 0) {
            effectiveBarDimension = barCoordinate + barDimension;
        }
        return (measurementDimension + 2 * Bar._LABEL_PADDING <= effectiveBarDimension);
    };
    Bar.prototype._calculateLabelProperties = function (barCoordinates, barDimensions, measurement, showLabelOnBar, aboveOrLeftOfBaseline) {
        var _this = this;
        var barCoordinate = this._isVertical ? barCoordinates.y : barCoordinates.x;
        var barDimension = this._isVertical ? barDimensions.height : barDimensions.width;
        var measurementDimension = this._isVertical ? measurement.height : measurement.width;
        var alignmentDimension = "center";
        var containerDimension = barDimension;
        var labelContainerOriginCoordinate = barCoordinate;
        var labelOriginCoordinate = barCoordinate;
        var updateCoordinates = function (position) {
            switch (position) {
                case "topLeft":
                    alignmentDimension = _this._isVertical ? "top" : "left";
                    labelContainerOriginCoordinate += Bar._LABEL_PADDING;
                    labelOriginCoordinate += Bar._LABEL_PADDING;
                    return;
                case "center":
                    labelOriginCoordinate += (barDimension + measurementDimension) / 2;
                    return;
                case "bottomRight":
                    alignmentDimension = _this._isVertical ? "bottom" : "right";
                    labelContainerOriginCoordinate -= Bar._LABEL_PADDING;
                    labelOriginCoordinate += containerDimension - Bar._LABEL_PADDING - measurementDimension;
                    return;
            }
        };
        if (showLabelOnBar) {
            switch (this._labelsPosition) {
                case exports.LabelsPosition.start:
                    aboveOrLeftOfBaseline ? updateCoordinates("bottomRight") : updateCoordinates("topLeft");
                    break;
                case exports.LabelsPosition.middle:
                    updateCoordinates("center");
                    break;
                case exports.LabelsPosition.end:
                    aboveOrLeftOfBaseline ? updateCoordinates("topLeft") : updateCoordinates("bottomRight");
                    break;
            }
        }
        else {
            if (aboveOrLeftOfBaseline) {
                alignmentDimension = this._isVertical ? "top" : "left";
                containerDimension = barDimension + Bar._LABEL_PADDING + measurementDimension;
                labelContainerOriginCoordinate -= Bar._LABEL_PADDING + measurementDimension;
                labelOriginCoordinate -= Bar._LABEL_PADDING + measurementDimension;
            }
            else {
                alignmentDimension = this._isVertical ? "bottom" : "right";
                containerDimension = barDimension + Bar._LABEL_PADDING + measurementDimension;
                labelOriginCoordinate += barDimension + Bar._LABEL_PADDING;
            }
        }
        return {
            containerDimensions: {
                width: this._isVertical ? barDimensions.width : containerDimension,
                height: this._isVertical ? containerDimension : barDimensions.height,
            },
            labelContainerOrigin: {
                x: this._isVertical ? barCoordinates.x : labelContainerOriginCoordinate,
                y: this._isVertical ? labelContainerOriginCoordinate : barCoordinates.y,
            },
            labelOrigin: {
                x: this._isVertical ? (barCoordinates.x + barDimensions.width / 2 - measurement.width / 2) : labelOriginCoordinate,
                y: this._isVertical ? labelOriginCoordinate : (barCoordinates.y + barDimensions.height / 2 - measurement.height / 2),
            },
            alignment: {
                x: this._isVertical ? "center" : alignmentDimension,
                y: this._isVertical ? alignmentDimension : "center",
            },
        };
    };
    Bar.prototype._createLabelContainer = function (labelArea, labelContainerOrigin, labelOrigin, measurement, showLabelOnBar, color) {
        var labelContainer = labelArea.append("g").attr("transform", "translate(" + labelContainerOrigin.x + ", " + labelContainerOrigin.y + ")");
        if (showLabelOnBar) {
            labelContainer.classed("on-bar-label", true);
            var dark = Utils.Color.contrast("white", color) * 1.6 < Utils.Color.contrast("black", color);
            labelContainer.classed(dark ? "dark-label" : "light-label", true);
        }
        else {
            labelContainer.classed("off-bar-label", true);
        }
        var hideLabel = labelOrigin.x < 0 ||
            labelOrigin.y < 0 ||
            labelOrigin.x + measurement.width > this.width() ||
            labelOrigin.y + measurement.height > this.height();
        labelContainer.style("visibility", hideLabel ? "hidden" : "inherit");
        return labelContainer;
    };
    Bar.prototype._generateDrawSteps = function () {
        var drawSteps = [];
        if (this._animateOnNextRender()) {
            var resetAttrToProjector = this._getAttrToProjector();
            var lengthScale = this.length().scale;
            var scaledBaseline_1 = lengthScale.scale(this.baselineValue());
            var lengthAttr = this._isVertical ? "y" : "x";
            var thicknessAttr = this._isVertical ? "height" : "width";
            resetAttrToProjector[lengthAttr] = function () { return scaledBaseline_1; };
            resetAttrToProjector[thicknessAttr] = function () { return 0; };
            drawSteps.push({ attrToProjector: resetAttrToProjector, animator: this._getAnimator(Plots.Animator.RESET) });
        }
        drawSteps.push({
            attrToProjector: this._getAttrToProjector(),
            animator: this._getAnimator(Plots.Animator.MAIN),
        });
        return drawSteps;
    };
    Bar.prototype._generateAttrToProjector = function () {
        var _this = this;
        var attrToProjector = _super.prototype._generateAttrToProjector.call(this);
        var lengthScale = this.length().scale;
        var scaledBaseline = lengthScale.scale(this.baselineValue());
        var lengthAttr = this._isVertical ? "y" : "x";
        var positionAttr = this._isVertical ? "x" : "y";
        var positionF = plot_1.Plot._scaledAccessor(this.position());
        var originalLengthFn = plot_1.Plot._scaledAccessor(this.length());
        var lengthFn = function (d, i, dataset) {
            return Math.abs(scaledBaseline - originalLengthFn(d, i, dataset));
        };
        var thicknessF = attrToProjector[Bar._BAR_THICKNESS_KEY];
        var gapF = attrToProjector["gap"];
        var thicknessMinusGap = gapF == null ? thicknessF : function (d, i, dataset) {
            return thicknessF(d, i, dataset) - gapF(d, i, dataset);
        };
        // re-interpret "width" attr from representing "thickness" to actually meaning
        // width (that is, x-direction specific) again
        attrToProjector["width"] = this._isVertical ? thicknessMinusGap : lengthFn;
        attrToProjector["height"] = this._isVertical ? lengthFn : thicknessMinusGap;
        attrToProjector[lengthAttr] = function (d, i, dataset) {
            var originalLength = originalLengthFn(d, i, dataset);
            // If it is past the baseline, it should start at the baseline then width/height
            // carries it over. If it's not past the baseline, leave it at original position and
            // then width/height carries it to baseline
            return (originalLength > scaledBaseline) ? scaledBaseline : originalLength;
        };
        attrToProjector[positionAttr] = function (d, i, dataset) {
            return _this._getPositionAttr(positionF(d, i, dataset), thicknessF(d, i, dataset));
        };
        return attrToProjector;
    };
    Bar.prototype._updateThicknessAttr = function () {
        var _this = this;
        var startProj = this.position();
        var endProj = this.barEnd();
        if (startProj != null && endProj != null) {
            this._fixedBarPixelThickness = false;
            this.attr(Bar._BAR_THICKNESS_KEY, function (d, i, data) {
                var v1 = startProj.accessor(d, i, data);
                var v2 = endProj.accessor(d, i, data);
                v1 = startProj.scale ? startProj.scale.scale(v1) : v1;
                v2 = endProj.scale ? endProj.scale.scale(v2) : v2;
                return Math.abs(v2 - v1);
            });
        }
        else {
            this._fixedBarPixelThickness = true;
            this._updateBarPixelWidth();
            this.attr(Bar._BAR_THICKNESS_KEY, function () { return _this._barPixelThickness; });
        }
    };
    Bar.prototype._updateBarPixelWidth = function () {
        if (this._fixedBarPixelThickness) {
            if (this._projectorsReady()) {
                this._barPixelThickness = computeBarPixelThickness(this.position(), this.datasets(), this._isVertical ? this.width() : this.height());
            }
            else {
                this._barPixelThickness = 0;
            }
        }
    };
    Bar.prototype.entities = function (datasets) {
        if (datasets === void 0) { datasets = this.datasets(); }
        if (!this._projectorsReady()) {
            return [];
        }
        var entities = _super.prototype.entities.call(this, datasets);
        return entities;
    };
    Bar.prototype._entityBounds = function (entity) {
        var datum = entity.datum, index = entity.index, dataset = entity.dataset;
        return this._pixelBounds(datum, index, dataset);
    };
    /**
     * The rectangular bounds of a bar. Note that the x/y coordinates are not the
     * same as the "pixel point" because they are always at the top/left of the
     * bar.
     */
    Bar.prototype._pixelBounds = function (datum, index, dataset) {
        var attrToProjector = this._getAttrToProjector();
        return {
            x: attrToProjector["x"](datum, index, dataset),
            y: attrToProjector["y"](datum, index, dataset),
            width: attrToProjector["width"](datum, index, dataset),
            height: attrToProjector["height"](datum, index, dataset),
        };
    };
    /**
     * The "pixel point" of a bar is the farthest point from the baseline.
     *
     * For example, in a vertical bar chart with positive bar values, the pixel
     * point will be at the top of the bar. For negative bar values, the pixel
     * point will be at the bottom of the bar.
     */
    Bar.prototype._pixelPoint = function (datum, index, dataset) {
        var rect = this._pixelBounds(datum, index, dataset);
        var originalPosition = (this._isVertical ? plot_1.Plot._scaledAccessor(this.y()) : plot_1.Plot._scaledAccessor(this.x()))(datum, index, dataset);
        var scaledBaseline = (this._isVertical ? this.y().scale : this.x().scale).scale(this.baselineValue());
        return this._pixelPointBar(originalPosition, scaledBaseline, rect);
    };
    Bar.prototype._pixelPointBar = function (originalPosition, scaledBaseline, rect) {
        var x, y;
        if (this._isVertical) {
            x = rect.x + rect.width / 2;
            y = originalPosition <= scaledBaseline ? rect.y : rect.y + rect.height;
        }
        else {
            x = originalPosition >= scaledBaseline ? rect.x + rect.width : rect.x;
            y = rect.y + rect.height / 2;
        }
        return { x: x, y: y };
    };
    Bar.prototype._uninstallScaleForKey = function (scale, key) {
        scale.offUpdate(this._updateBarPixelThicknessCallback);
        _super.prototype._uninstallScaleForKey.call(this, scale, key);
    };
    Bar.prototype._getDataToDraw = function () {
        var dataToDraw = new Utils.Map();
        var attrToProjector = this._getAttrToProjector();
        this.datasets().forEach(function (dataset) {
            var data = dataset.data().filter(function (d, i) { return Utils.Math.isValidNumber(attrToProjector["x"](d, i, dataset)) &&
                Utils.Math.isValidNumber(attrToProjector["y"](d, i, dataset)) &&
                Utils.Math.isValidNumber(attrToProjector["width"](d, i, dataset)) &&
                Utils.Math.isValidNumber(attrToProjector["height"](d, i, dataset)); });
            dataToDraw.set(dataset, data);
        });
        return dataToDraw;
    };
    return Bar;
}(xyPlot_1.XYPlot));
Bar._BAR_THICKNESS_RATIO = 0.95;
Bar._SINGLE_BAR_DIMENSION_RATIO = 0.4;
Bar._BAR_AREA_CLASS = "bar-area";
Bar._BAR_END_KEY = "barEnd";
// we special case the "width" property to represent the bar thickness
// (aka the distance between adjacent bar positions); in _generateAttrToProjector
// we re-assign "width" to specifically refer to <rect>'s width attribute
Bar._BAR_THICKNESS_KEY = "width";
Bar._LABEL_AREA_CLASS = "bar-label-text-area";
Bar._LABEL_PADDING = 10;
exports.Bar = Bar;
/**
 * Computes the barPixelThickness of all the bars in the plot.
 *
 * If the position scale of the plot is a CategoryScale and in bands mode, then the rangeBands function will be used.
 * If the position scale of the plot is a QuantitativeScale, then the bar thickness is equal to the smallest distance between
 * two adjacent data points, padded for visualisation.
 *
 * This is ignored when explicitly setting the barEnd.
 */
function computeBarPixelThickness(positionBinding, datasets, barWidthDimension) {
    var barPixelThickness;
    var positionScale = positionBinding.scale;
    if (positionScale instanceof Scales.Category) {
        barPixelThickness = positionScale.rangeBand();
    }
    else {
        var positionAccessor_1 = positionBinding.accessor;
        var numberBarAccessorData = d3.set(Utils.Array.flatten(datasets.map(function (dataset) {
            return dataset.data().map(function (d, i) { return positionAccessor_1(d, i, dataset); })
                .filter(function (d) { return d != null; })
                .map(function (d) { return d.valueOf(); });
        }))).values().map(function (value) { return +value; });
        numberBarAccessorData.sort(function (a, b) { return a - b; });
        var scaledData = numberBarAccessorData.map(function (datum) { return positionScale.scale(datum); });
        var barAccessorDataPairs = d3.pairs(scaledData);
        barPixelThickness = Utils.Math.min(barAccessorDataPairs, function (pair, i) {
            return Math.abs(pair[1] - pair[0]);
        }, barWidthDimension * Bar._SINGLE_BAR_DIMENSION_RATIO);
        barPixelThickness *= Bar._BAR_THICKNESS_RATIO;
    }
    return barPixelThickness;
}
