/**
 * Copyright 2017-present Palantir Technologies
 * @license MIT
 */
"use strict";
/**
 * Stores the deferred transformation state for a single scale.
 */
var DomainTransform = (function () {
    function DomainTransform() {
        var _this = this;
        this.scale = 0;
        this.translate = 0;
        this.cachedDomain = [null, null];
        this.lastSeenDomain = [null, null];
        this.updateDomain = function (scale) {
            _this.lastSeenDomain = scale.domain();
            var cachedLength = scale.scale(_this.cachedDomain[1]) - scale.scale(_this.cachedDomain[0]);
            var lastSeenLength = scale.scale(_this.lastSeenDomain[1]) - scale.scale(_this.lastSeenDomain[0]);
            _this.scale = (cachedLength / lastSeenLength) || 1;
            _this.translate = scale.scale(_this.cachedDomain[0]) - scale.scale(_this.lastSeenDomain[0]) * _this.scale || 0;
        };
    }
    DomainTransform.prototype.reset = function () {
        this.scale = 1;
        this.translate = 0;
        this.cachedDomain = this.lastSeenDomain;
    };
    DomainTransform.prototype.setDomain = function (domain) {
        this.cachedDomain = domain;
    };
    return DomainTransform;
}());
/**
 * Manages deferred rendering callbacks.
 *
 * Call `setDomains` when deferred rendering is initially enabled to fix the
 * current domain values.
 *
 * Call `updateDomains` when scale domains change, which uses the domain to
 * compute CSS-tyle transform parameters passed to `applyTransformCallback`,
 * mimicking the result of a full re-render. After a deferred timeout, invoke
 * `applyTransformCallback` again with an identity transform and finally invoke
 * `renderCallback`, which should actually redraw the plot.
 */
var DeferredRenderer = (function () {
    function DeferredRenderer(renderCallback, applyTransformCallback) {
        var _this = this;
        this.renderCallback = renderCallback;
        this.applyTransformCallback = applyTransformCallback;
        this.domainTransformX = new DomainTransform();
        this.domainTransformY = new DomainTransform();
        this.renderDeferred = function () {
            _this.applyTransform();
            clearTimeout(_this.timeoutToken);
            _this.timeoutToken = setTimeout(function () {
                _this.resetTransforms();
                _this.applyTransform();
                _this.renderCallback();
            }, DeferredRenderer.DEFERRED_RENDERING_DELAY);
        };
    }
    DeferredRenderer.prototype.setDomains = function (scaleX, scaleY) {
        if (scaleX) {
            this.domainTransformX.setDomain(scaleX.domain());
        }
        if (scaleY) {
            this.domainTransformY.setDomain(scaleY.domain());
        }
        this.renderDeferred();
    };
    DeferredRenderer.prototype.updateDomains = function (scaleX, scaleY) {
        if (scaleX) {
            this.domainTransformX.updateDomain(scaleX);
        }
        if (scaleY) {
            this.domainTransformY.updateDomain(scaleY);
        }
        this.renderDeferred();
    };
    DeferredRenderer.prototype.resetTransforms = function () {
        this.domainTransformX.reset();
        this.domainTransformY.reset();
    };
    DeferredRenderer.prototype.applyTransform = function () {
        this.applyTransformCallback(this.domainTransformX.translate, this.domainTransformY.translate, this.domainTransformX.scale, this.domainTransformY.scale);
    };
    return DeferredRenderer;
}());
DeferredRenderer.DEFERRED_RENDERING_DELAY = 200;
exports.DeferredRenderer = DeferredRenderer;
