/**
 * Copyright 2014-present Palantir Technologies
 * @license MIT
 */
"use strict";
var __extends = (this && this.__extends) || function (d, b) {
    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
    function __() { this.constructor = d; }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
};
var Typesettable = require("typesettable");
var formatters_1 = require("../core/formatters");
var Utils = require("../utils");
var barPlot_1 = require("./barPlot");
var plot_1 = require("./plot");
var StackedBar = (function (_super) {
    __extends(StackedBar, _super);
    /**
     * A StackedBar Plot stacks bars across Datasets based on the primary value of the bars.
     *   On a vertical StackedBar Plot, the bars with the same X value are stacked.
     *   On a horizontal StackedBar Plot, the bars with the same Y value are stacked.
     *
     * @constructor
     * @param {Scale} xScale
     * @param {Scale} yScale
     * @param {string} [orientation="vertical"] One of "vertical"/"horizontal".
     */
    function StackedBar(orientation) {
        if (orientation === void 0) { orientation = "vertical"; }
        var _this = _super.call(this, orientation) || this;
        _this._extremaFormatter = formatters_1.identity();
        _this.addClass("stacked-bar-plot");
        _this._stackingOrder = "bottomup";
        _this._stackingResult = new Utils.Map();
        _this._stackedExtent = [];
        return _this;
    }
    StackedBar.prototype.x = function (x, xScale) {
        if (x == null) {
            return _super.prototype.x.call(this);
        }
        if (xScale == null) {
            _super.prototype.x.call(this, x);
        }
        else {
            _super.prototype.x.call(this, x, xScale);
        }
        this._updateStackExtentsAndOffsets();
        return this;
    };
    StackedBar.prototype.y = function (y, yScale) {
        if (y == null) {
            return _super.prototype.y.call(this);
        }
        if (yScale == null) {
            _super.prototype.y.call(this, y);
        }
        else {
            _super.prototype.y.call(this, y, yScale);
        }
        this._updateStackExtentsAndOffsets();
        return this;
    };
    StackedBar.prototype.stackingOrder = function (stackingOrder) {
        if (stackingOrder == null) {
            return this._stackingOrder;
        }
        this._stackingOrder = stackingOrder;
        this._onDatasetUpdate();
        return this;
    };
    StackedBar.prototype.extremaFormatter = function (formatter) {
        if (arguments.length === 0) {
            return this._extremaFormatter;
        }
        else {
            this._extremaFormatter = formatter;
            this.render();
            return this;
        }
    };
    StackedBar.prototype._setup = function () {
        _super.prototype._setup.call(this);
        this._labelArea = this._renderArea.append("g").classed(barPlot_1.Bar._LABEL_AREA_CLASS, true);
        var context = new Typesettable.SvgContext(this._labelArea.node());
        this._measurer = new Typesettable.CacheMeasurer(context);
        this._writer = new Typesettable.Writer(this._measurer, context);
    };
    StackedBar.prototype._drawLabels = function () {
        var _this = this;
        _super.prototype._drawLabels.call(this);
        // remove all current labels before redrawing
        this._labelArea.selectAll("g").remove();
        var baselineValue = +this.baselineValue();
        var positionScale = this.position().scale;
        var lengthScale = this.length().scale;
        var _a = Utils.Stacking.stackedExtents(this._stackingResult), maximumExtents = _a.maximumExtents, minimumExtents = _a.minimumExtents;
        var anyTooWide = [];
        /**
         * Try drawing the text at the center of the bounds. This method does not draw
         * the text if the text would overflow outside of the plot.
         *
         * @param text
         * @param bounds
         * @returns {boolean}
         */
        var maybeDrawLabel = function (text, bounds, barThickness) {
            var _a = bounds.topLeft, x = _a.x, y = _a.y;
            var width = bounds.bottomRight.x - bounds.topLeft.x;
            var height = bounds.bottomRight.y - bounds.topLeft.y;
            var textTooLong = _this._isVertical
                ? (width > barThickness - (2 * StackedBar._LABEL_PADDING))
                : (height > barThickness - (2 * StackedBar._LABEL_PADDING));
            if (!textTooLong) {
                var labelContainer = _this._labelArea.append("g").attr("transform", "translate(" + x + ", " + y + ")");
                labelContainer.classed("stacked-bar-label", true);
                var writeOptions = {
                    xAlign: "center",
                    yAlign: "center",
                };
                _this._writer.write(text, width, height, writeOptions, labelContainer.node());
            }
            return textTooLong;
        };
        var drawLabelsForExtents = function (extents, computeLabelTopLeft) {
            extents.forEach(function (maximum) {
                if (maximum.extent !== baselineValue) {
                    // only draw sums for values not at the baseline
                    var text = _this.extremaFormatter()(maximum.extent);
                    var textDimensions = _this._measurer.measure(text);
                    var stackedDatum = maximum.stackedDatum;
                    var originalDatum = stackedDatum.originalDatum, originalIndex = stackedDatum.originalIndex, originalDataset = stackedDatum.originalDataset;
                    var barThickness = plot_1.Plot._scaledAccessor(_this.attr(barPlot_1.Bar._BAR_THICKNESS_KEY))(originalDatum, originalIndex, originalDataset);
                    /*
                     * The stackEdge is aligned at the edge of the stack in the length dimension,
                     * and in the center of the stack in the thickness dimension.
                     */
                    var stackEdgeLength = lengthScale.scale(maximum.extent);
                    var stackCenterPosition = _this._getPositionAttr(positionScale.scale(maximum.axisValue), barThickness) + barThickness / 2;
                    var stackEdge = _this._isVertical
                        ? {
                            x: stackCenterPosition,
                            y: stackEdgeLength,
                        }
                        : {
                            x: stackEdgeLength,
                            y: stackCenterPosition,
                        };
                    var topLeft = computeLabelTopLeft(stackEdge, textDimensions, barThickness);
                    var isTooWide = maybeDrawLabel(text, {
                        topLeft: topLeft,
                        bottomRight: {
                            x: topLeft.x + textDimensions.width,
                            y: topLeft.y + textDimensions.height,
                        },
                    }, barThickness);
                    anyTooWide.push(isTooWide);
                }
            });
        };
        drawLabelsForExtents(maximumExtents, function (stackEdge, measurement, thickness) {
            var primaryTextMeasurement = _this._isVertical ? measurement.width : measurement.height;
            var secondaryTextMeasurement = _this._isVertical ? measurement.height : measurement.width;
            return {
                x: _this._isVertical
                    ? stackEdge.x - primaryTextMeasurement / 2
                    : stackEdge.x + StackedBar._STACKED_BAR_LABEL_PADDING,
                y: _this._isVertical
                    ? stackEdge.y - secondaryTextMeasurement - StackedBar._STACKED_BAR_LABEL_PADDING
                    : stackEdge.y - primaryTextMeasurement / 2,
            };
        });
        drawLabelsForExtents(minimumExtents, function (stackEdge, measurement, thickness) {
            var primaryTextMeasurement = _this._isVertical ? measurement.width : measurement.height;
            var secondaryTextMeasurement = _this._isVertical ? measurement.height : measurement.width;
            return {
                x: _this._isVertical
                    ? stackEdge.x - primaryTextMeasurement / 2
                    : stackEdge.x - secondaryTextMeasurement - StackedBar._STACKED_BAR_LABEL_PADDING,
                y: _this._isVertical
                    ? stackEdge.y + StackedBar._STACKED_BAR_LABEL_PADDING
                    : stackEdge.y - primaryTextMeasurement / 2,
            };
        });
        if (anyTooWide.some(function (d) { return d; })) {
            this._labelArea.selectAll("g").remove();
        }
    };
    StackedBar.prototype._generateAttrToProjector = function () {
        var _this = this;
        var attrToProjector = _super.prototype._generateAttrToProjector.call(this);
        var valueAttr = this._isVertical ? "y" : "x";
        var lengthScale = this.length().scale;
        var lengthAccessor = this.length().accessor;
        var positionAccessor = this.position().accessor;
        var normalizedPositionAccessor = function (datum, index, dataset) {
            return Utils.Stacking.normalizeKey(positionAccessor(datum, index, dataset));
        };
        var getStart = function (d, i, dataset) {
            return lengthScale.scale(_this._stackingResult.get(dataset).get(normalizedPositionAccessor(d, i, dataset)).offset);
        };
        var getEnd = function (d, i, dataset) {
            return lengthScale.scale(+lengthAccessor(d, i, dataset) +
                _this._stackingResult.get(dataset).get(normalizedPositionAccessor(d, i, dataset)).offset);
        };
        var heightF = function (d, i, dataset) {
            return Math.abs(getEnd(d, i, dataset) - getStart(d, i, dataset));
        };
        attrToProjector[this._isVertical ? "height" : "width"] = heightF;
        var attrFunction = function (d, i, dataset) {
            return +lengthAccessor(d, i, dataset) < 0 ? getStart(d, i, dataset) : getEnd(d, i, dataset);
        };
        attrToProjector[valueAttr] = function (d, i, dataset) {
            return _this._isVertical ? attrFunction(d, i, dataset) : attrFunction(d, i, dataset) - heightF(d, i, dataset);
        };
        return attrToProjector;
    };
    StackedBar.prototype._onDatasetUpdate = function () {
        this._updateStackExtentsAndOffsets();
        _super.prototype._onDatasetUpdate.call(this);
        return this;
    };
    StackedBar.prototype._updateExtentsForProperty = function (property) {
        _super.prototype._updateExtentsForProperty.call(this, property);
        if ((property === "x" || property === "y") && this._projectorsReady()) {
            this._updateStackExtentsAndOffsets();
        }
    };
    StackedBar.prototype._extentsForProperty = function (attr) {
        var primaryAttr = this._isVertical ? "y" : "x";
        if (attr === primaryAttr) {
            return [this._stackedExtent];
        }
        else {
            return _super.prototype._extentsForProperty.call(this, attr);
        }
    };
    StackedBar.prototype._updateStackExtentsAndOffsets = function () {
        if (!this._projectorsReady()) {
            return;
        }
        var datasets = this.datasets();
        var positionAccessor = this.position().accessor;
        var lengthAccessor = this.length().accessor;
        var filter = this._filterForProperty(this._isVertical ? "y" : "x");
        this._stackingResult = Utils.Stacking.stack(datasets, positionAccessor, lengthAccessor, this._stackingOrder);
        this._stackedExtent = Utils.Stacking.stackedExtent(this._stackingResult, positionAccessor, filter);
    };
    StackedBar.prototype.invalidateCache = function () {
        _super.prototype.invalidateCache.call(this);
        this._measurer.reset();
    };
    return StackedBar;
}(barPlot_1.Bar));
StackedBar._STACKED_BAR_LABEL_PADDING = 5;
exports.StackedBar = StackedBar;
