/**
 * @module  point-cluster/quad-bucket
 *
 * Bucket based clustering
 */

'use strict'

const PointCluster = require('./cluster')
const search = require('binary-search-bounds')
const clamp = require('clamp')

module.exports = QuadCluster

function QuadCluster(coords, options) {
	if (!(this instanceof QuadCluster)) return new QuadCluster(coords, options)

	PointCluster.call(this, coords, options)

	let points = this.points

	// point indexes for levels [0: [a,b,c,d], 1: [a,b,c,d,e,f,...], ...]
	let levels = this.levels = []

	// starting indexes of subranges in sub levels
	let sublevels = this.sublevels = []

	// keep track of cx/cy
	let groups = this.groups = []

	let bounds = this.bounds
	this.diam = Math.max(bounds[2] - bounds[0], bounds[3] - bounds[1])

	sort(0, 0, 1, this.ids, 0, 1)

	// FIXME: it is possible to create one typed array heap and reuse that to avoid memory blow
	function sort (x, y, diam, ids, level, group) {
		if (!ids.length) return null

		// save first point as level representative
		let item = ids[0]
		let levelItems = levels[level] || (levels[level] = [])
		levelItems.push(item)

		let levelGroups = groups[level] || (groups[level] = [])
		levelGroups.push(group)

		let sublevel = sublevels[level] || (sublevels[level] = [])
		let offset = levelItems.length - 1
		if (ids.length <= 1) {
			sublevel.push(null, null, null, null)
			return offset
		}

		let d2 = diam * .5
		let cx = x + d2, cy = y + d2

		// distribute points by 4 buckets
		let lolo = [], lohi = [], hilo = [], hihi = []

		for (let i = 1; i < ids.length; i++) {
			let idx = ids[i],
				x = points[idx * 2],
				y = points[idx * 2 + 1]
			x < cx ? (y < cy ? lolo.push(idx) : lohi.push(idx)) : (y < cy ? hilo.push(idx) : hihi.push(idx))
		}

		level++
		group <<= 2
		sublevel.push(
			sort(x, y, d2, lolo, level, group),
			sort(x, cy, d2, lohi, level, group + 1),
			sort(cx, y, d2, hilo, level, group + 2),
			sort(cx, cy, d2, hihi, level, group + 3)
		)

		return offset
	}
}

QuadCluster.prototype = Object.create(PointCluster.prototype)

QuadCluster.prototype.closest = function (x, y) {

}

QuadCluster.prototype.range = function (l, t, r, b) {

}

QuadCluster.prototype.radius = function (x, y, r) {

}

// get group id closest to the x,y coordinate, corresponding to a level
QuadCluster.prototype.group = function (realx, realy, level) {
	let group = 1

	let [x, y] = this.normalize([realx, realy], this.bounds)

	x = clamp(x, 0, 1)
	y = clamp(y, 0, 1)

	let cx = .5, cy = .5
	let diam = .5

	for (let i = 0; i < level; i++) {
		group <<= 2

		group += x < cx ? (y < cy ? 0 : 1) : (y < cy ? 2 : 3)

		diam *= .5

		cx += x < cx ? -diam : diam
		cy += y < cy ? -diam : diam
	}

	return group
}

// get range offsets within levels to render lods appropriate for zoom level
QuadCluster.prototype.offsets = function (pxSize, lox, loy, hix, hiy) {
	let offsets = []
	let diam = this.diam

	for (let level = 0; level < this.levels.length; level++) {
		let levelGroups = this.groups[level]
		let levelPixelSize = diam * Math.pow(0.5, level)

		if (levelPixelSize && levelPixelSize < pxSize) {
			continue
		}

		let levelGroupStart = this.group(lox, loy, level)
		let levelGroupEnd = this.group(hix, hiy, level)

		// FIXME: utilize sublevels to speed up search range here
		let startOffset = search.ge(levelGroups, levelGroupStart)
		let endOffset = search.le(levelGroups, levelGroupEnd, startOffset, levelGroups.length - 1) + 1

		offsets[level] = [startOffset, endOffset]
	}

	return offsets
}

