"use strict";
/**
 * @license
 * Copyright (c) 2016 The Polymer Project Authors. All rights reserved.
 * This code may only be used under the BSD style license found at
 * http://polymer.github.io/LICENSE.txt
 * The complete set of authors may be found at
 * http://polymer.github.io/AUTHORS.txt
 * The complete set of contributors may be found at
 * http://polymer.github.io/CONTRIBUTORS.txt
 * Code distributed by Google as part of the polymer project is also
 * subject to an additional IP rights grant found at
 * http://polymer.github.io/PATENTS.txt
 */
Object.defineProperty(exports, "__esModule", { value: true });
const chai_1 = require("chai");
const multi_url_resolver_1 = require("../../url-loader/multi-url-resolver");
const url_resolver_1 = require("../../url-loader/url-resolver");
class MockResolver extends url_resolver_1.UrlResolver {
    constructor(_resolution) {
        super();
        this._resolution = _resolution;
        this.resetCounts();
    }
    resetCounts() {
        this.canResolveCount = 0;
        this.resolveCount = 0;
    }
    canResolve() {
        this.canResolveCount++;
        return this._resolution != null;
    }
    resolve() {
        if (this._resolution == null) {
            throw new Error('tried to resolve to a null resolution!');
        }
        this.resolveCount++;
        return this._resolution;
    }
}
const mockResolverArray = (resolutions) => {
    return resolutions.map((resolution) => {
        return new MockResolver(resolution);
    });
};
suite('MultiUrlResolver', function () {
    suite('canResolve', () => {
        test('canResolve is true if the first resolver is true', () => {
            const resolvers = mockResolverArray(['resolved.html', null, null]);
            const resolver = new multi_url_resolver_1.MultiUrlResolver(resolvers);
            chai_1.assert.isTrue(resolver.canResolve('test.html'));
            // Verify only the first resolver is called
            chai_1.assert.equal(resolvers[0].canResolveCount, 1);
            chai_1.assert.equal(resolvers[1].canResolveCount, 0);
            chai_1.assert.equal(resolvers[2].canResolveCount, 0);
        });
        test('canResolve is true if the last resolver is true', () => {
            const resolvers = mockResolverArray([null, null, 'resolved.html']);
            const resolver = new multi_url_resolver_1.MultiUrlResolver(resolvers);
            chai_1.assert.isTrue(resolver.canResolve('test.html'));
            // Verify all resolvers are called
            chai_1.assert.equal(resolvers[0].canResolveCount, 1);
            chai_1.assert.equal(resolvers[1].canResolveCount, 1);
            chai_1.assert.equal(resolvers[2].canResolveCount, 1);
        });
        test('canResolve is true if all resolvers are true', () => {
            const resolvers = mockResolverArray(['resolved.html', 'resolved2.html', 'resolved3.html']);
            const resolver = new multi_url_resolver_1.MultiUrlResolver(resolvers);
            chai_1.assert.isTrue(resolver.canResolve('test.html'));
            // Verify only the first resolver is called
            chai_1.assert.equal(resolvers[0].canResolveCount, 1);
            chai_1.assert.equal(resolvers[1].canResolveCount, 0);
            chai_1.assert.equal(resolvers[2].canResolveCount, 0);
        });
        test('canResolve is false if all resolvers are false', () => {
            const resolvers = mockResolverArray([null, null, null]);
            const resolver = new multi_url_resolver_1.MultiUrlResolver(resolvers);
            chai_1.assert.isFalse(resolver.canResolve('test.html'));
            // Verify only the first resolver is called
            chai_1.assert.equal(resolvers[0].canResolveCount, 1);
            chai_1.assert.equal(resolvers[1].canResolveCount, 1);
            chai_1.assert.equal(resolvers[2].canResolveCount, 1);
        });
    });
    suite('resolve', () => {
        test('only the first resolution is returned', () => {
            const resolvers = mockResolverArray(['resolved.html', 'resolved2.html', 'resolved3.html']);
            const resolver = new multi_url_resolver_1.MultiUrlResolver(resolvers);
            chai_1.assert.equal(resolver.resolve('test.html'), 'resolved.html');
            // Verify only the first resolver is called
            chai_1.assert.equal(resolvers[0].canResolveCount, 1);
            chai_1.assert.equal(resolvers[1].canResolveCount, 0);
            chai_1.assert.equal(resolvers[2].canResolveCount, 0);
        });
    });
});

//# sourceMappingURL=multi-url-resolver_test.js.map
