"use strict";
const constants_1 = require("./constants");
const tokenizer_1 = require("./tokenizer");
function parse(expr, astFactory) {
    return new Parser(expr, astFactory).parse();
}
exports.parse = parse;
class Parser {
    constructor(input, astFactory) {
        this._kind = null;
        this._token = null;
        this._value = null;
        this._tokenizer = new tokenizer_1.Tokenizer(input);
        this._ast = astFactory;
    }
    parse() {
        this._advance();
        return this._parseExpression();
    }
    _advance(kind, value) {
        if (!this._matches(kind, value)) {
            throw new Error(`Expected kind ${kind} (${value}), was ${this._token}`);
        }
        const t = this._tokenizer.nextToken();
        this._token = t;
        this._kind = t && t.kind;
        this._value = t && t.value;
    }
    _matches(kind, value) {
        return !(kind && (this._kind !== kind) || value && (this._value !== value));
    }
    _parseExpression() {
        if (!this._token)
            return this._ast.empty();
        const expr = this._parseUnary();
        return (!expr) ? null : this._parsePrecedence(expr, 0);
    }
    // _parsePrecedence and _parseBinary implement the precedence climbing
    // algorithm as described in:
    // http://en.wikipedia.org/wiki/Operator-precedence_parser#Precedence_climbing_method
    _parsePrecedence(left, precedence) {
        if (!left) {
            throw new Error('Expected left not to be null.');
        }
        while (this._token) {
            if (this._matches(tokenizer_1.Kind.GROUPER, '(')) {
                const args = this._parseArguments();
                left = this._ast.invoke(left, null, args);
            }
            else if (this._matches(tokenizer_1.Kind.GROUPER, '[')) {
                const indexExpr = this._parseIndex();
                left = this._ast.index(left, indexExpr);
            }
            else if (this._matches(tokenizer_1.Kind.DOT)) {
                this._advance();
                const right = this._parseUnary();
                left = this._makeInvokeOrGetter(left, right);
            }
            else if (this._matches(tokenizer_1.Kind.KEYWORD)) {
                break;
            }
            else if (this._matches(tokenizer_1.Kind.OPERATOR) &&
                this._token.precedence >= precedence) {
                left = this._value === '?' ? this._parseTernary(left) :
                    this._parseBinary(left, this._token);
            }
            else {
                break;
            }
        }
        return left;
    }
    _makeInvokeOrGetter(left, right) {
        if (right.type === 'ID') {
            return this._ast.getter(left, right.value);
        }
        else if (right.type === 'Invoke' && right.receiver.type === 'ID') {
            const method = right.receiver;
            return this._ast.invoke(left, method.value, right.arguments);
        }
        else {
            throw new Error(`expected identifier: ${right}`);
        }
    }
    _parseBinary(left, op) {
        if (constants_1.BINARY_OPERATORS.indexOf(op.value) === -1) {
            throw new Error(`unknown operator: ${op.value}`);
        }
        this._advance();
        let right = this._parseUnary();
        while ((this._kind === tokenizer_1.Kind.OPERATOR || this._kind === tokenizer_1.Kind.DOT ||
            this._kind === tokenizer_1.Kind.GROUPER) &&
            this._token.precedence > op.precedence) {
            right = this._parsePrecedence(right, this._token.precedence);
        }
        return this._ast.binary(left, op.value, right);
    }
    _parseUnary() {
        if (this._matches(tokenizer_1.Kind.OPERATOR)) {
            const value = this._value;
            this._advance();
            // handle unary + and - on numbers as part of the literal, not as a
            // unary operator
            if (value === '+' || value === '-') {
                if (this._matches(tokenizer_1.Kind.INTEGER)) {
                    return this._parseInteger(value);
                }
                else if (this._matches(tokenizer_1.Kind.DECIMAL)) {
                    return this._parseDecimal(value);
                }
            }
            if (constants_1.UNARY_OPERATORS.indexOf(value) === -1)
                throw new Error(`unexpected token: ${value}`);
            const expr = this._parsePrecedence(this._parsePrimary(), constants_1.POSTFIX_PRECEDENCE);
            return this._ast.unary(value, expr);
        }
        return this._parsePrimary();
    }
    _parseTernary(condition) {
        this._advance(tokenizer_1.Kind.OPERATOR, '?');
        const trueExpr = this._parseExpression();
        this._advance(tokenizer_1.Kind.COLON);
        const falseExpr = this._parseExpression();
        return this._ast.ternary(condition, trueExpr, falseExpr);
    }
    _parsePrimary() {
        switch (this._kind) {
            case tokenizer_1.Kind.KEYWORD:
                const keyword = this._value;
                if (keyword === 'this') {
                    this._advance();
                    // TODO(justin): return keyword node
                    return this._ast.id(keyword);
                }
                else if (constants_1.KEYWORDS.indexOf(keyword) !== -1) {
                    throw new Error(`unexpected keyword: ${keyword}`);
                }
                throw new Error(`unrecognized keyword: ${keyword}`);
            case tokenizer_1.Kind.IDENTIFIER:
                return this._parseInvokeOrIdentifier();
            case tokenizer_1.Kind.STRING:
                return this._parseString();
            case tokenizer_1.Kind.INTEGER:
                return this._parseInteger();
            case tokenizer_1.Kind.DECIMAL:
                return this._parseDecimal();
            case tokenizer_1.Kind.GROUPER:
                if (this._value === '(') {
                    return this._parseParen();
                }
                else if (this._value === '{') {
                    return this._parseMap();
                }
                else if (this._value === '[') {
                    return this._parseList();
                }
                return null;
            case tokenizer_1.Kind.COLON:
                throw new Error('unexpected token ":"');
            default:
                return null;
        }
    }
    _parseList() {
        const items = [];
        do {
            this._advance();
            if (this._matches(tokenizer_1.Kind.GROUPER, ']'))
                break;
            items.push(this._parseExpression());
        } while (this._matches(tokenizer_1.Kind.COMMA));
        this._advance(tokenizer_1.Kind.GROUPER, ']');
        return this._ast.list(items);
    }
    _parseMap() {
        const entries = {};
        do {
            this._advance();
            if (this._matches(tokenizer_1.Kind.GROUPER, '}'))
                break;
            const key = this._value;
            this._advance(tokenizer_1.Kind.STRING);
            this._advance(tokenizer_1.Kind.COLON);
            entries[key] = this._parseExpression();
        } while (this._matches(tokenizer_1.Kind.COMMA));
        this._advance(tokenizer_1.Kind.GROUPER, '}');
        return this._ast.map(entries);
    }
    _parseInvokeOrIdentifier() {
        const value = this._value;
        if (value === 'true') {
            this._advance();
            return this._ast.literal(true);
        }
        if (value === 'false') {
            this._advance();
            return this._ast.literal(false);
        }
        if (value === 'null') {
            this._advance();
            return this._ast.literal(null);
        }
        const identifier = this._parseIdentifier();
        const args = this._parseArguments();
        return (!args) ? identifier : this._ast.invoke(identifier, null, args);
    }
    _parseIdentifier() {
        if (!this._matches(tokenizer_1.Kind.IDENTIFIER)) {
            throw new Error(`expected identifier: ${this._value}`);
        }
        const value = this._value;
        this._advance();
        return this._ast.id(value);
    }
    _parseArguments() {
        if (this._matches(tokenizer_1.Kind.GROUPER, '(')) {
            const args = [];
            do {
                this._advance();
                if (this._matches(tokenizer_1.Kind.GROUPER, ')')) {
                    break;
                }
                const expr = this._parseExpression();
                args.push(expr);
            } while (this._matches(tokenizer_1.Kind.COMMA));
            this._advance(tokenizer_1.Kind.GROUPER, ')');
            return args;
        }
        return null;
    }
    _parseIndex() {
        if (this._matches(tokenizer_1.Kind.GROUPER, '[')) {
            this._advance();
            const expr = this._parseExpression();
            this._advance(tokenizer_1.Kind.GROUPER, ']');
            return expr;
        }
        return null;
    }
    _parseParen() {
        this._advance();
        const expr = this._parseExpression();
        this._advance(tokenizer_1.Kind.GROUPER, ')');
        return this._ast.paren(expr);
    }
    _parseString() {
        const value = this._ast.literal(this._value);
        this._advance();
        return value;
    }
    _parseInteger(prefix) {
        prefix = prefix || '';
        const value = this._ast.literal(parseInt(`${prefix}${this._value}`, 10));
        this._advance();
        return value;
    }
    _parseDecimal(prefix) {
        prefix = prefix || '';
        const value = this._ast.literal(parseFloat(`${prefix}${this._value}`));
        this._advance();
        return value;
    }
}
exports.Parser = Parser;
//# sourceMappingURL=parser.js.map