/**
@license
Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
This code may only be used under the BSD style license found at
http://polymer.github.io/LICENSE.txt The complete set of authors may be found at
http://polymer.github.io/AUTHORS.txt The complete set of contributors may be
found at http://polymer.github.io/CONTRIBUTORS.txt Code distributed by Google as
part of the polymer project is also subject to an additional IP rights grant
found at http://polymer.github.io/PATENTS.txt
*/
import {Polymer} from '@polymer/polymer/polymer-legacy.js';
/**
`app-media-devices` is an element that enumerates the connected input devices
and optionally filters them, allowing the user to cycle through categories of
devices one at a time.

@group App Elements
@demo demo/index.html
*/
Polymer({
  is: 'app-media-devices',

  properties: {
    /**
     * A string used to filter the devices based on their kind property.
     * The string is converted to a RegExp, and then used to match
     * the kind property. So, it is sufficient to use a value like
     * "audioinput" or "videoinput" to select for microphones and cameras
     * respectively.
     */
    kind: {type: String, value: ''},

    /**
     * A list of devices whose kind properties match the configured kind
     * value.
     */
    devices: {
      type: Array,
      readOnly: true,
      notify: true,
      value: function() {
        return [];
      }
    },

    /**
     * A selected device in the list of devices. This starts as the first
     * item in the devices array. The user can change this value by using
     * the selectNextDevice and selectPreviousDevice methods.
     *
     * @type {MediaDeviceInfo}
     */
    selectedDevice: {
      type: Object,
      readOnly: true,
      notify: true,
    },

    /**
     * The index of the currently selected device.
     */
    selectedDeviceIndex: {
      type: Number,
      readOnly: true,
      notify: true,
      value: 0,
    }
  },

  observers: [
    '_updateDevices(kind)',
    '_updateSelectedDevice(devices, selectedDeviceIndex)'
  ],

  /**
   * Select the next device in the list of devices. If the current device
   * is the last device, the first device is selected.
   */
  selectNextDevice: function() {
    var nextIndex = this.selectedDeviceIndex + 1;
    var minIndex = 0;

    if (this.devices != null && nextIndex < this.devices.length) {
      this._setSelectedDeviceIndex(nextIndex);
    } else {
      this._setSelectedDeviceIndex(minIndex);
    }
  },

  /**
   * Select the previous device in the list of devices. If the current
   * device is the first device, the last device is selected.
   */
  selectPreviousDevice: function() {
    var previousIndex = this.selectedDeviceIndex - 1;
    var maxIndex =
        this.devices && this.devices.length > 0 ? this.devices.length - 1 : 0;

    if (this.devices != null && previousIndex < this.devices.length &&
        previousIndex > -1) {
      this._setSelectedDeviceIndex(previousIndex);
    } else {
      this._setSelectedDeviceIndex(maxIndex);
    }
  },

  /**
   * Select a specific device from the list of devices.
   */
  selectDevice: function(device) {
    var index = this.devices.indexOf(device);
    if (index === -1) {
      throw new Error(
          'app-media-devices: ' +
          'The given object is not a known device.');
    }
    this._setSelectedDeviceIndex(index);
  },

  _updateSelectedDevice: function(devices, selectedDeviceIndex) {
    this.debounce('_updateSelectedDevice', function() {
      if (this.devices != null && this.selectedDeviceIndex != null) {
        this._setSelectedDevice(this.devices[this.selectedDeviceIndex]);
      }
    });
  },

  _updateDevices: function(kind) {
    try {
      var deviceKindRe = new RegExp(kind != null ? kind : '', 'i');
      var self = this;

      navigator.mediaDevices.enumerateDevices().then(function(devices) {
        var devicesOfKind = devices.filter(function(device) {
          return deviceKindRe.test(device.kind);
        });

        self._setSelectedDeviceIndex(0);
        self._setDevices(devicesOfKind);
      });
    } catch (e) {
      this._error(this._logf('Failed to look up media devices.', e));
    }
  }
});
