/**
@license
Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
This code may only be used under the BSD style license found at
http://polymer.github.io/LICENSE.txt The complete set of authors may be found at
http://polymer.github.io/AUTHORS.txt The complete set of contributors may be
found at http://polymer.github.io/CONTRIBUTORS.txt Code distributed by Google as
part of the polymer project is also subject to an additional IP rights grant
found at http://polymer.github.io/PATENTS.txt
*/
import {Base, Polymer} from '@polymer/polymer/polymer-legacy.js';

import {AppMedia as AppMedia$0} from './app-media-recorder.js';

export const AppMedia = AppMedia$0 || {};

/**
 * @enum {string}
 * @see https://www.w3.org/TR/image-capture/#MeteringMode
 */
AppMedia$0.MeteringMode = {
  SINGLE_SHOT: 'single-shot',
  CONTINUOUS: 'continuous',
  MANUAL: 'manual',
  NONE: 'none'
};

/**
 * @enum {string}
 * @see https://www.w3.org/TR/image-capture/#FillLightMode
 */
AppMedia$0.FillLightMode = {
  AUTO: 'auto',
  OFF: 'off',
  FLASH: 'flash'
};

var imageCaptureSupported = true;

if (window.ImageCapture == null) {
  Base._warn('Image Capture API is not supported in this browser!');
  imageCaptureSupported = false;
}

var TRACK_CONSTRAINT_NAMES = [
  'whiteBalanceMode',
  'colorTemperature',
  'exposureMode',
  'exposureCompensation',
  'iso',
  'focusMode',
  'pointsOfInterest',
  'brightness',
  'contrast',
  'saturation',
  'sharpness',
  'zoom',
  'torch'
];

var PHOTO_SETTING_NAMES = ['fillLightMode', 'imageHeight', 'imageWidth'];
/**
`app-media-image-capture` implements a helpful wrapper element for the emerging
[Image Capture API](https://www.w3.org/TR/image-capture/). This element enables
straight-forward photographic image control and capture that cooperates nicely
with other app-media elements.

NOTE: Image Capture API is still under development. As of today (April 7th,
2017), the API requires a polyfill or an [Origin
Trial](https://github.com/jpchase/OriginTrials/blob/gh-pages/developer-guide.md).
This element will politely neglect to register itself in browsers that do not
feature the appropriate ImageRecorder global constructor.

Chrome and Firefox have partial support for the API behind flags. See the
[ImageCapture API browser
support](https://github.com/w3c/mediacapture-image/blob/gh-pages/implementation-status.md)
page for details.

If you wish to support browsers that do not implement the Image Capture API,
please consider including a polyfill such as
[this one](https://github.com/GoogleChrome/imagecapture-polyfill) in your app.
Note that most polyfills will not enable full functionality, but they should
give your app a reasonable fallback in browsers that do not natively support the
API.

ELEMENT DESIGN NOTE: Many of the properties of this element have a default
configuration value of `null`. While this is weird, it is important to keep the
behavior of multi-property observers in the element implementation normalized
across Polymer 1.x and 2.x. Expect this aspect of the API to change as the
element graduates from the current hybrid implementation to something that can
rely on Polymer >=2.x observer semantics.

@group App Elements
@demo demo/index.html
*/
Polymer({
  is: 'app-media-image-capture',

  properties: {
    /**
     * The input media stream to capture images from. Note that the
     * stream must have at least one video track in order to be a suitable
     * source for image capture.
     *
     * @type {MediaStream}
     */
    stream: {type: Object},

    /**
     * MediaStreams can have multiple video tracks. This property enables
     * you to configure the index of the video track to use.
     */
    trackIndex: {
      type: Number,
      value: 0,
    },

    /**
     * The video track selected from the input MediaStream. This track will
     * be the source for any images captured.
     *
     * @type {MediaStreamTrack}
     */
    videoTrack: {
      type: Object,
      notify: true,
      computed: '__computeVideoTrack(stream, trackIndex)',
    },

    /**
     * An ImageCapture instance associated with the selected video track.
     *
     * @type {ImageCapture}
     */
    imageCapture: {
      type: Object,
      notify: true,
      computed: '__computeImageCapture(videoTrack)',
    },

    /**
     * The last photo captured by the image capture instance.
     *
     * @type {Blob}
     */
    lastPhoto: {
      type: Object,
      notify: true,
      readOnly: true,
    },

    /**
     * The last frame grabbed by the image capture instance.
     *
     * @type {ImageBitmap}
     */
    lastFrame: {
      type: Object,
      notify: true,
      readOnly: true,
    },

    /**
     * The PhotoCapabilities for the device providing the image data
     * associated with the chosen video track. This object contains
     * information about the minimum, maximum, current and incremental
     * values for various camera settings.
     *
     * @see https://w3c.github.io/mediacapture-image/##photocapabilities-section
     */
    photoCapabilities: {
      type: Object,
      readOnly: true,
      notify: true,
    },

    /**
     * The MediaTrackCapabilities for the device providing the image data
     * associated with the chosen video track. This object contains
     * information about the minimum, maximum, current and incremental
     * values for various camera settings.
     *
     * @see https://w3c.github.io/mediacapture-image/#mediatrackcapabilities-section
     */
    trackCapabilities: {
      type: Object,
      notify: true,
      computed: '__computeTrackCapabilities(imageCapture, videoTrack)',
    },

    /**
     * The PhotoSettings that will be used to configure the ImageCapture
     * instance used by this element. This configuration is generated
     * based on the individually configured properties on this element.
     * A full list of configurable properties can be found
     * [here](https://w3c.github.io/mediacapture-image/##photosettings-section).
     *
     * @type {!PhotoSettings}
     */
    photoSettings: {
      type: Object,
      readOnly: true,
      notify: true,
    },

    /**
     * @type {Polymer.AppMedia.FillLightMode}
     * @see https://w3c.github.io/mediacapture-image/##photosettings-section
     */
    fillLightMode: {
      type: String,
      value: null,
    },

    /** @see https://w3c.github.io/mediacapture-image/##photosettings-section */
    imageHeight: {
      type: Number,
      value: null,
    },

    /** @see https://w3c.github.io/mediacapture-image/##photosettings-section */
    imageWidth: {
      type: Number,
      value: null,
    },

    /**
     * Note that red eye reduction may not be controllable. If it is reported as
     * controllable, the value of this property will be respected. Otherwise it
     * will be ignored.
     *
     * @see https://w3c.github.io/mediacapture-image/##photosettings-section
     */
    redEyeReduction: {
      type: Boolean,
      value: null,
    },

    /**
     * The constraints that will be applied to the MediaStreamTrack that
     * is associated with the ImageCapture instance. This configuration is
     * generated based on the individually configured properties on this
     * element. A full list of configurable constraints can be found
     * [here](https://w3c.github.io/mediacapture-image/#constrainable-properties)
     *
     * Note that if a given setting is not supported by the current track, it
     * will be ignored. Also, any constraints that are suported will be clamped
     * to the bounds that are reported by the track PhotoCapabilities instance.
     *
     * @type {!MediaTrackConstraints}
     */
    trackConstraints: {
      type: Object,
      readOnly: true,
      notify: true,
    },

    /**
       @see https://w3c.github.io/mediacapture-image/#mediatracksettings-members
     */
    whiteBalanceMode: {
      type: String,
      value: null,
    },

    /**
       @see https://w3c.github.io/mediacapture-image/#mediatracksettings-members
     */
    colorTemperature: {
      type: Number,
      value: null,
    },

    /**
     * @type {Polymer.AppMedia.MeteringMode}
     * @see https://w3c.github.io/mediacapture-image/#mediatracksettings-members
     */
    exposureMode: {
      type: String,
      value: null,
    },

    /**
       @see https://w3c.github.io/mediacapture-image/#mediatracksettings-members
     */
    exposureCompensation: {
      type: Number,
      value: null,
    },

    /**
       @see https://w3c.github.io/mediacapture-image/#mediatracksettings-members
     */
    iso: {
      type: Number,
      value: null,
    },

    /**
     * @type {Polymer.AppMedia.MeteringMode}
     * @see https://w3c.github.io/mediacapture-image/#mediatracksettings-members
     */
    focusMode: {
      type: String,
      value: null,
    },

    /**
       @see https://w3c.github.io/mediacapture-image/#mediatracksettings-members
     */
    pointsOfInterest: {
      type: Array,
      value: null,
    },

    /**
       @see https://w3c.github.io/mediacapture-image/#mediatracksettings-members
     */
    brightness: {
      type: Number,
      value: null,
    },

    /**
       @see https://w3c.github.io/mediacapture-image/#mediatracksettings-members
     */
    contrast: {
      type: Number,
      value: null,
    },

    /**
       @see https://w3c.github.io/mediacapture-image/#mediatracksettings-members
     */
    saturation: {
      type: Number,
      value: null,
    },

    /**
       @see https://w3c.github.io/mediacapture-image/#mediatracksettings-members
     */
    sharpness: {
      type: Number,
      value: null,
    },

    /**
       @see https://w3c.github.io/mediacapture-image/#mediatracksettings-members
     */
    zoom: {
      type: Number,
      value: null,
    },

    /**
       @see https://w3c.github.io/mediacapture-image/#mediatracksettings-members
     */
    torch: {
      type: Boolean,
      value: null,
    }
  },

  observers: [
    '__updatePhotoCapabilities(imageCapture, videoTrack)',
    '__updatePhotoSettings(imageCapture, photoCapabilities, fillLightMode, imageHeight, imageWidth, redEyeReduction)',
    '__updateTrackConstraints(imageCapture, trackCapabilities, whiteBalanceMode, exposureMode, focusMode, pointsOfInterest, exposureCompensation, colorTemperature, iso, brightness, contrast, saturation, sharpness, zoom, torch)'
  ],

  /**
   * Take a photo. Returns a promise that resolves a photographic Blob.
   *
   * @return Promise<Blob>
   * @see https://www.w3.org/TR/image-capture/#dom-imagecapture-takephoto
   */
  takePhoto: function() {
    if (!imageCaptureSupported) {
      return;
    }

    if (this.imageCapture == null) {
      return Promise.reject(new Error('ImageCapture instance not ready.'));
    }

    return this.imageCapture.takePhoto(this.photoSettings)
        .then(function(photo) {
          this._setLastPhoto(photo);
          return photo;
        }.bind(this));
  },

  /**
   * Grab a frame from the camera. Returns a promise that resolves an
   * ImageBitmap suitable for drawing to a canvas or used in conjunction
   * with object detection.
   *
   * @return Promise<ImageBitmap>
   * @see https://www.w3.org/TR/image-capture/#dom-imagecapture-grabframe
   */
  grabFrame: function() {
    if (!imageCaptureSupported) {
      return;
    }

    if (this.imageCapture == null) {
      return Promise.reject(new Error('ImageCapture instance not ready.'));
    }

    return this.imageCapture.grabFrame().then(function(frame) {
      this._setLastFrame(frame);
      return frame;
    }.bind(this));
  },

  __computeVideoTrack: function(stream, trackIndex) {
    if (stream == null || trackIndex == null) {
      return;
    }

    return stream.getVideoTracks()[trackIndex];
  },

  __computeImageCapture: function(videoTrack) {
    if (!imageCaptureSupported) {
      return;
    }

    return new ImageCapture(videoTrack);
  },

  __computeTrackCapabilities: function(imageCapture, videoTrack) {
    if (imageCapture == null || videoTrack == null) {
      return;
    }

    return videoTrack.getCapabilities();
  },

  __updatePhotoCapabilities: function() {
    if (!imageCaptureSupported) {
      return;
    }

    this.debounce('updatePhotoCapabilities', function() {
      this.imageCapture.getPhotoCapabilities()
          .then(function(photoCapabilities) {
            this._setPhotoCapabilities(photoCapabilities);
          }.bind(this))
          .catch(function(e) {
            this.fire('image-capture-capabilities-error', e);
          }.bind(this));
    });
  },

  /**
   * ELEMENT DESIGN NOTE: While this observer does not rely on the
   * arguments passed to it, it also avoids the pitfall of relying on
   * properties that may be in a partial / indeterminant state at the
   * time that its debounce'd implementation is invoked. As long as
   * this remains true, it should not be subject to the undesirable
   * behavior exhibited by some elements such as iron-selector.
   */
  __updateTrackConstraints: function() {
    this.debounce('updateTrackConstraints', function() {
      var trackConstraints = this.__generateConfigurationObject(
          this.trackCapabilities, TRACK_CONSTRAINT_NAMES);

      // TODO(cdata): It's not clear how constraints from one controller
      // (such as this element) should attempt to compose with constraints
      // from some other controller that has a reference to the same
      // track. It's possible that we may benefit from some additional
      // abstraction dedicated to managing constraints (or perhaps
      // conflicts between controllers).
      // @see https://www.w3.org/TR/mediacapture-streams/#dfn-applyconstraints
      this._setTrackConstraints(trackConstraints);
    });
  },

  __updatePhotoSettings: function() {
    this.debounce('updatePhotoSettings', function() {
      // Don't update anything until we know the available capabilities:
      if (this.photoCapabilities == null) {
        return;
      }

      var photoSettings = this.__generateConfigurationObject(
          this.photoCapabilities, PHOTO_SETTING_NAMES);

      // Red eye reduction is a special case. The capability is reported
      // as one of three states, and only one (controllable) allows the
      // user to configure it.
      // @see https://w3c.github.io/mediacapture-image/#redeyereduction-section
      if (this.redEyeReduction != null &&
          this.photoCapabilities.redEyeReduction === 'controllable') {
        photoSettings.redEyeReduction = this.redEyeReduction;
      }

      this._setPhotoSettings(photoSettings);
    });
  },

  /**
   * The purpose of this method is to take a set of capabilities reported
   * by the platform, and sift the properties of this element through
   * the filter of those reported capabilities. Properties that are not
   * supported are ignored. Properties that are supported but don't have
   * values within a MediaSettingsRange are clamped to that range.
   * Properties with values that do not occur within a reported vector of
   * supported values are ignored.
   */
  __generateConfigurationObject: function(reportedCapabilities, allowedNames) {
    var configurationObject = {};

    if (reportedCapabilities != null) {
      for (var i = 0; i < allowedNames.length; ++i) {
        var name = allowedNames[i];
        var value = this[name];
        var capability = reportedCapabilities[name];
        var capabilityIsArray = Array.isArray(capability);

        // Don't set photo options if a value is provided, and skip
        // if the name is not present in the PhotoCapabilities object
        // (setting such properties will probably result in an error).
        if (value == null || capability == null || capability === false ||
            (capabilityIsArray && capability.indexOf(value) === -1)) {
          continue;
        }

        // If the capability is an object, then it is probably a
        // MediaSettingsRange. This means we need to clamp the value to the
        // min/max values of the range (otherwise it will likely throw when
        // we try to apply the configuration).
        // @see https://w3c.github.io/mediacapture-image/#mediasettingsrange
        if (!capabilityIsArray && capability instanceof Object) {
          if (capability.min >= value) {
            value = capability.min;
          } else if (capability.max <= value) {
            value = capability.max;
          }
        }

        configurationObject[name] = value;
      }
    }

    return configurationObject;
  }
});
