/**
@license
Copyright (c) 2017 The Polymer Project Authors. All rights reserved.
This code may only be used under the BSD style license found at
http://polymer.github.io/LICENSE.txt The complete set of authors may be found at
http://polymer.github.io/AUTHORS.txt The complete set of contributors may be
found at http://polymer.github.io/CONTRIBUTORS.txt Code distributed by Google as
part of the polymer project is also subject to an additional IP rights grant
found at http://polymer.github.io/PATENTS.txt
*/
import {IronResizableBehavior} from '@polymer/iron-resizable-behavior/iron-resizable-behavior.js';
import {html} from '@polymer/polymer/lib/utils/html-tag.js';
import {Polymer} from '@polymer/polymer/polymer-legacy.js';

/**
`app-media-video` is an element that converts a video source into a nicely
scaled video that is displayed to the viewer of the page.

@group App Elements
@demo demo/index.html
*/
Polymer({
  _template: html`
    <style>
      :host {
        display: flex;
        align-items: center;
        justify-content: center;
        flex-wrap: nowrap;
        width: 100%;
        height: 100%;
        background: var(--app-media-video-background, #000);
        overflow: hidden;
      }

      video {
        flex: 0 0 auto;
        transform-origin: center center;
        position: relative;
        will-change: transform;
      }
    </style>
    <video id="videoElement" on-loadedmetadata="_updateMetrics" muted="[[muted]]" autoplay="[[autoplay]]" loop="[[loop]]" webkit-playsinline playsinline>
    </video>
`,

  is: 'app-media-video',
  behaviors: [IronResizableBehavior],

  properties: {
    /**
     * The input source for the element. This can be a Media Stream, a
     * Blob or a string URL.
     *
     * @type {MediaStream|Blob|string}
     */
    source: {
      type: Object,
      notify: true,
      observer: '_sourceChanged',
    },

    /**
     * If true, the video will be scaled so that the source video is
     * flush with the edge of the element, but fully contained by it. If
     * false (the default), the video will be scaled to the smallest size
     * that is at full-bleed with respect to the element's bounding box.
     * Both settings preserve the aspect ratio of the source video.
     */
    contain: {
      type: Boolean,
      value: false,
    },

    /**
     * If true, the video will be inverted along the x-axis so that it is
     * effectively mirrored.
     */
    mirror: {
      type: Boolean,
      value: false,
    },

    /**
     * If true, the video is muted.
     */
    muted: {
      type: Boolean,
      value: false,
    },

    /**
     * If true, the video will automatically play when it has a source.
     */
    autoplay: {
      type: Boolean,
      value: false,
    },

    /**
     * If true, the video will loop when it reaches the end of the source.
     */
    loop: {
      type: Boolean,
      value: false,
    },

    /**
     * A bindable reference to the video element that actually plays the
     * source. This is sometimes useful in conjunction with
     * `app-media-audio`, which can accept an HTMLVideoElement as its
     * source.
     *
     * @type {HTMLVideoElement}
     */
    videoElement: {
      type: Object,
      notify: true,
      readOnly: true,
    },

    /**
     * A rect-like object that describes the projection of the source
     * video element to the viewport (the bounding box of the
     * app-media-video element).
     *
     * This rectangle is useful to understand the space within the
     * source video element that is visible to the user at any given
     * time. Depending on the value of contain, and also any transforms
     * that affect the scale of the video element, this rectangle could
     * have a range of values that are opportunistically calculated
     * and made available through this property.
     */
    sourceRect: {
      type: Object,
      notify: true,
      readOnly: true,
      value: function() {
        return {top: 0, left: 0, width: 0, height: 0};
      },
    },

    /**
     * This is the bounding ClientRect of the app-media-video element.
     * This is kept here for easy future access by users of the element
     * and is useful in conjunction with sourceRect.
     *
     * @type {ClientRect}
     */
    viewportRect: {
      type: Object,
      notify: true,
      readOnly: true,
    }
  },

  listeners: {
    'videoElement.loadedmetadata': '_updateMetrics',
    'iron-resize': '_updateMetrics'
  },

  observers: ['_updateMetrics(contain, mirror)'],

  attached: function() {
    this._setVideoElement(this.$.videoElement);
    this._updateMetrics();
  },

  /**
   * Play the video.
   */
  play: function() {
    this.$.videoElement.play();
  },

  /**
   * Pause the video.
   */
  pause: function() {
    this.$.videoElement.pause();
  },

  _sourceChanged: function() {
    var oldSrc = this.$.videoElement.src;
    var oldPaused = this.$.videoElement.paused;

    if (typeof this.source === 'string') {
      this.$.videoElement.src = this.source;
    } else {
      try {
        // NOTE(cdata): Chrome as of 55 does not support anything other than
        // MediaStream as the value of srcObject (even though the value is
        // allowed to be Blob, per the spec). We try the standardized way,
        // and then fall back to URL.createObjectURL if necessary.
        this.$.videoElement.srcObject = this.source;
      } catch (e) {
        if (this.source instanceof Blob) {
          this.$.videoElement.src = URL.createObjectURL(this.source);
        } else {
          this._error(this._logf(e));
        }
      }
    }

    if (typeof oldSrc === 'string') {
      try {
        URL.revokeObjectURL(oldSrc);
      } catch (e) {
      }
    }

    // Ensure that the video keeps playing if it was playing before we
    // changed the source:
    if (!oldPaused && this.$.videoElement.paused) {
      this.play();
    }
    // NOTE(cdata): No need to manually update metrics here, since the
    // new source will cause the video element to fire a loadmetadata
    // event when it is ready.
  },

  _updateMetrics: function() {
    this.debounce('_updateMetrics', function() {
      // These values are the width and the height of the video source
      // being displayed by the video element. They stay the same
      // regardless of the dimensions of the video element itself:
      var videoWidth = this.$.videoElement.videoWidth;
      var videoHeight = this.$.videoElement.videoHeight;

      // If the source video dimensions describe an effectively invisible
      // or unavailable video, we can just hide the element and exit early.
      if (!videoWidth || !videoHeight) {
        this.$.videoElement.style.visibility = 'hidden';
        return;
      } else {
        this.$.videoElement.style.visibility = '';
      }

      // Clear the current transform so that we can get the natural
      // bounding rect of the video element:
      this.$.videoElement.style.transform = '';

      var videoRect = this.$.videoElement.getBoundingClientRect();
      var selfRect = this.getBoundingClientRect();
      var selfRatio = selfRect.width / selfRect.height;
      var videoRatio = videoWidth / videoHeight;
      var scaleByHeight =
          this.contain ? videoRatio < selfRatio : videoRatio > selfRatio;
      // This is the scale of the source video's width compared to the
      // width of the video tag's bounding box.
      var sourceScale = videoWidth / videoRect.width;
      // This is the scale applied to the video to transform it based
      // on whether it should be contained within the viewport or not.
      var videoScale;

      if (scaleByHeight) {
        videoScale = selfRect.height / videoRect.height;
      } else {
        videoScale = selfRect.width / videoRect.width;
      }

      // Since we already grabbed this element's bounding rect, cache it
      // for future comparisons with the source rect:
      this._setViewportRect(selfRect);

      // The source rect needs to account for:
      //  - The (eventual) scale of the video element within this element
      //  - Any transforms on this or ancestor elements that may change
      //    the effective scale of the video element.
      //  - Vertical and horizontal centering of the video element within
      //    this element due to flex layout rules.
      var downScaledSelfWidth = selfRect.width / videoScale;
      var downScaledSelfHeight = selfRect.height / videoScale;
      var realHorizontalOverlap =
          (videoRect.width - downScaledSelfWidth) * sourceScale;
      var realVerticalOverlap =
          (videoRect.height - downScaledSelfHeight) * sourceScale;

      this._setSourceRect({
        left: realHorizontalOverlap / 2,
        top: realVerticalOverlap / 2,
        width: downScaledSelfWidth * sourceScale,
        height: downScaledSelfHeight * sourceScale
      });

      this.$.videoElement.style.transform = 'scale(' +
          (this.mirror ? -videoScale : videoScale) + ',' + videoScale + ')';
    });
  }
});
