/// <reference types="node" />
import { BaseError } from "make-error-cause";
import { SecureContext } from "tls";
import { Request, Response, CreateBody, ResponseOptions } from "servie/dist/node";
declare module "servie/dist/signal" {
    interface SignalEvents {
        error: [Error];
    }
}
/**
 * Address information from the HTTP request.
 */
export interface Connection {
    localPort: number;
    localAddress: string;
    remotePort: number;
    remoteAddress: string;
    encrypted: boolean;
}
/**
 * Extend response with URL.
 */
export interface HttpResponseOptions extends ResponseOptions {
    url: string;
    connection: Connection;
    httpVersion: string;
}
/**
 * HTTP responses implement a node.js body.
 */
export declare class HttpResponse extends Response implements HttpResponseOptions {
    url: string;
    httpVersion: string;
    connection: Connection;
    constructor(body: CreateBody, options: HttpResponseOptions);
}
export declare class Http2Response extends HttpResponse {
}
/**
 * Track HTTP connections for reuse.
 */
export declare class ConnectionManager<T> {
    connections: Map<string, T>;
    get(key: string): T | undefined;
    set(key: string, connection: T): T;
    delete(key: string, connection: T): T;
}
export interface ConcurrencyConnectionManagerOptions {
    maxConnections?: number;
    maxFreeConnections?: number;
}
export declare class ConnectionSet<T> {
    used: Set<T>;
    free: Set<T>;
    pending: Array<(connection?: T) => void>;
}
/**
 * Manage HTTP connection reuse.
 */
export declare class ConcurrencyConnectionManager<T> extends ConnectionManager<ConnectionSet<T>> {
    maxFreeConnections: number;
    maxConnections: number;
    constructor(maxFreeConnections?: number, maxConnections?: number);
    /**
     * Create a new connection.
     */
    ready(key: string, onReady: (existingConnection?: T) => void): void;
    getUsedConnection(key: string): T | undefined;
    getFreeConnection(key: string): T | undefined;
    use(key: string, connection: T): void;
    freed(key: string, connection: T, discard: () => void): void;
    remove(key: string, connection: T): void;
}
/**
 * Configure HTTP version negotiation.
 */
export declare enum NegotiateHttpVersion {
    HTTP1_ONLY = 0,
    HTTP2_FOR_HTTPS = 1,
    HTTP2_ONLY = 2
}
/**
 * Expose connection errors.
 */
export declare class ConnectionError extends BaseError {
    request: Request;
    code: string;
    constructor(request: Request, message: string, cause: Error);
}
/**
 * ALPN validation error.
 */
export declare class ALPNError extends Error {
    request: Request;
    code: string;
    constructor(request: Request, message: string);
}
/**
 * Node.js HTTP request options.
 */
export interface TransportOptions {
    keepAlive?: number;
    servername?: string;
    rejectUnauthorized?: boolean;
    ca?: string | Buffer | Array<string | Buffer>;
    cert?: string | Buffer;
    key?: string | Buffer;
    secureContext?: SecureContext;
    secureProtocol?: string;
    negotiateHttpVersion?: NegotiateHttpVersion;
}
/**
 * Custom abort error instance.
 */
export declare class AbortError extends Error {
    request: Request;
    code: string;
    constructor(request: Request, message: string);
}
/**
 * Forward request over HTTP1/1 or HTTP2, with TLS support.
 */
export declare function transport(options?: TransportOptions): (req: Request, next: () => Promise<HttpResponse>) => Promise<HttpResponse>;
