"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const pump = require("pump");
const url_1 = require("url");
const http_1 = require("http");
const https_1 = require("https");
const make_error_cause_1 = require("make-error-cause");
const net_1 = require("net");
const tls_1 = require("tls");
const http2_1 = require("http2");
const stream_1 = require("stream");
const node_1 = require("servie/dist/node");
const common_1 = require("servie/dist/common");
/**
 * HTTP responses implement a node.js body.
 */
class HttpResponse extends node_1.Response {
    constructor(body, options) {
        super(body, options);
        this.url = options.url;
        this.connection = options.connection;
        this.httpVersion = options.httpVersion;
    }
}
exports.HttpResponse = HttpResponse;
class Http2Response extends HttpResponse {
}
exports.Http2Response = Http2Response;
/**
 * Track HTTP connections for reuse.
 */
class ConnectionManager {
    constructor() {
        this.connections = new Map();
    }
    get(key) {
        return this.connections.get(key);
    }
    set(key, connection) {
        if (this.connections.has(key)) {
            throw new TypeError("Connection exists for key");
        }
        this.connections.set(key, connection);
        return connection;
    }
    delete(key, connection) {
        const existing = this.connections.get(key);
        if (existing !== connection) {
            throw new TypeError("Connection for key does not match");
        }
        this.connections.delete(key);
        return connection;
    }
}
exports.ConnectionManager = ConnectionManager;
class ConnectionSet {
    constructor() {
        this.used = new Set();
        this.free = new Set();
        this.pending = [];
    }
}
exports.ConnectionSet = ConnectionSet;
/**
 * Manage HTTP connection reuse.
 */
class ConcurrencyConnectionManager extends ConnectionManager {
    constructor(maxFreeConnections = 256, maxConnections = Infinity) {
        super();
        this.maxFreeConnections = maxFreeConnections;
        this.maxConnections = maxConnections;
    }
    /**
     * Create a new connection.
     */
    ready(key, onReady) {
        const pool = this.get(key);
        // No pool, zero connections.
        if (!pool)
            return onReady();
        // Reuse free connections first.
        if (pool.free.size)
            return onReady(this.getFreeConnection(key));
        // Add to "pending" queue.
        if (pool.used.size >= this.maxConnections) {
            pool.pending.push(onReady);
            return;
        }
        // Allow a new connection.
        return onReady();
    }
    getUsedConnection(key) {
        const pool = this.get(key);
        if (pool)
            return pool.used.values().next().value;
    }
    getFreeConnection(key) {
        const pool = this.get(key);
        if (pool)
            return pool.free.values().next().value;
    }
    use(key, connection) {
        const pool = this.get(key) || this.set(key, new ConnectionSet());
        pool.free.delete(connection);
        pool.used.add(connection);
    }
    freed(key, connection, discard) {
        const pool = this.get(key);
        if (!pool)
            return;
        // Remove from any possible "used".
        pool.used.delete(connection);
        pool.free.add(connection);
        // Discard when too many freed connections.
        if (pool.free.size >= this.maxFreeConnections)
            return discard();
        // Immediately send for connection.
        if (pool.pending.length) {
            const onReady = pool.pending.shift();
            return onReady(connection);
        }
    }
    remove(key, connection) {
        const pool = this.get(key);
        if (!pool)
            return;
        // Delete connection from pool.
        if (pool.used.has(connection))
            pool.used.delete(connection);
        if (pool.free.has(connection))
            pool.free.delete(connection);
        // Remove connection manager from pooling.
        if (!pool.free.size && !pool.used.size && !pool.pending.length) {
            this.delete(key, pool);
        }
    }
}
exports.ConcurrencyConnectionManager = ConcurrencyConnectionManager;
/**
 * Configure HTTP version negotiation.
 */
var NegotiateHttpVersion;
(function (NegotiateHttpVersion) {
    NegotiateHttpVersion[NegotiateHttpVersion["HTTP1_ONLY"] = 0] = "HTTP1_ONLY";
    NegotiateHttpVersion[NegotiateHttpVersion["HTTP2_FOR_HTTPS"] = 1] = "HTTP2_FOR_HTTPS";
    NegotiateHttpVersion[NegotiateHttpVersion["HTTP2_ONLY"] = 2] = "HTTP2_ONLY";
})(NegotiateHttpVersion = exports.NegotiateHttpVersion || (exports.NegotiateHttpVersion = {}));
/**
 * Write Servie body to node.js stream.
 */
function pumpBody(req, stream, onError) {
    const body = common_1.useRawBody(req);
    if (Buffer.isBuffer(body) || typeof body === "string" || body === null) {
        stream.end(body);
    }
    else {
        pump(body, stream, err => {
            if (err)
                return onError(err);
        });
    }
}
// Global connection caches.
const globalNetConnections = new ConcurrencyConnectionManager();
const globalTlsConnections = new ConcurrencyConnectionManager();
const globalHttp2Connections = new ConnectionManager();
/**
 * Expose connection errors.
 */
class ConnectionError extends make_error_cause_1.BaseError {
    constructor(request, message, cause) {
        super(message, cause);
        this.request = request;
        this.code = "EUNAVAILABLE";
    }
}
exports.ConnectionError = ConnectionError;
/**
 * Execute HTTP request.
 */
function execHttp1(req, url, keepAlive, socket) {
    return new Promise((resolve, reject) => {
        const encrypted = url.protocol === "https:";
        const request = encrypted ? https_1.request : http_1.request;
        const arg = {
            protocol: url.protocol,
            hostname: url.hostname,
            port: url.port,
            defaultPort: encrypted ? 443 : 80,
            method: req.method,
            path: url.pathname + url.search,
            headers: req.headers.asObject(),
            auth: url.username || url.password
                ? `${url.username}:${url.password}`
                : undefined,
            createConnection: () => socket
        };
        ref(socket);
        const rawRequest = request(arg);
        const requestStream = new stream_1.PassThrough();
        // Handle abort events correctly.
        const onAbort = () => {
            socket.emit("agentRemove"); // `abort` destroys the connection with no event.
            rawRequest.abort();
        };
        // Reuse HTTP connections where possible.
        if (keepAlive > 0) {
            rawRequest.shouldKeepAlive = true;
            rawRequest.setHeader("Connection", "keep-alive");
        }
        // Trigger unavailable error when node.js errors before response.
        const onRequestError = (err) => {
            unref(socket);
            req.signal.off("abort", onAbort);
            req.signal.emit("error", err);
            rawRequest.removeListener("response", onResponse);
            return reject(new ConnectionError(req, `Unable to connect to ${url.host}`, err));
        };
        // Track the node.js response.
        const onResponse = (rawResponse) => {
            // Trailers are populated on "end".
            let resolveTrailers;
            const trailer = new Promise(resolve => (resolveTrailers = resolve));
            rawRequest.removeListener("response", onResponse);
            rawRequest.removeListener("error", onRequestError);
            const { address: localAddress, port: localPort } = rawRequest.connection.address();
            const { address: remoteAddress, port: remotePort } = rawResponse.connection.address();
            let bytesTransferred = 0;
            req.signal.emit("responseStarted");
            const onData = (chunk) => {
                req.signal.emit("responseBytes", (bytesTransferred += chunk.length));
            };
            rawResponse.on("data", onData);
            const res = new HttpResponse(pump(rawResponse, new stream_1.PassThrough(), err => {
                unref(socket);
                req.signal.off("abort", onAbort);
                if (err)
                    req.signal.emit("error", err);
                rawResponse.removeListener("data", onData);
                resolveTrailers(rawResponse.trailers);
                req.signal.emit("responseEnded");
            }), {
                status: rawResponse.statusCode,
                statusText: rawResponse.statusMessage,
                url: req.url,
                headers: rawResponse.headers,
                omitDefaultHeaders: true,
                trailer,
                connection: {
                    localAddress,
                    localPort,
                    remoteAddress,
                    remotePort,
                    encrypted
                },
                httpVersion: rawResponse.httpVersion
            });
            return resolve(res);
        };
        let bytesTransferred = 0;
        req.signal.emit("requestStarted");
        const onData = (chunk) => {
            req.signal.emit("requestBytes", (bytesTransferred += chunk.length));
        };
        req.signal.on("abort", onAbort);
        rawRequest.once("error", onRequestError);
        rawRequest.once("response", onResponse);
        requestStream.on("data", onData);
        pump(requestStream, rawRequest, () => {
            requestStream.removeListener("data", onData);
            req.signal.emit("requestEnded");
        });
        return pumpBody(req, requestStream, reject);
    });
}
/**
 * ALPN validation error.
 */
class ALPNError extends Error {
    constructor(request, message) {
        super(message);
        this.request = request;
        this.code = "EALPNPROTOCOL";
    }
}
exports.ALPNError = ALPNError;
/**
 * Execute a HTTP2 connection.
 */
function execHttp2(req, url, client) {
    return new Promise((resolve, reject) => {
        // HTTP2 formatted headers.
        const headers = Object.assign(req.headers.asObject(), {
            [http2_1.constants.HTTP2_HEADER_PATH]: url.pathname + url.search,
            [http2_1.constants.HTTP2_HEADER_METHOD]: req.method
        });
        const http2Stream = client.request(headers, { endStream: false });
        const requestStream = new stream_1.PassThrough();
        ref(client.socket); // Request ref tracking.
        // Trigger unavailable error when node.js errors before response.
        const onRequestError = (err) => {
            unref(client.socket);
            req.signal.off("abort", onAbort);
            req.signal.emit("error", err);
            http2Stream.removeListener("response", onResponse);
            return reject(new ConnectionError(req, `Unable to connect to ${url.host}`, err));
        };
        const onResponse = (headers) => {
            const encrypted = client.socket.encrypted === true;
            const { localAddress, localPort, remoteAddress = "", remotePort = 0 } = client.socket;
            let resolveTrailers;
            const trailer = new Promise(resolve => (resolveTrailers = resolve));
            http2Stream.removeListener("error", onRequestError);
            http2Stream.removeListener("response", onResponse);
            let bytesTransferred = 0;
            req.signal.emit("responseStarted");
            const onTrailers = (headers) => {
                resolveTrailers(headers);
            };
            const onData = (chunk) => {
                req.signal.emit("responseBytes", (bytesTransferred += chunk.length));
            };
            http2Stream.on("data", onData);
            http2Stream.once("trailers", onTrailers);
            const res = new Http2Response(pump(http2Stream, new stream_1.PassThrough(), err => {
                unref(client.socket);
                req.signal.off("abort", onAbort);
                if (err)
                    req.signal.emit("error", err);
                http2Stream.removeListener("data", onData);
                http2Stream.removeListener("data", onTrailers);
                resolveTrailers({}); // Resolve in case "trailers" wasn't emitted.
                req.signal.emit("responseEnded");
            }), {
                status: Number(headers[http2_1.constants.HTTP2_HEADER_STATUS]),
                statusText: "",
                url: req.url,
                httpVersion: "2.0",
                headers,
                omitDefaultHeaders: true,
                trailer,
                connection: {
                    localAddress,
                    localPort,
                    remoteAddress,
                    remotePort,
                    encrypted
                }
            });
            return resolve(res);
        };
        let bytesTransferred = 0;
        req.signal.emit("requestStarted");
        const onAbort = () => http2Stream.destroy();
        const onData = (chunk) => {
            req.signal.emit("requestBytes", (bytesTransferred += chunk.length));
        };
        req.signal.on("abort", onAbort);
        http2Stream.once("error", onRequestError);
        http2Stream.once("response", onResponse);
        requestStream.on("data", onData);
        pump(requestStream, http2Stream, () => {
            requestStream.removeListener("data", onData);
            req.signal.emit("requestEnded");
        });
        return pumpBody(req, requestStream, reject);
    });
}
/**
 * Custom abort error instance.
 */
class AbortError extends Error {
    constructor(request, message) {
        super(message);
        this.request = request;
        this.code = "EABORT";
    }
}
exports.AbortError = AbortError;
/**
 * Forward request over HTTP1/1 or HTTP2, with TLS support.
 */
function transport(options = {}) {
    const { keepAlive = 5000, // Default to keeping a connection open briefly.
    negotiateHttpVersion = NegotiateHttpVersion.HTTP2_FOR_HTTPS } = options;
    // TODO: Allow configuration in options.
    const tlsConnections = globalTlsConnections;
    const netConnections = globalNetConnections;
    const http2Connections = globalHttp2Connections;
    return async function (req, next) {
        const url = new url_1.URL(req.url, "http://localhost");
        const { hostname: host, protocol } = url;
        if (req.signal.aborted) {
            throw new AbortError(req, "Request has been aborted");
        }
        if (protocol === "http:") {
            const port = Number(url.port) || 80;
            const connectionKey = `${host}:${port}:${negotiateHttpVersion}`;
            // Use existing HTTP2 session in HTTP2 mode.
            if (negotiateHttpVersion === NegotiateHttpVersion.HTTP2_ONLY) {
                const existingSession = http2Connections.get(connectionKey);
                if (existingSession)
                    return execHttp2(req, url, existingSession);
            }
            return new Promise(resolve => {
                return netConnections.ready(connectionKey, freeSocket => {
                    const socketOptions = { host, port };
                    const socket = freeSocket ||
                        setupSocket(connectionKey, keepAlive, netConnections, net_1.connect(socketOptions));
                    netConnections.use(connectionKey, socket);
                    if (negotiateHttpVersion === NegotiateHttpVersion.HTTP2_ONLY) {
                        const authority = `${protocol}//${host}:${port}`;
                        const client = manageHttp2(authority, connectionKey, keepAlive, http2Connections, socket);
                        return resolve(execHttp2(req, url, client));
                    }
                    return resolve(execHttp1(req, url, keepAlive, socket));
                });
            });
        }
        // Optionally negotiate HTTP2 connection.
        if (protocol === "https:") {
            const { ca, cert, key, secureProtocol, secureContext } = options;
            const port = Number(url.port) || 443;
            const servername = options.servername ||
                calculateServerName(host, req.headers.get("host"));
            const rejectUnauthorized = options.rejectUnauthorized !== false;
            const connectionKey = `${host}:${port}:${negotiateHttpVersion}:${servername}:${rejectUnauthorized}:${ca ||
                ""}:${cert || ""}:${key || ""}:${secureProtocol || ""}`;
            // Use an existing TLS session to speed up handshake.
            const existingSocket = tlsConnections.getFreeConnection(connectionKey) ||
                tlsConnections.getUsedConnection(connectionKey);
            const session = existingSocket ? existingSocket.getSession() : undefined;
            const socketOptions = {
                host,
                port,
                servername,
                rejectUnauthorized,
                ca,
                cert,
                key,
                session,
                secureProtocol,
                secureContext
            };
            // Use any existing HTTP2 session.
            if (negotiateHttpVersion === NegotiateHttpVersion.HTTP2_ONLY ||
                negotiateHttpVersion === NegotiateHttpVersion.HTTP2_FOR_HTTPS) {
                const existingSession = http2Connections.get(connectionKey);
                if (existingSession)
                    return execHttp2(req, url, existingSession);
            }
            return new Promise((resolve, reject) => {
                // Set up ALPN protocols for connection negotiation.
                if (negotiateHttpVersion === NegotiateHttpVersion.HTTP2_ONLY) {
                    socketOptions.ALPNProtocols = ["h2"];
                }
                else if (negotiateHttpVersion === NegotiateHttpVersion.HTTP2_FOR_HTTPS) {
                    socketOptions.ALPNProtocols = ["h2", "http/1.1"];
                }
                return tlsConnections.ready(connectionKey, freeSocket => {
                    const socket = freeSocket ||
                        setupSocket(connectionKey, keepAlive, tlsConnections, tls_1.connect(socketOptions));
                    tlsConnections.use(connectionKey, socket);
                    if (negotiateHttpVersion === NegotiateHttpVersion.HTTP1_ONLY) {
                        return resolve(execHttp1(req, url, keepAlive, socket));
                    }
                    if (negotiateHttpVersion === NegotiateHttpVersion.HTTP2_ONLY) {
                        const client = manageHttp2(`${protocol}//${host}:${port}`, connectionKey, keepAlive, http2Connections, socket);
                        return resolve(execHttp2(req, url, client));
                    }
                    const onError = (err) => {
                        socket.removeListener("connect", onConnect);
                        return reject(new ConnectionError(req, `Unable to connect to ${host}:${port}`, err));
                    };
                    // Execute HTTP connection according to negotiated ALPN protocol.
                    const onConnect = () => {
                        socket.removeListener("error", onError);
                        const alpnProtocol = socket.alpnProtocol;
                        // Successfully negotiated HTTP2 connection.
                        if (alpnProtocol === "h2") {
                            const existingClient = http2Connections.get(connectionKey);
                            if (existingClient) {
                                socket.destroy(); // Destroy socket in case of TLS connection race.
                                return resolve(execHttp2(req, url, existingClient));
                            }
                            const client = manageHttp2(`${protocol}//${host}:${port}`, connectionKey, keepAlive, http2Connections, socket);
                            return resolve(execHttp2(req, url, client));
                        }
                        if (alpnProtocol === "http/1.1" || alpnProtocol === false) {
                            return resolve(execHttp1(req, url, keepAlive, socket));
                        }
                        return reject(new ALPNError(req, `Unknown ALPN protocol negotiated: ${alpnProtocol}`));
                    };
                    // Existing socket may already have negotiated ALPN protocol.
                    if (socket.alpnProtocol != null)
                        return onConnect();
                    socket.once("secureConnect", onConnect);
                    socket.once("error", onError);
                });
            });
        }
        return next();
    };
}
exports.transport = transport;
/**
 * Setup the socket with the connection manager.
 *
 * Ref: https://github.com/nodejs/node/blob/531b4bedcac14044f09129ffb65dab71cc2707d9/lib/_http_agent.js#L254
 */
function setupSocket(key, keepAlive, manager, socket) {
    const onFree = () => {
        if (keepAlive > 0)
            socket.setKeepAlive(true, keepAlive);
        manager.freed(key, socket, () => socket.destroy());
    };
    const cleanup = () => {
        socket.removeListener("free", onFree);
        socket.removeListener("close", cleanup);
        socket.removeListener("agentRemove", cleanup);
        manager.remove(key, socket);
    };
    socket.on("free", onFree);
    socket.on("close", cleanup);
    socket.on("agentRemove", cleanup);
    return socket;
}
/**
 * Set up a HTTP2 working session.
 */
function manageHttp2(authority, key, keepAlive, manager, socket) {
    // TODO: Fix node.js types.
    const connectOptions = { createConnection: () => socket };
    const client = http2_1.connect(authority, connectOptions);
    manager.set(key, client);
    client.once("close", () => manager.delete(key, client));
    client.setTimeout(keepAlive, () => client.close());
    return client;
}
/**
 * Track socket usage.
 */
const SOCKET_REFS = new WeakMap();
/**
 * Track socket refs.
 */
function ref(socket) {
    const count = SOCKET_REFS.get(socket) || 0;
    if (count === 0)
        socket.ref();
    SOCKET_REFS.set(socket, count + 1);
}
/**
 * Track socket unrefs and globally unref.
 */
function unref(socket) {
    const count = SOCKET_REFS.get(socket);
    if (count) {
        if (count === 1) {
            socket.unref();
            SOCKET_REFS.delete(socket);
        }
        else {
            SOCKET_REFS.set(socket, count - 1);
        }
    }
}
/**
 * Ref: https://github.com/nodejs/node/blob/5823938d156f4eb6dc718746afbf58f1150f70fb/lib/_http_agent.js#L231
 */
function calculateServerName(host, hostHeader) {
    if (!hostHeader)
        return host;
    if (hostHeader.charAt(0) === "[") {
        const index = hostHeader.indexOf("]");
        if (index === -1)
            return hostHeader;
        return hostHeader.substr(1, index - 1);
    }
    return hostHeader.split(":", 1)[0];
}
//# sourceMappingURL=index.js.map