'use strict';

exports.__esModule = true;

var _createClass = function () { function defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; }();

var _typeof = typeof Symbol === "function" && typeof Symbol.iterator === "symbol" ? function (obj) { return typeof obj; } : function (obj) { return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj; };

var _mapGenerator = require('./map-generator');

var _mapGenerator2 = _interopRequireDefault(_mapGenerator);

var _stringify2 = require('./stringify');

var _stringify3 = _interopRequireDefault(_stringify2);

var _warnOnce = require('./warn-once');

var _warnOnce2 = _interopRequireDefault(_warnOnce);

var _result = require('./result');

var _result2 = _interopRequireDefault(_result);

var _parse = require('./parse');

var _parse2 = _interopRequireDefault(_parse);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

function isPromise(obj) {
  return (typeof obj === 'undefined' ? 'undefined' : _typeof(obj)) === 'object' && typeof obj.then === 'function';
}

/**
 * A Promise proxy for the result of PostCSS transformations.
 *
 * A `LazyResult` instance is returned by {@link Processor#process}.
 *
 * @example
 * const lazy = postcss([cssnext]).process(css)
 */

var LazyResult = function () {
  function LazyResult(processor, css, opts) {
    _classCallCheck(this, LazyResult);

    this.stringified = false;
    this.processed = false;

    var root = void 0;
    if ((typeof css === 'undefined' ? 'undefined' : _typeof(css)) === 'object' && css !== null && css.type === 'root') {
      root = css;
    } else if (css instanceof LazyResult || css instanceof _result2.default) {
      root = css.root;
      if (css.map) {
        if (typeof opts.map === 'undefined') opts.map = {};
        if (!opts.map.inline) opts.map.inline = false;
        opts.map.prev = css.map;
      }
    } else {
      var parser = _parse2.default;
      if (opts.syntax) parser = opts.syntax.parse;
      if (opts.parser) parser = opts.parser;
      if (parser.parse) parser = parser.parse;

      try {
        root = parser(css, opts);
      } catch (error) {
        this.error = error;
      }
    }

    this.result = new _result2.default(processor, root, opts);
  }

  /**
   * Returns a {@link Processor} instance, which will be used
   * for CSS transformations.
   *
   * @type {Processor}
   */


  /**
   * Processes input CSS through synchronous plugins
   * and calls {@link Result#warnings()}.
   *
   * @return {Warning[]} Warnings from plugins.
   */
  LazyResult.prototype.warnings = function warnings() {
    return this.sync().warnings();
  };

  /**
   * Alias for the {@link LazyResult#css} property.
   *
   * @example
   * lazy + '' === lazy.css
   *
   * @return {string} Output CSS.
   */


  LazyResult.prototype.toString = function toString() {
    return this.css;
  };

  /**
   * Processes input CSS through synchronous and asynchronous plugins
   * and calls `onFulfilled` with a Result instance. If a plugin throws
   * an error, the `onRejected` callback will be executed.
   *
   * It implements standard Promise API.
   *
   * @param {onFulfilled} onFulfilled Callback will be executed
   *                                  when all plugins will finish work.
   * @param {onRejected}  onRejected  Callback will be executed on any error.
   *
   * @return {Promise} Promise API to make queue.
   *
   * @example
   * postcss([cssnext]).process(css, { from: cssPath }).then(result => {
   *   console.log(result.css)
   * })
   */


  LazyResult.prototype.then = function then(onFulfilled, onRejected) {
    if (process.env.NODE_ENV !== 'production') {
      if (!('from' in this.opts)) {
        (0, _warnOnce2.default)('Without `from` option PostCSS could generate wrong source map ' + 'and will not find Browserslist config. Set it to CSS file path ' + 'or to `undefined` to prevent this warning.');
      }
    }
    return this.async().then(onFulfilled, onRejected);
  };

  /**
   * Processes input CSS through synchronous and asynchronous plugins
   * and calls onRejected for each error thrown in any plugin.
   *
   * It implements standard Promise API.
   *
   * @param {onRejected} onRejected Callback will be executed on any error.
   *
   * @return {Promise} Promise API to make queue.
   *
   * @example
   * postcss([cssnext]).process(css).then(result => {
   *   console.log(result.css)
   * }).catch(error => {
   *   console.error(error)
   * })
   */


  LazyResult.prototype.catch = function _catch(onRejected) {
    return this.async().catch(onRejected);
  };
  /**
   * Processes input CSS through synchronous and asynchronous plugins
   * and calls onFinally on any error or when all plugins will finish work.
   *
   * It implements standard Promise API.
   *
   * @param {onFinally} onFinally Callback will be executed on any error or
   *                              when all plugins will finish work.
   *
   * @return {Promise} Promise API to make queue.
   *
   * @example
   * postcss([cssnext]).process(css).finally(() => {
   *   console.log('processing ended')
   * })
   */


  LazyResult.prototype.finally = function _finally(onFinally) {
    return this.async().then(onFinally, onFinally);
  };

  LazyResult.prototype.handleError = function handleError(error, plugin) {
    try {
      this.error = error;
      if (error.name === 'CssSyntaxError' && !error.plugin) {
        error.plugin = plugin.postcssPlugin;
        error.setMessage();
      } else if (plugin.postcssVersion) {
        if (process.env.NODE_ENV !== 'production') {
          var pluginName = plugin.postcssPlugin;
          var pluginVer = plugin.postcssVersion;
          var runtimeVer = this.result.processor.version;
          var a = pluginVer.split('.');
          var b = runtimeVer.split('.');

          if (a[0] !== b[0] || parseInt(a[1]) > parseInt(b[1])) {
            console.error('Unknown error from PostCSS plugin. Your current PostCSS ' + 'version is ' + runtimeVer + ', but ' + pluginName + ' uses ' + pluginVer + '. Perhaps this is the source of the error below.');
          }
        }
      }
    } catch (err) {
      if (console && console.error) console.error(err);
    }
  };

  LazyResult.prototype.asyncTick = function asyncTick(resolve, reject) {
    var _this = this;

    if (this.plugin >= this.processor.plugins.length) {
      this.processed = true;
      return resolve();
    }

    try {
      var plugin = this.processor.plugins[this.plugin];
      var promise = this.run(plugin);
      this.plugin += 1;

      if (isPromise(promise)) {
        promise.then(function () {
          _this.asyncTick(resolve, reject);
        }).catch(function (error) {
          _this.handleError(error, plugin);
          _this.processed = true;
          reject(error);
        });
      } else {
        this.asyncTick(resolve, reject);
      }
    } catch (error) {
      this.processed = true;
      reject(error);
    }
  };

  LazyResult.prototype.async = function async() {
    var _this2 = this;

    if (this.processed) {
      return new Promise(function (resolve, reject) {
        if (_this2.error) {
          reject(_this2.error);
        } else {
          resolve(_this2.stringify());
        }
      });
    }
    if (this.processing) {
      return this.processing;
    }

    this.processing = new Promise(function (resolve, reject) {
      if (_this2.error) return reject(_this2.error);
      _this2.plugin = 0;
      _this2.asyncTick(resolve, reject);
    }).then(function () {
      _this2.processed = true;
      return _this2.stringify();
    });

    return this.processing;
  };

  LazyResult.prototype.sync = function sync() {
    if (this.processed) return this.result;
    this.processed = true;

    if (this.processing) {
      throw new Error('Use process(css).then(cb) to work with async plugins');
    }

    if (this.error) throw this.error;

    for (var _iterator = this.result.processor.plugins, _isArray = Array.isArray(_iterator), _i = 0, _iterator = _isArray ? _iterator : _iterator[Symbol.iterator]();;) {
      var _ref;

      if (_isArray) {
        if (_i >= _iterator.length) break;
        _ref = _iterator[_i++];
      } else {
        _i = _iterator.next();
        if (_i.done) break;
        _ref = _i.value;
      }

      var plugin = _ref;

      var promise = this.run(plugin);
      if (isPromise(promise)) {
        throw new Error('Use process(css).then(cb) to work with async plugins');
      }
    }

    return this.result;
  };

  LazyResult.prototype.run = function run(plugin) {
    this.result.lastPlugin = plugin;

    try {
      return plugin(this.result.root, this.result);
    } catch (error) {
      this.handleError(error, plugin);
      throw error;
    }
  };

  LazyResult.prototype.stringify = function stringify() {
    if (this.stringified) return this.result;
    this.stringified = true;

    this.sync();

    var opts = this.result.opts;
    var str = _stringify3.default;
    if (opts.syntax) str = opts.syntax.stringify;
    if (opts.stringifier) str = opts.stringifier;
    if (str.stringify) str = str.stringify;

    var map = new _mapGenerator2.default(str, this.result.root, this.result.opts);
    var data = map.generate();
    this.result.css = data[0];
    this.result.map = data[1];

    return this.result;
  };

  _createClass(LazyResult, [{
    key: 'processor',
    get: function get() {
      return this.result.processor;
    }

    /**
     * Options from the {@link Processor#process} call.
     *
     * @type {processOptions}
     */

  }, {
    key: 'opts',
    get: function get() {
      return this.result.opts;
    }

    /**
     * Processes input CSS through synchronous plugins, converts `Root`
     * to a CSS string and returns {@link Result#css}.
     *
     * This property will only work with synchronous plugins.
     * If the processor contains any asynchronous plugins
     * it will throw an error. This is why this method is only
     * for debug purpose, you should always use {@link LazyResult#then}.
     *
     * @type {string}
     * @see Result#css
     */

  }, {
    key: 'css',
    get: function get() {
      return this.stringify().css;
    }

    /**
     * An alias for the `css` property. Use it with syntaxes
     * that generate non-CSS output.
     *
     * This property will only work with synchronous plugins.
     * If the processor contains any asynchronous plugins
     * it will throw an error. This is why this method is only
     * for debug purpose, you should always use {@link LazyResult#then}.
     *
     * @type {string}
     * @see Result#content
     */

  }, {
    key: 'content',
    get: function get() {
      return this.stringify().content;
    }

    /**
     * Processes input CSS through synchronous plugins
     * and returns {@link Result#map}.
     *
     * This property will only work with synchronous plugins.
     * If the processor contains any asynchronous plugins
     * it will throw an error. This is why this method is only
     * for debug purpose, you should always use {@link LazyResult#then}.
     *
     * @type {SourceMapGenerator}
     * @see Result#map
     */

  }, {
    key: 'map',
    get: function get() {
      return this.stringify().map;
    }

    /**
     * Processes input CSS through synchronous plugins
     * and returns {@link Result#root}.
     *
     * This property will only work with synchronous plugins. If the processor
     * contains any asynchronous plugins it will throw an error.
     *
     * This is why this method is only for debug purpose,
     * you should always use {@link LazyResult#then}.
     *
     * @type {Root}
     * @see Result#root
     */

  }, {
    key: 'root',
    get: function get() {
      return this.sync().root;
    }

    /**
     * Processes input CSS through synchronous plugins
     * and returns {@link Result#messages}.
     *
     * This property will only work with synchronous plugins. If the processor
     * contains any asynchronous plugins it will throw an error.
     *
     * This is why this method is only for debug purpose,
     * you should always use {@link LazyResult#then}.
     *
     * @type {Message[]}
     * @see Result#messages
     */

  }, {
    key: 'messages',
    get: function get() {
      return this.sync().messages;
    }
  }]);

  return LazyResult;
}();

exports.default = LazyResult;

/**
 * @callback onFulfilled
 * @param {Result} result
 */

/**
 * @callback onRejected
 * @param {Error} error
 */

module.exports = exports['default'];
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
