'use strict';

exports.__esModule = true;
/**
 * Contains helpers for safely splitting lists of CSS values,
 * preserving parentheses and quotes.
 *
 * @example
 * const list = postcss.list
 *
 * @namespace list
 */
var list = {
  split: function split(string, separators, last) {
    var array = [];
    var current = '';
    var split = false;

    var func = 0;
    var quote = false;
    var escape = false;

    for (var i = 0; i < string.length; i++) {
      var letter = string[i];

      if (quote) {
        if (escape) {
          escape = false;
        } else if (letter === '\\') {
          escape = true;
        } else if (letter === quote) {
          quote = false;
        }
      } else if (letter === '"' || letter === '\'') {
        quote = letter;
      } else if (letter === '(') {
        func += 1;
      } else if (letter === ')') {
        if (func > 0) func -= 1;
      } else if (func === 0) {
        if (separators.indexOf(letter) !== -1) split = true;
      }

      if (split) {
        if (current !== '') array.push(current.trim());
        current = '';
        split = false;
      } else {
        current += letter;
      }
    }

    if (last || current !== '') array.push(current.trim());
    return array;
  },


  /**
   * Safely splits space-separated values (such as those for `background`,
   * `border-radius`, and other shorthand properties).
   *
   * @param {string} string Space-separated values.
   *
   * @return {string[]} Split values.
   *
   * @example
   * postcss.list.space('1px calc(10% + 1px)') //=> ['1px', 'calc(10% + 1px)']
   */
  space: function space(string) {
    var spaces = [' ', '\n', '\t'];
    return list.split(string, spaces);
  },


  /**
   * Safely splits comma-separated values (such as those for `transition-*`
   * and `background` properties).
   *
   * @param {string} string Comma-separated values.
   *
   * @return {string[]} Split values.
   *
   * @example
   * postcss.list.comma('black, linear-gradient(white, black)')
   * //=> ['black', 'linear-gradient(white, black)']
   */
  comma: function comma(string) {
    var comma = ',';
    return list.split(string, [comma], true);
  }
};

exports.default = list;
module.exports = exports['default'];
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
