'use strict';

exports.__esModule = true;

var _declaration = require('./declaration');

var _declaration2 = _interopRequireDefault(_declaration);

var _tokenize = require('./tokenize');

var _tokenize2 = _interopRequireDefault(_tokenize);

var _comment = require('./comment');

var _comment2 = _interopRequireDefault(_comment);

var _atRule = require('./at-rule');

var _atRule2 = _interopRequireDefault(_atRule);

var _root = require('./root');

var _root2 = _interopRequireDefault(_root);

var _rule = require('./rule');

var _rule2 = _interopRequireDefault(_rule);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

var Parser = function () {
  function Parser(input) {
    _classCallCheck(this, Parser);

    this.input = input;

    this.root = new _root2.default();
    this.current = this.root;
    this.spaces = '';
    this.semicolon = false;

    this.createTokenizer();
    this.root.source = { input: input, start: { line: 1, column: 1 } };
  }

  Parser.prototype.createTokenizer = function createTokenizer() {
    this.tokenizer = (0, _tokenize2.default)(this.input);
  };

  Parser.prototype.parse = function parse() {
    var token = void 0;
    while (!this.tokenizer.endOfFile()) {
      token = this.tokenizer.nextToken();

      switch (token[0]) {
        case 'space':
          this.spaces += token[1];
          break;

        case ';':
          this.freeSemicolon(token);
          break;

        case '}':
          this.end(token);
          break;

        case 'comment':
          this.comment(token);
          break;

        case 'at-word':
          this.atrule(token);
          break;

        case '{':
          this.emptyRule(token);
          break;

        default:
          this.other(token);
          break;
      }
    }
    this.endFile();
  };

  Parser.prototype.comment = function comment(token) {
    var node = new _comment2.default();
    this.init(node, token[2], token[3]);
    node.source.end = { line: token[4], column: token[5] };

    var text = token[1].slice(2, -2);
    if (/^\s*$/.test(text)) {
      node.text = '';
      node.raws.left = text;
      node.raws.right = '';
    } else {
      var match = text.match(/^(\s*)([^]*[^\s])(\s*)$/);
      node.text = match[2];
      node.raws.left = match[1];
      node.raws.right = match[3];
    }
  };

  Parser.prototype.emptyRule = function emptyRule(token) {
    var node = new _rule2.default();
    this.init(node, token[2], token[3]);
    node.selector = '';
    node.raws.between = '';
    this.current = node;
  };

  Parser.prototype.other = function other(start) {
    var end = false;
    var type = null;
    var colon = false;
    var bracket = null;
    var brackets = [];

    var tokens = [];
    var token = start;
    while (token) {
      type = token[0];
      tokens.push(token);

      if (type === '(' || type === '[') {
        if (!bracket) bracket = token;
        brackets.push(type === '(' ? ')' : ']');
      } else if (brackets.length === 0) {
        if (type === ';') {
          if (colon) {
            this.decl(tokens);
            return;
          } else {
            break;
          }
        } else if (type === '{') {
          this.rule(tokens);
          return;
        } else if (type === '}') {
          this.tokenizer.back(tokens.pop());
          end = true;
          break;
        } else if (type === ':') {
          colon = true;
        }
      } else if (type === brackets[brackets.length - 1]) {
        brackets.pop();
        if (brackets.length === 0) bracket = null;
      }

      token = this.tokenizer.nextToken();
    }

    if (this.tokenizer.endOfFile()) end = true;
    if (brackets.length > 0) this.unclosedBracket(bracket);

    if (end && colon) {
      while (tokens.length) {
        token = tokens[tokens.length - 1][0];
        if (token !== 'space' && token !== 'comment') break;
        this.tokenizer.back(tokens.pop());
      }
      this.decl(tokens);
    } else {
      this.unknownWord(tokens);
    }
  };

  Parser.prototype.rule = function rule(tokens) {
    tokens.pop();

    var node = new _rule2.default();
    this.init(node, tokens[0][2], tokens[0][3]);

    node.raws.between = this.spacesAndCommentsFromEnd(tokens);
    this.raw(node, 'selector', tokens);
    this.current = node;
  };

  Parser.prototype.decl = function decl(tokens) {
    var node = new _declaration2.default();
    this.init(node);

    var last = tokens[tokens.length - 1];
    if (last[0] === ';') {
      this.semicolon = true;
      tokens.pop();
    }
    if (last[4]) {
      node.source.end = { line: last[4], column: last[5] };
    } else {
      node.source.end = { line: last[2], column: last[3] };
    }

    while (tokens[0][0] !== 'word') {
      if (tokens.length === 1) this.unknownWord(tokens);
      node.raws.before += tokens.shift()[1];
    }
    node.source.start = { line: tokens[0][2], column: tokens[0][3] };

    node.prop = '';
    while (tokens.length) {
      var type = tokens[0][0];
      if (type === ':' || type === 'space' || type === 'comment') {
        break;
      }
      node.prop += tokens.shift()[1];
    }

    node.raws.between = '';

    var token = void 0;
    while (tokens.length) {
      token = tokens.shift();

      if (token[0] === ':') {
        node.raws.between += token[1];
        break;
      } else {
        node.raws.between += token[1];
      }
    }

    if (node.prop[0] === '_' || node.prop[0] === '*') {
      node.raws.before += node.prop[0];
      node.prop = node.prop.slice(1);
    }
    node.raws.between += this.spacesAndCommentsFromStart(tokens);
    this.precheckMissedSemicolon(tokens);

    for (var i = tokens.length - 1; i > 0; i--) {
      token = tokens[i];
      if (token[1].toLowerCase() === '!important') {
        node.important = true;
        var string = this.stringFrom(tokens, i);
        string = this.spacesFromEnd(tokens) + string;
        if (string !== ' !important') node.raws.important = string;
        break;
      } else if (token[1].toLowerCase() === 'important') {
        var cache = tokens.slice(0);
        var str = '';
        for (var j = i; j > 0; j--) {
          var _type = cache[j][0];
          if (str.trim().indexOf('!') === 0 && _type !== 'space') {
            break;
          }
          str = cache.pop()[1] + str;
        }
        if (str.trim().indexOf('!') === 0) {
          node.important = true;
          node.raws.important = str;
          tokens = cache;
        }
      }

      if (token[0] !== 'space' && token[0] !== 'comment') {
        break;
      }
    }

    this.raw(node, 'value', tokens);

    if (node.value.indexOf(':') !== -1) this.checkMissedSemicolon(tokens);
  };

  Parser.prototype.atrule = function atrule(token) {
    var node = new _atRule2.default();
    node.name = token[1].slice(1);
    if (node.name === '') {
      this.unnamedAtrule(node, token);
    }
    this.init(node, token[2], token[3]);

    var prev = void 0;
    var shift = void 0;
    var last = false;
    var open = false;
    var params = [];

    while (!this.tokenizer.endOfFile()) {
      token = this.tokenizer.nextToken();

      if (token[0] === ';') {
        node.source.end = { line: token[2], column: token[3] };
        this.semicolon = true;
        break;
      } else if (token[0] === '{') {
        open = true;
        break;
      } else if (token[0] === '}') {
        if (params.length > 0) {
          shift = params.length - 1;
          prev = params[shift];
          while (prev && prev[0] === 'space') {
            prev = params[--shift];
          }
          if (prev) {
            node.source.end = { line: prev[4], column: prev[5] };
          }
        }
        this.end(token);
        break;
      } else {
        params.push(token);
      }

      if (this.tokenizer.endOfFile()) {
        last = true;
        break;
      }
    }

    node.raws.between = this.spacesAndCommentsFromEnd(params);
    if (params.length) {
      node.raws.afterName = this.spacesAndCommentsFromStart(params);
      this.raw(node, 'params', params);
      if (last) {
        token = params[params.length - 1];
        node.source.end = { line: token[4], column: token[5] };
        this.spaces = node.raws.between;
        node.raws.between = '';
      }
    } else {
      node.raws.afterName = '';
      node.params = '';
    }

    if (open) {
      node.nodes = [];
      this.current = node;
    }
  };

  Parser.prototype.end = function end(token) {
    if (this.current.nodes && this.current.nodes.length) {
      this.current.raws.semicolon = this.semicolon;
    }
    this.semicolon = false;

    this.current.raws.after = (this.current.raws.after || '') + this.spaces;
    this.spaces = '';

    if (this.current.parent) {
      this.current.source.end = { line: token[2], column: token[3] };
      this.current = this.current.parent;
    } else {
      this.unexpectedClose(token);
    }
  };

  Parser.prototype.endFile = function endFile() {
    if (this.current.parent) this.unclosedBlock();
    if (this.current.nodes && this.current.nodes.length) {
      this.current.raws.semicolon = this.semicolon;
    }
    this.current.raws.after = (this.current.raws.after || '') + this.spaces;
  };

  Parser.prototype.freeSemicolon = function freeSemicolon(token) {
    this.spaces += token[1];
    if (this.current.nodes) {
      var prev = this.current.nodes[this.current.nodes.length - 1];
      if (prev && prev.type === 'rule' && !prev.raws.ownSemicolon) {
        prev.raws.ownSemicolon = this.spaces;
        this.spaces = '';
      }
    }
  };

  // Helpers

  Parser.prototype.init = function init(node, line, column) {
    this.current.push(node);

    node.source = { start: { line: line, column: column }, input: this.input };
    node.raws.before = this.spaces;
    this.spaces = '';
    if (node.type !== 'comment') this.semicolon = false;
  };

  Parser.prototype.raw = function raw(node, prop, tokens) {
    var token = void 0,
        type = void 0;
    var length = tokens.length;
    var value = '';
    var clean = true;
    var next = void 0,
        prev = void 0;
    var pattern = /^([.|#])?([\w])+/i;

    for (var i = 0; i < length; i += 1) {
      token = tokens[i];
      type = token[0];

      if (type === 'comment' && node.type === 'rule') {
        prev = tokens[i - 1];
        next = tokens[i + 1];

        if (prev[0] !== 'space' && next[0] !== 'space' && pattern.test(prev[1]) && pattern.test(next[1])) {
          value += token[1];
        } else {
          clean = false;
        }

        continue;
      }

      if (type === 'comment' || type === 'space' && i === length - 1) {
        clean = false;
      } else {
        value += token[1];
      }
    }
    if (!clean) {
      var raw = tokens.reduce(function (all, i) {
        return all + i[1];
      }, '');
      node.raws[prop] = { value: value, raw: raw };
    }
    node[prop] = value;
  };

  Parser.prototype.spacesAndCommentsFromEnd = function spacesAndCommentsFromEnd(tokens) {
    var lastTokenType = void 0;
    var spaces = '';
    while (tokens.length) {
      lastTokenType = tokens[tokens.length - 1][0];
      if (lastTokenType !== 'space' && lastTokenType !== 'comment') break;
      spaces = tokens.pop()[1] + spaces;
    }
    return spaces;
  };

  Parser.prototype.spacesAndCommentsFromStart = function spacesAndCommentsFromStart(tokens) {
    var next = void 0;
    var spaces = '';
    while (tokens.length) {
      next = tokens[0][0];
      if (next !== 'space' && next !== 'comment') break;
      spaces += tokens.shift()[1];
    }
    return spaces;
  };

  Parser.prototype.spacesFromEnd = function spacesFromEnd(tokens) {
    var lastTokenType = void 0;
    var spaces = '';
    while (tokens.length) {
      lastTokenType = tokens[tokens.length - 1][0];
      if (lastTokenType !== 'space') break;
      spaces = tokens.pop()[1] + spaces;
    }
    return spaces;
  };

  Parser.prototype.stringFrom = function stringFrom(tokens, from) {
    var result = '';
    for (var i = from; i < tokens.length; i++) {
      result += tokens[i][1];
    }
    tokens.splice(from, tokens.length - from);
    return result;
  };

  Parser.prototype.colon = function colon(tokens) {
    var brackets = 0;
    var token = void 0,
        type = void 0,
        prev = void 0;
    for (var i = 0; i < tokens.length; i++) {
      token = tokens[i];
      type = token[0];

      if (type === '(') {
        brackets += 1;
      } else if (type === ')') {
        brackets -= 1;
      } else if (brackets === 0 && type === ':') {
        if (!prev) {
          this.doubleColon(token);
        } else if (prev[0] === 'word' && prev[1] === 'progid') {
          continue;
        } else {
          return i;
        }
      }

      prev = token;
    }
    return false;
  };

  // Errors

  Parser.prototype.unclosedBracket = function unclosedBracket(bracket) {
    throw this.input.error('Unclosed bracket', bracket[2], bracket[3]);
  };

  Parser.prototype.unknownWord = function unknownWord(tokens) {
    throw this.input.error('Unknown word', tokens[0][2], tokens[0][3]);
  };

  Parser.prototype.unexpectedClose = function unexpectedClose(token) {
    throw this.input.error('Unexpected }', token[2], token[3]);
  };

  Parser.prototype.unclosedBlock = function unclosedBlock() {
    var pos = this.current.source.start;
    throw this.input.error('Unclosed block', pos.line, pos.column);
  };

  Parser.prototype.doubleColon = function doubleColon(token) {
    throw this.input.error('Double colon', token[2], token[3]);
  };

  Parser.prototype.unnamedAtrule = function unnamedAtrule(node, token) {
    throw this.input.error('At-rule without name', token[2], token[3]);
  };

  Parser.prototype.precheckMissedSemicolon = function precheckMissedSemicolon() /* tokens */{
    // Hook for Safe Parser
  };

  Parser.prototype.checkMissedSemicolon = function checkMissedSemicolon(tokens) {
    var colon = this.colon(tokens);
    if (colon === false) return;

    var founded = 0;
    var token = void 0;
    for (var j = colon - 1; j >= 0; j--) {
      token = tokens[j];
      if (token[0] !== 'space') {
        founded += 1;
        if (founded === 2) break;
      }
    }
    throw this.input.error('Missed semicolon', token[2], token[3]);
  };

  return Parser;
}();

exports.default = Parser;
module.exports = exports['default'];
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
