'use strict';

exports.__esModule = true;

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

var DEFAULT_RAW = {
  colon: ': ',
  indent: '    ',
  beforeDecl: '\n',
  beforeRule: '\n',
  beforeOpen: ' ',
  beforeClose: '\n',
  beforeComment: '\n',
  after: '\n',
  emptyBody: '',
  commentLeft: ' ',
  commentRight: ' '
};

function capitalize(str) {
  return str[0].toUpperCase() + str.slice(1);
}

var Stringifier = function () {
  function Stringifier(builder) {
    _classCallCheck(this, Stringifier);

    this.builder = builder;
  }

  Stringifier.prototype.stringify = function stringify(node, semicolon) {
    this[node.type](node, semicolon);
  };

  Stringifier.prototype.root = function root(node) {
    this.body(node);
    if (node.raws.after) this.builder(node.raws.after);
  };

  Stringifier.prototype.comment = function comment(node) {
    var left = this.raw(node, 'left', 'commentLeft');
    var right = this.raw(node, 'right', 'commentRight');
    this.builder('/*' + left + node.text + right + '*/', node);
  };

  Stringifier.prototype.decl = function decl(node, semicolon) {
    var between = this.raw(node, 'between', 'colon');
    var string = node.prop + between + this.rawValue(node, 'value');

    if (node.important) {
      string += node.raws.important || ' !important';
    }

    if (semicolon) string += ';';
    this.builder(string, node);
  };

  Stringifier.prototype.rule = function rule(node) {
    this.block(node, this.rawValue(node, 'selector'));
    if (node.raws.ownSemicolon) {
      this.builder(node.raws.ownSemicolon, node, 'end');
    }
  };

  Stringifier.prototype.atrule = function atrule(node, semicolon) {
    var name = '@' + node.name;
    var params = node.params ? this.rawValue(node, 'params') : '';

    if (typeof node.raws.afterName !== 'undefined') {
      name += node.raws.afterName;
    } else if (params) {
      name += ' ';
    }

    if (node.nodes) {
      this.block(node, name + params);
    } else {
      var end = (node.raws.between || '') + (semicolon ? ';' : '');
      this.builder(name + params + end, node);
    }
  };

  Stringifier.prototype.body = function body(node) {
    var last = node.nodes.length - 1;
    while (last > 0) {
      if (node.nodes[last].type !== 'comment') break;
      last -= 1;
    }

    var semicolon = this.raw(node, 'semicolon');
    for (var i = 0; i < node.nodes.length; i++) {
      var child = node.nodes[i];
      var before = this.raw(child, 'before');
      if (before) this.builder(before);
      this.stringify(child, last !== i || semicolon);
    }
  };

  Stringifier.prototype.block = function block(node, start) {
    var between = this.raw(node, 'between', 'beforeOpen');
    this.builder(start + between + '{', node, 'start');

    var after = void 0;
    if (node.nodes && node.nodes.length) {
      this.body(node);
      after = this.raw(node, 'after');
    } else {
      after = this.raw(node, 'after', 'emptyBody');
    }

    if (after) this.builder(after);
    this.builder('}', node, 'end');
  };

  Stringifier.prototype.raw = function raw(node, own, detect) {
    var value = void 0;
    if (!detect) detect = own;

    // Already had
    if (own) {
      value = node.raws[own];
      if (typeof value !== 'undefined') return value;
    }

    var parent = node.parent;

    // Hack for first rule in CSS
    if (detect === 'before') {
      if (!parent || parent.type === 'root' && parent.first === node) {
        return '';
      }
    }

    // Floating child without parent
    if (!parent) return DEFAULT_RAW[detect];

    // Detect style by other nodes
    var root = node.root();
    if (!root.rawCache) root.rawCache = {};
    if (typeof root.rawCache[detect] !== 'undefined') {
      return root.rawCache[detect];
    }

    if (detect === 'before' || detect === 'after') {
      return this.beforeAfter(node, detect);
    } else {
      var method = 'raw' + capitalize(detect);
      if (this[method]) {
        value = this[method](root, node);
      } else {
        root.walk(function (i) {
          value = i.raws[own];
          if (typeof value !== 'undefined') return false;
        });
      }
    }

    if (typeof value === 'undefined') value = DEFAULT_RAW[detect];

    root.rawCache[detect] = value;
    return value;
  };

  Stringifier.prototype.rawSemicolon = function rawSemicolon(root) {
    var value = void 0;
    root.walk(function (i) {
      if (i.nodes && i.nodes.length && i.last.type === 'decl') {
        value = i.raws.semicolon;
        if (typeof value !== 'undefined') return false;
      }
    });
    return value;
  };

  Stringifier.prototype.rawEmptyBody = function rawEmptyBody(root) {
    var value = void 0;
    root.walk(function (i) {
      if (i.nodes && i.nodes.length === 0) {
        value = i.raws.after;
        if (typeof value !== 'undefined') return false;
      }
    });
    return value;
  };

  Stringifier.prototype.rawIndent = function rawIndent(root) {
    if (root.raws.indent) return root.raws.indent;
    var value = void 0;
    root.walk(function (i) {
      var p = i.parent;
      if (p && p !== root && p.parent && p.parent === root) {
        if (typeof i.raws.before !== 'undefined') {
          var parts = i.raws.before.split('\n');
          value = parts[parts.length - 1];
          value = value.replace(/[^\s]/g, '');
          return false;
        }
      }
    });
    return value;
  };

  Stringifier.prototype.rawBeforeComment = function rawBeforeComment(root, node) {
    var value = void 0;
    root.walkComments(function (i) {
      if (typeof i.raws.before !== 'undefined') {
        value = i.raws.before;
        if (value.indexOf('\n') !== -1) {
          value = value.replace(/[^\n]+$/, '');
        }
        return false;
      }
    });
    if (typeof value === 'undefined') {
      value = this.raw(node, null, 'beforeDecl');
    } else if (value) {
      value = value.replace(/[^\s]/g, '');
    }
    return value;
  };

  Stringifier.prototype.rawBeforeDecl = function rawBeforeDecl(root, node) {
    var value = void 0;
    root.walkDecls(function (i) {
      if (typeof i.raws.before !== 'undefined') {
        value = i.raws.before;
        if (value.indexOf('\n') !== -1) {
          value = value.replace(/[^\n]+$/, '');
        }
        return false;
      }
    });
    if (typeof value === 'undefined') {
      value = this.raw(node, null, 'beforeRule');
    } else if (value) {
      value = value.replace(/[^\s]/g, '');
    }
    return value;
  };

  Stringifier.prototype.rawBeforeRule = function rawBeforeRule(root) {
    var value = void 0;
    root.walk(function (i) {
      if (i.nodes && (i.parent !== root || root.first !== i)) {
        if (typeof i.raws.before !== 'undefined') {
          value = i.raws.before;
          if (value.indexOf('\n') !== -1) {
            value = value.replace(/[^\n]+$/, '');
          }
          return false;
        }
      }
    });
    if (value) value = value.replace(/[^\s]/g, '');
    return value;
  };

  Stringifier.prototype.rawBeforeClose = function rawBeforeClose(root) {
    var value = void 0;
    root.walk(function (i) {
      if (i.nodes && i.nodes.length > 0) {
        if (typeof i.raws.after !== 'undefined') {
          value = i.raws.after;
          if (value.indexOf('\n') !== -1) {
            value = value.replace(/[^\n]+$/, '');
          }
          return false;
        }
      }
    });
    if (value) value = value.replace(/[^\s]/g, '');
    return value;
  };

  Stringifier.prototype.rawBeforeOpen = function rawBeforeOpen(root) {
    var value = void 0;
    root.walk(function (i) {
      if (i.type !== 'decl') {
        value = i.raws.between;
        if (typeof value !== 'undefined') return false;
      }
    });
    return value;
  };

  Stringifier.prototype.rawColon = function rawColon(root) {
    var value = void 0;
    root.walkDecls(function (i) {
      if (typeof i.raws.between !== 'undefined') {
        value = i.raws.between.replace(/[^\s:]/g, '');
        return false;
      }
    });
    return value;
  };

  Stringifier.prototype.beforeAfter = function beforeAfter(node, detect) {
    var value = void 0;
    if (node.type === 'decl') {
      value = this.raw(node, null, 'beforeDecl');
    } else if (node.type === 'comment') {
      value = this.raw(node, null, 'beforeComment');
    } else if (detect === 'before') {
      value = this.raw(node, null, 'beforeRule');
    } else {
      value = this.raw(node, null, 'beforeClose');
    }

    var buf = node.parent;
    var depth = 0;
    while (buf && buf.type !== 'root') {
      depth += 1;
      buf = buf.parent;
    }

    if (value.indexOf('\n') !== -1) {
      var indent = this.raw(node, null, 'indent');
      if (indent.length) {
        for (var step = 0; step < depth; step++) {
          value += indent;
        }
      }
    }

    return value;
  };

  Stringifier.prototype.rawValue = function rawValue(node, prop) {
    var value = node[prop];
    var raw = node.raws[prop];
    if (raw && raw.value === value) {
      return raw.raw;
    } else {
      return value;
    }
  };

  return Stringifier;
}();

exports.default = Stringifier;
module.exports = exports['default'];
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
