'use strict';

exports.__esModule = true;
exports.default = tokenizer;
var SINGLE_QUOTE = 39;
var DOUBLE_QUOTE = 34;
var BACKSLASH = 92;
var SLASH = 47;
var NEWLINE = 10;
var SPACE = 32;
var FEED = 12;
var TAB = 9;
var CR = 13;
var OPEN_SQUARE = 91;
var CLOSE_SQUARE = 93;
var OPEN_PARENTHESES = 40;
var CLOSE_PARENTHESES = 41;
var OPEN_CURLY = 123;
var CLOSE_CURLY = 125;
var SEMICOLON = 59;
var ASTERISK = 42;
var COLON = 58;
var AT = 64;

var RE_AT_END = /[ \n\t\r\f{}()'"\\;/[\]#]/g;
var RE_WORD_END = /[ \n\t\r\f(){}:;@!'"\\\][#]|\/(?=\*)/g;
var RE_BAD_BRACKET = /.[\\/("'\n]/;
var RE_HEX_ESCAPE = /[a-f0-9]/i;

function tokenizer(input) {
  var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

  var css = input.css.valueOf();
  var ignore = options.ignoreErrors;

  var code = void 0,
      next = void 0,
      quote = void 0,
      lines = void 0,
      last = void 0,
      content = void 0,
      escape = void 0;
  var nextLine = void 0,
      nextOffset = void 0,
      escaped = void 0,
      escapePos = void 0,
      prev = void 0,
      n = void 0,
      currentToken = void 0;

  var length = css.length;
  var offset = -1;
  var line = 1;
  var pos = 0;
  var buffer = [];
  var returned = [];

  function unclosed(what) {
    throw input.error('Unclosed ' + what, line, pos - offset);
  }

  function endOfFile() {
    return returned.length === 0 && pos >= length;
  }

  function nextToken() {
    if (returned.length) return returned.pop();
    if (pos >= length) return;

    code = css.charCodeAt(pos);
    if (code === NEWLINE || code === FEED || code === CR && css.charCodeAt(pos + 1) !== NEWLINE) {
      offset = pos;
      line += 1;
    }

    switch (code) {
      case NEWLINE:
      case SPACE:
      case TAB:
      case CR:
      case FEED:
        next = pos;
        do {
          next += 1;
          code = css.charCodeAt(next);
          if (code === NEWLINE) {
            offset = next;
            line += 1;
          }
        } while (code === SPACE || code === NEWLINE || code === TAB || code === CR || code === FEED);

        currentToken = ['space', css.slice(pos, next)];
        pos = next - 1;
        break;

      case OPEN_SQUARE:
        currentToken = ['[', '[', line, pos - offset];
        break;

      case CLOSE_SQUARE:
        currentToken = [']', ']', line, pos - offset];
        break;

      case OPEN_CURLY:
        currentToken = ['{', '{', line, pos - offset];
        break;

      case CLOSE_CURLY:
        currentToken = ['}', '}', line, pos - offset];
        break;

      case COLON:
        currentToken = [':', ':', line, pos - offset];
        break;

      case SEMICOLON:
        currentToken = [';', ';', line, pos - offset];
        break;

      case OPEN_PARENTHESES:
        prev = buffer.length ? buffer.pop()[1] : '';
        n = css.charCodeAt(pos + 1);
        if (prev === 'url' && n !== SINGLE_QUOTE && n !== DOUBLE_QUOTE && n !== SPACE && n !== NEWLINE && n !== TAB && n !== FEED && n !== CR) {
          next = pos;
          do {
            escaped = false;
            next = css.indexOf(')', next + 1);
            if (next === -1) {
              if (ignore) {
                next = pos;
                break;
              } else {
                unclosed('bracket');
              }
            }
            escapePos = next;
            while (css.charCodeAt(escapePos - 1) === BACKSLASH) {
              escapePos -= 1;
              escaped = !escaped;
            }
          } while (escaped);

          currentToken = ['brackets', css.slice(pos, next + 1), line, pos - offset, line, next - offset];

          pos = next;
        } else {
          next = css.indexOf(')', pos + 1);
          content = css.slice(pos, next + 1);

          if (next === -1 || RE_BAD_BRACKET.test(content)) {
            currentToken = ['(', '(', line, pos - offset];
          } else {
            currentToken = ['brackets', content, line, pos - offset, line, next - offset];
            pos = next;
          }
        }

        break;

      case CLOSE_PARENTHESES:
        currentToken = [')', ')', line, pos - offset];
        break;

      case SINGLE_QUOTE:
      case DOUBLE_QUOTE:
        quote = code === SINGLE_QUOTE ? '\'' : '"';
        next = pos;
        do {
          escaped = false;
          next = css.indexOf(quote, next + 1);
          if (next === -1) {
            if (ignore) {
              next = pos + 1;
              break;
            } else {
              unclosed('string');
            }
          }
          escapePos = next;
          while (css.charCodeAt(escapePos - 1) === BACKSLASH) {
            escapePos -= 1;
            escaped = !escaped;
          }
        } while (escaped);

        content = css.slice(pos, next + 1);
        lines = content.split('\n');
        last = lines.length - 1;

        if (last > 0) {
          nextLine = line + last;
          nextOffset = next - lines[last].length;
        } else {
          nextLine = line;
          nextOffset = offset;
        }

        currentToken = ['string', css.slice(pos, next + 1), line, pos - offset, nextLine, next - nextOffset];

        offset = nextOffset;
        line = nextLine;
        pos = next;
        break;

      case AT:
        RE_AT_END.lastIndex = pos + 1;
        RE_AT_END.test(css);
        if (RE_AT_END.lastIndex === 0) {
          next = css.length - 1;
        } else {
          next = RE_AT_END.lastIndex - 2;
        }

        currentToken = ['at-word', css.slice(pos, next + 1), line, pos - offset, line, next - offset];

        pos = next;
        break;

      case BACKSLASH:
        next = pos;
        escape = true;
        while (css.charCodeAt(next + 1) === BACKSLASH) {
          next += 1;
          escape = !escape;
        }
        code = css.charCodeAt(next + 1);
        if (escape && code !== SLASH && code !== SPACE && code !== NEWLINE && code !== TAB && code !== CR && code !== FEED) {
          next += 1;
          if (RE_HEX_ESCAPE.test(css.charAt(next))) {
            while (RE_HEX_ESCAPE.test(css.charAt(next + 1))) {
              next += 1;
            }
            if (css.charCodeAt(next + 1) === SPACE) {
              next += 1;
            }
          }
        }

        currentToken = ['word', css.slice(pos, next + 1), line, pos - offset, line, next - offset];

        pos = next;
        break;

      default:
        if (code === SLASH && css.charCodeAt(pos + 1) === ASTERISK) {
          next = css.indexOf('*/', pos + 2) + 1;
          if (next === 0) {
            if (ignore) {
              next = css.length;
            } else {
              unclosed('comment');
            }
          }

          content = css.slice(pos, next + 1);
          lines = content.split('\n');
          last = lines.length - 1;

          if (last > 0) {
            nextLine = line + last;
            nextOffset = next - lines[last].length;
          } else {
            nextLine = line;
            nextOffset = offset;
          }

          currentToken = ['comment', content, line, pos - offset, nextLine, next - nextOffset];

          offset = nextOffset;
          line = nextLine;
          pos = next;
        } else {
          RE_WORD_END.lastIndex = pos + 1;
          RE_WORD_END.test(css);
          if (RE_WORD_END.lastIndex === 0) {
            next = css.length - 1;
          } else {
            next = RE_WORD_END.lastIndex - 2;
          }

          currentToken = ['word', css.slice(pos, next + 1), line, pos - offset, line, next - offset];

          buffer.push(currentToken);

          pos = next;
        }

        break;
    }

    pos++;
    return currentToken;
  }

  function back(token) {
    returned.push(token);
  }

  return {
    back: back,
    nextToken: nextToken,
    endOfFile: endOfFile
  };
}
module.exports = exports['default'];
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
