"use strict";

exports.__esModule = true;
exports.default = void 0;
var DEFAULT_RAW = {
  colon: ': ',
  indent: '    ',
  beforeDecl: '\n',
  beforeRule: '\n',
  beforeOpen: ' ',
  beforeClose: '\n',
  beforeComment: '\n',
  after: '\n',
  emptyBody: '',
  commentLeft: ' ',
  commentRight: ' '
};

function capitalize(str) {
  return str[0].toUpperCase() + str.slice(1);
}

var Stringifier =
/*#__PURE__*/
function () {
  function Stringifier(builder) {
    this.builder = builder;
  }

  var _proto = Stringifier.prototype;

  _proto.stringify = function stringify(node, semicolon) {
    this[node.type](node, semicolon);
  };

  _proto.root = function root(node) {
    this.body(node);
    if (node.raws.after) this.builder(node.raws.after);
  };

  _proto.comment = function comment(node) {
    var left = this.raw(node, 'left', 'commentLeft');
    var right = this.raw(node, 'right', 'commentRight');
    this.builder('/*' + left + node.text + right + '*/', node);
  };

  _proto.decl = function decl(node, semicolon) {
    var between = this.raw(node, 'between', 'colon');
    var string = node.prop + between + this.rawValue(node, 'value');

    if (node.important) {
      string += node.raws.important || ' !important';
    }

    if (semicolon) string += ';';
    this.builder(string, node);
  };

  _proto.rule = function rule(node) {
    this.block(node, this.rawValue(node, 'selector'));

    if (node.raws.ownSemicolon) {
      this.builder(node.raws.ownSemicolon, node, 'end');
    }
  };

  _proto.atrule = function atrule(node, semicolon) {
    var name = '@' + node.name;
    var params = node.params ? this.rawValue(node, 'params') : '';

    if (typeof node.raws.afterName !== 'undefined') {
      name += node.raws.afterName;
    } else if (params) {
      name += ' ';
    }

    if (node.nodes) {
      this.block(node, name + params);
    } else {
      var end = (node.raws.between || '') + (semicolon ? ';' : '');
      this.builder(name + params + end, node);
    }
  };

  _proto.body = function body(node) {
    var last = node.nodes.length - 1;

    while (last > 0) {
      if (node.nodes[last].type !== 'comment') break;
      last -= 1;
    }

    var semicolon = this.raw(node, 'semicolon');

    for (var i = 0; i < node.nodes.length; i++) {
      var child = node.nodes[i];
      var before = this.raw(child, 'before');
      if (before) this.builder(before);
      this.stringify(child, last !== i || semicolon);
    }
  };

  _proto.block = function block(node, start) {
    var between = this.raw(node, 'between', 'beforeOpen');
    this.builder(start + between + '{', node, 'start');
    var after;

    if (node.nodes && node.nodes.length) {
      this.body(node);
      after = this.raw(node, 'after');
    } else {
      after = this.raw(node, 'after', 'emptyBody');
    }

    if (after) this.builder(after);
    this.builder('}', node, 'end');
  };

  _proto.raw = function raw(node, own, detect) {
    var value;
    if (!detect) detect = own; // Already had

    if (own) {
      value = node.raws[own];
      if (typeof value !== 'undefined') return value;
    }

    var parent = node.parent; // Hack for first rule in CSS

    if (detect === 'before') {
      if (!parent || parent.type === 'root' && parent.first === node) {
        return '';
      }
    } // Floating child without parent


    if (!parent) return DEFAULT_RAW[detect]; // Detect style by other nodes

    var root = node.root();
    if (!root.rawCache) root.rawCache = {};

    if (typeof root.rawCache[detect] !== 'undefined') {
      return root.rawCache[detect];
    }

    if (detect === 'before' || detect === 'after') {
      return this.beforeAfter(node, detect);
    } else {
      var method = 'raw' + capitalize(detect);

      if (this[method]) {
        value = this[method](root, node);
      } else {
        root.walk(function (i) {
          value = i.raws[own];
          if (typeof value !== 'undefined') return false;
        });
      }
    }

    if (typeof value === 'undefined') value = DEFAULT_RAW[detect];
    root.rawCache[detect] = value;
    return value;
  };

  _proto.rawSemicolon = function rawSemicolon(root) {
    var value;
    root.walk(function (i) {
      if (i.nodes && i.nodes.length && i.last.type === 'decl') {
        value = i.raws.semicolon;
        if (typeof value !== 'undefined') return false;
      }
    });
    return value;
  };

  _proto.rawEmptyBody = function rawEmptyBody(root) {
    var value;
    root.walk(function (i) {
      if (i.nodes && i.nodes.length === 0) {
        value = i.raws.after;
        if (typeof value !== 'undefined') return false;
      }
    });
    return value;
  };

  _proto.rawIndent = function rawIndent(root) {
    if (root.raws.indent) return root.raws.indent;
    var value;
    root.walk(function (i) {
      var p = i.parent;

      if (p && p !== root && p.parent && p.parent === root) {
        if (typeof i.raws.before !== 'undefined') {
          var parts = i.raws.before.split('\n');
          value = parts[parts.length - 1];
          value = value.replace(/[^\s]/g, '');
          return false;
        }
      }
    });
    return value;
  };

  _proto.rawBeforeComment = function rawBeforeComment(root, node) {
    var value;
    root.walkComments(function (i) {
      if (typeof i.raws.before !== 'undefined') {
        value = i.raws.before;

        if (value.indexOf('\n') !== -1) {
          value = value.replace(/[^\n]+$/, '');
        }

        return false;
      }
    });

    if (typeof value === 'undefined') {
      value = this.raw(node, null, 'beforeDecl');
    } else if (value) {
      value = value.replace(/[^\s]/g, '');
    }

    return value;
  };

  _proto.rawBeforeDecl = function rawBeforeDecl(root, node) {
    var value;
    root.walkDecls(function (i) {
      if (typeof i.raws.before !== 'undefined') {
        value = i.raws.before;

        if (value.indexOf('\n') !== -1) {
          value = value.replace(/[^\n]+$/, '');
        }

        return false;
      }
    });

    if (typeof value === 'undefined') {
      value = this.raw(node, null, 'beforeRule');
    } else if (value) {
      value = value.replace(/[^\s]/g, '');
    }

    return value;
  };

  _proto.rawBeforeRule = function rawBeforeRule(root) {
    var value;
    root.walk(function (i) {
      if (i.nodes && (i.parent !== root || root.first !== i)) {
        if (typeof i.raws.before !== 'undefined') {
          value = i.raws.before;

          if (value.indexOf('\n') !== -1) {
            value = value.replace(/[^\n]+$/, '');
          }

          return false;
        }
      }
    });
    if (value) value = value.replace(/[^\s]/g, '');
    return value;
  };

  _proto.rawBeforeClose = function rawBeforeClose(root) {
    var value;
    root.walk(function (i) {
      if (i.nodes && i.nodes.length > 0) {
        if (typeof i.raws.after !== 'undefined') {
          value = i.raws.after;

          if (value.indexOf('\n') !== -1) {
            value = value.replace(/[^\n]+$/, '');
          }

          return false;
        }
      }
    });
    if (value) value = value.replace(/[^\s]/g, '');
    return value;
  };

  _proto.rawBeforeOpen = function rawBeforeOpen(root) {
    var value;
    root.walk(function (i) {
      if (i.type !== 'decl') {
        value = i.raws.between;
        if (typeof value !== 'undefined') return false;
      }
    });
    return value;
  };

  _proto.rawColon = function rawColon(root) {
    var value;
    root.walkDecls(function (i) {
      if (typeof i.raws.between !== 'undefined') {
        value = i.raws.between.replace(/[^\s:]/g, '');
        return false;
      }
    });
    return value;
  };

  _proto.beforeAfter = function beforeAfter(node, detect) {
    var value;

    if (node.type === 'decl') {
      value = this.raw(node, null, 'beforeDecl');
    } else if (node.type === 'comment') {
      value = this.raw(node, null, 'beforeComment');
    } else if (detect === 'before') {
      value = this.raw(node, null, 'beforeRule');
    } else {
      value = this.raw(node, null, 'beforeClose');
    }

    var buf = node.parent;
    var depth = 0;

    while (buf && buf.type !== 'root') {
      depth += 1;
      buf = buf.parent;
    }

    if (value.indexOf('\n') !== -1) {
      var indent = this.raw(node, null, 'indent');

      if (indent.length) {
        for (var step = 0; step < depth; step++) {
          value += indent;
        }
      }
    }

    return value;
  };

  _proto.rawValue = function rawValue(node, prop) {
    var value = node[prop];
    var raw = node.raws[prop];

    if (raw && raw.value === value) {
      return raw.raw;
    }

    return value;
  };

  return Stringifier;
}();

var _default = Stringifier;
exports.default = _default;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
