'use strict';

var utils = require('../../utils');
var idbConstants = require('./idb-constants');
var DETECT_BLOB_SUPPORT_STORE = idbConstants.DETECT_BLOB_SUPPORT_STORE;

//
// Detect blob support. Chrome didn't support it until version 38.
// In version 37 they had a broken version where PNGs (and possibly
// other binary types) aren't stored correctly, because when you fetch
// them, the content type is always null.
//
// Furthermore, they have some outstanding bugs where blobs occasionally
// are read by FileReader as null, or by ajax as 404s.
//
// Sadly we use the 404 bug to detect the FileReader bug, so if they
// get fixed independently and released in different versions of Chrome,
// then the bug could come back. So it's worthwhile to watch these issues:
// 404 bug: https://code.google.com/p/chromium/issues/detail?id=447916
// FileReader bug: https://code.google.com/p/chromium/issues/detail?id=447836
//
function checkBlobSupport(txn, idb) {
  return new utils.Promise(function (resolve, reject) {
    var blob = utils.createBlob([''], {type: 'image/png'});
    txn.objectStore(DETECT_BLOB_SUPPORT_STORE).put(blob, 'key');
    txn.oncomplete = function () {
      // have to do it in a separate transaction, else the correct
      // content type is always returned
      var blobTxn = idb.transaction([DETECT_BLOB_SUPPORT_STORE],
        'readwrite');
      var getBlobReq = blobTxn.objectStore(
        DETECT_BLOB_SUPPORT_STORE).get('key');
      getBlobReq.onerror = reject;
      getBlobReq.onsuccess = function (e) {

        var storedBlob = e.target.result;
        var url = URL.createObjectURL(storedBlob);

        utils.ajax({
          url: url,
          cache: true,
          binary: true
        }, function (err, res) {
          if (err && err.status === 405) {
            // firefox won't let us do that. but firefox doesn't
            // have the blob type bug that Chrome does, so that's ok
            resolve(true);
          } else {
            resolve(!!(res && res.type === 'image/png'));
            if (err && err.status === 404) {
              utils.explain404('PouchDB is just detecting blob URL support.');
            }
          }
          URL.revokeObjectURL(url);
        });
      };
    };
  }).catch(function () {
    return false; // error, so assume unsupported
  });
}

module.exports = checkBlobSupport;