import forEach from "lodash/forEach.js";
import { builders, utils } from "prettier/doc";
import { BaseCstPrettierPrinter } from "../base-cst-printer.js";
import { isAnnotationCstNode } from "../types/utils.js";
import { printArgumentListWithBraces } from "../utils/index.js";
import { printTokenWithComments } from "./comments/format-comments.js";
import { handleCommentsBinaryExpression, handleCommentsParameters } from "./comments/handle-comments.js";
import { concat, dedent, group, indent } from "./prettier-builder.js";
import { binary, findDeepElementInPartsArray, isExplicitLambdaParameter, isUniqueMethodInvocation, putIntoBraces, rejectAndConcat, rejectAndJoin, rejectAndJoinSeps, sortAnnotationIdentifier, sortNodes, sortTokens } from "./printer-utils.js";
const { breakParent, conditionalGroup, ifBreak, label, line, lineSuffixBoundary, softline } = builders;
const { removeLines, willBreak } = utils;
export class ExpressionsPrettierVisitor extends BaseCstPrettierPrinter {
    expression(ctx, params) {
        const expression = this.visitSingle(ctx, params);
        return (params === null || params === void 0 ? void 0 : params.hug) && expression.label !== undefined
            ? label(expression.label, expression)
            : expression;
    }
    lambdaExpression(ctx, params) {
        const lambdaParameters = group(this.visit(ctx.lambdaParameters));
        const lambdaBody = this.visit(ctx.lambdaBody);
        const isLambdaBodyABlock = ctx.lambdaBody[0].children.block !== undefined;
        const suffix = [
            " ",
            ctx.Arrow[0],
            ...(isLambdaBodyABlock
                ? [" ", lambdaBody]
                : [group(indent([line, lambdaBody]))])
        ];
        if (params === null || params === void 0 ? void 0 : params.hug) {
            return willBreak(lambdaParameters)
                ? label({ huggable: false }, concat([lambdaParameters, ...suffix]))
                : concat([removeLines(lambdaParameters), ...suffix]);
        }
        return concat([lambdaParameters, ...suffix]);
    }
    lambdaParameters(ctx) {
        if (ctx.lambdaParametersWithBraces) {
            return this.visitSingle(ctx);
        }
        return printTokenWithComments(this.getSingle(ctx));
    }
    lambdaParametersWithBraces(ctx) {
        var _a, _b, _c, _d, _e, _f;
        const lambdaParameters = (_f = (_c = (_b = (_a = ctx.lambdaParameterList) === null || _a === void 0 ? void 0 : _a[0].children.normalLambdaParameterList) === null || _b === void 0 ? void 0 : _b[0].children.normalLambdaParameter) !== null && _c !== void 0 ? _c : (_e = (_d = ctx.lambdaParameterList) === null || _d === void 0 ? void 0 : _d[0].children.conciseLambdaParameterList) === null || _e === void 0 ? void 0 : _e[0].children.conciseLambdaParameter) !== null && _f !== void 0 ? _f : [];
        handleCommentsParameters(ctx.LBrace[0], lambdaParameters, ctx.RBrace[0]);
        const lambdaParameterList = this.visit(ctx.lambdaParameterList);
        if (findDeepElementInPartsArray(lambdaParameterList, ",")) {
            return concat([
                ctx.LBrace[0],
                indent([softline, lambdaParameterList]),
                softline,
                ctx.RBrace[0]
            ]);
        }
        // removing braces when only no comments attached
        if ((ctx.LBrace &&
            ctx.RBrace &&
            (!lambdaParameterList || isExplicitLambdaParameter(ctx))) ||
            ctx.LBrace[0].leadingComments ||
            ctx.LBrace[0].trailingComments ||
            ctx.RBrace[0].leadingComments ||
            ctx.RBrace[0].trailingComments) {
            return rejectAndConcat([
                ctx.LBrace[0],
                lambdaParameterList,
                ctx.RBrace[0]
            ]);
        }
        return lambdaParameterList;
    }
    lambdaParameterList(ctx) {
        return this.visitSingle(ctx);
    }
    conciseLambdaParameterList(ctx) {
        var _a;
        const conciseLambdaParameters = this.mapVisit(ctx.conciseLambdaParameter);
        const commas = (_a = ctx.Comma) === null || _a === void 0 ? void 0 : _a.map(comma => concat([comma, line]));
        return rejectAndJoinSeps(commas, conciseLambdaParameters);
    }
    normalLambdaParameterList(ctx) {
        var _a;
        const normalLambdaParameter = this.mapVisit(ctx.normalLambdaParameter);
        const commas = (_a = ctx.Comma) === null || _a === void 0 ? void 0 : _a.map(comma => concat([comma, line]));
        return rejectAndJoinSeps(commas, normalLambdaParameter);
    }
    normalLambdaParameter(ctx) {
        return this.visitSingle(ctx);
    }
    regularLambdaParameter(ctx) {
        const variableModifier = this.mapVisit(ctx.variableModifier);
        const lambdaParameterType = this.visit(ctx.lambdaParameterType);
        const variableDeclaratorId = this.visit(ctx.variableDeclaratorId);
        return rejectAndJoin(" ", [
            rejectAndJoin(" ", variableModifier),
            lambdaParameterType,
            variableDeclaratorId
        ]);
    }
    lambdaParameterType(ctx) {
        if (ctx.unannType) {
            return this.visitSingle(ctx);
        }
        return printTokenWithComments(this.getSingle(ctx));
    }
    conciseLambdaParameter(ctx) {
        return printTokenWithComments(this.getSingle(ctx));
    }
    lambdaBody(ctx) {
        return this.visitSingle(ctx);
    }
    conditionalExpression(ctx, params) {
        const binaryExpression = this.visit(ctx.binaryExpression, params);
        if (ctx.QuestionMark) {
            const expression1 = this.visit(ctx.expression[0]);
            const expression2 = this.visit(ctx.expression[1]);
            return indent(group(rejectAndConcat([
                rejectAndJoin(line, [
                    binaryExpression,
                    rejectAndJoin(" ", [ctx.QuestionMark[0], expression1]),
                    rejectAndJoin(" ", [ctx.Colon[0], expression2])
                ])
            ])));
        }
        return binaryExpression;
    }
    binaryExpression(ctx, params) {
        handleCommentsBinaryExpression(ctx);
        const sortedNodes = sortNodes([
            ctx.pattern,
            ctx.referenceType,
            ctx.expression,
            ctx.unaryExpression
        ]);
        const nodes = this.mapVisit(sortedNodes, sortedNodes.length === 1 ? params : undefined);
        const tokens = sortTokens([
            ctx.Instanceof,
            ctx.AssignmentOperator,
            ctx.Less,
            ctx.Greater,
            ctx.BinaryOperator
        ]);
        const hasTokens = tokens.length > 0;
        const content = binary(nodes, tokens, true);
        return hasTokens && (params === null || params === void 0 ? void 0 : params.addParenthesisToWrapStatement)
            ? group(concat([
                ifBreak("("),
                indent(concat([softline, content])),
                softline,
                ifBreak(")")
            ]))
            : content;
    }
    unaryExpression(ctx, params) {
        const unaryPrefixOperator = ctx.UnaryPrefixOperator
            ? ctx.UnaryPrefixOperator
            : [];
        const primary = this.visit(ctx.primary, params);
        const unarySuffixOperator = ctx.UnarySuffixOperator
            ? ctx.UnarySuffixOperator
            : [];
        return rejectAndConcat([
            rejectAndConcat(unaryPrefixOperator),
            primary,
            rejectAndConcat(unarySuffixOperator)
        ]);
    }
    unaryExpressionNotPlusMinus(ctx) {
        const unaryPrefixOperatorNotPlusMinus = ctx.UnaryPrefixOperatorNotPlusMinus // changed when moved to TS
            ? rejectAndJoin(" ", ctx.UnaryPrefixOperatorNotPlusMinus) // changed when moved to TS
            : "";
        const primary = this.visit(ctx.primary);
        const unarySuffixOperator = ctx.UnarySuffixOperator // changed when moved to TS
            ? rejectAndJoin(" ", ctx.UnarySuffixOperator) // changed when moved to TS
            : "";
        return rejectAndJoin(" ", [
            unaryPrefixOperatorNotPlusMinus,
            primary,
            unarySuffixOperator
        ]);
    }
    primary(ctx, params) {
        var _a, _b, _c, _d;
        const countMethodInvocation = isUniqueMethodInvocation(ctx.primarySuffix);
        const newExpression = (_a = ctx.primaryPrefix[0].children.newExpression) === null || _a === void 0 ? void 0 : _a[0].children;
        const isBreakableNewExpression = countMethodInvocation <= 1 &&
            this.isBreakableNewExpression(newExpression);
        const fqnOrRefType = (_b = ctx.primaryPrefix[0].children.fqnOrRefType) === null || _b === void 0 ? void 0 : _b[0].children;
        const firstMethodInvocation = (_c = ctx.primarySuffix) === null || _c === void 0 ? void 0 : _c.map(suffix => { var _a; return (_a = suffix.children.methodInvocationSuffix) === null || _a === void 0 ? void 0 : _a[0].children; }).find(methodInvocationSuffix => methodInvocationSuffix);
        const isCapitalizedIdentifier = this.isCapitalizedIdentifier(fqnOrRefType);
        const shouldBreakBeforeFirstMethodInvocation = countMethodInvocation > 1 &&
            !(isCapitalizedIdentifier !== null && isCapitalizedIdentifier !== void 0 ? isCapitalizedIdentifier : true) &&
            firstMethodInvocation !== undefined;
        const shouldBreakBeforeMethodInvocations = shouldBreakBeforeFirstMethodInvocation ||
            countMethodInvocation > 2 ||
            (countMethodInvocation > 1 && newExpression) ||
            !(firstMethodInvocation === null || firstMethodInvocation === void 0 ? void 0 : firstMethodInvocation.argumentList);
        const primaryPrefix = this.visit(ctx.primaryPrefix, Object.assign(Object.assign({}, params), { shouldBreakBeforeFirstMethodInvocation }));
        const suffixes = [];
        if (ctx.primarySuffix !== undefined) {
            // edge case: https://github.com/jhipster/prettier-java/issues/381
            let hasFirstInvocationArg = true;
            if (ctx.primarySuffix.length > 1 &&
                ctx.primarySuffix[1].children.methodInvocationSuffix &&
                Object.keys(ctx.primarySuffix[1].children.methodInvocationSuffix[0].children).length === 2) {
                hasFirstInvocationArg = false;
            }
            if (newExpression &&
                !isBreakableNewExpression &&
                ctx.primarySuffix[0].children.Dot !== undefined) {
                suffixes.push(softline);
            }
            suffixes.push(this.visit(ctx.primarySuffix[0]));
            for (let i = 1; i < ctx.primarySuffix.length; i++) {
                if (shouldBreakBeforeMethodInvocations &&
                    ctx.primarySuffix[i].children.Dot !== undefined &&
                    ctx.primarySuffix[i - 1].children.methodInvocationSuffix !== undefined) {
                    suffixes.push(softline);
                }
                suffixes.push(this.visit(ctx.primarySuffix[i]));
            }
            if (!newExpression && countMethodInvocation === 1) {
                return group(rejectAndConcat([
                    primaryPrefix,
                    hasFirstInvocationArg ? suffixes[0] : indent(suffixes[0]),
                    indent(rejectAndConcat(suffixes.slice(1)))
                ]));
            }
        }
        const methodInvocation = (_d = ctx.primarySuffix) === null || _d === void 0 ? void 0 : _d[0].children.methodInvocationSuffix;
        const isMethodInvocationWithArguments = (methodInvocation === null || methodInvocation === void 0 ? void 0 : methodInvocation[0].children.argumentList) !== undefined;
        const isUnqualifiedMethodInvocation = methodInvocation !== undefined && !(fqnOrRefType === null || fqnOrRefType === void 0 ? void 0 : fqnOrRefType.Dot);
        return group(rejectAndConcat([
            primaryPrefix,
            isCapitalizedIdentifier || isUnqualifiedMethodInvocation
                ? suffixes.shift()
                : "",
            !isBreakableNewExpression &&
                (shouldBreakBeforeMethodInvocations || !isMethodInvocationWithArguments)
                ? indent(concat(suffixes))
                : concat(suffixes)
        ]));
    }
    primaryPrefix(ctx, params) {
        if (ctx.This || ctx.Void) {
            return printTokenWithComments(this.getSingle(ctx));
        }
        return this.visitSingle(ctx, params);
    }
    primarySuffix(ctx, params) {
        var _a;
        if (ctx.Dot) {
            if (ctx.This) {
                return rejectAndConcat([ctx.Dot[0], ctx.This[0]]);
            }
            else if (ctx.Identifier) {
                const typeArguments = this.visit(ctx.typeArguments);
                return rejectAndConcat([ctx.Dot[0], typeArguments, ctx.Identifier[0]]);
            }
            const suffix = this.visit((_a = ctx.unqualifiedClassInstanceCreationExpression) !== null && _a !== void 0 ? _a : ctx.templateArgument);
            return rejectAndConcat([ctx.Dot[0], suffix]);
        }
        return this.visitSingle(ctx, params);
    }
    fqnOrRefType(ctx, params) {
        const fqnOrRefTypePartFirst = this.visit(ctx.fqnOrRefTypePartFirst);
        const fqnOrRefTypePartRest = this.mapVisit(ctx.fqnOrRefTypePartRest);
        const dims = this.visit(ctx.dims);
        const dots = ctx.Dot ? ctx.Dot : [];
        const isMethodInvocation = ctx.Dot && ctx.Dot.length === 1;
        if (params !== undefined &&
            params.shouldBreakBeforeFirstMethodInvocation === true) {
            // when fqnOrRefType is a method call from an object
            if (isMethodInvocation) {
                return rejectAndConcat([
                    indent(rejectAndJoin(concat([softline, dots[0]]), [
                        fqnOrRefTypePartFirst,
                        rejectAndJoinSeps(dots.slice(1), fqnOrRefTypePartRest),
                        dims
                    ]))
                ]);
                // otherwise it is a fully qualified name but we need to exclude when it is just a method call
            }
            else if (ctx.Dot) {
                return indent(rejectAndConcat([
                    rejectAndJoinSeps(dots.slice(0, dots.length - 1), [
                        fqnOrRefTypePartFirst,
                        ...fqnOrRefTypePartRest.slice(0, fqnOrRefTypePartRest.length - 1)
                    ]),
                    softline,
                    rejectAndConcat([
                        dots[dots.length - 1],
                        fqnOrRefTypePartRest[fqnOrRefTypePartRest.length - 1]
                    ]),
                    dims
                ]));
            }
        }
        return rejectAndConcat([
            rejectAndJoinSeps(dots, [fqnOrRefTypePartFirst, ...fqnOrRefTypePartRest]),
            dims
        ]);
    }
    fqnOrRefTypePartFirst(ctx) {
        const annotation = this.mapVisit(ctx.annotation);
        const fqnOrRefTypeCommon = this.visit(ctx.fqnOrRefTypePartCommon);
        return rejectAndJoin(" ", [
            rejectAndJoin(" ", annotation),
            fqnOrRefTypeCommon
        ]);
    }
    fqnOrRefTypePartRest(ctx) {
        const annotation = this.mapVisit(ctx.annotation);
        const fqnOrRefTypeCommon = this.visit(ctx.fqnOrRefTypePartCommon);
        const typeArguments = this.visit(ctx.typeArguments);
        return rejectAndJoin(" ", [
            rejectAndJoin(" ", annotation),
            rejectAndConcat([typeArguments, fqnOrRefTypeCommon])
        ]);
    }
    fqnOrRefTypePartCommon(ctx) {
        let keyWord = null;
        if (ctx.Identifier) {
            keyWord = ctx.Identifier[0];
        }
        else {
            keyWord = ctx.Super[0];
        }
        const typeArguments = this.visit(ctx.typeArguments);
        return rejectAndConcat([keyWord, typeArguments]);
    }
    parenthesisExpression(ctx, params) {
        const expression = this.visit(ctx.expression);
        const separator = (params === null || params === void 0 ? void 0 : params.addParenthesisToWrapStatement) ? softline : "";
        return putIntoBraces(expression, separator, ctx.LBrace[0], ctx.RBrace[0]);
    }
    castExpression(ctx) {
        return this.visitSingle(ctx);
    }
    primitiveCastExpression(ctx) {
        const primitiveType = this.visit(ctx.primitiveType);
        const unaryExpression = this.visit(ctx.unaryExpression);
        return rejectAndJoin(" ", [
            rejectAndConcat([ctx.LBrace[0], primitiveType, ctx.RBrace[0]]),
            unaryExpression
        ]);
    }
    referenceTypeCastExpression(ctx) {
        const referenceType = this.visit(ctx.referenceType);
        const hasAdditionalBounds = ctx.additionalBound !== undefined;
        const additionalBounds = rejectAndJoin(line, this.mapVisit(ctx.additionalBound));
        const expression = ctx.lambdaExpression
            ? this.visit(ctx.lambdaExpression)
            : this.visit(ctx.unaryExpressionNotPlusMinus);
        return rejectAndJoin(" ", [
            putIntoBraces(rejectAndJoin(line, [referenceType, additionalBounds]), hasAdditionalBounds ? softline : "", ctx.LBrace[0], ctx.RBrace[0]),
            expression
        ]);
    }
    newExpression(ctx) {
        return this.visitSingle(ctx);
    }
    unqualifiedClassInstanceCreationExpression(ctx) {
        const typeArguments = this.visit(ctx.typeArguments);
        const classOrInterfaceTypeToInstantiate = this.visit(ctx.classOrInterfaceTypeToInstantiate);
        let content = printArgumentListWithBraces.call(this, ctx.argumentList, ctx.RBrace[0], ctx.LBrace[0]);
        const classBody = this.visit(ctx.classBody);
        return rejectAndJoin(" ", [
            ctx.New[0],
            rejectAndConcat([
                typeArguments,
                classOrInterfaceTypeToInstantiate,
                content
            ]),
            classBody
        ]);
    }
    classOrInterfaceTypeToInstantiate(ctx) {
        const tokens = sortAnnotationIdentifier(ctx.annotation, ctx.Identifier);
        const segments = [];
        let currentSegment = [];
        forEach(tokens, token => {
            if (isAnnotationCstNode(token)) {
                currentSegment.push(this.visit([token]));
            }
            else {
                currentSegment.push(token);
                segments.push(rejectAndJoin(" ", currentSegment));
                currentSegment = [];
            }
        });
        const typeArgumentsOrDiamond = this.visit(ctx.typeArgumentsOrDiamond);
        const dots = ctx.Dot ? ctx.Dot : [];
        return rejectAndConcat([
            rejectAndJoinSeps(dots, segments),
            typeArgumentsOrDiamond
        ]);
    }
    typeArgumentsOrDiamond(ctx) {
        return this.visitSingle(ctx);
    }
    diamond(ctx) {
        return concat([ctx.Less[0], ctx.Greater[0]]);
    }
    methodInvocationSuffix(ctx) {
        return printArgumentListWithBraces.call(this, ctx.argumentList, ctx.RBrace[0], ctx.LBrace[0]);
    }
    argumentList(ctx) {
        var _a, _b;
        const headArgs = this.mapVisit(ctx.expression.slice(0, -1)).map((expression, index) => concat([expression, ctx.Comma[index], line]));
        const lastExpression = ctx.expression.at(-1);
        const lastArg = this.visit(lastExpression);
        if (this.isArgumentListHuggable(ctx)) {
            const huggedLastArg = this.visit(lastExpression, { hug: true });
            const lastArgNotHuggable = typeof huggedLastArg === "object" &&
                !Array.isArray(huggedLastArg) &&
                huggedLastArg.type === "label" &&
                ((_a = huggedLastArg.label) === null || _a === void 0 ? void 0 : _a.huggable) === false;
            if (lastArgNotHuggable || headArgs.some(willBreak)) {
                return group([indent([line, ...headArgs, lastArg]), line], {
                    shouldBreak: true
                });
            }
            const suffix = ((_b = lastExpression === null || lastExpression === void 0 ? void 0 : lastExpression.children.lambdaExpression) === null || _b === void 0 ? void 0 : _b[0].children.lambdaBody[0].children.block)
                ? ""
                : line;
            const hugged = [
                ...headArgs,
                group([huggedLastArg, suffix], { shouldBreak: true })
            ];
            const expanded = group([indent([line, ...headArgs, lastArg]), line], {
                shouldBreak: true
            });
            return willBreak(huggedLastArg)
                ? [breakParent, conditionalGroup([hugged, expanded])]
                : conditionalGroup([[...headArgs, huggedLastArg], hugged, expanded]);
        }
        return group([indent([softline, ...headArgs, lastArg]), softline]);
    }
    arrayCreationExpression(ctx) {
        const type = ctx.primitiveType
            ? this.visit(ctx.primitiveType)
            : this.visit(ctx.classOrInterfaceType);
        const suffix = ctx.arrayCreationExpressionWithoutInitializerSuffix
            ? this.visit(ctx.arrayCreationExpressionWithoutInitializerSuffix)
            : this.visit(ctx.arrayCreationWithInitializerSuffix);
        return rejectAndConcat([concat([ctx.New[0], " "]), type, suffix]);
    }
    arrayCreationExpressionWithoutInitializerSuffix(ctx) {
        const dimExprs = this.visit(ctx.dimExprs);
        const dims = this.visit(ctx.dims);
        return rejectAndConcat([dimExprs, dims]);
    }
    arrayCreationWithInitializerSuffix(ctx) {
        const dims = this.visit(ctx.dims);
        const arrayInitializer = this.visit(ctx.arrayInitializer);
        return rejectAndJoin(" ", [dims, arrayInitializer]);
    }
    dimExprs(ctx) {
        const dimExpr = this.mapVisit(ctx.dimExpr);
        return rejectAndConcat(dimExpr);
    }
    dimExpr(ctx) {
        const annotations = this.mapVisit(ctx.annotation);
        const expression = this.visit(ctx.expression);
        return rejectAndJoin(" ", [
            rejectAndJoin(" ", annotations),
            rejectAndConcat([ctx.LSquare[0], expression, ctx.RSquare[0]])
        ]);
    }
    classLiteralSuffix(ctx) {
        const squares = [];
        if (ctx.LSquare) {
            for (let i = 0; i < ctx.LSquare.length; i++) {
                squares.push(concat([ctx.LSquare[i], ctx.RSquare[i]]));
            }
        }
        return rejectAndConcat([...squares, ctx.Dot[0], ctx.Class[0]]);
    }
    arrayAccessSuffix(ctx) {
        const expression = this.visit(ctx.expression);
        return rejectAndConcat([ctx.LSquare[0], expression, ctx.RSquare[0]]);
    }
    methodReferenceSuffix(ctx) {
        const typeArguments = this.visit(ctx.typeArguments);
        const identifierOrNew = ctx.New ? ctx.New[0] : ctx.Identifier[0];
        return rejectAndConcat([ctx.ColonColon[0], typeArguments, identifierOrNew]);
    }
    templateArgument(ctx) {
        var _a;
        return ctx.template
            ? this.visit(ctx.template)
            : printTokenWithComments(((_a = ctx.StringLiteral) !== null && _a !== void 0 ? _a : ctx.TextBlock)[0]);
    }
    template(ctx) {
        return this.visitSingle(ctx);
    }
    stringTemplate(ctx) {
        const embeddedExpressions = this.mapVisit(ctx.embeddedExpression).flatMap(expression => group([softline, expression, lineSuffixBoundary, dedent(softline)]));
        return concat([
            ctx.StringTemplateBegin[0],
            rejectAndJoinSeps(ctx.StringTemplateMid, embeddedExpressions),
            ctx.StringTemplateEnd[0]
        ]);
    }
    textBlockTemplate(ctx) {
        const embeddedExpressions = this.mapVisit(ctx.embeddedExpression).flatMap(expression => group([softline, expression, lineSuffixBoundary, dedent(softline)]));
        return concat([
            ctx.TextBlockTemplateBegin[0],
            rejectAndJoinSeps(ctx.TextBlockTemplateMid, embeddedExpressions),
            ctx.TextBlockTemplateEnd[0]
        ]);
    }
    embeddedExpression(ctx) {
        return this.visit(ctx.expression);
    }
    pattern(ctx) {
        return this.visitSingle(ctx);
    }
    typePattern(ctx) {
        return this.visitSingle(ctx);
    }
    recordPattern(ctx) {
        var _a, _b;
        const componentPatterns = (_b = (_a = ctx.componentPatternList) === null || _a === void 0 ? void 0 : _a[0].children.componentPattern) !== null && _b !== void 0 ? _b : [];
        handleCommentsParameters(ctx.LBrace[0], componentPatterns, ctx.RBrace[0]);
        const referenceType = this.visit(ctx.referenceType);
        const componentPatternList = this.visit(ctx.componentPatternList);
        return concat([
            referenceType,
            putIntoBraces(componentPatternList, softline, ctx.LBrace[0], ctx.RBrace[0])
        ]);
    }
    componentPatternList(ctx) {
        var _a, _b;
        const componentPatterns = this.mapVisit(ctx.componentPattern);
        const commas = (_b = (_a = ctx.Comma) === null || _a === void 0 ? void 0 : _a.map(elt => concat([elt, line]))) !== null && _b !== void 0 ? _b : [];
        return rejectAndJoinSeps(commas, componentPatterns);
    }
    componentPattern(ctx) {
        return this.visitSingle(ctx);
    }
    matchAllPattern(ctx) {
        return printTokenWithComments(ctx.Underscore[0]);
    }
    guard(ctx) {
        const expression = this.visit(ctx.expression, {
            addParenthesisToWrapStatement: true
        });
        return concat([ctx.When[0], " ", expression]);
    }
    isRefTypeInMethodRef() {
        return "isRefTypeInMethodRef";
    }
    isArgumentListHuggable(argumentList) {
        var _a, _b, _c;
        const expressions = argumentList.expression;
        const lastArgument = expressions.at(-1);
        const lastArgumentLambdaBodyExpression = (_b = (_a = lastArgument === null || lastArgument === void 0 ? void 0 : lastArgument.children.lambdaExpression) === null || _a === void 0 ? void 0 : _a[0].children.lambdaBody[0].children.expression) === null || _b === void 0 ? void 0 : _b[0].children;
        const lastArgumentLambdaBodyTernaryExpression = (_c = lastArgumentLambdaBodyExpression === null || lastArgumentLambdaBodyExpression === void 0 ? void 0 : lastArgumentLambdaBodyExpression.conditionalExpression) === null || _c === void 0 ? void 0 : _c[0].children;
        return (!(lastArgument === null || lastArgument === void 0 ? void 0 : lastArgument.leadingComments) &&
            !(lastArgument === null || lastArgument === void 0 ? void 0 : lastArgument.trailingComments) &&
            (!lastArgumentLambdaBodyExpression ||
                (lastArgumentLambdaBodyTernaryExpression === null || lastArgumentLambdaBodyTernaryExpression === void 0 ? void 0 : lastArgumentLambdaBodyTernaryExpression.QuestionMark) !== undefined ||
                (lastArgumentLambdaBodyTernaryExpression === null || lastArgumentLambdaBodyTernaryExpression === void 0 ? void 0 : lastArgumentLambdaBodyTernaryExpression.binaryExpression[0].children.unaryExpression.length) === 1) &&
            expressions.findIndex(({ children }) => children.lambdaExpression) ===
                expressions.length - 1);
    }
    isBreakableNewExpression(newExpression) {
        var _a, _b, _c, _d, _e;
        const arrayCreationExpression = (_a = newExpression === null || newExpression === void 0 ? void 0 : newExpression.arrayCreationExpression) === null || _a === void 0 ? void 0 : _a[0].children;
        const classInstanceCreationExpression = (_b = newExpression === null || newExpression === void 0 ? void 0 : newExpression.unqualifiedClassInstanceCreationExpression) === null || _b === void 0 ? void 0 : _b[0].children;
        return [
            (_c = arrayCreationExpression === null || arrayCreationExpression === void 0 ? void 0 : arrayCreationExpression.classOrInterfaceType) === null || _c === void 0 ? void 0 : _c[0].children.classType[0].children.typeArguments,
            (_d = arrayCreationExpression === null || arrayCreationExpression === void 0 ? void 0 : arrayCreationExpression.arrayCreationWithInitializerSuffix) === null || _d === void 0 ? void 0 : _d[0].children.arrayInitializer[0].children.variableInitializerList,
            (_e = classInstanceCreationExpression === null || classInstanceCreationExpression === void 0 ? void 0 : classInstanceCreationExpression.classOrInterfaceTypeToInstantiate[0].children.typeArgumentsOrDiamond) === null || _e === void 0 ? void 0 : _e[0].children.typeArguments,
            classInstanceCreationExpression === null || classInstanceCreationExpression === void 0 ? void 0 : classInstanceCreationExpression.argumentList
        ].some(breakablePart => breakablePart !== undefined);
    }
    isCapitalizedIdentifier(fqnOrRefType) {
        var _a, _b, _c;
        const fqnOrRefTypeParts = [
            fqnOrRefType === null || fqnOrRefType === void 0 ? void 0 : fqnOrRefType.fqnOrRefTypePartFirst[0],
            ...((_a = fqnOrRefType === null || fqnOrRefType === void 0 ? void 0 : fqnOrRefType.fqnOrRefTypePartRest) !== null && _a !== void 0 ? _a : [])
        ];
        const nextToLastIdentifier = (_c = (_b = fqnOrRefTypeParts[fqnOrRefTypeParts.length - 2]) === null || _b === void 0 ? void 0 : _b.children.fqnOrRefTypePartCommon[0].children.Identifier) === null || _c === void 0 ? void 0 : _c[0].image;
        return (nextToLastIdentifier &&
            /^\p{Uppercase_Letter}/u.test(nextToLastIdentifier));
    }
}
