import forEach from "lodash/forEach.js";
import { builders } from "prettier/doc";
import { concat, group, indent, join } from "./prettier-builder.js";
import { printTokenWithComments } from "./comments/format-comments.js";
import { putIntoBraces, rejectAndConcat, rejectAndJoin, rejectAndJoinSeps, sortClassTypeChildren } from "./printer-utils.js";
import { BaseCstPrettierPrinter } from "../base-cst-printer.js";
import { isAnnotationCstNode, isCstNode, isTypeArgumentsCstNode } from "../types/utils.js";
const { line, softline } = builders;
export class TypesValuesAndVariablesPrettierVisitor extends BaseCstPrettierPrinter {
    primitiveType(ctx) {
        const annotations = this.mapVisit(ctx.annotation);
        const type = ctx.numericType
            ? this.visit(ctx.numericType)
            : this.getSingle(ctx);
        return rejectAndJoin(" ", [join(" ", annotations), type]);
    }
    numericType(ctx) {
        return this.visitSingle(ctx);
    }
    integralType(ctx) {
        return printTokenWithComments(this.getSingle(ctx));
    }
    floatingPointType(ctx) {
        return printTokenWithComments(this.getSingle(ctx));
    }
    referenceType(ctx) {
        const annotations = this.mapVisit(ctx.annotation);
        const type = ctx.primitiveType
            ? this.visit(ctx.primitiveType)
            : this.visit(ctx.classOrInterfaceType);
        const dims = this.visit(ctx.dims);
        return rejectAndJoin(" ", [join(" ", annotations), concat([type, dims])]);
    }
    classOrInterfaceType(ctx) {
        return this.visitSingle(ctx);
    }
    classType(ctx) {
        const tokens = sortClassTypeChildren(ctx.annotation, ctx.typeArguments, ctx.Identifier);
        const segments = [];
        let currentSegment = [];
        forEach(tokens, (token, i) => {
            if (isTypeArgumentsCstNode(token)) {
                currentSegment.push(this.visit([token]));
                segments.push(rejectAndConcat(currentSegment));
                currentSegment = [];
            }
            else if (isAnnotationCstNode(token)) {
                currentSegment.push(this.visit([token]), " ");
            }
            else {
                currentSegment.push(token);
                if ((i + 1 < tokens.length && !isTypeArgumentsCstNode(tokens[i + 1])) ||
                    i + 1 === tokens.length) {
                    segments.push(rejectAndConcat(currentSegment));
                    currentSegment = [];
                }
            }
        });
        return rejectAndJoinSeps(ctx.Dot, segments);
    }
    interfaceType(ctx) {
        return this.visitSingle(ctx);
    }
    typeVariable(ctx) {
        const annotations = this.mapVisit(ctx.annotation);
        const identifier = this.getSingle(ctx);
        return rejectAndJoin(" ", [join(" ", annotations), identifier]);
    }
    dims(ctx) {
        let tokens = [...ctx.LSquare];
        if (ctx.annotation) {
            tokens = [...tokens, ...ctx.annotation];
        }
        tokens = tokens.sort((a, b) => {
            const startOffset1 = isCstNode(a)
                ? a.children.At[0].startOffset
                : a.startOffset;
            const startOffset2 = isCstNode(b)
                ? b.children.At[0].startOffset
                : b.startOffset;
            return startOffset1 - startOffset2;
        });
        const segments = [];
        let currentSegment = [];
        forEach(tokens, token => {
            if (isCstNode(token)) {
                currentSegment.push(this.visit([token]));
            }
            else {
                segments.push(rejectAndConcat([
                    rejectAndJoin(" ", currentSegment),
                    concat([ctx.LSquare[0], ctx.RSquare[0]])
                ]));
                currentSegment = [];
            }
        });
        return rejectAndConcat(segments);
    }
    typeParameter(ctx) {
        const typeParameterModifiers = this.mapVisit(ctx.typeParameterModifier);
        const typeIdentifier = this.visit(ctx.typeIdentifier);
        const typeBound = this.visit(ctx.typeBound);
        return rejectAndJoin(" ", [
            join(" ", typeParameterModifiers),
            typeIdentifier,
            typeBound
        ]);
    }
    typeParameterModifier(ctx) {
        return this.visitSingle(ctx);
    }
    typeBound(ctx) {
        const classOrInterfaceType = this.visit(ctx.classOrInterfaceType);
        const additionalBound = this.mapVisit(ctx.additionalBound);
        return concat([
            rejectAndJoin(" ", [ctx.Extends[0], classOrInterfaceType]),
            indent(group(concat([
                additionalBound.length ? line : "",
                rejectAndJoin(line, additionalBound)
            ])))
        ]);
    }
    additionalBound(ctx) {
        const interfaceType = this.visit(ctx.interfaceType);
        return join(" ", [ctx.And[0], interfaceType]);
    }
    typeArguments(ctx) {
        const typeArgumentList = this.visit(ctx.typeArgumentList);
        return putIntoBraces(typeArgumentList, softline, ctx.Less[0], ctx.Greater[0]);
    }
    typeArgumentList(ctx) {
        const typeArguments = this.mapVisit(ctx.typeArgument);
        const commas = ctx.Comma ? ctx.Comma.map(elt => concat([elt, line])) : [];
        return rejectAndJoinSeps(commas, typeArguments);
    }
    typeArgument(ctx) {
        return this.visitSingle(ctx);
    }
    wildcard(ctx) {
        const annotations = this.mapVisit(ctx.annotation);
        const wildcardBounds = this.visit(ctx.wildcardBounds);
        return rejectAndJoin(" ", [
            join(" ", annotations),
            ctx.QuestionMark[0],
            wildcardBounds
        ]);
    }
    wildcardBounds(ctx) {
        const keyWord = ctx.Extends ? ctx.Extends[0] : ctx.Super[0];
        const referenceType = this.visit(ctx.referenceType);
        return join(" ", [keyWord, referenceType]);
    }
}
