declare namespace prettyMilliseconds {
	interface Options {
		/**
		Number of digits to appear after the seconds decimal point.

		@default 1
		*/
		readonly secondsDecimalDigits?: number;

		/**
		Number of digits to appear after the milliseconds decimal point.

		Useful in combination with [`process.hrtime()`](https://nodejs.org/api/process.html#process_process_hrtime).

		@default 0
		*/
		readonly millisecondsDecimalDigits?: number;

		/**
		Keep milliseconds on whole seconds: `13s` → `13.0s`.

		Useful when you are showing a number of seconds spent on an operation and don't want the width of the output to change when hitting a whole number.

		@default false
		*/
		readonly keepDecimalsOnWholeSeconds?: boolean;

		/**
		Only show the first unit: `1h 10m` → `~1h`.

		Also ensures that `millisecondsDecimalDigits` and `secondsDecimalDigits` are both set to `0`.

		@default false
		*/
		readonly compact?: boolean;

		/**
		Number of units to show. Setting `compact` to `true` overrides this option.

		@default Infinity
		*/
		readonly unitCount?: number;

		/**
		Use full-length units: `5h 1m 45s` → `5 hours 1 minute 45 seconds`.

		@default false
		*/
		readonly verbose?: boolean;

		/**
		Show milliseconds separately. This means they won't be included in the decimal part of the seconds.

		@default false
		*/
		readonly separateMilliseconds?: boolean;

		/**
		Show microseconds and nanoseconds.

		@default false
		*/
		readonly formatSubMilliseconds?: boolean;
	}
}

/**
Convert milliseconds to a human readable string: `1337000000` → `15d 11h 23m 20s`.

@param milliseconds - Milliseconds to humanize.

@example
```
import prettyMilliseconds = require('pretty-ms');

prettyMilliseconds(1337000000);
//=> '15d 11h 23m 20s'

prettyMilliseconds(1337);
//=> '1.3s'

prettyMilliseconds(133);
//=> '133ms'

// `compact` option
prettyMilliseconds(1337, {compact: true});
//=> '~1s'

// `verbose` option
prettyMilliseconds(1335669000, {verbose: true});
//=> '15 days 11 hours 1 minute 9 seconds'

// `formatSubMilliseconds` option
prettyMilliseconds(100.400080, {formatSubMilliseconds: true})
//=> '100ms 400µs 80ns'

// Can be useful for time durations
prettyMilliseconds(new Date(2014, 0, 1, 10, 40) - new Date(2014, 0, 1, 10, 5))
//=> '35m'
```
*/
declare function prettyMilliseconds(
	milliseconds: number,
	options?: prettyMilliseconds.Options
): string;

export = prettyMilliseconds;
