'use strict';

var utils = require('primevue/utils');
var OverlayEventBus = require('primevue/overlayeventbus');
var api = require('primevue/api');
var Ripple = require('primevue/ripple');
var VirtualScroller = require('primevue/virtualscroller');
var vue = require('vue');

function _interopDefaultLegacy (e) { return e && typeof e === 'object' && 'default' in e ? e : { 'default': e }; }

var OverlayEventBus__default = /*#__PURE__*/_interopDefaultLegacy(OverlayEventBus);
var Ripple__default = /*#__PURE__*/_interopDefaultLegacy(Ripple);
var VirtualScroller__default = /*#__PURE__*/_interopDefaultLegacy(VirtualScroller);

var script = {
    name: 'Dropdown',
    emits: ['update:modelValue', 'before-show', 'before-hide', 'show', 'hide', 'change', 'filter', 'focus', 'blur'],
    props: {
        modelValue: null,
        options: Array,
        optionLabel: null,
        optionValue: null,
        optionDisabled: null,
        optionGroupLabel: null,
        optionGroupChildren: null,
		scrollHeight: {
			type: String,
			default: '200px'
		},
		filter: Boolean,
        filterPlaceholder: String,
        filterLocale: String,
        filterMatchMode: {
            type: String,
            default: 'contains'
        },
        filterFields: {
            type: Array,
            default: null
        },
		editable: Boolean,
		placeholder: String,
		disabled: Boolean,
        dataKey: null,
        showClear: Boolean,
        inputId: String,
        tabindex: String,
        ariaLabelledBy: null,
        appendTo: {
            type: String,
            default: 'body'
        },
        emptyFilterMessage: {
            type: String,
            default: null
        },
        emptyMessage: {
            type: String,
            default: null
        },
        panelClass: null,
        loading: {
            type: Boolean,
            default: false
        },
        loadingIcon: {
            type: String,
            default: 'pi pi-spinner pi-spin'
        },
        virtualScrollerOptions: {
            type: Object,
            default: null
        }
    },
    data() {
        return {
            focused: false,
            filterValue: null,
            overlayVisible: false
        };
    },
    watch: {
        modelValue() {
            this.isModelValueChanged = true;
        }
    },
    outsideClickListener: null,
    scrollHandler: null,
    resizeListener: null,
    searchTimeout: null,
    currentSearchChar: null,
    previousSearchChar: null,
    searchValue: null,
    overlay: null,
    itemsWrapper: null,
    virtualScroller: null,
    isModelValueChanged: false,
    updated() {
        if (this.overlayVisible && this.isModelValueChanged) {
            this.scrollValueInView();
        }

        this.isModelValueChanged = false;
    },
    beforeUnmount() {
        this.unbindOutsideClickListener();
        this.unbindResizeListener();

        if (this.scrollHandler) {
            this.scrollHandler.destroy();
            this.scrollHandler = null;
        }

        this.itemsWrapper = null;

        if (this.overlay) {
            utils.ZIndexUtils.clear(this.overlay);
            this.overlay = null;
        }
    },
    methods: {
        getOptionIndex(index, fn) {
            return this.virtualScrollerDisabled ? index : (fn && fn(index)['index']);
        },
        getOptionLabel(option) {
            return this.optionLabel ? utils.ObjectUtils.resolveFieldData(option, this.optionLabel) : option;
        },
        getOptionValue(option) {
            return this.optionValue ? utils.ObjectUtils.resolveFieldData(option, this.optionValue) : option;
        },
        getOptionRenderKey(option) {
            return this.dataKey ? utils.ObjectUtils.resolveFieldData(option, this.dataKey) : this.getOptionLabel(option);
        },
        isOptionDisabled(option) {
            return this.optionDisabled ? utils.ObjectUtils.resolveFieldData(option, this.optionDisabled) : false;
        },
        getOptionGroupRenderKey(optionGroup) {
            return utils.ObjectUtils.resolveFieldData(optionGroup, this.optionGroupLabel);
        },
        getOptionGroupLabel(optionGroup) {
            return utils.ObjectUtils.resolveFieldData(optionGroup, this.optionGroupLabel);
        },
        getOptionGroupChildren(optionGroup) {
            return utils.ObjectUtils.resolveFieldData(optionGroup, this.optionGroupChildren);
        },
        getSelectedOption() {
            let index = this.getSelectedOptionIndex();
            return index !== -1 ? (this.optionGroupLabel ? this.getOptionGroupChildren(this.visibleOptions[index.group])[index.option]: this.visibleOptions[index]) : null;
        },
        getSelectedOptionIndex() {
            if (this.modelValue != null && this.visibleOptions) {
                if (this.optionGroupLabel) {
                    for (let i = 0; i < this.visibleOptions.length; i++) {
                        let selectedOptionIndex = this.findOptionIndexInList(this.modelValue, this.getOptionGroupChildren(this.visibleOptions[i]));
                        if (selectedOptionIndex !== -1) {
                            return {group: i, option: selectedOptionIndex};
                        }
                    }
                }
                else {
                    return this.findOptionIndexInList(this.modelValue, this.visibleOptions);
                }
            }

            return -1;
        },
        findOptionIndexInList(value, list) {
            for (let i = 0; i < list.length; i++) {
                if ((utils.ObjectUtils.equals(value, this.getOptionValue(list[i]), this.equalityKey))) {
                    return i;
                }
            }

            return -1;
        },
        isSelected(option) {
            return utils.ObjectUtils.equals(this.modelValue, this.getOptionValue(option), this.equalityKey);
        },
        show() {
            this.$emit('before-show');
            this.overlayVisible = true;
        },
        hide() {
            this.$emit('before-hide');
            this.overlayVisible = false;
        },
        onFocus(event) {
            this.focused = true;
            this.$emit('focus', event);
        },
        onBlur(event) {
            this.focused = false;
            this.$emit('blur', event);
        },
        onKeyDown(event) {
            switch(event.which) {
                //down
                case 40:
                    this.onDownKey(event);
                break;

                //up
                case 38:
                    this.onUpKey(event);
                break;

                //space
                case 32:
                    if (!this.overlayVisible) {
                        this.show();
                        event.preventDefault();
                    }
                break;

                //enter and escape
                case 13:
                case 27:
                    if (this.overlayVisible) {
                        this.hide();
                        event.preventDefault();
                    }
                break;

                //tab
                case 9:
                    this.hide();
                break;

                default:
                    this.search(event);
                break;
            }
        },
        onFilterKeyDown(event) {
            switch (event.which) {
                //down
                case 40:
                    this.onDownKey(event);
                    break;

                //up
                case 38:
                    this.onUpKey(event);
                    break;

                //enter and escape
                case 13:
                case 27:
                    this.overlayVisible = false;
                    event.preventDefault();
                break;
            }
        },
        onDownKey(event) {
            if (this.visibleOptions) {
                if (!this.overlayVisible && event.altKey) {
                    this.show();
                }
                else {
                    let nextOption = this.visibleOptions && this.visibleOptions.length > 0 ? this.findNextOption(this.getSelectedOptionIndex()) : null;
                    if (nextOption) {
                        this.updateModel(event, this.getOptionValue(nextOption));
                    }
                }
            }

            event.preventDefault();
        },
        onUpKey(event) {
            if (this.visibleOptions) {
                let prevOption = this.findPrevOption(this.getSelectedOptionIndex());
                if (prevOption) {
                    this.updateModel(event, this.getOptionValue(prevOption));
                }
            }

            event.preventDefault();
        },
        findNextOption(index) {
            if (this.optionGroupLabel) {
                let groupIndex = index === -1 ? 0 : index.group;
                let optionIndex = index === -1 ? -1 : index.option;
                let option = this.findNextOptionInList(this.getOptionGroupChildren(this.visibleOptions[groupIndex]), optionIndex);

                if (option)
                    return option;
                else if ((groupIndex + 1) !== this.visibleOptions.length)
                    return this.findNextOption({group: (groupIndex + 1), option: -1});
                else
                    return null;
            }
            else {
                return this.findNextOptionInList(this.visibleOptions, index);
            }
        },
        findNextOptionInList(list, index) {
                let i = index + 1;
                if (i === list.length) {
                    return null;
                }

                let option = list[i];
                if (this.isOptionDisabled(option))
                    return this.findNextOptionInList(i);
                else
                    return option;
        },
        findPrevOption(index) {
            if (index === -1) {
                return null;
            }

            if (this.optionGroupLabel) {
                let groupIndex = index.group;
                let optionIndex = index.option;
                let option = this.findPrevOptionInList(this.getOptionGroupChildren(this.visibleOptions[groupIndex]), optionIndex);

                if (option)
                    return option;
                else if (groupIndex > 0)
                    return this.findPrevOption({group: (groupIndex - 1), option: this.getOptionGroupChildren(this.visibleOptions[groupIndex - 1]).length});
                else
                    return null;
            }
            else {
                return this.findPrevOptionInList(this.visibleOptions, index);
            }
        },
        findPrevOptionInList(list, index) {
            let i = index - 1;
            if (i < 0) {
                return null;
            }

            let option = list[i];
            if (this.isOptionDisabled(option))
                return this.findPrevOption(i);
            else
                return option;
        },
        onClearClick(event) {
            this.updateModel(event, null);
        },
        onClick(event) {
            if (this.disabled || this.loading) {
                return;
            }

            if (utils.DomHandler.hasClass(event.target, 'p-dropdown-clear-icon') || event.target.tagName === 'INPUT') {
                return;
            }
            else if (!this.overlay || !this.overlay.contains(event.target)) {
                if (this.overlayVisible)
                    this.hide();
                else
                    this.show();

                this.$refs.focusInput.focus();
            }
        },
        onOptionSelect(event, option) {
            let value = this.getOptionValue(option);
            this.updateModel(event, value);
            this.$refs.focusInput.focus();

            setTimeout(() => {
                this.hide();
            }, 200);
        },
        onEditableInput(event) {
            this.$emit('update:modelValue', event.target.value);
        },
        onOverlayEnter(el) {
            utils.ZIndexUtils.set('overlay', el, this.$primevue.config.zIndex.overlay);
            this.alignOverlay();
            this.scrollValueInView();

            if (!this.virtualScrollerDisabled) {
                const selectedIndex = this.getSelectedOptionIndex();
                if (selectedIndex !== -1) {
                    setTimeout(() => {
                        this.virtualScroller && this.virtualScroller.scrollToIndex(selectedIndex);
                    }, 0);
                }
            }
        },
        onOverlayAfterEnter() {
            if (this.filter) {
                this.$refs.filterInput.focus();
            }

            this.bindOutsideClickListener();
            this.bindScrollListener();
            this.bindResizeListener();

            this.$emit('show');
        },
        onOverlayLeave() {
            this.unbindOutsideClickListener();
            this.unbindScrollListener();
            this.unbindResizeListener();
            this.$emit('hide');
            this.itemsWrapper = null;
            this.overlay = null;
        },
        onOverlayAfterLeave(el) {
            utils.ZIndexUtils.clear(el);
        },
        alignOverlay() {
            if (this.appendDisabled) {
                utils.DomHandler.relativePosition(this.overlay, this.$el);
            }
            else {
                this.overlay.style.minWidth = utils.DomHandler.getOuterWidth(this.$el) + 'px';
                utils.DomHandler.absolutePosition(this.overlay, this.$el);
            }
        },
        updateModel(event, value) {
            this.$emit('update:modelValue', value);
            this.$emit('change', {originalEvent: event, value: value});
        },
        bindOutsideClickListener() {
            if (!this.outsideClickListener) {
                this.outsideClickListener = (event) => {
                    if (this.overlayVisible && this.overlay && !this.$el.contains(event.target) && !this.overlay.contains(event.target)) {
                        this.hide();
                    }
                };
                document.addEventListener('click', this.outsideClickListener);
            }
        },
        unbindOutsideClickListener() {
            if (this.outsideClickListener) {
                document.removeEventListener('click', this.outsideClickListener);
                this.outsideClickListener = null;
            }
        },
        bindScrollListener() {
            if (!this.scrollHandler) {
                this.scrollHandler = new utils.ConnectedOverlayScrollHandler(this.$refs.container, () => {
                    if (this.overlayVisible) {
                        this.hide();
                    }
                });
            }

            this.scrollHandler.bindScrollListener();
        },
        unbindScrollListener() {
            if (this.scrollHandler) {
                this.scrollHandler.unbindScrollListener();
            }
        },
        bindResizeListener() {
            if (!this.resizeListener) {
                this.resizeListener = () => {
                    if (this.overlayVisible && !utils.DomHandler.isTouchDevice()) {
                        this.hide();
                    }
                };
                window.addEventListener('resize', this.resizeListener);
            }
        },
        unbindResizeListener() {
            if (this.resizeListener) {
                window.removeEventListener('resize', this.resizeListener);
                this.resizeListener = null;
            }
        },
        search(event) {
            if (!this.visibleOptions) {
                return;
            }

            if (this.searchTimeout) {
                clearTimeout(this.searchTimeout);
            }

            const char = event.key;
            this.previousSearchChar = this.currentSearchChar;
            this.currentSearchChar = char;

            if (this.previousSearchChar === this.currentSearchChar)
                this.searchValue = this.currentSearchChar;
            else
                this.searchValue = this.searchValue ? this.searchValue + char : char;

            if (this.searchValue) {
                let searchIndex = this.getSelectedOptionIndex();
                let newOption = this.optionGroupLabel ? this.searchOptionInGroup(searchIndex) : this.searchOption(++searchIndex);
                if (newOption) {
                    this.updateModel(event, this.getOptionValue(newOption));
                }
            }

            this.searchTimeout = setTimeout(() => {
                this.searchValue = null;
            }, 250);
        },
        searchOption(index) {
            let option;

            if (this.searchValue) {
                option = this.searchOptionInRange(index, this.visibleOptions.length);

                if (!option) {
                    option = this.searchOptionInRange(0, index);
                }
            }

            return option;
        },
        searchOptionInRange(start, end) {
            for (let i = start; i < end; i++) {
                let opt = this.visibleOptions[i];
                if (this.matchesSearchValue(opt)) {
                    return opt;
                }
            }

            return null;
        },
        searchOptionInGroup(index) {
            let searchIndex = index === -1 ? {group: 0, option: -1} : index;

            for (let i = searchIndex.group; i < this.visibleOptions.length; i++) {
                let groupOptions = this.getOptionGroupChildren(this.visibleOptions[i]);
                for (let j = (searchIndex.group === i ? searchIndex.option + 1 : 0); j < groupOptions.length; j++) {
                    if (this.matchesSearchValue(groupOptions[j])) {
                        return groupOptions[j];
                    }
                }
            }

            for (let i = 0; i <= searchIndex.group; i++) {
                let groupOptions = this.getOptionGroupChildren(this.visibleOptions[i]);
                for (let j = 0; j < (searchIndex.group === i ? searchIndex.option: groupOptions.length); j++) {
                    if (this.matchesSearchValue(groupOptions[j])) {
                        return groupOptions[j];
                    }
                }
            }

            return null;
        },
        matchesSearchValue(option) {
            let label = this.getOptionLabel(option).toLocaleLowerCase(this.filterLocale);
            return label.startsWith(this.searchValue.toLocaleLowerCase(this.filterLocale));
        },
        onFilterChange(event) {
            this.filterValue = event.target.value;
            this.$emit('filter', {originalEvent: event, value: event.target.value});
        },
        onFilterUpdated() {
            if (this.overlayVisible) {
                this.alignOverlay();
            }
        },
        overlayRef(el) {
            this.overlay = el;
        },
        itemsWrapperRef(el) {
            this.itemsWrapper = el;
        },
        virtualScrollerRef(el) {
            this.virtualScroller = el;
        },
        scrollValueInView() {
            if (this.overlay) {
                let selectedItem = utils.DomHandler.findSingle(this.overlay, 'li.p-highlight');
                if (selectedItem) {
                    selectedItem.scrollIntoView({ block: 'nearest', inline: 'start' });
                }
            }
        },
        onOverlayClick(event) {
            OverlayEventBus__default["default"].emit('overlay-click', {
                originalEvent: event,
                target: this.$el
            });
        }
    },
    computed: {
        visibleOptions() {
            if (this.filterValue) {
                if (this.optionGroupLabel) {
                    let filteredGroups = [];
                    for (let optgroup of this.options) {
                        let filteredSubOptions = api.FilterService.filter(this.getOptionGroupChildren(optgroup), this.searchFields, this.filterValue, this.filterMatchMode, this.filterLocale);
                        if (filteredSubOptions && filteredSubOptions.length) {
                            let filteredGroup = {...optgroup};
                            filteredGroup[this.optionGroupChildren] = filteredSubOptions;
                            filteredGroups.push(filteredGroup);
                        }
                    }
                    return filteredGroups
                }
                else {
                    return api.FilterService.filter(this.options, this.searchFields, this.filterValue, this.filterMatchMode, this.filterLocale);
                }
            }
            else {
                return this.options;
            }
        },
        containerClass() {
            return [
                'p-dropdown p-component p-inputwrapper',
                {
                    'p-disabled': this.disabled,
                    'p-dropdown-clearable': this.showClear && !this.disabled,
                    'p-focus': this.focused,
                    'p-inputwrapper-filled': this.modelValue,
                    'p-inputwrapper-focus': this.focused || this.overlayVisible
                }
            ];
        },
        labelClass() {
            return [
                'p-dropdown-label p-inputtext',
                {
                    'p-placeholder': this.label === this.placeholder,
                    'p-dropdown-label-empty': !this.$slots['value'] && (this.label === 'p-emptylabel' || this.label.length === 0)
                }
            ];
        },
        panelStyleClass() {
            return ['p-dropdown-panel p-component', this.panelClass, {
                'p-input-filled': this.$primevue.config.inputStyle === 'filled',
                'p-ripple-disabled': this.$primevue.config.ripple === false
            }];
        },
        label() {
            let selectedOption = this.getSelectedOption();
            if (selectedOption !== null)
                return this.getOptionLabel(selectedOption);
            else
                return this.placeholder||'p-emptylabel';
        },
        editableInputValue() {
            let selectedOption = this.getSelectedOption();
            if (selectedOption)
                return this.getOptionLabel(selectedOption);
            else
                return this.modelValue;
        },
        equalityKey() {
            return this.optionValue ? null : this.dataKey;
        },
        searchFields() {
            return this.filterFields || [this.optionLabel];
        },
        emptyFilterMessageText() {
            return this.emptyFilterMessage || this.$primevue.config.locale.emptyFilterMessage;
        },
        emptyMessageText() {
            return this.emptyMessage || this.$primevue.config.locale.emptyMessage;
        },
        appendDisabled() {
            return this.appendTo === 'self';
        },
        virtualScrollerDisabled() {
            return !this.virtualScrollerOptions;
        },
        appendTarget() {
            return this.appendDisabled ? null : this.appendTo;
        },
        dropdownIconClass() {
            return ['p-dropdown-trigger-icon', this.loading ? this.loadingIcon : 'pi pi-chevron-down'];
        }
    },
    directives: {
        'ripple': Ripple__default["default"]
    },
    components: {
        'VirtualScroller': VirtualScroller__default["default"]
    }
};

const _hoisted_1 = { class: "p-hidden-accessible" };
const _hoisted_2 = ["id", "disabled", "tabindex", "aria-expanded", "aria-labelledby"];
const _hoisted_3 = ["disabled", "placeholder", "value", "aria-expanded"];
const _hoisted_4 = ["aria-expanded"];
const _hoisted_5 = {
  key: 0,
  class: "p-dropdown-header"
};
const _hoisted_6 = { class: "p-dropdown-filter-container" };
const _hoisted_7 = ["value", "placeholder"];
const _hoisted_8 = /*#__PURE__*/vue.createElementVNode("span", { class: "p-dropdown-filter-icon pi pi-search" }, null, -1);
const _hoisted_9 = ["onClick", "aria-label", "aria-selected"];
const _hoisted_10 = { class: "p-dropdown-item-group" };
const _hoisted_11 = ["onClick", "aria-label", "aria-selected"];
const _hoisted_12 = {
  key: 2,
  class: "p-dropdown-empty-message"
};
const _hoisted_13 = {
  key: 3,
  class: "p-dropdown-empty-message"
};

function render(_ctx, _cache, $props, $setup, $data, $options) {
  const _component_VirtualScroller = vue.resolveComponent("VirtualScroller");
  const _directive_ripple = vue.resolveDirective("ripple");

  return (vue.openBlock(), vue.createElementBlock("div", {
    ref: "container",
    class: vue.normalizeClass($options.containerClass),
    onClick: _cache[11] || (_cache[11] = $event => ($options.onClick($event)))
  }, [
    vue.createElementVNode("div", _hoisted_1, [
      vue.createElementVNode("input", {
        ref: "focusInput",
        type: "text",
        id: $props.inputId,
        readonly: "",
        disabled: $props.disabled,
        onFocus: _cache[0] || (_cache[0] = (...args) => ($options.onFocus && $options.onFocus(...args))),
        onBlur: _cache[1] || (_cache[1] = (...args) => ($options.onBlur && $options.onBlur(...args))),
        onKeydown: _cache[2] || (_cache[2] = (...args) => ($options.onKeyDown && $options.onKeyDown(...args))),
        tabindex: $props.tabindex,
        "aria-haspopup": "true",
        "aria-expanded": $data.overlayVisible,
        "aria-labelledby": $props.ariaLabelledBy
      }, null, 40, _hoisted_2)
    ]),
    ($props.editable)
      ? (vue.openBlock(), vue.createElementBlock("input", {
          key: 0,
          type: "text",
          class: "p-dropdown-label p-inputtext",
          disabled: $props.disabled,
          onFocus: _cache[3] || (_cache[3] = (...args) => ($options.onFocus && $options.onFocus(...args))),
          onBlur: _cache[4] || (_cache[4] = (...args) => ($options.onBlur && $options.onBlur(...args))),
          placeholder: $props.placeholder,
          value: $options.editableInputValue,
          onInput: _cache[5] || (_cache[5] = (...args) => ($options.onEditableInput && $options.onEditableInput(...args))),
          "aria-haspopup": "listbox",
          "aria-expanded": $data.overlayVisible
        }, null, 40, _hoisted_3))
      : vue.createCommentVNode("", true),
    (!$props.editable)
      ? (vue.openBlock(), vue.createElementBlock("span", {
          key: 1,
          class: vue.normalizeClass($options.labelClass)
        }, [
          vue.renderSlot(_ctx.$slots, "value", {
            value: $props.modelValue,
            placeholder: $props.placeholder
          }, () => [
            vue.createTextVNode(vue.toDisplayString($options.label||'empty'), 1)
          ])
        ], 2))
      : vue.createCommentVNode("", true),
    ($props.showClear && $props.modelValue != null)
      ? (vue.openBlock(), vue.createElementBlock("i", {
          key: 2,
          class: "p-dropdown-clear-icon pi pi-times",
          onClick: _cache[6] || (_cache[6] = $event => ($options.onClearClick($event)))
        }))
      : vue.createCommentVNode("", true),
    vue.createElementVNode("div", {
      class: "p-dropdown-trigger",
      role: "button",
      "aria-haspopup": "listbox",
      "aria-expanded": $data.overlayVisible
    }, [
      vue.renderSlot(_ctx.$slots, "indicator", {}, () => [
        vue.createElementVNode("span", {
          class: vue.normalizeClass($options.dropdownIconClass)
        }, null, 2)
      ])
    ], 8, _hoisted_4),
    (vue.openBlock(), vue.createBlock(vue.Teleport, {
      to: $options.appendTarget,
      disabled: $options.appendDisabled
    }, [
      vue.createVNode(vue.Transition, {
        name: "p-connected-overlay",
        onEnter: $options.onOverlayEnter,
        onAfterEnter: $options.onOverlayAfterEnter,
        onLeave: $options.onOverlayLeave,
        onAfterLeave: $options.onOverlayAfterLeave
      }, {
        default: vue.withCtx(() => [
          ($data.overlayVisible)
            ? (vue.openBlock(), vue.createElementBlock("div", {
                key: 0,
                ref: $options.overlayRef,
                class: vue.normalizeClass($options.panelStyleClass),
                onClick: _cache[10] || (_cache[10] = (...args) => ($options.onOverlayClick && $options.onOverlayClick(...args)))
              }, [
                vue.renderSlot(_ctx.$slots, "header", {
                  value: $props.modelValue,
                  options: $options.visibleOptions
                }),
                ($props.filter)
                  ? (vue.openBlock(), vue.createElementBlock("div", _hoisted_5, [
                      vue.createElementVNode("div", _hoisted_6, [
                        vue.createElementVNode("input", {
                          type: "text",
                          ref: "filterInput",
                          value: $data.filterValue,
                          onVnodeUpdated: _cache[7] || (_cache[7] = (...args) => ($options.onFilterUpdated && $options.onFilterUpdated(...args))),
                          autoComplete: "off",
                          class: "p-dropdown-filter p-inputtext p-component",
                          placeholder: $props.filterPlaceholder,
                          onKeydown: _cache[8] || (_cache[8] = (...args) => ($options.onFilterKeyDown && $options.onFilterKeyDown(...args))),
                          onInput: _cache[9] || (_cache[9] = (...args) => ($options.onFilterChange && $options.onFilterChange(...args)))
                        }, null, 40, _hoisted_7),
                        _hoisted_8
                      ])
                    ]))
                  : vue.createCommentVNode("", true),
                vue.createElementVNode("div", {
                  ref: $options.itemsWrapperRef,
                  class: "p-dropdown-items-wrapper",
                  style: vue.normalizeStyle({'max-height': $options.virtualScrollerDisabled ? $props.scrollHeight : ''})
                }, [
                  vue.createVNode(_component_VirtualScroller, vue.mergeProps({ ref: $options.virtualScrollerRef }, $props.virtualScrollerOptions, {
                    items: $options.visibleOptions,
                    style: {'height': $props.scrollHeight},
                    disabled: $options.virtualScrollerDisabled
                  }), vue.createSlots({
                    content: vue.withCtx(({ styleClass, contentRef, items, getItemOptions, contentStyle }) => [
                      vue.createElementVNode("ul", {
                        ref: contentRef,
                        class: vue.normalizeClass(['p-dropdown-items', styleClass]),
                        style: vue.normalizeStyle(contentStyle),
                        role: "listbox"
                      }, [
                        (!$props.optionGroupLabel)
                          ? (vue.openBlock(true), vue.createElementBlock(vue.Fragment, { key: 0 }, vue.renderList(items, (option, i) => {
                              return vue.withDirectives((vue.openBlock(), vue.createElementBlock("li", {
                                class: vue.normalizeClass(['p-dropdown-item', {'p-highlight': $options.isSelected(option), 'p-disabled': $options.isOptionDisabled(option)}]),
                                key: $options.getOptionRenderKey(option),
                                onClick: $event => ($options.onOptionSelect($event, option)),
                                role: "option",
                                "aria-label": $options.getOptionLabel(option),
                                "aria-selected": $options.isSelected(option)
                              }, [
                                vue.renderSlot(_ctx.$slots, "option", {
                                  option: option,
                                  index: $options.getOptionIndex(i, getItemOptions)
                                }, () => [
                                  vue.createTextVNode(vue.toDisplayString($options.getOptionLabel(option)), 1)
                                ])
                              ], 10, _hoisted_9)), [
                                [_directive_ripple]
                              ])
                            }), 128))
                          : (vue.openBlock(true), vue.createElementBlock(vue.Fragment, { key: 1 }, vue.renderList(items, (optionGroup, i) => {
                              return (vue.openBlock(), vue.createElementBlock(vue.Fragment, {
                                key: $options.getOptionGroupRenderKey(optionGroup)
                              }, [
                                vue.createElementVNode("li", _hoisted_10, [
                                  vue.renderSlot(_ctx.$slots, "optiongroup", {
                                    option: optionGroup,
                                    index: $options.getOptionIndex(i, getItemOptions)
                                  }, () => [
                                    vue.createTextVNode(vue.toDisplayString($options.getOptionGroupLabel(optionGroup)), 1)
                                  ])
                                ]),
                                (vue.openBlock(true), vue.createElementBlock(vue.Fragment, null, vue.renderList($options.getOptionGroupChildren(optionGroup), (option, i) => {
                                  return vue.withDirectives((vue.openBlock(), vue.createElementBlock("li", {
                                    class: vue.normalizeClass(['p-dropdown-item', {'p-highlight': $options.isSelected(option), 'p-disabled': $options.isOptionDisabled(option)}]),
                                    key: $options.getOptionRenderKey(option),
                                    onClick: $event => ($options.onOptionSelect($event, option)),
                                    role: "option",
                                    "aria-label": $options.getOptionLabel(option),
                                    "aria-selected": $options.isSelected(option)
                                  }, [
                                    vue.renderSlot(_ctx.$slots, "option", {
                                      option: option,
                                      index: $options.getOptionIndex(i, getItemOptions)
                                    }, () => [
                                      vue.createTextVNode(vue.toDisplayString($options.getOptionLabel(option)), 1)
                                    ])
                                  ], 10, _hoisted_11)), [
                                    [_directive_ripple]
                                  ])
                                }), 128))
                              ], 64))
                            }), 128)),
                        ($data.filterValue && (!items || (items && items.length === 0)))
                          ? (vue.openBlock(), vue.createElementBlock("li", _hoisted_12, [
                              vue.renderSlot(_ctx.$slots, "emptyfilter", {}, () => [
                                vue.createTextVNode(vue.toDisplayString($options.emptyFilterMessageText), 1)
                              ])
                            ]))
                          : ((!$props.options || ($props.options && $props.options.length === 0)))
                            ? (vue.openBlock(), vue.createElementBlock("li", _hoisted_13, [
                                vue.renderSlot(_ctx.$slots, "empty", {}, () => [
                                  vue.createTextVNode(vue.toDisplayString($options.emptyMessageText), 1)
                                ])
                              ]))
                            : vue.createCommentVNode("", true)
                      ], 6)
                    ]),
                    _: 2
                  }, [
                    (_ctx.$slots.loader)
                      ? {
                          name: "loader",
                          fn: vue.withCtx(({ options }) => [
                            vue.renderSlot(_ctx.$slots, "loader", { options: options })
                          ])
                        }
                      : undefined
                  ]), 1040, ["items", "style", "disabled"])
                ], 4),
                vue.renderSlot(_ctx.$slots, "footer", {
                  value: $props.modelValue,
                  options: $options.visibleOptions
                })
              ], 2))
            : vue.createCommentVNode("", true)
        ]),
        _: 3
      }, 8, ["onEnter", "onAfterEnter", "onLeave", "onAfterLeave"])
    ], 8, ["to", "disabled"]))
  ], 2))
}

function styleInject(css, ref) {
  if ( ref === void 0 ) ref = {};
  var insertAt = ref.insertAt;

  if (!css || typeof document === 'undefined') { return; }

  var head = document.head || document.getElementsByTagName('head')[0];
  var style = document.createElement('style');
  style.type = 'text/css';

  if (insertAt === 'top') {
    if (head.firstChild) {
      head.insertBefore(style, head.firstChild);
    } else {
      head.appendChild(style);
    }
  } else {
    head.appendChild(style);
  }

  if (style.styleSheet) {
    style.styleSheet.cssText = css;
  } else {
    style.appendChild(document.createTextNode(css));
  }
}

var css_248z = "\n.p-dropdown {\n    display: -webkit-inline-box;\n    display: -ms-inline-flexbox;\n    display: inline-flex;\n    cursor: pointer;\n    position: relative;\n    -webkit-user-select: none;\n       -moz-user-select: none;\n        -ms-user-select: none;\n            user-select: none;\n}\n.p-dropdown-clear-icon {\n    position: absolute;\n    top: 50%;\n    margin-top: -.5rem;\n}\n.p-dropdown-trigger {\n    display: -webkit-box;\n    display: -ms-flexbox;\n    display: flex;\n    -webkit-box-align: center;\n        -ms-flex-align: center;\n            align-items: center;\n    -webkit-box-pack: center;\n        -ms-flex-pack: center;\n            justify-content: center;\n    -ms-flex-negative: 0;\n        flex-shrink: 0;\n}\n.p-dropdown-label {\n    display: block;\n    white-space: nowrap;\n    overflow: hidden;\n    -webkit-box-flex: 1;\n        -ms-flex: 1 1 auto;\n            flex: 1 1 auto;\n    width: 1%;\n    text-overflow: ellipsis;\n    cursor: pointer;\n}\n.p-dropdown-label-empty {\n    overflow: hidden;\n    visibility: hidden;\n}\ninput.p-dropdown-label  {\n    cursor: default;\n}\n.p-dropdown .p-dropdown-panel {\n    min-width: 100%;\n}\n.p-dropdown-panel {\n    position: absolute;\n    top: 0;\n    left: 0;\n}\n.p-dropdown-items-wrapper {\n    overflow: auto;\n}\n.p-dropdown-item {\n    cursor: pointer;\n    font-weight: normal;\n    white-space: nowrap;\n    position: relative;\n    overflow: hidden;\n}\n.p-dropdown-item-group {\n    cursor: auto;\n}\n.p-dropdown-items {\n    margin: 0;\n    padding: 0;\n    list-style-type: none;\n}\n.p-dropdown-filter {\n    width: 100%;\n}\n.p-dropdown-filter-container {\n    position: relative;\n}\n.p-dropdown-filter-icon {\n    position: absolute;\n    top: 50%;\n    margin-top: -.5rem;\n}\n.p-fluid .p-dropdown {\n    display: -webkit-box;\n    display: -ms-flexbox;\n    display: flex;\n}\n.p-fluid .p-dropdown .p-dropdown-label {\n    width: 1%;\n}\n";
styleInject(css_248z);

script.render = render;

module.exports = script;
