/**
 * @license 
 * Copyright (c) 2015, Immo Schulz-Gerlach, www.isg-software.de 
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are 
 * permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this list of
 * conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list
 * of conditions and the following disclaimer in the documentation and/or other materials
 * provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY 
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
 * SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED 
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY
 * WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
 
 
/**
 * Namespace of jQuery. Usually bound to the alias <code>$</code>.
 *  
 * @see http://jquery.com/
 * @namespace jQuery 
 */
 
/**
 * Namespace for jQuery plug-ins.
 *  
 * @see http://jquery.com/
 * @namespace fn
 * @memberOf jQuery
 */
 

(function( $ ) {
	"use strict";
	
	/**
	 * Namespace for this jQuery plugin
	 * @namespace progressPie
	 * @memberOf jQuery.fn
	 */
	 
	 var setupDataKey = "$.fn.setupProgressPie";
	 
	/**
	 * Stores options for the progressPie plug-in. If this plug-in function is called, any succeeding calls to the progressPie plug-in
	 * without argument will behave the same like when called with the options stored here.
	 * This is recommended, if the progressPie plug-in gets called repeatedly (to update its graphic due to changed values).
	 * Then, this setup provides the means to set the options only once and to keep update calls simple instead of
	 * calling the progressPie repeatedly with the same options argument over and over again.
	 * <p>The "update" option, if not specified in the options of this call, will default to true (regardless of
 	 * the value defined in $.fn.progressPie.defaults.</p>
	 * <p>Usage pattern:</p>
	 * <pre>code>$(selector).setupProgressPie({options}).progressPie();
	 * update value;
	 * $(selector).progresssPie(); //update the graphic using the same options.
	 * </code></pre>
	 * <p>Repeated calls of setupProgressPie are allowed and update the options: The options of the subsequent call
	 * get merged into the existing setup. Example:
	 * <pre><code>
	 * $(selector).setupProgressPie({color: "green", strokeWidth: 3});
	 * ...
	 * $(selector).setupProgressPie({color: "navy"});
	 * </code></pre>
	 * <p>In this example the second call will change the color for any following call of <code>progressPie()</code>, but
	 * will leave the <code>strokeWidth: 3</code> option untouched, i.e. will not reset it to the default.</p>
	 * @function setupProgressPie()
	 * @memberOf jQuery.fn
	 * @param options - object containing individual options (merged with default options)
	 * @return this / result set (for chainable method calls on the result set)
	 */
	$.fn.setupProgressPie = function( options ) {
		var existingSetup = $(this).data(setupDataKey);
		if (typeof existingSetup !== "object") {
			var opts = $.extend( {}, $.fn.progressPie.defaults, {update: true}, options );
			$(this).data(setupDataKey, opts);
		} else {
			$.extend(existingSetup, options);
		}
		return this;
	};

	/**
	 * This plug-in may be used to draw a piechart with only one filled pie (rest empty). 
	 * It is designed to be an alternative to a progress bar, since it simply depicts a single value (in percent).
	 * The plug-in assumes that values in percent are part of the document (either visible or as attribute value)
	 * and a small pie representing each value is to be dynamically inserted. This may be either a static display
	 * or the pie may be updated upon data changes.
	 * 
	 * <p>Typical application: Append or prepend to visible percent value:<br>
	 * This mode assumes by default, that a value (integer number between 0 and 100 inclusive, 
	 * floating point numbers are supported, but truncated) is the only text content of an HTML element, e.g. a span
	 * element, and the pie is to be prepended (or appended) to the same element. The pie will usually be auto-sized to fit 
	 * into the text line. A separator String to be placed between the pie and the number may be configured.
	 * Defaults are to prepend the pie and use a non-breaking space as separator.
	 * E.g. say you have HTML code like <code>&lt;p&gt;You have achieved 25 points out of 50 (&lt;span class="pie"&gt;50&lt;/span&gt;%)&lt;/p&gt;</code>,
	 * then you may insert a pie filled by 50% with <code>$(.pie).progressPie();</code>, resulting in a line like:
	 * <code>&lt;p&gt;You have achieved 25 points out of 50 (&lt;span class="pie"&gt;&lt;svg&gt;the pie chart&lt;/svg&gt;&amp;nbsp;50&lt;/span&gt;%)&lt;/p&gt;</code>
	 * 
	 * <p>Usage:
	 * Select the elements holding the percent number and to insert the pie into by a jQuery selector.
	 * On the jQuery result set call "progressPie(options)", where options is an optional object
	 * with configuration options. See <a href="index.html">Home</a> (or README) for a documentation of supported options.
	 * The plugin is applied to any element in the result set, i.e. if the selector did not found any matching
	 * element, nothing will happen, while if the selector found several matching elements, the plugin
	 * will try to insert a corresponding pie into each of the found elements individually.
	 * 
	 * <p>The progressPie method will <code>return this</code>, enabling chaining of method calls
	 * on the result set.</p>
	 *
	 * @function progressPie()
	 * @memberOf jQuery.fn
	 * @param options - object containing individual options (merged with default options)
	 * @return this / result set (for chainable method calls on the result set)
	 */
	$.fn.progressPie = function( options ) {
		//Note: Normally the @function directive for jsDoc should not contain the parentheses "()".
		//But I needed to add something to the name in order to be able to document the plugin function and its namespace
		//separately (though in reality both are the same).

		// Extend our default options with those provided.
		// Note that the first argument to extend is an empty
		// object – this is to keep from overriding our "defaults" object.
		var globalOpts = $.extend( {}, $.fn.progressPie.defaults, options );
		var noargs = typeof options === "undefined";
		//If noargs === true and the setupProgressPie plug-in has been called for a target element, don't use "globalOpts", but use the stored setup instead.
		//Since any element in the result set may have a different individual setup, this decision can't be made here globally, but has to be made individually in 
		//the forEach loop below...

		var NS = "http://www.w3.org/2000/svg";
		var contentPluginNS = "jQuery.fn.progressPie.contentPlugin";
		
		//Naming convention:
		// "self" is used to refer to the function object in order to simplify access to public members.
		// "me", on the other hand, may be used in functions to save the object reference "this" (or "$(this)").
		var self = $.fn.progressPie;
		
		var internalMode = $.extend( {USER_COLOR_CONST:{}, USER_COLOR_FUNC:{}, DATA_ATTR_FUNC:{}}, self.Mode );


		//private functions
		function angle(percent) {
			return 0.02 * Math.PI * percent; //2 Pi * percent / 100
		}

		function evalDataAttrFunc(functionString, percent) {
			var evalIndirect = eval;
			var handler = evalIndirect(functionString);
			if (typeof handler === "function") {
				return handler(percent);
			} else {
				throw "The value of the colorFunctionAttr attribute is NOT a function: "+ functionString;
			}
		}
		
		function evalContentPluginName(name) {
			var evalIndirect = eval;
			var f = evalIndirect(contentPluginNS + "." + name);
			if (typeof f === "function") {
				return f;
			} else {
				throw name + " is not the name of a function in namespace " + contentPluginNS + "!";
			}
		}
		
		function getContentPlugin(property) {
			var f;
			if (typeof property === 'function') {
				f = property;
			} else if (typeof property === 'string') {
				f = evalContentPluginName(property);
			} else {
				throw "contentPlugin option must either be a function or the name of a function in the namespace " + contentPluginNS + "!";
			}
			return f;
		}

		function drawPie(svg, rad, strokeWidth, strokeColor, ringWidth, ringEndsRounded, percent, color, rotation) {
			
			//strokeWidth or ringWidth must not be greater than the radius:
			if (typeof strokeWidth === 'number') {
				strokeWidth = Math.min(strokeWidth, rad);
			}
			if (typeof ringWidth === 'number') {
				ringWidth = Math.min(ringWidth, rad);
			}

			var circle;
			if (percent < 100 || ringWidth && ringWidth > 0 && ringWidth < strokeWidth) {
				//1. background Circle 						
				circle = document.createElementNS(NS, "circle");
				circle.setAttribute("cx", 0);
				circle.setAttribute("cy", 0);
				circle.setAttribute("r", rad - strokeWidth / 2);
				circle.style.stroke = typeof strokeColor === 'string' ? strokeColor : color;
				circle.style.strokeWidth = strokeWidth;
				circle.style.fill = "none";
				svg.appendChild(circle);
			}
			
			if (percent === 100) {
				//"value" circle (full pie or ring)
				circle = document.createElementNS(NS, "circle");
				circle.setAttribute("cx", 0);
				circle.setAttribute("cy", 0);
				var sw = ringWidth ? ringWidth : strokeWidth;
				circle.setAttribute("r", rad - sw / 2);
				circle.style.stroke = color;
				circle.style.strokeWidth = sw;
				circle.style.fill = !ringWidth ? color : "none";
				svg.appendChild(circle);
			}  
			if (percent > 0 && percent < 100) {
				//2. Pie (or ring)
				var arc = document.createElementNS(NS, "path");
				var alpha = angle(percent);
				var targetX = Math.sin(alpha)*rad;
				var targetY = Math.cos(alpha-Math.PI)*rad;
				var largeArcFlag = percent > 50 ? "1" : "0";
				var clockwiseFlag = "1";
				var innerrad = ringWidth ? rad - ringWidth : 0;
				var starty =  -innerrad;
				//start
				var path = "M0,"+starty;
				var rrad; //radius for rounded ring ends
				if (ringWidth && ringEndsRounded) {
					rrad = ringWidth / 2;
					path += "a"+rrad+","+rrad+" 0 0,"+clockwiseFlag+" 0,-"+ringWidth;
				} else {
					path += " v-"+(ringWidth?ringWidth:rad);
				}
				//outer arc
				path += " A"+rad+","+rad+" 0 "+largeArcFlag+","+clockwiseFlag+" "+targetX+","+targetY;
				if (!ringWidth) {
					//pie: simply close path
					path += " Z";
				} else {
					//ring: 
					var innerStartX = Math.sin(alpha)*innerrad;
					var innerStartY = Math.cos(alpha-Math.PI)*innerrad;
					var anticlockwiseFlag = "0";
					//-move inwards on radius by ringWidth
					if (ringEndsRounded) {
						path += " A"+rrad+","+rrad+" 0 0,"+clockwiseFlag+" "+innerStartX+","+innerStartY;
					} else {
						path += " L"+innerStartX+","+innerStartY;
					}
					//-inner arc
					path += " A"+innerrad+","+innerrad+" 0 "+largeArcFlag+","+anticlockwiseFlag+" 0,-"+innerrad;
				}

				arc.setAttribute("d", path);
				arc.style.fill = color;
				arc.style.stroke = "none";
				if (rotation) {
					//rotation is "truthy".
					//May be "true" or a String (i.e. duration) or an object holding properties "duration" and "clockwise".
					var anticlockwise = rotation.clockwise === false;
					var dur = typeof rotation === "string" ? rotation :
							  typeof rotation.duration === "string" ? rotation.duration :
							  "1s"; //Default duration for true or any other truthy value is 1 second.
					var anim = document.createElementNS(NS, "animateTransform");
					anim.setAttribute("attributeName", "transform");
					anim.setAttribute("attributeType", "XML");
					anim.setAttribute("type", "rotate");
					anim.setAttribute("from", "0");
					anim.setAttribute("to", anticlockwise ? "-360" : "360");
					anim.setAttribute("dur", dur);
					anim.setAttribute("repeatDur", "indefinite");
					arc.appendChild(anim);
				}
				svg.appendChild(arc);
			}
		}
		
		function getRawValueStringOrNumber(me, opts) {
			var stringOrNumber;
			if (typeof opts.valueData === "string") {
				stringOrNumber = me.data(opts.valueData);
				if (typeof opts.valueAttr !== "undefined" || typeof opts.valueSelector !== "undefined") {
					throw "options 'valueData', 'valueAttr' and 'valueSelector' are mutually exclusive, i.e. at least two must be undefined!";
				}
			} else if (typeof opts.valueData !== "undefined") {
				throw "option 'valueData' is not of type 'string'!";
			} else if (typeof opts.valueAttr === "string") {
				stringOrNumber = me.attr(opts.valueAttr);
				if (typeof opts.valueSelector !== "undefined") {
					throw "options 'valueAttr' and 'valueSelector' are mutually exclusive, i.e. at least one must be undefined!";
				}
			} else if (typeof opts.valueAttr !== "undefined") {
				throw "option 'valueAttr' is not of type 'string'!";
			} else if (typeof opts.valueSelector !== "undefined") {
				stringOrNumber = $(opts.valueSelector, me).text();
			}
			if (typeof stringOrNumber === "undefined") {
				stringOrNumber = me.text();
			}
			return stringOrNumber;
		}
		
		function getPercentValue(rawValueStringOrNumber, opts) {
			return Math.max(0, Math.min(100, opts.valueAdapter(rawValueStringOrNumber)));
		}
		
		function getModeAndColor(me, opts) {
			var mode = opts.mode;
			var color = opts.color;
			//color may be a function or a constant
			var ct = typeof color;
			if (ct !== "undefined" && ct !== "string" && ct !== "function") {
				throw "option 'color' has to be either a function or a string, but is of type '" + ct + "'!";
			}
			if (ct === 'function') {
				mode = internalMode.USER_COLOR_FUNC;
			} else {
				if (ct === 'undefined' && typeof opts.colorAttr === "string") {
					color = me.attr(opts.colorAttr);
				}
				if (typeof color === 'string') {
					mode = internalMode.USER_COLOR_CONST;
				} else if (typeof opts.colorFunctionAttr === "string") {
					color = me.attr(opts.colorFunctionAttr);
					if (typeof color === 'string') {
						mode = internalMode.DATA_ATTR_FUNC;
					}
				}
			}
			return {mode: mode, color: color};
		}
		
		function calcColor(mode, userdefinedPieColor, percent) {
			return mode === internalMode.GREY ? internalMode.GREY.color :
						mode === internalMode.GREEN ? self.colorByPercent(100) :
						mode === internalMode.RED ? self.colorByPercent(0) :
						mode === internalMode.COLOR || userdefinedPieColor === undefined ? self.colorByPercent(percent) :
						mode === internalMode.USER_COLOR_CONST ? userdefinedPieColor :
						mode === internalMode.USER_COLOR_FUNC ? userdefinedPieColor(percent) :
						mode === internalMode.DATA_ATTR_FUNC ? evalDataAttrFunc(userdefinedPieColor, percent)
						: "black";
		}
 
 		$(this).each(function () {
			var me = $(this);
			var opts = globalOpts;
			if (noargs) {
				var localOpts = $(this).data(setupDataKey);
				if (typeof localOpts === "object") {
					opts = localOpts; //use stored individual setup instead of gobalOpts (which in this case (noargs) are just defaults anyway).
				}
			}
			var existing = $("svg", me); //existing SVGs in target element
			if (!existing.length || opts.update) { //Only draw if no SVG already existing or update mode
				if (existing.length && opts.update) { //remove existing SVG
					existing.remove();
					opts.separator = ''; //reset any separator when applying an update in order not to repeatedly insert a new one with each update.
				}
				var raw = getRawValueStringOrNumber(me, opts);
				var p = getPercentValue(raw, opts);
				var mc = getModeAndColor(me, opts);
				
				var h = Math.ceil(typeof opts.size === "number" ? opts.size : me.height());
				if (h === 0) {
					h = 20;
				}
				h *= opts.sizeFactor;
				var mid = Math.floor(h / 2);
				var rad = mid;

				var svg = document.createElementNS(NS, "svg");
				var scaledSize = h;
				if (typeof opts.scale === "number") {
					scaledSize *= opts.scale;
				}
				svg.setAttribute("width", scaledSize);
				svg.setAttribute("height", scaledSize);
				svg.setAttribute("viewBox", "-" + rad + " -" + rad + " " + h + " " + h);
				svg.style.verticalAlign = opts.verticalAlign;
				if (opts.prepend) {
					me.prepend(svg, opts.separator);
				} else {
					me.append(opts.separator, svg);
				}
				var color = calcColor(mc.mode, mc.color, p);
				drawPie(svg, rad, opts.strokeWidth, opts.strokeColor, opts.ringWidth, opts.ringEndsRounded, p, color, opts.rotation);
				
				var w = typeof opts.ringWidth === 'number' ? opts.ringWidth : typeof opts.strokeWidth === 'number' ? opts.strokeWidth : 0;
				
				//Draw a second, inner pie?
				if (typeof opts.inner === 'object') {
					if (typeof opts.inner.valueAdapter === "undefined") {
						opts.inner.valueAdapter = $.fn.progressPie.defaults.valueAdapter;
					}
					raw = getRawValueStringOrNumber(me, opts.inner);
					p = getPercentValue(raw, opts.inner);
					mc = getModeAndColor(me, opts.inner);
					rad = Math.floor(typeof opts.inner.size === "number" ? opts.inner.size * opts.sizeFactor / 2 : rad * 0.6);
					color = calcColor(mc.mode, mc.color, p);
					drawPie(svg, rad, 0, undefined, opts.inner.ringWidth, opts.inner.ringEndsRounded, p, color);
					
					w = typeof opts.inner.ringWidth === 'number' ? opts.inner.ringWidth : 0;
				}
				
				if (opts.contentPlugin) {
					var f = getContentPlugin(opts.contentPlugin);
					var r = rad;
					if (w < rad) {
						r -= w;	
					}
					var args = {
						newSvgElement: function(name) {
							var el = document.createElementNS(NS, name);
							svg.appendChild(el);
							return el;
						},
						newSvgSubelement: function(parent, name) {
							var el = document.createElementNS(NS, name);
							parent.appendChild(el);
							return el;
						},
						getContentPlugin: getContentPlugin,
						radius: r,
						color: color,
						percentValue: p,
						rawValue: raw,
						pieOpts: opts
					};
					if (typeof opts.contentPluginOptions === 'object') {
						$.extend(args, opts.contentPluginOptions);
					}
					f(args);
				}
			}
		});
		
		return this;
	};
	
	/**
	 * Enum defining possible values for the <code>mode</code> option.
	 * @memberOf jQuery.fn.progressPie
	 * @enum 
	 * @readonly
	 */
	$.fn.progressPie.Mode = {
		/** Default Mode: Pie is drawn in a shade of grey. The HTML color code is "#888" and may be changed by
	 	 * overwriting <code>jQuery.fn.progressPie.Mode.GREY.color</code> (of type string).
	 	 * @type {Object}
	 	 */
		GREY:{color:"#888"}, 
		/** In mode RED the pie is drawn in red color regardless of the percentual value. 
		 * <code>jQuery.fn.progressPie.Mode.RED.value</code> is a variable of type "number" with the default value
		 * of 200 and means the red color will be <code>rgb(200, 0, 0)</code>.
		 * The variable RED.value is not only used in mode RED, but also in mode COLOR for calculating the
		 * color of any value between 0 and 50.
		 * @type {Object}
		 */
		RED:{value:200},
		/** In mode GREEN the pie is drawn in green color regardless of the percentual value. 
		 * <code>jQuery.fn.progressPie.Mode.GREEN.value</code> is a variable of type "number" with the default value
		 * of 200 and means the green color will be <code>rgb(0, 200, 0)</code>.
		 * The variable GREEN.value is not only used in mode GREE, but also in mode COLOR for calculating the
		 * color of any value between 50 and 100.
		 * @type {Object}
		 */
		GREEN:{value:200}, 
		/** In mode COLOR the color of the pie is depending on the percentual value.
		 * The color is calculated via {@link "$.fn.progressPie".colorByPercent}.
		 * It's the same green color as in mode GREEN for a value of 100 percent, the same red color
		 * as in mode RED for a value of 0%.
		 * The colors may be altered by overwriting progressPie.Mode.RED.value or progressPie.Mode.GREEN.value.
		 * @type {Object}
		 */ 
		COLOR:{}
	};
	
	/** 
	 * public static method to calculate a color for a percent value: green for 100%, red for 0%, yellow for 50%, 
	 * gradients for values in between.
	 * This is used internally in mode progressPie.Mode.COLOR
	 * @param {number} percent - a value between 0 and 100 (inclusive). 0 results in red color, 100 in green, 50 in yellow,
	 * any other value greater than 50 generates a gradient between green and yellow, values less than 50 a gradient
	 * between red and yellow.
	 * @memberOf jQuery.fn.progressPie
	 * @function colorByPercent
	 */
	$.fn.progressPie.colorByPercent = function(percent) { 
		var maxGreen = $.fn.progressPie.Mode.GREEN.value;
		var maxRed = $.fn.progressPie.Mode.RED.value;
		var green = percent > 50 ? maxGreen : Math.floor(maxGreen * percent / 50);
		var red = percent < 50 ? maxRed : Math.floor(maxRed * (100 - percent) / 50);
		return "rgb(" + red + "," + green + ",0)";
	};
	

	/**
	 * Default Options.
	 * This is a public (static) object in order to allow users to globally modify the defaults
	 * before using the plug-in.
	 * @memberOf jQuery.fn.progressPie
	 * @member defaults
	 * @property {Mode} mode - A value of the enum type Mode, defaults to $.fn.progressPie.Mode.GREY
	 * @property {number} strokeWidth - The default width of the outer circle stroke, defaults to 2
	 * @property {boolean} prepend - true for prepending the SVG graph to the selected element's content, false for appending. Defaults to true.
	 * @property {string} separator - String to be inserted between prepended or appended SVG and target element's content.
	 * @property {string} verticalAlign - CSS value for the verticalAlign style attribute of the inserted SVG node (defaults to "bottom").
	 * @property {boolean} update - true will remove any SVG child from the selected target element before inserting a new image,
	 * false will only insert a new SVG if none exists yet. Defaults to false.
	 * @property {function} valueAdapter - Function takes a value (string or number) and returns a number in range (0..100),
	 * defaults to a function returning number values unchanged and applying parseInt to string values.
	 * @property {boolean} ringEndsRounded - If setting a ringWidth, this flag controls if the ends of the ring are simply
	 * cut (false) or if half a circle is appended to each end of the ring section (true). Defaults to false.
	 * @property {number} sizeFactor - Defaults to 1. The "original" diameter for the pie chart as either auto-sized
	 * or specified by the <code>size</code> option, is multiplied with this factor to get the final diameter before
	 * drawing the pie chart.
	 * @property {number} scale - Defaults to 1. The already rendered SVG graphic is finally scaled by this factor.
	 * In difference to <code>sizeFactor</code> this does not simply change the diameter/radius of the chart, but scales
	 * all other aspects such as the <code>strokeWidth</code>, too.
	 */
	$.fn.progressPie.defaults = {
		mode: $.fn.progressPie.Mode.GREY,
		strokeWidth: 2,
		prepend: true,
		separator: "&nbsp;",
		verticalAlign: "bottom",
		update: false,
		valueAdapter: function(value) {
			if (typeof value === "string") {
				return parseInt(value);
			} else if (typeof value === "number") {
				return value;
			} else {
				return 0;
			}
		},
		ringEndsRounded: false,
		sizeFactor: 1,
		scale: 1
	};
	
	/**
	 * Default namespace for content plug-ins.
	 * If you write contentPlugin functions, it is recommended to add them as members to this object
	 * (see bundled jquery-progresspiesvg-controlIcons.js for eample).
	 * Though you may use any function as a plugin (if it conforms to the plug-in interface),
	 * only functions within this default namespace may be specified by a string holding their function name
	 * in the <code>contentPlugin</code> option. Functions not in this namespace have to be referred to
	 * by a function reference (an expression evaluating to the very function object).
	 * @namespace contentPlugin
	 * @memberOf jQuery.fn.progressPie
	 */
	$.fn.progressPie.contentPlugin = {};
 
}( jQuery ));